! This file is part of tblite.
! SPDX-Identifier: LGPL-3.0-or-later
!
! tblite is free software: you can redistribute it and/or modify it under
! the terms of the GNU Lesser General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.
!
! tblite is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU Lesser General Public License for more details.
!
! You should have received a copy of the GNU Lesser General Public License
! along with tblite.  If not, see <https://www.gnu.org/licenses/>.

module test_integral_overlap
   use mctc_env, only : wp
   use mctc_env_testing, only : new_unittest, unittest_type, error_type, check, &
      & test_failed
   use mctc_io, only : structure_type
   use mstore, only : get_structure
   use tblite_basis_type
   use tblite_basis_slater, only : slater_to_gauss
   use tblite_cutoff, only : get_lattice_points
   use tblite_integral_overlap

   implicit none
   private

   public :: collect_integral_overlap

   real(wp), parameter :: thr = 5e+6_wp*epsilon(1.0_wp)
   real(wp), parameter :: thr2 = sqrt(epsilon(1.0_wp))

contains


!> Collect all exported unit tests
subroutine collect_integral_overlap(testsuite)

   !> Collection of tests
   type(unittest_type), allocatable, intent(out) :: testsuite(:)

   testsuite = [ &
      new_unittest("overlap-alh3", test_overlap_alh3), &
      new_unittest("overlap-diat-alh3", test_overlap_diat_alh3), &
      new_unittest("overlap-bh3", test_overlap_bh3), &
      new_unittest("overlap-beh2", test_overlap_beh2), &
      new_unittest("overlap-ch4", test_overlap_ch4), &
      new_unittest("overlap-cl2", test_overlap_cl2), &
      new_unittest("overlap-diat-cl2", test_overlap_diat_cl2), &
      new_unittest("overlap-f2", test_overlap_f2), &
      new_unittest("overlap-h2", test_overlap_h2), &
      new_unittest("overlap-lih", test_overlap_lih), &
      new_unittest("overlap-diat-cecl3", test_overlap_diat_cecl3), &
      new_unittest("overlap-grad-ss", test_overlap_grad_ss), &
      new_unittest("overlap-grad-pp", test_overlap_grad_pp), &
      new_unittest("overlap-grad-dd", test_overlap_grad_dd), &
      new_unittest("overlap-grad-ff", test_overlap_grad_ff), &
      new_unittest("overlap-diat-grad-ss", test_overlap_diat_grad_ss), &
      new_unittest("overlap-diat-grad-ss_z", test_overlap_diat_grad_ss_z), &
      new_unittest("overlap-diat-grad-sp", test_overlap_diat_grad_sp), &
      new_unittest("overlap-diat-grad-sp_z", test_overlap_diat_grad_sp_z), &
      new_unittest("overlap-diat-grad-pp", test_overlap_diat_grad_pp), &
      new_unittest("overlap-diat-grad-pp_z", test_overlap_diat_grad_pp_z), &
      new_unittest("overlap-diat-grad-sd", test_overlap_diat_grad_sd), &
      new_unittest("overlap-diat-grad-sd_z", test_overlap_diat_grad_sd_z), &
      new_unittest("overlap-diat-grad-pd", test_overlap_diat_grad_pd), &
      new_unittest("overlap-diat-grad-pd_z", test_overlap_diat_grad_pd_z), &
      new_unittest("overlap-diat-grad-dd", test_overlap_diat_grad_dd), &
      new_unittest("overlap-diat-grad-dd_z", test_overlap_diat_grad_dd_z), &
      new_unittest("overlap-diat-grad-sf", test_overlap_diat_grad_sf), &
      new_unittest("overlap-diat-grad-sf_z", test_overlap_diat_grad_sf_z), &
      new_unittest("overlap-diat-grad-pf", test_overlap_diat_grad_pf), &
      new_unittest("overlap-diat-grad-pf_z", test_overlap_diat_grad_pf_z), &
      new_unittest("overlap-diat-grad-df", test_overlap_diat_grad_df), &
      new_unittest("overlap-diat-grad-df_z", test_overlap_diat_grad_df_z), &
      new_unittest("overlap-diat-grad-ff", test_overlap_diat_grad_ff), &
      new_unittest("overlap-diat-grad-ff_z", test_overlap_diat_grad_ff_z) &
      ]

end subroutine collect_integral_overlap


subroutine make_basis(bas, mol, ng)
   type(basis_type), intent(out) :: bas
   type(structure_type), intent(in) :: mol
   integer, intent(in) :: ng


   integer, parameter :: nsh(86) = [&
   & 1, 2, 2, 2, 2, 2, 2, 2, 2, 3, 2, 3, 3, 3, 3, 3, 3, 3, 2, 3, & ! 1-20
   & 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, & ! 21-40
   & 3, 3, 3, 3, 3, 3, 3, 2, 3, 3, 3, 3, 3, 3, 2, 3, 3, 4, 3, 3, & ! 41-60
   & 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 2, & ! 61-80
   & 3, 3, 3, 3, 3, 3]
   integer, parameter :: lsh(4, 86) = reshape([&
   & 0, 0, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0, & ! 1-6
   & 0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 0, 0,  0, 1, 2, 0, & ! 7-12
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 13-18
   & 0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 19-24
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 0, 0, & ! 25-30
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 31-36
   & 0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 37-42
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 0, 0, & ! 43-48
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 49-54
   & 0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 3,  0, 1, 2, 0,  0, 1, 2, 0, & ! 55-60
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 61-66
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 67-72
   & 0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 73-78
   & 0, 1, 2, 0,  0, 1, 0, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 0, & ! 79-84
   & 0, 1, 2, 0,  0, 1, 2, 0], &
      & shape(lsh))
   integer, parameter :: pqn(4, 86) = reshape([&
   & 1, 0, 0, 0,  1, 2, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0, & ! 1-6
   & 2, 2, 0, 0,  2, 2, 0, 0,  2, 2, 0, 0,  2, 2, 3, 0,  3, 3, 0, 0,  3, 3, 3, 0, & ! 7-12
   & 3, 3, 3, 0,  3, 3, 3, 0,  3, 3, 3, 0,  3, 3, 3, 0,  3, 3, 3, 0,  3, 3, 3, 0, & ! 13-18
   & 4, 4, 0, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0, & ! 19-24
   & 4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 3, 0,  4, 4, 0, 0, & ! 25-30
   & 4, 4, 4, 0,  4, 4, 4, 0,  4, 4, 4, 0,  4, 4, 4, 0,  4, 4, 4, 0,  4, 4, 4, 0, & ! 31-36
   & 5, 5, 0, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0, & ! 37-42
   & 5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 4, 0,  5, 5, 0, 0, & ! 43-48
   & 5, 5, 5, 0,  5, 5, 5, 0,  5, 5, 5, 0,  5, 5, 5, 0,  5, 5, 5, 0,  5, 5, 5, 0, & ! 49-54
   & 6, 6, 0, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 4,  6, 6, 5, 0,  6, 6, 5, 0, & ! 55-60
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 61-66
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 67-72
   & 6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 73-78
   & 6, 6, 5, 0,  6, 6, 0, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0,  6, 6, 5, 0, & ! 79-84
   & 6, 6, 5, 0,  6, 6, 5, 0], &
      & shape(pqn))
   real(wp), parameter :: zeta(4, 86) = reshape([&
   & 1.230000_wp, 0.000000_wp, 0.000000_wp, 0.000000_wp, &
   & 1.669667_wp, 1.500000_wp, 0.000000_wp, 0.000000_wp, & ! 2
   & 0.750060_wp, 0.557848_wp, 0.000000_wp, 0.000000_wp, &
   & 1.034720_wp, 0.949332_wp, 0.000000_wp, 0.000000_wp, &
   & 1.479444_wp, 1.479805_wp, 0.000000_wp, 0.000000_wp, &
   & 2.096432_wp, 1.800000_wp, 0.000000_wp, 0.000000_wp, &
   & 2.339881_wp, 2.014332_wp, 0.000000_wp, 0.000000_wp, &
   & 2.439742_wp, 2.137023_wp, 0.000000_wp, 0.000000_wp, &
   & 2.416361_wp, 2.308399_wp, 0.000000_wp, 0.000000_wp, &
   & 3.084104_wp, 2.312051_wp, 2.815609_wp, 0.000000_wp, & ! 10
   & 0.763787_wp, 0.573553_wp, 0.000000_wp, 0.000000_wp, &
   & 1.184203_wp, 0.717769_wp, 1.300000_wp, 0.000000_wp, &
   & 1.352531_wp, 1.391201_wp, 1.000000_wp, 0.000000_wp, &
   & 1.773917_wp, 1.718996_wp, 1.250000_wp, 0.000000_wp, &
   & 1.816945_wp, 1.903247_wp, 1.167533_wp, 0.000000_wp, &
   & 1.981333_wp, 2.025643_wp, 1.702555_wp, 0.000000_wp, &
   & 2.485265_wp, 2.199650_wp, 2.476089_wp, 0.000000_wp, &
   & 2.329679_wp, 2.149419_wp, 1.950531_wp, 0.000000_wp, & ! 18
   & 0.875961_wp, 0.631694_wp, 0.000000_wp, 0.000000_wp, &
   & 1.267130_wp, 0.786247_wp, 1.380000_wp, 0.000000_wp, &
   & 2.224492_wp, 1.554183_wp, 2.009535_wp, 0.000000_wp, &
   & 2.588796_wp, 0.994410_wp, 1.885617_wp, 0.000000_wp, &
   & 3.043706_wp, 4.030076_wp, 1.663291_wp, 0.000000_wp, &
   & 2.250127_wp, 2.706815_wp, 1.675019_wp, 0.000000_wp, &
   & 2.206053_wp, 2.820197_wp, 1.861022_wp, 0.000000_wp, &
   & 1.572970_wp, 1.986214_wp, 2.837906_wp, 0.000000_wp, &
   & 1.808266_wp, 1.736758_wp, 2.797674_wp, 0.000000_wp, &
   & 2.007589_wp, 2.250756_wp, 2.982916_wp, 0.000000_wp, &
   & 2.181599_wp, 2.384590_wp, 3.095025_wp, 0.000000_wp, &
   & 2.263767_wp, 2.203629_wp, 0.000000_wp, 0.000000_wp, &
   & 2.638221_wp, 2.067523_wp, 2.113616_wp, 0.000000_wp, &
   & 2.528919_wp, 2.194417_wp, 1.776619_wp, 0.000000_wp, &
   & 3.556676_wp, 2.420754_wp, 1.465797_wp, 0.000000_wp, &
   & 2.896526_wp, 2.454218_wp, 2.278836_wp, 0.000000_wp, &
   & 3.289210_wp, 2.565269_wp, 1.645016_wp, 0.000000_wp, &
   & 5.209881_wp, 2.843367_wp, 2.758388_wp, 0.000000_wp, & ! 36
   & 1.269729_wp, 1.887305_wp, 0.000000_wp, 0.000000_wp, &
   & 1.868807_wp, 1.785463_wp, 2.160122_wp, 0.000000_wp, &
   & 0.920018_wp, 1.457324_wp, 2.229013_wp, 0.000000_wp, &
   & 6.506473_wp, 1.432023_wp, 2.119714_wp, 0.000000_wp, &
   & 2.109733_wp, 2.799447_wp, 2.018973_wp, 0.000000_wp, &
   & 2.584133_wp, 3.027953_wp, 2.087336_wp, 0.000000_wp, &
   & 2.621415_wp, 3.134876_wp, 2.132598_wp, 0.000000_wp, &
   & 2.739844_wp, 2.181678_wp, 2.546096_wp, 0.000000_wp, &
   & 1.840571_wp, 2.974826_wp, 3.106937_wp, 0.000000_wp, &
   & 1.756228_wp, 3.394247_wp, 3.202653_wp, 0.000000_wp, &
   & 3.050188_wp, 2.349519_wp, 3.353329_wp, 0.000000_wp, &
   & 2.419991_wp, 2.288929_wp, 0.000000_wp, 0.000000_wp, &
   & 2.878139_wp, 2.446597_wp, 2.757735_wp, 0.000000_wp, &
   & 3.038232_wp, 2.320821_wp, 1.775133_wp, 0.000000_wp, &
   & 2.687507_wp, 2.385653_wp, 2.125961_wp, 0.000000_wp, &
   & 2.810717_wp, 2.452747_wp, 2.018718_wp, 0.000000_wp, &
   & 2.906869_wp, 2.493771_wp, 1.900737_wp, 0.000000_wp, &
   & 4.175313_wp, 2.869379_wp, 2.968948_wp, 0.000000_wp, & ! 54
   & 1.242993_wp, 1.991420_wp, 0.000000_wp, 0.000000_wp, &
   & 1.314003_wp, 1.164384_wp, 2.127596_wp, 0.000000_wp, &
   & 2.817373_wp, 1.698633_wp, 2.273697_wp, 0.000000_wp, &
   & 2.845039_wp, 1.460181_wp, 2.534989_wp, 2.534989_wp, &
   & 2.816971_wp, 1.475453_wp, 2.543502_wp, 0.000000_wp, &
   & 2.788903_wp, 1.490724_wp, 2.552016_wp, 0.000000_wp, &
   & 2.760835_wp, 1.505995_wp, 2.560529_wp, 0.000000_wp, &
   & 2.732767_wp, 1.521266_wp, 2.569042_wp, 0.000000_wp, &
   & 2.704699_wp, 1.536537_wp, 2.577556_wp, 0.000000_wp, &
   & 2.676631_wp, 1.551808_wp, 2.586069_wp, 0.000000_wp, &
   & 2.648563_wp, 1.567079_wp, 2.594583_wp, 0.000000_wp, &
   & 2.620495_wp, 1.582351_wp, 2.603096_wp, 0.000000_wp, &
   & 2.592427_wp, 1.597622_wp, 2.611609_wp, 0.000000_wp, &
   & 2.564359_wp, 1.612893_wp, 2.620123_wp, 0.000000_wp, &
   & 2.536291_wp, 1.628164_wp, 2.628636_wp, 0.000000_wp, &
   & 2.508223_wp, 1.643435_wp, 2.637150_wp, 0.000000_wp, &
   & 2.480155_wp, 1.658706_wp, 2.645663_wp, 0.000000_wp, &
   & 3.195377_wp, 2.248538_wp, 2.414921_wp, 0.000000_wp, &
   & 3.141220_wp, 2.487234_wp, 2.219335_wp, 0.000000_wp, &
   & 3.176612_wp, 3.395385_wp, 2.375027_wp, 0.000000_wp, &
   & 3.145383_wp, 2.583611_wp, 2.471393_wp, 0.000000_wp, &
   & 1.815656_wp, 2.481062_wp, 3.185853_wp, 0.000000_wp, &
   & 2.117984_wp, 2.858570_wp, 3.470484_wp, 0.000000_wp, &
   & 2.712412_wp, 3.378860_wp, 3.641249_wp, 0.000000_wp, &
   & 2.805724_wp, 2.825702_wp, 3.720644_wp, 0.000000_wp, &
   & 2.619513_wp, 2.696078_wp, 0.000000_wp, 0.000000_wp, &
   & 3.053831_wp, 2.616838_wp, 3.321796_wp, 0.000000_wp, &
   & 3.021350_wp, 2.592502_wp, 4.246744_wp, 0.000000_wp, &
   & 3.164052_wp, 2.632387_wp, 3.046255_wp, 0.000000_wp, &
   & 2.961334_wp, 2.713884_wp, 2.310225_wp, 0.000000_wp, &
   & 2.982405_wp, 2.959607_wp, 2.437783_wp, 0.000000_wp, &
   & 3.079362_wp, 2.685897_wp, 2.103113_wp, 0.000000_wp], & ! 86
      & shape(zeta))

   integer :: isp, izp, ish, stat
   integer, allocatable :: nshell(:)
   type(cgto_type), allocatable :: cgto(:, :)

   nshell = nsh(mol%num)
   allocate(cgto(maxval(nshell), mol%nid))
   do isp = 1, mol%nid
      izp = mol%num(isp)
      do ish = 1, nshell(isp)
         call slater_to_gauss(ng, pqn(ish, izp), lsh(ish, izp), zeta(ish, izp), &
            & cgto(ish, isp), .true., stat)
      end do
   end do

   call new_basis(bas, mol, nshell, cgto, 1.0_wp)

end subroutine make_basis


subroutine test_overlap_mol(error, mol, ref)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   type(structure_type), intent(in) :: mol
   real(wp), intent(in) :: ref(:, :)

   type(basis_type) :: bas
   real(wp), allocatable :: lattr(:, :), overlap(:, :)
   real(wp) :: cutoff
   integer :: ii, jj

   call make_basis(bas, mol, 6)
   call check(error, bas%nao, size(ref, 1))
   if (allocated(error)) return

   cutoff = get_cutoff(bas)
   call get_lattice_points(mol%periodic, mol%lattice, cutoff, lattr)

   allocate(overlap(bas%nao, bas%nao))
   call get_overlap(mol, lattr, cutoff, bas, overlap)

   !where(abs(overlap) < thr) overlap = 0.0_wp
   !print '(*(6x,"&", 3(es20.14e1, "_wp":, ","), "&", /))', overlap

   do ii = 1, size(overlap, 2)
      do jj = 1, size(overlap, 1)
         call check(error, overlap(jj, ii), ref(jj, ii), thr=thr)
         if (allocated(error)) return
      end do
   end do


end subroutine test_overlap_mol

subroutine test_overlap_diat_mol(error, mol, ref)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   type(structure_type), intent(in) :: mol
   real(wp), intent(in) :: ref(:, :)

   type(basis_type) :: bas
   real(wp), allocatable :: lattr(:, :), overlap(:, :), overlap_diat(:, :)
   real(wp) :: cutoff
   integer :: ii, jj
   real(wp) :: scalfac(3,86)

   scalfac = 1.2_wp

   call make_basis(bas, mol, 6)
   call check(error, bas%nao, size(ref, 1))
   if (allocated(error)) return

   cutoff = get_cutoff(bas)
   call get_lattice_points(mol%periodic, mol%lattice, cutoff, lattr)

   allocate(overlap(bas%nao, bas%nao), overlap_diat(bas%nao, bas%nao))
   call get_overlap(mol, lattr, cutoff, bas, scalfac, overlap, overlap_diat)

   do ii = 1, size(overlap_diat, 2)
      do jj = 1, size(overlap_diat, 1)
         call check(error, overlap_diat(jj, ii), ref(jj, ii), thr=thr)
         if (allocated(error)) return
      end do
   end do


end subroutine test_overlap_diat_mol

subroutine test_overlap_alh3(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 12
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 9.99999999869333E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 4.02664545809937E-1_wp, 4.02664545809937E-1_wp, 4.02664545809939E-1_wp,&
      & 0.00000000000000E+0_wp, 9.99999999998060E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 4.32775379754938E-1_wp,-4.32775379754938E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999998060E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999998060E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-2.49862982000156E-1_wp,-2.49862982000156E-1_wp, 4.99725964000314E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830206E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-3.40183623222704E-1_wp, 3.40183623222704E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830206E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999830206E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-2.26789082148469E-1_wp,-2.26789082148469E-1_wp,-2.26789082148469E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830206E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830206E-1_wp,&
      &-1.96405106441530E-1_wp,-1.96405106441530E-1_wp, 3.92810212883060E-1_wp,&
      & 4.02664545809937E-1_wp, 4.32775379754938E-1_wp, 0.00000000000000E+0_wp,&
      &-2.49862982000156E-1_wp,-3.40183623222704E-1_wp, 0.00000000000000E+0_wp,&
      &-2.26789082148469E-1_wp, 0.00000000000000E+0_wp,-1.96405106441530E-1_wp,&
      & 9.99999999881495E-1_wp, 3.54600353330803E-2_wp, 3.54600353330805E-2_wp,&
      & 4.02664545809937E-1_wp,-4.32775379754938E-1_wp, 0.00000000000000E+0_wp,&
      &-2.49862982000156E-1_wp, 3.40183623222704E-1_wp, 0.00000000000000E+0_wp,&
      &-2.26789082148469E-1_wp, 0.00000000000000E+0_wp,-1.96405106441530E-1_wp,&
      & 3.54600353330803E-2_wp, 9.99999999881495E-1_wp, 3.54600353330805E-2_wp,&
      & 4.02664545809939E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 4.99725964000314E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-2.26789082148469E-1_wp, 0.00000000000000E+0_wp, 3.92810212883060E-1_wp,&
      & 3.54600353330805E-2_wp, 3.54600353330805E-2_wp, 9.99999999881495E-1_wp],&
      & shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "AlH3")
   call test_overlap_mol(error, mol, overlap)

end subroutine test_overlap_alh3

subroutine test_overlap_diat_alh3(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 12
   real(wp), parameter :: overlap_diat(nao, nao) = reshape([&
      & 9.99999999869333E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 4.83197454971925E-1_wp, 4.83197454971925E-1_wp, 4.83197454971926E-1_wp,&
      & 0.00000000000000E+0_wp, 9.99999999998060E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 5.19330455705926E-1_wp,-5.19330455705926E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999998060E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999998060E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-2.99835578400188E-1_wp,-2.99835578400188E-1_wp, 5.99671156800377E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830206E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-4.08220347867244E-1_wp, 4.08220347867244E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830206E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999830206E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-2.72146898578163E-1_wp,-2.72146898578163E-1_wp,-2.72146898578163E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830206E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830206E-1_wp,&
      &-2.35686127729836E-1_wp,-2.35686127729836E-1_wp, 4.71372255459672E-1_wp,&
      & 4.83197454971925E-1_wp, 5.19330455705926E-1_wp, 0.00000000000000E+0_wp,&
      &-2.99835578400188E-1_wp,-4.08220347867244E-1_wp, 0.00000000000000E+0_wp,&
      &-2.72146898578163E-1_wp, 0.00000000000000E+0_wp,-2.35686127729836E-1_wp,&
      & 9.99999999881495E-1_wp, 4.25520423996964E-2_wp, 4.25520423996966E-2_wp,&
      & 4.83197454971925E-1_wp,-5.19330455705926E-1_wp, 0.00000000000000E+0_wp,&
      &-2.99835578400188E-1_wp, 4.08220347867244E-1_wp, 0.00000000000000E+0_wp,&
      &-2.72146898578163E-1_wp, 0.00000000000000E+0_wp,-2.35686127729836E-1_wp,&
      & 4.25520423996964E-2_wp, 9.99999999881495E-1_wp, 4.25520423996966E-2_wp,&
      & 4.83197454971927E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 5.99671156800377E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-2.72146898578163E-1_wp, 0.00000000000000E+0_wp, 4.71372255459672E-1_wp,&
      & 4.25520423996966E-2_wp, 4.25520423996966E-2_wp, 9.99999999881495E-1_wp],&
      & shape(overlap_diat))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "AlH3")
   call test_overlap_diat_mol(error, mol, overlap_diat)

end subroutine test_overlap_diat_alh3

subroutine test_overlap_bh3(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 7
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 1.00000000000600E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 4.80899928496809E-1_wp, 4.80899928496809E-1_wp,&
      & 4.80899928496810E-1_wp, 0.00000000000000E+0_wp, 9.99999999925689E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 4.26420603078617E-1_wp,&
      &-4.26420603078617E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999925689E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp,-2.46194049975443E-1_wp,-2.46194049975443E-1_wp,&
      & 4.92388099950885E-1_wp, 4.80899928496809E-1_wp, 4.26420603078617E-1_wp,&
      & 0.00000000000000E+0_wp,-2.46194049975443E-1_wp, 9.99999999881495E-1_wp,&
      & 1.10583333710332E-1_wp, 1.10583333710331E-1_wp, 4.80899928496809E-1_wp,&
      &-4.26420603078617E-1_wp, 0.00000000000000E+0_wp,-2.46194049975443E-1_wp,&
      & 1.10583333710332E-1_wp, 9.99999999881495E-1_wp, 1.10583333710331E-1_wp,&
      & 4.80899928496810E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 4.92388099950885E-1_wp, 1.10583333710331E-1_wp, 1.10583333710331E-1_wp,&
      & 9.99999999881495E-1_wp],shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "BH3")
   call test_overlap_mol(error, mol, overlap)

end subroutine test_overlap_bh3

subroutine test_overlap_beh2(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 6
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 1.00000000000600E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 4.75332491800271E-1_wp, 4.75332491800271E-1_wp,&
      & 0.00000000000000E+0_wp, 9.99999999925689E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999925689E-1_wp,&
      & 0.00000000000000E+0_wp,-5.55623013601629E-1_wp, 5.55623013601629E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 4.75332491800271E-1_wp, 0.00000000000000E+0_wp,-5.55623013601629E-1_wp,&
      & 0.00000000000000E+0_wp, 9.99999999881495E-1_wp, 4.00645037406879E-2_wp,&
      & 4.75332491800271E-1_wp, 0.00000000000000E+0_wp, 5.55623013601629E-1_wp,&
      & 0.00000000000000E+0_wp, 4.00645037406879E-2_wp, 9.99999999881495E-1_wp],&
      & shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "BeH2")
   call test_overlap_mol(error, mol, overlap)

end subroutine test_overlap_beh2

subroutine test_overlap_ch4(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 8
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 1.00000000000600E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 4.22144465267144E-1_wp, 4.22144465267144E-1_wp,&
      & 4.22144465267144E-1_wp, 4.22144465267144E-1_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 2.57682345348632E-1_wp,-2.57682345348632E-1_wp,-2.57682345348632E-1_wp,&
      & 2.57682345348632E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 2.57682345348632E-1_wp,&
      & 2.57682345348632E-1_wp,-2.57682345348632E-1_wp,-2.57682345348632E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp,-2.57682345348632E-1_wp, 2.57682345348632E-1_wp,&
      &-2.57682345348632E-1_wp, 2.57682345348632E-1_wp, 4.22144465267144E-1_wp,&
      & 2.57682345348632E-1_wp, 2.57682345348632E-1_wp,-2.57682345348632E-1_wp,&
      & 9.99999999881495E-1_wp, 1.71953274424808E-1_wp, 1.71953274424808E-1_wp,&
      & 1.71953274424808E-1_wp, 4.22144465267144E-1_wp,-2.57682345348632E-1_wp,&
      & 2.57682345348632E-1_wp, 2.57682345348632E-1_wp, 1.71953274424808E-1_wp,&
      & 9.99999999881495E-1_wp, 1.71953274424808E-1_wp, 1.71953274424808E-1_wp,&
      & 4.22144465267144E-1_wp,-2.57682345348632E-1_wp,-2.57682345348632E-1_wp,&
      &-2.57682345348632E-1_wp, 1.71953274424808E-1_wp, 1.71953274424808E-1_wp,&
      & 9.99999999881495E-1_wp, 1.71953274424808E-1_wp, 4.22144465267144E-1_wp,&
      & 2.57682345348632E-1_wp,-2.57682345348632E-1_wp, 2.57682345348632E-1_wp,&
      & 1.71953274424808E-1_wp, 1.71953274424808E-1_wp, 1.71953274424808E-1_wp,&
      & 9.99999999881495E-1_wp],shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "CH4")
   call test_overlap_mol(error, mol, overlap)

end subroutine test_overlap_ch4

subroutine test_overlap_cl2(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 18
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 9.99999999869332E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.17210205959034E-2_wp, 0.00000000000000E+0_wp,-1.66880110378166E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.08719248478375E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999998060E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 6.66413853324784E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,-7.28795473739356E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999998060E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.66880110378166E-1_wp, 0.00000000000000E+0_wp,-2.43857348208300E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.54878516243462E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999998060E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 6.66413853324784E-2_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp,-7.28795473739356E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830205E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 1.65407586749691E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830205E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 7.28795473739356E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,-8.83217466681701E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999830206E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.08719248478375E-1_wp, 0.00000000000000E+0_wp,-1.54878516243462E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.20587356016693E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830205E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 7.28795473739356E-2_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp,-8.83217466681701E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830206E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 1.65407586749691E-2_wp,&
      & 9.17210205959034E-2_wp, 0.00000000000000E+0_wp, 1.66880110378166E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.08719248478375E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999869332E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 6.66413853324784E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 7.28795473739356E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999998060E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-1.66880110378166E-1_wp, 0.00000000000000E+0_wp,-2.43857348208300E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-1.54878516243462E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999998060E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 6.66413853324784E-2_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 7.28795473739356E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999998060E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 1.65407586749691E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830205E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp,-7.28795473739356E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,-8.83217466681701E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830205E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.08719248478375E-1_wp, 0.00000000000000E+0_wp, 1.54878516243462E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.20587356016693E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999830206E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-7.28795473739356E-2_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp,-8.83217466681701E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830205E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 1.65407586749691E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830206E-1_wp],&
      & shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "Cl2")
   call test_overlap_mol(error, mol, overlap)

end subroutine test_overlap_cl2

subroutine test_overlap_diat_cl2(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 18
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 9.99999999869332E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.10065224715084E-1_wp, 0.00000000000000E+0_wp,-2.00256132453799E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.30463098174050E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999998060E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 7.99696623989741E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,-8.74554568487227E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999998060E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 2.00256132453799E-1_wp, 0.00000000000000E+0_wp,-2.92628817849960E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.85854219492155E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999998060E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 7.99696623989741E-2_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp,-8.74554568487227E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830205E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 1.98489104099629E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830205E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 8.74554568487227E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,-1.05986096001804E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999830206E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.30463098174050E-1_wp, 0.00000000000000E+0_wp,-1.85854219492155E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.44704827220032E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830205E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 8.74554568487227E-2_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp,-1.05986096001804E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830206E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 1.98489104099629E-2_wp,&
      & 1.10065224715084E-1_wp, 0.00000000000000E+0_wp, 2.00256132453799E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.30463098174050E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999869332E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 7.99696623989741E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 8.74554568487227E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999998060E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-2.00256132453799E-1_wp, 0.00000000000000E+0_wp,-2.92628817849960E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-1.85854219492155E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999998060E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 7.99696623989741E-2_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 8.74554568487227E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999998060E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 1.98489104099629E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830205E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp,-8.74554568487227E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,-1.05986096001804E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830205E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.30463098174050E-1_wp, 0.00000000000000E+0_wp, 1.85854219492155E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 1.44704827220032E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999830206E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-8.74554568487227E-2_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp,-1.05986096001804E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 9.99999999830205E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 1.98489104099629E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 9.99999999830206E-1_wp],&
      & shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "Cl2")
   call test_overlap_diat_mol(error, mol, overlap)

end subroutine test_overlap_diat_cl2

subroutine test_overlap_f2(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 8
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 1.00000000000600E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 1.47958427103274E-1_wp, 0.00000000000000E+0_wp,&
      &-2.00620208590218E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 7.94087596423658E-2_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 2.00620208590218E-1_wp,&
      & 0.00000000000000E+0_wp,-2.36369347922990E-1_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 7.94087596423658E-2_wp, 1.47958427103274E-1_wp,&
      & 0.00000000000000E+0_wp, 2.00620208590218E-1_wp, 0.00000000000000E+0_wp,&
      & 1.00000000000600E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 7.94087596423658E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      &-2.00620208590218E-1_wp, 0.00000000000000E+0_wp,-2.36369347922990E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 7.94087596423658E-2_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp],shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "F2")
   call test_overlap_mol(error, mol, overlap)

end subroutine test_overlap_f2

subroutine test_overlap_h2(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 2
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 9.99999999881495E-1_wp, 6.61346655776026E-1_wp, 6.61346655776026E-1_wp,&
      & 9.99999999881495E-1_wp],shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "H2")
   call test_overlap_mol(error, mol, overlap)

end subroutine test_overlap_h2

subroutine test_overlap_lih(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 5
   real(wp), parameter :: overlap(nao, nao) = reshape([&
      & 1.00000000000600E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 3.99089038384911E-1_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 4.65790780903622E-1_wp,&
      & 0.00000000000000E+0_wp, 0.00000000000000E+0_wp, 0.00000000000000E+0_wp,&
      & 9.99999999925689E-1_wp, 0.00000000000000E+0_wp, 3.99089038384911E-1_wp,&
      & 0.00000000000000E+0_wp, 4.65790780903622E-1_wp, 0.00000000000000E+0_wp,&
      & 9.99999999881495E-1_wp],shape(overlap))


   type(structure_type) :: mol

   call get_structure(mol, "MB16-43", "LiH")
   call test_overlap_mol(error, mol, overlap)

end subroutine test_overlap_lih

subroutine test_overlap_diat_cecl3(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: nao = 43
   real(wp), parameter :: overlap(nao, nao) = reshape([&
   &  1.00000000017687E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,&
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  8.69412276329442E-02_wp,  1.09663227357883E-01_wp, &
   &  6.85934472373425E-02_wp,  7.46686270912962E-02_wp,  5.39873838622696E-02_wp, &
   &  4.95948688317373E-02_wp, -1.55903398924374E-02_wp,  3.37686648082453E-02_wp, &
   & -2.12649579080578E-02_wp,  8.67139986554086E-02_wp, -1.13584328085048E-01_wp, &
   & -5.71130711953149E-02_wp,  7.77825760830674E-02_wp, -5.83831463741125E-02_wp, &
   &  4.28687369766487E-02_wp, -2.37076059225061E-02_wp, -2.93565217287235E-02_wp, &
   & -2.26374693772444E-02_wp,  8.73543363380361E-02_wp,  3.61731541134505E-02_wp, &
   & -5.71022949696438E-02_wp, -1.33816329028274E-01_wp, -3.17825201816373E-02_wp, &
   & -1.35622823871297E-02_wp, -2.40602114448212E-02_wp,  5.01713186690479E-02_wp, &
   &  5.44912525773983E-02_wp,  0.00000000000000E+00_wp,  1.00000000002709E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  2.77555756156289E-17_wp,  0.00000000000000E+00_wp, &
   &  5.55111512312578E-17_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, -3.40201809683547E-01_wp, &
   & -4.14280857819152E-02_wp, -1.19418786053541E-01_wp, -1.29995461121435E-01_wp, &
   & -4.72851062125490E-03_wp, -4.34379010897963E-03_wp, -1.71048248105265E-02_wp, &
   & -2.36191595818926E-02_wp, -3.36400249951174E-02_wp,  3.52869972519298E-01_wp, &
   & -5.63511977455039E-02_wp, -1.03417073813248E-01_wp,  1.40844227841612E-01_wp, &
   & -8.05377110703101E-03_wp,  5.91360720857850E-03_wp,  1.17908981123966E-02_wp, &
   & -2.13786714107248E-02_wp,  3.38405303098483E-02_wp, -1.11927689146485E-01_wp, &
   &  1.29184921393787E-01_wp,  3.25455034102347E-02_wp,  7.62687348214290E-02_wp, &
   & -5.17690901910778E-02_wp, -2.20909800759548E-02_wp, -3.71443619535530E-03_wp, &
   & -1.14661853993300E-02_wp, -2.84111485232627E-02_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  1.00000000002709E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, -5.55111512312578E-17_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   & -2.12793435363904E-01_wp, -1.19418786053541E-01_wp,  7.47962781286959E-02_wp, &
   & -8.13110923174573E-02_wp, -2.36191595818926E-02_wp,  2.68161297325115E-02_wp, &
   &  4.18598977604196E-02_wp,  1.82588424513004E-02_wp,  9.30329270286544E-03_wp, &
   &  1.77431941562336E-01_wp, -1.03417073813248E-01_wp,  9.73200705922398E-02_wp, &
   &  7.08200378325435E-02_wp, -2.13786714107248E-02_wp, -3.46284778301005E-02_wp, &
   & -3.79011924083843E-02_wp,  2.37135902185714E-02_wp, -8.28936173267027E-03_wp, &
   &  1.76687050868385E-01_wp,  3.25455034102347E-02_wp,  9.84260921377083E-02_wp, &
   & -1.20396462500221E-01_wp, -1.14661853993300E-02_wp,  1.10648156497081E-02_wp, &
   & -3.77677054156251E-02_wp, -4.09323723050175E-02_wp,  1.96588187822549E-02_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  1.00000000002709E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  5.55111512312578E-17_wp,  0.00000000000000E+00_wp, &
   & -2.77555756156289E-17_wp, -2.31640110135961E-01_wp, -1.29995461121435E-01_wp, &
   & -8.13110923174573E-02_wp,  6.09790994241015E-02_wp,  2.28025303637907E-02_wp, &
   & -2.36191595818926E-02_wp, -1.16465091900958E-02_wp,  1.42627952967513E-02_wp, &
   &  4.31596987206192E-02_wp, -2.41645444821936E-01_wp,  1.40844227841612E-01_wp, &
   &  7.08200378325435E-02_wp,  5.28706271385305E-02_wp, -2.12103821177911E-02_wp, &
   & -2.13786714107248E-02_wp, -8.07440995582121E-03_wp, -1.06651162567620E-02_wp, &
   &  4.57526496102248E-02_wp,  4.14057132845683E-01_wp,  7.62687348214290E-02_wp, &
   & -1.20396462500221E-01_wp, -1.32341154522262E-01_wp, -1.09127406617264E-02_wp, &
   & -1.14661853993300E-02_wp,  1.37409144503486E-02_wp,  1.72266574885550E-02_wp, &
   & -1.29632440844008E-02_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999778610E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  8.20155010090769E-02_wp, &
   &  9.28979457969490E-02_wp,  8.00569085802895E-02_wp,  3.56083920092022E-02_wp, &
   &  3.58620481214061E-02_wp,  6.49720464580809E-02_wp,  9.89663399650733E-03_wp, &
   &  2.86445384905872E-02_wp, -3.63460078236818E-02_wp, -8.86514552275003E-02_wp, &
   &  1.07106300572797E-01_wp,  7.22242203094390E-02_wp, -4.50177541095921E-02_wp, &
   &  4.63175554736653E-02_wp, -6.21718876923976E-02_wp,  4.41008058157470E-03_wp, &
   &  2.92431802609242E-02_wp,  4.19093396272880E-02_wp, -4.83242438420612E-02_wp, &
   &  3.82018892333236E-02_wp,  3.90978820690418E-02_wp,  7.46019922122533E-02_wp, &
   & -3.05116894672468E-02_wp, -1.58559058503109E-02_wp,  2.58807011549745E-03_wp, &
   & -4.23724796435138E-02_wp, -5.46710758908302E-02_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  9.99999999778610E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  7.53425656833336E-02_wp,  8.53395943076514E-02_wp,  2.20043365124130E-02_wp, &
   &  8.00569085802895E-02_wp,  6.49720464580809E-02_wp,  2.48213604227393E-02_wp, &
   & -4.12640911091508E-02_wp,  2.13736344720395E-02_wp, -4.31605738168390E-03_wp, &
   &  6.50937154430233E-02_wp, -7.86444738413934E-02_wp,  3.13009539865031E-04_wp, &
   &  7.22242203094390E-02_wp, -6.21718876923976E-02_wp,  7.29603146759239E-03_wp, &
   & -4.67218548895194E-02_wp, -5.71517669530143E-03_wp, -5.66727024069538E-03_wp, &
   & -2.06209903237694E-02_wp,  1.63015647136606E-02_wp, -3.38005258954584E-04_wp, &
   &  3.90978820690418E-02_wp, -1.58559058503109E-02_wp, -1.20233934496212E-04_wp, &
   &  1.48991414842628E-02_wp, -2.81564588294365E-03_wp, -3.12937259411050E-02_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999778609E-01_wp,  0.00000000000000E+00_wp,  4.80740671595891E-17_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp, -2.36842285309118E-02_wp, -7.20467680720084E-03_wp, &
   & -6.03429612836281E-02_wp, -4.90559450776114E-03_wp,  9.89663399650732E-03_wp, &
   & -4.12640911091508E-02_wp, -2.76120606165884E-02_wp, -2.80963191173944E-02_wp, &
   & -3.89816083520694E-03_wp, -3.59986382289629E-02_wp,  2.75278884074025E-02_wp, &
   &  6.03006889788791E-02_wp, -1.88511048183665E-02_wp,  4.41008058157471E-03_wp, &
   & -4.67218548895194E-02_wp, -1.18825152389580E-02_wp,  3.19951378324389E-02_wp, &
   &  1.70996375352673E-03_wp, -3.65827353559851E-02_wp, -8.92221837554665E-03_wp, &
   &  6.06254946164389E-02_wp,  3.30062041606793E-02_wp,  2.58807011549746E-03_wp, &
   &  1.48991414842628E-02_wp, -1.14945896141096E-02_wp, -5.51167977457504E-02_wp, &
   & -4.43725612524172E-03_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999778610E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  5.13000216813875E-02_wp, &
   &  8.00569085802895E-02_wp,  1.49825391521149E-02_wp,  2.22727564867189E-02_wp, &
   &  2.86445384905872E-02_wp,  2.13736344720395E-02_wp, -2.80963191173944E-02_wp, &
   &  7.98374374734485E-03_wp, -4.25294966518297E-02_wp, -4.45761925024143E-02_wp, &
   &  7.22242203094390E-02_wp, -2.14349010640342E-04_wp, -2.26360646654454E-02_wp, &
   &  2.92431802609242E-02_wp, -5.71517669530144E-03_wp,  3.19951378324389E-02_wp, &
   &  2.86402855372428E-03_wp,  3.82652025183376E-02_wp,  7.62837881761689E-02_wp, &
   &  3.90978820690418E-02_wp,  1.25039201181341E-03_wp, -1.17765372388225E-01_wp, &
   & -4.23724796435138E-02_wp, -2.81564588294364E-03_wp, -5.51167977457504E-02_wp, &
   &  9.53463736577520E-03_wp,  5.68398395712986E-02_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  4.80740671595891E-17_wp, &
   &  0.00000000000000E+00_wp,  9.99999999778609E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   & -3.23048840672788E-02_wp,  1.12518921460422E-03_wp, -3.15334189104662E-02_wp, &
   & -6.94186677160610E-02_wp, -3.63460078236819E-02_wp, -4.31605738168388E-03_wp, &
   & -3.89816083520693E-03_wp, -4.25294966518297E-02_wp, -4.20968008654698E-02_wp, &
   & -3.43736973355472E-02_wp,  2.34893769725024E-03_wp,  2.80042045879740E-02_wp, &
   & -7.46695782592206E-02_wp,  4.19093396272880E-02_wp, -5.66727024069537E-03_wp, &
   &  1.70996375352671E-03_wp,  3.82652025183376E-02_wp, -4.55187727956170E-02_wp, &
   &  8.28521011474394E-02_wp,  5.94862619285365E-02_wp, -6.70334685509428E-02_wp, &
   & -9.41199197927978E-02_wp, -5.46710758908302E-02_wp, -3.12937259411051E-02_wp, &
   & -4.43725612524171E-03_wp,  5.68398395712986E-02_wp,  3.13346757662612E-02_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999954551E-01_wp,  0.00000000000000E+00_wp,  1.11022302462516E-16_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp, -6.37703812927541E-03_wp, -2.24936604198012E-02_wp, &
   & -7.22571944109632E-03_wp,  1.99127981578240E-02_wp,  5.81512727763616E-03_wp, &
   & -1.72393727662213E-02_wp, -2.14384122602978E-03_wp,  1.29355636537831E-02_wp, &
   &  2.79559437246025E-02_wp,  7.40406770098628E-03_wp, -2.55601123198589E-02_wp, &
   & -7.00152224711212E-03_wp, -2.04737585704892E-02_wp,  5.71402985740158E-03_wp, &
   &  1.64261267820631E-02_wp,  6.55758393614611E-04_wp,  1.09396897005001E-02_wp, &
   & -3.25759068310625E-02_wp, -2.32970808181820E-02_wp,  1.42279516174685E-02_wp, &
   &  2.18832031916388E-02_wp,  3.48981234262437E-02_wp, -1.02121852619804E-02_wp, &
   & -6.78822012440303E-03_wp, -1.93575430061701E-03_wp, -2.57527981119962E-02_wp, &
   & -2.95098567978740E-02_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999954551E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, -3.40431849326325E-02_wp, &
   & -4.74827302104553E-02_wp, -1.58927481291258E-02_wp, -2.11545204817318E-02_wp, &
   & -2.63941704687171E-02_wp, -2.11135912416699E-02_wp,  1.60699196580794E-02_wp, &
   & -1.07444245139649E-02_wp,  2.19190965349443E-02_wp, -3.06985789568106E-02_wp, &
   &  4.54124683874554E-02_wp,  4.52717213649874E-03_wp, -2.15442376617703E-02_wp, &
   &  2.77251308119991E-02_wp, -9.78968090562189E-03_wp,  2.20763678154631E-02_wp, &
   &  5.41731475723263E-03_wp,  2.10798636529092E-02_wp, -1.66394753076522E-02_wp, &
   &  1.12164550839581E-02_wp,  2.25211882161450E-03_wp,  3.09187955546773E-02_wp, &
   & -1.08132873910535E-02_wp, -8.10584518942799E-04_wp,  1.20817835115180E-02_wp, &
   & -6.24737443153229E-03_wp, -2.73626654703935E-02_wp,  0.00000000000000E+00_wp, &
   &  2.77555756156289E-17_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  1.11022302462516E-16_wp, &
   &  0.00000000000000E+00_wp,  9.99999999954550E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   & -1.28090757033867E-03_wp, -1.23204005913030E-02_wp,  2.49038884936724E-02_wp, &
   & -8.75229503253804E-03_wp, -1.60649283629892E-02_wp,  1.43945681826134E-02_wp, &
   &  2.29720108902036E-02_wp,  1.37607056483274E-02_wp,  4.08008511942591E-03_wp, &
   & -6.45075714555441E-03_wp, -1.77211746261234E-03_wp,  2.88573809092836E-02_wp, &
   & -5.59343295542887E-04_wp, -6.89049708634549E-03_wp, -2.15630999197224E-02_wp, &
   & -1.47401248948007E-02_wp,  1.93252201033651E-02_wp, -3.00247894938941E-03_wp, &
   &  2.12528120885591E-03_wp, -2.57616878551625E-03_wp, -9.21701378652493E-03_wp, &
   & -4.47883417102252E-04_wp,  1.72123004852782E-03_wp,  3.87219849948905E-03_wp, &
   &  4.60884129645277E-03_wp,  1.05925682350414E-02_wp, -1.67051092576148E-03_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999954551E-01_wp,  0.00000000000000E+00_wp,  5.55111512312578E-17_wp, &
   &  0.00000000000000E+00_wp,  2.32909205984756E-02_wp,  2.60524728274164E-02_wp, &
   &  1.76031258469667E-02_wp,  1.77388758768520E-02_wp,  1.34622945224013E-02_wp, &
   &  1.98904123574332E-02_wp, -9.39476671196776E-03_wp,  1.35431887132267E-02_wp, &
   & -5.30263009400632E-03_wp, -2.25774856886379E-02_wp,  2.85240889524626E-02_wp, &
   &  8.00111168452364E-03_wp, -1.95333031990455E-02_wp,  1.78063068515174E-02_wp, &
   & -1.19714716143595E-02_wp,  1.69654313272323E-02_wp,  8.19806673481372E-03_wp, &
   &  6.90421382037361E-03_wp, -2.26881057605045E-02_wp, -9.07844125129397E-03_wp, &
   &  7.72420256658115E-03_wp,  3.35841236773789E-02_wp,  9.71809772688050E-03_wp, &
   &  3.71972593593363E-03_wp,  1.72717431621451E-02_wp, -1.37604829309815E-02_wp, &
   & -1.66617157727234E-02_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  2.77555756156289E-17_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999954550E-01_wp, &
   &  0.00000000000000E+00_wp, -1.66533453693773E-16_wp, -8.72157531858014E-04_wp, &
   & -8.75229503253805E-03_wp,  1.69568159524313E-02_wp, -5.42556064348706E-03_wp, &
   & -9.29097453288089E-03_wp,  1.37607056483274E-02_wp,  1.56414192434809E-02_wp, &
   &  3.55422714503404E-03_wp,  6.96070362833565E-03_wp,  4.41748009542106E-03_wp, &
   & -5.59343295542888E-04_wp, -1.97615416138547E-02_wp, -2.20587677537982E-03_wp, &
   &  4.95776309493031E-03_wp,  1.93252201033651E-02_wp,  1.00940411888978E-02_wp, &
   & -6.57678604430529E-03_wp,  1.43932135734599E-03_wp, -7.86211035482032E-03_wp, &
   & -4.47883417102248E-04_wp,  3.40967488112168E-02_wp, -1.04037311751583E-03_wp, &
   & -4.15510671194854E-03_wp,  1.05925682350414E-02_wp, -1.70496114723901E-02_wp, &
   & -3.24497991168268E-02_wp,  7.47010245689619E-03_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  9.99999999954550E-01_wp,  0.00000000000000E+00_wp, &
   &  1.34091864220632E-02_wp,  3.45519482653732E-03_wp,  6.25995549018286E-03_wp, &
   &  3.07262139246520E-02_wp,  2.19190965349443E-02_wp,  3.36169334446591E-03_wp, &
   & -6.32974114816765E-03_wp,  1.15088725091117E-02_wp,  2.06202786830572E-02_wp, &
   & -1.19030608012565E-02_wp,  4.39408073360094E-03_wp,  1.75536480937154E-03_wp, &
   & -2.76499015081222E-02_wp,  2.10798636529093E-02_wp, -2.01631062617650E-03_wp, &
   &  8.55988639565553E-03_wp,  4.69913239680737E-03_wp, -1.84673717546551E-02_wp, &
   &  2.85284441435993E-02_wp,  2.26838862024821E-02_wp, -3.86126634519725E-03_wp, &
   & -4.16800783537248E-02_wp, -2.73626654703935E-02_wp, -3.96213612481711E-03_wp, &
   & -2.07142641033214E-02_wp,  9.26442849949040E-03_wp,  2.01405830947493E-02_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp, -1.11022302462516E-16_wp,  0.00000000000000E+00_wp, &
   &  9.99999999954551E-01_wp,  2.81785548509627E-02_wp,  2.32672185743682E-02_wp, &
   &  3.19286677421413E-02_wp,  2.38149248312371E-02_wp,  2.22115769413314E-02_wp, &
   &  2.28581711018306E-02_wp,  9.47310434010206E-03_wp,  2.11477642204028E-02_wp, &
   & -1.90384369106144E-03_wp,  3.15424946469630E-02_wp, -2.93108178909011E-02_wp, &
   & -2.98275876071079E-02_wp,  2.89865919149946E-02_wp, -2.83688082507035E-02_wp, &
   &  2.34835834975460E-02_wp,  2.79363404761069E-03_wp, -2.12898936281053E-02_wp, &
   & -2.83300770467368E-03_wp,  2.29901709038050E-02_wp,  3.00639987805416E-02_wp, &
   & -2.15949193473986E-02_wp, -2.25160768370592E-02_wp, -3.08515669011558E-02_wp, &
   & -1.89302474912947E-02_wp,  1.91025315773589E-03_wp,  1.84855092937346E-02_wp, &
   &  5.24876072680659E-03_wp,  8.69412276329442E-02_wp, -3.40201809683547E-01_wp, &
   & -2.12793435363904E-01_wp, -2.31640110135961E-01_wp,  8.20155010090769E-02_wp, &
   &  7.53425656833336E-02_wp, -2.36842285309118E-02_wp,  5.13000216813875E-02_wp, &
   & -3.23048840672788E-02_wp, -6.37703812927541E-03_wp, -3.40431849326325E-02_wp, &
   & -1.28090757033867E-03_wp,  2.32909205984756E-02_wp, -8.72157531858011E-04_wp, &
   &  1.34091864220632E-02_wp,  2.81785548509627E-02_wp,  9.99999999869332E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.52137814488968E-05_wp, &
   & -3.36928188670201E-04_wp, -1.89691378004237E-04_wp,  4.99549878650670E-06_wp, &
   & -3.01084193947362E-06_wp,  1.14329075205538E-04_wp, -2.14716191083883E-05_wp, &
   & -1.69511123039559E-06_wp, -1.01512838117467E-04_wp,  1.12758480338272E-04_wp, &
   & -1.31040324396565E-04_wp, -2.23144270067549E-04_wp, -3.69794278272637E-04_wp, &
   &  7.55446878381371E-05_wp,  4.55857897636091E-05_wp, -2.44516968961120E-05_wp, &
   &  1.28642570924251E-04_wp,  9.32081069771156E-05_wp,  1.09663227357883E-01_wp, &
   & -4.14280857819152E-02_wp, -1.19418786053541E-01_wp, -1.29995461121435E-01_wp, &
   &  9.28979457969490E-02_wp,  8.53395943076514E-02_wp, -7.20467680720085E-03_wp, &
   &  8.00569085802895E-02_wp,  1.12518921460423E-03_wp, -2.24936604198012E-02_wp, &
   & -4.74827302104553E-02_wp, -1.23204005913030E-02_wp,  2.60524728274164E-02_wp, &
   & -8.75229503253804E-03_wp,  3.45519482653731E-03_wp,  2.32672185743682E-02_wp, &
   &  0.00000000000000E+00_wp,  9.99999999998060E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  3.36928188670201E-04_wp, -9.41758669062715E-04_wp, -5.84225011720663E-04_wp, &
   &  1.53854928347468E-05_wp, -1.07377169272165E-05_wp,  4.07737526840130E-04_wp, &
   & -4.66396749143034E-05_wp, -6.57520964570179E-06_wp, -3.30286014739083E-04_wp, &
   &  1.31040324396565E-04_wp, -2.04150976546664E-05_wp, -2.26089711420990E-04_wp, &
   & -3.74675458323373E-04_wp,  2.52051845112027E-05_wp,  1.52095173725851E-05_wp, &
   & -1.65836799186663E-05_wp,  1.63404821497647E-04_wp,  1.43467330644811E-04_wp, &
   &  6.85934472373425E-02_wp, -1.19418786053541E-01_wp,  7.47962781286959E-02_wp, &
   & -8.13110923174573E-02_wp,  8.00569085802895E-02_wp,  2.20043365124130E-02_wp, &
   & -6.03429612836281E-02_wp,  1.49825391521149E-02_wp, -3.15334189104662E-02_wp, &
   & -7.22571944109633E-03_wp, -1.58927481291258E-02_wp,  2.49038884936724E-02_wp, &
   &  1.76031258469667E-02_wp,  1.69568159524313E-02_wp,  6.25995549018286E-03_wp, &
   &  3.19286677421413E-02_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999998060E-01_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  1.89691378004237E-04_wp, -5.84225011720663E-04_wp, &
   & -2.32983327085689E-04_wp,  8.66206935257109E-06_wp, -6.57520964570179E-06_wp, &
   &  1.86202195665234E-04_wp, -8.81555136696621E-05_wp, -2.76074508981215E-06_wp, &
   & -2.21688220693919E-04_wp,  2.23144270067549E-04_wp, -2.26089711420990E-04_wp, &
   & -2.72645831427766E-04_wp, -6.38022547981310E-04_wp,  1.63404821497647E-04_wp, &
   &  7.35308666900574E-05_wp, -1.02189065578875E-04_wp,  2.07503254464807E-04_wp, &
   &  2.01611185625156E-04_wp,  7.46686270912962E-02_wp, -1.29995461121435E-01_wp, &
   & -8.13110923174573E-02_wp,  6.09790994241015E-02_wp,  3.56083920092022E-02_wp, &
   &  8.00569085802895E-02_wp, -4.90559450776115E-03_wp,  2.22727564867189E-02_wp, &
   & -6.94186677160610E-02_wp,  1.99127981578240E-02_wp, -2.11545204817318E-02_wp, &
   & -8.75229503253804E-03_wp,  1.77388758768520E-02_wp, -5.42556064348706E-03_wp, &
   &  3.07262139246520E-02_wp,  2.38149248312371E-02_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999998060E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, -4.99549878650671E-06_wp, &
   &  1.53854928347468E-05_wp,  8.66206935257109E-06_wp,  9.57086406720181E-05_wp, &
   & -6.33015324737798E-05_wp, -6.57520964570179E-06_wp,  6.91507707790898E-07_wp, &
   & -3.56389145477078E-05_wp,  6.77924535368498E-06_wp,  3.69794278272637E-04_wp, &
   & -3.74675458323373E-04_wp, -6.38022547981310E-04_wp, -9.44974883405076E-04_wp, &
   &  2.45721973530458E-04_wp,  1.63404821497647E-04_wp, -4.67989527259473E-05_wp, &
   &  4.18431888622898E-04_wp,  2.63356233815959E-04_wp,  5.39873838622696E-02_wp, &
   & -4.72851062125489E-03_wp, -2.36191595818926E-02_wp,  2.28025303637907E-02_wp, &
   &  3.58620481214061E-02_wp,  6.49720464580809E-02_wp,  9.89663399650732E-03_wp, &
   &  2.86445384905872E-02_wp, -3.63460078236818E-02_wp,  5.81512727763616E-03_wp, &
   & -2.63941704687171E-02_wp, -1.60649283629892E-02_wp,  1.34622945224013E-02_wp, &
   & -9.29097453288089E-03_wp,  2.19190965349443E-02_wp,  2.22115769413314E-02_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  9.99999999830205E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   & -3.01084193947362E-06_wp,  1.07377169272165E-05_wp,  6.57520964570179E-06_wp, &
   &  6.33015324737798E-05_wp, -3.73996719315490E-05_wp, -4.64599908927517E-06_wp, &
   &  2.45641225072744E-07_wp, -2.26199417185387E-05_wp,  4.42393841717257E-06_wp, &
   &  7.55446878381371E-05_wp, -2.52051845112026E-05_wp, -1.63404821497647E-04_wp, &
   & -2.45721973530458E-04_wp,  2.47089644023801E-05_wp,  1.61116888311556E-05_wp, &
   & -4.94256032779233E-06_wp,  1.09891385286233E-04_wp,  8.63251486754047E-05_wp, &
   &  4.95948688317373E-02_wp, -4.34379010897964E-03_wp,  2.68161297325115E-02_wp, &
   & -2.36191595818926E-02_wp,  6.49720464580809E-02_wp,  2.48213604227393E-02_wp, &
   & -4.12640911091509E-02_wp,  2.13736344720395E-02_wp, -4.31605738168389E-03_wp, &
   & -1.72393727662213E-02_wp, -2.11135912416699E-02_wp,  1.43945681826134E-02_wp, &
   &  1.98904123574332E-02_wp,  1.37607056483274E-02_wp,  3.36169334446591E-03_wp, &
   &  2.28581711018306E-02_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999830205E-01_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  1.14329075205538E-04_wp, -4.07737526840130E-04_wp, &
   & -1.86202195665234E-04_wp,  6.57520964570179E-06_wp, -4.64599908927517E-06_wp, &
   &  1.38897990982117E-04_wp, -4.86344418783074E-05_wp, -2.20750442352880E-06_wp, &
   & -1.45294009909431E-04_wp,  4.55857897636091E-05_wp, -1.52095173725851E-05_wp, &
   & -7.35308666900573E-05_wp, -1.63404821497647E-04_wp,  1.61116888311556E-05_wp, &
   &  7.73094926501920E-06_wp, -1.90066643814781E-05_wp,  5.65625287713878E-05_wp, &
   &  6.13425902955612E-05_wp, -1.55903398924374E-02_wp, -1.71048248105265E-02_wp, &
   &  4.18598977604196E-02_wp, -1.16465091900958E-02_wp,  9.89663399650731E-03_wp, &
   & -4.12640911091508E-02_wp, -2.76120606165884E-02_wp, -2.80963191173944E-02_wp, &
   & -3.89816083520695E-03_wp, -2.14384122602977E-03_wp,  1.60699196580794E-02_wp, &
   &  2.29720108902036E-02_wp, -9.39476671196776E-03_wp,  1.56414192434809E-02_wp, &
   & -6.32974114816765E-03_wp,  9.47310434010206E-03_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999830206E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, -2.14716191083883E-05_wp, &
   &  4.66396749143034E-05_wp,  8.81555136696621E-05_wp, -6.91507707790898E-07_wp, &
   &  2.45641225072744E-07_wp, -4.86344418783074E-05_wp, -2.10150359335817E-05_wp, &
   &  7.21083315540951E-07_wp,  8.28198172373845E-06_wp, -2.44516968961120E-05_wp, &
   &  1.65836799186663E-05_wp,  1.02189065578875E-04_wp,  4.67989527259473E-05_wp, &
   & -4.94256032779232E-06_wp, -1.90066643814781E-05_wp, -2.64669742429983E-05_wp, &
   & -5.36365868268805E-05_wp, -6.09820101131120E-06_wp,  3.37686648082453E-02_wp, &
   & -2.36191595818926E-02_wp,  1.82588424513004E-02_wp,  1.42627952967512E-02_wp, &
   &  2.86445384905872E-02_wp,  2.13736344720395E-02_wp, -2.80963191173944E-02_wp, &
   &  7.98374374734485E-03_wp, -4.25294966518297E-02_wp,  1.29355636537831E-02_wp, &
   & -1.07444245139649E-02_wp,  1.37607056483274E-02_wp,  1.35431887132267E-02_wp, &
   &  3.55422714503404E-03_wp,  1.15088725091117E-02_wp,  2.11477642204028E-02_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  9.99999999830205E-01_wp,  0.00000000000000E+00_wp, &
   & -1.69511123039559E-06_wp,  6.57520964570179E-06_wp,  2.76074508981216E-06_wp, &
   &  3.56389145477078E-05_wp, -2.26199417185387E-05_wp, -2.20750442352880E-06_wp, &
   &  7.21083315540952E-07_wp, -9.95740806845498E-06_wp,  2.82716000311223E-06_wp, &
   &  1.28642570924251E-04_wp, -1.63404821497647E-04_wp, -2.07503254464807E-04_wp, &
   & -4.18431888622897E-04_wp,  1.09891385286233E-04_wp,  5.65625287713878E-05_wp, &
   & -5.36365868268805E-05_wp,  1.47306249222992E-04_wp,  1.20892448538428E-04_wp, &
   & -2.12649579080578E-02_wp, -3.36400249951174E-02_wp,  9.30329270286542E-03_wp, &
   &  4.31596987206192E-02_wp, -3.63460078236819E-02_wp, -4.31605738168392E-03_wp, &
   & -3.89816083520695E-03_wp, -4.25294966518297E-02_wp, -4.20968008654698E-02_wp, &
   &  2.79559437246025E-02_wp,  2.19190965349444E-02_wp,  4.08008511942590E-03_wp, &
   & -5.30263009400631E-03_wp,  6.96070362833566E-03_wp,  2.06202786830572E-02_wp, &
   & -1.90384369106144E-03_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999830206E-01_wp, -1.01512838117467E-04_wp,  3.30286014739083E-04_wp, &
   &  2.21688220693919E-04_wp, -6.77924535368497E-06_wp,  4.42393841717257E-06_wp, &
   & -1.45294009909431E-04_wp,  8.28198172373844E-06_wp,  2.82716000311223E-06_wp, &
   &  1.11625581646469E-04_wp,  9.32081069771156E-05_wp, -1.43467330644811E-04_wp, &
   & -2.01611185625156E-04_wp, -2.63356233815959E-04_wp,  8.63251486754046E-05_wp, &
   &  6.13425902955612E-05_wp, -6.09820101131119E-06_wp,  1.20892448538428E-04_wp, &
   &  6.12520710210726E-05_wp,  8.67139986554086E-02_wp,  3.52869972519298E-01_wp, &
   &  1.77431941562336E-01_wp, -2.41645444821936E-01_wp, -8.86514552275003E-02_wp, &
   &  6.50937154430233E-02_wp, -3.59986382289629E-02_wp, -4.45761925024143E-02_wp, &
   & -3.43736973355472E-02_wp,  7.40406770098628E-03_wp, -3.06985789568106E-02_wp, &
   & -6.45075714555441E-03_wp, -2.25774856886379E-02_wp,  4.41748009542107E-03_wp, &
   & -1.19030608012565E-02_wp,  3.15424946469630E-02_wp,  9.52137814488968E-05_wp, &
   &  3.36928188670201E-04_wp,  1.89691378004237E-04_wp, -4.99549878650670E-06_wp, &
   & -3.01084193947362E-06_wp,  1.14329075205538E-04_wp, -2.14716191083883E-05_wp, &
   & -1.69511123039559E-06_wp, -1.01512838117467E-04_wp,  9.99999999869332E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  8.24984479851205E-05_wp, &
   &  1.96296194806693E-04_wp,  5.54539527348801E-07_wp, -2.76572801639242E-04_wp, &
   & -1.09131970490034E-04_wp,  2.18813964986787E-07_wp, -6.67465544040966E-05_wp, &
   & -3.08299360534131E-07_wp,  3.81531587038920E-05_wp, -1.13584328085048E-01_wp, &
   & -5.63511977455039E-02_wp, -1.03417073813248E-01_wp,  1.40844227841612E-01_wp, &
   &  1.07106300572798E-01_wp, -7.86444738413934E-02_wp,  2.75278884074025E-02_wp, &
   &  7.22242203094389E-02_wp,  2.34893769725024E-03_wp, -2.55601123198589E-02_wp, &
   &  4.54124683874554E-02_wp, -1.77211746261233E-03_wp,  2.85240889524626E-02_wp, &
   & -5.59343295542886E-04_wp,  4.39408073360095E-03_wp, -2.93108178909011E-02_wp, &
   & -3.36928188670201E-04_wp, -9.41758669062715E-04_wp, -5.84225011720663E-04_wp, &
   &  1.53854928347468E-05_wp,  1.07377169272165E-05_wp, -4.07737526840130E-04_wp, &
   &  4.66396749143034E-05_wp,  6.57520964570179E-06_wp,  3.30286014739083E-04_wp, &
   &  0.00000000000000E+00_wp,  9.99999999998060E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   & -1.96296194806693E-04_wp, -3.22041577165287E-04_wp, -1.14683436360897E-06_wp, &
   &  5.71975805721029E-04_wp,  2.33926970269965E-04_wp, -4.69032930059591E-07_wp, &
   &  1.53453012266193E-04_wp,  8.06130068702386E-07_wp, -1.36261835578487E-04_wp, &
   & -5.71130711953149E-02_wp, -1.03417073813248E-01_wp,  9.73200705922398E-02_wp, &
   &  7.08200378325435E-02_wp,  7.22242203094389E-02_wp,  3.13009539865045E-04_wp, &
   &  6.03006889788791E-02_wp, -2.14349010640352E-04_wp,  2.80042045879740E-02_wp, &
   & -7.00152224711211E-03_wp,  4.52717213649874E-03_wp,  2.88573809092836E-02_wp, &
   &  8.00111168452365E-03_wp, -1.97615416138547E-02_wp,  1.75536480937154E-03_wp, &
   & -2.98275876071079E-02_wp, -1.89691378004237E-04_wp, -5.84225011720663E-04_wp, &
   & -2.32983327085689E-04_wp,  8.66206935257109E-06_wp,  6.57520964570179E-06_wp, &
   & -1.86202195665234E-04_wp,  8.81555136696621E-05_wp,  2.76074508981215E-06_wp, &
   &  2.21688220693919E-04_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999998060E-01_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp, -5.54539527348801E-07_wp, -1.14683436360897E-06_wp, &
   &  8.39122168161397E-05_wp,  1.61583974295499E-06_wp,  8.06130068702386E-07_wp, &
   &  3.64987134453440E-05_wp,  6.12105527417005E-07_wp, -5.14250999299680E-05_wp, &
   & -2.81827665248565E-07_wp,  7.77825760830674E-02_wp,  1.40844227841612E-01_wp, &
   &  7.08200378325435E-02_wp,  5.28706271385305E-02_wp, -4.50177541095921E-02_wp, &
   &  7.22242203094389E-02_wp, -1.88511048183665E-02_wp, -2.26360646654454E-02_wp, &
   & -7.46695782592206E-02_wp, -2.04737585704892E-02_wp, -2.15442376617703E-02_wp, &
   & -5.59343295542893E-04_wp, -1.95333031990455E-02_wp, -2.20587677537982E-03_wp, &
   & -2.76499015081222E-02_wp,  2.89865919149946E-02_wp,  4.99549878650671E-06_wp, &
   &  1.53854928347468E-05_wp,  8.66206935257109E-06_wp,  9.57086406720181E-05_wp, &
   &  6.33015324737798E-05_wp,  6.57520964570179E-06_wp, -6.91507707790898E-07_wp, &
   &  3.56389145477078E-05_wp, -6.77924535368498E-06_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999998060E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  2.76572801639242E-04_wp, &
   &  5.71975805721029E-04_wp,  1.61583974295499E-06_wp, -7.21973578621338E-04_wp, &
   & -3.65551535952679E-04_wp,  8.06130068702386E-07_wp, -2.16208620672635E-04_wp, &
   & -1.03268826055672E-06_wp,  8.91322458394565E-05_wp, -5.83831463741125E-02_wp, &
   & -8.05377110703100E-03_wp, -2.13786714107248E-02_wp, -2.12103821177911E-02_wp, &
   &  4.63175554736653E-02_wp, -6.21718876923976E-02_wp,  4.41008058157470E-03_wp, &
   &  2.92431802609242E-02_wp,  4.19093396272880E-02_wp,  5.71402985740159E-03_wp, &
   &  2.77251308119991E-02_wp, -6.89049708634549E-03_wp,  1.78063068515174E-02_wp, &
   &  4.95776309493032E-03_wp,  2.10798636529093E-02_wp, -2.83688082507035E-02_wp, &
   & -3.01084193947362E-06_wp, -1.07377169272165E-05_wp, -6.57520964570179E-06_wp, &
   & -6.33015324737798E-05_wp, -3.73996719315490E-05_wp, -4.64599908927517E-06_wp, &
   &  2.45641225072744E-07_wp, -2.26199417185387E-05_wp,  4.42393841717257E-06_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  9.99999999830205E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   & -1.09131970490034E-04_wp, -2.33926970269965E-04_wp, -8.06130068702387E-07_wp, &
   &  3.65551535952679E-04_wp,  1.55708719431824E-04_wp, -3.37779234265429E-07_wp, &
   &  9.68084081921723E-05_wp,  5.17901618054598E-07_wp, -6.93663114292701E-05_wp, &
   &  4.28687369766487E-02_wp,  5.91360720857849E-03_wp, -3.46284778301005E-02_wp, &
   & -2.13786714107248E-02_wp, -6.21718876923976E-02_wp,  7.29603146759238E-03_wp, &
   & -4.67218548895194E-02_wp, -5.71517669530143E-03_wp, -5.66727024069538E-03_wp, &
   &  1.64261267820631E-02_wp, -9.78968090562189E-03_wp, -2.15630999197224E-02_wp, &
   & -1.19714716143595E-02_wp,  1.93252201033651E-02_wp, -2.01631062617651E-03_wp, &
   &  2.34835834975460E-02_wp,  1.14329075205538E-04_wp,  4.07737526840130E-04_wp, &
   &  1.86202195665234E-04_wp, -6.57520964570179E-06_wp, -4.64599908927517E-06_wp, &
   &  1.38897990982117E-04_wp, -4.86344418783074E-05_wp, -2.20750442352880E-06_wp, &
   & -1.45294009909431E-04_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999830205E-01_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  2.18813964986787E-07_wp,  4.69032930059591E-07_wp, &
   & -3.64987134453440E-05_wp, -8.06130068702386E-07_wp, -3.37779234265429E-07_wp, &
   & -1.27556892913121E-05_wp, -2.67921067336703E-07_wp,  2.12542538776661E-05_wp, &
   &  1.81700110721510E-07_wp, -2.37076059225061E-02_wp,  1.17908981123966E-02_wp, &
   & -3.79011924083843E-02_wp, -8.07440995582129E-03_wp,  4.41008058157471E-03_wp, &
   & -4.67218548895194E-02_wp, -1.18825152389580E-02_wp,  3.19951378324389E-02_wp, &
   &  1.70996375352672E-03_wp,  6.55758393614615E-04_wp,  2.20763678154631E-02_wp, &
   & -1.47401248948007E-02_wp,  1.69654313272323E-02_wp,  1.00940411888978E-02_wp, &
   &  8.55988639565554E-03_wp,  2.79363404761070E-03_wp, -2.14716191083883E-05_wp, &
   & -4.66396749143034E-05_wp, -8.81555136696621E-05_wp,  6.91507707790898E-07_wp, &
   &  2.45641225072744E-07_wp, -4.86344418783074E-05_wp, -2.10150359335817E-05_wp, &
   &  7.21083315540951E-07_wp,  8.28198172373845E-06_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999830206E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, -6.67465544040966E-05_wp, &
   & -1.53453012266193E-04_wp, -6.12105527417006E-07_wp,  2.16208620672635E-04_wp, &
   &  9.68084081921723E-05_wp, -2.67921067336703E-07_wp,  6.15385941191151E-05_wp, &
   &  3.77489132096825E-07_wp, -3.38447711064139E-05_wp, -2.93565217287235E-02_wp, &
   & -2.13786714107248E-02_wp,  2.37135902185714E-02_wp, -1.06651162567620E-02_wp, &
   &  2.92431802609242E-02_wp, -5.71517669530143E-03_wp,  3.19951378324389E-02_wp, &
   &  2.86402855372428E-03_wp,  3.82652025183376E-02_wp,  1.09396897005001E-02_wp, &
   &  5.41731475723262E-03_wp,  1.93252201033651E-02_wp,  8.19806673481372E-03_wp, &
   & -6.57678604430529E-03_wp,  4.69913239680737E-03_wp, -2.12898936281053E-02_wp, &
   & -1.69511123039559E-06_wp, -6.57520964570179E-06_wp, -2.76074508981216E-06_wp, &
   & -3.56389145477078E-05_wp, -2.26199417185387E-05_wp, -2.20750442352880E-06_wp, &
   &  7.21083315540952E-07_wp, -9.95740806845498E-06_wp,  2.82716000311223E-06_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  9.99999999830205E-01_wp,  0.00000000000000E+00_wp, &
   & -3.08299360534131E-07_wp, -8.06130068702386E-07_wp,  5.14250999299680E-05_wp, &
   &  1.03268826055672E-06_wp,  5.17901618054598E-07_wp,  2.12542538776661E-05_wp, &
   &  3.77489132096825E-07_wp, -2.76169058108372E-05_wp, -1.35914068297143E-07_wp, &
   & -2.26374693772444E-02_wp,  3.38405303098482E-02_wp, -8.28936173267023E-03_wp, &
   &  4.57526496102248E-02_wp,  4.19093396272880E-02_wp, -5.66727024069539E-03_wp, &
   &  1.70996375352672E-03_wp,  3.82652025183376E-02_wp, -4.55187727956170E-02_wp, &
   & -3.25759068310625E-02_wp,  2.10798636529092E-02_wp, -3.00247894938940E-03_wp, &
   &  6.90421382037361E-03_wp,  1.43932135734599E-03_wp, -1.84673717546551E-02_wp, &
   & -2.83300770467367E-03_wp, -1.01512838117467E-04_wp, -3.30286014739083E-04_wp, &
   & -2.21688220693919E-04_wp,  6.77924535368497E-06_wp,  4.42393841717257E-06_wp, &
   & -1.45294009909431E-04_wp,  8.28198172373844E-06_wp,  2.82716000311223E-06_wp, &
   &  1.11625581646469E-04_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999830206E-01_wp,  3.81531587038920E-05_wp,  1.36261835578487E-04_wp, &
   &  2.81827665248565E-07_wp, -8.91322458394565E-05_wp, -6.93663114292701E-05_wp, &
   &  1.81700110721510E-07_wp, -3.38447711064139E-05_wp, -1.35914068297143E-07_wp, &
   & -1.84534116858889E-05_wp,  8.73543363380361E-02_wp, -1.11927689146485E-01_wp, &
   &  1.76687050868385E-01_wp,  4.14057132845683E-01_wp, -4.83242438420612E-02_wp, &
   & -2.06209903237694E-02_wp, -3.65827353559851E-02_wp,  7.62837881761689E-02_wp, &
   &  8.28521011474394E-02_wp, -2.32970808181820E-02_wp, -1.66394753076522E-02_wp, &
   &  2.12528120885591E-03_wp, -2.26881057605045E-02_wp, -7.86211035482033E-03_wp, &
   &  2.85284441435993E-02_wp,  2.29901709038050E-02_wp,  1.12758480338272E-04_wp, &
   &  1.31040324396565E-04_wp,  2.23144270067549E-04_wp,  3.69794278272637E-04_wp, &
   &  7.55446878381371E-05_wp,  4.55857897636091E-05_wp, -2.44516968961120E-05_wp, &
   &  1.28642570924251E-04_wp,  9.32081069771156E-05_wp,  8.24984479851205E-05_wp, &
   & -1.96296194806693E-04_wp, -5.54539527348801E-07_wp,  2.76572801639242E-04_wp, &
   & -1.09131970490034E-04_wp,  2.18813964986787E-07_wp, -6.67465544040966E-05_wp, &
   & -3.08299360534131E-07_wp,  3.81531587038920E-05_wp,  9.99999999869332E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  3.61731541134506E-02_wp, &
   &  1.29184921393787E-01_wp,  3.25455034102347E-02_wp,  7.62687348214290E-02_wp, &
   &  3.82018892333236E-02_wp,  1.63015647136606E-02_wp, -8.92221837554666E-03_wp, &
   &  3.90978820690418E-02_wp,  5.94862619285365E-02_wp,  1.42279516174685E-02_wp, &
   &  1.12164550839581E-02_wp, -2.57616878551625E-03_wp, -9.07844125129397E-03_wp, &
   & -4.47883417102248E-04_wp,  2.26838862024821E-02_wp,  3.00639987805416E-02_wp, &
   & -1.31040324396565E-04_wp, -2.04150976546664E-05_wp, -2.26089711420990E-04_wp, &
   & -3.74675458323373E-04_wp, -2.52051845112027E-05_wp, -1.52095173725851E-05_wp, &
   &  1.65836799186663E-05_wp, -1.63404821497647E-04_wp, -1.43467330644811E-04_wp, &
   &  1.96296194806693E-04_wp, -3.22041577165287E-04_wp, -1.14683436360897E-06_wp, &
   &  5.71975805721029E-04_wp, -2.33926970269965E-04_wp,  4.69032930059591E-07_wp, &
   & -1.53453012266193E-04_wp, -8.06130068702386E-07_wp,  1.36261835578487E-04_wp, &
   &  0.00000000000000E+00_wp,  9.99999999998060E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   & -5.71022949696438E-02_wp,  3.25455034102347E-02_wp,  9.84260921377083E-02_wp, &
   & -1.20396462500221E-01_wp,  3.90978820690418E-02_wp, -3.38005258954589E-04_wp, &
   &  6.06254946164389E-02_wp,  1.25039201181343E-03_wp, -6.70334685509428E-02_wp, &
   &  2.18832031916388E-02_wp,  2.25211882161450E-03_wp, -9.21701378652493E-03_wp, &
   &  7.72420256658115E-03_wp,  3.40967488112168E-02_wp, -3.86126634519724E-03_wp, &
   & -2.15949193473986E-02_wp, -2.23144270067549E-04_wp, -2.26089711420990E-04_wp, &
   & -2.72645831427766E-04_wp, -6.38022547981310E-04_wp, -1.63404821497647E-04_wp, &
   & -7.35308666900574E-05_wp,  1.02189065578875E-04_wp, -2.07503254464807E-04_wp, &
   & -2.01611185625156E-04_wp,  5.54539527348801E-07_wp, -1.14683436360897E-06_wp, &
   &  8.39122168161397E-05_wp,  1.61583974295499E-06_wp, -8.06130068702386E-07_wp, &
   & -3.64987134453440E-05_wp, -6.12105527417005E-07_wp,  5.14250999299680E-05_wp, &
   &  2.81827665248565E-07_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999998060E-01_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp, -1.33816329028274E-01_wp,  7.62687348214290E-02_wp, &
   & -1.20396462500221E-01_wp, -1.32341154522262E-01_wp,  7.46019922122534E-02_wp, &
   &  3.90978820690418E-02_wp,  3.30062041606792E-02_wp, -1.17765372388225E-01_wp, &
   & -9.41199197927978E-02_wp,  3.48981234262437E-02_wp,  3.09187955546773E-02_wp, &
   & -4.47883417102249E-04_wp,  3.35841236773789E-02_wp, -1.04037311751581E-03_wp, &
   & -4.16800783537248E-02_wp, -2.25160768370592E-02_wp, -3.69794278272637E-04_wp, &
   & -3.74675458323373E-04_wp, -6.38022547981310E-04_wp, -9.44974883405076E-04_wp, &
   & -2.45721973530458E-04_wp, -1.63404821497647E-04_wp,  4.67989527259473E-05_wp, &
   & -4.18431888622898E-04_wp, -2.63356233815959E-04_wp, -2.76572801639242E-04_wp, &
   &  5.71975805721029E-04_wp,  1.61583974295499E-06_wp, -7.21973578621338E-04_wp, &
   &  3.65551535952679E-04_wp, -8.06130068702386E-07_wp,  2.16208620672635E-04_wp, &
   &  1.03268826055672E-06_wp, -8.91322458394565E-05_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999998060E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, -3.17825201816373E-02_wp, &
   & -5.17690901910778E-02_wp, -1.14661853993300E-02_wp, -1.09127406617264E-02_wp, &
   & -3.05116894672468E-02_wp, -1.58559058503109E-02_wp,  2.58807011549745E-03_wp, &
   & -4.23724796435138E-02_wp, -5.46710758908302E-02_wp, -1.02121852619804E-02_wp, &
   & -1.08132873910535E-02_wp,  1.72123004852782E-03_wp,  9.71809772688051E-03_wp, &
   & -4.15510671194854E-03_wp, -2.73626654703935E-02_wp, -3.08515669011558E-02_wp, &
   &  7.55446878381371E-05_wp,  2.52051845112026E-05_wp,  1.63404821497647E-04_wp, &
   &  2.45721973530458E-04_wp,  2.47089644023801E-05_wp,  1.61116888311556E-05_wp, &
   & -4.94256032779233E-06_wp,  1.09891385286233E-04_wp,  8.63251486754047E-05_wp, &
   & -1.09131970490034E-04_wp,  2.33926970269965E-04_wp,  8.06130068702387E-07_wp, &
   & -3.65551535952679E-04_wp,  1.55708719431824E-04_wp, -3.37779234265429E-07_wp, &
   &  9.68084081921723E-05_wp,  5.17901618054598E-07_wp, -6.93663114292701E-05_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  9.99999999830205E-01_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   & -1.35622823871297E-02_wp, -2.20909800759548E-02_wp,  1.10648156497081E-02_wp, &
   & -1.14661853993300E-02_wp, -1.58559058503109E-02_wp, -1.20233934496220E-04_wp, &
   &  1.48991414842628E-02_wp, -2.81564588294364E-03_wp, -3.12937259411051E-02_wp, &
   & -6.78822012440303E-03_wp, -8.10584518942798E-04_wp,  3.87219849948905E-03_wp, &
   &  3.71972593593363E-03_wp,  1.05925682350414E-02_wp, -3.96213612481712E-03_wp, &
   & -1.89302474912948E-02_wp,  4.55857897636091E-05_wp,  1.52095173725851E-05_wp, &
   &  7.35308666900573E-05_wp,  1.63404821497647E-04_wp,  1.61116888311556E-05_wp, &
   &  7.73094926501920E-06_wp, -1.90066643814781E-05_wp,  5.65625287713878E-05_wp, &
   &  6.13425902955612E-05_wp,  2.18813964986787E-07_wp, -4.69032930059591E-07_wp, &
   &  3.64987134453440E-05_wp,  8.06130068702386E-07_wp, -3.37779234265429E-07_wp, &
   & -1.27556892913121E-05_wp, -2.67921067336703E-07_wp,  2.12542538776661E-05_wp, &
   &  1.81700110721510E-07_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999830205E-01_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp, -2.40602114448212E-02_wp, -3.71443619535532E-03_wp, &
   & -3.77677054156251E-02_wp,  1.37409144503486E-02_wp,  2.58807011549745E-03_wp, &
   &  1.48991414842628E-02_wp, -1.14945896141096E-02_wp, -5.51167977457504E-02_wp, &
   & -4.43725612524170E-03_wp, -1.93575430061701E-03_wp,  1.20817835115180E-02_wp, &
   &  4.60884129645277E-03_wp,  1.72717431621451E-02_wp, -1.70496114723901E-02_wp, &
   & -2.07142641033214E-02_wp,  1.91025315773589E-03_wp, -2.44516968961120E-05_wp, &
   & -1.65836799186663E-05_wp, -1.02189065578875E-04_wp, -4.67989527259473E-05_wp, &
   & -4.94256032779232E-06_wp, -1.90066643814781E-05_wp, -2.64669742429983E-05_wp, &
   & -5.36365868268805E-05_wp, -6.09820101131120E-06_wp, -6.67465544040966E-05_wp, &
   &  1.53453012266193E-04_wp,  6.12105527417006E-07_wp, -2.16208620672635E-04_wp, &
   &  9.68084081921723E-05_wp, -2.67921067336703E-07_wp,  6.15385941191151E-05_wp, &
   &  3.77489132096825E-07_wp, -3.38447711064139E-05_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  9.99999999830206E-01_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  5.01713186690479E-02_wp, &
   & -1.14661853993300E-02_wp, -4.09323723050174E-02_wp,  1.72266574885550E-02_wp, &
   & -4.23724796435138E-02_wp, -2.81564588294364E-03_wp, -5.51167977457504E-02_wp, &
   &  9.53463736577521E-03_wp,  5.68398395712986E-02_wp, -2.57527981119962E-02_wp, &
   & -6.24737443153230E-03_wp,  1.05925682350414E-02_wp, -1.37604829309815E-02_wp, &
   & -3.24497991168268E-02_wp,  9.26442849949040E-03_wp,  1.84855092937346E-02_wp, &
   &  1.28642570924251E-04_wp,  1.63404821497647E-04_wp,  2.07503254464807E-04_wp, &
   &  4.18431888622897E-04_wp,  1.09891385286233E-04_wp,  5.65625287713878E-05_wp, &
   & -5.36365868268805E-05_wp,  1.47306249222992E-04_wp,  1.20892448538428E-04_wp, &
   & -3.08299360534131E-07_wp,  8.06130068702386E-07_wp, -5.14250999299680E-05_wp, &
   & -1.03268826055672E-06_wp,  5.17901618054598E-07_wp,  2.12542538776661E-05_wp, &
   &  3.77489132096825E-07_wp, -2.76169058108372E-05_wp, -1.35914068297143E-07_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  9.99999999830205E-01_wp,  0.00000000000000E+00_wp, &
   &  5.44912525773983E-02_wp, -2.84111485232627E-02_wp,  1.96588187822549E-02_wp, &
   & -1.29632440844007E-02_wp, -5.46710758908302E-02_wp, -3.12937259411051E-02_wp, &
   & -4.43725612524171E-03_wp,  5.68398395712986E-02_wp,  3.13346757662612E-02_wp, &
   & -2.95098567978740E-02_wp, -2.73626654703935E-02_wp, -1.67051092576148E-03_wp, &
   & -1.66617157727234E-02_wp,  7.47010245689617E-03_wp,  2.01405830947493E-02_wp, &
   &  5.24876072680661E-03_wp,  9.32081069771156E-05_wp,  1.43467330644811E-04_wp, &
   &  2.01611185625156E-04_wp,  2.63356233815959E-04_wp,  8.63251486754046E-05_wp, &
   &  6.13425902955612E-05_wp, -6.09820101131119E-06_wp,  1.20892448538428E-04_wp, &
   &  6.12520710210726E-05_wp,  3.81531587038920E-05_wp, -1.36261835578487E-04_wp, &
   & -2.81827665248565E-07_wp,  8.91322458394565E-05_wp, -6.93663114292701E-05_wp, &
   &  1.81700110721510E-07_wp, -3.38447711064139E-05_wp, -1.35914068297143E-07_wp, &
   & -1.84534116858889E-05_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  0.00000000000000E+00_wp,  0.00000000000000E+00_wp,  0.00000000000000E+00_wp, &
   &  9.99999999830206E-01_wp],&
   & shape(overlap))

   type(structure_type) :: mol

   call get_structure(mol, "f-block", "CeCl3")
   call test_overlap_diat_mol(error, mol, overlap)

end subroutine test_overlap_diat_cecl3

subroutine test_overlap_grad_ss(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6
   integer :: stat, i
   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3), r2
   real(wp) :: overlap(1, 1), doverlapi(3, 1, 1), doverlapj(3, 1, 1), sr(1, 1), sl(1, 1)
   real(wp), parameter :: step = 1.0e-6_wp

   call slater_to_gauss(ng, 2, 0, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 1, 0, 1.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))
   r2 = sum(vec**2)

   call overlap_grad_cgto(cgtoi, cgtoj, r2, vec, 100.0_wp, overlap, doverlapj)

   vec(:) = -vec

   call overlap_grad_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, overlap, doverlapi)

   do i = 1, 3
      call check(error, doverlapi(i, 1, 1), -doverlapj(i, 1, 1), thr=thr)
      if (allocated(error)) exit
   end do
   if (allocated(error)) return

   do i = 1, 3
      vec(i) = vec(i) + step
      r2 = sum(vec**2)
      call overlap_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, sr)
      vec(i) = vec(i) - 2*step
      r2 = sum(vec**2)
      call overlap_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, sl)
      vec(i) = vec(i) + step
      doverlapj(i, :, :) = 0.5_wp * (sr - sl) / step
   end do

   do i = 1, 3
      call check(error, doverlapi(i, 1, 1), doverlapj(i, 1, 1), thr=thr)
      if (allocated(error)) exit
   end do
   if (allocated(error)) return

end subroutine test_overlap_grad_ss


subroutine test_overlap_grad_pp(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6
   integer :: stat, i, j, k
   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3), r2
   real(wp) :: overlap(3, 3), doverlapi(3, 3, 3), doverlapj(3, 3, 3), sr(3, 3), sl(3, 3)
   real(wp), parameter :: step = 1.0e-6_wp

   call slater_to_gauss(ng, 3, 1, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 2, 1, 1.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))
   r2 = sum(vec**2)

   call overlap_grad_cgto(cgtoi, cgtoj, r2, vec, 100.0_wp, overlap, doverlapj)

   vec(:) = -vec

   call overlap_grad_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, overlap, doverlapi)

   lp: do i = 1, 3
      do j = 1, 3
         do k = 1, 3
            call check(error, doverlapi(i, j, k), -doverlapj(i, k, j), thr=thr)
            if (allocated(error)) exit
         end do
      end do
   end do lp
   if (allocated(error)) return

   do i = 1, 3
      vec(i) = vec(i) + step
      r2 = sum(vec**2)
      call overlap_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, sr)
      vec(i) = vec(i) - 2*step
      r2 = sum(vec**2)
      call overlap_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, sl)
      vec(i) = vec(i) + step
      doverlapj(i, :, :) = 0.5_wp * (sr - sl) / step
   end do

   num: do i = 1, 3
      do j = 1, 3
         do k = 1, 3
            call check(error, doverlapi(i, j, k), doverlapj(i, j, k), thr=thr)
            if (allocated(error)) exit num
         end do
      end do
   end do num
   if (allocated(error)) return

end subroutine test_overlap_grad_pp


subroutine test_overlap_grad_dd(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6
   integer :: stat, i, j, k
   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3), r2
   real(wp) :: overlap(5, 5), doverlapi(3, 5, 5), doverlapj(3, 5, 5), sl(5, 5), sr(5, 5)
   real(wp), parameter :: step = 1.0e-6_wp

   call slater_to_gauss(ng, 4, 2, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 2, 1.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))
   r2 = sum(vec**2)

   call overlap_grad_cgto(cgtoi, cgtoj, r2, vec, 100.0_wp, overlap, doverlapj)

   vec(:) = -vec

   call overlap_grad_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, overlap, doverlapi)

   lp: do i = 1, 3
      do j = 1, 5
         do k = 1, 5
            call check(error, doverlapi(i, j, k), -doverlapj(i, k, j), thr=thr)
            if (allocated(error)) exit lp
         end do
      end do
   end do lp
   if (allocated(error)) return

   do i = 1, 3
      vec(i) = vec(i) + step
      r2 = sum(vec**2)
      call overlap_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, sr)
      vec(i) = vec(i) - 2*step
      r2 = sum(vec**2)
      call overlap_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, sl)
      vec(i) = vec(i) + step
      doverlapj(i, :, :) = 0.5_wp * (sr - sl) / step
   end do

   num: do i = 1, 3
      do j = 1, 5
         do k = 1, 5
            call check(error, doverlapi(i, j, k), doverlapj(i, j, k), thr=thr)
            if (allocated(error)) exit num
         end do
      end do
   end do num
   if (allocated(error)) return

end subroutine test_overlap_grad_dd


subroutine test_overlap_grad_ff(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6
   integer :: stat, i, j, k
   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3), r2
   real(wp) :: overlap(7, 7), doverlapi(3, 7, 7), doverlapj(3, 7, 7), sl(7, 7), sr(7, 7)
   real(wp), parameter :: step = 1.0e-6_wp

   call slater_to_gauss(ng, 5, 3, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 1.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))
   r2 = sum(vec**2)

   call overlap_grad_cgto(cgtoi, cgtoj, r2, vec, 100.0_wp, overlap, doverlapj)

   vec(:) = -vec

   call overlap_grad_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, overlap, doverlapi)

   lp: do i = 1, 3
      do j = 1, 7
         do k = 1, 7
            call check(error, doverlapi(i, j, k), -doverlapj(i, k, j), thr=thr)
            if (allocated(error)) exit lp
         end do
      end do
   end do lp
   if (allocated(error)) return

   do i = 1, 3
      vec(i) = vec(i) + step
      r2 = sum(vec**2)
      call overlap_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, sr)
      vec(i) = vec(i) - 2*step
      r2 = sum(vec**2)
      call overlap_cgto(cgtoj, cgtoi, r2, vec, 100.0_wp, sl)
      vec(i) = vec(i) + step
      doverlapj(i, :, :) = 0.5_wp * (sr - sl) / step
   end do

   num: do i = 1, 3
      do j = 1, 7
         do k = 1, 7
            call check(error, doverlapi(i, j, k), doverlapj(i, j, k), thr=thr)
            if (allocated(error)) exit num
         end do
      end do
   end do num
   if (allocated(error)) return

end subroutine test_overlap_grad_ff

subroutine test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)
   !> Vector between the two centers
   real(wp), intent(inout) :: vec(3)
   !> Scaling factors for the diatomic frame overlap
   real(wp), intent(in) :: ksig, kpi, kdel
   !> CGTOs for the tested diatomic frame overlap calculation
   type(cgto_type), intent(in) :: cgtoi, cgtoj
   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer :: i, j, k
   real(wp) :: r2
   real(wp) :: overlap(msao(cgtoi%ang), msao(cgtoj%ang)), overlap_diat(msao(cgtoi%ang), msao(cgtoj%ang)), &
   & sl(msao(cgtoj%ang), msao(cgtoi%ang)), sr(msao(cgtoj%ang), msao(cgtoi%ang)), &
   & sl_diat(msao(cgtoj%ang), msao(cgtoi%ang)), sr_diat(msao(cgtoj%ang), msao(cgtoi%ang))

   real(wp) :: doverlapi(3, msao(cgtoi%ang), msao(cgtoj%ang)),  doverlapi_diat(3, msao(cgtoi%ang), msao(cgtoj%ang)), &
   & doverlapj(3, msao(cgtoj%ang), msao(cgtoi%ang)), doverlapj_diat(3, msao(cgtoj%ang), msao(cgtoi%ang)), &
   & doverlaptmp(3, msao(cgtoj%ang), msao(cgtoi%ang)), doverlaptmp_diat(3, msao(cgtoj%ang), msao(cgtoi%ang))

   real(wp), parameter :: step = 1.0e-6_wp

   r2 = sum(vec**2)

   ! Test antisymmetry w.r.t. the exchange of the two centers
   call overlap_grad_cgto_diat(cgtoi, cgtoj, r2, vec, 100.0_wp, &
   & ksig, kpi, kdel, overlap, doverlapi, overlap_diat, doverlapi_diat)

   vec(:) = -vec

   call overlap_grad_cgto_diat(cgtoj, cgtoi, r2, vec, 100.0_wp, &
   & ksig, kpi, kdel, overlap, doverlapj, overlap_diat, doverlapj_diat)

   lp: do i = 1, 3
      do j = 1, msao(cgtoi%ang)
         do k = 1, msao(cgtoj%ang)
            call check(error, doverlapi(i, j, k), -doverlapj(i, k, j), thr=thr2)
            if (allocated(error)) exit lp
            call check(error, doverlapi_diat(i, j, k), -doverlapj_diat(i, k, j), thr=thr2)
            if (allocated(error)) exit lp
         end do
      end do
   end do lp
   if (allocated(error)) return

   ! Test the analytical againts the numerical gradient
   do i = 1, 3
      vec(i) = vec(i) + step
      r2 = sum(vec**2)
      call overlap_cgto_diat(cgtoj, cgtoi, r2, vec,&
      & 100.0_wp, ksig, kpi, kdel, sr, sr_diat)

      vec(i) = vec(i) - 2*step
      r2 = sum(vec**2)
      call overlap_cgto_diat(cgtoj, cgtoi, r2, vec,&
      & 100.0_wp, ksig, kpi, kdel, sl, sl_diat)

      vec(i) = vec(i) + step
      doverlaptmp(i, :, :) = 0.5_wp * (sr - sl) / step
      doverlaptmp_diat(i, :, :) = 0.5_wp * (sr_diat - sl_diat) / step
   end do

   num: do i = 1, 3
      do j = 1, msao(cgtoi%ang)
         do k = 1, msao(cgtoj%ang)
            call check(error, doverlapj(i, k, j), doverlaptmp(i, k, j), thr=thr2)
            if (allocated(error)) then
               write(*,*) "Error", doverlapj(i, k, j), doverlaptmp(i, k, j)
               exit num
            end if
            call check(error, doverlapj_diat(i, k, j), doverlaptmp_diat(i, k, j), thr=thr2)
            if (allocated(error)) then
               write(*,*) "Error", doverlapj_diat(i, k, j), doverlaptmp_diat(i, k, j)
               exit num
            end if
         end do
      end do
   end do num
   if (allocated(error)) return

end subroutine test_overlap_diat_grad_gen

subroutine test_overlap_diat_grad_ss(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 2, 0, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 1, 0, 2.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_ss

subroutine test_overlap_diat_grad_ss_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 2, 0, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 1, 0, 2.0_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = 1e-7_wp
   vec(2) = 1e-7_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_ss_z

subroutine test_overlap_diat_grad_sp(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 4, 0, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 1, 2.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_sp

subroutine test_overlap_diat_grad_sp_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 4, 0, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 1, 2.0_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = 1e-7_wp
   vec(2) = 1e-7_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_sp_z

subroutine test_overlap_diat_grad_pp(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 3, 1, 1.5_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 2, 1, 1.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_pp

subroutine test_overlap_diat_grad_pp_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 3, 1, 1.5_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 2, 1, 1.0_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = 1e-7_wp
   vec(2) = 1e-7_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_pp_z

subroutine test_overlap_diat_grad_sd(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat


   call slater_to_gauss(ng, 4, 0, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 2, 1.8_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_sd

subroutine test_overlap_diat_grad_sd_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 4, 0, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 2, 1.8_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = 1e-7_wp
   vec(2) = 1e-7_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_sd_z

subroutine test_overlap_diat_grad_pd(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 4, 1, 1.5_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 2, 1.5_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_pd

subroutine test_overlap_diat_grad_pd_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 4, 1, 1.5_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 2, 1.5_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = 1e-7_wp
   vec(2) = 1e-7_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_pd_z

subroutine test_overlap_diat_grad_dd(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 4, 2, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 2, 1.3_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.7
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.7_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_dd

subroutine test_overlap_diat_grad_dd_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 4, 2, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 3, 2, 1.3_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = -1e-7_wp
   vec(2) = -1e-7_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_dd_z

subroutine test_overlap_diat_grad_sf(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 5, 0, 2.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 1.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_sf

subroutine test_overlap_diat_grad_sf_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 5, 0, 2.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 1.0_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = -1e-7_wp
   vec(2) = 1e-7_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_sf_z

subroutine test_overlap_diat_grad_pf(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 5, 1, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 2.0_wp, cgtoj, .true., stat)


   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_pf

subroutine test_overlap_diat_grad_pf_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 5, 1, 1.0_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 2.0_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = -1e-7_wp
   vec(2) = 1e-7_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_pf_z

subroutine test_overlap_diat_grad_df(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 5, 2, 1.3_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 1.4_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_df

subroutine test_overlap_diat_grad_df_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 5, 2, 1.3_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 1.4_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = 1e-7_wp
   vec(2) = -1e-7_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_df_z

subroutine test_overlap_diat_grad_ff(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 5, 3, 1.6_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 1.0_wp, cgtoj, .true., stat)

   ! Randomly oriented vector of length 0.5
   call random_number(vec)
   vec = vec - 0.5_wp
   vec = 0.5_wp * vec / sqrt(sum(vec**2))

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_ff

subroutine test_overlap_diat_grad_ff_z(error)

   !> Error handling
   type(error_type), allocatable, intent(out) :: error

   integer, parameter :: ng = 6

   type(cgto_type) :: cgtoi, cgtoj
   real(wp) :: vec(3)
   real(wp) :: ksig, kpi, kdel
   integer :: stat

   call slater_to_gauss(ng, 5, 3, 1.6_wp, cgtoi, .true., stat)
   call slater_to_gauss(ng, 4, 3, 1.0_wp, cgtoj, .true., stat)

   ksig = 0.1_wp
   kpi = 0.2_wp
   kdel = 0.5_wp

   ! Vector along the x-axis
   vec(1) = 0.5_wp
   vec(2) = 0.0_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the y-axis
   vec(1) = 0.0_wp
   vec(2) = 0.5_wp
   vec(3) = 0.0_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector along the z-axis to test the ill-defined gradient
   vec(1) = 0.0_wp
   vec(2) = 0.0_wp
   vec(3) = -0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

   ! Vector nearly along the z-axis to test the ill-defined gradient
   vec(1) = -1e-7_wp
   vec(2) = -1e-7_wp
   vec(3) = 0.5_wp

   call test_overlap_diat_grad_gen(vec, ksig, kpi, kdel, cgtoi, cgtoj, error)

end subroutine test_overlap_diat_grad_ff_z

end module test_integral_overlap
