/* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   Copyright (c) 2013-2015 The plumed team
   (see the PEOPLE file at the root of the distribution for a list of names)

   See http://www.plumed-code.org for more information.

   This file is part of plumed, version 2.

   plumed is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   plumed is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with plumed.  If not, see <http://www.gnu.org/licenses/>.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
#include <complex>
#include "Steinhardt.h"

namespace PLMD {
namespace crystallization {

void Steinhardt::registerKeywords( Keywords& keys ){
  VectorMultiColvar::registerKeywords( keys ); 
  keys.add("compulsory","NN","12","The n parameter of the switching function ");
  keys.add("compulsory","MM","24","The m parameter of the switching function ");
  keys.add("compulsory","D_0","0.0","The d_0 parameter of the switching function");
  keys.add("compulsory","R_0","The r_0 parameter of the switching function");
  keys.add("optional","SWITCH","This keyword is used if you want to employ an alternative to the continuous swiching function defined above. "
                               "The following provides information on the \\ref switchingfunction that are available. "
                               "When this keyword is present you no longer need the NN, MM, D_0 and R_0 keywords.");
  keys.use("SPECIES"); keys.use("SPECIESA"); keys.use("SPECIESB");
  keys.use("MEAN"); keys.use("LESS_THAN"); keys.use("MORE_THAN"); keys.use("VMEAN");
  keys.use("BETWEEN"); keys.use("HISTOGRAM"); keys.use("MOMENTS"); keys.use("MIN"); keys.use("ALT_MIN");
  keys.use("LOWEST"); keys.use("HIGHEST");
}

Steinhardt::Steinhardt( const ActionOptions& ao ):
Action(ao),
VectorMultiColvar(ao),
tmom(0)
{
  // Read in the switching function
  std::string sw, errors; parse("SWITCH",sw);
  if(sw.length()>0){
     switchingFunction.set(sw,errors);
  } else {
     double r_0=-1.0, d_0; int nn, mm;
     parse("NN",nn); parse("MM",mm);
     parse("R_0",r_0); parse("D_0",d_0);
     if( r_0<0.0 ) error("you must set a value for R_0");
     switchingFunction.set(nn,mm,r_0,d_0);
  }
  log.printf("  Steinhardt parameter of central atom and those within %s\n",( switchingFunction.description() ).c_str() );
  // Set the link cell cutoff
  setLinkCellCutoff( switchingFunction.get_dmax() );
  rcut = switchingFunction.get_dmax(); rcut2 = rcut*rcut;
}

void Steinhardt::setAngularMomentum( const unsigned& ang ){
  tmom=ang; setVectorDimensionality( 2*(2*ang + 1), 2 );
} 

void Steinhardt::calculateVector( multicolvar::AtomValuePack& myatoms ) const {
  double dfunc, dpoly_ass, md, tq6, itq6, real_z, imag_z; 
  Vector dz, myrealvec, myimagvec, real_dz, imag_dz;
  // The square root of -1
  std::complex<double> ii( 0.0, 1.0 ), dp_x, dp_y, dp_z;

  unsigned ncomp=2*tmom+1;
  double sw, poly_ass, d2, dlen, nbond=0.0; std::complex<double> powered;
  for(unsigned i=1;i<myatoms.getNumberOfAtoms();++i){
      Vector& distance=myatoms.getPosition(i);  // getSeparation( myatoms.getPosition(0), myatoms.getPosition(i) );
      if ( (d2=distance[0]*distance[0])<rcut2 &&
           (d2+=distance[1]*distance[1])<rcut2 &&
           (d2+=distance[2]*distance[2])<rcut2) {
         dlen = sqrt(d2);
         sw = switchingFunction.calculate( dlen, dfunc ); 
   
         nbond += sw;  // Accumulate total number of bonds
         double dlen3 = d2*dlen;

         // Store derivatives of weight
         myatoms.addAtomsDerivatives( 1, 0, (-dfunc)*distance );
         myatoms.addAtomsDerivatives( 1, i, (+dfunc)*distance );
         myatoms.addBoxDerivatives( 1, (-dfunc)*Tensor( distance,distance ) ); 

         // Do stuff for m=0
         poly_ass=deriv_poly( 0, distance[2]/dlen, dpoly_ass );
         // Derivatives of z/r wrt x, y, z
         dz = -( distance[2] / dlen3 )*distance; dz[2] += (1.0 / dlen);
         // Derivative wrt to the vector connecting the two atoms
         myrealvec = (+sw)*dpoly_ass*dz + poly_ass*(+dfunc)*distance;
         // Accumulate the derivatives
         myatoms.addAtomsDerivatives( 2 + tmom, 0, -myrealvec );      
         myatoms.addAtomsDerivatives( 2 + tmom, i, myrealvec ); 
         myatoms.addBoxDerivatives( 2 + tmom, Tensor( -myrealvec,distance ) );
         // And store the vector function
         myatoms.addValue( 2 + tmom, sw*poly_ass );

         // The complex number of which we have to take powers
         std::complex<double> com1( distance[0]/dlen ,distance[1]/dlen );

         // Do stuff for all other m values
         for(unsigned m=1;m<=tmom;++m){
             // Calculate Legendre Polynomial
             poly_ass=deriv_poly( m, distance[2]/dlen, dpoly_ass );
             // Calculate powe of complex number
             powered=pow(com1,m-1); md=static_cast<double>(m);
             // Real and imaginary parts of z
             real_z = real(com1*powered); imag_z = imag(com1*powered );
 
             // Calculate steinhardt parameter
             tq6=poly_ass*real_z;   // Real part of steinhardt parameter
             itq6=poly_ass*imag_z;  // Imaginary part of steinhardt parameter

             // Derivatives wrt ( x/r + iy )^m
             dp_x = md*powered*( (1.0/dlen)-(distance[0]*distance[0])/dlen3-ii*(distance[0]*distance[1])/dlen3 );
             dp_y = md*powered*( ii*(1.0/dlen)-(distance[0]*distance[1])/dlen3-ii*(distance[1]*distance[1])/dlen3 );
             dp_z = md*powered*( -(distance[0]*distance[2])/dlen3-ii*(distance[1]*distance[2])/dlen3 );

             // Derivatives of real and imaginary parts of above
             real_dz[0] = real( dp_x ); real_dz[1] = real( dp_y ); real_dz[2] = real( dp_z );
             imag_dz[0] = imag( dp_x ); imag_dz[1] = imag( dp_y ); imag_dz[2] = imag( dp_z );  

             // Complete derivative of steinhardt parameter
             myrealvec = (+sw)*dpoly_ass*real_z*dz + (+dfunc)*distance*tq6 + (+sw)*poly_ass*real_dz; 
             myimagvec = (+sw)*dpoly_ass*imag_z*dz + (+dfunc)*distance*itq6 + (+sw)*poly_ass*imag_dz;

             // Real part
             myatoms.addValue( 2+tmom+m, sw*tq6 );
             myatoms.addAtomsDerivatives( 2+tmom+m, 0, -myrealvec );
             myatoms.addAtomsDerivatives( 2+tmom+m, i, myrealvec );
             myatoms.addBoxDerivatives( 2+tmom+m, Tensor( -myrealvec,distance ) );
             // Imaginary part 
             myatoms.addValue( 2+ncomp+tmom+m, sw*itq6 );
             myatoms.addAtomsDerivatives( 2+ncomp+tmom+m, 0, -myimagvec );
             myatoms.addAtomsDerivatives( 2+ncomp+tmom+m, i, myimagvec );
             myatoms.addBoxDerivatives( 2+ncomp+tmom+m, Tensor( -myimagvec,distance ) );
             // Store -m part of vector
             double pref=pow(-1.0,m); 
             // -m part of vector is just +m part multiplied by (-1.0)**m and multiplied by complex
             // conjugate of Legendre polynomial
             // Real part
             myatoms.addValue( 2+tmom-m, pref*sw*tq6 );
             myatoms.addAtomsDerivatives( 2+tmom-m, 0, -pref*myrealvec );
             myatoms.addAtomsDerivatives( 2+tmom-m, i, pref*myrealvec );
             myatoms.addBoxDerivatives( 2+tmom-m, pref*Tensor( -myrealvec,distance ) );
             // Imaginary part
             myatoms.addValue( 2+ncomp+tmom-m, -pref*sw*itq6 );
             myatoms.addAtomsDerivatives( 2+ncomp+tmom-m, 0, pref*myimagvec );
             myatoms.addAtomsDerivatives( 2+ncomp+tmom-m, i, -pref*myimagvec );
             myatoms.addBoxDerivatives( 2+ncomp+tmom-m, pref*Tensor( myimagvec,distance ) );
         }
     }
  } 

  // Normalize 
  myatoms.setValue(1, nbond ); updateActiveAtoms( myatoms );
  for(unsigned i=0;i<getNumberOfComponentsInVector();++i) myatoms.getUnderlyingMultiValue().quotientRule( 2+i, 1, 2+i ); 
  // Clear tempory stuff
  myatoms.getUnderlyingMultiValue().clear(1);
}

double Steinhardt::deriv_poly( const unsigned& m, const double& val, double& df ) const { 
  double fact=1.0;
  for(unsigned j=1;j<=m;++j) fact=fact*j;
  double res=coeff_poly[m]*fact;

  double pow=1.0, xi=val, dxi=1.0; df=0.0;
  for(int i=m+1;i<=tmom;++i){
      double fact=1.0;
      for(unsigned j=i-m+1;j<=i;++j) fact=fact*j;
      res=res+coeff_poly[i]*fact*xi;
      df = df + pow*coeff_poly[i]*fact*dxi;
      xi=xi*val; dxi=dxi*val; pow+=1.0;
  }
  df = df*normaliz[m];
  return normaliz[m]*res;
}

}
}
