! Copyright (C) 2022  Light and Molecules Group

! This program is free software: you can redistribute it and/or modify
! it under the terms of the GNU General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.

! This program is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.

! You should have received a copy of the GNU General Public License
! along with this program.  If not, see <https://www.gnu.org/licenses/>.

program nx_moldyn
  !! Main driver for performing Molecular Dynamics with Newton-X.
  use mod_kinds, only: dp
  use mod_constants, only: &
       & MAX_CMD_SIZE, MAX_STR_SIZE, TRAJ_KILLED_KILLSTAT, au2fs, &
       & convert_status_to_str
  use mod_numerics, only: uvip3p
  use mod_tools, only: &
       & nx_init_random, to_str, nx_step, clear, title_message, nx_restart_nad
  use mod_configuration, only: &
       & nx_config_t
  use mod_trajectory, only: &
       & nx_traj_t, traj_init_txt_files, traj_check_consistency, traj_t
  use mod_thermostat, only: &
       & nx_thermo_t, th_run
  use mod_qm_interfaces, only: nx_qm_create_item, nx_qminfo_t, nx_qm_item_t
  use mod_overlap, only: &
       & nx_moovlp_t, &
       & evaluate_metrics, ovl_run_new, ovl_compute_mo_ovl,&
       & read_mo_overlap, ovl_prepare_run_new
  use mod_nad_setup, only: &
       & nx_nad_t, nad_write_pairs
  use mod_cioverlap, only: &
       & nx_cioverlap_t, do_cioverlap
  use mod_aux_nad, only: nx_auxnac_t
  use mod_adaptive_step, only: &
       & nx_adaptive_ts_t
  use mod_sh_t, only: &
       & nx_sh_t
  use mod_sh, only: &
       & nx_sh_solver_t
  use mod_zpecorrect, only: &
       & nx_zpe_t, zpecorrect_model1
  use mod_md_utils, only: &
       & type_of_dynamics, is_section_here, create_directories, copy_initial_files, &
       & clean_pwd_content, &
       & nx_opts_t
  use mod_logger, only: &
       & nx_log, &
       & LOG_DEBUG, LOG_INFO, LOG_WARN, LOG_ERROR, &
       & check_error
  use mod_timers, only: &
       & time_r, nx_timers_t, T_QM_EXE, T_QM_READ
  use mod_input_parser, only: parser_t
  use mod_export_configuration, only: &
       & nx_export_config
  use mod_cs_fssh, only: &
       & nx_csfssh_params_t

#ifdef USE_HDF5
  use hdf5, only: h5open_f, h5close_f
#endif
  use mod_io, only: nx_output_t

  use iso_fortran_env, only: &
       & compiler_version, compiler_options, &
       & stdout => output_unit

  implicit none

  ! ========================
  ! Main NX-specific objects
  ! ========================
  type(nx_config_t) :: nx_conf
  ! Main configuration
  type(nx_traj_t) :: nx_traj
  ! Main classical trajectory
  type(nx_cioverlap_t) :: nx_cio
  ! Handling of cioverlap
  type(nx_auxnac_t) :: nx_auxnac
  ! Auxiliary non-adiabatic derivatives
  type(nx_moovlp_t) :: moovlp
  ! Handles information related to MO OVL
  type(nx_sh_t) :: nx_sh
  ! Surface hopping configuration
  type(nx_sh_solver_t) :: sh_solver
  type(nx_adaptive_ts_t) :: nx_adapt_ts
  ! Object to handle adaptive time steps
  type(nx_thermo_t) :: nx_thermo
  ! Object that handles thermostats
  type(nx_nad_t) :: nx_nad
  ! Informations about NAC computation
  type(nx_zpe_t) :: nx_zpe
  ! Object related to ZPE correction

  ! =============
  ! File handling
  ! =============
  character(len=:), allocatable :: conffile
  ! Hold the file names of the main configuration, QM
  ! configuration and file containing the computed gradient

  ! ============
  ! Main control
  ! ============
  integer :: ierr
  ! Error handling for running external commands
  character(len=30) :: curstep
  ! Current step for pretty printing
  character(len=MAX_CMD_SIZE) :: msg
  ! General messages.
  integer :: res
  logical :: forceprt
  real(dp) :: time_on_traj
  logical :: print_at_this_step

  integer :: maxstep
  integer :: status
  integer :: check_en_cons
  integer :: init_cycle

  ! ======
  ! Timers
  ! ======
  real(dp) :: t
  type(nx_timers_t) :: timers

  ! =============
  ! Output handling
  ! =============
  type(nx_output_t) :: nx_out

  ! ================
  ! MO OVL checkings
  ! ================
  real(dp), dimension(:, :), allocatable :: Smo
  ! MO OVL matrix

  ! ==========================
  ! Adaptive time-step related
  ! ==========================
  type(nx_traj_t), dimension(:), allocatable :: backup_traj
  ! Stores information about the current trajectory when using
  ! adaptive time step algorithm

  ! =======================
  ! Miscalleneous variables
  ! =======================
  integer :: i, k
  ! Loop handling
  logical :: has_h5, has_io, ext

  ! ==========
  ! Thermostat
  ! ==========
  logical :: call_thermo
  logical, dimension(:), allocatable :: freeze_me
  integer :: lts
  integer :: ktherm
  logical :: run_thermo

  ! =================
  ! Input file parser
  ! =================
  type(parser_t) :: parser
  type(nx_opts_t) :: opts

  character(len=*), parameter :: nl = NEW_LINE('c')

  type(nx_csfssh_params_t) :: csfssh
  logical :: csfssh_run_ref, csfssh_require_gamma

  integer :: tmpcount, tmpmax, tmpstart, tmpend

  type(nx_qm_item_t) :: new_qm
  type(nx_qminfo_t) :: qminfo
  logical :: is_test

  ! ==================================================
  ! END OF VARIABLE DECLARATION. MAIN CODE STARTS HERE
  ! ==================================================
  ! Default user-defined parameters parsing
  conffile = 'user_config.nml'

  call opts%parse()
  if (opts%status < 0) then
     print *, 'Termination required after command line parsing'
     stop
  end if

  if (opts%input /= 'UNALLOCATED') then
     conffile = opts%input
  end if

  ! Dummy initialization
  call nx_log%init(1, 0, 5 )

  call title_message(stdout)
  call nx_log%log(LOG_INFO, 'NX EXECUTABLE FOUND AT '//trim(opts%nx_exe_path))

  inquire(file='INFO_RESTART', exist=ext)
  if (ext) then
     conffile = 'INFO_RESTART/nx-restart-config.nml'
     write(msg, '(A)') &
          & '"INFO_RESTART" found in the current directory.'//nl//nl//&
          & '!! Dynamics will be restarted from INFO_RESTART/'//conffile//' !!'&
          & //nl
     call nx_log%log(LOG_WARN, msg)
  end if

  call parser%init(conffile)
  if (parser%status /= 0) then
     call nx_log%log(LOG_ERROR, parser%errmsg)
     error stop
  end if
  call parser%parse()
  if (parser%status /= 0) then
     call nx_log%log(LOG_ERROR, parser%errmsg)
     error stop
  end if

  check_en_cons = 0
  status = 0

  t = time_r()

  nx_conf = nx_config_t()
  call nx_log%log(LOG_INFO, 'Reading &nxconfig from '//conffile, force=.true.)
  call nx_conf%init( parser )
  call nx_conf%print()

  if (opts%clean_pwd) then
     call clean_pwd_content(nx_conf, ierr)
     call check_error(ierr, 101, &
          & 'Problem in cleaning working directory content', system=.true.)
  end if

  csfssh_run_ref = .false.
  csfssh_require_gamma = .false.
  if (nx_conf%run_complex) then
     call nx_log%log(LOG_INFO, 'CS-FSSH requested: initializing', force=.true.)
     call csfssh%init(nx_conf)

     csfssh_run_ref = (nx_conf%run_complex .and. nx_conf%gamma_model == 2)
     csfssh_require_gamma = (nx_conf%run_complex .and. nx_conf%gamma_model > 0)
     call csfssh%print()
  end if

  ! Timers initialization
  has_h5 = .false.
  has_io = .true.

#ifdef USE_HDF5
  has_io = .false.
  has_h5 = .true.
  if (nx_conf%use_txt_outputs) has_io = .true.
#endif

  call timers%init(&
       & with_cio=(nx_conf%dc_method == 2),&
       & with_sh=(nx_conf%thres > 0),&
       & with_auxnac=(nx_conf%dc_method == 3),&
       & with_io_txt=has_io,&
       & with_io_h5=has_h5&
       & )

  call system_clock(tmpstart, tmpcount, tmpmax)
  call timers%start('main')

  ! Surface hopping setup
  ! This should only be done for thres > 0 (else there is no need to
  ! compute NAD, and no transition will be possible anyway)
  if (nx_conf%thres > 0) then
     call nx_log%log(LOG_INFO, 'Reading &sh from '//conffile, force=.true.)
     call nx_sh% init( parser, nx_conf%dc_method, nx_conf%use_locdiab )
     call nx_sh% print()

     ! TODO: Move the ``seed`` to ``nxconfig``
     call nx_init_random(nx_sh% seed)
  else
     call nx_log%log(LOG_INFO, &
          & 'ADIABATIC DYNAMICS REQUESTED: NO SURFACE HOPPING', force=.true.)
  end if

  call nx_log%log(LOG_INFO, 'Initializing trajectory ', force=.true.)
  if (nx_conf%thres > 0) then
     nx_traj = traj_t(nx_conf%nat, nx_conf%nstat, nx_sh%ms, nx_conf%run_complex)
  else
     nx_traj = traj_t(nx_conf%nat, nx_conf%nstat, 1, nx_conf%run_complex)
  end if
  call nx_traj%init(nx_conf)

  if (nx_conf%thres > 0) then
    call nx_log%log(LOG_INFO, 'Reading &nad_setup from '//conffile, force=.true.)
    nx_nad = nx_nad_t()
    call nx_nad% init(parser, nx_conf)
    call nx_nad% print()
    
    if (nx_conf%dc_method == 2) then
      call nx_log%log(LOG_INFO, 'I found dc_method = 2 in &nxconfig', force=.true.)
      call nx_log%log(LOG_INFO, 'Reading &cioverlap from '//conffile, force=.true.)
      call nx_cio%init(nx_conf, parser)
      call nx_cio%print()
    else if (nx_conf%dc_method == 3) then
      call nx_log%log(LOG_INFO, 'I found dc_method = 3 in &nxconfig', force=.true.)
      call nx_log%log(LOG_INFO, 'Reading &auxnac from '//conffile, force=.true.)
      call nx_auxnac%init(parser, nx_conf%lvprt)
      call nx_auxnac%print()
    end if

    if ( (nx_conf%dc_method /= 2) .and. (nx_conf%check_mo_ovl == 1) ) then
      call nx_log%log(LOG_INFO, 'Found check_mo_ovl == 1 in &nxconfig', force=.true.)
      call nx_log%log(LOG_INFO, 'Reading &cioverlap from '//conffile, force=.true.)
      call nx_cio%init(nx_conf, parser)
      call nx_cio%print()
    end if

  end if
   
  if (nx_conf%with_adt) then
     call nx_log%log(LOG_INFO, 'Reading &adapt_dt from '//conffile, force=.true.)
     call nx_adapt_ts% init(parser)
     call nx_adapt_ts% print()

     ! Now set the maximum number of times we can go back in time
     if (.not. allocated(backup_traj)) then
        allocate(backup_traj( nx_adapt_ts%max_subtraj ))
     end if
  end if

  ! call_thermo = is_section_here(conffile, '&thermostat')
  call_thermo = .false.
  lts = -1
  ktherm = -1
  if (call_thermo) then
     call nx_log%log(LOG_INFO, 'Reading &thermostat from '//conffile, force=.true.)
     call nx_thermo% init(parser)

     ! Freeze atoms
     if (.not. allocated(freeze_me)) then
        allocate(freeze_me(nx_conf%nat))
     end if

     ! For now, let's not freeze any atom !!
     freeze_me(:) = .false.

     ! lts and ktherm (this should be done properly in the future)
     lts = nx_thermo%lts
     ktherm = nx_thermo%ktherm

  end if

  if (nx_conf%with_zpe) then
     call nx_log%log(LOG_INFO, 'Reading &zpe_correct from '//conffile, force=.true.)
     call nx_zpe% init(parser, nx_conf)
     call nx_zpe% print()
  end if

  call nx_log%log(LOG_INFO, 'Initializing QM computation', force=.true.)
  new_qm = nx_qm_create_item(nx_conf)
  if (new_qm%check_error() /= 0) call nx_main_terminate(1)
  
  call new_qm%setup(parser, nx_conf)
  if (new_qm%check_error() /= 0) call nx_main_terminate(1)
  
  call new_qm%print()
  if (new_qm%is_qmmm()) call new_qm%set_qmmm_in_traj(nx_traj)

  call nx_log%log(LOG_INFO, 'Initializing outputs', force=.true.)

  nx_out = nx_output_t(nx_conf)
  call nx_out%set_options( parser )

  if (opts%dry_run) then
     call nx_log%log(LOG_WARN, 'Dry-run requested: I am stopping now !')
     call nx_log%log(LOG_INFO, 'Normal termination of Newton-X', force=.true.)
     stop
  end if

  if (nx_conf%nxrestart < 2) then
     call nx_log%log(LOG_INFO, 'Creating directories', force=.true.)
     call create_directories(nx_conf, ierr)
     call check_error(ierr, 101, &
       & 'Error in directory creation', system=.true.)
  end if

  ! HDF5
#ifdef USE_HDF5
  call timers%start('io_h5')
  call h5open_f(ierr)

  if (nx_conf%nxrestart == 0) then
     call nx_log%log(LOG_INFO, 'Reading &h5md from '//conffile, force=.true.)
     call nx_out%init_h5()
     call nx_out%create_particles('all', ['none', 'none', 'none'], nx_traj)
     call nx_out%create_obs(nx_conf, nx_sh%ms)
  end if

  call timers%update('io_h5')
#else
  nx_conf%use_txt_outputs = .true.
#endif

  call timers%start('io_txt')
  call nx_log%log(LOG_INFO, 'Initializing TXT outputs', force=.true.)

  if (nx_conf%nxrestart == 0) then
     call nx_out%init_txt( )
     call timers%update('io_txt')
  end if

  call parser%clean()

  ! Main logger initialization
  call nx_log%init( nx_conf%kt, nx_conf%init_step, nx_conf%lvprt )

  t = time_r() - t
  write(msg, '(A)')&
       & 'Initialization done in '//to_str(t, fmt='(F6.3)')//' s.'
  call nx_log%log(LOG_INFO, msg, force=.true.)


  call nx_log%log(LOG_DEBUG, 'Copying input files to "TEMP/"')
  call copy_initial_files(nx_conf, conffile, ierr)
  call check_error(ierr, 101, &
       & 'Error in copying input files', system=.true.)

  call nx_log%log(LOG_DEBUG, 'Changing directory to "TEMP/"')
  ierr = chdir('TEMP')

  call nx_log%log(LOG_DEBUG, 'Setting correct path for outputs')
  call nx_out%reset_path('../'//nx_conf%output_path)

  maxstep = nx_conf%init_step&
        & + int((nx_conf%tmax - nx_conf%init_time + nx_conf%dt * au2fs * 1E-4_dp)&
        & / (nx_conf%dt * au2fs))
  call nx_log%log(LOG_INFO, 'Total number of steps: '//to_str(maxstep))

  ! ===============================================================
  ! STEP 0: Further initialization
  ! ===============================================================
  call nx_log%log(&
       & LOG_INFO, &
       & nx_step('Starting', nx_traj%step, nx_traj%t, nx_traj%is_virtual), &
       & force=.true.)

  write(msg, '(A)') 'Running on surface '//to_str(nx_traj%nstatdyn)
  call nx_log%log(LOG_INFO, msg)

  ! First we determine the type of dynamics and we write the pairs for which
  ! cioverlap will do stuff
  call nx_log%log(LOG_DEBUG, 'Deterimining type of dyn')
  call type_of_dynamics(nx_conf, nx_traj)
  write(msg, '(A)') 'Type of dynamics is: '//to_str(nx_traj%typeofdyn)
  call nx_log%log(LOG_DEBUG, msg)

  if (nx_conf%thres > 0) then
     call nx_log%log(LOG_DEBUG, 'Determining pairs for NAD')
     call nad_write_pairs(nx_nad, nx_conf, nx_traj)
     call nx_log%log_file(LOG_INFO, 'transmomin', 'Content of transmom&
          &in')
     call nx_log%log(LOG_DEBUG, nx_traj%couplings, 'Couplings matri&
          &x', fmt='I2', expand=.true.)
  end if

  if (nx_conf%with_zpe) then
     call nx_log%log(LOG_INFO, 'Preparation of ZPE')
     call nx_zpe%get_ahbond(nx_traj, nx_conf)
  end if

  call timers%start('qm_exe')
  if (csfssh_run_ref) then
     call nx_log%log(LOG_INFO, 'CSFSSH: Start reference QM job')
     call csfssh%run_reference(new_qm, nx_conf, nx_traj)
  end if

  if (nx_conf%nxrestart < 2) then
    call nx_log%log(LOG_INFO, 'Start QM job')
    call new_qm%update(nx_conf, nx_traj)
    if (new_qm%check_error() /= 0) call nx_main_terminate(1)

    call new_qm%run()
    if (new_qm%check_error() /= 0) call nx_main_terminate(1)
  else
    call nx_log%log(LOG_INFO, 'Skiping QM job')
  end if
  call timers%update('qm_exe')

  call timers%start('qm_read')
  qminfo = new_qm%read_output(nx_conf, nx_traj)
  if (new_qm%check_error() /= 0) call nx_main_terminate(1)

  if (nx_conf%nxrestart == 2) then 
    call nx_restart_nad(qminfo%rnad)
  end if
 
  call new_qm%transfer_to_traj( &
       & qminfo, nx_traj, nx_conf, update_epot=.true., report=.true. &
       & )
  call timers%update('qm_read')

  if (nx_conf%run_complex) then
     call nx_log%log(LOG_INFO, 'CS-FSSH: Computing resonances')
     call csfssh%get_resonance(nx_traj)
  end if

  write(msg, '(A)') &
       & 'QM computation done in '//&
       & to_str( timers%sum([T_QM_READ, T_QM_EXE]), fmt='(F10.3)' )//'s.'
  call nx_log%log(LOG_INFO, msg)


  if (nx_conf%lvprt >= 5) then
     call nx_log%log(LOG_DEBUG, nx_traj%grad(nx_traj%nstatdyn, :, :), &
          & title='Gradient for current state', &
          & transpose=.true., fmt='E16.7' &
          & )

     do k=1, size(nx_traj%grad, 1)
        call nx_log%log(LOG_DEBUG, nx_traj%grad(k, :, :), &
             & title='Gradient for state '//to_str(k), &
             & transpose=.true., fmt='E16.7' &
             & )
     end do

     do k=1, size(nx_traj%nad, 1)
        call nx_log%log(LOG_DEBUG, nx_traj%nad(k, :, :), &
             & title='NAD for couplings '//to_str(k), &
             & transpose=.true., fmt='E16.7' &
             & )
     end do
  end if

  ! Initialize sh_traj information with first run
  if (nx_conf%thres > 0) then
   ! Generate inputs for double molecule run
    if (nx_conf%nxrestart < 2) then
      call nx_log%log(LOG_INFO, 'Preparing overlap <t | t-dt>')
      call ovl_prepare_run_new( new_qm )
    end if

    if (nx_conf%dc_method == 2) then
       ! Init cio: prepare double molecule input, generate
       ! list of Slater determinants and generate the first
       ! Casida WF in cioverlap.old
       call nx_log%log(LOG_INFO, 'Preparation of cioverlap')
       call timers%start('cio')
       if (nx_conf%nxrestart == 2) then
         call nx_cio%prepare( new_qm, nx_conf, nx_traj, call_cio= .false. )
       else
         call nx_cio%prepare( new_qm, nx_conf, nx_traj )
       end if
       call timers%update('cio')
       write(msg, '(A, A, A)') &
             & 'Cioverlap preparation done in ', timers%t_cio%print(), ' s.'
       call nx_log%log(LOG_INFO, msg)
    end if

    ! time_on_traj = 0.0_dp
    sh_solver = nx_sh_solver_t(&
          & nx_sh, nx_conf, nx_traj%masses, &
          & wf=nx_traj%wf, phase=nx_traj%sh_phase, geom=nx_traj%geom &
          & )

    if (nx_traj%is_qmmm) then
    call sh_solver%init_qmmm(nx_traj%is_qm_atom)
    end if
    
    if (nx_conf%run_complex) then
      call sh_solver%init_csfssh(csfssh)
    end if
    
    ! Set data from trajectory
    call sh_solver%set(nx_traj, is_init=.true.)
  end if
    
  call nx_log%log(LOG_INFO, 'Back up QM information')
  call new_qm%backup(nx_conf, nx_traj)

  call nx_log%log(LOG_DEBUG, 'Computing energies')
  call nx_traj% compute_energies()
  nx_traj%etot0 = nx_traj%etot

  if (nx_conf%nxrestart == 0) then
     ! We only print this step if it is not a restart (else, it is
     ! redundant with the information already contained in the output)
#ifdef USE_HDF5
     call timers%start('io_h5')
     call nx_out%write_h5(nx_traj)
     call timers%update('io_h5')
#endif

     if (nx_conf%use_txt_outputs) then
        call timers%start('io_txt')
        call nx_out%write_txt(nx_traj)
        call timers%update('io_txt')
     end if
  end if

  write(msg, '(A18, F20.12)') 'Total energy: ', nx_traj%etot
  call nx_log%log(LOG_INFO, msg, force=.true.)
  write(msg, '(A18, F20.12)') 'Kinetic energy: ', nx_traj%ekin
  call nx_log%log(LOG_INFO, msg, force=.true.)
  write(msg, '(A18, F20.12)') &
       & 'Potential energy: ', nx_traj%epot(nx_traj%nstatdyn)
  call nx_log%log(LOG_INFO, msg, force=.true.)

  if (nx_conf%thres > 0) then
     call nx_log%log(LOG_INFO, nx_traj%wf, title='Wavefunction')
     call nx_log%log(LOG_INFO, abs(nx_traj%wf)**2, title='Electronic populations')
  end if

  if (nx_conf%with_adt) nx_adapt_ts% dt_init = nx_traj%dt

  call nx_log%log(&
       & LOG_INFO, &
       & nx_step('Finished', nx_traj%step, nx_traj%t, nx_traj%is_virtual), &
       & force=.true.)
  call nx_log%log_blank(LOG_INFO, n=5, force=.true.)

  ! ===============================================================
  ! MAIN LOOP STARTING NOW
  ! ===============================================================
  init_cycle = 0
  MAIN_LOOP: do while ((nx_traj%step < maxstep) .or. nx_traj%is_virtual)

     ! Update dynamics properties
     call nx_traj% update_s_and_t()
     ! call update_logger(nx_traj%step)
     call nx_log%update(nx_traj%step)

     call nx_log%log(&
          & LOG_INFO, &
          & nx_step('Starting', nx_traj%step, nx_traj%t, nx_traj%is_virtual), &
          & force=.true.)

     print_at_this_step = (mod(nx_traj%step, nx_conf%kt) == 0)

     ! write(msg, '(A)') 'Running on surface '//to_str(nx_traj%nstatdyn)
     call nx_log%log(LOG_INFO, 'Running on surface '//to_str(nx_traj%nstatdyn))

     ! First we determine the type of dynamics and we write the pairs for which
     ! cioverlap will do stuff
     call nx_log%log(LOG_DEBUG, 'Deterimining type of dyn')
     call type_of_dynamics(nx_conf, nx_traj)
     ! write(msg, '(A)') 'Type of dynamics is: '//to_str(nx_traj%typeofdyn)
     call nx_log%log(LOG_DEBUG, 'Type of dynamics is: '//to_str(nx_traj%typeofdyn))

     if (nx_conf%thres > 0) then
        call nx_log%log(LOG_DEBUG, 'Determining pairs for NAD')
        call nad_write_pairs(nx_nad, nx_conf, nx_traj)
        call nx_log%log_file(LOG_INFO, 'transmomin', 'Content of trans&
             &momin')
        call nx_log%log(LOG_DEBUG, nx_traj%couplings, 'Couplings matri&
             &x', fmt='I2', expand=.true.)
     end if

     ! First step of the Velocity Verlet
     ! After this step nx_traj contains:
     ! geom -> r(t + dt)
     ! veloc -> v(t + 0.5*dt)
     ! acc -> a(t)
     ! old_geom -> r(t)
     ! old_veloc -> v(t)
     ! old_acc -> a(t)
     ! curstep = 'Updating coordinates'
     ! write(msg, '(A30, A5)') curstep, ' ... '
     call nx_log%log(LOG_DEBUG, 'Updating coordinates')
     if (nx_conf%read_geom_list) then
       call nx_traj% read_pos_from_file(nx_conf%geom_file_list)
     else
       call nx_traj% update_pos()
     end if


     ! Call to external QM program: acc(t + dt)
     call timers%start('qm_exe')
     if (csfssh_run_ref) then
        call nx_log%log(LOG_INFO, 'CSFSSH: Start reference QM job')
        call csfssh%run_reference(new_qm, nx_conf, nx_traj)
     end if
     call nx_log%log(LOG_INFO, 'Start QM job')

     call new_qm%update(nx_conf, nx_traj)
     call new_qm%run()
     qminfo = new_qm%read_output(nx_conf, nx_traj)
     call new_qm%transfer_to_traj(&
          & qminfo, nx_traj, nx_conf, update_epot=.true., report=.true.)
     call timers%update('qm_exe')

     if (nx_conf%run_complex) then
        call nx_log%log(LOG_INFO, 'CS-FSSH: Computing resonances')
        call csfssh%get_resonance(nx_traj)
     end if

     write(msg, '(A)') &
          & 'QM computation done in '//&
          & to_str( timers%sum([T_QM_EXE, T_QM_READ]), fmt='(F10.3)')&
          & //'s.'
     call nx_log%log(LOG_INFO, &
          & 'QM computation done in '// &
          & to_str( timers%sum([T_QM_EXE, T_QM_READ]), fmt='(F10.3)')//'s.' &
          & )

     if (nx_conf%thres > 0) then
      ! Overlap computation
      if ((nx_conf%check_mo_ovl == 1) &
            & .or. (nx_conf%dc_method == 2)) then
          call nx_log%log(LOG_INFO, 'Double molecule computation')
          ! call ovl_run( nx_traj, nx_qm )
          call ovl_run_new(new_qm, nx_traj)
          if (nx_conf%dc_method /= 2) then
            call nx_log%log(LOG_DEBUG, 'Start MO OVL computation')
            call ovl_compute_mo_ovl(new_qm, Smo, nx_cio%cio_path)
            call nx_log%log(LOG_DEBUG, 'End MO OVL computation')
          end if
      end if

      CIOVERLAP: if (nx_conf%dc_method == 2) then
          call nx_log%log(LOG_INFO, 'Start time derivatives computation')
          call timers%start('cio')
          call do_cioverlap(nx_conf, new_qm, nx_cio, nx_traj)

          if (nx_conf%check_mo_ovl == 1) then
            call read_mo_overlap('cioverlap/cioverlap.out', Smo) ! Required for checking MO OVL
          end if

          call timers%update('cio')
          ! write(msg, '(A, A, A)') &
          !      &'Time derivatives done in ', timers%print('cio'), ' s.'
          call nx_log%log(LOG_INFO, 'Time derivatives done in '//timers%print('cio')//' s.')

      end if CIOVERLAP

      AUXNAC: if (nx_conf%dc_method == 3) then
          call nx_log%log(LOG_INFO, 'Start time derivatives computation (AUXNAD)')
          call timers%start('auxnad')
          call timers%update('auxnad')

          call nx_auxnac%run( nx_traj )

          ! write(msg, '(A, A, A)') &
          !      &'Time derivatives done in ', timers%print('auxnad'), ' s.'
          call nx_log%log(LOG_INFO, 'Time derivatives done in '//timers%print('auxnad')//' s.')
      end if AUXNAC
    end if

     call nx_log%log(LOG_INFO, 'Back up QM information')
     ! call qm_backup(nx_qm, nx_conf, nx_traj)
     call new_qm%backup(nx_conf, nx_traj)

     ! Debug printing if required.
     call nx_log%log(LOG_DEBUG, nx_traj%grad(nx_traj%nstatdyn, :, :), &
          & title='Gradient for current state', &
          & transpose=.true., fmt='E16.7' &
          & )

     do k=1, size(nx_traj%grad, 1)
        call nx_log%log(LOG_DEBUG, nx_traj%grad(k, :, :), &
             & title='Gradient for state '//to_str(k), &
             & transpose=.true., fmt='E16.7' &
             & )
     end do

     do k=1, size(nx_traj%nad, 1)
        write(curstep, '(I0)') k
        call nx_log%log(LOG_DEBUG, nx_traj%nad(k, :, :), &
             & title='NAD for couplings '//to_str(k), &
             & transpose=.true., fmt='E16.7' &
             & )
     end do

     ! Now we can perform the checkings on the MO OVL. If the values
     ! are too low or too high (depending on diagnostics), then the
     ! user is warned.
     ! If analytical model, the computation is not carried out !
     ! TODO: Implement killing the trajectory if the conditions are
     ! not satisfied ?

     ! This requires cioverlap run (for now ?), and should not be called in
     ! adiabatic trajectories.
     if (nx_conf%check_mo_ovl == 1) then
        call evaluate_metrics(nx_conf, Smo, moovlp)
     end if


     ADAPT_DT: if (moovlp%ill_conditioned .or. new_qm%request_adapt_dt) then

        if (.not. nx_conf%with_adt) then
           msg = 'Adaptive time-step should be called here, but is diabled'
           msg = trim(msg)//NEW_LINE('a')//&
                & '    Please set "with_adt = 1" in the configuration file !'
           call nx_log%log(LOG_WARN, msg)
           exit ADAPT_DT
        end if

        ! Update ratio if needed (for model 1)
        if (nx_adapt_ts%model == 1) then
           nx_adapt_ts%ratio = 1 / moovlp%norm_sim
        end if

        if(new_qm%request_adapt_dt) new_qm%request_adapt_dt = .false.

        if (nx_traj%is_virtual) then
           ! We are already in a subtrajectory
           ierr = chdir('../')
           if ( ierr /= 0 ) then
              call nx_log%log(LOG_ERROR, 'ADT: Cannot go back in directory.')
              ERROR STOP
           end if
        end if

        call nx_adapt_ts%create(nx_traj, new_qm)

        call nx_log%log(&
             & LOG_INFO, &
             & nx_step('Finished', nx_traj%step, nx_traj%t, nx_traj%is_virtual), &
             & force=.true.)
        call nx_log%log_blank(LOG_INFO, n=5, force=.true.)

        if (.not. nx_traj%is_virtual) then
           ! Backup the main trajectory
           call backup_traj(1)%copy(nx_traj)
        else
           call nx_traj%copy(backup_traj(1))
        end if

        nx_traj%is_virtual = .true.
        call nx_traj%go_back()
        nx_traj%step = nx_conf%init_step + 1
        nx_traj%dt = nx_adapt_ts%dt

        ierr = chdir ( nx_adapt_ts%new_folder )
        if ( ierr /= 0 ) then
           call nx_log%log(LOG_ERROR, 'ADT: Cannot change directory.')
           ERROR STOP
        end if

        CYCLE MAIN_LOOP
     end if ADAPT_DT

     ! Second step of velocity Verlet
     ! After this step nx_traj contains:
     ! geom -> r(t + dt)
     ! veloc -> v(t + dt)
     ! acc -> a(t + dt)
     ! old_geom -> r(t)
     ! old_veloc -> v(t)
     ! old_acc -> a(t)
     call nx_log%log(LOG_DEBUG, 'Updating velocities')
     call nx_traj% update_veloc()

     call nx_log%log(LOG_DEBUG, 'Computing energies')
     call nx_traj% compute_energies()

     ! Time-dependent Schrodinger equation integration and
     ! surface hopping routines
     SH_ROUTINE: if (nx_conf%thres > 0) then
        call nx_log%log(LOG_INFO, 'Time-Dependent Schrödinger Equation')

        ! Decide if we should skip this section, by comparing the energy separation
        ! between the current state and the adjacent states to ``thres``.
        call nx_traj%check_de_sup_inf(nx_conf%thres, res, msg)
        if (res < 0) then
           call nx_log%log(LOG_WARN, ' !! SKIPPING TDSE INTEGRATION !!'//NEW_LINE('c')//msg)
           exit SH_ROUTINE
        end if

        call timers%start('sh')
        call sh_solver%set(nx_traj)
        if (.not. nx_conf%run_complex) then
           call sh_solver%run(nx_traj, nx_out, new_qm, nx_conf)
        else
           call sh_solver%run(nx_traj, nx_out, new_qm, nx_conf)
        end if
        call timers%update('sh')

        ! write(msg, '(A ,A, A)') &
        !      & 'Integration done in ', timers%print('sh'), ' s.'
        call nx_log%log(LOG_INFO, 'Integration done in '//timers%print('sh')//' s.')

        ! Check against killstat and timekill
        if (nx_traj%old_nstatdyn /= nx_traj%nstatdyn) then
           time_on_traj = 0.0_dp
           nx_traj%time_on_traj = time_on_traj
        else
           time_on_traj = time_on_traj + nx_traj%dt
           nx_traj%time_on_traj = time_on_traj
        end if

        if ((nx_traj%nstatdyn == nx_conf%killstat) &
             & .and. (nx_conf%timekill /= 0)) then
           if (time_on_traj >= nx_conf%timekill) then

              msg = ' !! ENDING THE TRAJECTORY NOW !!'//NEW_LINE('c')
              msg = msg//'  The trajectory spent '//&
                   & to_str(time_on_traj, fmt='(F10.3)')// &
                   & ' fs. on state '//to_str(nx_traj%nstatdyn)//NEW_LINE('c')
              msg = msg//'  Setup is: killstat = '//to_str(nx_conf%killstat)// &
                   &' ; timekill = '//to_str(nx_conf%timekill, fmt='(F10.3)')//NEW_LINE('c')
              call nx_log%log(LOG_WARN, msg)

              status = TRAJ_KILLED_KILLSTAT

              call nx_log%log(&
                   & LOG_INFO, &
                   & nx_step('Finished', nx_traj%step, nx_traj%t, nx_traj%is_virtual), &
                   & force=.true.)
              call nx_log%log_blank(LOG_INFO, n=5, force=.true.)
              exit MAIN_LOOP
           end if
        end if

     end if SH_ROUTINE


     ! Thermostat
     if (call_thermo) then
        call nx_log%log(LOG_INFO, 'Starting thermostat')

        if (nx_thermo%lts == -1) lts = nx_traj%step + 1

        ! In the following loop we decide if the thermostat has to be
        ! run or not, based on the current step and on the current
        ! energy surface.
        if ((nx_traj%step >= nx_thermo%kts) &
             & .and. (nx_traj%step <= lts)) then
           ! We are in the right range of steps where thermostat
           ! should be run.

           run_thermo = .true.
           call nx_log%log(LOG_DEBUG, ' step >= kts and step <= lts')

           if (nx_thermo%nstherm == 0) then
              ! Do not use thermostat on the excited state !
              if (nx_traj%nstatdyn >= 2) then
                 call nx_log%log(LOG_DEBUG, ' nstherm = 0 and nstatdyn > 1')
                 run_thermo = .false.
              end if
           end if
        else
           run_thermo = .false.
        end if

        ! Finally, run the thermostat if required !
        if (run_thermo) then
           call nx_log%log(LOG_INFO, ' Running thermostat at this step')
           call th_run(nx_thermo, nx_traj, freeze_me)
        else
           call nx_log%log(LOG_INFO, ' Thermostat NOT running at this step')
        end if
     end if

     call nx_log%log(LOG_DEBUG, 'Computing momentum')
     call nx_traj% compute_momentum()

!    *************************************
!    ZPE correction takes place
!    *************************************
     ZPECORRECT: if (nx_conf%with_zpe) then

       select case(nx_zpe%kmodel)

       case(1)
          call nx_log%log(LOG_DEBUG, 'ZPE correction Model 1 is switched on')
          call zpecorrect_model1(nx_traj, nx_conf, nx_zpe)

       case(2)
         call nx_log%log(LOG_DEBUG, 'ZPE correction Model 2 is switched on')

      end select

     end if ZPECORRECT
! *************************************


     if (nx_traj%old_nstatdyn /= nx_traj%nstatdyn) then
        ! If surface hopping occured, we need to recompute the gradient
        ! for the next step (the current one corresponds to the old PES).
        ! res = message('Surface hopping occured: running second QM job')
        ! res = message('')
        call nx_log%log(LOG_WARN, 'Surface hopping occured: running second QM job')

        ! Do not update epot !!
        ! nx_qm%update_epot = 0

        call nx_log%log(LOG_DEBUG, 'Setting up QM')

        ! write(backup_cmd, '(A)') &
        !      & backup_script//' '//to_str(nx_traj%step)//'_2 '//trim(backup_folder)
        ! call qm_update_input(nx_qm, nx_traj, nx_conf )
        ! call new_qm%update(nx_conf, nx_traj)
        call nx_traj% write_coordinates( nx_conf%progname )

        call nx_log%log(LOG_INFO, 'Start QM job')
        call timers%start('qm_exe')
        ! call qm_run(nx_qm, nx_conf, nx_traj)
        call timers%update('qm_exe')

        call new_qm%update(nx_conf, nx_traj)
        call new_qm%run()
        qminfo = new_qm%read_output(nx_conf, nx_traj)
        call new_qm%transfer_to_traj(&
             & qminfo, nx_traj, nx_conf, update_epot=.false., report=.true.)

        ! Process information
        call timers%start('qm_read')
        ! call qm_read(nx_qm, nx_traj, nx_conf)
        call timers%update('qm_read')
        ! write(msg, '(A)') &
        msg = 'QM computation done in '//&
             & to_str( timers%sum([T_QM_EXE, T_QM_READ]), fmt='(F10.3)')//'s.'
        call nx_log%log(LOG_INFO, msg)

        call nx_log%log(LOG_DEBUG, nx_traj%grad(nx_traj%nstatdyn, :, :), &
             & title='Gradient for current state', &
             & transpose=.true., fmt='E16.7' &
             & )

        do k=1, size(nx_traj%grad, 1)
           call nx_log%log(LOG_DEBUG, nx_traj%grad(k, :, :), &
                & title='Gradient for state '//to_str(k), &
                & transpose=.true., fmt='E16.7' &
                & )
        end do

        ! Reset: read epot at the next step
        ! nx_qm%update_epot = 1
     end if

     ! Report
     ! Recompute energies (only necessary if surface hopping occured)
     call nx_log%log(LOG_DEBUG, 'Computing energies')
     call nx_traj% compute_energies()

     forceprt = .false.
     if (nx_traj%nstatdyn /= nx_traj%old_nstatdyn) then
        forceprt = .true.
     end if

     write(msg, '(A18, F20.12)') 'Total energy: ', nx_traj%etot
     call nx_log%log(LOG_INFO, msg, force=forceprt)
     write(msg, '(A18, F20.12)') 'Kinetic energy: ', nx_traj%ekin
     call nx_log%log(LOG_INFO, msg, force=forceprt)
     write(msg, '(A18, F20.12)') &
          & 'Potential energy: ', nx_traj%epot(nx_traj%nstatdyn)
     call nx_log%log(LOG_INFO, msg, force=forceprt)

     if (nx_conf%thres > 0) then
        call nx_log%log(LOG_INFO, nx_traj%wf, title='Wavefunction')
        call nx_log%log(LOG_INFO, abs(nx_traj%wf)**2, title='Electronic populations')
        call nx_log%log(LOG_INFO, nx_traj%nrofhops, &
             & title='Hopping events (hopping, rejected(ekin), rejected(veloc))')
     end if

     ! Energy checks
     status = 0
     call traj_check_consistency(&
          &nx_traj, nx_conf%etot_drift, nx_conf%etot_jump, nx_conf%epot_diff, status, msg &
          &)
     if (status /= 0) then
        check_en_cons = 1
        call nx_log%log(LOG_ERROR, 'Consistency check failed ('//to_str(status)//'): '//trim(msg))
        exit MAIN_LOOP
     end if

     ! Before the end of the loop we check the actual time to
     ! increase dt
     if (nx_traj%is_virtual) then
        call nx_adapt_ts%increase_dt(nx_traj)
     end if

     if ((nx_traj%nstatdyn /= nx_traj%old_nstatdyn) &
          & .or. (mod(nx_traj%step, nx_conf%kt) == 0)) then

        if (.not. nx_traj%is_virtual) then
#ifdef USE_HDF5
           call timers%start('io_h5')
           call nx_out%open_h5()
           call nx_out%write_h5(nx_traj)
           call nx_out%h5%close()

           call timers%update('io_h5')

           if (mod(nx_traj%step, nx_out%freq_backup_file) == 0) then
              call nx_out%backup_h5_file()
           end if
#endif
           if (nx_conf%use_txt_outputs) then
              call timers%start('io_txt')
              call nx_out%write_txt(nx_traj)
              call timers%update('io_txt')
           end if
        end if
     end if

     ! ! Some additional inputs for exciton models and mopac
     ! if (nx_qm%qmcode .eq. 'exc_mopac' .or. &
     !   & nx_qm%qmcode .eq. 'exc_gaussian' .or. &
     !   & nx_qm%qmcode .eq. 'mopac') then
     !    if (nx_qm%verboseexc .eq. 2 .or. nx_qm%verboseexc .eq. 3) then
     !       init_cycle = init_cycle + 1
     !       call wdyn(nx_qm, nx_conf, nx_traj, &
     !                  & check_en_cons)
     !    end if
     ! end if
     ! ! Some additional inputs for exciton models and mopac
     ! if (nx_qm%qmcode .eq. 'exc_mopac' .or. &
     !   & nx_qm%qmcode .eq. 'exc_gaussian' .or. &
     !   & nx_qm%qmcode .eq. 'mopac') then
     !    if (nx_qm%verboseexc .eq. 2 .or. nx_qm%verboseexc .eq. 3) then
     !       if (init_cycle .eq. 1) then
     !          call winf(nx_qm, nx_conf)
     !       end if
     !    end if
     ! end if

     call nx_log%log(&
          & LOG_INFO, &
          & nx_step('Finished', nx_traj%step, nx_traj%t, nx_traj%is_virtual), &
          & force=.true.)
     call nx_log%log_blank(LOG_INFO, n=5, force=.true.)

  end do MAIN_LOOP


  ! Clean up the different objects
  call nx_log%log(LOG_INFO, 'Cleaning memory', force=.true.)

  call nx_traj%destroy()
  if ((nx_conf%dc_method == 2) .and. (nx_conf%thres > 0)) then
     call nx_cio%destroy()
  end if
  ! call nx_qm% destroy()
  if (nx_conf%with_zpe) then
     call nx_zpe%destroy()
  end if
  if (allocated(Smo)) deallocate(Smo)
  if (nx_conf%thres > 0 .and. maxstep > 0) then
     call sh_solver% destroy()
  end if

  if (nx_conf%thres > 0) then
     call nx_nad%destroy()
  end if

  call nx_conf%destroy()
  deallocate(conffile)


#ifdef USE_HDF5
  call timers%start('io_h5')
  call nx_out%open_h5()

  if ((status == 0) .and. (nx_traj%step == maxstep)) then
     call nx_out%h5%finalize( convert_status_to_str(1) )
  else
     call nx_out%h5%finalize( convert_status_to_str(status) )
  end if

  call nx_out%h5%close()
  call h5close_f(ierr)
  call timers%update('io_h5')
#endif

  call timers%update('main')
  call timers%summary()
  call system_clock(tmpend)
  ierr = chdir('../')

  call nx_main_terminate(status)

contains

  subroutine nx_main_terminate(code)
    integer, intent(in) :: code

    if (code == 0) then
       call nx_log%log(LOG_INFO, 'Normal termination of Newton-X', force=.true.)
    else
       call nx_log%log(LOG_INFO, 'ABNORMAL TERMINATION OF  NEWTON-X', force=.true.)
       stop 1
    end if
  end subroutine nx_main_terminate

end program nx_moldyn
