! Copyright (C) 2022  Light and Molecules Group

! This program is free software: you can redistribute it and/or modify
! it under the terms of the GNU General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.

! This program is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.

! You should have received a copy of the GNU General Public License
! along with this program.  If not, see <https://www.gnu.org/licenses/>.

module mod_nx_geninp
  !! author: Baptiste Demoulin <baptiste.demoulin@univ-amu.fr>
  !! date: 2023-02-05
  !!
  !! # Helper module for Newton-X input generator.
  !!
  !! The input generator is interactive, and populates a parser (type ``parser_t``,
  !! defined in module ``mod_input_parser.f90``), that is then printed as
  !! ``user_basic.nml`` and propagated to every trajectory folder (if trajectory
  !! generation is handled).
  !!
  !! The ``parser_t`` type object is flexible, and can be passed around each routine to
  !! serve as a basis for determining default values, or to be updated.  As a side
  !! effect, only the elements that the user has interacted with will be printed in the
  !! input file.
  use mod_adaptive_step, only: nx_adaptive_ts_t
  use mod_aux_nad, only: nx_auxnac_t
  use mod_cioverlap, only: nx_cioverlap_t
  use mod_constants, only: MAX_STR_SIZE, fs2au, au2ev, cm2au
  use mod_configuration, only: nx_config_t
  use mod_cs_fssh, only: nx_csfssh_params_t
  use mod_fortran_menu, only: &
       & fmenu => fmenu_selection_menu, &
       & ask, ask_y_or_n
  use mod_gen_traj, only: nx_gen_traj_t
  use mod_kinds, only: dp
  use mod_implemented, only: nx_interfaces_t
  use mod_input_parser, only: parser_t
  use mod_interface, only: copy, rm
  use mod_keyval_pairs, only: pair_t
  use mod_nad_setup, only: nx_nad_t
  use mod_qm_interfaces
  use mod_sh_t, only: nx_sh_t
  use mod_tools, only: to_str, split_pattern
  use mod_zpecorrect, only: nx_zpe_t
  use iso_fortran_env, only: &
       & stdout => output_unit, stdin => input_unit

  implicit none

  private

  character(len=1), parameter :: nl = NEW_LINE('a')
  character(len=1), parameter :: tab = achar(9)

  public :: main_loop

contains

  subroutine main_loop()
    !! Main loop for the ``nx_geninp`` program.
    !!
    logical :: has_generated
    type(parser_t) :: config
    integer :: temp

    call nxinp_greeter()

    config = nxinp_init_parser()

    has_generated = .false.
    MAIN: do while (.true.)
       call nxinp_greeter()
       temp = nxinp_select_main()

       select case(temp)
       case(1)
          call nxinp_basic_configuration(config)
         !  exit MAIN
       ! case(2)
       !    if (.not. config%has_section('nxconfig')) then
       !       call no_basic_found()
       !    else
       !       GENERAL: do while(.true.)
       !          call nxinp_greeter()
       !          temp = nxinp_select_general()
       ! 
       !          select case(temp)
       !          case(1)
       !             call nxinp_set_general_dynamics(config)
       !          case(2)
       !             call nxinp_set_qm(config)
       !          case(3)
       !             call nxinp_set_zpe_corrections(config)
       !          case(4)
       !             call nxinp_set_adaptive_ts(config)
       !          case(5)
       !             exit GENERAL
       !          end select
       !       end do GENERAL
       !    end if
       case(2)
          if (.not. config%has_section('nxconfig')) then
             call no_basic_found()
          else
             call nxinp_greeter()
             temp = nxinp_select_nad_options()

             select case(temp)
             case(1)
                call nxinp_set_tdse_parameters(config)
             case(2)
                call nxinp_set_which_nad(config)
             case(3)
                call nxinp_set_how_nad(config)
             end select
          end if
       case(3)
          if (.not. config%has_section('nxconfig')) then
             call no_basic_found()
          else
             call nxinp_set_gen_traj(config, has_generated)
          end if
       case(4)
          if (.not. config%has_section('nxconfig')) then
             call no_basic_found()
          else
             if (.not. has_generated) then
                open(newunit=config%fileu, file=config%filename)
                call config%print(config%fileu)
                close(config%fileu)
             end if

             exit MAIN
          end if
       end select
    end do MAIN
  end subroutine main_loop


  subroutine no_basic_found()
    write(stdout, '(A)') ''
    write(stdout, '(A)') '     ! No basic configuration found !'
    write(stdout, '(A)') '! Please use option (1) before proceeding !'
    write(stdout, '(A)') ''
    read(stdin, *)
  end subroutine no_basic_found
  

  subroutine nxinp_greeter()
    !! Print the greeter of ``nx_geninp``.
    !!
    call execute_command_line('clear')
    write(stdout, '(A)') "nxinp  Copyright (C) 2022  Light and Molecules Group"
    write(stdout, '(A)') "This program comes with ABSOLUTELY NO WARRANTY."
    write(stdout, '(A)') "This is free software, and you are welcome to redistribute it"
    write(stdout, '(A)') "under certain conditions (see GPLv3)."
    write(stdout, '(A)') ""
    write(stdout, '(A)') tab//"============================================================"
    write(stdout, '(A)') tab//"                       NEWTON-X NS 3"
    write(stdout, '(A)') tab//"         Newtonian dynamics close to the crossing seam"
    write(stdout, '(A)') tab//"                       www.newtonx.org"
    write(stdout, '(A)') tab//"============================================================"
    write(stdout, '(A)') ""
  end subroutine nxinp_greeter


  function nxinp_select_main() result(res)
    !! Main menu for the input generator.
    !!
    !! Currently the following options are implemented:
    !!
    !! - basic input generation (basics in ``nxconfig`` section) ;
    !! - trajectories generation (finds initial conditions from ``initial_condition`` file).
    !!
    !! Return the option number chosen, and keeps on until a valid option has been chosen.
    integer :: res

    character(len=256) :: options(3)

    options(1) = "Generate a basic input"
    ! options(2) = "Set up general parameters (more dynamics control, QM, ZPE, ...)"
    options(2) = "Set up a non-adiabatic dynamics"
    options(3) = "Create trajectories"

    res = fmenu(options, with_exit=.true., exit_message='Save configuration and exit')
  end function nxinp_select_main


  function nxinp_select_nad_options() result(res)
    !! Sub-menu for non-adiabatic dynamics.
    !!
    !! This menu contains entries for:
    !!
    !! 1. Setting up SH algorithm and TDSE integration ;
    !! 2. Defining which non-adiabatic derivatives will be computed ;
    !! 3. Defining how to compute the non-adiabatic derivatives (``cioverlap``, auxiliary
    !! method, ...)
    integer :: res
    character(len=256) :: options(3)

    options(1) = "Set TDSE integration method and Surface Hopping algorithm"
    options(2) = "Define which non-adiabatic couplings (NAC) will be computed"
    options(3) = "Define how to compute time-derivative couplings"

    res = fmenu(options, with_exit=.true., exit_message='Return to main menu')
  end function nxinp_select_nad_options


  function nxinp_select_general() result(res)
    !! Sub-menu for general dynamics parameters
    !!
    integer :: res

    character(len=256) :: options(4)

    options(1) = "Set general dynamics parameters"
    options(2) = "Set electronic structure computation parameters"
    options(3) = "Set ZPE corrections"
    options(4) = "Set Adaptive time-step (EXPERIMENTAL)"

    res = fmenu(options, with_exit=.true., exit_message='Return to main menu')
  end function nxinp_select_general


  function nxinp_init_parser() result(parser)
    !! Initialize a parser for input generation.
    !!
    type(parser_t) :: parser

    character(len=:), allocatable :: init_file
    logical :: ext, found_init_file, parse_file
    integer :: u, ierr

    init_file = ''
    found_init_file = .false.
    inquire(file='user_config.nml', exist=ext)
    if (ext) then
       init_file = 'user_config.nml'
       found_init_file = .true.
    else
       inquire(file='basic.nml', exist=ext)
       if (ext) then
          init_file = 'basic.nml'
          found_init_file = .true.
       end if
    end if

    parse_file = .false.
    if (found_init_file) then
       parse_file = &
            & ask_y_or_n('I found '//trim(init_file)//' in working directory: read it')
       if (parse_file) then
          call parser%init(init_file)
          call parser%parse()
       end if

       ierr = copy(trim(init_file), trim(init_file)//'.bck')
    end if

    if (.not. parse_file) then
       open(newunit=u, file='user_config.nml')
       call parser%init('user_config.nml')
       close(u)
    end if
  end function nxinp_init_parser


  subroutine nxinp_basic_configuration(parser)
    !! Basic configuration for a Newton-X input.
    !!
    !! This routine will populate ``parser`` with information coming from interactive
    !! input from the user.  The following parameters can be set:
    !!
    !! - ``nat``: Number of atoms (can be derived from ``geom.orig`` or ``initial_condition``
    !! if one of those files is present) ;
    !! - ``nstat``: Number of states in the dynamics ;
    !! - ``nstatdyn``: Starting state for the dynamics ;
    !! - ``progname`` and ``methodname``: QM Program and method to be used ;
    !! - ``dt``: Time-step (in fs) ;
    !! - ``tmax``: Maximum duration of the dynamics (in fs).
    !!
    !! The routine does not print the parser !
    type(parser_t) :: parser

    integer :: dum, u, nat, nstat, nstatdyn
    character(len=:), allocatable :: dumc
    real(dp) :: dumr
    type(pair_t) :: options
    type(nx_config_t) :: config
    character(len=:), allocatable :: prog, filename, val, progname, methodname
    logical :: ext, found_geom, parse_file, has_nat

    character(len=*), parameter :: section = 'nxconfig'

    call options%set('0.1', '(Built-in) Spin-Boson Hamiltonian')
    call options%set('0.2', '(Built-in) Recoherence Model Hamiltonian')
    call options%set('0.3', '(Built-in) 1D models (including models from Tully et al.)')
    call options%set('0.4', '(Built-in) Conical intersection (from Granucci et al.)')
    call options%set('0.5', '(Built-in) Analytical CS-FSSH (Kossosski and Barbatti)')
    call options%set('1.1', 'Columbus with MCSCF gradients')
    call options%set('1.2', 'Columbus with MRCI gradients')
    call options%set('2.1', 'Turbomole TDDFT')
    call options%set('2.2', 'Turbomole RICC2')
    call options%set('2.3', 'Turbomole ADC2')
    call options%set('3.1', 'Gaussian TDDFT (G16)')
    call options%set('4.1', 'Orca TDDFT')
    call options%set('6.1', 'OpenMolcas RASSCF')
    call options%set('6.2', 'OpenMolcas CASPT2')
    call options%set('6.3', 'OpenMolcas CMS-PDFT')
    call options%set('10.1', 'Mopac FOMO-CI')
    call options%set('11.1', 'Exciton model with Mopac (FOMO-CI)')
    call options%set('11.2', 'Exciton model with Gaussian (TDDFT)')
    call options%set('12.1', 'Tinker-Gaussian MMPol')
    call options%set('12.2', 'Tinker-MNDO QM/MM')
    call options%set('13.1', 'External')

    nat = -1
    found_geom = .false.
    inquire(file='geom.orig', exist=ext)
    if (ext) then
       filename = 'geom.orig'
       found_geom = .true.
    else
       inquire(file='initial_condition', exist=ext)
       if (ext) then
          filename = 'initial_condition'
          found_geom = .true.
       end if
    end if

    if (found_geom) then
       nat = nxinp_count_atoms(filename)
       if (nat < 0) then
          write(stdout, '(A)') 'Number of atoms found from '//trim(filename)
       end if
    else
       val = parser%get(section, 'nat')
       if (val /= 'UNDEFINED_ELEMENT') then
          read(val, *) nat
          write(stdout, '(A)') 'Number of atoms found configuration'
       else
          write(stdout, '(A)') 'Number of atoms still undefined !'
       end if
    end if

    ! Check if the initial configuration has nstat defined
    nstat = -1
    dumc = parser%get(section, 'nstat')
    if (dumc /= 'UNDEFINED_ELEMENT') then
       read(dumc, *) nstat
    end if

    ! Check if the initial configuration has nstatdyn defined
    nstatdyn = -1
    dumc = parser%get(section, 'nstatdyn')
    if (dumc /= 'UNDEFINED_ELEMENT') then
       read(dumc, *) nstatdyn
    end if

    has_nat = .true.
    if (nat < 0) has_nat = .false.
    dum = ask('nat', 'Number of atoms in the system', nat, has_def=has_nat)
    call parser%set(section, 'nat', to_str(dum))

    if (nstat > 0) then
       dum = ask('nstat', 'Number of states to include', nstat)
    else
       dum = ask('nstat', 'Number of states to include', -1, has_def=.false.)
    end if
    call parser%set(section, 'nstat', to_str(dum))

    if (nstatdyn < 0) then
       dum = ask('nstatdyn', 'State on which the dynamics will start', dum)
    else
       dum = ask('nstatdyn', 'State on which the dynamics will start', nstatdyn)
    end if
    call parser%set(section, 'nstatdyn', to_str(dum))

    prog = ask('progname / methodname', 'QM program and method to use', '', &
         & options, has_def=.false.)
    call prog_key_to_str(prog, progname, methodname)
    call parser%set(section, 'progname', progname)
    call parser%set(section, 'methodname', methodname)

    ! At this point we should have enough data to generate ``configuration``.
    config = nx_config_t()
    call config%init(parser)

    dumr = ask('dt', 'Time step', config%dt/fs2au, unit='fs')
    call parser%set(section, 'dt', to_str(dumr))
    dumr = ask('tmax', 'Maximum duration of the simulation', config%tmax, unit='fs')
    call parser%set(section, 'tmax', to_str(dumr))

    ! call options%clean()
    ! call options%set('0', 'Do not compute (or read) the couplings (for adiabatic dynamics for instance)')
    ! call options%set('1', 'Read non-adiabatic coupling vectors directly from the QM code')
    ! call options%set('2', 'Read a state-overlap matrix (and compute it if required)')
    ! call options%set('3', 'Auxiliary method (Baeck An for instance)')
    ! dumc = ask('dc_method', &
    !      & 'Define how to obtain the derivative couplings for integrating the TDSE', &
    !      & to_str(config%dc_method), options=options)

    call options%clean()
    call options%set('1', 'Original algorithm from Tully')
    call options%set('2', 'Local diabatization algorithm from Persico and Granucci')
    ! call options%set('2', 'Read a state-overlap matrix (and compute it if required)')
    ! call options%set('3', 'Auxiliary method (Baeck An for instance)')
    dumc = ask('dc_method', &
         & 'Which type of surface hopping do you want to perform ?', &
         & '1', options=options)

    if (dumc == '2') then
       config%dc_method = 2
       config%use_locdiab = .true.
    else if (dumc == '1') then
       if (ask_y_or_n('Do you want to use approximated Baeck-An couplings')) then
          config%dc_method = 3
       else
          config%dc_method = 1
       end if
    end if
    call parser%set(section, 'dc_method', to_str(config%dc_method))
    call parser%set(section, 'use_locdiab', to_str(config%use_locdiab))

    open(newunit=u, file='user_config.nml', action='write')
    write(u, '(a)') write_basic_input(parser)
    close(u)

    write(stdout, '(A)')&
         & 'The  basic configuration is now finished.'//nl
    write(stdout, '(A)') 'Press any key to go back to the General Option menu.'//nl
    read(stdin, *)
  end subroutine nxinp_basic_configuration


  subroutine nxinp_set_tdse_parameters(parser)
    !! Set parameters for TDSE integration.
    !!
    !! This routine sets the parameters for the ``sh`` section of the input (object of
    !! type ``nx_sh_t``).  The following parameters are set in the ``sh`` section of
    !! ``parser`` in this routine:
    !!
    !! - ``integrator``
    !! - ``ms``
    !! - ``getphase``
    !! - ``nohop``
    !! - ``nrelax``
    !! -  ``seed``
    !! - ``probmin``
    !! - ``popdev``
    !! - ``tully``
    !! - ``decay``
    !! - ``mom``
    !! - ``adjmom``
    !! - ``if adjmom == 2 -> adjtheta``
    type(parser_t), intent(inout) :: parser
    !! Current configuration.

    type(nx_sh_t) :: sh
    type(nx_config_t) :: config
    integer :: dum
    real(dp) :: dumr
    character(len=:), allocatable :: dumc
    type(pair_t) :: options

    character(len=*), parameter :: section = 'sh'

    config = nx_config_t()
    call config%init(parser)
    call sh%init(parser, config%dc_method, config%use_locdiab)

    call options%set('0', 'Finite element method')
    call options%set('1', 'Butcher, 5th order')
    call options%set('2', 'Unitary propagators for local-diabatization method')
    call options%set('3', 'Unitary propagators (Experimental)')
    dumc = ask('integrator',&
         & 'Integrator type for the time-dependent Schrödinger equation', &
         & to_str(sh%integrator), options=options)
    call parser%set(section, 'integrator', trim(dumc))

    dum = ask('ms', &
         & 'Number of sub-timesteps for integration of the time-dependent Schrödinger equa&
         &tion'//nl//&
         & ' - For local-diabatization (integrator = 2), this value will be used'//nl//&
         & ' - as a dummy value (the number of integration step is determined automaticall&
         &y).', sh%ms)
    call parser%set(section, 'ms', to_str(dum))

    call options%clean()
    call options%set('0', 'use phase provided by the overlap of CI vectors (NOT IMPLEMENTE&
         &D).')
    call options%set('1', 'use phase provided by the scalar product between h(t) and h(t-dt).')
    dumc = ask('getphase', 'Phase to use', to_str(sh%getphase), options=options)
    call parser%set(section, 'getphase', trim(dumc))

    dum = ask('nohop',&
         & 'Force the hopping at a certain time step'//nl//&
         & '- 0 - Normal surface hopping'//nl//&
         & '- -1 - Hopping is not allowed at any time'//nl//&
         & '- n - Hopping is forced at (and only at) step $n$ ($n$ = positive integer)', &
         & sh%nohop)
    call parser%set(section, 'nohop', to_str(dum))

    dum = ask('nrelax', &
         & 'Number of substeps after hopping, in which other hopping is forbiden', &
         & sh%nrelax)
    call parser%set(section, 'nrelax', to_str(dum))

    dum = ask('seed', &
         & 'Random number generation'//nl//&
         & '- -1 - A randomized seed is used'//nl// &
         & '-  0 - A default random-number seed is used'//nl// &
         & '- superior to 0 - Use this number as a random seed', sh%seed)
    call parser%set(section, 'seed', to_str(dum))

    dumr = ask('probmin', 'Do not hop if probability is smaller than =probmin=', &
         & sh%probmin)
    call parser%set(section, 'probmin', to_str(dumr))

    dumr = ask('popdev', &
         & 'Kill trajectory if total adiabatic population deviate more '//&
         & 'than =popdev= from the unity', sh%popdev)
    call parser%set(section, 'popdev', to_str(dumr))

    call options%clean()
    call options%set('0', 'Tully')
    call options%set('1', 'Hammes-Schiffer and Tully')
    dumc = ask('tully', 'Fewest-switch algorithm', to_str(sh%tully), options=options)
    call parser%set(section, 'tully', trim(dumc))

    dumr = ask('decay', '(EDC model) Decay time.', sh%decay)
    call parser%set(section, 'decay', to_str(dumr))

    call options%clean()
    call options%set('-1', 'Invert momentum direction')
    call options%set('1', 'Keep momentum direction')
    dumc = ask('mom', 'What to do after a frustrated hopping', &
         & to_str(sh%mom), options=options)
    call parser%set(section, 'mom', trim(dumc))

    call options%clean()
    call options%set('0', '(Large systems, NAC not available) Rescale in '//&
         & 'the direction of velocity, and scale kinetic enregy in the hopping '//&
         & 'condition with the number of degrees of freedom.')
    call options%set('1', '(NAC not available) Rescale in the direction of velocity,'//&
         & ' and don''t rescale the kinetic energy in the hopping condition.')
    call options%set('2', '(NAC available) Rescale in a direction given by the angle ``adj&
         &theta`` in the plane $(h, g)$.')
    dumc = ask('adjmom', &
         & 'Condition for accepting a hop, and direction of velocity '//&
         & 'rescaling (see manual for more details).', &
         & to_str(sh%adjmom), options=options)
    call parser%set(section, 'adjmom', trim(dumc))

    if (dumc == '2') then
       dumr = ask('adjtheta', &
            & 'Direction (in degrees) in which the velocity is rescaled after '//&
            & 'a hopping (for ``adjmom = 2`` only).'//nl//&
            & '- 0 - adjust along non adiabatic vector $h$'//nl//&
            & '- 90 - adjust along gradient vector $g$'//nl//&
            & '- n - adjust along the given ``n`` angle in the plane $(h, g)$', &
            & sh%adjtheta)
       call parser%set(section, 'adjtheta', trim(to_str(dumr)))
    end if

  end subroutine nxinp_set_tdse_parameters


  subroutine nxinp_set_which_nad(parser)
    !! Set parameters related to NAD computation.
    !!
    !! This routine sets parameters defining which NAD are computed during the
    !! computation. The following parameters are set in the ``nad_setup`` section of
    !! ``parser``:
    !!
    !! - ``kross``
    !! - ``cascade``
    !! - ``current``
    !! - ``never_state``
    !! - ``include_pair``
    !!
    type(parser_t), intent(inout) :: parser
    !! Current configuration.

    type(nx_nad_t) :: nad
    type(nx_config_t) :: config
    type(pair_t) :: options
    integer :: dum
    character(len=:), allocatable :: dumc
    character(len=*), parameter :: section = 'nad_setup'
    logical :: keep_on

    ! Generate main configuration from parser
    config = nx_config_t()
    call config%init(parser)

    ! Create NAD object from parser and configuration
    nad = nx_nad_t()
    call nad%init(parser, config)

    call options%set('0', &
         & 'Do not calculate nonadiabatic couplings between non-consecutive states')
    call options%set('1', &
         & 'Calculate nonadiabatic couplings between non-consecutive states')
    dumc = ask('kross', 'Control the computation of NAC between non-consecutive states', &
         & to_str(nad%kross), options=options)
    call parser%set(section, 'kross', trim(dumc))

    call options%clean()
    call options%set('0', 'Compute NAC above AND below')
    call options%set('1', 'Compute NAC below only')
    dumc = ask('cascade', &
         & 'Control the computation of NAC between state of interest and &
         &  states above / below',&
         & to_str(nad%cascade), options=options)
    call parser%set(section, 'cascade', trim(dumc))

    call options%clean()
    call options%set('0', 'Deactivate this screening')
    call options%set('1', 'Activate')
    dumc = ask('current', &
         & 'Compute NAC only for pairs of states including the current state.',&
         & to_str(nad%current), options=options)
    call parser%set(section, 'current', trim(dumc))

    dumc = ask('never_state', &
         & 'Array of states for which the nonadiabatic coupling vectors '//&
         & 'should never be computed (COMMA-SEPARATED array)'//nl//&
         & '- 1,2 will never compute couplings involving S0 or S1', &
         & to_str(nad%never_state), &
         & should_be='array of int')
    call parser%set(section, 'never_state', trim(dumc))

    dumc = ask('include_pair', &
         & 'Pairs of state for which the NACs should always be computed '//&
         & 'if current state is part of this pair (COMMA-SEPARATED array).'//nl//&
         & '- 1,2,1,3 will always compute NAC S0-S1 and S0-S2 '//&
         & '(when the system is in S0)', &
         & to_str(nad%include_pair), should_be='array of int')
    call parser%set(section, 'include_pair', trim(dumc))

  end subroutine nxinp_set_which_nad


  subroutine nxinp_set_how_nad(parser)
    !! Set how to compute NAD.
    !!
    !! This routine sets how to compute the NAD during the dynamics, by looking at the
    !! value of ``dc_method`` from a ``nx_config_t`` initialized from ``parser``.  It
    !! then decides wether to set the section ``cioverlap`` with  either
    !! ``[[nxinp_how_nad_cioverlap]]`` or ``[[nxinp_how_nad_cioverlap_od]]``, or the
    !! section ``aux_nad``.
    type(parser_t), intent(inout) :: parser
    !! Current configuration.

    type(nx_config_t) :: config
    type(nx_cioverlap_t) :: cio
    type(pair_t) :: options
    type(nx_auxnac_t) :: auxnac
    integer :: dum, ovl_prog
    logical :: go_on
    real(dp) :: dumr
    character(len=:), allocatable :: dumc
    character(len=*), parameter :: ciosection = 'cioverlap'
    character(len=*), parameter :: auxsection = 'aux_nad'

    config = nx_config_t()
    call config%init(parser)

    if (config%dc_method == 1) then
       write(stdout, '(A)') &
            & "Reading from QM program requested: nothing else to do here."//nl
       write(stdout, '(A)') &
            & "The Time Derivative couplings configuration is now finished."
       write(stdout, '(A)') &
            & "Press any key to go back to the General Option menu."
       read(stdin, *)
       return

    else if (config%dc_method == 2) then
       call cio%init(config, parser)

       go_on = .false.

       do while (.not. go_on)
          call options%set('1', 'cioverlap (based on Tully and Hammes-Schiffer)')
          call options%set('2', 'cioverlap-od (orbital derivative method)')
          dumc = ask('ovl_prog', &
               & 'Program to use for computing the state overlap matrix',&
               & to_str(cio%ovl_prog), options=options)
          call parser%set(ciosection, 'ovl_prog', trim(dumc))

          read(dumc, *) dum
          if (dum == 2) then
             ! ovl_prog = cioverlap-od should only be used with single-ref methods
             if (config%progname == 'columbus') then
                write(stdout, '(A)') &
                     & '!! cioverlap-od is not compatible with Columbus !!'
             else
                go_on = .true.
             end if

          else
             go_on = .true.
          end if
       end do

       ! Reinit to take into account the change in ovl_prog
       call cio%init(config, parser)

       ! CIOVERLAP CONFIGURATION
       if (cio%ovl_prog == 1) then
          call nxinp_how_nad_cioverlap(parser, config, cio)
       else if (cio%ovl_prog == 2) then
          call nxinp_how_nad_cioverlap_od(parser, config, cio)
       end if

    else if (config%dc_method == 3) then
       call nxinp_how_nad_auxnad(parser)
    end if
  end subroutine nxinp_set_how_nad


  subroutine nxinp_how_nad_cioverlap(parser, config, cio)
    !! Set NAD computing through cioverlap.
    !!
    !! This routine populates the ``cioverlap`` section of ``parser``, with the following
    !! parameters:
    !!
    !! - ``cio_options``
    !! - ``blasthread``
    !! - For Columbus: ``ci_cons``
    !! - For single reference QM methods (Gaussian, Orca, Turbomole, ...) :
    !! ``cisc_options``, ``ncore``, ``ndisc``
    !! - For Gaussian: ``coptda``
    type(parser_t), intent(inout) :: parser
    !! Current configuration.
    type(nx_config_t), intent(in) :: config
    !! ``nx_config_t`` obtained from ``parser``.
    type(nx_cioverlap_t), intent(in) :: cio
    !! ``nx_cioverlap_t`` obtained from ``parser``.

    integer :: dum
    character(len=:), allocatable :: dumc
    type(pair_t) :: options

    character(len=*), parameter :: section = 'cioverlap'

    dumc = ask('cio_options', 'Options to be passed to cioverlap program',&
         & trim(cio%cio_options))
    call parser%set(section, 'cio_options', trim(dumc))

    dum = ask('blasthread', &
         & 'Number of CPU cores used by BLAS in cioverlap programs.', &
         & cio%blasthread)
    call parser%set(section, 'blasthread', to_str(dum))

    if (config%progname == 'columbus') then
       call options%clean()
       call options%set('0', 'Compute all determinant overlap terms')
       call options%set('1', &
            & 'Neglect determinant overlap terms when the rank is too high '//&
            & 'or when the determinants are orthogonal')
       dumc = ask('ci_cons', &
            & "COLUMBUS: Computation of determinant overlap in the "//&
            & "time-derivative coupling.", to_str(cio%ci_cons), options=options)
       call parser%set(section, 'ci_cons', to_str(dum))
    end if

    ! CIS-CASIDA OPTIONS
    if (config%progname == 'gaussian' &
         & .or. config%progname == 'turbomole' &
         & .or. config%progname == 'orca') then

       dumc = ask('cisc_options', 'Options to be passed to cis_casida program', &
            & trim(cio%cisc_options))
       call parser%set(section, 'cisc_options', trim(dumc))

       dum = ask('ncore', &
            & "CIS Casida: Number of core orbitals that should be ignored.", &
            & cio%ncore)
       call parser%set(section, 'ncore', to_str(dum))

       dum = ask('ndisc', &
            & "CIS Casida: Number of virtual orbitals that should be ignored.", &
            & cio%ndisc)
       call parser%set(section, 'ndisc', to_str(dum))
    end if

    if (config%progname == 'gaussian') then
       call options%clean()
       call options%set('0', '|X>')
       call options%set('1', '|X+Y>')
       dumc = ask('coptda', &
            & "GAUSSIAN: Linear response vectors used in the evaluation of NAD.", &
            & to_str(cio%coptda), options=options)
       call parser%set(section, 'coptda', trim(dumc))
    end if
  end subroutine nxinp_how_nad_cioverlap


  subroutine nxinp_how_nad_cioverlap_od(parser, config, cio)
    !! Set NAD computing through cioverlap-od.
    !!
    !! This routine populates the ``cioverlap`` section of ``parser``, with the following
    !! parameters:
    !!
    !! - For Gaussian: ``coptda``
    type(parser_t), intent(inout) :: parser
    !! Current configuration.
    type(nx_config_t), intent(in) :: config
    !! ``nx_config_t`` obtained from ``parser``.
    type(nx_cioverlap_t), intent(in) :: cio
    !! ``nx_cioverlap_t`` obtained from ``parser``.

    integer :: dum
    character(len=:), allocatable :: dumc
    type(pair_t) :: options

    character(len=*), parameter :: section = 'cioverlap'

    if (config%progname /= 'gaussian') then
       return
    else
       call options%set('0', '|X>')
       call options%set('1', '|X+Y>')
       dumc = ask('coptda', &
            & "GAUSSIAN: Linear response vectors used in the evaluation of NAD.", &
            & to_str(cio%coptda), options=options)
       call parser%set(section, 'coptda', trim(dumc))
    end if
  end subroutine nxinp_how_nad_cioverlap_od


  subroutine nxinp_how_nad_auxnad(parser)
    !! Set NAD computing through auxiliary method.
    !!
    !! This routine populates the ``aux_nac`` section of ``parser``, with the following
    !! parameters:
    !!
    !! - ``model``
    !! - ``ba_smooth``
    !! - ``ba_dh``
    !! - ``ba_de``
    !! - ``ba_dv``
    type(parser_t), intent(inout) :: parser

    character(len=*), parameter :: section = 'auxnac'

    type(pair_t) :: options
    type(nx_auxnac_t) :: auxnac
    character(len=:), allocatable :: dumc
    real(dp) :: dumr

    call auxnac%init(parser, 3)

    call options%clean()
    call options%set('0', 'Baeck An model')
    dumc = ask('model', 'Model to use', to_str(auxnac%model), options=options)
    call parser%set(section, 'model', trim(dumc))

    call options%clean()
    call options%set('0', 'Use backward approximation')
    call options%set('1', 'Use quadratic regression')
    dumc = ask('ba_smooth', &
         & 'Method for estimating the second time derivative of the energy.',&
         & to_str(auxnac%ba_smooth), options=options)
    call parser%set(section, 'ba_smooth', trim(dumc))

    dumr = ask('ba_dh', &
         & 'If the variation of the couplings between time-step exceeds '//&
         & '``ba_dh``, keep the old values.', &
         & auxnac%ba_dh)
    call parser%set(section, 'ba_dh', to_str(dumr))

    dumr = ask('ba_de', &
         & 'If the energy gap exceeds ``ba_de``, the couplings are set to 0.', &
         & auxnac%ba_de)
    call parser%set(section, 'ba_de', to_str(dumr))

    dumr = ask('ba_dv', &
         & 'If the product energy_gap x second_derivatives is higher '//&
         & 'than ``ba_dv``, set the couplings to 0.', &
         & auxnac%ba_dv)
    call parser%set(section, 'ba_dv', to_str(dumr))
  end subroutine nxinp_how_nad_auxnad


  subroutine nxinp_set_gen_traj(config, has_generated)
    !! Set the options for generating trajectories.
    !!
    !! The routine starts by setting up all the different parameters for the trajectory
    !! generation out of files ``initial_condition`` and ``initial_condition.1.N`` by populating a
    !! section ``generate_traj`` in the parser ``config``.
    !!
    !! When that is completed, and after a validation from the user, the trajectories are
    !! effectively created in a directory ``TRAJECTORIES``.  If the directory already
    !! exists it is either deleted or backed up as ``TRAJECTORIES.bck``.
    type(parser_t), intent(inout) :: config
    !! Configuration.
    logical, intent(inout) :: has_generated
    !! Indicate if the trajectory generation has been performed (``.true.``) or not
    !! (``.false.``).

    logical :: ext

    character(len=*), parameter :: section = 'generate_traj'
    integer :: dum, nis, ierr, nat, proceed
    real(dp) :: dumr, max_intensity
    character(len=:), allocatable :: dumc, msg
    type(pair_t) :: options
    character(len=256), allocatable :: split(:)

    type(nx_gen_traj_t) :: gentraj

    inquire(file='initial_condition', exist=ext)
    if (.not. ext) then
       inquire(file='JOB_NAD', exist=ext)
       if (.not. ext) then
          write(stdout, '(A)') nl//'No initial_condition file found !!'//nl
          write(stdout, '(A)') 'Press any key to go back to the General Option menu.'//nl
          read(stdin, *)
          return
       end if
       return
    end if

    inquire(file='JOB_AD', exist=ext)
    if (.not. ext) then
       inquire(file='JOB_NAD', exist=ext)
       if (.not. ext) then
          write(stdout, '(A)') 'No JOB_AD or JOB_NAD folder found !!'
          write(stdout, '(A)') 'Press any key to go back to the General Option menu.'//nl
          read(stdin, *)
          return
       end if
    end if

    gentraj = nx_gen_traj_t()

    nis = ask('nis', 'Lowest state considered', gentraj%nis)
    call config%set(section, 'nis', to_str(nis))

    call options%set('0', 'Do not apply any restriction.')
    call options%set('1', 'Use the original energy restriction written in the initial_condition files.')
    call options%set('2', 'Apply new energy restriction.')
    dumc = ask('screen', 'Energy restriction criterion', to_str(gentraj%screen), options=options)
    call config%set(section, 'screen', dumc)

    read(dumc, *) dum
    if (dum == 2) then
       dumc = ask('e_center_ref / e_center_real', &
            & 'Center of the energy restriction'//nl//&
            & ' - x - Value of the center of restriction'//nl//&
            & ' - ref n - Use the vertical excitation of initial_condition.nis.n file.', '0.0'&
            &, unit='eV')

       if (index(dumc, 'ref') /= 0) then
          split = split_pattern(dumc)
          dumc = 'initial_condition.'//to_str(nis)//'.'//trim(split(2))
          call config%set(section, 'e_center_ref', trim(dumc))
       else
          call config%set(section, 'e_center_ref', '')
          call config%set(section, 'e_center_real', dumc)
       end if

       dumr = ask('e_var', 'Width of the energy restriction', gentraj%e_var, unit='eV')
       call config%set(section, 'e_var', to_str(dumr))
    end if

    dumc = ask('read_os_from_file / os_condon', &
         & 'Oscillator strength:'//nl//&
         & ' - -1 - Try to read from initial_condition file.'//nl//&
         & '- x - oscillator strength is always x (Condon approximation).', '-1')
    if (dumc == '-1') then
       call config%set(section, 'read_os_from_file', to_str(.true.))
    else
       call config%set(section, 'read_os_from_file', to_str(.false.))
       call config%set(section, 'os_condon', dumc)
    end if

    call options%clean()
    call options%set('local', 'Use the energy-restricted data set.')
    call options%set('global', 'Use complete data set.')
    dumc = ask('norm', ' Normalization of transition intensities.', trim(gentraj%norm), options=options)
    call config%set(section, 'norm', dumc)

    dum = ask('seed', &
         & 'Random number generation'//nl//&
         & '- -1 - A randomized seed is used'//nl// &
         & '- 0 - A default seed is used (4357)'//nl// &
         & '- superior to 0 - Use this number as a random seed', gentraj%seed)
    call config%set(section, 'seed', to_str(dum))

    call options%clean()
    call options%set('0', 'No (default)')
    call options%set('1', 'Yes')
    dumc = ask('run_is', &
         & 'Will you compute the dynamical observables for a target distribution '//nl//&
         & 'different from the sampling distribution ?', to_str(gentraj%run_is), options=options)
    read(dumc, *) dum
    call config%set(section, 'run_is', to_str(dum))

    call gentraj%init(config)
    call gentraj%set_energy_window(ierr, msg)
    if (ierr /= 0) then
       write(stdout, '(A)') trim(msg)
       read(stdin, *)
       return
    end if

    if (gentraj%screen == 2) then
       write(stdout, '(A,A,F10.3,A,F10.3,A,F10.3)') &
            & 'Energy transitions will be restricted to the range (in eV): '//nl, &
            & 'e_min = ', gentraj%window%e_min, ' to e_max = ', gentraj%window%e_max, &
            & ' with center at ', gentraj%window%e_0
    end if

    write(stdout, '(A)') ''
    write(stdout, '(A,F12.8)') 'Maximal intensity: ', gentraj%max_intensity()
    write(stdout, '(A)') ''

    write(stdout, '(A)')&
         & 'The configuration for generating trajectories is now finished.'
    write(stdout, '(A)') ''
    write(stdout, '(A)')&
         & 'The full configuration from `nx_geninp` is:'
    write(stdout, '(A)') ''
    call config%print(stdout)

    call options%clean()
    call options%set('1', 'Proceed to trajectory generation')
    call options%set('2', 'Go back to main menu')
    dumc = ask('proceed', &
         & 'After this step the configuration will be saved as "user_config.nml"'//&
         & nl//'Please review the configuration above, and press:', &
         & '1', options=options)
    read(dumc, *) proceed

    if (proceed == 1) then
       inquire(file='TRAJECTORIES', exist=ext)
       if (ext) then
          if (ask_y_or_n('I found an existing TRAJECTORIES folder, delete it')) then
             ierr = rm(['TRAJECTORIES'])
          else
             write(stdout, '(A)') 'I will backup TRAJECTORIES as TRAJECTORIES.bck'
             ierr = rename('TRAJECTORIES', 'TRAJECTORIES.bck')
          end if
       end if

       inquire(file='user_config.nml', exist=ext)
       if (ext) then
          write(stdout, '(A)') 'I will backup user_config.nml as user_config.nml.bck'
          ierr = copy('user_config.nml', 'user_config.nml.bck')
          ierr = rm(['user_config.nml'])
       end if

       open(newunit=config%fileu, file='user_config.nml', action='write')
       call config%print(config%fileu)
       close(config%fileu)
       dumc = config%get('nxconfig', 'nat')
       read(dumc, *) nat
       call gentraj%generate_dir(nat)
       has_generated = .true.

       write(stdout, '(A)')&
            & 'The generation of trajectories is now finished.'//nl
       write(stdout, '(A)') 'Press any key to go back to the General Option menu.'//nl
       read(stdin, *)

    else if (proceed == 2) then
       has_generated = .false.
       return
    end if
  end subroutine nxinp_set_gen_traj


  subroutine nxinp_set_general_dynamics(parser)
    !! Set up general parameters for the dynamics.
    !!
    !! This routine keeps on setting up the section ``nxconfig`` from the input, this
    !! time with parameters that will mostly be kept default in the majority of dynamics.
    !! The parameters set include:
    !!
    !! - ``killstat``
    !! - ``timekill``
    !! - ``lvprt``
    !! - ``save_cwd``
    !! - ``kt``
    !! - ``etot_jump``
    !! - ``etot_drift``
    type(parser_t), intent(inout) :: parser
    !! Current configuration.

    integer :: dum
    real(dp) :: dumr
    character(len=:), allocatable :: dumc
    character(len=*), parameter :: section = 'nxconfig'
    type(nx_config_t) :: config
    type(pair_t) :: options

    config = nx_config_t()
    call config%init(parser)

    dum = ask('killstat', &
         & 'Finish dynamics if after hopping the system remains more than '//&
         & ' =timekill= (see =timekill= description) fs on state killstat '//&
         & '(1 - ground state)', config%killstat)
    call parser%set(section, 'killstat', to_str(dum))

    dumr = ask('timekill', &
         & 'Finish dynamics if after hopping the system remains more than '//&
         & ' =timekill= fs on state =killstat= (see =killstat= description). '//&
         & '0 means deactivate =killstat=', config%timekill)
    call parser%set(section, 'timekill', to_str(dumr))

    call options%set('1', 'Only error messages')
    call options%set('2', 'Warning and error messages')
    call options%set('3', '(DEFAULT) Info, warning and error messages')
    call options%set('4', 'Same as 3, with supplementary messages')
    call options%set('5', 'Full debug output (warning, huge amount of text can be created ! )')
    dumc = ask('lvprt', 'Debug level: amount of output to print and output files to keep', &
         & to_str(config%lvprt), options=options)
    call parser%set(section, 'lvprt', trim(dumc))

    dum = ask('save_cwd', &
         & 'Save the working directory every ``save_cwd`` step'//nl//&
         & '- -1 - Never save the working directory (DEFAULT for =lvprt > 3=)'//nl//&
         & '-  N - Save the working directory every ``N`` step.', config%save_cwd)
    call parser%set(section, 'save_cwd', to_str(dum))

    dum = ask('kt', 'Print output at each ``kt`` steps', config%kt)
    call parser%set(section, 'kt', to_str(dum))

    dumr = ask('etot_jump', &
         & 'Kill trajectory if total energy deviate more than ``etot_jump`` '//&
         & 'in one time step', config%etot_jump * au2ev, unit='fs')
    call parser%set(section, 'etot_jump', to_str(dumr))

    dumr = ask('etot_drift',&
         & 'Kill trajectory if total energy deviate more than Etot_drift in '//&
         & 'comparison to the value in t = 0', config%etot_drift * au2ev, unit='fs')
    call parser%set(section, 'etot_drift', to_str(dumr))

    write(stdout, '(A)')&
         & 'The setup of general dynamics parameters is now finished.'//nl
    write(stdout, '(A)') 'Press any key to go back to the previous menu.'//nl
    read(stdin, *)
  end subroutine nxinp_set_general_dynamics
  

  subroutine nxinp_set_qm(parser)
    type(parser_t), intent(inout) :: parser

    type(nx_config_t) :: config
    ! type(nx_qm_t) :: qm

    config = nx_config_t()
    call config%init(parser)

    write(stdout, '(A)') 'NOT IMPLEMENTED !'
    write(stdout, '(A)') 'PLEASE SEE THE MANUAL TO SETUP QM COMPUTATION'

    ! select case (config%progname)
    ! case ('analytical')
    !    call nxinp_set_qm_analytical(parser, config)
    ! case('columbus')
    !    call nxinp_set_qm_columbus(parser, config)
    ! case('turbomole')
    !    call nxinp_set_qm_turbomole(parser, config)
    ! case('gaussian')
    !    call nxinp_set_qm_gaussian(parser, config)
    ! case('orca')
    !    call nxinp_set_qm_orca(parser, config)
    ! case('exc_mopac','exc_gaussian')
    !    call nxinp_set_qm_exc_inp(parser, config)
    !    if (config%progname == 'exc_mopac') then
    !       call nxinp_set_qm_exc_mopac(parser, config)
    !    else
    !       call nxinp_set_qm_exc_gaussian(parser, config)
    !    end if
    ! case('tinker_g16mmp')
    !    call nxinp_set_qm_tinker_g16_mndo(parser, config, 'g16')
    ! case('tinker_mndo')
    !    call nxinp_set_qm_tinker_g16_mndo(parser, config, 'mndo')
    ! end select

    write(stdout, '(a)') ''
    ! write(stdout, '(A)')&
    !      & 'The setup of QM parameters is now finished.'//nl
    write(stdout, '(A)') 'Press any key to go back to the previous menu.'//nl
    read(stdin, *)
  end subroutine nxinp_set_qm


  ! subroutine nxinp_set_qm_analytical(parser, config)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config

  !   real(dp) :: dumr
  !   integer :: dum
  !   character(len=:), allocatable :: dumc, sbh_jw
  !   type(pair_t) :: options
  !   type(nx_qm_item_t) :: qm

  !   call qm%init(config)
    
  !   select case(qm%qmmethod)
  !   case('sbh')
  !      call qm%sbh%setup(parser, parse=.false.)

  !      dumr = ask('e0', 'SBH model: epsilon_0 parameter', &
  !           & qm%sbh%e0/cm2au, unit='cm-1')
  !      call parser%set('sbh', 'e0', to_str(dumr))
       
  !      dumr = ask('v0', 'SBH model: nu_0 parameter', &
  !           & qm%sbh%v0/cm2au, unit='cm-1')
  !      call parser%set('sbh', 'v0', to_str(dumr))
       
  !      dum = ask('n', 'SBH model: Number of oscillators in the system', &
  !           & config%nat)
  !      call parser%set('sbh', 'n', to_str(dum))

  !      call options%set('user ', 'User-provided file (named user_sd_sbh.inp)')
  !      call options%set('ohmic', 'Ohmic model')
  !      call options%set('debye', 'Debye model')
  !      dumc = ask('jw', 'Type of spectral density', trim(qm%sbh%jw), options=options)
  !      call parser%set('sbh', 'jw', trim(dumc))

  !      sbh_jw = dumc
  !      if (sbh_jw == 'ohmic' .or. sbh_jw == 'debye') then
  !         dumr = ask('wc', 'SBH model: Omega_c parameter', qm%sbh%wc/cm2au, unit='cm-1')
  !         call parser%set('sbh', 'wc', to_str(dumr))

  !         dumr = ask('wc', 'SBH model: Omega_max parameter', qm%sbh%wmax/cm2au, unit='cm-1')
  !         call parser%set('sbh', 'wmax', to_str(dumr))
  !      end if

  !      if (sbh_jw == 'ohmic') then
  !         dumr = ask('xi', 'SBH model: Xi parameter', qm%sbh%xi/cm2au, unit='cm-1')
  !         call parser%set('sbh', 'xi', to_str(dumr))
  !      else if (sbh_jw == 'debye') then
  !         dumr = ask('er', 'SBH model: Er parameter', qm%sbh%er/cm2au, unit='cm-1')
  !         call parser%set('sbh', 'er', to_str(dumr))
  !      end if
  !   case('recohmodel')

  !      call qm%recohmod%setup(parser)

  !      call options%clean()
  !      call options%set('1', 'Two flat adiabats with one avoided crossing.')
  !      call options%set('2', 'Two flat adiabats with two avoided crossings.')
  !      dumc = ask('modelnum', 'Determine which model is to be run.', &
  !           & to_str(qm%recohmod%mnum), options=options)
  !      call parser%set('recohmodel', 'modelnum', trim(dumc))

  !      dumr = ask('a', 'Recoherence model: a parameter', qm%recohmod%a)
  !      call parser%set('recohmodel', 'a', to_str(dumr))
  !      dumr = ask('b', 'Recoherence model: b parameter', qm%recohmod%b)
  !      call parser%set('recohmodel', 'b', to_str(dumr))
  !      dumr = ask('c', 'Recoherence model: c parameter', qm%recohmod%c)
  !      call parser%set('recohmodel', 'c', to_str(dumr))
  !   case('onedim_model')

  !      call options%clean()
  !      call options%set('1', 'Simple avoided crossing [Tully, JCP 93, 1061 (1990)]')
  !      call options%set('2', 'Dual avoided crossing [Tully, JCP 93, 1061 (1990)]')
  !      call options%set('3', 'Extended coupling with reflection [Tully, JCP 93, 1061 (1990)]')
  !      call options%set('4', 'Double arch [Subotniki and Shenvi, JCP 134, 024105 (2011)]')
  !      call options%set('5', 'Nikitin Hamiltonian [Nikitin, Adv Chem Phys 5, 135 (2011)]')
  !      dumc = ask('onedim_mod', '1D model to be used', '1', options=options)
  !      call parser%set('onedim_model', 'onedim_mod', trim(dumc))
       
  !   case('con_int')

  !      dumr = ask('alpha', '2D conical intersection: alpha parameter', &
  !           & qm%conint%alpha, unit='a.u')
  !      call parser%set('con_int', 'alpha', to_str(dumr))
  !      dumr = ask('beta', '2D conical intersection: beta parameter', &
  !           & qm%conint%beta, unit='a.u')
  !      call parser%set('con_int', 'beta', to_str(dumr))
  !      dumr = ask('gamma', '2D conical intersection: gamma parameter', &
  !           & qm%conint%gamma, unit='a.u')
  !      call parser%set('con_int', 'gamma', to_str(dumr))
  !      dumr = ask('delta', '2D conical intersection: delta parameter', &
  !           & qm%conint%delta, unit='a.u')
  !      call parser%set('con_int', 'delta', to_str(dumr))
  !      dumr = ask('delta', '2D conical intersection: delta parameter', &
  !           & qm%conint%delta, unit='a.u')
  !      call parser%set('con_int', 'delta', to_str(dumr))
  !      dumr = ask('kx', '2D conical intersection: kx parameter', &
  !           & qm%conint%kx, unit='a.u')
  !      call parser%set('con_int', 'kx', to_str(dumr))
  !      dumr = ask('ky', '2D conical intersection: ky parameter', &
  !           & qm%conint%ky, unit='a.u')
  !      call parser%set('con_int', 'ky', to_str(dumr))

  !      dumr = ask('x1', '2D conical intersection: x1 parameter', &
  !           & qm%conint%x1, unit='a.u')
  !      call parser%set('con_int', 'x1', to_str(dumr))

  !      dumr = ask('x2', '2D conical intersection: x2 parameter', &
  !           & qm%conint%x2, unit='a.u')
  !      call parser%set('con_int', 'x2', to_str(dumr))

  !      dumr = ask('x3', '2D conical intersection: x3 parameter', &
  !           & qm%conint%x3, unit='a.u')
  !      call parser%set('con_int', 'x3', to_str(dumr))
  !   case('cs_fssh')

  !      call options%clean()
  !      call options%set('1', '2RHE: 2 resonances with 1st harmonic and 2nd exponential')
  !      call options%set('2', '2REH: 2 resonances with 1st exponential and 2nd harmonic')
  !      call options%set('3', '2REE: 2 exponential resonances')
  !      call options%set('4', '2RHH: 2 harmonic resonances')
  !      dumc = ask('cs_mod', 'Which analytical model to use', '1', options=options)
  !      call parser%set('cs_fssh', 'cs_mod', trim(dumc))
       
  !      call options%clean()
  !      call options%set('0', 'Use home-made diagonalizer')
  !      call options%set('1', '(EXPERIMENTAL) Use Lapack')
  !      dumc = ask('use_lapack', 'Which diagonalizer to use', '0', options=options)
  !      call parser%set('cs_fssh', 'use_lapack', trim(dumc))
       
  !   end select
  ! end subroutine nxinp_set_qm_analytical


  ! subroutine nxinp_set_qm_columbus(parser, config)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config

  !   integer :: dum
  !   character(len=:), allocatable :: dumc
  !   character(len=*), parameter :: section = 'columbus'
  !   type(pair_t) :: options

  !   type(nx_qm_t) :: qm

  !   call qm%init(config)

  !   call options%set('0', 'Compute only the gradient for the current state.')
  !   call options%set('1', 'Compute the gradient for all states.')
  !   dumc = ask('all_grads', 'Which gradients to compute', &
  !        & to_str(qm%col_all_grads), options=options)
  !   call parser%set(section, 'all_grads', trim(dumc))

  !   dum = ask('mem', 'Core memory', qm%col_mem, unit='MB')
  !   call parser%set(section, 'mem', trim(dumc))

  !   call options%clean()
  !   call options%set('0', 'do not use previous CI vector')
  !   call options%set('1', 'use previous CI vector')
  !   dumc = ask('cirestart', 'CI restart options.', &
  !        & to_str(qm%col_cirestart), options=options)
  !   call parser%set(section, 'cirestart', trim(dumc))

  !   dum = ask('ivmode', &
  !        & 'Initial CI-vector generation mode.'//nl//&
  !        & 'See COLUMBUS docs (=ciudg= program) for a list of options.'//nl//&
  !        & 'Cirestart keyword has priority over ivmode keyword. The default is generate'&
  !        &//nl//&
  !        & 'vectors by iterative diagonalization of the reference space hamiltonian matrix&
  !        &.', qm%col_ivmode)
  !   call parser%set(section, 'ivmode', to_str(dum))

  !   call options%clean()
  !   call options%set('0', 'Use the same mocoef file for all time steps')
  !   call options%set('1', 'Use the mocoef from the previous time step')
  !   dumc = ask('mocoef', 'Molecular orbitals usage.', &
  !        & to_str(qm%col_mocoef), options=options)
  !   call parser%set(section, 'mocoef', trim(dumc))

  !   dum = ask('prt_mo', 'Save molecular orbitals file every ``prt_mo`` step'//nl//&
  !        & '- -1 - Deactivate this option (orbitals will never be saved)'//nl//&
  !        & '-  N - Save every N steps', qm%col_prt_mo)
  !   call parser%set(section, 'prt_mo', to_str(dum))

  !   call options%clean()
  !   call options%set('0', 'keep =rtolci= and =rtolbk= in ciudgin as =1E-4= for all states')
  !   call options%set('1', 'use =citol= only for =nstatdyn=. For all other states use =10 * citol=')
  !   call options%set('2', &
  !        & 'use =citol= for =nstatdyn=, =10 * citol= for states coupled to '//&
  !        & '=nstatdyn= (as defined in =transmomin=), and =1000 * citol= for other states')
  !   dumc = ask('reduce_tol', 'Tolerance in CI calculations.', &
  !        & to_str(qm%col_reduce_tol), options=options)
  !   call parser%set(section, 'reduce_tol', trim(dumc))

  !   dum = ask('quad_conv', 'Set value of =NCOUPL= in =mcscfin=.', qm%col_quad_conv)
  !   call parser%set(section, 'quad_conv', to_str(dum))

  !   call options%clean()
  !   call options%set('0', 'warn and continue')
  !   call options%set('1', 'kill trajectory')
  !   dumc = ask('mc_conv', 'What to do if MCSCF calculation do not converge.', &
  !        & to_str(qm%col_mc_conv), options=options)
  !   call parser%set(section, 'mc_conv', trim(dumc))

  !   dumc = ask('ci_conv', 'What to do if CI calculation do not converge.', &
  !        & to_str(qm%col_ci_conv), options=options)
  !   call parser%set(section, 'ci_conv', trim(dumc))
  ! end subroutine nxinp_set_qm_columbus

  ! subroutine nxinp_set_qm_turbomole(parser, config)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config

  !   integer :: dum
  !   character(len=:), allocatable :: dumc
  !   character(len=*), parameter :: section = 'turbomole'
  !   type(pair_t) :: options

  !   type(nx_qm_t) :: qm

  !   call qm%init(config)

  !   dum = ask('nnodes', 'Number of cores to use for parallel Turbomole (smp, no mpi!)', &
  !        & qm%tm_nnodes)
  !   call parser%set(section, 'nnodes', to_str(dum))

  !   dum = ask('multiplicity',&
  !        & 'Multiplicity of the excited states given as one of the irrep options '//&
  !        & 'in control file of Turbomole.', qm%tm_mult)
  !   call parser%set(section, 'multiplicity', to_str(dum))

  !   if (qm%qmmethod /= 'tddft') then
  !      dum = ask('npre',&
  !           & 'Number of roots used in preoptimization steps given as one of the irrep options '//&
  !           & 'in control file of Turbomole.', qm%tm_npre)
  !      call parser%set(section, 'npre', to_str(dum))

  !      dum = ask('nstatdyn',&
  !           & 'Number of start vectors generated or read from file given as one of '//&
  !           & 'the irrep options in control file of Turbomole.', qm%tm_nstart)
  !      call parser%set(section, 'nstart', to_str(dum))
  !   end if
  ! end subroutine nxinp_set_qm_turbomole
  
  ! subroutine nxinp_set_qm_gaussian(parser, config)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config

  !   integer :: dum
  !   character(len=:), allocatable :: dumc
  !   character(len=*), parameter :: section = 'gaussian'
  !   type(pair_t) :: options

  !   type(nx_qm_t) :: qm

  !   call qm%init(config)

  !   call options%clean()
  !   call options%set('0', 'Compute the initial guess at every time step')
  !   call options%set('1', 'Use the mocoef from the previous time step')
  !   call options%set('2', 'Use the same checkpoint file for all time steps')
  !   dumc = ask('mocoef', 'Molecular orbitals usage.', &
  !        & to_str(qm%gau_mocoef), options=options)
  !   call parser%set(section, 'mocoef', trim(dumc))

  !   dum = ask('prt_mo', 'Save mocoef to DEBUG directory every prt_mo timesteps.', &
  !        & qm%gau_prt_mo)
  !   call parser%set(section, 'mocoef', to_str(dum))

  !   call options%clean()
  !   call options%set('0', 'Compute the excited states without previous reference')
  !   call options%set('1', &
  !        & 'Read states from the checkpoint file, what file is controlled '//&
  !        & 'by mocoef parameter')
  !   dumc = ask('td_st', 'Reading of the excited states from previous steps.', &
  !        & to_str(qm%gau_td_st), options=options)
  !   call parser%set(section, 'td_st', trim(dumc))

  !   dum = ask('ld_thr', 'Linear dependence threshold.'//nl//&
  !        & '- N - Linear dependence threshold for Gaussian to use with IOp(3/59=N)',&
  !        & qm%gau_ld_thr)
  !   call parser%set(section, 'ld_thr', to_str(dum))
  ! end subroutine nxinp_set_qm_gaussian

  ! subroutine nxinp_set_qm_orca(parser, config)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config

  !   integer :: dum
  !   character(len=:), allocatable :: dumc
  !   character(len=*), parameter :: section = 'orca'
  !   type(pair_t) :: options

  !   type(nx_qm_t) :: qm

  !   call qm%init(config)

  !   call options%clean()
  !   call options%set('0', 'Compute the initial guess at every time step')
  !   call options%set('1', 'Use the mocoef from the previous time step')
  !   call options%set('2', 'Use the same checkpoint file for all time steps')
  !   dumc = ask('mocoef', 'Molecular orbitals usage.', &
  !        & to_str(qm%gau_mocoef), options=options)
  !   call parser%set(section, 'mocoef', trim(dumc))

  !   dum = ask('prt_mo', 'Save mocoef to DEBUG directory every prt_mo timesteps.', &
  !        & qm%gau_prt_mo)
  !   call parser%set(section, 'mocoef', to_str(dum))

  !   call options%clean()
  !   call options%set('F', 'Full TD-DFT computation')
  !   call options%set('T', 'TDA computation')
  !   dumc = ask('is_tda', 'Is the dynamics done with TDA or full TD-DFT ?', &
  !        & to_str(qm%orca_is_tda), options=options)
  !   call parser%set(section, 'is_tda', trim(dumc))
  ! end subroutine nxinp_set_qm_orca


  ! subroutine nxinp_set_qm_exc_inp(parser, config)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config

  !   integer :: dum
  !   character(len=:), allocatable :: dumc
  !   character(len=*), parameter :: section = 'exc_inp'
  !   type(pair_t) :: options

  !   type(nx_qm_t) :: qm

  !   call qm%init(config)

  !   dum = ask('tresp', 'Exciton couplings using transition monopol approximation', &
  !        & qm%trespexc)
  !   call parser%set(section, 'tresp', to_str(dum))

  !   dum = ask('gs', 'Compute only the ground state informations if "gs > 0"', &
  !        & qm%gsexc)
  !   call parser%set(section, 'gs', to_str(dum))

  !   dum = ask('dip', 'Compute the transition dipole moments if "dip > 0"', &
  !        & qm%dipexc)
  !   call parser%set(section, 'dip', to_str(dum))

  !   dumc = ask('nat_array', 'Array with the number of atom bellongin to each chromophore', &
  !        & '1', should_be='array of int')
  !   call parser%set(section, 'nat_array', trim(dumc))

  !   dumc = ask('nstat', 'Array with the number of states for each chromophore', &
  !        & '1', should_be='array of int')
  !   call parser%set(section, 'nstat', trim(dumc))
  ! end subroutine nxinp_set_qm_exc_inp


  ! subroutine nxinp_set_qm_exc_mopac(parser, config)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config

  !   integer :: dum
  !   character(len=:), allocatable :: dumc
  !   character(len=*), parameter :: section = 'exc_mopac'
  !   type(pair_t) :: options

  !   type(nx_qm_t) :: qm

  !   call qm%init(config)

  !   dum = ask('nchrom', 'Number of chromophores in the system', qm%nchromexc)
  !   call parser%set(section, 'nchrom', to_str(dum))

  !   dum = ask('nproc', 'Number of CPU cores to allocate in TOTAL', qm%nprocexc)
  !   call parser%set(section, 'nproc', to_str(dum))
  ! end subroutine nxinp_set_qm_exc_mopac


  ! subroutine nxinp_set_qm_exc_gaussian(parser, config)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config

  !   integer :: dum
  !   character(len=:), allocatable :: dumc
  !   character(len=*), parameter :: section = 'exc_gaussian'
  !   type(pair_t) :: options

  !   type(nx_qm_t) :: qm

  !   call qm%init(config)

  !   dum = ask('nchrom', 'Number of chromophores in the system', qm%nchromexc)
  !   call parser%set(section, 'nchrom', to_str(dum))

  !   dum = ask('nproc', 'Number of CPU cores to allocate in TOTAL', qm%nprocexc)
  !   call parser%set(section, 'nproc', to_str(dum))

  !   dum = ask('nprocgau', 'Number of CPU cores per Gaussian job', qm%nproc_gauexc)
  !   call parser%set(section, 'nprocgau', to_str(dum))

  !   dum = ask('nstattd', 'Number of excited state to include in TDDFT', qm%nstattdexc)
  !   call parser%set(section, 'nstattd', to_str(dum))

  !   dumc = ask('functional', 'DFT functional to use', trim(qm%functionalexc))
  !   call parser%set(section, 'nstattd', trim(dumc))

  !   dumc = ask('basis', 'Basis set', trim(qm%basisexc))
  !   call parser%set(section, 'basis', trim(dumc))

  !   dumc = ask('force_field', 'Force Field (UFF or Amber or Dreiding', &
  !        & trim(qm%force_fieldexc))
  !   call parser%set(section, 'force_field', trim(dumc))
  ! end subroutine nxinp_set_qm_exc_gaussian


  ! subroutine nxinp_set_qm_tinker_g16_mndo(parser, config, type)
  !   type(parser_t), intent(inout) :: parser
  !   type(nx_config_t), intent(in) :: config
  !   character(len=*), intent(in) :: type

  !   integer :: dum
  !   character(len=:), allocatable :: dumc
  !   character(len=:), allocatable :: section
  !   type(pair_t) :: options

  !   type(nx_qm_t) :: qm

  !   character(len=:), allocatable :: def

  !   call qm%init(config)

  !   if (type == 'g16') then
  !      section = 'tinker_g16mmp'
  !      def = trim(qm%tg16mmp_interface)
  !   else if (type == 'mndo') then
  !      section = 'tinker_mndo'
  !      def = trim(qm%tmndo_interface)

  !      call options%clean()
  !      call options%set('0', 'Discard NAC on MM atoms')
  !      call options%set('1', 'Retain NAC on MM atoms')
  !      dumc = ask('nac_mm', 'For QM/MM system discard or retain NAC vector on MM atoms', &
  !           & to_str(qm%compute_nac_mm), options=options)
  !      call parser%set(section, 'nac_mm', trim(dumc))
  !   end if

  !   dumc = ask('interface_path', &
  !        & 'Path to gradinterf.x (the interface program of modified '//&
  !        & 'Tinker-g16-MMP / Tinker MNDO)', trim(def))
  !   call parser%set(section, 'interface_path', trim(dumc))
  ! end subroutine nxinp_set_qm_tinker_g16_mndo


  subroutine nxinp_set_zpe_corrections(parser)
    type(parser_t), intent(inout) :: parser

    integer :: dum
    real(dp) :: dumr
    character(len=:), allocatable :: dumc
    character(len=*), parameter :: section = 'zpe_correction'
    type(pair_t) :: options
    
    type(nx_zpe_t) :: zpe
    type(nx_config_t) :: config

    logical :: ans

    ans = ask_y_or_n('Do you want to include ZPE corrections')
    if (.not. ans) then
       write(stdout, '(A)')&
            & 'No ZPE correction will be included.'//nl
       write(stdout, '(A)') 'Press any key to go back to the previous menu.'//nl
       read(stdin, *)
       return
    end if

    call parser%set('nxconfig', 'with_zpe', '1')
    config = nx_config_t()
    call config%init(parser)

    call zpe%init(parser, config)

    call options%clean()
    call options%set('0', 'Just check the values, don''t correct anything.')
    call options%set('1', 'Use ZPE correction.')
    dumc = ask('kcheck', 'Flag to trigger the ZPE corrections.', to_str(zpe%kcheck), &
         & options=options)
    call parser%set(section, 'kcheck', trim(dumc))

    call options%clean()
    call options%set('1', 'Model')
    dumc = ask('kmodel', 'Model used for ZPE correction.', to_str(zpe%kmodel), &
         & options=options)
    call parser%set(section, 'kmodel', trim(dumc))

    dumr = ask('tcheck', 'Duration of time for checking ZPE leakage.', zpe%tcheck, &
         & unit='fs')
    call parser%set(section, 'tcheck', to_str(dumr))

    dumr = ask('tcycle', 'ZPE correction will take place after every this time.', zpe%tcycle, &
         & unit='fs')
    call parser%set(section, 'tcycle', to_str(dumr))

    dumr = ask('zthresh', 'Threshold for the energy leak parameter', zpe%zthresh, &
         & unit='Hartree')
    call parser%set(section, 'zthresh', to_str(dumr))

    write(stdout, '(A)')&
         & nl//'The configuration of ZPE corrections is now finished.'//nl
    write(stdout, '(A)') 'Press any key to go back to the previous menu.'//nl
    read(stdin, *)
  end subroutine nxinp_set_zpe_corrections


  subroutine nxinp_set_adaptive_ts(parser)
    type(parser_t), intent(inout) :: parser

    integer :: dum
    real(dp) :: dumr
    character(len=:), allocatable :: dumc
    character(len=*), parameter :: section = 'adapt_dt'
    type(pair_t) :: options

    type(nx_adaptive_ts_t) :: adt
    type(nx_config_t) :: config

    logical :: ans

    ans = ask_y_or_n('Do you want to enable adaptive time-steps')
    if (.not. ans) then
       write(stdout, '(A)')&
            & 'Adaptive time-step will remain switched off.'//nl
       write(stdout, '(A)') 'Press any key to go back to the previous menu.'//nl
       read(stdin, *)
       return
    end if

    call parser%set('nxconfig', 'with_adt', '1')
    config = nx_config_t()
    call config%init(parser)

    call adt%init(parser)

    call options%clean()
    call options%set('0', 'Use a constant value')
    call options%set('1', 'Use the value of the F-norm')
    dumc = ask('model', 'Model used for adapting the value of the time-step.', &
         & to_str(adt%model), options=options)
    call parser%set(section, 'model', trim(dumc))

    dumr = ask('ratio', &
         & 'After time-step adaptation the new time step will be ``dt * ratio``.', &
         & adt%ratio)
    call parser%set(section, 'ratio', to_str(dumr))

    dum = ask('max_subtraj', &
         & 'Maximum number of virtual trajectories that can be created.', &
         & adt%max_subtraj)
    call parser%set(section, 'max_subtraj', to_str(dum))

    write(stdout, '(A)')&
         & nl//'The configuration of adaptive time-step is now finished.'//nl
    write(stdout, '(A)') 'Press any key to go back to the previous menu.'//nl
    read(stdin, *)
  end subroutine nxinp_set_adaptive_ts
  
  

  ! ===============
  ! HELPER ROUTINES
  ! ===============
  subroutine prog_key_to_str(key, prog, method)
    !! Convert a key to a program / method pair.
    !!
    character(len=*), intent(in) :: key
    character(len=:), allocatable, intent(out) :: prog
    character(len=:), allocatable, intent(out) :: method

    character(len=2), allocatable :: split(:)
    integer :: dum

    split = split_pattern(key, pattern='.')
    read(split(1), *) dum
    select case(dum)
    case(0)
       prog = 'analytical'

       read(split(2), *) dum
       select case(dum)
       case(1)
          method = 'sbh'
       case(2)
          method = 'recohmodel'
       case(3)
          method = 'onedim_model'
       case(4)
          method = 'con_int'
       case(5)
          method = 'cs_fssh'
       end select

    case(1)
       prog = 'columbus'

       read(split(2), *) dum
       select case(dum)
       case(1)
          method = 'mcscf'
       case(2)
          method = 'mrci'
       end select

    case(2)
       prog = 'turbomole'
       read(split(2), *) dum
       select case(dum)
       case(1)
          method = 'tddft'
       case(2)
          method = 'ricc2'
       case(3)
          method = 'adc2'
       end select

    case(3)
       prog = 'gaussian'
       read(split(2), *) dum
       select case(dum)
       case(1)
          method = 'tddft'
       end select

    case(4)
       prog = 'orca'
       read(split(2), *) dum
       select case(dum)
       case(1)
          method = 'tddft'
       end select

    case(6)
      prog = 'openmolcas'
      read(split(2), *) dum
      select case(dum)
      case(1)
        method = 'rasscf'
      case(2)
        method = 'caspt2'    
      case(3)
        method = 'cspdft'
      end select

    case(10)
       prog = 'mopac'
       read(split(2), *) dum
       select case(dum)
       case(1)
          method = 'fomo-ci'
       end select

    case(11)
       read(split(2), *) dum
       select case(dum)
       case(1)
          prog = 'exc_mopac'
          method = 'mopac-fomo-ci'
       case(2)
          prog = 'exc_gaussian'
          method = 'gaussian-tddft'
       end select

    case(12)
       read(split(2), *) dum
       select case(dum)
       case(1)
          prog = 'tinker_g16mmp'
          method = 'g16mmpol'
       case(2)
          prog = 'tinker_mndo'
          method = 'mndo'
       end select

    case(13)
      prog = 'external'
      read(split(2),*) dum
      select case(dum)
      case(1)
        method = 'generic'
      end select

    end select
  end subroutine prog_key_to_str


  function nxinp_count_atoms(filename) result(res)
    !! Wrapper function to count atoms in file.
    !!
    character(len=*), intent(in) :: filename
    !! Name of the file to read.

    integer :: res

    if (filename == 'geom.orig') then
       res = nxinp_atoms_in_geom(filename)
    else if (filename == 'initial_condition' .or. filename == 'initial_condition.old') then
       res = nxinp_atoms_in_initial_condition(filename)
    end if
  end function nxinp_count_atoms


  function nxinp_atoms_in_geom(filename) result(res)
    !! Count the atoms in a file of type ``geom.orig``.
    !!
    character(len=*), intent(in) :: filename
    !! Name of the file to read.

    integer :: res
    integer :: u, ierr
    character(len=MAX_STR_SIZE) :: buf

    res = 0
    open(newunit=u, file=filename, action='read')
    do
       read(u, '(A)', iostat=ierr) buf
       if (ierr /= 0) exit
       if (buf == '') exit
       res = res + 1
    end do
    close(u)
  end function nxinp_atoms_in_geom


  function nxinp_atoms_in_initial_condition(filename) result(res)
    !! Count the atoms in a file of type ``initial_condition``.
    !!
    character(len=*), intent(in) :: filename
    !! Name of the file to read.

    integer :: res
    integer :: u, ierr
    character(len=MAX_STR_SIZE) :: buf

    res = 0
    open(newunit=u, file=filename, action='read')
    MAIN_LOOP: do
       read(u, '(A)', iostat=ierr) buf
       if (ierr /= 0) exit

       if (index(buf, 'Geometry in') /= 0) then
          do
             read(u, '(A)', iostat=ierr) buf
             if (ierr /= 0) exit
             res = res + 1
             if (index(buf, 'Velocity in') /= 0) then
                res = res - 1
                exit MAIN_LOOP
             end if
          end do
       end if
    end do MAIN_LOOP
    close(u)
  end function nxinp_atoms_in_initial_condition


  function write_basic_input(parser) result(res)
    type(parser_t), intent(in) :: parser

    character(len=:), allocatable :: res

    type(nx_config_t) :: conf
    type(nx_csfssh_params_t) :: csfssh
    type(nx_sh_t) :: sh
    type(nx_nad_t) :: nad
    type(nx_cioverlap_t) :: cio
    type(nx_auxnac_t) :: auxnac
    type(nx_adaptive_ts_t) :: adaptive_ts
    type(nx_qm_item_t), target :: qm
    class(nx_qm_generic_t), pointer :: qm_obj

    ! 1. Start with main configuration
    conf = nx_config_t()
    call conf%init(parser)
    res = conf%to_str()//nl

    ! 2. CS-FSSH if required
    if (conf%run_complex) then
       call csfssh%init(conf)
       res = res//csfssh%to_str()//nl
    end if

    ! 3. SH and NAD
    if (conf%thres > 0) then
       call sh%init(parser, conf%dc_method, conf%use_locdiab)
       res = res//sh%to_str()//nl

       nad = nx_nad_t()
       call nad%init(parser, conf)
       res = res//nad%to_str()//nl
    end if

    ! 4. CIO or AUXNAD
    if (conf%dc_method == 2) then
       call cio%init(conf, parser)
       res = res//cio%to_str()//nl
    else if (conf%dc_method == 3) then
      call auxnac%init(parser, lvprt=conf%lvprt)
       res = res//auxnac%to_str()//nl
    end if

    ! 5. Adaptive time-step
    if (conf%with_adt) then
       call adaptive_ts%init(parser)
       res = res//adaptive_ts%to_str()//nl
    end if

    ! 6. Thermostat
    if (conf%with_thermo) then
    end if

    ! 7. ZPE correction
    if (conf%with_zpe) then
    end if

    ! 8. QM
    qm = nx_qm_create_item(conf)
    ! Some QM specific parameters
    qm_obj => qm%qm
    select type(qm_obj)
    type is (nx_turbomole_t)
       call qm_obj%complete_setup(conf%nstat)
    end select
    res = res//qm%qm%to_str()//nl
  end function write_basic_input
  
end module mod_nx_geninp
