! Copyright (C) 2022  Light and Molecules Group

! This program is free software: you can redistribute it and/or modify
! it under the terms of the GNU General Public License as published by
! the Free Software Foundation, either version 3 of the License, or
! (at your option) any later version.

! This program is distributed in the hope that it will be useful,
! but WITHOUT ANY WARRANTY; without even the implied warranty of
! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
! GNU General Public License for more details.

! You should have received a copy of the GNU General Public License
! along with this program.  If not, see <https://www.gnu.org/licenses/>.

module mod_constants
  !! author: Baptiste Demoulin <baptiste.demoulin@univ-amu.fr>
  !! date: 2020-03-27
  !!
  !! Module for storing constants used across the program.
  !!
  !! The constants stored can be of several type:
  !!
  !! - conversion factors ;
  !! - general status of the trajectory, error codes ;
  !! - specification for atoms ;
  !! - common size parameters.
  use mod_kinds, only: dp
  implicit none

  private
  public :: pi, BK, proton, au2fs, au2ev, au2kcalm, au2ang,&
       & deg2rad, au2cm, cm2au, h_planck, c_speed,&
       & light, bk_ev, h_evs, e_charge, e_mass,&
       & eps0, cm2K, ev2nm, fs2au, au2debye, &
       & hbar

  public :: MAX_CMD_SIZE, MAX_STR_SIZE

  public :: convert_status_to_str

  ! Mathematical constants
  real(dp), parameter :: pi = 4*atan(1.0_dp)

  ! DOUBLE PRECISION CONSTANTS AND CONVERSION COEFFICIENTS:
  ! Values obtained from: https://physics.nist.gov/cuu/Constants/Table/allascii.txt
  ! Fundamental constants
  real(dp), parameter :: light    = 299792458_dp
  !! speed of light m/s
  real(dp), parameter :: bk_ev    = 8.617333262E-5_dp
  !! Boltzmann constant eV/kelvin
  real(dp), parameter :: e_mass   = 9.1093837139E-31_dp
  !! electron mass kg
  real(dp), parameter :: e_rel_m  = 5.485799090441E-4_dp
  !! electron mass in u
  real(dp), parameter :: h_evs    = 4.135667696E-15_dp
  !! Planck constant eV.s
  real(dp), parameter :: e_charge = -1.602176634E-19_dp
  !! electron charge C
  real(dp), parameter :: eps0     = 8.8541878188E-12_dp
  !! vacuum permitivity \(C^2*s^2*kg^{-1}*m^{-3}\)
  real(dp), parameter :: mol      = 6.02214076E23_dp
  !! Avogadro constant
  
  ! Conversion factors
  real(dp), parameter :: au2ev    = 27.211386245981_dp
  !! a.u. to eV conversion.
  real(dp), parameter :: au2fs    = 2.4188843265864E-2_dp
  !! a.u. to femtoseconds conversion.
  real(dp), parameter :: fs2au    = 1.0_dp / au2fs
  !! Femtoseconds to a.u. conversion.
  real(dp), parameter :: au2ang   = 5.29177210544E-1_dp
  !! a.u. to angstrom conversion.
  real(dp), parameter :: deg2rad  = pi / 180.0_dp
  !! degree to radian (\(\pi\)/180) conversion.
  real(dp), parameter :: cm2au    = 4.5563352529132E-6_dp
  !! cm\(^{-1}\) to hartree conversion.
  real(dp), parameter :: au2cm    = 1.0_dp / cm2au
  !! hartree to cm\(^{-1}\) conversion.
  real(dp), parameter :: cm2K     = 1.438776877_dp
  !! cm-1 to Kelvin conversion.
  real(dp), parameter :: ev2nm    = au2cm / au2ev * 1.0E-7_dp
  !! eV to nm conversion.
  real(dp), parameter :: au2debye = 2.541747760_dp ! not in NIST
  !! a.u. to Debye conversion
  real(dp), parameter :: cal2j    = 4.184_dp
  !! calorie to Joule conversion.
  real(dp), parameter :: au2j     = -au2ev * e_charge
  !! a.u. to Joule conversion.
  real(dp), parameter :: au2kcalm = au2j / cal2j * mol * 1E-3_dp
  !! a.u. to kcal/mol conversion.

  ! Derived constants
  real(dp), parameter :: c_speed  = light * 1.0E-13_dp
  !! speed of light (cm/fs)
  real(dp), parameter :: proton   = 1.0_dp / e_rel_m
  !! (a.m.u.)/(electron mass).
  real(dp), parameter :: BK       = bk_ev / au2ev
  !! Botzmann constant Hartree/kelvin
  real(dp), parameter :: h_planck = h_evs / au2ev
  !! Planck constant in Hartree.sec.

  real(dp), parameter :: hbar = 1.0_dp

  ! CONSTANTS USED IN THE CODE
  integer, parameter :: MAX_STR_SIZE = 1024 ! String-size limit
  integer, parameter :: MAX_CMD_SIZE = 1024 ! Maximum length for command line

  ! ERROR CHECKS
  integer, parameter, public :: ERR_ETOT_JUMP = 90
  integer, parameter, public :: ERR_ETOT_DRIFT = 91

  ! TRAJECTORY STATUS
  integer, parameter, public :: TRAJ_NORMAL_TERM = 1
  integer, parameter, public :: TRAJ_KILLED_EDRIFT = 2
  integer, parameter, public :: TRAJ_KILLED_ETOT0 = 3
  integer, parameter, public :: TRAJ_KILLED_KILLSTAT = 4
  integer, parameter, public :: TRAJ_KILLED_EDIFF = 5

  character(len=2), parameter, public :: ATOM_NAMES(94) = [&
       & "H ", "He", "Li", "Be", "B ", "C ", "N ", "O ", "F ", "Ne", &
       & "Na", "Mg", "Al", "Si", "P ", "S ", "Cl", "Ar", "K ", "Ca", &
       & "Sc", "Ti", "V ", "Cr", "Mn", "Fe", "Co", "Ni", "Cu", "Zn", &
       & "Ga", "Ge", "As", "Se", "Br", "Kr", "Rb", "Sr", "Y ", "Zr", &
       & "Nb", "Mo", "Tc", "Ru", "Rh", "Pd", "Ag", "Cd", "In", "Sn", &
       & "Sb", "Te", "I ", "Xe", "Cs", "Ba", "La", "Ce", "Pr", "Nd", &
       & "Pm", "Sm", "Eu", "Gd", "Tb", "Dy", "Ho", "Er", "Tm", "Yb", &
       & "Lu", "Hf", "Ta", "W ", "Re", "Os", "Ir", "Pt", "Au", "Hg", &
       & "Tl", "Pb", "Bi", "Po", "At", "Rn", "Fr", "Ra", "Ac", "Th", &
       & "Pa", "U ", "Np", "Pu" &
  &]


contains

  pure function convert_status_to_str(status) result(msg)
    !! Convert the status code to a message.
    !!
    !! This function is used to write the termination status of the trajectory in the
    !! H5MD file at the end of the dynamics.
    integer, intent(in) :: status
    !! Status code.
    character(len=:), allocatable :: msg

    select case(status)
    case(TRAJ_NORMAL_TERM)
       msg = "FINISHED"
    case(TRAJ_KILLED_EDRIFT)
       msg = "KILLED (EDRIFT)"
    case(TRAJ_KILLED_ETOT0)
       msg = "KILLED (ETOT0)"
    case(TRAJ_KILLED_KILLSTAT)
       msg = "KILLED (KILLSTAT)"
    case(TRAJ_KILLED_EDIFF)
       msg = "KILLED (EDIFF)"
    end select

  end function convert_status_to_str


end module mod_constants
