#!/usr/bin/env python3

import socket
import os
import json
import time
import random
import argparse
import fcntl
import threading
from single_point import set_model, predict

def start_server(host='127.0.0.1', retries=5):
    # Create a socket object
    server_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    for retry in range(retries):
        print(f'Attempt {retry} to create new server')
        try:
            # Bind to an available port
            server_socket.bind((host, 0))  # Bind to the specified host and port
            port = server_socket.getsockname()[1]

            # Write port to the `port_id` file
            with open('port_id', 'w') as f:
                f.write(str(port))

            # Signal readiness
            print(f"Server started on {host}:{port}")

            return server_socket, port
            
        except OSError as e:
            print(f"Failed to bind or start server: {e}. Retrying...")
            time.sleep(random.uniform(0.5, 1.5))

    raise RuntimeError("Failed to start server after multiple attempts.")


def stop(host='127.0.0.1', port = None):
    if port is None:
        with open('port_id', 'r') as f:
            port = int(f.read().strip())

    print(f"Request to stop the server. Shutting down.")
    command = {"command": "shutdown"}
    with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as killer_socket:
        killer_socket.connect((host, port))
        killer_socket.sendall(json.dumps(command).encode())

    return


def handle_client(client_socket, model):
    try:
        # Receive data from the client
        data = client_socket.recv(1024)
        if not data:
            return
        
        if data.decode() == "PING":
            client_socket.sendall(b"PONG")
            return

        request = json.loads(data.decode())

        # Handle commands or predict (mock logic here)
        if request.get("command") == "shutdown":
            response = {"status": "shutting down"}
            client_socket.sendall(json.dumps(response).encode())
            try:
                os.remove('port_id')
            except OSError:
                pass
            print("Shutdown command received. Exiting server.")
            exit()

        # Example response logic
        folder_path = request.get('folder_path')
        curr_state = request.get('curr_state')
        n_states = request.get('n_states')
        response = predict(model, folder_path, n_states, curr_state)

        # Send a response back to the client
        client_socket.sendall(json.dumps(response).encode())
    except Exception as e:
        print(f"Error handling client: {e}")


def server_manager(lock_file, host='127.0.0.1', retries=5, timeout=300):
    model = set_model()

    lock = open(lock_file, 'w')
    try:
        # Acquire the lock
        fcntl.flock(lock, fcntl.LOCK_EX)
        
        # Start the server and release the lock
        server_socket, port = start_server(host = host, retries = retries)
    
    finally:
        # Release the lock file immediately after startup
        fcntl.flock(lock, fcntl.LOCK_UN)
        lock.close()

    last_request_time = time.time()

    def shutdown_if_inactive():
        """Shut down the server if no requests have been received for a timeout period."""
        while True:
            time.sleep(5)  # Check every 5 seconds
            if time.time() - last_request_time > timeout:
                print(f"No requests received for {timeout} seconds. Shutting down.")
                stop(port = port)


    # Start the inactivity timer thread
    timer_thread = threading.Thread(target=shutdown_if_inactive, daemon=True)
    timer_thread.start()

    # Enter the server loop
    print(f"Server is now running on port {port}.")
    with server_socket:

        server_socket.listen()
        while True:
            client_socket, client_address = server_socket.accept()
            with client_socket:
                handle_client(client_socket, model)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Start or manage the server.")
    parser.add_argument("--lock_file", default="/tmp/server_lock", help="Path to lock file for synchronization.")
    parser.add_argument("--stop", action='store_true', help="Stop the server.")
    args = parser.parse_args()

    # Use a lock file to ensure only one instance starts at a time
    if args.stop:
        stop()

    else:
        server_manager(args.lock_file)