#!/usr/bin/env python3

import socket
import json
import sys
import os


def is_server_running(port, host='127.0.0.1'):
    try:
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as client_socket:
            client_socket.settimeout(1)  # Timeout for quick checks
            client_socket.connect((host, port))
            # Send a lightweight ping message
            client_socket.sendall(b'PING')
            # Expect a simple acknowledgment or no response
            client_socket.recv(1024)
        return True
    except (ConnectionRefusedError, socket.timeout):
        return False


def start_new_server(server_path = '../SERVER/',
                     sp_scripts = ['single_point.py'],
                     server_script='server.py'):
    """Start a new server and return its port number."""
    import shutil
    import subprocess
    import time

    # Start the server as a subprocess
    if not os.path.exists(server_path):
        os.mkdir(server_path)
    shutil.copy(server_script, server_path)
    for sp_script in sp_scripts:
        shutil.copy(sp_script, server_path)

    process = subprocess.Popen(
        [f"./{server_script}"],
        cwd=server_path,
        start_new_session=True,
        text=True
    )

    for _ in range(100):
        if os.path.exists(f'{server_path}/port_id'):
            return
        
        time.sleep(1)

    raise RuntimeError("Failed to start the server.")


def connect_to_server(port, n_states=2, curr_state=1, host='127.0.0.1', shutdown=False):
    """Connect to the server and send a request."""
    request = {
        'folder_path': os.path.abspath('./'),
        'n_states': n_states,
        'curr_state': curr_state
    }

    if shutdown:
        request = {"command": "shutdown"}

    with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as client_socket:
        # Connect to the server
        client_socket.connect((host, port))
        # Send data to the server
        client_socket.sendall(json.dumps(request).encode())
        # Receive a response from the server
        data = client_socket.recv(1024)
        response = json.loads(data.decode())
        print(f"Received from server: {response}")

        if not response.get("status") == "success":
            command = {"command": "shutdown"}
            client_socket.sendall(json.dumps(command).encode())

    return response


if __name__ == "__main__":
    # Parse arguments
    if len(sys.argv) == 1:
        n_states = 2
        curr_state = 1
    else:
        n_states = int(sys.argv[2])
        curr_state = int(sys.argv[3]) 
    server_path = '../SERVER'
    server_script = 'server.py'
    sp_scripts = ['single_point.py', 'mndokw']

    # Check if the `port_id` file exists
    if os.path.exists(f'{server_path}/port_id'):
        with open(f'{server_path}/port_id', 'r') as f:
            port = int(f.read().strip())
    else:
        port = None

    # Check if the server is running
    if port is None or not is_server_running(port):
        print("No running server detected. Starting a new server...")
        start_new_server(server_path = server_path,
                         server_script = server_script,
                         sp_scripts=sp_scripts)
        with open(f'{server_path}/port_id', 'r') as f:
            port = int(f.read().strip())

    # Connect to the server
    connect_to_server(port=port, n_states=n_states, curr_state=curr_state)
