//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "transfer_logger.h"

#include <cstdio>
#include <filesystem>
#include <string>

#include <gmock/gmock.h>
#include <gtest/gtest.h>

#include "data_types.h"
#include "error_codes.h"

using namespace mcl;
using ::testing::ContainsRegex;
using ::testing::ElementsAreArray;
using ::testing::MatchesRegex;
using ::testing::internal::CaptureStderr;
using ::testing::internal::GetCapturedStderr;

class LoggerTest : public ::testing::Test {
  protected:
    std::string file_name = "MCL_test_log";
    std::vector<int> tags = {1, 2, 3, 4, 5};
    std::vector<int> ids = {9, 8, 7, 10, 0};
    
    std::string ref_chars = "Transfer logger test!";
    std::vector<int> ref_ints = {5, 2, 1, 4, 9};
    std::vector<long int> ref_longints = {0};
    std::vector<float> ref_floats = {0.1, 0.2};
    std::vector<double> ref_doubles = {4.3, 2.312, 5.23};
};


TEST_F(LoggerTest, Write) {
    if (std::filesystem::exists(file_name))
        std::remove(file_name.c_str());

    TransferLogger *logger = new TransferLogger(file_name, WRITE);
    int size;

    size = static_cast<int>(ref_chars.size());
    ASSERT_EQ(logger->Write(ref_chars.data(), size, kTypeChar, tags[0], ids[0]), kSuccess);

    size = static_cast<int>(ref_ints.size());
    ASSERT_EQ(logger->Write(ref_ints.data(), size, kTypeInt, tags[1], ids[1]), kSuccess);
    
    size = static_cast<int>(ref_longints.size());
    ASSERT_EQ(logger->Write(ref_longints.data(), size, kTypeLongInt, tags[2], ids[2]), kSuccess);
    
    size = static_cast<int>(ref_floats.size());
    ASSERT_EQ(logger->Write(ref_floats.data(), size, kTypeFloat, tags[3], ids[3]), kSuccess);

    size = static_cast<int>(ref_doubles.size());
    ASSERT_EQ(logger->Write(ref_doubles.data(), size, kTypeDouble, tags[4], ids[4]), kSuccess);

    delete logger;
}

TEST_F(LoggerTest, Read) {
    TransferLogger *logger = new TransferLogger(file_name, READ);
    int size;
    
    size = static_cast<int>(ref_chars.size());
    std::string chars(size, ' ');
    ASSERT_EQ(logger->Read(chars.data(), size, kTypeChar, tags[0], ids[0]), kSuccess);
    ASSERT_THAT(chars, MatchesRegex(ref_chars));
    

    size = static_cast<int>(ref_ints.size());
    std::vector<int> ints(size);
    ASSERT_EQ(logger->Read(ints.data(), size, kTypeInt, tags[1], ids[1]), kSuccess);
    ASSERT_THAT(ints, ElementsAreArray(ref_ints));
    
    size = static_cast<int>(ref_longints.size());
    std::vector<long int> longints(size);
    ASSERT_EQ(logger->Read(longints.data(), size, kTypeLongInt, tags[2], ids[2]), kSuccess);
    ASSERT_THAT(longints, ElementsAreArray(ref_longints));
    
    size = static_cast<int>(ref_floats.size());
    std::vector<float> floats(size);
    ASSERT_EQ(logger->Read(floats.data(), size, kTypeFloat, tags[3], ids[3]), kSuccess);
    ASSERT_THAT(floats, ElementsAreArray(ref_floats));

    size = static_cast<int>(ref_doubles.size());
    std::vector<double> doubles(size);
    ASSERT_EQ(logger->Read(doubles.data(), size, kTypeDouble, tags[4], ids[4]), kSuccess);
    ASSERT_THAT(doubles, ElementsAreArray(ref_doubles));

    delete logger;
}

TEST_F(LoggerTest, ReadMismatch) {
    TransferLogger *logger = new TransferLogger(file_name, READ);
    int size;

    size = static_cast<int>(ref_ints.size())-1;
    std::vector<int> ints(size);
    CaptureStderr();
    ASSERT_EQ(logger->Read(ints.data(), size, kTypeInt, tags[0], ids[0]), kErrLogMismatch);
    std::string error = GetCapturedStderr();
    ASSERT_THAT(error, ContainsRegex("~~~ MCL ~ ERROR ~~~"));
    ASSERT_THAT(error, ContainsRegex("Mismatch between the call and logged data!"));
    
    delete logger;
    std::remove(file_name.c_str());
}

int main(int argc, char** argv) {
    ::testing::InitGoogleMock(&argc, argv);
    auto result = RUN_ALL_TESTS();
    return result;
}
