//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "error_print.h"

#include <csignal>
#include <string>

#include <gmock/gmock.h>
#include <gtest/gtest.h>

#include "config.h"

using namespace mcl;
using ::testing::StrEq;
using ::testing::ContainsRegex;
using ::testing::internal::CaptureStderr;
using ::testing::internal::GetCapturedStderr;

const std::string kIndent("  ");
const int  kLineWidth = 85;

TEST(ErrorPrinter, AddQuotes) {
    std::string name = "1234";
    ASSERT_THAT(AddQuotes(name), StrEq('\"'+ name + '\"'));
    ASSERT_THAT(AddQuotes(name.c_str()), StrEq('\"'+ name + '\"'));
}

TEST(ErrorPrinter, PrintWarning) {
    std::string ref =
        "~~~ MCL ~ WARNING ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
        "  There is something fishy going on,\n"
        "  so here's a warning!\n"
        "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n";

    CaptureStderr();
    PrintWarning("There is something fishy going on,\nso here's a warning!");
    std::string error = GetCapturedStderr();
    
    ASSERT_THAT(error, StrEq(ref));
}

TEST(ErrorPrinter, PrintError) {
    std::string ref =
        "~~~ MCL ~ ERROR ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
        "  There is something fishy going on,\n"
        "  so here's an error!\n"
        "   > Additional MPI error code: 25\n"
        "   > Called in \"error_print_test.cpp\" on the line no. 68\n"
        "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n";

    CaptureStderr();
    PrintErrorMessage("There is something fishy going on,\nso here's an error!", __FILE__, __LINE__, 25);
    std::string error = GetCapturedStderr();
    
    ASSERT_THAT(error, StrEq(ref));
}

int main(int argc, char** argv) {
    ::testing::InitGoogleTest(&argc, argv);
    auto result = RUN_ALL_TESTS();
    return result;
}
