//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "env_controls.h"

#include <stdlib.h>

#include <cstdlib>
#include <stdexcept>

#include <gmock/gmock.h>
#include <gtest/gtest.h>

#include "test_macros.h"

using namespace mcl;
using ::testing::ContainsRegex;
using ::testing::internal::CaptureStderr;
using ::testing::internal::GetCapturedStderr;

std::string err_header = "~~~ MCL ~ ERROR ~~~";
std::string err_not_assigned = "The required environment variable, \"ENV_VAR_NOT_DEFINED\" is not assigned!";
std::string err_invalid = "The environment variable, \"ENV_VAR\" is present but invalid!";
std::string err_invalid_bool = "Must be one of the following: ON/OFF, TRUE/FALSE, 0/1";
std::string err_invalid_int = "Must be an integer.";


TEST(String, Assigned) {
    ADD_ENV_VAR("ENV_VAR", "abcd");
    EnvVariableString var = EnvVariableString("ENV_VAR", "default");
    ASSERT_TRUE(var.value() == "abcd");
    ASSERT_TRUE(var.default_value() == "default");
    ASSERT_TRUE(var.found());
    ASSERT_STREQ(var.name(), "ENV_VAR");
    ASSERT_TRUE(var.IsDefined());
}

TEST(String, NotAssigned) {
    EnvVariableString var = EnvVariableString("ENV_VAR_NOT_DEFINED", "default");
    ASSERT_TRUE(var.value() == var.default_value());
    ASSERT_TRUE(!var.found());
    ASSERT_STREQ(var.name(), "ENV_VAR_NOT_DEFINED");
    CaptureStderr();
    ASSERT_TRUE(!var.IsDefined());
    std::string error = GetCapturedStderr();
    ASSERT_THAT(error, ContainsRegex(err_header));
    ASSERT_THAT(error, ContainsRegex(err_not_assigned));
}


TEST(Boolean, Valid) {
    ADD_ENV_VAR("ENV_VAR", "ON");
    EnvVariableBool var = EnvVariableBool("ENV_VAR", false);
    ASSERT_TRUE(var.value());
    ASSERT_TRUE(!var.default_value());
    ASSERT_TRUE(var.found());
    ASSERT_TRUE(var.IsDefined());
}

TEST(Boolean, True) {
    EnvVariableBool var = EnvVariableBool("ENV_VAR_TRUE", false);
    ASSERT_TRUE(!var.value());
    ASSERT_TRUE(!var.default_value());
    ADD_ENV_VAR("ENV_VAR_TRUE", "ON");
    ASSERT_TRUE(var.value());
    ADD_ENV_VAR("ENV_VAR_TRUE", "TRUE");
    ASSERT_TRUE(var.value());
    ADD_ENV_VAR("ENV_VAR_TRUE", "1");
    ASSERT_TRUE(var.value());
}

TEST(Boolean, False) {
    EnvVariableBool var = EnvVariableBool("ENV_VAR_FALSE", true);
    ASSERT_TRUE(var.value());
    ASSERT_TRUE(var.default_value());
    ADD_ENV_VAR("ENV_VAR_FALSE", "OFF");
    ASSERT_TRUE(!var.value());
    ADD_ENV_VAR("ENV_VAR_FALSE", "FALSE");
    ASSERT_TRUE(!var.value());
    ADD_ENV_VAR("ENV_VAR_FALSE", "0");
    ASSERT_TRUE(!var.value());
}

TEST(Boolean, NotAssigned) {
    EnvVariableBool var = EnvVariableBool("ENV_VAR_NOT_DEFINED", false);
    ASSERT_TRUE(var.value() == var.default_value());
    ASSERT_TRUE(!var.found());
    ASSERT_STREQ(var.name(), "ENV_VAR_NOT_DEFINED");
    CaptureStderr();
    ASSERT_TRUE(!var.IsDefined());
    std::string error = GetCapturedStderr();
    ASSERT_THAT(error, ContainsRegex(err_header));
    ASSERT_THAT(error, ContainsRegex(err_not_assigned));
}

TEST(Boolean, Invalid) {
    ADD_ENV_VAR("ENV_VAR", "abcd");
    EnvVariableBool var = EnvVariableBool("ENV_VAR", false);
    ASSERT_TRUE(var.value() == var.default_value());
    ASSERT_TRUE(var.found());
    CaptureStderr();
    ASSERT_TRUE(!var.IsDefined());
    std::string error = GetCapturedStderr();
    ASSERT_THAT(error, ContainsRegex(err_header));
    ASSERT_THAT(error, ContainsRegex(err_invalid));
    ASSERT_THAT(error, ContainsRegex(err_invalid_bool));
}


TEST(Integer, Valid) {
    ADD_ENV_VAR("ENV_VAR", "23");
    EnvVariableInt var = EnvVariableInt("ENV_VAR", 32);
    ASSERT_EQ(var.value(), 23);
    ASSERT_EQ(var.default_value(), 32);
    ASSERT_TRUE(var.found());
    ASSERT_TRUE(var.IsDefined());
    
    ADD_ENV_VAR("ENV_VAR", "-1238");
    ASSERT_EQ(var.value(), -1238);
    ASSERT_TRUE(var.IsDefined());
    ADD_ENV_VAR("ENV_VAR", "-0");
    ASSERT_EQ(var.value(), 0);
    ASSERT_TRUE(var.IsDefined());
}

TEST(Integer, NotAssigned) {
    EnvVariableInt var = EnvVariableInt("ENV_VAR_NOT_DEFINED", 32);
    ASSERT_EQ(var.value(), 32);
    ASSERT_TRUE(!var.found());
    CaptureStderr();
    ASSERT_TRUE(!var.IsDefined());
    std::string error = GetCapturedStderr();
    ASSERT_THAT(error, ContainsRegex(err_header));
    ASSERT_THAT(error, ContainsRegex(err_not_assigned));
}
    
void CheckInvalidInteger(EnvVariableInt &var) {
    CaptureStderr();
    ASSERT_TRUE(!var.IsDefined());
    std::string error = GetCapturedStderr();
    ASSERT_THAT(error, ContainsRegex(err_header));
    ASSERT_THAT(error, ContainsRegex(err_invalid));
    ASSERT_THAT(error, ContainsRegex(err_invalid_int));
}

TEST(Integer, Invalid) {
    EnvVariableInt var = EnvVariableInt("ENV_VAR", 32);
    ASSERT_EQ(var.default_value(), 32);
    
    ADD_ENV_VAR("ENV_VAR", "abcd");
    ASSERT_EQ(var.value(), var.default_value());
    CheckInvalidInteger(var);

    ADD_ENV_VAR("ENV_VAR", "abc-d");
    ASSERT_EQ(var.value(), var.default_value());
    CheckInvalidInteger(var);

    ADD_ENV_VAR("ENV_VAR", "-");
    ASSERT_EQ(var.value(), var.default_value());
    CheckInvalidInteger(var);

    ADD_ENV_VAR("ENV_VAR", "1.238");
    ASSERT_EQ(var.value(), var.default_value());
    CheckInvalidInteger(var);
}


int main(int argc, char** argv) {
    ::testing::InitGoogleTest(&argc, argv);
    auto result = RUN_ALL_TESTS();
    return result;
}
