//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "mcl.h"

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include "mpi.h"


void ASSERT_TRUE(bool a) { if (!a) abort(); }
void ASSERT_EQ_ARRAYS(double * a, double * b, int len) {
    for (int i = 0; i < len; ++i) {
        ASSERT_TRUE(a[i] == b[i]);
    }
}

int APICTest() {
    MPI_Comm local_comm = MPI_COMM_WORLD;
    MCL_Initialize(&local_comm);
    
    int local_rank, local_size;
    MPI_Comm_rank(local_comm, &local_rank); 
    MPI_Comm_size(local_comm, &local_size);

    int program_id = MCL_GetProgramID();
    int num_programs = MCL_GetNumPrograms();
    ASSERT_TRUE(num_programs == 3);

    int ref_num_atoms[2] = {3, 2};
    double ref_energy[2] = {0.410, 2.56};
    double ref_coords[2][9] = {{ 1.0,  2.0,  3.0,  4.0,  5.0,  6.0,  7.0,  8.0,  9.0 },
                               {-1.0, -2.0, -3.0, -4.0, -5.0, -6.0,  0.0,  0.0,  0.0 }};

    // server
    if (program_id == 0) {
        ASSERT_TRUE(local_size == 3);
        int request;

        // MCL_SEND_CLIENT_ID
        for (int i = 1; i <= 2; ++i) {
            request = MCL_SEND_CLIENT_ID;
            MCL_Send(&request, 1, MCL_INT, MCL_REQUEST, i);
            int client_id = -1;
            MCL_Receive(&client_id, 1, MCL_INT, MCL_DATA, i);
            ASSERT_TRUE((client_id == i) == (local_rank == 0));
            MPI_Bcast(&client_id, 1, MPI_INT, 0, local_comm);
            ASSERT_TRUE(client_id == i);
        }

        int num_atoms[2] = {-1, -1};
        // MCL_SEND_NUM_PARTICLES
        for (int i = 1; i <= 2; ++i) {
            request = MCL_SEND_NUM_PARTICLES;
            MCL_Send(&request, 1, MCL_INT, MCL_REQUEST, i);
            MCL_Receive(&(num_atoms[i-1]), 1, MCL_INT, MCL_DATA, i);
            ASSERT_TRUE((num_atoms[i-1] == ref_num_atoms[i-1]) == (local_rank == 0));
            MPI_Bcast(&(num_atoms[i-1]), 1, MPI_INT, 0, local_comm);
            ASSERT_TRUE(num_atoms[i-1] == ref_num_atoms[i-1]);
        }

        // MCL_SEND_PARTICLE_POSITIONS
        for (int i = 1; i <= 2; ++i) {
            request = MCL_SEND_PARTICLE_POSITIONS;
            MCL_Send(&request, 1, MCL_INT, MCL_REQUEST, i);
            MCL_Send(ref_coords[i-1], 3*num_atoms[i-1], MCL_DOUBLE, MCL_DATA, i);
        }

        // MCL_COMPUTE_ENERGY
        for (int i = 1; i <= 2; ++i) {
            request = MCL_COMPUTE_ENERGY;
            MCL_Send(&request, 1, MCL_INT, MCL_REQUEST, i);
        }

        // MCL_SEND_ENERGY
        for (int i = 1; i <= 2; ++i) {
            request = MCL_SEND_ENERGY;
            MCL_Send(&request, 1, MCL_INT, MCL_REQUEST, i);
            double energy = 0.0;
            MCL_Receive(&energy, 1, MCL_DOUBLE, MCL_DATA, i);
            ASSERT_TRUE((energy == ref_energy[i-1]) == (local_rank == 0));
            MPI_Bcast(&energy, 1, MPI_DOUBLE, 0, local_comm);
            ASSERT_TRUE(energy == ref_energy[i-1]);
        }

        // MCL_EXIT
        request = MCL_EXIT;
        MCL_Send(&request, 1, MCL_INT, MCL_REQUEST, 1);
        MCL_Send(&request, 1, MCL_INT, MCL_REQUEST, 2);
    }

    // clients
    if (program_id == 1 || program_id == 2) {
        ASSERT_TRUE(local_size == 2);
        
        int num_atoms = ref_num_atoms[program_id-1];

        bool computed_energy = false;
        bool terminate = false;
        while (!terminate) {
            int request = -1;
            MCL_Receive(&request, 1, MCL_INT, MCL_REQUEST, 0);
            MPI_Bcast(&request, 1, MPI_INT, 0, local_comm);

            if (request == MCL_SEND_CLIENT_ID) {
                MCL_Send(&program_id, 1, MCL_INT, MCL_DATA, 0);
            } else if (request == MCL_SEND_NUM_PARTICLES) {
                MCL_Send(&num_atoms, 1, MCL_INT, MCL_DATA, 0);
            } else if (request == MCL_SEND_PARTICLE_POSITIONS) {
                double coords[9] = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
                MCL_Receive(coords, 3*num_atoms, MCL_DOUBLE, MCL_DATA, 0);
                MPI_Bcast(coords, 3*num_atoms, MPI_DOUBLE, 0, local_comm);
                ASSERT_EQ_ARRAYS(coords, ref_coords[program_id-1], 3*num_atoms);
            } else if (request == MCL_COMPUTE_ENERGY) {
                computed_energy = true;
            } else if (request == MCL_SEND_ENERGY) {
                ASSERT_TRUE(computed_energy);
                double energy = ref_energy[program_id - 1];
                MCL_Send(&energy, 1, MCL_DOUBLE, MCL_DATA, 0);
            } else if (request == MCL_EXIT) {
                terminate = true;
            } else {
                fprintf(stderr, "ABORT: Unknown MCL request!");
                abort();
            }
        }
    }

    MCL_Finalize();
    return 0;
}

int main(int argc, char** argv) {
    MPI_Init(&argc, &argv);
    
    int world_rank;
    MPI_Comm_rank(MPI_COMM_WORLD, &world_rank);

    int result = APICTest();
    
    MPI_Finalize();
    return result;
}
