//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "stub_transport.h"

#include <cstring>
#include <filesystem>
#include <string>

#include "mpi.h"

#include "data_types.h"
#include "env_controls.h"
#include "error_codes.h"
#include "error_print.h"
#include "transfer_logger.h"

namespace mcl {

int StubTransport::Initialize(void *args) {
    int world_rank = 0;
    MPI_Comm_rank(MPI_COMM_WORLD, &world_rank);

    num_programs_ = 1;

    program_id_ = env_program_number.value();
    if (!env_program_number.IsDefined()) return kErrInvalidEnvVar;
    std::string file_name = env_test_data_file.value();
    if (!env_test_data_file.IsDefined()) return kErrInvalidEnvVar;

    is_server_ = false;
    is_client_ = false;
    if (world_rank == 0) {
        if (!std::filesystem::exists(file_name)) {
            PrintErrorMessage("File " + AddQuotes(file_name) + " doesn't exist!", 
                               __FILE__, __LINE__);
            return kErrFileMissing;
        }

        transfer_log_ = new TransferLogger(file_name, READ);
        transfer_log_->Read(&skipped_send_, 1, kTypeChar, kSkipSendTag, kSkipSendTag);
        is_server_ = program_id_ == 0;
        is_client_ = !is_server_;
    }
    
    return kSuccess;
}

int StubTransport::Finalize() {
    if (is_server_ || is_client_)
        delete transfer_log_;
    return kSuccess;
}

int StubTransport::Abort(int error_code) {
    return MPI_Abort(MPI_COMM_WORLD, error_code);
}

int StubTransport::Send(void *data, int length, int data_type, int tag, int destination) {
    if (skipped_send_) return kSuccess;
    int num_bytes = length * TypeSizeInBytes(data_type);
    char *ref_data = new char[num_bytes];
    int result = transfer_log_->Read(reinterpret_cast<void*>(ref_data), length, data_type, tag, destination);
    delete[] ref_data;
    return result;
}

int StubTransport::Receive(void *data, int length, int data_type, int tag, int source) {
    return transfer_log_->Read(data, length, data_type, tag, source);
}

int StubTransport::ProbeSize(int data_type, int source) {
    int size;
    int result = transfer_log_->Read(reinterpret_cast<void*>(&size), 1, data_type, kProbeTag, source);
    if (result == kErrLogMismatch) return result;
    return size;
}

} // namespace mcl
