//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef MCL_TRANSFERLOGGER_H_
#define MCL_TRANSFERLOGGER_H_

#include <fstream>
#include <string>

namespace mcl {

/// Possible I/O modes for a logger object.
enum LoggerIoMode {
  READ,
  WRITE
};

constexpr int kProbeTag = -1;     ///< Tag for storing the probe data.
constexpr int kSkipSendTag = -2;  ///< Tag with the information about type of recording.

/// Class for writing and reading binary files that are used for storing data recorded during
/// the data transfers. It is used by the StubTransport class to perform integration tests
/// of MCL interfaces in external programs.
///   \note Binary files are stored in the little-endian representation.
///   \note Constructor assumes the READ file exists.
class TransferLogger {

  public:
    /// Open file constructor.
    ///   \param[in]  file_name  Name of the log file.
    ///   \param[in]  io_mode    Mode in which the file is opened.
    TransferLogger(const std::string &file_name, LoggerIoMode io_mode);
    ~TransferLogger();

    /// Logs the transfer to a file.
    ///   \param[in]  data       Pointer to the array containing data to be written.
    ///   \param[in]  length     Number of elements in the data array.
    ///   \param[in]  data_type  Type of the data.
    ///   \param[in]  tag        Message tag used in the message.
    ///   \param[in]  id         ID of the program to/from which the data is transfered.
    ///   \return Error code.
    int Write(void *data, int length, int data_type, int tag, int id);

    /// Loads a transfer data from a file.
    ///   \param[out] data       Pointer to the array to serve as buffer for the data to be read.
    ///   \param[in]  length     Number of elements in the data array.
    ///   \param[in]  data_type  Type of the data.
    ///   \param[in]  tag        Message tag used in the message.
    ///   \param[in]  id         ID of the program to/from which the data is transfered.
    ///   \return Error code.
    int Read(void *data, int length, int data_type, int tag, int id);


  private:
    std::fstream file_;     ///< File in which the logged data is (to be) stored.
    LoggerIoMode io_mode_;  ///< Mode in which the file is open.

};

} // namespace mcl

#endif // MCL_TRANSFERLOGGER_H_
