//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "transfer_logger.h"

#include <fstream>
#include <string>

#include "data_types.h"
#include "error_codes.h"
#include "error_print.h"

namespace mcl {

TransferLogger::TransferLogger(const std::string &file_name, LoggerIoMode io_mode) {
    std::fstream::openmode open_mode = std::fstream::binary;
    switch (io_mode) {
      case READ:
        open_mode |= std::fstream::in;
        break;
      case WRITE:
        open_mode |= std::fstream::out | std::fstream::trunc;
        break;
    }

    file_.open(file_name, open_mode);
}

TransferLogger::~TransferLogger() {
    file_.close();
}

int TransferLogger::Write(void *data, int length, int data_type, int tag, int id) {
    file_.write(reinterpret_cast<char*>(&id), sizeof(int));
    file_.write(reinterpret_cast<char*>(&tag), sizeof(int));
    file_.write(reinterpret_cast<char*>(&data_type), sizeof(int));
    file_.write(reinterpret_cast<char*>(&length), sizeof(int));
    file_.write(reinterpret_cast<char*>(data), length*TypeSizeInBytes(data_type));
    return kSuccess;
}

int TransferLogger::Read(void *data, int length, int data_type, int tag, int id) {
    int ref_length, ref_data_type, ref_tag, ref_id;
    file_.read(reinterpret_cast<char*>(&ref_id), sizeof(int));
    file_.read(reinterpret_cast<char*>(&ref_tag), sizeof(int));
    file_.read(reinterpret_cast<char*>(&ref_data_type), sizeof(int));
    file_.read(reinterpret_cast<char*>(&ref_length), sizeof(int));
    
    if ((ref_length != length) || (ref_data_type != data_type) || (ref_tag != tag) || (ref_id != id)) {
        PrintErrorMessage("Mismatch between the call and logged data!", __FILE__, __LINE__);
        return kErrLogMismatch;
    }

    file_.read(reinterpret_cast<char*>(data), length*TypeSizeInBytes(data_type));
    return kSuccess;
}

} // namespace mcl
