//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef MCL_MCL_H_
#define MCL_MCL_H_

#include "requests.h"

#ifndef __cplusplus
#include <stdbool.h>
#endif

/// External C and C++ API of the library
///   \file  mcl.h
///   \note  All API calls are blocking.

/// Initializes communicator. For the MPI-based modes of communication, it should
/// be called immediately after MPI_Init.
///   \param[in,out] args  Arguments depend on the chosen transport type.
///   \return Error code.
///   \note   All calls are blocking.
#ifdef __cplusplus
extern "C"
#endif
int MCL_Initialize(void *args);

/// Destroys the endpoint.
///   \return Error code.
#ifdef __cplusplus
extern "C"
#endif
int MCL_Finalize();

/// Terminates all processes.
///   \param[in]  error_code  Error code to return.
///   \return Error code.
#ifdef __cplusplus
extern "C"
#endif
int MCL_Abort(int error_code);

/// Returns the version number of the library.
///   \param[out] version  Will contain the API version in the order: major, minor, patch.
#ifdef __cplusplus
extern "C"
#endif
void MCL_GetAPIVersion(int version[3]);

/// Returns the number of running programs.
#ifdef __cplusplus
extern "C"
#endif
int MCL_GetNumPrograms();

/// Returns the program's ID from MCL.
#ifdef __cplusplus
extern "C"
#endif
int MCL_GetProgramID();

/// Returns the initialization status of MCL.
#ifdef __cplusplus
extern "C"
#endif
bool MCL_IsInitialized();

/// Sends data to the specified program.
///   \param[in]  data         Pointer to the array containing data to be sent.
///   \param[in]  length       Number of elements in the data array.
///   \param[in]  data_type    Type of the data to be sent.
///   \param[in]  tag          Message tag used for matching.
///   \param[in]  destination  ID of the receiving program.
///   \return Error code.
///   \note   All calls are blocking.
///   \note   Transfers occurs only between rank-0 processes.
#ifdef __cplusplus
extern "C"
#endif
int MCL_Send(void *data, int length, int data_type, int tag, int destination);

#ifdef __cplusplus
int MCL_Send(char *data, int length, int tag, int destination);
int MCL_Send(int *data, int length, int tag, int destination);
int MCL_Send(long int *data, int length, int tag, int destination);
int MCL_Send(float *data, int length, int tag, int destination);
int MCL_Send(double *data, int length, int tag, int destination);
#endif

/// Receives data from the specified program.
///   \param[out] data       Pointer to the array to serve as buffer for the data to be received.
///   \param[in]  length     Number of elements in the data array.
///   \param[in]  data_type  Type of the data to be received.
///   \param[in]  tag        Message tag used for matching.
///   \param[in]  source     ID of the sending program.
///   \return Error code.
///   \note   All calls are blocking.
///   \note   Transfers occurs only between rank-0 processes.
#ifdef __cplusplus
extern "C"
#endif
int MCL_Receive(void *data, int length, int data_type, int tag, int source);
#ifdef __cplusplus
int MCL_Receive(char *data, int length, int tag, int source);
int MCL_Receive(int *data, int length, int tag, int source);
int MCL_Receive(long int *data, int length, int tag, int source);
int MCL_Receive(float *data, int length, int tag, int source);
int MCL_Receive(double *data, int length, int tag, int source);
#endif

/// Tags for data type implemented in the framework.
extern const int MCL_CHAR;
extern const int MCL_INT;
extern const int MCL_LONG_INT;
extern const int MCL_FLOAT;
extern const int MCL_DOUBLE;

#endif // MCL_MCL_H_
