//    MCL: MiMiC Communication Library
//    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
//
//    This file is part of MCL.
//
//    MCL is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as
//    published by the Free Software Foundation, either version 3 of
//    the License, or (at your option) any later version.
//
//    MCL is distributed in the hope that it will be useful, but
//    WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "endpoint.h"

#include "data_types.h"
#include "env_controls.h"
#include "error_codes.h"
#include "error_print.h"
#include "transfer_logger.h"

namespace mcl {

int Endpoint::Initialize() {
    is_test_run_ = (env_communication_mode.value() == "TEST_STUB");
    is_recorded_ = env_record_communication.value();
    if (env_record_communication.found())
        if (!env_record_communication.IsDefined()) return kErrInvalidEnvVar;
    skip_send_record_ = env_record_discard_send.value();
    if (env_record_discard_send.found())
        if (!env_record_discard_send.IsDefined()) return kErrInvalidEnvVar;
    
    if (is_test_run_) is_recorded_ = false;
    if (is_recorded_) {
        std::string file_name = "MCL_LOG_" + std::to_string(id_);
        transfer_log_ = new TransferLogger(file_name, WRITE);
        transfer_log_->Write(&skip_send_record_, 1, kTypeChar, kSkipSendTag, kSkipSendTag);
    }

    constexpr int kTag = 410;
    if (protocol_->is_server()) {
        int num_programs = protocol_->num_programs();
        for (int i = 1; i < num_programs; ++i) {
            location_ids_.push_back(i);

            // check that the ID is correctly assigned
            int client_id;
            int result = Receive(&client_id, 1, kTypeInt, kTag, i);
            if (result != 0) return result;
            if (client_id != i) return kErrEndpointNotFound;
        }
    }

    if (protocol_->is_client()) {
        location_ids_.push_back(0);
        return Send(&id_, 1, kTypeInt, kTag, 0);
    }
    
    return kSuccess;
}

int Endpoint::Send(void *data, int length, int data_type, int tag, int destination) {
    for (auto & id : location_ids_) {
        if (id == destination || is_test_run_) {
            if (is_recorded_ && !skip_send_record_)
                transfer_log_->Write(data, length, data_type, tag, destination);
            return protocol_->Send(data, length, data_type, tag, destination);
        }
    }
    return kErrEndpointNotFound;
}

int Endpoint::Receive(void *data, int length, int data_type, int tag, int source) {
    for (auto & id : location_ids_) {
        if (id == source || is_test_run_) {
            int result = protocol_->Receive(data, length, data_type, tag, source);
            if (is_recorded_)
                transfer_log_->Write(data, length, data_type, tag, source);
            return result;
        }
    }
    return kErrEndpointNotFound;
}

int Endpoint::ProbeSize(int data_type, int source) {
    int size = -1;
    for (auto & id : location_ids_) {
        if (id == source || is_test_run_) {
            size = protocol_->ProbeSize(data_type, source);
            if (is_recorded_)
                transfer_log_->Write(reinterpret_cast<void*>(&size), 1, data_type, kProbeTag, source);
            break;
        }
    }
    return size;
}

int Endpoint::Finalize() {
    if (is_recorded_) delete transfer_log_;
    location_ids_.clear();
    protocol_ = nullptr;
    return kSuccess;
}

} // namespace mcl
