!    MCL: MiMiC Communication Library
!    Copyright (C) 2015-2025  The MiMiC Authors (see CONTRIBUTORS file for details).
!
!    This file is part of MCL.
!
!    MCL is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Lesser General Public License as
!    published by the Free Software Foundation, either version 3 of
!    the License, or (at your option) any later version.
!
!    MCL is distributed in the hope that it will be useful, but
!    WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU Lesser General Public License for more details.
!
!    You should have received a copy of the GNU Lesser General Public License
!    along with this program.  If not, see <http://www.gnu.org/licenses/>.

!> Contains bindings to the C API functions.
module c_bindings

    use, intrinsic :: iso_c_binding, only: c_int

#include "preproc_constants.inc"

    implicit none
    
    !> MCL datatypes.
    integer(kind=c_int), parameter :: MCL_CHAR = PP_TYPE_CHAR
    integer(kind=c_int), parameter :: MCL_INT = PP_TYPE_INT
    integer(kind=c_int), parameter :: MCL_LONG_INT = PP_TYPE_LONG_INT
    integer(kind=c_int), parameter :: MCL_FLOAT = PP_TYPE_FLOAT
    integer(kind=c_int), parameter :: MCL_DOUBLE = PP_TYPE_DOUBLE

    !> Bindings to the MCL library functions.
    interface
        !> Initializes the server endpoint.
        function c_mcl_initialize(args) result(err) bind(C, name="MCL_InitializeWrapper")
            use iso_c_binding, only: c_ptr, c_int
            !> Argument list (MPI_COMM_WORLD or null)
            type(c_ptr), value :: args
            !> Error status
            integer(kind=c_int) :: err
        end function c_mcl_initialize


        !> Deinitializes the communication layer.
        function c_mcl_finalize() result(err) bind(C, name="MCL_Finalize")
            use iso_c_binding, only: c_int
            !> Error status.
            integer(kind=c_int) :: err
        end function c_mcl_finalize


        !> Terminates all processes.
        function c_mcl_abort(error_code) result(err) bind(C, name="MCL_Abort")
            use iso_c_binding, only: c_int
            !> Error code to return.
            integer(kind=c_int), value :: error_code
            !> Error status.
            integer(kind=c_int) :: err
        end function c_mcl_abort


        !> Returns a version number of the library.
        subroutine c_mcl_get_api_version(version) bind(C, name="MCL_GetAPIVersion")
            use iso_c_binding, only: c_ptr
            !> Pointer to the array to store the version number.
            type(c_ptr), value :: version
        end subroutine c_mcl_get_api_version


        !> Returns the number of programs.
        integer(kind=c_int) function c_mcl_get_num_programs() bind(C, name="MCL_GetNumPrograms")
            use iso_c_binding, only: c_int
        end function c_mcl_get_num_programs


        !> Returns the invoking program ID.
        integer(kind=c_int) function c_mcl_get_program_id() bind(C, name="MCL_GetProgramID")
            use iso_c_binding, only: c_int
        end function c_mcl_get_program_id


        !> Returns the initialization status of MCL.
        logical(kind=c_bool) function c_mcl_is_initialized() bind(C, name="MCL_IsInitialized")
            use iso_c_binding, only: c_bool
        end function c_mcl_is_initialized


        !> Sends data to another program.
        function c_mcl_send(buffer, length, data_type, tag, destination) result(err) &
            bind(C, name="MCL_Send")
            use iso_c_binding, only: c_ptr, c_int
            !> Pointer to the buffer to send data from.
            type(c_ptr), value :: buffer
            !> Size of data buffer.
            integer(kind=c_int), value :: length
            !> Type of data.
            integer(kind=c_int), value :: data_type
            !> Message tag.
            integer(kind=c_int), value :: tag
            !> ID of the program that receives data.
            integer(kind=c_int), value :: destination
            !> Error status.
            integer(kind=c_int) :: err
        end function c_mcl_send


        !> Receives data from another program.
        function c_mcl_receive(buffer, length, data_type, tag, source) result(err) &
            bind(C, name="MCL_Receive")
            use iso_c_binding, only: c_ptr, c_int
            !> Pointer to the buffer to receive data.
            type(c_ptr), value :: buffer
            !> Size of data buffer.
            integer(kind=c_int), value :: length
            !> Type of data.
            integer(kind=c_int), value :: data_type
            !> Message tag.
            integer(kind=c_int), value :: tag
            !> ID of the program to receive data from.
            integer(kind=c_int), value :: source
            !> Error status.
            integer(kind=c_int) :: err
        end function c_mcl_receive


        !> Returns the request name associated with the given request value.
        function c_mcl_get_request_name(request) result(request_name) bind(C, name="MCL_GetRequestName")
            use iso_c_binding, only: c_ptr, c_int
            !> Request value.
            integer(kind=c_int), value :: request
            !> Pointer to a C string with the request name.
            type(c_ptr) :: request_name
        end function c_mcl_get_request_name

 
        !> Returns length of a C string.
        function c_strlen(string) result(length) bind(c,name="strlen")
            use iso_c_binding, only: c_ptr, c_size_t
            !> Pointer to a C string.
            type(c_ptr), intent(in), value :: string
            !> Length of the string.
            integer(c_size_t) :: length
        end function c_strlen
    end interface
   
end module c_bindings 
