/***********************************************************************************

    LibBQB - File Format and Compression Algorithms for Trajectories of
             Volumetric Data and Atom Positions

    https://brehm-research.de/bqb

    Free software, licensed under GNU LGPL v3

    Copyright (c) Martin Brehm and Martin Thomas,
                  Martin Luther University Halle-Wittenberg, Germany,
                  2016 - 2022.

    Please cite:  M. Brehm, M. Thomas: "An Efficient Lossless Compression Algorithm
                  for Trajectories of Atom Positions and Volumetric Data",
                  J. Chem. Inf. Model. 2018, 58 (10), pp 2092-2107.

    --------------------------------------------------------------------------------

    LibBQB is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/








/*****************************************************************************
******                                                                  ******
******      libbqb example: Writing a single BQB compressed CUBE file   ******
******                                                                  ******
******      Released under GNU L-GPL v3                                 ******
******                                                                  ******
******      (c) Martin Brehm, 2019 - 2022                               ******
******          https://brehm-research.de/bqb                           ******
******                                                                  ******
*****************************************************************************/



// Either compile this example together with all libbqb source files (*.cpp),
// or link it together with a static libbqb library (*.a) you have created before.



// This is the only include required for libbqb
#include "bqb.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>



// Do anything with the messages and errors which libbqb wants to print.
// Only used with variant 1C below.

void PrintCallback( const char *s ) {

	// Here, we simply print it to stdout. But you might do anything complicated here.
	printf( "%s", s );
	fflush( stdout );
}



void ErrorCallback( const char *s ) {

	// Here, we simply print it to stderr. But you might do anything complicated here.
	fprintf( stderr, "%s", s );
	fflush( stderr );
}



int bqbex1_main( int argc, const char *argv[] ) {

	CBQBInterface *ifc;
	CBQBVolHeader *volhead;
	CBQBAtomHeader *atomhead;
	CBQBCellInfo *cell;
	CBQBVolFrame *volframe;
	CBQBAtomPosFrame *atomframe;
	CBQBFileWriter *file;
	unsigned int z, z2, ix, iy, iz;

	// This minimalistic example does not care for error checking / output redirection / etc.
	// See all further examples for these techniques.

	printf( "\n*** libbqb example: Writing a single BQB compressed CUBE file ***\n\n" );

	if (argc != 2) {
		printf( "This example requires the output BQB file name as command line argument.\n\n" );
		return 0;
	}

	ifc = BQBCreateInterface( 0 );

	printf( "Trying to open \"%s\" for writing...\n", argv[1] );

	file = ifc->CreateBQBFileWriter(
		argv[1],       // Output file name
		1,             // The frame history depth. For a single CUBE frame, always use 1.                  
		BQB_CONTENTS_VOLUMETRIC | BQB_CONTENTS_ATOMPOS | BQB_CONTENTS_CELLINFO,  // File contents. Here: Volumetric data, atom positions & cell info
		BQB_OVERWRITE  // Flags. Here: Overwrite file if already existing
	);

	volhead = ifc->CreateVolumetricHeader();

	// The resolution of the volumetric data (X, Y, Z)
	volhead->SetResolution( 80, 60, 40 );

	file->SetVolumetricHeader( volhead );


	cell = ifc->CreateCellInfo();

	// Set the unit cell vector
	cell->SetOrthorhombic( 30.0, 20.0, 15.0, BQB_UNIT_BOHR );

	file->SetCellInfo( cell );


	atomhead = ifc->CreateAtomHeader();

	atomhead->SetAtomCount( 10 );

	for (z=0;z<atomhead->GetAtomCount();z++)
		atomhead->SetAtomOrd( z, 6 ); // Set the ordinal numbers for all atoms (6 = carbon)

	file->SetAtomHeader( atomhead );

	// Creates a new volumetric frame (with uninitialized volumetric data)
	volframe = ifc->CreateVolumetricFrame( volhead );

	// Set the volumetric data to some values
	for ( iz=0; iz<volframe->GetResolutionZ(); iz++ )
		for ( iy=0; iy<volframe->GetResolutionY(); iy++ )
			for ( ix=0; ix<volframe->GetResolutionX(); ix++ )
				volframe->SetVoxel(
					ix,
					iy,
					iz,
					sin( ix/10.0 ) * cos( iy/20.0 ) + sin( z/10.0 )
				);

	// Hand over the frame to the file. The volframe pointer is no longer valid after this.
	file->PushVolumetricFrame( volframe );

	// The two CUBE comment lines
	file->PushCommentLine( "-Quickstep-", " ELECTRON DENSITY" );

	// Creates a new atom position frame (with uninitialized atom positions)
	atomframe = ifc->CreateAtomPosFrame( file->GetAtomHeader() );

	// Set the atom positions to some values
	for ( z2=0; z2<atomframe->GetAtomCount(); z2++ )
		atomframe->SetAtomPosition(
			z2,
			5.0 * sin( (z+z2)/2.0 ),
			5.0 * cos( (z+z2)/2.0 ),
			2.0 * z,
			BQB_UNIT_BOHR
		);

	// Hand over the frame to the file. The atomframe pointer is no longer valid after this.
	file->PushAtomPosFrame( atomframe );


	// Call this to optimize the compression parameters for the dataset
	//   (otherwise, default parameters are used)
	file->Optimize( 1 ); // 1 - fast optimization ... 4 - very slow (and thorough) optimization


	// Compresses the data and writes the BQB frame to disk.
	// This function call might take some time (compression)...
	file->WritePendingFrames( true );


	// Writes the BQB index. Should always be called at the very end.
	file->WriteIndex();


	file->CloseFile();


	// After this, all BQB pointers are invalid
	BQBReleaseInterface( ifc );


	printf( "All finished. Leaving.\n\n" );

	return 0;
}






/**************************************************************************
******                                                               ******
******      libbqb example: Writing volumetric BQB trajectories      ******
******                                                               ******
******      Released under GNU L-GPL v3                              ******
******                                                               ******
******      (c) Martin Brehm, 2019 - 2022                            ******
******          https://brehm-research.de/bqb                        ******
******                                                               ******
**************************************************************************/



// Either compile this example together with all libbqb source files (*.cpp),
// or link it together with a static libbqb library (*.a) you have created before.



// This is the only include required for libbqb
#include "bqb.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>



// Do anything with the messages and errors which libbqb wants to print.
// Only used with variant 1C below.
/*
void PrintCallback( const char *s ) {

	// Here, we simply print it to stdout. But you might do anything complicated here.
	printf( "%s", s );
	fflush( stdout );
}



void ErrorCallback( const char *s ) {

	// Here, we simply print it to stderr. But you might do anything complicated here.
	fprintf( stderr, "%s", s );
	fflush( stderr );
}
*/


int bqbex2_main( int argc, const char *argv[] ) {

	CBQBInterface *ifc;
	CBQBVolHeader *volhead;
	CBQBAtomHeader *atomhead;
	CBQBCellInfo *cell;
	CBQBVolFrame *volframe;
	CBQBAtomPosFrame *atomframe;
	CBQBFileWriter *file;
	unsigned int z, z2, ix, iy, iz;
	double *voldat;


	printf( "\n*** libbqb example: Writing volumetric BQB trajectories ***\n\n" );


	if (argc != 2) {
		printf( "This example requires the output BQB file name as command line argument.\n\n" );
		return 0;
	}


	ifc = BQBCreateInterface( 0 );

	if (BQBGetLastError() != 0) {
		printf("BQBCreateInterface() returned an error.\n");
		abort();
	}

	/************************************************************************************************************
	***   Variant 1A: Print all libbqb messages and errors directly to the screen (i.e., stdout and stderr)   ***
	************************************************************************************************************/

	// This is the default. No code required.

	/*************************************************************************
	***   Variant 1B: Print all libbqb messages and errors to a log file   ***
	*************************************************************************/

	//FILE *logfile;

	//logfile = fopen( "log.txt", "wt" );

	//if (logfile == NULL) {
	//	printf("Error: Could not open \"log.txt\" for writing.\n");
	//	abort();
	//}
	
	//ifc->SetPrintFile( logfile );

	//ifc->SetEPrintFile( logfile );

	/*****************************************************************************************************
	***   Variant 1C: Redirect all libbqb messages and errors into user-specified callback functions   ***
	*****************************************************************************************************/

	ifc->SetPrintCallback( &PrintCallback ); // Messages

	ifc->SetEPrintCallback( &ErrorCallback ); // Errors

	/************************
	***   End Variant 1   ***
	************************/


	// Decide how much information shall be printed. Possible options are:
	//   BQB_PL_SILENT
	//   BQB_PL_QUIET
	//   BQB_PL_STANDARD
	//   BQB_PL_VERBOSE
	//   BQB_PL_DEBUG
	ifc->SetPrintLevel( BQB_PL_STANDARD );

	printf( "Trying to open \"%s\" for writing...\n", argv[1] );

	file = ifc->CreateBQBFileWriter(
		argv[1],       // Output file name
		10,            // The frame history depth. If the frames are independent (i.e., no trajectory), pass a value of 1 here.                  
		BQB_CONTENTS_VOLUMETRIC | BQB_CONTENTS_ATOMPOS | BQB_CONTENTS_CELLINFO,  // File contents. Here: Volumetric data, atom positions & cell info
		BQB_OVERWRITE  // Flags. Here: Overwrite file if already existing
	);

	if (ifc->GetLastError() != 0) {
		printf( "CBQBInterface::CreateBQBFileWriter() returned an error: %lu.\n", ifc->GetLastError() );
		abort();
	}
	// All further calls to ifc->GetLastError() are omitted for clarity

	volhead = ifc->CreateVolumetricHeader();

	// The resolution of the volumetric data (X, Y, Z)
	volhead->SetResolution( 80, 60, 40 );

	file->SetVolumetricHeader( volhead );


	cell = ifc->CreateCellInfo();

	/****************************************
	***   Variant 2A: Orthorhombic cell   ***
	****************************************/

	// X, Y, Z cell vector
	cell->SetOrthorhombic( 30.0, 20.0, 15.0, BQB_UNIT_BOHR );

	/********************************************
	***   Variant 2B: Non-orthorhombic cell   ***
	********************************************/

	cell->SetGeneric(
		30.0,  0.0,  0.0,  // Cell vector A (X, Y, Z)
		 0.0, 20.0,  0.0,  // Cell vector B (X, Y, Z)
		 0.0,  0.0, 15.0,  // Cell vector C (X, Y, Z)
		BQB_UNIT_BOHR
	);

	/************************
	***   End Variant 2   ***
	************************/

	file->SetCellInfo( cell );


	atomhead = ifc->CreateAtomHeader();

	atomhead->SetAtomCount( 10 );

	for (z=0;z<atomhead->GetAtomCount();z++)
		atomhead->SetAtomOrd( z, 6 ); // Set the ordinal numbers for all atoms (6 = carbon)

	file->SetAtomHeader( atomhead );


	// Create local working array for volumetric data (only needed for variant 3A below)
	voldat = new double[ volhead->GetResolutionX() * volhead->GetResolutionY() * volhead->GetResolutionZ() ];


	// Do 20 hypothetical MD steps
	for ( z=0; z<20; z++ ) {

		printf( "    Frame %2u ...\n", z+1 );

		// Creates a new volumetric frame (with uninitialized volumetric data)
		volframe = ifc->CreateVolumetricFrame( volhead, z+1 );

		/****************************************************************
		***   Variant 3A: Modify local array and load it into frame   ***
		****************************************************************/

		for ( iz=0; iz<volframe->GetResolutionZ(); iz++ )
			for ( iy=0; iy<volframe->GetResolutionY(); iy++ )
				for ( ix=0; ix<volframe->GetResolutionX(); ix++ )
					voldat[ iz*volframe->GetResolutionY()*volframe->GetResolutionX() + iy*volframe->GetResolutionX() + ix ]
						= sin( ix/10.0 ) * cos( iy/20.0 ) + sin( z/10.0 );

		volframe->LoadFromArray( voldat, BQB_ORDER_ZYX );

		/****************************************************
		***   Variant 3B: Directly modify data in frame   ***
		****************************************************/

		for ( iz=0; iz<volframe->GetResolutionZ(); iz++ )
			for ( iy=0; iy<volframe->GetResolutionY(); iy++ )
				for ( ix=0; ix<volframe->GetResolutionX(); ix++ )
					volframe->SetVoxel(
						ix,
						iy,
						iz,
						sin( ix/10.0 ) * cos( iy/20.0 ) + sin( z/10.0 )
					);

		/************************
		***   End Variant 3   ***
		************************/

		// Hand over the frame to the file. The volframe pointer is no longer valid after this.
		file->PushVolumetricFrame( volframe );

		// The two CUBE comment lines
		file->PushCommentLine( "-Quickstep-", " ELECTRON DENSITY" );

		// Creates a new atom position frame (with uninitialized atom positions)
		atomframe = ifc->CreateAtomPosFrame( file->GetAtomHeader() );

		for ( z2=0; z2<atomframe->GetAtomCount(); z2++ )
			atomframe->SetAtomPosition(
				z2,
				5.0 * sin( (z+z2)/2.0 ),
				5.0 * cos( (z+z2)/2.0 ),
				2.0 * z,
				BQB_UNIT_BOHR
			);

		// Hand over the frame to the file. The atomframe pointer is no longer valid after this.
		file->PushAtomPosFrame( atomframe );


		// Compresses the data and writes the BQB frame to disk.
		// This function call might take some time (compression)...
		file->WritePendingFrames( true );
	}


	// Writes the BQB index. Should always be called at the very end.
	file->WriteIndex();


	file->CloseFile();


	// After this, all BQB pointers are invalid
	BQBReleaseInterface( ifc );

	delete[] voldat;

	// We do not need to free any pointers that have been allocated via the "Create..." functions.
	// This is all done inside BQBReleaseEngine().

	//fclose( logfile );

	printf( "All finished. Leaving.\n\n" );

	return 0;
}







/**************************************************************************
******                                                               ******
******      libbqb example: Writing volumetric BQB trajectories      ******
******                      with automatic parameter optimization    ******
******                                                               ******
******      Released under GNU L-GPL v3                              ******
******                                                               ******
******      (c) Martin Brehm, 2019 - 2022                            ******
******          https://brehm-research.de/bqb                        ******
******                                                               ******
**************************************************************************/



// Either compile this example together with all libbqb source files (*.cpp),
// or link it together with a static libbqb library (*.a) you have created before.



// This is the only include required for libbqb
#include "bqb.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>



// Do anything with the messages and errors which libbqb wants to print.
// Only used with variant 1C below.
/*
void PrintCallback( const char *s ) {

	// Here, we simply print it to stdout. But you might do anything complicated here.
	printf( "%s", s );
	fflush( stdout );
}



void ErrorCallback( const char *s ) {

	// Here, we simply print it to stderr. But you might do anything complicated here.
	fprintf( stderr, "%s", s );
	fflush( stderr );
}
*/


int bqbex3_main( int argc, const char *argv[] ) {

	CBQBInterface *ifc;
	CBQBVolHeader *volhead;
	CBQBAtomHeader *atomhead;
	CBQBCellInfo *cell;
	CBQBVolFrame *volframe;
	CBQBAtomPosFrame *atomframe;
	CBQBFileWriter *file;
	unsigned int z, z2, ix, iy, iz;
	double *voldat;


	printf( "\n*** libbqb example: Writing volumetric BQB trajectories with automatic parameter optimization ***\n\n" );


	if (argc != 2) {
		printf( "This example requires the output BQB file name as command line argument.\n\n" );
		return 0;
	}


	ifc = BQBCreateInterface( 0 );

	if (BQBGetLastError() != 0) {
		printf("BQBCreateInterface() returned an error.\n");
		abort();
	}

	/************************************************************************************************************
	***   Variant 1A: Print all libbqb messages and errors directly to the screen (i.e., stdout and stderr)   ***
	************************************************************************************************************/

	// This is the default. No code required.

	/*************************************************************************
	***   Variant 1B: Print all libbqb messages and errors to a log file   ***
	*************************************************************************/

	//FILE *logfile;

	//logfile = fopen( "log.txt", "wt" );

	//if (logfile == NULL) {
	//	printf("Error: Could not open \"log.txt\" for writing.\n");
	//	abort();
	//}
	
	//ifc->SetPrintFile( logfile );

	//ifc->SetEPrintFile( logfile );

	/*****************************************************************************************************
	***   Variant 1C: Redirect all libbqb messages and errors into user-specified callback functions   ***
	*****************************************************************************************************/

	ifc->SetPrintCallback( &PrintCallback ); // Messages

	ifc->SetEPrintCallback( &ErrorCallback ); // Errors

	/************************
	***   End Variant 1   ***
	************************/


	// Decide how much information shall be printed. Possible options are:
	//   BQB_PL_SILENT
	//   BQB_PL_QUIET
	//   BQB_PL_STANDARD
	//   BQB_PL_VERBOSE
	//   BQB_PL_DEBUG
	ifc->SetPrintLevel( BQB_PL_STANDARD );

	printf( "Trying to open \"%s\" for writing...\n", argv[1] );

	file = ifc->CreateBQBFileWriter(
		argv[1],       // Output file name
		10,            // The frame history depth. If the frames are independent (i.e., no trajectory), pass a value of 1 here.                  
		BQB_CONTENTS_VOLUMETRIC | BQB_CONTENTS_ATOMPOS | BQB_CONTENTS_CELLINFO,  // File contents. Here: Volumetric data, atom positions & cell info
		BQB_OVERWRITE  // Flags. Here: Overwrite file if already existing
	);

	if (ifc->GetLastError() != 0) {
		printf( "CBQBInterface::CreateBQBFileWriter() returned an error: %lu.\n", ifc->GetLastError() );
		abort();
	}
	// All further calls to ifc->GetLastError() are omitted for clarity

	volhead = ifc->CreateVolumetricHeader();

	// The resolution of the volumetric data (X, Y, Z)
	volhead->SetResolution( 80, 60, 40 );

	file->SetVolumetricHeader( volhead );


	cell = ifc->CreateCellInfo();

	/****************************************
	***   Variant 2A: Orthorhombic cell   ***
	****************************************/

	// X, Y, Z cell vector
	cell->SetOrthorhombic( 30.0, 20.0, 15.0, BQB_UNIT_BOHR );

	/********************************************
	***   Variant 2B: Non-orthorhombic cell   ***
	********************************************/

	cell->SetGeneric(
		30.0,  0.0,  0.0,  // Cell vector A (X, Y, Z)
		 0.0, 20.0,  0.0,  // Cell vector B (X, Y, Z)
		 0.0,  0.0, 15.0,  // Cell vector C (X, Y, Z)
		BQB_UNIT_BOHR
	);

	/************************
	***   End Variant 2   ***
	************************/

	file->SetCellInfo( cell );


	atomhead = ifc->CreateAtomHeader();

	// Sets the total number of atoms in the system
	atomhead->SetAtomCount( 10 );

	for (z=0;z<atomhead->GetAtomCount();z++)
		atomhead->SetAtomOrd( z, 6 ); // Set the ordinal numbers for all atoms (6 = carbon)

	file->SetAtomHeader( atomhead );


	// Create local working array for volumetric data (only needed for variant 3A below)
	voldat = new double[ volhead->GetResolutionX() * volhead->GetResolutionY() * volhead->GetResolutionZ() ];


	// Do 20 hypothetical MD steps
	for ( z=0; z<20; z++ ) {

		printf( "    Frame %2u ...\n", z+1 );

		// Creates a new volumetric frame (with uninitialized volumetric data)
		volframe = ifc->CreateVolumetricFrame( volhead, z+1 );

		/****************************************************************
		***   Variant 3A: Modify local array and load it into frame   ***
		****************************************************************/

		for ( iz=0; iz<volframe->GetResolutionZ(); iz++ )
			for ( iy=0; iy<volframe->GetResolutionY(); iy++ )
				for ( ix=0; ix<volframe->GetResolutionX(); ix++ )
					voldat[ iz*volframe->GetResolutionY()*volframe->GetResolutionX() + iy*volframe->GetResolutionX() + ix ]
						= sin( ix/10.0 ) * cos( iy/20.0 ) + sin( z/10.0 );

		volframe->LoadFromArray( voldat, BQB_ORDER_ZYX );

		/****************************************************
		***   Variant 3B: Directly modify data in frame   ***
		****************************************************/

		for ( iz=0; iz<volframe->GetResolutionZ(); iz++ )
			for ( iy=0; iy<volframe->GetResolutionY(); iy++ )
				for ( ix=0; ix<volframe->GetResolutionX(); ix++ )
					volframe->SetVoxel(
						ix,
						iy,
						iz,
						sin( ix/10.0 ) * cos( iy/20.0 ) + sin( z/10.0 )
					);

		/************************
		***   End Variant 3   ***
		************************/

		// Hand over the frame to the file. The volframe pointer is no longer valid after this.
		file->PushVolumetricFrame( volframe );

		// The two CUBE comment lines
		file->PushCommentLine( "-Quickstep-", " ELECTRON DENSITY" );

		// Creates a new atom position frame (with uninitialized atom positions)
		atomframe = ifc->CreateAtomPosFrame( file->GetAtomHeader() );

		for ( z2=0; z2<atomframe->GetAtomCount(); z2++ )
			atomframe->SetAtomPosition(
				z2,
				5.0 * sin( (z+z2)/2.0 ),
				5.0 * cos( (z+z2)/2.0 ),
				2.0 * z,
				BQB_UNIT_BOHR
			);

		// Hand over the frame to the file. The atomframe pointer is no longer valid after this.
		file->PushAtomPosFrame( atomframe );

		// Call this to optimize the compression parameters for the dataset after 10 steps have been cached
		//   (otherwise, default parameters are used)
		if (z == 9)
			file->Optimize( 1 ); // 1 - fast optimization ... 4 - very slow (and thorough) optimization

		// After parameters have been optimized, actually start writing to output BQB file
		if (z >= 9) {

			// Compresses the data and writes the BQB frame to disk.
			// This function call might take some time (compression)...
			file->WritePendingFrames( true );
		}
	}


	// Writes the BQB index. Should always be called at the very end.
	file->WriteIndex();


	file->CloseFile();


	// After this, all BQB pointers are invalid
	BQBReleaseInterface( ifc );

	delete[] voldat;

	// We do not need to free any pointers that have been allocated via the "Create..." functions.
	// This is all done inside BQBReleaseEngine().

	//fclose( logfile );

	printf( "All finished. Leaving.\n\n" );

	return 0;
}













/*********************************************************************************
******                                                                      ******
******      libbqb example: Read BQB trajectory, write CUBE trajectory      ******
******                                                                      ******
******      Released under GNU L-GPL v3                                     ******
******                                                                      ******
******      (c) Martin Brehm, 2019 - 2022                                   ******
******          https://brehm-research.de/bqb                               ******
******                                                                      ******
*********************************************************************************/



// Either compile this example together with all libbqb source files (*.cpp),
// or link it together with a static libbqb library (*.a) you have created before.



// This is the only include required for libbqb
#include "bqb.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>


/*
// Do anything with the messages and errors which libbqb wants to print.
// Only used with variant 1C below.

void PrintCallback( const char *s ) {

	// Here, we simply print it to stdout. But you might do anything complicated here.
	printf( "%s", s );
	fflush( stdout );
}



void ErrorCallback( const char *s ) {

	// Here, we simply print it to stderr. But you might do anything complicated here.
	fprintf( stderr, "%s", s );
	fflush( stderr );
}
*/


int bqbex4_main( int argc, const char *argv[] ) {

	CBQBInterface *ifc;
	CBQBFileReader *file;
	unsigned int z, ix, iy, iz;
	double sum;
	FILE *outcube;


	printf( "\n*** libbqb example: Read BQB trajectory, write CUBE trajectory ***\n\n" );


	if (argc != 3) {
		printf( "This example requires the input BQB file name and the output CUBE file name as command line arguments.\n\n" );
		return 0;
	}

	
	ifc = BQBCreateInterface( 0 );

	if (BQBGetLastError() != 0) {
		printf("BQBCreateInterface() returned an error.\n");
		abort();
	}

	/************************************************************************************************************
	***   Variant 1A: Print all libbqb messages and errors directly to the screen (i.e., stdout and stderr)   ***
	************************************************************************************************************/

	// This is the default. No code required.

	/*************************************************************************
	***   Variant 1B: Print all libbqb messages and errors to a log file   ***
	*************************************************************************/

	//FILE *logfile;

	//logfile = fopen( "log.txt", "wt" );

	//if (logfile == NULL) {
	//	printf("Error: Could not open \"log.txt\" for writing.\n");
	//	abort();
	//}
	
	//ifc->SetPrintFile( logfile );

	//ifc->SetEPrintFile( logfile );

	/*****************************************************************************************************
	***   Variant 1C: Redirect all libbqb messages and errors into user-specified callback functions   ***
	*****************************************************************************************************/

	ifc->SetPrintCallback( &PrintCallback ); // Messages

	ifc->SetEPrintCallback( &ErrorCallback ); // Errors

	/************************
	***   End Variant 1   ***
	************************/


	// Decide how much information shall be printed. Possible options are:
	//   BQB_PL_SILENT
	//   BQB_PL_QUIET
	//   BQB_PL_STANDARD
	//   BQB_PL_VERBOSE
	//   BQB_PL_DEBUG
	ifc->SetPrintLevel( BQB_PL_STANDARD );

	printf( "Trying to open \"%s\" for reading...\n", argv[1] );

	file = ifc->CreateBQBFileReader( argv[1] );

	if (ifc->GetLastError() != 0) {
		printf( "CBQBInterface::CreateBQBFileReader() returned an error: %lu.\n", ifc->GetLastError() );
		abort();
	}
	// All further calls to ifc->GetLastError() are omitted for clarity


	if (!file->ContainsAtomPositions()) {
		printf( "Error: BQB file does not contain atom positions.\n\n" );
		abort();
	}

	if (!file->ContainsVolumetricData()) {
		printf( "Error: BQB file does not contain volumetric data.\n\n" );
		abort();
	}


	if (file->HasIndex())
		printf( "The BQB file contains %lu frames in total.\n", file->GetTotalFrameCount() );
	else
		printf( "The BQB file does not possess an index - total frame count is unknown.\n" );


	printf( "Trying to open \"%s\" for writing...\n", argv[2] );

	outcube = fopen( argv[2], "wt" );
	
	if (outcube == NULL) {
		printf( "Error opening file for writing.\n\n" );
		abort();
	}

	z = 0;

	while (true) {

		z++;

		printf( "    Frame %u:\n", z );

		printf( "        Reading BQB...\n" );

		file->ReadFrame();

		if (file->IsEOF()) {
			printf( "Reached end of BQB file.\n\n" );
			break;
		}

		if (ifc->GetLastError() != 0) {
			printf( "Error while reading the frame: %lu.\n\n", ifc->GetLastError() );
			break;
		}

		// Here, you can compute something from the data.
		// Important: Do NOT attempt modify the data here; you would need to create a local copy first.
		// As example, we compute the sum over all bins.

		sum = 0;
		for ( iz=0; iz<file->GetCurrentVolumetricFrame()->GetResolutionZ(); iz++ )
			for ( iy=0; iy<file->GetCurrentVolumetricFrame()->GetResolutionY(); iy++ )
				for ( ix=0; ix<file->GetCurrentVolumetricFrame()->GetResolutionX(); ix++ )
					sum += file->GetCurrentVolumetricFrame()->GetVoxel( ix, iy, iz );

		printf( "        The total sum is %16.8f\n", sum );

		printf( "        Writing CUBE...\n" );

		ifc->WriteCubeFrame(
			outcube,
			file->GetVolumetricHeader(),
			file->GetCurrentVolumetricFrame(),
			file->GetAtomHeader(),
			file->GetCurrentAtomPosFrame(),
			file->GetCellInfo()
		);
	}

	printf( "Have converted %u frames in total.\n\n", z );


	file->CloseFile();


	fclose( outcube );


	// After this, all BQB pointers are invalid
	BQBReleaseInterface( ifc );

	//fclose( logfile );

	printf( "All finished. Leaving.\n\n" );

	return 0;
}






/*
int bqbex4_main( int argc, const char *argv[] ) {

	return 0;
}



int bqbex5_main( int argc, const char *argv[] ) {

	return 0;
}
*/





