/*****************************************************************************

    LibVori - Voronoi Integration for Electromagnetic Moments

    https://brehm-research.de/voronoi

    Free software, licensed under GNU LGPL v3

    Copyright (c) 2020-2022 Martin Brehm

    Please cite:  J. Chem. Phys. 2020, 152 (16), 164105.        (DOI 10.1063/5.0005078 )
                  Phys. Chem. Chem. Phys. 2015, 17, 3207-3213.  (DOI 10.1039/C4CP05272B )
				  Molecules 2021, 26 (7), 1875.                 (DOI 10.3390/molecules26071875 )

    ---------------------------------------------------------------------------

    LibVori is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*****************************************************************************/


#ifndef TOOLS_H
#define TOOLS_H


#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <vector>




class CTetraPak;
class CSnapshot;
class CBQBInterface;
class CBQBVolHeader;
class CBQBFileWriter;



#define LV_PL_SILENT   0
#define LV_PL_ERROR    1
#define LV_PL_WARNING  2
#define LV_PL_INFO     3
#define LV_PL_VERBOSE  4



#define UNUSED(a) ((void)a)


#define Pi (3.1415926535897932385)

#define LEN_BOHR2ANGSTROM (0.52917721092)

#define LEN_AU2PM (52.917721092) // Atomic units to pm: CONST_BOHR_RADIUS*1E12



#define MIN(a,b) (((a)<=(b)) ? (a) : (b))
#define MAX(a,b) (((a)>=(b)) ? (a) : (b))

#define MIN3(a,b,c) MIN(MIN(a,b),c)
#define MED3(a,b,c) MAX( MIN(a,b), MIN( MAX(a,b), c ) )
#define MAX3(a,b,c) MAX(MAX(a,b),c)

#define MIN4(a,b,c,d) MIN(MIN(a,b),MIN(c,d))
#define MAX4(a,b,c,d) MAX(MAX(a,b),MAX(c,d))


#define SQR(x)   ((x)*(x))


inline int floori(int a, int b) {
	if (0 < (a ^ b)) {
		return a / b;
	} else {
		if (a % b != 0)
			return a / b - 1;
		else
			return a / b;
	}
}


#ifdef _MSC_VER
	#define isfinite(X) _finite(X)
#endif


class CElement {
public:
	char m_sLabel[16];
	int m_iOrd;
	double m_fMass;
	double m_fRVdW;
	double m_fRCov;
};


#ifdef __GNUG__ // Variadic Argument Type Checking of GCC

	void libvori_printf( int level, const char *s, ... ) __attribute__ ((format (printf, 2, 3)));

#else

	void libvori_printf( int level, const char *s, ... );

#endif


char *FormatBytes(double i);

bool FileExist( const char *s );

void AddElement( const char *label, int ord, double mass, double rcov, double rvdw );

void AddElementData();

void WriteHeader();

double FindOrdCovalentRadius( int ord );

double FindOrdVdWRadius( int ord );

extern bool glv_bJitter;
extern int glv_iJitterSeed;
extern double glv_fJitterAmplitude;

extern int glv_iPrintLevel;

extern FILE *glv_fLogFile;

extern char *glv_pPrintPrefix;

extern std::vector<CElement*> glv_oaElements;

extern CTetraPak *glv_pTetraPak;
extern CSnapshot *glv_pSnapshot;

extern int glv_iVoroMemory;

extern std::vector<double> glv_faVoronoiRadii;

extern int glv_iInitialized;

extern int glv_iInterpolation;

extern CBQBInterface *glv_pBQBInterface;

extern bool glv_bEMPOutput;

extern int glv_iStepCounter;

extern bool glv_bVoronoi;
extern bool glv_bVoronoiSkipFirstFrame;
extern int glv_iVoronoiStepCounter;

extern bool glv_bBQBInitialized;
extern int glv_iBQBOptimization;
extern int glv_iBQBHistory;
extern char *glv_pBQBFilename;
extern CBQBInterface *glv_BIF;
extern CBQBVolHeader *glv_pBQBVolHead;
extern CBQBFileWriter *glv_pBQBFile;
extern int glv_iBQBStepCounter;
extern bool glv_bBQBSkipFirstFrame;
extern char *glv_pBQBParmString;
extern bool glv_bBQBStoreStep;
extern bool glv_bBQBCheck;
extern bool glv_bVoriOverwrite;
extern bool glv_bBQBOverwrite;
extern bool glv_bRadiiWritten;



#endif


