/*****************************************************************************

    LibVori - Voronoi Integration for Electromagnetic Moments

    https://brehm-research.de/voronoi

    Free software, licensed under GNU LGPL v3

    Copyright (c) 2020-2022 Martin Brehm

    Please cite:  J. Chem. Phys. 2020, 152 (16), 164105.        (DOI 10.1063/5.0005078 )
                  Phys. Chem. Chem. Phys. 2015, 17, 3207-3213.  (DOI 10.1039/C4CP05272B )
				  Molecules 2021, 26 (7), 1875.                 (DOI 10.3390/molecules26071875 )

    ---------------------------------------------------------------------------

    LibVori is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*****************************************************************************/


#include <stdio.h>
#include <stdlib.h>

#include <vector>
#include <set>

#include "tools.h"
#include "tetrapak.h"
#include "snapshot.h"
#include "bqb.h"



//#define DEBUG_INTERFACE


//#define TEST



#ifndef _MSC_VER


#include <chrono>


double GetElapsedTime() {

	static std::chrono::time_point<std::chrono::high_resolution_clock> last;
	static bool first=true;
	double t;

	if (first) {
		first = false;
		t = 0;
	} else
		t = (double)std::chrono::duration_cast<std::chrono::microseconds>( std::chrono::high_resolution_clock::now() - last ).count() / 1.0e6;

	last = std::chrono::high_resolution_clock::now();
	return t;
}


#endif




void lv_setPrintPrefix( const char *s ) {

	if (glv_pPrintPrefix != NULL)
		delete[] glv_pPrintPrefix;
	glv_pPrintPrefix = new char[strlen(s)+1];
	strcpy(glv_pPrintPrefix,s);
}



bool libvori_Initialize( ) {

	WriteHeader();

	AddElementData();

	glv_pBQBInterface = BQBCreateInterface(0);
	glv_pBQBInterface->SetPrintLevel( BQB_PL_QUIET );

	return true;
}



void libvori_PrintRadiusInfo( CSnapshot *snap ) {

	int z, z2;
	std::set<int> setord;
	std::set<int> setrad;
	std::set<int>::iterator setit, setit2;


	libvori_printf( LV_PL_INFO, "The following atom radii are set:\n" );

	for (z=0;z<(int)snap->m_oaAtoms.size();z++)
		setord.insert( snap->m_oaAtoms[z]->m_iOrd );

	for (setit=setord.begin();setit!=setord.end();++setit) {
		setrad.clear();
		for (z2=0;z2<(int)snap->m_oaAtoms.size();z2++) {
			if (snap->m_oaAtoms[z2]->m_iOrd != *setit)
				continue;
			setrad.insert( (int)floor(snap->m_oaAtoms[z2]->m_fRadius*100.0 + 0.5) );
		}
		libvori_printf( LV_PL_INFO, "  Element %s:\n", GetAtomOrdLabel(*setit) );
		for (setit2=setrad.begin();setit2!=setrad.end();++setit2) {
			libvori_printf( LV_PL_INFO, "    %7.2f pm", (*setit2)/100.0 );
			if (*setit2 == (int)floor(FindOrdVdWRadius(*setit)*100.0 + 0.5))
				libvori_printf( LV_PL_INFO, "  (VdW radius)" );
			else if (*setit2 == (int)floor(FindOrdCovalentRadius(*setit)*100.0 + 0.5))
				libvori_printf( LV_PL_INFO, "  (covalent radius)" );
			libvori_printf( LV_PL_INFO, "\n" );
		}
	}

	libvori_printf( LV_PL_INFO, "----------------------------------------------------------------------\n");
}


bool libvori_InitializeVoronoi( CSnapshot *snap, int ifac ) {

	libvori_printf( LV_PL_INFO, "Initializing Voronoi tessellation...\n" );

	if (glv_pTetraPak != NULL)
		delete glv_pTetraPak;

	glv_pTetraPak = new CTetraPak();

	glv_pTetraPak->m_bIntegrateCharge = true;
	glv_pTetraPak->m_bIntegrateDipoleMoment = true;
	glv_pTetraPak->m_bIntegrateQuadrupoleMoment = true;
	glv_pTetraPak->m_bIntegrateTotalCurrent = false;
	glv_pTetraPak->m_bIntegrateMagneticMoment = false;

	if (!glv_pTetraPak->Init( snap, ifac ))
		return false;

	if (glv_bEMPOutput) {
		if (glv_bVoriOverwrite) {
			if (FileExist("properties.emp")) {
				FILE *a;
				a = fopen("properties.emp","wb");
				if (a == NULL) {
					libvori_printf( LV_PL_ERROR, "Error: Could not overwrite existing file \"properties.emp\".\n\n");
					return false;
				}
				fclose(a);
				libvori_printf( LV_PL_INFO, "  File \"properties.emp\" already exists, overwriting.\n");
			} else
				libvori_printf( LV_PL_INFO, "  File \"properties.emp\" does not yet exist, creating.\n");
		} else {
			if (FileExist("properties.emp"))
				libvori_printf( LV_PL_INFO, "  File \"properties.emp\" already exists, appending.\n");
			else
				libvori_printf( LV_PL_INFO, "  File \"properties.emp\" does not yet exist, creating.\n");
		}
	}

	libvori_printf( LV_PL_INFO, "Initialization done.\n" );

	libvori_printf( LV_PL_INFO, "----------------------------------------------------------------------\n");

	return true;
}



bool libvori_processStep( CSnapshot *snap, int step ) {

#ifndef _MSC_VER
	GetElapsedTime();
#endif

	if (!glv_bRadiiWritten) {
		libvori_PrintRadiusInfo( glv_pSnapshot );
		glv_bRadiiWritten = true;
	}

	if (!glv_pTetraPak->ProcessStep( snap, false ))
		return false;

	if (glv_bEMPOutput) {
		if ((glv_iStepCounter > 0) || (step <= 1)) {
			libvori_printf( LV_PL_INFO, "Writing binary data to \"properties.emp\"...\n" );
			if (!glv_pTetraPak->WriteEMPFrame( snap, "properties.emp", step ))
				return false;
		} else
			libvori_printf( LV_PL_INFO, "Not writing binary data in first step of MD restart.\n" );
	}

#ifndef _MSC_VER
	libvori_printf( LV_PL_INFO, "The Voronoi integration took %7.3f s.\n", GetElapsedTime() );
#endif

/*	FILE *a;
	int z;

	a = fopen("voronoi.txt","wt");

	fprintf(a,"#ID;  PX;  PY;  PZ;  WPX;  WPY;  WPZ;  Ord;  CoreCharge;  Charge;  DipX;  DipY;  DipZ;  QuadXX;  QuadXY;  QuadXZ;  QuadYX;  QuadYY;  QuadYZ;  QuadZX;  QuadZY;  QuadZZ\n");

	for (z=0;z<(int)snap->m_oaAtoms.size();z++)
		fprintf(
			a,
			"%d;  %f;  %f;  %f;  %f;  %f;  %f;  %d;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f;  %f\n",
			z+1,
			snap->m_oaAtoms[z]->m_vPos[0],
 			snap->m_oaAtoms[z]->m_vPos[1],
			snap->m_oaAtoms[z]->m_vPos[2],
			snap->m_oaAtoms[z]->m_vWrappedPos[0],
 			snap->m_oaAtoms[z]->m_vWrappedPos[1],
			snap->m_oaAtoms[z]->m_vWrappedPos[2],
			snap->m_oaAtoms[z]->m_iOrd,
			snap->m_oaAtoms[z]->m_fCoreCharge,
			snap->m_oaAtoms[z]->m_fCharge,
			snap->m_oaAtoms[z]->m_vDipole[0],
			snap->m_oaAtoms[z]->m_vDipole[1],
			snap->m_oaAtoms[z]->m_vDipole[2],
			snap->m_oaAtoms[z]->m_mQuadrupole(0,0),
			snap->m_oaAtoms[z]->m_mQuadrupole(0,1),
			snap->m_oaAtoms[z]->m_mQuadrupole(0,2),
			snap->m_oaAtoms[z]->m_mQuadrupole(1,0),
			snap->m_oaAtoms[z]->m_mQuadrupole(1,1),
			snap->m_oaAtoms[z]->m_mQuadrupole(1,2),
			snap->m_oaAtoms[z]->m_mQuadrupole(2,0),
			snap->m_oaAtoms[z]->m_mQuadrupole(2,1),
			snap->m_oaAtoms[z]->m_mQuadrupole(2,2)
		);

	fclose(a);*/

	glv_iStepCounter++;

	return true;
}



void libvori_printinfo( const char *s ) {

	#ifdef __GNUG__
		#pragma GCC diagnostic push
		#pragma GCC diagnostic ignored "-Wformat-security"
	#endif
	
	libvori_printf( LV_PL_INFO, s );

	#ifdef __GNUG__
		#pragma GCC diagnostic pop
	#endif
}



void libvori_printerror( const char *s ) {

	#ifdef __GNUG__
		#pragma GCC diagnostic push
		#pragma GCC diagnostic ignored "-Wformat-security"
	#endif

	libvori_printf( LV_PL_ERROR, s );

	#ifdef __GNUG__
		#pragma GCC diagnostic pop
	#endif
}




/******* Interface ********/



extern "C" int libvori_setBQBParmString( int len, const char *s ) {

	const char *p, *q;


	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setBQBParmString( %d, \"%s\" );\n", len, s );
	#endif

	if (!glv_bBQBInitialized) {

		if (glv_pBQBParmString != NULL)
			delete[] glv_pBQBParmString;

		p = s;
		while ((*p == ' ') && ((p-s) < len))
			p++;
		q = p;
		while ((*q != 0) && (*q != ' ') && (*q != '\r') && (*q != '\n') && (*q != '\t') && ((q-s) < len) && (q >= p))
			q++;
		if (q > p)
			q--;

		if (q == p) {
			glv_pBQBParmString = NULL;
			return 0;
		}

		glv_pBQBParmString = new char[q-p+3];
		glv_pBQBParmString[0] = '@';
		memcpy( &glv_pBQBParmString[1], p, q-p+1 );
		glv_pBQBParmString[q-p+2] = 0;
	}

	return 0;
}



extern "C" int libvori_setBQBFilename( int len, const char *s ) {

	const char *p, *q;
	const char *def = "density.bqb";
	bool ext_missing;


	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setBQBFilename( %d, \"%s\" );\n", len, s );
	#endif

	if (!glv_bBQBInitialized) {

		if (glv_pBQBFilename != NULL)
			delete[] glv_pBQBFilename;

		p = s;
		while (((*p == ' ') || (*p == '=')) && ((p-s) < len))
			p++;
		q = p;
		while ((*q != 0) && (*q != ' ') && (*q != '\r') && (*q != '\n') && (*q != '\t') && ((q-s) < len) && (q >= p))
			q++;
		if (q > p)
			q--;

		if (q == p) {
			glv_pBQBFilename = new char[strlen(def)+1];
			strcpy( glv_pBQBFilename, def );
			return 0;
		}

		ext_missing = true;

		if (q > (p+4))
			if ((tolower(*q) == 'b') && (tolower(*(q-1)) == 'q') && (tolower(*(q-2)) == 'b') && (*(q-3) == '.'))
				ext_missing = false;

		if (ext_missing) {
			glv_pBQBFilename = new char[q-p+6];
			memcpy( glv_pBQBFilename, p, q-p+1 );
			glv_pBQBFilename[q-p+1] = '.';
			glv_pBQBFilename[q-p+2] = 'b';
			glv_pBQBFilename[q-p+3] = 'q';
			glv_pBQBFilename[q-p+4] = 'b';
			glv_pBQBFilename[q-p+5] = 0;
		} else {
			glv_pBQBFilename = new char[q-p+2];
			memcpy( glv_pBQBFilename, p, q-p+1 );
			glv_pBQBFilename[q-p+1] = 0;
		}
	}

	return 0;
}



extern "C" int libvori_setBQBHistory( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setBQBHistory( %d );\n", i );
	#endif

	if (!glv_bBQBInitialized)
		glv_iBQBHistory = i;

	return 0;
}



extern "C" int libvori_setBQBCheck( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setBQBCheck( %d );\n", i );
	#endif

	if (!glv_bBQBInitialized) {
		if (i != 0)
			glv_bBQBCheck = true;
		else
			glv_bBQBCheck = false;
	}

	return 0;
}



extern "C" int libvori_setBQBStoreStep( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setBQBStoreStep( %d );\n", i );
	#endif

	if (!glv_bBQBInitialized) {
		if (i != 0)
			glv_bBQBStoreStep = true;
		else
			glv_bBQBStoreStep = false;
	}

	return 0;
}



extern "C" int libvori_setBQBOptimization( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setBQBOptimization( %d );\n", i );
	#endif

	if (!glv_bBQBInitialized)
		glv_iBQBOptimization = i;

	return 0;
}



extern "C" int libvori_processBQBFrame( int step, double t ) {

	CBQBAtomHeader *atomhead;
	CBQBCellInfo *cell;
	CBQBVolFrame *volframe;
	CBQBAtomPosFrame *atomframe;
	unsigned int z;
	char buf[128];


	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_processBQBFrame( %d, %.20E );\n", step, t );
	#endif

	if (!glv_bBQBInitialized) {

		glv_bBQBInitialized = true;

		libvori_printf( LV_PL_INFO, "Writing electron density to losslessly compressed BQB format.\n" );
		libvori_printf( LV_PL_INFO, "  File name:           \"%s\"\n", glv_pBQBFilename );
		libvori_printf( LV_PL_INFO, "  History:             %d\n", glv_iBQBHistory );
		libvori_printf( LV_PL_INFO, "  Skip first step:     %s\n", glv_bBQBSkipFirstFrame?"true":"false" );
		libvori_printf( LV_PL_INFO, "  Store step number:   %s\n", glv_bBQBStoreStep?"true":"false" );
		libvori_printf( LV_PL_INFO, "  De-compress check:   %s\n", glv_bBQBCheck?"true":"false" );
		libvori_printf( LV_PL_INFO, "  Optimization:        " );
		switch( glv_iBQBOptimization ) {
			case 0:
				libvori_printf( LV_PL_INFO, "None" );
				break;
			case 1:
				libvori_printf( LV_PL_INFO, "Quick" );
				break;
			case 2:
				libvori_printf( LV_PL_INFO, "Standard" );
				break;
			case 3:
				libvori_printf( LV_PL_INFO, "Patient" );
				break;
			case 4:
				libvori_printf( LV_PL_INFO, "Exhaustive" );
				break;
		}
		libvori_printf( LV_PL_INFO, "\n" );

		libvori_printf( LV_PL_INFO, "The history requires %s of buffer memory on MPI rank 0.\n",
			FormatBytes( ((double)glv_iBQBHistory+1)*glv_pSnapshot->m_iRes[0]*glv_pSnapshot->m_iRes[1]*glv_pSnapshot->m_iRes[2]*sizeof(double) ) );

		libvori_printf( LV_PL_INFO, "----------------------------------------------------------------------\n");

		libvori_printf( LV_PL_INFO, "Initializing LibBQB...\n" );

		if (glv_pBQBInterface == NULL)
			glv_pBQBInterface = BQBCreateInterface( 0 );

		glv_pBQBInterface->SetPrintCallback( &libvori_printinfo );
		glv_pBQBInterface->SetEPrintCallback( &libvori_printerror );

		glv_pBQBInterface->SetPrintLevel( BQB_PL_STANDARD );

		if (glv_bBQBOverwrite) {
			if (FileExist( glv_pBQBFilename ))
				libvori_printf( LV_PL_INFO, "File \"%s\" already exists, overwriting.\n", glv_pBQBFilename );
			else
				libvori_printf( LV_PL_INFO, "File \"%s\" does not yet exist, creating.\n", glv_pBQBFilename );
		} else {
			if (FileExist( glv_pBQBFilename ))
				libvori_printf( LV_PL_INFO, "File \"%s\" already exists, appending.\n", glv_pBQBFilename );
			else
				libvori_printf( LV_PL_INFO, "File \"%s\" does not yet exist, creating.\n", glv_pBQBFilename );
		}

		glv_pBQBFile = glv_pBQBInterface->CreateBQBFileWriter(
				glv_pBQBFilename,
				glv_iBQBHistory,
				BQB_CONTENTS_VOLUMETRIC | BQB_CONTENTS_ATOMPOS | BQB_CONTENTS_CELLINFO,
				glv_bBQBOverwrite?BQB_OVERWRITE:BQB_APPEND
			);

		if (glv_pBQBInterface->GetLastError() != 0) {
			libvori_printf( LV_PL_ERROR, "CBQBInterface::CreateBQBFileWriter() returned an error: %lu.\n", glv_pBQBInterface->GetLastError() );
			return 1;
		}

		glv_pBQBVolHead = glv_pBQBInterface->CreateVolumetricHeader();

		glv_pBQBVolHead->SetResolution( glv_pSnapshot->m_iRes[0], glv_pSnapshot->m_iRes[1], glv_pSnapshot->m_iRes[2] );

		glv_pBQBFile->SetVolumetricHeader( glv_pBQBVolHead );

		cell = glv_pBQBInterface->CreateCellInfo();

		if (glv_pSnapshot->m_bBoxNonOrtho)
			cell->SetGeneric(
				glv_pSnapshot->m_fCellVecA[0] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecA[1] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecA[2] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecB[0] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecB[1] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecB[2] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecC[0] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecC[1] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecC[2] / LEN_AU2PM,
				BQB_UNIT_BOHR
			);
		else
			cell->SetOrthorhombic(
				glv_pSnapshot->m_fCellVecA[0] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecB[1] / LEN_AU2PM,
				glv_pSnapshot->m_fCellVecC[2] / LEN_AU2PM,
				BQB_UNIT_BOHR
			);

		glv_pBQBFile->SetCellInfo( cell );

		atomhead = glv_pBQBInterface->CreateAtomHeader();

		atomhead->SetAtomCount( glv_pSnapshot->m_oaAtoms.size() );

		for (z=0;z<atomhead->GetAtomCount();z++)
			atomhead->SetAtomOrd( z, glv_pSnapshot->m_oaAtoms[z]->m_iOrd );

		glv_pBQBFile->SetAtomHeader( atomhead );

		if (glv_pTetraPak == NULL) {
			libvori_printf( LV_PL_INFO, "  Grid resolution: ( %d | %d | %d )\n",glv_pSnapshot->m_iRes[0],glv_pSnapshot->m_iRes[1],glv_pSnapshot->m_iRes[2]);
			libvori_printf( LV_PL_INFO, "  Volumetric cell matrix:\n");
			libvori_printf( LV_PL_INFO, "    A = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_fCellVecA[0],glv_pSnapshot->m_fCellVecA[1],glv_pSnapshot->m_fCellVecA[2]);
			libvori_printf( LV_PL_INFO, "    B = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_fCellVecB[0],glv_pSnapshot->m_fCellVecB[1],glv_pSnapshot->m_fCellVecB[2]);
			libvori_printf( LV_PL_INFO, "    C = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_fCellVecC[0],glv_pSnapshot->m_fCellVecC[1],glv_pSnapshot->m_fCellVecC[2]);
			libvori_printf( LV_PL_INFO, "  Simulation cell matrix (should match):\n");
			libvori_printf( LV_PL_INFO, "    A = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(0,0),glv_pSnapshot->m_mBoxFromOrtho(0,1),glv_pSnapshot->m_mBoxFromOrtho(0,2));
			libvori_printf( LV_PL_INFO, "    B = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(1,0),glv_pSnapshot->m_mBoxFromOrtho(1,1),glv_pSnapshot->m_mBoxFromOrtho(1,2));
			libvori_printf( LV_PL_INFO, "    C = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(2,0),glv_pSnapshot->m_mBoxFromOrtho(2,1),glv_pSnapshot->m_mBoxFromOrtho(2,2));

			if (glv_pSnapshot->m_bBoxNonOrtho)
				libvori_printf( LV_PL_INFO, "  Cell is non-orthorhombic.\n" );
			else
				libvori_printf( LV_PL_INFO, "  Cell is orthorhombic.\n" );
		}

		libvori_printf( LV_PL_INFO, "Finished initialization of LibBQB.\n" );

		if (glv_pBQBParmString != NULL) {
			libvori_printf( LV_PL_INFO, "Trying to import compression parameter key \"%s\" ...\n", glv_pBQBParmString );
			if (!glv_pBQBFile->m_pParmPosVol->FromKey(glv_pBQBParmString)) {
				libvori_printf( LV_PL_ERROR, "Error: Invalid parameter key: \"%s\".\n",glv_pBQBParmString);
				abort();
			}
			libvori_printf( LV_PL_INFO, "Using the following parameter set:\n" );
			libvori_printf( LV_PL_INFO, "\n" );
			libvori_printf( LV_PL_INFO, "%s",glv_pBQBFile->m_pParmPosVol->ToString(4).c_str() );
			libvori_printf( LV_PL_INFO, "\n" );
		}

		libvori_printf( LV_PL_INFO, "----------------------------------------------------------------------\n");

		if (glv_bBQBSkipFirstFrame) {
			libvori_printf( LV_PL_INFO, "Skipping first frame of MD run.\n" );
			return 0;
		}

	} // End of one-time initialization


	libvori_printf( LV_PL_INFO, "Processing electron density for compressed BQB storage...\n" );

	if ((glv_pSnapshot->m_iRes[0] != (int)glv_pBQBVolHead->GetResolutionX()) ||
		(glv_pSnapshot->m_iRes[1] != (int)glv_pBQBVolHead->GetResolutionY()) ||
		(glv_pSnapshot->m_iRes[2] != (int)glv_pBQBVolHead->GetResolutionZ())) {

		libvori_printf( LV_PL_ERROR, "Error: Grid resolution has changed. This is not yet supported by libbqb.\n\n" );
		return 1;
	}

	glv_iBQBStepCounter++;

	volframe = glv_pBQBInterface->CreateVolumetricFrame( glv_pBQBVolHead, step );

	volframe->LoadFromArray( glv_pSnapshot->m_pBin, BQB_ORDER_ZYX );

	glv_pBQBFile->PushVolumetricFrame( volframe );

	if (glv_bBQBStoreStep) {
		sprintf( buf, "ELECTRON DENSITY, Step %6d, Time %12.4f fs", step, t );
		glv_pBQBFile->PushCommentLine( "-Quickstep-", buf );
	} else
		glv_pBQBFile->PushCommentLine( "-Quickstep-", "ELECTRON DENSITY" );

	atomframe = glv_pBQBInterface->CreateAtomPosFrame( glv_pBQBFile->GetAtomHeader() );

	for ( z=0; z<atomframe->GetAtomCount(); z++ )
		atomframe->SetAtomPosition(
			z,
			glv_pSnapshot->m_oaAtoms[z]->m_vPos[0] / LEN_AU2PM,
			glv_pSnapshot->m_oaAtoms[z]->m_vPos[1] / LEN_AU2PM,
			glv_pSnapshot->m_oaAtoms[z]->m_vPos[2] / LEN_AU2PM,
			BQB_UNIT_BOHR
		);

	glv_pBQBFile->PushAtomPosFrame( atomframe );

	if (glv_iBQBStepCounter < glv_iBQBHistory)
		libvori_printf( LV_PL_INFO, "Caching frame %d / %d to fill optimization history buffer.\n", glv_iBQBStepCounter, glv_iBQBHistory );

	if ((glv_iBQBStepCounter == glv_iBQBHistory) && (glv_iBQBOptimization > 0)) {

		#ifndef _MSC_VER
			GetElapsedTime();
		#endif

		libvori_printf( LV_PL_INFO, "Optimizing compression parameters over first %d frames...\n", glv_iBQBHistory );
		libvori_printf( LV_PL_INFO, "To disable parameter optimization, specify \"OPTIMIZE OFF\" in the input.\n" );
		glv_pBQBFile->Optimize( glv_iBQBOptimization );
		libvori_printf( LV_PL_INFO, "Optimization finished.\n" );

		#ifndef _MSC_VER
			libvori_printf( LV_PL_INFO, "The parameter optimization took %9.3f s.\n", GetElapsedTime() );
		#endif

		libvori_printf( LV_PL_INFO, "To disable parameter optimization, specify \"OPTIMIZE OFF\" in the input.\n" );

		libvori_printf( LV_PL_INFO, "Writing %d pending frames to BQB file...\n", glv_iBQBHistory );
		libvori_printf( LV_PL_INFO, "----------------------------------------------------------------------\n");
	}

	if (glv_iBQBStepCounter >= glv_iBQBHistory) {

		#ifndef _MSC_VER
			GetElapsedTime();
		#endif

		if (!glv_pBQBFile->WritePendingFrames( glv_bBQBCheck )) {
			libvori_printf( LV_PL_ERROR, "CBQBFileWriter::WritePendingFrames() returned an error.\n" );
			return 1;
		}

		if (!glv_pBQBFile->WriteIndex()) {
			libvori_printf( LV_PL_ERROR, "CBQBFileWriter::WriteIndex() returned an error.\n" );
			return 1;
		}

		#ifndef _MSC_VER
			libvori_printf( LV_PL_INFO, "The BQB compression took %9.3f s.\n", GetElapsedTime() );
		#endif
	}

	return 0;
}



extern "C" int libvori_setPrefix_Voronoi() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setPrefix_Voronoi();\n");
	#endif

	lv_setPrintPrefix( " VORONOI| " );
	glv_bVoronoi = true;

	return 0;
}



extern "C" int libvori_setPrefix_BQB() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setPrefix_BQB();\n");
	#endif

	lv_setPrintPrefix( " BQB| " );
	glv_bVoronoi = false;

	return 0;
}



extern "C" int libvori_setVoronoiSkipFirst( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setVoronoiSkipFirst( %d );\n", i);
	#endif

	if (i != 0)
		glv_bVoronoiSkipFirstFrame = true;
	else
		glv_bVoronoiSkipFirstFrame = false;

	return 0;
}



extern "C" int libvori_setBQBSkipFirst( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setBQBSkipFirst( %d );\n", i );
	#endif

	if (i != 0)
		glv_bBQBSkipFirstFrame = true;
	else
		glv_bBQBSkipFirstFrame = false;

	return 0;
}



extern "C" int libvori_setRefinementFactor( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setRefinementFactor( %d );\n", i);
	#endif

	glv_iInterpolation = i;

	return 0;
}



extern "C" int libvori_setJitter( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setJitter( %d );\n", i);
	#endif

	if (i != 0)
		glv_bJitter = true;
	else
		glv_bJitter = false;

	return 0;
}



extern "C" int libvori_setJitterAmplitude( double f ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setJitterAmplitude( %.20E );\n", f);
	#endif

	glv_fJitterAmplitude = f;

	return 0;
}



extern "C" int libvori_setJitterSeed( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setJitterSeed( %d );\n", i);
	#endif

	glv_iJitterSeed = i;

	return 0;
}



extern "C" int libvori_setEMPOutput( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setEMPOutput( %d );\n", i);
	#endif

	if (i != 0)
		glv_bEMPOutput = true;
	else
		glv_bEMPOutput = false;
	
	return 0;
}



extern "C" int libvori_setBQBOverwrite( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setBQBOverwrite( %d );\n", i);
	#endif

	if (i != 0)
		glv_bBQBOverwrite = true;
	else
		glv_bBQBOverwrite = false;
	
	return 0;
}



extern "C" int libvori_setVoriOverwrite( int i ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setVoriOverwrite( %d );\n", i);
	#endif

	if (i != 0)
		glv_bVoriOverwrite = true;
	else
		glv_bVoriOverwrite = false;
	
	return 0;
}



extern "C" int libvori_setPrintPrefix( const char *s ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setEMPOutput( \"%s\" );\n", s);
	#endif

	lv_setPrintPrefix( s );

	return 0;
}



extern "C" int libvori_setPrintLevel_Silent() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setPrintLevel_Silent();\n");
	#endif

	glv_iPrintLevel = LV_PL_SILENT;

	return 0;
}



extern "C" int libvori_setPrintLevel_Error() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setPrintLevel_Error();\n");
	#endif

	glv_iPrintLevel = LV_PL_ERROR;

	return 0;
}



extern "C" int libvori_setPrintLevel_Warning() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setPrintLevel_Warning();\n");
	#endif

	glv_iPrintLevel = LV_PL_WARNING;

	return 0;
}



extern "C" int libvori_setPrintLevel_Info() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setPrintLevel_Info();\n");
	#endif

	glv_iPrintLevel = LV_PL_INFO;

	return 0;
}



extern "C" int libvori_setPrintLevel_Verbose() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setPrintLevel_Verbose();\n");
	#endif

	glv_iPrintLevel = LV_PL_VERBOSE;

	return 0;
}



extern "C" int libvori_setRadii_Unity() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setRadii_Unity();\n");
	#endif

	glv_pSnapshot->SetUnityRadii();

	return 0;
}



extern "C" int libvori_setRadii_VdW() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setRadii_VdW();\n");
	#endif

	glv_pSnapshot->SetVdWRadii();

	return 0;
}



extern "C" int libvori_setRadii_Covalent() {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setRadii_Covalent();\n");
	#endif

	glv_pSnapshot->SetCovalentRadii();

	return 0;
}



extern "C" int libvori_setRadii_User( double factor, const double *rad ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setRadii_User( %.20E, ... );\n", factor);
	#endif

	glv_pSnapshot->SetCustomRadii( factor, rad );

	return 0;
}



extern "C" int libvori_sanitycheck( int step, double t ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_sanitycheck( %d, %.20E );\n", step, t );
	#endif

	libvori_printf( LV_PL_VERBOSE, "libvori_sanitycheck() called. step=%d, t=%f\n",step,t);

	if (glv_iInitialized == 0) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_sanitycheck(): LibVori not initialized.\n");
		return 1;
	}

	if (!glv_bRadiiWritten) {
		libvori_PrintRadiusInfo( glv_pSnapshot );
		glv_bRadiiWritten = true;
	}

	if (!glv_pTetraPak->ProcessStep( glv_pSnapshot, true ))
		return 1;

	libvori_printf( LV_PL_VERBOSE, "libvori_sanitycheck() done.\n");

	return 0;
}



extern "C" int libvori_step( int step, double t ) {

	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_step( %d, %.20E );\n", step, t );
	#endif

	libvori_printf( LV_PL_VERBOSE, "libvori_step() called. step=%d, t=%f\n",step,t);

	if (glv_iInitialized == 0) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_step(): LibVori not initialized.\n");
		return 1;
	}

	glv_iVoronoiStepCounter++;

	if (glv_bVoronoiSkipFirstFrame && (glv_iVoronoiStepCounter == 1)) {
		libvori_printf( LV_PL_INFO, "Skipping first frame of MD run.\n" );
		return 0;
	}

	if (!libvori_processStep( glv_pSnapshot, step ))
		return 1;

	libvori_printf( LV_PL_VERBOSE, "libvori_step() done.\n");

	return 0;
}



extern "C" int libvori_setGrid(
		int rx,
		int ry,
		int rz,
		double ax,
		double ay,
		double az,
		double bx,
		double by,
		double bz,
		double cx,
		double cy,
		double cz,
		double tax,
		double tay,
		double taz,
		double tbx,
		double tby,
		double tbz,
		double tcx,
		double tcy,
		double tcz
	) {

	bool init;


	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_setGrid( %d, %d, %d, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E, %.20E );\n", rx, ry, rz, ax, ay, az, bx, by, bz, cx, cy, cz, tax, tay, taz, tbx, tby, tbz, tcx, tcy, tcz );
	#endif

	libvori_printf( LV_PL_VERBOSE, "libvori_setgrid() called.\n");

	init = false;

	if (glv_iInitialized == 0) {

		if (!libvori_Initialize())
			return 1;

		glv_iInitialized = 1;
		init = true;

		glv_pSnapshot = new CSnapshot();
		glv_pSnapshot->m_iRes[0] = rx;
		glv_pSnapshot->m_iRes[1] = ry;
		glv_pSnapshot->m_iRes[2] = rz;
		glv_pSnapshot->m_pBin = new double[rx*ry*rz];
	}

	glv_pSnapshot->m_mBoxFromOrtho(0,0) = tax * LEN_AU2PM;
	glv_pSnapshot->m_mBoxFromOrtho(0,1) = tay * LEN_AU2PM;
	glv_pSnapshot->m_mBoxFromOrtho(0,2) = taz * LEN_AU2PM;
	glv_pSnapshot->m_mBoxFromOrtho(1,0) = tbx * LEN_AU2PM;
	glv_pSnapshot->m_mBoxFromOrtho(1,1) = tby * LEN_AU2PM;
	glv_pSnapshot->m_mBoxFromOrtho(1,2) = tbz * LEN_AU2PM;
	glv_pSnapshot->m_mBoxFromOrtho(2,0) = tcx * LEN_AU2PM;
	glv_pSnapshot->m_mBoxFromOrtho(2,1) = tcy * LEN_AU2PM;
	glv_pSnapshot->m_mBoxFromOrtho(2,2) = tcz * LEN_AU2PM;

	glv_pSnapshot->m_mBoxToOrtho = CxDMatrix3(glv_pSnapshot->m_mBoxFromOrtho);
	if (!glv_pSnapshot->m_mBoxToOrtho.Invert()) {
		libvori_printf( LV_PL_ERROR, "Error: Encountered singular cell matrix (cell volume is zero).\n");
		return 1;
	}

	libvori_printf( LV_PL_VERBOSE, "  Volumetric Resolution:\n");
	libvori_printf( LV_PL_VERBOSE, "     ( %d | %d | %d )\n",rx,ry,rz);
	libvori_printf( LV_PL_VERBOSE, "  Volumetric Cell Size:\n");
	libvori_printf( LV_PL_VERBOSE, "     A = ( %16.8f %16.8f %16.8f ) pm\n",ax*LEN_AU2PM,ay*LEN_AU2PM,az*LEN_AU2PM);
	libvori_printf( LV_PL_VERBOSE, "     B = ( %16.8f %16.8f %16.8f ) pm\n",bx*LEN_AU2PM,by*LEN_AU2PM,bz*LEN_AU2PM);
	libvori_printf( LV_PL_VERBOSE, "     C = ( %16.8f %16.8f %16.8f ) pm\n",cx*LEN_AU2PM,cy*LEN_AU2PM,cz*LEN_AU2PM);
	libvori_printf( LV_PL_VERBOSE, "  Simulation Cell Size (should match):\n");
	libvori_printf( LV_PL_VERBOSE, "     A = ( %16.8f %16.8f %16.8f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(0,0),glv_pSnapshot->m_mBoxFromOrtho(0,1),glv_pSnapshot->m_mBoxFromOrtho(0,2));
	libvori_printf( LV_PL_VERBOSE, "     B = ( %16.8f %16.8f %16.8f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(1,0),glv_pSnapshot->m_mBoxFromOrtho(1,1),glv_pSnapshot->m_mBoxFromOrtho(1,2));
	libvori_printf( LV_PL_VERBOSE, "     C = ( %16.8f %16.8f %16.8f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(2,0),glv_pSnapshot->m_mBoxFromOrtho(2,1),glv_pSnapshot->m_mBoxFromOrtho(2,2));

	glv_pSnapshot->m_fCellVecA[0] = ax * LEN_AU2PM;
	glv_pSnapshot->m_fCellVecA[1] = ay * LEN_AU2PM;
	glv_pSnapshot->m_fCellVecA[2] = az * LEN_AU2PM;
	glv_pSnapshot->m_fCellVecB[0] = bx * LEN_AU2PM;
	glv_pSnapshot->m_fCellVecB[1] = by * LEN_AU2PM;
	glv_pSnapshot->m_fCellVecB[2] = bz * LEN_AU2PM;
	glv_pSnapshot->m_fCellVecC[0] = cx * LEN_AU2PM;
	glv_pSnapshot->m_fCellVecC[1] = cy * LEN_AU2PM;
	glv_pSnapshot->m_fCellVecC[2] = cz * LEN_AU2PM;

	glv_pSnapshot->m_fStrideA[0] = ax / rx * LEN_AU2PM;
	glv_pSnapshot->m_fStrideA[1] = ay / ry * LEN_AU2PM;
	glv_pSnapshot->m_fStrideA[2] = az / rz * LEN_AU2PM;
	glv_pSnapshot->m_fStrideB[0] = bx / rx * LEN_AU2PM;
	glv_pSnapshot->m_fStrideB[1] = by / ry * LEN_AU2PM;
	glv_pSnapshot->m_fStrideB[2] = bz / rz * LEN_AU2PM;
	glv_pSnapshot->m_fStrideC[0] = cx / rx * LEN_AU2PM;
	glv_pSnapshot->m_fStrideC[1] = cy / ry * LEN_AU2PM;
	glv_pSnapshot->m_fStrideC[2] = cz / rz * LEN_AU2PM;

	if ((ay != 0) || (az != 0) || (bx != 0) || (bz != 0) || (cx != 0) || (cy != 0))
		glv_pSnapshot->m_bBoxNonOrtho = true;
	else
		glv_pSnapshot->m_bBoxNonOrtho = false;

	if ((rx != glv_pSnapshot->m_iRes[0]) || (ry != glv_pSnapshot->m_iRes[1]) || (rz != glv_pSnapshot->m_iRes[2])) {

		libvori_printf( LV_PL_WARNING, "Grid resolution has changed from (%d|%d|%d) to (%d|%d|%d).\n",glv_pSnapshot->m_iRes[0],glv_pSnapshot->m_iRes[1],glv_pSnapshot->m_iRes[2],rx,ry,rz);

		libvori_printf( LV_PL_WARNING, "Re-initializing...\n");

		if (glv_pSnapshot->m_pBin != NULL)
			delete[] glv_pSnapshot->m_pBin;

		glv_pSnapshot->m_iRes[0] = rx;
		glv_pSnapshot->m_iRes[1] = ry;
		glv_pSnapshot->m_iRes[2] = rz;
		glv_pSnapshot->m_pBin = new double[rx*ry*rz];

		init = true;
	}

	if (init && glv_bVoronoi)
		if (!libvori_InitializeVoronoi( glv_pSnapshot, glv_iInterpolation ))
			return 1;

	libvori_printf( LV_PL_VERBOSE, "libvori_setgrid() done.\n");

	return 0;
}



extern "C" int libvori_pushAtoms(
		int n,
		const int *pord,
		const double *pchg,
		const double *posx,
		const double *posy,
		const double *posz
	) {

	int z, z2;
	CSnapshotAtom *sa;
	CxDVector3 tv;


	#ifdef DEBUG_INTERFACE
		printf("@@  int pord[%d] = {",n);
		for (z=0;z<n;z++) {
			if (z != 0)
				printf(",");
			printf(" %d",pord[z]);
		}
		printf(" };\n");
		printf("@@  double pchg[%d] = {",n);
		for (z=0;z<n;z++) {
			if (z != 0)
				printf(",");
			printf(" %.20E",pchg[z]);
		}
		printf(" };\n");
		printf("@@  double posx[%d] = {",n);
		for (z=0;z<n;z++) {
			if (z != 0)
				printf(",");
			printf(" %.20E",posx[z]);
		}
		printf(" };\n");
		printf("@@  double posy[%d] = {",n);
		for (z=0;z<n;z++) {
			if (z != 0)
				printf(",");
			printf(" %.20E",posy[z]);
		}
		printf(" };\n");
		printf("@@  double posz[%d] = {",n);
		for (z=0;z<n;z++) {
			if (z != 0)
				printf(",");
			printf(" %.20E",posz[z]);
		}
		printf(" };\n");
		printf("@@  libvori_pushAtoms( %d, pord, pchg, posx, posy, posz );\n", n );
	#endif

	libvori_printf( LV_PL_VERBOSE, "libvori_pushatoms() called with %d atoms.\n",n);
	for (z=0;z<MIN(10,n);z++)
		libvori_printf( LV_PL_VERBOSE, "     %5d  %3d  %12.6f  %12.6f  %12.6f  %12.6f\n",z+1,pord[z],pchg[z],posx[z]*LEN_AU2PM,posy[z]*LEN_AU2PM,posz[z]*LEN_AU2PM);

	if (glv_iInitialized == 0) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_pushatoms(): LibVori not initialized.\n");
		return 1;
	}

	if (glv_pSnapshot->m_oaAtoms.size() == 0) {

		for (z=0;z<n;z++) {
			sa = new CSnapshotAtom();
			sa->m_vPos[0] = posx[z] * LEN_AU2PM;
			sa->m_vPos[1] = posy[z] * LEN_AU2PM;
			sa->m_vPos[2] = posz[z] * LEN_AU2PM;
			sa->m_fCoreCharge = pchg[z];
			sa->m_iOrd = pord[z];

			for (z2=0;z2<(int)glv_oaElements.size();z2++) {
				if (glv_oaElements[z2]->m_iOrd == sa->m_iOrd) {
					sa->m_sLabel = glv_oaElements[z2]->m_sLabel;
					goto _done;
				}
			}
			libvori_printf( LV_PL_WARNING, "Warning: Did not find element entry for ordinal number %d, using \"X\" as label.\n", sa->m_iOrd );
			sa->m_sLabel = "X";
_done:

			glv_pSnapshot->m_oaAtoms.push_back( sa );
		}

		glv_pSnapshot->SetVdWRadii();

	} else {

		if (n != (int)glv_pSnapshot->m_oaAtoms.size()) {
			libvori_printf( LV_PL_ERROR, "Error: libvori_pushatoms(): Atom count has changed.\n");
			return 1;
		}

		for (z=0;z<n;z++) {
			glv_pSnapshot->m_oaAtoms[z]->m_vPos[0] = posx[z] * LEN_AU2PM;
			glv_pSnapshot->m_oaAtoms[z]->m_vPos[1] = posy[z] * LEN_AU2PM;
			glv_pSnapshot->m_oaAtoms[z]->m_vPos[2] = posz[z] * LEN_AU2PM;
		}
	}

	for (z=0;z<n;z++) {
		tv = glv_pSnapshot->m_mBoxToOrtho * glv_pSnapshot->m_oaAtoms[z]->m_vPos;
		tv[0] = tv[0] - floor(tv[0]);
		tv[1] = tv[1] - floor(tv[1]);
		tv[2] = tv[2] - floor(tv[2]);
		glv_pSnapshot->m_oaAtoms[z]->m_vWrappedPos = glv_pSnapshot->m_mBoxFromOrtho * tv;
	}

	libvori_printf( LV_PL_VERBOSE, "libvori_pushatoms() done.\n");

	return 0;
}



extern "C" int libvori_push_rho_zrow( int ix, int iy, int length, const double *buf ) {

	int z;


	if (glv_iInitialized == 0) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_push_rho_zrow(): LibVori not initialized.\n");
		return -1;
	}

	if (length != glv_pSnapshot->m_iRes[2]) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_push_rho_zrow(): Length mismatch (%d vs %d).\n",length,glv_pSnapshot->m_iRes[2]);
		return -1;
	}

	for (z=0;z<glv_pSnapshot->m_iRes[2];z++)
		glv_pSnapshot->m_pBin[z*glv_pSnapshot->m_iRes[1]*glv_pSnapshot->m_iRes[0]+iy*glv_pSnapshot->m_iRes[0]+ix] = buf[z];

	return 0;
}



extern "C" int libvori_get_radius( int length, double *rad ) {

	int z;

	if (length != (int)glv_pSnapshot->m_oaAtoms.size()) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_get_radius(): Length mismatch (%d vs %lu).\n",length,(unsigned long)glv_pSnapshot->m_oaAtoms.size());
		return -1;
	}

	for (z=0;z<(int)glv_pSnapshot->m_oaAtoms.size();z++)
		rad[z] = glv_pSnapshot->m_oaAtoms[z]->m_fRadius;

	return 0;
}



extern "C" int libvori_get_volume( int length, double *vol ) {

	int z;

	if (length != (int)glv_pSnapshot->m_oaAtoms.size()) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_get_volume(): Length mismatch (%d vs %lu).\n",length,(unsigned long)glv_pSnapshot->m_oaAtoms.size());
		return -1;
	}

	for (z=0;z<(int)glv_pSnapshot->m_oaAtoms.size();z++)
		vol[z] = glv_pSnapshot->m_oaAtoms[z]->m_fVolume;

	return 0;
}



extern "C" int libvori_get_charge( int length, double *buf ) {

	int z;

	if (length != (int)glv_pSnapshot->m_oaAtoms.size()) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_get_charge(): Length mismatch (%d vs %lu).\n",length,(unsigned long)glv_pSnapshot->m_oaAtoms.size());
		return -1;
	}

	for (z=0;z<(int)glv_pSnapshot->m_oaAtoms.size();z++)
		buf[z] = glv_pSnapshot->m_oaAtoms[z]->m_fCharge;

	return 0;
}



extern "C" int libvori_get_dipole( int component, int length, double *buf ) {

	int z;

	if (length != (int)glv_pSnapshot->m_oaAtoms.size()) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_get_dipole(): Length mismatch (%d vs %lu).\n",length,(unsigned long)glv_pSnapshot->m_oaAtoms.size());
		return -1;
	}

	for (z=0;z<(int)glv_pSnapshot->m_oaAtoms.size();z++)
		buf[z] = glv_pSnapshot->m_oaAtoms[z]->m_vDipole[component-1];

	return 0;
}



extern "C" int libvori_get_quadrupole( int component, int length, double *buf ) {

	int z;

	if (length != (int)glv_pSnapshot->m_oaAtoms.size()) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_get_quadrupole(): Length mismatch (%d vs %lu).\n",length,(unsigned long)glv_pSnapshot->m_oaAtoms.size());
		return -1;
	}

	for (z=0;z<(int)glv_pSnapshot->m_oaAtoms.size();z++)
		buf[z] = glv_pSnapshot->m_oaAtoms[z]->m_mQuadrupole[component-1];

	return 0;
}



extern "C" int libvori_get_wrapped_pos( int component, int length, double *buf ) {

	int z;

	if (length != (int)glv_pSnapshot->m_oaAtoms.size()) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_get_wrapped_pos(): Length mismatch (%d vs %lu).\n",length,(unsigned long)glv_pSnapshot->m_oaAtoms.size());
		return -1;
	}

	for (z=0;z<(int)glv_pSnapshot->m_oaAtoms.size();z++)
		buf[z] = glv_pSnapshot->m_oaAtoms[z]->m_vWrappedPos[component-1];

	return 0;
}



extern "C" int libvori_get_charge_center( int component, int length, double *buf ) {

	int z;

	if (length != (int)glv_pSnapshot->m_oaAtoms.size()) {
		libvori_printf( LV_PL_ERROR, "Error: libvori_get_charge_center(): Length mismatch (%d vs %lu).\n",length,(unsigned long)glv_pSnapshot->m_oaAtoms.size());
		return -1;
	}

	for (z=0;z<(int)glv_pSnapshot->m_oaAtoms.size();z++)
		buf[z] = glv_pSnapshot->m_oaAtoms[z]->m_vChargeCenter[component-1];

	return 0;
}



extern "C" int libvori_finalize() {

	int z;


	#ifdef DEBUG_INTERFACE
		printf("@@  libvori_finalize();\n" );
	#endif

	if (glv_iInitialized != 0) {

		if (LV_PL_INFO <= glv_iPrintLevel)
			printf( "\n" );
		libvori_printf( LV_PL_INFO, "Cleaning up...\n");

		if (glv_pSnapshot != NULL) {
			delete glv_pSnapshot;
			glv_pSnapshot = NULL;
		}

		if (glv_pTetraPak != NULL) {
			delete glv_pTetraPak;
			glv_pTetraPak = NULL;
		}

		if (glv_pBQBFile != NULL)
			glv_pBQBFile->CloseFile();

		if (glv_pBQBInterface != NULL) {
			BQBReleaseInterface( glv_pBQBInterface );
			glv_pBQBInterface = NULL;
		}

		for (z=0;z<(int)glv_oaElements.size();z++)
			delete glv_oaElements[z];

		if (glv_pPrintPrefix != NULL) {
			delete[] glv_pPrintPrefix;
			glv_pPrintPrefix = NULL;
		}
	}

	return 0;
}



/******* Test Main Program ********/


#ifdef TEST 


int main( int argc, char *argv[] ) {

	UNUSED(argc);
	UNUSED(argv);

	int i, x, y, z;
	double buf[90];


	libvori_setPrefix_Voronoi();

	libvori_setRefinementFactor( 1 );

	libvori_setVoronoiSkipFirst( 0 );

	libvori_setVoriOverwrite( 0 );

	libvori_setEMPOutput( 1 );

	libvori_setBQBFilename( 64, "=out.bqb                                                                   " );

	libvori_setBQBParmString( 64, "                                                                             " );

	libvori_setBQBOptimization( 1 );

	libvori_setBQBHistory( 1 );

	libvori_setBQBSkipFirst( 0 );

	libvori_setBQBCheck( 1 );

	libvori_setBQBOverwrite( 0 );

	libvori_setBQBStoreStep( 1 );

	libvori_setGrid( 90, 90, 90, 1.88972613288564303957E+01, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 1.88972613288564303957E+01, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 1.88972613288564303957E+01, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 1.88972613288564303957E+01 );

	int pord[6] = { 8, 1, 1, 8, 1, 1 };
	double pchg[6] = { 6.00000000000000000000E+00, 1.00000000000000000000E+00, 1.00000000000000000000E+00, 6.00000000000000000000E+00, 1.00000000000000000000E+00, 1.00000000000000000000E+00 };
	double posx[6] = { -3.88006128508611780603E+00, -2.51522548287079050766E+00, -3.44880688430028481406E+00, 2.46382493205630126454E+00, 2.75697814705085120224E+00, 2.89143216140566483574E+00 };
	double posy[6] = { 7.19362047005577709413E-01, 1.68283891585732270357E+00, -1.07430930654548806302E+00, 1.15893124277610715112E+00, -3.64150225807063410155E-01, 2.59151372685538428087E+00 };
	double posz[6] = { -3.08252126796306036294E-01, 5.34490139425375265780E-01, 2.30546588212048432845E-03, 1.64646168779927415393E+00, 5.95566088040239249857E-01, 5.24871433408987342339E-01 };
	libvori_pushAtoms( 6, pord, pchg, posx, posy, posz );

	for (y=0;y<90;y++) {
		for (x=0;x<90;x++) {
			for (z=0;z<90;z++)
				buf[z] = cos( x/90.0*2.0*Pi ) * cos( y/90.0*4.0*Pi ) * cos( z/90.0*6.0*Pi );
			libvori_push_rho_zrow( x, y, 90, buf );
		}
	}

	libvori_setRadii_VdW();
//	double radii[6] = { 152.0, 109.0, 109.0, 152.0, 109.0, 109.5 };
//	libvori_setRadii_User( 1.0, radii );

	i = libvori_sanitycheck( 0, 0.00000000000000000000E+00 );

	if (i != 0)
		goto _ende;

	i = libvori_step( 0, 0.00000000000000000000E+00 );

	if (i != 0)
		goto _ende;

	i = libvori_processBQBFrame( 0, 0.00000000000000000000E+00 );

_ende:
	libvori_finalize();


	return i;
}


#endif


/*
int main( int argc, char *argv[] ) {

	UNUSED(argc);
	UNUSED(argv);

	libvori_setPrefix_Voronoi();

	libvori_setRefinementFactor( 1 );

	libvori_setVoronoiSkipFirst( 0 );

	libvori_setVoriOverwrite( 0 );

	libvori_setEMPOutput( 0 );

	libvori_setGrid( 243, 243, 243, 3.77945226577128607914E+01, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 3.77945226577128607914E+01, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 3.77945226577128607914E+01, 3.77945226577128607914E+01, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 3.77945226577128607914E+01, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 0.00000000000000000000E+00, 3.77945226577128607914E+01 );

	int pord[79] = { 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79 };

	double pchg[79] = { 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01, 1.10000000000000000000E+01 };

	double posx[79] = { 1.88972613288564303957E+01, 1.50895145085258199913E+01, 1.88972613288564303957E+01, 2.27050081489980648541E+01, 1.88972613288564303957E+01, 1.12200194978544640634E+01, 2.65745031598583913990E+01, 2.32818050097572140089E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 1.45127176479556467825E+01, 1.45127176479556467825E+01, 1.88972613288564303957E+01, 1.07640071182907686165E+01, 2.70305155394220903986E+01, 1.45127176479556467825E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 2.27986075911055117160E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.45127176479556467825E+01, 1.07640071182907686165E+01, 3.01848425933925383902E+01, 2.32818050097572140089E+01, 7.60968006432032240127E+00, 1.49959150662294007361E+01, 2.70305155394220903986E+01, 1.88972613288564303957E+01, 7.60968006432032240127E+00, 2.27986075914834565026E+01, 2.27050081491870372474E+01, 2.65745031598583913990E+01, 1.88972613288564303957E+01, 2.63618290590597759149E+01, 3.01848425933925383902E+01, 3.01848425933925383902E+01, 1.88972613288564303957E+01, 2.27986075914834565026E+01, 2.65745031598583913990E+01, 1.49959150664183731294E+01, 1.14326935992199985037E+01, 1.12200194978544640634E+01, 1.50895145085258199913E+01, 1.88972613288564303957E+01, 1.49959150664183731294E+01, 2.27050081491870372474E+01, 7.60968006413135000798E+00, 1.50895145085258199913E+01, 1.12200194980434382330E+01, 2.70305155394220903986E+01, 2.32818050097572140089E+01, 1.07640071182907686165E+01, 2.70305155394220903986E+01, 1.07640071182907686165E+01, 1.45127176479556467825E+01, 7.60968006413135000798E+00, 3.01848425935815107835E+01, 1.88972613288564303957E+01, 2.27986075912944841093E+01, 1.88972613288564303957E+01, 1.45127176479556467825E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 1.88972613288564303957E+01, 1.49959150664183731294E+01, 2.32818050097572140089E+01, 1.45127176479556467825E+01, 1.88972613288564303957E+01, 1.45127176479556467825E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 1.88972613288564303957E+01, 1.12200194978544640634E+01, 2.65745031598583913990E+01, 1.88972613288564303957E+01, 2.27050081491870372474E+01, 1.50895145085258199913E+01, 1.88972613288564303957E+01 };

	double posy[79] = { 2.27050081491870372474E+01, 1.88972613288564303957E+01, 1.50895145087147941609E+01, 1.88972613288564303957E+01, 2.65745031598583913990E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.45127176479556467825E+01, 1.12200194978544640634E+01, 2.32818050097572140089E+01, 1.45127176479556467825E+01, 2.32818050097572140089E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 1.45127176479556467825E+01, 2.70305155394220903986E+01, 3.01848425933925383902E+01, 2.70305155394220903986E+01, 1.88972613288564303957E+01, 7.60968006432032240127E+00, 2.27986075912944841093E+01, 1.07640071182907686165E+01, 1.45127176479556467825E+01, 1.88972613288564303957E+01, 1.07640071182907686165E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 1.49959150662294007361E+01, 1.50895145085258199913E+01, 2.27986075914834565026E+01, 3.01848425933925383902E+01, 1.12200194978544640634E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 2.27050081491870372474E+01, 1.50895145085258199913E+01, 2.63618290586818311283E+01, 1.49959150662294007361E+01, 2.65745031598583913990E+01, 2.27986075912944841093E+01, 1.88972613288564303957E+01, 1.12200194978544640634E+01, 7.60968006432032240127E+00, 1.14326935992199985037E+01, 1.49959150664183731294E+01, 7.60968006432032240127E+00, 2.27050081491870372474E+01, 3.01848425935815107835E+01, 2.65745031596694190057E+01, 1.45127176479556467825E+01, 2.70305155394220903986E+01, 2.32818050097572140089E+01, 2.32818050097572140089E+01, 1.45127176479556467825E+01, 1.07640071182907686165E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 3.01848425935815107835E+01, 1.88972613288564303957E+01, 7.60968006413135000798E+00, 2.70305155394220903986E+01, 2.27986075912944841093E+01, 1.07640071182907686165E+01, 1.49959150664183731294E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 1.45127176479556467825E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 1.12200194978544640634E+01, 1.45127176479556467825E+01, 2.65745031598583913990E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 2.27050081491870372474E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.50895145085258199913E+01 };

	double posz[79] = { 7.60968006394237761469E+00, 7.60968006413135000798E+00, 7.60968006413135000798E+00, 7.60968006432032240127E+00, 1.12200194978544640634E+01, 1.12200194978544640634E+01, 1.12200194978544640634E+01, 1.07640071182907686165E+01, 1.12200194978544640634E+01, 1.07640071182907686165E+01, 1.07640071182907686165E+01, 1.07640071182907686165E+01, 1.14326935984641089306E+01, 1.45127176479556467825E+01, 1.45127176479556467825E+01, 1.45127176479556467825E+01, 1.50895145087147941609E+01, 1.45127176479556467825E+01, 1.49959150666073455227E+01, 1.50895145087147941609E+01, 1.49959150664183731294E+01, 1.45127176479556467825E+01, 1.45127176479556467825E+01, 1.50895145087147941609E+01, 1.45127176479556467825E+01, 1.50895145085258199913E+01, 1.49959150662294007361E+01, 1.45127176479556467825E+01, 1.49959150662294007361E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 1.88972613288564303957E+01, 2.32818050097572140089E+01, 2.32818050097572140089E+01, 2.32818050097572140089E+01, 2.32818050097572140089E+01, 2.32818050097572140089E+01, 2.32818050097572140089E+01, 2.27050081491870372474E+01, 2.27050081491870372474E+01, 2.27050081491870372474E+01, 2.27986075912944841093E+01, 2.27050081491870372474E+01, 2.32818050097572140089E+01, 2.27986075912944841093E+01, 2.32818050097572140089E+01, 2.27986075912944841093E+01, 2.27986075912944841093E+01, 2.70305155394220903986E+01, 2.70305155394220903986E+01, 2.63618290586818311283E+01, 2.70305155394220903986E+01, 2.65745031598583913990E+01, 2.70305155394220903986E+01, 2.65745031598583913990E+01, 2.65745031598583913990E+01, 2.65745031598583913990E+01, 3.01848425935815107835E+01, 3.01848425935815107835E+01, 3.01848425935815107835E+01, 3.01848425933925383902E+01 };

	libvori_pushAtoms( 79, pord, pchg, posx, posy, posz );

	libvori_setRadii_VdW();

	int fail=0, run=100;

	for (int z=0;z<run;z++) {
		srand(z*123);
		if (libvori_sanitycheck( 0, 0.00000000000000000000E+00 ) != 0)
			fail++;
	}

	printf("\n\n###################\n");
	printf("%d out of %d tests failed.\n",fail,run);

	return 0;
}
*/


/*
int main( int argc, char *argv[] ) {

	// @OYNm20Ot100WKCf0A6rkm0JZo8kB2
	// @OYNm20Ot100WKCf0A6rkm0JZo8kB2

	// @OYNm20Ot100WKCf0A6rkm0JZo8kB2
	// @OYNm20Ot100WKCf0A6rkm0JZo8kB2

	glv_pBQBInterface = BQBCreateInterface( 0 );

	CBQBParameterSet_PosAndVol pov(*glv_pBQBInterface);

	pov.FromKey( "@OYNm20Ot100WKCf0A6rkm0JZo8kB2" );

	printf( "%s\n", pov.ToString(4).c_str() );

	printf( "Key: %s\n", pov.ToKey().c_str() );


	pov.SetPosBlockLength( 40 );
	pov.SetPosMaxIter( 10 );

	printf( "%s\n", pov.ToString(4).c_str() );

	printf( "Key: %s\n", pov.ToKey().c_str() );


	return 0;
}
*/


/*
int main( int argc, char *argv[] ) {

	libvori_setPrintPrefix( "VORONOI| " );

	libvori_setPrintLevel_Verbose();

	AddElementData();

	glv_pSnapshot = new CSnapshot();

	if (!glv_pSnapshot->ReadCube( "snapshot.cube" )) {
		libvori_printf( LV_PL_ERROR, "Error: Reading cube file failed.\n");
		abort();
	}

	glv_pSnapshot->SetVdWRadii();

	libvori_InitializeVoronoi( glv_pSnapshot, 1 );

	libvori_sanityCheck( glv_pSnapshot );

	libvori_processStep( glv_pSnapshot );


	if (glv_pTetraPak != NULL) {
		delete glv_pTetraPak;
		glv_pTetraPak = NULL;
	}

	if (glv_pSnapshot != NULL) {
		delete glv_pSnapshot;
		glv_pSnapshot = NULL;
	}

	return 0;
}
*/



