//>>>BETA
/***********************************************************************************

    LibBQB - File Format and Compression Algorithms for Trajectories of
             Volumetric Data and Atom Positions

    https://brehm-research.de/bqb

    Free software, licensed under GNU LGPL v3

    Copyright (c) Martin Brehm and Martin Thomas,
                  Martin Luther University Halle-Wittenberg, Germany,
                  2016 - 2021.

    Please cite:  M. Brehm, M. Thomas: "An Efficient Lossless Compression Algorithm
                  for Trajectories of Atom Positions and Volumetric Data",
                  J. Chem. Inf. Model. 2018, 58 (10), pp 2092-2107.

    --------------------------------------------------------------------------------

    LibBQB is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/



// This must always be the first include directive
#include "bqb_config.h"

#include "bqb_arithmeticcoding_model.h"



const char *GetRevisionInfo_bqb_arithmeticcoding_model(unsigned int len) {
	static char buf[256];
	GET_REVISION_INFO( buf, len );
	return buf;
}


const char *GetSourceVersion_bqb_arithmeticcoding_model() {
	static char buf[256];
	GET_SOURCE_VERSION( buf );
	return buf;
}



#ifdef BQI_DEVELOPMENT



CBQBACProbability CBQBACModel1::GetSymbol( BQB_AC_CODE_VALUE scaled_value, unsigned int &sym ) {

	unsigned int z;


	for ( z=0; z<m_iSymbolRange; z++ ) {

		if (scaled_value < m_iaCumulativeCount[z+1]) {

			sym = z;

			CBQBACProbability prob = CBQBACProbability(
				m_iaCumulativeCount[z],
				m_iaCumulativeCount[z+1],
				m_iSum
			);

			Update( sym );

			return prob;
		}
	}

	printf( "CBQBACModel1::GetSymbol(): Internal error: %lu >= %lu.\n", scaled_value, m_iSum );
	abort();
}



void CBQBACModel1::Update( unsigned int sym ) {

	UNUSED(sym);

}



void CBQBACModel1::InitUniform() {

	unsigned int z;


	m_iaCount.resize( m_iSymbolRange );
	m_iaCumulativeCount.resize( m_iSymbolRange+1 );

	for (z=0;z<m_iSymbolRange;z++)
		m_iaCount[z] = 1;

	for (z=0;z<=m_iSymbolRange;z++)
		m_iaCumulativeCount[z] = z;

	m_iSum = m_iSymbolRange;
}



void CBQBACModel1::InitFromSample( std::vector<int> &ia ) {

	unsigned int z;
	BQB_AC_CODE_VALUE m;


	m_iaCount.resize( m_iSymbolRange );
	m_iaCumulativeCount.resize( m_iSymbolRange+1 );

	for (z=0;z<m_iSymbolRange;z++)
		m_iaCount[z] = 0;

	for (z=0;z<ia.size();z++)
		m_iaCount[ia[z]]++;

	m = 0;
	for (z=0;z<m_iSymbolRange;z++)
		m += m_iaCount[z];

	if (m > BQB_AC_MAX_FREQ) {
		for (z=0;z<m_iSymbolRange;z++) {
			m_iaCount[z] /= (m / BQB_AC_MAX_FREQ) + 1;
			if (m_iaCount[z] == 0)
				m_iaCount[z] = 1;
		}
	}

	m_iaCumulativeCount[0] = 0;
	m_iSum = 0;

	for (z=1;z<=m_iSymbolRange;z++) {
		m_iaCumulativeCount[z] = m_iaCumulativeCount[z-1] + m_iaCount[z-1];
		m_iSum += m_iaCount[z-1];
	}
}








CBQBACProbability CBQBACModel2::GetSymbol( BQB_AC_CODE_VALUE scaled_value, unsigned int &sym ) {

	unsigned int z;


	for ( z=0; z<m_iSymbolRange; z++ ) {

		if (scaled_value < m_iaCumulativeCount[z+1]) {

			sym = z;

			CBQBACProbability prob = CBQBACProbability(
				m_iaCumulativeCount[z],
				m_iaCumulativeCount[z+1],
				m_iSum
			);

			Update( sym );

			return prob;
		}
	}

	printf( "CBQBACModel2::GetSymbol(): Internal error: %lu >= %lu.\n", scaled_value, m_iSum );
	abort();
}



void CBQBACModel2::Update( unsigned int sym ) {

	unsigned int z;
	BQB_AC_CODE_VALUE m;


	m_pRingBuffer->Push( sym );

	for (z=0;z<m_iSymbolRange;z++)
		m_iaCount2[z] = m_pRingBuffer->m_iaaCount[0][z] * BQB_AC_MAX_FREQ/2 / 50;

	m = 0;
	for (z=0;z<m_iSymbolRange;z++)
		m += m_iaCount[z];

	if (m > BQB_AC_MAX_FREQ/2) {
		for (z=0;z<m_iSymbolRange;z++) {
			m_iaCount[z] /= (m / BQB_AC_MAX_FREQ) + 1;
			if (m_iaCount[z] == 0)
				m_iaCount[z] = 1;
		}
	}

	m_iaCumulativeCount[0] = 0;
	m_iSum = 0;

	for (z=1;z<=m_iSymbolRange;z++) {
		m_iaCumulativeCount[z] = m_iaCumulativeCount[z-1] + m_iaCount[z-1] + m_iaCount2[z-1];
		m_iSum += m_iaCount[z-1] + m_iaCount2[z-1];
	}
}



void CBQBACModel2::InitUniform() {

	unsigned int z;


	m_iaCount.resize( m_iSymbolRange );
	m_iaCumulativeCount.resize( m_iSymbolRange+1 );

	for (z=0;z<m_iSymbolRange;z++)
		m_iaCount[z] = 1;

	for (z=0;z<=m_iSymbolRange;z++)
		m_iaCumulativeCount[z] = z;

	m_iSum = m_iSymbolRange;
}



void CBQBACModel2::InitFromSample( std::vector<int> &ia ) {

	unsigned int z;
	BQB_AC_CODE_VALUE m;


	m_iaCount.resize( m_iSymbolRange );
	m_iaCount2.resize( m_iSymbolRange );
	m_iaCumulativeCount.resize( m_iSymbolRange+1 );

	for (z=0;z<m_iSymbolRange;z++)
		m_iaCount[z] = 0;

	for (z=0;z<ia.size();z++)
		m_iaCount[ia[z]]++;

	m = 0;
	for (z=0;z<m_iSymbolRange;z++)
		m += m_iaCount[z];

	if (m > BQB_AC_MAX_FREQ/2) {
		for (z=0;z<m_iSymbolRange;z++) {
			m_iaCount[z] /= (m / BQB_AC_MAX_FREQ) + 1;
			if (m_iaCount[z] == 0)
				m_iaCount[z] = 1;
		}
	}

	m_iaCumulativeCount[0] = 0;
	m_iSum = 0;

	for (z=1;z<=m_iSymbolRange;z++) {
		m_iaCumulativeCount[z] = m_iaCumulativeCount[z-1] + m_iaCount[z-1];
		m_iSum += m_iaCount[z-1];
	}

	m_pRingBuffer = new CBQBIntRingBuffer();
	m_pRingBuffer->AddInterval( 2 );
	m_pRingBuffer->Initialize( 55, m_iSymbolRange );
}








CBQBACProbability CBQBACModel3::GetSymbol( BQB_AC_CODE_VALUE scaled_value, unsigned int &sym ) {

	unsigned int z;


	for ( z=0; z<m_iSymbolRange; z++ ) {

		if (scaled_value < m_iaCumulativeCount[z+1]) {

			sym = z;

			CBQBACProbability prob = CBQBACProbability(
				m_iaCumulativeCount[z],
				m_iaCumulativeCount[z+1],
				m_iSum
			);

			Update( sym );

			return prob;
		}
	}

	printf( "CBQBACModel3::GetSymbol(): Internal error: %lu >= %lu.\n", scaled_value, m_iSum );
	abort();
}



void CBQBACModel3::Update( unsigned int sym ) {

	unsigned int z;

	m_iLastSym = sym;

	m_iaCumulativeCount[0] = 0;
	m_iSum = 0;

	for (z=0;z<m_iSymbolRange;z++)
		m_iaCumulativeCount[z+1] = m_iaCumulativeCount[z] + m_iaaCount[m_iLastSym][z];

	m_iSum = m_iaCumulativeCount[m_iSymbolRange];
}



void CBQBACModel3::InitUniform() {

}



void CBQBACModel3::InitFromSample( std::vector<int> &ia ) {

	unsigned int z, z2;
	BQB_AC_CODE_VALUE m;


	m_iaaCount.resize( m_iSymbolRange );
	m_iaSum.resize( m_iSymbolRange );
	m_iaCumulativeCount.resize( m_iSymbolRange+1 );

	for (z=0;z<m_iSymbolRange;z++) {
		m_iaaCount[z].resize( m_iSymbolRange );
		for (z2=0;z2<m_iSymbolRange;z2++)
			m_iaaCount[z][z2] = 0;
	}

	for (z=1;z<(int)ia.size();z++)
		m_iaaCount[ia[z-1]][ia[z]]++;

	for (z=0;z<m_iSymbolRange;z++) {
		m_iaSum[z] = 0;
		for (z2=0;z2<m_iSymbolRange;z2++)
			m_iaSum[z] += m_iaaCount[z][z2];
		if (m_iaSum[z] > BQB_AC_MAX_FREQ) {
			m = (m_iaSum[z] / BQB_AC_MAX_FREQ) + 1;
			m_iaSum[z] = 0;
			for (z2=0;z2<m_iSymbolRange;z2++) {
				m_iaaCount[z][z2] /= m;
				m_iaSum[z] += m_iaaCount[z][z2];
			}
		}
	}

	for (z=0;z<=m_iSymbolRange;z++)
		m_iaCumulativeCount[z] = z;

	m_iSum = m_iSymbolRange;
}








CBQBACProbability CBQBACModelB::GetProbability( unsigned int sym ) const {

	switch( m_iState ) {

		case 0:
			if (sym != 0)
				return CBQBACProbability(
					m_iZeroCounter,
					m_iZeroSum,
					m_iZeroSum
				);
			else
				return CBQBACProbability(
					0,
					m_iZeroCounter,
					m_iZeroSum
				);

		case 1:
			if (sym != 0)
				return CBQBACProbability(
					m_iSignCounter,
					m_iSignSum,
					m_iSignSum
				);
			else
				return CBQBACProbability(
					0,
					m_iSignCounter,
					m_iSignSum
				);

		case 2:
			if (sym != 0)
				return CBQBACProbability(
					m_iaExpoCounter[m_iSubState],
					m_iaExpoSum[m_iSubState],
					m_iaExpoSum[m_iSubState]
				);
			else
				return CBQBACProbability(
					0,
					m_iaExpoCounter[m_iSubState],
					m_iaExpoSum[m_iSubState]
				);

		case 3:
			if (sym != 0)
				return CBQBACProbability(
					m_iaMantisCounter[m_iSubState],
					m_iaMantisSum[m_iSubState],
					m_iaMantisSum[m_iSubState]
				);
			else
				return CBQBACProbability(
					0,
					m_iaMantisCounter[m_iSubState],
					m_iaMantisSum[m_iSubState]
				);
	}
	printf("Internal error in CBQBACModelB::GetProbability().\n");
	abort();
}



CBQBACProbability CBQBACModelB::GetSymbol( BQB_AC_CODE_VALUE scaled_value, unsigned int &sym ) {

	unsigned int z;


	for ( z=0; z<m_iSymbolRange; z++ ) {

		if (scaled_value < m_iaCumulativeCount[z+1]) {

			sym = z;

			CBQBACProbability prob = CBQBACProbability(
				m_iaCumulativeCount[z],
				m_iaCumulativeCount[z+1],
				m_iSum
			);

			Update( sym );

			return prob;
		}
	}

	printf( "CBQBACModelB::GetSymbol(): Internal error: %lu >= %lu.\n", scaled_value, m_iSum );
	abort();
}



void CBQBACModelB::Update( unsigned int sym ) {

	ChangeState( sym );
}



void CBQBACModelB::InitUniform() {

}



void CBQBACModelB::ChangeState( int sym ) {

	switch( m_iState ) {

		case 0:
			if (sym != 0)
				m_iState = 0;
			else
				m_iState = 1;
			break;

		case 1:
			m_iState = 2;
			break;

		case 2:
			if (sym == 0) {
				if (m_iSubState != 0) {
					m_iState = 3;
					m_iSubState--;
				} else
					m_iState = 0; // +/- 1 --> no mantissa
			} else 
				m_iSubState++;
			break;

		case 3:
			if (m_iSubState == 0)
				m_iState = 0;
			else
				m_iSubState--;
			break;
	}

//	printf(" %d  --> %d (%d)\n",sym,m_iState,m_iSubState);
//	if (m_iState == 0)
//		printf("\n");
}



void CBQBACModelB::InitFromSample( std::vector<int> &ia ) {

	unsigned int z;


	m_iState = 0;
	m_iSubState = 0;

	m_iZeroCounter = 0;
	m_iZeroSum = 0;
	m_iSignCounter = 0;
	m_iSignSum = 0;

	m_iaExpoCounter.resize(12);
	m_iaExpoSum.resize(12);
	m_iaMantisCounter.resize(12);
	m_iaMantisSum.resize(12);
	for (z=0;z<12;z++) {
		m_iaExpoCounter[z] = 0;
		m_iaExpoSum[z] = 0;
		m_iaMantisCounter[z] = 0;
		m_iaMantisSum[z] = 0;
	}

	for (z=0;z<(int)ia.size();z++) {

		switch( m_iState ) {

			case 0:
				if (ia[z] == 0)
					m_iZeroCounter++;
				m_iZeroSum++;
				break;

			case 1:
				if (ia[z] == 0)
					m_iSignCounter++;
				m_iSignSum++;
				break;

			case 2:
				if (ia[z] == 0)
					m_iaExpoCounter[m_iSubState]++;
				m_iaExpoSum[m_iSubState]++;
				break;

			case 3:
				if (ia[z] == 0)
					m_iaMantisCounter[m_iSubState]++;
				m_iaMantisSum[m_iSubState]++;
				break;
		}

		ChangeState( ia[z] );
	}

	while (m_iZeroSum > BQB_AC_MAX_FREQ) {
		m_iZeroCounter >>= 1;
		m_iZeroSum >>= 1;
	}

	while (m_iSignSum > BQB_AC_MAX_FREQ) {
		m_iSignCounter >>= 1;
		m_iSignSum >>= 1;
	}

	for (z=0;z<12;z++) {

		while (m_iaExpoSum[z] > BQB_AC_MAX_FREQ) {
			m_iaExpoCounter[z] >>= 1;
			m_iaExpoSum[z] >>= 1;
		}

		while (m_iaMantisSum[z] > BQB_AC_MAX_FREQ) {
			m_iaMantisCounter[z] >>= 1;
			m_iaMantisSum[z] >>= 1;
		}
	}

	m_iState = 0;
	m_iSubState = 0;

/*	printf("\n");
	printf("*** B-Model Statistics:\n\n");
	printf("  Zero Bit:       %6d / %6d (%10.6f%%)\n",m_iZeroSum-m_iZeroCounter,m_iZeroSum,100.0-(double)m_iZeroCounter/m_iZeroSum*100.0);
	printf("  Sign Bit:       %6d / %6d (%10.6f%%)\n",m_iSignSum-m_iSignCounter,m_iSignSum,100.0-(double)m_iSignCounter/m_iSignSum*100.0);
	for (z=0;z<12;z++)
		printf("  Expo Bit %2d:    %6d / %6d (%10.6f%%)\n",z+1,m_iaExpoSum[z]-m_iaExpoCounter[z],m_iaExpoSum[z],100.0-(double)m_iaExpoCounter[z]/m_iaExpoSum[z]*100.0);
	for (z=0;z<12;z++)
		printf("  Mantis Bit %2d:  %6d / %6d (%10.6f%%)\n",z+1,m_iaMantisSum[z]-m_iaMantisCounter[z],m_iaMantisSum[z],100.0-(double)m_iaMantisCounter[z]/m_iaMantisSum[z]*100.0);
	printf("\n");*/
}



#endif



//<<<BETA



