//>>>BETA
/***********************************************************************************

    LibBQB - File Format and Compression Algorithms for Trajectories of
             Volumetric Data and Atom Positions

    https://brehm-research.de/bqb

    Free software, licensed under GNU LGPL v3

    Copyright (c) Martin Brehm and Martin Thomas,
                  Martin Luther University Halle-Wittenberg, Germany,
                  2016 - 2021.

    Please cite:  M. Brehm, M. Thomas: "An Efficient Lossless Compression Algorithm
                  for Trajectories of Atom Positions and Volumetric Data",
                  J. Chem. Inf. Model. 2018, 58 (10), pp 2092-2107.

    --------------------------------------------------------------------------------

    LibBQB is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/



// This must always be the first include directive
#include "bqb_config.h"

#include "bqb_arithmeticcoding.h"
#include "bqb_interface.h"
#include <typeinfo>
#include <limits>



const char *GetRevisionInfo_bqb_arithmeticcoding(unsigned int len) {
	static char buf[256];
	GET_REVISION_INFO( buf, len );
	return buf;
}


const char *GetSourceVersion_bqb_arithmeticcoding() {
	static char buf[256];
	GET_SOURCE_VERSION( buf );
	return buf;
}



#ifdef BQI_DEVELOPMENT



//#define LOG


bool CBQBACEngine::Compress(
		std::vector<int> *iain,
		CBQBBitSet *bsout,
		unsigned int symrange
	) {

	unsigned int pending_bits;
	unsigned int z;
	BQB_AC_CODE_VALUE low;
	BQB_AC_CODE_VALUE high;
	BQB_AC_CODE_VALUE range;
	CBQBACProbability prob;


//	m_IF.printf("*** CBQBACEngine::Compress() ***\n");
//	m_IF.printf("    Will compress %lu symbols with a range of 0 .. %u.\n",(unsigned long)iain->size(),symrange);

	m_iSymbolRange = symrange;

	bsout->WriteProgressiveUnsignedInt( iain->size() );
	bsout->WriteProgressiveUnsignedInt( m_iSymbolRange );

	pending_bits = 0;
	low = 0;
	high = BQB_AC_MAX_CODE;

	#ifdef LOG
		FILE *a;
		a = fopen("compress.log","wt");
	#endif

	for ( z=0; z<iain->size(); z++ ) {

		#ifdef LOG
			fprintf(a,"%7u: sym=%5d, l=%08X, h=%08X --> ",z+1,(*iain)[z],low,high);
		#endif

		prob = m_pModel->GetProbability( (*iain)[z] );

/*		if (prob.m_iCount == 0) {
			printf("CBQBACEngine::Compress(): Error: Model returned zero count for symbol %d.\n",(*iain)[z]);
			abort();
		}

		if (prob.m_iLow == prob.m_iHigh) {
			printf("CBQBACEngine::Compress(): Error: Model returned low == high for symbol %d.\n",(*iain)[z]);
			abort();
		}*/

		range = high - low + 1;
		high = low + (range * prob.m_iHigh / prob.m_iCount) - 1;
		low = low + (range * prob.m_iLow / prob.m_iCount);

		#ifdef LOG
			fprintf(a,"pl=%04X, ph=%04X, pc=%04X;  l=%08X, h=%08X\n",prob.m_iLow,prob.m_iHigh,prob.m_iCount,low,high);
			fflush(a);
		#endif

		m_pModel->Update( (*iain)[z] );

		// On each pass there are six possible configurations of high/low,
		// each of which has its own set of actions. When high or low
		// is converging, we output their MSB and upshift high and low.
		// When they are in a near-convergent state, we upshift over the
		// next-to-MSB, increment the pending count, leave the MSB intact,
		// and don't output anything. If we are not converging, we do
		// no shifting and no output.
		// high: 0xxx, low anything : converging (output 0)
		// low: 1xxx, high anything : converging (output 1)
		// high: 10xxx, low: 01xxx : near converging
		// high: 11xxx, low: 01xxx : not converging
		// high: 11xxx, low: 00xxx : not converging
		// high: 10xxx, low: 00xxx : not converging

		while (true) {

			if (high < BQB_AC_ONE_HALF)
				put_bit_plus_pending( 0, pending_bits, bsout );
			else if (low >= BQB_AC_ONE_HALF)
				put_bit_plus_pending( 1, pending_bits, bsout );
			else if ((low >= BQB_AC_ONE_FOURTH) && (high < BQB_AC_THREE_FOURTHS)) {
				pending_bits++;
				low -= BQB_AC_ONE_FOURTH;  
				high -= BQB_AC_ONE_FOURTH;  
			} else
				break;

			high <<= 1;
			high++;
			low <<= 1;
			high &= BQB_AC_MAX_CODE;
			low &= BQB_AC_MAX_CODE;
		}
	}

	pending_bits++;

	if (low < BQB_AC_ONE_FOURTH)
		put_bit_plus_pending( 0, pending_bits, bsout );
	else
		put_bit_plus_pending( 1, pending_bits, bsout );

	#ifdef LOG
		fclose(a);
	#endif 

//	m_IF.printf("*** CBQBACEngine::Compress() leaving ***\n\n");

	return true;
}



bool CBQBACEngine::Decompress(
		CBQBBitSet *bsin,
		std::vector<int> *iaout
	) {

	BQB_AC_CODE_VALUE high;
	BQB_AC_CODE_VALUE low;
	BQB_AC_CODE_VALUE value;
	BQB_AC_CODE_VALUE range;
	BQB_AC_CODE_VALUE scaled_value;
	unsigned int sym, i, symcount, co;
	CBQBACProbability prob;


	m_IF.printf("*** CBQBACEngine::Decompress() ***\n");

	#ifdef LOG
		FILE *a;
		a = fopen("decompress.log","wt");
	#endif

	symcount = bsin->ReadProgressiveUnsignedInt();
	m_iSymbolRange = bsin->ReadProgressiveUnsignedInt();

	m_IF.printf("    Will decompress %u symbols with a range of 0 .. %u.\n",symcount,m_iSymbolRange);

	value = 0;

	for ( i=0; i<BQB_AC_CODE_VALUE_BITS; i++ ) {
		value <<= 1;
		value += bsin->ReadBitOrZero() ? 1 : 0;
	}

	co = 0;
	low = 0;
	high = BQB_AC_MAX_CODE;

	while (true) {

		range = high - low + 1;
		scaled_value = ((value - low + 1) * m_pModel->GetSum() - 1 ) / range;

		prob = m_pModel->GetSymbol( scaled_value, sym );

		iaout->push_back( sym );
		co++;

		if (co == symcount)
			break;

		#ifdef LOG
			fprintf(a,"%7u: sym=%5u, l=%08X, h=%08X --> ",co,sym,low,high);
		#endif 

		high = low + (range*prob.m_iHigh) / prob.m_iCount - 1;
		low = low + (range*prob.m_iLow) / prob.m_iCount;

		#ifdef LOG
			fprintf(a,"pl=%04X, ph=%04X, pc=%04X;  l=%08X, h=%08X\n",prob.m_iLow,prob.m_iHigh,prob.m_iCount,low,high);
			fflush(a);
		#endif

		while (true) {

			if (high < BQB_AC_ONE_HALF) {
				// do nothing, bit is a zero
			} else if (low >= BQB_AC_ONE_HALF) {
				value -= BQB_AC_ONE_HALF;  // subtract one half from all three code values
				low -= BQB_AC_ONE_HALF;
				high -= BQB_AC_ONE_HALF;
			} else if ((low >= BQB_AC_ONE_FOURTH) && (high < BQB_AC_THREE_FOURTHS)) {
				value -= BQB_AC_ONE_FOURTH;
				low -= BQB_AC_ONE_FOURTH;
				high -= BQB_AC_ONE_FOURTH;
			} else
				break;

			low <<= 1;
			high <<= 1;
			high++;
			value <<= 1;
			value += bsin->ReadBitOrZero() ? 1 : 0;
		}
	}

	#ifdef LOG
		fclose(a);
	#endif 

	m_IF.printf("*** CBQBACEngine::Decompress() leaving ***\n");
	m_IF.printf("\n");

	return true;
}



#endif


//<<<BETA

