/*****************************************************************************

    TRAVIS - Trajectory Analyzer and Visualizer

    http://www.travis-analyzer.de/

    Copyright (c) 2009-2020 Martin Brehm
                  2012-2020 Martin Thomas
                  2016-2020 Sascha Gehrke

    Please cite:  J. Chem. Phys. 2020, 152 (16), 164105.         (DOI 10.1063/5.0005078 )
                  J. Chem. Inf. Model. 2011, 51 (8), 2007-2023.  (DOI 10.1021/ci200217w )

    This file was written by Martin Brehm and Martin Thomas.

    ---------------------------------------------------------------------------

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

*****************************************************************************/


// This must always be the first include directive
#include "config.h"

#include "tetrapak.h"

#include "tools.h"
#include "voro++.h"
#include "bqb.h"



bool CTetraPak::Init( CSnapshot *snap, int ifac ) {

	libvori_printf( LV_PL_INFO, "  Grid resolution: ( %d | %d | %d )\n",snap->m_iRes[0],snap->m_iRes[1],snap->m_iRes[2]);
	libvori_printf( LV_PL_INFO, "  Volumetric cell matrix:\n");
	libvori_printf( LV_PL_INFO, "    A = ( %12.6f | %12.6f | %12.6f ) pm\n",snap->m_fCellVecA[0],snap->m_fCellVecA[1],snap->m_fCellVecA[2]);
	libvori_printf( LV_PL_INFO, "    B = ( %12.6f | %12.6f | %12.6f ) pm\n",snap->m_fCellVecB[0],snap->m_fCellVecB[1],snap->m_fCellVecB[2]);
	libvori_printf( LV_PL_INFO, "    C = ( %12.6f | %12.6f | %12.6f ) pm\n",snap->m_fCellVecC[0],snap->m_fCellVecC[1],snap->m_fCellVecC[2]);
	libvori_printf( LV_PL_INFO, "  Simulation cell matrix (should match):\n");
	libvori_printf( LV_PL_INFO, "    A = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(0,0),glv_pSnapshot->m_mBoxFromOrtho(0,1),glv_pSnapshot->m_mBoxFromOrtho(0,2));
	libvori_printf( LV_PL_INFO, "    B = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(1,0),glv_pSnapshot->m_mBoxFromOrtho(1,1),glv_pSnapshot->m_mBoxFromOrtho(1,2));
	libvori_printf( LV_PL_INFO, "    C = ( %12.6f | %12.6f | %12.6f ) pm\n",glv_pSnapshot->m_mBoxFromOrtho(2,0),glv_pSnapshot->m_mBoxFromOrtho(2,1),glv_pSnapshot->m_mBoxFromOrtho(2,2));

	if (snap->m_bBoxNonOrtho)
		libvori_printf( LV_PL_INFO, "  Cell is non-orthorhombic.\n" );
	else
		libvori_printf( LV_PL_INFO, "  Cell is orthorhombic.\n" );

	libvori_printf( LV_PL_INFO, "  Refinement factor: %d\n",ifac);

	m_iInterpolationFactor = ifac;
	m_iInterpolationFactorCube = ifac * ifac * ifac;

	libvori_printf( LV_PL_INFO, "  This will require %s of buffer memory on MPI rank 0.\n", 
			FormatBytes( ((double)snap->m_iRes[0]*snap->m_iRes[1]*snap->m_iRes[2])*(m_iInterpolationFactorCube*sizeof(int) + sizeof(double)) ) );

	if (m_hitCount != NULL)
		delete[] m_hitCount;

	try { m_hitCount = new int[snap->m_iRes[0] * snap->m_iRes[1] * snap->m_iRes[2] * m_iInterpolationFactorCube]; } catch (...) { m_hitCount = NULL; }
	if (m_hitCount == NULL) {
		libvori_printf( LV_PL_ERROR, "Error: Could not allocate hit count array (%d x %d x %d x %d).\n",snap->m_iRes[0], snap->m_iRes[1], snap->m_iRes[2], m_iInterpolationFactorCube );
		return false;
	}
	memset(m_hitCount, 0, snap->m_iRes[0] * snap->m_iRes[1] * snap->m_iRes[2] * m_iInterpolationFactorCube * sizeof(int));

	return true;
}



bool CTetraPak::ProcessStep( CSnapshot *snap, bool sanity ) {

	voronoicell_neighbor c;
	container_periodic_poly *con;
	int ijk, q, z, z2, fc, id, ti, cc;
	double *pp, tx, ty, tz;
	CxDVector3 vec1;
	vector<int> nb;
	vector<int> fv;
	vector<int> fo;
	vector<double> fare;
	CTetraFace *fa;
	double tfx, tfy, tfz, lastvert[2];
	double mi[2], ma[2];
	bool b;
	int blocksx, blocksy, blocksz;
	double tf, integralFactor, totcharge, gridsum, cv;


	if (sanity) {
		libvori_printf( LV_PL_INFO, "Starting sanity check.\n");
		libvori_printf( LV_PL_INFO, "    Using the following cell vectors for Voronoi tessellation:\n");
		libvori_printf( LV_PL_INFO, "      A = ( %12.6f | %12.6f | %12.6f ) pm\n",snap->m_fCellVecA[0],snap->m_fCellVecA[1],snap->m_fCellVecA[2]);
		libvori_printf( LV_PL_INFO, "      B = ( %12.6f | %12.6f | %12.6f ) pm\n",snap->m_fCellVecB[0],snap->m_fCellVecB[1],snap->m_fCellVecB[2]);
		libvori_printf( LV_PL_INFO, "      C = ( %12.6f | %12.6f | %12.6f ) pm\n",snap->m_fCellVecC[0],snap->m_fCellVecC[1],snap->m_fCellVecC[2]);
		memset(m_hitCount, 0, snap->m_iRes[0] * snap->m_iRes[1] * snap->m_iRes[2] * m_iInterpolationFactorCube * sizeof(int));
	}

	if (snap->m_bBoxNonOrtho) {
		integralFactor = 
			  snap->m_fCellVecA[0] * snap->m_fCellVecB[1] * snap->m_fCellVecC[2]
			+ snap->m_fCellVecA[1] * snap->m_fCellVecB[2] * snap->m_fCellVecC[0]
			+ snap->m_fCellVecA[2] * snap->m_fCellVecB[0] * snap->m_fCellVecC[1]
			- snap->m_fCellVecA[0] * snap->m_fCellVecB[2] * snap->m_fCellVecC[1]
			- snap->m_fCellVecA[1] * snap->m_fCellVecB[0] * snap->m_fCellVecC[2]
			- snap->m_fCellVecA[2] * snap->m_fCellVecB[1] * snap->m_fCellVecC[0];
		integralFactor /= snap->m_iRes[0] * snap->m_iRes[1] * snap->m_iRes[2];
	} else
		integralFactor = snap->m_fStrideA[0] * snap->m_fStrideB[1] * snap->m_fStrideC[2];

	integralFactor /= LEN_AU2PM * LEN_AU2PM * LEN_AU2PM;

	blocksx = (int)(pow(snap->m_oaAtoms.size()/optimal_particles/snap->m_fCellVecA[0]/snap->m_fCellVecB[1]/snap->m_fCellVecC[2],1/3.0)*snap->m_fCellVecA[0])+1;
	blocksy = (int)(pow(snap->m_oaAtoms.size()/optimal_particles/snap->m_fCellVecA[0]/snap->m_fCellVecB[1]/snap->m_fCellVecC[2],1/3.0)*snap->m_fCellVecB[1])+1;
	blocksz = (int)(pow(snap->m_oaAtoms.size()/optimal_particles/snap->m_fCellVecA[0]/snap->m_fCellVecB[1]/snap->m_fCellVecC[2],1/3.0)*snap->m_fCellVecC[2])+1;

	b = true;

	if (snap->m_bBoxNonOrtho) {

		if ((snap->m_fCellVecA[1] != 0) || (snap->m_fCellVecA[2] != 0) || (snap->m_fCellVecB[2] != 0)) {
			libvori_printf( LV_PL_ERROR, "This cell matrix cannot be handled by the Voro++ library.\n");
			libvori_printf( LV_PL_ERROR, "The first cell vector has to be parallel to the x axis and the second cell vector has to be in the xy plane.\n");
			abort();
		}

		try { con = new container_periodic_poly(
							snap->m_fCellVecA[0] / 1000.0,
							snap->m_fCellVecB[0] / 1000.0,
							snap->m_fCellVecB[1] / 1000.0,
							snap->m_fCellVecC[0] / 1000.0,
							snap->m_fCellVecC[1] / 1000.0,
							snap->m_fCellVecC[2] / 1000.0,
							blocksx,
							blocksy,
							blocksz,
							glv_iVoroMemory
						); } catch(...) { con = NULL; }

	} else {

		try { con = new container_periodic_poly(
							snap->m_fCellVecA[0] / 1000.0,
							0,
							snap->m_fCellVecB[1] / 1000.0,
							0,
							0,
							snap->m_fCellVecC[2] / 1000.0,
							blocksx,
							blocksy,
							blocksz,
							glv_iVoroMemory
						); } catch(...) { con = NULL; }
	}

	if (con == NULL) {
		libvori_printf( LV_PL_ERROR, "Error: Could not allocate Voro++ container object.\n" );
		return false;
	}

	for (z=0;z<(int)snap->m_oaAtoms.size();z++)
		con->put(
			z,
			snap->m_oaAtoms[z]->m_vWrappedPos[0]/1000.0,
			snap->m_oaAtoms[z]->m_vWrappedPos[1]/1000.0,
			snap->m_oaAtoms[z]->m_vWrappedPos[2]/1000.0,
			snap->m_oaAtoms[z]->m_fRadius/1000.0
		);
	
	c_loop_all_periodic vl(*con);

	tfx = 0;
	tfy = 0;
	cv = 0;

	if (sanity) {
		libvori_printf( LV_PL_INFO, "    Voronoi tessellation and integration over grid:\n");
		libvori_printf( LV_PL_INFO, "      [");
	} else {
		libvori_printf( LV_PL_INFO, "Voronoi tessellation and integration over grid...\n");
		libvori_printf( LV_PL_VERBOSE, "  [");
	}
	
	m_fRayCount = 0;
	m_fRayHisto[0] = 0;
	m_fRayHisto[1] = 0;
	m_fRayHisto[2] = 0;
	m_fRayHisto[3] = 0;
	cc = 0;
	totcharge = 0;
	m_oaFaces.clear();

	if (vl.start()) {

		do {

			if (fmod(cc,snap->m_oaAtoms.size()/60.0) < 1.0) {
				if (sanity)
					libvori_printf( LV_PL_INFO, "#");
				else
					libvori_printf( LV_PL_VERBOSE, "#");
			}

			if (con->compute_cell(c,vl)) {

				ijk = vl.ijk;
				q = vl.q;
				pp = con->p[ijk]+con->ps*q;

				id = con->id[ijk][q];

				m_iFaces = c.number_of_faces();
				c.face_vertices(fv);
				c.neighbors(nb);
				c.face_areas(fare);
				c.face_orders(fo);

				if (sanity)
					cv += c.volume()*1000.0;

				for (z=(int)m_oaFaces.size();z<m_iFaces;z++) {
					fa = new CTetraFace();
					m_oaFaces.push_back(fa);
				}

				mi[0] = 1.0E20;
				mi[1] = 1.0E20;
				ma[0] = -1.0E20;
				ma[1] = -1.0E20;

				ti = 0;
				for (z=0;z<m_iFaces;z++) {

					fc = fv[ti];

					fa = m_oaFaces[z];

					fa->m_fMin[0] = 1.0E20;
					fa->m_fMin[1] = 1.0E20;
					fa->m_fMax[0] = -1.0E20;
					fa->m_fMax[1] = -1.0E20;

					tx = 0;
					ty = 0;
					tz = 0;
					for (z2=0;z2<fc;z2++) {

						tfy = pp[1]+c.pts[fv[ti+z2+1]*3+1]*0.5;
						tfz = pp[2]+c.pts[fv[ti+z2+1]*3+2]*0.5;

						tx += *pp+c.pts[fv[ti+z2+1]*3]*0.5;
						ty += tfy;
						tz += tfz;

						if (snap->m_bBoxNonOrtho) {
							CxDVector3 coord((*pp+c.pts[fv[ti+z2+1]*3]*0.5) * 1000.0, tfy * 1000.0, tfz * 1000.0);
							coord = snap->m_mBoxToOrtho * coord;
							if (fa->m_fMin[0] > coord[1])
								fa->m_fMin[0] = coord[1];
							if (fa->m_fMax[0] < coord[1])
								fa->m_fMax[0] = coord[1];
							if (fa->m_fMin[1] > coord[2])
								fa->m_fMin[1] = coord[2];
							if (fa->m_fMax[1] < coord[2])
								fa->m_fMax[1] = coord[2];
						} else {
							if (fa->m_fMin[0] > tfy)
								fa->m_fMin[0] = tfy;
							if (fa->m_fMax[0] < tfy)
								fa->m_fMax[0] = tfy;
							if (fa->m_fMin[1] > tfz)
								fa->m_fMin[1] = tfz;
							if (fa->m_fMax[1] < tfz)
								fa->m_fMax[1] = tfz;
						}
					}

					tx /= fc;
					ty /= fc;
					tz /= fc;

					if (fa->m_fMin[0] < mi[0])
						mi[0] = fa->m_fMin[0];
					if (fa->m_fMax[0] > ma[0])
						ma[0] = fa->m_fMax[0];
					if (fa->m_fMin[1] < mi[1])
						mi[1] = fa->m_fMin[1];
					if (fa->m_fMax[1] > ma[1])
						ma[1] = fa->m_fMax[1];

					fa->m_vCenter[0] = tx;
					fa->m_vCenter[1] = ty;
					fa->m_vCenter[2] = tz;

					fa->m_vSpan1[0] = *pp+c.pts[fv[ti+1]*3]*0.5;
					fa->m_vSpan1[1] = pp[1]+c.pts[fv[ti+1]*3+1]*0.5;
					fa->m_vSpan1[2] = pp[2]+c.pts[fv[ti+1]*3+2]*0.5;

					vec1[0] = *pp+c.pts[fv[ti+2]*3]*0.5;
					vec1[1] = pp[1]+c.pts[fv[ti+2]*3+1]*0.5;
					vec1[2] = pp[2]+c.pts[fv[ti+2]*3+2]*0.5;

					fa->m_vSpan1 -= fa->m_vCenter;
					vec1 -= fa->m_vCenter;

					fa->m_vSpan1.Normalize();
					fa->m_vSpan1.Chop(1.0E-14);
					
					fa->m_vNormal = CrossP(fa->m_vSpan1,vec1);

					fa->m_vNormal.Normalize();
					fa->m_vNormal.Chop(1.0E-14);
					
					fa->m_vSpan2 = CrossP(fa->m_vSpan1,fa->m_vNormal);

					fa->m_vSpan2.Normalize();
 					fa->m_vSpan2.Chop(1.0E-14);
										
					fa->m_vaEdges.RemoveAll_KeepSize();
					fa->m_faEdgesLength.clear();

					#define vec1x (fa->m_vSpan1[0])
					#define vec1y (fa->m_vSpan1[1])
					#define vec1z (fa->m_vSpan1[2])
					#define vec2x (fa->m_vSpan2[0])
					#define vec2y (fa->m_vSpan2[1])
					#define vec2z (fa->m_vSpan2[2])
					#define vecnx (fa->m_vNormal[0])
					#define vecny (fa->m_vNormal[1])
					#define vecnz (fa->m_vNormal[2])

					for (z2=0;z2<fc+1;z2++) {

						tx = (*pp+c.pts[fv[ti+(z2%fc)+1]*3]*0.5)-fa->m_vCenter[0];
						ty = (pp[1]+c.pts[fv[ti+(z2%fc)+1]*3+1]*0.5)-fa->m_vCenter[1];
						tz = (pp[2]+c.pts[fv[ti+(z2%fc)+1]*3+2]*0.5)-fa->m_vCenter[2];

						lastvert[0] = tfx;
						lastvert[1] = tfy;

						tfx = - (-tz*vec2y*vecnx + ty*vec2z*vecnx + tz*vec2x*vecny - tx*vec2z*vecny - ty*vec2x*vecnz + tx*vec2y*vecnz) /*/ tf*/;
						tfy = - ( tz*vec1y*vecnx - ty*vec1z*vecnx - tz*vec1x*vecny + tx*vec1z*vecny + ty*vec1x*vecnz - tx*vec1y*vecnz) /*/ tf*/;

						if (z2 > 0) {
							fa->m_vaEdges.Add(CxDVector3(tfy-lastvert[1],-tfx+lastvert[0],-(tfy-lastvert[1])*tfx + (tfx-lastvert[0])*tfy));
							fa->m_faEdgesLength.push_back(sqrt((tfx-lastvert[0])*(tfx-lastvert[0]) + (tfy-lastvert[1])*(tfy-lastvert[1])));
						}
					}

					ti += fv[ti]+1;
				}

				double charge;
				CxDVector3 dipoleMoment, totalCurrent, magneticMoment;
				CxDMatrix3 quadrupoleMoment;

				if (sanity) {

					integrateCell( snap, NULL, mi, ma, pp, &charge, &dipoleMoment, &quadrupoleMoment, &totalCurrent, &magneticMoment, true );

				} else {

					integrateCell( snap, NULL, mi, ma, pp, &charge, &dipoleMoment, &quadrupoleMoment, &totalCurrent, &magneticMoment);

					if (m_bIntegrateCharge) {
						tf = snap->m_oaAtoms[id]->m_fCoreCharge - charge * integralFactor;
						snap->m_oaAtoms[id]->m_fCharge = tf;
						totcharge += tf;
					}

					if (m_bIntegrateDipoleMoment) {
						snap->m_oaAtoms[id]->m_vDipole = dipoleMoment * 1000.0 * integralFactor; // Conversion to e*pm
						snap->m_oaAtoms[id]->m_vChargeCenter = dipoleMoment / (-charge) * 1000.0;
					}

					if (m_bIntegrateQuadrupoleMoment)
						snap->m_oaAtoms[id]->m_mQuadrupole = quadrupoleMoment * integralFactor; // Unit is e*nm^2

/*					if (m_bIntegrateTotalCurrent)
						m_totalCurrent[id] = totalCurrent;

					if (m_bIntegrateMagneticMoment)
						m_magneticMoments[id] = magneticMoment;*/
				}
			}

			cc++;

		} while (vl.inc());
	}

	if (sanity) {
		libvori_printf( LV_PL_INFO, "]");
		libvori_printf( LV_PL_INFO, " done.\n");
	} else {
		libvori_printf( LV_PL_VERBOSE, "]");
		libvori_printf( LV_PL_VERBOSE, " done.\n");
	}

	gridsum = 0;
	for (z=0;z<snap->m_iRes[0]*snap->m_iRes[1]*snap->m_iRes[2];z++)
		gridsum += snap->m_pBin[z];

	if (!sanity) {
		libvori_printf( LV_PL_INFO, "Sum over grid: %.8E,  electron count: %.8E\n",gridsum,gridsum*integralFactor);
		libvori_printf( LV_PL_INFO, "Total charge: %10.7f\n",totcharge);
	}

	if (sanity) {

		libvori_printf( LV_PL_INFO, "      %d Voronoi cells processed.\n",cc);
		libvori_printf( LV_PL_INFO, "    Volume: %.6f / %.6f Angstrom^3.\n",cv,snap->m_mBoxFromOrtho.Det()/1.0E6);
		libvori_printf( LV_PL_INFO, "    %10d rays cast. Intersection histogram:\n",m_fRayCount);
		libvori_printf( LV_PL_INFO, "    %10d rays hit   0 times (%10.6f%c).\n",m_fRayHisto[0],(double)m_fRayHisto[0]/m_fRayCount*100.0,'%');
		libvori_printf( LV_PL_INFO, "    %10d rays hit   1 times (%10.6f%c).\n",m_fRayHisto[1],(double)m_fRayHisto[1]/m_fRayCount*100.0,'%');
		libvori_printf( LV_PL_INFO, "    %10d rays hit   2 times (%10.6f%c).\n",m_fRayHisto[2],(double)m_fRayHisto[2]/m_fRayCount*100.0,'%');
		libvori_printf( LV_PL_INFO, "    %10d rays hit > 2 times (%10.6f%c).\n",m_fRayHisto[3],(double)m_fRayHisto[3]/m_fRayCount*100.0,'%');

		if ((m_fRayHisto[1] == 0) && (m_fRayHisto[3] == 0))
			b = true;
		else
			b = false;

		libvori_printf( LV_PL_INFO, "    Grid analysis:\n");
		m_fRayHisto[0] = 0;
		m_fRayHisto[1] = 0;
		m_fRayHisto[2] = 0;

		int z2a, z2b, z2c, z2d, z2e, z2f;

		for (z2a = 0; z2a < snap->m_iRes[0]; z2a++) {
			for (z2b = 0; z2b < snap->m_iRes[1]; z2b++) {
				for (z2c = 0; z2c < snap->m_iRes[2]; z2c++) {
					for (z2d = 0; z2d < m_iInterpolationFactor; z2d++) {
						for (z2e = 0; z2e < m_iInterpolationFactor; z2e++) {
							for (z2f = 0; z2f < m_iInterpolationFactor; z2f++) {
								if (m_hitCount[(z2d + z2e * m_iInterpolationFactor + z2f * m_iInterpolationFactor * m_iInterpolationFactor) * snap->m_iRes[0] * snap->m_iRes[1] * snap->m_iRes[2] + z2a + z2b * snap->m_iRes[0] + z2c * snap->m_iRes[0]*snap->m_iRes[1]] == 0) {
									m_fRayHisto[0]++;
									b = false;
									libvori_printf( LV_PL_VERBOSE, "  Missed: ( %d.%d | %d.%d | %d.%d ).\n", z2a, z2d, z2b, z2e, z2c, z2f);
								} else if (m_hitCount[(z2d + z2e * m_iInterpolationFactor + z2f * m_iInterpolationFactor * m_iInterpolationFactor) * snap->m_iRes[0] * snap->m_iRes[1] * snap->m_iRes[2] + z2a + z2b * snap->m_iRes[0] + z2c * snap->m_iRes[0]*snap->m_iRes[1]] == 1) {
									m_fRayHisto[1]++;
								} else {
									m_fRayHisto[2]++;
									b = false;
									libvori_printf( LV_PL_VERBOSE, "  Multiple: ( %d.%d | %d.%d | %d.%d ).\n", z2a, z2d, z2b, z2e, z2c, z2f);
								}
							}
						}
					}
				}
			}
		}
		
		libvori_printf( LV_PL_INFO, "      %10d points missed             (%10.6f%c).\n",m_fRayHisto[0],(double)m_fRayHisto[0]/(snap->m_iRes[0]*snap->m_iRes[1]*snap->m_iRes[2]*m_iInterpolationFactorCube)*100.0,'%');
		libvori_printf( LV_PL_INFO, "      %10d points hit                (%10.6f%c).\n",m_fRayHisto[1],(double)m_fRayHisto[1]/(snap->m_iRes[0]*snap->m_iRes[1]*snap->m_iRes[2]*m_iInterpolationFactorCube)*100.0,'%');
		libvori_printf( LV_PL_INFO, "      %10d points hit multiple times (%10.6f%c).\n",m_fRayHisto[2],(double)m_fRayHisto[2]/(snap->m_iRes[0]*snap->m_iRes[1]*snap->m_iRes[2]*m_iInterpolationFactorCube)*100.0,'%');

		if (!b) {
			libvori_printf( LV_PL_WARNING, "Warning: Sanity check failed!\n" );
			return false;
		} else
			libvori_printf( LV_PL_INFO, "Sanity check passed.\n");

		libvori_printf( LV_PL_INFO, "---------------------------------------------------------------\n");
	}

	delete con;

	for (z=0;z<(int)m_oaFaces.size();z++)
		delete m_oaFaces[z];
	m_oaFaces.clear();

	return true;
}



void CTetraPak::integrateCell(CSnapshot *electronDensity, CSnapshot *currentDensity, double mi[2], double ma[2], double refPoint[3], double *charge, CxDVector3 *dipoleMoment, CxDMatrix3 *quadrupoleMoment, CxDVector3 *totalCurrent, CxDVector3 *magneticMoment, bool sanity) {

	int z2, z3, z4, ti, ixa, ixb, iya, iyb, iza, izb;
	int ix1, ix2, iy1, iy2, iz1, iz2, z2i, z3i, z4a;
	double xv[4], tfy, tfz, tf;
	double cx1, cx2, cy1, cy2, cz1, cz2, c11, c12, c21, c22;

	
	*charge = 0.0;
	*dipoleMoment = CxDVector3(0.0, 0.0, 0.0);
	*totalCurrent = CxDVector3(0.0, 0.0, 0.0);
	*magneticMoment = CxDVector3(0.0, 0.0, 0.0);
	*quadrupoleMoment = CxDMatrix3(0.0);
	
	if (electronDensity->m_bBoxNonOrtho) {

		iya = (int)floor(mi[0] * electronDensity->m_iRes[1]);
		iyb = (int)ceil(ma[0] * electronDensity->m_iRes[1]);
		
		iza = (int)floor(mi[1] * electronDensity->m_iRes[2]);
		izb = (int)ceil(ma[1] * electronDensity->m_iRes[2]);

	} else {

		iya = (int)floor(mi[0] / (electronDensity->m_fCellVecB[1] / 1000.0) * electronDensity->m_iRes[1]);
		iyb = (int)ceil(ma[0] / (electronDensity->m_fCellVecB[1] / 1000.0) * electronDensity->m_iRes[1]);
		
		iza = (int)floor(mi[1] / (electronDensity->m_fCellVecC[2] / 1000.0) * electronDensity->m_iRes[2]);
		izb = (int)ceil(ma[1] / (electronDensity->m_fCellVecC[2] / 1000.0) * electronDensity->m_iRes[2]);
	}
	
	for (z2 = iya; z2 <= iyb; z2++) {
		
		iy1 = z2 % electronDensity->m_iRes[1];
		if (iy1 < 0)
			iy1 += electronDensity->m_iRes[1];
		iy1 *= electronDensity->m_iRes[0];
		
		iy2 = (z2 + 1) % electronDensity->m_iRes[1];
		if (iy2 < 0)
			iy2 += electronDensity->m_iRes[1];
		iy2 *= electronDensity->m_iRes[0];
		
		for (z3 = iza; z3 <= izb; z3++) {
			
			iz1 = z3 % electronDensity->m_iRes[2];
			if (iz1 < 0)
				iz1 += electronDensity->m_iRes[2];
			iz1 *= electronDensity->m_iRes[0]*electronDensity->m_iRes[1];
			
			iz2 = (z3 + 1) % electronDensity->m_iRes[2];
			if (iz2 < 0)
				iz2 += electronDensity->m_iRes[2];
			iz2 *= electronDensity->m_iRes[0]*electronDensity->m_iRes[1];
			
			for (z2i = 0; z2i < m_iInterpolationFactor; z2i++) {

				if (electronDensity->m_bBoxNonOrtho)
					tfy = ((double)z2 + (double)z2i / m_iInterpolationFactor) / electronDensity->m_iRes[1];
				else
					tfy = ((double)z2 + (double)z2i / m_iInterpolationFactor) * (electronDensity->m_fCellVecB[1] / 1000.0) / electronDensity->m_iRes[1];
				
				cy2 = (double)z2i / m_iInterpolationFactor;
				cy1 = 1.0 - cy2;
				
				for (z3i = 0; z3i < m_iInterpolationFactor; z3i++) {

					if (electronDensity->m_bBoxNonOrtho)
						tfz = ((double)z3 + (double)z3i / m_iInterpolationFactor) / electronDensity->m_iRes[2];
					else
						tfz = ((double)z3 + (double)z3i / m_iInterpolationFactor) * (electronDensity->m_fCellVecC[2] / 1000.0) / electronDensity->m_iRes[2];
					
					cz2 = (double)z3i / m_iInterpolationFactor;
					cz1 = 1.0 - cz2;
					
					c11 = cy1 * cz1;
					c12 = cy1 * cz2;
					c21 = cy2 * cz1;
					c22 = cy2 * cz2;
					
					ti = 0;
					for (z4 = 0; z4 < m_iFaces; z4++)
						if (((CTetraFace *)m_oaFaces[z4])->XRay_Hit( electronDensity, tfy, tfz, xv[ti] ))
							ti++;
						
					if (sanity) {
						if (ti == 0)
							m_fRayHisto[0]++;
						else if (ti == 1)
							m_fRayHisto[1]++;
						else if (ti == 2)
							m_fRayHisto[2]++;
						else
							m_fRayHisto[3]++;
						
						m_fRayCount++;
					}
					
					if (ti == 2) {

						if (xv[0] > xv[1]) {
							tf = xv[0];
							xv[0] = xv[1];
							xv[1] = tf;
						}
						
						if (electronDensity->m_bBoxNonOrtho) {

							while (fabs(xv[0] * electronDensity->m_iRes[0] * m_iInterpolationFactor - ceil(xv[0] * electronDensity->m_iRes[0] * m_iInterpolationFactor)) < VORI_EPS)
								xv[0] += VORI_EPS;
							while (fabs(xv[1] * electronDensity->m_iRes[0] * m_iInterpolationFactor - ceil(xv[1] * electronDensity->m_iRes[0] * m_iInterpolationFactor)) < VORI_EPS)
								xv[1] += VORI_EPS;
							
							ixa = (int)ceil(xv[0] * electronDensity->m_iRes[0] * m_iInterpolationFactor);
							ixb = (int)floor(xv[1] * electronDensity->m_iRes[0] * m_iInterpolationFactor);

						} else {

							while (fabs(xv[0] / (electronDensity->m_fCellVecA[0] / 1000.0) * electronDensity->m_iRes[0] * m_iInterpolationFactor - ceil(xv[0] / (electronDensity->m_fCellVecA[0] / 1000.0) * electronDensity->m_iRes[0] * m_iInterpolationFactor)) < VORI_EPS)
								xv[0] += VORI_EPS * (electronDensity->m_fCellVecA[0] / 1000.0) * electronDensity->m_iRes[0] * m_iInterpolationFactor;
							while (fabs(xv[1] / (electronDensity->m_fCellVecA[0] / 1000.0) * electronDensity->m_iRes[0] * m_iInterpolationFactor - ceil(xv[1] / (electronDensity->m_fCellVecA[0] / 1000.0) * electronDensity->m_iRes[0] * m_iInterpolationFactor)) < VORI_EPS)
								xv[1] += VORI_EPS * (electronDensity->m_fCellVecA[0] / 1000.0) * electronDensity->m_iRes[0] * m_iInterpolationFactor;
							
							ixa = (int)ceil(xv[0] / (electronDensity->m_fCellVecA[0] / 1000.0) * electronDensity->m_iRes[0] * m_iInterpolationFactor);
							ixb = (int)floor(xv[1] / (electronDensity->m_fCellVecA[0] / 1000.0) * electronDensity->m_iRes[0] * m_iInterpolationFactor);
						}
						
						for (z4 = ixa; z4 <= ixb; z4++) {
							
							ix1 = floori(z4, m_iInterpolationFactor) % electronDensity->m_iRes[0];
							if (ix1 < 0)
								ix1 += electronDensity->m_iRes[0];
							ix2 = (ix1 + 1) % electronDensity->m_iRes[0];
							
							z4a = z4 % m_iInterpolationFactor;
							if (z4a < 0)
								z4a += m_iInterpolationFactor;
							
							cx2 = (double)z4a / m_iInterpolationFactor;
							cx1 = 1.0 - cx2;
							
							double x, y, z;

							if (electronDensity->m_bBoxNonOrtho) {
								x = (((double)z4 / m_iInterpolationFactor) / electronDensity->m_iRes[0] * electronDensity->m_fCellVecA[0] + tfy * electronDensity->m_fCellVecB[0] + tfz * electronDensity->m_fCellVecC[0]) / 1000.0;
								y = (((double)z4 / m_iInterpolationFactor) / electronDensity->m_iRes[0] * electronDensity->m_fCellVecA[1] + tfy * electronDensity->m_fCellVecB[1] + tfz * electronDensity->m_fCellVecC[1]) / 1000.0;
								z = (((double)z4 / m_iInterpolationFactor) / electronDensity->m_iRes[0] * electronDensity->m_fCellVecA[2] + tfy * electronDensity->m_fCellVecB[2] + tfz * electronDensity->m_fCellVecC[2]) / 1000.0;
							} else {
								x = ((double)z4 / m_iInterpolationFactor) * (electronDensity->m_fCellVecA[0] / 1000.0) / electronDensity->m_iRes[0];
								y = tfy;
								z = tfz;
							}
							
							if (sanity) {

								m_hitCount[(z4a + z2i * m_iInterpolationFactor + z3i * m_iInterpolationFactor * m_iInterpolationFactor) * electronDensity->m_iRes[0] * electronDensity->m_iRes[1] * electronDensity->m_iRes[2] + ix1 + iy1 + iz1] += 1;

							} else {

								double ed = (cx1 * (c11 * electronDensity->m_pBin[ix1+iy1+iz1] + c12 * electronDensity->m_pBin[ix1+iy1+iz2] + c21 * electronDensity->m_pBin[ix1+iy2+iz1] + c22 * electronDensity->m_pBin[ix1+iy2+iz2])
									+ cx2 * (c11 * electronDensity->m_pBin[ix2+iy1+iz1] + c12 * electronDensity->m_pBin[ix2+iy1+iz2] + c21 * electronDensity->m_pBin[ix2+iy2+iz1] + c22 * electronDensity->m_pBin[ix2+iy2+iz2])) / m_iInterpolationFactorCube;
						
								if (m_bIntegrateCharge) {
									*charge += ed;
								}

								if (m_bIntegrateDipoleMoment) {
									(*dipoleMoment)[0] += -(x - refPoint[0]) * ed;
									(*dipoleMoment)[1] += -(y - refPoint[1]) * ed;
									(*dipoleMoment)[2] += -(z - refPoint[2]) * ed;
								}

								if (m_bIntegrateQuadrupoleMoment) {
									(*quadrupoleMoment)(0,0) -= ed * (x - refPoint[0]) * (x - refPoint[0]);
									(*quadrupoleMoment)(0,1) -= ed * (x - refPoint[0]) * (y - refPoint[1]);
									(*quadrupoleMoment)(0,2) -= ed * (x - refPoint[0]) * (z - refPoint[2]);
									(*quadrupoleMoment)(1,0) -= ed * (y - refPoint[1]) * (x - refPoint[0]);
									(*quadrupoleMoment)(1,1) -= ed * (y - refPoint[1]) * (y - refPoint[1]);
									(*quadrupoleMoment)(1,2) -= ed * (y - refPoint[1]) * (z - refPoint[2]);
									(*quadrupoleMoment)(2,0) -= ed * (z - refPoint[2]) * (x - refPoint[0]);
									(*quadrupoleMoment)(2,1) -= ed * (z - refPoint[2]) * (y - refPoint[1]);
									(*quadrupoleMoment)(2,2) -= ed * (z - refPoint[2]) * (z - refPoint[2]);
								}

								if ((currentDensity != NULL) && (m_bIntegrateTotalCurrent || m_bIntegrateMagneticMoment))  {

									double cdx = (cx1 * (c11 * currentDensity->m_pBin[(ix1+iy1+iz1)*3] + c12 * currentDensity->m_pBin[(ix1+iy1+iz2)*3] + c21 * currentDensity->m_pBin[(ix1+iy2+iz1)*3] + c22 * currentDensity->m_pBin[(ix1+iy2+iz2)*3])
									+ cx2 * (c11 * currentDensity->m_pBin[(ix2+iy1+iz1)*3] + c12 * currentDensity->m_pBin[(ix2+iy1+iz2)*3] + c21 * currentDensity->m_pBin[(ix2+iy2+iz1)*3] + c22 * currentDensity->m_pBin[(ix2+iy2+iz2)*3])) / m_iInterpolationFactorCube;

									double cdy = (cx1 * (c11 * currentDensity->m_pBin[(ix1+iy1+iz1)*3 + 1] + c12 * currentDensity->m_pBin[(ix1+iy1+iz2)*3 + 1] + c21 * currentDensity->m_pBin[(ix1+iy2+iz1)*3 + 1] + c22 * currentDensity->m_pBin[(ix1+iy2+iz2)*3 + 1])
									+ cx2 * (c11 * currentDensity->m_pBin[(ix2+iy1+iz1)*3 + 1] + c12 * currentDensity->m_pBin[(ix2+iy1+iz2)*3 + 1] + c21 * currentDensity->m_pBin[(ix2+iy2+iz1)*3 + 1] + c22 * currentDensity->m_pBin[(ix2+iy2+iz2)*3 + 1])) / m_iInterpolationFactorCube;

									double cdz = (cx1 * (c11 * currentDensity->m_pBin[(ix1+iy1+iz1)*3 + 2] + c12 * currentDensity->m_pBin[(ix1+iy1+iz2)*3 + 2] + c21 * currentDensity->m_pBin[(ix1+iy2+iz1)*3 + 2] + c22 * currentDensity->m_pBin[(ix1+iy2+iz2)*3 + 2])
									+ cx2 * (c11 * currentDensity->m_pBin[(ix2+iy1+iz1)*3 + 2] + c12 * currentDensity->m_pBin[(ix2+iy1+iz2)*3 + 2] + c21 * currentDensity->m_pBin[(ix2+iy2+iz1)*3 + 2] + c22 * currentDensity->m_pBin[(ix2+iy2+iz2)*3 + 2])) / m_iInterpolationFactorCube;

									if (m_bIntegrateTotalCurrent) {
										(*totalCurrent)[0] += cdx;
										(*totalCurrent)[1] += cdy;
										(*totalCurrent)[2] += cdz;
									}

									if (m_bIntegrateMagneticMoment) {
										(*magneticMoment)[0] += (y - refPoint[1]) * cdz;
										(*magneticMoment)[0] -= (z - refPoint[2]) * cdy;
										(*magneticMoment)[1] += (z - refPoint[2]) * cdx;
										(*magneticMoment)[1] -= (x - refPoint[0]) * cdz;
										(*magneticMoment)[2] += (x - refPoint[0]) * cdy;
										(*magneticMoment)[2] -= (y - refPoint[1]) * cdx;
									}
								}
							}
						}
					}
				}
			}
		}
	}
}



inline bool CTetraFace::XRay_Hit( const CSnapshot *snap, double py, double pz, double &px) const {

	double tf, a, b, ty, tz;
	int z;

	
	#define lvec1x (m_vSpan1[0])
	#define lvec1y (m_vSpan1[1])
	#define lvec1z (m_vSpan1[2])
	#define lvec2x (m_vSpan2[0])
	#define lvec2y (m_vSpan2[1])
	#define lvec2z (m_vSpan2[2])
	
	if (snap->m_bBoxNonOrtho) {
		
		CxDVector3 veca = CxDVector3( snap->m_fCellVecA[0] / 1000.0, snap->m_fCellVecA[1] / 1000.0, snap->m_fCellVecA[2] / 1000.0 );
		CxDVector3 vecb = CxDVector3( snap->m_fCellVecB[0] / 1000.0, snap->m_fCellVecB[1] / 1000.0, snap->m_fCellVecB[2] / 1000.0 );
		CxDVector3 vecc = CxDVector3( snap->m_fCellVecC[0] / 1000.0, snap->m_fCellVecC[1] / 1000.0, snap->m_fCellVecC[2] / 1000.0 );
		
		double coorda;
		CxDVector3 intersection;
		bool run;

		do {
			run = false;
		
			coorda = DotP(m_vNormal, m_vCenter - py * vecb - pz * vecc) / DotP(m_vNormal, veca);
			
			intersection = coorda * veca + py * vecb + pz * vecc - m_vCenter;
			
			a = DotP(intersection, m_vSpan1);
			b = DotP(intersection, m_vSpan2);
			
			if (!isfinite(a) || !isfinite(b))
				return false;
			
			for (z=0;z<m_vaEdges.GetSize();z++) {
				if (m_faEdgesLength[z] < VORI_EPS)
					return false;
				double area = m_vaEdges[z][0]*a + m_vaEdges[z][1]*b + m_vaEdges[z][2];
				double dist = area / m_faEdgesLength[z];
				if (fabs(dist) < VORI_EPS) {
					py += VORI_EPS;
					pz += VORI_EPS;
					run = true;
					break;
				}
				if (area * m_vaEdges[z][2] < 0)
					return false;
			}
		} while (run);
		
		px = coorda;
		return true;

	} else {
		
		ty = py - m_vCenter[1];
		tz = pz - m_vCenter[2];
		
		tf = -lvec1z*lvec2y + lvec1y*lvec2z;
		
		bool run;

		do {
			run = false;
			
			a = (-tz*lvec2y + ty*lvec2z)/tf;
			b = ( tz*lvec1y - ty*lvec1z)/tf;
						
			if (!isfinite(a) || !isfinite(b))
				return false;
					
			for (z=0;z<m_vaEdges.GetSize();z++) {
				double area = m_vaEdges[z][0]*a + m_vaEdges[z][1]*b + m_vaEdges[z][2];
				double dist = 2.0 * area / m_faEdgesLength[z];
				if (fabs(dist) < VORI_EPS) {
					ty += VORI_EPS;
					tz += VORI_EPS;
					run = true;
					break;
				}
				if (area * m_vaEdges[z][2] < 0)
					return false;
			}
		} while (run);
		
		px = a*lvec1x + b*lvec2x + m_vCenter[0];
		
		return true;
	}
}



bool CTetraPak::WriteEMPFrame( CSnapshot *snap, const char *s, int step ) {

	CBQBFile *bqwrite;
	CBQBTrajectoryFrame *btf;
	CBQBTrajectoryFrameColumn *bco;
	std::vector<unsigned char> ia;
	int z;


	bqwrite = new CBQBFile(*glv_pBQBInterface);
	bqwrite->OpenWriteAppend( s );

	btf = new CBQBTrajectoryFrame(*glv_pBQBInterface);
	btf->m_iAtomCount = (int)snap->m_oaAtoms.size();
	btf->AddColumn(BQB_TYPE_STRING,"Label");
	btf->AddColumn(BQB_TYPE_DOUBLE,"PosX");    // Unit: pm
	btf->AddColumn(BQB_TYPE_DOUBLE,"PosY");    // Unit: pm
	btf->AddColumn(BQB_TYPE_DOUBLE,"PosZ");    // Unit: pm

//	if (m_bMagMom) {
//		btf->AddColumn(BQB_TYPE_DOUBLE,"VelX");    // Unit: pm/ps
//		btf->AddColumn(BQB_TYPE_DOUBLE,"VelY");    // Unit: pm/ps
//		btf->AddColumn(BQB_TYPE_DOUBLE,"VelZ");    // Unit: pm/ps
//	}

	btf->AddColumn(BQB_TYPE_DOUBLE,"EChg");    // Unit: e
	btf->AddColumn(BQB_TYPE_DOUBLE,"ECCg");    // Unit: e
	btf->AddColumn(BQB_TYPE_DOUBLE,"EDipX");   // Unit: e*pm
	btf->AddColumn(BQB_TYPE_DOUBLE,"EDipY");   // Unit: e*pm
	btf->AddColumn(BQB_TYPE_DOUBLE,"EDipZ");   // Unit: e*pm
	btf->AddColumn(BQB_TYPE_DOUBLE,"EChZX");   // Unit: pm
	btf->AddColumn(BQB_TYPE_DOUBLE,"EChZY");   // Unit: pm
	btf->AddColumn(BQB_TYPE_DOUBLE,"EChZZ");   // Unit: pm
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQXX");    // Unit: e*nm^2
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQXY");    // Unit: e*nm^2
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQXZ");    // Unit: e*nm^2
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQYX");    // Unit: e*nm^2
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQYY");    // Unit: e*nm^2
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQYZ");    // Unit: e*nm^2
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQZX");    // Unit: e*nm^2
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQZY");    // Unit: e*nm^2
	btf->AddColumn(BQB_TYPE_DOUBLE,"EQZZ");    // Unit: e*nm^2

//	if (m_bMagMom) {
//		btf->AddColumn(BQB_TYPE_DOUBLE,"ECurX");   // Unit: MB/pm
//		btf->AddColumn(BQB_TYPE_DOUBLE,"ECurY");   // Unit: MB/pm
//		btf->AddColumn(BQB_TYPE_DOUBLE,"ECurZ");   // Unit: MB/pm
//		btf->AddColumn(BQB_TYPE_DOUBLE,"MDipX");   // Unit: MB
//		btf->AddColumn(BQB_TYPE_DOUBLE,"MDipY");   // Unit: MB
//		btf->AddColumn(BQB_TYPE_DOUBLE,"MDipZ");   // Unit: MB
//	}

	btf->m_oaColumns[0]->m_aString.resize( snap->m_oaAtoms.size() );
	for (z=1;z<(int)btf->m_oaColumns.size();z++)
		btf->m_oaColumns[z]->m_aReal.resize( snap->m_oaAtoms.size() );

	btf->m_pCellMatrix = new CBQBDMatrix3();

	for (z=0;z<9;z++)
		(*btf->m_pCellMatrix)[z] = snap->m_mBoxFromOrtho[z];

	for (z=0;z<(int)snap->m_oaAtoms.size();z++)
		btf->m_oaColumns[0]->m_aString[z] = snap->m_oaAtoms[z]->m_sLabel;

	bco = btf->GetColumn("PosX");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vPos[0];
	bco = btf->GetColumn("PosY");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vPos[1];
	bco = btf->GetColumn("PosZ");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vPos[2];

//	if (m_bMagMom) {
//		bco = btf->GetColumn("VelX");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_vaVelocities[z][0];
//		bco = btf->GetColumn("VelY");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_vaVelocities[z][1];
//		bco = btf->GetColumn("VelZ");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_vaVelocities[z][2];
//	}

	bco = btf->GetColumn("EChg");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_fCharge;
	bco = btf->GetColumn("ECCg");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_fCoreCharge;
	bco = btf->GetColumn("EDipX"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vDipole[0];
	bco = btf->GetColumn("EDipY"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vDipole[1];
	bco = btf->GetColumn("EDipZ"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vDipole[2];
	bco = btf->GetColumn("EChZX"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vChargeCenter[0];
	bco = btf->GetColumn("EChZY"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vChargeCenter[1];
	bco = btf->GetColumn("EChZZ"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_vChargeCenter[2];
	bco = btf->GetColumn("EQXX");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[0];
	bco = btf->GetColumn("EQXY");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[1];
	bco = btf->GetColumn("EQXZ");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[2];
	bco = btf->GetColumn("EQYX");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[3];
	bco = btf->GetColumn("EQYY");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[4];
	bco = btf->GetColumn("EQYZ");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[5];
	bco = btf->GetColumn("EQZX");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[6];
	bco = btf->GetColumn("EQZY");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[7];
	bco = btf->GetColumn("EQZZ");  for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = snap->m_oaAtoms[z]->m_mQuadrupole[8];

//	if (m_bMagMom) {
//		bco = btf->GetColumn("ECurX"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_totalCurrents[z][0];
//		bco = btf->GetColumn("ECurY"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_totalCurrents[z][1];
//		bco = btf->GetColumn("ECurZ"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_totalCurrents[z][2];
//		bco = btf->GetColumn("MDipX"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_magneticDipoleMoments[z][0];
//		bco = btf->GetColumn("MDipY"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_magneticDipoleMoments[z][1];
//		bco = btf->GetColumn("MDipZ"); for (z=0;z<(int)snap->m_oaAtoms.size();z++) bco->m_aReal[z] = ts->m_magneticDipoleMoments[z][2];
//	}

	ia.clear();
	btf->WriteFrame( &ia );

	bqwrite->CreateShortFrame( BQB_FRAMETYPE_TRAJ, 0, step );
	bqwrite->PushPayload( ia );
	bqwrite->FinalizeFrame( NULL );

	bqwrite->WriteIndexFrame( true, NULL );
	bqwrite->Close();

	delete bqwrite;
	delete btf;

	return true;
}



