/*
 *  libgrpp - a library for the evaluation of integrals over
 *            generalized relativistic pseudopotentials.
 *
 *  Copyright (C) 2021-2023 Alexander Oleynichenko
 */

/*
 * This file contains pre-tabulated values of the scaled modified spherical Bessel functions K_lambda(x).
 * These pre-tabulated points are used to calculate K_lambda(x) for any 'x' in the range x=[0,16]
 * using the Taylor series.
 * This set of points was pre-calculated using GSL - GNU Scientific Library (gsl_sf_bessel_il_scaled).
 *
 * Range: x = [0.00,16.00]
 * Step: dx = 0.01
 * Overall 1601 x-points.
 * Values of lambda: from 0 to 59
 *
 * More on the Taylor series and the approach used:
 * R. Flores-Moreno, R. J. Alvarez-Mendez, A. Vela, A. M. Koster.
 * Half-numerical evaluation of pseudopotential integrals.
 * J. Comput. Chem. 27, 1009 (2006).
 * DOI: 10.1002/jcc.20410
 *
 */

const double bessel_table_0[1601][60] = {
// x = 0.0000
{
    1.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,
    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00,    0.0000000000000000e+00
},
// x = 0.0100
{
    9.9006633466223481e-01,    3.3001991142762154e-03,    6.6003793703555928e-06,    9.4290984194145026e-09,
    1.0476765438987720e-11,    9.5243255568974565e-15,    7.3264005174095088e-18,    4.8842650962088553e-21,
    2.8730962259137219e-24,    1.5121555293893733e-27,    7.2007391253038690e-31,    3.1307555969573125e-34,
    1.2523020532567190e-37,    4.6381551604457471e-41,    1.5993636705248390e-44,    5.1592371425301078e-48,
    1.5634050593463575e-51,    4.4668712532003322e-55,    1.2072624172015911e-58,    3.0955444658987465e-62,
    7.5501080251581624e-66,    1.7558389848771481e-69,    3.9018642263528107e-73,    8.3018384189947754e-77,
    1.6942526707731389e-80,    3.3220639374366463e-84,    6.2680449499475386e-88,    1.1396445000018026e-91,
    1.9993762563404746e-95,    3.3887732216725705e-99,   5.5553657926101031e-103,   8.8180407253141914e-107,
   1.3566216188974393e-110,   2.0248083426155096e-114,   2.9345047844702280e-118,   4.1331052504962683e-122,
   5.6617879109667951e-126,   7.5490504172362421e-130,   9.8039614196564509e-134,   1.2410077552462807e-137,
   1.5321083170211665e-141,   1.8459136088006464e-145,   2.1716630398105910e-149,   2.4961643813377019e-153,
   2.8046790455247789e-157,   3.0820648487739525e-161,   3.3140481869776730e-165,   3.4884717379095233e-169,
   3.5963625789511644e-173,   3.6326894373575173e-177,   3.5967221806315691e-181,   3.4919632498784245e-185,
   3.3256792559974382e-189,   3.1081114275538817e-193,   2.8514783503347609e-197,   2.5688993942352603e-201,
   2.2733622782892846e-205,   1.9768367490376034e-209,   1.6896040468711550e-213,   1.4198353236445199e-217
},
// x = 0.0200
{
    9.8026402119191980e-01,    6.5349192120920342e-03,    2.6139378114514981e-05,    7.4683463291239302e-08,
    1.6596258120430172e-10,    3.0174930359192023e-13,    4.6422874557112374e-16,    6.1897068982828512e-19,
    7.2819990976803858e-22,    7.6652545236252930e-25,    7.3002363577124669e-28,    6.3480271993858416e-31,
    5.0784187500768791e-34,    3.7617897449927985e-37,    2.5943366008468304e-40,    1.6737648944801392e-43,
    1.0144026150434430e-46,    5.7965845812264793e-50,    3.1332880942770130e-53,    1.6068140053669136e-56,
    7.8381153209926115e-60,    3.6456342794004704e-63,    1.6202815955195835e-66,    6.8948141025474222e-70,
    2.8142093873208423e-73,    1.1036113611230518e-76,    4.1645706025816600e-80,    1.5143891168062257e-83,
    5.3136453918690309e-87,    1.8012355258644294e-90,    5.9056896340401190e-94,    1.8748219229117100e-97,
   5.7686823098838570e-101,   1.7219945703746373e-104,   4.9912882022463225e-108,   1.4059965681807926e-111,
   3.8520451108482062e-115,   1.0272119584106466e-118,   2.6680828334136360e-122,   6.7546396623605567e-126,
   1.6678121630810196e-129,   4.0188242614950715e-133,   9.4560565743869917e-137,   2.1738059967566172e-140,
   4.8849570548200639e-144,   1.0736168843812581e-147,   2.3088534102662785e-151,   4.8607438106205512e-155,
   1.0022151769387311e-158,   2.0246770441335717e-162,   4.0092613193739400e-166,   7.7849731477451354e-170,
   1.4828519753479146e-173,   2.7716858401706444e-177,   5.0856619239212842e-181,   9.1633545256540994e-185,
   1.6218326094860635e-188,   2.8205783674279130e-192,   4.8215013442637271e-196,   8.1033633786955318e-200
},
// x = 0.0300
{
    9.7059110692918804e-01,    9.7053287645393459e-03,    5.8230475253548412e-05,    2.4955561461026515e-07,
    8.3184448652587331e-10,    2.2686525032675604e-12,    5.2353277676514950e-15,    1.0470618580281558e-17,
    1.8477510715289030e-20,    2.9174951110839635e-23,    4.1678423925375806e-26,    5.4363076551875130e-29,
    6.5235604881543650e-32,    7.2483922109167779e-35,    7.4983292632505041e-38,    7.2564412901524233e-41,
    6.5967596689008325e-44,    5.6543615008167964e-47,    4.5846145736693410e-50,    3.5266246101643280e-53,
    2.5804557145223599e-56,    1.8003171030080002e-59,    1.2002108912774947e-62,    7.6609175887780395e-66,
    4.6903560182118396e-69,    2.7590320332329858e-72,    1.5617157630499631e-75,    8.5184471711495908e-79,
    4.4833920481364686e-82,    2.2796903018530006e-85,    1.1211589022793075e-88,    5.3388507422422020e-92,
    2.4640844487327797e-95,    1.1033211801617938e-98,   4.7970477281290061e-102,   2.0269212233018962e-105,
   8.3298118771622087e-109,   3.3319242316040424e-112,   1.2981521059616720e-115,   4.9296908482805842e-119,
   1.8258111808699695e-122,   6.5993166793479732e-126,   2.3291703092420782e-129,   8.0316208224597119e-133,
   2.7072876168362035e-136,   8.9251230624034214e-140,   2.8790716623299953e-143,   9.0918043614953527e-147,
   2.8118979925995251e-150,   8.5209022409195154e-154,   2.5309608426969849e-157,   7.3717299963382864e-161,
   2.1062084016607461e-164,   5.9052567452264158e-168,   1.6252999007218920e-171,   4.3927021191932182e-175,
   1.1662040216724159e-178,   3.0422711573881772e-182,   7.8006947711077430e-186,   1.9665615840833244e-189
},
// x = 0.0400
{
    9.6104567016705256e-01,    1.2812575656629447e-02,    1.0249591984426384e-04,    5.8567609646547569e-07,
    2.6029628055648290e-09,    9.4652133893464874e-12,    2.9123494545174333e-14,    7.7662164830864074e-17,
    1.8273360030609481e-19,    3.8470077377924959e-22,    7.3276095126842598e-25,    1.2743633257251100e-27,
    2.0389764880405993e-30,    3.0206997356527704e-33,    4.1664749787095626e-36,    5.3760883383843577e-39,
    6.5164616860712406e-42,    7.4473755827009384e-45,    8.0512079189838460e-48,    8.2576408848780988e-51,
    8.0562276982624313e-54,    7.4941591039932181e-57,    6.6614697196988723e-60,    5.6693319929155044e-63,
    4.6280231535474066e-66,    3.6298199325977791e-69,    2.7394852379214306e-72,    1.9923518834766716e-75,
    1.3981410074279890e-78,    9.4789178702363596e-82,    6.2156812614908877e-85,    3.9464627510471860e-88,
    2.4285915699337064e-91,    1.4499049130802351e-94,    8.4052431278003333e-98,   4.7353467792059321e-101,
   2.5947098056773050e-104,   1.3838448462917629e-107,   7.1888025054641752e-111,   3.6398990926499080e-114,
   1.7974806056268598e-117,   8.6625551709749098e-121,   4.0764956690467828e-124,   1.8742504950285866e-127,
   8.4235960663402125e-131,   3.7026788895782150e-134,   1.5925497716262627e-137,   6.7054715583663884e-141,
   2.7651424499535252e-144,   1.1172290939339894e-147,   4.4246689984245918e-151,   1.7183178034281267e-154,
   6.5459716522626272e-158,   2.4470918707466831e-161,   8.9801524665601035e-165,   3.2360905661410673e-168,
   1.1455186434335737e-171,   3.9844121990065016e-175,   1.3621920482485039e-178,   4.5787962920448848e-182
},
// x = 0.0500
{
    9.5162581964040427e-01,    1.5857787551510367e-02,    1.5856654978239885e-04,    1.1325732704825561e-06,
    6.2919148409423396e-09,    2.8599112935292309e-11,    1.0999517803189779e-13,    3.6664699888906015e-16,
    1.0783651797168716e-18,    2.8377853240003136e-21,    6.7565967519294437e-24,    1.4688189947083376e-26,
    2.9376271093510087e-29,    5.4400328333521053e-32,    9.3793408714136675e-35,    1.5127932177990197e-37,
    2.2921059748003539e-40,    3.2744307855736487e-43,    4.4248988008585650e-46,    5.6729383110909256e-49,
    6.9182076422851568e-52,    8.0444170977377912e-55,    8.9382306544356681e-58,    9.5087456932379207e-61,
    9.7027920211280866e-64,    9.5125323990672690e-67,    8.9740794724714207e-70,    8.1582475601083940e-73,
    7.1563521889538267e-76,    6.0647010321107144e-79,    4.9710631859220322e-82,    3.9452858341972352e-85,
    3.0348335149210226e-88,    2.2647999057929159e-91,    1.6411585145275663e-94,    1.1557448752970771e-97,
   7.9160571750695859e-101,   5.2773691654719352e-104,   3.4268616860863517e-107,   2.1688989539622363e-110,
   1.3388260169387809e-113,   8.0652141115336309e-117,   4.7442419911521471e-120,   2.7265749766488095e-123,
   1.5317832219156878e-126,   8.4163888429768812e-130,   4.5249389577584454e-133,   2.3815461737671800e-136,
   1.2276008009107895e-139,   6.2000024948487212e-143,   3.0693074281653478e-146,   1.4899547177976990e-149,
   7.0950208869273233e-153,   3.3154296103216652e-156,   1.5208389565409223e-159,   6.8506245649369374e-163,
   3.0312492243374928e-166,   1.3179342004860221e-169,   5.6321964266630554e-173,   2.3664686759594969e-176
},
// x = 0.0600
{
    9.4232969402368749e-01,    1.8842072248189783e-02,    2.2608161419807043e-04,    1.9377316839134449e-06,
    1.2917741501860034e-08,    7.0458634440905433e-11,    3.2518769403919543e-13,    1.3007324130803193e-15,
    4.5907691154546768e-18,    1.4497034828412264e-20,    4.1419790792744551e-23,    1.0805095166567892e-25,
    2.5932090095918306e-28,    5.7626601930985514e-31,    1.1922697483429816e-33,    2.3076107471615908e-36,
    4.1956428266382274e-39,    7.1925105653771626e-42,    1.1663501548490313e-44,    1.7943808137403869e-47,
    2.6259177800134872e-50,    3.6640645040964584e-53,    4.8854110232150469e-56,    6.2366851742153510e-59,
    7.6367463548814376e-62,    8.9843955104473075e-65,    1.0171001224327187e-67,    1.1095624957892151e-70,
    1.1679592716160126e-73,    1.1877540033909976e-76,    1.1682815318782599e-79,    1.1126480998279274e-82,
    1.0270589354532678e-85,    9.1975355432969170e-89,    7.9978511170036445e-92,    6.7587427283461592e-95,
    5.5551273569153735e-98,   4.4440991151864405e-101,   3.4629323260618724e-104,   2.6300737047039269e-107,
   1.9482017010119012e-110,   1.4083378604035254e-113,   9.9412035868489165e-117,   6.8559992860889514e-120,
   4.6220199361379159e-123,   3.0474843758372074e-126,   1.9661181510186200e-129,   1.2417583471079751e-132,
   7.6809765871819190e-136,   4.6551356495489437e-139,   2.7654261615573963e-142,   1.6109273248705589e-145,
   9.2052960496336726e-149,   5.1618466588323534e-152,   2.8413826364863994e-155,   1.5358820653913648e-158,
   8.1551237517173091e-162,   4.2548460363889828e-165,   2.1819717621720148e-168,   1.1001535546091389e-171
},
// x = 0.0700
{
    9.3315546143710126e-01,    2.1766518032880312e-02,    3.0468859937371023e-04,    3.0466490438722194e-06,
    2.3694986488141614e-08,    1.5078111115614538e-10,    8.1187789019862511e-13,    3.7886906864063642e-15,
    1.5600254404373421e-17,    5.7473915676940105e-20,    1.9157777539704677e-22,    5.8305782605604075e-25,
    1.6325500619269250e-27,    4.2325107107563382e-30,    1.0216349479942802e-32,    2.3069065748509543e-35,
    4.8934174292036016e-38,    9.7867978274060378e-41,    1.8515500584299139e-43,    3.3232847927727501e-46,
    5.6738850960178145e-49,    9.2365337434344299e-52,    1.4367908091394536e-54,    2.1398966521386406e-57,
    3.0569892232552654e-60,    4.1958599550861321e-63,    5.5416925121203238e-66,    7.0530521733508126e-69,
    8.6616303996454643e-72,    1.0276496652324742e-74,    1.1792686040244508e-77,    1.3102968810412102e-80,
    1.4110873611395034e-83,    1.4742688147077298e-86,    1.4956335334762264e-89,    1.4745668783894045e-92,
    1.4139669740843965e-95,    1.3197013893997060e-98,   1.1997275694097961e-101,   1.0630489310232185e-104,
   9.1868359180402026e-108,   7.7479285255158896e-111,   6.3806427931388605e-114,   5.1338472743663547e-117,
   4.0378549638518094e-120,   3.1060404815178273e-123,   2.3378786357096007e-126,   1.7226464997834663e-129,
   1.2431463242825606e-132,   8.7899191975390780e-136,   6.0920203367654156e-139,   4.1402061200392669e-142,
   2.7601362095622938e-145,   1.8056958270645908e-148,   1.1596207037071368e-151,   7.3129204998754387e-155,
   4.5301260351202578e-158,   2.7574670171721517e-161,   1.6497660108592490e-164,   9.7045026437760312e-168
},
// x = 0.0800
{
    9.2410131896117920e-01,    2.4632194024081684e-02,    3.9404305811605012e-04,    4.5028918285502243e-06,
    4.0023117905380621e-08,    2.9106419490827472e-10,    1.7911054928331457e-12,    9.5523228886191661e-15,
    4.4951217051165378e-17,    1.8926524652739346e-19,    7.2100090943259357e-22,    2.5078013375457411e-24,
    8.0248881929394927e-27,    2.3777252151071744e-29,    6.5591952779517643e-32,    1.6926849660366673e-34,
    4.1034559678911353e-37,    9.3792815736076609e-40,    2.0279437783626865e-42,    4.1598680237466589e-45,
    8.1167861908273921e-48,    1.5100947618194468e-50,    2.6846047862970395e-53,    4.5695273631715870e-56,
    7.4604337315235099e-59,    1.1702613438808036e-61,    1.7664283389218673e-64,    2.5693450659351149e-67,
    3.6060914755484219e-70,    4.8896068650304189e-73,    6.4125884879654224e-76,    8.1429567820692012e-79,
    1.0022085926665853e-81,    1.1966653196780248e-84,    1.3874362392713241e-87,    1.5633065082374254e-90,
    1.7132106091121096e-93,    1.8274226245260798e-96,    1.8986189110399380e-99,   1.9226501388638258e-102,
   1.8989119097174779e-105,   1.8302748791395842e-108,   1.7226101601238123e-111,   1.5840080333770041e-114,
   1.4238263205955090e-117,   1.2517145000902271e-120,   1.0767428760086139e-123,   9.0673021321295108e-127,
   7.4781823415734036e-130,   6.0429717616679310e-133,   4.7865093417766095e-136,   3.7176749586516277e-139,
   2.8325126406752987e-142,   2.1177652982078244e-145,   1.5543223324595120e-148,   1.1202317400904398e-151,
   7.9308402717173405e-155,   5.5171036517151381e-158,   3.7723768312236687e-161,   2.5360505240314423e-164
},
// x = 0.0900
{
    9.1516549215959986e-01,    2.7440150511512548e-02,    4.9380844251504510e-04,    6.3481495655907059e-06,
    6.3476302434402636e-08,    5.1932215044441673e-10,    3.5951578628012026e-12,    2.1570262020959886e-14,
    1.1419264122050842e-16,    5.4090153052089379e-19,    2.3181105419199998e-21,    9.0707395603673320e-24,
    3.2654270570129130e-26,    1.0884644257956028e-28,    3.3779626103328471e-31,    9.8069105740664591e-34,
    2.6745932179752133e-36,    6.8774824004345322e-39,    1.6728917340250284e-41,    3.8604998302985606e-44,
    8.4742289859829682e-47,    1.7736684096454793e-49,    3.5473232338455068e-52,    6.7927227270442604e-55,
    1.2476389059097480e-57,    2.2017091185331824e-60,    3.7387409444308235e-63,    6.1179239202123539e-66,
    9.6598566076911029e-69,    1.4735341322648240e-71,    2.1740621701984916e-74,    3.1057969569603438e-77,
    4.3003262498023613e-80,    5.7765475278639502e-83,    7.5346147525549139e-86,    9.5509051827258889e-89,
    1.1775071160814073e-91,    1.4130065574210014e-94,    1.6515639068901219e-97,   1.8815261198446274e-100,
   2.0905820588344685e-103,   2.2668936057073845e-106,   2.4002376593352417e-109,   2.4830018776934248e-112,
   2.5108982639897112e-115,   2.4833035985883472e-118,   2.4031948276252834e-121,   2.2767088881000834e-124,
   2.1124085267600273e-127,   1.9203698323097433e-130,   1.7112193102610097e-133,   1.4952390541413715e-136,
   1.2816325509715136e-139,   1.0780079390527377e-142,   8.9009770240103493e-146,   7.2170037372631302e-149,
   5.7480524910716577e-152,   4.4984731544684665e-155,   3.4603619518380560e-158,   2.6170789955683099e-161
},
// x = 0.1000
{
    9.0634623461009067e-01,    3.0191419289002229e-02,    6.0365594002390120e-04,    8.6222878071640092e-06,
    9.5793522420571370e-08,    8.7078931258835740e-10,    6.6980358520539093e-12,    4.4651821349417524e-14,
    2.6264964127958046e-16,    1.3823318888581865e-18,    6.5823965249550875e-21,    2.8618617618176025e-23,
    1.1447277460140548e-25,    4.2396782466537443e-28,    1.4619417543804468e-30,    4.7158950448810599e-33,
    1.4290467319059744e-35,    4.0829591341650270e-38,    1.1034948215143151e-40,    2.8294562062583599e-43,
    6.9010735544708151e-46,    1.6048925326420775e-48,    3.5664109879256484e-51,    7.5880755364237732e-54,
    1.5485806473897324e-56,    3.0364214084556662e-59,    5.7290773433672266e-62,    1.0416471034383602e-64,
    1.8274456247043880e-67,    3.0973568594633986e-70,    5.0776209832249718e-73,    8.0596961646363193e-76,
    1.2399504089159741e-78,    1.8506682489455993e-81,    2.6821224221966289e-84,    3.7776299258740381e-87,
    5.1748260631752028e-90,    6.8997561366235741e-93,    8.9607075246409704e-96,    1.1342650027076145e-98,
   1.4003250809522902e-101,   1.6871362603479345e-104,   1.9848635045865602e-107,   2.2814493576591754e-110,
   2.5634230794540037e-113,   2.8169451104221929e-116,   3.0289698086412700e-119,   3.1883858122621152e-122,
   3.2869922598672533e-125,   3.3201908813663719e-128,   3.2873145443582216e-131,   3.1915645677433256e-134,
   3.0395825971565336e-137,   2.8407289635376882e-140,   2.6061712070069488e-143,   2.3479001164151662e-146,
   2.0777861147309499e-149,   1.8067691917259889e-152,   1.5442460632830111e-155,   1.2976848662326891e-158
},
// x = 0.1100
{
    8.9764182744327958e-01,    3.2887013981451073e-02,    7.2326431279564913e-04,    1.1363399830658289e-05,
    1.3886902648520939e-07,    1.3885727775223509e-09,    1.1748732974297091e-11,    8.6153287240035403e-14,
    5.5744156499465986e-16,    3.2271954061439915e-18,    1.6903933424956252e-20,    8.0843197797340770e-23,
    3.5570369408876057e-25,    1.4491408043108729e-27,    5.4966670009076964e-30,    1.9504071570382471e-32,
    6.5012890820210961e-35,    2.0432431917739575e-37,    6.0744558505548702e-40,    1.7132950956046724e-42,
    4.5966138307036545e-45,    1.1758706037443931e-47,    2.8743339206150382e-50,    6.7271291507800482e-53,
    1.5101645380866325e-55,    3.2572030503336121e-58,    6.7602046848567323e-61,    1.3520357186066469e-63,
    2.6091823499128302e-66,    4.8645609077281015e-69,    8.7721313942439495e-72,    1.5316374637662027e-74,
    2.5919946601386040e-77,    4.2555024830194968e-80,    6.7841176372219853e-83,    1.0510580252585874e-85,
    1.5837825652336892e-88,    2.3228762287069951e-91,    3.3183880116792326e-94,    4.6205315323849111e-97,
   6.2747846148907663e-100,   8.3159653474433203e-103,   1.0761819899555815e-105,   1.3606877460116254e-108,
   1.6817463870199012e-111,   2.0328773415374504e-114,   2.4044752829602310e-117,   2.7841256192412490e-120,
   3.1572518786666365e-123,   3.5080533977987394e-126,   3.8206477715344995e-129,   4.0802988802274805e-132,
   4.2745942232079201e-135,   4.3944381002842416e-138,   4.4347496110465395e-141,   4.3947926722767026e-144,
   4.2781127771590787e-147,   4.0921041938030193e-150,   3.8472741036345632e-153,   3.5563008047900039e-156
},
// x = 0.1200
{
    8.8905057888936079e-01,    3.5527930365965910e-02,    8.5231974021294284e-04,    1.4607857093287711e-05,
    1.9474310449269637e-07,    2.1242563354963797e-09,    1.9607072194852708e-11,    1.5684772067061590e-13,
    1.1071110257138108e-15,    6.9920278260645070e-18,    3.9953253597295535e-20,    2.0844653778395613e-22,
    1.0005220372853890e-24,    4.4466828333920521e-27,    1.8399772177117447e-29,    7.1223922003611898e-32,
    2.5899285104725943e-34,    8.8796561550005211e-37,    2.8798597442365800e-39,    8.8610271068766922e-42,
    2.5934501654815138e-44,    7.2374814855691680e-47,    1.9299819225981563e-49,    4.9275826088130529e-52,
    1.2067479709945500e-54,    2.8393918640342115e-57,    6.4287800098177959e-60,    1.4026364685284562e-62,
    2.9529062371399913e-65,    6.0058869606603421e-68,    1.1814815323926558e-70,    2.2504331004965116e-73,
    4.1546319865620452e-76,    7.4411087382831165e-79,    1.2941020636863184e-81,    2.1872086787902792e-84,
    3.5954020704021505e-87,    5.7526289684597137e-90,    8.9651148320951841e-93,    1.3617865302566783e-95,
    2.0174572051215315e-98,   2.9167996442815799e-101,   4.1178267734556365e-104,   5.6797505039705980e-107,
   7.6580769511345172e-110,   1.0098545829592146e-112,   1.3030360477696974e-115,   1.6459376988141021e-118,
   2.0362085328075492e-121,   2.4681280004633943e-124,   2.9324252483764489e-127,   3.4164132162049010e-130,
   3.9044672427069019e-133,   4.3788363610938622e-136,   4.8207315407319423e-139,   5.2115956824866860e-142,
   5.5344317601065153e-145,   5.7750530472848466e-148,   5.9231252045040555e-151,   5.9728933925840185e-154
},
// x = 0.1300
{
    8.8057082383243734e-01,    3.8115146687275163e-02,    9.9051566454897767e-04,    1.8390358468319405e-05,
    2.6559317793275837e-07,    3.1384575899942802e-09,    3.1381856319291236e-11,    2.7195806515602477e-13,
    2.0795705191395656e-15,    1.4228037773998717e-17,    8.8075247445797231e-20,    4.9780198531462999e-22,
    2.5885055165263367e-24,    1.2462905719229788e-26,    5.5867147841261988e-29,    2.3427771750099620e-31,
    9.2289871674870804e-34,    3.4278647855790470e-36,    1.2043708195681532e-38,    4.0145269690073985e-41,
    1.2728865933520055e-43,    3.8482281842752295e-46,    1.1117014813120569e-48,    3.0748964266433413e-51,
    8.1578333101738249e-54,    2.0794347053152004e-56,    5.1004706502622152e-59,    1.2055592912263683e-61,
    2.7495073735503501e-64,    6.0582081380902992e-67,    1.2910878599534652e-69,    2.6641385574700543e-72,
    5.3282564381996699e-75,    1.0338370221240968e-77,    1.9478021629835054e-80,    3.5663866978515013e-83,
    6.3510799947240720e-86,    1.1008506442241727e-88,    1.8585738461889218e-91,    3.0584045809441869e-94,
    4.9085382231578945e-97,   7.6880534993745643e-100,   1.1758172598410158e-102,   1.7569644845308951e-105,
   2.5663472627260593e-108,   3.6662030541941351e-111,   5.1247901653087298e-114,   7.0128578912049138e-117,
   9.3986589839361567e-120,   1.2341652553804620e-122,   1.5885269560204531e-125,   2.0049338017890815e-128,
   2.4822952587589704e-131,   3.0158683742184912e-134,   3.5969022111254979e-137,   4.2125824993129173e-140,
   4.8463275461250818e-143,   5.4784503449174313e-146,   6.0871596612077234e-149,   6.6498304805048939e-152
},
// x = 0.1400
{
    8.7220092337240906e-01,    4.0649623967526492e-02,    1.1375526396986209e-03,    2.2743978290032444e-05,
    3.5372519699820559e-07,    4.5013401478051442e-09,    4.8471099230041238e-11,    4.5236215845941223e-13,
    3.7251093899202300e-15,    2.7446826241603207e-17,    1.8297141693670112e-19,    1.1137010980360597e-21,
    6.2365450627448067e-24,    3.2336831630474505e-26,    1.5610543900809103e-28,    7.0497879850186018e-31,
    2.9907683934941850e-33,    1.1962892516441140e-35,    4.5264383900705973e-38,    1.6248554028071781e-40,
    5.5482250600053996e-43,    1.8063805597274042e-45,    5.6197985512492377e-48,    1.6739683007536299e-50,
    4.7827290622533126e-53,    1.3128964970749669e-55,    3.4680051646658602e-58,    8.8275943203168329e-61,
    2.1681684248325920e-63,    5.1447784137059506e-66,    1.1807627941838798e-68,    2.6239047616047449e-71,
    5.6514617441267399e-74,    1.1808974473997535e-76,    2.3960142203413808e-79,    4.7245172162228431e-82,
    9.0606855126039212e-85,    1.6913222221261111e-87,    3.0751214046555768e-90,    5.4495655442445621e-93,
    9.4189747155630377e-96,    1.5887383093312102e-98,   2.6167385151736403e-101,   4.2108329287017808e-104,
   6.6237661052161606e-107,   1.0190385792129978e-109,   1.5340331676906322e-112,   2.2606756492768845e-115,
   3.2628241745026734e-118,   4.6140857477305276e-121,   6.3957503725650336e-124,   8.6932371788769869e-127,
   1.1590962684209206e-129,   1.5165720081745720e-132,   1.9478874971896501e-135,   2.4567912024639508e-138,
   3.0438075183631491e-141,   3.7054994071424943e-144,   4.4339246727134017e-147,   5.2163748673485835e-150
},
// x = 0.1500
{
    8.6393926439427382e-01,    4.3132306310567692e-02,    1.2931381829200124e-03,    2.7700213233946233e-05,
    4.6156533585401222e-07,    6.2930827055170254e-09,    7.2604116097019973e-11,    7.2597710862809604e-13,
    6.4052342103605143e-15,    5.0564787237887285e-17,    3.6116022812325743e-19,    2.3553006312423962e-21,
    1.4131332756564194e-23,    7.8505148364523920e-26,    4.0605094988831682e-28,    1.9647194492870855e-30,
    8.9303689861004411e-33,    3.8272344987275667e-35,    1.5515573611962693e-37,    5.9674443436978405e-40,
    2.1831834830561017e-42,    7.6156677829077137e-45,    2.5385289222729086e-47,    8.1016088988913834e-50,
    2.4800620274993103e-52,    7.2942393634852425e-55,    2.0643914327277692e-57,    5.6301180456184773e-60,
    1.4816000994881123e-62,    3.7667563653220900e-65,    9.2624614228615150e-68,    2.2053358406568794e-70,
    5.0892102622108008e-73,    1.1393698865852189e-75,    2.4768796820946451e-78,    5.2328216826031979e-81,
    1.0752329132935707e-83,    2.1504574482124901e-86,    4.1891873261441049e-89,    7.9541251829087655e-92,
    1.4729812153428176e-94,    2.6620057548793315e-97,   4.6976429215032683e-100,   8.0993608119535897e-103,
   1.3650570187202597e-105,   2.2500880047614847e-108,   3.6291649588951840e-111,   5.7302464700685235e-114,
   8.8611851196814570e-117,   1.3426007848649943e-119,   1.9939572490838605e-122,   2.9038151953175919e-125,
   4.1482991142231717e-128,   5.8153613711876275e-131,   8.0027759953828517e-134,   1.0814542756513487e-136,
   1.4355562874200309e-139,   1.8724615915201645e-142,   2.4005879045817349e-145,   3.0259464118665988e-148
},
// x = 0.1600
{
    8.5578425914471579e-01,    4.5564121200810014e-02,    1.4569866295280551e-03,    3.3289028058291434e-05,
    5.9165197780428124e-07,    8.6043068006268679e-09,    1.0588526118409528e-10,    1.1293294191320218e-12,
    1.0628140468173700e-14,    8.9494388566754883e-17,    6.8182587155361797e-19,    4.7429253424874847e-21,
    3.0353571042268464e-23,    1.7986713303784654e-25,    9.9234213186299535e-28,    5.1216376786099678e-30,
    2.4831632314281066e-32,    1.1351378949736300e-34,    4.9086173292055364e-37,    2.0137594849684617e-39,
    7.8584594910275325e-42,    2.9240392656540826e-44,    1.0396458218130859e-46,    3.5391804780384217e-49,
    1.1556389299731618e-51,    3.6254995613017870e-54,    1.0944808217719816e-56,    3.1839182095294302e-59,
    8.9372462402859001e-62,    2.4236427578987894e-64,    6.3570534114155118e-67,    1.6144796623774668e-69,
    3.9740804236029778e-72,    9.4902887576150155e-75,    2.2006351689464357e-77,    4.9591533512564528e-80,
    1.0869326385672216e-82,    2.3187793500686804e-85,    4.8182225281711085e-88,    9.7583863340144811e-91,
    1.9275751458912025e-93,    3.7157940269486278e-96,    6.9944116022690812e-99,   1.2863243176847121e-101,
   2.3124858459107085e-104,   4.0658968805970454e-107,   6.9950711388513204e-110,   1.1781139715529004e-112,
   1.9432755973831115e-115,   3.1406393892929858e-118,   4.9752580764436353e-121,   7.7285379412050489e-124,
   1.1776792885464562e-126,   1.7610118932042444e-129,   2.5849661168947738e-132,   3.7260696807989805e-135,
   5.2758404821425028e-138,   7.3402858353575478e-141,   1.0037979266924903e-143,   1.3496418716573460e-146
},
// x = 0.1700
{
    8.4773434481585375e-01,    4.7945979796771059e-02,    1.6288189904823233e-03,    3.9538900232144160e-05,
    7.4662798226941783e-07,    1.1536464939705564e-08,    1.5083911199928511e-10,    1.7093162308159993e-12,
    1.7091633167521999e-14,    1.5291406379950288e-16,    1.2378016952067342e-18,    9.1485092591826415e-21,
    6.2207199671005041e-23,    3.9166050543205059e-25,    2.2958670911482670e-27,    1.2589883228563159e-29,
    6.4855351456381269e-32,    3.1500467738052508e-34,    1.4472898025734781e-36,    6.3085851212105664e-39,
    2.6157119289048075e-41,    1.0341032250322110e-43,    3.9065588037232484e-46,    1.4129928999039923e-48,
    4.9021635738915710e-51,    1.6340370539673047e-53,    5.2411989658114899e-56,    1.6199920192996168e-58,
    4.8315136259151304e-61,    1.3921198661051429e-63,    3.8796491399732520e-66,    1.0468820622660841e-68,
    2.7379810705057206e-71,    6.9470727207040518e-74,    1.7115875299624389e-76,    4.0981444744518086e-79,
    9.5435737427140770e-82,    2.1631992230393411e-84,    4.7758716987258943e-87,    1.0277145848291494e-89,
    2.1569225728294352e-92,    4.4177751246184574e-95,    8.8355157197641252e-98,   1.7264736392991072e-100,
   3.2977468694157447e-103,   6.1606049822239562e-106,   1.1261284098791974e-108,   2.0151708345790683e-111,
   3.5317320711448373e-114,   6.0645728968783906e-117,   1.0207668597857008e-119,   1.6847563345207530e-122,
   2.7276937156011969e-125,   4.3337082488399997e-128,   6.7589783718973437e-131,   1.0351564646782674e-133,
   1.5573115720086456e-136,   2.3021078139334263e-139,   3.3449360344333723e-142,   4.7784704584183883e-145
},
// x = 0.1800
{
    8.3978798313602487e-01,    5.0278777219392709e-02,    1.8083628128130339e-03,    4.6476863475103173e-05,
    9.2923322568682743e-07,    1.5202190761830377e-08,    2.1045690830412941e-10,    2.5251620877033714e-12,
    2.6734328848352434e-14,    2.5325202564284576e-16,    2.1705860520406918e-18,    1.6986238098231944e-20,
    1.2229504438802384e-22,    8.1526656196781176e-25,    5.0600928520179438e-27,    2.9380253809838669e-29,
    1.6025143461939358e-31,    8.2412961649814875e-34,    4.0091891980034396e-36,    1.8503575219790886e-38,
    8.1233715412859287e-41,    3.4004241738429370e-43,    1.3601488335077500e-45,    5.2090073564429469e-48,
    1.9134880980165552e-50,    6.7534064547320778e-53,    2.2935842465945666e-55,    7.5061981408559836e-58,
    2.3703555238660293e-60,    7.2315280223718859e-63,    2.1338755244209572e-65,    6.0967389750554297e-68,
    1.6883151556576744e-70,    4.5357402717403009e-73,    1.1832287672814230e-75,    2.9997161512347600e-78,
    7.3965166019864495e-81,    1.7751540252240948e-83,    4.1496886058616706e-86,    9.4549388240227464e-89,
    2.1010873907736361e-91,    4.5565541385016841e-94,    9.6491311938328381e-97,    1.9963636175381318e-99,
   4.0375731866161477e-102,   7.9863779258251719e-105,   1.5457448977025967e-107,   2.9287695086354221e-110,
   5.4348116793247329e-113,   9.8814437616104469e-116,   1.7610438985268089e-118,   3.0775432242296609e-121,
   5.2757731699232211e-124,   8.8751077802599253e-127,   1.4656102040921882e-129,   2.3766590566589873e-132,
   3.7858191467946144e-135,   5.9256156998879744e-138,   9.1163106316024285e-141,   1.3789346397836668e-143
},
// x = 0.1900
{
    8.3194365996748454e-01,    5.2563392835228130e-02,    1.9953520428298206e-03,    5.4128550232860469e-05,
    1.1422974086448188e-06,    1.9725612842761022e-08,    2.8824406391741770e-10,    3.6505747271828438e-12,
    4.0795981930055210e-14,    4.0792291474733505e-16,    3.6904553216811801e-18,    3.0484456257329791e-20,
    2.3166947809999312e-22,    1.6301915412251684e-24,    1.0680136416569776e-26,    6.5456590833641319e-29,
    3.7685950198694501e-31,    2.0457516982080320e-33,    1.0504948623449437e-35,    5.1176799457185200e-38,
    2.3715592194902890e-40,    1.0478787105366417e-42,    4.4243012625627239e-45,    1.7885192839046549e-47,
    6.9349745945565214e-50,    2.5835834815056424e-52,    9.2617883482074097e-55,    3.1994900417090724e-57,
    1.0664852325379559e-59,    3.4344095204489730e-62,    1.0697240643453354e-64,    3.2261235000175645e-67,
    9.4301289851264143e-70,    2.6741948002984646e-72,    7.3636705510044682e-75,    1.9705460000003173e-77,
    5.1287845391853863e-80,    1.2992839613775885e-82,    3.2060063330760972e-85,    7.7106046436487123e-88,
    1.8086506367812182e-90,    4.1402634048529253e-93,    9.2546612563225324e-96,    2.0211234950546874e-98,
   4.3147387909729107e-101,   9.0087568500174749e-104,   1.8404911910367660e-106,   3.6809679618408799e-109,
   7.2101163259364102e-112,   1.3837547030803614e-114,   2.6030938736813899e-117,   4.8018076223065684e-120,
   8.6889573022790235e-123,   1.5428941845756647e-125,   2.6894405541099441e-128,   4.6035336450320958e-131,
   7.7404332985809087e-134,   1.2788507659868008e-136,   2.0767608164967368e-139,   3.3158282846134631e-142
},
// x = 0.2000
{
    8.2419988491090168e-01,    5.4800690534589518e-02,    2.1895268920590805e-03,    6.2518233112495087e-05,
    1.3887331217511350e-06,    2.5242633694050987e-08,    3.8826857833092879e-10,    5.1761025406193146e-12,
    6.0887784480158038e-14,    6.4085980588229582e-16,    6.1029213399271612e-18,    5.3065189944014456e-20,
    4.2449636549689022e-22,    3.1442569031946053e-24,    2.1683565618568403e-26,    1.3988850218595921e-28,
    8.4777974472667864e-31,    4.8443060572248849e-33,    2.6184712324366570e-35,    1.3427721706878071e-37,
    6.5499595433631200e-40,    3.0464298367637618e-42,    1.3539432102932105e-44,    5.7613604038844722e-47,
    2.3515380359474359e-49,    9.2215813255091594e-52,    3.4797942600568092e-54,    1.2653635861318403e-56,
    4.4398194247851606e-59,    1.5050068070553481e-61,    4.9343971883782927e-64,    1.5664599968418562e-66,
    4.8198326443267974e-69,    1.4387435647627667e-71,    4.1702371527725086e-74,    1.1747056251526428e-76,
    3.2183480628783118e-79,    8.5822020575759422e-82,    2.2291287335817865e-84,    5.6433286063797016e-87,
    1.3934061803746984e-89,    3.3575862173844989e-92,    7.9001601322133090e-95,    1.8161193840789245e-97,
   4.0811469990280317e-100,   8.9695114502053339e-103,   1.9289184605530985e-105,   4.0608633424712965e-108,
   8.3728792322297402e-111,   1.6914839873992127e-113,   3.3494603635974965e-116,   6.5037824766689424e-119,
   1.2388112993137078e-121,   2.3155271974841065e-124,   4.2486597105794472e-127,   7.6552183091210791e-130,
   1.3549017248996221e-132,   2.3563438208226772e-135,   4.0279265832768777e-138,   6.7696057039933381e-141
},
// x = 0.2100
{
    8.1655519091653161e-01,    5.6991519004747228e-02,    2.3906337058567490e-03,    7.1668865300814703e-05,
    1.6715291629249101e-06,    3.1901175461419082e-08,    5.1521018391025967e-10,    7.2116955458727459e-12,
    8.9073490777598479e-14,    9.8438768620360968e-16,    9.8429782242457693e-18,    8.9863779032206951e-20,
    7.5480643262743712e-22,    5.8703860834905067e-24,    4.2507607228974856e-26,    2.8794234635789744e-28,
    1.8322904281161826e-30,    1.0993368211382333e-32,    6.2392885794068048e-35,    3.3595242748933938e-37,
    1.7206889050353714e-39,    8.4031729097600013e-42,    3.9213989274476590e-44,    1.7520808644989147e-46,
    7.5087854848923158e-49,    3.0918024040924284e-51,    1.2250352498909669e-53,    4.6773415225322317e-56,
    1.7232084905899966e-58,    6.1333787956534561e-61,    2.1114668311124209e-63,    7.0381469757132763e-66,
    2.2738398437308878e-68,    7.1268927195589055e-71,    2.1690347808210586e-73,    6.4154003999822524e-76,
    1.8455112773600359e-78,    5.1673921168089190e-81,    1.4092785423361146e-83,    3.7461576497372430e-86,
    9.7121968659401475e-89,    2.4572874613709941e-91,    6.0709092892053016e-94,    1.4653835513963547e-96,
    3.4576277548579943e-99,   7.9790993945231857e-102,   1.8017231280661072e-104,   3.9827373283556139e-107,
   8.6223814236352699e-110,   1.8289819323701480e-112,   3.8028176000763204e-115,   7.7532858218326643e-118,
   1.5506510777217655e-120,   3.0433223835225305e-123,   5.8632602852335518e-126,   1.1092615593231656e-128,
   2.0614525392762921e-131,   3.7643792554009048e-134,   6.7565567499266321e-137,   1.1923298923516302e-139
},
// x = 0.2200
{
    8.0900813390195137e-01,    5.9136711998269184e-02,    2.5984248346443469e-03,    8.1602119988568542e-05,
    1.9937440989837637e-06,    3.9861393778217428e-08,    6.7441007289210689e-10,    9.8894709565733483e-12,
    1.2796221665108502e-13,    1.4814880804231664e-15,    1.5518796356959701e-17,    1.4842816792173235e-19,
    1.3060742331396074e-21,    1.0641428595163063e-23,    8.0723733231541561e-26,    5.7285100531191944e-28,
    3.8188466801991167e-30,    2.4003282051686383e-32,    1.4271743083083033e-34,    8.0504831946533296e-37,
    4.3196528849527208e-39,    2.2099996871142166e-41,    1.0804195675025853e-43,    5.0571768020239112e-46,
    2.2705252022483907e-48,    9.7942470678981785e-51,    4.0654690176652062e-53,    1.6261625017606978e-55,
    6.2763263463407081e-58,    2.3402936060031193e-60,    8.4402968705862290e-63,    2.9473704209512305e-65,
    9.9756044065723555e-68,    3.2755373036421303e-70,    1.0443638949884678e-72,    3.2360269209331259e-75,
    9.7523236871024507e-78,    2.8606576400386840e-80,    8.1732425122352088e-83,    2.2760756361205510e-85,
    6.1818893223517350e-88,    1.6385618323067490e-90,    4.2409558092458472e-93,    1.0724189034920090e-95,
    2.6509072903730000e-98,   6.4087502043271792e-101,   1.5160401302871414e-103,   3.5108113356392147e-106,
   7.9626247527907483e-109,   1.7694636022663404e-111,   3.8542592215749094e-114,   8.2323614954970421e-117,
   1.7248683112701560e-119,   3.5464434928474603e-122,   7.1579307093830476e-125,   1.4186834951367968e-127,
   2.7620283759344676e-130,   5.2838613644408898e-133,   9.9354312754295210e-136,   1.8367962465826896e-138
},
// x = 0.2300
{
    8.0155729237624795e-01,    6.1237088596587372e-02,    2.8126585077171669e-03,    9.2338428822870790e-05,
    2.3585000645764747e-06,    4.9295861182689951e-08,    8.7192105652019653e-10,    1.3366683722341685e-11,
    1.8081375878215047e-13,    2.1885080088254730e-15,    2.3966748741821547e-17,    2.3964544176518239e-19,
    2.2045653033115096e-21,    1.8778361757595997e-23,    1.4892307197829835e-25,    1.1048559844676366e-27,
    7.7001587473636208e-30,    5.0598976333850313e-32,    3.1452264727443372e-34,    1.8548157892349223e-36,
    1.0404751990470629e-38,    5.5651803201350052e-41,    2.8443543561118554e-43,    1.3918861181593556e-45,
    6.5332047083808430e-48,    2.9462895610902026e-50,    1.2785552865679895e-52,    5.3465955267788959e-55,
    2.1573642601686045e-57,    8.4099404530256675e-60,    3.1709175063921333e-62,    1.1576215957552288e-64,
    4.0961497374848246e-67,    1.4061248648466917e-69,    4.6870322721922564e-72,    1.5183189015467775e-74,
    4.7836982641756560e-77,    1.4669873632724559e-79,    4.3818721998253180e-82,    1.2757243980285845e-84,
    3.6223988003727459e-87,    1.0037897313439806e-89,    2.7161174905421788e-92,    7.1804914356040925e-95,
    1.8556204979637734e-97,   4.6900005142270358e-100,   1.1598856554597777e-102,   2.8081280981171504e-105,
   6.6584113887356596e-108,   1.5468954721150694e-110,   3.5226153406676861e-113,   7.8659957806663991e-116,
   1.7230195343925285e-118,   3.7036700509244270e-121,   7.8150494243929022e-124,   1.6193277359410532e-126,
   3.2959633905242769e-129,   6.5919008642718346e-132,   1.2958388361624234e-134,   2.5045532550874342e-137
},
// x = 0.2400
{
    7.9420126707053995e-01,    6.3293453468877242e-02,    3.0330987095744419e-03,    1.0389701940969447e-04,
    2.7689767916862028e-06,    6.0389721461884545e-08,    1.1145580164948362e-09,    1.7828901747577956e-11,
    2.5165727121411637e-13,    3.1783699113782008e-15,    3.6319896675541874e-17,    3.7895226828683147e-19,
    3.6376313871735837e-21,    3.2332122916673673e-23,    2.6755904779371061e-25,    2.0713082669788099e-27,
    1.5063308947718785e-29,    1.0328666747579610e-31,    6.6994083185828087e-34,    4.1225643111562609e-36,
    2.4131295388480724e-38,    1.3468229080460137e-40,    7.1828598961590267e-43,    3.6677516197390078e-45,
    1.7964083680045750e-47,    8.4535062999173041e-50,    3.8279272148659337e-52,    1.6703375502843755e-54,
    7.0328797575485643e-57,    2.8607866591301066e-59,    1.1255385372096959e-61,    4.2877055462510839e-64,
    1.5831318785560664e-66,    5.6708495072025163e-69,    1.9724462030280417e-71,    6.6673496897066181e-74,
    2.1919823168127352e-76,    7.0142734539175841e-79,    2.1862463491651384e-81,    6.6417012766142098e-84,
    1.9678946293634220e-86,    5.6902512662074796e-89,    1.6066466668940663e-91,    4.4320957662348831e-94,
    1.1951633920871881e-96,    3.1520578227625084e-99,   8.1342897370486646e-102,   2.0549656150341647e-104,
   5.0844205094530299e-107,   1.2325796898406942e-109,   2.9288860165265695e-112,   6.8245524294281040e-115,
   1.5598897008961579e-117,   3.4988007412741315e-120,   7.7037447734425708e-123,   1.6656668965628861e-125,
   3.5376839226944208e-128,   7.3829609284629646e-131,   1.5144472584843133e-133,   3.0543351939272286e-136
},
// x = 0.2500
{
    7.8693868057473315e-01,    6.5306597126334243e-02,    3.2595150587226840e-03,    1.1629595188782926e-04,
    3.2284058630996886e-06,    7.3340816240528004e-08,    1.4099485164549209e-09,    2.3493384872158080e-11,
    3.4542412543554494e-13,    4.5443425410308293e-15,    5.4092317201869912e-17,    5.8789607375420376e-19,
    5.8784164831595437e-21,    5.4425438249641724e-23,    4.6915219823531756e-25,    3.7832543449057687e-27,
    2.8659467002319117e-29,    2.0470059964328955e-31,    1.3830522585823730e-33,    8.8653730983691105e-36,
    5.4055236791993669e-38,    3.1426448214996625e-40,    1.7458621994495233e-42,    9.2862490521952953e-45,
    4.7377636807384735e-47,    2.3223794788653456e-49,    1.0954385316969109e-51,    4.9791667893269335e-54,
    2.1838044986161827e-56,    9.2532482037480681e-59,    3.7922531638557928e-61,    1.5048393993047481e-63,
    5.7877607825937833e-66,    2.1595830365036572e-68,    7.8244763980530612e-71,    2.7550641012745767e-73,
    9.4350433269538126e-76,    3.1449804060831596e-78,    1.0210870433498830e-80,    3.2312565519890679e-83,
    9.9729213374552010e-86,    3.0038653581990266e-88,    8.8348234440595729e-91,    2.5387218770682778e-93,
    7.1311861967474376e-96,    1.9591026189755740e-98,   5.2663676349660547e-101,   1.3858768201904665e-103,
   3.5718242280636426e-106,   9.0197017732406200e-109,   2.2325860357852272e-111,   5.4188668295949544e-114,
   1.2901992106940765e-116,   3.0144679832065372e-119,   6.9138816777030955e-122,   1.5571727989061149e-124,
   3.4450559948865907e-127,   7.4892173746895632e-130,   1.6002529393325342e-132,   3.3618613306034000e-135
},
// x = 0.2600
{
    7.7976817698039536e-01,    6.7277296171929790e-02,    3.4916826888992483e-03,    1.2955215459523910e-04,
    3.7400651834043138e-06,    8.8359785089865225e-08,    1.7665834484750340e-09,    3.0612666113608817e-11,
    4.6809576683234621e-13,    6.4044361092773911e-15,    7.9281923613646393e-17,    8.9612509825688448e-19,
    9.3187678450882892e-21,    8.9728536856850769e-23,    8.0440226147516537e-25,    6.7461538515589179e-27,
    5.3148404679272592e-29,    3.9479611277801240e-31,    2.7741130017031123e-33,    1.8493304877221571e-35,
    1.1727011987757453e-37,    7.0905037327870710e-40,    4.0966045575063852e-42,    2.2661402602367665e-44,
    1.2024092453948332e-46,    6.1297761887826422e-49,    3.0069902853512420e-51,    1.4214556664813196e-53,
    6.4837025375450961e-56,    2.8571711817649400e-58,    1.2177892465617572e-60,    5.0257139276936673e-63,
    2.0102543675853435e-65,    7.8008730310924866e-68,    2.9394188432893433e-70,    1.0763928615288707e-72,
    3.8336806659421871e-75,    1.3289937409935467e-77,    4.4874614648721077e-80,    1.4768704500050129e-82,
    4.7405241491516038e-85,    1.4849692397763380e-87,    4.5422173296186569e-90,    1.3574324085073818e-92,
    3.9654997801754185e-95,    1.1329908872582065e-97,   3.1674771698236291e-100,   8.6688212929742424e-103,
   2.3235852269932606e-105,   6.1023037851572826e-108,   1.5708798755969782e-110,   3.9653030560262397e-113,
   9.8187865841142863e-116,   2.3858595469583844e-118,   5.6910093243294510e-121,   1.3330220268723329e-123,
   3.0671143722657062e-126,   6.9343106984166100e-129,   1.5409504512777086e-131,   3.3667666924648965e-134
},
// x = 0.2700
{
    7.7268842152964878e-01,    6.9206313545726400e-02,    3.7293821326905281e-03,    1.4368145884830638e-04,
    4.3072736612375134e-06,    1.0567014038936908e-07,    2.1938675965527534e-09,    3.9478333125393275e-11,
    6.2686736423863462e-13,    8.9064881459082381e-15,    1.1449471175893670e-16,    1.3438979909355370e-18,
    1.4512531094646173e-20,    1.4511180163273775e-22,    1.3509313723982354e-24,    1.1765338112461717e-26,
    9.6255782260786010e-29,    7.4250281008856410e-31,    5.4179900462673497e-33,    3.7507451927447198e-35,
    2.4699008053266691e-37,    1.5508095236972212e-39,    9.3045364427805863e-42,    5.3449900457622554e-44,
    2.9451126851100486e-46,    1.5591352550036085e-48,    7.9425658789758227e-51,    3.8989871316564749e-53,
    1.8468486078478348e-55,    8.4515088824515999e-58,    3.7407608411239574e-60,    1.6031546784255810e-62,
    6.6591464267736173e-65,    2.6834943044030740e-67,    1.0500473635771650e-69,    3.9930816986930266e-72,
    1.4768735665989803e-74,    5.3166777261353052e-77,    1.8642672507500046e-79,    6.3714737144093225e-82,
    2.1238015425343794e-84,    6.9086806184708398e-87,    2.1945004455999560e-89,    6.8104545047271605e-92,
    2.0660743428315123e-94,    6.1300578838434179e-97,    1.7796795398354566e-99,   5.0579965733195346e-102,
   1.4078852656873401e-104,   3.8396590945201771e-107,   1.0264363275167463e-109,   2.6906401977533116e-112,
   6.9187441868975984e-115,   1.7458404245082320e-117,   4.3245327932807159e-120,   1.0519072684915716e-122,
   2.5133926480440139e-125,   5.9009768541620329e-128,   1.3617567593806974e-130,   3.0896845678217685e-133
},
// x = 0.2800
{
    7.6569810026997343e-01,    7.1094398765835745e-02,    3.9723992074418082e-03,    1.5869863285401228e-04,
    4.9333860966717933e-06,    1.2550831813331645e-07,    2.7021700057907320e-09,    5.0425007318182313e-11,
    8.3032803096484577e-13,    1.2234009602409788e-14,    1.6309365846723067e-16,    1.9852173674826338e-18,
    2.2231852585797974e-20,    2.3052946495852298e-22,    2.2256076546943118e-24,    2.0100722326198733e-26,
    1.7054000802551749e-28,    1.3642374770167015e-30,    1.0323398429844038e-32,    7.4113073025673403e-35,
    5.0611555381579603e-37,    3.2955026461231000e-39,    2.0504589748399809e-41,    1.2215084456025050e-43,
    6.9798292919001088e-46,    3.8319519986025519e-48,    2.0243730260622750e-50,    1.0305641324564224e-52,
    5.0623022873404941e-55,    2.4023962108243721e-57,    1.1027167485184013e-59,    4.9008694998248731e-62,
    2.1111057804328428e-64,    8.8223820061176620e-67,    3.5800397544990075e-69,    1.4118253082532823e-71,
    5.4151428186091903e-74,    2.0216258741983694e-76,    7.3512720704584428e-79,    2.6054822297819922e-81,
    9.0065002109718204e-84,    3.0383036572121242e-86,    1.0008423588535869e-88,    3.2210692306920076e-91,
    1.0133602856022745e-93,    3.1180027635396126e-96,    9.3874519004172297e-99,   2.7668043891307623e-101,
   7.9865866420099427e-104,   2.2588148748727396e-106,   6.2620138472558922e-109,   1.7022826863523816e-111,
   4.5393888204206600e-114,   1.1878694633937560e-116,   3.0513880233864126e-119,   7.6971469032344401e-122,
   1.9072461332196459e-124,   4.6437026577508207e-127,   1.1113071988519522e-129,   2.6148262306971007e-132
},
// x = 0.2900
{
    7.5879591971482407e-01,    7.2942288165092795e-02,    4.2205249035234321e-03,    1.7461741477467100e-04,
    5.6217882677736265e-06,    1.4812370583446727e-07,    3.3028740524527723e-09,    6.3834517618832393e-11,
    1.0886583752350287e-12,    1.6612863675535732e-14,    2.2937580337772521e-16,    2.8917068038943684e-18,
    3.3539620585633985e-20,    3.6020168454262001e-22,    3.6016799417713675e-24,    3.3690365483181124e-26,
    2.9604529338684400e-28,    2.4527874367863444e-30,    1.9223429871533802e-32,    1.4293593548076397e-34,
    1.0109620517478925e-36,    6.8178198550170735e-39,    4.3935314325706640e-41,    2.7108033846244189e-43,
    1.6042990350173305e-45,    9.1222008928056636e-48,    4.9912489438246743e-50,    2.6316788470448585e-52,
    1.3388908442476485e-54,    6.5808351204830205e-57,    3.1285252827347159e-59,    1.4400852378815963e-61,
    6.4248716073860908e-64,    2.7808639886940901e-66,    1.1687488594686460e-68,    4.7736854899031813e-71,
    1.8963664764276181e-73,    7.3325102619984581e-76,    2.7615565948349680e-78,    1.0137226423284204e-80,
    3.6293319615745222e-83,    1.2680647260454332e-85,    4.3262892766936910e-88,    1.4420807626777230e-90,
    4.6988660521492837e-93,    1.4974259494650834e-95,    4.6693482987158941e-98,   1.4253669985863286e-100,
   4.2613691712257044e-103,   1.2482693593274960e-105,   3.5841107701375022e-108,   1.0091107193243142e-110,
   2.7870468386011987e-113,   7.5536238800335400e-116,   2.0096657788304142e-118,   5.2504429572572430e-121,
   1.3474500917744116e-123,   3.3978963844329950e-126,   8.4220854653114974e-129,   2.0524290082279181e-131
},
// x = 0.3000
{
    7.5198060650995602e-01,    7.4750705123524053e-02,    4.4735552747137636e-03,    1.9145054496441690e-04,
    6.3758922105259521e-06,    1.7377864863844495e-07,    4.0084271163009537e-09,    8.0140265403706931e-11,
    1.4138461156069579e-12,    2.2318852645986809e-14,    3.1878136112663658e-16,    4.1573671222344285e-18,
    4.9881755330409431e-20,    5.5417803364462984e-22,    5.7323023927251574e-24,    5.5469014531395294e-26,
    5.0422448097816712e-28,    4.3216237968893061e-30,    3.5038007748010653e-32,    2.6950796799345167e-34,
    1.9719088619238355e-36,    1.3756863860933244e-38,    9.1708523401693290e-41,    5.8535067926223226e-43,
    3.5836506102023068e-45,    2.1079595855405366e-47,    1.1931478339346259e-49,    6.5078922697453566e-52,
    3.4251147977477734e-54,    1.7415402458813069e-56,    8.5647514536732778e-59,    4.0783634406687171e-61,
    1.8822826896899865e-63,    8.4279673747465428e-66,    3.6642663257740672e-68,    1.5482546619363553e-70,
    6.3625858026976427e-73,    2.5449946594114250e-75,    9.9154169082820748e-78,    3.7652952359885256e-80,
    1.3945351228694670e-82,    5.0404240964597517e-85,    1.7789515602407651e-87,    6.1342444249728860e-90,
    2.0676998628432259e-92,    6.8165204646973309e-95,    2.1988551702721932e-97,   6.9436853531303125e-100,
   2.1475114260656370e-102,   6.5075518087001539e-105,   1.9329194585773193e-107,   5.6298156510250397e-110,
   1.6085058723078737e-112,   4.5097947480030269e-115,   1.2412186793925516e-117,   3.3546210091371788e-120,
   8.9060117952699091e-123,   2.3232918845395308e-125,   5.9571201709830441e-128,   1.5017856143329737e-130
},
// x = 0.3100
{
    7.4525090710536379e-01,    7.6520360296688739e-02,    4.7312913309608415e-03,    2.0920979739622890e-04,
    7.1991316864518076e-06,    2.0274843472487737e-07,    4.8323897626094908e-09,    9.9831776737490514e-11,
    1.8199204728530637e-12,    2.9686290709638542e-14,    4.3813903650764200e-16,    5.9043655724584494e-18,
    7.3203712337682142e-20,    8.4038393569715136e-22,    8.9824543818230909e-24,    8.9816107249449578e-26,
    8.4365687813521792e-28,    7.4718350553481646e-30,    6.2597692683569816e-32,    4.9754150273657558e-34,
    3.7616941296792352e-36,    2.7117843515588621e-38,    1.8680332339091470e-40,    1.2320556171584826e-42,
    7.7943378543563527e-45,    4.7375663446504686e-47,    2.7709380235496541e-49,    1.5617535589740832e-51,
    8.4935047248366179e-54,    4.4625697672784107e-56,    2.2678066157671555e-58,    1.1158786562144529e-60,
    5.3217653911712484e-63,    2.4622581772325192e-65,    1.1062102354064419e-67,    4.8298424633482049e-70,
    2.0509929918059976e-72,    8.4772966336056326e-75,    3.4128836893807429e-77,    1.3392127281655663e-79,
    5.1253087848036307e-82,    1.9142458811224564e-84,    6.9812766087244113e-87,    2.4875504472051072e-89,
    8.6643987500174055e-92,    2.9515748497036910e-94,    9.8384758184615069e-97,    3.2104165239471704e-99,
   1.0259991371888363e-101,   3.2126936942064668e-104,   9.8606519326907425e-107,   2.9677426577233873e-109,
   8.7618319494056292e-112,   2.5384537609168108e-114,   7.2193983084788407e-117,   2.0162129090308284e-119,
   5.5311626522519839e-122,   1.4909984136573296e-124,   3.9504813503623389e-127,   1.0291101221171591e-129
},
// x = 0.3200
{
    7.3860558743273663e-01,    7.8251951839961489e-02,    4.9935389330965927e-03,    2.2790601029787618e-04,
    8.0949578324210312e-06,    2.3532126103488059e-07,    5.7894843470081567e-09,    1.2345943767432620e-10,
    2.3232060241130793e-12,    3.9117643318955292e-14,    5.9595205010733131e-16,    8.2900306616759512e-18,
    1.0609629937228855e-19,    1.2572732159118510e-21,    1.3871779726096393e-23,    1.4317823436560707e-25,
    1.3882719282152868e-27,    1.2691768405558547e-29,    1.0975885732089998e-31,    9.0052782949088477e-34,
    7.0281016983902802e-36,    5.2299384630025822e-38,    3.7188873055445748e-40,    2.5318958052915255e-42,
    1.6534152264626705e-44,    1.0373977056186018e-46,    6.2633133024195704e-49,    3.6439905359947771e-51,
    2.0456867854865082e-53,    1.1094934693430227e-55,    5.8201385310102060e-58,    2.9561869201655781e-60,
    1.4553193422353918e-62,    6.9506249940373085e-65,    3.2234108829759958e-67,    1.4527762031894634e-69,
    6.3682149375159722e-72,    2.7170568630476275e-74,    1.1291474810015740e-76,    4.5736887590194038e-79,
    1.8068618658316103e-81,    6.9661133140317261e-84,    2.6225004632546647e-86,    9.6458511521145568e-89,
    3.4681273518969880e-91,    1.2195465105908224e-93,    4.1962404336972710e-96,    1.4134547552873298e-98,
   4.6628938011108159e-101,   1.5071825613776397e-103,   4.7751848744309885e-106,   1.4835385358415753e-108,
   4.5212190919844512e-111,   1.3521284177367402e-113,   3.9695177223500022e-116,   1.1443561271524805e-118,
   3.2406289835574307e-121,   9.0173337626312899e-124,   2.4662611811212375e-126,   6.6319156767498735e-129
},
// x = 0.3300
{
    7.3204343258833460e-01,    7.9946165628833374e-02,    5.2601086898525643e-03,    2.4754911601477137e-04,
    9.0668349872628912e-06,    2.7179818032903148e-07,    6.8956429618407187e-09,    1.5163940803357878e-10,
    2.9425966780481942e-12,    5.1094315944556128e-14,    8.0272972526369715e-16,    1.1515245957206377e-17,
    1.5197673113171624e-19,    1.8572351066687823e-21,    2.1131495179439373e-23,    2.2492423320053305e-25,
    2.2490302984063184e-27,    2.1203359901066783e-29,    1.8909677802486725e-31,    1.5999407028736694e-33,
    1.2876776160955189e-35,    9.8816209539530923e-38,    7.2461489509554937e-40,    5.0874810469605073e-42,
    3.4261134507044432e-44,    2.2168076300219160e-46,    1.3802248875541494e-48,    8.2810616767072689e-51,
    4.7941436271497182e-53,    2.6813890320813854e-55,    1.4505464057752295e-57,    7.5978981717843153e-60,
    3.8572980041233099e-62,    1.8998184446807088e-64,    9.0858862475291290e-67,    4.2229288254862781e-69,
    1.9089572532518047e-71,    8.3992535312195901e-74,    3.5996156443229865e-76,    1.5036113262469383e-78,
    6.1257246980188709e-81,    2.4354915034477466e-83,    9.4552983632825388e-86,    3.5864420424478657e-88,
    1.3297864726000071e-90,    4.8222405421921194e-93,    1.7110965210597832e-95,    5.9437387274352992e-98,
   2.0220737499841685e-100,   6.7401724262304840e-103,   2.2022115022916089e-105,   7.0555580408304235e-108,
   2.2174396052740470e-110,   6.8387685729668994e-113,   2.0704342354283053e-115,   6.1552915836311303e-118,
   1.7975479652551777e-120,   5.1581393698638840e-123,   1.4548484431270729e-125,   4.0344231535938122e-128
},
// x = 0.3400
{
    7.2556324652119175e-01,    8.1603675475302714e-02,    5.5308158567560356e-03,    2.6814817011733044e-04,
    1.0118236689945210e-05,    3.1249303054554877e-07,    8.1680546480394487e-09,    1.8505870874646289e-10,
    3.6998504013811283e-12,    6.6188677406557452e-14,    1.0713698382112249e-15,    1.5834458217059810e-17,
    2.1531176306145576e-19,    2.7109331292511768e-21,    3.1779267979944702e-23,    3.4850743236292200e-25,
    3.5903233253552715e-27,    3.4874313733758413e-29,    3.2044099778790876e-31,    2.7933862507090660e-33,
    2.3163147751239872e-35,    1.8313953014021097e-37,    1.3836452719976736e-39,    1.0008846401369789e-41,
    6.9445925967993437e-44,    4.6295304132217467e-46,    2.9697696672416040e-48,    1.8357899218799191e-50,
    1.0949949467561163e-52,    6.3099377018238265e-55,    3.5169086687849987e-57,    1.8979606250525516e-59,
    9.9275305269491582e-62,    5.0377253447791510e-64,    2.4822988436314347e-66,    1.1886799771153920e-68,
    5.5362008117696621e-71,    2.5096941316163725e-73,    1.1081555707795054e-75,    4.7691909845709007e-78,
    2.0018482148738901e-80,    8.2002077222765082e-83,    3.2800318156155156e-85,    1.2818323771593334e-87,
    4.8968178398443655e-90,    1.8295553014016614e-92,    6.6886092859920362e-95,    2.3937880311054160e-97,
   8.3904962196041599e-100,   2.8815512465744624e-102,   9.7001637343165952e-105,   3.2019615708332072e-107,
   1.0368149835395690e-109,   3.2945196302105716e-112,   1.0276385065293149e-114,   3.1476925417232982e-117,
   9.4708490619958077e-120,   2.8000530572406815e-122,   8.1368387621165390e-125,   2.3247924108224290e-127
},
// x = 0.3500
{
    7.1916385172655795e-01,    8.3225143340419486e-02,    5.8054802372508041e-03,    2.8971137977006322e-04,
    1.1252641844214887e-05,    3.5773234739496739e-07,    9.6252118016219455e-09,    2.2448047758089818e-10,
    4.6199052977389102e-12,    8.5077405008248060e-14,    1.4175972911591060e-15,    2.1567538701682776e-17,
    3.0189076280966334e-19,    3.9127867068232885e-21,    4.7216854723904777e-23,    5.3302969975253097e-25,
    5.6527458234601515e-27,    5.6522112002151784e-29,    5.3462324497466883e-31,    4.7975334011100291e-33,
    4.0951708122524027e-35,    3.3330675721787796e-37,    2.5922357561433288e-39,    1.9302856593063942e-41,
    1.3787078902756550e-43,    9.4612920477394693e-46,    6.2477605046793970e-48,    3.9756922495291103e-50,
    2.4411256222321276e-52,    1.4480760821894029e-54,    8.3083684276836946e-57,    4.6156221673182716e-59,
    2.4852651080258029e-61,    1.2982384178353016e-63,    6.5851026795660076e-66,    3.2461006575067241e-68,
    1.5563148093945000e-70,    7.2626483907652956e-73,    3.3011373366568301e-75,    1.4625012027330946e-77,
    6.3193344986406828e-80,    2.6647333393379435e-82,    1.0972249642126194e-84,    4.4140535867024523e-87,
    1.7358375153265232e-89,    6.6762015007129533e-92,    2.5125141155611589e-94,    9.2565079018500062e-97,
    3.3399344728678246e-99,   1.1807704489359921e-101,   4.0917306020206740e-104,   1.3903781454424605e-106,
   4.6345432862688478e-109,   1.5159561806022216e-111,   4.8676999197437564e-114,   1.5348453452765793e-116,
   4.7539009465039033e-119,   1.4468262461001640e-121,   4.3280746222395432e-124,   1.2729522945907253e-126
},
// x = 0.3600
{
    7.1284408894448370e-01,    8.4811219543061339e-02,    6.0839260856356530e-03,    3.1224613137924171e-04,
    1.2473531044725869e-05,    4.0785526109510334e-07,    1.1286955708817937e-08,    2.7074938778900820e-10,
    5.7312176092598291e-12,    1.0855624062747487e-13,    1.8604650319879584e-15,    2.9113761510581749e-17,
    4.1915770079024484e-19,    5.5878445925431461e-21,    6.9356349508723899e-23,    8.0532656260977908e-25,
    8.7843951040604029e-27,    9.0344737574907708e-29,    8.7895094438013262e-31,    8.1127358385025237e-33,
    7.1228542359119133e-35,    5.9629204728850952e-37,    4.7700441026907034e-39,    3.6534452171945986e-41,
    2.6840246441985199e-43,    1.8945147994554120e-45,    1.2867830335280006e-47,    8.4222316965957095e-50,
    5.3190992545726571e-52,    3.2454352233454116e-54,    1.9152742323346699e-56,    1.0944077833052567e-58,
    6.0611550469916741e-61,    3.2566487316064233e-63,    1.6990761304807747e-65,    8.6148184941125167e-68,
    4.2483030806173264e-70,    2.0391397181980816e-72,    9.5334371325207179e-75,    4.3442631149841564e-77,
    1.9307463881358335e-79,    8.3741678531285956e-82,    3.5466442296957042e-84,    1.4675523595767314e-86,
    5.9360718603039053e-89,    2.3483001598795902e-91,    9.0900608276815732e-94,    3.4446061850599132e-96,
    1.2783932902145675e-98,   4.6486426218244981e-101,   1.6569212830241433e-103,   5.7911117872290747e-106,
   1.9855011380790564e-108,   6.6801165158531276e-111,   2.2062533820865437e-113,   7.1553424428085925e-116,
   2.2795553894339865e-118,   7.1359307465583869e-121,   2.1956505612658582e-123,   6.6422444351245174e-126
},
// x = 0.3700
{
    7.0660281686346693e-01,    8.6362542965000774e-02,    6.3659820121105375e-03,    3.3575901753563639e-04,
    1.3784383059809071e-05,    4.6321337811858154e-07,    1.3174521148538561e-08,    3.2479722398350840e-10,
    7.0661221801109489e-12,    1.3755624868110967e-13,    2.4229235134274840e-15,    3.8968189279389227e-17,
    5.7661227626303947e-19,    7.9003426973689172e-21,    1.0078213071303042e-22,    1.2027225638306143e-24,
    1.3483473168162139e-26,    1.4252450804459872e-28,    1.4251098892865304e-30,    1.3519115945714596e-32,
    1.2199230580321139e-34,    1.0496269803564565e-36,    8.6297077312672115e-39,    6.7931959093394035e-41,
    5.1292751175353753e-43,    3.7210503579780610e-45,    2.5975924129027812e-47,    1.7473949569950940e-49,
    1.1342277494045179e-51,    7.1126831376558159e-54,    4.3140967509168811e-56,    2.5335911717280231e-58,
    1.4421527186803036e-60,    7.9638911273367521e-63,    4.2703730134494579e-65,    2.2253468776686698e-67,
    1.1278873390710494e-69,    5.5641123082772963e-72,    2.6736041868340575e-74,    1.2521675776982017e-76,
    5.7196613572312485e-79,    2.5496790820535115e-81,    1.1098397607509442e-83,    4.7199247287556628e-86,
    1.9621827316026101e-88,    7.9779765598934056e-91,    3.1739845043578700e-93,    1.2361650740699116e-95,
    4.7152016204224436e-98,   1.7622229433315723e-100,   6.4555832560035822e-103,   2.3189665729929131e-105,
   8.1714969256778845e-108,   2.8256246486857018e-110,   9.5914613823282730e-113,   3.1971188996689788e-115,
   1.0468332133961292e-117,   3.3680378095931551e-120,   1.0650955013611163e-122,   3.3116099896312024e-125
},
// x = 0.3800
{
    7.0043891182906681e-01,    8.7879741252336371e-02,    6.6514808895676318e-03,    3.6025586326844829e-04,
    1.5188671466163169e-05,    5.2417064879454076e-07,    1.5310580005413714e-08,    3.8764860933462591e-10,
    8.6612158890081249e-12,    1.7316166847299905e-13,    3.1324653581714692e-15,    5.1740784575685990e-17,
    7.8629174837018014e-19,    1.1064288174168774e-20,    1.4495704766166348e-22,    1.7766420944527664e-24,
    2.0455745779937996e-26,    2.2206619499275571e-28,    2.2804516578695659e-30,    2.2217779140108242e-32,
    2.0590401635161135e-34,    1.8194811185915135e-36,    1.5363458361106891e-38,    1.2420740780303093e-40,
    9.6318546885571574e-43,    7.1762926886488637e-45,    5.1450115999376699e-47,    3.5545715685383765e-49,
    2.3696126370282195e-51,    1.5261299604737428e-53,    9.5066818988495990e-56,    5.7339868937492335e-58,
    3.3520658113204319e-60,    1.9011122744453560e-62,    1.0469585098706803e-64,    5.6032838068627838e-67,
    2.9167009476662412e-69,    1.4777581973425055e-71,    7.2926595508704930e-74,    3.5077824011063560e-76,
    1.6455909675170546e-78,    7.5338767371002376e-81,    3.3680203055836059e-83,    1.4710618953807655e-85,
    6.2808264485000676e-88,    2.6227179433673478e-90,    1.0716306773640165e-92,    4.2864555412658217e-95,
    1.6792047560738178e-97,   6.4453393183352325e-100,   2.4249454901961440e-102,   8.9462813899580896e-105,
   3.2376602242297656e-107,   1.1498090215483583e-109,   4.0084606755807672e-112,   1.3722500191535418e-114,
   4.6145948054383200e-117,   1.5248062672045574e-119,   4.9523108738397823e-122,   1.5813943361892284e-124
},
// x = 0.3900
{
    6.9435126755740573e-01,    8.9363431013348774e-02,    6.9402597624153390e-03,    3.8574175162622338e-04,
    1.6689861430868046e-05,    5.9110322157641500e-07,    1.7719283840952835e-08,    4.6042687798711966e-10,
    1.0557764525151742e-11,    2.1662945486434138e-13,    4.0218522735762655e-15,    6.8178594850105297e-17,
    1.0633465187735993e-18,    1.5356467182274524e-20,    2.0648307767045943e-22,    2.5973042147197080e-24,
    3.0691372226293903e-26,    3.4194941791509324e-28,    3.6039518624691499e-30,    3.6036091097073244e-32,
    3.4275276182648897e-34,    3.1084435192722767e-36,    2.6937906724923881e-38,    2.2351255029236372e-40,
    1.7788691774856087e-42,    1.3602351864169777e-44,    1.0008755571448856e-46,    7.0967732902483257e-49,
    4.8554673968120071e-51,    3.2094106154431663e-53,    2.0518370568093650e-55,    1.2701376697970368e-57,
    7.6205598766410358e-60,    4.4357023531599460e-62,    2.5070582779190616e-64,    1.3770761083885692e-66,
    7.3567775769800145e-69,    3.8254235898339110e-71,    1.9375037634325534e-73,    9.5646646560793130e-76,
    4.6051047256284654e-78,    2.1637977404097770e-80,    9.9278089711994872e-83,    4.4503097067853094e-85,
    1.9500990918187161e-87,    8.3574173365202765e-90,    3.5046630652352723e-92,    1.4387326693447583e-94,
    5.7845036502945897e-97,    2.2787092001108463e-99,   8.7988474750142162e-102,   3.3315555882975672e-104,
   1.2374181807845925e-106,   4.5101569818142583e-109,   1.6137056064656134e-111,   5.6697077034593963e-114,
   1.9567788708975382e-116,   6.6359576782547645e-119,   2.2119617287190441e-121,   7.2492097635054355e-124
},
// x = 0.4000
{
    6.8833879485347305e-01,    9.0814218012844394e-02,    7.2321597571391848e-03,    4.1222104860018865e-04,
    1.8291406636241333e-05,    6.6439928475879254e-07,    2.0426305374534360e-08,    5.4436008642628151e-10,
    1.2802133548795652e-11,    2.6941060246644413e-13,    5.1299316395544851e-15,    8.9191389833643900e-17,
    1.4267241199596203e-18,    2.1132336167700911e-20,    2.9142863980846470e-22,    3.7597815872234921e-24,
    4.5566798644081951e-26,    5.2069908672912424e-28,    5.6285552835931941e-30,    5.7722996753896773e-32,
    5.6310008825951407e-34,    5.2377072965849636e-36,    4.6553876630239885e-38,    3.9617568297932568e-40,
    3.2338801690978954e-42,    2.5362264833597453e-44,    1.9140281422013199e-46,    1.3919494299537837e-48,
    9.7676014868103603e-51,    6.6218083308118119e-53,    4.3419886293539070e-55,    2.7567104710252527e-57,
    1.6963748913336759e-59,    1.0127260802941422e-61,    5.8706840987737758e-64,    3.3073255666201588e-66,
    1.8121802299646488e-68,    9.6646934671216901e-71,    5.0204879332762318e-73,    2.5419556494124006e-75,
    1.2552568673962594e-77,    6.0492934975838090e-80,    2.8466647619528114e-82,    1.3087843408490547e-84,
    5.8820606118039101e-87,    2.5854722684436861e-89,    1.1120109452658646e-91,    4.6820700550827919e-94,
    1.9307183701063859e-96,    7.8007574807777612e-99,   3.0893613890507238e-101,   1.1997342468115752e-103,
   4.5703510918907205e-106,   1.7085190260837071e-108,   6.2697116800598303e-111,   2.2593267404961089e-113,
   7.9975183128208495e-116,   2.7817124217858544e-118,   9.5100186519377980e-121,   3.1966094050539930e-123
},
// x = 0.4100
{
    6.8240042133414724e-01,    9.2232697363054408e-02,    7.5270259947255863e-03,    4.3969742740511923e-04,
    1.9996746343175070e-05,    7.4445889639831958e-07,    2.3458878829906830e-08,    6.4078715737151561e-10,
    1.5446243144056791e-11,    3.3317334964881336e-13,    6.5025505752027300e-15,    1.1588116281957811e-16,
    1.8999780068894261e-18,    2.8845326320455270e-20,    4.0773700578582655e-22,    5.3917648724172996e-24,
    6.6978846957722898e-26,    7.8450752740737297e-28,    8.6921790447154048e-30,    9.1369713544114419e-32,
    9.1361003135341734e-34,    8.7104087726688176e-36,    7.9355195467855712e-38,    6.9219692855908859e-40,
    5.7914623016787834e-42,    4.6555919429034578e-44,    3.6012879887456477e-46,    2.6844525664357562e-48,
    1.9308255134100000e-50,    1.3416974380560575e-52,    9.0175719576796258e-55,    5.8683376893100692e-57,
    3.7014239829589439e-59,    2.2649681418364166e-61,    1.3458044577451114e-63,    7.7712948303602312e-66,
    4.3645658340084030e-68,    2.3858932087270717e-70,    1.2703755644491565e-72,    6.5929151826885082e-75,
    3.3370711614446251e-77,    1.6483934669016668e-79,    7.9508936373464631e-82,    3.7468915205069253e-84,
    1.7260602689274166e-86,    7.7766005946489404e-89,    3.4283285823016949e-91,    1.4795674509906589e-93,
    6.2537323337476117e-96,    2.5898860127257514e-98,   1.0513228786995837e-100,   4.1848124338023176e-103,
   1.6340451684963824e-105,   6.2612043287557666e-108,   2.3550991828498583e-110,   8.6988984194789901e-113,
   3.1561966523904946e-115,   1.1252386615519797e-117,   3.9430964210551446e-120,   1.3585299658689237e-122
},
// x = 0.4200
{
    6.7653509115585753e-01,    9.3619453711148160e-02,    7.8247075047976483e-03,    4.6817389213251755e-04,
    2.1809302588703071e-05,    8.3169380316640691e-07,    2.6845839106695383e-08,    7.5116414964521624e-10,
    1.8548047937645123e-11,    4.0982835958044108e-13,    8.1935756727823796e-15,    1.4957594132198878e-16,
    2.5122194353751481e-18,    3.9070168706183180e-20,    5.6573283479972500e-22,    7.6634462021662713e-24,
    9.7519877929032300e-26,    1.1700791708704473e-27,    1.3280356495049943e-29,    1.4300344938074952e-31,
    1.4647668374913082e-33,    1.4305720883649223e-35,    1.3350892722085723e-37,    1.1929671288129597e-39,
    1.0224710835483944e-41,    8.4198006499524846e-44,    6.6718911307068725e-46,    5.0946120333281999e-48,
    3.7537277708772390e-50,    2.6720142337621995e-52,    1.8396630653000436e-54,    1.2263892165990979e-56,
    7.9240401395522369e-59,    4.9671192204497729e-61,    3.0233550140246467e-63,    1.7884026642742191e-65,
    1.0289108489741900e-67,    5.7617247648569073e-70,    3.1426678314106607e-72,    1.6707393736155757e-74,
    8.6628657553066401e-77,    4.3835092152889263e-79,    2.1659175947945689e-81,    1.0455915705806688e-83,
    4.9341448898105744e-86,    2.2772501761419373e-88,    1.0284150304320947e-90,    4.5465899442289004e-93,
    1.9685904128794124e-95,    8.3514483592192935e-98,   3.4728206292814894e-100,   1.4160785189831294e-102,
   5.6642251436491516e-105,   2.2233070841099232e-107,   8.5667464342826418e-110,   3.2414260382136490e-112,
   1.2047614657510964e-114,   4.3999407125455593e-117,   1.5794458859772754e-119,   5.5744466007805661e-122
},
// x = 0.4300
{
    6.7074176474680369e-01,    9.4975061423420382e-02,    8.1250571415428328e-03,    4.9765280079056722e-04,
    2.3732477513660539e-05,    9.2652724883507379e-07,    3.0617659740055396e-08,    8.7707064735219419e-10,
    2.2172041723037854e-11,    5.0155597627940540e-13,    1.0266026971106610e-14,    1.9186838815199395e-16,
    3.2992327441489504e-18,    5.2530934031620317e-20,    7.7874681465164269e-22,    1.0800020230449763e-23,
    1.4070501456682502e-25,    1.7284148562096085e-27,    2.0084410229044544e-29,    2.2141789647395987e-31,
    2.3219441970156609e-33,    2.3217223629373053e-35,    2.2183407835515758e-37,    2.0293829030845714e-39,
    1.7807552890483597e-41,    1.5013184341725852e-43,    1.2179735529293447e-45,    9.5217771299822342e-48,
    7.1827001384874269e-50,    5.2345803378470312e-52,    3.6897725545366780e-54,    2.5183024848762716e-56,
    1.6658832260479044e-58,    1.0691061781120663e-60,    6.6622943056719599e-63,    4.0347667031321357e-65,
    2.3765631424921383e-67,    1.3625192454788293e-69,    7.6086424414888850e-72,    4.1412933145226155e-74,
    2.1984038919090230e-76,    1.1389022887539788e-78,    5.7613616459709550e-81,    2.8475015534176854e-83,
    1.3757277703439881e-85,    6.5005496407398174e-88,    3.0055675794144943e-90,    1.3603875033023846e-92,
    6.0304676666384737e-95,    2.6192456031909530e-97,    1.1151045464987978e-99,   4.6552112380978386e-102,
   1.9063884664751900e-104,   7.6610658419158976e-107,   3.0222091466427531e-109,   1.1707484473594203e-111,
   4.4549970761280028e-114,   1.6657586246986589e-116,   6.1219375776849417e-119,   2.2121002989395659e-121
},
// x = 0.4400
{
    6.6501941854365754e-01,    9.6300084766211477e-02,    8.4279315013064005e-03,    5.2813588774500164e-04,
    2.5769650816405471e-05,    1.0293937730716145e-06,    3.4806489615107856e-08,    1.0202162616099674e-09,
    2.6389787495324806e-11,    6.1083565423624351e-13,    1.2793335123386404e-14,    2.4465971098305777e-16,
    4.3047765447513630e-18,    7.0134576730305426e-20,    1.0638817553486355e-21,    1.5097400508988086e-23,
    2.0126494265717963e-25,    2.5298096996095669e-27,    3.0080188236855097e-29,    3.3932514675229087e-31,
    3.6411383565983332e-33,    3.7254432900530105e-35,    3.6423222834535223e-37,    3.4095470282028659e-39,
    3.0613969136812842e-41,    2.6410160325396188e-43,    2.1923960127225389e-45,    1.7538085110445232e-47,
    1.3537391688567439e-49,    1.0095138919607924e-51,    7.2813727499422423e-54,    5.0851627890305545e-56,
    3.4421110303722705e-58,    2.2603961697607844e-60,    1.4413550925513788e-62,    8.9320077941702150e-65,
    5.3834855731310305e-67,    3.1582056645211515e-69,    1.8046315179653653e-71,    1.0050808176254593e-73,
    5.4595410563047038e-76,    2.8941351184533815e-78,    1.4981013128163317e-80,    7.5764149103734699e-83,
    3.7455537941114218e-85,    1.8109955718354884e-87,    8.5679633480324869e-90,    3.9682364985680000e-92,
    1.7999885157484653e-94,    7.9997940702621458e-97,    3.4849939482530995e-99,   1.4887086298571314e-101,
   6.2382905723779306e-104,   2.5652357852544795e-106,   1.0354914544285866e-108,   4.1045874573782931e-111,
   1.5982226335424598e-113,   6.1148507882656321e-116,   2.2995700306389864e-118,   8.5024975601850045e-121
},
// x = 0.4500
{
    6.5936704473266761e-01,    9.7595078083626510e-02,    8.7331908418238086e-03,    5.5962428557474115e-04,
    2.7924177328677427e-05,    1.1407390011929161e-06,    3.9446188406132011e-08,    1.1824472379919323e-09,
    3.1280473067594061e-11,    7.4047766060126088e-13,    1.5860731096377654e-14,    3.1020943697037389e-16,
    5.5820956696500247e-18,    9.3010878705903486e-20,    1.4429472958152602e-21,    2.0941864475618744e-23,
    2.8552082819189448e-25,    3.6704082131330957e-27,    4.4633837098287541e-29,    5.1494060723264553e-31,
    5.6511381249095501e-33,    5.9133629775544531e-35,    5.9127969084712540e-37,    5.6606908320268259e-39,
    5.1981724323458407e-41,    4.5862791691162098e-43,    3.8937401413750630e-45,    3.1855816335517208e-47,
    2.5147814518244415e-49,    1.9179457409582265e-51,    1.4148034587775906e-53,    1.0105239304866079e-55,
    6.9956096338155075e-58,    4.6983379924025884e-60,    3.0640068276348819e-62,    1.9419002910505315e-64,
    1.1970175515507527e-66,    7.1818534864611591e-69,    4.1970473893090699e-71,    2.3906475453171238e-73,
    1.3280975233868706e-75,    7.2003220757267314e-78,    3.8118308385275772e-80,    1.9715850797244214e-82,
    9.9684393695568714e-85,    4.9293300951351680e-87,    2.3851050575599115e-89,    1.1297617801749948e-91,
    5.2410523810456919e-94,    2.3822482924255796e-96,    1.0613770941608352e-98,   4.6369975704128821e-101,
   1.9872488554189212e-103,   8.3574435397672614e-106,   3.4502629815822100e-108,   1.3987326813673913e-110,
   5.5700875979222094e-113,   2.1795666924730076e-115,   8.3828268967431894e-118,   3.1699319779342188e-120
},
// x = 0.4600
{
    6.5378365099509339e-01,    9.8860585972094850e-02,    9.0406990031691621e-03,    5.9211854635582775e-04,
    3.0199384709772018e-05,    1.2610194255060754e-06,    4.4572360713685747e-08,    1.3657531627828645e-09,
    3.6931492505381877e-11,    8.9365714918640094e-13,    1.9566778117487562e-14,    3.9119164892495481e-16,
    7.1956712398196036e-18,    1.2255980432435918e-19,    1.9435946941734873e-21,    2.8834474291494792e-23,
    4.0186148579455115e-25,    5.2807453639488437e-27,    6.5642885400092701e-29,    7.7414698486560325e-31,
    8.6845093133195653e-33,    9.2893895815786660e-35,    9.4948706124428731e-37,    9.2919984107859447e-39,
    8.7223620505626321e-41,    7.8666127357867113e-43,    6.8271305997251837e-45,    5.7095784247821715e-47,
    4.6074400736999979e-49,    3.5920302348014921e-51,    2.7085949808403520e-53,    1.9776020885081557e-55,
    1.3994657918193922e-57,    9.6078328580398993e-60,    6.4049452711298581e-62,    4.1495134511556978e-64,
    2.6146608519673656e-66,    1.6035999009059827e-68,    9.5796142396146116e-71,    5.5778187884816143e-73,
    3.1675504835404107e-75,    1.7554572567704700e-77,    9.4998498042821305e-80,    5.0227718336533056e-82,
    2.5959715522689119e-84,    1.3122165475877058e-86,    6.4903780189166386e-89,    3.1426371951196565e-91,
    1.4902899560442903e-93,    6.9244330610644686e-96,    3.1536380415812300e-98,   1.4083933176536260e-100,
   6.1699878548435485e-103,   2.6524700206931998e-105,   1.1193714414958371e-107,   4.6387583499696414e-110,
   1.8883133597934629e-112,   7.5531346566391697e-115,   2.9695633670953195e-117,   1.1478815760097435e-119
},
// x = 0.4700
{
    6.4826826025689244e-01,    1.0009714345184707e-01,    9.3503233302082503e-03,    6.2561866238690746e-04,
    3.2598571255310283e-05,    1.3907021788385604e-06,    5.0222388875877486e-08,    1.5722737610986090e-09,
    4.3439053581452941e-11,    1.0739507056305258e-12,    2.4025055963596464e-14,    4.9075832089610183e-16,
    9.2232388936176555e-18,    1.6050740580105613e-19,    2.6006880250703478e-21,    3.9421275438882664e-23,
    5.6134718659856518e-25,    7.5368053663596242e-27,    9.5723146252797132e-29,    1.1534273096409907e-30,
    1.3220559183008490e-32,    1.4448729344207701e-34,    1.5089319248412287e-36,    1.5087872172251142e-38,
    1.4470761611381821e-40,    1.3334710232945147e-42,    1.1824224413382952e-44,    1.0103625345836700e-46,
    8.3305123362371090e-49,    6.6357635765675188e-51,    5.1125075873550912e-53,    3.8138872367298557e-55,
    2.7575939804951449e-57,    1.9343391742348231e-59,    1.3175339443920301e-61,    8.7213318864752090e-64,
    5.6148775563129099e-66,    3.5185220195680181e-68,    2.1475912990488026e-70,    1.2776367955592100e-72,
    7.4132045023295338e-75,    4.1977072630715180e-77,    2.3210158621936213e-79,    1.2538463624497649e-81,
    6.6212552568052533e-84,    3.4196799408109695e-86,    1.7281821380235004e-88,    8.5497487874581711e-91,
    4.1425664933166779e-93,    1.9666295354069690e-95,    9.1514480722350734e-98,   4.1758181984628874e-100,
   1.8691390151385323e-102,   8.2100812974105759e-105,   3.5400621656856129e-107,   1.4989188436906675e-109,
   6.2343352335080098e-112,   2.5479038741006527e-114,   1.0235007024354111e-116,   4.0423357183689741e-119
},
// x = 0.4800
{
    6.4281991044259168e-01,    1.0130527613534245e-01,    9.6619345967017770e-03,    6.6012408636921885e-04,
    3.5125003816980308e-05,    1.5302648008384309e-06,    5.6435464432930826e-08,    1.8043057798883752e-09,
    5.0908811419069021e-11,    1.2853754630149905e-12,    2.9366008058999574e-14,    6.1261043375844124e-16,
    1.1758108074254209e-17,    2.0897155770132892e-19,    3.4579535544562875e-21,    5.3530452928231817e-23,
    7.7846950798235555e-25,    1.0674254444865591e-26,    1.3845471090623445e-28,    1.7038125100197585e-30,
    1.9944467129101857e-32,    2.2260940897511856e-34,    2.3742417474684751e-36,    2.4245149949126473e-38,
    2.3748161650792439e-40,    2.2349306091907719e-42,    2.0239281404754265e-44,    1.7662074915502993e-46,
    1.4872307404226405e-48,    1.2098729841434372e-50,    9.5197412998944954e-53,    7.2527281844916480e-55,
    5.3555774920232136e-57,    3.8366404354733885e-59,    2.6688417493949980e-61,    1.8042071807905286e-63,
    1.1862780900963347e-65,    7.5918769017781947e-68,    4.7324193492324201e-70,    2.8752905118898579e-72,
    1.7038174702726929e-74,    9.8530804689030182e-77,    5.5639191547408252e-79,    3.0696571613278452e-81,
    1.6554983410601066e-83,    8.7320612230273392e-86,    4.5067527840547450e-88,    2.2770392127304242e-90,
    1.1267552577804018e-92,    5.4629299195934892e-95,    2.5961864244795296e-97,    1.2098475114402554e-99,
   5.5306180648278603e-102,   2.4809759160776066e-104,   1.0925190486070136e-106,   4.7243199180764851e-109,
   2.0067551827091266e-111,   8.3758782081411366e-114,   3.4362008645328364e-116,   1.3860084234883925e-118
},
// x = 0.4900
{
    6.3743765423326570e-01,    1.0248550039272229e-01,    9.9754069308851771e-03,    6.9563375105376581e-04,
    3.7781915829724445e-05,    1.6801949976030707e-06,    6.3252618226967939e-08,    2.0643099488179910e-09,
    5.9456528458010997e-11,    1.5324308869967318e-12,    3.5738962219362028e-14,    7.6107759550070892e-16,
    1.4911818308362629e-17,    2.7053895159521796e-19,    4.5699551567408680e-21,    7.2217828921674712e-23,
    1.0721025124694683e-24,    1.5006653322730838e-26,    1.9870370297991908e-28,    2.4961589410287902e-30,
    2.9828081709298290e-32,    3.3985923036188919e-34,    3.7002693773816997e-36,    3.8573255406848190e-38,
    3.8569550034093064e-40,    3.7053727551322600e-42,    3.4254398593312739e-44,    3.0515218210604429e-46,
    2.6230507975614885e-48,    2.1783206034473356e-50,    1.7496892081945767e-52,    1.3607896020823885e-54,
    1.0257694579057448e-56,    7.5015064564908230e-59,    5.3268956998810357e-61,    3.6761380120474037e-63,
    2.4674364908495952e-65,    1.6119914902327682e-67,    1.0257722801260891e-69,    6.3621463199598398e-72,
    3.8485683583904901e-74,    2.2719690617483732e-76,    1.3096808202323989e-78,    7.3761345231739084e-81,
    4.0608974990743648e-83,    2.1865750822251208e-85,    1.1520351342760421e-87,    5.9419211408076726e-90,
    3.0015139860308266e-92,    1.4855621603784070e-94,    7.2070164248840720e-97,    3.4285045159209976e-99,
   1.5999345830519132e-101,   7.3266524038813677e-104,   3.2935673736937875e-106,   1.4538892988735787e-108,
   6.3043592413914862e-111,   2.6861573092269840e-113,   1.1249524652387684e-115,   4.6320799721888061e-118
},
// x = 0.5000
{
    6.3212055882855767e-01,    1.0363832351432700e-01,    1.0290617742596098e-02,    7.3214608836806756e-04,
    4.0572505442938585e-05,    1.8409903951734988e-06,    7.0716749121595140e-08,    2.3549180120250378e-09,
    6.9208760843996659e-11,    1.8201433291514065e-12,    4.3314336243215879e-14,    9.4120693633887082e-16,
    1.8817171627822839e-17,    3.4835494772934661e-19,    6.0044504381170173e-21,    9.6822318559579119e-23,
    1.4666874231142941e-24,    2.0948634035633371e-26,    2.8304061995824859e-28,    3.6281587229831937e-30,
    4.4239565559456560e-32,    5.1434710775543698e-34,    5.7142924889719167e-36,    6.0783747964711215e-38,
    6.2018028906178720e-40,    6.0796366560503881e-42,    5.7350144647818782e-44,    5.2132338159637267e-46,
    4.5726722177926251e-48,    3.8748768013552779e-50,    3.1759219339714927e-52,    2.5204191005615433e-54,
    1.9386726394783378e-56,    1.4466923970485925e-58,    1.0482743322267890e-60,    7.3818575624440448e-63,
    5.0558359734947931e-65,    3.3704114164030293e-67,    2.1884889025053452e-69,    1.3850654479856062e-71,
    8.5494688085579001e-74,    5.1500999227376885e-76,    3.0293681333774882e-78,    1.7409599595730481e-80,
    9.7803720384971961e-83,    5.3736720547099477e-85,    2.8889892509510816e-87,    1.5204794093495129e-89,
    7.8373187006260004e-92,    3.9581428069045413e-94,    1.9594295500811218e-96,    9.5115740673742397e-99,
   4.5292202029080311e-101,   2.1164126737213985e-103,   9.7081144236138585e-106,   4.3729373578976219e-108,
   1.9348908114040719e-110,   8.4124124416045797e-113,   3.5949835019231171e-115,   1.5104710448792652e-117
},
// x = 0.5100
{
    6.2686770571266826e-01,    1.0476424387033507e-01,    1.0607447651873827e-02,    7.6965904803462349e-04,
    4.3499933752342081e-05,    2.0131582874106344e-06,    7.8872651328408887e-08,    2.6789398237414011e-09,
    8.0303571308841268e-11,    2.1541134466923224e-12,    5.2286039911637333e-14,    1.1588620954891447e-15,
    2.3631683695541012e-17,    4.4622806066413141e-19,    7.8451897928769880e-21,    1.2903322602805120e-22,
    1.9937009758355215e-24,    2.9045238693969068e-26,    4.0028115136922904e-28,    5.2335946328579869e-30,
    6.5091209500927874e-32,    7.7190827358623260e-34,    8.7472181670523873e-36,    9.4905905140015409e-38,
    9.8769336469129060e-40,    9.8759834012162894e-42,    9.5024569661684470e-44,    8.8106382554560488e-46,
    7.8825930408472260e-48,    6.8132745032793235e-50,    5.6959587702965326e-52,    4.6107214032996748e-54,
    3.6174269105089433e-56,    2.7534088631742413e-58,    2.0350202513464828e-60,    1.4617017606009880e-62,
    1.0211403922637404e-64,    6.9434419553522911e-67,    4.5987064874522345e-69,    2.9686645381918325e-71,
    1.8690868449500915e-73,    1.1484327109873929e-75,    6.8903539236853831e-78,    4.0390373162519029e-80,
    2.3144302037201140e-82,    1.2970583837043911e-84,    7.1126914240694217e-87,    3.8182844672793727e-89,
    2.0074972549036076e-91,    1.0341383525700354e-93,    5.2217561774857972e-96,    2.5854675789077175e-98,
   1.2557694668219803e-100,   5.9853097757889763e-103,   2.8004058410022199e-105,   1.2866462737541269e-107,
   5.8068713826570831e-110,   2.5751714418978164e-112,   1.1224896593112954e-114,   4.8105830728042916e-117
},
// x = 0.5200
{
    6.2167819042424988e-01,    1.0586375106757753e-01,    1.0925780418993886e-02,    8.0817011569295552e-04,
    4.6567323129265264e-05,    2.1972153787496686e-06,    8.7767040329954968e-08,    3.0393705007944778e-09,
    9.2891268575787070e-11,    2.5405665860551703e-12,    6.2874085309676289e-14,    1.4208331643972579e-15,
    2.9541499797603187e-17,    5.6875105095068935e-19,    1.0195229009693381e-20,    1.7097156394315422e-22,
    2.6934669284171776e-24,    4.0008870525610377e-26,    5.6218150109466945e-28,    7.4944861819883394e-30,
    9.5037445544027646e-32,    1.1491294784678375e-33,    1.3277132264122947e-35,    1.4687868796749145e-37,
    1.5585439842957240e-39,    1.5889447317467599e-41,    1.5588200563300464e-43,    1.4736664113553881e-45,
    1.3442893492430824e-47,    1.1847083893269069e-49,    1.0098442986117734e-51,    8.3346724633841033e-54,
    6.6673240237545053e-56,    5.1743369097443122e-58,    3.8992850702376656e-60,    2.8556654433375975e-62,
    2.0340721902551629e-64,    1.4102240246154973e-66,    9.5231675120029421e-69,    6.2681491984961977e-71,
    4.0238351833072860e-73,    2.5208603677529902e-75,    1.5421170155286747e-77,    9.2169292656847184e-80,
    5.3849923731081829e-82,    3.0770401878940838e-84,    1.7204429353794080e-86,    9.4168850142728818e-89,
    5.0480848785001610e-91,    2.6514476296707861e-93,    1.3650662693286314e-95,    6.8914244021295867e-98,
   3.4128137578996947e-100,   1.6585252443257902e-102,   7.9120537010730805e-105,   3.7064677780014188e-107,
   1.7055957005493734e-109,   7.7121038356778132e-112,   3.4275351394869283e-114,   1.4977183222428671e-116
},
// x = 0.5300
{
    6.1655112233013443e-01,    1.0693732610357401e-01,    1.1245502875940266e-02,    8.4767633053682979e-04,
    4.9777755644188438e-05,    2.3936875223093334e-06,    9.7448577390975654e-08,    3.4393976250248229e-09,
    1.0713517329193164e-10,    2.9864062270172256e-12,    7.5327417729234479e-14,    1.7349585003866442e-15,
    3.6765825663112265e-17,    7.2144080587872361e-19,    1.3180835328215847e-20,    2.2528791974339977e-22,
    3.6173809603860863e-24,    5.4765606152847286e-26,    7.8432765088909445e-28,    1.0656939835045072e-29,
    1.3773849898548370e-31,    1.6974606963336565e-33,    1.9989660300148167e-35,    2.2538783051161012e-37,
    2.4375944016023881e-39,    2.5329231389349548e-41,    2.5326791026310193e-43,    2.4403630393360882e-45,
    2.2689199923028868e-47,    2.0380233109769722e-49,    1.7706121531384136e-51,    1.4894604408075674e-53,
    1.2144048036103808e-55,    9.6059021252050646e-58,    7.3780235400375148e-60,    5.5072402415556273e-62,
    3.9982021771351565e-64,    2.8252587889764362e-66,    1.9445688548461381e-68,    1.3045269053846127e-70,
    8.5354367186003027e-73,    5.4501221170999672e-75,    3.3981823620215110e-77,    2.0700819754352394e-79,
    1.2327026932744752e-81,    7.1792389421473916e-84,    4.0912641621549436e-86,    2.2824251701748404e-88,
    1.2470618117343368e-90,    6.6760019545778122e-93,    3.5031539613236352e-95,    1.8025469692077002e-97,
   9.0983429426601731e-100,   4.5065473137268217e-102,   2.1912060790686023e-104,   1.0462281215351100e-106,
   4.9069815825481687e-109,   2.2614312575367148e-111,   1.0243883643040898e-113,   4.5623129573788846e-116
},
// x = 0.5400
{
    6.1148562440283416e-01,    1.0798544151784351e-01,    1.1566504859259500e-02,    8.8817430247801782e-04,
    5.3134271581452283e-05,    2.6031094538138339e-06,    1.0796789265196736e-07,    3.8824084886943485e-09,
    1.2321241045765286e-10,    3.4992705830561368e-12,    8.9926979751800216e-14,    2.1102593750153380e-15,
    4.5561927072861149e-17,    9.1089941991537819e-19,    1.6956077092230831e-20,    2.9527977705625199e-22,
    4.8306315941448831e-24,    7.4512969620197094e-26,    1.0872669098879320e-27,    1.5051720468389290e-29,
    1.9820939315056556e-31,    2.4887662167239201e-33,    2.9861078105120467e-35,    3.4304129721618116e-37,
    3.7800140824784181e-39,    4.0019361658088211e-41,    4.0770365897578580e-43,    4.0025363907007587e-45,
    3.7915478486427698e-47,    3.4699491117069800e-49,    3.0715251847653098e-51,    2.6325484246151294e-53,
    2.1868938099290560e-55,    1.7624599682145592e-57,    1.3792344062066858e-59,    1.0489361712846289e-61,
    7.7588406526616092e-64,    5.5860832225067479e-66,    3.9173251335285273e-68,    2.6775432718077277e-70,
    1.7849514314913349e-72,    1.1612457072499534e-74,    7.3770347889471305e-77,    4.5786767534038855e-79,
    2.7779735198243791e-81,    1.6484110074172304e-83,    9.5711028675969162e-86,    5.4402442206647969e-88,
    3.0284976069482897e-90,    1.6518596139147461e-92,    8.8314771255061255e-95,    4.6299699710971517e-97,
    2.3810656156319023e-99,   1.2016292400726650e-101,   5.9528821249592076e-104,   2.8959293863063250e-106,
   1.3838644065291622e-108,   6.4980050855379067e-111,   2.9990164608328686e-113,   1.3608706669192453e-115
},
// x = 0.5500
{
    6.0648083300174582e-01,    1.0900856154053441e-01,    1.1888679144286311e-02,    9.2966022884769515e-04,
    5.6639868042140091e-05,    2.8260245217669469e-06,    1.1937760680115523e-07,    4.3719973760050077e-09,
    1.4131472829140305e-10,    4.0875924525515575e-12,    1.0698902144015599e-13,    2.5570884728718937e-15,
    5.6230756422310977e-17,    1.1449991304865343e-18,    2.1708198426545458e-20,    3.8503163231916640e-22,
    6.4155140106344693e-24,    1.0079168109801127e-25,    1.4979407610285061e-27,    2.1120810639064233e-29,
    2.8327934940610327e-31,    3.6227742456276047e-33,    4.4272020672273641e-35,    5.1800880521831655e-37,
    5.8136808902264919e-39,    6.2689543592560468e-41,    6.5048437088711725e-43,    6.5042161655081677e-45,
    6.2754336300715934e-47,    5.8494888851803944e-49,    5.2737142353428553e-51,    4.6036961831438478e-53,
    3.8951646899325556e-55,    3.1973140537544504e-57,    2.5484263167866604e-59,    1.9740178573192643e-61,
    1.4871915633893206e-63,    1.0905500252769917e-65,    7.7892557060350129e-68,    5.4226432090351532e-70,
    3.6818760269966918e-72,    2.4396927639029020e-74,    1.5785601594970962e-76,    9.9790134661692676e-79,
    6.1665757594371429e-81,    3.7269180734257117e-83,    2.2040158733143694e-85,    1.2759673050656775e-87,
    7.2346382743884529e-90,    4.0191218987102957e-92,    2.1885670992178090e-94,    1.1686196486414006e-96,
    6.1211762095920303e-99,   3.1463173831167501e-101,   1.5875516489904310e-103,   7.8660572078320550e-106,
   3.8285227962622648e-108,   1.8309914775963613e-110,   8.6070378966037470e-113,   3.9779591564490673e-115
},
// x = 0.5600
{
    6.0153589765800042e-01,    1.1000714223842721e-01,    1.2211921380711708e-02,    9.7212991064633955e-04,
    6.0297497632215839e-05,    3.0629844143000006e-06,    1.3173235132297039e-07,    4.9119728739025700e-09,
    1.6164934343726408e-10,    4.7606624141962799e-12,    1.2686866989032619e-13,    3.0872933090454573e-15,
    6.9123268816390034e-17,    1.4330940280457342e-18,    2.7663892756403382e-20,    4.9958173198759291e-22,
    8.4754499473600203e-24,    1.3557437530594812e-25,    2.0514907382634706e-27,    2.9451527825973452e-29,
    4.0219324031997302e-31,    5.2370168325738357e-33,    6.5162104481922441e-35,    7.7629384792701726e-37,
    8.8708166192013705e-39,    9.7393746897526490e-41,    1.0289553194949285e-42,    1.0475588185042431e-44,
    1.0290846397593331e-46,    9.7667320636013949e-49,    8.9654487043853833e-51,    7.9686775303085293e-53,
    6.8648278828265468e-55,    5.7373774200396879e-57,    4.6561242191998376e-59,    3.6722138274720459e-61,
    2.8168794063863942e-63,    2.1031557549250352e-65,    1.5294889751027332e-67,    1.0841415877670887e-69,
    7.4949502733121000e-72,    5.0566091588341477e-74,    3.3312718292497524e-76,    2.1441800862895779e-78,
    1.3490947844528218e-80,    8.3018141342070184e-83,    4.9987644182008374e-85,    2.9465398062894686e-87,
    1.7010395977070186e-89,    9.6217404099880316e-92,    5.3346655563010571e-94,    2.9003165933567201e-96,
    1.5467923423548822e-98,   8.0951441315218363e-101,   4.1588653387863488e-103,   2.0981138412833434e-105,
   1.0397481400882914e-107,   5.0630033754356904e-110,   2.4232632748036064e-112,   1.1403343529933334e-114
},
// x = 0.5700
{
    5.9664998086289134e-01,    1.1098163165835177e-01,    1.2536130029460201e-02,    1.0155787683527563e-03,
    6.4110067233078393e-05,    3.3145488830982348e-06,    1.4508878732299336e-07,    5.5063652054028439e-09,
    1.8443981239224771e-10,    5.5286954586161886e-12,    1.4996377170807732e-13,    3.7143956870223110e-15,
    8.4647494896396629e-17,    1.7862617417585726e-18,    3.5096602569560530e-20,    6.4511979847467927e-22,
    1.1139845499033092e-23,    1.8137484644745942e-25,    2.7935241890953561e-27,    4.0820137760884445e-29,
    5.6739492958014261e-31,    7.5200191548951373e-33,    9.5238947701990346e-35,    1.1548626327474793e-36,
    1.3432370180611914e-38,    1.5010844926222107e-40,    1.6141939921385809e-42,    1.6727186177588192e-44,
    1.6725570463237308e-46,    1.6157143508886872e-48,    1.5096382491499499e-50,    1.3657548259968434e-52,
    1.1975725870240302e-54,    1.0187588172127242e-56,    8.4152756441950775e-59,    6.7555020688996841e-61,
    5.2745310951223347e-63,    4.0084181317736617e-65,    2.9671120962680129e-67,    2.1407190283585027e-69,
    1.5063591149225673e-71,    1.0344399485507318e-73,    6.9365278519661024e-76,    4.5444310172559325e-78,
    2.9103615441831557e-80,    1.8229037346177356e-82,    1.1172224957740044e-84,    6.7030986465841208e-87,
    3.9388009984181313e-89,    2.2677208309235459e-91,    1.2797628776202647e-93,    7.0819701673602737e-96,
    3.8443869198231499e-98,   2.0478873867886042e-100,   1.0708848313696789e-102,   5.4989958869029917e-105,
   2.7737604359864345e-107,   1.3747871935828382e-109,   6.6975249130200021e-112,   3.2079857679891046e-114
},
// x = 0.5800
{
    5.9182225785982312e-01,    1.1193246996806819e-01,    1.2861206300849245e-02,    1.0600018573026122e-03,
    6.8080436851770617e-05,    3.5812854647933545e-06,    1.5950562293111315e-07,    6.1594335787483536e-09,
    2.0992692900052699e-10,    6.4029011467044490e-12,    1.7671902227771917e-13,    4.4537883732375873e-15,
    1.0327644243548917e-16,    2.2175786044345409e-18,    4.4334987674312746e-20,    8.2922071890266039e-22,
    1.4569939860205736e-23,    2.4138202888579558e-25,    3.7829446835936225e-27,    5.6247346202458306e-29,
    7.9554239383913310e-31,    1.0728706933365851e-32,    1.3825912407853079e-34,    1.7059272729520591e-36,
    2.0189891036275821e-38,    2.2958195634323305e-40,    2.5121164398080773e-42,    2.6488567667323676e-44,
    2.6950583772960519e-46,    2.6491318279862022e-48,    2.5186262043188568e-50,    2.3185447843846450e-52,
    2.0686955622418782e-54,    1.7906807909220026e-56,    1.5051065956350664e-58,    1.2294439063086279e-60,
    9.7675843471107689e-63,    7.5531586064314077e-65,    5.6890775981529693e-67,    4.1765715936449202e-69,
    2.9904818833355831e-71,    2.0896346936510438e-73,    1.4258035219330518e-75,    9.5049438811651208e-78,
    6.1939758283909822e-80,    3.9476518585390882e-82,    2.4618826930892411e-84,    1.5029893047670596e-86,
    8.9866315608214274e-89,    5.2647181407516667e-91,    3.0232057289315800e-93,    1.7023347425734397e-95,
    9.4030913231347277e-98,   5.0968557669765287e-100,   2.7120129871140570e-102,   1.4170498632154256e-104,
   7.2731650053812889e-107,   3.6681132518046985e-109,   1.8183369955958319e-111,   8.8622757933690653e-114
},
// x = 0.5900
{
    5.8705191643971921e-01,    1.1286008959465482e-01,    1.3187054094015764e-02,    1.1053938826466557e-03,
    7.2211418548156979e-05,    3.8637692001939708e-06,    1.7504362928634086e-07,    6.8756735457813251e-09,
    2.3836964783262830e-10,    7.3955573835589462e-12,    2.0763039598428761e-13,    5.3229502198909922e-15,
    1.2555690379131521e-16,    2.7424321234011042e-18,    5.5772720417165372e-20,    1.0611198455180721e-21,
    1.8965822148045517e-23,    3.1962367823707665e-25,    5.0954729647091227e-27,    7.7068585826534108e-29,
    1.1088168534798162e-30,    1.5211262682462709e-32,    1.9940374100763009e-34,    2.5027751011037899e-36,
    3.0131258684070955e-38,    3.4853242671030879e-40,    3.8794260612186998e-42,    4.1611041510124183e-44,
    4.3066620712398878e-46,    4.3062456032381890e-48,    4.1646800170017134e-50,    3.8999243980496564e-52,
    3.5396484699476862e-54,    3.1167692084958206e-56,    2.6648741853396932e-58,    2.2143242058660478e-60,
    1.7895455173894563e-62,    1.4076909604008864e-64,    1.0785586290632147e-66,    8.0546199741956068e-69,
    5.8666416683659515e-71,    4.1700576099397640e-73,    2.8943743374449353e-75,    1.9627633265366013e-77,
    1.3011017909950979e-79,    8.4353679588796168e-82,    5.3512591631655120e-84,    3.3232880516719174e-86,
    2.0213081242269419e-88,    1.2045760389418047e-90,    7.0363968658633852e-93,    4.0304277055576860e-95,
    2.2646463556134056e-97,    1.2486929653977299e-99,   6.7587858199722432e-102,   3.5924080824362260e-104,
   1.8756318544649505e-106,   9.6225579624697879e-109,   4.8522797110367747e-111,   2.4056939686510659e-113
},
// x = 0.6000
{
    5.8233815673983158e-01,    1.1376491536044932e-01,    1.3513579937585585e-02,    1.1517492138985374e-03,
    7.6505775436455227e-05,    4.1625823517098303e-06,    1.9176565510831518e-07,    7.6598243630030409e-09,
    2.7004603323910115e-10,    8.5200878951768915e-12,    2.4324989183294349e-13,    6.3416810238618054e-15,
    1.5211925157439512e-16,    3.3788749286758909e-18,    6.9879783980060282e-20,    1.3520363063094180e-21,
    2.4574820740358918e-23,    4.2116558967709666e-25,    6.8280091949487388e-27,    1.0502265525776208e-28,
    1.5366031942032773e-30,    2.1436987204781181e-32,    2.8577786062513492e-34,    3.6476578960721691e-36,
    4.4658766148334182e-38,    5.2532729154361655e-40,    5.9463671267719669e-42,    6.4862012092605792e-44,
    6.8268494976956746e-46,    6.9418644969185322e-48,    6.8274239244806399e-50,    6.5017369654750675e-52,
    6.0011073182051118e-54,    5.3737075286170502e-56,    4.6724458273583761e-58,    3.9482715492032545e-60,
    3.2449413452681328e-62,    2.5957912702301592e-64,    2.0225748043597609e-66,    1.5360463513219870e-68,
    1.1377511914402684e-70,    8.2242877624350230e-73,    5.8050970090794684e-75,    4.0033290577397889e-77,
    2.6987535677613730e-79,    1.7793222708483550e-81,    1.1479030803474365e-83,    7.2496309828431045e-86,
    4.4841397277832140e-88,    2.7175626024284775e-90,    1.6143377622568013e-92,    9.4035962862674650e-95,
    5.3733114219341505e-97,    3.0129788270036556e-99,   1.6584711096272416e-101,   8.9644514166957868e-104,
   4.7597538522245576e-106,   2.4832833952960357e-108,   1.2734457382323622e-110,   6.4205742929947365e-113
},
// x = 0.6100
{
    5.7768019104588997e-01,    1.1464736461658549e-01,    1.3840692931534679e-02,    1.1990618990819811e-03,
    8.0966220758581036e-05,    4.4783141193107798e-06,    2.0973663986205834e-07,    8.5168763488808616e-09,
    3.0525423384038052e-10,    9.7911434932121852e-12,    2.8419060918118192e-13,    7.5323574665734206e-15,
    1.8368831038081133e-16,    4.1480247368948761e-18,    8.7215469070933332e-20,    1.7155515554200785e-21,
    3.1701500404781425e-23,    5.5235319419740556e-25,    9.1040164056996186e-27,    1.4236303201901410e-28,
    2.1176372708459777e-30,    3.0035126087798456e-32,    4.0707121428096291e-34,    5.2824113339954571e-36,
    6.5750841968558837e-38,    7.8632340630561644e-40,    9.0489823515541560e-42,    1.0034953123878169e-43,
    1.0737972377739353e-45,    1.1100823840203039e-47,    1.1099749200222934e-49,    1.0746399801006426e-51,
    1.0084221343266518e-53,    9.1804178801080690e-56,    8.1154052486669303e-58,    6.9718899302259361e-60,
    5.8254299411785328e-62,    4.7377111063614905e-64,    3.7530220948925433e-66,    2.8977346434559856e-68,
    2.1821238397081549e-70,    1.6036449925737116e-72,    1.1507932097378775e-74,    8.0683922569848004e-77,
    5.5297676111194182e-79,    3.7066043351637608e-81,    2.4311112102227533e-83,    1.5609662743181816e-85,
    9.8160055093553582e-88,    6.0480207575862334e-90,    3.6526343671465696e-92,    2.1631362223986914e-94,
    1.2566375205778557e-96,    7.1637797464420610e-99,   4.0089644575366357e-101,   2.2030592578268281e-103,
   1.1892280257132658e-105,   6.3079046515467185e-108,   3.2886487244852946e-110,   1.6857343558131811e-112
},
// x = 0.6200
{
    5.7307724359753975e-01,    1.1550784737417029e-01,    1.4168304690264989e-02,    1.2473256784867855e-03,
    8.5595417026832537e-05,    4.8115603553464815e-06,    2.2902362552396629e-07,    9.4520782310601929e-09,
    3.4431348218743603e-10,    1.1224687211142279e-11,    3.3113216855976291e-13,    8.9202115374000260e-15,
    2.2109539814907670e-16,    5.0745152598383644e-18,    1.0844328514780051e-19,    2.1680513122109301e-21,
    4.0719537254079784e-23,    7.2110352603618873e-25,    1.2080139133669092e-26,    1.9199709142038148e-28,
    2.9027378709068153e-30,    4.1845118479186032e-32,    5.7642799552457954e-34,    7.6026749827090220e-36,
    9.6182319218189345e-38,    1.1691090134120647e-39,    1.3674566551748105e-41,    1.5413076262976626e-43,
    1.6763184623817789e-45,    1.7613668860817760e-47,    1.7900627170199276e-49,    1.7614836862069602e-51,
    1.6800361608021229e-53,    1.5545310796117242e-55,    1.3967154429165328e-57,    1.2195765622843959e-59,
    1.0357320375752961e-61,    8.5614816706897788e-64,    6.8932249920284021e-66,    5.4095478353900143e-68,
    4.1404048305830082e-70,    3.0926640253594573e-72,    2.2557082760743969e-74,    1.6074364455809398e-76,
    1.1197340436782329e-78,    7.6286107347390259e-81,    5.0855192343587190e-83,    3.3188320094923957e-85,
    2.1212303972346579e-87,    1.3283962527083655e-89,    8.1542103557528766e-92,    4.9081851848464720e-94,
    2.8980673374189067e-96,    1.6791986283829846e-98,   9.5511016013200971e-101,   5.3346860580550068e-103,
   2.9269093129768294e-105,   1.5779408110465828e-107,   8.3615067845730562e-110,   4.3562990054710711e-112
},
// x = 0.6300
{
    5.6852855039700756e-01,    1.1634676643313914e-01,    1.4496329286821033e-02,    1.2965339980429719e-03,
    9.0395975233510366e-05,    5.1629232785388867e-06,    2.4969576695837211e-07,    1.0470944477240968e-08,
    3.8756511930141998e-10,    1.2838083393127699e-11,    3.8482649280678240e-13,    1.0533632901617079e-14,
    2.6529163663504761e-16,    6.1870034009040574e-18,    1.3434802487367104e-19,    2.7292400525357019e-21,
    5.2085780644450595e-23,    9.3725687400303046e-25,    1.5954310948908782e-26,    2.5765954330854742e-28,
    3.9582679034692681e-30,    5.7981336737891097e-32,    8.1158643860386020e-34,    1.0876837615408585e-35,
    1.3982285115644675e-37,    1.7269699071625328e-39,    2.0525338527423378e-41,    2.3507866501010213e-43,
    2.5979313033539609e-45,    2.7737563791496373e-47,    2.8644034080930496e-49,    2.8641258334996099e-51,
    2.7757459343763026e-53,    2.6098057167966854e-55,    2.3826730681155065e-57,    2.1140409875000980e-59,
    1.8243140904898893e-61,    1.5323185308084016e-63,    1.2536333703118129e-65,    9.9967093963529835e-68,
    7.7747594533816153e-70,    5.9009914805146316e-72,    4.3734413218130941e-74,    3.1668124299551541e-76,
    2.2415663693000894e-78,    1.5517808677264467e-80,    1.0511591744464524e-82,    6.9705448351073915e-85,
    4.5270739781546692e-87,    2.8807509147493945e-89,    1.7968354847616512e-91,    1.0989949658811531e-93,
    6.5937368655933020e-96,    3.8821615602030908e-98,   2.2437445471912770e-100,   1.2734363405889508e-102,
   7.0994725031279637e-105,   3.8891615203530028e-107,   2.0941042010369892e-109,   1.1086128431002897e-111
},
// x = 0.6400
{
    5.6403335902094198e-01,    1.1716451750883602e-01,    1.4824683198273592e-02,    1.3466800223221596e-03,
    9.5370454125135273e-05,    5.5330111874455252e-06,    2.7182434091529557e-07,    1.1579262603584882e-08,
    4.3537364377487817e-10,    1.4650190814683034e-11,    4.4610396397562241e-13,    1.2404496732910366e-14,
    3.1736263665633334e-16,    7.5187385223453576e-18,    1.6585524488853547e-19,    3.4227383335960829e-21,
    6.6356854975221584e-23,    1.2129989362190491e-24,    2.0975650742370651e-26,    3.4412767574575917e-28,
    5.3705016134529299e-30,    7.9916133930734572e-32,    1.1363649816998654e-33,    1.5471154962820062e-35,
    2.0203911776730224e-37,    2.5350087609843489e-39,    3.0607126362004757e-41,    3.5610913082845573e-43,
    3.9979314343427895e-45,    4.3362457301166072e-47,    4.5490189154357376e-49,    4.6207634191682803e-51,
    4.5492469196148086e-53,    4.3451643449025561e-55,    4.0299604494520664e-57,    3.6323533704786134e-59,
    3.1842907735693086e-61,    2.7170687611624682e-63,    2.2581908204301093e-65,    1.8293033249121043e-67,
    1.4452874173096658e-69,    1.1143737955970569e-71,    8.3901144732563397e-74,    6.1717117699034692e-76,
    4.4378604381097698e-78,    3.1209815707005164e-80,    2.1476726997418621e-82,    1.4467888812312731e-84,
    9.5454164190770471e-87,    6.1705214908308291e-89,    3.9098794810699323e-91,    2.4293476734403826e-93,
    1.4806912681686303e-95,    8.8561601223573952e-98,   5.1997712003049622e-100,   2.9979683800892880e-102,
   1.6979108760548306e-104,   9.4489554976341298e-107,   5.1685073697896417e-109,   2.7796223744467434e-111
},
// x = 0.6500
{
    5.5959092843537495e-01,    1.1796148935635591e-01,    1.5153285252194225e-02,    1.3977566471750629e-03,
    1.0052135953898894e-04,    5.9224381736784562e-06,    2.9548275366119125e-07,    1.2783100454630958e-08,
    4.8812778509219508e-10,    1.6681459912016896e-11,    5.1587997170130783e-13,    1.4568518590011285e-14,
    3.7854467013947823e-16,    9.1082000313577577e-18,    2.0405345230971660e-19,    4.2767744627126739e-21,
    8.4208703419818311e-23,    1.5633660142039377e-24,    2.7456501144789684e-26,    4.5748750052515830e-28,
    7.2511132801828254e-30,    1.0958592901078195e-31,    1.5825917772470270e-33,    2.1882893680133792e-35,
    2.9023422196837283e-37,    3.6984855949196913e-39,    4.5352213135506881e-41,    5.3590848605231240e-43,
    6.1104695419946644e-45,    6.7310831241953736e-47,    7.1716772501956243e-49,    7.3985862716920179e-51,
    7.3978686335985283e-53,    7.1763809349315086e-55,    6.7597759220568387e-57,    6.1880228655647797e-59,
    5.5094582454417535e-61,    4.7745145326641680e-63,    4.0301544463895200e-65,    3.3157309504518419e-67,
    2.6606045574257258e-69,    2.0834812132178068e-71,    1.5931593221984669e-73,    1.1902265058052400e-75,
    8.6922120684163500e-78,    6.2084129770199495e-80,    4.3390058842282536e-82,    2.9686573952781363e-84,
    1.9892189866814898e-86,    1.3059976883636993e-88,    8.4045942933357224e-91,    5.3036629994137383e-93,
    3.2830964933599887e-95,    1.9943321683560267e-97,    1.1892391237360565e-99,   6.9637783255778440e-102,
   4.0055829686824552e-104,   2.2639540680238331e-106,   1.2577140952057942e-108,   6.8696653404208150e-111
},
// x = 0.6600
{
    5.5520052881374971e-01,    1.1873806389269002e-01,    1.5482056574249626e-02,    1.4497565120138089e-03,
    1.0585114379976837e-04,    6.3318238351507611e-06,    3.2074654725563754e-07,    1.4088813448812723e-08,
    5.4624160080282372e-10,    1.8954034194546090e-11,    5.9516186892074103e-13,    1.7065637977055084e-14,
    4.5024244760959472e-16,    1.0999810024985328e-17,    2.5021931474023409e-19,    5.3249833689830337e-21,
    1.0645953042500668e-22,    2.0068477326991519e-24,    3.5787024294110485e-26,    6.0546166568521402e-28,
    9.7440490751363347e-30,    1.4952616916884299e-31,    2.1925989844376025e-33,    3.0783866279276979e-35,
    4.1456757999829050e-37,    5.3641278887534971e-39,    6.6788594611068880e-41,    8.0135180402716616e-43,
    9.2776088050681284e-45,    1.0377086219144554e-46,    1.1226394692104603e-48,    1.1759734115072124e-50,
    1.1939458993907480e-52,    1.1760150117299856e-54,    1.1247839212618336e-56,    1.0454859261427620e-58,
    9.4515865832791115e-61,    8.3167689162032524e-63,    7.1281486633020696e-65,    5.9547568416902805e-67,
    4.8517097582153583e-69,    3.8577478961840738e-71,    2.9952513538596023e-73,    2.2721318307081070e-75,
    1.6848610800739060e-77,    1.2219242662795357e-79,    8.6712930910007613e-82,    6.0239820337527177e-84,
    4.0986059912049979e-86,    2.7322849688840851e-88,    1.7853787886951213e-90,    1.1439830518686091e-92,
    7.1904718353719723e-95,    4.4350786704061627e-97,    2.6853638015988375e-99,   1.5966473535875730e-101,
   9.3252383374356721e-104,   5.3517026616818398e-106,   3.0188147488992008e-108,   1.6742499689399812e-110
},
// x = 0.6700
{
    5.5086144135796566e-01,    1.1949461631671240e-01,    1.5810920536865385e-02,    1.5026720117475747e-03,
    1.1136220517420412e-04,    6.7617929896099849e-06,    3.4769340448796565e-07,    1.5503051783789427e-08,
    6.1015559715749633e-10,    2.1491855912648736e-11,    6.8505635104008988e-13,    1.9940432287697521e-14,
    5.3404862659310916e-16,    1.3244728253154997e-17,    3.0584624208670485e-19,    6.6073270140415641e-21,
    1.3409666090119189e-22,    2.5661039828575161e-24,    4.6452841470920026e-26,    7.9781207536663755e-28,
    1.3034098832171032e-29,    2.0304235318606662e-31,    3.0224336622791900e-33,    4.3077361345000272e-35,
    5.8891121200697525e-37,    7.7353922513383628e-39,    9.7771979726572414e-41,    1.1908699826351496e-42,
    1.3996077418340166e-44,    1.5891868680750621e-46,    1.7452964851467174e-48,    1.8559055342313756e-50,
    1.9128131721454074e-52,    1.9126274702606864e-54,    1.8570188472049253e-56,    1.7522463173431982e-58,
    1.6080942332880583e-60,    1.4364525322198827e-62,    1.2498080311634103e-64,    1.0598894253047604e-66,
    8.7664013020996377e-69,    7.0760424057857348e-71,    5.5772478298722596e-73,    4.2948729107487775e-75,
    3.2330397459274948e-77,    2.3802436747929193e-79,    1.7147120875262189e-81,    1.2092643563143200e-83,
    8.3522602929285420e-86,    5.6522860486567155e-88,    3.7493745071055010e-90,    2.4388122901885617e-92,
    1.5561323353307526e-94,    9.7436312008427568e-97,    5.9889823028946076e-99,   3.6148419332474031e-101,
   2.1432393249430368e-103,   1.2486282108580595e-105,   7.1500335920197425e-108,   4.0255236702152651e-110
},
// x = 0.6800
{
    5.4657295812238538e-01,    1.2023151522704779e-01,    1.6139802708939784e-02,    1.5564953083798516e-03,
    1.1705688738155331e-04,    7.2129753887058032e-06,    3.7640315248882436e-07,    1.7032767595926665e-08,
    6.8033787279505856e-10,    2.4320776050197764e-11,    7.8677727482698480e-13,    2.3242562893808584e-14,
    6.3176518345957010e-16,    1.5901737206742211e-17,    3.7267672127673106e-19,    8.1711522933817978e-21,
    1.6830790197284904e-22,    3.2688152876473068e-24,    6.0056285120095713e-26,    1.0468325832747408e-27,
    1.7357549926809869e-29,    2.7442592296887801e-31,    4.1459743661031281e-33,    5.9972270877067918e-35,
    8.3211430578721789e-37,    1.1092960047820632e-38,    1.4230220067013850e-40,    1.7591132363080572e-42,
    2.0983027575082861e-44,    2.4180722022954777e-46,    2.6952316367997082e-48,    2.9088104868052709e-50,
    3.0427402423642117e-52,    3.0878454536399390e-54,    3.0428068956456181e-56,    2.9139776423891787e-58,
    2.7141609221274689e-60,    2.4606422292242244e-62,    2.1728695369245226e-64,    1.8701829498756514e-66,
    1.5699221604438872e-68,    1.2861169984474721e-70,    1.0288292712057904e-72,    8.0409440233908555e-75,
    6.1432919019333217e-77,    4.5903409577272689e-79,    3.3562026889259101e-81,    2.4022140212558722e-83,
    1.6839452432385157e-85,    1.1565957740219782e-87,    7.7866353577083212e-90,    5.1404715300806806e-92,
    3.3289302729445649e-94,    2.1154979862967123e-96,    1.3197097767800286e-98,   8.0844105197608989e-101,
   4.8647819058621070e-103,   2.8764678415815416e-105,   1.6717377567594301e-107,   9.5524804289493008e-110
},
// x = 0.6900
{
    5.4233438184075611e-01,    1.2094912273785821e-01,    1.6468630806589717e-02,    1.6112183422754853e-03,
    1.2293747915793717e-04,    7.6860054328276469e-06,    4.0695776503260694e-07,    1.8685222068387815e-08,
    7.5728528504579465e-10,    2.7468668708815788e-11,    9.0165393347592682e-13,    2.7027255200622586e-14,
    7.4542678850747049e-16,    1.9038225714231871e-17,    4.5273882013660173e-19,    1.0072404142804188e-20,
    2.1051806858766872e-22,    4.1486886113330512e-24,    7.7341925847332166e-26,    1.3679499543690738e-27,
    2.3015382993251791e-29,    3.6922578454700549e-31,    5.6601881485394082e-33,    8.3079207479094617e-35,
    1.1696680793400647e-36,    1.5822134655235947e-38,    2.0595264871247473e-40,    2.5833773212104678e-42,
    3.1268036278089966e-44,    3.6562867260195501e-46,    4.1352874880259069e-48,    4.5285979374785941e-50,
    4.8067554544949394e-52,    4.9497309930564124e-54,    4.9492500537747653e-56,    4.8093928165201910e-58,
    4.5454692065727206e-60,    4.1814869687214376e-62,    3.7467535377022680e-64,    3.2722374933986817e-66,
    2.7872641971962838e-68,    2.3169669000178236e-70,    1.8807108788921574e-72,    1.4915065791983985e-74,
    1.1562685939344386e-76,    8.7668181514296610e-79,    6.5040629947029846e-81,    4.7237596042419162e-83,
    3.3600335541772553e-85,    2.3417300722465576e-87,    1.5997225828184409e-89,    1.0716116449222293e-91,
    7.0417209895941637e-94,    4.5407375291659843e-96,    2.8742987299938867e-98,   1.7866584230925189e-100,
   1.0909284505256644e-102,   6.5453391067663614e-105,   3.8599397938003718e-107,   2.2380423529799823e-109
},
// x = 0.7000
{
    5.3814502575599532e-01,    1.2164779459258265e-01,    1.6797334644927133e-02,    1.6668328431054260e-03,
    1.2900621387254929e-04,    8.1815218869364766e-06,    4.3944136354752595e-07,    2.0467992482426056e-08,
    8.4152463839615055e-10,    3.0965549948115213e-11,    1.0311398044521234e-12,    3.1355814551494175e-14,
    8.7732633160079941e-16,    2.2731280037055782e-17,    5.4838731436203311e-19,    1.2377013485836151e-20,
    2.6243141786106880e-22,    5.2466438143285007e-24,    9.9227144643920536e-26,    1.7804545784149426e-27,
    3.0389560802318546e-29,    4.9458856485696261e-31,    7.6918182479611625e-33,    1.1453463088844529e-34,
    1.6358883084061851e-36,    2.2449300012336966e-38,    2.9645036449184874e-40,    3.7724150969956538e-42,
    4.6321156475960468e-44,    5.4949823887443988e-46,    6.3049136862460173e-48,    7.0046215858209969e-50,
    7.5425900711863758e-52,    7.8794829079177321e-54,    7.9928789368994157e-56,    7.8795583116867212e-58,
    7.5550647431391263e-60,    7.0507955591258434e-62,    6.4092979001225242e-64,    5.6786899106757213e-66,
    4.9071521706742229e-68,    4.1382746697756317e-70,    3.4077651169281167e-72,    2.7417064863171754e-74,
    2.1562679105731324e-76,    1.6585715990566008e-78,    1.2483179955212856e-80,    9.1976435464459659e-83,
    6.6371360761794750e-85,    4.6926945441103213e-87,    3.2522093773560740e-89,    2.2101392512771374e-91,
    1.4733619114426323e-93,    9.6384113189386804e-96,    6.1895547718644270e-98,   3.9031703551052053e-100,
   2.4178019760500497e-102,   1.4716519583624655e-104,   8.8044454571659604e-107,   5.1789093298261970e-109
},
// x = 0.7100
{
    5.3400421345279125e-01,    1.2232788027567174e-01,    1.7125846090798182e-02,    1.7233303404769614e-03,
    1.3526526919381600e-04,    8.7001675976041075e-06,    4.7394021685094085e-07,    2.2388979206598260e-08,
    9.3361389463958389e-10,    3.4843701143443810e-11,    1.1768217868617642e-12,    3.6296179926823843e-14,
    1.0300427533864179e-15,    2.7068892090843416e-17,    6.6234993180888165e-19,    1.5162481748238165e-20,
    3.2608082946921451e-22,    6.6122095281780798e-24,    1.2683864353460059e-25,    2.3083862341076119e-27,
    3.9963069534626855e-29,    6.5968351648335125e-31,    1.0405860283024976e-32,    1.5716051697997053e-34,
    2.2767646325577256e-36,    3.1690225986651687e-38,    4.2445605175915152e-40,    5.4784609685374909e-42,
    6.8230252726109281e-44,    8.2096094843723116e-46,    9.5541940013398098e-48,    1.0766099817663543e-49,
    1.1758532721581162e-51,    1.2459176244521701e-53,    1.2818992860321539e-55,    1.2817746317565974e-57,
    1.2465427555694918e-59,    1.1799574852666237e-61,    1.0879225967904564e-63,    9.7767620636274462e-66,
    8.5691119232853084e-68,    7.3296804843279016e-70,    6.1220216959363081e-72,    4.9958074217785345e-74,
    3.9851713726225646e-76,    3.1091243475433582e-78,    2.3734971488073348e-80,    1.7737797882125029e-82,
    1.2982655116830567e-84,    9.3103237059721894e-87,    6.5445639047266955e-89,    4.5110910745031144e-91,
    3.0502199682748168e-93,    2.0238902627027280e-95,    1.3182589183192697e-97,   8.4317677371848085e-100,
   5.2976308686755635e-102,   3.2705887017474931e-104,   1.9846443666496009e-106,   1.1840740941976332e-108
},
// x = 0.7200
{
    5.2991127869297094e-01,    1.2298972312234699e-01,    1.7454099016524761e-02,    1.7807021742570481e-03,
    1.4171676680364690e-04,    9.2425892114630324e-06,    5.1054273962829839e-07,    2.4456412618755814e-08,
    1.0341434042189358e-09,    3.9137796919845858e-11,    1.3404299452252331e-12,    4.1923517443205247e-14,
    1.2064713450657660e-15,    3.2151295088383844e-17,    7.9777925137111269e-19,    1.8519685936250129e-20,
    4.0388467145490347e-22,    8.3051612336980065e-24,    1.6155592791725276e-25,    2.9816046169619701e-27,
    5.2344498479389359e-29,    8.7623133007498416e-31,    1.4016266577779526e-32,    2.1466896376430725e-34,
    3.1536653872419121e-36,    4.4513799232711929e-38,    6.0460825814959203e-40,    7.9135633669528157e-42,
    9.9945396252041236e-44,    1.2194969995558321e-45,    1.4392121771582601e-47,    1.6446057452855332e-49,
    1.8215003341782604e-51,    1.9572137460380875e-53,    2.0420939281723430e-55,    2.0706487292670162e-57,
    2.0420903403508882e-59,    1.9602307791957225e-61,    1.8327868867832442e-63,    1.6702527475309483e-65,
    1.4845546456430276e-67,    1.2877118254166992e-69,    1.0906912099982312e-71,    9.0258057675875965e-74,
    7.3013081396365157e-76,    5.7765053690546389e-78,    4.4718708134582427e-80,    3.3890167107716318e-82,
    2.5154230122555089e-84,    1.8293037185044293e-86,    1.3039931191767123e-88,    9.1148548208068220e-91,
    6.2498977958589935e-93,    4.2053517912143241e-95,    2.7777280436958499e-97,    1.8016950810839365e-99,
   1.1479369144905481e-101,   7.1868064046786747e-104,   4.4224854371450378e-106,   2.6756931798995845e-108
},
// x = 0.7300
{
    5.2586556525358985e-01,    1.2363366042642408e-01,    1.7782029254586496e-02,    1.8389395045961567e-03,
    1.4836277215796181e-04,    9.8094368952590508e-06,    5.4933948967472068e-07,    2.6678859955801714e-08,
    1.1437371582471213e-09,    4.3885037718076234e-11,    1.5238477766437993e-12,    4.8320855720340819e-14,
    1.4098566878560082e-15,    3.8092437600813285e-17,    9.5831083962603740e-19,    2.2554930737826196e-20,
    4.9871240328121049e-22,    1.0397438812557469e-23,    2.0506295311285578e-25,    3.8370794401191682e-27,
    6.8298092790642043e-29,    1.1591600968077444e-30,    1.8799417035109895e-32,    2.9192341056035613e-34,
    4.3481360184585484e-36,    6.2225759713937010e-38,    8.5691515609798754e-40,    1.1371668466691216e-41,
    1.4561408700563359e-43,    1.8013991280483426e-45,    2.1554738712749485e-47,    2.4972876379542639e-49,
    2.8043030072069740e-51,    3.0550824942457594e-53,    3.2318372114036287e-55,    3.3225332999957495e-57,
    3.3222099680590285e-59,    3.2333193671843215e-61,    3.0650862308241835e-63,    2.8320590401953017e-65,
    2.5521472241563444e-67,    2.2444900812624333e-69,    1.9274820042759273e-71,    1.6172011922979958e-73,
    1.3263811256248898e-75,    1.0639530326705363e-77,    8.3509556138970812e-80,    6.4166790497530843e-82,
    4.8287791703919437e-84,    3.5604251995093669e-86,    2.5732448198124283e-88,    1.8236661819921989e-90,
    1.2678220699623871e-92,    8.6492320272788448e-95,    5.7923498227086490e-97,    3.8092213853690815e-99,
   2.4607235292448545e-101,   1.5619626403006822e-103,   9.7452058848587843e-106,   5.9779302768226910e-108
},
// x = 0.7400
{
    5.2186642676769412e-01,    1.2426002354623279e-01,    1.8109574553237261e-02,    1.8980333216599044e-03,
    1.5520529429173776e-04,    1.0401364057689632e-05,    5.9042316391895637e-07,    2.9065232086341874e-08,
    1.2630540606752507e-09,    4.9125287045580659e-11,    1.7291230184499333e-12,    5.5579765244709865e-14,
    1.6438284116535047e-15,    4.5021607767154108e-17,    1.1481282573407584e-18,    2.7392276773041741e-20,
    6.1396009171263738e-22,    1.2975385856548525e-23,    2.5940930831524420e-25,    4.9204407863184094e-27,
    8.8780387652390562e-29,    1.5274163886617757e-30,    2.5111013935958163e-32,    3.9527093459154500e-34,
    5.9680902789429921e-36,    8.6578283157735497e-38,    1.2086018558176723e-39,    1.6258349172446742e-41,
    2.1103894663106277e-43,    2.6465265405505932e-45,    3.2100830407438112e-47,    3.7700696447782556e-49,
    4.2915397313330907e-51,    4.7393482351554531e-53,    5.0822112477243535e-55,    5.2963930436818818e-57,
    5.3684094849502431e-59,    5.2963285258504141e-61,    5.0895199374429910e-63,    4.7669905162704608e-65,
    4.3546737046966365e-67,    3.8821788630468086e-69,    3.3795290089346985e-71,    2.8743386505756547e-73,
    2.3897379845808020e-75,    1.9431776901484924e-77,    1.5460885765642168e-79,    1.2042500697601486e-81,
    9.1865385647798843e-84,    6.8663295758566038e-86,    5.0305113389057144e-88,    3.6139707988291057e-90,
    2.5468648141907123e-92,    1.7613004498657911e-94,    1.1956911449962461e-96,    7.9709263213944174e-99,
   5.2196787083743863e-101,   3.3586130920034625e-103,   2.1241674738423667e-105,   1.3208606350518679e-107
},
// x = 0.7500
{
    5.1791322656771344e-01,    1.2486913800866864e-01,    1.8436674533039314e-02,    1.9579744550755804e-03,
    1.6224628566686501e-04,    1.1019027073201026e-05,    6.3388859324997308e-07,    3.1624790201495625e-08,
    1.3927892200602902e-09,    5.4901213462372683e-11,    1.9584790135161564e-12,    6.3801083920276692e-14,
    1.9124399610038148e-15,    5.3085220149550976e-17,    1.3720356199796648e-18,    3.3176177003918606e-20,
    7.5363715102971800e-22,    1.6142358611014957e-23,    3.2708251568678225e-25,    6.2878371336921051e-27,
    1.1498473479285308e-28,    2.0049650161391062e-30,    3.3407200877555525e-32,    5.3296348577524264e-34,
    8.1557689736558775e-36,    1.1991282972515511e-37,    1.6965523453300375e-39,    2.3130655165824915e-41,
    3.0429983621002820e-43,    3.8676138627822857e-45,    4.7545671155125205e-47,    5.6594216543387022e-49,
    6.5292586800947710e-51,    7.3079825656898888e-53,    7.9425474513417129e-55,    8.3891045551808091e-57,
    8.6180577054826420e-59,    8.6172184436242606e-61,    8.3926185838290780e-63,    7.9669755974181345e-65,
    7.3762121531090816e-67,    6.6647206032386742e-69,    5.8801885828675523e-71,    5.0687598877954270e-73,
    4.2711302485619186e-75,    3.5199283528374986e-77,    2.8384711908740694e-79,    2.2407615365237356e-81,
    1.7324461066925800e-83,    1.3123853466697303e-85,    9.7449088536073039e-88,    7.0954383946247408e-90,
    5.0679165599709252e-92,    3.5521066544331133e-94,    2.4439964634921152e-96,    1.6512749123941474e-98,
   1.0959314877804785e-100,   7.1470804892136343e-103,   4.5812767721546690e-105,   2.8872465235440783e-107
},
// x = 0.7600
{
    5.1400533753143762e-01,    1.2546132361140749e-01,    1.8763270644302495e-02,    2.0187535831005070e-03,
    1.6948764206115285e-04,    1.1663085007908109e-05,    6.7983273616704018e-07,    3.4367152419271364e-08,
    1.5336752603683075e-09,    6.1258437348713587e-11,    2.2143266504675469e-12,    7.3095691057639982e-14,
    2.2202105652848503e-15,    6.2448778533082277e-17,    1.6355384516640321e-18,    4.0074456428415126e-20,
    9.2246576815205241e-22,    2.0021758654940278e-23,    4.1109325348740289e-25,    8.0081562114553578e-27,
    1.4839526798322888e-28,    2.6220176232712120e-30,    4.4270877094470307e-32,    7.1569004599702816e-34,
    1.1097934127774171e-35,    1.6534565369358038e-37,    2.3705246523259719e-39,    3.2750307690242165e-41,
    4.3659579528940952e-43,    5.6230435364491827e-45,    7.0047065065213188e-47,    8.4489305706947246e-49,
    9.8774397174363859e-51,    1.1202860451823274e-52,    1.2337928027109862e-54,    1.3205324735355179e-56,
    1.3746559227926640e-58,    1.3928453729994120e-60,    1.3746259587687504e-62,    1.3223056265131363e-64,
    1.2405752483173051e-66,    1.1358554337727825e-68,    1.0155090755662380e-70,    8.8704521068723408e-73,
    7.5742279535261846e-75,    6.3252971668085619e-77,    5.1687221593146690e-79,    4.1347185771727158e-81,
    3.2393784877334478e-83,    2.4866519713372284e-85,    1.8710401784612754e-87,    1.3804996105904313e-89,
    9.9916740033907406e-92,    7.0965385131705013e-94,    4.9478090068778502e-96,    3.3875330621805879e-98,
   2.2782395620448116e-100,   1.5055545606437211e-102,   9.7792649707488819e-105,   6.2453304749017904e-107
},
// x = 0.7700
{
    5.1014214193053375e-01,    1.2603689452331843e-01,    1.9089306125397356e-02,    2.0803612415190250e-03,
    1.7693120249687226e-04,    1.2334199347791835e-05,    7.2835467127463136e-07,    3.7302300298052407e-08,
    1.6864836502339162e-09,    6.8245682498427345e-11,    2.4992768960981608e-12,    8.3585332113866879e-14,
    2.5721706190198904e-15,    7.3299028805542247e-17,    1.9449336307462103e-18,    4.8281673542080567e-20,
    1.1259946364739500e-21,    2.4760550339825258e-23,    5.1507557280253963e-25,    1.0165672690246112e-26,
    1.9085212773635191e-28,    3.4165380507523849e-30,    5.8444382647191405e-32,    9.5724669574709688e-34,
    1.5038880810150248e-35,    2.2700782844458122e-37,    3.2973677169504050e-39,    4.6154420722082765e-41,
    6.2337965877888493e-43,    8.1342929964911511e-45,    1.0266294384878511e-46,    1.2545876912897640e-48,
    1.4860015986182938e-50,    1.7075739120453387e-52,    1.9053229312886821e-54,    2.0660998145519259e-56,
    2.1790748106298091e-58,    2.2369538340333633e-60,    2.2367358432676002e-62,    2.1799076576297240e-64,
    2.0720751762261189e-66,    1.9221250874262048e-68,    1.7410796150908910e-70,    1.5408375444281257e-72,
    1.3329865911123742e-74,    1.1278327226519202e-76,    9.3373432832267095e-79,    7.5676755707613175e-81,
    6.0069592378361829e-83,    4.6718024041259132e-85,    3.5614681715267815e-87,    2.6623108429022047e-89,
    1.9522582643153363e-91,    1.4048247219954984e-93,    9.9235048864367860e-96,    6.8835473927950925e-98,
   4.6903448541952743e-100,   3.1403533968931820e-102,   2.0666403013106000e-104,   1.3371827837502336e-106
},
// x = 0.7800
{
    5.0632303128156109e-01,    1.2659615938310423e-01,    1.9414725961929676e-02,    2.1427878322747515e-03,
    1.8457874920727071e-04,    1.3033033729320803e-05,    7.7955558864395570e-07,    4.0440585254881400e-08,
    1.8520260500821304e-09,    7.5914932578575916e-11,    2.8161539373321766e-12,    9.5403496555777591e-14,
    2.9739106874522097e-15,    8.5846316924760867e-17,    2.3074092874100774e-18,    5.8022905668236436e-20,
    1.3707288006820767e-21,    3.0533331686992527e-23,    6.4340447087399528e-25,    1.2863196815834659e-26,
    2.4463008226247165e-28,    4.4360815252467856e-30,    7.6869973225780025e-32,    1.2753776056334519e-33,
    2.0296988892490666e-35,    3.1035469493620024e-37,    4.5665312775655572e-39,    6.4749152899553744e-41,
    8.8588080215249466e-43,    1.1709665333018648e-44,    1.4970644980189215e-46,    1.8532331270801247e-48,
    2.2235691574362185e-50,    2.5882921661004868e-52,    2.9255322169721542e-54,    3.2135877896669001e-56,
    3.4333150613118439e-58,    3.5702715709478495e-60,    3.6162770814317387e-62,    3.5701620113216072e-64,
    3.4376227267527133e-66,    3.2302584763678767e-68,    2.9639933561521413e-70,    2.6571645849022213e-72,
    2.3285760735627360e-74,    1.9957788832985029e-76,    1.6737638114954707e-78,    1.3741574621091180e-80,
    1.1049226000619302e-82,    8.7049211589700178e-85,    6.7222192658858762e-87,    5.0903263061275806e-89,
    3.7811805074064753e-91,    2.7562308040352427e-93,    1.9722494787131888e-95,    1.3858378219818091e-97,
   9.5655123691550004e-100,   6.4876193969327037e-102,   4.3248906191227488e-104,   2.8346824857942231e-106
},
// x = 0.7900
{
    5.0254740619944083e-01,    1.2713942139620357e-01,    1.9739476846769081e-02,    2.2060236318445802e-03,
    1.9243200763953750e-04,    1.3760253672634966e-05,    8.3353878006330545e-07,    4.3792734884371505e-08,
    2.0311556765049386e-09,    8.4321592493069801e-11,    3.1680089500959176e-12,    1.0869635127955876e-13,
    3.4316343619287504e-15,    1.0032716788947411e-16,    2.7311555796347352e-18,    6.9558004148380930e-20,
    1.6642776096612885e-21,    3.7547035947330143e-23,    8.0133351375134076e-25,    1.6225809609121321e-26,
    3.1253304788336454e-28,    5.7400354277191932e-30,    1.0073979231954492e-31,    1.6928272387892700e-33,
    2.7285708031430199e-35,    4.2256342208052281e-37,    6.2972388901174727e-39,    9.0433249856754424e-41,
    1.2531405965959766e-42,    1.6776431475100992e-44,    2.1723326564948687e-46,    2.7236211532761404e-48,
    3.3097729997120704e-50,    3.9020414595533576e-52,    4.4669857188123255e-54,    4.9697097044258666e-56,
    5.3775660680108244e-58,    5.6637575763099948e-60,    5.8102721020105245e-62,    5.8097055290001981e-64,
    5.6657301032638919e-66,    5.3922059038522433e-68,    5.0111567229621605e-70,    4.5499939928559235e-72,
    4.0384475373961809e-74,    3.5056465009804373e-76,    2.9777044403352865e-78,    2.4760286421301666e-80,
    2.0164283698524109e-82,    1.6089686834615124e-84,    1.2584247659085648e-86,    9.6514388537856144e-89,
    7.2611554238023943e-91,    5.3607531407774539e-93,    3.8851161014259260e-95,    2.7649450624317962e-97,
    1.9329218631479776e-99,   1.3277716949761080e-101,   8.9648942251456283e-104,   5.9512264666619252e-106
},
// x = 0.8000
{
    4.9881467625334036e-01,    1.2766697842998420e-01,    2.0063507140899436e-02,    2.2700587993607986e-03,
    2.0049264649274481e-04,    1.4516526317421448e-05,    8.9040962819982954e-07,    4.7369859174216398e-08,
    2.2247686832720848e-09,    9.3524654684615015e-11,    3.5581345124775527e-12,    1.2362373207924520e-13,
    3.9522151992534444e-15,    1.1700710257510644e-16,    3.2254873436011640e-18,    8.3186369564236512e-20,
    2.0155229870013034e-21,    4.6046350432697317e-23,    9.9515557079667686e-25,    2.0405283351022616e-26,
    3.9800743432441324e-28,    7.4023418964356242e-30,    1.3155739099829322e-31,    2.2386527816325445e-33,
    3.6540077381220989e-35,    5.7304203275969850e-37,    8.6477927902104411e-39,    1.2576040825639319e-40,
    1.7647225834139892e-42,    2.3924188146509397e-44,    3.1370760892096641e-46,    3.9829662857068886e-48,
    4.9013921548929180e-50,    5.8515985639302213e-52,    6.7835760135610986e-54,    7.6425223377443396e-56,
    8.3743881300273508e-58,    8.9316909437330385e-60,    9.2787027763323655e-62,    9.3952151313093390e-64,
    9.2783416438836114e-66,    8.9421687718790352e-68,    8.4154305911830832e-70,    7.7376874700386401e-72,
    6.9546751606775335e-74,    6.1135378488251815e-76,    5.2585763888953666e-78,    4.4279673431981865e-80,
    3.6516884751718113e-82,    2.9506705236940238e-84,    2.3370210045464121e-86,    1.8150545417890030e-88,
    1.3828199307091297e-90,    1.0338273327005291e-92,    7.5873222171528445e-95,    5.4680613454625085e-97,
    3.8710032361688943e-99,   2.6927437394097494e-101,   1.8411076738230196e-103,   1.2376644358238217e-105
},
// x = 0.8100
{
    4.9512425982492919e-01,    1.2817912310725696e-01,    2.0386766835088662e-02,    2.3348833844874929e-03,
    2.0876227778932998e-04,    1.5302520161604897e-05,    9.5027559469562339e-07,    5.1183456613413130e-08,
    2.4338055583432722e-09,    1.0358687040619044e-10,    3.9900796795468783e-12,    1.4036019571573478e-13,
    4.5432579889767819e-15,    1.3618371027865370e-16,    3.8009796883232802e-18,    9.9252301647399219e-20,
    2.4348104598547397e-21,    5.6319949613452647e-23,    1.2323901376523711e-24,    2.5585300936928648e-26,
    5.0527772617702432e-28,    9.5147971039655400e-30,    1.7121325045830055e-31,    2.9498562821782186e-33,
    4.8750134379234032e-35,    7.7407898998750953e-37,    1.1827602242711876e-38,    1.7415188413971994e-40,
    2.4743072987985808e-42,    3.3963113153142770e-44,    4.5090863139128129e-46,    5.7964681203413546e-48,
    7.2222031398202399e-50,    8.7300813992964737e-52,    1.0246984268588590e-53,    1.1688742024951708e-55,
    1.2968146185807805e-57,    1.4004007051541208e-59,    1.4729899363231465e-61,    1.5101260001907990e-63,
    1.5099786545297457e-65,    1.4734566105331002e-67,    1.4039929212592675e-69,    1.3070550794240326e-71,
    1.1894706308490459e-73,    1.0586775038934114e-75,    9.2200549040428667e-78,    7.8607503304497384e-80,
    6.5636833020721579e-82,    5.3699340036731965e-84,    4.3063091602987720e-86,    3.3863095497343062e-88,
    2.6121434033520858e-90,    1.9773057419395382e-92,    1.4692943195466851e-94,    1.0721316683994788e-96,
    7.6848036289024688e-99,   5.4125108145737772e-101,   3.7469463105803047e-103,   2.5503262445104721e-105
},
// x = 0.8200
{
    4.9147558396896829e-01,    1.2867614289814347e-01,    2.0709207512345860e-02,    2.4004873350573228e-03,
    2.1724245697873025e-04,    1.6118904802967027e-05,    1.0132462072213548e-06,    5.5245420189453071e-08,
    2.6592525362376290e-09,    1.1457492599001913e-10,    4.4676657371883077e-12,    1.5909613519653455e-13,
    5.2131646025828588e-15,    1.5818999583772703e-16,    4.4696176820896092e-18,    1.1815098333838966e-19,
    2.9341656382945364e-21,    6.8707650926516297e-23,    1.5220011871359125e-24,    3.1987604530005024e-26,
    6.3950826981967484e-28,    1.2191039021285760e-29,    2.2207724005559905e-31,    3.8734084785026408e-33,
    6.4802873130553588e-35,    1.0416694817562935e-36,    1.6112679857303651e-38,    2.4017390617938056e-40,
    3.4544428330069743e-42,    4.8001933773871191e-44,    6.4515903331573628e-46,    8.3959296517722574e-48,
    1.0590153323340757e-49,    1.2959198558083143e-51,    1.5398673459897642e-53,    1.7782076814697608e-55,
    1.9971935131405575e-57,    2.1833440560466255e-59,    2.3248626864424331e-61,    2.4128950921903677e-63,
    2.4424396634706602e-65,    2.4127827422768445e-67,    2.3274140995320498e-69,    2.1934642045050645e-71,
    2.0207767911646222e-73,    1.8207750931597690e-75,    1.6052924341632319e-77,    1.3855197464004326e-79,
    1.1711821151072088e-81,    9.7000480930451824e-84,    7.8747776143955694e-86,    6.2688507015235531e-88,
    4.8953809159581720e-90,    3.7513840638221724e-92,    2.8219853167776075e-94,    2.0845981291556391e-96,
    1.5126389619332928e-98,   1.0785234512552189e-100,   7.5585172925209267e-103,   5.2081468696880733e-105
},
// x = 0.8300
{
    4.8786808427619244e-01,    1.2915832021032267e-01,    2.1030782311170712e-02,    2.4668605044745901e-03,
    2.2593468307181719e-04,    1.6966350683803109e-05,    1.0794330455109037e-06,    5.9568043270880844e-08,
    2.9021430251056436e-09,    1.2655962413875284e-10,    4.9950026522661586e-12,    1.8003896093431653e-13,
    5.9712046887138668e-15,    1.8339802124827466e-16,    5.2449613603544502e-18,    1.4033516360105388e-19,
    3.5275391102718060e-21,    8.3608614343492425e-23,    1.8746500040544107e-24,    3.9879222994594888e-26,
    8.0699587465125385e-28,    1.5571355195603500e-29,    2.8711150552998362e-31,    5.0687512068130324e-33,
    8.5834782739390526e-35,    1.3965631622679640e-36,    2.1865539792733539e-38,    3.2989839461775609e-40,
    4.8027999906648697e-42,    6.7551885349493103e-44,    9.1898389367421256e-46,    1.2105211990704678e-47,
    1.5454980144796589e-49,    1.9142869483104602e-51,    2.3023692784160784e-53,    2.6911444649271451e-55,
    3.0594094828016906e-57,    3.3853427023475579e-59,    3.6487224667001597e-61,    3.8330552933831183e-63,
    3.9272962465963349e-65,    3.9269128308230432e-67,    3.8341577329375124e-69,    3.6575492318006094e-71,
    3.4106827904244352e-73,    3.1105893585419883e-75,    2.7759009526744169e-77,    2.4250795499198165e-79,
    2.0749192281894789e-81,    1.7394589114923297e-83,    1.4293616850714609e-85,    1.1517423669650161e-87,
    9.1036910055921542e-90,    7.0613245526929005e-92,    5.3766657841402809e-94,    4.0201689421885127e-96,
    2.9527109299551283e-98,   2.1309779177061899e-100,   1.5116445860121400e-102,   1.0542899028141949e-104
},
// x = 0.8400
{
    4.8430120473844646e-01,    1.2962593247768867e-01,    2.1351445889558385e-02,    2.5339926588903937e-03,
    2.3484039880482339e-04,    1.7845528838715914e-05,    1.1489497264006990e-06,    6.4164025371760156e-08,
    3.1635590478388148e-09,    1.3961607026031764e-10,    5.5765062363923194e-12,    2.0341435050951509e-13,
    6.8275914889294975e-15,    2.1222286280403693e-16,    6.1423273711637413e-18,    1.6632260909827138e-19,
    4.2310830132496847e-21,    1.0149072060514202e-22,    2.3029880354344298e-24,    4.9580949101713824e-26,
    1.0153985691441401e-27,    1.9828464916507722e-29,    3.7000794195865429e-31,    6.6108830083920183e-33,
    1.1329744148171686e-34,    1.8655886252012798e-36,    2.9560665924847461e-38,    4.5137041923744009e-40,
    6.6503795388768569e-42,    9.4664813653588122e-44,    1.3033417795982131e-45,    1.7374897051137383e-47,
    2.2450076290930320e-49,    2.8142069651110396e-51,    3.4254977827409372e-53,    4.0521500241369522e-55,
    4.6621472041699505e-57,    5.2209670354028622e-59,    5.6949398882697948e-61,    6.0547115554935969e-63,
    6.2783012700428941e-65,    6.3533080937738590e-67,    6.2779643301239113e-69,    6.0609305323445640e-71,
    5.7199305275596355e-73,    5.2794957304233669e-75,    4.7681960098758455e-77,    4.2157663225488632e-79,
    3.6504985034169789e-81,    3.0971741259258520e-83,    2.5756929008129548e-85,    2.1004280551273356e-87,
    1.6802369253251079e-89,    1.3189847095031629e-91,    1.0164060084029983e-93,    7.6912884986916412e-96,
    5.7171075889013553e-98,   4.1757552671952123e-100,   2.9978262219598063e-102,   2.1160089405423916e-104
},
// x = 0.8500
{
    4.8077439761603841e-01,    1.3007925224744585e-01,    2.1671154389759265e-02,    2.6018734841555232e-03,
    2.4396099083148889e-04,    1.8757110645641145e-05,    1.2219118878976509e-06,    6.9046477794729395e-08,
    3.4446326965432403e-09,    1.5382386386459609e-10,    6.2169160405028969e-12,    2.2946756981857780e-13,
    7.7935630590304014e-15,    2.4512690591890489e-16,    7.1789886651885046e-18,    1.9670440070889439e-19,
    5.0634628641229164e-21,    1.2290127823973688e-22,    2.8219954278728147e-24,    6.1477261743778945e-26,
    1.2740066876616012e-27,    2.5174456572330059e-29,    4.7535517902059126e-31,    8.5941535928950940e-33,
    1.4903917815669123e-34,    2.4833226858339200e-36,    3.9817006656045541e-38,    6.1521198637290099e-40,
    9.1722436813514997e-42,    1.3211597638908236e-43,    1.8406143446062089e-45,    2.4829305577872048e-47,
    3.2463706980990910e-49,    4.1178865260623015e-51,    5.0720108527524560e-53,    6.0712794560352974e-55,
    7.0683653474009390e-57,    8.0098120860089073e-59,    8.8409504014363338e-61,    9.5113411952222483e-63,
    9.9799646506191259e-65,    1.0219399263783893e-66,    1.0218401007399259e-68,    9.9825638462988580e-71,
    9.5330589336896372e-73,    8.9037455323943287e-75,    8.1371665571166499e-77,    7.2800519611025952e-79,
    6.3789470786613737e-81,    5.4764780667735054e-83,    4.6085971340561457e-85,    3.8029571855346471e-87,
    3.0783864357811968e-89,    2.4452956963783772e-91,    1.9067681076715291e-93,    1.4600536429938292e-95,
    1.0982093841159504e-97,   8.1167639682385655e-100,   5.8964883674230885e-102,   4.2115660914054966e-104
},
// x = 0.8600
{
    4.7728712330727141e-01,    1.3051854726566894e-01,    2.1989865403774973e-02,    2.6704925925566383e-03,
    2.5329778994219384e-04,    1.9701767580191722e-05,    1.2984371722997185e-06,    7.4228929149467993e-08,
    3.7465475996947556e-09,    1.6926729503682890e-10,    6.9213139973719103e-12,    2.5846486844500015e-13,
    8.8814691916745889e-15,    2.8262450097146916e-16,    8.3743937331148131e-18,    2.3215415713237433e-19,
    6.0462085757377589e-21,    1.4847922615795792e-22,    3.4493716348205808e-24,    7.6027915677122561e-26,
    1.5940634161861135e-27,    3.1869091505524251e-29,    6.0884090990020926e-31,    1.1136917722635106e-32,
    1.9540645387163044e-34,    3.2941880887236013e-36,    5.3439307788904215e-38,    8.3539940741126853e-40,
    1.2601500974265252e-41,    1.8364516345572523e-43,    2.5885976515406283e-45,    3.5330032494564272e-47,
    4.6736414812759407e-49,    5.9980430600847313e-51,    7.4746934946357776e-53,    9.0525624905094610e-55,
    1.0663222384343223e-56,    1.2225596134603373e-58,    1.3652902123774060e-60,    1.4860935498546076e-62,
    1.5776541794034193e-64,    1.6345064906193271e-66,    1.6535706151159399e-68,    1.6344079543255735e-70,
    1.5791713541759567e-72,    1.4922732952554938e-74,    1.3798361491020452e-76,    1.2490145909662158e-78,
    1.1072884866696319e-80,    9.6181583026215503e-83,    8.1891367848885454e-85,    6.8370641523564997e-87,
    5.5995125227515423e-89,    4.5002574124434100e-91,    3.5504447892419116e-93,    2.7506328796577053e-95,
    2.0932828836388151e-97,    1.5653255088754837e-99,   1.1505200299447392e-101,   8.3142555315098207e-104
},
// x = 0.8700
{
    4.7383885022011679e-01,    1.3094408056135820e-01,    2.2307537939571419e-02,    2.7398395293411456e-03,
    2.6285207130888197e-04,    2.0680170973402281e-05,    1.3786452083933161e-06,    7.9725330743537764e-08,
    4.0705404012857898e-09,    1.8603554599933276e-10,    7.6951438290977757e-12,    2.9069495214492525e-13,
    1.0104864346878809e-14,    3.2528700472963282e-16,    9.7504069936538095e-18,    2.7343827450584200e-19,
    7.2041089628910129e-21,    1.7896901687241080e-22,    4.2059852652917148e-24,    9.3781451958626412e-26,
    1.9891430084606766e-27,    4.0229720824411722e-29,    7.7749644952087087e-31,    1.4387228504637938e-32,
    2.5536938525775418e-34,    4.3550820747714416e-36,    7.1470529773049478e-38,    1.1302610109556812e-39,
    1.7247471253565359e-41,    2.5427365312314590e-43,    3.6258118807552994e-45,    5.0061483981153297e-47,
    6.6993729247283898e-49,    8.6977619182293927e-51,    1.0965049425323343e-52,    1.3434084901932239e-54,
    1.6008271947348790e-56,    1.8567162257733297e-58,    2.0975872338837443e-60,    2.3097279576288234e-62,
    2.4805396791526562e-64,    2.5998048669594799e-66,    2.6607044418728517e-68,    2.6604444002598390e-70,
    2.6004161301260670e-72,    2.4858898144886751e-74,    2.3253106319946931e-76,    2.1293201159000817e-78,
    1.9096520724765143e-80,    1.6780520787709322e-82,    1.4453456476303656e-84,    1.2207405062278564e-86,
    1.0114025715659468e-88,    8.2230199988922654e-91,    6.5629173449779215e-93,    5.1435978199810832e-95,
    3.9598845033411757e-97,    2.9955702070358510e-99,   2.2273542022306823e-101,   1.6283162217520204e-103
},
// x = 0.8800
{
    4.7042905464599405e-01,    1.3135611052901280e-01,    2.2624132387995884e-02,    2.8099037790360469e-03,
    2.7262505475461412e-04,    2.1692991772948982e-05,    1.4626575927518096e-06,    8.5550061842698512e-08,
    4.4179022512668979e-09,    2.0422289776982937e-10,    8.5442312364906770e-12,    3.2647053539272793e-13,
    1.1478606908016351e-14,    3.7374823317253206e-16,    1.1331572040935458e-17,    3.2142727806811991e-19,
    8.5656544448738833e-21,    2.1523638534903886e-22,    5.1163912189332661e-24,    1.1539091661744964e-25,
    2.4755961144768949e-27,    5.0643102048849439e-29,    9.8999163538668688e-31,    1.8529784757523566e-32,
    3.3267674622379976e-34,    5.7386609710754388e-36,    9.5258127382325261e-38,    1.5237537308470243e-39,
    2.3519204386272501e-41,    3.5071946346500899e-43,    5.0585403230299203e-45,    7.0645618616603355e-47,
    9.5626297763285353e-49,    1.2557768722153338e-50,    1.6013174163338707e-52,    1.9844349900394644e-54,
    2.3918573384933516e-56,    2.8060697111369099e-58,    3.2065286530446005e-60,    3.5713972288462064e-62,
    3.8795896675531392e-64,    4.1128484843222116e-66,    4.2575620378110155e-68,    4.3060689111658149e-70,
    4.2572790844935388e-72,    4.1165525757476912e-74,    3.8948911808659758e-76,    3.6075960523866809e-78,
    3.2726067580928707e-80,    2.9087585249507600e-82,    2.5341752326592923e-84,    2.1649655770664488e-86,
    1.8143222924568025e-88,    1.4920538503549112e-90,    1.2045168435412423e-92,    9.5487332235779126e-95,
    7.4357385392889778e-97,    5.6896290001382493e-99,   4.2791410831568973e-101,   3.1642366828289549e-103
},
// x = 0.8900
{
    4.6705722063562050e-01,    1.3175489100974982e-01,    2.2939610490396587e-02,    2.8806747715659380e-03,
    2.8261790504662698e-04,    2.2740900307914180e-05,    1.5505978701594977e-06,    9.1717934797939751e-08,
    4.7899803065797839e-09,    2.2392894192139095e-10,    9.4748048871676923e-12,    3.6612997698466449e-13,
    1.3018965092093401e-14,    4.2871035282760053e-16,    1.3145399570687555e-17,    3.7710838946658869e-19,
    1.0163533087281942e-20,    2.5828623029187435e-22,    6.2094241183423357e-24,    1.4163211359750640e-25,
    3.0730730583485653e-27,    6.3579448864573141e-29,    1.2569896107606351e-30,    2.3794387911857270e-32,
    4.3204687603807089e-34,    7.5374333567351894e-36,    1.2653761841401679e-37,    2.0470916308155234e-39,
    3.1955835529398043e-41,    4.8193960688530950e-43,    7.0301291359614478e-45,    9.9295319412267006e-47,
    1.3593348683877084e-48,    1.8053743501818067e-50,    2.3282948679729117e-52,    2.9181209044891571e-54,
    3.5571944785543768e-56,    4.2206252882185684e-58,    4.8777499940560054e-60,    5.4945066449588796e-62,
    6.0364628351662357e-64,    6.4721070646395578e-66,    6.7759510912233180e-68,    6.9310111166874555e-70,
    6.9303333782792384e-72,    6.7773840821577400e-74,    6.4853023031330285e-76,    6.0751820974576752e-78,
    5.5736764449935536e-80,    5.0102819503797934e-82,    4.4146648119974941e-84,    3.8143339387135501e-86,
    3.2328741889289270e-88,    2.6888435919753719e-90,    2.1953340797422019e-92,    1.7601116750761969e-94,
    1.3861981570971284e-96,    1.0727336860524464e-98,   8.1596467563836147e-101,   6.1022572802784400e-103
},
// x = 0.9000
{
    4.6372283987689633e-01,    1.3214067137099650e-01,    2.3253935306907843e-02,    2.9521418881752037e-03,
    2.9283173221180439e-04,    2.3824566057159734e-05,    1.6425915131854483e-06,    9.8244200036602517e-08,
    5.1881792420730090e-09,    2.4525879744571488e-10,    1.0493518219027052e-11,    4.1003900175023412e-13,
    1.4743729854401454e-14,    4.9095023908269753e-16,    1.5222681920524995e-17,    4.4159942133668148e-19,
    1.2035185594861386e-20,    3.0928285843058834e-22,    7.5188781162860844e-24,    1.7342476104932949e-25,
    3.8051374818052547e-27,    7.9609100423463182e-29,    1.5915726842335987e-30,    3.0466211783292880e-32,
    5.5940221719196502e-34,    9.8688472859109028e-36,    1.6753765701392119e-37,    2.7408173133197682e-39,
    4.3265644379795105e-41,    6.5983593274531015e-43,    9.7332331581210590e-45,    1.3901869488240451e-46,
    1.9245163527438020e-48,    2.5847184241106104e-50,    3.3708146145744585e-52,    4.2721960352651721e-54,
    5.2663119860229087e-56,    6.3186882436372869e-58,    7.3844965850706519e-60,    8.4116529906150611e-62,
    9.3451553075978059e-64,    1.0132137770415503e-65,    1.0726971035101364e-67,    1.1095705975462881e-69,
    1.1219254872479554e-71,    1.1094904553201192e-73,    1.0736020205746148e-75,    1.0170072635057877e-77,
    9.4353540727464294e-80,    8.5768998673088079e-82,    7.6421870673823777e-84,    6.6771391302767871e-86,
    5.7228495450225141e-88,    4.8132775051761917e-90,    3.9739997970675605e-92,    3.2219539437243776e-94,
    2.5659980749685520e-96,    2.0080515274839922e-98,   1.5445651678473725e-100,   1.1680928240960400e-102
},
// x = 0.9100
{
    4.6042541157479078e-01,    1.3251369658477974e-01,    2.3567071185407244e-02,    3.0242944671593427e-03,
    3.0326759187350650e-04,    2.4944657421367552e-05,    1.7387659009316714e-06,    1.0514455091510581e-07,
    5.6139627705866324e-09,    2.6832333271815887e-10,    1.1607472075623123e-11,    4.5859251147132458e-13,
    1.6672335139096277e-14,    5.6132633131710629e-16,    1.7597836281034256e-17,    5.1616411931138754e-19,
    1.4223425371613248e-20,    3.6957286826931256e-22,    9.0842843320026217e-24,    2.1185696810712922e-25,
    4.6999845542249335e-27,    9.9422257434504263e-29,    2.0097523747282448e-30,    3.8898244646109884e-32,
    7.2215674234905525e-34,    1.2881596545361745e-35,    2.2111178482594939e-37,    3.6574291251340085e-39,
    5.8376164003869076e-41,    9.0017104550501525e-43,    1.3425888708764855e-44,    1.9389030209608710e-46,
    2.7139482665144734e-48,    3.6854487910470984e-50,    4.8596980946596387e-52,    6.2276322825233837e-54,
    7.7620389963783044e-56,    9.4165905506940928e-58,    1.1127183337261729e-59,    1.2815730110915502e-61,
    1.4396145988136843e-63,    1.5781874090922846e-65,    1.6893997127072225e-67,    1.7668843169167916e-69,
    1.8064049007078738e-71,    1.8062281788142312e-73,    1.7672189364272970e-75,    1.6926576215864851e-77,
    1.5878202382392247e-79,    1.4593906775099929e-81,    1.3147918989218977e-83,    1.1615233294142559e-85,
    1.0065793653146241e-87,    8.5600205073889018e-90,    7.1459500764120195e-92,    5.8580047780497391e-94,
    4.7172076893414216e-96,    3.7325172467262980e-98,   2.9028963236793555e-100,   2.2197342426816015e-102
},
// x = 0.9200
{
    4.5716444233319531e-01,    1.3287420730463589e-01,    2.3878983731121687e-02,    3.0971218094102537e-03,
    3.1392648560872632e-04,    2.6101841498800606e-05,    1.8392502969800085e-06,    1.1243512843092257e-07,
    6.0688551714878740e-09,    2.9323939255969617e-10,    1.2824238189800695e-11,    5.1221648815847424e-13,
    1.8825985838838602e-14,    6.4078601612115311e-16,    2.0309278761284558e-17,    6.0222908066128954e-19,
    1.6777130306324221e-20,    4.4071097792082874e-22,    1.0951798466608843e-23,    2.5821350286444612e-25,
    5.7912799638464683e-27,    1.2385230172307487e-28,    2.5310789636202143e-30,    4.9526328607912560e-32,
    9.2956734641925187e-34,    1.6763418191539355e-35,    2.9090319261427245e-37,    4.8647039780000554e-39,
    7.8498277312565497e-41,    1.2237531998059195e-42,    1.8452542403256903e-44,    2.6941002475476420e-46,
    3.8124472241401735e-48,    5.2340440512842260e-50,    6.9775200926937586e-52,    9.0398176390831260e-54,
    1.1390886187442668e-55,    1.3970773079549161e-57,    1.6690030275944338e-59,    1.9433920740141208e-61,
    2.2070316926380335e-63,    2.4460549585289670e-65,    2.6471918255272730e-67,    2.7990233529106272e-69,
    2.8930701396285998e-71,    2.9245696556779261e-73,    2.8928455586869814e-75,    2.8012352695745166e-77,
    2.6566075677334493e-79,    2.4685576859807585e-81,    2.2484042806770598e-83,    2.0081262876817878e-85,
    1.7593685940733667e-87,    1.5126183718364935e-89,    1.2766161135812843e-91,    1.0580248475547931e-93,
    8.6134466262891047e-96,    6.8903238847812479e-98,   5.4177031254884235e-100,   4.1882301879633234e-102
},
// x = 0.9300
{
    4.5393944603871111e-01,    1.3322243994116947e-01,    2.4189639776874241e-02,    3.1706131837801869e-03,
    3.2480936132458680e-04,    2.7296783864832992e-05,    1.9441758265621928e-06,    1.2013252579158588e-07,
    6.5544428269361374e-09,    3.2013003038768760e-10,    1.4151883531769969e-11,    5.7136999288158346e-13,
    2.1227793838340960e-14,    7.3037357134243628e-16,    2.3399831625079416e-17,    7.0140238834668361e-19,
    1.9752013523315647e-20,    5.2448913225687434e-22,    1.3175212573073453e-23,    3.1400838190915411e-25,
    7.1191381734410745e-27,    1.5390329508517495e-28,    3.1793684276057442e-30,    6.2887297800397437e-32,
    1.1931624243676688e-33,    2.1750710057884528e-35,    3.8155022561407702e-37,    6.4498884801369506e-39,
    1.0520797156937443e-40,    1.6579645946485108e-42,    2.5271478769979558e-44,    3.7297640253861414e-46,
    5.3353722024732443e-48,    7.4044301237047262e-50,    9.9781227736456908e-52,    1.3067755159897268e-53,
    1.6645332940148131e-55,    2.0637122540994803e-57,    2.4921812813646665e-59,    2.9334372228442712e-61,
    3.3675873350934907e-63,    3.7728582736151586e-65,    4.1274681329682275e-67,    4.4116283774483641e-69,
    4.6094116169386919e-71,    4.7102360381057125e-73,    4.7097750191859684e-75,    4.6101891974497734e-77,
    4.4196802759597481e-79,    4.1514617993050866e-81,    3.8223153820349861e-83,    3.4509408429563924e-85,
    3.0563123382036245e-87,    2.6562240391326040e-89,    2.2661576073546861e-91,    1.8985416851041585e-93,
    1.5624126262687436e-95,    1.2634350880190158e-97,    1.0042065383204403e-99,   7.8475293300555910e-102
},
// x = 0.9400
{
    4.5074994374633831e-01,    1.3355862673628047e-01,    2.4499007353954064e-02,    3.2447578322689816e-03,
    3.3591711365320775e-04,    2.8530148355292419e-05,    2.0536754529774276e-06,    1.2825379283863588e-07,
    7.0723757651529548e-09,    3.4912474544412730e-10,    1.5598995537613545e-11,    6.3654726339912835e-13,
    2.3902922528491436e-14,    8.3123870520734372e-16,    2.6917166152829750e-17,    8.1549410940502908e-19,
    2.3211480961768790e-20,    6.2296925782662301e-22,    1.5817106522205944e-23,    3.8102237809119096e-25,
    8.7312609756824281e-27,    1.9078234334026222e-28,    3.9835675640487567e-30,    7.9640806013338316e-32,
    1.5272633818409138e-33,    2.8140364184308379e-35,    4.9894205396916748e-37,    8.5249711532044840e-39,
    1.4055032422421323e-40,    2.2387268426224912e-42,    3.4490485141973742e-44,    5.1450894121542121e-46,
    7.4390817912626868e-48,    1.0434926640573496e-49,    1.4213134071081609e-51,    1.8814181838085483e-53,
    2.4222572085405113e-55,    3.0354313346795039e-57,    3.7050533878172488e-59,    4.4079359516175165e-61,
    5.1147103231242996e-63,    5.7918381892114509e-65,    6.4043265035963789e-67,    6.9188276791666284e-69,
    7.3067287833039278e-71,    7.5468220865448827e-73,    7.6272079771338653e-75,    7.5461980605876499e-77,
    7.3121377400577020e-79,    6.9422244299961182e-81,    6.4605314689931584e-83,    5.8955373751059272e-85,
    5.2774946219673283e-87,    4.6359524880540224e-89,    3.9976854413012984e-91,    3.3851888549778802e-93,
    2.8158063593060862e-95,    2.3014645035045944e-97,    1.8489225259282833e-99,   1.4604038109385019e-101
},
// x = 0.9500
{
    4.4759546356703411e-01,    1.3388299583608784e-01,    2.4807055663598920e-02,    3.3195449750390653e-03,
    3.4725058436401595e-04,    2.9802596853654303e-05,    2.1678839532817902e-06,    1.3681644032454862e-07,
    7.6243692099699297e-09,    3.8035972508679573e-10,    1.7174708234013060e-11,    7.0827991387531433e-13,
    2.6878740189661799e-14,    9.4464572631987459e-16,    3.0914283728531858e-17,    9.4653881732277948e-19,
    2.7227584314852301e-20,    7.3852007001447239e-22,    1.8950156424399209e-23,    4.6134611681902230e-25,
    1.0684260249876261e-26,    2.3593761383627688e-28,    4.9787814974082375e-30,    1.0059553799215764e-31,
    1.9496177962255507e-33,    3.6304292102990847e-35,    6.5053595972549259e-37,    1.1233297252828269e-38,
    1.8717140385611301e-40,    3.0130214614888332e-42,    4.6913089964497074e-44,    7.0726376849363494e-46,
    1.0334790708646736e-47,    1.4650948096368599e-49,    2.0167880499317339e-51,    2.6980494960180823e-53,
    3.5105817085148309e-55,    4.4460421194667625e-57,    5.4845630423098880e-59,    6.5944306822322654e-61,
    7.7331708515969747e-63,    8.8500876537320084e-65,    9.8900675743385966e-67,    1.0798240606304323e-68,
    1.1524927756903903e-70,    1.2030235206643967e-72,    1.2287694870541254e-74,    1.2286491667415108e-76,
    1.2032031261456493e-78,    1.1544846647997692e-80,    1.0858072273194947e-82,    1.0013891272687086e-84,
    9.0594596579537374e-87,    8.0428231717022191e-89,    7.0092772070434352e-91,    5.9984993660096234e-93,
    5.0426360061468406e-95,    4.1653764548285645e-97,    3.3819240700025143e-99,   2.6996861494310653e-101
},
// x = 0.9600
{
    4.4447554055710931e-01,    1.3419577136256844e-01,    2.5113755049082680e-02,    3.3949638152626469e-03,
    3.5881056279258954e-04,    3.1114789082122106e-05,    2.2869378932736948e-06,    1.4583844404081718e-07,
    8.2122051359261185e-09,    4.1397809212554173e-10,    1.8888729274766773e-11,    7.8713924001858471e-13,
    3.0184982654847324e-14,    1.0719833819358293e-15,    3.5450037902117133e-17,    1.0968203093742242e-18,
    3.1882078574496158e-20,    8.7385837591806697e-22,    2.2658619149974216e-23,    5.5742951281077010e-25,
    1.3045192036682853e-26,    2.9110291077327366e-28,    6.2074916299614776e-30,    1.2674061882954480e-31,
    2.4821664316744316e-33,    4.6707212829109663e-35,    8.4575012798042573e-37,    1.4757846856981481e-38,
    2.4848513252855143e-40,    4.0421130987435195e-42,    6.3598334939303152e-44,    9.6889947529692436e-46,
    1.4306872942471404e-47,    2.0495315042307951e-49,    2.8509858606454235e-51,    3.8541689189727096e-53,
    5.0676432185744196e-55,    6.4855484840700602e-57,    8.0846539468837257e-59,    9.8229750707596044e-61,
    1.1640449044623084e-62,    1.3461893588769802e-64,    1.5202126658640037e-66,    1.6772764322665382e-68,
    1.8089912245313723e-70,    1.9081785724503925e-72,    1.9695272943847015e-74,    1.9900601521273802e-76,
    1.9693550864754683e-78,    1.9095016779270688e-80,    1.8148111317797349e-82,    1.6913303380620649e-84,
    1.5462323397805583e-86,    1.3871642707910298e-88,    1.2216296138860661e-90,    1.0564669122674174e-92,
    8.9746576865138672e-95,    7.4913750680489695e-97,    6.1463624687385622e-99,   4.9580927384897893e-101
},
// x = 0.9700
{
    4.4138971660943149e-01,    1.3449717348393808e-01,    2.5419076968385617e-02,    3.4710035438053660e-03,
    3.7059778628516460e-04,    3.2467382396624009e-05,    2.4109756017995137e-06,    1.5533824879547920e-07,
    8.8377338281851560e-09,    4.5013015718275230e-10,    2.0751367904440116e-11,    8.7373863301746796e-13,
    3.3853925675420040e-14,    1.2147754035493671e-15,    4.0589700334554020e-17,    1.2688987018758578e-18,
    3.7267594191566946e-20,    1.0320953586312072e-21,    2.7040014152279536e-23,    6.7213838961647010e-25,
    1.5893332648269911e-26,    3.5834984423673295e-28,    7.7209965384390534e-30,    1.5928317512772452e-31,
    3.1519703946669933e-33,    5.9928386814557557e-35,    1.0964487469562342e-36,    1.9331568495246788e-38,
    3.2888382368485460e-40,    5.4056601573979287e-42,    8.5937822506349361e-44,    1.3228657615219155e-45,
    1.9736964204310856e-47,    2.8568597531193357e-49,    4.0153941051305597e-51,    5.4848245945500916e-53,
    7.2868025369020158e-55,    9.4227296402997871e-57,    1.1868356392994697e-58,    1.4570396751301667e-60,
    1.7446058521033179e-62,    2.0386048719006217e-64,    2.3261119866251539e-66,    2.5931661053775189e-68,
    2.8259314222547588e-70,    3.0119217469843872e-72,    3.1411322816987137e-74,    3.2069339696059743e-76,
    3.2066198152033539e-78,    3.1415440262096428e-80,    3.0168536052306459e-82,    2.8408674268080850e-84,
    2.6242006329885889e-86,    2.3787563486897574e-88,    2.1167103184544768e-90,    1.8495991101621611e-92,
    1.5875940293251694e-94,    1.3390074211711727e-96,    1.1100422532545178e-98,   9.0476523971429454e-101
},
// x = 0.9800
{
    4.3833754034640560e-01,    1.3478741848379283e-01,    2.5722993967448048e-02,    3.5476533437506395e-03,
    3.8261294065795458e-04,    3.3861031585752758e-05,    2.5401371444033796e-06,    1.6533477223851342e-07,
    9.5028754465417861e-09,    4.8897367605389402e-10,    2.2773563864249462e-11,    9.6873610569080263e-13,
    3.7920567424499479e-14,    1.3746918005106191e-15,    4.6405573696670964e-17,    1.4656400989275506e-18,
    4.3488934677038045e-20,    1.2167883742276253e-21,    3.2207026051442298e-23,    8.0881922419194712e-25,
    1.9322231558684622e-26,    4.4014877759033965e-28,    9.5811133942035496e-30,    1.9969316262593627e-31,
    3.9923294902788970e-33,    7.6688111991367617e-35,    1.4175397077241983e-36,    2.5250247099747141e-38,
    4.3400314655257130e-40,    7.2069431179942738e-42,    1.1575475495689341e-43,    1.8002074121288427e-45,
    2.7135605753490264e-47,    3.9682643816234997e-49,    5.6349811663330206e-51,    7.7764206249695300e-53,
    1.0437747734639571e-54,    1.3636389624943744e-56,    1.7352665703879238e-58,    2.1522861814804149e-60,
    2.6036286799768566e-62,    3.0737456076842438e-64,    3.5433877489544014e-66,    3.9909073399685785e-68,
    4.3939612515419047e-70,    4.7314213970572254e-72,    4.9852570305199944e-74,    5.1421503314977117e-76,
    5.1946427221092883e-78,    5.1416777729675002e-80,    4.9884941762858437e-82,    4.7459128514879475e-84,
    4.4291400769732537e-86,    4.0562615945960761e-88,    3.6466246529315701e-90,    3.2192960075856495e-92,
    2.7917495190923784e-94,    2.3788863218925513e-96,    1.9924340213956029e-98,   1.6407185902663341e-100
},
// x = 0.9900
{
    4.3531856701470062e-01,    1.3506671882903620e-01,    2.6025479653984867e-02,    3.6249023947687617e-03,
    3.9485666067046036e-04,    3.5296388673670379e-05,    2.6745642963450281e-06,    1.7584740853364785e-07,
    1.0209621592787697e-08,    5.3067411204091306e-10,    2.4966917255025602e-11,    1.0728369343089546e-12,
    4.2422821585247544e-14,    1.5535609441177713e-15,    5.2977654762879693e-17,    1.6902490435169052e-18,
    5.0664511340247579e-20,    1.4319988419853165e-21,    3.8289653898025327e-23,    9.7137306921873558e-25,
    2.3442080317554274e-26,    5.3943990588211297e-28,    1.1862183280667421e-29,    2.4975676086669339e-31,
    5.0441284102610596e-33,    9.7879954782361813e-35,    1.8277093515111302e-36,    3.2888489341701559e-38,
    5.7105474993345164e-40,    9.5794970482721534e-42,    1.5543089501021248e-43,    2.4419011782527201e-45,
    3.7183666857563668e-47,    5.4931386722614820e-49,    7.8798836728365461e-51,    1.0985366481309414e-52,
    1.4895296752546909e-54,    1.9658455929938524e-56,    2.5271086540951151e-58,    3.1663980876255242e-60,
    3.8694780808076870e-62,    4.6147605559842465e-64,    5.3741265928015430e-66,    6.1146115463593297e-68,
    6.8008236455921956e-70,    7.3978416028095193e-72,    7.8742482789300286e-74,    8.2049229949125086e-76,
    8.3732421600283886e-78,    8.3724215742266037e-80,    8.2058580178473427e-82,    7.8864692779624557e-84,
    7.4351653287224529e-86,    6.8786871136969871e-88,    6.2471088833255857e-90,    5.5713104558844103e-92,
    4.8806915215761558e-94,    4.2013327727975019e-96,    3.5547236693796202e-98,   2.9570898522515107e-100
},
// x = 1.0000
{
    4.3233235838169365e-01,    1.3533528323661276e-01,    2.6326508671855375e-02,    3.7027398773347982e-03,
    4.0732953051197805e-04,    3.6774102726997153e-05,    2.8144005150093787e-06,    1.8689603187524469e-07,
    1.0960036880706861e-08,    5.7540490322815869e-10,    2.7343719371837057e-11,    1.1867964195805775e-12,
    4.7401721483767339e-14,    1.7533824863944216e-15,    6.0394351117949848e-17,    1.9463039738761783e-18,
    5.8927927788323458e-20,    1.6823568615029297e-21,    4.5437635720952915e-23,    1.1643398276719230e-24,
    2.8382441747915573e-26,    6.5971600738432344e-28,    1.4653430389658362e-29,    3.1163984969754033e-31,
    6.3574538739592886e-33,    1.2460987353527267e-34,    2.3503236603850439e-36,    4.2719534865258739e-38,
    7.4924279581425406e-40,    1.2695503846262243e-41,    2.0806888478172017e-43,    3.3018745773081178e-45,
    5.0786411308683170e-47,    7.5784224371345086e-49,    1.0980979881941413e-50,    1.5463185949362284e-52,
    2.1178578941976985e-54,    2.8233217190627814e-56,    3.6660490061359245e-58,    4.6398433811945876e-60,
    5.7273499219965226e-62,    6.8994437740520893e-64,    8.1158953329025801e-66,    9.3274108489448798e-68,
    1.0478943205347992e-69,    1.1513961851685842e-71,    1.2379203138762471e-73,    1.3029326367460560e-75,
    1.3430896749356533e-77,    1.3565205847222566e-79,    1.3429606061892012e-81,    1.3037247116316297e-83,
    1.2415320862309549e-85,    1.1602108912718582e-87,    1.0643257006577266e-89,    9.5877554936236476e-92,
    8.4840865502564196e-94,    7.3769183388646073e-96,    6.3046056212495859e-98,   5.2976200259044475e-100
},
// x = 1.0100
{
    4.2937848263360312e-01,    1.3559331673906708e-01,    2.6626056675978036e-02,    3.7811549767991924e-03,
    4.2003208430053669e-04,    3.8294819665699392e-05,    2.9597909117314500e-06,    1.9850099984904752e-07,
    1.1756260507971988e-08,    6.2334773466746798e-10,    2.9916984524576337e-11,    1.3114227703359125e-12,
    5.2901635738724127e-14,    1.9763411595733969e-15,    6.8753255078868096e-17,    2.2377959227283717e-18,
    6.8429727799260337e-20,    1.9731332475875419e-21,    5.3823179890085963e-23,    1.3929941883542389e-24,
    3.4295389278726475e-26,    8.0511862375783688e-28,    1.8061732600738508e-29,    3.8796332889407233e-31,
    7.9935333905743046e-33,    1.5824360878438881e-34,    3.0145311645694182e-36,    5.5339752528324412e-38,
    9.8028431412875171e-40,    1.6776384424584285e-41,    2.7769922729331123e-43,    4.4508949882448022e-45,
    6.9143868140470947e-47,    1.0420881154844723e-48,    1.5250578629690967e-50,    2.1690216895138878e-52,
    3.0004162765089408e-54,    4.0398471039376904e-56,    5.2981318887542675e-58,    6.7724815473393642e-60,
    8.4434182545629529e-62,    1.0273035809609796e-63,    1.2205100321306210e-65,    1.4167273223898536e-67,
    1.6075442846988644e-69,    1.7839824926987075e-71,    1.9372197032640542e-73,    2.0593424764760164e-75,
    2.1440360839489107e-77,    2.1871268349847982e-79,    2.1869124103988924e-81,    2.1442458590520191e-83,
    2.0623730623936853e-85,    1.9465557343546264e-87,    1.8035372096160853e-89,    1.6409229370358491e-91,
    1.4665505193452930e-93,    1.2879162972823021e-95,    1.1117095455332718e-97,   9.4348496235228963e-100
},
// x = 1.0200
{
    4.2645651427528142e-01,    1.3584102074895255e-01,    2.6924100307773917e-02,    3.8601368873148792e-03,
    4.3296480659351462e-04,    3.9859182077987608e-05,    3.1108822230599214e-06,    2.1068315663568797e-07,
    1.2600507829215126e-08,    6.7469281543598612e-10,    3.2700482858522093e-11,    1.4475801134722596e-12,
    5.8970495912298457e-14,    2.2248215041610068e-15,    7.8161978624738267e-17,    2.5691707125669479e-18,
    7.9339321232975190e-20,    2.3103197353961597e-21,    6.3644038008964195e-23,    1.6634546788310814e-24,
    4.1359112481710417e-26,    9.8054966428969275e-28,    2.2214869497858353e-29,    4.8189232535404932e-31,
    1.0027055073723807e-32,    2.0046396927831555e-34,    3.8566098080277478e-36,    7.1498861187488416e-38,
    1.2790577219988669e-39,    2.2106134610598087e-41,    3.6954354270325206e-43,    5.9815665800684886e-45,
    9.3842169609937512e-47,    1.4283206116892889e-48,    2.1109822503800292e-50,    3.0320702044609516e-52,
    4.2357864740695755e-54,    5.7596321507535739e-56,    7.6283380959708271e-58,    9.8476457951649269e-60,
    1.2398820685794870e-61,    1.5234858572409560e-63,    1.8279258929715798e-65,    2.1427976463951753e-67,
    2.4554751687193943e-69,    2.7519525766386564e-71,    3.0179152216242728e-73,    3.2399221652577554e-75,
    3.4065594302863852e-77,    3.5094233835189525e-79,    3.5438157681385179e-81,    3.5090722492847225e-83,
    3.4084974316199082e-85,    3.2489320540561087e-87,    3.0400236497001769e-89,    2.7933035690062662e-91,
    2.5211872277065192e-93,    2.2360105688491292e-95,    1.9491969034504581e-97,    1.6706195007530711e-99
},
// x = 1.0300
{
    4.2356603403164039e-01,    1.3607859312210674e-01,    2.7220617171135379e-02,    3.9396748156246926e-03,
    4.4612813290922498e-04,    4.1467829039234053e-05,    3.2678227814825718e-06,    2.2346383605597428e-07,
    1.3495071929547668e-08,    7.2963916052736058e-10,    3.5708774188590861e-11,    1.5961916337603987e-12,
    6.5660036659623957e-14,    2.5014235753513745e-15,    8.8739053325793597e-17,    2.9453749164094011e-18,
    9.1847103763087342e-20,    2.7007181745639321e-21,    7.5126957516827678e-23,    1.9828074545893813e-24,
    4.9782051792852215e-26,    1.1918006991476213e-27,    2.7265323582560799e-29,    5.9724165710211311e-31,
    1.2548938095431276e-32,    2.5333994081933391e-34,    4.9216082603717179e-36,    9.2137130303664172e-38,
    1.6644092051854527e-39,    2.9048074954685672e-41,    4.9034855782693661e-43,    8.0147338670835338e-45,
    1.2697178202123354e-46,    1.9515065202546117e-48,    2.9124878457792801e-50,    4.2242939276590948e-52,
    5.9591506281731240e-54,    8.1823973056580176e-56,    1.0943376454421722e-57,    1.4265581776999955e-59,
    1.8137322567116121e-61,    2.2504378115454862e-63,    2.7266100020198765e-65,    3.2276133490813609e-67,
    3.7348386377589184e-69,    4.2268150326551572e-71,    4.6807492766491395e-73,    5.0743335059346581e-75,
    5.3876146668677398e-77,    5.6047014659361627e-79,    5.7151028832182898e-81,    5.7145423667479272e-83,
    5.6051647036502577e-85,    5.3951351222520706e-87,    5.0977082528617401e-89,    4.7299058285545749e-91,
    4.3109781749165014e-93,    3.8608325782751439e-95,    3.3985936658416671e-97,    2.9414231907724114e-99
},
// x = 1.0400
{
    4.2070662875068149e-01,    1.3630622821981703e-01,    2.7515585808901403e-02,    4.0197579847126635e-03,
    4.5952245025876264e-04,    4.3121395934913744e-05,    3.4307624856365326e-06,    2.3686486445708495e-07,
    1.4442325197807830e-08,    7.8839487753394922e-10,    3.8957242860679971e-11,    1.7582428471419678e-12,
    7.3026048886438768e-14,    2.8089796794981320e-15,    1.0061489946802271e-16,    3.3719058705757185e-18,
    1.0616678740037638e-19,    3.1520396022346370e-21,    8.8531555941556751e-23,    2.3592466215596845e-24,
    5.9807633068606237e-26,    1.4457025088616349e-27,    3.3394721442421762e-29,    7.3860029531052778e-31,
    1.5669635119055391e-32,    3.1940989349484930e-34,    6.2653419041246947e-36,    1.1843107310999921e-37,
    2.1601531151269142e-39,    3.8065838620175708e-41,    6.4880879002425745e-43,    1.0707666829828816e-44,
    1.7128014039625068e-46,    2.6580550631489218e-48,    4.0054597233645393e-50,    5.8659283975566263e-52,
    8.3552887876760545e-54,    1.1583831301603196e-55,    1.5642913275990579e-57,    2.0589722640193265e-59,
    2.6431935360774549e-61,    3.3114457438576189e-63,    4.0510588339526512e-65,    4.8419687708714599e-67,
    5.6572755056380109e-69,    6.4646316200826070e-71,    7.2283806573107350e-73,    7.9122460281411907e-75,
    8.4822775868477620e-77,    8.9097117741496166e-79,    9.1734030149074349e-81,    9.2615390297144651e-83,
    9.1724509407744716e-85,    8.9144527869567633e-87,    8.5047727087820905e-89,    7.9677486784505946e-91,
    7.3325389732627763e-93,    6.6306327083885278e-95,    5.8934379469424333e-97,    5.1501807829400669e-99
},
// x = 1.0500
{
    4.1787789130810388e-01,    1.3652411696989250e-01,    2.7808985679839717e-02,    4.1003756373228254e-03,
    4.7314809768744641e-04,    4.4820514287571928e-05,    3.5998527700261306e-06,    2.5090856343891203e-07,
    1.5444720898816413e-08,    8.5117745807481003e-10,    4.2462133653181631e-11,    1.9347850111774586e-12,
    8.1128646437291432e-14,    3.1505721943308892e-15,    1.1393286878280134e-16,    3.8548660439967478e-18,
    1.2253796004021807e-19,    3.6730141613216925e-21,    1.0415466282834618e-22,    2.8022330847327746e-24,
    7.1719681128873066e-26,    1.7502977958262725e-27,    4.0819014082825398e-29,    9.1147799089908012e-31,
    1.9523061628487031e-32,    4.0178156968507089e-34,    7.9568152121693050e-36,    1.5184945176290516e-37,
    2.7963103028407848e-39,    4.9749608691297027e-41,    8.5610018695136786e-43,    1.4264496979453056e-44,
    2.3036818418442071e-46,    3.6093870841543419e-48,    5.4913100286469736e-50,    8.1192247204648464e-52,
    1.1675986183654558e-53,    1.6343261145280354e-55,    2.2282227400219396e-57,    2.9610517861343647e-59,
    3.8377712084503363e-61,    4.8542532982016485e-63,    5.9955368141527515e-65,    7.2349626846750745e-67,
    8.5344685054946833e-69,    9.8461811291212094e-71,    1.1115269229708694e-72,    1.2283828077982750e-74,
    1.3295401052467661e-76,    1.4099628459658733e-78,    1.4656476465714600e-80,    1.4939545427799933e-82,
    1.4938079680020287e-84,    1.4657477796522112e-86,    1.4118302310777394e-88,    1.3353977151115377e-90,
    1.2407510268591294e-92,    1.1327668218818828e-94,    1.0165051230562632e-96,    8.9684762047113192e-99
},
// x = 1.0600
{
    4.1507942051346375e-01,    1.3673244692666464e-01,    2.8100797136110765e-02,    4.1815170393489790e-03,
    4.8700536682520405e-04,    4.6565811587817180e-05,    3.7752465742708553e-06,    2.6561775241990299e-07,
    1.6504794743925921e-08,    9.1821407392137750e-10,    4.6240588731416559e-11,    2.1269386763322312e-12,
    9.0032546849272900e-14,    3.5295525286256732e-15,    1.2885036541139959e-16,    4.4010220873826825e-18,
    1.4122889360420068e-19,    4.2735129122819821e-21,    1.2233517979570798e-22,    3.3226741299114625e-24,
    8.5848600852273356e-26,    2.1150403423982857e-27,    4.9794509701358400e-29,    1.1224777179775045e-30,
    2.4271262733190666e-32,    5.0425200925661906e-34,    1.0081155749561969e-35,    1.9422177852068535e-37,
    3.6106376396151959e-39,    6.4848843264482987e-41,    1.1265522524640659e-42,    1.8949490607489817e-44,
    3.0894277363297645e-46,    4.8865639582442324e-48,    7.5051743954148497e-50,    1.1202480213600831e-51,
    1.6263278142974311e-53,    2.2980962694556909e-55,    3.1630289010320521e-57,    4.2433191354041815e-59,
    5.5520577799173155e-61,    7.0894509000677303e-63,    8.8396193975576163e-65,    1.0768548564321828e-66,
    1.2823683500821855e-68,    1.4935494808319827e-70,    1.7021087737065289e-72,    1.8989635361275100e-74,
    2.0749132810039629e-76,    2.2213747294034158e-78,    2.3310920265968814e-80,    2.3987387242428205e-82,
    2.4213417225504562e-84,    2.3984813151819823e-86,    2.3322514977562291e-88,    2.2269956473039026e-90,
    2.0888595686023784e-92,    1.9252228438470803e-94,    1.7440782488587706e-96,    1.5534274824985016e-98
},
// x = 1.0700
{
    4.1231082101786015e-01,    1.3693140232993403e-01,    2.8391001401222710e-02,    4.2631714830988392e-03,
    5.0109450244526367e-04,    4.8357911129335714e-05,    3.9570983119057248e-06,    2.8101575104185561e-07,
    1.7625166459151283e-08,    9.8974187776988223e-10,    5.0310685667392040e-11,    2.3358973818139248e-12,
    9.9807366718890433e-14,    3.9495612791019607e-15,    1.4554004996242829e-16,    5.0178689053067954e-18,
    1.6253962176413789e-19,    4.9646826745052829e-21,    1.4345951396507654e-22,    3.9331261802075907e-24,
    1.0257842479990481e-25,    2.5510243233875916e-27,    6.0624888066993567e-29,    1.3794981401998019e-30,
    3.0109946067665184e-32,    6.3145111980819499e-34,    1.2743160919433324e-35,    2.4782192972212275e-37,
    4.6505131559832099e-39,    8.4313001521818641e-41,    1.4784926305951288e-42,    2.5103889759848763e-44,
    4.1314006197729977e-46,    6.5962752464586256e-48,    1.0226617101175175e-49,    1.5408541120216378e-51,
    2.2580400970215957e-53,    3.2208321250991574e-55,    4.4748597078598266e-57,    6.0598019068473510e-59,
    8.0035607533197994e-61,    1.0316169510560416e-62,    1.2984226981713618e-64,    1.5966746196898833e-66,
    1.9193262912022104e-68,    2.2564850299821091e-70,    2.5958346041750755e-72,    2.9233663367939522e-74,
    3.2243599352872660e-76,    3.4845153353337584e-78,    3.6911100649885345e-80,    3.8340483056378108e-82,
    3.9066797269792716e-84,    3.9062963011446800e-86,    3.8342583458913458e-88,    3.6957495630407789e-90,
    3.4992068081708641e-92,    3.2555067341712950e-94,    2.9770135124900663e-96,    2.6765976627020038e-98
},
// x = 1.0800
{
    4.0957170322311920e-01,    1.3712116416288156e-01,    2.8679580548448363e-02,    4.3453282904358731e-03,
    5.1541570303055453e-04,    5.0197431847922054e-05,    4.1455638387561176e-06,    2.9712638141322774e-07,
    1.8808541350175930e-08,    1.0660083086068480e-09,    5.4691476536939072e-11,    2.5629314996988032e-12,
    1.1052793224234922e-13,    4.4145496444237142e-15,    1.6419113175632779e-16,    5.7136991149134370e-18,
    1.8680530973845041e-19,    5.7590951274489196e-21,    1.6796764519852515e-22,    4.6480234253727041e-24,
    1.2235483784422530e-25,    3.0712479530397019e-27,    7.3669343570472761e-29,    1.6919709366712673e-30,
    3.7275030149089701e-32,    7.8901324034575130e-34,    1.6071577207005263e-35,    3.1547000197434942e-37,
    5.9752546078417661e-39,    1.0934211603410105e-40,    1.9353059788398319e-42,    3.3167229258682798e-44,
    5.5093875000113794e-46,    8.8785641698765964e-48,    1.3893575879372278e-49,    2.1129136109750684e-51,
    3.1252887028545432e-53,    4.4995071227338913e-55,    6.3097873378693236e-57,    8.6244666039254726e-59,
    1.1497294423590513e-60,    1.4957862325861949e-62,    1.9002287151303803e-64,    2.3585493728435683e-66,
    2.8616478417299228e-68,    3.3957734390843797e-70,    3.9429583475243199e-72,    4.4819538288161685e-74,
    4.9896106565215436e-76,    5.4425768108171303e-78,    5.8191327250784125e-80,    6.1009569750172113e-82,
    6.2746186755555202e-84,    6.3326267156636043e-86,    6.2739244435314885e-88,    6.1037913645478631e-90,
    5.8331885730088060e-92,    5.4776501087041307e-94,    5.0558688866784118e-96,    4.5881480251826762e-98
},
// x = 1.0900
{
    4.0686168319245414e-01,    1.3730191020896396e-01,    2.8966517479709467e-02,    4.4279768158020927e-03,
    5.2996912134721801e-04,    5.2084988164516079e-05,    4.3408004209088293e-06,    3.1397397019062482e-07,
    2.0057711863530880e-08,    1.1472714016112212e-09,    5.9403028105920219e-11,    2.8093922311067700e-12,
    1.2227460550215216e-13,    4.9288021583273060e-15,    1.8501075459501041e-16,    6.4976782765709553e-18,
    2.1439994024014858e-19,    6.6709115022339179e-21,    1.9635989318798849e-22,    5.4839353021989811e-24,
    1.4569430196995789e-25,    3.6909161730444130e-27,    8.9352024169081861e-29,    2.0711385410339573e-30,
    4.6050381379110337e-32,    9.8378178954739565e-34,    2.0224253498251024e-35,    4.0065614619940556e-37,
    7.6589652534962570e-39,    1.4144945694087303e-40,    2.5267585315700573e-42,    4.3704256677092871e-44,
    7.3268693074438498e-46,    1.1916770317653156e-47,    1.8820479689699823e-49,    2.8886792193261243e-51,
    4.3122977590236991e-53,    6.2659172309434346e-55,    8.8682030115437302e-57,    1.2233604033706185e-58,
    1.6459595915398728e-60,    2.1611974924568386e-62,    2.7709728374778205e-64,    3.4711460716182416e-66,
    4.2505554399601752e-68,    5.0906100855670719e-70,    5.9656118399795841e-72,    6.8438723586671472e-74,
    7.6895857243446387e-76,    8.4653058063686803e-78,    9.1347819326836755e-80,    9.6658434068600526e-82,
    1.0033005131719715e-83,    1.0219496988567366e-85,    1.0218493645933126e-87,    1.0033426342406588e-89,
    9.6773761082418042e-92,    9.1716597242066889e-94,    8.5438086778019632e-96,    7.8251959357111486e-98
},
// x = 1.1000
{
    4.0418038256257549e-01,    1.3747381510781043e-01,    2.9251795904910657e-02,    4.5111064491249629e-03,
    5.4475486502466664e-04,    5.4021189832236603e-05,    4.5429667023009399e-06,    3.3158335049823048e-07,
    2.1375559143250941e-08,    1.2338001025340807e-09,    6.4466463116831537e-11,    3.0767157582052425e-12,
    1.3513362708556299e-13,    5.4969608060854686e-15,    2.0822548164686856e-16,    7.3799263043898580e-18,
    2.4574034133626408e-19,    7.7160643019355206e-21,    2.2920445649751774e-22,    6.4598561099237847e-24,
    1.7319441840181832e-25,    4.4277876741924607e-27,    1.0817295611297935e-28,    2.5303786614907738e-30,
    5.6776940191735277e-32,    1.2240529498389811e-33,    2.5394335564318816e-35,    5.0769083089301992e-37,
    9.7940196678106283e-39,    1.8253901556050952e-40,    3.2906513832978978e-42,    5.7438850353291450e-44,
    9.7177215484113366e-46,    1.5950294497224151e-47,    2.5421728293525971e-49,    3.9376585578592105e-51,
    5.9321473438135301e-53,    8.6986605568261135e-55,    1.2424187048185610e-56,    1.7296230961904952e-58,
    2.3484482720529827e-60,    3.1118676966249443e-62,    4.0264641779803153e-64,    5.0901364015606267e-66,
    6.2902401872951000e-68,    7.6025002186318878e-70,    8.9909733599407352e-72,    1.0409234092668939e-73,
    1.1802799084669682e-75,    1.3112634602244194e-77,    1.4279426499090776e-79,    1.5248167153851334e-81,
    1.5972550299924589e-83,    1.6418675596763898e-85,    1.6567648906093674e-87,    1.6416807256258371e-89,
    1.5979475056971269e-91,    1.5283340969715457e-93,    1.4367704505623787e-95,    1.3279955519818511e-97
},
// x = 1.1100
{
    4.0152742845722611e-01,    1.3763705041013766e-01,    2.9535400321718856e-02,    4.5947066186115236e-03,
    5.5977299714164263e-04,    5.6006641787396494e-05,    4.7522226719476704e-06,    3.4997986368503641e-07,
    2.2765054582316579e-08,    1.3258745864941642e-09,    6.9904002686739984e-11,    3.3664275558403778e-12,
    1.4919747563308384e-13,    6.1240505907423178e-15,    2.3408288529765982e-16,    8.3716054881430622e-18,
    2.8129058375450806e-19,    8.9124576036304265e-21,    2.6714580219757829e-22,    7.5975303778189363e-24,
    2.0554567961564419e-25,    5.3025721960441008e-27,    1.3072066979156024e-28,    3.0855828727449098e-30,
    6.9863468127089370e-32,    1.5198653328556493e-33,    3.1817698586586229e-35,    6.4188682945952372e-37,
    1.2495325078309373e-38,    2.3500111390744543e-40,    4.2748796253424187e-42,    7.5296658898071588e-44,
    1.2854716680834831e-45,    2.1290947237010263e-47,    3.4242044412454283e-49,    5.3520617008533309e-51,
    8.1362358248989821e-53,    1.2039060999416405e-54,    1.7351496176276431e-56,    2.4375257832122564e-58,
    3.3397011621765243e-60,    4.4655675911714598e-62,    5.8305346273144995e-64,    7.4377746215736330e-66,
    9.2749148651864817e-68,    1.1311711087934329e-69,    1.3499192223066349e-71,    1.5770632030653417e-73,
    1.8044490836882561e-75,    2.0229209452406265e-77,    2.2229468985889884e-79,    2.3953301100705306e-81,
    2.5319284786577396e-83,    2.6263025915557852e-85,    2.6742193279214255e-87,    2.6739566898184807e-89,
    2.6263810294526017e-91,    2.5347965502044243e-93,    2.4045942099113135e-95,    2.2427489250011908e-97
},
// x = 1.1200
{
    3.9890245340211927e-01,    1.3779178463170277e-01,    2.9817315995772666e-02,    4.6787667934327587e-03,
    5.7502353681774100e-04,    5.8041944004485451e-05,    4.9687296308302690e-06,    3.6918936091983468e-07,
    2.4229261368197346e-08,    1.4237865811249828e-09,    7.5739009827113137e-11,    3.6801468666107702e-12,
    1.6456524492768424e-13,    6.8155066178203544e-15,    2.6285324808641625e-16,    9.4850155827900716e-18,
    3.2156677704827060e-19,    1.0280187617807735e-20,    3.1091399177902094e-22,    8.9218179650791210e-24,
    2.4354478073031604e-25,    6.3393847729046235e-27,    1.5768677058495199e-28,    3.7555976163780041e-30,
    8.5799183375045090e-32,    1.8833437197808497e-33,    3.9781849309986593e-35,    8.0977921898340697e-37,
    1.5905520622872327e-38,    3.0183014079813045e-40,    5.5399915422388820e-42,    9.8458586905515513e-44,
    1.6960288036343482e-45,    2.8343837450669500e-47,    4.5995613894059695e-49,    7.2538909362988204e-51,
    1.1126708593985542e-52,    1.6612277189519679e-54,    2.4158331464764503e-56,    3.4243074940899583e-58,
    4.7339618085445634e-60,    6.3868612469385762e-62,    8.4142018829892753e-64,    1.0830321699294317e-65,
    1.3627058588695888e-67,    1.6769279199080525e-69,    2.0192394429621226e-71,    2.3802530557600404e-73,
    2.7479745137325114e-75,    3.1084297384875108e-77,    3.4465560515832421e-79,    3.7472768476423116e-81,
    3.9966491219032443e-83,    4.1829585801980730e-85,    4.2976403544090371e-87,    4.3359242499357460e-89,
    4.2971384771920837e-91,    4.1846491158767687e-93,    4.0054571147483485e-95,    3.7695136286994312e-97
},
// x = 1.1300
{
    3.9630509524125440e-01,    1.3793818330630800e-01,    3.0097528941321533e-02,    4.7632764863011663e-03,
    5.9050645980989521e-04,    6.0127691355101500e-05,    5.1926501584645816e-06,    3.8923820462402157e-07,
    2.5771336021818637e-08,    1.5278396940066807e-09,    8.1996034095700828e-11,    4.0195913432132458e-12,
    1.8134303914805710e-13,    7.5772027695050399e-15,    2.9483138112258491e-16,    1.0733696447549348e-17,
    3.6714229600996518e-19,    1.1841785311424908e-20,    3.6133503662749436e-22,    1.0461103268012453e-23,
    2.8810967839231106e-25,    7.5662643976220804e-27,    1.8988273058572138e-28,    4.5627371287570585e-30,
    1.0516859317110252e-31,    2.3291062933228096e-33,    4.9636569807573455e-35,    1.0193908437011592e-36,
    2.0201308844458452e-38,    3.8676924617601037e-40,    7.1623627021578735e-42,    1.2842774093918072e-43,
    2.2320126283261882e-45,    3.7634000063773312e-47,    6.1616436794771053e-49,    9.8041448423382889e-51,
    1.5172741141109654e-52,    2.2855210904509651e-54,    3.3533726297971262e-56,    4.7956395204008589e-58,
    6.6889429564043887e-60,    9.1049855346639724e-62,    1.2102185538627421e-63,    1.5716339265319972e-65,
    1.9951308147003029e-67,    2.4770964414806748e-69,    3.0093704772446880e-71,    3.5790711115248496e-73,
    4.1688790963795531e-75,    4.7578100434896069e-77,    5.3224411874700244e-79,    5.8384937923003124e-81,
    6.2826174496302044e-83,    6.6341875012363396e-85,    6.8769182233973710e-87,    7.0001141056357652e-89,
    6.9994264011693418e-91,    6.8770446641939415e-93,    6.6413229940375802e-95,    6.3059072140611915e-97
},
// x = 1.1400
{
    3.9373499705458881e-01,    1.3807640903787705e-01,    3.0376025902280605e-02,    4.8482252559443936e-03,
    6.0622169911332105e-04,    6.2264473470811913e-05,    5.4241480791707185e-06,    4.1015326974234905e-07,
    2.7394529929280068e-08,    1.6383497443133911e-09,    8.8700857390223564e-11,    4.3865818619027929e-12,
    1.9964438692166797e-13,    8.4154820416556278e-15,    3.3033856666620816e-16,    1.2132538743314664e-17,
    4.1865347080948382e-19,    1.3622483040127665e-20,    4.1934238451187338e-22,    1.2247753338795913e-23,
    3.4029660569749526e-25,    9.0157654649420188e-27,    2.2825921284002872e-28,    5.5333791514194975e-30,
    1.2866887799662702e-31,    2.8747463013201901e-33,    6.1806621776456854e-35,    1.2805520638623096e-36,
    2.5601151520882907e-38,    4.9448781816444161e-40,    9.2381246530419375e-42,    1.6711304553085220e-43,
    2.9300316001028751e-45,    4.9840261828616312e-47,    8.2322947367122422e-49,    1.3214737989601401e-50,
    2.0631817850586701e-52,    3.1353308923614152e-54,    4.6409271563572843e-56,    6.6956745342458440e-58,
    9.4217209919935081e-60,    1.2938294083545921e-61,    1.7349487100788110e-63,    2.2729996249526144e-65,
    2.9110156234264412e-67,    3.6462069863879480e-69,    4.4688867816828786e-71,    5.3619080454635949e-73,
    6.3007712988370902e-75,    7.2544908466853699e-77,    8.1872145242781660e-79,    9.0604885997578898e-81,
    9.8359642674458149e-83,    1.0478271104223655e-84,    1.0957748850123201e-86,    1.1252738427063563e-88,
    1.1351185086823344e-90,    1.1251392474397216e-92,    1.0961872121254817e-94,    1.0500341620113882e-96
},
// x = 1.1500
{
    3.9119180707704182e-01,    1.3820662155161742e-01,    3.0652794333691859e-02,    4.9336027094777852e-03,
    6.2216914556644974e-04,    6.4452874609922123e-05,    5.6633884280639101e-06,    4.3196194485186593e-07,
    2.9102190865661815e-08,    1.7556450986044149e-09,    9.5880540893215188e-11,    4.7830475109194700e-12,
    2.1959067482580456e-13,    9.3371886193722600e-15,    3.6972463184721102e-16,    1.3697903225202548e-17,
    4.7680577653354592e-19,    1.5650507283404253e-20,    4.8598964732974854e-22,    1.4318630186256274e-23,
    4.0131927410126127e-25,    1.0725631341738029e-26,    2.7392827975210657e-28,    6.6986557860428894e-30,
    1.5713023557101953e-31,    3.5414008429258490e-33,    7.6806884742848951e-35,    1.6052852119401303e-36,
    3.2374606581872433e-38,    6.3079875167095618e-40,    1.1888017884255582e-41,    2.1693346261656535e-43,
    3.8368887393894361e-45,    6.5838216292940203e-47,    1.0970075398753071e-48,    1.7763900421819460e-50,
    2.7977470210642130e-52,    4.2888968031407721e-54,    6.4041032023251982e-56,    9.3205027955203211e-58,
    1.3230210547150946e-59,    1.8327579618360444e-61,    2.4791704212442324e-63,    3.2765048161128254e-65,
    4.2329948931249632e-67,    5.3485534654993055e-69,    6.6128129507878871e-71,    8.0038355779535570e-73,
    9.4877769581513992e-75,    1.1019697736863506e-76,    1.2545585906390231e-78,    1.4005495120865790e-80,
    1.5337546582682183e-82,    1.6482410255381098e-84,    1.7387798493305159e-86,    1.8012486831738354e-88,
    1.8329426707586613e-90,    1.8327625445396605e-92,    1.8012621900117597e-94,    1.7405557117573473e-96
},
// x = 1.1600
{
    3.8867517861880974e-01,    1.3832897774428535e-01,    3.0927822383589054e-02,    5.0193985046785823e-03,
    6.3834864845939215e-04,    6.6693473528129326e-05,    5.9105374167863979e-06,    4.5469213310935435e-07,
    3.0897764510262138e-08,    1.8800670106852007e-09,    1.0356347317694527e-10,    5.2110307577434666e-12,
    2.4131160099720296e-13,    1.0349701769267558e-14,    4.1337016080285487e-16,    1.5447749196179853e-17,
    5.4238056011748421e-19,    1.7957399734174753e-20,    5.6246468979789518e-22,    1.6715663034997250e-23,
    4.7257051781500322e-25,    1.2739560501449921e-26,    3.2818888194574974e-28,    8.0952535544607332e-30,
    1.9153965294407444e-31,    4.3544214782772202e-33,    9.5260364644964754e-35,    2.0082660503802599e-36,
    4.0853635555878092e-38,    8.0292392762990838e-40,    1.5263374701724558e-41,    2.8094762542669064e-43,
    5.0122863094607091e-45,    8.6754637944032442e-47,    1.4580833829726093e-48,    2.3816025834110866e-50,
    3.7835347099700800e-52,    5.8505041267159283e-54,    8.8117976581669439e-56,    1.2936122430440918e-57,
    1.8522098494185316e-59,    2.5881370873809691e-61,    3.5314034421754081e-63,    4.7077199381637052e-65,
    6.1348855262906620e-67,    7.8190506138569243e-69,    9.7513093738856305e-71,    1.1905123796469605e-72,
    1.4235060872412884e-74,    1.6677221243545701e-76,    1.9151559385286217e-78,    2.1566063568417131e-80,
    2.3822512606653551e-82,    2.5823296358965106e-84,    2.7478617460633754e-86,    2.8713313006451352e-88,
    2.9472561846262670e-90,    2.9725872471831586e-92,    2.9468957399774109e-94,    2.8723358525300986e-96
},
// x = 1.1700
{
    3.8618476998696882e-01,    1.3844363173357069e-01,    3.1201098875249043e-02,    5.1056023521644520e-03,
    6.5476001614548447e-04,    6.8986843353035173e-05,    6.1657623989997738e-06,    4.7837225303767802e-07,
    3.2784795952617242e-08,    2.0119699654617986e-09,    1.1177941947692470e-10,    5.6726927990460832e-12,
    2.6494564952311736e-13,    1.1460971629046030e-14,    4.6168885282426544e-16,    1.7401772718084706e-17,
    6.1624234509816765e-19,    2.0578369162012113e-20,    6.5010520891673166e-22,    1.9486486867610359e-23,
    5.5564666305299640e-25,    1.5108076864321259e-26,    3.9255607366782460e-28,    9.7663386357114997e-30,
    2.3306864351617131e-31,    5.3441636667193453e-33,    1.1791958225113185e-34,    2.5073767108061937e-36,
    5.1446273062894962e-38,    1.0198179984916876e-39,    1.9553480835506889e-41,    3.6301476013170952e-43,
    6.5322130302072694e-45,    1.1403623130591003e-46,    1.9331178175446587e-48,    3.1847194301949460e-50,
    5.1030007593226692e-52,    7.9587950643862241e-54,    1.2090513827857808e-55,    1.7902375054430912e-57,
    2.5853723018975836e-59,    3.6437336014783349e-61,    5.0145640440683753e-63,    6.7425322353130456e-65,
    8.8622806636686855e-67,    1.1392518385744989e-68,    1.4330303114755776e-70,    1.7646278621922188e-72,
    2.1281653317689089e-74,    2.5147602483323180e-76,    2.9127548771567456e-78,    3.3082446196450687e-80,
    3.6858806749757357e-82,    4.0298825658688743e-84,    4.3251644608097326e-86,    4.5584596921063267e-88,
    4.7193240116801125e-90,    4.8009108194050390e-92,    4.8004388883426623e-94,    4.7193106237964498e-96
},
// x = 1.1800
{
    3.8372024440834646e-01,    1.3855073490661413e-01,    3.1472613289835867e-02,    5.1922040174789807e-03,
    6.7140301665549555e-04,    7.1333551462491722e-05,    6.4292318356570364e-06,    5.0303123915151198e-07,
    3.4766931188662716e-08,    2.1517220267102297e-09,    1.2055957214206492e-10,    6.1703190972088670e-12,
    2.9064058629892123e-13,    1.2679556977485446e-14,    5.1513003442370461e-16,    1.9581555207951598e-17,
    6.9934675717996988e-19,    2.3552676647353175e-20,    7.5041594491777386e-22,    2.2685154168743740e-23,
    6.5237493387041255e-25,    1.7889517314155178e-26,    4.6879445628288662e-28,    1.1762625401038679e-29,
    2.8310556354912866e-31,    6.5469146765577813e-33,    1.4569193519075420e-34,    3.1243671425470370e-36,
    6.4653122883831394e-38,    1.2925625653506830e-39,    2.4994616297251530e-41,    4.6799405208679348e-43,
    8.4931773633930741e-45,    1.4953630666673432e-46,    2.5565611293411281e-48,    4.2477917126035884e-50,
    6.8645480846147983e-52,    1.0797619520567673e-53,    1.6543215421295010e-55,    2.4704744683811592e-57,
    3.5982177262241079e-59,    5.1145292223873099e-61,    7.0988335845847515e-63,    9.6265722032614752e-65,
    1.2761126885792649e-66,    1.6544673670159585e-68,    2.0988859238200571e-70,    2.6066434421952006e-72,
    3.1705086629227211e-74,    3.7784640279425268e-76,    4.4138524212787043e-78,    5.0559955990431065e-80,
    5.6812719870581292e-82,    6.2645801680307561e-84,    6.7810587769690759e-86,    7.2078930508626734e-88,
    7.5260200503361437e-90,    7.7215520179060347e-92,    7.7867695314087769e-94,    7.7205879733537308e-96
},
// x = 1.1900
{
    3.8128126995363731e-01,    1.3865043596767587e-01,    3.1742355749412546e-02,    5.2791933230866856e-03,
    6.8827737831408053e-04,    7.3734159366751684e-05,    6.7011152600731607e-06,    5.2869854242302913e-07,
    3.6847918606404453e-08,    2.2997051886800547e-09,    1.2993660226912348e-10,    6.7063251072855735e-12,
    3.1855397704944030e-13,    1.4014665070440469e-14,    5.7418133356408496e-16,    2.2010723080414976e-17,
    7.9274911629986957e-19,    2.6924056973209283e-20,    8.6508767606686857e-22,    2.6372927432714026e-23,
    7.6484423842547829e-25,    2.1151150827783730e-26,    5.5895641312474453e-28,    1.4143608780799544e-29,
    3.4329321081280085e-31,    8.0059826253901274e-33,    1.7966972465260063e-34,    3.8856450642152285e-36,
    8.1087230888423841e-38,    1.6348451898852386e-39,    3.1881137811164712e-41,    6.0199116166848706e-43,
    1.1017487538917512e-44,    1.9562382845511751e-46,    3.3728275789580414e-48,    5.6515053348735561e-50,
    9.2103462172029139e-52,    1.4610167574748320e-53,    2.2574095884319567e-55,    3.3996496003638841e-57,
    4.9934953492255210e-59,    7.1579122718878093e-61,    1.0019158920056193e-62,    1.3701861333842969e-64,
    1.8317264062905080e-66,    2.3949338551351048e-68,    3.0639942248620911e-70,    3.8374662943774340e-72,
    4.7071254397059258e-74,    5.6572587764681632e-76,    6.6645760215810627e-78,    7.6988420187981291e-80,
    8.7242547848552488e-82,    9.7014980820132861e-84,    1.0590304399373985e-85,    1.1352288051880619e-87,
    1.1953761912194519e-89,    1.2368243345606306e-91,    1.2578387094348305e-93,    1.2577150285667953e-95
},
// x = 1.2000
{
    3.7886751946274483e-01,    1.3874288098496781e-01,    3.2010317000325181e-02,    5.3665601502800543e-03,
    7.0538279035808828e-04,    7.6189222594393681e-05,    6.9815832428128164e-06,    5.5540413058818715e-07,
    3.9031610460475621e-08,    2.4563157314492840e-09,    1.3994471252862863e-10,    7.2832621982820176e-12,
    3.4885372822324653e-13,    1.5476193631049576e-14,    6.3937152463110822e-16,    2.4715119131125190e-17,
    8.9761374370776729e-19,    3.0741179161562652e-20,    9.9601816219041361e-22,    3.0619160691565778e-23,
    8.9543971452579841e-25,    2.4970445267671405e-26,    6.6542576757247032e-28,    1.6978983703781397e-29,
    4.1557250769862804e-31,    9.7729727540882602e-33,    2.2116564987687879e-34,    4.8232178594447182e-36,
    1.0149798566254775e-37,    2.0635404737011135e-39,    4.0579038909685388e-41,    7.7266245877316202e-43,
    1.4259824094350655e-44,    2.5532032916849089e-46,    4.4390491099080448e-48,    7.5005348778441683e-50,
    1.2326405169129441e-51,    1.9717332904173864e-53,    3.0721040207869664e-55,    4.6654374574913568e-57,
    6.9102793848808378e-59,    9.9887269679348947e-61,    1.4098987258684034e-62,    1.9443263670336934e-64,
    2.6210976897568666e-66,    3.4558046401788076e-68,    4.4583762127027203e-70,    5.6307533420024547e-72,
    6.9648361744104027e-74,    8.4410102071428387e-76,    1.0027535175598079e-77,    1.1681010144549575e-79,
    1.3348015263608863e-81,    1.4967888918172825e-83,    1.6476449047581970e-85,    1.7810332858746244e-87,
    1.8911532416924280e-89,    1.9731661425438157e-91,    2.0235508793804221e-93,    2.0403514790398870e-95
},
// x = 1.2100
{
    3.7647867047132799e-01,    1.3882821343666546e-01,    3.2276488396950992e-02,    5.4542944410010397e-03,
    7.2271890355633871e-04,    7.8699290581991791e-05,    7.2708073564133808e-06,    5.8317848829432561e-07,
    4.1321964334964671e-08,    2.6219645799457122e-09,    1.5061969118902062e-10,    7.9038237726270833e-12,
    3.8171865148104073e-13,    1.7074775084919872e-14,    7.1127355307652457e-16,    2.7722986391599379e-17,
    1.0152240355489817e-18,    3.5058149354397645e-20,    1.1453352151660660e-21,    3.5502279071656589e-23,
    1.0468814515148905e-24,    2.9436499664428379e-26,    7.9096757239380859e-28,    2.0350277881789158e-29,
    5.0223318381626390e-31,    1.1909280965247825e-32,    2.7175470251283121e-34,    5.9758138593429536e-36,
    1.2679981542435978e-37,    2.5994137326158585e-39,    5.1542511685647400e-41,    9.8958978570021769e-43,
    1.8415405388122989e-44,    3.3247146220572766e-46,    5.8285552438487211e-48,    9.9303672205695366e-50,
    1.6455524401535367e-51,    2.6541518746335542e-53,    4.1698066993006478e-55,    6.3852053314173231e-57,
    9.5363168106973882e-59,    1.3899457438676678e-60,    1.9782370596717795e-62,    2.7508211476397398e-64,
    3.7392045973565486e-66,    4.9710495599787684e-68,    6.4666381879776990e-70,    8.2351467359375490e-72,
    1.0271143077217057e-73,    1.2551781685043143e-75,    1.5035167272089386e-77,    1.7660281751458706e-79,
    2.0348720873427236e-81,    2.3008282367216283e-83,    2.5538214260520953e-85,    2.7835705486591666e-87,
    2.9803017467880899e-89,    3.1354546037034050e-91,    3.2423087163055333e-93,    3.2964661458463195e-95
},
// x = 1.2200
{
    3.7411440513853228e-01,    1.3890657425611339e-01,    3.2540861885794567e-02,    5.5423861995794023e-03,
    7.4028533083050332e-04,    8.1264906567496515e-05,    7.5689601399612369e-06,    6.1205261708987580e-07,
    4.3723044593909434e-08,    2.7970776665478024e-09,    1.6199896734530082e-10,    8.5708515877045027e-12,
    4.1733905251105566e-13,    1.8821823133692053e-14,    7.9050774901836221e-16,    3.1065165222780255e-17,
    1.1469933575528755e-18,    3.9935059464743405e-20,    1.3154220889250170e-21,    4.1110866197819848e-23,
    1.2222678471702952e-24,    3.4651661768926135e-26,    9.3878482289801135e-28,    2.4352727606067737e-29,
    6.0597250031965464e-31,    1.4487839130781634e-32,    3.3332354107813092e-34,    7.3902150923377489e-36,
    1.5810658749478921e-37,    3.2679716795652180e-39,    6.5334138766365486e-41,    1.2647412469431921e-42,
    2.3730112741597106e-44,    4.3196153023237920e-46,    7.6352616221668281e-48,    1.3115979835405321e-49,
    2.1913900866854593e-51,    3.5637429431286000e-53,    5.6450689328802461e-55,    8.7156745828074305e-57,
    1.3124407344504301e-58,    1.9287229316234692e-60,    2.7677277216879086e-62,    3.8804356222069539e-64,
    5.3182716327833660e-66,    7.1287370105519458e-68,    9.3501015855347663e-70,    1.2005560049526344e-71,
    1.5097436904825137e-73,    1.8602153620967195e-75,    2.2466714173268918e-77,    2.6607398183378850e-79,
    3.0911168097302891e-81,    3.5240012738907901e-83,    3.9438099000548798e-85,    4.3341236634930422e-87,
    4.6787835219389501e-89,    4.9630300860419651e-91,    5.1745722726244467e-93,    5.3044754147295354e-95
},
// x = 1.2300
{
    3.7177441017588569e-01,    1.3897810187623974e-01,    3.2803429989935233e-02,    5.6308254943902326e-03,
    7.5808164787673552e-04,    8.3886607487293816e-05,    7.8762150635388622e-06,    6.4205803525704029e-07,
    4.6239023818860769e-08,    2.9820962971762303e-09,    1.7412166735801388e-10,    9.2873422833091399e-12,
    4.5591734491606801e-13,    2.0729581763045893e-14,    8.7774523945067206e-16,    3.4775304452816965e-17,
    1.2944768186997830e-18,    4.5438585261790443e-20,    1.5089453968035478e-21,    4.7544870138998817e-23,
    1.4251241035860022e-24,    4.0733352798727581e-26,    1.1125829800781138e-27,    2.9097430016127492e-29,
    7.2996320169731164e-31,    1.7595151763036809e-32,    4.0812859578538967e-34,    9.1228389509074367e-36,
    1.9677277733360262e-37,    4.1004891063446809e-39,    8.2649468288635492e-41,    1.6130367456848500e-42,
    3.0513021850436857e-44,    5.5998123087058198e-46,    9.9791932587996114e-48,    1.7282854767726977e-49,
    2.9112384043651095e-51,    4.7731808446040988e-53,    7.6227960651325280e-55,    1.1865599276398836e-56,
    1.8014079332742214e-58,    2.6689841051864898e-60,    3.8613871748413204e-62,    5.4581362947892571e-64,
    7.5418584404206050e-66,    1.0192118832914600e-67,    1.3477591585834737e-69,    1.7447070395550995e-71,
    2.2120120351822219e-73,    2.7478416749480909e-75,    3.3458949229465570e-77,    3.9950244725850373e-79,
    4.6792557859977021e-81,    5.3782600163330059e-83,    6.0682869988166164e-85,    6.7235071503272171e-87,
    7.3176559449211554e-89,    7.8258304192321621e-91,    8.2262612686206231e-93,    8.5018810523150941e-95
},
// x = 1.2400
{
    3.6945837677734034e-01,    1.3904293227319162e-01,    3.3064185793812359e-02,    5.7196024594329022e-03,
    7.7610739378771506e-04,    8.6564923876908387e-05,    8.1927464925600599e-06,    6.7322677748837602e-07,
    4.8874184232930160e-08,    3.1774775207854504e-09,    1.8702867250793052e-10,    1.0056454118883608e-11,
    4.9766868992814576e-13,    2.2811176783897306e-14,    9.7371156909142700e-16,    3.8890087404250545e-17,
    1.4593839851633439e-18,    5.1642637806709548e-20,    1.7288857804135092e-21,    5.4916939532251278e-23,
    1.6594564152843021e-24,    4.7816123657406700e-26,    1.3166432935856998e-27,    3.4713809538589081e-29,
    8.7793204240328048e-31,    2.1333669427205172e-32,    4.9886434887396235e-34,    1.1241612430970323e-35,
    2.4444266156916749e-37,    5.1352459360089523e-39,    1.0434687196737404e-40,    2.0530408397442987e-42,
    3.9151883591087136e-44,    7.2436117924750044e-46,    1.3013422068951668e-47,    2.2720928164188560e-49,
    3.8583620372554082e-51,    6.3774609913647933e-53,    1.0267602202534280e-54,    1.6112365652817663e-56,
    2.4660204972191340e-58,    3.6833725959033366e-60,    5.3722738061001587e-62,    7.6555139919580604e-64,
    1.0664085166804463e-65,    1.4528641709864432e-67,    1.9368152104020931e-69,    2.5276318487337664e-71,
    3.2306822702952590e-73,    4.0458890602432136e-75,    4.9664961720955947e-77,    5.9782329438177817e-79,
    7.0590424695545610e-81,    8.1794943690198604e-83,    9.3039306161449276e-85,    1.0392306344052258e-86,
    1.1402597755675766e-88,    1.2293570251415709e-90,    1.3027644434561897e-92,    1.3573575145690098e-94
},
// x = 1.2500
{
    3.6716600055044046e-01,    1.3910119900920709e-01,    3.3323122928343309e-02,    5.8087072958336742e-03,
    7.9436207167477284e-04,    8.9300379775313941e-05,    8.5187296520101583e-06,    7.0559139440828458e-07,
    5.1632919110744284e-08,    3.3836945021626928e-09,    2.0076267787132802e-10,    1.0881513924380945e-11,
    5.4282166271863473e-13,    2.5080670008257037e-14,    1.0791905402827638e-15,    4.3449473696909339e-17,
    1.6435925994123648e-18,    5.8629072422883304e-20,    1.9785715716332970e-21,    6.3353902537722693e-23,
    1.9298124563486791e-24,    5.6053969486036230e-26,    1.5559060290309915e-27,    4.1352440920569044e-29,
    1.0542504175952136e-30,    2.5824550836680395e-32,    6.0874345865367213e-34,    1.3828189764682069e-35,
    3.0310900766107909e-37,    6.4190153368704304e-39,    1.3148376079466259e-40,    2.6078100908951997e-42,
    5.0132213544489816e-44,    9.3498658173004806e-46,    1.6932763759240449e-47,    2.9802221997528534e-49,
    5.1016646442539439e-51,    8.5004750854670891e-53,    1.3795929736942036e-54,    2.1823675107936908e-56,
    3.3670687259083267e-58,    5.0697640509569906e-60,    7.4539607288521826e-62,    1.0707553375029918e-63,
    1.5035798313626390e-65,    2.0649757279262819e-67,    2.7750143230569513e-69,    3.6507157191094362e-71,
    4.7037653377884219e-73,    5.9381698562211151e-75,    7.3481166129916195e-77,    8.9163292388443093e-79,
    1.0613201839139961e-80,    1.2396939667404904e-82,    1.4214838413617854e-84,    1.6005707301364860e-86,
    1.7703300058661321e-88,    1.9240483350113187e-90,    2.0553765571963962e-92,    2.1587747549199957e-94
},
// x = 1.2600
{
    3.6489698144859822e-01,    1.3915303327473647e-01,    3.3580235556368423e-02,    5.8981302732741669e-03,
    8.1284514928978992e-04,    9.2093492632815360e-05,    8.8543405906078733e-06,    7.3918495194046692e-07,
    5.4519734173740270e-08,    3.6012368979401645e-09,    2.1536825242029458e-10,    1.1766024268587294e-11,
    5.9161894608213542e-13,    2.7553116163971190e-14,    1.1950282827528226e-15,    4.8496957755433455e-17,
    1.8491633731111000e-18,    6.6488459666514713e-20,    2.2617157079141058e-21,    7.2998402370360665e-23,
    2.2413488315131911e-24,    6.5622932233019447e-26,    1.8360648307811414e-27,    4.9188276550134006e-29,
    1.2640388316973693e-30,    3.1210873014099188e-32,    7.4159065049771955e-34,    1.6980572528437619e-35,
    3.7518298655391978e-37,    8.0088509983828437e-39,    1.6536043916765976e-40,    3.3059275675678991e-42,
    6.4060789264728017e-44,    1.2043118478042106e-45,    2.1984659266013958e-47,    3.9003085582764758e-49,
    6.7300884243059568e-51,    1.1303441944476459e-52,    1.8491716414012014e-54,    2.9485803580069986e-56,
    4.5855979363806427e-58,    6.9597033373115138e-60,    1.0314522786130402e-61,    1.4935212711628714e-63,
    2.1140090534324065e-65,    2.9265405585404962e-67,    3.9642781982569380e-69,    5.2569792228128511e-71,
    6.8275248534214670e-73,    8.6881978203978037e-75,    1.0837088231927309e-76,    1.3255075036902903e-78,
    1.5903843003329809e-80,    1.8725341280612859e-82,    2.1642967776098072e-84,    2.4564584327939046e-86,
    2.7387252946380784e-88,    3.0003363435602487e-90,    3.2307630928149595e-92,    3.4204308921528954e-94
},
// x = 1.2700
{
    3.6265102370445867e-01,    1.3919856392982580e-01,    3.3835518358413502e-02,    5.9878617313477052e-03,
    8.3155605964656200e-04,    9.4944773222464021e-05,    9.1997561449050595e-06,    7.7404103052248094e-07,
    5.7539248970245916e-08,    3.8306112357244871e-09,    2.3089190035201036e-10,    1.2713670848724008e-11,
    6.4431805228428876e-13,    3.0244622655340397e-14,    1.3221375644530800e-15,    5.4079844994480734e-17,
    2.0783559263849143e-18,    7.5320923061688002e-20,    2.5824561809147272e-21,    8.4010704329827750e-23,
    2.5999062192286082e-24,    7.6724024022312013e-26,    2.1636735912706004e-27,    5.8424331621493858e-29,
    1.5132871680712709e-30,    3.7661357326709334e-32,    9.0195258924299515e-34,    2.0816200820552818e-35,
    4.6357732929060305e-37,    9.9742303446512093e-39,    2.0757312176750943e-40,    4.1827636920915729e-42,
    8.1694522021894918e-44,    1.5479980575810041e-45,    2.8482763684409129e-47,    5.0932197137555562e-49,
    8.8581980591828406e-51,    1.4995695008483282e-52,    2.4726604722465163e-54,    3.9740350201496791e-56,
    6.2294002742356610e-58,    9.5295547969330909e-60,    1.4235171848997682e-61,    2.0775751629873570e-63,
    2.9640395567380925e-65,    4.1358385593730989e-67,    5.6468348650220046e-69,    7.5476057745871644e-71,
    9.8802576388710800e-73,    1.2672630084686267e-74,    1.5932421471350818e-76,    1.9641901474280624e-78,
    2.3753937846422188e-80,    2.8190028292123391e-82,    3.2840884914483647e-84,    3.7569875665670661e-86,
    4.2219311022032206e-88,    4.6619216577521649e-90,    5.0597904181273183e-92,    5.3993396558092901e-94
},
// x = 1.2800
{
    3.6042783576433607e-01,    1.3923791754477646e-01,    3.4088966518766188e-02,    6.0778920808456970e-03,
    8.5049420164134894e-04,    9.7854725554968267e-05,    9.5551539033402766e-06,    8.1019372416857415e-07,
    6.0696198239798832e-08,    4.0723412962457850e-09,    2.4738212365047777e-10,    1.3728330105132725e-11,
    7.0119207387408940e-13,    3.3172412279418799e-14,    1.4615023550992964e-15,    6.0249546700352877e-17,
    2.3336459501255469e-18,    8.5237048678617931e-20,    2.9454003195884759e-21,    9.6570690513570770e-23,
    3.0120930031168775e-24,    8.9586507483231610e-26,    2.5462673520641584e-27,    6.9295887226106273e-29,
    1.8089929805680646e-30,    4.5374688735082808e-32,    1.0952262796085005e-33,    2.5475595043335120e-35,
    5.7180509019463731e-37,    1.2399620605180209e-38,    2.6007792461205147e-40,    5.2820103871250919e-42,
    1.0397587073834285e-43,    1.9857011936212935e-45,    3.6823884728207324e-47,    6.6365749136927471e-49,
    1.1633253817865576e-50,    1.9848456912790169e-52,    3.2985955275945022e-54,    5.3431921044799068e-56,
    8.4415061081118671e-58,    1.3015204403654503e-59,    1.9595026171553621e-61,    2.8823366071662366e-63,
    4.1445447206198511e-65,    5.8285611024468984e-67,    8.0206184901001026e-69,    1.0804807335435050e-70,
    1.4255458319019772e-72,    1.8428280528769990e-74,    2.3350975492377272e-76,    2.9014293160423076e-78,
    3.5364648493310288e-80,    4.2299432544858612e-82,    4.9666003425385615e-84,    5.7265029287012688e-86,
    6.4858405542880744e-88,    7.2181436882746570e-90,    7.8958435381319697e-92,    8.4920420090482388e-94
},
// x = 1.2900
{
    3.5822713022370328e-01,    1.3927121844009260e-01,    3.4340575711860138e-02,    6.1682118049760275e-03,
    8.6965894067231916e-04,    1.0082384679705943e-04,    9.9207121702618378e-06,    8.4767763938205722e-07,
    6.3995433261169567e-08,    4.3269684984273042e-09,    2.6488948588369487e-10,    1.4814077064828493e-11,
    7.6253046427220200e-13,    3.6354889010625049e-14,    1.6141826544681212e-15,    6.7061894675051815e-17,
    2.6177436723024530e-18,    9.6358871965764766e-20,    3.3556732313176411e-21,    1.1088005975490708e-22,
    3.4853782623118857e-24,    1.0447157290140101e-25,    2.9924989318507762e-27,    8.2075278699564612e-29,
    2.1593203162516622e-30,    5.4584516361927671e-32,    1.3278089196374079e-33,    3.1126640390775254e-35,
    7.0409677489659381e-37,    1.5387546507166017e-38,    3.2526797969825169e-40,    6.6575446905484892e-42,
    1.3207620634878161e-43,    2.5420450672478410e-45,    4.7509057614666014e-47,    8.6291576570476948e-49,
    1.5244153395807429e-50,    2.6212392651866000e-52,    4.3902261178995045e-54,    7.1669868844725608e-56,
    1.1411266333605420e-57,    1.7731401156673619e-59,    2.6903955906191999e-61,    3.9883502680325290e-63,
    5.7796799261075062e-65,    8.1915622192398144e-67,    1.1360350158552604e-68,    1.5423374925871044e-70,
    2.0507945390420619e-72,    2.6718028415909929e-74,    3.4119549548554879e-76,    4.2725678165714575e-78,
    5.2483774804611498e-80,    6.3265805656937757e-82,    7.4863914925607359e-84,    8.6992469284527962e-86,
    9.9297272926195188e-88,    1.1137186387979133e-89,    1.2277994598278753e-91,    1.3308220984072769e-93
},
// x = 1.3000
{
    3.5604862376371771e-01,    1.3929858872573012e-01,    3.4590342088956075e-02,    6.2588114605153788e-03,
    8.8904960925763777e-04,    1.0385262719327387e-04,    1.0296609929935542e-05,    8.8652789391850439e-07,
    6.7441923183564641e-08,    4.5950522872749265e-09,    2.8346667723878147e-10,    1.5975193417685716e-11,
    8.2863984895726180e-13,    3.9811706969152760e-14,    1.7813195979349523e-15,    7.4577476757664182e-17,
    2.9336137137299403e-18,    1.0882094759644992e-19,    3.8189707485963657e-21,    1.2716475178416624e-22,
    4.0281950713781690e-24,    1.2167645608555580e-25,    3.5122931636275883e-27,    9.7077344600933558e-29,
    2.5737819015281891e-30,    6.5565235640295549e-32,    1.6072725627482262e-33,    3.7969620560160458e-35,
    8.6553904912970286e-37,    1.9062252165807042e-38,    4.0606621999903898e-40,    8.3756889290463262e-42,
    1.6744882217847966e-43,    3.2478201223450543e-45,    6.1169719157656583e-47,    1.1196439771300667e-48,
    1.9932714399124320e-50,    3.4539933308468548e-52,    5.8297980847835796e-54,    9.5908062888826853e-56,
    1.5388784625614646e-57,    2.4097144612515963e-59,    3.6846037775392308e-61,    5.5045286055994774e-63,
    8.0386456111849987e-65,    1.1481487112873520e-66,    1.6046321735356046e-68,    2.1954098880413751e-70,
    2.9417842844590633e-72,    3.8622963730489530e-74,    4.9704652178352156e-76,    6.2724226928497387e-78,
    7.7646888504051358e-80,    9.4323675328056264e-82,    1.1248041728111306e-83,    1.3171607738424061e-85,
    1.5151206876889439e-87,    1.7125300509237796e-89,    1.9025802560120371e-91,    2.0782051294622074e-93
},
// x = 1.3100
{
    3.5389203708876504e-01,    1.3932014833965858e-01,    3.4838262265119629e-02,    6.3496816788974836e-03,
    9.0866550765192854e-04,    1.0694154999110938e-04,    1.0683026810552136e-05,    9.2678011540114955e-07,
    7.1040756340495951e-08,    4.8771705244845025e-09,    3.0316858079708135e-10,    1.7216175828999501e-11,
    8.9984486808253725e-13,    4.3563842691546843e-14,    1.9641408521819800e-15,    8.2861994388506928e-17,
    3.2844964234200687e-18,    1.2277150845935195e-19,    4.3416172541774330e-21,    1.4567761617255267e-22,
    4.6500551472383221e-24,    1.4153904524642021e-25,    4.1170208291064720e-27,    1.1466562062553347e-28,
    3.0634478850405548e-30,    7.8638666001282096e-32,    1.9425674333886427e-33,    4.6243123726302284e-35,
    1.0622387728984747e-36,    2.3574065834339652e-38,    5.0603684806238552e-40,    1.0517947465207126e-41,
    2.1189362876019379e-43,    4.1414580219415240e-45,    7.8600157077637994e-47,    1.4497483865466631e-48,
    2.6007883124236257e-50,    4.5413611212163260e-52,    7.7240333031190867e-54,    1.2804773982353717e-55,
    2.0703671958500988e-57,    3.2668935096049786e-59,    5.0336915381502751e-61,    7.5777874411207040e-63,
    1.1151459556100663e-64,    1.6049946095642885e-66,    2.2603598907230871e-68,    3.1163370744148224e-70,
    4.2079095661163610e-72,    5.5670866454022987e-74,    7.2194859859479612e-76,    9.1806081646609903e-78,
    1.1452152603105840e-79,    1.4018797288267648e-81,    1.6845890246356677e-83,    1.9878466426244394e-85,
    2.3041905312794065e-87,    2.6244388724160876e-89,    2.9381121948245574e-91,    3.2340070255224033e-93
},
// x = 1.3200
{
    3.5175709486500523e-01,    1.3933601508574830e-01,    3.5084333306486186e-02,    6.4408131672390994e-03,
    9.2850590446087321e-04,    1.1009109136951491e-04,    1.1080143048249004e-05,    9.6847043978988061e-07,
    7.4797141545814492e-08,    5.1739198816635861e-09,    3.2405234005077634e-10,    1.8541744492140114e-11,
    9.7648905136545332e-13,    4.7633670824715318e-14,    2.1639663144583364e-15,    9.1986643433681918e-17,
    3.6739307885348500e-18,    1.3837372031058091e-19,    4.9306287846005810e-21,    1.6670134829196679e-22,
    5.3616759742883227e-24,    1.6444302998099138e-25,    4.8196946045116266e-27,    1.3525937264056026e-28,
    3.6411847341006313e-30,    9.4181753491389661e-32,    2.3442582969403161e-33,    5.6230962725417164e-35,
    1.3015167146018305e-36,    2.9104594883569125e-38,    6.2951904836239220e-40,    1.3184315307061776e-41,
    2.6763597989782506e-43,    5.2708423961576953e-45,    1.0079766860870455e-46,    1.8733552481393815e-48,
    3.3863595146936933e-50,    5.9581955877685735e-52,    1.0211125524602304e-53,    1.7056984172290319e-55,
    2.7789366406713639e-57,    4.4184227160460756e-59,    6.8599346057537317e-61,    1.0405835531397669e-62,
    1.5430055143611194e-64,    2.2377451749687277e-66,    3.1755282964822391e-68,    4.4114790169613170e-70,
    6.0021609341069791e-72,    8.0014871603792036e-74,    1.0455638225763921e-75,    1.3397300598398984e-77,
    1.6839709525955830e-79,    2.0771118432070903e-81,    2.5150393528957639e-83,    2.9904422498345933e-85,
    3.4927917122058679e-87,    4.0085984017336145e-89,    4.5219554400975181e-91,    5.0153437445053761e-93
},
// x = 1.3300
{
    3.4964352565990398e-01,    1.3934630467099524e-01,    3.5328552717809457e-02,    6.5321967093055870e-03,
    9.4857003725371566e-04,    1.1330172037067253e-04,    1.1488139451160494e-05,    1.0116355097052984e-06,
    7.8716409371416698e-08,    5.4859162360628232e-09,    3.4617742766206252e-10,    1.9956851924992477e-11,
    1.0589357261024452e-12,    5.2045043367569132e-14,    2.3822141292371239e-15,    1.0202851954161442e-16,
    4.1057790190442081e-18,    1.5580703901373366e-19,    5.5937818406909477e-21,    1.9055171631634851e-22,
    6.1751216401302307e-24,    1.9082365067980185e-25,    5.6351895802387544e-27,    1.5934157397949894e-28,
    4.3219283316488039e-30,    1.1263544506934035e-31,    2.8247989147003113e-33,    6.8270273012144316e-35,
    1.5921360026141003e-36,    3.5872900111447382e-38,    7.8178714387877678e-40,    1.6497271894453440e-41,
    3.3742256256107174e-43,    6.6955286867813886e-45,    1.2901217433992101e-46,    2.4158826802273273e-48,
    4.4001184928425179e-50,    7.8004881752122962e-52,    1.3469654671658435e-53,    2.2670495153109055e-55,
    3.7214604131953344e-57,    5.9618095886041001e-59,    9.3262481834284498e-61,    1.4254112250127093e-62,
    2.1296438716324216e-64,    3.1119061206624730e-66,    4.4494696862385740e-68,    6.2280622497501161e-70,
    8.5379355991924607e-72,    1.1468127450898060e-73,    1.5099026593172781e-75,    1.9493613457994398e-77,
    2.4688023951633560e-79,    3.0682330205403879e-81,    3.7432600680260729e-83,    4.4845351150220007e-85,
    5.2775398515591520e-87,    6.1027873489964463e-89,    6.9364753966975355e-91,    7.7515791666073553e-93
},
// x = 1.3400
{
    3.4755106188273210e-01,    1.3935113074209468e-01,    3.5570918430281241e-02,    6.6238231664178075e-03,
    9.6885711317343725e-04,    1.1657389883502733e-04,    1.1907197363510779e-05,    1.0563124726093473e-06,
    8.2804013406144317e-08,    5.8137950687110339e-09,    3.6960571546543981e-10,    2.1466692013843467e-11,
    1.1475689591713240e-12,    5.6823372587427731e-14,    2.6204070365866313e-15,    1.1307104935881712e-16,
    4.5842529125049820e-18,    1.7526867772425322e-19,    6.3396883640410838e-21,    2.1758110520834357e-22,
    7.1039587236248415e-24,    2.2117411235962035e-25,    6.5804911892580453e-27,    1.8746794423836949e-28,
    5.1229958226997108e-30,    1.3451490084268963e-31,    3.3988503286888652e-33,    8.2760976637467150e-35,
    1.9445711808852082e-36,    4.4142823693396592e-38,    9.6924214610427677e-40,    2.0606594619794749e-41,
    4.2463786019554337e-43,    8.4894610557209909e-45,    1.6480740949360472e-46,    3.1093728413474589e-48,
    5.7057452060338060e-50,    1.0191097021737716e-51,    1.7729923746422611e-53,    3.0065107336752631e-55,
    4.9724060982950463e-57,    8.0256978049754754e-59,    1.2649205863568272e-60,    1.9478168911227749e-62,
    2.9320182189152865e-64,    4.3165617157665956e-66,    6.2182984991256800e-68,    8.7693652265485506e-70,
    1.2112116472272708e-71,    1.6391234854766880e-73,    2.1743034523949499e-75,    2.8282359691237238e-77,
    3.6087911328232298e-79,    4.5187250850609927e-81,    5.5543057303216135e-83,    6.7042382923689554e-85,
    7.9490604583882388e-87,    9.2611477290099497e-89,    1.0605417005973308e-90,    1.1940745407220493e-92
},
// x = 1.3500
{
    3.4547943972601858e-01,    1.3935060492137499e-01,    3.5811428789629569e-02,    6.7156834783021592e-03,
    9.8936630954439596e-04,    1.1990808133952313e-04,    1.2337498629762829e-05,    1.1025389788440681e-06,
    8.7065531495408765e-08,    6.1582118648467928e-09,    3.9440154571318541e-10,    2.3076709308349787e-11,
    1.2427945338928590e-12,    6.1995717741290178e-14,    2.8801790670553731e-15,    1.2520444898956067e-16,
    5.1139421099073112e-18,    1.9697519182641001e-19,    7.1778773707561193e-21,    2.4818240568673755e-22,
    8.1634286948147161e-24,    2.5605273310546252e-25,    7.6749736777580124e-27,    2.2027718019560648e-28,
    6.0644413183737653e-30,    1.6042123241059746e-31,    4.0836495289741023e-33,    1.0017682865121748e-34,
    2.3713246652429145e-36,    5.4231674294461306e-38,    1.1996405220132126e-39,    2.5695151642067005e-41,
    5.3344538341837643e-43,    1.0744292293325980e-44,    2.1013626812402183e-46,    3.9941447648731188e-48,
    7.3839749213420673e-50,    1.3286962955535994e-51,    2.3288349331846106e-53,    3.9785218163645802e-55,
    6.6290731200369753e-57,    1.0779443423968949e-58,    1.7116075227356188e-60,    2.6553178557894484e-62,
    4.0268233797407957e-64,    5.9725729366586918e-66,    8.6680762271514938e-68,    1.2315357321097035e-69,
    1.7136678609829153e-71,    2.3363940346124358e-73,    3.1223560046226505e-75,    4.0917189474832791e-77,
    5.2599283911365197e-79,    6.6353215487515660e-81,    8.2168212601510329e-83,    9.9919796295912591e-85,
    1.1935647093361313e-86,    1.4009514443764650e-88,    1.6162708951335132e-90,    1.8333526075378643e-92
},
// x = 1.3600
{
    3.4342839910794010e-01,    1.3934483684210394e-01,    3.6050082544475705e-02,    6.8077686638853673e-03,
    1.0100967744772294e-03,    1.2330471513900124e-04,    1.2779225558836981e-05,    1.1503531795300816e-06,
    9.1506666961083505e-08,    6.5198425165382663e-09,    4.2063180356358195e-10,    2.4792608571186578e-11,
    1.3450409626325634e-12,    6.7590875735056041e-14,    3.1632825983628008e-15,    1.3852621114337973e-16,
    5.6998443592946638e-18,    2.2116418990618126e-19,    8.1188837679414277e-21,    2.8279327836303311e-22,
    9.3706384132745161e-24,    2.9609090402201024e-25,    8.9407125785990282e-27,    2.5850252424849330e-28,
    7.1694611878588261e-30,    1.9105498005037966e-31,    4.8994359695870661e-33,    1.2107829441319672e-34,
    2.8872984651382185e-36,    6.6520506668459239e-38,    1.4823670212378520e-39,    3.1985862937914478e-41,
    6.6895867269497314e-43,    1.3573434110636764e-44,    2.6743342095603491e-46,    5.1208415438198101e-48,
    9.5369771324365700e-50,    1.7288183207646579e-51,    3.0525693961825753e-53,    5.2535380834743594e-55,
    8.8183298791010698e-57,    1.4445525391645680e-58,    2.3107062584910491e-60,    3.6112760766198035e-62,
    5.5170947691069608e-64,    8.2435271892310120e-66,    1.2052527832718013e-67,    1.7250683282535640e-69,
    2.4181868291467374e-71,    3.3213393567081719e-73,    4.4715036652454371e-75,    5.9031119501376923e-77,
    7.6446771468812112e-79,    9.7150585440465113e-81,    1.2119688446134671e-82,    1.4847158944633166e-84,
    1.7866604414313907e-86,    2.1126297988063233e-88,    2.4553802898597503e-90,    2.8057885344983758e-92
},
// x = 1.3700
{
    3.4139768361563194e-01,    1.3933393418317683e-01,    3.6286878834952616e-02,    6.9000698220357337e-03,
    1.0310476274708086e-03,    1.2676424011071668e-04,    1.3232560888411899e-05,    1.1997937243264421e-06,
    9.6133249801218372e-08,    6.8993837273820086e-09,    4.4836599081098136e-10,    2.6620364585942593e-11,
    1.4547605360327034e-12,    7.3639475856775142e-14,    3.4715957896893326e-15,    1.5314162247541541e-16,
    6.3473979098589814e-18,    2.4809617954210724e-19,    9.1743449146390594e-21,    3.2190082557798573e-22,
    1.0744770448227952e-23,    3.4200194488637045e-25,    1.0402835006098102e-26,    3.0298483205341564e-28,
    8.4648553604820975e-30,    2.2723156901942213e-31,    5.8699444306218846e-33,    1.4612754280899875e-34,
    3.5102302606226876e-36,    8.1466272142519633e-38,    1.8287595944689643e-39,    3.9750053025596436e-41,
    8.3744803642646553e-43,    1.7116990764153659e-44,    3.3972898245449973e-46,    6.5529617007498407e-48,
    1.2293810907031596e-49,    2.2449400687633032e-51,    3.9930123419893533e-53,    6.9225611235235975e-55,
    1.1705262348714014e-56,    1.9315596853839972e-58,    3.1124299642530649e-60,    4.8999953355593235e-62,
    7.5409422634233155e-64,    1.1350330434465995e-65,    1.6716828656130031e-67,    2.4102518375792374e-69,
    3.4035050331292486e-71,    4.7090171400323139e-73,    6.3863266794147540e-75,    8.4929608420976598e-77,
    1.1079441150569580e-78,    1.4183544348409582e-80,    1.7824258993231321e-82,    2.1996019700705046e-84,
    2.6663890104534643e-86,    3.1760378406921943e-88,    3.7184490892196033e-90,    4.2803457034539838e-92
},
// x = 1.3800
{
    3.3938704044940554e-01,    1.3931800270319916e-01,    3.6521817181581311e-02,    6.9925781322524132e-03,
    1.0522179600110748e-03,    1.3028708870192916e-04,    1.3697687749320643e-05,    1.2508997590535382e-06,
    1.0095123786914281e-07,    7.2975534191702847e-09,    4.7767630085628663e-10,    2.8566232226790788e-11,
    1.5724304097733421e-12,    8.0174078723021790e-14,    3.8071304098710795e-15,    1.6916431268754104e-16,
    7.0625161654497936e-18,    2.7805655721980220e-19,    1.0357105527272684e-20,    3.6604670596950634e-22,
    1.2307315090981068e-23,    3.9459094760508290e-25,    1.2089911982106041e-26,    3.5468731901665682e-28,
    9.9815518285880058e-30,    2.6989901606835647e-31,    7.0229738878743448e-33,    1.7610588187807504e-34,
    4.2612043278750813e-36,    9.9616161495621451e-38,    2.2524958159052156e-39,    4.9317459231475061e-41,
    1.0465901204842709e-42,    2.1547759389861101e-44,    4.3078778811550211e-46,    8.3699840859999785e-48,
    1.5817209666291552e-49,    2.9094074544418317e-51,    5.2126312815950218e-53,    9.1029007359549514e-55,
    1.5504255048688184e-56,    2.5771204205877828e-58,    4.1829538486243350e-60,    6.6333991337426908e-62,
    1.0283077865517941e-63,    1.5590610535808599e-65,    2.3129469774937355e-67,    3.3591657421211096e-69,
    4.7780719297182212e-71,    6.6590746409989569e-73,    9.0968726244790792e-75,    1.2185897497724892e-76,
    1.6013037134074150e-78,    2.0648957119482639e-80,    2.6138602723926487e-82,    3.2491708740508312e-84,
    3.9674326480614992e-86,    4.7602455119798807e-88,    5.6138807826471404e-90,    6.5093538774178551e-92
},
// x = 1.3900
{
    3.3739622036785671e-01,    1.3929714627397299e-01,    3.6754897474389925e-02,    7.0852848553043411e-03,
    1.0736068361665687e-03,    1.3387368588052318e-04,    1.4174789630054306e-05,    1.3037109231807768e-06,
    1.0596671803153538e-07,    7.7150911404158454e-09,    5.0863769491595998e-10,    3.0636756793426285e-11,
    1.6985537297698245e-12,    8.7229279580517144e-14,    4.1720400763260955e-15,    1.8671683702304884e-16,
    7.8516247329225702e-18,    3.1135775222513544e-19,    1.1681331570242534e-20,    4.1583272947112901e-22,
    1.4082326088549614e-23,    4.5476570815476232e-25,    1.4034397431064125e-26,    4.1471218506064445e-28,
    1.1755202394852690e-29,    3.2015819173667963e-31,    8.3910433630071625e-33,    2.1193401050720238e-34,
    5.1652494128225680e-36,    1.2162451375902084e-37,    2.7700518281592646e-39,    6.1088206707792755e-41,
    1.3057687413195654e-42,    2.7078516590075359e-44,    5.4527978355847658e-46,    1.0671219186963145e-47,
    2.0312005172638876e-49,    3.7632329873261568e-51,    6.7912122459529622e-53,    1.1945489205597417e-54,
    2.0493161529871801e-56,    3.4310479023397536e-58,    5.6093072675362609e-60,    8.9597614851289495e-62,
    1.3989998656277645e-63,    2.1364462173637604e-65,    3.1924850472753852e-67,    4.6701307159608801e-69,
    6.6909032370319895e-71,    9.3924839179601681e-73,    1.2923889884183633e-74,    1.7437862871647006e-76,
    2.3080440423327576e-78,    2.9978036657437680e-80,    3.8222769546811272e-82,    4.7857178518761930e-84,
    5.8859814690276636e-86,    7.1133388967231632e-88,    8.4497173871856659e-90,    9.8685139422122263e-92
},
// x = 1.4000
{
    3.3542497763385071e-01,    1.3927146691339873e-01,    3.6986119962282189e-02,    7.1781813338202957e-03,
    1.0952132931804847e-03,    1.3752444908861274e-04,    1.4664050341384971e-05,    1.3582673471807935e-06,
    1.1118590730504945e-07,    8.1527584766218537e-09,    5.4132797946714176e-10,    3.2838784614721194e-11,
    1.8336607967223305e-12,    9.4841816108161669e-14,    4.5686289220700046e-15,    2.0593129385443600e-16,
    8.7217010074934651e-18,    3.4834153494712962e-19,    1.3162633815226570e-20,    4.7192697328463243e-22,
    1.6096702297529693e-23,    5.2354885697705397e-25,    1.6273118948744022e-26,    4.8431933885376125e-28,
    1.3826858653463904e-29,    3.7928598252477937e-31,    1.0012147204075986e-32,    2.5469551333131556e-34,
    6.2520374885913026e-36,    1.4827272438406595e-37,    3.4012466913806929e-39,    7.5547116764263328e-41,
    1.6264369888419178e-42,    3.3970853745762882e-44,    6.8898815183678148e-46,    1.3580548092952563e-47,
    2.6035569418752185e-49,    4.8583246031940901e-51,    8.8304730695813038e-53,    1.5644149243740605e-54,
    2.7031391848198024e-56,    4.5582458545995710e-58,    7.5057102148790134e-60,    1.2075098515972323e-61,
    1.8989942390736888e-63,    2.9208532895869312e-65,    4.3960084218300169e-67,    6.4769508556471454e-69,
    9.3462692659533835e-71,    1.3214356651616476e-72,    1.8313480245876460e-74,    2.4887599484635600e-76,
    3.3177678945588189e-78,    4.3402754444615901e-80,    5.5737629174470518e-82,    7.0288730638664288e-84,
    8.7070252438663462e-86,    1.0598313171621371e-87,    1.2679957487940515e-89,    1.4915566996555859e-91
},
// x = 1.4100
{
    3.3347306996136905e-01,    1.3924106481780185e-01,    3.7215485242641705e-02,    7.2712589928316173e-03,
    1.1170363420590966e-03,    1.4123978819908608e-04,    1.5165653981120087e-05,    1.4146096497520043e-06,
    1.1661515397109685e-07,    8.6113394621842563e-09,    5.7582788492601123e-10,    3.5179473924506491e-11,
    1.9783102709412275e-12,    1.0305068086769761e-13,    4.9993607087198877e-15,    2.2694997913253613e-16,
    9.6803164442725434e-18,    3.8938150062539900e-19,    1.4818201798103422e-20,    5.3507046240136823e-22,
    1.8380497640045135e-23,    6.0209130785724570e-25,    1.8847825958906282e-26,    5.6494746661991442e-28,
    1.6243738242461621e-29,    4.4876174004237851e-31,    1.1930623907502604e-32,    3.0566408660696918e-34,
    7.5566994293342271e-36,    1.8049265515999767e-37,    4.1698871782981934e-39,    9.3280779721201073e-41,
    2.0225524572918422e-42,    4.2545874414057517e-44,    8.6906314868262080e-46,    1.7252244482476847e-47,
    3.3310736641098863e-49,    6.2602640355745239e-51,    1.1459856127986901e-52,    2.0447316384176926e-54,
    3.5582957175536560e-56,    6.0431131240192610e-58,    1.0021764643037753e-59,    1.6238002021756678e-61,
    2.5719062091624930e-63,    3.9841057664740762e-65,    6.0390597709561179e-67,    8.9613031153087077e-69,
    1.3023528403195671e-70,    1.8544975074342172e-72,    2.5884573809484297e-74,    3.5427710945597874e-76,
    4.7565939057169333e-78,    6.2669668547382796e-80,    8.1054786589622748e-82,    1.0294517107072507e-83,
    1.2843406286027501e-85,    1.5744806816695289e-87,    1.8971802121949169e-89,    2.2476091204469312e-91
},
// x = 1.4200
{
    3.3154025846320395e-01,    1.3920603839369472e-01,    3.7442994251172861e-02,    7.3645093402690577e-03,
    1.1390749681563889e-03,    1.4502010547504479e-04,    1.5679784898999203e-05,    1.4727789349113155e-06,
    1.2226093866840317e-07,    9.0916409938127876e-09,    6.1222114555599598e-10,    3.7666306012844129e-11,
    2.1330904183796644e-12,    1.1189723855427905e-13,    5.4668684039356956e-15,    2.4992607953593075e-16,
    1.0735681672421939e-17,    4.3488574020942007e-19,    1.6666948950326550e-20,    6.0608446147459436e-22,
    2.0967261939807877e-23,    6.9168715619767362e-25,    2.1805801427614550e-26,    6.5823771693129929e-28,
    1.9060092564496310e-29,    5.3029745220102554e-31,    1.4198154459478831e-32,    3.6633505146372039e-34,
    9.1207760248674877e-36,    2.1939412749016793e-37,    5.1045305576984595e-39,    1.1499789889595152e-40,
    2.5110996386358100e-42,    5.3197127410957859e-44,    1.0943312173746246e-45,    2.1878115993737552e-47,
    4.2541768774751516e-49,    8.0517644638545146e-51,    1.4843789604468177e-52,    2.6672839668413502e-54,
    4.6745776747393147e-56,    7.9951857455823339e-58,    1.3353023214746563e-59,    2.1788916847372593e-61,
    3.4755688211123407e-63,    5.4221234994603348e-65,    8.2770526099940124e-67,    1.2369309431305266e-68,
    1.8103848249952293e-70,    2.5961901406722015e-72,    3.6493818855436578e-74,    5.0302489818936753e-76,
    6.8015924050096293e-78,    9.0248523162426110e-80,    1.1755188268144895e-81,    1.5035752018115896e-83,
    1.8891553923318321e-85,    2.3323467428568706e-87,    2.8303015901490880e-89,    3.3768618473496485e-91
},
// x = 1.4300
{
    3.2962630759948697e-01,    1.3916648428898371e-01,    3.7668648251968660e-02,    7.4579239674151133e-03,
    1.1613281317547478e-03,    1.4886579553209032e-04,    1.6206627661744475e-05,    1.5328167889588376e-06,
    1.2812987546295752e-07,    9.5944932453562352e-09,    6.5059458060196069e-10,    4.0307096656108797e-11,
    2.2986203988123096e-12,    1.2142534820130001e-13,    5.9739642423085113e-15,    2.7502440623227052e-16,
    1.1896694615936482e-17,    4.8529971065920473e-19,    1.8729669732168159e-20,    6.8587842828051651e-22,
    2.3894415426803700e-23,    7.9379016936134438e-25,    2.5200543910116205e-26,    7.6606030174412222e-28,
    2.2338188456232338e-29,    6.2587212497225474e-31,    1.6874908270804855e-32,    4.3846178857828562e-34,
    1.0993325486168103e-35,    2.6629717158516861e-37,    6.2393858010086994e-39,    1.4155350018200644e-40,
    3.1127160671553769e-42,    6.6406220398483820e-44,    1.3756708627766593e-45,    2.7696250519126324e-47,
    5.4234049836042726e-49,    1.0336966461125579e-50,    1.9190774195322558e-52,    3.4726636441982010e-54,
    6.1288882136127476e-56,    1.0556350888810179e-57,    1.7754599676842462e-59,    2.9175144632567740e-61,
    4.6865087767747043e-63,    7.3627351587395737e-65,    1.1318575767950327e-66,    1.7033655970519504e-68,
    2.5106127100967931e-70,    3.6256992335632225e-72,    5.1324070686960763e-74,    7.1242280585458054e-76,
    9.7007547776809987e-78,    1.2962287800384662e-79,    1.7002697008593957e-81,    2.1900805473169332e-83,
    2.7710818681238158e-85,    3.4452565265665938e-87,    4.2102508075174602e-89,    5.0586587049250272e-91
},
// x = 1.4400
{
    3.2773098512703691e-01,    1.3912249742363195e-01,    3.7892448827803592e-02,    7.5514945493133524e-03,
    1.1837947686415902e-03,    1.5277724530341573e-04,    1.6746367018275609e-05,    1.5947652773165376e-06,
    1.3422871289499854e-07,    1.0120750083917236e-08,    6.9103817664603985e-10,    4.3110007829150606e-11,
    2.4755515971064523e-12,    1.3168149049691615e-13,    6.5236502892750591e-15,    3.0242217123788186e-16,
    1.3172991792880215e-17,    5.4110931771009948e-19,    2.1029209648635478e-20,    7.7545868266008782e-22,
    2.7203659924763128e-23,    9.1003202447103979e-25,    2.9092527364014433e-26,    8.9054434558907086e-28,
    2.6149417704479674e-29,    7.3777092274868937e-31,    2.0030857130245307e-32,    5.2409781604933023e-34,
    1.3232211694507013e-35,    3.2276980842759310e-37,    7.6153769875756232e-39,    1.7397770390349363e-40,
    3.8524417977734533e-42,    8.2761642886357535e-44,    1.7264690332064267e-45,    3.5001711882962540e-47,
    6.9018342146826362e-49,    1.3246766863029507e-50,    2.4764735214378536e-52,    4.5126164520963896e-54,
    8.0199563497090243e-56,    1.3910053850649578e-57,    2.3558663485058159e-59,    3.8983212749664225e-61,
    6.3057821360144185e-63,    9.9759367973318422e-65,    1.5442988116584919e-66,    2.3403053707515297e-68,
    3.4735178768984239e-70,    5.0513422967549387e-72,    7.2004787940329133e-74,    1.0064759401870886e-75,
    1.3800551947094345e-77,    1.8569404479255126e-79,    2.4527854256004312e-81,    3.1814660296398191e-83,
    4.0536108640529792e-85,    5.0750437602141788e-87,    6.2452777082242061e-89,    7.5562228201203717e-91
},
// x = 1.4500
{
    3.2585406204951473e-01,    1.3907417101978539e-01,    3.8114397870648513e-02,    7.6452128451359310e-03,
    1.2064737906817779e-03,    1.5675483400765314e-04,    1.7299187865099618e-05,    1.6586669412427045e-06,
    1.4056433500268042e-07,    1.0671289487140799e-08,    7.3364517118024532e-10,    4.6083559702759779e-11,
    2.6645689985384466e-12,    1.4271490037277159e-13,    7.1191295282166406e-15,    3.3230980843881366e-16,
    1.4575002973037817e-17,    6.0284422484946704e-19,    2.3590649085164467e-20,    8.7593784871849635e-22,
    3.0941429977336601e-23,    1.0422424627712396e-24,    3.3550046879144314e-26,    1.0341113495200795e-27,
    3.0575549872772727e-29,    8.6862968155250784e-31,    2.3747280236240770e-32,    6.2564533129336003e-34,
    1.5905600975388771e-35,    3.9067226083613214e-37,    9.2813965392533320e-39,    2.1350987444006496e-40,
    4.7606153056822970e-42,    1.0298142672061266e-43,    2.1631744539870374e-45,    4.4159599850199726e-47,
    8.7680614967302644e-49,    1.6945418385238298e-50,    3.1899181587347571e-52,    5.8529905778651063e-54,
    1.0474301048014033e-55,    1.8293027676119900e-57,    3.1196885798893820e-59,    5.1980698717912101e-61,
    8.4665681465584752e-63,    1.3487335587661739e-64,    2.1023639569727193e-66,    3.2081394918822208e-68,
    4.7946229126418622e-70,    7.0209514938821255e-72,    1.0077547499281215e-73,    1.4184081758988098e-75,
    1.9583877241552959e-77,    2.6534116566397519e-79,    3.5291546251245041e-81,    4.6093844271212446e-83,
    5.9137401789783833e-85,    7.4552902084750647e-87,    9.2380673953759379e-89,    1.1254825510325890e-90
},
// x = 1.4600
{
    3.2399531256837127e-01,    1.3902159663137448e-01,    3.8334497572396251e-02,    7.7390706985107577e-03,
    1.2293640863857039e-03,    1.6079893311943404e-04,    1.7865275211885724e-05,    1.7245647944241945e-06,
    1.4714376232208499e-07,    1.1247013961560733e-08,    7.7851213739054487e-10,    4.9236642929603780e-11,
    2.8663926091151949e-12,    1.5457770502854113e-13,    7.7638174914897753e-15,    3.6489184141538130e-16,
    1.6114009382365453e-17,    6.7108140301121784e-19,    2.6441501959550942e-20,    9.8854513218717567e-22,
    3.5159387427764775e-23,    1.1924715444657253e-24,    3.8650159253673475e-26,    1.1995126744200873e-27,
    3.5710145629556307e-29,    1.0214854829241104e-30,    2.8118486316810396e-32,    7.4591114948600530e-34,
    1.9093699186923308e-35,    4.7220862667093245e-37,    1.1295780358229931e-38,    2.6163910105151447e-40,
    5.8739429933444269e-42,    1.2794038892021684e-43,    2.7059675536524112e-45,    5.5620973630313664e-47,
    1.1119866988026373e-48,    2.1638690181724559e-50,    4.1014837158225500e-52,    7.5774338933031561e-54,
    1.3653762954982309e-55,    2.4010210184495393e-57,    4.1229185910981882e-59,    6.9170171429149552e-61,
    1.1344031968112770e-62,    1.8195733119663465e-64,    2.8558455828010284e-66,    4.3879689445813809e-68,
    6.6031064191032274e-70,    9.7358394430796595e-72,    1.4070707135918288e-73,    1.9940956293371945e-75,
    2.7722166216447523e-77,    3.7819596254412662e-79,    5.0648519121995246e-81,    6.6607455258305438e-83,
    8.6045078041304001e-85,    1.0922253734482235e-86,    1.3627392985059773e-88,    1.6716848110478536e-90
},
// x = 1.4700
{
    3.2215451403457468e-01,    1.3896486417319759e-01,    3.8552750415804159e-02,    7.8330600378095378e-03,
    1.2524645214729310e-03,    1.6490990634261571e-04,    1.8444814147235067e-05,    1.7925023194484383e-06,
    1.5397415286324886e-07,    1.1848850961887383e-08,    8.2573907014659344e-10,    5.2578531221754275e-11,
    3.0817789218666851e-12,    1.6732506755898622e-13,    8.4613544567337436e-15,    4.0038780029326174e-16,
    1.7802205651356158e-17,    7.4644893628649612e-19,    2.9611930249115433e-20,    1.1146374992366982e-21,
    3.9914963243858576e-23,    1.3630142039076603e-24,    4.4479728192315320e-26,    1.3898714898443261e-27,
    4.1660149673588372e-29,    1.1998340580474132e-30,    3.3253782147494308e-32,    8.8817109693796589e-34,
    2.2890765461572987e-35,    5.6998720350317410e-37,    1.3728042057843267e-38,    3.2015212328319559e-40,
    7.2367742777356822e-42,    1.5870283910099494e-43,    3.3795704114178531e-45,    6.9942238524438465e-47,
    1.4078704415732642e-48,    2.7583942908922594e-50,    5.2641560782674767e-52,    9.7920227596148781e-54,
    1.7764999710436553e-55,    3.1453681498410571e-57,    5.4380480686845892e-59,    9.1858631595320456e-61,
    1.5168110023581281e-62,    2.4496078399748877e-64,    3.8710142129352475e-66,    5.9884811789362965e-68,
    9.0732858865955569e-70,    1.3469544480757132e-71,    1.9600117999365267e-73,    2.7967396966390742e-75,
    3.9146827785636298e-77,    5.3771195076651712e-79,    7.2504169170110979e-81,    9.6002498347389557e-83,
    1.2486744531134843e-84,    1.5958754312107094e-86,    2.0047631599076800e-88,    2.4761005969958871e-90
},
// x = 1.4800
{
    3.2033144690110615e-01,    1.3890406194949784e-01,    3.8769159165637751e-02,    7.9271728763979001e-03,
    1.2757739394312700e-03,    1.6908810958612641e-04,    1.9037989804654964e-05,    1.8625234641571824e-06,
    1.6106280306189258e-07,    1.2477753311119925e-08,    8.7542947319083966e-10,    5.6118894222866190e-11,
    3.3115224300815617e-12,    1.8101533635332252e-13,    9.2156182304077551e-15,    4.3903318992733982e-16,
    1.9652765713478581e-17,    8.2963009977662065e-19,    3.3134975517957579e-20,    1.2557118276814401e-21,
    4.5271950676381565e-23,    1.5564373222211414e-24,    5.1136584821351301e-26,    1.6087296800556108e-27,
    4.8547684450131115e-29,    1.4076948823510808e-30,    3.9279720465501170e-32,    1.0562440594850213e-33,
    2.7407444098345190e-35,    6.8709083253725780e-37,    1.6660909359929828e-38,    3.9118999960905045e-40,
    8.9026198148401060e-42,    1.9656179512689487e-43,    4.2142246360790894e-45,    8.7808715099566513e-47,
    1.7795198161048623e-48,    3.5102548448179424e-50,    6.7445555523092401e-52,    1.2631047651616132e-53,
    2.3071436763038938e-55,    4.1126664394152730e-57,    7.1587582100485462e-59,    1.2174674279308690e-60,
    2.0240053197913841e-62,    3.2909373199552364e-64,    5.2358927836843492e-66,    8.1550318061314709e-68,
    1.2439892080634709e-69,    1.8592938235915380e-71,    2.7239363399604475e-73,    3.9132132123203364e-75,
    5.5146922399809236e-77,    7.6263665819401284e-79,    1.0353192539386681e-80,    1.3801846891280444e-82,
    1.8073709263530199e-84,    2.3256292607623765e-86,    2.9413589576725699e-88,    3.6576044014226562e-90
},
// x = 1.4900
{
    3.1852589467620857e-01,    1.3883927668204074e-01,    3.8983726860019316e-02,    8.0214013128489161e-03,
    1.2992911620711936e-03,    1.7333389094238596e-04,    1.9644987328746762e-05,    1.9346726378839950e-06,
    1.6841714870654809e-07,    1.3134699621365867e-08,    9.2769044752023284e-10,    5.9867810678014666e-11,
    3.5564571884634626e-12,    1.9571020043978803e-13,    1.0030737541127153e-14,    4.8108051180981753e-16,
    2.1679912869212895e-17,    9.2136772657873967e-19,    3.7046808638478593e-20,    1.4134181064523400e-21,
    5.1301154155611684e-23,    1.7756095529580373e-24,    5.8730815211934725e-26,    1.8601002486004895e-27,
    5.6512068160854969e-29,    1.6496849213671061e-30,    4.6342664396970676e-32,    1.2545770453306879e-33,
    3.2773461945286067e-35,    8.2715883316716847e-37,    2.0192712492145776e-38,    4.7731503234093147e-40,
    1.0935956925190017e-41,    2.4308594002482454e-43,    5.2468703824691175e-45,    1.1006325276470300e-46,
    2.2455862987106219e-48,    4.4595176867035729e-50,    8.6263090681183154e-52,    1.6264233135689265e-53,
    2.9908283904506659e-55,    5.3673889440375710e-57,    9.4058940960227239e-59,    1.6104328836669913e-60,
    2.6953793094229830e-62,    4.4121562239007014e-64,    7.0671630242136181e-66,    1.1081618076381160e-67,
    1.7018345746180634e-69,    2.5607785320305472e-71,    3.7769763132357367e-73,    5.4626601169127453e-75,
    7.7502436316975720e-77,    1.0790341728446605e-78,    1.4747394573860963e-80,    1.9792549510123620e-82,
    2.6093683326292388e-84,    3.3802760897841648e-86,    4.3041098374332236e-88,    5.3883488693457464e-90
},
// x = 1.5000
{
    3.1673764387737868e-01,    1.3877059353770219e-01,    3.9196456801974346e-02,    8.1157375311211675e-03,
    1.3230149900754866e-03,    1.7764759066825499e-04,    2.0265991841616623e-05,    2.0089947075775826e-06,
    1.7604476584079403e-07,    1.3820694715250587e-08,    9.8263278095331669e-10,    6.3835781904138612e-11,
    3.8174584231912348e-12,    2.1147485095140215e-13,    1.0911106065994170e-14,    5.2680034218160986e-16,
    2.3898994240900582e-17,    1.0224688817968421e-18,    4.1386998974286279e-20,    1.5895737644468123e-21,
    5.8081098669163968e-23,    2.0237341563307015e-24,    6.7386187683788666e-26,    2.1485258170429204e-27,
    6.5712083110497203e-29,    1.9311021000113962e-30,    5.4611710109732434e-32,    1.4883428008502248e-33,
    3.9140745224174981e-35,    9.9448233157621859e-37,    2.4440182177047995e-38,    5.8158970959977391e-40,
    1.3414373857490554e-41,    3.0017577518346886e-43,    6.5225659622715466e-45,    1.3774091897787956e-46,
    2.8291306519168644e-48,    5.6560584592028813e-50,    1.1014223154243974e-51,    2.0905733576419087e-53,
    3.8701373299179698e-55,    6.9919948620522137e-57,    1.2335062490799426e-58,    2.1261172658703772e-60,
    3.5823487512357638e-62,    5.9034013715757473e-64,    9.5191914647734458e-66,    1.5026634162092063e-67,
    2.3231621151356166e-69,    3.5191508817530287e-71,    5.2253317861717067e-73,    7.6081239741231720e-75,
    1.0866572737964310e-76,    1.5230575481555249e-78,    2.0955611215605158e-80,    2.8313315488400440e-82,
    3.7577541888396946e-84,    4.9005991413894025e-86,    6.2818044115346086e-88,    7.9170039240723169e-90
},
// x = 1.5100
{
    3.1496648398608840e-01,    1.3869809615557491e-01,    3.9407352551171400e-02,    8.2101738007024829e-03,
    1.3469442035440409e-03,    1.8202954116846640e-04,    2.0901188409517444e-05,    2.0855349938129636e-06,
    1.8395337164031763e-07,    1.4536770047798856e-08,    1.0403710388749836e-09,    6.8033745563978236e-11,
    4.0954441918712107e-12,    2.2837814889199119e-13,    1.1861397113753235e-14,    5.7648246891575398e-16,
    2.6326559853652905e-17,    1.1338098624119937e-18,    4.6198804368293011e-20,    1.7861792154099283e-21,
    6.5698804725348629e-23,    2.3043851190044138e-24,    7.7241733832197813e-26,    2.4791438726925257e-27,
    7.6328523225156894e-29,    2.2580196907461005e-30,    6.4282014527001230e-32,    1.7635517189047679e-33,
    4.6687017224899036e-35,    1.1941150509628948e-36,    2.9541723699837022e-38,    7.0766971120035906e-40,
    1.6431113331978416e-41,    3.7013068473174485e-43,    8.0961948744994417e-45,    1.7211099634016056e-46,
    3.5586227435503771e-48,    7.1618671830888640e-50,    1.4039440029302642e-51,    2.6825323848668346e-53,
    4.9990753635746471e-55,    9.0917659169220500e-57,    1.6146277830604926e-58,    2.8015751907962984e-60,
    4.7518968779147703e-62,    7.8828924388234728e-64,    1.2795803456594509e-65,    2.0333562851341445e-67,
    3.1645766114740649e-69,    4.8256782961380286e-71,    7.2130473456230792e-73,    1.0572239000733401e-74,
    1.5200802221536053e-76,    2.1447407248854045e-78,    2.9705948646395411e-80,    4.0403451646481303e-82,
    5.3980983197540414e-84,    7.0867370636384518e-86,    9.1446334064923159e-88,    1.1601857603026776e-89
},
// x = 1.5200
{
    3.1321220740322425e-01,    1.3862186667360196e-01,    3.9616417915851798e-02,    8.3047024767205317e-03,
    1.3710775625337044e-03,    1.8648006698149464e-04,    2.1550762009730697e-05,    2.1643392666925707e-06,
    1.9215082526453695e-07,    1.5283984128670900e-08,    1.1010236561507301e-09,    7.2473089746333068e-11,
    4.3913770943744664e-12,    2.4649279938464931e-13,    1.2886578989245516e-14,    6.3043708983353327e-16,
    2.8980446587935530e-17,    1.2563415428272616e-18,    5.1529483360436478e-20,    2.0054347113740502e-21,
    7.4250634391759586e-23,    2.6215468594830581e-24,    8.8453498488831930e-26,    2.8577594847448221e-27,
    8.8567052643354969e-29,    2.6373929524572714e-30,    7.5578580643922135e-32,    2.0871800047220943e-33,
    5.5619946741083663e-35,    1.4320019314572536e-36,    3.5661243729755731e-38,    8.5991335521369942e-40,
    2.0098086030021730e-41,    4.5572892987586108e-43,    1.0034515756795746e-44,    2.1472775815981185e-46,
    4.4691585413769324e-48,    9.0538738419423121e-50,    1.7865798922422537e-51,    3.4362299256360726e-53,
    6.4460194455690831e-55,    1.1800895104432129e-56,    2.1096188068035247e-58,    3.6846716492606399e-60,
    6.2911281881652735e-62,    1.0505390849203054e-63,    1.7165613397611131e-65,    2.7458098751899786e-67,
    4.3016776737648828e-69,    6.6030706370542925e-71,    9.9350884136352313e-73,    1.4658358887737444e-74,
    2.1215357605168467e-76,    3.0131736408080732e-78,    4.2010547432051431e-80,    5.7517364122854556e-82,
    7.7354739142474759e-84,    1.0222523514722413e-85,    1.3278340298629657e-87,    1.6957848561659650e-89
},
// x = 1.5300
{
    3.1147460940523253e-01,    1.3854198575474627e-01,    3.9823656944945704e-02,    8.3993160000213774e-03,
    1.3954138075931055e-03,    1.9099948476781591e-04,    2.2214897497696165e-05,    2.2454537416393914e-06,
    2.0064512868252811e-07,    1.6063422944634699e-08,    1.1647130302017769e-09,    7.7165667355405107e-11,
    4.7062660355564132e-12,    2.6589553258788705e-13,    1.3991931064289608e-14,    6.8899607520822740e-16,
    3.1879867260169416e-17,    1.3910950869259619e-18,    5.7430631144149138e-20,    2.2497586034013892e-21,
    8.3843214309828853e-23,    2.9796578438833548e-24,    1.0119647519856078e-25,    3.2909262786278584e-27,
    1.0266141064615591e-28,    3.0771795026044777e-30,    8.8760559339993092e-32,    2.4673163171348735e-33,
    6.6181926647961611e-35,    1.7151282892456476e-36,    4.2992618227466906e-38,    1.0435102420649375e-39,
    2.4549436557701195e-41,    5.6032294691360182e-43,    1.2418620961763005e-44,    2.6749177527044735e-46,
    5.6039394089168401e-48,    1.1427406068940695e-49,    2.2697672792443787e-51,    4.3942714363086651e-53,
    8.2974023532888026e-55,    1.5290139792985115e-56,    2.7513544798858734e-58,    4.8371269524848992e-60,
    8.3131082583244523e-62,    1.3973120216562064e-63,    2.2981948790436328e-65,    3.7003635517524260e-67,
    5.8352338036066252e-69,    9.0159783188904653e-71,    1.3654796186750207e-72,    2.0278949573184876e-74,
    2.9543172205257125e-76,    4.2235499691987271e-78,    5.9273226255029422e-80,    8.1685691231933284e-82,
    1.1058106371948535e-83,    1.4709530614009517e-85,    1.9232306995525732e-87,    2.4723233133677632e-89
},
// x = 1.5400
{
    3.0975348810095965e-01,    1.3845853261270283e-01,    4.0029073920369702e-02,    8.4940068972169987e-03,
    1.4199516602923761e-03,    1.9558810330051694e-04,    2.2893779574397423e-05,    2.3289250750842394e-06,
    2.0944442747301207e-07,    1.6876200382156923e-08,    1.2315656152319658e-09,    8.2123810811926199e-11,
    5.0411680408602392e-12,    2.8666729146779151e-13,    1.5183060580777403e-14,    7.5251429730790634e-16,
    3.5045505098770050e-17,    1.5391880485480574e-18,    6.3958540859648174e-20,    2.5218071149523307e-21,
    9.4594442024215454e-23,    3.3836584634782368e-24,    1.1566674527761338e-25,    3.7860365349920075e-27,
    1.1887700188321115e-28,    3.5864750716536184e-30,    1.0412613364324852e-31,    2.9133293859734557e-33,
    7.8655572768009465e-35,    2.0516926120672323e-36,    5.1764903096003218e-38,    1.2646322904807147e-39,
    2.9945758155780019e-41,    6.8795273675331174e-43,    1.5346881447616313e-44,    3.3272384063299422e-46,
    7.0160677834701546e-48,    1.4400414382422362e-49,    2.8789608618686941e-51,    5.6100730788894081e-53,
    1.0662304903589031e-54,    1.9776425860883396e-56,    3.5818746713649482e-58,    6.3383890638646721e-60,
    1.0964339868556863e-61,    1.8549839844048665e-63,    3.0708698008399889e-65,    4.9767610539246557e-67,
    7.8992991244049501e-69,    1.2284885527098345e-70,    1.8727141274411282e-72,    2.7993666078335196e-74,
    4.1048713689121451e-76,    5.9067448432918099e-78,    8.3436740419805550e-80,    1.1573720198617198e-81,
    1.5770157045252119e-83,    2.1114576205071871e-85,    2.7787102958511460e-87,    3.5953859430451179e-89
},
// x = 1.5500
{
    3.0804864438917490e-01,    1.3837158503716387e-01,    4.0232673349502833e-02,    8.5887677807029317e-03,
    1.4446898237477045e-03,    2.0024622345820239e-04,    2.3587592754010562e-05,    2.4148003600492801e-06,
    2.1855701159817030e-07,    1.7723458649994141e-08,    1.3017120175968771e-09,    8.7360347068702387e-11,
    5.3971901258092585e-12,    3.0889342661759502e-13,    1.6465920212444291e-14,    8.2137102992749016e-16,
    3.8499613894495133e-17,    1.7018308833999566e-18,    7.1174591915478010e-20,    2.8244957401594953e-21,
    1.0663458243264990e-22,    3.8390435539167276e-24,    1.3208384012128857e-25,    4.3514213631895374e-27,
    1.3751491489615131e-28,    4.1756664724940568e-30,    1.2201805925018985e-31,    3.4360594230476349e-33,
    9.3370045273781676e-35,    2.4513065279231228e-36,    6.2248404449899484e-38,    1.5306108625622092e-39,
    3.6479068338708444e-41,    8.4348061637178249e-43,    1.8938469734607399e-44,    4.1325399248580433e-46,
    8.7707239673458106e-48,    1.8118628177516109e-49,    3.6458169347959004e-51,    7.1505014332590046e-53,
    1.3678171993743372e-54,    2.5534881415027048e-56,    4.6548489224438994e-58,    8.2905500121144576e-60,
    1.4434317732024895e-61,    2.4578950162889470e-63,    4.0953783285043411e-65,    6.6801918634137552e-67,
    1.0671864120384917e-68,    1.6704461406161928e-70,    2.5629641911694535e-72,    3.8560314448405088e-74,
    5.6910201737475146e-76,    8.2423036638507265e-78,    1.1718380570108121e-79,    1.6360371295349962e-81,
    2.2437069865415262e-83,    3.0235870149519963e-85,    4.0049157714241832e-87,    5.2156168342053616e-89
},
// x = 1.5600
{
    3.0635988191676511e-01,    1.3828121941864185e-01,    4.0434459957838537e-02,    8.6835913486469317e-03,
    1.4696269831406411e-03,    2.0497413822015959e-04,    2.4296521331823562e-05,    2.5031271216299403e-06,
    2.2799131615106507e-07,    1.8606368701666602e-08,    1.3752870923051712e-09,    9.2888612943141229e-11,
    5.7754912203968990e-12,    3.3266389831910139e-13,    1.7846826412451675e-14,    8.9597142096141849e-16,
    4.2266124115792875e-17,    1.8803338965690770e-18,    7.9145667127691909e-20,    3.1610223866413228e-21,
    1.2010746165884234e-22,    4.3519199666200834e-24,    1.5069334816051638e-25,    4.9964619898057034e-27,
    1.5891641637015395e-28,    4.8566038200996513e-30,    1.4282994381921475e-31,    4.0480364981220735e-33,
    1.1070830850067369e-34,    2.9252259820742401e-36,    7.4761742737724608e-38,    1.8501442529559259e-39,
    4.4378676042995062e-41,    1.0327511644654333e-42,    2.3337569159104997e-44,    5.1252858194365657e-46,
    1.0947801412954086e-47,    2.2761838875318998e-49,    4.6096459738059361e-51,    9.0991328153331806e-53,
    1.7517916820807939e-54,    3.2913891444420499e-56,    6.0386805072980545e-58,    1.0824577732377402e-59,
    1.8967719337349755e-61,    3.2506745304213079e-63,    5.4512432254657678e-65,    8.9491523979269682e-67,
    1.4388805743450254e-68,    2.2667753970050245e-70,    3.5003393799037568e-72,    5.3002924682272083e-74,
    7.8730152297305737e-76,    1.1476020624006902e-77,    1.6421094181425860e-79,    2.3073818567904097e-81,
    3.1848158018028554e-83,    4.3194907423693793e-85,    5.7583146645425005e-87,    7.5474396250592243e-89
},
// x = 1.5700
{
    3.0468700703759050e-01,    1.3818751077286143e-01,    4.0634438681804309e-02,    8.7784703849497509e-03,
    1.4947618062321165e-03,    2.0977213266373652e-04,    2.5020749352433384e-05,    2.5939533123773575e-06,
    2.3775592207646289e-07,    1.9526130657696805e-08,    1.4524300406417250e-09,    9.8722470769259707e-11,
    6.1772841493859784e-12,    3.5807348604347610e-13,    1.9332478574607498e-14,    9.7674804117178316e-16,
    4.6370755292048499e-17,    2.0761146510178048e-18,    8.7944600568790523e-20,    3.5348923902589374e-21,
    1.3517176617996515e-22,    4.9290696356432835e-24,    1.7176978973503919e-25,    5.7317133013997010e-27,
    1.8346797339203053e-28,    5.6427942598950643e-30,    1.6701335722271649e-31,    4.7637294466708326e-33,
    1.3111546036255223e-34,    3.4866182214841405e-36,    8.9680064740599124e-38,    2.2335404799701020e-39,
    5.3918092117284015e-41,    1.2627808468794648e-42,    2.8718396950153301e-44,    6.3473887909725056e-46,
    1.3645092889126027e-47,    2.8551546272753426e-49,    5.8191919508336748e-51,    1.1560271849427053e-52,
    2.2398737334338110e-54,    4.2353903099404829e-56,    7.8204091750583179e-58,    1.4108202592271046e-59,
    2.4879933539837524e-61,    4.2912225542310819e-63,    7.2423019376209309e-65,    1.1965657040341535e-66,
    1.9362125735052051e-68,    3.0698027616050577e-70,    4.7707414915080036e-72,    7.2702502984287280e-74,
    1.0868371248281526e-75,    1.5943680493612110e-77,    2.2960074121238609e-79,    3.2468680388222794e-81,
    4.5102798836457907e-83,    6.1563849123086733e-85,    8.2596698657220216e-87,    1.0895346686954909e-88
},
// x = 1.5800
{
    3.0302982877199008e-01,    1.3809053276472505e-01,    4.0832614661752593e-02,    8.8733977591789644e-03,
    1.5200929438710920e-03,    2.1464048396388423e-04,    2.5760460578227318e-05,    2.6873273075835123e-06,
    2.4785955686485349e-07,    2.0483974227494946e-08,    1.5332845089016604e-09,    1.0487632437159848e-10,
    6.6038376695316153e-12,    3.8522200559192193e-13,    2.0929979036745680e-14,    1.0641625124125625e-15,
    5.0841134980219057e-17,    2.2907058636830481e-18,    9.7650658126218559e-20,    3.9499455374249472e-21,
    1.5200245560278824e-22,    5.5780186184149360e-24,    1.9561978516651178e-25,    6.5690408877072975e-27,
    2.1160686129463715e-28,    6.5496197090585025e-30,    1.9508587566020413e-31,    5.5998293174740899e-33,
    1.5510827977696973e-34,    4.1508698239188112e-36,    1.0744458000149311e-37,    2.6930010763544818e-39,
    6.5423159510542658e-41,    1.5419825663364127e-42,    3.5291191213733702e-44,    7.8497535690333607e-46,
    1.6982137697737615e-47,    3.5760251419219180e-49,    7.3348088677591341e-51,    1.4663899127196830e-52,
    2.8593041607187135e-54,    5.4410880692499895e-56,    1.0110610494164595e-57,    1.8355882120725065e-59,
    3.2576808539588107e-61,    5.6545256406277180e-63,    9.6039005297245590e-65,    1.5968478151645109e-66,
    2.6003752543938810e-68,    4.1490582143267679e-70,    6.4890491065372770e-72,    9.9517786939371135e-74,
    1.4971731478938818e-75,    2.2103060347318394e-77,    3.2032690459714595e-79,    4.5587010596221424e-81,
    6.3728889514458769e-83,    8.7541713236842021e-85,    1.1819753466157133e-86,    1.5690734286060896e-88
},
// x = 1.5900
{
    3.0138815876692665e-01,    1.3799035773186155e-01,    4.1028993235112450e-02,    8.9683664264767361e-03,
    1.5456190304978050e-03,    2.1957946139482205e-04,    2.6515838458155244e-05,    2.7832979004711893e-06,
    2.5831109521948597e-07,    2.1481159130774035e-08,    1.6179986882239478e-09,    1.1136513536339710e-10,
    7.0564785647445790e-12,    4.1421453407980194e-13,    2.2646853955489169e-14,    1.1587072186788672e-15,
    5.5706924643339205e-17,    2.5257638171092940e-18,    1.0835005288306480e-19,    4.4103852392339680e-21,
    1.7079229808063607e-22,    6.3071126263736221e-24,    2.2258554348771229e-25,    7.5217729478069968e-27,
    2.4382742046143487e-28,    7.5945813854191107e-30,    2.2764017440676643e-31,    6.5755718602295648e-33,
    1.8328615983182977e-34,    4.9359417300110059e-36,    1.2857362387419169e-37,    3.2429524476759480e-39,
    7.9281607786133835e-41,    1.8804312616733667e-42,    4.3309338262389222e-44,    9.6941255081637692e-46,
    2.1104861154805608e-47,    4.4722735674714786e-49,    9.2311195628039127e-51,    1.8571754217844958e-52,
    3.6441967048421704e-54,    6.9785516678678421e-56,    1.3049536910335792e-57,    2.3841371851841739e-59,
    4.2579725355994757e-61,    7.4375330773811883e-63,    1.2712145825700625e-64,    2.1270283864818970e-66,
    3.4856548968414584e-68,    5.5967583018857062e-70,    8.8085981828419749e-72,    1.3594561810540894e-73,
    2.0581420513332559e-75,    3.0576979627884207e-77,    4.4593801023055774e-79,    6.3864737139254199e-81,
    8.9845296141588398e-83,    1.2419755609605133e-84,    1.6875065827995015e-86,    2.2543399484287966e-88
},
// x = 1.6000
{
    2.9976181125676055e-01,    1.3788705670776405e-01,    4.1223579929702943e-02,    9.0633694274425865e-03,
    1.5713386846415653e-03,    2.2458932633378411e-04,    2.7287066096799164e-05,    2.8819142972909770e-06,
    2.6911955969624478e-07,    2.2518975518377421e-08,    1.7067254155128471e-09,    1.1820443977131361e-10,
    7.5365938002139381e-12,    4.4516164297076187e-13,    2.4491075082333858e-14,    1.2609071034607014e-15,
    6.0999952782778422e-17,    2.7830773158960749e-18,    1.2013649755179475e-19,    4.9208100108223940e-21,
    1.9175353799885414e-22,    7.1255996017573542e-24,    2.5304870162522647e-25,    8.6048685478615580e-27,
    2.8068803179316266e-28,    8.7975741959545096e-30,    2.6535429711253722e-31,    7.7131041017144556e-33,
    2.1634361610283875e-34,    5.8627780508290612e-36,    1.5367547851717257e-37,    3.9004323618545827e-39,
    9.5954269148353822e-41,    2.2901777027075188e-42,    5.3077847476507465e-44,    1.1955302831341707e-45,
    2.6191162428666067e-47,    5.5849732628006437e-49,    1.1600259288061249e-50,    2.3484804211679739e-52,
    4.6372085443740640e-54,    8.9359557860448200e-56,    1.6814803633148466e-57,    3.0913559346955110e-59,
    5.5557382416334054e-61,    9.7653778692879846e-63,    1.6795784758648797e-64,    2.8279783233717356e-66,
    4.6634636291093636e-68,    7.5349822418648378e-70,    1.1933669554935888e-71,    1.8533353115579159e-73,
    2.8234867818053013e-75,    4.2211099818748813e-77,    6.1948142647338645e-79,    8.9276402493795952e-81,
    1.2638417267713709e-82,    1.7580540567816535e-84,    2.4037345955769445e-86,    3.2313376601409640e-88
},
// x = 1.6100
{
    2.9815060302464164e-01,    1.3778069944452503e-01,    4.1416380457203805e-02,    9.1583998879918461e-03,
    1.5972505094130382e-03,    2.2967033226679631e-04,    2.8074326223745773e-05,    2.9832261123274396e-06,
    2.8029412131620084e-07,    2.3598744392401730e-08,    1.7996222754349832e-09,    1.2541036498889325e-10,
    8.0456327365092770e-12,    4.7817963937038813e-13,    2.6471082471709132e-14,    1.3713215569937752e-15,
    6.6754355679938078e-17,    3.0645772186466349e-18,    1.3311179631558079e-19,    5.4862474190013462e-21,
    2.1511970623143972e-22,    8.0437199398369277e-24,    2.8743454635399290e-25,    9.8351038557689634e-27,
    3.2281888744551862e-28,    1.0181194383613271e-29,    3.0900324410451502e-31,    9.0379006696055889e-33,
    2.5508458382726709e-34,    6.9557763420973862e-36,    1.8346321624491330e-37,    4.6855402714761890e-39,
    1.1598823062760032e-40,    2.7856188090971858e-42,    6.4963416723586251e-44,    1.4723780863476790e-45,
    3.2457636204171229e-47,    6.9644479208751895e-49,    1.4555827423469932e-50,    2.9652400852259678e-52,
    5.8916015536898923e-54,    1.1424091459591657e-55,    2.1630993662450993e-57,    4.0016378007606062e-59,
    7.2366024402557000e-61,    1.2799300602514976e-62,    2.2151431820208365e-64,    3.7530293511396448e-66,
    6.2275743533189742e-68,    1.0125047300777856e-69,    1.6135907288616755e-71,    2.5216104033677243e-73,
    3.8655776307078899e-75,    5.8151377562118073e-77,    8.5874925035282306e-79,    1.2453159349329772e-80,
    1.7739459157822473e-82,    2.4830460133669616e-84,    3.4162052013308386e-86,    4.6210929423990485e-88
},
// x = 1.6200
{
    2.9655435336451014e-01,    1.3767135443517492e-01,    4.1607400706778846e-02,    9.2534510191908930e-03,
    1.6233530929910011e-03,    2.3482272479644797e-04,    2.8877801163267907e-05,    3.0872833628166588e-06,
    2.9184410015068726e-07,    2.4721818025498752e-08,    1.8968517034794896e-09,    1.3299964706092109e-10,
    8.5851094046843759e-12,    5.1339081579191829e-13,    2.8595808152399335e-14,    1.4905463971153840e-15,
    7.3006726117311069e-17,    3.3723465775649913e-18,    1.4738647856128857e-19,    6.1121906713647959e-21,
    2.4114758398799805e-22,    9.0728050018834194e-24,    3.2621665405526036e-25,    1.1231278126194322e-26,
    3.7093064098055324e-28,    1.1771084189933310e-29,    3.5947203820837938e-31,    1.0579236202380358e-32,
    3.0043874484877262e-34,    8.2433280716976968e-36,    2.1877186718965008e-37,    5.6219614701306062e-39,
    1.4003224012653818e-40,    3.3839341892624192e-42,    7.9406373092344627e-44,    1.8108908847798396e-45,
    4.0167648289970136e-47,    8.6722726883790961e-49,    1.8237695622501713e-50,    3.7383492887736800e-52,
    7.4737821856161717e-54,    1.4581959655958471e-55,    2.7781643535018628e-57,    5.1713814116523293e-59,
    9.4100279968707482e-61,    1.6746726307448861e-62,    2.9163056494744635e-64,    4.9716530584287684e-66,
    8.3009051932052501e-68,    1.3579736083416846e-69,    2.1775866727338138e-71,    3.4241114344244091e-73,
    5.2816813064140538e-75,    7.9947656344756225e-77,    1.1879552480543864e-78,    1.7334099121637197e-80,
    2.4845638665281842e-82,    3.4993119035618799e-84,    4.8442881450403249e-86,    6.5935436609044600e-88
},
// x = 1.6300
{
    2.9497288404369570e-01,    1.3755908893563215e-01,    4.1796646738851216e-02,    9.3485161170698359e-03,
    1.6496450091035243e-03,    2.4004674165160921e-04,    2.9697672804320466e-05,    3.1941364637773309e-06,
    3.0377896587875834e-07,    2.5889580379240259e-08,    1.9985810900681643e-09,    1.4098964830071120e-10,
    9.1566048434050364e-12,    5.5092370860945410e-13,    3.0874700794444884e-14,    1.6192159475512805e-15,
    7.9796270463485516e-17,    3.7086314193648551e-18,    1.6308047712363643e-19,    6.8046380307773789e-21,
    2.7011933203042552e-22,    1.0225384613303300e-23,    3.6992198622781151e-25,    1.2814441369863722e-26,
    4.2582402928677594e-28,    1.3596317684571562e-29,    4.1777044435322904e-31,    1.2370720938945100e-32,
    3.5348015569522794e-34,    9.7584391733270604e-36,    2.6057825578228176e-37,    6.7375765668184310e-39,
    1.6885473347118504e-40,    4.1056001760863376e-42,    9.6934822237295799e-44,    2.2242654278061842e-45,
    4.9641026106263159e-47,    1.0783690594266987e-48,    2.2817850433488192e-50,    4.7060336621347287e-52,
    9.4664268229336300e-54,    1.8583697566669459e-55,    3.5624178197412025e-57,    6.6721263012184871e-59,
    1.2215730785765957e-60,    2.1874059229638123e-62,    3.8326882379258230e-64,    6.5741911656400463e-66,
    1.1044297307032900e-67,    1.8179215775267723e-69,    2.9331245083306684e-71,    4.6406009488510424e-73,
    7.2022776835439741e-75,    1.0969195742479243e-76,    1.6399875606668256e-78,    2.4077601059961572e-80,
    3.4724308662116884e-82,    4.9208218069149241e-84,    6.8541961526987539e-86,    9.3867890416371404e-88
},
// x = 1.6400
{
    2.9340601926610482e-01,    1.3744396898627029e-01,    4.1984124779025071e-02,    9.4435885624135994e-03,
    1.6761248175035399e-03,    2.4534261269905475e-04,    3.0534122570855934e-05,    3.3038362227576014e-06,
    3.1610833831690144e-07,    2.7103447521429824e-08,    2.1049828857022613e-09,    1.4939837524231069e-10,
    9.7617694991242266e-12,    5.9091336541705695e-13,    3.3317751404384222e-14,    1.7580052175798211e-15,
    8.7164974521770677e-17,    4.0758522027276896e-18,    1.8032385380169536e-19,    7.5701352504183232e-21,
    3.0234479784497901e-22,    1.1515304293849961e-23,    4.1913648183951157e-25,    1.4608145814595432e-26,
    4.8840056756921461e-28,    1.5689832344507204e-29,    4.8504953880796194e-31,    1.4450907420622868e-32,
    4.1544848219507443e-34,    1.1539441891626364e-35,    3.1002389876007876e-37,    8.0651694527007352e-39,
    2.0336490633096946e-40,    4.9749944610940580e-42,    1.1818139602922762e-43,    2.7284086448473985e-45,
    6.1265673029365983e-47,    1.3390526865942483e-48,    2.8507483897290647e-50,    5.9155239218872030e-52,
    1.1972322346185754e-53,    2.3647143198468221e-55,    4.5608494000187519e-57,    8.5944788588215380e-59,
    1.5831761317195951e-60,    2.8522903432465378e-62,    5.0283150352832917e-64,    8.6779275052600990e-66,
    1.4667852850618722e-67,    2.4291728819584444e-69,    3.9433802598155149e-71,    6.2772195010916538e-73,
    9.8020730063692930e-75,    1.5020275015589760e-76,    2.2594291247492918e-78,    3.3375401219086711e-80,
    4.8428614037598846e-82,    6.9049614728504837e-84,    9.6768804155993388e-86,    1.3333714777866104e-87
},
// x = 1.6500
{
    2.9185358563598784e-01,    1.3732605943311035e-01,    4.2169841212150852e-02,    9.5386618205321218e-03,
    1.7027910644387826e-03,    2.5071055995694774e-04,    3.1387331392464552e-05,    3.4164338344998034e-06,
    3.2884198792088065e-07,    2.8364868042246586e-08,    2.2162347071320544e-09,    1.5824449692951518e-10,
    1.0402325690328017e-11,    6.3350162151500680e-13,    3.5935520082447452e-14,    1.9076321871425224e-15,
    9.5157778557667004e-17,    4.4766159891801922e-18,    1.9925757505695181e-19,    8.4158212364236959e-21,
    3.3816401421003499e-22,    1.2957853022824319e-23,    4.7451119097693768e-25,    1.6638723453297961e-26,
    5.5967442845036329e-28,    1.8088911438686752e-29,    5.6262034550006623e-31,    1.6863977169479799e-32,
    4.8777318407284107e-34,    1.3630810599838404e-35,    3.6844141195270997e-37,    9.6432488594312244e-39,
    2.4463731988124209e-40,    6.0211065338132255e-42,    1.4390305367474440e-43,    3.3424710511895675e-45,
    7.5511472041897235e-47,    1.6604699419894168e-48,    3.5565587832835601e-50,    7.4250979042243253e-52,
    1.5119079276693364e-53,    3.0044411203114513e-55,    5.8300048236498015e-57,    1.1053020676134714e-58,
    2.0484671424084826e-60,    3.7130595071361567e-62,    6.5856574551205708e-64,    1.1434869772899380e-65,
    1.9445555724425616e-67,    3.2400439946280495e-69,    5.2917566573192772e-71,    8.4749529927887621e-73,
    1.3314557602343686e-74,    2.0527003882331950e-76,    3.1065998617539174e-78,    4.6169168060815753e-80,
    6.7401038994998969e-82,    9.6686278770444504e-84,    1.3632579234624505e-85,    1.8898743107139536e-87
},
// x = 1.6600
{
    2.9031541212227518e-01,    1.3720542394864343e-01,    4.2353802576534108e-02,    9.6337294410105585e-03,
    1.7296422831160764e-03,    2.5615079761014684e-04,    3.2257479675343994e-05,    3.5319808755253383e-06,
    3.4198983625960855e-07,    2.9675323469106440e-08,    2.3325194445349723e-09,    1.6754736354351486e-10,
    1.1080070136875762e-11,    6.7883738574738883e-13,    3.8739163876069522e-14,    2.0688602015955172e-15,
    1.0382276193639611e-16,    4.9137293659551200e-18,    2.2003434095685916e-19,    9.3494771576555751e-21,
    3.7794990350539210e-22,    1.4569902401918275e-23,    5.3676899787045166e-25,    1.8935592177115503e-26,
    6.4078562681994356e-28,    2.0835722791866333e-29,    6.5197478067697268e-31,    1.9660517240090307e-32,
    5.7210103542644978e-34,    1.6084095929087946e-35,    4.3738493476983908e-37,    1.1517000798668010e-38,
    2.9394060352309952e-40,    7.2783715586938101e-42,    1.7500446738555447e-43,    4.0894806195587241e-45,
    9.2946910452508494e-47,    2.0562442375647517e-48,    4.4309357715499186e-50,    9.3065678557318780e-52,
    1.9064908100479042e-53,    3.8114934499352320e-55,    7.4408508028991786e-57,    1.4192436314347180e-58,
    2.6462285124028090e-60,    4.8256153200745936e-62,    8.6107790408758937e-64,    1.5041706682193843e-65,
    2.5734094998973712e-67,    4.3138213477619433e-69,    7.0881659356122526e-71,    1.1420736846085811e-72,
    1.8051214746757655e-74,    2.7998074740359241e-76,    4.2629562851358372e-78,    6.3738319371875387e-80,
    9.3613436585820483e-82,    1.3510129720461973e-83,    1.9164425993558830e-85,    2.6728454837725338e-87
},
// x = 1.6700
{
    2.8879133002347346e-01,    1.3708212505229095e-01,    4.2536015558280182e-02,    9.7287850574401084e-03,
    1.7566769941599349e-03,    2.6166353202728863e-04,    3.3144747273603090e-05,    3.6505292986418246e-06,
    3.5556195646094346e-07,    3.1036328680126163e-08,    2.4540253696876784e-09,    1.7732702537090477e-10,
    1.1796876555457121e-11,    7.2707693591800316e-13,    4.1740465764853671e-14,    2.2425004804241717e-15,
    1.1321133781815867e-16,    5.3902121611554509e-18,    2.4281947058937076e-19,    1.0379579235259632e-20,
    4.2211120306796020e-22,    1.6370058142049162e-23,    6.0671198525687755e-25,    2.1531593211152837e-26,
    7.3301464359410603e-28,    2.3977920068286052e-29,    7.5480917339434644e-31,    2.2898397687028778e-32,
    6.7032741435690927e-34,    1.8958993409726971e-35,    5.1866515098635435e-37,    1.3739391662523313e-38,
    3.5277090313942650e-40,    8.7876481144988907e-42,    2.1256561162657948e-43,    4.9970951971061679e-45,
    1.1425893523947232e-46,    2.5429381352163766e-48,    5.5126771671058715e-50,    1.1648306226510466e-51,
    2.4005689362465316e-53,    4.8281525003483223e-55,    9.4823248059967758e-57,    1.8195152601393273e-58,
    3.4129717380101284e-60,    6.2613030159325210e-62,    1.1239868133669675e-63,    1.9752528709656266e-65,
    3.3997060208945884e-67,    5.7332589511797519e-69,    9.4772049461467914e-71,    1.5361993185185438e-72,
    2.4426822300291537e-74,    3.8114993733833289e-76,    5.8382911887657260e-78,    8.7817832067289832e-80,
    1.2975603410747164e-81,    1.8838928701181251e-83,    2.6884368798329024e-85,    3.7721179208086323e-87
},
// x = 1.6800
{
    2.8728117293311350e-01,    1.3695622413050940e-01,    4.2716486985775383e-02,    9.8238223871303636e-03,
    1.7838937060654753e-03,    2.6724896177960722e-04,    3.4049313460903873e-05,    3.7721314273747499e-06,
    3.6956857362932770e-07,    3.2449432316077564e-08,    2.5809462451171068e-09,    1.8760425211371578e-10,
    1.2554698322187927e-11,    7.7838422401444679e-13,    4.4951864812887014e-14,    2.4294147443735148e-15,
    1.2337845837571683e-16,    5.9093119933619629e-18,    2.6779184734264471e-19,    1.1515355458480711e-20,
    4.7109562791386971e-22,    1.8378824868410910e-23,    6.8522949620921545e-25,    2.4463362806941487e-26,
    8.3779863406648034e-28,    2.7569313335802831e-29,    8.7305065818026053e-31,    2.6643762258881754e-32,
    7.8463184782223274e-34,    2.2324564913142315e-35,    6.1438956306823586e-37,    1.6372444593267362e-38,
    4.2289082070956683e-40,    1.0597363432942053e-41,    2.5787427675839839e-43,    6.0984946506953041e-45,
    1.4027663972800748e-46,    3.1406625140165584e-48,    6.8491780839762233e-50,    1.4558921941224241e-51,
    3.0183616148225980e-53,    6.1070126152899407e-55,    1.2065727448305207e-56,    2.3290848974235939e-58,
    4.3949437901684931e-60,    8.1110382242619102e-62,    1.4647520266274028e-63,    2.5895054316458096e-65,
    4.4835992292677968e-67,    7.6063855666537176e-69,    1.2648774897142360e-70,    2.0625630145541548e-72,
    3.2992721972532610e-74,    5.1789127086590128e-76,    7.9803161924651389e-78,    1.2075599762750375e-79,
    1.7949206479283983e-81,    2.6215951367443469e-83,    3.7635789506485981e-85,    5.3122468550256876e-87
},
// x = 1.6900
{
    2.8578477670573982e-01,    1.3682778145654426e-01,    4.2895223824300249e-02,    9.9188352308037658e-03,
    1.8112909156456133e-03,    2.7290727766145022e-04,    3.4971356902446170e-05,    3.8968399503257722e-06,
    3.8402006523519065e-07,    3.3916217190718339e-08,    2.7134814342150589e-09,    1.9840055254303313e-10,
    1.3355571203365474e-11,    8.3293119147297195e-13,    4.8386487525085942e-14,    2.6305179655799279e-15,
    1.3438283101628983e-16,    6.4745196997100523e-18,    2.9514492762015536e-19,    1.2766846487717144e-20,
    5.2539328822116148e-22,    2.0618785310266291e-23,    7.7330695403070924e-25,    2.7771741401265400e-26,
    9.5674938013283921e-28,    3.1670616348757314e-29,    1.0088867679736398e-30,    3.0972146033687188e-32,
    9.1751835660136708e-34,    2.6260633522742260e-35,    7.2700876598663020e-37,    1.9488714940802649e-38,
    5.0637470227080272e-40,    1.2764853464084032e-41,    3.1246434557782797e-43,    7.4334375336632381e-45,
    1.7199948901737439e-46,    3.8738068772353645e-48,    8.4982631189627288e-50,    1.8171721458366375e-51,
    3.7897745192756531e-53,    7.7134074002321237e-55,    1.5330150196405687e-56,    2.9768280754151304e-58,
    5.6506365763201880e-60,    1.0490500157416740e-61,    1.9057223265598620e-63,    3.3891296021130452e-65,
    5.9030170341622567e-67,    1.0073990731698750e-68,    1.6851854156502608e-70,    2.7642831380197309e-72,
    4.4480490450806824e-74,    7.0237036605235532e-76,    1.0887393054126813e-77,    1.6572528322972273e-79,
    2.4780017604132205e-81,    3.6408125170022977e-83,    5.2578730514784220e-85,    7.4655828646969262e-87
},
// x = 1.7000
{
    2.8430197942343349e-01,    1.3669685620984096e-01,    4.3072233170773098e-02,    1.0013817472272986e-02,
    1.8388671084725470e-03,    2.7863866271244742e-04,    3.5911055627298383e-05,    4.0247079154598624e-06,
    3.9892696147604768e-07,    3.5438300699387884e-08,    2.8518360123006343e-09,    2.0973819449769107e-10,
    1.4201616155402069e-11,    8.9089809471951695e-13,    5.2058180445032977e-14,    2.8467812454249383e-15,
    1.4628714610763898e-16,    7.0895856884148494e-18,    3.2508781674512090e-19,    1.4144971021039234e-20,
    5.8554038010327900e-22,    2.3114795018978057e-23,    8.7203550559839350e-25,    3.1502223726479437e-26,
    1.0916731604321907e-27,    3.6350278728061855e-29,    1.1647985903350302e-30,    3.5969735263842394e-32,
    1.0718612107173555e-33,    3.0859375084009568e-35,    8.5936956643545700e-37,    2.3172994266691824e-38,
    6.0566125805399663e-40,    1.5357929333127543e-41,    3.7816080720535191e-43,    9.0495112632576696e-45,
    2.1063091635533516e-46,    4.7719138815028954e-48,    1.0530393608980278e-49,    2.2650115530053337e-51,
    4.7516861907816513e-53,    9.7283857407656793e-55,    1.9449173489975746e-56,    3.7989957778017976e-58,
    7.2539212253778408e-60,    1.3546656922220095e-61,    2.4754611306165359e-63,    4.4283841413959942e-65,
    7.7587513053822559e-67,    1.3319259720015172e-68,    2.2412331381243231e-70,    3.6981346883103177e-72,
    5.9859167748492535e-74,    9.5079796225030910e-76,    1.4825418620073131e-77,    2.2700366914991978e-79,
    3.4143402842571503e-81,    5.0462019885542598e-83,    7.3305670573909117e-85,    1.0470137616934956e-86
},
// x = 1.7100
{
    2.8283262136285975e-01,    1.3656350649511706e-01,    4.3247522248619205e-02,    1.0108763078101816e-02,
    1.8666207593135010e-03,    2.8444329224129470e-04,    3.6868587001078516e-05,    4.1557887243234974e-06,
    4.1429994560924333e-07,    3.7017335225756605e-08,    2.9962208786143542e-09,    2.2164022522944232e-10,
    1.5095042194954062e-11,    9.5247384122502924e-13,    5.5981544032555215e-14,    3.0792348249699027e-15,
    1.5915831672659164e-16,    7.7585372637460660e-18,    3.5784641599969521e-19,    1.5661595916405047e-20,
    6.5212316940455533e-22,    2.5894193839685225e-23,    9.8262255866917051e-25,    3.5705453654438610e-26,
    1.2445927284597395e-27,    4.1685412025043521e-29,    1.3435978882651744e-30,    4.1734786415313557e-32,
    1.2509567784808976e-33,    3.6227132617003015e-35,    1.0147759059393067e-36,    2.7524276478036688e-38,
    7.2361464321744781e-40,    1.8456706613254134e-41,    4.5713263965404188e-43,    1.1003609669991540e-44,
    2.5761826842860914e-46,    5.8707253783846942e-48,    1.3031323629812415e-49,    2.8194164223791850e-51,
    5.9495147272853372e-53,    1.2252357703416065e-54,    2.4639121769562076e-56,    4.8410343355050217e-58,
    9.2979573435378393e-60,    1.7465953011152243e-61,    3.2104194743685736e-63,    5.7769224810567287e-65,
    1.0180960037231464e-66,    1.7580154926832474e-68,    2.9756059073987037e-70,    4.9387486296047512e-72,
    8.0410126250979129e-74,    1.2847370007672633e-75,    2.0150179577262198e-77,    3.1034935113513339e-79,
    4.6953807710926853e-81,    6.9803103862217260e-83,    1.0199850371412291e-84,    1.4653952554187533e-86
},
// x = 1.7200
{
    2.8137654496283115e-01,    1.3642778936110420e-01,    4.3421098402765625e-02,    1.0203666097250361e-02,
    1.8945503325607466e-03,    2.9032133385111110e-04,    3.7844127698988765e-05,    4.2901361261959905e-06,
    4.3014985425628959e-07,    3.8655008546618260e-08,    3.1468528692275418e-09,    2.3413049209591852e-10,
    1.6038149340260656e-11,    1.0178563363158877e-12,    6.0171967860090252e-14,    3.3289712329707030e-15,
    1.7306773096712442e-16,    8.4856969735471544e-18,    3.9366464494405118e-19,    1.7329611378610184e-20,
    7.2578228951797484e-22,    2.8987035449152937e-23,    1.1064032891510261e-24,    4.0437767876113374e-26,
    1.4177716060675736e-27,    4.7762819537142248e-29,    1.5484686290535648e-30,    4.8379223282380998e-32,
    1.4585822332428124e-33,    4.2486483055255459e-35,    1.1970586730027535e-36,    3.2658024343840899e-38,
    8.6359529230190073e-40,    2.2155739408614842e-41,    5.5195487329905872e-43,    1.3363677431667732e-44,
    3.1470024765870828e-46,    7.2134321992807620e-48,    1.6105292262879976e-49,    3.5048490379767649e-51,
    7.4391233355323099e-53,    1.5409555458523164e-54,    3.1169224077616414e-56,    6.1598387708678881e-58,
    1.1900062180006398e-59,    2.2484567948084283e-61,    4.1570446827317163e-63,    7.5240240116421931e-65,
    1.3337460688749319e-66,    2.3165301963588318e-68,    3.9438608699976745e-70,    6.5840641833613038e-72,
    1.0782485661980138e-73,    1.7328199431912561e-75,    2.7336898019754283e-77,    4.2349891613049858e-79,
    6.4447113325827852e-81,    9.6369280582878249e-83,    1.4164098903448369e-84,    2.0468274537612943e-86
},
// x = 1.7300
{
    2.7993359479237928e-01,    1.3628976081896216e-01,    4.3592969094757107e-02,    1.0298520660704989e-02,
    1.9226542826558755e-03,    2.9627294746633064e-04,    3.8837853679206706e-05,    4.4278042121762233e-06,
    4.4648767767876362e-07,    4.0353044234615280e-08,    3.3039548708500914e-09,    2.4723366360260514e-10,
    1.7033331624704262e-11,    1.0872528409828986e-12,    6.4645667167685246e-14,    3.5971485766146728e-15,
    1.8809151736450859e-16,    9.2757020315592076e-18,    4.3280574336857625e-19,    1.9163011537638658e-20,
    8.0721737556593942e-22,    3.2426336364946539e-23,    1.2448532002156383e-24,    4.5761792863487447e-26,
    1.6137409185798784e-27,    5.4680140704756650e-29,    1.7830134107523591e-30,    5.6030433152551263e-32,
    1.6990619718277328e-33,    4.9778589440997574e-35,    1.4106556318631105e-36,    3.8708779928317444e-38,
    1.0295419879662963e-39,    2.6566507618449362e-41,    6.6568135582929808e-43,    1.6210767454257190e-44,
    3.8396319793407341e-46,    8.8521657095590145e-48,    1.9878856012795524e-49,    4.3511836326833088e-51,
    9.2891352243287780e-53,    1.9353484600105467e-54,    3.9374104053481006e-56,    7.8265391465825503e-58,
    1.5207767198832496e-59,    2.8901252178513384e-61,    5.3744344121716795e-63,    9.7839512320863888e-65,
    1.7444292340667654e-66,    3.0474341397849107e-68,    5.2183615516933516e-70,    8.7623966335704851e-72,
    1.4433247698839880e-73,    2.3330013266245399e-75,    3.7019214511041434e-77,    5.7682853002342634e-79,
    8.8290586713118785e-81,    1.3279022097385458e-82,    1.9630574429334580e-84,    2.8532636736340806e-86
},
// x = 1.7400
{
    2.7850361751932440e-01,    1.3614947586037418e-01,    4.3763141897989556e-02,    1.0393320981093898e-02,
    1.9509310545083435e-03,    3.0229828536108816e-04,    3.9849940156636438e-05,    4.5688474092068401e-06,
    4.6332456002574177e-07,    4.2113202058788771e-08,    3.4677559355194841e-09,    2.6097525079497547e-10,
    1.8083080183601223e-11,    1.1608803409349654e-12,    6.9419720817281881e-14,    3.8849939802674113e-15,
    2.0431082401191337e-16,    1.0133524869050514e-17,    4.7555365744916963e-19,    2.1176980763574793e-20,
    8.9719205870006783e-22,    3.6248345929508466e-23,    1.3996018214103246e-24,    5.1747099931115610e-26,
    1.8353290181208258e-27,    6.2547121965924194e-29,    2.0513053264957791e-30,    6.4833285305590377e-32,
    1.9773426983830659e-33,    5.8265875800307420e-35,    1.6607028554006517e-36,    4.5833168445472014e-38,
    1.2260668576987293e-39,    3.1820313048242152e-41,    8.0192997905958841e-43,    1.9641464844678168e-44,
    4.6790781122726498e-46,    1.0849775948059609e-47,    2.4505484538546269e-49,    5.3948591626964286e-51,
    1.1583742395806221e-52,    2.4273577521553596e-54,    4.9669113869745233e-56,    9.9299415056402395e-58,
    1.9406341544035785e-59,    3.7093365644632427e-61,    6.9376724176774598e-63,    1.2702722563253648e-64,
    2.2779147516138202e-66,    4.0023962110909908e-68,    6.8932116549811615e-70,    1.1641584869830466e-71,
    1.9286573228986953e-73,    3.1355075107512433e-75,    5.0040536775716257e-77,    7.8422997361851490e-79,
    1.2072941431686693e-80,    1.8262776759696926e-82,    2.7154123467726975e-84,    3.9696003632731657e-86
},
// x = 1.7500
{
    2.7708646187933755e-01,    1.3600698847532669e-01,    4.3931624493063287e-02,    1.0488061352288752e-02,
    1.9793790839082739e-03,    3.0839749218906253e-04,    4.0880561577022523e-05,    4.7133204740381400e-06,
    4.8067179955275114e-07,    4.3937278382844513e-08,    3.6384913961535555e-09,    2.7538162900182932e-10,
    1.9189986415227829e-11,    1.2389659271461156e-12,    7.4512110687751330e-14,    4.1938071776652856e-15,
    2.2181211196628034e-16,    1.1064494872568951e-17,    5.2221451490138042e-19,    2.3387986082623557e-20,
    9.9653934576965891e-22,    4.0492838877262295e-23,    1.5724476426418969e-24,    5.8470923613546078e-26,
    2.0856941638021398e-27,    7.1487027086169048e-29,    2.3579458623558862e-30,    7.4952397676477414e-32,
    2.2990782380259368e-33,    6.8135066489728779e-35,    1.9531392293673404e-36,    5.4213351781471854e-38,
    1.4585652343526945e-39,    3.8071648371801871e-41,    9.6498240370786660e-43,    2.3770739891709593e-44,
    5.6952810136353218e-46,    1.3281949116517424e-47,    3.0171065564281401e-49,    6.6802682336170076e-51,
    1.4426109667444364e-52,    3.0403303999057581e-54,    6.2569136056303312e-56,    1.2580771710254760e-57,
    2.4728125322500164e-59,    4.7536890975331777e-61,    8.9420153276391517e-63,    1.6466662735149347e-64,
    2.9698428437916736e-66,    5.2481152755010847e-68,    9.0905936534599684e-70,    1.5440812184744234e-71,
    2.5727675819369634e-73,    4.2066931224457710e-75,    6.7521564154826716e-77,    1.0642693737081665e-78,
    1.6478165337322857e-80,    2.5069764103268850e-82,    3.7489266033331411e-84,    5.5119552701339896e-86
},
// x = 1.7600
{
    2.7568197864548522e-01,    1.3586235166958002e-01,    4.4098424663246538e-02,    1.0582736148993003e-02,
    2.0079967979335289e-03,    3.1457070501473579e-04,    4.1929891591429217e-05,    4.8612784871337253e-06,
    4.9854084881223230e-07,    4.5827106561028340e-08,    3.8164029829491253e-09,    2.9048005993079980e-10,
    2.0356745217082756e-11,    1.3217471881471269e-12,    7.9941762552959192e-14,    4.5249642631400175e-15,
    2.4068746356122587e-16,    1.2074321367029392e-17,    5.7311819416412209e-19,    2.5813876079106940e-20,
    1.1061674111841322e-21,    4.5203432203868198e-23,    1.7653743850791491e-24,    6.6018949003597206e-26,
    2.3683605512689751e-27,    8.1638201222343089e-29,    2.7081294851691718e-30,    8.6574680316859563e-32,
    2.6707252673110511e-33,    7.9600636899415006e-35,    2.2948257965708266e-36,    6.4060995539746561e-38,
    1.7333425912616378e-39,    4.5502112470192257e-41,    1.1599006349992599e-42,    2.8735302607664331e-44,
    6.9240480371078351e-46,    1.6239726478397807e-47,    3.7100491542284216e-49,    8.2614286484585143e-51,
    1.7942495226115951e-52,    3.8030046210595469e-54,    7.8711608919543305e-56,    1.5916902861474236e-57,
    3.1464092937733403e-59,    6.0831372795009279e-61,    1.1508140313841800e-62,    2.1313182095203352e-64,
    3.8658874536282813e-66,    6.8705610020247487e-68,    1.1968899893617327e-69,    2.0445858011685745e-71,
    3.4261708694219630e-73,    5.6340747933442941e-75,    9.0948937741832561e-77,    1.4417172876224695e-78,
    2.2449715642775743e-80,    3.4349919425324590e-82,    5.1660182738806985e-84,    7.6388546404063641e-86
},
// x = 1.7700
{
    2.7429002059824942e-01,    1.3571561748183564e-01,    4.4263550290053334e-02,    1.0677339826317585e-02,
    2.0367826153510722e-03,    3.2081805334603381e-04,    4.2998103031087626e-05,    5.0127768465201093e-06,
    5.1694331481551927e-07,    4.7784557331504855e-08,    4.0017389406084134e-09,    3.0629871411690237e-10,
    2.1586158298383152e-11,    1.4094726143154819e-12,    8.5728588485962949e-14,    4.8799216076149425e-15,
    2.6103490626621630e-16,    1.3169117905822590e-17,    6.2861999288825450e-19,    2.8473986689591608e-20,
    1.2270658294562972e-21,    5.0427928174548344e-23,    1.9805687186820514e-24,    7.4486174157228180e-26,
    2.6872580098904122e-27,    9.3155804329754617e-29,    3.1077156432454185e-30,    9.9912187377655196e-32,
    3.0996512843817515e-33,    9.2908728073387850e-35,    2.6936819354898403e-36,    7.5621822043097294e-38,
    2.0577610745109083e-39,    5.4324956538527023e-41,    1.3926631655352040e-42,    3.4697543086879908e-44,
    8.4081572250754563e-46,    1.9832497585620191e-47,    4.5565532025006974e-49,    1.0203992820541767e-50,
    2.2287232758389835e-52,    4.7507107701517520e-54,    9.8884689776693254e-56,    2.0109786156335590e-57,
    3.9978160806478251e-59,    7.7730954510426259e-61,    1.4788713547238162e-62,    2.7544346971516072e-64,
    5.0244947295311127e-66,    8.9803746506571284e-68,    1.5733147567863933e-69,    2.7028746443718211e-71,
    4.5550045702899941e-73,    7.5328911504513677e-75,    1.2229143674144367e-76,    1.9495658087964772e-78,
    3.0530088105285780e-80,    4.6978851551956512e-82,    7.1054585343819071e-84,    1.0566324686273879e-85
},
// x = 1.7800
{
    2.7291044249601559e-01,    1.3556683700060487e-01,    4.4427009348928785e-02,    1.0771866919344380e-02,
    2.0657349470126021e-03,    3.2713965916830734e-04,    4.4085367882613142e-05,    5.1678712615823486e-06,
    5.3589095916634621e-07,    4.9811539207134196e-08,    4.1947541463746898e-09,    3.2286669372490124e-10,
    2.2881137569786282e-11,    1.5024020144198651e-12,    9.1893530833268330e-14,    5.2602199452682862e-15,
    2.8295875275328035e-16,    1.4355427932188474e-17,    6.8910240125973437e-19,    3.1389254317597526e-20,
    1.3603122786646886e-21,    5.6218685433357914e-23,    2.2204395442436097e-24,    8.3977854165598995e-26,
    3.0467657137470190e-27,    1.0621373099006207e-28,    3.5633089755782116e-30,    1.1520531273326199e-31,
    3.5942562920256176e-33,    1.0834158412712961e-34,    3.1588406207608130e-36,    8.9180831393799110e-38,
    2.4404085083721121e-39,    6.4790357738878528e-41,    1.6703238188191987e-42,    4.1850155437660819e-44,
    1.0198659686840910e-45,    2.4191553425577834e-47,    5.5894232547487513e-49,    1.2587660653464806e-50,
    2.7648748402134211e-52,    5.9268300217100524e-54,    1.2406166070492409e-55,    2.5372352389584962e-57,
    5.0724868188614703e-59,    9.9182952776104480e-61,    1.8976600830943179e-62,    3.5543940277754104e-64,
    6.5203456249700447e-66,    1.1719737187023512e-67,    2.0648299401166163e-69,    3.5673020921444709e-71,
    6.0457219032219436e-73,    1.0054638393916691e-74,    1.6415203619076666e-76,    2.6316833585173998e-78,
    4.1444732659107890e-80,    6.4134139425679824e-82,    9.7549402698250178e-84,    1.4588213908628699e-85
},
// x = 1.7900
{
    2.7154310104602036e-01,    1.3541606038078388e-01,    4.4588809905041826e-02,    1.0866312042678175e-02,
    2.0948521962444911e-03,    3.3353563697961861e-04,    4.5191857263594735e-05,    5.3266177468078379e-06,
    5.5539569816589942e-07,    5.1909998863542280e-08,    4.3957102288584088e-09,    3.4021405570618614e-10,
    2.4244708611324005e-11,    1.6008069446778137e-12,    9.8458607803903633e-14,    5.6674886369169978e-15,
    3.0656995785510557e-16,    1.5640251878736730e-17,    7.5497698595158789e-19,    3.4582336720675853e-20,
    1.5070797469185088e-21,    6.2633020307767328e-23,    2.4876389665554836e-24,    9.4610534026135068e-26,
    3.4517602826640064e-27,    1.2100673533267142e-28,    4.0823486046523659e-30,    1.3272636810411548e-31,
    4.1641098331767725e-33,    1.2622257837468436e-34,    3.7008252961161193e-36,    1.0506828357633428e-37,
    2.8912931501635665e-39,    7.7191531288560763e-41,    2.0011969045887302e-42,    5.0421558604146246e-44,
    1.2356415192421120e-45,    2.9474299858975808e-47,    6.8482124033279039e-49,    1.5509073710526211e-50,
    3.4256819224791412e-52,    7.3845641122242447e-54,    1.5544291014193965e-55,    3.1968708472214717e-57,
    6.4271198773179881e-59,    1.2637573818015082e-60,    2.4315117349000326e-62,    4.5798880875375201e-64,
    8.4487279911483238e-66,    1.5271090940168833e-67,    2.7056275912600526e-69,    4.7006207685281956e-71,
    8.0111551477846568e-73,    1.3398159170825426e-74,    2.1996657828637307e-76,    3.5463031077474201e-78,
    5.6162107622377064e-80,    8.7396734037426621e-82,    1.3367877661654765e-83,    2.0103511526771175e-85
},
// x = 1.8000
{
    2.7018785487575209e-01,    1.3526333685994127e-01,    4.4748960109183313e-02,    1.0960669889987474e-02,
    2.1241327592320370e-03,    3.4000609382729631e-04,    4.6317741398559154e-05,    5.4890726154803617e-06,
    5.7546960288945701e-07,    5.4081921524383128e-08,    4.6048756876347721e-09,    3.5837183531075788e-10,
    2.5680014219532278e-11,    1.7049711505875580e-12,    1.0544696071890343e-13,    6.1034501163357287e-15,
    3.3198649312150913e-16,    1.7031075774723526e-17,    8.2668639077403850e-19,    3.8077742146070145e-20,
    1.6686442758523953e-21,    6.9733640543348968e-23,    2.7850850946133850e-24,    1.0651317801439242e-25,
    3.9076686820262827e-27,    1.3775278145475942e-28,    4.6732074747663071e-30,    1.5278358664027781e-31,
    4.8201052022646685e-33,    1.4692190189667332e-34,    4.3317512070444984e-36,    1.2366654683202149e-37,
    3.4220679237439442e-39,    9.1871807934623484e-41,    2.3950728440689954e-42,    6.0682245312024249e-44,
    1.4953900947063246e-45,    3.5869248934383695e-47,    8.3805577367111975e-49,    1.9085282896903026e-50,
    4.2391319593120528e-52,    9.1890799987965362e-54,    1.9450709780969647e-55,    4.0226022275286513e-57,
    8.1323479145051843e-59,    1.6079809121979355e-60,    3.1110806156253648e-62,    5.8926079149724322e-64,
    1.0931049454692751e-65,    1.9868199435620858e-67,    3.5397651012759616e-69,    6.1841451279598405e-71,
    1.0598336100548283e-72,    1.7824026400065448e-74,    2.9426293982645493e-76,    4.7705994634720211e-78,
    7.5972912346961905e-80,    1.1888550238536101e-81,    1.8285822981281046e-83,    2.7653007034236838e-85
},
// x = 1.8100
{
    2.6884456450479605e-01,    1.3510871477432213e-01,    4.4907468193764920e-02,    1.1054935233534872e-02,
    2.1535750253980418e-03,    3.4655112934572224e-04,    4.7463189595310469e-05,    5.6552924733264812e-06,
    5.9612489923463893e-07,    5.6329331343689062e-08,    4.8225260135932839e-09,    3.7737206995482372e-10,
    2.7190318034751113e-11,    1.8151910217973369e-12,    1.1288290296763172e-13,    6.5699245258886313e-15,
    3.5933373970496285e-16,    1.8535901433505680e-17,    9.0470646037791932e-19,    4.1901967216724563e-20,
    1.8463932772243690e-21,    7.7589113852128020e-23,    3.1159868146966107e-24,    1.1982840386978554e-25,
    4.4205263651626542e-27,    1.5669564162529089e-28,    5.3453027925810054e-30,    1.7572559943812684e-31,
    5.5746328589209666e-33,    1.7086299696937414e-34,    5.0655543944084147e-36,    1.4541793130038220e-37,
    4.0462883730125094e-39,    1.0923282203244865e-40,    2.8634690489398094e-42,    7.2952210986354114e-44,
    1.8077340054764588e-45,    4.3601925701279077e-47,    1.0243769704211639e-48,    2.3457898831769491e-50,
    5.2392748757810054e-52,    1.1420105898701287e-53,    2.4307343875989523e-55,    5.0549072144897598e-57,
    1.0276049436545974e-58,    2.0431269063790865e-60,    3.9749348610669109e-62,    7.5706063752436193e-64,
    1.4121779279948162e-65,    2.5810152714017402e-67,    4.6239280822861820e-69,    8.1230835807025708e-71,
    1.3998568035047077e-72,    2.3673172774676245e-74,    3.9299859290682531e-76,    6.4066824972885649e-78,
    1.0259445432224316e-79,    1.6143545186819454e-81,    2.4968328638956962e-83,    3.7968400903711377e-85
},
// x = 1.8200
{
    2.6751309231711867e-01,    1.3495224157457433e-01,    4.5064342468919279e-02,    1.1149102923697305e-02,
    2.1831773777757375e-03,    3.5317083579531299e-04,    4.8628370221647531e-05,    5.8253342121163509e-06,
    6.1737396794133410e-07,    5.8654291785210124e-08,    5.0489438100196906e-09,    3.9724782344441875e-10,
    2.8779008249565463e-11,    1.9317760602778514e-12,    1.2079197071821716e-13,    7.0688345480178845e-15,
    3.8874490033014355e-16,    2.0163278295496962e-17,    9.8954849366350100e-19,    4.6083644096124009e-20,
    2.0418344608439297e-21,    8.6274373815702148e-23,    3.4838706927241896e-24,    1.3471383076347727e-25,
    4.9970411398840878e-27,    1.7810776659804621e-28,    6.1092187300444320e-30,    2.0194643741171534e-31,
    6.4417752947966843e-33,    1.9852982742354185e-34,    5.9182519456016921e-36,    1.7083364238622376e-37,
    4.7797091554805224e-39,    1.2974397620519730e-40,    3.4199215529269522e-42,    8.7609638186558926e-44,
    2.1829203743824045e-45,    5.2941851440331376e-47,    1.2506721709490055e-48,    2.8798054027292512e-50,
    6.4674888520940863e-52,    1.4175070170471776e-53,    3.0337743514660826e-55,    6.3438037345801779e-57,
    1.2967420788547851e-58,    2.5924698507304837e-60,    5.0715348954386420e-62,    9.7125031403452564e-64,
    1.8217177858971086e-65,    3.3479078167143363e-67,    6.0309438766169148e-69,    1.0653357346206646e-70,
    1.8460378296388244e-72,    3.1390982903509601e-74,    5.2399959732473255e-76,    8.5894373580593390e-78,
    1.3830790463011232e-79,    2.1883343215931228e-81,    3.4032661316120657e-83,    5.2037984251062163e-85
},
// x = 1.8300
{
    2.6619330253378143e-01,    1.3479396384120135e-01,    4.5219591318697232e-02,    1.1243167888476803e-02,
    2.2129381933760749e-03,    3.5986529810266501e-04,    4.9813450682460216e-05,    5.9992550032209772e-06,
    6.3922934458334586e-07,    6.1058905998641486e-08,    5.2844189143903601e-09,    4.1803321055537163e-10,
    3.0449601399350009e-11,    2.0550493620656751e-12,    1.2920097543021601e-13,    7.6022104393010440e-15,
    4.2036143112732886e-16,    2.1922337004944495e-17,    1.0817616338552614e-18,    5.0653697488382529e-20,
    2.2566054143227491e-21,    9.5871265851000953e-23,    3.8926101735448144e-24,    1.5134355071876301e-25,
    5.6446632814994531e-27,    2.0229345457002662e-28,    6.9768426626451346e-30,    2.3189111636967416e-31,
    7.4375258516520450e-33,    2.3047508870886506e-34,    6.9082375410958146e-36,    2.0050400567111190e-37,
    5.6406245493404875e-39,    1.5395337213284844e-40,    4.0803237116109556e-42,    1.0510103932067256e-43,
    2.6331150165583959e-45,    6.4210791293807759e-47,    1.5252094351276567e-48,    3.5312329420040116e-50,
    7.9740005488556747e-52,    1.7572892318291266e-53,    3.7816285377831048e-55,    7.9510225773837110e-57,
    1.6341977154393620e-58,    3.2850542086696204e-60,    6.4616905173699387e-62,    1.2442736127670721e-63,
    2.3466259561745283e-65,    4.3362506800570108e-67,    7.8542434040705239e-69,    1.3950308159012044e-70,
    2.4306151183935781e-72,    4.1558403603673166e-74,    6.9753063782299528e-76,    1.1496760664171639e-77,
    1.8613852078173289e-79,    2.9612934963221076e-81,    4.6306531983047245e-83,    7.1194330761253912e-85
},
// x = 1.8400
{
    2.6488506118607957e-01,    1.3463392729974677e-01,    4.5373223197361999e-02,    1.1337125133002170e-02,
    2.2428558435494046e-03,    3.6663459390182243e-04,    5.1018597397205961e-05,    6.1771122911280277e-06,
    6.6170371953183049e-07,    6.3545317192641434e-08,    5.5292485208588415e-09,    4.3976342196983739e-10,
    3.2205746235874453e-11,    2.1853481128476662e-12,    1.3813805821846965e-13,    8.1721952739605131e-15,
    4.5433349413490278e-16,    2.3822824801930808e-17,    1.1819354025233453e-18,    5.5645512059187536e-20,
    2.4924838775228962e-21,    1.0646913612301435e-22,    4.3464572567990956e-24,    1.6990973390609913e-25,
    6.3716624585188074e-27,    2.2959234771786338e-28,    7.9615163389004358e-30,    2.6606186910064193e-31,
    8.5800342616863669e-33,    2.6732947231427696e-34,    8.0566168264015141e-36,    2.3511013466157389e-37,
    6.6502591845779719e-39,    1.8250042376511085e-40,    4.8633192179593603e-42,    1.2595309163486441e-43,
    3.1727472662257365e-45,    7.7792485691067591e-47,    1.8579038180910292e-48,    4.3249825301690042e-50,
    9.8197089890177104e-52,    2.1758556557742579e-53,    4.7079330794856791e-55,    9.9526579446113956e-57,
    2.0567687183346485e-58,    4.1570787535803122e-60,    8.2216085742924003e-62,    1.5918111390453297e-63,
    3.0184542214023662e-65,    5.6081580603940200e-67,    1.0213519512360709e-68,    1.8239802394971624e-70,
    3.1953456102971615e-72,    5.4932057758647557e-74,    9.2703411493824704e-76,    1.5362906327534873e-77,
    2.5009191848054010e-79,    4.0004643277830959e-81,    6.2897994096761568e-83,    9.7231184771246370e-85
},
// x = 1.8500
{
    2.6358823608909743e-01,    1.3447217683571472e-01,    4.5525246625776283e-02,    1.1430969739022034e-02,
    2.2729286943415539e-03,    3.7347879357663686e-04,    5.2243975777767677e-05,    6.3589637869181703e-06,
    6.8480993789060108e-07,    6.6115709004541855e-08,    5.7837373034142282e-09,    4.6247474956953130e-10,
    3.4051227684918593e-11,    2.3230240976585605e-12,    1.4763274611797090e-13,    8.7810504038818347e-15,
    4.9082043130227710e-16,    2.5875142814190233e-17,    1.2907023851648273e-18,    6.1095110893700711e-20,
    2.7513987570821151e-21,    1.1816546647545787e-22,    4.8500768417416411e-24,    1.9062437903969553e-25,
    7.1872120845088281e-27,    2.6038328784012017e-28,    9.0782035109218453e-30,    3.0502509479162313e-31,
    9.8898819816955643e-33,    3.0981211235447248e-34,    9.3875876880648576e-36,    2.7543723449597771e-37,
    7.8332160396668375e-39,    2.1613039957549120e-40,    5.7907577436409521e-42,    1.5078643428817584e-43,
    3.8189142028568939e-45,    9.4144121390129471e-47,    2.2606329867816513e-48,    5.2910588947229640e-50,
    1.2078371431969564e-51,    2.6908623471012915e-53,    5.8538747043183516e-55,    1.2442397117773375e-56,
    2.5852489330374654e-58,    5.2536020796142220e-60,    1.0446668488487266e-61,    2.0335962125179291e-63,
    3.8771269087059968e-65,    7.2426579100168648e-67,    1.3261892915662726e-68,    2.3812371415376664e-70,
    4.1942357503209069e-72,    7.2495892309685188e-74,    1.2300870039793404e-75,    2.0495859010566167e-77,
    3.3546334537102177e-79,    5.3952122280594070e-81,    8.5287951112794674e-83,    1.3255901150947870e-84
},
// x = 1.8600
{
    2.6230269681567442e-01,    1.3430875650923177e-01,    4.5675670187881282e-02,    1.1524696864389792e-02,
    2.3031551068443592e-03,    3.8039796030418706e-04,    5.3489750206693726e-05,    6.5448674617040066e-06,
    7.0856099940338055e-07,    6.8772305866655944e-08,    6.0481975396905088e-09,    4.8620461208570024e-10,
    3.5989970888838635e-11,    2.4684442249659660e-12,    1.5771601030040975e-13,    9.4311611423735182e-15,
    5.2999126085117118e-16,    2.8090385336607302e-17,    1.4087410763027125e-18,    6.7041345639360387e-20,
    3.0354419290274555e-21,    1.3106655864766737e-22,    5.4085839469704280e-24,    2.1372122096362999e-25,
    8.1014817604209898e-27,    2.9508856544270248e-28,    1.0343675701724369e-29,    3.4941910324474942e-31,
    1.1390390723715012e-32,    3.5874235670882094e-34,    1.0928871123382323e-35,    3.2238975918545179e-37,
    9.2179896815348118e-39,    2.5571117482661249e-40,    6.8882227267269666e-42,    1.8033173835413872e-43,
    4.5918540689863208e-45,    1.1380984015928826e-46,    2.7476109505076623e-48,    6.4655649024821967e-50,
    1.4839220340402412e-51,    3.3237865005023136e-53,    7.2698274082266986e-55,    1.5535451298958742e-56,
    3.2453490363217304e-58,    6.6306412902487689e-60,    1.3256093720631604e-61,    2.5944299329668342e-63,
    4.9730952633605968e-65,    9.3401605297664460e-67,    1.7194975171652291e-68,    3.1041193531017251e-70,
    5.4970335083400177e-72,    9.5527581300469879e-74,    1.6296378291146927e-75,    2.7299916866931729e-77,
    4.4924189457462999e-79,    7.2641320213832579e-81,    1.1545239233198462e-82,    1.8041166293870003e-84
},
// x = 1.8700
{
    2.6102831467077403e-01,    1.3414370956945371e-01,    4.5824502527265387e-02,    1.1618301742540766e-02,
    2.3335334375406340e-03,    3.8739215009921823e-04,    5.4756084015821185e-05,    6.7348815400335171e-06,
    7.3297005833308737e-07,    7.1517373369087547e-08,    6.3229492354059963e-09,    5.1099158110577093e-10,
    3.8026045335018315e-11,    2.6219910654190598e-12,    1.6842032629392289e-13,    1.0125042679079664e-14,
    5.7202519688043390e-16,    3.0480381189645426e-17,    1.5367788924183597e-18,    7.3526099014775871e-20,
    3.3468808802532880e-21,    1.4528827125456040e-22,    6.0275840254278260e-24,    2.3945780843629003e-25,
    9.1257385264189387e-27,    3.3417859964409473e-28,    1.1776717943626479e-29,    3.9996273917302784e-31,
    1.3107967949190485e-32,    4.1505302123251445e-34,    1.2712199073719338e-35,    3.7700866733787238e-37,
    1.0837553780326020e-38,    3.0225251370578094e-40,    8.1856422044589688e-42,    2.1544840756250177e-43,
    5.5155002142477057e-45,    1.3743663197625107e-46,    3.3358301734679378e-48,    7.8918951100689691e-50,
    1.8210093211532834e-51,    4.1007243248052956e-53,    9.0173313809886624e-55,    1.9373334468127829e-56,
    4.0688209505836648e-58,    8.3577514462211894e-60,    1.6798724760255773e-61,    3.3054424576228279e-63,
    6.3700290168084673e-65,    1.2028071927686529e-66,    2.2262315626575098e-68,    4.0404930122793904e-70,
    7.1936872822047913e-72,    1.2568377793738265e-73,    2.1556034347617625e-75,    3.6305002071494211e-77,
    6.0063800461989990e-79,    9.7643591686979033e-81,    1.5602365719477631e-82,    2.4512051744661999e-84
},
// x = 1.8800
{
    2.5976496266625076e-01,    1.3397707846872223e-01,    4.5971752343821705e-02,    1.1711779681962126e-02,
    2.3640620386436457e-03,    3.9446141185957474e-04,    5.6043139465282413e-05,    6.9290644932601350e-06,
    7.5805042331326046e-07,    7.4353218618949263e-08,    6.6083202494116164e-09,    5.3687540743652668e-10,
    4.0163669071130261e-11,    2.7840634055392240e-12,    1.7977973625843107e-13,    1.0865346233637773e-14,
    6.1711219312746369e-16,    3.3057737251434084e-17,    1.6755953614038638e-18,    8.0594500400597008e-20,
    3.6881722425428288e-21,    1.6095681322681493e-22,    6.7132166103585004e-24,    2.6811776610629796e-25,
    1.0272457701050202e-26,    3.7817708956400810e-28,    1.3398356494663707e-29,    4.5746498040360070e-31,
    1.5072493494532725e-32,    4.7980530340656481e-34,    1.4773866347978737e-35,    4.4049105193558723e-37,
    1.2730033115968681e-38,    3.5692824518097492e-40,    9.7179951573354639e-42,    2.5714632132229891e-43,
    6.6181286741863764e-45,    1.6579301612544390e-46,    4.0455840755859262e-48,    9.6221540275646057e-50,
    2.2321171944145316e-51,    5.0533495020062010e-53,    1.1171483160500434e-54,    2.4129666735565648e-56,
    5.0948307439942839e-58,    1.0521192188444274e-59,    2.1260144813660958e-61,    4.2056582822041770e-63,
    8.1481748632511883e-65,    1.5467836118105381e-66,    2.8781840361231484e-68,    5.2516687203673406e-70,
    9.4000315593721075e-72,    1.6510941222793968e-73,    2.8469272500068459e-75,    4.8204647755591740e-77,
    8.0177082031342196e-79,    1.3103768667948483e-80,    2.1050285912688805e-82,    3.3247754772602604e-84
},
// x = 1.8900
{
    2.5851251549600712e-01,    1.3380890487647590e-01,    4.6117428390489983e-02,    1.1805126065655883e-02,
    2.3947392584310933e-03,    4.0160578741259018e-04,    5.7351077722896133e-05,    7.1274750328812760e-06,
    7.8381555717172737e-07,    7.7282190595896735e-08,    6.9046464193265847e-09,    5.6389704782356050e-10,
    4.2407213008123746e-11,    2.9550768166327746e-12,    1.9182991336980606e-13,    1.1654865455857397e-14,
    6.6545351182774571e-16,    3.5835884261827582e-17,    1.8260254976055229e-18,    8.8295155264437514e-20,
    4.0619762758596416e-21,    1.7820959764107468e-22,    7.4722025441846956e-24,    3.0001325572484702e-25,
    1.1555444148818219e-26,    4.2766668141159631e-28,    1.5232110727529100e-29,    5.2283561316127174e-31,
    1.7317751936533787e-32,    5.5420555152283761e-34,    1.7155354603937063e-35,    5.1421245396932638e-37,
    1.4939471626181488e-38,    4.2110174615366144e-40,    1.1526127612531853e-41,    3.0661109297494899e-43,
    7.9331145311313773e-45,    1.9979097886902272e-46,    4.9010839479392953e-48,    1.1718839742185942e-49,
    2.7329445917295108e-51,    6.2200633451889865e-53,    1.3823819375234966e-54,    3.0017214060641092e-56,
    6.3716330351015985e-58,    1.3227810691774083e-59,    2.6871464696334385e-61,    5.3439364349412773e-63,
    1.0408541867646029e-64,    1.9863758848504327e-66,    3.7158041437654234e-68,    6.8160696520564002e-70,
    1.2265026976515704e-71,    2.1657762143411567e-73,    3.7542286583589453e-75,    6.3905158388309178e-77,
    1.0685625376136214e-78,    1.7556933103226074e-80,    2.8353926917785878e-82,    4.5021541205387625e-84
},
// x = 1.9000
{
    2.5727084951153539e-01,    1.3363922969291969e-01,    4.6261539470083227e-02,    1.1898336350595447e-02,
    2.4255634415736448e-03,    4.0882531156240308e-04,    5.8680058843942767e-05,    7.3301721038473784e-06,
    8.1027907672662553e-07,    8.0306680503888568e-08,    7.2122716877398822e-09,    5.9209869202666406e-10,
    4.4761205311846690e-11,    3.1354642392078324e-12,    2.0460822836697089e-13,    1.2496543080381381e-14,
    7.1726231864314477e-16,    3.8829125000437508e-17,    1.9889633719279160e-18,    9.6680389209678392e-20,
    4.4711723608333970e-21,    1.9719616011558887e-22,    8.3118950595415562e-24,    3.3548765276270751e-25,
    1.2989964885110658e-26,    4.8329519932726422e-28,    1.7304271589357642e-29,    5.9709709781414707e-31,
    1.9881915790228308e-32,    6.3962410729802335e-34,    1.9904037289690101e-35,    5.9975220797270395e-37,
    1.7516709542538422e-38,    4.9635520165289735e-40,    1.3657694778354056e-41,    3.6523338635543992e-43,
    9.4998145455204821e-45,    2.4051171175794737e-46,    5.9311866541381842e-48,    1.4256840603161085e-49,
    3.3424033501505320e-51,    7.6473735719834603e-53,    1.7085792314430912e-54,    3.7296418433146359e-56,
    7.9586108323220255e-58,    1.6609797532647062e-59,    3.3920140542264006e-61,    6.7813724862559974e-63,
    1.3278110984013112e-64,    2.5474049440444727e-66,    4.7904860971746319e-68,    8.8338712532453181e-70,
    1.5979967311211948e-71,    2.8366865229220710e-73,    4.9432084402857564e-75,    8.4589138971697838e-77,
    1.4219003602538187e-78,    2.3485967128579774e-80,    3.8129720820087237e-82,    6.0864130523789938e-84
},
// x = 1.9100
{
    2.5603984269783725e-01,    1.3346809306245733e-01,    4.6404094432197451e-02,    1.1991406067176039e-02,
    2.4565329294580387e-03,    4.1612001213816593e-04,    6.0030241751324328e-05,    7.5372148778433086e-06,
    8.3745475255489828e-07,    8.3429122119081587e-08,    7.5315482289555706e-09,    6.2152379025070962e-10,
    4.7230335884199343e-11,    3.3256765831790901e-12,    2.1815381831689867e-13,    1.3393477843979914e-14,
    7.7276430465932536e-16,    4.2052684944439125e-17,    2.1653658868810009e-18,    1.0580650747733360e-19,
    4.9188755637183924e-21,    2.1807914620586071e-22,    9.2403350000581652e-24,    3.7491845579937107e-25,
    1.4592894000341682e-26,    5.4578249217610496e-28,    1.9644209251973635e-29,    6.8139774961104817e-31,
    2.2808085163352529e-32,    7.3761646387842324e-34,    2.3073975485021552e-35,    6.9892221019521552e-37,
    2.0520384341664676e-38,    5.8452317521377992e-40,    1.6168247778159774e-41,    4.3464291376734671e-43,
    1.1364596232229162e-44,    2.8923578162450938e-46,    7.1702522091648693e-48,    1.7325800896239575e-49,
    4.0832520814830920e-51,    9.3915454474026894e-53,    2.1092954389950223e-54,    4.6285722939516181e-56,
    9.9287578145773066e-58,    2.0830505098362195e-59,    4.2763279230697901e-61,    8.5942687553614260e-63,
    1.6916311360940098e-64,    3.2624621091300818e-66,    6.1674445066878741e-68,    1.1432862218340712e-69,
    2.0790172161837142e-71,    3.7099827757907985e-73,    6.4990199724411437e-75,    1.1179747641798705e-76,
    1.8891439612715337e-78,    3.1367714487868151e-80,    5.1193713287567406e-82,    8.2147253268726170e-84
},
// x = 1.9200
{
    2.5481937464971616e-01,    1.3329553438688999e-01,    4.6545102170200511e-02,    1.2084330818659398e-02,
    2.4876460605048078e-03,    4.2348991004311821e-04,    6.1401784216109454e-05,    7.7486627465441299e-06,
    8.6535650873340496e-07,    8.6651992133774638e-08,    7.8628365762599411e-09,    6.5221708093149798e-10,
    4.9819460934708613e-11,    3.5261833441461520e-12,    2.3250765765334073e-13,    1.4348931673818284e-14,
    8.3219833648305002e-16,    4.5522765515858443e-17,    2.3562567668802054e-18,    1.1573407077117953e-19,
    5.4084543406207039e-21,    2.4103537250825376e-22,    1.0266310487932693e-23,    4.1872044733151110e-25,
    1.6382870963404316e-26,    6.1592795296346027e-28,    2.2284712812402243e-29,    7.7702637110669956e-31,
    2.6144890074911775e-32,    8.4994700775630113e-34,    2.6726815733776278e-35,    8.1379954778369546e-37,
    2.4018072123754448e-38,    6.8773109409234810e-40,    1.9122486156872846e-41,    5.1674782973034697e-43,
    1.3582036636058103e-44,    3.4747846355400628e-46,    8.6591534797409106e-48,    2.1032921023023407e-49,
    4.9828504760699318e-51,    1.1520577103387583e-52,    2.6009990846754255e-54,    5.7374056057471997e-56,
    1.2371695712259282e-57,    2.6091557936766623e-59,    5.3843984792721928e-61,    1.0877802791941697e-62,
    2.1523061760388386e-64,    4.1726317454789275e-66,    7.9293227626701717e-68,    1.4775838659707508e-69,
    2.7009816812481337e-71,    4.8450903817806028e-73,    8.5318723498376081e-75,    1.4753498415484166e-76,
    2.5060783858255482e-78,    4.1829155733798231e-80,    6.8624551992494694e-82,    1.1069363371779343e-83
},
// x = 1.9300
{
    2.5360932654843488e-01,    1.3312159233838641e-01,    4.6684571618300551e-02,    1.2177106280613177e-02,
    2.5189011704806431e-03,    4.3093501930449167e-04,    6.2794842838462606e-05,    7.9645753148470676e-06,
    8.9399842255276251e-07,    8.9977810496312271e-08,    8.2065057496881917e-09,    6.8422461887593253e-10,
    5.2533607643400299e-11,    3.7374732360325490e-12,    2.4771263154597300e-13,    1.5366337155232609e-14,
    8.9581713550112246e-16,    4.9256600032065156e-17,    2.5627307745521608e-18,    1.2652818831908519e-19,
    5.9435494514768670e-21,    2.6625696646776697e-22,    1.1399421365990796e-23,    4.6734912601290859e-25,
    1.8380473448453970e-26,    6.9461877236764020e-28,    2.5262365163983015e-29,    8.8542848624139622e-31,
    2.9947162285193809e-32,    9.7861564240865454e-34,    3.0932802234941554e-35,    9.4676348044951606e-37,
    2.8087588143874579e-38,    8.0843933456907280e-40,    2.2595700284990788e-41,    6.1378033883220557e-43,
    1.6216317588421906e-44,    4.1703108360434135e-46,    1.0446463900874717e-47,    2.5506268861426477e-49,
    6.0740560421155678e-51,    1.4116560319707642e-52,    3.2036766454413782e-54,    7.1035910798111459e-56,
    1.5397338166639378e-57,    3.2641532261482255e-59,    6.7711417954145340e-61,    1.3750553008076822e-62,
    2.7348743404474026e-64,    5.3296389888415243e-66,    1.0180715597805136e-67,    1.9069920626687736e-69,
    3.5040716453675755e-71,    6.3184109573921486e-73,    1.1184202848797775e-74,    1.9440635115438462e-76,
    3.3194404367986253e-78,    5.5693455812094151e-80,    9.1845835778232389e-82,    1.4892153787474324e-83
},
// x = 1.9400
{
    2.5240958113873391e-01,    1.3294630487222805e-01,    4.6822511748690489e-02,    1.2269728200345286e-02,
    2.5502965928054320e-03,    4.3845534712421449e-04,    6.4209573028958290e-05,    8.1850123940815450e-06,
    9.2339472420407601e-07,    9.3409140746864596e-08,    8.5629333842681387e-09,    7.1759380375580238e-10,
    5.5377978915837271e-11,    3.9600548403743951e-12,    2.6381361165752082e-13,    1.6449305287743396e-14,
    9.6388798739442905e-16,    5.3272512477329529e-17,    2.7859581642570546e-18,    1.3825882912799775e-19,
    6.5280941581304156e-21,    2.9395259019002855e-22,    1.2650148763799482e-23,    5.2130443179336313e-25,
    2.0608405919031353e-26,    7.8283899308677405e-28,    2.8617956440900989e-29,    1.0082243405095855e-30,
    3.4276684185938883e-32,    1.1258876237525785e-33,    3.5771917174901187e-35,    1.1005373252727885e-36,
    3.2818467588976285e-38,    9.4949368337651231e-40,    2.6675431127665337e-41,    7.2834945439486784e-43,
    1.9342848265551210e-44,    5.0000935092953330e-46,    1.2589853327001911e-47,    3.0898691966367088e-49,
    7.3962891513971323e-51,    1.7278499193483581e-52,    3.9415583087987822e-54,    8.7849539012835895e-56,
    1.9140334808847359e-57,    4.0786539257763579e-59,    8.5045384531176676e-61,    1.7360075292276055e-62,
    3.4706553536182587e-64,    6.7985241847639046e-66,    1.3053830428628575e-67,    2.4578275912989643e-69,
    4.5396088454103893e-71,    8.2280383457836058e-73,    1.4639845915935642e-74,    2.5579084795599930e-76,
    4.3901823707437729e-78,    7.4039763191649659e-80,    1.2273361872220969e-81,    2.0003446811905197e-83
},
// x = 1.9500
{
    2.5122002270620403e-01,    1.3276970923933237e-01,    4.6958931568769610e-02,    1.2362192396333601e-02,
    2.5818306588540297e-03,    4.4605089393039241e-04,    6.5646128990278936e-05,    8.4100339951992329e-06,
    9.5355979643870348e-07,    9.6948590348998431e-08,    8.9325058587183796e-09,    7.5237340895434925e-10,
    5.8357958231180415e-11,    4.1944572725490597e-12,    2.8085753434727967e-13,    1.7601633538234884e-14,
    1.0366934830328421e-15,    5.7589979217549779e-17,    3.0271893845127349e-18,    1.5100115243639165e-19,
    7.1663357849000564e-21,    3.2434875387771932e-22,    1.4029930160598926e-23,    5.8113478697516902e-25,
    2.3091705299976300e-26,    8.8167943730295862e-28,    3.2396939745062412e-29,    1.1472288474161734e-30,
    3.9203023067790273e-32,    1.2943269730731883e-33,    4.1335164550219713e-35,    1.2782358611866285e-36,
    3.8313650343994771e-38,    1.1141830534695984e-39,    3.1463378629072480e-41,    8.6350197934142076e-43,
    2.3050150487435830e-44,    5.9890981177083782e-46,    1.5157727019008312e-47,    3.7392435358673758e-49,
    8.9967967763701376e-51,    2.1125672213189120e-52,    4.8439881922094774e-54,    1.0851888173437402e-55,
    2.3765455989475984e-57,    5.0903115739972143e-59,    1.0668644155659726e-60,    2.1889766825749787e-62,
    4.3987790123929134e-64,    8.6609692311961511e-66,    1.6715565548675994e-67,    3.1634854716585580e-69,
    5.8730573513077108e-71,    1.0699748005588958e-72,    1.9135738819720455e-74,    3.3606653349632179e-76,
    5.7976822375286470e-78,    9.8280757482081314e-80,    1.6375655084685102e-81,    2.6826973970709059e-83
},
// x = 1.9600
{
    2.5004053705500762e-01,    1.3259184199856000e-01,    4.7093840118436203e-02,    1.2454494757651357e-02,
    2.6135016982527545e-03,    4.5372165342952536e-04,    6.7104663699296625e-05,    8.6397003219457964e-06,
    9.8450817420120766e-07,    1.0059881101695755e-07,    9.3156184245779973e-09,    7.8861361076469568e-10,
    6.1479114584117670e-11,    4.4412308652357486e-12,    2.9889348138031795e-13,    1.8827314200428403e-14,
    1.1145322919097512e-15,    6.2229693784622312e-17,    3.2877600414963206e-18,    1.6483585841623640e-19,
    7.8628587242712108e-21,    3.5769122484884862e-22,    1.5551240342380932e-23,    6.4744147785812044e-25,
    2.5857965170900574e-26,    9.9234858560590854e-28,    3.6649933195802805e-29,    1.3044736785876970e-30,
    4.4804459923549402e-32,    1.4868338722306789e-33,    4.7726014564652933e-35,    1.4834189430105787e-36,
    4.4691396455754489e-38,    1.3063054472920733e-39,    3.7077594239743459e-41,    1.0227929337583766e-42,
    2.7442047475740296e-44,    7.1667573335426752e-46,    1.8231148408646684e-47,    4.5204587171928393e-49,
    1.0932150573493477e-50,    2.5801638613539092e-52,    5.9464667396827479e-54,    1.3389997567973933e-55,
    2.9474110412585956e-57,    6.3453908303088441e-59,    1.3367271866334186e-60,    2.7567304666020834e-62,
    5.5680741301995776e-64,    1.1019429829068266e-65,    2.1376349048934042e-67,    4.0662885461280241e-69,
    7.5878116286025537e-71,    1.3894593766575791e-72,    2.4976846045419648e-74,    4.4089760107899698e-76,
    7.6451679866593024e-78,    1.3026307261838230e-79,    2.1815830297299338e-81,    3.5922374299601512e-83
},
// x = 1.9700
{
    2.4887101148594279e-01,    1.3241273902880701e-01,    4.7227246467454916e-02,    1.2546631243388544e-02,
    2.6453080391707047e-03,    4.6146761265944495e-04,    6.8585328889538021e-05,    8.8740717640163807e-06,
    1.0162545442356642e-06,    1.0436249903857053e-07,    9.7126753357449407e-09,    8.2636601793913851e-10,
    6.4747207521495312e-11,    4.7009478694013622e-12,    3.1797276320251104e-13,    2.0130543069980938e-14,
    1.1977199693086499e-15,    6.7213634861414751e-17,    3.5690961363064971e-18,    1.7984956022675601e-19,
    8.6226089747796091e-21,    3.9424653844937669e-22,    1.7227678676457058e-23,    7.2088340340361516e-25,
    2.8937580025621372e-26,    1.1161844922206162e-27,    4.1433272701864817e-29,    1.4822317135932096e-30,
    5.1169022866583037e-32,    1.7066864890303629e-33,    5.5062027602619796e-35,    1.7201520964506790e-36,
    5.2087462318873342e-38,    1.5302432898839199e-39,    4.3654997948130258e-41,    1.2103668275812062e-42,
    3.2640202817783196e-44,    8.5677392730977943e-46,    2.1906092791604640e-47,    5.4593494756792484e-49,
    1.3270021114237193e-50,    3.1479007898468600e-52,    7.2918981311369743e-54,    1.6503271735460489e-55,
    3.6512225833554870e-57,    7.9006736006738961e-59,    1.6728490340425401e-60,    3.4675008791638800e-62,
    7.0394002950157938e-64,    1.4002262880834221e-65,    2.7301163224506879e-67,    5.2198063408484507e-69,
    9.7899703246323752e-71,    1.8018531002317242e-72,    3.2555162671683461e-74,    5.7760215380697595e-76,
    1.0066695757412197e-77,    1.7239716353509113e-79,    2.9019469880987909e-81,    4.8027693690887004e-83
},
// x = 1.9800
{
    2.4771133477484553e-01,    1.3223243554088904e-01,    4.7359159712892507e-02,    1.2638597882069636e-02,
    2.6772480086058766e-03,    4.6928875204293189e-04,    7.0088275034032032e-05,    9.1132088901963822e-06,
    1.0488137446654887e-06,    1.0824239559370710e-07,    1.0124089978399782e-08,    8.6568370158817891e-10,
    6.8168192274473739e-11,    4.9742031731062982e-12,    3.3814900484240947e-13,    2.1515728444744596e-14,
    1.2865897984285008e-15,    7.2565137602909372e-17,    3.8727195891941359e-18,    1.9613517857448333e-19,
    9.4509203027984215e-21,    4.3430361754639914e-22,    1.9074063154395822e-23,    8.0218221922145040e-25,
    3.2364011260373005e-26,    1.2546678283814982e-27,    4.6809620243497394e-29,    1.6830440858608731e-30,
    5.8395636250931045e-32,    1.9575877280104521e-33,    6.3476678902419242e-35,    1.9930749561204996e-36,
    6.0657571312430602e-38,    1.7910493992933352e-39,    5.1354265635938108e-41,    1.4310513742802152e-42,
    3.8787063263738907e-44,    1.0232842534762624e-45,    2.6296086607769706e-47,    6.5866317410723671e-49,
    1.6091277228450015e-50,    3.8365112519064824e-52,    8.9320815693080488e-54,    2.0317903348446533e-55,
    4.5179768088098346e-57,    9.8257734932394222e-59,    2.0910114119185759e-60,    4.3562554400503167e-62,
    8.8885229847417554e-64,    1.7770081313776779e-65,    3.4823278533622794e-67,    6.6917587469721081e-69,
    1.2614344828495015e-70,    2.3334591338430327e-72,    4.2373880301393141e-74,    7.5562230169236680e-76,
    1.3236115608970227e-77,    2.2782501881279035e-79,    3.8544152980641557e-81,    6.4114836272142829e-83
},
// x = 1.9900
{
    2.4656139715132341e-01,    1.3205096608921760e-01,    4.7489588976623580e-02,    1.2730390771067890e-02,
    2.7093199326661726e-03,    4.7718504544199005e-04,    7.1613651328538162e-05,    9.3571724414895006e-06,
    1.0822007645469641e-06,    1.1224128706820463e-07,    1.0550285001291671e-08,    9.0662122542809843e-10,
    7.1748224987017957e-11,    5.2616150384261873e-12,    3.5947823450182227e-13,    2.2987500459930792e-14,
    1.3814936687296659e-15,    7.8308968433818803e-17,    4.2002540645114171e-18,    2.1379235998844544e-19,
    1.0353542124798061e-20,    4.7817550768135507e-22,    2.1106531683347053e-23,    8.9212790717588943e-25,
    3.6174076680308258e-26,    1.4094361532234699e-27,    5.2848632872625779e-29,    1.9097500831843749e-30,
    6.6595397680763408e-32,    2.2437174539661195e-33,    7.3121407396049835e-35,    2.3074785093143001e-36,
    7.0580216752867528e-38,    2.0945450246585284e-39,    6.0359138736149462e-41,    1.6904654654076910e-42,
    4.6049266829853674e-44,    1.2210038100417191e-45,    3.1535295618050828e-47,    7.9387909201351379e-49,
    1.9492467765604105e-50,    4.6708748389964565e-52,    1.0929492402807535e-53,    2.4986870434879223e-55,
    5.5842227347160164e-57,    1.2205942933202833e-58,    2.6106396333156109e-60,    5.4662548937022113e-62,
    1.1209654678212175e-63,    2.2523621113526988e-65,    4.4361342967420037e-67,    8.5676476504723734e-69,
    1.6232012713789967e-70,    3.0178266133431446e-72,    5.5078078745567720e-74,    9.8712426431213710e-76,
    1.7378575393243106e-77,    3.0063645461331895e-79,    5.1119356895073752e-81,    8.5461813136694896e-83
},
// x = 2.0000
{
    2.4542109027781644e-01,    1.3186836458327533e-01,    4.7618543402903446e-02,    1.2822006076016617e-02,
    2.7415221368453043e-03,    4.8515646021275402e-04,    7.3161605675156811e-05,    9.6060233242346886e-06,
    1.1164307433966472e-06,    1.1636200536318782e-07,    1.0991692446363043e-08,    9.4923467637578487e-10,
    7.5493668041519827e-11,    5.5638258567868155e-12,    3.8201897489784020e-13,    2.4550720768131777e-14,
    1.4828029917978136e-15,    8.4471403467835660e-17,    4.5534311106897238e-18,    2.3292792007579461e-19,
    1.1336669211727725e-20,    5.2620123537633149e-22,    2.3342651136591303e-23,    9.9158480302731982e-25,
    4.0408265449292446e-26,    1.5822995196548732e-27,    5.9627698093189855e-29,    2.1655201853414506e-30,
    7.5892996299955555e-32,    2.5697907927181400e-33,    8.4167914770429954e-35,    2.6693922200249213e-36,
    8.2059839644918585e-38,    2.4474315650686581e-39,    7.0882215118518452e-41,    1.9951434797717712e-42,
    5.4621586620607093e-44,    1.4555681196095831e-45,    3.7782135247794510e-47,    9.5591256949395301e-49,
    2.3588752783354512e-50,    5.6808176809570052e-52,    1.3359407382948711e-53,    3.0695432037999786e-55,
    6.8944464188117070e-57,    1.5145474287674178e-58,    3.2556179199576307e-60,    6.8509598711970657e-62,
    1.4119811390231151e-63,    2.8513469349579497e-65,    5.6440621893008994e-67,    1.0955293609953511e-68,
    2.0859801748417789e-70,    3.8976920341651277e-72,    7.1493656343526892e-74,    1.2877634429128747e-75,
    2.2785261862389783e-77,    3.9614768785096904e-79,    6.7698109590691803e-81,    1.1374674542204951e-82
},
// x = 2.0100
{
    2.4429030722897938e-01,    1.3168466429889158e-01,    4.7746032156006805e-02,    1.2913440030217656e-02,
    2.7738529462936561e-03,    4.9320295726099528e-04,    7.4732284666318541e-05,    9.8598226032138411e-06,
    1.1515189706928479e-06,    1.2060742819970777e-07,    1.1448753879689472e-08,    9.9358169548940658e-10,
    7.9411095482338223e-11,    5.8815029230105870e-12,    4.0583233741988988e-13,    2.6210492574364484e-14,
    1.5909096560596446e-15,    9.1080310698666098e-17,    4.9340966301357912e-18,    2.5365631308191693e-19,
    1.2406973322478904e-20,    5.7874779752144292e-22,    2.5801534706741640e-23,    1.1014981167795636e-24,
    4.5111080552833793e-26,    1.7752575313770423e-27,    6.7232741774480836e-29,    2.4538925474491444e-30,
    8.6428287066459202e-32,    2.9411231764207536e-33,    9.6810743660983801e-35,    3.0856822316915449e-36,
    9.5330428860734911e-38,    2.8574177871834043e-39,    8.3169287954761040e-41,    2.3526782886230246e-42,
    6.4731490962684113e-44,    1.7335919683261787e-45,    4.5223487826786282e-47,    1.1498973497887264e-48,
    2.8517362453233278e-50,    6.9020615096777452e-52,    1.6312438146084381e-53,    3.7667712838522597e-55,
    8.5027383656514482e-57,    1.8771816483836349e-58,    4.0552809295193079e-60,    8.5763621798584768e-62,
    1.7764166508719469e-63,    3.6052079888844881e-65,    7.1719366018494954e-67,    1.3990496466811188e-68,
    2.6772183591906659e-70,    5.0274262629316437e-72,    9.2676648575923907e-74,    1.6776585656238317e-75,
    2.9832265354150589e-77,    5.2125964422065482e-79,    8.9523763519405961e-81,    1.5117000321740112e-82
},
// x = 2.0200
{
    2.4316894247138035e-01,    1.3149989788932245e-01,    4.7872064417931140e-02,    1.3004688934047284e-02,
    2.8063106860841277e-03,    5.0132449109820457e-04,    7.6325833569151921e-05,    1.0118631494751809e-05,
    1.1874808853513551e-06,    1.2498047941862429e-07,    1.1921920522710901e-08,    1.0397215092534440e-09,
    8.3507298538025718e-11,    6.2153392283739284e-12,    4.3098211916626118e-13,    2.7972171036515176e-14,
    1.7062270217212746e-15,    9.8165236118114803e-17,    5.3442176945332174e-18,    2.7610012914004199e-19,
    1.3571636878942035e-20,    6.3621229022839946e-22,    2.8503968139469224e-23,    1.2229009827980917e-24,
    5.0331410998765123e-26,    1.9905179765615555e-27,    7.5759115280320338e-29,    2.7788132660218814e-30,
    9.8358037150309025e-32,    3.3637028695967050e-33,    1.1127016703895674e-34,    3.5641619846431629e-36,
    1.1065959711552570e-37,    3.3333645891184752e-39,    9.7504308130707072e-41,    2.7718856438272398e-42,
    7.6644412026939196e-44,    2.0628329527021798e-45,    5.4079624630557667e-47,    1.3819148046844814e-48,
    3.4441675075469160e-50,    8.3773482655981917e-52,    1.9897548506297427e-53,    4.6174584928402966e-55,
    1.0474799510015094e-56,    2.3240552593863261e-58,    5.0456187202626548e-60,    1.0723832257967472e-61,
    2.2322623076543616e-63,    4.5528598252461114e-65,    9.1021508324699086e-67,    1.7844090111561328e-68,
    3.4316171688400988e-70,    6.4761151152713937e-72,    1.1997562953484896e-73,    2.1826304701422755e-75,
    3.9004690397252182e-77,    6.8491821677999469e-79,    1.1821620173269122e-80,    2.0061268470427625e-82
},
// x = 2.0300
{
    2.4205689184351073e-01,    1.3131409739613911e-01,    4.7996649386162320e-02,    1.3095749154359904e-02,
    2.8388936814730242e-03,    5.0952100989822479e-04,    7.7942396310229324e-05,    1.0382511359810600e-05,
    1.2243320751756431e-06,    1.2948412927566223e-07,    1.2411653383730712e-08,    1.0877149612065243e-09,
    8.7789291243004131e-11,    6.5660542729762880e-12,    4.5753490292541569e-13,    2.9841374041780580e-14,
    1.8291909573379817e-15,    1.0575749392174704e-16,    5.7858897216543034e-18,    3.0039062065877610e-19,
    1.4838388800968385e-20,    6.9902418601576507e-22,    3.1472545462022373e-23,    1.3569220793091445e-24,
    5.6122936145609912e-26,    2.2305171737328996e-27,    8.5312569069113375e-29,    3.1446807954566897e-30,
    1.1185786215873981e-31,    3.8442717876416687e-33,    1.2779542432684241e-34,    4.1137167365510428e-36,
    1.2835319249969196e-37,    3.8854496150942863e-39,    1.1421505569337174e-40,    3.2629930043116281e-42,
    9.0669828285331387e-44,    2.4523812430576721e-45,    6.4609945762267513e-47,    1.6591624199389571e-48,
    4.1556020800724980e-50,    1.0157771514083903e-51,    2.4245644618415146e-53,    5.6543093287906864e-55,
    1.2890352169352570e-56,    2.8741412420232045e-58,    6.2707395933800179e-60,    1.3393593417176344e-61,
    2.8017873122713412e-63,    5.7427132196916292e-65,    1.1537716335516925e-66,    2.2730724639979309e-68,
    4.3929960200401906e-70,    8.3314328748728656e-72,    1.5511056983343499e-73,    2.8357754421226558e-75,
    5.0927431654347687e-77,    8.9870471315346383e-79,    1.5588298363273556e-80,    2.6584196971761860e-82
},
// x = 2.0400
{
    2.4095405253610164e-01,    1.3112729425992697e-01,    4.8119796271503132e-02,    1.3186617123889740e-02,
    2.8716002581560107e-03,    5.1779245555440528e-04,    7.9582115460688087e-05,    1.0651523697079268e-05,
    1.2620882762817244e-06,    1.3412139473156916e-07,    1.2918423389658280e-08,    1.1376245439103330e-09,
    9.2264316159429082e-11,    6.9343948977219344e-12,    4.8556016016821067e-13,    3.1823993369917633e-14,
    1.9602609194623391e-15,    1.1389026096803159e-16,    6.2613440304239601e-18,    3.2666825936176015e-19,
    1.6215542625606127e-20,    7.6764776869589824e-22,    3.4731814859245900e-23,    1.5047938595919674e-24,
    6.2544564725065574e-26,    2.4979421761771338e-27,    9.6010320637219663e-29,    3.5563949160339689e-30,
    1.2712437159797362e-31,    4.3904155023545168e-33,    1.4666834360281379e-34,    4.7444436429263022e-36,
    1.4876051244263607e-37,    4.5253543129006132e-39,    1.3367961697857237e-40,    3.8378562724339861e-42,
    1.0716828111505430e-43,    2.9128795913520680e-45,    7.7119668287170173e-47,    1.9901511010359070e-48,
    5.0091335289493890e-50,    1.2304351295318044e-51,    2.9513842856758740e-53,    6.9167716685697798e-55,
    1.5846034917026056e-56,    3.5505528084147032e-58,    7.7846441957582646e-60,    1.6708956425552406e-61,
    3.5125269961117936e-63,    7.2349244326059581e-65,    1.4607273471380405e-66,    2.8919786382046920e-68,
    5.6166216214265459e-70,    1.0704507057353646e-71,    2.0027236969353085e-73,    3.6794609836236606e-75,
    6.6404408125838967e-77,    1.1775923394388707e-78,    2.0526245706578906e-80,    3.5177685566892007e-82
},
// x = 2.0500
{
    2.3986032307274119e-01,    1.3093951933079973e-01,    4.8241514295961095e-02,    1.3277289340650823e-02,
    2.9044287425192025e-03,    5.2613876373725670e-04,    8.1245132221726961e-05,    1.0925730136061304e-05,
    1.3007653724978937e-06,    1.3889533973730741e-07,    1.3442711517971484e-08,    1.1895144312578957e-09,
    9.6939850199977667e-11,    7.3211361362171162e-12,    5.1513035711805835e-13,    3.3926206254337511e-14,
    2.0999210768570880e-15,    1.2259867566240507e-16,    6.7729557916359004e-18,    3.5508332556202349e-19,
    1.7712037041307991e-20,    8.4258473586351245e-22,    3.8308435390402429e-23,    1.6678614400453696e-24,
    6.9660911313425679e-26,    2.7957549927562224e-27,    1.0798222534404384e-28,    4.0194106906975733e-30,
    1.4437754484069260e-31,    5.0106634197764226e-33,    1.6820739346879808e-34,    5.4678092414522548e-36,
    1.7228019489944864e-37,    5.2664763477884796e-39,    1.5633377660554628e-40,    4.5102083822889924e-42,
    1.2655946285446193e-43,    3.4567782032677546e-45,    9.1967612958762352e-47,    2.3849360361914185e-48,
    6.0321807971137916e-50,    1.4889895269449841e-51,    3.5890538731243105e-53,    8.4523808856245329e-55,
    1.9458875178009417e-56,    4.3814180984923932e-58,    9.6533749456240128e-60,    2.0821475507666858e-61,
    4.3984908515634055e-63,    9.1041612448984318e-65,    1.8471284173325628e-66,    3.6748960892123257e-68,
    7.1721153320049805e-70,    1.3736020878231861e-71,    2.5824833668821180e-73,    4.7678543220702849e-75,
    8.6468517576891176e-77,    1.5409141733525479e-78,    2.6990779065338778e-80,    4.6483157467558396e-82
},
// x = 2.0600
{
    2.3877560329078887e-01,    1.3075080287873109e-01,    4.8361812690695033e-02,    1.3367762367335402e-02,
    2.9373774618854033e-03,    5.3455986395257744e-04,    8.2931586410475228e-05,    1.1205192430161031e-05,
    1.3403793947395564e-06,    1.4380907551421077e-07,    1.3985008928874680e-08,    1.2434505111193113e-09,
    1.0182361055227190e-10,    7.7270820868854967e-12,    5.4632106396684098e-13,    3.6154487352298303e-14,
    2.2486814807988232e-15,    1.3191994144334953e-16,    7.3232523923992349e-18,    3.8579653132449079e-19,
    1.9337478974409769e-20,    9.2437697944188994e-22,    4.2231345282933021e-23,    1.8475921933539164e-24,
    7.7542813207725228e-26,    3.1272189954039415e-27,    1.2137205937518365e-28,    4.5397978870806784e-30,
    1.6386336089374344e-31,    5.7146002149619509e-33,    1.9277221279437903e-34,    6.2968263905316308e-36,
    1.9936687035330627e-37,    6.1241706457571869e-39,    1.8267944761131982e-40,    5.2959442129796216e-42,
    1.4933153289606170e-43,    4.0986297696625725e-45,    1.0957526271568434e-46,    2.8553866005857932e-48,
    7.2572693219426851e-50,    1.8001196277490766e-51,    4.3601421769900209e-53,    1.0318362726110360e-54,
    2.3870450890432409e-56,    5.4009336419849602e-58,    1.1957617586238016e-59,    2.5917219843038576e-61,
    5.5016392056588008e-63,    1.1443000858795152e-64,    2.3329676941400290e-66,    4.6641059726784040e-68,
    9.1470780082700635e-70,    1.7603859388109927e-71,    3.3258018246096486e-73,    6.1701219715043364e-75,
    1.1244516145942961e-76,    2.0136001861056446e-78,    3.5442332172769960e-80,    6.1335881725424245e-82
},
// x = 2.0700
{
    2.3769979432258173e-01,    1.3056117460370731e-01,    4.8480700694020447e-02,    1.3458032830711168e-02,
    2.9704447447555628e-03,    5.4305567960002835e-04,    8.4641616446233365e-05,    1.1489972449770607e-05,
    1.3809465203593558e-06,    1.4886576082904092e-07,    1.4545817097627897e-08,    1.2995004183230031e-09,
    1.0692356070564055e-10,    8.1530668056057047e-12,    5.7921106730527405e-13,    3.8515621135678259e-14,
    2.4070792830391957e-15,    1.4189343505338949e-16,    7.9149222330941019e-18,    4.1897967924370771e-19,
    2.1102189372071930e-20,    1.0136095553264869e-21,    4.6531942584500850e-23,    2.0455860982085510e-24,
    8.6267890876717982e-26,    3.4959276968689776e-27,    1.3633892487366977e-28,    5.1243063837098489e-30,
    1.8585670746605525e-31,    6.5129897170240478e-33,    2.2076867165003674e-34,    7.2462529408381149e-36,
    2.3053866800710681e-37,    7.1160227405608131e-39,    2.1329427880985525e-40,    6.2134468989725896e-42,
    1.7605184026409125e-43,    4.8554307220217699e-45,    1.3043729228803135e-46,    3.4155016457260952e-48,
    8.7229480127200748e-50,    2.1741624878042985e-51,    5.2916606545320050e-53,    1.2583543008175135e-54,
    2.9251873279652193e-56,    6.6506318995465826e-58,    1.4795846862865850e-59,    3.2224394184497269e-61,
    6.8736864057499819e-63,    1.4366099581891528e-64,    2.9431274540343212e-66,    5.9124873279252937e-68,
    1.1651604289826534e-69,    2.2532678683553686e-71,    4.2776273132731290e-73,    7.9744735647505675e-75,
    1.4603292764490332e-76,    2.6277561157402521e-78,    4.6476770444956504e-80,    8.0822102531924128e-82
},
// x = 2.0800
{
    2.3663279857692721e-01,    1.3037066364570393e-01,    4.8598187549469582e-02,    1.3548097421017443e-02,
    3.0036289210454950e-03,    5.5162612803213583e-04,    8.6375359337084216e-05,    1.1780132175359439e-05,
    1.4224830724728688e-06,    1.5406860226388198e-07,    1.5125647947023614e-08,    1.3577335679704710e-09,
    1.1224791658090775e-10,    8.5999552191757992e-12,    6.1388248583725734e-13,    4.1016714714038373e-14,
    2.5756800030317984e-15,    1.5256081978386740e-16,    8.5508239763371016e-18,    4.5481635864021512e-19,
    2.3017251833068016e-20,    1.1109138537780602e-21,    5.1244279002342397e-23,    2.2635869004610814e-24,
    9.5921155385271464e-26,    3.9058360965119803e-27,    1.5305878810272262e-28,    5.7804381252983708e-30,
    2.1066459723688850e-31,    7.4179125567158753e-33,    2.5265452235169230e-34,    8.3328146710521788e-36,
    2.6638568040017386e-37,    8.2621585467502444e-39,    2.4884259619952872e-40,    7.2839612851092196e-42,
    2.0737925589683826e-43,    5.7470156528691375e-45,    1.5513379819160158e-46,    4.0817775069655875e-48,
    1.0474864819705805e-49,    2.6234185224640770e-51,    6.4159079501127258e-53,    1.5330620814730856e-54,
    3.5809747204922609e-56,    8.1809049322316738e-58,    1.8288126409050469e-59,    4.0022589543029353e-61,
    8.5783004553313396e-63,    1.8015304224548461e-64,    3.7085407626000812e-66,    7.4860984615231325e-68,
    1.4823898266087135e-69,    2.8805868156865500e-71,    5.4949353918960429e-73,    1.0293266641062558e-74,
    1.8940594059282518e-76,    3.4246761638760761e-78,    6.0864224676696722e-80,    1.0635258118860000e-81
},
// x = 2.0900
{
    2.3557451972087906e-01,    1.3017929859448982e-01,    4.8714282503908010e-02,    1.3637952891360491e-02,
    3.0369283223178685e-03,    5.6027112061368971e-04,    8.8132950666871209e-05,    1.2075733690567155e-05,
    1.4650055192600281e-06,    1.5942085448080466e-07,    1.5725023979985688e-08,    1.4182211890823300e-09,
    1.1780515276387438e-10,    9.0686440599092602e-12,    6.5042088944852973e-13,    4.3665211101907465e-14,
    2.7550788460745370e-15,    1.6396616388843558e-16,    9.2339962682005120e-18,    4.9350268105811279e-19,
    2.5094564245299497e-20,    1.2169709828594102e-21,    5.6405267809216571e-23,    2.5034941442215983e-24,
    1.0659566643420520e-25,    4.3612948072170855e-27,    1.7172616240072354e-28,    6.5165262417977565e-30,
    2.3862972183522753e-31,    8.4429190188255940e-33,    2.8894570570120170e-34,    9.5754552976949174e-36,
    3.0757950274237659e-37,    9.5855951808111389e-39,    2.9008785512945439e-40,    8.5320210348960227e-42,
    2.4407834590545729e-43,    6.7965128393707799e-45,    1.8434449263037471e-46,    4.8736371990799940e-48,
    1.2567027280549267e-49,    3.1625114030075324e-51,    7.7714695624123435e-53,    1.8658871939003026e-54,
    4.3793294302233399e-56,    1.0052834224204262e-57,    2.2580694454063011e-59,    4.9654001475152829e-61,
    1.0693783538997765e-62,    2.2565911383298809e-64,    4.6676205930610996e-66,    9.4673689844058867e-68,
    1.8837251481492704e-69,    3.6780487277344024e-71,    7.0498610342654689e-73,    1.3269443136576262e-74,
    2.4534354139656313e-76,    4.4574055658997411e-78,    7.9598775285429808e-80,    1.3975714522363818e-81
},
// x = 2.1000
{
    2.3452486266179101e-01,    1.2998710749926085e-01,    4.8828994805704018e-02,    1.3727596057108339e-02,
    3.0703412820095913e-03,    5.6899056278152712e-04,    8.9914524582544050e-05,    1.2376839175302201e-05,
    1.5085304732425883e-06,    1.6492582048124604e-07,    1.6344478412267252e-08,    1.4810363585735053e-09,
    1.2360400884315428e-10,    9.5600628216686032e-12,    6.8891542170085833e-13,    4.6468902942460528e-14,
    2.9459020740605290e-15,    1.7615606436646995e-16,    9.9676679526980265e-18,    5.3524805702853095e-19,
    2.7346893596740491e-20,    1.3325153778833412e-21,    6.2054906748244067e-23,    2.7673761352550418e-24,
    1.1839324491695263e-25,    4.8670871928122625e-27,    1.9255594865483720e-28,    7.3418219997038357e-30,
    2.7013437687968799e-31,    9.6031986837350108e-33,    3.3022338427577998e-34,    1.0995616676638246e-35,
    3.5488397663261739e-37,    1.1112638009609557e-38,    3.3790680221723191e-40,    9.9859367576571187e-42,
    2.8703564882466682e-43,    8.0308699424997513e-45,    2.1886516396156939e-46,    5.8139305755405163e-48,
    1.5063278647365751e-49,    3.8088115565853453e-51,    9.4043999093947511e-53,    2.2687361160399501e-54,
    5.3502858006730093e-56,    1.2340386118656140e-57,    2.7851492553488148e-59,    6.1537021113924274e-61,
    1.3316335284306771e-62,    2.8234324507295498e-64,    5.8680165817450007e-66,    1.1959042712400887e-67,
    2.3908704245679294e-69,    4.6905895612380248e-71,    9.0336241331617703e-73,    1.7084635493019333e-74,
    3.1739439944092247e-76,    5.7939992934937978e-78,    1.0396194007187257e-79,    1.8340609180390818e-81
},
// x = 2.1100
{
    2.3348373352964413e-01,    1.2979411787810746e-01,    4.8942333702951084e-02,    1.3817023795285087e-02,
    3.1038661356545239e-03,    5.7778435410466413e-04,    9.1720213781867191e-05,    1.2683510898847000e-05,
    1.5530746905377940e-06,    1.7058685186003827e-07,    1.6984555305221871e-08,    1.5462540355551445e-09,
    1.2965349585299722e-10,    1.0075174737642072e-11,    7.2945892582371719e-13,    4.9435946699984964e-14,
    3.1488084305734734e-15,    1.8917977632397739e-16,    1.0755268801338332e-17,    5.8027601614871148e-19,
    2.9787934134661762e-20,    1.4583386505447685e-21,    6.8236516924753506e-23,    3.0574839031070213e-24,
    1.3142524417053379e-25,    5.4284697628731863e-27,    2.1578544705855561e-28,    8.2665903123114575e-30,
    3.0560479451291409e-31,    1.0915768597663154e-32,    3.7714178204870208e-34,    1.2617552653275488e-35,
    4.0916728265675580e-37,    1.2873329726134738e-38,    3.9330557096075510e-40,    1.1678353484458056e-41,
    3.3727835576840698e-43,    9.4814602430210495e-45,    2.5962675107120971e-46,    6.9295167257144505e-48,
    1.8039025062120825e-49,    4.5829340946087531e-51,    1.1369618873128166e-52,    2.7558755760787046e-54,
    6.5300049363506224e-56,    1.5133043478722645e-57,    3.4316824939373280e-59,    7.6182684775512502e-61,
    1.6564021488494246e-62,    3.5287411344652572e-64,    7.3687722828484207e-66,    1.5089042866269767e-67,
    3.0309784608137299e-69,    5.9746963436203300e-71,    1.1561443806703659e-72,    2.1969363658333670e-74,
    4.1008395233949111e-76,    7.5216610054564873e-78,    1.3560370086745703e-79,    2.4036587775914101e-81
},
// x = 2.1200
{
    2.3245103965964331e-01,    1.2960035672731740e-01,    4.9054308441741350e-02,    1.3906233043965146e-02,
    3.1375012211016929e-03,    5.8665238834476553e-04,    9.3550149501492577e-05,    1.2995811212971761e-05,
    1.5986550700885635e-06,    1.7640734905403535e-07,    1.7645809698624652e-08,    1.6139510959609647e-09,
    1.3596290282173143e-10,    1.0614977780170542e-11,    7.7214807427634308e-13,    5.2574877333773495e-14,
    3.3644906221082519e-15,    2.0308934812614639e-16,    1.1600440780364523e-17,    6.2882507261469371e-19,
    3.2432369056482761e-20,    1.5952936921495675e-21,    7.4996998732083399e-23,    3.3762662327043647e-24,
    1.4581338439227382e-25,    6.0512160904891166e-27,    2.4167655503562392e-28,    9.3022145985146688e-30,
    3.4551592321516511e-31,    1.2399681880516400e-32,    4.3043691777479517e-34,    1.4468680392569032e-35,
    4.7141554279085342e-37,    1.4901957943809630e-38,    4.5743796290772717e-40,    1.3646886907189853e-41,
    3.9599573297280892e-43,    1.1184781239342966e-44,    3.0771743001570001e-46,    8.2519416029161726e-48,
    2.1583254854070512e-49,    5.5093237665913445e-51,    1.3732560340251267e-52,    3.3443848868267791e-54,
    7.9619839338354585e-56,    1.8538957354762906e-57,    4.2239372155363329e-59,    9.4214564195501277e-61,
    2.0581596058502964e-62,    4.4053927822689991e-64,    9.2429714920670214e-66,    1.9016468974435216e-67,
    3.8379809027477618e-69,    7.6012957316719031e-71,    1.4778683661885553e-72,    2.8215847590534980e-74,
    5.2917630677153540e-76,    9.7519937502948210e-78,    1.7664578856192428e-79,    3.1459853207003681e-81
},
// x = 2.1300
{
    2.3142668957507889e-01,    1.2940585053051792e-01,    4.9164928264490267e-02,    1.3995220801667381e-02,
    3.1712448787289431e-03,    5.9559455351692410e-04,    9.5404461505390432e-05,    1.3313802545057844e-05,
    1.6452886528703864e-06,    1.8239076158527530e-07,    1.8328807743517662e-08,    1.6842063674954062e-09,
    1.4254180342642300e-10,    1.1180505682930351e-11,    8.1708350195373799e-13,    5.5894623466307668e-14,
    3.5936768572417410e-15,    2.1793976256235306e-16,    1.2507049879133553e-17,    6.8114963843680543e-19,
    3.5295935924442569e-20,    1.7442990461268054e-21,    8.2387105920206989e-23,    3.7263858407420690e-24,
    1.6169065500639701e-25,    6.7416645385693101e-27,    2.7051816741352877e-28,    1.0461311847698103e-29,
    3.9039669831954931e-31,    1.4076258865100143e-32,    4.9093632851232601e-34,    1.6579973432591518e-35,
    5.4274811173178268e-37,    1.7237628569971338e-38,    5.3162619747302987e-40,    1.5934849014030083e-41,
    4.6456367202759900e-43,    1.3183259093571212e-44,    3.6440816945785517e-46,    9.8182258459330505e-48,
    2.5800897489205773e-49,    6.6169415871938089e-51,    1.6571116572632287e-52,    4.0546920119068414e-54,
    9.6984958296990411e-56,    2.2688717978513336e-57,    5.1937825407626290e-59,    1.1639279408895514e-60,
    2.5546353866945168e-62,    5.4938450754188183e-64,    1.1580981195731998e-65,    2.3938980998902818e-67,
    4.8542338212406051e-69,    9.6593448998338953e-71,    1.8868527794436342e-72,    3.6194087843223480e-74,
    6.8200451508676533e-76,    1.2627653174401998e-77,    2.2981318243224845e-79,    4.1121675601814931e-81
},
// x = 2.1400
{
    2.3041059297044869e-01,    1.2921062526765939e-01,    4.9274202408309396e-02,    1.4083984126749501e-02,
    3.2050954516521547e-03,    6.0461073195072919e-04,    9.7283278073640229e-05,    1.3637547391232605e-05,
    1.6929926210752184e-06,    1.8854058829862883e-07,    1.9034126835056054e-08,    1.7571006649011151e-09,
    1.4940006275435270e-10,    1.1772828985779888e-11,    8.6436994311113934e-13,    5.9404523058845165e-14,
    3.8371324456251751e-15,    2.3378908425133167e-16,    1.3479198523956576e-17,    7.3752098666193074e-19,
    3.8395496005504192e-20,    1.9063435658223029e-21,    9.0461738981265497e-23,    4.1107367770021277e-24,
    1.7920229009728605e-25,    7.5067701016525759e-27,    3.0262879621996581e-28,    1.1757858821644443e-29,
    4.4083585059937214e-31,    1.5969343063037865e-32,    5.5956988944950644e-34,    1.8986401159416010e-35,
    6.2443475641800154e-37,    1.9924913074850032e-38,    6.1738444839731626e-40,    1.8592075123492390e-41,
    5.4457280481204885e-43,    1.5526174296312696e-44,    4.3118227772574744e-46,    1.1671780012436537e-47,
    3.0815577131458109e-49,    7.9400701556636529e-51,    1.9977929585076928e-52,    4.9112083200132653e-54,
    1.1802302780006074e-55,    2.7739862724226586e-57,    6.3798458723466974e-59,    1.4364307018966574e-60,
    3.1675228054574677e-62,    6.8438353509593362e-64,    1.4494421631949719e-65,    3.0101882167134991e-67,
    6.1325515062446734e-69,    1.2260290700840460e-70,    2.4061558244427051e-72,    4.6372025107943572e-74,
    8.7788660077434864e-76,    1.6330767989635129e-77,    2.9860133133927050e-79,    5.3680986563227795e-81
},
// x = 2.1500
{
    2.2940266069483700e-01,    1.2901470642384349e-01,    4.9382140103427508e-02,    1.4172520136802796e-02,
    3.2390512859300090e-03,    6.1370080035160249e-04,    9.9186725991575771e-05,    1.3967108309516783e-05,
    1.7417842972726028e-06,    1.9486037759387571e-07,    1.9762355745329284e-08,    1.8327168255429759e-09,
    1.5654784417187445e-10,    1.2393056102576051e-11,    9.1411637208213218e-13,    6.3114339607753149e-14,
    4.0956614587151094e-15,    2.5069861352119673e-16,    1.4521238602607313e-17,    7.9822826702432341e-19,
    4.1749107747494926e-20,    2.0824913744200110e-21,    9.9280259094687248e-23,    4.5324631358608710e-24,
    1.9850682237981344e-25,    8.3541606930303378e-27,    3.3835942886127382e-28,    1.3207330403579834e-29,
    4.9748830458036491e-31,    1.8105584472489379e-32,    6.3738184693548267e-34,    2.1727419901252826e-35,
    7.1791494528365815e-37,    2.3014578723636283e-38,    7.1644552382938643e-40,    2.1675865855978050e-41,
    6.3786067848399283e-43,    1.8270726056399254e-44,    5.0976953983319729e-46,    1.3863467024284882e-47,
    3.6772824319437328e-49,    9.5192574271013749e-51,    2.4063089232019946e-52,    5.9430795586058787e-54,
    1.4348692544997204e-55,    3.3882260256384516e-57,    7.8289015972457003e-59,    1.7709161321717334e-60,
    3.9233388124603751e-62,    8.5164468961035507e-64,    1.8121021615304974e-65,    3.7809279835392232e-67,
    7.7387174193986828e-69,    1.5543601119584218e-70,    3.0647691869989939e-72,    5.9340785195482394e-74,
    1.1286489996727849e-75,    2.1093584773176666e-77,    3.8748854938241914e-79,    6.9985974822360010e-81
},
// x = 2.1600
{
    2.2840280473554542e-01,    1.2881811899799839e-01,    4.9488750571658822e-02,    1.4260826008047414e-02,
    3.2731107307644096e-03,    6.2286462986238761e-04,    1.0111493053928524e-04,    1.4302547912986113e-05,
    1.7916811435483143e-06,    2.0135372765216363e-07,    2.0514094756133915e-08,    1.9111397453060284e-09,
    1.6399561630122090e-10,    1.3042334412268350e-11,    9.6643614786652557e-13,    6.7034278875180898e-14,
    4.3701084542077949e-15,    2.6873304700621407e-16,    1.5637785125623728e-17,    8.6357957654831583e-19,
    4.5376104612481791e-20,    2.2738871447264093e-21,    1.0890682394677257e-22,    4.9949791686483845e-24,
    2.1977722155307507e-25,    9.2921982314038136e-27,    3.7809664492947135e-28,    1.4826851189931833e-29,
    5.6108222287388096e-31,    2.0514752982201897e-32,    7.2554419336645597e-34,    2.4847521389932518e-35,
    8.2481949342397923e-37,    2.6564411896122678e-38,    8.3079109054383692e-40,    2.5252059305678535e-41,
    7.4654855162643290e-43,    2.1483255340733444e-44,    6.0218562874412453e-46,    1.6452834577153978e-47,
    4.3843819080786501e-49,    1.1402421859047171e-50,    2.8957307596009670e-52,    7.1850735803924587e-54,
    1.7427897206723147e-55,    4.1344535481985792e-57,    9.5975360717293491e-59,    2.1810728706669692e-60,
    4.8544646292489785e-62,    1.0586621616407197e-63,    2.2630550623455912e-65,    4.7437785875367163e-67,
    9.7545810354721753e-69,    1.9683619599621256e-70,    3.8991041783105177e-72,    7.5846257394834889e-74,
    1.4492844092825622e-75,    2.7211909219360670e-77,    5.0220547400812693e-79,    9.1127105871415109e-81
},
// x = 2.1700
{
    2.2741093820197289e-01,    1.2862088751140369e-01,    4.9594043024917114e-02,    1.4348898974728361e-02,
    3.3072721386965516e-03,    6.3210208612515860e-04,    1.0306801548146128e-04,    1.4643928862948266e-05,
    1.8427007606207358e-06,    2.0802428665679299e-07,    2.1289955791672157e-08,    1.9924564148043046e-09,
    1.7175416010578727e-10,    1.3721851373576882e-11,    1.0214471626647410e-12,    7.1175006167898639e-14,
    4.6613602661889487e-15,    2.8796064520952390e-16,    1.6833730551467314e-17,    9.3390308773107052e-19,
    4.9297177498767969e-20,    2.4817617174822490e-21,    1.1941074681551472e-22,    5.5019908932370689e-24,
    2.4320212328317299e-25,    1.0330044907357492e-26,    4.2226601348082215e-28,    1.6635361512524172e-29,
    6.3242675785916253e-31,    2.3230084887189551e-32,    8.2537152543729850e-34,    2.8396844942896678e-35,
    9.4699483707257181e-37,    3.0640145791988045e-38,    9.6268589137382205e-40,    2.9396250608475805e-41,
    8.7308344673381185e-43,    2.5240649227721955e-44,    7.1077757291500532e-46,    1.9509543641124145e-47,
    5.2229750080849154e-49,    1.3646145507631589e-50,    3.4815650737588011e-52,    8.6786288528695584e-54,
    2.1147963410518686e-55,    5.0401729794004482e-57,    1.1754142065203401e-58,    2.6835228847048162e-60,
    6.0004040532814227e-62,    1.3146212564396143e-63,    2.8232059301824978e-65,    5.9453317494938231e-67,
    1.2281873352017429e-68,    2.4898049692589166e-70,    4.9548400034089997e-72,    9.6828551891226289e-74,
    1.8587776365977580e-75,    3.5062059177107967e-77,    6.5007652060751212e-79,    1.1850462325059965e-80
},
// x = 2.1800
{
    2.2642697530973949e-01,    1.2842303601606805e-01,    4.9698026663774250e-02,    1.4436736328512379e-02,
    3.3415338657987463e-03,    6.4141302934324514e-04,    1.0504610305760128e-04,    1.4991313862136787e-05,
    1.8948608869353019e-06,    2.1487575300828660e-07,    2.2090562551152737e-08,    2.0767559558977797e-09,
    1.7983457608442383e-10,    1.4432835663561952e-11,    1.0792719944364194e-12,    7.5547664178384645e-14,
    4.9703478630602845e-15,    3.0845340728856081e-16,    1.8114259803575961e-17,    1.0095482370422896e-18,
    5.3534461993719588e-20,    2.7074380778385050e-21,    1.3086688039126229e-22,    6.0575193032764802e-24,
    2.6898715548490675e-25,    1.1477735037753573e-26,    4.7133579434139896e-28,    1.8653799178278408e-29,
    7.1242057749420714e-31,    2.6288665815207773e-32,    9.3833754133632669e-34,    3.2431860383938401e-35,
    1.0865302408587574e-36,    3.5316505122257507e-38,    1.1147164590999842e-39,    3.4195178267193676e-41,
    1.0202861776217932e-42,    2.9631952335437368e-44,    8.3827617326029520e-46,    2.3115023032448435e-47,
    6.2166887340209662e-49,    1.6317185856791558e-50,    4.1821922150081520e-52,    1.0473091851505596e-53,
    2.5638155540868370e-55,    6.1384426189669727e-57,    1.4381305731004439e-58,    3.2984309696698569e-60,
    7.4093035440469096e-62,    1.6307688783434528e-63,    3.5182708358156269e-65,    7.4431679584582721e-67,
    1.5446903055256497e-68,    3.1458446879407642e-70,    6.2892199509076280e-72,    1.2347124869513985e-73,
    2.3811412009690262e-75,    4.5122222892111519e-77,    8.4045207323566198e-79,    1.5391438637914246e-80
},
// x = 2.1900
{
    2.2545083136505151e-01,    1.2822458810296145e-01,    4.9800710676063169e-02,    1.4524335417885796e-02,
    3.3758942718619751e-03,    6.5079731434343946e-04,    1.0704931397255238e-04,    1.5344765647923013e-05,
    1.9481793977372130e-06,    2.2191187553369050e-07,    2.2916550641268373e-08,    2.1641296585142606e-09,
    1.8824829157523096e-10,    1.5176558340390958e-11,    1.1400380635615443e-12,    8.0163891402476136e-14,
    5.2980482753525842e-15,    3.3028725332753993e-16,    1.9484866008340015e-17,    1.0908869765899441e-18,
    5.8111630710869663e-20,    2.9523377106493132e-21,    1.4335602688768774e-22,    6.6659252858697165e-24,
    2.9735636902251645e-25,    1.2746252945834714e-26,    5.2582096883124628e-28,    2.0905298320533570e-29,
    8.0206123793731116e-31,    2.9731853672049849e-32,    1.0660933478408221e-33,    3.7016129451591034e-35,
    1.2457882745041844e-36,    4.0678381832152069e-38,    1.2898348903993850e-39,    3.9748299020775877e-41,
    1.1914061641856450e-42,    3.4760214587665053e-44,    9.8785638889706693e-46,    2.7364384434848703e-47,
    7.3932480891738935e-49,    1.9494242013756136e-50,    5.0193807640365731e-52,    1.2627176171262526e-53,
    3.1052987405014552e-55,    7.4689610054657101e-57,    1.7578660841967369e-58,    4.0502369590202939e-60,
    9.1397868565017819e-62,    2.0208627615157269e-63,    4.3798523895928866e-65,    9.3083754823004556e-67,
    1.9406330771460661e-68,    3.9703179015131179e-70,    7.9739010540850465e-72,    1.5726280187363328e-73,
    3.0467125447348766e-75,    5.7999337084672715e-77,    1.0852549004654823e-78,    1.9965689941485138e-80
},
// x = 2.2000
{
    2.2448242274930261e-01,    1.2802556691010528e-01,    4.9902104235522687e-02,    1.4611693647553639e-02,
    3.4103517205792888e-03,    6.6025479063837734e-04,    1.0907776738740197e-04,    1.5704346985547622e-05,
    2.0026743041227430e-06,    2.2913645369006734e-07,    2.3768567708526034e-08,    2.2546710177732484e-09,
    1.9700706816935216e-10,    1.5954334030610056e-11,    1.2038777936833161e-12,    8.5035841148161502e-14,
    5.6454865955861457e-15,    3.5354221436927412e-16,    2.0951366984059050e-17,    1.1783150919174802e-18,
    6.3053990976453692e-20,    3.2179873562978958e-21,    1.5696537608569512e-22,    7.3319363632252833e-24,
    3.2855378042766028e-25,    1.4147617336487547e-26,    5.8628762726708547e-28,    2.3415406871401401e-29,
    9.0245548205041983e-31,    3.3605745549622875e-32,    1.2104877650791508e-33,    4.2221154246400058e-35,
    1.4274389322396229e-36,    4.6822157502143840e-38,    1.4914083106963520e-39,    4.6169575758261555e-41,
    1.3901839524574300e-42,    4.0744608537691903e-44,    1.1632068542979134e-45,    3.2368637264948530e-47,
    8.7851614748906135e-49,    2.3270016487647569e-50,    6.0188909144823145e-52,    1.5210681693176258e-53,
    3.7576994535217230e-55,    9.0793584747422925e-57,    2.1466298783218208e-58,    4.9685345636899895e-60,
    1.1263167284158737e-61,    2.5017156745467520e-63,    5.4467486983554783e-65,    1.1628630290084404e-66,
    2.4354261796009874e-68,    5.0053419887799066e-70,    1.0098486943968763e-71,    2.0007301772025673e-73,
    3.8937771739981973e-75,    7.4462873985585625e-77,    1.3996702062223669e-78,    2.5867563759766957e-80
},
// x = 2.2100
{
    2.2352166690390865e-01,    1.2782599513052181e-01,    5.0002216500485336e-02,    1.4698808477840025e-02,
    3.4449045797250708e-03,    6.6978530248906743e-04,    1.1113158091070810e-04,    1.6070120661372766e-05,
    2.0583637520693833e-06,    2.3655333776213324e-07,    2.4647273571404836e-08,    2.3484757714084724e-09,
    2.0612300923522466e-10,    1.6767522141225562e-11,    1.2709287768126302e-12,    9.0176201150326870e-14,
    6.0137380523893052e-15,    3.7830263048653935e-16,    2.2519922511987098e-17,    1.2722535890178500e-18,
    6.8388588142678358e-20,    3.5060261898811305e-21,    1.7178897304545440e-22,    8.0606753809293427e-24,
    3.6284503473986578e-25,    1.5694972669876644e-26,    6.5335774271225182e-28,    2.6212324288710749e-29,
    1.0148305497671694e-30,    3.7961692905930629e-32,    1.3735898350845517e-33,    4.8127322148886289e-35,
    1.6344978086072810e-36,    5.3857189848602863e-38,    1.7232747356104411e-39,    5.3589506014450756e-41,
    1.6209224765032148e-42,    4.7722853937292535e-44,    1.3686098530161687e-45,    3.8257248493907093e-47,
    1.0430516502581492e-48,    2.7753620192130425e-50,    7.2111815551354833e-52,    1.8306518532410743e-53,
    4.5430378516176062e-55,    1.1026731778744278e-56,    2.6188838993739503e-58,    6.0891252701942305e-60,
    1.3866112814065018e-61,    3.0938540661481676e-63,    6.7665448944951134e-65,    1.4511958132956436e-66,
    3.0530949724288113e-68,    6.3032865752046914e-70,    1.2774903161311780e-71,    2.5424820690236928e-73,
    4.9705974371223861e-75,    9.5487266699670864e-77,    1.8030160987886558e-78,    3.3473234389136989e-80
},
// x = 2.2200
{
    2.2256848231537124e-01,    1.2762589502004704e-01,    5.0101056612604979e-02,    1.4785677424089984e-02,
    3.4795512213302020e-03,    6.7938868896755429e-04,    1.1321087059006947e-04,    1.6442149476156400e-05,
    2.1152660214451372e-06,    2.4416642905400329e-07,    2.5553340352317541e-08,    2.4456419374858976e-09,
    2.1560856755374290e-10,    1.7617528096901806e-11,    1.3413339427748798e-12,    9.5598213806532504e-14,
    6.4039301611381581e-15,    4.0465735718144676e-16,    2.4197052421660363e-17,    1.3731501537741227e-18,
    7.4144314817672487e-20,    3.8182134477391771e-21,    1.8792821731899869e-22,    8.8576912729912132e-24,
    4.0051919711263731e-25,    1.7402688072644264e-26,    7.2771436270139272e-28,    2.9327161304707158e-29,
    1.1405465937835968e-30,    4.2856869722910073e-32,    1.5577137603110656e-33,    5.4824957606021727e-35,
    1.8703687888269329e-36,    6.1907482710654728e-38,    1.9898061179835085e-39,    6.2157421978148936e-41,
    1.8885682318990331e-42,    5.5853992279791790e-44,    1.6090332574187966e-45,    4.5181099049744649e-47,
    1.2373903332886607e-48,    3.3073375698948013e-50,    8.6322382348830453e-52,    2.2013087909302232e-53,
    5.4875677240639159e-55,    1.3379465982923383e-56,    3.1920283610902154e-58,    7.4552810589734511e-60,
    1.7053854033777591e-61,    3.8223144850441893e-63,    8.3975464183633328e-65,    1.8091325455566514e-66,
    3.8233466365297579e-68,    7.9292017095464092e-70,    1.6142810276585404e-71,    3.2272944661947437e-73,
    6.3379456116891651e-75,    1.2230512358423719e-76,    2.3198404336355934e-78,    4.3262892631600670e-80
},
// x = 2.2300
{
    2.2162278850056760e-01,    1.2742528840500750e-01,    5.0198633695624773e-02,    1.4872298056072949e-02,
    3.5142900218531978e-03,    6.8906478401969452e-04,    1.1531575090403068e-04,    1.6820496238349639e-05,
    2.1733995249972032e-06,    2.5197968007499541e-07,    2.6487452609350327e-08,    2.5462698524137743e-09,
    2.2547655306475863e-10,    1.8505804602579350e-11,    1.4152417330804264e-12,    1.0131569704913546e-13,
    6.8172449534320768e-15,    4.3269998040958394e-16,    2.5989655523810687e-17,    1.4814806871644857e-18,
    8.0352026315186528e-20,    4.1564365265275044e-21,    2.0549239559785047e-22,    9.7289920417331412e-24,
    4.4189068239862688e-25,    1.9286464364207003e-26,    8.1010725308560545e-28,    3.2794223607403591e-29,
    1.2811103021093453e-30,    4.8354898779048984e-32,    1.7654465201569970e-33,    6.2415492243225461e-35,
    2.1388917609411252e-36,    7.1113561085048576e-38,    2.2959794620860498e-39,    7.2044096735850854e-41,
    2.1988036800376155e-42,    6.5321559731513203e-44,    1.8902361526435839e-45,    5.3315895837913562e-47,
    1.4667485201866506e-48,    3.9380082254999481e-50,    1.0324541933967568e-51,    2.6447080683631050e-53,
    6.6225641204768279e-55,    1.6219395629350345e-56,    3.8869802486627745e-58,    9.1192560033452930e-60,
    2.0954041883209190e-61,    4.7176057161863467e-63,    1.0411125251858885e-64,    2.2530236403945442e-66,
    4.7828769423795993e-68,    9.9638052971050815e-70,    2.0376294188646144e-71,    4.0919941345474358e-73,
    8.0722633203475795e-75,    1.5647390349815640e-76,    2.9813013843601833e-78,    5.5848803025715123e-80
},
// x = 2.2400
{
    2.2068450599226172e-01,    1.2722419668976429e-01,    5.0294956854184628e-02,    1.4958667997387915e-02,
    3.5491193623473508e-03,    6.9881341652803478e-04,    1.1744633475432155e-04,    1.7205223757418846e-05,
    2.2327828073203885e-06,    2.5999709471946808e-07,    2.7450307467757497e-08,    2.6504622092413932e-09,
    2.3574014072533804e-10,    1.9433852931819356e-11,    1.4928062793009670e-12,    1.0734306586934048e-13,
    7.2549212877735077e-15,    4.6252904053428447e-16,    2.7905029425315257e-17,    1.5977509197023303e-18,
    8.7044662640759069e-20,    4.5227195812945262e-21,    2.2459924983729102e-22,    1.0681080098950226e-23,
    4.8730133253182655e-25,    2.1363449816518666e-26,    9.0135903073201698e-28,    3.6651321520057499e-29,
    1.4381898377482718e-30,    5.4526541641633000e-32,    1.9996784374032409e-33,    7.1012765919725859e-35,
    2.4443959109506012e-36,    8.1634575177205479e-38,    2.6474569895442655e-39,    8.3444695708025343e-41,
    2.5581523702499073e-42,    7.6337213273828464e-44,    2.2188900994029945e-45,    6.2866106850539015e-47,
    1.7372238705898040e-48,    4.6850815818485084e-50,    1.2338201727232026e-51,    3.1746764256404090e-53,
    7.9852526465135754e-55,    1.9644366238537117e-56,    4.7288621078727292e-58,    1.1144094244091889e-59,
    2.5721382801874716e-61,    5.8168700657892022e-63,    1.2894564681359568e-64,    2.8029549976099674e-66,
    5.9769691351726276e-68,    1.2507154977964254e-69,    2.5692126629164722e-71,    5.1826450225016408e-73,
    1.0269598018198987e-74,    1.9995936780429498e-76,    3.8269032463441685e-78,    7.2010740782685160e-80
},
// x = 2.2500
{
    2.1975355632483506e-01,    1.2702264086412715e-01,    5.0390035172665507e-02,    1.5044784924870496e-02,
    3.5840376286239341e-03,    7.0863441037476345e-04,    1.1960273345842556e-04,    1.7596394837193016e-05,
    2.2934345438054622e-06,    2.6822272844063803e-07,    2.8442614751185192e-08,    2.7583240962428846e-09,
    2.4641287848015379e-10,    2.0403224241176601e-11,    1.5741875860345544e-12,    1.1369535450900108e-13,
    7.7182572438733644e-15,    4.9424826552502915e-16,    2.9950891261802670e-17,    1.7224981087186998e-18,
    9.4257377345214388e-20,    4.9192326503478383e-21,    2.4537558301121354e-22,    1.1720990123571469e-23,
    5.3712265216510697e-25,    2.3652365309140866e-26,    1.0023718245805381e-27,    4.0940107910399916e-29,
    1.6136312152069230e-30,    6.1450458491228345e-32,    2.2636369924897726e-33,    8.0744472616721302e-35,
    2.7917592215754465e-36,    9.3650660097298842e-38,    3.0506764558805935e-39,    9.6582116960677090e-41,
    2.9740984547850361e-42,    8.9144872095817893e-44,    2.6027182577745690e-45,    7.4069496421528059e-47,
    2.0559389742494438e-48,    5.5693348548139140e-50,    1.4732278069780035e-51,    3.8075840081231730e-53,
    9.6199050370932049e-55,    2.3771267840842017e-56,    5.7478214637675197e-58,    1.3605790602935194e-59,
    3.1543203059936897e-61,    7.1652837624033181e-63,    1.5954505362288060e-64,    3.4835775540116978e-66,
    7.4614483678412334e-68,    1.5683156857887114e-69,    3.2359954238000701e-71,    6.5568048112318335e-73,
    1.3050502590327817e-74,    2.5523992005264420e-76,    4.9066765260061065e-78,    9.2740700326744525e-80
},
// x = 2.2600
{
    2.1882986202023127e-01,    1.2682064151063974e-01,    5.0483877714072367e-02,    1.5130646568001910e-02,
    3.6190432114115165e-03,    7.1852758450472983e-04,    1.2178505674247777e-04,    1.7994072269238429e-05,
    2.3553735395678456e-06,    2.7666068841836064e-07,    2.9465097112601235e-08,    2.8699630357826285e-09,
    2.5750869534440399e-10,    2.1415520910905389e-11,    1.6595517185429735e-12,    1.2038823933626477e-13,
    8.2086126030589212e-15,    5.2796681372296106e-16,    3.2135399384746998e-17,    1.8562928222362145e-18,
    1.0202767359108114e-19,    5.3483013360712632e-21,    2.6795790476942880e-22,    1.2856329600339298e-23,
    5.9175821369996974e-25,    2.6173639587743762e-26,    1.1141345075220552e-27,    4.5706446739801087e-29,
    1.8094762437107370e-30,    6.9214044441819549e-32,    2.5609242119074997e-33,    9.1753766440093450e-35,
    3.1864748606472234e-36,    1.0736558077069198e-37,    3.5129528435223552e-39,    1.1171076934577811e-40,
    3.4552224824261332e-42,    1.0402544440288825e-43,    3.0506548966590190e-45,    8.7202348576514129e-47,
    2.4312074799855492e-48,    6.6151284996652233e-50,    1.7576327633665857e-51,    4.5627967378003581e-53,
    1.1579129744115802e-54,    2.8739624627218229e-56,    6.9800048376708886e-58,    1.6595870430940696e-59,
    3.8646122754645178e-61,    8.8177442124587457e-63,    1.9721115998472419e-64,    4.3251157962888038e-66,
    9.3050684836304176e-68,    1.9645096463538773e-69,    4.0714854063052651e-71,    8.2863180841898631e-73,
    1.6566127429884434e-74,    3.2543692476446762e-76,    6.2839131084239686e-78,    1.1929923986415272e-79
},
// x = 2.2700
{
    2.1791334657411390e-01,    1.2661821881173907e-01,    5.0576493518952097e-02,    1.5216250708320077e-02,
    3.6541345065114091e-03,    7.2849275298851027e-04,    1.2399341273448516e-04,    1.8398318826260761e-05,
    2.4186187283568039e-06,    2.8531513372082159e-07,    3.0518490164905423e-08,    2.9854890234584458e-09,
    2.6904190959959021e-10,    2.2472397912299593e-11,    1.7490709951456380e-12,    1.2743806242139659e-13,
    8.7274114173152319e-15,    5.6379952650549282e-16,    3.4467176041113828e-17,    1.9997408133292011e-18,
    1.1039554779276062e-19,    5.8124170722917377e-21,    2.9249311939282024e-22,    1.4095322213804531e-23,
    6.5164624356359142e-25,    2.8949555383388642e-26,    1.2377305447277906e-27,    5.1000814854327936e-29,
    2.0279821779946206e-30,    7.7914349617208965e-32,    2.8955579878680553e-33,    1.0420104454314080e-34,
    3.6347252170084957e-36,    1.2300969487996691e-37,    4.0425927980178813e-39,    1.2912084330007841e-40,
    4.0113555925228142e-42,    1.2130221894818332e-43,    3.5730281555261775e-45,    1.0258547879313064e-46,
    2.8727261617190204e-48,    7.8510026945343951e-50,    2.0952205456526081e-51,    5.4632064078839318e-53,
    1.3925391067843468e-54,    3.4715839276815827e-56,    8.4687145824999544e-58,    2.0224467454780024e-59,
    4.7304053893704026e-61,    1.0840901082719855e-62,    2.4353136467631515e-64,    5.3645927163999226e-66,
    1.1592424054788851e-67,    2.4582417178553052e-69,    5.1172789951184759e-71,    1.0460770274913139e-72,
    2.1005787512597592e-74,    4.1447730473703284e-76,    8.0385942016548263e-78,    1.5328641385476455e-79
},
// x = 2.2800
{
    2.1700393444223096e-01,    1.2641539255679052e-01,    5.0667891604348675e-02,    1.5301595178832920e-02,
    3.6893099149493476e-03,    7.3852972508550621e-04,    1.2622790795787154e-04,    1.8809197255536753e-05,
    2.4831891714455724e-06,    2.9419027546012493e-07,    3.1603542611197212e-08,    3.1050145675189550e-09,
    2.8102723710250922e-10,    2.3575564201969012e-11,    1.8429241844551146e-12,    1.3486185582939012e-13,
    9.2761446695482754e-15,    6.0186719119132854e-16,    3.6955331084899286e-17,    2.1534849890158501e-18,
    1.1940364120713810e-19,    6.3142480102960933e-21,    3.1913925857136878e-22,    1.5446854282241815e-23,
    7.1726240234877667e-25,    3.2004407202318441e-26,    1.3743465074430891e-27,    5.6878739825586156e-29,
    2.2716432293877842e-30,    8.7659090891566906e-32,    3.2720177200453357e-33,    1.1826592544165050e-34,
    4.1434644207817294e-36,    1.4084327024162535e-37,    4.6490233304573660e-39,    1.4914313567877936e-40,
    4.6537544953811940e-42,    1.4134701087252467e-43,    4.1817693113082052e-45,    1.2059114851948535e-46,
    3.3917968172639812e-48,    9.3103695633462434e-50,    2.4956165721489617e-51,    6.5358513873897268e-53,
    1.6732795923555107e-54,    4.1898207388294830e-56,    1.0265781735694419e-57,    2.4623992699162352e-59,
    5.7847777101071193e-61,    1.3315599145230452e-62,    3.0043970465292794e-64,    6.6473163068881011e-66,
    1.4427499964979746e-67,    3.0729019632173287e-69,    6.4249621615711684e-71,    1.3191754486579163e-72,
    2.6606352102627710e-74,    5.2729620312158782e-76,    1.0271682021321263e-77,    1.9673097483891679e-79
},
// x = 2.2900
{
    2.1610155102698603e-01,    1.2621218214900259e-01,    5.0758080962790456e-02,    1.5386677863433930e-02,
    3.7245678431233969e-03,    7.4863830530705030e-04,    1.2848864732533885e-04,    1.9226770272375667e-05,
    2.5491040565025724e-06,    3.0329037694173081e-07,    3.2721016374675307e-08,    3.2286547285511160e-09,
    2.9347979970775129e-10,    2.4726784143351275e-11,    1.9412967075398138e-12,    1.4267736664621636e-13,
    9.8563730277135339e-15,    6.4229681453659568e-16,    3.9609486761201377e-17,    2.3182074778834382e-18,
    1.2909739986773600e-19,    6.8566505571604803e-21,    3.4806626166577372e-22,    1.6920524426940414e-23,
    7.8912277223696367e-25,    3.5364671651656633e-26,    1.5252813048728456e-27,    6.3401276870248157e-29,
    2.5432141026935084e-30,    9.8567763903213554e-32,    3.6952947067844579e-33,    1.3414944301610131e-34,
    4.7205102716275996e-36,    1.6116019697735387e-37,    5.3429364823093324e-39,    1.7215449729113416e-40,
    5.3952999204582293e-42,    1.6458716298961756e-43,    4.8906522393131876e-45,    1.4165101271138029e-46,
    4.0015824577047449e-48,    1.1032315937844622e-49,    2.9701308877016551e-51,    7.8126428821374439e-53,
    2.0089193786965817e-54,    5.0522837096793600e-56,    1.2433193916838204e-57,    2.9953504393917804e-59,
    7.0676398854031697e-61,    1.6339813825766228e-62,    3.7029039583822112e-64,    8.2286818097479144e-66,
    1.7937994126261909e-67,    3.8373413745500679e-69,    8.0584460497922290e-71,    1.6618137319358318e-72,
    3.3663884438755061e-74,    6.7008932477312317e-76,    1.3110487266349502e-77,    2.5220243124911975e-79
},
// x = 2.3000
{
    2.1520612266421008e-01,    1.2600860661222030e-01,    5.0847070561314045e-02,    1.5471496696320208e-02,
    3.7599067029481770e-03,    7.5881829347952743e-04,    1.3077573413304648e-04,    1.9651100553612354e-05,
    2.6163826964441569e-06,    3.1261975380772761e-07,    3.3871686728146407e-08,    3.3565271594342386e-09,
    3.0641513380402014e-10,    2.5927878955759846e-11,    2.0443808451003175e-12,    1.5090308275584041e-13,
    1.0469729695512517e-14,    6.8522190718250693e-16,    4.2439803604808293e-17,    2.4946318008085169e-18,
    1.3952524327257830e-19,    7.4426816016870161e-21,    3.7945680625572263e-22,    1.8526696683753564e-23,
    8.6778706597580203e-25,    3.9059191225605095e-26,    1.6919561906458255e-27,    7.0635528114321353e-29,
    2.8457357381422094e-30,    1.1077286471008808e-31,    4.1709477529947681e-33,    1.5207647848765188e-34,
    5.3746465938681587e-36,    1.8429213920392431e-37,    6.1364518408001597e-39,    1.9858397991955940e-40,
    6.2507215442019887e-42,    1.9151351662655993e-43,    5.7155672489495741e-45,    1.6626524868288861e-46,
    4.7174028851418390e-48,    1.3062533658907134e-49,    3.5320430144839238e-51,    9.3312140752411396e-53,
    2.4098642840146963e-54,    6.0870631842723822e-56,    1.5045024112753702e-57,    3.6403908545799012e-59,
    8.6271047062854473e-61,    2.0032175812569214e-62,    4.5594652230483247e-64,    1.0176354835262498e-65,
    2.2280576916437324e-67,    4.7871125411048233e-69,    1.0096833906187642e-70,    2.0912551289456609e-72,
    4.2548056238011747e-74,    8.5062681725752529e-76,    1.6715375135486254e-77,    3.2295166974690620e-79
},
// x = 2.3100
{
    2.1431757661013262e-01,    1.2580468459760216e-01,    5.0934869340519467e-02,    1.5556049661412923e-02,
    3.7953249119954029e-03,    7.6906948480746343e-04,    1.3308927005510190e-04,    2.0082250731132259e-05,
    2.6850445282690472e-06,    3.2218277417389994e-07,    3.5056342423117795e-08,    3.4887521455562632e-09,
    3.1984919896452584e-10,    2.7180728191264492e-11,    2.1523759497462319e-12,    1.5955825938537475e-13,
    1.1117923362416498e-14,    7.3078277942470968e-16,    4.5457007496654009e-17,    2.6835251492991025e-18,
    1.5073874225358329e-19,    8.0756114649401500e-21,    4.1350719192676960e-22,    2.0276557275812152e-23,
    9.5386207258284849e-25,    4.3119372539609118e-26,    1.8759256044652457e-27,    7.8655207722533624e-29,
    3.1825634525403891e-30,    1.2442123127730819e-31,    4.7051645052446488e-33,    1.7229845483043472e-34,
    6.1157371418809238e-36,    2.1061317577675009e-37,    7.0432990054892244e-39,    2.2891976863044516e-40,
    7.2368527785743718e-42,    2.2268948069197579e-43,    6.6748340297095542e-45,    1.9501303499062599e-46,
    5.5570754846816372e-48,    1.5454396932189268e-49,    4.1969332889588912e-51,    1.1135912210655984e-52,
    2.8884304001450916e-54,    7.3275520883128150e-56,    1.8189714158015839e-57,    4.4204143064402392e-59,
    1.0521122871989025e-60,    2.4536199340274028e-62,    5.6088690144413636e-64,    1.2572913495761462e-65,
    2.7647284524884558e-67,    5.9659844503012888e-69,    1.2637936908881312e-70,    2.6289391841362497e-72,
    5.3719980967369930e-74,    1.0786433081953948e-75,    2.1288135131687782e-77,    4.1308723978011442e-79
},
// x = 2.3200
{
    2.1343584102854790e-01,    1.2560043439018145e-01,    5.1021486213658145e-02,    1.5640334791780446e-02,
    3.8308208936308594e-03,    7.7939166993659908e-04,    1.3542935513836252e-04,    2.0520283385430068e-05,
    2.7551091118748967e-06,    3.3198385876057656e-07,    3.6275785818450860e-08,    3.6254526452884691e-09,
    3.3379838671172560e-10,    2.8487271239702703e-11,    2.2654886634613332e-12,    1.6866294643602422e-13,
    1.1802741255837206e-14,    7.7912684868460518e-16,    4.8672417922908128e-17,    2.8857007761571265e-18,
    1.6279280647365151e-19,    8.7589376141473102e-21,    4.5042828040387924e-22,    2.2182175278964137e-23,
    1.0480053558988148e-24,    4.7579400066762443e-26,    2.0788889139505196e-27,    8.7541256685909776e-29,
    3.5573976897276643e-30,    1.3967551587664352e-31,    5.3048290716450580e-33,    1.9509632045864828e-34,
    6.9548522938314205e-36,    2.4054498492321816e-37,    8.0790223419268591e-39,    2.6371699522105221e-40,
    8.3729192136274620e-42,    2.5876134398104243e-43,    7.7895590679029962e-45,    2.2856457219295141e-46,
    6.5413078843943101e-48,    1.8270209125346115e-49,    4.9830679989330829e-51,    1.3278956841275266e-52,
    3.4591834548586087e-54,    8.8134153090496881e-56,    2.1972776020501402e-57,    5.3628525555925857e-59,
    1.2819435100709939e-60,    3.0025350371406600e-62,    6.8933473941892658e-64,    1.5519043255050672e-65,
    3.4273283693550653e-67,    7.4277902971418290e-69,    1.5802581732715815e-70,    3.3014658227645217e-72,
    6.7754255234982509e-74,    1.3663220951132464e-75,    2.7082416868996954e-77,    5.2779611641704832e-79
},
// x = 2.3300
{
    2.1256084497817374e-01,    1.2539587391531390e-01,    5.1106930065752419e-02,    1.5724350169064105e-02,
    3.8663930771477770e-03,    7.8978463501691305e-04,    1.3779608779754344e-04,    2.0965261039202610e-05,
    2.8265961288571729e-06,    3.4202748101722571e-07,    3.7530833008551006e-08,    3.7667543307143363e-09,
    3.4827952939662674e-10,    2.9849508862116165e-11,    2.3839331403452199e-12,    1.7823801661771849e-13,
    1.2526052298322537e-14,    8.3040895907303875e-16,    5.2097977482902961e-17,    3.1020205033350067e-18,
    1.7574588201658141e-19,    9.4963991805715900e-21,    4.9044649530293401e-22,    2.4256567424807817e-23,
    1.1509292231194567e-24,    5.2476466501645154e-26,    2.3027031263656816e-27,    9.7382511352839461e-29,
    3.9743176076682731e-30,    1.5671580043966019e-31,    5.9775965352480846e-33,    2.2078385162010802e-34,
    7.9044098975221032e-36,    2.7456263344007977e-37,    9.2612106185167109e-39,    3.0360653979956418e-40,
    9.6808649677652197e-42,    3.0046999404411032e-43,    9.0840435985826738e-45,    2.6769486348122366e-46,
    7.6941500776590888e-48,    2.1582644672331451e-49,    5.9118467427376829e-51,    1.5821791099560498e-52,
    4.1393364640206432e-54,    1.0591730554611685e-55,    2.6519984856074057e-57,    6.5005476808378395e-59,
    1.5605899868267600e-60,    3.6709116448911399e-62,    8.4641238803495886e-64,    1.9137396752232233e-65,
    4.2446293628150942e-67,    9.2386794603394439e-69,    1.9739883244487638e-70,    4.1418051928726281e-72,
    8.5366174978673124e-74,    1.7288956236714761e-75,    3.4416728793433908e-77,    6.7361988101814288e-79
},
// x = 2.3400
{
    2.1169251840019984e-01,    1.2519102074501387e-01,    5.1191209752746193e-02,    1.5808093922906791e-02,
    3.9020398978967148e-03,    8.0024816176558545e-04,    1.4018956481062956e-04,    2.1417246150976838e-05,
    2.8995253812907301e-06,    3.5231816724077916e-07,    3.8822313951070430e-08,    3.9127856286084132e-09,
    3.6330990919287554e-10,    3.1269504752906757e-11,    2.5079312747204377e-12,    1.8830519440560221e-13,
    1.3289810372716437e-14,    8.8479171344724220e-16,    5.5746282693588325e-17,    3.3333973520420221e-18,
    1.8966015955464478e-19,    1.0291992323886118e-20,    5.3380488494246886e-22,    2.6513767300184500e-23,
    1.2634049814980376e-24,    5.7851020952083629e-26,    2.5493966449327708e-27,    1.0827643010204167e-28,
    4.4378177480312827e-30,    1.7574136794634176e-31,    6.7319750252289850e-33,    2.4971130575705163e-34,
    8.9783317699018432e-36,    3.1320103755596767e-37,    1.0609754410458543e-38,    3.4930493988260583e-40,
    1.1185721712813030e-41,    3.4866422647254298e-43,    1.0586248949471563e-44,    3.1329950275233249e-46,
    9.0435146705925959e-48,    2.5476415489642938e-49,    7.0083217021536585e-51,    1.8836657033486685e-52,
    4.9492153445073349e-54,    1.2718330018616425e-55,    3.1981150453904059e-57,    7.8727869366098387e-59,
    1.8981266812828817e-60,    4.4840270184076666e-62,    1.0383273411759003e-63,    2.3577252980085086e-65,
    5.2518008668567152e-67,    1.1479859574176280e-68,    2.4633692632864513e-70,    5.1907836539235776e-72,
    1.0744530919339255e-73,    2.1853894563288425e-75,    4.3690613128958218e-77,    8.5879988093122040e-79
},
// x = 2.3500
{
    2.1083079210602218e-01,    1.2498589210418115e-01,    5.1274334100684514e-02,    1.5891564230384337e-02,
    3.9377597974119948e-03,    8.1078202752990036e-04,    1.4260988131458815e-04,    2.1876301108774679e-05,
    2.9739167904943841e-06,    3.6286049668766641e-07,    4.0151072594100607e-08,    4.0636777616609883e-09,
    3.7890726720581871e-10,    3.2749387131005232e-11,    2.6377129346946813e-12,    1.9888708583682374e-13,
    1.4096057698283640e-14,    9.4244581837252071e-16,    5.9630616139719397e-17,    3.5807983003431500e-18,
    2.0460179359903887e-19,    1.1149986487579649e-20,    5.8076425183673149e-22,    2.8968899216709932e-23,
    1.3862675025327795e-24,    6.3747036239217463e-26,    2.8211841497616382e-27,    1.2032988289114260e-28,
    4.9528480540465957e-30,    1.9697264405498347e-31,    7.5774160704170097e-33,    2.8226946118013741e-34,
    1.0192217502687527e-35,    3.5706216963141633e-37,    1.2147134472673712e-38,    4.0162554014586336e-40,
    1.2916025713587406e-41,    4.0431585144561941e-43,    1.2328327025092280e-44,    3.6641275110648209e-46,
    1.0621775129682595e-47,    3.0050195572065564e-49,    8.3017999743324199e-51,    2.2408430863183565e-52,
    5.9128037070207969e-54,    1.5259377019686535e-55,    3.8534568443112870e-57,    9.5265294333546121e-59,
    2.3066478487830633e-60,    5.4723553989944472e-62,    1.2725955896582804e-63,    2.9020136545989805e-65,
    6.4917934468572631e-67,    1.4250932372240267e-68,    3.0710475819028642e-70,    6.4989072440207592e-72,
    1.3509687696547272e-73,    2.7595431278139509e-75,    5.5404753300683900e-77,    1.0937081628789594e-78
},
// x = 2.3600
{
    2.0997559776516111e-01,    1.2478050487671979e-01,    5.1356311904924082e-02,    1.5974759315439860e-02,
    3.9735512235346644e-03,    8.2138600535005039e-04,    1.4505713080137822e-04,    2.2342488223814323e-05,
    3.0497903957787117e-06,    3.7365910167953059e-07,    4.1517967002829706e-08,    4.2195647899440812e-09,
    3.9508981269670522e-10,    3.4291350360341907e-11,    2.7735162012680831e-12,    2.1000720916630475e-13,
    1.4946928320860278e-14,    1.0035504424109086e-15,    6.3764980020536382e-17,    3.8452471736855743e-18,
    2.2064113335953239e-19,    1.2074941592004081e-20,    6.3160435267814907e-22,    3.1638257039384322e-23,
    1.5204201141388453e-24,    7.0212296671007334e-26,    3.1204826891107864e-27,    1.3366000877354816e-28,
    5.5248575238597417e-30,    2.2065332439338286e-31,    8.5244140251708103e-33,    3.1889408176442087e-34,
    1.1563537391775225e-35,    4.0682309264553896e-37,    1.3898744634924492e-38,    4.6149103122130623e-40,
    1.4904288859769929e-41,    4.6853683011744899e-43,    1.4347224681517207e-44,    4.2822822048911259e-46,
    1.2466453279953361e-47,    3.5418842291698461e-49,    9.8265417712803568e-51,    2.6636759544610187e-52,
    7.0583798350801269e-54,    1.8293217401225179e-55,    4.6392269805897656e-57,    1.1517859070594163e-58,
    2.8006597997372513e-60,    6.6726055724116112e-62,    1.5583096154320645e-63,    3.5686589099790837e-65,
    8.0170133102433016e-67,    1.7673948284606674e-68,    3.8248930699328563e-70,    8.1285972049325913e-72,
    1.6969269348985649e-73,    3.4809488504763472e-75,    7.0185945458297696e-77,    1.3913849590468333e-78
},
// x = 2.3700
{
    2.0912686789336007e-01,    1.2457487561155131e-01,    5.1437151929371087e-02,    1.6057677448321175e-02,
    4.0094126305321555e-03,    8.3205986402185678e-04,    1.4753140511425590e-04,    2.2815869724250373e-05,
    3.1271663531775412e-06,    3.8471866770262103e-07,    4.2923869485643570e-08,    4.3805836526145509e-09,
    4.1187623242218810e-10,    3.5897656599910889e-11,    2.9155876130765400e-12,    2.2169002640047204e-13,
    1.5844651720156220e-14,    1.0682935881701683e-15,    6.8164131145295962e-17,    4.1278276739870915e-18,
    2.3785296576152689e-19,    1.3071726213843064e-20,    6.8662517281301053e-22,    3.4539388279574002e-23,
    1.6668398425558484e-24,    7.7298707744214608e-26,    3.4499290727496083e-27,    1.4839514686050347e-28,
    6.1598418096550538e-30,    2.4705270424265714e-31,    9.5846154285716690e-33,    3.6007084904025453e-34,
    1.3111846488948781e-35,    4.6324491296744539e-37,    1.5893253160235912e-38,    5.2994754285579786e-40,
    1.7187530378684779e-41,    5.4259870221539894e-43,    1.6685371868368629e-44,    5.0012252578517155e-46,
    1.4621008862907629e-47,    4.1715958453124042e-49,    1.1622569197527723e-50,    3.1638546379040297e-52,
    8.4192609179916618e-54,    2.1912551982997240e-55,    5.5806207226787832e-57,    1.3913704131142763e-58,
    3.3975477481357037e-60,    8.1289594481165744e-62,    1.9064596819456519e-63,    4.3844322300414051e-65,
    9.8913471246770438e-67,    2.1898330325870230e-68,    4.7591716470223572e-70,    1.0156928172474790e-71,
    2.1293383949266897e-73,    4.3864582673600944e-75,    8.8818080681300901e-77,    1.7682084097989836e-78
},
// x = 2.3800
{
    2.0828453584086176e-01,    1.2436902052852405e-01,    5.1516862905747453e-02,    1.6140316945021150e-02,
    4.0453424792145625e-03,    8.4280336815936924e-04,    1.5003279444436853e-04,    2.3296507748952579e-05,
    3.2060649341632446e-06,    3.9604393350082995e-07,    4.4369666719644406e-08,    4.5468742098496231e-09,
    4.2928570008923479e-10,    3.7570637483712829e-11,    3.0641824168621904e-12,    2.3396097572816158e-13,
    1.6791556537395678e-14,    1.1368724785575548e-15,    7.2843617431636358e-17,    4.4296865531245920e-18,
    2.5631677119134902e-19,    1.4145536803032428e-20,    7.4614827941851926e-22,    3.7691183774716340e-23,
    1.8265830270618576e-24,    8.5062629325156808e-26,    3.8123986656401259e-27,    1.6467584661620466e-28,
    6.8643950975810066e-30,    2.7646822876040379e-31,    1.0770939234862615e-32,    4.0634080804252660e-34,
    1.4859021972201709e-35,    5.2718275130347517e-37,    1.8163006935805445e-38,    6.0818047550210933e-40,
    1.9807876708914885e-41,    6.2795459841706876e-43,    1.9391464679527318e-44,    5.8368231471441150e-46,
    1.7135745729612559e-47,    4.9096845448856646e-49,    1.3736602490284320e-50,    3.7550840698344092e-52,
    1.0034671982067822e-53,    2.6226983888780827e-55,    6.7075540045690039e-57,    1.6793870927149864e-58,
    4.1181302864951545e-60,    9.8945494175150064e-62,    2.3303187052968444e-63,    5.3818025998266221e-65,
    1.2192608340627213e-66,    2.7106848440285918e-68,    5.9159746163139274e-70,    1.2678979016245201e-71,
    2.6692768046133435e-73,    5.5219187112563176e-75,    1.1228054768871842e-76,    2.2447282394902489e-78
},
// x = 2.3900
{
    2.0744853578085953e-01,    1.2416295552422021e-01,    5.1595453532884361e-02,    1.6222676166721392e-02,
    4.0813392370476764e-03,    8.5361627825735354e-04,    1.5256138732763918e-04,    2.3784464341324649e-05,
    3.2865065243463435e-06,    4.0763969116237675e-07,    4.5856259875566890e-08,    4.7185792850105566e-09,
    4.4733788592557852e-10,    3.9312695830864399e-11,    3.2195648237631344e-12,    2.4684650486820782e-13,
    1.7790074426549295e-14,    1.2094939576943603e-15,    7.7819815962443187e-17,    4.7520369368720781e-18,
    2.7611699256407187e-19,    1.5301917990511199e-20,    8.1051825780325922e-22,    4.1113973295029995e-23,
    2.0007913319582848e-24,    9.3565233960555202e-26,    4.2110256870228821e-27,    1.8265596381343120e-28,
    7.6457666301181235e-30,    3.0922828341735829e-31,    1.2097708936474755e-32,    4.5830637762608940e-34,
    1.6829526247290899e-35,    5.9959684202758011e-37,    2.0744483337800058e-38,    6.9753227519397826e-40,
    2.2813237895264992e-41,    7.2626416705526784e-43,    2.2521355873763569e-44,    6.8073513903571799e-46,
    2.0068851244019049e-47,    5.7741905013521600e-49,    1.6223143088967062e-50,    4.4534195019475035e-52,
    1.1950759702099200e-53,    3.1366003289717900e-55,    8.0555206584954163e-57,    2.0253447037008138e-58,
    4.9873174252298630e-60,    1.2033219129628436e-61,    2.8459029151555171e-63,    6.6001157913093185e-65,
    1.5015490263976785e-66,    3.3522862149129798e-68,    7.3469587178346372e-70,    1.5811929716610278e-71,
    3.3428243461848949e-73,    6.9443141461303232e-75,    1.4179578795078784e-76,    2.8467025143543954e-78
},
// x = 2.4000
{
    2.0661880269812083e-01,    1.2395669617766215e-01,    5.1672932476043175e-02,    1.6304753519238976e-02,
    4.1174013782627585e-03,    8.6449835075363480e-04,    1.5511727064193313e-04,    2.4279801443163769e-05,
    3.3685116221595760e-06,    4.1951078620010668e-07,    4.7384564742065915e-08,    4.8958447070295808e-09,
    4.6605296636576880e-10,    4.1126307386155923e-11,    3.3820082715146857e-12,    2.6037410535346170e-13,
    1.8842744032472932e-14,    1.2863749069586994e-15,    8.3109972658582367e-17,    5.0961618055538472e-18,
    2.9734331833236476e-19,    1.6546784042617019e-20,    8.8010423547630383e-22,    4.4829627436317225e-23,
    2.1906981817587149e-24,    1.0287289207677257e-25,    4.6492251272977590e-27,    2.0250384894705436e-28,
    8.5119222610983268e-30,    3.4569524596909737e-31,    1.3580797691347469e-32,    5.1663798068258757e-34,
    1.9050698429544401e-35,    6.8156488242670965e-37,    2.3678795132074490e-38,    7.9952237956832356e-40,
    2.6258069844934676e-41,    8.3942178489329276e-43,    2.6139067019273828e-44,    7.9338469159106786e-46,
    2.3487587734489122e-47,    6.7860555206037165e-49,    1.9145725734603784e-50,    5.2776562648763698e-52,
    1.4221774426943941e-53,    3.7482482179902422e-55,    9.6666003976071478e-57,    2.4405639174734825e-58,
    6.0348909412766249e-60,    1.4621620408458078e-61,    3.4725227876694250e-63,    8.0870103492423106e-65,
    1.8475127862671900e-66,    4.1419093233546367e-68,    9.1154627158095529e-70,    1.9700065911318292e-71,
    4.1822318248467527e-73,    8.7244026648182195e-75,    1.7888812880219502e-76,    3.6063857112160031e-78
},
// x = 2.4100
{
    2.0579527237778017e-01,    1.2375025775592101e-01,    5.1749308366260183e-02,    1.6386547452476551e-02,
    4.1535273839631631e-03,    8.7544933809130543e-04,    1.5770052960450864e-04,    2.4782580888561733e-05,
    3.4521008375268574e-06,    4.3166211762540942e-07,    4.8955511849355629e-08,    5.0788193530157217e-09,
    4.8545163385291045e-10,    4.3014022591341304e-11,    3.5517956926553000e-12,    2.7457234777128629e-13,
    1.9952215099336573e-14,    1.3677426766360272e-15,    8.8732243626634657e-17,    5.4634176379024809e-18,
    3.2009098008001520e-19,    1.7886441520478491e-20,    9.5530149886366499e-22,    4.8861666177702274e-23,
    2.3976356470642711e-24,    1.1305758593909775e-25,    5.1307164028660264e-27,    2.2440363540505936e-28,
    9.4716114642564315e-30,    3.8626882306080466e-31,    1.5237787664121044e-32,    5.8208135483633046e-34,
    2.1553077113229722e-35,    7.7429576581952579e-37,    2.7012254346624506e-38,    9.1586958836816989e-40,
    3.0204232873463202e-41,    9.6958846658639248e-43,    3.0317938202043237e-44,    9.2405100243813291e-46,
    2.7469657390749746e-47,    7.9695735484313079e-49,    2.2578365031966813e-50,    6.2497819699834095e-52,
    1.6911447379404446e-53,    4.4756763444234614e-55,    1.1590643199682705e-56,    2.9385121807564887e-58,
    7.2964290658401388e-60,    1.7751708540265103e-61,    4.2334414158814806e-63,    9.9001168613900007e-65,
    2.2711390218542137e-66,    5.1128242658360779e-68,    1.1299079767840136e-69,    2.4520886801677125e-71,
    5.2273405116707450e-73,    1.0949962723002413e-74,    2.2545650098455521e-76,    4.5641274866609734e-78
},
// x = 2.4200
{
    2.0497788139429890e-01,    1.2354365521962724e-01,    5.1824589799719746e-02,    1.6468056459875775e-02,
    4.1897157422277707e-03,    8.8646898878076599e-04,    1.6031124776974307e-04,    2.5292864397848419e-05,
    3.5372948905173604e-06,    4.4409863801573395e-07,    5.0570046592175011e-08,    5.2676551910745133e-09,
    5.0555510675610886e-10,    4.4978468387437541e-11,    3.7292197888305261e-12,    2.8947091798080397e-13,
    2.1121252712789910e-14,    1.4538355336687037e-15,    9.4705738242544631e-17,    5.8552382248400877e-18,
    3.4446106536965783e-19,    1.9327613205388592e-20,    1.0365332077942467e-21,    5.3235374503857294e-23,
    2.6230418102403451e-24,    1.2419735436271385e-25,    5.6595488773679822e-27,    2.4855663523323056e-28,
    1.0534440249104247e-29,    4.3138969647766450e-31,    1.7088144896725588e-32,    6.5546560978794820e-34,
    2.4370757617833649e-35,    8.7914484613556842e-37,    3.0797001683614894e-38,    1.0485171399374051e-39,
    3.4721958178626363e-41,    1.1192279375178206e-42,    3.5141933239385001e-44,    1.0755162646615544e-45,
    3.2104764896227512e-47,    9.3529086220406540e-49,    2.6607225592219997e-50,    7.3955008063569069e-52,
    2.0094594160043007e-53,    5.3401441923291548e-55,    1.3886659962298168e-56,    3.5351985534120502e-58,
    8.8144013830941763e-60,    2.1533709321534590e-61,    5.1566606482080118e-63,    1.2109095542473057e-64,
    2.7894049240244732e-66,    6.3055828627367758e-68,    1.3992781036663201e-69,    3.0492553140994142e-71,
    6.5273232319996748e-73,    1.3729784962593247e-74,    2.8386423375949387e-76,    5.7703552127583372e-78
},
// x = 2.4300
{
    2.0416656710058917e-01,    1.2333690322838649e-01,    5.1898785337149075e-02,    1.6549279077874068e-02,
    4.2259649482113684e-03,    8.9755704746160745e-04,    1.6294950702713625e-04,    2.5810713571578479e-05,
    3.6241146099851045e-06,    4.5682535357570010e-07,    5.2229129352059115e-08,    5.4625073233370754e-09,
    5.2638513940366796e-10,    4.7022350048314084e-11,    3.9145833112892192e-12,    3.0510065432748116e-13,
    2.2352741679381166e-14,    1.5449031260083922e-15,    1.0105056403395291e-16,    6.2731386601387336e-18,
    3.7056084654115077e-19,    2.0877463353885690e-20,    1.1242522131321255e-21,    5.7977925513014020e-23,
    2.8684686417308053e-24,    1.3637677029405022e-25,    6.2401293865443046e-27,    2.7518285090277671e-28,
    1.1710950473226041e-29,    4.8154350611648500e-31,    1.9153411138547998e-32,    7.3771210355166654e-34,
    2.7541787208493763e-35,    9.9763089648379766e-37,    3.5091708734726550e-38,    1.1996608063773481e-39,
    3.9890935223807263e-41,    1.2911473905445057e-42,    4.0707120581790144e-44,    1.2509770474524145e-45,
    3.7496405353603182e-47,    1.0968689989760258e-48,    3.1332549072264852e-50,    8.7448410256376906e-52,
    2.3858980475596424e-53,    6.3666950694526367e-55,    1.6624454189444662e-56,    4.2496389244312681e-58,
    1.0639464301843927e-59,    2.6099644359570437e-61,    6.2758590559752457e-63,    1.4798077533399337e-64,
    3.4228998430162907e-66,    7.7695697486987314e-68,    1.7312705440102736e-69,    3.7882961423642246e-71,
    8.1428144883825698e-73,    1.7198576431090733e-74,    3.5704983602079694e-76,    7.2880300894071901e-78
},
// x = 2.4400
{
    2.0336126761729967e-01,    1.2313001614610211e-01,    5.1971903503239714e-02,    1.6630213885364925e-02,
    4.2622735042419490e-03,    9.0871325496430684e-04,    1.6561538759958286e-04,    2.6336189884561881e-05,
    3.7125809321942951e-06,    4.6984732419178861e-07,    5.3933735618891991e-08,    5.6635340291934916e-09,
    5.4796403223201061e-10,    4.9148453045844448e-11,    4.1081993476663218e-12,    3.2149358587585846e-13,
    2.3649691046809729e-14,    1.6412069640873343e-15,    1.0778787342584079e-16,    6.7187195151579708e-18,
    3.9850412618472256e-19,    2.2543624349109791e-20,    1.2189429831972918e-21,    6.3118511454838865e-23,
    3.1355904196577282e-24,    1.4968745351561605e-25,    6.8772519132938533e-27,    3.0452261210194198e-28,
    1.3012706077951532e-29,    5.3726519897588514e-31,    2.1457414190378026e-32,    8.2984421643397508e-34,
    3.1108602123750854e-35,    1.1314549406125418e-36,    3.9962361029415776e-38,    1.3717803544730435e-39,
    4.5801534503440938e-41,    1.4885435094408271e-42,    4.7123352595858327e-44,    1.4541037518549603e-45,
    4.3763908751000346e-47,    1.2854695471625845e-48,    3.6870876207513967e-50,    1.0332858352533234e-51,
    2.8307491514340484e-53,    7.5848083680514713e-55,    1.9886535132136075e-56,    5.1044037537061963e-58,
    1.2831992706207461e-59,    3.1607516483126888e-61,    7.6315101846655229e-63,    1.8068587585161595e-64,
    4.1965728964927510e-66,    9.5648748444263739e-68,    2.1400753713474662e-69,    4.7020789807298279e-71,
    1.0148514252901545e-72,    2.1522980336198189e-74,    4.4866354310888410e-76,    9.1956871246493323e-78
},
// x = 2.4500
{
    2.0256192182226032e-01,    1.2292300804620482e-01,    5.2043952786091148e-02,    1.6710859503161646e-02,
    4.2986399199150304e-03,    9.1993734837173956e-04,    1.6830896804191617e-04,    2.6869354679939281e-05,
    3.8027148994308086e-06,    4.8316966348061915e-07,    5.5684856111720866e-08,    5.8708968087257755e-09,
    5.7031464205033037e-10,    5.1359644946896238e-11,    4.3103916151474476e-12,    3.3868297168154907e-13,
    2.5015238768665100e-14,    1.7430209199374332e-15,    1.1493991241605935e-16,    7.1936712051087989e-18,
    4.2841160004176944e-19,    2.4334224817961470e-20,    1.3211236448944122e-21,    6.8688483166159865e-23,
    3.4262127272645085e-24,    1.6422862086966530e-25,    7.5761295694335279e-27,    3.3683834722573681e-28,
    1.4452386814951073e-29,    5.9914377585468722e-31,    2.4026498450450697e-32,    9.3299810877157536e-34,
    3.5118510610197240e-35,    1.2825211540977938e-36,    4.5483130788168349e-38,    1.5676747575430953e-39,
    5.2556181817440569e-41,    1.7150544111912405e-42,    5.4516168729169475e-44,    1.6891082744852541e-45,
    5.1044776379651337e-47,    1.5054635661696010e-48,    4.3357607375064170e-50,    1.2200449939055270e-51,
    3.3560652704649912e-53,    9.0291606314244894e-55,    2.3770359358952986e-56,    6.1262625158032924e-58,
    1.5463889511661349e-59,    3.8246275128807761e-61,    9.2722147145400314e-63,    2.2043040295782708e-64,
    5.1406310068958353e-66,    1.1764551943415050e-67,    2.6430152819131700e-69,    5.8308932707494878e-71,
    1.2636367980974988e-72,    2.6908958508164001e-74,    5.6323546939277599e-76,    1.1591194304906685e-77
},
// x = 2.4600
{
    2.0176846934008363e-01,    1.2271589271679299e-01,    5.2114941636677618e-02,    1.6791214593464537e-02,
    4.3350627121849748e-03,    9.3122906108048594e-04,    1.7103032523971319e-04,    2.7410269163302558e-05,
    3.8945376586000311e-06,    4.9679753883079016e-07,    5.7483496898807530e-08,    6.0847604263352686e-09,
    5.9346039242089566e-10,    5.3658877342427076e-11,    4.5214947601105473e-12,    3.5670334112384605e-13,
    2.6452656517369520e-14,    1.8506317444987376e-15,    1.2253007124926229e-16,    7.6997785545492464e-18,
    4.6041123811575535e-19,    2.6257919286655634e-20,    1.4313481457581085e-21,    7.4721498397566244e-23,
    3.7422820647695670e-24,    1.8010767654641470e-25,    8.3424290512139820e-27,    3.7241649993458950e-28,
    1.6043890074786571e-29,    6.6782747002262991e-31,    2.6889777495858100e-32,    1.0484345562296162e-33,
    3.9624226563039910e-35,    1.4529600517327970e-36,    5.1737349179168888e-38,    1.7905015853774110e-39,
    6.0270902015980462e-41,    1.9748182365248560e-42,    6.3028951198311356e-44,    1.9608209679562633e-45,
    5.9497349309379015e-47,    1.7619053303686745e-48,    5.0949951411558579e-50,    1.4395295627887502e-51,
    3.9759556834480222e-53,    1.0740512973929016e-54,    2.8390955311909086e-56,    7.3469413780209437e-58,
    1.8620721990801242e-59,    4.6241702659824264e-61,    1.1256286237817363e-62,    2.6868919192000366e-64,
    6.2916167765635260e-66,    1.4457340814572550e-67,    3.2612190055474328e-69,    7.2240826753905233e-71,
    1.5719447004905424e-72,    3.3610837063186286e-74,    7.0638249606162350e-76,    1.4596396840787624e-77
},
// x = 2.4700
{
    2.0098085053192011e-01,    1.2250868366568304e-01,    5.2184878468337484e-02,    1.6871277859331932e-02,
    4.3715404054534331e-03,    9.4258812286193813e-04,    1.7377953440836456e-04,    2.7958994396861734e-05,
    3.9880704598115233e-06,    5.1073617143830150e-07,    5.9330679516897234e-08,    6.3052929545597107e-09,
    6.1742528415497482e-10,    5.6049187808957991e-11,    4.7418546643408399e-12,    3.7559053532059883e-13,
    2.7965354649115395e-14,    1.9643396036722954e-15,    1.3058293715979635e-16,    8.2389255700827010e-18,
    4.9463868480638190e-19,    2.8323919450453544e-20,    1.5502085434251422e-21,    8.1253679550164775e-23,
    4.0858961143144319e-24,    1.9744084514160644e-25,    9.1823077468506585e-27,    4.1156960189595946e-28,
    1.7802441475046008e-29,    7.4402939489764785e-31,    3.0079410689243726e-32,    1.1775519649076017e-33,
    4.4684458806923760e-35,    1.6451541990825543e-36,    5.8818588894939040e-38,    2.0438210457963719e-39,
    6.9077052209356163e-41,    2.2725392040625550e-42,    7.2825365351494053e-44,    2.2747781252913591e-45,
    6.9303854320980343e-47,    2.0606354130577989e-48,    5.9830319442995775e-50,    1.6972945741379241e-51,
    4.7069260809799555e-53,    1.2766745135672647e-54,    3.3883994163509535e-56,    8.8040133795788346e-58,
    2.2404242052481562e-59,    5.5863387257563241e-61,    1.3653637506815623e-62,    3.2723768155235041e-64,
    7.6937011912904601e-66,    1.7750945295839184e-67,    4.0204356030386442e-69,    8.9420273690713873e-71,
    1.9536679498830457e-72,    4.1942380274900921e-74,    8.8506259008838641e-76,    1.8362849661571051e-77
},
// x = 2.4800
{
    2.0019900648536584e-01,    1.2230139412537373e-01,    5.2253771656284732e-02,    1.6951048044154562e-02,
    4.4080715316548750e-03,    9.5401425992317564e-04,    1.7655666909240024e-04,    2.8515591293658348e-05,
    4.0833346549506218e-06,    5.2499083633553331e-07,    6.1227441089680812e-08,    6.5326658180746200e-09,
    6.4223390592425604e-10,    5.8533701902686125e-11,    4.9718287579150497e-12,    3.9538174964723168e-13,
    2.9556887324654394e-14,    2.0844586336853619e-15,    1.3912434925613558e-16,    8.8131004285003903e-18,
    5.3123767891175752e-19,    3.0542027136659280e-20,    1.6783374293582073e-21,    8.8323781369256203e-23,
    4.4593146989157530e-24,    2.1635385035625957e-25,    1.0102453686221843e-26,    4.5463851361526418e-28,
    1.9744714915579172e-29,    8.2853370074289964e-31,    3.3630905969865411e-32,    1.3219006776616323e-33,
    5.0364561525551586e-35,    1.8617667096759347e-36,    6.6831868983972057e-38,    2.3316452025191394e-39,
    7.9123266690280750e-41,    2.6135620390981998e-42,    8.4092120778744885e-44,    2.6373213710407256e-45,
    8.0673878657254013e-47,    2.4083987675105562e-48,    7.0210228473280115e-50,    1.9998077408751288e-51,
    5.5682724838092118e-53,    1.5164059597258454e-54,    4.0409380409887752e-56,    1.0541943550557860e-57,
    2.6935357347539651e-59,    6.7432976895296355e-61,    1.6548022401188140e-62,    3.9821148522056683e-64,
    9.4002327533152726e-66,    2.1775977778306607e-67,    4.9520183200813531e-69,    1.1058548916794236e-70,
    2.4258613549033524e-72,    5.2290335652377978e-74,    1.1078871519544985e-75,    2.3078886782700149e-77
},
// x = 2.4900
{
    1.9942287900451880e-01,    1.2209403705792343e-01,    5.2321629537141215e-02,    1.7030523931133815e-02,
    4.4446546303393156e-03,    9.6550719496762386e-04,    1.7936180116507201e-04,    2.9080120611826532e-05,
    4.1803516962375139e-06,    5.3956686241380424e-07,    6.3174834445431679e-08,    6.7670538378741544e-09,
    6.6791144498769046e-10,    6.1115635186503959e-11,    5.2117863388521881e-12,    4.1611557738210227e-13,
    3.1230957789870333e-14,    2.2113175163508405e-15,    1.4818145562157453e-16,    9.4244006888903674e-18,
    5.7036049437607154e-19,    3.2922669043405155e-20,    1.8164104939743182e-21,    9.5973369170513893e-23,
    4.8649714786740966e-24,    2.3698264238911918e-25,    1.1110128535511313e-26,    5.0199484610722496e-28,
    2.1888962861768895e-29,    9.2220228359981817e-31,    3.7583451170787087e-32,    1.4831986929859677e-33,
    5.6737251865010319e-35,    2.1057728150160461e-36,    7.5894995100628473e-38,    2.6584929504390415e-39,
    9.0577638310481480e-41,    3.0039557943997478e-42,    9.7042093621691905e-44,    3.0557105161032759e-45,
    9.3848331668274621e-47,    2.8129799063917566e-48,    8.2334788549436783e-50,    2.3545944631908535e-51,
    6.5825377708195573e-53,    1.7998382794460524e-54,    4.8155447315745487e-56,    1.2613315085698861e-57,
    3.2357631352239066e-59,    8.1333942643716883e-61,    2.0039698713079932e-62,    4.8417743272162110e-64,
    1.1475592466691116e-65,    2.6690701777883424e-67,    6.0941123396002831e-69,    1.3663825416346395e-70,
    3.0094431567108226e-72,    6.5130987430542510e-74,    1.3855042962098452e-75,    2.8978331525695145e-77
},
// x = 2.5000
{
    1.9865241060018290e-01,    1.2188662515974392e-01,    5.2388460408490177e-02,    1.7109704342763550e-02,
    4.4812882487522470e-03,    9.7706664725546837e-04,    1.8219500082818581e-04,    2.9652642948901991e-05,
    4.2791431347738987e-06,    5.5446963243948749e-07,    6.5173928233792994e-08,    7.0086352756260269e-09,
    6.9448369803358469e-10,    6.3798295290181221e-11,    5.4621088996276385e-12,    4.3783205450060498e-13,
    3.2991423820135507e-14,    2.3452600748162488e-15,    1.5778277270802217e-16,    1.0075038737521037e-17,
    6.1216840269394723e-19,    3.5476933340306658e-20,    1.9651492406723499e-21,    1.0424700820438266e-22,
    5.3054864299552433e-24,    2.5947417725991557e-25,    1.2213213852964945e-26,    5.5404357705868263e-28,
    2.4255157673928388e-29,    1.0259820931155629e-30,    4.1980276401106485e-32,    1.6633489285638392e-33,
    6.3883401297736570e-35,    2.3804948226885039e-36,    8.6140049684671194e-38,    3.0294513915794657e-39,
    1.0363016381441201e-40,    3.4506081986329935e-42,    1.1191785542222109e-43,    3.5382516285835126e-45,
    1.0910395898208404e-46,    3.2833575639900435e-48,    9.6487857614640929e-50,    2.7704050922514105e-51,
    7.7760404291822553e-53,    2.1346994625280380e-54,    5.7343855801971692e-56,    1.5080266946913987e-57,
    3.8841403698498732e-59,    9.8023118964891952e-61,    2.4248588401511167e-62,    5.8821822786806367e-64,
    1.3997413346959113e-65,    3.2686729578091980e-67,    7.4930875357390951e-69,    1.6867922269546816e-70,
    3.7300480603116499e-72,    8.1050369381467263e-74,    1.7310687641567846e-75,    3.6351218929729026e-77
},
// x = 2.5100
{
    1.9788754448021617e-01,    1.2167917086631123e-01,    5.2454272528449697e-02,    1.7188588140315743e-02,
    4.5179709419118147e-03,    9.8869233266382487e-04,    1.8505633661218383e-04,    3.0233218736179760e-05,
    4.3797306190777237e-06,    5.6970458306370665e-07,    6.7225807041696796e-08,    7.2575918781954588e-09,
    7.2197708213682574e-10,    6.6585084003970563e-11,    5.7231904606485463e-12,    4.6057270564074884e-13,
    3.4842303332525630e-14,    2.4866458904118128e-15,    1.6795824710195665e-16,    1.0767347474603539e-17,
    6.5683215791757953e-19,    3.8216608220768159e-20,    2.1253238566108925e-21,    1.1319246479600329e-22,
    5.7836791558510981e-24,    2.8398725149975589e-25,    1.3420260836139506e-26,    6.1122587609284326e-28,
    2.6865144862071695e-29,    1.1409130896217859e-30,    4.6869050244849032e-32,    1.8644581732639304e-33,
    7.1892907945584766e-35,    2.6896408085490256e-36,    9.7715048060957346e-38,    3.4502443235063039e-39,
    1.1849548368745837e-40,    3.9613317993423271e-42,    1.2899565930125710e-43,    4.0944422918100751e-45,
    1.2675848333041948e-46,    3.8298815455442205e-48,    1.1299795983374928e-49,    3.2574077479307130e-51,
    9.1794865632937661e-53,    2.5300422209704289e-54,    6.8235310902236269e-56,    1.8016178198428421e-57,
    4.6588637268102439e-59,    1.1804433464537025e-60,    2.9318025506206328e-62,    7.1403325575824124e-64,
    1.7059234055417069e-65,    3.9995861928012809e-67,    9.2052653876957680e-69,    2.0805065110893759e-70,
    4.6190637547056918e-72,    1.0076892497612268e-73,    2.1608175526602465e-75,    4.5556983593323518e-77
},
// x = 2.5200
{
    1.9712822454002193e-01,    1.2147168635679474e-01,    5.2519074115266275e-02,    1.7267174223329872e-02,
    4.5547012726833026e-03,    1.0003839637466564e-03,    1.8794587537646987e-04,    3.0821908233121098e-05,
    4.4821358936061600e-06,    5.8527720482559849e-07,    6.9331571508391882e-08,    7.5141089223327535e-09,
    7.5041864583102808e-10,    6.9479499405889013e-11,    5.9954379107881786e-12,    4.8438059126315231e-13,
    3.6787780170038346e-14,    2.6358509412212074e-15,    1.7873931974382572e-16,    1.1503786252338817e-17,
    7.0453250524865783e-19,    4.1154222499546544e-20,    2.2977562484584724e-21,    1.2286091993098714e-22,
    6.3025830789496959e-24,    3.1069339585415509e-25,    1.4740543806084159e-26,    6.7402215476583843e-28,
    2.9742809210698849e-29,    1.2681369047934249e-30,    5.2302312757569886e-32,    2.0888578839937740e-33,
    8.0865657725532413e-35,    3.0373474225014995e-36,    1.1078577807192327e-37,    3.9273086274099076e-39,
    1.3541595045374744e-40,    4.5449833132505708e-42,    1.4858994033768988e-43,    4.7351362656022989e-45,
    1.4717645571294824e-46,    4.4644748289648545e-48,    1.3224507640749063e-49,    3.8274104582258344e-51,
    1.0828677826536029e-52,    2.9964639333474265e-54,    8.1136227814353191e-56,    2.1507640082015016e-57,
    5.5838616280678483e-59,    1.4204450166559270e-60,    3.5419197767069206e-62,    8.6605853132744635e-64,
    2.0773669161695601e-65,    4.8898292346258852e-67,    1.1298998799967718e-68,    2.5638806715990128e-70,
    5.7148893530194318e-72,    1.2517156815673509e-73,    2.6947744382689132e-75,    5.7040665678427648e-77
},
// x = 2.5300
{
    1.9637439535317999e-01,    1.2126418355860656e-01,    5.2582873346927156e-02,    1.7345461529106130e-02,
    4.5914778118509589e-03,    1.0121412497944252e-03,    1.9086368230997815e-04,    3.1418771521810573e-05,
    4.5863807972672596e-06,    6.0119304214916168e-07,    7.1492338439561658e-08,    7.7783752595192837e-09,
    7.7983608029543177e-10,    7.2485138022929432e-11,    6.2792713550778907e-12,    5.0930035602871330e-13,
    3.8832210062036997e-14,    2.7932682630131967e-15,    1.9015899268449319e-16,    1.2286947073968077e-17,
    7.5546071423428671e-19,    4.4303088353159985e-20,    2.4833232517252273e-21,    1.3330719599195629e-22,
    6.8654605703067516e-24,    3.3977783186511215e-25,    1.6184117689867155e-26,    7.4295535801363371e-28,
    3.2914254788164145e-29,    1.4089062644809829e-30,    5.8337948489068972e-32,    2.3391269975015187e-33,
    9.0912582904648723e-35,    3.4282272239835484e-36,    1.2553784263377277e-37,    4.4678794261101686e-39,
    1.5466506309218810e-40,    5.2115977584925670e-42,    1.7105839379198044e-43,    5.4727300409882866e-45,
    1.7077591131839135e-46,    5.2008643915703887e-48,    1.5466843288066626e-49,    4.4941169234963303e-51,
    1.2765329798615231e-52,    3.5463619281264039e-54,    9.6406500182913943e-56,    2.5656763434015666e-57,
    6.6874640167346284e-59,    1.7079259183313967e-60,    4.2756406552500761e-62,    1.0496093185204539e-63,
    2.5276193762920007e-65,    5.9732440717276619e-67,    1.3857174980899877e-68,    3.1568269921417345e-70,
    7.0644620171738061e-72,    1.5534429692423847e-73,    3.3576115266036085e-75,    7.1352808577563082e-77
},
// x = 2.5400
{
    1.9562600216221601e-01,    1.2105667415187216e-01,    5.2645678360792196e-02,    1.7423449032202398e-02,
    4.6282991381871863e-03,    1.0239638968934752e-03,    1.9380982093197789e-04,    3.2023868501463030e-05,
    4.6924872619205052e-06,    6.1745769333368341e-07,    7.3709240920509574e-08,    8.0505833609664024e-09,
    8.1025773065629985e-10,    7.5605697026447331e-11,    6.5751244696550324e-12,    5.3537827841742420e-13,
    4.0980126765210983e-14,    2.9593086331867348e-15,    2.0225189846465117e-16,    1.3119561063862777e-17,
    8.0981913762432097e-19,    4.7677346304840658e-20,    2.6829600226873924e-21,    1.4458999738690040e-22,
    7.4758190715169536e-24,    3.7144049543161602e-25,    1.7761879779781087e-26,    8.1859451492403616e-28,
    3.6407999929918595e-29,    1.5645952368865113e-30,    6.5039703027196757e-32,    2.6181169420220694e-33,
    1.0215682743673655e-34,    3.8674210032217760e-36,    1.4217892655580232e-37,    5.0800849736049731e-39,
    1.7655130936416895e-40,    5.9725391229464300e-42,    1.9680770236503388e-43,    6.3213740851649073e-45,
    1.9803593667650290e-46,    6.0548446937537234e-48,    1.8077542392068484e-49,    5.2734208174165117e-51,
    1.5038017452882323e-52,    4.1942295955375086e-54,    1.1446854696765968e-55,    3.0583876665040970e-57,
    8.0031881876523227e-59,    2.0520200384465152e-60,    5.1573291403155139e-62,    1.2710495808965337e-63,
    3.0729658138249688e-65,    7.2906730747131563e-67,    1.6980225087956719e-68,    3.8835684640151725e-70,
    8.7251074179784796e-72,    1.9261875129343205e-73,    4.1797019372653617e-75,    8.9173868216550778e-77
},
// x = 2.5500
{
    1.9488299086950672e-01,    1.2084916957382395e-01,    5.2707497253243259e-02,    1.7501135743935192e-02,
    4.6651638385191846e-03,    1.0358516079851349e-03,    1.9678435309311457e-04,    3.2637258882982142e-05,
    4.8004773108666588e-06,    6.3407681053776686e-07,    7.5983428428393468e-08,    8.3309293627620304e-09,
    8.4171260740260596e-10,    7.8844976461973198e-11,    6.8834448640662350e-12,    5.6266232161212466e-13,
    4.3236248389424374e-14,    3.1344012783973738e-15,    2.1505437220553522e-16,    1.4004505219020450e-17,
    8.6782179698717371e-19,    5.1292012560007803e-20,    2.8976636223357110e-21,    1.5677216584824908e-22,
    8.1374282699431810e-24,    4.0589713169388438e-25,    1.9485636065493569e-26,    9.0155856793960828e-28,
    4.0255188364769854e-29,    1.7367103741645502e-30,    7.2477746844965838e-32,    2.9289790496814392e-33,
    1.1473502930674687e-34,    4.3606555878941479e-36,    1.6094131109663029e-37,    5.7730523382716000e-39,
    2.0142246240152971e-40,    6.8406695219211356e-42,    2.2629999208457809e-43,    7.2972119162298015e-45,
    2.2950526804690991e-46,    7.0445782691381321e-48,    2.1115183574729809e-49,    6.1837442281926462e-51,
    1.7703266778433760e-52,    4.9569996412545196e-54,    1.3581784152932367e-55,    3.6430678321278084e-57,
    9.5706606915114171e-59,    2.4635690822912902e-60,    6.2160190864066914e-62,    1.5379931655023214e-63,
    3.7329669267306171e-65,    8.8913684853723108e-67,    2.0789740449804231e-68,    4.7735480050139246e-70,
    1.0766780920102235e-71,    2.3862639490114257e-73,    5.1984049526724234e-75,    1.1134413146666209e-76
},
// x = 2.5600
{
    1.9414530802831914e-01,    1.2064168102311867e-01,    5.2768338079351912e-02,    1.7578520711884399e-02,
    4.7020705077930299e-03,    1.0478040829245321e-03,    1.9978733897668186e-04,    3.3259002183569861e-05,
    4.9103730573271212e-06,    6.5105609975695300e-07,    7.8316066943487169e-08,    8.6196131111592632e-09,
    8.7423039791570712e-10,    8.2206881513688371e-11,    7.2046944510249622e-12,    5.9120218567119060e-13,
    4.5605483912893499e-14,    3.3189946065475154e-15,    2.2860452650169774e-16,    1.4944809452667136e-17,
    9.2969499622178178e-19,    5.5163028802653287e-20,    3.1284968022148128e-21,    1.6992095122107025e-22,
    8.8543383904770243e-24,    4.4338046584584202e-25,    2.1368172454396887e-26,    9.9252050090627782e-28,
    4.4489817738576021e-29,    1.9269028207960307e-30,    8.0729290542505382e-32,    3.2751945878946248e-33,
    1.2879873103605272e-34,    4.9143076823492727e-36,    1.8208466206782973e-37,    6.5570250523005588e-39,
    2.2967038293151336e-40,    7.8305390191190230e-42,    2.6006010576087588e-43,    8.4186505301757915e-45,
    2.6581205623219901e-46,    8.1909384538705614e-48,    2.4647354811456849e-49,    7.2464266290245008e-51,
    2.0826814381408300e-52,    5.8544417380196329e-54,    1.6103515791634453e-55,    4.3363918399291771e-57,
    1.1436698147276738e-58,    2.9554325626016725e-60,    7.4862840905779534e-62,    1.8595424908386645e-63,
    4.5311000942700529e-65,    1.0834677982125345e-66,    2.5432814285971282e-68,    5.8625244266421948e-70,
    1.3274779827404725e-71,    2.9536434518238961e-73,    6.4596336645758702e-75,    1.3890035607056306e-76
},
// x = 2.5700
{
    1.9341290083398141e-01,    1.2043421946408032e-01,    5.2828208852564694e-02,    1.7655603019401930e-02,
    4.7390177491352728e-03,    1.0598210185391215e-03,    2.0281883710012114e-04,    3.3889157721388284e-05,
    5.0221967029132278e-06,    6.6840132079495794e-07,    8.0708339059451253e-08,    8.9168382080017377e-09,
    9.0784147811274754e-10,    8.5695424803807106e-11,    7.5393498237233462e-12,    6.2104936101447005e-13,
    4.8092939891215811e-14,    3.5135569638387786e-15,    2.4294232920911501e-16,    1.5943663941014647e-17,
    9.9567796414558178e-19,    5.9307314567622825e-20,    3.3765920024752968e-21,    1.8410829860016686e-22,
    9.6308996706879884e-24,    4.8414145475380864e-25,    2.3423331215522766e-26,    1.0922117877747218e-27,
    4.9148986880859065e-29,    2.1369814677308992e-30,    8.9879255908464554e-32,    3.6606076467198136e-33,
    1.4451593050800670e-34,    5.5354743383217183e-36,    2.0589911985292728e-37,    7.4434940213377570e-39,
    2.6173638406325819e-40,    8.9585985292487426e-42,    2.9868379335343236e-43,    9.7066661303386553e-45,
    3.0767495002609645e-46,    9.5178999441778410e-48,    2.8751992229853461e-49,    8.4861716582108832e-51,
    2.4485060034520297e-52,    6.9096229100815833e-54,    1.9080080534920104e-55,    5.1579694295609003e-57,
    1.3656573508200154e-58,    3.5428525359436704e-60,    9.0092646817306611e-62,    2.2465715281171632e-63,
    5.4955223505953257e-65,    1.3192058894998855e-66,    3.1087250510523009e-68,    7.1938924813224704e-70,
    1.6353023099112605e-71,    3.6527529188711007e-73,    8.0197656427028179e-75,    1.7312060453050999e-76
},
// x = 2.5800
{
    1.9268571711518329e-01,    1.2022679563086971e-01,    5.2887117544404609e-02,    1.7732381785124377e-02,
    4.7760041739120795e-03,    1.0719021086869043e-03,    2.0587890431674681e-04,    3.4527784610273708e-05,
    5.1359705360856802e-06,    6.8611828722853525e-07,    8.3161444092591324e-08,    9.2228120562800311e-09,
    9.4257692420346262e-10,    8.9314728727099102e-11,    7.8879026407978748e-12,    6.5225718324700714e-13,
    5.0703927364844528e-14,    3.7185774175999982e-15,    2.5810968422442694e-16,    1.7004426784573771e-17,
    1.0660235273815693e-18,    6.3742822308518194e-20,    3.6431555729324881e-21,    1.9941115271902015e-22,
    1.0471783089872235e-23,    5.2845062454746863e-25,    2.5666093003670686e-26,    1.2014271852412231e-27,
    5.4273163257017505e-29,    2.3689272373481144e-30,    1.0002100758391080e-31,    4.0894611393690121e-33,
    1.6207278536513518e-34,    6.2320507125071554e-36,    2.3270872250745802e-37,    8.4453431216522610e-39,
    2.9811722167883735e-40,    1.0243438491354755e-41,    3.4284693015274062e-43,    1.1185149586918177e-44,
    3.5591566958685227e-46,    1.1052983610025648e-47,    3.3518910928311762e-49,    9.9315600004567089e-51,
    2.8766740725184124e-52,    8.1494402187756536e-54,    2.2591116324261170e-55,    6.1308460768582014e-57,
    1.6295498620507829e-58,    4.2438823261892659e-60,    1.0833880479247154e-61,    2.7120610576268782e-63,
    6.6599778685407353e-65,    1.6049483370423680e-66,    3.7967808068822010e-68,    8.8202715802749125e-70,
    2.0128014150906383e-71,    4.5134444128087002e-73,    9.9479697823331847e-75,    2.1557905878397858e-76
},
// x = 2.5900
{
    1.9196370532540463e-01,    1.2001942003158203e-01,    5.2945072084189887e-02,    1.7808856162489464e-02,
    4.8130284017859530e-03,    1.0840470443143453e-03,    2.0896759581769152e-04,    3.5174941754503691e-05,
    5.2517169306045141e-06,    7.0421286636596667e-07,    8.5676598190085500e-08,    9.5377459058135172e-09,
    9.7846852456004743e-10,    9.3069027820784641e-11,    8.2508600190492290e-12,    6.8488088934535575e-13,
    5.3443968969682260e-14,    3.9345665656203115e-15,    2.7415051535371668e-16,    1.8130631995788781e-17,
    1.1409988148122571e-18,    6.8488595285863812e-20,    3.9294722284176026e-21,    2.1591178053866700e-22,
    1.1382002426347887e-23,    5.7659949965289901e-25,    2.8112664842925719e-26,    1.3210298941873185e-27,
    5.9906472152545038e-29,    2.6249085907678527e-30,    1.1125715049744004e-31,    4.5664361949388545e-33,
    1.8167548541380909e-34,    7.0128158278163801e-36,    2.6287519616525762e-37,    9.5770110585528939e-39,
    3.3937178021686270e-40,    1.1706056301540824e-41,    3.9331588653026902e-43,    1.2881296586883329e-44,
    4.1147326239894827e-46,    1.2827762823556593e-47,    3.9051554365988991e-49,    1.1615637807303961e-50,
    3.3774858828981947e-52,    9.6052367271471729e-54,    2.6729787654067271e-55,    7.2820868836353760e-57,
    1.9430358879587841e-58,    5.0798901217829177e-60,    1.3018259645417071e-61,    3.2714955778026969e-63,
    8.0648765492089835e-65,    1.9510308260798101e-66,    4.6333681070624742e-68,    1.0805416403914001e-69,
    2.4753625279050523e-71,    5.5721701969679372e-73,    1.2329037687099639e-74,    2.6821298225395547e-76
},
// x = 2.6000
{
    1.9124681453446907e-01,    1.1981210295227360e-01,    5.3002080358768799e-02,    1.7885025339256646e-02,
    4.8500890607701087e-03,    1.0962555135139722e-03,    2.1208496513407099e-04,    3.5830687843617528e-05,
    5.3694583439698622e-06,    7.2269097919921153e-07,    8.8255034437162528e-08,    9.8618548990523134e-09,
    1.0155487916997578e-09,    9.6962671170034269e-11,    8.6287449340167940e-12,    7.1897767523158836e-13,
    5.6318806255546177e-14,    4.1620573727329480e-15,    2.9111085337190055e-16,    1.9325997825132415e-17,
    1.2208859949144033e-18,    7.3564828405276269e-20,    4.2369097502185728e-21,    2.3369811303176018e-22,
    1.2366937721367908e-23,    6.2890212905740913e-25,    3.0780574472575433e-26,    1.4519571164482857e-27,
    6.6097009246314891e-29,    2.9072983559985697e-30,    1.2370039865502602e-31,    5.0966952460365152e-33,
    2.0355230875657934e-34,    7.8875271220339226e-36,    2.9680215032050744e-37,    1.0854671220458311e-38,
    3.8612853106912279e-40,    1.3370155825237144e-41,    4.5095918728236524e-43,    1.4826048028636271e-44,
    4.7542025841688185e-46,    1.4878439494954334e-47,    4.5468992410891777e-49,    1.3576591393559760e-50,
    3.9628901672577137e-52,    1.1313513331435413e-53,    3.1605012553182572e-55,    8.6434566431897270e-57,
    2.3151741527793204e-58,    6.0761501284175546e-60,    1.5631423434064709e-61,    3.9433328770288632e-63,
    9.7585750657566888e-65,    2.3698697040450838e-66,    5.6497452634568169e-68,    1.3226512881886576e-69,
    3.0416869590897967e-71,    6.8734058424494821e-73,    1.5266826217571104e-74,    3.3340445424868841e-76
},
// x = 2.6100
{
    1.9053499442022206e-01,    1.1960485446091894e-01,    5.3058150212269235e-02,    1.7960888537031514e-02,
    4.8871847872804623e-03,    1.1085272015816611e-03,    2.1523106413936209e-04,    3.6495081347290417e-05,
    5.4892173158538415e-06,    7.4155860034971299e-07,    9.0898002963209424e-08,    1.0195358116993267e-08,
    1.0538509743798314e-09,    1.0100012484929458e-10,    9.0220966285080836e-12,    7.5460675476028639e-13,
    5.9334407217330971e-14,    4.4016060354111812e-15,    3.0903892637647396e-16,    2.0594435438180776e-17,
    1.3059830473357267e-18,    7.8992932140635869e-20,    4.5669239459389199e-21,    2.5286410720626445e-22,
    1.3432360232240118e-23,    6.8569671593283150e-25,    3.3688771483637353e-26,    1.5952260352146419e-27,
    7.2897178347982444e-29,    3.2186919828407532e-30,    1.3747452131410427e-31,    5.6859291394640767e-33,
    2.2795587876445767e-34,    8.8670246404193761e-36,    3.3493971920924644e-37,    1.2296431439299310e-38,
    4.3909384899708976e-40,    1.5262482675716226e-41,    5.1676061443984118e-43,    1.7054586878001467e-44,
    5.4898096776596394e-46,    1.7246499057756882e-47,    5.2908202120535713e-49,    1.5858520417653885e-50,
    4.6467395022771874e-52,    1.3316750884929740e-53,    3.7344044664358381e-55,    1.0252211422728712e-56,
    2.7566305767275318e-58,    7.2625360401867400e-60,    1.8755270015258806e-61,    4.7495591862045741e-63,
    1.1798897286743810e-64,    2.8763696984416296e-66,    6.8835804888499637e-68,    1.6176935196144344e-69,
    3.7344858920258758e-71,    8.4713724307845225e-73,    1.8888439790533258e-74,    4.1408005454789080e-76
},
// x = 2.6200
{
    1.8982819526033007e-01,    1.1939768441129966e-01,    5.3113289445864784e-02,    1.8036445010794298e-02,
    4.9243142261853673e-03,    1.1208617910735854e-03,    2.1840594305199296e-04,    3.7168180510261924e-05,
    5.6110164665238598e-06,    7.6082175800789014e-07,    9.3606771046792140e-08,    1.0538478625204696e-08,
    1.0934090698043679e-09,    1.0518597439966593e-10,    9.4314710291843789e-12,    7.9182942014414147e-13,
    6.2496974043780102e-14,    4.6537928751549565e-15,    3.2798525354215507e-16,    2.1940057956582613e-17,
    1.3966045701238468e-18,    8.4795599682724597e-20,    4.9210638806564670e-21,    2.7351012946468868e-22,
    1.4584458961671122e-23,    7.4734735709886414e-25,    3.6857735700867479e-26,    1.7519402492550509e-27,
    8.0364056200377828e-29,    3.5619273384338485e-30,    1.5271537305077895e-31,    6.3404086256334110e-33,
    2.5516564050988564e-34,    9.9633458080044170e-36,    3.7778969450546570e-37,    1.3922555761241282e-38,
    4.9906128098839925e-40,    1.7413199350748067e-41,    5.9183392500061570e-43,    1.9606898439904587e-44,
    6.3355219433223262e-46,    1.9979454824108658e-47,    6.1526669825655274e-49,    1.8512323418960705e-50,
    5.4450839030045034e-52,    1.5664359136248927e-53,    4.4095465210987663e-55,    1.2152019371625370e-56,
    3.2799550513986263e-58,    8.6743340045053452e-60,    2.2486909175990276e-61,    5.7163450587115529e-63,
    1.4254937544094939e-64,    3.4884093605268388e-66,    8.3802320196849630e-68,    1.9769553907294122e-69,
    4.5813185031045018e-71,    1.0432119095306941e-72,    2.3349307369517726e-74,    5.1383234230754332e-76
},
// x = 2.6300
{
    1.8912636792420054e-01,    1.1919060244682590e-01,    5.3167505817554986e-02,    1.8111694048432297e-02,
    4.9614760308530747e-03,    1.1332589618628435e-03,    2.2160965043814074e-04,    3.7850043347318829e-05,
    5.7348784952576542e-06,    7.8048653386632427e-07,    9.6382623219569170e-08,    1.0891443519953814e-08,
    1.1342578359427054e-09,    1.0952492734254767e-10,    9.8574411713031270e-12,    8.3070910384398564e-13,
    6.5812951088849522e-14,    4.9192232614628484e-15,    3.4800274238575118e-16,    2.3367189876236588e-17,
    1.4930826240679013e-18,    9.0996877459428208e-20,    5.3009773928374934e-21,    2.9574336144838256e-22,
    1.5829868854987731e-23,    8.1424589918149781e-25,    4.0309593293289769e-26,    1.9232966931485077e-27,
    8.8559786381831153e-29,    3.9401061658548971e-30,    1.6957201474407649e-31,    7.0670406120576967e-33,
    2.8549057691490155e-34,    1.1189851803478679e-35,    4.2591119890462688e-37,    1.5755710543606934e-38,
    5.6692187189245306e-40,    1.9856304766961155e-41,    6.7743937454956698e-43,    2.2528401821744622e-44,
    7.3072667206674640e-46,    2.3131693453205984e-47,    7.1505358153845436e-49,    2.1596711468857347e-50,
    6.3765081959917997e-52,    1.8413771394597478e-53,    5.2032647812660250e-55,    1.4394031185305120e-56,
    3.8999044447805319e-58,    1.0353195049935485e-59,    2.6941408124209238e-61,    6.8748197258505852e-63,
    1.7209197874248345e-64,    4.2274186411721720e-66,    1.0194277083984296e-67,    2.4140702433219300e-69,
    5.6156007624656714e-71,    1.2836039355373833e-72,    2.8839340702674856e-74,    6.3706777580297995e-76
},
// x = 2.6400
{
    1.8842946386501908e-01,    1.1898361800429184e-01,    5.3220807041960159e-02,    1.8186634970276384e-02,
    4.9986688631969934e-03,    1.1457183911957416e-03,    2.2484223321473526e-04,    3.8540727638333335e-05,
    5.8608261787504023e-06,    8.0055906304666766e-07,    9.9226861369081059e-08,    1.1254483974431417e-08,
    1.1764328039588950e-09,    1.1402181572976831e-10,    1.0300597631718688e-11,    8.7131144194937470e-13,
    6.9289033070733830e-14,    5.1985285652014916e-15,    3.6914678965346566e-16,    2.4880376876404955e-17,
    1.5957676156652518e-18,    9.7622239179449905e-20,    5.7084169090594800e-21,    3.1967822957246806e-22,
    1.7175700761753230e-23,    8.8681391871481783e-25,    4.4068241126068531e-26,    2.1105930778359500e-27,
    9.7552004486253498e-29,    4.3566173372956857e-30,    1.8820793305444550e-31,    7.8734295982700231e-33,
    3.1927218663809742e-34,    1.2561366650202664e-35,    4.7992695492362679e-37,    1.7821237426062918e-38,
    6.4367566208616024e-40,    2.2630103225903425e-41,    7.7500225935854910e-43,    2.5870661279067925e-44,
    8.4231956825891831e-46,    2.6765434760215236e-47,    8.3052087336373367e-49,    2.5179367822279583e-50,
    7.4625194770228070e-52,    2.1631706558754536e-53,    6.1357768460574075e-55,    1.7038123779566847e-56,
    4.6338193193382958e-58,    1.2348250179929085e-59,    3.2255018648888216e-61,    8.2619847104872453e-63,
    2.0760119335717154e-64,    5.1190655997396291e-66,    1.2391336772431340e-67,    2.9454929403318801e-69,
    6.8778187632095222e-71,    1.5780909278028001e-72,    3.5590398261117919e-74,    7.8918668507518582e-76
},
// x = 2.6500
{
    1.8773743511190374e-01,    1.1877674031756655e-01,    5.3273200790130322e-02,    1.8261267128641501e-02,
    5.0358913937187241e-03,    1.1582397537477332e-03,    2.2810373665266159e-04,    3.9240290923356041e-05,
    5.9888823695141052e-06,    8.2104553402028068e-07,    1.0214080484039584e-07,    1.1627835285067690e-08,
    1.2199702907517524e-09,    1.1868159873040741e-10,    1.0761548970242348e-11,    9.1370433907604596e-13,
    7.2932173503698118e-14,    5.4923671432012507e-15,    3.9147538594576898e-16,    2.6484396033911506e-17,
    1.7050292202789768e-18,    1.0469866355750967e-19,    6.1452455722152257e-21,    3.4543685951536944e-22,
    1.8629573263388399e-23,    9.6550483384803053e-25,    4.8159479897998895e-26,    2.3152358880524187e-27,
    1.0741429691095975e-28,    4.8151620430955784e-30,    2.0880236656773144e-31,    8.7679447440197766e-33,
    3.5688774763864710e-34,    1.4094330241642713e-35,    5.4053020843493254e-37,    2.0147455978434661e-38,
    7.3044448441937032e-40,    2.5777728391606374e-41,    8.8613371352993184e-43,    2.9692187151756188e-44,
    9.7039843984132999e-46,    3.0951820757054771e-47,    9.6405386565242680e-49,    2.9338273863706746e-50,
    8.7279918356418170e-52,    2.5395623249321413e-53,    7.2306443654044345e-55,    2.0154344317178607e-56,
    5.5020630196464717e-58,    1.4717415076431748e-59,    3.8588967531032199e-61,    9.9217910194669996e-63,
    2.5025078009529430e-64,    6.1940722949597301e-66,    1.5050252522207975e-67,    3.5910688440005825e-69,
    8.4169869602507629e-71,    1.9385552898793649e-72,    4.3885324723874890e-74,    9.7680207054980128e-76
},
// x = 2.6600
{
    1.8705023426217302e-01,    1.1856997842122055e-01,    5.3324694689367964e-02,    1.8335589907371164e-02,
    5.0731423015490150e-03,    1.1708227216790134e-03,    2.3139420438016318e-04,    3.9948790497765097e-05,
    6.1190699942697052e-06,    8.4195218852263546e-07,    1.0512579053659544e-07,    1.2011736917933674e-08,
    1.2649074116050623e-09,    1.2350936525452196e-10,    1.1240922179463521e-11,    9.5795803480691032e-13,
    7.6749593367947905e-14,    5.8014253549249952e-15,    4.1504922419796892e-16,    2.8184266457007516e-17,
    1.8212573471069557e-18,    1.1225471588526621e-19,    6.6134436985160097e-21,    3.7314955698792384e-22,
    2.0199646473746314e-23,    1.0508061557548460e-24,    5.2611156642128739e-26,    2.5387489755887764e-27,
    1.1822669574428731e-28,    5.3197810683336302e-30,    2.3155174741358091e-31,    9.7597930597820316e-33,
    3.9875389242749078e-34,    1.5806966629064306e-35,    6.0849237211143875e-37,    2.2765999105944955e-38,
    8.2848620087815429e-40,    2.9347728461749983e-41,    1.0126540243015773e-42,    3.4059337230651351e-44,
    1.1173170754656733e-45,    3.5772150681606856e-47,    1.1183887839561778e-48,    3.4163224360556897e-50,
    1.0201676522494525e-51,    2.9795394315808612e-53,    8.5153091915813773e-55,    2.3824585843644416e-56,
    6.5285331369463394e-58,    1.7528915682206451e-59,    4.6133905612315945e-61,    1.1906408357160725e-62,
    3.0143928573310306e-64,    7.4891834855967902e-66,    1.8265680120206142e-67,    4.3747145348816070e-69,
    1.0292399459591838e-70,    2.3794261076282101e-72,    5.4068890187508616e-74,    1.2080053956885892e-75
},
// x = 2.6700
{
    1.8636781447372727e-01,    1.1836334115409025e-01,    5.3375296323064043e-02,    1.8409602721385997e-02,
    5.1104202744865533e-03,    1.1834669646897552e-03,    2.3471367838643844e-04,    4.0666283407470926e-05,
    6.2514120523320683e-06,    8.6328532146149200e-07,    1.0818317301808279e-07,    1.2406432555222241e-08,
    1.3112820929473637e-09,    1.2851033661397384e-10,    1.1739363143133180e-11,    1.0041451717033742e-12,
    8.0748790022841537e-14,    6.1264186120737009e-15,    4.3993181213761609e-16,    2.9985260353929142e-17,
    1.9448631476397329e-18,    1.2032063361863739e-19,    7.1151155792797013e-21,    4.0295531617056540e-22,
    2.1894657925176512e-23,    1.1432418882998368e-24,    5.7453317202106201e-26,    2.7827827898630683e-27,
    1.3007621241755188e-28,    5.8748843198254241e-30,    2.5667126785147422e-31,    1.0859099248672078e-32,
    4.4533052325650471e-34,    1.7719469019348162e-35,    6.8467146012731301e-37,    2.5712184597386311e-38,
    9.3921053389424485e-40,    3.3394719378849089e-41,    1.1566187580348665e-42,    3.9047330652560233e-44,
    1.2857537081782196e-45,    4.1319280861427364e-47,    1.2966626731748950e-48,    3.9757558106722579e-50,
    1.1916786861814168e-51,    3.4935234008688019e-53,    1.0021712785671284e-54,    2.8144530507142091e-56,
    7.7412569128548663e-58,    2.0863070927798306e-59,    5.5115126599886158e-61,    1.4277719601669186e-62,
    3.6283192697926452e-64,    9.0483159677712789e-66,    2.2151179142654858e-67,    5.3252316312816309e-69,
    1.2575731708765531e-70,    2.9182114670024289e-72,    6.6561018262291502e-74,    1.4926892166228414e-75
},
// x = 2.6800
{
    1.8569012945754043e-01,    1.1815683716278029e-01,    5.3425013230547568e-02,    1.8483305016236341e-02,
    5.1477240090347168e-03,    1.1961721500749903e-03,    2.3806219902542910e-04,    4.1392826444177415e-05,
    6.3859316139882896e-06,    8.8505128081888190e-07,    1.1131432460069383e-07,    1.2812170141802946e-08,
    1.3591330852208314e-09,    1.3368986922056376e-10,    1.2257537103211753e-11,    1.0523408649141622e-12,
    8.4937546368841271e-14,    6.4680924620112972e-15,    4.6618958884302481e-16,    3.1892914551627385e-17,
    2.0762800693428188e-18,    1.2892841615889157e-19,    7.6524966441852558e-21,    4.3500235727344239e-22,
    2.3723960658470034e-23,    1.2433750849977418e-24,    6.2718369333902672e-26,    3.0491242899654697e-27,
    1.4305741296950313e-28,    6.4852827782774558e-30,    2.8439658205171478e-31,    1.2076992772002721e-32,
    4.9712509791943983e-34,    1.9854203061090279e-35,    7.7002139218272307e-37,    2.9025426535105285e-38,
    1.0641966631434572e-39,    3.7980113660341778e-41,    1.3203480219527264e-42,    4.4741387819414578e-44,
    1.4787541416250501e-45,    4.7699210597749732e-47,    1.5024701275786042e-48,    4.6240133502943378e-50,
    1.3911668487253496e-51,    4.0935914676173070e-53,    1.1787011388760165e-54,    3.3225901744143949e-56,
    9.1730839330177718e-58,    2.4814374639340644e-59,    6.5798684852200354e-61,    1.7109079369616260e-62,
    4.3641006287132296e-64,    1.0923921299497410e-65,    2.6842889885160214e-67,    6.4772790404609340e-69,
    1.5353560863082805e-70,    3.5761392357158935e-72,    8.1872769884918012e-74,    1.8429385011911605e-75
},
// x = 2.6900
{
    1.8501713347026158e-01,    1.1795047490510588e-01,    5.3473852906946492e-02,    1.8556696267658900e-02,
    5.1850522104362933e-03,    1.2089379427791241e-03,    2.4143980501979714e-04,    4.2128476140699677e-05,
    6.5226518188694714e-06,    9.0725646754691523e-07,    1.1452063545259708e-07,    1.3229201931844618e-08,
    1.4084999758586704e-09,    1.3905345732166066e-10,    1.2796129135682654e-11,    1.1026227734089379e-12,
    8.9323940263671156e-14,    6.8272237059085428e-15,    4.9389204553030075e-16,    3.3913042480612936e-17,
    2.2159649563587115e-18,    1.3811191901176272e-19,    8.2279610033951616e-21,    4.6944869474336494e-22,
    2.5697563641200000e-23,    1.3518105728072696e-24,    6.8441257121649276e-26,    3.3397075851837457e-27,
    1.5727303796674767e-28,    7.1562230630688185e-30,    3.1498565408587429e-31,    1.3425702757169236e-32,
    5.5469731945711723e-34,    2.2235930138524049e-35,    8.6560225224319707e-37,    3.2749690650230502e-38,
    1.2052127761468679e-39,    4.3172933233680135e-41,    1.5064592227007917e-42,    5.1238011389821093e-44,
    1.6997803974955194e-45,    5.5032877874223515e-47,    1.7399277704077906e-48,    5.3747582527522532e-50,
    1.6230566923773115e-51,    4.7937315039003967e-53,    1.3854401284789207e-54,    3.9199073004105564e-56,
    1.0862491517724346e-57,    2.9493925322609312e-59,    7.8498562435908027e-61,    2.0487381989558790e-62,
    5.2452959158808984e-64,    1.3178600432598326e-65,    3.2503906466180688e-67,    7.8725336626622692e-69,
    1.8730386050148572e-70,    4.3789278414855185e-72,    1.0062564243946927e-73,    2.2735048989366967e-75
},
// x = 2.7000
{
    1.8434878130692356e-01,    1.1774426265347510e-01,    5.3521822803062356e-02,    1.8629775981137374e-02,
    5.2224035927061884e-03,    1.2217640054500887e-03,    2.4484653346508651e-04,    4.2873288766338635e-05,
    6.6615958743164260e-06,    9.2990733545744492e-07,    1.1780351368996425e-07,    1.3657784535500135e-08,
    1.4594232023704750e-09,    1.4460673577352413e-10,    1.3355844635233834e-11,    1.1550711728643908e-12,
    9.3916354198243528e-14,    7.2046215525248263e-15,    5.2311185069953296e-16,    3.6051746642338654e-17,
    2.3643991990858828e-18,    1.4790695251599529e-19,    8.8440293866969082e-21,    5.0646273771359789e-22,
    2.7826174645389099e-23,    1.4691978528327268e-24,    7.4659647437594014e-26,    3.6566253540292557e-27,
    1.7283467033140472e-28,    7.8934248107124203e-30,    3.4872076398522721e-31,    1.4918661416543059e-32,
    6.1866426588910182e-34,    2.4892052546165763e-35,    9.7259159535889551e-37,    3.6933998105106805e-38,
    1.3642377801616222e-39,    4.9050715551803937e-41,    1.7179037226241690e-42,    5.8646425114711080e-44,
    1.9527655641028928e-45,    6.3458191624313970e-47,    2.0137475036125783e-48,    6.2436880955026128e-50,
    1.8924506172919242e-51,    5.6101347996028619e-53,    1.6274070554038439e-54,    4.6216087674042268e-56,
    1.2854520579110917e-57,    3.5032260199363561e-59,    9.3585060109458144e-61,    2.4515491751436817e-62,
    6.2998983606620845e-64,    1.5887015528707730e-65,    3.9329474334617612e-67,    9.5610751028577254e-69,
    2.2832245091426286e-70,    5.3577127422729659e-72,    1.2357485448838057e-73,    2.8023811098096593e-75
},
// x = 2.7100
{
    1.8368502829375802e-01,    1.1753820849821320e-01,    5.3568930325256357e-02,    1.8702543691467160e-02,
    5.2597768786621889e-03,    1.2346499984931104e-03,    2.4828241983406644e-04,    4.3627320322312391e-05,
    6.8027870537394367e-06,    9.5301039110559983e-07,    1.2116438547139391e-07,    1.4098178965646862e-08,
    1.5119440655349090e-09,    1.5035548285249865e-10,    1.3937409808908494e-11,    1.2097690302305508e-12,
    9.8723485237987299e-14,    7.6011288085645184e-15,    5.5392497977402204e-16,    3.8315431576024523e-17,
    2.5220899345549970e-18,    1.5835138534006643e-19,    9.5033774985898820e-21,    5.4622392436748016e-22,
    3.0121245722140001e-23,    1.5962341885649815e-24,    8.1414129219692812e-26,    4.0021410949734927e-27,
    1.8986345454814866e-28,    8.7031210825446615e-30,    3.8591068463015509e-31,    1.6570616699841621e-32,
    6.8970599915030622e-34,    2.7852882583370709e-35,    1.0922969046091807e-36,    4.1632982620371366e-38,
    1.5434854038796718e-39,    5.5700523243614968e-41,    1.9580078386683271e-42,    6.7090189201237773e-44,
    2.2421756064138029e-45,    7.3132330566926024e-47,    2.3293196776936163e-48,    7.2488277642848678e-50,
    2.2052293807622825e-51,    6.5615322611304898e-53,    1.9104296077643297e-54,    5.4454152971825603e-56,
    1.5201862427680514e-57,    4.1582658725582672e-59,    1.1149461508245649e-60,    2.9315097065674825e-62,
    7.5611474910162167e-64,    1.9138152143344950e-65,    4.7553163375806908e-67,    1.1603036470590317e-68,
    2.7811042392731587e-70,    6.5501592299045107e-72,    1.5163741845302906e-73,    3.4514959545481251e-75
},
// x = 2.7200
{
    1.8302583028111416e-01,    1.1733232035082913e-01,    5.3615182835346761e-02,    1.8774998962324121e-02,
    5.2971707999537824e-03,    1.2475955801241158e-03,    2.5174749798125294e-04,    4.4390626537245513e-05,
    6.9462486949725246e-06,    9.7657219366723912e-07,    1.2460469509107460e-07,    1.4550650684677446e-08,
    1.5661047426991621e-09,    1.5630562310427076e-10,    1.4541572178827629e-11,    1.2668020800055207e-12,
    1.0375435523530786e-13,    8.0176231065643299e-15,    5.8641084936983330e-16,    4.0710817342335037e-17,
    2.6895712995882415e-18,    1.6948525295347428e-19,    1.0208844809051222e-20,    5.8892339196516297e-22,
    3.2595021417889986e-23,    1.7336678928801990e-24,    8.8748426386264570e-26,    4.3787022654842675e-27,
    2.0849087095769054e-28,    9.5921020326679097e-30,    4.2689304320269975e-31,    1.8397754960301728e-32,
    7.6857169571133348e-34,    3.1151937788246896e-35,    1.2261693096635840e-36,    4.6907506339813338e-38,
    1.7454309405152486e-39,    6.3220068631984257e-41,    2.2305186774494674e-42,    7.6709013010340478e-44,
    2.5730789873718788e-45,    8.4234342280899451e-47,    2.6928074767873745e-48,    8.4108631293962004e-50,
    2.5681670172661136e-51,    7.6695802533996035e-53,    2.2412707243363836e-54,    6.4119679847589864e-56,
    1.7966120124728590e-57,    4.9324991021039320e-59,    1.3274128100306806e-60,    3.5030060341251066e-62,
    9.0684857889477143e-64,    2.3037994210243452e-65,    5.7454194769393221e-67,    1.4070571038104253e-68,
    3.3849724188374638e-70,    8.0017980809566266e-72,    1.8592596094803137e-73,    4.2475548832178843e-75
},
// x = 2.7300
{
    1.8237114363648074e-01,    1.1712660594722593e-01,    5.3660587650518163e-02,    1.8847141385837332e-02,
    5.3345840970890953e-03,    1.2606004064227509e-03,    2.5524180014760742e-04,    4.5163262862715286e-05,
    7.0920041986223882e-06,    1.0005993548103253e-06,    1.2812590507066882e-07,    1.5015469651334452e-08,
    1.6219483011844577e-09,    1.6246323023135980e-10,    1.5169101094085457e-11,    1.3262589022468550e-12,
    1.0901832131897881e-13,    8.4550181712861196e-15,    6.2065245633626873e-16,    4.3244953541823203e-17,
    2.8674057387942529e-18,    1.8135087128680953e-19,    1.0963443800548360e-20,    6.3476468436410480e-22,
    3.5260589884286430e-23,    1.8823018256304772e-24,    9.6709625255516908e-26,    4.7889543695261195e-27,
    2.2885956909329599e-28,    1.0567762083676499e-29,    4.7203688197236249e-31,    2.0417834477189014e-32,
    8.5608634492346041e-34,    3.4826264725684004e-35,    1.3758186887242604e-36,    5.2825340345567339e-38,
    1.9728409091398279e-39,    7.1718965637812978e-41,    2.5396553537574032e-42,    8.7660788245205075e-44,
    2.9512250500030675e-45,    9.6968080278931746e-47,    3.1112539769392480e-48,    9.7535209394645311e-50,
    2.9890621648402109e-51,    8.9593031722611149e-53,    2.6277740865060307e-54,    7.5452951526616439e-56,
    2.1219271549079944e-57,    5.8470198383702571e-59,    1.5793014330659472e-60,    4.1830347196819354e-62,
    1.0868684973055396e-63,    2.7712685067826029e-65,    6.9366131382278651e-67,    1.7050193582921835e-68,
    4.1168462706372961e-70,    9.7676275366829305e-72,    2.2778943789193918e-73,    5.2230555998228706e-75
},
// x = 2.7400
{
    1.8172092523760874e-01,    1.1692107285085523e-01,    5.3705152043241673e-02,    1.8918970582165982e-02,
    5.3720155194599854e-03,    1.2736641313850136e-03,    2.5876535696540552e-04,    4.5945284468856092e-05,
    7.2400770264123880e-06,    1.0250985385602741e-06,    1.3172949624990344e-07,    1.5492910367583112e-08,
    1.6795187117970253e-09,    1.6883453001902168e-10,    1.5820788251920178e-11,    1.3882310023482414e-12,
    1.1452508666636389e-13,    8.9142651256099422e-15,    6.5673652171140547e-16,    4.5925233886583719e-17,
    3.0561853695204704e-18,    1.9399295580297652e-19,    1.1770369692734138e-20,    6.8396449894889771e-22,
    3.8131937041362797e-23,    2.0429971143076233e-24,    1.0534841738878128e-25,    5.2357560574750474e-27,
    2.5112426428216296e-28,    1.1638150875307686e-29,    5.2174543429679262e-31,    2.2650330744050550e-32,
    9.5315806497322416e-34,    3.8916793955833372e-35,    1.5430302867871422e-36,    5.9461916303160612e-38,
    2.2288059388209960e-39,    8.1320122893121407e-41,    2.8901661990617185e-42,    1.0012386839423505e-43,
    3.3831322206363549e-45,    1.1156552140848596e-46,    3.5927035179862148e-48,    1.1304001110341586e-49,
    3.4768880602372887e-51,    1.0459600811708218e-52,    3.0790315312745736e-54,    8.8733515426961874e-56,
    2.5045365441428334e-57,    6.9265506576959623e-59,    1.8777298658252690e-60,    4.9916632305534800e-62,
    1.3017171135588393e-63,    3.3312260396808570e-65,    8.3687178565772866e-67,    2.0645565898955573e-68,
    5.0032040806852594e-70,    1.1914033355772376e-71,    2.7886211457610334e-73,    6.4175143840212593e-75
},
// x = 2.7500
{
    1.8107513246573381e-01,    1.1671572845581754e-01,    5.3748883241205594e-02,    1.8990486199080100e-02,
    5.4094638253652967e-03,    1.2867864069755050e-03,    2.6231819746327543e-04,    4.6736746240021485e-05,
    7.3904906995218282e-06,    1.0500764611592868e-06,    1.3541696787585085e-07,    1.5983251925516058e-08,
    1.7388608624437962e-09,    1.7542590329973034e-10,    1.6497448228261396e-11,    1.4528128926102140e-12,
    1.2028471156443940e-13,    9.3963538369406489e-15,    6.9475363974028484e-16,    4.8759411344087370e-17,
    3.2565334059551701e-18,    2.0745874621027980e-19,    1.2633010667157563e-20,    7.3675347497422545e-22,
    4.1224003961715004e-23,    2.2166771109398750e-24,    1.1471935883003872e-25,    5.7221953064011840e-27,
    2.7545270201510427e-28,    1.2812028269932842e-29,    5.7645913290867830e-31,    2.5116594503969075e-32,
    1.0607860904778333e-33,    4.3468729038462101e-35,    1.7297829953021853e-36,    6.6901156334538846e-38,
    2.5167772104568619e-39,    9.2161293320243489e-41,    3.2873926320415765e-42,    1.1429962307934255e-43,
    3.8761872168262912e-45,    1.2829051100456697e-46,    4.1463392339063178e-48,    1.3093468382626321e-49,
    4.0419637663544450e-51,    1.2203829697391434e-52,    3.6055755812797966e-54,    1.0428640699752124e-55,
    2.9542486381531102e-57,    8.2000488120680294e-59,    2.2310658086191036e-60,    5.9525695033187367e-62,
    1.5579582851776057e-63,    4.0015053678805221e-65,    1.0089238568197427e-66,    2.4980808495433315e-68,
    6.0758664043485879e-70,    1.4521088473642591e-71,    3.4112244616869702e-73,    7.8789457380332876e-75
},
// x = 2.7600
{
    1.8043372319889664e-01,    1.1651057998990888e-01,    5.3791788427256522e-02,    1.9061687911545548e-02,
    5.4469277820323233e-03,    1.2999668831792854e-03,    2.6590034907140265e-04,    4.7537702770504648e-05,
    7.5432687969208495e-06,    1.0755398909197213e-06,    1.3918983769088595e-07,    1.6486778054284055e-08,
    1.8000205718521728e-09,    1.8224388895640706e-10,    1.7199919017756328e-11,    1.5201021756340059e-12,
    1.2630762476568505e-13,    9.9023143051621580e-15,    7.3479843210699612e-16,    5.1755613872720253e-17,
    3.4691056446446646e-18,    2.2179813705677120e-19,    1.3554958615256413e-20,    7.9337702541678321e-22,
    4.4552747651771885e-23,    2.4043315990941661e-24,    1.2488114677097863e-25,    6.2516067529093196e-27,
    3.0202669488711898e-28,    1.4098923718821496e-29,    6.3665886883341252e-31,    2.7840023590273017e-32,
    1.1800694902240948e-33,    4.8531972676602023e-35,    1.9382694515084056e-36,    7.5236388891941217e-38,
    2.8406068244110829e-39,    1.0439679701030666e-40,    3.7373404353606772e-42,    1.3041529915738135e-43,
    4.4387565805554574e-45,    1.4744298890605413e-46,    4.7826388154252930e-48,    1.5157611215927631e-49,
    4.6961495350606427e-51,    1.4230468811791058e-52,    4.2196007382283551e-54,    1.2248932978129350e-55,
    3.4825030026717950e-57,    9.7014107683346770e-59,    2.6491401169662140e-60,    7.0936736394730158e-62,
    1.8633602063579011e-63,    4.8032891983884116e-65,    1.2154808957272214e-66,    3.0204435236636357e-68,
    7.3730468858608016e-70,    1.7685305394639224e-71,    4.1696380944579319e-73,    9.6656463936627341e-75
},
// x = 2.7700
{
    1.7979665580536000e-01,    1.1630563451761471e-01,    5.3833874739351271e-02,    1.9132575421312813e-02,
    5.4844061656365330e-03,    1.3132052080533352e-03,    2.6951183762689704e-04,    4.8348208360316846e-05,
    7.6984349537011501e-06,    1.1014956480715121e-06,    1.4304964201930322e-07,    1.7003777167046567e-08,
    1.8630446033930883e-09,    1.8929518696455193e-10,    1.7929062583376078e-11,    1.5901996295677455e-12,
    1.3260463514498520e-13,    1.0433218093192453e-14,    7.7696970753547757e-16,    5.4922360769106239e-17,
    3.6945920137655177e-18,    2.3706381445415538e-19,    1.4540020433861548e-20,    8.5409621452759319e-22,
    4.8135205415014494e-23,    2.6070212656036622e-24,    1.3589691473047590e-25,    6.8275902552053487e-27,
    3.3104323722890930e-28,    1.5509200314438895e-29,    7.0286952064406233e-31,    3.0846249714046955e-32,
    1.3122166783734108e-33,    5.4161593371255292e-35,    2.1709181300515395e-36,    8.4571359141157620e-38,
    3.2045924984024907e-39,    1.1819943596556953e-40,    4.2467592625227980e-42,    1.4872722396103634e-43,
    5.0803120094880955e-45,    1.6936375578878694e-46,    5.5135508349149702e-48,    1.7537276793041230e-49,
    5.4530695915522756e-51,    1.6583881555578782e-52,    4.9352176978753765e-54,    1.4378093397913482e-55,
    4.1026336090186198e-57,    1.1470290519762009e-58,    3.1434954574673551e-60,    8.4478770110066567e-62,
    2.2271104143393270e-63,    5.7617219947658473e-65,    1.4632900073927804e-66,    3.6494024982301133e-68,
    8.9406044858376878e-70,    2.1522928161812354e-71,    5.0927941105384868e-73,    1.1848344736451132e-74
},
// x = 2.7800
{
    1.7916388913712039e-01,    1.1610089894305209e-01,    5.3875149270517432e-02,    1.9203148456509996e-02,
    5.5218977613195599e-03,    1.3265010277776241e-03,    2.7315268737931918e-04,    4.9168317011024447e-05,
    7.8560128594029319e-06,    1.1279506046036829e-06,    1.4699793585257975e-07,    1.7534542407936226e-08,
    1.9279806790066627e-09,    1.9658666147343946e-10,    1.8685765415703782e-11,    1.6632092952345559e-12,
    1.3918694366377848e-13,    1.0990179801213524e-14,    8.2137062691763621e-16,    5.8268579647866832e-17,
    3.9337181885689038e-18,    2.5331139918878533e-19,    1.5592229893445306e-20,    9.1918868337579812e-22,
    5.1989562991162815e-23,    2.8258824524247071e-24,    1.4783454740022225e-25,    7.4540307657926127e-27,
    3.6271570288642216e-28,    1.7054123876765265e-29,    7.7566377521697377e-31,    3.4163341428786033e-32,
    1.4585557876500361e-33,    6.0418336249212658e-35,    2.4304176150552715e-36,    9.5021343164546746e-38,
    3.6135270398709971e-39,    1.3376262117153592e-40,    4.8232312863990160e-42,    1.6952438998042100e-43,
    5.8115711280586805e-45,    1.9443984633719626e-46,    6.3526942503062155e-48,    2.0279192135873405e-49,
    6.3283660580639587e-51,    1.9315187395263880e-52,    5.7687452257838974e-54,    1.6867035627100998e-55,
    4.8301733572950255e-57,    1.3553049488871441e-58,    3.7276760062730042e-60,    1.0053926512682192e-61,
    2.6600681209710946e-63,    6.9066313165474923e-65,    1.7603840265542957e-66,    4.4061764605142750e-68,
    1.0833534712576517e-69,    2.6173866972009892e-71,    6.2156414213468754e-73,    1.4512788643552323e-74
},
// x = 2.7900
{
    1.7853538252351395e-01,    1.1589638001286098e-01,    5.3915619068824673e-02,    1.9273406771239648e-02,
    5.5594013632055041e-03,    1.3398539867057740e-03,    2.7682292099636302e-04,    4.9998082421644601e-05,
    8.0160262563382079e-06,    1.1549116840999884e-06,    1.5103629293326998e-07,    1.8079371699030985e-08,
    1.9948774932296575e-09,    2.0412534392652392e-10,    1.9470939102005911e-11,    1.7392385651722322e-12,
    1.4606615564777291e-13,    1.1574358585670308e-14,    8.6810887413110149e-16,    6.1803624075055105e-17,
    4.1872472754928221e-18,    2.7059959648676632e-19,    1.6715860105381259e-20,    9.8894962577852511e-22,
    5.6135226674909339e-23,    3.0621322048492646e-24,    1.6076701637437892e-25,    8.1351196012061927e-27,
    3.9727513196300727e-28,    1.8745937445827692e-29,    8.5566626270869944e-31,    3.7822024598046119e-32,
    1.6205459786254972e-33,    6.7369182039942181e-35,    2.7197432577091386e-36,    1.0671437616780030e-37,
    4.0727530805309981e-39,    1.5130273455183856e-40,    5.4752699977058502e-42,    1.9313246461616189e-43,
    6.6446555277864645e-45,    2.2311058365172951e-46,    7.3155840253380630e-48,    2.3436782601788854e-49,
    7.3399882211100993e-51,    2.2483258462734930e-52,    6.7390450926109818e-54,    1.9774820698336219e-55,
    5.6832060807234652e-57,    1.6003858544973996e-58,    4.4175647649515258e-60,    1.1957424540146604e-61,
    3.1750601662222990e-63,    8.2733769444040275e-65,    2.1163201605377744e-66,    5.3161021596804030e-68,
    1.3117744283678340e-69,    3.1806397676687274e-71,    7.5803667642305742e-73,    1.7762858946162610e-74
},
// x = 2.8000
{
    1.7791109576491373e-01,    1.1569208431904561e-01,    5.3955291137364941e-02,    1.9343350145179725e-02,
    5.5969157744155951e-03,    1.3532637274153204e-03,    2.8052255956969381e-04,    5.0837557984599458e-05,
    8.1784989379109471e-06,    1.1823858615687163e-06,    1.5516630583751830e-07,    1.8638567787328020e-08,
    2.0637847273239635e-09,    2.1191843622120625e-10,    2.0285520905188168e-11,    1.8183982746145131e-12,
    1.5325429338463652e-13,    1.2186959724153897e-14,    9.1729683271283893e-16,    6.5537291877099882e-17,
    4.4559815675193088e-18,    2.8899035271001150e-19,    1.7915436615557762e-20,    1.0636928171192329e-21,
    6.0592899627775517e-23,    3.3170736331613590e-24,    1.7477273805076334e-25,    8.8753772004823083e-27,
    4.3497161271808600e-28,    2.0597941578411010e-29,    9.4355803013977751e-31,    4.1855921794423937e-32,
    1.7997897652390743e-33,    7.5087958516860613e-35,    3.0421864427666245e-36,    1.1979260582595948e-37,
    4.5882236083680650e-39,    1.7106175064906158e-40,    6.2124302682051140e-42,    2.1991827342093769e-43,
    7.5932681143710799e-45,    2.5587439806007124e-46,    8.4198861618304614e-48,    2.7071100450369912e-49,
    8.5085218940952656e-51,    2.6155858424244482e-52,    7.8679062158121460e-54,    2.3169921725575616e-55,
    6.6827732061318433e-57,    1.8885975761734542e-58,    5.2317760899897914e-60,    1.4212008557077658e-61,
    3.7872279219400778e-63,    9.9038498023630667e-65,    2.5424617513338909e-66,    6.4094132418592536e-68,
    1.5872161682576228e-69,    3.8622771052750265e-71,    9.2378573395139876e-73,    2.1724312638268704e-74
},
// x = 2.8100
{
    1.7729098912651856e-01,    1.1548801830176665e-01,    5.3994172434240997e-02,    1.9412978383188416e-02,
    5.6344398070811872e-03,    1.3667298907575750e-03,    2.8425162262093269e-04,    5.1686796781729076e-05,
    8.3434547469340831e-06,    1.2103801632666801e-06,    1.5938958605617409e-07,    1.9212438291713026e-08,
    2.1347530635052741e-09,    2.1997331390809729e-10,    2.1130474352736746e-11,    1.9008027944440315e-12,
    1.6076380904815149e-13,    1.2829236227304458e-14,    9.6905176855866170e-16,    6.9479844147704707e-17,
    4.7407643734366107e-18,    3.0854901927017454e-19,    1.9195751152798474e-20,    1.1437516986688885e-21,
    6.5384662607091398e-23,    3.5921016057293341e-24,    1.8993595506433442e-25,    9.6796774696457447e-27,
    4.7607576521837955e-28,    2.2624580874348306e-29,    1.0400813797496120e-30,    4.6301812168112140e-32,
    1.9980464431825581e-33,    8.3656009085010061e-35,    3.4013867072292258e-36,    1.3441378294013060e-37,
    5.1665688842908212e-39,    1.9331014538664458e-40,    7.0454309106346404e-42,    2.5029481053517787e-43,
    8.6728920299960384e-45,    2.9329650326359385e-46,    9.6857058400649223e-48,    3.1251877764434582e-49,
    9.8575642435411607e-51,    3.0410943254387344e-52,    9.1804850029133564e-54,    2.7131678908974983e-55,
    7.8533432954721805e-57,    2.2273227095933542e-58,    6.1921122073420863e-60,    1.6880727906814966e-61,
    4.5144336696993352e-63,    1.1847646372602757e-64,    3.0523107367200966e-66,    7.7221625497209921e-68,
    1.9191245527296758e-69,    4.6865903454213394e-71,    1.1249451729006258e-72,    2.6549280059231922e-74
},
// x = 2.8200
{
    1.7667502333223026e-01,    1.1528418825208528e-01,    5.4032269872565090e-02,    1.9482291314913105e-02,
    5.6719722823552268e-03,    1.3802521159070689e-03,    2.8801012810779298e-04,    5.2545851580363638e-05,
    8.5109175739438956e-06,    1.2389016665174575e-06,    1.6370776407449797e-07,    1.9801295749919106e-08,
    2.2078341992711806e-09,    2.2829752942992311e-10,    2.2006789835746460e-11,    1.9865701261474426e-12,
    1.6860759795542160e-13,    1.3502490498891678e-14,    1.0234960188227794e-15,    7.3642024975775812e-17,
    5.0424819237523090e-18,    3.2934452405784927e-19,    2.0561876061696402e-20,    1.2294805201409640e-21,
    7.0534059346994736e-23,    3.8887087924732728e-24,    2.0634714269085291e-25,    1.0553273815399818e-26,
    5.2088033376421374e-28,    2.4841537186992453e-29,    1.1460451001871258e-30,    5.1199913441143893e-32,
    2.2172467147614232e-33,    9.3162923593354392e-35,    3.8013669760526741e-36,    1.5075290270388807e-37,
    5.8151703874090882e-39,    2.1835012202135711e-40,    7.9862910964037563e-42,    2.8472683586448693e-43,
    9.9010136752229669e-45,    3.3601753361449500e-46,    1.1135912810525132e-47,    3.6058719795729018e-49,
    1.1414150140647438e-50,    3.5338146167957019e-52,    1.0705809852799424e-53,    3.1751972554652172e-55,
    9.2233538917747359e-57,    2.6251721242616802e-58,    7.3240938347979919e-60,    2.0037649900573995e-61,
    5.3777363613923190e-63,    1.4163448581320044e-64,    3.6618996857742948e-66,    9.2973136108991310e-68,
    2.3187963778298259e-69,    5.6827351980477538e-71,    1.3689034486495585e-72,    3.2421664166063995e-74
},
// x = 2.8300
{
    1.7606315955861984e-01,    1.1508060031465939e-01,    5.4069590320465051e-02,    1.9551288794403071e-02,
    5.7095120304220832e-03,    1.3938300404105980e-03,    2.9179809243035689e-04,    5.3414774829454057e-05,
    8.6809113555119137e-06,    1.2679574995238925e-06,    1.6812248945044450e-07,    2.0405457665468930e-08,
    2.2830808618277818e-09,    2.3689881540019616e-10,    2.2915485218135927e-11,    2.0758219988032638e-12,
    1.7679901216379309e-13,    1.4208076045250615e-14,    1.0807571871954346e-15,    7.8035081918085726e-17,
    5.3620653560906196e-18,    3.5144955069514148e-19,    2.2019179450658245e-20,    1.3212555433321725e-21,
    7.6066186837704093e-23,    4.2084920786388128e-24,    2.2410344173965477e-25,    1.1501826977432237e-26,
    5.6970189556531980e-28,    2.7265830003533387e-29,    1.2623301206311114e-30,    5.6594187809776201e-32,
    2.4595086113585016e-33,    1.0370733687775870e-34,    4.2465722030815935e-36,    1.6900401110507567e-37,
    6.5422424948901431e-39,    2.4651918741466437e-40,    9.0484821340228602e-42,    3.2373712499642693e-43,
    1.1297372638501355e-44,    3.8476325840236240e-46,    1.2796508679423971e-47,    4.1582456807356665e-49,
    1.3209236879696160e-50,    4.1040471916445489e-52,    1.2477358860331503e-53,    3.7137145744794975e-55,
    1.0825836460400862e-56,    3.0921834587060928e-58,    8.6575765880276434e-60,    2.3769732262623198e-61,
    6.4019482743229828e-63,    1.6920644110368974e-64,    4.3902537947819288e-66,    1.1186031526920955e-67,
    2.7997329860791604e-69,    6.8856814458596560e-71,    1.6645540160498016e-72,    3.9563618375091551e-74
},
// x = 2.8400
{
    1.7545535942897969e-01,    1.1487726049039367e-01,    5.4106140601099077e-02,    1.9619970699726354e-02,
    5.7470578905058972e-03,    1.4074633002358421e-03,    2.9561553043749449e-04,    5.4293618655761938e-05,
    8.8534600725545415e-06,    1.2975548411748956e-06,    1.7263543089150737e-07,    2.1025246554593800e-08,
    2.3605468226139424e-09,    2.4578508792178247e-10,    2.3857606456150468e-11,    2.1686839681332549e-12,
    1.8535187441425039e-13,    1.4947399235273347e-14,    1.1409683457408593e-15,    8.2670787241034013e-17,
    5.7004927829973937e-18,    3.7494072593081322e-19,    2.3573341087191151e-20,    1.4194763098268319e-21,
    8.2007790761180293e-23,    4.5531593698482165e-24,    2.4330911953975482e-25,    1.2531434775327715e-26,
    6.2288269361949669e-28,    2.9915924513869222e-29,    1.3898956201009448e-30,    6.2532673672860142e-32,
    2.7271548227088510e-33,    1.1539780100265404e-34,    4.7419117307440648e-36,    1.8938219232296244e-37,
    7.3569226700060978e-39,    2.7819411505890867e-40,    1.0247096267319178e-41,    3.6791344497322496e-43,
    1.2884241655518086e-44,    4.4035550246283201e-46,    1.4697041287437712e-47,    4.7926674726639423e-49,
    1.5278254981273001e-50,    4.7636228988097371e-52,    1.4533721002368717e-53,    4.3410212735035359e-55,
    1.2699136775893026e-56,    3.6400507513440982e-58,    1.0227466630143344e-59,    2.8179004832595496e-61,
    7.6162859273740141e-63,    2.0201227867635993e-64,    5.2599349900288057e-66,    1.3449208722619126e-67,
    3.3780597093590047e-69,    8.3373438315347451e-71,    2.0225944769774233e-72,    4.8243313108565558e-74
},
// x = 2.8500
{
    1.7485158500746117e-01,    1.1467417463904366e-01,    5.4141927492678391e-02,    1.9688336932590374e-02,
    5.7846087108774213e-03,    1.4211515298195821e-03,    2.9946245543342281e-04,    5.5182434860108563e-05,
    9.0285877486409764e-06,    1.3277009208466284e-06,    1.7724827633011841e-07,    2.1660989993124284e-08,
    2.4402869119223828e-09,    2.5496444994549147e-10,    2.4834228228253534e-11,    2.2652855176484640e-12,
    1.9428049242813371e-13,    1.5721921112178784e-14,    1.2042682434819850e-15,    8.7561459956536533e-17,
    6.0587914451690141e-18,    3.9989881550863984e-19,    2.5230369073741883e-20,    1.5245669758735411e-21,
    8.8387366353667279e-23,    4.9245368104886161e-24,    2.6407606071315935e-25,    1.3648663893019599e-26,
    6.8079260225488550e-28,    3.2811847921878160e-29,    1.5297856265310296e-30,    6.9067845249101717e-32,
    3.0227315508859418e-33,    1.2835373766795435e-34,    5.2928057094716027e-36,    2.1212575443137310e-37,
    8.2693710058205414e-39,    3.1379533491686621e-40,    1.1599034324064020e-41,    4.1791633689097701e-43,
    1.4686740068516504e-44,    5.0372441735018645e-46,    1.6871072002308662e-47,    5.5209447431377376e-49,
    1.7661733782944279e-50,    5.5261232007503469e-52,    1.6919352478270368e-53,    5.0713394200164572e-55,
    1.4887744882172243e-56,    4.2823899438013759e-58,    1.2074551072719491e-59,    3.3385110130638032e-61,
    9.0551307696154188e-63,    2.4102032054907331e-64,    6.2976823344134291e-66,    1.6159267150724950e-67,
    4.0730230784503139e-69,    1.0087927395626504e-70,    2.4558837238311612e-72,    5.8784241091095162e-74
},
// x = 2.8600
{
    1.7425179879329658e-01,    1.1447134848177454e-01,    5.4176957728498100e-02,    1.9756387417966662e-02,
    5.8221633488593763e-03,    1.4348943621154938e-03,    3.0333887918439846e-04,    5.6081274913682890e-05,
    9.2063184482992206e-06,    1.3584030181980356e-06,    1.8196273299758157e-07,    2.2313020663345216e-08,
    2.5223570336165839e-09,    2.6444519466879554e-10,    2.5846454575505552e-11,    2.3657601619210502e-12,
    2.0359967356410450e-13,    1.6533159258304024e-14,    1.2708015219224647e-15,    9.2719988677744897e-17,
    6.4380399532155028e-18,    4.2640892885133947e-19,    2.6996617338717121e-20,    1.6369777176785636e-21,
    9.5235264978493659e-23,    5.3245764386363684e-24,    2.8652428952345775e-25,    1.4860583831034440e-26,
    7.4383123433386470e-28,    3.5975314590286014e-29,    1.6831361426497664e-30,    7.6257002301458414e-32,
    3.3490290162416011e-33,    1.4270647778568771e-34,    5.9052359475174350e-36,    2.3749863229506640e-37,
    9.2908800525036324e-39,    3.5379179409934249e-40,    1.3123214234162878e-41,    4.7448779495738190e-43,
    1.6733184640782208e-44,    5.7592226389955236e-46,    1.9356702438571025e-47,    6.3565296300073019e-49,
    2.0406011625745522e-50,    6.4071310864910618e-52,    1.9685441455783193e-53,    5.9211026175065998e-55,
    1.7443250778752907e-56,    5.0350456917992454e-58,    1.4246460986290624e-59,    3.9528260112975881e-61,
    1.0758917631482740e-62,    2.8737340049739196e-64,    7.5351653215701960e-66,    1.9402285712921886e-67,
    4.9075798417741362e-69,    1.2197526891199205e-70,    2.9798680401907280e-72,    7.1576358734997611e-74
},
// x = 2.8700
{
    1.7365596371510333e-01,    1.1426878760367599e-01,    5.4211237996975137e-02,    1.9824122103719338e-02,
    5.8597206708303727e-03,    1.4486914286415326e-03,    3.0724481192554191e-04,    5.6990189954408948e-05,
    9.3866762753209483e-06,    1.3896684629608453e-06,    1.8678052749653806e-07,    2.2981676400809952e-08,
    2.6068141799426764e-09,    2.7423580897480421e-10,    2.6895419552531350e-11,    2.4702455520134632e-12,
    2.1332473984239172e-13,    1.7382689714184637e-14,    1.3407189377010442e-15,    9.8159855321006352e-17,
    6.8393706211664240e-18,    4.5456073291462313e-19,    2.8878803978685105e-20,    1.7571862105017030e-21,
    1.0258380670598785e-22,    5.7553643019110529e-24,    3.1078252568684108e-25,    1.6174803164511569e-26,
    8.1243019968565975e-28,    3.9429860650030832e-29,    1.8511828383366317e-30,    8.4162692353273328e-32,
    3.7091037525833348e-33,    1.5860039581103533e-34,    6.5858015940082030e-36,    2.6579302825977462e-37,
    1.0433995944097127e-38,    3.9870633674628688e-40,    1.4840802643644574e-41,    5.3846094118858630e-43,
    1.9055482007531974e-44,    6.5813888541896372e-46,    2.2197167889483036e-47,    7.3147405796755938e-49,
    2.3564041685942452e-50,    7.4245167875161796e-52,    2.2890896256999603e-53,    6.9092896034181790e-55,
    2.0425444291343406e-56,    5.9164457167494535e-58,    1.6798787573515212e-59,    4.6772675332957493e-61,
    1.2775171791797124e-62,    3.4241948333858473e-64,    9.0098694178348717e-66,    2.3280510025705727e-67,
    5.9090943040879010e-69,    1.4738027032091339e-70,    3.6130891384131317e-72,    8.7089416963487719e-74
},
// x = 2.8800
{
    1.7306404312527007e-01,    1.1406649745623337e-01,    5.4244774941693621e-02,    1.9891540960237514e-02,
    5.8972795522274062e-03,    1.4625423595268803e-03,    3.1118026236779268e-04,    5.7909230783371679e-05,
    9.5696853710650238e-06,    1.4215046347240041e-06,    1.9170340587194013e-07,    2.3667300241106690e-08,
    2.6937164464352773e-09,    2.8434497691158426e-10,    2.7982287889173950e-11,    2.5788835830965158e-12,
    2.2347154334341545e-13,    1.8272148953211816e-14,    1.4141775925775201e-15,    1.0389515968116874e-16,
    7.2639718950268392e-18,    4.8444867557833679e-19,    3.0884030489168018e-20,    1.8856991850867046e-21,
    1.1046739921137133e-22,    6.2191290599003133e-24,    3.3698877563667723e-25,    1.7599508253123860e-26,
    8.8705552493692591e-28,    4.3200988747199400e-29,    2.0352693519379168e-30,    9.2853167958777527e-32,
    4.1063028396586109e-33,    1.7619414703804987e-34,    7.3417800928676026e-36,    2.9733231309693212e-37,
    1.1712651936290871e-38,    4.4912165622577232e-40,    1.6775472078058888e-41,    6.1077080544765357e-43,
    2.1689567515607593e-44,    7.5171907119007014e-46,    2.5441505615989483e-47,    8.4130127371350737e-49,
    2.7196305893947927e-50,    8.5987629636648407e-52,    2.6603474034713331e-53,    8.0578066219970148e-55,
    2.3903580209283053e-56,    6.9480098426576144e-58,    1.9796375147490517e-59,    5.5310582946187729e-61,
    1.5159718819211401e-62,    4.0774751146149070e-64,    1.0766136440735148e-65,    2.7915311422310997e-67,
    7.1101633861209167e-69,    1.7795358711551356e-70,    4.3777894945083957e-72,    1.0588890111001581e-73
},
// x = 2.8900
{
    1.7247600079442271e-01,    1.1386448335975630e-01,    5.4277575161457675e-02,    1.9958643980071675e-02,
    5.9348388775470164e-03,    1.4764467835584950e-03,    3.1514523770499118e-04,    5.8838447861303132e-05,
    9.7553699127603848e-06,    1.4539189627126721e-06,    1.9673313368053467e-07,    2.4370240466572028e-08,
    2.7831230469164265e-09,    2.9478158321194759e-10,    2.9108255662935712e-11,    2.6918205042883503e-12,
    2.3405648198800722e-13,    1.9203235913179541e-14,    1.4913411709541768e-15,    1.0994064490807606e-16,
    7.7130908797941710e-18,    5.1617221895460360e-19,    3.3019801922902999e-20,    2.0230540651107662e-21,
    1.1892266331613898e-22,    6.7182511001051555e-24,    3.6529096134212023e-25,    1.9143504557619935e-26,
    9.6821024554712897e-28,    4.7316323645427441e-29,    2.2368562460153729e-30,    1.0240288178114616e-31,
    4.5442902326018483e-33,    1.9566201674474236e-34,    8.1811938829063416e-36,    3.3247421168449067e-37,
    1.3142315571107888e-38,    5.0568687795901138e-40,    1.8953686358463923e-41,    6.9246633211790769e-43,
    2.4675895725264753e-44,    8.5818203263266396e-46,    2.9145307094831655e-47,    9.6711807879121712e-49,
    3.1371850943802344e-50,    9.9533346304845920e-52,    3.0901069266095307e-53,    9.3919254781417888e-55,
    2.7957832757246193e-56,    8.1546219034486182e-58,    2.3314818096950486e-59,    6.5366861752595588e-61,
    1.7978095160372485e-62,    4.8522934633277278e-64,    1.2856386114114839e-65,    3.3450673466997312e-67,
    8.5495921885391134e-69,    2.1472176169639103e-70,    5.3006331463874269e-72,    1.2865507793828002e-73
},
// x = 2.9000
{
    1.7189180090596967e-01,    1.1366275050576488e-01,    5.4309645210350824e-02,    2.0025431177573823e-02,
    5.9723975403450178e-03,    1.4904043282272198e-03,    3.1913974362108227e-04,    5.9777891305126365e-05,
    9.9437541118080801e-06,    1.4869189255617275e-06,    2.0187149605883189e-07,    2.5090850652943998e-08,
    2.8750943285864657e-09,    3.0555471685378031e-10,    3.0274550982303401e-11,    2.8092070307461638e-12,
    2.4509651570649425e-13,    2.0177714086054994e-14,    1.5723801851408737e-15,    1.1631172391282123e-16,
    8.1880359684503424e-18,    5.4983608300603977e-19,    3.5294048015955106e-20,    2.1698206894953230e-21,
    1.2798856551364989e-22,    7.2552721957213036e-24,    3.9584758889592779e-25,    2.0816260726759171e-26,
    1.0564371815298040e-27,    5.1805779449228404e-29,    2.4575306661918122e-30,    1.1289302243509493e-31,
    5.0270753604378262e-33,    2.1719539079888684e-34,    9.1128833600940232e-36,    3.7161430009800500e-37,
    1.4740150798045572e-38,    5.6912483685754496e-40,    2.1405017247002942e-41,    7.8472374746686042e-43,
    2.8059988422949986e-44,    9.7924323963631685e-46,    3.3371564393356470e-47,    1.1111798310403166e-48,
    3.6169462146976241e-50,    1.1515099779422534e-51,    3.5873183960714480e-53,    1.0940785124439039e-54,
    3.2680966862457066e-56,    9.5651738930519872e-58,    2.7442193072691040e-59,    7.7204435975197658e-61,
    2.1307191844990043e-62,    5.7706881268159304e-64,    1.5342549496773750e-65,    4.0057297526959269e-67,
    1.0273546799683895e-68,    2.5891031639257979e-70,    6.4135634264188983e-72,    1.5620574050704072e-73
},
// x = 2.9100
{
    1.7131140805072492e-01,    1.1346130395933521e-01,    5.4340991597802035e-02,    2.0091902588541426e-02,
    6.0099544432349761e-03,    1.5044146197734747e-03,    3.2316378429744189e-04,    6.0727610884559568e-05,
    1.0134862212083190e-05,    1.5205120510839186e-06,    2.0712029778956130e-07,    2.5829489715950168e-08,
    2.9696917872060571e-09,    3.1667367466102768e-10,    3.1482434681058111e-11,    2.9311984580431397e-12,
    2.5660918300404844e-13,    2.1197413667332073e-14,    1.6574722285768380e-15,    1.2302450673309830e-16,
    8.6901795765527449e-18,    5.8555049988085368e-19,    3.7715145323636732e-20,    2.3266031235847115e-21,
    1.3770655783556422e-22,    7.8329057350038977e-24,    4.2882845920724491e-25,    2.2627955627958818e-26,
    1.1523219090544066e-27,    5.6701739264251088e-29,    2.6990167551581593e-30,    1.2441209426903324e-31,
    5.5590441790886308e-33,    2.4100435811882567e-34,    1.0146586662060444e-35,    4.1518984316572377e-37,
    1.6525196504990719e-38,    6.4024011956438334e-40,    2.4162495599404732e-41,    8.8886143580474682e-43,
    3.1893046629812450e-44,    1.1168388924417425e-45,    3.8191622023920827e-47,    1.2760498186256251e-48,
    4.1678992845451684e-50,    1.3314807406760267e-51,    4.1622604380076451e-53,    1.2737965704904656e-54,
    3.8180257535460356e-56,    1.1213193084544520e-57,    3.2281062267666315e-59,    9.1130534988209263e-61,
    2.5237168746889074e-62,    6.8585901496107606e-64,    1.8297750038491803e-65,    4.7937434402966354e-67,
    1.2336915710828210e-68,    3.1198137550965555e-70,    7.7548229895849037e-72,    1.8952335108734456e-73
},
// x = 2.9200
{
    1.7073478722160743e-01,    1.1326014866140373e-01,    5.4371620788658398e-02,    2.0158058269865334e-02,
    6.0475084978853155e-03,    1.5184772832325260e-03,    3.2721736242031851e-04,    6.1687656018778655e-05,
    1.0328718488236383e-05,    1.5547059160326203e-06,    2.1248136336659073e-07,    2.6586521957823042e-08,
    3.0669780823681095e-09,    3.2814796494539918e-10,    3.2733201023665042e-11,    3.0579547788628085e-12,
    2.6861261792974738e-13,    2.2264233766347449e-14,    1.7468022372265495e-15,    1.3009582888774233e-16,
    9.2209609861561392e-18,    6.2343147938556523e-19,    4.0291940409795868e-20,    2.4940415633689709e-21,
    1.4812072543227155e-22,    8.4540475534570441e-24,    4.6441542326150727e-25,    2.4589528505022528e-26,
    1.2564959408776615e-27,    6.2039248164074436e-29,    2.9631868775274169e-30,    1.3705654449483131e-31,
    6.1449928786573209e-33,    2.6731945622660592e-34,    1.1293026882458729e-35,    4.6368400412228725e-37,
    1.8518563046952660e-38,    7.1992794847012363e-40,    2.7263000631010461e-41,    1.0063564879984249e-42,
    3.6232633792792412e-44,    1.2731533353477461e-45,    4.3686246948941267e-47,    1.4646399162363883e-48,
    4.8002869295466602e-50,    1.5387630526454816e-51,    4.8267312336869543e-53,    1.4822145191569926e-54,
    4.4579693063269404e-56,    1.3137564387975575e-57,    3.7950788968630134e-59,    1.0750395401454045e-60,
    2.9873682679225675e-62,    8.1464928283736813e-64,    2.1808273883746297e-65,    5.7330567012660148e-67,
    1.4804916604952381e-68,    3.7567822262192159e-70,    9.3701660753346825e-72,    2.2978741159369921e-73
},
// x = 2.9300
{
    1.7016190380841645e-01,    1.1305928943103184e-01,    5.4401539203264362e-02,    2.0223898299181413e-02,
    6.0850586250152357e-03,    1.5325919424793318e-03,    3.3130047918839564e-04,    6.2658075773139413e-05,
    1.0525347243995925e-05,    1.5895081458593264e-06,    2.1795653705831288e-07,    2.7362317113738279e-08,
    3.1670170528587895e-09,    3.3998731118889816e-10,    3.4028178421844250e-11,    3.1896408020437062e-12,
    2.8112556745692970e-13,    2.3380144678966964e-14,    1.8405627593722095e-15,    1.3754328075136000e-16,
    9.7818893029125600e-18,    6.6360108603067622e-19,    4.3033774134714799e-20,    2.6728143371042385e-21,
    1.5927794225772590e-22,    9.1217874016554420e-24,    5.0280318454248427e-25,    2.6712732456917831e-26,
    1.3696401293513383e-27,    6.7856220390071880e-29,    3.2520737150792754e-30,    1.5093144132244576e-31,
    6.7901644601337404e-33,    2.9639357203858678e-34,    1.2564007374245387e-35,    5.1763046079454450e-37,
    2.0743648507285038e-38,    8.0918399187922896e-40,    3.0747691263826443e-41,    1.1388631029027792e-42,
    4.1143438120118673e-44,    1.4506497527408442e-45,    4.9946830872946256e-47,    1.6802564267022246e-48,
    5.5257793392182686e-50,    1.7773782707363075e-51,    5.5942662829802717e-53,    1.7237850121941902e-54,
    5.2022502687323237e-56,    1.5383361974835532e-57,    4.4590212622964094e-59,    1.2674347124355631e-60,
    3.5340479324880350e-62,    9.6702331868984201e-64,    2.5975877845161739e-65,    6.8520090054916942e-67,
    1.7754991592523248e-68,    4.5207803888703031e-70,    1.1314298322406554e-71,    2.7841304349206136e-73
},
// x = 2.9400
{
    1.6959272359268082e-01,    1.1285873096763183e-01,    5.4430753217546257e-02,    2.0289422774525920e-02,
    6.1226037543893087e-03,    1.5467582202729469e-03,    3.3541313432045749e-04,    6.3638918855957619e-05,
    1.0724772810469758e-05,    1.6249264144658332e-06,    2.2354768296947309e-07,    2.8157250398167748e-08,
    3.2698737321063036e-09,    3.5220165576722322e-10,    3.5368730162415718e-11,    3.3264262740068314e-12,
    2.9416740928246428e-13,    2.4547190224068510e-14,    1.9389542340296783e-15,    1.4538523798072161e-16,
    1.0374546530309986e-17,    7.0618772809932380e-19,    4.5950507088576561e-20,    2.8636400088665978e-21,
    1.7122803526700762e-22,    9.8394210831378716e-24,    5.4420015135063686e-25,    2.9010191442717797e-26,
    1.4924883065961914e-27,    7.4193661771227636e-29,    3.5678832960452792e-30,    1.6615120702283272e-31,
    7.5002884131498846e-33,    3.2850401101556169e-34,    1.3972515856468678e-35,    5.7761846578670943e-37,
    2.3226376585563528e-38,    9.0911519278452552e-40,    3.4662483917481448e-41,    1.2882330409637188e-42,
    4.6698122878115767e-44,    1.6521045258353865e-45,    5.7076740586144036e-47,    1.9266517461872886e-48,
    6.3576668356959488e-50,    2.0519217759287937e-51,    6.4803863925404542e-53,    2.0036313496710164e-54,
    6.0674055082578938e-56,    1.8002806199419223e-57,    5.2360737595860823e-59,    1.4933760024856067e-60,
    4.1782406739242507e-62,    1.1471903698018921e-63,    3.0920491350494591e-65,    8.1841156926982131e-67,
    2.1279041327894960e-68,    5.4365428610279103e-70,    1.3652585778988396e-71,    3.3709693896742174e-73
},
// x = 2.9500
{
    1.6902721274258162e-01,    1.1265847785315340e-01,    5.4459269163103577e-02,    2.0354631813994765e-02,
    6.1601428248108953e-03,    1.5609757383005247e-03,    3.3955532606317452e-04,    6.4630233615349446e-05,
    1.0927019544448514e-05,    1.6609684439512547e-06,    2.2925668510144421e-07,    2.8971702551143332e-08,
    3.3756143637166450e-09,    3.6480116371414349e-10,    3.6756255146519142e-11,    3.4684860025992269e-12,
    3.0775817005270132e-13,    2.5767490145277717e-14,    2.0421852782201143e-15,    1.5364089302544095e-16,
    1.1000590765131655e-17,    7.5132645920445509e-19,    4.9052546219250766e-20,    3.0672795887660943e-21,
    1.8402395755365993e-22,    1.0610463298523470e-23,    5.8882934189824486e-25,    3.1495461029576700e-26,
    1.6258311773232901e-27,    8.1095908414841590e-29,    3.9130090264575062e-30,    1.8284041012705681e-31,
    8.2816237440880434e-33,    3.6395474884578029e-34,    1.5532838099135701e-35,    6.4429839142000099e-37,
    2.5995458179757353e-38,    1.0209517175298827e-39,    3.9058581540016484e-41,    1.4565383497916585e-42,
    5.2978274409352582e-44,    1.8806456704516796e-45,    6.5192833932474728e-47,    2.2080825668140280e-48,
    7.3110775586468939e-50,    2.3676423408969351e-51,    7.5028799478753172e-53,    2.3276454650555023e-54,
    7.0725180355633626e-56,    2.1056364126350967e-57,    6.1449897710021450e-59,    1.7585588343615092e-60,
    4.9368917060332709e-62,    1.3600915361073388e-63,    3.6783377795313315e-65,    9.7689892457616966e-67,
    2.5486057023562173e-68,    6.5335045327903365e-70,    1.6463082176110156e-71,    4.0787205703836526e-73
},
// x = 2.9600
{
    1.6846533780794698e-01,    1.1245853455423306e-01,    5.4487093327305364e-02,    2.0419525555406461e-02,
    6.1976747841143744e-03,    1.5752441172208690e-03,    3.4372705119898683e-04,    6.5632068036129435e-05,
    1.1132111826709171e-05,    1.6976420043538317e-06,    2.3508544741092588e-07,    2.9806059884423481e-08,
    3.4843064170949575e-09,    3.7779622652688087e-10,    3.8192188640301483e-11,    3.6159999833865034e-12,
    3.2191854402390702e-13,    2.7043242579431320e-14,    2.1504729833340057e-15,    1.6233028775626614e-16,
    1.1661759518336916e-17,    7.9915929281559291e-19,    5.2350872704980374e-20,    3.2845388547437652e-21,
    1.9772197087325594e-22,    1.1438661233784352e-23,    6.3692934521485446e-25,    3.4183093112964075e-26,
    1.7705204807249897e-27,    8.8610882786934079e-29,    4.2900467962353447e-30,    2.0113462127322490e-31,
    9.1410056227859925e-33,    4.0307888096494270e-34,    1.7260682025465941e-35,    7.1838780374336990e-37,
    2.9082678918903957e-38,    1.1460601352650147e-39,    4.3993059148524413e-41,    1.6460966040754839e-42,
    6.0075458671958726e-44,    2.1397958225601003e-45,    7.4427160982938848e-47,    2.5293754148960955e-48,
    8.4032234303657052e-50,    2.7305321172173965e-51,    8.6821240554710451e-53,    2.7025997889931863e-54,
    8.2395975537198223e-56,    2.4614014566234413e-57,    7.2075467698019443e-59,    2.0696196302097159e-60,
    5.8298133289492072e-62,    1.6115236567873723e-63,    4.3730831299865498e-65,    1.1653420288128332e-66,
    3.0505219384264705e-68,    7.8466718380845175e-70,    1.9838932247153195e-71,    4.9317268786936965e-73
},
// x = 2.9700
{
    1.6790706571531763e-01,    1.1225890542430604e-01,    5.4514231953392341e-02,    2.0484104155968758e-02,
    6.2351985891561995e-03,    1.5895629767075669e-03,    3.4792830505409804e-04,    6.6644469736768045e-05,
    1.1340074060319992e-05,    1.7349549133876414e-06,    2.4103589386706879e-07,    3.0660714327558004e-08,
    3.5960186031515976e-09,    3.9119746601258682e-10,    3.9678003037170740e-11,    3.7691535284279588e-12,
    3.3666991216512172e-13,    2.8376726593272781e-14,    2.2640432208297825e-15,    1.7147434724518852e-16,
    1.2359873165691697e-17,    8.4983553025266626e-19,    5.5857071124464785e-20,    3.5162707910795982e-21,
    2.1238183801993780e-22,    1.2328008932475978e-23,    6.8875534105745248e-25,    3.7088704851404400e-26,
    1.9274734388526795e-27,    9.6790368373164085e-29,    4.7018112376362546e-30,    2.2118133753727957e-31,
    1.0085895936384825e-32,    4.4624128643335832e-34,    1.9173313140711889e-35,    8.0067811378936906e-37,
    3.2523215107453856e-38,    1.2859579495930992e-39,    4.9529511668549616e-41,    1.8594989269974920e-42,
    6.8092398245954684e-44,    2.4335202896383371e-45,    8.4928872223385074e-47,    2.8960004168775961e-48,
    9.6536779495924424e-50,    3.1474286158878185e-51,    1.0041449734145123e-52,    3.1362748928806163e-54,
    9.5940161768236829e-56,    2.8756701469567971e-57,    8.4490203010245830e-59,    2.4342870124101918e-60,
    6.8801569757709486e-62,    1.9082836022806636e-63,    5.1958497092940610e-65,    1.3892645262790527e-66,
    3.6489545137665253e-68,    9.4176514902732482e-70,    2.3891219098789817e-71,    5.9591180227181360e-73
},
// x = 2.9800
{
    1.6735236376308268e-01,    1.1205959470568148e-01,    5.4540691240584541e-02,    2.0548367791949074e-02,
    6.2727132058047804e-03,    1.6039319354916972e-03,    3.5215908150656663e-04,    6.7667485966407158e-05,
    1.1550930668946711e-05,    1.7729150361742280e-06,    2.4710996850700675e-07,    3.1536063473843799e-08,
    3.7108208900916093e-09,    4.0501573817594449e-10,    4.1215208631709078e-11,    3.9281373975675150e-12,
    3.5203436171143259e-13,    2.9770304789905090e-14,    2.3831309575139068e-15,    1.8109491473241764e-16,
    1.3096838532601837e-17,    9.0351210266085286e-19,    5.9583359978788249e-20,    3.7633781489503971e-21,
    2.2806702554369478e-22,    1.3282762493672350e-23,    7.4458018218811468e-25,    4.0229052071653373e-26,
    2.0976775099482246e-27,    1.0569030418734251e-28,    5.1513532189606486e-30,    2.4314098042987484e-31,
    1.1124438060607912e-32,    4.9384152399762281e-34,    2.1289702272098687e-35,    8.9204185842498772e-37,
    3.6355980765109574e-38,    1.4423296152553161e-39,    5.5738770428831353e-41,    2.0996410875257310e-42,
    7.7144283014983463e-44,    2.7662807426228601e-45,    9.6866358038035705e-47,    3.3141542896269029e-48,
    1.1084689792149809e-49,    3.6261302264550424e-51,    1.1607557001864363e-52,    3.6376050678409486e-54,
    1.1165007069389527e-55,    3.3578002879826575e-57,    9.8987301182827551e-59,    2.8615566183672815e-60,
    8.1149608387995692e-62,    2.2583362399153612e-63,    6.1696417880359054e-65,    1.6551832176644659e-66,
    4.3620175407477096e-68,    1.1295864413402386e-69,    2.8752335622274715e-71,    7.1957294891594255e-73
},
// x = 2.9900
{
    1.6680119961668405e-01,    1.1186060653158210e-01,    5.4566477344193655e-02,    2.0612316658348600e-02,
    6.3102176089293426e-03,    1.6183506114041072e-03,    3.5641937299449967e-04,    6.8701163601934942e-05,
    1.1764706095160446e-05,    1.8115302849692231e-06,    2.5330963548979705e-07,    3.2432510626166359e-08,
    3.8287845192866026e-09,    4.1926213714796177e-10,    4.2805354405341550e-11,    4.0931479322743768e-12,
    3.6803470617577532e-13,    3.1226425986554601e-14,    2.5079805806548071e-15,    1.9121478781617558e-16,
    1.3874652617735296e-17,    9.6035392749766739e-19,    6.3542623621679628e-20,    4.0268161345966153e-21,
    2.4484491731810414e-22,    1.4307456139389836e-23,    8.0469554255859957e-25,    4.3622107414653822e-26,
    2.2821954663051452e-27,    1.1537110047547463e-28,    5.6419786620010946e-30,    2.6718797311109433e-31,
    1.2265516182050156e-32,    5.4631697956924799e-34,    2.3630686685735173e-35,    9.9344066766720573e-37,
    4.0624008687825853e-38,    1.6170441851691425e-39,    6.2699695288860046e-41,    2.3697579978305794e-42,
    8.7360229122314666e-44,    3.1430951859355605e-45,    1.1042964651152670e-46,    3.7908536680787052e-48,
    1.2721536676166800e-49,    4.1755270123341126e-51,    1.3410986454282793e-52,    4.2168442796017000e-54,
    1.2986234812494426e-55,    3.9186046453168221e-57,    1.1590669137723448e-58,    3.3618931092364372e-60,
    9.5657848353955799e-62,    2.6710098500271366e-63,    7.3214924891045411e-65,    1.9707820927071404e-66,
    5.2111415964287247e-68,    1.3539977357481407e-69,    3.4579973974159367e-71,    8.6831936799183499e-73
},
// x = 3.0000
{
    1.6625354130388895e-01,    1.1166194492814806e-01,    5.4591596375740875e-02,    2.0675950968579968e-02,
    6.3477107823875769e-03,    1.6328186214172469e-03,    3.6070917052433751e-04,    6.9745549145118041e-05,
    1.1981424798747193e-05,    1.8508086188839645e-06,    2.5963687914874901e-07,    3.3350464842720166e-08,
    3.9499820212277051e-09,    4.3394799915598328e-10,    4.4450028823853404e-11,    4.2643871920670311e-12,
    3.8469450582743710e-13,    3.2747627965221360e-14,    2.6388462331880435e-15,    2.0185775590214562e-16,
    1.4695406460151068e-17,    1.0205342800810036e-18,    6.7748445656674658e-20,    4.3075952308840630e-21,
    2.6278703949101968e-22,    1.5406919197405500e-23,    8.6941313512628694e-25,    4.7287143507757178e-26,
    2.4821708174056994e-27,    1.2589797704888252e-28,    6.1772687776777152e-30,    2.9351190276893915e-31,
    1.3518819529988182e-32,    6.0414628586212763e-34,    2.6219145733995855e-35,    1.1059339802230000e-36,
    4.5374868718203016e-38,    1.8121747467259011e-39,    7.0500050055508774e-41,    2.6734619678422032e-42,
    9.8884902330736039e-44,    3.5696049123298173e-45,    1.2583308961119467e-46,    4.3340400126325582e-48,
    1.4592924485047094e-49,    4.8057487352548071e-51,    1.5486654774173391e-52,    4.8857552610560201e-54,
    1.5096447495985275e-55,    4.5705706874507116e-57,    1.3564227397929790e-58,    3.9474634810134008e-60,
    1.1269446450452708e-61,    3.1572233549541585e-63,    8.6831511162013131e-65,    2.3451160676827327e-66,
    6.2216657752009740e-68,    1.6219590236977595e-69,    4.1561843595651911e-71,    1.0471234673353481e-72
},
// x = 3.0100
{
    1.6570935721012880e-01,    1.1146361381640646e-01,    5.4616054403079178e-02,    2.0739270954148726e-02,
    6.3851917190123368e-03,    1.6473355816866025e-03,    3.6502846367923828e-04,    7.0800688719793178e-05,
    1.2201111255019695e-05,    1.8907580436022325e-06,    2.6609370404214684e-07,    3.4290340982603066e-08,
    4.0744872315585993e-09,    4.4908490653497657e-10,    4.6150860646851002e-11,    4.4420630935547903e-12,
    4.0203808864555950e-13,    3.4336540297820425e-14,    2.7759921592756470e-15,    2.1304863895035325e-16,
    1.5561289154791724e-17,    1.0842351807650856e-18,    7.2215143861921725e-20,    4.6067841582827083e-21,
    2.8196929737444277e-22,    1.6586294047595710e-23,    9.3906600321989668e-25,    5.1244821464635264e-26,
    2.6988336003154816e-27,    1.3734132577066959e-28,    6.7611018206292810e-30,    3.2231877452803752e-31,
    1.4894911902913397e-32,    6.6785303655227473e-34,    2.9080192281357910e-35,    1.2306885743031672e-36,
    5.0661126698508984e-38,    2.0300198210540746e-39,    7.9237469586908442e-41,    3.0147851099057792e-42,
    1.1188032360057749e-43,    4.0521492257890341e-45,    1.4331837117885085e-46,    4.9526974826144433e-48,
    1.6731437238694180e-49,    5.5283323028758624e-51,    1.7874462557858423e-52,    5.6578248664502560e-54,
    1.7540221885843588e-55,    5.3281125405330264e-57,    1.5865024954231863e-58,    4.6324063887784027e-60,
    1.3268873030026759e-61,    3.7297503969731021e-63,    1.0291884640646718e-64,    2.7888493634945665e-66,
    7.4235327431710194e-68,    1.9417223705611740e-69,    4.9921247479342002e-71,    1.2619203674821251e-72
},
// x = 3.0200
{
    1.6516861607390448e-01,    1.1126561701420661e-01,    5.4639857450520427e-02,    2.0802276864338307e-02,
    6.4226594205972071e-03,    1.6619011075916692e-03,    3.6937724062754718e-04,    7.1866628069115724e-05,
    1.2423789953131269e-05,    1.9313866110920709e-06,    2.7268213500234262e-07,    3.5252559751276925e-08,
    4.2023753071872184e-09,    4.6468469178009790e-10,    4.7909519749257258e-11,    4.6263895521307674e-12,
    4.2009057175598919e-13,    3.5995887247440442e-14,    2.9196930604876890e-15,    2.2481332755816384e-16,
    1.6474592021334125e-17,    1.1516477983297237e-18,    7.6957806705590070e-20,    4.9255129815281198e-21,
    3.0247222485442015e-22,    1.7851055082231927e-23,    1.0140098895763520e-24,    5.5517285031045653e-26,
    2.9335065606198973e-27,    1.4977709881579547e-28,    7.3976764702511183e-30,    3.5383236370332789e-31,
    1.6405306903545792e-32,    7.3800981906429359e-34,    3.2241381258478289e-35,    1.3689889864785969e-36,
    5.6540847888792403e-38,    2.2731269150696068e-39,    8.9020527792285904e-41,    3.3982263258925631e-42,
    1.2652787655332232e-43,    4.5978488001431559e-45,    1.6315787389129113e-46,    5.6569853220433811e-48,
    1.9174044169706385e-49,    6.3564111034187921e-51,    2.0619983908045997e-52,    6.5485092191981675e-54,
    2.0368814541829502e-55,    6.2078597363721466e-57,    1.8545869616175785e-58,    5.4331428828907572e-60,
    1.5614088435791211e-61,    4.4035260759318206e-63,    1.2191411794362251e-64,    3.3145342580310882e-66,
    8.8521042479891687e-68,    2.3230659026887890e-69,    5.9923668894726706e-71,    1.5197898798409623e-72
},
// x = 3.0300
{
    1.6463128698225590e-01,    1.1106795823812168e-01,    5.4663011498967157e-02,    2.0864968965898650e-02,
    6.4601128978811215e-03,    1.6765148137765254e-03,    3.7375548813136018e-04,    7.2943412552867502e-05,
    1.2649485394392398e-05,    1.9727024193128306e-06,    2.7940421718322977e-07,    3.6237547745890723e-08,
    4.3337227424751175e-09,    4.8075944164059796e-10,    4.9727717954936835e-11,    4.8175866263509050e-12,
    4.3887788335996001e-13,    3.7728490747375797e-14,    3.0702344628799704e-15,    2.3717882441907051e-16,
    1.7437712931539386e-17,    1.2229728701876129e-18,    8.1992331517154691e-20,    5.2649763684848561e-21,
    3.2438124692750707e-22,    1.9207028733089729e-23,    1.0946246873830907e-24,    6.0128260712229509e-26,
    3.1876117485621474e-27,    1.6328722442671667e-28,    8.0915369528141998e-30,    3.8829567369154438e-31,
    1.8062549326642240e-32,    8.1524259195835704e-34,    3.5732936803975431e-35,    1.5222488975306007e-36,
    6.3078148967995586e-38,    2.5443184336755013e-39,    9.9969916621808778e-41,    3.8288033523081759e-42,
    1.4303053852632131e-43,    5.2146986342400322e-45,    1.8565844647891681e-46,    6.4583864816405521e-48,
    2.1962670903957428e-49,    7.3049289929923063e-51,    2.3775248435970599e-52,    7.5755126448980155e-54,
    2.3641133485059620e-55,    7.2289879647403987e-57,    2.1667857689997089e-58,    6.3687347413706057e-60,
    1.8363354644874540e-61,    5.1960030477467006e-63,    1.4432991115556998e-64,    3.9369368697625978e-66,
    1.0549117416796428e-67,    2.7775691091413317e-69,    7.1884547088681464e-71,    1.8291720536327702e-72
},
// x = 3.0400
{
    1.6409733936629570e-01,    1.1087064110531755e-01,    5.4685522486048098e-02,    2.0927347542738440e-02,
    6.4975511705319391e-03,    1.6911763141899442e-03,    3.7816319155516744e-04,    7.4031087144821060e-05,
    1.2878222090589831e-05,    2.0147136119174014e-06,    2.8626201610607174e-07,    3.7245737500457790e-08,
    4.4686073855030534e-09,    4.9732150125505029e-10,    5.1607209882539741e-11,    5.0158806650330619e-12,
    4.5842678516312981e-13,    3.9537273459610426e-14,    3.2279130952468684e-15,    2.5017328719787971e-16,
    1.8453160800386620e-17,    1.2984211400338009e-18,    8.7335464382188020e-20,    5.6264370079919354e-21,
    3.4778695599690664e-22,    2.0660414620741824e-23,    1.1813159778825862e-24,    6.5103164236205991e-26,
    3.4626775564913005e-27,    1.7796005199409885e-28,    8.8476000268829549e-30,    4.2597250730265658e-31,
    1.9880303176584442e-32,    9.0023543489730383e-34,    3.9587999571480934e-35,    1.6920235709278729e-36,
    7.0343803099697449e-38,    2.8467201785583322e-39,    1.1221974710705374e-40,    4.3121103862478874e-42,
    1.6161535916450675e-43,    5.9116716672798696e-45,    2.1116561706305615e-46,    7.3698743983618781e-48,
    2.5144842572609322e-49,    8.3908820387540556e-51,    2.7399627654753662e-52,    8.7591048984929706e-54,
    2.7424846963133587e-55,    8.4135977562664118e-57,    2.5301638450211770e-58,    7.4612974789455903e-60,
    2.1584491942174099e-61,    6.1275647078317007e-63,    1.7076687634869962e-64,    4.6734176974086825e-66,
    1.2563805515892976e-67,    3.3189365668823612e-69,    8.6178451208841129e-71,    2.2001223315699724e-72
},
// x = 3.0500
{
    1.6356674299680560e-01,    1.1067366913538929e-01,    5.4707396306258781e-02,    2.0989412895620781e-02,
    6.5349732671290909e-03,    1.7058852221251140e-03,    3.8260033487458480e-04,    7.5129696430162771e-05,
    1.3110024562308803e-05,    2.0574283779497678e-06,    2.9325761770369491e-07,    3.8277567530883225e-08,
    4.6071084544117464e-09,    5.1438347832795068e-10,    5.3549793803657265e-11,    5.2215044571112290e-12,
    4.7876489531364042e-13,    4.1425261914461680e-14,    3.3930372788344100e-15,    2.6382607286393196e-16,
    1.9523560246428484e-17,    1.3782138135819306e-18,    9.3004841830755031e-20,    6.0112291937426798e-21,
    3.7278540258916046e-22,    2.2217807884038917e-23,    1.2745166593292896e-24,    7.0469213726678092e-26,
    3.7603462252584193e-27,    1.9389082840598563e-28,    9.6711839623054623e-30,    4.6714915987635395e-31,
    2.1873446826674390e-32,    9.9373570132599222e-34,    4.3842895906697011e-35,    1.8800234367811795e-36,
    7.8415902938440812e-38,    3.1837926807908403e-39,    1.2591898456776789e-40,    4.8543818668629483e-42,
    1.8253621295711965e-43,    6.6988342312463585e-45,    2.4006830353034446e-46,    8.4061000728231273e-48,
    2.8774407604540289e-49,    9.6335914982574427e-51,    3.1560839247598170e-52,    1.0122481776693890e-53,
    3.1797648059822073e-55,    9.7871478324472228e-57,    2.9528865976981129e-58,    8.7364761354348431e-60,
    2.5356404856937355e-61,    7.2220043580471635e-63,    2.0192846739105838e-64,    5.5443758749166351e-66,
    1.4954210720714789e-67,    3.9633783895145706e-69,    1.0324989758211563e-70,    2.6446134137435195e-72
},
// x = 3.0600
{
    1.6303946797989499e-01,    1.1047704575216546e-01,    5.4728638811105373e-02,    2.1051165341862556e-02,
    6.5723782251452464e-03,    1.7206411502588949e-03,    3.8706690068516348e-04,    7.6239284602972631e-05,
    1.3344917337258262e-05,    2.1008549515378645e-06,    3.0039312836302717e-07,    3.9333482379834158e-08,
    4.7493065538163451e-09,    5.3195824734766982e-10,    5.5557312513378447e-11,    5.4346973842791271e-12,
    4.9992071185790263e-13,    4.3395589733115502e-14,    3.5659273288039737e-15,    2.7816778352496584e-16,
    2.0651656426957452e-17,    1.4625830330523304e-18,    9.9019034392014414e-20,    6.4207625815308558e-21,
    3.9947840118097629e-22,    2.3886222750518593e-23,    1.3746886723330019e-24,    7.6255549979666013e-26,
    4.0823818488092402e-27,    2.1118220772919298e-28,    1.0568039651597113e-29,    5.1213624310677418e-31,
    2.4058175869406775e-32,    1.0965596063034692e-33,    4.8537430736051203e-35,    2.0881289219220337e-36,
    8.7380586879892072e-38,    3.5593656370245286e-39,    1.4123303125170460e-40,    5.4625630437269578e-42,
    2.0607685483208793e-43,    7.5874746422791928e-45,    2.7280407745623893e-46,    9.5836018281110948e-48,
    3.2912351974711781e-49,    1.1055011936749660e-50,    3.6336084377825217e-52,    1.1692174861602845e-53,
    3.6848696378482171e-55,    1.1378950776800208e-56,    3.4443865305037288e-58,    1.0223993111427333e-59,
    2.9770845004114586e-61,    8.5070806036965314e-63,    2.3863808182273607e-64,    6.5737675271793615e-66,
    1.7788720936714973e-67,    4.7300570852657667e-69,    1.2362609735789551e-70,    3.1768921697628547e-72
},
// x = 3.0700
{
    1.6251548475272024e-01,    1.1028077428548164e-01,    5.4749255809252877e-02,    2.1112605215037161e-02,
    6.6097650909270802e-03,    1.7354437106906818e-03,    3.9156287021128061e-04,    7.7359895463762022e-05,
    1.3582924948599600e-05,    2.1450016115818538e-06,    3.0767067496598291e-07,    4.0413932661447690e-08,
    4.8952836912935167e-09,    5.5005895384578305e-10,    5.7631654213338841e-11,    5.6557055764584129e-12,
    5.2192363672295271e-13,    4.5451500934814119e-14,    3.7469159677431649e-15,    2.9323031380529646e-16,
    2.1840320053731413e-17,    1.5517723710985658e-18,    1.0539759209024929e-19,    6.8565261274865564e-21,
    4.2797385185573264e-22,    2.5673117411344759e-23,    1.4823248268783382e-24,    8.2493364259439671e-26,
    4.4306789079208528e-27,    2.2994479641410273e-28,    1.1544384001614974e-29,    5.6127064911460254e-31,
    2.6452114246903225e-32,    1.2095982844043957e-33,    5.3715206155819228e-35,    2.3184066351690073e-36,
    9.7332834321041075e-38,    3.9776757435978867e-39,    1.5834547092996092e-40,    6.1443880255765196e-42,
    2.3255431089089593e-43,    8.5902463704714329e-45,    3.0986504427422971e-46,    1.0921040403180561e-47,
    3.7627714826606737e-49,    1.2680078854828995e-50,    4.1813335094097286e-52,    1.3498516877698455e-53,
    4.2680260828448386e-55,    1.3222739719648631e-56,    4.0155543587995014e-58,    1.1958278647006070e-59,
    3.4934446574032916e-61,    1.0015160773313392e-62,    2.8185898315208707e-64,    7.7897102588457308e-66,
    2.1147867916323711e-67,    5.6416121207139246e-69,    1.4793197037026821e-70,    3.8139013910031481e-72
},
// x = 3.0800
{
    1.6199476407926416e-01,    1.1008485797292283e-01,    5.4769253066677022e-02,    2.1173732864680831e-02,
    6.6471329196751252e-03,    1.7502925149808063e-03,    3.9608822331510112e-04,    7.8491572417067499e-05,
    1.3824071933278798e-05,    2.1898766814378072e-06,    3.1509240492868405e-07,    4.1519375105870138e-08,
    5.0451232939396384e-09,    5.6869901869775714e-10,    5.9774753407348377e-11,    5.8847820701261810e-12,
    5.4480400023426483e-13,    4.7596353330481040e-14,    3.9363487515258830e-15,    3.0904689981299005e-16,
    2.3092552595165288e-17,    1.6460373448807342e-18,    1.1216109196021717e-19,    7.3200922152247865e-21,
    4.5838607854018141e-22,    2.7586420267354949e-23,    1.5979507365768609e-24,    8.9216034051818376e-26,
    4.8072713658183647e-27,    2.5029773634865066e-28,    1.2606935763045298e-29,    6.1491766496120697e-31,
    2.9074434293327966e-32,    1.3338243552599048e-33,    5.9423967868994849e-35,    2.5731270259297796e-36,
    1.0837733619801486e-37,    4.4434082495541457e-39,    1.7745999133933281e-40,    6.9084660708239595e-42,
    2.6232263962728324e-43,    9.7213273791762483e-45,    3.5180440922208088e-46,    1.2439462333105044e-47,
    4.2998617658126749e-49,    1.4537100724259309e-50,    4.8092790996995458e-52,    1.5576169971724943e-53,
    4.9409590700957524e-55,    1.5357315903909719e-56,    4.6789581247726005e-58,    1.3979196051515231e-59,
    4.0971075451059530e-61,    1.1783965926755556e-62,    3.3271744473634979e-64,    9.2251877094205153e-66,
    2.5126429944169654e-67,    6.7247753582690677e-69,    1.7690781801384859e-70,    4.5757778728735457e-72
},
// x = 3.0900
{
    1.6147727704617471e-01,    1.0988929996153572e-01,    5.4788636306819655e-02,    2.1234548656002342e-02,
    6.6844807754227412e-03,    1.7651871741885367e-03,    4.0064293850561935e-04,    7.9634358469102889e-05,
    1.4068382830362556e-05,    2.2354885285969328e-06,    3.2266048623901573e-07,    4.2650272603622236e-08,
    5.1989102249990345e-09,    5.8789214246500272e-10,    6.1988591809689276e-11,    6.1221869695370889e-12,
    5.6859308617804196e-13,    4.9833622004598742e-14,    4.1345845078299425e-15,    3.2565216974181887e-16,
    2.4411491671062919e-17,    1.7456459510164479e-18,    1.1933118766249307e-19,    7.8131209801420541e-21,
    4.9083618460419991e-22,    2.9634557615903411e-23,    1.7221268659883128e-24,    9.6459267236556019e-26,
    5.2143423602934773e-27,    2.7236932822972244e-28,    1.3762953965442373e-29,    6.7347324849752317e-31,
    3.1945986366654534e-32,    1.4702990371528200e-33,    6.5715981787564820e-35,    2.8547836440909071e-36,
    1.2062944761196191e-37,    4.9617425785363763e-39,    1.9880250185564696e-40,    7.7643769555324757e-42,
    2.9577710255129100e-43,    1.0996597391837276e-44,    3.9924380614748571e-46,    1.4162594899964333e-47,
    4.9113420674223889e-49,    1.6658200916027784e-50,    5.5288526698708092e-52,    1.7964725154478773e-53,
    5.7171045779800307e-55,    1.7827288327875191e-56,    5.4490942918610246e-58,    1.6332875513905976e-59,
    4.8024538923665757e-61,    1.3857433048681471e-62,    3.9252962212107063e-64,    1.0918870300662590e-65,
    2.9835889338563968e-67,    8.0110927087698532e-69,    2.1143011426979167e-70,    5.4864403020109763e-72
},
// x = 3.1000
{
    1.6096299505866199e-01,    1.0969410330951158e-01,    5.4807411210747538e-02,    2.1295052969596169e-02,
    6.7218077310142507e-03,    1.7801272989096458e-03,    4.0522699294776591e-04,    8.0788296225466763e-05,
    1.4315882179378239e-05,    2.2818455643603041e-06,    3.3037710749250520e-07,    4.3807094249783855e-08,
    5.3567308005606951e-09,    6.0765230977826009e-10,    6.4275199266168740e-11,    6.3681876108751693e-12,
    5.9332315741705334e-13,    5.2166902887177879e-14,    4.3419957876262543e-15,    3.4308219615483063e-16,
    2.5800416646128501e-17,    1.8508792221633289e-18,    1.2693066128231400e-19,    8.3373648394168075e-21,
    5.2545242663985077e-22,    3.1826482851419717e-23,    1.8554506971394077e-24,    1.0426125516531781e-25,
    5.6542345289313425e-27,    2.9629769787058241e-28,    1.5020279136385870e-29,    7.3736647718300434e-31,
    3.5089438795780180e-32,    1.6201798521296335e-33,    7.2648443300996055e-35,    3.1661141397221232e-36,
    1.3421622994103419e-37,    5.5384024009925413e-39,    2.2262346540825977e-40,    8.7227763357683834e-42,
    3.3335878706536932e-43,    1.2433835028089186e-44,    4.5288147459446442e-46,    1.6117176305481790e-47,
    5.6072021481406290e-49,    1.9079815658699875e-50,    6.3530354255042459e-52,    2.0709382187145701e-53,
    6.6118520241079444e-55,    2.0683918162781882e-56,    6.3426753485692450e-58,    1.9072672282071570e-59,
    5.6261709777268893e-61,    1.6286712351464847e-62,    4.6283273825657008e-64,    1.2916070830627647e-65,
    3.5407303087053332e-67,    9.5377698630409366e-69,    2.5253595127286627e-70,    6.5742827424353017e-72
},
// x = 3.1100
{
    1.6045188983645256e-01,    1.0949927098783925e-01,    5.4825583417314999e-02,    2.1355246201159139e-02,
    6.7591128680822028e-03,    1.7951124993135458e-03,    4.0984036247158694e-04,    8.1953427888907201e-05,
    1.4566594518658107e-05,    2.3289562435092221e-06,    3.3824447792652474e-07,    4.4990315387992586e-08,
    5.5186728063222485e-09,    6.2799379376231309e-10,    6.6636654688015549e-11,    6.6230587293707484e-12,
    6.1902748206921213e-13,    5.4599916417687875e-14,    4.5589693299599228e-15,    3.6137454999749222e-16,
    2.7262754428670917e-17,    1.9620318060106399e-18,    1.3498347739838351e-19,    8.8946732366034104e-21,
    5.6237060727082697e-22,    3.4171707255977721e-23,    1.9985590216745268e-24,    1.1266283515779412e-25,
    6.1294610061433201e-27,    3.2223150822198083e-28,    1.6387377494809932e-29,    8.0706218230023540e-31,
    3.8529428920908231e-32,    1.7847289693565485e-33,    8.0283921908616059e-35,    3.5101231525271431e-36,
    1.4927759047232321e-37,    6.1797105722863568e-39,    2.4920046541751149e-40,    9.7955121100261677e-42,
    3.7555972874391891e-43,    1.4052936953041163e-44,    5.1350137979793219e-46,    1.8333325123400405e-47,
    6.3987313046127970e-49,    2.1843254894316205e-50,    7.2965927710735839e-52,    2.3861720366273369e-53,
    7.6428199611374610e-55,    2.3986082339816025e-56,    7.3789590738366569e-58,    2.2260267227510659e-59,
    6.5876126353810181e-61,    1.9131320275702761e-62,    5.4542125508050636e-64,    1.5269856482457215e-65,
    4.1994654264465789e-67,    1.1348662892465070e-68,    3.0145176572438982e-70,    7.8729922227550020e-72
},
// x = 3.1200
{
    1.5994393340980045e-01,    1.0930480588193019e-01,    5.4843158523329175e-02,    2.1415128761210354e-02,
    6.7963952770238563e-03,    1.8101423851800164e-03,    4.1448302158148968e-04,    8.3129795257142782e-05,
    1.4820544383687951e-05,    2.3768290639712435e-06,    3.4626482745281023e-07,    4.6200417654251220e-08,
    5.6848255144199037e-09,    6.4893116050200490e-10,    6.9075086998705911e-11,    6.8870826294179717e-12,
    6.4574036025815681e-13,    5.7136511302844547e-14,    4.7859065403497245e-15,    3.8056835638949091e-16,
    2.8802085481089468e-17,    2.0794125674819629e-18,    1.4351483951111905e-19,    9.4869976100533551e-21,
    6.0173448787943396e-22,    3.6680332459682548e-23,    2.1521303653922155e-24,    1.2170766295580096e-25,
    6.6427171329041277e-27,    3.5033072005254564e-28,    1.7873388320731615e-29,    8.8306378183132974e-31,
    4.2292726060530570e-32,    1.9653222369430604e-33,    8.8690844125103606e-35,    3.8901072532690728e-36,
    1.6596752827540552e-37,    6.8926493893894789e-39,    2.7884103046617870e-40,    1.0995752884425712e-41,
    4.2292858488668436e-43,    1.5876161406020363e-44,    5.8198338037339359e-46,    2.0844953540090378e-47,
    7.2986819779664179e-49,    2.4995333711812499e-50,    8.3763120211065199e-52,    2.7480571975757850e-53,
    8.8301695135873902e-55,    2.7801372617566166e-56,    8.5801253212011718e-58,    2.5966930344431114e-59,
    7.7092139050029392e-61,    2.2460471825069157e-62,    6.4238880721686542e-64,    1.8042342172265581e-65,
    4.9778762664793101e-67,    1.3495437901401345e-68,    3.5962707704838249e-70,    9.4225120870005085e-72
},
// x = 3.1300
{
    1.5943909811555390e-01,    1.0911071079321483e-01,    5.4860142083718595e-02,    2.1474701074814518e-02,
    6.8336540569768241e-03,    1.8252165659354805e-03,    4.1915494346554953e-04,    8.4317439720738607e-05,
    1.5077756305460202e-05,    2.4254725664818996e-06,    3.5444040668828874e-07,    4.7437889020536246e-08,
    5.8552797003227419e-09,    6.7047927354950304e-10,    7.1592676093801896e-11,    7.1605493577282764e-12,
    6.7349715144517690e-13,    5.9780668370176594e-14,    5.0232239831385033e-15,    4.0070435224539582e-16,
    3.0422150048907704e-17,    2.2033452149756327e-18,    1.5255124892293687e-19,    1.0116396594752819e-20,
    6.4369622217567827e-22,    3.9363084654339500e-23,    2.3168875522552732e-24,    1.3144239570385221e-25,
    7.1968929224078903e-27,    3.8076740441142432e-28,    1.9488174716506381e-29,    9.6591632615593508e-31,
    4.6408407297464217e-32,    2.1634589561994470e-33,    9.7944017795531475e-35,    4.3096821126508866e-36,
    1.8445549578293701e-37,    7.6849266590619687e-39,    3.1188574151621470e-40,    1.2338129749334907e-41,
    4.7607691638797610e-43,    1.7928398719548383e-44,    6.5931455969640191e-46,    2.3690229390507230e-47,
    8.3214532767050149e-49,    2.8589082943444352e-50,    9.6112707834969858e-52,    3.1633011648194174e-53,
    1.0196960562651162e-54,    3.2207348604301367e-56,    9.9717069775695649e-58,    3.0274970610431891e-59,
    9.0169683859102410e-61,    2.6354619359200157e-62,    7.5617679088990687e-64,    2.1306194024506485e-65,
    5.8971845533227710e-67,    1.6038927845991118e-68,    4.2877408979177844e-70,    1.1270176458194242e-71
},
// x = 3.1400
{
    1.5893735659327668e-01,    1.0891698844071165e-01,    5.4876539611705050e-02,    2.1533963581308654e-02,
    6.8708883157940085e-03,    1.8403346506888970e-03,    4.2385610000488232e-04,    8.5516402261039825e-05,
    1.5338254808832038e-05,    2.4748953342422193e-06,    3.6277348698421910e-07,    4.8703223838204625e-08,
    6.0301276597903172e-09,    6.9265329847281397e-10,    7.4191653813890685e-11,    7.4437568795558640e-12,
    7.0233430235196070e-13,    6.2536504519326233e-14,    5.2713538881337587e-15,    4.2182494577558789e-16,
    3.2126854615308538e-17,    2.3341689514954959e-18,    1.6212056616639552e-19,    1.0785041467535021e-20,
    6.8841681157196435e-22,    4.2231350647684091e-23,    2.4936004153116260e-24,    1.4191688605474166e-25,
    7.7950863272967176e-27,    4.1372661018352671e-28,    2.1242377988413025e-29,    1.0562097716811573e-30,
    5.0908047039477687e-32,    2.3807724563013510e-33,    1.0812520120046040e-34,    4.7728120874128631e-36,
    2.0492788634792034e-37,    8.5650481140711902e-39,    3.4871164876793991e-40,    1.3838892691769047e-41,
    5.3568614062952422e-43,    2.0237471708070156e-44,    7.4660184932633571e-46,    2.6912092548317576e-47,
    9.4832967549854089e-49,    3.2684548518179656e-50,    1.1023139844169774e-51,    3.6395476530336236e-53,
    1.1769556329186895e-54,    3.7292965650611655e-56,    1.1583082656471242e-57,    3.5279398904792407e-59,
    1.0540977510679069e-60,    3.0907228954738113e-62,    8.8963063574843460e-64,    2.5146376209667050e-65,
    6.9822833663809460e-67,    1.9050719187824484e-68,    5.1091415407708909e-70,    1.3472045461616520e-71
},
// x = 3.1500
{
    1.5843868178142398e-01,    1.0872364146256840e-01,    5.4892356578977890e-02,    2.1592916734032075e-02,
    6.9080971700177128e-03,    1.8554962482671918e-03,    4.2858646178307348e-04,    8.6726723448158906e-05,
    1.5602064410888157e-05,    2.5251059925720394e-06,    3.7126636045362732e-07,    4.9996922881189871e-08,
    6.2094632258917970e-09,    7.1546870744546087e-10,    7.6874304930703628e-11,    7.7370112580304187e-12,
    7.3228937548365655e-13,    6.5408276773059596e-14,    5.5307446718835060e-15,    4.4397427791999691e-16,
    3.3920278588307461e-17,    2.4722391515504332e-18,    1.7225207507931015e-19,    1.1495221846004686e-20,
    7.3606658336712452e-22,    4.5297215849413941e-23,    2.6830886623275263e-24,    1.5318438803014862e-25,
    8.4406173566788575e-27,    4.4940729024540833e-28,    2.3147475891847324e-29,    1.1545824985235007e-30,
    5.5825921377298809e-32,    2.6190415316212143e-33,    1.1932372059356725e-34,    5.2838424287908528e-36,
    2.2758965891634249e-37,    9.5423967613210731e-39,    3.8973602774564456e-40,    1.5516083094205125e-41,
    6.0251522430959588e-43,    2.2834469101239708e-44,    8.4508608645425453e-46,    3.0558831839187381e-47,
    1.0802547052700077e-48,    3.7349690288731375e-50,    1.2636524844344829e-51,    4.1855033999044797e-53,
    1.3578082728139664e-54,    4.3160201281371232e-56,    1.3448039708711485e-57,    4.1089834391549065e-59,
    1.2316082268464708e-60,    3.6226830000003144e-62,    1.0460649416018907e-63,    2.9662179493308681e-65,
    8.2623564720358692e-67,    2.2615006322859910e-68,    6.0843224203454335e-70,    1.6094475862599635e-71
},
// x = 3.1600
{
    1.5794304691357111e-01,    1.0853067241757727e-01,    5.4907598415871194e-02,    2.1651561000059605e-02,
    6.9452797448530494e-03,    1.8707009672503023e-03,    4.3334599809566581e-04,    8.7948443439019153e-05,
    1.5869209619309025e-05,    2.5761132085592538e-06,    3.7992133999704986e-07,    5.1319493388984687e-08,
    6.3933817860855737e-09,    7.3894128387731363e-10,    7.9642968146501651e-11,    8.0406268366334758e-12,
    7.6340107826190826e-13,    6.8400386429999482e-14,    5.8018614739397304e-15,    4.6719828576844606e-16,
    3.5806681227896299e-17,    2.6179280647284520e-18,    1.8297654962937523e-19,    1.2249351651906887e-20,
    7.8682569278545017e-22,    4.8573504284399691e-23,    2.8862249043161716e-24,    1.6530177529931861e-25,
    9.1370430938533509e-27,    4.8802328993865336e-28,    2.5215844998746304e-29,    1.2617250894365273e-30,
    6.1199228334528222e-32,    2.8802028085732414e-33,    1.3163714009554639e-34,    5.8475343350471117e-36,
    2.5266611189301530e-37,    1.0627319797068397e-38,    4.3542050689851966e-40,    1.7389723908264195e-41,
    6.7740919191442942e-43,    2.5754115520921780e-44,    9.5615766237010455e-46,    3.4684729320865393e-47,
    1.2299880298232793e-48,    4.2661392298344878e-50,    1.4479351557527472e-51,    4.8110815748207600e-53,
    1.5656949680247896e-54,    4.9925906940348766e-56,    1.5605417290525400e-57,    4.7832688985959904e-59,
    1.4382589405555724e-60,    4.2439378891819812e-62,    1.2293388388886264e-63,    3.4969575724803388e-65,
    9.7715994774910028e-67,    2.6830757826043541e-68,    7.2414078865524096e-70,    1.9215967605829194e-71
},
// x = 3.1700
{
    1.5745042551469490e-01,    1.0833808378666318e-01,    5.4922270511543636e-02,    2.1709896859938193e-02,
    6.9824351741406433e-03,    1.8859484160057799e-03,    4.3813467695970283e-04,    8.9181601975453208e-05,
    1.6139714930744243e-05,    2.6279256907049492e-06,    3.8874075932656558e-07,    5.2671449109402157e-08,
    6.5819802993576291e-09,    7.6308712708648515e-10,    8.2500037106805885e-11,    8.3549264248538703e-12,
    7.9570929277753998e-13,    7.1517383321113157e-14,    6.0851867084674937e-15,    4.9154476802257413e-16,
    3.7790508820681852e-17,    2.7716255468783110e-18,    1.9432632359432246e-19,    1.3049975350077231e-20,
    8.4088464995963290e-22,    5.2073820732744463e-23,    3.1039378555423027e-24,    1.7832977257345410e-25,
    9.8881736684938682e-27,    5.2980440179780873e-28,    2.7460827462099750e-29,    1.3783843883180038e-30,
    6.7068325180424155e-32,    3.1663641137230788e-33,    1.4517198816977406e-34,    6.4691030876200800e-36,
    2.8040481929278136e-37,    1.1831223779999283e-38,    4.8627560194675893e-40,    1.9482029242046849e-41,
    7.6130853297285401e-43,    2.9035181857210337e-44,    1.0813739354856538e-45,    3.9350779519306077e-47,
    1.3998603501240087e-48,    4.8706597843360952e-50,    1.6583300149095660e-51,    5.5275639389444866e-53,
    1.8045442479688460e-54,    5.7723915319301004e-56,    1.8099840537673595e-57,    5.5653669299931903e-59,
    1.6787105834536703e-60,    4.9690983643109045e-62,    1.4439431342159078e-63,    4.1203949249783707e-65,
    1.1550059111791219e-66,    3.1814245606772689e-68,    8.6135446654849676e-70,    2.2929333531651249e-71
},
// x = 3.1800
{
    1.5696079139750671e-01,    1.0814587797434648e-01,    5.4936378214160975e-02,    2.1767924807426682e-02,
    7.0195626003286436e-03,    1.9012382027229918e-03,    4.4295246512332466e-04,    9.0426238382356231e-05,
    1.6413604829191614e-05,    2.6805521885645835e-06,    3.9772697298812558e-07,    5.4053310341111828e-08,
    6.7753573134174668e-09,    7.8792265701226236e-10,    8.5447961426556846e-11,    8.6802414870587624e-12,
    8.2925510617272167e-13,    7.4763970172025086e-14,    6.3812206315652143e-15,    5.1706345255559994e-16,
    3.9876402109747590e-17,    2.9337398198604223e-18,    2.0633536320726975e-19,    1.3899774474532007e-20,
    8.9844487299165510e-22,    5.5812595100842041e-23,    3.3372157139995397e-24,    1.9233320084973169e-25,
    1.0698089240030054e-26,    5.7499749070241088e-28,    2.9896802469602122e-29,    1.5053678578417498e-30,
    7.3476984058010179e-32,    3.4798189201588672e-33,    1.6004454522899932e-34,    7.1542595296382860e-36,
    3.1107774336470970e-37,    1.3166678813178758e-38,    5.4286569539960261e-40,    2.1817635263857461e-41,
    8.5525959936969758e-43,    3.2720940289705075e-44,    1.2224786006658169e-45,    4.4625492033867187e-47,
    1.5924978524700674e-48,    5.5583584233205774e-50,    1.8984294443680929e-51,    6.3477841308694806e-53,
    2.0788392341283385e-54,    6.6707437484501488e-56,    2.0982558372546748e-57,    6.4720640878144009e-59,
    1.9583496921504294e-60,    5.8151042988315270e-62,    1.6951010341592368e-63,    4.8523263989291371e-65,
    1.3644609481234586e-66,    3.7701996225058010e-68,    1.0239777204172416e-69,    2.7344247442528788e-71
},
// x = 3.1900
{
    1.5647411865883648e-01,    1.0795405731018032e-01,    5.4949926831080707e-02,    2.1825645349238726e-02,
    7.0566611744439980e-03,    1.9165699354468898e-03,    4.4779932807541376e-04,    9.1682391565892809e-05,
    1.6690903784381730e-05,    2.7340014923851886e-06,    4.0688235638216620e-07,    5.5465603975943035e-08,
    6.9736129819498917e-09,    8.1346461896896880e-10,    8.8489247729778823e-11,    9.0169123346158489e-12,
    8.6408084166245157e-13,    7.8145007073243750e-14,    6.6904819246676003e-15,    5.4380606612741000e-16,
    4.2069203987665937e-17,    3.1046982608573270e-18,    2.1903934288061361e-19,    1.4801574453688711e-20,
    9.5971926827214609e-22,    5.9805129132230074e-23,    3.5871097317905798e-24,    2.0738123727683193e-25,
    1.1571158052099227e-26,    6.2386769386769698e-28,    3.2539262696688155e-29,    1.6435482570453054e-30,
    8.0472667266753065e-32,    3.8230619547261565e-33,    1.7638169726951102e-34,    7.9092551661991717e-36,
    3.4498353905941410e-37,    1.4647532551126624e-38,    6.0581450314580343e-40,    2.4423854503782173e-41,
    9.6042609289059096e-43,    3.6859668629251730e-44,    1.3814232266742493e-45,    5.0585786834425519e-47,
    1.8108584626447165e-48,    6.3403393867562604e-50,    2.1723052941852345e-51,    7.2863347428830260e-53,
    2.3936936401320450e-54,    7.7051788451464913e-56,    2.4312399141690149e-57,    7.5226905674737080e-59,
    2.2833986524271165e-60,    6.8015861305317512e-62,    1.9888845058075902e-63,    5.7111733835583095e-65,
    1.6110087073638334e-66,    4.4654233039766384e-68,    1.2166071841675243e-69,    3.2590234876003966e-71
},
// x = 3.2000
{
    1.5599038167606655e-01,    1.0776262405016261e-01,    5.4962921629039110e-02,    2.1883059004788999e-02,
    7.0937300560631579e-03,    1.9319432221113792e-03,    4.5267523005529110e-04,    9.2950100011759490e-05,
    1.6971636250168620e-05,    2.7882824327387117e-06,    4.1620930578251557e-07,    5.6908863540952254e-08,
    7.1768490819214749e-09,    8.3973008844072778e-10,    9.1626460702833042e-11,    9.3652883213035033e-12,
    9.0023009020538814e-13,    8.1665516060432305e-14,    7.0135082944102477e-15,    5.7182640631380072e-16,
    4.4373967460802828e-17,    3.2849482222643467e-18,    2.3247572412566690e-19,    1.5758351747157759e-20,
    1.0249328392872407e-21,    6.4067645571886852e-23,    3.8547379852940771e-24,    2.2354769045369232e-25,
    1.2512055621240429e-26,    6.7669970034722619e-28,    3.5404896088454070e-29,    1.7938686610703499e-30,
    8.8106823631538588e-32,    4.1988060547217689e-33,    1.9432186080156432e-34,    8.7409311880373422e-36,
    3.8245006698601256e-37,    1.6289034918921472e-38,    6.7581107379083986e-40,    2.7330955829384920e-41,
    1.0781017528991217e-42,    4.1505209125918616e-44,    1.5603911956081260e-45,    5.7317992577753511e-47,
    2.0582724004528126e-48,    7.2291440159655316e-50,    2.4845709126150388e-51,    8.3598011780761540e-53,
    2.7549378736794338e-54,    8.8957484853743918e-56,    2.8156860167309320e-57,    8.7394950673866135e-59,
    2.6610419158667845e-60,    7.9512809487276359e-62,    2.3323486359792074e-63,    6.7184074234628062e-65,
    1.9010609655472678e-66,    5.2858888740160365e-68,    1.4446514477551382e-69,    3.8820181613128288e-71
},
// x = 3.2100
{
    1.5550955510361553e-01,    1.0757158037812418e-01,    5.4975367834340629e-02,    2.1940166305942550e-02,
    7.1307684132820700e-03,    1.9473576705722690e-03,    4.5758013406245777e-04,    9.4229401783499749e-05,
    1.7255826662926338e-05,    2.8434038801515027e-06,    4.2571023835357769e-07,    5.8383629240246383e-08,
    7.3851690309398956e-09,    8.6673647591700680e-10,    9.4862224161332040e-11,    9.7257280420444750e-12,
    9.3774774283402109e-13,    8.5330685806869259e-14,    7.3508570893411215e-15,    6.0118041570993908e-16,
    4.6795963893263343e-17,    3.4749578832109781e-18,    2.4668383778915009e-19,    1.6773241307006505e-20,
    1.0943233251919060e-21,    6.8617339902645982e-23,    4.1412893554740752e-24,    2.4091129201483326e-25,
    1.3525785126464684e-26,    7.3379911499310837e-28,    3.8511673320327297e-29,    1.9573478467018046e-30,
    9.6435207488125621e-32,    4.6100003690426187e-33,    2.1401598474388742e-34,    9.6567717441031954e-36,
    4.2383713288225633e-37,    1.8107974506702613e-38,    7.5361637051325610e-40,    3.0572472584011726e-41,
    1.2097243646803425e-42,    4.6717587404975356e-44,    1.7618243024748704e-45,    6.4918959379101892e-47,
    2.3384875272491103e-48,    8.2389308955214547e-50,    2.8404509486357667e-51,    9.5870256421288922e-53,
    3.1692165415462720e-54,    1.0265376397169726e-55,    3.2593349425231180e-57,    1.0148073343094177e-58,
    3.0995707514665321e-60,    9.2905111943058012e-62,    2.7336866979304772e-63,    7.8990424546507174e-65,
    2.2421108082681540e-66,    6.2536280369602467e-68,    1.7144710394054292e-69,    4.6214447348518887e-71
},
// x = 3.2200
{
    1.5503161386947015e-01,    1.0738092840709192e-01,    5.4987270633049118e-02,    2.1996967796767210e-02,
    7.1677754226855891e-03,    1.9628128886398036e-03,    4.6251400186638121e-04,    9.5520334520873520e-05,
    1.7543499439951884e-05,    2.8993747447300611e-06,    4.3538759216580335e-07,    5.9890447996559391e-08,
    7.5986779046650214e-09,    8.9450153176889612e-10,    9.8199222130798087e-11,    1.0098599535000091e-11,
    9.7668002365438621e-13,    8.9145876430287376e-14,    7.7031059338717220e-15,    6.3192625846951812e-16,
    4.9340691539049436e-17,    3.6752171337954176e-18,    2.6170496973172690e-19,    1.7849544376876848e-20,
    1.1681418704808047e-21,    7.3472434777651490e-23,    4.4480277291821145e-24,    2.5955600539929233e-25,
    1.4617699069977841e-26,    7.9549391210701732e-28,    4.1878941308897850e-29,    2.1350860688899719e-30,
    1.0551822192011696e-31,    5.0598500056247729e-33,    2.3562863537782359e-34,    1.0664961814270258e-35,
    4.6953947310688000e-37,    2.0122827685699233e-38,    8.4007048966202396e-40,    3.4185541607359147e-41,
    1.3568912206038864e-42,    5.2563697758304505e-44,    1.9884522997639137e-45,    7.3497298698068318e-47,
    2.6557200334962522e-48,    9.3856778451443092e-50,    3.2458598685127307e-51,    1.0989405029901829e-52,
    3.6440988211977316e-54,    1.1840257970776129e-55,    3.7710599963751775e-57,    1.1777857922307617e-58,
    3.6085491773996901e-60,    1.0849735134717640e-61,    3.2024090307848360e-63,    9.2822044211502513e-65,
    2.6429104504333544e-66,    7.3944553437531824e-68,    2.0335419500754972e-69,    5.4985686341427924e-71
},
// x = 3.2300
{
    1.5455653317176563e-01,    1.0719067018062893e-01,    5.4998635171181469e-02,    2.2053464033289220e-02,
    7.2047502693162605e-03,    1.9783084841107986e-03,    4.6747679401632581e-04,    9.6822935438279845e-05,
    1.7834678977874398e-05,    2.9562039757830433e-06,    4.4524382620943852e-07,    6.1429873492573724e-08,
    7.8174824542698804e-09,    9.2304335116599398e-10,    1.0164019994113629e-10,    1.0484280487060743e-11,
    1.0170745235255011e-12,    9.3116624416297135e-14,    8.0708533798672142e-15,    6.6412439924236076e-16,
    5.2013884371218617e-17,    3.8862384931461491e-18,    2.7758245007790514e-19,    1.8990736642829074e-20,
    1.2466537271414139e-21,    7.8652237278211194e-23,    4.7762964328160944e-24,    2.7957135274584377e-25,
    1.5793522283153129e-26,    8.6213598431787969e-28,    4.5527523167096226e-29,    2.3282712550714266e-30,
    1.1542128799405990e-31,    5.5518372343098278e-33,    2.5933917091483617e-34,    1.1774450061006416e-35,
    5.1999000726496735e-37,    2.2353921581441661e-38,    9.3610057514574541e-40,    3.8211276109654973e-41,
    1.5213761789517657e-42,    5.9118061638545415e-44,    2.2433257012195462e-45,    8.3174764346797406e-47,
    3.0147110736102225e-48,    1.0687408323394981e-49,    3.7074902388829572e-51,    1.2591226920831108e-52,
    4.1882033443964702e-54,    1.3650313820036069e-55,    4.3610280386660682e-57,    1.3662677457959099e-58,
    4.1990040846827013e-60,    1.2664179578803243e-61,    3.7495492896756437e-63,    1.0901790114545512e-64,
    3.1136775562860196e-66,    8.7386018420984677e-68,    2.4106465605602686e-69,    6.5384493569380740e-71
},
// x = 3.2400
{
    1.5408428847541289e-01,    1.0700080767415103e-01,    5.5009466554902652e-02,    2.2109655583251835e-02,
    7.2416921466425402e-03,    1.9938440648003515e-03,    4.7246846985122073e-04,    9.8137241323232284e-05,
    1.8129389651071078e-05,    3.0139005614396350e-06,    4.5528142040655173e-07,    6.3002466211983846e-08,
    8.0416911239503241e-09,    9.5238037903383879e-10,    1.0518796533500220e-10,    1.0883158442769603e-11,
    1.0589802344288799e-12,    9.7248647660640450e-14,    8.4547195762825201e-15,    6.9783768457466872e-16,
    5.4821521217062506e-17,    4.1085580624578753e-18,    2.9436174617094586e-19,    2.0200476750295733e-20,
    1.3301389907300047e-21,    8.4177199132079303e-23,    5.1275229102378212e-24,    3.0105276090504614e-25,
    1.7059376355868690e-26,    9.3410279254141198e-28,    4.9479825021994551e-29,    2.5381856458355907e-30,
    1.2621524185806455e-31,    6.0897443620732830e-33,    2.8534301272196256e-34,    1.2995017068526111e-35,
    5.7566338080413460e-37,    2.4823612136645153e-38,    1.0427294929044316e-39,    4.2695175644785082e-41,
    1.7051484791161088e-42,    6.6463666882366816e-44,    2.5298521912810942e-45,    9.4087790115857480e-47,
    3.4207900219584415e-48,    1.2164445094969031e-49,    4.2329119517065521e-51,    1.4420048403934130e-52,
    4.8113394419144771e-54,    1.5729704375356568e-55,    5.0408827777489924e-57,    1.5841395336553682e-58,
    4.8836429803876181e-60,    1.4774566778988209e-61,    4.3879021415113342e-63,    1.2797228842063097e-64,
    3.6663345082378956e-66,    1.0321452221041406e-67,    2.8560964485083900e-69,    7.7706014277827375e-71
},
// x = 3.2500
{
    1.5361485550877269e-01,    1.0681134279622033e-01,    5.5019769850723155e-02,    2.2165543025877050e-02,
    7.2786002565264057e-03,    2.0094192385731570e-03,    4.7748898750956742e-04,    9.9463288534886853e-05,
    1.8427655810089615e-05,    3.0724735282643119e-06,    4.6550287562133549e-07,    6.4608793480294597e-08,
    8.2714140684815729e-09,    9.8253141505174730e-10,    1.0884538959013286e-10,    1.1295631016715544e-11,
    1.1024475845384770e-12,    1.0154785063254181e-13,    8.8553469572578165e-15,    7.3313142683743896e-16,
    5.7769835208550842e-17,    4.3427365141822106e-18,    3.1209055937078937e-19,    2.1482615202053016e-20,
    1.4188933718686700e-21,    9.0068980033070488e-23,    5.5032236574103914e-24,    3.2410192760865961e-25,
    1.8421805571538637e-26,    1.0117991232126949e-27,    5.3759950138954432e-29,    2.7662129123189058e-30,
    1.3797676169565175e-31,    6.6776784058708671e-33,    3.1385302077533928e-34,    1.4337349409823110e-35,
    6.3707982228043803e-37,    2.7556478600950019e-38,    1.1610853354380341e-39,    4.7687576725833550e-41,
    1.9103934870021857e-42,    7.4692895932955236e-44,    2.8518370220983454e-45,    1.0638920115232976e-46,
    3.8799450975189029e-48,    1.3839694335056333e-49,    4.8306837031313450e-51,    1.6507123018937375e-52,
    5.5246668265731478e-54,    1.8117413472904761e-55,    5.8239532883152834e-57,    1.8358638448022640e-58,
    5.6771032496494981e-60,    1.7227949155027690e-61,    5.1322970711523946e-63,    1.5014362547349970e-64,
    4.3147857497908136e-66,    1.2184401923441171e-67,    3.3819922654759837e-69,    9.2297677276354047e-71
},
// x = 3.2600
{
    1.5314821026037514e-01,    1.0662227738981650e-01,    5.5029550085697362e-02,    2.2221126951630341e-02,
    7.3154738091904840e-03,    2.0250336133744158e-03,    4.8253830393938784e-04,    1.0080111300262340e-04,
    1.8729501780077785e-05,    3.1319319408680862e-06,    4.7591071366869103e-07,    6.6249429505350372e-08,
    8.5067631708203902e-09,    1.0135156186909933e-09,    1.1261540865572030e-10,    1.1722106109432360e-11,
    1.1475284740016091e-12,    1.0602032966146278e-13,    9.2734009490952264e-15,    7.7007349075007406e-16,
    6.0865323557527881e-17,    4.5893601185886108e-18,    3.3081892583758241e-19,    2.2841203652662251e-20,
    1.5132290047215813e-21,    9.6350514209038918e-23,    5.9050094268016483e-24,    3.4882720888939251e-25,
    1.9887804434596147e-26,    1.0956589593373091e-27,    5.8393820822756913e-29,    3.0138457826551243e-30,
    1.5078882666096462e-31,    7.3200976972408525e-33,    3.4510098147467065e-34,    1.5813120016232560e-35,
    7.0480934199850677e-37,    3.0579535897881152e-38,    1.2924118325295173e-39,    5.3244147950853315e-41,
    2.1395355609850100e-42,    8.3908552146040182e-44,    3.2135278189565991e-45,    1.2025011803071043e-46,
    4.3989021861063713e-48,    1.5738963700885893e-49,    5.5104781903703370e-51,    1.8887881731256321e-52,
    6.3408760471735107e-54,    2.0857909911836108e-55,    6.7254911251229649e-57,    2.1265628479683220e-58,
    6.5962373702925336e-60,    2.0078667391309769e-61,    5.9999136356434847e-63,    1.7606462317764855e-64,
    5.0752391039209148e-66,    1.4375853253694066e-67,    4.0025266975503952e-69,    1.0956823835292758e-70
},
// x = 3.2700
{
    1.5268432897568493e-01,    1.0643361323358574e-01,    5.5038812247624601e-02,    2.2276407961988431e-02,
    7.3523120231845478e-03,    2.0406867972603323e-03,    4.8761637490819934e-04,    1.0215075022467751e-04,
    1.9034951859219843e-05,    3.1922849015162616e-06,    4.8650747732107889e-07,    6.7924955417588782e-08,
    8.7478520597510681e-09,    1.0453525142931676e-09,    1.1650102430289277e-10,    1.2163002126840209e-11,
    1.1942763114413426e-12,    1.1067237834958549e-13,    9.7095706965445624e-15,    8.0873438256756192e-16,
    6.4114757665382324e-17,    4.8490418089457000e-18,    3.5059932144800277e-19,    2.4280504615390610e-20,
    1.6134752940703424e-21,    1.0304608039159394e-22,    6.3345907152078026e-24,    3.7534402889865335e-25,
    2.1464846881828837e-26,    1.1861474722792444e-27,    6.3409308594759130e-29,    3.2826942118722512e-30,
    1.6474121006436352e-31,    8.0218405622736990e-33,    3.7933921655369170e-34,    1.7435075360938882e-35,
    7.7947630089049016e-37,    3.3922466441682597e-38,    1.4380797509815831e-39,    5.9426433859402497e-41,
    2.3952632467822190e-42,    9.4224994154882374e-44,    3.6196642576826529e-45,    1.3586207444637860e-46,
    4.9852127790043012e-48,    1.7891318295875472e-49,    6.2832226597519440e-51,    2.1602474562594342e-52,
    7.2743923315697005e-54,    2.4001897081780669e-55,    7.7629398379373096e-57,    2.4621129741877651e-58,
    7.6604391195827381e-60,    2.3389449639503175e-61,    7.0106442712948686e-63,    2.0635401853636235e-64,
    5.9665778587157161e-66,    1.6952372447357733e-67,    4.7343374737014909e-69,    1.2999835031303497e-70
},
// x = 3.2800
{
    1.5222318815391053e-01,    1.0624535204306793e-01,    5.5047561285250859e-02,    2.2331386669209932e-02,
    7.3891141253515664e-03,    2.0563783984282181e-03,    4.9272315501302860e-04,    1.0351223526682545e-04,
    1.9344030317180241e-05,    3.2535415497327607e-06,    4.9729573031364747e-07,    6.9635959310014018e-08,
    8.9947961275738920e-09,    1.0780619961886374e-09,    1.2050530528937989e-10,    1.2618748203266432e-11,
    1.2427460511910677e-12,    1.1551049311239219e-13,    1.0164569808834662e-14,    8.4918734200114358e-16,
    6.7525193577162708e-17,    5.1224222866098946e-18,    3.7148677099640806e-19,    2.5805001598201122e-20,
    1.7199798026726469e-21,    1.1018137534754157e-22,    6.7937835492872202e-24,    4.0377531332734310e-25,
    2.3160917273846467e-26,    1.2837631415966422e-27,    6.8836373175157991e-29,    3.5744941318134243e-30,
    1.7993100922940680e-31,    8.7881562306681367e-33,    4.1684232246619537e-34,    1.9217130007324157e-35,
    8.6176438082501734e-37,    3.7617873108816198e-38,    1.5995993734279319e-39,    6.6302452123894451e-41,
    2.6805570292549531e-42,    1.0576938924467293e-43,    4.0755331245118816e-45,    1.5343937165158075e-46,
    5.6473520461752719e-48,    2.0329478889800329e-49,    7.1612566270122418e-51,    2.4696380283913947e-52,
    8.3416057566779045e-54,    2.7607162031128915e-55,    8.9562411847231087e-57,    2.8492529390074218e-58,
    8.8920164985875587e-60,    2.7232672034777515e-61,    8.1875116326942218e-63,    2.4173011494897523e-64,
    7.0107914331732668e-66,    1.9980033050594843e-67,    5.5969184830027241e-69,    1.5415291103416136e-70
},
// x = 3.2900
{
    1.5176476454485777e-01,    1.0605749547190280e-01,    5.5055802108472232e-02,    2.2386063696109126e-02,
    7.4258793507932302e-03,    2.0721080252461936e-03,    4.9785859769045366e-04,    1.0488560276111975e-04,
    1.9656761393554673e-05,    3.3157110619011140e-06,    5.0827805734763893e-07,    7.1383036277885116e-08,
    9.2477125478343714e-09,    1.1116643338549498e-09,    1.2463138853842744e-10,    1.3089784428081987e-11,
    1.2929942312719431e-12,    1.2054137884972866e-13,    1.0639137125893574e-14,    8.9150843694383556e-16,
    7.1103982790357411e-17,    5.4101711673448774e-18,    3.9353896183772732e-19,    2.7419409685995082e-20,
    1.8331091806546295e-21,    1.1778359113888588e-22,    7.2845155837550789e-24,    4.3425194769458661e-25,
    2.4984543268072859e-26,    1.3890400106495808e-27,    7.4707210831137766e-29,    3.8911168199745547e-30,
    1.9646321460370864e-31,    9.6247381383680544e-33,    4.5790905031653922e-34,    2.1174469115410944e-35,
    9.5242199004185963e-37,    4.1701555211608621e-38,    1.7786341541255132e-39,    7.3947349096619711e-41,
    2.9987198905712451e-42,    1.1868309775319760e-43,    4.5870293203638696e-45,    1.7322169516450103e-46,
    6.3948281718977157e-48,    2.3090267243508307e-49,    8.1585087996746072e-51,    2.8221092452427706e-52,
    9.5611310404656503e-54,    3.1739536769105955e-55,    1.0328182892735135e-56,    3.2957068004161110e-58,
    1.0316617876192707e-59,    3.1691803546084826e-61,    9.5571484386704595e-63,    2.8302639851405685e-64,
    8.2334736065383973e-66,    2.3535974845230472e-67,    6.6130983271337726e-69,    1.8269548128621724e-70
},
// x = 3.3000
{
    1.5130903514582605e-01,    1.0587004511301450e-01,    5.5063539588540190e-02,    2.2440439675832431e-02,
    7.4626069428349812e-03,    2.0878752862824920e-03,    5.0302265522667619e-04,    1.0627088690467646e-04,
    1.9973169296328668e-05,    3.3788026508621481e-06,    5.1945706409206238e-07,    7.3166788458113172e-08,
    9.5067202930917553e-09,    1.1461801771150768e-09,    1.2888248033203400e-10,    1.3576562075990416e-11,
    1.3450790121240638e-12,    1.2577195474977770e-13,    1.1134037505208744e-14,    9.3577666107368624e-16,
    7.4858783428814476e-17,    5.7129881702336088e-18,    4.1681636213409186e-19,    2.9128686586906548e-20,
    1.9532501387561213e-21,    1.2588149628535680e-22,    7.8088325278798469e-24,    4.6691326183198964e-25,
    2.6944830680022914e-26,    1.5025500861392307e-27,    8.1056412685164952e-29,    4.2345789286055151e-30,
    2.1445132087783044e-31,    1.0537759799765302e-32,    5.0286433714096584e-34,    2.3323659545110482e-35,
    1.0522681401040411e-36,    4.6212809475619983e-38,    1.9770156581328753e-39,    8.2444119186223767e-41,
    3.3534109475184207e-42,    1.3312320168070636e-43,    5.1607234279287888e-45,    1.9547702193447461e-46,
    7.2383042017334802e-48,    2.6215103923837102e-49,    9.2906954632473849e-51,    3.2234891049021657e-52,
    1.0954100650241918e-53,    3.6473986189344358e-55,    1.1904793438616930e-56,    3.8103240850111644e-58,
    1.1963718733603484e-59,    3.6863061373278889e-61,    1.1150348934269574e-62,    3.3120954024857752e-64,
    9.6643986356064413e-66,    2.7710212023559558e-67,    7.8095970932650807e-69,    2.1640511074278198e-70
},
// x = 3.3100
{
    1.5085597719854782e-01,    1.0568300249977609e-01,    5.5070778558267378e-02,    2.2494515251638033e-02,
    7.4992961529905759e-03,    2.1036797903343620e-03,    5.0821527876762000e-04,    1.0766812145851230e-04,
    2.0293278200343713e-05,    3.4428255655084110e-06,    5.3083537718364529e-07,    7.4987825068362874e-08,
    9.7719401527251528e-09,    1.1816305613753255e-09,    1.3326185751857234e-10,    1.4079543841005690e-11,
    1.3990602161022064e-12,    1.3120936022839186e-13,    1.1650062629786013e-14,    9.8207403440919018e-16,
    7.8797571792536326e-17,    6.0316043505831945e-18,    4.4138234387238266e-19,    3.0938044161080009e-20,
    2.0808104673189597e-21,    1.3450552101081718e-22,    8.3689049166401693e-24,    5.0190754195645798e-25,
    2.9051500445251393e-26,    1.6249058901953055e-27,    8.7921133613029724e-29,    4.6070532180196360e-30,
    2.3401798304871103e-31,    1.1533913437698339e-32,    5.5206150013441628e-34,    2.5682770243739505e-35,
    1.1621988336599080e-36,    5.1194758188626808e-38,    2.1967598994820459e-39,    9.1884394029398233e-41,
    3.7486824661962820e-42,    1.4926419196653398e-43,    5.8039365218685363e-45,    2.2050484906393252e-46,
    8.1897337833002744e-48,    2.9750564588889035e-49,    1.0575542850422611e-50,    3.6803710058138903e-52,
    1.2544495367633561e-53,    4.1895838789808144e-55,    1.3715790017232255e-56,    4.4032392815434145e-58,
    1.3867177384173722e-59,    4.2857306725958128e-61,    1.3002702368520954e-62,    3.8740013910303068e-64,
    1.1338187120232142e-65,    3.2607729450127261e-67,    9.2176738133036317e-69,    2.5619596455814049e-70
},
// x = 3.3200
{
    1.5040556818616921e-01,    1.0549636910715331e-01,    5.5077523812235490e-02,    2.2548291076678141e-02,
    7.5359462409261774e-03,    2.1195211464565852e-03,    5.1343641832905438e-04,    1.0907733974643331e-04,
    2.0617112245771311e-05,    3.5077890903754496e-06,    5.4241564422505971e-07,    7.6846762445853166e-08,
    1.0043494750775810e-08,    1.2180369129028125e-09,    1.3777286873486682e-10,    1.4599204074156053e-11,
    1.4549993677471887e-12,    1.3686096100627401e-13,    1.2188031837674205e-14,    1.0304857068928492e-15,
    8.2928654294351823e-17,    6.3667833782629823e-18,    4.6730331082523841e-19,    3.2852960451018943e-20,
    2.2162201029822669e-21,    1.4368784676259233e-22,    8.9670352436500643e-24,    5.3939257179311529e-25,
    3.1314927800264844e-26,    1.7567631741104438e-27,    9.5341272398544019e-29,    5.0108800408081590e-30,
    2.5529572055788147e-31,    1.2620451572528090e-32,    6.0588460626043710e-34,    2.8271502657558162e-35,
    1.2831940054286194e-36,    5.6694706868613695e-38,    2.4400852039139931e-39,    1.0236930795783178e-40,
    4.1890205800427439e-42,    1.6729983028289811e-43,    6.5248229702935021e-45,    2.4863977839582351e-46,
    9.2625123306546644e-48,    3.3749001381608190e-49,    1.2033036298204697e-50,    4.2002112540810102e-52,
    1.4359516948795313e-53,    4.8102178365968835e-55,    1.5795086654000602e-56,    5.0860533015190807e-58,
    1.6065869167321501e-59,    4.9802214927032508e-61,    1.5155320356198313e-62,    4.5289661200718093e-64,
    1.3295075235345247e-65,    3.8350911741763527e-67,    1.0873879011497719e-68,    3.0314020521583507e-70
},
// x = 3.3300
{
    1.4995778583027264e-01,    1.0531014635282866e-01,    5.5083780107003585e-02,    2.2601767813784261e-02,
    7.5725564744240298e-03,    2.1353989639895971e-03,    5.1868602280674095e-04,    1.1049857465397246e-04,
    2.0944695536594863e-05,    3.5737025452299339e-06,    5.5420053378141855e-07,    7.8744224085851778e-08,
    1.0321508563823708e-08,    1.2554210541423132e-09,    1.4241893564278688e-10,    1.5136029024950020e-11,
    1.5129597348438254e-12,    1.4273435532651674e-13,    1.2748792973529270e-14,    1.0811000650804107e-15,
    8.7260679794734695e-17,    6.7193228629564386e-18,    4.9464883163351533e-19,    3.4879192233262587e-20,
    2.3599322451244079e-21,    1.5346250020072022e-22,    9.6056654737385193e-24,    5.7953620428131609e-25,
    3.3746183806872021e-26,    1.8988238033264323e-27,    1.0335966385115652e-28,    5.4485796265681349e-30,
    2.7842767284040420e-31,    1.3805231785473223e-32,    6.6475103058244390e-34,    3.1111331963133571e-35,
    1.4163250621786572e-36,    6.2764533991808134e-38,    2.7094317325291209e-39,    1.1401044684047798e-40,
    4.6793900673301396e-42,    1.8744520271786459e-43,    7.3324620500329356e-45,    2.8025549480167306e-46,
    1.0471645304541862e-47,    3.8269236797149122e-49,    1.3685699314925243e-50,    4.7914386096645129e-52,
    1.6430008074525624e-53,    5.5203417096040634e-55,    1.8181370297323901e-56,    5.8720398451621086e-58,
    1.8604409885551286e-59,    5.7844758441357426e-61,    1.7655671661654311e-62,    5.2920269576084889e-64,
    1.5581802959352079e-65,    4.5082356661885376e-67,    1.2820928971240974e-68,    3.5849465633334164e-70
},
// x = 3.3400
{
    1.4951260808793976e-01,    1.0512433559830582e-01,    5.5089552161317779e-02,    2.2654946135255247e-02,
    7.6091261293456955e-03,    2.1513128525872138e-03,    5.2396403998660422e-04,    1.1193185862737766e-04,
    2.1276052139099686e-05,    3.6405752846547489e-06,    5.6619273537505368e-07,    8.0680840679859251e-08,
    1.0606107938897133e-08,    1.2938052090723629e-09,    1.4720355418042437e-10,    1.5690517086641725e-11,
    1.5730063702766992e-12,    1.4883738031505016e-13,    1.3333223262701013e-14,    1.1340088420149916e-15,
    9.1802652346318419e-17,    7.0900557278496703e-18,    5.2349177819387443e-19,    3.7022788111847026e-20,
    2.5124245241703803e-21,    1.6386545186241375e-22,    1.0287384953882767e-23,    6.2251696547156974e-25,
    3.6357079350938875e-26,    2.0518388238101251e-27,    1.1202228363428790e-28,    5.9228652198358628e-30,
    3.0356840983895203e-31,    1.5096764886197883e-32,    7.2911421761719406e-34,    3.4225659974422923e-35,
    1.5627630710780818e-36,    6.9461115530677328e-38,    3.0074828143702122e-39,    1.2693088801074518e-40,
    5.2252835771364912e-42,    2.0993898438122612e-43,    8.2369592796763953e-45,    3.1576917988064957e-46,
    1.1833935479839214e-47,    4.3377338193978441e-49,    1.5558906030721531e-50,    5.4635773107110809e-52,
    1.8790925405578160e-53,    6.3325072946680282e-55,    2.0918753717838099e-56,    6.7763799924348529e-58,
    2.1533980688697198e-59,    6.7154046707644640e-61,    2.0558539840979746e-62,    6.1805909237699133e-64,
    1.8252639229446806e-65,    5.2968132085643971e-67,    1.5108720916702191e-68,    4.2373185818924934e-70
},
// x = 3.3500
{
    1.4907001314885471e-01,    1.0493893814999464e-01,    5.5094844656322195e-02,    2.2707826722648070e-02,
    7.6456544895948594e-03,    2.1672624222439812e-03,    5.2927041655492031e-04,    1.1337722367264927e-04,
    2.1611206080371240e-05,    3.7084166976310758e-06,    5.7839495947856914e-07,    8.2657250153477053e-08,
    1.0897421111413470e-08,    1.3332120086004293e-09,    1.5213029582791643e-10,    1.6263179045332009e-11,
    1.6352061546949146e-12,    1.5517811848657291e-13,    1.3942230208332169e-14,    1.1893072303668461e-15,
    9.6563944359934386e-17,    7.4798516333213460e-18,    5.5390846954091602e-19,    3.9290102174718876e-20,
    2.6742002239645094e-21,    1.7493471971559377e-22,    1.1014938742038430e-23,    6.6852469229918453e-25,
    3.9160211753307693e-26,    2.2166117205166822e-27,    1.2137846659598087e-28,    6.4366571271791024e-30,
    3.3088480126927886e-31,    1.6504266730411977e-32,    7.9946666103936984e-34,    3.7639980645851845e-35,
    1.7237876496743227e-36,    7.6846787277016889e-38,    3.3371882485728290e-39,    1.4126633967124404e-40,
    5.8327757285654396e-42,    2.3504593578122014e-43,    9.2495584635707488e-45,    3.5564640703691275e-46,
    1.3368191268835911e-47,    4.9147482007804837e-49,    1.7681230894898946e-50,    6.2273851816706700e-52,
    2.1481872250779677e-53,    7.2609777135707885e-55,    2.4057515152563026e-56,    7.8164287682053557e-58,
    2.4913268230149293e-59,    7.7924572630398321e-61,    2.3927121335491353e-62,    7.2147986560845210e-64,
    2.1370564584434957e-65,    6.2201544691437548e-67,    1.7795511254907496e-68,    5.0057622059142240e-70
},
// x = 3.3600
{
    1.4862997943244674e-01,    1.0475395526027749e-01,    5.5099662235770404e-02,    2.2760410266571482e-02,
    7.6821408470797662e-03,    2.1832472833221241e-03,    5.3460509810852300e-04,    1.1483470135462555e-04,
    2.1950181346801591e-05,    3.7772362071175057e-06,    5.9080993750616801e-07,    8.4674097703954010e-08,
    1.1195578223149426e-08,    1.3736644959969810e-09,    1.5720280888797139e-10,    1.6854538332941732e-11,
    1.6996278399971208e-12,    1.6176490439857467e-13,    1.4576752511968525e-14,    1.2470939989212268e-15,
    1.0155431020428743e-16,    7.8896184522427208e-18,    5.8597882141924277e-19,    4.1687808235008515e-20,
    2.8457895604926340e-21,    1.8671047782426733e-22,    1.1791236374290024e-23,    7.1776120600202906e-25,
    4.2169014137765184e-26,    2.3940018792266742e-27,    1.3148113943965964e-28,    6.9930977328528099e-30,
    3.6055694866943570e-31,    1.8037713950033913e-32,    8.7634311816426382e-34,    4.1382059160644516e-35,
    1.9007966149304207e-36,    8.4989848168066649e-38,    3.7017897503632138e-39,    1.5716638890965541e-40,
    6.5085825469095527e-42,    2.6305965380011674e-43,    1.0382765537623201e-44,    4.0040656857742330e-46,
    1.5095458386349314e-47,    5.5662917709908815e-49,    2.0084839915677380e-50,    7.0950086158912849e-52,
    2.4547698127604975e-53,    8.3219540529494409e-55,    2.7654935878930148e-56,    9.0120179138920107e-58,
    2.8809535749299817e-59,    9.0379922358139032e-61,    2.7834284059102894e-62,    8.4179428311218577e-64,
    2.5008634861057307e-65,    7.3007498734118886e-67,    2.0949282415440608e-68,    5.9104608923897450e-70
},
// x = 3.3700
{
    1.4819248558507186e-01,    1.0456938812855665e-01,    5.5104009506238047e-02,    2.2812697466482377e-02,
    7.7185845016752574e-03,    2.1992670465781324e-03,    5.3996802916503003e-04,    1.1630432279611826e-04,
    2.2293001882604585e-05,    3.8470432696263367e-06,    6.0344042180327057e-07,    8.6732035837409336e-08,
    1.1500711340239535e-08,    1.4151861323683218e-09,    1.6242481978116705e-10,    1.7465131284094799e-11,
    1.7663420936482543e-12,    1.6860633145609505e-13,    1.5237761018187045e-14,    1.3074716124985170e-15,
    1.0678390025168098e-16,    8.3203037985413336e-18,    6.1978650174696147e-19,    4.4222914679832650e-20,
    3.0277510193251077e-21,    1.9923517035777878e-22,    1.2619361091662008e-23,    7.7044102303497536e-25,
    4.5397807708409204e-26,    2.5849282636624499e-27,    1.4238706862151126e-28,    7.5955675460736398e-30,
    3.9277918452622953e-31,    1.9707903876336572e-32,    9.6032407680803847e-34,    4.5482125667545446e-35,
    2.0953164532239306e-36,    9.3965108095263559e-38,    4.1048487299349714e-39,    1.7479586826275514e-40,
    7.2601267427886173e-42,    2.9430560225723216e-43,    1.1650485414653349e-44,    4.5062889060464774e-46,
    1.7039277382031538e-47,    6.3017042636655545e-49,    2.2805928216545349e-50,    8.0801564290052446e-52,
    2.8039173355482201e-53,    9.5338311364845251e-55,    3.1776248302139424e-56,    1.0385799641551247e-57,
    3.3299842798827919e-59,    1.0477701263616701e-60,    3.2364009303067837e-62,    9.8169489745875873e-64,
    2.9251552942553178e-65,    8.5647534940852173e-67,    2.4649327722322444e-68,    6.9750266913750394e-70
},
// x = 3.3800
{
    1.4775751047723307e-01,    1.0438523790228381e-01,    5.5107891037336175e-02,    2.2864689030484695e-02,
    7.7549847611844119e-03,    2.2153213231889371e-03,    5.4535915317307907e-04,    1.1778611867709471e-04,
    2.2639691588339207e-05,    3.9178473747969590e-06,    6.1628918563439875e-07,    8.8831724405722180e-08,
    1.1812954471200537e-08,    1.4578008021679367e-09,    1.6780013435606718e-10,    1.8095507396946138e-11,
    1.8354215438393252e-12,    1.7571125886988071e-13,    1.5926259683755962e-14,    1.3705463553923678e-15,
    1.1226327538248648e-16,    8.7728966107265411e-18,    6.5541909217832192e-19,    4.6902779950072165e-20,
    3.2206727542415681e-21,    2.1255363118551544e-22,    1.3502579548881393e-23,    8.2679210542241228e-25,
    4.8861857096573175e-26,    2.7903733204390460e-27,    1.5415712441232282e-28,    8.2477023468276879e-30,
    4.2776114304935327e-31,    2.1526518955510466e-32,    1.0520394933764317e-33,    4.9973084808177648e-35,
    2.3090136780255486e-36,    1.0385448395716450e-37,    4.5502766091777162e-39,    1.9433635163664396e-40,
    8.0956093863317230e-42,    3.2914444940451267e-43,    1.3068173143988440e-44,    5.0695909711103330e-46,
    1.9225969828705352e-47,    7.1314600015254862e-49,    2.5885209213661916e-50,    9.1982948078607462e-52,
    3.2013747781116348e-53,    1.0917486061085158e-54,    3.6495708707155213e-56,    1.1963636757708779e-57,
    3.8472433663960212e-59,    1.2141092869070744e-60,    3.7613042937411914e-62,    1.1442927716019707e-63,
    3.4197479477388177e-65,    1.0042565295254310e-66,    2.8988088209311623e-68,    8.2270689541373213e-70
},
// x = 3.3900
{
    1.4732503320083778e-01,    1.0420150567797112e-01,    5.5111311361925298e-02,    2.2916385675131434e-02,
    7.7913409412998990e-03,    2.2314097247777371e-03,    5.5077841252258264e-04,    1.1928011923390974e-04,
    2.2990274319442211e-05,    3.9896580449665747e-06,    6.2935902316936262e-07,    9.0973830643089287e-08,
    1.2132443584980918e-08,    1.5015328187462620e-09,    1.7333263921422865e-10,    1.8746229597993028e-11,
    1.9069408255019335e-12,    1.8308881877065660e-13,    1.6643286571851468e-14,    1.4364284584135234e-15,
    1.1800342196137911e-16,    9.2484287921233437e-18,    6.9296825597974640e-19,    4.9735128675387734e-20,
    3.4251740495920013e-21,    2.2671320930864085e-22,    1.4444352028378022e-23,    8.8705665258319460e-25,
    5.2577428945666307e-26,    3.0113871250887526e-27,    1.6685656211576661e-28,    8.9534115012090142e-30,
    4.6572890745752084e-31,    2.3506195976315904e-32,    1.1521728223687360e-33,    5.4890742261988831e-35,
    2.5437071471417545e-36,    1.1474764802179705e-37,    5.0423678984579902e-39,    2.1598779133533401e-40,
    9.0240885785245420e-42,    3.6797574226977777e-43,    1.4653000827923812e-44,    5.7011679086194660e-46,
    2.1684956245539789e-47,    8.0673013833574887e-49,    2.9368461305813413e-50,    1.0466865833454098e-51,
    3.6536403803418999e-53,    1.2496603567243578e-54,    4.1897810583783259e-56,    1.3775045231446541e-57,
    4.4428317146296689e-59,    1.4062044540486778e-60,    4.3692785468201488e-62,    1.3331808823182244e-63,
    3.9960117958731220e-65,    1.1769503605168893e-66,    3.4073280429660322e-68,    9.6988551091409194e-70
},
// x = 3.4000
{
    1.4689503306649340e-01,    1.0401819250218500e-01,    5.5114274976330202e-02,    2.2967788125228844e-02,
    7.8276523655648918e-03,    2.2475318634394251e-03,    5.5622574855498962e-04,    1.2078635425858326e-04,
    2.3344773884769027e-05,    4.0624848347381294e-06,    6.4265274946771520e-07,    9.3159029202239949e-08,
    1.2459316629032993e-08,    1.5464069299385821e-09,    1.7902630305014141e-10,    1.9417874510904464e-11,
    1.9809766271889562e-12,    1.9074842348223217e-13,    1.7389914871860882e-14,    1.5052322296285027e-15,
    1.2401576729864024e-16,    9.7479769096065225e-18,    7.3252991243994967e-19,    5.2728068489554867e-20,
    3.6419068490440427e-21,    2.4176390039068057e-22,    1.5448343183836463e-23,    9.5149193675903827e-25,
    5.6561853910844550e-26,    3.2490917831152260e-27,    1.8055532149210376e-28,    9.7168975215986450e-30,
    5.0692623895233809e-31,    2.5660600451009832e-32,    1.2614653589087509e-33,    6.0274049626367888e-35,
    2.8013814169518086e-36,    1.2674273299071571e-37,    5.5858362742587836e-39,    2.3997030897266653e-40,
    1.0055565776236400e-41,    4.1124195056409758e-43,    1.6424041877546645e-44,    6.4090362543144169e-46,
    2.4449109153869837e-47,    9.1223875661379794e-49,    3.3307138618193589e-50,    1.1905532337965314e-51,
    4.1680615055205920e-53,    1.4298042803629718e-54,    4.8078656401741868e-56,    1.5853696053451787e-57,
    5.1283063343830527e-59,    1.6279432562942945e-60,    5.0731454568868191e-62,    1.5525068805794321e-63,
    4.6671114657336123e-65,    1.3786581444367783e-66,    4.0030360209825550e-68,    1.1428078045455564e-69
},
// x = 3.4100
{
    1.4646748960083877e-01,    1.0383529937252228e-01,    5.5116786340555259e-02,    2.3018897113643608e-02,
    7.8639183653337590e-03,    2.2636873517656621e-03,    5.6170110157356126e-04,    1.2230485309812675e-04,
    2.3703214045144212e-05,    4.1363373305456825e-06,    6.5617320046151526e-07,    9.5388002190309127e-08,
    1.2793713547406722e-08,    1.5924483236909589e-09,    1.8488517800616871e-10,    2.0111032729406901e-11,
    2.0576077388332359e-12,    1.9869977295622328e-13,    1.8167253945314208e-14,    1.5770761888843788e-15,
    1.3031219561017313e-16,    1.0272663952678317e-17,    7.7420441804178939e-19,    5.5890107552067673e-20,
    3.8715573534665663e-21,    2.5775848465953048e-22,    1.6518433338690722e-23,    1.0203711842787414e-24,
    6.0833592259417935e-26,    3.5046861007876688e-27,    1.9532834549641211e-28,    1.0542676951555929e-29,
    5.5161589288606566e-31,    2.8004506513627309e-32,    1.3807209174350799e-33,    6.6165369046529183e-35,
    3.0842012170239436e-36,    1.3994709850919065e-37,    6.1858539185803425e-39,    2.6652615415063082e-40,
    1.1201080486547573e-41,    4.5943291592308996e-43,    1.8402474343336447e-44,    7.2021235018710949e-46,
    2.7555145014306204e-47,    1.0311460014049318e-48,    3.7759053060229392e-50,    1.3536452168579964e-51,
    4.7529423423026395e-53,    1.6352250592982083e-54,    5.5147507917116365e-56,    1.8237984100113409e-57,
    5.9168846392577486e-59,    1.8837850199520640e-60,    5.8876558304497203e-62,    1.8070565531810502e-63,
    5.4482819718135777e-65,    1.6141402340285984e-66,    4.7005374062827460e-68,    1.3458746881843995e-69
},
// x = 3.4200
{
    1.4604238254391233e-01,    1.0365282723856946e-01,    5.5118849878500527e-02,    2.3069713381112470e-02,
    7.9001382797323130e-03,    2.2798758028695475e-03,    5.6720441085365109e-04,    1.2383564465391462e-04,
    2.4065618511920328e-05,    4.2112251502171127e-06,    6.6992323293636277e-07,    9.7661439204358644e-08,
    1.3135776298862834e-08,    1.6396826337239055e-09,    1.9091340104253051e-10,    2.0826309094261301e-11,
    2.1369151003958844e-12,    2.0695286237117394e-13,    1.8976450398491054e-14,    1.6520832062123848e-15,
    1.3690506449017092e-16,    1.0823661154780202e-17,    8.1809675463028028e-19,    5.9230172802830155e-20,
    4.1148476908014413e-21,    2.7475267146448991e-22,    1.7658730366065472e-23,    1.0939845049958145e-24,
    6.5412303267278343e-26,    3.7794505411760901e-27,    2.1125591950368280e-28,    1.1435602660109609e-29,
    6.0008102797936491e-31,    3.0553882724023387e-32,    1.5108108713042295e-33,    7.2610759113509387e-35,
    3.3945271348832983e-36,    1.5447816424591537e-37,    6.8480944026981190e-39,    2.9592184598697103e-40,
    1.2472814110192140e-41,    5.1309074557824451e-43,    2.0611805223052564e-44,    8.0903681816604504e-46,
    3.1044059179490924e-47,    1.1651026763690564e-48,    4.2789135752873658e-50,    1.5384585277552921e-51,
    5.4176648559974666e-53,    1.8693726916834367e-54,    6.3228537559377728e-56,    2.0971672690653326e-57,
    6.8236765889692596e-59,    2.1788426274006154e-60,    6.8297722480358285e-62,    2.1023495178517920e-63,
    6.3571462361581600e-65,    1.8889193550883068e-66,    5.5168257735040913e-68,    1.5842220474324721e-69
},
// x = 3.4300
{
    1.4561969184655571e-01,    1.0347077700284493e-01,    5.5120469978178224e-02,    2.3120237676054547e-02,
    7.9363114556178851e-03,    2.2960968304099601e-03,    5.7273561465299853e-04,    1.2537875738110395e-04,
    2.4432010945546697e-05,    4.2871579425343841e-06,    6.8390572451074619e-07,    9.9980037366547646e-08,
    1.3485648875004271e-08,    1.6881359452336615e-09,    1.9711519532239216e-10,    2.1564322974369287e-11,
    2.2189818514161600e-12,    2.1551798989892220e-13,    1.9818689182261601e-14,    1.7303806442052189e-15,
    1.4380722191072378e-16,    1.1402189878782670e-17,    8.6431672481847089e-19,    6.2757628977670815e-20,
    4.3725376608821159e-21,    2.9280525078344504e-22,    1.8873582177831414e-23,    1.1726398723474041e-24,
    7.0318918616476483e-26,    4.0747524817603926e-27,    2.2842403225982480e-28,    1.2400887635228235e-29,
    6.5262671481622282e-31,    3.3325984191954723e-32,    1.6526795797891635e-33,    7.9660283659618372e-35,
    3.7349326075587491e-36,    1.7046431507348647e-37,    7.5787794212918279e-39,    3.2845051387408918e-40,
    1.3884203783648477e-41,    5.7281519317771801e-43,    2.3078117831982182e-44,    9.0848305571052307e-46,
    3.4961608420804100e-47,    1.3159567450524291e-48,    4.8470286760075606e-50,    1.7478037442564028e-51,
    6.1728245685426536e-53,    2.1361549017748847e-54,    7.2462806209769721e-56,    2.4104623618405834e-57,
    7.8659483944633611e-59,    2.5189757803693724e-60,    7.9189921449068659e-62,    2.4447488975131150e-63,
    7.4140800685108560e-65,    2.2093997238222492e-66,    6.4716650251803978e-68,    1.8638405974502499e-69
},
// x = 3.4400
{
    1.4519939766785250e-01,    1.0328914952172526e-01,    5.5121650991929325e-02,    2.3170470754386094e-02,
    7.9724372475389934e-03,    2.3123500486154900e-03,    5.7829465022201753e-04,    1.2693421928809833e-04,
    2.4802414954146843e-05,    4.3641453867912681e-06,    6.9812357361367611e-07,    1.0234450135894213e-07,
    1.3843477318423635e-08,    1.7378348006308026e-09,    2.0349487161210113e-10,    2.2325708552043696e-11,
    2.3038933814746990e-12,    2.2440576464106028e-13,    2.0695194719726729e-14,    1.8121005044644963e-15,
    1.5103202376296879e-16,    1.2009523568647090e-17,    9.1297915488009190e-19,    6.6482298413342073e-20,
    4.6454265582665463e-21,    3.1197825198720114e-22,    2.0167589851646752e-23,    1.2566641565969573e-24,
    7.5575720009353049e-26,    4.3920517908174460e-27,    2.4692475986776440e-28,    1.3444130371623755e-29,
    7.0958155023620220e-31,    3.6339451462963994e-32,    1.8073502307987011e-33,    8.7368345199343247e-35,
    4.1082223238860400e-36,    1.8804588431962236e-37,    8.3847297082261851e-39,    3.6443445525720210e-40,
    1.5450067145088971e-41,    6.3926957342117178e-43,    2.5830344520002306e-44,    1.0197815025101746e-45,
    3.9358846053124539e-47,    1.4857761357310074e-48,    5.4884323031043284e-50,    1.9848444847793391e-51,
    7.0303829288262941e-53,    2.4399961277391097e-54,    8.3010495771110726e-56,    2.7693623663818448e-57,
    9.0634219561328707e-59,    2.9108972133803487e-60,    9.1777168393395835e-62,    2.8415869617037921e-63,
    8.6426315163066340e-65,    2.5830043123654103e-66,    7.5880302013443717e-68,    2.1917148151447428e-69
},
// x = 3.4500
{
    1.4478148037260166e-01,    1.0310794560635436e-01,    5.5122397236641321e-02,    2.3220413379337938e-02,
    8.0085150176947961e-03,    2.3286350723080366e-03,    5.8388145381410160e-04,    1.2850205793605904e-04,
    2.5176854092105923e-05,    4.4421971923487501e-06,    7.1257969946062870e-07,    1.0475554345796452e-07,
    1.4209409740865500e-08,    1.7888062053160220e-09,    2.1005682969662814e-10,    2.3111115112480927e-11,
    2.3917373815821736e-12,    2.3362711473839414e-13,    2.1607232062232746e-14,    1.8973795782169012e-15,
    1.5859335195473652e-16,    1.2646989769308385e-17,    9.6420410538548488e-19,    7.0414481671634185e-20,
    4.9343550762656292e-21,    3.3233711018032013e-22,    2.1545621426113231e-23,    1.3464042139429645e-24,
    8.1206421225399815e-26,    4.7329067407062386e-27,    2.6685667419174060e-28,    1.4571341953721037e-29,
    7.7129938466061237e-31,    3.9614416637044274e-32,    1.9759311296076849e-33,    9.5794044890577340e-35,
    4.5174521494275280e-36,    2.0737622153101036e-37,    9.2734204925243949e-39,    4.0422792974114343e-40,
    1.7186739031541737e-41,    7.1318726146806337e-43,    2.8900567237611370e-44,    1.1443005414062302e-45,
    4.4292715193397576e-47,    1.6768740982926351e-48,    6.2123035533878388e-50,    2.2531404228703427e-51,
    8.0038382376191220e-53,    2.7859038875751151e-54,    9.5053428383600937e-56,    3.1803319972318099e-57,
    1.0438614740485088e-58,    3.3622936059144398e-60,    1.0631672866386135e-61,    3.3013089982430679e-63,
    1.0070002473660743e-64,    3.0183329570834921e-66,    8.8926167157693520e-68,    2.5759839095397321e-69
},
// x = 3.4600
{
    1.4436592052882496e-01,    1.0292716602353776e-01,    5.5122712993965650e-02,    2.3270066321275273e-02,
    8.0445441358942688e-03,    2.3449515169260251e-03,    5.8949596069592135e-04,    1.3008230043845990e-04,
    2.5555351858667109e-05,    4.5213230981880040e-06,    7.2727704202778866e-07,    1.0721388356847390e-07,
    1.4583596341401389e-08,    1.8410776334927894e-09,    2.1680555981025536e-10,    2.3921207337470156e-11,
    2.4826038965053293e-12,    2.4319329565632030e-13,    2.2556108074343761e-14,    1.9863596011992996e-15,
    1.6650563307997975e-16,    1.3315972216880606e-17,    1.0181170898449152e-18,    7.4564979013214861e-20,
    5.2402072954645799e-21,    3.5395084045057051e-22,    2.3012826395477223e-23,    1.4422280343007562e-24,
    8.7236254858276185e-26,    5.0989802771310026e-27,    2.8832527714063477e-28,    1.5788974940618793e-29,
    8.3816116983004033e-31,    4.3172617222084252e-32,    2.1596224659805719e-33,    1.0500157102662594e-34,
    4.9659506943184469e-36,    2.2862285169904639e-37,    1.0253041882473200e-38,    4.4822021048091573e-40,
    1.9112221203736354e-41,    7.9537883275250521e-43,    3.2324348690705890e-44,    1.2835614490369585e-45,
    4.9826706250576890e-47,    1.8918373891713163e-48,    7.0289367747478573e-50,    2.5566953807065739e-51,
    9.1084173165136073e-53,    3.1795434180775858e-54,    1.0879790800280685e-55,    3.6507278222407113e-57,
    1.2017225402266899e-58,    3.8819631712757176e-60,    1.2310392830429069e-61,    3.8336379902623215e-63,
    1.1727601552835837e-64,    3.5253443939680449e-66,    1.0416428375570331e-67,    3.0261282405071759e-69
},
// x = 3.4700
{
    1.4395269890530815e-01,    1.0274681149662032e-01,    5.5122602510535530e-02,    2.3319430357520127e-02,
    8.0805239795150915e-03,    2.3612989985472778e-03,    5.9513810515772784e-04,    1.3167497346068855e-04,
    2.5937931696538187e-05,    4.6015328724611660e-06,    7.4221856202460923e-07,    1.0972024925747721e-07,
    1.4966189424616313e-08,    1.8946770340167449e-09,    2.2374564408256500e-10,    2.4756665603377183e-11,
    2.5765853780426205e-12,    2.5311589864908768e-13,    2.3543172648365368e-14,    2.0791874129152144e-15,
    1.7478385767563734e-16,    1.4017913001344196e-17,    1.0748493016313789e-18,    7.8945112752832859e-20,
    5.5639127601558750e-21,    3.7689222037236589e-22,    2.4574650936664146e-23,    1.5445259506946612e-24,
    9.3692063982188854e-26,    5.4920466644705221e-27,    3.1144346237304935e-28,    1.7103954166196592e-29,
    9.1057693489769607e-31,    4.7037518257056400e-32,    2.3597235943011890e-33,    1.1506061821433527e-34,
    5.4573426534068268e-36,    2.5196873344224517e-37,    1.1332564597499864e-38,    4.9683891559103867e-40,
    2.1246346291781379e-41,    8.8673990392292483e-43,    3.6141097083183097e-44,    1.4392549112349639e-45,
    5.6031585361246348e-47,    2.1335575891641119e-48,    7.9498728988645346e-50,    2.9000110822641549e-51,
    1.0361290360360763e-52,    3.6273215869388972e-54,    1.2447792437111336e-55,    4.1889179237802599e-57,
    1.3828571135358660e-58,    4.4799721587629456e-60,    1.4247763957065658e-61,    4.4497630276270772e-63,
    1.3651678489127026e-64,    4.1155657499750695e-66,    1.2195456068444417e-67,    3.5531851105254625e-69
},
// x = 3.4800
{
    1.4354179646917517e-01,    1.0256688270634920e-01,    5.5122069998184448e-02,    2.3368506272176140e-02,
    8.1164539334623292e-03,    2.3776771339115072e-03,    6.0080782052365561e-04,    1.3328010321969036e-04,
    2.6324616990507304e-05,    4.6828363120398062e-06,    7.5740724086467394e-07,    1.1227537578746198e-07,
    1.5357343418804330e-08,    1.9496328362814781e-09,    2.3088175799976482e-10,    2.5618186283437150e-11,
    2.6737767392614739e-12,    2.6340685940592415e-13,    2.4569819949019984e-14,    2.1760151203667812e-15,
    1.8344360008192012e-16,    1.4754314803929080e-17,    1.1345378494629742e-18,    8.3566750528559185e-20,
    5.9064486462260665e-21,    4.0123798112323185e-22,    2.6236853902838624e-23,    1.6537119133332109e-24,
    1.0060239900914088e-25,    5.9139985283177291e-27,    3.3633200605290968e-28,    1.8523709574198439e-29,
    9.8898789931713722e-31,    5.1234443274895171e-32,    2.5776408636957480e-33,    1.2604683954845370e-34,
    5.9955740577576822e-36,    2.7761362422628099e-37,    1.2521811501630212e-38,    5.5055364423303231e-40,
    2.3610957259651024e-41,    9.8825974115527187e-43,    4.0394467729313229e-44,    1.6132592611260494e-45,
    6.2986201162012283e-47,    2.4052658906781481e-48,    8.9880457497480614e-50,    3.2881472109937492e-51,
    1.1781811690411221e-52,    4.1364811952199600e-54,    1.4235876423402852e-55,    4.8044171633861163e-57,
    1.5906083495666030e-58,    5.1678327972268087e-60,    1.6482653618077668e-61,    5.1625547829438904e-63,
    1.5884051791127462e-64,    4.8023335260624092e-66,    1.4271460748812988e-67,    4.1699984442496751e-69
},
// x = 3.4900
{
    1.4313319438349498e-01,    1.0238738029172134e-01,    5.5121119634164328e-02,    2.3417294855955818e-02,
    8.1523333901268452e-03,    2.3940855404424757e-03,    6.0650503916202130e-04,    1.3489771548365819e-04,
    2.6715431066069014e-05,    4.7652432420613215e-06,    7.7284608063489324e-07,    1.1488000614935293e-07,
    1.5757214894172441e-08,    2.0059739561405818e-09,    2.3821867188140669e-10,    2.6506482054394294e-11,
    2.7742754097096253e-12,    2.7407846688206806e-13,    2.5637489688886259e-14,    2.2770002663685609e-15,
    1.9250103892239644e-16,    1.5526743211467762e-17,    1.1973260017333737e-18,    8.8442329518832444e-20,
    6.2688420241683282e-21,    4.2706900758660058e-22,    2.8005523619151238e-23,    1.7702248315817800e-24,
    1.0799762001144556e-25,    6.3668543174833248e-27,    3.6312008837501395e-28,    2.0056211215173190e-29,
    1.0738687314862918e-30,    5.5790714712174726e-32,    2.8148960376656582e-33,    1.3804233425766992e-34,
    6.5849395869892705e-36,    3.0577556133259705e-37,    1.3831535429166012e-38,    6.0987994412059415e-40,
    2.6230103800118480e-41,    1.1010307081026821e-42,    4.5132805126331356e-44,    1.8076606132363660e-45,
    7.0778378032266113e-47,    2.7105717258550150e-48,    1.0157944980012051e-49,    3.7267884879072166e-51,
    1.3391789433662833e-52,    4.7152069178334402e-54,    1.6274108003513788e-55,    5.5080400257224481e-57,
    1.8287870292808724e-58,    5.9587055364431155e-60,    1.9059622338417946e-61,    5.9868118369421018e-63,
    1.8472942986049251e-64,    5.6010706854648794e-66,    1.6692876351224351e-67,    4.8915075472630137e-69
},
// x = 3.5000
{
    1.4272687400492079e-01,    1.0220830485081614e-01,    5.5119755561364100e-02,    2.3465796906010274e-02,
    8.1881617493435276e-03,    2.4105238362697886e-03,    6.1222969249562177e-04,    1.3652783557176442e-04,
    2.7110397188059978e-05,    4.8487635154731271e-06,    7.8853810406300750e-07,    1.1753489109508052e-07,
    1.6165962581050082e-08,    2.0637298018656097e-09,    2.4576125237252768e-10,    2.7422282207522668e-11,
    2.8781813916126424e-12,    2.8514337231774228e-13,    2.6747668435220768e-14,    2.3823060025512982e-15,
    2.0197297822059440e-16,    1.6336829110049403e-17,    1.2633634399872307e-18,    9.3584881642164316e-20,
    6.6521722210239617e-21,    4.5447054782885110e-22,    2.9887095517845283e-23,    1.8945299871939598e-24,
    1.1591000479732969e-25,    6.8527662088778135e-27,    3.9194584767514411e-28,    2.1710006539582287e-29,
    1.1657299626630098e-30,    6.0735804412111936e-32,    3.0731353454381427e-33,    1.5113617347426563e-34,
    7.2301121030418947e-36,    3.3669246796308697e-37,    1.5273503832893519e-38,    6.7538363942968283e-40,
    2.9130257194949445e-41,    1.2262586322816142e-42,    5.0409629414031495e-44,    2.0247750837065175e-45,
    7.9505904755212182e-47,    3.0535056454070662e-48,    1.1475797462847978e-49,    4.2223195646033464e-51,
    1.5215787496030745e-52,    5.3727442748274318e-54,    1.8596547233152564e-55,    6.3120732605498758e-57,
    2.1017352105791291e-58,    6.8676288124866828e-60,    2.2029736189145036e-61,    6.9395421529425005e-63,
    2.1473932670249644e-64,    6.5296051190416348e-66,    1.9515850541417165e-67,    5.7350809107583638e-69
},
// x = 3.5100
{
    1.4232281688136056e-01,    1.0202965694161364e-01,    5.5117981888528372e-02,    2.3514013225761233e-02,
    8.2239384183492581e-03,    2.4269916402503353e-03,    6.1798171101202744e-04,    1.3817048835393708e-04,
    2.7509538559304788e-05,    4.9334070125748936e-06,    8.0448635448342831e-07,    1.2024078916976488e-07,
    1.6583747388103370e-08,    2.1229302801397727e-09,    2.5351446395126668e-10,    2.8366332964065475e-11,
    2.9855973170702030e-12,    2.9661459844817556e-13,    2.7901890948793089e-14,    2.4921012671660564e-15,
    2.1187686917024299e-16,    1.7186271160369111e-17,    1.3328065218463005e-18,    9.9008059775520929e-20,
    7.0575732851942158e-21,    4.8353243235382352e-22,    3.1888370651482789e-23,    2.0271205223168866e-24,
    1.2437386304154041e-25,    7.3740284799064036e-27,    4.2295696903963726e-28,    2.3494260129231387e-29,
    1.2651205662533938e-30,    6.6101494909282501e-32,    3.3541392101090534e-33,    1.6542496696780765e-34,
    7.9361745779037692e-36,    3.7062389459515192e-37,    1.6860590827878258e-38,    7.4768555052935793e-40,
    3.2340545308654206e-41,    1.3652741757995117e-42,    5.6284171514770462e-44,    2.2671733050692804e-45,
    8.9277628437903601e-47,    3.4385669002138549e-48,    1.2959769161849414e-49,    4.7819086127303092e-51,
    1.7281463576489060e-52,    6.1195331877496175e-54,    2.1241764887571493e-55,    7.2304708130930062e-57,
    2.4143983053550791e-58,    7.9117799795219374e-60,    2.5451492045169828e-61,    8.0402845831605279e-63,
    2.4951055687837100e-64,    7.6085355092602552e-66,    2.2805443822029377e-67,    6.7209019171314239e-69
},
// x = 3.5200
{
    1.4192100474967922e-01,    1.0185143708279829e-01,    5.5115802690476154e-02,    2.3561944624735525e-02,
    8.2596628117407155e-03,    2.4434885719893914e-03,    6.2376102427386668e-04,    1.3982569825067825e-04,
    2.7912878319271815e-05,    5.0191836405587284e-06,    8.2069389580140849e-07,    1.2299846674350508e-07,
    1.7010732420551660e-08,    2.1836058020870685e-09,    2.6148337045198729e-10,    2.9339397795127370e-11,
    3.0966285062634219e-12,    3.0850554890778113e-13,    2.9101741555371096e-14,    2.6065609678009371e-15,
    2.2223083257663080e-16,    1.8076838357225892e-17,    1.4058185537007704e-18,    1.0472616502854962e-19,
    7.4862365582037911e-21,    5.1434930355369552e-22,    3.4016535124680761e-23,    2.1685190059451668e-24,
    1.3342565678753509e-25,    7.9330863742866762e-27,    4.5631130943455214e-28,    2.5418796017447336e-29,
    1.3726307131933726e-30,    7.1922052228816195e-32,    3.6598327017008683e-33,    1.8101347388440876e-34,
    8.7086546005794924e-36,    4.0785290648160140e-37,    1.8606877249921354e-38,    8.2746663957192600e-40,
    3.5893009528678202e-41,    1.5195453040475991e-42,    6.2821961646751555e-44,    2.5377074641091010e-45,
    1.0021466450960780e-46,    3.8707762241302692e-48,    1.4630189714914392e-49,    5.4136005875457991e-51,
    1.9619947390218231e-52,    6.9673578549828906e-54,    2.4253423078844475e-55,    8.2790738371743411e-57,
    2.7724066658197646e-58,    9.1107715187908662e-60,    2.9391870964038238e-61,    9.3114759494826171e-63,
    2.8978055158417530e-64,    8.8616514678886234e-66,    2.6637010485954459e-67,    7.8724143185377167e-69
},
// x = 3.5300
{
    1.4152141953343139e-01,    1.0167364575454838e-01,    5.5113222008319453e-02,    2.3609591918401823e-02,
    8.2953343514319804e-03,    2.4600142518613768e-03,    6.2956756092911006e-04,    1.4149348923292584e-04,
    2.8320439542739465e-05,    5.1061033330474841e-06,    8.3716381245555340e-07,    1.2580869804277405e-07,
    1.7447082998385374e-08,    2.2457872893366087e-09,    2.6967313660395327e-10,    3.0342257746056592e-11,
    3.2113830266858774e-12,    3.2083001783165339e-13,    3.0348855550506099e-14,    2.7258661691249353e-15,
    2.3305368198697189e-16,    1.9010372675688183e-17,    1.4825700734896409e-18,    1.1075417511204140e-19,
    7.9394133576441252e-21,    5.4702085579150157e-22,    3.6279180486456454e-23,    2.3192790826650104e-24,
    1.4310412765320956e-25,    8.5325454885387542e-27,    4.9217756148539450e-28,    2.7494142757146026e-29,
    1.4888947147994814e-30,    7.8234410979946421e-32,    3.9922967665039333e-33,    1.9801526074776275e-34,
    9.5535616622484255e-36,    4.4868812902752585e-37,    2.0527759397308511e-38,    9.1547361881335595e-40,
    3.9822885621509023e-41,    1.6906909543668857e-42,    7.0095476338651315e-44,    2.8395411132411826e-45,
    1.1245173469016018e-46,    4.3557343674554934e-48,    1.6509802204416152e-49,    6.1264212495371627e-51,
    2.2266263735403124e-52,    7.9295148796479094e-54,    2.7680928458564719e-55,    9.4758589254262291e-57,
    3.1821678992448779e-58,    1.0486987162422525e-59,    3.3927536947058782e-61,    1.0778869987954672e-62,
    3.3639817796219565e-64,    1.0316415793733225e-65,    3.1097788306791880e-67,    9.2168365245845199e-69
},
// x = 3.5400
{
    1.4112404334062451e-01,    1.0149628339931206e-01,    5.5110243849682057e-02,    2.3656955928009739e-02,
    8.3309524666119374e-03,    2.4765683010302486e-03,    6.3540124872133483e-04,    1.4317388482195558e-04,
    2.8732245238472266e-05,    5.1941760496311226e-06,    8.5389920937866532e-07,    1.2867226518141099e-07,
    1.7892966674582554e-08,    2.3095061801218244e-09,    2.7808902958558338e-10,    3.1375711765351316e-11,
    3.3299717534107835e-12,    3.3360219965756249e-13,    3.1644920638273697e-14,    2.8502042857753742e-15,
    2.4436494752804350e-16,    1.9988791806508290e-17,    1.5632391439032203e-18,    1.1710777384024492e-19,
    8.4184177756738476e-21,    5.8165208656736662e-22,    3.8684325127041667e-23,    2.4799872077031914e-24,
    1.5345043108813493e-25,    9.1751817077988203e-27,    5.3073595815465024e-28,    2.9731581405123887e-29,
    1.6145941651573480e-30,    8.5078372573719290e-32,    4.3537802875193231e-33,    2.1655341020711219e-34,
    1.0477427433174884e-35,    4.9346596367506197e-37,    2.2640067177814063e-38,    1.0125250615921907e-39,
    4.4168910643708979e-41,    1.8804960157991724e-42,    7.8184849547572001e-44,    3.1761820297883219e-45,
    1.2613864599458782e-46,    4.8996869865373542e-48,    1.8624040845814041e-49,    6.9304931455279422e-51,
    2.5259805555440199e-52,    9.0210018056307447e-54,    3.1580166811843311e-55,    1.0841218070926303e-56,
    3.6509712820462222e-58,    1.2065963162083539e-59,    3.9146200605180803e-61,    1.2472015290587898e-62,
    3.9034016049259774e-64,    1.2004517801583795e-65,    3.6288727730486165e-67,    1.0785755067188511e-68
},
// x = 3.5500
{
    1.4072885846151184e-01,    1.0131935042256933e-01,    5.5106872188918049e-02,    2.3704037480431223e-02,
    8.3665165937015402e-03,    2.4931503414695896e-03,    6.4126201449999482e-04,    1.4486690808932721e-04,
    2.9148318347908059e-05,    5.2834117754013887e-06,    8.7090321195695792e-07,    1.3158995819121317e-07,
    1.8348553253323428e-08,    2.3747944354143813e-09,    2.8673642059433461e-10,    3.2440577038127188e-11,
    3.4525084304070911e-12,    3.4683669913167748e-13,    3.2991678404639254e-14,    2.9797692805077358e-15,
    2.5618490046978699e-16,    2.1014091983434858e-17,    1.6480116563505687e-18,    1.2380338180793626e-19,
    8.9246295976097694e-21,    6.1835355923806687e-22,    4.1240436724865036e-23,    2.6512644724764347e-24,
    1.6450827804704421e-25,    9.8639517210774394e-27,    5.7217902068676676e-28,    3.2143196600599314e-29,
    1.7504612958490551e-30,    9.2496817447637197e-32,    4.7467130344925148e-33,    2.3676128426159740e-34,
    1.1487349260570616e-35,    5.4255298789150084e-37,    2.4962192440320590e-38,    1.1195180591951917e-39,
    4.8973658230403350e-41,    2.0909277409185293e-42,    8.7178653998314459e-44,    3.5515184236760532e-45,
    1.4144192512667790e-46,    5.5095965565212788e-48,    2.1001339613252583e-49,    7.8371648804617150e-51,
    2.8644862721229052e-52,    1.0258728463921596e-53,    3.6014328884315937e-55,    1.2398274299312275e-56,
    4.1871058142318465e-58,    1.3876820598358630e-59,    4.5148169800910265e-61,    1.4424800332668839e-62,
    4.5272986086569638e-64,    1.3962507929774548e-65,    4.2326595750662175e-67,    1.2615809133779430e-68
},
// x = 3.5600
{
    1.4033584736641516e-01,    1.0114284719358059e-01,    5.5103110967330407e-02,    2.3750837408004193e-02,
    8.4020261763108808e-03,    2.5097599959823243e-03,    6.4714978423066323e-04,    1.4657258165686919e-04,
    2.9568681743854663e-05,    5.3738205204845655e-06,    8.8817896598759643e-07,    1.3456257505211527e-07,
    1.8814014808200067e-08,    2.4416845450923511e-09,    2.9562078643223065e-10,    3.3537689324179260e-11,
    3.5791097329170453e-12,    3.6054854152124825e-13,    3.4390925826122425e-14,    3.1147618677285896e-15,
    2.6853457853393890e-16,    2.2088350905137299e-17,    1.7370816460443608e-18,    1.3085818828443949e-19,
    9.4594973453003346e-21,    6.5724167777749356e-22,    4.3956455791297940e-23,    2.8337685250243057e-24,
    1.7632408446177186e-25,    1.0602004147621253e-26,    6.1671235231859422e-28,    3.4741930926233633e-29,
    1.8972825565943256e-30,    1.0053593223613572e-31,    5.1737195659283782e-33,    2.5878334595085038e-34,
    1.2591037133331173e-35,    5.9634855389062053e-37,    2.7514228332155747e-38,    1.2374354703980924e-39,
    5.4283904782330802e-41,    2.3241537214644552e-42,    9.7174759423588148e-44,    3.9698588226029199e-45,
    1.5854662402244421e-46,    6.1932220418138323e-48,    2.3673475136029842e-49,    8.8591551543545321e-51,
    3.2471212892737083e-52,    1.1661753805384423e-53,    4.1054838437701312e-55,    1.4173237384895689e-56,
    4.7999936458184675e-58,    1.5952755374922501e-59,    5.2048112209729506e-61,    1.6676074752519215e-62,
    5.2485874627466369e-64,    1.6232525261625752e-65,    4.9346394686577234e-67,    1.4749479801211457e-68
},
// x = 3.5700
{
    1.3994499270357610e-01,    1.0096677404612246e-01,    5.5098964093389199e-02,    2.3797356548378445e-02,
    8.4374806651961402e-03,    2.5263968882201253e-03,    6.5306448300527723e-04,    1.4829092769670684e-04,
    2.9993358229198176e-05,    5.4654123195727286e-06,    9.0572963763463012e-07,    1.3759092172196214e-07,
    1.9289525700420640e-08,    2.5102095341425688e-09,    3.0474771110708823e-10,    3.4667903300676236e-11,
    3.7098953309081806e-12,    3.7475318303756069e-13,    3.5844516814449795e-14,    3.2553897225349448e-15,
    2.8143581196722169e-16,    2.3213730754527912e-17,    1.8306516185660078e-18,    1.3829018436816269e-19,
    1.0024541450139995e-20,    6.9843897408440757e-22,    4.6841820362742911e-23,    3.0281955899050161e-24,
    1.8894712891215071e-25,    1.1392691307650709e-26,    6.6455548038841457e-28,    3.7541642750625529e-29,
    2.0559024361989635e-30,    1.0924545288531536e-31,    5.6376341496272949e-33,    2.8277604377946398e-34,
    1.3796864377454302e-35,    6.5528760182675686e-37,    3.0318120589667954e-38,    1.3673538143108670e-39,
    6.0151029287417771e-41,    2.5825615711366987e-42,    1.0828127499595603e-43,    4.4359759948865915e-45,
    1.7765831384584660e-46,    6.9592071329917190e-48,    2.6675947546953481e-49,    1.0010713197114942e-50,
    3.6794781521562327e-52,    1.3251551200251542e-53,    4.6782394814353263e-55,    1.6195804594170643e-56,
    5.5003408202681661e-58,    1.8331593381923880e-59,    5.9977057979832014e-61,    1.9270357269297617e-62,
    6.0821092099949622e-64,    1.8863131218305959e-65,    5.7504141821203051e-67,    1.7235999592367707e-68
},
// x = 3.5800
{
    1.3955627729703665e-01,    1.0079113127921009e-01,    5.5094435442949972e-02,    2.3843595744363739e-02,
    8.4728795182163752e-03,    2.5430606427024704e-03,    6.5900603505236060e-04,    1.5002196793132856e-04,
    3.0422370535620820e-05,    5.5581972314531823e-06,    9.2355841338326627e-07,    1.4067581216586145e-07,
    1.9775262597005372e-08,    2.5804029688966739e-09,    3.1412288744944860e-10,    3.5832092909519179e-11,
    3.8449879536123407e-12,    3.8946652147244978e-13,    3.7354363797888980e-14,    3.4018676953849789e-15,
    2.9491125039898989e-16,    2.4392481318333191e-17,    1.9289328882838697e-18,    1.4611819744656950e-19,
    1.0621357560750593e-20,    7.4207440836303746e-22,    4.9906491891681281e-23,    3.2352825923364001e-24,
    2.0242971891538620e-25,    1.2239581672431345e-26,    7.1594274961826386e-28,    4.0557167762595581e-29,
    2.2272275390677470e-30,    1.1867892477337761e-31,    6.1415167726895794e-33,    3.0890876343889826e-34,
    1.5113922365062144e-35,    7.1984370439053133e-37,    3.3397831740969508e-38,    1.5104518612985200e-39,
    6.6631449745192697e-41,    2.8687804715349677e-42,    1.2061758390713270e-43,    4.9551553041624335e-45,
    1.9900528642556602e-46,    7.8171779402610304e-48,    3.0048403345604247e-49,    1.1307797408530337e-50,
    4.1678378833782408e-52,    1.5052305522264422e-53,    5.3288143721560891e-55,    1.8499611991651357e-56,
    6.3003075189688938e-58,    2.1056419256546673e-59,    6.9084674311995514e-61,    2.2258642000568033e-62,
    7.0449114703576015e-64,    2.1910264523081342e-65,    6.6980062392936915e-67,    2.0132400250008037e-68
},
// x = 3.5900
{
    1.3916968414454725e-01,    1.0061591915780729e-01,    5.5089528859471507e-02,    2.3889555843780125e-02,
    8.5082222002901848e-03,    2.5597508848353807e-03,    6.6497436374723517e-04,    1.5176572363369381e-04,
    3.0855741322330306e-05,    5.6521853385364504e-06,    9.4166849999253605e-07,    1.4381806838512135e-07,
    2.0271404488973449e-08,    2.6522989633007083e-09,    3.2375211874526863e-10,    3.7031151709402345e-11,
    3.9845134551648058e-12,    4.0470490705174722e-13,    3.8922439339972888e-14,    3.5544180325280596e-15,
    3.0898439050371707e-16,    2.5626943209841949e-17,    2.0321459290083133e-18,    1.5436192700795377e-19,
    1.1251619990533575e-20,    7.8828368312223263e-22,    5.3160982390434979e-23,    3.4558093915770931e-24,
    2.1682736627337914e-25,    1.3146473030405933e-26,    7.7112426949329820e-28,    4.3804384419521344e-29,
    2.4122309334525722e-30,    1.2889398096504569e-31,    6.6886703166215909e-33,    3.3736485170593713e-34,
    1.6552079540387464e-35,    7.9053236101050883e-37,    3.6779519276560614e-38,    1.6680199812146531e-39,
    7.3787099413639654e-41,    3.1857047509177082e-42,    1.3431547877336501e-43,    5.5332479271717333e-45,
    2.2284098396646650e-46,    8.7778511228432722e-48,    3.3835104759803525e-49,    1.2768274203362695e-50,
    4.7192522492755728e-52,    1.7091245710037515e-53,    6.0674991538067704e-55,    2.1122742503144879e-56,
    7.2137002582430863e-58,    2.4176288227759809e-59,    7.9541847882056461e-61,    2.5699316494779773e-62,
    8.1565683712338690e-64,    2.5438334628582783e-65,    7.7982255875745909e-67,    2.3504719217127476e-68
},
// x = 3.6000
{
    1.3878519641550324e-01,    1.0044113791352363e-01,    5.5084248154233531e-02,    2.3935237699310389e-02,
    8.5435081833522258e-03,    2.5764672409298215e-03,    6.7096939162221490e-04,    1.5352221562737976e-04,
    3.1293493174799386e-05,    5.7473867463826838e-06,    9.6006312444632833e-07,    1.4701852044576163e-07,
    2.0778132709518311e-08,    2.7259321852178583e-09,    3.3364132038437577e-10,    3.8265993232608715e-11,
    4.1286008813563277e-12,    4.2048515350902863e-13,    4.0550777796328086e-14,    3.7132706023235981e-15,
    3.2367960448909225e-16,    2.6919551197826307e-17,    2.1405207372778805e-18,    1.6304198185282240e-19,
    1.1917085310474837e-20,    8.3720957135938184e-22,    5.6616382883589235e-23,    3.6906011287626155e-24,
    2.3219897193818870e-25,    1.4117406407961600e-26,    8.3036691881805573e-28,    4.7300283544542015e-29,
    2.6119567885703845e-30,    1.3995263979898561e-31,    7.2826589781523172e-33,    3.6834271773284311e-34,
    1.8122045087915080e-35,    8.6791456123147326e-37,    4.0491728926058556e-38,    1.8414703129984795e-39,
    8.1685946369697104e-41,    3.5365196802953648e-42,    1.4952040733177271e-43,    6.1767294062849402e-45,
    2.4944667988665361e-46,    9.8531535315618063e-48,    3.8085450550791637e-49,    1.4412139273969930e-50,
    5.3416355593197976e-52,    1.9399016913811653e-53,    6.9059080215957301e-55,    2.4108297668060266e-56,
    8.2561887896022035e-58,    2.7747031720501539e-59,    9.1543615642419826e-61,    2.9659206546664401e-62,
    9.4395456871301941e-64,    2.9521473172388082e-65,    9.0750903951119068e-67,    2.7429388274402153e-68
},
// x = 3.6100
{
    1.3840279744890852e-01,    1.0026678774529965e-01,    5.5078597106554245e-02,    2.3980642168354705e-02,
    8.5787369463096765e-03,    2.5932093382197937e-03,    6.7699104037678671e-04,    1.5529146428676694e-04,
    3.1735648603516517e-05,    5.8438115832266017e-06,    9.7874553390281869e-07,    1.5027800650660182e-07,
    2.1295630952170179e-08,    2.8013378627640763e-09,    3.4379652152472992e-10,    3.9537551346576161e-11,
    4.2773825375111180e-12,    4.3682454938308099e-13,    4.2241477010333924e-14,    3.8786631275810156e-15,
    3.3902216943097736e-16,    2.8272837644708961e-17,    2.2542972086821492e-18,    1.7217991875420534e-19,
    1.2619596092771953e-20,    8.8900225951678686e-22,    6.0284393227319689e-23,    3.9405306946410227e-24,
    2.4860702087752648e-25,    1.5156680785343507e-26,    8.9395541069364504e-28,    5.1063042320688766e-29,
    2.8275253187439953e-30,    1.5192162308220354e-31,    7.9273280216564219e-33,    4.0205701730247124e-34,
    1.9835437591200562e-35,    9.5260063830435785e-37,    4.4565604267934250e-38,    2.0323478248738663e-39,
    9.0402560168184216e-41,    3.9247296874610611e-42,    1.6639283874711829e-43,    6.8927640524566392e-45,
    2.7913443583646874e-46,    1.1056354549236382e-47,    4.2854553715832703e-49,    1.6261763717154143e-50,
    6.0438670690196538e-52,    2.2010096796630151e-53,    7.8571441820222824e-55,    2.7505040833741216e-56,
    9.4455507886381239e-58,    3.1832168727281773e-59,    1.0531248972881842e-60,    3.4214764840764218e-62,
    1.0919617411408279e-63,    3.4244965718895989e-65,    1.0556309820915699e-66,    3.1994831163670148e-68
},
// x = 3.6200
{
    1.3802247075136612e-01,    1.0009286882007971e-01,    5.5072579464007239e-02,    2.4025770112887368e-02,
    8.6139079749984956e-03,    2.6099768048800762e-03,    6.8303923088776732e-04,    1.5707348953726482e-04,
    3.2182230042747354e-05,    5.9414699995010690e-06,    9.9771899564228684e-07,    1.5359737284691745e-07,
    2.1824085288943732e-08,    2.8785517906762889e-09,    3.5422386677251017e-10,    4.0846780620268500e-11,
    4.4309940575038745e-12,    4.5374086954254312e-13,    4.3996700048349363e-14,    4.0508414240553382e-15,
    3.5503829747691532e-16,    2.9689436057125177e-17,    2.3737255276381545e-18,    1.8179828261774760e-19,
    1.3361084811044438e-20,    9.4381970582022156e-22,    6.4177353356232744e-23,    4.2065213228911345e-24,
    2.6611778744477191e-25,    1.6268868650251752e-26,    9.6219342133203767e-28,    5.5112102943924468e-29,
    3.0601380537938560e-30,    1.6487269624474283e-31,    8.6268249546933080e-33,    4.3873992600292666e-34,
    2.1704859055804822e-35,    1.0452544355913827e-36,    4.9035113994416684e-38,    2.2423423399781054e-39,
    1.0001872970112870e-40,    4.3541892069934085e-42,    1.8510976177713585e-43,    7.6892757623157090e-45,
    3.1225036236609167e-46,    1.2402212431134190e-47,    4.8203882100561852e-49,    1.8342167730186412e-50,
    6.8359051730105027e-52,    2.4963261069871362e-53,    8.9359853937492205e-55,    3.1368120460548272e-56,
    1.0801947792117875e-57,    3.6503936449456490e-59,    1.2110222803382271e-60,    3.9453422697736784e-62,
    1.2626340816572510e-63,    3.9706889098846618e-65,    1.2273837657148955e-66,    3.7303300879004713e-68
},
// x = 3.6300
{
    1.3764419999509553e-01,    9.9919381273473218e-02,    5.5066198942638339e-02,    2.4070622399315705e-02,
    8.6490207621397162e-03,    2.6267692700436885e-03,    6.8911388321945004e-04,    1.5886831085557454e-04,
    3.2633259849307142e-05,    6.0403721673592895e-06,    1.0169867970133433e-06,    1.5697747389366485e-07,
    2.2363684188469446e-08,    2.9576103367128417e-09,    3.6492961787804698e-10,    4.2194656695385471e-11,
    4.5895744739288701e-12,    4.7125238694118952e-13,    4.5818676975251754e-14,    4.2300596452351405e-15,
    3.7175516694032728e-16,    3.1172084752104199e-17,    2.4990665710487573e-18,    1.9192064819401668e-19,
    1.4143577903085005e-20,    1.0018280146322184e-21,    6.8308276020786200e-23,    4.4895493149543050e-24,
    2.8480155178157055e-25,    1.7458832433776548e-26,    1.0354047863042666e-27,    5.9468256211730955e-29,
    3.3110834560433151e-30,    1.7888303189199461e-31,    9.3856222241312072e-33,    4.7864250768237169e-34,
    2.3743974696389801e-35,    1.1465978100696748e-36,    5.3937298256037724e-38,    2.4733016082682066e-39,
    1.1062413670210070e-40,    4.8291364031496593e-42,    2.0586632705343367e-43,    8.5750258653969360e-45,
    3.4917821334194060e-46,    1.3907136076626168e-47,    5.4201968547194724e-49,    2.0681324865534419e-50,
    7.7289147035772524e-52,    2.8302103909256542e-53,    1.0159091961830420e-54,    3.5759883227006629e-56,
    1.2348236261046255e-57,    4.1844455385222371e-59,    1.3920210856733618e-60,    4.5475126710987311e-62,
    1.4593598001961260e-63,    4.6019983118915007e-65,    1.4264506988187647e-66,    4.3472991698616294e-68
},
// x = 3.6400
{
    1.3726796901597627e-01,    9.9746325210403886e-02,    5.5059459227181869e-02,    2.4115199898340885e-02,
    8.6840748072955221e-03,    2.6435863638189956e-03,    6.9521491663372150e-04,    1.6067594726999019e-04,
    3.3088760301344248e-05,    6.1405282801957222e-06,    1.0365522453775584e-06,    1.6041917224826799e-07,
    2.2914618534106539e-08,    3.0385504480858662e-09,    3.7592015544762930e-10,    4.3582176662446148e-11,
    4.7532662894343331e-12,    4.8937788460738199e-13,    4.7709706671046058e-14,    4.4165805335622170e-15,
    3.8920095430802326e-16,    3.2723630642175176e-17,    2.6305923262833850e-18,    2.0257166339708720e-19,
    1.4969200002313952e-20,    1.0632018274762412e-21,    7.2690881080908799e-23,    4.7906469025690079e-24,
    3.0473282780568786e-25,    1.8731741875544931e-26,    1.1139347680089906e-27,    6.4153730339312922e-29,
    3.5817429054977951e-30,    1.9403559828087389e-31,    1.0208541536653125e-32,    5.2203618497692338e-34,
    2.5967598916432646e-35,    1.2574154990154333e-36,    5.9312535620160617e-38,    2.7272455120359411e-39,
    1.2231708971021030e-40,    5.3542300230205012e-42,    2.2887764683081720e-43,    9.5596986744916290e-45,
    3.9034334708878087e-46,    1.5589363804399639e-47,    6.0925197879009054e-49,    2.3310500147323019e-50,
    8.7354087918748293e-52,    3.2075619524719371e-53,    1.1545239822032372e-54,    4.0750787761417776e-56,
    1.4110318114706630e-57,    4.7947045917874806e-59,    1.5994177309362872e-60,    5.2394084892601783e-62,
    1.6860212992556547e-63,    5.3313789335543433e-65,    1.6570757426069280e-66,    5.0640466035042393e-68
},
// x = 3.6500
{
    1.3689376181161728e-01,    9.9573700705747850e-02,    5.5052363971276573e-02,    2.4159503484821005e-02,
    8.7190696168253212e-03,    2.6604277173065288e-03,    7.0134224960016452e-04,    1.6249641736073729e-04,
    3.3548753597134502e-05,    6.2419485521657494e-06,    1.0564186680525064e-06,    1.6392333871296607e-07,
    2.3477081642035954e-08,    3.1214096579252371e-09,    3.8720198067119389e-10,    4.5010359441779008e-11,
    4.9222155492352516e-12,    5.0813666787122005e-13,    4.9672158689309001e-14,    4.6106756782245094e-15,
    4.0740486718410507e-16,    3.4347033142781970e-17,    2.7685863239336343e-18,    2.1377709428512850e-19,
    1.5840178344307433e-20,    1.1281247314122407e-21,    7.7339631424073167e-23,    5.1109052544653512e-24,
    3.2599060336242117e-25,    2.0093092367269023e-26,    1.1981513983466249e-27,    6.9192285311709574e-29,
    3.8735970760215115e-30,    2.1041957434017690e-31,    1.1100779914155000e-32,    5.6921431916460826e-34,
    2.8391787939336602e-35,    1.3785603778759866e-36,    6.5204832295775998e-38,    3.0063814993423849e-39,
    1.3520532370793421e-40,    5.9345896594618160e-42,    2.5438076674770655e-43,    1.0653995474129395e-44,
    4.3621709037572843e-46,    1.7469160858204354e-47,    6.8458678761988037e-49,    2.6264625697809378e-50,
    9.8694069059529536e-52,    3.6338851852219974e-53,    1.3115581652189531e-54,    4.6420431092759721e-56,
    1.6117535602550518e-57,    5.4917715559596393e-59,    1.8369671381149377e-60,    6.0340750136669499e-62,
    1.9470654655366114e-63,    6.1737094029183257e-65,    1.9241468089201125e-66,    5.8963441881009836e-68
},
// x = 3.6600
{
    1.3652156253945186e-01,    9.9401507804960154e-02,    5.5044916797681254e-02,    2.4203534037636081e-02,
    8.7540047038417168e-03,    2.6772929626154751e-03,    7.0749579980613553e-04,    1.6432973926034866e-04,
    3.4013261853886997e-05,    6.3446432177041857e-06,    1.0765894122532437e-06,    1.6749085231671821e-07,
    2.4051269279331863e-08,    3.2062260917738265e-09,    3.9878171706593102e-10,    4.6480246169454040e-11,
    5.0965719148179671e-12,    5.2754857683298800e-13,    5.1708475158247496e-14,    4.8126257796665700e-15,
    4.2639717819384439e-16,    3.6045368205474644e-17,    2.9133440858087577e-18,    2.2556387176044234e-19,
    1.6758847354987785e-20,    1.1967896854693491e-21,    8.2269770578821507e-23,    5.4514776339559543e-24,
    3.4865859314464020e-25,    2.1548724326313235e-26,    1.2884469007929627e-27,    7.4609313097184683e-29,
    4.1882327266689332e-30,    2.2813079295501748e-31,    1.2067937601672777e-32,    6.2049390708805566e-34,
    3.1033939582083366e-35,    1.5109591393979807e-36,    7.1662135403452841e-38,    3.3131213471960898e-39,
    1.4940687107821981e-40,    6.5758397272916096e-42,    2.8263682543203815e-43,    1.1869737749739243e-44,
    4.8732154479383631e-46,    1.9569037539397280e-47,    7.6897209297496465e-49,    2.9582717902276035e-50,
    1.1146610852860511e-51,    4.1153620104749070e-53,    1.4893939280845203e-54,    5.2858701323754486e-56,
    1.8403115963480625e-57,    6.2876838359264244e-59,    2.1089448613754815e-60,    6.9464072398838677e-62,
    2.2475837057835080e-63,    7.1460717544573496e-65,    2.2332911316096263e-66,    6.8623993118219774e-68
},
// x = 3.6700
{
    1.3615135551485774e-01,    9.9229746524690562e-02,    5.5037121298489422e-02,    2.4247292439555056e-02,
    8.7888795881664421e-03,    2.6941817328798730e-03,    7.1367548416682110e-04,    1.6617593065407701e-04,
    3.4482307106560952e-05,    6.4486225310426550e-06,    1.0970678450322052e-06,    1.7112260034066172e-07,
    2.4637379682009266e-08,    3.2930384741136499e-09,    4.1066611223580822e-10,    4.7992900588188321e-11,
    5.2764887388497087e-12,    5.4763399907648894e-13,    5.3821172725156161e-14,    5.0227209209639584e-15,
    4.4620925987157477e-16,    3.7821832480426469e-17,    3.0651735886486339e-18,    2.3796014004813510e-19,
    1.7727653427284131e-20,    1.2693994658673465e-21,    8.7497362097568599e-23,    5.8135827144486223e-24,
    3.7282550501435704e-25,    2.3104843653429257e-26,    1.3852391962846887e-27,    8.0431944065206421e-29,
    4.5273499337335075e-30,    2.4727221434051261e-31,    1.3116047953031438e-32,    6.7621740340939501e-34,
    3.3912900697096055e-35,    1.6556184262846625e-36,    7.8736672205161362e-38,    3.6500993643831966e-39,
    1.6505100999168808e-40,    7.2841574821815980e-42,    3.1393341918880422e-43,    1.3219980533536300e-44,
    5.4423487881576050e-46,    2.1913990047827037e-47,    8.6346346092121841e-49,    3.3308340559109948e-50,
    1.2584600723231308e-51,    4.6589328771077166e-53,    1.6907131033374951e-54,    6.0167071598395474e-56,
    2.1004671972414132e-57,    7.1961050616618889e-59,    2.4202174108350683e-60,    7.9934055032419410e-62,
    2.5934030425473525e-63,    8.2680697855093771e-65,    2.5909844177110826e-66,    7.9832222342655425e-68
},
// x = 3.6800
{
    1.3578312520930183e-01,    9.9058416853387884e-02,    5.5028981035344302e-02,    2.4290779577104831e-02,
    8.8236937962861885e-03,    2.7110936622744908e-03,    7.1988121883526655e-04,    1.6803500878034323e-04,
    3.4955911306693634e-05,    6.5538967657258929e-06,    1.1178573532175518e-06,    1.7481947834312086e-07,
    2.5235613573046784e-08,    3.3818861349226314e-09,    4.2286203964703724e-10,    4.9549409443260953e-11,
    5.4621231413066185e-12,    5.6841388263092817e-13,    5.6012844545074605e-14,    5.2412608462108764e-15,
    4.6687362055717509e-16,    3.9679747611914341e-17,    3.2243957440451507e-18,    2.5099530701444248e-19,
    1.8749159893303206e-20,    1.3461671307855945e-21,    9.3039330785460748e-23,    6.1985080602085152e-24,
    3.9858532038785193e-25,    2.4768043331462360e-26,    1.4889734975588084e-27,    8.6689159971224674e-29,
    4.8927697905600423e-30,    2.6795443144131372e-31,    1.4251609427429266e-32,    7.3675467701508148e-34,
    3.7049082844823067e-35,    1.8136314520144626e-36,    8.6485317354439094e-38,    4.0201921525566583e-39,
    1.8227929683691664e-40,    8.0663254397435690e-42,    3.4858719052643661e-43,    1.4719136823155984e-44,
    6.0759715291206274e-46,    2.4531766315332875e-47,    9.6923587529883126e-49,    3.7490118922185318e-50,
    1.4203052967647480e-51,    5.2723871592786669e-53,    1.9185338066279538e-54,    6.8460052182499051e-56,
    2.3964765199659549e-57,    8.2325389992164377e-59,    2.7763218242181362e-60,    9.1944655268571018e-62,
    2.9911897585220382e-63,    9.5621922649674694e-65,    3.0046757197052404e-66,    9.2830474263522844e-68
},
// x = 3.6900
{
    1.3541685624850958e-01,    9.8887518751894374e-02,    5.5020499539652358e-02,    2.4333996340441279e-02,
    8.8584468613084593e-03,    2.7280283860304103e-03,    7.2611291921238536e-04,    1.6990699043122176e-04,
    3.5434096321239958e-05,    6.6604762141270570e-06,    1.1389613433499649e-06,    1.7858239018416351e-07,
    2.5846174180382324e-08,    3.4728090162615873e-09,    4.3537650041949164e-10,    5.1150882883468041e-11,
    5.6536360868334640e-12,    5.8990974918500487e-13,    5.8286162314450250e-14,    5.4685552460725092e-15,
    4.8842394132618250e-16,    4.1622564670483296e-17,    3.3913448950758537e-18,    2.6470009638754351e-19,
    1.9826052199283005e-20,    1.4273165054656663e-21,    9.8913505855095706e-23,    6.6076137800091610e-24,
    4.2603758937657960e-25,    2.6545326224598009e-26,    1.6001239968280582e-27,    9.3411913890282052e-29,
    5.2864426037294314e-30,    2.9029620941129727e-31,    1.5481619838669276e-32,    8.0250511097749268e-34,
    4.0464586798830198e-35,    1.9861851469755711e-36,    9.4969990383665805e-38,    4.4265400535466630e-39,
    2.0124668984835960e-40,    8.9297885826701132e-42,    3.8694666092899419e-43,    1.6383114116045249e-44,
    6.7811672955337263e-46,    2.7453159337885161e-47,    1.0875968304593050e-48,    4.2182310049031375e-50,
    1.6023983026699604e-51,    5.9644640092060216e-53,    2.1762514195648407e-54,    7.7866819418205976e-56,
    2.7331539618018098e-57,    9.4145708398390887e-59,    3.1835556696074748e-60,    1.0571707394246550e-61,
    3.4485672731484187e-63,    1.1054227152473387e-64,    3.4829302366578482e-66,    1.0789816728989897e-67
},
// x = 3.7000
{
    1.3505253341065815e-01,    9.8717052154028803e-02,    5.5011680312796955e-02,    2.4376943623222090e-02,
    8.8931383229172972e-03,    2.7449855404503031e-03,    7.3237049995693629e-04,    1.7179189195295992e-04,
    3.5916883931423079e-05,    6.7683711869620598e-06,    1.1603832416178911e-06,    1.8241224804970183e-07,
    2.6469267254879995e-08,    3.5658476788911045e-09,    4.4821662513409026e-10,    5.2798454867153056e-11,
    5.8511924633486550e-12,    6.1214370755693423e-13,    6.0643878350627812e-14,    5.7049240506564148e-15,
    5.1089511397912156e-16,    4.3653868725607630e-17,    3.5663693301677689e-18,    2.7910660194562995e-19,
    2.0961143290847309e-20,    1.5130826884627792e-21,    1.0513866609007969e-22,    7.0423363616379333e-24,
    4.5528774140772046e-25,    2.8444129140617580e-26,    1.7191956519266595e-27,    1.0063325750237945e-28,
    5.7104566158738554e-30,    3.1442506135230623e-31,    1.6813613007767215e-32,    8.7389985610688063e-34,
    4.4183336527086405e-35,    2.1745678697694966e-36,    1.0425808581285387e-37,    4.8725704209161787e-39,
    2.2212277167057236e-40,    9.8827167766247892e-42,    4.2939533007037627e-43,    1.8229464197226254e-44,
    7.5657732478588738e-46,    3.0712330754909630e-47,    1.2200008137859424e-48,    4.7445435425121121e-50,
    1.8072015193191855e-51,    6.7449648374905357e-53,    2.4676844230692960e-54,    8.8533042473215913e-56,
    3.1159434095023594e-57,    1.0762139274357842e-58,    3.6490788096840845e-60,    1.2150348531763346e-61,
    3.9742501550821075e-63,    1.2773733809872505e-64,    4.0355925541682241e-66,    1.2535733178388345e-67
},
// x = 3.7100
{
    1.3469014162459375e-01,    9.8547016967159964e-02,    5.5002526826351227e-02,    2.4419622322481731e-02,
    8.9277677273290491e-03,    2.7619647629234124e-03,    7.3865387499547877e-04,    1.7368972924653359e-04,
    3.6404295831596496e-05,    6.8775920128030222e-06,    1.1821264937912594e-06,    1.8630997247513496e-07,
    2.7105101088266613e-08,    3.6610433089179556e-09,    4.6138967565615554e-10,    5.4493283573343826e-11,
    6.0549611619079408e-12,    6.3513846742413457e-13,    6.3088827717993506e-14,    5.9506977298596173e-15,
    5.3432328011613105e-16,    4.5777383562749391e-17,    3.7498318147228674e-18,    2.9424834373892040e-19,
    2.2157379216326555e-20,    1.6037125798905289e-21,    1.1173458710364787e-22,    7.5041926955584169e-24,
    4.8644741208104238e-25,    3.0472348221652592e-26,    1.8467260763245602e-27,    1.0838847615442714e-28,
    6.1670472871168595e-30,    3.4047786262239764e-31,    1.8255697979218444e-32,    9.5140424879320136e-34,
    4.8231223337777452e-35,    2.3801777264203602e-36,    1.1442293844405021e-37,    5.3620228646011858e-39,
    2.4509307922707311e-40,    1.0934072851072791e-41,    4.7635506560645335e-43,    2.0277547420894960e-44,
    8.4384576336395932e-46,    3.4347167689456886e-47,    1.3680653206775002e-48,    5.3346982446776801e-50,
    2.0374682667731912e-51,    7.6248787208186650e-53,    2.7971256382930353e-54,    1.0062293120638619e-55,
    3.5509983313715700e-57,    1.2297843171906421e-58,    4.1810284228173177e-60,    1.3959126432366470e-61,
    4.5781964219587707e-63,    1.4754581117379203e-64,    4.6739731754592267e-66,    1.4557895580564622e-67
},
// x = 3.7200
{
    1.3432966596807178e-01,    9.8377413072769121e-02,    5.4993042522290210e-02,    2.4462033338508098e-02,
    8.9623346272480579e-03,    2.7789656919402386e-03,    7.4496295753229613e-04,    1.7560051776823626e-04,
    3.6896353628117708e-05,    6.9881490375908537e-06,    1.2041945651536684e-06,    1.9027649236853007e-07,
    2.7753886531036042e-08,    3.7584377244706679e-09,    4.7490304697475216e-10,    5.6236551818029414e-11,
    6.2651151578403837e-12,    6.5891735331634900e-13,    6.5623930401616030e-14,    6.2062176013509777e-15,
    5.5874587132351396e-16,    4.7996976548808388e-17,    3.9421101410512439e-18,    3.1016032641433359e-19,
    2.3417844956169276e-20,    1.6994654325341317e-21,    1.1872209078197025e-22,    7.9947842963834927e-24,
    5.1963478705303917e-25,    3.2638365732091476e-26,    1.9832875386851570e-27,    1.1671523214667047e-28,
    6.6586071689644158e-30,    3.6860150616319774e-31,    1.9816600971336048e-32,    1.0355204045454496e-33,
    5.2636260925588435e-35,    2.6045315433099227e-36,    1.2554432659808644e-37,    5.8989766291025433e-39,
    2.7036054994096427e-40,    1.2093686839614361e-41,    5.2828980978361927e-43,    2.2548712838111883e-44,
    9.4088050519649785e-46,    3.8399676152660860e-47,    1.5335885588257150e-48,    5.9962182017996544e-50,
    2.2962761080612244e-51,    8.6165221776458497e-53,    3.1693994941840332e-54,    1.1432153113930187e-55,
    4.0452717786333831e-57,    1.4047287142411657e-58,    4.7886489612091513e-60,    1.6030777578398069e-61,
    5.2717805581182016e-63,    1.7035560458354647e-64,    5.4110615838366198e-66,    1.6899025320120839e-67
},
// x = 3.7300
{
    1.3397109166602048e-01,    9.8208240327003538e-02,    5.4983230813202638e-02,    2.4504177574721203e-02,
    8.9968385818223612e-03,    2.7959879671069312e-03,    7.5129766005929373e-04,    1.7752427253030372e-04,
    3.7393078838233196e-05,    7.1000526241470797e-06,    1.2265909404330594e-06,    1.9431274503333999e-07,
    2.8415837010319659e-08,    3.8580733824039236e-09,    4.8876426905802106e-10,    5.8029467475609480e-11,
    6.4818315931701714e-12,    6.8350431887602007e-13,    6.8252193529234250e-14,    6.4718361463507994e-15,
    5.8420165049936089e-16,    5.0316663650050288e-17,    4.1435976971723548e-18,    3.2687909971356125e-19,
    2.4745770486731589e-20,    1.8006134267388075e-21,    1.2612309700525401e-22,    8.5158017311766741e-24,
    5.5497496377494466e-25,    3.4951078315590845e-26,    2.1294890779312185e-27,    1.2565371671567568e-28,
    7.1876964064013583e-30,    3.9895360144291252e-31,    2.1505710241802110e-32,    1.1267899993883080e-33,
    5.7428752105111942e-35,    2.8492745429523968e-36,    1.3770901625681544e-37,    6.4878802781896375e-39,
    2.9814709408581795e-40,    1.3372336915844000e-41,    5.8570963138052738e-43,    2.5066495644314743e-44,
    1.0487410171540965e-45,    4.2916414632550556e-47,    1.7185690141221494e-48,    6.7374870255712710e-50,
    2.5870639096261983e-51,    9.7336949079700054e-53,    3.5899260094935049e-54,    1.2983729448721125e-55,
    4.6066174874032081e-57,    1.6039470776441698e-58,    5.4824389333336683e-60,    1.8402579836623106e-61,
    6.0679899954259716e-63,    1.9661083724038405e-64,    6.2617695160110301e-66,    1.9608298481022858e-67
},
// x = 3.7400
{
    1.3361440408882719e-01,    9.8039498561219005e-02,    5.4973095082501755e-02,    2.4546055937553577e-02,
    9.0312791565993634e-03,    2.8130312291593904e-03,    7.5765789436586491e-04,    1.7946100810157121e-04,
    3.7894492888974989e-05,    7.2133131516848947e-06,    1.2493191237308785e-06,    1.9841967619065268e-07,
    2.9091168547720941e-08,    3.9599933850313603e-09,    5.0298100872450115e-10,    5.9873263905547470e-11,
    6.7052918603378721e-12,    7.0892396138973897e-13,    7.0976713642454376e-14,    6.7479173333724489e-15,
    6.1073075434594914e-16,    5.2740614606695956e-17,    4.3547040550597733e-18,    3.4444282121740750e-19,
    2.6144537087005251e-20,    1.9074422700125367e-21,    1.3396067774338878e-22,    9.0690292639779521e-24,
    5.9260033194839089e-25,    3.7419926796594140e-26,    2.2859787400874003e-27,    1.3524681121140495e-28,
    7.7570539059804086e-30,    4.3170321976684739e-31,    2.3333124060945635e-32,    1.2257972520753116e-33,
    6.2641468072085629e-35,    3.1161907752431533e-36,    1.5101134929514603e-37,    7.1335838725001152e-39,
    3.2869530383340310e-40,    1.4781837605555278e-41,    6.4917515396812849e-43,    2.7856833552353459e-44,
    1.1685980711364708e-45,    4.7948971833163717e-47,    1.9252270675139901e-48,    7.5678443109967202e-50,
    2.9136730094759164e-51,    1.0991853265063910e-52,    4.0647922531176428e-54,    1.4740495949417957e-55,
    5.2439034116289783e-57,    1.8307226930184256e-58,    6.2743166297362296e-60,    2.1116966514512267e-61,
    6.9816481563885979e-63,    2.2681977830124522e-64,    7.2432086230366502e-66,    2.2742298164956158e-67
},
// x = 3.7500
{
    1.3325958875064697e-01,    9.7871187582513841e-02,    5.4962638684635898e-02,    2.4587669336332676e-02,
    9.0656559234815054e-03,    2.8300951199770829e-03,    7.6404357154872659e-04,    1.8141073860816444e-04,
    3.8400617116068748e-05,    7.3279410153195389e-06,    1.2723826384497434e-06,    2.0259824000097239e-07,
    2.9780099777112893e-08,    4.0642414868864486e-09,    5.1756107153045539e-10,    6.1769200384260589e-11,
    6.9356816872347605e-12,    7.3520153659465003e-13,    7.3800679018029368e-14,    7.0348369500934835e-15,
    6.3837473705716760e-16,    5.5273158268451780e-17,    4.5758555789200345e-18,    3.6289132141140233e-19,
    2.7617683897123964e-20,    2.0202518223159792e-21,    1.4225911362662522e-22,    9.6563497263425535e-24,
    6.3265097360113142e-25,    4.0054927605358840e-26,    2.4534459434856287e-27,    1.4554025799260699e-28,
    8.3696092098309084e-30,    4.6703168887128378e-31,    2.5309701997302027e-32,    1.3331721209266541e-33,
    6.8309841090723191e-35,    3.4072143605749570e-36,    1.6555387922406429e-37,    7.8413738408378479e-39,
    3.6227031041358128e-40,    1.6335135904492539e-41,    7.1930239414643255e-43,    3.0948303840071984e-44,
    1.3017450567625426e-45,    5.3554492907629759e-47,    2.1560288707271234e-48,    8.4976913597091708e-50,
    3.2803929341336964e-51,    1.2408303416677175e-52,    4.6008321309208058e-54,    1.6728877397143089e-55,
    5.9671391721710116e-57,    2.0887715064253266e-58,    7.1778071707426832e-60,    2.4222221294545449e-61,
    8.0296675572330278e-63,    2.6156388749917503e-64,    8.3750072583247810e-66,    2.6366103944191670e-67
},
// x = 3.7600
{
    1.3290663130773328e-01,    9.7703307174252532e-02,    5.4951864945297781e-02,    2.4629018683165076e-02,
    9.0999684606819491e-03,    2.8471792825965851e-03,    7.7045460202172875e-04,    1.8337347773422471e-04,
    3.8911472762853592e-05,    7.4439466255780907e-06,    1.2957850272196208e-06,    2.0684939908552888e-07,
    3.0482851962396298e-08,    4.1708621015110667e-09,    5.3251240367319487e-10,    6.3718562542277803e-11,
    7.1731912235640257e-12,    7.6236297376371304e-13,    7.6727372040105460e-14,    7.3329829435264762e-15,
    6.6717661522978043e-16,    5.7918788095360766e-17,    4.8074960541118426e-18,    3.8226617114992513e-19,
    2.9168914737776777e-20,    2.1393567480470767e-21,    1.5104395309566857e-22,    1.0279749624089196e-23,
    6.7527508372555427e-25,    4.2866705909255842e-26,    2.6326239792526709e-27,    1.5658284159281319e-28,
    9.0284951177691046e-30,    5.0513343988980063e-31,    2.7447119732765872e-32,    1.4495937300015345e-33,
    7.4472171566713635e-35,    3.7244416052308507e-36,    1.8144805811954454e-37,    8.6170107614561491e-39,
    3.9916180171578615e-40,    1.8046415983099108e-41,    7.9676804631127985e-43,    3.4372382977091599e-44,
    1.4496104051712939e-45,    5.9796258941293533e-47,    2.4137127091368811e-48,    9.5386082322796730e-50,
    3.6920121513245768e-51,    1.4002416362549850e-52,    5.2057154404417716e-54,    1.8978610299342001e-55,
    6.7876190794677528e-57,    2.3822976360251960e-58,    8.2082535449854391e-60,    2.7773264416520346e-61,
    9.2313369169345638e-63,    3.0150809775498751e-64,    9.6796717687244118e-66,    3.0554537712712932e-67
},
// x = 3.7700
{
    1.3255551755679068e-01,    9.7535857096580497e-02,    5.4940777161633779e-02,    2.4670104892822137e-02,
    9.1342163526802137e-03,    2.8642833612248023e-03,    7.7689089552562059e-04,    1.8534923872266362e-04,
    3.9427080979213326e-05,    7.5613404079085983e-06,    1.3195298518225143e-06,    2.1117412454710926e-07,
    3.1199649015216767e-08,    4.2799003082712963e-09,    5.4784309391039797e-10,    6.5722662806696640e-11,
    7.4180151285423694e-12,    7.9043489107374345e-13,    7.9760171624328546e-14,    7.6427557686623967e-15,
    6.9718091402790155e-16,    6.0682167828452615e-17,    5.0500873373275735e-18,    4.0261075159832202e-19,
    3.0802105199951955e-20,    2.2650871957634436e-21,    1.6034207423957693e-22,    1.0941324490876616e-23,
    7.2062941246456571e-25,    4.5866530537025802e-26,    2.8242926543475822e-27,    1.6842658073866032e-28,
    9.7370611020643931e-30,    5.4621690996398369e-31,    2.9757927638150628e-32,    1.5757940402722123e-33,
    8.1169850530734197e-35,    4.0701440537497321e-36,    1.9881497868386208e-37,    9.4667702862369120e-39,
    4.3968621364553595e-40,    1.9931213216113918e-41,    8.8231525363811658e-43,    3.8163730904383540e-44,
    1.6137712292661409e-45,    6.6744324891071256e-47,    2.7013181023521962e-48,    1.0703483304627031e-49,
    4.1538743957811372e-51,    1.5795867206068051e-52,    5.8880472380687094e-54,    2.1523146520579229e-55,
    7.7180825788959658e-57,    2.7160558136891091e-58,    9.3830546324405152e-60,    3.1832541771312939e-61,
    1.0608646722382794e-62,    3.4741250643020669e-64,    1.1182998384447867e-65,    3.5393587837638472e-67
},
// x = 3.7800
{
    1.3220623343334853e-01,    9.7368837086929616e-02,    5.4929378602452068e-02,    2.4710928882627985e-02,
    9.1683991901779331e-03,    2.8814070012519732e-03,    7.8335236113779661e-04,    1.8733803437595374e-04,
    3.9947462820519548e-05,    7.6801328021887619e-06,    1.3436206931157015e-06,    2.1557339599041633e-07,
    3.1930717512639886e-08,    4.3914018592002222e-09,    5.6356137549543878e-10,    6.7782840848971764e-11,
    7.6703526599560471e-12,    8.1944461126022330e-13,    8.2902555694720649e-14,    7.9645687457625684e-15,
    7.2843371463065947e-16,    6.3568137334779005e-17,    5.3041100286757109e-18,    4.2397032673486086e-19,
    3.2521310014742080e-20,    2.3977895067228323e-21,    1.7018174943402948e-22,    1.1643284499667475e-23,
    7.6887972987314082e-25,    4.9066350786779625e-26,    3.0292810847819239e-27,    1.8112693183320054e-28,
    1.0498887561915628e-29,    5.9050550396361049e-31,    3.2255613354270455e-32,    1.7125617824835637e-33,
    8.8447598626824246e-35,    4.4467825475378442e-36,    2.1778617567662124e-37,    1.0397487458522874e-38,
    4.8418910960873248e-40,    2.2006538335751837e-41,    9.7675990836161025e-43,    4.2360502223568068e-44,
    1.7959682953611559e-45,    7.4476221678430269e-47,    3.0222179174642343e-48,    1.2006656621501071e-49,
    4.6719411610552974e-51,    1.7812901332715569e-52,    6.6574786774086094e-54,    2.4400104719029120e-55,
    8.7728941810986707e-57,    3.0954215977254843e-58,    1.0721933568857649e-59,    3.6471030019850081e-61,
    1.2186658266753391e-62,    4.0014566328030069e-64,    1.2914542617262480e-65,    4.0982036503343333e-67
},
// x = 3.7900
{
    1.3185876501015642e-01,    9.7202246860514749e-02,    5.4917672508429688e-02,    2.4751491572348867e-02,
    9.2025165700544854e-03,    2.8985498492643392e-03,    7.8983890728200122e-04,    1.8933987705694813e-04,
    4.0472639246586242e-05,    7.8003342622341111e-06,    1.3680611509534903e-06,    2.2004820154194391e-07,
    3.2676286714781239e-08,    4.5054131858671525e-09,    5.7967562812869813e-10,    6.9900464038064211e-11,
    7.9304077645848500e-12,    8.4942017756284629e-13,    8.6158103714238870e-14,    8.2988484264774306e-15,
    7.6098270299356108e-16,    6.6581718631517357e-17,    5.5700641663188565e-18,    4.4639211849670704e-19,
    3.4330770713250758e-20,    2.5378269533596367e-21,    1.8059271289680761e-22,    1.2387960343592613e-23,
    8.2020131432911403e-25,    5.2478835212797800e-26,    3.2484706470370812e-27,    1.9474300454927606e-28,
    1.1317800967320044e-29,    6.3823861898510370e-31,    3.4954668638786800e-32,    1.8607466694604075e-33,
    9.6353722773915303e-35,    4.8570223638833170e-36,    2.3850449126396856e-37,    1.1416604694509611e-38,
    5.3304776363629833e-40,    2.4291012572892450e-41,    1.0809975280996836e-42,    4.7004686751793263e-44,
    1.9981224516588775e-45,    8.3077728669209851e-47,    3.3801537965840785e-48,    1.3464078469285225e-49,
    5.2528610112644451e-51,    2.0080630428993814e-52,    7.5248306037929186e-54,    2.7651775086559260e-55,
    9.9682451765524128e-57,    3.5264702970767152e-58,    1.2247240214451475e-59,    4.1769372476658776e-61,
    1.3993921815266283e-62,    4.6069966761926311e-64,    1.4908153995767214e-65,    4.7433318528859898e-67
},
// x = 3.8000
{
    1.3151309849559992e-01,    9.7036086110821151e-02,    5.4905662092320083e-02,    2.4791793884084222e-02,
    9.2365680953227884e-03,    2.9157115530565655e-03,    7.9635044173799777e-04,    1.9135477868973162e-04,
    4.1002631120636375e-05,    7.9219552553057676e-06,    1.3928548441075375e-06,    2.2459953786937199e-07,
    3.3436588582391053e-08,    4.6219814062730191e-09,    5.9619437992487376e-10,    7.2076927898983523e-11,
    8.1983891700080923e-12,    8.8039036996585570e-13,    8.9530499269950060e-14,    8.6460349689739054e-15,
    7.9487721995472847e-16,    6.9728122093941288e-17,    5.8484699443396032e-18,    4.6992538465659592e-19,
    3.6234923586967656e-20,    2.6855805088550239e-21,    1.9160623128180973e-22,    1.3177809398202972e-23,
    8.7477946571415452e-25,    5.6117412490654836e-26,    3.4827980960931508e-27,    2.0933779021212350e-28,
    1.2197889944786511e-29,    6.8967273551223001e-31,    3.7870660755083325e-32,    2.0212639067700842e-33,
    1.0494039174753821e-34,    5.3037495147441190e-36,    2.6112500916749042e-37,    1.2532223718654283e-38,
    5.8667396388820130e-40,    2.6805014721580594e-41,    1.1960107589408669e-42,    5.2142482113837347e-44,
    2.2223526503800207e-45,    9.2643723357935231e-47,    3.7792752294659485e-48,    1.5093484732128508e-49,
    5.9040464338210409e-51,    2.2629361585321478e-52,    8.5022313293967648e-54,    3.1325683503746462e-55,
    1.1322379697070144e-56,    4.0160656578954730e-58,    1.3984291943414343e-59,    4.7819162318878243e-61,
    1.6062950268330179e-62,    5.3020731475334977e-64,    1.7202585005121489e-65,    5.4877643776703497e-67
},
// x = 3.8100
{
    1.3116922023213740e-01,    9.6870354510083745e-02,    5.4893350539158087e-02,    2.4831836742159746e-02,
    9.2705533750850437e-03,    2.9328917616438998e-03,    8.0288687165121046e-04,    1.9338275076050441e-04,
    4.1537459208279894e-05,    8.0450062616178282e-06,    1.4180054101857187e-06,    2.2922841020047660e-07,
    3.4211857794390676e-08,    4.7411543317714321e-09,    6.1312630939626644e-10,    7.4313656576752707e-11,
    8.4745104778063293e-12,    9.1238472173722010e-13,    9.3023532713757219e-14,    9.0065825222553999e-15,
    8.3016831271771560e-16,    7.3012752852157257e-17,    6.1398684545233467e-18,    4.9462149951934022e-19,
    3.8238407959306180e-20,    2.8414496489984249e-21,    2.0325517743749804e-22,    1.4015422177585266e-23,
    9.3281004453489198e-25,    5.9996314464839365e-26,    3.7332588589015415e-27,    2.2497840368606826e-28,
    1.3143522360255341e-29,    7.4508257934984567e-31,    4.1020308696371287e-32,    2.1950990218620750e-33,
    1.1426393201244768e-34,    5.7900882902395415e-36,    2.8581606285046393e-37,    1.3753161765598236e-38,
    6.4551705459495455e-40,    2.9570841138578397e-41,    1.3228775603072706e-42,    5.7824701277070396e-44,
    2.4709957127503082e-45,    1.0327911571146707e-46,    4.2241826339041205e-48,    1.6914590750819081e-49,
    6.6337590279723675e-51,    2.5492963067915098e-52,    9.6032701676548462e-54,    3.5475221900934546e-55,
    1.2855847979630659e-56,    4.5719594865797215e-58,    1.5961757477635498e-59,    5.4724391721938147e-61,
    1.8430755496032863e-62,    6.0996156276357890e-64,    1.9842184271297935e-65,    6.3464419616198705e-67
},
// x = 3.8200
{
    1.3082711669475675e-01,    9.6705051709757170e-02,    5.4880741006465755e-02,    2.4871621073021863e-02,
    9.3044720244885252e-03,    2.9500901252740436e-03,    8.0944810354232287e-04,    1.9542380431849518e-04,
    4.2077144176503876e-05,    8.1694977738444916e-06,    1.4435165055495819e-06,    2.3393583234155342e-07,
    3.5002331765359868e-08,    4.8629804740151061e-09,    6.3048024745204989e-10,    7.6612103305826160e-11,
    8.7589902581727748e-12,    9.4543353627074879e-13,    9.6641103859632144e-14,    9.3809596198619345e-15,
    8.6690878774328780e-16,    7.6441217381623144e-17,    6.4448224527652496e-18,    5.2053403752991823e-19,
    4.0346074779341497e-20,    3.0058531875801089e-21,    2.1557410746045707e-22,    1.4903529097327294e-23,
    9.9450003820537669e-25,    6.4130621487907088e-26,    4.0009105125712695e-27,    2.4173633951768288e-28,
    1.4159363457652110e-29,    8.0476235868099327e-31,    4.4421564556158985e-32,    2.3833130330952696e-33,
    1.2438514522599832e-34,    6.3194201377077081e-36,    3.1276032336017820e-37,    1.5089012384742922e-38,
    7.1006723591343751e-40,    3.2612879772144136e-41,    1.4627801312483531e-42,    6.4107218188627149e-44,
    2.7466279995149182e-45,    1.1509987532145760e-46,    4.7199748414520031e-48,    1.8949305574136124e-49,
    7.4512039050562907e-51,    2.8709270720946031e-52,    1.0843168476035870e-53,    4.0160352369521504e-55,
    1.4591790013055982e-56,    5.2029035205315372e-58,    1.8212089054025597e-59,    6.2603087807811808e-61,
    2.1139455420605866e-62,    7.0143762564055089e-64,    2.2877685353681222e-65,    7.3365014822072134e-67
},
// x = 3.8300
{
    1.3048677448945223e-01,    9.6540177340977987e-02,    5.4867836624456441e-02,    2.4911147805134034e-02,
    9.3383236646814374e-03,    2.9673062954387644e-03,    8.1603404331684044e-04,    1.9747794997690372e-04,
    4.2621706592674159e-05,    8.2954402966268280e-06,    1.4693918052303596e-06,    2.3872282669535194e-07,
    3.5808250662972674e-08,    4.9875090519270996e-09,    6.4826517941349215e-10,    7.8973750884990139e-11,
    9.0520521459482182e-12,    9.7956790423525545e-13,    1.0038722473831357e-13,    9.7696495831403280e-15,
    9.0515326508309499e-16,    8.0019330292571822e-17,    6.7639171508262485e-18,    5.4771885988747551e-19,
    4.2562995549133945e-20,    3.1792301465973520e-21,    2.2859934117964743e-22,    1.5845007557844890e-23,
    1.0600681557651406e-24,    6.8536310165257526e-26,    4.2868764569960583e-27,    2.5968774312707152e-28,
    1.5250395114723516e-29,    8.6902708085006208e-31,    4.8093700375171911e-32,    2.5870479814298427e-33,
    1.3536964892681808e-34,    6.8954039735158520e-36,    3.4215597285240752e-37,    1.6550211208964344e-38,
    7.8085914269900845e-40,    3.5957799403073856e-41,    1.6170146428390314e-42,    7.1051454949129693e-44,
    3.0520891642845488e-45,    1.2823416027203853e-46,    5.2723014241021594e-48,    2.1221968679025888e-49,
    8.3666342654498802e-51,    3.2320539374767582e-52,    1.2238970144418955e-53,    4.5448393401666793e-55,
    1.6556253109092942e-56,    5.9187750106488756e-58,    2.0772008846739404e-59,    7.1589158849211431e-61,
    2.4236960936627943e-62,    8.0631803807783893e-64,    2.6367104003539782e-65,    8.4775911852966122e-67
},
// x = 3.8400
{
    1.3014818035172115e-01,    9.6375731015018160e-02,    5.4854640496238186e-02,    2.4950417868874645e-02,
    9.3721079227688103e-03,    2.9845399248852448e-03,    8.2264459627462026e-04,    1.9954519791387428e-04,
    4.3171166923548975e-05,    8.4228443460794006e-06,    1.4956350028435900e-06,    2.4359042427851828e-07,
    3.6629857425380649e-08,    5.1147899986966023e-09,    6.6649024704514078e-10,    8.1400112157776208e-11,
    9.3539249380935294e-12,    1.0148197210349436e-12,    1.0426602241044753e-13,    1.0173150934277773e-14,
    9.4495823418888040e-16,    8.3753121323582318e-17,    7.0977610341819777e-18,    5.7623420426231166e-19,
    4.4894471596385474e-20,    3.3620406626012943e-21,    2.4236904621204511e-22,    1.6842889362130565e-23,
    1.1297454523624860e-24,    7.3230303624918187e-26,    4.5923497921282920e-27,    2.7891369788020386e-28,
    1.6421936281194118e-29,    9.3821395374113860e-31,    5.2057400814866310e-32,    2.8075328490100598e-33,
    1.4728824202407459e-34,    7.5219980315553245e-36,    3.7421797025974184e-37,    1.8148107076127084e-38,
    8.5847572483556767e-40,    3.9634755376830684e-41,    1.7870018467798746e-42,    7.8724914255313906e-44,
    3.3905081829189037e-45,    1.4282355745391361e-46,    5.8874203391797687e-48,    2.3759611440362772e-49,
    9.3914672150183386e-51,    3.6373944092486688e-52,    1.3809753674110744e-53,    5.1414897559998729e-55,
    1.8778547340252506e-56,    6.7307176495784682e-58,    2.3683056270825584e-59,    8.1834477035001447e-61,
    2.7777752895498218e-62,    9.2652108144642492e-64,    3.0376758430816852e-65,    9.7922300747386691e-67
},
// x = 3.8500
{
    1.2981132114507965e-01,    9.6211712323730839e-02,    5.4841155698016633e-02,    2.4989432196436488e-02,
    9.4058244317684406e-03,    3.0017906676271782e-03,    8.2927966711935947e-04,    2.0162555787349746e-04,
    4.3725545534304300e-05,    8.5517204492967120e-06,    1.5222498105023298e-06,    2.4853966473854722e-07,
    3.7467397778541272e-08,    5.2448739687987551e-09,    6.8516475060194151e-10,    8.3892730498416534e-11,
    9.6648426926136562e-12,    1.0512217045851952e-12,    1.0828174183915446e-13,    1.0591977819294971e-14,
    9.8638211123143266e-16,    8.7648842544653820e-17,    7.4469867067269369e-18,    6.0614077771566420e-19,
    4.7346043704544121e-20,    3.5547669305571126e-21,    2.5692332573560499e-22,    1.7900368482550048e-23,
    1.2037759848900770e-24,    7.8230524437219736e-26,    4.9185974106059011e-27,    2.9950052891843655e-28,
    1.7679664667844781e-29,    1.0126838769013620e-30,    5.6334862028872485e-32,    3.0460898904005894e-33,
    1.6021729680969903e-34,    8.2034833595416259e-36,    4.0917941603102212e-37,    1.9895038921181708e-38,
    9.4355245352710794e-40,    4.3675613208797128e-41,    1.9742986361573731e-42,    8.7201761166765506e-44,
    3.7653318694266626e-45,    1.5902444495433332e-46,    6.5722614151480639e-48,    2.6592245854898511e-49,
    1.0538411991287518e-50,    4.0922136589699564e-52,    1.5576868221918294e-53,    5.8144630900916495e-55,
    2.1291643231549914e-56,    7.6512996689422043e-58,    2.6992203580656539e-59,    9.3511227303606381e-61,
    3.1823760655748466e-62,    1.0642330101991168e-63,    3.4982429021619747e-65,    1.1306217498934380e-66
},
// x = 3.8600
{
    1.2947618385959808e-01,    9.6048120839987913e-02,    5.4827385279296592e-02,    2.5028191721728089e-02,
    9.4394728305668783e-03,    3.0190581789555902e-03,    8.3593915996804386e-04,    2.0371903916684158e-04,
    4.4284862687571232e-05,    8.6820791438595306e-06,    1.5492399587289839e-06,    2.5357159637023506e-07,
    3.8321120253490413e-08,    5.3778123450381033e-09,    7.0429815089227804e-10,    8.6453180303369649e-11,
    9.9850448289470256e-12,    1.0888074134079622e-12,    1.1243874882301922e-13,    1.1026660441197759e-14,
    1.0294852979641675e-15,    9.1712975775257144e-17,    7.8122517631156150e-18,    6.3750185292490751e-19,
    4.9923502112851491e-20,    3.7579141866372823e-21,    2.7230431013092760e-22,    1.9020809191900603e-23,
    1.2824175002196570e-24,    8.3555950314989948e-26,    5.2669643169614369e-27,    3.2154012466669726e-28,
    1.9029639758860478e-29,    1.0928230278551435e-30,    6.0949897126114869e-32,    3.3041414038674744e-33,
    1.7423917932975744e-34,    8.9444890819022503e-36,    4.4729302336669646e-37,    2.1804418887838434e-38,
    1.0367818798475388e-39,    4.8115191555886507e-41,    2.1806106406531695e-42,    9.6563458601676068e-44,
    4.1803561076211818e-45,    1.7700948813066223e-46,    7.3344962506469944e-48,    2.9753183269064978e-49,
    1.1821611887475223e-50,    4.6023862715333545e-52,    1.7564196236826578e-53,    6.5732665605947283e-55,
    2.4132616586760329e-56,    8.6946911391968296e-58,    3.0752548054075086e-59,    1.0681455533146520e-60,
    3.6445355130437689e-62,    1.2219445738589599e-63,    4.0270676090452381e-65,    1.3049099773716719e-66
},
// x = 3.8700
{
    1.2914275561045493e-01,    9.5884956118109668e-02,    5.4813332263083110e-02,    2.5066697380276184e-02,
    9.4730527638755158e-03,    3.0363421154494222e-03,    8.4262297836036203e-04,    2.0582565067301222e-04,
    4.4849138542485119e-05,    8.8139309773411540e-06,    1.5766091963657532e-06,    2.5868727613163920e-07,
    3.9191276203556501e-08,    5.5136572456152741e-09,    7.2390007135692471e-10,    8.9083067488446440e-11,
    1.0314776229834546e-11,    1.1276112650510410e-12,    1.1674153299051409e-13,    1.1477745503489841e-14,
    1.0743302421668978e-15,    9.5952240222961947e-17,    8.1942396895422342e-18,    6.7038336781975447e-19,
    5.2632896899221810e-20,    3.9720117314164331e-21,    2.8855625264864282e-22,    2.0207754574555434e-23,
    1.3659421575449305e-24,    8.9226672730884701e-26,    5.6388781851897522e-27,    3.4513027698934099e-28,
    2.0478327223733840e-29,    1.1790445493671839e-30,    6.5928048643044278e-32,    3.5832169718211592e-33,
    1.8944270007212857e-34,    9.7500195562002577e-36,    4.8883270390655193e-37,    2.3890822142246111e-38,
    1.1389185738793099e-39,    5.2991526167524244e-41,    2.4078059453283852e-42,    1.0689946134522468e-43,
    4.6397600481407162e-45,    1.9696928204023609e-46,    8.1826151532531238e-48,    3.3279386128515647e-49,
    1.3256801291914300e-50,    5.1744647490152952e-52,    1.9798445596098421e-53,    7.4285598526559686e-55,
    2.7343145877119535e-56,    9.8768627398158751e-58,    3.5024090031700880e-59,    1.2196555176615609e-60,
    4.1722470865406832e-62,    1.4024923925759362e-63,    4.6340336664562132e-65,    1.5054701589414163e-66
},
// x = 3.8800
{
    1.2881102363650959e-01,    9.5722217694286932e-02,    5.4798999646081564e-02,    2.5104950109130272e-02,
    9.5065638821867134e-03,    3.0536421349858468e-03,    8.4933102526807381e-04,    2.0794540084023935e-04,
    4.5418393153746576e-05,    8.9472865068136783e-06,    1.6043612904837170e-06,    2.6388776965953662e-07,
    4.0078119821515297e-08,    5.6524615312164238e-09,    7.4398030016387796e-10,    9.1784029991563887e-11,
    1.0654287344682002e-11,    1.1676685548354717e-12,    1.2119471085687393e-13,    1.1945796664251586e-14,
    1.1209814997059642e-15,    1.0037360034835514e-16,    8.5936607937803360e-18,    7.0485402873791189e-19,
    5.5480548759235764e-20,    4.1976139949859882e-21,    3.0572562926530437e-22,    2.1464935434147738e-23,
    1.4546372864060665e-24,    9.5263958594703324e-26,    6.0358541670216445e-27,    3.7037504101245224e-28,
    2.2032624809147691e-29,    1.2717903437408855e-30,    7.1296708467439610e-32,    3.8849612013674371e-33,
    2.0592359705613331e-34,    1.0625483558738513e-35,    5.3409527639411173e-37,    2.6170083906915535e-38,
    1.2507844749553982e-39,    5.8346156557982183e-41,    2.6579300294913646e-42,    1.1830797376415392e-43,
    5.1481435425524806e-45,    2.1911415405830579e-46,    9.1260118035598779e-48,    3.7211856060769619e-49,
    1.4861479402360268e-50,    5.8157554868959621e-52,    2.2309474455624386e-53,    8.3922909731165972e-55,
    3.0970068227242944e-56,    1.1215808527652064e-57,    3.9874607171517829e-59,    1.3921461426702264e-60,
    4.7745873477866825e-62,    1.6091058148255664e-63,    5.3304224016277651e-65,    1.7361730976062601e-66
},
// x = 3.8900
{
    1.2848097529889319e-01,    9.5559905086995425e-02,    5.4784390398896710e-02,    2.5142950846768340e-02,
    9.5400058417300041e-03,    3.0709578967503476e-03,    8.5606320310434194e-04,    2.1007829768699267e-04,
    4.5992646470694328e-05,    9.0821562983542441e-06,    1.6325000262905593e-06,    2.6917415128438177e-07,
    4.0981908156683142e-08,    5.7942788121250492e-09,    7.6454879231905968e-10,    9.4557738281155208e-11,
    1.1003834294430146e-11,    1.2090154749354097e-12,    1.2580302894445620e-13,    1.2431395000993627e-14,
    1.1695057982476259e-15,    1.0498427396209652e-16,    9.0112531653244879e-18,    7.4098541721178493e-19,
    5.8473060194951406e-20,    4.4353016455586901e-21,    3.2386124289662868e-22,    2.2796279614904788e-23,
    1.5488061820365763e-24,    1.0169031514004933e-25,    6.4594999638488945e-27,    3.9738511568360731e-28,
    2.3699889795611631e-29,    1.3715329805854256e-30,    7.7085245682690855e-32,    4.2111419978503163e-33,
    2.2378505354632268e-34,    1.1576725644256279e-35,    5.8340230744858281e-37,    2.8659404271480666e-38,
    1.3732746858646543e-39,    6.4224437280919475e-41,    2.9332220301267332e-42,    1.3089677686487464e-43,
    5.7105681102567327e-45,    2.4367614177718039e-46,    1.0175076394671104e-47,    4.1596061923374094e-49,
    1.6655102330383954e-50,    6.5344030128765936e-52,    2.5130652368974633e-53,    9.4778476672394383e-55,
    3.5066000705701617e-56,    1.2731795520487545e-57,    4.5380636494223954e-59,    1.5884523391381404e-60,
    5.4618590796310695e-62,    1.8454599651043993e-63,    6.1291056720493712e-65,    2.0014467759251157e-66
},
// x = 3.9000
{
    1.2815259807961799e-01,    9.5398017797403048e-02,    5.4769507466231021e-02,    2.5180700533004318e-02,
    9.5733783044283788e-03,    3.0882890612465353e-03,    8.6281941373302334e-04,    2.1222434880312455e-04,
    4.6571918336390315e-05,    9.2185509265514754e-06,    1.6610292070369601e-06,    2.7454750404476302e-07,
    4.1902901131947550e-08,    5.9391634553556763e-09,    7.8561567179286586e-10,    9.7405895870264061e-11,
    1.1363678977946366e-11,    1.2516891337947739e-12,    1.3057136696762924e-13,    1.2935139486496699e-14,
    1.2199721026622636e-15,    1.0979174056008110e-16,    9.4477836664969606e-18,    7.7885210050085726e-19,
    6.1617327127651402e-20,    4.6856827431859297e-21,    3.4301433214307361e-22,    2.4205921754417993e-23,
    1.6487689397428418e-24,    1.0852955817646090e-25,    6.9115211758694721e-27,    4.2627824619474552e-28,
    2.5487968108193759e-29,    1.4787777248495671e-30,    8.3325142829400829e-32,    4.5636594063226587e-33,
    2.4313825275758652e-34,    1.2610059834494920e-35,    6.3710209422280773e-37,    3.1377461377949316e-38,
    1.5073637463980280e-39,    7.0675875836442809e-41,    3.2361324429642079e-42,    1.4478413081260503e-43,
    6.3326017599417727e-45,    2.7091116265175262e-46,    1.1341298067544008e-47,    4.6482411801097533e-49,
    1.8659295481779529e-50,    7.3394833594630961e-52,    2.8299261603966075e-53,    1.0700226128292498e-54,
    3.9690034373074316e-56,    1.4447643234028965e-57,    5.1628577148664206e-59,    1.8117825812562080e-60,
    6.2457528295248616e-62,    2.1157357784902053e-63,    7.0447647460518093e-65,    2.3063546746645244e-66
},
// x = 3.9100
{
    1.2782587958020417e-01,    9.5236555309769155e-02,    5.4754353767082029e-02,    2.5218200108897002e-02,
    9.6066809378546562e-03,    3.1056352903057337e-03,    8.6959955847791212e-04,    2.1438356135103839e-04,
    4.7156228486716113e-05,    9.3564809740119168e-06,    1.6899526539216324e-06,    2.8000891970135305e-07,
    4.2841361560732700e-08,    6.0871705918090516e-09,    8.0719123366252609e-10,    1.0033023983635048e-10,
    1.1734089179952109e-11,    1.2957275758850651e-12,    1.3550474108324381e-13,    1.3457647475852489e-14,
    1.2724516821576398e-15,    1.1480374990280793e-16,    9.9040489554037299e-18,    8.1853174598760963e-19,
    6.4920550949083985e-20,    4.9493939402656676e-21,    3.6323868474919191e-22,    2.5698213486339150e-23,
    1.7548633300991851e-24,    1.1580688387020269e-25,    7.3937269426744005e-27,    4.5717964945112252e-28,
    2.7405225175491960e-29,    1.5940646923049390e-30,    9.0050141110656807e-32,    4.9445550580544196e-33,
    2.6410297207275592e-34,    1.3732305800916733e-35,    6.9557179941717127e-37,    3.4344533622051470e-38,
    1.6541124243155556e-39,    7.7754499401857341e-41,    3.5693423835462026e-42,    1.6009975954856005e-43,
    7.0203680155828739e-45,    3.0110139334890413e-46,    1.2637377539327110e-47,    5.1926773319547791e-49,
    2.0898088286081578e-50,    8.2411075304501381e-52,    3.1856942998111211e-53,    1.2076218917572200e-54,
    4.4908509366035896e-56,    1.6389036667421774e-57,    5.8715928334338972e-59,    2.0657668846466718e-60,
    7.1395291939874441e-62,    2.4246879188424933e-63,    8.0941385683463177e-65,    2.6566848360517448e-66
},
// x = 3.9200
{
    1.2750080752032461e-01,    9.5075517091837508e-02,    5.4738932194938750e-02,    2.5255450516660453e-02,
    9.6399134151879430e-03,    3.1229962470963168e-03,    8.7640353813195164e-04,    2.1655594206688533e-04,
    4.7745596549481930e-05,    9.4959570308667608e-06,    1.7192742059950620e-06,    2.8555949875035478e-07,
    4.3797555163898950e-08,    6.2383561234483076e-09,    8.2928594627026815e-10,    1.0333254134683521e-10,
    1.2115338680500109e-11,    1.3411698018087438e-12,    1.4060830720775610e-13,    1.3999555204923532e-14,
    1.3270181791801737e-15,    1.2002833084518661e-16,    1.0380876541645875e-17,    8.6010523955806620e-19,
    6.8390251026199308e-20,    5.2271017305748860e-21,    3.8459075596478610e-22,    2.7277734112218925e-23,
    1.8674457168164748e-24,    1.2354894422425569e-25,    7.9080358901782183e-27,    4.9022246382929166e-28,
    2.9460578636027141e-29,    1.7179711400662740e-30,    9.7296395098474725e-32,    5.3560222614894147e-33,
    2.8680821945731172e-34,    1.4950827718042959e-35,    7.5921974985734800e-37,    3.7582631559330718e-38,
    1.8146750647135208e-39,    8.5519252750332487e-41,    3.9357845406642745e-42,    1.7698592470408910e-43,
    7.7805995276042269e-45,    3.3455787838086974e-46,    1.4077350905464208e-47,    5.7991047062200403e-49,
    2.3398173553397242e-50,    9.2505361191933140e-52,    3.5850191148812229e-53,    1.3624624218729977e-54,
    5.0795880210511985e-56,    1.8584876628551501e-57,    6.6752678496411763e-59,    2.3545107865583209e-60,
    8.1582244377678825e-62,    2.7777215895167922e-63,    9.2963052568857843e-65,    3.0590511095561756e-66
},
// x = 3.9300
{
    1.2717736973646665e-01,    9.4914902595221201e-02,    5.4723245617977180e-02,    2.5292452699575928e-02,
    9.6730754151701705e-03,    3.1403715961328049e-03,    8.8323125296639290e-04,    2.1874149726178557e-04,
    4.8340042043547173e-05,    9.6369896942787877e-06,    1.7489977200619248e-06,    2.9120035043644030e-07,
    4.4771750586574490e-08,    6.3927767304957972e-09,    8.5191045339724716e-10,    1.0641460619041101e-10,
    1.2507707366015676e-11,    1.3880557887526126e-12,    1.4588736440208179e-13,    1.4561518300443448e-14,
    1.3837476801238972e-15,    1.2547380042314775e-16,    1.0879125875714926e-17,    9.0365680809143624e-19,
    7.2034277674842345e-20,    5.5195037486188146e-21,    4.0712979200274474e-22,    2.8949301762459106e-23,
    1.9868920192205613e-24,    1.3178392643570753e-25,    8.4564823995070084e-27,    5.2554822452993886e-28,
    3.1663532996538790e-29,    1.8511139001626470e-30,    1.0510263753182286e-31,    5.8004167794984722e-33,
    3.1139291492937196e-34,    1.6273575975782010e-35,    8.2848791063144546e-37,    4.1115640254961813e-38,
    1.9903075418822468e-39,    9.4034429907873242e-41,    4.3386659653321399e-42,    1.9559859661828749e-43,
    8.6206966830298726e-45,    3.7162338936654511e-46,    1.5676725687482836e-47,    6.4743798336356682e-49,
    2.6189193961494316e-50,    1.0380306243942437e-51,    4.0330904234748497e-53,    1.5366478778057666e-54,
    5.7435681569397111e-56,    2.1067671730195166e-57,    7.5862863776644491e-59,    2.6826560591493576e-60,
    9.3188823571045522e-62,    3.1809793712026606e-63,    1.0673001168015305e-64,    3.5210082057385251e-66
},
// x = 3.9400
{
    1.2685555418061134e-01,    9.4754711255781707e-02,    5.4707296879254727e-02,    2.5329207601905171e-02,
    9.7061666220627386e-03,    3.1577610032847204e-03,    8.9008260273991413e-04,    2.2094023282307515e-04,
    4.8939584377952672e-05,    9.7795895679494607e-06,    1.7791270705821933e-06,    2.9693259276517688e-07,
    4.5764219414916660e-08,    6.5504898786498544e-09,    8.7507557645319646e-10,    1.0957827531414346e-10,
    1.2911481341915765e-11,    1.4364265112956165e-12,    1.5134735832526814e-13,    1.5144212301981541e-14,
    1.4427187878874454e-15,    1.3114877320355398e-16,    1.1399689473022831e-17,    9.4927414618667494e-19,
    7.5860825618330100e-20,    5.8273301211483588e-21,    4.3091795879547278e-22,    3.0717985067127510e-23,
    2.1135987213577996e-24,    1.4054163631670819e-25,    9.0412232141999246e-27,    5.6330736589714216e-28,
    3.4024216352217359e-29,    1.9941519646224001e-30,    1.1351035483134904e-31,    6.2802683373551260e-33,
    3.3800662012724861e-34,    1.7709131951783046e-35,    9.0385454762229773e-37,    4.4969472870116361e-38,
    2.1823758610814776e-39,    1.0337014229917602e-40,    4.7814928501049303e-42,    2.1610873092247730e-43,
    9.5487916644418481e-45,    4.1267555818162593e-46,    1.7452630297699442e-47,    7.2260953034475930e-49,
    2.9304058422242036e-50,    1.1644372083437203e-51,    4.5356994316074445e-53,    1.7325317133851907e-54,
    6.4921605750937714e-56,    2.3873976876122749e-57,    8.6186315781993358e-59,    3.0554489733735240e-60,
    1.0640815649560623e-61,    3.6414393624817665e-63,    1.2248982418411284e-64,    4.0511823951677941e-66
},
// x = 3.9500
{
    1.2653534891892923e-01,    9.4594942494000092e-02,    5.4691088796903813e-02,    2.5365716168805306e-02,
    9.7391867256032688e-03,    3.1751641357852254e-03,    8.9695748670770006e-04,    2.2315215421558036e-04,
    4.9544242851065760e-05,    9.9237672616265455e-06,    1.8096661495709889e-06,    3.0275735251494947e-07,
    4.6775236192803244e-08,    6.7115538263214653e-09,    8.9879231668181176e-10,    1.1282542536639406e-10,
    1.3326953046820682e-11,    1.4863239625756345e-12,    1.5699388475809617e-13,    1.5748333196000380e-14,
    1.5040126963202552e-15,    1.3706217090405852e-16,    1.1943494073541445e-17,    9.9704854725737773e-19,
    7.9878447947331445e-20,    6.1513448727595192e-21,    4.5602047625915525e-22,    3.2589115358197028e-23,
    2.2479839298295261e-24,    1.4985358597348048e-25,    9.6645444028569877e-27,    6.0365975214460859e-28,
    3.6553419284749625e-29,    2.1477892310073152e-30,    1.2256397399260828e-31,    6.7982929085903398e-33,
    3.6681031921234799e-34,    1.9266756058329547e-35,    9.8583709216723683e-37,    4.9172236335299892e-38,
    2.3923654612390016e-39,    1.1360282634822818e-40,    5.2680974661062880e-42,    2.3870365986501012e-43,
    1.0573818446571811e-44,    4.5813030934140231e-46,    1.9423978200269921e-47,    8.0626563884472844e-49,
    3.2779291346068306e-50,    1.3058260425447185e-51,    5.0993064567777328e-53,    1.9527460015511856e-54,
    7.3358704543867648e-56,    2.7044883597677782e-57,    9.7880621044256315e-59,    3.4788170275417098e-60,
    1.2145900447668688e-61,    4.1670260567409854e-63,    1.4052434372147521e-64,    4.6594199276714129e-66
},
// x = 3.9600
{
    1.2621674213049305e-01,    9.4435595715342047e-02,    5.4674624164324850e-02,    2.5401979346245010e-02,
    9.7721354209624328e-03,    3.1925806622395058e-03,    9.0385580363046964e-04,    2.2537726648291361e-04,
    5.0154036649736171e-05,    1.0069533390611925e-05,    1.8406188664971283e-06,    3.0867576524836374e-07,
    4.7805078438449475e-08,    6.8760276318899264e-09,    9.2307185738177992e-10,    1.1615796924558368e-10,
    1.3754421368371677e-11,    1.5377911758196826e-12,    1.6283269319771589e-13,    1.6374597962236417e-14,
    1.5677132665997336e-15,    1.4322323228968068e-16,    1.2511501838048475e-17,    1.0470750391298737e-18,
    8.4096070597951000e-20,    6.4923473875507268e-21,    4.8250575828255658e-22,    3.4568299425600883e-23,
    2.3904884825444101e-24,    1.5975308594649110e-25,    1.0328868695174085e-26,    6.4677523800144556e-28,
    3.9262636060171141e-29,    2.3127774178284640e-30,    1.3231106155824338e-31,    7.3574058287692448e-33,
    3.9797725455243027e-34,    2.0956439292618553e-35,    1.0749952225335037e-36,    5.3754410022657637e-38,
    2.6218912734141971e-39,    1.2481579373088152e-40,    5.8026674386512355e-42,    2.6358860831704500e-43,
    1.1705589261610976e-44,    5.0844561922244778e-46,    2.1611648165841780e-47,    8.9933653979984597e-49,
    3.6655418118198989e-50,    1.4639242783052309e-51,    5.7311160568212092e-53,    2.2002335099225692e-54,
    8.2864729317004230e-56,    3.0626568201828344e-57,    1.1112331711450858e-58,    3.9594551622712365e-60,
    1.3858908114824829e-61,    4.7667355661676422e-63,    1.6115435297591678e-64,    5.3569555155571804e-66
},
// x = 3.9700
{
    1.2589972210600667e-01,    9.4276670310616331e-02,    5.4657905750377177e-02,    2.5437998080922113e-02,
    9.8050124087008996e-03,    3.2100102526329303e-03,    9.1077745178346273e-04,    2.2761557424879616e-04,
    5.0768984848464661e-05,    1.0216898575269921e-05,    1.8719891481804140e-06,    3.1468897532314202e-07,
    4.8854026660951744e-08,    7.0439711609773675e-09,    9.4792556614346475e-10,    1.1957785665482894e-10,
    1.4194191760669384e-11,    1.5908722462421538e-12,    1.6886969052445339e-13,    1.7023745132638953e-14,
    1.6339071055820759e-15,    1.4964152335302986e-16,    1.3104711582002773e-17,    1.0994525242831956e-18,
    8.8523007365441465e-20,    6.8511739288804513e-21,    5.1044555866489485e-22,    3.6661432850368686e-23,
    2.5415771106689250e-24,    1.7027534202392075e-25,    1.1036763210152928e-26,    6.9283426086563397e-28,
    4.2164108254956345e-29,    2.4899191597999606e-30,    1.4280253541041152e-31,    7.9607357902895563e-33,
    4.3169382085023924e-34,    2.2788958534521594e-35,    1.1719341779171714e-36,    5.8749038395102090e-38,
    2.8727085939813036e-39,    1.3709982772535863e-40,    6.3897775569307583e-42,    2.9098834522894269e-43,
    1.2954878110102485e-44,    5.6412563218670058e-46,    2.4038682145369776e-47,    1.0028514513382828e-48,
    4.0977390423449345e-50,    1.6406525789414269e-51,    6.4391603499410413e-53,    2.4782833644859522e-54,
    9.3571624837898297e-56,    3.4670904357446322e-57,    1.2611435309330558e-58,    4.5049226034981756e-60,
    1.5807878894858459e-61,    5.4507770019391089e-63,    1.8474481180938056e-64,    6.1566035266375923e-66
},
// x = 3.9800
{
    1.2558427724655027e-01,    9.4118165656326538e-02,    5.4640936299570494e-02,    2.5473773320182740e-02,
    9.8378173947264533e-03,    3.2274525783390052e-03,    9.1772232896538395e-04,    2.2986708171840442e-04,
    5.1389106408583332e-05,    1.0365873440536032e-05,    1.9037809386876882e-06,    3.2079813590250066e-07,
    4.9922364376754490e-08,    7.2154450937414837e-09,    9.7336499710118874e-10,    1.2308707466247136e-10,
    1.4646576363346655e-11,    1.6456123533156245e-12,    1.7511094474191972e-13,    1.7696535363104709e-14,
    1.7026836461700362e-15,    1.5632694778524010e-16,    1.3724160048094684e-17,    1.1542839248731994e-18,
    9.3168975471477229e-20,    7.2286992193333915e-21,    5.3991512323527692e-22,    3.8874713938998645e-23,
    2.7017396561500708e-24,    1.8145755694990825e-25,    1.1790947596154557e-26,    7.4202846613225353e-28,
    4.5270870945515252e-29,    2.6800712934370586e-30,    1.5409289016778055e-31,    8.6116397745315035e-33,
    4.6816052161566939e-34,    2.4775935852130128e-35,    1.2773083217601242e-36,    6.4191938680486509e-38,
    3.1467248358941789e-39,    1.5055382937385185e-40,    7.0344243289266522e-42,    3.2114898220220185e-43,
    1.4333511943850935e-44,    6.2572516633914311e-46,    2.6730502428314703e-47,    1.1179487930964387e-48,
    4.5795055409754485e-50,    1.8381461752869358e-51,    7.2323913923677047e-53,    2.7905706911101476e-54,
    1.0562719394209096e-55,    3.9236147484499291e-57,    1.4307884557272833e-58,    5.1237516088799647e-60,
    1.8024542557358903e-61,    6.2307320418812668e-63,    2.1171079561275857e-64,    7.0729748730406749e-66
},
// x = 3.9900
{
    1.2527039606234147e-01,    9.3960081115016539e-02,    5.4623718532253861e-02,    2.5509306011941554e-02,
    9.8705500902510988e-03,    3.2449073121270694e-03,    9.2469033250729626e-04,    2.3213179267973967e-04,
    5.2014420177447700e-05,    1.0516468615426214e-05,    1.9359981992276271e-06,    3.2700440896501458e-07,
    5.1010378126039455e-08,    7.3905109321861157e-09,    9.9940189320107485e-10,    1.2668764826852745e-10,
    1.5111894122290406e-11,    1.7020577834188650e-12,    1.8156268879158002e-13,    1.8393752018249594e-14,
    1.7741352297418052e-15,    1.6328975774483285e-16,    1.4370923218543601e-17,    1.2116763326870661e-18,
    9.8044111703468012e-20,    7.6258380830733998e-21,    5.7099334839451123e-22,    4.1214658284153490e-23,
    2.8714923472826281e-24,    1.9333903725881656e-25,    1.2594302603386342e-26,    7.9456136744625664e-28,
    4.8596801603348725e-29,    2.8841483440845392e-30,    1.6624043701685065e-31,    9.3137189811077134e-33,
    5.0759299212698855e-34,    2.6929902092627414e-35,    1.3918249723383536e-36,    7.0121924694276233e-38,
    3.4460122261174675e-39,    1.6528551745548472e-40,    7.7420635096407869e-42,    3.5433993180879218e-43,
    1.5854470200958883e-44,    6.9385464454632104e-46,    2.9715149902446255e-47,    1.2458874216339175e-48,
    5.1163673067258224e-50,    2.0587781441814112e-51,    8.1207835689385022e-53,    3.1412006656506250e-54,
    1.1919695203803529e-55,    4.4387699128544897e-57,    1.6227016449374071e-58,    5.8255695417781612e-60,
    2.0544791795736427e-61,    7.1197349655869643e-63,    2.4252421237661040e-64,    8.1227229645567669e-66
},
// x = 4.0000
{
    1.2495806717151219e-01,    9.3802416035609751e-02,    5.4606255144804869e-02,    2.5544597104603672e-02,
    9.9032102117483992e-03,    3.2623741281697922e-03,    9.3168135928146836e-04,    2.3440971050502066e-04,
    5.2644944887640942e-05,    1.0668694732546653e-05,    1.9686449080443222e-06,    3.3330896531396268e-07,
    5.2118357489035791e-08,    7.5692310074890921e-09,    1.0260481884844151e-09,    1.3038164097708247e-10,
    1.5590470912026154e-11,    1.7602559528666525e-12,    1.8823132444293958e-13,    1.9116201769461926e-14,
    1.8483571906858594e-15,    1.7054056493187149e-16,    1.5046117668239406e-17,    1.2717411641782525e-18,
    1.0315898914493235e-19,    8.0435471528316692e-21,    6.0376294632859928e-22,    4.3688113977724010e-23,
    3.0513791348945363e-24,    2.0596130547689242e-25,    1.3449879110372215e-26,    8.5064904371597986e-28,
    5.2156671845532893e-29,    3.1031262260703073e-30,    1.7930755885523103e-31,    1.0070835817574213e-32,
    5.5022309328875948e-34,    2.9264365054344028e-35,    1.5162485198093242e-36,    7.6581048014529407e-38,
    3.7728215223679110e-39,    1.8141218657922778e-40,    8.5186508489361958e-42,    3.9085603933356331e-43,
    1.7531993431194504e-44,    7.6918548948556799e-46,    3.3023545397842234e-47,    1.3880589857355634e-48,
    5.7144486622602264e-50,    2.3051851374588613e-51,    9.1154470495453503e-53,    3.5347574486591392e-54,
    1.3446619248069580e-55,    5.0198960408731180e-57,    1.8397338733982973e-58,    6.6212358627609284e-60,
    2.3409214821400313e-61,    8.1326757153376803e-63,    2.7772139804484941e-64,    9.3248225258383517e-66
},
// x = 4.0100
{
    1.2464727929890099e-01,    9.3645169753742166e-02,    5.4588548809817096e-02,    2.5579647546987678e-02,
    9.9357974809109219e-03,    3.2798527020504505e-03,    9.3869530571018332e-04,    2.3670083815210076e-04,
    5.3280699156190087e-05,    1.0822562427604239e-05,    2.0017250603096410e-06,    3.3971298458615854e-07,
    5.3246595102249371e-08,    7.7516684873470358e-09,    1.0533160103865256e-09,    1.3417115537465330e-10,
    1.6082639659780052e-11,    1.8202554313260943e-12,    1.9512342626053055e-13,    1.9864715206485809e-14,
    1.9254479430875268e-15,    1.7809035197492961e-16,    1.5750901959851482e-17,    1.3345943207360252e-18,
    1.0852463451652267e-19,    8.4828266438524707e-21,    6.3831061715207874e-22,    4.6302277503301813e-23,
    3.2419730918296095e-24,    2.1936821794326487e-25,    1.4360907626958806e-26,    9.1052087481414004e-28,
    5.5966202197995998e-29,    3.3380461683201057e-30,    1.9336098167501906e-31,    1.0887132016784900e-32,
    5.9630008107206208e-34,    3.1793882555327733e-35,    1.6514048502219569e-36,    8.3614857798871554e-38,
    4.1295968277289950e-39,    1.9906152803651122e-40,    9.3706863248989773e-42,    4.3101990274224939e-43,
    1.9381701812770245e-44,    8.5245602490478698e-46,    3.6689776277576181e-47,    1.5460015096287035e-48,
    6.3805351210380903e-50,    2.5802958115092043e-51,    1.0228753471727044e-52,    3.9763585305723965e-54,
    1.5164228610912764e-55,    5.6752284630673564e-57,    2.0850916443752028e-58,    7.5229958130674068e-60,
    2.6663694363452528e-61,    9.2864288525995905e-63,    3.1791170211816158e-64,    1.0700885560961074e-65
},
// x = 4.0200
{
    1.2433802127486081e-01,    9.3488341592089214e-02,    5.4570602176286780e-02,    2.5614458288249951e-02,
    9.9683116246077307e-03,    3.2973427107699667e-03,    9.4573206777448689e-04,    2.3900517816590586e-04,
    5.3921701483793722e-05,    1.0978082338917518e-05,    2.0352426680143892e-06,    3.4621765526025723e-07,
    5.4395386674608885e-08,    7.9378873833363921e-09,    1.0812176820510288e-09,    1.3805833371454152e-10,
    1.6588740471231755e-11,    1.8821059656240908e-12,    2.0224574564887707e-13,    2.0640147462785616e-14,
    2.0055090696136888e-15,    1.8595048413853878e-16,    1.6486478082054406e-17,    1.4003563543477513e-18,
    1.1415254614786216e-19,    8.9447221971930824e-21,    6.7472722824839552e-22,    4.9064710336125766e-23,
    3.4438778785158357e-24,    2.3360608851318467e-25,    1.5330808297453976e-26,    9.7442031798754558e-28,
    6.0042120037209171e-29,    3.5900188784424909e-30,    2.0847206316764237e-31,    1.1767047953108793e-32,
    6.4609185651705274e-34,    3.4534140734440139e-35,    1.7981860983939936e-36,    9.1272680624836535e-38,
    4.5189915875041950e-39,    2.1837251840873415e-40,    1.0305262149736462e-41,    4.7518439689348811e-43,
    2.1420724429633236e-44,    9.4447792897121017e-46,    4.0751410633214528e-47,    1.7214143222172033e-48,
    7.1221426589069204e-50,    2.8873622322012986e-51,    1.1474475127599179e-52,    4.4717150681719058e-54,
    1.7095724074156237e-55,    6.4140040266202994e-57,    2.3623804296219336e-58,    8.5446527697862647e-60,
    3.0360081137872103e-61,    1.0600111628183840e-62,    3.6378718938239775e-64,    1.2275519293161513e-65
},
// x = 4.0300
{
    1.2403028203408169e-01,    9.3331930860687098e-02,    5.4552417869798482e-02,    2.5649030277810580e-02,
    1.0000752374842119e-02,    3.3148438327537670e-03,    9.5279154102290034e-04,    2.4132273267989465e-04,
    5.4567970254061977e-05,    1.1135265106928473e-05,    2.0692017598582673e-06,    3.5282417466454770e-07,
    5.5565031003528681e-08,    8.1279525582904663e-09,    1.1097657246595365e-09,    1.4204535850720330e-10,
    1.7109120757972956e-11,    1.9458585039507187e-12,    2.0960521497664756e-13,    2.1443378854947152e-14,
    2.0886454126429370e-15,    1.9413272135895075e-16,    1.7254092932047596e-17,    1.4691526388163827e-18,
    1.2005471260095447e-19,    9.4303267948525120e-21,    7.1310800108393742e-22,    5.1983356279633802e-23,
    3.6577292775210586e-24,    2.4872381841784186e-25,    1.6363201428585347e-26,    1.0426057270957853e-27,
    6.4402220884480595e-29,    3.8602289590017696e-30,    2.2471709958765443e-31,    1.2715343232997750e-32,
    6.9988630159000941e-34,    3.7502037942972413e-35,    1.9575557530544457e-36,    9.9607921832887335e-38,
    4.9438858588872939e-39,    2.3949638130648536e-40,    1.1330114857944179e-41,    5.2373541932528334e-43,
    2.3667840251952909e-44,    1.0461432895988657e-45,    4.5249841655956173e-47,    1.9161744614148712e-48,
    7.9475940221000880e-50,    3.2299945570343534e-51,    1.2867939062840093e-52,    5.0271988535850678e-54,
    1.9267054917966792e-55,    7.2465796731267591e-57,    2.6756530260488829e-58,    9.7017614809038547e-60,
    3.4556950803278582e-61,    1.2095374773628151e-62,    4.1613359922819484e-64,    1.4076731519268457e-65
},
// x = 4.0400
{
    1.2372405061442839e-01,    9.3175936857247760e-02,    5.4533998492709741e-02,    2.5683364465280207e-02,
    1.0033119468709358e-02,    3.3323557478583902e-03,    9.5987362058007916e-04,    2.4365350341754236e-04,
    5.5219523732768404e-05,    1.1294121373714955e-05,    2.1036063811386735e-06,    3.5953374898422024e-07,
    5.6755829990885417e-08,    8.3219297336918232e-09,    1.1389728597766786e-09,    1.4613445311665720e-10,
    1.7644135366685546e-11,    2.0115652204632449e-12,    2.1720895178120429e-13,    2.2275315536374375e-14,
    2.1749651676893412e-15,    2.0264923061620685e-16,    1.8055039843573527e-17,    1.5411135466998360e-18,
    1.2624363196653529e-19,    9.9407827492832062e-21,    7.5355270578199934e-22,    5.5066559568851144e-23,
    3.8841968001155797e-24,    2.6477303256697980e-25,    1.7461918568103453e-26,    1.1153512169026104e-27,
    6.9065433236017934e-29,    4.1499395904409689e-30,    2.4217765197212054e-31,    1.3737118639901921e-32,
    7.5799270651762893e-34,    4.0715774608115975e-35,    2.1305541392385243e-36,    1.0867838995598601e-37,
    5.4074049506784629e-39,    2.6259762802680592e-40,    1.2453681810915236e-41,    5.7709487636155869e-43,
    2.6143631842068329e-44,    1.1584323159935698e-45,    5.0230664985459459e-47,    2.1323546946272962e-48,
    8.8661027639351145e-50,    3.6121993260446072e-51,    1.4426197637683713e-52,    5.6499166236782363e-54,
    2.1707235731782528e-55,    8.1845646753490392e-57,    3.0294636179866849e-58,    1.1011844642391952e-59,
    3.9320454483039298e-61,    1.3796730056695043e-62,    4.7584272149953446e-64,    1.6136389505558766e-65
},
// x = 4.0500
{
    1.2341931615579273e-01,    9.3020358867468303e-02,    5.4515346624334750e-02,    2.5717461800388416e-02,
    1.0065412648354723e-02,    3.3498781373779234e-03,    9.6697820115542463e-04,    2.4599749169384403e-04,
    5.5876380067113379e-05,    1.1454661782503893e-05,    2.1384605936383266e-06,    3.6634759326811510e-07,
    5.7968088658907765e-08,    8.5198854970796019e-09,    1.1688520117104282e-09,    1.5032788236295221e-10,
    1.8194146710053311e-11,    2.0792795402955467e-12,    2.2506426305477047e-13,    2.3136890165543273e-14,
    2.2645799791686607e-15,    2.1151259865065863e-16,    1.8890660161666392e-17,    1.6163746321432915e-18,
    1.3273233185535002e-19,    1.0477283769920945e-20,    7.9616586375289245e-22,    5.8323083771979724e-23,
    4.1239853679827527e-24,    2.8180822259284647e-25,    1.8631014161037435e-26,    1.1929475747517494e-27,
    7.4051887121326079e-29,    4.4604974958989380e-30,    2.6094089287451621e-31,    1.4837839518292634e-32,
    8.2074329457142110e-34,    4.4194949474407594e-35,    2.3183043046146674e-36,    1.1854664593117678e-37,
    5.9129395373911310e-39,    2.8785518335417290e-40,    1.3685162478454902e-41,    6.3572392981058483e-43,
    2.8870652894192432e-44,    1.2824217653683104e-45,    5.5744092089659245e-47,    2.3722433094684123e-48,
    9.8878657684112473e-50,    4.0384217254725947e-51,    1.6168217256015413e-52,    6.3477924909709722e-54,
    2.4448698731218170e-55,    9.2409680625547743e-57,    3.4289281999377132e-58,    1.2494635561918893e-59,
    4.4725274117958937e-61,    1.5731919130342977e-62,    5.4392636738144111e-64,    1.8490739323568985e-65
},
// x = 4.0600
{
    1.2311606789896039e-01,    9.2865196165335062e-02,    5.4496464821126599e-02,    2.5751323232913147e-02,
    1.0097631660931512e-02,    3.3674106840502031e-03,    9.7410517705165075e-04,    2.4835469841684010e-04,
    5.6538557284999932e-05,    1.1616896977185372e-05,    2.1737684755117476e-06,    3.7326693143494897e-07,
    5.9202115165977598e-08,    8.7218873094712982e-09,    1.1994163098876812e-09,    1.5462795313072050e-10,
    1.8759524899421674e-11,    2.1490561649777536e-12,    2.3317864961346002e-13,    2.4029062589079658e-14,
    2.3576050385568868e-15,    2.2073584503226673e-16,    1.9762344865393243e-17,    1.6950768197800462e-18,
    1.3953439010699204e-19,    1.1041077109453319e-20,    8.4105695868675800e-22,    6.1762131522735555e-23,
    4.3778370733456475e-24,    2.9988689694690225e-25,    1.9874777811849340e-26,    1.2757032220721152e-27,
    7.9382986592378425e-29,    4.7933382039882854e-30,    2.8109997483775241e-31,    1.6023360686586649e-32,
    8.8849485064602751e-34,    4.7960662655595719e-35,    2.5220183382253658e-36,    1.2928037890777915e-37,
    6.4641673597108919e-39,    3.1546360320228078e-40,    1.5034584885941167e-41,    7.0012652617282641e-43,
    3.1873610809181746e-44,    1.4192941488911019e-45,    6.1845403001884905e-47,    2.6383658429996113e-48,
    1.1024165090876270e-49,    4.5135922237544498e-51,    1.8115087140049766e-52,    7.1296593578337301e-54,
    2.7527685446959319e-55,    1.0430362930457101e-56,    3.8797920877786345e-58,    1.4172349967672430e-59,
    5.0855695233657822e-61,    1.7932328747959433e-62,    6.2153213575682172e-64,    2.1180993390931077e-65
},
// x = 4.0700
{
    1.2281429518449194e-01,    9.2710448013421584e-02,    5.4477355616859333e-02,    2.5784949712611600e-02,
    1.0129776258559283e-02,    3.3849530720628171e-03,    9.8125444217329123e-04,    2.5072512408916080e-04,
    5.7206073294320450e-05,    1.1780837601827363e-05,    2.2095341211705617e-06,    3.8029299627901148e-07,
    6.0458220822339503e-08,    8.9280035127983221e-09,    1.2306790912449993e-09,    1.5903701498382985e-10,
    1.9340647879218945e-11,    2.2209510982708847e-12,    2.4155981055040510e-13,    2.4952820539929057e-14,
    2.4541591849915925e-15,    2.3033243559115066e-16,    2.0671536239877071e-17,    1.7773665998809495e-18,
    1.4666395623958055e-19,    1.1633465792632399e-20,    8.8834065622616801e-22,    6.5393365117189243e-23,
    4.6465330209083859e-24,    3.1906973837400543e-25,    2.1197747182018376e-26,    1.3639452282753103e-27,
    8.5081486356320080e-29,    5.1499916264640199e-30,    3.0275442190005913e-31,    1.7299952974078087e-32,
    9.6163046036782881e-34,    5.2035625957310119e-35,    2.7430041520407172e-36,    1.4095281059387172e-37,
    7.0650766314132130e-39,    3.4563439129917413e-40,    1.6512877646272906e-41,    7.7085323204958074e-43,
    3.5179565606189792e-44,    1.5703477864626084e-45,    6.8595442038113080e-47,    2.9335089332433986e-48,
    1.2287480024501120e-49,    5.0431780183171696e-51,    2.0290250216113522e-52,    8.0053602644268729e-54,
    3.0984682060274533e-55,    1.1769069515747063e-56,    4.3885053273657228e-58,    1.6069990369576092e-59,
    5.7806811177004297e-61,    2.0433458024522284e-62,    7.0996120000290083e-64,    2.4253994954877230e-65
},
// x = 4.0800
{
    1.2251398745161796e-01,    9.2556113663181561e-02,    5.4458021522807987e-02,    2.5818342189152139e-02,
    1.0161846198282263e-02,    3.4025049870589205e-03,    9.8842589003517319e-04,    2.5310876880959340e-04,
    5.7878945882256435e-05,    1.1946494300191677e-05,    2.2457616411677428e-06,    3.8742702947534996e-07,
    6.1736720105720772e-08,    9.1383033373552444e-09,    1.2626539026345460e-09,    1.6355746078617038e-10,
    1.9937901563154401e-11,    2.2950216724214561e-12,    2.5021564777426738e-13,    2.5909180350894124e-14,
    2.5543650083677638e-15,    2.4031629621806652e-16,    2.1619729598923718e-17,    1.8633962299374178e-18,
    1.5413577366424714e-19,    1.2255810930528037e-20,    9.3813703264666127e-22,    6.9226928010134099e-23,
    4.9308952551475079e-24,    3.3942076910274238e-25,    2.2604721553932480e-26,    1.4580203797320826e-27,
    9.1171572775306569e-29,    5.5320879686210337e-30,    3.2601054550035967e-31,    1.8674331482599614e-32,
    1.0405613668861255e-33,    5.6444280960582849e-35,    2.9826727577647672e-36,    1.5364313021837315e-37,
    7.7199912815618279e-39,    3.7859742266043672e-40,    1.8131948028435875e-41,    8.4850540135592279e-43,
    3.8818146581636207e-44,    1.7370077859174024e-45,    7.6061160439239515e-47,    3.2607464935611063e-48,
    1.3691610387062271e-49,    5.6332397742434351e-51,    2.2719758383348711e-52,    8.9858607183857046e-54,
    3.4864903110295220e-55,    1.3275359118552855e-56,    4.9623068992370358e-58,    1.8215686767644750e-59,
    6.5685874513326407e-61,    2.3275444100901290e-62,    8.1068836766444561e-64,    2.7762969347335091e-65
},
// x = 4.0900
{
    1.2221513423714808e-01,    9.2402192355236265e-02,    5.4438465027928332e-02,    2.5851501612047578e-02,
    1.0193841242027820e-02,    3.4200661161428287e-03,    9.9561941377082912e-04,    2.5550563227466801e-04,
    5.8557192714589284e-05,    1.2113877715250529e-05,    2.2824551620806849e-06,    3.9467028158441995e-07,
    6.3037930676855911e-08,    9.3528569092617893e-09,    1.2953545032450629e-09,    1.6819172732858022e-10,
    2.0551679972205875e-11,    2.3713265748407841e-12,    2.5915427063437858e-13,    2.6899187683813845e-14,
    2.6583489549802482e-15,    2.5070182704361485e-16,    2.2608475059601279e-17,    1.9533239428674666e-18,
    1.6196520268906171e-19,    1.2909534123207158e-20,    9.9057181288444949e-22,    7.3273467247290662e-23,
    5.2317887766285955e-24,    3.6100752410490029e-25,    2.4100776093358542e-26,    1.5582963066430263e-27,
    9.7678949468398961e-29,    5.9413639912102316e-30,    3.5098188622638389e-31,    2.0153685679554790e-32,
    1.1257289529389480e-33,    6.1212925387728818e-35,    3.2425460735049201e-36,    1.6743696233062403e-37,
    8.4335981701253299e-39,    4.1460248217619973e-40,    1.9904766549640834e-41,    9.3373970200133562e-43,
    4.2821788243235193e-44,    1.9208380287360386e-45,    8.4316210232866336e-47,    3.6234684288092051e-48,
    1.5251812116106611e-49,    6.2904941824548312e-51,    2.5432554663375957e-52,    1.0083373161498751e-53,
    3.9218828794762137e-55,    1.4969681180843993e-56,    5.6093187175144976e-58,    2.0641077932501637e-59,
    7.4613813098967804e-61,    2.6503653324699460e-62,    9.2538469617253580e-64,    3.1768373064580503e-65
},
// x = 4.1000
{
    1.2191772517439366e-01,    9.2248683319656702e-02,    5.4418688599035085e-02,    2.5884428930589509e-02,
    1.0225761156565196e-02,    3.4376361478854326e-03,    1.0028349061408611e-03,    2.5791571378026238e-04,
    5.9240831335022837e-05,    1.2282998488704292e-05,    2.3196188263931953e-06,    4.0202401205621460e-07,
    6.4362173394917501e-08,    9.5717352579372732e-09,    1.3287948670378905e-09,    1.7294229596194401e-10,
    2.1182385374411897e-11,    2.4499258752140042e-12,    2.6838400063381831e-13,    2.7923918274667026e-14,
    2.7662414357661826e-15,    2.6150391700520237e-16,    2.3639379370154873e-17,    2.0473141620393642e-18,
    1.7016824433776160e-19,    1.3596119953920532e-20,    1.0457766182623210e-21,    7.7544156871013020e-23,
    5.5501236511702737e-24,    3.8390123279214917e-25,    2.5691276844224914e-26,    1.6651626707539129e-27,
    1.0463092776233871e-28,    6.3796696436653304e-30,    3.7778968292961519e-31,    2.1745711435222416e-32,
    1.2176068562514970e-33,    6.6369848305500238e-35,    3.5242652972248688e-36,    1.8242686981365818e-37,
    9.2109764251139381e-39,    4.5392092726075441e-40,    2.1845458615743714e-41,    1.0272730319211230e-42,
    4.7225987173549629e-44,    2.1235542513868133e-45,    9.3441593988647194e-47,    4.0254121321324656e-48,
    1.6984946362667475e-49,    7.0223829160132202e-51,    2.8460784977013370e-52,    1.1311494846510590e-53,
    4.4102801633121584e-55,    1.6874916077011130e-56,    6.3386505312024229e-58,    2.3381737960790250e-59,
    8.4726930372593237e-61,    3.0169345904778224e-62,    1.0559429821529322e-63,    3.6338853097004359e-65
},
// x = 4.1100
{
    1.2162174999210393e-01,    9.2095585776240621e-02,    5.4398694680979409e-02,    2.5917125093783910e-02,
    1.0257605713464230e-02,    3.4552147723294025e-03,    1.0100722595412599e-03,    2.6033901222322726e-04,
    5.9929879164518159e-05,    1.2453867260500139e-05,    2.3572567923763988e-06,    4.0948948923386870e-07,
    6.5709772332850323e-08,    9.7950103235870810e-09,    1.3629891851979655e-09,    1.7781169323647728e-10,
    2.1830428426482173e-11,    2.5308810530436971e-12,    2.7791337623171955e-13,    2.8984478694883844e-14,
    2.8781769372014551e-15,    2.7273795881093274e-16,    2.4714107792669342e-17,    2.1455377233123941e-18,
    1.7876156500935648e-19,    1.4317118577974084e-20,    1.1038892242766427e-21,    8.2050722338544484e-23,
    5.8868572158286863e-24,    4.0817700953352585e-25,    2.7381896490965481e-26,    1.7790324170066459e-27,
    1.1205652225045454e-28,    6.8489750894822945e-30,    4.0656337081613482e-31,    2.3458645123832297e-32,
    1.3167032268178645e-33,    7.1945474755846656e-35,    3.8295998863572846e-36,    1.9871289465179618e-37,
    1.0057629059986396e-38,    4.9684748418105987e-40,    2.3969403775211622e-41,    1.1298878567320095e-42,
    5.2069581614352313e-44,    2.3470383190270875e-45,    1.0352637555308619e-46,    4.4706970229479870e-48,
    1.8909643385044095e-49,    7.8371486188469513e-51,    3.1840142566035808e-52,    1.2685360526716285e-53,
    4.9579688863625617e-55,    1.9016656446965316e-56,    7.1605169579639128e-58,    2.6477653338411676e-59,
    9.6178811666015801e-61,    3.4330422937732103e-62,    1.2045064803062749e-63,    4.1552330473188800e-65
},
// x = 4.1200
{
    1.2132719851341570e-01,    9.1942898934784370e-02,    5.4378485696825130e-02,    2.5949591050287889e-02,
    1.0289374689054442e-02,    3.4728016809942031e-03,    1.0173313660116648e-03,    2.6277552610302831e-04,
    6.0624353500639250e-05,    1.2626494668351759e-05,    2.3953732339685472e-06,    4.1706799035673357e-07,
    6.7081054792607084e-08,    1.0022754964700389e-08,    1.3979518685996916e-09,    1.8280249154721624e-10,
    2.2496228317240073e-11,    2.6142550256330377e-12,    2.8775115773610855e-13,    3.0082007129152656e-14,
    2.9942941349061631e-15,    2.8441986430978252e-16,    2.5834386041924271e-17,    2.2481721042992684e-18,
    1.8776252200542039e-19,    1.5074148409560599e-20,    1.1650538288207032e-21,    8.6805465993303078e-23,
    6.2429963858320607e-24,    4.3391405339349221e-25,    2.9178630925256531e-26,    1.9003430923638272e-27,
    1.1998655173197997e-28,    7.3513781456967793e-30,    4.3744111021190506e-31,    2.5301299914877207e-32,
    1.4235631351379749e-33,    7.7972520442185820e-35,    4.1604571855640435e-36,    2.1640313916851954e-37,
    1.0979517041458156e-38,    5.4370218839569059e-40,    2.6293343195462862e-41,    1.2424380038595242e-42,
    5.7395055710781237e-44,    2.5933537965313942e-45,    1.1466845729654858e-46,    4.9638624102732475e-48,
    2.1046482665569814e-49,    8.7439186221049318e-51,    3.5610248375919426e-52,    1.4221811503323794e-53,
    5.5719617609952836e-55,    2.1423520204745103e-56,    8.0863680153631765e-58,    2.9973756332817695e-59,
    1.0914246084628517e-60,    3.9052265754031373e-62,    1.3735012508114301e-63,    4.7497223706763361e-65
},
// x = 4.1300
{
    1.2103406065481574e-01,    9.1790621995349972e-02,    5.4358064048023999e-02,    2.5981827748347532e-02,
    1.0321067864384107e-02,    3.4903965668809430e-03,    1.0246121172435851e-03,    2.6522525352341077e-04,
    6.1324271516911870e-05,    1.2800891347260303e-05,    2.4339723406538141e-06,    4.2476080156293359e-07,
    6.8476351320286279e-08,    1.0255042965559034e-08,    1.4336975502877139e-09,    1.8791730978573977e-10,
    2.3180212912912242e-11,    2.7001121765135870e-12,    2.9790633228862032e-13,    3.1217674170016395e-14,
    3.1147360100151478e-15,    2.9656608027763291e-16,    2.7002002281921028e-17,    2.3554016610601986e-18,
    1.9718918995266531e-19,    1.5868898909931726e-20,    1.2294213312324760e-21,    9.1821293631171152e-23,
    6.6196000667584614e-24,    4.6119585750719842e-25,    3.1087816655628810e-26,    2.0295582351987263e-27,
    1.2845374581771911e-28,    7.8891121595639598e-30,    4.7057034779528607e-31,    2.7283104388479776e-32,
    1.5387711410333904e-33,    8.4486157138948074e-35,    4.5188927474093857e-36,    2.3561439063749419e-37,
    1.1983095990005464e-38,    5.9483248000206672e-40,    2.8835496017198942e-41,    1.3658549507564373e-42,
    6.3248870513709880e-44,    2.8647629307272112e-45,    1.2697542988999641e-46,    5.5099089912148557e-48,
    2.3418190805658039e-49,    9.7527971498953883e-51,    3.9815071041965681e-52,    1.5939582731864770e-53,
    6.2600790583822005e-55,    2.4127498689679263e-56,    9.1290346638411229e-58,    3.3920521188179740e-59,
    1.2381269439304884e-60,    4.4408678701463814e-62,    1.5656725822231709e-63,    5.4273829766235033e-65
},
// x = 4.1400
{
    1.2074232642511659e-01,    9.1638754148526749e-02,    5.4337432114589940e-02,    2.6013836135736972e-02,
    1.0352685025179595e-02,    3.5079991244769903e-03,    1.0319144045885585e-03,    2.6768819219407703e-04,
    6.2029650262192303e-05,    1.2977067929036238e-05,    2.4730583173399838e-06,    4.3256921789138701e-07,
    6.9895995721167598e-08,    1.0491949043756598e-08,    1.4702410879724151e-09,    1.9315881399813198e-10,
    2.3882818904277872e-11,    2.7885183843225410e-12,    3.0838811894240475e-13,    3.2392683629551339e-14,
    3.2396499683702221e-15,    3.0919360462885879e-16,    2.8218809181590212e-17,    2.4674178724435337e-18,
    2.0706038814913079e-19,    1.6703133480391578e-20,    1.2971496225677608e-21,    9.7111742205275725e-23,
    7.0177816764164909e-24,    4.9011042852694352e-25,    3.3116149100145964e-26,    2.1671688388023006e-27,
    1.3749285750224530e-28,    8.4645543467587336e-30,    5.0610841218854114e-31,    2.9414143616379042e-32,
    1.6629540332484497e-33,    9.1524189534384313e-35,    4.9071213936616201e-36,    2.5647279236869024e-37,
    1.3075355708348979e-38,    6.5061546621028344e-40,    3.1615685292572023e-41,    1.5011546478142949e-42,
    6.9681824010767468e-44,    3.1637451673604105e-45,    1.4056550115459832e-46,    6.1143443223329680e-48,
    2.6049858898664066e-49,    1.0874966848877495e-50,    4.4503390478742716e-52,    1.7859509860552528e-53,
    7.0310390894640922e-55,    2.7164343795768531e-56,    1.0302891040821391e-57,    3.8374630320927191e-59,
    1.4040882311929121e-60,    5.0482947787287471e-62,    1.7841259904823534e-63,    6.1995882351410895e-65
},
// x = 4.1500
{
    1.2045198592444482e-01,    9.1487294575688938e-02,    5.4316592255272099e-02,    2.6045617159698487e-02,
    1.0384225961804773e-02,    3.5256090497604378e-03,    1.0392381190662628e-03,    2.7016433943238567e-04,
    6.2740506660049269e-05,    1.3155035041822554e-05,    2.5126353842351363e-06,    4.4049454328330601e-07,
    7.1340325074646066e-08,    1.0733548857727387e-08,    1.5075975665401531e-09,    1.9852971804921010e-10,
    2.4604491955692816e-11,    2.8795410521346182e-12,    3.1920597383458613e-13,    3.3608273368430143e-14,
    3.3691879625919987e-15,    3.2232000306342332e-16,    2.9486726031226410e-17,    2.5844195922937750e-18,
    2.1739570886315910e-19,    1.7578692463704505e-20,    1.3684038875133801e-21,    1.0269100871434746e-22,
    7.4387117810646488e-24,    5.2075051659205296e-25,    3.5270701804134505e-26,    2.3136948927267353e-27,
    1.4714078201779507e-28,    9.0802346166917316e-30,    5.4422314590447559e-31,    3.1705202858286402e-32,
    1.7967837506603221e-33,    9.9127244261437581e-35,    5.3275290680771843e-36,    2.7911456458379652e-37,
    1.4263862747082059e-38,    7.1146036364075488e-40,    3.4655474266965184e-41,    1.6494449197809309e-42,
    7.6749442642060533e-44,    3.4930173367417278e-45,    1.5556851109737296e-46,    6.7832326319236834e-48,
    2.8969181237296411e-49,    1.2122800555606376e-50,    4.9729309464336248e-52,    2.0004758261892361e-53,
    7.8945585422684350e-55,    3.0573998321621421e-56,    1.1624035247971532e-57,    4.3399718515474393e-59,
    1.5917765519349403e-60,    5.7369029053117535e-62,    2.0323733545748404e-63,    7.0792309682297863e-65
},
// x = 4.1600
{
    1.2016302934324211e-01,    9.1336242449247748e-02,    5.4295546807726920e-02,    2.6077171766883699e-02,
    1.0415690469220673e-02,    3.5432260402043698e-03,    1.0465831513725775e-03,    2.7265369216506607e-04,
    6.3456857508156466e-05,    1.3334803309619051e-05,    2.5527077767232975e-06,    4.4853809058316633e-07,
    7.2809679749061889e-08,    1.0979919014284898e-08,    1.5457823005781433e-09,    2.0403278429303584e-10,
    2.5345686856001627e-11,    2.9732491372535862e-12,    3.3036959545463259e-13,    3.4865716142671233e-14,
    3.5035066170899626e-15,    3.3596342615957974e-16,    3.0807740921228433e-17,    2.7066133097530088e-18,
    2.2821554661511888e-19,    1.8497496257617257e-20,    1.4433569183688845e-21,    1.0857398032136910e-22,
    7.8836208507844859e-24,    5.5321385629315196e-25,    3.7558946626805520e-26,    2.4696870058593682e-27,
    1.5743668230063975e-28,    9.7388449118735056e-30,    5.8509357575392389e-31,    3.4167814031980004e-32,
    1.9409804965733411e-33,    1.0733897191893485e-34,    5.7826855348533155e-36,    3.0368677862110185e-37,
    1.5556806231426917e-38,    7.7781113418357012e-40,    3.7978313822035007e-41,    1.8119334929508652e-42,
    8.4512406956829933e-44,    3.8555556533013090e-45,    1.7212704086383562e-46,    7.5232493742147424e-48,
    3.2206717383542155e-49,    1.3509984302271806e-50,    5.5552818039540751e-52,    2.2401076325611088e-53,
    8.8614637186416989e-55,    3.4401074231781649e-56,    1.3110490755553543e-57,    4.9067204009183558e-59,
    1.8039685795429884e-60,    6.5172882174461958e-62,    2.3143848162753589e-63,    8.0809216717537303e-65
},
// x = 4.1700
{
    1.1987544696127854e-01,    9.1185596932899265e-02,    5.4274298088689131e-02,    2.6108500903295986e-02,
    1.0447078346945289e-02,    3.5608497947809620e-03,    1.0539493918875938e-03,    2.7515624692995113e-04,
    6.4178719477697480e-05,    1.3516383351807970e-05,    2.5932797452390923e-06,    4.5670118153915310e-07,
    7.4304403416425757e-08,    1.1231137076169053e-08,    1.5848108369139168e-09,    2.0967082424972556e-10,
    2.6106867671352469e-11,    3.0697131814685157e-12,    3.4188893001001502e-13,    3.6166320468383302e-14,
    3.6427673560705256e-15,    3.5014262692250306e-16,    3.2183912984761039e-17,    2.8342134178877086e-18,
    2.3954112847269326e-19,    1.9461548544311968e-20,    1.5221894415396140e-21,    1.1477626575091850e-22,
    8.3538021390104235e-24,    5.8760341912140743e-25,    3.9988774942548415e-26,    2.6357281153034519e-27,
    1.6842212141709993e-28,    1.0443249089664130e-29,    6.2891062393521527e-31,    3.6814305124604151e-32,
    2.0963160582624297e-33,    1.1620626293567657e-34,    6.2753579804774756e-36,    3.3034818825080034e-37,
    1.6963047188928653e-38,    8.5014932916458156e-40,    4.1609701959776568e-41,    1.9899366991115800e-42,
    9.3037014283176482e-44,    4.2546196864695786e-45,    1.9039762400823085e-46,    8.3417409622738179e-48,
    3.5796179815572748e-49,    1.5051652656178918e-50,    6.2040416011285518e-52,    2.5077075507838977e-53,
    9.9438138206044527e-55,    3.8695384014818477e-56,    1.4782430710977202e-57,    5.5457216340922391e-59,
    2.0437872026638468e-60,    7.4013966578339597e-62,    2.6346471460880820e-63,    9.2212119753134210e-65
},
// x = 4.1800
{
    1.1958922914667834e-01,    9.1035357181867546e-02,    5.4252848394141837e-02,    2.6139605514233760e-02,
    1.0478389399013530e-02,    3.5784800139653837e-03,    1.0613367306835700e-03,    2.7767199987772625e-04,
    6.4906109112781762e-05,    1.3699785782680881e-05,    2.6343555551413892e-06,    4.6498514680307803e-07,
    7.5824843067037210e-08,    1.1487281569601777e-08,    1.6246989571692807e-09,    2.1544669928857739e-10,
    2.6888507899927675e-11,    3.1690053417798646e-12,    3.5377417689053275e-13,    3.7511431504814974e-14,
    3.7871365346035438e-15,    3.6487697879934367e-16,    3.3617374705986354e-17,    2.9674424908784450e-18,
    2.5139454539138510e-19,    2.0472939639720303e-20,    1.6050904569986169e-21,    1.2131422801534755e-22,
    8.8506146914118274e-24,    6.2402767791355666e-25,    4.2568519904688155e-26,    2.8124352853321594e-27,
    1.8014120231347452e-28,    1.1196493376217085e-29,    6.7587786214794977e-31,    3.9657852722221965e-32,
    2.2636173446586498e-33,    1.2577947818329102e-34,    6.8085255804625944e-36,    3.5927012213797738e-37,
    1.8492171634787752e-38,    9.2899715764081308e-40,    4.5577356273996219e-41,    2.1848889112249234e-42,
    1.0239568151574206e-43,    4.6937784734781822e-45,    2.1055206920741490e-46,    9.2467901553158185e-48,
    3.9774749569161728e-49,    1.6764537591692808e-50,    6.9265799362946663e-52,    2.8064539880324770e-53,
    1.1155037554354903e-54,    4.3512530868671966e-56,    1.6662427685283078e-57,    6.2659631929019427e-59,
    2.3147436189407262e-60,    8.4026919377773772e-62,    2.9982293608244206e-63,    1.0518846474576239e-64
},
// x = 4.1900
{
    1.1930436635495757e-01,    9.0885522343143205e-02,    5.4231199999485100e-02,    2.6170486544235014e-02,
    1.0509623433937344e-02,    3.5961163997395343e-03,    1.0687450575328364e-03,    2.8020094677369458e-04,
    6.5639042829872325e-05,    1.3885021210966840e-05,    2.6759394865859655e-06,    4.7339132592976781e-07,
    7.7371349023994995e-08,    1.1748431991850430e-08,    1.6654626803287188e-09,    2.2136332131753163e-10,
    2.7691090628604568e-11,    3.2711994216005180e-12,    3.6603579423271443e-13,    3.8902431956025726e-14,
    3.9367855728090128e-15,    3.8018649417142750e-16,    3.5110334295553664e-17,    3.1065315700161374e-18,
    2.6379878463279446e-19,    2.1533849966751408e-20,    1.6922575911902312e-21,    1.2820501852174486e-22,
    9.3754864895243113e-24,    6.6260088382465005e-25,    4.5306979821599867e-26,    3.0004616008811641e-27,
    1.9264071527208677e-28,    1.2001817423971804e-29,    7.2621231120104747e-31,    4.2712537844780544e-32,
    2.4437701558400354e-33,    1.3611269530020817e-34,    7.3853950964530284e-36,    3.9063744176353039e-37,
    2.0154547689565255e-38,    1.0149207958010737e-39,    4.9911400427055215e-41,    2.3983527701205861e-42,
    1.1266749137716188e-43,    5.1769389589138902e-45,    2.3277890433041294e-46,    1.0247287618328087e-47,
    4.4183422512217308e-49,    1.8667132206184792e-50,    7.7310616935813360e-52,    3.1398768196371377e-53,
    1.2510084449471396e-54,    4.8914564044499167e-56,    1.8775731663726974e-57,    7.0775229553133533e-59,
    2.6207844161774392e-60,    9.5363436653644050e-62,    3.4108564712180684e-63,    1.1995046449989730e-64
},
// x = 4.2000
{
    1.1902084912807395e-01,    9.0736091555717990e-02,    5.4209355159703938e-02,    2.6201144937022771e-02,
    1.0540780264666013e-02,    3.6137586555955977e-03,    1.0761742619156442e-03,    2.8274308299955733e-04,
    6.6377536917224770e-05,    1.4072100239361886e-05,    2.7180358343971905e-06,    4.8192106737592590e-07,
    7.8944274957597452e-08,    1.2014668818798380e-08,    1.7071182653221211e-09,    2.2742365347897931e-10,
    2.8515108691559920e-11,    3.3763709024369074e-12,    3.7868450458569538e-13,    4.0340742991496679e-14,
    4.0918910932263686e-15,    3.9609184333449878e-16,    3.6665078135069518e-17,    3.2517204587543981e-18,
    2.7677776329409628e-19,    2.2646553656607397e-20,    1.7838974638636689e-21,    1.3546661261348226e-22,
    9.9299177347336941e-24,    7.0344335638225507e-25,    4.8213442697252675e-26,    3.2004981602518862e-27,
    2.0597029347437107e-28,    1.2862666005667735e-29,    7.8014528871899262e-31,    4.5993405284292859e-32,
    2.6377231988038965e-33,    1.4726397174890633e-34,    8.0094175734274162e-36,    4.2464956940376643e-37,
    2.1961387023234054e-38,    1.1085339556811115e-39,    5.4644565726388848e-41,    2.6320302660983841e-42,
    1.2393878577950559e-43,    5.7083769612405604e-45,    2.5728495262683523e-46,    1.1353010217780836e-47,
    4.9067389131627027e-49,    2.0779870714598051e-50,    8.6265304360194438e-52,    3.5118951789380744e-53,
    1.4025592433215452e-54,    5.4970706342128274e-56,    2.1150579387530032e-57,    7.9916979252735196e-59,
    2.9663442164307919e-60,    1.0819438209716879e-61,    3.8789923416486022e-63,    1.3673829410059665e-64
},
// x = 4.2100
{
    1.1873866809348810e-01,    9.0587063950814020e-02,    5.4187316109535144e-02,    2.6231581635451612e-02,
    1.0571859708546696e-02,    3.6314064865394400e-03,    1.0836242330279687e-03,    2.8529840355521238e-04,
    6.7121607534338304e-05,    1.4261033464060105e-05,    2.7606489079387839e-06,    4.9057572849847058e-07,
    8.0543977899633659e-08,    1.2286073512522582e-08,    1.7496822136218414e-09,    2.3363071085194020e-10,
    2.9361064830832553e-11,    3.4845969760554923e-12,    3.9173130068001459e-13,    4.1827825186006173e-14,
    4.2526350614301935e-15,    4.1261437397815062e-16,    3.8283973292311854e-17,    3.4032580270739153e-18,
    2.9035636298310005e-19,    2.3813422282488731e-20,    1.8802260693411842e-21,    1.4311784660207613e-22,
    1.0515484278432182e-23,    7.4668178719863277e-25,    5.1297711990532549e-26,    3.4132761719140742e-27,
    2.2018257709154361e-28,    1.3782701379552506e-29,    8.3792330769126654e-31,    4.9516526655270250e-32,
    2.8464923648564989e-33,    1.5929562569195455e-34,    8.6843062112297028e-36,    4.6152159107872515e-37,
    2.3924810940082374e-38,    1.2105017327223081e-39,    5.9812408977498858e-41,    2.8877747443024632e-42,
    1.3630381020331946e-43,    6.2927708831211837e-45,    2.8429705271905664e-46,    1.2574706666960498e-47,
    5.4476450973419044e-49,    2.3125326291509382e-50,    9.6230002886291078e-52,    3.9268591947862822e-53,
    1.5720073356254745e-54,    6.1758161479257919e-56,    2.3818538486092503e-57,    9.0211479636057979e-59,
    3.3564045231610391e-60,    1.2271214978773850e-61,    4.4099327596315148e-63,    1.5582368871541914e-64
},
// x = 4.2200
{
    1.1845781396323680e-01,    9.0438438652109945e-02,    5.4165085063632573e-02,    2.6261797581455242e-02,
    1.0602861587285019e-02,    3.6490595990938276e-03,    1.0910948597892586e-03,    2.8786690306056585e-04,
    6.7871270711417721e-05,    1.4451831474285898e-05,    2.8037830309835990e-06,    4.9935667555234267e-07,
    8.2170818257560964e-08,    1.2562728528877172e-08,    1.7931712718539258e-09,    2.3998756116061903e-10,
    3.0229471858857055e-11,    3.5959565771396472e-12,    4.0518745130074399e-13,    4.3365179479095466e-14,
    4.4192049299566146e-15,    4.2977613117574394e-16,    3.9969470108985869e-17,    3.5614025254207512e-18,
    3.0456046567417292e-19,    2.5036928730118409e-20,    1.9814691727417943e-21,    1.5117845634705211e-22,
    1.1133841204386828e-23,    7.9244955794083414e-25,    5.4570133650047876e-26,    3.6395691605211246e-27,
    2.3533338634385421e-28,    1.4765816362234419e-29,    8.9980902875786971e-31,    5.3299067378254810e-32,
    3.0711652848629139e-33,    1.7227453583941430e-34,    9.4140554894384354e-36,    5.0148543970822430e-37,
    2.6057921440871190e-38,    1.3215447531136194e-39,    6.5453547878205083e-41,    3.1676039080665607e-42,
    1.4986541332574785e-43,    6.9352384003144496e-45,    3.1406393508077025e-46,    1.3924191188469605e-47,
    6.0465477162034174e-49,    2.5728428475965453e-50,    1.0731557149913282e-51,    4.3895960770143621e-53,
    1.7614118337942847e-54,    6.9363009859299727e-56,    2.6814890205694022e-57,    1.0180056023228408e-58,
    3.7965594832568540e-60,    1.3913331095122200e-61,    5.0119099415633241e-63,    1.7751399318676549e-64
},
// x = 4.2300
{
    1.1817827753301730e-01,    9.0290214775962038e-02,    5.4142664216732188e-02,    2.6291793715994979e-02,
    1.0633785726905952e-02,    3.6667177013014695e-03,    1.0985860308501221e-03,    2.9044857575735955e-04,
    6.8626542348847042e-05,    1.4644504851827841e-05,    2.8474425415824962e-06,    5.0826528368778739e-07,
    8.3825159828569607e-08,    1.2844717325082897e-08,    1.8376024344234718e-09,    2.4649732548935711e-10,
    3.1120852822982709e-11,    3.7105304164422836e-12,    4.1906450726641924e-13,    4.4954348154457425e-14,
    4.5917937856061746e-15,    4.4759987789637681e-16,    4.1724104862861681e-17,    3.7264219084872321e-18,
    3.1941699078130968e-19,    2.6319651209650089e-20,    2.0878627216987458e-21,    1.5966911744356960e-22,
    1.1786726569591836e-23,    8.4088707318318491e-25,    5.8041624483586886e-26,    3.8801952874871606e-27,
    2.5148190399087357e-28,    1.5816148147497645e-29,    9.6608226927963964e-31,    5.7359357829765412e-32,
    3.3129061795588989e-33,    1.8627246149120918e-34,    1.0202961629660010e-35,    5.4479116406481955e-37,
    2.8374877622009287e-38,    1.4424436433653460e-39,    7.1609915313384915e-41,    3.4737139001639242e-42,
    1.6473580646732651e-43,    7.6413763832957979e-45,    3.4685826876451044e-46,    1.5414445920038095e-47,
    6.7094904734270077e-49,    2.8616702013999897e-50,    1.1964470150431741e-51,    4.9054609895754432e-53,
    1.9730624987996318e-54,    7.7881202146546957e-56,    3.0179054945419841e-57,    1.1484306733279186e-58,
    4.2930893552473570e-60,    1.5770157795698231e-61,    5.6942098447667356e-63,    2.0215671875353006e-64
},
// x = 4.2400
{
    1.1790004968128329e-01,    9.0142391431621616e-02,    5.4120055743815176e-02,    2.6321570979009359e-02,
    1.0664631957714821e-02,    3.6843805027279054e-03,    1.1060976345999714e-03,    2.9304341551101204e-04,
    6.9387438216673985e-05,    1.4839064170573941e-05,    2.8916317919322735e-06,    5.1730293694710731e-07,
    8.5507369813531684e-08,    1.3132124367321531e-08,    1.8829929461540122e-09,    2.5316317900399269e-10,
    3.2035741171991074e-11,    3.8284010144393632e-12,    4.3337430751521815e-13,    4.6596915839582590e-14,
    4.7706005001896435e-15,    4.6610911605064928e-16,    4.3550502506171692e-17,    3.8985941691104018e-18,
    3.3495393348559577e-19,    2.7664277413659418e-20,    2.1996532741263810e-21,    1.6861148707965414e-22,
    1.2475965310113264e-23,    8.9214210879191031e-25,    6.1723701923947664e-26,    4.1360197917195310e-27,
    2.6869086773764130e-28,    1.6938092911352229e-29,    1.0370410724054479e-30,    6.1716968905084470e-32,
    3.5729610231381847e-33,    2.0136638407320962e-34,    1.1055644484714176e-35,    5.9170828948520635e-37,
    3.0890977796413465e-38,    1.5740438461757605e-39,    7.8327034010605583e-41,    3.8084945480544085e-42,
    1.8103738779107450e-43,    8.4173043266651070e-45,    3.8297889330926738e-46,    1.7059732853351765e-47,
    7.4431286860263460e-49,    3.1820529187722158e-50,    1.3335314364802873e-51,    5.4803931941733396e-53,
    2.2095048768588777e-54,    8.7419661030022778e-56,    3.3955065257550633e-57,    1.2951685377237618e-58,
    4.8530425631418658e-60,    1.7869112260471772e-61,    6.4673038157955932e-63,    2.3014466884877647e-64
},
// x = 4.2500
{
    1.1762312136835169e-01,    8.9994967721447924e-02,    5.4097261800270809e-02,    2.6351130309364611e-02,
    1.0695400114258515e-02,    3.7020477144642334e-03,    1.1136295591746136e-03,    2.9565141581247665e-04,
    7.0153973954107052e-05,    1.5035519996048590e-05,    2.9363551482427456e-06,    5.2647102826089684e-07,
    8.7217818830835476e-08,    1.3425035138335249e-08,    1.9293603049409071e-09,    2.5998835167965133e-10,
    3.2974680924626868e-11,    3.9496527354897066e-12,    4.4812898529988466e-13,    4.8294510526004478e-14,
    4.9558298847846235e-15,    4.8532810808221720e-16,    4.5451379482188943e-17,    4.0782076825700588e-18,
    3.5120040435534663e-19,    2.9073608826061355e-20,    2.3170984426102963e-21,    1.7802824762707917e-22,
    1.3203473318682191e-23,    9.4637017651802332e-25,    6.5628515255525905e-26,    4.4079575563564923e-27,
    2.8702677306494029e-28,    1.8136321245642311e-29,    1.1130028395202857e-30,    6.6392792254137721e-32,
    3.8526630393974958e-33,    2.1763887154536423e-34,    1.1977070949893930e-35,    6.4252727669776235e-37,
    3.3622747747282645e-38,    1.7172608083755187e-39,    8.5654313125484015e-41,    4.1745458658372902e-42,
    1.9890363657951205e-43,    9.2697115839761130e-45,    4.2275325202364703e-46,    1.8875716164541290e-47,
    8.2547893390056891e-49,    3.5373437869482147e-50,    1.4859105879260960e-51,    6.1209779944328918e-53,
    2.4735680988307646e-54,    9.8097502627657336e-56,    3.8192091462150532e-57,    1.4602099531832148e-58,
    5.4843273130133609e-60,    2.0241028996612019e-61,    7.3429962830512772e-63,    2.6192170329677177e-64
},
// x = 4.2600
{
    1.1734748363552064e-01,    8.9847942741117709e-02,    5.4074284522057467e-02,    2.6380472644806113e-02,
    1.0726090035286852e-02,    3.7197190491296637e-03,    1.1211816924637768e-03,    2.9827256978011023e-04,
    7.0926165069022285e-05,    1.5233882884950878e-05,    2.9816169906028753e-06,    5.3577095944374004e-07,
    8.8956880930100233e-08,    1.3723536145029781e-08,    1.9767222644185481e-09,    2.6697612903496635e-10,
    3.3938226840153943e-11,    4.0743718225061793e-12,    4.6334097449285587e-13,    5.0048804610484907e-14,
    5.1476928475713648e-15,    5.0528189911731430e-16,    4.7429546621943377e-17,    4.2655615615746475e-18,
    3.6818667029819703e-19,    3.0550565186935372e-20,    2.4404673560129754e-21,    1.8794315203216043e-22,
    1.3971261701045013e-23,    1.0037349055017069e-24,    6.9768878368799782e-26,    4.6969758076253514e-27,
    3.0656008701637640e-28,    1.9415794464679150e-29,    1.1943055296384229e-30,    7.1409125465254534e-32,
    4.1534385508486153e-33,    2.3517846714434133e-34,    1.2972579997538603e-35,    6.9756108554497574e-37,
    3.6588035554314326e-38,    1.8730855685761135e-39,    9.3645368441718181e-41,    4.5746959127207335e-42,
    2.1848008333708569e-43,    1.0205908729974306e-44,    4.6654004420737163e-46,    2.0879595869712113e-47,
    9.1525368562242182e-49,    3.9312417746181403e-50,    1.6552450421492709e-51,    6.8345150623571938e-53,
    2.7683956168984363e-54,    1.1004739015976718e-55,    4.2945025568640504e-57,    1.6457825878565054e-58,
    6.1958138576636840e-60,    2.2920575372646215e-61,    8.3345904000347713e-63,    2.9798921846495543e-64
},
// x = 4.2700
{
    1.1707312760419823e-01,    8.9701315579830243e-02,    5.4051126025863171e-02,    2.6409598921910870e-02,
    1.0756701563714201e-02,    3.7373942208739270e-03,    1.1287539221186017e-03,    3.0090687016155846e-04,
    7.1704026937482847e-05,    1.5434163384694688e-05,    3.0274217128460750e-06,    5.4520414118939516e-07,
    9.0724933605774030e-08,    1.4027714926081342e-08,    2.0250968366414026e-09,    2.7412985287275579e-10,
    3.4926944590951901e-11,    4.2026464321437556e-12,    4.7902301600311781e-13,    5.1861515957489874e-14,
    5.3464065553177625e-15,    5.2599633968468089e-16,    4.9487912123081394e-17,    4.4609660222309863e-18,
    3.8594419688552602e-19,    3.2098189118393419e-20,    2.5700411389065921e-21,    1.9838107107506935e-22,
    1.4781441218345329e-23,    1.0644084414198840e-24,    7.4158304112736835e-26,    5.0040969522149016e-27,
    3.2736547350056335e-28,    2.0781781831618805e-29,    1.2813089294955527e-30,    7.6789762486926410e-32,
    4.4768132023978496e-33,    2.5408010401225309e-34,    1.4047909442588147e-35,    7.5714685082744753e-37,
    3.9806113462070637e-38,    2.0425907741168982e-39,    1.0235836799513175e-40,    5.0120201154461608e-42,
    2.3992536191958975e-43,    1.1233883398506504e-44,    5.1473211535281413e-46,    2.3090253819922964e-47,
    1.0145245114365172e-48,    4.3678267383826883e-50,    1.8433706873693454e-51,    7.6290937854062217e-53,
    3.0974791800233606e-54,    1.2341703381567221e-55,    4.8275129795842248e-57,    1.8543784970486806e-58,
    6.9974486145910494e-60,    2.5946716416801676e-61,    9.4570737662466387e-63,    3.3891343011125381e-64
},
// x = 4.2800
{
    1.1680004447504200e-01,    8.9555085320508945e-02,    5.4027788409264706e-02,    2.6438510076040811e-02,
    1.0787234546581153e-02,    3.7550729453795081e-03,    1.1363461355590545e-03,    3.0355430933565094e-04,
    7.2487574803268214e-05,    1.5636372032950180e-05,    3.0737737224146030e-06,    5.5477199306543903e-07,
    9.2522357810608874e-08,    1.4337660059546406e-08,    2.0745022947787025e-09,    2.8145292202715449e-10,
    3.5941410937164828e-11,    4.3345666705095611e-12,    4.9518816430628042e-13,    5.3734408983306160e-14,
    5.5521945985841721e-15,    5.4749810901853453e-16,    5.1629484612910330e-17,    4.6647427613001765e-18,
    4.0450569209063204e-19,    3.3719650916772531e-20,    2.7061134094640123e-21,    2.0936804256863973e-22,
    1.5636226923082394e-23,    1.1285718640378137e-24,    7.8811040318126331e-26,    5.3304015598446531e-27,
    3.4952203069337668e-28,    2.2239878753656098e-29,    1.3743959982925615e-30,    8.2560089593792717e-32,
    4.8244185824485421e-33,    2.7444554735720511e-34,    1.5209224553598927e-35,    8.2164767797205275e-37,
    4.3297787292343671e-38,    2.2269371590066996e-39,    1.1185640506423328e-40,    5.4898621702906818e-42,
    2.6341235038068481e-43,    1.2362360972701024e-44,    5.6775960596699429e-46,    2.5528413137826442e-47,
    1.1242676273885990e-48,    4.8515975047456713e-50,    2.0523167114455482e-51,    8.5136763344484364e-53,
    3.4646963796870886e-54,    1.3837085216091637e-55,    5.4250756641796418e-57,    2.0887847036013382e-58,
    7.9003814752367382e-60,    2.9363234570584787e-61,    1.0727326598305370e-62,    3.8533355591152169e-64
},
// x = 4.2900
{
    1.1652822552710881e-01,    8.9409251039998694e-02,    5.4004273750885949e-02,    2.6467207041297104e-02,
    1.0817688835016562e-02,    3.7727549398637477e-03,    1.1439582199813134e-03,    3.0621487931431447e-04,
    7.3276823777416244e-05,    1.5840519357187279e-05,    3.1206774402231724e-06,    5.6447594350740538e-07,
    9.4349537969017400e-08,    1.4653461170474253e-08,    2.1249571758228143e-09,    2.8894879311723263e-10,
    3.6982213903418162e-11,    4.4702246294005812e-12,    5.1184979408942776e-13,    5.5669295762157836e-14,
    5.7652871607202323e-15,    5.6981473895750130e-16,    5.3857376297720383e-17,    4.8772253450501526e-18,
    4.2390515148324442e-19,    3.5418253516594262e-20,    2.8489907964596878e-21,    2.2093132257013064e-22,
    1.6537942996475240e-23,    1.1964156239561398e-24,    8.3742107567956542e-26,    5.6770314980156550e-27,
    3.7311354115319323e-28,    2.3796025997577982e-29,    1.4739742912526882e-30,    8.8747187220064455e-32,
    5.1979992656082662e-33,    2.9638386589110379e-34,    1.6463148630929232e-35,    8.9145456673288940e-37,
    4.7085513936629857e-38,    2.4273805168013848e-39,    1.2220790061239396e-40,    6.0118566490565217e-42,
    2.8912940775122629e-43,    1.3600870534578337e-44,    6.2609338138282420e-46,    2.8216812293337549e-47,
    1.2455567051856883e-48,    5.3875136458371410e-50,    2.2843253771173316e-51,    9.4981892205767506e-53,
    3.8743521303740300e-54,    1.5509181199751984e-55,    6.0948148181707082e-57,    2.3521172232010130e-58,
    8.9171077904687316e-60,    3.3219310728846641e-61,    1.2164354997021019e-62,    4.3797100607515506e-64
},
// x = 4.3000
{
    1.1625766211701541e-01,    8.9263811809260102e-02,    5.3980584110554876e-02,    2.6495690750475361e-02,
    1.0848064284199632e-02,    3.7904399230807803e-03,    1.1515900623650793e-03,    3.0888857174449387e-04,
    7.4071788837775038e-05,    1.6046615874220500e-05,    3.1681373005216592e-06,    5.7431742981238376e-07,
    9.6206861990304449e-08,    1.4975208938521158e-08,    2.1764802833110521e-09,    2.9662098130708995e-10,
    3.8049952957614185e-11,    4.6097144230739534e-12,    5.2902160701223968e-13,    5.7668037154679722e-14,
    5.9859211907262615e-15,    5.9297463845257449e-16,    5.6174806200510632e-17,    5.0987596100217563e-18,
    4.4417790492400895e-19,    3.7197437631886515e-20,    2.9989934760524476e-21,    2.3309943868191861e-22,
    1.7489027795349694e-23,    1.2681399993760811e-24,    8.8967338794169060e-26,    6.0451932262408714e-27,
    3.9822873529111741e-28,    2.5456529979797037e-29,    1.5804774663727695e-30,    9.5379938001434116e-32,
    5.5994203025789827e-33,    3.2001193439545818e-34,    1.7816795681458249e-35,    9.6698847167107400e-37,
    5.1193527501298416e-38,    2.6452792047551823e-39,    1.3348703741932815e-40,    6.5819534428807159e-42,
    3.1728171453231489e-43,    1.4959816514194857e-44,    6.9024876679599122e-46,    3.1180395118762623e-47,
    1.3795723117004850e-48,    5.9810412960994406e-50,    2.5418737613110168e-51,    1.0593624182864413e-52,
    4.3312244853556127e-54,    1.7378346530941712e-55,    6.8452323073228971e-57,    2.6478589123211466e-58,
    1.0061626679937662e-59,    3.7570173607838419e-61,    1.3789552259943015e-62,    4.9763970319073015e-64
},
// x = 4.3100
{
    1.1598834567810891e-01,    8.9118766693559545e-02,    5.3956721529459600e-02,    2.6523962135021714e-02,
    1.0878360753322240e-02,    3.8081276153233107e-03,    1.1592415494808455e-03,    3.1157537791008812e-04,
    7.4872484828566628e-05,    1.6254672089755750e-05,    3.2161577507570425e-06,    5.8429789813210480e-07,
    9.8094721281775250e-08,    1.5302995105566239e-08,    2.2290906900609733e-09,    3.0447306107245307e-10,
    3.9145239191822047e-11,    4.7531322255556425e-12,    5.4671763858597401e-13,    5.9732543959115811e-14,
    6.2143405800537323e-15,    6.1700711869750394e-16,    5.8585103489300897e-17,    5.3297040760336648e-18,
    4.6536066480382799e-19,    3.9060787080633691e-20,    3.1564557290441726e-21,    2.4590224551974978e-22,
    1.8492039116970345e-23,    1.3439555737383955e-24,    9.4503420783494605e-26,    6.4361612573815441e-27,
    4.2496156886870017e-28,    2.7228084187696929e-29,    1.6943668789797542e-30,    1.0248914138533066e-31,
    6.0306751842843574e-33,    3.4545496937739780e-34,    1.9277805327324083e-35,    1.0487025087308474e-36,
    5.5647974716910567e-38,    2.8821022181433613e-39,    1.4577422830292256e-40,    7.2044441878114088e-42,
    3.4809272529092254e-43,    1.6450556513939280e-44,    7.6078961378217261e-46,    3.4446518175467535e-47,
    1.5276122347444224e-48,    6.6382033887745184e-50,    2.8276976477232882e-51,    1.1812149330150547e-52,
    4.8406152279147329e-54,    1.9467220381477400e-55,    7.6858060630339724e-57,    2.9799015567821435e-58,
    1.1349617492955236e-59,    4.2477825270571492e-61,    1.5626991526653298e-62,    5.6525766663480776e-64
},
// x = 4.3200
{
    1.1572026771964784e-01,    8.8974114752655875e-02,    5.3932688030303488e-02,    2.6552022124989812e-02,
    1.0908578105551471e-02,    3.8258177384242606e-03,    1.1669125678971054e-03,    3.1427528873389745e-04,
    7.5678926459961322e-05,    1.6464698497938733e-05,    3.2647432514344929e-06,    5.9441880346549521e-07,
    1.0001351076171855e-07,    1.5636912483327933e-08,    2.2828077409189627e-09,    3.1250866697377876e-10,
    4.0268695505273819e-11,    4.9005763084925878e-12,    5.6495226517184187e-13,    6.1864778085609860e-14,
    6.4507963434196936e-15,    6.4194241889517742e-16,    6.1091710898248251e-17,    5.5704303717586495e-18,
    4.8749157587405657e-19,    4.1012034298282744e-20,    3.3217265193298958e-21,    2.5937098242985436e-22,
    1.9549659690547877e-23,    1.4240837351253485e-24,    1.0036793767854397e-25,    6.8512817940559903e-27,
    4.5341151522739520e-28,    2.9117791791947832e-29,    1.8161332689448940e-30,    1.1010763518912321e-31,
    6.4938943088419404e-33,    3.7284709989583760e-34,    2.0854380094765797e-35,    1.1370843178360196e-36,
    6.0477060264513868e-38,    3.1394378763849275e-39,    1.5915662100718236e-40,    7.8839908269409020e-42,
    3.8180574240038433e-43,    1.8085485821870225e-44,    8.3833272672922104e-46,    3.8045177005044895e-47,
    1.6911027756871770e-48,    7.3656347254150759e-50,    3.1448177795548121e-51,    1.3167231548504691e-52,
    5.4084057215173567e-54,    2.1800975372226337e-55,    8.6270992298128907e-57,    3.3525926633637591e-58,
    1.2798636447679734e-59,    4.8011851512351581e-61,    1.7703753419476745e-62,    6.4186001268760054e-64
},
// x = 4.3300
{
    1.1545341982599266e-01,    8.8829855040983646e-02,    5.3908485617458972e-02,    2.6579871648998665e-02,
    1.0938716207992324e-02,    3.8435100157582576e-03,    1.1746030039875123e-03,    3.1698829477958299e-04,
    7.6491128307663251e-05,    1.6676705580905186e-05,    3.3138982759777047e-06,    6.0468160965072014e-07,
    1.0196362887226288e-07,    1.5977054960981013e-08,    2.3376510555220849e-09,    3.2073149443587681e-10,
    4.1420956789481107e-11,    5.0521470795539596e-12,    5.8374021110037841e-13,    6.4066753753963222e-14,
    6.6955468037117817e-15,    6.6781173267386414e-16,    6.3698188243840612e-17,    5.8213236732118006e-18,
    5.1061026671481878e-19,    4.3055066046410624e-20,    3.4951700942779997e-21,    2.7353833353904279e-22,
    2.0664702904477170e-23,    1.5087571983502409e-24,    1.0657941656396363e-25,    7.2919765484432980e-27,
    4.8368387298724152e-28,    3.1133189512418508e-29,    1.9462985456617146e-30,    1.1827042450668691e-31,
    6.9913539816288092e-33,    4.0233197576228052e-34,    2.2555325238272580e-35,    1.2326585920754529e-36,
    6.5711202715803152e-38,    3.4190031655098828e-39,    1.7372864251904476e-40,    8.6256564755167844e-42,
    4.1868562067061700e-43,    1.9878129169225119e-44,    9.2355267997379530e-46,    4.2009252928824329e-47,
    1.8716110965872322e-48,    8.1706423290485794e-50,    3.4965686985273013e-51,    1.4673771282817404e-52,
    6.0411185499522486e-54,    2.4407593556754783e-55,    9.6808811929791439e-57,    3.7707874659493346e-58,
    1.4428335695633481e-59,    5.4250326777777957e-61,    2.0050292759431115e-62,    7.2861353910952026e-64
},
// x = 4.3400
{
    1.1518779365580657e-01,    8.8685986607832459e-02,    5.3884116277120517e-02,    2.6607511634191325e-02,
    1.0968774931650643e-02,    3.8612041722429933e-03,    1.1823127439379846e-03,    3.1971438625363809e-04,
    7.7309104812507280e-05,    1.6890703808333041e-05,    3.3636273105884423e-06,    6.1508778935670041e-07,
    1.0394547759210575e-07,    1.6323517512773387e-08,    2.3936405310730577e-09,    3.2914530053406037e-10,
    4.2602670115484918e-11,    5.2079471213868749e-12,    6.0309655591337561e-13,    6.6340538715237062e-14,
    6.9488577810610348e-15,    6.9464723516739201e-16,    6.6408216038480388e-17,    6.0827831554992027e-18,
    5.3475790288990738e-19,    4.5193929322830147e-20,    3.6771666078020254e-21,    2.8843849022475043e-22,
    2.1840118778691228e-23,    1.5982205506951939e-24,    1.1315737523121358e-25,    7.7597467541733919e-27,
    5.1589008998685734e-28,    3.3282272803409394e-29,    2.0854176761594681e-30,    1.2701481838559279e-31,
    7.5254859804165643e-33,    4.3406341545224362e-34,    2.4390091264896187e-35,    1.3359897847304210e-36,
    7.1383201830922842e-38,    3.7226537846257129e-39,    1.8959258577673908e-40,    9.4349387679199032e-42,
    4.5902061336778670e-43,    2.1843240331651045e-44,    1.0171870589743256e-45,    4.6374782201220872e-47,
    2.0708587171861932e-48,    9.0612715722999452e-50,    3.8866304173115414e-51,    1.6348250906727892e-52,
    6.7459855304985171e-54,    2.7318171650834677e-55,    1.0860261744342322e-56,    4.2399067117796984e-58,
    1.6260707302083493e-59,    6.1280824353750971e-61,    2.2700848900637283e-62,    8.2683308254150673e-64
},
// x = 4.3500
{
    1.1492338094126578e-01,    8.8542508497523667e-02,    5.3859581977456363e-02,    2.6634943006194546e-02,
    1.0998754151396171e-02,    3.8788999343404101e-03,    1.1900416737537492e-03,    3.2245355300736819e-04,
    7.8132870280065816e-05,    1.7106703636996218e-05,    3.4139348541052947e-06,    6.2563882407410385e-07,
    1.0595946244911246e-07,    1.6676396205641895e-08,    2.4507963451282597e-09,    3.3775390478683191e-10,
    4.3814494923251344e-11,    5.3680812311319438e-12,    6.2303674172998615e-13,    6.8688255497580466e-14,
    7.2110027861612176e-15,    7.2248211077359672e-16,    6.9225599203823306e-17,    6.3552224581837411e-18,
    5.5997724183802150e-19,    4.7432837479594821e-20,    3.8681127669098911e-21,    3.0410721609497171e-22,
    2.3079000191862111e-23,    1.6927308222956889e-24,    1.2012237221944162e-25,    8.2561773793806838e-27,
    5.5014810427304254e-28,    3.5573522427179436e-29,    2.2340806820028665e-30,    1.3638057472018593e-31,
    8.0988877193743550e-33,    4.6820609620235161e-34,    2.6368819333812899e-35,    1.4476850061234560e-36,
    7.7528418007605315e-38,    4.0523949473568350e-39,    2.0685924194623061e-40,    1.0317805878764808e-41,
    5.0312437093441739e-43,    2.3996910221557885e-44,    1.1200421615973914e-45,    5.1181249475725270e-47,
    2.2907362654598821e-48,    1.0046378616010712e-49,    4.3190631953762650e-51,    1.8208898010647612e-52,
    7.5310227401670282e-54,    3.0567258520293287e-55,    1.2179839773042932e-56,    4.7660008530552363e-58,
    1.8320354901990590e-59,    6.9201543772204701e-61,    2.5693904741064960e-62,    9.3799985892097817e-64
},
// x = 4.3600
{
    1.1466017348727960e-01,    8.8399419749582861e-02,    5.3834884668759263e-02,    2.6662166689079090e-02,
    1.1028653745925881e-02,    3.8965970300577813e-03,    1.1977896792663384e-03,    3.2520578453888461e-04,
    7.8962438880267083e-05,    1.7324715510320391e-05,    3.4648254178616900e-06,    6.3633620410582733e-07,
    1.0800599253278570e-07,    1.7035788206826943e-08,    2.5091389583986604e-09,    3.4656118995511286e-10,
    4.5057103213227031e-11,    5.5326564605042773e-12,    6.4357658073860418e-13,    7.1112082676672026e-14,
    7.4822632179143745e-15,    7.5135058160563897e-16,    7.2154270886284174e-17,    6.6390701646330811e-18,
    5.8631268955138856e-19,    4.9776176555548547e-20,    4.0684225025407892e-21,    3.2058191457107701e-22,
    2.4384589373534451e-23,    1.7925580822093226e-24,    1.2749605923398205e-25,    8.7829415503990880e-27,
    5.8658270298602735e-28,    3.8015932488135239e-29,    2.3929147509213181e-30,    1.4641005382998817e-31,
    8.7143330476586102e-33,    5.0493628891500284e-34,    2.8502389717051430e-35,    1.5683971230395971e-36,
    8.4184964728446503e-38,    4.4103929927703828e-39,    2.2564858167090785e-40,    1.1280735424702707e-41,
    5.5133810459236354e-43,    2.6356684170516420e-44,    1.2329991985574614e-45,    5.6471907708196502e-47,
    2.5333195944637821e-48,    1.1135709741972263e-49,    4.7983457132000723e-51,    2.0275865063834814e-52,
    8.4051132574465433e-54,    3.4193228241064087e-55,    1.3655868011086043e-56,    5.3558213348998624e-58,
    2.0634796088691814e-59,    7.8122568673501483e-61,    2.9072700038900593e-62,    1.0637820213983668e-63
},
// x = 4.3700
{
    1.1439816317071973e-01,    8.8256719398909611e-02,    5.3810026283596421e-02,    2.6689183605321042e-02,
    1.1058473597727488e-02,    3.9142951889486388e-03,    1.2055566461405326e-03,    3.2797106999510859e-04,
    7.9797824647024923e-05,    1.7544749857940786e-05,    3.5163035255432061e-06,    6.4718142855695454e-07,
    1.1008548050660403e-07,    1.7401791791485425e-08,    2.5686891175636338e-09,    3.5557110284803189e-10,
    4.6331179740067642e-11,    5.7017821564454002e-12,    6.6473226281617306e-13,    7.3614256171167006e-14,
    7.7629285654837740e-15,    7.8128793665106556e-16,    7.5198296377174355e-17,    6.9347702957236235e-18,
    6.1381035909416358e-19,    5.2228511830257776e-20,    4.2785276655255675e-21,    3.3790169916966631e-22,
    2.5760284671645503e-23,    1.8979860612467809e-24,    1.3530123604822843e-25,    9.3418051959943671e-27,
    6.2532590002584046e-28,    4.0619040003605465e-29,    2.5625864694178572e-30,    1.5714838121828433e-31,
    9.3747837203324834e-33,    5.4444264064767917e-34,    3.0802473518775463e-35,    1.6988280742897934e-36,
    9.1393914909634321e-38,    4.7989878640773061e-39,    2.4609048894410477e-40,    1.2330756468857576e-41,
    6.0403292794713994e-43,    2.8941689156098576e-44,    1.3570210352476128e-45,    6.2294126801541369e-47,
    2.8008873880176517e-48,    1.2339988215648046e-49,    5.3294169669503408e-51,    2.2571427044279147e-52,
    9.3780983902682881e-54,    3.8238692375936069e-55,    1.5306435519638797e-56,    6.0168997432014605e-58,
    2.3234798921124091e-59,    8.8167269840138821e-61,    3.2885805298094072e-62,    1.2060576972678074e-63
},
// x = 4.3800
{
    1.1413734193965908e-01,    8.8114406475943860e-02,    5.3785008736957789e-02,    2.6715994675763757e-02,
    1.1088213593043112e-02,    3.9319941421135443e-03,    1.2133424598812425e-03,    3.3074939817378255e-04,
    8.0639041477877704e-05,    1.7766817095261642e-05,    3.5683737130440947e-06,    6.5817600532419773e-07,
    1.1219834262022670e-07,    1.7774506350300964e-08,    2.6294678580974395e-09,    3.6478765513526893e-10,
    4.7637422208550826e-11,    5.8755700023514727e-12,    6.8652036327651711e-13,    7.6197070563542670e-14,
    8.0532966148359469e-15,    8.1233056165368175e-16,    7.8361877139978908e-17,    7.2427828182822346e-18,
    6.4251813101429303e-19,    5.4794594606353596e-20,    4.4988787485320016e-21,    3.5610746658281343e-22,
    2.7209647606262538e-23,    2.0093128026846495e-24,    1.4356190799898971e-25,    9.9346319224629759e-27,
    6.6651733342630762e-28,    4.3392956090835772e-29,    2.7438041829292556e-30,    1.6864362003243979e-31,
    1.0083401581488703e-32,    5.8692700762776199e-34,    3.3281587862646430e-35,    1.8397324168548056e-36,
    9.9199522114539979e-38,    5.2207065174222728e-39,    2.6832555151523868e-40,    1.3475494866269017e-41,
    6.6161239071719183e-43,    3.1772771787262191e-44,    1.4931595206486313e-45,    6.8699773489892531e-47,
    3.0959403884047009e-48,    1.3671009371537285e-49,    5.9177222353142892e-51,    2.5120198780586976e-52,
    1.0460878235618080e-53,    4.2750955475136378e-55,    1.7151669774254708e-56,    6.7576356557341186e-58,
    2.6154756298385585e-59,    9.9473869725351808e-61,    3.7187763190480064e-62,    1.3669407954887866e-63
},
// x = 4.3900
{
    1.1387770181261973e-01,    8.7972480006829085e-02,    5.3759833926403974e-02,    2.6742600819580825e-02,
    1.1117873621833162e-02,    3.9496936222007664e-03,    1.2211470058403424e-03,    3.3354075752549509e-04,
    8.1486103133639689e-05,    1.7990927623017871e-05,    3.6210405283231445e-06,    6.6932145108482303e-07,
    1.1434499872156574e-07,    1.8154032397091390e-08,    2.6914965071083656e-09,    3.7421492416596828e-10,
    4.8976541471688211e-11,    6.0541340598834727e-12,    7.0895785074938485e-13,    7.8862880446744467e-14,
    8.3536736598555470e-15,    8.4451596973365127e-16,    8.1649354947340562e-17,    7.5635841686571154e-18,
    6.7248571570399858e-19,    5.7479369227517934e-20,    4.7299456348835608e-21,    3.7524197265944888e-22,
    2.8736410220763125e-23,    2.1268513420233222e-24,    1.5230334619000021e-25,    1.0563388130380119e-26,
    7.1030468340616704e-28,    4.6348398853731529e-29,    2.9373204904464726e-30,    1.8094695377559953e-31,
    1.0843561501697401e-32,    6.3260534190683765e-34,    3.5953154769651026e-35,    1.9919211179701588e-36,
    1.0764945765952906e-37,    5.6782773273682747e-39,    2.9250591201954021e-40,    1.4723222206332180e-41,
    7.2451521979101425e-43,    3.4872647926649286e-44,    1.6425634527257331e-45,    7.5745625170215003e-47,
    3.4212223907833892e-48,    1.5141744675518807e-49,    6.5692635023805904e-51,    2.7949373914257006e-52,
    1.1665522497968176e-53,    4.7782518205551700e-55,    1.9213960396344368e-56,    7.5873941279846555e-58,
    2.9433102351274503e-59,    1.1219718658868839e-60,    4.2039805262729688e-62,    1.5488099098062650e-63
},
// x = 4.4000
{
    1.1361923487783004e-01,    8.7830939013572182e-02,    5.3734503732212655e-02,    2.6769002954239642e-02,
    1.1147453577740491e-02,    3.9673933634068173e-03,    1.2289701692234506e-03,    3.3634513615571403e-04,
    8.2339023238061891e-05,    1.8217091826838602e-05,    3.6743085312588349e-06,    6.8061929128506511e-07,
    1.1652587226872282e-07,    1.8540471576412701e-08,    2.7547966861903882e-09,    3.8385705379422902e-10,
    5.0349261731048537e-11,    6.2375908113648440e-12,    7.3206209519184761e-13,    8.1614101797036236e-14,
    8.6643747181172958e-15,    8.7788283276145312e-16,    8.5065216130364368e-17,    7.8976677918177792e-18,
    7.0376471776539210e-19,    6.0287980339548800e-20,    4.9722183751675632e-21,    3.9534991139403808e-22,
    3.0344482742087319e-23,    2.2509304169980196e-24,    1.6155215052296032e-25,    1.1230148384249758e-26,
    7.5684411211151434e-28,    4.9496728056938624e-29,    3.1439348808578967e-30,    1.9411287984890119e-31,
    1.1658865114265472e-32,    6.8170863495111544e-34,    3.8831563962378568e-35,    2.1562656094905491e-36,
    1.1679506470732463e-37,    6.1746455602556907e-39,    3.1879618431959825e-40,    1.6082908627089071e-41,
    7.9321828397352520e-43,    3.8266064897308485e-44,    1.8064872337359858e-45,    8.3493820616081675e-47,
    3.7797431614948142e-48,    1.6766455585507912e-49,    7.2906547555373474e-51,    3.1088987569901626e-52,
    1.3005392583103816e-53,    5.3391632949344015e-55,    2.1518206786060898e-56,    8.5166138420467535e-58,
    3.3112775442956745e-59,    1.2651057832854180e-60,    4.7510652542439490e-62,    1.7543406796409018e-63
},
// x = 4.4100
{
    1.1336193329249070e-01,    8.7689782514200773e-02,    5.3709020017524187e-02,    2.6795201995465832e-02,
    1.1176953358054628e-02,    3.9850931014767934e-03,    1.2368118350966424e-03,    3.3916252182682671e-04,
    8.3197815277503707e-05,    1.8445320076812613e-05,    3.7281822935037637e-06,    6.9207106012801363e-07,
    1.1874139034178926e-07,    1.8933926671158665e-08,    2.8193903142871874e-09,    3.9371825521116685e-10,
    5.1756320739303645e-11,    6.4260592027720128e-12,    7.5585087603371915e-13,    8.4453213373462168e-14,
    8.9857237514006412e-15,    9.1247101350159463e-16,    8.8614095942909825e-17,    8.2455446963930934e-18,
    7.3640870243918501e-19,    6.3225780402147893e-20,    5.2262079925766011e-21,    4.1647799703225329e-22,
    3.2037961562107338e-23,    2.3818952090973363e-24,    1.7133631568037500e-25,    1.1937101046798520e-26,
    8.0630072611040951e-28,    5.2849981679099497e-29,    3.3644965186482988e-30,    2.0819941453432357e-31,
    1.2533155397285702e-32,    7.3448392165744748e-34,    4.1932239846126259e-35,    2.3337021219094093e-36,
    1.2669163051553117e-37,    6.7129899914901904e-39,    3.4737443986552371e-40,    1.7564279796423480e-41,
    8.6823980002644236e-43,    4.1979977295734183e-44,    1.9863002731974709e-45,    9.2012350753495262e-47,
    4.1748034509742813e-48,    1.8560818104507213e-49,    8.0891826155202288e-51,    3.4572205016431068e-52,
    1.4495276081681023e-53,    5.9642917190537753e-55,    2.4092092140110443e-56,    9.5569270536916687e-58,
    3.7241732859313704e-59,    1.4260810828970280e-60,    5.3677409617965640e-62,    1.9865420122252202e-63
},
// x = 4.4200
{
    1.1310578928204947e-01,    8.7549009522916707e-02,    5.3683384628486083e-02,    2.6821198857208461e-02,
    1.1206372863676300e-02,    4.0027925737046929e-03,    1.2446718883931294e-03,    3.4199290196019403e-04,
    8.4062492600615879e-05,    1.8675622727056109e-05,    3.7826663983384906e-06,    7.0367830056100854e-07,
    1.2099198365451066e-07,    1.9334501610156449e-08,    2.8852996105685536e-09,    4.0380280778357591e-10,
    5.3198470005013157e-11,    6.6196606873237286e-12,    7.8034239045872726e-13,    8.7382758144338522e-14,
    9.3180538910346164e-15,    9.4832159854233188e-16,    9.2300783043597675e-17,    8.6077440260666399e-18,
    7.7047326415600892e-19,    6.6298337459297865e-20,    5.4924473179569340e-21,    4.3867504940703964e-22,
    3.3821137552597032e-23,    2.5201081178922220e-24,    1.8168530018955866e-25,    1.2686554190166617e-26,
    8.5884906274930350e-28,    5.6420914441574472e-29,    3.5999071869752184e-30,    2.2326831006088314e-31,
    1.3470532151079249e-32,    7.9119534848811815e-34,    4.5271712932448168e-35,    2.5252363165099513e-36,
    1.3739867808478738e-37,    7.2967407480096859e-39,    3.7843326922090788e-40,    1.9177878376840116e-41,
    9.5014279894185721e-43,    4.6043737513130387e-44,    2.1834972009379109e-45,    1.0137559294356499e-46,
    4.6100222866267656e-48,    2.0542058999117507e-49,    8.9728727972807066e-51,    3.8435638815420416e-52,
    1.6151534863734076e-53,    6.6608030531759013e-55,    2.6966386588796905e-56,    1.0721292590240457e-57,
    4.1873522808442429e-59,    1.6070695774400158e-60,    6.0626562831153860e-62,    2.2487966153551357e-63
},
// x = 4.4300
{
    1.1285079513948500e-01,    8.7408619050247521e-02,    5.3657599394396363e-02,    2.6846994451606012e-02,
    1.1235711999082122e-02,    4.0204915189335283e-03,    1.2525502139198665e-03,    3.4483626363820811e-04,
    8.4933068418033020e-05,    1.8908010115282217e-05,    3.8377654405246335e-06,    7.1544256426250365e-07,
    1.2327808656581165e-07,    1.9742301475756399e-08,    2.9525470973188806e-09,    4.1411505989916671e-10,
    5.4676474999656067e-11,    6.8185192696742916e-12,    8.0555526182307206e-13,    9.0405344741186111e-14,
    9.6617076681602787e-15,    9.8547693202777973e-16,    9.6130224098300103e-17,    8.9848136477565477e-18,
    8.0601609727123461e-19,    6.9511443176302194e-20,    5.7714918555660825e-21,    4.6199208262215229e-22,
    3.5698504726721531e-23,    2.6659495695275383e-24,    1.9263009870241420e-25,    1.3480941796776351e-26,
    9.1467360153149592e-28,    6.0223038413549353e-29,    3.8511243965506386e-30,    2.3938528443103801e-31,
    1.4475368423394875e-32,    8.5212530963223427e-34,    4.8867695986863124e-35,    2.7319482352899233e-36,
    1.4898027853264872e-37,    7.9295984627125290e-39,    4.1218092426502388e-40,    2.0935130313649039e-41,
    1.0395388728194177e-42,    5.0489302152830162e-44,    2.3997089576693476e-45,    1.1166489250301564e-46,
    5.0893667468894416e-48,    2.2729104742294470e-49,    9.9505629455280558e-51,    4.2719697185347682e-52,
    1.7992268122162592e-53,    7.4366421756891267e-55,    3.0175282461988219e-56,    1.2024143281194518e-57,
    4.7067919943266012e-59,    1.8105011241779844e-60,    6.8455094401857967e-62,    2.5449063411038637e-63
},
// x = 4.4400
{
    1.1259694322459890e-01,    8.7268610103194130e-02,    5.3631666127846117e-02,    2.6872589688953036e-02,
    1.1264970672289527e-02,    4.0381896775553480e-03,    1.2604466963641158e-03,    3.4769259360636321e-04,
    8.5809555802077517e-05,    1.9142492562372912e-05,    3.8934840261574020e-06,    7.2736541162843567e-07,
    1.2560013709118257e-07,    2.0157432511416701e-08,    3.0211556028378054e-09,    4.2465942981841018e-10,
    5.6191115366926016e-11,    7.0227615507167188e-12,    8.3150854821314289e-13,    9.3523648940533123e-14,
    1.0017037249000923e-14,    1.0239806502092479e-15,    1.0010752850595337e-16,    9.3773207570184898e-18,
    8.4309706904587983e-19,    7.2871121151803087e-20,    6.0639206805739428e-21,    4.8648239720421787e-22,
    3.7674769260436175e-23,    2.8198188607805858e-24,    2.0420331763110055e-25,    1.4322830263212705e-26,
    9.7396930173100726e-28,    6.4270665799320849e-29,    4.1191646691830068e-30,    2.5662026472013285e-31,
    1.5552327937640139e-32,    9.1757565532848684e-34,    5.2739165199492523e-35,    2.9549975895399443e-36,
    1.6150538560612958e-37,    8.6155548335261433e-39,    4.4884254696981825e-40,    2.2848416311152972e-41,
    1.1370922242533393e-42,    5.5351455624320669e-44,    2.6367148359564708e-45,    1.2296919550269321e-46,
    5.6171844348825710e-48,    2.5142744332175871e-49,    1.1031982437952203e-50,    4.7468966563862813e-52,
    2.0037491830886021e-53,    8.3006152983693055e-55,    3.3756764996021962e-56,    1.3481549345898038e-57,
    5.2891631276883696e-59,    2.0390935339795986e-60,    7.7271725610296625e-62,    2.8791428974582588e-63
},
// x = 4.4500
{
    1.1234422596331661e-01,    8.7128981685376264e-02,    5.3605586624860702e-02,    2.6897985477667592e-02,
    1.1294148794821813e-02,    4.0558867915111187e-03,    1.2683612202999436e-03,    3.5056187827532979e-04,
    8.6691967686472295e-05,    1.9379080371952592e-05,    3.9498267725174092e-06,    7.3944841175806593e-07,
    1.2795857691392371e-07,    2.0580002129281143e-08,    3.0911482643527358e-09,    4.3544040653296946e-10,
    5.7743185134298733e-11,    7.2325167730010947e-12,    8.5822175114406910e-13,    9.6740415174008549e-14,
    1.0384404675230053e-14,    1.0638777168327596e-15,    1.0423797325057346e-16,    9.7858525011187085e-18,
    8.8177829493763309e-19,    7.6383635513309086e-20,    6.3703373693717142e-21,    5.1220167584825611e-22,
    3.9754858887650897e-23,    2.9821350401436726e-24,    2.1643925428534098e-25,    1.5214925221021372e-26,
    1.0369421675101899e-27,    6.8578954018609648e-29,    4.4051070052803737e-30,    2.7504764459937533e-31,
    1.6706383582476738e-32,    9.8786897672302609e-34,    5.6906446695405641e-35,    3.1956294092655181e-36,
    1.7504819383800672e-37,    9.3589146860710120e-39,    4.8866149106429061e-40,    2.4931148887932916e-41,
    1.2435240417819582e-42,    6.0668052293746631e-44,    2.8964555502855076e-45,    1.3538573721806368e-46,
    6.1982388886413957e-48,    2.7805807239730031e-49,    1.2227839802580271e-50,    5.2732631626495864e-52,
    2.2309336222806286e-53,    9.2624808638345846e-55,    3.7753022125366173e-56,    1.5111399419054893e-57,
    5.9419080083200731e-59,    2.2958858601667290e-60,    8.7198303611151504e-62,    3.2563045476887081e-63
},
// x = 4.4600
{
    1.1209263584699648e-01,    8.6989732797174701e-02,    5.3579362665040396e-02,    2.6923182724259469e-02,
    1.1323246281673511e-02,    4.0735826042905049e-03,    1.2762936701946838e-03,    3.5344410372304205e-04,
    8.7580316866066289e-05,    1.9617783829964401e-05,    4.0067983079219871e-06,    7.5169314243933846e-07,
    1.3035385139625124e-07,    2.1010118917750873e-08,    3.1625485309433009e-09,    4.4646255063074844e-10,
    5.9333492926888618e-11,    7.4479168667741519e-12,    8.8571482440089619e-13,    1.0005845806716649e-13,
    1.0764182109529923e-14,    1.1052144593801814e-15,    1.0852700788243021e-16,    1.0211016620235831e-17,
    9.2212421626778441e-19,    8.0055499805016297e-20,    6.6913709637868248e-21,    5.3920808288583189e-22,
    4.1943932683515981e-23,    3.1533378274437979e-24,    2.2937397966306084e-25,    1.6160078688928939e-26,
    1.1038098418672097e-27,    7.3163953196069796e-29,    4.7100965450843488e-30,    2.9474655687250211e-31,
    1.7942837024338823e-32,    1.0633499718887690e-33,    6.1391308720523904e-35,    3.4551800770334116e-36,
    1.8968852194999760e-37,    1.0164319645562516e-38,    5.3190074334103450e-40,    2.7197855430687920e-41,
    1.3596172267187167e-42,    6.6480278677656906e-44,    3.1810474212326923e-45,    1.4902079096623686e-46,
    6.8377481850455586e-48,    3.0743357847473818e-49,    1.3549918453835668e-50,    5.8564936321287552e-52,
    2.4832263045050955e-53,    1.0333049773707530e-54,    4.2210897366950757e-56,    1.6933601067105582e-57,
    6.6733276172300352e-59,    2.5842754392706666e-60,    9.8371348056709763e-62,    3.6817794870261261e-63
},
// x = 4.4700
{
    1.1184216543174708e-01,    8.6850862435871021e-02,    5.3552996011699404e-02,    2.6948182333299021e-02,
    1.1352263051275863e-02,    4.0912768609315106e-03,    1.2842439304153234e-03,    3.5633925569678910e-04,
    8.8474615996568065e-05,    1.9858613204248184e-05,    4.0644032715757928e-06,    7.6410119013371060e-07,
    1.3278640959025770e-07,    2.1447892649048065e-08,    3.2353801664774462e-09,    4.5773049516753195e-10,
    6.0962862183601887e-11,    7.6690964966452249e-12,    9.1400818302407544e-13,    1.0348066400746823e-13,
    1.1156752086432975e-14,    1.1480386061814682e-15,    1.1298025963137153e-16,    1.0653442107258498e-17,
    9.6420168033131847e-19,    8.3893486176933370e-20,    7.0276769703335723e-21,    5.6756236760910055e-22,
    4.4247391250681170e-23,    3.3338885735688491e-24,    2.4304542505209288e-25,    1.7161296571606072e-26,
    1.1748022307991846e-27,    7.8042656181684371e-29,    5.0353484338931157e-30,    3.1580116185777872e-31,
    1.9267339507817316e-32,    1.1443868978983126e-33,    6.6217059859061965e-35,    3.7350837712944109e-36,
    2.0551222320992288e-37,    1.1036773530706870e-38,    5.7884445183125601e-40,    2.9664267696403211e-41,
    1.4862214985761091e-42,    7.2832937281575763e-44,    3.4927977655062160e-45,    1.6399048245315552e-46,
    7.5414270163325024e-48,    3.3982907860236465e-49,    1.5011181515031733e-50,    6.5025689806804348e-52,
    2.7633304515299964e-53,    1.1524295878427728e-54,    4.7182390204483599e-56,    1.8970302837780072e-57,
    7.4926781808228995e-59,    2.9080590957076685e-60,    1.1094377548142614e-61,    4.1616166636285362e-63
},
// x = 4.4800
{
    1.1159280733775284e-01,    8.6712369595784447e-02,    5.3526488412004337e-02,    2.6972985207386745e-02,
    1.1381199025462554e-02,    4.1089693080200110e-03,    1.2922118852348493e-03,    3.5924731961531347e-04,
    8.9374877594290792e-05,    2.0101578744120901e-05,    4.1226463134208863e-06,    7.7667414996048956e-07,
    1.3525670424873024e-07,    2.1893434286772089e-08,    3.3096672525591812e-09,    4.6924894654523993e-10,
    6.2632131375602169e-11,    7.8961931088846220e-12,    9.4312271244106891e-13,    1.0700999274186984e-13,
    1.1562507768541293e-14,    1.1923993244160220e-15,    1.1760353865536786e-16,    1.1113779886657902e-17,
    1.0080800230192751e-18,    8.7904634884583839e-20,    7.3799383956645174e-21,    5.9732797158847111e-22,
    4.6670887323910669e-23,    3.5242712619294950e-24,    2.5749347260706885e-25,    1.8221746520742355e-26,
    1.2501621591294988e-27,    8.3233051229537151e-29,    5.3821519020459167e-30,    3.3830095249067697e-31,
    2.0685913912424088e-32,    1.2313731141235203e-33,    7.1408653659794936e-35,    4.0368793457955717e-36,
    2.2261162456050981e-37,    1.1981669589925965e-38,    6.2979956857800526e-40,    3.2347418245059468e-41,
    1.6242589083236505e-42,    7.9774753808379989e-44,    3.8342215909329009e-45,    1.8042167517466170e-46,
    8.3155325416365752e-48,    3.7554648297703572e-49,    1.6625886563685190e-50,    7.2180821538301580e-52,
    3.0742326084609247e-53,    1.2849477749863937e-54,    5.2725208818140065e-56,    2.1246140093258314e-57,
    8.4102783504513923e-59,    3.2714789664391929e-60,    1.2508682136504608e-61,    4.7026048966959868e-63
},
// x = 4.4900
{
    1.1134455424860783e-01,    8.6574253268405996e-02,    5.3499841597111623e-02,    2.6997592247123597e-02,
    1.1410054129435620e-02,    4.1266596936891733e-03,    1.3001974188385354e-03,    3.6216828057092018e-04,
    9.0281114035906787e-05,    2.0346690679959129e-05,    4.1815320939862038e-06,    7.8941362568066127e-07,
    1.3776519183582378e-07,    2.2346855993446978e-08,    3.3854341914877702e-09,    4.8102268539678771e-10,
    6.4342154227098492e-11,    8.1293469793598208e-12,    9.7307977774582665e-13,    1.1064947900445780e-13,
    1.1981853208219748e-14,    1.2383472590213403e-15,    1.2240284342739544e-16,    1.1592703512924194e-17,
    1.0538311540241039e-18,    9.2096264108804362e-20,    7.7488668194219935e-21,    6.2857114012604505e-22,
    4.9220336808972178e-23,    3.7249935533468003e-24,    2.7276005007216326e-25,    1.9344766174869849e-26,
    1.3301460595128020e-27,    8.8754177468489386e-29,    5.7518745709779794e-30,    3.6234107706833140e-31,
    2.2204978137999880e-32,    1.3247287222306797e-33,    7.6992800070940505e-35,    4.3622176733473522e-36,
    2.4108599645480633e-37,    1.3004819708307228e-38,    6.8509761527416092e-40,    3.5265744319768550e-41,
    1.7747297908953238e-42,    8.7358709593979323e-44,    4.2080597033160013e-45,    1.9845293286837526e-46,
    9.1669143414688153e-48,    4.1491702818039675e-49,    1.8409711210622574e-50,    8.0102990135875602e-52,
    3.4192315310363997e-53,    1.4323272856556545e-54,    5.8903380485351022e-56,    2.3788507106084918e-57,
    9.4376280998585756e-59,    3.6792734503184784e-60,    1.4099218196549678e-61,    5.3123612398337864e-63
},
// x = 4.5000
{
    1.1109739891065704e-01,    8.6436512442530283e-02,    5.3473057282303518e-02,    2.7022004351081947e-02,
    1.1438828291731590e-02,    4.1443477676187598e-03,    1.3082004153301754e-03,    3.6510212333158861e-04,
    9.1193337558213036e-05,    2.0593959222783916e-05,    4.2410652842365041e-06,    8.0232122968022237e-07,
    1.4031233253758919e-07,    2.2808271138060145e-08,    3.4627057092283315e-09,    4.9305656747756088e-10,
    6.6093799938469793e-11,    8.3687012621153856e-12,    1.0039012331279354e-12,    1.1440223417458563e-13,
    1.2415203614860034e-14,    1.2859345725275077e-15,    1.2738436626384420e-16,    1.2090909889066940e-17,
    1.1015296447005474e-18,    9.6475980105425432e-20,    8.1352035057255107e-21,    6.6136103799162816e-22,
    5.1901930272277054e-23,    3.9365878761190576e-24,    2.8888922982717939e-25,    2.0533871795070007e-26,
    1.4150246961991437e-27,    9.4626183304606970e-29,    6.1459669972175242e-30,    3.8802268060495035e-31,
    2.3831369894977267e-32,    1.4249023086522023e-33,    8.2998084107234814e-35,    4.7128694839619806e-36,
    2.6104205545776267e-37,    1.4112485722255575e-38,    7.4509658060438526e-40,    3.8439199728294460e-41,
    1.9387191906922414e-42,    9.5642401270138753e-44,    4.6172983405469238e-45,    2.1823556550235201e-46,
    1.0103068830600953e-47,    4.5830404273163924e-49,    2.0379890504883487e-50,    8.8872251092095972e-52,
    3.8019699359315269e-53,    1.5961925369359452e-54,    6.5787925505000253e-56,    2.6627858148291964e-57,
    1.0587540588008044e-58,    4.1367338405085360e-60,    1.5887440041789117e-61,    5.9994296433669709e-63
},
// x = 4.5100
{
    1.1085133413234603e-01,    8.6299146104384342e-02,    5.3446137167123618e-02,    2.7046222415777210e-02,
    1.1467521444187823e-02,    4.1620332810343198e-03,    1.3162207587382642e-03,    3.6804883234309413e-04,
    9.2111560257905945e-05,    2.0843394563847798e-05,    4.3012505654207381e-06,    8.1539858295300863e-07,
    1.4289859027235498e-07,    2.3277794303591318e-08,    3.5415068583936353e-09,    5.0535552456349598e-10,
    6.7887953411735443e-11,    8.6144020386020874e-12,    1.0356094314532139e-12,    1.1827144796596291e-13,
    1.2862985627813925e-14,    1.3352149858363065e-15,    1.3255449899769075e-16,    1.2609120005690318e-17,
    1.1512528186563150e-18,    1.0105168769488009e-19,    8.5397205545668934e-21,    6.9576986959282355e-22,
    5.4722144898317781e-23,    4.1596125630864752e-24,    3.0592733244150748e-25,    2.1792767314420719e-26,
    1.5050839252092372e-27,    1.0087038790176803e-28,    6.5659674667845813e-30,    4.1545326581773694e-31,
    2.5572372979857989e-32,    1.5323727956446274e-33,    8.9455092197901704e-35,    5.0907337292328174e-36,
    2.8259450180573265e-37,    1.5311412987951014e-38,    8.1018295874079908e-40,    4.1889375319535532e-41,
    2.1174037963221934e-42,    1.0468842980709307e-43,    5.0651904584534305e-45,    2.3993476581393804e-46,
    1.1132198514597916e-47,    5.0610596560165905e-49,    2.2555367244424743e-50,    9.8576788837749239e-52,
    4.2264693896657793e-53,    1.7783408943278574e-54,    7.3457601074467459e-56,    2.9798040580236979e-57,
    1.1874288364031079e-58,    4.6497672575948452e-60,    1.7897352425601055e-61,    6.7733910863684519e-63
},
// x = 4.5200
{
    1.1060635278357804e-01,    8.6162153237754210e-02,    5.3419082935511085e-02,    2.7070247335640184e-02,
    1.1496133521909039e-02,    4.1797159867062606e-03,    1.3242583330221212e-03,    3.7100839173113406e-04,
    9.3035794091366657e-05,    2.1095006874224182e-05,    4.3620926289199033e-06,    8.2864731508302723e-07,
    1.4552443270096011e-07,    2.3755541294531422e-08,    3.6218630212370316e-09,    5.1792456535577666e-10,
    6.9725515478385887e-11,    8.8665983675610073e-12,    1.0682272339975838e-12,    1.2226039014716022e-13,
    1.3325637595095762e-14,    1.3862438199641387e-15,    1.3791983879974758e-16,    1.3148079701165672e-17,
    1.2030808451487250e-18,    1.0583160110206747e-19,    8.9632220944237687e-21,    7.3187300373564225e-22,
    5.7687756932548147e-23,    4.3946530375804009e-24,    3.2392303492783491e-25,    2.3125353819794460e-26,
    1.6006254926466327e-27,    1.0750934589377481e-28,    7.0135070530680413e-30,    4.4474707481580712e-31,
    2.7435745120630775e-32,    1.6476514073894516e-33,    9.6396546690774555e-35,    5.4978465067881479e-36,
    3.0586659425602243e-37,    1.6608866359427956e-38,    8.8077393909331192e-40,    4.5639628690891822e-41,
    2.3120594234917846e-42,    1.1456482125269065e-43,    5.5552788026598609e-45,    2.6373084398114680e-46,
    1.2263276507438034e-47,    5.5875964010945126e-49,    2.4956956368193086e-50,    1.0931371918623285e-51,
    4.6971686374553964e-53,    1.9807605950348049e-54,    8.1999722181365554e-56,    3.3336663248845829e-57,
    1.3313765433272834e-58,    5.2249665663704408e-60,    2.0155806445393000e-61,    7.6449864788931059e-63
},
// x = 4.5300
{
    1.1036244779507887e-01,    8.6025532824109033e-02,    5.3391896255933831e-02,    2.7094080002989997e-02,
    1.1524664463234056e-02,    4.1973956389488266e-03,    1.3323130220779628e-03,    3.7398078530345914e-04,
    9.3966050874455569e-05,    2.1348806304398753e-05,    4.4235961760943459e-06,    8.4206906422628301e-07,
    1.4819033123683895e-07,    2.4241629144390640e-08,    3.7037999126563174e-09,    5.3076877639213261e-10,
    7.1607403129582889e-11,    9.1254423355682405e-12,    1.1017780203360104e-12,    1.2637241229398957e-13,
    1.3803609857954949e-14,    1.4390780387681452e-15,    1.4348719415135434e-16,    1.3708560443434685e-17,
    1.2570968353652567e-18,    1.1082425515707089e-19,    9.4065455174399943e-21,    7.6974910313706479e-22,
    6.0805854627955049e-23,    4.6423230502134223e-24,    3.4292748389504795e-25,    2.4535739485398536e-26,
    1.7019678729491174e-27,    1.1456691548846845e-28,    7.4903149519013551e-30,    4.7602549262003149e-31,
    2.9429747481440947e-32,    1.7712837579584761e-33,    1.0385744901585961e-34,    5.9363905807306187e-36,
    3.3099076470748831e-37,    1.8012668742417901e-38,    9.5731975810946242e-40,    4.9715223807678138e-41,
    2.5240690878298660e-42,    1.2534548166089283e-43,    6.0914209112686657e-45,    2.8982056862224361e-46,
    1.3506116762449329e-47,    6.1674390754509371e-49,    2.7607524708872400e-50,    1.2118996872322077e-51,
    5.2189657014255432e-53,    2.2056504779426961e-54,    9.1511067264604350e-56,    3.7285503837828993e-57,
    1.4923666859959737e-58,    5.8696880313520041e-60,    2.2692828933681490e-61,    8.6262537786400850e-63
},
// x = 4.5400
{
    1.1011961215776907e-01,    8.5889283842722319e-02,    5.3364578781520823e-02,    2.7117721308007724e-02,
    1.1553114209702772e-02,    4.2150719936189665e-03,    1.3403847097449208e-03,    3.7696599655201451e-04,
    9.4902342282317920e-05,    2.1604802983863702e-05,    4.4857659181306147e-06,    8.5566547709212790e-07,
    1.5089676105595813e-07,    2.4736176123195665e-08,    3.7873435832085642e-09,    5.4389332296476056e-10,
    7.3534549748744671e-11,    9.3910891082462181e-12,    1.1362856983884001e-12,    1.3061094957423802e-13,
    1.4297365041421803e-14,    1.4937762926754019e-15,    1.4926359097195589e-16,    1.4291360133989512e-17,
    1.3133869416680273e-18,    1.1603851686762685e-19,    9.8705627585626186e-21,    8.0948025885621401e-22,
    6.4083851714211262e-23,    4.9032659685395320e-24,    3.6299441380767467e-25,    2.6028249978262800e-26,
    1.8094471489617604e-27,    1.2206833013186590e-28,    7.9982241082329490e-30,    5.0941747369996961e-31,
    3.1563175920651352e-32,    1.9038520683505312e-33,    1.1187523204126214e-34,    6.4087055361744295e-36,
    3.5810927523124693e-37,    1.9531242401083270e-38,    1.0403062246585776e-39,    5.4143481264083349e-41,
    2.7549317124873148e-42,    1.3711068889147205e-43,    6.6778162045048568e-45,    3.1841862298055833e-46,
    1.4871450506385526e-47,    6.8058352694695323e-49,    3.0532187498220218e-50,    1.3432323830233693e-51,
    5.7972641082098900e-53,    2.4554416972937430e-54,    1.0209887715822798e-55,    4.1690959177865986e-57,
    1.6723687754242700e-58,    6.5921375455707987e-60,    2.5541989030333141e-61,    9.7306809254733009e-63
},
// x = 4.5500
{
    1.0987783892214395e-01,    8.5753405270791463e-02,    5.3337132150193534e-02,    2.7141172138710645e-02,
    1.1581482706023310e-02,    4.2327448081150880e-03,    1.3484732798110027e-03,    3.7996400865508072e-04,
    9.5844679849198190e-05,    2.1863007020713814e-05,    4.5486065758877691e-06,    8.6943820892410097e-07,
    1.5364420110660099e-07,    2.5239301744974724e-08,    3.8725204221355512e-09,    5.5730345004484114e-10,
    7.5507905346522986e-11,    9.6636969821468401e-12,    1.1717747146242269e-12,    1.3497952256521805e-13,
    1.4807378350929564e-14,    1.5503989634351501e-15,    1.5525627890505713e-16,    1.4897303934585912e-17,
    1.3720404598839150e-18,    1.2148359737452004e-19,    1.0356181620064208e-20,    8.5115212981602749e-22,
    6.7529501408934146e-23,    5.1781561216876115e-24,    3.8418027056730412e-25,    2.7607439356719968e-26,
    1.9234179357968279e-27,    1.3004027390814001e-28,    8.5391771494914630e-30,    5.4505999277499122e-31,
    3.3845394101496504e-32,    2.0459775202115262e-33,    1.2048992218564408e-34,    6.9172986083194059e-36,
    3.8737492031807376e-37,    2.1173653206442364e-38,    1.1302574313231043e-39,    5.8953939966785240e-41,
    3.0062715186395488e-42,    1.4994762416549507e-43,    7.3190353296486041e-45,    3.4975918586419978e-46,
    1.6371009407071147e-47,    7.5085344971606881e-49,    3.3758523139093776e-50,    1.4884305845367675e-51,
    6.4380236393324438e-53,    2.7328216138954432e-54,    1.1388195666232156e-55,    4.6604542925773549e-57,
    1.8735743680807610e-58,    7.4014663548044095e-60,    2.8740806027231211e-61,    1.0973376373527675e-62
},
// x = 4.5600
{
    1.0963712119766054e-01,    8.5617896083554232e-02,    5.3309557984795940e-02,    2.7164433380927128e-02,
    1.1609769900039381e-02,    4.2504138413757206e-03,    1.3565786160190075e-03,    3.8297480447942585e-04,
    9.6793074968265001e-05,    2.2123428501245213e-05,    4.6121228797432409e-06,    8.8338892348028413e-07,
    1.5643313411900187e-07,    2.5751126775230557e-08,    3.9593571603998959e-09,    5.7100448321367071e-10,
    7.7528436798187639e-11,    9.9434274373127097e-12,    1.2082700644278079e-12,    1.3948173910462219e-13,
    1.5334137875118701e-14,    1.6090082099145969e-15,    1.6147273776613035e-16,    1.5527245117259813e-17,
    1.4331499347242942e-18,    1.2716906430139315e-19,    1.0864347142923189e-20,    8.9485408759290187e-22,
    7.1150910991260233e-23,    5.4677002021496379e-24,    4.0654434063993749e-25,    2.9278101483804522e-26,
    2.0442543505327621e-27,    1.3851096085946593e-28,    9.1152326414860144e-30,    5.8309852119074987e-31,
    3.6286368559867305e-32,    2.1983227542773605e-33,    1.2974431188442445e-34,    7.4648562289666355e-36,
    4.1895177732693264e-37,    2.2949658027704897e-38,    1.2277386647602479e-39,    6.4178531077474769e-41,
    3.2798481505319311e-42,    1.6395094647875780e-43,    8.0200519426851808e-45,    3.8409764767854092e-46,
    1.8017616048912055e-47,    8.2818348019249163e-49,    3.7316807890851206e-50,    1.6489194521468447e-51,
    7.1478160341357994e-53,    3.0407600761007042e-54,    1.2699188899494207e-55,    5.2083435461715225e-57,
    2.0984214734577360e-58,    8.3078772959933396e-60,    3.2331203015210874e-61,    1.2371259195816200e-62
},
// x = 4.5700
{
    1.0939745215213219e-01,    8.5482755254403450e-02,    5.3281857893224024e-02,    2.7187505918272084e-02,
    1.1637975742697867e-02,    4.2680788538780436e-03,    1.3647006020723724e-03,    3.8599836658245615e-04,
    9.7747538891444324e-05,    2.2386077489556210e-05,    4.6763195694381758e-06,    8.9751929301316150e-07,
    1.5926404661482641e-07,    2.6271773238400125e-08,    4.0478808737315447e-09,    5.8500182960038422e-10,
    7.9597128083426346e-11,    1.0230445190521649e-11,    1.2457973026260396e-12,    1.4412129617515829e-13,
    1.5878144894929540e-14,    1.6696680149589431e-15,    1.6792068415610384e-16,    1.6182065938227967e-17,
    1.4968112684201559e-18,    1.3310485451075809e-19,    1.1396043026576232e-20,    9.4067936665731794e-22,
    7.4956556958625013e-23,    5.7726387269842451e-24,    4.3014888596191299e-25,    3.1045281978402000e-26,
    2.1723510298942417e-27,    1.4751021841825833e-28,    9.7285716834522026e-30,    6.2368753024916505e-31,
    3.8896705839356319e-32,    2.3615945220287670e-33,    1.3968414304176641e-34,    8.0542563359861630e-36,
    4.5301600830751776e-37,    2.4869755480927129e-38,    1.3333595291916119e-39,    6.9851765109393479e-41,
    3.5775675904670198e-42,    1.7922341320863631e-43,    8.7862771221844304e-45,    4.2171247271682644e-46,
    1.9825282337929438e-47,    9.1326335596076462e-49,    4.1240272258615645e-50,    1.8262666565673234e-51,
    7.9338861146448542e-53,    3.3825383228820365e-54,    1.4157437437458965e-55,    5.8191091336192019e-57,
    2.3496215758066357e-58,    9.3227426750004735e-60,    3.6360011347436107e-61,    1.3943270952284765e-62
},
// x = 4.5800
{
    1.0915882501112992e-01,    8.5347981754998914e-02,    5.3254033468554195e-02,    2.7210390632123171e-02,
    1.1666100188016604e-02,    4.2857396076363945e-03,    1.3728391216409873e-03,    3.8903467721437905e-04,
    9.8708082729265012e-05,    2.2650964027150996e-05,    4.7412013939224315e-06,    9.1183099824900693e-07,
    1.6213742891650279e-07,    2.6801364425301731e-08,    4.1381189856847415e-09,    5.9930097882629772e-10,
    8.1714980528576102e-11,    1.0524918249220911e-11,    1.2843825541805487e-12,    1.4890198182346342e-13,
    1.6439914199092192e-14,    1.7324442333368367e-15,    1.7460807824414688e-16,    1.6862678536268232e-17,
    1.5631238326606890e-18,    1.3930128727832073e-19,    1.1952293098602710e-20,    9.8872522025435853e-22,
    7.8955300788361295e-23,    6.0937475607797856e-24,    4.5505928486618860e-25,    3.2914290727930450e-26,
    2.3081241981495327e-27,    1.5706957515324768e-28,    1.0381504859663875e-29,    6.6699102294046036e-31,
    4.1687691809582156e-32,    2.5365464995213073e-33,    1.5035830213720826e-34,    8.6885814940066417e-36,
    4.8975671656967817e-37,    2.6945240263423389e-38,    1.4477772980319638e-39,    7.6010933135842934e-41,
    3.9014939231558476e-42,    1.9587655050007048e-43,    9.6235966260623226e-45,    4.6290721976582792e-46,
    2.1809316509117829e-47,    1.0068482845114058e-48,    4.5565381032863088e-50,    2.0221962321108282e-51,
    8.8042188449256005e-53,    3.7617807632557148e-54,    1.5779070507222555e-55,    6.4997910127950466e-57,
    2.6301895426709491e-58,    1.0458735026104813e-59,    4.0879531468153002e-61,    1.5710611751320236e-62
},
// x = 4.5900
{
    1.0892123305739111e-01,    8.5213574555377730e-02,    5.3226086289170381e-02,    2.7233088401597376e-02,
    1.1694143193052372e-02,    4.3033958662005607e-03,    1.3809940583669315e-03,    3.9208371832036347e-04,
    9.9674717450710937e-05,    2.2918098132545245e-05,    4.8067731111990102e-06,    9.2632572836676524e-07,
    1.6505377515639404e-07,    2.7340024900567120e-08,    4.2300992707050287e-09,    6.1390750395582774e-10,
    8.3883013051290639e-11,    1.0827017966156083e-11,    1.3240525250460586e-12,    1.5382767711377721e-13,
    1.7019974406121934e-14,    1.7974046407925294e-15,    1.8154313072350725e-16,    1.7570025856184711e-17,
    1.6321905839252212e-18,    1.4576907789803486e-19,    1.2534162835944425e-20,    1.0390930821188459e-21,
    8.3156405326449552e-23,    6.4318395028052243e-24,    4.8134417928006904e-25,    3.4890714987311454e-26,
    2.4520127875585468e-27,    1.6722235304009519e-28,    1.1076479565872985e-29,    7.1318309559938053e-31,
    4.4671333289983021e-32,    2.7239822728520538e-33,    1.6181902769458634e-34,    9.3711328775123865e-36,
    5.2937686158439003e-37,    2.9188261317263951e-38,    1.5717005096829264e-39,    8.2696323135680843e-41,
    4.2538620131577360e-42,    2.1403137728464551e-43,    1.0538411218142612e-44,    5.0801273404351854e-46,
    2.3986439464123797e-47,    1.1097649759786127e-48,    5.0332139094895084e-50,    2.2386037380016442e-51,
    9.7676128845137993e-53,    4.1824899102519145e-54,    1.7581939046369201e-55,    7.2581977154381137e-57,
    2.9434767205140177e-58,    1.1729972124732547e-59,    4.5948156246015535e-61,    1.7697003199174628e-62
},
// x = 4.6000
{
    1.0868466963023508e-01,    8.5079532624061607e-02,    5.3198017918890561e-02,    2.7255600103528396e-02,
    1.1722104717869093e-02,    4.3210473946540904e-03,    1.3891652958701807e-03,    3.9514547154270951e-04,
    1.0064745388308479e-04,    2.3187489800874555e-05,    4.8730394881681511e-06,    9.4100518097646659e-07,
    1.6801358328581891e-07,    2.7887880510059897e-08,    4.3238498572064509e-09,    6.2882706245401412e-10,
    8.6102262407661779e-11,    1.1136919094701096e-11,    1.3648345131969563e-12,    1.5890235811688495e-13,
    1.7618868292932390e-14,    1.8646189842266364e-15,    1.8873430994426281e-16,    1.8305082597979254e-17,
    1.7041181823008893e-18,    1.5251935173065204e-19,    1.3142760939312438e-20,    1.0918887342260947e-21,
    8.7569551826527230e-23,    6.7877659408666265e-24,    5.0907562845528581e-25,    3.6980433090012712e-26,
    2.6044796138071459e-27,    1.7800376447695724e-28,    1.1816087729724821e-29,    7.6244853108504934e-31,
    4.7860402107768087e-32,    2.9247585052502475e-33,    1.7412213086189992e-34,    1.0105445170620523e-35,
    5.7209423602569999e-37,    3.1611884081024903e-38,    1.7058928245986053e-39,    8.9951452572418309e-41,
    4.6370911637368780e-42,    2.3381918707438045e-43,    1.1537680308917840e-44,    5.5738952451977531e-46,
    2.6374911226823418e-47,    1.2229182150208798e-48,    5.5584425287099651e-50,    2.4775728476049589e-51,
    1.0833761246689079e-52,    4.6490847737558766e-54,    1.9585794691711361e-55,    8.1029881112304562e-57,
    3.2932075481355238e-58,    1.3152177767101068e-59,    5.1631063602592516e-61,    1.9928981224271314e-62
},
// x = 4.6100
{
    1.0844912812498564e-01,    8.4945854928162812e-02,    5.3169829907092257e-02,    2.7277926612444530e-02,
    1.1749984725506223e-02,    4.3386939596124210e-03,    1.3973527177542420e-03,    3.9821991822301911e-04,
    1.0162630271188004e-04,    2.3459149003505040e-05,    4.9400053004709701e-06,    9.5587106209714439e-07,
    1.7101735508391510e-07,    2.8445058388278238e-08,    4.4193992306585614e-09,    6.4406539715062181e-10,
    8.8373783441800028e-11,    1.1454799844894573e-11,    1.4067564198239037e-12,    1.6413009793482259e-13,
    1.8237153130177455e-14,    1.9341590330274892e-15,    1.9619034922688774e-16,    1.9068856192360942e-17,
    1.7790171138803302e-18,    1.5956365870884892e-19,    1.3779240962478723e-20,    1.1472224807854612e-21,
    9.2204857663028721e-23,    7.1624185744773100e-24,    5.3832926950133808e-25,    3.9189628797997203e-26,
    2.7660126089685715e-27,    1.8945101427548746e-28,    1.2603073945217213e-29,    8.1498342516474229e-31,
    5.1268481725525631e-32,    3.1397882963314956e-33,    1.8732722989395914e-34,    1.0895302441071606e-35,
    6.1814250900101891e-37,    3.4230157105764205e-38,    1.8511771618804899e-39,    9.7823318379896397e-41,
    5.0537998303853281e-42,    2.5538239198006256e-43,    1.2628969173682397e-44,    6.1143034178417218e-46,
    2.8994668368655804e-47,    1.3472980185297480e-48,    6.1370356845595885e-50,    2.7413934960847696e-51,
    1.2013339728107711e-52,    5.1664430450533670e-54,    2.1812486877545313e-55,    9.0437616424404255e-57,
    3.6835200519566783e-58,    1.4742859986956078e-59,    5.8000985945339733e-61,    2.2436222086781746e-62
},
// x = 4.6200
{
    1.0821460199240007e-01,    8.4812540433487363e-02,    5.3141523788836839e-02,    2.7300068800547046e-02,
    1.1777783181947386e-02,    4.3563353292209684e-03,    1.4055562076117445e-03,    4.0130703940437299e-04,
    1.0261127448066331e-04,    2.3733085687646694e-05,    5.0076753323327258e-06,    9.7092508613429013e-07,
    1.7406559616634767e-07,    2.9011686965742555e-08,    4.5167762366833308e-09,    6.5962833721083659e-10,
    9.0698649337890675e-11,    1.1780841940188838e-11,    1.4498467607025029e-12,    1.6951506876186323e-13,
    1.8875401024437275e-14,    2.0060986315756758e-15,    2.0392025436063664e-16,    1.9862387803241656e-17,
    1.8570018168361883e-18,    1.6691398831238248e-19,    1.4444802998201070e-20,    1.2052093286904656e-21,
    9.7072894743149945e-23,    7.5567312100446870e-24,    5.6918448500352275e-25,    4.1524806318517810e-26,
    2.9371261146433956e-27,    2.0160340686831859e-28,    1.3440344042238953e-29,    8.7099584786674724e-31,
    5.4910016581152643e-32,    3.3700447446374864e-33,    2.0149799937528404e-34,    1.1744755049418698e-35,
    6.6777233976955719e-37,    3.7058183329062517e-38,    2.0084401348841776e-39,    1.0636266560902222e-40,
    5.5068214675261793e-42,    2.7887543373435144e-43,    1.3820500031379137e-44,    6.7056297282326794e-46,
    3.1867473324927690e-47,    1.4839874297649087e-48,    6.7742687108196922e-50,    3.0325817285689462e-51,
    1.3318103836980492e-52,    5.7399474370146673e-54,    2.4286179825310067e-55,    1.0091157883072750e-56,
    4.1190106239191674e-58,    1.6521508551836435e-59,    6.5139064707067567e-61,    2.5251905241406271e-62
},
// x = 4.6300
{
    1.0798108473810533e-01,    8.4679588104636508e-02,    5.3113101084993099e-02,    2.7322027537689315e-02,
    1.1805500056089184e-02,    4.3739712731530990e-03,    1.4137756490299775e-03,    4.0440681583350855e-04,
    1.0360237959096520e-04,    2.4009309775968969e-05,    5.0760543764056780e-06,    9.8616897585681426e-07,
    1.7715881599385890e-07,    2.9587895976366616e-08,    4.6160100841616003e-09,    6.7552179911252290e-10,
    9.3077951874730744e-11,    1.2115230674916758e-11,    1.4941346777358545e-12,    1.7506154398228510e-13,
    1.9534199267362134e-14,    2.0805137529444242e-15,    2.1193331129079379e-16,    2.0686753357876367e-17,
    1.9381908112712719e-18,    1.7458278502704813e-19,    1.5140695422577786e-20,    1.2659691746457713e-21,
    1.0218470864313997e-22,    7.9716816308731489e-24,    6.0172457801778299e-25,    4.3992806016828473e-26,
    3.1183622380433106e-27,    2.1450245898501907e-28,    1.4330974113228562e-29,    9.3070654165611660e-31,
    5.8800364290276213e-32,    3.6165647251948067e-33,    2.1670243506876904e-34,    1.2658137658043451e-35,
    7.2125256651580174e-37,    4.0112196319921636e-38,    2.1786368066235401e-39,    1.1562427607693736e-40,
    5.9992215925358094e-42,    3.0446576685375850e-43,    1.5121207289152031e-44,    7.3525327035920717e-46,
    3.5017076598101011e-47,    1.6341710037432573e-48,    7.4759239443893745e-50,    3.3539014034575353e-51,
    1.4760995013003158e-52,    6.3755365777691146e-54,    2.7033591376902211e-55,    1.1256966360175000e-56,
    4.6047835229956906e-58,    1.8509813770830617e-59,    7.3135799168116786e-61,    2.8413117116438781e-62
},
// x = 4.6400
{
    1.0774856992204043e-01,    8.4546996905106031e-02,    5.3084563302359798e-02,    2.7343803691356285e-02,
    1.1833135319710218e-02,    4.3916015626080203e-03,    1.4220109255963752e-03,    4.0751922796300396e-04,
    1.0459962830218126e-04,    2.4287831166219116e-05,    5.1451472336115915e-06,    1.0016044623735442e-06,
    1.8029752788066081e-07,    3.0173816464812073e-08,    4.7171303483491215e-09,    6.9175178763006160e-10,
    9.5512801682762513e-11,    1.2458154972482725e-11,    1.5396499506730425e-12,    1.8077390030544972e-13,
    2.0214150691891848e-14,    2.1574825538191407e-15,    2.2023909399893595e-16,    2.1543064605323288e-17,
    2.0227068329457269e-18,    1.8258296430160096e-19,    1.5868216699682038e-20,    1.3296269991982996e-21,
    1.0755183849529566e-22,    8.4082935448839158e-24,    6.3603695474580039e-25,    4.6600820855073021e-26,
    3.3102922739036987e-27,    2.2819201806000174e-28,    1.5278220020051066e-29,    9.9434965838033426e-31,
    6.2955850869238342e-32,    3.8804528944713538e-33,    2.3301313532638586e-34,    1.3640088408484802e-35,
    7.7887147502827219e-37,    4.3409641827455776e-38,    2.3627957871455776e-39,    1.2566727845271136e-40,
    6.5343161572395464e-42,    3.3233491945283901e-43,    1.6540797280850648e-44,    8.0600843578922272e-46,
    3.8469392924754305e-47,    1.7991440432814450e-48,    8.2483380602782880e-50,    3.7083879191448839e-51,
    1.6356257004085799e-52,    7.0797608927102076e-54,    3.0084255810093738e-55,    1.2554247666636747e-56,
    5.1465055854965113e-58,    2.0731908854297741e-59,    8.2092099701289438e-61,    3.1961300308824028e-62
},
// x = 4.6500
{
    1.0751705115790557e-01,    8.4414765797383565e-02,    5.3055911933787149e-02,    2.7365398126644680e-02,
    1.1860688947440325e-02,    4.4092259703085628e-03,    1.4302619209039401e-03,    4.1064425595346219e-04,
    1.0560303073148119e-04,    2.4568659730842784e-05,    5.2149587129838518e-06,    1.0172332851092396e-06,
    1.8348224900266707e-07,    3.0769580793825668e-08,    4.8201669740018728e-09,    7.0832439682473978e-10,
    9.8004328503609711e-11,    1.2809807444283149e-11,    1.5864230090054506e-12,    1.8665661993870479e-13,
    2.0915874035668553e-14,    2.2370854306595298e-15,    2.2884747258041409e-16,    2.2432470203911614e-17,
    2.1106769709848835e-18,    1.9092792901715202e-19,    1.6628717248377180e-20,    1.3963130679066897e-21,
    1.1318633765290174e-22,    8.8676386130557395e-24,    6.7221331520526368e-25,    4.9356413588835563e-26,
    3.5135181952334362e-27,    2.4271838664777434e-28,    1.6285527405289357e-29,    1.0621735370291740e-30,
    6.7393829144989730e-32,    4.1628859357861622e-33,    2.5050760005164088e-34,    1.4695569339636884e-35,
    8.4093815243356568e-37,    4.6969264987677626e-38,    2.5620246965246811e-39,    1.3655548131247896e-40,
    7.1156913234612106e-42,    3.6267963762999617e-43,    1.8089812852686892e-44,    8.8338057625835870e-46,
    4.2252692569883762e-47,    1.9803226500313131e-48,    9.0984536959019968e-50,    4.0993741472232069e-51,
    1.8119563343067097e-52,    7.8598439499920208e-54,    3.3470812977922202e-55,    1.3997466995208877e-56,
    5.7504666778449468e-58,    2.3214638295339574e-59,    9.2120456641927326e-61,    3.5942753177232238e-62
},
// x = 4.6600
{
    1.0728652211261770e-01,    8.4282893743043721e-02,    5.3027148458297708e-02,    2.7386811706243620e-02,
    1.1888160916730036e-02,    4.4268442704989168e-03,    1.4385285185566294e-03,    4.1378187967570148e-04,
    1.0661259685372816e-04,    2.4851805316607365e-05,    5.2854936315092759e-06,    1.0330571917801429e-06,
    1.8671350040556816e-07,    3.1375322651558804e-08,    4.9251502785106792e-09,    7.2524581104170453e-10,
    1.0055368145213062e-10,    1.3170384449362543e-11,    1.6344849440429134e-12,    1.9271429279864511e-13,
    2.1640004311762644e-14,    2.3194050771282584e-15,    2.3776862152334791e-16,    2.3356156838419596e-17,
    2.2022328096738815e-18,    1.9963158648393821e-19,    1.7423601373269096e-20,    1.4661631398905897e-21,
    1.1910079516127439e-22,    9.3508385616980138e-24,    7.1034985222227150e-25,    5.2267534754111753e-26,
    3.7286742160393748e-27,    2.5813045313337780e-28,    1.7356542233298253e-29,    1.1344415244551328e-30,
    7.2132740526934842e-32,    4.4651170589416731e-33,    2.6926854825989429e-34,    1.5829888123677644e-35,
    9.0778393145337369e-37,    5.0811203565368706e-38,    2.7775160186950510e-39,    1.4835773080232444e-40,
    7.7472247460748414e-42,    3.9571311977921397e-43,    1.9779703176030546e-44,    9.6797055800352035e-46,
    4.6397809005978817e-47,    2.1792546601848886e-48,    1.0033875741571635e-49,    4.5305187712936695e-51,
    2.0068156952584280e-52,    8.7237497886276922e-54,    3.7229326336139012e-55,    1.5602641333378439e-56,
    6.4236464786961899e-58,    2.5987854994964723e-59,    1.0334623715625179e-60,    4.0409185347859109e-62
},
// x = 4.6700
{
    1.0705697650577230e-01,    8.4151379702841783e-02,    5.2998274341205825e-02,    2.7408045290415847e-02,
    1.1915551207820195e-02,    4.4444562389422317e-03,    1.4468106021746756e-03,    4.1693207871294427e-04,
    1.0762833650140625e-04,    2.5137277744227862e-05,    5.3567568139695495e-06,    1.0490779383690563e-06,
    1.8999180701274315e-07,    3.1991177058868550e-08,    5.0321109550448370e-09,    7.4252230591345624e-10,
    1.0316202928099609e-10,    1.3540086154810867e-11,    1.6838675211716663e-12,    1.9895161876126079e-13,
    2.2387193186833240e-14,    2.4045265428104080e-15,    2.4701302819350107e-16,    2.4315350367680400e-17,
    2.2975105744477019e-18,    2.0870836598074763e-19,    1.8254329261806999e-20,    1.5393186840087237e-21,
    1.2530835806399708e-22,    9.8590673817786201e-24,    7.5054745908524937e-25,    5.5342541478782492e-26,
    3.9564284292969681e-27,    2.7447982903883402e-28,    1.8495121887571718e-29,    1.2114328413076434e-30,
    7.7192180324895077e-32,    4.7884807685974988e-33,    2.8938425534209103e-34,    1.7048721201137330e-35,
    9.7976393098851107e-37,    5.4957087632101114e-38,    3.0105533730100228e-39,    1.6114829465955888e-40,
    8.4331084743441010e-42,    4.3166634764770663e-43,    2.1622899196074892e-44,    1.0604321798339317e-45,
    5.0938364335374263e-47,    2.3976315402911171e-48,    1.1062932706150163e-49,    5.0058372479337327e-51,
    2.2221002998678381e-52,    9.6802567961944053e-54,    4.1399632665492507e-55,    1.7387501679031177e-56,
    7.1737882356316880e-58,    2.9084749130086067e-59,    1.1590912377089797e-60,    4.5418335235820285e-62
},
// x = 4.6800
{
    1.0682840810911151e-01,    8.4020222636804837e-02,    5.2969291034236601e-02,    2.7429099736979417e-02,
    1.1942859803711840e-02,    4.4620616529181863e-03,    1.4551080553998617e-03,    4.2009483236301448e-04,
    1.0865025936455827e-04,    2.5425086807995320e-05,    5.4287530927823883e-06,    1.0652972890999607e-06,
    1.9331769763301310e-07,    3.2617280376600103e-08,    5.1410800757047395e-09,    7.6016024936988562e-10,
    1.0583056064780404e-10,    1.3919116596907927e-11,    1.7346031922961082e-12,    2.0537340995151155e-13,
    2.3158109366845649e-14,    2.4925372932483629e-15,    2.5659150152948170e-16,    2.5311317003347560e-17,
    2.3966512821876747e-18,    2.1817323685267042e-19,    1.9122419049600400e-20,    1.6159271029222121e-21,
    1.3182275457439835e-22,    1.0393553618642537e-23,    7.9291194621256023e-25,    5.8590217154038673e-26,
    4.1974845235775130e-27,    2.9182099323991541e-28,    1.9705346852226802e-29,    1.2934434955459870e-30,
    8.2592966806919025e-32,    5.1343979166928663e-33,    3.1094891120005779e-34,    1.8358138401317231e-35,
    1.0572586991899237e-36,    5.9430146107014280e-38,    3.2625182321922476e-39,    1.7500727446273241e-40,
    9.1778735901703917e-42,    4.7078952145788637e-43,    2.3632895155937100e-44,    1.1614766924697330e-45,
    5.5911013923278782e-47,    2.6373013248524361e-48,    1.2194743601567478e-49,    5.5297356252673300e-51,
    2.4598956214167770e-52,    1.0739038755327243e-53,    4.6025726560601019e-55,    1.9371671768467387e-56,
    8.0094802056591476e-58,    3.2542212068147729e-59,    1.2996470964725394e-60,    5.1034656334255240e-62
},
// x = 4.6900
{
    1.0660081074599857e-01,    8.3889421504321535e-02,    5.2940199975643643e-02,    2.7449975901289961e-02,
    1.1970086690136243e-02,    4.4796602912204088e-03,    1.4634207619007408e-03,    4.2327011964052986e-04,
    1.0967837499073083e-04,    2.5715242275407730e-05,    5.5014873078423433e-06,    1.0817170164121561e-06,
    1.9669170496823000e-07,    3.3253770312849423e-08,    5.2520890946831940e-09,    7.7816610265483133e-10,
    1.0856048438473911e-10,    1.4307683743020551e-11,    1.7867251084663424e-12,    2.1198459307286780e-13,
    2.3953438990470062e-14,    2.5835272713171142e-15,    2.6651518095279598e-16,    2.6345364520567281e-17,
    2.4998008959381866e-18,    2.2804172718325656e-19,    2.0029448956078870e-20,    1.6961419653075959e-21,
    1.3865831814331541e-22,    1.0955582755576004e-23,    8.3755426719933012e-25,    6.2019792002631802e-26,
    4.4525835818876235e-27,    3.1021144342166299e-28,    2.0991533006721732e-29,    1.3807872460130083e-30,
    8.8357214200723250e-32,    5.5043810550547404e-33,    3.3406300048757413e-34,    1.9764629139345265e-35,
    1.1406759655536954e-36,    6.4255320614060709e-38,    3.5348971172272534e-39,    1.9002104810833092e-40,
    9.9864167102415326e-42,    5.1335360694566981e-43,    2.5824336668841500e-44,    1.2718776914568582e-45,
    6.1355711826229600e-47,    2.9002826840930193e-48,    1.3439290827463489e-49,    6.1070474750816030e-51,
    2.7224944022014721e-52,    1.1910753735024403e-53,    5.1156183046456823e-55,    2.1576864952357576e-56,
    8.9402455589205360e-58,    3.6401238966954480e-59,    1.4568626724874947e-60,    5.7330079739803257e-62
},
// x = 4.7000
{
    1.0637417829089822e-01,    8.3758975264229790e-02,    5.2911002590326016e-02,    2.7470674636223404e-02,
    1.1997231855525198e-02,    4.4972519341538852e-03,    1.4717486053778068e-03,    4.2645791927910150e-04,
    1.1071269278492946e-04,    2.6007753886803685e-05,    5.5749643063613213e-06,    1.0983389009339521e-06,
    2.0011436562070433e-07,    3.3900785930206058e-08,    5.3651698514354296e-09,    7.9654642134915544e-10,
    1.1135302977078832e-10,    1.4705999554258445e-11,    1.8402671326935428e-12,    2.1879021177736680e-13,
    2.4773886030287471e-14,    2.6775889599654297e-15,    2.7679554549736274e-16,    2.7418843501320273e-17,
    2.6071104841598919e-18,    2.3832994305756667e-19,    2.0977059492680198e-20,    1.7801232464919545e-21,
    1.4583001245520470e-22,    1.1546499694792122e-23,    8.8459075462272423e-25,    6.5640964582295203e-26,
    4.7225059664268240e-27,    3.2971185511572981e-28,    2.2358244564292200e-29,    1.4737966186741399e-30,
    9.4508409853068651e-32,    5.9000401051999544e-33,    3.5883370626095051e-34,    2.1275130286509884e-35,
    1.2304525089758889e-36,    6.9459387138173629e-38,    3.8292893017596528e-39,    2.0628274464800825e-40,
    1.0864028488008655e-41,    5.5965200273804505e-43,    2.8213115846508181e-44,    1.3924764135217995e-45,
    6.7315998727113193e-47,    3.1887802175481988e-48,    1.4807499577879451e-49,    6.7430742166570806e-51,
    3.0124166907556442e-52,    1.3207141564683962e-53,    5.6844621999316295e-55,    2.4027101012747386e-56,
    9.9766416019294079e-58,    4.0707374066039559e-59,    1.6326670877168355e-60,    6.4384861173700145e-62
},
// x = 4.7100
{
    1.0614850466886315e-01,    8.3628882874902638e-02,    5.2881700289944281e-02,    2.7491196792159254e-02,
    1.2024295290981474e-02,    4.5148363635322368e-03,    1.4800914695686069e-03,    4.2965820973353363e-04,
    1.1175322200958240e-04,    2.6302631354998646e-05,    5.6491889427088700e-06,    1.1151647314558943e-06,
    2.0358622010046990e-07,    3.4558467652975698e-08,    5.4803545738575844e-09,    8.1530785640033893e-10,
    1.1420944680652427e-10,    1.5114280048894587e-11,    1.8952638529551899e-12,    2.2579542907673980e-13,
    2.5620172701930304e-14,    2.7748174463486757e-15,    2.8744442316318810e-16,    2.8533148611212076e-17,
    2.7187363846385581e-18,    2.4905458843303571e-19,    2.1966955745815140e-20,    1.8680375777909550e-21,
    1.5335345738572221e-22,    1.2167711339540600e-23,    9.3414336599920944e-25,    6.9463924264216646e-26,
    5.0080732931259611e-27,    3.5038624867794178e-28,    2.3810307686022144e-29,    1.5728239782545934e-30,
    1.0107149577244091e-31,    6.3230883632567588e-33,    3.8537533841617067e-34,    2.2897055816174476e-35,
    1.3270561491253495e-36,    7.5071085993308080e-38,    4.1474150606852673e-39,    2.2389275376503992e-40,
    1.1816424260961563e-41,    6.1000233710382477e-43,    3.0816474029916240e-44,    1.5241874685358979e-45,
    7.3839314224037129e-47,    3.5052010769595378e-48,    1.6311324335896611e-49,    7.4436291345723478e-51,
    3.3324317617387308e-52,    1.4641130696192705e-53,    6.3150218393258975e-55,    2.6748944886491763e-56,
    1.1130369260961949e-57,    4.5511203079849814e-59,    1.8292075860616747e-60,    7.2288521629871952e-62
},
// x = 4.7200
{
    1.0592378385502643e-01,    8.3499143294332404e-02,    5.2852294473035487e-02,    2.7511543216964272e-02,
    1.2051276990249512e-02,    4.5324133626749464e-03,    1.4884492382528144e-03,    4.3287096918202545e-04,
    1.1279997178451300e-04,    2.6599884364923734e-05,    5.7241660782521576e-06,    1.1321963049035363e-06,
    2.0710781283238445e-07,    3.5226957274381452e-08,    5.5976758814735183e-09,    8.3445715515855861e-10,
    1.1713100649146219e-10,    1.5532745366555729e-11,    1.9517505953921597e-12,    2.3300552979516205e-13,
    2.6493039881286188e-14,    2.8753104873794337e-15,    2.9847400049896331e-16,    2.9689719910505083e-17,
    2.8348403731711125e-18,    2.6023298563547851e-19,    2.3000909736917331e-20,    1.9600585048389524e-21,
    1.6124495595496003e-22,    1.2820689281172644e-23,    9.8633994030194448e-25,    7.3499374728019366e-26,
    5.3101504999912654e-27,    3.7230216458035955e-28,    2.5352824803954628e-29,    1.6782426581415183e-30,
    1.0807295479197232e-31,    6.7753488588935787e-33,    4.1380978836679684e-34,    2.4638328333542891e-35,
    1.4309878689390819e-36,    8.1121260647625843e-38,    4.4911244998991771e-39,    2.4295927232286792e-40,
    1.2849776998855795e-41,    6.6474840376495636e-43,    3.3653112709393535e-44,    1.6680050417361433e-45,
    8.0977335466917958e-47,    3.8521730304137771e-48,    1.7963843069679227e-49,    8.2150854188568180e-51,
    3.6855820902698644e-52,    1.6226955332753770e-53,    7.0138262769540651e-55,    2.9771769448838391e-56,
    1.2414393858592690e-57,    5.0868897546567366e-59,    2.0488736018014541e-60,    8.1140891743193166e-62
},
// x = 4.7300
{
    1.0570000987409987e-01,    8.3369755480213012e-02,    5.2822786525127549e-02,    2.7531714755976735e-02,
    1.2078176949686298e-02,    4.5499827164044975e-03,    1.4968217952572365e-03,    4.3609617552837284e-04,
    1.1385295108692057e-04,    2.6899522573267314e-05,    5.7999005811956778e-06,    1.1494354263097506e-06,
    2.1067969216306456e-07,    3.5906397963743580e-08,    5.7171667886297931e-09,    8.5400116241925798e-10,
    1.2011900110400313e-10,    1.5961619833189010e-11,    2.0097634376997416e-12,    2.4042592306418909e-13,
    2.7393247529895534e-14,    2.9791685767224775e-15,    3.0989683241845273e-16,    3.0890044200203072e-17,
    2.9555898371535688e-18,    2.7188309649802628e-19,    2.4080762861953297e-20,    2.0563667552091822e-21,
    1.6952152231165758e-22,    1.3506972595127229e-23,    1.0413144654617932e-24,    7.7758558516359838e-26,
    5.6296480134482886e-27,    3.9553084740873390e-28,    2.6991189688209292e-29,    1.7904481515585050e-30,
    1.1554090160170884e-31,    7.2587610881559531e-33,    4.4426701149820026e-34,    2.6507412603855189e-35,
    1.5427840765175821e-36,    8.7643005985386655e-38,    4.8624070065696674e-39,    2.6359889058289309e-40,
    1.3970752719416194e-41,    7.2426224715490687e-43,    3.6743313264710768e-44,    1.8250096035113027e-45,
    8.8786344293489692e-47,    4.2325640887869877e-48,    1.9779359794811225e-49,    9.0644285841655498e-51,
    4.0752095676961389e-52,    1.7980283782996899e-53,    7.7880776735839529e-55,    3.3128044716765825e-56,
    1.3843078317556040e-57,    5.6842816473349711e-59,    2.2943234183501267e-60,    9.1053271031575569e-62
},
// x = 4.7400
{
    1.0547717679987817e-01,    8.3240718390020937e-02,    5.2793177818852260e-02,    2.7551712251991125e-02,
    1.2104995168232453e-02,    4.5675442110434325e-03,    1.5052090244607760e-03,    4.3933380640417231e-04,
    1.1491216875137020e-04,    2.7201555608118981e-05,    5.8763973264207348e-06,    1.1668839087865913e-06,
    2.1430241036765612e-07,    3.6596934273636990e-08,    5.8388607076986606e-09,    8.7394682147218169e-10,
    1.2317474448397146e-10,    1.6401132026810220e-11,    2.0693392227146459e-12,    2.4806214486044881e-13,
    2.8321575128674882e-14,    3.0864950132610990e-15,    3.2172585225561297e-16,    3.2135656404015042e-17,
    3.0811579541981048e-18,    2.8402354416121933e-19,    2.5208428412831871e-20,    2.1571505156309620e-21,
    1.7820091078484272e-22,    1.4228170749942288e-23,    1.0992073572911000e-24,    8.2253282693951904e-26,
    5.9675240170553496e-27,    4.2014743897351470e-28,    2.8731103294696989e-29,    1.9098593671664875e-30,
    1.2350517891205435e-31,    7.7753881411758182e-33,    4.7688553901927561e-34,    2.8513351200244690e-35,
    1.6630190151972129e-36,    9.4671826621440248e-38,    5.2634013618674567e-39,    2.8593722086352236e-40,
    1.5186548549639906e-41,    7.8894640825815731e-43,    4.0109066202714642e-44,    1.9963751669208874e-45,
    9.7327625186315793e-47,    4.6495038253954245e-48,    2.1773516223413999e-49,    9.9993136553096696e-51,
    4.5049841622953441e-52,    1.9918359085806211e-53,    8.6457188749552964e-55,    3.6853656056678433e-56,
    1.5432330036937114e-57,    6.3502171153235068e-59,    2.5685136890631494e-60,    1.0214971433455341e-61
},
// x = 4.7500
{
    1.0525527875474867e-01,    8.3112030981093710e-02,    5.2763469714057896e-02,    2.7571536545243313e-02,
    1.2131731647383535e-02,    4.5850976344113514e-03,    1.5136108097993492e-03,    4.4258383917102680e-04,
    1.1597763346979111e-04,    2.7505993068616596e-05,    5.9536611953247340e-06,    1.1845435734967126e-06,
    2.1797652365644041e-07,    3.7298712147026619e-08,    5.9627914522890080e-09,    8.9430117515688552e-10,
    1.2629957231775802e-10,    1.6851514844040008e-11,    2.1305155722000909e-12,    2.5591986058666562e-13,
    2.9278822120100400e-14,    3.1973959710622136e-15,    3.3397438206349036e-16,    3.3428140987042718e-17,
    3.2117238759097231e-18,    2.9667363555295835e-19,    2.6385894183221349e-20,    2.2626057191201588e-21,
    1.8730164604055179e-22,    1.4985966633538368e-23,    1.1601657502857966e-24,    8.6995945657597229e-26,
    6.3247868271388337e-27,    4.4623118096057883e-28,    3.0578590431722545e-29,    2.0369199523988747e-30,
    1.3199745902359140e-31,    8.3274242468356667e-33,    5.1181302092300980e-34,    3.0665802399513213e-35,
    1.7923073311056140e-36,    1.0224580592274550e-37,    5.6964065607993757e-39,    3.1010957159869933e-40,
    1.6504933623199341e-41,    8.5923634296429319e-43,    4.3774210620252817e-44,    2.1833771361848793e-45,
    1.0666789655524660e-46,    5.1064065303696089e-48,    2.3963413280635734e-49,    1.1028127539691128e-50,
    4.9789352463395374e-52,    2.2060153045838863e-53,    9.5955075926749394e-55,    4.0988254226912140e-56,
    1.7199762807049467e-57,    7.0923759615481793e-59,    2.8747321195983457e-60,    1.1456845905939670e-61
},
// x = 4.7600
{
    1.0503430990920690e-01,    8.2983692210707644e-02,    5.2733663557920564e-02,    2.7591188473396144e-02,
    1.2158386391161524e-02,    4.6026427758218301e-03,    1.5220270352707403e-03,    4.4584625092275069e-04,
    1.1704935379148340e-04,    2.7812844524595717e-05,    6.0316970756601756e-06,    1.2024162496243299e-06,
    2.2170259218127070e-07,    3.8011878924379090e-08,    6.0889932404649297e-09,    9.1507136692467600e-10,
    1.2949484242607497e-10,    1.7313005567433181e-11,    2.1933309008309793e-12,    2.6400486769658187e-13,
    3.0265808358985320e-14,    3.3119805708678820e-15,    3.4665614316199482e-16,    3.4769133412050790e-17,
    3.3474729169555226e-18,    3.0985338456745626e-19,    2.7615225161349295e-20,    2.3729363423484028e-21,
    1.9684305438247083e-22,    1.5782119701167083e-23,    1.2243438007780668e-24,    9.1999565145624759e-26,
    6.7024973800916528e-27,    4.7386562756652388e-28,    3.2540017285533032e-29,    2.1720996879991155e-30,
    1.4105135109245753e-31,    8.9172027576431237e-33,    5.4920680186213759e-34,    3.2975080461485888e-35,
    1.9313068080648868e-36,    1.1040578643214762e-37,    6.1638933866829377e-39,    3.3626166995226211e-40,
    1.7934293017644013e-41,    9.3560302572458917e-43,    4.7764584673934372e-44,    2.3874007926781718e-45,
    1.1687977804648648e-46,    5.6069963527128773e-48,    2.6367743334047371e-49,    1.2160057043159076e-50,
    5.5014858304005664e-52,    2.4426534922530315e-53,    1.0647097814968717e-54,    4.5575640354056774e-56,
    1.9164875262204670e-57,    7.9192777814557898e-59,    3.2166336418628450e-60,    1.2846350826530219e-61
},
// x = 4.7700
{
    1.0481426448137764e-01,    8.2855701036153251e-02,    5.2703760685054842e-02,    2.7610668871525548e-02,
    1.2184959406086538e-02,    4.6201794260792529e-03,    1.5304575849394086e-03,    4.4912101848757548e-04,
    1.1812733812313308e-04,    2.8122119516241748e-05,    6.1105098613735181e-06,    1.2205037743457429e-06,
    2.2548118004184459e-07,    3.8736583350750825e-08,    6.2175006979718999e-09,    9.3626464190699337e-10,
    1.3276193505433483e-10,    1.7785845933607389e-11,    2.2578244303813292e-12,    2.7232309836437319e-13,
    3.1283374571989643e-14,    3.4303609531415745e-15,    3.5978526693978218e-16,    3.6160321634198837e-17,
    3.4885967495629723e-18,    3.2358353596284466e-19,    2.8898566312433232e-20,    2.4883547135865066e-21,
    2.0684529623661734e-22,    1.6618469249571038e-23,    1.2919030029293919e-24,    9.7277807497050354e-26,
    7.1017718362736891e-27,    5.0313886858293483e-28,    3.4622109846721969e-29,    2.3158959573971691e-30,
    1.5070251439514791e-31,    9.5472045993060014e-33,    5.8923453184613809e-34,    3.5452198435431045e-35,
    2.0807212802987776e-36,    1.1919556243292463e-37,    6.6685167908662922e-39,    3.6455043635545541e-40,
    1.9483674949185031e-41,    1.0185557522126281e-42,    5.2108187895908364e-44,    2.6099504685030431e-45,
    1.2804229679113105e-46,    6.1553345953477839e-48,    2.9006934051664197e-49,    1.3405163024758983e-50,
    6.0774899668927178e-52,    2.7040456127211476e-53,    1.1811129132490438e-54,    5.0664189235430779e-56,
    2.1349247515990736e-57,    8.8403715368815799e-59,    3.5982804431382308e-60,    1.4400638617998235e-61
},
// x = 4.7800
{
    1.0459513673654161e-01,    8.2728056414809500e-02,    5.2673762417623532e-02,    2.7629978572107080e-02,
    1.2211450701148714e-02,    4.6377073774756094e-03,    1.5389023429412546e-03,    4.5240811843035727e-04,
    1.1921159472883577e-04,    2.8433827553744926e-05,    6.1901044524437859e-06,    1.2388079927994116e-06,
    2.2931285529180628e-07,    3.9472975582852060e-08,    6.3483488614703319e-09,    9.5788834799020503e-10,
    1.3610225316565983e-10,    1.8270282202176555e-11,    2.3240362041160028e-12,    2.8088062219913385e-13,
    3.2332382825999145e-14,    3.5526523526976647e-15,    3.7337630591553189e-16,    3.7603447635129777e-17,
    3.6352936035863658e-18,    3.3788558999755474e-19,    3.0238145453460745e-20,    2.6090818315673582e-21,
    2.1732939986144811e-22,    1.7496937822056606e-23,    1.3630125180717004e-24,    1.0284501821272520e-25,
    7.5237843056570222e-27,    5.3414376341426575e-28,    3.6831973281323915e-29,    2.4688352947367490e-30,
    1.6098877791197348e-31,    1.0220067210792113e-32,    6.3207481377093403e-34,    3.8108913645271209e-35,
    2.2413037340246998e-36,    1.2866208543842462e-37,    7.2131291315538392e-39,    3.9514481455929678e-40,
    2.1162841457857881e-41,    1.1086451556665307e-42,    5.6835356256559091e-44,    2.8526594615110033e-45,
    1.4024143573370117e-46,    6.7558493417557739e-48,    3.1903304880121476e-49,    1.4774461228012638e-50,
    6.7122736077530143e-52,    2.9927152404543958e-53,    1.3099324727481491e-54,    5.6307314680324140e-56,
    2.3776757782792506e-57,    9.8661344433875931e-59,    4.0241862508562694e-60,    1.6138808446514706e-61
},
// x = 4.7900
{
    1.0437692098666740e-01,    8.2600757304216246e-02,    5.2643670065446367e-02,    2.7649118405002886e-02,
    1.2237860287780362e-02,    4.6552264237871722e-03,    1.5473611934883195e-03,    4.5570752705478238e-04,
    1.2030213173012818e-04,    2.8747978116957909e-05,    6.2704857547210463e-06,    1.2573307580555994e-06,
    2.3319818994468287e-07,    4.0221207196085986e-08,    6.4815731817764180e-09,    9.7994993689680800e-10,
    1.3951722273653293e-10,    1.8766565225495430e-11,    2.3920071013888920e-12,    2.8968364900503518e-13,
    3.3413717005515620e-14,    3.6789731749433260e-15,    3.8744424506402969e-16,    3.9100308997328974e-17,
    3.7877684722840531e-18,    3.5278182782611565e-19,    3.1636276223114239e-20,    2.7353476956239100e-21,
    2.2831729632612590e-22,    1.8419534749343932e-23,    1.4378495179230620e-24,    1.0871625387278929e-25,
    7.9697697005716492e-27,    5.6697818653518424e-28,    3.9177112292450050e-29,    2.6314750155491807e-30,
    1.7195026656453271e-31,    1.0938594002019655e-32,    6.7791788990346782e-34,    4.0957776014016170e-35,
    2.4138596096723627e-36,    1.3885568343969215e-37,    7.8007943290562367e-39,    4.2822666103235302e-40,
    2.2982322831874759e-41,    1.2064664526337596e-42,    6.1978950941124084e-44,    3.1173007497222453e-45,
    1.5357072742609577e-46,    7.4133676071548707e-48,    3.5081237216425600e-49,    1.6280010372014842e-50,
    7.4116792260085900e-52,    3.3114365106655764e-53,    1.4524598843313729e-54,    6.2563980952304690e-56,
    2.6473820963376795e-57,    1.1008181114437672e-58,    4.4993653140207648e-60,    1.8082121944368638e-61
},
// x = 4.8000
{
    1.0415961158994887e-01,    8.2473802662145088e-02,    5.2613484926108171e-02,    2.7668089197449102e-02,
    1.2264188179828226e-02,    4.6727363602711769e-03,    1.5558340208734521e-03,    4.5901922040557559e-04,
    1.2139895710602899e-04,    2.9064580655056228e-05,    6.3516586797647194e-06,    1.2760739310855788e-06,
    2.3713775997965122e-07,    4.0981431191562857e-08,    6.6172095271100410e-09,    1.0024569652730239e-09,
    1.4300829305509863e-10,    1.9274950519220430e-11,    2.4617788524496455e-12,    2.9873853158774733e-13,
    3.4528283299198254e-14,    3.8094450737621263e-15,    4.0200451341252747e-16,    4.0652760519684297e-17,
    3.9462333239518805e-18,    3.6829533767549857e-19,    3.3095361149708411e-20,    2.8673916474682809e-21,
    2.3983185580101548e-22,    1.9388359831223751e-23,    1.5165995422234937e-24,    1.1490731546684699e-25,
    8.4410267211272754e-27,    6.0174528491511856e-28,    4.1665452520407491e-29,    2.8044049342612790e-30,
    1.8362953445940995e-31,    1.1705764357750424e-32,    7.2696636955969312e-34,    4.4012179397014138e-35,
    2.5992503171685190e-36,    1.4983029479537737e-37,    8.4348029984520817e-39,    4.6399169778841608e-40,
    2.4953476037040671e-41,    1.3126629349536355e-42,    6.7574561877995937e-44,    3.4057985674640959e-45,
    1.6813189693566128e-46,    8.1331502226089727e-48,    3.8567359435123692e-49,    1.7935008135026534e-50,
    8.1821145379266431e-52,    3.6632583312001459e-53,    1.6101174626309710e-54,    6.9499264756527258e-56,
    2.9469651362777761e-57,    1.2279383998793711e-58,    5.0293865667835194e-60,    2.0254242258880766e-61
},
// x = 4.8100
{
    1.0394320295034790e-01,    8.2347191446669132e-02,    5.2583208285065700e-02,    2.7686891774043692e-02,
    1.2290434393526024e-02,    4.6902369836623516e-03,    1.5643207094749106e-03,    4.6234317427070484e-04,
    1.2250207869308606e-04,    2.9383644586201316e-05,    6.4336281446816711e-06,    1.2950393807303967e-06,
    2.4113214534713431e-07,    4.1753802003087959e-08,    6.7552941863495870e-09,    1.0254170957827461e-09,
    1.4657693702211954e-10,    1.9795698333692782e-11,    2.5333940534610585e-12,    3.0805176860772751e-13,
    3.5677010695700662e-14,    3.9441930310692719e-15,    4.1707299591296676e-16,    4.2262715875197405e-17,
    4.1109073195617887e-18,    3.8445004182367802e-19,    3.4617894820088589e-20,    3.0054627249888309e-21,
    2.5189692520590122e-22,    2.0405607164223072e-23,    1.5994568713564374e-24,    1.2143478319547481e-25,
    8.9389209791160502e-27,    6.3855374796070098e-28,    4.4305363031437696e-29,    2.9882491729256602e-30,
    1.9607170550801364e-31,    1.2524744217758985e-32,    7.7943600033719367e-34,    4.7286416103294818e-35,
    2.7983969774654096e-36,    1.6164371770270564e-37,    9.1186886245445837e-39,    5.0265053299880061e-40,
    2.7088547435305098e-41,    1.4279297260034231e-42,    7.3660727132238701e-44,    3.7202409102426732e-45,
    1.8403555779917900e-46,    8.9209296771100376e-48,    4.2390748028219779e-49,    1.9753895716695937e-50,
    9.0306056916642472e-52,    4.0515308697073072e-53,    1.7844713313262332e-54,    7.7184972632496592e-56,
    3.2796551916965411e-57,    1.3694006249464479e-58,    5.6204335085706211e-60,    2.2681498887288739e-61
},
// x = 4.8200
{
    1.0372768951714237e-01,    8.2220922616230568e-02,    5.2552841415754038e-02,    2.7705526956734686e-02,
    1.2316598947467153e-02,    4.7077280921694609e-03,    1.5728211437609158e-03,    4.6567936418358844e-04,
    1.2361150418543310e-04,    2.9705179297206311e-05,    6.5163990719642298e-06,    1.3142289836692092e-06,
    2.4518192997422859e-07,    4.2538475504123222e-08,    6.8958638722935157e-09,    1.0488380982078579e-09,
    1.5022465145460099e-10,    2.0329073726149765e-11,    2.6068961817292847e-12,    3.1763000748098529e-13,
    3.6860851488947757e-14,    4.0833454380687569e-15,    4.3266604559574397e-16,    4.3932149311814418e-17,
    4.2820170365576511e-18,    4.0127072440260302e-19,    3.6206467152517017e-20,    3.1498200284531833e-21,
    2.6453736726285564e-22,    2.1473569120658851e-23,    1.6866249145419774e-24,    1.2831605280393801e-25,
    9.4648882664388854e-27,    6.7751809055057530e-28,    4.7105679947489210e-29,    3.1836680657713974e-30,
    2.0932462181097511e-31,    1.3398897264924113e-32,    8.3555648539253085e-34,    5.0795734794473645e-35,
    3.0122844042698468e-36,    1.7435787624593591e-37,    9.8562448481291798e-39,    5.4442975403126889e-40,
    2.9400740095908784e-41,    1.5530178205051679e-42,    8.0279169358926953e-44,    4.0628930403945301e-45,
    2.0140196525884046e-46,    9.7829511606049712e-48,    4.6583146218229479e-49,    2.1752471722858400e-50,
    9.9648553200121097e-52,    4.4799345238666369e-53,    1.9772455826831961e-54,    8.5700319067138396e-56,
    3.6490232532459184e-57,    1.5267848274235786e-58,    6.2793703888331165e-60,    2.5393181010176380e-61
},
// x = 4.8300
{
    1.0351306578447914e-01,    8.2094995129707388e-02,    5.2522385579691938e-02,    2.7723995564808892e-02,
    1.2342681862577611e-02,    4.7252094854717281e-03,    1.5813352082941733e-03,    4.6902776542530068e-04,
    1.2472724113485431e-04,    3.0029194143204815e-05,    6.5999763893280518e-06,    1.3336446243871990e-06,
    2.4928770176996243e-07,    4.3335609014722719e-08,    7.0389557249286389e-09,    1.0727278505548883e-09,
    1.5395295739209089e-10,    2.0875346633769974e-11,    2.6823296111490554e-12,    3.2748004732794218e-13,
    3.8080781793001111e-14,    4.2270341782433287e-15,    4.4880049601080014e-16,    4.5663097397365285e-17,
    4.4597966989640987e-18,    4.1878306004834017e-19,    3.7863766776656930e-20,    3.3007330995165902e-21,
    2.7777910100218080e-22,    2.2594640484642418e-23,    1.7783166142092313e-24,    1.3556937351134124e-25,
    1.0020437974345055e-26,    7.1875894976184633e-28,    5.0075731271821815e-29,    3.3913601643924121e-30,
    2.2343900021494379e-31,    1.4331796754541728e-32,    8.9557234938912713e-34,    5.4556401961438521e-35,
    3.2419653407536126e-36,    1.8803910407981165e-37,    1.0651543936974251e-38,    5.8957309786314213e-40,
    3.1904286023248167e-41,    1.6887384285483704e-42,    8.7475050597697630e-44,    4.4362120709630652e-45,
    2.2036183134332258e-46,    1.0726017070252775e-47,    5.1179201516014212e-49,    2.3948016179407973e-50,
    1.0993305889163703e-51,    4.9525116008618634e-53,    2.1903377937617266e-54,    9.5132671138737171e-56,
    4.0590160391944260e-57,    1.7018409340174210e-58,    7.0138153434242935e-60,    2.8421862314127828e-61
},
// x = 4.8400
{
    1.0329932629093236e-01,    8.1969407946478334e-02,    5.2491842026586279e-02,    2.7742298414880913e-02,
    1.2368683162089101e-02,    4.7426809647152270e-03,    1.5898627877363157e-03,    4.7238835302677504e-04,
    1.2584929695085597e-04,    3.0355698447322188e-05,    6.6843650295498565e-06,    1.3532881951430548e-06,
    2.5345005263038141e-07,    4.4145361308439837e-08,    7.1846073147046814e-09,    1.0970943401679759e-09,
    1.5776340040566124e-10,    2.1434791947557981e-11,    2.7597396278659739e-12,    3.3760884197099199e-13,
    3.9337802066658479e-14,    4.3753947121081718e-15,    4.6549367396188945e-16,    4.7457660809615812e-17,
    4.6444884139669349e-18,    4.3701364342165806e-19,    3.9592584523844212e-20,    3.4584823134464124e-21,
    2.9164914377134824e-22,    2.3771322750782853e-23,    1.8747548671770627e-24,    1.4321388760085778e-25,
    1.0607156670029720e-26,    7.6240339591313576e-28,    5.3225362967713938e-29,    3.6120643486194543e-30,
    2.3846859746988865e-31,    1.5327238019890442e-32,    9.5974385588400453e-34,    5.8585767190364652e-35,
    3.4885649668952298e-36,    2.0275844687009894e-37,    1.1508956519567498e-38,    6.3834270414083190e-40,
    3.4614523647587758e-41,    1.8359676459329271e-42,    9.5297246782730914e-44,    4.8428627110310400e-45,
    2.4105720670842629e-46,    1.1757535262991256e-47,    5.6216723815068063e-49,    2.6359425549358955e-50,
    1.2125208812648215e-51,    5.4737009522966573e-53,    2.4258360256835860e-54,    1.0557836604287117e-55,
    4.5139945350951549e-57,    1.8965065738504562e-58,    7.8322211942784060e-60,    3.1803760600153765e-61
},
// x = 4.8500
{
    1.0308646561906668e-01,    8.1844160026486590e-02,    5.2461211994435805e-02,    2.7760436320882667e-02,
    1.2394602871512370e-02,    4.7601423325092027e-03,    1.5984037668523253e-03,    4.7576110177101029e-04,
    1.2697767890074735e-04,    3.0684701500349732e-05,    6.7695699303050677e-06,    1.3731615959360343e-06,
    2.5766957844346448e-07,    4.4967892619207357e-08,    7.3328566458152478e-09,    1.1219456648481450e-09,
    1.6165755090958161e-10,    2.2007689587074396e-11,    2.8391724461581363e-12,    3.4802350298139628e-13,
    4.0632937647939886e-14,    4.5285661637602168e-15,    4.8276341254001404e-16,    4.9318006172461636e-17,
    4.8363424151279130e-18,    4.5599001962294862e-19,    4.1395817030926778e-20,    3.6233592849858626e-21,
    3.0617565479846475e-22,    2.5006228591535114e-23,    1.9761729632952477e-24,    1.5126967173921510e-25,
    1.1226711837401528e-26,    8.0858525857610823e-28,    5.6564966350138761e-29,    3.8465620483613650e-30,
    2.5447038443632503e-31,    1.6389251690902382e-32,    1.0283479790718833e-33,    6.2902332441552221e-35,
    3.7532856940224900e-36,    2.1859198468175541e-37,    1.2433172664617320e-38,    6.9102045650339345e-40,
    3.7547980948204584e-41,    1.9956514744846139e-42,    1.0379864344171977e-43,    5.2857342619460425e-45,
    2.6364243452930791e-46,    1.2885571359884702e-47,    6.1736965743459531e-49,    2.9007359700344134e-50,
    1.3370699839885166e-51,    6.0483758324690568e-53,    2.6860374437964301e-54,    1.1714360844095912e-55,
    5.0187763847897823e-57,    2.1129267163008302e-58,    8.7439646950348662e-60,    3.5579135810786898e-61
},
// x = 4.8600
{
    1.0287447839500548e-01,    8.1719250330302043e-02,    5.2430496709633831e-02,    2.7778410094053227e-02,
    1.2420441018610688e-02,    4.7775933929223293e-03,    1.6069580305148844e-03,    4.7914598619527084e-04,
    1.2811239410972754e-04,    3.1016212560421578e-05,    6.8555960340053328e-06,    1.3932667344725966e-06,
    2.6194687909386511e-07,    4.5803364648188449e-08,    7.4837421594847812e-09,    1.1472900339788450e-09,
    1.6563700447568922e-10,    2.2594324576016847e-11,    2.9206752245391870e-12,    3.5873130277613327e-13,
    4.1967239298610455e-14,    4.6866914092549069e-15,    5.0062806446208518e-16,    5.1246367939307894e-17,
    5.0356173123996010e-18,    4.7574071552591056e-19,    4.3276470461034450e-20,    3.7956672882920259e-21,
    3.2138798036337671e-22,    2.6302086499335860e-23,    2.0828150422213879e-24,    1.5975778009620454e-25,
    1.1880855789106641e-26,    8.5744546823511757e-28,    6.0105506852962189e-29,    4.0956795819507029e-30,
    2.7150472981410910e-31,    1.7522117664698293e-32,    1.1016794329627108e-33,    6.7525825577155388e-35,
    4.0374122640946663e-36,    2.3562117557762260e-37,    1.3429224394529582e-38,    7.4790941815449089e-40,
    4.0722464603517534e-41,    2.1688112176468155e-42,    1.1303645416360614e-43,    5.7679589605254369e-45,
    2.8828518214180385e-46,    1.4118905431822608e-47,    6.7784927134329244e-49,    3.1914401848834699e-50,
    1.4740881272489683e-51,    6.6818852714992313e-53,    2.9734687090139431e-54,    1.2994545521460879e-55,
    5.5786825074475574e-57,    2.3534753115731508e-58,    9.7594450831051006e-60,    3.9792730478499724e-61
},
// x = 4.8700
{
    1.0266335928800395e-01,    8.1594677819182088e-02,    5.2399697387070433e-02,    2.7796220542929087e-02,
    1.2446197633373591e-02,    4.7950339514789154e-03,    1.6155254637086921e-03,    4.8254298059328970e-04,
    1.2925344956098232e-04,    3.1350240852694542e-05,    6.9424482876360177e-06,    1.4136055261326414e-06,
    2.6628255846748589e-07,    4.6651940570599689e-08,    7.6373027372616209e-09,    1.1731357696577839e-09,
    1.6970338215047095e-10,    2.3194987118658263e-11,    3.0042960820850954e-12,    3.6973967776534861e-13,
    4.3341783758895635e-14,    4.8499171668432645e-15,    5.1910651572100891e-16,    5.3245050324705213e-17,
    5.2425803491103288e-18,    4.9629527205506758e-19,    4.5237664344737337e-20,    3.9757216913963063e-21,
    3.3731670063122619e-22,    2.7661745609879316e-23,    2.1949365690334035e-24,    1.6870028933770130e-25,
    1.2571429757183790e-26,    9.0913241430374148e-28,    6.3858554237035565e-29,    4.3602906167905604e-30,
    2.8963559388779713e-31,    1.8730379868751456e-32,    1.1800517612347308e-33,    7.2477278387140412e-35,
    4.3423171722884880e-36,    2.5393322176718378e-37,    1.4502509726632307e-38,    8.0933536805723136e-40,
    4.4157155589357696e-41,    2.3565492784152554e-42,    1.2307256352849928e-43,    6.2929317724559516e-45,
    3.1516755656592728e-46,    1.5467093422081349e-47,    7.4409685626543746e-49,    3.5105232592807908e-50,
    1.6247911608763718e-51,    7.3800992803988004e-53,    3.2909083041499362e-54,    1.4411289590938539e-55,
    6.1995883508048966e-57,    2.6207791329629933e-58,    1.0890192884075704e-59,    4.4494257004526292e-61
},
// x = 4.8800
{
    1.0245310301002714e-01,    8.1470441455131243e-02,    5.2368815230233318e-02,    2.7813868473334800e-02,
    1.2471872747990796e-02,    4.8124638151550566e-03,    1.6241059515347260e-03,    4.8595205901746810e-04,
    1.3040085209578680e-04,    3.1686795569030665e-05,    7.0301316425935863e-06,    1.4341798939353056e-06,
    2.7067722445587491e-07,    4.7513785042503841e-08,    7.7935777043166962e-09,    1.1994913078349739e-09,
    1.7385833077485427e-10,    2.3809972677147239e-11,    3.0900841149867445e-12,    3.8105623155102850e-13,
    4.4757674312541509e-14,    5.0183940891018976e-15,    5.3821819955344214e-16,    5.5316429285324832e-17,
    5.4575076660921112e-18,    5.1768427743273017e-19,    4.7282635545135154e-20,    4.1638504056476284e-21,
    3.5399367820492154e-22,    2.9088180713093818e-23,    2.3128048294020287e-24,    1.7812034556844765e-25,
    1.3300368170019518e-26,    9.6380232023693705e-28,    6.7836314307469627e-29,    4.6413187583711161e-30,
    3.0893073280789682e-31,    2.0018861859541088e-32,    1.2637984911792464e-33,    7.7779109376759028e-35,
    4.6694664325323661e-36,    2.7362145972590320e-37,    1.5658818341800587e-38,    8.7564844454038706e-40,
    4.7872711674841587e-41,    2.5600553885627638e-42,    1.3397389631464731e-43,    6.8643317467387304e-45,
    3.4448731051218410e-46,    1.6940533689168321e-47,    8.1664755569569633e-49,    3.8606819236895179e-50,
    1.7905103268026763e-51,    8.1494582329392444e-53,    3.6414109737835880e-54,    1.5978803791481051e-55,
    6.8879802294976427e-57,    2.9177440390182230e-58,    1.2148990007979947e-59,    4.9738936623763843e-61
},
// x = 4.8900
{
    1.0224370431533258e-01,    8.1346540200959327e-02,    5.2337851431308473e-02,    2.7831354688373982e-02,
    1.2497466396826272e-02,    4.8298827923747081e-03,    1.6326993792144520e-03,    4.8937319528107232e-04,
    1.3155460841361632e-04,    3.2025885867682465e-05,    7.1186510545228944e-06,    1.4549917685043631e-06,
    2.7513148896045225e-07,    4.8389064207573118e-08,    7.9526068327478527e-09,    1.2263651994570046e-09,
    1.7810352330672210e-10,    2.4439582049676712e-11,    3.1780894133305644e-12,    3.9268873817754947e-13,
    4.6216041362378956e-14,    5.1922768569896601e-15,    5.5798311073162636e-16,    5.7462954551384096e-17,
    5.6806845731289012e-18,    5.3993940142169150e-19,    4.9414742350520272e-20,    4.3603943506127316e-21,
    3.7145210845480155e-22,    3.0584497458602847e-23,    2.4366994450739743e-24,    1.8804221330376841e-25,
    1.4069703123583054e-26,    1.0216196365090618e-27,    7.2051662211440643e-29,    4.9397402740100725e-30,
    3.2946191395292609e-31,    2.1392683301762839e-32,    1.3532743553367249e-33,    8.3455211593549992e-35,
    5.0204257067757954e-36,    2.9478577579144909e-37,    1.6904358986374726e-38,    9.4722490354480005e-40,
    5.1891377295466758e-41,    2.7806133001081137e-42,    1.4581281492723789e-43,    7.4861450502145490e-45,
    3.7645914597351886e-46,    1.8550539083401573e-47,    8.9608477581834807e-49,    4.2448621713795091e-50,
    1.9727029101902076e-51,    8.9970267993094038e-53,    4.0283344721983801e-54,    1.7712740625899224e-55,
    7.6510172396627307e-57,    3.2475838948524638e-58,    1.3550002279638391e-59,    5.5588095402110850e-61
},
// x = 4.9000
{
    1.0203515800005793e-01,    8.1222973020338401e-02,    5.2306807171279328e-02,    2.7848679988420711e-02,
    1.2522978616392601e-02,    4.8472906930057402e-03,    1.6413056320940006e-03,    4.9280636296043345e-04,
    1.3271472507226569e-04,    3.2367520872981286e-05,    7.2080114831545459e-06,    1.4760430880332231e-06,
    2.7964596789656259e-07,    4.9277945703822121e-08,    8.1144303448897375e-09,    1.2537661116175343e-09,
    1.8244065914615857e-10,    2.5084121449526893e-11,    3.2683630781094997e-12,    4.0464514543475519e-13,
    4.7718043016550305e-14,    5.3717242758647932e-15,    5.7842182018578247e-16,    5.9687151709650943e-17,
    5.9124058279060403e-18,    5.6309343059054282e-19,    5.1637468698349931e-20,    4.5657079349210098e-21,
    3.8972657168554109e-22,    3.2153937762677646e-23,    2.5669129104428502e-24,    1.9849132655236972e-25,
    1.4881569055239006e-26,    1.0827574522605219e-27,    7.6518177391050974e-29,    5.2565869579679583e-30,
    3.5130514294387897e-31,    2.2857277375471844e-32,    1.4488565846162849e-33,    8.9531045787365363e-35,
    5.3968668199852483e-36,    3.1753304883373913e-37,    1.8245788719873084e-38,    1.0244689992061378e-39,
    5.6237101315131131e-41,    3.0196079721229969e-42,    1.5866754713262204e-43,    8.1626898099676771e-45,
    4.1131612304460857e-46,    2.0309415002978268e-47,    9.8304441300670791e-49,    4.6662816513471981e-50,
    2.1729638459703824e-51,    9.9305528392348957e-53,    4.4553688313337632e-54,    1.9630336880205814e-55,
    8.4965992871397115e-57,    3.6138524148922929e-58,    1.5108925657801694e-59,    6.2109823151546564e-61
},
// x = 4.9100
{
    1.0182745890181302e-01,    8.1099738877858471e-02,    5.2275683620025770e-02,    2.7865845171111250e-02,
    1.2548409445325422e-02,    4.8646873283558923e-03,    1.6499245956482678e-03,    4.9625153539713653e-04,
    1.3388120848797369e-04,    3.2711709675028066e-05,    7.2982178921420347e-06,    1.4973357982494951e-06,
    2.8422128119735219e-07,    5.0180598670309339e-08,    8.2790889166287368e-09,    1.2817028287139259e-09,
    1.8687146446342679e-10,    2.5743902584987312e-11,    3.3609572384663787e-12,    4.1693357821420082e-13,
    4.9264865685557472e-14,    5.5568993734968730e-15,    5.9955548996365111e-16,    6.1991624339177396e-17,
    6.1529759226455504e-18,    5.8718030462913610e-19,    5.3954428534360401e-20,    4.7801595535553951e-21,
    4.0885308720220264e-22,    3.3799885423921072e-23,    2.7037511510125192e-24,    2.0949434209546385e-25,
    1.5738207628767077e-26,    1.1473979264496291e-27,    8.1250180269075723e-29,    5.5929491448995906e-30,
    3.7454090291063926e-31,    2.4418409160955931e-32,    1.5509462768929443e-33,    9.6033739213231334e-35,
    5.8005746841291385e-36,    3.4197762169163346e-37,    1.9690244128007447e-38,    1.1078149949665750e-39,
    6.0935663222927606e-41,    3.2785332882537884e-42,    1.7262264632992690e-43,    8.8986429007936264e-45,
    4.4931118218864426e-46,    2.2230543910026565e-47,    1.0782194406544912e-48,    5.1284540148022258e-50,
    2.3930383640041269e-51,    1.0958531697678917e-52,    4.9265683795955421e-54,    2.1750569859903406e-55,
    9.4334418169727942e-57,    4.0204782145223983e-58,    1.6843147521154341e-59,    6.9379701740124671e-61
},
// x = 4.9200
{
    1.0162060189927656e-01,    8.0976836739081848e-02,    5.2244481936421783e-02,    2.7882851031336132e-02,
    1.2573758924358178e-02,    4.8820725111687292e-03,    1.6585561554850062e-03,    4.9970868570021890e-04,
    1.3505406493555827e-04,    3.3058461329387534e-05,    7.3892752488990913e-06,    1.5188718523791971e-06,
    2.8885805281747667e-07,    5.1097193753808378e-08,    8.4466236807232956e-09,    1.3101842536100815e-09,
    1.9139769252969650e-10,    2.6419242740165144e-11,    3.4559250691721561e-12,    4.2956234191925001e-13,
    5.0857724690296979e-14,    5.7479695001086982e-15,    6.2140588853391564e-16,    6.4379056200937703e-17,
    6.4027093786163407e-18,    6.1223515374242066e-19,    5.6369370310761343e-20,    5.0041321021053918e-21,
    4.2886916933928303e-22,    3.5525871955164536e-23,    2.8475341045862304e-24,    2.2107919505069195e-25,
    1.6641972839563803e-26,    1.2157327393816322e-27,    8.6262770748888507e-29,    5.9499788789277031e-30,
    3.9925440663918046e-31,    2.6082195053679238e-32,    1.6599698452897916e-33,    1.0299219040404363e-34,
    6.2334546557567378e-36,    3.6824180317052688e-37,    2.1245374627934465e-38,    1.1977293139352965e-39,
    6.6014808346912769e-41,    3.5590003427692075e-42,    1.8776948675059839e-43,    9.6990688249693076e-45,
    4.9071878879249994e-46,    2.4328476824496487e-47,    1.1823648849484991e-48,    5.6352153805748890e-50,
    2.6348357631062681e-51,    1.2090276384681576e-52,    5.4463867629160690e-54,    2.4094328627696954e-55,
    1.0471157886875759e-56,    4.4718033856277965e-58,    1.8771924534364278e-59,    7.7481609923903530e-61
},
// x = 4.9300
{
    1.0141458191179760e-01,    8.0854265570596412e-02,    5.2213203268432637e-02,    2.7899698361232673e-02,
    1.2599027096296993e-02,    4.8994460556194986e-03,    1.6672001973488258e-03,    5.0317778674835521e-04,
    1.3623330054855556e-04,    3.3407784856784490e-05,    7.4811885244367953e-06,    1.5406532111105140e-06,
    2.9355691073662745e-07,    5.2027903115445859e-08,    8.6170762301288840e-09,    1.3392194088053660e-09,
    1.9602112405051518e-10,    2.7110464856683399e-11,    3.5533208083409723e-12,    4.4253992592965317e-13,
    5.2497864881241676e-14,    5.9451064304830431e-15,    6.4399540644027537e-16,    6.6852213482561880e-17,
    6.6619310487116966e-18,    6.3829433715146840e-19,    5.8886181627542894e-20,    5.2380235085123671e-21,
    4.4981388551849504e-22,    3.7335582639301444e-23,    2.9985963260437695e-24,    2.3327515681259448e-25,
    1.7595336349354802e-26,    1.2879635655232077e-27,    9.1571868613432224e-29,    6.3288932459617593e-30,
    4.2553586225888323e-31,    2.7855123264292786e-32,    1.7763805505752037e-33,    1.1043718025815854e-34,
    6.6975403531899517e-36,    3.9645640250221479e-37,    2.2919378000537368e-38,    1.2947128377771279e-39,
    7.1504392705696983e-41,    3.8627463355394233e-42,    2.0420679614577095e-43,    1.0569450842323495e-44,
    5.3583670951701118e-46,    2.6619032353905051e-47,    1.2963032215544814e-48,    6.1907530983626330e-50,
    2.9004444117815565e-51,    1.3335994162561804e-52,    6.0197152417728983e-54,    2.6684601644039751e-55,
    1.1620348287275942e-56,    4.9726259411309904e-58,    2.0916578755378637e-59,    8.6508612541149138e-61
},
// x = 4.9400
{
    1.0120939389900109e-01,    8.0732024340067488e-02,    5.2181848753211121e-02,    2.7916387950177708e-02,
    1.2624214005995749e-02,    4.9168077773109626e-03,    1.6758566071251864e-03,    5.0665881119204919e-04,
    1.3741892131936904e-04,    3.3759689242803734e-05,    7.5739626932008492e-06,    1.5626818425571964e-06,
    2.9831848696289006e-07,    5.2972900437308018e-08,    8.7904886213279272e-09,    1.3688174376096741e-09,
    2.0074356750204165e-10,    2.7817897616276777e-11,    3.6531997753845393e-12,    4.5587500712130958e-13,
    5.4186561268939212e-14,    6.1484864681705248e-15,    6.6734707231314343e-16,    6.9413947099386561e-17,
    6.9309764282913676e-18,    6.6539548273128274e-19,    6.1508894021042285e-20,    5.4822472828530474e-21,
    4.7172791640312082e-22,    3.9232862817040909e-23,    3.1572876165997631e-24,    2.4611289546485467e-25,
    1.8600893060119236e-26,    1.3643025685490210e-27,    9.7194255911863855e-29,    6.7309778772392996e-30,
    4.5348075316140100e-31,    2.9744075461506160e-32,    1.9006601223402438e-33,    1.1840148980600315e-34,
    7.1950019608412734e-36,    4.2676129828179696e-37,    2.4721038292869425e-38,    1.3993033640030498e-39,
    7.7436538159849617e-41,    4.1916441191194222e-42,    2.2204122870237102e-43,    1.1515724520128784e-44,
    5.8498793066357919e-46,    2.9119403860132099e-47,    1.4209302277335210e-48,    6.7996370033784102e-50,
    3.1921480817477607e-51,    1.4706870108875769e-52,    6.6519245621184104e-54,    2.9546682339201587e-55,
    1.2892700475949887e-56,    5.5282465065203895e-58,    2.3300713807282801e-59,    9.6563942692031830e-61
},
// x = 4.9500
{
    1.0100503286039837e-01,    8.0610112016288829e-02,    5.2150419517193007e-02,    2.7932920584780745e-02,
    1.2649319700331365e-02,    4.9341574932691718e-03,    1.6845252708443175e-03,    5.1015173145581478e-04,
    1.3861093309942366e-04,    3.4114183437592047e-05,    7.6676027329087556e-06,    1.5849597222215454e-06,
    3.0314341753592141e-07,    5.3932360929013797e-08,    8.9669033776642019e-09,    1.3989876053245521e-09,
    2.0556685947003928e-10,    2.8541875524289027e-11,    3.7556183892079321e-12,    4.6957645344186175e-13,
    5.5925119665991101e-14,    6.3582905518346098e-15,    6.9148456924597077e-16,    7.2067195053060306e-17,
    7.2101919744890068e-18,    6.9357752781560204e-19,    6.4241687904017062e-20,    5.7372330857218522e-21,
    4.9465361821876861e-22,    4.1221724414821885e-23,    3.3239736784670691e-24,    2.5962453876305262e-25,
    1.9661366937308611e-26,    1.4449729196062245e-27,    1.0314762142638060e-28,    7.1575906324344900e-30,
    4.8319013287587595e-31,    3.1756349618532591e-32,    2.0333204738673904e-33,    1.2692002503983982e-34,
    7.7281550497447844e-36,    4.5930604401608762e-37,    2.6659766242710677e-38,    1.5120782321666909e-39,
    8.3845798568843132e-41,    4.5477124440573533e-42,    2.4138798112209470e-43,    1.2544313884664460e-44,
    6.3852272954438478e-46,    3.1848275410861884e-47,    1.5572213271471670e-48,    7.4668533717658722e-50,
    3.5124437292183178e-51,    1.6215158272471406e-52,    7.3489107244979324e-54,    3.2708394284045270e-55,
    1.4301097166570316e-56,    6.1445196723131400e-58,    2.5950453115546134e-59,    1.0776208638601688e-60
},
// x = 4.9600
{
    1.0080149383500150e-01,    8.0488527569232068e-02,    5.2118916676191791e-02,    2.7949297048877428e-02,
    1.2674344228179292e-02,    4.9514950219392007e-03,    1.6932060746850969e-03,    5.1365651974036037e-04,
    1.3980934159932897e-04,    3.4471276355563761e-05,    7.7621136243871102e-06,    1.6074888329570214e-06,
    3.0803234252995692e-07,    5.4906461334255004e-08,    9.1463634926817211e-09,    1.4297393004304125e-09,
    2.1049286499164272e-10,    2.9282738994079098e-11,    3.8606341866491881e-12,    4.8365332754282486e-13,
    5.7714877340685574e-14,    6.5747043637708192e-15,    7.1643225154338642e-16,    7.4814984848970390e-17,
    7.4999354341906672e-18,    7.2288076119969091e-19,    6.7088897661587775e-20,    6.0034273157891823e-21,
    5.1863508731256472e-22,    4.3306352721402490e-23,    3.4990367958819028e-24,    2.7384373979037672e-25,
    2.0779617092842185e-26,    1.5302093398240411e-27,    1.0945060731580829e-28,    7.6101654710638555e-30,
    5.1477093566037397e-31,    3.3899684126890496e-32,    2.1749055158608525e-33,    1.3602994921192899e-34,
    8.2994699450487181e-36,    4.9425051254463152e-37,    2.8745642386539715e-38,    1.6336571301404163e-39,
    9.0769337705798559e-41,    4.9331269516916687e-42,    2.6237145500384924e-43,    1.3662170369496759e-44,
    6.9682091066522454e-46,    3.4825947213135439e-47,    1.7062384675615149e-48,    8.1978418032003591e-50,
    3.8640608485383241e-51,    1.7874281093017899e-52,    8.1171450042195334e-54,    3.6200337777393945e-55,
    1.5859735488232266e-56,    6.8279104606695003e-58,    2.8894702405812203e-59,    1.2022998008149157e-60
},
// x = 4.9700
{
    1.0059877190094242e-01,    8.0367269970095342e-02,    5.2087341335492511e-02,    2.7965518123523374e-02,
    1.2699287640389165e-02,    4.9688201831808217e-03,    1.7018989049788927e-03,    5.1717314802476866e-04,
    1.4101415238904763e-04,    3.4830976875108674e-05,    7.8575003514088389e-06,    1.6302711649304541e-06,
    3.1298590605663863e-07,    5.5895379937302165e-08,    9.3289124334678059e-09,    1.4610820357797589e-09,
    2.1552347789989895e-10,    3.0040834432340252e-11,    3.9683058411648787e-12,    4.9811489046891437e-13,
    5.9557203682452120e-14,    6.7979184406372191e-15,    7.4221516184838535e-16,    7.7660435973775149e-17,
    7.8005761808936688e-18,    7.5334686637276793e-19,    7.0055016907519383e-20,    6.2812937171283199e-21,
    5.4371822702492721e-22,    4.5491113421911707e-23,    3.6828765434803337e-24,    2.8880574539250099e-25,
    2.1958644138770523e-26,    1.6202586681380134e-27,    1.1612285803673876e-28,    8.0902165213193028e-30,
    5.4833630360591078e-31,    3.6182283244549491e-32,    2.3259930744799919e-33,    1.4577082302849245e-34,
    8.9115816729681348e-36,    5.3176558172809326e-37,    3.0989463022113185e-38,    1.7647050923544349e-39,
    9.8247119731745052e-41,    5.3502319670482066e-42,    2.8512596889058594e-43,    1.4876814769639364e-44,
    7.6029421941078158e-46,    3.8074471280046835e-47,    1.8691375748380290e-48,    8.9985352754904475e-50,
    4.2499825331678742e-51,    1.9698937811841279e-52,    8.9637286064667194e-54,    4.0056159831785877e-55,
    1.7584257717768388e-56,    7.5855564022717004e-58,    3.2165438870085682e-59,    1.3410833257567562e-60
},
// x = 4.9800
{
    1.0039686217509594e-01,    8.0246338191350658e-02,    5.2055694589944961e-02,    2.7981584586988250e-02,
    1.2724149989760670e-02,    4.9861327982641517e-03,    1.7106036482133463e-03,    5.2070158806867095e-04,
    1.4222537089807246e-04,    3.5193293838303082e-05,    7.9537679005305484e-06,    1.6533087155838766e-06,
    3.1800475626766856e-07,    5.6899296569476616e-08,    9.5145941440002539e-09,    1.4930254497964269e-09,
    2.2066062117109154e-10,    3.0816514325338608e-11,    4.0786931817640009e-12,    5.1297060540527556e-13,
    6.1453500879312928e-14,    7.0281282864342197e-15,    7.6885904865598534e-16,    8.0606762434355712e-17,
    8.1124955606601885e-18,    7.8501896601252323e-19,    7.3144703905432080e-20,    6.5713140069207114e-21,
    5.6995081695036679e-22,    4.7780559898441280e-23,    3.8759105230504684e-24,    3.0454746750179602e-25,
    2.3201596822928321e-26,    1.7153804555429681e-27,    1.2318507164741613e-28,    8.5993423558776855e-30,
    5.8400593108775219e-31,    3.8612843948778423e-32,    2.4871969194029755e-33,    1.5618475319024720e-34,
    9.5673005215433278e-36,    5.7203386394016013e-37,    3.3402789207327876e-38,    1.9059357026360462e-39,
    1.0632211308371222e-40,    5.8015531480162054e-42,    3.0979652357677327e-43,    1.6196382425862469e-44,
    8.2938894686661879e-46,    4.1617798139174364e-47,    2.0471766300200340e-48,    9.8754036358410960e-50,
    4.6734683902496462e-51,    2.1705222664131850e-52,    9.8964523739249989e-54,    4.4312849717698205e-55,
    1.9491894679023790e-56,    8.4253357664279726e-58,    3.5798029641932379e-59,    1.4955308383620069e-60
},
// x = 4.9900
{
    1.0019575981270733e-01,    8.0125731206790202e-02,    5.2023977524055909e-02,    2.7997497214750207e-02,
    1.2748931331019562e-02,    5.0034326898652191e-03,    1.7193201910361137e-03,    5.2424181141442117e-04,
    1.4344300241560885e-04,    3.5558236050623588e-05,    8.0509212609298938e-06,    1.6766034895959709e-06,
    3.2308954535728465e-07,    5.7918392615586782e-08,    9.7034530484983055e-09,    1.5255793076808081e-09,
    2.2590624727485186e-10,    3.1610137326076300e-11,    4.1918572121924103e-12,    5.2823014148329949e-13,
    6.3405204607504456e-14,    7.2655344877717499e-15,    7.9639038422082358e-16,    8.3657275359520471e-17,
    8.4360872473838859e-18,    8.1794166777488201e-19,    7.6362787159642887e-20,    6.8739885241660813e-21,
    5.9738258466603658e-22,    5.0179440806546117e-23,    4.0785751297199060e-24,    3.2110755746508276e-25,
    2.4511778958337846e-26,    1.8158475869346445e-27,    1.3065905360413574e-28,    9.1392304846720985e-30,
    6.2190642743861003e-31,    4.1200584267573485e-32,    2.6591689079469490e-33,    1.6731654975481412e-34,
    1.0269623251495080e-35,    6.1525048204830979e-37,    3.5997998988126954e-38,    2.0581145150851271e-39,
    1.1504050868686946e-40,    6.2898110507813720e-42,    3.3653962452387063e-43,    1.7629671879572601e-44,
    9.0458874042428985e-46,    4.5481935453145259e-47,    2.2417244200052893e-48,    1.0835500814840130e-49,
    5.1380794671466136e-51,    2.3910753709977964e-52,    1.0923862001037930e-53,    4.9011062420171016e-55,
    2.1601623003892182e-56,    9.3559425385101729e-58,    3.9831582475962621e-59,    1.6673701460820682e-60
},
// x = 5.0000
{
    9.9995460007023751e-02,    8.0005447991571485e-02,    5.1992191212080846e-02,    2.8013256779490642e-02,
    1.2773631720793951e-02,    5.0207196820615382e-03,    1.7280484202585667e-03,    5.2779378938926709e-04,
    1.4466705209076564e-04,    3.5925812280663839e-05,    8.1489654242430199e-06,    1.7001574988431434e-06,
    3.2824092956456261e-07,    5.8952851020330558e-08,    9.8955340547775050e-09,    1.5587535026210391e-09,
    2.3126233852706858e-10,    3.2422068342385482e-11,    4.3078601303702357e-12,    5.4390337764574180e-13,
    6.5413784733448636e-14,    7.5103428314630522e-15,    8.2483638286635891e-16,    8.6815385665826132e-17,
    8.7717576075931747e-18,    8.5216111141301047e-19,    7.9714271180468717e-20,    7.1898369000417802e-21,
    6.2606528000916223e-22,    5.2692707937329455e-23,    4.2913263486750853e-24,    3.3852648349337286e-25,
    2.5892656658586534e-26,    1.9219469317478980e-27,    1.3856777316469807e-28,    9.7116620750660511e-30,
    6.6217169876019204e-31,    4.3955273167240441e-32,    2.8426012515856029e-33,    1.7921389282215709e-34,
    1.1021744995520408e-35,    6.6162389472653307e-37,    3.8788343059956660e-38,    2.2220627072699022e-39,
    1.2445195346035730e-40,    6.8179356755416531e-42,    3.6552416549923073e-43,    1.9186197255964732e-44,
    9.8641763590080494e-46,    4.9695119489118573e-47,    2.4542700162572012e-48,    1.1886516072309408e-49,
    5.6477053614641689e-51,    2.6334813234651898e-52,    1.2055329248660265e-53,    5.4195472572525787e-55,
    2.3934337559548245e-56,    1.0386968794670793e-57,    4.4309331805433853e-59,    1.8585152199271499e-60
},
// x = 5.0100
{
    9.9795957988929998e-02,    7.9885487522261628e-02,    5.1960336718114661e-02,    2.8028864051089297e-02,
    1.2798251217590680e-02,    5.0379936003275960e-03,    1.7367882228594325e-03,    5.3135749310751163e-04,
    1.4589752493275086e-04,    3.6296031259854066e-05,    8.2479053844021013e-06,    1.7239727623602205e-06,
    3.3345956917553831e-07,    6.0002856294660387e-08,    1.0090882557607862e-08,    1.5925580570100967e-09,
    2.3673090744559186e-10,    3.3252678625957519e-11,    4.4267653480843885e-12,    5.6000040657182705e-13,
    6.7480746028248501e-14,    7.7627644244839729e-15,    8.5422501970338553e-16,    9.0084606788896150e-17,
    9.1199260750182839e-18,    8.8772501726024774e-19,    8.3204342428933455e-20,    7.5193987505727718e-21,
    6.5605275198684860e-22,    5.5325524375093961e-23,    4.5146405835463197e-24,    3.5684661135632956e-25,
    2.7347865891872647e-26,    2.0339800246487273e-27,    1.4693542251839557e-28,    1.0318516910345045e-29,
    7.0494334983315112e-31,    4.6887262077564025e-32,    3.0382289115343941e-33,    1.9192750907133912e-34,
    1.1827071886532346e-35,    7.1137677410949162e-37,    4.1788004079695481e-38,    2.3986609809255740e-39,
    1.3460980014957658e-40,    7.3890820608395751e-42,    3.9693237783980283e-43,    2.0876244651618969e-44,
    1.0754433280657779e-45,    5.4288000445230285e-47,    2.6864330402692151e-48,    1.3036829609039333e-49,
    6.2065937002516169e-51,    2.8998500727555355e-52,    1.3301129695883580e-53,    5.9915161660940095e-55,
    2.6513040465712852e-56,    1.1528995184721970e-57,    4.9279063656442009e-59,    2.0710857768443425e-60
},
// x = 5.0200
{
    9.9597249026588244e-02,    7.9765848776880213e-02,    5.1928415096181728e-02,    2.8044319796619557e-02,
    1.2822789881771995e-02,    5.0552542715303246e-03,    1.7455394859884148e-03,    5.3493289347268202e-04,
    1.4713442581107666e-04,    3.6668901682183713e-05,    8.3477461374729617e-06,    1.7480513063007825e-06,
    3.3874612852515680e-07,    6.1068594522113609e-08,    1.0289544442075614e-08,    1.6270031236688284e-09,
    2.4231399710874847e-10,    3.4102345862314444e-11,    4.5486375109386478e-12,    5.7653153866306809e-13,
    6.9607628894892334e-14,    8.0230158163380271e-15,    8.8458504976575618e-16,    9.3468557481651422e-17,
    9.4810255351552100e-18,    9.2468273611255744e-19,    8.6838375445957131e-20,    7.8632343922930020e-21,
    6.8740102840433877e-22,    5.8083272960868986e-23,    4.7490155176319611e-24,    3.7611228844873879e-25,
    2.8881220366910333e-26,    2.1522637775892328e-27,    1.5578747876719292e-28,    1.0961778597936760e-29,
    7.5037110713122455e-31,    5.0007518140053865e-32,    3.2468321304203765e-33,    2.0551135870548084e-34,
    1.2689234457627985e-35,    7.6474693897355066e-37,    4.5012159858132228e-38,    2.5888537262920448e-39,
    1.4557137458481914e-40,    8.0066469994346714e-42,    4.3096085004723923e-43,    2.2710932822469604e-44,
    1.1722806977149509e-45,    5.9293842718566237e-47,    2.9399747790827621e-48,    1.4295572905247280e-49,
    6.8193821893978100e-51,    3.1924899534731251e-52,    1.4672527612459366e-53,    6.6224041542614231e-55,
    2.9363048255826126e-56,    1.2793690300559224e-57,    5.4793583230287831e-59,    2.3074288719240746e-60
},
// x = 5.0300
{
    9.9399328425081757e-02,    7.9646530734941753e-02,    5.1896427390325246e-02,    2.8059624780344090e-02,
    1.2847247775532276e-02,    5.0725015239245098e-03,    1.7543020969697299e-03,    5.3851996117968218e-04,
    1.4837775945576772e-04,    3.7044432203926470e-05,    8.4484926814927467e-06,    1.7723951638971026e-06,
    3.4410127599903802e-07,    6.2150253365104210e-08,    1.0491566086947857e-08,    1.6620989870748049e-09,
    2.4801368151666006e-10,    3.4971454261726462e-11,    4.6735425185634844e-12,    5.9350730609049624e-13,
    7.1796010108344977e-14,    8.2913191238672464e-15,    9.1594602757127570e-16,    9.6970964680885068e-17,
    9.8555027200632997e-18,    9.6308530054680637e-19,    9.0621939171226469e-20,    8.2219255815967058e-21,
    7.2016839830029885e-22,    6.0971565072446700e-23,    4.9949710091747799e-24,    3.9636993136086097e-25,
    3.0496719764391622e-26,    2.2771312245878006e-27,    1.6515076888810196e-28,    1.1643540039285479e-29,
    7.9861326399291618e-31,    5.3327659268987563e-32,    3.4692391074161901e-33,    2.2002283339180793e-34,
    1.3612101858966810e-35,    8.2198834681563802e-37,    4.8477050676970050e-38,    2.7936534672370848e-39,
    1.5739826153737432e-40,    8.6742869535228401e-42,    4.6782162274644436e-43,    2.4702278489182722e-44,
    1.2775956147623098e-45,    6.4748741281337324e-47,    3.2168102190900898e-48,    1.5672694175500727e-49,
    7.4911334507767322e-51,    3.5139258372355378e-52,    1.6181868585822588e-53,    7.3181317590750835e-55,
    3.2512218877569894e-56,    1.4193919780056346e-57,    6.0911229329738410e-59,    2.5701427015807031e-60
},
// x = 5.0400
{
    9.9202191526062647e-02,    7.9527532377496352e-02,    5.1864374634695759e-02,    2.8074779763710873e-02,
    1.2871624962875104e-02,    5.0897351871481882e-03,    1.7630759433056489e-03,    5.4211866671695125e-04,
    1.4962753045757969e-04,    3.7422631443368962e-05,    8.5501500163078031e-06,    1.7970063754197637e-06,
    3.4952568403507457e-07,    6.3248022071179685e-08,    1.0696994368040558e-08,    1.6978560645970740e-09,
    2.5383206595538755e-10,    3.5860394651083128e-11,    4.8015475450881786e-12,    6.1093846690405341e-13,
    7.4047503568715523e-14,    8.5679021585502699e-15,    9.4833832711594193e-16,    1.0059566644365118e-16,
    1.0243818613639444e-17,    1.0029854777121391e-18,    9.4560803457085753e-20,    8.5960762784985290e-21,
    7.5441549728041656e-22,    6.3996249731910081e-23,    5.2530500219458373e-24,    4.1766811708940777e-25,
    3.2198558328235786e-26,    2.4089323006527465e-27,    1.7505353781237190e-28,    1.2366009173846787e-29,
    8.4983714905405313e-31,    5.6859991119419320e-32,    3.7063288245995375e-33,    2.3552296581528766e-34,
    1.4599796939273201e-35,    8.8337214839701324e-37,    5.2200050989056648e-38,    3.0141456053830903e-39,
    1.7015661041999010e-40,    9.3959372523103342e-42,    5.0774336438690712e-43,    2.6863266599549708e-44,
    1.3921090382516843e-45,    7.0691855420074117e-47,    3.5190210716572207e-48,    1.7179029360994447e-49,
    8.2273728825627460e-51,    3.8669188988724066e-52,    1.7842680591559565e-53,    8.0851995073453536e-55,
    3.5991200382300967e-56,    1.5743866075493394e-57,    6.7696440202929769e-59,    2.8621028384711197e-60
},
// x = 5.0500
{
    9.9005833707403443e-02,    7.9408852687170128e-02,    5.1832257853639006e-02,    2.8089785505349352e-02,
    1.2895921509590387e-02,    5.1069550922179753e-03,    1.7718609126799443e-03,    5.4572898036861343e-04,
    1.5088374326822140e-04,    3.7803507980541532e-05,    8.6527231434115961e-06,    1.8218869881368743e-06,
    3.5502002912484266e-07,    6.4362091479238142e-08,    1.0905876661589048e-08,    1.7342849077366795e-09,
    2.5977128736388745e-10,    3.6769564566721477e-11,    4.9327210598772738e-12,    6.2883600920483161e-13,
    7.6363761067681451e-14,    8.8529985563285399e-15,    9.8179316230970382e-16,    1.0434661495494779e-16,
    1.0646448867615156e-17,    1.0444378236324087e-18,    9.8660945782909185e-20,    8.9863134355396976e-21,
    7.9020539584321057e-22,    6.7163423051973373e-23,    5.5238195924329497e-24,    4.4005767803075910e-25,
    3.3991133831398376e-26,    2.5480346563247631e-27,    1.8552551976287561e-28,    1.3131515010226559e-29,
    9.0421961909638648e-31,    6.0617546060955062e-32,    3.9590340326981913e-33,    2.5207665149749434e-34,
    1.5656712242336935e-35,    9.4918780852569009e-37,    5.6199745776090041e-38,    3.2514934825902554e-39,
    1.8391746215893660e-40,    1.0175832660481059e-41,    5.5097263343629813e-43,    2.9207925911609779e-44,
    1.5166014357178773e-45,    7.7165661187199721e-47,    3.8488698698031754e-48,    1.8826379113620481e-49,
    9.0341297974154541e-51,    4.2544881370520786e-52,    1.9669784255599208e-53,    8.9307432692798788e-55,
    3.9833703320671601e-56,    1.7459159901872443e-57,    7.5220375813560633e-59,    3.1864911403949929e-60
},
// x = 5.0600
{
    9.8810250382852571e-02,    7.9290490648204204e-02,    5.1800078061782884e-02,    2.8104642761066979e-02,
    1.2920137483231732e-02,    5.1241610715243840e-03,    1.7806568929613282e-03,    5.4935087221662411e-04,
    1.5214640220058466e-04,    3.8187070356952515e-05,    8.7562170657827869e-06,    1.8470390562729475e-06,
    3.6058499181484137e-07,    6.5492654025708646e-08,    1.1118260847621325e-08,    1.7713962033729904e-09,
    2.6583351470379820e-10,    3.7699368348218216e-11,    5.0671328485339678e-12,    6.4721115538092225e-13,
    7.8746473068362235e-14,    9.1468479100030232e-15,    1.0163426078621725e-15,    1.0822787960821768e-16,
    1.1063884228529490e-17,    1.0874987390586441e-18,    1.0292855817556585e-19,    9.3932878125989359e-21,
    8.2760369079481581e-22,    7.0479438032826992e-23,    5.8078718349759672e-24,    4.6359180090324717e-25,
    3.5879056931584604e-26,    2.6948245093739160e-27,    1.9659801299716171e-28,    1.3942513958075751e-29,
    9.6194757752195482e-31,    6.4614124260970684e-32,    4.2283444048007956e-33,    2.6975288356728157e-34,
    1.6787526971694712e-35,    1.0197442970697762e-36,    6.0496011854521882e-38,    3.5069437823529891e-39,
    1.9875709865035358e-40,    1.1018529411986191e-41,    5.9777523321073094e-43,    3.1751410286986465e-44,
    1.6519175459652743e-45,    8.4216224015803031e-47,    4.2088152213018302e-48,    2.0627592262417656e-49,
    9.9179821140496818e-51,    4.6799338006261649e-52,    2.1679413120829903e-53,    9.8625947556968909e-55,
    4.4076799044234433e-56,    1.9357024474573819e-57,    8.3561602021228489e-59,    3.5468275989371198e-60
},
// x = 5.0700
{
    9.8615437001693729e-02,    7.9172445246492903e-02,    5.1767836264123956e-02,    2.8119352283845996e-02,
    1.2944272953093991e-02,    5.1413529588270829e-03,    1.7894637722068274e-03,    5.5298431214291250e-04,
    1.5341551142898044e-04,    3.8573327075324898e-05,    8.8606367877234458e-06,    1.8724646409674172e-06,
    3.6622125670755194e-07,    6.6639903750692067e-08,    1.1334195313333753e-08,    1.8092007750157625e-09,
    2.7202094933205569e-10,    3.8650217233150497e-11,    5.2048540341726118e-12,    6.6607536640759180e-13,
    8.1197369498828669e-14,    9.4496959042438015e-15,    1.0520196206267521e-15,    1.1224365016020519e-16,
    1.1496630975934693e-17,    1.1322265269113022e-18,    1.0737005434171850e-19,    9.8176748183865787e-21,
    8.6667859985231105e-22,    7.3950914721542433e-23,    6.1058249862358138e-24,    4.8832612975057583e-25,
    3.7867160932795527e-26,    2.8497075352486003e-27,    2.0830395810988931e-28,    1.4801596474960016e-29,
    1.0232185197195621e-30,    6.8864336986029673e-32,    4.5153098670526476e-33,    2.8862500120645834e-34,
    1.7997224979627325e-35,    1.0953713544242296e-36,    6.5110104437772976e-38,    3.7818322919407678e-39,
    2.1475741630419958e-40,    1.1928928809847735e-41,    6.4843766590691399e-43,    3.4510086111440577e-44,
    1.7989715110845738e-45,    9.1893493057235136e-47,    4.6015283101602181e-48,    2.2596656291911137e-49,
    1.0886104900165519e-50,    5.1468628846448525e-52,    2.3889344800640780e-53,    1.0889347623463975e-54,
    4.8761246311378820e-56,    2.1456433748376463e-57,    9.2806842674792877e-59,    3.9470054193507116e-60
},
// x = 5.0800
{
    9.8421389048408803e-02,    7.9054715469620979e-02,    5.1735533456112953e-02,    2.8133914823840502e-02,
    1.2968327990191007e-02,    5.1585305892501345e-03,    1.7982814386651219e-03,    5.5662926983151748e-04,
    1.5469107498938112e-04,    3.8962286599336485e-05,    8.9659873146974560e-06,    1.8981658102328042e-06,
    3.7192951246232022e-07,    6.7804036304063825e-08,    1.1553728956469119e-08,    1.8477095840629451e-09,
    2.7833582537634470e-10,    3.9622529452831451e-11,    5.3459570989628120e-12,    6.8544034621254137e-13,
    8.3718220559443166e-14,    9.7617944532557879e-15,    1.0888580614118615e-15,    1.1639823996174468e-16,
    1.1945211372097744e-17,    1.1786814513530560e-18,    1.1199207701785014e-19,    1.0260175379422278e-20,
    9.0750105953812387e-22,    7.7584750746479254e-23,    6.4183244904318032e-24,    5.1431887318396463e-25,
    3.9960511969246335e-26,    3.0131097979386925e-27,    2.2067802005492494e-28,    1.5711494043059744e-29,
    1.0882411066491313e-30,    7.3383652235589842e-32,    4.8210441158173102e-33,    3.0877095253222981e-34,
    1.9291113839720641e-35,    1.1764208358967704e-36,    7.0064749281286883e-38,    4.0775900484659572e-39,
    2.3200632527539831e-40,    1.2912302499333458e-41,    7.0326869285016774e-43,    3.7501626289046277e-44,
    1.9587524054046639e-45,    1.0025161891786377e-46,    5.0299107455072884e-48,    2.4748795403772004e-49,
    1.1946323088950321e-50,    5.6592168736880267e-52,    2.6319043977509097e-53,    1.2020429694935281e-54,
    5.3931848811933050e-56,    2.3778285981668018e-57,    1.0305180618853425e-58,    4.3913296513477935e-60
},
// x = 5.0900
{
    9.8228102042344723e-02,    7.8937300306899952e-02,    5.1703170623739657e-02,    2.8148331128373758e-02,
    1.2992302667233502e-02,    5.1756937992771951e-03,    1.8071097807798406e-03,    5.6028571477072470e-04,
    1.5597309677966914e-04,    3.9353957353362391e-05,    9.0722736531690309e-06,    1.9241446389125398e-06,
    3.7771045179606047e-07,    6.8985248951536343e-08,    1.1776911188696665e-08,    1.8869337310641584e-09,
    2.8478041011338867e-10,    4.0616730329025044e-11,    5.4905159059473769e-12,    7.0531804610697618e-13,
    8.6310837544221527e-14,    1.0083401841144032e-14,    1.1268927172679747e-15,    1.2069608926607557e-16,
    1.2410164123465065e-17,    1.2269257985336855e-18,    1.1680150554404744e-19,    1.0721516837317525e-20,
    9.5014482647100074e-22,    8.1388132239522593e-23,    6.7460441268294505e-24,    5.4163091602620912e-25,
    4.2164419628821985e-26,    3.1854787229816154e-27,    2.3375667405414697e-28,    1.6675086491215896e-29,
    1.1572357680314532e-30,    7.8188442827652611e-32,    5.1467283312677149e-33,    3.3027357271859765e-34,
    2.0674845065511252e-35,    1.2632681397349665e-36,    7.5384240756389486e-38,    4.3957498934914415e-39,
    2.5059817608392242e-40,    1.3974319528946735e-41,    7.6260100845036628e-43,    4.0745111287786772e-44,
    2.1323301910841126e-45,    1.0934929659644098e-46,    5.4971138645333304e-48,    2.7100576778351988e-49,
    1.3107168717507738e-50,    6.2213019249565708e-52,    2.8989818286656641e-53,    1.3266181841495746e-54,
    5.9637846459636642e-56,    2.6345594071492207e-57,    1.1440209378981390e-58,    4.8845597212787569e-60
},
// x = 5.1000
{
    9.8035571537383903e-02,    7.8820198749403561e-02,    5.1670748743617115e-02,    2.8162601941935810e-02,
    1.3016197058607184e-02,    5.1928424267466645e-03,    1.8159486871928093e-03,    5.6395361625519158e-04,
    1.5726158055989135e-04,    3.9748347722220768e-05,    9.1795008104414212e-06,    1.9504032086384331e-06,
    3.8356477148378146e-07,    7.0183740580682059e-08,    1.2003791938994147e-08,    1.9268844569898305e-09,
    2.9135700435008291e-10,    4.1633252371646497e-11,    5.6386057211365551e-12,    7.2572066928324972e-13,
    8.8977073676413560e-14,    1.0414782865022929e-14,    1.1661593242594169e-15,    1.2514176861631460e-16,
    1.2892044854164311e-17,    1.2770239390496583e-18,    1.2180546366773018e-19,    1.1202453875207225e-20,
    9.9468658216217468e-22,    8.5368545159388704e-23,    7.0896871810114071e-24,    5.7032593552675127e-25,
    4.4484448033891405e-26,    3.3672841144097610e-27,    2.4757829556709131e-28,    1.7695409678528575e-29,
    1.2304353365956193e-30,    8.3296037061846495e-32,    5.4936150978777869e-33,    3.5322087820147648e-34,
    2.2154435541187856e-35,    1.3563137237872130e-36,    8.1094546219494386e-38,    4.7379534623069871e-39,
    2.7063421543401615e-40,    1.5121075321214924e-41,    8.2679303586754693e-43,    4.4261137747897822e-44,
    2.3208621321952020e-45,    1.1925013555771357e-46,    6.0065596042928314e-48,    2.9670025701231286e-49,
    1.4377943063771591e-50,    6.8378216995650658e-52,    3.1924988213506189e-53,    1.4637944129436076e-54,
    6.5933343557089456e-56,    2.9183694237818492e-57,    1.2697419730267501e-58,    5.4319562498941009e-60
},
// x = 5.1100
{
    9.7843793121618339e-02,    7.8703409790002329e-02,    5.1638268783065107e-02,    2.8176728006181293e-02,
    1.3040011240351011e-02,    5.2099763108468368e-03,    1.8247980467472524e-03,    5.6763294338807208e-04,
    1.5855652995252015e-04,    4.0145466050921274e-05,    9.2876737944958087e-06,    1.9769436077878008e-06,
    3.8949317235893322e-07,    7.1399711706915461e-08,    1.2234421657031470e-08,    1.9675731445059559e-09,
    2.9806794280747412e-10,    4.2672535377452630e-11,    5.7903032358808702e-12,    7.4666067537982823e-13,
    9.1718824958498838e-14,    1.0756208980913928e-14,    1.2066945907299336e-15,    1.2973998231373936e-16,
    1.3391426591821649e-17,    1.3290423921618854e-18,    1.2701132758366208e-19,    1.1703769474196613e-20,
    1.0412060414284769e-21,    8.9533787029647054e-23,    7.4499876615151880e-24,    6.0047052232284085e-25,
    4.6926427397972983e-26,    3.5590192175094341e-27,    2.6218325450289643e-28,    1.8775663556431590e-29,
    1.3080857149038031e-30,    8.8724772091528583e-32,    5.8630325428163961e-33,    3.7770637785691528e-34,
    2.3736290233946798e-35,    1.4559847160755007e-36,    8.7223417064915861e-38,    5.1059586356078611e-39,
    2.9222307315780491e-40,    1.6359122683273639e-41,    8.9623085293178441e-43,    4.8071935200073700e-44,
    2.5255997014512961e-45,    1.3002305902170777e-46,    6.5619630659605690e-48,    3.2476750272198405e-49,
    1.5768784088873442e-50,    7.5139130677616361e-52,    3.5150072229708656e-53,    1.6148149878580229e-54,
    7.2877777215415163e-56,    3.2320474779997149e-57,    1.4089659506932798e-58,    6.0393325767475763e-60
},
// x = 5.1200
{
    9.7652762417027181e-02,    7.8586932423397191e-02,    5.1605731700192899e-02,    2.8190710059927574e-02,
    1.3063745290135658e-02,    5.2270952921109849e-03,    1.8336577484909687e-03,    5.7132366508313537e-04,
    1.5985794844272004e-04,    4.0545320644416562e-05,    9.3967976138304092e-06,    2.0037679314402678e-06,
    3.9549635931357879e-07,    7.2633364479435388e-08,    1.2468851316556087e-08,    2.0090113192544617e-09,
    3.0491559450759467e-10,    4.3735026529729056e-11,    5.9456865895250613e-12,    7.6815078511435635e-13,
    9.4538031036800451e-14,    1.1107958452477521e-14,    1.2485362210712406e-15,    1.3449557196857318e-16,
    1.3908900265979506e-17,    1.3830498918161837e-18,    1.3242673421673970e-19,    1.2226275900714097e-20,
    1.0897860645374706e-21,    9.3891979105539983e-23,    7.8277115634748227e-24,    6.3213430632816778e-25,
    4.9496466077446414e-26,    3.7612018293364718e-27,    2.7761401386366558e-28,    1.9919220626916753e-29,
    1.3904465763435908e-30,    9.4494050142876239e-32,    6.2563887038030096e-33,    4.0382940208877118e-34,
    2.5427226261422604e-35,    1.5627366248589926e-36,    9.3800506872529799e-38,    5.5016474830094218e-39,
    3.1548128233067294e-40,    1.7695504995768686e-41,    9.7133025744047181e-43,    5.2201491478866729e-44,
    2.7478960161868245e-45,    1.4174273470227486e-46,    7.1673569045580709e-48,    3.5542076470348280e-49,
    1.7290739624727654e-50,    8.2551849324769616e-52,    3.8692988496493223e-53,    1.7810428340713198e-54,
    8.0536429712662022e-56,    3.5786626785834617e-57,    1.5631095541632433e-58,    6.7131114526134094e-60
},
// x = 5.1300
{
    9.7462475079158020e-02,    7.8470765646152296e-02,    5.1573138443981224e-02,    2.8204548839153066e-02,
    1.3087399287242160e-02,    5.2441992124124329e-03,    1.8425276816794384e-03,    5.7502575006687933e-04,
    1.6116583937862076e-04,    4.0947919767356664e-05,    9.5068772772997616e-06,    2.0308782813342208e-06,
    4.0157504129838423e-07,    7.3884902687124858e-08,    1.2707132418779548e-08,    2.0512106511391023e-09,
    3.1190236316314652e-10,    4.4821180498977430e-11,    6.1048353923453617e-12,    7.9020398498555470e-13,
    9.7436676080912900e-14,    1.1470316502624727e-14,    1.2917229400038314e-15,    1.3941352013498107e-16,
    1.4445075219404231e-17,    1.4391174545118551e-18,    1.3805958975420872e-19,    1.2770815725682908e-20,
    1.1405127732028922e-21,    9.8451578984110549e-23,    8.2236581809573387e-24,    6.6539008773669909e-25,
    5.2200963138223727e-26,    3.9743754590106608e-27,    2.9391523301625958e-28,    2.1129634815357769e-29,
    1.4777921019514098e-30,    1.0062439772559851e-31,    6.6751761385669859e-33,    4.3169545081138718e-34,
    2.7234498391629181e-35,    1.6770551540844309e-36,    1.0085749708576615e-37,    5.9270347296297545e-39,
    3.4053383473525597e-40,    1.9137791727652739e-41,    1.0525389815720663e-42,    5.6675687457378863e-44,
    2.9892138428274679e-45,    1.5449003939362755e-46,    7.8271176872881286e-48,    3.8889194409511384e-49,
    1.8955846781946479e-50,    9.0677604357763856e-52,    4.2584274576625635e-53,    1.9639716766264034e-54,
    8.8980988802883825e-56,    3.9615918845174683e-57,    1.7337345795030024e-58,    7.4603874053589979e-60
},
// x = 5.1400
{
    9.7272926796811515e-02,    7.8354908456727101e-02,    5.1540489954363791e-02,    2.8218245076995812e-02,
    1.3110973312540686e-02,    5.2612879149595932e-03,    1.8514077357789096e-03,    5.7873916688063951e-04,
    1.6248020597159531e-04,    4.1353271643846043e-05,    9.6179177939542236e-06,    2.0582767658229312e-06,
    4.0772993132243201e-07,    7.5154531764409853e-08,    1.2949316995765365e-08,    2.0941829556169017e-09,
    3.1903068757004767e-10,    4.5931459544610111e-11,    6.2678307487725960e-12,    8.1283353204475487e-13,
    1.0041678967815210e-13,    1.1843575468055133e-14,    1.3362945173795288e-15,    1.4449895403202798e-16,
    1.5000579732580203e-17,    1.4973184490649176e-18,    1.4391807843412680e-19,    1.3338262876450422e-20,
    1.1934756705522445e-21,    1.0322139367261121e-22,    8.6386614697421887e-24,    7.0031397333619826e-25,
    5.5046621458046376e-26,    4.1991105398919012e-27,    3.1113387579781997e-28,    2.2410650777198392e-29,
    1.5704117548074927e-30,    1.0713752798689036e-31,    7.1209767886649682e-33,    4.6141656136461407e-34,
    2.9165826057081263e-35,    1.7994581304541813e-36,    1.0842823068107121e-37,    6.3842767788782327e-39,
    3.6751477398834717e-40,    2.0694116434086072e-41,    1.1403390658117184e-42,    6.1522441772874383e-44,
    3.2511342118997326e-45,    1.6835255998060381e-46,    8.5459943744829611e-48,    4.2543316683352524e-49,
    2.0777218091308060e-50,    9.9583228345787410e-52,    4.6857326718060656e-53,    2.1652382694293936e-54,
    9.8290160344174431e-56,    4.3845498006545784e-57,    1.9225624392609834e-58,    8.2889953328857157e-60
},
// x = 5.1500
{
    9.7084113291729543e-02,    7.8239359855507437e-02,    5.1507787162307736e-02,    2.8231799503752368e-02,
    1.3134467448469571e-02,    5.2783612442909734e-03,    1.8602978004694352e-03,    5.8246388388269227e-04,
    1.6380105129654504e-04,    4.1761384457204017e-05,    9.7299241728797248e-06,    2.0859654998303641e-06,
    4.1396174645285739e-07,    7.6442458797076040e-08,    1.3195457613817827e-08,    2.1379401949950716e-09,
    3.2630304200283908e-10,    4.7066333617656250e-11,    6.4347552809035097e-12,    8.3605295873783157e-13,
    1.0348044774323115e-13,    1.2228034956768277e-14,    1.3822917935153769e-15,    1.4975714935237181e-16,
    1.5576061561691845e-17,    1.5577286683134939e-18,    1.5001067159705543e-19,    1.3929523722438142e-20,
    1.2487677651918199e-21,    1.0821059313062329e-22,    9.0735914623491204e-24,    7.3698551833271204e-25,
    5.8040461385845744e-26,    4.4360056958240439e-27,    3.2931932366886938e-28,    2.3766213658612545e-29,
    1.6686110938227623e-30,    1.1405640636753759e-31,    7.5954671110511249e-33,    4.9311169745280802e-34,
    3.1229421969212128e-35,    1.9304975486952978e-36,    1.1654885431704432e-37,    6.8756813266035255e-39,
    3.9656782879070783e-40,    2.2373217404881428e-41,    1.2352494034828339e-42,    6.6771866259369766e-44,
    3.5353656886342151e-45,    1.8342513363796677e-46,    9.3291390888356319e-48,    4.6531849769512847e-49,
    2.2769134933064866e-50,    1.0934165355532961e-51,    5.1548660404150132e-53,    2.3866357373033182e-54,
    1.0855033800098670e-55,    4.8516219418609722e-57,    2.1314900797476629e-58,    9.2075859293895961e-60
},
// x = 5.1600
{
    9.6896030318286794e-02,    7.8124118844835930e-02,    5.1475030989893825e-02,    2.8245212846876779e-02,
    1.3157881779014469e-02,    5.2954190462701552e-03,    1.8691977656478608e-03,    5.8619986925035109e-04,
    1.6512837829218925e-04,    4.2172266349727486e-05,    9.8429014230376879e-06,    2.1139466048066387e-06,
    4.2027120781430246e-07,    7.7748892528042307e-08,    1.3445607376871526e-08,    2.1824944797333630e-09,
    3.3372193661295179e-10,    4.8226280464484283e-11,    6.6056931523026096e-12,    8.5987607781830009e-13,
    1.0662977344336761e-13,    1.2624002008595875e-14,    1.4297567050685912e-15,    1.5519353416049910e-16,
    1.6172188490401142e-17,    1.6204264028138811e-18,    1.5634613700812906e-19,    1.4545538195500666e-20,
    1.3064856994980031e-21,    1.1342872430179404e-22,    9.5293557371797067e-24,    7.7548787389441849e-25,
    6.1189834980406349e-26,    4.6856890637186730e-27,    3.4852349413673330e-28,    2.5200479332124308e-29,
    1.7727126288215273e-30,    1.2140531972670830e-31,    8.1004234914661271e-33,    5.2690716025642981e-34,
    3.3434022423883190e-35,    2.0707617419845812e-36,    1.2525796948005144e-37,    7.4037176038881362e-39,
    4.2784708891428159e-40,    2.4184481141804899e-41,    1.3378284677225035e-42,    7.2456432852793496e-44,
    3.8437543474293723e-45,    1.9981043018710977e-46,    1.0182140351102493e-47,    5.0884579537484800e-49,
    2.4947148852630590e-50,    1.2003245364345175e-51,    5.6698194007826957e-53,    2.6301281296466081e-54,
    1.1985633519645800e-55,    5.3673007319260346e-57,    2.3626074461809770e-58,    1.0225708608706686e-59
},
// x = 5.1700
{
    9.6708673663185640e-02,    7.8009184429041492e-02,    5.1442222350395604e-02,    2.8258485830979993e-02,
    1.3181216389687681e-02,    5.3124611680807430e-03,    1.8781075214307846e-03,    5.8994709098206092e-04,
    1.6646218976136177e-04,    4.2585925422457164e-05,    9.9568545531053033e-06,    2.1422222086831927e-06,
    4.2665904058819704e-07,    7.9074043363092195e-08,    1.3699819929881707e-08,    2.2278580697518740e-09,
    3.4128991782985100e-10,    4.9411785731548223e-11,    6.7807300920971822e-12,    8.8431698733240060e-13,
    1.0986693813903649e-13,    1.3031791258803648e-14,    1.4787323114625156e-15,    1.6081369288234576e-16,
    1.6789648895735198e-17,    1.6854925165770410e-18,    1.6293354845683830e-19,    1.5187280946009653e-20,
    1.3667298822675645e-21,    1.1888572565159960e-22,    1.0006900943700159e-23,    8.1590794063060364e-25,
    6.4502440851404515e-26,    4.9488196748411769e-27,    3.6880096468132561e-28,    2.6717825129094136e-29,
    1.8830567189112063e-30,    1.2920994910995486e-31,    8.6377279544148773e-33,    5.6293702292480292e-34,
    3.5788919393708881e-35,    2.2208776848767326e-36,    1.3459679317333600e-37,    7.9710272880354159e-39,
    4.6151772670522262e-40,    2.6137988854600517e-41,    1.4486772335309143e-42,    7.8611152787159157e-44,
    4.1782945018769842e-45,    2.1761957979604993e-46,    1.1111058973900067e-47,    5.5633871985902476e-49,
    2.7328191408861063e-50,    1.3174243212273144e-51,    6.2349557541392768e-53,    2.8978662927281500e-54,
    1.3231218494700823e-55,    5.9365250275819164e-57,    2.6182166423838552e-58,    1.1353902651336783e-59
},
// x = 5.1800
{
    9.6522039145154392e-02,    7.7894555614468314e-02,    5.1409362148358069e-02,    2.8271619177829270e-02,
    1.3204471367507702e-02,    5.3294874582212942e-03,    1.8870269581574607e-03,    5.9370551689948354e-04,
    1.6780248837131189e-04,    4.3002369734946258e-05,    1.0071788571315926e-05,    2.1707944458276131e-06,
    4.3312597401185785e-07,    8.0418123376561105e-08,    1.3958149462214886e-08,    2.2740433757442003e-09,
    3.4900956876504066e-10,    5.0623343071159633e-11,    6.9599534193675786e-12,    9.0939007567692027e-13,
    1.1319416234057384e-13,    1.3451725104810863e-14,    1.5292628218735466e-15,    1.6662337038816682e-16,
    1.7429152328402907e-17,    1.7530105248961063e-18,    1.6978229564201662e-19,    1.5855762535703185e-20,
    1.4296046258631813e-21,    1.2459194222802687e-22,    1.0507214385654967e-23,    8.5833652822898705e-25,
    6.7986339626744304e-26,    5.2260888972533480e-27,    3.9020910242487234e-28,    2.8322861091926962e-29,
    2.0000025162208694e-30,    1.3749744634326100e-31,    9.2093741852381225e-33,    6.0134358972128560e-34,
    3.8303994508089100e-35,    2.3815134364950337e-36,    1.4460932872851300e-37,    8.5804361236822275e-39,
    4.9775676705474773e-40,    2.8244566177870880e-41,    1.5684423083674876e-42,    8.5273768956534289e-44,
    4.5411402457189823e-45,    2.3697284944140048e-46,    1.2122466819629660e-47,    6.0814890421974521e-49,
    2.9930693252490404e-50,    1.4456626152097194e-51,    6.8550428660278137e-53,    3.1922051768399832e-54,
    1.4603201370709037e-55,    6.5647233841485673e-57,    2.9008529452573631e-58,    1.2603797370352871e-59
},
// x = 5.1900
{
    9.6336122614648786e-02,    7.7780231409503631e-02,    5.1376451279675592e-02,    2.8284613606347982e-02,
    1.3227646800978880e-02,    5.3464977665002187e-03,    1.8959559663926779e-03,    5.9747511464957877e-04,
    1.6914927665401152e-04,    4.3421607305032275e-05,    1.0187708485299859e-05,    2.1996654569981524e-06,
    4.3967274137741153e-07,    8.1781346316980343e-08,    1.4220650711039792e-08,    2.3210629604959421e-09,
    3.5688350961894817e-10,    5.1861454248293611e-11,    7.1434520678354646e-12,    9.3511002673057539e-13,
    1.1661371668084988e-13,    1.3884133876077538e-14,    1.5813936227892721e-15,    1.7262847617056357e-16,
    1.8091430107867758e-17,    1.8230666743168358e-18,    1.7690209434972706e-19,    1.6552030668370761e-20,
    1.4952182879945851e-21,    1.3055814126260721e-22,    1.1031325664367815e-23,    9.0286852148254657e-25,
    7.1649970071009597e-26,    5.5182219419640615e-27,    4.1280819979721659e-28,    3.0020441769855794e-29,
    2.1239289571855541e-30,    1.4629651464470589e-31,    9.8174738805560156e-33,    6.4227788115815210e-34,
    4.0989755027293823e-35,    2.5533807321794944e-36,    1.5534254735202777e-37,    9.2349662984994133e-39,
    5.3675390897405011e-40,    3.0515836323985988e-41,    1.6978192856639466e-42,    9.2484962377700828e-44,
    4.9346178640270477e-45,    2.5800037180038254e-46,    1.3223488644027491e-47,    6.6465830389377743e-49,
    3.2774713187596080e-50,    1.5860717747818403e-51,    7.5352898259463065e-53,    3.5157227044859833e-54,
    1.6114099589005144e-55,    7.2578614077889802e-57,    3.2133078490069693e-58,    1.3988222166333998e-59
},
// x = 5.2000
{
    9.6150919953556888e-02,    7.7666210824605258e-02,    5.1343490631669229e-02,    2.8297469832615611e-02,
    1.3250742780071314e-02,    5.3634919440306373e-03,    1.9048944369295829e-03,    6.0125585170667977e-04,
    1.7050255700646797e-04,    4.3843646108611668e-05,    1.0304619301925295e-05,    2.2288373892979457e-06,
    4.4630008003053638e-07,    8.3163927612676033e-08,    1.4487378964718318e-08,    2.3689295402085074e-09,
    3.6491439809067854e-10,    5.3126629248431348e-11,    7.3313166108522311e-12,    9.6149182505973433e-13,
    1.2012792290422271e-13,    1.4329356007209011e-14,    1.6351713061481122e-15,    1.7883508861959852e-16,
    1.8777235932508820e-17,    1.8957500248039205e-18,    1.8430299693190831e-19,    1.7277171459470366e-20,
    1.5636834182794951e-21,    1.3679552832975015e-22,    1.1580308384252249e-23,    9.4960305294494756e-25,
    7.5502165880760817e-26,    5.8259794354383730e-27,    4.3666161645865217e-28,    3.1815678583183146e-29,
    2.2552358036519081e-30,    1.5563749345440340e-31,    1.0464263444168243e-32,    6.8590014652755672e-34,
    4.3857371922661973e-35,    2.7372377322452770e-36,    1.6684658105468522e-37,    9.9378496206140168e-39,
    5.7871240210457897e-40,    3.2964276901028579e-41,    1.8375563365788795e-42,    1.0028857375246455e-43,
    5.3612391780842841e-45,    2.8084293050776273e-46,    1.4421847263418701e-47,    7.2628173751484446e-49,
    3.5882078028514852e-50,    1.7397773236790384e-51,    8.2813868196154542e-53,    3.8712403365690614e-54,
    1.7777639631498646e-55,    8.0224935703151242e-57,    3.5586543301741638e-58,    1.5521327423257744e-59
},
// x = 5.2100
{
    9.5966427074906879e-02,    7.7552492872327891e-02,    5.1310481083163370e-02,    2.8310188569867820e-02,
    1.3273759396200867e-02,    5.3804698432252489e-03,    1.9138422607924682e-03,    6.0504769537456213e-04,
    1.7186233169104147e-04,    4.4268494079417276e-05,    1.0422526027139635e-05,    2.2583123961289071e-06,
    4.5300873136903003e-07,    8.4566084377324556e-08,    1.4758390066196406e-08,    2.4176559858281942e-09,
    3.7310492979065142e-10,    5.4419386386446819e-11,    7.5236392866902560e-12,    9.8855076119931130e-13,
    1.2373915487199412e-13,    1.4787738214328818e-14,    1.6906436980709982e-15,    1.8524945939697360e-16,
    1.9487346505209646e-17,    1.9711525341573190e-18,    1.9199540309384886e-19,    1.8032310745804457e-20,
    1.6351169097327965e-21,    1.4331576408288935e-22,    1.2155281922725012e-23,    9.9864368246209494e-25,
    7.9552173183362467e-26,    6.1501590612197740e-27,    4.6183592775305953e-28,    3.3713952781942580e-29,
    2.3943447361841183e-30,    1.6555244769290420e-31,    1.1152111046342572e-32,    7.3238040530761748e-34,
    4.6918720180999817e-35,    2.9338919369873878e-36,    1.7917492764989039e-37,    1.0692541547715661e-38,
    6.2384998170187481e-40,    3.5603280639486447e-41,    1.9884580563342859e-42,    1.0873184119689164e-43,
    5.8237158919232447e-45,    3.0565280599827354e-46,    1.5725912301539875e-47,    7.9346963444866003e-49,
    3.9276534128755136e-50,    1.9080061340166699e-51,    9.0995483883017095e-53,    4.2618454852200544e-54,
    1.9608870846316410e-55,    8.8658198961450694e-57,    3.9402745420386967e-58,    1.7218717286580692e-59
},
// x = 5.2200
{
    9.5782639922578391e-02,    7.7439076567349135e-02,    5.1277423504561631e-02,    2.8322770528496979e-02,
    1.3296696742209381e-02,    5.3974313177911504e-03,    1.9227993292395232e-03,    6.0885061278850643e-04,
    1.7322860283576903e-04,    4.4696159108798985e-05,    1.0541433665811047e-05,    2.2880926371453310e-06,
    4.5979944084119685e-07,    8.5988035415461756e-08,    1.5033740416394625e-08,    2.4672553243804862e-09,
    3.8145783865611480e-10,    5.5740252416540371e-11,    7.7205140241392044e-12,    1.0163024370096188e-12,
    1.2744983958458308e-13,    1.5259635674770738e-14,    1.7478598881956994e-15,    1.9187801791126616e-16,
    2.0222562174720373e-17,    2.0493691447338627e-18,    1.9999007099873023e-19,    1.8818615436409383e-20,
    1.7096401553364119e-21,    1.5013098158649915e-22,    1.2757413266785282e-23,    1.0500985838358002e-24,
    8.3809668767001099e-26,    6.4915972735264877e-27,    4.8840107997525012e-28,    3.5720929026057373e-29,
    2.5417005020563207e-30,    1.7607526166817458e-31,    1.1883524065307768e-32,    7.8189901899870782e-34,
    5.0186421457584100e-35,    3.1442032775756614e-36,    1.9238466854581628e-37,    1.1502736120797967e-38,
    6.7239986585019373e-40,    3.8447220287000820e-41,    2.1513895825547911e-42,    1.1786565527754554e-43,
    6.3249750132521979e-45,    3.3259468646068863e-46,    1.7144752790860542e-47,    8.6671100534095767e-49,
    4.2983911527463931e-50,    2.0920953057635774e-51,    9.9965604723921901e-53,    4.6909159345579438e-54,
    2.1624289711505825e-55,    9.7957479675994432e-57,    4.3618901660304761e-58,    1.9097595462175849e-59
},
// x = 5.2300
{
    9.5599554471016618e-02,    7.7325960926494547e-02,    5.1244318757922229e-02,    2.8335216416052651e-02,
    1.3319554912345067e-02,    5.4143762227246298e-03,    1.9317655337655407e-03,    6.1266457091735610e-04,
    1.7460137243469244e-04,    4.5126649045506774e-05,    1.0661347221570323e-05,    2.3181802782071901e-06,
    4.6667295794405768e-07,    8.7430001227947551e-08,    1.5313486977598158e-08,    2.5177407403095393e-09,
    3.8997589736953982e-10,    5.7089762643225989e-11,    7.9220364684099123e-12,    1.0447627711099972e-12,
    1.3126245822064012e-13,    1.5745412210142112e-14,    1.8068702596245745e-15,    1.9872737589626326e-16,
    2.0983707593145348e-17,    2.1304978725306135e-18,    2.0829812869768942e-19,    1.9637294905843956e-20,
    1.7873792098467567e-21,    1.5725380426363177e-22,    1.3387918918597176e-23,    1.1040807388845100e-24,
    8.8284779070573622e-26,    6.8511710857944392e-27,    5.1643055274807575e-28,    3.7842569615240363e-29,
    2.6977721205300058e-30,    1.8724173786165873e-31,    1.2661156930694737e-32,    8.3464729502495266e-34,
    5.3673889208827309e-35,    3.3690873930199002e-36,    2.0653670009850989e-37,    1.2372381858647980e-38,
    7.2461181889736916e-40,    4.1511517947048141e-41,    2.3272810041796583e-42,    1.2774483257240486e-43,
    6.8681754266049422e-45,    3.6184664875679309e-46,    1.8688193923778259e-47,    9.4653665323587552e-49,
    4.7032301733297243e-50,    2.2935018025654489e-51,    1.0979831561021356e-52,    5.1621464443641062e-54,
    2.3841975468480394e-55,    1.0820960734494495e-56,    4.8275956685629011e-58,    2.1176925281123297e-59
},
// x = 5.2400
{
    9.5417166724949778e-02,    7.7213144968762068e-02,    5.1211167697032559e-02,    2.8347526937242452e-02,
    1.3342334002243031e-02,    5.4313044143059503e-03,    1.9407407661045861e-03,    6.1648953656556662e-04,
    1.7598064234819238e-04,    4.5559971695477142e-05,    1.0782271696653073e-05,    2.3485774913331339e-06,
    4.7363003622138128e-07,    8.8892204017383867e-08,    1.5597687276846202e-08,    2.5691255768228199e-09,
    3.9866191777989952e-10,    5.8468461033376224e-11,    8.1283040073483126e-12,    1.0739480043900551e-12,
    1.3517954719332587e-13,    1.6245440472810796e-14,    1.8677265194967473e-15,    2.0580433209446379e-16,
    2.1771632389915637e-17,    2.2146398986877089e-18,    2.1693108589405030e-19,    2.0489602431093418e-20,
    1.8684649570010261e-21,    1.6469736447921244e-22,    1.4048066872488449e-23,    1.1607081391751181e-24,
    9.2988099963175493e-26,    7.2297999372543130e-27,    5.4600152881690752e-28,    4.0085149398089932e-29,
    2.8630541481317366e-30,    1.9908970083478140e-31,    1.3487819389640192e-32,    8.9082812441968505e-34,
    5.7395376442647642e-35,    3.6095191039487692e-36,    2.2169597933702871e-37,    1.3305698672538179e-38,
    7.8075328534525600e-40,    4.4812719155011712e-41,    2.5171320807398253e-42,    1.3842840905682328e-43,
    7.4567257019972231e-45,    3.9360121450833887e-46,    2.0366878270973172e-47,    1.0335226441612928e-48,
    5.1452250245477248e-50,    2.5138129065007815e-51,    1.2057448296126035e-52,    5.6795777264737413e-54,
    2.6281738137768717e-55,    1.1950990657202607e-56,    5.3418947343154754e-58,    2.3477605391338404e-59
},
// x = 5.2500
{
    9.5235472719109551e-02,    7.7100627715345788e-02,    5.1177971167483391e-02,    2.8359702793933027e-02,
    1.3365034108906028e-02,    5.4482157500941250e-03,    1.9497249182326218e-03,    6.2032547637525296e-04,
    1.7736641430332717e-04,    4.5996134821622174e-05,    1.0904212091742238e-05,    2.3792864546532030e-06,
    4.8067143326154031e-07,    9.0374867693487203e-08,    1.5886399409320449e-08,    2.6214233372408562e-09,
    4.0751875132683334e-10,    5.9876900329330740e-11,    8.3394157979618924e-12,    1.1038747055993343e-12,
    1.3920369922398030e-13,    1.6760102135869032e-14,    1.9304817301958046e-15,    2.1311587704785649e-16,
    2.2587211862612866e-17,    2.3018996634696483e-18,    2.2590084605057798e-19,    2.1376836673336765e-20,
    1.9530332822881271e-21,    1.7247532277994783e-22,    1.4739178665856580e-23,    1.2201039957093813e-24,
    9.7930717344004909e-26,    7.6284476407495839e-27,    5.7719507158162379e-28,    4.2455271391098482e-29,
    3.0380680067684491e-30,    2.1165910650858175e-31,    1.4366485217282747e-32,    9.5065665510147813e-34,
    6.1366026231931183e-35,    3.8665360945399781e-36,    2.3793178491821534e-37,    1.4307195864109194e-38,
    8.4111059869161264e-40,    4.8368572003760808e-41,    2.7220172930919722e-42,    1.4997995470299853e-43,
    8.0943032281893129e-45,    4.2806648692932834e-46,    2.2192331807693215e-47,    1.1282940575160272e-48,
    5.6276964997832452e-50,    2.7547575593780327e-51,    1.3238235908014773e-52,    6.2476279997285202e-54,
    2.8965280017334840e-55,    1.3196300759304185e-56,    5.9097401716143948e-58,    2.6022662563538097e-59
},
// x = 5.2600
{
    9.5054468517954363e-02,    7.6988408189658880e-02,    5.1144730006742065e-02,    2.8371744685151225e-02,
    1.3387655330685299e-02,    5.4651100889216301e-03,    1.9587178823701015e-03,    6.2417235682822418e-04,
    1.7875868989417713e-04,    4.6435146143621386e-05,    1.1027173405810869e-05,    2.4103093523612372e-06,
    4.8779791069518647e-07,    9.1878217878413589e-08,    1.6179682041732540e-08,    2.6746476863520362e-09,
    4.1654928946769533e-10,    6.1315642163072150e-11,    8.5554727932609253e-12,    1.1345597770162245e-12,
    1.4333756443340680e-13,    1.7289788086627563e-14,    1.9951903412042434e-15,    2.2066919799810915e-16,
    2.3431347685016420e-17,    2.3923849627851290e-18,    2.3521971884881076e-19,    2.2300343205856249e-20,
    2.0412252514545809e-21,    1.8060188781231573e-22,    1.5462631506552573e-23,    1.2823969568580772e-24,
    1.0312422859461427e-25,    8.0481244151252703e-27,    6.1009631069919777e-28,    4.4959883139596805e-29,
    3.2233633776442868e-30,    2.2499215708098913e-31,    1.5300301394443117e-32,    1.0143610026393968e-33,
    6.5601925144180550e-35,    4.1412428145639073e-36,    2.5531799421775647e-37,    1.5381691273163513e-38,
    9.0599026999502988e-40,    5.2198111650647713e-41,    2.9430912480831090e-42,    1.6246791077524963e-43,
    8.7848747658615324e-45,    4.6546737438113644e-46,    2.4177035113902707e-47,    1.2315290381377328e-48,
    6.1542542004132825e-50,    3.0182186635990300e-51,    1.4531823890538822e-52,    6.8711273466898019e-54,
    3.1916371858820498e-55,    1.4568373216955298e-56,    6.5365776123554053e-58,    2.8837463237004563e-59
},
// x = 5.2700
{
    9.4874150215395914e-02,    7.6876485417356269e-02,    5.1111445044225365e-02,    2.8383653307085532e-02,
    1.3410197767261664e-02,    5.4819872908891424e-03,    1.9677195509845181e-03,    6.2803014424801676e-04,
    1.8015747058219305e-04,    4.6877013337716828e-05,    1.1151160635965350e-05,    2.4416483746670270e-06,
    4.9501023419275176e-07,    9.3402481912037408e-08,    1.6477594415710208e-08,    2.7288124517724633e-09,
    4.2575646410749079e-10,    6.2785257171475684e-11,    8.7765777694172632e-12,    1.1660204601969795e-12,
    1.4758385145100517e-13,    1.7834898623694876e-14,    2.0619082216161720e-15,    2.2847168389835312e-16,
    2.4304968632755370e-17,    2.4862070473070471e-18,    2.4490043300975435e-19,    2.3261516089403166e-20,
    2.1331872949206471e-21,    1.8909183694076933e-22,    1.6219860479396957e-23,    1.3477213348462129e-24,
    1.0858076491661214e-25,    8.4898890056459781e-27,    6.4479463610353031e-28,    4.7606293854027155e-29,
    3.4195196640757239e-30,    2.3913342185863298e-31,    1.6292597776394107e-32,    1.0821830005028106e-33,
    7.0120159748557803e-35,    4.4348146141613344e-36,    2.7393337751594824e-37,    1.6534331646038883e-38,
    9.7572036122673911e-40,    5.6321750558845947e-41,    3.1815944610884123e-42,    1.7596595140338567e-43,
    9.5327185226356459e-45,    5.0604690705601771e-46,    2.6334500141192124e-47,    1.3439631735881789e-48,
    6.7288209582558563e-50,    3.3062464214518412e-51,    1.5948717357431154e-52,    7.5553551140906380e-54,
    3.5161045020137147e-55,    1.6079806167491497e-56,    7.2283933580742076e-58,    3.1949945581033875e-59
},
// x = 5.2800
{
    9.4694513934528560e-02,    7.6764858426356192e-02,    5.1078117101371633e-02,    2.8395429353087594e-02,
    1.3432661519626724e-02,    5.4988472173602291e-03,    1.9767298167929137e-03,    6.3189880480191965e-04,
    1.8156275769655057e-04,    4.7321744036510628e-05,    1.1276178777288851e-05,    2.4733057177481308e-06,
    5.0230917346176855e-07,    9.4947888857181383e-08,    1.6780196351181966e-08,    2.7839316253107569e-09,
    4.3514324803169908e-10,    6.4286325112635720e-11,    9.0028353532427788e-12,    1.1982743418056465e-12,
    1.5194532854197999e-13,    1.8395843657695948e-14,    2.1306926933197770e-15,    2.3653093053876828e-16,
    2.5209031326951606e-17,    2.5834807242553236e-18,    2.5495614948542074e-19,    2.4261799496364893e-20,
    2.2290713982866148e-21,    1.9796053758892919e-22,    1.7012360834565943e-23,    1.4162173411028203e-24,
    1.1431301458910524e-25,    8.9548508960333599e-27,    6.8138390080311064e-28,    5.0402192356349658e-29,
    3.6271475264391658e-30,    2.5412996429296113e-31,    1.7346897277787581e-32,    1.1543789918922110e-33,
    7.4938876370032436e-35,    4.7485021246678840e-36,    2.9386191029112461e-37,    1.7770614299397337e-38,
    1.0506519487822390e-39,    6.0761374848229185e-41,    3.4388595419260330e-42,    1.9055337112288665e-43,
    1.0342447858943909e-44,    5.5006765365203624e-46,    2.8679352968225171e-47,    1.4663942219715325e-48,
    7.3556592644155452e-50,    3.6210727979860152e-51,    1.7500374557519433e-52,    8.3060806193151964e-54,
    3.8727801004539939e-55,    1.7744419479937468e-56,    7.9917667554946037e-58,    3.5390874011928383e-59
},
// x = 5.2900
{
    9.4515555827361417e-02,    7.6653526246861364e-02,    5.1044746991712249e-02,    2.8407073513673997e-02,
    1.3455046690064231e-02,    5.5156897309560325e-03,    1.9857485727643523e-03,    6.3577830450299154e-04,
    1.8297455243450903e-04,    4.7769345828765699e-05,    1.1402232822685224e-05,    2.5052835837014769e-06,
    5.0969550224401691e-07,    9.6514669504800632e-08,    1.7087548249760279e-08,    2.8400193643378327e-09,
    4.4471265534198769e-10,    6.5819434983276181e-11,    9.2343520499902630e-12,    1.2313393595257908e-12,
    1.5642482475286458e-13,    1.8973042915686895e-14,    2.2016025648614220e-15,    2.4485474578822894e-16,
    2.6144520996249880e-17,    2.6843244619068279e-18,    2.6540047503094063e-19,    2.5302689385115643e-20,
    2.3290352991143985e-21,    2.0722396932724072e-22,    1.7841690360674955e-23,    1.4880313307996947e-24,
    1.2033424718142359e-25,    9.4441726158515073e-27,    7.1996263283169589e-28,    5.3355665872868469e-29,
    3.8468904926294844e-30,    2.7003147552394460e-31,    1.8466926600025275e-32,    1.2312206653537859e-33,
    8.0077344269296652e-35,    5.0836358995272587e-36,    3.1519310469129174e-37,    1.9096410158674022e-38,
    1.1311606828532312e-39,    6.5540447154634638e-41,    3.7163178113122978e-42,    2.0631550018400662e-43,
    1.1219036741291474e-44,    5.9781324539167162e-46,    3.1227422997438690e-47,    1.5996872175108503e-48,
    8.0393998645214939e-50,    3.9651271994001895e-51,    1.9199291065533502e-52,    9.1296074471630046e-54,
    4.2647839917371315e-55,    1.9577370295401582e-56,    8.8339275193975585e-58,    3.9194118284117952e-59
},
// x = 5.3000
{
    9.4337272074553627e-02,    7.6542487911379645e-02,    5.1011335520942530e-02,    2.8418586476528219e-02,
    1.3477353382131655e-02,    5.5325146955499333e-03,    1.9947757121223588e-03,    6.3966860921207307e-04,
    1.8439285586177519e-04,    4.8219826259209239e-05,    1.1529327762723203e-05,    2.5375841804946425e-06,
    5.1716999831249164e-07,    9.8103056379116786e-08,    1.7399711098122852e-08,    2.8970899931615672e-09,
    4.5446774189481472e-10,    6.7385185137248010e-11,    9.4712362714790227e-12,    1.2652338080548263e-12,
    1.6102523107558966e-13,    1.9566926149321114e-14,    2.2746981660032047e-15,    2.5345115495428738e-16,
    2.7112452257636027e-17,    2.7888604968977431e-18,    2.7624747616717776e-19,    2.6385735225965762e-20,
    2.4332426901744382e-21,    2.1689874673124590e-22,    1.8709471845474713e-23,    1.5633160569142675e-24,
    1.2665833875793756e-25,    9.9590721471094804e-27,    7.6063425674218027e-28,    5.6475219711288791e-29,
    4.0794266475570187e-30,    2.8689041474867297e-31,    1.9656627528630665e-32,    1.3129959364915082e-33,
    8.5556022436487130e-35,    5.4416313301013464e-36,    3.3802236131511999e-37,    2.0517988255080363e-38,
    1.2176484487066556e-39,    7.0684116421410192e-41,    4.0155063767832848e-42,    2.2334414955256458e-43,
    1.2167847067502260e-44,    6.4959001525903214e-46,    3.3995839078833070e-47,    1.7447799831563211e-48,
    8.7850726927217063e-50,    4.3410534661706730e-51,    2.1059091207327141e-52,    1.0032821644907117e-53,
    4.6955309502797994e-55,    2.1595279216221343e-56,    9.7628184581870806e-58,    4.3396959468735525e-59
},
// x = 5.3100
{
    9.4159658885152250e-02,    7.6431742454743712e-02,    5.0977883486991969e-02,    2.8429968926502704e-02,
    1.3499581700641892e-02,    5.5493219762622050e-03,    2.0038111283473092e-03,    6.4356968463979156e-04,
    1.8581766891287129e-04,    4.8673192828339064e-05,    1.1657468585481036e-05,    2.5702097219168780e-06,
    5.2473344346819472e-07,    9.9713283742704530e-08,    1.7716746471391839e-08,    2.9551580044063168e-09,
    4.6441160574292003e-10,    6.8984183405118783e-11,    9.7135983645477119e-12,    1.2999763451818276e-12,
    1.6574950163033817e-13,    2.0177933346824163e-14,    2.3500413829861217e-15,    2.6232840626382839e-16,
    2.8113869916453937e-17,    2.8972149443853257e-18,    2.8751169354402681e-19,    2.7512541780162114e-20,
    2.5418634293533876e-21,    2.2700214303535958e-22,    1.9617395627169704e-23,    1.6422309341633407e-24,
    1.3329979811311198e-25,    1.0500825434098674e-26,    8.0350732504956341e-28,    5.9769797861418073e-29,
    4.3254704053671767e-30,    3.0476215674686660e-31,    2.0920168829514034e-32,    1.4000098782072841e-33,
    9.1396630196624649e-35,    5.8239938519929613e-36,    3.6245134249745945e-37,    2.2042041769975165e-38,
    1.3105451362845681e-39,    7.6219335073736146e-41,    4.3380756988827733e-42,    2.4173808765371365e-43,
    1.3194657997122644e-44,    7.0572876088944401e-46,    3.7003133082169647e-47,    1.9026890816970179e-48,
    9.5981403300969694e-50,    4.7517282880117457e-51,    2.3094627323770092e-52,    1.1023244149338689e-53,
    5.1687576564937297e-55,    2.3816368099669118e-56,    1.0787164098347028e-57,    4.8040425345163705e-59
},
// x = 5.3200
{
    9.3982712496333096e-02,    7.6321288914130606e-02,    5.0944391680094042e-02,    2.8441221545621179e-02,
    1.3521731751645118e-02,    5.5661114394546241e-03,    2.0128547151787904e-03,    6.4748149634855578e-04,
    1.8724899239150824e-04,    4.9129452992232724e-05,    1.1786660276391392e-05,    2.6031624275298560e-06,
    5.3238662353674831e-07,    1.0134558760152823e-07,    1.8038716536510929e-08,    3.0142380603972313e-09,
    4.7454738757970338e-10,    7.0617047214859743e-11,    9.9615506398367958e-12,    1.3355859979496687e-12,
    1.7060065486742529e-13,    2.0806514948834950e-14,    2.4276956945110880e-15,    2.7149497646675013e-16,
    2.9149849786037977e-17,    3.0095179111372424e-18,    2.9920815671479785e-19,    2.8684770933428223e-20,
    2.6550737564232444e-21,    2.3755211460775796e-22,    2.0567222239462782e-23,    1.7249423131659691e-24,
    1.4027379407630126e-25,    1.1070769000634073e-26,    8.4869576004508737e-28,    6.3248804560575921e-29,
    4.5857743682287526e-30,    3.2370514691062798e-31,    2.2261958774412805e-32,    1.4925857020216937e-33,
    9.7622221835024827e-35,    6.2323244583451431e-36,    3.8858836836191228e-37,    2.3675715720606491e-38,
    1.4103105249264852e-39,    8.2174984054386171e-41,    4.6857976804060876e-42,    2.6160355104669949e-43,
    1.4305697429308103e-44,    7.6658664014204257e-46,    4.0269351476862309e-47,    2.0745162392051980e-48,
    1.0484534187448141e-49,    5.2002811562226813e-51,    2.5322087526578095e-52,    1.2109087807248942e-53,
    5.6885522731617898e-55,    2.6260610495512578e-56,    1.1916545748356247e-57,    5.3169657962190562e-59
},
// x = 5.3300
{
    9.3806429173144135e-02,    7.6211126329080253e-02,    5.0910860882855076e-02,    2.8452345013081110e-02,
    1.3543803642410835e-02,    5.5828829527251097e-03,    2.0219063666179278e-03,    6.5140400975455422e-04,
    1.8868682697096445e-04,    4.9588614162359892e-05,    1.1916907818086814e-05,    2.6364445226181805e-06,
    5.4013032836483904e-07,    1.0300020570993095e-07,    1.8365684055620123e-08,    3.0743449945493958e-09,
    4.8487827118650284e-10,    7.2284403713635582e-11,    1.0215207400903460e-11,    1.3720821689023850e-12,
    1.7558177478844771e-13,    2.1453132068171763e-14,    2.5077262084503030e-15,    2.8095957656508792e-16,
    3.0221499527387616e-17,    3.1259036116183241e-18,    3.1135239933235037e-19,    2.9904143585572113e-20,
    2.7730565168783429e-21,    2.4856732627276702e-22,    2.1560785153529503e-23,    1.8116237652060351e-24,
    1.4759618392721462e-25,    1.1670302679027346e-26,    8.9631910642054090e-28,    6.6922126866506565e-29,
    4.8611312757069019e-30,    3.4378106414178107e-31,    2.3686658327248973e-32,    1.5910657931560194e-33,
    1.0425726546181293e-34,    6.6683255374745909e-36,    4.1654883697409861e-37,    2.5426636386712318e-38,
    1.5174362904266693e-39,    8.8582006229479538e-41,    5.0605743136038137e-42,    2.8305479136420459e-43,
    1.5507675780279518e-44,    8.3254920902394797e-46,    4.3816175519508114e-47,    2.2614552771710503e-48,
    1.1450693627769971e-49,    5.6901159780913637e-51,    2.7759112652226871e-52,    1.3299319380804938e-53,
    6.2593866685637864e-55,    2.8949895857309284e-56,    1.3161483590869086e-57,    5.8834316377147334e-59
},
// x = 5.3400
{
    9.3630805208251885e-02,    7.6101253741513677e-02,    5.0877291870322854e-02,    2.8463340005256327e-02,
    1.3565797481410055e-02,    5.5996363849023017e-03,    2.0309659769296570e-03,    6.5533719012972856e-04,
    1.9013117319446649e-04,    5.0050683705396860e-05,    1.2048216190245416e-05,    2.6700582381395889e-06,
    5.4796535181647430e-07,    1.0467737757557139e-07,    1.8697712389427726e-08,    3.1354938127616191e-09,
    4.9540748388274867e-10,    7.3986889890699310e-11,    1.0474684973670795e-11,    1.4094846424185585e-12,
    1.8069601218694089e-13,    2.2118256713581262e-14,    2.5901996993014706e-15,    2.9073115767000227e-16,
    3.1329959509315393e-17,    3.2465104871456948e-18,    3.2396047477785196e-19,    3.1172441597724243e-20,
    2.8960013930518493e-21,    2.6006717750787927e-22,    2.2599993620224201e-23,    1.9024563779772561e-24,
    1.5528354296442080e-25,    1.2300892455281450e-26,    9.4650279515905977e-28,    7.0800158282368894e-29,
    5.1523760489121430e-30,    3.6505499199678887e-31,    2.5199195024664376e-32,    1.6958128021901231e-33,
    1.1132772634606812e-34,    7.1338070531347266e-36,    4.4645567000405735e-37,    2.7302942583252834e-38,
    1.6324481420729047e-39,    9.5473548704430703e-41,    5.4644469224934924e-42,    3.0621466100468729e-43,
    1.6807822222837779e-44,    9.0403261231571498e-46,    4.7667050692500182e-47,    2.4647995923966533e-48,
    1.2503608260221993e-49,    6.2249344878336073e-51,    3.0424923177115623e-52,    1.4603726961693368e-53,
    6.8861515168750274e-55,    3.1908208755564880e-56,    1.4533526444390441e-57,    6.5089017855492718e-59
},
// x = 5.3500
{
    9.3455836921690350e-02,    7.5991670195750691e-02,    5.0843685410054441e-02,    2.8474207195699819e-02,
    1.3587713378297681e-02,    5.6163716060401740e-03,    2.0400334406449872e-03,    6.5928100260375635e-04,
    1.9158203147557687e-04,    5.0515668943044661e-05,    1.2180590369437043e-05,    2.7040058106749408e-06,
    5.5589249176907470e-07,    1.0637734446431285e-07,    1.9034865500579784e-08,    3.1976996948149719e-09,
    5.0613829697901827e-10,    7.5725152701400045e-11,    1.0740101736214483e-11,    1.4478135911316776e-12,
    1.8594658590879715e-13,    2.2802372017531080e-14,    2.6751846463977682e-15,    3.0081891698913124e-16,
    3.2476403689511471e-17,    3.3714813281850609e-18,    3.3704897223332866e-19,    3.2491509798812826e-20,
    3.0241051427296781e-21,    2.7207182954336565e-22,    2.3686835615926654e-23,    1.9976290627064894e-24,
    1.6335319527079351e-25,    1.2964073435167192e-26,    9.9937841916703122e-28,    7.4893823480250483e-29,
    5.4603879338012215e-30,    3.8759559847671525e-31,    2.6804777585590673e-32,    1.8072107962514278e-33,
    1.1886115496211521e-34,    7.6306930866928429e-36,    4.7843978538519606e-37,    2.9313318890663361e-38,
    1.7559080977731594e-39,    1.0288511462388364e-40,    5.8996060398122790e-42,    3.3121524023096364e-43,
    1.8213923561520476e-44,    9.8148593797562222e-46,    5.1847326083733967e-47,    2.6859502266255058e-48,
    1.3650863655134508e-49,    6.8087615991079273e-51,    3.3340456918646074e-52,    1.6032993252938875e-53,
    7.5741945258881434e-55,    3.5161824427457331e-56,    1.6045349892645208e-57,    7.1993821111743649e-59
},
// x = 5.3600
{
    9.3281520660612655e-02,    7.5882374738526756e-02,    5.0810042262183504e-02,    2.8484947255146650e-02,
    1.3609551443894962e-02,    5.6330884874125709e-03,    2.0491086525631940e-03,    6.6323541216601971e-04,
    1.9303940209858570e-04,    5.0983577151849482e-05,    1.2314035328969882e-05,    2.7382894823779252e-06,
    5.6391255010938235e-07,    1.0810034940506033e-07,    1.9377207957026351e-08,    3.2609779957758899e-09,
    5.1707402623296998e-10,    7.7499849192305043e-11,    1.1011578148888803e-11,    1.4870895824382771e-12,
    1.9133678413268592e-13,    2.3505972468105235e-14,    2.7627512728864262e-15,    3.1123230394681430e-16,
    3.3662040516972719e-17,    3.5009633998621856e-18,    3.5063503320938983e-19,    3.3863258052918724e-20,
    3.1575718454852049e-21,    2.8460223339323030e-22,    2.4823380895542471e-23,    2.0973388730641462e-24,
    1.7182324572133515e-25,    1.3661452936015580e-26,    1.0550840211403466e-27,    7.9214604171576489e-29,
    5.7860927481959534e-30,    4.1147532487780487e-31,    2.8508911286371787e-32,    1.9256664728371716e-33,
    1.2688678000302920e-34,    8.1610287615429052e-36,    5.1264059854025739e-37,    3.1467030960463893e-38,
    1.8884169058622267e-39,    1.1085472506497062e-40,    6.3684019606738060e-42,    3.5819850850350732e-43,
    1.9734365928803405e-44,    1.0653937471772642e-45,    5.6384404447223503e-47,    2.9264245709999214e-48,
    1.4900690748622908e-49,    7.4459728555069295e-51,    3.6528518413178007e-52,    1.7598775215139147e-53,
    8.3293620632405371e-55,    3.8739522113378784e-56,    1.7710863541730249e-57,    7.9614755497475467e-59
},
// x = 5.3700
{
    9.3107852799045307e-02,    7.5773366419009558e-02,    5.0776363179486896e-02,    2.8495560851517113e-02,
    1.3631311790172221e-02,    5.6497869015078037e-03,    2.0581915077540055e-03,    6.6720038366756599e-04,
    1.9450328521890460e-04,    5.1454415563026209e-05,    1.2448556038737425e-05,    2.7729115009245296e-06,
    5.7202633272920029e-07,    1.0984663719454768e-07,    1.9724804935384899e-08,    3.3253442474039360e-09,
    5.2821803230817437e-10,    7.9311646627444220e-11,    1.1289236784795037e-11,    1.5273335850948079e-12,
    1.9686996567073478e-13,    2.4229564145064371e-14,    2.8529715854892054e-15,    3.2198102643976450e-16,
    3.4888113856255631e-17,    3.6351085707653704e-18,    3.6473636853981210e-19,    3.5289663389198022e-20,
    3.2966131569646380e-21,    2.9768015884713329e-22,    2.6011784156282919e-23,    2.2017913362844948e-24,
    1.8071261328044781e-25,    1.4394713709242824e-26,    1.1137643941779029e-27,    8.3774566174804459e-29,
    6.1304652372860600e-30,    4.3677058413727933e-31,    3.0317414139700426e-32,    2.0516104395264585e-33,
    1.3543560662962853e-34,    8.7269875711689749e-36,    5.4920655383241504e-37,    3.3773963020875860e-38,
    2.0306166226947233e-39,    1.1942309167094197e-40,    6.8733560176807269e-42,    3.8731706306345917e-43,
    2.1378179500352219e-44,    1.1562787926648459e-45,    6.1307903737746809e-47,    3.1878657537793517e-48,
    1.6262019225830677e-49,    8.1413241476541018e-51,    4.0013940933304054e-52,    1.9313790614731959e-53,
    9.1580444752001484e-55,    4.2672817755840837e-56,    1.9545328232371004e-57,    8.8024400395657171e-59
},
// x = 5.3800
{
    9.2934829737645122e-02,    7.5664644288814906e-02,    5.0742648907450928e-02,    2.8506048649919975e-02,
    1.3652994530231645e-02,    5.6664667220231698e-03,    2.0672819015597373e-03,    6.7117588182306139e-04,
    1.9597368086346931e-04,    5.1928191362284751e-05,    1.2584157465065947e-05,    2.8078741194622636e-06,
    5.8023464952095309e-07,    1.1161645440207376e-07,    2.0077722224300079e-08,    3.3908141595640553e-09,
    5.3957372123582588e-10,    8.1161222615678506e-11,    1.1573202360594233e-11,    1.5685669759040593e-12,
    2.0254956128971537e-13,    2.4973664960131547e-14,    2.9459194150579159e-15,    3.3307505723077808e-16,
    3.6155903934017580e-17,    3.7740734451159859e-18,    3.7937127585488671e-19,    3.6772772196097850e-20,
    3.4414485713586918e-21,    3.1132822445372692e-22,    2.7254288315959910e-23,    2.3112007969323057e-24,
    1.9004106563735714e-25,    1.5165617298809679e-26,    1.1755713956752644e-27,    8.8586387732870415e-29,
    6.4945315425880831e-30,    4.6356196912864099e-31,    3.2236433917443505e-32,    2.1854985629996298e-33,
    1.4454052023703959e-34,    9.3308791334202147e-36,    5.8829568799174062e-37,    3.6244657714276307e-38,
    2.1831933556738803e-39,    1.2863380071211501e-40,    7.4171726250952053e-42,    4.1873488797853645e-43,
    2.3155086440749763e-44,    1.2547049389961331e-45,    6.6649830969628397e-47,    3.4720527633318056e-48,
    1.7744535193769326e-49,    8.8999838786849979e-51,    4.3823762183911771e-52,    2.1191912056037972e-53,
    1.0067225416802686e-54,    4.6996217772086515e-56,    2.1565484107615584e-57,    9.7302519465244708e-59
},
// x = 5.3900
{
    9.2762447903458639e-02,    7.5556207402022138e-02,    5.0708900184336846e-02,    2.8516411312655872e-02,
    1.3674599778290266e-02,    5.6831278238595080e-03,    2.0763797295973905e-03,    6.7516187121273951e-04,
    1.9745058893114127e-04,    5.2404911689658969e-05,    1.2720844570562285e-05,    2.8431795965591254e-06,
    5.8853831437307294e-07,    1.1341004937418591e-07,    2.0436026227800154e-08,    3.4574036216433292e-09,
    5.5114454487934123e-10,    8.3049265239197830e-11,    1.1863601767667133e-11,    1.6108115464920568e-12,
    2.0837907505299367e-13,    2.5738804901566192e-14,    3.0416704579386180e-15,    3.4452464048313370e-16,
    3.7466728308322245e-17,    3.9180194983859867e-18,    3.9455865754574430e-19,    3.8314702481637256e-20,
    3.5923056923029027e-21,    3.2556992852677345e-22,    2.8553227909650870e-23,    2.4257907737671072e-24,
    1.9982925523013308e-25,    1.5976007541013185e-26,    1.2406642750525117e-27,    9.3663389135033358e-29,
    6.8793717895490824e-30,    4.9193447138146344e-31,    3.4272466059335812e-32,    2.3278133909517583e-33,
    1.5423639603545910e-34,    9.9751573947666239e-36,    6.3007622736452037e-37,    3.8890358405909264e-38,
    2.3468801819405113e-39,    1.3853350930340891e-40,    8.0027521426982247e-42,    4.5262817707596966e-43,
    2.5075552305431447e-44,    1.3612802991887935e-45,    7.2444769310743410e-47,    3.7809113622468652e-48,
    1.9358743478293835e-49,    9.7275677740405833e-51,    4.7987414799529388e-52,    2.3248269126478981e-53,
    1.1064535538900964e-54,    5.1747495758954078e-56,    2.3789690515589624e-57,    1.0753675479848382e-58
},
// x = 5.4000
{
    9.2590703749684145e-02,    7.5448054815189139e-02,    5.0675117741245716e-02,    2.8526649499220891e-02,
    1.3696127649663105e-02,    5.6997700831157138e-03,    2.0854848877607255e-03,    6.7915831628433869e-04,
    1.9893400919311779e-04,    5.2884583639338929e-05,    1.2858622313962251e-05,    2.8788301961523483e-06,
    5.9693814516521562e-07,    1.1522767223931323e-07,    2.0799783968649701e-08,    3.5251287039722084e-09,
    5.6293400140184519e-10,    8.4976473183154215e-11,    1.2160564103623797e-11,    1.6540895101763489e-12,
    2.1436208568350246e-13,    2.6525526283089737e-14,    3.1403023181581642e-15,    3.5634029843837542e-16,
    3.8821942861192559e-17,    4.0671132164428699e-18,    4.1031803923213245e-19,    3.9917646201568313e-20,
    3.7494205124553484e-21,    3.4042968120630302e-22,    2.9911032608706746e-23,    2.5457943301711044e-24,
    2.1009875671051451e-25,    1.6827814211208387e-26,    1.3092100158919602e-27,    9.9019563700012211e-29,
    7.2861227992077521e-30,    5.2197771072212990e-31,    3.6432372511505426e-32,    2.4790656506638192e-33,
    1.6456021475694985e-34,    1.0662429309572048e-35,    6.7472722093515488e-37,    4.1723054111271820e-38,
    2.5224602535537551e-39,    1.4917215455251761e-40,    8.6332046131804021e-42,    4.8918621441072917e-43,
    2.7150841139795314e-44,    1.4766606032949440e-45,    7.8730079388010348e-47,    4.1165258525279946e-48,
    2.1116034905046594e-49,    1.0630176546713559e-50,    5.2536932854898984e-52,    2.5499359335810478e-53,
    1.2158310906635147e-54,    5.6967994148034380e-56,    2.6238078813011906e-57,    1.1882338650858553e-58
},
// x = 5.4100
{
    9.2419593755436250e-02,    7.5340185587366659e-02,    5.0641302302183028e-02,    2.8536763866310259e-02,
    1.3717578260746455e-02,    5.7163933770832528e-03,    2.0945972722222813e-03,    6.8316518135503701e-04,
    2.0042394129334294e-04,    5.3367214259505123e-05,    1.2997495649979258e-05,    2.9148281874968859e-06,
    6.0543496376329468e-07,    1.1706957491234581e-07,    2.1169063091698195e-08,    3.5940056592500849e-09,
    5.7494563573653202e-10,    8.6943555866431641e-11,    1.2464220704164937e-11,    1.6984235089265068e-12,
    2.2050224794798564e-13,    2.7334383997225212e-14,    3.2418945504468679e-15,    3.6853283824019009e-16,
    4.0222942814902116e-17,    4.2215262383039716e-18,    4.2666958874638915e-19,    4.1583871657274663e-20,
    3.9130377020069628e-21,    3.5593283760799213e-22,    3.1330230866198213e-23,    2.6714544586221266e-24,
    2.2087210590356687e-25,    1.7723056823264376e-26,    1.3813836930838895e-27,    1.0466961017964696e-28,
    7.7159809295590727e-30,    5.5378617635438033e-31,    3.8723401540866775e-32,    2.6397958281801800e-33,
    1.7555118481608029e-34,    1.1395464020753272e-35,    7.2243921117804505e-37,    4.4755527198031931e-38,
    2.7107701006371124e-39,    1.6060317646020627e-40,    9.3118644303152077e-42,    5.2861231615561545e-43,
    2.9393074532680355e-44,    1.6015528155055600e-45,    8.5546115850335697e-47,    4.4811517562237684e-48,
    2.3028758951351159e-49,    1.1614436645447523e-50,    5.7507175697084710e-52,    2.7963168585729324e-53,
    1.3357656555137603e-54,    6.2702953001971858e-56,    2.8932719228608846e-57,    1.3126816376051554e-58
},
// x = 5.4200
{
    9.2249114425512785e-02,    7.5232598780112167e-02,    5.0607454584122309e-02,    2.8546755067822224e-02,
    1.3738951729001331e-02,    5.7329975842406768e-03,    2.1037167794353833e-03,    6.8718243061337324e-04,
    2.0192038474892602e-04,    5.3852810552166644e-05,    1.3137469529153660e-05,    2.9511758451136978e-06,
    6.1402959601435135e-07,    1.1893601109916382e-07,    2.1543931867224885e-08,    3.6640509239752960e-09,
    5.8718304005992089e-10,    8.8951233573561552e-11,    1.2774705175298348e-11,    1.7438366204178647e-12,
    2.2680329406279328e-13,    2.8165945773117560e-14,    3.3465287041115618e-15,    3.8111335890718421e-16,
    4.1671163772508840e-17,    4.3814355025841728e-18,    4.4363413564668305e-19,    4.3315725965316443e-20,
    4.0834109063870280e-21,    3.7210573209494061e-22,    3.2813453693025475e-23,    2.8030244797091078e-24,
    2.3217284031811735e-25,    1.8663848587776613e-26,    1.4573688456019515e-27,    1.1062896664476007e-28,
    8.1702050525126237e-30,    5.8745947992199454e-31,    4.1153208573590842e-32,    2.8105758322352456e-33,
    1.8725087126898546e-34,    1.2177202569596233e-35,    7.7341494491039871e-37,    4.8001404027245991e-38,
    2.9127031446422936e-39,    1.7288375544448159e-40,    1.0042305998783247e-41,    5.7112483805320259e-43,
    3.1815294898693986e-44,    1.7367190175357966e-45,    9.2936460309498103e-47,    4.8772294805732364e-48,
    2.5110302185174319e-49,    1.2687544330978865e-50,    6.2936070511237600e-52,    3.0659301966143078e-53,
    1.4672515622136896e-54,    6.9001868329971565e-56,    3.1897803046980500e-57,    1.4498721379598602e-58
},
// x = 5.4300
{
    9.2079262290164332e-02,    7.5125293457503392e-02,    5.0573575297068532e-02,    2.8556623754861999e-02,
    1.3760248172937049e-02,    5.7495825842481052e-03,    2.1128433061360988e-03,    6.9121002812116473e-04,
    2.0342333895056051e-04,    5.4341379473001037e-05,    1.3278548897702351e-05,    2.9878754487377747e-06,
    6.2272287174124334e-07,    1.2082723630111167e-07,    2.1924459194279019e-08,    3.7352811198793532e-09,
    5.9964985426797645e-10,    9.1000237587783167e-11,    1.3092153425911871e-11,    1.7903523651792793e-12,
    2.3326903512146994e-13,    2.9020792438897060e-14,    3.4542883677730621e-15,    3.9409325845734628e-16,
    4.3168082783137951e-17,    4.5470233977221081e-18,    4.6123319127282943e-19,    4.5115637600569540e-20,
    4.2608030534327003e-21,    3.8897571370694351e-22,    3.4363438569041264e-23,    2.9407684562035011e-24,
    2.4402554126582263e-25,    1.9652400535275506e-26,    1.5373578655538344e-27,    1.1691384591739694e-28,
    8.6501196725860493e-30,    6.2310262112020299e-31,    4.3729878108116905e-32,    2.9920107472744144e-33,
    1.9970333193295952e-34,    1.3010768163968942e-35,    8.2787012643607988e-37,    5.1475208708023421e-38,
    3.1292134345996817e-39,    1.8607506541352613e-40,    1.0828360450349813e-41,    6.1695825283870281e-43,
    3.4431533282263838e-44,    1.8829805772676553e-45,    1.0094817185907383e-46,    5.3073990417884375e-48,
    2.7375172938525318e-49,    1.3857313344314113e-50,    6.8864875143910737e-52,    3.3609125739031975e-53,
    1.6113744533712439e-54,    7.5918882503307043e-56,    3.5159841483503804e-57,    1.6010803608894869e-58
},
// x = 5.4400
{
    9.1910033904866073e-02,    7.5018268686151163e-02,    5.0539665144120956e-02,    2.8566370575745903e-02,
    1.3781467712094970e-02,    5.7661482579417231e-03,    2.1219767493451626e-03,    6.9524793781541546e-04,
    2.0493280316294981e-04,    5.4832927931197863e-05,    1.3420738697369016e-05,    3.0249292832659680e-06,
    6.3151562473716911e-07,    1.2274350781942048e-07,    2.2310714604015838e-08,    3.8077130553654942e-09,
    6.1234976645511756e-10,    9.3091310325257128e-11,    1.3416703700706258e-11,    1.8379947138359090e-12,
    2.3990336254441622e-13,    2.9899518188653904e-14,    3.5652592149826593e-15,    4.0748424118707798e-16,
    4.4715219432536144e-17,    4.7184779160723001e-18,    4.7948896935833798e-19,    4.6986119014962397e-20,
    4.4454866702991859e-21,    4.0657118278342343e-22,    3.5983033493676068e-23,    3.0849616227172765e-24,
    2.5645587764877953e-25,    2.0691025810904945e-26,    1.6215524041781226e-27,    1.2354127261626828e-28,
    9.1571181937400239e-30,    6.6082626644794124e-31,    4.6461946755544265e-32,    3.1847406801273949e-33,
    2.1295526104714477e-34,    1.3899477035719408e-35,    8.8603421539646132e-37,    5.5192420149701663e-38,
    3.3613196199826022e-39,    2.0024254336863777e-40,    1.1674133485169029e-41,    6.6636430232907229e-43,
    3.7256881995890628e-44,    2.0412226229345543e-45,    1.0963205645664425e-46,    5.7745159270045781e-48,
    2.9839092696199536e-49,    1.5132226451358869e-50,    7.5338462828216003e-52,    3.6835921440595085e-53,
    1.7693194759006172e-54,    8.3513209571205745e-56,    3.8747882740256508e-57,    1.7677058940408267e-58
},
// x = 5.4500
{
    9.1741425850092123e-02,    7.4911523535212063e-02,    5.0505724821535022e-02,    2.8575996176005612e-02,
    1.3802610467032404e-02,    5.7826944873282547e-03,    2.1311170063698696e-03,    6.9929612351021721e-04,
    2.0644877652523401e-04,    5.5327462789304133e-05,    1.3564043865274647e-05,    3.0623396387045564e-06,
    6.4040869276000616e-07,    1.2468508475957528e-07,    2.2702768263027600e-08,    3.8813637269513471e-09,
    6.2528651339608987e-10,    9.5225205470433494e-11,    1.3748496613489774e-11,    1.8867880944478712e-12,
    2.4671024955088092e-13,    3.0802730854088607e-14,    3.6795290507321357e-15,    4.2129832510768683e-16,
    4.6314136959423575e-17,    4.8959928119519556e-18,    4.9842440721261586e-19,    4.8929769333856517e-20,
    4.6377442103941145e-21,    4.2492162881717248e-22,    3.7675201180687553e-23,    3.2358908314951418e-24,
    2.6949065147762877e-25,    2.1782144147278931e-26,    1.7101637954840208e-27,    1.3052912188491952e-28,
    9.6926663410334206e-30,    7.0074704171947223e-31,    4.9358427462704599e-32,    3.3894427051136191e-33,
    2.2705614087416095e-34,    1.4846849919668090e-35,    9.4815127187633227e-37,    5.9169532606027286e-38,
    3.6101091746123448e-39,    2.1545617657797463e-40,    1.2584024411300474e-41,    7.1961322917753991e-43,
    4.0307572425927039e-44,    2.2123988445265633e-45,    1.1904295654544871e-46,    6.2816681797148911e-48,
    3.2519094716732750e-49,    1.6521491170047367e-50,    8.2405630584698902e-52,    4.0365053107601008e-53,
    1.9423801692435190e-54,    9.1849598515192615e-56,    4.2693748862443544e-57,    1.9512848022209259e-58
},
// x = 5.4600
{
    9.1573434731092099e-02,    7.4805057076400200e-02,    5.0471755018784287e-02,    2.8585501198392579e-02,
    1.3823676559306637e-02,    5.7992211555794393e-03,    2.1402639748059346e-03,    7.0335454889864193e-04,
    2.0797125805142384e-04,    5.5824990863073530e-05,    1.3708469333768780e-05,    3.1001088101166476e-06,
    6.4940291752649395e-07,    1.2665222803563104e-07,    2.3100690976670228e-08,    3.9562503207158645e-09,
    6.3846388103072433e-10,    9.7402688112582340e-11,    1.4087675180837723e-11,    1.9367573999457813e-12,
    2.5369375265356612e-13,    3.1731052180907617e-14,    3.7971878588723258e-15,    4.3554784954231512e-16,
    4.7966443398184799e-17,    5.0797677637334648e-18,    5.1806318748760464e-19,    5.0949277132172649e-20,
    4.8378683906275682e-21,    4.4405766957716410e-22,    3.9443023401802028e-23,    3.3938550149068704e-24,
    2.8315784518428941e-25,    2.2928286522473611e-26,    1.8034134982578041e-27,    1.3789615987500036e-28,
    1.0258305744063858e-29,    7.4298783898161045e-31,    5.2428834975803097e-32,    3.6068329125924814e-33,
    2.4205840166279954e-34,    1.5856624188312731e-35,    1.0144808514522518e-36,    6.3424119916861472e-38,
    3.8767428868771132e-39,    2.3179080842502063e-40,    1.3562746460101834e-41,    7.7699509361581758e-43,
    4.3601058361417338e-44,    2.3975366455827844e-45,    1.2924006238884047e-46,    6.8321948002037868e-48,
    3.5433630441009835e-49,    1.8035100009586892e-50,    9.0119433211339862e-52,    4.4224148715015829e-53,
    2.1319681268136334e-54,    1.0099883773733608e-55,    4.7032294155274975e-57,    2.1535026174690574e-58
},
// x = 5.4700
{
    9.1406057177670155e-02,    7.4698868383998931e-02,    5.0437756418621367e-02,    2.8594886282882488e-02,
    1.3844666111459139e-02,    5.8157281470264952e-03,    2.1494175525393166e-03,    7.0742317755462627e-04,
    2.0950024663083524e-04,    5.6325518921317429e-05,    1.3854020030281051e-05,    3.1382390975693160e-06,
    6.5849914470623213e-07,    1.2864520037447208e-07,    2.3504554192384421e-08,    4.0323902137502519e-09,
    6.5188570495155152e-10,    9.9624534883486261e-11,    1.4434384856118587e-11,    1.9879279956639851e-12,
    2.6085801317610082e-13,    3.2685118110029820e-14,    3.9183278504550042e-15,    4.5024548288627432e-16,
    4.9673792748443086e-17,    5.2700085400747320e-18,    5.3842976054338319e-19,    5.3047423292408448e-20,
    5.0461625392763519e-21,    4.6401109153971697e-22,    4.1289705484155391e-23,    3.5591656652234572e-24,
    2.9748667079562524e-25,    2.4132100010362003e-26,    1.9015335571874270e-27,    1.4566208605991440e-28,
    1.0855657689453060e-29,    7.8767813851143739e-31,    5.5683212605199663e-32,    3.8376685662137026e-33,
    2.5801759041308003e-34,    1.6932766678151147e-35,    1.0852989530169622e-36,    6.7974903664521731e-38,
    4.1624596324302670e-39,    2.4932646410238802e-40,    1.4615348460008797e-41,    8.3882118084912160e-43,
    4.7156105225145631e-44,    2.5977426701224925e-45,    1.4028724669466968e-46,    7.4297055591207407e-48,
    3.8602684285320250e-49,    1.9683895576631330e-50,    9.8537544926226019e-52,    4.8443296999466242e-53,
    2.3396234961261151e-54,    1.1103830435635324e-55,    5.1801687073544051e-57,    2.3762085352594609e-58
},
// x = 5.4800
{
    9.1239289843966259e-02,    7.4592956534872024e-02,    5.0403729697138487e-02,    2.8604152066679947e-02,
    1.3865579246999894e-02,    5.8322153471545776e-03,    2.1585776377480090e-03,    7.1150197293484924e-04,
    2.1103574102853017e-04,    5.6829053685759723e-05,    1.4000700877173408e-05,    3.1767328060805780e-06,
    6.6769822391551428e-07,    1.3066426632001905e-07,    2.3914430003011883e-08,    4.1098009756130483e-09,
    6.6555587089429529e-10,    1.0189153409630349e-10,    1.4788773563890104e-11,    2.0403257269725245e-12,
    2.6820725879367066e-13,    3.3665579063674427e-14,    4.0430435130135675e-15,    4.6540423053385832e-16,
    5.1437886172078590e-17,    5.4669271703821007e-18,    5.5954936742765489e-19,    5.5227083946752940e-20,
    5.2629409547695233e-21,    4.8481489166847546e-22,    4.3218580966595878e-23,    3.7321473322803869e-24,
    3.1250762103667580e-25,    2.5396352830765503e-26,    2.0047670838859357e-27,    1.5384757745720040e-28,
    1.1486427049946464e-29,    8.3495434659950354e-31,    5.9132160354723218e-32,    4.0827503743781185e-33,
    2.7499254890724998e-34,    1.8079487245411507e-35,    1.1608990223677279e-36,    7.2841825474120165e-38,
    4.4685814464771820e-39,    2.6814869739224292e-40,    1.5747237956358703e-41,    9.0542550513364626e-43,
    5.0992885611192140e-44,    2.8142087311577815e-45,    1.5225342421761715e-46,    8.0781023294343175e-48,
    4.2047897460002058e-49,    2.1479640930275192e-50,    1.0772265088804674e-51,    5.3055260937443086e-53,
    2.5670263884637458e-54,    1.2205256218698205e-55,    5.7043717661299769e-57,    2.6214309260122902e-58
},
// x = 5.4900
{
    9.1073129408239806e-02,    7.4487320608474239e-02,    5.0369675523827664e-02,    2.8613299184223215e-02,
    1.3886416090391851e-02,    5.8486826425972246e-03,    2.1677441289037896e-03,    7.1559089838059994e-04,
    2.1257773988575830e-04,    5.7335601830893110e-05,    1.4148516791592631e-05,    3.2155922455660966e-06,
    6.7700100871098674e-07,    1.3270969223737910e-07,    2.4330391150105828e-08,    4.1885003697890867e-09,
    6.7947831523121575e-10,    1.0420448588560156e-10,    1.5150991734666895e-11,    2.0939769270086746e-12,
    2.7574580509706821e-13,    3.4673100236398253e-14,    4.1714316607976754e-15,    4.8103744297470236e-16,
    5.3260473218255152e-17,    5.6707421196018046e-18,    5.8144806348430994e-19,    5.7491233505549388e-20,
    5.4885292757097523e-21,    5.0650332058475294e-22,    4.5233116420059232e-23,    3.9131381396501621e-24,
    3.2825252243440099e-25,    2.6723939607164402e-26,    2.1133687586239839e-27,    1.6247433484113479e-28,
    1.2152406398015442e-29,    8.8496014985465984e-31,    6.2786864481839419e-32,    4.3429248816847924e-33,
    2.9304560149358264e-34,    1.9301253090952038e-35,    1.2415930147083870e-36,    7.8046123700144006e-38,
    4.7965189137552575e-39,    2.8834895984931955e-40,    1.6964205870362293e-41,    9.7716641695288782e-43,
    5.5133081560034754e-44,    3.0482181690320203e-45,    1.6521293814587625e-46,    8.7816020495959492e-48,
    4.5792701502338255e-49,    2.3435095596830183e-50,    1.1774287099341442e-51,    5.8095709248829482e-53,
    2.8160092747511224e-54,    1.3413401260415267e-55,    6.2804132798007234e-57,    2.8913942807175699e-58
},
// x = 5.5000
{
    9.0907572572655429e-02,    7.4381959686861759e-02,    5.0335594561639933e-02,    2.8622328267189100e-02,
    1.3907176767035614e-02,    5.8651299211308246e-03,    2.1769169247739637e-03,    7.1968991711963796e-04,
    2.1412624172040549e-04,    5.7845169983839497e-05,    1.4297472685323588e-05,    3.2548197307857598e-06,
    6.8640835658313987e-07,    1.3478174631694337e-07,    2.4752511027236175e-08,    4.2685063551525401e-09,
    6.9365702546732793e-10,    1.0656420234857019e-10,    1.5521192340063138e-11,    2.1489084245090969e-12,
    2.8347805718045587e-13,    3.5708361891154132e-14,    4.3035914859778246e-15,    4.9715882406285799e-16,
    5.5143353077036882e-17,    5.8816784674385533e-18,    6.0415274260665528e-19,    5.9842947774420150e-20,
    5.7232648624538618e-21,    5.2911192717102731e-22,    4.7336916437390661e-23,    4.1024903199670472e-24,
    3.4475459049536595e-25,    2.8117886839994638e-26,    2.2276053536130956e-27,    1.7156513103035231e-28,
    1.2855480312185492e-29,    9.3784688679934486e-31,    6.6659128558066087e-32,    4.6190869864193034e-33,
    3.1224275313428696e-34,    2.0602803896168180e-35,    1.3277125193855500e-36,    8.3610414755126709e-38,
    5.1477768953640246e-39,    3.1002499378157738e-40,    1.8272452796107682e-41,    1.0544283201202460e-42,
    5.9599994030698272e-44,    3.3011526697496239e-45,    1.7924597520504120e-46,    9.5447614388664467e-48,
    4.9862462263323149e-49,    2.5564097686572276e-50,    1.2867221853707779e-51,    6.3603467405955080e-53,
    3.0885704505937860e-54,    1.4738360284634734e-55,    6.9133001701619453e-57,    3.1885377199320198e-58
},
// x = 5.5100
{
    9.0742616063071102e-02,    7.4276872854701845e-02,    5.0301487467044680e-02,    2.8631239944497969e-02,
    1.3927861403254152e-02,    5.8815570716690422e-03,    2.1860959244230367e-03,    7.2379899226804767e-04,
    2.1568124492744322e-04,    5.8357764724212086e-05,    1.4447573464642883e-05,    3.2944175812899935e-06,
    6.9592112894962925e-07,    1.3688069857842724e-07,    2.5180863683288761e-08,    4.3498370874337627e-09,
    7.0809604073946587e-10,    1.0897150768741309e-10,    1.5899530928311890e-11,    2.2051475517434341e-12,
    2.9140851125310476e-13,    3.6772059660439638e-14,    4.4396246108352957e-15,    5.1378243946174186e-16,
    5.7088375862177986e-17,    6.0999680921009368e-18,    6.2769116215115583e-19,    6.2285407162413701e-20,
    5.9674971905836792e-21,    5.5267760465149703e-22,    4.9533728798142264e-23,    4.3005707700668354e-24,
    3.6204848703324768e-25,    2.9581358603843057e-26,    2.3477562787130329e-27,    1.8114386133865289e-28,
    1.3597629884657985e-29,    9.9377393755789673e-31,    7.0761406102217557e-32,    4.9121825904283167e-33,
    3.3265389825438683e-34,    2.1989167813918040e-35,    1.4196099503584321e-36,    8.9558779350562382e-38,
    5.5239606127112853e-39,    3.3328125050709335e-40,    1.9678617040598194e-41,    1.1376235060712316e-42,
    6.4418660059705511e-44,    3.5744995755086202e-45,    1.9443901154700115e-46,    1.0372503594442960e-47,
    5.4284635142495170e-49,    2.7881652588159888e-50,    1.4059109652225499e-51,    6.9620789746472260e-53,
    3.3868886602236924e-54,    1.6191159669414024e-55,    7.6085114349212785e-57,    3.5155352067577183e-58
},
// x = 5.5200
{
    9.0578256628828333e-02,    7.4172059199282478e-02,    5.0267354890087844e-02,    2.8640034842318828e-02,
    1.3948470126277764e-02,    5.8979639842572590e-03,    2.1952810272143838e-03,    7.2791808683208030e-04,
    2.1724274777938296e-04,    5.8873392583980986e-05,    1.4598824030173086e-05,    3.3343881213659193e-06,
    7.0554019114842681e-07,    1.3900682087485655e-07,    2.5615523825758599e-08,    4.4325109206900324e-09,
    7.2279945231820671e-10,    1.1142723835292561e-10,    1.6286165660164089e-11,    2.2627221525503977e-12,
    2.9954175627540961e-13,    3.7864904852609187e-14,    4.5796351409538288e-15,    5.3092272526822075e-16,
    5.9097443923687480e-17,    6.3258498586760905e-18,    6.5209196852797235e-19,    6.4821899983598818e-20,
    6.2215882566075383e-21,    5.7723863819491618e-22,    5.1827449814048525e-23,    4.5077616266264689e-24,
    3.8017037972468013e-25,    3.1117662477158774e-26,    2.4741141504702438e-27,    1.9123559628072698e-28,
    1.4380937439155408e-29,    1.0529091324759654e-30,    7.5106834862407786e-32,    5.2232113880306048e-33,
    3.5435304095527805e-34,    2.3465678360765252e-35,    1.5176598060906019e-36,    9.5916853935320107e-38,
    5.9267821100141521e-39,    3.5822933545425291e-40,    2.1189804518315268e-41,    1.2271941130718880e-42,
    6.9615978128786876e-44,    3.8698597218290309e-45,    2.1088529163577048e-46,    1.1270146609317225e-47,
    5.9088932423540732e-49,    3.0404028752412746e-50,    1.5358683462471942e-51,    7.6193654415627825e-53,
    3.7133389764177724e-54,    1.7783841287267556e-55,    8.3720415703652391e-57,    3.8753176167326273e-58
},
// x = 5.5300
{
    9.0414491042544773e-02,    7.4067517810521261e-02,    5.0233197474450063e-02,    2.8648713584074569e-02,
    1.3969003064229085e-02,    5.9143505500670176e-03,    2.2044721328118733e-03,    7.3204716370998924e-04,
    2.1881074842673224e-04,    5.9392060047340584e-05,    1.4751229276737414e-05,    3.3747336799832887e-06,
    7.1526641243079871e-07,    1.4116038689649596e-07,    2.6056566824036870e-08,    4.5165464087800352e-09,
    7.3777140411263557e-10,    1.1393224318925970e-10,    1.6681257345169386e-11,    2.3216605904772108e-12,
    3.0788247561944535e-13,    3.8987624763420676e-14,    4.7237297194290422e-15,    5.4859449681909844e-16,
    6.1172513190774317e-17,    6.5595698122374637e-18,    6.7738472348485500e-19,    6.7455825854582465e-20,
    6.4859129962411354e-21,    6.0283475408613398e-22,    5.4222129861047149e-23,    4.7244608630096998e-24,
    3.9915800397459274e-25,    3.2730255713404141e-26,    2.6069853854285675e-27,    2.0186663662817365e-28,
    1.5207591468292890e-29,    1.1154291805457434e-30,    7.9709272826218236e-32,    5.5532298009323434e-33,
    3.7741852718471984e-34,    2.5037992259232474e-35,    1.6222600027499606e-36,    1.0271192763269636e-37,
    6.3580671180297337e-39,    3.8498848176574955e-40,    2.2813620618666148e-41,    1.3236142185612465e-42,
    7.5220842297521642e-44,    4.1889558379928493e-45,    2.2868534249595198e-46,    1.2243434359729083e-47,
    6.4307503626030191e-49,    3.3148861115755821e-50,    1.6775427004691420e-51,    8.3372083000591293e-53,
    4.0705100413611379e-54,    1.9529553697309233e-55,    9.2104478881512326e-57,    4.2710968309334038e-58
},
// x = 5.5400
{
    9.0251316099908663e-02,    7.3963247780974239e-02,    5.0199015857503852e-02,    2.8657276790447300e-02,
    1.3989460346108333e-02,    5.9307166613904406e-03,    2.2136691411814664e-03,    7.3618618569385873e-04,
    2.2038524489845641e-04,    5.9913773550580870e-05,    1.4904794093215097e-05,    3.4154565907402522e-06,
    7.2510066595412177e-07,    1.4334167217472303e-07,    2.6504068712691724e-08,    4.6019623068421378e-09,
    7.5301609317796578e-10,    1.1648738357988341e-10,    1.7084969478341727e-11,    2.3819917570234365e-12,
    3.1643544875437122e-13,    4.0140962992891022e-14,    4.8720175821122345e-15,    5.6681295768336332e-16,
    6.3315594545794774e-17,    6.8013813757921823e-18,    7.0359993110138413e-19,    7.0190699190492736e-20,
    6.7608597156259968e-21,    6.2950717051410927e-22,    5.6721979103894710e-23,    4.9510829080480512e-24,
    4.1905072717498120e-25,    3.4422751662893384e-26,    2.7466908186892788e-27,    2.1306457091500947e-28,
    1.6079891800169636e-29,    1.1815201185502227e-30,    8.4583336042079394e-32,    5.9033540664477928e-33,
    4.0193328948447526e-34,    2.6712108281278746e-35,    1.7338332848119699e-36,    1.0997304499400132e-37,
    6.8197623429965008e-39,    4.1368605416580200e-40,    2.4558204171957928e-41,    1.4273920733674209e-42,
    8.1264285693490563e-44,    4.5336415499869527e-45,    2.4794752585369783e-46,    1.3298569621703643e-47,
    6.9975129855785711e-49,    3.6135262755084660e-50,    1.8319637574855492e-51,    9.1210486867188266e-53,
    4.4612227819580402e-54,    2.1442651317042838e-55,    1.0130902066477228e-56,    4.7063920331009034e-58
},
// x = 5.5500
{
    9.0088728619475686e-02,    7.3859248205844025e-02,    5.0164810670370805e-02,    2.8665725079383808e-02,
    1.4009842101778625e-02,    5.9470622116346806e-03,    2.2228719525927750e-03,    7.4033511547142263e-04,
    2.2196623510244224e-04,    6.0438539481960515e-05,    1.5059523362397195e-05,    3.4565591918089482e-06,
    7.3504382877452895e-07,    1.4555095408584499e-07,    2.6958106194742446e-08,    4.6887775727763339e-09,
    7.6853777022599557e-10,    1.1909353359473572e-10,    1.7497468277211843e-11,    2.4437450799901172e-12,
    3.2520555295695954e-13,    4.1325679767533736e-14,    5.0246106139051235e-15,    5.8559370884357186e-16,
    6.5528755229830537e-17,    7.0515455531759642e-18,    7.3076906551085595e-19,    7.3030152802023079e-20,
    7.0468305358523997e-21,    6.5729865002545847e-22,    5.9331373419600298e-23,    5.1880592875090180e-24,
    4.3988961544385212e-25,    3.6198926454894093e-26,    2.8935663487342123e-27,    2.2485833549579086e-28,
    1.7000255004276818e-29,    1.2513777818794208e-30,    8.9744438338697995e-32,    6.2747634866734620e-33,
    4.2798510496756459e-34,    2.8494387146849698e-35,    1.8528287173799707e-36,    1.1773111490419249e-37,
    7.3139432061434920e-39,    4.4445808495444158e-40,    2.6432263647245164e-41,    1.5390724870901001e-42,
    8.7779633991194852e-44,    4.9059110277755001e-45,    2.6878863087513191e-46,    1.4442249687513823e-47,
    7.6129433208261629e-49,    3.9383945410272114e-50,    2.0002493980905109e-51,    9.9768042368410668e-53,
    4.8885507222973310e-54,    2.3538802253702968e-55,    1.1141246304824289e-56,    5.1850584073392221e-58
},
// x = 5.5600
{
    8.9926725442467728e-02,    7.3755518182987700e-02,    5.0130582537977954e-02,    2.8674059066101057e-02,
    1.4030148461951454e-02,    5.9633870953163248e-03,    2.2320804676205981e-03,    7.4449391562787771e-04,
    2.2355371682596475e-04,    6.0966364181582960e-05,    1.5215421960843024e-05,    3.4980438258808807e-06,
    7.4509678183938746e-07,    1.4778851185485927e-07,    2.7418756644926880e-08,    4.7770113687298074e-09,
    7.8434074013840002e-10,    1.2175158013858204e-10,    1.7918922719269095e-11,    2.5069505319351762e-12,
    3.3419776504754165e-13,    4.2542552268053247e-14,    5.1816234061224025e-15,    6.0495275806980350e-16,
    6.7814120280537843e-17,    7.3103311370060176e-18,    7.5892459936752063e-19,    7.5977941596199027e-20,
    7.3442418511636493e-21,    6.8625355369398024e-22,    6.2054860526090770e-23,    5.4358392890277953e-24,
    4.6171750293404685e-25,    3.8062725949912289e-26,    3.0479636095641578e-27,    2.3727827726357845e-28,
    1.7971220047240299e-29,    1.3252082980132119e-30,    9.5208833033327340e-32,    6.6687038466280435e-33,
    4.5566686720940986e-34,    3.0391572534123076e-35,    1.9797232647670542e-36,    1.2601902599365522e-37,
    7.8428220605787536e-39,    4.7744984410324010e-40,    2.8445115723565914e-41,    1.6592393745296480e-42,
    9.4802669552135135e-44,    5.3079093215361632e-45,    2.9133451039409294e-46,    1.5681704665049964e-47,
    8.2811102356317980e-49,    4.2917349558253846e-50,    2.1836129996614325e-51,    1.0910909726534701e-52,
    5.3558420259095765e-54,    2.5835105531419841e-55,    1.2250054482809703e-56,    5.7113184500157723e-58
},
// x = 5.5700
{
    8.9765303432573856e-02,    7.3652056812924541e-02,    5.0096332079113605e-02,    2.8682279363091860e-02,
    1.4050379558172298e-02,    5.9796912080558326e-03,    2.2412945871464183e-03,    7.4866254864767697e-04,
    2.2514768773615647e-04,    6.1497253941275056e-05,    1.5372494758737019e-05,    3.5399128401121224e-06,
    7.5526040997960171e-07,    1.5005462655915490e-07,    2.7886098112961687e-08,    4.8666830625860358e-09,
    8.0042936248284944e-10,    1.2446242310057176e-10,    1.8349504579795275e-11,    2.5716386387360233e-12,
    3.4341716315166191e-13,    4.3792374962570813e-14,    5.3431733149391565e-15,    6.2490652948967717e-16,
    7.0173874002919413e-17,    7.5780149218043350e-18,    7.8810003307723663e-19,    7.9037946383588532e-20,
    7.6535248012276426e-21,    7.1641789705797394e-22,    6.4897166322707487e-23,    5.6948906513029980e-24,
    4.8457906380463481e-25,    4.0018272972432604e-26,    3.2102506712441202e-27,    2.5035621913920669e-28,
    1.8995454209352368e-29,    1.4032286037085498e-30,    1.0099365672379635e-31,    7.0864910097495125e-33,
    4.8507687277087875e-34,    3.2410813260974646e-35,    2.1150234601257633e-36,    1.3487176892977101e-37,
    8.4087569139005279e-39,    5.1281644554354972e-40,    3.0606726384723173e-41,    1.7885184736656874e-42,
    1.0237180694214241e-43,    5.7419434344829380e-45,    3.1572076372035412e-46,    1.7024738656048617e-47,
    9.0064135536191138e-49,    4.6759784773801396e-50,    2.3833713768572615e-51,    1.1930361088553852e-52,
    5.8667434111625216e-54,    2.8350218511387240e-55,    1.3466698757615121e-56,    6.2897961280086082e-58
},
// x = 5.5800
{
    8.9604459475753254e-02,    7.3548863198842934e-02,    5.0062059906482874e-02,    2.8690386580130671e-02,
    1.4070535522806399e-02,    5.9959744465719628e-03,    2.2505142123598774e-03,    7.5284097691633467e-04,
    2.2674814538048310e-04,    6.2031215004468971e-05,    1.5530746619746438e-05,    3.5821685860684017e-06,
    7.6553560190176905e-07,    1.5234958113216115e-07,    2.8360209326796062e-08,    4.9578122294575354e-09,
    8.1680805203197258e-10,    1.2722697550500824e-10,    1.8789388470093311e-11,    2.6378404882604886e-12,
    3.5286892848775240e-13,    4.5075959945460679e-14,    5.5093805209407343e-15,    6.4547187335800286e-16,
    7.2610261473686280e-17,    7.8548819224064542e-18,    8.1832992481006275e-19,    8.2214177794749479e-20,
    7.9751257578726133e-21,    7.4783940787867649e-22,    6.7863201449338281e-23,    5.9657002783828950e-24,
    5.0852088695074809e-25,    4.2069874834755799e-26,    3.3808127699883920e-27,    2.6412552844774931e-28,
    2.0075759273303061e-29,    1.4856669869741423e-30,    1.0711697526353615e-31,    7.5295146995391902e-33,
    5.1631912310668327e-34,    3.4559686700243175e-35,    2.2592671711667575e-36,    1.4432656598252726e-37,
    9.0142606864934107e-39,    5.5072349186180644e-40,    3.2927754696927437e-41,    1.9275802463491693e-42,
    1.1052828058852345e-43,    6.2104941830816910e-45,    3.4209346933215339e-46,    1.8479774022997816e-47,
    9.7936102233729507e-49,    5.0937581166991620e-50,    2.6009533645162528e-51,    1.3042763074227303e-52,
    6.4252260946933376e-54,    3.1104495368027506e-55,    1.4801422071182128e-56,    6.9255541355313137e-58
},
// x = 5.5900
{
    8.9444190480040250e-02,    7.3445936446607296e-02,    5.0027766626762622e-02,    2.8698381324279352e-02,
    1.4090616489024630e-02,    6.0122367086761896e-03,    2.2597392447602140e-03,    7.5702916272220322e-04,
    2.2835508718721751e-04,    6.2568253566085814e-05,    1.5690182400879269e-05,    3.6248134196699180e-06,
    7.7592325018014740e-07,    1.5467366036693262e-07,    2.8841169695857462e-08,    5.0504186531819165e-09,
    8.3348127928511899e-10,    1.3004616366332419e-10,    1.9238751876112216e-11,    2.7055877391467929e-12,
    3.6255834718109578e-13,    4.6394137281869435e-14,    5.6803680897921075e-15,    6.6666607602960337e-16,
    7.5125590079876313e-17,    8.1412255977718752e-18,    8.4964992131356259e-19,    8.5510780308756467e-20,
    8.3095068266924458e-21,    7.8056758577445633e-22,    7.0958068071173861e-23,    6.2487749798934740e-24,
    5.3359155359082594e-25,    4.4222031162943721e-26,    3.5600530689600584e-27,    2.7862118830258536e-28,
    2.1215077996989208e-29,    1.5727636549621265e-30,    1.1359783202330317e-31,    7.9992424765551100e-33,
    5.4950364264948743e-34,    3.6846223494561120e-35,    2.4130254672742479e-36,    1.5442300828006024e-37,
    9.6620110371747498e-39,    5.9134775974666797e-40,    3.5419599438296826e-41,    2.0771429735721607e-42,
    1.1931634538890519e-43,    6.7162288988503281e-45,    3.7060997098351182e-46,    2.0055898969341302e-47,
    1.0647842496739666e-48,    5.5479252746227445e-50,    2.8379090932257468e-51,    1.4256380855190949e-52,
    7.0356139302286405e-54,    3.4120137555227382e-55,    1.6265417078107289e-56,    7.6241345235373655e-58
},
// x = 5.6000
{
    8.9284493375351270e-02,    7.3343275664764562e-02,    4.9993452840655976e-02,    2.8706264199893160e-02,
    1.4110622590789528e-02,    6.0284778932671286e-03,    2.2689695861576697e-03,    7.6122706825825317e-04,
    2.2996851046592066e-04,    6.3108375772422842e-05,    1.5850806952343131e-05,    3.6678497011360903e-06,
    7.8642425124847541e-07,    1.5702715091968281e-07,    2.9329059314290463e-08,    5.1445223278214869e-09,
    8.5045357099295668e-10,    1.3292092732727754e-10,    1.9697775197472132e-11,    2.7749126296938277e-12,
    3.7249081210440580e-13,    4.7747755358000445e-14,    5.8562620340449812e-15,    6.8850687013902765e-16,
    7.7722231092424271e-17,    8.4373480803160213e-18,    8.8209678954618821e-19,    8.8932036396738148e-20,
    8.6571463639390505e-21,    8.1465376378710821e-22,    7.4187066896303456e-23,    6.5446422380879810e-24,
    5.5984171781368471e-25,    4.6479442036283718e-26,    3.7483934510044112e-27,    2.9387987212226986e-28,
    2.2416500882774779e-29,    1.6647713289555989e-30,    1.2045629854801505e-31,    8.4972239203930346e-33,
    5.8474681389898356e-34,    3.9278933639880800e-35,    2.5769045936076737e-36,    1.6520320119288381e-37,
    1.0354860789658353e-38,    6.3487792867070018e-40,    3.8094448759479034e-41,    2.2379760579233709e-42,
    1.2878349113613859e-43,    7.2620150295432627e-45,    4.0143972099872330e-46,    2.1762918673432749e-47,
    1.1574668266567441e-48,    6.0415673618796150e-50,    3.0959200118888763e-51,    1.5580195881626757e-52,
    7.7026139235732869e-54,    3.7421357273513574e-55,    1.7870912048180733e-56,    8.3916029993120122e-58
},
// x = 5.6100
{
    8.9125365113293914e-02,    7.3240879964550282e-02,    4.9959119142946179e-02,    2.8714035808626772e-02,
    1.4130553962841463e-02,    6.0446979003249638e-03,    2.2782051386748704e-03,    7.6543465562383815e-04,
    2.3158841240792269e-04,    6.3651587721042546e-05,    1.6012625117404335e-05,    3.7112797949300588e-06,
    7.9703950539161150e-07,    1.5941034131325368e-07,    2.9823958964187281e-08,    5.2401434591660863e-09,
    8.6772951068486310e-10,    1.3585221984336073e-10,    2.0166641786890690e-11,    2.8458479868624522e-12,
    3.8267182474530382e-13,    4.9137681237237914e-14,    6.0371913761001481e-15,    7.1101244499078674e-16,
    8.0402621275372485e-17,    8.7435604108844422e-18,    9.1570844915031740e-19,    9.2482370783401298e-20,
    9.0185395091282419e-21,    8.5015117193791826e-22,    7.7555704433557397e-23,    6.8538510026235697e-24,
    5.8732419019110928e-25,    4.8847016452075013e-26,    3.9462753445801374e-27,    3.0994002141025317e-28,
    2.3683273256079825e-29,    1.7619558676811686e-30,    1.2771352772196524e-31,    9.0250950267438186e-33,
    6.2217173038561541e-34,    4.1866834010331472e-35,    2.7515480580734491e-36,    1.7671191830952864e-37,
    1.1095848995196267e-38,    6.8151535543465502e-40,    4.0965333065503752e-41,    2.4109035466247921e-42,
    1.3898067167928332e-43,    7.8509347013549579e-45,    4.3476518478406831e-46,    2.3611410233584451e-47,
    1.2580093724464821e-48,    6.5780268008592607e-50,    3.3768097157487905e-51,    1.7023966338442353e-52,
    8.4313493190205644e-54,    4.1034555031943768e-55,    1.9631264344033686e-56,    9.2345972194398396e-58
},
// x = 5.6200
{
    8.8966802666977859e-02,    7.3138748459894390e-02,    4.9924766122550264e-02,    2.8721696749440445e-02,
    1.4150410740684922e-02,    6.0608966309058671e-03,    2.2874458047481756e-03,    7.6965188682645829e-04,
    2.3321479008680943e-04,    6.4197895460665032e-05,    1.6175641732247901e-05,    3.7551060697030926e-06,
    8.0776991673702428e-07,    1.6182352194053172e-07,    3.0325950118811128e-08,    5.3373024662393385e-09,
    8.8531373919914113e-10,    1.3884100830843693e-10,    2.0645537990014302e-11,    2.9184272353890429e-12,
    3.9310699710101692e-13,    5.0564801022194010e-14,    6.2232882123437583e-15,    7.3420145716394073e-16,
    8.3169264531440670e-17,    9.0601827794938045e-18,    9.5052400578514679e-19,    9.6166354829610508e-20,
    9.3941987337980692e-21,    8.8711500283296667e-22,    8.1069700498239155e-23,    7.1769725140007430e-24,
    6.1609402456530791e-25,    5.1329881127967523e-26,    4.1541605842002315e-27,    3.2684192693265353e-28,
    2.5018802666709916e-29,    1.8645969202290417e-30,    1.3539180956080653e-31,    9.5845828300912068e-33,
    6.6190856842123992e-34,    4.4619477400739752e-35,    2.9376388373613449e-36,    1.8899676449163577e-37,
    1.1888212668762370e-38,    7.3147489735630171e-40,    4.4046181320474142e-41,    2.5968078893809377e-42,
    1.4996254979990397e-43,    8.4863003078628019e-45,    4.7078281086175363e-46,    2.5612781699611876e-47,
    1.3670608510734678e-48,    7.1609215143182163e-50,    3.6825556427742539e-51,    1.8598292567442280e-52,
    9.2273954680391871e-54,    4.4988512488034309e-55,    2.1561062124696746e-56,    1.0160379427062927e-57
},
// x = 5.6300
{
    8.8808803030827721e-02,    7.3036880267426676e-02,    4.9890394362571942e-02,    2.8729247618606145e-02,
    1.4170193060574947e-02,    6.0770739871364270e-03,    2.2966914871289960e-03,    7.7387872378350609e-04,
    2.3484764045890913e-04,    6.4747304991062647e-05,    1.6339861625838007e-05,    3.7993308982387992e-06,
    8.1861639324610844e-07,    1.6426698506780279e-07,    3.0835114945811045e-08,    5.4360199828080647e-09,
    9.0321095521601919e-10,    1.4188827372659852e-10,    2.1134653185655990e-11,    2.9926844070121012e-12,
    4.0380205360058556e-13,    5.2030020222755327e-14,    6.4146877784755055e-15,    7.5809304133479585e-16,
    8.6024733584668104e-17,    9.3875447719650627e-18,    9.8658378533986639e-19,    9.9988711039141502e-20,
    9.7846544068678146e-21,    9.2560247937855004e-22,    8.4734995973596960e-23,    7.5146011566287665e-24,
    6.4620860812407843e-25,    5.3933389654501538e-26,    4.3725323067421414e-27,    3.4462781343449169e-28,
    2.6426666626881508e-29,    1.9729886099177088e-30,    1.4351462975396641e-31,    1.0177510263105703e-32,
    7.0409497859327024e-34,    4.7546983166998698e-35,    3.1359017085649131e-36,    2.0210834852309349e-37,
    1.2735399238246485e-38,    7.8498578704791715e-40,    4.7351880988848518e-41,    2.7966339461570089e-42,
    1.6178775884568835e-43,    9.1716711957713252e-45,    5.0970407102385109e-46,    2.7779335485540666e-47,
    1.4853223541019552e-48,    7.7941670140082009e-50,    4.0153017060787668e-51,    2.0314687852160349e-52,
    1.0096818707900127e-53,    4.9314601845685805e-55,    2.3676234980295860e-56,    1.1176893814379430e-57
},
// x = 5.6400
{
    8.8651363220397891e-02,    7.2935274506481934e-02,    4.9856004440354304e-02,    2.8736689009713934e-02,
    1.4189901059503671e-02,    6.0932298722080735e-03,    2.3059420888850864e-03,    7.7811512832400974e-04,
    2.3648696036378432e-04,    6.5299822262957361e-05,    1.6505289619779008e-05,    3.8439566573971938e-06,
    8.2957984670534341e-07,    1.6674102483804884e-07,    3.1351536310428555e-08,    5.5363168588949047e-09,
    9.2142591579345211e-10,    1.4499501116725536e-10,    2.1634179826442560e-11,    3.0686541498129851e-12,
    4.1476283305489851e-13,    5.3534264130212028e-14,    6.6115285160476032e-15,    7.8270682132160789e-16,
    8.8971671700867979e-17,    9.7259856225777240e-18,    1.0239293690481291e-18,    1.0395431769281527e-19,
    1.0190455377058644e-20,    9.6567292466940688e-22,    8.8557760836126030e-23,    7.8673553425115140e-24,
    6.7772775488035686e-25,    5.6663132010949238e-26,    4.6018958850377739e-27,    3.6334192804031527e-28,
    2.7910620700473530e-29,    2.0874402504978883e-30,    1.5210673108671946e-31,    1.0805801264308181e-32,
    7.4887649800548664e-34,    5.0660069548532869e-35,    3.3471057132479156e-36,    2.1610046589583381e-37,
    1.3641079748352772e-38,    8.4229256189815828e-40,    5.0898341839934443e-41,    3.0113932609415992e-42,
    1.7451918223016220e-43,    9.9108715221434627e-45,    5.5175657551632777e-46,    3.0124336478729045e-47,
    1.6135511707443522e-48,    8.4820002105456048e-50,    4.3773719351667672e-51,    2.2185654992813527e-52,
    1.1046218496021782e-53,    5.4047013195413280e-55,    2.5994174261981555e-56,    1.2292829023876975e-57
},
// x = 5.6500
{
    8.8494480272189216e-02,    7.2833930299104732e-02,    4.9821596927531825e-02,    2.8744021513678306e-02,
    1.4209534875187047e-02,    6.1093641903715110e-03,    2.3151975134018048e-03,    7.8236106219036687e-04,
    2.3813274652472485e-04,    6.5855453177920739e-05,    1.6671930528177180e-05,    3.8889857280586277e-06,
    8.4066119271728401e-07,    1.6924593727418593e-07,    3.1875297778696180e-08,    5.6382141622940713e-09,
    9.3996343690570670e-10,    1.4816222992445594e-10,    2.2144313479873797e-11,    3.1463717376717937e-12,
    4.2599529063486539e-13,    5.5078478197550448e-14,    6.8139521402334064e-15,    8.0806292135521780e-16,
    9.2012794446637364e-17,    1.0075854472876154e-17,    1.0626036295251918e-18,    1.0806821361328631e-19,
    1.0612169572847774e-20,    1.0073878341140739e-21,    9.2544402453022065e-23,    8.2358784265785970e-24,
    7.1071380267663598e-25,    5.9524944458012358e-26,    4.8427799002057548e-27,    3.8303063249079266e-28,
    2.9474606958626446e-29,    2.2082770961498016e-30,    1.6119417787689390e-31,    1.1471486146114662e-32,
    7.9640698431723728e-34,    5.3970087761346267e-35,    3.5720667611774922e-36,    2.3103029230431679e-37,
    1.4609162863237115e-38,    9.0365605155499313e-40,    5.4702563855816519e-41,    3.2421686185440752e-42,
    1.8822425197808938e-43,    1.0708009362719222e-44,    5.9718526849585921e-46,    3.2662085182623326e-47,
    1.7525651666199770e-48,    9.2290050747650370e-50,    4.7712852042960581e-51,    2.4224769131940237e-52,
    1.2082773065323224e-53,    5.9223001293777812e-55,    2.8533863934824942e-56,    1.3517686237041128e-57
},
// x = 5.6600
{
    8.8338151243467639e-02,    7.2732846770053955e-02,    4.9787172390082131e-02,    2.8751245718744677e-02,
    1.4229094646051625e-02,    6.1254768469311350e-03,    2.3244576643833420e-03,    7.8661648704006677e-04,
    2.3978499554924322e-04,    6.6414203588276342e-05,    1.6839789157502928e-05,    3.9344204950675368e-06,
    8.5186135069139099e-07,    1.7178202028224161e-07,    3.2406483620626795e-08,    5.7417331800901369e-09,
    9.5882839398472252e-10,    1.5139095367744250e-10,    2.2665252869795520e-11,    3.2258730798393083e-12,
    4.3750549987803357e-13,    5.6663628425990889e-14,    7.0221037088446260e-15,    8.3418197757958109e-16,
    9.5150891487680848e-17,    1.0437510636761005e-17,    1.1026507677495469e-18,    1.1233560306383176e-19,
    1.1050384620439666e-20,    1.0508109498634061e-21,    9.6701574160347119e-23,    8.6208396547180516e-24,
    7.4523171383869307e-25,    6.2524919821402863e-26,    5.0957371542418960e-27,    4.0374249937271185e-28,
    3.1122762817423510e-29,    2.3358411267896907e-30,    1.7080442356719810e-31,    1.2176707235936098e-32,
    8.4684907268364587e-34,    5.7489057954592234e-35,    3.8116503813218283e-36,    2.4695858845170253e-37,
    1.5643809714386475e-38,    9.6935442689750148e-40,    5.8782709497328315e-41,    3.4901189025307830e-42,
    2.0297526757134459e-43,    1.1567497156150022e-44,    6.4625370935655972e-46,    3.5407996253813922e-47,
    1.9032474938681187e-48,    1.0040140290360384e-49,    5.1997711321440084e-51,    2.6446767326878632e-52,
    1.3214288886892620e-53,    6.4883153400417445e-55,    3.1316022850210512e-56,    1.4861853337450624e-57
},
// x = 5.6700
{
    8.8182373212084469e-02,    7.2632023046807187e-02,    4.9752731388376985e-02,    2.8758362210495908e-02,
    1.4248580511221541e-02,    6.1415677482394874e-03,    2.3337224458539305e-03,    7.9088136444740650e-04,
    2.4144370392957491e-04,    6.6976079297005069e-05,    1.7008870306453817e-05,    3.9802633471761040e-06,
    8.6318124383470564e-07,    1.7434957365447518e-07,    3.2945178813394965e-08,    5.8468954201799089e-09,
    9.7802572246426342e-10,    1.5468222065244762e-10,    2.3197199918289953e-11,    3.3071947306261304e-12,
    4.4929965472397846e-13,    5.8290701757856252e-14,    7.2361316926168220e-15,    8.6108514978627662e-16,
    9.8388828427223172e-17,    1.0811323872002366e-17,    1.1441163510114123e-18,    1.1676186078457615e-19,
    1.1505708480250822e-20,    1.0960083376101858e-21,    1.0103618413072454e-22,    9.0229351456005539e-24,
    7.8134917960722899e-25,    6.5669418180826995e-26,    5.3613457244402963e-27,    4.2552841250595409e-28,
    3.2859430274032149e-29,    2.4704918702663265e-30,    1.8096638162043744e-31,    1.2923724803596099e-32,
    9.0037465675274806e-34,    6.1229707128273722e-35,    4.0667746281063198e-36,    2.6394991680330918e-37,
    1.6749449642846370e-38,    1.0396843141862502e-39,    6.3158180597905089e-41,    3.7564842735217290e-42,
    2.1884973642990283e-43,    1.2494073574538258e-44,    6.9924544590146227e-46,    3.8378682819091384e-47,
    2.0665516569450905e-48,    1.0920769047859036e-49,    5.6657872432956778e-51,    2.8867645403509135e-52,
    1.4449254248914866e-53,    7.1071679922174266e-55,    3.4363259408621400e-56,    1.6336685676506447e-57
},
// x = 5.6800
{
    8.8027143276298736e-02,    7.2531458259564183e-02,    4.9718274477233142e-02,    2.8765371571858944e-02,
    1.4267992610505568e-02,    6.1576368016916733e-03,    2.3429917621590147e-03,    7.9515565590519634e-04,
    2.4310886804317926e-04,    6.7541086057653187e-05,    1.7179178765818083e-05,    4.0265166769877432e-06,
    8.7462179914235780e-07,    1.7694889907243647e-07,    3.3491469044508642e-08,    5.9537226127971863e-09,
    9.9756041832681962e-10,    1.5803708378573094e-10,    2.3740359787984854e-11,    3.3903738992099121e-12,
    4.6138407157876748e-13,    5.9960706475852879e-14,    7.4561880467823584e-15,    8.8879413338704928e-16,
    1.0172954868529033e-16,    1.1197674658312486e-17,    1.1870473518507790e-18,    1.2135253716964916e-19,
    1.1978770102415718e-20,    1.1430484658294276e-21,    1.0555540453962324e-22,    9.4428889074173415e-24,
    8.1913672848011690e-25,    6.8965077979388452e-26,    5.6402100612741042e-27,    4.4844167165726138e-28,
    3.4689165558340594e-29,    2.6126072630947359e-30,    1.9171049987138853e-31,    1.3714923288940231e-32,
    9.5716539493098575e-34,    6.5205509114559119e-35,    4.3384131513371647e-36,    2.8207287095716434e-37,
    1.7930796887640210e-38,    1.1149619782947669e-39,    6.7849700171236630e-41,    4.0425916882518246e-42,
    2.3593073744725983e-43,    1.3492826916580376e-44,    7.5646548573566601e-46,    4.1592046984901368e-47,
    2.2435069601660407e-48,    1.1876691140940536e-49,    6.1725374888619062e-51,    3.1504762666807090e-52,
    1.5796897285172970e-53,    7.7836729755209190e-55,    3.7700239664063984e-56,    1.7954594014786090e-57
},
// x = 5.6900
{
    8.7872458554601171e-02,    7.2431151541251002e-02,    4.9683802205962690e-02,    2.8772274383111550e-02,
    1.4287331084384326e-02,    6.1736839157198187e-03,    2.3522655179664016e-03,    7.9943932282645902e-04,
    2.4478048415324202e-04,    6.8109229574242098e-05,    1.7350719318338751e-05,    4.0731828809004460e-06,
    8.8618394738791407e-07,    1.7958030010996441e-07,    3.4045440714971980e-08,    6.0622367120404337e-09,
    1.0174375386532767e-09,    1.6145661088786067e-10,    2.4294940924784357e-11,    3.4754484595617131e-12,
    4.7376519140882427e-13,    6.1674672608849494e-14,    7.6824282839508300e-15,    9.1733117162856078e-16,
    1.0517607541965819e-16,    1.1596954482119377e-17,    1.2314921880082753e-18,    1.2611336358886630e-19,
    1.2470220101835058e-20,    1.1920022875309559e-21,    1.1026668103955924e-22,    9.8814538906895914e-24,
    8.5866783860240666e-25,    7.2418827568953465e-26,    5.9329621314253192e-27,    4.7253810175719305e-28,
    3.6616749217817895e-29,    2.7625845514445608e-30,    2.0306883849592624e-31,    1.4552817844153007e-32,
    1.0174132431868523e-33,    6.9430726730329358e-35,    4.6275984386380403e-36,    3.0140031833752942e-37,
    1.9192868275114399e-38,    1.1955245791495824e-39,    7.2879399435665839e-41,    4.3498607810549358e-42,
    2.5430730908954883e-43,    1.4569220125898445e-44,    8.1824187268658757e-46,    4.5067376980100806e-47,
    2.4352243649035189e-48,    1.2914177537857612e-49,    6.7234922308219615e-51,    3.4376955087957491e-52,
    1.7267248812681516e-53,    8.5230732368531350e-55,    4.1353870008215708e-56,    1.9729140260454949e-57
},
// x = 5.7000
{
    8.7718316185540116e-02,    7.2331102027522992e-02,    4.9649315118422750e-02,    2.8779071221889021e-02,
    1.4306596073997632e-02,    6.1897089997874971e-03,    2.3615436182673755e-03,    8.0373232654611332e-04,
    2.4645854840918310e-04,    6.8680515501181360e-05,    1.7523496738578505e-05,    4.1202643590500081e-06,
    8.9786862311356521e-07,    1.8224408223612669e-07,    3.4607180942438733e-08,    6.1724598974032732e-09,
    1.0376622021753318e-09,    1.6494188480924450e-10,    2.4861155101023959e-11,    3.5624569604925328e-12,
    4.8644958186451883e-13,    6.3433652344240673e-14,    7.9150115483170360e-15,    9.4671906805357870e-16,
    1.0873151348928049e-16,    1.2009566128185236e-17,    1.2775007634125950e-18,    1.3105025785759960e-19,
    1.2980731453300193e-20,    1.2429433245977249e-21,    1.1517774255181118e-22,    1.0339413078342118e-23,
    9.0001905434564624e-25,    7.6037897207555388e-26,    6.2402626077142951e-27,    4.9787616680342923e-28,
    3.8647196654037784e-29,    2.9208412341727285e-30,    2.1507515176507206e-31,    1.5440061205965669e-32,
    1.0813210157232706e-33,    7.3920456213931777e-35,    4.9354252397031401e-36,    3.2200965695512757e-37,
    2.0541001966983016e-38,    1.2817315057435007e-39,    7.8270910376264994e-41,    4.6798101307561190e-42,
    2.7407486366299052e-43,    1.5729117543824468e-44,    8.8492737551287706e-46,    4.8825451403313646e-47,
    2.6429027863326678e-48,    1.4040007613291543e-49,    7.3224098024991539e-51,    3.7504657635277054e-52,
    1.8871210366257440e-53,    9.3310768836843643e-55,    4.5353495666182004e-56,    2.1675141678588835e-57
},
// x = 5.7100
{
    8.7564713327549060e-02,    7.2231308856768081e-02,    4.9614813753064951e-02,    2.8785762663191074e-02,
    1.4325787721131925e-02,    6.2057119643841906e-03,    2.3708259683777946e-03,    8.0803462832265986e-04,
    2.4814305684716493e-04,    6.9254949443184647e-05,    1.7697515792785178e-05,    4.1677635152531409e-06,
    9.0967676462016347e-07,    1.8494055281809849e-07,    3.5176777564356377e-08,    6.2844145753078010e-09,
    1.0582395898306684e-09,    1.6849400360691223e-10,    2.5439217459052184e-11,    3.6514386358210508e-12,
    4.9944393943380357e-13,    6.5238720446980654e-14,    8.1541006912167631e-15,    9.7698119921287827e-16,
    1.1239905146101877e-16,    1.2435923978215915e-17,    1.3251245102286852e-18,    1.3616932985858999e-19,
    1.3511000207240782e-20,    1.2959477547853149e-21,    1.2029661138551895e-22,    1.0817580614270971e-23,
    9.4327010722272052e-25,    7.9829831525470414e-26,    6.5628021077434531e-27,    5.2451708864059948e-28,
    4.0785769130050104e-29,    3.0878160497656294e-30,    2.2776497375902439e-31,    1.6379450913636403e-32,
    1.1491029749130151e-33,    7.8690674064737804e-35,    5.2630541821503757e-36,    3.4398308701764927e-37,
    2.1980877328050925e-38,    1.3739657923371107e-39,    8.4049464184543579e-41,    5.0340639373637108e-42,
    2.9533562945554190e-43,    1.6978813513002783e-44,    9.5690129664937590e-46,    5.2888651078616309e-47,
    2.8678358617269255e-48,    1.5261509239436553e-49,    7.9733597659591714e-51,    4.0910036467219311e-52,
    2.0620627848238466e-53,    1.0213897420796256e-54,    4.9731116337078862e-56,    2.3808784301990792e-57
},
// x = 5.7200
{
    8.7411647158775974e-02,    7.2131771170109368e-02,    4.9580298642984341e-02,    2.8792349279388771e-02,
    1.4344906168207879e-02,    6.2216927210197370e-03,    2.3801124739391520e-03,    8.1234618933984657e-04,
    2.4983400539060294e-04,    6.9832536955187065e-05,    1.7872781238757779e-05,    4.2156827569504326e-06,
    9.2160931395709260e-07,    1.8767002112398057e-07,    3.5754319141100147e-08,    6.3981233806405682e-09,
    1.0791749453208558e-09,    1.7211408071255188e-10,    2.6029346555241122e-11,    3.7424334146635185e-12,
    5.1275509162621935e-13,    6.7090974685374922e-14,    8.3998623480508576e-15,    1.0081415276321862e-15,
    1.1618196366051212e-16,    1.2876454316610563e-17,    1.3744164319914083e-18,    1.4147688731953430e-19,
    1.4061746226655878e-20,    1.3510945014600023e-21,    1.2563161369431552e-22,    1.1316802971671604e-23,
    9.8850404128911093e-25,    8.3802502477166827e-26,    6.9013024831353330e-27,    5.5252497081405654e-28,
    4.3037985268550040e-29,    3.2639700091342079e-30,    2.4117570822390661e-31,    1.7373936889310707e-32,
    1.2209854519579275e-33,    8.3758286409985121e-35,    5.6117155892625840e-36,    3.6740789821595420e-37,
    2.3518535977966016e-38,    1.4726356217276118e-39,    9.0241995935838024e-41,    5.4143591344412256e-42,
    3.1819912256594918e-43,    1.8325062955720251e-44,    1.0345714092525625e-45,    5.7281079057865447e-47,
    3.1114192245664558e-48,    1.6586601949516258e-49,    8.6807479961066680e-51,    4.4617131760552154e-52,
    2.2528371244276930e-53,    1.1178297378126064e-54,    5.4521620422087390e-56,    2.6147746335459629e-57
},
// x = 5.7300
{
    8.7259114876914387e-02,    7.2032488111407694e-02,    4.9545770315967942e-02,    2.8798831640231472e-02,
    1.4363951558268074e-02,    6.2376511822187916e-03,    2.3894030409196106e-03,    8.1666697070833193e-04,
    2.5153138985067978e-04,    7.0413283542266182e-05,    1.8049297825713340e-05,    4.2640244951492089e-06,
    9.3366721691198114e-07,    1.9043279832555643e-07,    3.6339894959097556e-08,    6.5136091782912445e-09,
    1.1004735756719847e-09,    1.7580324510180288e-10,    2.6631764404428321e-11,    3.8354819318469068e-12,
    5.2638999918760175e-13,    6.8991536263717405e-14,    8.6524670165984417e-15,    1.0402246150385788e-15,
    1.2008361226803640e-16,    1.3331595643503310e-17,    1.4254311478498932e-18,    1.4697944175037162e-19,
    1.4633713945802838e-20,    1.4084653261549240e-21,    1.3119139028093393e-22,    1.1837960162433224e-23,
    1.0358073431864009e-24,    8.7964122796921631e-26,    7.2565181613134126e-27,    5.8196692770257627e-28,
    4.5409633061597718e-29,    3.4497874762870525e-30,    2.5534672276199372e-31,    1.8426629398131088e-32,
    1.2972074998023683e-33,    8.9141181029551648e-35,    5.9827135104336258e-36,    3.9237677355541241e-37,
    2.5160404094906267e-38,    1.5781759207427250e-39,    9.6877255885683545e-41,    5.8225529646155430e-42,
    3.4278265034731547e-43,    1.9775114058893104e-44,    1.1183760313614986e-45,    6.2028689344866874e-47,
    3.3751583211345903e-48,    1.8023843402358119e-49,    9.4493437308917030e-51,    4.8652012005677551e-52,
    2.4608420891344738e-53,    1.2231635607968594e-54,    5.9763039400803506e-56,    2.8711332411864720e-57
},
// x = 5.7400
{
    8.7107113699036232e-02,    7.1933458827264068e-02,    4.9511229294542831e-02,    2.8805210312853925e-02,
    1.4382924034964864e-02,    6.2535872615152857e-03,    2.3986975756150179e-03,    8.2099693346733538e-04,
    2.5323520592685921e-04,    7.0997194659564589e-05,    1.8227070294154205e-05,    4.3127911443662440e-06,
    9.4585142300026310e-07,    1.9322919750098808e-07,    3.6933595033942402e-08,    6.6308950646938300e-09,
    1.1221408517980173e-09,    1.7956264146480893e-10,    2.7246696524792486e-11,    3.9306255384462535e-12,
    5.4035575834581476e-13,    7.0941550261862463e-14,    8.9120891367401658e-15,    1.0732556358507845e-15,
    1.2410744946021958e-16,    1.3801798995251705e-17,    1.4782249379483143e-18,    1.5268371454427727e-19,
    1.5227673151232333e-20,    1.4681449240259010e-21,    1.3698490776052672e-22,    1.2381966988944559e-23,
    1.0852700769888715e-24,    9.2323259976506477e-26,    7.6292375418444640e-27,    6.1291321914273154e-28,
    4.7906782413466686e-29,    3.6457772989911449e-30,    2.7031944755451009e-31,    1.9540807406256182e-32,
    1.3780215799039715e-33,    9.4858282175763017e-35,    6.3774299756912882e-36,    4.1898811064806958e-37,
    2.6913316042846493e-38,    1.6910500534113245e-39,    1.0398592778891653e-40,    6.2606310473464796e-42,
    3.6921184851304611e-43,    2.1336743206156993e-44,    1.2087862465751901e-45,    6.7159424955910358e-47,
    3.6606768088548414e-48,    1.9582479393153563e-49,    1.0284308737365915e-50,    5.3042940664290374e-52,
    2.6875960821959442e-53,    1.3381918552002870e-54,    6.5496824044289902e-56,    3.1520619630193269e-57
},
// x = 5.7500
{
    8.6955640861426209e-02,    7.1834682467021405e-02,    4.9476676096023728e-02,    2.8811485861783385e-02,
    1.4401823742548309e-02,    6.2695008734469006e-03,    2.4079959846498877e-03,    8.2533603858628497e-04,
    2.5494544920740567e-04,    7.1584275712215751e-05,    1.8406103375736217e-05,    4.3619851225704097e-06,
    9.5816288545458070e-07,    1.9605953363745251e-07,    3.7535510113499016e-08,    6.7500043693704819e-09,
    1.1441822090668648e-09,    1.8339343037803638e-10,    2.7874371983165854e-11,    4.0279063124473979e-12,
    5.5465960308785498e-13,    7.2942186081823014e-14,    9.1789071716130971e-15,    1.1072603909379561e-15,
    1.2825701959849389e-16,    1.4287528272529727e-17,    1.5328557899694094e-18,    1.5859664324647116e-19,
    1.5844419785772586e-20,    1.5302210222907346e-21,    1.4302147009374532e-22,    1.2949774339697217e-23,
    1.1369860240192481e-24,    9.6888850783976548e-26,    8.0202844494342335e-27,    6.4543739076580628e-28,
    5.0535798239062719e-29,    3.8524739916182404e-30,    2.8613747882479444e-31,    2.0719927355777931e-32,
    1.4636942845409080e-33,    1.0092960833209982e-34,    6.7973294862511919e-36,    4.4734636142995317e-37,
    2.8784539397990859e-38,    1.8117516175725369e-39,    1.1160075466898338e-40,    6.7307159698493422e-42,
    3.9762125408114459e-43,    2.3018292306558312e-44,    1.3063082812709390e-45,    7.2703365973128637e-47,
    3.9697255783741061e-48,    2.1272497673189995e-49,    1.1191228754400529e-50,    5.7820556152472189e-52,
    2.9347479749386580e-53,    1.4637855802547807e-54,    7.1768144290978910e-56,    3.4598616383400651e-57
},
// x = 5.7600
{
    8.6804693619417894e-02,    7.1736158182766491e-02,    4.9442111232560329e-02,    2.8817658848946749e-02,
    1.4420650825854218e-02,    6.2853919335495402e-03,    2.4172981749783565e-03,    8.2968424696644569e-04,
    2.5666211516990362e-04,    7.2174532055272212e-05,    1.8586401793137469e-05,    4.4116088511251740e-06,
    9.7060256121403246e-07,    1.9892412363371443e-07,    3.8145731680995967e-08,    6.8709606564777579e-09,
    1.1666031478691605e-09,    1.8729678847735548e-10,    2.8515023440784895e-11,    4.1273670695359380e-12,
    5.6930890746864535e-13,    7.4994637901482256e-14,    9.4531036902183487e-15,    1.1422653216514525e-15,
    1.3253596146517423e-16,    1.4789260576185238e-17,    1.5893834468674643e-18,    1.6472538799502504e-19,
    1.6484776776079551e-20,    1.5947844817378454e-21,    1.4931073050092670e-22,    1.3542370530113988e-23,
    1.1910528278049161e-24,    1.0167021634323637e-25,    8.4305196457427467e-27,    6.7961642027640398e-28,
    5.3303354141242205e-29,    4.0704389724659379e-30,    3.0284668725864697e-31,    2.1967632376385578e-32,
    1.5545070964141394e-33,    1.0737633306173895e-34,    7.2439637536661068e-36,    4.7756239131886378e-37,
    3.0781801454099202e-38,    1.9408063510117487e-39,    1.1975667248998097e-40,    7.2350764339308551e-42,
    4.2815491646856664e-43,    2.4828708678953602e-44,    1.4114860490516020e-45,    7.8692888291823761e-47,
    4.3041924443367205e-48,    2.3104685860227470e-49,    1.2176147384741610e-50,    6.3018066185198139e-52,
    3.2040880302388188e-53,    1.6008920308238014e-54,    7.8626214760222273e-56,    3.7970435067860621e-57
},
// x = 5.7700
{
    8.6654269247231602e-02,    7.1637885129331377e-02,    4.9407535211184114e-02,    2.8823729833677890e-02,
    1.4439405430292386e-02,    6.3012603583518147e-03,    2.4266040538851127e-03,    8.3404151944254940e-04,
    2.5838519918177852e-04,    7.2767968993635754e-05,    1.8767970259927634e-05,    4.4616647547309960e-06,
    9.8317141091326543e-07,    2.0182328630263897e-07,    3.8764351958108863e-08,    6.9937877263552728e-09,
    1.1894092341897273e-09,    1.9127390863239200e-10,    2.9168887199482261e-11,    4.2290513740135980e-12,
    5.8431118795186694e-13,    7.7100125135513862e-14,    9.7348654515035637e-15,    1.1782975241343225e-15,
    1.3694801054807391e-16,    1.5307486551025795e-17,    1.6478694558181660e-18,    1.7107733813796204e-19,
    1.7149594884386072e-20,    1.6619294013925450e-21,    1.5586270376906984e-22,    1.4160782690073398e-23,
    1.2475721443515070e-24,    1.0667707779473327e-25,    8.8608424022641545e-27,    7.1553086991064041e-28,
    5.6216446691288903e-29,    4.3002618579381095e-30,    3.2049533160812038e-31,    2.3287761954527039e-32,
    1.6507571873862883e-33,    1.1422084910428203e-34,    7.7189767007759932e-36,    5.0975385888171905e-37,
    3.2913317290803332e-38,    2.0787741535569338e-39,    1.2849095221044236e-40,    7.7761369934780867e-42,
    4.6096704919136163e-43,    2.6777587661436760e-44,    1.5249038738158733e-45,    8.5162833810555015e-47,
    4.6661125529335556e-48,    2.5090693741400813e-49,    1.3245602621793672e-50,    6.8671457596492112e-52,
    3.4975597165214008e-53,    1.7505413602095778e-54,    8.6124648038733170e-56,    4.1663479857000951e-57
},
// x = 5.7800
{
    8.6504365037813738e-02,    7.1539862464294598e-02,    4.9372948533854606e-02,    2.8829699372724862e-02,
    1.4458087701834878e-02,    6.3171060653695234e-03,    2.4359135289863066e-03,    8.3840781678440660e-04,
    2.6011469650082146e-04,    7.3364591781990352e-05,    1.8950813480438061e-05,    4.5121552613676208e-06,
    9.9587039887140785e-07,    2.0475734237364112e-07,    3.9391463908032485e-08,    7.1185096170766912e-09,
    1.2126061001817355e-09,    1.9532600012214666e-10,    2.9836203248322033e-11,    4.3330035498429430e-12,
    5.9967410578315606e-13,    7.9259892903599438e-14,    1.0024383489941931e-14,    1.2153847639131766e-15,
    1.4149700137457682e-16,    1.5842710737698543e-17,    1.7083772184131751e-18,    1.7766011903102341e-19,
    1.7839753585097234e-20,    1.7317532264313970e-21,    1.6268777896359874e-22,    1.4806078199645141e-23,
    1.3066497979163369e-24,    1.1191957255831202e-25,    9.3121921365955107e-27,    7.5326504532070131e-28,
    5.9282410337764092e-29,    4.5425618160678011e-30,    3.3913417771479662e-31,    2.4684362081754872e-32,
    1.7527582582854033e-33,    1.2148683588677176e-34,    8.2241097383351743e-36,    5.4404561713683118e-37,
    3.5187819493599056e-38,    2.2262512319319480e-39,    1.3784335072583160e-40,    8.3564884194324709e-42,
    4.9622272477888352e-43,    2.8875218125951101e-44,    1.6471894035964954e-45,    9.2150692863538570e-47,
    5.0576795576606421e-48,    2.7243100292134462e-49,    1.4406666210143121e-50,    7.4819722833436508e-52,
    3.8172724829192975e-53,    1.9138536459897091e-54,    9.4321838048262042e-56,    4.5707650819832752e-57
},
// x = 5.7900
{
    8.6354978302677771e-02,    7.1442089347982365e-02,    4.9338351697505563e-02,    2.8835568020257343e-02,
    1.4476697787004462e-02,    6.3329289731001532e-03,    2.4452265082304217e-03,    8.4278309969851642e-04,
    2.6185060227571617e-04,    7.3964405624738020e-05,    1.9134936149632616e-05,    4.5630828022362896e-06,
    1.0087004930808653e-06,    2.0772661449507582e-07,    4.0027161238542451e-08,    7.2451506060030885e-09,
    1.2361994447435455e-09,    1.9945428881189384e-10,    3.0517215310681571e-11,    4.4392686918216225e-12,
    6.1540546939602948e-13,    8.1475212506041347e-14,    1.0321853202630647e-14,    1.2535554907772966e-15,
    1.4618686989611100e-16,    1.6395451932834544e-17,    1.7709720421279767e-18,    1.8448159902059367e-19,
    1.8556161966896376e-20,    1.8043568594376856e-21,    1.6979673255726683e-22,    1.5479366174597970e-23,
    1.3683959424701317e-24,    1.1740827121998637e-25,    9.7855501145030438e-27,    7.9290716114192442e-28,
    6.2508932969947073e-29,    4.7979889819689653e-30,    3.5881662319884153e-31,    2.6161695904925459e-32,
    1.8608414217927162e-33,    1.2919933062323054e-34,    8.7612073318983682e-36,    5.8057013767583154e-37,
    3.7614589619038621e-38,    2.3838723745469856e-39,    1.4785627123626704e-40,    8.9788987312973057e-42,
    5.3409861567218256e-43,    3.1132631089759361e-44,    1.7790167281104936e-45,    9.9696799749027224e-47,
    5.4812576182999844e-48,    2.9575485757670603e-49,    1.5666986053427998e-50,    8.1505104412385672e-52,
    4.1655155716758945e-53,    2.0920465428961245e-54,    1.0328137598963308e-55,    5.0135565771145975e-57
},
// x = 5.8000
{
    8.6206106371746874e-02,    7.1344564943469163e-02,    4.9303745194090406e-02,    2.8841336327874008e-02,
    1.4495235832863154e-02,    6.3487290010173876e-03,    2.4545428998991397e-03,    8.4716732882966208e-04,
    2.6359291154656517e-04,    7.4567415675936766e-05,    1.9320342952979154e-05,    4.6144498117018538e-06,
    1.0216626651959470e-06,    2.1073142723656415e-07,    4.0671538405045120e-08,    7.3737352113384545e-09,
    1.2601950340982178e-09,    2.0366001733135237e-10,    3.1212170891781041e-11,    4.5478926768870549e-12,
    6.3151323685085674e-13,    8.3747381906859774e-14,    1.0627474437930630e-14,    1.2928388539497581e-15,
    1.5102165592396629e-16,    1.6966243557628848e-17,    1.8357211930919331e-18,    1.9154989661634873e-19,
    1.9299759661040043e-20,    1.8798447750931664e-21,    1.7720074198887376e-22,    1.6181799003286162e-23,
    1.4329252290412642e-24,    1.2315419506510023e-25,    1.0281941220278763e-26,    8.3454951350804025e-28,
    6.5904072163106209e-29,    5.0672259379094170e-30,    3.7959882807069326e-31,    2.7724254901936181e-32,
    1.9753561305278329e-33,    1.3738480318542873e-34,    9.3322228742840533e-36,    6.1946795885202202e-37,
    4.0203491503726193e-38,    2.5523133638097846e-39,    1.5857493360714682e-40,    9.6463249365681380e-42,
    5.7478378404798758e-43,    3.3561651627658597e-44,    1.9211097138146841e-45,    1.0784454226495003e-46,
    5.9393942819517762e-48,    3.2102509168511504e-49,    1.7034831898786868e-50,    8.8773358722414078e-52,
    4.5447729497280927e-53,    2.2864435702189836e-54,    1.1307250155969541e-55,    5.4982801354731781e-57
},
// x = 5.8100
{
    8.6057746593198065e-02,    7.1247288416578528e-02,    4.9269129510627403e-02,    2.8847004844610009e-02,
    1.4513701987000884e-02,    6.3645060695656118e-03,    2.4638626126081598e-03,    8.5156046476250128e-04,
    2.6534161924542050e-04,    7.5173627039240280e-05,    1.9507038566321518e-05,    4.6662587272347404e-06,
    1.0347578905213460e-06,    2.1377210709125592e-07,    4.1324690613616324e-08,    7.5042881936873643e-09,
    1.2845987023756678e-09,    2.0794444525413473e-10,    3.1921321326663845e-11,    4.6589221755526154e-12,
    6.4800551830723211e-13,    8.6077726224471371e-14,    1.0941451585670699e-14,    1.3332647175555210e-15,
    1.5600550561742937e-16,    1.7555634035032547e-17,    1.9026939501902306e-18,    1.9887338785831547e-19,
    2.0071517796538364e-20,    1.9583251384039854e-21,    1.8491139966484715e-22,    1.6914573936570484e-23,
    1.5003569791430966e-24,    1.2916883428112211e-25,    1.0802435797970120e-26,    8.7828865979025871e-28,
    6.9476272133919143e-29,    5.3509892608088241e-30,    4.0153985153335343e-31,    2.9376770607753201e-32,
    2.0966711525440131e-33,    1.4607123493650135e-34,    9.9392248797089959e-36,    6.6088815934737712e-37,
    4.2965006521077640e-38,    2.7322935339673912e-39,    1.7004755532331170e-40,    1.0361925521955719e-41,
    6.1848052369143567e-43,    3.6174954301805870e-44,    2.0742455711636401e-45,    1.1664058622451778e-46,
    6.4348343090229001e-48,    3.4839991687498330e-49,    1.8519144543862647e-50,    9.6674040664802323e-52,
    4.9577394477003786e-53,    2.4984830849548625e-54,    1.2377059235505615e-55,    6.0288154984825016e-57
},
// x = 5.8200
{
    8.5909896333307950e-02,    7.1150258935883101e-02,    4.9234505129244509e-02,    2.8852574116944540e-02,
    1.4532096397524270e-02,    6.3802601001544243e-03,    2.4731855553080059e-03,    8.5596246802314177e-04,
    2.6709672019681482e-04,    7.5783044767841224e-05,    1.9695027655752471e-05,    4.7185119893528985e-06,
    1.0479871480004692e-06,    2.1684898247803270e-07,    4.1986713824028706e-08,    7.6368345576147685e-09,
    1.3094163521974898e-09,    2.1230884927847905e-10,    3.2644921828630144e-11,    4.7724046634764698e-12,
    6.6489057853010195e-13,    8.8467598230045741e-14,    1.1263993668939289e-14,    1.3748636763914867e-15,
    1.6114267402521787e-16,    1.8164187175735203e-17,    1.9719616605279082e-18,    2.0646071388312280e-19,
    2.0872439982933145e-20,    2.0399099265610760e-21,    1.9294072741708911e-22,    1.7678934732478083e-23,
    1.5708153644912561e-24,    1.3546416685409072e-25,    1.1348151566156292e-26,    9.2422560584613001e-28,
    7.3234381435468495e-29,    5.6500311400805899e-30,    4.2470179525473368e-31,    3.1124226916590287e-32,
    2.2251755965493420e-33,    1.5528820172832709e-34,    1.0584403516498815e-35,    7.0498885849898532e-37,
    4.5910270895431482e-38,    2.9245784829366979e-39,    1.8232554367212850e-40,    1.1129073742891484e-41,
    6.6540525723310204e-43,    3.8986122339786261e-44,    2.2392586697276717e-45,    1.2613511598988896e-46,
    6.9705345114235343e-48,    3.7805006214479298e-49,    2.0129588830259530e-50,    1.0526081072864394e-51,
    5.4073382013015080e-53,    2.7297279959401145e-54,    1.3545769461075155e-55,    6.6093929404711126e-57
},
// x = 5.8300
{
    8.5762552976300019e-02,    7.1053475672705119e-02,    4.9199872527223618e-02,    2.8858044688808343e-02,
    1.4550419213045514e-02,    6.3959910151531766e-03,    2.4825116372848041e-03,    8.6037329908071603e-04,
    2.6885820911829573e-04,    7.6395673864416514e-05,    1.9884314877487235e-05,    4.7712120415636311e-06,
    1.0613514202036204e-06,    2.1996238374364713e-07,    4.2657704752767599e-08,    7.7713995532078591e-09,
    1.3346539552644179e-09,    2.1675452340926517e-10,    3.3383231538126215e-11,    4.8883884331640746e-12,
    6.8217683942999657e-13,    9.0918378853635543e-14,    1.1595314437486995e-14,    1.4176670720035678e-15,
    1.6643752768121444e-16,    1.8792482573120161e-17,    2.0435977962868400e-18,    2.1432078869433623e-19,
    2.1703563321403926e-20,    2.1247150545380147e-21,    2.0130119143087279e-22,    1.8476173357356979e-23,
    1.6444295932243262e-24,    1.4205267818353902e-25,    1.1920255609040098e-26,    9.7246600107480637e-28,
    7.7187671422392906e-29,    5.9651410688562681e-30,    4.4914995340151084e-31,    3.2971872987257217e-32,
    2.3612799892776066e-33,    1.6506696127332536e-34,    1.1270077496130384e-35,    7.5193774483779460e-37,
    4.9051115189022612e-38,    3.1299829470550209e-39,    1.9546369982759785e-40,    1.1951371760578548e-41,
    7.1578949226982450e-43,    4.2009710806171829e-44,    2.4170446178500407e-45,    1.3638209213166300e-46,
    7.5496796748629626e-48,    4.1015973704702704e-49,    2.1876610706545167e-50,    1.1459176622203706e-51,
    5.8967394973812993e-53,    2.9818762785396217e-54,    1.4822309867467068e-55,    7.2446241765922951e-57
},
// x = 5.8400
{
    8.5615713924193343e-02,    7.0956937801116018e-02,    4.9165232177044708e-02,    2.8863417101591439e-02,
    1.4568670582671405e-02,    6.4116987378855014e-03,    2.4918407681610412e-03,    8.6479291834893810e-04,
    2.7062608062096004e-04,    7.7011519281074641e-05,    2.0074904877737743e-05,    4.8243613303053393e-06,
    1.0748516933160308e-06,    2.2311264316479904e-07,    4.3337760876034796e-08,    7.9080086776398802e-09,
    1.3603175529464119e-09,    2.2128277914131530e-10,    3.4136513572091290e-11,    5.0069226058053592e-12,
    6.9987288263770521e-13,    9.3431477698178105e-14,    1.1935632462763496e-14,    1.4617070090758857e-15,
    1.7189454725551205e-16,    1.9441116007377041e-17,    2.1176780130070447e-18,    2.2246280714186638e-19,
    2.2565959444947953e-20,    2.2128605045318566e-21,    2.1000571765693942e-22,    1.9307631745331641e-23,
    1.7213341028479726e-24,    1.4894738144154190e-25,    1.2519966446720886e-26,    1.0231203415862692e-27,
    8.1345855517961042e-29,    6.2971476117549416e-30,    4.7495296973836690e-31,    3.4925236779898718e-32,
    2.5054174075451522e-33,    1.7544054511149576e-34,    1.1998701337252065e-35,    8.0191263436809270e-37,
    5.2400106083553817e-38,    3.3493738481798247e-39,    2.0952043554630208e-40,    1.2832665678780974e-41,
    7.6988084010469585e-43,    4.5261314028320350e-44,    2.6085646246078944e-45,    1.4743952739611991e-46,
    8.1756996420827073e-48,    4.4492766689310947e-49,    2.3771498664431113e-50,    1.2472979850631604e-51,
    6.4293811347086670e-53,    3.2567723541191628e-54,    1.6216396289087204e-55,    7.9395359287463783e-57
},
// x = 5.8500
{
    8.5469376596652921e-02,    7.0860644497936331e-02,    4.9130584546429178e-02,    2.8868691894150696e-02,
    1.4586850655992445e-02,    6.4273831926238640e-03,    2.5011728578962925e-03,    8.6922128618765766e-04,
    2.7240032920999163e-04,    7.7630585919306048e-05,    2.0266802292587527e-05,    4.8779623048891959e-06,
    1.0884889571257373e-06,    2.2630009495014929e-07,    4.4026980432740320e-08,    8.0466876767358089e-09,
    1.3864132568753687e-09,    2.2589494564398245e-10,    3.4905035073764822e-11,    5.1280571432476615e-12,
    7.1798745211376211e-13,    9.6008333561465605e-14,    1.2285171234612688e-14,    1.5070163721372919e-15,
    1.7751833026180073e-16,    2.0110699858960519e-17,    2.1942802093243973e-18,    2.3089625311555618e-19,
    2.3460735588399560e-20,    2.3044704593552836e-21,    2.1906770772235355e-22,    2.0174703617920787e-23,
    1.8016687601296556e-24,    1.5616183870240188e-25,    1.3148556187615946e-26,    1.0763041818035810e-27,
    8.5719109316080542e-29,    6.6469202524897444e-30,    5.0218300210949891e-31,    3.6990139253605414e-32,
    2.6580446676478678e-33,    1.8644385540419390e-34,    1.2772873024263410e-35,    8.5510206019563992e-37,
    5.5970590584656277e-38,    3.5836735230918871e-39,    2.2455800322692487e-40,    1.3777061535644720e-41,
    8.2794410107044313e-43,    4.8757637553621878e-44,    2.8148501629958144e-45,    1.5936978224113136e-46,
    8.8522876391443112e-48,    4.8256820520778224e-49,    2.5826449873759382e-50,    1.3574297821805790e-51,
    7.0089904179165134e-53,    3.5564194035429884e-54,    1.7738598985853458e-55,    8.6996063722973834e-57
},
// x = 5.8600
{
    8.5323538430841464e-02,    7.0764594942735071e-02,    4.9095930098383228e-02,    2.8873869602817643e-02,
    1.4604959583072064e-02,    6.4430443045840943e-03,    2.5105078167879254e-03,    8.7365836290439884e-04,
    2.7418094928519777e-04,    7.8252878629934551e-05,    2.0460011747867318e-05,    4.9320174174407057e-06,
    1.1022642050113139e-06,    2.2952507524226878e-07,    4.4725462427481766e-08,    8.1874625465398805e-09,
    1.4129472495404268e-09,    2.3059236994702812e-10,    3.5689067262955876e-11,    5.2518428601054874e-12,
    7.3652945679308091e-13,    9.8650414966183750e-14,    1.2644159259650155e-14,    1.5536288425904705e-15,
    1.8331359382214400e-16,    2.0801863531586812e-17,    2.2734845881974315e-18,    2.3963090795816354e-19,
    2.4389035689071504e-20,    2.3996734398911586e-21,    2.2850105535506544e-22,    2.1078836365732701e-23,
    1.8855790681780512e-24,    1.6371018287035857e-25,    1.3807352766101504e-26,    1.1321383548289722e-27,
    9.0318091552531664e-29,    7.0153713247371748e-30,    5.3091589463274697e-31,    3.9172709255685173e-32,
    2.8196435748751017e-33,    1.9811376679726943e-34,    1.3595342081013552e-35,    9.1170589519566413e-37,
    5.9776742784412765e-38,    3.8338631457088740e-39,    2.4064274012819978e-40,    1.4788942309118784e-41,
    8.9026242064257314e-43,    5.2516574932850953e-44,    3.0370079544769521e-45,    1.7223988128573206e-46,
    9.5834199325169256e-48,    5.2331252902941686e-49,    2.8054641365455010e-50,    1.4770497061073834e-51,
    7.6396079120617570e-53,    3.8829926893306969e-54,    1.9400415934965076e-55,    9.5308047045632349e-57
},
// x = 5.8700
{
    8.5178196881272580e-02,    7.0668788317829176e-02,    4.9061269291240640e-02,    2.8878950761406175e-02,
    1.4622997514435987e-02,    6.4586819999199888e-03,    2.5198455554717855e-03,    8.7810410875589838e-04,
    2.7596793514155193e-04,    7.8878402213073434e-05,    2.0654537859031550e-05,    4.9865291228412723e-06,
    1.1161784339294660e-06,    2.3278792211952767e-07,    4.5433306633512126e-08,    8.3303595348850366e-09,
    1.4399257848859075e-09,    2.3537641712779688e-10,    3.6488885486778366e-11,    5.3783314360082737e-12,
    7.5550797326513641e-13,    1.0135922069812178e-13,    1.3012830161347780e-14,    1.6015789160690190e-15,
    1.8928517749021233e-16,    2.1515253884963954e-17,    2.3553737196566674e-18,    2.4867685910307572e-19,
    2.5352041518820831e-20,    2.4986024467235049e-21,    2.3832016333755663e-22,    2.2021532994210920e-23,
    1.9732163809488111e-24,    1.7160714043362851e-25,    1.4497742268552869e-26,    1.1907492019168534e-27,
    9.5153965981051960e-29,    7.4034580298347259e-30,    5.6123135795076411e-31,    4.1479399134751837e-32,
    2.9907222360458668e-33,    2.1048923360786223e-34,    1.4469018081204586e-35,    9.7193600949950404e-37,
    6.3833613324368624e-38,    4.1009863531990766e-39,    2.5784532758570172e-40,    1.5872985998009916e-41,
    9.5713852080530872e-43,    5.6557289642803909e-44,    3.2762252963503529e-45,    1.8612185210787073e-46,
    1.0373376910713532e-47,    5.6740992304607896e-49,    3.0470306636793036e-50,    1.6069548330567881e-51,
    8.3256130949182158e-53,    4.2388539669092915e-54,    2.1214352250953693e-55,    1.0439634095693317e-56
},
// x = 5.8800
{
    8.5033349419665519e-02,    7.0573223808282542e-02,    4.9026602578705031e-02,    2.8883935901220424e-02,
    1.4640964601061684e-02,    6.4742962057178456e-03,    2.5291859849228566e-03,    8.8255848394962427e-04,
    2.7776128096973355e-04,    7.9507161418081317e-05,    2.0850385231035275e-05,    5.0414998786696089e-06,
    1.1302326444024618e-06,    2.3608897559791829e-07,    4.6150613595694197e-08,    8.4754051429638850e-09,
    1.4673551889118030e-09,    2.4024847049967973e-10,    3.7304769270852792e-11,    5.5075754279870066e-12,
    7.7493224849000037e-13,    1.0413628035264940e-13,    1.3391422781849492e-14,    1.6509019201278050e-15,
    1.9543804613404221e-16,    2.2251535677453329e-17,    2.4400326051102926e-18,    2.5804450894218492e-19,
    2.6350973848357988e-20,    2.6013951060615867e-21,    2.4853996100533809e-22,    2.3004354135458499e-23,
    2.0647381254255535e-24,    1.7986805507399902e-25,    1.5221171351072870e-26,    1.2522688114093505e-27,
    1.0023842419126410e-28,    7.8121845450158886e-30,    5.9321315789817603e-31,    4.3917001111205176e-32,
    3.1718164381060578e-33,    2.2361140260131160e-34,    1.5396979618158235e-35,    1.0360169646703583e-36,
    6.8157181709056641e-38,    4.3861530876977116e-39,    2.7624106611588907e-40,    1.7034184844361771e-41,
    1.0288960114047300e-42,    6.0900302481028087e-44,    3.5337757547670789e-45,    2.0109308792009289e-46,
    1.1226765690622485e-47,    6.1512915897708597e-49,    3.3088818080268825e-50,    1.7480074891096947e-51,
    9.0717520544989264e-53,    4.6265670716350223e-54,    2.3194006233152119e-55,    1.1433178303735215e-56
},
// x = 5.8900
{
    8.4888993534801255e-02,    7.0477900601904814e-02,    4.8991930409892165e-02,    2.8888825551062560e-02,
    1.4658860994367903e-02,    6.4898868499910665e-03,    2.5385290164558934e-03,    8.8702144864529376e-04,
    2.7956098085667245e-04,    8.0139160943522696e-05,    2.1047558458212149e-05,    5.0969321451431557e-06,
    1.1444278405054364e-06,    2.3942857763281635e-07,    4.6877484633442683e-08,    8.6226261269015055e-09,
    1.4952418602768717e-09,    2.4520993180187764e-10,    3.8137002370977947e-11,    5.6396282830009437e-12,
    7.9481170255063464e-13,    1.0698315488956702e-13,    1.3780181285543705e-14,    1.7016340322721831e-15,
    2.0177729287942942e-16,    2.3011392018865672e-17,    2.5275487432410185e-18,    2.6774458392950311e-19,
    2.7387093644640696e-20,    2.7081938200770874e-21,    2.5917592230653696e-22,    2.4028920128231670e-23,
    2.1603080317330652e-24,    1.8850891216218830e-25,    1.5979149752314435e-26,    1.3168352675031760e-27,
    1.0558370940688091e-28,    8.2426042260426735e-30,    6.2694931295894526e-31,    4.6492664440163898e-32,
    3.3634910959656941e-33,    2.3752373163740832e-34,    1.6382483757732271e-35,    1.1041867465350995e-36,
    7.2764411628080256e-38,    4.6905436659749630e-39,    2.9591016734652222e-40,    1.8277865766720662e-41,
    1.1058807865112736e-42,    6.5567584786287366e-44,    3.8110252476994014e-45,    2.1723673575432565e-46,
    1.2148544355505752e-47,    6.6675997705845744e-49,    3.5926775666193426e-50,    1.9011404514827193e-51,
    9.8831673904501182e-53,    5.0489147749703618e-54,    2.5354162579292756e-55,    1.2519152258561879e-56
},
// x = 5.9000
{
    8.4745126732380077e-02,    7.0382817889250410e-02,    4.8957253229371389e-02,    2.8893620237240760e-02,
    1.4676686846204386e-02,    6.5054538616747528e-03,    2.5478745617260323e-03,    8.9149296295637666e-04,
    2.8136702878609203e-04,    8.0774405437128555e-05,    2.1246062124152756e-05,    5.1528283850593685e-06,
    1.1587650298535317e-06,    2.4280707212067700e-07,    4.7614021843653427e-08,    8.7720494993296793e-09,
    1.5235922709042839e-09,    2.5026222139046154e-10,    3.8985872825274389e-11,    5.7745443506054456e-12,
    8.1515593144178474e-13,    1.0990143719642946e-13,    1.4179355264417292e-14,    1.7538122983315566e-15,
    2.0830814211506050e-16,    2.3795524833596736e-17,    2.6180121975293622e-18,    2.7778814392618545e-19,
    2.8461703302210692e-20,    2.8191459217771471e-21,    2.7024408443922091e-22,    2.5096913168251805e-23,
    2.2600963714474109e-24,    1.9754636417015969e-25,    1.6773252904920081e-26,    1.3845929092300613e-27,
    1.1120264130409140e-28,    8.6958219082507583e-30,    6.6253230090388440e-31,    4.9213913403453221e-32,
    3.5663417729006267e-33,    2.5227211447852465e-34,    1.7428975999349741e-35,    1.1766975387401103e-36,
    7.7673309453203970e-38,    5.0154130900844414e-39,    3.1693806376600412e-40,    1.9609712077935028e-41,
    1.1884625111179372e-42,    7.0582657860483070e-44,    4.1094385437294130e-45,    2.3464211189479607e-46,
    1.3144047938906447e-47,    7.2261467696955308e-49,    3.9002102339966934e-50,    2.0673625532153124e-51,
    1.0765430489919370e-52,    5.5089170104156119e-54,    2.7710893346566256e-55,    1.3705956944001484e-56
},
// x = 5.9100
{
    8.4601746534880534e-02,    7.0287974863616834e-02,    4.8922571477207022e-02,    2.8898320483577065e-02,
    1.4694442308841614e-02,    6.5209971706202963e-03,    2.5572225327293889e-03,    8.9597298695159107e-04,
    2.8317941863905614e-04,    8.1412899495761105e-05,    2.1445900801583925e-05,    5.2091910637369808e-06,
    1.1732452235889048e-06,    2.4622480490066808e-07,    4.8360328103619257e-08,    8.9237025309627109e-09,
    1.5524129665898175e-09,    2.5540677843073206e-10,    3.9851673006802016e-11,    5.9123788957619349e-12,
    8.3597470989584804e-13,    1.1289275266044718e-13,    1.4589199845216718e-14,    1.8074746511829381e-15,
    2.1503595256051789e-16,    2.4604655334312049e-17,    2.7115156654394125e-18,    2.8818659179277536e-19,
    2.9576147909353965e-20,    2.9344038345392875e-21,    2.8176106708358478e-22,    2.6210079531049592e-23,
    2.3642802043760047e-24,    2.0699775703261604e-25,    1.7605124649224394e-26,    1.4556926000470006e-27,
    1.1710864189156958e-28,    9.1729963101845230e-30,    7.0005927501470904e-31,    5.2088666168859297e-32,
    3.7809962769956530e-33,    2.6790501206609268e-34,    1.8540100771340057e-35,    1.2538165392052069e-36,
    8.2902986085747097e-38,    5.3620956127371713e-39,    3.3941573734058975e-40,    2.1035786565455704e-41,
    1.2770362038227363e-42,    7.5970698991168160e-44,    4.4305862041832026e-45,    2.5340514641290156e-46,
    1.4219016276459949e-47,    7.8302982604959817e-49,    4.2334146627962400e-50,    2.2477647216964218e-51,
    1.1724576361338512e-52,    6.0098505775514975e-54,    3.0281667287374920e-55,    1.5002738934130482e-56
},
// x = 5.9200
{
    8.4458850481419739e-02,    7.0193370721043147e-02,    4.8887885588999237e-02,    2.8902926811415420e-02,
    1.4712127534960720e-02,    6.5365167075900048e-03,    2.5665728418036147e-03,    9.0046148065639053e-04,
    2.8499814419451664e-04,    8.2054647665380376e-05,    2.1647079052248788e-05,    5.2660226489572438e-06,
    1.1878694363675979e-06,    2.4968212375624041e-07,    4.9116507073933410e-08,    9.0776127521748085e-09,
    1.5817105676126306e-09,    2.6064506109088506e-10,    4.0734699676655011e-11,    6.0531881117913002e-12,
    8.5727799424609734e-13,    1.1595875974907242e-13,    1.5009975798442538e-14,    1.8626599298302507e-15,
    2.2196622039831081e-16,    2.5439524506394527e-17,    2.8081545493039003e-18,    2.9895168323460342e-19,
    3.0731816549984948e-20,    3.0541252364374763e-21,    2.9374409224658542e-22,    2.7370231869618694e-23,
    2.4730436340891718e-24,    2.1688115749095792e-25,    1.8476480052992246e-26,    1.5302920084470834e-27,
    1.2331576249514042e-28,    9.6753425441673192e-30,    7.3963229031812679e-31,    5.5125254556540551e-32,
    4.0081163372647913e-33,    2.8447359058648324e-34,    1.9719712488084175e-35,    1.3358268217610600e-36,
    8.8473722338874407e-38,    5.7320095719004186e-39,    3.6344006810786896e-40,    2.2562556016667297e-41,
    1.3720239214849341e-42,    8.1758654498618621e-44,    4.7761519978994517e-45,    2.7362885877998109e-46,
    1.5379623855864299e-47,    8.4836809319876456e-49,    4.5943792981222961e-50,    2.4435264836794453e-51,
    1.2767141223338284e-52,    6.5552704408872080e-54,    3.3085468236443875e-55,    1.6419454968402226e-56
},
// x = 5.9300
{
    8.4316436127615135e-02,    7.0099004660308148e-02,    4.8853195995924671e-02,    2.8907439739629656e-02,
    1.4729742677643459e-02,    6.5520124042517253e-03,    2.5759254016284525e-03,    9.0495840405444309e-04,
    2.8682319912986143e-04,    8.2699654441012059e-05,    2.1849601426787364e-05,    5.3233256109050325e-06,
    1.2026386863462489e-06,    2.5317937841663255e-07,    4.9882663201378602e-08,    9.2338079545786591e-09,
    1.6114917693485204e-09,    2.6597854673697561e-10,    4.1635254037534579e-11,    6.1970291334714727e-12,
    8.7907592532759926e-13,    1.1910115059936952e-13,    1.5441949649202458e-14,    1.9194078988450431e-15,
    2.2910458247108409e-16,    2.6300893603370192e-17,    2.9080270289458555e-18,    3.1009553690637888e-19,
    3.1930143642174406e-20,    3.1784732294915843e-21,    3.0621100473706563e-22,    2.8579251589220865e-23,
    2.5865780724931872e-24,    2.2721538145405843e-25,    1.9389108341093755e-26,    1.6085559000164475e-27,
    1.2983871189177066e-28,    1.0204134738325235e-29,    7.8135854027111826e-31,    5.8332444754223530e-32,
    4.2483993632498369e-33,    3.0203186666251252e-34,    2.0971887197807565e-35,    1.4230282453720405e-36,
    9.4407038049128186e-38,    6.1266625099131755e-39,    3.8911420391757965e-40,    2.4196917276596545e-41,
    1.4738765522050434e-42,    8.7975360257722626e-44,    5.1479408197918690e-45,    2.9542386666376407e-46,
    1.6632511804063770e-47,    9.1902021744097692e-49,    4.9853580423872495e-50,    2.6559229718203525e-51,
    1.3900203060789487e-52,    7.1490327491683673e-54,    3.6142923279162093e-55,    1.7966941981245135e-56
},
// x = 5.9400
{
    8.4174501045447492e-02,    7.0004875882928350e-02,    4.8818503124776605e-02,    2.8911859784631556e-02,
    1.4747287890362325e-02,    6.5674841931734817e-03,    2.5852801252262509e-03,    9.0946371708909503e-04,
    2.8865457702146523e-04,    8.3347924266719638e-05,    2.2053472464618143e-05,    5.3811024221099929e-06,
    1.2175539951686793e-06,    2.5671692055831275e-07,    5.0658901721803469e-08,    9.3923161926055270e-09,
    1.6417633428857396e-09,    2.7140873212919260e-10,    4.2553641787803732e-11,    6.3439600502805572e-12,
    9.0137883141623884e-13,    1.2232165161628259e-13,    1.5885393789950273e-14,    1.9777592681745969e-15,
    2.3645681954520648e-16,    2.7189544653534114e-17,    3.0112341360752161e-18,    3.2163064478218269e-19,
    3.3172610314265742e-20,    3.3076165139762075e-21,    3.1918029328993834e-22,    2.9839091301755458e-23,
    2.7050825137440614e-24,    2.3802002341136027e-25,    2.0344875939154952e-26,    1.6906564413786371e-27,
    1.3669288563927381e-28,    1.0760708774765402e-29,    8.2535060435714464e-31,    6.1719459024639636e-32,
    4.5025802920719055e-33,    3.2063686002277692e-34,    2.2300934851285746e-35,    1.5157384134691226e-36,
    1.0072576512182810e-37,    6.5476565932469292e-39,    4.1654795255698861e-40,    2.5946224930439479e-41,
    1.5830757233606578e-42,    9.4651670172858230e-44,    5.5478871463531902e-45,    3.1890893015249578e-46,
    1.7984822160081742e-47,    9.9540712074777133e-49,    5.4087830113467919e-50,    2.8863324703304099e-51,
    1.5131425375825363e-52,    7.7953197104314837e-54,    3.9476441488319811e-55,    1.9656993035159143e-56
},
// x = 5.9500
{
    8.4033042823125431e-02,    6.9910983593156006e-02,    4.8783807398004767e-02,    2.8916187460378908e-02,
    1.4764763326970744e-02,    6.5829320078181156e-03,    2.5946369259624642e-03,    9.1397737966483251e-04,
    2.9049227134523973e-04,    8.3999461535576135e-05,    2.2258696693820100e-05,    5.4393555573875341e-06,
    1.2326163879523214e-06,    2.6029510380635392e-07,    5.1445328662984176e-08,    9.5531657850864045e-09,
    1.6725321356432859e-09,    2.7693713361943446e-10,    4.3490173176024499e-11,    6.4940399197863848e-12,
    9.2419723120618473e-13,    1.2562202407990082e-13,    1.6340586595136380e-14,    2.0377557133232333e-15,
    2.4402885964192144e-16,    2.8106280977999275e-17,    3.1178798304991047e-18,    3.3356988279715292e-19,
    3.4460745819541923e-20,    3.4417295679278233e-21,    3.3267111235845283e-22,    3.1151777362171809e-23,
    2.8287638178101450e-24,    2.4931548693492506e-25,    2.1345729635358385e-26,    1.7767735164833207e-27,
    1.4389439664986866e-28,    1.1346465148795204e-29,    8.7172670707208782e-31,    6.5295998450540948e-32,
    4.7714335270902962e-33,    3.4034875401758059e-34,    2.3711412223207447e-35,    1.6142936860455893e-36,
    1.0745412472572182e-37,    6.9966943499177716e-39,    4.4585819756767594e-40,    2.7818320708718555e-41,
    1.7001358318335081e-42,    1.0182059311345868e-43,    5.9780640633562743e-45,    3.4421153379755240e-46,
    1.9444234591922771e-47,    1.0779821753858134e-48,    5.8672782463597887e-50,    3.1362445400709435e-51,
    1.6469104378703452e-52,    8.4986664684366223e-54,    4.3110364078056428e-55,    2.1502439642335309e-56
},
// x = 5.9600
{
    8.3892059064951049e-02,    6.9817326997976617e-02,    4.8749109233754771e-02,    2.8920423278383688e-02,
    1.4782169141693387e-02,    6.5983557825379634e-03,    2.6039957175461410e-03,    9.1849935164873121e-04,
    2.9233627547718683e-04,    8.4654270589640992e-05,    2.2465278631015818e-05,    5.4980874937799151e-06,
    1.2478268932745299e-06,    2.6391428373575003e-07,    5.2242050847473638e-08,    9.7163853168345941e-09,
    1.7038050719917255e-09,    2.8256528735020094e-10,    4.4445163055982219e-11,    6.6473287811838401e-12,
    9.4754183682622269e-13,    1.2900406476183741e-13,    1.6807812537798008e-14,    2.0994398959129733e-15,
    2.5182678143730115e-16,    2.9051927720398582e-17,    3.2280710781856992e-18,    3.4592652176734190e-19,
    3.5796128990432948e-20,    3.5809928319932096e-21,    3.4670330459413479e-22,    3.2519412489479270e-23,
    2.9578370040015257e-24,    2.6112301630828917e-25,    2.2393699864715260e-26,    1.8670950557132375e-27,
    1.5146010705758169e-28,    1.1962871954265948e-29,    9.2061098879880415e-31,    6.9072266764616465e-32,
    5.0557749725121343e-33,    3.6123106436766402e-34,    2.5208136519498237e-35,    1.7190502473032453e-36,
    1.1461780886361959e-37,    7.4755847424827836e-39,    4.7716933913413714e-40,    2.9821564718564962e-41,
    1.8256062039907772e-42,    1.0951743884930972e-43,    6.4406929032486838e-45,    3.7146850902166550e-46,
    2.1019005726609930e-47,    1.1672336367591161e-48,    6.3636744525109164e-50,    3.4072687663487100e-51,
    1.7922219881610981e-52,    9.2639901372292028e-54,    4.7071126890200171e-55,    2.3517240998638884e-56
},
// x = 5.9700
{
    8.3751547391187064e-02,    6.9723905307106551e-02,    4.8714409045907375e-02,    2.8924567747720148e-02,
    1.4799505489116583e-02,    6.6137554525695190e-03,    2.6133564140303702e-03,    9.2302959287189417e-04,
    2.9418658269395205e-04,    8.5312355719936931e-05,    2.2673222781255006e-05,    5.5573007104971655e-06,
    1.2631865431587317e-06,    2.6757481787266222e-07,    5.3049175895435302e-08,    9.8820036402292391e-09,
    1.7355891538764671e-09,    2.8829474945478852e-10,    4.5418930942196541e-11,    6.8038876689807346e-12,
    9.7142355689528201e-13,    1.3246960655082289e-13,    1.7287362308114999e-14,    2.1628554846294719e-15,
    2.5985681773222012e-16,    3.0027332388469795e-17,    3.3419179312218227e-18,    3.5871423859430298e-19,
    3.7180389733270035e-20,    3.7255928997650241e-21,    3.6129742403445217e-22,    3.3944178474977174e-23,
    3.0925255547933305e-24,    2.7346472932116622e-25,    2.3490904120274099e-26,    1.9618173782998771e-27,
    1.5940766143134692e-28,    1.2611468000323673e-29,    9.7213378908973657e-31,    7.3058995313668776e-32,
    5.3564641684913428e-33,    3.8335081655001614e-34,    2.6796199705528032e-35,    1.8303852317776257e-36,
    1.2224406655760838e-37,    7.9862495955247827e-39,    5.1061376150191005e-40,    3.1964868610606278e-41,
    1.9600733934348412e-42,    1.1777997355777071e-43,    6.9381535321113274e-45,    4.0082669960582883e-46,
    2.2718011273748273e-47,    1.2636872534719536e-48,    6.9010248371541357e-50,    3.7011441758060453e-51,
    1.9500490178072419e-52,    1.0096621162520084e-53,    5.1387436199565846e-55,    2.5716580695621457e-56
},
// x = 5.9800
{
    8.3611505437925060e-02,    6.9630717732990483e-02,    4.8679707244117118e-02,    2.8928621375033020e-02,
    1.4816772524178816e-02,    6.6291309540281113e-03,    2.6227189298127238e-03,    9.2756806313088564e-04,
    2.9604318617337913e-04,    8.5973721166430052e-05,    2.2882533637899076e-05,    5.6169976888580391e-06,
    1.2786963730604551e-06,    2.7127706569560323e-07,    5.3866812227463559e-08,    1.0050049876799999e-08,
    1.7678914614435247e-09,    2.9412709625880178e-10,    4.6411801065923526e-11,    6.9637786268335763e-12,
    9.9585349961757987e-13,    1.3602051908762899e-13,    1.7779532933959906e-14,    2.2280471765595391e-15,
    2.6812535899362512e-16,    3.1033365407761530e-17,    3.4595336097057034e-18,    3.7194712776115750e-19,
    3.8615210564622269e-20,    3.8757227137547195e-21,    3.7647476001885858e-22,    3.5428338980407883e-23,
    3.2330617302806542e-24,    2.8636365127034165e-25,    2.4639550495888061e-26,    2.0611455485565137e-27,
    1.6775552138765576e-28,    1.3293866065061971e-29,    1.0264319428987451e-30,    7.7267469208561480e-32,
    5.6744065314614569e-33,    4.0677873224410061e-34,    2.8480983591833496e-35,    1.9486979120191108e-36,
    1.3036179489996514e-37,    8.5307303975557434e-39,    5.4633232846468426e-40,    3.4257730797276086e-41,
    2.1041636250140149e-42,    1.2664858551029222e-43,    7.4729953285756690e-45,    4.3244367314460922e-46,
    2.4550791135078466e-47,    1.3679090671446078e-48,    7.4826221286965659e-50,    4.0197493721585698e-51,
    2.1214431211596160e-52,    1.1002337191401168e-53,    5.6090458901608134e-55,    2.8116971521316929e-56
},
// x = 5.9900
{
    8.3471930856955243e-02,    6.9537763490798474e-02,    4.8645004233850833e-02,    2.8932584664545707e-02,
    1.4833970402161346e-02,    6.6444822239026206e-03,    2.6320831796356691e-03,    9.3211472218914795e-04,
    2.9790607899506490e-04,    8.6638371118011992e-05,    2.3093215682506110e-05,    5.6771809122309095e-06,
    1.2943574218532073e-06,    2.7502138863655737e-07,    5.4695069067389352e-08,    1.0220553418812666e-08,
    1.8007191536677220e-09,    3.0006392448297371e-10,    4.7424102431649248e-11,    7.1270647215341528e-12,
    1.0208429759177339e-12,    1.3965870940942894e-13,    1.8284627903469125e-14,    2.2950607189263742e-15,
    2.7663895696836049e-16,    3.2070920687698730e-17,    3.5810345856166939e-18,    3.8563971312698676e-19,
    4.0102328190269930e-20,    4.0315817671560923e-21,    3.9225736185436742e-22,    3.6974242428808286e-23,
    3.3796868936125156e-24,    2.9984375020836890e-25,    2.5841941365316865e-26,    2.1652937464551138e-27,
    1.7652300165860710e-28,    1.4011756291786759e-29,    1.0836490903255564e-30,    8.1709554713660428e-32,
    6.0105557046591283e-33,    4.3158942528157569e-34,    3.0268175715762109e-35,    2.0744109510616486e-36,
    1.3900163523397976e-37,    9.1111954983445011e-39,    5.8447490854571847e-40,    3.6710273845034153e-41,
    2.2585453940914813e-42,    1.3616646158281178e-43,    8.0479488997680657e-45,    4.6648848154714406e-46,
    2.6527597705326726e-47,    1.4805084153731740e-48,    8.1120168610596339e-50,    4.3651134441254752e-51,
    2.3075420359676052e-52,    1.1987399645684448e-53,    6.1214028228448792e-55,    3.0736369009149965e-56
},
// x = 6.0000
{
    8.3332821315637226e-02,    6.9445041798423243e-02,    4.8610300416425598e-02,    2.8936458118068571e-02,
    1.4851099278678935e-02,    6.6598092000501683e-03,    2.6414490785869626e-03,    9.3666952977841786e-04,
    2.9977525414091768e-04,    8.7306309712484345e-05,    2.3305273384717050e-05,    5.7378528659746496e-06,
    1.3101707318142278e-06,    2.7880815008203467e-07,    5.5534056445071508e-08,    1.0393543930855770e-08,
    1.8340794689833570e-09,    3.0610685144730491e-10,    4.8456168874079457e-11,    7.2938100571481851e-12,
    1.0464035026162643e-12,    1.4338612260371148e-13,    1.8802957289664354e-14,    2.3639429312289559e-15,
    2.8540432837086139e-16,    3.3140916200253700e-17,    3.7065406687049571e-18,    3.9980696002658814e-19,
    4.1643535127898001e-20,    4.1933763115571798e-21,    4.0866806425241876e-22,    3.8584324990919562e-23,
    3.5326518477633705e-24,    3.1392997348305349e-25,    2.7100477202605295e-26,    2.2744856530928420e-27,
    1.8573030767334319e-28,    1.4766909733830690e-29,    1.1439360004596522e-30,    8.6397727931820209e-32,
    6.3659160240190260e-33,    4.5786160756335293e-34,    3.2163786059313803e-35,    2.2079717230738693e-36,
    1.4819607474270064e-37,    9.7299477238095183e-39,    6.2520093158964323e-40,    3.9333284170044549e-41,
    2.4239322306043603e-42,    1.4637977527407087e-43,    8.6659385821907071e-45,    5.0314247386062352e-46,
    2.8659447583341826e-47,    1.6021411521414520e-48,    8.7930370152589512e-50,    4.7394277027435565e-51,
    2.5095765183369622e-52,    1.3058593208943355e-53,    6.6794866228188357e-55,    3.3594294446631435e-56
},
// x = 6.0100
{
    8.3194174496772286e-02,    6.9352551876476931e-02,    4.8575596189046701e-02,    2.8940242235007144e-02,
    1.4868159309670649e-02,    6.6751118211908593e-03,    2.6508165421000236e-03,    9.4123244560012696e-04,
    3.0165070449571327e-04,    8.7977541036545401e-05,    2.3518711202142237e-05,    5.7990160373794872e-06,
    1.3261373486101037e-06,    2.8263771537406258e-07,    5.6383885199173554e-08,    1.0569051351428069e-08,
    1.8679797259172977e-09,    3.1225751527651594e-10,    4.9508339115625015e-11,    7.4640797893070448e-12,
    1.0725468056458408e-12,    1.4720474247185627e-13,    1.9334837877151724e-14,    2.4347417277918993e-15,
    2.9442835864601589e-16,    3.4244294571469506e-17,    3.8361750944445010e-18,    4.1446428768272759e-19,
    4.3240681374612576e-20,    4.3613195707615262e-21,    4.2573051355933636e-22,    4.0261113670090533e-23,
    3.6922171840110117e-24,    3.2864828561205392e-25,    2.8417660548844220e-26,    2.3889548516130793e-27,
    1.9539857471300735e-28,    1.5561182054086257e-29,    1.2074509099344123e-30,    9.1345104843362793e-32,
    6.7415451048516034e-33,    4.8567830542938891e-34,    3.4174164645400292e-35,    2.3498537057573572e-36,
    1.5797955373805118e-37,    1.0389432431807094e-38,    6.6867997857605516e-40,    4.2138254174291136e-41,
    2.6010856380090154e-42,    1.5733788695763272e-43,    9.3300957784583610e-45,    5.4260016490488468e-46,
    3.0958176927062159e-47,    1.7335131009698572e-48,    9.5298091159537041e-50,    5.1450583093689740e-51,
    2.7288777518498866e-52,    1.4223268453209308e-53,    7.2872824337979036e-55,    3.6711968111890555e-56
},
// x = 6.0200
{
    8.3055988098476655e-02,    6.9260292948288005e-02,    4.8540891944844761e-02,    2.8943937512370437e-02,
    1.4885150651390771e-02,    6.6903900269025089e-03,    2.6601854859542922e-03,    9.4580342932679726e-04,
    3.0353242284765497e-04,    8.8652069125778185e-05,    2.3733533580248844e-05,    5.8606729156077806e-06,
    1.3422583212822385e-06,    2.8651045181110852e-07,    5.7244666979924913e-08,    1.0747105894526671e-08,
    1.9024273237244701e-09,    3.1851757510681446e-10,    5.0580956824386717e-11,    7.6379401396535899e-12,
    1.0992848233086742e-12,    1.5111659220248444e-13,    1.9880593290927743e-14,    2.5075061407322904e-15,
    3.0371810580853335e-16,    3.5382023686088721e-17,    3.9700646140940772e-18,    4.2962758193820836e-19,
    4.4895676120412146e-20,    4.5356319608841593e-21,    4.4346919480328965e-22,    4.2007229488697550e-23,
    3.8586536425012050e-24,    3.4402570753837420e-25,    2.9796100130596120e-26,    2.5089452441662666e-27,
    2.0554990870163246e-28,    1.6396517385671014e-29,    1.2743598768275875e-30,    9.6565472760015512e-32,
    7.1385565549582500e-33,    5.1512708698923925e-34,    3.6306020056786339e-35,    2.5005579482359403e-36,
    1.6838857895796956e-37,    1.1092246033400481e-38,    7.1509240656698670e-40,    4.5137426966969125e-41,
    2.7908182178060388e-42,    1.6909355713762265e-43,    1.0043773184006208e-44,    5.8507016343199245e-46,
    3.3436500701358859e-47,    1.8753837571284374e-48,    1.0326780887665654e-49,    5.5845598600727763e-51,
    2.9668853312204595e-52,    1.5489387848321508e-53,    7.9491143484122371e-55,    4.0112453566829537e-56
},
// x = 6.0300
{
    8.2918259834056290e-02,    6.9168264239897900e-02,    4.8506188072913056e-02,    2.8947544444779275e-02,
    1.4902073460399807e-02,    6.7056437576153878e-03,    2.6695558262755507e-03,    9.5038244060342088e-04,
    3.0542040188893173e-04,    8.9329897964641851e-05,    2.3949744952249140e-05,    5.9228259916348743e-06,
    1.3585347022322076e-06,    2.9042672864894252e-07,    5.8116514251867901e-08,    1.0927738051236071e-08,
    1.9374297430257809e-09,    3.2488871129398370e-10,    5.1674370672641861e-11,    7.8154584104433481e-12,
    1.1266297095754511e-12,    1.5512373505470284e-13,    2.0440554127321283e-14,    2.5822863433501146e-15,
    3.1328080436017872e-16,    3.6555097305545150e-17,    4.1083395869112533e-18,    4.4531320831521786e-19,
    4.6610489508770311e-20,    4.7165413168916931e-21,    4.6190945958128504e-22,    4.3825390779187999e-23,
    4.0322424852907770e-24,    3.6009035731397995e-25,    3.1238515135443978e-26,    2.6347114855166210e-27,
    2.1620742869769954e-28,    1.7274952360389205e-29,    1.3448371505701935e-30,    1.0207332325738128e-31,
    7.5581228200902611e-33,    5.4630030094496279e-34,    3.8566438924110262e-35,    2.6606146183668756e-36,
    1.7946184319515752e-37,    1.1841145005512512e-38,    7.6463001080623725e-40,    4.8343843824274356e-41,
    2.9939969909692628e-42,    1.8170317352503219e-43,    1.0810559961271760e-44,    6.3077616376388808e-46,
    3.6108076084289940e-47,    2.0285702563517073e-48,    1.1188745582678752e-49,    6.0606899968436295e-51,
    3.2251558638241839e-52,    1.6865575415219399e-53,    8.6696735252948269e-55,    4.3820813901085961e-56
},
// x = 6.0400
{
    8.2780987431882530e-02,    6.9076464980057442e-02,    4.8471484958344073e-02,    2.8951063524474619e-02,
    1.4918927893555601e-02,    6.7208729546069925e-03,    2.6789274795362482e-03,    9.5496943904884075e-04,
    3.0731463421627953e-04,    9.0011031486464778e-05,    2.4167349738989540e-05,    5.9854777581898562e-06,
    1.3749675472069645e-06,    2.9438691710143200e-07,    5.8999540296589306e-08,    1.1110978591317776e-08,
    1.9729945464484149e-09,    3.3137262562278829e-10,    5.2788934395833974e-11,    7.9967029993021259e-12,
    1.1545938374262115e-12,    1.5922827505135402e-13,    2.1015058087100617e-14,    2.6591336739489103e-15