/*
 *  libgrpp - a library for the evaluation of integrals over
 *            generalized relativistic pseudopotentials.
 *
 *  Copyright (C) 2021-2023 Alexander Oleynichenko
 */

#include "specfunc_fermi_sk.h"

#include <assert.h>
#include <math.h>
#include <stdio.h>


/*
 * This file contains pre-tabulated values of the special function S_k(x)
 * used in the formulas for Fermi nuclear model and its electrostatic potential.
 *
 * The pre-tabulated points are given for the range x = [-10.00, -0.005] with the step = 0.005;
 * for this range the Taylor series expansion of the 3rd order (max x^3) can be used to achieve
 * convergence ~1e-12 at each point within the given range.
 *
 * The asymtotic tail is given by the first one to three terms of the exponential series for S_k(x),
 * depending on the argument:
 * x <= -30:       1 term
 * -30 < x <= -14: 2 terms
 * -14 < x <= -10: 3 terms
 *
 * k = -2, ..., +7 (overall 10 numbers per 1 x-point)
 * overall 2000 x-points.
 */

extern double specfunc_fermi_sk_table[2001][10];


/**
 * The S_k(x) special function.
 * This special function is needed for the Fermi nuclear model.
 *
 * allowed values of k: 2, 3, 4, 5, 6, 7
 */
double specfunc_fermi_sk(int k, double x)
{
    assert(k >= 2 && k <= 7);

    if (x <= -30) {
        // asymptotic tail: 1 exponential term
        double d1 = -1.0 * exp(x);
        return d1;
    }
    if (x <= -14) {
        // asymptotic tail: 2 exponential terms
        double d1 = -1.0 * exp(x);
        double d2 = 1.0 * exp(2 * x) / pow(2, k);
        return d1 + d2;
    }
    if (x <= -10) {
        // asymptotic tail: 3 exponential terms
        double d1 = -1.0 * exp(x);
        double d2 = 1.0 * exp(2 * x) / pow(2, k);
        double d3 = -1.0 * exp(3 * x) / pow(3, k);
        return d1 + d2 + d3;
    }

    // taylor power series (max 3rd order) in range [-10.0,0.0]
    int i0 = (int) (floor((x + 10.0) * 200) + 1e-10);
    double x0 = -10.0 + i0 * 0.005;
    double dx = x - x0;

    //printf("x=%10.6f  x0=%10.6f  i0=%d\n", x, x0, i0);

    if (i0 == 2000) { // special case: x == 0.0
        return specfunc_fermi_sk_table[i0][k + 2];
    }

    double *derivatives = specfunc_fermi_sk_table[i0];

    double d0 = derivatives[k + 2]; // f(x0)
    double d1 = derivatives[k + 1]; // f'(x0)
    double d2 = derivatives[k + 0]; // f''(x0)
    double d3 = derivatives[k - 1]; // f'''(x0)

    double dx2 = dx * dx;
    double dx3 = dx2 * dx;

    double sum = d0 +
                 d1 * dx +
                 d2 * dx2 * 0.5 +
                 d3 * dx3 * (1.0 / 6.0);

    return sum;
}


/**
 * auxiliary special function Sk(x)
 * used in the formula for the electrostatic potential induced by the Fermi charge distribution.
 * "exact" implementation, direct summation, no approximate Taylor series.
 */
double fermi_model_Sk(int k, double x)
{
    const double tol = 1e-12;
    const int max_iter = 10000000; // to achieve such tolerance at x=0
    double prev_sum = 0.0;
    double sum = 0.0;
    int converged = 0;

    for (int i = 1; i < max_iter; i++) {
        double d = pow(-1, i) * exp(i * x) / pow(i, k);
        sum += d;

        if (fabs(sum - prev_sum) < tol) {
            converged = 1;
            break;
        }

        prev_sum = sum;
    }

    if (!converged) {
        printf("prev_sum = %e\n", prev_sum);
        printf("sum   = %e\n", sum);
        printf("delta = %e\n", sum - prev_sum);
        printf("S(%d,%f) not converged!\n", k, x);
    }

    return sum;
}


double specfunc_fermi_sk_table[2001][10] = {
// x = -10.0000
        {
                -4.5391685990113197e-05, -4.5395807735951667e-05, -4.5397868702434395e-05, -4.5398899216864647e-05,
                -4.5399414484476340e-05, -4.5399672121747767e-05, -4.5399800941538702e-05, -4.5399865351819237e-05,
                -4.5399897557087868e-05, -4.5399913659764962e-05
        },
// x = -9.9950
        {
                -4.5619171238036491e-05, -4.5623334405267251e-05, -4.5625416083860072e-05, -4.5626456954816718e-05,
                -4.5626977400848726e-05, -4.5627237627382691e-05, -4.5627367741822347e-05, -4.5627432799433060e-05,
                -4.5627465328368721e-05, -4.5627481592879991e-05
        },
// x = -9.9900
        {
                -4.5847796343220638e-05, -4.5852001348092632e-05, -4.5854103946941398e-05, -4.5855155278504492e-05,
                -4.5855680954999217e-05, -4.5855943796817704e-05, -4.5856075218917340e-05, -4.5856140930363966e-05,
                -4.5856173786219542e-05, -4.5856190214191426e-05
        },
// x = -9.9850
        {
                -4.6077567015030746e-05, -4.6081814277975649e-05, -4.6083938007317909e-05, -4.6084999904613441e-05,
                -4.6085530864136284e-05, -4.6085796347522806e-05, -4.6085929090424451e-05, -4.6085995462278069e-05,
                -4.6086028648339152e-05, -4.6086045241414441e-05
        },
// x = -9.9800
        {
                -4.6308488991408086e-05, -4.6312778937082356e-05, -4.6314924009268341e-05, -4.6315996578478763e-05,
                -4.6316532874123416e-05, -4.6316801025625619e-05, -4.6316935102603370e-05, -4.6317002141501126e-05,
                -4.6317035661086304e-05, -4.6317052420924329e-05
        },
// x = -9.9750
        {
                -4.6540568039012988e-05, -4.6544901096340245e-05, -4.6547067725854120e-05, -4.6548151074228989e-05,
                -4.6548692759622679e-05, -4.6548963606055025e-05, -4.6549099030516383e-05, -4.6549166743162122e-05,
                -4.6549200599623356e-05, -4.6549217527900089e-05
        },
// x = -9.9700
        {
                -4.6773809953368244e-05, -4.6778186555582192e-05, -4.6780374959063509e-05, -4.6781469194930133e-05,
                -4.6782016324239080e-05, -4.6782289892685500e-05, -4.6782426678172707e-05, -4.6782495071337658e-05,
                -4.6782529268060579e-05, -4.6782546366468853e-05
        },
// x = -9.9650
        {
                -4.7008220559003428e-05, -4.7012641143691174e-05, -4.7014851539956509e-05, -4.7015956772730870e-05,
                -4.7016509400665596e-05, -4.7016785718482219e-05, -4.7016923878673622e-05, -4.7016992959197035e-05,
                -4.7017027499601318e-05, -4.7017044769850981e-05
        },
// x = -9.9600
        {
                -4.7243805709599856e-05, -4.7248270718745617e-05, -4.7250503328810437e-05, -4.7251619669008088e-05,
                -4.7252177850828961e-05, -4.7252456945646829e-05, -4.7252596494358255e-05, -4.7252666269148137e-05,
                -4.7252701156687805e-05, -4.7252718600505883e-05
        },
// x = -9.9550
        {
                -4.7480571288136236e-05, -4.7485081168165515e-05, -4.7487336215266336e-05, -4.7488463774513412e-05,
                -4.7489027566036163e-05, -4.7489309465764015e-05, -4.7489450416950115e-05, -4.7489520892983903e-05,
                -4.7489556131147710e-05, -4.7489573750278582e-05
        },
// x = -9.9500
        {
                -4.7718523207035076e-05, -4.7723078408859345e-05, -4.7725356118475977e-05, -4.7726495009520431e-05,
                -4.7727064467121690e-05, -4.7727349199948741e-05, -4.7727491567704435e-05, -4.7727562752029674e-05,
                -4.7727598344341432e-05, -4.7727616140547009e-05
        },
// x = -9.9450
        {
                -4.7957667408309867e-05, -4.7962268387371585e-05, -4.7964568987249715e-05, -4.7965719323972531e-05,
                -4.7966294504595516e-05, -4.7966582098994292e-05, -4.7966725897556129e-05, -4.7966797797291196e-05,
                -4.7966833747310112e-05, -4.7966851722370035e-05
        },
// x = -9.9400
        {
                -4.8198009863712943e-05, -4.8202657080031075e-05, -4.8204980800205017e-05, -4.8206142697631644e-05,
                -4.8206723658791838e-05, -4.8207014143520991e-05, -4.8207159387268596e-05, -4.8207232009603411e-05,
                -4.8207268320924501e-05, -4.8207286476636265e-05
        },
// x = -9.9350
        {
                -4.8439556574884015e-05, -4.8444250493100064e-05, -4.8446597565915780e-05, -4.8447771140227580e-05,
                -4.8448357940018467e-05, -4.8448651344125663e-05, -4.8448798047583198e-05, -4.8448871399779951e-05,
                -4.8448908076034324e-05, -4.8448926414213511e-05
        },
// x = -9.9300
        {
                -4.8682313573499624e-05, -4.8687054662924059e-05, -4.8689425323062340e-05, -4.8690610691608254e-05,
                -4.8691203388707147e-05, -4.8691499741531995e-05, -4.8691647919369583e-05, -4.8691722008763423e-05,
                -4.8691759053618705e-05, -4.8691777576099129e-05
        },
// x = -9.9250
        {
                -4.8926286921423170e-05, -4.8931075656082300e-05, -4.8933470140582302e-05, -4.8934667421890544e-05,
                -4.8935266075564439e-05, -4.8935565406741410e-05, -4.8935715073776577e-05, -4.8935789907776400e-05,
                -4.8935827324937065e-05, -4.8935846033570978e-05
        },
// x = -9.9200
        {
                -4.9171482710855758e-05, -4.9176319569539184e-05, -4.9178738117822047e-05, -4.9179947431611989e-05,
                -4.9180552101723497e-05, -4.9180854441184859e-05, -4.9181005612384095e-05, -4.9181081198473240e-05,
                -4.9181118991680986e-05, -4.9181137888339251e-05
        },
// x = -9.9150
        {
                -4.9417907064487823e-05, -4.9422792530796241e-05, -4.9425235384689064e-05, -4.9426456851883185e-05,
                -4.9427067598896519e-05, -4.9427372976875363e-05, -4.9427525667355544e-05, -4.9427602013092554e-05,
                -4.9427640186126705e-05, -4.9427659272698987e-05
        },
// x = -9.9100
        {
                -4.9665566135651427e-05, -4.9670500698044986e-05, -4.9672968101805021e-05, -4.9674201844540985e-05,
                -4.9674818729528000e-05, -4.9675127176561282e-05, -4.9675281401591192e-05, -4.9675358514610592e-05,
                -4.9675397071288442e-05, -4.9675416349683409e-05
        },
// x = -9.9050
        {
                -4.9914466108473402e-05, -4.9919450260320532e-05, -4.9921942460659575e-05, -4.9923188602302480e-05,
                -4.9923811686948784e-05, -4.9924123233880313e-05, -4.9924279008882226e-05, -4.9924356896895241e-05,
                -4.9924395841072443e-05, -4.9924415313217931e-05
        },
// x = -9.9000
        {
                -5.0164613198029215e-05, -5.0169647437655957e-05, -5.0172164683765011e-05, -5.0173423348919684e-05,
                -5.0174052695530798e-05, -5.0174367373514379e-05, -5.0174524714065537e-05, -5.0174603384860913e-05,
                -5.0174642720431866e-05, -5.0174662388275101e-05
        },
// x = -9.8950
        {
                -5.0416013650497568e-05, -5.0421098481237356e-05, -5.0423641024811555e-05, -5.0424912339335038e-05,
                -5.0425548010842643e-05, -5.0425865851345177e-05, -5.0426024773179366e-05, -5.0426104234624114e-05,
                -5.0426143965522380e-05, -5.0426163831030137e-05
        },
// x = -9.8900
        {
                -5.0668673743315841e-05, -5.0673809673559848e-05, -5.0676377768823616e-05, -5.0677661859837738e-05,
                -5.0678303919805953e-05, -5.0678624954610555e-05, -5.0678785473619705e-05, -5.0678865733659909e-05,
                -5.0678905863858553e-05, -5.0678925929017390e-05
        },
// x = -9.8850
        {
                -5.0922599785336250e-05, -5.0927787328584167e-05, -5.0930381232316645e-05, -5.0931678228220727e-05,
                -5.0932326740852480e-05, -5.0932651002061685e-05, -5.0932813134297437e-05, -5.0932894200959033e-05,
                -5.0932934734471065e-05, -5.0932955001287502e-05
        },
// x = -9.8800
        {
                -5.1177798116982817e-05, -5.1183037791894235e-05, -5.1185657763454915e-05, -5.1186967793938624e-05,
                -5.1187622824082038e-05, -5.1187950344121028e-05, -5.1188114105796316e-05, -5.1188195987185897e-05,
                -5.1188236928064670e-05, -5.1188257398565378e-05
        },
// x = -9.8750
        {
                -5.1434275110409072e-05, -5.1439567440855276e-05, -5.1442213742209980e-05, -5.1443536938266278e-05,
                -5.1444198551421185e-05, -5.1444529363041002e-05, -5.1444694770531719e-05, -5.1444777474837363e-05,
                -5.1444818827176938e-05, -5.1444839503408980e-05
        },
// x = -9.8700
        {
                -5.1692037169656647e-05, -5.1697382684772993e-05, -5.1700055580520014e-05, -5.1701392074458273e-05,
                -5.1702060336782764e-05, -5.1702394473063569e-05, -5.1702561542910188e-05, -5.1702645078402249e-05,
                -5.1702686846337856e-05, -5.1702707730368858e-05
        },
// x = -9.8650
        {
                -5.1951090730814508e-05, -5.1956489965053320e-05, -5.1959189722449937e-05, -5.1960539647909133e-05,
                -5.1961214626226154e-05, -5.1961552120580584e-05, -5.1961720869489799e-05, -5.1961805244521751e-05,
                -5.1961847432230176e-05, -5.1961868526148536e-05
        },
// x = -9.8600
        {
                -5.2211442262179087e-05, -5.2216895755363102e-05, -5.2219622644352211e-05, -5.2220986136314342e-05,
                -5.2221667898118383e-05, -5.2222008784294861e-05, -5.2222179229141269e-05, -5.2222264452150542e-05,
                -5.2222307063850532e-05, -5.2222328369765653e-05
        },
// x = -9.8550
        {
                -5.2473098264415231e-05, -5.2478606561791511e-05, -5.2481360855028707e-05, -5.2482738049832218e-05,
                -5.2483426663296082e-05, -5.2483770975382181e-05, -5.2483943133209970e-05, -5.2484029212718793e-05,
                -5.2484072252671512e-05, -5.2484093772713974e-05
        },
// x = -9.8500
        {
                -5.2736065270717824e-05, -5.2741628923012297e-05, -5.2744410895893050e-05, -5.2745801931246530e-05,
                -5.2746497465228136e-05, -5.2746845237654004e-05, -5.2747019125678656e-05, -5.2747106070294897e-05,
                -5.2747149542804329e-05, -5.2747171279126144e-05
        },
// x = -9.8450
        {
                -5.3000349846974389e-05, -5.3005969410446859e-05, -5.3008779341134106e-05, -5.3010184356130035e-05,
                -5.3010886880179257e-05, -5.3011238147721078e-05, -5.3011413783331083e-05, -5.3011501601749135e-05,
                -5.3011545511162512e-05, -5.3011567465937309e-05
        },
// x = -9.8400
        {
                -5.3265958591928294e-05, -5.3271634628428116e-05, -5.3274472797880015e-05, -5.3275891933008633e-05,
                -5.3276601517374338e-05, -5.3276956315157776e-05, -5.3277133715916389e-05, -5.3277222416918011e-05,
                -5.3277266767626257e-05, -5.3277288943049518e-05
        },
// x = -9.8350
        {
                -5.3532898137343009e-05, -5.3538631214365154e-05, -5.3541497906363208e-05, -5.3542931303526630e-05,
                -5.3543648019163637e-05, -5.3544006382667379e-05, -5.3544185566314350e-05, -5.3544275158769556e-05,
                -5.3544319955207729e-05, -5.3544342353497006e-05
        },
// x = -9.8300
        {
                -5.3801175148166954e-05, -5.3806965838908789e-05, -5.3809861340086230e-05, -5.3811309142612554e-05,
                -5.3812033061188775e-05, -5.3812395026248033e-05, -5.3812576010701409e-05, -5.3812666503569362e-05,
                -5.3812711750217092e-05, -5.3812734373612206e-05
        },
// x = -9.8250
        {
                -5.4070796322699305e-05, -5.4076645206117898e-05, -5.4079569805988297e-05, -5.4081032158645998e-05,
                -5.4081763352549560e-05, -5.4082128955359711e-05, -5.4082311758717605e-05, -5.4082403161047503e-05,
                -5.4082448862429438e-05, -5.4082471713192733e-05
        },
// x = -9.8200
        {
                -5.4341768392756595e-05, -5.4347676053626539e-05, -5.4350630044612781e-05, -5.4352107093625178e-05,
                -5.4352845635971678e-05, -5.4353214913091832e-05, -5.4353399553634251e-05, -5.4353491874566244e-05,
                -5.4353538035252506e-05, -5.4353561115669051e-05
        },
// x = -9.8150
        {
                -5.4614098123840131e-05, -5.4620065152812042e-05, -5.4623048830275552e-05, -5.4624540723335394e-05,
                -5.4625286687975225e-05, -5.4625659676331924e-05, -5.4625846172522567e-05, -5.4625939421288663e-05,
                -5.4625986045895305e-05, -5.4626009358273151e-05
        },
// x = -9.8100
        {
                -5.4887792315304259e-05, -5.4893819308963777e-05, -5.4896832971234039e-05, -5.4898339857518289e-05,
                -5.4899093319044011e-05, -5.4899470055934881e-05, -5.4899658426423034e-05, -5.4899752612348017e-05,
                -5.4899799705537472e-05, -5.4899823252207864e-05
        },
// x = -9.8050
        {
                -5.5162857800525427e-05, -5.5168945361452854e-05, -5.5171989309857272e-05, -5.5173511340042020e-05,
                -5.5174272373795824e-05, -5.5174652896893353e-05, -5.5174843160515688e-05, -5.5174938293018054e-05,
                -5.5174985859499638e-05, -5.5175009642817238e-05
        },
// x = -9.8000
        {
                -5.5439301447072168e-05, -5.5445450183902745e-05, -5.5448524722796682e-05, -5.5450062049072219e-05,
                -5.5450830731153449e-05, -5.5451215078508688e-05, -5.5451407254291232e-05, -5.5451503342884145e-05,
                -5.5451551387414481e-05, -5.5451575409757617e-05
        },
// x = -9.7950
        {
                -5.5717130156875839e-05, -5.5723340684360512e-05, -5.5726446121157812e-05, -5.5727998897243854e-05,
                -5.5728775304516611e-05, -5.5729163514563056e-05, -5.5729357621722994e-05, -5.5729454676015223e-05,
                -5.5729503203398757e-05, -5.5729527467169654e-05
        },
// x = -9.7900
        {
                -5.5996350866402288e-05, -5.6002623805469197e-05, -5.6005760450672860e-05, -5.6007328831833888e-05,
                -5.6008113041934737e-05, -5.6008505153492099e-05, -5.6008701211439801e-05, -5.6008799241136669e-05,
                -5.6008848256226107e-05, -5.6008872763851163e-05
        },
// x = -9.7850
        {
                -5.6276970546824299e-05, -5.6283306524640844e-05, -5.6286474691874082e-05, -5.6288058834934870e-05,
                -5.6288850926280603e-05, -5.6289246978558755e-05, -5.6289445006899619e-05, -5.6289544021804001e-05,
                -5.6289593529500839e-05, -5.6289618283430810e-05
        },
// x = -9.7800
        {
                -5.6558996204194992e-05, -5.6565395854230503e-05, -5.6568595860268100e-05, -5.6570195923629414e-05,
                -5.6570995975424877e-05, -5.6571396008027705e-05, -5.6571596026564195e-05, -5.6571696036577473e-05,
                -5.6571746041832462e-05, -5.6571771044542739e-05
        },
// x = -9.7750
        {
                -5.6842434879621974e-05, -5.6848898841711024e-05, -5.6852131006511009e-05, -5.6853747150165426e-05,
                -5.6854555242411444e-05, -5.6854959295340871e-05, -5.6855161324074430e-05, -5.6855262339197509e-05,
                -5.6855312847011159e-05, -5.6855338101002006e-05
        },
// x = -9.7700
        {
                -5.7127293649442453e-05, -5.7133822569848827e-05, -5.7137087216584431e-05, -5.7138719602132365e-05,
                -5.7139535815633707e-05, -5.7139943929293659e-05, -5.7140147988426789e-05, -5.7140250018761071e-05,
                -5.7140301034184131e-05, -5.7140326541980953e-05
        },
// x = -9.7650
        {
                -5.7413579625399144e-05, -5.7420174156880455e-05, -5.7423471611972432e-05, -5.7425120402638245e-05,
                -5.7425944819011771e-05, -5.7426357034212235e-05, -5.7426563144150427e-05, -5.7426666199898838e-05,
                -5.7426717728032831e-05, -5.7426743492186412e-05
        },
// x = -9.7600
        {
                -5.7701299954817158e-05, -5.7707960756690020e-05, -5.7711291349839298e-05, -5.7712956710487646e-05,
                -5.7713789412170419e-05, -5.7714205770131508e-05, -5.7714413951485337e-05, -5.7714518042953353e-05,
                -5.7714570088951066e-05, -5.7714596112037821e-05
        },
// x = -9.7550
        {
                -5.7990461820781607e-05, -5.7997189558987627e-05, -5.8000553623208231e-05, -5.8002235720360546e-05,
                -5.8003076790618101e-05, -5.8003497332974174e-05, -5.8003707606561358e-05, -5.8003812744158012e-05,
                -5.8003865313224029e-05, -5.8003891597846267e-05
        },
// x = -9.7500
        {
                -5.8281072442316272e-05, -5.8287867789488549e-05, -5.8291265661140928e-05, -5.8292964662992075e-05,
                -5.8293814185926699e-05, -5.8294238954730536e-05, -5.8294451341578011e-05, -5.8294557535816947e-05,
                -5.8294610633208150e-05, -5.8294637181994323e-05
        },
// x = -9.7450
        {
                -5.8573139074563026e-05, -5.8580002710093385e-05, -5.8583434728918023e-05, -5.8585150805353112e-05,
                -5.8586008865912308e-05, -5.8586437903639305e-05, -5.8586652424985325e-05, -5.8586759686485842e-05,
                -5.8586813317511949e-05, -5.8586840133116949e-05
        },
// x = -9.7400
        {
                -5.8866669008962203e-05, -5.8873601619069155e-05, -5.8877068128220539e-05, -5.8878801450831862e-05,
                -5.8879668134816817e-05, -5.8880101484369247e-05, -5.8880318161665503e-05, -5.8880426501153647e-05,
                -5.8880480671177732e-05, -5.8880507756283110e-05
        },
// x = -9.7350
        {
                -5.9161669573433914e-05, -5.9168671851231190e-05, -5.9172173197312078e-05, -5.9173923939416353e-05,
                -5.9174799333490517e-05, -5.9175237038201813e-05, -5.9175455893115571e-05, -5.9175565321425158e-05,
                -5.9175620035864206e-05, -5.9175647393178476e-05
        },
// x = -9.7300
        {
                -5.9458148132560138e-05, -5.9465220778125966e-05, -5.9468757311222109e-05, -5.9470525647877715e-05,
                -5.9471409839575485e-05, -5.9471851943214543e-05, -5.9472072997630851e-05, -5.9472183525704612e-05,
                -5.9472238790030029e-05, -5.9472266422288916e-05
        },
// x = -9.7250
        {
                -5.9756112087767848e-05, -5.9763255808214953e-05, -5.9766827881930026e-05, -5.9768613989954583e-05,
                -5.9769507067690011e-05, -5.9769953614465635e-05, -5.9770176890489465e-05, -5.9770288529380067e-05,
                -5.9770344349118276e-05, -5.9770372259085006e-05
        },
// x = -9.7200
        {
                -6.0055568877512961e-05, -6.0062784387059234e-05, -6.0066392358550215e-05, -6.0068196416538234e-05,
                -6.0069098469613896e-05, -6.0069549504179156e-05, -6.0069775024137637e-05, -6.0069887785008848e-05,
                -6.0069944165741775e-05, -6.0069972356207349e-05
        },
// x = -9.7150
        {
                -6.0356525977465225e-05, -6.0363813997505113e-05, -6.0367458227517969e-05, -6.0369280415858689e-05,
                -6.0370191534474627e-05, -6.0370647101931345e-05, -6.0370874888375999e-05, -6.0370988782503777e-05,
                -6.0371045729869485e-05, -6.0371074203652935e-05
        },
// x = -9.7100
        {
                -6.0658990900694079e-05, -6.0666352159870680e-05, -6.0670033012776459e-05, -6.0671873513671886e-05,
                -6.0672793788934633e-05, -6.0673253934837892e-05, -6.0673484010546865e-05, -6.0673599049320484e-05,
                -6.0673656569013682e-05, -6.0673685328962403e-05
        },
// x = -9.7050
        {
                -6.0962971197855376e-05, -6.0970406432133270e-05, -6.0974124275964502e-05, -6.0975983273447654e-05,
                -6.0976912797379275e-05, -6.0977377567741987e-05, -6.0977609955722360e-05, -6.0977726150645564e-05,
                -6.0977784248418182e-05, -6.0977813297408162e-05
        },
// x = -9.7000
        {
                -6.1268474457379021e-05, -6.1275984410117832e-05, -6.1279739616605300e-05, -6.1281617296558594e-05,
                -6.1282556162105980e-05, -6.1283025603403460e-05, -6.1283260326893535e-05, -6.1283377689585678e-05,
                -6.1283436371247462e-05, -6.1283465712183596e-05
        },
// x = -9.6950
        {
                -6.1575508305657621e-05, -6.1583093727686294e-05, -6.1586886672296284e-05, -6.1588783222470109e-05,
                -6.1589731523514164e-05, -6.1590205682688815e-05, -6.1590442765160403e-05, -6.1590561307357621e-05,
                -6.1590620578776733e-05, -6.1590650214593103e-05
        },
// x = -9.6900
        {
                -6.1884080407235983e-05, -6.1891742056927862e-05, -6.1895573118899591e-05, -6.1897488728931044e-05,
                -6.1898446560296210e-05, -6.1898925484762187e-05, -6.1899164949923022e-05, -6.1899284683479398e-05,
                -6.1899344550582928e-05, -6.1899374484243120e-05
        },
// x = -9.6850
        {
                -6.2194198465001657e-05, -6.2201937108350252e-05, -6.2205806670733823e-05, -6.2207741532165759e-05,
                -6.2208708989629387e-05, -6.2209192727277307e-05, -6.2209434599073377e-05, -6.2209555535962122e-05,
                -6.2209616004736736e-05, -6.2209646239234136e-05
        },
// x = -9.6800
        {
                -6.2505870220376240e-05, -6.2513686631071841e-05, -6.2517595080766535e-05, -6.2519549387066644e-05,
                -6.2520526567368597e-05, -6.2521015166570426e-05, -6.2521259469188370e-05, -6.2521381621503014e-05,
                -6.2521442697995578e-05, -6.2521473236353594e-05
        },
// x = -9.6750
        {
                -6.2819103453507923e-05, -6.2826998413014886e-05, -6.2830946140807838e-05, -6.2832920087388017e-05,
                -6.2833907088240336e-05, -6.2834400597854139e-05, -6.2834647355723661e-05, -6.2834770735679287e-05,
                -6.2834832425997410e-05, -6.2834863271269914e-05
        },
// x = -9.6700
        {
                -6.3133905983464770e-05, -6.3141880281099557e-05, -6.3145867681704839e-05, -6.3147861465940748e-05,
                -6.3148858386037434e-05, -6.3149356855412287e-05, -6.3149606093208609e-05, -6.3149730713143084e-05,
                -6.3149793023455756e-05, -6.3149824178727247e-05
        },
// x = -9.6650
        {
                -6.3450285668428864e-05, -6.3458340101439119e-05, -6.3462367573537156e-05, -6.3464381394787877e-05,
                -6.3465388333814815e-05, -6.3465891812795720e-05, -6.3466143555442062e-05, -6.3466269427817207e-05,
                -6.3466332364355444e-05, -6.3466363832741446e-05
        },
// x = -9.6600
        {
                -6.3768250405891894e-05, -6.3776385779535914e-05, -6.3780453725813393e-05, -6.3782487785441413e-05,
                -6.3783504844086222e-05, -6.3784013383019164e-05, -6.3784267655689182e-05, -6.3784394793092096e-05,
                -6.3784458362149490e-05, -6.3784490146796846e-05
        },
// x = -9.6550
        {
                -6.4087808132851082e-05, -6.4096025260478433e-05, -6.4100134087668499e-05, -6.4102188589059859e-05,
                -6.4103215869022045e-05, -6.4103729518758925e-05, -6.4103986346879341e-05, -6.4104114762023569e-05,
                -6.4104178969957013e-05, -6.4104211074044183e-05
        },
// x = -9.6500
        {
                -6.4408966826006513e-05, -6.4417266529139298e-05, -6.4421416648062269e-05, -6.4423491796646880e-05,
                -6.4424529400647986e-05, -6.4425048212551743e-05, -6.4425307621804759e-05, -6.4425437327531672e-05,
                -6.4425502180761907e-05, -6.4425534607499309e-05
        },
// x = -9.6450
        {
                -6.4731734501959390e-05, -6.4740117610374354e-05, -6.4744309435978701e-05, -6.4746405439250891e-05,
                -6.4747453471044754e-05, -6.4747977496994543e-05, -6.4748239513320456e-05, -6.4748370522600460e-05,
                -6.4748436027612790e-05, -6.4748468780243069e-05
        },
// x = -9.6400
        {
                -6.5056119217411166e-05, -6.5064586569222595e-05, -6.5068820520626549e-05, -6.5070937588165742e-05,
                -6.5071996152548864e-05, -6.5072525444945221e-05, -6.5072790094545063e-05, -6.5072922420478897e-05,
                -6.5072988583823773e-05, -6.5073021665622223e-05
        },
// x = -9.6350
        {
                -6.5382129069363628e-05, -6.5390681511107160e-05, -6.5394958011640508e-05, -6.5397096355132272e-05,
                -6.5398165557954335e-05, -6.5398700169724377e-05, -6.5398967479062480e-05, -6.5399101134882561e-05,
                -6.5399167963176301e-05, -6.5399201377451053e-05
        },
// x = -9.6300
        {
                -6.5709772195320196e-05, -6.5718410582037400e-05, -6.5722730059283865e-05, -6.5724889892541027e-05,
                -6.5725969840715412e-05, -6.5726509825318180e-05, -6.5726779821124830e-05, -6.5726914820196593e-05,
                -6.5726982320121960e-05, -6.5727016070214457e-05
        },
// x = -9.6250
        {
                -6.6039056773488020e-05, -6.6047781968811873e-05, -6.6052144854651789e-05, -6.6054326393635821e-05,
                -6.6055417195150392e-05, -6.6055962606582149e-05, -6.6056235315856271e-05, -6.6056371671679428e-05,
                -6.6056439849986368e-05, -6.6056473939271636e-05
        },
// x = -9.6200
        {
                -6.6369991022981114e-05, -6.6378803899222317e-05, -6.6383210629875864e-05, -6.6385414092718444e-05,
                -6.6386515856646229e-05, -6.6387066749445936e-05, -6.6387342199457776e-05, -6.6387479925667742e-05,
                -6.6387548789174057e-05, -6.6387583221061004e-05
        },
// x = -9.6150
        {
                -6.6702583204024624e-05, -6.6711484642258757e-05, -6.6715935658329529e-05, -6.6718161265354430e-05,
                -6.6719274101864628e-05, -6.6719830531119284e-05, -6.6720108749413200e-05, -6.6720247859782393e-05,
                -6.6720317415374405e-05, -6.6720352193306201e-05
        },
// x = -9.6100
        {
                -6.7036841618159916e-05, -6.7045832508315519e-05, -6.7050328254834586e-05, -6.7052576228579617e-05,
                -6.7053700248948545e-05, -6.7054262270298820e-05, -6.7054543284695943e-05, -6.7054683793135204e-05,
                -6.7054754047768411e-05, -6.7054789175222850e-05
        },
// x = -9.6050
        {
                -6.7372774608450887e-05, -6.7381855849398388e-05, -6.7386396775868829e-05, -6.7388667341108100e-05,
                -6.7389802657730376e-05, -6.7390370327376027e-05, -6.7390654165977128e-05, -6.7390796086537136e-05,
                -6.7390867047236936e-05, -6.7390902527726779e-05
        },
// x = -9.6000
        {
                -6.7710390559691102e-05, -6.7719563059332713e-05, -6.7724149619774446e-05, -6.7726443003540869e-05,
                -6.7727589729940578e-05, -6.7728163104646271e-05, -6.7728449795834475e-05, -6.7728593142707028e-05,
                -6.7728664816569498e-05, -6.7728700653642770e-05
        },
// x = -9.5950
        {
                -6.8049697898612136e-05, -6.8058962573972520e-05, -6.8063595226967760e-05, -6.8065911658575731e-05,
                -6.8067069909417865e-05, -6.8067649046518635e-05, -6.8067938618962350e-05, -6.8068083406481997e-05,
                -6.8068155800674431e-05, -6.8068191997914854e-05
        },
// x = -9.5900
        {
                -6.8390705094092895e-05, -6.8400062871410845e-05, -6.8404742080149911e-05, -6.8407081791218273e-05,

                -6.8408251682320085e-05, -6.8408836639727237e-05, -6.8409129122382954e-05, -6.8409275365028212e-05,
                -6.8409348486790005e-05, -6.8409385047817268e-05
        },
// x = -9.5850
        {
                -6.8733420657369852e-05, -6.8742872472190862e-05, -6.8747598704518495e-05, -6.8749961928993628e-05,
                -6.8751143577336346e-05, -6.8751734413543088e-05, -6.8752029835658366e-05, -6.8752177548053308e-05,
                -6.8752251404696589e-05, -6.8752288333166799e-05
        },
// x = -9.5800
        {
                -6.9077853142248527e-05, -6.9087399939518295e-05, -6.9092173667980464e-05, -6.9094560642159677e-05,
                -6.9095754165900074e-05, -6.9096350939987577e-05, -6.9096649331103823e-05, -6.9096798528019482e-05,
                -6.9096873126929837e-05, -6.9096910426535841e-05
        },
// x = -9.5750
        {
                -6.9424011145315849e-05, -6.9433653879474706e-05, -6.9438475581365787e-05, -6.9440886543920997e-05,
                -6.9442092062403304e-05, -6.9442694834046369e-05, -6.9442996224001964e-05, -6.9443146920357816e-05,
                -6.9443222268995112e-05, -6.9443259943466863e-05
        },
// x = -9.5700
        {
                -6.9771903306153774e-05, -6.9781642941232102e-05, -6.9786513098642566e-05, -6.9788948290644174e-05,
                -6.9790165924411913e-05, -6.9790774753885132e-05, -6.9791079172818282e-05, -6.9791231383683721e-05,
                -6.9791307489582756e-05, -6.9791345542687680e-05
        },
// x = -9.5650
        {
                -7.0121538307553713e-05, -7.0131375817268225e-05, -7.0136294917132931e-05, -7.0138754582073812e-05,
                -7.0139984452881939e-05, -7.0140599401065609e-05, -7.0140906879417417e-05, -7.0141060620013327e-05,
                -7.0141137490784631e-05, -7.0141175926328062e-05
        },
// x = -9.5600
        {
                -7.0472924875732111e-05, -7.0482861243583293e-05, -7.0487829777730079e-05, -7.0490314161550030e-05,
                -7.0491556392377063e-05, -7.0492177520763350e-05, -7.0492488089280813e-05, -7.0492643374981019e-05,
                -7.0492721018311627e-05, -7.0492759840137101e-05
        },
// x = -9.5550
        {
                -7.0826071780547383e-05, -7.0836107999917627e-05, -7.0841126465116477e-05, -7.0843635816226762e-05,
                -7.0844890531287128e-05, -7.0845517901986084e-05, -7.0845831591725267e-05, -7.0845988438058104e-05,
                -7.0846066861712299e-05, -7.0846106073701965e-05
        },
// x = -9.5500
        {
                -7.1180987835717366e-05, -7.1191124909970364e-05, -7.1196193807983104e-05, -7.1198728377291307e-05,
                -7.1199995702047635e-05, -7.1200629377793598e-05, -7.1200946220122637e-05, -7.1201104642772513e-05,
                -7.1201183854592600e-05, -7.1201223460667665e-05
        },
// x = -9.5450
        {
                -7.1537681899038364e-05, -7.1547920841619371e-05, -7.1553040679249802e-05, -7.1555600720184901e-05,
                -7.1556880781360705e-05, -7.1557520825518434e-05, -7.1557840852120685e-05, -7.1558000866929624e-05,
                -7.1558080874836730e-05, -7.1558120878957798e-05
        },
// x = -9.5400
        {
                -7.1896162872605069e-05, -7.1906504707142140e-05, -7.1911675996286708e-05, -7.1914261764824392e-05,
                -7.1915554690416691e-05, -7.1916201166987752e-05, -7.1916524409865014e-05, -7.1916686032834267e-05,
                -7.1916766844829085e-05, -7.1916807250996565e-05
        },
// x = -9.5350
        {
                -7.2256439703031507e-05, -7.2266885463437817e-05, -7.2272108721136805e-05, -7.2274720475825084e-05,
                -7.2276026395117182e-05, -7.2276679368746315e-05, -7.2277005860222016e-05, -7.2277169107513596e-05,
                -7.2277250731677309e-05, -7.2277291543931804e-05
        },
// x = -9.5300
        {
                -7.2618521381673202e-05, -7.2629072112250415e-05, -7.2634347860739559e-05, -7.2636985862724614e-05,
                -7.2638304906299053e-05, -7.2638964442280654e-05, -7.2639294215003035e-05, -7.2639459102941448e-05,
                -7.2639541547436391e-05, -7.2639582769859137e-05
        },
// x = -9.5250
        {
                -7.2982416944850445e-05, -7.2993073700393049e-05, -7.2998402467155788e-05, -7.3001066980208097e-05,
                -7.3002399279959671e-05, -7.3003065444244363e-05, -7.3003398531189798e-05, -7.3003565076263551e-05,
                -7.3003648349334133e-05, -7.3003689986047321e-05
        },
// x = -9.5200
        {
                -7.3348135474072542e-05, -7.3358899319973239e-05, -7.3364281637793474e-05, -7.3366972928334132e-05,
                -7.3368318617483098e-05, -7.3368991476684275e-05, -7.3369327911160527e-05, -7.3369496130023913e-05,
                -7.3369580239997334e-05, -7.3369622295164653e-05
        },
// x = -9.5150
        {
                -7.3715686096263375e-05, -7.3726558108619614e-05, -7.3731994515634886e-05, -7.3734712852762275e-05,
                -7.3736072065867732e-05, -7.3736751687268189e-05, -7.3737091502917753e-05, -7.3737261412392373e-05,
                -7.3737346367679617e-05, -7.3737388845506564e-05
        },
// x = -9.5100
        {
                -7.4085077983987858e-05, -7.4096059249709336e-05, -7.4101550289464645e-05, -7.4104295944981351e-05,
                -7.4105668817954601e-05, -7.4106355269513337e-05, -7.4106698500316857e-05, -7.4106870117393364e-05,
                -7.4106955926489873e-05, -7.4106998831224217e-05
        },
// x = -9.5050
        {
                -7.4456320355679694e-05, -7.4467411972596991e-05, -7.4472958194099194e-05, -7.4475731442539140e-05,
                -7.4477118112657321e-05, -7.4477811463016314e-05, -7.4478158143295877e-05, -7.4478331485135728e-05,
                -7.4478418156622325e-05, -7.4478461492554540e-05
        },
// x = -9.5000
        {
                -7.4829422475870176e-05, -7.4840625552844505e-05, -7.4846227510617084e-05, -7.4849028629273012e-05,
                -7.4850429235192830e-05, -7.4851129553683840e-05, -7.4851479718106498e-05, -7.4851654802043579e-05,
                -7.4851742344587364e-05, -7.4851786116051031e-05
        },
// x = -9.4950
        {
                -7.5204393655418206e-05, -7.5215709312452162e-05, -7.5221367566590748e-05, -7.5224196835541922e-05,
                -7.5225611517313470e-05, -7.5226318873965062e-05, -7.5226672557546264e-05, -7.5226849401088678e-05,
                -7.5226937823443850e-05, -7.5226982034816084e-05
        },
// x = -9.4900
        {
                -7.5581243251741321e-05, -7.5592672620090841e-05, -7.5598387736319170e-05, -7.5601245438459362e-05,
                -7.5602674337540166e-05, -7.5603388803084659e-05, -7.5603746041191729e-05, -7.5603924662023572e-05,
                -7.5604013973032248e-05, -7.5604058628734195e-05
        },
// x = -9.4850
        {
                -7.5959980669047943e-05, -7.5971524891335374e-05, -7.5977297441061820e-05, -7.5980183862127618e-05,
                -7.5981627121396794e-05, -7.5982348767277325e-05, -7.5982709595633031e-05, -7.5982890011616088e-05,
                -7.5982980220209328e-05, -7.5983025324706546e-05
        },
// x = -9.4800
        {
                -7.6340615358570852e-05, -7.6352275588899171e-05, -7.6358106149273876e-05, -7.6361021577873246e-05,
                -7.6362479341645720e-05, -7.6363208240023424e-05, -7.6363572694709554e-05, -7.6363754923885097e-05,
                -7.6363846039083681e-05, -7.6363891596886586e-05
        },
// x = -9.4750
        {
                -7.6723156818801695e-05, -7.6734934222869746e-05, -7.6740823376842307e-05, -7.6743768104483426e-05,
                -7.6745240518524434e-05, -7.6745976742285642e-05, -7.6746344859746594e-05, -7.6746528920337257e-05,
                -7.6746620951252616e-05, -7.6746666966916998e-05
        },
// x = -9.4700
        {
                -7.7107614595726707e-05, -7.7119510350945808e-05, -7.7125458687323534e-05, -7.7128433008443985e-05,
                -7.7129920219983601e-05, -7.7130663842747091e-05, -7.7131035659793549e-05, -7.7131221570205018e-05,
                -7.7131314526040213e-05, -7.7131361004167617e-05
        },
// x = -9.4650
        {
                -7.7493998283063579e-05, -7.7506013578675171e-05, -7.7512021692181972e-05, -7.7515025904178073e-05,
                -7.7516528061925941e-05, -7.7517279158050391e-05, -7.7517654711862927e-05, -7.7517842490685990e-05,
                -7.7517936380736482e-05, -7.7517983325974706e-05
        },
// x = -9.4600
        {
                -7.7882317522499691e-05, -7.7894453559694058e-05, -7.7900522051029870e-05, -7.7903556454286542e-05,
                -7.7905073708446770e-05, -7.7905832353038077e-05, -7.7906211681170946e-05, -7.7906401347183158e-05,
                -7.7906496180837854e-05, -7.7906543597881412e-05
        },
// x = -9.4550
        {
                -7.8272582003931118e-05, -7.8284839995967584e-05, -7.8290969471868423e-05, -7.8294034369789126e-05,
                -7.8295566872075222e-05, -7.8296333140994131e-05, -7.8296716281379001e-05, -7.8296907853546622e-05,
                -7.8297003640288835e-05, -7.8297051533879417e-05
        },
// x = -9.4500
        {
                -7.8664801465703230e-05, -7.8677182638031346e-05, -7.8683373711329940e-05, -7.8686469410366993e-05,
                -7.8688017314017180e-05, -7.8688791283886751e-05, -7.8689178274836508e-05, -7.8689371772316415e-05,
                -7.8689468521724731e-05, -7.8689516896651680e-05
        },
// x = -9.4450
        {
                -7.9058985694852306e-05, -7.9071491285234312e-05, -7.9077744574921374e-05, -7.9080871384606692e-05,
                -7.9082434844399068e-05, -7.9083216592612431e-05, -7.9083607472824988e-05, -7.9083802914966584e-05,
                -7.9083900636715856e-05, -7.9083949497816650e-05
        },
// x = -9.4400
        {
                -7.9455144527348305e-05, -7.9467775785982848e-05, -7.9474091917268913e-05, -7.9477250150244863e-05,
                -7.9478829322512966e-05, -7.9479618927241018e-05, -7.9480013735803190e-05, -7.9480211142150372e-05,
                -7.9480309846012676e-05, -7.9480359198173407e-05
        },
// x = -9.4350
        {
                -7.9853287848338951e-05, -7.9866046037985989e-05, -7.9872425642363990e-05, -7.9875615614414660e-05,
                -7.9877210657063090e-05, -7.9878008197262310e-05, -7.9878406973653735e-05, -7.9878606363946775e-05,
                -7.9878706059792403e-05, -7.9878755907948265e-05
        },
// x = -9.4300
        {
                -8.0253425592395031e-05, -8.0266311988502030e-05, -8.0272755703810313e-05, -8.0275977733893102e-05,
                -8.0277588806413298e-05, -8.0278394361833655e-05, -8.0278797145930746e-05, -8.0278998540108298e-05,
                -8.0279099237906760e-05, -8.0279149587042563e-05
        },
// x = -9.4250
        {
                -8.0655567743756718e-05, -8.0668583634586129e-05, -8.0675092105072290e-05, -8.0678346515349782e-05,
                -8.0679973778835986e-05, -8.0680787430028889e-05, -8.0681194262108795e-05, -8.0681397680309929e-05,
                -8.0681499390130901e-05, -8.0681550245281531e-05
        },
// x = -9.4200
        {
                -8.1059724336581381e-05, -8.1072871023339391e-05, -8.1079444899724619e-05, -8.1082732015596814e-05,
                -8.1084375632762130e-05, -8.1085197461088523e-05, -8.1085608381833139e-05, -8.1085813844399303e-05,
                -8.1085916576413672e-05, -8.1085967942664619e-05
        },
// x = -9.4150
        {
                -8.1465905455192412e-05, -8.1479184252158924e-05, -8.1485824191703123e-05, -8.1489144341839891e-05,
                -8.1490804477032591e-05, -8.1491634564671069e-05, -8.1492049615171182e-05, -8.1492257142648251e-05,
                -8.1492360907129132e-05, -8.1492412789617021e-05
        },
// x = -9.4100
        {
                -8.1874121234329456e-05, -8.1887533468989493e-05, -8.1894240135556878e-05, -8.1897593651930937e-05,
                -8.1899270471150898e-05, -8.1900108901105891e-05, -8.1900528122865235e-05, -8.1900737736005571e-05,
                -8.1900842543329306e-05, -8.1900894947242357e-05
        },
// x = -9.4050
        {
                -8.2284381859399760e-05, -8.2297928872576080e-05, -8.2304702936701580e-05, -8.2308090154621573e-05,
                -8.2309783825536853e-05, -8.2310630681646993e-05, -8.2311054116586380e-05, -8.2311265836350917e-05,
                -8.2311371696998123e-05, -8.2311424627576717e-05
        },
// x = -9.4000
        {
                -8.2696697566730747e-05, -8.2710380712717874e-05, -8.2717222851674168e-05, -8.2720644109818265e-05,
                -8.2722354801781893e-05, -8.2723210168728301e-05, -8.2723637859189893e-05, -8.2723851706750171e-05,
                -8.2723958631306837e-05, -8.2724012093844009e-05
        },
// x = -9.3950
        {
                -8.3111078643824089e-05, -8.3124899290523598e-05, -8.3131810188388791e-05, -8.3135265828838498e-05,
                -8.3136993712905349e-05, -8.3137857676220193e-05, -8.3138289664971618e-05, -8.3138505661712036e-05,
                -8.3138613660870480e-05, -8.3138667660712472e-05
        },
// x = -9.3900
        {
                -8.3527535429610644e-05, -8.3541494958667875e-05, -8.3548475306393887e-05, -8.3551965674668214e-05,
                -8.3553710923612193e-05, -8.3554583569687236e-05, -8.3555019899925941e-05, -8.3555238067445744e-05,
                -8.3555347152005813e-05, -8.3555401694552562e-05
        },
// x = -9.3850
        {
                -8.3946078314706986e-05, -8.3960178121649133e-05, -8.3967228617130765e-05, -8.3970754062220879e-05,
                -8.3972516850552132e-05, -8.3973398266647272e-05, -8.3973838982004872e-05, -8.3974059342120376e-05,
                -8.3974169522990405e-05, -8.3974224613696179e-05
        },
// x = -9.3800
        {
                -8.4366717741673073e-05, -8.4380959236048640e-05, -8.4388080584193309e-05, -8.4391641458597298e-05,
                -8.4393421962579681e-05, -8.4394312236831800e-05, -8.4394757381378349e-05, -8.4394979956125177e-05,
                -8.4395091244323126e-05, -8.4395146888696946e-05
        },
// x = -9.3750
        {
                -8.4789464205271164e-05, -8.4803848810790800e-05, -8.4811041723589032e-05, -8.4814638383347234e-05,
                -8.4816436781015919e-05, -8.4817336002447610e-05, -8.4817785620696100e-05, -8.4818010432331282e-05,
                -8.4818122838985864e-05, -8.4818179042592154e-05
        },
// x = -9.3700
        {
                -8.5214328252726203e-05, -8.5228857407404967e-05, -8.5236122604001531e-05, -8.5239755408732069e-05,
                -8.5241571879911379e-05, -8.5242480138439879e-05, -8.5242934275350641e-05, -8.5243161346354880e-05,
                -8.5243274882706641e-05, -8.5243331651165742e-05
        },
// x = -9.3650
        {
                -8.5641320483987183e-05, -8.5655995640288252e-05, -8.5663333847054029e-05, -8.5667003159988809e-05,
                -8.5668837886310214e-05, -8.5669755272756401e-05, -8.5670213973741528e-05, -8.5670443326821513e-05,
                -8.5670558004223988e-05, -8.5670615343212716e-05
        },
// x = -9.3600
        {
                -8.6070451551990062e-05, -8.6085274176969919e-05, -8.6092686127574571e-05, -8.6096392315595561e-05,
                -8.6098245480515704e-05, -8.6099172086613179e-05, -8.6099635397541253e-05, -8.6099867055631818e-05,
                -8.6099982885552600e-05, -8.6100040800804838e-05
        },
// x = -9.3550
        {
                -8.6501732162921929e-05, -8.6516703738376868e-05, -8.6524190173862105e-05, -8.6527933607538006e-05,
                -8.6529805396357202e-05, -8.6530741314761435e-05, -8.6531209281961975e-05, -8.6531443268228454e-05,
                -8.6531560262250441e-05, -8.6531618759557672e-05
        },
// x = -9.3500
        {
                -8.6935173076486364e-05, -8.6950295099100672e-05, -8.6957856767954340e-05, -8.6961637821577668e-05,
                -8.6963528421458378e-05, -8.6964473745755979e-05, -8.6964946416024099e-05, -8.6965182753864632e-05,
                -8.6965300923687083e-05, -8.6965360008899053e-05
        },
// x = -9.3450
        {
                -8.7370785106170273e-05, -8.7386059087665743e-05, -8.7393696745896636e-05, -8.7397515797521085e-05,
                -8.7399425397506603e-05, -8.7400380222224714e-05, -8.7400857642825773e-05, -8.7401096355873705e-05,
                -8.7401215713313489e-05, -8.7401275392338652e-05
        },
// x = -9.3400
        {
                -8.7808579119511999e-05, -8.7824006586798914e-05, -8.7831720998012385e-05, -8.7835578429490681e-05,
                -8.7837507220524065e-05, -8.7838471641139753e-05, -8.7838953859814182e-05, -8.7839194971940296e-05,
                -8.7839315528933016e-05, -8.7839375807739254e-05
        },
// x = -9.3350
        {
                -8.8248566038370634e-05, -8.8264148533700483e-05, -8.8271940469174674e-05, -8.8275836666196723e-05,
                -8.8277784841139827e-05, -8.8278758954089717e-05, -8.8279246019057651e-05, -8.8279489554372707e-05,
                -8.8279611322973918e-05, -8.8279672207589111e-05
        },
// x = -9.3300
        {
                -8.8690756839196802e-05, -8.8706495920316342e-05, -8.8714366159079339e-05, -8.8718301511210747e-05,
                -8.8720269264863625e-05, -8.8721253167553415e-05, -8.8721745127519683e-05, -8.8721991110376664e-05,
                -8.8722114102763108e-05, -8.8722175599275662e-05
        },
// x = -9.3250
        {
                -8.9135162553304807e-05, -8.9151059793611659e-05, -8.9159009122519368e-05, -8.9162984023240452e-05,
                -8.9164971552360673e-05, -8.9165965343175016e-05, -8.9166462247333855e-05, -8.9166710702330530e-05,
                -8.9166834930801315e-05, -8.9166897045360864e-05
        },
// x = -9.3200
        {
                -8.9581794267146054e-05, -8.9597851255845994e-05, -8.9605880469660729e-05, -8.9609895316405814e-05,
                -8.9611902819728299e-05, -8.9612906598040531e-05, -8.9613408496080533e-05, -8.9613659448061910e-05,
                -8.9613784925039744e-05, -8.9613847663857710e-05
        },
// x = -9.3150
        {
                -9.0030663122583744e-05, -9.0046881464849487e-05, -9.0054991366319469e-05, -9.0059046560516618e-05,
                -9.0061074238773363e-05, -9.0062088104955466e-05, -9.0062595047064702e-05, -9.0062848521125439e-05,
                -9.0062975259157861e-05, -9.0063038628508088e-05
        },
// x = -9.3100
        {
                -9.0481780317169133e-05, -9.0498161634300779e-05, -9.0506353034240327e-05, -9.0510448981351459e-05,
                -9.0512497037291672e-05, -9.0513521092724382e-05, -9.0514033129595199e-05, -9.0514289151082155e-05,
                -9.0514417162842838e-05, -9.0514481169062257e-05
        },
// x = -9.3050
        {
                -9.0935157104419029e-05, -9.0951703034005961e-05, -9.0959976751376565e-05, -9.0964113860938044e-05,
                -9.0966182499348447e-05, -9.0967216846431272e-05, -9.0967734029265482e-05, -9.0967992623780275e-05,
                -9.0968121922070246e-05, -9.0968186571559418e-05
        },
// x = -9.3000
        {
                -9.1390804794094720e-05, -9.1407516990179332e-05, -9.1415873852171496e-05, -9.1420052537834979e-05,
                -9.1422141965560232e-05, -9.1423186707721775e-05, -9.1423709088235796e-05, -9.1423970281637285e-05,
                -9.1424100879386222e-05, -9.1424166178610082e-05
        },
// x = -9.2950
        {
                -9.1848734752482142e-05, -9.1865614885725116e-05, -9.1874055727840984e-05, -9.1878276407414877e-05,
                -9.1880386833378258e-05, -9.1881442075086536e-05, -9.1881969705516484e-05, -9.1882233523923478e-05,
                -9.1882365434190990e-05, -9.1882431389679416e-05
        },
// x = -9.2900
        {
                -9.2308958402673654e-05, -9.2326008160520813e-05, -9.2334533826657792e-05, -9.2338796922148940e-05,
                -9.2340928557373272e-05, -9.2341994404146146e-05, -9.2342527337253106e-05, -9.2342793807046847e-05,
                -9.2342927043023812e-05, -9.2342993661372332e-05
        },
// x = -9.2850
        {
                -9.2771487224851105e-05, -9.2788708311702117e-05, -9.2797319654237107e-05, -9.2801625591893008e-05,
                -9.2803778649521704e-05, -9.2804855207937454e-05, -9.2805393497012782e-05, -9.2805662644839651e-05,
                -9.2805797219849506e-05, -9.2805864507719917e-05
        },
// x = -9.2800
        {
                -9.3236332756570182e-05, -9.3253726893948851e-05, -9.3262424773823526e-05, -9.3266773984174912e-05,
                -9.3268948679493349e-05, -9.3270036057201326e-05, -9.3270579756071865e-05, -9.3270851608846057e-05,
                -9.3270987536346148e-05, -9.3271055500467200e-05
        },
// x = -9.2750
        {
                -9.3703506593046423e-05, -9.3721075519772679e-05, -9.3729860806579483e-05, -9.3734253724483434e-05,
                -9.3736450274940406e-05, -9.3737548580671761e-05, -9.3738097743705358e-05, -9.3738372328611548e-05,
                -9.3738509622194464e-05, -9.3738578269362532e-05
        },
// x = -9.2700
        {
                -9.4173020387442475e-05, -9.4190765859806079e-05, -9.4199639431875176e-05, -9.4204076496558512e-05,
                -9.4206295121788023e-05, -9.4207404465366623e-05, -9.4207959147477507e-05, -9.4208236491973563e-05,
                -9.4208375165368501e-05, -9.4208444502448253e-05
        },
// x = -9.2650
        {
                -9.4644885851156640e-05, -9.4662809643092854e-05, -9.4671772387579827e-05, -9.4676254042683075e-05,
                -9.4678494964526265e-05, -9.4679615456879653e-05, -9.4680175713533928e-05, -9.4680455845353673e-05,
                -9.4680595912427769e-05, -9.4680665946352894e-05
        },
// x = -9.2600
        {
                -9.5119114754113298e-05, -9.5137218657379930e-05, -9.5146271470354632e-05, -9.5150798163976273e-05,
                -9.5153061606503660e-05, -9.5154193359674142e-05, -9.5154759246895298e-05, -9.5155042194051265e-05,
                -9.5155183668811083e-05, -9.5155254406584916e-05
        },
// x = -9.2550
        {
                -9.5595718925054276e-05, -9.5614004749410822e-05, -9.5623148535946728e-05, -9.5627720720688151e-05,
                -9.5630006910221911e-05, -9.5631150037377792e-05, -9.5631721611752374e-05, -9.5632007402538633e-05,
                -9.5632150299131426e-05, -9.5632221747827737e-05
        },
// x = -9.2500
        {
                -9.6074710251831919e-05, -9.6093179825220378e-05, -9.6102415499485196e-05, -9.6107033632495829e-05,
                -9.6109342797632522e-05, -9.6110497413079353e-05, -9.6111074731762569e-05, -9.6111363394757552e-05,
                -9.6111507727472833e-05, -9.6111579894236419e-05
        },
// x = -9.2450
        {
                -9.6556100681703590e-05, -9.6574755850431128e-05, -9.6584084335778103e-05, -9.6588748878801084e-05,
                -9.6591081250434519e-05, -9.6592247469626564e-05, -9.6592830590348059e-05, -9.6593122154417361e-05,
                -9.6593267937688219e-05, -9.6593340829735747e-05
        },
// x = -9.2400
        {
                -9.7039902221627598e-05, -9.7058744850550980e-05, -9.7068167079611173e-05, -9.7072878499029647e-05,
                -9.7075234310373866e-05, -9.7076412249925715e-05, -9.7077001230995211e-05, -9.7077295725294600e-05,
                -9.7077442973699164e-05, -9.7077516598319778e-05
        },
// x = -9.2350
        {
                -9.7526126938560562e-05, -9.7545158911272557e-05, -9.7554675826048060e-05, -9.7559434592931662e-05,
                -9.7561814079544400e-05, -9.7563003857242489e-05, -9.7563598757555799e-05, -9.7563896211533955e-05,
                -9.7564044939796896e-05, -9.7564119304352994e-05
        },
// x = -9.2300
        {
                -9.8014786959756326e-05, -9.8034010178773816e-05, -9.8043622730731981e-05, -9.8048429320883986e-05,
                -9.8050832720690062e-05, -9.8052034455504560e-05, -9.8052635334549323e-05, -9.8052935777950966e-05,
                -9.8053086000944909e-05, -9.8053161112872939e-05
        },
// x = -9.2250
        {
                -9.8505894473066205e-05, -9.8525310860020406e-05, -9.8535020010188809e-05, -9.8539874904193689e-05,
                -9.8542302457508901e-05, -9.8543516269605593e-05, -9.8544123187467318e-05, -9.8544426650336070e-05,
                -9.8544578383083097e-05, -9.8544654249894180e-05
        },
// x = -9.2200
        {
                -9.8999461727240974e-05, -9.9019073223069354e-05, -9.9028879942131924e-05, -9.9033783625403403e-05,
                -9.9036235574958553e-05, -9.9037461585710755e-05, -9.9038074603078797e-05, -9.9038381115760203e-05,
                -9.9038534373433411e-05, -9.9038611002714172e-05
        },
// x = -9.2150
        {
                -9.9495501032234034e-05, -9.9515309597374266e-05, -9.9525214865768346e-05, -9.9530167828597893e-05,
                -9.9532644419562981e-05, -9.9533882751563833e-05, -9.9534501929737397e-05, -9.9534811522882001e-05,
                -9.9534966320806927e-05, -9.9535043720220276e-05
        },
// x = -9.2100
        {
                -9.9994024759506354e-05, -1.0001403237409220e-04, -1.0002403718210641e-04, -1.0002903991971230e-04,
                -1.0003154139972109e-04, -1.0003279217679569e-04, -1.0003341757769009e-04, -1.0003373028225643e-04,
                -1.0003388663591267e-04, -1.0003396481319849e-04
        },
// x = -9.2050
        {
                -1.0049504534233279e-04, -1.0051525400639194e-04, -1.0052535935426525e-04, -1.0053041236684200e-04,
                -1.0053293898601681e-04, -1.0053420233323460e-04, -1.0053483401938738e-04, -1.0053514986664515e-04,
                -1.0053530779166784e-04, -1.0053538675464383e-04
        },
// x = -9.2000
        {
                -1.0099857527610983e-04, -1.0101898700976381e-04, -1.0102919390778547e-04, -1.0103429770055387e-04,
                -1.0103684971153048e-04, -1.0103812575521789e-04, -1.0103876378979501e-04, -1.0103908281132814e-04,
                -1.0103924232350961e-04, -1.0103932208007198e-04
        },
// x = -9.1950
        {
                -1.0150462711866517e-04, -1.0152524396233101e-04, -1.0153555343094153e-04, -1.0154070851419925e-04,
                -1.0154328617215222e-04, -1.0154457503990507e-04, -1.0154521948670738e-04, -1.0154554171441725e-04,
                -1.0154570282970846e-04, -1.0154578338783284e-04
        },
// x = -9.1900
        {
                -1.0201321349056835e-04, -1.0203403750516268e-04, -1.0204445057505586e-04, -1.0204965746422821e-04,
                -1.0205226102689643e-04, -1.0205356284759291e-04, -1.0205421377106237e-04, -1.0205453923717093e-04,
                -1.0205470197168320e-04, -1.0205478333942533e-04
        },
// x = -9.1850
        {
                -1.0252434707544348e-04, -1.0254538034258829e-04, -1.0255589805481412e-04, -1.0256115727050583e-04,
                -1.0256378699821821e-04, -1.0256510190203162e-04, -1.0256575936725784e-04, -1.0256608810431090e-04,
                -1.0256625247431742e-04, -1.0256633465981406e-04
        },
// x = -9.1800
        {
                -1.0303804062028301e-04, -1.0305928524251368e-04, -1.0306990864858252e-04, -1.0307522071662964e-04,
                -1.0307787687233116e-04, -1.0307920499074302e-04, -1.0307986906346973e-04, -1.0308020110434015e-04,
                -1.0308036712627772e-04, -1.0308045013774730e-04
        },
// x = -9.1750
        {
                -1.0355430693576345e-04, -1.0357576503673879e-04, -1.0358649519872637e-04, -1.0359186065024887e-04,
                -1.0359454349952692e-04, -1.0359588496534000e-04, -1.0359655571197168e-04, -1.0359689108986267e-04,
                -1.0359705878033324e-04, -1.0359714262607689e-04
        },
// x = -9.1700
        {
                -1.0407315889656223e-04, -1.0409483262127677e-04, -1.0410567061193033e-04, -1.0411108998338522e-04,
                -1.0411379979449608e-04, -1.0411515474184781e-04, -1.0411583222945621e-04, -1.0411617097790473e-04,
                -1.0411634035367710e-04, -1.0411642504207934e-04
        },
// x = -9.1650
        {
                -1.0459460944167666e-04, -1.0461650095667484e-04, -1.0462744785952045e-04, -1.0463292169275537e-04,
                -1.0463565873665103e-04, -1.0463702730102682e-04, -1.0463771159735780e-04, -1.0463805375023779e-04,
                -1.0463822482824933e-04, -1.0463831036777892e-04
        },
// x = -9.1600
        {
                -1.0511867157474401e-04, -1.0514078306833670e-04, -1.0515183997778740e-04, -1.0515736882009478e-04,
                -1.0516013337045013e-04, -1.0516151568869692e-04, -1.0516220686217719e-04, -1.0516255245370304e-04,
                -1.0516272525106124e-04, -1.0516281165027211e-04
        },
// x = -9.1550
        {
                -1.0564535836436329e-04, -1.0566769204684641e-04, -1.0567886006831175e-04, -1.0568444447248352e-04,

                -1.0568723680572358e-04, -1.0568863301606361e-04, -1.0568933113580745e-04, -1.0568968020053744e-04,
                -1.0568985473452180e-04, -1.0568994200205379e-04
        },
// x = -9.1500
        {
                -1.0617468294441883e-04, -1.0619724104829423e-04, -1.0620852129829061e-04, -1.0621416182267349e-04,
                -1.0621698221800112e-04, -1.0621839246004565e-04, -1.0621909759586200e-04, -1.0621945016870165e-04,
                -1.0621962645676532e-04, -1.0621971460134512e-04
        },
// x = -9.1450
        {
                -1.0670665851440520e-04, -1.0672944329460365e-04, -1.0674083690086593e-04, -1.0674653410941723e-04,
                -1.0674938284884105e-04, -1.0675080726360437e-04, -1.0675151948600369e-04, -1.0675187560220934e-04,
                -1.0675205366198088e-04, -1.0675214269242288e-04
        },
// x = -9.1400
        {
                -1.0724129833975382e-04, -1.0726431207386036e-04, -1.0727582017545470e-04, -1.0728157463779863e-04,
                -1.0728445200616115e-04, -1.0728589073607466e-04, -1.0728661011627603e-04, -1.0728696981145836e-04,
                -1.0728714966074345e-04, -1.0728723958595062e-04
        },
// x = -9.1350
        {
                -1.0777861575216128e-04, -1.0780186074064275e-04, -1.0781348448808022e-04, -1.0781929677956497e-04,
                -1.0782220306457112e-04, -1.0782365625349759e-04, -1.0782438286343580e-04, -1.0782474617356336e-04,
                -1.0782492783034667e-04, -1.0782501865931149e-04
        },
// x = -9.1300
        {
                -1.0831862414991911e-04, -1.0834210271635402e-04, -1.0835384327170576e-04, -1.0835971397346086e-04,
                -1.0836264946570676e-04, -1.0836411725895467e-04, -1.0836485117128747e-04, -1.0836521813269028e-04,
                -1.0836540161513721e-04, -1.0836549335694249e-04
        },
// x = -9.1250
        {
                -1.0886133699824539e-04, -1.0888505148955601e-04, -1.0889691002656933e-04, -1.0890283972556383e-04,
                -1.0890580471856583e-04, -1.0890728726290393e-04, -1.0890802855101922e-04, -1.0890839920039245e-04,
                -1.0890858452685093e-04, -1.0890867719067080e-04
        },
// x = -9.1200
        {
                -1.0940676782961783e-04, -1.0943072061630464e-04, -1.0944269832052031e-04, -1.0944868760962142e-04,
                -1.0945168239984540e-04, -1.0945317984351744e-04, -1.0945392858154069e-04, -1.0945430295594820e-04,
                -1.0945449014495061e-04, -1.0945458374005139e-04
        },
// x = -9.1150
        {
                -1.0995493024410844e-04, -1.0997912372048710e-04, -1.0999122178935782e-04, -1.0999727126739022e-04,
                -1.1000029615428126e-04, -1.1000180864702069e-04, -1.1000256490982227e-04, -1.1000294304670049e-04,
                -1.1000313211696545e-04, -1.1000322665270656e-04
        },
// x = -9.1100
        {
                -1.1050583790972012e-04, -1.1053027449416062e-04, -1.1054249413717069e-04, -1.1054860440897633e-04,
                -1.1055165969498869e-04, -1.1055318738803372e-04, -1.1055395125123643e-04, -1.1055433318839800e-04,
                -1.1055452415883221e-04, -1.1055461964466715e-04
        },
// x = -9.1050
        {
                -1.1105950456272478e-04, -1.1108418669789298e-04, -1.1109652913667921e-04, -1.1110270081317777e-04,
                -1.1110578680380507e-04, -1.1110732984991375e-04, -1.1110810138990038e-04, -1.1110848716553794e-04,
                -1.1110868005523816e-04, -1.1110877650071543e-04
        },
// x = -9.1000
        {
                -1.1161594400800289e-04, -1.1164087416110466e-04, -1.1165334062957840e-04, -1.1165957432782843e-04,
                -1.1166269133163419e-04, -1.1166424988509976e-04, -1.1166502917902071e-04, -1.1166541883171074e-04,
                -1.1166561365996563e-04, -1.1166571107472971e-04
        },
// x = -9.0950
        {
                -1.1217517011938517e-04, -1.1220035078241286e-04, -1.1221294252688339e-04, -1.1221923887014387e-04,
                -1.1222238719879242e-04, -1.1222396141545861e-04, -1.1222474854123963e-04, -1.1222514210994628e-04,
                -1.1222533889623835e-04, -1.1222543729003063e-04
        },
// x = -9.0900
        {
                -1.1273719683999554e-04, -1.1276263052997688e-04, -1.1277534880927593e-04, -1.1278170842706869e-04,
                -1.1278488839535627e-04, -1.1278647843263298e-04, -1.1278727346898294e-04, -1.1278767099306194e-04,
                -1.1278786975706950e-04, -1.1278796913972930e-04
        },
// x = -9.0850
        {
                -1.1330203818259610e-04, -1.1332772744184573e-04, -1.1334057352745325e-04, -1.1334699705562587e-04,
                -1.1335020898151209e-04, -1.1335181499839109e-04, -1.1335261802480986e-04, -1.1335301954401250e-04,
                -1.1335322030561161e-04, -1.1335332068707710e-04
        },
// x = -9.0800
        {
                -1.1386970822993352e-04, -1.1389565562630701e-04, -1.1390863080247829e-04, -1.1391511888326755e-04,
                -1.1391836308790724e-04, -1.1391998524497808e-04, -1.1392079634176447e-04, -1.1392120189624151e-04,
                -1.1392140467550800e-04, -1.1392150606581726e-04
        },
// x = -9.0750
        {
                -1.1444022113508740e-04, -1.1446642926223768e-04, -1.1447953482613170e-04, -1.1448608810822792e-04,
                -1.1448936491600320e-04, -1.1449100337546923e-04, -1.1449182262372904e-04, -1.1449223225403472e-04,
                -1.1449243707124619e-04, -1.1449253948053814e-04
        },
// x = -9.0700
        {
                -1.1501359112182012e-04, -1.1504006259945674e-04, -1.1505329986126571e-04, -1.1505991899987755e-04,
                -1.1506322873843022e-04, -1.1506488366412487e-04, -1.1506571114577898e-04, -1.1506612489287515e-04,
                -1.1506633176851297e-04, -1.1506643520702847e-04
        },
// x = -9.0650
        {
                -1.1558983248492864e-04, -1.1561656995907939e-04, -1.1562994024215979e-04, -1.1563662589907971e-04,
                -1.1563996889934407e-04, -1.1564164045674718e-04, -1.1564247625453973e-04, -1.1564289415979986e-04,
                -1.1564310311455127e-04, -1.1564320759263407e-04
        },
// x = -9.0600
        {
                -1.1616895959059790e-04, -1.1619596573387315e-04, -1.1620947037487798e-04, -1.1621622321854837e-04,
                -1.1621959981478430e-04, -1.1622128817103869e-04, -1.1622213236854539e-04, -1.1622255447375879e-04,
                -1.1622276552851886e-04, -1.1622287105661669e-04
        },
// x = -9.0550
        {
                -1.1675098687675590e-04, -1.1677826438861570e-04, -1.1679190473762796e-04, -1.1679872544320803e-04,
                -1.1680213597303436e-04, -1.1680384129696251e-04, -1.1680469397859899e-04, -1.1680512032597490e-04,
                -1.1680533350184877e-04, -1.1680544009051435e-04
        },
// x = -9.0500
        {
                -1.1733592885343074e-04, -1.1736348046045442e-04, -1.1737725788112209e-04, -1.1738414713055535e-04,
                -1.1738759193498366e-04, -1.1738931439710466e-04, -1.1739017564813486e-04, -1.1739060628030671e-04,
                -1.1739082159861160e-04, -1.1739092925850370e-04
        },
// x = -9.0450
        {
                -1.1792380010310927e-04, -1.1795162855926781e-04, -1.1796554442894023e-04, -1.1797250291102257e-04,
                -1.1797598233449122e-04, -1.1797772210703774e-04, -1.1797859201358249e-04, -1.1797902697361220e-04,
                -1.1797924445587956e-04, -1.1797935319776409e-04
        },
// x = -9.0400
        {
                -1.1851461528109751e-04, -1.1854272336802880e-04, -1.1855677907789412e-04, -1.1856380748834273e-04,
                -1.1856732187875137e-04, -1.1856907913568690e-04, -1.1856995778473250e-04, -1.1857039711611477e-04,
                -1.1857061678409244e-04, -1.1857072661884347e-04
        },
// x = -9.0350
        {
                -1.1910838911588301e-04, -1.1913677964316956e-04, -1.1915097659839392e-04, -1.1915807563991683e-04,
                -1.1916162534866115e-04, -1.1916340026569742e-04, -1.1916428774510435e-04, -1.1916473149177096e-04,
                -1.1916495336742536e-04, -1.1916506430602627e-04
        },
// x = -9.0300
        {
                -1.1970513640949867e-04, -1.1973381221494843e-04, -1.1974815183481628e-04, -1.1975532221718260e-04,
                -1.1975890759918942e-04, -1.1976070035380394e-04, -1.1976159675231570e-04, -1.1976204495863994e-04,
                -1.1976226906415824e-04, -1.1976238111770280e-04
        },
// x = -9.0250
        {
                -1.2030487203788876e-04, -1.2033383598781855e-04, -1.2034831970587460e-04, -1.2035556214598541e-04,
                -1.2035918355974824e-04, -1.2036099433120206e-04, -1.2036189973845392e-04, -1.2036235244925504e-04,
                -1.2036257880704737e-04, -1.2036269198674081e-04
        },
// x = -9.0200
        {
                -1.2090761095127645e-04, -1.2093686594079835e-04, -1.2095149520499068e-04, -1.2095881042695080e-04,
                -1.2096246823456557e-04, -1.2096429720392118e-04, -1.2096521171044925e-04, -1.2096566897099693e-04,
                -1.2096589760369868e-04, -1.2096601192085887e-04
        },
// x = -9.0150
        {
                -1.2151336817453329e-04, -1.2154291712784380e-04, -1.2155769340066869e-04, -1.2156508213585891e-04,
                -1.2156877670306031e-04, -1.2157062405319981e-04, -1.2157154775045004e-04, -1.2157200960646884e-04,
                -1.2157224053694288e-04, -1.2157235600300145e-04
        },
// x = -9.0100
        {
                -1.2212215880755042e-04, -1.2215200467822262e-04, -1.2216692943687064e-04, -1.2217439242402103e-04,
                -1.2217812412021890e-04, -1.2217999003586222e-04, -1.2218092301619955e-04, -1.2218138951387364e-04,
                -1.2218162276521257e-04, -1.2218173939171597e-04
        },
// x = -9.0050
        {
                -1.2273399802561176e-04, -1.2276414379689034e-04, -1.2277921853339402e-04, -1.2278675651865745e-04,
                -1.2279052571697392e-04, -1.2279241038469727e-04, -1.2279335274141488e-04, -1.2279382392739257e-04,
                -1.2279405952292109e-04, -1.2279417732153191e-04
        },
// x = -9.0000
        {
                -1.2334890107976878e-04, -1.2337934976486818e-04, -1.2339457598625100e-04, -1.2340218972327810e-04,
                -1.2340599680058469e-04, -1.2340790040883928e-04, -1.2340885223616796e-04, -1.2340932815756630e-04,
                -1.2340956612084351e-04, -1.2340968510334146e-04
        },
// x = -8.9950
        {
                -1.2396688329721758e-04, -1.2399763793962277e-04, -1.2401301716804993e-04, -1.2402070741806409e-04,
                -1.2402455275501955e-04, -1.2402647549415041e-04, -1.2402743688726781e-04, -1.2402791759167739e-04,
                -1.2402815794649921e-04, -1.2402827812478247e-04
        },
// x = -8.9900
        {
                -1.2458796008167722e-04, -1.2461902375544796e-04, -1.2463455752837827e-04, -1.2464232506025205e-04,
                -1.2464620904134024e-04, -1.2464815110360520e-04, -1.2464912215864556e-04, -1.2464960769413531e-04,
                -1.2464985046453674e-04, -1.2464997185062299e-04
        },
// x = -8.9850
        {
                -1.2521214691377035e-04, -1.2524352272384821e-04, -1.2525921259418787e-04, -1.2526705818451954e-04,
                -1.2527098119808806e-04, -1.2527294277767706e-04, -1.2527392359174076e-04, -1.2527441400686257e-04,
                -1.2527465921712022e-04, -1.2527478182314795e-04
        },
// x = -8.9800
        {
                -1.2583945935140578e-04, -1.2587115043392411e-04, -1.2588699797018207e-04, -1.2589492240337339e-04,
                -1.2589888484167226e-04, -1.2590086613472665e-04, -1.2590185680588984e-04, -1.2590235214968363e-04,
                -1.2590259982431804e-04, -1.2590272366254772e-04
        },
// x = -8.9750
        {
                -1.2646991303016236e-04, -1.2650192255275999e-04, -1.2651792933920423e-04, -1.2652593340753898e-04,
                -1.2652993566675985e-04, -1.2653193687139211e-04, -1.2653293749871657e-04, -1.2653343782071514e-04,
                -1.2653368798449323e-04, -1.2653381306730859e-04
        },
// x = -8.9700
        {
                -1.2710352366367565e-04, -1.2713585482581287e-04, -1.2715202246262917e-04, -1.2716010696635195e-04,
                -1.2716414944666788e-04, -1.2716617076298143e-04, -1.2716718144652442e-04, -1.2716768679675827e-04,
                -1.2716793947469603e-04, -1.2716806581460519e-04
        },
// x = -8.9650
        {
                -1.2774030704402543e-04, -1.2777296307730395e-04, -1.2778929318075558e-04, -1.2779745892815223e-04,
                -1.2780154203375746e-04, -1.2780358366386657e-04, -1.2780460450469101e-04, -1.2780511493369341e-04,
                -1.2780537015105812e-04, -1.2780549776069493e-04
        },
// x = -8.9600
        {
                -1.2838027904212623e-04, -1.2841326321061180e-04, -1.2842975741320103e-04, -1.2843800522067912e-04,
                -1.2844212935982969e-04, -1.2844419150787972e-04, -1.2844522260806407e-04, -1.2844573816687626e-04,
                -1.2844599594918912e-04, -1.2844612484131447e-04
        },
// x = -8.9550
        {
                -1.2902345560811897e-04, -1.2905677120866749e-04, -1.2907343115929872e-04, -1.2908176185146938e-04,
                -1.2908592743652377e-04, -1.2908801030871164e-04, -1.2908905177136024e-04, -1.2908957251153634e-04,
                -1.2908983288457504e-04, -1.2908996307207796e-04
        },
// x = -8.9500
        {
                -1.2966985277176472e-04, -1.2970350313435151e-04, -1.2972033049849616e-04, -1.2972874490825638e-04,
                -1.2973295235571680e-04, -1.2973505616031155e-04, -1.2973610808956487e-04, -1.2973663406317713e-04,
                -1.2973689705297851e-04, -1.2973702854887764e-04
        },
// x = -8.9450
        {
                -1.3031948664284063e-04, -1.3035347513089325e-04, -1.3037047159075604e-04, -1.3037897055937181e-04,
                -1.3038322028992592e-04, -1.3038534523728950e-04, -1.3038640773833463e-04, -1.3038693899797862e-04,
                -1.3038720463084112e-04, -1.3038733744828590e-04
        },
// x = -8.9400
        {
                -1.3097237341153767e-04, -1.3100670342227168e-04, -1.3102387067695883e-04, -1.3103245505414950e-04,
                -1.3103674749271227e-04, -1.3103889379532078e-04, -1.3103996697440187e-04, -1.3104050357320160e-04,
                -1.3104077187568801e-04, -1.3104090602796003e-04
        },
// x = -8.9350
        {
                -1.3162852934886022e-04, -1.3166320431361867e-04, -1.3168054407930769e-04, -1.3168921472333063e-04,
                -1.3169355029908706e-04, -1.3169571817155164e-04, -1.3169680213598054e-04, -1.3169734412759413e-04,
                -1.3169761512653406e-04, -1.3169775062704843e-04
        },
// x = -8.9300
        {
                -1.3228797080702779e-04, -1.3232299419162400e-04, -1.3234050820173511e-04, -1.3234926597947169e-04,
                -1.3235364512591949e-04, -1.3235583478500804e-04, -1.3235692964317505e-04, -1.3235747708179976e-04,
                -1.3235775080429256e-04, -1.3235788766659918e-04
        },
// x = -8.9250
        {
                -1.3295071421987888e-04, -1.3298608952494245e-04, -1.3300377953031180e-04, -1.3301262531735387e-04,
                -1.3301704847234696e-04, -1.3301926013700578e-04, -1.3302036599839047e-04, -1.3302091893876820e-04,
                -1.3302119541218563e-04, -1.3302133364997054e-04
        },
// x = -8.9200
        {
                -1.3361677610327605e-04, -1.3365250686460273e-04, -1.3367037463365739e-04, -1.3367930931439503e-04,
                -1.3368377692018723e-04, -1.3368601081156279e-04, -1.3368712778674495e-04, -1.3368768628416780e-04,
                -1.3368796553615660e-04, -1.3368810516324343e-04
        },
// x = -8.9150
        {
                -1.3428617305551413e-04, -1.3432226284441884e-04, -1.3434031016335365e-04, -1.3434933463106328e-04,
                -1.3435384713435255e-04, -1.3435610347581375e-04, -1.3435723167648449e-04, -1.3435779578680021e-04,
                -1.3435807784528494e-04, -1.3435821887563626e-04
        },
// x = -8.9100
        {
                -1.3495892175772954e-04, -1.3499537418140337e-04, -1.3501360285435909e-04, -1.3502271801129297e-04,
                -1.3502727586326598e-04, -1.3502955488042641e-04, -1.3503069441939925e-04, -1.3503126419901683e-04,
                -1.3503154909220278e-04, -1.3503169153992147e-04
        },
// x = -8.9050
        {
                -1.3563503897431211e-04, -1.3567185767618232e-04, -1.3569026952542623e-04, -1.3569947628290250e-04,
                -1.3570407993927997e-04, -1.3570638186002042e-04, -1.3570753285124256e-04, -1.3570810835713794e-04,
                -1.3570839611351380e-04, -1.3570853999284448e-04
        },
// x = -8.9000
        {
                -1.3631454155331864e-04, -1.3635173021341276e-04, -1.3637032707952053e-04, -1.3637962635801442e-04,
                -1.3638427627909636e-04, -1.3638660133358775e-04, -1.3638776389215157e-04, -1.3638834518187322e-04,
                -1.3638863583021401e-04, -1.3638878115554442e-04
        },
// x = -8.8950
        {
                -1.3699744642688854e-04, -1.3703500876220196e-04, -1.3705379250424180e-04, -1.3706318523347758e-04,
                -1.3706788188418950e-04, -1.3707023030491564e-04, -1.3707140454707019e-04, -1.3707199167874496e-04,
                -1.3707228524811491e-04, -1.3707243203397741e-04
        },
// x = -8.8900
        {
                -1.3768377061166196e-04, -1.3772171037652916e-04, -1.3774068287224727e-04, -1.3775016999129113e-04,
                -1.3775491384123058e-04, -1.3775728586301169e-04, -1.3775847190617414e-04, -1.3775906493851300e-04,
                -1.3775936145826845e-04, -1.3775950971934146e-04
        },
// x = -8.8850
        {
                -1.3837353120919914e-04, -1.3841185219566867e-04, -1.3843101534167718e-04, -1.3844059779903118e-04,
                -1.3844538932251439e-04, -1.3844778518253047e-04, -1.3844898314529808e-04, -1.3844958213760216e-04,
                -1.3844988163739432e-04, -1.3845003138850381e-04
        },
// x = -8.8800
        {
                -1.3906674540640267e-04, -1.3910545144461603e-04, -1.3912480715658237e-04, -1.3913448591027912e-04,
                -1.3913932558638878e-04, -1.3914174552420320e-04, -1.3914295552636512e-04, -1.3914356053853127e-04,
                -1.3914386304830960e-04, -1.3914401430443045e-04
        },
// x = -8.8750
        {
                -1.3976343047594135e-04, -1.3980252543451536e-04, -1.3982207564735367e-04, -1.3983185166505219e-04,
                -1.3983673997768512e-04, -1.3983918423526878e-04, -1.3984040639781784e-04, -1.3984101749034517e-04,
                -1.3984132304035985e-04, -1.3984147581661753e-04
        },
// x = -8.8700
        {
                -1.4046360377667610e-04, -1.4050309156308949e-04, -1.4052283823115402e-04, -1.4053271249023637e-04,
                -1.4053764992815200e-04, -1.4054011874990735e-04, -1.4054135319505244e-04, -1.4054197042904782e-04,
                -1.4054227904985317e-04, -1.4054243336152514e-04
        },
// x = -8.8650
        {
                -1.4116728275408830e-04, -1.4120716731507197e-04, -1.4122711241235228e-04, -1.4123708590002142e-04,
                -1.4124207295689048e-04, -1.4124456658967611e-04, -1.4124581344085417e-04, -1.4124643687803864e-04,
                -1.4124674860049611e-04, -1.4124690446301331e-04
        },
// x = -8.8600
        {
                -1.4187448494070991e-04, -1.4191477026264112e-04, -1.4193491578295942e-04, -1.4194498949633762e-04,
                -1.4195002667079178e-04, -1.4195254536394689e-04, -1.4195380474583535e-04, -1.4195443444855029e-04,
                -1.4195474930383146e-04, -1.4195490673277991e-04
        },
// x = -8.8550
        {
                -1.4258522795655606e-04, -1.4262591806585645e-04, -1.4264626602306674e-04, -1.4265644096929549e-04,
                -1.4266152876497693e-04, -1.4266407277034645e-04, -1.4266534480887578e-04, -1.4266598084008904e-04,
                -1.4266629885967866e-04, -1.4266645787080112e-04
        },
// x = -8.8500
        {
                -1.4329952950955919e-04, -1.4334062847309723e-04, -1.4336118090128662e-04, -1.4337145809762707e-04,
                -1.4337659702323897e-04, -1.4337916659519872e-04, -1.4338045141756487e-04, -1.4338109384087709e-04,
                -1.4338141505657636e-04, -1.4338157566577375e-04
        },
// x = -8.8450
        {
                -1.4401740739600601e-04, -1.4405891932150303e-04, -1.4407967827519491e-04, -1.4409005874912973e-04,
                -1.4409524931848706e-04, -1.4409784471396903e-04, -1.4409914244864621e-04, -1.4409979132829723e-04,
                -1.4410011577222702e-04, -1.4410027799556004e-04
        },
// x = -8.8400
        {
                -1.4473887950097633e-04, -1.4478080853741674e-04, -1.4480177609177609e-04, -1.4481226088111204e-04,
                -1.4481750361319297e-04, -1.4482012509171123e-04, -1.4482143586846472e-04, -1.4482209126933998e-04,
                -1.4482241897394393e-04, -1.4482258282763467e-04
        },
// x = -8.8350
        {
                -1.4546396379878359e-04, -1.4550631413682948e-04, -1.4552749238787017e-04, -1.4553808254084193e-04,
                -1.4554337795983963e-04, -1.4554602578351612e-04, -1.4554734973341532e-04, -1.4554801172105233e-04,
                -1.4554834271910010e-04, -1.4554850821953377e-04
        },
// x = -8.8300
        {
                -1.4619267835341845e-04, -1.4623545422582786e-04, -1.4625684529062221e-04, -1.4626754186599701e-04,
                -1.4627289050137227e-04, -1.4627556493496295e-04, -1.4627690219039445e-04, -1.4627757083098941e-04,
                -1.4627790515557999e-04, -1.4627807231930640e-04
        },
// x = -8.8250
        {
                -1.4692504131899375e-04, -1.4696824700104376e-04, -1.4698985301793400e-04, -1.4700065708511759e-04,
                -1.4700605947165135e-04, -1.4700876078257288e-04, -1.4701011147725369e-04, -1.4701078683766790e-04,
                -1.4701112452223303e-04, -1.4701129336596832e-04
        },
// x = -8.8200
        {
                -1.4766107094019227e-04, -1.4770471075010568e-04, -1.4772653387891757e-04, -1.4773744651806100e-04,
                -1.4774290319590825e-04, -1.4774563165426455e-04, -1.4774699592325544e-04, -1.4774767807102226e-04,
                -1.4774801914932959e-04, -1.4774818968995792e-04
        },
// x = -8.8150
        {
                -1.4840078555271617e-04, -1.4844486385209301e-04, -1.4846690627435172e-04, -1.4847792857645929e-04,
                -1.4848344009120283e-04, -1.4848619596981216e-04, -1.4848757394953123e-04, -1.4848826295286267e-04,
                -1.4848860745901918e-04, -1.4848877971359440e-04
        },
// x = -8.8100
        {
                -1.4914420358373893e-04, -1.4918872477799215e-04, -1.4921098869714010e-04, -1.4922212176417848e-04,
                -1.4922768866688346e-04, -1.4923047224130564e-04, -1.4923186406954190e-04, -1.4923255999733554e-04,
                -1.4923290796579100e-04, -1.4923308195153833e-04
        },
// x = -8.8050
        {
                -1.4989134355235977e-04, -1.4993631209115485e-04, -1.4995879973277214e-04, -1.4997004467778047e-04,
                -1.4997566752504948e-04, -1.4997847907361351e-04, -1.4997988488954069e-04, -1.4998058781138648e-04,
                -1.4998093927693685e-04, -1.4998111501125463e-04
        },
// x = -8.8000
        {
                -1.5064222407005948e-04, -1.5068764444775928e-04, -1.5071035805978581e-04, -1.5072171600698692e-04,
                -1.5072739536101568e-04, -1.5073023516484749e-04, -1.5073165510903793e-04, -1.5073236509522515e-04,
                -1.5073272009301621e-04, -1.5073289759347762e-04
        },
// x = -8.7950
        {
                -1.5139686384115945e-04, -1.5144274059727286e-04, -1.5146568245023295e-04, -1.5147715453514576e-04,
                -1.5148289096377930e-04, -1.5148575930683001e-04, -1.5148719352126873e-04, -1.5148791064279306e-04,
                -1.5148826920832368e-04, -1.5148844849267851e-04
        },
// x = -8.7900
        {
                -1.5215528166328231e-04, -1.5220161938291760e-04, -1.5222479177014707e-04, -1.5223637913969998e-04,
                -1.5224217321648943e-04, -1.5224507038556350e-04, -1.5224651901366246e-04, -1.5224724334223310e-04,
                -1.5224760551135890e-04, -1.5224778659753539e-04
        },
// x = -8.7850
        {
                -1.5291749642781514e-04, -1.5296429974213799e-04, -1.5298770498001305e-04, -1.5299940879265876e-04,
                -1.5300526109691858e-04, -1.5300818738170271e-04, -1.5300965056831500e-04, -1.5301038217636174e-04,
                -1.5301074798529875e-04, -1.5301093089140522e-04
        },
// x = -8.7800
        {
                -1.5368352712037480e-04, -1.5373080070707078e-04, -1.5375444113523959e-04, -1.5376626256107065e-04,
                -1.5377217367793669e-04, -1.5377512937102860e-04, -1.5377660726246302e-04, -1.5377734622314362e-04,
                -1.5377771570847186e-04, -1.5377790045279868e-04
        },
// x = -8.7750
        {
                -1.5445339282127554e-04, -1.5450114140501726e-04, -1.5452501938663368e-04, -1.5453695960749968e-04,
                -1.5454293012798754e-04, -1.5454591552492534e-04, -1.5454740826896114e-04, -1.5454815465616850e-04,
                -1.5454852785483554e-04, -1.5454871445585682e-04
        },
// x = -8.7700
        {
                -1.5522711270599910e-04, -1.5527534105891819e-04, -1.5529945898087790e-04, -1.5531151919050332e-04,
                -1.5531754971156753e-04, -1.5532056511085923e-04, -1.5532207285676058e-04, -1.5532282674513031e-04,
                -1.5532320369445501e-04, -1.5532339217083068e-04
        },
// x = -8.7650
        {
                -1.5600470604566698e-04, -1.5605341898783042e-04, -1.5607777926100947e-04, -1.5608996066511315e-04,
                -1.5609605178970679e-04, -1.5609909749286018e-04, -1.5610062039139136e-04, -1.5610138185630904e-04,
                -1.5610176259398540e-04, -1.5610195296456275e-04
        },
// x = -8.7600
        {
                -1.5678619220751492e-04, -1.5683539460740667e-04, -1.5685999966690218e-04, -1.5687230348331783e-04,
                -1.5687845582045281e-04, -1.5688153213200547e-04, -1.5688307033544590e-04, -1.5688383945305475e-04,
                -1.5688422401715553e-04, -1.5688441630097141e-04
        },
// x = -8.7550
        {
                -1.5757159065536992e-04, -1.5762128743037707e-04, -1.5764613973575070e-04, -1.5765856719454857e-04,
                -1.5766478135935646e-04, -1.5766788858690640e-04, -1.5766944224906580e-04, -1.5767021909627422e-04,
                -1.5767060752525482e-04, -1.5767080174153731e-04
        },
// x = -8.7500
        {
                -1.5836092095012955e-04, -1.5841111706703335e-04, -1.5843621910255687e-04, -1.5844877144616674e-04,
                -1.5845504805996036e-04, -1.5845818651419669e-04, -1.5845975579043041e-04, -1.5846054044491974e-04,
                -1.5846093277762207e-04, -1.5846112894579251e-04
        },
// x = -8.7450
        {
                -1.5915420275024343e-04, -1.5920490322571544e-04, -1.5923025750061917e-04, -1.5924293598395433e-04,
                -1.5924927567428985e-04, -1.5925244566902366e-04, -1.5925403071624840e-04, -1.5925482325648070e-04,
                -1.5925521953213695e-04, -1.5925541767181187e-04
        },
// x = -8.7400
        {
                -1.5995145581219748e-04, -1.6000266571330034e-04, -1.6002827476202365e-04, -1.6004108065260656e-04,
                -1.6004748405334609e-04, -1.6005068590554218e-04, -1.6005228688225151e-04, -1.6005308738747728e-04,
                -1.6005348764571399e-04, -1.6005368777670704e-04
        },
// x = -8.7350
        {
                -1.6075269999099993e-04, -1.6080442443569358e-04, -1.6083029081813822e-04, -1.6084322539622711e-04,
                -1.6084969314760224e-04, -1.6085292717741046e-04, -1.6085454424369079e-04, -1.6085535279395698e-04,
                -1.6085575707479891e-04, -1.6085595921712286e-04
        },
// x = -8.7300
        {
                -1.6155795524067034e-04, -1.6161019939832315e-04, -1.6163632570010902e-04, -1.6164939025882573e-04,
                -1.6165592300750135e-04, -1.6165918953828893e-04, -1.6166082285583533e-04, -1.6166163953199334e-04,
                -1.6166204787586752e-04, -1.6166225204973633e-04
        },
// x = -8.7250
        {
                -1.6236724161473073e-04, -1.6242001070663562e-04, -1.6244639953935894e-04, -1.6245959538481835e-04,
                -1.6246619378395747e-04, -1.6246949314234106e-04, -1.6247114287447361e-04, -1.6247196775818747e-04,
                -1.6247238020592708e-04, -1.6247258643175785e-04
        },
// x = -8.7200
        {
                -1.6318057926669908e-04, -1.6323387856659495e-04, -1.6326053256808935e-04, -1.6327386101952973e-04,
                -1.6328052572885869e-04, -1.6328385824473718e-04, -1.6328552455641723e-04, -1.6328635773017150e-04,
                -1.6328677432302029e-04, -1.6328698262143524e-04
        },
// x = -8.7150
        {
                -1.6399798845058531e-04, -1.6405182328518381e-04, -1.6407874511978389e-04, -1.6409220750969878e-04,
                -1.6409893919557312e-04, -1.6410230520216062e-04, -1.6410398826000727e-04, -1.6410482980711561e-04,
                -1.6410525058673162e-04, -1.6410546097856031e-04
        },
// x = -8.7100
        {

                -1.6481948952138965e-04, -1.6487386527090699e-04, -1.6490105762971438e-04, -1.6491465530398588e-04,
                -1.6492145463945711e-04, -1.6492485447331610e-04, -1.6492655444562296e-04, -1.6492740445023620e-04,
                -1.6492782945869632e-04, -1.6492804196497756e-04
        },
// x = -8.7050
        {
                -1.6564510293560363e-04, -1.6570002503429795e-04, -1.6572749063545030e-04, -1.6574122495348345e-04,
                -1.6574809261836620e-04, -1.6575152661944142e-04, -1.6575324367619322e-04, -1.6575410222330800e-04,
                -1.6575453150311179e-04, -1.6575474614509590e-04
        },
// x = -8.7000
        {
                -1.6647484925171337e-04, -1.6653032318842740e-04, -1.6655806477736977e-04, -1.6657193711222852e-04,
                -1.6657887379316851e-04, -1.6658234230482074e-04, -1.6658407661771063e-04, -1.6658494379317754e-04,
                -1.6658537738725189e-04, -1.6658559418640271e-04
        },
// x = -8.6950
        {
                -1.6730874913070530e-04, -1.6736478044941444e-04, -1.6739280079917387e-04, -1.6740681253771814e-04,
                -1.6741381892826086e-04, -1.6741732229730140e-04, -1.6741907403974778e-04, -1.6741994993028044e-04,
                -1.6742038788198344e-04, -1.6742060685998055e-04
        },
// x = -8.6900
        {
                -1.6814682333657447e-04, -1.6820341763694045e-04, -1.6823171954840265e-04, -1.6824587209142712e-04,
                -1.6825294889208722e-04, -1.6825648746881240e-04, -1.6825825681597646e-04, -1.6825914150915981e-04,
                -1.6825958386228545e-04, -1.6825980504102630e-04
        },
// x = -8.6850
        {
                -1.6898909273683531e-04, -1.6904625567476530e-04, -1.6907484197695486e-04, -1.6908913673932877e-04,
                -1.6909628465765987e-04, -1.6909985879588627e-04, -1.6910164592468958e-04, -1.6910253950898875e-04,
                -1.6910298630777105e-04, -1.6910320970937311e-04
        },
// x = -8.6800
        {
                -1.6983557830303487e-04, -1.6989331559124648e-04, -1.6992218914160914e-04, -1.6993662755241786e-04,
                -1.6994384730308350e-04, -1.6994745736018311e-04, -1.6994926244932510e-04, -1.6995016501409431e-04,
                -1.6995061630321182e-04, -1.6995084195001498e-04
        },
// x = -8.6750
        {
                -1.7068630111126873e-04, -1.7074461851986018e-04, -1.7077378220454860e-04, -1.7078836570723642e-04,
                -1.7079565801208130e-04, -1.7079930434901741e-04, -1.7080112757899334e-04, -1.7080203921448475e-04,
                -1.7080249503906514e-04, -1.7080272295363365e-04
        },
// x = -8.6700
        {
                -1.7154128234269910e-04, -1.7160018569972556e-04, -1.7162964243388755e-04, -1.7164437248640173e-04,
                -1.7165173807452415e-04, -1.7165542105588734e-04, -1.7165726260900635e-04, -1.7165818340637916e-04,
                -1.7165864381200353e-04, -1.7165887401712846e-04
        },
// x = -8.6650
        {
                -1.7240054328407581e-04, -1.7246003847613142e-04, -1.7248979120420110e-04, -1.7250466927913789e-04,
                -1.7251210888696226e-04, -1.7251582888100697e-04, -1.7251768894141028e-04, -1.7251861899273977e-04,
                -1.7251908402544737e-04, -1.7251931654414885e-04
        },
// x = -8.6600
        {
                -1.7326410532825966e-04, -1.7332419830106533e-04, -1.7335424999705736e-04, -1.7336927758180895e-04,
                -1.7337679195315970e-04, -1.7338054933184086e-04, -1.7338242808552021e-04, -1.7338336748380705e-04,
                -1.7338383719009973e-04, -1.7338407204562921e-04
        },
// x = -8.6550
        {
                -1.7413198997474858e-04, -1.7419268673374554e-04, -1.7422304040155229e-04, -1.7423821899845547e-04,
                -1.7424580888463124e-04, -1.7424960402364161e-04, -1.7425150165845785e-04, -1.7425245049763728e-04,
                -1.7425292492448429e-04, -1.7425316214032694e-04
        },
// x = -8.6500
        {
                -1.7500421883020578e-04, -1.7506552544115539e-04, -1.7509618411484698e-04, -1.7511151524133336e-04,
                -1.7511918140118219e-04, -1.7512301467998987e-04, -1.7512493138569177e-04, -1.7512588976064300e-04,
                -1.7512636895548563e-04, -1.7512660855536263e-04
        },
// x = -8.6450
        {
                -1.7588081360899127e-04, -1.7594273619858073e-04, -1.7597370294270806e-04, -1.7598918813145574e-04,
                -1.7599693133145086e-04, -1.7600080313333715e-04, -1.7600273910158031e-04, -1.7600370710813614e-04,
                -1.7600419111889236e-04, -1.7600443312676335e-04
        },
// x = -8.6400
        {
                -1.7676179613369534e-04, -1.7682434089014940e-04, -1.7685561880005052e-04, -1.7687125959913708e-04,
                -1.7687908061345356e-04, -1.7688299132555144e-04, -1.7688494674991740e-04, -1.7688592448487372e-04,
                -1.7688641335994321e-04, -1.7688665780000849e-04
        },
// x = -8.6350
        {
                -1.7764718833567509e-04, -1.7771036150937404e-04, -1.7774195371148319e-04, -1.7775775168454055e-04,
                -1.7776565129513255e-04, -1.7776960130846530e-04, -1.7777157638448112e-04, -1.7777256394560650e-04,
                -1.7777305773387526e-04, -1.7777330463057841e-04
        },
// x = -8.6300
        {
                -1.7853701225559339e-04, -1.7860082015969692e-04, -1.7863272981185694e-04, -1.7864868653822761e-04,
                -1.7865666553490685e-04, -1.7866065524442705e-04, -1.7866265016958460e-04, -1.7866364765563021e-04,
                -1.7866414640647555e-04, -1.7866439578450579e-04
        },
// x = -8.6250
        {
                -1.7943129004396025e-04, -1.7949573905503814e-04, -1.7952796934681597e-04, -1.7954408642171087e-04,
                -1.7955214560222527e-04, -1.7955617540685437e-04, -1.7955819038063025e-04, -1.7955919789133966e-04,
                -1.7955970165463514e-04, -1.7955995353892982e-04
        },
// x = -8.6200
        {
                -1.8033004396167751e-04, -1.8039514052034617e-04, -1.8042769467335135e-04, -1.8044397370800913e-04,
                -1.8045211387812272e-04, -1.8045618418079096e-04, -1.8045821940466638e-04, -1.8045923704078555e-04,
                -1.8045974586690589e-04, -1.8046000028265304e-04
        },
// x = -8.6150
        {
                -1.8123329638058529e-04, -1.8129904699215103e-04, -1.8133192826035765e-04, -1.8134837088220587e-04,
                -1.8135659285577920e-04, -1.8136070406346585e-04, -1.8136275974094674e-04, -1.8136378760423409e-04,
                -1.8136430154406033e-04, -1.8136455851670106e-04
        },
// x = -8.6100
        {
                -1.8214106978401218e-04, -1.8220748101912033e-04, -1.8224069268919213e-04, -1.8225730054200978e-04,
                -1.8226560514108145e-04, -1.8226975766485551e-04, -1.8227183400149288e-04, -1.8227287219472945e-04,
                -1.8227339129965394e-04, -1.8227365085488504e-04
        },
// x = -8.6050
        {
                -1.8305338676732688e-04, -1.8312046526261830e-04, -1.8315401065423706e-04, -1.8317078539831878e-04,
                -1.8317917345318744e-04, -1.8318336770824865e-04, -1.8318546491165926e-04, -1.8318651353865900e-04,
                -1.8318703786059063e-04, -1.8318730002436708e-04
        },
// x = -8.6000
        {
                -1.8397027003849378e-04, -1.8403802249726732e-04, -1.8407190496346423e-04, -1.8408884827578642e-04,
                -1.8409732062509384e-04, -1.8410155703081429e-04, -1.8410367531070123e-04, -1.8410473447632143e-04,
                -1.8410526406769070e-04, -1.8410552886622845e-04
        },
// x = -8.5950
        {
                -1.8489174241863033e-04, -1.8496017561151217e-04, -1.8499439853900297e-04, -1.8501151211339120e-04,
                -1.8502006960420609e-04, -1.8502434858417216e-04, -1.8502648815234594e-04, -1.8502755796249759e-04,
                -1.8502809287626192e-04, -1.8502836033604036e-04
        },
// x = -8.5900
        {
                -1.8581782684256763e-04, -1.8588694760818722e-04, -1.8592151441771057e-04, -1.8593879996500892e-04,
                -1.8594744345291163e-04, -1.8595176543496621e-04, -1.8595392650536590e-04, -1.8595500706702439e-04,
                -1.8595554735667343e-04, -1.8595581750443800e-04
        },
// x = -8.5850
        {
                -1.8674854635941361e-04, -1.8681836160508659e-04, -1.8685327575174572e-04, -1.8687073499998769e-04,
                -1.8687946534915562e-04, -1.8688383076544100e-04, -1.8688601355415556e-04, -1.8688710497537133e-04,
                -1.8688765069493237e-04, -1.8688792355769731e-04
        },
// x = -8.5800
        {
                -1.8768392413311897e-04, -1.8775444083553669e-04, -1.8778970580914458e-04, -1.8780734050372581e-04,
                -1.8781615858701992e-04, -1.8782056787402097e-04, -1.8782277259931095e-04, -1.8782387498922034e-04,
                -1.8782442619326350e-04, -1.8782470179831457e-04
        },
// x = -8.5750
        {
                -1.8862398344304576e-04, -1.8869520864897200e-04, -1.8873082797439994e-04, -1.8874863987825271e-04,
                -1.8875754657730464e-04, -1.8876200017589235e-04, -1.8876422705821173e-04, -1.8876534052704783e-04,
                -1.8876589727069167e-04, -1.8876617564558891e-04
        },
// x = -8.5700
        {
                -1.8956874768453916e-04, -1.8964068851151354e-04, -1.8967666574904322e-04, -1.8969465664281254e-04,
                -1.8970365284811291e-04, -1.8970815120358857e-04, -1.8971040046560664e-04, -1.8971152512471033e-04,
                -1.8971208746362741e-04, -1.8971236863620774e-04
        },
// x = -8.5650
        {
                -1.9051824036950144e-04, -1.9059090400655028e-04, -1.9062724275222963e-04, -1.9064541443445109e-04,
                -1.9065450104543833e-04, -1.9065904460757805e-04, -1.9066131647420181e-04, -1.9066245243603292e-04,
                -1.9066302042645522e-04, -1.9066330442483537e-04
        },
// x = -8.5600
        {
                -1.9147248512696940e-04, -1.9154587883532322e-04, -1.9158258272132541e-04, -1.9160093700860509e-04,
                -1.9161011493375532e-04, -1.9161470415685500e-04, -1.9161699885525161e-04, -1.9161814623340015e-04,
                -1.9161871993212485e-04, -1.9161900678470406e-04
        },
// x = -8.5550
        {
                -1.9243150570369394e-04, -1.9250563681751250e-04, -1.9254270951249891e-04, -1.9256124823969481e-04,
                -1.9257051839661289e-04, -1.9257515373953351e-04, -1.9257747149915304e-04, -1.9257863040835049e-04,
                -1.9257920987274553e-04, -1.9257949960820852e-04
        },
// x = -8.5500
        {
                -1.9339532596472303e-04, -1.9347020189182769e-04, -1.9350764710131436e-04, -1.9352637212171974e-04,
                -1.9353573543723078e-04, -1.9354041736344434e-04, -1.9354275841604261e-04, -1.9354392897217359e-04,
                -1.9354451426018340e-04, -1.9354480690750317e-04
        },
// x = -8.5450
        {
                -1.9436396989398736e-04, -1.9443959811660041e-04, -1.9447741958332821e-04, -1.9449633276885688e-04,
                -1.9450579017909887e-04, -1.9451051915673481e-04, -1.9451288373639665e-04, -1.9451406605651027e-04,
                -1.9451465722666167e-04, -1.9451495281510233e-04
        },
// x = -8.5400
        {
                -1.9533746159488852e-04, -1.9541384967038057e-04, -1.9545205117468898e-04, -1.9547115441606217e-04,
                -1.9548070686657967e-04, -1.9548548336847157e-04, -1.9548787171163427e-04, -1.9548906591395590e-04,
                -1.9548966302536388e-04, -1.9548996158448369e-04
        },
// x = -8.5350
        {
                -1.9631582529089066e-04, -1.9639298085253476e-04, -1.9643156621273957e-04, -1.9645086141967510e-04,
                -1.9646050986551371e-04, -1.9646533436924653e-04, -1.9646774671472331e-04, -1.9646895291866657e-04,
                -1.9646955603104023e-04, -1.9646985759069445e-04
        },
// x = -8.5300
        {
                -1.9729908532611448e-04, -1.9737701608384841e-04, -1.9741598915662311e-04, -1.9743547825802615e-04,
                -1.9744522366382783e-04, -1.9745009665178583e-04, -1.9745253324078980e-04, -1.9745375156696827e-04,
                -1.9745436074061672e-04, -1.9745466533096077e-04
        },
// x = -8.5250
        {
                -1.9828726616593443e-04, -1.9836597990713044e-04, -1.9840534458789145e-04, -1.9842502953204712e-04,
                -1.9843487287214689e-04, -1.9843979483156164e-04, -1.9844225590773002e-04, -1.9844348647796940e-04,
                -1.9844410177380785e-04, -1.9844440942530010e-04
        },
// x = -8.5200
        {
                -1.9928039239757895e-04, -1.9935989698782099e-04, -1.9939965721111689e-04, -1.9941953996588507e-04,
                -1.9942948222440810e-04, -1.9943445364740731e-04, -1.9943693945682564e-04, -1.9943818239417593e-04,
                -1.9943880387373185e-04, -1.9943911461713681e-04
        },
// x = -8.5150
        {
                -2.0027848873073299e-04, -2.0035879211460222e-04, -2.0039895185450658e-04, -2.0041903440751863e-04,
                -2.0042907657847876e-04, -2.0043409796213546e-04, -2.0043660875336224e-04, -2.0043786418211007e-04,
                -2.0043849190752916e-04, -2.0043880577392056e-04
        },
// x = -8.5100
        {
                -2.0128157999814444e-04, -2.0136269020001218e-04, -2.0140325347052065e-04, -2.0142353782937788e-04,
                -2.0143368091677703e-04, -2.0143875276315917e-04, -2.0144128878725085e-04, -2.0144255683293181e-04,
                -2.0144319086698442e-04, -2.0144350788774820e-04
        },
// x = -8.5050
        {
                -2.0228969115623264e-04, -2.0237161628106152e-04, -2.0241258713649281e-04, -2.0243307532896715e-04,
                -2.0244332034689545e-04, -2.0244844316311624e-04, -2.0245100467365207e-04, -2.0245228546306344e-04,
                -2.0245292586915068e-04, -2.0245324607598826e-04
        },
// x = -8.5000
        {
                -2.0330284728570038e-04, -2.0338559551985350e-04, -2.0342697805525420e-04, -2.0344767212949078e-04,
                -2.0345802010222823e-04, -2.0346319440049676e-04, -2.0346578165360434e-04, -2.0346707531481754e-04,
                -2.0346772215697773e-04, -2.0346804558190909e-04
        },
// x = -8.4950
        {
                -2.0432107359214866e-04, -2.0440465320420683e-04, -2.0444645155576058e-04, -2.0446735358048253e-04,
                -2.0447780554260108e-04, -2.0448303184027349e-04, -2.0448564509465428e-04, -2.0448695175702791e-04,
                -2.0448760509994286e-04, -2.0448793177530985e-04
        },
// x = -8.4900
        {
                -2.0534439540669471e-04, -2.0542881474828185e-04, -2.0547103309372241e-04, -2.0549214515843743e-04,
                -2.0550270215490457e-04, -2.0550798097453588e-04, -2.0551062049149112e-04, -2.0551194028568364e-04,
                -2.0551260019468530e-04, -2.0551293015315473e-04
        },
// x = -8.4850
        {
                -2.0637283818659256e-04, -2.0645810569320933e-04, -2.0650074825223806e-04, -2.0652207246744722e-04,
                -2.0653273555373044e-04, -2.0653806742312663e-04, -2.0654073346658341e-04, -2.0654206652456642e-04,
                -2.0654273306564328e-04, -2.0654306634021027e-04
        },
// x = -8.4800
        {
                -2.0740642751585707e-04, -2.0749255170772325e-04, -2.0753562274243033e-04, -2.0755716123983909e-04,
                -2.0756793148201115e-04, -2.0757331693428208e-04, -2.0757600977081977e-04, -2.0757735622589117e-04,
                -2.0757802946569481e-04, -2.0757836608968611e-04
        },
// x = -8.4750
        {
                -2.0844518910589097e-04, -2.0853217858879573e-04, -2.0857568240408608e-04, -2.0859743733681708e-04,
                -2.0860831581166292e-04, -2.0861375538527548e-04, -2.0861647528415238e-04, -2.0861783527094957e-04,
                -2.0861851527680152e-04, -2.0861885528387873e-04
        },
// x = -8.4700
        {
                -2.0948914879611458e-04, -2.0957701226227566e-04, -2.0962095320629877e-04, -2.0964292674910732e-04,
                -2.0965391454423127e-04, -2.0965940878306334e-04, -2.0966215601624362e-04, -2.0966352967075703e-04,
                -2.0966421651065536e-04, -2.0966455993481846e-04
        },
// x = -8.4650
        {
                -2.1053833255459895e-04, -2.1062707878353047e-04, -2.1067146124811487e-04, -2.1069365559760585e-04,
                -2.1070475381154073e-04, -2.1071030326493546e-04, -2.1071307810711625e-04, -2.1071446556670303e-04,
                -2.1071515930932904e-04, -2.1071550618491974e-04
        },
// x = -8.4600
        {
                -2.1159276647870218e-04, -2.1168240433809082e-04, -2.1172723275918263e-04, -2.1174965013403025e-04,
                -2.1176085987634690e-04, -2.1176646509916760e-04, -2.1176926782780659e-04, -2.1177066923120427e-04,
                -2.1177136994592963e-04, -2.1177172030763461e-04
        },
// x = -8.4550
        {
                -2.1265247679570838e-04, -2.1274301524229874e-04, -2.1278829410040486e-04, -2.1281093674157425e-04,
                -2.1282225913299256e-04, -2.1282792068567826e-04, -2.1283075158102133e-04, -2.1283216706836163e-04,
                -2.1283287482525516e-04, -2.1283322870810988e-04
        },
// x = -8.4500
        {
                -2.1371748986347020e-04, -2.1380893794395889e-04, -2.1385467176459439e-04, -2.1387754193556555e-04,
                -2.1388897810806635e-04, -2.1389469655668785e-04, -2.1389755590179714e-04, -2.1389898561462002e-04,
                -2.1389970048445471e-04, -2.1390005792384662e-04
        },
// x = -8.4450
        {
                -2.1478783217105418e-04, -2.1488019902299280e-04, -2.1492639237713306e-04, -2.1494949236412704e-04,
                -2.1496104346106540e-04, -2.1496681937738173e-04, -2.1496970745816390e-04, -2.1497115153943180e-04,
                -2.1497187359369185e-04, -2.1497223462536405e-04
        },
// x = -8.4400
        {
                -2.1586353033938976e-04, -2.1595682519209695e-04, -2.1600348269663400e-04, -2.1602681480884138e-04,
                -2.1603848198506077e-04, -2.1604431594657636e-04, -2.1604723305181136e-04, -2.1604869164592336e-04,
                -2.1604942095681138e-04, -2.1604978561686622e-04
        },
// x = -8.4350
        {
                -2.1694461112192038e-04, -2.1703884329740331e-04, -2.1708596961560723e-04, -2.1710953618541871e-04,
                -2.1712132060736647e-04, -2.1712721319738903e-04, -2.1713015961875859e-04, -2.1713163287156496e-04,
                -2.1713236951200919e-04, -2.1713273783691180e-04
        },
// x = -8.4300
        {
                -2.1803110140525904e-04, -2.1812628031914373e-04, -2.1817388016112831e-04, -2.1819768354436786e-04,
                -2.1820958639021172e-04, -2.1821553819791043e-04, -2.1821851423002754e-04, -2.1822000228884425e-04,
                -2.1822074633250590e-04, -2.1822111835908794e-04
        },
// x = -8.4250
        {
                -2.1912302820984629e-04, -2.1921916337231754e-04, -2.1926724149551056e-04, -2.1929128407167087e-04,
                -2.1930330653141680e-04, -2.1930931815188117e-04, -2.1931232409231954e-04, -2.1931382710594301e-04,
                -2.1931457862722341e-04, -2.1931495439268665e-04
        },
// x = -8.4200
        {
                -2.2022041869061148e-04, -2.2031751970736198e-04, -2.2036608091698069e-04, -2.2039036508946083e-04,
                -2.2040250836507198e-04, -2.2040858039937139e-04, -2.2041161654869494e-04, -2.2041313466741698e-04,
                -2.2041389374146531e-04, -2.2041427328338542e-04
        },
// x = -8.4150
        {
                -2.2132330013763737e-04, -2.2142137671082683e-04, -2.2147042586035762e-04, -2.2149495405670325e-04,
                -2.2150721936221990e-04, -2.2151335241746373e-04, -2.2151641907925692e-04, -2.2151795245487963e-04,
                -2.2151871915760025e-04, -2.2151910251393052e-04
        },
// x = -8.4100
        {
                -2.2243169997682811e-04, -2.2253076190605141e-04, -2.2258030389773461e-04, -2.2260507856988046e-04,
                -2.2261746713154155e-04, -2.2262366182093987e-04, -2.2262675930183790e-04, -2.2262830808768890e-04,
                -2.2262908249574904e-04, -2.2262946970482409e-04
        },
// x = -8.4050
        {
                -2.2354564577058012e-04, -2.2364570295384546e-04, -2.2369574273916509e-04, -2.2372076636367999e-04,
                -2.2373327942004569e-04, -2.2373953636297040e-04, -2.2374266497268987e-04, -2.2374422932363775e-04,
                -2.2374501151447499e-04, -2.2374540261501489e-04
        },
// x = -8.4000
        {
                -2.2466516521845641e-04, -2.2476622765317343e-04, -2.2481677023335160e-04, -2.2484204531168596e-04,
                -2.2485468411376133e-04, -2.2486100393580828e-04, -2.2486416398717821e-04, -2.2486574405964781e-04,
                -2.2486653411147811e-04, -2.2486692914259188e-04
        },
// x = -8.3950
        {
                -2.2579028615786440e-04, -2.2589236394184191e-04, -2.2594341436833832e-04, -2.2596894342707410e-04,
                -2.2598170923843406e-04, -2.2598809257148549e-04, -2.2599128438047857e-04, -2.2599288033246676e-04,
                -2.2599367832429204e-04, -2.2599407732548191e-04
        },
// x = -8.3900
        {
                -2.2692103656473664e-04, -2.2702413989719047e-04, -2.2707570327220705e-04, -2.2710148886331006e-04,
                -2.2711438296022600e-04, -2.2712083044251370e-04, -2.2712405432827781e-04, -2.2712566631936926e-04,
                -2.2712647233098541e-04, -2.2712687534215046e-04
        },
// x = -8.3850
        {
                -2.2805744455421512e-04, -2.2816158373678618e-04, -2.2821366521377651e-04, -2.2823970991485152e-04,
                -2.2825273358641852e-04, -2.2825924586258758e-04, -2.2826250214747794e-04, -2.2826413033886101e-04,
                -2.2826494445086587e-04, -2.2826535151230623e-04
        },
// x = -8.3800
        {
                -2.2919953838133872e-04, -2.2930472381912123e-04, -2.2935732860330526e-04, -2.2938363501785351e-04,
                -2.2939678956611936e-04, -2.2940336728729280e-04, -2.2940665629690385e-04, -2.2940830085138684e-04,
                -2.2940912314518816e-04, -2.2940953429760895e-04
        },
// x = -8.3750
        {
                -2.3034734644173429e-04, -2.3045358864431421e-04, -2.3050672199319828e-04, -2.3053329275087757e-04,
                -2.3054657949097282e-04, -2.3055322331481644e-04, -2.3055654537801462e-04, -2.3055820646004192e-04,
                -2.3055903701786566e-04, -2.3055945230238110e-04
        },
// x = -8.3700
        {
                -2.3150089727231069e-04, -2.3160820685481470e-04, -2.3166187407871637e-04, -2.3168871183560391e-04,
                -2.3170213209587335e-04, -2.3170884268666162e-04, -2.3171219813561823e-04, -2.3171387591128682e-04,
                -2.3171471481618524e-04, -2.3171513427432271e-04
        },
// x = -8.3650
        {
                -2.3266021955195671e-04, -2.3276860723611152e-04, -2.3282281369869012e-04, -2.3284992113754784e-04,
                -2.3286347625968320e-04, -2.3287025428836569e-04, -2.3287364345858999e-04, -2.3287533809566603e-04,
                -2.3287618543152609e-04, -2.3287660910523028e-04
        },
// x = -8.3600
        {
                -2.3382534210224159e-04, -2.3393481871744441e-04, -2.3398956983623653e-04, -2.3401694966677908e-04,
                -2.3403064100595327e-04, -2.3403748715022154e-04, -2.3404091038059443e-04, -2.3404262204853007e-04,
                -2.3404347790008170e-04, -2.3404390583171893e-04
        },
// x = -8.3550
        {
                -2.3499629388812007e-04, -2.3510687037251897e-04, -2.3516217161947975e-04, -2.3518982657864528e-04,
                -2.3520365550364766e-04, -2.3521057044800314e-04, -2.3521402808081101e-04, -2.3521575695076123e-04,
                -2.3521662140358587e-04, -2.3521705363594824e-04
        },
// x = -8.3500
        {
                -2.3617310401863979e-04, -2.3628479142022565e-04, -2.3634064832227507e-04, -2.3636858117449870e-04,
                -2.3638254906787191e-04, -2.3638953350369444e-04, -2.3639302588466320e-04, -2.3639477212950309e-04,
                -2.3639564527004234e-04, -2.3639608184635193e-04
        },
// x = -8.3450
        {
                -2.3735580174765269e-04, -2.3746861122536184e-04, -2.3752502936493663e-04, -2.3755324290242668e-04,
                -2.3756735116060496e-04, -2.3757440578622157e-04, -2.3757793326455151e-04, -2.3757969705889343e-04,
                -2.3758057897445750e-04, -2.3758101993837078e-04
        },
// x = -8.3400
        {
                -2.3854441647452995e-04, -2.3865835929935751e-04, -2.3871534431496897e-04, -2.3874384135798586e-04,
                -2.3875809139143437e-04, -2.3876521691218943e-04, -2.3876877984058993e-04, -2.3877056136080099e-04,
                -2.3877145213957760e-04, -2.3877189753518979e-04
        },
// x = -8.3350
        {
                -2.3973897774487991e-04, -2.3985406530100512e-04, -2.3991162288780176e-04, -2.3994040628493984e-04,
                -2.3995479951829590e-04, -2.3996199664662137e-04, -2.3996559538134625e-04, -2.3996739480556595e-04,
                -2.3996829453662902e-04, -2.3996874440847851e-04
        },
// x = -8.3300
        {
                -2.4093951525126995e-04, -2.4105575903719193e-04, -2.4111389494752868e-04, -2.4114296757600076e-04,
                -2.4115750544821601e-04, -2.4116477490370287e-04, -2.4116840980458593e-04, -2.4117022731274396e-04,
                -2.4117113608606268e-04, -2.4117159047913541e-04
        },
// x = -8.3250
        {
                -2.4214605883395162e-04, -2.4226347046363706e-04, -2.4232219050764951e-04, -2.4235155527357446e-04,
                -2.4236623923805877e-04, -2.4237358174752872e-04, -2.4237725317801986e-04, -2.4237908895185413e-04,
                -2.4238000685830171e-04, -2.4238046581803580e-04
        },
// x = -8.3200
        {
                -2.4335863848158951e-04, -2.4347722968563176e-04, -2.4353653973181632e-04, -2.4356619957050928e-04,
                -2.4358103109527600e-04, -2.4358844739285436e-04, -2.4359215572005569e-04, -2.4359400994313047e-04,
                -2.4359493707449341e-04, -2.4359540064678364e-04
        },
// x = -8.3150
        {
                -2.4457728433199339e-04, -2.4469706695878297e-04, -2.4475697293458305e-04, -2.4478693081084890e-04,
                -2.4480191137866145e-04, -2.4480940220585035e-04, -2.4481314780055308e-04, -2.4481502065827734e-04,
                -2.4481595710726471e-04, -2.4481642533846694e-04
        },
// x = -8.3100
        {
                -2.4580202667285391e-04, -2.4592301268976113e-04, -2.4598352058215916e-04, -2.4601377949058840e-04,
                -2.4602891059910878e-04, -2.4603647670486134e-04, -2.4604025994158274e-04, -2.4604215162122878e-04,
                -2.4604309748148105e-04, -2.4604357041841730e-04
        },
// x = -8.3050
        {
                -2.4703289594248242e-04, -2.4715509743705142e-04, -2.4721621329316647e-04, -2.4724677625843471e-04,
                -2.4726205942037269e-04, -2.4726970156116799e-04, -2.4727352281818908e-04, -2.4727543350891104e-04,
                -2.4727638887501004e-04, -2.4727686656497244e-04
        },
// x = -8.3000
        {
                -2.4826992273055344e-04, -2.4839335191170861e-04, -2.4845508183940046e-04, -2.4848595191657058e-04,
                -2.4850138865983531e-04, -2.4850910759975387e-04, -2.4851296725915666e-04, -2.4851489715200966e-04,
                -2.4851586211948766e-04, -2.4851634461024410e-04
        },
// x = -8.2950
        {
                -2.4951313777885162e-04, -2.4963780697811593e-04, -2.4970015714659500e-04, -2.4973133742142224e-04,
                -2.4974692928927478e-04, -2.4975472580007481e-04, -2.4975862424778128e-04, -2.4976057353574046e-04,
                -2.4976154820108959e-04, -2.4976203554088767e-04
        },
// x = -8.2900
        {
                -2.5076257198202161e-04, -2.5088849365474718e-04, -2.5095147029519045e-04, -2.5098296388443062e-04,
                -2.5099871243563831e-04, -2.5100658729683320e-04, -2.5101052492264325e-04, -2.5101249380062295e-04,
                -2.5101347826130541e-04, -2.5101397049887773e-04
        },
// x = -8.2850
        {
                -2.5201825638832210e-04, -2.5214544311493321e-04, -2.5220905252110680e-04, -2.5224086257282804e-04,
                -2.5225676938181972e-04, -2.5226472338075589e-04, -2.5226870057838658e-04, -2.5227068924326004e-04,
                -2.5227168359771715e-04, -2.5227218078228606e-04
        },
// x = -8.2800
        {
                -2.5328022220038362e-04, -2.5340868668763178e-04, -2.5347293521651900e-04, -2.5350506491041592e-04,
                -2.5352113156744020e-04, -2.5352916549937544e-04, -2.5353318266650027e-04, -2.5353519131711915e-04,
                -2.5353619566478169e-04, -2.5353669784606425e-04
        },
// x = -8.2750
        {
                -2.5454850077596898e-04, -2.5467825585820126e-04, -2.5474314993063788e-04, -2.5477560247834933e-04,
                -2.5479183058963275e-04, -2.5479994525781619e-04, -2.5480400279610495e-04, -2.5480603163331912e-04,
                -2.5480704607461717e-04, -2.5480755330283011e-04
        },
// x = -8.2700
        {
                -2.5582312362873878e-04, -2.5595418226917845e-04, -2.5601972837049328e-04, -2.5605250701592343e-04,
                -2.5606889820383133e-04, -2.5607709441958333e-04, -2.5608119273474219e-04, -2.5608324196142013e-04,
                -2.5608426659779295e-04, -2.5608477892365757e-04
        },
// x = -8.2650
        {
                -2.5710412242901958e-04, -2.5723649772105945e-04, -2.5730270240172277e-04, -2.5733581042136476e-04,
                -2.5735236632456327e-04, -2.5736064490735689e-04, -2.5736478440916902e-04, -2.5736685423021773e-04,
                -2.5736788916412405e-04, -2.5736840663887145e-04
        },
// x = -8.2600
        {
                -2.5839152900457632e-04, -2.5852523417308566e-04, -2.5859210404936237e-04, -2.5862554475262616e-04,
                -2.5864226702624612e-04, -2.5865062880378869e-04, -2.5865480990615498e-04, -2.5865690052854080e-04,
                -2.5865794586346912e-04, -2.5865846853884530e-04
        },
// x = -8.2550
        {
                -2.5968537534138867e-04, -2.5982042374403228e-04, -2.5988796549864332e-04, -2.5992174222818590e-04,
                -2.5993863254398819e-04, -2.5994707835230426e-04, -2.5995130147328510e-04, -2.5995341310605424e-04,
                -2.5995446894653282e-04, -2.5995499687480380e-04
        },
// x = -8.2500
        {
                -2.6098569358442983e-04, -2.6112209871300150e-04, -2.6119031909579082e-04, -2.6122443522785045e-04,
                -2.6124149527439290e-04, -2.6125002595790773e-04, -2.6125429151976453e-04, -2.6125642437406401e-04,
                -2.6125749082567186e-04, -2.6125802405962880e-04
        },
// x = -8.2450
        {
                -2.6229251603845157e-04, -2.6243029152021966e-04, -2.6249919734882849e-04, -2.6253365629356172e-04,
                -2.6255088777636806e-04, -2.6255950418799207e-04, -2.6256381261722964e-04, -2.6256596690632815e-04,
                -2.6256704407570528e-04, -2.6256758266866994e-04
        },
// x = -8.2400
        {
                -2.6360587516877001e-04, -2.6374503476783799e-04, -2.6381463292838507e-04, -2.6384943813020760e-04,
                -2.6386684277193804e-04, -2.6387554577315204e-04, -2.6387989750056092e-04, -2.6388207343987073e-04,
                -2.6388316143472856e-04, -2.6388370544055880e-04
        },
// x = -8.2350
        {
                -2.6492580360205811e-04, -2.6506636122073708e-04, -2.6513665866850691e-04, -2.6517181360643773e-04,

                -2.6518939314706074e-04, -2.6519818360800198e-04, -2.6520257906870187e-04, -2.6520477687579970e-04,
                -2.6520587580493250e-04, -2.6520642527802708e-04
        },
// x = -8.2300
        {
                -2.6625233412713972e-04, -2.6639430380733626e-04, -2.6646530756747330e-04, -2.6650081575548202e-04,
                -2.6651857195244833e-04, -2.6652745075199759e-04, -2.6653189038548070e-04, -2.6653411028013005e-04,
                -2.6653522025342513e-04, -2.6653577524872978e-04
        },
// x = -8.2250
        {
                -2.6758549969578901e-04, -2.6772889562040576e-04, -2.6780061278861564e-04, -2.6783647777597392e-04,
                -2.6785441240439247e-04, -2.6786338043026191e-04, -2.6786786468043676e-04, -2.6787010688460931e-04,
                -2.6787122801305856e-04, -2.6787178858607110e-04
        },
// x = -8.2200
        {
                -2.6892533342353286e-04, -2.6907016991788385e-04, -2.6914260766114253e-04, -2.6917883303277825e-04,
                -2.6919694788559307e-04, -2.6920600603441480e-04, -2.6921053534965098e-04, -2.6921280008754928e-04,
                -2.6921393248325974e-04, -2.6921449869003581e-04
        },
// x = -8.2150
        {
                -2.7027186859045747e-04, -2.7041816012369758e-04, -2.7049132568096623e-04, -2.7052791505782195e-04,
                -2.7054621194599166e-04, -2.7055536112340750e-04, -2.7055993595658011e-04, -2.7056222345465974e-04,
                -2.7056336723086538e-04, -2.7056393912802375e-04
        },
// x = -8.2100
        {
                -2.7162513864201909e-04, -2.7177289982858739e-04, -2.7184680051153571e-04, -2.7188375755093024e-04,
                -2.7190223830360913e-04, -2.7191147942436106e-04, -2.7191610023289602e-04, -2.7191841071988845e-04,
                -2.7191956599096088e-04, -2.7192014363568959e-04
        },
// x = -8.2050
        {
                -2.7298517718985811e-04, -2.7313442279093607e-04, -2.7320906598467232e-04, -2.7324639438066597e-04,
                -2.7326506084538642e-04, -2.7327439483340826e-04, -2.7327906207932819e-04, -2.7328139578626330e-04,
                -2.7328256266772385e-04, -2.7328314611778549e-04
        },
// x = -8.2000
        {
                -2.7435201801261761e-04, -2.7450276293760191e-04, -2.7457815610141000e-04, -2.7461585958517401e-04,
                -2.7463471362803134e-04, -2.7464414141654074e-04, -2.7464885556651120e-04, -2.7465121272674052e-04,
                -2.7465239133527128e-04, -2.7465298064900917e-04
        },
// x = -8.1950
        {
                -2.7572569505676555e-04, -2.7587795436475563e-04, -2.7595410503284009e-04, -2.7599218737302890e-04,
                -2.7601123087886800e-04, -2.7602075341045981e-04, -2.7602551493583578e-04, -2.7602789578505614e-04,
                -2.7602908623851189e-04, -2.7602968147485523e-04
        },
// x = -8.1900
        {
                -2.7710624243742121e-04, -2.7726003133872148e-04, -2.7733694712095977e-04, -2.7737541212408774e-04,
                -2.7739464699669117e-04, -2.7740426522343190e-04, -2.7740907460030499e-04, -2.7741147937658165e-04,
                -2.7741268179400146e-04, -2.7741328301247214e-04
        },
// x = -8.1850
        {
                -2.7849369443918497e-04, -2.7864902829682263e-04, -2.7872671687952506e-04, -2.7876556839034672e-04,
                -2.7878499655262505e-04, -2.7879471143614769e-04, -2.7879956914539388e-04, -2.7880199808918434e-04,
                -2.7880321259080362e-04, -2.7880381985152157e-04
        },
// x = -8.1800
        {
                -2.7988808551697333e-04, -2.8004497984823089e-04, -2.8012344899490773e-04, -2.8016269089680177e-04,
                -2.8018231429098573e-04, -2.8019212680258630e-04, -2.8019703332991347e-04, -2.8019948668409208e-04,
                -2.8020071339135449e-04, -2.8020132675504386e-04
        },
// x = -8.1750
        {
                -2.8128945029685666e-04, -2.8144792077482015e-04, -2.8152717832695683e-04, -2.8156681454231437e-04,
                -2.8158663513014884e-04, -2.8159654625088297e-04, -2.8160150208688019e-04, -2.8160398009676168e-04,
                -2.8160521913233157e-04, -2.8160583866032655e-04
        },
// x = -8.1700
        {
                -2.8269782357690160e-04, -2.8285788603202397e-04, -2.8293793990986438e-04, -2.8297797440048018e-04,
                -2.8299799416342070e-04, -2.8300800488420194e-04, -2.8301301052438804e-04, -2.8301551343775240e-04,
                -2.8301676492552665e-04, -2.8301739067977823e-04
        },
// x = -8.1650
        {
                -2.8411324032801775e-04, -2.8427491074969831e-04, -2.8435576895303517e-04, -2.8439620572050402e-04,
                -2.8441642665991411e-04, -2.8442653798161315e-04, -2.8443159392648625e-04, -2.8443412199360377e-04,
                -2.8443538605872447e-04, -2.8443601810180581e-04
        },
// x = -8.1600
        {
                -2.8553573569480787e-04, -2.8569903023298760e-04, -2.8578070084196136e-04, -2.8582154392807740e-04,
                -2.8584196806542887e-04, -2.8585218099897331e-04, -2.8585728775406129e-04, -2.8585984122771695e-04,
                -2.8586111799658369e-04, -2.8586175639169704e-04
        },
// x = -8.1550
        {
                -2.8696534499642279e-04, -2.8713027996319523e-04, -2.8721277113910051e-04, -2.8725402462626123e-04,
                -2.8727465400333636e-04, -2.8728496956981192e-04, -2.8729012764572233e-04, -2.8729270678124174e-04,
                -2.8729399638152448e-04, -2.8729464119250724e-04
        },
// x = -8.1500
        {
                -2.8840210372741971e-04, -2.8856869559865822e-04, -2.8865201558475931e-04, -2.8869368359637316e-04,
                -2.8871452027546851e-04, -2.8872493950622094e-04, -2.8873014941869249e-04, -2.8873275447396690e-04,
                -2.8873405703461864e-04, -2.8873470832594978e-04
        },
// x = -8.1450
        {
                -2.8984604755862557e-04, -2.9001431297562703e-04, -2.9009847009798021e-04, -2.9014055679887858e-04,
                -2.9016160286301129e-04, -2.9017212679974946e-04, -2.9017738906970351e-04, -2.9018002030521580e-04,
                -2.9018133595648543e-04, -2.9018199379329183e-04
        },
// x = -8.1400
        {
                -2.9129721233800350e-04, -2.9146716810914862e-04, -2.9155217077743363e-04, -2.9159468037428718e-04,
                -2.9161593792740263e-04, -2.9162656762230286e-04, -2.9163188277589545e-04, -2.9163454045474631e-04,
                -2.9163586932819162e-04, -2.9163653377625476e-04
        },
// x = -8.1350
        {
                -2.9275563409152469e-04, -2.9292729719395451e-04, -2.9301315390231424e-04, -2.9305609064405236e-04,
                -2.9307756181123505e-04, -2.9308829832704617e-04, -2.9309366689572053e-04, -2.9309635128365447e-04,
                -2.9309769351215556e-04, -2.9309836463791789e-04
        },
// x = -8.1300
        {
                -2.9422134902404345e-04, -2.9439473660535268e-04, -2.9448145593324122e-04, -2.9452482411147730e-04,
                -2.9454651103916294e-04, -2.9455735544931238e-04, -2.9456277796985206e-04, -2.9456548933528426e-04,
                -2.9456684505305635e-04, -2.9456752292362815e-04
        },
// x = -8.1250
        {
                -2.9569439352017667e-04, -2.9586952290012466e-04, -2.9595711351316373e-04, -2.9600091746262338e-04,
                -2.9602282231881293e-04, -2.9603377570751430e-04, -2.9603925272209716e-04, -2.9604199133614011e-04,
                -2.9604336067874730e-04, -2.9604404536191325e-04
        },
// x = -8.1200
        {
                -2.9717480414518787e-04, -2.9735169281742632e-04, -2.9744016346827028e-04, -2.9748440756722453e-04,
                -2.9750653254170172e-04, -2.9751759600406274e-04, -2.9752312806031464e-04, -2.9752589419680540e-04,
                -2.9752727730117428e-04, -2.9752796886540036e-04
        },
// x = -8.1150
        {
                -2.9866261764587574e-04, -2.9884128327969305e-04, -2.9893064280890315e-04, -2.9897533147960553e-04,
                -2.9899767878415509e-04, -2.9900885342628740e-04, -2.9901444107733742e-04, -2.9901723501286475e-04,
                -2.9901863201729783e-04, -2.9901933053173802e-04
        },
// x = -8.1100
        {
                -3.0015787095146563e-04, -3.0033833139355012e-04, -3.0042858873047669e-04, -3.0047372643960519e-04,
                -3.0049629830823394e-04, -3.0050758524736369e-04, -3.0051322905189900e-04, -3.0051605106583138e-04,
                -3.0051746211002113e-04, -3.0051816764452440e-04
        },
// x = -8.1050
        {
                -3.0166060117450718e-04, -3.0184287445072697e-04, -3.0193403861440097e-04, -3.0197962987350417e-04,
                -3.0200242856266425e-04, -3.0201382892724370e-04, -3.0201952944956567e-04, -3.0202237982407892e-04,
                -3.0202380504912166e-04, -3.0202451767423889e-04
        },
// x = -8.1000
        {
                -3.0317084561177468e-04, -3.0335494992897579e-04, -3.0344703002900921e-04, -3.0349307939495708e-04,
                -3.0351610718377156e-04, -3.0352762211359149e-04, -3.0353337992367226e-04, -3.0353625894377780e-04,
                -3.0353769849218769e-04, -3.0353841827917888e-04
        },
// x = -8.0950
        {
                -3.0468864174517274e-04, -3.0487459549299509e-04, -3.0496760073049037e-04, -3.0501411280592908e-04,
                -3.0503737199641970e-04, -3.0504900264272393e-04, -3.0505481831626287e-04, -3.0505772626983639e-04,
                -3.0505918028555997e-04, -3.0505990730640123e-04
        },
// x = -8.0900
        {
                -3.0621402724264598e-04, -3.0640184899535772e-04, -3.0649578866382641e-04, -3.0654276809763837e-04,
                -3.0656626101495529e-04, -3.0657800854055542e-04, -3.0658388265903720e-04, -3.0658681983684730e-04,
                -3.0658828846527754e-04, -3.0658902279266827e-04
        },
// x = -8.0850
        {
                -3.0774703995909303e-04, -3.0793674847744319e-04, -3.0803163196373341e-04, -3.0807908345150157e-04,
                -3.0810281244415626e-04, -3.0811467802354780e-04, -3.0812061117430019e-04, -3.0812357787003751e-04,
                -3.0812506125802857e-04, -3.0812580296539895e-04
        },
// x = -8.0800
        {
                -3.0928771793728508e-04, -3.0947933217037496e-04, -3.0957516895560876e-04, -3.0962309724008553e-04,
                -3.0964706468018468e-04, -3.0965904949966478e-04, -3.0966504227591767e-04, -3.0966803878622399e-04,
                -3.0966953708210605e-04, -3.0967028624362392e-04
        },
// x = -8.0750
        {
                -3.1083609940878886e-04, -3.1102963849596150e-04, -3.1112643815648148e-04, -3.1117484802806243e-04,
                -3.1119905631154577e-04, -3.1121116156933158e-04, -3.1121721457027580e-04, -3.1122024119477424e-04,
                -3.1122175454836783e-04, -3.1122251123894663e-04
        },
// x = -8.0700
        {
                -3.1239222279489374e-04, -3.1258770606764366e-04, -3.1268547827596825e-04, -3.1273437457317099e-04,
                -3.1275882612005019e-04, -3.1277105302639848e-04, -3.1277716685724631e-04, -3.1278022389857077e-04,
                -3.1278175246120217e-04, -3.1278251675650814e-04
        },
// x = -8.0650
        {
                -3.1395612670754398e-04, -3.1415357369144451e-04, -3.1425232821723448e-04, -3.1430171582718140e-04,
                -3.1432641308178197e-04, -3.1433876285911041e-04, -3.1434493813115553e-04, -3.1434802589498125e-04,
                -3.1434956981949753e-04, -3.1435034179595739e-04
        },
// x = -8.0600
        {
                -3.1552784995027430e-04, -3.1572728036692566e-04, -3.1582702707795855e-04, -3.1587691093686536e-04,
                -3.1590185636807107e-04, -3.1591433025108021e-04, -3.1592056758175907e-04, -3.1592368637683305e-04,
                -3.1592524581761729e-04, -3.1592602555242580e-04
        },
// x = -8.0550
        {
                -3.1710743151915148e-04, -3.1730886528814726e-04, -3.1740961415130294e-04, -3.1745999924497157e-04,
                -3.1748519534647093e-04, -3.1749779458226752e-04, -3.1750409459522122e-04, -3.1750724473339313e-04,
                -3.1750881984637990e-04, -3.1750960741750751e-04
        },
// x = -8.0500
        {
                -3.1869491060371912e-04, -3.1889836784463309e-04, -3.1900012892688843e-04, -3.1905102029120507e-04,
                -3.1907646958174052e-04, -3.1908919542996252e-04, -3.1909555875509881e-04, -3.1909874055135215e-04,
                -3.1910033149404305e-04, -3.1910112698024394e-04
        },
// x = -8.0450
        {
                -3.2029032658794732e-04, -3.2049582762234019e-04, -3.2059861109177420e-04, -3.2065001381321220e-04,
                -3.2067571883683199e-04, -3.2068857256977416e-04, -3.2069499984333047e-04, -3.2069821361581412e-04,
                -3.2069982054729366e-04, -3.2070062402811335e-04
        },
// x = -8.0400
        {
                -3.2189371905118802e-04, -3.2210128440463383e-04, -3.2220510053144227e-04, -3.2225701974757017e-04,
                -3.2228298307388268e-04, -3.2229596597662356e-04, -3.2230245784123097e-04, -3.2230570391129096e-04,
                -3.2230732699224223e-04, -3.2230813854802569e-04
        },
// x = -8.0350
        {
                -3.2350512776913281e-04, -3.2371477817326612e-04, -3.2381963733078714e-04, -3.2387207823078205e-04,
                -3.2389830245521184e-04, -3.2391141582574273e-04, -3.2391797293048982e-04, -3.2392125162270132e-04,
                -3.2392289101542239e-04, -3.2392371072732210e-04
        },
// x = -8.0300
        {
                -3.2512459271477720e-04, -3.2533634910936085e-04, -3.2544226177511006e-04, -3.2549522960027567e-04,
                -3.2552171734432359e-04, -3.2553496249367790e-04, -3.2554158549417570e-04, -3.2554489713637581e-04,
                -3.2554655300479562e-04, -3.2554738095477953e-04
        },
// x = -8.0250
        {
                -3.2675215405938876e-04, -3.2696603759440159e-04, -3.2707301435111837e-04, -3.2712651439540855e-04,
                -3.2715326830691320e-04, -3.2716664655929768e-04, -3.2717333611774528e-04, -3.2717668104106543e-04,
                -3.2717835355076041e-04, -3.2717918982162026e-04
        },
// x = -8.0200
        {
                -3.2838785217347987e-04, -3.2860388421122655e-04, -3.2871193574792969e-04, -3.2876597335847738e-04,
                -3.2879299611188000e-04, -3.2880650880480678e-04, -3.2881326559005761e-04, -3.2881664412895694e-04,
                -3.2881833344716743e-04, -3.2881917812252703e-04
        },
// x = -8.0150
        {
                -3.3003172762778517e-04, -3.3024992974502616e-04, -3.3035906685808148e-04, -3.3041364743573263e-04,
                -3.3044094173234406e-04, -3.3045459021676472e-04, -3.3046141490439322e-04, -3.3046482739669184e-04,
                -3.3046653369233887e-04, -3.3046738685666235e-04
        },
// x = -8.0100
        {
                -3.3168382119424383e-04, -3.3190421518434723e-04, -3.3201444877854492e-04, -3.3206957777839782e-04,
                -3.3209714634666878e-04, -3.3211093198710911e-04, -3.3211782525947831e-04, -3.3212127204639117e-04,
                -3.3212299549009332e-04, -3.3212385722869366e-04
        },
// x = -8.0050
        {
                -3.3334417384698684e-04, -3.3356678172210159e-04, -3.3367812281174473e-04, -3.3373380574369492e-04,
                -3.3376165133948811e-04, -3.3377557551418456e-04, -3.3378253806051452e-04, -3.3378601948668552e-04,
                -3.3378776025077606e-04, -3.3378863064982382e-04
        },
// x = -8.0000
        {
                -3.3501282676332781e-04, -3.3523767075657897e-04, -3.3535013046658284e-04, -3.3540637289587369e-04,
                -3.3543449830273947e-04, -3.3544856240377694e-04, -3.3545559492021321e-04, -3.3545911133374970e-04,
                -3.3546086959229372e-04, -3.3546174873882512e-04
        },
// x = -7.9950
        {
                -3.3668982132476041e-04, -3.3691692389246583e-04, -3.3703051345946851e-04, -3.3708732100724647e-04,
                -3.3711572903670104e-04, -3.3712993447015201e-04, -3.3713703765983575e-04, -3.3714058941234303e-04,
                -3.3714236534115503e-04, -3.3714325332308113e-04
        },
// x = -7.9900
        {
                -3.3837519911795869e-04, -3.3860458294186860e-04, -3.3871931371535233e-04, -3.3877669205922888e-04,
                -3.3880538555103513e-04, -3.3881973373710013e-04, -3.3882690831023792e-04, -3.3883049575685487e-04,
                -3.3883228953351588e-04, -3.3883318643963139e-04
        },
// x = -7.9850
        {
                -3.4006900193578327e-04, -3.4030068992534209e-04, -3.4041657336876622e-04, -3.4047452824338470e-04,
                -3.4050351006583573e-04, -3.4051800243898526e-04, -3.4052524911292045e-04, -3.4052887261235471e-04,
                -3.4053068441623063e-04, -3.4053159033622238e-04
        },
// x = -7.9800
        {
                -3.4177127177829257e-04, -3.4200528707292320e-04, -3.4212233476486810e-04, -3.4218087196247649e-04,
                -3.4221014501268260e-04, -3.4222478302180058e-04, -3.4223210252108465e-04, -3.4223576243564856e-04,
                -3.4223759244790775e-04, -3.4223850747236396e-04
        },
// x = -7.9750
        {
                -3.4348205085375777e-04, -3.4371841682516891e-04, -3.4383664046049207e-04, -3.4389576583152134e-04,
                -3.4392533303569920e-04, -3.4394011814422774e-04, -3.4394751120069310e-04, -3.4395120789633990e-04,
                -3.4395305629997139e-04, -3.4395398052039066e-04
        },
// x = -7.9700
        {
                -3.4520138157968376e-04, -3.4544012183420036e-04, -3.4555953322520320e-04, -3.4561925267885183e-04,
                -3.4564911699261699e-04, -3.4566405067870264e-04, -3.4567151803153561e-04, -3.4567525187789618e-04,
                -3.4567711885772788e-04, -3.4567805236652841e-04
        },
// x = -7.9650
        {
                -3.4692930658383454e-04, -3.4717044496475162e-04, -3.4729105604235859e-04, -3.4735137554718268e-04,
                -3.4738153995584464e-04, -3.4739662371248631e-04, -3.4740416610830088e-04, -3.4740793747872040e-04,
                -3.4740982322143771e-04, -3.4741076611196649e-04
        },
// x = -7.9600
        {
                -3.4866586870526306e-04, -3.4890942929522296e-04, -3.4903125211017261e-04, -3.4909217769468179e-04,
                -3.4912264521354264e-04, -3.4913788054874044e-04, -3.4914549874165315e-04, -3.4914930801322868e-04,
                -3.4915121270739306e-04, -3.4915216507393502e-04
        },
// x = -7.9550
        {
                -3.5041111099534672e-04, -3.5065711811874069e-04, -3.5078016484278778e-04, -3.5084170259604765e-04,
                -3.5087247627070306e-04, -3.5088786470760931e-04, -3.5089555945931443e-04, -3.5089940701293258e-04,
                -3.5090133084900047e-04, -3.5090229278678824e-04
        },
// x = -7.9500
        {
                -3.5216507671882757e-04, -3.5241355494422072e-04, -3.5253783787135172e-04, -3.5259999394359132e-04,
                -3.5263107685023506e-04, -3.5264661992730645e-04, -3.5265439200715231e-04, -3.5265827822752711e-04,
                -3.5266022139786887e-04, -3.5266119300309223e-04
        },
// x = -7.9450
        {
                -3.5392780935485731e-04, -3.5417878349743850e-04, -3.5430431504509803e-04, -3.5436709564832484e-04,
                -3.5439849089405589e-04, -3.5441419016520715e-04, -3.5442204035027289e-04, -3.5442596562598451e-04,
                -3.5442792832490381e-04, -3.5442890969471886e-04
        },
// x = -7.9400
        {
                -3.5569935259804750e-04, -3.5595284772210311e-04, -3.5607964043243357e-04, -3.5614305184105321e-04,
                -3.5617476256418656e-04, -3.5619061959894576e-04, -3.5619854867411906e-04, -3.5620251339765264e-04,
                -3.5620449582140563e-04, -3.5620548705394525e-04
        },
// x = -7.9350
        {
                -3.5747975035952469e-04, -3.5773579178093788e-04, -3.5786385832203095e-04, -3.5792790687347450e-04,
                -3.5795993624385412e-04, -3.5797595262751974e-04, -3.5798396138557518e-04, -3.5798796595335987e-04,
                -3.5798996830017525e-04, -3.5799096949455824e-04
        },
// x = -7.9300
        {
                -3.5926904676799026e-04, -3.5952766005676505e-04, -3.5965701322392617e-04, -3.5972170531928228e-04,
                -3.5975405653859840e-04, -3.5977023387239756e-04, -3.5977832311407624e-04, -3.5978236792652492e-04,
                -3.5978439039662316e-04, -3.5978540165296458e-04
        },
// x = -7.9250
        {
                -3.6106728617078636e-04, -3.6132849715359667e-04, -3.6145914987062184e-04, -3.6152449197527655e-04,
                -3.6155716827738539e-04, -3.6157350817863385e-04, -3.6158167871272283e-04, -3.6158576417427224e-04,
                -3.6158780696988617e-04, -3.6158882838930716e-04
        },
// x = -7.9200
        {
                -3.6287451313496546e-04, -3.6313834789773020e-04, -3.6327031321819586e-04, -3.6333631186247803e-04,
                -3.6336931651372468e-04, -3.6338582061598871e-04, -3.6339407325940246e-04, -3.6339819977855358e-04,
                -3.6340026310394808e-04, -3.6340129478858613e-04
        },
// x = -7.9150
        {
                -3.6469077244836648e-04, -3.6495725733884989e-04, -3.6509054844741556e-04, -3.6515721022724925e-04,
                -3.6519054652679408e-04, -3.6520721648005365e-04, -3.6521555205791518e-04, -3.6521972004727394e-04,
                -3.6522180410876703e-04, -3.6522284616178588e-04
        },
// x = -7.9100
        {
                -3.6651610912069485e-04, -3.6678527075113365e-04, -3.6691990096485763e-04, -3.6698723254242058e-04,
                -3.6702090382256905e-04, -3.6703774129338238e-04, -3.6704616063910629e-04, -3.6705037051542509e-04,
                -3.6705247552140777e-04, -3.6705352804700813e-04
        },
// x = -7.9050
        {
                -3.6835056838460824e-04, -3.6862243363436424e-04, -3.6875841640403306e-04, -3.6882642450842195e-04,
                -3.6886043413495752e-04, -3.6887744080662816e-04, -3.6888594476200339e-04, -3.6889019694622232e-04,
                -3.6889232310718018e-04, -3.6889338621060968e-04
        },
// x = -7.9000
        {
                -3.7019419569680806e-04, -3.7046879171504822e-04, -3.7060614062651829e-04, -3.7067483205442095e-04,
                -3.7070918342694105e-04, -3.7072636099968581e-04, -3.7073495041496006e-04, -3.7073924533224966e-04,
                -3.7074139286078319e-04, -3.7074246664834737e-04
        },
// x = -7.8950
        {
                -3.7204703673913460e-04, -3.7232439094753725e-04, -3.7246311972309110e-04, -3.7253250133946508e-04,
                -3.7256719789172121e-04, -3.7258454808284004e-04, -3.7259322381680483e-04, -3.7259756189660787e-04,
                -3.7259973100745426e-04, -3.7260081558652697e-04
        },
// x = -7.8900
        {
                -3.7390913741966941e-04, -3.7418927751515786e-04, -3.7432940001487334e-04, -3.7439947875363155e-04,
                -3.7443452395387197e-04, -3.7445204849791960e-04, -3.7446081141799602e-04, -3.7446519309407099e-04,
                -3.7446738400412545e-04, -3.7446847948315954e-04
        },
// x = -7.8850
        {
                -3.7578054387384075e-04, -3.7606349783134456e-04, -3.7620502805447802e-04, -3.7627581091918117e-04,
                -3.7631120827049747e-04, -3.7632890891945711e-04, -3.7633775990178231e-04, -3.7634218561224643e-04,
                -3.7634439854058381e-04, -3.7634550502912214e-04
        },
// x = -7.8800
        {
                -3.7766130246553667e-04, -3.7794709854077998e-04, -3.7809005062716325e-04, -3.7816154469171910e-04,
                -3.7819729773239634e-04, -3.7821517625585426e-04, -3.7822411618536951e-04, -3.7822858637274256e-04,
                -3.7823082154063930e-04, -3.7823193914932561e-04
        },
// x = -7.8750
        {
                -3.7955145978822090e-04, -3.7984012652053943e-04, -3.7998451475199069e-04, -3.8005672716136057e-04,
                -3.8009283946523068e-04, -3.8011089765055341e-04, -3.8011992742109218e-04, -3.8012444253234109e-04,
                -3.8012670016329719e-04, -3.8012782900388704e-04
        },
// x = -7.8700
        {
                -3.8145106266605636e-04, -3.8174262888124181e-04, -3.8188846768299106e-04, -3.8196140565390337e-04,
                -3.8199788083070218e-04, -3.8201612048321498e-04, -3.8202524099759241e-04, -3.8202980148417556e-04,
                -3.8203208180393714e-04, -3.8203322198930925e-04
        },
// x = -7.8650
        {
                -3.8336015815503222e-04, -3.8365465296820618e-04, -3.8380195691033430e-04, -3.8387562773200515e-04,
                -3.8391246942773299e-04, -3.8393089237090031e-04, -3.8394010454100334e-04, -3.8394471085891589e-04,
                -3.8394701409549776e-04, -3.8394816573966527e-04
        },
// x = -7.8600
        {
                -3.8527879354409764e-04, -3.8557624636261319e-04, -3.8572503016150633e-04, -3.8579944119636713e-04,
                -3.8583665309365335e-04, -3.8585526116926113e-04, -3.8586456591613977e-04, -3.8586921852595905e-04,
                -3.8587154490966739e-04, -3.8587270812778915e-04
        },
// x = -7.8550
        {
                -3.8720701635629981e-04, -3.8750745688267323e-04, -3.8765773540249066e-04, -3.8773289408692386e-04,
                -3.8777047990539419e-04, -3.8778927497373387e-04, -3.8779867322769344e-04, -3.8780337259462522e-04,
                -3.8780572235808060e-04, -3.8780689726647291e-04
        },
// x = -7.8500
        {
                -3.8914487434992830e-04, -3.8944833258479941e-04, -3.8960012083895710e-04, -3.8967603468403845e-04,
                -3.8971399818068739e-04, -3.8973298212074145e-04, -3.8974247482143515e-04, -3.8974722141536013e-04,
                -3.8974959479352091e-04, -3.8975078150966879e-04
        },
// x = -7.8450
        {
                -3.9109241551966425e-04, -3.9139892176478684e-04, -3.9155223491745564e-04, -3.9162891150970396e-04,
                -3.9166725647926936e-04, -3.9168643118889999e-04, -3.9169601928542322e-04, -3.9170081358094370e-04,
                -3.9170321081112938e-04, -3.9170440945369880e-04
        },
// x = -7.8400
        {
                -3.9304968809773539e-04, -3.9335927295899734e-04, -3.9351412632661586e-04, -3.9359157332875122e-04,
                -3.9363030360409375e-04, -3.9364967100023160e-04, -3.9365935545121651e-04, -3.9366419792770440e-04,
                -3.9366661924961934e-04, -3.9366782993846864e-04
        },
// x = -7.8350
        {
                -3.9501674055507650e-04, -3.9532943494555021e-04, -3.9548584399835325e-04, -3.9556406915006143e-04,
                -3.9560318860254760e-04, -3.9562275062138394e-04, -3.9563253239509544e-04, -3.9563742353673965e-04,
                -3.9563986919249715e-04, -3.9564109204868922e-04
        },
// x = -7.8300
        {
                -3.9699362160249501e-04, -3.9730945674551873e-04, -3.9746743710908074e-04, -3.9754644822778662e-04,
                -3.9758596076767527e-04, -3.9760571936485510e-04, -3.9761559943928779e-04, -3.9762053973514276e-04,
                -3.9762300996928917e-04, -3.9762424511510368e-04
        },
// x = -7.8250
        {
                -3.9898038019184253e-04, -3.9929938762413221e-04, -3.9945895508092651e-04, -3.9953876006257436e-04,
                -3.9957866963940750e-04, -3.9959862679022517e-04, -3.9960860615320063e-04, -3.9961359609723553e-04,
                -3.9961609115677483e-04, -3.9961733871572008e-04
        },
// x = -7.8200
        {
                -4.0097706551719210e-04, -4.0129927709198432e-04, -4.0146044758295775e-04, -4.0154105440279975e-04,
                -4.0158136500579694e-04, -4.0160152270539374e-04, -4.0161160235465976e-04, -4.0161664244580742e-04,
                -4.0161916258022579e-04, -4.0162042267705145e-04
        },
// x = -7.8150
        {
                -4.0298372701602018e-04, -4.0330917490624695e-04, -4.0347196453241062e-04, -4.0355338124580312e-04,
                -4.0359409690426013e-04, -4.0361445716782329e-04, -4.0362463811115338e-04, -4.0362972885336055e-04,
                -4.0363227431465125e-04, -4.0363354707536063e-04
        },
// x = -7.8100
        {
                -4.0500041437039540e-04, -4.0532913107189065e-04, -4.0549355609592602e-04, -4.0557579083913401e-04,
                -4.0561691562282542e-04, -4.0563748048578981e-04, -4.0564776374108386e-04, -4.0565290564336130e-04,
                -4.0565547668605011e-04, -4.0565676223791288e-04
        },
// x = -7.8050
        {
                -4.0702717750817219e-04, -4.0735919584291018e-04, -4.0752527269079188e-04, -4.0760833368180122e-04,
                -4.0764987170138696e-04, -4.0767064321963815e-04, -4.0768102981502433e-04, -4.0768622339149789e-04,
                -4.0768882027266833e-04, -4.0769011874423308e-04
        },
// x = -7.8000
        {
                -4.0906406660418994e-04, -4.0939941972355627e-04, -4.0956716498619111e-04, -4.0965106052552933e-04,
                -4.0969301593296505e-04, -4.0971399618304515e-04, -4.0972448715698260e-04, -4.0972973292694467e-04,
                -4.0973235590626372e-04, -4.0973366742737093e-04
        },
// x = -7.7950
        {
                -4.1111113208147876e-04, -4.1144985346957429e-04, -4.1161928390445632e-04, -4.1170402237602116e-04,
                -4.1174639936497333e-04, -4.1176759044428830e-04, -4.1177818684567095e-04, -4.1178348533363210e-04,
                -4.1178613467337632e-04, -4.1178745937517136e-04
        },
// x = -7.7900
        {
                -4.1316842461246997e-04, -4.1351054808944738e-04, -4.1368168062232982e-04, -4.1376727049422641e-04,
                -4.1381007330049124e-04, -4.1383147732752053e-04, -4.1384218021578185e-04, -4.1384753195152353e-04,
                -4.1385020791660511e-04, -4.1385154593155129e-04
        },
// x = -7.7850
        {
                -4.1523599512021249e-04, -4.1558155484564690e-04, -4.1575440657223118e-04, -4.1584085639761754e-04,
                -4.1588408929954393e-04, -4.1590570841405208e-04, -4.1591651885927126e-04, -4.1592192437789853e-04,
                -4.1592462723589194e-04, -4.1592597869778377e-04
        },
// x = -7.7800
        {
                -4.1731389477959579e-04, -4.1766292525588885e-04, -4.1783751344352940e-04, -4.1792483186147068e-04,
                -4.1796849918038773e-04, -4.1799033554363848e-04, -4.1800125462664706e-04, -4.1800671446864224e-04,
                -4.1800944448981090e-04, -4.1801080953378752e-04
        },
// x = -7.7750
        {
                -4.1940217501857790e-04, -4.1975471109439544e-04, -4.1993105318382346e-04, -4.2001924892015396e-04,
                -4.2006335502080288e-04, -4.2008541081577474e-04, -4.2009643962826468e-04, -4.2010195433954156e-04,
                -4.2010471179686475e-04, -4.2010609055942325e-04
        },
// x = -7.7700
        {
                -4.2150088751941891e-04, -4.2185696439316414e-04, -4.2203507800022680e-04, -4.2212415986842190e-04,
                -4.2216870915939137e-04, -4.2219098659099620e-04, -4.2220212623562931e-04, -4.2220769636758720e-04,
                -4.2221048153678762e-04, -4.2221187415579687e-04
        },
// x = -7.7650
        {
                -4.2361008421992134e-04, -4.2396973744324206e-04, -4.2414964036066014e-04, -4.2423961726271574e-04,
                -4.2428461419688311e-04, -4.2430711549218614e-04, -4.2431836708270399e-04, -4.2432399319228330e-04,
                -4.2432680635185436e-04, -4.2432821296656899e-04
        },
// x = -7.7600
        {
                -4.2572981731467570e-04, -4.2609308279600681e-04, -4.2627479299514983e-04, -4.2636567392247132e-04,
                -4.2641112299744699e-04, -4.2643385040588947e-04, -4.2644521506722468e-04, -4.2645089771696260e-04,
                -4.2645373914819640e-04, -4.2645515989927027e-04
        },
// x = -7.7550
        {
                -4.2786013925631174e-04, -4.2822705326445328e-04, -4.2841058889713239e-04, -4.2850238293143232e-04,
                -4.2854828869000919e-04, -4.2857124448363314e-04, -4.2858272335202216e-04, -4.2858846311010857e-04,
                -4.2859133309712397e-04, -4.2859276812662447e-04
        },
// x = -7.7500
        {
                -4.3000110275675661e-04, -4.3037170192448748e-04, -4.3055708132476634e-04, -4.3064979763897127e-04,
                -4.3069616466957810e-04, -4.3071935114325367e-04, -4.3073094536634992e-04, -4.3073674280668452e-04,
                -4.3073964163645563e-04, -4.3074109108787791e-04
        },
// x = -7.7450
        {
                -4.3215276078849757e-04, -4.3252708211622581e-04, -4.3271432380224940e-04, -4.3280797166141561e-04,
                -4.3285480459857612e-04, -4.3287822407023040e-04, -4.3288993480721931e-04, -4.3289579050946868e-04,
                -4.3289871847185360e-04, -4.3290018249013488e-04
        },
// x = -7.7400
        {
                -4.3431516658585167e-04, -4.3469324744530111e-04, -4.3488237012114330e-04, -4.3497695888338183e-04,
                -4.3502426240817717e-04, -4.3504791721902614e-04, -4.3505974564074089e-04, -4.3506566019039670e-04,
                -4.3506861757816622e-04, -4.3507009630970025e-04
        },
// x = -7.7350
        {
                -4.3648837364624173e-04, -4.3687025178417528e-04, -4.3706127434170445e-04, -4.3715681345911546e-04,
                -4.3720459229965183e-04, -4.3722848481443421e-04, -4.3724043210347283e-04, -4.3724640609191039e-04,
                -4.3724939320077740e-04, -4.3725088679342925e-04
        },
// x = -7.7300
        {
                -4.3867243573147654e-04, -4.3905814927345756e-04, -4.3925109079422147e-04, -4.3934758981383767e-04,
                -4.3939584874571881e-04, -4.3941998135293243e-04, -4.3943204870377589e-04, -4.3943808272831292e-04,
                -4.3944109985696232e-04, -4.3944260846008284e-04
        },
// x = -7.7250
        {
                -4.4086740686903966e-04, -4.4125699432322988e-04, -4.4145187408035886e-04, -4.4154934264509926e-04,
                -4.4159808649190318e-04, -4.4162246160404369e-04, -4.4163465022317532e-04, -4.4164074488713204e-04,
                -4.4164379233725004e-04, -4.4164531610169116e-04
        },
// x = -7.7200
        {
                -4.4307334135338191e-04, -4.4346684161437839e-04, -4.4366367907450822e-04, -4.4376212692414020e-04,
                -4.4381136055790153e-04, -4.4383598061170328e-04, -4.4384829171772934e-04, -4.4385444763048865e-04,
                -4.4385752570679318e-04, -4.4385906478492277e-04
        },
// x = -7.7150
        {
                -4.4529029374722185e-04, -4.4568774609993184e-04, -4.4588656092514546e-04, -4.4598599789725721e-04,
                -4.4603572623895348e-04, -4.4606059369563320e-04, -4.4607302851940478e-04, -4.4607924629647318e-04,
                -4.4608235530674457e-04, -4.4608390985246171e-04
        },
// x = -7.7100
        {
                -4.4751831888285089e-04, -4.4791976300640580e-04, -4.4812057505619454e-04, -4.4822101108717762e-04,
                -4.4827123910722060e-04, -4.4829635645272340e-04, -4.4830891623745951e-04, -4.4831519650052849e-04,
                -4.4831833675563988e-04, -4.4831990692439038e-04
        },
// x = -7.7050
        {
                -4.4975747186344579e-04, -4.5016294783515427e-04, -4.5036577716839874e-04, -4.5046722229443971e-04,
                -4.5051795501317265e-04, -4.5054332475842005e-04, -4.5055601075983152e-04, -4.5056235413684013e-04,
                -4.5056552595078867e-04, -4.5056711189958034e-04
        },
// x = -7.7000
        {
                -4.5200780806438638e-04, -4.5241735636372644e-04, -4.5262222324069776e-04, -4.5272468759878027e-04,
                -4.5277593008697867e-04, -4.5280155476811997e-04, -4.5281436825453543e-04, -4.5282077537973253e-04,
                -4.5282397906967102e-04, -4.5282558095708953e-04
        },
// x = -7.6950
        {
                -4.5426938313457998e-04, -4.5468304464723231e-04, -4.5488996953161270e-04, -4.5499346336052942e-04,
                -4.5504522073990780e-04, -4.5507110291857333e-04, -4.5508404517106564e-04, -4.5509051668507348e-04,
                -4.5509375257134178e-04, -4.5509537055756671e-04
        },
// x = -7.6900
        {
                -4.5654225299779236e-04, -4.5696006901971254e-04, -4.5716907258063681e-04, -4.5727360622201157e-04,
                -4.5732588366573517e-04, -4.5735202592929209e-04, -4.5736509824180656e-04, -4.5737163479168502e-04,
                -4.5737490319784212e-04, -4.5737653744466257e-04
        },
// x = -7.6850
        {
                -4.5882647385398390e-04, -4.5924848609551707e-04, -4.5945958920963384e-04, -4.5956517310895409e-04,
                -4.5961797584215529e-04, -4.5964438080396659e-04, -4.5965758448345001e-04, -4.5966418672276126e-04,
                -4.5966748797561762e-04, -4.5966913864644877e-04
        },
// x = -7.6800
        {
                -4.6112210218065338e-04, -4.6154835277068852e-04, -4.6176157652424267e-04, -4.6186822123190225e-04,
                -4.6192155453220294e-04, -4.6194822483188792e-04, -4.6196156119841981e-04, -4.6196822978729319e-04,
                -4.6197156421694366e-04, -4.6197323147684281e-04
        },
// x = -7.6750
        {
                -4.6342919473418685e-04, -4.6385972622435355e-04, -4.6407509191528972e-04, -4.6418280808764199e-04,

                -4.6423667728568032e-04, -4.6426361558937897e-04, -4.6427708597630287e-04, -4.6428382158150163e-04,
                -4.6428718952135805e-04, -4.6428887353704135e-04
        },
// x = -7.6700
        {
                -4.6574780855121391e-04, -4.6618266392012124e-04, -4.6640019306020791e-04, -4.6650899146062961e-04,
                -4.6656340194059271e-04, -4.6659061094123170e-04, -4.6660421669528879e-04, -4.6661101999027600e-04,
                -4.6661442177710084e-04, -4.6661612271695985e-04
        },
// x = -7.6650
        {
                -4.6807800094996968e-04, -4.6851722360748696e-04, -4.6873693792446171e-04, -4.6884682942442766e-04,
                -4.6890178662458928e-04, -4.6892926904215133e-04, -4.6894301152361484e-04, -4.6894988318862133e-04,
                -4.6895331916256121e-04, -4.6895503719667978e-04
        },
// x = -7.6600
        {
                -4.7041982953166316e-04, -4.7086346332324405e-04, -4.7108538476298134e-04, -4.7119638034314923e-04,
                -4.7125188975641303e-04, -4.7127964833820844e-04, -4.7129352892101995e-04, -4.7130046964311174e-04,
                -4.7130394014773176e-04, -4.7130567544790273e-04
        },
// x = -7.6550
        {
                -4.7277335218185270e-04, -4.7322144139290255e-04, -4.7344559212160199e-04, -4.7355770287290879e-04,
                -4.7361377004735694e-04, -4.7364180756829843e-04, -4.7365582764020476e-04, -4.7366283811335218e-04,
                -4.7366634349566965e-04, -4.7366809623541266e-04
        },
// x = -7.6500
        {
                -4.7513862707182725e-04, -4.7559121643211375e-04, -4.7581761883851089e-04, -4.7593085596328013e-04,
                -4.7598748650272752e-04, -4.7601580576560711e-04, -4.7602996672829924e-04, -4.7603704765344630e-04,
                -4.7604058826396603e-04, -4.7604235861854435e-04
        },
// x = -7.6450
        {
                -4.7751571265999387e-04, -4.7797284734810256e-04, -4.7820152404570211e-04, -4.7831589885876199e-04,
                -4.7837309842331581e-04, -4.7840170225908529e-04, -4.7841600552833859e-04, -4.7842315761347258e-04,
                -4.7842673380622157e-04, -4.7842852195265984e-04
        },
// x = -7.6400
        {
                -4.7990466769327303e-04, -4.8036639334110684e-04, -4.8059736717043797e-04, -4.8071289110025066e-04,
                -4.8077066540687602e-04, -4.8079955667492937e-04, -4.8081400368074487e-04, -4.8082122764096774e-04,
                -4.8082483977353058e-04, -4.8082664589063243e-04
        },
// x = -7.6350
        {
                -4.8230555120849897e-04, -4.8277191390582309e-04, -4.8300520793671711e-04, -4.8312189252651980e-04,
                -4.8318024734961040e-04, -4.8320942893807003e-04, -4.8322402112481748e-04, -4.8323131768241692e-04,
                -4.8323496611597187e-04, -4.8323679038433779e-04
        },
// x = -7.6300
        {
                -4.8471842253382735e-04, -4.8518946883285982e-04, -4.8542510636675091e-04, -4.8554296327570816e-04,
                -4.8560190444766341e-04, -4.8563137927366828e-04, -4.8564611810023034e-04, -4.8565348798475211e-04,
                -4.8565717308410685e-04, -4.8565901568615229e-04
        },
// x = -7.6250
        {
                -4.8714334129014994e-04, -4.8761911821019744e-04, -4.8785712278244699e-04, -4.8797616378681413e-04,
                -4.8803569719862137e-04, -4.8806546820861878e-04, -4.8808035514853699e-04, -4.8808779909685802e-04,
                -4.8809152123048625e-04, -4.8809338235045982e-04
        },
// x = -7.6200
        {
                -4.8958036739251483e-04, -4.9006092242465571e-04, -4.9030131780689891e-04, -4.9042155480119793e-04,
                -4.9048168640302087e-04, -4.9051175657306083e-04, -4.9052679311468279e-04, -4.9053431187108434e-04,
                -4.9053807141116301e-04, -4.9053995123516476e-04
        },
// x = -7.6150
        {
                -4.9202956105155390e-04, -4.9251494216336793e-04, -4.9275775236588457e-04, -4.9287919736409132e-04,
                -4.9293993316586475e-04, -4.9297030550189680e-04, -4.9298549314852531e-04, -4.9299308746476801e-04,
                -4.9299688478721322e-04, -4.9299878350321416e-04
        },
// x = -7.6100
        {
                -4.9449098277491753e-04, -4.9498123841526271e-04, -4.9522648768937183e-04, -4.9534915282611479e-04,
                -4.9541049889814410e-04, -4.9544117643631890e-04, -4.9545651670636153e-04, -4.9546418734175998e-04,
                -4.9546802282626637e-04, -4.9546994062412558e-04
        },
// x = -7.6050
        {
                -4.9696469336871500e-04, -4.9745987247255190e-04, -4.9770758531303004e-04, -4.9783148284480213e-04,
                -4.9789344531837041e-04, -4.9792443112534176e-04, -4.9793992555246310e-04, -4.9794767327396218e-04,
                -4.9795154730403982e-04, -4.9795348437552480e-04
        },
// x = -7.6000
        {
                -4.9945075393896218e-04, -4.9995090593222787e-04, -5.0020110707975178e-04, -5.0032624938613375e-04,
                -5.0038883445411339e-04, -5.0042013162734510e-04, -5.0043578176061992e-04, -5.0044360734287157e-04,
                -5.0044752030588399e-04, -5.0044947684468963e-04
        },
// x = -7.5950
        {
                -5.0194922589303606e-04, -5.0245440069756524e-04, -5.0270711514117884e-04, -5.0283351472607454e-04,
                -5.0289672864354629e-04, -5.0292834031162256e-04, -5.0294414771569033e-04, -5.0295205194113019e-04,
                -5.0295600422833420e-04, -5.0295798043010153e-04
        },
// x = -7.5900
        {
                -5.0446017094113514e-04, -5.0497041897963222e-04, -5.0522567195923878e-04, -5.0535334145212336e-04,
                -5.0541719053700097e-04, -5.0544911985993745e-04, -5.0546508611515968e-04, -5.0547306977408580e-04,
                -5.0547706178066972e-04, -5.0547905784300608e-04
        },
// x = -7.5850
        {
                -5.0698365109774898e-04, -5.0749902329880788e-04, -5.0775684030768720e-04, -5.0788579246486762e-04,
                -5.0795028309852874e-04, -5.0798253326808922e-04, -5.0799865997070608e-04, -5.0800672386135841e-04,
                -5.0801075598648211e-04, -5.0801277210898066e-04
        },
// x = -7.5800
        {
                -5.0951972868313201e-04, -5.1004027648630804e-04, -5.1030068327365759e-04, -5.1043093097954604e-04,
                -5.1049606960746957e-04, -5.1052864384748509e-04, -5.1054493260977590e-04, -5.1055307753841532e-04,
                -5.1055715018524957e-04, -5.1055918656950903e-04
        },
// x = -7.5750
        {
                -5.1206846632478474e-04, -5.1259424168571669e-04, -5.1285726425922011e-04, -5.1298882052762005e-04,
                -5.1305461366002968e-04, -5.1308751522671987e-04, -5.1310396767716387e-04, -5.1311219445815317e-04,
                -5.1311630803392051e-04, -5.1311836488356598e-04
        },
// x = -7.5700
        {
                -5.1462992695894402e-04, -5.1516098235452690e-04, -5.1542664698294696e-04, -5.1555952495835160e-04,
                -5.1562597917086659e-04, -5.1565921135316546e-04, -5.1567582913660470e-04, -5.1568413859248989e-04,
                -5.1568829350850508e-04, -5.1569037102920765e-04
        },
// x = -7.5650
        {
                -5.1720417383207759e-04, -5.1774056226568793e-04, -5.1800889548148587e-04, -5.1814310844039066e-04,
                -5.1821023037468175e-04, -5.1824379649456642e-04, -5.1826058127236996e-04, -5.1826897423396261e-04,
                -5.1827317090567360e-04, -5.1827526930517149e-04
        },
// x = -7.5600
        {
                -5.1979127050238625e-04, -5.2033304550915980e-04, -5.2060407411114098e-04, -5.2073963546336811e-04,
                -5.2080743182782152e-04, -5.2084133524064402e-04, -5.2085828869087439e-04, -5.2086676599733439e-04,
                -5.2087100484436396e-04, -5.2087312433248315e-04
        },
// x = -7.5550
        {
                -5.2239128084131524e-04, -5.2293849649347602e-04, -5.2321224754946184e-04, -5.2334917083949933e-04,
                -5.2341764840988685e-04, -5.2345189250470878e-04, -5.2346901632228915e-04, -5.2347757882120890e-04,
                -5.2348186026739715e-04, -5.2348400105607185e-04
        },
// x = -7.5500
        {
                -5.2500426903506867e-04, -5.2555697994731421e-04, -5.2583348079684122e-04, -5.2597177970519382e-04,
                -5.2604094532534857e-04, -5.2607553352528112e-04, -5.2609282942216488e-04, -5.2610147796965394e-04,
                -5.2610580244309962e-04, -5.2610796474639415e-04
        },
// x = -7.5450
        {
                -5.2763029958613516e-04, -5.2818856092107214e-04, -5.2846783917811775e-04, -5.2860752752267360e-04,
                -5.2867738810517427e-04, -5.2871232386771886e-04, -5.2872979357306102e-04, -5.2873852903383156e-04,
                -5.2874289696693574e-04, -5.2874508100106562e-04
        },
// x = -7.5400
        {
                -5.3026943731481727e-04, -5.3083330478845465e-04, -5.3111538834419065e-04, -5.3125648008159896e-04,
                -5.3132704260846022e-04, -5.3136232942585439e-04, -5.3137997468618409e-04, -5.3138879793363750e-04,
                -5.3139320976314654e-04, -5.3139541574649999e-04
        },
// x = -7.5350
        {
                -5.3292174736077006e-04, -5.3349127724806563e-04, -5.3377619427363886e-04, -5.3391870350070374e-04,
                -5.3398997502407310e-04, -5.3402561642363971e-04, -5.3404343900303466e-04, -5.3405235091934862e-04,
                -5.3405680708639770e-04, -5.3405903523955794e-04
        },
// x = -7.5300
        {
                -5.3558729518454675e-04, -5.3616254432500832e-04, -5.3645032327434951e-04, -5.3659426422943655e-04,
                -5.3666625187229905e-04, -5.3670225141679853e-04, -5.3672025309706183e-04, -5.3672925457327880e-04,
                -5.3673375552343598e-04, -5.3673600606920265e-04
        },
// x = -7.5250
        {
                -5.3826614656914975e-04, -5.3884717237249503e-04, -5.3913784198515551e-04, -5.3928322904961293e-04,
                -5.3935594000650204e-04, -5.3939230129448825e-04, -5.3941048387532581e-04, -5.3941957581144190e-04,
                -5.3942412199475310e-04, -5.3942639515816484e-04
        },
// x = -7.5200
        {
                -5.4095836762159085e-04, -5.4154522807346240e-04, -5.4183881737747939e-04, -5.4198566507707288e-04,
                -5.4205910661478949e-04, -5.4209583328096922e-04, -5.4211419858017011e-04, -5.4212338188522509e-04,
                -5.4212797375625846e-04, -5.4213026976461585e-04
        },
// x = -7.5150
        {
                -5.4366402477445811e-04, -5.4425677844219440e-04, -5.4455331675698558e-04, -5.4470163976334861e-04,
                -5.4477581922168631e-04, -5.4481291493728272e-04, -5.4483146479090061e-04, -5.4484074038306865e-04,
                -5.4484537840096081e-04, -5.4484769748384813e-04
        },
// x = -7.5100
        {
                -5.4638318478748989e-04, -5.4698189082595650e-04, -5.4728140776524198e-04, -5.4743122089733940e-04,
                -5.4750614568981786e-04, -5.4754361416293692e-04, -5.4756235042547382e-04, -5.4757171923215577e-04,
                -5.4757640386065664e-04, -5.4757874624996575e-04
        },
// x = -7.5050
        {
                -5.4911591474915552e-04, -5.4972063290663288e-04, -5.5002315838138891e-04, -5.5017447660699612e-04,
                -5.5025015422160105e-04, -5.5028799919760215e-04, -5.5030692374219349e-04, -5.5031638670010891e-04,
                -5.5032111840762896e-04, -5.5032348433758211e-04
        },
// x = -7.5000
        {
                -5.5186228207824495e-04, -5.5247307270237607e-04, -5.5277863692381542e-04, -5.5293147536101238e-04,
                -5.5300791336094352e-04, -5.5304613862281382e-04, -5.5306525334141582e-04, -5.5307481139669687e-04,
                -5.5307959065635374e-04, -5.5308198036352666e-04
        },
// x = -7.4950
        {
                -5.5462235452546449e-04, -5.5523927856926121e-04, -5.5554791205184636e-04, -5.5570228597052551e-04,
                -5.5577949199495200e-04, -5.5581810136368429e-04, -5.5583740816726268e-04, -5.5584706227554913e-04,
                -5.5585188956521468e-04, -5.5585430328856076e-04
        },
// x = -7.4900
        {
                -5.5739620017503989e-04, -5.5801931920295008e-04, -5.5833105276743454e-04, -5.5848697759082510e-04,
                -5.5856495935564842e-04, -5.5860395669062284e-04, -5.5862345750934427e-04, -5.5863320863587822e-04,
                -5.5863808443822667e-04, -5.5864052241910061e-04
        },
// x = -7.4850
        {
                -5.6018388744632759e-04, -5.6081326364036330e-04, -5.6112812841686399e-04, -5.6128561972307116e-04,
                -5.6136438502169515e-04, -5.6140377422106457e-04, -5.6142347100448966e-04, -5.6143332012421247e-04,
                -5.6143824492676836e-04, -5.6144070740895039e-04
        },
// x = -7.4800
        {
                -5.6298548509543279e-04, -5.6362118126135851e-04, -5.6393920869245932e-04, -5.6409828221601907e-04,
                -5.6417783892012837e-04, -5.6421762392120842e-04, -5.6423751863848588e-04, -5.6424746673613589e-04,
                -5.6425244103132279e-04, -5.6425492826104305e-04
        },
// x = -7.4750
        {
                -5.6580106221683650e-04, -5.6644314179042014e-04, -5.6676436363430592e-04, -5.6692503526775434e-04,
                -5.6700539132810142e-04, -5.6704557610776313e-04, -5.6706567074782702e-04, -5.6707571881803754e-04,
                -5.6708074310322758e-04, -5.6708325532919064e-04
        },
// x = -7.4700
        {
                -5.6863068824502660e-04, -5.6927921529835348e-04, -5.6960366363197664e-04, -5.6976594942743648e-04,
                -5.6984711287463386e-04, -5.6988770144970164e-04, -5.6990799802147051e-04, -5.6991814706886933e-04,
                -5.6992322184643257e-04, -5.6992575931984213e-04
        },
// x = -7.4650
        {
                -5.7147443295614119e-04, -5.7212947220399032e-04, -5.7245717942626710e-04, -5.7262109559704974e-04,
                -5.7270307454237350e-04, -5.7274407097002600e-04, -5.7276457150260242e-04, -5.7277482254191236e-04,
                -5.7277994831926777e-04, -5.7278251129385196e-04
        },
// x = -7.4600
        {
                -5.7433236646961636e-04, -5.7499398327590069e-04, -5.7532498211094161e-04, -5.7549054503316433e-04,
                -5.7557334766936293e-04, -5.7561475604753860e-04, -5.7563546259041369e-04, -5.7564581664655405e-04,
                -5.7565099393621874e-04, -5.7565358266825596e-04
        },
// x = -7.4550
        {
                -5.7720455924984131e-04, -5.7787281963411264e-04, -5.7820714313448459e-04, -5.7837436934870504e-04,
                -5.7845800395081842e-04, -5.7849982841862461e-04, -5.7852074304188140e-04, -5.7853120115007020e-04,
                -5.7853643046971233e-04, -5.7853904521805596e-04
        },
// x = -7.4500
        {
                -5.8009108210782393e-04, -5.8076605275184361e-04, -5.8110373430186304e-04, -5.8127264051472988e-04,
                -5.8135711544091519e-04, -5.8139936017904189e-04, -5.8142048497356285e-04, -5.8143104817942072e-04,
                -5.8143633005191002e-04, -5.8143897107801508e-04
        },
// x = -7.4450
        {
                -5.8299200620286044e-04, -5.8367375445723426e-04, -5.8401482777629683e-04, -5.8418543086221633e-04,
                -5.8427075455458235e-04, -5.8431342378572053e-04, -5.8433476086339631e-04, -5.8434543022305022e-04,
                -5.8435076517651099e-04, -5.8435343274445975e-04
        },
// x = -7.4400
        {
                -5.8590740304421680e-04, -5.8659599693509739e-04, -5.8694049608103724e-04, -5.8711281308385750e-04,
                -5.8719899406930824e-04, -5.8724209205857016e-04, -5.8726364355251106e-04, -5.8727442013270081e-04,
                -5.8727980870056443e-04, -5.8728250307709317e-04
        },
// x = -7.4350
        {
                -5.8883734449281387e-04, -5.8953285272866972e-04, -5.8988081210115497e-04, -5.9005486023586639e-04,
                -5.9014190712695209e-04, -5.9018543818229895e-04, -5.9020720624704725e-04, -5.9021809112523159e-04,
                -5.9022353384629016e-04, -5.9022625530081555e-04
        },
// x = -7.4300
        {
                -5.9178190276292333e-04, -5.9248439474137587e-04, -5.9283584908533654e-04, -5.9301164573978972e-04,
                -5.9309956723556725e-04, -5.9314353570823848e-04, -5.9316552251998484e-04, -5.9317651678444906e-04,
                -5.9318201420290831e-04, -5.9318476300755456e-04
        },
// x = -7.4250
        {
                -5.9474115042386898e-04, -5.9545069623859821e-04, -5.9580568064769037e-04, -5.9598324338433050e-04,
                -5.9607204827123164e-04, -5.9611645855618017e-04, -5.9613866631298001e-04, -5.9614977106294518e-04,
                -5.9615532372848034e-04, -5.9615810015810648e-04
        },
// x = -7.4200
        {
                -5.9771516040173906e-04, -5.9843183084945701e-04, -5.9879038076955920e-04, -5.9896972732717958e-04,
                -5.9905942447988807e-04, -5.9910428101621934e-04, -5.9912671193821363e-04, -5.9913792828394604e-04,
                -5.9914353675175557e-04, -5.9914634108398251e-04
        },
// x = -7.4150
        {
                -6.0070400598110317e-04, -6.0142787256859884e-04, -6.0179002380134582e-04, -6.0197117209685638e-04,
                -6.0206177047919486e-04, -6.0210707775061026e-04, -6.0212973408024632e-04, -6.0214106314316880e-04,
                -6.0214672797403025e-04, -6.0214956048926910e-04
        },
// x = -7.4100
        {
                -6.0370776080673899e-04, -6.0443889575799274e-04, -6.0480468446434286e-04, -6.0498765259455877e-04,
                -6.0507916126038353e-04, -6.0512492379562785e-04, -6.0514780779788435e-04, -6.0515925071068797e-04,
                -6.0516497247101446e-04, -6.0516783345249397e-04
        },
// x = -7.4050
        {
                -6.0672649888536610e-04, -6.0746497514873547e-04, -6.0783443785257497e-04, -6.0801924409602205e-04,
                -6.0811167219012674e-04, -6.0815789456344167e-04, -6.0818100852605395e-04, -6.0819256643281181e-04,
                -6.0819834569470811e-04, -6.0820123542850368e-04
        },
// x = -7.4000
        {
                -6.0976029458738898e-04, -6.1050618584286608e-04, -6.1087935943464941e-04, -6.1106602225338756e-04,
                -6.1115937901241684e-04, -6.1120606584399680e-04, -6.1122941207768645e-04, -6.1124108613396758e-04,
                -6.1124692347528753e-04, -6.1124984225034912e-04
        },
// x = -7.3950
        {
                -6.1280922264864623e-04, -6.1356260331518861e-04, -6.1393952505561420e-04, -6.1412806309707978e-04,
                -6.1422235785045108e-04, -6.1426951380690509e-04, -6.1429309464561092e-04, -6.1430488601859560e-04,
                -6.1431078202300003e-04, -6.1431373013118165e-04
        },
// x = -7.3900
        {
                -6.1587335817216973e-04, -6.1663430341510350e-04, -6.1701501093882716e-04, -6.1720544303769389e-04,
                -6.1730068520852890e-04, -6.1734831500334772e-04, -6.1737213280445768e-04, -6.1738404267305477e-04,
                -6.1738999793006951e-04, -6.1739297566615770e-04
        },
// x = -7.3850
        {
                -6.1895277662994989e-04, -6.1972136236844791e-04, -6.2010589368783416e-04, -6.2029823886789099e-04,
                -6.2039443797395693e-04, -6.2044254636798245e-04, -6.2046660351257083e-04, -6.2047863306753518e-04,
                -6.2048464817261002e-04, -6.2048765583435435e-04
        },
// x = -7.3800
        {
                -6.2204755386471139e-04, -6.2282385677934484e-04, -6.2321225028825426e-04, -6.2340652776430422e-04,
                -6.2350369341896436e-04, -6.2355228522086600e-04, -6.2357658411393076e-04, -6.2358873455798205e-04,
                -6.2359481011255082e-04, -6.2359784800069228e-04
        },
// x = -7.3750
        {
                -6.2515776609169496e-04, -6.2594186363206211e-04, -6.2633415810967714e-04, -6.2653038728945466e-04,
                -6.2662852920262731e-04, -6.2667760926938241e-04, -6.2670215234008481e-04, -6.2671442488802841e-04,
                -6.2672056149956873e-04, -6.2672362991787083e-04
        },
// x = -7.3700
        {
                -6.2828348990044756e-04, -6.2907546029287824e-04, -6.2947169490756795e-04, -6.2966989539367459e-04,
                -6.2976902337280333e-04, -6.2981859661018235e-04, -6.2984338631209087e-04, -6.2985578219093845e-04,
                -6.2986198047303187e-04, -6.2986507972831126e-04
        },
// x = -7.3650
        {
                -6.3142480225662268e-04, -6.3222472451195891e-04, -6.3262493882518245e-04, -6.3282513041704267e-04,
                -6.3292525436807522e-04, -6.3297532573113155e-04, -6.3300036454246630e-04, -6.3301288499155917e-04,
                -6.3301914556395322e-04, -6.3302227596610946e-04
        },
// x = -7.3600
        {
                -6.3458178050378655e-04, -6.3538973442524250e-04, -6.3579396839549059e-04, -6.3599617109132676e-04,
                -6.3609730101970435e-04, -6.3614787551326951e-04, -6.3617316593715006e-04, -6.3618581220828267e-04,
                -6.3619213569695318e-04, -6.3619529755899978e-04
        },
// x = -7.3550
        {
                -6.3775450236523451e-04, -6.3857056855633372e-04, -6.3897886254311062e-04, -6.3918309654193881e-04,
                -6.3928524255359479e-04, -6.3933632523277800e-04, -6.3936186979747360e-04, -6.3937464315501900e-04,
                -6.3938103019223181e-04, -6.3938422383032841e-04
        },
// x = -7.3500
        {
                -6.4094304594581350e-04, -6.4176730581840652e-04, -6.4217970058625144e-04, -6.4238598628989609e-04,
                -6.4248915859226486e-04, -6.4254075456295881e-04, -6.4256655582214115e-04, -6.4257945754317685e-04,
                -6.4258590876755132e-04, -6.4258913450103438e-04
        },
// x = -7.3450
        {
                -6.4414748973375569e-04, -6.4498002551611696e-04, -6.4539656223866655e-04, -6.4560492025379562e-04,
                -6.4570912915683232e-04, -6.4576124357622246e-04, -6.4578730410922032e-04, -6.4580033548365635e-04,
                -6.4580685154022935e-04, -6.4581010969164468e-04
        },
// x = -7.3400
        {
                -6.4736791260251790e-04, -6.4820880734752504e-04, -6.4862952761161451e-04, -6.4883997875179577e-04,
                -6.4894523466900510e-04, -6.4899787274608526e-04, -6.4902419515814290e-04, -6.4903735748885079e-04,
                -6.4904393902914098e-04, -6.4904722992427573e-04
        },
// x = -7.3350
        {
                -6.5060439381263132e-04, -6.5145373140602383e-04, -6.5187867721583240e-04, -6.5209124250360936e-04,
                -6.5219755595308530e-04, -6.5225072294917768e-04, -6.5227730987171514e-04, -6.5229060447465743e-04,
                -6.5229725215673064e-04, -6.5230057612464579e-04
        },
// x = -7.3300
        {
                -6.5385701301355842e-04, -6.5471487818228192e-04, -6.5514409196351653e-04, -6.5535879263250508e-04,
                -6.5546617423798152e-04, -6.5551987546726251e-04, -6.5554672955813809e-04, -6.5556015776250100e-04,
                -6.5556687225103594e-04, -6.5557022962409921e-04
        },
// x = -7.3250
        {
                -6.5712585024555901e-04, -6.5799232856619046e-04, -6.5842585317031339e-04, -6.5864271066732151e-04,
                -6.5875117115923192e-04, -6.5880541198926296e-04, -6.5883253593303909e-04, -6.5884609908136457e-04,
                -6.5885288104771907e-04, -6.5885627216163848e-04
        },
// x = -7.3200
        {
                -6.6041098594156477e-04, -6.6128616384882326e-04, -6.6172404255732174e-04, -6.6194307854448723e-04,
                -6.6205262876103694e-04, -6.6210741461330050e-04, -6.6213481112151228e-04, -6.6214851056983206e-04,
                -6.6215536069211039e-04, -6.6215878588596745e-04
        },
// x = -7.3150
        {
                -6.6371250092906191e-04, -6.6459646572440440e-04, -6.6503874225310193e-04, -6.6525997861005440e-04,
                -6.6537062949830232e-04, -6.6542596584874395e-04, -6.6545363766016923e-04, -6.6546747477814138e-04,
                -6.6547439374126181e-04, -6.6547785335754558e-04
        },
// x = -7.3100
        {
                -6.6703047643198283e-04, -6.6792331629228518e-04, -6.6837003479569795e-04, -6.6859349362174102e-04,
                -6.6870525623869228e-04, -6.6876114861826692e-04, -6.6878909849920113e-04, -6.6880307467024572e-04,
                -6.6881006316600936e-04, -6.6881355755065059e-04
        },
// x = -7.3050
        {
                -6.7036499407260690e-04, -6.7126679805893253e-04, -6.7171800313466773e-04, -6.7194370675098319e-04,
                -6.7205659226469365e-04, -6.7211304625991800e-04, -6.7214127700444988e-04, -6.7215539362588820e-04,
                -6.7216245235304794e-04, -6.7216598185545389e-04
        },
// x = -7.3000
        {
                -6.7371613587346806e-04, -6.7462699393992364e-04, -6.7508273063312271e-04, -6.7531070158499803e-04,
                -6.7542472127568804e-04, -6.7548174252919982e-04, -6.7551025695949073e-04, -6.7552451544268424e-04,
                -6.7553164510701462e-04, -6.7553521008010371e-04
        },
// x = -7.2950
        {
                -6.7708398425927346e-04, -6.7800398726195399e-04, -6.7846430106978004e-04, -6.7869456212885603e-04,
                -6.7880972739003769e-04, -6.7886732160115835e-04, -6.7889612256772410e-04, -6.7891052433821609e-04,
                -6.7891772565258383e-04, -6.7892132645282102e-04
        },
// x = -7.2900
        {
                -6.8046862205882908e-04, -6.8139786176485114e-04, -6.8186279864102168e-04, -6.8209537280756463e-04,
                -6.8221169514717892e-04, -6.8226986807248238e-04, -6.8229895845447878e-04, -6.8231350495213657e-04,
                -6.8232077863657160e-04, -6.8232441562400439e-04
        },
// x = -7.2850
        {
                -6.8387013250697517e-04, -6.8480870160360157e-04, -6.8527830796296610e-04, -6.8551321846816175e-04,
                -6.8563070950972734e-04, -6.8568946696361639e-04, -6.8571884966912595e-04, -6.8573354234828477e-04,
                -6.8574088913005290e-04, -6.8574456266834666e-04
        },
// x = -7.2800
        {
                -6.8728859924652933e-04, -6.8823659135038484e-04, -6.8871091407354888e-04, -6.8894818438181959e-04,
                -6.8906685586559303e-04, -6.8912620372087922e-04, -6.8915588168720252e-04, -6.8917072201681083e-04,
                -6.8917814263048569e-04, -6.8918185308696042e-04
        },
// x = -7.2750
        {
                -6.9072410633023995e-04, -6.9168161599661871e-04, -6.9216070243461334e-04, -6.9240035624595865e-04,
                -6.9252022003010694e-04, -6.9258016421859731e-04, -6.9261014041254604e-04, -6.9262512987631260e-04,
                -6.9263262506384974e-04, -6.9263637280951671e-04
        },
// x = -7.2700
        {
                -6.9417673822274770e-04, -6.9514386095501340e-04, -6.9562775893401306e-04, -6.9586982018637357e-04,
                -6.9599088824815684e-04, -6.9605143476124669e-04, -6.9608171217944040e-04, -6.9609685227598264e-04,
                -6.9610442278679290e-04, -6.9610820819639195e-04
        },
// x = -7.2650
        {
                -6.9764657980255483e-04, -6.9862341206163633e-04, -6.9911216988772140e-04, -6.9935666275936653e-04,
                -6.9947894719633538e-04, -6.9954010208560604e-04, -6.9957068375477150e-04, -6.9958597599776478e-04,
                -6.9959362258879007e-04, -6.9959744604082671e-04
        },
// x = -7.2600
        {
                -7.0113371636400582e-04, -7.0212035557798598e-04, -7.0261402204195621e-04, -7.0286097095389533e-04,
                -7.0298448398509661e-04, -7.0304625336292117e-04, -7.0307714234019495e-04, -7.0309258825852403e-04,
                -7.0310031169431219e-04, -7.0310417357109572e-04
        },
// x = -7.2550
        {
                -7.0463823361927513e-04, -7.0563477819307705e-04, -7.0613340257530974e-04, -7.0638283219372840e-04,
                -7.0650758616092654e-04, -7.0656997620107895e-04, -7.0660117557431231e-04, -7.0661677671222457e-04,
                -7.0662457776500683e-04, -7.0662847845268854e-04
        },
// x = -7.2500
        {
                -7.0816021770036478e-04, -7.0916676702553305e-04, -7.0967039910089299e-04, -7.0992233433961229e-04,
                -7.1004834170852087e-04, -7.1011135864679329e-04, -7.1014287153486140e-04, -7.1015862945212119e-04,
                -7.1016650890188800e-04, -7.1017044879049954e-04
        },
// x = -7.2450
        {
                -7.1169975516110974e-04, -7.1271640962569107e-04, -7.1322509966848825e-04, -7.1347956569145065e-04,
                -7.1360683905297585e-04, -7.1367048918780204e-04, -7.1370231874091548e-04, -7.1371823501295898e-04,
                -7.1372619364753944e-04, -7.1373017313103174e-04
        },
// x = -7.2400
        {
                -7.1525693297919504e-04, -7.1628379397771708e-04, -7.1679759276671335e-04, -7.1705461499049165e-04,
                -7.1718316706199047e-04, -7.1724745675507383e-04, -7.1727960615509305e-04, -7.1729568237318688e-04,
                -7.1730372098832641e-04, -7.1730774046460980e-04
        },
// x = -7.2350
        {
                -7.1883183855817859e-04, -7.1986900850172855e-04, -7.2038796732519560e-04, -7.2064757142152808e-04,
                -7.2077741504807710e-04, -7.2084235072502668e-04, -7.2087482318578043e-04, -7.2089106095718026e-04,
                -7.2089918035662048e-04, -7.2090324022760382e-04
        },
// x = -7.2300
        {
                -7.2242455972952656e-04, -7.2347214205593002e-04, -7.2399631271675752e-04, -7.2425852461510828e-04,
                -7.2438967277078622e-04, -7.2445526092175842e-04, -7.2448805968936404e-04, -7.2450446063747559e-04,
                -7.2451266163303478e-04, -7.2451676230466560e-04
        },
// x = -7.2250
        {
                -7.2603518475465459e-04, -7.2709328393875733e-04, -7.2762271875961241e-04, -7.2788756464975748e-04,
                -7.2802003043893958e-04, -7.2808627761928647e-04, -7.2811940597247520e-04, -7.2813597173701581e-04,
                -7.2814425514866989e-04, -7.2814839703097545e-04
        },
// x = -7.2200
        {
                -7.2966380232698130e-04, -7.3073252389103244e-04, -7.3126727571957062e-04, -7.3153478205420909e-04,
                -7.3166857871287600e-04, -7.3173549154379998e-04, -7.3176895279424437e-04, -7.3178568503140735e-04,
                -7.3179405168737218e-04, -7.3179823519449992e-04
        },
// x = -7.2150
        {
                -7.3331050157398919e-04, -7.3438995209812842e-04, -7.3493007431225726e-04, -7.3520026780965031e-04,
                -7.3533540870670904e-04, -7.3540299387592407e-04, -7.3543679136856834e-04, -7.3545369175118829e-04,
                -7.3546214248800207e-04, -7.3546636803826137e-04
        },
// x = -7.2100
        {
                -7.3697537205929490e-04, -7.3806565919214617e-04, -7.3861120570533961e-04, -7.3888411335197492e-04,
                -7.3902061199059314e-04, -7.3908887625299274e-04, -7.3912301336638748e-04, -7.3914008358410774e-04,
                -7.3914861924671491e-04, -7.3915288726261903e-04
        },
// x = -7.2050
        {
                -7.4065850378472991e-04, -7.4175973625409842e-04, -7.4231076152076782e-04, -7.4258641057405007e-04,
                -7.4272428059300407e-04, -7.4279323077133656e-04, -7.4282771091797590e-04, -7.4284495267741731e-04,
                -7.4285357411925260e-04, -7.4285788502756055e-04
        },
// x = -7.2000
        {
                -7.4435998719242947e-04, -7.4547227481610722e-04, -7.4602883383702333e-04, -7.4630725182799312e-04,
                -7.4644650700302919e-04, -7.4651614998857789e-04, -7.4655097661524116e-04, -7.4656839164017252e-04,
                -7.4657709972324692e-04, -7.4658145395500603e-04
        },
// x = -7.1950
        {
                -7.4807991316693130e-04, -7.4920336686361050e-04, -7.4976551519138066e-04, -7.5004672992745939e-04,
                -7.5018738417266864e-04, -7.5025772692594148e-04, -7.5029290351403719e-04, -7.5031049354554744e-04,
                -7.5031928914053383e-04, -7.5032368713112382e-04
        },
// x = -7.1900
        {
                -7.5181837303728435e-04, -7.5295310483757926e-04, -7.5352089858217930e-04, -7.5380493814994299e-04,
                -7.5394700551915060e-04, -7.5401805507057437e-04, -7.5405358513648795e-04, -7.5407135193316050e-04,
                -7.5408023591948118e-04, -7.5408467810865595e-04
        },
// x = -7.1850
        {
                -7.5557545857916615e-04, -7.5672158163674562e-04, -7.5729507747110750e-04, -7.5758197023908667e-04,
                -7.5772546492725425e-04, -7.5779722837787731e-04, -7.5783311547332249e-04, -7.5785106081141139e-04,
                -7.5786003407732501e-04, -7.5786452090925835e-04
        },
// x = -7.1800
        {
                -7.5935126201700988e-04, -7.6050889061984026e-04, -7.6108814578549537e-04, -7.6137792040700510e-04,
                -7.6152285675164775e-04, -7.6159534127384913e-04, -7.6163158898622278e-04, -7.6164971465982902e-04,
                -7.6165877810252048e-04, -7.6166331002584971e-04
        },
// x = -7.1750
        {
                -7.6314587602614140e-04, -7.6431512560784379e-04, -7.6490019792062271e-04, -7.6519288333661798e-04,
                -7.6533927581923548e-04, -7.6541248865744072e-04, -7.6544910061018183e-04, -7.6546740843143335e-04,
                -7.6547656295710331e-04, -7.6548114042497410e-04
        },
// x = -7.1700
        {
                -7.6695939373492552e-04, -7.6814038088624382e-04, -7.6873132874203331e-04, -7.6902695418399670e-04,
                -7.6917481743151838e-04, -7.6924876590292392e-04, -7.6928574575587489e-04, -7.6930423755510824e-04,
                -7.6931348407906295e-04, -7.6931810754917427e-04
        },
// x = -7.1650
        {
                -7.7079190872692211e-04, -7.7198475120730749e-04, -7.7258163358786484e-04, -7.7288022858072053e-04,
                -7.7302957736696571e-04, -7.7310426886226908e-04, -7.7314162031204194e-04, -7.7316029793798413e-04,
                -7.7316963738472755e-04, -7.7317430731937867e-04
        },
// x = -7.1600
        {
                -7.7464351504305138e-04, -7.7584833179236213e-04, -7.7645120827118800e-04, -7.7675280263624590e-04,
                -7.7690365188339819e-04, -7.7697909386753637e-04, -7.7701682064788185e-04, -7.7703568596783573e-04,
                -7.7704511927116204e-04, -7.7704983613729735e-04
        },
// x = -7.1550
        {
                -7.7851430718376975e-04, -7.7973121833408777e-04, -7.8034014908235771e-04, -7.8064477294028669e-04,
                -7.8079713772038334e-04, -7.8087333773327816e-04, -7.8091144361545906e-04, -7.8093049851549006e-04,
                -7.8094002661857642e-04, -7.8094479088783276e-04
        },
// x = -7.1500
        {
                -7.8240438011125454e-04, -7.8363350699881993e-04, -7.8424855279137537e-04, -7.8455623656520704e-04,
                -7.8471013210164341e-04, -7.8478709775895491e-04, -7.8482558655212204e-04, -7.8484483293724634e-04,
                -7.8485445679274803e-04, -7.8485926894150161e-04
        },
// x = -7.1450
        {
                -7.8631382925159899e-04, -7.8755529442886389e-04, -7.8817651665026274e-04, -7.8848729106842461e-04,
                -7.8864273273747344e-04, -7.8872047173135973e-04, -7.8875934728293395e-04, -7.8877878707730889e-04,
                -7.8878850764745349e-04, -7.8879336815686836e-04
        },
// x = -7.1400
        {
                -7.9024275049701667e-04, -7.9149667774481951e-04, -7.9212413839544827e-04, -7.9243803449482759e-04,
                -7.9259503782717357e-04, -7.9267355792705984e-04, -7.9271282412311491e-04, -7.9273245927023175e-04,
                -7.9274227752691630e-04, -7.9274718688299129e-04
        },
// x = -7.1350
        {
                -7.9419124020805633e-04, -7.9545775454791835e-04, -7.9609151625016389e-04, -7.9640856537920219e-04,

                -7.9656714606149220e-04, -7.9664645511484623e-04, -7.9668611588049835e-04, -7.9670594834337571e-04,
                -7.9671586526826321e-04, -7.9672082396188115e-04
        },
// x = -7.1300
        {
                -7.9815939521582643e-04, -7.9943862292236969e-04, -8.0007874892685409e-04, -8.0039898274867315e-04,
                -8.0055915662508113e-04, -8.0063926255819731e-04, -8.0067932185799631e-04, -8.0069935361937715e-04,
                -8.0070937020399499e-04, -8.0071437873097161e-04
        },
// x = -7.1250
        {
                -8.0214731282422932e-04, -8.0343938143771928e-04, -8.0408593562959643e-04, -8.0440938612515482e-04,
                -8.0457116919896421e-04, -8.0465208001775446e-04, -8.0469254185608081e-04, -8.0471277491863020e-04,
                -8.0472289216446927e-04, -8.0472795102560243e-04
        },
// x = -7.1200
        {
                -8.0615509081220446e-04, -8.0746012915121931e-04, -8.0811317605653551e-04, -8.0843987552781636e-04,
                -8.0860328396301648e-04, -8.0868500775381057e-04, -8.0872587617527474e-04, -8.0874631256177990e-04,
                -8.0875653148039452e-04, -8.0876164118151529e-04
        },
// x = -7.1150
        {
                -8.1018282743598475e-04, -8.1150096561020892e-04, -8.1216057040232501e-04, -8.1249055147555838e-04,
                -8.1265560159845759e-04, -8.1273814652881016e-04, -8.1277942561865709e-04, -8.1280006737222978e-04,
                -8.1281038898533902e-04, -8.1281555003736130e-04
        },
// x = -7.1100
        {
                -8.1423062143135948e-04, -8.1556199085450632e-04, -8.1622821936058688e-04, -8.1656151498950042e-04,
                -8.1672822329035557e-04, -8.1681159760986250e-04, -8.1685329149437929e-04, -8.1687414067866000e-04,
                -8.1688456601824977e-04, -8.1688977893722169e-04
        },
// x = -7.1050
        {
                -8.1829857201594889e-04, -8.1964330541881220e-04, -8.2031622412637704e-04, -8.2065286759548336e-04,
                -8.2082125073014575e-04, -8.2090546277126654e-04, -8.2094757561819594e-04, -8.2096863431755989e-04,
                -8.2097916442598596e-04, -8.2098442973314140e-04
        },
// x = -7.1000
        {
                -8.2238677889148889e-04, -8.2374501033512508e-04, -8.2442468639866841e-04, -8.2476471132658159e-04,
                -8.2493478611815857e-04, -8.2501984429705043e-04, -8.2506238031600546e-04, -8.2508365063577150e-04,
                -8.2509428656586377e-04, -8.2509960478767462e-04
        },
// x = -7.0950
        {
                -8.2649534224612554e-04, -8.2786720713516648e-04, -8.2855370838284142e-04, -8.2889714872562946e-04,
                -8.2906893216616422e-04, -8.2915484498352089e-04, -8.2919780842640692e-04, -8.2921929249304700e-04,
                -8.2923003530821471e-04, -8.2923540697644343e-04
        },
// x = -7.0900
        {
                -8.3062436275672035e-04, -8.3200999785281999e-04, -8.3270339279318943e-04, -8.3305028284775997e-04,
                -8.3322379209992614e-04, -8.3331056814182772e-04, -8.3335396330326637e-04, -8.3337566326461881e-04,
                -8.3338651403895721e-04, -8.3339193969070985e-04
        },
// x = -7.0850
        {
                -8.3477394159116444e-04, -8.3617348502658070e-04, -8.3687384285543629e-04, -8.3722421726295396e-04,
                -8.3739946966176950e-04, -8.3748711760053926e-04, -8.3753094881829830e-04, -8.3755286684378228e-04,
                -8.3756382666217940e-04, -8.3756930683995950e-04
        },
// x = -7.0800
        {
                -8.3894418041070525e-04, -8.4035777170201405e-04, -8.4106516230926433e-04, -8.4141905605860615e-04,
                -8.4159606911316214e-04, -8.4168459770823348e-04, -8.4172886936365889e-04, -8.4175100764449123e-04,
                -8.4176207760273643e-04, -8.4176761285449945e-04
        },
// x = -7.0750
        {
                -8.4313518137228054e-04, -8.4456296143423126e-04, -8.4527745541085732e-04, -8.4563490384209866e-04,
                -8.4581369523730767e-04, -8.4590311333609784e-04, -8.4594782985455291e-04, -8.4597019060396689e-04,
                -8.4598137180886092e-04, -8.4598696268806833e-04
        },
// x = -7.0700
        {
                -8.4734704713086586e-04, -8.4878915829037199e-04, -8.4951082693545267e-04, -8.4987186574339210e-04,
                -8.5005245334175163e-04, -8.5014276988054584e-04, -8.5018793573185297e-04, -8.5021052118531952e-04,
                -8.5022181475478465e-04, -8.5022746182045964e-04
        },
// x = -7.0650
        {
                -8.5157988084182991e-04, -8.5303646685210065e-04, -8.5376538217990961e-04, -8.5413004741762531e-04,
                -8.5431244926100175e-04, -8.5440367326584531e-04, -8.5444929296473186e-04, -8.5447210538018320e-04,
                -8.5448351244337513e-04, -8.5448921626015903e-04
        },
// x = -7.0600
        {
                -8.5583378616330185e-04, -8.5730499221811445e-04, -8.5804122696528655e-04, -8.5840955504773138e-04,
                -8.5859378935915958e-04, -8.5868592994675901e-04, -8.5873200805330918e-04, -8.5875504971136469e-04,
                -8.5876657140878526e-04, -8.5877233254699314e-04
        },
// x = -7.0550
        {
                -8.6010886725854834e-04, -8.6159484000666311e-04, -8.6233846763943383e-04, -8.6271049534706462e-04,
                -8.6289658053256629e-04, -8.6298964691120048e-04, -8.6303618803131022e-04, -8.6305946123550357e-04,
                -8.6307109871911522e-04, -8.6307691775479333e-04
        },
// x = -7.0500
        {
                -8.6440522879836128e-04, -8.6590611635808056e-04, -8.6665721107959690e-04, -8.6703297556204098e-04,
                -8.6722093021246124e-04, -8.6731493168290042e-04, -8.6736194046873791e-04, -8.6738544754574821e-04,
                -8.6739720197908851e-04, -8.6740307949407169e-04
        },
// x = -7.0450
        {
                -8.6872297596345593e-04, -8.7023892793732861e-04, -8.7099756469503323e-04, -8.7137710347479102e-04,
                -8.7156694636765443e-04, -8.7166189232408878e-04, -8.7170937347455957e-04, -8.7173311677444295e-04,
                -8.7174498933274142e-04, -8.7175092591471157e-04
        },
// x = -7.0400
        {
                -8.7306221444688036e-04, -8.7459338193655184e-04, -8.7535963642964209e-04, -8.7574298740582799e-04,
                -8.7593473750721088e-04, -8.7603063743818879e-04, -8.7607859569940492e-04, -8.7610257759583002e-04,
                -8.7611456946612529e-04, -8.7612056570866978e-04
        },
// x = -7.0350
        {
                -8.7742305045643363e-04, -8.7896958607764676e-04, -8.7974353476460683e-04, -8.8013073621672606e-04,
                -8.8032441268315017e-04, -8.8042127617252576e-04, -8.8046971633827954e-04, -8.8049393922876474e-04,
                -8.8050605161002320e-04, -8.8051210811269454e-04
        },
// x = -7.0300
        {
                -8.8180559071709779e-04, -8.8336764861484036e-04, -8.8414936872105042e-04, -8.8454045931281452e-04,
                -8.8473608149315782e-04, -8.8483391822104804e-04, -8.8488284513329129e-04, -8.8490731143944389e-04,
                -8.8491954554268000e-04, -8.8492566291105542e-04
        },
// x = -7.0250
        {
                -8.8620994247347632e-04, -8.8778767833728306e-04, -8.8857724786270214e-04, -8.8897226664588303e-04,
                -8.8916985408331089e-04, -8.8926867382706166e-04, -8.8931809237638979e-04, -8.8934280454414842e-04,
                -8.8935516159254510e-04, -8.8936134043828745e-04
        },
// x = -7.0200
        {
                -8.9063621349224785e-04, -8.9222978457165160e-04, -8.9302728229858112e-04, -8.9342626871690279e-04,
                -8.9362584115081729e-04, -8.9372565378598059e-04, -8.9377556891212062e-04, -8.9380052941199931e-04,
                -8.9381301064103062e-04, -8.9381925158194919e-04
        },
// x = -7.0150
        {
                -8.9508451206462680e-04, -8.9669407718476690e-04, -8.9749958268568785e-04, -8.9790257657875838e-04,
                -8.9810415394676847e-04, -8.9820496944808777e-04, -8.9825538614039205e-04, -8.9828059746772763e-04,
                -8.9829320412528188e-04, -8.9829950778539433e-04
        },
// x = -7.0100
        {
                -8.9955494700883778e-04, -9.0118066658622276e-04, -9.0199426023171371e-04, -9.0240130183899504e-04,
                -9.0260490427890587e-04, -9.0270673272131220e-04, -9.0275765601925705e-04, -9.0278312069445778e-04,
                -9.0279585404096313e-04, -9.0280222105055782e-04
        },
// x = -7.0050
        {
                -9.0404762767259787e-04, -9.0568966373102641e-04, -9.0651142669776006e-04, -9.0692255666257931e-04,
                -9.0712820451440107e-04, -9.0723105607401911e-04, -9.0728249106770780e-04, -9.0730821163650607e-04,
                -9.0732107294505537e-04, -9.0732750394075467e-04
        },
// x = -7.0000
        {
                -9.0856266393561354e-04, -9.1022118012225263e-04, -9.1105119440107224e-04, -9.1146645377467202e-04,
                -9.1167416758264948e-04, -9.1177805253781429e-04, -9.1183000436848558e-04, -9.1185598340219208e-04,
                -9.1186897395867080e-04, -9.1187546958349430e-04
        },
// x = -6.9950
        {
                -9.1310016621208427e-04, -9.1477532781370982e-04, -9.1561367621778697e-04, -9.1603310646341612e-04,
                -9.1624290697807896e-04, -9.1634783571036102e-04, -9.1640030957090368e-04, -9.1642654966666366e-04,
                -9.1643967076987856e-04, -9.1644623167330729e-04
        },
// x = -6.9900
        {
                -9.1766024545322030e-04, -9.1935221941261927e-04, -9.2019898558569227e-04, -9.2062262858273855e-04,
                -9.2083453676297113e-04, -9.2094051975821377e-04, -9.2099352089368446e-04, -9.2102002467473799e-04,
                -9.2103327763654627e-04, -9.2103990447458834e-04
        },
// x = -6.9850
        {
                -9.2224301314977088e-04, -9.2395196808230584e-04, -9.2480723650700152e-04, -9.2523513455516524e-04,
                -9.2544917157029775e-04, -9.2555621941966420e-04, -9.2560975312781129e-04, -9.2563652324375548e-04,
                -9.2564990938919683e-04, -9.2565660282445210e-04
        },
// x = -6.9800
        {
                -9.2684858133456152e-04, -9.2857468754490192e-04, -9.2943854355114039e-04, -9.2987073937464957e-04,
                -9.3008692660657066e-04, -9.3019505000760112e-04, -9.3024912163939438e-04, -9.3027616076644774e-04,
                -9.3028968143387596e-04, -9.3029644213560292e-04
        },
// x = -6.9750
        {
                -9.3147706258504478e-04, -9.3322049208406535e-04, -9.3409302185754831e-04, -9.3452955860941667e-04,
                -9.3474791765470579e-04, -9.3485712741238575e-04, -9.3491174237255120e-04, -9.3493905321382124e-04,
                -9.3495270975503823e-04, -9.3495953839922089e-04
        },
// x = -6.9700
        {
                -9.3612857002586120e-04, -9.3788949654770727e-04, -9.3877078713849199e-04, -9.3921170840481887e-04,
                -9.3943226107690166e-04, -9.3954256810474068e-04, -9.3959773185230014e-04, -9.3962531713805337e-04,
                -9.3963911091844471e-04, -9.3964600818785950e-04
        },
// x = -6.9650
        {
                -9.4080321733141050e-04, -9.4258181635073588e-04, -9.4347195568189406e-04, -9.4391730548620855e-04,
                -9.4414007381753303e-04, -9.4425148913865439e-04, -9.4430720718747022e-04, -9.4433506967540534e-04,
                -9.4434900207407619e-04, -9.4435596865836123e-04
        },
// x = -6.9600
        {
                -9.4550111872843633e-04, -9.4729756747781121e-04, -9.4819664435417570e-04, -9.4864646716182185e-04,
                -9.4887147340605655e-04, -9.4898400815429814e-04, -9.4904028607362492e-04, -9.4906842854914839e-04,
                -9.4908250095906152e-04, -9.4908953755478451e-04
        },
// x = -6.9550
        {
                -9.5022238899861926e-04, -9.5203686648611264e-04, -9.5294497060311042e-04, -9.5339931132568036e-04,
                -9.5362657795993366e-04, -9.5374024338095928e-04, -9.5379708679600012e-04, -9.5382551207250402e-04,
                -9.5383972590061897e-04, -9.5384683321134795e-04
        },
// x = -6.9500
        {
                -9.5496714348118313e-04, -9.5679983050812162e-04, -9.5771705246069558e-04, -9.5817595646050283e-04,
                -9.5840550618756675e-04, -9.5852031363998890e-04, -9.5857772823245693e-04, -9.5860643915160015e-04,
                -9.5862079581901429e-04, -9.5862797455538722e-04
        },
// x = -6.9450
        {
                -9.5973549807551299e-04, -9.6158657725441447e-04, -9.6251300854603478e-04, -9.6297652164063557e-04,
                -9.6320837739124945e-04, -9.6332433834776332e-04, -9.6338232985644949e-04, -9.6341132928844282e-04,
                -9.6342583023053283e-04, -9.6343308111032817e-04
        },
// x = -6.9400
        {
                -9.6452756924378185e-04, -9.6639722501647000e-04, -9.6733295806823499e-04, -9.6780112653499322e-04,
                -9.6803531147013192e-04, -9.6815243751866150e-04, -9.6821101174000791e-04, -9.6824030258389900e-04,
                -9.6825494925046468e-04, -9.6826227299867456e-04
        },
// x = -6.9350
        {
                -9.6934347401359151e-04, -9.7123189266949130e-04, -9.7217702082931989e-04, -9.7264989141001776e-04,
                -9.7288642892320182e-04, -9.7300473176805617e-04, -9.7306389455673493e-04, -9.7309347974069928e-04,
                -9.7310827359610864e-04, -9.7311567094501025e-04
        },
// x = -6.9300
        {
                -9.7418332998062442e-04, -9.7609069967523787e-04, -9.7704531722715335e-04, -9.7752293713264924e-04,
                -9.7776185085227943e-04, -9.7788134231532138e-04, -9.7794109958481952e-04, -9.7797098206645300e-04,
                -9.7798592458978755e-04, -9.7799339627901668e-04
        },
// x = -6.9250
        {
                -9.7904725531130490e-04, -9.8097376608487327e-04, -9.8193796825838103e-04, -9.8242038517331286e-04,
                -9.8266169896502617e-04, -9.8278239098685289e-04, -9.8284274871006275e-04, -9.8287293147667691e-04,
                -9.8288802416188138e-04, -9.8289557093850623e-04
        },
// x = -6.9200
        {
                -9.8393536874547471e-04, -9.8588121254182602e-04, -9.8685509552138293e-04, -9.8734235760892055e-04,
                -9.8758609557797190e-04, -9.8770800021910632e-04, -9.8776896442892320e-04, -9.8779945049784360e-04,
                -9.8781469485387328e-04, -9.8782231747246950e-04
        },
// x = -6.9150
        {
                -9.8884778959907813e-04, -9.9081316028466326e-04, -9.9179682121924395e-04, -9.9228897712588587e-04,
                -9.9253516361955184e-04, -9.9265829306164764e-04, -9.9271986985157161e-04, -9.9275066227043959e-04,
                -9.9276605982141276e-04, -9.9277375904413827e-04
        },
// x = -6.9100
        {
                -9.9378463776685945e-04, -9.9576973114997658e-04, -9.9676326816273476e-04, -9.9726036702315669e-04,
                -9.9750902663316523e-04, -9.9763339318022248e-04, -9.9769558870496699e-04, -9.9772669055204436e-04,
                -9.9774224283739324e-04, -9.9775001943406497e-04
        },
// x = -6.9050
        {
                -9.9874603372507179e-04, -1.0007510475752849e-03, -1.0017545597733116e-03, -1.0022566512152592e-03,
                -1.0025078087802419e-03, -1.0026334248598354e-03, -1.0026962453359425e-03, -1.0027276597204188e-03,
                -1.0027433682950448e-03, -1.0027512230432153e-03
        },
// x = -6.9000
        {
                -1.0037320985341995e-03, -1.0057572326019478e-03, -1.0067708200861246e-03, -1.0072779542353609e-03,
                -1.0075316348433316e-03, -1.0076585130078515e-03, -1.0077219647143128e-03, -1.0077536947766166e-03,
                -1.0077695612110439e-03, -1.0077774948960795e-03
        },
// x = -6.8950
        {
                -1.0087429538416868e-03, -1.0107884098780931e-03, -1.0118121737530477e-03, -1.0123244012383446e-03,
                -1.0125806302292010e-03, -1.0127087831571071e-03, -1.0127728724359899e-03, -1.0128049213481011e-03,
                -1.0128209472286362e-03, -1.0128289606437949e-03
        },
// x = -6.8900
        {
                -1.0137787218846858e-03, -1.0158447036615590e-03, -1.0168787460457188e-03, -1.0173961180039012e-03,
                -1.0176549209719528e-03, -1.0177843614690382e-03, -1.0178490947261214e-03, -1.0178814656918900e-03,
                -1.0178976526207784e-03, -1.0179057465672897e-03
        },
// x = -6.8850
        {
                -1.0188395254928111e-03, -1.0209262388228501e-03, -1.0219706628585941e-03, -1.0224932309396354e-03,
                -1.0227546337361567e-03, -1.0228853747368284e-03, -1.0229507584422395e-03, -1.0229834546977051e-03,
                -1.0229998042932916e-03, -1.0230079795804361e-03
        },
// x = -6.8800
        {
                -1.0239254880909060e-03, -1.0260331408481055e-03, -1.0270880507120210e-03, -1.0276158670841880e-03,
                -1.0278798958199967e-03, -1.0280119503885655e-03, -1.0280779910774287e-03, -1.0281110158911438e-03,
                -1.0281275297880344e-03, -1.0281357872332256e-03
        },
// x = -6.8750
        {
                -1.0290367337018249e-03, -1.0311655358420822e-03, -1.0322310367553232e-03, -1.0327641541103719e-03,
                -1.0330308351584340e-03, -1.0331642164904187e-03, -1.0332309207635084e-03, -1.0332642774368635e-03,
                -1.0332809572860893e-03, -1.0332892977149546e-03
        },
// x = -6.8700
        {
                -1.0341733869492218e-03, -1.0363235505311526e-03, -1.0373997487699005e-03, -1.0379382203283263e-03,
                -1.0382075803263861e-03, -1.0383423017498315e-03, -1.0384096762742284e-03, -1.0384433681417855e-03,
                -1.0384602156109674e-03, -1.0384686398574306e-03
        },
// x = -6.8650
        {
                -1.0393355730603567e-03, -1.0415073122663151e-03, -1.0425943151723450e-03, -1.0431381946886837e-03,
                -1.0434102605419203e-03, -1.0435463355187278e-03, -1.0436143870284878e-03, -1.0436484174583104e-03,
                -1.0436654342318255e-03, -1.0436739431381899e-03
        },
// x = -6.8600
        {
                -1.0445234178689086e-03, -1.0467169490262196e-03, -1.0478148650175712e-03, -1.0483642067857539e-03,
                -1.0486390056694663e-03, -1.0487764477967363e-03, -1.0488451830935630e-03, -1.0488795554875539e-03,
                -1.0488967432667037e-03, -1.0489053376837372e-03
        },
// x = -6.8550
        {
                -1.0497370478178049e-03, -1.0519525894202059e-03, -1.0530615280019639e-03, -1.0536163868607239e-03,
                -1.0538939462230400e-03, -1.0540327692344312e-03, -1.0541021951883534e-03, -1.0541369129825945e-03,
                -1.0541542734857762e-03, -1.0541629542727957e-03
        },
// x = -6.8500
        {
                -1.0549765899620597e-03, -1.0572143626913575e-03, -1.0583344344665352e-03, -1.0588948658048730e-03,
                -1.0591752133694878e-03, -1.0593154311365868e-03, -1.0593855546866466e-03, -1.0594206213517417e-03,
                -1.0594381563146184e-03, -1.0594469243395764e-03
        },
// x = -6.8450
        {
                -1.0602421719716263e-03, -1.0625023987195692e-03, -1.0636337154001049e-03, -1.0641997751628031e-03,
                -1.0644829389317436e-03, -1.0646245654654508e-03, -1.0646953936203954e-03, -1.0647308126618181e-03,
                -1.0647485238374933e-03, -1.0647573799770636e-03
        },
// x = -6.8400
        {
                -1.0655339221342594e-03, -1.0678168280246294e-03, -1.0689595024424914e-03, -1.0695312471356872e-03,
                -1.0698172553921057e-03, -1.0699603048440326e-03, -1.0700318446830153e-03, -1.0700676196414584e-03,
                -1.0700855088006506e-03, -1.0700944539403165e-03
        },
// x = -6.8350
        {
                -1.0708519693583940e-03, -1.0731577817693146e-03, -1.0743119278877164e-03, -1.0748894145845313e-03,
                -1.0751782958955251e-03, -1.0753227825594082e-03, -1.0753950412326948e-03, -1.0754311756844246e-03,
                -1.0754492446156444e-03, -1.0754582796497875e-03
        },
// x = -6.8300
        {
                -1.0761964431760310e-03, -1.0785253917625025e-03, -1.0796911246872313e-03, -1.0802744110334534e-03,
                -1.0805661942529157e-03, -1.0807121325660421e-03, -1.0807851172957260e-03, -1.0808216148529386e-03,
                -1.0808398653626681e-03, -1.0808489911946575e-03
        },
// x = -6.8250
        {
                -1.0815674737456383e-03, -1.0839197904622950e-03, -1.0850972264531511e-03, -1.0856863706729791e-03,
                -1.0859810849444750e-03, -1.0861284894891252e-03, -1.0862022075698470e-03, -1.0862390718810310e-03,
                -1.0862575057939039e-03, -1.0862667233361854e-03
        },
// x = -6.8200
        {
                -1.0869651918550647e-03, -1.0893411109791588e-03, -1.0905303674615145e-03, -1.0911254283633533e-03,
                -1.0914231031230279e-03, -1.0915719886279297e-03, -1.0916464474276075e-03, -1.0916836821779075e-03,
                -1.0917023013368907e-03, -1.0917116115110800e-03
        },
// x = -6.8150
        {
                -1.0923897289244627e-03, -1.0947894870790786e-03, -1.0959906826555467e-03, -1.0965917196378680e-03,
                -1.0968923846173797e-03, -1.0970427659591792e-03, -1.0971179729197444e-03, -1.0971555818313284e-03,
                -1.0971743880979078e-03, -1.0971837918348816e-03
        },
// x = -6.8100
        {
                -1.0978412170092296e-03, -1.1002650531867285e-03, -1.1014783076489499e-03, -1.1020853807062053e-03,
                -1.1023890659356936e-03, -1.1025409581404404e-03, -1.1026169207785795e-03, -1.1026549076110126e-03,
                -1.1026739028653777e-03, -1.1026834011053667e-03
        },
// x = -6.8050
        {
                -1.1033197888029535e-03, -1.1057679443886520e-03, -1.1069933787292022e-03, -1.1076065484577999e-03,
                -1.1079132842688778e-03, -1.1080667025135246e-03, -1.1081434284214318e-03, -1.1081817969720510e-03,
                -1.1082009831132836e-03, -1.1082105768059664e-03
        },
// x = -6.8000
        {
                -1.1088255776403786e-03, -1.1112982964364636e-03, -1.1125360328608773e-03, -1.1131553604652139e-03,
                -1.1134651774939949e-03, -1.1136201371079125e-03, -1.1136976339540471e-03, -1.1137363880583404e-03,
                -1.1137557670046038e-03, -1.1137654571092021e-03
        },
// x = -6.7950
        {
                -1.1143587175003779e-03, -1.1168562457500592e-03, -1.1181064076889746e-03, -1.1187319549875328e-03,
                -1.1190448841776849e-03, -1.1192014006441923e-03, -1.1192796761740440e-03, -1.1193188197060340e-03,
                -1.1193383933947661e-03, -1.1193481808801403e-03
        },
// x = -6.7900
        {
                -1.1199193430089431e-03, -1.1224419294208472e-03, -1.1237046415422708e-03, -1.1243364709737757e-03,
                -1.1246525435796065e-03, -1.1248106325375158e-03, -1.1248896945743802e-03, -1.1249292314470109e-03,
                -1.1249490018351161e-03, -1.1249588876798629e-03
        },
// x = -6.7850
        {
                -1.1255075894421828e-03, -1.1280554852149886e-03, -1.1293308734366860e-03, -1.1299690480663207e-03,
                -1.1302882956558962e-03, -1.1304479729010749e-03, -1.1305278293468326e-03, -1.1305677635123604e-03,
                -1.1305877325764060e-03, -1.1305977177689555e-03
        },
// x = -6.7800
        {
                -1.1311235927293350e-03, -1.1336970515766574e-03, -1.1349852430786636e-03, -1.1356298266043514e-03,
                -1.1359522810626417e-03, -1.1361135625495887e-03, -1.1361942213854951e-03, -1.1362345568358854e-03,
                -1.1362547265722990e-03, -1.1362648121110137e-03
        },
// x = -6.7750
        {
                -1.1367674894557926e-03, -1.1393667676313126e-03, -1.1406678908685711e-03, -1.1413189476273168e-03,
                -1.1416446411593771e-03, -1.1418075430028138e-03, -1.1418890122902984e-03, -1.1419297530576213e-03,
                -1.1419501254828909e-03, -1.1419603123761653e-03
        },
// x = -6.7700
        {
                -1.1424394168661448e-03, -1.1450647731889869e-03, -1.1463789579041121e-03, -1.1470365528784097e-03,
                -1.1473655180125910e-03, -1.1475300564890718e-03, -1.1476123443705409e-03, -1.1476534945273761e-03,
                -1.1476740716782510e-03, -1.1476843609446127e-03
        },
// x = -6.7650
        {
                -1.1481395128672205e-03, -1.1507912087475901e-03, -1.1521185859837595e-03, -1.1527827848080610e-03,
                -1.1531150543992548e-03, -1.1532812459487912e-03, -1.1533643606484404e-03, -1.1534059243082848e-03,
                -1.1534267082419787e-03, -1.1534371009101908e-03
        },
// x = -6.7600
        {
                -1.1538679160311592e-03, -1.1565462154962288e-03, -1.1578869176102013e-03, -1.1585577865774546e-03,
                -1.1588933938103667e-03, -1.1590612550380671e-03, -1.1591452048627063e-03, -1.1591871861803829e-03,
                -1.1592081789747809e-03, -1.1592186760839430e-03
        },
// x = -6.7550
        {
                -1.1596247655984832e-03, -1.1623299353185387e-03, -1.1636840959938074e-03, -1.1643617020620539e-03,
                -1.1647006804545166e-03, -1.1648702281322448e-03, -1.1649550214721245e-03, -1.1649974246441988e-03,
                -1.1650186283980637e-03, -1.1650292309977174e-03
        },
// x = -6.7500
        {
                -1.1654102014811876e-03, -1.1681425107960366e-03, -1.1695102650561080e-03, -1.1701946758551525e-03,
                -1.1705370592614633e-03, -1.1707083103295120e-03, -1.1707939556591621e-03, -1.1708367849243616e-03,
                -1.1708582017575448e-03, -1.1708689109077783e-03
        },
// x = -6.7450
        {
                -1.1712243642658441e-03, -1.1739840852114847e-03, -1.1753655694332940e-03, -1.1760568532714353e-03,
                -1.1764026758857348e-03, -1.1765756474545164e-03, -1.1766621533335936e-03, -1.1767054129732304e-03,
                -1.1767270450268829e-03, -1.1767378617984373e-03
        },
// x = -6.7400
        {
                -1.1770673952167124e-03, -1.1798548025522694e-03, -1.1812501544797276e-03, -1.1819483803505626e-03,
                -1.1822976767102437e-03, -1.1824723860619974e-03, -1.1825597611361391e-03, -1.1826034554745387e-03,
                -1.1826253049113254e-03, -1.1826362303857000e-03
        },
// x = -6.7350
        {
                -1.1829394362788720e-03, -1.1857548075138009e-03, -1.1871641662714800e-03, -1.1878694038607695e-03,
                -1.1882222088499201e-03, -1.1883986734404384e-03, -1.1884869264421260e-03, -1.1885310598470586e-03,
                -1.1885531288513744e-03, -1.1885641641209371e-03
        },
// x = -6.7300
        {
                -1.1888406300813608e-03, -1.1916842455029226e-03, -1.1931077516098741e-03, -1.1938200713024830e-03,
                -1.1941764201553651e-03, -1.1943546576157341e-03, -1.1944437973651655e-03, -1.1944883742482826e-03,
                -1.1945106650264711e-03, -1.1945218111945664e-03
        },
// x = -6.7250
        {
                -1.1947711199403297e-03, -1.1976432626413386e-03, -1.1990810580250531e-03, -1.1998005309119574e-03,
                -1.2001604592165188e-03, -1.2003404873548789e-03, -1.2004305227608502e-03, -1.2004755475781251e-03,
                -1.2004980623586986e-03, -1.2005093205397575e-03
        },
// x = -6.7200
        {
                -1.2007310498622112e-03, -1.2036320057690606e-03, -1.2050842337795644e-03, -1.2058109316649267e-03,
                -1.2061744753663508e-03, -1.2063563121696738e-03, -1.2064472522304677e-03, -1.2064927294826412e-03,
                -1.2065154705165038e-03, -1.2065268418361560e-03
        },
// x = -6.7150
        {
                -1.2067205645468967e-03, -1.2096506224478622e-03, -1.2111174278719573e-03, -1.2118514232802770e-03,
                -1.2122186186845660e-03, -1.2124022823204498e-03, -1.2124941361247336e-03, -1.2125400703577643e-03,
                -1.2125630399184362e-03, -1.2125745255136228e-03
        },
// x = -6.7100
        {
                -1.2127398093909330e-03, -1.2156992609647580e-03, -1.2171807900404014e-03, -1.2179221562237336e-03,
                -1.2182930400013272e-03, -1.2184785488198104e-03, -1.2185713255475463e-03, -1.2186177213530628e-03,
                -1.2186409217369082e-03, -1.2186525227559943e-03
        },
// x = -6.7050
        {
                -1.2187889304907269e-03, -1.2217780703354941e-03, -1.2232744707663225e-03, -1.2240232817115706e-03,
                -1.2243978909010037e-03, -1.2245852634363947e-03, -1.2246789723597536e-03, -1.2247258343755149e-03,
                -1.2247492679019715e-03, -1.2247609855048625e-03
        },
// x = -6.7000
        {
                -1.2248680746457664e-03, -1.2278872003080533e-03, -1.2293986212780538e-03, -1.2301549517143334e-03,
                -1.2305333237259278e-03, -1.2307225786986555e-03, -1.2308172291829456e-03, -1.2308645620933043e-03,
                -1.2308882311051147e-03, -1.2309000664633702e-03
        },
// x = -6.6950
        {
                -1.2309773893618521e-03, -1.2340268013661827e-03, -1.2355533935545064e-03, -1.2363173189605856e-03,
                -1.2366994915801791e-03, -1.2368906478986618e-03, -1.2369862494032622e-03, -1.2370340579396322e-03,
                -1.2370579648030781e-03, -1.2370699191000308e-03
        },
// x = -6.6900
        {
                -1.2371170228543467e-03, -1.2401970247329283e-03, -1.2417389403288576e-03, -1.2425105369406682e-03,
                -1.2428965483333827e-03, -1.2430896250959149e-03, -1.2431861871752203e-03, -1.2432344761165489e-03,
                -1.2432586232216879e-03, -1.2432706976525610e-03
        },
// x = -6.6850
        {
                -1.2432871240514313e-03, -1.2463980223741957e-03, -1.2479554150922538e-03, -1.2487347599104820e-03,
                -1.2491246486245279e-03, -1.2493196651211845e-03, -1.2494171974255603e-03, -1.2494659715988072e-03,
                -1.2494903613597113e-03, -1.2495025571317390e-03
        },
// x = -6.6800
        {
                -1.2494878425973819e-03, -1.2526299470023191e-03, -1.2542029720975377e-03, -1.2549901428952864e-03,
                -1.2553839478658044e-03, -1.2555809235803689e-03, -1.2556794358571139e-03, -1.2557287001377319e-03,
                -1.2557533349927289e-03, -1.2557656533252752e-03
        },
// x = -6.6750
        {
                -1.2557193288558540e-03, -1.2588929520796523e-03, -1.2604817663629860e-03, -1.2612768416935174e-03,
                -1.2616746022464619e-03, -1.2618735568583669e-03, -1.2619730589526874e-03, -1.2620228182651104e-03,
                -1.2620477006770278e-03, -1.2620601428017101e-03
        },
// x = -6.6700
        {
                -1.2619817339131825e-03, -1.2651871918221711e-03, -1.2667919536760693e-03, -1.2675950128806215e-03,
                -1.2679967687366802e-03, -1.2681977221229747e-03, -1.2682982239789668e-03, -1.2683484832971014e-03,
                -1.2683736157535128e-03, -1.2683861829143241e-03
        },
// x = -6.6650
        {
                -1.2682752095816974e-03, -1.2715128212030975e-03, -1.2731336905972312e-03, -1.2739448138129170e-03,
                -1.2743506050914681e-03, -1.2745535773287992e-03, -1.2746550889904435e-03, -1.2747058533381665e-03,
                -1.2747312383516384e-03, -1.2747439318050724e-03
        },
// x = -6.6600
        {
                -1.2745999084030504e-03, -1.2778699959565373e-03, -1.2795071344636809e-03, -1.2803264026314614e-03,
                -1.2807362698545775e-03, -1.2809412812211942e-03, -1.2810438128333568e-03, -1.2810950872850159e-03,
                -1.2811207273933619e-03, -1.2811335484085376e-03
        },
// x = -6.6550
        {
                -1.2809559836515539e-03, -1.2842588725811342e-03, -1.2859124433932079e-03, -1.2867399382659505e-03,
                -1.2871539223624341e-03, -1.2873609933402111e-03, -1.2874645551496577e-03, -1.2875163448305799e-03,
                -1.2875422425971108e-03, -1.2875551924559003e-03
        },
// x = -6.6500
        {
                -1.2873435893375378e-03, -1.2906796083437431e-03, -1.2923497762880147e-03, -1.2931855804386274e-03,
                -1.2936037227480918e-03, -1.2938128740245729e-03, -1.2939174763809923e-03, -1.2939697864679950e-03,
                -1.2939959444817780e-03, -1.2940090244789343e-03
        },
// x = -6.6450
        {
                -1.2937628802107161e-03, -1.2971323612831157e-03, -1.2988192928385658e-03, -1.2996634896682150e-03,
                -1.3000858319452062e-03, -1.3002970844156682e-03, -1.3004027377727053e-03, -1.3004555734946141e-03,
                -1.3004819943707297e-03, -1.3004952058140156e-03
        },
// x = -6.6400
        {
                -1.3002140117635698e-03, -1.3036172902136098e-03, -1.3053211535274563e-03, -1.3061738272738676e-03,
                -1.3066004116920246e-03, -1.3068137864615645e-03, -1.3069205013778637e-03, -1.3069738680160346e-03,
                -1.3070005543958387e-03, -1.3070138986061585e-03
        },
// x = -6.6350
        {
                -1.3066971402347423e-03, -1.3101345547289092e-03, -1.3118555196333015e-03, -1.3127167553791418e-03,
                -1.3131476245354012e-03, -1.3133631429210397e-03, -1.3134709300613003e-03, -1.3135248329501467e-03,
                -1.3135517875015347e-03, -1.3135652658130652e-03
        },
// x = -6.6300
        {
                -1.3132124226124468e-03, -1.3166843152057619e-03, -1.3184225532346398e-03, -1.3192924369159842e-03,
                -1.3197276338348257e-03, -1.3199453173676355e-03, -1.3200541875036760e-03, -1.3201086320312025e-03,
                -1.3201358574488750e-03, -1.3201494712091995e-03
        },
// x = -6.6250
        {
                -1.3197600166378926e-03, -1.3232667328077396e-03, -1.3250224172138593e-03, -1.3259010356287419e-03,
                -1.3263406037664776e-03, -1.3265604741937317e-03, -1.3266704382055652e-03, -1.3267254298139056e-03,
                -1.3267529288196389e-03, -1.3267666793898823e-03
        },
// x = -6.6200
        {
                -1.3263400808087205e-03, -1.3298819694890097e-03, -1.3316552752611404e-03, -1.3325427160781932e-03,
                -1.3329866993272974e-03, -1.3332087786146385e-03, -1.3333198474915606e-03, -1.3333753916775209e-03,
                -1.3334031670204384e-03, -1.3334170557754022e-03
        },
// x = -6.6150
        {
                -1.3329527743824526e-03, -1.3365301879981247e-03, -1.3383212918784200e-03, -1.3392176436455936e-03,
                -1.3396660863390791e-03, -1.3398903966727121e-03, -1.3400025815143962e-03, -1.3400586838300053e-03,
                -1.3400867382868515e-03, -1.3401007666151519e-03
        },
// x = -6.6100
        {
                -1.3395982573799600e-03, -1.3432115518818332e-03, -1.3450206323833702e-03, -1.3459259845367466e-03,

                -1.3463789314525822e-03, -1.3466054952414879e-03, -1.3467188072590949e-03, -1.3467754733121570e-03,
                -1.3468038096875767e-03, -1.3468179789917836e-03
        },
// x = -6.6050
        {
                -1.3462766905889384e-03, -1.3499262254889027e-03, -1.3517534629134014e-03, -1.3526679057860919e-03,
                -1.3531254021516659e-03, -1.3533542420298360e-03, -1.3534686925471327e-03, -1.3535259280017904e-03,
                -1.3535545491286075e-03, -1.3535688608253835e-03
        },
// x = -6.6000
        {
                -1.3529882355674038e-03, -1.3566743739739644e-03, -1.3585199504296797e-03, -1.3594435752608137e-03,
                -1.3599056667574409e-03, -1.3601368055861378e-03, -1.3602524060406281e-03, -1.3603102166179258e-03,
                -1.3603391253574270e-03, -1.3603535808776707e-03
        },
// x = -6.5950
        {
                -1.3597330546471968e-03, -1.3634561633013712e-03, -1.3653202627211683e-03, -1.3662531616649698e-03,
                -1.3667198944324452e-03, -1.3669533553024808e-03, -1.3670701172465488e-03, -1.3671285087250057e-03,
                -1.3671577079672264e-03, -1.3671723087562144e-03
        },
// x = -6.5900
        {
                -1.3665113109375051e-03, -1.3702717602490793e-03, -1.3721545684086840e-03, -1.3730968345436411e-03,
                -1.3735682551848372e-03, -1.3738040614188795e-03, -1.3739219965209412e-03, -1.3739809747371271e-03,
                -1.3740104674011407e-03, -1.3740252149186717e-03
        },
// x = -6.5850
        {
                -1.3733231683283977e-03, -1.3771213324125391e-03, -1.3790230369489754e-03, -1.3799747642871004e-03,
                -1.3804509198726113e-03, -1.3806890950275098e-03, -1.3808082150731796e-03, -1.3808677859222993e-03,
                -1.3808975749565074e-03, -1.3809124706770492e-03
        },
// x = -6.5800
        {
                -1.3801687914943759e-03, -1.3840050482086133e-03, -1.3859258386388199e-03, -1.3868871221350023e-03,
                -1.3873680602078369e-03, -1.3876086280769724e-03, -1.3877289449702387e-03, -1.3877891144067212e-03,
                -1.3878192027891500e-03, -1.3878342482019862e-03
        },
// x = -6.5750
        {
                -1.3870483458979324e-03, -1.3909230768795037e-03, -1.3928631446191422e-03, -1.3938340801805940e-03,
                -1.3943198487609120e-03, -1.3945628333765720e-03, -1.3946843591409867e-03, -1.3947451331790784e-03,
                -1.3947755239176753e-03, -1.3947907205270546e-03
        },
// x = -6.5700
        {
                -1.3939619977931341e-03, -1.3978755884967034e-03, -1.3998351268791479e-03, -1.4008158113749476e-03,
                -1.4013064589648448e-03, -1.4015518846006202e-03, -1.4016746313804990e-03, -1.4017360160948661e-03,
                -1.4017667122278000e-03, -1.4017820615530858e-03
        },
// x = -6.5650
        {
                -1.4009099142292107e-03, -1.4048627539649614e-03, -1.4068419582604834e-03, -1.4078324895312095e-03,
                -1.4083280651195523e-03, -1.4085759562927590e-03, -1.4086999363543938e-03, -1.4087619378807279e-03,
                -1.4087929424766944e-03, -1.4088084460525175e-03
        },
// x = -6.5600
        {
                -1.4078922630541603e-03, -1.4118847450262676e-03, -1.4138838124614084e-03, -1.4148842893288731e-03,
                -1.4153848423961799e-03, -1.4156352238703059e-03, -1.4157604496031889e-03, -1.4158230741388212e-03,
                -1.4158543902973479e-03, -1.4158700496737593e-03
        },
// x = -6.5550
        {
                -1.4149092129183722e-03, -1.4189417342638589e-03, -1.4209608640409954e-03, -1.4219713863180751e-03,
                -1.4224769668414470e-03, -1.4227298636286219e-03, -1.4228563475466843e-03, -1.4229196013512042e-03,
                -1.4229512322029595e-03, -1.4229670489455857e-03
        },
// x = -6.5500
        {
                -1.4219609332782601e-03, -1.4260338951062350e-03, -1.4280732884233436e-03, -1.4290939569239084e-03,
                -1.4296046153820085e-03, -1.4298600527454990e-03, -1.4299878074883591e-03, -1.4300516968842384e-03,
                -1.4300836455913471e-03, -1.4300996212815464e-03
        },
// x = -6.5450
        {
                -1.4290475943999111e-03, -1.4331614018312012e-03, -1.4352212619018164e-03, -1.4362521784507557e-03,
                -1.4367679658288410e-03, -1.4370259692855717e-03, -1.4371550076197972e-03, -1.4372195389930230e-03,
                -1.4372518087493807e-03, -1.4372679449844009e-03
        },
// x = -6.5400
        {
                -1.4361693673627511e-03, -1.4403244295699265e-03, -1.4424049616432995e-03, -1.4434462290866515e-03,
                -1.4439671968816507e-03, -1.4442277922047499e-03, -1.4443581270251324e-03, -1.4444233068258451e-03,
                -1.4444559008574365e-03, -1.4444721992505759e-03
        },
// x = -6.5350
        {
                -1.4433264240632192e-03, -1.4475231543110152e-03, -1.4496245656924740e-03, -1.4506762879076549e-03,
                -1.4512024881333020e-03, -1.4514657013546721e-03, -1.4515973456855152e-03, -1.4516631804286518e-03,
                -1.4516961019938747e-03, -1.4517125641746414e-03
        },
// x = -6.5300
        {
                -1.4505189372184650e-03, -1.4547577529046062e-03, -1.4568802529761179e-03, -1.4579425348822536e-03,
                -1.4584740200742690e-03, -1.4587398774871837e-03, -1.4588728444836021e-03, -1.4589393407495484e-03,
                -1.4589725931395381e-03, -1.4589892207538151e-03
        },
// x = -6.5250
        {
                -1.4577470803700514e-03, -1.4620284030664830e-03, -1.4641722033074214e-03, -1.4652451508757817e-03,
                -1.4657819740971081e-03, -1.4660505022588365e-03, -1.4661848052080705e-03, -1.4662519696433166e-03,
                -1.4662855561822763e-03, -1.4663023508924835e-03
        },
// x = -6.5200
        {
                -1.4650110278876789e-03, -1.4693352833822064e-03, -1.4715005973903264e-03, -1.4725843176548660e-03,
                -1.4731265325009547e-03, -1.4733977582354085e-03, -1.4735334105581504e-03, -1.4736012498759567e-03,
                -1.4736351739214877e-03, -1.4736521374067500e-03
        },
// x = -6.5150
        {
                -1.4723109549729209e-03, -1.4766785733112617e-03, -1.4788656168238858e-03, -1.4799602178918883e-03,
                -1.4805078784960388e-03, -1.4807818288964498e-03, -1.4809188441481836e-03, -1.4809873651292496e-03,
                -1.4810216300726884e-03, -1.4810387640290031e-03
        },
// x = -6.5100
        {
                -1.4796470376629734e-03, -1.4840584531912321e-03, -1.4862674441066439e-03, -1.4873730351694723e-03,
                -1.4879261962082262e-03, -1.4882028986398486e-03, -1.4883412905122033e-03, -1.4884105000053461e-03,
                -1.4884451092721026e-03, -1.4884624154125083e-03
        },
// x = -6.5050
        {
                -1.4870194528344236e-03, -1.4914751042419826e-03, -1.4937062626410359e-03, -1.4948229539849928e-03,
                -1.4953816706835812e-03, -1.4956611527864214e-03, -1.4958009351085377e-03, -1.4958708400313759e-03,
                -1.4959057970812758e-03, -1.4959232771360241e-03
        },
// x = -6.5000
        {
                -1.4944283782070274e-03, -1.4989287085698671e-03, -1.5011822567378118e-03, -1.5023101597551052e-03,
                -1.5028744878929513e-03, -1.5031567775845250e-03, -1.5032979643244372e-03, -1.5033685716640810e-03,
                -1.5034038799917112e-03, -1.5034215357084403e-03
        },
// x = -6.4950
        {
                -1.5018739923475034e-03, -1.5064194491719546e-03, -1.5086956116204774e-03, -1.5098348388202989e-03,
                -1.5104048347365747e-03, -1.5106899602146918e-03, -1.5108325654807203e-03, -1.5109038822944713e-03,
                -1.5109395454295292e-03, -1.5109573785734370e-03
        },
// x = -6.4900
        {
                -1.5093564746733474e-03, -1.5139475099402726e-03, -1.5162465134297605e-03, -1.5173971784494722e-03,
                -1.5179728990487116e-03, -1.5182608887942888e-03, -1.5184049268364501e-03, -1.5184769602525042e-03,
                -1.5185129817601518e-03, -1.5185309941141733e-03
        },
// x = -6.4850
        {
                -1.5168760054566513e-03, -1.5215130756660717e-03, -1.5238351492280948e-03, -1.5249973668445289e-03,
                -1.5255788696022959e-03, -1.5258697523821976e-03, -1.5260152375936268e-03, -1.5260879948117878e-03,
                -1.5261243782930061e-03, -1.5261425716579919e-03
        },
// x = -6.4800
        {
                -1.5244327658279467e-03, -1.5291163320441067e-03, -1.5314617070041269e-03, -1.5326355931449998e-03,
                -1.5332229361136155e-03, -1.5335167409835238e-03, -1.5336636879019103e-03, -1.5337371761943084e-03,
                -1.5337739252862580e-03, -1.5337923014811536e-03
        },
// x = -6.4750
        {
                -1.5320269377800558e-03, -1.5367574656769403e-03, -1.5391263756772454e-03, -1.5403120474326828e-03,
                -1.5409052892470080e-03, -1.5412020455543205e-03, -1.5413504688633604e-03, -1.5414246955751782e-03,
                -1.5414618139515627e-03, -1.5414803748135910e-03
        },
// x = -6.4700
        {
                -1.5396587041719624e-03, -1.5444366640792611e-03, -1.5468293451021305e-03, -1.5480269207363089e-03,
                -1.5486261206195853e-03, -1.5489258580063422e-03, -1.5490757725372048e-03, -1.5491507450874115e-03,
                -1.5491882364588442e-03, -1.5492069838436884e-03
        },
// x = -6.4650
        {
                -1.5473282487326961e-03, -1.5521541156822282e-03, -1.5545708060733237e-03, -1.5557804050362310e-03,
                -1.5563856228059808e-03, -1.5566883712118216e-03, -1.5568397919446285e-03, -1.5569155178267217e-03,
                -1.5569533859410970e-03, -1.5569723217230861e-03
        },
// x = -6.4600
        {
                -1.5550357560652299e-03, -1.5599100098378281e-03, -1.5623509503298247e-03, -1.5635726932691340e-03,
                -1.5641839893431183e-03, -1.5644897790082686e-03, -1.5646427210735895e-03, -1.5647192078563420e-03,
                -1.5647574564992106e-03, -1.5647765825715072e-03
        },
// x = -6.4550
        {
                -1.5627814116503948e-03, -1.5677045368232547e-03, -1.5701699705597024e-03, -1.5714039793327675e-03,
                -1.5720214147350055e-03, -1.5723302762032925e-03, -1.5724847548836572e-03, -1.5725620102118743e-03,
                -1.5726006432068185e-03, -1.5726199614816089e-03
        },
// x = -6.4500
        {
                -1.5705654018508095e-03, -1.5755378878453106e-03, -1.5780280604047352e-03, -1.5792744580907052e-03,
                -1.5798980944575515e-03, -1.5802100585794513e-03, -1.5803660893108726e-03, -1.5804441209061552e-03,
                -1.5804831421151748e-03, -1.5805026545238580e-03
        },
// x = -6.4450
        {
                -1.5783879139148220e-03, -1.5834102550448220e-03, -1.5859254144650686e-03, -1.5871843253771228e-03,
                -1.5878142249634080e-03, -1.5881293228991215e-03, -1.5882869212726386e-03, -1.5883657369341548e-03,
                -1.5884051502580503e-03, -1.5884248587514321e-03
        },
// x = -6.4400
        {
                -1.5862491359804715e-03, -1.5913218315010810e-03, -1.5938622283038952e-03, -1.5951337780016020e-03,
                -1.5957700036868345e-03, -1.5960882669093957e-03, -1.5962474486726291e-03, -1.5963270562778925e-03,
                -1.5963668656566567e-03, -1.5963867722051455e-03
        },
// x = -6.4350
        {
                -1.5941492570794604e-03, -1.5992728112363020e-03, -1.6018386984521607e-03, -1.6031230137539593e-03,
                -1.6037656290485854e-03, -1.6040870893470011e-03, -1.6042478704057250e-03, -1.6043282779113835e-03,
                -1.6043684873245927e-03, -1.6043885939183949e-03
        },
// x = -6.4300
        {
                -1.6020884671411454e-03, -1.6072633892200996e-03, -1.6098550224132888e-03, -1.6111522314090935e-03,
                -1.6118013004608263e-03, -1.6121259899432461e-03, -1.6122883863629771e-03, -1.6123696018056038e-03,
                -1.6124102152728186e-03, -1.6124305239221385e-03
        },
// x = -6.4250
        {
                -1.6100669569965400e-03, -1.6152937613739884e-03, -1.6179113986679280e-03, -1.6192216307318629e-03,
                -1.6198772183320697e-03, -1.6202051694289878e-03, -1.6203691974365868e-03, -1.6204512289334869e-03,
                -1.6204922505146514e-03, -1.6205127632498925e-03
        },
// x = -6.4200
        {
                -1.6180849183823356e-03, -1.6233641245759010e-03, -1.6260080266787243e-03, -1.6273314124819810e-03,
                -1.6279935840721373e-03, -1.6283248295396280e-03, -1.6284905055249216e-03, -1.6285733612749406e-03,
                -1.6286147950707890e-03, -1.6286355139427551e-03
        },
// x = -6.4150
        {
                -1.6261425439449350e-03, -1.6314746766647248e-03, -1.6341451068951139e-03, -1.6354817784189402e-03,
                -1.6361506000971478e-03, -1.6364851730201304e-03, -1.6366525135375462e-03, -1.6367362018218895e-03,
                -1.6367780519743560e-03, -1.6367989790544587e-03
        },
// x = -6.4100
        {
                -1.6342400272445043e-03, -1.6396256164448633e-03, -1.6423228407581392e-03, -1.6436729313069570e-03,
                -1.6443484698345249e-03, -1.6446864036300662e-03, -1.6448554254002844e-03, -1.6449399545833455e-03,
                -1.6449822252759781e-03, -1.6450033626564413e-03
        },
// x = -6.4050
        {
                -1.6423775627590366e-03, -1.6478171436908129e-03, -1.6505414307052876e-03, -1.6519050749199417e-03,
                -1.6525873977280383e-03, -1.6529287261486821e-03, -1.6530994460603066e-03, -1.6531848245905001e-03,
                -1.6532275200488780e-03, -1.6532488698429494e-03
        },
// x = -6.4000
        {
                -1.6505553458884337e-03, -1.6560494591517630e-03, -1.6588010801753539e-03, -1.6601784140464948e-03,
                -1.6608675892428595e-03, -1.6612123463799944e-03, -1.6613847814912377e-03, -1.6614710179018455e-03,
                -1.6615141423940020e-03, -1.6615357067361622e-03
        },
// x = -6.3950
        {
                -1.6587735729586004e-03, -1.6643227645562158e-03, -1.6671019936133256e-03, -1.6684931544949215e-03,
                -1.6691892508706519e-03, -1.6695374711579109e-03, -1.6697116386982960e-03, -1.6697987416083178e-03,
                -1.6698422994451665e-03, -1.6698640804913443e-03
        },
// x = -6.3900
        {
                -1.6670324412255593e-03, -1.6726372626166275e-03, -1.6754443764752903e-03, -1.6768495030982807e-03,
                -1.6775525901346791e-03, -1.6779043083513729e-03, -1.6780802257234540e-03, -1.6781682038384698e-03,
                -1.6782121993742358e-03, -1.6782341993020215e-03
        },
// x = -6.3850
        {
                -1.6753321488795739e-03, -1.6809931570340672e-03, -1.6838284352333676e-03, -1.6852476677194470e-03,
                -1.6859578155949428e-03, -1.6863130668695293e-03, -1.6864907516506286e-03, -1.6865796137636665e-03,
                -1.6866240513963206e-03, -1.6866462724051858e-03
        },
// x = -6.3800
        {
                -1.6836728950492918e-03, -1.6893906525028998e-03, -1.6922543773806639e-03, -1.6936878572562089e-03,
                -1.6944051368533444e-03, -1.6947639566669301e-03, -1.6949434266108926e-03, -1.6950331816033089e-03,
                -1.6950780657750072e-03, -1.6951005100865233e-03
        },
// x = -6.3750
        {
                -1.6920548798059034e-03, -1.6978299547154884e-03, -1.7007224114362497e-03, -1.7021702816463799e-03,
                -1.7028947645588719e-03, -1.7032571887487518e-03, -1.7034384617877160e-03, -1.7035291186300802e-03,
                -1.7035744538276094e-03, -1.7035971236856698e-03
        },
// x = -6.3700
        {
                -1.7004783041673142e-03, -1.7063112703669166e-03, -1.7092327469501634e-03, -1.7106951518729469e-03,
                -1.7114269104128131e-03, -1.7117929751760436e-03, -1.7119760694222332e-03, -1.7120676371752256e-03,
                -1.7121134279304488e-03, -1.7121363256014934e-03
        },
// x = -6.3650
        {
                -1.7089433701023323e-03, -1.7148348071597337e-03, -1.7177855945084343e-03, -1.7192626799692337e-03,
                -1.7200017871739949e-03, -1.7203715290710026e-03, -1.7205564628185329e-03, -1.7206489506338490e-03,
                -1.7206952015241608e-03, -1.7207183292974025e-03
        },
// x = -6.3600
        {
                -1.7174502805348748e-03, -1.7234007738087184e-03, -1.7263811657381341e-03, -1.7278730790240948e-03,
                -1.7286196086640471e-03, -1.7289930646222750e-03, -1.7291798563489800e-03, -1.7292732734702442e-03,
                -1.7293199891190271e-03, -1.7293433493066799e-03
        },
// x = -6.3550
        {
                -1.7259992393481845e-03, -1.7320093800456662e-03, -1.7350196733124477e-03, -1.7365265631871336e-03,
                -1.7372805897726928e-03, -1.7376577970902836e-03, -1.7378464654595595e-03, -1.7379408212232484e-03,
                -1.7379880063003337e-03, -1.7380116012378462e-03
        },
// x = -6.3500
        {
                -1.7345904513890666e-03, -1.7406608366241953e-03, -1.7437013309557722e-03, -1.7452233476739447e-03,
                -1.7459849464630678e-03, -1.7463659428125808e-03, -1.7465565066752491e-03, -1.7466518105116241e-03,
                -1.7466994697337608e-03, -1.7467233017800460e-03
        },
// x = -6.3450
        {
                -1.7432241224721410e-03, -1.7493553553245761e-03, -1.7524263534488380e-03, -1.7539636487713848e-03,
                -1.7547328957770599e-03, -1.7551177192092295e-03, -1.7553101976054185e-03, -1.7554064590394674e-03,
                -1.7554545971707914e-03, -1.7554786687084651e-03
        },
// x = -6.3400
        {
                -1.7519004593841031e-03, -1.7580931489585799e-03, -1.7611949566338522e-03, -1.7627476838428635e-03,
                -1.7635246558406806e-03, -1.7639133447882097e-03, -1.7641077569492539e-03, -1.7642049856016430e-03,
                -1.7642536074541542e-03, -1.7642779208897727e-03
        },
// x = -6.3350
        {
                -1.7606196698880133e-03, -1.7668744313743519e-03, -1.7700073574196716e-03, -1.7715756713336678e-03,
                -1.7723604458694630e-03, -1.7727530391508537e-03, -1.7729494045012149e-03, -1.7730476100892485e-03,
                -1.7730967205232915e-03, -1.7731212782875912e-03
        },
// x = -6.3300
        {
                -1.7693819627275877e-03, -1.7756994174613009e-03, -1.7788637737869925e-03, -1.7804478307763057e-03,
                -1.7812404861738793e-03, -1.7816370229973056e-03, -1.7818353611565106e-03, -1.7819345534951020e-03,
                -1.7819841574198520e-03, -1.7820089619679925e-03
        },
// x = -6.3250
        {
                -1.7781875476315160e-03, -1.7845683231550160e-03, -1.7877644247935739e-03, -1.7893643827958793e-03,
                -1.7901649981647962e-03, -1.7905655181320115e-03, -1.7907658489166097e-03, -1.7908660379192625e-03,
                -1.7909161402932139e-03, -1.7909411941050253e-03
        },
// x = -6.3200
        {
                -1.7870366353177874e-03, -1.7934813654422008e-03, -1.7967095305794758e-03, -1.7983255491154826e-03,
                -1.7991342043589466e-03, -1.7995387474692333e-03, -1.7997410908947755e-03, -1.7998422865745723e-03,
                -1.7998928924060371e-03, -1.7999181979862647e-03
        },
// x = -6.3150
        {
                -1.7959294374980373e-03, -1.8024387623656311e-03, -1.8056993123723331e-03, -1.8073315525616274e-03,
                -1.8081483283844363e-03, -1.8085569350385937e-03, -1.8087613113216292e-03, -1.8088635237922318e-03,
                -1.8089146381398374e-03, -1.8089401980183942e-03
        },
// x = -6.3100
        {
                -1.8048661668819068e-03, -1.8114407330291345e-03, -1.8147339924926426e-03, -1.8163826170696934e-03,
                -1.8172075949862735e-03, -1.8176203059906454e-03, -1.8178267355507395e-03, -1.8179299750273991e-03,
                -1.8179816030005954e-03, -1.8180074197328147e-03
        },
// x = -6.3050
        {
                -1.8138470371814185e-03, -1.8204874976025897e-03, -1.8238137943590842e-03, -1.8254789676894056e-03,
                -1.8263122300319267e-03, -1.8267290866024696e-03, -1.8269375900642417e-03, -1.8270418668648197e-03,
                -1.8270940136243881e-03, -1.8271200897912781e-03
        },
// x = -6.3000
        {
                -1.8228722631153684e-03, -1.8295792773269526e-03, -1.8329389424938632e-03, -1.8346208305903395e-03,
                -1.8354624605169101e-03, -1.8358835042833032e-03, -1.8360941024784857e-03, -1.8361994270244826e-03,
                -1.8362520977830534e-03, -1.8362784359915558e-03
        },
// x = -6.2950
        {
                -1.8319420604137337e-03, -1.8387162945192974e-03, -1.8421096625280766e-03, -1.8438084330674510e-03,
                -1.8446585145703970e-03, -1.8450837875801927e-03, -1.8452965015497078e-03, -1.8454028843673054e-03,
                -1.8454560843898773e-03, -1.8454826872731280e-03
        },
// x = -6.2900
        {
                -1.8410566458220983e-03, -1.8478987725778857e-03, -1.8513261812071089e-03, -1.8530420035466355e-03,
                -1.8539006214608610e-03, -1.8543301661836748e-03, -1.8545450171797374e-03, -1.8546524689008496e-03,
                -1.8547062035053164e-03, -1.8547330737229079e-03
        },
// x = -6.2850
        {
                -1.8502162371060887e-03, -1.8571269359872573e-03, -1.8605887263960490e-03, -1.8623217715903124e-03,
                -1.8631890116017438e-03, -1.8636228709334908e-03, -1.8638398804217272e-03, -1.8639484117850614e-03,
                -1.8640026863427424e-03, -1.8640298265809902e-03
        },
// x = -6.2800
        {
                -1.8594210530558301e-03, -1.8664010103233357e-03, -1.8698975270851348e-03, -1.8716479679030363e-03,
                -1.8725239165571516e-03, -1.8729621338243218e-03, -1.8731813234859126e-03, -1.8732909453380423e-03,
                -1.8733457652742201e-03, -1.8733731782464302e-03
        },
// x = -6.2750
        {
                -1.8686713134904197e-03, -1.8757212222585689e-03, -1.8792528133952226e-03, -1.8810208243371382e-03,
                -1.8819055690475815e-03, -1.8823481880115588e-03, -1.8825695797454025e-03, -1.8826803030418506e-03,
                -1.8827356738363122e-03, -1.8827633622830544e-03
        },
// x = -6.2700
        {
                -1.8779672392624115e-03, -1.8850877995670809e-03, -1.8886548165832839e-03, -1.8904405738983917e-03,
                -1.8913342029556733e-03, -1.8917812678170985e-03, -1.8920048837419965e-03, -1.8921167195483290e-03,
                -1.8921726467359139e-03, -1.8922006134252933e-03
        },
// x = -6.2650
        {
                -1.8873090522623182e-03, -1.8945009711298514e-03, -1.8981037690479253e-03, -1.8999074507517050e-03,
                -1.9008100533319907e-03, -1.9012616087351673e-03, -1.9014874711920321e-03, -1.9016004306849639e-03,
                -1.9016569198561162e-03, -1.9016851675840519e-03
        },
// x = -6.2600
        {
                -1.8966969754231329e-03, -1.9039609669399169e-03, -1.9075999043349358e-03, -1.9094216902268472e-03,
                -1.9103333564008318e-03, -1.9107894474381768e-03, -1.9110175789922603e-03, -1.9111316734607706e-03,
                -1.9111887302620994e-03, -1.9112172618526017e-03
        },
// x = -6.2550
        {
                -1.9061312327248612e-03, -1.9134680181075955e-03, -1.9171434571428612e-03, -1.9189835288241943e-03,
                -1.9199043495660671e-03, -1.9203650217826070e-03, -1.9205954452257530e-03, -1.9207106860722120e-03,
                -1.9207683162070536e-03, -1.9207971345125078e-03
        },
// x = -6.2500
        {
                -1.9156120491990725e-03, -1.9230223568657305e-03, -1.9267346633286024e-03, -1.9285932042205100e-03,
                -1.9295232714170089e-03, -1.9299885708149169e-03, -1.9302213091678361e-03, -1.9303377079091452e-03,
                -1.9303959171381336e-03, -1.9304250250395851e-03
        },
// x = -6.2450
        {
                -1.9251396509334670e-03, -1.9326242165749634e-03, -1.9363737599130434e-03, -1.9382509552747496e-03,
                -1.9391903617343056e-03, -1.9396603347774904e-03, -1.9398954112920554e-03, -1.9400129795607965e-03,
                -1.9400717737024387e-03, -1.9401011741098807e-03
        },
// x = -6.2400
        {
                -1.9347142650764579e-03, -1.9422738317290233e-03, -1.9460609850867003e-03, -1.9479570220338940e-03,
                -1.9489058614958680e-03, -1.9493805551146035e-03, -1.9496179932761710e-03, -1.9497367428217682e-03,
                -1.9497961277530258e-03, -1.9498258236056889e-03
        },
// x = -6.2350
        {
                -1.9443361198417696e-03, -1.9519714379600428e-03, -1.9557965782154014e-03, -1.9577116457388127e-03,
                -1.9586700128828237e-03, -1.9591494744784273e-03, -1.9593892980081821e-03, -1.9595092406980734e-03,
                -1.9595692223549508e-03, -1.9595992166215985e-03
        },
// x = -6.2300
        {
                -1.9540054445130518e-03, -1.9617172720438948e-03, -1.9655807798459940e-03, -1.9675150688301551e-03,
                -1.9684830592855028e-03, -1.9689673367350603e-03, -1.9692095695923784e-03, -1.9693307174132057e-03,
                -1.9693913017913402e-03, -1.9694215974705646e-03
        },
// x = -6.2250
        {
                -1.9637224694485107e-03, -1.9715115719055568e-03, -1.9754138317120752e-03, -1.9773675349542678e-03,
                -1.9783452453094492e-03, -1.9788343869705858e-03, -1.9790790533554320e-03, -1.9792014184142298e-03,
                -1.9792626115694937e-03, -1.9792932116900158e-03
        },
// x = -6.2200
        {
                -1.9734874260855565e-03, -1.9813545766244919e-03, -1.9852959767397505e-03, -1.9872692889691475e-03,
                -1.9882568167814657e-03, -1.9887508714971692e-03, -1.9889979958525039e-03, -1.9891215903779171e-03,
                -1.9891833984270208e-03, -1.9892143060479925e-03
        },
// x = -6.2150
        {
                -1.9833005469454628e-03, -1.9912465264400594e-03, -1.9952274590534203e-03, -1.9972205769504120e-03,
                -1.9982180207556866e-03, -1.9987170378591725e-03, -1.9989666448733969e-03, -1.9990914812168963e-03,
                -1.9991539103379995e-03, -1.9991851285493093e-03
        },
// x = -6.2100
        {
                -1.9931620656380490e-03, -2.0011876627569433e-03, -2.0052085239815917e-03, -2.0072216461973092e-03,
                -2.0082291055196809e-03, -2.0087331348393102e-03, -2.0089852494487273e-03, -2.0091113400858428e-03,
                -2.0091743965191737e-03, -2.0092059284417567e-03
        },
// x = -6.2050
        {
                -2.0030722168663706e-03, -2.0111782281506075e-03, -2.0152394180627219e-03, -2.0172727452387523e-03,
                -2.0182903206005854e-03, -2.0187994124648309e-03, -2.0190540598561328e-03, -2.0191814173877035e-03,
                -2.0192451074361796e-03, -2.0192769562223274e-03
        },
// x = -6.2000
        {
                -2.0130312364314345e-03, -2.0212184663727767e-03, -2.0253203890510832e-03, -2.0273741238393838e-03,
                -2.0284019167712683e-03, -2.0289161220137329e-03, -2.0291733276265145e-03, -2.0293019647799420e-03,
                -2.0293662948098010e-03, -2.0293984636434795e-03
        },
// x = -6.1950
        {
                -2.0230393612369205e-03, -2.0313086223569315e-03, -2.0354516859226570e-03, -2.0375260330056694e-03,
                -2.0385641460565224e-03, -2.0390835160210017e-03, -2.0393433055502998e-03, -2.0394732351808240e-03,
                -2.0395382116222592e-03, -2.0395707037194227e-03
        },
// x = -6.1900
        {
                -2.0330968292939290e-03, -2.0414489422238397e-03, -2.0456335588810597e-03, -2.0477287249920225e-03,
                -2.0487772617392912e-03, -2.0493018482848932e-03, -2.0495642476837514e-03, -2.0496954827757268e-03,
                -2.0497611121235316e-03, -2.0497939307324489e-03
        },
// x = -6.1850
        {
                -2.0432038797257384e-03, -2.0516396732871032e-03, -2.0558662593634920e-03, -2.0579824533069548e-03,
                -2.0590415183669229e-03, -2.0595713738732349e-03, -2.0598364093552890e-03, -2.0599689630234896e-03,
                -2.0600352518377030e-03, -2.0600684002392788e-03
        },
// x = -6.1800
        {
                -2.0533607527725762e-03, -2.0618810640587295e-03, -2.0661500400467147e-03, -2.0682874727192594e-03,
                -2.0693571717574559e-03, -2.0698923491297666e-03, -2.0701600471718620e-03, -2.0702939326627868e-03,
                -2.0703608875693512e-03, -2.0703943690774536e-03
        },
// x = -6.1750
        {
                -2.0635676897964176e-03, -2.0721733642547310e-03, -2.0764851548530560e-03, -2.0786440392642274e-03,
                -2.0797244790059363e-03, -2.0802650316805050e-03, -2.0805354190253414e-03, -2.0806706497185348e-03,
                -2.0807382774099574e-03, -2.0807720953717520e-03
        },
// x = -6.1700
        {
                -2.0738249332857839e-03, -2.0825168248007437e-03, -2.0868718589564478e-03, -2.0890524102498854e-03,
                -2.0901436984907653e-03, -2.0906896804401514e-03, -2.0909627840989477e-03, -2.0910993735083388e-03,
                -2.0911676807443577e-03, -2.0912018385406406e-03
        },
// x = -6.1650
        {
                -2.0841327268605707e-03, -2.0929116978376716e-03, -2.0973104087884838e-03, -2.0995128442632730e-03,
                -2.1006150898800792e-03, -2.1011665556185180e-03, -2.1014424028737051e-03, -2.1015803646489573e-03,
                -2.1016493582572222e-03, -2.1016838593027570e-03
        },
// x = -6.1600
        {
                -2.0944913152768901e-03, -2.1033582367273583e-03, -2.1078010620445144e-03, -2.1100256011767431e-03,
                -2.1111389141381621e-03, -2.1116959187269979e-03, -2.1119745371349467e-03, -2.1121138850628165e-03,
                -2.1121835719395661e-03, -2.1122184196834251e-03
        },
// x = -6.1550
        {
                -2.1049009444319183e-03, -2.1138566960582782e-03, -2.1183440776897607e-03, -2.1205909421542985e-03,
                -2.1217154335318850e-03, -2.1222780325850566e-03, -2.1225594499788216e-03, -2.1227001979845415e-03,
                -2.1227705850952938e-03, -2.1228057830212045e-03
        },
// x = -6.1500
        {
                -2.1153618613687719e-03, -2.1244073316512534e-03, -2.1289397159654686e-03, -2.1312091296579565e-03,
                -2.1323449116371798e-03, -2.1329131613267655e-03, -2.1331974058188658e-03, -2.1333395679675278e-03,
                -2.1334106623477770e-03, -2.1334462139744690e-03
        },
// x = -6.1450
        {
                -2.1258743142813967e-03, -2.1350104005651979e-03, -2.1395882383950762e-03, -2.1418804274541416e-03,
                -2.1430276133455493e-03, -2.1436015704073630e-03, -2.1438886703925835e-03, -2.1440322608905485e-03,
                -2.1441040696464652e-03, -2.1441399785280223e-03
        },
// x = -6.1400
        {
                -2.1364385525194682e-03, -2.1456661611028815e-03, -2.1502899077904267e-03, -2.1526051006201157e-03,
                -2.1537638048706015e-03, -2.1543435266098451e-03, -2.1546335107680724e-03, -2.1547785439643857e-03,
                -2.1548510742735298e-03, -2.1548873439997430e-03
        },
// x = -6.1350
        {
                -2.1470548265933118e-03, -2.1563748728167238e-03, -2.1610449882579993e-03, -2.1633834155504301e-03,
                -2.1645537537546208e-03, -2.1651392980515968e-03, -2.1654321953506767e-03, -2.1655786857385029e-03,
                -2.1656519448505374e-03, -2.1656885790472691e-03
        },
// x = -6.1300
        {
                -2.1577233881788422e-03, -2.1671367965146065e-03, -2.1718537452051680e-03, -2.1742156399634249e-03,
                -2.1753977288751728e-03, -2.1759891541910481e-03, -2.1762849938896794e-03, -2.1764329561077488e-03,
                -2.1765069513451679e-03, -2.1765439536747064e-03
        },
// x = -6.1250
        {
                -2.1684444901225093e-03, -2.1779521942657137e-03, -2.1827164453464996e-03, -2.1851020429077364e-03,
                -2.1862960004517362e-03, -2.1868933658343703e-03, -2.1871921774850216e-03, -2.1873416263190888e-03,
                -2.1874163650779468e-03, -2.1874537392393783e-03
        },
// x = -6.1200
        {
                -2.1792183864462740e-03, -2.1888213294064024e-03, -2.1936333567100738e-03, -2.1960428947688576e-03,
                -2.1972488400523742e-03, -2.1978522051422007e-03, -2.1981540185940585e-03, -2.1983049689783822e-03,
                -2.1983804587290333e-03, -2.1984182084586108e-03
        },
// x = -6.1150
        {
                -2.1900453323525837e-03, -2.1997444665460848e-03, -2.2046047486438281e-03, -2.2070384672757170e-03,
                -2.2082565206004304e-03, -2.2088659456364012e-03, -2.2091707910383484e-03, -2.2093232580571820e-03,
                -2.2093995063450235e-03, -2.2094376354165383e-03
        },
// x = -6.1100
        {
                -2.2009255842293822e-03, -2.2107218715731493e-03, -2.2156308918219463e-03, -2.2180890335072993e-03,
                -2.2193193163812671e-03, -2.2199348622068535e-03, -2.2202427700104771e-03, -2.2203967688995724e-03,
                -2.2204737833458046e-03, -2.2205122955709622e-03
        },
// x = -6.1050
        {
                -2.2118593996551155e-03, -2.2217538116608977e-03, -2.2267120582512589e-03, -2.2291948678992844e-03,
                -2.2304375030490313e-03, -2.2310592311182856e-03, -2.2313702320809121e-03, -2.2315257782290440e-03,
                -2.2316035665314242e-03, -2.2316424657602306e-03
        },
// x = -6.1000
        {
                -2.2228470374037737e-03, -2.2328405552735093e-03, -2.2378485212776939e-03, -2.2403562462507353e-03,
                -2.2416113576334529e-03, -2.2422393300171317e-03, -2.2425534552048981e-03, -2.2427105641553975e-03,
                -2.2427891340890153e-03, -2.2428284242101562e-03
        },
// x = -6.0950
        {
                -2.2338887574499338e-03, -2.2439823721720343e-03, -2.2490405555927358e-03, -2.2515734457308045e-03,
                -2.2528411585466799e-03, -2.2534754379384282e-03, -2.2537927187293804e-03, -2.2539514061816861e-03,
                -2.2540307655997374e-03, -2.2540704505409686e-03
        },
// x = -6.0900
        {
                -2.2449848209738273e-03, -2.2551795334204076e-03, -2.2602884372399279e-03, -2.2628467448854777e-03,
                -2.2641271855901447e-03, -2.2647678353127417e-03, -2.2650883033999629e-03, -2.2652485852111930e-03,
                -2.2653287420457630e-03, -2.2653688257743038e-03
        },
// x = -6.0850
        {
                -2.2561354903664174e-03, -2.2664323113914895e-03, -2.2715924436214081e-03, -2.2741764236443546e-03,
                -2.2754697199614681e-03, -2.2761168039731349e-03, -2.2764404913679097e-03, -2.2766023835544395e-03,
                -2.2766833458172974e-03, -2.2767238323402267e-03
        },
// x = -6.0800
        {
                -2.2673410292344993e-03, -2.2777409797731332e-03, -2.2829528535044576e-03, -2.2855627633274556e-03,
                -2.2868690442613884e-03, -2.2875226271621622e-03, -2.2878495661971671e-03, -2.2880130849362323e-03,
                -2.2880948607196297e-03, -2.2881357540842865e-03
        },
// x = -6.0750
        {
                -2.2786017024058109e-03, -2.2891058135742753e-03, -2.2943699470281014e-03, -2.2970060466520610e-03,
                -2.2983254425007369e-03, -2.2989855895389055e-03, -2.2993158128714348e-03, -2.2994809745027437e-03,
                -2.2995635719802242e-03, -2.2996048762746108e-03
        },
// x = -6.0700
        {
                -2.2899177759341640e-03, -2.3005270891310547e-03, -2.3058440057097227e-03, -2.3085065577395920e-03,
                -2.3098392001074390e-03, -2.3105059771860396e-03, -2.3108395178012640e-03, -2.3110063388286301e-03,
                -2.3110897662558466e-03, -2.3111314856090370e-03
        },
// x = -6.0650
        {
                -2.3012895171045837e-03, -2.3120050841129555e-03, -2.3173753124517164e-03, -2.3200645821225183e-03,
                -2.3214106039335502e-03, -2.3220840776169362e-03, -2.3224209688311919e-03, -2.3225894659241802e-03,
                -2.3226737316397179e-03, -2.3227158702222731e-03
        },
// x = -6.0600
        {
                -2.3127171944384737e-03, -2.3235400775289744e-03, -2.3289641515481709e-03, -2.3316804067512937e-03,
                -2.3330399422623318e-03, -2.3337201797828029e-03, -2.3340604552469155e-03, -2.3342306452425079e-03,
                -2.3343157576687160e-03, -2.3343583196930996e-03
        },
// x = -6.0550
        {
                -2.3242010776987941e-03, -2.3351323497338180e-03, -2.3406108086915819e-03, -2.3433543200013368e-03,
                -2.3447275048153562e-03, -2.3454145740798571e-03, -2.3457582677824948e-03, -2.3459301676867720e-03,
                -2.3460161353306081e-03, -2.3460591250516042e-03

        },
// x = -6.0500
        {
                -2.3357414378952493e-03, -2.3467821824341237e-03, -2.3523155709795967e-03, -2.3550866116800423e-03,
                -2.3564735827596504e-03, -2.3571675523565319e-03, -2.3575146986276026e-03, -2.3576883256174381e-03,
                -2.3577751570713168e-03, -2.3578185787864591e-03
        },
// x = -6.0450
        {
                -2.3473385472894993e-03, -2.3584898586947014e-03, -2.3640787269217857e-03, -2.3668775730338156e-03,
                -2.3682784687148717e-03, -2.3689794079207269e-03, -2.3693300414347955e-03, -2.3695054128595738e-03,
                -2.3695931168022238e-03, -2.3696369748522229e-03
        },
// x = -6.0400
        {
                -2.3589926794003861e-03, -2.3702556629448131e-03, -2.3759005664464533e-03, -2.3787274967551554e-03,
                -2.3801424567605235e-03, -2.3808504355470866e-03, -2.3812045913268352e-03, -2.3813817247101830e-03,
                -2.3814703099075147e-03, -2.3815146086766936e-03
        },
// x = -6.0350
        {
                -2.3707041090091706e-03, -2.3820798809844686e-03, -2.3877813809074764e-03, -2.3906366769897640e-03,
                -2.3920658424432012e-03, -2.3927809314843165e-03, -2.3931386449040394e-03, -2.3933175579455720e-03,
                -2.3934070332515549e-03, -2.3934517771682837e-03
        },
// x = -6.0300
        {
                -2.3824731121647900e-03, -2.3939627999907498e-03, -2.3997214630911696e-03, -2.4026054093436917e-03,
                -2.4040489227838760e-03, -2.4047711934625372e-03, -2.4051325002516677e-03, -2.4053132108287608e-03,
                -2.4054035851863043e-03, -2.4054487787234474e-03
        },
// x = -6.0250
        {
                -2.3942999661891334e-03, -2.4059047085241633e-03, -2.4117211072231890e-03, -2.4146339908905131e-03,
                -2.4160919962852145e-03, -2.4168215207006766e-03, -2.4171864569473519e-03, -2.4173689831169221e-03,
                -2.4174602655587709e-03, -2.4175059132341302e-03
        },
// x = -6.0200
        {
                -2.4061849496823228e-03, -2.4179058965350208e-03, -2.4237806089754664e-03, -2.4267227201785471e-03,
                -2.4281953629389382e-03, -2.4289322139138974e-03, -2.4293008160685527e-03, -2.4294851760688634e-03,
                -2.4295773757184993e-03, -2.4296234820952670e-03
        },
// x = -6.0150
        {
                -2.4181283425280271e-03, -2.4299666553698395e-03, -2.4359002654731720e-03, -2.4388718972380998e-03,
                -2.4403593242332071e-03, -2.4411035753210547e-03, -2.4414758802000672e-03, -2.4416620924525450e-03,
                -2.4417552185250994e-03, -2.4418017882123131e-03
        },
// x = -6.0100
        {
                -2.4301304258987719e-03, -2.4420872777777755e-03, -2.4480803753017125e-03, -2.4510818235887527e-03,
                -2.4525841831600577e-03, -2.4533359086522035e-03, -2.4537119534415565e-03, -2.4539000365526339e-03,
                -2.4539940983558050e-03, -2.4540411360088107e-03
        },
// x = -6.0050
        {
                -2.4421914822612806e-03, -2.4542680579170786e-03, -2.4603212385137605e-03, -2.4633528022466778e-03,
                -2.4648702442228608e-03, -2.4656295191561343e-03, -2.4660093414151296e-03, -2.4661993141780990e-03,
                -2.4662943211130857e-03, -2.4663418314340006e-03
        },
// x = -6.0000
        {
                -2.4543117953818240e-03, -2.4665092913615824e-03, -2.4726231566363165e-03, -2.4756851377319956e-03,
                -2.4772178134438239e-03, -2.4779847136079483e-03, -2.4783683512729499e-03, -2.4785602326698442e-03,
                -2.4786561942322814e-03, -2.4787041819704634e-03
        },
// x = -5.9950
        {
                -2.4664916503315897e-03, -2.4788112751072096e-03, -2.4849864326778018e-03, -2.4880791360761601e-03,
                -2.4896271983715313e-03, -2.4904018003166690e-03, -2.4907892917048887e-03, -2.4909831009083702e-03,
                -2.4910800266892802e-03, -2.4911284966418041e-03
        },
// x = -5.9900
        {
                -2.4787313334920609e-03, -2.4911743075785135e-03, -2.4974113711351887e-03, -2.5005351048293873e-03,
                -2.5020987080885187e-03, -2.5028810891328970e-03, -2.5032724729462130e-03, -2.5034682293214924e-03,
                -2.5035661290082403e-03, -2.5036150860203764e-03
        },
// x = -5.9850
        {
                -2.4910311325604174e-03, -2.5035986886352405e-03, -2.5098982780011538e-03, -2.5130533530681133e-03,
                -2.5146326532188863e-03, -2.5154228914564939e-03, -2.5158182067853141e-03, -2.5160159298920791e-03,
                -2.5161148132693415e-03, -2.5161642622350401e-03
        },
// x = -5.9800
        {
                -2.5033913365549508e-03, -2.5160847195789251e-03, -2.5224474607712767e-03, -2.5256341914024929e-03,
                -2.5272293459359450e-03, -2.5280275202443146e-03, -2.5284268065714696e-03, -2.5286265161658409e-03,
                -2.5287263931165834e-03, -2.5287763389789638e-03
        },
// x = -5.9750
        {
                -2.5158122358204891e-03, -2.5286327031595053e-03, -2.5350592284512619e-03, -2.5382779319839291e-03,
                -2.5398890999699078e-03, -2.5406952900179662e-03, -2.5410985872226497e-03, -2.5413003032591524e-03,
                -2.5414011837656160e-03, -2.5414516315174587e-03
        },
// x = -5.9700
        {
                -2.5282941220338478e-03, -2.5412429435819725e-03, -2.5477338915641978e-03, -2.5509848885126459e-03,
                -2.5526122306156140e-03, -2.5534265168716164e-03, -2.5538338652333647e-03, -2.5540376078669155e-03,
                -2.5541395020116145e-03, -2.5541904566958639e-03
        },
// x = -5.9650
        {
                -2.5408372882092896e-03, -2.5539157465130427e-03, -2.5604717621578511e-03, -2.5637553762452910e-03,
                -2.5653990547402916e-03, -2.5662215184798368e-03, -2.5666329586825393e-03, -2.5668387482704652e-03,
                -2.5669416662371927e-03, -2.5669931329474565e-03
        },
// x = -5.9600
        {
                -2.5534420287039957e-03, -2.5666514190878596e-03, -2.5732731538119892e-03, -2.5765897120025773e-03,
                -2.5782498907913577e-03, -2.5790806141054788e-03, -2.5794961872414408e-03, -2.5797040443455059e-03,
                -2.5798079964203533e-03, -2.5798599803014133e-03
        },
// x = -5.9550
        {
                -2.5661086392235639e-03, -2.5794502699167201e-03, -2.5861383816457415e-03, -2.5894882141769618e-03,
                -2.5911650588042621e-03, -2.5920041246076018e-03, -2.5924238721816363e-03, -2.5926338175700967e-03,
                -2.5927388141424785e-03, -2.5927913203908060e-03
        },
// x = -5.9500
        {
                -2.5788374168275131e-03, -2.5923126090918311e-03, -2.5990677623249883e-03, -2.6024512027403618e-03,
                -2.6041448804103584e-03, -2.6049923724494255e-03, -2.6054163363829971e-03, -2.6056283910326720e-03,
                -2.6057344425963655e-03, -2.6057874764606388e-03
        },
// x = -5.9450
        {
                -2.5916286599348055e-03, -2.6052387481940958e-03, -2.6120616140697926e-03, -2.6154789992519053e-03,
                -2.6171896788448274e-03, -2.6180456817063344e-03, -2.6184739043417370e-03, -2.6186880894401026e-03,
                -2.6187952065942935e-03, -2.6188487733759236e-03
        },
// x = -5.9400
        {
                -2.6044826683293862e-03, -2.6182290002999195e-03, -2.6251202566618557e-03, -2.6285719268657214e-03,
                -2.6302997789546295e-03, -2.6311643780739134e-03, -2.6315969021784924e-03, -2.6318132391257997e-03,
                -2.6319214325761430e-03, -2.6319755376298017e-03
        },
// x = -5.9350
        {
                -2.6173997431657337e-03, -2.6312836799880532e-03, -2.6382440114520156e-03, -2.6417303103387696e-03,
                -2.6434755072064979e-03, -2.6443487888760304e-03, -2.6447856576464470e-03, -2.6450041680578531e-03,
                -2.6451134486175417e-03, -2.6451680973516968e-03
        },
// x = -5.9300
        {
                -2.6303801869744319e-03, -2.6444031033464577e-03, -2.6514332013677731e-03, -2.6549544760386999e-03,
                -2.6567171916949757e-03, -2.6575992430729525e-03, -2.6580405001394897e-03, -2.6582612058472180e-03,
                -2.6583715844380631e-03, -2.6584267823155207e-03
        },
// x = -5.9250
        {
                -2.6434243036677543e-03, -2.6575875879791991e-03, -2.6646881509208571e-03, -2.6682447519517593e-03,
                -2.6700251621504874e-03, -2.6709160712695052e-03, -2.6713617607004208e-03, -2.6715846837559393e-03,
                -2.6716961714094581e-03, -2.6717519239479078e-03
        },
// x = -5.9200
        {
                -2.6565323985452568e-03, -2.6708374530133696e-03, -2.6780091862148199e-03, -2.6816014676907260e-03,
                -2.6833997499474504e-03, -2.6842996057232721e-03, -2.6847497720291916e-03, -2.6849749347054135e-03,
                -2.6850875425639352e-03, -2.6851438553364983e-03
        },
// x = -5.9150
        {
                -2.6697047782993995e-03, -2.6841530191060396e-03, -2.6913966349526733e-03, -2.6950249545028959e-03,
                -2.6968412881124295e-03, -2.6977501803528378e-03, -2.6982048684911913e-03, -2.6984322932846984e-03,
                -2.6985460326024728e-03, -2.6986029112382641e-03
        },
// x = -5.9100
        {
                -2.6829417510211714e-03, -2.6975346084512348e-03, -2.7048508264445525e-03, -2.7085155452780908e-03,
                -2.7103501113323303e-03, -2.7112681307460661e-03, -2.7117273861255705e-03, -2.7119570957588571e-03,
                -2.7120719779031846e-03, -2.7121294280878709e-03
        },
// x = -5.9050
        {
                -2.6962436262057366e-03, -2.7109825447869480e-03, -2.7183720916154216e-03, -2.7220735745567197e-03,
                -2.7239265559626286e-03, -2.7248537941684230e-03, -2.7253176626536113e-03, -2.7255496800773532e-03,
                -2.7256657165297122e-03, -2.7257237440060826e-03
        },
// x = -5.9000
        {
                -2.7096107147580950e-03, -2.7244971534021649e-03, -2.7319607630128060e-03, -2.7356993785378693e-03,
                -2.7375709600356468e-03, -2.7385075095713403e-03, -2.7389760374871325e-03, -2.7392103858824789e-03,
                -2.7393275882396785e-03, -2.7393861988082168e-03
        },
// x = -5.8950
        {
                -2.7230433289987511e-03, -2.7380787611439349e-03, -2.7456171748145718e-03, -2.7493932950874351e-03,
                -2.7512836632688654e-03, -2.7522296176006231e-03, -2.7527028517369450e-03, -2.7529395545178277e-03,
                -2.7530579344931631e-03, -2.7531171340126285e-03
        },
// x = -5.8900
        {
                -2.7365417826694092e-03, -2.7517276964244605e-03, -2.7593416628367271e-03, -2.7631556637462956e-03,
                -2.7650650070732809e-03, -2.7660204606048916e-03, -2.7664984482213399e-03, -2.7667375290368152e-03,
                -2.7668570984612385e-03, -2.7669168928492480e-03
        },
// x = -5.8850
        {
                -2.7501063909386734e-03, -2.7654442892282183e-03, -2.7731345645412714e-03, -2.7769868257385183e-03,
                -2.7789153345617944e-03, -2.7798803826440724e-03, -2.7803631714746280e-03, -2.7806046542112311e-03,
                -2.7807254250345327e-03, -2.7807858202681577e-03
        },
// x = -5.8800
        {
                -2.7637374704077709e-03, -2.7792288711191043e-03, -2.7869962190440706e-03, -2.7908871239796171e-03,
                -2.7928349905576564e-03, -2.7938097294979303e-03, -2.7942973677557137e-03, -2.7945412765398490e-03,
                -2.7946632608318485e-03, -2.7947242629482081e-03
        },
// x = -5.8750
        {
                -2.7774353391162784e-03, -2.7930817752476185e-03, -2.8009269671227742e-03, -2.8048569030848305e-03,
                -2.8068243216029408e-03, -2.8078088486746399e-03, -2.8083013850567196e-03, -2.8085477442570629e-03,
                -2.8086709542088197e-03, -2.8087325693056829e-03
        },
// x = -5.8700
        {
                -2.7912003165478771e-03, -2.8070033363580631e-03, -2.8149271512247650e-03, -2.8188965093774612e-03,
                -2.8208836759670607e-03, -2.8218780894193993e-03, -2.8223755731116473e-03, -2.8226244073415794e-03,
                -2.8227488552666049e-03, -2.8228110895030037e-03
        },
// x = -5.8650
        {
                -2.8050327236361140e-03, -2.8209938907957826e-03, -2.8289971154751496e-03, -2.8330062908972376e-03,
                -2.8350134036553357e-03, -2.8360178027230908e-03, -2.8365202834050858e-03, -2.8367716175251505e-03,
                -2.8368973158606403e-03, -2.8369601754574827e-03
        },
// x = -5.8600
        {
                -2.8189328827701735e-03, -2.8350537765144245e-03, -2.8431372056847734e-03, -2.8471865974087221e-03,
                -2.8492138564175937e-03, -2.8502283413309828e-03, -2.8507358691809586e-03, -2.8509897283013411e-03,
                -2.8511166896094176e-03, -2.8511801808501088e-03
        },
// x = -5.8550
        {
                -2.8329011178006781e-03, -2.8491833330832341e-03, -2.8573477693582907e-03, -2.8614377804097673e-03,
                -2.8634853877568130e-03, -2.8645100597514692e-03, -2.8650226854513221e-03, -2.8652790949343569e-03,
                -2.8654073319033207e-03, -2.8654714611343912e-03
        },
// x = -5.8500
        {
                -2.8469377540454878e-03, -2.8633829016943774e-03, -2.8716291557022563e-03, -2.8757601931399986e-03,
                -2.8778283529378138e-03, -2.8788633142648600e-03, -2.8793810890052006e-03, -2.8796400744678981e-03,
                -2.8797695999135003e-03, -2.8798343735452403e-03
        },
// x = -5.8450
        {
                -2.8610431182955233e-03, -2.8776528251702863e-03, -2.8859817156332559e-03, -2.8901541905893445e-03,
                -2.8922431089959846e-03, -2.8932884629322146e-03, -2.8938114384174656e-03, -2.8940730257340768e-03,
                -2.8942038526007918e-03, -2.8942692771078918e-03
        },
// x = -5.8400
        {
                -2.8752175388206012e-03, -2.8919934479710459e-03, -2.9004058017860802e-03, -2.9046201295066100e-03,
                -2.9067300147460600e-03, -2.9077858656042120e-03, -2.9083140940577680e-03, -2.9085783093623580e-03,
                -2.9087104507246821e-03, -2.9087765326468798e-03
        },
// x = -5.8350
        {
                -2.8894613453752816e-03, -2.9064051162017968e-03, -2.9149017685219289e-03, -2.9191583684080851e-03,
                -2.9212894307909308e-03, -2.9223558839300708e-03, -2.9228894180995018e-03, -2.9231562877885609e-03,
                -2.9232897568523164e-03, -2.9233565027950484e-03
        },
// x = -5.8300
        {
                -2.9037748692047325e-03, -2.9208881776201785e-03, -2.9294699719366567e-03, -2.9337692675862016e-03,
                -2.9359217195305074e-03, -2.9369988813665164e-03, -2.9375377745288235e-03, -2.9378073252639019e-03,
                -2.9379421353675525e-03, -2.9380095520026186e-03
        },
// x = -5.8250
        {
                -2.9181584430506054e-03, -2.9354429816437931e-03, -2.9441107698690515e-03, -2.9484531891182213e-03,
                -2.9506272451706203e-03, -2.9517152231867804e-03, -2.9522595291537096e-03, -2.9525317878640778e-03,
                -2.9526679524800653e-03, -2.9527360465462895e-03
        },
// x = -5.8200
        {
                -2.9326124011569278e-03, -2.9500698793577061e-03, -2.9588245219091566e-03, -2.9632104968749782e-03,
                -2.9654063737319684e-03, -2.9665052764896614e-03, -2.9670550496130651e-03, -2.9673300434983987e-03,
                -2.9674675762344856e-03, -2.9675363545383915e-03
        },
// x = -5.8150
        {
                -2.9471370792760083e-03, -2.9647692235219708e-03, -2.9736115894066243e-03, -2.9780415565296490e-03,
                -2.9802594730591043e-03, -2.9813694102086129e-03, -2.9819247053858690e-03, -2.9822024619189664e-03,
                -2.9823413765195953e-03, -2.9824108459360868e-03
        },
// x = -5.8100
        {
                -2.9617328146743561e-03, -2.9795413685791854e-03, -2.9884723354791087e-03, -2.9929467355665758e-03,
                -2.9951869128294730e-03, -2.9963079951208929e-03, -2.9968688678003756e-03, -2.9971494147298956e-03,
                -2.9972897250775611e-03, -2.9973598925506100e-03
        },
// x = -5.8050
        {
                -2.9763999461386100e-03, -2.9943866706620766e-03, -3.0034071250207022e-03, -3.0079264032901276e-03,
                -3.0101890645624842e-03, -3.0113214038567475e-03, -3.0118879100433514e-03, -3.0121712753965850e-03,
                -3.0123129955132214e-03, -3.0123838680565593e-03
        },
// x = -5.8000
        {
                -2.9911388139814894e-03, -3.0093054876011221e-03, -3.0184163247103991e-03, -3.0229809308336009e-03,
                -3.0252663016286430e-03, -3.0264100109086414e-03, -3.0269822071693699e-03, -3.0272684192550344e-03,
                -3.0274115633034133e-03, -3.0274831480012337e-03
        },
// x = -5.7950
        {
                -3.0059497600477525e-03, -3.0242981789321832e-03, -3.0335003030206096e-03, -3.0381106911681701e-03,
                -3.0404189992587075e-03, -3.0415741926405416e-03, -3.0421521361101386e-03, -3.0424412235212032e-03,
                -3.0425858058063505e-03, -3.0426581098140148e-03
        },
// x = -5.7900
        {
                -3.0208331277201675e-03, -3.0393651059041947e-03, -3.0486594302256988e-03, -3.0533160591118694e-03,
                -3.0556475345529083e-03, -3.0568143272972359e-03, -3.0573980756838834e-03, -3.0576900673004231e-03,
                -3.0578361022710462e-03, -3.0579091328157989e-03
        },
// x = -5.7850
        {
                -3.0357892619255022e-03, -3.0545066314868591e-03, -3.0638940784105781e-03, -3.0685974113386304e-03,
                -3.0709522864901992e-03, -3.0721307950137091e-03, -3.0727204066047765e-03, -3.0730153315968525e-03,
                -3.0731628338467824e-03, -3.0732365982284750e-03
        },
// x = -5.7800
        {
                -3.0508185091405221e-03, -3.0697231203783838e-03, -3.0792046214793236e-03, -3.0839551263873513e-03,
                -3.0863336359375645e-03, -3.0875239778245540e-03, -3.0881195114924090e-03, -3.0884173993229805e-03,
                -3.0885663835926291e-03, -3.0886408891844484e-03
        },
// x = -5.7750
        {
                -3.0659212173980064e-03, -3.0850149390132526e-03, -3.0945914351638419e-03, -3.0993895846710188e-03,
                -3.1017919656593641e-03, -3.1029942596734403e-03, -3.1035957748813165e-03, -3.1038966553091772e-03,
                -3.1040471364870100e-03, -3.1041223907362133e-03
        },
// x = -5.7700
        {
                -3.0810977362927725e-03, -3.1003824555700110e-03, -3.1100548970325673e-03, -3.1149011684858661e-03,
                -3.1173276603267269e-03, -3.1185420264226221e-03, -3.1191495832305419e-03, -3.1194534863132916e-03,
                -3.1196054794373151e-03, -3.1196814898659755e-03
        },
// x = -5.7650
        {
                -3.0963484169877131e-03, -3.1158260399790981e-03, -3.1255953864992030e-03, -3.1304902620205763e-03,
                -3.1329411065269888e-03, -3.1341676658624938e-03, -3.1347813249332579e-03, -3.1350882810303017e-03,
                -3.1352418012895612e-03, -3.1353185754953200e-03
        },
// x = -5.7600
        {
                -3.1116736122198531e-03, -3.1313460639307003e-03, -3.1412132848315029e-03, -3.1461572513655357e-03,
                -3.1486326927731814e-03, -3.1498715677211993e-03, -3.1504913903264284e-03, -3.1508014301020047e-03,
                -3.1509564928381053e-03, -3.1510340384949282e-03
        },
// x = -5.7550
        {
                -3.1270736763064117e-03, -3.1469429008826374e-03, -3.1569089751600833e-03, -3.1619025245221196e-03,
                -3.1644028095135614e-03, -3.1656541236742795e-03, -3.1662801717005236e-03, -3.1665933261267637e-03,
                -3.1667499468353987e-03, -3.1668282716943441e-03
        },
// x = -5.7500
        {
                -3.1425489651508839e-03, -3.1626169260682777e-03, -3.1726828424872909e-03, -3.1777264714120340e-03,
                -3.1802518491411890e-03, -3.1815157273543766e-03, -3.1821480633092785e-03, -3.1824643636692968e-03,
                -3.1826225580017975e-03, -3.1827016698917927e-03
        },
// x = -5.7450
        {
                -3.1580998362491261e-03, -3.1783685165044850e-03, -3.1885352736960898e-03, -3.1936294838866949e-03,
                -3.1961802060035535e-03, -3.1974567743609781e-03, -3.1980954613795064e-03, -3.1984149392705200e-03,
                -3.1985747230354172e-03, -3.1986546298640355e-03
        },
// x = -5.7400
        {
                -3.1737266486954619e-03, -3.1941980509995979e-03, -3.2044666575590055e-03, -3.2096119557366515e-03,
                -3.2121882764122446e-03, -3.2134776622702176e-03, -3.2141227641209543e-03, -3.2144454514574381e-03,
                -3.2146068406220399e-03, -3.2146875503762948e-03
        },
// x = -5.7350
        {
                -3.1894297631887975e-03, -3.2101059101614336e-03, -3.2204773847471007e-03, -3.2256742827010596e-03,
                -3.2282764586526740e-03, -3.2295787906447154e-03, -3.2302303717362114e-03, -3.2305563007530799e-03,
                -3.2307193114450667e-03, -3.2308008321922104e-03
        },
// x = -5.7300
        {
                -3.2052095420387517e-03, -3.2260924764053287e-03, -3.2365678478389944e-03, -3.2418168624771945e-03,
                -3.2444451529938380e-03, -3.2457605610434737e-03, -3.2464186864306689e-03, -3.2467478896864919e-03,
                -3.2469125381955266e-03, -3.2469948780838517e-03
        },
// x = -5.7250
        {
                -3.2210663491717890e-03, -3.2421581339622078e-03, -3.2527384413299125e-03, -3.2580400947300123e-03,
                -3.2606947616981357e-03, -3.2620233770318191e-03, -3.2626881124225219e-03, -3.2630206228027744e-03,
                -3.2631869255821287e-03, -3.2632700928417846e-03
        },
// x = -5.7200
        {
                -3.2370005501373813e-03, -3.2583032688866826e-03, -3.2689895616407989e-03, -3.2743443811017580e-03,
                -3.2770256890312308e-03, -3.2783676441913963e-03, -3.2790390559528319e-03, -3.2793749066731734e-03,
                -3.2795428803413696e-03, -3.2796268832851853e-03
        },
// x = -5.7150
        {
                -3.2530125121141623e-03, -3.2745282690651812e-03, -3.2853216071274398e-03, -3.2907301252216104e-03,
                -3.2934383412719617e-03, -3.2947937701302038e-03, -3.2954719252956269e-03, -3.2958111499052159e-03,
                -3.2959808112476884e-03, -3.2960656582720011e-03
        },
// x = -5.7100
        {
                -3.2691026039161119e-03, -3.2908335242241133e-03, -3.3017349780896537e-03, -3.3071977327153879e-03,
                -3.3099331267222969e-03, -3.3113021644926906e-03, -3.3119871307680654e-03, -3.3123297631529039e-03,
                -3.3125011291236725e-03, -3.3125868287091696e-03
        },
// x = -5.7050
        {
                -3.2852711959987361e-03, -3.3072194259380571e-03, -3.3182300767805074e-03, -3.3237476112152847e-03,
                -3.3265104557173514e-03, -3.3278932389698927e-03, -3.3285850847406357e-03, -3.3289311591269541e-03,
                -3.3291042468503178e-03, -3.3291908075628791e-03
        },
// x = -5.7000
        {
                -3.3015186604652730e-03, -3.3236863676379866e-03, -3.3348073074155791e-03, -3.3403801703696628e-03,
                -3.3431707406354340e-03, -3.3445674073096271e-03, -3.3452662016474248e-03, -3.3456157526050904e-03,
                -3.3457905793773382e-03, -3.3458780098688957e-03
        },
// x = -5.6950
        {
                -3.3178453710728997e-03, -3.3402347446195245e-03, -3.3514670761822579e-03, -3.3570958218528879e-03,
                -3.3599143959081551e-03, -3.3613250853267293e-03, -3.3620308979964170e-03, -3.3623839604423876e-03,
                -3.3625605437335232e-03, -3.3626488527429242e-03
        },
// x = -5.6900
        {
                -3.3342517032389531e-03, -3.3568649540512258e-03, -3.3682097912490919e-03, -3.3738949793752134e-03,
                -3.3767418380305863e-03, -3.3781666909133484e-03, -3.3788795923798639e-03, -3.3792362015816675e-03,
                -3.3794145590371541e-03, -3.3795037553910326e-03
        },
// x = -5.6850
        {
                -3.3507380340471668e-03, -3.3735773949828966e-03, -3.3850358627751705e-03, -3.3907780586927053e-03,
                -3.3936534855714571e-03, -3.3950926440492885e-03, -3.3958127054846887e-03, -3.3961728970639438e-03,
                -3.3963530465064662e-03, -3.3964431391201263e-03
        },
// x = -5.6800
        {
                -3.3673047422539135e-03, -3.3903724683539407e-03, -3.4019457029195533e-03, -3.4077454776172225e-03,
                -3.4106497591834158e-03, -3.4121033668124035e-03, -3.4128306601029570e-03, -3.4131944700389245e-03,
                -3.4133764294701691e-03, -3.4134674273484755e-03
        },
// x = -5.6750
        {
                -3.3839522082944585e-03, -3.4072505770017369e-03, -3.4189397258507369e-03, -3.4247976560264383e-03,
                -3.4277310816133266e-03, -3.4291992833890409e-03, -3.4299338811423891e-03, -3.4303013457755624e-03,
                -3.4304851333780122e-03, -3.4305770456162887e-03
        },
// x = -5.6700
        {
                -3.4006808142892299e-03, -3.4242121256700532e-03, -3.4360183477561697e-03, -3.4419350158739121e-03,
                -3.4448978777126328e-03, -3.4463808200845400e-03, -3.4471227956369326e-03, -3.4474939516726598e-03,
                -3.4476795858114115e-03, -3.4477724215963503e-03
        },
// x = -5.6650
        {
                -3.4174909440500920e-03, -3.4412575210174836e-03, -3.4531819868518013e-03, -3.4591579811992096e-03,
                -3.4621505744477513e-03, -3.4636484053337815e-03, -3.4643978327573775e-03, -3.4647727172695273e-03,
                -3.4649602164941238e-03, -3.4650539851047040e-03
        },
// x = -5.6600
        {
                -3.4343829830866326e-03, -3.4583871716259238e-03, -3.4704310633916781e-03, -3.4764669781380624e-03,
                -3.4794896009105358e-03, -3.4810024697117838e-03, -3.4817594238220374e-03, -3.4821380742566929e-03,
                -3.4823274573029766e-03, -3.4824221681113874e-03
        },
// x = -5.6550
        {
                -3.4513573186124628e-03, -3.4756014880090763e-03, -3.4877659996775828e-03, -3.4938624349325942e-03,
                -3.4969153883287795e-03, -3.4984434459443647e-03, -3.4992080023074721e-03, -3.4995904564866674e-03,
                -3.4997817422786478e-03, -3.4998774047512301e-03
        },
// x = -5.6500
        {
                -3.4684143395515214e-03, -3.4929008826209808e-03, -3.5051872200687136e-03, -3.5113447819415714e-03,
                -3.5144283700767730e-03, -3.5159717689188397e-03, -3.5167440038592693e-03, -3.5171302999847625e-03,
                -3.5173235076365068e-03, -3.5174201313346951e-03
        },
// x = -5.6450
        {
                -3.4855544365443984e-03, -3.5102857698645902e-03, -3.5226951509914088e-03, -3.5289144516507239e-03,
                -3.5320289816859153e-03, -3.5335878756947857e-03, -3.5343678663028806e-03, -3.5347580429599635e-03,
                -3.5349531917775066e-03, -3.5350507863587833e-03
        },
// x = -5.6400
        {
                -3.5027780019546563e-03, -3.5277565661003640e-03, -3.5402902209489095e-03, -3.5465718786830995e-03,
                -3.5497176608553721e-03, -3.5512922055148518e-03, -3.5520800296545076e-03, -3.5524741258158546e-03,
                -3.5526712352991247e-03, -3.5527698105179873e-03
        },
// x = -5.6350
        {
                -3.5200854298751728e-03, -3.5453136896548961e-03, -3.5579728605311723e-03, -3.5643174998094764e-03,
                -3.5674948474627884e-03, -3.5690851998156203e-03, -3.5698809361320461e-03, -3.5702789911616008e-03,
                -3.5704780810063641e-03, -3.5705776467152995e-03
        },
// x = -5.6300
        {
                -3.5374771161344851e-03, -3.5629575608295883e-03, -3.5757435024247213e-03, -3.5821517539588190e-03,
                -3.5853609835750561e-03, -3.5869673022385366e-03, -3.5877710301660832e-03, -3.5881730838229842e-03,
                -3.5883741739228150e-03, -3.5884747400732812e-03
        },
// x = -5.6250
        {
                -3.5549534583031494e-03, -3.5806886019093321e-03, -3.5936025814225386e-03, -3.6000750822287857e-03,
                -3.6033165134591286e-03, -3.6049389586408740e-03, -3.6057507584109460e-03, -3.6061568508534945e-03,
                -3.6063599613017578e-03, -3.6064615379451802e-03
        },
// x = -5.6200
        {
                -3.5725148557001008e-03, -3.5985072371712422e-03, -3.6115505344340121e-03, -3.6180879278962883e-03,
                -3.6213618835928916e-03, -3.6230006171067601e-03, -3.6238205697558118e-03, -3.6242307415454752e-03,
                -3.6244358926373330e-03, -3.6245384899261071e-03
        },
// x = -5.6150
        {
                -3.5901617093990383e-03, -3.6164138928934109e-03, -3.6295878004949131e-03, -3.6361907364280952e-03,
                -3.6394975426760827e-03, -3.6411527279582661e-03, -3.6419809153358704e-03, -3.6423952074413251e-03,
                -3.6426024196757659e-03, -3.6427060478642715e-03
        },
// x = -5.6100
        {
                -3.6078944222347996e-03, -3.6344089973637012e-03, -3.6477148207774202e-03, -3.6543839554914934e-03,
                -3.6577239416412744e-03, -3.6593957437665390e-03, -3.6602322485435363e-03, -3.6606507023447593e-03,
                -3.6608599964266385e-03, -3.6609646658722645e-03
        },
// x = -5.6050
        {
                -3.6257133988097613e-03, -3.6524929808885646e-03, -3.6659320386002001e-03, -3.6726680349649884e-03,
                -3.6760415336648962e-03, -3.6777301193629906e-03, -3.6785750250397284e-03, -3.6789976823321207e-03,
                -3.6792090791742320e-03, -3.6793148003384109e-03
        },
// x = -5.6000
        {
                -3.6436190455002323e-03, -3.6706662758018991e-03, -3.6842398994385126e-03, -3.6910434269490718e-03,
                -3.6944507741783232e-03, -3.6961563118505505e-03, -3.6970097027651970e-03, -3.6974366057637505e-03,
                -3.6976501264889116e-03, -3.6977569099381638e-03
        },
// x = -5.5950
        {
                -3.6616117704628727e-03, -3.6889293164739274e-03, -3.7026388509343758e-03, -3.7095105857770216e-03,
                -3.7129521208790085e-03, -3.7146747806149649e-03, -3.7155367419519077e-03, -3.7159679332954190e-03,
                -3.7161835992385803e-03, -3.7162914556455691e-03
        },
// x = -5.5900
        {
                -3.6796919836411049e-03, -3.7072825393201251e-03, -3.7211293429067673e-03, -3.7280699680257616e-03,
                -3.7315460337416749e-03, -3.7332859873361562e-03, -3.7341566051344857e-03, -3.7345921278898065e-03,
                -3.7348099606001805e-03, -3.7349189007447794e-03
        },
// x = -5.5850
        {
                -3.6978600967715455e-03, -3.7257263828101646e-03, -3.7397118273618741e-03, -3.7467220325267774e-03,
                -3.7502329750295658e-03, -3.7519903959996347e-03, -3.7528697571617164e-03, -3.7533096548280484e-03,
                -3.7535296760712613e-03, -3.7536397108416295e-03
        },
// x = -5.5800
        {
                -3.7161165233904336e-03, -3.7442612874768943e-03, -3.7583867585033875e-03, -3.7654672403770739e-03,
                -3.7690134093057381e-03, -3.7707884729079760e-03, -3.7716766652080998e-03, -3.7721209817213317e-03,
                -3.7723432134815957e-03, -3.7724543538752674e-03
        },
// x = -5.5750
        {
                -3.7344616788400780e-03, -3.7628876959253585e-03, -3.7771545927428366e-03, -3.7843060549501905e-03,
                -3.7878878034444210e-03, -3.7896806866923398e-03, -3.7905777987854656e-03, -3.7910265785225565e-03,
                -3.7912510430048629e-03, -3.7913633001298427e-03
        },
// x = -5.5700
        {
                -3.7528959802753037e-03, -3.7816060528418419e-03, -3.7960157887099773e-03, -3.8032389419072644e-03,
                -3.8068566266424245e-03, -3.8086675083240616e-03, -3.8095736297546444e-03, -3.8100269175380475e-03,
                -3.8102536371703859e-03, -3.8103670222462598e-03
        },
// x = -5.5650
        {
                -3.7714198466699102e-03, -3.8004168050029489e-03, -3.8149708072632189e-03, -3.8222663692081488e-03,
                -3.8259203504306108e-03, -3.8277494111262906e-03, -3.8286646323371981e-03, -3.8291224734393328e-03,
                -3.8293514708749271e-03, -3.8294659952339853e-03
        },
// x = -5.5600
        {
                -3.7900336988231330e-03, -3.8193204012847096e-03, -3.8340201115000970e-03, -3.8413888071225834e-03,
                -3.8450794486854060e-03, -3.8469268707856876e-03, -3.8478512831272094e-03, -3.8483137232749724e-03,
                -3.8485450213945433e-03, -3.8486606964829081e-03
        },
// x = -5.5550
        {
                -3.8087379593661165e-03, -3.8383172926717327e-03, -3.8531641667677958e-03, -3.8606067282414150e-03,
                -3.8643343976403864e-03, -3.8662003653641853e-03, -3.8671340611031273e-03, -3.8676011464824542e-03,
                -3.8678347683965019e-03, -3.8679516057752734e-03
        },
// x = -5.5500
        {
                -3.8275330527683918e-03, -3.8574079322663732e-03, -3.8724034406737149e-03, -3.8799206074878719e-03,
                -3.8836856758979084e-03, -3.8855703753108029e-03, -3.8865134476396713e-03, -3.8869852249001436e-03,
                -3.8872211939512567e-03, -3.8873392052976657e-03
        },
// x = -5.5450
        {
                -3.8464194053443594e-03, -3.8765927752979429e-03, -3.8917384030960752e-03, -3.8993309221288922e-03,
                -3.9031337644407985e-03, -3.9050373834735139e-03, -3.9059899265198009e-03, -3.9064664427792958e-03,
                -3.9067047825444767e-03, -3.9068239796530542e-03
        },
// x = -5.5400
        {
                -3.8653974452597774e-03, -3.8958722791319514e-03, -3.9111695261945861e-03, -3.9188381517865044e-03,
                -3.9226791466440997e-03, -3.9246018751111838e-03, -3.9255639839467374e-03, -3.9260452867961228e-03,
                -3.9262860210891506e-03, -3.9264064158729007e-03
        },
// x = -5.5350
        {
                -3.8844676025382640e-03, -3.9152469032793799e-03, -3.9306972844211363e-03, -3.9384427784492611e-03,
                -3.9423223082868785e-03, -3.9442643379055597e-03, -3.9452361085560448e-03, -3.9457222460639362e-03,
                -3.9459653989377275e-03, -3.9460870034293220e-03
        },
// x = -5.5300
        {
                -3.9036303090677943e-03, -3.9347171094059876e-03, -3.9503221545305629e-03, -3.9581452864837323e-03,
                -3.9620637375640845e-03, -3.9640252619733141e-03, -3.9650067914277845e-03, -3.9654978121453233e-03,
                -3.9657434078943475e-03, -3.9658662342473274e-03
        },
// x = -5.5250
        {
                -3.9228859986072100e-03, -3.9542833613416435e-03, -3.9700446155914301e-03, -3.9779461626460308e-03,
                -3.9819039250984685e-03, -3.9838851398781667e-03, -3.9848765260987089e-03, -3.9853724790644149e-03,
                -3.9856205422271118e-03, -3.9857446027170952e-03
        },
// x = -5.5200
        {
                -3.9422351067927314e-03, -3.9739461250897050e-03, -3.9898651489968847e-03, -3.9978458960934302e-03,
                -4.0018433639525596e-03, -4.0038444666430362e-03, -4.0048458085745336e-03, -4.0053467433191874e-03,
                -4.0055972986804194e-03, -4.0057226057063338e-03
        },
// x = -5.5150
        {
                -3.9616780711444807e-03, -3.9937058688364144e-03, -4.0097842384755429e-03, -4.0178449783959983e-03,
                -4.0218825496407002e-03, -4.0239037397622841e-03, -4.0249151373422639e-03, -4.0254211038938479e-03,
                -4.0256741764873715e-03, -4.0258007425726895e-03
        },
// x = -5.5100
        {
                -3.9812153310729986e-03, -4.0135630629603456e-03, -4.0298023701024277e-03, -4.0379439035483078e-03,
                -4.0420219801411408e-03, -4.0440634592139989e-03, -4.0450850133825792e-03, -4.0455960622712665e-03,
                -4.0458516773822289e-03, -4.0459795151762210e-03
        },
// x = -5.5050
        {
                -4.0008473278857776e-03, -4.0335181800418528e-03, -4.0499200323099579e-03, -4.0581431679811977e-03,
                -4.0622621559081870e-03, -4.0643241274723411e-03, -4.0653559401822824e-03, -4.0658721224454721e-03,
                -4.0661303056129441e-03, -4.0662594278919372e-03
        },
// x = -5.5000
        {

                -4.0205745047937926e-03, -4.0535716948725958e-03, -4.0701377158989796e-03, -4.0784432705735837e-03,
                -4.0826035798844206e-03, -4.0846862495199634e-03, -4.0857284237468183e-03, -4.0862497909342297e-03,
                -4.0865105679537403e-03, -4.0866409876223996e-03
        },
// x = -5.4950
        {
                -4.0403973069180373e-03, -4.0737240844650521e-03, -4.0904559140498373e-03, -4.0988447126643311e-03,
                -4.1030467575129560e-03, -4.1051503328604728e-03, -4.1062029726128364e-03, -4.1067295767916422e-03,
                -4.1069929737177656e-03, -4.1071247038103774e-03
        },
// x = -5.4900
        {
                -4.0603161812960671e-03, -4.0939758280620978e-03, -4.1108751223335200e-03, -4.1193479980641762e-03,
                -4.1235921967497811e-03, -4.1257168875309730e-03, -4.1267800978608384e-03, -4.1273119916208582e-03,
                -4.1275780347698066e-03, -4.1277110884515881e-03
        },
// x = -5.4850
        {
                -4.0803315768885460e-03, -4.1143274071466030e-03, -4.1313958387228190e-03, -4.1399536330677146e-03,
                -4.1442404080761384e-03, -4.1463864261146524e-03, -4.1474603131278660e-03, -4.1479975495868123e-03,
                -4.1482662655390667e-03, -4.1484006561074687e-03
        },
// x = -5.4800
        {
                -4.1004439445857833e-03, -4.1347793054510567e-03, -4.1520185636035661e-03, -4.1606621264654300e-03,
                -4.1649919045109833e-03, -4.1671594637534464e-03, -4.1682441346202730e-03, -4.1687867674290420e-03,
                -4.1690581830319991e-03, -4.1691939239180470e-03
        },
// x = -5.4750
        {
                -4.1206537372143045e-03, -4.1553320089672448e-03, -4.1727437997858960e-03, -4.1814739895557904e-03,
                -4.1858472016234822e-03, -4.1880365181607556e-03, -4.1891320811265438e-03, -4.1896801644745680e-03,
                -4.1899543068452238e-03, -4.1900914116148498e-03
        },
// x = -5.4700
        {
                -4.1409614095433865e-03, -4.1759860059559324e-03, -4.1935720525155705e-03, -4.2023897361574041e-03,
                -4.2068068175455916e-03, -4.2090181096342263e-03, -4.2101246740301858e-03, -4.2106782626508306e-03,
                -4.2109551591784904e-03, -4.2110936415338928e-03
        },
// x = -5.4650
        {
                -4.1613674182916279e-03, -4.1967417869566079e-03, -4.2145038294853551e-03, -4.2234098826212174e-03,
                -4.2278712729846906e-03, -4.2301047610686024e-03, -4.2312224373226839e-03, -4.2317815864987120e-03,
                -4.2320612648477120e-03, -4.2322011386287219e-03
        },
// x = -5.4600
        {
                -4.1818722221335031e-03, -4.2175998447972346e-03, -4.2355396408464176e-03, -4.2445349478427794e-03,
                -4.2490410912362662e-03, -4.2512969979686243e-03, -4.2524258976165104e-03, -4.2529906631855888e-03,
                -4.2532731512980697e-03, -4.2534144304835330e-03
        },
// x = -5.4550
        {
                -4.2024762817059243e-03, -4.2385606746040572e-03, -4.2566799992198147e-03, -4.2657654532745693e-03,
                -4.2703167981966673e-03, -4.2725953484620072e-03, -4.2737355841582191e-03, -4.2743060225184830e-03,
                -4.2745913486171710e-03, -4.2747340473263439e-03
        },
// x = -5.4500
        {
                -4.2231800596148079e-03, -4.2596247738114195e-03, -4.2779254197080029e-03, -4.2871019229383705e-03,
                -4.2916989223759274e-03, -4.2940003433124818e-03, -4.2951520288415836e-03, -4.2957281969572651e-03,
                -4.2960163895482892e-03, -4.2961605220422491e-03
        },
// x = -5.4450
        {
                -4.2439840204416381e-03, -4.2807926421716321e-03, -4.2992764199063956e-03, -4.3085448834377016e-03,
                -4.3131879949106387e-03, -4.3155125159328863e-03, -4.3166757662208184e-03, -4.3172577216279160e-03,
                -4.3175488095036534e-03, -4.3176943901867245e-03
        },
// x = -5.4400
        {
                -4.2648886307500335e-03, -4.3020647817648677e-03, -4.3207335199149837e-03, -4.3300948639703185e-03,
                -4.3347845495768870e-03, -4.3371324023983330e-03, -4.3383073335238470e-03, -4.3388951343358703e-03,
                -4.3391891465778204e-03, -4.3393361899990062e-03
        },
// x = -5.4350
        {
                -4.2858943590923168e-03, -4.3234416970090858e-03, -4.3422972423500057e-03, -4.3517523963407610e-03,
                -4.3564891228032648e-03, -4.3588605414594467e-03, -4.3600472706656511e-03, -4.3606409755794096e-03,
                -4.3609379415610961e-03, -4.3610864624155400e-03
        },
// x = -5.4300
        {
                -4.3070016760160855e-03, -4.3449238946699888e-03, -4.3639681123556449e-03, -4.3735180149729680e-03,
                -4.3783022536839260e-03, -4.3806974745556485e-03, -4.3818961202616892e-03, -4.3824957885631403e-03,
                -4.3827957379530364e-03, -4.3829457510834871e-03
        },
// x = -5.4250
        {
                -4.3282110540707724e-03, -4.3665118838710259e-03, -4.3857466576158145e-03, -4.3953922569229401e-03,
                -4.4002244839917177e-03, -4.4026437458285226e-03, -4.4038544276413594e-03, -4.4044601192115215e-03,
                -4.4047630819760222e-03, -4.4049146023743104e-03
        },
// x = -5.4200
        {
                -4.3495229678142235e-03, -4.3882061761034066e-03, -4.4076334083659489e-03, -4.4173756618914725e-03,
                -4.4222563581913779e-03, -4.4246999021352422e-03, -4.4259227408615523e-03, -4.4265345161824746e-03,
                -4.4268405225888780e-03, -4.4269935653974178e-03
        },
// x = -5.4150
        {
                -4.3709378938192696e-03, -4.4100072852361659e-03, -4.4296288974048897e-03, -4.4394687722369456e-03,
                -4.4443984234527872e-03, -4.4468664930620615e-03, -4.4481016107202663e-03, -4.4487195308810538e-03,
                -4.4490286115005809e-03, -4.4491831920138846e-03
        },
// x = -5.4100
        {
                -4.3924563106802822e-03, -4.4319157275262535e-03, -4.4517336601067803e-03, -4.4616721329881568e-03,
                -4.4666512296642836e-03, -4.4691440709378749e-03, -4.4703915907702754e-03, -4.4710157174731842e-03,
                -4.4713279031840308e-03, -4.4714840368502299e-03
        },
// x = -5.4050
        {
                -4.4140786990197597e-03, -4.4539320216286588e-03, -4.4739482344330461e-03, -4.4839862918572499e-03,
                -4.4890153294460554e-03, -4.4915331908478434e-03, -4.4927932373328887e-03, -4.4934236328994645e-03,
                -4.4937389548898874e-03, -4.4938966573122808e-03
        },
// x = -5.4000
        {
                -4.4358055414948751e-03, -4.4760566886065702e-03, -4.4962731609443987e-03, -4.5064117992526589e-03,
                -4.5114912781635888e-03, -4.5140344106470841e-03, -4.5153071095117544e-03, -4.5159438368890577e-03,
                -4.5162623266604826e-03, -4.5164216135990896e-03
        },
// x = -5.3950
        {
                -4.4576373228040608e-03, -4.4982902519415573e-03, -4.5187089828129124e-03, -4.5289492082921509e-03,
                -4.5340796339411781e-03, -4.5366482909744311e-03, -4.5379337692067574e-03, -4.5385768919736268e-03,
                -4.5388985813437886e-03, -4.5390594687169236e-03
        },
// x = -5.3900
        {
                -4.4795745296935608e-03, -4.5206332375438037e-03, -4.5412562458341313e-03, -4.5515990748159025e-03,
                -4.5567809576755056e-03, -4.5593753952662746e-03, -4.5606737811279578e-03, -4.5613233635013522e-03,
                -4.5616482846074736e-03, -4.5618107884933430e-03
        },
// x = -5.3850
        {
                -4.5016176509639965e-03, -4.5430861737623552e-03, -4.5639154984392482e-03, -4.5743619573996577e-03,
                -4.5795958130492916e-03, -4.5822162897704386e-03, -4.5835277128096243e-03, -4.5841838196510228e-03,
                -4.5845120049530196e-03, -4.5846761415913179e-03
        },
// x = -5.3800
        {
                -4.5237671774769235e-03, -4.5656495913954072e-03, -4.5866872917073170e-03, -4.5972384173679267e-03,
                -4.6025247665449961e-03, -4.6051715435601560e-03, -4.6064961346243211e-03, -4.6071588314461863e-03,
                -4.6074903137299072e-03, -4.6076560995234406e-03
        },
// x = -5.3750
        {
                -4.5460236021613934e-03, -4.5883240237006339e-03, -4.6095721793775251e-03, -4.6202290188072604e-03,
                -4.6255683874586078e-03, -4.6282417285481003e-03, -4.6295796197970713e-03, -4.6302489727693781e-03,
                -4.6305837851498763e-03, -4.6307512366662021e-03
        },
// x = -5.3700
        {
                -4.5683874200205098e-03, -4.6111100064055366e-03, -4.6325707178615253e-03, -4.6433343285795826e-03,
                -4.6487272479134754e-03, -4.6514274195004921e-03, -4.6527787444195900e-03, -4.6534548203764192e-03,
                -4.6537929963012545e-03, -4.6539621302743375e-03
        },
// x = -5.3650
        {
                -4.5908591281379743e-03, -4.6340080777178263e-03, -4.6556834662557951e-03, -4.6665549163355828e-03,
                -4.6720019228742278e-03, -4.6747291940512602e-03, -4.6760940874645872e-03, -4.6767769539107894e-03,
                -4.6771185271633651e-03, -4.6772893604952450e-03
        },
// x = -5.3600
        {
                -4.6134392256846303e-03, -4.6570187783358565e-03, -4.6789109863540814e-03, -4.6898913545281706e-03,
                -4.6953929901607447e-03, -4.6981476327162923e-03, -4.6995262308001402e-03, -4.7002159559180624e-03,
                -4.7005609606209954e-03, -4.7007335103834763e-03
        },
// x = -5.3550
        {
                -4.6361282139250146e-03, -4.6801426514590551e-03, -4.7022538426598646e-03, -4.7133442184259930e-03,
                -4.7189010304622024e-03, -4.7216833189077404e-03, -4.7230757592041398e-03, -4.7237724118604175e-03,
                -4.7241208824789493e-03, -4.7242951659153005e-03
        },
// x = -5.3500
        {
                -4.6589265962238956e-03, -4.7033802427984273e-03, -4.7257126023988998e-03, -4.7369140861270144e-03,
                -4.7425266273511922e-03, -4.7453368389484066e-03, -4.7467432603788108e-03, -4.7474469101312323e-03,
                -4.7477988814766583e-03, -4.7479749160033420e-03
        },
// x = -5.3450
        {
                -4.6818348780527938e-03, -4.7267321005870664e-03, -4.7492878355317961e-03, -4.7606015385721588e-03,
                -4.7662703672978939e-03, -4.7691087820861938e-03, -4.7705293249652959e-03, -4.7712400420697352e-03,
                -4.7715955493028836e-03, -4.7717733525112901e-03
        },
// x = -5.3400
        {
                -4.7048535669965358e-03, -4.7501987755907017e-03, -4.7729801147666602e-03, -4.7844071595590223e-03,
                -4.7901328396843306e-03, -4.7929997405086302e-03, -4.7944345465583240e-03, -4.7951524019757387e-03,
                -4.7955114806104773e-03, -4.7956910702686819e-03
        },
// x = -5.3350
        {
                -4.7279831727597508e-03, -4.7737808211182862e-03, -4.7967900155717752e-03, -4.8083315357556299e-03,
                -4.8141146368186815e-03, -4.8170103093574607e-03, -4.8184595217209370e-03, -4.8191845871244426e-03,
                -4.8195472730312196e-03, -4.8197286670857578e-03
        },
// x = -5.3300
        {
                -4.7512242071734044e-03, -4.7974787930326057e-03, -4.8207181161883531e-03, -4.8323752567142773e-03,
                -4.8382163539496710e-03, -4.8411410867433211e-03, -4.8426048499993022e-03, -4.8433371977813153e-03,
                -4.8437035271907420e-03, -4.8438867437683979e-03
        },
// x = -5.3250
        {
                -4.7745771842013021e-03, -4.8212932497609332e-03, -4.8447649976433227e-03, -4.8565389148854292e-03,
                -4.8624385892810252e-03, -4.8653926737604681e-03, -4.8668711339375954e-03, -4.8676108372170427e-03,
                -4.8679808467235047e-03, -4.8681659041331276e-03
        },
// x = -5.3200
        {
                -4.7980426199466016e-03, -4.8452247523057042e-03, -4.8689312437621875e-03, -4.8808231056316704e-03,
                -4.8867819439859959e-03, -4.8897656745015968e-03, -4.8912589790929542e-03, -4.8920061117225591e-03,
                -4.8923798382878685e-03, -4.8925667550221916e-03
        },
// x = -5.3150
        {
                -4.8216210326583013e-03, -4.8692738642552320e-03, -4.8932174411819174e-03, -4.9052284272417370e-03,
                -4.9112470222219592e-03, -4.9142606960727205e-03, -4.9157689940505008e-03, -4.9165236306241483e-03,
                -4.9169011115812282e-03, -4.9170899063187253e-03
        },
// x = -5.3100
        {
                -4.8453129427377369e-03, -4.8934411517944539e-03, -4.9176241793639185e-03, -4.9297554809446097e-03,
                -4.9358344311450772e-03, -4.9388783486081344e-03, -4.9404017904384600e-03, -4.9411640062986210e-03,
                -4.9415452793552363e-03, -4.9417359709619767e-03
        },
// x = -5.3050
        {
                -4.8691188727450563e-03, -4.9177271837157095e-03, -4.9421520506070290e-03, -4.9544048709236604e-03,
                -4.9605447809250415e-03, -4.9636192452854394e-03, -4.9651579829433235e-03, -4.9659278541885647e-03,
                -4.9663129574310768e-03, -4.9665055649626179e-03
        },
// x = -5.3000
        {
                -4.8930393474057024e-03, -4.9421325314295455e-03, -4.9668016500605911e-03, -4.9791772043308790e-03,
                -4.9853786847598707e-03, -4.9884840023406481e-03, -4.9900381893251133e-03, -4.9908157928176839e-03,
                -4.9912047647148548e-03, -4.9913993074181357e-03
        },
// x = -5.2950
        {
                -4.9170748936168693e-03, -4.9666577689755700e-03, -4.9915735757375713e-03, -5.0040730913011528e-03,
                -5.0103367588907972e-03, -5.0134732390833686e-03, -5.0150430304327122e-03, -5.0158284438061940e-03,
                -5.0162213232130210e-03, -5.0164178205282907e-03
        },
// x = -5.2900
        {
                -4.9412260404539665e-03, -4.9913034730333112e-03, -5.0164684285277277e-03, -5.0290931449666239e-03,
                -5.0354196226172132e-03, -5.0385875779120513e-03, -5.0401731302192601e-03, -5.0409664318863106e-03,
                -5.0413632580479035e-03, -5.0415617296106609e-03
        },
// x = -5.2850
        {
                -4.9654933191770586e-03, -5.0160702229331390e-03, -5.0414868122108363e-03, -5.0542379814710985e-03,
                -5.0606278983117000e-03, -5.0638276443293250e-03, -5.0654291157576489e-03, -5.0662303849178065e-03,
                -5.0666311974733036e-03, -5.0668316631162537e-03
        },
// x = -5.2800
        {
                -4.9898772632373011e-03, -5.0409586006671964e-03, -5.0666293334699760e-03, -5.0795082199845392e-03,
                -5.0859622114351119e-03, -5.0891940669573876e-03, -5.0908116172560716e-03, -5.0916209339036530e-03,
                -5.0920257728901712e-03, -5.0922282526452128e-03
        },
// x = -5.2750
        {
                -5.0143784082833731e-03, -5.0659691909003737e-03, -5.0918966019048639e-03, -5.1049044827176033e-03,
                -5.1114231905517501e-03, -5.1146874775534982e-03, -5.1163212680736675e-03, -5.1171387130057302e-03,
                -5.1175476188623642e-03, -5.1177521329625799e-03
        },
// x = -5.2700
        {
                -5.0389972921678905e-03, -5.0911025809813118e-03, -5.1172892300452499e-03, -5.1304273949362728e-03,
                -5.1370114673446009e-03, -5.1403085110255315e-03, -5.1419587047362323e-03, -5.1427843595606287e-03,
                -5.1431973731324810e-03, -5.1434039420141624e-03
        },
// x = -5.2650
        {
                -5.0637344549538032e-03, -5.1163593609534399e-03, -5.1428078333643551e-03, -5.1560775849765310e-03,
                -5.1627276766306514e-03, -5.1660578054476001e-03, -5.1677245669520065e-03, -5.1685585140955120e-03,
                -5.1689756766377685e-03, -5.1691843209424595e-03
        },
// x = -5.2600
        {
                -5.0885904389207951e-03, -5.1417401235660356e-03, -5.1684530302923753e-03, -5.1818556842591085e-03,
                -5.1885724563762725e-03, -5.1919360020757695e-03, -5.1936194976275494e-03, -5.1944618203440774e-03,
                -5.1948831735261214e-03, -5.1950939141026742e-03
        },
// x = -5.2550
        {
                -5.1135657885716607e-03, -5.1672454642853385e-03, -5.1942254422300480e-03, -5.2077623273043125e-03,
                -5.2145464477126834e-03, -5.2179437453638445e-03, -5.2196441428836926e-03, -5.2204949252625844e-03,
                -5.2209205111721528e-03, -5.2211333690788111e-03
        },
// x = -5.2500
        {
                -5.1386610506386730e-03, -5.1928759813056639e-03, -5.2201256935622518e-03, -5.2337981517469080e-03,
                -5.2406502949514794e-03, -5.2440816829792307e-03, -5.2457991520715608e-03, -5.2466584790459639e-03,
                -5.2470883401933496e-03, -5.2473033366998477e-03
        },
// x = -5.2450
        {
                -5.1638767740899332e-03, -5.2186322755605813e-03, -5.2461544116716840e-03, -5.2599637983510722e-03,
                -5.2668846456002550e-03, -5.2703504658188731e-03, -5.2720851777886867e-03, -5.2729531351440092e-03,
                -5.2733873144662959e-03, -5.2736044710559905e-03
        },
// x = -5.2400
        {
                -5.1892135101357217e-03, -5.2445149507341039e-03, -5.2723122269525769e-03, -5.2862599110254172e-03,
                -5.2932501503782725e-03, -5.2967507480252782e-03, -5.2985028758951880e-03, -5.2993795502776575e-03,
                -5.2998180911430067e-03, -5.3000374295150091e-03
        },
// x = -5.2350
        {
                -5.2146718122348138e-03, -5.2705246132719102e-03, -5.2985997728244869e-03, -5.3126871368380877e-03,
                -5.3197474632322378e-03, -5.3232831870026103e-03, -5.3250529055300505e-03, -5.3259383844553336e-03,
                -5.3263813306673041e-03, -5.3266028727386571e-03
        },
// x = -5.2300
        {
                -5.2402522361007960e-03, -5.2966618723926124e-03, -5.3250176857461218e-03, -5.3392461260319164e-03,
                -5.3463772413521200e-03, -5.3499484434328696e-03, -5.3517359291274606e-03, -5.3526303009893868e-03,
                -5.3530776967913099e-03, -5.3533014646991711e-03
        },
// x = -5.2250
        {
                -5.2659553397083657e-03, -5.3229273400990373e-03, -5.3515666052292420e-03, -5.3659375320396557e-03,
                -5.3731401451870761e-03, -5.3767471812921500e-03, -5.3785526124332499e-03, -5.3794559665126113e-03,
                -5.3799078565920037e-03, -5.3801338726958553e-03
        },
// x = -5.2200
        {
                -5.2917816832996033e-03, -5.3493216311895518e-03, -5.3782471738526051e-03, -5.3927620114992804e-03,
                -5.4000368384614318e-03, -5.4036800678669665e-03, -5.4055036245213997e-03, -5.4064160509948453e-03,
                -5.4068724804878579e-03, -5.4071007673717397e-03
        },
// x = -5.2150
        {
                -5.3177318293902483e-03, -5.3758453632694152e-03, -5.4050600372759711e-03, -5.4197202242693525e-03,
                -5.4270679881907487e-03, -5.4307477737706809e-03, -5.4325896378106253e-03, -5.4335112277596499e-03,
                -5.4339722422555780e-03, -5.4342028227303404e-03
        },
// x = -5.2100
        {
                -5.3438063427759400e-03, -5.4024991567621595e-03, -5.4320058442541711e-03, -5.4468128334444667e-03,
                -5.4542342646979628e-03, -5.4579509729599934e-03, -5.4598113280810694e-03, -5.4607421735010748e-03,
                -5.4612078190469024e-03, -5.4614407161524877e-03
        },
// x = -5.2050
        {
                -5.3700057905384522e-03, -5.4292836349210160e-03, -5.4590852466512218e-03, -5.4740405053707612e-03,
                -5.4815363416296078e-03, -5.4852903427515254e-03, -5.4871693744910502e-03, -5.4881095683005131e-03,
                -5.4885798914055053e-03, -5.4888151284132412e-03
        },
// x = -5.2000
        {
                -5.3963307420519038e-03, -5.4561994238403440e-03, -5.4862988994544984e-03, -5.5014039096614904e-03,
                -5.5089748959721027e-03, -5.5127665638384643e-03, -5.5146644595938975e-03, -5.5156140956436270e-03,
                -5.5160891432839655e-03, -5.5163267436988935e-03
        },
// x = -5.1950
        {
                -5.4227817689889634e-03, -5.4832471524671215e-03, -5.5136474607889868e-03, -5.5289037192126911e-03,
                -5.5365506080681474e-03, -5.5403803203073233e-03, -5.5422972693548889e-03, -5.5432564424373670e-03,
                -5.5437362620608431e-03, -5.5439762496240609e-03
        },
// x = -5.1900
        {
                -5.4493594453270122e-03, -5.5104274526124484e-03, -5.5411315919315501e-03, -5.5565406102188932e-03,
                -5.5642641616331472e-03, -5.5681322996547471e-03, -5.5700684931682537e-03, -5.5710372990270800e-03,
                -5.5715219385578151e-03, -5.5717643372488520e-03
        },
// x = -5.1850
        {
                -5.4760643473543138e-03, -5.5377409589630871e-03, -5.5687519573253009e-03, -5.5843152621889228e-03,
                -5.5921162437717661e-03, -5.5960231928044290e-03, -5.5979788238742587e-03, -5.5989573592136865e-03,
                -5.5994468670569238e-03, -5.5996917010961251e-03
        },
// x = -5.1800
        {
                -5.5028970536761439e-03, -5.5651883090930285e-03, -5.5965092245939931e-03, -5.6122283579617691e-03,
                -5.6201075449945296e-03, -5.6240536941240893e-03, -5.6260289577763981e-03, -5.6270173202709640e-03,
                -5.6275117453178855e-03, -5.6277590391688387e-03
        },
// x = -5.1750
        {
                -5.5298581452209164e-03, -5.5927701434750994e-03, -5.6244040645564994e-03, -5.6402805837225184e-03,
                -5.6482387592345189e-03, -5.6522245014425541e-03, -5.6542195946586464e-03, -5.6552178829628996e-03,
                -5.6557172745955054e-03, -5.6559670529674798e-03
        },
// x = -5.1700
        {
                -5.5569482052462661e-03, -5.6204871054925856e-03, -5.6524371512413178e-03, -5.6684726290183686e-03,
                -5.6765105838641359e-03, -5.6805363160669026e-03, -5.6825514378028158e-03, -5.6835597515611349e-03,
                -5.6840641596571711e-03, -5.6843164475075814e-03
        },
// x = -5.1650
        {
                -5.5841678193451396e-03, -5.6483398414508956e-03, -5.6806091619011718e-03, -5.6968051867747173e-03,
                -5.7049237197119573e-03, -5.7089898427997079e-03, -5.7110251940059760e-03, -5.7120436338625050e-03,
                -5.7125531088004335e-03, -5.7128079313373313e-03
        },
// x = -5.1600
        {
                -5.6115175754518432e-03, -5.6763290005892595e-03, -5.7089207770276386e-03, -5.7252789533113182e-03,
                -5.7334788710796695e-03, -5.7375857899563693e-03, -5.7396415735979974e-03, -5.7406702412066543e-03,
                -5.7411848338706790e-03, -5.7414422165552740e-03
        },
// x = -5.1550
        {
                -5.6389980638480624e-03, -5.7044552350924418e-03, -5.7373726803658506e-03, -5.7538946283585083e-03,
                -5.7621767457590733e-03, -5.7663248693825058e-03, -5.7684012904591344e-03, -5.7694402884937443e-03,
                -5.7699600502788819e-03, -5.7702200188280792e-03
        },
// x = -5.1500
        {
                -5.6666098771688857e-03, -5.7327192001024934e-03, -5.7659655589292515e-03, -5.7826529150735110e-03,
                -5.7910180550491850e-03, -5.7952077964714584e-03, -5.7973050620377369e-03, -5.7983544942022498e-03,
                -5.7988794770194611e-03, -5.7991420574084254e-03
        },
// x = -5.1450
        {
                -5.6943536104087774e-03, -5.7611215537305407e-03, -5.7947001030144137e-03, -5.8115545200568292e-03,
                -5.8200035137734056e-03, -5.8242352901818669e-03, -5.8263536093680285e-03, -5.8274135804068494e-03,
                -5.8279438366882061e-03, -5.8282090551529612e-03
        },
// x = -5.1400
        {
                -5.7222298609275381e-03, -5.7896629570685894e-03, -5.8235770062159157e-03, -5.8406001533686759e-03,
                -5.8491338402967838e-03, -5.8534080730553353e-03, -5.8555476570879717e-03, -5.8566182727963880e-03,
                -5.8571538555003124e-03, -5.8574217385403545e-03
        },
// x = -5.1350
        {
                -5.7502392284562468e-03, -5.8183440742013764e-03, -5.8525969654412676e-03, -5.8697905285455187e-03,
                -5.8784097565433508e-03, -5.8827268712341842e-03, -5.8848879334572326e-03, -5.8859693006919527e-03,
                -5.8865102633084880e-03, -5.8867808376894312e-03
        },
// x = -5.1300
        {
                -5.7783823151031505e-03, -5.8471655722182247e-03, -5.8817606809259109e-03, -5.8991263626166773e-03,
                -5.9078319880135430e-03, -5.9121924144792844e-03, -5.9143751703752299e-03, -5.9154673970650143e-03,
                -5.9160137936211665e-03, -5.9162870863774183e-03
        },
// x = -5.1250
        {
                -5.8066597253595706e-03, -5.8761281212249663e-03, -5.9110688562482730e-03, -5.9286083761209965e-03,
                -5.9374012638017176e-03, -5.9418054361879916e-03, -5.9440101033992722e-03, -5.9451132985556864e-03,
                -5.9456651836208045e-03, -5.9459412220582587e-03
        },
// x = -5.1200
        {
                -5.8350720661057339e-03, -5.9052323943558464e-03, -5.9405221983448754e-03, -5.9582372931236060e-03,
                -5.9671183166137287e-03, -5.9715666734121492e-03, -5.9737934717627754e-03, -5.9749077454910446e-03,
                -5.9754651741822677e-03, -5.9757439858810326e-03
        },
// x = -5.1150
        {
                -5.8636199466166198e-03, -5.9344790677855058e-03, -5.9701214175255039e-03, -5.9880138412327383e-03,
                -5.9969838827846102e-03, -6.0014768668761988e-03, -6.0037260183935957e-03, -6.0048514819035607e-03,
                -6.0054145098913120e-03, -6.0056961227084619e-03
        },
// x = -5.1100
        {
                -5.8923039785677474e-03, -5.9638688207409567e-03, -5.9998672274884464e-03, -6.0179387516166513e-03,
                -6.0269987022963257e-03, -6.0315367609953697e-03, -6.0338084899324322e-03, -6.0349452555496282e-03,
                -6.0355139390631607e-03, -6.0357983811355174e-03
        },
// x = -5.1050
        {
                -5.9211247760409510e-03, -5.9934023355135962e-03, -6.0297603453357794e-03, -6.0480127590205835e-03,
                -6.0571635187956193e-03, -6.0617471038939465e-03, -6.0640416367513184e-03, -6.0651898179281544e-03,
                -6.0657642137611637e-03, -6.0660515135081035e-03
        },
// x = -5.1000
        {
                -5.9500829555301166e-03, -6.0230802974712665e-03, -6.0598014915887281e-03, -6.0782366017838379e-03,
                -6.0874790796119392e-03, -6.0921086474236569e-03, -6.0944262129722249e-03, -6.0955859242992783e-03,
                -6.0961660898155639e-03, -6.0964562759418554e-03
        },
// x = -5.0950
        {
                -5.9791791359468990e-03, -6.0529033950703177e-03, -6.0899913902030655e-03, -6.1086110218568921e-03,
                -6.1179461357754482e-03, -6.1226221471821149e-03, -6.1249629764857363e-03, -6.1261343337031633e-03,
                -6.1267203268423427e-03, -6.1270134283410126e-03
        },
// x = -5.0900
        {
                -6.0084139386263986e-03, -6.0828723198677152e-03, -6.1203307685845998e-03, -6.1391367648186302e-03,
                -6.1485654420351298e-03, -6.1532883625313737e-03, -6.1556526889698284e-03, -6.1568358089788859e-03,
                -6.1574276882621782e-03, -6.1577237344173979e-03
        },
// x = -5.0850
        {
                -6.0377879873328068e-03, -6.1129877665331680e-03, -6.1508203576046928e-03, -6.1698145798936175e-03,
                -6.1793377568769630e-03, -6.1841080566165706e-03, -6.1864961159087415e-03, -6.1876911167834166e-03,
                -6.1882889413194765e-03, -6.1885879617094930e-03
        },
// x = -5.0800
        {
                -6.0673019082650237e-03, -6.1432504328612922e-03, -6.1814608916158694e-03, -6.2006452199694778e-03,
                -6.2102638425422067e-03, -6.2150819963846500e-03, -6.2174940266119310e-03, -6.2187010276107089e-03,
                -6.2193048571015132e-03, -6.2196068816015963e-03
        },
// x = -5.0750
        {
                -6.0969563300622472e-03, -6.1736610197837983e-03, -6.2122531084674625e-03, -6.2316294416143257e-03,
                -6.2413444650457435e-03, -6.2462109526031876e-03, -6.2486471942331352e-03, -6.2498663158108599e-03,
                -6.2504762105576664e-03, -6.2507812693430925e-03
        },
// x = -5.0700
        {
                -6.1267518838095114e-03, -6.2042202313817053e-03, -6.2431977495213337e-03, -6.2627680050943036e-03,
                -6.2725803941945431e-03, -6.2774956998793135e-03, -6.2799563957895207e-03, -6.2811877596093933e-03,
                -6.2818037805187441e-03, -6.2821119040678075e-03
        },
// x = -5.0650
        {
                -6.1566892030432098e-03, -6.2349287748975782e-03, -6.2742955596676430e-03, -6.2940616743911723e-03,
                -6.3039724036061838e-03, -6.3089370166787015e-03, -6.3114224121809271e-03, -6.3126661411266076e-03,
                -6.3132883497164149e-03, -6.3135995688134655e-03
        },
// x = -5.0600
        {
                -6.1867689237565769e-03, -6.2657873607478085e-03, -6.3055472873406997e-03, -6.3255112172200022e-03,
                -6.3355212707274771e-03, -6.3405356853446868e-03, -6.3430460282092125e-03, -6.3443022463970551e-03,
                -6.3449307048027235e-03, -6.3452450505412433e-03
        },
// x = -5.0550
        {
                -6.2169916844051184e-03, -6.2967967025349099e-03, -6.3369536845348491e-03, -6.3571174050469272e-03,
                -6.3672277768531821e-03, -6.3722924921174465e-03, -6.3748280325976779e-03, -6.3760968653890924e-03,
                -6.3767316363697131e-03, -6.3770491401554186e-03
        },
// x = -5.0500
        {
                -6.2473581259120383e-03, -6.3279575170598302e-03, -6.3685155068204432e-03, -6.3888810131069789e-03,
                -6.3990927071448005e-03, -6.4042082271532890e-03, -6.4067692180106153e-03, -6.4080507920245286e-03,
                -6.4086919389691431e-03, -6.4090126325231176e-03
        },
// x = -5.0450
        {
                -6.2778688916735933e-03, -6.3592705243343262e-03, -6.4002335133598615e-03, -6.4208028204220304e-03,
                -6.4311168506494751e-03, -6.4362836845440431e-03, -6.4388703810729269e-03, -6.4401648241984013e-03,
                -6.4408124111323111e-03, -6.4411363264941712e-03
        },
// x = -5.0400
        {
                -6.3085246275644364e-03, -6.3907364475933202e-03, -6.4321084669235898e-03, -6.4528836098187610e-03,
                -6.4633010003189544e-03, -6.4685196623365179e-03, -6.4711323223898493e-03, -6.4724397637988037e-03,
                -6.4730938553899595e-03, -6.4734210249210505e-03
        },
// x = -5.0350
        {
                -6.3393259819428992e-03, -6.4223560133073202e-03, -6.4641411339063813e-03, -6.4851241679467712e-03,
                -6.4956459530286781e-03, -6.5009169625520899e-03, -6.5035558465667857e-03, -6.5048764167268533e-03,
                -6.5055370782922967e-03, -6.5058675346789299e-03
        },
// x = -5.0300
        {
                -6.3702736056562553e-03, -6.4541299511948454e-03, -6.4963322843434477e-03, -6.5175252852967226e-03,
                -6.5281525095969217e-03, -6.5334763912063597e-03, -6.5361417622292204e-03, -6.5374755929167331e-03,
                -6.5381428904291144e-03, -6.5384766666858156e-03
        },
// x = -5.0250
        {
                -6.4013681520459272e-03, -6.4860589942348831e-03, -6.5286826919267469e-03, -6.5500877562185899e-03,
                -6.5608214748040595e-03, -6.5661987583289141e-03, -6.5688908820427487e-03, -6.5702381063558503e-03,
                -6.5709121064499972e-03, -6.5712492359228120e-03
        },
// x = -5.0200
        {
                -6.4326102769526623e-03, -6.5181438786793711e-03, -6.5611931340213000e-03, -6.5828123789399813e-03,
                -6.5936536574119015e-03, -6.5990848779831886e-03, -6.6018040227331818e-03, -6.6031647751050738e-03,
                -6.6038455450846403e-03, -6.6041860614544541e-03
        },
// x = -5.0150
        {
                -6.4640006387216599e-03, -6.5503853440657149e-03, -6.5938643916816026e-03, -6.6156999555845499e-03,
                -6.6266498701831236e-03, -6.6321355682864257e-03, -6.6348820051067795e-03, -6.6362564213191021e-03,
                -6.6369440291632713e-03, -6.6372879664491789e-03
        },
// x = -5.0100
        {
                -6.4955398982076600e-03, -6.5827841332293186e-03, -6.6266972496680770e-03, -6.6487512921904780e-03,
                -6.6598109299008069e-03, -6.6653516514297165e-03, -6.6681256540705620e-03, -6.6695138712669052e-03,
                -6.6702083856371684e-03, -6.6705557781998595e-03
        },
// x = -5.0050
        {
                -6.5272287187799890e-03, -6.6153409923161389e-03, -6.6596924964635934e-03, -6.6819671987290476e-03,
                -6.6931376573880407e-03, -6.6987339536981666e-03, -6.7015357986527323e-03, -6.7029379553522851e-03,
                -6.7036394455992831e-03, -6.7039903281444768e-03
        },
// x = -5.0000
        {
                -6.5590677663275527e-03, -6.6480566707952833e-03, -6.6928509242900518e-03, -6.7153484891233003e-03,
                -6.7266308775276468e-03, -6.7322833054911362e-03, -6.7351132720231991e-03, -6.7365295081345264e-03,
                -6.7372380443049655e-03, -6.7375924518868697e-03
        },
// x = -4.9950
        {
                -6.5910577092637953e-03, -6.6809319214715997e-03, -6.7261733291250330e-03, -6.7488959812667644e-03,
                -6.7602914192819779e-03, -6.7660005413425839e-03, -6.7688589115141879e-03, -6.7702893683491633e-03,
                -6.7710050211927933e-03, -6.7713629892176067e-03
        },
// x = -4.9900
        {
                -6.6231992185316084e-03, -6.7139675004983411e-03, -6.7596605107185098e-03, -6.7826104970422803e-03,
                -6.7941201157128229e-03, -6.7998864999415307e-03, -6.8027735586409831e-03, -6.8042183789288379e-03,
                -6.8049412199055071e-03, -6.8053027841349480e-03
        },
// x = -4.9850
        {
                -6.6554929676081905e-03, -6.7471641673898007e-03, -6.7933132726096133e-03, -6.8164928623409109e-03,
                -6.8281178040013940e-03, -6.8339420241525888e-03, -6.8368580591227357e-03, -6.8383173870242737e-03,
                -6.8390474883110487e-03, -6.8394126848659248e-03
        },
// x = -4.9800
        {
                -6.6879396325098504e-03, -6.7805226850339998e-03, -6.8271324221434830e-03, -6.8505439070809127e-03,
                -6.8622853254684197e-03, -6.8681679610366215e-03, -6.8711132629034064e-03, -6.8725872440253384e-03,
                -6.8733246785237000e-03, -6.8736935438875116e-03
        },
// x = -4.9750
        {
                -6.7205398917967903e-03, -6.8140438197054091e-03, -6.8611187704881536e-03, -6.8847644652268274e-03,
                -6.8966235255943304e-03, -6.9025651618714835e-03, -6.9055400241727948e-03, -6.9070288055822293e-03,
                -6.9077736469253343e-03, -6.9081462179479153e-03
        },
// x = -4.9700
        {
                -6.7532944265778026e-03, -6.8477283410776609e-03, -6.8952731326515273e-03, -6.9191553748086214e-03,
                -6.9311332540395315e-03, -6.9371344821728756e-03, -6.9401392013876727e-03, -6.9416429316267476e-03,
                -6.9423952541867746e-03, -6.9427715680879694e-03
        },
// x = -4.9650
        {
                -6.7862039205149431e-03, -6.8815770222363159e-03, -6.9295963274983961e-03, -6.9537174779409445e-03,
                -6.9658153646647838e-03, -6.9718767817152890e-03, -6.9749116572930311e-03, -6.9764304863936949e-03,
                -6.9771903652892436e-03, -6.9775704596626220e-03
        },
// x = -4.9600

        {
                -6.8192690598281435e-03, -6.9155906396916392e-03, -6.9640891777675374e-03, -6.9884516208424379e-03,
                -7.0006707155516714e-03, -7.0067929245530673e-03, -7.0098582589434230e-03, -7.0113923384423566e-03,
                -7.0121598495459488e-03, -7.0125437623625537e-03
        },
// x = -4.9550
        {
                -6.8524905332997699e-03, -6.9497699733913959e-03, -6.9987525100888709e-03, -7.0233586538551666e-03,
                -7.0357001690231728e-03, -7.0418837790415540e-03, -7.0449798777244173e-03, -7.0465293606781160e-03,
                -7.0473045806237469e-03, -7.0476923502358849e-03
        },
// x = -4.9500
        {
                -6.8858690322791249e-03, -6.9841158067336833e-03, -7.0335871550006755e-03, -7.0584394314641028e-03,
                -7.0709045916643306e-03, -7.0771502178583563e-03, -7.0802773893741565e-03, -7.0818424303741500e-03,
                -7.0826254365649322e-03, -7.0830171017099968e-03
        },
// x = -4.9450
        {
                -6.9194052506869131e-03, -7.0186289265797685e-03, -7.0685939469668649e-03, -7.0936948123167236e-03,
                -7.1062848543429992e-03, -7.1125931180246972e-03, -7.1157516740050194e-03, -7.1173324291932604e-03,
                -7.1181232998091268e-03, -7.1185188996134713e-03
        },
// x = -4.9400
        {
                -6.9530998850196252e-03, -7.0533101232669627e-03, -7.1037737243943569e-03, -7.1291256592426640e-03,
                -7.1418418322307308e-03, -7.1482133609269026e-03, -7.1514036161253970e-03, -7.1530002432097818e-03,
                -7.1537990572152913e-03, -7.1541986311981220e-03
        },
// x = -4.9350
        {
                -6.9869536343538895e-03, -7.0881601906215017e-03, -7.1391273296504644e-03, -7.1647328392735040e-03,
                -7.1775764048237116e-03, -7.1840118323379509e-03, -7.1872341046615692e-03, -7.1888467629316385e-03,
                -7.1896536000838306e-03, -7.1900571881611574e-03
        },
// x = -4.9300
        {
                -7.0209672003507511e-03, -7.1231799259714708e-03, -7.1746556090803810e-03, -7.2005172236625824e-03,
                -7.2134894559638379e-03, -7.2199894224391544e-03, -7.2232440329796912e-03, -7.2248728833224640e-03,
                -7.2256878241788178e-03, -7.2260954666674331e-03
        },
// x = -4.9250
        {
                -7.0551412872599060e-03, -7.1583701301597309e-03, -7.2103594130247251e-03, -7.2364796879049540e-03,
                -7.2495818738598670e-03, -7.2561470258419511e-03, -7.2594342989078945e-03, -7.2610795038238803e-03,
                -7.2619026297503332e-03, -7.2623143673718332e-03
        },
// x = -4.9200
        {
                -7.0894766019238553e-03, -7.1937316075568625e-03, -7.2462395958371416e-03, -7.2726211117573976e-03,
                -7.2858545511086791e-03, -7.2924855416097740e-03, -7.2958058047584828e-03, -7.2974675283778425e-03,
                -7.2982989215569045e-03, -7.2987147954417554e-03
        },
// x = -4.9150
        {
                -7.1239738537820264e-03, -7.2292651660741591e-03, -7.2822970159019743e-03, -7.3089423792585314e-03,
                -7.3223083847166279e-03, -7.3290058732800676e-03, -7.3323594573502576e-03, -7.3340378654491271e-03,
                -7.3348776088880659e-03, -7.3352976605797022e-03
        },
// x = -4.9100
        {
                -7.1586337548748174e-03, -7.2649716171766061e-03, -7.3185325356520045e-03, -7.3454443787490118e-03,
                -7.3589442761210168e-03, -7.3657089288863657e-03, -7.3690961680309376e-03, -7.3707914280479181e-03,
                -7.3716396055870329e-03, -7.3720638770460060e-03
        },
// x = -4.9050
        {
                -7.1934570198475701e-03, -7.3008517758959062e-03, -7.3549470215862490e-03, -7.3821280028918100e-03,
                -7.3957631312116419e-03, -7.4025956209805175e-03, -7.4060168526996935e-03, -7.4077291337525065e-03,
                -7.4085858300734792e-03, -7.4090143636816415e-03
        },
// x = -4.9000
        {
                -7.2284443659545120e-03, -7.3369064608435055e-03, -7.3915413442878219e-03, -7.4189941486926024e-03,
                -7.4327658603524673e-03, -7.4396668666549936e-03, -7.4431224318298001e-03, -7.4448519047321067e-03,
                -7.4457172053664384e-03, -7.4461500439311733e-03
        },
// x = -4.8950
        {
                -7.2635965130625956e-03, -7.3731364942236522e-03, -7.4283163784418810e-03, -7.4560437175202263e-03,
                -7.4699533784033795e-03, -7.4769235875653192e-03, -7.4804138304913935e-03, -7.4821606677697786e-03,
                -7.4830346591073169e-03, -7.4834718458658047e-03
        },
// x = -4.8900
        {
                -7.2989141836553011e-03, -7.4095427018464714e-03, -7.4652730028536077e-03, -7.4932776151272326e-03,
                -7.5073266047420697e-03, -7.5143667099526017e-03, -7.5178919783743386e-03, -7.5196563542854773e-03,
                -7.5205391235830194e-03, -7.5209807022065559e-03
        },
// x = -4.8850
        {
                -7.3343981028363625e-03, -7.4461259131410511e-03, -7.5024121004662750e-03, -7.5306967516705426e-03,
                -7.5448864632859788e-03, -7.5519971646661735e-03, -7.5555578098112182e-03, -7.5573399003591975e-03,
                -7.5582315357491867e-03, -7.5586775503475381e-03
        },
// x = -4.8800
        {
                -7.3700489983334243e-03, -7.4828869611685608e-03, -7.5397345583793811e-03, -7.5683020417321766e-03,
                -7.5826338825144028e-03, -7.5898158871863589e-03, -7.5934122638004302e-03, -7.5952122467542607e-03,
                -7.5961128372535628e-03, -7.5965633323793616e-03
        },
// x = -4.8750
        {
                -7.4058676005016285e-03, -7.5198266826353695e-03, -7.5772412678668348e-03, -7.6060944043400726e-03,
                -7.6205697954906428e-03, -7.6278238176473202e-03, -7.6314562840294016e-03, -7.6332743389406852e-03,
                -7.6341839744594638e-03, -7.6346389951126581e-03
        },
// x = -4.8700
        {
                -7.4418546423271519e-03, -7.5569459179062048e-03, -7.6149331243952172e-03, -7.6440747629890242e-03,
                -7.6586951398843013e-03, -7.6660219008600492e-03, -7.6696908188979077e-03, -7.6715271271187036e-03,
                -7.6724458984693709e-03, -7.6729054901017108e-03
        },
// x = -4.8650
        {
                -7.4780108594306336e-03, -7.5942455110173043e-03, -7.6528110276421046e-03, -7.6822440456616658e-03,
                -7.6970108579936664e-03, -7.7044110863354410e-03, -7.7081168215415215e-03, -7.7099715662423702e-03,
                -7.7108995651486408e-03, -7.7113637736682134e-03
        },
// x = -4.8600
        {
                -7.5143369900705819e-03, -7.6317263096896040e-03, -7.6908758815144654e-03, -7.7206031848495969e-03,
                -7.7355178967682039e-03, -7.7429923283075137e-03, -7.7467352498551724e-03, -7.7486086160433150e-03,
                -7.7495459351493365e-03, -7.7500148069251451e-03
        },
// x = -4.8550
        {
                -7.5508337751466542e-03, -7.6693891653419350e-03, -7.7291285941671070e-03, -7.7591531175745547e-03,
                -7.7742172078311507e-03, -7.7817665857566895e-03, -7.7855470665168071e-03, -7.7874392410545810e-03,
                -7.7883859739341678e-03, -7.7888595558007574e-03
        },
// x = -4.8500
        {
                -7.5875019582029146e-03, -7.7072349331042357e-03, -7.7675700780212055e-03, -7.7978947854097118e-03,
                -7.8131097475022237e-03, -7.8207348224332545e-03, -7.8245532390111960e-03, -7.8264644106346148e-03,
                -7.8274206518005734e-03, -7.8278989910626987e-03
        },
// x = -4.8450
        {
                -7.6243422854309724e-03, -7.7452644718307810e-03, -7.8062012497828936e-03, -7.8368291345010509e-03,
                -7.8521964768204326e-03, -7.8598980068808715e-03, -7.8637547396538336e-03, -7.8656850989913488e-03,
                -7.8666509439048824e-03, -7.8671340883422342e-03
        },
// x = -4.8400
        {
                -7.6613555056730762e-03, -7.7834786441134335e-03, -7.8450230304619104e-03, -7.8759571155888400e-03,
                -7.8914783615669868e-03, -7.8992571124602545e-03, -7.9031525456149665e-03, -7.9051022852064234e-03,
                -7.9060778302866546e-03, -7.9065658281586137e-03
        },
// x = -4.8350
        {
                -7.6985423704251160e-03, -7.8218783162948974e-03, -7.8840363453903314e-03, -7.9152796840291917e-03,
                -7.9309563722883298e-03, -7.9388131173729129e-03, -7.9427476389437347e-03, -7.9447169532595192e-03,
                -7.9457022958930850e-03, -7.9461951959435458e-03
        },
// x = -4.8300
        {
                -7.7359036338395413e-03, -7.8604643584819864e-03, -7.9232421242413364e-03, -7.9547977998157202e-03,
                -7.9706314843192745e-03, -7.9785670046850637e-03, -7.9825410065924359e-03, -7.9845300920528012e-03,
                -7.9855253306035552e-03, -7.9860231820657913e-03
        },
// x = -4.8250
        {
                -7.7734400527282249e-03, -7.8992376445589443e-03, -7.9626413010481008e-03, -7.9945124276012969e-03,
                -8.0105046778062364e-03, -8.0185197623516089e-03, -8.0225336404409174e-03, -8.0245426954355138e-03,
                -8.0255479292543289e-03, -8.0260507818558837e-03
        },
// x = -4.8200
        {
                -7.8111523865652130e-03, -7.9381990522007142e-03, -8.0022348142226706e-03, -8.0344245367199134e-03,
                -8.0505769377305948e-03, -8.0586723832402670e-03, -8.0627265373210619e-03, -8.0647557622286672e-03,
                -8.0657710916633308e-03, -8.0662789956309992e-03
        },
// x = -4.8150
        {
                -7.8490413974894122e-03, -7.9773494628862726e-03, -8.0420236065749818e-03, -8.0745351012085926e-03,
                -8.0908492539321364e-03, -8.0990258651557866e-03, -8.1031206990414360e-03, -8.1051702962498598e-03,
                -8.1061958226550639e-03, -8.1067088287198885e-03
        },
// x = -4.8100
        {
                -7.8871078503071953e-03, -8.0166897619119593e-03, -8.0820086253318990e-03, -8.1148450998294371e-03,
                -8.1313226211326486e-03, -8.1395812108643276e-03, -8.1437171324120119e-03, -8.1457873063382289e-03,
                -8.1468231320856715e-03, -8.1473412914880097e-03
        },
// x = -4.8050
        {
                -7.9253525124949112e-03, -8.0562208384048258e-03, -8.1221908221563426e-03, -8.1553555160917869e-03,
                -8.1719980389595801e-03, -8.1803394281178953e-03, -8.1845168492690554e-03, -8.1866078063795224e-03,
                -8.1876540348680988e-03, -8.1881773993627300e-03
        },
// x = -4.8000
        {
                -7.9637761542013189e-03, -8.0959435853359702e-03, -8.1625711531664782e-03, -8.1960673382744081e-03,
                -8.2128765119698476e-03, -8.2213015296789625e-03, -8.2255208665001026e-03, -8.2276328153312751e-03,
                -8.2286895509973840e-03, -8.2292181728586683e-03
        },
// x = -4.7950
        {
                -8.0023795482499216e-03, -8.1358588995339379e-03, -8.2031505789549441e-03, -8.2369815594478413e-03,
                -8.2539590496737217e-03, -8.2624685333451530e-03, -8.2667302060690836e-03, -8.2688633572481473e-03,
                -8.2699307055760802e-03, -8.2704646376031785e-03
        },
// x = -4.7900
        {
                -8.0411634701412389e-03, -8.1759676816980723e-03, -8.2439300646082233e-03, -8.2780991774968286e-03,
                -8.2952466665588611e-03, -8.3038414619740729e-03, -8.3081458950415576e-03, -8.3103004613073669e-03,
                -8.3113785288398290e-03, -8.3119178243619626e-03
        },
// x = -4.7850
        {
                -8.0801286980549513e-03, -8.2162708364119184e-03, -8.2849105797259733e-03, -8.3194211951427930e-03,
                -8.3367403821144274e-03, -8.3454213435082622e-03, -8.3497689656100629e-03, -8.3519451618342851e-03,
                -8.3530340561829945e-03, -8.3535787690647689e-03
        },
// x = -4.7800
        {
                -8.1192760128519868e-03, -8.2567692721566242e-03, -8.3260930984405228e-03, -8.3609486199665087e-03,
                -8.3784412208553309e-03, -8.3872092110002568e-03, -8.3916004551196285e-03, -8.3937984983280950e-03,
                -8.3948983281845069e-03, -8.3954485128312807e-03
        },
// x = -4.7750
        {
                -8.1586061980764932e-03, -8.2974639013243465e-03, -8.3674785994363714e-03, -8.4026824644307628e-03,
                -8.4203502123465692e-03, -8.4292061026377957e-03, -8.4336414060933694e-03, -8.4358615154876450e-03,
                -8.4369723906337745e-03, -8.4375281019970841e-03
        },
// x = -4.7700
        {
                -8.1981200399577365e-03, -8.3383556402316806e-03, -8.4090680659697926e-03, -8.4446237459031952e-03,
                -8.4624683912277211e-03, -8.4714130617690991e-03, -8.4758928662582202e-03, -8.4781352632374001e-03,
                -8.4792572945567527e-03, -8.4798185881397922e-03
        },
// x = -4.7650
        {
                -8.2378183274118919e-03, -8.3794454091330774e-03, -8.4508624858884735e-03, -8.4867734866791895e-03,
                -8.5047967972375106e-03, -8.5138311369283484e-03, -8.5183558885708315e-03, -8.5206207967535291e-03,
                -8.5217540962421439e-03, -8.5223210281053074e-03
        },
// x = -4.7600
        {
                -8.2777018520437239e-03, -8.4207341322342869e-03, -8.4928628516512481e-03, -8.5291327140049090e-03,
                -8.5473364752385005e-03, -8.5564613818611898e-03, -8.5610315312435164e-03, -8.5633191764900984e-03,
                -8.5644638572677078e-03, -8.5650364840341667e-03
        },
// x = -4.7550
        {
                -8.3177714081482095e-03, -8.4622227377058323e-03, -8.5350701603478935e-03, -8.5717024601003524e-03,
                -8.5900884752419127e-03, -8.5993048555504641e-03, -8.6039208577704217e-03, -8.6062314682054386e-03,
                -8.6073876445267282e-03, -8.6079660233880917e-03
        },
// x = -4.7500
        {
                -8.3580277927120285e-03, -8.5039121576964017e-03, -8.5774854137189646e-03, -8.6144837621825992e-03,
                -8.6330538524325498e-03, -8.6423626222419791e-03, -8.6470249369537434e-03, -8.6493587429886127e-03,
                -8.6505265302545924e-03, -8.6511107189765971e-03
        },
// x = -4.7450
        {
                -8.3984718054149494e-03, -8.5458033283463790e-03, -8.6201096181757458e-03, -8.6574776624891020e-03,
                -8.6762336671938455e-03, -8.6856357514704646e-03, -8.6903448429301224e-03, -8.6927020772860127e-03,
                -8.6938815920555197e-03, -8.6944716489838047e-03
        },
// x = -4.7400
        {
                -8.4391042486311445e-03, -8.5878971898012708e-03, -8.6629437848202238e-03, -8.7006852083010861e-03,
                -8.7196289851330152e-03, -8.7291253180856055e-03, -8.7338816551971437e-03, -8.7362625529281129e-03,
                -8.7374539129294049e-03, -8.7380498969953195e-03
        },
// x = -4.7350
        {
                -8.4799259274303800e-03, -8.6301946862251937e-03, -8.7059889294651641e-03, -8.7441074519670550e-03,
                -8.7632408771063409e-03, -8.7728324022782380e-03, -8.7776364586399872e-03, -8.7800412571563246e-03,
                -8.7812445812988204e-03, -8.7818465520253016e-03
        },
// x = -4.7300
        {
                -8.5209376495791102e-03, -8.6726967658143535e-03, -8.7492460726542095e-03, -8.7877454509263948e-03,
                -8.8070704192445715e-03, -8.8167580896066598e-03, -8.8216103435581998e-03, -8.8240392826500168e-03,
                -8.8252546910361218e-03, -8.8258627085436320e-03
        },
// x = -4.7250
        {
                -8.5621402255414356e-03, -8.7154043808105354e-03, -8.7927162396821065e-03, -8.8316002677330790e-03,
                -8.8511186929784141e-03, -8.8609034710230607e-03, -8.8658044056925799e-03, -8.8682577275536432e-03,
                -8.8694853414907140e-03, -8.8700994665032330e-03
        },
// x = -4.7200
        {
                -8.6035344684800234e-03, -8.7583184875145922e-03, -8.8364004606149484e-03, -8.8756729700794710e-03,
                -8.8953867850641959e-03, -8.9052696429000825e-03, -8.9102197462522445e-03, -8.9126976955040248e-03,
                -8.9139376375164444e-03, -8.9145579313675317e-03
        },
// x = -4.7150
        {
                -8.6451211942568124e-03, -8.8014400462999298e-03, -8.8802997703105058e-03, -8.9199646308202440e-03,
                -8.9398757876095903e-03, -8.9498577070575170e-03, -8.9548574719417601e-03, -8.9573602956577562e-03,
                -8.9586126894991309e-03, -8.9592392141380310e-03
        },
// x = -4.7100
        {
                -8.6869012214337397e-03, -8.8447700216260222e-03, -8.9244152084386325e-03, -8.9644763279963756e-03,
                -8.9845867980995128e-03, -8.9946687707891312e-03, -8.9997186949884837e-03, -9.0022466427187430e-03,
                -9.0035116133842345e-03, -9.0041444313820624e-03
        },
// x = -4.7050
        {
                -8.7288753712732349e-03, -8.8883093820519045e-03, -8.9687478195017174e-03, -9.0092091448592659e-03,
                -9.0295209194220898e-03, -9.0397039468896098e-03, -9.0448045331699735e-03, -9.0473578569659017e-03,
                -9.0486355307046670e-03, -9.0492747052606324e-03
        },
// x = -4.7000
        {
                -8.7710444677386572e-03, -8.9320591002496636e-03, -9.0132986528552242e-03, -9.0541641698949557e-03,
                -9.0746792598947832e-03, -9.0849643536816457e-03, -9.0901161098415721e-03, -9.0926950642809656e-03,
                -9.0939855686087111e-03, -9.0946311635564502e-03
        },
// x = -4.6950
        {
                -8.8134093374946272e-03, -8.9760201530179710e-03, -9.0580687627282823e-03, -9.0993424968484234e-03,
                -9.1200629332906236e-03, -9.1304511150431412e-03, -9.1356545539641072e-03, -9.1382593961764364e-03,
                -9.1395628598881402e-03, -9.1402149397020460e-03
        },
// x = -4.6900
        {
                -8.8559708099072154e-03, -9.0201935212955633e-03, -9.1030592082443681e-03, -9.1447452247480493e-03,
                -9.1656730588645652e-03, -9.1761653604345590e-03, -9.1814210001317544e-03, -9.1840519898237112e-03,
                -9.1853685430064273e-03, -9.1860271728080936e-03
        },
// x = -4.6850
        {
                -8.8987297170440207e-03, -9.0645801901747779e-03, -9.1482710534420263e-03, -9.1903734579300810e-03,
                -9.2115107613799600e-03, -9.2221082249263984e-03, -9.2274165885999886e-03, -9.2300739880812888e-03,
                -9.2314037621270915e-03, -9.2320690076918115e-03
        },
// x = -4.6800
        {
                -8.9416868936741320e-03, -9.1091811489150271e-03, -9.1937053672956660e-03, -9.2362283060633005e-03,
                -9.2575771711351645e-03, -9.2682808492267951e-03, -9.2736424653137306e-03, -9.2763265395231515e-03,
                -9.2776696671422413e-03, -9.2783415949055491e-03
        },
// x = -4.6750
        {
                -8.9848431772679618e-03, -9.1539973909563234e-03, -9.2393632237364586e-03, -9.2823108841737535e-03,
                -9.3038734239902672e-03, -9.3146843797092867e-03, -9.3200997819355910e-03, -9.3228107984673000e-03,
                -9.3241674137011909e-03, -9.3248460907655005e-03
        },
// x = -4.6700
        {
                -9.0281994079969806e-03, -9.1990299139327932e-03, -9.2852457016732379e-03, -9.3286223126695638e-03,
                -9.3504006613939258e-03, -9.3613199684406487e-03, -9.3667896958742700e-03, -9.3695279250043985e-03,
                -9.3708981632392614e-03, -9.3715836573805521e-03
        },
// x = -4.6650
        {
                -9.0717564287332783e-03, -9.2442797196861711e-03, -9.3313538850135266e-03, -9.3751637173658964e-03,
                -9.3971600304103460e-03, -9.4081887732089479e-03, -9.4137133703130899e-03, -9.4164790850265676e-03,
                -9.4178630830067320e-03, -9.4185554626812981e-03
        },
// x = -4.6600
        {
                -9.1155150850490609e-03, -9.2897478142792925e-03, -9.3776888626846116e-03, -9.4219362295099845e-03,
                -9.4441526837463746e-03, -9.4552919575516502e-03, -9.4608719742386767e-03, -9.4636654502563630e-03,
                -9.4650633460978961e-03, -9.4657626804491805e-03
        },
// x = -4.6550
        {
                -9.1594762252159567e-03, -9.3354352080096108e-03, -9.4242517286546607e-03, -9.4689409858063050e-03,
                -9.4913797797787393e-03, -9.5026306907839350e-03, -9.5082666824697817e-03, -9.5110881982758202e-03,
                -9.5125001314803007e-03, -9.5132064903457811e-03
        },
// x = -4.6500
        {
                -9.2036407002042409e-03, -9.3813429154226847e-03, -9.4710435819539557e-03, -9.5161791284418124e-03,
                -9.5388424825813799e-03, -9.5502061480270874e-03, -9.5558986756862276e-03, -9.5587485125557241e-03,
                -9.5601746240241096e-03, -9.5608880779422707e-03
        },
// x = -4.6450
        {
                -9.2480093636819090e-03, -9.4274719553256673e-03, -9.5180655266961494e-03, -9.5636518051113138e-03,
                -9.5865419619529241e-03, -9.5980195102370620e-03, -9.6037691404580201e-03, -9.6066475824849624e-03,
                -9.6080880145316232e-03, -9.6088086347489790e-03
        },
// x = -4.6400
        {
                -9.2925830720136156e-03, -9.4738233508007922e-03, -9.5653186720996083e-03, -9.6113601690429387e-03,
                -9.6344793934443011e-03, -9.6460719642331768e-03, -9.6518792692745926e-03, -9.6547866034000611e-03,
                -9.6562414997669512e-03, -9.6569693582451467e-03
        },
// x = -4.6350
        {
                -9.3373626842594912e-03, -9.5203981292188589e-03, -9.6128041325088381e-03, -9.6593053790237134e-03,
                -9.6826559583864727e-03, -9.6943647027269483e-03, -9.7002302605741983e-03, -9.7031667766148619e-03,
                -9.7046362824858064e-03, -9.7053714519088029e-03
        },
// x = -4.6300
        {
                -9.3823490621737903e-03, -9.5671973222526963e-03, -9.6605230274159314e-03, -9.7074885994252510e-03,
                -9.7310728439182793e-03, -9.7428989243510487e-03, -9.7488233187734332e-03, -9.7517893094503181e-03,
                -9.7532735714654806e-03, -9.7540161252467877e-03
        },
// x = -4.6250
        {
                -9.4275430702034594e-03, -9.6142219658906531e-03, -9.7084764814821447e-03, -9.7559110002295342e-03,
                -9.7797312430144313e-03, -9.7916758336884287e-03, -9.7976596542969172e-03, -9.8006554152644827e-03,
                -9.8021545815349526e-03, -9.8029045938249505e-03
        },
// x = -4.6200
        {
                -9.4729455754864900e-03, -9.6614731004500218e-03, -9.7566656245594834e-03, -9.8045737570548186e-03,
                -9.8286323545136410e-03, -9.8406966413015495e-03, -9.8467404836071228e-03, -9.8497663134826090e-03,
                -9.8512805336051432e-03, -9.8520380792984780e-03
        },
// x = -4.6150
        {
                -9.5185574478502052e-03, -9.7089517705905171e-03, -9.8050915917123858e-03, -9.8534780511816463e-03,
                -9.8777773831468484e-03, -9.8899625637618067e-03, -9.8960670292343542e-03, -9.8991232296274257e-03,
                -9.9006526546993288e-03, -9.9014178094423821e-03
        },
// x = -4.6100
        {
                -9.5643795598093426e-03, -9.7566590253277130e-03, -9.8537555232394870e-03, -9.9026250695789537e-03,
                -9.9271675395656081e-03, -9.9394748236790124e-03, -9.9456405198068490e-03, -9.9487273953495371e-03,
                -9.9502721779837176e-03, -9.9510450181821281e-03
        },
// x = -4.6050
        {
                -9.6104127865640297e-03, -9.8045959180464680e-03, -9.9026585646954032e-03, -9.9520160049303089e-03,
                -9.9768040403705929e-03, -9.9892346497311227e-03, -9.9954621900810465e-03, -9.9985800484580032e-03,
                -1.0000140342798142e-02, -1.0000920945624440e-02
        },
// x = -4.6000
        {
                -9.6566580059975985e-03, -9.8527635065143518e-03, -9.9518018669126421e-03, -1.0001652055660231e-02,
                -1.0026688108140237e-02, -1.0039243276694009e-02, -1.0045533280972019e-02, -1.0048682432951050e-02,
                -1.0050258394686944e-02, -1.0051046838088237e-02
        },
// x = -4.5950
        {
                -9.7031160986742332e-03, -9.9011628528950513e-03, -1.0001186586023506e-02, -1.0051534425960644e-02,
                -1.0076820971459507e-02, -1.0089501945471454e-02, -1.0095855039584017e-02, -1.0099035799046927e-02,
                -1.0100627585429992e-02, -1.0101423948135735e-02
        },
// x = -4.5900
        {
                -9.7497879478365204e-03, -9.9497950237617712e-03, -1.0050813883482170e-02, -1.0101664325817432e-02,
                -1.0127203864948796e-02, -1.0140011903125235e-02, -1.0146428719241200e-02, -1.0149641403214964e-02,
                -1.0151249173073864e-02, -1.0152053534603704e-02
        },
// x = -4.5850
        {
                -9.7966744394027642e-03, -9.9986610901106211e-03, -1.0100684926086699e-02, -1.0152042971037094e-02,
                -1.0177838029292976e-02, -1.0190774402905377e-02, -1.0197255579518484e-02, -1.0200500508206707e-02,
                -1.0202124421963180e-02, -1.0202936862634881e-02
        },
// x = -4.5800
        {
                -9.8437764619642380e-03, -1.0047762127373993e-02, -1.0150800886001236e-02, -1.0202671583273534e-02,
                -1.0228724711270551e-02, -1.0241790704280562e-02, -1.0248336886272580e-02, -1.0251614383087278e-02,
                -1.0253254602772079e-02, -1.0254075203709533e-02
        },
// x = -4.5750
        {
                -9.8910949067822027e-03, -1.0097099215433896e-02, -1.0201162940778202e-02, -1.0253551390054927e-02,
                -1.0279865163782976e-02, -1.0293062072968652e-02, -1.0299673911673149e-02, -1.0302984303266842e-02,
                -1.0304640992535880e-02, -1.0305469835677188e-02
        },
// x = -4.5700
        {
                -9.9386306677848226e-03, -1.0146673438635319e-02, -1.0251772273380574e-02, -1.0304683624810735e-02,
                -1.0331260645884077e-02, -1.0344589780967377e-02, -1.0351267934234116e-02, -1.0354611550532270e-02,
                -1.0356284874682882e-02, -1.0357122042788521e-02
        },
// x = -4.5650
        {
                -9.9863846415638793e-03, -1.0196485885799569e-02, -1.0302630072204258e-02, -1.0356069526898815e-02,
                -1.0382912422809658e-02, -1.0396375106585202e-02, -1.0403120238845183e-02, -1.0406497413078926e-02,
                -1.0408187539066329e-02, -1.0409033115727400e-02
        },
// x = -4.5600
        {
                -1.0034357727371346e-02, -1.0246537650237544e-02, -1.0353737531100459e-02, -1.0407710341632613e-02,
                -1.0434821766007157e-02, -1.0448419334472265e-02, -1.0455232116803412e-02, -1.0458643185542627e-02,
                -1.0460350281996524e-02, -1.0461204351643084e-02
        },
// x = -4.5550
        {
                -1.0082550827115790e-02, -1.0296829829763045e-02, -1.0405095849398209e-02, -1.0459607320308529e-02,
                -1.0486989953165568e-02, -1.0500723755651543e-02, -1.0507604865845050e-02, -1.0511050169031755e-02,
                -1.0512774406273117e-02, -1.0513637054182595e-02
        },
// x = -4.5500
        {
                -1.0130964845358626e-02, -1.0347363526706068e-02, -1.0456706231926884e-02, -1.0511761720233357e-02,
                -1.0539418268245339e-02, -1.0553289667550123e-02, -1.0560239790177454e-02, -1.0563719671159551e-02,
                -1.0565461221217558e-02, -1.0566332533523248e-02
        },
// x = -4.5450
        {
                -1.0179600689310158e-02, -1.0398139847926017e-02, -1.0508569889038831e-02, -1.0564174804751815e-02,
                -1.0592108001508567e-02, -1.0606118374030634e-02, -1.0613138200511180e-02, -1.0616653006076523e-02,
                -1.0618412042705682e-02, -1.0619292106405330e-02
        },
// x = -4.5400
        {
                -1.0228459268825497e-02, -1.0449159904824977e-02, -1.0560688036632042e-02, -1.0616847843274255e-02,
                -1.0645060449549208e-02, -1.0659211185422835e-02, -1.0666301414092267e-02, -1.0669851494503061e-02,
                -1.0671628193200491e-02, -1.0672517096164966e-02
        },
// x = -4.5350
        {
                -1.0277541496400287e-02, -1.0500424813360906e-02, -1.0613061896172903e-02, -1.0669782111304420e-02,
                -1.0698276915323499e-02, -1.0712569418555343e-02, -1.0719730754734627e-02, -1.0723316463762195e-02,
                -1.0725111001785080e-02, -1.0726008832767130e-02
        },
// x = -4.5300
        {
                -1.0326848287166256e-02, -1.0551935694060837e-02, -1.0665692694719016e-02, -1.0722978890467364e-02,
                -1.0751758708180483e-02, -1.0766194396787531e-02, -1.0773427552852641e-02, -1.0777049247812517e-02,
                -1.0778861804195744e-02, -1.0779768652838839e-02
        },
// x = -4.5250
        {
                -1.0376380558886580e-02, -1.0603693672034032e-02, -1.0718581664942063e-02, -1.0776439468537440e-02,
                -1.0805507143892692e-02, -1.0820087450041564e-02, -1.0827393145493858e-02, -1.0831051187281242e-02,
                -1.0832881942855215e-02, -1.0833797899702482e-02
        },
// x = -4.5200
        {
                -1.0426139231951086e-02, -1.0655699876985142e-02, -1.0771730045150746e-02, -1.0830165139466460e-02,
                -1.0859523544686962e-02, -1.0874249914834587e-02, -1.0881628876371936e-02, -1.0885323629497489e-02,
                -1.0887172766906110e-02, -1.0888097923409350e-02
        },
// x = -4.5150
        {
                -1.0476125229371277e-02, -1.0707955443227338e-02, -1.0825139079313824e-02, -1.0884157203411906e-02,
                -1.0913809239275388e-02, -1.0928683134311081e-02, -1.0936136095899657e-02, -1.0939867928525688e-02,
                -1.0941735632244516e-02, -1.0942670080773327e-02
        },
// x = -4.5100
        {
                -1.0526339476775146e-02, -1.0760461509695368e-02, -1.0878810017083159e-02, -1.0938416966765311e-02,
                -1.0968365562886431e-02, -1.0983388458275364e-02, -1.0990916161222171e-02, -1.0994685445199123e-02,
                -1.0996571901553750e-02, -1.0997515735404721e-02
        },
// x = -4.5050
        {
                -1.0576782902401834e-02, -1.0813219219958667e-02, -1.0932744113816862e-02, -1.0992945742180711e-02,
                -1.1023193857296146e-02, -1.1038367243224260e-02, -1.1045970436250361e-02, -1.1049777547153724e-02,
                -1.1051682944338281e-02, -1.1052636257744292e-02
        },
// x = -4.5000
        {
                -1.0627456437096094e-02, -1.0866229722234371e-02, -1.0986942630602519e-02, -1.1047744848603261e-02,
                -1.1078295470859581e-02, -1.1093620852379900e-02, -1.1101300291694399e-02, -1.1105145608861958e-02,
                -1.1107070136957845e-02, -1.1108033025097460e-02
        },
// x = -4.4950
        {
                -1.0678361014302546e-02, -1.0919494169400353e-02, -1.1041406834280449e-02, -1.1102815611297919e-02,
                -1.1133671758542296e-02, -1.1149150655722700e-02, -1.1156907105097445e-02, -1.1160791011666900e-02,
                -1.1162734862661694e-02, -1.1163707421668645e-02
        },
// x = -4.4900
        {
                -1.0729497570059763e-02, -1.0973013719008188e-02, -1.1096137997467019e-02, -1.1158159361878278e-02,
                -1.1189324081952032e-02, -1.1204958030024515e-02, -1.1212792260869530e-02, -1.1216715143816513e-02,
                -1.1218678511623052e-02, -1.1219660838595805e-02
        },
// x = -4.4850
        {
                -1.0780867042994144e-02, -1.1026789533296118e-02, -1.1151137398578086e-02, -1.1213777438335491e-02,
                -1.1245253809370536e-02, -1.1261044358881884e-02, -1.1268957150321586e-02, -1.1272919400498032e-02,
                -1.1274902480973712e-02, -1.1275894673985152e-02
        },
// x = -4.4800
        {
                -1.0832470374313627e-02, -1.1080822779201978e-02, -1.1206406321852434e-02, -1.1269671185067356e-02,
                -1.1301462315785509e-02, -1.1317411032749523e-02, -1.1325403171699668e-02, -1.1329405183872569e-02,
                -1.1331408174838816e-02, -1.1332410332946029e-02
        },
// x = -4.4750
        {
                -1.0884308507801154e-02, -1.1135114628376080e-02, -1.1261946057375308e-02, -1.1325841952907456e-02,
                -1.1357950982922716e-02, -1.1374059448973918e-02, -1.1382131730219312e-02, -1.1386173903109890e-02,
                -1.1388197004371834e-02, -1.1389209227625948e-02
        },
// x = -4.4700
        {
                -1.0936382389807958e-02, -1.1189666257194066e-02, -1.1317757901102021e-02, -1.1382291099154472e-02,
                -1.1414721199278248e-02, -1.1430991011827086e-02, -1.1439144238100085e-02, -1.1443226974423324e-02,
                -1.1445270387789665e-02, -1.1446292777245843e-02
        },
// x = -4.4650
        {
                -1.0988692969246665e-02, -1.1244478846769750e-02, -1.1373843154881597e-02, -1.1439019987601586e-02,
                -1.1471774360150915e-02, -1.1488207132540534e-02, -1.1496442114600311e-02, -1.1500565821104883e-02,
                -1.1502629750407970e-02, -1.1503662408135462e-02
        },
// x = -4.4600
        {
                -1.1041241197584154e-02, -1.1299553582967902e-02, -1.1430203126480510e-02, -1.1496029988566008e-02,
                -1.1529111867674781e-02, -1.1545709229339316e-02, -1.1554026786051927e-02, -1.1558191873560535e-02,
                -1.1560276524676626e-02, -1.1561319553768962e-02
        },
// x = -4.4550
        {
                -1.1094028028834243e-02, -1.1354891656416991e-02, -1.1486839129606448e-02, -1.1553322478918623e-02,
                -1.1586735130851877e-02, -1.1603498727476336e-02, -1.1611899685895561e-02, -1.1616106569345674e-02,
                -1.1618212150215411e-02, -1.1619265654800632e-02
        },
// x = -4.4500
        {
                -1.1147054419550138e-02, -1.1410494262521899e-02, -1.1543752483932189e-02, -1.1610898842113755e-02,
                -1.1644645565585022e-02, -1.1661577059266728e-02, -1.1670062254715736e-02, -1.1674311353200736e-02,
                -1.1676438073849818e-02, -1.1677502159100888e-02
        },
// x = -4.4450
        {

                -1.1200321328816674e-02, -1.1466362601476618e-02, -1.1600944515119499e-02, -1.1668760468219063e-02,
                -1.1702844594710847e-02, -1.1719945664122473e-02, -1.1728515940276291e-02, -1.1732807677087016e-02,
                -1.1734955749647086e-02, -1.1736030521792350e-02
        },
// x = -4.4400
        {
                -1.1253829718242379e-02, -1.1522497878276874e-02, -1.1658416554843115e-02, -1.1726908753945529e-02,
                -1.1761333648032895e-02, -1.1778605988587126e-02, -1.1787262197555931e-02, -1.1791597000222646e-02,
                -1.1793766638952383e-02, -1.1794852205286151e-02
        },
// x = -4.4350
        {
                -1.1307580551951251e-02, -1.1578901302732727e-02, -1.1716169940814779e-02, -1.1785345102677590e-02,
                -1.1820114162354955e-02, -1.1837559486370762e-02, -1.1846302488783980e-02, -1.1850680789118771e-02,
                -1.1852872210425190e-02, -1.1853968679318431e-02
        },
// x = -4.4300
        {
                -1.1361574796574375e-02, -1.1635574089481125e-02, -1.1774206016807344e-02, -1.1844070924503394e-02,
                -1.1879187581514473e-02, -1.1896807618385035e-02, -1.1905638283476308e-02, -1.1910060517615874e-02,
                -1.1912273940075850e-02, -1.1913381420986975e-02
        },
// x = -4.4250
        {
                -1.1415813421241280e-02, -1.1692517457998423e-02, -1.1832526132678954e-02, -1.1903087636245143e-02,
                -1.1938555356416157e-02, -1.1956351852778435e-02, -1.1965271058471410e-02, -1.1969737666920312e-02,
                -1.1971973311302300e-02, -1.1973091914788086e-02
        },
// x = -4.4200
        {
                -1.1470297397571116e-02, -1.1749732632612857e-02, -1.1891131644397240e-02, -1.1962396661489603e-02,
                -1.1998218945065739e-02, -1.2016193664971732e-02, -1.2025202297966702e-02, -1.2029713725641031e-02,
                -1.2031971814927017e-02, -1.2033101652653600e-02
        },
// x = -4.4150
        {
                -1.1525027699663534e-02, -1.1807220842516954e-02, -1.1950023914063643e-02, -1.2021999430618702e-02,
                -1.2058179812603834e-02, -1.2076334537693526e-02, -1.2085433493554953e-02, -1.2089990189826410e-02,
                -1.2092270949234089e-02, -1.2093412133988098e-02
        },
// x = -4.4100
        {
                -1.1580005304089406e-02, -1.1864983321779928e-02, -1.2009204309937755e-02, -1.2081897380840262e-02,
                -1.2118439431340047e-02, -1.2136775961016049e-02, -1.2145966144260912e-02, -1.2150568563001377e-02,
                -1.2152872220006529e-02, -1.2154024865706315e-02
        },
// x = -4.4050
        {
                -1.1635231189881275e-02, -1.1923021309360007e-02, -1.2068674206461721e-02, -1.2142091956218837e-02,
                -1.2178999280787136e-02, -1.2197519432391059e-02, -1.2206801756578142e-02, -1.2211450356204641e-02,
                -1.2213777140563750e-02, -1.2214941362270734e-02
        },
// x = -4.4000
        {
                -1.1690706338523548e-02, -1.1981336049116704e-02, -1.2128434984284727e-02, -1.2202584607706716e-02,
                -1.2239860847695386e-02, -1.2258566456685971e-02, -1.2267941844505963e-02, -1.2272637088026110e-02,
                -1.2274987231799206e-02, -1.2276163145729345e-02
        },
// x = -4.3950
        {
                -1.1746431733942539e-02, -1.2039928789823085e-02, -1.2188488030287549e-02, -1.2263376793174989e-02,
                -1.2301025626087134e-02, -1.2319918546220101e-02, -1.2329387929586667e-02, -1.2334130284644551e-02,
                -1.2336504022218269e-02, -1.2337691745753627e-02
        },
// x = -4.3900
        {
                -1.1802408362496157e-02, -1.2098800785177907e-02, -1.2248834737607120e-02, -1.2324469977444789e-02,
                -1.2362495117291436e-02, -1.2381577220801143e-02, -1.2391141540942842e-02, -1.2395931479865367e-02,
                -1.2398329047976248e-02, -1.2399528699676688e-02
        },
// x = -4.3850
        {
                -1.1858637212963435e-02, -1.2157953293817790e-02, -1.2309476505661212e-02, -1.2385865632318620e-02,
                -1.2424270829978889e-02, -1.2443544007761763e-02, -1.2453204215314912e-02, -1.2458042215158600e-02,
                -1.2460463852916635e-02, -1.2461675552531618e-02
        },
// x = -4.3800
        {
                -1.1915119276533779e-02, -1.2217387579329267e-02, -1.2370414740173160e-02, -1.2447565236611832e-02,
                -1.2486354280196619e-02, -1.2505820441996423e-02, -1.2515577497098852e-02, -1.2520464039697128e-02,
                -1.2522909988609500e-02, -1.2524133857090029e-02
        },
// x = -4.3750
        {
                -1.1971855546795973e-02, -1.2277104910260832e-02, -1.2431650853196616e-02, -1.2509570276184205e-02,
                -1.2548746991403429e-02, -1.2568408065998318e-02, -1.2578262938384085e-02, -1.2583198510395015e-02,
                -1.2585669014390127e-02, -1.2586905173900778e-02
        },
// x = -4.3700
        {
                -1.2028847019726939e-02, -1.2337106560134894e-02, -1.2493186263140412e-02, -1.2571882243971665e-02,
                -1.2611450494505108e-02, -1.2631308429896553e-02, -1.2641262098991584e-02, -1.2646247191946092e-02,
                -1.2648742497397789e-02, -1.2649991071328892e-02
        },
// x = -4.3650
        {
                -1.2086094693680232e-02, -1.2397393807459692e-02, -1.2555022394793456e-02, -1.2634502640018121e-02,
                -1.2674466327889870e-02, -1.2694523091493442e-02, -1.2704576546512138e-02, -1.2709611656862706e-02,
                -1.2712132012614777e-02, -1.2713393125594686e-02
        },
// x = -4.3600
        {
                -1.2143599569374285e-02, -1.2457967935741153e-02, -1.2617160679349678e-02, -1.2697432971507433e-02,
                -1.2737796037463982e-02, -1.2758053616302032e-02, -1.2768207856344794e-02, -1.2773293485514649e-02,
                -1.2775839142905558e-02, -1.2777112920813075e-02
        },
// x = -4.3550
        {
                -1.2201362649880410e-02, -1.2518830233494678e-02, -1.2679602554433064e-02, -1.2760674752795491e-02,
                -1.2801441176687559e-02, -1.2821901577583766e-02, -1.2832157611735536e-02, -1.2837294266168331e-02,
                -1.2839865479056174e-02, -1.2841152049033086e-02
        },
// x = -4.3500
        {
                -1.2259384940610501e-02, -1.2579981994256887e-02, -1.2742349464122716e-02, -1.2824229505442434e-02,
                -1.2865403306610488e-02, -1.2886068556386337e-02, -1.2896427403816087e-02, -1.2901615595026070e-02,
                -1.2904212619813825e-02, -1.2905512110277557e-02
        },
// x = -4.3450
        {
                -1.2317667449304497e-02, -1.2641424516597281e-02, -1.2805402858978002e-02, -1.2888098758244988e-02,
                -1.2929683995908528e-02, -1.2950556141581755e-02, -1.2961018831642971e-02, -1.2966259076265647e-02,
                -1.2968882171926648e-02, -1.2970194712583056e-02
        },
// x = -4.3400
        {
                -1.2376211186017599e-02, -1.2703159104129854e-02, -1.2868764196063745e-02, -1.2952284047268923e-02,
                -1.2994284820919588e-02, -1.3015365929904538e-02, -1.3025933502236698e-02, -1.3031226322080021e-02,
                -1.3033875750183679e-02, -1.3035201472039969e-02
        },
// x = -4.3350
        {
                -1.2435017163107144e-02, -1.2765187065524640e-02, -1.2932434938975473e-02, -1.3016786915881654e-02,
                -1.3059207365680144e-02, -1.3080499525990125e-02, -1.3091173030621199e-02, -1.3096518952717240e-02,
                -1.3099194977455058e-02, -1.3100534012832820e-02
        },
// x = -4.3300
        {
                -1.2494086395219291e-02, -1.2827509714519188e-02, -1.2996416557864727e-02, -1.3081608914784959e-02,
                -1.3124453221961830e-02, -1.3145958542413461e-02, -1.3156739039863407e-02, -1.3162138596520564e-02,
                -1.3164841484732362e-02, -1.3166193967280762e-02
        },
// x = -4.3250
        {
                -1.2553419899275381e-02, -1.2890128369929966e-02, -1.3060710529464434e-02, -1.3146751602047806e-02,
                -1.3190023989308202e-02, -1.3211744599727770e-02, -1.3222633161113061e-02, -1.3228086889968788e-02,
                -1.3230816911169235e-02, -1.3232182975878307e-02
        },
// x = -4.3200
        {
                -1.2613018694458019e-02, -1.2953044355663716e-02, -1.3125318337114307e-02, -1.3212216543139338e-02,
                -1.3255921275071633e-02, -1.3277859326503492e-02, -1.3288857033642692e-02, -1.3294365477716731e-02,
                -1.3297122904122114e-02, -1.3298502687336212e-02
        },
// x = -4.3150
        {
                -1.2672883802196886e-02, -1.3016259000728704e-02, -1.3190241470786358e-02, -1.3278005310961970e-02,
                -1.3322146694450417e-02, -1.3344304359367426e-02, -1.3355412304887812e-02, -1.3360976012635972e-02,
                -1.3363761119191249e-02, -1.3365154758622615e-02
        },
// x = -4.3100
        {
                -1.2733016246154279e-02, -1.3079773639245944e-02, -1.3255481427110414e-02, -1.3344119485884607e-02,
                -1.3388701870526020e-02, -1.3411081343042070e-02, -1.3422300630487289e-02, -1.3427920155855763e-02,
                -1.3430733220261868e-02, -1.3432140855004352e-02
        },
// x = -4.3050
        {
                -1.2793417052210316e-02, -1.3143589610460310e-02, -1.3321039709399712e-02, -1.3410560655775998e-02,
                -1.3455588434300480e-02, -1.3478191930385098e-02, -1.3489523674323914e-02, -1.3495199576804135e-02,
                -1.3498040879545578e-02, -1.3499462650088480e-02
        },
// x = -4.3000
        {
                -1.2854087248447919e-02, -1.3207708258751583e-02, -1.3386917827676547e-02, -1.3477330416038229e-02,
                -1.3522808024734002e-02, -1.3545637782429083e-02, -1.3557083108565203e-02, -1.3562815953249234e-02,
                -1.3565685777621961e-02, -1.3567121825864016e-02
        },
// x = -4.2950
        {
                -1.2915027865137417e-02, -1.3272130933645442e-02, -1.3453117298697968e-02, -1.3544430369640318e-02,
                -1.3590362288782697e-02, -1.3613420568421372e-02, -1.3624980613704348e-02, -1.3630770971340838e-02,
                -1.3633669603480369e-02, -1.3635120072743888e-02
        },
// x = -4.2900
        {
                -1.2976239934720957e-02, -1.3336858989824346e-02, -1.3519639645981546e-02, -1.3611862127151956e-02,
                -1.3658252881436517e-02, -1.3681541965864160e-02, -1.3693217878601397e-02, -1.3699066325652093e-02,
                -1.3701994054561951e-02, -1.3703459089607073e-02
        },
// x = -4.2850
        {
                -1.3037724491796496e-02, -1.3401893787138373e-02, -1.3586486399831165e-02, -1.3679627306777388e-02,
                -1.3726481465757334e-02, -1.3750003660554753e-02, -1.3761796600524630e-02, -1.3767703719221449e-02,
                -1.3770660836801870e-02, -1.3772140583840985e-02
        },
// x = -4.2800
        {
                -1.3099482573101591e-02, -1.3467236690615930e-02, -1.3653659097362930e-02, -1.3747727534389378e-02,
                -1.3795049712917194e-02, -1.3818807346626032e-02, -1.3830718485192132e-02, -1.3836684863594797e-02,
                -1.3839671664671718e-02, -1.3841166271384021e-02
        },
// x = -4.2750
        {
                -1.3161515217496835e-02, -1.3532889070474442e-02, -1.3721159282531026e-02, -1.3816164443563383e-02,
                -1.3863959302236757e-02, -1.3887954726587094e-02, -1.3899985246813592e-02, -1.3906011478867842e-02,
                -1.3909028261222192e-02, -1.3910537876768379e-02
        },
// x = -4.2700
        {
                -1.3223823465948966e-02, -1.3598852302130898e-02, -1.3788988506153737e-02, -1.3884939675611755e-02,
                -1.3933211921223876e-02, -1.3957447511364079e-02, -1.3969598608132251e-02, -1.3975685293728644e-02,
                -1.3978732358125922e-02, -1.3980257133163041e-02
        },
// x = -4.2650
        {
                -1.3286408361513721e-02, -1.3665127766212358e-02, -1.3857148325939445e-02, -1.3954054879618162e-02,
                -1.4002809265612393e-02, -1.4027287420341228e-02, -1.4039560300467116e-02, -1.4045708045500401e-02,
                -1.4048785695720538e-02, -1.4050325782416993e-02
        },
// x = -4.2600
        {
                -1.3349270949318310e-02, -1.3731716848566342e-02, -1.3925640306512712e-02, -1.4023511712472096e-02,
                -1.4072753039401056e-02, -1.4097476181402096e-02, -1.4109872063755349e-02, -1.4116081480184443e-02,
                -1.4119190023051982e-02, -1.4120745575102676e-02
        },
// x = -4.2550
        {
                -1.3412412276543620e-02, -1.3798620940271148e-02, -1.3994466019440420e-02, -1.4093311838903515e-02,
                -1.4143044954892660e-02, -1.4168015530970969e-02, -1.4180535646594860e-02, -1.4186807352503413e-02,
                -1.4189947097917977e-02, -1.4191518270559606e-02
        },
// x = -4.2500
        {
                -1.3475833392406071e-02, -1.3865841437646079e-02, -1.4063627043257939e-02, -1.4163456931517618e-02,
                -1.4213686732733316e-02, -1.4238907214054493e-02, -1.4251552806287126e-02, -1.4257887425944684e-02,
                -1.4261058686911755e-02, -1.4262645636938265e-02
        },
// x = -4.2450
        {
                -1.3539535348139134e-02, -1.3933379742261571e-02, -1.4133124963495348e-02, -1.4233948670829760e-02,
                -1.4284680101951915e-02, -1.4310152984283486e-02, -1.4322925308880189e-02, -1.4329323472803985e-02,
                -1.4332526565465985e-02, -1.4334129451244184e-02
        },
// x = -4.2400
        {
                -1.3603519196974577e-02, -1.4001237260949220e-02, -1.4202961372703733e-02, -1.4304788745300483e-02,
                -1.4356026799999784e-02, -1.4381754603954947e-02, -1.4394654929211905e-02, -1.4401117274229247e-02,
                -1.4404352517896931e-02, -1.4405971499382246e-02
        },
// x = -4.2350
        {
                -1.3667785994123296e-02, -1.4069415405811751e-02, -1.4273137870481525e-02, -1.4375978851370706e-02,
                -1.4427728572790495e-02, -1.4453713844074272e-02, -1.4466743450953355e-02, -1.4473270620264650e-02,
                -1.4476538337448819e-02, -1.4478173576201214e-02
        },
// x = -4.2300
        {
                -1.3732336796755876e-02, -1.4137915594232843e-02, -1.4343656063500848e-02, -1.4447520693497000e-02,
                -1.4499787174739829e-02, -1.4526032484397664e-02, -1.4539192666652521e-02, -1.4545785309894903e-02,
                -1.4549085826338426e-02, -1.4550737485538486e-02
        },
// x = -4.2250
        {
                -1.3797172663982801e-02, -1.4206739248886883e-02, -1.4414517565533972e-02, -1.4519415984187031e-02,
                -1.4572204368805979e-02, -1.4598712313474744e-02, -1.4612004377778123e-02, -1.4618663151089744e-02,
                -1.4621996795799898e-02, -1.4623665040265072e-02
        },
// x = -4.2200
        {
                -1.3862294656834273e-02, -1.4275887797748622e-02, -1.4485723997479789e-02, -1.4591666444035150e-02,
                -1.4644981926529866e-02, -1.4671755128691366e-02, -1.4685180394763696e-02, -1.4691905960848638e-02,
                -1.4695273066129780e-02, -1.4696958062330782e-02
        },
// x = -4.2150
        {
                -1.3927703838239760e-02, -1.4345362674102732e-02, -1.4557276987390335e-02, -1.4664273801758059e-02,
                -1.4718121628075691e-02, -1.4745162736312618e-02, -1.4758722537051901e-02, -1.4765515565245739e-02,
                -1.4768916466732297e-02, -1.4770618382809670e-02
        },
// x = -4.2100
        {
                -1.3993401273007149e-02, -1.4415165316553213e-02, -1.4629178170497368e-02, -1.4737239794230670e-02,
                -1.4791625262271599e-02, -1.4818936951526078e-02, -1.4832632633139001e-02, -1.4839493799475008e-02,
                -1.4842928836164808e-02, -1.4844647841945667e-02
        },
// x = -4.2050
        {
                -1.4059388027801531e-02, -1.4485297169032766e-02, -1.4701429189238977e-02, -1.4810566166522047e-02,
                -1.4865494626650594e-02, -1.4893079598485180e-02, -1.4906912520619596e-02, -1.4913842507895634e-02,
                -1.4917312022183543e-02, -1.4919048289198471e-02
        },
// x = -4.2000
        {
                -1.4125665171123677e-02, -1.4555759680812028e-02, -1.4774031693286256e-02, -1.4884254671931522e-02,
                -1.4939731527491579e-02, -1.4967592510352887e-02, -1.4981564046231556e-02, -1.4988563544077604e-02,
                -1.4992067881789542e-02, -1.4993821583289655e-02
        },
// x = -4.1950
        {
                -1.4192233773288128e-02, -1.4626554306508666e-02, -1.4846987339570019e-02, -1.4958307072024923e-02,
                -1.5014337779860609e-02, -1.5042477529345509e-02, -1.5056589065901183e-02, -1.5063658770847553e-02,
                -1.5067198281274806e-02, -1.5068969592249007e-02
        },
// x = -4.1900
        {
                -1.4259094906400916e-02, -1.4697682506096413e-02, -1.4920297792307552e-02, -1.5032725136670930e-02,
                -1.5089315207652309e-02, -1.5117736506776730e-02, -1.5131989444788592e-02, -1.5139130060334819e-02,
                -1.5142705096268716e-02, -1.5144494193461087e-02
        },
// x = -4.1850
        {
                -1.4326249644336929e-02, -1.4769145744913970e-02, -1.4993964723029421e-02, -1.5107510644077586e-02,
                -1.5164665643631477e-02, -1.5193371303101871e-02, -1.5207767057333284e-02, -1.5214979294017706e-02,
                -1.5218590211784648e-02, -1.5220397273712058e-02
        },
// x = -4.1800
        {
                -1.4393699062716900e-02, -1.4840945493673785e-02, -1.5067989810606317e-02, -1.5182665380828929e-02,
                -1.5240390929474884e-02, -1.5269383787962346e-02, -1.5283923787300013e-02, -1.5291208362770026e-02,
                -1.5294855522266842e-02, -1.5296680729236706e-02
        },
// x = -4.1750
        {
                -1.4461444238884029e-02, -1.4913083228470712e-02, -1.5142374741275947e-02, -1.5258191141921751e-02,
                -1.5316492915813225e-02, -1.5345775840230298e-02, -1.5360461527824764e-02, -1.5367819166907805e-02,
                -1.5371502931637486e-02, -1.5373346465765705e-02
        },
// x = -4.1700
        {
                -1.4529486251880223e-02, -1.4985560430790603e-02, -1.5217121208669956e-02, -1.5334089730802507e-02,
                -1.5392973462273302e-02, -1.5422549348053536e-02, -1.5437382181461082e-02, -1.5444813616236291e-02,
                -1.5448534353344066e-02, -1.5450396398573134e-02
        },
// x = -4.1650
        {
                -1.4597826182421940e-02, -1.5058378587518689e-02, -1.5292230913840922e-02, -1.5410362959404357e-02,
                -1.5469834437520336e-02, -1.5499706208900533e-02, -1.5514687660226510e-02, -1.5522193630097127e-02,
                -1.5525951710406898e-02, -1.5527832452524230e-02
        },
// x = -4.1600
        {
                -1.4666465112875690e-02, -1.5131539190947943e-02, -1.5367705565289366e-02, -1.5487012648184322e-02,
                -1.5547077719300521e-02, -1.5577248329605810e-02, -1.5592379885649352e-02, -1.5599961137415812e-02,
                -1.5603756935466954e-02, -1.5605656562123352e-02
        },
// x = -4.1550
        {
                -1.4735404127233086e-02, -1.5205043738787220e-02, -1.5443546878990808e-02, -1.5564040626160616e-02,
                -1.5624705194483720e-02, -1.5655177626415391e-02, -1.5670460788815576e-02, -1.5678118076749348e-02,
                -1.5681951970833902e-02, -1.5683870671562228e-02
        },
// x = -4.1500
        {
                -1.4804644311085562e-02, -1.5278893734169362e-02, -1.5519756578422864e-02, -1.5641448730950053e-02,
                -1.5702718759106362e-02, -1.5733496025032559e-02, -1.5748932310416009e-02, -1.5756666396334180e-02,
                -1.5760538768534348e-02, -1.5762476734768396e-02
        },
// x = -4.1450
        {
                -1.4874186751598666e-02, -1.5353090685659113e-02, -1.5596336394592403e-02, -1.5719238808805672e-02,
                -1.5781120318414540e-02, -1.5812205460663774e-02, -1.5827796400793754e-02, -1.5835608054134288e-02,
                -1.5839519290360416e-02, -1.5841476715453929e-02
        },
// x = -4.1400
        {
                -1.4944032537485932e-02, -1.5427636107260913e-02, -1.5673288066062694e-02, -1.5797412714654409e-02,
                -1.5859911786907273e-02, -1.5891307878064854e-02, -1.5907055019991758e-02, -1.5914945017889634e-02,
                -1.5918895507918432e-02, -1.5920872587164361e-02
        },
// x = -4.1350
        {
                -1.5014182758982399e-02, -1.5502531518426597e-02, -1.5750613338980674e-02, -1.5875972312134983e-02,
                -1.5939095088380009e-02, -1.5970805231587323e-02, -1.5986710137800755e-02, -1.5994679265164711e-02,
                -1.5998669402677988e-02, -1.6000666333327921e-02
        },
// x = -4.1300
        {
                -1.5084638507817681e-02, -1.5577778444062908e-02, -1.5828313967104140e-02, -1.5954919473635833e-02,
                -1.6018672155968201e-02, -1.6050699485225015e-02, -1.6066763733807311e-02, -1.6074812783397452e-02,
                -1.6078842966021148e-02, -1.6080859947304921e-02
        },
// x = -4.1250
        {
                -1.5155400877188648e-02, -1.5653378414538940e-02, -1.5906391711829091e-02, -1.6034256080333285e-02,
                -1.6098644932191226e-02, -1.6130992612660894e-02, -1.6147217797442159e-02, -1.6155347569948301e-02,
                -1.6159418199291967e-02, -1.6161455432437480e-02
        },
// x = -4.1200
        {
                -1.5226470961731672e-02, -1.5729332965693357e-02, -1.5984848342217046e-02, -1.6113984022229823e-02,
                -1.6179015368996397e-02, -1.6211686597314062e-02, -1.6228074328028762e-02, -1.6236285632149593e-02,
                -1.6240397113846186e-02, -1.6242454802099442e-02
        },
// x = -4.1150
        {
                -1.5297849857494487e-02, -1.5805643638841575e-02, -1.6063685635022400e-02, -1.6194105198192425e-02,
                -1.6259785427803163e-02, -1.6292783432387031e-02, -1.6309335334832146e-02, -1.6317628987355091e-02,
                -1.6321781731101267e-02, -1.6323860079746580e-02
        },
// x = -4.1100
        {
                -1.5369538661907566e-02, -1.5882311980782677e-02, -1.6142905374719834e-02, -1.6274621515991131e-02,
                -1.6340957079547573e-02, -1.6374285120913150e-02, -1.6391002837107867e-02, -1.6399379662989839e-02,
                -1.6403574082586581e-02, -1.6405673298966972e-02
        },
// x = -4.1050
        {
                -1.5441538473755149e-02, -1.5959339543806322e-02, -1.6222509353531752e-02, -1.6355534892337736e-02,
                -1.6422532304726851e-02, -1.6456193675804382e-02, -1.6473078864151360e-02, -1.6481539696600250e-02,
                -1.6485776209993919e-02, -1.6487896503531767e-02
        },
// x = -4.1000
        {
                -1.5513850393145769e-02, -1.6036727885699414e-02, -1.6302499371455729e-02, -1.6436847252924554e-02,
                -1.6504513093444206e-02, -1.6538511119899141e-02, -1.6555565455347383e-02, -1.6564111135904398e-02,
                -1.6568390165228195e-02, -1.6570531747446077e-02
        },
// x = -4.0950
        {
                -1.5586475521482412e-02, -1.6114478569752642e-02, -1.6382877236292066e-02, -1.6518560532463408e-02,
                -1.6586901445453873e-02, -1.6621239486010471e-02, -1.6638464660219846e-02, -1.6647096038842641e-02,
                -1.6651418010458476e-02, -1.6653581095000161e-02
        },
// x = -4.0900
        {
                -1.5659414961432142e-02, -1.6192593164766886e-02, -1.6463644763671244e-02, -1.6600676674724639e-02,
                -1.6669699370206233e-02, -1.6704380816974403e-02, -1.6721778538481714e-02, -1.6730496473628374e-02,
                -1.6734861818169161e-02, -1.6737046620820911e-02
        },
// x = -4.0850
        {
                -1.5732669816895386e-02, -1.6271073245059445e-02, -1.6544803777081573e-02, -1.6683197632576419e-02,
                -1.6752908886893285e-02, -1.6787937165698562e-02, -1.6805509160085320e-02, -1.6814314518799150e-02,
                -1.6818723671211511e-02, -1.6820930409923520e-02
        },
// x = -4.0800
        {
                -1.5806241192974725e-02, -1.6349920390470157e-02, -1.6626356107896752e-02, -1.6766125368024055e-02,
                -1.6836532024494175e-02, -1.6871910595211009e-02, -1.6889658605272813e-02, -1.6898552263267988e-02,
                -1.6903005662855429e-02, -1.6905234557763472e-02
        },
// x = -4.0750
        {
                -1.5880130195943191e-02, -1.6429136186367297e-02, -1.6708303595403470e-02, -1.6849461852249462e-02,
                -1.6920570821821000e-02, -1.6956303178709196e-02, -1.6974228964626862e-02, -1.6983211806374937e-02,
                -1.6987709896841365e-02, -1.6989961170288753e-02
        },
// x = -4.0700
        {
                -1.5954337933212229e-02, -1.6508722223653387e-02, -1.6790648086829126e-02, -1.6933209065650868e-02,
                -1.7005027327564811e-02, -1.7041116999609387e-02, -1.7059222339121695e-02, -1.7068295257938935e-02,
                -1.7072838487432709e-02, -1.7075112363992336e-02
        },
// x = -4.0650
        {
                -1.6028865513299028e-02, -1.6588680098770757e-02, -1.6873391437369451e-02, -1.7017368997882528e-02,
                -1.7089903600341742e-02, -1.7126354151596045e-02, -1.7144640840174221e-02, -1.7153804738309866e-02,
                -1.7158393559468209e-02, -1.7160690265964926e-02
        },
// x = -4.0600
        {
                -1.6103714045793579e-02, -1.6669011413707028e-02, -1.6956535510216246e-02, -1.7101943647894649e-02,
                -1.7175201708739443e-02, -1.7212016738671641e-02, -1.7230486589695587e-02, -1.7239742378420901e-02,
                -1.7244377248414811e-02, -1.7246697013947970e-02
        },
// x = -4.0550
        {
                -1.6178884641325070e-02, -1.6749717776000390e-02, -1.7040082176585161e-02, -1.7186935023973473e-02,
                -1.7260923731363644e-02, -1.7298106875206613e-02, -1.7316761720142817e-02, -1.7326110319841131e-02,
                -1.7330791700420699e-02, -1.7333134756386939e-02
        },
// x = -4.0500
        {
                -1.6254378411528016e-02, -1.6830800798744702e-02, -1.7124033315743387e-02, -1.7272345143781401e-02,
                -1.7347071756884904e-02, -1.7384626685989571e-02, -1.7403468374570841e-02, -1.7412910714828395e-02,
                -1.7417639072368596e-02, -1.7420005652484839e-02
        },
// x = -4.0450
        {
                -1.6330196469007685e-02, -1.6912262100594433e-02, -1.7208390815037507e-02, -1.7358176034397380e-02,
                -1.7433647884085638e-02, -1.7471578306277727e-02, -1.7490608706684667e-02, -1.7500145726382418e-02,
                -1.7504921531929358e-02, -1.7507311872256064e-02
        },
// x = -4.0400
        {
                -1.6406339927305277e-02, -1.6994103305769448e-02, -1.7293156569921260e-02, -1.7444429732357332e-02,
                -1.7520654221907284e-02, -1.7558963881847588e-02, -1.7578184880891867e-02, -1.7587817528298161e-02,
                -1.7592641257615782e-02, -1.7595055596580429e-02
        },
// x = -4.0350
        {
                -1.6482809900862420e-02, -1.7076326044059561e-02, -1.7378332483983436e-02, -1.7531108283694790e-02,
                -1.7608092889497669e-02, -1.7646785569045875e-02, -1.7666199072355289e-02, -1.7675928305219517e-02,
                -1.7680800438836766e-02, -1.7683239017257532e-02
        },
// x = -4.0300
        {
                -1.6559607504985308e-02, -1.7158931950828994e-02, -1.7463920468975706e-02, -1.7618213743981620e-02,
                -1.7695966016258644e-02, -1.7735045534840638e-02, -1.7754653467046085e-02, -1.7764480252693188e-02,
                -1.7769401275951648e-02, -1.7771864337061374e-02
        },
// x = -4.0250
        {
                -1.6636733855808269e-02, -1.7241922667020564e-02, -1.7549922444840479e-02, -1.7705748178368903e-02,
                -1.7784275741893809e-02, -1.7823745956872698e-02, -1.7843550261796862e-02, -1.7853475577222843e-02,
                -1.7858445980324874e-02, -1.7860933769795230e-02
        },
// x = -4.0200
        {
                -1.6714190070256880e-02, -1.7325299839159768e-02, -1.7636340339738873e-02, -1.7793713661627984e-02,
                -1.7873024216456607e-02, -1.7912889023507243e-02, -1.7932891664355244e-02, -1.7942916496323586e-02,
                -1.7947936774380911e-02, -1.7950449540346844e-02
        },
// x = -4.0150
        {
                -1.6791977266010550e-02, -1.7409065119358628e-02, -1.7723176090078563e-02, -1.7882112278191593e-02,
                -1.7962213600398420e-02, -1.8002476933885674e-02, -1.8022679893437601e-02, -1.8032805238576675e-02,
                -1.8037875891659443e-02, -1.8040413884743815e-02
        },
// x = -4.0100
        {
                -1.6870096561464626e-02, -1.7493220165319370e-02, -1.7810431640541777e-02, -1.7970946122195210e-02,
                -1.8051846064617069e-02, -1.8092511897977798e-02, -1.8112917178783043e-02, -1.8123144043684480e-02,
                -1.8128265576870844e-02, -1.8130829050209386e-02
        },
// x = -4.0050
        {
                -1.6948549075691956e-02, -1.7577766640337885e-02, -1.7898108944113238e-02, -1.8060217297518426e-02,
                -1.8141923790505391e-02, -1.8182996136634072e-02, -1.8203605761207733e-02, -1.8213935162525716e-02,
                -1.8219108085951917e-02, -1.8221697295218376e-02
        },
// x = -4.0000
        {
                -1.7027335928403945e-02, -1.7662706213307088e-02, -1.7986209962108129e-02, -1.8149927917826613e-02,
                -1.8232448970000047e-02, -1.8273931881638301e-02, -1.8294747892659384e-02, -1.8305180857211008e-02,
                -1.8310405686121911e-02, -1.8313020889553451e-02
        },
// x = -3.9950
        {
                -1.7106458239911103e-02, -1.7748040558719914e-02, -1.8074736664200095e-02, -1.8240080106612565e-02,
                -1.8323423805630602e-02, -1.8365321375760406e-02, -1.8386345836272064e-02, -1.8396883401138646e-02,
                -1.8402160655938825e-02, -1.8404802114361737e-02
        },
// x = -3.9900
        {
                -1.7185917131083094e-02, -1.7833771356672294e-02, -1.8163691028449270e-02, -1.8330675997238428e-02,
                -1.8414850510568712e-02, -1.8457166872809554e-02, -1.8478401866421284e-02, -1.8489045079050692e-02,
                -1.8494375285355996e-02, -1.8497043262211570e-02
        },
// x = -3.9850
        {
                -1.7265713723308165e-02, -1.7919900292865835e-02, -1.8253075041330273e-02, -1.8421717732977676e-02,
                -1.8506731308677651e-02, -1.8549470637687474e-02, -1.8570918268779282e-02, -1.8581668187089274e-02,
                -1.8587051875778939e-02, -1.8589746637149691e-02
        },
// x = -3.9800
        {
                -1.7345849138452125e-02, -1.8006429058610253e-02, -1.8342890697760232e-02, -1.8513207467057216e-02,
                -1.8599068434561869e-02, -1.8642234946442054e-02, -1.8663897340370689e-02, -1.8674755032853270e-02,
                -1.8680192740122489e-02, -1.8682914554758616e-02
        },
// x = -3.9750
        {
                -1.7426324498816804e-02, -1.8093359350825733e-02, -1.8433140001126870e-02, -1.8605147362699727e-02,
                -1.8691864133616960e-02, -1.8735462086321139e-02, -1.8757341389628316e-02, -1.8768307935455152e-02,
                -1.8773800202868254e-02, -1.8776549342214309e-02
        },
// x = -3.9700
        {
                -1.7507140927097806e-02, -1.8180692872044955e-02, -1.8523824963316549e-02, -1.8697539593166074e-02,
                -1.8785120662079709e-02, -1.8829154355826664e-02, -1.8851252736449345e-02, -1.8862329225578216e-02,
                -1.8867876600122294e-02, -1.8870653338344191e-02
        },
// x = -3.9650
        {
                -1.7588299546341977e-02, -1.8268431330414992e-02, -1.8614947604742291e-02, -1.8790386341797836e-02,
                -1.8878840287078369e-02, -1.8923314064768923e-02, -1.8945633712251712e-02, -1.8956821245534005e-02,
                -1.8962424279673134e-02, -1.8965228893685401e-02
        },
// x = -3.9600
        {
                -1.7669801479904088e-02, -1.8356576439698912e-02, -1.8706509954371932e-02, -1.8883689802060056e-02,
                -1.8973025286683199e-02, -1.9017943534321204e-02, -1.9040486660030782e-02, -1.9051786349320071e-02,
                -1.9057445601050060e-02, -1.9060278370543339e-02
        },
// x = -3.9550
        {
                -1.7751647851403048e-02, -1.8445129919277277e-02, -1.8798514049756153e-02, -1.8977452177584033e-02,
                -1.9067677949957201e-02, -1.9113045097074614e-02, -1.9135813934416324e-02, -1.9147226902678033e-02,
                -1.9152942935581660e-02, -1.9155804143050528e-02
        },
// x = -3.9500
        {
                -1.7833839784677619e-02, -1.8534093494149265e-02, -1.8890961937056583e-02, -1.9071675682210361e-02,
                -1.9162800577007005e-02, -1.9208621097093148e-02, -1.9231617901729697e-02, -1.9243145283151832e-02,
                -1.9248918666454739e-02, -1.9251808597225761e-02
        },
// x = -3.9450
        {
                -1.7916378403741415e-02, -1.8623468894933718e-02, -1.8983855671073885e-02, -1.9166362540032011e-02,
                -1.9258395479034088e-02, -1.9304673889969058e-02, -1.9327900940041401e-02, -1.9339543880146370e-02,
                -1.9345375188773422e-02, -1.9348294131033516e-02
        },
// x = -3.9400
        {
                -1.7999264832737481e-02, -1.8713257857869918e-02, -1.9077197315275893e-02, -1.9261514985437621e-02,
                -1.9354464978386086e-02, -1.9401205842878435e-02, -1.9424665439228821e-02, -1.9436425094986408e-02,
                -1.9442314909618643e-02, -1.9445263154443736e-02
        },
// x = -3.9350
        {
                -1.8082500195892207e-02, -1.8803462124818048e-02, -1.9170988941825652e-02, -1.9357135263154895e-02,
                -1.9451011408608433e-02, -1.9498219334637110e-02, -1.9521913801034314e-02, -1.9533791340975722e-02,
                -1.9539740248107895e-02, -1.9542718089491803e-02
        },
// x = -3.9300
        {

                -1.8166085617468693e-02, -1.8894083443259567e-02, -1.9265232631609615e-02, -1.9453225628294140e-02,
                -1.9548037114496129e-02, -1.9595716755756706e-02, -1.9619648439123505e-02, -1.9631645043456585e-02,
                -1.9637653635455245e-02, -1.9640661370338930e-02
        },
// x = -3.9250
        {
                -1.8250022221719515e-02, -1.8985123566297227e-02, -1.9359930474265628e-02, -1.9549788346391987e-02,
                -1.9645544452145756e-02, -1.9693700508501073e-02, -1.9717871779143963e-02, -1.9729988639869533e-02,
                -1.9736057515031703e-02, -1.9739095443332754e-02
        },
// x = -3.9200
        {
                -1.8334311132838902e-02, -1.9076584252654915e-02, -1.9455084568211160e-02, -1.9646825693455158e-02,
                -1.9743535789007768e-02, -1.9792173006942911e-02, -1.9816586258784047e-02, -1.9828824579813421e-02,
                -1.9834954342425843e-02, -1.9838022767068291e-02
        },
// x = -3.9150
        {
                -1.8418953474914328e-02, -1.9168467266677251e-02, -1.9550697020671335e-02, -1.9744339956004487e-02,
                -1.9842013503938896e-02, -1.9891136677020655e-02, -1.9915794327832151e-02, -1.9928155325105744e-02,
                -1.9934346585504765e-02, -1.9937445812449165e-02
        },
// x = -3.9100
        {
                -1.8503950371877475e-02, -1.9260774378328918e-02, -1.9646769947707036e-02, -1.9842333431119036e-02,
                -1.9940979987254869e-02, -1.9990593956595668e-02, -2.0015498448236121e-02, -2.0027983349843351e-02,
                -2.0034236724475320e-02, -2.0037367062749171e-02
        },
// x = -3.9050
        {
                -1.8589302947454665e-02, -1.9353507363193755e-02, -1.9743305474243029e-02, -1.9940808426480265e-02,
                -2.0040437640783275e-02, -2.0090547295509624e-02, -2.0115701094163035e-02, -2.0128311140463331e-02,
                -2.0134627251945675e-02, -2.0137789013674093e-02
        },
// x = -3.9000
        {
                -1.8675012325116570e-02, -1.9446668002473647e-02, -1.9840305734096047e-02, -2.0039767260416529e-02,
                -2.0140388877916724e-02, -2.0190999155642226e-02, -2.0216404752059273e-02, -2.0229141195804278e-02,
                -2.0235520672987162e-02, -2.0238714173423909e-02
        },
// x = -3.8950
        {
                -1.8761079628027387e-02, -1.9540258082987049e-02, -1.9937772870002875e-02, -2.0139212261947519e-02,
                -2.0240836123666194e-02, -2.0291952010969193e-02, -2.0317611920710802e-02, -2.0330476027167858e-02,
                -2.0336919505196425e-02, -2.0340145062755246e-02
        },
// x = -3.8900
        {
                -1.8847505978993387e-02, -1.9634279397167394e-02, -2.0035709033648420e-02, -2.0239145770828987e-02,
                -2.0341781814714549e-02, -2.0393408347620438e-02, -2.0419325111303861e-02, -2.0432318158380633e-02,
                -2.0438826278757923e-02, -2.0442084215044131e-02
        },
// x = -3.8850
        {
                -1.8934292500410812e-02, -1.9728733743061128e-02, -2.0134116385693790e-02, -2.0339570137597532e-02,
                -2.0443228399470395e-02, -2.0495370663938563e-02, -2.0521546847485805e-02, -2.0534670125856212e-02,
                -2.0541243536506666e-02, -2.0544534176349133e-02
        },
// x = -3.8800
        {
                -1.9021440314213135e-02, -1.9823622924325539e-02, -2.0232997095804367e-02, -2.0440487723615560e-02,
                -2.0545178338122038e-02, -2.0597841470537670e-02, -2.0624279665426390e-02, -2.0637534478657744e-02,
                -2.0644173833991353e-02, -2.0647497505474739e-02
        },
// x = -3.8750
        {
                -1.9108950541817746e-02, -1.9918948750226324e-02, -2.0332353342677828e-02, -2.0541900901116381e-02,
                -2.0647634102691784e-02, -2.0700823290362352e-02, -2.0727526113879235e-02, -2.0740913778560660e-02,
                -2.0747619739537738e-02, -2.0750976774035132e-02
        },
// x = -3.8700
        {
                -1.9196824304071873e-02, -2.0014713035634860e-02, -2.0432187314072196e-02, -2.0643812053249399e-02,
                -2.0750598177090348e-02, -2.0804318658747004e-02, -2.0831288754243622e-02, -2.0844810600115716e-02,
                -2.0851583834312362e-02, -2.0854974566518147e-02
        },
// x = -3.8650
        {
                -1.9285062721198058e-02, -2.0110917601025229e-02, -2.0532501206833854e-02, -2.0746223574125568e-02,
                -2.0854073057171602e-02, -2.0908330123475424e-02, -2.0935570160626613e-02, -2.0949227530712447e-02,
                -2.0956068712386609e-02, -2.0959493480349731e-02
        },
// x = -3.8600
        {
                -1.9373666912738743e-02, -2.0207564272470929e-02, -2.0633297226925591e-02, -2.0849137868862822e-02,
                -2.0958061250787457e-02, -2.1012860244840642e-02, -2.1040372919905435e-02, -2.1054167170642801e-02,
                -2.1061076980801025e-02, -2.1064536125958554e-02
        },
// x = -3.8550
        {
                -1.9462637997500417e-02, -2.0304654881641390e-02, -2.0734577589454507e-02, -2.0952557353631749e-02,
                -2.1062565277843032e-02, -2.1117911595705058e-02, -2.1145699631790187e-02, -2.1159632133165168e-02,
                -2.1166611259630000e-02, -2.1170105126841074e-02
        },
// x = -3.8500
        {
                -1.9551977093496913e-02, -2.0402191265798071e-02, -2.0836344518700044e-02, -2.1056484455701411e-02,
                -2.1167587670352025e-02, -2.1223486761560856e-02, -2.1251552908886799e-02, -2.1265625044568699e-02,
                -2.1272674182046786e-02, -2.1276203119626805e-02
        },
// x = -3.8450
        {
                -1.9641685317892225e-02, -2.0500175267790442e-02, -2.0938600248141929e-02, -2.1160921613485258e-02,
                -2.1273130972492317e-02, -2.1329588340590656e-02, -2.1357935376760409e-02, -2.1372148544237971e-02,
                -2.1379268394388768e-02, -2.1382832754144043e-02
        },
// x = -3.8400
        {
                -1.9731763786942476e-02, -2.0598608736051509e-02, -2.1041347020488110e-02, -2.1265871276587182e-02,
                -2.1379197740661812e-02, -2.1436218943728508e-02, -2.1464849673998875e-02, -2.1479205284717882e-02,
                -2.1486396556223145e-02, -2.1489996693485804e-02
        },
// x = -3.8350
        {
                -1.9822213615937351e-02, -2.0697493524593193e-02, -2.1144587087702636e-02, -2.1371335905847767e-02,
                -2.1485790543534521e-02, -2.1543381194721122e-02, -2.1572298452276768e-02, -2.1586797931778992e-02,
                -2.1594061340412873e-02, -2.1597697614076160e-02
        },
// x = -3.8300
        {
                -1.9913035919140776e-02, -2.0796831493001326e-02, -2.1248322711033548e-02, -2.1477317973390586e-02,
                -2.1592911962116830e-02, -2.1651077730189382e-02, -2.1680284376419535e-02, -2.1694929164483061e-02,
                -2.1702265433182967e-02, -2.1705938205736863e-02
        },
// x = -3.8250
        {
                -2.0004231809730873e-02, -2.0896624506430345e-02, -2.1352556161040741e-02, -2.1583819962668747e-02,
                -2.1700564589804055e-02, -2.1759311199690146e-02, -2.1788810124468028e-02, -2.1803601675248994e-02,
                -2.1811011534187099e-02, -2.1814721171754335e-02
        },
// x = -3.8200
        {
                -2.0095802399739387e-02, -2.0996874435597761e-02, -2.1457289717623809e-02, -2.1690844368511469e-02,
                -2.1808751032437197e-02, -2.1868084265778340e-02, -2.1897878387743389e-02, -2.1912818169919079e-02,
                -2.1920302356574585e-02, -2.1924049228946987e-02
        },
// x = -3.8150
        {
                -2.0187748799990163e-02, -2.1097583156778243e-02, -2.1562525670049799e-02, -2.1798393697170897e-02,
                -2.1917473908359922e-02, -2.1977399604069331e-02, -2.2007491870912117e-02, -2.2022581367825537e-02,
                -2.2030140627057628e-02, -2.2033925107732846e-02
        },
// x = -3.8100
        {
                -2.0280072120037171e-02, -2.1198752551797425e-02, -2.1668266316981037e-02, -2.1906470466368986e-02,
                -2.2026735848475858e-02, -2.2087259903301595e-02, -2.2117653292051617e-02, -2.2132894001857438e-02,
                -2.2140529085978987e-02, -2.2144351552197565e-02
        },
// x = -3.8050
        {
                -2.0372773468101658e-02, -2.1300384508025419e-02, -2.1774513966502845e-02, -2.2015077205344537e-02,
                -2.2136539496305976e-02, -2.2197667865399621e-02, -2.2228365382715880e-02, -2.2243758818527876e-02,
                -2.2251470487379882e-02, -2.2255331320162709e-02
        },
// x = -3.8000
        {
                -2.0465853951008642e-02, -2.1402480918369965e-02, -2.1881270936151213e-02, -2.2124216454900401e-02,
                -2.2246887508046331e-02, -2.2308626205537169e-02, -2.2339630888001647e-02, -2.2355178578041576e-02,
                -2.2362967599068279e-02, -2.2366867183254445e-02
        },
// x = -3.7950
        {
                -2.0559314674122697e-02, -2.1505043681269303e-02, -2.1988539552940528e-02, -2.2233890767450790e-02,
                -2.2357782552626021e-02, -2.2420137652200803e-02, -2.2451452566614755e-02, -2.2467156054362718e-02,
                -2.2475023202687575e-02, -2.2478961926972536e-02
        },
// x = -3.7900
        {
                -2.0653156741282975e-02, -2.1608074700684717e-02, -2.2096322153391166e-02, -2.2344102707068745e-02,
                -2.2469227311765330e-02, -2.2532204947253682e-02, -2.2563833190936917e-02, -2.2579694035283177e-02,
                -2.2587640093785496e-02, -2.2591618350759703e-02
        },
// x = -3.7850
        {
                -2.0747381254737574e-02, -2.1711575886092748e-02, -2.2204621083557093e-02, -2.2454854849533731e-02,
                -2.2581224480034169e-02, -2.2644830845999660e-02, -2.2676775547092732e-02, -2.2692795322491077e-02,
                -2.2700821081883482e-02, -2.2704839268071316e-02
        },
// x = -3.7800
        {
                -2.0841989315077063e-02, -2.1815549152477035e-02, -2.2313438699053448e-02, -2.2566149782379377e-02,
                -2.2693776764910682e-02, -2.2758018117247685e-02, -2.2790282435017035e-02, -2.2806462731639651e-02,
                -2.2814568990546232e-02, -2.2818627506445401e-02
        },
// x = -3.7750
        {
                -2.0936982021167356e-02, -2.1919996420319901e-02, -2.2422777365084027e-02, -2.2677990104941318e-02,
                -2.2806886886840197e-02, -2.2871769543376497e-02, -2.2904356668522620e-02, -2.2920699092416473e-02,
                -2.2928886657451839e-02, -2.2932985907573096e-02
        },
// x = -3.7700
        {
                -2.1032360470081796e-02, -2.2024919615593545e-02, -2.2532639456468762e-02, -2.2790378428405301e-02,
                -2.2920557579294319e-02, -2.2986087920399603e-02, -2.3019001075368233e-02, -2.3035507248613033e-02,
                -2.3043776934462010e-02, -2.3047917327369338e-02
        },
// x = -3.7650
        {
                -2.1128125757032504e-02, -2.2130320669750920e-02, -2.2643027357671156e-02, -2.2903317375855207e-02,
                -2.3034791588830265e-02, -2.3100976058030544e-02, -2.3134218497326900e-02, -2.3150890058194592e-02,
                -2.3159242687692854e-02, -2.3163424636043990e-02
        },
// x = -3.7600
        {
                -2.1224278975300968e-02, -2.2236201519716205e-02, -2.2753943462825683e-02, -2.3016809582321423e-02,
                -2.3149591675150592e-02, -2.3216436779748512e-02, -2.3250011790254613e-02, -2.3266850393370501e-02,
                -2.3275286797585861e-02, -2.3279510718173271e-02
        },
// x = -3.7550
        {
                -2.1320821216167855e-02, -2.2342564107875072e-02, -2.2865390175765041e-02, -2.3130857694829265e-02,
                -2.3264960611162910e-02, -2.3332472922864170e-02, -2.3366383824159308e-02, -2.3383391140664685e-02,
                -2.3391912158979324e-02, -2.3396178472771579e-02
        },
// x = -3.7500
        {
                -2.1417753568842079e-02, -2.2449410382064394e-02, -2.2977369910047545e-02, -2.3245464372447480e-02,
                -2.3380901183040096e-02, -2.3449087338585875e-02, -2.3483337483270195e-02, -2.3500515200986682e-02,
                -2.3509121681180120e-02, -2.3513430813363639e-02
        },
// x = -3.7450
        {
                -2.1515077120389071e-02, -2.2556742295561767e-02, -2.3089885088984272e-02, -2.3360632286336988e-02,
                -2.3497416190280535e-02, -2.3566282892086135e-02, -2.3600875666107388e-02, -2.3618225489702837e-02,
                -2.3626918288035748e-02, -2.3631270668057026e-02
        },
// x = -3.7400
        {
                -2.1612792955658277e-02, -2.2664561807074639e-02, -2.3202938145666256e-02, -2.3476364119799718e-02,
                -2.3614508445768779e-02, -2.3684062462568389e-02, -2.3719001285551947e-02, -2.3736524936708001e-02,
                -2.3745304918006853e-02, -2.3749700979615075e-02
        },
// x = -3.7350
        {
                -2.1710902157209919e-02, -2.2772870880728994e-02, -2.3316531522991655e-02, -2.3592662568327549e-02,
                -2.3732180775836335e-02, -2.3802428943334081e-02, -2.3837717268916134e-02, -2.3855416486497458e-02,
                -2.3864284524240022e-02, -2.3868724705530087e-02
        },
// x = -3.7300
        {
                -2.1809405805240854e-02, -2.2881671486057718e-02, -2.3430667673692743e-02, -2.3709530339651414e-02,
                -2.3850436020322736e-02, -2.3921385241850061e-02, -2.3957026558014117e-02, -2.3974903098239277e-02,
                -2.3983860074640966e-02, -2.3988344818096974e-02
        },
// x = -3.7250
        {
                -2.1908304977509761e-02, -2.2990965597988636e-02, -2.3545349060362972e-02, -2.3826970153790577e-02,
                -2.3969277032636894e-02, -2.4040934279816302e-02, -2.4076932109232962e-02, -2.4094987745847025e-02,
                -2.4104034551948077e-02, -2.4108564304487223e-02
        },
// x = -3.7200
        {
                -2.2007600749261426e-02, -2.3100755196832128e-02, -2.3660578155483881e-02, -2.3944984743101951e-02,
                -2.4088706679818613e-02, -2.4161078993233821e-02, -2.4197436893603933e-02, -2.4215673418052749e-02,
                -2.4224810953806308e-02, -2.4229386166823238e-02
        },
// x = -3.7150
        {
                -2.2107294193150325e-02, -2.3211042268268329e-02, -2.3776357441452015e-02, -2.4063576852329641e-02,
                -2.4208727842600451e-02, -2.4281822332473085e-02, -2.4318543896874226e-02, -2.4336963118480440e-02,
                -2.4346192292841511e-02, -2.4350813422253110e-02
        },
// x = -3.7100
        {
                -2.2207386379163269e-02, -2.3321828803334020e-02, -2.3892689410605697e-02, -2.4182749238654593e-02,
                -2.4329343415469752e-02, -2.4403167262342571e-02, -2.4440256119578955e-02, -2.4458859865719766e-02,
                -2.4468181596735000e-02, -2.4472849103025621e-02
        },
// x = -3.7050
        {
                -2.2307878374541318e-02, -2.3433116798409096e-02, -2.4009576565251788e-02, -2.4302504671744343e-02,
                -2.4450556306730951e-02, -2.4525116762157705e-02, -2.4562576577113467e-02, -2.4581366693400161e-02,
                -2.4590781908298644e-02, -2.4595496256565774e-02
        },
// x = -3.7000
        {
                -2.2408771243700849e-02, -2.3544908255202602e-02, -2.4127021417692390e-02, -2.4422845933802922e-02,
                -2.4572369438568151e-02, -2.4647673825810110e-02, -2.4685508299806150e-02, -2.4704486650265367e-02,
                -2.4713996285550176e-02, -2.4718757945550611e-02
        },
// x = -3.6950
        {
                -2.2510066048153823e-02, -2.3657205180738453e-02, -2.4245026490251450e-02, -2.4543775819620941e-02,
                -2.4694785747107914e-02, -2.4770841461837157e-02, -2.4809054332991413e-02, -2.4828222800248240e-02,
                -2.4837827801789013e-02, -2.4842637247985435e-02
        },
// x = -3.6900
        {
                -2.2611763846427126e-02, -2.3770009587340706e-02, -2.4363594315301291e-02, -2.4665297136625724e-02,
                -2.4817808182482328e-02, -2.4894622693491839e-02, -2.4933217737083117e-02, -2.4952578222545992e-02,
                -2.4962279545672369e-02, -2.4967137257280420e-02
        },
// x = -3.6850
        {
                -2.2713865693981178e-02, -2.3883323492618406e-02, -2.4482727435289064e-02, -2.4787412704931656e-02,
                -2.4941439708892336e-02, -2.5019020558812957e-02, -2.5058001587648319e-02, -2.5077556011695722e-02,
                -2.5087354621291739e-02, -2.5092261082327563e-02
        },
// x = -3.6800
        {
                -2.2816372643127632e-02, -2.3997148919450057e-02, -2.4602428402763152e-02, -2.4910125357390574e-02,
                -2.5065683304671238e-02, -2.5144038110695630e-02, -2.5183408975481388e-02, -2.5203159277650437e-02,
                -2.5213056148249847e-02, -2.5218011847578072e-02
        },
// x = -3.6750
        {
                -2.2919285742946199e-02, -2.4111487895967690e-02, -2.4722699780399454e-02, -2.5033437939642366e-02,
                -2.5190541962348555e-02, -2.5269678416962080e-02, -2.5309443006678454e-02, -2.5329391145855328e-02,
                -2.5339387261737834e-02, -2.5344392693120114e-02
        },
// x = -3.6700
        {
                -2.3022606039200727e-02, -2.4226342455540471e-02, -2.4843544141027667e-02, -2.5157353310165699e-02,
                -2.5316018688714102e-02, -2.5395944560432839e-02, -2.5436106802712223e-02, -2.5456254757324465e-02,
                -2.5466351112613014e-02, -2.5471406774756931e-02
        },
// x = -3.6650
        {
                -2.3126334574254264e-02, -2.4341714636757928e-02, -2.4964964067657351e-02, -2.5281874340328789e-02,
                -2.5442116504882269e-02, -2.5522839638998131e-02, -2.5563403500507145e-02, -2.5583753268717892e-02,
                -2.5593950867476789e-02, -2.5599057264085387e-02
        },
// x = -3.6600
        {
                -2.3230472386983371e-02, -2.4457606483412709e-02, -2.5086962153504051e-02, -2.5407003914440437e-02,
                -2.5568838446356625e-02, -2.5650366765689742e-02, -2.5691336252514938e-02, -2.5711889852419057e-02,
                -2.5722189708753194e-02, -2.5727347348574878e-02
        },
// x = -3.6550
        {
                -2.3335020512691484e-02, -2.4574020044482964e-02, -2.5209541002015251e-02, -2.5532744929801073e-02,
                -2.5696187563094733e-02, -2.5778529068753039e-02, -2.5819908226790516e-02, -2.5840667696612661e-02,
                -2.5851070834767630e-02, -2.5856280231646631e-02
        },
// x = -3.6500
        {
                -2.3439979983021476e-02, -2.4690957374114236e-02, -2.5332703226896236e-02, -2.5659100296754007e-02,
                -2.5824166919573260e-02, -2.5907329691719480e-02, -2.5949122607068165e-02, -2.5970090005362802e-02,
                -2.5980597459826074e-02, -2.5985859132753444e-02
        },
// x = -3.6450
        {
                -2.3545351825867214e-02, -2.4808420531600962e-02, -2.5456451452135907e-02, -2.5786072938736769e-02,
                -2.5952779594853304e-02, -2.6036771793479338e-02, -2.6078982592838217e-02, -2.6100159998691658e-02,
                -2.6110772814294702e-02, -2.6116087287459726e-02
        },
// x = -3.6400
        {
                -2.3651137065284343e-02, -2.4926411581367453e-02, -2.5580788312032466e-02, -2.5913665792332594e-02,
                -2.6082028682645961e-02, -2.6166858548354759e-02, -2.6209491399423995e-02, -2.6230880912658374e-02,
                -2.6241600144679866e-02, -2.6246967947522057e-02
        },
// x = -3.6350
        {
                -2.3757336721400093e-02, -2.5044932592948561e-02, -2.5705716451219027e-02, -2.6041881807322028e-02,
                -2.6211917291378253e-02, -2.6297593146173241e-02, -2.6340652258059174e-02, -2.6362255999438467e-02,
                -2.6373082713708449e-02, -2.6378504380970062e-02
        },
// x = -3.6300
        {
                -2.3863951810322252e-02, -2.5163985640969735e-02, -2.5831238524689114e-02, -2.6170723946734718e-02,
                -2.6342448544259187e-02, -2.6428978792341341e-02, -2.6472468415965487e-02, -2.6494288527403563e-02,
                -2.6505223800408699e-02, -2.6510699872187724e-02
        },
// x = -3.6250
        {
                -2.3970983344047094e-02, -2.5283572805126719e-02, -2.5957357197822047e-02, -2.6300195186901160e-02,
                -2.6473625579346158e-02, -2.6561018707918733e-02, -2.6604943136430775e-02, -2.6626981781201535e-02,
                -2.6638026700191338e-02, -2.6643557721995077e-02
        },
// x = -3.6200
        {
                -2.4078432330366592e-02, -2.5403696170164809e-02, -2.6084075146408280e-02, -2.6430298517504825e-02,
                -2.6605451549611560e-02, -2.6693716129692645e-02, -2.6738079698887498e-02, -2.6760339061837002e-02,
                -2.6771494724931236e-02, -2.6777081247730355e-02
        },
// x = -3.6150
        {
                -2.4186299772774533e-02, -2.5524357825857481e-02, -2.6211395056674549e-02, -2.6561036941634170e-02,
                -2.6737929623009707e-02, -2.6827074310252601e-02, -2.6871881398991503e-02, -2.6894363686752291e-02,
                -2.6905631203049302e-02, -2.6911273783332475e-02
        },
// x = -3.6100
        {
                -2.4294586670371800e-02, -2.5645559866984819e-02, -2.6339319625309006e-02, -2.6692413475834967e-02,
                -2.6871062982543974e-02, -2.6961096518065501e-02, -2.7006351548701266e-02, -2.7029058989908719e-02,
                -2.7040439479594947e-02, -2.7046138679424007e-02
        },
// x = -3.6050
        {
                -2.4403294017770652e-02, -2.5767304393311166e-02, -2.6467851559486159e-02, -2.6824431150162618e-02,
                -2.7004854826334222e-02, -2.7095786037551001e-02, -2.7141493476357428e-02, -2.7164428321868335e-02,
                -2.7175922916328841e-02, -2.7181679303394508e-02
        },
// x = -3.6000
        {
                -2.4512422804998189e-02, -2.5889593509562537e-02, -2.6596993576891756e-02, -2.6957093008234668e-02,
                -2.7139308367684464e-02, -2.7231146169157304e-02, -2.7277310526762789e-02, -2.7300475049876000e-02,
                -2.7312084891806157e-02, -2.7317899039484311e-02
        },
// x = -3.5950
        {
                -2.4621974017398709e-02, -2.6012429325403406e-02, -2.6726748405747531e-02, -2.7090402107283407e-02,
                -2.7274426835150811e-02, -2.7367180229437231e-02, -2.7413806061262602e-02, -2.7437202557941902e-02,
                -2.7448928801460139e-02, -2.7454801288868631e-02
        },
// x = -3.5900
        {
                -2.4731948635535260e-02, -2.6135813955413065e-02, -2.6857118784835838e-02, -2.7224361518208645e-02,
                -2.7410213472609704e-02, -2.7503891551124662e-02, -2.7550983457825313e-02, -2.7574614246924513e-02,
                -2.7586458057686201e-02, -2.7592389469742289e-02
        },
// x = -3.5850
        {
                -2.4842347635090120e-02, -2.6259749519061435e-02, -2.6988107463524161e-02, -2.7358974325630551e-02,
                -2.7546671539326335e-02, -2.7641283483211272e-02, -2.7688846111123652e-02, -2.7712713534613832e-02,
                -2.7724676089926327e-02, -2.7730667017404610e-02
        },
// x = -3.5800
        {
                -2.4953171986764413e-02, -2.6384238140684479e-02, -2.7119717201789517e-02, -2.7494243627942633e-02,
                -2.7683804310023430e-02, -2.7779359391023734e-02, -2.7827397432616099e-02, -2.7851503855815175e-02,
                -2.7863586344753957e-02, -2.7869637384344954e-02
        },
// x = -3.5750
        {
                -2.5064422656176696e-02, -2.6509281949458979e-02, -2.7251950770242703e-02, -2.7630172537364880e-02,
                -2.7821615074950221e-02, -2.7918122656301066e-02, -2.7966640850628791e-02, -2.7990988662433297e-02,
                -2.8003192285959259e-02, -2.8009304040328531e-02
        },
// x = -3.5700
        {
                -2.5176100603760559e-02, -2.6634883079376875e-02, -2.7384810950152452e-02, -2.7766764179996940e-02,
                -2.7960107139951701e-02, -2.8057576677272534e-02, -2.8106579810437723e-02, -2.8131171423556894e-02,
                -2.8143497394634877e-02, -2.8149670472482742e-02
        },
// x = -3.5650
        {
                -2.5288206784661368e-02, -2.6761043669219109e-02, -2.7518300533469455e-02, -2.7904021695871532e-02,
                -2.8099283826538195e-02, -2.8197724868735733e-02, -2.8247217774351456e-02, -2.8272055625543604e-02,
                -2.8284505169262012e-02, -2.8290740185383870e-02
        },
// x = -3.5600
        {
                -2.5400742148631809e-02, -2.6887765862528938e-02, -2.7652422322850197e-02, -2.8041948239007856e-02,
                -2.8239148471955113e-02, -2.8338570662135128e-02, -2.8388558221794095e-02, -2.8413644772105371e-02,
                -2.8426219125797014e-02, -2.8432516701144273e-02
        },
// x = -3.5550
        {
                -2.5513707639926671e-02, -2.7015051807584650e-02, -2.7787179131680769e-02, -2.8180546977465235e-02,
                -2.8379704429253066e-02, -2.8480117505640910e-02, -2.8530604649388759e-02, -2.8555942384394215e-02,
                -2.8568642797758354e-02, -2.8575003559499951e-02
        },
// x = -3.5500
        {
                -2.5627104197196469e-02, -2.7142903657371861e-02, -2.7922573784100388e-02, -2.8319821093396761e-02,
                -2.8520955067358154e-02, -2.8622368864228170e-02, -2.8673360571041411e-02, -2.8698952001088442e-02,
                -2.8711779736314037e-02, -2.8718204317898580e-02
        },
// x = -3.5450
        {
                -2.5740932753380141e-02, -2.7271323569555167e-02, -2.8058609115024932e-02, -2.8459773783103184e-02,
                -2.8662903771142584e-02, -2.8765328219756445e-02, -2.8816829518025049e-02, -2.8842677178479260e-02,
                -2.8855633510369457e-02, -2.8862122551587977e-02
        },
// x = -3.5400
        {
                -2.5855194235596731e-02, -2.7400313706449419e-02, -2.8195287970170246e-02, -2.8600408257086818e-02,
                -2.8805553941495567e-02, -2.8908999071049692e-02, -2.8961015039064349e-02, -2.8987121490557856e-02,
                -2.9000207706655689e-02, -2.9006761853705036e-02
        },
// x = -3.5350
        {
                -2.5969889565036017e-02, -2.7529876234990217e-02, -2.8332613206075292e-02, -2.8741727740105590e-02,
                -2.8948908995394429e-02, -2.9053384933976457e-02, -2.9105920700420691e-02, -2.9132288529102827e-02,
                -2.9145505929818193e-02, -2.9152125835365035e-02
        },
// x = -3.5300
        {
                -2.6085019656848191e-02, -2.7660013326704109e-02, -2.8470587690125219e-02, -2.8883735471227256e-02,
                -2.9092972365976043e-02, -2.9198489341530583e-02, -2.9251550085977580e-02, -2.9278181903768123e-02,
                -2.9291531802505967e-02, -2.9298218125751469e-02
        },
// x = -3.5250
        {
                -2.6200585420032441e-02, -2.7790727157678045e-02, -2.8609214300574226e-02, -2.9026434703883618e-02,
                -2.9237747502608498e-02, -2.9344315843912113e-02, -2.9397906797326463e-02, -2.9424805242171316e-02,
                -2.9438288965461152e-02, -2.9445042372206250e-02
        },
// x = -3.5200
        {
                -2.6316587757324577e-02, -2.7922019908528346e-02, -2.8748495926568248e-02, -2.9169828705925011e-02,
                -2.9383237870963086e-02, -2.9490868008608669e-02, -2.9544994453852978e-02, -2.9572162189982400e-02,
                -2.9585781077609067e-02, -2.9592602240320436e-02
        },
// x = -3.5150
        {
                -2.6433027565083516e-02, -2.8053893764369134e-02, -2.8888435468167620e-02, -2.9313920759674740e-02,
                -2.9529446953086503e-02, -2.9638149420477152e-02, -2.9692816692823580e-02, -2.9720256411012974e-02,
                -2.9734011816148712e-02, -2.9740901414025349e-02
        },
// x = -3.5100
        {
                -2.6549905733176875e-02, -2.8186350914780150e-02, -2.9029035836369364e-02, -2.9458714161983795e-02,
                -2.9676378247473333e-02, -2.9786163681825774e-02, -2.9841377169472615e-02, -2.9869091587305812e-02,
                -2.9882984876643662e-02, -2.9889943595684199e-02
        },
// x = -3.5050
        {
                -2.6667223144865360e-02, -2.8319393553773973e-02, -2.9170299953129527e-02, -2.9604212224285470e-02,
                -2.9824035269138894e-02, -2.9934914412496478e-02, -2.9990679557089773e-02, -3.0018671419225013e-02,
                -3.0032703973113500e-02, -3.0039732506184094e-02
        },
// x = -3.5000
        {
                -2.6784980676686214e-02, -2.8453023879762774e-02, -2.9312230751385226e-02, -2.9750418272650351e-02,
                -2.9972421549692194e-02, -3.0084405249947758e-02, -3.0140727547107952e-02, -3.0168999625546415e-02,
                -3.0183172838125605e-02, -3.0190271885028583e-02
        },
// x = -3.4950
        {
                -2.6903179198335569e-02, -2.8587244095524315e-02, -2.9454831175076582e-02, -2.9897335647841156e-02,
                -3.0121540637409339e-02, -3.0234639849337795e-02, -3.0291524849191544e-02, -3.0320079943548565e-02,
                -3.0334395222887448e-02, -3.0341565490430610e-02
        },
// x = -3.4900
        {
                -2.7021819572549718e-02, -2.8722056408167509e-02, -2.9598104179168409e-02, -3.0044967705367881e-02,
                -3.0271396097307061e-02, -3.0385621883607935e-02, -3.0443075191325178e-02, -3.0471916129104101e-02,
                -3.0486374897339356e-02, -3.0493617099405942e-02
        },
// x = -3.4850
        {
                -2.7140902654985356e-02, -2.8857463029097294e-02, -2.9742052729671876e-02, -3.0193317815542957e-02,
                -3.0421991511216651e-02, -3.0537355043566652e-02, -3.0595382319902817e-02, -3.0624511956771579e-02,
                -3.0639115650247680e-02, -3.0646430507867094e-02
        },
// x = -3.4800
        {
                -2.7260429294098693e-02, -2.8993466173978931e-02, -2.9886679803665786e-02, -3.0342389363536518e-02,
                -3.0573330477858075e-02, -3.0689843037973761e-02, -3.0748449999817309e-02, -3.0777871219887714e-02,

                -3.0792621289298484e-02, -3.0800009530717650e-02
        },
// x = -3.4750
        {
                -2.7380400331023558e-02, -2.9130068062701752e-02, -3.0031988389317883e-02, -3.0492185749431806e-02,
                -3.0725416612914409e-02, -3.0843089593625032e-02, -3.0902282014550340e-02, -3.0931997730660125e-02,
                -3.0946895641191654e-02, -3.0954358001947135e-02
        },
// x = -3.4700
        {
                -2.7500816599448349e-02, -2.9267270919342139e-02, -3.0177981485905819e-02, -3.0642710388280675e-02,
                -3.0878253549106525e-02, -3.0997098455437223e-02, -3.1056882166262859e-02, -3.1086895320260512e-02,
                -3.1101942551735453e-02, -3.1109479774726296e-02
        },
// x = -3.4650
        {
                -2.7621678925491933e-02, -2.9405076972126073e-02, -3.0324662103838024e-02, -3.0793966710159194e-02,
                -3.1031844936268135e-02, -3.1151873386533475e-02, -3.1212254275885885e-02, -3.1242567838918274e-02,
                -3.1257765885941667e-02, -3.1265378721502896e-02
        },
// x = -3.4600
        {
                -2.7742988127578434e-02, -2.9543488453390943e-02, -3.0472033264674365e-02, -3.0945958160223320e-02,
                -3.1186194441420966e-02, -3.1307418168328979e-02, -3.1368402183211720e-02, -3.1399019156014542e-02,
                -3.1414369528121017e-02, -3.1422058734097966e-02
        },
// x = -3.4550
        {
                -2.7864745016310956e-02, -2.9682507599546749e-02, -3.0620098001146574e-02, -3.1098688198764764e-02,
                -3.1341305748850364e-02, -3.1463736600617229e-02, -3.1525329746985686e-02, -3.1556253160176795e-02,
                -3.1571757381979225e-02, -3.1579523723802544e-02
        },
// x = -3.4500
        {
                -2.7986950394344099e-02, -2.9822136651036710e-02, -3.0768859357178543e-02, -3.1252160301266818e-02,
                -3.1497182560181081e-02, -3.1620832501656405e-02, -3.1683040844998171e-02, -3.1714273759373780e-02,
                -3.1729933370713469e-02, -3.1737777621474859e-02
        },
// x = -3.4450
        {
                -2.8109605056255496e-02, -2.9962377852297155e-02, -3.0918320387906390e-02, -3.1406377958460456e-02,
                -3.1653828594453384e-02, -3.1778709708256365e-02, -3.1841539374177205e-02, -3.1873084881011038e-02,
                -3.1888901437109332e-02, -3.1896824377638071e-02
        },
// x = -3.4400
        {
                -2.8232709788416129e-02, -3.0103233451716858e-02, -3.1068484159698276e-02, -3.1561344676380401e-02,
                -3.1811247588199434e-02, -3.1937372075865805e-02, -3.2000829250681428e-02, -3.2032690472026733e-02,
                -3.2048665543638180e-02, -3.2056667962578451e-02
        },
// x = -3.4350
        {
                -2.8356265368859587e-02, -3.0244705701595636e-02, -3.1219353750174150e-02, -3.1717063976421350e-02,
                -3.1969443295519921e-02, -3.2096823478660046e-02, -3.2160914409993496e-02, -3.2193094498988155e-02,
                -3.2209229672555122e-02, -3.2217312366444040e-02
        },
// x = -3.4300
        {
                -2.8480272567150165e-02, -3.0386796858102381e-02, -3.1370932248225111e-02, -3.1873539395394210e-02,
                -3.2128419488161052e-02, -3.2257067809628941e-02, -3.2321798807013975e-02, -3.2354300948188489e-02,
                -3.2370597825997403e-02, -3.2378761599343830e-02
        },
// x = -3.4250
        {
                -2.8604732144249800e-02, -3.0529509181232352e-02, -3.1523222754032729e-02, -3.2030774485582621e-02,
                -3.2288179955591745e-02, -3.2418108980665362e-02, -3.2483486416155567e-02, -3.2516313825744131e-02,
                -3.2532774026083212e-02, -3.2541019691447362e-02
        },
// x = -3.4200
        {
                -2.8729644852383954e-02, -3.0672844934763809e-02, -3.1676228379087942e-02, -3.2188772814799357e-02,
                -3.2448728505081098e-02, -3.2579950922653998e-02, -3.2645981231437880e-02, -3.2679137157692557e-02,
                -3.2695762315011111e-02, -3.2704090693084983e-02
        },
// x = -3.4150
        {
                -2.8855011434906257e-02, -3.0816806386214042e-02, -3.1829952246210054e-02, -3.2347537966443018e-02,
                -3.2610068961776290e-02, -3.2742597585560564e-02, -3.2809287266582624e-02, -3.2842774990090563e-02,
                -3.2859566755159823e-02, -3.2867978674848426e-02
        },
// x = -3.4100
        {
                -2.8980832626162099e-02, -3.0961395806794644e-02, -3.1984397489565150e-02, -3.2507073539554614e-02,
                -3.2772205168780495e-02, -3.2906052938521328e-02, -3.2973408555109225e-02, -3.3007231389112976e-02,
                -3.3024191429188651e-02, -3.3032687727691974e-02
        },
// x = -3.4050
        {
                -2.9107109151350952e-02, -3.1106615471366117e-02, -3.2139567254684497e-02, -3.2667383148874359e-02,
                -3.2935140987231361e-02, -3.3070320969933169e-02, -3.3138349150430894e-02, -3.3172510441151923e-02,
                -3.3189640440138231e-02, -3.3198221963034097e-02
        },
// x = -3.4000
        {
                -2.9233841726387672e-02, -3.1252467658391805e-02, -3.2295464698482754e-02, -3.2828470424898656e-02,
                -3.3098880296379585e-02, -3.3235405687543844e-02, -3.3304113125951190e-02, -3.3338616252916543e-02,
                -3.3355917911531960e-02, -3.3364585512859667e-02
        },
// x = -3.3950
        {
                -2.9361031057762499e-02, -3.1398954649891127e-02, -3.2452092989275740e-02, -3.2990339013936935e-02,
                -3.3263426993667869e-02, -3.3401311118542901e-02, -3.3470704575160951e-02, -3.3505552951533180e-02,
                -3.3523027987477728e-02, -3.3531782529822578e-02
        },
// x = -3.3900
        {
                -2.9488677842399995e-02, -3.1546078731392119e-02, -3.2609455306798193e-02, -3.3152992578168720e-02,
                -3.3428784994810119e-02, -3.3568041309652651e-02, -3.3638127611735756e-02, -3.3673324684646032e-02,
                -3.3690974832770365e-02, -3.3699817187348935e-02
        },
// x = -3.3850
        {
                -2.9616782767516722e-02, -3.1693842191883190e-02, -3.2767554842221172e-02, -3.3316434795700831e-02,
                -3.3594958233870896e-02, -3.3735600327219895e-02, -3.3806386369633859e-02, -3.3841935620518321e-02,
                -3.3859762632994353e-02, -3.3868693679740720e-02
        },
// x = -3.3800
        {
                -2.9745346510477874e-02, -3.1842247323764297e-02, -3.2926394798169202e-02, -3.3480669360624556e-02,
                -3.3761950663345260e-02, -3.3903992257307745e-02, -3.3975485003194444e-02, -3.4011389948134046e-02,
                -3.4029395594627351e-02, -3.4038416222280092e-02
        },
// x = -3.3750
        {
                -2.9874369738652550e-02, -3.1991296422797279e-02, -3.3085978388737267e-02, -3.3645699983072958e-02,
                -3.3929766254238838e-02, -3.4073221205788068e-02, -3.4145427687236538e-02, -3.4181691877299997e-02,
                -3.4199877945143842e-02, -3.4208989051334023e-02
        },
// x = -3.3700
        {
                -3.0003853109267977e-02, -3.2140991788055510e-02, -3.3246308839507417e-02, -3.3811530389278345e-02,
                -3.4098408996148077e-02, -3.4243291298434161e-02, -3.4316218617158209e-02, -3.4352845638748450e-02,
                -3.4371213933119704e-02, -3.4380416424459506e-02
        },
// x = -3.3650
        {
                -3.0133797269262467e-02, -3.2291335721872871e-02, -3.3407389387565274e-02, -3.3978164321629596e-02,
                -3.4267882897341014e-02, -3.4414206681014026e-02, -3.4487862009036384e-02, -3.4524855484240391e-02,
                -3.4543407828336946e-02, -3.4552702620509489e-02
        },
// x = -3.3600
        {
                -3.0264202855137232e-02, -3.2442330529791896e-02, -3.3569223281516195e-02, -3.4145605538729890e-02,
                -3.4438191984838100e-02, -3.4585971519383765e-02, -3.4660362099726928e-02, -3.4697725686669061e-02,
                -3.4716463921889214e-02, -3.4725851939738976e-02
        },
// x = -3.3550
        {
                -3.0395070492806962e-02, -3.2593978520511220e-02, -3.3731813781501099e-02, -3.4313857815454235e-02,
                -3.4609340304493502e-02, -3.4758589999581674e-02, -3.4833723146965520e-02, -3.4871460540164136e-02,
                -3.4890386526287613e-02, -3.4899868703911985e-02
        },
// x = -3.3500
        {
                -3.0526400797449168e-02, -3.2746282005832253e-02, -3.3895164159212184e-02, -3.4482924943007177e-02,
                -3.4781331921076512e-02, -3.4932066327922576e-02, -3.5007949429468582e-02, -3.5046064360196374e-02,
                -3.5065179975567275e-02, -3.5074757256408830e-02
        },
// x = -3.3450
        {
                -3.0658194373352415e-02, -3.2899243300605147e-02, -3.4059277697908369e-02, -3.4652810728980722e-02,
                -3.4954170918353414e-02, -3.5106404731092597e-02, -3.5183045247035041e-02, -3.5221541483682804e-02,
                -3.5240848625394207e-02, -3.5250521962334080e-02
        },
// x = -3.3400
        {
                -3.0790451813763237e-02, -3.3052864722673950e-02, -3.4224157692430285e-02, -3.4823518997412053e-02,
                -3.5127861399169548e-02, -3.5281609456244332e-02, -3.5359014920648399e-02, -3.5397896269092351e-02,
                -3.5417396853172828e-02, -3.5427167208624870e-02
        },
// x = -3.3350
        {
                -3.0923173700731874e-02, -3.3207148592821048e-02, -3.4389807449215146e-02, -3.4995053588841550e-02,
                -3.5302407485531664e-02, -3.5457684771092512e-02, -3.5535862792579219e-02, -3.5575133096552025e-02,
                -3.5594829058154059e-02, -3.5604697404159981e-02
        },
// x = -3.3300
        {
                -3.1056360604956777e-02, -3.3362097234710709e-02, -3.4556230286311301e-02, -3.5167418360370802e-02,
                -3.5477813318690606e-02, -3.5634634964009974e-02, -3.5713593226488344e-02, -3.5753256367953630e-02,
                -3.5773149661543761e-02, -3.5783116979869259e-02
        },
// x = -3.3250
        {
                -3.1190013085627945e-02, -3.3517712974832060e-02, -3.4723429533392532e-02, -3.5340617185720721e-02,
                -3.5654083059224205e-02, -3.5812464344124023e-02, -3.5892210607530214e-02, -3.5932270507060979e-02,
                -3.5952363106611929e-02, -3.5962430388843704e-02
        },
// x = -3.3200
        {
                -3.1324131690268917e-02, -3.3673998142440979e-02, -3.4891408531771993e-02, -3.5514653955289639e-02,
                -3.5831220887120624e-02, -3.5991177241413451e-02, -3.6071719342457086e-02, -3.6112179959617588e-02,
                -3.6132473858802285e-02, -3.6142642106446070e-02
        },
// x = -3.3150
        {
                -3.1458716954577616e-02, -3.3830955069501574e-02, -3.5060170634415931e-02, -3.5689532576211623e-02,
                -3.6009231001861812e-02, -3.6170778006805605e-02, -3.6252123859723415e-02, -3.6292989193454958e-02,
                -3.6313486405842400e-02, -3.6323756630421956e-02
        },
// x = -3.3100
        {
                -3.1593769402265916e-02, -3.3988586090626445e-02, -3.5229719205957015e-02, -3.5865256972414689e-02,
                -3.6188117622507329e-02, -3.6351271012274200e-02, -3.6433428609590833e-02, -3.6474702698601320e-02,
                -3.6495405257854482e-02, -3.6505778481011573e-02
        },
// x = -3.3050
        {
                -3.1729289544898051e-02, -3.4146893543016546e-02, -3.5400057622707558e-02, -3.6041831084679267e-02,
                -3.6367884987778501e-02, -3.6532660650937347e-02, -3.6615638064233687e-02, -3.6657324987391000e-02,
                -3.6678234947466630e-02, -3.6688712201061964e-02
        },
// x = -3.3000
        {
                -3.1865277881727543e-02, -3.4305879766399830e-02, -3.5571189272672117e-02, -3.6219258870696547e-02,
                -3.6548537356142823e-02, -3.6714951337156090e-02, -3.6798756717844994e-02, -3.6840860594574115e-02,
                -3.6861980029924538e-02, -3.6872562356139789e-02
        },
// x = -3.2950
        {
                -3.2001734899533250e-02, -3.4465547102969521e-02, -3.5743117555560110e-02, -3.6397544305126941e-02,
                -3.6730079005898600e-02, -3.6898147506633366e-02, -3.6982789086742886e-02, -3.7025314077427068e-02,
                -3.7046645083203947e-02, -3.7057333534644708e-02
        },
// x = -3.2900
        {
                -3.2138661072453750e-02, -3.4625897897321123e-02, -3.5915845882798024e-02, -3.6576691379658785e-02,
                -3.6912514235260034e-02, -3.7082253616513371e-02, -3.7167739709477524e-02, -3.7210690015863419e-02,
                -3.7232234708123577e-02, -3.7243030347923332e-02
        },
// x = -3.2850
        {
                -3.2276056861820793e-02, -3.4786934496388908e-02, -3.6089377677541112e-02, -3.6756704103066827e-02,
                -3.7095847362442425e-02, -3.7267274145481290e-02, -3.7353613146938670e-02, -3.7396993012545197e-02,
                -3.7418753528458450e-02, -3.7429657430383748e-02
        },
// x = -3.2800
        {
                -3.2413922715991292e-02, -3.4948659249381446e-02, -3.6263716374685055e-02, -3.6937586501270950e-02,
                -3.7280082725747794e-02, -3.7453213593863610e-02, -3.7540413982463550e-02, -3.7584227692995055e-02,
                -3.7606206191054085e-02, -3.7617219439610466e-02
        },
// x = -3.2750
        {
                -3.2552259070178176e-02, -3.5111074507716312e-02, -3.6438865420877105e-02, -3.7119342617394829e-02,
                -3.7465224683650752e-02, -3.7640076483728691e-02, -3.7728146821945294e-02, -3.7772398705708574e-02,
                -3.7794597365940950e-02, -3.7805721056480196e-02
        },
// x = -3.2700
        {
                -3.2691066346279919e-02, -3.5274182624953888e-02, -3.6614828274527000e-02, -3.7301976511824882e-02,
                -3.7651277614884682e-02, -3.7827867358987938e-02, -3.7916816293942009e-02, -3.7961510722267493e-02,
                -3.7983931746449685e-02, -3.7995166985278002e-02
        },
// x = -3.2650
        {
                -3.2830344952708687e-02, -3.5437985956730457e-02, -3.6791608405817473e-02, -3.7485492262268903e-02,
                -3.7838245918528128e-02, -3.8016590785497142e-02, -3.8106427049786097e-02, -3.8151568437453182e-02,
                -3.8174214049326763e-02, -3.8185561953814062e-02
        },
// x = -3.2600
        {
                -3.2970095284217495e-02, -3.5602486860690272e-02, -3.6969209296714552e-02, -3.7669893963815128e-02,
                -3.8026134014091634e-02, -3.8206251351158661e-02, -3.8296983763694385e-02, -3.8342576569360902e-02,
                -3.8365449014850839e-02, -3.8376910713541088e-02
        },
// x = -3.2550
        {
                -3.3110317721725742e-02, -3.5767687696416792e-02, -3.7147634440977281e-02, -3.7855185728991075e-02,
                -3.8214946341604751e-02, -3.8396853666023550e-02, -3.8488491132878506e-02, -3.8534539859514362e-02,
                -3.8557641406949526e-02, -3.8569218039672268e-02
        },
// x = -3.2500
        {
                -3.3251012632143744e-02, -3.5933590825363210e-02, -3.7326887344167371e-02, -3.8041371687822505e-02,
                -3.8404687361703382e-02, -3.8588402362394587e-02, -3.8680953877655945e-02, -3.8727463072981118e-02,
                -3.8750796013316914e-02, -3.8762488731299768e-02
        },
// x = -3.2450
        {
                -3.3392180368195808e-02, -3.6100198610781814e-02, -3.7506971523658318e-02, -3.8228455987892637e-02,
                -3.8595361555717460e-02, -3.8780902094929362e-02, -3.8874376741561537e-02, -3.8921350998488251e-02,
                -3.8944917645531485e-02, -3.8956727611513912e-02
        },
// x = -3.2400
        {
                -3.3533821268242128e-02, -3.6267513417652684e-02, -3.7687890508644246e-02, -3.8416442794400991e-02,
                -3.8786973425758792e-02, -3.8974357540744034e-02, -3.9068764491459511e-02, -3.9116208448538810e-02,
                -3.9140011139174771e-02, -3.9151939527522855e-02
        },
// x = -3.2350
        {
                -3.3675935656099297e-02, -3.6435537612611459e-02, -3.7869647840148304e-02, -3.8605336290222683e-02,
                -3.8979527494809395e-02, -3.9168773399517534e-02, -3.9264121917656054e-02, -3.9312040259528773e-02,
                -3.9336081353950510e-02, -3.9348129350772934e-02
        },
// x = -3.2300
        {
                -3.3818523840859642e-02, -3.6604273563876064e-02, -3.8052247071030838e-02, -3.8795140675967510e-02,
                -3.9173028306809958e-02, -3.9364154393595953e-02, -3.9460453834012331e-02, -3.9508851291864398e-02,
                -3.9533133173804365e-02, -3.9545301977069534e-02
        },
// x = -3.2250
        {
                -3.3961586116709075e-02, -3.6773723641172755e-02, -3.8235691765997094e-02, -3.8985860170039202e-02,
                -3.9367480426748667e-02, -3.9560505268097748e-02, -3.9657765078058162e-02, -3.9706646430080442e-02,
                -3.9731171507044333e-02, -3.9743462326698641e-02
        },
// x = -3.2200
        {
                -3.4105122762743897e-02, -3.6943890215661024e-02, -3.8419985501604539e-02, -3.9177499008694612e-02,
                -3.9562888440750385e-02, -3.9757830791019091e-02, -3.9856060511106121e-02, -3.9905430582958708e-02,
                -3.9930201286461638e-02, -3.9942615344548960e-02
        },
// x = -3.2150
        {
                -3.4249134042786122e-02, -3.7114775659857782e-02, -3.8605131866269862e-02, -3.9370061446103076e-02,
                -3.9759256956165952e-02, -3.9956135753339804e-02, -4.0055345018366141e-02, -4.0105208683647246e-02,
                -4.0130227469452301e-02, -4.0142766000234573e-02
        },
// x = -3.2100
        {
                -3.4393620205197518e-02, -3.7286382347560432e-02, -3.8791134460275621e-02, -3.9563551754405617e-02,
                -3.9956590601661977e-02, -4.0155424969129751e-02, -4.0255623509060749e-02, -4.0305985689780145e-02,
                -4.0331255038139224e-02, -4.0343919288218323e-02
        },
// x = -3.2050
        {
                -3.4538581482692392e-02, -3.7458712653769176e-02, -3.8977996895776301e-02, -3.9757974223774376e-02,
                -4.0154894027310838e-02, -4.0355703275655561e-02, -4.0456900916540771e-02, -4.0507766583597886e-02,
                -4.0533288999494932e-02, -4.0546080227935752e-02
        },
// x = -3.2000
        {
                -3.4684018092149091e-02, -3.7631768954608250e-02, -3.9165722796804359e-02, -3.9953333162471988e-02,
                -4.0354171904680998e-02, -4.0556975533487929e-02, -4.0659182198401568e-02, -4.0710556372068225e-02,
                -4.0736334385464978e-02, -4.0749253863919636e-02
        },
// x = -3.1950
        {
                -3.4829930234420117e-02, -3.7805553627246302e-02, -3.9354315799275411e-02, -4.0149632896910897e-02,
                -4.0554428926927524e-02, -4.0759246626609347e-02, -4.0862472336599884e-02, -4.0914360087007828e-02,
                -4.0940396253091746e-02, -4.0953445265925169e-02
        },
// x = -3.1900
        {
                -3.4976318094140972e-02, -3.7980069049815830e-02, -3.9543779550993391e-02, -4.0346877771712891e-02,
                -4.0755669808883094e-02, -4.0962521462522244e-02, -4.1066776337571061e-02, -4.1119182785204225e-02,
                -4.1145479684639150e-02, -4.1158659529055737e-02
        },
// x = -3.1850
        {
                -3.5123181839537700e-02, -3.8155317601331479e-02, -3.9734117711655158e-02, -4.0545072149768484e-02,
                -4.0957899287149116e-02, -4.1166804972357557e-02, -4.1272099232347065e-02, -4.1325029548538582e-02,
                -4.1351589787717677e-02, -4.1364901773889420e-02
        },
// x = -3.1800
        {
                -3.5270521622233084e-02, -3.8331301661607683e-02, -3.9925333952854664e-02, -4.0744220412296396e-02,
                -4.1161122120187264e-02, -4.1372102110983855e-02, -4.1478446076674697e-02, -4.1531905484108955e-02,
                -4.1558731695410231e-02, -4.1572177146605910e-02
        },
// x = -3.1750
        {
                -3.5418337577051595e-02, -3.8508023611175023e-02, -4.0117431958086751e-02, -4.0944326958903147e-02,
                -4.1365343088411242e-02, -4.1578417857116919e-02, -4.1685821951134808e-02, -4.1739815724354222e-02,
                -4.1766910566398550e-02, -4.1780490819114321e-02
        },
// x = -3.1700
        {
                -3.5566629821822932e-02, -3.8685485831195812e-02, -4.0310415422750524e-02, -4.1145396207642397e-02,
                -4.1570566994278878e-02, -4.1785757213429559e-02, -4.1894231961261574e-02, -4.1948765427178424e-02,
                -4.1976131585090067e-02, -4.1989847989181379e-02
        },
// x = -3.1650
        {
                -3.5715398457184334e-02, -3.8863690703378605e-02, -4.0504288054152328e-02, -4.1347432595074625e-02,
                -4.1776798662384519e-02, -4.1994125206662250e-02, -4.2103681237662692e-02, -4.2158759776075964e-02,
                -4.2186399961745759e-02, -4.2200253880560458e-02
        },
// x = -3.1600
        {
                -3.5864643566381471e-02, -3.9042640609891752e-02, -4.0699053571508229e-02, -4.1550440576326578e-02,
                -4.1984042939551693e-02, -4.2203526887733989e-02, -4.2314174936139981e-02, -4.2369803980257172e-02,
                -4.2397720932608254e-02, -4.2411713743121131e-02
        },
// x = -3.1550
        {
                -3.6014365215068206e-02, -3.9222337933275939e-02, -4.0894715705946078e-02, -4.1754424625150909e-02,
                -4.2192304694926103e-02, -4.2413967331853604e-02, -4.2525718237810498e-02, -4.2581903274774666e-02,
                -4.2610099760030842e-02, -4.2624232852979407e-02
        },
// x = -3.1500
        {
                -3.6164563451104727e-02, -3.9402785056355728e-02, -4.1091278200507203e-02, -4.1959389233985657e-02,
                -4.2401588820068919e-02, -4.2625451638631677e-02, -4.2738316349228311e-02, -4.2795062920650162e-02,
                -4.2823541732606885e-02, -4.2837816512628621e-02
        },
// x = -3.1450
        {
                -3.6315238304354708e-02, -3.9583984362150076e-02, -4.1288744810147562e-02, -4.2165338914013847e-02,
                -4.2611900229050294e-02, -4.2837984932192839e-02, -4.2951974502506818e-02, -4.3009288205002022e-02,
                -4.3038052165300079e-02, -4.3052470051070928e-02
        },
// x = -3.1400
        {
                -3.6466389786480889e-02, -3.9765938233781821e-02, -4.1487119301738491e-02, -4.2372278195223108e-02,
                -4.2823243858543365e-02, -4.3051572361288516e-02, -4.3166697955441487e-02, -4.3224584441173289e-02,
                -4.3253636399575211e-02, -4.3268198823949520e-02
        },
// x = -3.1350
        {
                -3.6618017890739429e-02, -3.9948649054386200e-02, -4.1686405454067044e-02, -4.2580211626465192e-02,
                -4.3035624667918271e-02, -4.3266219099410211e-02, -4.3382491991633416e-02, -4.3440956968860488e-02,
                -4.3470299803529601e-02, -4.3485008213681398e-02
        },
// x = -3.1300
        {
                -3.6770122591773063e-02, -4.0132119207018271e-02, -4.1886607057835812e-02, -4.2789143775515563e-02,
                -4.3249047639336700e-02, -4.3481930344903212e-02, -4.3599361920613228e-02, -4.3658411154242890e-02,
                -4.3688047772025236e-02, -4.3702903629590943e-02
        },
// x = -3.1250
        {
                -3.6922703845402617e-02, -4.0316351074559366e-02, -4.2087727915662176e-02, -4.2999079229133055e-02,
                -4.3463517777846718e-02, -4.3698711321080851e-02, -4.3817313077965603e-02, -4.3876952390112522e-02,
                -4.3906885726821422e-02, -4.3921890508044056e-02
        },
// x = -3.1200
        {
                -3.7075761588417543e-02, -4.0501347039622410e-02, -4.2289771842077349e-02, -4.3210022593119249e-02,
                -4.3679040111477656e-02, -4.3916567276339004e-02, -4.4036350825454498e-02, -4.4096586096004725e-02,
                -4.4126819116708293e-02, -4.4141974312582877e-02
        },
// x = -3.1150
        {
                -3.7229295738364920e-02, -4.0687109484456238e-02, -4.2492742663524770e-02, -4.3421978492378277e-02,
                -4.3895619691335701e-02, -4.4135503484271456e-02, -4.4256480551148780e-02, -4.4317317718329356e-02,
                -4.4347853417640755e-02, -4.4363160534061426e-02
        },
// x = -3.1100
        {
                -3.7383306193337233e-02, -4.0873640790848903e-02, -4.2696644218357928e-02, -4.3634951570976216e-02,
                -4.4113261591699343e-02, -4.4355525243785318e-02, -4.4477707669548505e-02, -4.4539152730502643e-02,
                -4.4569994132873228e-02, -4.4585454690781665e-02
        },
// x = -3.1050
        {
                -3.7537792831758794e-02, -4.1060943340029847e-02, -4.2901480356838009e-02, -4.3848946492200683e-02,
                -4.4331970910115509e-02, -4.4576637879217298e-02, -4.4700037621711819e-02, -4.4762096633079687e-02,
                -4.4793246793095015e-02, -4.4808862328630357e-02
        },
// x = -3.1000
        {
                -3.7692755512170849e-02, -4.1249019512570984e-02, -4.3107254941130609e-02, -4.4063967938620351e-02,
                -4.4551752767495671e-02, -4.4798846740450177e-02, -4.4923475875382327e-02, -4.4986154953887518e-02,
                -4.5017616956566288e-02, -4.5033389021216563e-02
        },
// x = -3.0950
        {
                -3.7848194073015373e-02, -4.1437871688286797e-02, -4.3313971845302446e-02, -4.4280020612144512e-02,
                -4.4772612308212535e-02, -4.5022157203029919e-02, -4.5148027925117198e-02, -4.5211333248158861e-02,
                -4.5243110209254873e-02, -4.5259040370009905e-02
        },
// x = -3.0900
        {
                -3.8004108332417540e-02, -4.1627502246133270e-02, -4.3521634955317148e-02, -4.4497109234082531e-02,
                -4.4994554700196795e-02, -4.5246574668283183e-02, -4.5373699292415706e-02, -4.5437637098666536e-02,
                -4.5469732164973485e-02, -4.5485822004479350e-02
        },
// x = -3.0850
        {
                -3.8160498087966913e-02, -4.1817913564105753e-02, -4.3730248169030782e-02, -4.4715238545203262e-02,
                -4.5217585135034308e-02, -4.5472104563435381e-02, -4.5600495525848481e-02, -4.5665072115858463e-02,
                -4.5697488465517855e-02, -4.5713739582232855e-02
        },
// x = -3.0800
        {
                -3.8317363116497201e-02, -4.2009108019135794e-02, -4.3939815396186750e-02, -4.4934413305794639e-02,
                -4.5441708828063587e-02, -4.5698752341729244e-02, -4.5828422201187255e-02, -4.5893643937993410e-02,
                -4.5926384780805483e-02, -4.5942798789157593e-02
        },
// x = -3.0750
        {
                -3.8474703173864894e-02, -4.2201087986986713e-02, -4.4150340558410307e-02, -4.5154638295722992e-02,
                -4.5666931018473453e-02, -4.5926523482543731e-02, -4.6057484921535254e-02, -4.6123358231277244e-02,
                -4.6156426809014897e-02, -4.6173005339560938e-02
        },
// x = -3.0700
        {
                -3.8632517994726424e-02, -4.2393855842148193e-02, -4.4361827589202298e-02, -4.5375918314492426e-02,
                -4.5893256969401205e-02, -4.6155423491513574e-02, -4.6287689317458212e-02, -4.6354220689999913e-02,
                -4.6387620276725827e-02, -4.6404364976312062e-02
        },
// x = -3.0650
        {
                -3.8790807292314128e-02, -4.2587413957729782e-02, -4.4574280433932661e-02, -4.5598258181304321e-02,
                -4.6120691968030779e-02, -4.6385457900649310e-02, -4.6519041047115926e-02, -4.6586237036673236e-02,
                -4.6619970939059979e-02, -4.6636883470984374e-02
        },
// x = -3.0600
        {
                -3.8949570758210740e-02, -4.2781764705353130e-02, -4.4787703049833164e-02, -4.5821662735116467e-02,
                -4.6349241325691549e-02, -4.6616632268457678e-02, -4.6751545796394428e-02, -4.6819413022168979e-02,
                -4.6853484579822413e-02, -4.6870566623998508e-02
        },
// x = -3.0550
        {
                -3.9108808062122843e-02, -4.2976910455043212e-02, -4.5002099405989780e-02, -4.6046136834702496e-02,
                -4.6578910377957188e-02, -4.6848952180062610e-02, -4.6985209279038757e-02, -4.7053754425857984e-02,
                -4.7088167011643804e-02, -4.7105420264766162e-02
        },
// x = -3.0500
        {
                -3.9268518851652764e-02, -4.3172853575118353e-02, -4.5217473483334381e-02, -4.6271685358711082e-02,
                -4.6809704484744905e-02, -4.7082423247326824e-02, -4.7220037236786395e-02, -4.7289267055749727e-02,
                -4.7324024076123149e-02, -4.7341450251834519e-02
        },
// x = -3.0450
        {
                -3.9428702752069278e-02, -4.3369596432079188e-02, -4.5433829274635948e-02, -4.6498313205725172e-02,
                -4.7041629030414943e-02, -4.7317051108973579e-02, -4.7456035439501248e-02, -4.7525956748632629e-02,
                -4.7561061643971426e-02, -4.7578662473031462e-02
        },
// x = -3.0400
        {
                -3.9589359366077104e-02, -4.3567141390496361e-02, -4.5651170784491123e-02, -4.6726025294321107e-02,
                -4.7274689423870420e-02, -4.7552841430709314e-02, -4.7693209685308185e-02, -4.7763829370215040e-02,
                -4.7799285615155770e-02, -4.7817062845611553e-02
        },
// x = -3.0350
        {
                -3.9750488273584871e-02, -4.3765490812897129e-02, -4.5869502029314405e-02, -4.6954826563127793e-02,
                -4.7508891098657455e-02, -4.7789799905346528e-02, -4.7931565800728396e-02, -4.8002890815266895e-02,
                -4.8038701919044462e-02, -4.8056657316402622e-02
        },
// x = -3.0300
        {
                -3.9912089031472045e-02, -4.3964647059650856e-02, -4.6088827037327564e-02, -4.7184721970885656e-02,
                -4.7744239513065577e-02, -4.8027932252927286e-02, -4.8171109640815120e-02, -4.8243147007761930e-02,
                -4.8279316514552618e-02, -4.8297451861953132e-02
        },
// x = -3.0250
        {
                -4.0074161173354453e-02, -4.4164612488853221e-02, -4.6309149848548531e-02, -4.7415716496505819e-02,
                -4.7980740150228372e-02, -4.8267244220847140e-02, -4.8411847089290132e-02, -4.8484603901020829e-02,
                -4.8521135390288468e-02, -4.8539452488680281e-02
        },
// x = -3.0200
        {
                -4.0236704209348446e-02, -4.4365389456209298e-02, -4.6530474514779901e-02, -4.7647815139128920e-02,
                -4.8218398518224584e-02, -4.8507741583979642e-02, -4.8653784058680823e-02, -4.8727267477854830e-02,
                -4.8764164564700542e-02, -4.8782665233018892e-02
        },
// x = -3.0150
        {
                -4.0399717625833970e-02, -4.4566980314915572e-02, -4.6752805099596416e-02, -4.7881022918184017e-02,
                -4.8457220150179330e-02, -4.8749430144801334e-02, -4.8896926490457931e-02, -4.8971143750710086e-02,
                -4.9008410086225473e-02, -4.9027096161570881e-02
        },
// x = -3.0100
        {
                -4.0563200885216269e-02, -4.4769387415540540e-02, -4.6976145678332443e-02, -4.8115344873447492e-02,
                -4.8697210604365701e-02, -4.8992315733517179e-02, -4.9141280355173851e-02, -4.9216238761812747e-02,
                -4.9253878033436470e-02, -4.9272751371255669e-02
        },
// x = -3.0050
        {
                -4.0727153425686197e-02, -4.4972613105904301e-02, -4.7200500338068281e-02, -4.8350786065101743e-02,
                -4.8938375464306724e-02, -4.9236404208186577e-02, -4.9386851652601545e-02, -4.9462558583314768e-02,
                -4.9500574515192672e-02, -4.9519636989461206e-02
        },
// x = -3.0000
        {
                -4.0891574660979521e-02, -4.5176659730956886e-02, -4.7425873177616221e-02, -4.8587351573793944e-02,
                -4.9180720338877396e-02, -4.9481701454849952e-02, -4.9633646411874219e-02, -4.9710109317440285e-02,
                -4.9748505670789062e-02, -4.9767759174195708e-02
        },
// x = -2.9950
        {
                -4.1056463980134690e-02, -4.5381529632655321e-02, -4.7652268307505852e-02, -4.8825046500694573e-02,
                -4.9424250862407267e-02, -4.9728213387655609e-02, -4.9881670691625461e-02, -4.9958897096632739e-02,
                -4.9997677670107245e-02, -5.0017124114240329e-02
        },
// x = -2.9900
        {
                -4.1221820747249574e-02, -4.5587225149839629e-02, -4.7879689849968828e-02, -4.9063875967556056e-02,
                -4.9668972694783195e-02, -4.9975945948987424e-02, -5.0130930580130163e-02, -5.0208928083702885e-02,
                -5.0248096713766906e-02, -5.0267738029302386e-02
        },
// x = -2.9850
        {
                -4.1387644301236884e-02, -4.5793748618107422e-02, -4.8108141938922967e-02, -4.9303845116771182e-02,
                -4.9914891521552346e-02, -5.0224905109592738e-02, -5.0381432195445941e-02, -5.0460208471977151e-02,
                -5.0499769033277998e-02, -5.0519607170169442e-02
        },
// x = -2.9800
        {
                -4.1553933955578166e-02, -4.6001102369687394e-02, -4.8337628719955693e-02, -4.9544959111431496e-02,
                -5.0162013054025581e-02, -5.0475096868711086e-02, -5.0633181685555403e-02, -5.0712744485447062e-02,
                -5.0752700891193693e-02, -5.0772737818864183e-02
        },
// x = -2.9750
        {

                -4.1720688998076870e-02, -4.6209288733311628e-02, -4.8568154350306862e-02, -4.9787223135385687e-02,
                -5.0410343029381070e-02, -5.0726527254203110e-02, -5.0886185228508837e-02, -5.0966542378919177e-02,
                -5.1006898581264157e-02, -5.1027136288799985e-02
        },
// x = -2.9700
        {
                -4.1887908690610001e-02, -4.6418310034086412e-02, -4.8799722998850999e-02, -5.0030642393297672e-02,
                -5.0659887210768292e-02, -5.0979202322680259e-02, -5.1140449032567564e-02, -5.1221608438165735e-02,
                -5.1262368428590867e-02, -5.1282808924937258e-02
        },
// x = -2.9650
        {
                -4.2055592268878499e-02, -4.6628168593362149e-02, -4.9032338846078848e-02, -5.0275222110704787e-02,
                -5.0910651387412131e-02, -5.1233128159634843e-02, -5.1395979336348235e-02, -5.1477948980076113e-02,
                -5.1519116789781938e-02, -5.1539762103940658e-02
        },
// x = -2.9600
        {
                -4.2223738942156523e-02, -4.6838866728601762e-02, -4.9266006084078197e-02, -5.0520967534075839e-02,
                -5.1162641374717523e-02, -5.1488310879570759e-02, -5.1652782408967347e-02, -5.1735570352808985e-02,
                -5.1777150053107991e-02, -5.1798002234336941e-02
        },
// x = -2.9550
        {
                -4.2392347893039438e-02, -4.7050406753247953e-02, -4.9500728916514171e-02, -5.0767883930869048e-02,
                -5.1415863014374168e-02, -5.1744756626134543e-02, -5.1910864550186779e-02, -5.1994478935945145e-02,
                -5.2036474638659021e-02, -5.2057535756673733e-02
        },
// x = -2.9500
        {
                -4.2561418277190698e-02, -4.7262790976589170e-02, -4.9736511558608823e-02, -5.1015976589589854e-02,
                -5.1670322174461665e-02, -5.2002471572247154e-02, -5.2170232090559723e-02, -5.2254681140641142e-02,
                -5.2297096998501706e-02, -5.2318369143678961e-02
        },
// x = -2.9450
        {
                -4.2730949223087365e-02, -4.7476021703624360e-02, -4.9973358237119882e-02, -5.1265250819848611e-02,
                -5.1926024749554871e-02, -5.2261461920236126e-02, -5.2430891391577486e-02, -5.2516183409783514e-02,
                -5.2559023616837827e-02, -5.2580508900421201e-02
        },
// x = -2.9400
        {
                -4.2900939831764573e-02, -4.7690101234926306e-02, -5.0211273190319157e-02, -5.1515711952418317e-02,
                -5.2182976660829578e-02, -5.2521733901968272e-02, -5.2692848845816757e-02, -5.2778992218143972e-02,
                -5.2822261010163206e-02, -5.2843961564470625e-02
        },
// x = -2.9350
        {
                -4.3071389176558783e-02, -4.7905031866503828e-02, -5.0450260667969947e-02, -5.1767365339292000e-02,
                -5.2441183856168465e-02, -5.2783293778983109e-02, -5.2956110877087730e-02, -5.3043114072535077e-02,
                -5.3086815727427607e-02, -5.3108733706061007e-02
        },
// x = -2.9300
        {
                -4.3242296302849779e-02, -4.8120815889662681e-02, -5.0690324931303848e-02, -5.2020216353740253e-02,
                -5.2700652310267340e-02, -5.3046147842626477e-02, -5.3220683940582721e-02, -5.3308555511966872e-02,
                -5.3352694350195240e-02, -5.3374831928252320e-02
        },
// x = -2.9250
        {
                -4.3413660227801652e-02, -4.8337455590865074e-02, -5.0931470252996998e-02, -5.2274270390368416e-02,
                -5.2961388024741728e-02, -5.3310302414184989e-02, -5.3486574523025604e-02, -5.3575323107804135e-02,
                -5.3619903492806223e-02, -5.3642262867094267e-02
        },
// x = -2.9200
        {
                -4.3585479940102471e-02, -4.8554953251587943e-02, -5.1173700917145404e-02, -5.2529532865173756e-02,
                -5.3223397028233546e-02, -5.3575763845020830e-02, -5.3753789142821778e-02, -5.3843423463924373e-02,
                -5.3888449802538693e-02, -5.3911033191790504e-02
        },
// x = -2.9150
        {
                -4.3757754399702843e-02, -4.8773311148179947e-02, -5.1417021219239682e-02, -5.2786009215602551e-02,
                -5.3486685376518385e-02, -5.3842538516707207e-02, -5.4022334350208957e-02, -5.4112863216876646e-02,
                -5.4158339959771790e-02, -5.4181149604863740e-02
        },
// x = -2.9100
        {
                -4.3930482537553453e-02, -4.8992531551717088e-02, -5.1661435466139147e-02, -5.3043704900606940e-02,
                -5.3751259152612745e-02, -5.4110632841164255e-02, -5.4292216727408520e-02, -5.4383649036041107e-02,
                -5.4429580678149396e-02, -5.4452618842321686e-02
        },
// x = -2.9050
        {
                -4.4103663255341263e-02, -4.9212616727857081e-02, -5.1906947976044927e-02, -5.3302625400701749e-02,
                -5.4017124466881804e-02, -5.4380053260795529e-02, -5.4563442888777611e-02, -5.4655787623789309e-02,
                -5.4702178704744746e-02, -5.4725447673823820e-02
        },
// x = -2.9000
        {
                -4.4277295425224759e-02, -4.9433568936692374e-02, -5.2153563078472603e-02, -5.3562776218021045e-02,
                -5.4284287457147355e-02, -5.4650806248624964e-02, -5.4836019480961860e-02, -5.4929285715645197e-02,
                -5.4976140820225700e-02, -5.4999642902848818e-02
        },
// x = -2.8950
        {
                -4.4451377889568058e-02, -4.9655390432601837e-02, -5.2401285114224032e-02, -5.3824162876374801e-02,
                -5.4552754288795961e-02, -5.4922898308434526e-02, -5.5109953183048850e-02, -5.5204150080447038e-02,
                -5.5251473839021001e-02, -5.5275211366863121e-02
        },
// x = -2.8900
        {
                -4.4625909460673865e-02, -4.9878083464101067e-02, -5.2650118435358349e-02, -5.4086790921305063e-02,
                -5.4822531154887610e-02, -5.5196335974902207e-02, -5.5385250706722323e-02, -5.5480387520509926e-02,
                -5.5528184609487150e-02, -5.5552159937490013e-02
        },
// x = -2.8850
        {
                -4.4800888920515272e-02, -5.0101650273691567e-02, -5.2900067405162288e-02, -5.4350665920142288e-02,
                -5.5093624276264401e-02, -5.5471125813740745e-02, -5.5661918796416854e-02, -5.5758004871789231e-02,
                -5.5806280014076330e-02, -5.5830495520679715e-02
        },
// x = -2.8800
        {
                -4.4976315020466680e-02, -5.0326093097708191e-02, -5.3151136398119726e-02, -5.4615793462061392e-02,
                -5.5366039901659690e-02, -5.5747274421836739e-02, -5.5939964229473498e-02, -5.6037009004044618e-02,
                -5.6085766969504888e-02, -5.6110225056880340e-02
        },
// x = -2.8750
        {
                -4.5152186481033536e-02, -5.0551414166165723e-02, -5.3403329799880543e-02, -5.4882179158137624e-02,
                -5.5639784307807458e-02, -5.6024788427390448e-02, -5.6219393816295975e-02, -5.6317406821005139e-02,
                -5.6366652426922813e-02, -5.6391355521209573e-02
        },
// x = -2.8700
        {
                -4.5328501991580887e-02, -5.0777615702603654e-02, -5.3656652007228481e-02, -5.5149828641402318e-02,
                -5.5914863799551920e-02, -5.6303674490056001e-02, -5.6500214400507585e-02, -5.6599205260534802e-02,
                -5.6648943372084065e-02, -5.6673893923627310e-02
        },
// x = -2.8650
        {
                -4.5505260210061210e-02, -5.1004699923929912e-02, -5.3911107428048513e-02, -5.5418747566898498e-02,
                -5.6191284709957399e-02, -5.6583939301082242e-02, -5.6782432859108813e-02, -5.6882411294799082e-02,
                -5.6932646825517567e-02, -5.6957847309109022e-02
        },
// x = -2.8600
        {
                -4.5682459762740810e-02, -5.1232669040263128e-02, -5.4166700481293260e-02, -5.5688941611736320e-02,
                -5.6469053400418655e-02, -5.6865589583454168e-02, -5.7066056102635647e-02, -5.7167031930432308e-02,
                -5.7217769842699169e-02, -5.7243222757820199e-02
        },
// x = -2.8550
        {
                -4.5860099243925562e-02, -5.1461525254773370e-02, -5.4423435596948580e-02, -5.5960416475148260e-02,
                -5.6748176260771242e-02, -5.7148632092034760e-02, -5.7351091075318668e-02, -5.7453074208705573e-02,
                -5.7504319514224439e-02, -5.7530027385291380e-02
        },
// x = -2.8500
        {
                -4.6038177215685441e-02, -5.1691270763521868e-02, -5.4681317215998532e-02, -5.6233177878544267e-02,
                -5.7028659709402296e-02, -5.7433073613707517e-02, -5.7637544755242777e-02, -5.7740545205695763e-02,
                -5.7792302965982228e-02, -5.7818268342594271e-02
        },
// x = -2.8450
        {
                -4.6216692207578111e-02, -5.1921907755299007e-02, -5.4940349790389344e-02, -5.6507231565566633e-02,
                -5.7310510193361586e-02, -5.7718920967519481e-02, -5.7925424154507611e-02, -5.8029452032455110e-02,
                -5.8081727359329142e-02, -5.8107952816518635e-02
        },
// x = -2.8400
        {
                -4.6395642716371394e-02, -5.2153438411461056e-02, -5.5200537782992679e-02, -5.6782583302144654e-02,
                -5.7593734188472724e-02, -5.8006181004824860e-02, -5.8214736319388748e-02, -5.8319801835181713e-02,
                -5.8372599891264851e-02, -5.8399088029750003e-02
        },
// x = -2.8350
        {
                -4.6575027205765167e-02, -5.2385864905765633e-02, -5.5461885667568045e-02, -5.7059238876549311e-02,
                -5.7878338199444920e-02, -5.8294860609429122e-02, -5.8505488330499636e-02, -5.8611601795390877e-02,
                -5.8664927794608211e-02, -5.8691681241048378e-02
        },
// x = -2.8300
        {
                -4.6754844106111845e-02, -5.2619189404205582e-02, -5.5724397928724405e-02, -5.7337204099447453e-02,
                -5.8164328759984570e-02, -5.8584966697733727e-02, -5.8797687302954149e-02, -5.8904859130087112e-02,
                -5.8958718338174350e-02, -5.8985739745427854e-02
        },
// x = -2.8250
        {
                -4.6935091814136159e-02, -5.2853414064841502e-02, -5.5988079061880934e-02, -5.7616484803956043e-02,
                -5.8451712432907581e-02, -5.8876506218881401e-02, -5.9091340386529960e-02, -5.9199581091937112e-02,
                -5.9253978826952443e-02, -5.9281270874336878e-02
        },
// x = -2.8200
        {
                -4.7115768692654061e-02, -5.3088541037633073e-02, -5.6252933573226957e-02, -5.7897086845696076e-02,
                -5.8740495810251668e-02, -5.9169486154901973e-02, -5.9386454765832666e-02, -5.9495774969443437e-02,
                -5.9550716602284566e-02, -5.9578281995839699e-02
        },
// x = -2.8150
        {
                -4.7296873070290540e-02, -5.3324572464268589e-02, -5.6518965979680903e-02, -5.8179016102846322e-02,
                -5.9030685513389060e-02, -5.9463913520858697e-02, -5.9683037660460465e-02, -5.9793448087119123e-02,
                -5.9848939042045253e-02, -5.9876780514798493e-02
        },
// x = -2.8100
        {
                -4.7478403241196676e-02, -5.3561510477993422e-02, -5.6786180808848681e-02, -5.8462278476196915e-02,
                -5.9322288193139311e-02, -5.9759795364995330e-02, -5.9981096325169805e-02, -6.0092607805662973e-02,
                -6.0148653560821955e-02, -6.0176773873056554e-02
        },
// x = -2.8050
        {
                -4.7660357464765844e-02, -5.3799357203437002e-02, -5.7054582598980941e-02, -5.8746879889202595e-02,
                -5.9615310529882622e-02, -6.0057138768883522e-02, -6.0280638050041573e-02, -6.0393261522135880e-02,
                -6.0449867610096492e-02, -6.0478269549622259e-02
        },
// x = -2.8000
        {
                -4.7842733965348952e-02, -5.4038114756438153e-02, -5.7324175898929568e-02, -5.9032826288035911e-02,
                -5.9909759233673199e-02, -6.0355950847571012e-02, -6.0581670160648186e-02, -6.0695416670137732e-02,
                -6.0752588678427248e-02, -6.0781275060853963e-02
        },
// x = -2.7950
        {
                -4.8025530931968839e-02, -5.4277785243869288e-02, -5.7594965268103301e-02, -5.9320123641640098e-02,
                -6.0205641044353081e-02, -6.0656238749730182e-02, -6.0884200018221299e-02, -6.0999080719985409e-02,
                -6.1056824291632318e-02, -6.1085797960645991e-02
        },
// x = -2.7900
        {
                -4.8208746518033994e-02, -5.4518370763458973e-02, -5.7866955276422459e-02, -5.9608777941781703e-02,
                -6.0502962731666006e-02, -6.0958009657807358e-02, -6.1188235019820236e-02, -6.1304261178891421e-02,
                -6.1362582012973566e-02, -6.1391845840615196e-02
        },
// x = -2.7850
        {
                -4.8392378841051277e-02, -5.4759873403613289e-02, -5.8140150504272771e-02, -5.9898795203103067e-02,
                -6.0801731095371687e-02, -6.1261270788172575e-02, -6.1493782598501377e-02, -6.1610965591143499e-02,
                -6.1669869443341511e-02, -6.1699426330288770e-02
        },
// x = -2.7800
        {
                -4.8576425982338051e-02, -5.5002295243235397e-02, -5.8414555542458328e-02, -6.0190181463174494e-02,
                -6.1101952965360418e-02, -6.1566029391269962e-02, -6.1800850223488045e-02, -6.1919201538285021e-02,
                -6.1978694221441172e-02, -6.2008547097292803e-02
        },
// x = -2.7750
        {
                -4.8760885986733261e-02, -5.5245638351543967e-02, -5.8690174992153579e-02, -6.0482942782546352e-02,
                -6.1403635201767753e-02, -6.1872292751768629e-02, -6.2109445400341266e-02, -6.2228976639296232e-02,
                -6.2289064023978810e-02, -6.2319215847541881e-02
        },
// x = -2.7700
        {
                -4.8945756862308121e-02, -5.5489904787890142e-02, -5.8967013464854480e-02, -6.0777085244800570e-02,
                -6.1706784695089500e-02, -6.2180068188714223e-02, -6.2419575671131322e-02, -6.2540298550776377e-02,
                -6.2600986565849587e-02, -6.2631440325429458e-02
        },
// x = -2.7650
        {
                -4.9131036580075720e-02, -5.5735096601572852e-02, -5.9245075582328686e-02, -6.1072614956602371e-02,
                -6.2011408366297142e-02, -6.2489363055681056e-02, -6.2731248614609963e-02, -6.2853174967126724e-02,
                -6.2914469600326012e-02, -6.2945228314019344e-02
        },
// x = -2.7600
        {
                -4.9316723073700094e-02, -5.5981215831652845e-02, -5.9524365976564866e-02, -6.1369538047751311e-02,
                -6.2317513166953248e-02, -6.2800184740924705e-02, -6.3044471846383426e-02, -6.3167613620734350e-02,
                -6.3229520919247564e-02, -6.3260587635238077e-02
        },
// x = -2.7550
        {
                -4.9502814239204704e-02, -5.6228264506765165e-02, -5.9804889289720949e-02, -6.1667860671232415e-02,
                -6.2625106079327364e-02, -6.3112540667535161e-02, -6.3359253019086292e-02, -6.3483622282156843e-02,
                -6.3546148353211027e-02, -6.3577526150068175e-02
        },
// x = -2.7500
        {
                -4.9689307934679926e-02, -5.6476244644930389e-02, -6.0086650174071631e-02, -6.1967589003266724e-02,
                -6.2934194116511979e-02, -6.3426438293591028e-02, -6.3675599822555995e-02, -6.3801208760307870e-02,
                -6.3864359771761650e-02, -6.3896051758742411e-02
        },
// x = -2.7450
        {
                -4.9876201979990209e-02, -5.6725158253363911e-02, -6.0369653291954740e-02, -6.2268729243361895e-02,
                -6.3244784322538911e-02, -6.3741885112313421e-02, -6.3993519984008110e-02, -6.4120380902643700e-02,
                -6.4184163083585682e-02, -6.4216172400939131e-02
        },
// x = -2.7400
        {
                -5.0063494156480426e-02, -5.6975007328284477e-02, -6.0653903315716809e-02, -6.2571287614362286e-02,
                -6.3556883772495881e-02, -6.4058888652221313e-02, -6.4313021268212561e-02, -6.4441146595350537e-02,
                -6.4505566236703324e-02, -6.4537896055978330e-02
        },
// x = -2.7350
        {
                -5.0251182206681552e-02, -5.7225793854720486e-02, -6.0939404927657495e-02, -6.2875270362498917e-02,
                -6.3870499572643222e-02, -6.4377456477287057e-02, -6.4634111477670370e-02, -6.4763513763532551e-02,
                -6.4828577218662933e-02, -6.4861230743018805e-02
        },
// x = -2.7300
        {
                -5.0439263834016007e-02, -5.7477519806315443e-02, -6.1226162819973326e-02, -6.3180683757439016e-02,
                -6.4185638860531025e-02, -6.4697596187092352e-02, -6.4956798452791439e-02, -6.5087490371401338e-02,
                -6.5153204056736039e-02, -6.5186184521256366e-02
        },
// x = -2.7250
        {
                -5.0627736702502188e-02, -5.7730187145131520e-02, -6.1514181694700083e-02, -6.3487534092335526e-02,
                -6.4502308805116343e-02, -6.5019315416985191e-02, -6.5281090072072862e-02, -6.5413084422465392e-02,
                -6.5479454818113472e-02, -6.5512765490122873e-02
        },
// x = -2.7200
        {
                -5.0816598436458554e-02, -5.7983797821451889e-02, -6.1803466263654572e-02, -6.3795827683875983e-02,
                -6.4820516606880901e-02, -6.5342621838237150e-02, -6.5606994252278339e-02, -6.5740303959721566e-02,
                -6.5807337610102254e-02, -6.5840981789486386e-02
        },
// x = -2.7150
        {
                -5.1005846620207122e-02, -5.8238353773581407e-02, -6.2094021248375063e-02, -6.4105570872331250e-02,
                -6.5140269497948725e-02, -6.5667523158201166e-02, -6.5934518948617876e-02, -6.6069157065846346e-02,
                -6.6136860580323392e-02, -6.6170841599852148e-02
        },
// x = -2.7100
        {
                -5.1195478797776625e-02, -5.8493856927645992e-02, -6.2385851380061051e-02, -6.4416770021604230e-02,
                -6.5461574742204348e-02, -6.5994027120470242e-02, -6.6263672154928940e-02, -6.6399651863388962e-02,
                -6.6468031916910947e-02, -6.6502353142564738e-02
        },
// x = -2.7050
        {
                -5.1385492472605071e-02, -5.8750309197390299e-02, -6.2678961399511790e-02, -6.4729431519277694e-02,
                -6.5784439635410891e-02, -6.6322141505036472e-02, -6.6594461903857829e-02, -6.6731796514964623e-02,
                -6.6800859848711869e-02, -6.6835524680011096e-02
        },
// x = -2.7000
        {
                -5.1575885107241794e-02, -5.9007712483974101e-02, -6.2973356057063820e-02, -6.5043561776662415e-02,
                -6.6108871505328765e-02, -6.6651874128450569e-02, -6.6926896267042096e-02, -6.7065599223449235e-02,
                -6.7135352645486659e-02, -6.7170364515824657e-02
        },
// x = -2.6950
        {
                -5.1766654123049191e-02, -5.9266068675767052e-02, -6.3269040112527694e-02, -6.5359167228844439e-02,
                -6.6434877711834261e-02, -6.6983232843982388e-02, -6.7260983355293902e-02, -6.7401068232174760e-02,
                -6.7471518618111265e-02, -6.7506880995090196e-02
        },
// x = -2.6900
        {
                -5.1957796899904124e-02, -5.9525379648142050e-02, -6.3566018335123292e-02, -6.5676254334732495e-02,
                -6.6762465647038818e-02, -6.7316225541781710e-02, -6.7596731318783757e-02, -6.7738211825125474e-02,
                -6.7809366118779912e-02, -6.7845082504550291e-02
        },
// x = -2.6850
        {
                -5.2149310775898683e-02, -5.9785647263267086e-02, -6.3864295503414520e-02, -6.5994829577104758e-02,
                -6.7091642735407853e-02, -6.7650860149039690e-02, -6.7934148347225526e-02, -6.8077038327135367e-02,
                -6.8148903541208755e-02, -6.8184977472812047e-02
        },
// x = -2.6800
        {
                -5.2341193047040963e-02, -6.0046873369895407e-02, -6.4163876405242654e-02, -6.6314899462655438e-02,
                -6.7422416433880625e-02, -6.7987144630150959e-02, -6.8273242670061934e-02, -6.8417556104086272e-02,
                -6.8490139320840665e-02, -6.8526574370555496e-02
        },
// x = -2.6750
        {
                -5.2533440966955107e-02, -6.0309059803154548e-02, -6.4464765837658713e-02, -6.6636470522041066e-02,
                -6.7754794231989587e-02, -6.8325086986876185e-02, -6.8614022556651083e-02, -6.8759773563106794e-02,
                -6.8833081935050888e-02, -6.8869881710742514e-02
        },
// x = -2.6700
        {
                -5.2726051746581598e-02, -6.0572208384333609e-02, -6.4766968606854922e-02, -6.6959549309926272e-02,
                -6.8088783651980608e-02, -6.8664695258505548e-02, -6.8956496316453564e-02, -6.9103699152772660e-02,
                -6.9177739903353949e-02, -6.9214908048827239e-02
        },
// x = -2.6650
        {
                -5.2919022553876545e-02, -6.0836320920668985e-02, -6.5070489528094883e-02, -6.7284142405029293e-02,
                -6.8424392248932986e-02, -6.9005977522022519e-02, -6.9300672299220628e-02, -6.9449341363307493e-02,
                -6.9524121787611179e-02, -6.9561661982967080e-02
        },
// x = -2.6600
        {
                -5.3112350513511583e-02, -6.1101399205128790e-02, -6.5375333425642709e-02, -6.7610256410167366e-02,
                -6.8761627610879728e-02, -6.9348941892268159e-02, -6.9646558895183008e-02, -6.9796708726784756e-02,
                -6.9872236192239687e-02, -6.9910152154235031e-02
        },
// x = -2.6550
        {
                -5.3306032706572831e-02, -6.1367445016195656e-02, -6.5681505132691415e-02, -6.7937897952301179e-02,
                -6.9100497358928398e-02, -6.9693596522106410e-02, -6.9994164535240613e-02, -7.0145809817330637e-02,
                -7.0222091764421901e-02, -7.0260387246833103e-02
        },
// x = -2.6500
        {
                -5.3500066170260117e-02, -6.1634460117648095e-02, -6.5989009491289594e-02, -6.8267073682579782e-02,
                -6.9441009147381744e-02, -7.0039949602589677e-02, -7.0343497691153126e-02, -7.0496653251328048e-02,
                -7.0573697194316498e-02, -7.0612375988306497e-02
        },
// x = -2.6450
        {
                -5.3694447897585866e-02, -6.1902446258340224e-02, -6.6297851352267553e-02, -6.8597790276384324e-02,
                -6.9783170663858834e-02, -7.0388009363125170e-02, -7.0694566875731224e-02, -7.0849247687621103e-02,
                -7.0927061215270093e-02, -7.0966127149759070e-02
        },
// x = -2.6400
        {
                -5.3889174837073971e-02, -6.2171405171980131e-02, -6.6608035575162192e-02, -6.8930054433371940e-02,
                -7.0126989629416325e-02, -7.0737784071641663e-02, -7.1047380643028957e-02, -7.1203601827721064e-02,
                -7.1282192604030126e-02, -7.1321649546069885e-02
        },
// x = -2.6350
        {
                -5.4084243892458445e-02, -6.2441338576906688e-02, -6.6919567028140475e-02, -6.9263872877519048e-02,
                -7.0472473798669930e-02, -7.1089282034757090e-02, -7.1401947588536560e-02, -7.1559724416012840e-02,
                -7.1639100180958676e-02, -7.1678952036110660e-02
        },
// x = -2.6300
        {
                -5.4279651922382148e-02, -6.2712248175864876e-02, -6.7232450587922293e-02, -6.9599252357164312e-02,
                -7.0819630959916086e-02, -7.1442511597946567e-02, -7.1758276349374608e-02, -7.1917624239963063e-02,
                -7.1997792810247360e-02, -7.2038043522964404e-02
        },
// x = -2.6250
        {
                -5.4475395740095439e-02, -6.2984135655779441e-02, -6.7546691139701714e-02, -6.9936199645051239e-02,
                -7.1168468935254067e-02, -7.1797481145711273e-02, -7.2116375604488261e-02, -7.2277310130328176e-02,
                -7.2358279400133355e-02, -7.2398932954145123e-02
        },
// x = -2.6200
        {
                -5.4671472113154720e-02, -6.3257002687527375e-02, -6.7862293577067467e-02, -7.0274721538370277e-02,
                -7.1518995580707762e-02, -7.2154199101747529e-02, -7.2476254074843399e-02, -7.2638790961364455e-02,
                -7.2720568903116123e-02, -7.2761629321818538e-02
        },
// x = -2.6150
        {
                -5.4867877763121263e-02, -6.3530850925708510e-02, -6.8179262801922089e-02, -7.0614824858800718e-02,
                -7.1871218786348348e-02, -7.2512673929116880e-02, -7.2837920523622562e-02, -7.3002075651038403e-02,
                -7.3084670316175729e-02, -7.3126141663024044e-02
        },
// x = -2.6100
        {
                -5.5064609365259941e-02, -6.3805682008414882e-02, -6.8497603724399780e-02, -7.0956516452551854e-02,
                -7.2225146476416674e-02, -7.2872914130416591e-02, -7.3201383756422378e-02, -7.3367173161238447e-02,
                -7.3450592680991716e-02, -7.3492479059897503e-02
        },
// x = -2.6050
        {
                -5.5261663548237992e-02, -6.4081497556998532e-02, -6.8817321262783712e-02, -7.1299803190404215e-02,
                -7.2580786609446127e-02, -7.3234928247950845e-02, -7.3566652621451692e-02, -7.3734092497987402e-02,
                -7.3818345084163337e-02, -7.3860650639895470e-02
        },
// x = -2.6000
        {
                -5.5459036893824205e-02, -6.4358299175837644e-02, -6.9138420343421256e-02, -7.1644691967749793e-02,
                -7.2938147178385315e-02, -7.3598724863902440e-02, -7.3933736009730466e-02, -7.4102842711656111e-02,
                -7.4187936657430759e-02, -7.4230665576020233e-02
        },
// x = -2.5950
        {
                -5.5656725936587874e-02, -6.4636088452101462e-02, -6.9460905900638872e-02, -7.1991189704632363e-02,
                -7.3297236210721581e-02, -7.3964312600505219e-02, -7.4302642855289674e-02, -7.4473432897177894e-02,
                -7.4559376577897310e-02, -7.4602533087046050e-02
        },
// x = -2.5900
        {
                -5.5854727163598419e-02, -6.4914866955513364e-02, -6.9784782876655280e-02, -7.2339303345787170e-02,
                -7.3658061768604149e-02, -7.4331700120217087e-02, -7.4673382135371891e-02, -7.4845872194264265e-02,
                -7.4932674068252769e-02, -7.4976262437746591e-02
        },
// x = -2.5850
        {
                -5.6053037014124889e-02, -6.5194636238112738e-02, -7.0110056221493730e-02, -7.2689039860679996e-02,
                -7.4020631948967755e-02, -7.4700896125893590e-02, -7.5045962870632990e-02, -7.5220169787621291e-02,
                -7.5307838396997992e-02, -7.5351862939123451e-02
        },
// x = -2.5800
        {
                -5.6251651879335916e-02, -6.5475397834015181e-02, -7.0436730892893012e-02, -7.3040406243546074e-02,
                -7.4384954883656318e-02, -7.5071909360962141e-02, -7.5420394125344320e-02, -7.5596334907167370e-02,
                -7.5684878878670231e-02, -7.5729343948635675e-02
        },
// x = -2.5750
        {
                -5.6450568101999950e-02, -6.5757153259171322e-02, -7.0764811856217122e-02, -7.3393409513428501e-02,
                -7.4751038739547013e-02, -7.5444748609596976e-02, -7.5796685007596384e-02, -7.5974376828251630e-02,
                -7.6063804874069829e-02, -7.6108714870430555e-02
        },
// x = -2.5700
        {
                -5.6649781976185795e-02, -6.6039904011123862e-02, -7.1094304084363874e-02, -7.3748056714215984e-02,
                -7.5118891718674183e-02, -7.5819422696894512e-02, -7.6174844669502612e-02, -7.6354304871873738e-02,
                -7.6444625790487872e-02, -7.6489985155575604e-02
        },
// x = -2.5650
        {
                -5.6849289746963766e-02, -6.6323651568763631e-02, -7.1425212557672313e-02, -7.4104354914680412e-02,
                -7.5488522058353991e-02, -7.6195940489049457e-02, -7.6554882307404593e-02, -7.6736128404904319e-02,
                -7.6827351081935100e-02, -7.6873164302291339e-02
        },
// x = -2.5600
        {
                -5.7049087610106929e-02, -6.6608397392083629e-02, -7.1757542263828827e-02, -7.4462311208513848e-02,
                -7.5859938031308666e-02, -7.6574310893531658e-02, -7.6936807162078139e-02, -7.7119856840306727e-02,
                -7.7211990249371726e-02, -7.7258261856185742e-02
        },
// x = -2.5550
        {
                -5.7249171711793098e-02, -6.6894142921931812e-02, -7.2091298197771880e-02, -7.4821932714364742e-02,
                -7.6233147945791224e-02, -7.6954542859263245e-02, -7.7320628518939796e-02, -7.7505499637359698e-02,
                -7.7598552840938473e-02, -7.7645287410489441e-02
        },
// x = -2.5500
        {
                -5.7449538148307167e-02, -6.7180889579762454e-02, -7.2426485361595974e-02, -7.5183226575874423e-02,
                -7.6608160145710452e-02, -7.7336645376796728e-02, -7.7706355708254876e-02, -7.7893066301881170e-02,
                -7.7987048452188637e-02, -7.8034250606292085e-02
        },
// x = -2.5450
        {
                -5.7650182965744111e-02, -6.7468638767385972e-02, -7.2763108764453702e-02, -7.5546199961712163e-02,
                -7.6984983010755917e-02, -7.7720627478493676e-02, -7.8093998105345799e-02, -7.8282566386452840e-02,
                -7.8377486726321541e-02, -7.8425161132780252e-02
        },
// x = -2.5400
        {
                -5.7851102159712516e-02, -6.7757391866717046e-02, -7.3101173422457200e-02, -7.5910860065610541e-02,
                -7.7363624956523075e-02, -7.8106498238703664e-02, -7.8483565130801791e-02, -7.8674009490646124e-02,
                -7.8769877354416692e-02, -7.8818028727476022e-02
        },
// x = -2.5350
        {
                -5.8052291675038760e-02, -6.8047150239521598e-02, -7.3440684358577896e-02, -7.6277214106400010e-02,
                -7.7744094434638866e-02, -7.8494266773944249e-02, -7.8875066250689005e-02, -7.9067405261249141e-02,
                -7.9164230075669501e-02, -7.9212863176477105e-02
        },
// x = -2.5300
        {
                -5.8253747405471908e-02, -6.8337915227162233e-02, -7.3781646602545103e-02, -7.6645269328042540e-02,
                -7.8126399932887211e-02, -7.8883942243081664e-02, -7.9268510976762158e-02, -7.9462763392494265e-02,
                -7.9560554677627737e-02, -7.9609674314697981e-02
        },
// x = -2.5250
        {
                -5.8455465193389157e-02, -6.8629688150341789e-02, -7.4124065190743382e-02, -7.7015032999665597e-02,
                -7.8510549975334659e-02, -7.9275533847511556e-02, -7.9663908866676417e-02, -7.9860093626287390e-02,
                -7.9958860996429487e-02, -8.0008472026112207e-02
        },
// x = -2.5200
        {
                -5.8657440829502305e-02, -6.8922470308846087e-02, -7.4467945166108676e-02, -7.7386512415594993e-02,
                -7.8896553122456456e-02, -7.9669050831340946e-02, -8.0061269524200687e-02, -8.0259405752437923e-02,
                -8.0359158917041998e-02, -8.0409266243996039e-02
        },
// x = -2.5150
        {
                -5.8859670052564775e-02, -6.9216262981284707e-02, -7.4813291578023008e-02, -7.7759714895387341e-02,
                -7.9284417971262455e-02, -8.0064502481570349e-02, -8.0460602599431552e-02, -8.0660709608889755e-02,
                -8.0761458373501807e-02, -8.0812066951173087e-02
        },
// x = -2.5100
        {
                -5.9062148549079714e-02, -6.9511067424830500e-02, -7.5160109482208012e-02, -7.8134647783862129e-02,
                -7.9674153155423519e-02, -8.0461898128277226e-02, -8.0861917789008225e-02, -8.1064015081953669e-02,
                -8.1165769349156022e-02, -8.1216884180260421e-02
        },
// x = -2.5050
        {
                -5.9264871953008690e-02, -6.9806884874957578e-02, -7.5508403940616933e-02, -7.8511318451132892e-02,
                -8.0065767345397762e-02, -8.0861247144799003e-02, -8.1265224836328406e-02, -8.1469332106540268e-02,
                -8.1572101876904640e-02, -8.1623728013915500e-02
        },
// x = -2.5000
        {
                -5.9467835845481688e-02, -7.0103716545177858e-02, -7.5858180021325730e-02, -7.8889734292638569e-02,
                -8.0459269248557258e-02, -8.1262558947917868e-02, -8.1670533531764899e-02, -8.1876670666394646e-02,
                -8.1980466039444241e-02, -8.2032608585084779e-02
        },
// x = -2.4950
        {
                -5.9671035754507745e-02, -7.0401563626776395e-02, -7.6209442798422400e-02, -7.9269902729173281e-02,
                -8.0854667609314562e-02, -8.1665842998045302e-02, -8.2077853712883619e-02, -8.2286040794331550e-02,
                -8.2390871969512766e-02, -8.2443536077253146e-02
        },
// x = -2.4900
        {
                -5.9874467154686875e-02, -7.0700427288544587e-02, -7.6562197351895150e-02, -7.9651831206916518e-02,
                -8.1251971209249721e-02, -8.2071108799407697e-02, -8.2487195264661778e-02, -8.2697452572472088e-02,
                -8.2803329850135263e-02, -8.2856520724694807e-02
        },
// x = -2.4850
        {
                -6.0078125466922913e-02, -7.1000308676513080e-02, -7.6916448767519546e-02, -8.0035527197462278e-02,
                -8.1651188867237129e-02, -8.2478365900232595e-02, -8.2898568119707805e-02, -8.3110916132481102e-02,
                -8.3217849914871286e-02, -8.3271572812725334e-02
        },
// x = -2.4800
        {
                -6.0282006058137473e-02, -7.1301208913681913e-02, -7.7272202136743712e-02, -8.0420998197847551e-02,
                -8.2052329439572774e-02, -8.2887623892935372e-02, -8.3311982258481576e-02, -8.3526441655806166e-02,
                -8.3634442448062965e-02, -8.3688702677954932e-02
        },
// x = -2.4750
        {
                -6.0486104240985293e-02, -7.1603129099750293e-02, -7.7629462556572632e-02, -8.0808251730580538e-02,
                -8.2455401820101554e-02, -8.3298892414306699e-02, -8.3727447709515915e-02, -8.3944039373917131e-02,
                -8.4053117785084688e-02, -8.4107920708542963e-02
        },
// x = -2.4700
        {
                -6.0690415273570539e-02, -7.1906070310844478e-02, -7.7988235129450995e-02, -8.1197295343668166e-02,
                -8.2860414940344559e-02, -8.3712181145700601e-02, -8.4144974549638971e-02, -8.4363719568547213e-02,
                -8.4473886312593685e-02, -8.4529237344453848e-02
        },
// x = -2.4650
        {
                -6.0894934359164384e-02, -7.2210033599244175e-02, -7.8348524963144706e-02, -8.1588136610642917e-02,
                -8.3267377769626769e-02, -8.4127499813223219e-02, -8.4564572904197444e-02, -8.4785492571935003e-02,
                -8.4896758468782058e-02, -8.4952663077713944e-02
        },
// x = -2.4600
        {
                -6.1099656645924268e-02, -7.2515019993107929e-02, -7.8710337170620717e-02, -8.1980783130589097e-02,
                -8.3676299315204650e-02, -8.4544858187922053e-02, -8.4986252947280780e-02, -8.5209368767067473e-02,
                -8.5321744743629913e-02, -8.5378208452670024e-02
        },
// x = -2.4550
        {
                -6.1304577226613956e-02, -7.2821030496196523e-02, -7.9073676869925988e-02, -8.2375242528168691e-02,
                -8.4087188622394074e-02, -8.4964266085975973e-02, -8.5410024901946505e-02, -8.5635358587924523e-02,
                -8.5748855679159547e-02, -8.5805884066248617e-02
        },
// x = -2.4500
        {
                -6.1509691138325620e-02, -7.3128066087595617e-02, -7.9438549184064705e-02, -8.2771522453646260e-02,
                -8.4500054774698213e-02, -8.5385733368885858e-02, -8.5835899040446251e-02, -8.6063472519724071e-02,
                -8.6178101869691312e-02, -8.6235700568217166e-02
        },
// x = -2.4450
        {
                -6.1714993362202833e-02, -7.3436127721436609e-02, -7.9804959240874190e-02, -8.3169630582913642e-02,
                -8.4914906893935549e-02, -8.5809269943665833e-02, -8.6263885684452771e-02, -8.6493721099168877e-02,
                -8.6609493962100145e-02, -8.6667668661445887e-02
        },
// x = -2.4400
        {
                -6.1920478823165516e-02, -7.3745216326616014e-02, -8.0172912172899471e-02, -8.3569574617513603e-02,

                -8.5331754140368296e-02, -8.6234885763035216e-02, -8.6693995205288157e-02, -8.6926114914694014e-02,
                -8.7043042656073724e-02, -8.7101799102171279e-02
        },
// x = -2.4350
        {
                -6.2126142389636214e-02, -7.4055332806513996e-02, -8.0542413117266440e-02, -8.3971362284663106e-02,
                -8.5750605712830316e-02, -8.6662590825610897e-02, -8.7126238024152688e-02, -8.7360664606715835e-02,
                -8.7478758704371828e-02, -8.7538102700260995e-02
        },
// x = -2.4300
        {
                -6.2331978873268121e-02, -7.4366478038710923e-02, -8.0913467215553342e-02, -8.4375001337275990e-02,
                -8.6171470848856027e-02, -8.7092395176100762e-02, -8.7560624612354990e-02, -8.7797380867881789e-02,
                -8.7916652913086688e-02, -8.7976590319479547e-02
        },
// x = -2.4250
        {
                -6.2537983028674748e-02, -7.4678652874702919e-02, -8.1286079613661397e-02, -8.4780499553984867e-02,
                -8.6594358824808593e-02, -8.7524308905497283e-02, -8.7997165491542767e-02, -8.8236274443321647e-02,
                -8.8356736141904965e-02, -8.8417272877755818e-02
        },
// x = -2.4200
        {
                -6.2744149553161349e-02, -7.4991858139616105e-02, -8.1660255461683512e-02, -8.5187864739162372e-02,
                -8.7019278956008708e-02, -8.7958342151272240e-02, -8.8435871233934815e-02, -8.8677356130899859e-02,
                -8.8799019304370660e-02, -8.8860161347451797e-02
        },
// x = -2.4150
        {
                -6.2950473086458000e-02, -7.5306094631919174e-02, -8.2035999913771543e-02, -8.5597104722941755e-02,
                -8.7446240596863248e-02, -8.8394505097571435e-02, -8.8876752462553901e-02, -8.9120636781468698e-02,
                -8.9243513368149466e-02, -8.9305266755632162e-02
        },
// x = -2.4100
        {
                -6.3156948210454894e-02, -7.5621363123135180e-02, -8.2413318128002511e-02, -8.6008227361236908e-02,
                -8.7875253140994355e-02, -8.8832807975410932e-02, -8.9319819851460619e-02, -8.9566127299123166e-02,
                -8.9690229355294318e-02, -8.9752600184336007e-02
        },
// x = -2.4050
        {
                -6.3363569448939033e-02, -7.5937664357551254e-02, -8.2792215266243210e-02, -8.6421240535761723e-02,
                -8.8306326021368051e-02, -8.9273261062873061e-02, -8.9765084125988229e-02, -9.0013838641456609e-02,
                -9.0139178342512324e-02, -9.0202172770849021e-02
        },
// x = -2.4000
        {
                -6.3570331267333302e-02, -7.6254999051927772e-02, -8.3172696494012946e-02, -8.6836152154048454e-02,
                -8.8739468710423583e-02, -8.9715874685303690e-02, -9.0212556062978469e-02, -9.0463781819817762e-02,
                -9.0590371461432714e-02, -9.0653995707977691e-02
        },
// x = -2.3950
        {
                -6.3777228072437395e-02, -7.6573367895205788e-02, -8.3554766980345641e-02, -8.7252970149466050e-02,
                -8.9174690720202468e-02, -9.0160659215509739e-02, -9.0662246491018395e-02, -9.0915967899568725e-02,
                -9.1043819898876596e-02, -9.1108080244324610e-02
        },
// x = -2.3900
        {
                -6.3984254212170771e-02, -7.6892771548213265e-02, -8.3938431897649601e-02, -8.7671702481236874e-02,
                -8.9612001602477565e-02, -9.0607625073957782e-02, -9.1114166290678117e-02, -9.1370408000344572e-02,
                -9.1499534897127255e-02, -9.1564437684564876e-02
        },
// x = -2.3850
        {
                -6.4191403975317946e-02, -7.7213210643369945e-02, -8.4323696421566424e-02, -8.8092357134453514e-02,
                -9.0051410948882596e-02, -9.1056782728972663e-02, -9.1568326394749697e-02, -9.1827113296313420e-02,
                -9.1957527754202448e-02, -9.2023079389724274e-02
        },
// x = -2.3800
        {
                -6.4398671591275616e-02, -7.7534685784391183e-02, -8.4710565730828150e-02, -8.8514942120094739e-02,
                -9.0492928391041505e-02, -9.1508142696937453e-02, -9.2024737788486924e-02, -9.2286095016438710e-02,
                -9.2417809824127742e-02, -9.2484016777458669e-02
        },
// x = -2.3750
        {
                -6.4606051229802564e-02, -7.7857197545990395e-02, -8.5099045007113044e-02, -8.8939465475040141e-02,
                -9.0936563600697740e-02, -9.1961715542493494e-02, -9.2483411509845875e-02, -9.2747364444741520e-02,
                -9.2880392517210877e-02, -9.2947261322334482e-02
        },
// x = -2.3700
        {
                -6.4813537000771165e-02, -7.8180746473580046e-02, -8.5489139434899802e-02, -8.9365935262085014e-02,
                -9.1382326289844093e-02, -9.2417511878741462e-02, -9.2944358649727082e-02, -9.3210932920565218e-02,
                -9.3345287300318047e-02, -9.3412824556111135e-02
        },
// x = -2.3650
        {
                -6.5021122953922247e-02, -7.8505333082971862e-02, -8.5880854201320286e-02, -8.9794359569953547e-02,
                -9.1830226210852003e-02, -9.2875542367442915e-02, -9.3407590352218114e-02, -9.3676811838840315e-02,
                -9.3812505697150994e-02, -9.3880718068024172e-02
        },
// x = -2.3600
        {
                -6.5228803078621375e-02, -7.8830957860075379e-02, -8.6274194496010814e-02, -9.0224746513311999e-02,
                -9.2280273156601450e-02, -9.3335817719222447e-02, -9.3873117814837384e-02, -9.4145012650351426e-02,
                -9.4282059288525594e-02, -9.4350953505070487e-02
        },
// x = -2.3550
        {
                -6.5436571303618213e-02, -7.9157621260595348e-02, -8.6669165510961890e-02, -9.0657104232780741e-02,
                -9.2732476960610696e-02, -9.3798348693770586e-02, -9.4340952288779020e-02, -9.4615546862004754e-02,
                -9.4753959712651770e-02, -9.4823542572294292e-02
        },
// x = -2.3500
        {
                -6.5644421496808464e-02, -7.9485323709728675e-02, -8.7065772440366326e-02, -9.1091440894945711e-02,
                -9.3186847497165798e-02, -9.4263146100047338e-02, -9.4811105079158728e-02, -9.5088426037097079e-02,
                -9.5228218665414990e-02, -9.5298497033074997e-02
        },
// x = -2.3450
        {
                -6.5852347464998184e-02, -7.9814065601858650e-02, -8.7464020480466032e-02, -9.1527764692368965e-02,
                -9.3643394681450914e-02, -9.4730220796486339e-02, -9.5283587545260520e-02, -9.5563661795586327e-02,
                -9.5704847900658710e-02, -9.5775828709416158e-02
        },
// x = -2.3400
        {
                -6.6060342953670928e-02, -8.0143847300249635e-02, -8.7863914829397191e-02, -9.1966083843598501e-02,
                -9.4102128469677873e-02, -9.5199583691199616e-02, -9.5758411100784757e-02, -9.6041265814362495e-02,
                -9.6183859230468294e-02, -9.6255549482236094e-02
        },
// x = -2.3350
        {
                -6.6268401646757694e-02, -8.0474669136739577e-02, -8.8265460687033906e-02, -9.2406406593177476e-02,
                -9.4563058859216176e-02, -9.5671245742183136e-02, -9.6235587214097001e-02, -9.6521249827520916e-02,
                -9.6665264525456596e-02, -9.6737671291659680e-02
        },
// x = -2.3300
        {
                -6.6476517166409524e-02, -8.0806531411431709e-02, -8.8668663254830180e-02, -9.2848741211652500e-02,
                -9.5026195888723172e-02, -9.6145217957522899e-02, -9.6715127408477847e-02, -9.7003625626635670e-02,
                -9.7149075715050423e-02, -9.7222206137311862e-02
        },
// x = -2.3250
        {
                -6.6684683072773254e-02, -8.1139434392385051e-02, -8.9073527735660615e-02, -9.3293095995580902e-02,
                -9.5491549638273845e-02, -9.6621511395601614e-02, -9.7197043262374119e-02, -9.7488405061035274e-02,
                -9.7635304787778682e-02, -9.7709166078612256e-02
        },
// x = -2.3200
        {
                -6.6892892863769926e-02, -8.1473378315304101e-02, -8.9480059333659268e-02, -9.3739479267537798e-02,
                -9.5959130229491069e-02, -9.7100137165306258e-02, -9.7681346409650607e-02, -9.7975600038078814e-02,
                -9.8123963791562088e-02, -9.8198563235071590e-02
        },
// x = -2.3150
        {
                -6.7101139974876461e-02, -8.1808363383226529e-02, -8.9888263254057243e-02, -9.4187899376121584e-02,
                -9.6428947825675612e-02, -9.7581106426235978e-02, -9.8168048539843233e-02, -9.8465222523433718e-02,
                -9.8615064834003563e-02, -9.8690409786589120e-02
        },
// x = -2.3100
        {
                -6.7309417778910322e-02, -8.2144389766211329e-02, -9.0298144703018585e-02, -9.4638364695959376e-02,
                -9.6901012631936401e-02, -9.8064430388910953e-02, -9.8657161398413171e-02, -9.8957284541354884e-02,
                -9.9108620082680898e-02, -9.9184717973752154e-02
        },
// x = -2.3050
        {
                -6.7517719585817162e-02, -8.2481457601024058e-02, -9.0709708887474413e-02, -9.5090883627711051e-02,
                -9.7375334895320437e-02, -9.8550120314981671e-02, -9.9148696787001958e-02, -9.9451798174964898e-02,
                -9.9604641765440136e-02, -9.9681500098136214e-02
        },
// x = -2.3000
        {
                -6.7726038642462169e-02, -8.2819566990822671e-02, -9.1122961014955844e-02, -9.5545464598072644e-02,
                -9.7851924904943224e-02, -9.9038187517439064e-02, -9.9642666563687587e-02, -9.9948775566535392e-02,
                -1.0010314217069068e-01, -1.0018076852260746e-01
        },
// x = -2.2950
        {
                -6.7934368132424122e-02, -8.3158718004841203e-02, -9.1537906293425050e-02, -9.6002116059779041e-02,
                -9.8330792992118762e-02, -9.9528643360824828e-02, -1.0013908264324183e-01, -1.0044822891777021e-01,
                -1.0060413364770175e-01, -1.0068253567162616e-01
        },
// x = -2.2900
        {
                -6.8142701175793421e-02, -8.3498910678072885e-02, -9.1954549931104967e-02, -9.6460846491605717e-02,
                -9.8811949530489926e-02, -1.0002149926144338e-01, -1.0063795699738848e-01, -1.0095017049008911e-01,
                -1.0110762860690033e-01, -1.0118681403155154e-01
        },
// x = -2.2850
        {
                -6.8351030828972884e-02, -8.3840145010952452e-02, -9.2372897136307239e-02, -9.6921664398369453e-02,
                -9.9295404936158432e-02, -1.0051676668757305e-01, -1.0113930165506271e-01, -1.0145461260491360e-01,
                -1.0161363952017029e-01, -1.0169361615094866e-01
        },
// x = -2.2800
        {
                -6.8559350084482523e-02, -8.4182420969036831e-02, -9.2792953117258589e-02, -9.7384578310928491e-02,
                -9.9781169667815320e-02, -1.0101445715967936e-01, -1.0164312870267142e-01, -1.0196156764395327e-01,
                -1.0212217892115329e-01, -1.0220295464089613e-01
        },
// x = -2.2750
        {
                -6.8767651870767971e-02, -8.4525738482685575e-02, -9.3214723081925674e-02, -9.7849596786181714e-02,
                -1.0026925422687091e-01, -1.0151458225062798e-01, -1.0214945028435483e-01, -1.0247104804949404e-01,
                -1.0263325940555081e-01, -1.0271484217529590e-01
        },
// x = -2.2700
        {
                -6.8975929052012150e-02, -8.4870097446739831e-02, -9.3638212237838261e-02, -9.8316728407066692e-02,
                -1.0075966915758523e-01, -1.0201715358589859e-01, -1.0265827860224880e-01, -1.0298306632468743e-01,
                -1.0314689363142776e-01, -1.0322929149118414e-01
        },
// x = -2.2650
        {
                -6.9184174427950898e-02, -8.5215497720200553e-02, -9.4063425791910918e-02, -9.8785981782557236e-02,
                -1.0125242504719796e-01, -1.0252218284379977e-01, -1.0316962591674873e-01, -1.0349763503384110e-01,
                -1.0366309431951770e-01, -1.0374631538904387e-01
        },
// x = -2.2600
        {
                -6.9392380733692485e-02, -8.5561939125905997e-02, -9.4490368950262818e-02, -9.9257365547659807e-02,
                -1.0174753252605886e-01, -1.0302968175568376e-01, -1.0368350454677396e-01, -1.0401476680271100e-01,
                -1.0418187425352909e-01, -1.0426592673311914e-01
        },
// x = -2.2550
        {
                -6.9600540639540714e-02, -8.5909421450208051e-02, -9.4919046918036459e-02, -9.9730888363409084e-02,
                -1.0224500226775768e-01, -1.0353966210616281e-01, -1.0419992687003375e-01, -1.0453447431879445e-01,
                -1.0470324628045354e-01, -1.0478813845173066e-01
        },
// x = -2.2500
        {
                -6.9808646750822392e-02, -8.6257944442647547e-02, -9.5349464899214018e-02, -1.0020655891686274e-01,
                -1.0274484498925449e-01, -1.0405213573332540e-01, -1.0471890532329391e-01, -1.0505677033162486e-01,
                -1.0522722331087500e-01, -1.0531296353759283e-01
        },
// x = -2.2450
        {
                -7.0016691607718340e-02, -8.6607507815629328e-02, -9.5781628096432717e-02, -1.0068438592109487e-01,
                -1.0324707145100964e-01, -1.0456711452895352e-01, -1.0524045240264493e-01, -1.0558166765306772e-01,
                -1.0575381831928082e-01, -1.0584041504813263e-01
        },
// x = -2.2400
        {
                -7.0224667685098857e-02, -8.6958111244095943e-02, -9.6215541710798311e-02, -1.0116437811518920e-01,
                -1.0375169245711394e-01, -1.0508461043874046e-01, -1.0576458066377087e-01, -1.0610917915761778e-01,
                -1.0628304434437419e-01, -1.0637050610581002e-01
        },
// x = -2.2350
        {
                -7.0432567392363088e-02, -8.7309754365200595e-02, -9.6651210941696986e-02, -1.0164654426423067e-01,
                -1.0425871885541856e-01, -1.0560463546250938e-01, -1.0629130272221948e-01, -1.0663931778269778e-01,
                -1.0681491448938786e-01, -1.0690324989843950e-01
        },
// x = -2.2300
        {
                -7.0640383073282978e-02, -8.7662436777979441e-02, -9.7088640986605371e-02, -1.0213089315929645e-01,
                -1.0476816153766501e-01, -1.0612720165443228e-01, -1.0682063125367337e-01, -1.0717209652895845e-01,
                -1.0734944192239976e-01, -1.0743865967951376e-01
        },
// x = -2.2250
        {
                -7.0848107005851202e-02, -8.8016158043023263e-02, -9.7527837040899554e-02, -1.0261743361744596e-01,
                -1.0528003143961528e-01, -1.0665232112324990e-01, -1.0735257899422242e-01, -1.0770752846057981e-01,
                -1.0788663987664966e-01, -1.0797674876852874e-01
        },
// x = -2.2200
        {
                -7.1055731402133712e-02, -8.8370917682147901e-02, -9.7968804297661416e-02, -1.0310617448170986e-01,
                -1.0579433954118167e-01, -1.0718000603249217e-01, -1.0788715874063702e-01, -1.0824562670557389e-01,
                -1.0842652165085777e-01, -1.0851753055131004e-01
        },
// x = -2.2150
        {
                -7.1263248408126728e-02, -8.8726715178064575e-02, -9.8411547947484634e-02, -1.0359712462107799e-01,
                -1.0631109686655631e-01, -1.0771026860069911e-01, -1.0842438335064268e-01, -1.0878640445608893e-01,
                -1.0896910060954453e-01, -1.0906101848034126e-01
        },
// x = -2.2100
        {
                -7.1470650103618139e-02, -8.9083549974048798e-02, -9.8856073178277812e-02, -1.0409029293048663e-01,
                -1.0683031448434126e-01, -1.0824312110164275e-01, -1.0896426574319563e-01, -1.0932987496871467e-01,
                -1.0951439018335209e-01, -1.0960722607509384e-01
        },
// x = -2.2050
        {
                -7.1677928502053775e-02, -8.9441421473609478e-02, -9.9302385175066801e-02, -1.0458568833080444e-01,
                -1.0735200350767810e-01, -1.0877857586454948e-01, -1.0950681889875946e-01, -1.0987605156478931e-01,
                -1.1006240386936732e-01, -1.1015616692235826e-01
        },
// x = -2.2000
        {
                -7.1885075550408156e-02, -8.9800329040156376e-02, -9.9750489119794672e-02, -1.0508331976881752e-01,
                -1.0787617509437752e-01, -1.0931664527432282e-01, -1.1005205585958297e-01, -1.1042494763070776e-01,
                -1.1061315523144624e-01, -1.1070785467657739e-01
        },
// x = -2.1950
        {
                -7.2092083129060472e-02, -9.0160271996667846e-02, -1.0020039019112070e-01, -1.0558319621721372e-01,
                -1.0840284044704897e-01, -1.0985734177176751e-01, -1.1059998972997923e-01, -1.1097657661823115e-01,
                -1.1116665790054027e-01, -1.1126230306018085e-01
        },
// x = -2.1900
        {
                -7.2298943051674697e-02, -9.0521249625357431e-02, -1.0065209356421738e-01, -1.0608532667456537e-01,
                -1.0893201081322997e-01, -1.1040067785381351e-01, -1.1115063367660540e-01, -1.1153095204479795e-01,
                -1.1172292557502346e-01, -1.1181952586392151e-01
        },
// x = -2.1850
        {
                -7.2505647065085402e-02, -9.0883261167339732e-02, -1.0110560441056556e-01, -1.0658972016531168e-01,
                -1.0946369748551567e-01, -1.1094666607374098e-01, -1.1170400092874418e-01, -1.1208808749383638e-01,
                -1.1228197202102218e-01, -1.1237953694721334e-01
        },
// x = -2.1800
        {
                -7.2712186849188315e-02, -9.1246305822296578e-02, -1.0156092789774819e-01, -1.0709638573973949e-01,
                -1.0999791180168798e-01, -1.1149531904140612e-01, -1.1226010477858604e-01, -1.1264799661507822e-01,
                -1.1284381107274534e-01, -1.1294235023847099e-01
        },
// x = -2.1750
        {
                -7.2918554016835654e-02, -9.1610382748141414e-02, -1.0201806918924246e-01, -1.0760533247396345e-01,
                -1.1053466514484479e-01, -1.1204664942346718e-01, -1.1281895858151257e-01, -1.1321069312487406e-01,
                -1.1340845663281701e-01, -1.1350797973545104e-01
        },
// x = -2.1700
        {
                -7.3124740113737499e-02, -9.1975491060684228e-02, -1.0247703344420996e-01, -1.0811656946990479e-01,
                -1.1107396894352921e-01, -1.1260066994361143e-01, -1.1338057575638115e-01, -1.1377619080650990e-01,
                -1.1397592267261025e-01, -1.1407643950559503e-01
        },
// x = -2.1650
        {
                -7.3330736618367531e-02, -9.2341629833295524e-02, -1.0293782581728525e-01, -1.0863010585526946e-01,
                -1.1161583467185829e-01, -1.1315739338278283e-01, -1.1394496978581073e-01, -1.1434450351052541e-01,
                -1.1454622323258248e-01, -1.1464774368637377e-01
        },
// x = -2.1600
        {
                -7.3536534941874718e-02, -9.2708798096569886e-02, -1.0340045145836320e-01, -1.0914595078352474e-01,
                -1.1216027384965213e-01, -1.1371683257941007e-01, -1.1451215421646842e-01, -1.1491564515503332e-01,
                -1.1511937242261272e-01, -1.1522190648563392e-01
        },
// x = -2.1550
        {
                -7.3742126428000487e-02, -9.3076994837989147e-02, -1.0386491551238376e-01, -1.0966411343387514e-01,
                -1.1270729804256242e-01, -1.1427900042963551e-01, -1.1508214265935779e-01, -1.1548962972604043e-01,
                -1.1569538442234004e-01, -1.1579894218194568e-01
        },
// x = -2.1500
        {
                -7.3947502353000905e-02, -9.3446219001585187e-02, -1.0433122311911594e-01, -1.1018460301123688e-01,
                -1.1325691886220129e-01, -1.1484390988754453e-01, -1.1565494879010782e-01, -1.1606647127776991e-01,
                -1.1627427348150403e-01, -1.1637886512495262e-01
        },
// x = -2.1450
        {
                -7.4152653925574893e-02, -9.3816469487602347e-02, -1.0479937941293958e-01, -1.1070742874621192e-01,
                -1.1380914796626959e-01, -1.1541157396539582e-01, -1.1623058634926339e-01, -1.1664618393298538e-01,
                -1.1685605392028658e-01, -1.1696168973572302e-01
        },
// x = -2.1400
        {
                -7.4357572286798360e-02, -9.4187745152159247e-02, -1.0526938952262552e-01, -1.1123259989505985e-01,
                -1.1436399705868529e-01, -1.1598200573385205e-01, -1.1680906914257637e-01, -1.1722878188331601e-01,
                -1.1744074012965543e-01, -1.1754743050710269e-01
        },
// x = -2.1350
        {
                -7.4562248510063517e-02, -9.4560044806910923e-02, -1.0574125857111413e-01, -1.1176012573966998e-01,
                -1.1492147788971190e-01, -1.1655521832221129e-01, -1.1739041104129867e-01, -1.1781427938958335e-01,
                -1.1802834657170930e-01, -1.1813610200407025e-01
        },
// x = -2.1300
        {
                -7.4766673601024444e-02, -9.4933367218709944e-02, -1.0621499167529236e-01, -1.1229001558753102e-01,
                -1.1548160225608617e-01, -1.1713122491863903e-01, -1.1797462598247552e-01, -1.1840269078212962e-01,
                -1.1861888778002470e-01, -1.1872771886409311e-01
        },
// x = -2.1250
        {
                -7.4970838497548681e-02, -9.5307711109267682e-02, -1.0669059394576851e-01, -1.1282227877170062e-01,
                -1.1604438200114646e-01, -1.1771003877040088e-01, -1.1856172796924082e-01, -1.1899403046114727e-01,
                -1.1921237836000431e-01, -1.1932229579748611e-01
        },
// x = -2.1200
        {
                -7.5174734069674690e-02, -9.5683075154815292e-02, -1.0716807048664603e-01, -1.1335692465077325e-01,
                -1.1660982901496011e-01, -1.1829167318409597e-01, -1.1915173107111299e-01, -1.1958831289701034e-01,
                -1.1980883298922718e-01, -1.1991984758777140e-01
        },
// x = -2.1150
        {
                -7.5378351119575554e-02, -9.6059457985764610e-02, -1.0764742639529502e-01, -1.1389396260884727e-01,
                -1.1717795523445128e-01, -1.1887614152589086e-01, -1.1974464942429214e-01, -1.2018555263060705e-01,
                -1.2040826641780024e-01, -1.2052038909204012e-01
        },
// x = -2.1100
        {
                -7.5581680381528585e-02, -9.6436858186368352e-02, -1.0812866676212252e-01, -1.1443340205549039e-01,
                -1.1774877264352832e-01, -1.1946345722175403e-01, -1.2034049723195883e-01, -1.2078576427367385e-01,
                -1.2101069346871191e-01, -1.2112393524131607e-01
        },
// x = -2.1050
        {
                -7.5784712521891992e-02, -9.6815274294381137e-02, -1.0861179667034052e-01, -1.1497525242570472e-01,
                -1.1832229327321091e-01, -1.2005363375769137e-01, -1.2093928876457333e-01, -1.2138896250913131e-01,
                -1.2161612903818719e-01, -1.2173050104092095e-01
        },
// x = -2.1000
        {
                -7.5987438139086969e-02, -9.7194704800719248e-02, -1.0909682119573295e-01, -1.1551952317988963e-01,
                -1.1889852920175739e-01, -1.2064668467998182e-01, -1.2154103836017648e-01, -1.2199516209142099e-01,
                -1.2222458809604417e-01, -1.2234010157084134e-01
        },
// x = -2.0950
        {
                -7.6189847763587198e-02, -9.7575148149121438e-02, -1.0958374540642060e-01, -1.1606622380380478e-01,
                -1.1947749255479144e-01, -1.2124262359541399e-01, -1.2214576042469186e-01, -1.2260437784684433e-01,
                -1.2283608568605278e-01, -1.2295275198609769e-01
        },
// x = -2.0900
        {
                -7.6391931857914042e-02, -9.7956602735808682e-02, -1.1007257436262412e-01, -1.1661536380853037e-01,
                -1.2005919550542889e-01, -1.2184146417152328e-01, -1.2275346943222856e-01, -1.2321662467390267e-01,
                -1.2345063692629483e-01, -1.2356846751711494e-01
        },
// x = -2.0850
        {
                -7.6593680816639143e-02, -9.8339066909145123e-02, -1.1056331311642591e-01, -1.1716695273042770e-01,
                -1.2064365027440416e-01, -1.2244322013682971e-01, -1.2336417992538586e-01, -1.2383191754363899e-01,
                -1.2406825700952577e-01, -1.2418726347009500e-01
        },
// x = -2.0800
        {
                -7.6795084966393409e-02, -9.8722538969297349e-02, -1.1105596671152979e-01, -1.1772100013109747e-01,
                -1.2123086913019658e-01, -1.2304790528107622e-01, -1.2397790651555854e-01, -1.2445027149998122e-01,
                -1.2468896120353835e-01, -1.2480915522739083e-01
        },
// x = -2.0750
        {
                -7.6996134565882751e-02, -9.9107017167895814e-02, -1.1155054018301921e-01, -1.1827751559733762e-01,
                -1.2182086438915656e-01, -1.2365553345546779e-01, -1.2459466388324364e-01, -1.2507170166008677e-01,
                -1.2531276485152798e-01, -1.2543415824788301e-01
        },
// x = -2.0700
        {
                -7.7196819805911046e-02, -9.9492499707694346e-02, -1.1204703855711366e-01, -1.1883650874109933e-01,
                -1.2241364841563128e-01, -1.2426611857291113e-01, -1.2521446677834838e-01, -1.2569622321468904e-01,
                -1.2593968337245950e-01, -1.2606228806735700e-01
        },
// x = -2.0650
        {
                -7.7397130809409867e-02, -9.9878984742231239e-02, -1.1254546685092359e-01, -1.1939798919944229e-01,
                -1.2300923362209062e-01, -1.2487967460825491e-01, -1.2583733002049932e-01, -1.2632385142844499e-01,
                -1.2656973226143614e-01, -1.2669356029888357e-01
        },
// x = -2.0600
        {
                -7.7597057631475411e-02, -1.0026647037548990e-01, -1.1304583007220341e-01, -1.1996196663448840e-01,
                -1.2360763246925231e-01, -1.2549621559853072e-01, -1.2646326849935252e-01, -1.2695460164028480e-01,
                -1.2720292709007008e-01, -1.2732799063320005e-01
        },
// x = -2.0550
        {
                -7.7796590259412629e-02, -1.0065495466155965e-01, -1.1354813321910277e-01, -1.2052845073337443e-01,
                -1.2420885746620740e-01, -1.2611575564319460e-01, -1.2709229717490536e-01, -1.2758848926376262e-01,
                -1.2783928350685445e-01, -1.2796559483909381e-01
        },
// x = -2.0500
        {
                -7.7995718612786522e-02, -1.0104443560429699e-01, -1.1405238127991638e-01, -1.2109745120820334e-01,
                -1.2481292117054477e-01, -1.2673830890436918e-01, -1.2772443107780868e-01, -1.2822552978740906e-01,
                -1.2847881723753735e-01, -1.2860638876378758e-01
        },
// x = -2.0450
        {
                -7.8194432543481018e-02, -1.0143491115698695e-01, -1.1455857923283198e-01, -1.2166897779599450e-01,
                -1.2541983618847649e-01, -1.2736388960708689e-01, -1.2835968530968161e-01, -1.2886573877508520e-01,
                -1.2912154408549795e-01, -1.2925038833332717e-01
        },
// x = -2.0400
        {
                -7.8392721835765219e-02, -1.0182637922200502e-01, -1.1506673204567654e-01, -1.2224304025863234e-01,
                -1.2602961517496158e-01, -1.2799251203953249e-01, -1.2899807504342581e-01, -1.2950913186633850e-01,
                -1.2976747993212348e-01, -1.2989760955296958e-01
        },
// x = -2.0350
        {
                -7.8590576206367002e-02, -1.0221883765047900e-01, -1.1557684467566086e-01, -1.2281964838281398e-01,
                -1.2664227083383081e-01, -1.2862419055328797e-01, -1.2963961552354245e-01, -1.3015572477675966e-01,
                -1.3041664073718923e-01, -1.3054806850757494e-01
        },
// x = -2.0300
        {
                -7.8787985304554664e-02, -1.0261228424195187e-01, -1.1608892206912268e-01, -1.2339881197999555e-01,
                -1.2725781591791024e-01, -1.2925893956357690e-01, -1.3028432206644958e-01, -1.3080553329834152e-01,
                -1.3106904253923896e-01, -1.3120178136199892e-01
        },
// x = -2.0250
        {
                -7.8984938712225727e-02, -1.0300671674404424e-01, -1.1660296916126776e-01, -1.2398054088633716e-01,
                -1.2787626322914486e-01, -1.2989677354950979e-01, -1.3093221006080094e-01, -1.3145857329983937e-01,
                -1.3172470145596840e-01, -1.3185876436148758e-01
        },
// x = -2.0200
        {
                -7.9181425944004513e-02, -1.0340213285211812e-01, -1.1711899087590952e-01, -1.2456484496264655e-01,
                -1.2849762561872238e-01, -1.3053770705432971e-01, -1.3158329496780610e-01, -1.3211486072713338e-01,
                -1.3238363368460956e-01, -1.3251903383207428e-01
        },
// x = -2.0150
        {
                -7.9377436447346336e-02, -1.0379853020894068e-01, -1.1763699212520663e-01, -1.2515173409432157e-01,
                -1.2912191598719580e-01, -1.3118175468565924e-01, -1.3223759232155166e-01, -1.3277441160359146e-01,
                -1.3304585550231754e-01, -1.3318260618097785e-01
        },
// x = -2.0100
        {
                -7.9572959602650872e-02, -1.0419590640434871e-01, -1.1815697780939968e-01, -1.2574121819129089e-01,
                -1.2974914728460665e-01, -1.3182893111574734e-01, -1.3289511772932394e-01, -1.3343724203043503e-01,
                -1.3371138326655868e-01, -1.3384949789700404e-01
        },
// x = -2.0050
        {
                -7.9767984723382937e-02, -1.0459425897491344e-01, -1.1867895281654518e-01, -1.2633330718795466e-01,
                -1.3037933251060702e-01, -1.3247925108171715e-01, -1.3355588687193215e-01, -1.3410336818710553e-01,
                -1.3438023341550079e-01, -1.3451972555094699e-01
        },
// x = -2.0000
        {
                -7.9962501056201660e-02, -1.0499358540360650e-01, -1.1920292202224886e-01, -1.2692801104312157e-01,
                -1.3101248471458252e-01, -1.3313272938581414e-01, -1.3421991550403448e-01, -1.3477280633163291e-01,
                -1.3505242246840532e-01, -1.3519330579599467e-01
        },
// x = -1.9950
        {
                -8.0156497781097985e-02, -1.0539388311946603e-01, -1.1972889028939637e-01, -1.2752533973994709e-01,
                -1.3164861699577324e-01, -1.3378938089565570e-01, -1.3488721945446311e-01, -1.3544557280100583e-01,
                -1.3572796702602102e-01, -1.3587025536813457e-01
        },
// x = -1.9900
        {
                -8.0349964011540667e-02, -1.0579514949726389e-01, -1.2025686246788329e-01, -1.2812530328586869e-01,
                -1.3228774250339595e-01, -1.3444922054448000e-01, -1.3555781462655259e-01, -1.3612168401154309e-01,
                -1.3640688377097973e-01, -1.3655059108656278e-01
        },
// x = -1.9850
        {
                -8.0542888794629938e-02, -1.0619738185717302e-01, -1.2078684339434245e-01, -1.2872791171254011e-01,
                -1.3292987443676538e-01, -1.3511226333139667e-01, -1.3623171699846795e-01, -1.3680115645926716e-01,
                -1.3708918946819421e-01, -1.3723432985409376e-01
        },
// x = -1.9800
        {
                -8.0735261111261017e-02, -1.0660057746443656e-01, -1.2131883789187051e-01, -1.2933317507576483e-01,
                -1.3357502604541455e-01, -1.3577852432163723e-01, -1.3690894262353528e-01, -1.3748400672027902e-01,
                -1.3777490096525696e-01, -1.3792148865757337e-01
        },
// x = -1.9750
        {
                -8.0927069876295082e-02, -1.0700473352903675e-01, -1.2185285076975230e-01, -1.2994110345542709e-01,
                -1.3422321062921588e-01, -1.3644801864680683e-01, -1.3758950763057243e-01, -1.3817025145113510e-01,
                -1.3846403519284239e-01, -1.3861208456829277e-01
        },
// x = -1.9700
        {
                -8.1118303938739714e-02, -1.0740984720536564e-01, -1.2238888682318369e-01, -1.3055170695542262e-01,
                -1.3487444153850159e-01, -1.3712076150513591e-01, -1.3827342822422187e-01, -1.3885990738922513e-01,
                -1.3915660916510941e-01, -1.3930613474240514e-01
        },
// x = -1.9650
        {
                -8.1308952081937627e-02, -1.0781591559189586e-01, -1.2292695083299290e-01, -1.3116499570358725e-01,
                -1.3552873217418265e-01, -1.3779676816173322e-01, -1.3896072068528401e-01, -1.3955299135315227e-01,
                -1.3985263998010680e-01, -1.4000365642134394e-01
        },
// x = -1.9600
        {
                -8.1499003023764824e-02, -1.0822293573085315e-01, -1.2346704756535999e-01, -1.3178097985162449e-01,
                -1.3618609598786935e-01, -1.3847605394883850e-01, -1.3965140137105314e-01, -1.4024952024311507e-01,
                -1.4055214482017997e-01, -1.4070466693224354e-01
        },
// x = -1.9550
        {
                -8.1688445416837285e-02, -1.0863090460788931e-01, -1.2400918177153490e-01, -1.3239966957503149e-01,
                -1.3684654648199002e-01, -1.3915863426607664e-01, -1.4034548671565242e-01, -1.4094951104129014e-01,
                -1.4125514095238056e-01, -1.4140918368836145e-01
        },
// x = -1.9500
        {
                -8.1877267848726967e-02, -1.0903981915175658e-01, -1.2455335818755367e-01, -1.3302107507302399e-01,
                -1.3751009720990959e-01, -1.3984452458071206e-01, -1.4104299323037264e-01, -1.4165298081221772e-01,
                -1.4196164572887643e-01, -1.4211722418950334e-01
        },
// x = -1.9450
        {
                -8.2065458842187092e-02, -1.0944967623398293e-01, -1.2509958153395309e-01, -1.3364520656845885e-01,
                -1.3817676177604832e-01, -1.4053374042790343e-01, -1.4174393750401029e-01, -1.4235994670318797e-01,
                -1.4267167658736507e-01, -1.4282880602244899e-01
        },
// x = -1.9400
        {
                -8.2253006855386362e-02, -1.0986047266854840e-01, -1.2564785651548399e-01, -1.3427207430775651e-01,
                -1.3884655383599956e-01, -1.4122629741095960e-01, -1.4244833620320782e-01, -1.4307042594462963e-01,
                -1.4338525105148806e-01, -1.4354394686138178e-01
        },
// x = -1.9350
        {
                -8.2439900282152839e-02, -1.1027220521156310e-01, -1.2619818782082268e-01, -1.3490168856082105e-01,
                -1.3951948709664769e-01, -1.4192221120159532e-01, -1.4315620607279553e-01, -1.4378443585049988e-01,
                -1.4410238673124764e-01, -1.4426266446831867e-01
        },
// x = -1.9300
        {
                -8.2626127452227000e-02, -1.1068487056094579e-01, -1.2675058012228044e-01, -1.3553405962095935e-01,
                -1.4019557531628513e-01, -1.4262149754018841e-01, -1.4386756393613354e-01, -1.4450199381867634e-01,
                -1.4482310132342577e-01, -1.4498497669354365e-01
        },
// x = -1.9250
        {
                -8.2811676631524750e-02, -1.1109846535610440e-01, -1.2730503807551194e-01, -1.3616919780479794e-01,
                -1.4087483230472905e-01, -1.4332417223603688e-01, -1.4458242669545590e-01, -1.4522311733135032e-01,
                -1.4554741261200410e-01, -1.4571090147604226e-01
        },
// x = -1.9200
        {
                -8.2996536022409501e-02, -1.1151298617761732e-01, -1.2786156631922216e-01, -1.3680711345219967e-01,
                -1.4155727192343801e-01, -1.4403025116761686e-01, -1.4530081133221642e-01, -1.4594782395542241e-01,
                -1.4627533846858712e-01, -1.4644045684393897e-01
        },
// x = -1.9150
        {
                -8.3180693763974148e-02, -1.1192842954691655e-01, -1.2842016947487117e-01, -1.3744781692617794e-01,
                -1.4224290808562762e-01, -1.4473975028284108e-01, -1.4602273490743450e-01, -1.4667613134289909e-01,
                -1.4700689685282611e-01, -1.4717366091493622e-01
        },
// x = -1.9100
        {
                -8.3364137932333299e-02, -1.1234479192597205e-01, -1.2898085214637778e-01, -1.3809131861280949e-01,
                -1.4293175475638645e-01, -1.4545268559931812e-01, -1.4674821456204332e-01, -1.4740805723129155e-01,
                -1.4774210581284641e-01, -1.4791053189675554e-01
        },
// x = -1.9050
        {
                -8.3546856540924488e-02, -1.1276206971697722e-01, -1.2954361891982102e-01, -1.3873762892114616e-01,
                -1.4362382595279072e-01, -1.4616907320461162e-01, -1.4747726751723869e-01, -1.4814361944401622e-01,
                -1.4848098348567504e-01, -1.4865108808758132e-01
        },
// x = -1.9000
        {
                -8.3728837540820208e-02, -1.1318025926203704e-01, -1.3010847436314113e-01, -1.3938675828312533e-01,
                -1.4431913574401961e-01, -1.4688892925650135e-01, -1.4820991107483003e-01, -1.4888283589079696e-01,
                -1.4922354809767205e-01, -1.4939534787650618e-01
        },
// x = -1.8950
        {

                -8.3910068821049552e-02, -1.1359935684285623e-01, -1.3067542302583796e-01, -1.4003871715347779e-01,
                -1.4501769825146829e-01, -1.4761226998324289e-01, -1.4894616261759147e-01, -1.4962572456806886e-01,
                -1.4996981796496256e-01, -1.5014332974397876e-01
        },
// x = -1.8900
        {
                -8.4090538208929935e-02, -1.1401935868043046e-01, -1.3124446943866783e-01, -1.4069351600963514e-01,
                -1.4571952764886301e-01, -1.4833911168382991e-01, -1.4968603960961516e-01, -1.5037230355938405e-01,
                -1.5071981149387151e-01, -1.5089505226225366e-01
        },
// x = -1.8850
        {
                -8.4270233470409406e-02, -1.1444026093473839e-01, -1.3181561811334030e-01, -1.4135116535163547e-01,
                -1.4642463816237342e-01, -1.4906947072825572e-01, -1.5042955959666576e-01, -1.5112259103581907e-01,
                -1.5147354718136030e-01, -1.5165053409584364e-01
        },
// x = -1.8800
        {
                -8.4449142310418737e-02, -1.1486205970443560e-01, -1.3238887354221118e-01, -1.4201167570202688e-01,
                -1.4713304407072567e-01, -1.4980336355777563e-01, -1.5117674020653576e-01, -1.5187660525638433e-01,
                -1.5223104361546522e-01, -1.5240979400197346e-01
        },
// x = -1.8750
        {
                -8.4627252373234485e-02, -1.1528475102655067e-01, -1.3296424019797567e-01, -1.4267505760577040e-01,
                -1.4784475970531494e-01, -1.5054080668517064e-01, -1.5192759914940285e-01, -1.5263436456843510e-01,
                -1.5299231947573858e-01, -1.5317285083103702e-01
        },
// x = -1.8700
        {
                -8.4804551242852008e-02, -1.1570833087618275e-01, -1.3354172253335952e-01, -1.4334132163014068e-01,
                -1.4855979945031675e-01, -1.5128181669501017e-01, -1.5268215421818779e-01, -1.5339588740808394e-01,
                -1.5375739353369094e-01, -1.5393972352705543e-01
        },
// x = -1.8650
        {
                -8.4981026443369453e-02, -1.1613279516620084e-01, -1.3412132498080850e-01, -1.4401047836462547e-01,
                -1.4927817774279889e-01, -1.5202641024391694e-01, -1.5344042328891433e-01, -1.5416119230061578e-01,
                -1.5452628465323617e-01, -1.5471043112813831e-01
        },
// x = -1.8600
        {
                -8.5156665439381982e-02, -1.1655813974694565e-01, -1.3470305195217644e-01, -1.4468253842082332e-01,
                -1.4999990907283148e-01, -1.5277460406083132e-01, -1.5420242432106993e-01, -1.5493029786090434e-01,
                -1.5529901179113859e-01, -1.5548499276694677e-01
        },
// x = -1.8550
        {
                -8.5331455636386944e-02, -1.1698436040593303e-01, -1.3528690783841199e-01, -1.4535751243234005e-01,
                -1.5072500798359822e-01, -1.5352641494727706e-01, -1.5496817535796803e-01, -1.5570322279382992e-01,
                -1.5607559399746146e-01, -1.5626342767115875e-01
        },
// x = -1.8500
        {
                -8.5505384381200158e-02, -1.1741145286755926e-01, -1.3587289700924382e-01, -1.4603541105468321e-01,
                -1.5145348907150530e-01, -1.5428185977762648e-01, -1.5573769452711200e-01, -1.5647998589469997e-01,
                -1.5685605041601866e-01, -1.5704575516393679e-01
        },
// x = -1.8450
        {
                -8.5678438962382156e-02, -1.1783941279280900e-01, -1.3646102381286407e-01, -1.4671624496515517e-01,
                -1.5218536698629101e-01, -1.5504095549936747e-01, -1.5651100004055946e-01, -1.5726060604967046e-01,
                -1.5764040028482715e-01, -1.5783199466439768e-01
        },
// x = -1.8400
        {
                -8.5850606610676478e-02, -1.1826823577896492e-01, -1.3705129257561055e-01, -1.4740002486274514e-01,
                -1.5292065643113434e-01, -1.5580371913337010e-01, -1.5728811019528899e-01, -1.5804510223616985e-01,
                -1.5842866293656299e-01, -1.5862216568808485e-01
        },
// x = -1.8350
        {
                -8.6021874499458320e-02, -1.1869791735931985e-01, -1.3764370760164779e-01, -1.4808676146801827e-01,
                -1.5365937216276365e-01, -1.5657016777415408e-01, -1.5806904337356723e-01, -1.5883349352332468e-01,
                -1.5922085779901840e-01, -1.5941628784744274e-01
        },
// x = -1.8300
        {
                -8.6192229745194276e-02, -1.1912845300289100e-01, -1.3823827317264617e-01, -1.4877646552300475e-01,
                -1.5440152899156315e-01, -1.5734031859015699e-01, -1.5885381804331863e-01, -1.5962579907238678e-01,
                -1.6001700439556119e-01, -1.6021438085229348e-01
        },
// x = -1.8250
        {
                -8.6361659407913641e-02, -1.1955983811413666e-01, -1.3883499354745998e-01, -1.4946914779108600e-01,
                -1.5514714178168121e-01, -1.5811418882400249e-01, -1.5964245275849470e-01, -1.6042203813716241e-01,
                -1.6081712234559689e-01, -1.6101646451031576e-01
        },
// x = -1.8200
        {
                -8.6530150491690938e-02, -1.1999206803267516e-01, -1.3943387296180348e-01, -1.5016481905688001e-01,
                -1.5589622545113593e-01, -1.5889179579276991e-01, -1.6043496615944630e-01, -1.6122223006444344e-01,
                -1.6162123136503206e-01, -1.6182255872752666e-01
        },
// x = -1.8150
        {
                -8.6697689945139295e-02, -1.2042513803300624e-01, -1.4003491562792689e-01, -1.5086349012612518e-01,
                -1.5664879497192150e-01, -1.5967315688826331e-01, -1.6123137697329631e-01, -1.6202639429444041e-01,
                -1.6242935126674091e-01, -1.6263268350876478e-01
        },
// x = -1.8100
        {
                -8.6864264661915880e-02, -1.2085904332423524e-01, -1.4063812573428885e-01, -1.5156517182556126e-01,
                -1.5740486537011322e-01, -1.6045828957728206e-01, -1.6203170401431419e-01, -1.6283455036121711e-01,
                -1.6324150196103282e-01, -1.6344685895817643e-01
        },
// x = -1.8050
        {
                -8.7029861481238480e-02, -1.2129377904979943e-01, -1.4124350744523023e-01, -1.5226987500281050e-01,
                -1.5816445172597252e-01, -1.6124721140189108e-01, -1.6283596618429119e-01, -1.6364671789312712e-01,
                -1.6405770345612325e-01, -1.6426510527970450e-01
        },
// x = -1.8000
        {
                -8.7194467188413602e-02, -1.2172934028719716e-01, -1.4185106490064386e-01, -1.5297761052625600e-01,
                -1.5892756917405082e-01, -1.6203993997969265e-01, -1.6364418247291776e-01, -1.6446291661325313e-01,
                -1.6487797585860603e-01, -1.6508744277757817e-01
        },
// x = -1.7950
        {
                -8.7358068515376405e-02, -1.2216572204771961e-01, -1.4246080221564467e-01, -1.5368838928491793e-01,
                -1.5969423290329288e-01, -1.6283649300409744e-01, -1.6445637195816185e-01, -1.6528316633984608e-01,
                -1.6570233937392823e-01, -1.6591389185680666e-01
        },
// x = -1.7900
        {
                -8.7520652141242189e-02, -1.2260291927618548e-01, -1.4307272348023811e-01, -1.5440222218833002e-01,
                -1.6046445815714031e-01, -1.6363688824459713e-01, -1.6527255380664824e-01, -1.6610748698676880e-01,
                -1.6653081430686670e-01, -1.6674447302367482e-01
        },
// x = -1.7850
        {
                -8.7682204692869734e-02, -1.2304092685067805e-01, -1.4368683275898719e-01, -1.5511912016641224e-01,
                -1.6123826023363305e-01, -1.6444114354703715e-01, -1.6609274727404033e-01, -1.6693589856393934e-01,
                -1.6736342106200786e-01, -1.6757920688624037e-01
        },
// x = -1.7800
        {
                -8.7842712745436480e-02, -1.2347973958228538e-01, -1.4430313409067741e-01, -1.5583909416934322e-01,
                -1.6201565448551158e-01, -1.6524927683388968e-01, -1.6691697170542200e-01, -1.6776842117777777e-01,
                -1.6820018014422877e-01, -1.6841811415483474e-01
        },
// x = -1.7750
        {
                -8.8002162823025573e-02, -1.2391935221484365e-01, -1.4492163148798218e-01, -1.5656215516743005e-01,
                -1.6279665632031759e-01, -1.6606130610452766e-01, -1.6774524653568146e-01, -1.6860507503165362e-01,
                -1.6904111215918050e-01, -1.6926121564256513e-01
        },
// x = -1.7700
        {
                -8.8160541399224968e-02, -1.2435975942468291e-01, -1.4554232893712529e-01, -1.5728831415097774e-01,
                -1.6358128120049409e-01, -1.6687724943549898e-01, -1.6857759128989711e-01, -1.6944588042633638e-01,
                -1.6988623781377468e-01, -1.7010853226582051e-01
        },
// x = -1.7650
        {
                -8.8317834897738368e-02, -1.2480095582037599e-01, -1.4616523039754273e-01, -1.5801758213015515e-01,
                -1.6436954464348588e-01, -1.6769712498080114e-01, -1.6941402558372301e-01, -1.7029085776044717e-01,
                -1.7073557791667104e-01, -1.7096008504477803e-01
        },
// x = -1.7600
        {
                -8.8474029693008577e-02, -1.2524293594249111e-01, -1.4679033980154368e-01, -1.5874997013486092e-01,
                -1.6516146222183745e-01, -1.6852095097215666e-01, -1.7025456912377782e-01, -1.7114002753091248e-01,
                -1.7158915337876790e-01, -1.7181589510391379e-01
        },
// x = -1.7550
        {
                -8.8629112110852781e-02, -1.2568569426334650e-01, -1.4741766105396956e-01, -1.5948548921458644e-01,
                -1.6595704956329205e-01, -1.6934874571928857e-01, -1.7109924170803345e-01, -1.7199341033342044e-01,
                -1.7244698521369500e-01, -1.7267598367251505e-01
        },
// x = -1.7500
        {
                -8.8783068429110171e-02, -1.2612922518676972e-01, -1.4804719803185210e-01, -1.6022415043827815e-01,
                -1.6675632235088908e-01, -1.7018052761019695e-01, -1.7194806322620626e-01, -1.7285102686287790e-01,
                -1.7330909453830848e-01, -1.7354037208519471e-01
        },
// x = -1.7450
        {
                -8.8935884878301757e-02, -1.2657352304785860e-01, -1.4867895458407071e-01, -1.6096596489419726e-01,
                -1.6755929632306105e-01, -1.7101631511143561e-01, -1.7280105366014856e-01, -1.7371289791387057e-01,
                -1.7417550257318778e-01, -1.7440908178240969e-01
        },
// x = -1.7400
        {
                -8.9087547642302797e-02, -1.2701858211274672e-01, -1.4931293453100780e-01, -1.6171094368977854e-01,
                -1.6836598727372970e-01, -1.7185612676838891e-01, -1.7365823308424264e-01, -1.7457904438112479e-01,
                -1.7504623064313563e-01, -1.7528213431097955e-01
        },
// x = -1.7350
        {
                -8.9238042859027117e-02, -1.2746439657837211e-01, -1.4994914166420348e-01, -1.6245909795148636e-01,
                -1.6917641105240161e-01, -1.7269998120555008e-01, -1.7451962166579535e-01, -1.7544948725997053e-01,
                -1.7592130017767971e-01, -1.7615955132460978e-01
        },
// x = -1.7300
        {
                -8.9387356621124320e-02, -1.2791096057224857e-01, -1.5058757974600931e-01, -1.6321043882467012e-01,
                -1.6999058356426289e-01, -1.7354789712679880e-01, -1.7538523966543415e-01, -1.7632424764680785e-01,
                -1.7680073271157662e-01, -1.7704135458441633e-01
        },
// x = -1.7250
        {
                -8.9535474976689733e-02, -1.2835826815224166e-01, -1.5122825250924027e-01, -1.6396497747341737e-01,
                -1.7080852077027345e-01, -1.7439989331568012e-01, -1.7625510743750514e-01, -1.7720334673957394e-01,
                -1.7768454988531890e-01, -1.7792756595945300e-01
        },
// x = -1.7200
        {
                -8.9682383929986134e-02, -1.2880631330634754e-01, -1.5187116365682599e-01, -1.6472272508040506e-01,
                -1.7163023868726002e-01, -1.7525598863568356e-01, -1.7712924543047193e-01, -1.7808680583821279e-01,
                -1.7857277344564335e-01, -1.7881820742724150e-01
        },
// x = -1.7150
        {
                -8.9828069442178735e-02, -1.2925508995247559e-01, -1.5251631686146122e-01, -1.6548369284674933e-01,
                -1.7245575338800906e-01, -1.7611620203052281e-01, -1.7800767418731586e-01, -1.7897464634514682e-01,
                -1.7946542524604270e-01, -1.7971330107430386e-01
        },
// x = -1.7100
        {
                -8.9972517432082538e-02, -1.2970459193823466e-01, -1.5316371576525428e-01, -1.6624789199185355e-01,
                -1.7328508100135798e-01, -1.7698055252441519e-01, -1.7889041434593800e-01, -1.7986688976575058e-01,
                -1.8036252724727883e-01, -1.8061286909669760e-01
        },
// x = -1.7050
        {
                -9.0115713776922873e-02, -1.3015481304072365e-01, -1.5381336397937545e-01, -1.6701533375325439e-01,
                -1.7411823771228732e-01, -1.7784905922236288e-01, -1.7977748663956250e-01, -1.8076355770882616e-01,
                -1.8126410151789885e-01, -1.8151693380055320e-01
        },
// x = -1.7000
        {
                -9.0257644313107968e-02, -1.3060574696632515e-01, -1.5446526508370420e-01, -1.6778602938646633e-01,
                -1.7495523976200955e-01, -1.7872174131043317e-01, -1.8066891189714093e-01, -1.8166467188708102e-01,
                -1.8217017023475354e-01, -1.8242551760261450e-01
        },
// x = -1.6950
        {
                -9.0398294837014836e-02, -1.3105738735050329e-01, -1.5511942262647394e-01, -1.6855999016482423e-01,
                -1.7579610344805979e-01, -1.7959861805604060e-01, -1.8156471104375851e-01, -1.8257025411760758e-01,
                -1.8308075568351795e-01, -1.8333864303078151e-01
        },
// x = -1.6900
        {
                -9.0537651105788083e-02, -1.3150972775760567e-01, -1.5577584012391826e-01, -1.6933722737932427e-01,
                -1.7664084512438399e-01, -1.8047970880822803e-01, -1.8246490510104138e-01, -1.8348032632236463e-01,
                -1.8399588025921443e-01, -1.8425633272465541e-01
        },
// x = -1.6850
        {
                -9.0675698838151186e-02, -1.3196276168066934e-01, -1.5643452105991401e-01, -1.7011775233846305e-01,
                -1.7748948120142677e-01, -1.8136503299794945e-01, -1.8336951518756575e-01, -1.8439491052866153e-01,
                -1.8491556646673848e-01, -1.8517860943608744e-01
        },
// x = -1.6800
        {
                -9.0812423715231017e-02, -1.3241648254123062e-01, -1.5709546888562498e-01, -1.7090157636807493e-01,
                -1.7834202814621877e-01, -1.8225461013835240e-01, -1.8427856251926775e-01, -1.8531402886964382e-01,
                -1.8583983692138653e-01, -1.8610549602972884e-01
        },
// x = -1.6750
        {
                -9.0947811381395335e-02, -1.3287088368913885e-01, -1.5775868701914283e-01, -1.7168871081116774e-01,
                -1.7919850248246261e-01, -1.8314845982506125e-01, -1.8519206840985544e-01, -1.8623770358478056e-01,
                -1.8676871434938627e-01, -1.8703701548358437e-01
        },
// x = -1.6700
        {
                -9.1081847445103281e-02, -1.3332595840237585e-01, -1.5842417884512947e-01, -1.7247916702775598e-01,
                -1.8005892079061880e-01, -1.8404660173646062e-01, -1.8611005427122182e-01, -1.8716595702035510e-01,
                -1.8770222158842947e-01, -1.8797319088956854e-01
        },
// x = -1.6650
        {
                -9.1214517479768789e-02, -1.3378169988687696e-01, -1.5909194771445631e-01, -1.7327295639469339e-01,
                -1.8092329970799015e-01, -1.8494905563397987e-01, -1.8703254161385927e-01, -1.8809881162995640e-01,
                -1.8864038158820748e-01, -1.8891404545406390e-01
        },
// x = -1.6600
        {
                -9.1345807024636960e-02, -1.3423810127635930e-01, -1.5976199694384408e-01, -1.7407009030550280e-01,
                -1.8179165592880497e-01, -1.8585584136237671e-01, -1.8795955204727541e-01, -1.8903628997497310e-01,
                -1.8958321741094897e-01, -1.8985960249848285e-01
        },
// x = -1.6550
        {
                -9.1475701585674238e-02, -1.3469515563215220e-01, -1.6043432981550126e-01, -1.7487058017020446e-01,
                -1.8266400620430101e-01, -1.8676697885002277e-01, -1.8889110728041072e-01, -1.8997841472509008e-01,
                -1.9053075223195995e-01, -1.9080988545983119e-01
        },
// x = -1.6500
        {
                -9.1604186636470894e-02, -1.3515285594303414e-01, -1.6110894957676167e-01, -1.7567443741514266e-01,
                -1.8354036734280677e-01, -1.8768248810918847e-01, -1.8982722912205716e-01, -1.9092520865878612e-01,
                -1.9148300934016677e-01, -1.9176491789127506e-01
        },
// x = -1.6450
        {
                -9.1731247619156950e-02, -1.3561119512507191e-01, -1.6178585943972112e-01, -1.7648167348281060e-01,
                -1.8442075620982304e-01, -1.8860238923632880e-01, -1.9076793948127815e-01, -1.9187669466383458e-01,
                -1.9244001213866099e-01, -1.9272472346271055e-01
        },
// x = -1.6400
        {
                -9.1856869945331668e-02, -1.3607016602146663e-01, -1.6246506258087390e-01, -1.7729229983167247e-01,
                -1.8530518972810295e-01, -1.8952670241236932e-01, -1.9171326036783085e-01, -1.9283289573780579e-01,
                -1.9340178414524747e-01, -1.9368932596133573e-01
        },
// x = -1.6350
        {
                -9.1981038997005701e-02, -1.3652976140240289e-01, -1.6314656214074763e-01, -1.7810632793598605e-01,
                -1.8619368487773161e-01, -1.9045544790299254e-01, -1.9266321389258845e-01, -1.9379383498857128e-01,
                -1.9436834899299443e-01, -1.9465874929222551e-01
        },
// x = -1.6300
        {
                -9.2103740127556802e-02, -1.3698997396490320e-01, -1.6383036122353817e-01, -1.7892376928562037e-01,
                -1.8708625869620440e-01, -1.9138864605892511e-01, -1.9361782226796515e-01, -1.9475953563481066e-01,
                -1.9533973043078642e-01, -1.9563301747890960e-01
        },
// x = -1.6250
        {
                -9.2224958662698864e-02, -1.3745079633268770e-01, -1.6451646289674310e-01, -1.7974463538587443e-01,
                -1.8798292827850493e-01, -1.9232631731622474e-01, -1.9457710780834211e-01, -1.9573002100652054e-01,
                -1.9631595232387911e-01, -1.9661215466395268e-01
        },
// x = -1.6200
        {
                -9.2344679901463819e-02, -1.3791222105603773e-01, -1.6520487019079522e-01, -1.8056893775729163e-01,
                -1.8888371077718125e-01, -1.9326848219656792e-01, -1.9554109293049465e-01, -1.9670531454552503e-01,
                -1.9729703865445783e-01, -1.9759618510953800e-01
        },
// x = -1.6150
        {
                -9.2462889117197328e-02, -1.3837424061166548e-01, -1.6589558609869495e-01, -1.8139668793547442e-01,
                -1.8978862340242189e-01, -1.9421516130753791e-01, -1.9650980015402103e-01, -1.9768543980598902e-01,
                -1.9828301352219754e-01, -1.9858513319805296e-01
        },
// x = -1.6100
        {
                -9.2579571558567400e-02, -1.3883684740258731e-01, -1.6658861357564178e-01, -1.8222789747089599e-01,
                -1.9069768342213081e-01, -1.9516637534291328e-01, -1.9748325210177292e-01, -1.9867042045493349e-01,
                -1.9927390114482590e-01, -1.9957902343267847e-01
        },
// x = -1.6050
        {
                -9.2694712450586511e-02, -1.3930003375800365e-01, -1.6728395553866596e-01, -1.8306257792871011e-01,
                -1.9161090816200099e-01, -1.9612214508295653e-01, -1.9846147150028676e-01, -1.9966028027275248e-01,
                -2.0026972585868894e-01, -2.0057788043798028e-01
        },
// x = -1.6000
        {
                -9.2808296995647049e-02, -1.3976379193318289e-01, -1.6798161486625898e-01, -1.8390074088855937e-01,
                -1.9252831500558745e-01, -1.9708249139470321e-01, -1.9944448118021746e-01, -2.0065504315373317e-01,
                -2.0127051211931923e-01, -2.0158172896050364e-01
        },
// x = -1.5950
        {
                -9.2920310374570006e-02, -1.4022811410935107e-01, -1.6868159439800370e-01, -1.8474239794438213e-01,
                -1.9344992139437936e-01, -1.9804743523225149e-01, -2.0043230407677229e-01, -2.0165473310657692e-01,
                -2.0227628450200663e-01, -2.0259059386937059e-01
        },
// x = -1.5900
        {
                -9.3030737747666736e-02, -1.4069299239358676e-01, -1.6938389693420372e-01, -1.8558756070421598e-01,
                -1.9437574482787082e-01, -1.9901699763705197e-01, -2.0142496323014747e-01, -2.0265937425492345e-01,
                -2.0328706770237151e-01, -2.0360450015688014e-01
        },
// x = -1.5850
        {
                -9.3139564255814780e-02, -1.4115841881872077e-01, -1.7008852523551210e-01, -1.8643624079000173e-01,
                -1.9530580286363095e-01, -1.9999119973819779e-01, -2.0242248178596547e-01, -2.0366899083787690e-01,
                -2.0430288653694076e-01, -2.0462347293911154e-01
        },
// x = -1.5800
        {
                -9.3246775021546158e-02, -1.4162438534324201e-01, -1.7079548202256084e-01, -1.8728844983738319e-01,
                -1.9624011311737299e-01, -2.0097006275271526e-01, -2.0342488299571382e-01, -2.0468360721053405e-01,
                -2.0532376594372695e-01, -2.0564753745653011e-01
        },
// x = -1.5750
        {
                -9.3352355150149557e-02, -1.4209088385120869e-01, -1.7150476997558806e-01, -1.8814419949550651e-01,
                -1.9717869326302226e-01, -2.0195360798585507e-01, -2.0443219021718570e-01, -2.0570324784451458e-01,
                -2.0634973098280882e-01, -2.0667671907459609e-01
        },
// x = -1.5700
        {
                -9.3456289730785805e-02, -1.4255790615216446e-01, -1.7221639173406561e-01, -1.8900350142681735e-01,
                -1.9812156103278300e-01, -2.0294185683138272e-01, -2.0544442691492174e-01, -2.0672793732849407e-01,
                -2.0738080683691584e-01, -2.0771104328437667e-01
        },
// x = -1.5650
        {
                -9.3558563837616182e-02, -1.4302544398106101e-01, -1.7293034989632716e-01, -1.8986636730685605e-01,
                -1.9906873421720456e-01, -2.0393483077187102e-01, -2.0646161666065319e-01, -2.0775770036873820e-01,
                -2.0841701881201463e-01, -2.0875053570316038e-01
        },
// x = -1.5600
        {
                -9.3659162530944431e-02, -1.4349348899818562e-01, -1.7364664701919436e-01, -1.9073280882405089e-01,
                -2.0002023066524641e-01, -2.0493255137899122e-01, -2.0748378313374685e-01, -2.0879256178964051e-01,
                -2.0945839233789806e-01, -2.0979522207507489e-01
        },
// x = -1.5550
        {
                -9.3758070858372233e-02, -1.4396203278909475e-01, -1.7436528561760317e-01, -1.9160283767950995e-01,
                -2.0097606828434172e-01, -2.0593504031380561e-01, -2.0851095012165127e-01, -2.0983254653426120e-01,
                -2.1050495296877755e-01, -2.1084512827170773e-01
        },
// x = -1.5500
        {
                -9.3855273855967425e-02, -1.4443106686455373e-01, -1.7508626816423059e-01, -1.9247646558680992e-01,
                -2.0193626504046083e-01, -2.0694231932706006e-01, -2.0954314152034437e-01, -2.1087767966486928e-01,
                -2.1155672638387762e-01, -2.1190028029272975e-01
        },
// x = -1.5450
        {
                -9.3950756549445499e-02, -1.4490058266048170e-01, -1.7580959708912050e-01, -1.9335370427178519e-01,
                -2.0290083895817235e-01, -2.0795441025947670e-01, -2.1058038133478241e-01, -2.1192798636348523e-01,
                -2.1261373838803324e-01, -2.1296070426652158e-01
        },
// x = -1.5400
        {
                -9.4044503955365122e-02, -1.4537057153790284e-01, -1.7653527477930894e-01, -1.9423456547231266e-01,
                -2.0386980812070427e-01, -2.0897133504204690e-01, -2.1162269367935130e-01, -2.1298349193242816e-01,
                -2.1367601491228999e-01, -2.1402642645080353e-01
        },
// x = -1.5350
        {
                -9.4136501082335913e-02, -1.4584102478290420e-01, -1.7726330357844988e-01, -1.9511906093809595e-01,
                -2.0484319067000384e-01, -2.0999311569632531e-01, -2.1267010277831772e-01, -2.1404422179486352e-01,
                -2.1474358201450691e-01, -2.1509747323326767e-01
        },
// x = -1.5300
        {
                -9.4226732932239515e-02, -1.4631193360659819e-01, -1.7799368578644023e-01, -1.9600720243044800e-01,
                -2.0582100480679588e-01, -2.1101977433472313e-01, -2.1372263296628333e-01, -2.1511020149535387e-01,
                -2.1581646587996228e-01, -2.1617387113221384e-01
        },
// x = -1.5250
        {
                -9.4315184501464769e-02, -1.4678328914509245e-01, -1.7872642365904495e-01, -1.9689900172207084e-01,
                -2.0680326879064018e-01, -2.1205133316080210e-01, -2.1478030868863968e-01, -2.1618145670041120e-01,
                -2.1689469282196161e-01, -2.1725564679718806e-01
        },
// x = -1.5200
        {
                -9.4401840782154342e-02, -1.4725508245946564e-01, -1.7946151940752203e-01, -1.9779447059683436e-01,
                -2.0779000093998812e-01, -2.1308781446956912e-01, -2.1584315450202438e-01, -2.1725801319905272e-01,
                -2.1797828928244922e-01, -2.1834282700962412e-01
        },
// x = -1.5150
        {
                -9.4486686763466096e-02, -1.4772730453574875e-01, -1.8019897519824712e-01, -1.9869362084955264e-01,
                -2.0878121963223809e-01, -2.1412924064777061e-01, -2.1691119507477966e-01, -2.1833989690335781e-01,
                -2.1906728183262172e-01, -2.1943543868348853e-01
        },
// x = -1.5100
        {
                -9.4569707432846414e-02, -1.4819994628491392e-01, -1.8093879315233874e-01, -1.9959646428575856e-01,
                -2.0977694330378918e-01, -2.1517563417418736e-01, -2.1798445518741100e-01, -2.1942713384902776e-01,
                -2.2016169717354522e-01, -2.2053350886592801e-01
        },
// x = -1.5050
        {
                -9.4650887777316742e-02, -1.4867299854286906e-01, -1.8168097534528199e-01, -2.0050301272147683e-01,
                -2.1077719045009466e-01, -2.1622701761993005e-01, -2.1906295973304890e-01, -2.2051975019594766e-01,
                -2.2126156213677420e-01, -2.2163706473792089e-01
        },
// x = -1.5000
        {
                -9.4730212784773518e-02, -1.4914645207045873e-01, -1.8242552380655447e-01, -2.0141327798299480e-01,
                -2.1178197962571341e-01, -2.1728341364873416e-01, -2.2014673371791055e-01, -2.2161777222875101e-01,
                -2.2236690368497453e-01, -2.2274613361493040e-01
        },
// x = -1.4950
        {
                -9.4807667445300198e-02, -1.4962029755347184e-01, -1.8317244051925008e-01, -2.0232727190663105e-01,
                -2.1279132944436124e-01, -2.1834484501725593e-01, -2.2123580226176429e-01, -2.2272122635738581e-01,
                -2.2347774891254815e-01, -2.2386074294756253e-01
        },
// x = -1.4900
        {
                -9.4883236752493064e-02, -1.5009452560265610e-01, -1.8392172741970386e-01, -2.0324500633850315e-01,
                -2.1380525857895991e-01, -2.1941133457536810e-01, -2.2233019059839426e-01, -2.2383013911768393e-01,
                -2.2459412504626117e-01, -2.2498092032222575e-01
        },
// x = -1.4850
        {
                -9.4956905704799074e-02, -1.5056912675373849e-01, -1.8467338639711756e-01, -2.0416649313429253e-01,
                -2.1482378576168593e-01, -2.2048290526645623e-01, -2.2342992407606804e-01, -2.2494453717193205e-01,
                -2.2571605944587483e-01, -2.2610669346179460e-01
        },
// x = -1.4800
        {
                -9.5028659306866520e-02, -1.5104409146745329e-01, -1.8542741929318343e-01, -2.0509174415900833e-01,
                -2.1584692978401746e-01, -2.2155958012771473e-01, -2.2453502815800422e-01, -2.2606444730944569e-01,
                -2.2684357960477916e-01, -2.2723809022627625e-01
        },
// x = -1.4750
        {
                -9.5098482570909137e-02, -1.5151941012957679e-01, -1.8618382790171051e-01, -2.0602077128674823e-01,
                -2.1687470949678087e-01, -2.2264138229044411e-01, -2.2564552842284272e-01, -2.2718989644714424e-01,
                -2.2797671315062945e-01, -2.2837513861348016e-01
        },
// x = -1.4700
        {
                -9.5166360518081694e-02, -1.5199507305096721e-01, -1.8694261396824910e-01, -2.0695358640045849e-01,
                -2.1790714381019483e-01, -2.2372833498034728e-01, -2.2676145056511574e-01, -2.2832091163013035e-01,
                -2.2911548784598595e-01, -2.2951786675969116e-01
        },
// x = -1.4650
        {
                -9.5232278179868929e-02, -1.5247107046761471e-01, -1.8770377918971662e-01, -2.0789020139169190e-01,
                -2.1894425169391452e-01, -2.2482046151782689e-01, -2.2788282039572058e-01, -2.2945752003226982e-01,
                -2.3025993158895589e-01, -2.3066630294034551e-01
        },
// x = -1.4600
        {
                -9.5296220599486864e-02, -1.5294739254069509e-01, -1.8846732521402365e-01, -2.0883062816036341e-01,
                -2.1998605217707401e-01, -2.2591778531828252e-01, -2.2900966384239402e-01, -2.3059974895677485e-01,
                -2.3141007241383907e-01, -2.3182047557071001e-01
        },
// x = -1.4550
        {
                -9.5358172833295909e-02, -1.5342402935663260e-01, -1.8923325363969937e-01, -2.0977487861450411e-01,
                -2.2103256434832716e-01, -2.2702032989240814e-01, -2.3014200695018788e-01, -2.3174762583678943e-01,
                -2.3256593849177590e-01, -2.3298041320656504e-01
        },
// x = -1.4500
        {
                -9.5418119952227293e-02, -1.5390097092716917e-01, -1.9000156601551865e-01, -2.1072296467001372e-01,

                -2.2208380735588784e-01, -2.2812811884649001e-01, -2.3127987588194640e-01, -2.3290117823597711e-01,
                -2.3372755813139862e-01, -2.3414614454488991e-01
        },
// x = -1.4450
        {
                -9.5476047043220810e-02, -1.5437820718944073e-01, -1.9077226384012838e-01, -2.1167489825041041e-01,
                -2.2313980040756867e-01, -2.2924117588270426e-01, -2.3242329691878466e-01, -2.3406043384911107e-01,
                -2.3489495977948530e-01, -2.3531769842455239e-01
        },
// x = -1.4400
        {
                -9.5531939210675487e-02, -1.5485572800606121e-01, -1.9154534856167471e-01, -2.1263069128657971e-01,
                -2.2420056277081818e-01, -2.3035952479941540e-01, -2.3357229646056921e-01, -2.3522542050266693e-01,
                -2.3606817202161681e-01, -2.3649510382700076e-01
        },
// x = -1.4350
        {
                -9.5585781577911616e-02, -1.5533352316521279e-01, -1.9232082157743091e-01, -2.1359035571652066e-01,
                -2.2526611377275743e-01, -2.3148318949147403e-01, -2.3472690102639940e-01, -2.3639616615541731e-01,
                -2.3724722358283692e-01, -2.3767838987695969e-01
        },
// x = -1.4300
        {
                -9.5637559288645493e-02, -1.5581158238074522e-01, -1.9309868423342513e-01, -2.1455390348509090e-01,
                -2.2633647280021443e-01, -2.3261219395051583e-01, -2.3588713725509053e-01, -2.3757269889902941e-01,
                -2.3843214332831539e-01, -2.3886758584312887e-01
        },
// x = -1.4250
        {
                -9.5687257508475537e-02, -1.5628989529228002e-01, -1.9387893782406931e-01, -2.1552134654374919e-01,
                -2.2741165929975840e-01, -2.3374656226525997e-01, -2.3705303190565899e-01, -2.3875504695866495e-01,
                -2.3962296026401358e-01, -2.4006272113888572e-01
        },
// x = -1.4200
        {
                -9.5734861426380199e-02, -1.5676845146532548e-01, -1.9466158359178809e-01, -2.1649269685029657e-01,
                -2.2849169277773174e-01, -2.3488631862180784e-01, -2.3822461185780777e-01, -2.3994323869358228e-01,
                -2.4081970353735374e-01, -2.4126382532299084e-01
        },
// x = -1.4150
        {
                -9.5780356256228244e-02, -1.5724724039139573e-01, -1.9544662272664870e-01, -2.1746796636861551e-01,
                -2.2957659280028098e-01, -2.3603148730394197e-01, -2.3940190411241460e-01, -2.4113730259774080e-01,
                -2.4202240243789097e-01, -2.4247092810029700e-01
        },
// x = -1.4100
        {
                -9.5823727238299780e-02, -1.5772625148814004e-01, -1.9623405636599098e-01, -2.1844716706840711e-01,
                -2.3066637899338696e-01, -2.3718209269342524e-01, -2.4058493579202100e-01, -2.4233726730040830e-01,
                -2.4323108639798813e-01, -2.4368405932246140e-01
        },
// x = -1.4050
        {
                -9.5864959640819258e-02, -1.5820547409947805e-01, -1.9702388559405876e-01, -2.1943031092492660e-01,
                -2.3176107104289304e-01, -2.3833815927030022e-01, -2.4177373414132280e-01, -2.4354316156677072e-01,
                -2.4444578499349398e-01, -2.4490324898866159e-01
        },
// x = -1.4000
        {
                -9.5904038761499791e-02, -1.5868489749574355e-01, -1.9781611144163155e-01, -2.2041740991871683e-01,
                -2.3286068869453189e-01, -2.3949971161318812e-01, -2.4296832652766229e-01, -2.4475501429854354e-01,
                -2.4566652794442428e-01, -2.4612852724631440e-01
        },
// x = -1.3950
        {
                -9.5940949929098712e-02, -1.5916451087383623e-01, -1.9861073488565692e-01, -2.2140847603534031e-01,
                -2.3396525175395166e-01, -2.4066677439958881e-01, -2.4416874044152209e-01, -2.4597285453458670e-01,
                -2.4689334511564592e-01, -2.4735992439179866e-01
        },
// x = -1.3900
        {
                -9.5975678504984355e-02, -1.5964430335737997e-01, -1.9940775684888337e-01, -2.2240352126510879e-01,
                -2.3507478008674063e-01, -2.4183937240618034e-01, -2.4537500349702002e-01, -2.4719671145152158e-01,
                -2.4812626651756436e-01, -2.4859747087118100e-01
        },
// x = -1.3850
        {
                -9.6008209884713758e-02, -1.6012426399689064e-01, -2.0020717819949493e-01, -2.2340255760281152e-01,
                -2.3618929361844948e-01, -2.4301753050911820e-01, -2.4658714343240593e-01, -2.4842661436435023e-01,
                -2.4936532230681371e-01, -2.4984119728094542e-01
        },
// x = -1.3800
        {
                -9.6038529499621147e-02, -1.6060438176995048e-01, -2.0100899975074582e-01, -2.2440559704744154e-01,
                -2.3730881233461371e-01, -2.4420127368433592e-01, -2.4780518811055977e-01, -2.4966259272707705e-01,
                -2.5061054278695022e-01, -2.5109113436872615e-01
        },
// x = -1.3750
        {
                -9.6066622818417935e-02, -1.6108464558139127e-01, -2.0181322226059600e-01, -2.2541265160192009e-01,
                -2.3843335628077372e-01, -2.4539062700784470e-01, -2.4902916551949120e-01, -2.5090467613333378e-01,
                -2.5186195840914949e-01, -2.5234731303404384e-01
        },
// x = -1.3700
        {
                -9.6092475348802414e-02, -1.6156504426348520e-01, -2.0261984643134842e-01, -2.2642373327281903e-01,
                -2.3956294556249369e-01, -2.4658561565603329e-01, -2.5025910377284083e-01, -2.5215289431700610e-01,
                -2.5311959977290527e-01, -2.5360976432904564e-01
        },
// x = -1.3650
        {
                -9.6116072639080494e-02, -1.6204556657614355e-01, -2.0342887290928610e-01, -2.2743885407008227e-01,
                -2.4069760034537938e-01, -2.4778626490596856e-01, -2.5149503111038252e-01, -2.5340727715286254e-01,
                -2.5438349762673257e-01, -2.5487851945924839e-01
        },
// x = -1.3600
        {
                -9.6137400279796642e-02, -1.6252620120712449e-01, -2.0424030228431125e-01, -2.2845802600674389e-01,
                -2.4183734085509384e-01, -2.4899260013569552e-01, -2.5273697589852834e-01, -2.5466785465718761e-01,
                -2.5565368286887385e-01, -2.5615360978428553e-01
        },
// x = -1.3550
        {
                -9.6156443905375574e-02, -1.6300693677224729e-01, -2.0505413508958512e-01, -2.2948126109864597e-01,
                -2.4298218737737243e-01, -2.5020464682453741e-01, -2.5398496663083281e-01, -2.5593465698841472e-01,
                -2.5693018654800831e-01, -2.5743506681865730e-01
        },
// x = -1.3500
        {
                -9.6173189195773195e-02, -1.6348776181561706e-01, -2.0587037180116843e-01, -2.3050857136415370e-01,
                -2.4413216025803655e-01, -2.5142243055339708e-01, -2.5523903192850167e-01, -2.5720771444776469e-01,
                -2.5821303986396393e-01, -2.5872292223248566e-01
        },
// x = -1.3450
        {
                -9.6187621878138049e-02, -1.6396866480985578e-01, -2.0668901283766367e-01, -2.3153996882386937e-01,
                -2.4528727990300458e-01, -2.5264597700505526e-01, -2.5649920054089947e-01, -2.5848705747988432e-01,
                -2.5950227416843297e-01, -2.6001720785226989e-01
        },
// x = -1.3400
        {
                -9.6199727728482690e-02, -1.6444963415634242e-01, -2.0751005855985818e-01, -2.3257546550034344e-01,
                -2.4644756677830343e-01, -2.5387531196447455e-01, -2.5776550134605997e-01, -2.5977271667348834e-01,
                -2.6079792096569088e-01, -2.6131795566164995e-01
        },
// x = -1.3350
        {
                -9.6209492573364003e-02, -1.6493065818546204e-01, -2.0833350927036898e-01, -2.3361507341778542e-01,
                -2.4761304141007656e-01, -2.5511046131909620e-01, -2.5903796335119761e-01, -2.6106472276200471e-01,
                -2.6210001191331833e-01, -2.6262519780216931e-01
        },
// x = -1.3300
        {
                -9.6216902291574036e-02, -1.6541172515686209e-01, -2.0915936521328746e-01, -2.3465880460177138e-01,
                -2.4878372438459237e-01, -2.5635145105914381e-01, -2.6031661569322129e-01, -2.6236310662422152e-01,
                -2.6340857882292606e-01, -2.6393896657404475e-01
        },
// x = -1.3250
        {
                -9.6221942815838860e-02, -1.6589282325971824e-01, -2.0998762657382741e-01, -2.3570667107895119e-01,
                -2.4995963634825000e-01, -2.5759830727792121e-01, -2.6160148763924823e-01, -2.6366789928493661e-01,
                -2.6472365366088368e-01, -2.6525929443693747e-01
        },
// x = -1.3200
        {
                -9.6224600134528215e-02, -1.6637394061300811e-01, -2.1081829347797276e-01, -2.3675868487675264e-01,
                -2.5114079800758315e-01, -2.5885105617211607e-01, -2.6289260858712088e-01, -2.6497913191561012e-01,
                -2.6604526854905097e-01, -2.6658621401072902e-01
        },
// x = -1.3150
        {
                -9.6224860293372197e-02, -1.6685506526579394e-01, -2.1165136599212758e-01, -2.3781485802308455e-01,
                -2.5232723012926539e-01, -2.6010972404209798e-01, -2.6419000806592358e-01, -2.6629683583501912e-01,
                -2.6737345576551319e-01, -2.6791975807630025e-01
        },
// x = -1.3100
        {
                -9.6222709397188702e-02, -1.6733618519751367e-01, -2.1248684412276653e-01, -2.3887520254603850e-01,
                -2.5351895354010895e-01, -2.6137433729222104e-01, -2.6549371573650332e-01, -2.6762104250991542e-01,
                -2.6870824774531915e-01, -2.6925995957631471e-01
        },
// x = -1.3050
        {
                -9.6218133611617906e-02, -1.6781728831828030e-01, -2.1332472781608855e-01, -2.3993973047358710e-01,
                -2.5471598912706706e-01, -2.6264492243112286e-01, -2.6680376139198830e-01, -2.6895178355568539e-01,
                -2.7004967708122263e-01, -2.7060685161600517e-01
        },
// x = -1.3000
        {
                -9.6211119164866188e-02, -1.6829836246919111e-01, -2.1416501695767026e-01, -2.4100845383328351e-01,
                -2.5591835783723210e-01, -2.6392150607202725e-01, -2.6812017495831242e-01, -2.7028909073701302e-01,
                -2.7139777652442731e-01, -2.7196046746396407e-01
        },
// x = -1.2950
        {
                -9.6201652349457989e-02, -1.6877939542264431e-01, -2.1500771137212205e-01, -2.4208138465195567e-01,
                -2.5712608067783199e-01, -2.6520411493304302e-01, -2.6944298649473669e-01, -2.7163299596854495e-01,
                -2.7275257898533523e-01, -2.7332084055293715e-01
        },
// x = -1.2900
        {
                -9.6189719523995645e-02, -1.6926037488266532e-01, -2.1585281082274513e-01, -2.4315853495540210e-01,
                -2.5833917871622664e-01, -2.6649277583746561e-01, -2.7077222619437646e-01, -2.7298353131555875e-01,
                -2.7411411753429726e-01, -2.7468800448062236e-01
        },
// x = -1.2850
        {
                -9.6175307114927028e-02, -1.6974128848524117e-01, -2.1670031501119041e-01, -2.4423991676808332e-01,
                -2.5955767307990185e-01, -2.6778751571407783e-01, -2.7210792438472631e-01, -2.7434072899463297e-01,
                -2.7548242540236950e-01, -2.7606199301047013e-01
        },
// x = -1.2800
        {
                -9.6158401618321579e-02, -1.7022212379866486e-01, -2.1755022357711953e-01, -2.4532554211281363e-01,
                -2.6078158495646153e-01, -2.6908836159744998e-01, -2.7345011152818949e-01, -2.7570462137432067e-01,
                -2.7685753598206980e-01, -2.7744284007249009e-01
        },
// x = -1.2750
        {
                -9.6138989601652652e-02, -1.7070286832388709e-01, -2.1840253609786733e-01, -2.4641542301044966e-01,
                -2.6201093559361871e-01, -2.7039534062824055e-01, -2.7479881822260760e-01, -2.7707524097582575e-01,
                -2.7823948282814087e-01, -2.7883057976405951e-01
        },
// x = -1.2700
        {
                -9.6117057705588149e-02, -1.7118350949487834e-01, -2.1925725208810537e-01, -2.4750957147957858e-01,
                -2.6324574629918568e-01, -2.7170848005349724e-01, -2.7615407520179081e-01, -2.7845262047368091e-01,
                -2.7962829965831398e-01, -2.8022524635073637e-01
        },
// x = -1.2650
        {
                -9.6092592645787747e-02, -1.7166403467899927e-01, -2.2011437099950856e-01, -2.4860799953620319e-01,
                -2.6448603844106100e-01, -2.7302780722695669e-01, -2.7751591333605086e-01, -2.7983679269642836e-01,
                -2.8102402035407781e-01, -2.8162687426707683e-01
        },
// x = -1.2600
        {
                -9.6065581214707166e-02, -1.7214443117737985e-01, -2.2097389222042307e-01, -2.4971071919342674e-01,
                -2.6573183344721607e-01, -2.7435334960934532e-01, -2.7888436363273633e-01, -2.8122779062730502e-01,
                -2.8242667896145018e-01, -2.8303549811745449e-01
        },
// x = -1.2550
        {
                -9.6036010283408663e-02, -1.7262468622530830e-01, -2.2183581507553554e-01, -2.5081774246113503e-01,
                -2.6698315280567991e-01, -2.7568513476867973e-01, -2.8025945723676687e-01, -2.8262564740492752e-01,
                -2.8383630969175377e-01, -2.8445115267688725e-01
        },
// x = -1.2500
        {
                -9.6003866803378690e-02, -1.7310478699262843e-01, -2.2270013882554560e-01, -2.5192908134567754e-01,
                -2.6824001806452274e-01, -2.7702319038056644e-01, -2.8164122543117165e-01, -2.8403039632398269e-01,
                -2.8525294692239372e-01, -2.8587387289186367e-01
        },
// x = -1.2450
        {
                -9.5969137808351104e-02, -1.7358472058414595e-01, -2.2356686266683937e-01, -2.5304474784954661e-01,
                -2.6950245083183633e-01, -2.7836754422850296e-01, -2.8302969963762709e-01, -2.8544207083591877e-01,
                -2.8667662519764020e-01, -2.8730369388117660e-01
        },
// x = -1.2400
        {
                -9.5931810416136548e-02, -1.7406447404004524e-01, -2.2443598573116488e-01, -2.5416475397105476e-01,
                -2.7077047277571475e-01, -2.7971822420417641e-01, -2.8442491141699744e-01, -2.8686070454963986e-01,
                -2.8810737922941426e-01, -2.8874065093675882e-01
        },
// x = -1.2350
        {
                -9.5891871830458067e-02, -1.7454403433631344e-01, -2.2530750708531086e-01, -2.5528911170401175e-01,
                -2.7204410562423315e-01, -2.8107525830776475e-01, -2.8582689246987653e-01, -2.8828633123220382e-01,
                -2.8954524389807623e-01, -2.9018477952452332e-01
        },
// x = -1.2300
        {
                -9.5849309342791067e-02, -1.7502338838517506e-01, -2.2618142573078615e-01, -2.5641783303739762e-01,
                -2.7332337116542299e-01, -2.8243867464823680e-01, -2.8723567463713101e-01, -2.8971898480952196e-01,
                -2.9099025425321901e-01, -2.9163611528520789e-01
        },
// x = -1.2250
        {
                -9.5804110334209588e-02, -1.7550252303553499e-01, -2.2705774060350223e-01, -2.5755092995503581e-01,
                -2.7460829124724839e-01, -2.8380850144365061e-01, -2.8865128990044492e-01, -2.9115869936706112e-01,
                -2.9244244551446347e-01, -2.9309469403522326e-01
        },
// x = -1.2200
        {
                -9.5756262277237617e-02, -1.7598142507343140e-01, -2.2793645057345774e-01, -2.5868841443526536e-01,
                -2.7589888777757826e-01, -2.8518476702145379e-01, -2.9007377038286569e-01, -2.9260550915055067e-01,
                -2.9390185307225775e-01, -2.9456055176750434e-01
        },
// x = -1.2150
        {
                -9.5705752737704569e-02, -1.7646008122249784e-01, -2.2881755444442559e-01, -2.5983029845060934e-01,
                -2.7719518272416005e-01, -2.8656749981878399e-01, -2.9150314834935298e-01, -2.9405944856668914e-01,
                -2.9536851248868151e-01, -2.9603372465236782e-01
        },
// x = -1.2100
        {
                -9.5652569376605240e-02, -1.7693847814443389e-01, -2.2970105095364129e-01, -2.6097659396744421e-01,
                -2.7849719811458767e-01, -2.8795672838276565e-01, -2.9293945620732570e-01, -2.9552055218385592e-01,
                -2.9684245949825122e-01, -2.9751424903837059e-01
        },
// x = -1.2050
        {
                -9.5596699951965111e-02, -1.7741660243948604e-01, -2.3058693877149544e-01, -2.6212731294566433e-01,
                -2.7980495603627098e-01, -2.8935248137081232e-01, -2.9438272650721503e-01, -2.9698885473282471e-01,
                -2.9832373000873108e-01, -2.9900216145317593e-01
        },
// x = -1.2000
        {
                -9.5538132320708180e-02, -1.7789444064693713e-01, -2.3147521650122707e-01, -2.6328246733834954e-01,
                -2.8111847863640266e-01, -2.9075478755092299e-01, -2.9583299194301482e-01, -2.9846439110748019e-01,
                -2.9981236010194690e-01, -3.0049749860442010e-01
        },
// x = -1.1950
        {
                -9.5476854440530445e-02, -1.7837197924560602e-01, -2.3236588267862041e-01, -2.6444206909142687e-01,
                -2.8243778812192277e-01, -2.9216367580198249e-01, -2.9729028535283675e-01, -2.9994719636553729e-01,
                -3.0130838603460297e-01, -3.0200029738058509e-01
        },
// x = -1.1900
        {
                -9.5412854371774589e-02, -1.7884920465435583e-01, -2.3325893577170351e-01, -2.6560613014333306e-01,
                -2.8376290675948235e-01, -2.9357917511405934e-01, -2.9875463971946398e-01, -3.0143730572926242e-01,
                -3.0281184423910307e-01, -3.0351059485187637e-01
        },
// x = -1.1850
        {
                -9.5346120279309873e-02, -1.7932610323261183e-01, -2.3415437418044996e-01, -2.6677466242467407e-01,
                -2.8509385687540595e-01, -2.9500131458870493e-01, -3.0022608817090923e-01, -3.0293475458619906e-01,
                -3.0432277132437541e-01, -3.0502842827110155e-01
        },
// x = -1.1800
        {
                -9.5276640434414181e-02, -1.7980266128088906e-01, -2.3505219623648327e-01, -2.6794767785788698e-01,
                -2.8643066085564989e-01, -2.9643012343925101e-01, -3.0170466398097184e-01, -3.0443957848989489e-01,
                -3.0584120407670085e-01, -3.0655383507455591e-01
        },
// x = -1.1750
        {
                -9.5204403216658476e-02, -1.8027886504132853e-01, -2.3595240020278260e-01, -2.6912518835689375e-01,
                -2.8777334114576203e-01, -2.9786563099110830e-01, -3.0319040056979807e-01, -3.0595181316063241e-01,
                -3.0736717946054398e-01, -3.0808685288291160e-01
        },
// x = -1.1700
        {
                -9.5129397115794878e-02, -1.8075470069824426e-01, -2.3685498427339402e-01, -2.7030720582675943e-01,
                -2.8912192025083777e-01, -2.9930786668206405e-01, -3.0468333150444221e-01, -3.0747149448616151e-01,
                -3.0890073461938949e-01, -3.0962751950211009e-01
        },
// x = -1.1650
        {
                -9.5051610733644959e-02, -1.8123015437867826e-01, -2.3775994657314112e-01, -2.7149374216334721e-01,
                -2.9047642073547514e-01, -3.0075686006257984e-01, -3.0618349049942889e-01, -3.0899865852243630e-01,
                -3.1044190687658124e-01, -3.1117587292425969e-01
        },
// x = -1.1600
        {
                -9.4971032785991616e-02, -1.8170521215296534e-01, -2.3866728515734062e-01, -2.7268480925296934e-01,
                -2.9183686522372743e-01, -3.0221264079608867e-01, -3.0769091141731719e-01, -3.1053334149435297e-01,
                -3.1199073373616432e-01, -3.1273195132853687e-01
        },
// x = -1.1550
        {
                -9.4887652104472017e-02, -1.8217986003530862e-01, -2.3957699801152085e-01, -2.7388041897204124e-01,
                -2.9320327639905497e-01, -3.0367523865929269e-01, -3.0920562826926673e-01, -3.1207557979649175e-01,
                -3.1354725288373286e-01, -3.1429579308209171e-01
        },
// x = -1.1500
        {
                -9.4801457638471476e-02, -1.8265408398436270e-01, -2.4048908305114161e-01, -2.7508058318673106e-01,
                -2.9457567700427478e-01, -3.0514468354245966e-01, -3.1072767521560413e-01, -3.1362540999386174e-01,
                -3.1511150218727885e-01, -3.1586743674095774e-01
        },
// x = -1.1450
        {
                -9.4712438457018641e-02, -1.8312786990382759e-01, -2.4140353812131840e-01, -2.7628531375260795e-01,
                -2.9595408984150867e-01, -3.0662100544971943e-01, -3.1225708656639234e-01, -3.1518286882264723e-01,
                -3.1668351969804814e-01, -3.1744692105096683e-01
        },
// x = -1.1400
        {
                -9.4620583750680998e-02, -1.8360120364305138e-01, -2.4232036099654941e-01, -2.7749462251429041e-01,
                -2.9733853777212826e-01, -3.0810423449936075e-01, -3.1379389678199943e-01, -3.1674799319095909e-01,
                -3.1826334365139552e-01, -3.1903428494866609e-01
        },
// x = -1.1350
        {
                -9.4525882833461652e-02, -1.8407407099764286e-01, -2.4323954938044459e-01, -2.7870852130509222e-01,
                -2.9872904371670117e-01, -3.0959440092412777e-01, -3.1533814047367198e-01, -3.1832082017958613e-01,
                -3.1985101246764902e-01, -3.2062956756224253e-01
        },
// x = -1.1300
        {
                -9.4428325144695069e-02, -1.8454645771009304e-01, -2.4416110090545884e-01, -2.7992702194666763e-01,
                -3.0012563065493153e-01, -3.1109153507151438e-01, -3.1688985240410594e-01, -3.1990138704275228e-01,
                -3.2144656475297312e-01, -3.2223280821244793e-01
        },
// x = -1.1250
        {
                -9.4327900250943633e-02, -1.8501834947040782e-01, -2.4508501313262848e-01, -2.8115013624865481e-01,
                -3.0152832162560289e-01, -3.1259566740406192e-01, -3.1844906748802315e-01, -3.2148973120887503e-01,
                -3.2305003930023896e-01, -3.2384404641353215e-01
        },
// x = -1.1200
        {
                -9.4224597847892402e-02, -1.8548973191674670e-01, -2.4601128355130938e-01, -2.8237787600831754e-01,
                -3.0293713972651531e-01, -3.1410682849965149e-01, -3.2001582079274560e-01, -3.2308589028132600e-01,
                -3.2466147508989657e-01, -3.2546332187417765e-01
        },
// x = -1.1150
        {
                -9.4118407762245004e-02, -1.8596059063607523e-01, -2.4693990957892059e-01, -2.8361025301018722e-01,
                -3.0435210811442426e-01, -3.1562504905180233e-01, -3.2159014753877446e-01, -3.2468990203919673e-01,
                -3.2628091129085202e-01, -3.2709067449844009e-01
        },
// x = -1.1100
        {
                -9.4009319953616111e-02, -1.8643091116482347e-01, -2.4787088856068884e-01, -2.8484727902570151e-01,
                -3.0577325000497474e-01, -3.1715035986996387e-01, -3.2317208310036777e-01, -3.2630180443806495e-01,
                -3.2790838726134752e-01, -3.2872614438669201e-01
        },
// x = -1.1050
        {
                -9.3897324516424283e-02, -1.8690067898955634e-01, -2.4880421776939854e-01, -2.8608896581284349e-01,
                -3.0720058867263522e-01, -3.1868279187981130e-01, -3.2476166300612108e-01, -3.2792163561076587e-01,
                -3.2954394254984631e-01, -3.3036977183657340e-01
        },
// x = -1.1000
        {
                -9.3782411681781613e-02, -1.8736987954765114e-01, -2.4973989440514308e-01, -2.8733532511577814e-01,
                -3.0863414745063061e-01, -3.2022237612353815e-01, -3.2635892293954982e-01, -3.2954943386816415e-01,
                -3.3118761689592024e-01, -3.3202159734394293e-01
        },
// x = -1.0950
        {
                -9.3664571819382508e-02, -1.8783849822798684e-01, -2.5067791559508174e-01, -2.8858636866448961e-01,
                -3.1007394973087044e-01, -3.2176914376015170e-01, -3.2796389873967186e-01, -3.3118523769993102e-01,
                -3.3283945023114236e-01, -3.3368166160383755e-01
        },
// x = -1.0900
        {
                -9.3543795439389177e-02, -1.8830652037164114e-01, -2.5161827839319800e-01, -2.8984210817441380e-01,
                -3.1152001896387882e-01, -3.2332312606576513e-01, -3.2957662640159363e-01, -3.3282908577532233e-01,
                -3.3449948267998270e-01, -3.3535000551143357e-01
        },
// x = -1.0850
        {
                -9.3420073194314626e-02, -1.8877393127259770e-01, -2.5256097978006309e-01, -2.9110255534607404e-01,
                -3.1297237865871874e-01, -3.2488435443389074e-01, -3.3119714207709433e-01, -3.3448101694396154e-01,
                -3.3616775456070819e-01, -3.3702667016301352e-01
        },
// x = -1.0800
        {
                -9.3293395880902547e-02, -1.8924071617846228e-01, -2.5350601666260164e-01, -2.9236772186471194e-01,
                -3.1443105238291913e-01, -3.2645286037573190e-01, -3.3282548207521584e-01, -3.3614107023662271e-01,
                -3.3784430638628610e-01, -3.3871169685693842e-01
        },
// x = -1.0750
        {
                -9.3163754442003699e-02, -1.8970686029118949e-01, -2.5445338587386213e-01, -2.9363761939991900e-01,
                -3.1589606376239504e-01, -3.2802867552047593e-01, -3.3446168286285038e-01, -3.3780928486602119e-01,
                -3.3952917886529266e-01, -3.4040512709462178e-01
        },
// x = -1.0700
        {
                -9.3031139968449056e-02, -1.9017234876781650e-01, -2.5540308417278956e-01, -2.9491225960526751e-01,
                -3.1736743648137011e-01, -3.2961183161558566e-01, -3.3610578106533173e-01, -3.3948570022760099e-01,
                -3.4122241290282357e-01, -3.4210700258151178e-01
        },
// x = -1.0650
        {
                -9.2895543700917474e-02, -1.9063716672120903e-01, -2.5635510824400426e-01, -2.9619165411793874e-01,
                -3.1884519428229458e-01, -3.3120236052709034e-01, -3.3775781346702749e-01, -3.4117035590033146e-01,
                -3.4292404960141076e-01, -3.4381736522807460e-01
        },
// x = -1.0600
        {
                -9.2756957031800355e-02, -1.9110129922081465e-01, -2.5730945469758143e-01, -2.9747581455834937e-01,
                -3.2032936096576353e-01, -3.3280029423987673e-01, -3.3941781701193224e-01, -3.4286329164750173e-01,
                -3.4463413026194140e-01, -3.4553625715078529e-01
        },
// x = -1.0550
        {
                -9.2615371507061200e-02, -1.9156473129342594e-01, -2.5826612006883676e-01, -2.9876475252978185e-01,
                -3.2181996039043137e-01, -3.3440566485798051e-01, -3.4108582880426236e-01, -3.4456454741752135e-01,
                -3.4635269638458088e-01, -3.4726372067312028e-01
        },
// x = -1.0500
        {
                -9.2470778828089017e-02, -1.9202744792395288e-01, -2.5922510081811467e-01, -3.0005847961800569e-01,
                -3.2331701647292416e-01, -3.3601850460487481e-01, -3.4276188610905367e-01, -3.4627416334472322e-01,
                -3.4807978966970199e-01, -3.4899979832655792e-01
        },
// x = -1.0450
        {
                -9.2323170853547701e-02, -1.9248943405620556e-01, -2.6018639333058152e-01, -3.0135700739090610e-01,
                -3.2482055318775382e-01, -3.3763884582376097e-01, -3.4444602635275823e-01, -3.4799217975016827e-01,
                -3.4981545201881525e-01, -3.5074453285157992e-01
        },
// x = -1.0400
        {
                -9.2172539601218142e-02, -1.9295067459368334e-01, -2.6114999391602112e-01, -3.0266034739810466e-01,
                -3.2633059456722541e-01, -3.3926672097785743e-01, -3.4613828712384381e-01, -3.4971863714245455e-01,
                -3.5155972553550519e-01, -3.5249796719868132e-01
        },
// x = -1.0350
        {
                -9.2018877249835554e-02, -1.9341115440037696e-01, -2.6211589880863634e-01, -3.0396851117058293e-01,
                -3.2784716470134523e-01, -3.4090216265068835e-01, -3.4783870617339535e-01, -3.5145357621852896e-01,
                -3.5331265252636945e-01, -3.5426014452938120e-01
        },
// x = -1.0300
        {
                -9.1862176140918780e-02, -1.9387085830157672e-01, -2.6308410416685157e-01, -3.0528151022030492e-01,
                -3.2937028773772808e-01, -3.4254520354637197e-01, -3.4954732141571598e-01, -3.5319703786450130e-01,
                -3.5507427550196363e-01, -3.5603110821724199e-01
        },
// x = -1.0250
        {
                -9.1702428780594231e-02, -1.9432977108469129e-01, -2.6405460607312409e-01, -3.0659935603983623e-01,
                -3.3089998788149955e-01, -3.4419587648990851e-01, -3.5126417092893258e-01, -3.5494906315646224e-01,
                -3.5684463717774773e-01, -3.5781090184889031e-01
        },
// x = -1.0200
        {
                -9.1539627841410537e-02, -1.9478787750007534e-01, -2.6502740053375468e-01, -3.0792206010196466e-01,
                -3.3243628939519893e-01, -3.4585421442746728e-01, -3.5298929295559883e-01, -3.5670969336130437e-01,
                -3.5862378047503957e-01, -3.5959956922504438e-01
        },
// x = -1.0150
        {
                -9.1373766164148038e-02, -1.9524516226186597e-01, -2.6600248347870464e-01, -3.0924963385931903e-01,
                -3.3397921659867963e-01, -3.4752025042667423e-01, -3.5472272590330367e-01, -3.5847896993754474e-01,
                -3.6041174852196961e-01, -3.6139715436154585e-01
        },
// x = -1.0100
        {
                -9.1204836759617908e-02, -1.9570161004882877e-01, -2.6697985076141745e-01, -3.1058208874398591e-01,
                -3.3552879386900597e-01, -3.4919401767689634e-01, -3.5646450834527771e-01, -3.6025693453615260e-01,
                -3.6220858465444206e-01, -3.6320370149039555e-01
        },
// x = -1.0050
        {
                -9.1032832810454614e-02, -1.9615720550521173e-01, -2.6795949815864412e-01, -3.1191943616712864e-01,
                -3.3708504564035185e-01, -3.5087554948952981e-01, -3.5821467902100379e-01, -3.6204362900137838e-01,
                -3.6401433241709863e-01, -3.6501925506079608e-01
        },
// x = -1.0000
        {
                -9.0857747672898739e-02, -1.9661193324160967e-01, -2.6894142137027183e-01, -3.1326168751860123e-01,

                -3.3864799640389287e-01, -3.5256487929828423e-01, -3.5997327683682706e-01, -3.6383909537158732e-01,
                -3.6582903556428725e-01, -3.6684385974019634e-01
        },
// x = -0.9950
        {
                -9.0679574878570812e-02, -1.9706577783583637e-01, -2.6992561601915854e-01, -3.1460885416656487e-01,
                -3.4021767070770031e-01, -3.5426204065946659e-01, -3.6174034086656781e-01, -3.6564337588009449e-01,
                -3.6765273806103455e-01, -3.6867756041534377e-01
        },
// x = -0.9900
        {
                -9.0498308136236050e-02, -1.9751872383380562e-01, -2.7091207765097186e-01, -3.1596094745710224e-01,
                -3.4179409315663123e-01, -3.5596706725226696e-01, -3.6351591035213471e-01, -3.6745651295600384e-01,
                -3.6948548408402349e-01, -3.7052040219333909e-01
        },
// x = -0.9850
        {
                -9.0313941333558773e-02, -1.9797075575042161e-01, -2.7190080173402964e-01, -3.1731797871383138e-01,
                -3.4337728841221682e-01, -3.5767999287904151e-01, -3.6530002470414064e-01, -3.6927854922505093e-01,
                -3.7132731802257296e-01, -3.7237243040269852e-01
        },
// x = -0.9800
        {
                -9.0126468538846941e-02, -1.9842185807047771e-01, -2.7289178365915007e-01, -3.1867995923751741e-01,
                -3.4496728119254971e-01, -3.5940085146559453e-01, -3.6709272350251848e-01, -3.7110952751044707e-01,
                -3.7317828447962409e-01, -3.7423369059441786e-01
        },
// x = -0.9750
        {
                -8.9935884002785668e-02, -1.9887201524956433e-01, -2.7388501873950150e-01, -3.2004690030568633e-01,
                -3.4656409627216805e-01, -3.6112967706146321e-01, -3.6889404649713914e-01, -3.7294949083372775e-01,
                -3.7503842827273015e-01, -3.7610422854304348e-01
        },
// x = -0.9700
        {
                -8.9742182160160500e-02, -1.9932121171498460e-01, -2.7488050221045823e-01, -3.2141881317223503e-01,
                -3.4816775848193871e-01, -3.6286650384019725e-01, -3.7070403360843140e-01, -3.7479848241560404e-01,
                -3.7690779443504868e-01, -3.7798409024774893e-01
        },
// x = -0.9650
        {
                -8.9545357631568359e-02, -1.9976943186667911e-01, -2.7587822922946359e-01, -3.2279570906704247e-01,
                -3.4977829270893918e-01, -3.6461136609964201e-01, -3.7252272492800276e-01, -3.7665654567681683e-01,
                -3.7878642821634106e-01, -3.7987332193341405e-01
        },
// x = -0.9600
        {
                -8.9345405225117566e-02, -2.0021666007815997e-01, -2.7687819487589321e-01, -3.2417759919557998e-01,
                -3.5139572389633411e-01, -3.6636429826221817e-01, -3.7435016071926014e-01, -3.7852372423899455e-01,
                -3.8067437508397439e-01, -3.8177197005171137e-01
        },
// x = -0.9550
        {
                -8.9142319938115652e-02, -2.0066288069745225e-01, -2.7788039415092647e-01, -3.2556449473852039e-01,
                -3.5302007704325439e-01, -3.6812533487520205e-01, -3.7618638141803573e-01, -3.8040006192551307e-01,
                -3.8257168072392750e-01, -3.8368008128219599e-01
        },
// x = -0.9500
        {
                -8.8936096958743535e-02, -2.0110807804804345e-01, -2.7888482197742143e-01, -3.2695640685134669e-01,
                -3.5465137720467110e-01, -3.6989451061100581e-01, -3.7803142763321052e-01, -3.8228560276235979e-01,
                -3.8447839104180265e-01, -3.8559770253340209e-01
        },
// x = -0.9450
        {
                -8.8726731667719175e-02, -2.0155223642984296e-01, -2.7989147319979385e-01, -3.2835334666395932e-01,
                -3.5628964949126929e-01, -3.7167186026745486e-01, -3.7988534014734049e-01, -3.8418039097900092e-01,
                -3.8639455216384005e-01, -3.8752488094394277e-01
        },
// x = -0.9400
        {
                -8.8514219639945407e-02, -2.0199534012014858e-01, -2.8090034258390223e-01, -3.2975532528028556e-01,
                -3.5793491906931879e-01, -3.7345741876806815e-01, -3.8174815991728556e-01, -3.8608447100925036e-01,
                -3.8832021043793802e-01, -3.8946166388361564e-01
        },
// x = -0.9350
        {
                -8.8298556646145845e-02, -2.0243737337462150e-01, -2.8191142481693698e-01, -3.3116235377788483e-01,
                -3.5958721116054315e-01, -3.7525122116233423e-01, -3.8361992807483752e-01, -3.8799788749214392e-01,
                -3.9025541243467621e-01, -3.9140809895451478e-01
        },
// x = -0.9300
        {
                -8.8079738654486095e-02, -2.0287832042826937e-01, -2.8292471450731510e-01, -3.3257444320755530e-01,
                -3.6124655104198661e-01, -3.7705330262598824e-01, -3.8550068592735143e-01, -3.8992068527281448e-01,
                -3.9220020494834473e-01, -3.9336423399214537e-01
        },
// x = -0.9250
        {
                -8.7857761832180081e-02, -2.0331816549643755e-01, -2.8394020618457866e-01, -3.3399160459294075e-01,
                -3.6291296404588008e-01, -3.7886369846128876e-01, -3.8739047495837703e-01, -3.9185290940337253e-01,
                -3.9415463499797665e-01, -3.9533011706654614e-01
        },
// x = -0.9200
        {
                -8.7632622547083511e-02, -2.0375689277580872e-01, -2.8495789429929946e-01, -3.3541384893013509e-01,
                -3.6458647555950435e-01, -3.8068244409729396e-01, -3.8928933682829231e-01, -3.9379460514378717e-01,
                -3.9611874982838396e-01, -3.9730579648341441e-01
        },
// x = -0.9150
        {
                -8.7404317369269707e-02, -2.0419448644540941e-01, -2.8597777322298829e-01, -3.3684118718728701e-01,
                -3.6626711102505022e-01, -3.8250957509013467e-01, -3.9119731337493829e-01, -3.9574581796277380e-01,
                -3.9809259691120091e-01, -3.9929132078523860e-01
        },
// x = -0.9100
        {
                -8.7172843072591266e-02, -2.0463093066762547e-01, -2.8699983724800915e-01, -3.3827363030420554e-01,
                -3.6795489593947900e-01, -3.8434512712329022e-01, -3.9311444661425454e-01, -3.9770659353868171e-01,
                -4.0007622394592884e-01, -4.0128673875243426e-01
        },
// x = -0.9050
        {
                -8.6938196636225457e-02, -2.0506620958922445e-01, -2.8802408058749795e-01, -3.3971118919196247e-01,
                -3.6964985585437915e-01, -3.8618913600786120e-01, -3.9504077874091709e-01, -3.9967697776038608e-01,
                -4.0206967886098632e-01, -4.0329209940448624e-01
        },
// x = -0.9000
        {
                -8.6700375246203720e-02, -2.0550030734238678e-01, -2.8905049737528848e-01, -3.4115387473249903e-01,
                -3.7135201637582205e-01, -3.8804163768284250e-01, -3.9697635212897725e-01, -4.0165701672818382e-01,
                -4.0407300981476502e-01, -4.0530745200109658e-01
        },
// x = -0.8950
        {
                -8.6459376296923865e-02, -2.0593320804574283e-01, -2.9007908166583990e-01, -3.4260169777822497e-01,
                -3.7306140316421493e-01, -3.8990266821539515e-01, -3.9892120933250164e-01, -4.0364675675469208e-01,
                -4.0608626519668817e-01, -4.0733284604333519e-01
        },
// x = -0.8900
        {
                -8.6215197392645632e-02, -2.0636489580541939e-01, -2.9110982743417413e-01, -3.4405466915162519e-01,
                -3.7477804193415248e-01, -3.9177226380111818e-01, -4.0087539308621334e-01, -4.0564624436574975e-01,
                -4.0810949362827492e-01, -4.0936833127480049e-01
        },
// x = -0.8850
        {
                -8.5967836348967333e-02, -2.0679535471609195e-01, -2.9214272857581314e-01, -3.4551279964486048e-01,
                -3.7650195845426643e-01, -3.9365046076431809e-01, -4.0283894630613531e-01, -4.0765552630132190e-01,
                -4.1014274396420852e-01, -4.1141395768277950e-01
        },
// x = -0.8800
        {
                -8.5717291194288087e-02, -2.0722456886204574e-01, -2.9317777890672803e-01, -3.4697610001937024e-01,
                -3.7823317854707217e-01, -3.9553729555827966e-01, -4.0481191209023437e-01, -4.0967464951640864e-01,
                -4.1218606529340929e-01, -4.1346977549941871e-01
        },
// x = -0.8750
        {
                -8.5463560171247205e-02, -2.0765252231824241e-01, -2.9421497216328518e-01, -3.4844458100547548e-01,
                -3.7997172808881530e-01, -3.9743280476553439e-01, -4.0679433371906687e-01, -4.1170366118195578e-01,
                -4.1423950694011197e-01, -4.1553583520289689e-01
        },
// x = -0.8700
        {
                -8.5206641738149194e-02, -2.0807919915139605e-01, -2.9525430200220598e-01, -3.4991825330197884e-01,
                -3.8171763300931344e-01, -3.9933702509812985e-01, -4.0878625465642549e-01, -4.1374260868577056e-01,
                -4.1630311846494777e-01, -4.1761218751860441e-01
        },
// x = -0.8650
        {
                -8.4946534570366744e-02, -2.0850458342105407e-01, -2.9629576200052704e-01, -3.5139712757576802e-01,
                -3.8347091929179922e-01, -4.0124999339789524e-01, -4.1078771854998691e-01, -4.1579153963343751e-01,
                -4.1837694966603123e-01, -4.1969888342032757e-01
        },
// x = -0.8600
        {
                -8.4683237561725980e-02, -2.0892865918068626e-01, -2.9733934565556702e-01, -3.5288121446141657e-01,
                -3.8523161297275982e-01, -4.0317174663670968e-01, -4.1279876923196274e-01, -4.1785050184924138e-01,
                -4.2046105058005084e-01, -4.2179597413144021e-01
        },
// x = -0.8550
        {
                -8.4416749825870829e-02, -2.0935141047878053e-01, -2.9838504638489854e-01, -3.5437052456078560e-01,
                -3.8699974014177219e-01, -4.0510232191676721e-01, -4.1481945071974941e-01, -4.1991954337709042e-01,
                -4.2255547148336531e-01, -4.2390351112609753e-01
        },
// x = -0.8500
        {
                -8.4147070697609044e-02, -2.0977282135994538e-01, -2.9943285752632687e-01, -3.5586506844262428e-01,
                -3.8877532694134093e-01, -4.0704175647084273e-01, -4.1684980721658060e-01, -4.2199871248144438e-01,
                -4.2466026289310438e-01, -4.2602154613043836e-01
        },
// x = -0.8450
        {
                -8.3874199734235308e-02, -2.1019287586601909e-01, -3.0048277233787324e-01, -3.5736485664217077e-01,
                -3.9055839956673039e-01, -4.0899008766255474e-01, -4.1888988311218056e-01, -4.2408805764824442e-01,
                -4.2677547556827239e-01, -4.2815013112379136e-01
        },
// x = -0.8400
        {
                -8.3598136716835386e-02, -2.1061155803718540e-01, -3.0153478399776285e-01, -3.5886989966075500e-01,
                -3.9234898426579679e-01, -4.1094735298663032e-01, -4.2093972298342019e-01, -4.2618762758584894e-01,
                -4.2890116051085964e-01, -4.3028931833988721e-01
        },
// x = -0.8350
        {
                -8.3318881651567811e-02, -2.1102885191309523e-01, -3.0258888560442010e-01, -3.6038020796539805e-01,
                -3.9414710733881708e-01, -4.1291359006916689e-01, -4.2299937159497103e-01, -4.2829747122596934e-01,
                -4.3103736896695655e-01, -4.3243916026807616e-01
        },
// x = -0.8300
        {
                -8.3036434770924969e-02, -2.1144474153399617e-01, -3.0364507017646908e-01, -3.6189579198841304e-01,
                -3.9595279513831738e-01, -4.1488883666789378e-01, -4.2506887389996556e-01, -4.3041763772461122e-01,
                -4.3318415242787189e-01, -4.3459970965455186e-01
        },
// x = -0.8250
        {
                -8.2750796534970930e-02, -2.1185921094186644e-01, -3.0470333065273852e-01, -3.6341666212700796e-01,
                -3.9776607406889647e-01, -4.1687313067243309e-01, -4.2714827504065306e-01, -4.3254817646301857e-01,
                -4.3534156263125717e-01, -4.3677101950357944e-01
        },
// x = -0.8200
        {
                -8.2461967632557726e-02, -2.1227224418155580e-01, -3.0576365989227422e-01, -3.6494282874288425e-01,
                -3.9958697058705195e-01, -4.1886651010455983e-01, -4.2923762034906227e-01, -4.3468913704862078e-01,
                -4.3750965156223537e-01, -4.3895314307873057e-01
        },
// x = -0.8150
        {
                -8.2169948982519919e-02, -2.1268382530193328e-01, -3.0682605067435520e-01, -3.6647430216183968e-01,
                -4.0141551120099972e-01, -4.2086901311846087e-01, -4.3133695534766137e-01, -4.3684056931598297e-01,
                -4.3968847145453410e-01, -4.4114613390412410e-01
        },
// x = -0.8100
        {
                -8.1874741734842424e-02, -2.1309393835703874e-01, -3.0789049569851795e-01, -3.6801109267336934e-01,
                -4.0325172247049396e-01, -4.2288067800099349e-01, -4.3344632575002129e-01, -4.3900252332776035e-01,
                -4.4187807479162389e-01, -4.4335004576567072e-01
        },
// x = -0.8050
        {
                -8.1576347271810221e-02, -2.1350256740724322e-01, -3.0895698758458412e-01, -3.6955321053026624e-01,
                -4.0509563100664508e-01, -4.2490154317194145e-01, -4.3556577746147868e-01, -4.4117504937565499e-01,
                -4.4407851430785983e-01, -4.4556493271232550e-01
        },
// x = -0.8000
        {
                -8.1274767209128726e-02, -2.1390969652041122e-01, -3.1002551887269530e-01, -3.7110066594822361e-01,
                -4.0694726347173438e-01, -4.2693164718427251e-01, -4.3769535657980296e-01, -4.4335819798137655e-01,
                -4.4628984298963137e-01, -4.4779084905734512e-01
        },
// x = -0.7950
        {
                -8.0970003397024010e-02, -2.1431530977307245e-01, -3.1109608202335359e-01, -3.7265346910543584e-01,
                -4.0880664657902871e-01, -4.2897102872439363e-01, -4.3983510939586101e-01, -4.4555201989760579e-01,
                -4.4851211407651231e-01, -4.5002784937955032e-01
        },
// x = -0.7900
        {
                -8.0662057921316782e-02, -2.1471939125159581e-01, -3.1216866941746885e-01, -3.7421163014220105e-01,
                -4.1067380709259066e-01, -4.3101972661240434e-01, -4.4198508239428652e-01, -4.4775656610896231e-01,
                -4.5074538106242024e-01, -4.5227598852459527e-01
        },
// x = -0.7850
        {
                -8.0350933104471292e-02, -2.1512192505337077e-01, -3.1324327335640800e-01, -3.7577515916052351e-01,
                -4.1254877182708932e-01, -4.3307777980235229e-01, -4.4414532225414821e-01, -4.4997188783297443e-01,
                -4.5298969769677738e-01, -4.5453532160624166e-01
        },
// x = -0.7800
        {
                -8.0036631506619771e-02, -2.1552289528799248e-01, -3.1431988606205696e-01, -3.7734406622371452e-01,
                -4.1443156764760675e-01, -4.3514522738248324e-01, -4.4631587584962018e-01, -4.5219803652105278e-01,
                -4.5524511798567774e-01, -4.5680590400764021e-01
        },
// x = -0.7750
        {
                -7.9719155926560537e-02, -2.1592228607845346e-01, -3.1539849967688083e-01, -3.7891836135599749e-01,
                -4.1632222146944431e-01, -4.3722210857549521e-01, -4.4849679025065431e-01, -4.5443506385946947e-01,
                -4.5751169619305943e-01, -4.5908779138261568e-01
        },
// x = -0.7700
        {
                -7.9398509402731215e-02, -2.1632008156233848e-01, -3.1647910626399656e-01, -3.8049805454210883e-01,
                -4.1822076025792604e-01, -4.3930846273878682e-01, -4.5068811272365217e-01, -4.5668302177033593e-01,
                -4.5978948684188070e-01, -4.6138103965695942e-01
        },
// x = -0.7650
        {
                -7.9074695214153376e-02, -2.1671626589302626e-01, -3.1756169780724819e-01, -3.8208315572690255e-01,
                -4.2012721102819961e-01, -4.4140432936470858e-01, -4.5288989073213975e-01, -4.5894196241258822e-01,
                -4.6207854471530263e-01, -4.6368570502972833e-01
        },
// x = -0.7600
        {
                -7.8747716881354124e-02, -2.1711082324089298e-01, -3.1864626621128800e-01, -3.8367367481495424e-01,
                -4.2204160084503667e-01, -4.4350974808081089e-01, -4.5510217193744279e-01, -4.6121193818297501e-01,
                -4.6437892485787446e-01, -4.6600184397454725e-01
        },
// x = -0.7550
        {
                -7.8417578167256766e-02, -2.1750373779452409e-01, -3.1973280330166570e-01, -3.8526962167016415e-01,
                -4.2396395682263044e-01, -4.4562475865009138e-01, -4.5732500419936301e-01, -4.6349300171704655e-01,
                -4.6669068257672580e-01, -4.6832951324092015e-01
        },
// x = -0.7500
        {
                -7.8084283078047670e-02, -2.1789499376192689e-01, -3.2082130082492211e-01, -3.8687100611536385e-01,
                -4.2589430612439100e-01, -4.4774940097124205e-01, -4.5955843557685672e-01, -4.6578520589015071e-01,
                -4.6901387344276380e-01, -4.7066876985554584e-01
        },
// x = -0.7450
        {
                -7.7747835864012585e-02, -2.1828457537174981e-01, -3.2191175044868825e-01, -3.8847783793191926e-01,
                -4.2783267596273927e-01, -4.4988371507889413e-01, -4.6180251432871267e-01, -4.6808860381842865e-01,
                -4.7134855329187320e-01, -4.7301967112363891e-01
        },
// x = -0.7400
        {
                -7.7408241020347296e-02, -2.1867246687450406e-01, -3.2300414376179348e-01, -3.9009012685933758e-01,
                -4.2977909359889815e-01, -4.5202774114386379e-01, -4.6405728891423409e-01, -4.7040324885981688e-01,
                -4.7369477822612382e-01, -4.7538227463025895e-01
        },
// x = -0.7350
        {
                -7.7065503287938972e-02, -2.1905865254379178e-01, -3.2409847227437655e-01, -3.9170788259487443e-01,
                -4.3173358634268255e-01, -4.5418151947339408e-01, -4.6632280799391873e-01, -4.7272919461505070e-01,
                -4.7605260461498183e-01, -4.7775663824164361e-01
        },
// x = -0.7300
        {
                -7.6719627654118244e-02, -2.1944311667753438e-01, -3.2519472741800459e-01, -3.9333111479313815e-01,
                -4.3369618155228767e-01, -4.5634509051139821e-01, -4.6859912043014207e-01, -4.7506649492867314e-01,
                -4.7842208909652550e-01, -4.8014282010654891e-01
        },
// x = -0.7250
        {
                -7.6370619353383282e-02, -2.1982584359920784e-01, -3.2629290054579735e-01, -3.9495983306569982e-01,
                -4.3566690663407420e-01, -4.5851849483870183e-01, -4.7088627528784138e-01, -4.7741520389004410e-01,
                -4.8080328857866850e-01, -4.8254087865759565e-01
        },
// x = -0.7200
        {
                -7.6018483868095441e-02, -2.2020681765908018e-01, -3.2739298293255892e-01, -3.9659404698069906e-01,
                -4.3764578904235174e-01, -4.6070177317328109e-01, -4.7318432183520087e-01, -4.7977537583435703e-01,
                -4.8319626024038459e-01, -4.8495087261262054e-01
        },
// x = -0.7150
        {
                -7.5663226929143457e-02, -2.2058602323545229e-01, -3.2849496577491438e-01, -3.9823376606245614e-01,
                -4.3963285627916232e-01, -4.6289496637050426e-01, -4.7549330954433833e-01, -4.8214706534365531e-01,
                -4.8560106153294080e-01, -4.8737286097603638e-01
        },
// x = -0.7100
        {
                -7.5304854516579325e-02, -2.2096344473590177e-01, -3.2959884019145219e-01, -3.9987899979107688e-01,
                -4.4162813589405742e-01, -4.6509811542336776e-01, -4.7781328809199186e-01, -4.8453032724785394e-01,
                -4.8801775018113286e-01, -4.8980690304019492e-01
        },
// x = -0.7050
        {
                -7.4943372860222193e-02, -2.2133906659853125e-01, -3.3070459722287548e-01, -4.0152975760206699e-01,
                -4.4363165548388012e-01, -4.6731126146273433e-01, -4.8014430736020913e-01, -4.8692521662576543e-01,
                -4.9044638418452891e-01, -4.9225305838675854e-01
        },
// x = -0.7000
        {
                -7.4578788440235327e-02, -2.2171287329321748e-01, -3.3181222783215619e-01, -4.0318604888594017e-01,
                -4.4564344269253614e-01, -4.6953444575756953e-01, -4.8248641743703780e-01, -4.8933178880612643e-01,
                -4.9288702181871474e-01, -4.9471138688807687e-01
        },
// x = -0.6950
        {
                -7.4211107987668487e-02, -2.2208484932286543e-01, -3.3292172290469774e-01, -4.0484788298783214e-01,
                -4.4766352521077257e-01, -4.7176770971517384e-01, -4.8483966861721589e-01, -4.9175009936863046e-01,
                -4.9533972163654721e-01, -4.9718194870857030e-01
        },
// x = -0.6900
        {
                -7.3840338484971083e-02, -2.2245497922466348e-01, -3.3403307324850351e-01, -4.0651526920711073e-01,
                -4.4969193077594677e-01, -4.7401109488142007e-01, -4.8720411140286379e-01, -4.9418020414496361e-01,
                -4.9780454246941092e-01, -4.9966480430611937e-01
        },
// x = -0.6850
        {
                -7.3466487166474179e-02, -2.2282324757134148e-01, -3.3514626959435095e-01, -4.0818821679699208e-01,
                -4.5172868717179776e-01, -4.7626464294098259e-01, -4.8957979650417854e-01, -4.9662215921984110e-01,
                -5.0028154342848064e-01, -5.0216001443345981e-01
        },
// x = -0.6800
        {
                -7.3089561518842633e-02, -2.2318963897243230e-01, -3.3626130259597253e-01, -4.0986673496415216e-01,
                -4.5377382222821538e-01, -4.7852839571757044e-01, -4.9196677484012841e-01, -4.9907602093205217e-01,
                -5.0277078390599139e-01, -5.0466764013958676e-01
        },
// x = -0.6750
        {
                -7.2709569281489472e-02, -2.2355413807553323e-01, -3.3737816283024330e-01, -4.1155083286834426e-01,
                -4.5582736382100453e-01, -4.8080239517415690e-01, -4.9436509753914726e-01, -5.0154184587550332e-01,
                -5.0527232357650687e-01, -5.0718774277115997e-01
        },
// x = -0.6700
        {
                -7.2326518446965332e-02, -2.2391672956757117e-01, -3.3849684079737424e-01, -4.1324051962201408e-01,
                -4.5788933987165137e-01, -4.8308668341320843e-01, -4.9677481593983375e-01, -5.0401969090026877e-01,
                -5.0778622239820148e-01, -5.0972038397392216e-01
        },
// x = -0.6650
        {
                -7.1940417261312797e-02, -2.2427739817607023e-01, -3.3961732692111141e-01, -4.1493580428991711e-01,
                -4.5995977834708496e-01, -4.8538130267691304e-01, -4.9919598159164669e-01, -5.0650961311364284e-01,
                -5.1031254061414333e-01, -5.1226562569411738e-01
        },
// x = -0.6600
        {
                -7.1551274224385022e-02, -2.2463612867041824e-01, -3.4073961154894272e-01, -4.1663669588873681e-01,
                -4.6203870725943857e-01, -4.8768629534740593e-01, -5.0162864625560710e-01, -5.0901166988119473e-01,
                -5.1285133875358169e-01, -5.1482353017992077e-01
        },
// x = -0.6550
        {
                -7.1159098090136499e-02, -2.2499290586313792e-01, -3.4186368495231056e-01, -4.1834320338670389e-01,
                -4.6412615466580753e-01, -4.9000170394699549e-01, -5.0407286190499567e-01, -5.1152591882782972e-01,
                -5.1540267763324055e-01, -5.1739415998287164e-01
        },
// x = -0.6500
        {
                -7.0763897866878533e-02, -2.2534771461115727e-01, -3.4298953732683068e-01, -4.2005533570321651e-01,
                -4.6622214866800660e-01, -4.9232757113838849e-01, -5.0652868072605617e-01, -5.1405241783885292e-01,
                -5.1796661835862046e-01, -5.1997757795931399e-01
        },
// x = -0.6450
        {
                -7.0365682817499192e-02, -2.2570053981708238e-01, -3.4411715879251642e-01, -4.2177310170846216e-01,
                -4.6832671741232434e-01, -4.9466393972491091e-01, -5.0899615511869756e-01, -5.1659122506103217e-01,
                -5.2054322232530037e-01, -5.2257384727184453e-01
        },
// x = -0.6400
        {
                -6.9964462459655555e-02, -2.2605136643047080e-01, -3.4524653939401145e-01, -4.2349651022303925e-01,
                -4.7043988908927686e-01, -4.9701085265073108e-01, -5.1147533769719711e-01, -5.1914239890367175e-01,
                -5.2313255122025082e-01, -5.2518303139076516e-01
        },
// x = -0.6350
        {
                -6.9560246565924896e-02, -2.2640017944910512e-01, -3.4637766910082740e-01, -4.2522557001758238e-01,
                -4.7256169193335806e-01, -4.9936835300108023e-01, -5.1396628129090527e-01, -5.2170599803968487e-01,
                -5.2573466702314575e-01, -5.2780519409554472e-01
        },
// x = -0.6300
        {
                -6.9153045163928145e-02, -2.2674696392026888e-01, -3.4751053780758767e-01, -4.2696028981238549e-01,
                -4.7469215422279004e-01, -5.0173648400247206e-01, -5.1646903894495277e-01, -5.2428208140667043e-01,
                -5.2834963200768525e-01, -5.3044039947628374e-01
        },
// x = -0.6250
        {
                -6.8742868536414939e-02, -2.2709170494202094e-01, -3.4864513533427849e-01, -4.2870067827702935e-01,
                -4.7683130427926856e-01, -5.0411528902291824e-01, -5.1898366392095485e-01, -5.2687070820799276e-01,
                -5.3097750874292193e-01, -5.3308871193519036e-01
        },
// x = -0.6200
        {
                -6.8329727221316791e-02, -2.2743438766447308e-01, -3.4978145142650668e-01, -4.3044674403000910e-01,
                -4.7897917046771088e-01, -5.0650481157214944e-01, -5.2151020969772255e-01, -5.2947193791386948e-01,
                -5.3361836009459196e-01, -5.3575019618805841e-01
        },
// x = -0.6150
        {
                -6.7913632011763758e-02, -2.2777499729106279e-01, -3.5091947575576082e-01, -4.3219849563836105e-01,
                -4.8113578119599781e-01, -5.0890509530182693e-01, -5.2404872997197038e-01, -5.3208583026245482e-01,
                -5.3627224922645123e-01, -5.3842491726575414e-01
        },
// x = -0.6100
        {
                -6.7494593956067717e-02, -2.2811351907983290e-01, -3.5205919791968476e-01, -4.3395594161729550e-01,
                -4.8330116491471575e-01, -5.1131618400575873e-01, -5.2659927865902434e-01, -5.3471244526093253e-01,
                -5.3893923960161894e-01, -5.4111294051571224e-01
        },
// x = -0.6050
        {
                -6.7072624357670765e-02, -2.2844993834470501e-01, -3.5320060744234988e-01, -4.3571909042982720e-01,
                -4.8547535011689696e-01, -5.1373812162011179e-01, -5.2916190989353762e-01, -5.3735184318661255e-01,
                -5.4161939498392508e-01, -5.4381433160343307e-01
        },
// x = -0.6000
        {
                -6.6647734775056283e-02, -2.2878424045675527e-01, -3.5434369377454067e-01, -4.3748795048640615e-01,
                -4.8765836533775786e-01, -5.1617095222362364e-01, -5.3173667803020042e-01, -5.4000408458802696e-01,
                -5.4431277943926359e-01, -5.4652915651399070e-01
        },
// x = -0.5950
        {
                -6.6219937021628719e-02, -2.2911641084548998e-01, -3.5548844629404247e-01, -4.3926253014455585e-01,
                -4.8985023915443426e-01, -5.1861472003781428e-01, -5.3432363764445145e-01, -5.4266923028603253e-01,
                -5.4701945733695145e-01, -5.4925748155354714e-01
        },
// x = -0.5900
        {
                -6.5789243165551778e-02, -2.2944643500011980e-01, -3.5663485430593589e-01, -4.4104283770850544e-01,
                -4.9205100018571685e-01, -5.2106946942719135e-01, -5.3692284353319664e-01, -5.4534734137491736e-01,
                -5.4973949335109473e-01, -5.5199937335087146e-01
        },
// x = -0.5850
        {
                -6.5355665529557336e-02, -2.2977429847083175e-01, -3.5778290704290039e-01, -4.4282888142882826e-01,
                -4.9426067709178334e-01, -5.2353524489946079e-01, -5.3953435071552081e-01, -5.4803847922350890e-01,
                -5.5247295246195249e-01, -5.5475489885886942e-01
        },
// x = -0.5800
        {
                -6.4919216690714945e-02, -2.3009998687006336e-01, -3.5893259366552194e-01, -4.4462066950208157e-01,
                -4.9647929857393031e-01, -5.2601209110573410e-01, -5.4215821443340728e-01, -5.5074270547628690e-01,
                -5.5521989995732191e-01, -5.5752412535611551e-01
        },
// x = -0.5750
        {
                -6.4479909480164993e-02, -2.3042348587377193e-01, -3.6008390326260536e-01, -4.4641821007044535e-01,
                -4.9870689337429974e-01, -5.2850005284072932e-01, -5.4479449015245229e-01, -5.5346008205449981e-01,
                -5.5798040143391048e-01, -5.6030712044839504e-01
        },
// x = -0.5700
        {
                -6.4037756982818425e-02, -2.3074478122270642e-01, -3.6123682485149827e-01, -4.4822151122136544e-01,
                -5.0094349027560903e-01, -5.3099917504297733e-01, -5.4744323356258606e-01, -5.5619067115728504e-01,
                -5.6075452279872662e-01, -5.6310395207025177e-01
        },
// x = -0.5650
        {
                -6.3592772537017464e-02, -2.3106385872367213e-01, -3.6239134737841566e-01, -4.5003058098719489e-01,
                -5.0318911810087530e-01, -5.3350950279502340e-01, -5.5010450057879123e-01, -5.5893453526279091e-01,
                -5.6354233027047285e-01, -5.6591468848654392e-01
        },
// x = -0.5600
        {
                -6.3144969734161366e-02, -2.3138070425080073e-01, -3.6354745971877506e-01, -4.5184542734484295e-01,
                -5.0544380571313785e-01, -5.3603108132362964e-01, -5.5277834734182352e-01, -5.6169173712930576e-01,
                -5.6634389038094124e-01, -5.6873939829400422e-01
        },
// x = -0.5550
        {
                -6.2694362418295541e-02, -2.3169530374681144e-01, -3.6470515067753490e-01, -4.5366605821541789e-01,
                -5.0770758201518273e-01, -5.3856395599996987e-01, -5.5546483021893345e-01, -5.6446233979638782e-01,
                -5.6915926997641897e-01, -5.7157815042281168e-01
        },
// x = -0.5500
        {
                -6.2240964685663001e-02, -2.3200764322427520e-01, -3.6586440898954214e-01, -4.5549248146387794e-01,
                -5.0998047594925922e-01, -5.4110817233983355e-01, -5.5816400580458880e-01, -5.6724640658599801e-01,
                -5.7198853621909451e-01, -5.7443101413816400e-01
        },
// x = -0.5450
        {
                -6.1784790884222923e-02, -2.3231770876687294e-01, -3.6702522331988258e-01, -4.5732470489868154e-01,
                -5.1226251649680221e-01, -5.4366377600381732e-01, -5.6087593092119803e-01, -5.7004400110363995e-01,
                -5.7483175658847729e-01, -5.7729805904186371e-01
        },
// x = -0.5400
        {
                -6.1325855613124786e-02, -2.3262548653065421e-01, -3.6818758226424231e-01, -4.5916273627143794e-01,

                -5.1455373267814641e-01, -5.4623081279752272e-01, -5.6360066261983532e-01, -5.7285518723950102e-01,
                -5.7768899888281311e-01, -5.8017935507390694e-01
        },
// x = -0.5350
        {
                -6.0864173722154670e-02, -2.3293096274529068e-01, -3.6935147434927035e-01, -4.6100658327656296e-01,
                -5.1685415355224118e-01, -5.4880932867175169e-01, -5.6633825818096684e-01, -5.7568002916959726e-01,
                -5.8056033122051420e-01, -5.8307497251408036e-01
        },
// x = -0.5300
        {
                -6.0399760311137293e-02, -2.3323412371532842e-01, -3.7051688803295030e-01, -4.6285625355093279e-01,
                -5.1916380821636510e-01, -5.5139936972269410e-01, -5.6908877511517364e-01, -5.7851859135692085e-01,
                -5.8344582204158912e-01, -5.8598498198356730e-01
        },
// x = -0.5250
        {
                -5.9932630729306061e-02, -2.3353495582143718e-01, -3.7168381170497955e-01, -4.6471175467354392e-01,
                -5.2148272580583499e-01, -5.5400098219212346e-01, -5.7185227116388404e-01, -5.8137093855259392e-01,
                -5.8634554010908213e-01, -5.8890945444655673e-01
        },
// x = -0.5200
        {
                -5.9462800574631040e-02, -2.3383344552165308e-01, -3.7285223368715092e-01, -4.6657309416517040e-01,
                -5.2381093549371927e-01, -5.5661421246758214e-01, -5.7462880430009811e-01, -5.8423713579702641e-01,
                -5.8925955451051459e-01, -5.9184846121186385e-01
        },
// x = -0.5150
        {
                -5.8990285693117719e-02, -2.3412957935262502e-01, -3.7402214223374342e-01, -4.6844027948802724e-01,
                -5.2614846649054237e-01, -5.5923910708257429e-01, -5.7741843272911841e-01, -5.8711724842107160e-01,
                -5.9218793465933639e-01, -5.9480207393455697e-01
        },
// x = -0.5100
        {
                -5.8515102178056250e-02, -2.3442334393084821e-01, -3.7519352553191843e-01, -4.7031331804543158e-01,
                -5.2849534804399489e-01, -5.6187571271674719e-01, -5.8022121488928158e-01, -5.9001134204719119e-01,
                -5.9513075029637985e-01, -5.9777036461758704e-01
        },
// x = -0.5050
        {
                -5.8037266369245132e-02, -2.3471472595390203e-01, -3.7636637170212228e-01, -4.7219221718147070e-01,
                -5.3085160943863230e-01, -5.6452407619608114e-01, -5.8303720945268755e-01, -5.9291948259062388e-01,
                -5.9808807149132182e-01, -6.0075340561343316e-01
        },
// x = -0.5000
        {
                -5.7556794852177302e-02, -2.3500371220167976e-01, -3.7754066879849429e-01, -4.7407698418066735e-01,
                -5.3321727999558499e-01, -5.6718424449306992e-01, -5.8586647532593128e-01, -5.9584173626055581e-01,
                -6.0105996864414835e-01, -6.0375126962574588e-01
        },
// x = -0.4950
        {
                -5.7073704457173816e-02, -2.3529028953761394e-01, -3.7871640480928170e-01, -4.7596762626765055e-01,
                -5.3559238907225637e-01, -5.6985626472690221e-01, -5.8870907165083886e-01, -5.9877816956129082e-01,
                -6.0404651248662966e-01, -6.0676402971100596e-01
        },
// x = -0.4900
        {
                -5.6588012258498535e-02, -2.3557444490989890e-01, -3.7989356765726012e-01, -4.7786415060682630e-01,
                -5.3797696606201928e-01, -5.7254018416364605e-01, -5.9156505780519775e-01, -6.0172884929343451e-01,
                -6.0704777408379862e-01, -6.0979175928018603e-01
        },
// x = -0.4850
        {
                -5.6099735573428074e-02, -2.3585616535270829e-01, -3.8107214520016036e-01, -4.7976656430205217e-01,
                -5.4037104039392192e-01, -5.7523605021642477e-01, -5.9443449340349441e-01, -6.0469384255507019e-01,
                -6.1006382483543098e-01, -6.1283453210041750e-01
        },
// x = -0.4800
        {
                -5.5608891961279897e-02, -2.3613543798740788e-01, -3.8225212523110225e-01, -4.8167487439631190e-01,
                -5.4277464153237598e-01, -5.7794391044559490e-01, -5.9731743829764861e-01, -6.0767321674295005e-01,
                -6.1309473647754353e-01, -6.1589242229667296e-01
        },
// x = -0.4750
        {
                -5.5115499222414134e-02, -2.3641225002376332e-01, -3.8343349547903288e-01, -4.8358908787139565e-01,
                -5.4518779897685710e-01, -5.8066381255892419e-01, -6.0021395257775045e-01, -6.1066703955367785e-01,
                -6.1614058108388281e-01, -6.1896550435344866e-01
        },
// x = -0.4700
        {
                -5.4619575397184567e-02, -2.3668658876114163e-01, -3.8461624360917068e-01, -4.8550921164757721e-01,
                -5.4761054226159600e-01, -5.8339580441176331e-01, -6.0312409657279631e-01, -6.1367537898490643e-01,
                -6.1920143106743197e-01, -6.2205385311645622e-01
        },
// x = -0.4650
        {
                -5.4121138764862041e-02, -2.3695844158971033e-01, -3.8580035722345796e-01, -4.8743525258329817e-01,
                -5.5004290095526942e-01, -5.8613993400722042e-01, -6.0604793085143149e-01, -6.1669830333653242e-01,
                -6.2227735918191673e-01, -6.2515754379432320e-01
        },
// x = -0.4600
        {
                -5.3620207842520480e-02, -2.3722779599162722e-01, -3.8698582386101826e-01, -4.8936721747485451e-01,
                -5.5248490466069333e-01, -5.8889624949633412e-01, -6.0898551622268415e-01, -6.1973588121189593e-01,
                -6.2536843852332114e-01, -6.2827665196030169e-01
        },
// x = -0.4550
        {
                -5.3116801383878282e-02, -2.3749463954222860e-01, -3.8817263099861710e-01, -4.9130511305607932e-01,
                -5.5493658301450510e-01, -5.9166479917824077e-01, -6.1193691373670889e-01, -6.2278818151898474e-01,
                -6.2847474253140589e-01, -6.3141125355398242e-01
        },
// x = -0.4500
        {
                -5.2610938378110622e-02, -2.3775895991120771e-01, -3.8936076605113296e-01, -4.9324894599803615e-01,
                -5.5739796568685929e-01, -5.9444563150034291e-01, -6.1490218468552627e-01, -6.2585527347164216e-01,
                -6.3159634499123729e-01, -6.3456142488301692e-01
        },
// x = -0.4450
        {
                -5.2102638048620274e-02, -2.3802074486379043e-01, -3.9055021637203030e-01, -4.9519872290870848e-01,
                -5.5986908238110444e-01, -5.9723879505847788e-01, -6.1788139060376535e-01, -6.2893722659077667e-01,
                -6.3473332003471727e-01, -6.3772724262484937e-01
        },
// x = -0.4400
        {
                -5.1591919851770952e-02, -2.3827998226190231e-01, -3.9174096925384150e-01, -4.9715445033269562e-01,
                -5.6234996283347305e-01, -6.0004433859708328e-01, -6.2087459326940486e-01, -6.3203411070557958e-01,
                -6.3788574214212146e-01, -6.4090878382845240e-01
        },
// x = -0.4350
        {
                -5.1078803475587090e-02, -2.3853666006533011e-01, -3.9293301192865310e-01, -4.9911613475090605e-01,
                -5.6484063681275898e-01, -6.0286231100935683e-01, -6.2388185470451807e-01, -6.3514599595474031e-01,
                -6.4105368614364677e-01, -6.4410612591607586e-01
        },
// x = -0.4300
        {
                -5.0563308838415817e-02, -2.3879076633287707e-01, -3.9412633156859678e-01, -5.0108378258026065e-01,
                -5.6734113412000486e-01, -6.0569276133742267e-01, -6.2690323717601693e-01, -6.3827295278767238e-01,
                -6.4423722722095833e-01, -6.4731934668499669e-01
        },
// x = -0.4250
        {
                -5.0045456087546875e-02, -2.3904228922351228e-01, -3.9532091528634872e-01, -5.0305740017339240e-01,
                -5.6985148458817292e-01, -6.0853573877249056e-01, -6.2993880319639683e-01, -6.4141505196573612e-01,
                -6.4743644090874941e-01, -6.5054852430928367e-01
        },
// x = -0.4200
        {
                -4.9525265597803264e-02, -2.3929121699750863e-01, -3.9651675013563187e-01, -5.0503699381834732e-01,
                -5.7237171808182941e-01, -6.1139129265501313e-01, -6.3298861552448071e-01, -6.4457236456347011e-01,
                -6.5065140309629921e-01, -6.5379373734156521e-01
        },
// x = -0.4150
        {
                -4.9002757970094499e-02, -2.3953753801758115e-01, -3.9771382311172832e-01, -5.0702256973829740e-01,
                -5.7490186449681890e-01, -6.1425947247484380e-01, -6.3605273716616839e-01, -6.4774496196982412e-01,
                -6.5388219002904846e-01, -6.5705506471480346e-01
        },
// x = -0.4100
        {
                -4.8477954029925129e-02, -2.3978124075000987e-01, -3.9891212115198998e-01, -5.0901413409124208e-01,
                -5.7744195375993834e-01, -6.1714032787139417e-01, -6.3913123137518280e-01, -6.5093291588939739e-01,
                -6.5712887831016942e-01, -6.6033258574408360e-01
        },
// x = -0.4050
        {
                -4.7950874825877343e-02, -2.4002231376576177e-01, -4.0011163113636150e-01, -5.1101169296972693e-01,
                -5.7999201582861182e-01, -6.2003390863378505e-01, -6.4222416165381813e-01, -6.5413629834367915e-01,
                -6.6039154490215046e-01, -6.6362638012840391e-01
        },
// x = -0.4000
        {
                -4.7421541628049214e-02, -2.4026074574160022e-01, -4.0131233988790849e-01, -5.1301525240055501e-01,
                -5.8255208069056086e-01, -6.2294026470100117e-01, -6.4533159175368804e-01, -6.5735518167229179e-01,
                -6.6367026712837995e-01, -6.6693652795247649e-01
        },
// x = -0.3950
        {
                -4.6889975926467112e-02, -2.4049652546119035e-01, -4.0251423417334475e-01, -5.1502481834449998e-01,
                -5.8512217836347691e-01, -6.2585944616204103e-01, -6.4845358567647660e-01, -6.6058963853424135e-01,
                -6.6696512267474539e-01, -6.7026310968853564e-01
        },
// x = -0.3900
        {
                -4.6356199429451142e-02, -2.4072964181619408e-01, -4.0371730070357376e-01, -5.1704039669603330e-01,
                -5.8770233889468937e-01, -6.2879150325606603e-01, -6.5159020767468923e-01, -6.6383974190916906e-01,
                -6.7027618959122759e-01, -6.7360620619815492e-01
        },
// x = -0.3850
        {
                -4.5820234061944409e-02, -2.4096008380735578e-01, -4.0492152613422794e-01, -5.1906199328304070e-01,
                -5.9029259236083398e-01, -6.3173648637254620e-01, -6.5474152225240112e-01, -6.6710556509860808e-01,
                -6.7360354629351094e-01, -6.7696589873407154e-01
        },
// x = -0.3800
        {
                -4.5282101963823665e-02, -2.4118784054558401e-01, -4.0612689706621985e-01, -5.2108961386654806e-01,
                -5.9289296886751863e-01, -6.3469444605140768e-01, -6.5790759416601119e-01, -6.7038718172724010e-01,
                -6.7694727156459533e-01, -6.8034226894201888e-01
        },
// x = -0.3750
        {
                -4.4741825488154988e-02, -2.4141290125301987e-01, -4.0733340004629348e-01, -5.2312326414045296e-01,
                -5.9550349854898788e-01, -6.3766543298317846e-01, -6.6108848842499635e-01, -6.7368466574416352e-01,
                -6.8030744455641312e-01, -6.8373539886256618e-01
        },
// x = -0.3700
        {
                -4.4199427199420556e-02, -2.4163525526409896e-01, -4.0854102156758393e-01, -5.2516294973125011e-01,
                -5.9812421156778928e-01, -6.4064949800912618e-01, -6.6428427029265957e-01, -6.7699809142415757e-01,
                -6.8368414479145811e-01, -6.8714537093296835e-01
        },
// x = -0.3650
        {
                -4.3654929871721024e-02, -2.4185489202660415e-01, -4.0974974807018066e-01, -5.2720867619776812e-01,
                -6.0075513811443171e-01, -6.4364669212140191e-01, -6.6749500528688976e-01, -6.8032753336895491e-01,
                -6.8707745216441252e-01, -6.9057226798902371e-01
        },
// x = -0.3600
        {
                -4.3108356486920650e-02, -2.4207180110270982e-01, -4.1095956594169963e-01, -5.2926044903090363e-01,
                -6.0339630840704928e-01, -6.4665706646317966e-01, -6.7072075918091512e-01, -6.8367306650851345e-01,
                -6.9048744694378872e-01, -6.9401617326693699e-01
        },
// x = -0.3550
        {
                -4.2559730232781583e-02, -2.4228597217001338e-01, -4.1217046151785397e-01, -5.3131827365336270e-01,
                -6.0604775269106059e-01, -6.4968067232879101e-01, -6.7396159800405442e-01, -6.8703476610229886e-01,
                -6.9391420977357043e-01, -6.9747717040519253e-01
        },
// x = -0.3500
        {
                -4.2009074501052159e-02, -2.4249739502256382e-01, -4.1338242108303597e-01, -5.3338215541940059e-01,
                -6.0870950123882650e-01, -6.5271756116386259e-01, -6.7721758804247811e-01, -6.9041270774056662e-01,
                -6.9735782167486793e-01, -7.0095534344643862e-01
        },
// x = -0.3450
        {
                -4.1456412885514243e-02, -2.4270605957187288e-01, -4.1459543087090134e-01, -5.3545209961456708e-01,
                -6.1138158434930823e-01, -6.5576778456545004e-01, -6.8048879583996313e-01, -6.9380696734564273e-01,
                -7.0081836404757047e-01, -7.0445077683937363e-01
        },
// x = -0.3400
        {
                -4.0901769180019303e-02, -2.4291195584792463e-01, -4.1580947706495935e-01, -5.3752811145545676e-01,
                -6.1406403234772022e-01, -6.5883139428216853e-01, -6.8377528819865052e-01, -6.9721762117322161e-01,
                -7.0429591867201158e-01, -7.0796355544064715e-01
        },
// x = -0.3350
        {
                -4.0345167376466824e-02, -2.4311507400016599e-01, -4.1702454579916826e-01, -5.3961019608945460e-01,
                -6.1675687558518910e-01, -6.6190844221432532e-01, -6.8707713217980326e-01, -7.0064474581365233e-01,
                -7.0779056771063986e-01, -7.1149376451676383e-01
        },
// x = -0.3300
        {
                -3.9786631662749768e-02, -2.4331540429849455e-01, -4.1824062315853155e-01, -5.4169835859449356e-01,
                -6.1946014443840380e-01, -6.6499898041404604e-01, -6.9039439510456591e-01, -7.0408841819324508e-01,
                -7.1130239370969506e-01, -7.1504148974600013e-01
        },
// x = -0.3250
        {
                -3.9226186420701088e-02, -2.4351293713423541e-01, -4.1945769517970849e-01, -5.4379260397881257e-01,
                -6.2217386930927132e-01, -6.6810306108540474e-01, -6.9372714455472273e-01, -7.0754871557556098e-01,
                -7.1483147960089199e-01, -7.1860681722032715e-01
        },
// x = -0.3200
        {
                -3.8663856223958876e-02, -2.4370766302110039e-01, -4.2067574785161954e-01, -5.4589293718071241e-01,
                -6.2489808062455965e-01, -6.7122073658454628e-01, -6.9707544837345847e-01, -7.1102571556272642e-01,
                -7.1837790870310780e-01, -7.2218983344734067e-01
        },
// x = -0.3150
        {
                -3.8099665835831235e-02, -2.4389957259614584e-01, -4.2189476711606017e-01, -5.4799936306831887e-01,
                -6.2763280883555861e-01, -6.7435205941980980e-01, -7.0043937466612038e-01, -7.1451949609673848e-01,
                -7.2194176472408000e-01, -7.2579062535220229e-01
        },
// x = -0.3100
        {
                -3.7533640207127753e-02, -2.4408865662071405e-01, -4.2311473886832357e-01, -5.5011188643935527e-01,
                -6.3037808441772070e-01, -6.7749708225185346e-01, -7.0381899180097740e-01, -7.1803013546077832e-01,
                -7.2552313176210759e-01, -7.2940928027958929e-01
        },
// x = -0.3050
        {
                -3.6965804473939726e-02, -2.4427490598136581e-01, -4.2433564895782089e-01, -5.5223051202090134e-01,
                -6.3313393787030792e-01, -6.8065585789377070e-01, -7.0721436840998142e-01, -7.2155771228052479e-01,
                -7.2912209430776254e-01, -7.3304588599564879e-01
        },
// x = -0.3000
        {
                -3.6396183955404005e-02, -2.4445831169080129e-01, -4.2555748318871151e-01, -5.5435524446917306e-01,
                -6.3590039971604573e-01, -6.8382843931121118e-01, -7.1062557338953336e-01, -7.2510230552548149e-01,
                -7.3273873724560357e-01, -7.3670053068996622e-01
        },
// x = -0.2950
        {
                -3.5824804151435109e-02, -2.4463886488876763e-01, -4.2678022732053583e-01, -5.5648608836929225e-01,
                -6.3867750050075756e-01, -6.8701487962249541e-01, -7.1405267590124233e-01, -7.2866399451029351e-01,
                -7.3637314585589864e-01, -7.4037330297753801e-01
        },
// x = -0.2900
        {
                -3.5251690740424071e-02, -2.4481655684296089e-01, -4.2800386706885307e-01, -5.5862304823507536e-01,
                -6.4146527079301952e-01, -6.9021523209873159e-01, -7.1749574537269289e-01, -7.3224285889607954e-01,
                -7.4002540581635634e-01, -7.4406429190075407e-01
        },
// x = -0.2850
        {
                -3.4676869576895746e-02, -2.4499137894990836e-01, -4.2922838810588049e-01, -5.6076612850879959e-01,
                -6.4426374118379315e-01, -6.9342955016392327e-01, -7.2095485149820415e-01, -7.3583897869176307e-01,
                -7.4369560320385775e-01, -7.4777358693139173e-01
        },
// x = -0.2800
        {
                -3.4100366689162151e-02, -2.4516332273584657e-01, -4.3045377606114360e-01, -5.6291533356100087e-01,
                -6.4707294228607615e-01, -6.9665788739508838e-01, -7.2443006423960099e-01, -7.3945243425540785e-01,
                -7.4738382449620244e-01, -7.5150127797261335e-01
        },
// x = -0.2750
        {
                -3.3522208276914663e-02, -2.4533237985758288e-01, -4.3168001652212490e-01, -5.6507066769025627e-01,
                -6.4989290473453654e-01, -6.9990029752236071e-01, -7.2792145382697437e-01, -7.4308330629555530e-01,
                -7.5109015657385414e-01, -7.5524745536097493e-01
        },
// x = -0.2700
        {
                -3.2942420708810434e-02, -2.4549854210334690e-01, -4.3290709503491925e-01, -5.6723213512297788e-01,
                -6.5272365918515540e-01, -7.0315683442910337e-01, -7.3142909075944951e-01, -7.4673167587257216e-01,
                -7.5481468672170071e-01, -7.5901220986844753e-01
        },
// x = -0.2650
        {
                -3.2361030520019117e-02, -2.4566180139362775e-01, -4.3413499710489245e-01, -5.6939974001320404e-01,
                -6.5556523631486763e-01, -7.0642755215201036e-01, -7.3495304580595189e-01, -7.5039762439999258e-01,
                -7.5855750263081190e-01, -7.6279563270443929e-01
        },
// x = -0.2600
        {
                -3.1778064409729939e-02, -2.4582214978200195e-01, -4.3536370819734516e-01, -5.7157348644240002e-01,
                -6.5841766682119096e-01, -7.0971250488121307e-01, -7.3849339000597258e-01, -7.5408123364587376e-01,
                -7.6231869240021277e-01, -7.6659781551783412e-01
        },
// x = -0.2550
        {
                -3.1193549238672416e-02, -2.4597957945594798e-01, -4.3659321373818150e-01, -5.7375337841926066e-01,
                -6.6128098142187564e-01, -7.1301174696037994e-01, -7.4205019467033950e-01, -7.5778258573414581e-01,
                -7.6609834453865577e-01, -7.7041885039903546e-01
        },
// x = -0.2500
        {
                -3.0607512026537727e-02, -2.4613408273764487e-01, -4.3782349911457696e-01, -5.7593941987951069e-01,
                -6.6415521085452767e-01, -7.1632533288681854e-01, -7.4562353138198112e-01, -7.6150176314597540e-01,
                -7.6989654796640827e-01, -7.7425882988201478e-01
        },
// x = -0.2450
        {
                -3.0019979949458389e-02, -2.4628565208476341e-01, -4.3905454967565921e-01, -5.7813161468571672e-01,
                -6.6704038587625192e-01, -7.1965331731157323e-01, -7.4921347199669819e-01, -7.6523884872112358e-01,
                -7.7371339201703593e-01, -7.7811784694638186e-01
        },
// x = -0.2400
        {
                -2.9430980337374551e-02, -2.4643428009123952e-01, -4.4028635073318295e-01, -5.8032996662709757e-01,
                -6.6993653726328217e-01, -7.2299575503952340e-01, -7.5282008864392802e-01, -7.6899392565931524e-01,
                -7.7754896643920590e-01, -7.8199599501944905e-01
        },
// x = -0.2350
        {
                -2.8840540671449210e-02, -2.4657995948803904e-01, -4.4151888756221491e-01, -5.8253447941933745e-01,
                -6.7284369581061299e-01, -7.2635270102947347e-01, -7.5644345372752197e-01, -7.7276707752160956e-01,
                -7.8140336139848365e-01, -7.8589336797831411e-01
        },
// x = -0.2300
        {
                -2.8248688581390336e-02, -2.4672268314390586e-01, -4.4275214540182112e-01, -5.8474515670440663e-01,
                -6.7576189233163475e-01, -7.2972421039425195e-01, -7.6008363992650663e-01, -7.7655838823176893e-01,
                -7.8527666747914904e-01, -7.8981006015194721e-01
        },
// x = -0.2250
        {
                -2.7655451842803221e-02, -2.4686244406609748e-01, -4.4398610945575767e-01, -5.8696200205038185e-01,
                -6.7869115765776400e-01, -7.3311033840079798e-01, -7.6374072019586015e-01, -7.8036794207764404e-01,
                -7.8916897568600863e-01, -7.9374616632329320e-01
        },
// x = -0.2200
        {
                -2.7060858374488794e-02, -2.4699923540110966e-01, -4.4522076489316281e-01, -5.8918501895127251e-01,
                -6.8163152263806936e-01, -7.3651114047025346e-01, -7.6741476776727879e-01, -7.8419582371255037e-01,
                -7.9308037744622173e-01, -7.9770178173137407e-01
        },
// x = -0.2150
        {
                -2.6464936235715036e-02, -2.4713305043538392e-01, -4.4645609684925958e-01, -5.9141421082684786e-01,
                -6.8458301813890554e-01, -7.3992667217804631e-01, -7.7110585614995220e-01, -7.8804211815665481e-01,
                -7.9701096461113030e-01, -8.0167700207340742e-01
        },
// x = -0.2100
        {
                -2.5867713623467944e-02, -2.4726388259600446e-01, -4.4769209042605340e-01, -5.9364958102246890e-01,
                -6.8754567504354269e-01, -7.4335698925398208e-01, -7.7481405913133150e-01, -7.9190691079836661e-01,
                -8.0096082945809821e-01, -8.0567192350693284e-01
        },
// x = -0.2050
        {
                -2.5269218869685708e-02, -2.4739172545137539e-01, -4.4892873069303657e-01, -5.9589113280892758e-01,
                -6.9051952425178720e-01, -7.4680214758232177e-01, -7.7853945077790532e-01, -7.9579028739572866e-01,
                -8.0493006469235384e-01, -8.0968664265194468e-01
        },
// x = -0.2000
        {
                -2.4669480438446353e-02, -2.4751657271189803e-01, -4.5016600268790141e-01, -5.9813886938228189e-01,
                -6.9350459667961495e-01, -7.5026220320186643e-01, -7.8228210543596677e-01, -7.9969233407781726e-01,
                -8.0891876344884417e-01, -8.1372125659303751e-01
        },
// x = -0.1950
        {
                -2.4068526923188265e-02, -2.4763841823061222e-01, -4.5140389141724491e-01, -6.0039279386369993e-01,
                -6.9650092325879520e-01, -7.5373721230603663e-01, -7.8604209773239142e-01, -8.0361313734613815e-01,
                -8.1292701929408984e-01, -8.1777586288155835e-01
        },
// x = -0.1900
        {
                -2.3466387043828058e-02, -2.4775725600384946e-01, -4.5264238185729050e-01, -6.0265290929930693e-01,
                -6.9950853493651410e-01, -7.5722723124295210e-01, -7.8981950257540767e-01, -8.0755278407603714e-01,
                -8.1695492622805499e-01, -8.2185055953777020e-01
        },
// x = -0.1850
        {
                -2.2863089643911785e-02, -2.4787308017184390e-01, -4.5388145895460114e-01, -6.0491921866003673e-01,
                -7.0252746267500232e-01, -7.6073231651550421e-01, -7.9361439515537280e-01, -8.1151136151810577e-01,
                -8.2100257868601767e-01, -8.2594544505302259e-01
        },
// x = -0.1800
        {
                -2.2258663687712076e-02, -2.4798588501935523e-01, -4.5512110762680047e-01, -6.0719172484148076e-01,
                -7.0555773745115369e-01, -7.6425252478143191e-01, -7.9742685094554389e-01, -8.1548895729959403e-01,
                -8.2507007154044887e-01, -8.3006061839193479e-01
        },
// x = -0.1750
        {
                -2.1653138257360684e-02, -2.4809566497625796e-01, -4.5636131276330166e-01, -6.0947043066374884e-01,
                -7.0859939025615626e-01, -7.6778791285339665e-01, -8.0125694570285644e-01, -8.1948565942582552e-01,
                -8.2915750010290157e-01, -8.3419617899458376e-01
        },
// x = -0.1700
        {
                -2.1046542549863340e-02, -2.4820241461812642e-01, -4.5760205922603003e-01, -6.1175533887132605e-01,
                -7.1165245209510530e-01, -7.7133853769905047e-01, -8.0510475546869265e-01, -8.2350155628162214e-01,
                -8.3326496012590234e-01, -8.3835222677870813e-01
        },
// x = -0.1650
        {
                -2.0438905874222112e-02, -2.4830612866680360e-01, -4.5884333185015380e-01, -6.1404645213294529e-01,
                -7.1471695398663171e-01, -7.7490445644110273e-01, -8.0897035656966632e-01, -8.2753673663272276e-01,
                -8.3739254780485672e-01, -8.4252886214191236e-01
        },
// x = -0.1600
        {
                -1.9830257648377699e-02, -2.4840680199094622e-01, -4.6008511544481595e-01, -6.1634377304144750e-01,
                -7.1779292696251817e-01, -7.7848572635739099e-01, -8.1285382561838770e-01, -8.3159128962721629e-01,
                -8.4154035977994901e-01, -8.4672618596389293e-01
        },
// x = -0.1550
        {
                -1.9220627396288728e-02, -2.4850442960657171e-01, -4.6132739479387480e-01, -6.1864730411365310e-01,
                -7.2088040206732440e-01, -7.8208240488094327e-01, -8.1675523951424467e-01, -8.3566530479697099e-01,
                -8.4570849313806729e-01, -8.5094429960866036e-01
        },
// x = -0.1500
        {
                -1.8610044744903645e-02, -2.4859900667757001e-01, -4.6257015465663370e-01, -6.2095704779024374e-01,
                -7.2397941035800173e-01, -7.8569454960004115e-01, -8.2067467544418193e-01, -8.3975887205906785e-01,
                -8.4989704541471933e-01, -8.5518330492678141e-01
        },
// x = -0.1450
        {
                -1.7998539421117055e-02, -2.4869052851622778e-01, -4.6381337976858822e-01, -6.2327300643563721e-01,
                -7.2708998290351323e-01, -7.8932221825828175e-01, -8.2461221088346981e-01, -8.4387208171724726e-01,
                -8.5410611459596497e-01, -8.5944330425762439e-01
        },
// x = -0.1400
        {
                -1.7386141248761918e-02, -2.4877899058369615e-01, -4.6505705484216908e-01, -6.2559518233786859e-01,
                -7.3021215078445711e-01, -7.9296546875463614e-01, -8.2856792359648768e-01, -8.4800502446334380e-01,
                -8.5833579912035107e-01, -8.6372440043162213e-01
        },
// x = -0.1350
        {
                -1.6772880145500903e-02, -2.4886438849049308e-01, -4.6630116456748283e-01, -6.2792357770848028e-01,

                -7.3334594509267770e-01, -7.9662435914350549e-01, -8.3254189163749603e-01, -8.5215779137874303e-01,
                -8.6258619788085966e-01, -8.6802669677253097e-01
        },
// x = -0.1300
        {
                -1.6158786119776808e-02, -2.4894671799694293e-01, -4.6754569361306564e-01, -6.3025819468241140e-01,
                -7.3649139693088606e-01, -8.0029894763477938e-01, -8.3653419335141921e-01, -8.5633047393582562e-01,
                -8.6685741022685425e-01, -8.7235029709971534e-01
        },
// x = -0.1250
        {
                -1.5543889267713650e-02, -2.4902597501364110e-01, -4.6879062662662813e-01, -6.3259903531789252e-01,
                -7.3964853741227643e-01, -8.0398929259388752e-01, -8.4054490737461651e-01, -8.6052316399942508e-01,
                -8.7114953596604150e-01, -8.7669530573043086e-01
        },
// x = -0.1200
        {
                -1.4928219770056077e-02, -2.4910215560187354e-01, -4.7003594823581279e-01, -6.3494610159633980e-01,
                -7.4281739766014188e-01, -8.0769545254184716e-01, -8.4457411263566529e-01, -8.6473595382828672e-01,
                -8.7546267536643685e-01, -8.8106182748212014e-01
        },
// x = -0.1150
        {
                -1.4311807888851091e-02, -2.4917525597403181e-01, -4.7128164304894565e-01, -6.3729939542225866e-01,
                -7.4599800880749145e-01, -8.1141748615532161e-01, -8.4862188835613694e-01, -8.6896893607653269e-01,
                -8.7979692915833629e-01, -8.8544996767472350e-01
        },
// x = -0.1100
        {
                -1.3694683964607036e-02, -2.4924527249402043e-01, -4.7252769565579195e-01, -6.3965891862315394e-01,
                -7.4919040199666120e-01, -8.1515545226665909e-01, -8.5268831405137469e-01, -8.7322220379512372e-01,
                -8.8415239853629857e-01, -8.8985983213298914e-01
        },
// x = -0.1050
        {
                -1.3076878412869137e-02, -2.4931220167762674e-01, -4.7377409062831666e-01, -6.4202467294942822e-01,
                -7.5239460837893690e-01, -8.1890940986394445e-01, -8.5677346953127276e-01, -8.7749585043333844e-01,
                -8.8852918516113255e-01, -8.9429152718880645e-01
        },
// x = -0.1000
        {
                -1.2458421721225505e-02, -2.4937604019291232e-01, -4.7502081252144607e-01, -6.4439666007430640e-01,
                -7.5561065911415803e-01, -8.2267941809103762e-01, -8.6087743490105617e-01, -8.8178996984023816e-01,
                -8.9292739116189201e-01, -8.9874515968353597e-01
        },
// x = -0.0950
        {
                -1.1839344446063479e-02, -2.4943678486053178e-01, -4.7626784587382542e-01, -6.4677488159374241e-01,
                -7.5883858537034132e-01, -8.2646553624762154e-01, -8.6500029056205752e-01, -8.8610465626615331e-01,
                -8.9734711913787868e-01, -9.0322083697035882e-01
        },
// x = -0.0900
        {
                -1.1219677209303568e-02, -2.4949443265409932e-01, -4.7751517520858650e-01, -6.4915933902635048e-01,
                -7.6207841832329004e-01, -8.3026782378923614e-01, -8.6914211721249957e-01, -8.9044000436415971e-01,
                -9.0178847216065239e-01, -9.0771866691663405e-01
        },
// x = -0.0850
        {
                -1.0599450695238931e-02, -2.4954898070049195e-01, -4.7876278503411307e-01, -6.5155003381331433e-01,
                -7.6533018915620254e-01, -8.3408634032732143e-01, -8.7330299584827031e-01, -8.9479610919157204e-01,
                -9.0625155377604893e-01, -9.1223875790626174e-01
        },
// x = -0.0800
        {
                -9.9786956475741376e-03, -2.4960042628015960e-01, -4.8001065984480529e-01, -6.5394696731833579e-01,
                -7.6859392905929291e-01, -8.3792114562924969e-01, -8.7748300776370569e-01, -8.9917306621142690e-01,
                -9.1073646800620178e-01, -9.1678121884206087e-01
        },
// x = -0.0750
        {
                -9.3574428656029787e-03, -2.4964876682740961e-01, -4.8125878412185191e-01, -6.5635014082754173e-01,
                -7.7186966922939815e-01, -8.4177229961836086e-01, -8.8168223455236772e-01, -9.0357097129398234e-01,
                -9.1524331935157932e-01, -9.2134615914815698e-01
        },
// x = -0.0700
        {
                -8.7357232016522252e-03, -2.4969399993067815e-01, -4.8250714233399794e-01, -6.5875955554944698e-01,
                -7.7515744086959404e-01, -8.4563986237399547e-01, -8.8590075810782565e-01, -9.0798992071821283e-01,
                -9.1977221279302224e-01, -9.2593368877237836e-01
        },
// x = -0.0650
        {
                -8.1135675572901363e-03, -2.4973612333279044e-01, -4.8375571893831482e-01, -6.6117521261486967e-01,
                -7.7845727518880214e-01, -8.4952389413152252e-01, -8.9013866062443670e-01, -9.1243001117331379e-01,
                -9.2432325379379188e-01, -9.3054391818866078e-01
        },
// x = -0.0600
        {
                -7.4910068803816475e-03, -2.4977513493119199e-01, -4.8500449838097753e-01, -6.6359711307689473e-01,
                -7.8176920340140876e-01, -8.5342445528237176e-01, -8.9439602459812306e-01, -9.1689133976020332e-01,
                -9.2889654830162038e-01, -9.3517695839946857e-01
        },
// x = -0.0550
        {
                -6.8680721618988019e-03, -2.4981103277818800e-01, -4.8625346509803441e-01, -6.6602525791080480e-01,
                -7.8509325672686536e-01, -8.5734160637405887e-01, -8.9867293282715466e-01, -9.2137400399303837e-01,
                -9.3349220275078171e-01, -9.3983292093821980e-01
        },
// x = -0.0500
        {
                -6.2447944320520839e-03, -2.4984381508112699e-01, -4.8750260351617825e-01, -6.6845964801404378e-01,
                -7.8842946638930422e-01, -8.6127540811020398e-01, -9.0296946841292991e-01, -9.2587810180072017e-01,
                -9.3811032406415373e-01, -9.4451191787172417e-01
        },
// x = -0.0450
        {
                -5.6212047579633677e-03, -2.4987348020260194e-01, -4.8875189805353408e-01, -6.7090028420616721e-01,
                -7.9177786361715941e-01, -8.6522592135056187e-01, -9.0728571476075659e-01, -9.3040373152842160e-01,
                -9.4275101965529995e-01, -9.4921406180263224e-01
        },
// x = -0.0400
        {
                -4.9973342395361747e-03, -2.4990002666063019e-01, -4.9000133312042360e-01, -6.7334716722879639e-01,
                -7.9513847964275608e-01, -8.6919320711103898e-01, -9.1162175558062919e-01, -9.3495099193909903e-01,
                -9.4741439743055422e-01, -9.5393946587189149e-01
        },
// x = -0.0350
        {
                -4.3732140065135157e-03, -2.4992345312880662e-01, -4.9125089312014891e-01, -6.7580029774559414e-01,
                -7.9851134570194260e-01, -8.7317732656371294e-01, -9.1597767488801196e-01, -9.3951998221502575e-01,
                -9.5210056579111135e-01, -9.5868824376122075e-01
        },
// x = -0.0300
        {
                -3.7488752148550756e-03, -2.4994375843643160e-01, -4.9250056244976942e-01, -6.7825967634221707e-01,
                -8.0189649303368260e-01, -8.7717834103684778e-01, -9.2035355700462573e-01, -9.4411080195931241e-01,
                -9.5680963363512972e-01, -9.6346050969558250e-01
        },
// x = -0.0250
        {
                -3.1243490441768217e-03, -2.4996094156864662e-01, -4.9375032550087977e-01, -6.8072530352629701e-01,
                -8.0529395287968042e-01, -8.8119631201490745e-01, -9.2474948655921740e-01, -9.4872355119745133e-01,
                -9.6154171035983937e-01, -9.6825637844568013e-01
        },
// x = -0.0200
        {
                -2.4996666932229380e-03, -2.4997500166655595e-01, -4.9500016666038937e-01, -6.8319717972740612e-01,
                -8.0870375648398152e-01, -8.8523130113857551e-01, -9.2916554848834498e-01, -9.5335833037884699e-01,
                -9.6629690586365391e-01, -9.7307596533045226e-01
        },
// x = -0.0150
        {
                -1.8748593790223443e-03, -2.4998593802731167e-01, -4.9625007031130802e-01, -6.8567530529705289e-01,
                -8.1212593509257747e-01, -8.8928337020475756e-01, -9.3360182803716518e-01, -9.5801524037835295e-01,
                -9.7107533054830186e-01, -9.7791938621959140e-01
        },
// x = -0.0100
        {
                -1.2499583298593480e-03, -2.4999375010410788e-01, -4.9750002083351741e-01, -6.8815968050863952e-01,
                -8.1556051995302425e-01, -8.9335258116658889e-01, -9.3805841076020968e-01, -9.6269438249782646e-01,
                -9.7587709532094902e-01, -9.8278675753605549e-01
        },
// x = -0.0050
        {
                -6.2499476305724871e-04, -2.4999843750647555e-01, -4.9875000260455199e-01, -6.9065030555747164e-01,
                -8.1900754231406159e-01, -8.9743899613345723e-01, -9.4253538252216795e-01, -9.6739585846767051e-01,
                -9.8070231159634247e-01, -9.8767819625861208e-01
        },
// x = 0.0000
// at this point Sk(x) values cannot be calculated for k <= 1,
// these values are set to zero and not used
        {
                0.0,                     0.0,                     0.0,                     0.0,
                -8.2246703342407823e-01, -9.0154267736968796e-01, -9.4703282949724632e-01, -9.7211977044690967e-01,
                -9.8555109129743457e-01, -9.9259381992283091e-01
        }
};
