/*
 *  libgrpp - a library for the evaluation of integrals over
 *            generalized relativistic pseudopotentials.
 *
 *  Copyright (C) 2021-2023 Alexander Oleynichenko
 */

#include "scaled_mod_sph_bessel.h"

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "libgrpp.h"


/*
 * assumed max angular momentum
 */
#define LMAX (LIBGRPP_MAX_BASIS_L + LIBGRPP_MAX_RPP_L)

/*
 * number of pretabulated points
 */
#define NPOINTS 2000


/*
 * array containing pre-tabulated values of the Bessel function Kn(z).
 * 1st dim: number of point (z)
 * 2nd dim: order of Bessel function
 * 3rd dim: order of derivative
 */
double bessel_table[NPOINTS][3 * LMAX][10];

double modified_scaled_bessel_derivative(int n, int lambda, int ix);

double bessel_asymptotic_R_function(int n, double x);

extern double rfun_coeffs[50][50];

extern double bessel_table_0[1601][60];


/*
 * evaluates scaled modified spherical Bessel function of the n-th order.
 */
double modified_bessel_scaled(int n, double x)
{
    static int bessel_pretabulated = 0;

    /*
     * pre-tabulation step.
     * this code is invoked only once at start of the LIBGRPP library
     */
    if (bessel_pretabulated == 0) {


        /*FILE *f_bessel_tab = fopen("bessel_table.dat", "w");

        for (int i = 0; i <= 1600; i++) {
            double xi = 0.01 * i;

            fprintf(f_bessel_tab, "// x = %.4f\n", xi);
            fprintf(f_bessel_tab, "{\n");

            // Bessel functions
            for (int lam = 0; lam <= 3 * LMAX - 1; lam++) {
                bessel_table[i][lam][0] = gsl_sf_bessel_il_scaled(lam, xi);
                fprintf(f_bessel_tab, "%26.16e", bessel_table[i][lam][0]);

                if (lam != 3 * LMAX - 1) {
                    fprintf(f_bessel_tab, ",");
                }

                if ((lam + 1) % 4 == 0) {
                    fprintf(f_bessel_tab, "\n");
                }
            }

            fprintf(f_bessel_tab, "},\n");

            // and their 1st, 2nd, 3rd, 4th derivatives
            for (int k = 1; k < 5; k++) {
                for (int lam = 0; lam <= 3 * LMAX - 1 - 5; lam++) {
                    bessel_table[i][lam][k] = modified_scaled_bessel_derivative(k, lam, i);
                }
            }
        }

        fclose(f_bessel_tab);*/

        for (int i = 0; i <= 1600; i++) {

            // Bessel functions
            for (int lam = 0; lam <= 3 * LMAX - 1; lam++) {
                bessel_table[i][lam][0] = bessel_table_0[i][lam];
            }

            // and their 1st, 2nd, 3rd, 4th derivatives
            for (int k = 1; k < 5; k++) {
                for (int lam = 0; lam <= 3 * LMAX - 1 - 5; lam++) {
                    bessel_table[i][lam][k] = modified_scaled_bessel_derivative(k, lam, i);
                }
            }
        }

        bessel_pretabulated = 1;
    }

    /*
     * use Taylor expansion
     */
    if (x <= 1e-7) {
        switch (n) {
            case 0:
                return 1.0 - x;
            case 1:
                return (1.0 - x) * x * (1.0 / 3.0);
            case 2:
                return (1.0 - x) * x * x * (1.0 / 15.0);
            default:
                return 0.0;
        }
    }
    else if (x <= 16.0) {

        int i0 = (int) (round(x * 100) + 1e-5);
        double x0 = i0 * 0.01;
        double dx = x - x0;

        double *k_derivatives = bessel_table[i0][n];

        double d0 = k_derivatives[0];
        double d1 = k_derivatives[1];
        double d2 = k_derivatives[2];
        double d3 = k_derivatives[3];
        double d4 = k_derivatives[4];

        double dx2 = dx * dx;
        double dx3 = dx2 * dx;
        double dx4 = dx3 * dx;

        double sum = d0 +
                     d1 * dx +
                     d2 * dx2 * 0.5 +
                     d3 * dx3 * (1.0 / 6.0) +
                     d4 * dx4 * (1.0 / 24.0);

        return sum;
    }
    else {
        return bessel_asymptotic_R_function(n, -x) / (2.0 * x);
    }


    /*
     * GSL version (works slower)
     */

    /*
    gsl_sf_result result;

    gsl_set_error_handler_off();

    int status = gsl_sf_bessel_il_scaled_e(n, x, &result);

    if (status == 0) {

    }
    else if (status == 11) {
        //printf ("error: %s\n", gsl_strerror (status));
        //printf("val = %30.16e\n", result.val);
        //printf("err = %30.16e\n", result.err);
    }
    else {
        printf("error in modified bessel function!\n");
        printf("error: %s\n", gsl_strerror(status));
        exit(0);
    }

    return result.val;
    */
}


/**
 * Calculates n-th order derivative on the scaled modified spherical Bessel function K_lambda(z).
 *
 * Note that the formula (45) for derivatives in the paper of Flores-Moreno et al is wrong
 * (you must also add: - K^(n-1)_{lambda}).
 *
 * R. Flores-Moreno, R. J. Alvarez-Mendez, A. Vela, A. M. Koster,
 * Half-numerical evaluation of pseudopotential integrals.
 * J. Comput. Chem. 27 (9), 1009 (2006), doi: 10.1002/jcc.20410
 */
double modified_scaled_bessel_derivative(int n, int lambda, int ix)
{
    /*
     * base of recursion
     */
    if (n == 0) {
        return bessel_table[ix][lambda][0];
    }

    /*
     * upward recursion
     */
    double deriv = 0.0;

    if (lambda > 0) {
        deriv += lambda / (2.0 * lambda + 1.0) * bessel_table[ix][lambda - 1][n - 1];
    }
    deriv += (lambda + 1.0) / (2.0 * lambda + 1.0) * bessel_table[ix][lambda + 1][n - 1];
    deriv -= bessel_table[ix][lambda][n - 1];

    return deriv;
}


/**
 * Asymptotic formula for the scaled spherical modified Bessel function of order 'n'.
 *
 * For the formula, see:
 * R. Flores-Moreno, R. J. Alvarez-Mendez, A. Vela, A. M. Koster,
 * Half-numerical evaluation of pseudopotential integrals.
 * J. Comput. Chem. 27 (9), 1009 (2006), doi: 10.1002/jcc.20410
 *
 * In the optimized version all factors used are pretabulated above in the 'rfun_coeffs' array.
 * Unoptimized version of this code:
 *
 * double sum = 0.0;
 *
 * for (int k = 0; k <= n; k++) {
 *     sum += factorial(k + n) / (factorial(k) * factorial(n - k) * pow(2 * x, k));
 * }
 */
double bessel_asymptotic_R_function(int n, double x)
{
    double sum = 0.0;

    double denom = 1.0;
    double inv_x = 1.0 / x;
    double *coeffs = rfun_coeffs[n];

    for (int k = 0; k <= n; k++) {
        sum += coeffs[k] * denom;
        denom *= inv_x;
    }

    return sum;
}


/*
 * pretabulated coefficients which are required for the fast calculation
 * of the R_lambda(z) asymptotic relation.
 */
double rfun_coeffs[50][50] = {
        // lambda = 0
        {1.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 1
        {1.0000000000000000e+00, 1.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 2
        {1.0000000000000000e+00, 3.0000000000000000e+00, 3.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 3
        {1.0000000000000000e+00, 6.0000000000000000e+00, 1.5000000000000000e+01, 1.5000000000000000e+01, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 4
        {1.0000000000000000e+00, 1.0000000000000000e+01, 4.5000000000000000e+01, 1.0500000000000000e+02, 1.0500000000000000e+02,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 5
        {1.0000000000000000e+00, 1.5000000000000000e+01, 1.0500000000000000e+02, 4.2000000000000000e+02, 9.4500000000000000e+02,
                9.4500000000000000e+02, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 6
        {1.0000000000000000e+00, 2.1000000000000000e+01, 2.1000000000000000e+02, 1.2600000000000000e+03, 4.7250000000000000e+03,
                1.0395000000000000e+04, 1.0395000000000000e+04, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 7
        {1.0000000000000000e+00, 2.8000000000000000e+01, 3.7800000000000000e+02, 3.1500000000000000e+03, 1.7325000000000000e+04,
                6.2370000000000000e+04, 1.3513500000000000e+05, 1.3513500000000000e+05, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 8
        {1.0000000000000000e+00, 3.6000000000000000e+01, 6.3000000000000000e+02, 6.9300000000000000e+03, 5.1975000000000000e+04,
                2.7027000000000000e+05, 9.4594500000000000e+05, 2.0270250000000000e+06, 2.0270250000000000e+06, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 9
        {1.0000000000000000e+00, 4.5000000000000000e+01, 9.9000000000000000e+02, 1.3860000000000000e+04, 1.3513500000000000e+05,
                9.4594500000000000e+05, 4.7297250000000000e+06, 1.6216200000000000e+07, 3.4459425000000000e+07, 3.4459425000000000e+07,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 10
        {1.0000000000000000e+00, 5.5000000000000000e+01, 1.4850000000000000e+03, 2.5740000000000000e+04, 3.1531500000000000e+05,
                2.8378350000000000e+06, 1.8918900000000000e+07, 9.1891800000000000e+07, 3.1013482500000000e+08, 6.5472907500000000e+08,
                6.5472907500000000e+08, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 11
        {1.0000000000000000e+00, 6.6000000000000000e+01, 2.1450000000000000e+03, 4.5045000000000000e+04, 6.7567500000000000e+05,
                7.5675600000000000e+06, 6.4324260000000000e+07, 4.1351310000000000e+08, 1.9641872250000000e+09, 6.5472907500000000e+09,
                1.3749310575000000e+10, 1.3749310575000000e+10, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 12
        {1.0000000000000000e+00, 7.8000000000000000e+01, 3.0030000000000000e+03, 7.5075000000000000e+04, 1.3513500000000000e+06,
                1.8378360000000000e+07, 1.9297278000000000e+08, 1.5713497800000000e+09, 9.8209361250000000e+09, 4.5831035250000000e+10,
                1.5124241632500000e+11, 3.1623414322500000e+11, 3.1623414322500000e+11, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 13
        {1.0000000000000000e+00, 9.1000000000000000e+01, 4.0950000000000000e+03, 1.2012000000000000e+05, 2.5525500000000000e+06,
                4.1351310000000000e+07, 5.2378326000000000e+08, 5.2378326000000000e+09, 4.1247931725000000e+10, 2.5207069387500000e+11,
                1.1595251918250000e+12, 3.7948097187000000e+12, 7.9058535806250000e+12, 7.9058535806250000e+12, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 14
        {1.0000000000000000e+00, 1.0500000000000000e+02, 5.4600000000000000e+03, 1.8564000000000000e+05, 4.5945900000000000e+06,
                8.7297210000000000e+07, 1.3094581500000000e+09, 1.5713497800000000e+10, 1.5124241632500000e+11, 1.1595251918250000e+12,
                6.9571511509500000e+12, 3.1623414322500000e+13, 1.0277609654812500e+14, 2.1345804667687500e+14, 2.1345804667687500e+14,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 15
        {1.0000000000000000e+00, 1.2000000000000000e+02, 7.1400000000000000e+03, 2.7846000000000000e+05, 7.9361100000000000e+06,
                1.7459442000000000e+08, 3.0554023500000000e+09, 4.3212118950000000e+10, 4.9693936792500000e+11, 4.6381007673000000e+12,
                3.4785755754750000e+13, 2.0555219309625000e+14, 9.2498486893312500e+14, 2.9884126534762500e+15, 6.1902833536293750e+15,
                6.1902833536293750e+15, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 16
        {1.0000000000000000e+00, 1.3600000000000000e+02, 9.1800000000000000e+03, 4.0698000000000000e+05, 1.3226850000000000e+07,
                3.3331662000000000e+08, 6.7218851700000000e+09, 1.1043097065000000e+11, 1.4908181037750000e+12, 1.6564645597500000e+13,
                1.5073827493725000e+14, 1.1099818427197500e+15, 6.4748940825318750e+15, 2.8887988983603752e+16, 9.2854250304440624e+16,
                1.9189878396251062e+17, 1.9189878396251062e+17, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 17
        {1.0000000000000000e+00, 1.5300000000000000e+02, 1.1628000000000000e+04, 5.8140000000000000e+05, 2.1366450000000000e+07,
                6.1108047000000000e+08, 1.4054850810000000e+10, 2.6503432956000000e+11, 4.1411613993750000e+12, 5.3835098191875000e+13,
                5.8141906047225000e+14, 5.1799152660255000e+15, 3.7554385678684872e+16, 2.1665991737702813e+17, 9.5949391981255309e+17,
                3.0703805434001700e+18, 6.3326598707628503e+18, 6.3326598707628503e+18, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 18
        {1.0000000000000000e+00, 1.7100000000000000e+02, 1.4535000000000000e+04, 8.1396000000000000e+05, 3.3575850000000000e+07,
                1.0811423700000000e+09, 2.8109701620000000e+10, 6.0235074900000000e+11, 1.0767019638375000e+13, 1.6150529457562500e+14,
                2.0349667116528750e+15, 2.1459648959248500e+16, 1.8777192839342438e+17, 1.3432914877375744e+18, 7.6759513585004247e+18,
                3.3774185977401868e+19, 1.0765521780296847e+20, 2.2164309547669976e+20, 2.2164309547669976e+20, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 19
        {1.0000000000000000e+00, 1.9000000000000000e+02, 1.7955000000000000e+04, 1.1191950000000000e+06, 5.1482970000000000e+07,
                1.8533869200000000e+09, 5.4057118500000000e+10, 1.3050932895000000e+12, 2.6428139112375000e+13, 4.5221482481175000e+14,
                6.5571149597703750e+15, 8.0473683597181872e+16, 8.3156139717087936e+17, 7.1642212679337298e+18, 5.0661278966102802e+19,
                2.8708058080791590e+20, 1.2559775410346320e+21, 3.9895757185805960e+21, 8.2007945326378919e+21, 8.2007945326378919e+21,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 20
        {1.0000000000000000e+00, 2.1000000000000000e+02, 2.1945000000000000e+04, 1.5142050000000000e+06, 7.7224455000000000e+07,
                3.0889782000000000e+09, 1.0039179150000000e+11, 2.7105783705000000e+12, 6.1665657928875000e+13, 1.1922027199582500e+15,
                1.9671344879311124e+16, 2.7718713239029312e+17, 3.3262455886835174e+18, 3.3774185977401868e+19, 2.8708058080791590e+20,
                2.0095640656554112e+21, 1.1303797869311689e+22, 4.9204767195827347e+22, 1.5581509612011996e+23, 3.1983098677287775e+23,
                3.1983098677287775e+23, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 21
        {1.0000000000000000e+00, 2.3100000000000000e+02, 2.6565000000000000e+04, 2.0189400000000000e+06, 1.1356537500000000e+08,
                5.0195895750000000e+09, 1.8070522470000000e+11, 5.4211567410000000e+12, 1.3756185230287500e+14, 2.9805067998956250e+15,
                5.5437426478058624e+16, 8.8699882364893798e+17, 1.2196233825172898e+19, 1.4354029040395795e+20, 1.4354029040395795e+21,
                1.2057384393932467e+22, 8.3648104232906486e+22, 4.6744528836035980e+23, 2.0255962495615593e+24, 6.3966197354575558e+24,
                1.3113070457687988e+25, 1.3113070457687988e+25, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 22
        {1.0000000000000000e+00, 2.5300000000000000e+02, 3.1878000000000000e+04, 2.6565000000000000e+06, 1.6403887500000000e+08,
                7.9722893250000000e+09, 3.1623414322500000e+11, 1.0480903032600000e+13, 2.9477539779187500e+14, 7.1073623689818750e+15,
                1.4783313727482301e+17, 2.6609964709468140e+18, 4.1467195005587849e+19, 5.5821224045983649e+20, 6.4593130681781075e+21,
                6.3731888939357331e+22, 5.2977132680840781e+23, 3.6460732492108065e+24, 2.0255962495615594e+25, 8.7420469717919930e+25,
                2.7537447961144777e+26, 5.6386202968058351e+26, 5.6386202968058351e+26, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 23
        {1.0000000000000000e+00, 2.7600000000000000e+02, 3.7950000000000000e+04, 3.4534500000000000e+06, 2.3310787500000000e+08,
                1.2401338950000000e+10, 5.3945824432500000e+11, 1.9651693186125000e+13, 6.0920248876987500e+14, 1.6245399700530000e+16,
                3.7526873308224301e+17, 7.5394900010159729e+18, 1.3194107501777953e+20, 2.0095640656554112e+21, 2.6554953724732222e+22,
                3.0272647246194728e+23, 2.9515831065039861e+24, 2.4307154994738711e+25, 1.6609889246404785e+26, 9.1791493203815915e+26,
                3.9470342077640844e+27, 1.2404964652972836e+28, 2.5373791335626256e+28, 2.5373791335626256e+28, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 24
        {1.0000000000000000e+00, 3.0000000000000000e+02, 4.4850000000000000e+04, 4.4401500000000000e+06, 3.2635102500000000e+08,
                1.8928359450000000e+10, 8.9909707387500000e+11, 3.5835440515875000e+13, 1.2184049775397500e+15, 3.5739879341166000e+16,
                9.1136692319973299e+17, 2.0298626925812236e+19, 3.9582322505333860e+20, 6.7594427662954744e+21, 1.0090882415398244e+23,
                1.3118147140017716e+24, 1.4757915532519932e+25, 1.4237047925489816e+26, 1.1626922472483349e+27, 7.8940684155281688e+27,
                4.3417376285404926e+28, 1.8607446979459256e+29, 5.8359720071940396e+29, 1.1925681927744342e+30, 1.1925681927744342e+30,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 25
        {1.0000000000000000e+00, 3.2500000000000000e+02, 5.2650000000000000e+04, 5.6511000000000000e+06, 4.5067522500000000e+08,
                2.8392539175000000e+10, 1.4669478573750000e+12, 6.3707449806000000e+13, 2.3651390740477500e+15, 7.5947243599977744e+16,
                2.1265228207993769e+18, 5.2196469237802893e+19, 1.1265737943825791e+21, 2.1404902093269001e+22, 3.5776764927321044e+23,
                5.2472588560070862e+24, 6.7230504092590799e+25, 7.4744501608821538e+26, 7.1422523759540579e+27, 5.7889835047206574e+28,
                3.9075638656864439e+29, 2.1398564026378145e+30, 9.1430228112706621e+30, 2.8621636626586418e+31, 5.8435841445947271e+31,
                5.8435841445947271e+31, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 26
        {1.0000000000000000e+00, 3.5100000000000000e+02, 6.1425000000000000e+04, 7.1253000000000000e+06, 6.1455712500000000e+08,
                4.1912795925000000e+10, 2.3471165718000000e+12, 1.1064978124200000e+14, 4.4674849176457500e+15, 1.5636197211760125e+17,
                4.7846763467985981e+18, 1.2875129078658046e+20, 3.0578431561812861e+21, 6.4214706279807002e+22, 1.1925588309107015e+24,
                1.9557964826935507e+25, 2.8236811718888136e+26, 3.5711261879770289e+27, 3.9282388067747316e+28, 3.7214893958918512e+29,
                2.9957989636929402e+30, 2.0114650184795456e+31, 1.0971627373524793e+32, 4.6748673156757817e+32, 1.4608960361486818e+33,
                2.9802279137433107e+33, 2.9802279137433107e+33, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 27
        {1.0000000000000000e+00, 3.7800000000000000e+02, 7.1253000000000000e+04, 8.9066250000000000e+06, 8.2831612500000000e+08,
                6.0964066800000000e+10, 3.6883260414000000e+12, 1.8810462811140000e+14, 8.2295774798737500e+15, 3.1272394423520250e+17,
                1.0413707343032244e+19, 3.0578431561812863e+20, 7.9503922060713431e+21, 1.8347058937087716e+23, 3.7611470821029817e+24,
                6.8452876894274268e+25, 1.1038026399201726e+27, 1.5712955227098926e+28, 1.9641194033873659e+29, 2.1398564026378145e+30,
                2.0114650184795456e+31, 1.6091720147836365e+32, 1.0752194826054298e+33, 5.8435841445947272e+33, 2.4835232614527592e+34,
                7.7485925757326079e+34, 1.5795207942839547e+35, 1.5795207942839547e+35, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 28
        {1.0000000000000000e+00, 4.0600000000000000e+02, 8.2215000000000000e+04, 1.1044215000000000e+07, 1.1044215000000000e+09,
                8.7470182800000000e+10, 5.7001402458000000e+12, 3.1350771351900000e+14, 1.4813239463772750e+16, 6.0898873351065754e+17,
                2.1984493279734735e+19, 7.0150519465335390e+20, 1.9875980515178361e+22, 5.0148627761373093e+23, 1.1283441246308945e+25,
                2.2642105434259949e+26, 4.0472763463739662e+27, 6.4280271383586516e+28, 9.0349492555818833e+29, 1.1174805658219697e+31,
                1.2068790110877273e+32, 1.1264204103485455e+33, 8.9601623550452488e+33, 5.9604558274866214e+34, 3.2285802398885867e+35,
                1.3689180217127607e+36, 4.2647061445666780e+36, 8.6873643685617512e+36, 8.6873643685617512e+36, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 29
        {1.0000000000000000e+00, 4.3500000000000000e+02, 9.4395000000000000e+04, 1.3592880000000000e+07, 1.4578363800000000e+09,
                1.2391609230000000e+11, 8.6741264610000000e+12, 5.1301262212200000e+14, 2.6099517150456752e+16, 1.1570785936702492e+18,
                4.5126065153139720e+19, 1.5589004325630084e+21, 4.7936188301312512e+22, 1.3164014787360436e+24, 3.2345864906085645e+25,
                7.1160902793388421e+26, 1.4009802737448345e+28, 2.4640770697041499e+29, 3.8603874092031681e+30, 5.3639067159454546e+31,
                6.5707857270331817e+32, 7.0401275646784093e+33, 6.5281182872472521e+34, 5.1657283838217391e+35, 3.4222950542819017e+36,
                1.8480393293122271e+37, 7.8186279317055763e+37, 2.4324620231972902e+38, 4.9517976900801979e+38, 4.9517976900801979e+38,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 30
        {1.0000000000000000e+00, 4.6500000000000000e+02, 1.0788000000000000e+05, 1.6613520000000000e+07, 1.9064014200000000e+09,
                1.7348252922000000e+11, 1.3011189691500000e+13, 8.2528117471800000e+14, 4.5080984168970752e+16, 2.1488602453876058e+18,
                9.0252130306279440e+19, 3.3639430386885974e+21, 1.1185110603639586e+23, 3.3297213873911694e+24, 8.8951128491735526e+25,
                2.1348270838016525e+27, 4.6032208994473134e+28, 8.9085863289303885e+29, 1.5441549636812672e+31, 2.3893766280120661e+32,
                3.2853928635165913e+33, 3.9894056199844321e+34, 4.2432768867107142e+35, 3.9111943477507451e+36, 3.0800655488537117e+37,
                2.0328432622434498e+38, 1.0946079104387807e+39, 4.6216778440748515e+39, 1.4360213301232574e+40, 2.9215606371473171e+40,
                2.9215606371473171e+40, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 31
        {1.0000000000000000e+00, 4.9600000000000000e+02, 1.2276000000000000e+05, 2.0173560000000000e+07, 2.4712611000000000e+09,
                2.4020657892000000e+11, 1.9256560743420000e+13, 1.3066951933035000e+15, 7.6441668808254752e+16, 3.9070186279774648e+18,
                1.7620654012178366e+20, 7.0642803812460543e+21, 2.5313671366131695e+23, 8.1393189469561908e+24, 2.3545886953694697e+26,
                6.1376278659297511e+27, 1.4423425484934914e+29, 3.0543724556332761e+30, 5.8202764015678539e+31, 9.9557359500502762e+32,
                1.5232276003576923e+34, 2.0744909223919045e+35, 2.4988186110629760e+36, 2.6400561847317531e+37, 2.4200515026707736e+38,
                1.8973203780938864e+39, 1.2478530179002099e+40, 6.7014328739085347e+40, 2.8241752825757396e+41, 8.7646819114419503e+41,
                1.7821519886598634e+42, 1.7821519886598634e+42, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 32
        {1.0000000000000000e+00, 5.2800000000000000e+02, 1.3912800000000000e+05, 2.4347400000000000e+07, 3.1773357000000000e+09,
                3.2917197852000000e+11, 2.8144204163460000e+13, 2.0384445015534600e+15, 1.2740278134709125e+17, 6.9646853803076547e+18,
                3.3639430386885971e+20, 1.4464955066360969e+22, 5.5690077005489726e+23, 1.9277334348054137e+25, 6.0172822214997554e+26,
                1.6968735864629310e+28, 4.3270276454804741e+29, 9.9776166884020347e+30, 2.0786701434170907e+32, 3.9057117957889542e+33,
                6.6006529348833326e+34, 9.9952744442519035e+35, 1.3493620499740072e+37, 1.6133676684471823e+38, 1.6940360518695416e+39,
                1.5449608793050218e+40, 1.2062579173035363e+41, 7.9076907912120706e+41, 4.2362629238636097e+42, 1.7821519886598633e+43,
                5.5246711648455764e+43, 1.1227557528557138e+44, 1.1227557528557138e+44, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 33
        {1.0000000000000000e+00, 5.6100000000000000e+02, 1.5708000000000000e+05, 2.9216880000000000e+07, 4.0538421000000000e+09,
                4.4673339942000000e+11, 4.0652739347220000e+13, 3.1360684639284000e+15, 2.0894056140922966e+17, 1.2188199415538397e+19,
                6.2891108984178121e+20, 2.8929910132721939e+22, 1.1933587929747799e+24, 4.4337869000524515e+25, 1.4884856021604660e+27,
                4.5249962305678166e+28, 1.2472020860502543e+30, 3.1180052151256360e+31, 7.0674784876181083e+32, 1.4506929527216116e+34,
                2.6910354272985894e+35, 4.4978734999133567e+36, 6.7468102498700352e+37, 9.0348589433042215e+38, 1.0728894995173763e+40,
                1.1200966374961408e+41, 1.0167031017272664e+42, 7.9076907912120706e+42, 5.1682407671136041e+43, 2.7623355824227882e+44,
                1.1601809446175710e+45, 3.5928184091382843e+45, 7.2979123935621403e+45, 7.2979123935621403e+45, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 34
        {1.0000000000000000e+00, 5.9500000000000000e+02, 1.7671500000000000e+05, 3.4871760000000000e+07, 5.1348666600000000e+09,
                6.0077939922000000e+11, 5.8075341924600000e+13, 4.7621780378172000e+15, 3.3751936843029408e+17, 2.0963702994726040e+19,
                1.1530036647099323e+21, 5.6601998085760311e+22, 2.4952047489472671e+24, 9.9232373477364390e+25, 3.5723654451851181e+27,
                1.1669727120938052e+29, 3.4644502390284845e+30, 9.3540156453769076e+31, 2.2969305084758852e+33, 5.1257817662830279e+34,
                1.0379708076723131e+36, 1.9029464807325740e+37, 3.1485114499393499e+38, 4.6816996342576421e+39, 6.2227590972007826e+40,
                7.3428557346969232e+41, 7.6252732629544975e+42, 6.8909876894848052e+43, 5.3405154593507234e+44, 3.4805428338527132e+45,
                1.8562895113881136e+46, 7.7844398864662830e+46, 2.4083110898755061e+47, 4.8896013036866339e+47, 4.8896013036866339e+47,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 35
        {1.0000000000000000e+00, 6.3000000000000000e+02, 1.9813500000000000e+05, 4.1410215000000000e+07, 6.4599935400000000e+09,
                8.0103919896000000e+11, 8.2106517893400000e+13, 7.1432670567258000e+15, 5.3753084601861645e+17, 3.5477035837228687e+19,
                2.0754065964778780e+21, 1.0848716299770726e+23, 5.0988966608922417e+24, 2.1650699667788595e+26, 8.3355193720986085e+27,
                2.9174317802345130e+29, 9.2993137994975104e+30, 2.7022711864422177e+32, 7.1610186440718767e+33, 1.7299513461205220e+35,
                3.8058929614651479e+36, 7.6117859229302960e+37, 1.3805011742041766e+39, 2.2628214898911938e+40, 3.3376616975895104e+41,
                4.4057134408181539e+42, 5.1682407671136041e+43, 5.3405154593507234e+44, 4.8064639134156512e+45, 3.7125790227762272e+46,
                2.4131763648045479e+47, 1.2844325812669368e+48, 5.3785614340552972e+48, 1.6624644432534555e+49, 3.3738248995437775e+49,
                3.3738248995437775e+49, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 36
        {1.0000000000000000e+00, 6.6600000000000000e+02, 2.2144500000000000e+05, 4.8939345000000000e+07, 8.0749919250000000e+09,
                1.0594389405600000e+12, 1.1494912505076000e+14, 1.0591740808248600e+16, 8.4469132945782579e+17, 5.9128393062047810e+19,
                3.6718732091531691e+21, 2.0395586643568966e+23, 1.0197793321784483e+25, 4.6125403640071350e+26, 1.8944362209315020e+28,
                7.0851914662838170e+29, 2.4178215878693527e+31, 7.5379143621809230e+32, 2.1483055932215630e+34, 5.5969014139193357e+35,
                1.3320625365128018e+37, 2.8924786507135127e+38, 5.7192191502744452e+39, 1.0269728300275418e+41, 1.6688308487947554e+42,
                2.4431683626355217e+43, 3.2043092756104344e+44, 3.7383608215455063e+45, 3.8451711307325209e+46, 3.4473948068636396e+47,
                2.6544940012850026e+48, 1.7211396588976951e+49, 9.1435544378940057e+49, 3.8236682194829480e+50, 1.1808387148403220e+51,
                2.3954156786760820e+51, 2.3954156786760820e+51, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 37
        {1.0000000000000000e+00, 7.0300000000000000e+02, 2.4675300000000000e+05, 5.7575700000000000e+07, 1.0032565725000000e+10,
                1.3905136094850000e+12, 1.5944556055428000e+14, 1.5534553185431280e+16, 1.3107279250207642e+18, 9.7139502887649968e+19,
                6.3917792900073684e+21, 3.7653390726588858e+23, 1.9987674910697586e+25, 9.6094590916815319e+26, 4.2007064029350700e+28,
                1.6746816193034478e+30, 6.1021211503369381e+31, 2.0352368777888494e+33, 6.2187793487992616e+34, 1.7412582176637932e+36,
                4.4663273283076299e+37, 1.0485235108836483e+39, 2.2495595324412818e+40, 4.4013121286894641e+41, 7.8306678289600057e+42,
                1.2623036540283529e+44, 1.8351953123950669e+45, 2.3925509257891243e+46, 2.7770680388623764e+47, 2.8441007156625026e+48,
                2.5407299726585025e+49, 1.9506249467507211e+50, 1.2618105124293728e+51, 6.6914193840951586e+51, 2.7946516251220958e+52,
                8.6234964432338948e+52, 1.7486534454335398e+53, 1.7486534454335398e+53, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 38
        {1.0000000000000000e+00, 7.4100000000000000e+02, 2.7417000000000000e+05, 6.7445820000000000e+07, 1.2393169425000000e+10,
                1.8118813699350000e+12, 2.1923764576213500e+14, 2.2550157849819600e+16, 2.0097828183651720e+18, 1.5743298743860514e+20,
                1.0957335925726917e+22, 6.8333931318624231e+23, 3.8437836366726128e+25, 1.9603296547030326e+27, 9.1015305396926512e+28,
                3.8590489488296838e+30, 1.4977933732645212e+32, 5.3303822989707952e+33, 1.7412582176637933e+35, 5.2237746529913798e+36,
                1.4391499168991251e+38, 3.6389933613020737e+39, 8.4358482466548073e+40, 1.7898669323337157e+42, 3.4678671813965737e+43,
                6.1173177079835560e+44, 9.7877083327736896e+45, 1.4137800925117553e+47, 1.8328649056491685e+48, 2.1172749772154188e+49,
                2.1596204767597270e+50, 1.9227588760828539e+51, 1.4721122645009349e+52, 9.5018155254151250e+52, 5.0303729252197726e+53,
                2.0983841345202479e+54, 6.4700177481040970e+54, 1.3114900840751548e+55, 1.3114900840751548e+55, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 39
        {1.0000000000000000e+00, 7.8000000000000000e+02, 3.0381000000000000e+05, 7.8686790000000000e+07, 1.5225893865000000e+10,
                2.3447876552100000e+12, 2.9896042603927500e+14, 3.2415851909115676e+16, 3.0470900794568735e+18, 2.5189277990176820e+20,
                1.8514119322779964e+22, 1.2202487735468612e+24, 7.2604802026038244e+25, 3.9206593094060651e+27, 1.9295244744148420e+29,
                8.6828601348667891e+30, 3.5816798056325503e+32, 1.3568245851925663e+34, 4.7262723050874392e+35, 1.5148946493675002e+37,
                4.4689392156341257e+38, 1.2129977871006912e+40, 3.0269808414467250e+41, 6.9357343627931483e+42, 1.4565042161865610e+44,
                2.7964880950781971e+45, 4.8938541663868447e+46, 7.7757905088146541e+47, 1.1163813516226753e+49, 1.4397469845064846e+50,
                1.6557090321824573e+51, 1.6824140165724971e+52, 1.4931424397080912e+53, 1.1402178630498150e+54, 7.3443444708208672e+54,
                3.8820106488624587e+55, 1.6175044370260244e+56, 4.9836623194855884e+56, 1.0098473647378693e+57, 1.0098473647378693e+57,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 40
        {1.0000000000000000e+00, 8.2000000000000000e+02, 3.3579000000000000e+05, 9.1446810000000000e+07, 1.8609425835000000e+10,
                3.0147269852700000e+12, 4.0447587052372500e+14, 4.6168031506922328e+16, 4.5706351191853102e+18, 3.9815310371569815e+20,
                3.0856865537966605e+22, 2.1459547396858593e+24, 1.3483748947692816e+26, 7.6961090147600541e+27, 4.0074739084000568e+29,
                1.9102292296706936e+31, 8.3572528798092841e+32, 3.3625652763467943e+34, 1.2460172440685066e+36, 4.2561325863182151e+37,
                1.3406817646902376e+39, 3.8943613164811663e+40, 1.0426267342760942e+42, 2.5703015579762841e+43, 5.8260168647462440e+44,
                1.2118115078672187e+46, 2.3071026784395125e+47, 4.0075228006967830e+48, 6.3261609925284934e+49, 9.0311401755406768e+50,
                1.1589963225277201e+52, 1.3272377241849699e+53, 1.3438281957372820e+54, 1.1890843428948071e+55, 9.0580248473457363e+55,
                5.8230159732936878e+56, 3.0732584303494464e+57, 1.2791399953346345e+58, 3.9384047224776901e+58, 7.9777941814291672e+58,
                7.9777941814291672e+58, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 41
        {1.0000000000000000e+00, 8.6100000000000000e+02, 3.7023000000000000e+05, 1.0588578000000000e+08, 2.2633085475000000e+10,
                3.8521511478450000e+12, 5.4315331184614500e+14, 6.5178397421537400e+16, 6.7867006315175813e+18, 6.2211422455577838e+20,
                5.0764520723751509e+22, 3.7196548821221565e+24, 2.4642713594059285e+26, 1.4842495957037247e+28, 8.1633727763704851e+29,
                4.1143398792907248e+31, 1.9054536565965168e+33, 8.1261994178380867e+34, 3.1963051043496476e+36, 1.1607634326322404e+38,
                3.8943613164811664e+39, 1.2072520081091615e+41, 3.4571307504944172e+42, 9.1388499839156768e+43, 2.2275946835794461e+45,
                4.9987224699522775e+46, 1.0305058630363156e+48, 1.9465110746241519e+49, 3.3577316037266620e+50, 5.2681651023987281e+51,
                7.4807944454061944e+52, 9.5561116141317828e+53, 1.0899939809869065e+55, 1.0999030171776966e+56, 9.7050266221561463e+56,
                7.3758202328386708e+57, 4.7328179827381474e+58, 2.4943229909025372e+59, 1.0371132435857918e+60, 3.1911176725716667e+60,
                6.4620132869576254e+60, 6.4620132869576254e+60, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 42
        {1.0000000000000000e+00, 9.0300000000000000e+02, 4.0725300000000000e+05, 1.2217590000000000e+08, 2.7397945575000000e+10,
                4.8932730796950000e+12, 7.2420441579486000e+14, 9.1249756390152352e+16, 9.9804421051729142e+18, 9.6144925613165747e+20,
                8.2492346176096214e+22, 6.3594099597572353e+24, 4.4356884469306716e+26, 2.8149561297829258e+28, 1.6326745552740970e+30,
                8.6858286340581957e+31, 4.2506273877922301e+33, 1.9177830626097884e+35, 7.9907627608741190e+36, 3.0785464952420287e+38,
                1.0975018255537833e+40, 3.6217560243274846e+41, 1.1062818401582135e+43, 3.1264486787079947e+44, 8.1678471731246364e+45,
                1.9700847381576624e+47, 4.3796499179043412e+48, 8.9539509432710981e+49, 1.6788658018633310e+51, 2.8772286328485359e+52,
                4.4884766672437164e+53, 6.3417831621056375e+54, 8.0659554593031079e+55, 9.1658584764808051e+56, 9.2197752910483396e+57,
                8.1134022561225381e+58, 6.1526633775595913e+59, 3.9410303256260089e+60, 2.0742264871715836e+61, 8.6160177159435002e+61,
                2.6494254476526262e+62, 5.3634710281748292e+62, 5.3634710281748292e+62, 0.0000000000000000e+00, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 43
        {1.0000000000000000e+00, 9.4600000000000000e+02, 4.4698500000000000e+05, 1.4050228500000000e+08, 3.3018036975000000e+10,
                6.1809765217200000e+12, 9.5908152362022000e+14, 1.2673577276410050e+17, 1.4542929924680532e+19, 1.4704518034954760e+21,
                1.3248770749494239e+23, 1.0731504307090333e+25, 7.8697698251995782e+26, 5.2545847755947955e+28, 3.2090499879525356e+30,
                1.7992073599120551e+32, 9.2884079955459836e+33, 4.4256532214072039e+35, 1.9497461136532850e+37, 7.9529117793752411e+38,
                3.0062006526038411e+40, 1.0536017525316319e+42, 3.4242056957278036e+43, 1.0317280639736383e+45, 2.8802408452597401e+46,
                7.4425423441511693e+47, 1.7776226137376445e+49, 3.9173535376811057e+50, 7.9466314621530998e+51, 1.4797175826078185e+53,
                2.5204522823753176e+54, 3.9107662832984770e+55, 5.4995150858884829e+56, 6.9660524421254113e+57, 7.8880299712302451e+58,
                7.9105671997194750e+59, 6.9437200975315386e+60, 5.2547071008346777e+61, 3.3602469092179653e+62, 1.7662836317684177e+63,
                7.3300770718389326e+63, 2.2526578318334282e+64, 4.5589503739486049e+64, 4.5589503739486049e+64, 0.0000000000000000e+00,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 44
        {1.0000000000000000e+00, 9.9000000000000000e+02, 4.8955500000000000e+05, 1.6106359500000000e+08, 3.9621644370000000e+10,
                7.7658422965200000e+12, 1.2619493731845000e+15, 1.7468984894511149e+17, 2.1006454335649657e+19, 2.2266841595788638e+21,
                2.1042165308020262e+23, 1.7885840511817223e+25, 1.3772097194099262e+27, 9.6616558777065585e+28, 6.2041633100415692e+30,
                3.6604563529245253e+32, 1.9903731419027108e+34, 9.9986980187347942e+35, 4.6493945787116795e+37, 2.0041337684025609e+39,
                8.0165350736102435e+40, 2.9775701701980905e+42, 1.0272617087183411e+44, 3.2917038231539888e+45, 9.7928188738831166e+46,
                2.7028180091917401e+48, 6.9129768312019513e+49, 1.6360711833844617e+51, 3.5759841579688950e+52, 7.2012922353580503e+53,
                1.3322390635412394e+55, 2.2562113172875826e+56, 3.4830262210627056e+57, 4.8762367094877882e+58, 6.1526633775595913e+59,
                6.9437200975315386e+60, 6.9437200975315394e+61, 6.0804467881086989e+62, 4.5923374425978860e+63, 2.9320308287355731e+64,
                1.5393161850861759e+65, 6.3825305235280469e+65, 1.9603486607978999e+66, 3.9662868253352865e+66, 3.9662868253352865e+66,
                0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 45
        {1.0000000000000000e+00, 1.0350000000000000e+03, 5.3509500000000000e+05, 1.8407268000000000e+08, 4.7352696930000000e+10,
                9.7073028706500000e+12, 1.6502414880105000e+15, 2.3904926697752099e+17, 3.0090326480795455e+19, 3.3400262393682954e+21,
                3.3066259769746125e+23, 2.9459031431228367e+25, 2.3788167880716906e+27, 1.7511751278343138e+29, 1.1807923719111372e+31,
                7.3209127058490507e+32, 4.1866469536574267e+34, 2.2139974184341332e+36, 1.0848587350327252e+38, 4.9332523529909190e+39,
                2.0842991191386631e+41, 8.1883179680447479e+42, 2.9924580210490806e+44, 1.0174357271566874e+46, 3.2176404871330240e+47,
                9.4598630321710909e+48, 2.5832702895544132e+50, 6.5442847335378469e+51, 1.5355696678337018e+53, 3.3305976588530985e+54,
                6.6611953177061970e+55, 1.2248004293846877e+57, 2.0630232232448336e+58, 3.1695538611670622e+59, 4.4187309711564336e+60,
                5.5549760780252308e+61, 6.2493480877783854e+62, 6.2324579578114167e+63, 5.4452001105089213e+64, 4.1048431602298024e+65,
                2.6168375146464991e+66, 1.3722440625585301e+67, 5.6850111163139101e+67, 1.7451662031475260e+68, 3.5299952745484048e+68,
                3.5299952745484048e+68, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 46
        {1.0000000000000000e+00, 1.0810000000000000e+03, 5.8374000000000000e+05, 2.0975724000000000e+08, 5.6372258250000000e+10,
                1.2074937717150000e+13, 2.1453139344136500e+15, 3.2486182435406701e+17, 4.2759937630604067e+19, 4.9649038693312506e+21,
                5.1436404086271749e+23, 4.7976136902286196e+25, 4.0579815796517076e+27, 3.1308888649158942e+29, 2.2139856973333827e+31,
                1.4405666937315875e+33, 8.6524037042253478e+34, 4.8097185297017371e+36, 2.4796771086462288e+38, 1.1876348257200359e+40,
                5.2909131485827601e+41, 2.1944692154359926e+43, 8.4786310596390627e+44, 3.0523071814700622e+46, 1.0237947004514167e+48,
                3.1983346442102258e+49, 9.2997730423958877e+50, 2.5143830818329623e+52, 6.3128975233163300e+53, 1.4693813200822492e+55,
                3.1640677759104434e+56, 6.2873088708413973e+57, 1.1493986529506929e+59, 1.9261135002476763e+60, 2.9458206474376228e+61,
                4.0904823847276701e+62, 5.1244654319782758e+63, 5.7477112277594171e+64, 5.7174601160343675e+65, 4.9844524088504742e+66,
                3.7508004376599823e+67, 2.3877046688518421e+68, 1.2507024455890602e+69, 5.1773264026709932e+69, 1.5884978735467820e+70,
                3.2122956998390481e+70, 3.2122956998390481e+70, 0.0000000000000000e+00, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 47
        {1.0000000000000000e+00, 1.1280000000000000e+03, 6.3562800000000000e+05, 2.3836050000000000e+08, 6.6860120250000000e+10,
                1.4949922887900000e+13, 2.7732106957054500e+15, 4.3856346287799046e+17, 6.0302476145723687e+19, 7.3167004390144743e+21,
                7.9239865754526747e+23, 7.7294887231461096e+25, 6.8405975199843072e+27, 5.5250979969104019e+29, 4.0925190162829194e+31,
                2.7910979691049507e+33, 1.7583917205361192e+35, 1.0260732863363706e+37, 5.5578969676553410e+38, 2.7994249463400848e+40,
                1.3129302998334999e+42, 5.7393810249864419e+43, 2.3401021724603811e+45, 8.9025626126210148e+46, 3.1604097274804605e+48,
                1.0467277017415285e+50, 3.2327782480709515e+51, 9.3032174027819598e+52, 2.4919332328880251e+54, 6.2040544625694973e+55,
                1.4331365808535538e+57, 3.0650630745351811e+58, 6.0534995722069831e+59, 1.1006362858558150e+61, 1.8354728649419034e+62,
                2.7951629628972415e+63, 3.8666420986745172e+64, 4.8280774313179104e+65, 5.3998234429213471e+66, 5.3582863395142599e+67,
                4.6617091153774061e+68, 3.5019668476493689e+69, 2.2262503531485273e+70, 1.1648984406009735e+71, 4.8184435497585726e+71,
                1.4776560219259621e+72, 2.9874350008503150e+72, 2.9874350008503150e+72, 0.0000000000000000e+00, 0.0000000000000000e+00},
        // lambda = 48
        {1.0000000000000000e+00, 1.1760000000000000e+03, 6.9090000000000000e+05, 2.7014190000000000e+08, 7.9016505750000000e+10,
                1.8426649140900000e+13, 3.5655566087641500e+15, 5.8831684044608474e+17, 8.4423466604013158e+19, 1.0693639103175000e+22,
                1.2094505825690927e+24, 1.2325400936908661e+26, 1.1400995866640512e+28, 9.6294565089009858e+29, 7.4628287943982639e+31,
                5.3284597592003606e+33, 3.5167834410722384e+35, 2.1514439874794866e+37, 1.2227373328841749e+39, 6.4676369449926094e+40,
                3.1885450138813566e+42, 1.4667307063854240e+44, 6.3002750797010264e+45, 2.5283277819843683e+47, 9.4812291824413815e+48,
                3.3222227055274598e+50, 1.0873890470784109e+52, 3.3225776438506998e+53, 9.4693462849744954e+54, 2.5142747032518486e+56,
                6.2102585170320663e+57, 1.4243528405192900e+59, 3.0267497861034914e+60, 5.9434359436214007e+61, 1.0750626780374005e+63,
                1.7846040455420849e+64, 2.7066494690721621e+65, 3.7307871060183856e+66, 4.6438481609123586e+67, 5.1796767948637847e+68,
                5.1278800269151473e+69, 4.4525007062970545e+70, 3.3393755297227907e+71, 2.1201151618937719e+72, 1.1082420164444717e+73,
                4.5807336679704831e+73, 1.4040944503996481e+74, 2.8380632508077990e+74, 2.8380632508077990e+74, 0.0000000000000000e+00},
        // lambda = 49
        {1.0000000000000000e+00, 1.2250000000000000e+03, 7.4970000000000000e+05, 3.0537780000000000e+08, 9.3063884550000000e+10,
                2.2614523945650000e+13, 4.5605956623727500e+15, 7.8442245392811302e+17, 1.1736920966899391e+20, 1.5505776699603750e+22,
                1.8296816505532427e+24, 1.9461159374066307e+26, 1.8796236428785709e+28, 1.6584063987551699e+30, 1.3433091829916875e+32,
                1.0030041899671267e+34, 6.9269976869604691e+35, 4.4373532241764414e+37, 2.6426903646206363e+39, 1.4659977075316582e+41,
                7.5865381364763308e+42, 3.6668267659635603e+44, 1.6567390024398995e+46, 7.0015230885720968e+47, 2.7685189212728831e+49,
                1.0243520008709669e+51, 3.5458338491687312e+52, 1.1477995496938781e+54, 3.4720936378239817e+55, 9.8056713426822102e+56,
                2.5821601202396487e+58, 6.3304570689746226e+59, 1.4421572510257811e+61, 3.0460109211059682e+62, 5.9486801518069496e+63,
                1.0707624273252509e+65, 1.7697323451625674e+66, 2.6737307593131764e+67, 3.6728617272670476e+68, 4.5581155794801306e+69,
                5.0709035821716453e+70, 5.0090632945841862e+71, 4.3411881886396280e+72, 3.2508432482371166e+73, 2.0613301505867174e+74,
                1.0764724119730636e+75, 4.4462990929322190e+75, 1.3622703603877436e+76, 2.7529213532835652e+76, 2.7529213532835652e+76},
};
