! ***************************************************************************************************
!  Copyright (C) 2020-2023 GreenX library                                                          
!  This file is distributed under the terms of the APACHE2 License.                                 
!                                                                                                   
! ***************************************************************************************************
!> \brief This module contains the tabulated minimax coefficients that approximate                  
!>        1/x ~ sum_{i}^{k} w_i exp(-a_i * x) with x \in [1:rc]                                     
!> The arrays containing the coefficients and weights are stored in the `er_aw_aux` derived type.
!> To extend this module, add the new entries to `tau_npoints_supported`, `energy_ranges_grids`,
!> and fill the corresponding arrays in the derived type.
!> reference: [https://doi.org/10.1021/ct5001268](https://doi.org/10.1021/ct5001268)
!> reference: [https://doi.org/10.1103/PhysRevB.94.165109](https://doi.org/10.1103/PhysRevB.94.165109)
! ***************************************************************************************************
module minimax_tau
#include "gx_common.h"
  use kinds,          only: dp
  use error_handling, only: register_exc
  use minimax_utils,  only: er_aw_aux
  implicit none

  private

  !> list wth the number of points supported for the imag-time meshes
  integer, parameter :: ngrids = 15
  integer, parameter, public :: tau_npoints_supported(ngrids) = &
       [6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34]
  integer, parameter :: energy_ranges_grids(ngrids) = &
       [13, 13, 15, 18, 21, 36, 37, 39, 40, 40, 19, 28, 28, 18, 7]

  public :: get_points_weights_tau

contains

  !> \brief Stores the minimax coefficients for all supported grid sizes
  !! @param[in] grid_size  - size of the grid
  !! @param[inout] aw - derived type of energy ranges and coefficients:weights
  subroutine set_aw_array_tau(grid_size, aw)
    integer, intent(in)            :: grid_size
    type(er_aw_aux), intent(inout) :: aw

    select case(grid_size)
    case(6)
       aw%energy_range(:) = [1.5849_dp, 2.5119_dp, 3.9811_dp, 6.3096_dp, 10.0000_dp,&
            15.8489_dp, 25.1189_dp, 39.8107_dp, 63.0957_dp,&
            100.0000_dp, 158.4893_dp, 251.1886_dp, 398.1072_dp]
       aw%aw_erange_matrix(:, 1) = [0.1534423273_dp,  0.8208030144_dp,  2.0763266378_dp,  4.0375449448_dp,  6.9549672561_dp,  11.4891014500_dp,  0.3951503587_dp,&
            0.9483584804_dp,  1.5813287669_dp,  2.3797403876_dp,  3.5481339356_dp,  5.8817260742_dp]
       aw%aw_erange_matrix(:, 2) = [0.1205040984_dp,  0.6471706375_dp,  1.6495023836_dp,  3.2457669505_dp,  5.6881108969_dp,  9.6288261179_dp,  0.3106082604_dp,&
            0.7514093561_dp,  1.2720076125_dp,  1.9595162536_dp,  3.0177199430_dp,  5.2126406909_dp]
       aw%aw_erange_matrix(:, 3) = [0.1005838109_dp,  0.5423052006_dp,  1.3926894069_dp,  2.7733849876_dp,  4.9443770209_dp,  8.5653072999_dp,  0.2594925337_dp,&
            0.6326994282_dp,  1.0873449642_dp,  1.7143811393_dp,  2.7218526271_dp,  4.8651609155_dp]
       aw%aw_erange_matrix(:, 4) = [0.0745374837_dp,  0.4051625322_dp,  1.0572745177_dp,  2.1597944537_dp,  3.9894964633_dp,  7.2237871301_dp,  0.1926489629_dp,&
            0.4774754563_dp,  0.8471482105_dp,  1.4011198768_dp,  2.3561324362_dp,  4.4544447949_dp]
       aw%aw_erange_matrix(:, 5) = [0.0543212595_dp,  0.2985187630_dp,  0.7962225874_dp,  1.6836229509_dp,  3.2535534712_dp,  6.1993107282_dp,  0.1407385830_dp,&
            0.3565694185_dp,  0.6603802522_dp,  1.1604683175_dp,  2.0803300697_dp,  4.1513349990_dp]
       aw%aw_erange_matrix(:, 6) = [0.0391835514_dp,  0.2184067186_dp,  0.5994995153_dp,  1.3240424530_dp,  2.6961080409_dp,  5.4203992106_dp,  0.1018333153_dp,&
            0.2654513291_dp,  0.5191592345_dp,  0.9779410789_dp,  1.8692768235_dp,  3.9186061085_dp]
       aw%aw_erange_matrix(:, 7) = [0.0280585989_dp,  0.1592861279_dp,  0.4535621551_dp,  1.0550856007_dp,  2.2736909699_dp,  4.8223631523_dp,  0.0732079266_dp,&
            0.1979229248_dp,  0.4135142943_dp,  0.8384872273_dp,  1.7033414620_dp,  3.7333533452_dp]
       aw%aw_erange_matrix(:, 8) = [0.0199972626_dp,  0.1162433512_dp,  0.3464841610_dp,  0.8546910330_dp,  1.9521830237_dp,  4.3586015820_dp,  0.0524378861_dp,&
            0.1485110005_dp,  0.3348353302_dp,  0.7306689896_dp,  1.5701588368_dp,  3.5827524038_dp]
       aw%aw_erange_matrix(:, 9) = [0.0142171937_dp,  0.0852259462_dp,  0.2684487069_dp,  0.7053207912_dp,  1.7059999114_dp,  3.9958883070_dp,  0.0375251243_dp,&
            0.1126916200_dp,  0.2761723166_dp,  0.6463232558_dp,  1.4619350572_dp,  3.4590599788_dp]
       aw%aw_erange_matrix(:, 10) = [0.0101053619_dp,  0.0630473824_dp,  0.2117607106_dp,  0.5936601317_dp,  1.5164074935_dp,  3.7104671974_dp,  0.0269037518_dp,&
            0.0868884069_dp,  0.2322198587_dp,  0.5797744541_dp,  1.3735367201_dp,  3.3571561550_dp]
       aw%aw_erange_matrix(:, 11) = [0.0071978808_dp,  0.0472825494_dp,  0.1706047888_dp,  0.5098889800_dp,  1.3697999561_dp,  3.4851720987_dp,  0.0193881718_dp,&
            0.0683650657_dp,  0.1990907295_dp,  0.5270527982_dp,  1.3013617514_dp,  3.2733697689_dp]
       aw%aw_erange_matrix(:, 12) = [0.0051532116_dp,  0.0361337473_dp,  0.1407180752_dp,  0.4469079904_dp,  1.2563367882_dp,  3.3075069418_dp,  0.0141042618_dp,&
            0.0550896004_dp,  0.1740170543_dp,  0.4853305543_dp,  1.2427571345_dp,  3.2049369357_dp]
       aw%aw_erange_matrix(:, 13) = [0.0037247921_dp,  0.0282947016_dp,  0.1190443970_dp,  0.3996406559_dp,  1.1688995465_dp,  3.1683119524_dp,  0.0104189611_dp,&
            0.0455916166_dp,  0.1550538183_dp,  0.4525504643_dp,  1.1957054331_dp,  3.1497211714_dp]
       aw%aw_erange_matrix(:, 14) = [0.0027359974_dp,  0.0228248008_dp,  0.1034065810_dp,  0.3644298766_dp,  1.1022400211_dp,  3.0606897855_dp,  0.0078759332_dp,&
            0.0388225207_dp,  0.1408179499_dp,  0.4271560097_dp,  1.1585955269_dp,  3.1059901069_dp]
    case(8)
       aw%energy_range(:) = [10.0000_dp, 15.8489_dp, 25.1189_dp, 39.8107_dp, 63.0957_dp,&
            100.0000_dp, 158.4893_dp, 251.1886_dp, 398.1072_dp, 630.9573_dp,&
            1000.0000_dp, 1584.8932_dp, 2511.8864_dp]
       aw%aw_erange_matrix(:, 1) = [0.0410961677_dp,  0.2215921004_dp,  0.5688643520_dp,  1.1299764766_dp,  1.9937655343_dp,  3.3208105202_dp,  5.4098231211_dp,&
            8.9485547791_dp,  0.1060264691_dp,  0.2585290567_dp,  0.4435891050_dp,  0.6929754149_dp,  1.0604512559_dp,  1.6406263618_dp,  2.6364875094_dp,  4.7875207638_dp]
       aw%aw_erange_matrix(:, 2) = [0.0294981977_dp,  0.1602023889_dp,  0.4171350374_dp,  0.8475736875_dp,  1.5433057317_dp,  2.6707425150_dp,  4.5354087346_dp,&
            7.8277568864_dp,  0.0762269210_dp,  0.1885752622_dp,  0.3328646094_dp,  0.5428410724_dp,  0.8754790293_dp,  1.4275727338_dp,  2.4016019104_dp,  4.5261221056_dp]
       aw%aw_erange_matrix(:, 3) = [0.0209899113_dp,  0.1150165283_dp,  0.3048862124_dp,  0.6373750483_dp,  1.2053450775_dp,  2.1776820981_dp,  3.8637697754_dp,&
            6.9570683332_dp,  0.0543478976_dp,  0.1368874674_dp,  0.2503214962_dp,  0.4296737848_dp,  0.7335320894_dp,  1.2599694377_dp,  2.2127956985_dp,  4.3145944146_dp]
       aw%aw_erange_matrix(:, 4) = [0.0148401762_dp,  0.0822239432_dp,  0.2229043947_dp,  0.4824765203_dp,  0.9529144401_dp,  1.8027560234_dp,  3.3434086590_dp,&
            6.2719911449_dp,  0.0385180029_dp,  0.0992010434_dp,  0.1894294562_dp,  0.3445892444_dp,  0.6234527401_dp,  1.1253809532_dp,  2.0573355093_dp,  4.1393353027_dp]
       aw%aw_erange_matrix(:, 5) = [0.0104449886_dp,  0.0586763377_dp,  0.1635760280_dp,  0.3689646813_dp,  0.7643187927_dp,  1.5159884551_dp,  2.9363772533_dp,&
            5.7267555407_dp,  0.0271911499_dp,  0.0719923583_dp,  0.1447925468_dp,  0.2804290097_dp,  0.5370201382_dp,  1.0156145492_dp,  1.9274472597_dp,  3.9922032180_dp]
       aw%aw_erange_matrix(:, 6) = [0.0073303679_dp,  0.0418998419_dp,  0.1209028303_dp,  0.2859242910_dp,  0.6229028126_dp,  1.2950331297_dp,  2.6150915219_dp,&
            5.2886811820_dp,  0.0191536972_dp,  0.0524874990_dp,  0.1121483173_dp,  0.2317178191_dp,  0.4683927372_dp,  0.9251988681_dp,  1.8180761389_dp,  3.8678537709_dp]
       aw%aw_erange_matrix(:, 7) = [0.0051372225_dp,  0.0300161640_dp,  0.0903142204_dp,  0.2250889082_dp,  0.5162665984_dp,  1.1235047008_dp,  2.3595117299_dp,&
            4.9341336214_dp,  0.0134857752_dp,  0.0385736952_dp,  0.0882435850_dp,  0.1944219012_dp,  0.4134424375_dp,  0.8503209725_dp,  1.7256834115_dp,  3.7624829337_dp]
       aw%aw_erange_matrix(:, 8) = [0.0036002613_dp,  0.0216336337_dp,  0.0684129319_dp,  0.1803598683_dp,  0.4353504991_dp,  0.9894641326_dp,  2.1550074529_dp,&
            4.6457957380_dp,  0.0095074576_dp,  0.0286789793_dp,  0.0706649087_dp,  0.1656344893_dp,  0.3692115544_dp,  0.7881982435_dp,  1.6476414680_dp,  3.6732307680_dp]
       aw%aw_erange_matrix(:, 9) = [0.0025268045_dp,  0.0157375491_dp,  0.0527163930_dp,  0.1473061940_dp,  0.3735693722_dp,  0.8841524703_dp,  1.9907155422_dp,&
            4.4106669852_dp,  0.0067246311_dp,  0.0216513436_dp,  0.0576569758_dp,  0.1432645576_dp,  0.3335125532_dp,  0.7366874142_dp,  1.5818713728_dp,  3.5978167045_dp]
       aw%aw_erange_matrix(:, 10) = [0.0017790505_dp,  0.0115988461_dp,  0.0414413896_dp,  0.1227558108_dp,  0.3261731917_dp,  0.8011585238_dp,  1.8585679204_dp,&
            4.2189781344_dp,  0.0047836527_dp,  0.0166593084_dp,  0.0479683727_dp,  0.1258102636_dp,  0.3047080911_dp,  0.6941204359_dp,  1.5267225054_dp,  3.5344255520_dp]
       aw%aw_erange_matrix(:, 11) = [0.0012593172_dp,  0.0086977894_dp,  0.0333179986_dp,  0.1044408163_dp,  0.2897047768_dp,  0.7357044190_dp,  1.7524150854_dp,&
            4.0631467031_dp,  0.0034335494_dp,  0.0131079750_dp,  0.0407111652_dp,  0.1121712160_dp,  0.2815295945_dp,  0.6591400198_dp,  1.4808101311_dp,  3.4815299149_dp]
       aw%aw_erange_matrix(:, 12) = [0.0008992926_dp,  0.0066689908_dp,  0.0274568200_dp,  0.0907619245_dp,  0.2616803521_dp,  0.6842860055_dp,  1.6676681182_dp,&
            3.9374424568_dp,  0.0024984514_dp,  0.0105791706_dp,  0.0352648241_dp,  0.1015481471_dp,  0.2630137361_dp,  0.6306824085_dp,  1.4430323891_dp,  3.4379149107_dp]
       aw%aw_erange_matrix(:, 13) = [0.0006511970_dp,  0.0052555004_dp,  0.0232331300_dp,  0.0805769027_dp,  0.2402766535_dp,  0.6442562263_dp,  1.6007710076_dp,&
            3.8373360452_dp,  0.0018548778_dp,  0.0087795311_dp,  0.0311892270_dp,  0.0933425139_dp,  0.2484004110_dp,  0.6078709938_dp,  1.4124537209_dp,  3.4025456698_dp]
       aw%aw_erange_matrix(:, 14) = [0.0004817072_dp,  0.0042773567_dp,  0.0202081663_dp,  0.0730638184_dp,  0.2241387198_dp,  0.6135835799_dp,  1.5489149535_dp,&
            3.7591686639_dp,  0.0014161889_dp,  0.0075045226_dp,  0.0281686456_dp,  0.0870986616_dp,  0.2370801484_dp,  0.5899701185_dp,  1.3882618018_dp,  3.3745190976_dp]
    case(10)
       aw%energy_range(:) = [10.0000_dp, 15.8489_dp, 25.1189_dp, 39.8107_dp, 63.0957_dp, 100.0000_dp,&
            158.4893_dp, 251.1886_dp, 398.1072_dp, 630.9573_dp, 1000.0000_dp,&
            1584.8932_dp, 2511.8864_dp, 3981.0717_dp, 6309.5734_dp]
       aw%aw_erange_matrix(:, 1) = [0.0330697624_dp,  0.1768078438_dp,  0.4464853449_dp,  0.8637994095_dp,  1.4665304201_dp,  2.3172114001_dp,  3.5172089430_dp,&
            5.2328732180_dp,  7.7617287679_dp,  11.7952949789_dp,  0.0851548885_dp,  0.2041221377_dp,  0.3387624704_dp,  0.5020218013_dp,  0.7136389791_dp,&
            1.0042283384_dp,  1.4225951252_dp,  2.0553775439_dp,  3.1013168391_dp,  5.3215604239_dp]
       aw%aw_erange_matrix(:, 2) = [0.0236720154_dp,  0.1271138180_dp,  0.3236822915_dp,  0.6344940443_dp,  1.0975870642_dp,  1.7773429154_dp,  2.7785871376_dp,&
            4.2719025722_dp,  6.5582697626_dp,  10.3263813643_dp,  0.0610159587_dp,  0.1475426219_dp,  0.2490140072_dp,  0.3787936644_dp,  0.5578985972_dp,&
            0.8187105789_dp,  1.2113333425_dp,  1.8225764417_dp,  2.8490231857_dp,  5.0376131531_dp]
       aw%aw_erange_matrix(:, 3) = [0.0167859311_dp,  0.0906140453_dp,  0.2331247893_dp,  0.4645377048_dp,  0.8225114978_dp,  1.3719280550_dp,  2.2188430810_dp,&
            3.5357560691_dp,  5.6260677965_dp,  9.1774382056_dp,  0.0433183177_dp,  0.1058666501_dp,  0.1824134653_dp,  0.2865259976_dp,  0.4400162724_dp,&
            0.6761202158_dp,  1.0455810781_dp,  1.6359664124_dp,  2.6436405908_dp,  4.8056680998_dp]
       aw%aw_erange_matrix(:, 4) = [0.0118170372_dp,  0.0641991517_dp,  0.1672645684_dp,  0.3401155680_dp,  0.6193913122_dp,  1.0690495497_dp,  1.7944040288_dp,&
            2.9682685375_dp,  4.8960358518_dp,  8.2659053716_dp,  0.0305390218_dp,  0.0756000893_dp,  0.1335956518_dp,  0.2180513278_dp,  0.3509424253_dp,&
            0.5655362558_dp,  0.9130793425_dp,  1.4827338153_dp,  2.4721207512_dp,  4.6114350814_dp]
       aw%aw_erange_matrix(:, 5) = [0.0082736336_dp,  0.0452968206_dp,  0.1198570122_dp,  0.2498047575_dp,  0.4701898609_dp,  0.8428723827_dp,  1.4710843779_dp,&
            2.5270093394_dp,  4.3178190934_dp,  7.5333932640_dp,  0.0214183290_dp,  0.0538519398_dp,  0.0981228090_dp,  0.1674591236_dp,  0.2833958314_dp,&
            0.4787566255_dp,  0.8054477304_dp,  1.3548062293_dp,  2.3266241431_dp,  4.4464016762_dp]
       aw%aw_erange_matrix(:, 6) = [0.0057695951_dp,  0.0318850374_dp,  0.0859854511_dp,  0.1845984841_dp,  0.3607477727_dp,  0.6734363414_dp,  1.2230817482_dp,&
            2.1806860083_dp,  3.8549835580_dp,  6.9382542461_dp,  0.0149667333_dp,  0.0383470288_dp,  0.0724914643_dp,  0.1300892819_dp,  0.2318018691_dp,&
            0.4098643125_dp,  0.7169870686_dp,  1.2469010615_dp,  2.2020857144_dp,  4.3050004436_dp]
       aw%aw_erange_matrix(:, 7) = [0.0040123212_dp,  0.0224292598_dp,  0.0619085873_dp,  0.1376297836_dp,  0.2803265917_dp,  0.5457959215_dp,  1.0313322018_dp,&
            1.9064099527_dp,  3.4810591957_dp,  6.4503500759_dp,  0.0104341288_dp,  0.0273555299_dp,  0.0540240668_dp,  0.1023930872_dp,  0.1920401273_dp,&
            0.3546405619_dp,  0.6437134573_dp,  1.1553442821_dp,  2.0949658889_dp,  4.1832825912_dp]
       aw%aw_erange_matrix(:, 8) = [0.0027856541_dp,  0.0157937535_dp,  0.0448481728_dp,  0.1037925661_dp,  0.2209854440_dp,  0.4489803369_dp,  0.8818934417_dp,&
            1.6874467002_dp,  3.1766710704_dp,  6.0475787906_dp,  0.0072661315_dp,  0.0195941905_dp,  0.0407225136_dp,  0.0817423486_dp,  0.1611258077_dp,&
            0.3100532566_dp,  0.5827362547_dp,  1.0774384067_dp,  2.0026449628_dp,  4.0782983130_dp]
       aw%aw_erange_matrix(:, 9) = [0.0019327474_dp,  0.0111525104_dp,  0.0327753346_dp,  0.0793548313_dp,  0.1769378304_dp,  0.3749891456_dp,  0.7645313671_dp,&
            1.5113972490_dp,  2.9273291880_dp,  5.7132711850_dp,  0.0050602718_dp,  0.0141268602_dp,  0.0311202095_dp,  0.0662243130_dp,  0.1368964938_dp,&
            0.2738596643_dp,  0.5318462411_dp,  1.0110650398_dp,  1.9230352100_dp,  3.9876880962_dp]
       aw%aw_erange_matrix(:, 10) = [0.0013414763_dp,  0.0079135387_dp,  0.0242305726_dp,  0.0616336183_dp,  0.1440265775_dp,  0.3180415012_dp,  0.6717744197_dp,&
            1.3691124338_dp,  2.7222502593_dp,  5.4349420610_dp,  0.0035286728_dp,  0.0102803623_dp,  0.0241590001_dp,  0.0544702109_dp,  0.1177892252_dp,&
            0.2443886234_dp,  0.4893474221_dp,  0.9545715678_dp,  1.8545048283_dp,  3.9096100475_dp]
       aw%aw_erange_matrix(:, 11) = [0.0009324064_dp,  0.0056560504_dp,  0.0181721289_dp,  0.0487120808_dp,  0.1192629003_dp,  0.2739191229_dp,  0.5980643983_dp,&
            1.2536520059_dp,  2.5531126726_dp,  5.2028197179_dp,  0.0024672655_dp,  0.0075741023_dp,  0.0190815196_dp,  0.0454982798_dp,  0.1026475177_dp,&
            0.2203452052_dp,  0.4538671149_dp,  0.9065751913_dp,  1.7956644412_dp,  3.8424989882_dp]
       aw%aw_erange_matrix(:, 12) = [0.0006497444_dp,  0.0040833478_dp,  0.0138630492_dp,  0.0392312547_dp,  0.1005045507_dp,  0.2395418039_dp,  0.5392569146_dp,&
            1.1597390133_dp,  2.4134857725_dp,  5.0092570960_dp,  0.0017326137_dp,  0.0056676592_dp,  0.0153513377_dp,  0.0386046831_dp,  0.0906088701_dp,&
            0.2007206044_dp,  0.4242960863_dp,  0.8659292391_dp,  1.7453482692_dp,  3.7850453570_dp]
       aw%aw_erange_matrix(:, 13) = [0.0004546223_dp,  0.0029878372_dp,  0.0107866942_dp,  0.0322343304_dp,  0.0862209138_dp,  0.2126637392_dp,  0.4922674783_dp,&
            1.0833747559_dp,  2.2984274495_dp,  4.8483158018_dp,  0.0012247077_dp,  0.0043218651_dp,  0.0125921514_dp,  0.0332845904_dp,  0.0810286101_dp,&
            0.1847320073_dp,  0.3997482877_dp,  0.8317002528_dp,  1.7026006523_dp,  3.7361789668_dp]
       aw%aw_erange_matrix(:, 14) = [0.0003200556_dp,  0.0022246315_dp,  0.0085816023_dp,  0.0270450051_dp,  0.0753063709_dp,  0.1916196153_dp,  0.4547477247_dp,&
            1.0214393206_dp,  2.2039990618_dp,  4.7151828791_dp,  0.0008740345_dp,  0.0033691497_dp,  0.0105393561_dp,  0.0291696470_dp,  0.0734134385_dp,&
            0.1717543874_dp,  0.3794887739_dp,  0.8030879371_dp,  1.6665845213_dp,  3.6949629981_dp]
       aw%aw_erange_matrix(:, 15) = [0.0002273927_dp,  0.0016930353_dp,  0.0069960068_dp,  0.0231854324_dp,  0.0669605830_dp,  0.1751716668_dp,  0.4249074065_dp,&
            0.9714990965_dp,  2.1270679156_dp,  4.6059719000_dp,  0.0006324539_dp,  0.0026928112_dp,  0.0090069991_dp,  0.0259895933_dp,  0.0673853109_dp,&
            0.1612907871_dp,  0.3629137459_dp,  0.7794159919_dp,  1.6365795128_dp,  3.6605913731_dp]
       aw%aw_erange_matrix(:, 16) = [0.0001638280_dp,  0.0013235929_dp,  0.0058560160_dp,  0.0203202808_dp,  0.0606089867_dp,  0.1624111567_dp,  0.4014057161_dp,&
            0.9317006239_dp,  2.0652164939_dp,  4.5176548597_dp,  0.0004668065_dp,  0.0022124398_dp,  0.0078651680_dp,  0.0235476809_dp,  0.0626594967_dp,&
            0.1529569671_dp,  0.3495462844_dp,  0.7601415052_dp,  1.6120023672_dp,  3.6324120602_dp]
    case(12)
       aw%energy_range(:) = [10.0000_dp, 15.8489_dp, 25.1189_dp, 39.8107_dp, 63.0957_dp, 100.0000_dp,&
            158.4893_dp, 251.1886_dp, 398.1072_dp, 630.9573_dp, 1000.0000_dp, 1584.8932_dp, 2511.8864_dp,&
            3981.0717_dp, 6309.5734_dp, 10000.0000_dp, 15848.9319_dp, 25118.8643_dp]
       aw%aw_erange_matrix(:, 1) = [0.0276731293_dp,  0.1472866131_dp,  0.3687581752_dp,  0.7039332404_dp,  1.1723654412_dp,  1.8045683155_dp,  2.6469524305_dp,&
            3.7693192226_dp,  5.2773524399_dp,  7.3381749736_dp,  10.2488461667_dp,  14.7099922874_dp,  0.0711846766_dp,  0.1690895322_dp,  0.2757926388_dp,&
            0.3977636094_dp,  0.5441057435_dp,  0.7279147631_dp,  0.9682939485_dp,  1.2936786640_dp,  1.7493131133_dp,  2.4187704026_dp,  3.5024366141_dp,  5.7855524080_dp]
       aw%aw_erange_matrix(:, 2) = [0.0197744403_dp,  0.1055540607_dp,  0.2657286900_dp,  0.5115778444_dp,  0.8622848926_dp,  1.3486298025_dp,  2.0182853816_dp,&
            2.9432989908_dp,  4.2321628138_dp,  6.0546789845_dp,  8.7089828257_dp,  12.8889838191_dp,  0.0509006255_dp,  0.1216171100_dp,  0.2005723407_dp,&
            0.2942623895_dp,  0.4121942175_dp,  0.5683208329_dp,  0.7828128482_dp,  1.0849323300_dp,  1.5201863096_dp,  2.1715313996_dp,  3.2362170757_dp,  5.4827819151_dp]
       aw%aw_erange_matrix(:, 3) = [0.0139914438_dp,  0.0749462508_dp,  0.1899352964_dp,  0.3694953283_dp,  0.6321343467_dp,  1.0083502424_dp,  1.5460638227_dp,&
            2.3179957425_dp,  3.4336834940_dp,  5.0643777475_dp,  7.5093175535_dp,  11.4582109486_dp,  0.0360435732_dp,  0.0867256318_dp,  0.1449645046_dp,&
            0.2171731804_dp,  0.3130978055_dp,  0.4472033444_dp,  0.6401597909_dp,  0.9215937590_dp,  1.3374215538_dp,  1.9708906690_dp,  3.0177535019_dp,  5.2339591533_dp]
       aw%aw_erange_matrix(:, 4) = [0.0098233039_dp,  0.0528381746_dp,  0.1349852452_dp,  0.2659580878_dp,  0.4633451027_dp,  0.7567923091_dp,  1.1933915052_dp,&
            1.8450943819_dp,  2.8211687374_dp,  4.2936117871_dp,  6.5630286139_dp,  10.3169372291_dp,  0.0253298174_dp,  0.0614579856_dp,  0.1044042198_dp,&
            0.1604027682_dp,  0.2392238170_dp,  0.3554023577_dp,  0.5295956187_dp,  0.7916439851_dp,  1.1882714021_dp,  1.8038125350_dp,  2.8336933495_dp,  5.0241742006_dp]
       aw%aw_erange_matrix(:, 5) = [0.0068554341_dp,  0.0370564464_dp,  0.0955849405_dp,  0.1912568606_dp,  0.3405607339_dp,  0.5717808682_dp,  0.9302915137_dp,&
            1.4862655552_dp,  2.3478924769_dp,  3.6875108667_dp,  5.8073060489_dp,  9.3940946176_dp,  0.0176966619_dp,  0.0433653612_dp,  0.0751114198_dp,&
            0.1189110538_dp,  0.1843254395_dp,  0.2855556571_dp,  0.4429546072_dp,  0.6866338193_dp,  1.0644429943_dp,  1.6622940695_dp,  2.6760843207_dp,  4.8445700066_dp]
       aw%aw_erange_matrix(:, 6) = [0.0047621174_dp,  0.0258922313_dp,  0.0675671842_dp,  0.1377372356_dp,  0.2516682757_dp,  0.4358989283_dp,  0.7335111111_dp,&
            1.2123305473_dp,  1.9789993531_dp,  3.2058934683_dp,  5.1968506439_dp,  8.6390361850_dp,  0.0123090460_dp,  0.0305206067_dp,  0.0541046151_dp,&
            0.0887139624_dp,  0.1434876371_dp,  0.2320235826_dp,  0.3742832810_dp,  0.6007236316_dp,  0.9604392059_dp,  1.5411563269_dp,  2.5398045385_dp,  4.6893655829_dp]
       aw%aw_erange_matrix(:, 7) = [0.0032965068_dp,  0.0180490022_dp,  0.0477641980_dp,  0.0995665181_dp,  0.1874300857_dp,  0.3359256681_dp,  0.5855771225_dp,&
            1.0016134105_dp,  1.6888502670_dp,  2.8194249642_dp,  4.6987721274_dp,  8.0151195661_dp,  0.0085339378_dp,  0.0214595543_dp,  0.0391102006_dp,&
            0.0667625447_dp,  0.1129751216_dp,  0.1906296580_dp,  0.3193090639_dp,  0.5298124247_dp,  0.8724391547_dp,  1.4368081150_dp,  2.4212834056_dp,  4.5544745698_dp]
       aw%aw_erange_matrix(:, 8) = [0.0022762644_dp,  0.0125677039_dp,  0.0338269925_dp,  0.0724068993_dp,  0.1409732198_dp,  0.2620656724_dp,  0.4736158009_dp,&
            0.8381994746_dp,  1.4586379799_dp,  2.5065758758_dp,  4.2889044537_dp,  7.4953768133_dp,  0.0059035786_dp,  0.0150972863_dp,  0.0284353503_dp,&
            0.0507789773_dp,  0.0900226433_dp,  0.1583351587_dp,  0.2749467359_dp,  0.4709220586_dp,  0.7976398186_dp,  1.3465944433_dp,  2.3178650924_dp,  4.4368325878_dp]
       aw%aw_erange_matrix(:, 9) = [0.0015692112_dp,  0.0087519455_dp,  0.0240448765_dp,  0.0530894602_dp,  0.1072765322_dp,  0.2071777069_dp,  0.3882400731_dp,&
            0.7104394593_dp,  1.2745126394_dp,  2.2513920738_dp,  3.9492406835_dp,  7.0596133347_dp,  0.0040787492_dp,  0.0106444112_dp,  0.0208412874_dp,&
            0.0390928325_dp,  0.0726187626_dp,  0.1329358715_dp,  0.2389268087_dp,  0.4218108128_dp,  0.7338882151_dp,  1.2684530084_dp,  2.2274784048_dp,  4.3340445584_dp]
       aw%aw_erange_matrix(:, 10) = [0.0010808702_dp,  0.0061030127_dp,  0.0171885376_dp,  0.0393301517_dp,  0.0827196472_dp,  0.1661050396_dp,  0.3226299268_dp,&
            0.6097857548_dp,  1.1261984299_dp,  2.0419146289_dp,  3.6661682448_dp,  6.6924544132_dp,  0.0028168777_dp,  0.0075342598_dp,  0.0154330800_dp,&
            0.0304965669_dp,  0.0593131883_dp,  0.1128213789_dp,  0.2095432740_dp,  0.3807404710_dp,  0.6794745486_dp,  1.2007255924_dp,  2.1484537810_dp,  4.2441834572_dp]
       aw%aw_erange_matrix(:, 11) = [0.0007444375_dp,  0.0042674833_dp,  0.0123834195_dp,  0.0294995195_dp,  0.0647151559_dp,  0.1351401380_dp,  0.2718256132_dp,&
            0.5299318244_dp,  1.0060016190_dp,  1.8690725994_dp,  3.4292526441_dp,  6.3820141516_dp,  0.0019463550_dp,  0.0053641267_dp,  0.0115705684_dp,&
            0.0241262403_dp,  0.0490612250_dp,  0.0968006464_dp,  0.1854888062_dp,  0.3463341687_dp,  0.6330104563_dp,  1.1420475986_dp,  2.0794138297_dp,  4.1656658317_dp]
       aw%aw_erange_matrix(:, 12) = [0.0005130705_dp,  0.0029969040_dp,  0.0090121194_dp,  0.0224440417_dp,  0.0514233406_dp,  0.1116185588_dp,  0.2322060971_dp,&
            0.4661910446_dp,  0.9081058970_dp,  1.7259054260_dp,  3.2303898959_dp,  6.1189723302_dp,  0.0013467807_dp,  0.0038499840_dp,  0.0087994505_dp,&
            0.0193674948_dp,  0.0411066473_dp,  0.0839811159_dp,  0.1657464888_dp,  0.3174854170_dp,  0.5933519921_dp,  1.0912778783_dp,  2.0192022809_dp,  4.0971686784_dp]
       aw%aw_erange_matrix(:, 13) = [0.0003541405_dp,  0.0021177005_dp,  0.0066414300_dp,  0.0173517266_dp,  0.0415390587_dp,  0.0936217813_dp,  0.2011144128_dp,&
            0.4150562059_dp,  0.8280777004_dp,  1.6070223727_dp,  3.0632229700_dp,  5.8959440901_dp,  0.0009342370_dp,  0.0027926556_dp,  0.0067994457_dp,&
            0.0157843491_dp,  0.0348978173_dp,  0.0736865554_dp,  0.1495177964_dp,  0.2932983996_dp,  0.5595503883_dp,  1.0474552726_dp,  1.9668404301_dp,  4.0375772596_dp]
       aw%aw_erange_matrix(:, 14) = [0.0002450328_dp,  0.0015091465_dp,  0.0049687254_dp,  0.0136527233_dp,  0.0341354342_dp,  0.0797616748_dp,  0.1765865877_dp,&
            0.3738789013_dp,  0.7625021866_dp,  1.5081991527_dp,  2.9226994333_dp,  5.7069942693_dp,  0.0006505188_dp,  0.0020530149_dp,  0.0053457983_dp,&
            0.0130665496_dp,  0.0300282048_dp,  0.0653990603_dp,  0.1361698664_dp,  0.2730416562_dp,  0.5308111938_dp,  1.0097601352_dp,  1.9214869313_dp,  3.9859370380_dp]
       aw%aw_erange_matrix(:, 15) = [0.0001701376_dp,  0.0010875693_dp,  0.0037833625_dp,  0.0109476101_dp,  0.0285522959_dp,  0.0690278231_dp,  0.1571594905_dp,&
            0.3406410667_dp,  0.7087222805_dp,  1.4260874074_dp,  2.8047517783_dp,  5.5472853843_dp,  0.0004554191_dp,  0.0015341915_dp,  0.0042812667_dp,&
            0.0109920399_dp,  0.0261952113_dp,  0.0587184401_dp,  0.1251974104_dp,  0.2561136781_dp,  0.5064635464_dp,  0.9774854852_dp,  1.8824077828_dp,  3.9414177238_dp]
       aw%aw_erange_matrix(:, 16) = [0.0001187248_dp,  0.0007951840_dp,  0.0029392994_dp,  0.0089567848_dp,  0.0243186289_dp,  0.0606825937_dp,  0.1417400945_dp,&
            0.3138020680_dp,  0.6646691425_dp,  1.3580347536_dp,  2.7061116698_dp,  5.4128869315_dp,  0.0003212717_dp,  0.0011689985_dp,  0.0034961439_dp,&
            0.0094011682_dp,  0.0231726278_dp,  0.0533353063_dp,  0.1161983613_dp,  0.2420232525_dp,  0.4859467156_dp,  0.9500293558_dp,  1.8489717513_dp,  3.9033068726_dp]
       aw%aw_erange_matrix(:, 17) = [0.0000834432_dp,  0.0005922059_dp,  0.0023356082_dp,  0.0074845398_dp,  0.0210976147_dp,  0.0541859891_dp,  0.1295099966_dp,&
            0.2921841803_dp,  0.6287314551_dp,  1.3019414676_dp,  2.6241566957_dp,  5.3006121791_dp,  0.0002290980_dp,  0.0009110177_dp,  0.0029139485_dp,&
            0.0081785934_dp,  0.0207908695_dp,  0.0490106127_dp,  0.1088539140_dp,  0.2303716185_dp,  0.4687957030_dp,  0.9268838333_dp,  1.8206410245_dp,  3.8709984304_dp]
       aw%aw_erange_matrix(:, 18) = [0.0000592658_dp,  0.0004512821_dp,  0.0019025862_dp,  0.0063934448_dp,  0.0186470165_dp,  0.0491400281_dp,  0.1198519988_dp,&
            0.2748801818_dp,  0.5996436460_dp,  1.2561293960_dp,  2.5567595977_dp,  5.2078458291_dp,  0.0001658923_dp,  0.0007282576_dp,  0.0024811714_dp,&
            0.0072403151_dp,  0.0189219666_dp,  0.0455586869_dp,  0.1029100002_dp,  0.2208331616_dp,  0.4546218887_dp,  0.9076157149_dp,  1.7969509606_dp,  3.8439691363_dp]
       aw%aw_erange_matrix(:, 19) = [0.0000427539_dp,  0.0003536020_dp,  0.0015919883_dp,  0.0055863488_dp,  0.0167910186_dp,  0.0452482974_dp,  0.1122951892_dp,&
            0.2611820892_dp,  0.5763969601_dp,  1.2192345007_dp,  2.5021613308_dp,  5.1323956397_dp,  0.0001227292_dp,  0.0005986865_dp,  0.0021600744_dp,&
            0.0065244896_dp,  0.0174683766_dp,  0.0428338303_dp,  0.0981617995_dp,  0.2131383094_dp,  0.4430948941_dp,  0.8918475783_dp,  1.7774899045_dp,  3.8217552341_dp]
    case(14)
       aw%energy_range(:) = [10.0000_dp, 15.8489_dp, 25.1189_dp, 39.8107_dp, 63.0957_dp, 100.0000_dp,&
            158.4893_dp, 251.1886_dp, 398.1072_dp, 630.9573_dp, 1000.0000_dp, 1584.8932_dp, 2511.8864_dp,&
            3981.0717_dp, 6309.5734_dp, 10000.0000_dp, 15848.9319_dp, 25118.8643_dp, 39810.7171_dp, 63095.7344_dp, 100000.0000_dp]
       aw%aw_erange_matrix(:, 1) = [0.0237936782_dp,  0.1262969955_dp,  0.3146137237_dp,  0.5959508382_dp,  0.9817872624_dp,  1.4893066488_dp,  2.1434983519_dp,&
            2.9801772260_dp,  4.0503393145_dp,  5.4269061190_dp,  7.2167997449_dp,  9.5870429977_dp,  12.8357453001_dp,  17.6753249521_dp,  0.0611672950_dp,&
            0.1445085894_dp,  0.2333087911_dp,  0.3312554161_dp,  0.4432468756_dp,  0.5759073007_dp,  0.7383747432_dp,  0.9433758083_dp,  1.2089173155_dp,&
            1.5616542882_dp,  2.0450063723_dp,  2.7419835160_dp,  3.8566734312_dp,  6.1981791899_dp]
       aw%aw_erange_matrix(:, 2) = [0.0169816642_dp,  0.0903275498_dp,  0.2258900214_dp,  0.4304310095_dp,  0.7149888745_dp,  1.0965252424_dp,  1.6002842943_dp,&
            2.2629949541_dp,  3.1372091996_dp,  4.2978145261_dp,  5.8537407167_dp,  7.9734653507_dp,  10.9544068995_dp,  15.4993236149_dp,  0.0436764843_dp,&
            0.1036202428_dp,  0.1686129519_dp,  0.2422818311_dp,  0.3296217363_dp,  0.4376054730_dp,  0.5759638625_dp,  0.7581137515_dp,  1.0026174373_dp,&
            1.3363743540_dp,  1.8026148317_dp,  2.4832977293_dp,  3.5786612501_dp,  5.8790633067_dp]
       aw%aw_erange_matrix(:, 3) = [0.0119972526_dp,  0.0639735360_dp,  0.1607328068_dp,  0.3084839990_dp,  0.5176460378_dp,  0.8046770125_dp,  1.1946063310_dp,&
            1.7242874128_dp,  2.4466652263_dp,  3.4372421876_dp,  4.8058180424_dp,  6.7218376735_dp,  9.4826338752_dp,  13.7842321234_dp,  0.0308743065_dp,&
            0.0736138461_dp,  0.1209192548_dp,  0.1762894116_dp,  0.2447353626_dp,  0.3334443879_dp,  0.4524878835_dp,  0.6155970910_dp,  0.8415625308_dp,&
            1.1575185632_dp,  1.6069457576_dp,  2.2716047438_dp,  3.3492956544_dp,  5.6157341385_dp]
       aw%aw_erange_matrix(:, 4) = [0.0084076681_dp,  0.0449636994_dp,  0.1135988127_dp,  0.2199152900_dp,  0.3735947871_dp,  0.5903443786_dp,  0.8944811045_dp,&
            1.3221417879_dp,  1.9255697207_dp,  2.7798165137_dp,  3.9948200990_dp,  5.7408169122_dp,  8.3156700660_dp,  12.4109325959_dp,  0.0216511589_dp,&
            0.0519267386_dp,  0.0862556649_dp,  0.1279584309_dp,  0.1819750580_dp,  0.2555243887_dp,  0.3587034665_dp,  0.5052195178_dp,  0.7139586455_dp,&
            1.0124827029_dp,  1.4449804822_dp,  2.0936661885_dp,  3.1548999120_dp,  5.3926769614_dp]
       aw%aw_erange_matrix(:, 5) = [0.0058545394_dp,  0.0314169341_dp,  0.0798961491_dp,  0.1562801892_dp,  0.2694516528_dp,  0.4341668678_dp,  0.6735857092_dp,&
            1.0224344459_dp,  1.5314708133_dp,  2.2746179259_dp,  3.3615119894_dp,  4.9630904952_dp,  7.3782044923_dp,  11.2956335062_dp,  0.0150881610_dp,&
            0.0364360730_dp,  0.0613313227_dp,  0.0928838706_dp,  0.1358762369_dp,  0.1973630960_dp,  0.2871909934_dp,  0.4188526951_dp,  0.6113456809_dp,&
            0.8928419449_dp,  1.3085276567_dp,  1.9414929173_dp,  2.9873859616_dp,  5.2007139221_dp]
       aw%aw_erange_matrix(:, 6) = [0.0040562245_dp,  0.0218539742_dp,  0.0560102454_dp,  0.1109232899_dp,  0.1946568562_dp,  0.3208703328_dp,  0.5112371968_dp,&
            0.7986245426_dp,  1.2318540903_dp,  1.8833267668_dp,  2.8620729079_dp,  4.3396108414_dp,  6.6160518775_dp,  10.3786630825_dp,  0.0104630611_dp,&
            0.0254712518_dp,  0.0435516941_dp,  0.0675857103_dp,  0.1021182426_dp,  0.1538700771_dp,  0.2322649417_dp,  0.3505246702_dp,  0.5277975034_dp,&
            0.7929346940_dp,  1.1922412386_dp,  1.8099472487_dp,  2.8415520058_dp,  5.0338617688_dp]
       aw%aw_erange_matrix(:, 7) = [0.0027992651_dp,  0.0151526724_dp,  0.0391950273_dp,  0.0787783513_dp,  0.1411563608_dp,  0.2388040142_dp,  0.3917244612_dp,&
            0.6307171506_dp,  1.0024644268_dp,  1.5776184281_dp,  2.4643581015_dp,  3.8346158305_dp,  5.9898844950_dp,  9.6168480481_dp,  0.0072283419_dp,&
            0.0177636198_dp,  0.0309408098_dp,  0.0494041369_dp,  0.0773938528_dp,  0.1211829834_dp,  0.1897088377_dp,  0.2959270382_dp,  0.4591303320_dp,&
            0.7088154109_dp,  1.0924186101_dp,  1.6954725243_dp,  2.7137745874_dp,  4.8878976412_dp]
       aw%aw_erange_matrix(:, 8) = [0.0019260163_dp,  0.0104833755_dp,  0.0274163538_dp,  0.0560830704_dp,  0.1029553545_dp,  0.1792971830_dp,  0.3033947804_dp,&
            0.5039497133_dp,  0.8254537344_dp,  1.3366813377_dp,  2.1447535633_dp,  3.4218231610_dp,  5.4707784720_dp,  8.9783978369_dp,  0.0049795399_dp,&
            0.0123739468_dp,  0.0220303943_dp,  0.0363538024_dp,  0.0592326432_dp,  0.0964440642_dp,  0.1564474990_dp,  0.2519410443_dp,  0.4023081489_dp,&
            0.6375997771_dp,  1.0063349415_dp,  1.5954396637_dp,  2.6013624022_dp,  4.7596612523_dp]
       aw%aw_erange_matrix(:, 9) = [0.0013222495_dp,  0.0072441400_dp,  0.0191950478_dp,  0.0400934258_dp,  0.0756713875_dp,  0.1360117507_dp,  0.2377411946_dp,&
            0.4075418317_dp,  0.6877504424_dp,  1.1452008897_dp,  1.8857902883_dp,  3.0816948930_dp,  5.0371415973_dp,  8.4394873058_dp,  0.0034234922_dp,&
            0.0086197096_dp,  0.0157487515_dp,  0.0269783975_dp,  0.0458232460_dp,  0.0775717368_dp,  0.1302433050_dp,  0.2162717795_dp,  0.3550545602_dp,&
            0.5770852050_dp,  0.9318838821_dp,  1.5078058926_dp,  2.5022210559_dp,  4.6466884816_dp]
       aw%aw_erange_matrix(:, 10) = [0.0009063749_dp,  0.0050043736_dp,  0.0134702129_dp,  0.0288347782_dp,  0.0561436492_dp,  0.1043728737_dp,  0.1886107108_dp,&
            0.3336543186_dp,  0.5797709752_dp,  0.9918421828_dp,  1.6744195586_dp,  2.7995206256_dp,  4.6726106098_dp,  7.9819594252_dp,  0.0023507281_dp,&
            0.0060117822_dp,  0.0113238920_dp,  0.0202237097_dp,  0.0358553075_dp,  0.0630595051_dp,  0.1094556677_dp,  0.1871955718_dp,  0.3156131499_dp,&
            0.5255332814_dp,  0.8673791226_dp,  1.4309274629_dp,  2.4146664138_dp,  4.5470018340_dp]
       aw%aw_erange_matrix(:, 11) = [0.0006207558_dp,  0.0034593750_dp,  0.0094888452_dp,  0.0209005466_dp,  0.0421152117_dp,  0.0811052514_dp,  0.1515714938_dp,&
            0.2765844244_dp,  0.4944571337_dp,  0.8681492042_dp,  1.5007904059_dp,  2.5640827408_dp,  4.3646060695_dp,  7.5917613733_dp,  0.0016132125_dp,&
            0.0042033180_dp,  0.0082051681_dp,  0.0153343742_dp,  0.0283888063_dp,  0.0518157294_dp,  0.0928668378_dp,  0.1633931970_dp,  0.2825994311_dp,&
            0.4815408039_dp,  0.8114386960_dp,  1.3634496528_dp,  2.3373123760_dp,  4.4589805898_dp]
       aw%aw_erange_matrix(:, 12) = [0.0004250257_dp,  0.0023953290_dp,  0.0067207167_dp,  0.0152965867_dp,  0.0319860564_dp,  0.0638742350_dp,  0.1234325709_dp,&
            0.2321694171_dp,  0.4265771556_dp,  0.7677581817_dp,  1.3573866615_dp,  2.3667230075_dp,  4.1033285208_dp,  7.2578630071_dp,  0.0011072202_dp,&
            0.0029503441_dp,  0.0060027300_dp,  0.0117733548_dp,  0.0227515996_dp,  0.0430443066_dp,  0.0795610664_dp,  0.1438398396_dp,  0.2549060018_dp,&
            0.4439586001_dp,  0.7629125934_dp,  1.3042372856_dp,  2.2689984236_dp,  4.3812747921_dp]
       aw%aw_erange_matrix(:, 13) = [0.0002911017_dp,  0.0016631492_dp,  0.0047946445_dp,  0.0113241508_dp,  0.0246263552_dp,  0.0510161558_dp,  0.1018875389_dp,&
            0.1973480201_dp,  0.3722122939_dp,  0.6858159813_dp,  1.2383805242_dp,  2.2006300353_dp,  3.8809743266_dp,  6.9713912766_dp,  0.0007605525_dp,&
            0.0020822672_dp,  0.0044420437_dp,  0.0091605927_dp,  0.0184618417_dp,  0.0361581540_dp,  0.0688392550_dp,  0.1277263816_dp,  0.2316313965_dp,&
            0.4118257407_dp,  0.7208180838_dp,  1.2523064711_dp,  2.2087140692_dp,  4.3127157224_dp]
       aw%aw_erange_matrix(:, 14) = [0.0001995700_dp,  0.0011595295_dp,  0.0034524399_dp,  0.0084955264_dp,  0.0192432510_dp,  0.0413511687_dp,  0.0852771242_dp,&
            0.1698836547_dp,  0.3284577878_dp,  0.6186813704_dp,  1.1393608161_dp,  2.0606221816_dp,  3.6915967870_dp,  6.7255839033_dp,  0.0005232710_dp,&
            0.0014805383_dp,  0.0033311544_dp,  0.0072290816_dp,  0.0151751660_dp,  0.0307257083_dp,  0.0601730847_dp,  0.1144276403_dp,  0.2120682416_dp,&
            0.3843826899_dp,  0.6843807450_dp,  1.2068922048_dp,  2.1556860505_dp,  4.2524123622_dp]
       aw%aw_erange_matrix(:, 15) = [0.0001370440_dp,  0.0008130158_dp,  0.0025145919_dp,  0.0064696769_dp,  0.0152763632_dp,  0.0340306034_dp,  0.0723813392_dp,&
            0.1480920563_dp,  0.2930715707_dp,  0.5634732104_dp,  1.0567504907_dp,  1.9423970762_dp,  3.5301492307_dp,  6.5145924693_dp,  0.0003609177_dp,&
            0.0010628352_dp,  0.0025356684_dp,  0.0057896189_dp,  0.0126401225_dp,  0.0264193690_dp,  0.0531457819_dp,  0.1034316774_dp,  0.1956149122_dp,&
            0.3609571463_dp,  0.6528859764_dp,  1.1672606510_dp,  2.1091543249_dp,  4.1994938535_dp]
       aw%aw_erange_matrix(:, 16) = [0.0000943292_dp,  0.0005743617_dp,  0.0018567491_dp,  0.0050088333_dp,  0.0123298613_dp,  0.0284433849_dp,  0.0623023416_dp,&
            0.1307057577_dp,  0.2643293512_dp,  0.5179294356_dp,  0.9876865837_dp,  1.8424511225_dp,  3.3924602893_dp,  6.3335236354_dp,  0.0002498085_dp,&
            0.0007721791_dp,  0.0019618732_dp,  0.0047081086_dp,  0.0106724872_dp,  0.0229904052_dp,  0.0474305891_dp,  0.0943252116_dp,  0.1817720546_dp,&
            0.3409759754_dp,  0.6257092922_dp,  1.1327575630_dp,  2.0684341083_dp,  4.1531784722_dp]
       aw%aw_erange_matrix(:, 17) = [0.0000651365_dp,  0.0004097585_dp,  0.0013931604_dp,  0.0039478705_dp,  0.0101248429_dp,  0.0241506943_dp,  0.0543825199_dp,&
            0.1167789486_dp,  0.2409226560_dp,  0.4803083615_dp,  0.9299382185_dp,  1.7580291538_dp,  3.2752284213_dp,  6.1784868742_dp,  0.0001737312_dp,&
            0.0005692888_dp,  0.0015447657_dp,  0.0038896495_dp,  0.0091375731_dp,  0.0202515001_dp,  0.0427754289_dp,  0.0867830166_dp,  0.1701401832_dp,&
            0.3239747287_dp,  0.6023404762_dp,  1.1028471436_dp,  2.0329658726_dp,  4.1128293420_dp]
       aw%aw_erange_matrix(:, 18) = [0.0000451780_dp,  0.0002960620_dp,  0.0010648809_dp,  0.0031722807_dp,  0.0084646811_dp,  0.0208370642_dp,  0.0481400189_dp,&
            0.1056065372_dp,  0.2218612717_dp,  0.4492752495_dp,  0.8817787002_dp,  1.6869834288_dp,  3.1758682753_dp,  6.0464269241_dp,  0.0001216233_dp,&
            0.0004271658_dp,  0.0012393754_dp,  0.0032669191_dp,  0.0079367228_dp,  0.0180617935_dp,  0.0389867897_dp,  0.0805512332_dp,  0.1604032849_dp,&
            0.3095824040_dp,  0.5823698798_dp,  1.0770993952_dp,  2.0023024581_dp,  4.0779388476_dp]
       aw%aw_erange_matrix(:, 19) = [0.0000315328_dp,  0.0002174308_dp,  0.0008313641_dp,  0.0026022985_dp,  0.0072095763_dp,  0.0182730950_dp,  0.0432167413_dp,&
            0.0966537970_dp,  0.2063800292_dp,  0.4237806872_dp,  0.8418287528_dp,  1.6275738925_dp,  3.0922598344_dp,  5.9348141031_dp,  0.0000859396_dp,&
            0.0003272551_dp,  0.0010144705_dp,  0.0027916184_dp,  0.0069966844_dp,  0.0163137262_dp,  0.0359133252_dp,  0.0754269174_dp,  0.1523033750_dp,&
            0.2974898789_dp,  0.5654495050_dp,  1.0551430219_dp,  1.9760540876_dp,  4.0480653472_dp]
       aw%aw_erange_matrix(:, 20) = [0.0000222079_dp,  0.0001629932_dp,  0.0006645802_dp,  0.0021817704_dp,  0.0062585553_dp,  0.0162885708_dp,  0.0393399077_dp,&
            0.0895029361_dp,  0.1938660784_dp,  0.4029632862_dp,  0.8089283219_dp,  1.5783019361_dp,  3.0225368925_dp,  5.8413807575_dp,  0.0000615230_dp,&
            0.0002567640_dp,  0.0008481266_dp,  0.0024284357_dp,  0.0062618334_dp,  0.0149229402_dp,  0.0334327161_dp,  0.0712410515_dp,  0.1456187328_dp,&
            0.2874223434_dp,  0.5512586346_dp,  1.0366236919_dp,  1.9538396852_dp,  4.0227772765_dp]
       aw%aw_erange_matrix(:, 21) = [0.0000158399_dp,  0.0001252648_dp,  0.0005450424_dp,  0.0018707369_dp,  0.0055376345_dp,  0.0147550601_dp,  0.0362978083_dp,&
            0.0838207306_dp,  0.1838173647_dp,  0.3860983826_dp,  0.7820753769_dp,  1.5378394885_dp,  2.9650064681_dp,  5.7640301975_dp,  0.0000448370_dp,&
            0.0002068443_dp,  0.0007247732_dp,  0.0021511315_dp,  0.0056892147_dp,  0.0138220852_dp,  0.0314441670_dp,  0.0678498508_dp,  0.1401543526_dp,&
            0.2791293861_dp,  0.5394940065_dp,  1.0211944165_dp,  1.9352770155_dp,  4.0016416470_dp]
       aw%aw_erange_matrix(:, 22) = [0.0000115002_dp,  0.0000991173_dp,  0.0004592690_dp,  0.0016408940_dp,  0.0049930058_dp,  0.0135768119_dp,  0.0339290060_dp,&
            0.0793477695_dp,  0.1758354289_dp,  0.3726002908_dp,  0.7604460875_dp,  1.5050773582_dp,  2.9182351984_dp,  5.7009682775_dp,  0.0000334643_dp,&
            0.0001714169_dp,  0.0006333913_dp,  0.0019403728_dp,  0.0052461953_dp,  0.0129586872_dp,  0.0298673394_dp,  0.0651362091_dp,  0.1357480131_dp,&
            0.2723982760_dp,  0.5298926968_dp,  1.0085489528_dp,  1.9200248668_dp,  3.9842719598_dp]
    case(16)
       aw%energy_range(:) = [100.0_dp, 200.0_dp, 300.0_dp, 400.0_dp, 500.0_dp, 600.0_dp, 700.0_dp, 800.0_dp,&
            900.0_dp, 1000.0_dp, 2.00e+03_dp, 3.00e+03_dp, 4.00e+03_dp, 5.00e+03_dp, 6.00e+03_dp, 7.00e+03_dp, 8.00e+03_dp,&
            9.00e+03_dp, 1.00e+04_dp, 2.00e+04_dp, 3.00e+04_dp, 4.00e+04_dp, 5.00e+04_dp, 6.00e+04_dp, 7.00e+04_dp,&
            8.00e+04_dp, 1.00e+05_dp, 2.00e+05_dp, 3.00e+05_dp, 4.00e+05_dp, 5.00e+05_dp, 6.00e+05_dp, 7.00e+05_dp,&
            9.00e+05_dp, 1.00e+06_dp, 2.00e+06_dp]
       aw%aw_erange_matrix(:, 1) = [3.53350980972745024e-03_dp,  1.89307470173197823e-02_dp,  4.79913291058656280e-02_dp,  9.34063314619771184e-02_dp,  1.59900975899146841e-01_dp,&
            2.55237720565199833e-01_dp,  3.91507506232646407e-01_dp,  5.86692524556495676e-01_dp,  8.66706221259449516e-01_dp,  1.26834755965607449e+00_dp,  1.84378922435133030e+00_dp,&
            2.66770358673057162e+00_dp,  3.84969929504600383e+00_dp,  5.55989190488239782e+00_dp,  8.09609615483757494e+00_dp,  1.21489357832949842e+01_dp,  9.10308460639413716e-03_dp,&
            2.19107094627195229e-02_dp,  3.66477217677682598e-02_dp,  5.49529774439788052e-02_dp,  7.93083736311463450e-02_dp,  1.13354729405511695e-01_dp,  1.62148972334804270e-01_dp,&
            2.32492103990712351e-01_dp,  3.33595921973548981e-01_dp,  4.78268763696850785e-01_dp,  6.84867425639795080e-01_dp,  9.80851423476391093e-01_dp,  1.41057748263778815e+00_dp,&
            2.05646355120530222e+00_dp,  3.11470981048028772e+00_dp,  5.34707526399065092e+00_dp]
       aw%aw_erange_matrix(:, 2) = [2.01256609288612623e-03_dp,  1.08393025800470229e-02_dp,  2.77607372768341396e-02_dp,  5.49153548548539372e-02_dp,  9.62139067787890073e-02_dp,&
            1.58268585901329639e-01_dp,  2.51551762218636565e-01_dp,  3.91889894075743928e-01_dp,  6.02624857991079899e-01_dp,  9.17865362174561272e-01_dp,  1.38738289791511860e+00_dp,&
            2.08421352166685470e+00_dp,  3.11754692365379737e+00_dp,  4.65845614288767074e+00_dp,  7.00624402509797584e+00_dp,  1.08487716247577879e+01_dp,  5.19100503916200377e-03_dp,&
            1.26277612849116580e-02_dp,  2.15606407131872894e-02_dp,  3.33846640063611144e-02_dp,  5.02953701646382048e-02_dp,  7.55353808513282848e-02_dp,  1.13633681039463183e-01_dp,&
            1.70879885021969491e-01_dp,  2.56195443232918985e-01_dp,  3.82459276964640105e-01_dp,  5.68547819274359711e-01_dp,  8.42929673227647003e-01_dp,  1.25134422119431932e+00_dp,&
            1.87743223838423345e+00_dp,  2.91674707410050216e+00_dp,  5.12003624180075434e+00_dp]
       aw%aw_erange_matrix(:, 3) = [1.44321148556243663e-03_dp,  7.80153177980970076e-03_dp,  2.01254698329391621e-02_dp,  4.02748735892814805e-02_dp,  7.17270043310115762e-02_dp,&
            1.20437842219770330e-01_dp,  1.95922508665984957e-01_dp,  3.12738745192699397e-01_dp,  4.92726432022653815e-01_dp,  7.68398119460578344e-01_dp,  1.18803361602970625e+00_dp,&
            1.82355128165130798e+00_dp,  2.78370305964699760e+00_dp,  4.23999745998180622e+00_dp,  6.49272540108700547e+00_dp,  1.02289376129522349e+01_dp,  3.72555641699100692e-03_dp,&
            9.13029870564178482e-03_dp,  1.58171136039656351e-02_dp,  2.50466904073013037e-02_dp,  3.88327073275191448e-02_dp,  6.01510240904654450e-02_dp,  9.32081506757678202e-02_dp,&
            1.44008985109400189e-01_dp,  2.21302503955418645e-01_dp,  3.37961419891918136e-01_dp,  5.13089899907686053e-01_dp,  7.75684683956141319e-01_dp,  1.17230167874908897e+00_dp,&
            1.78743631474834719e+00_dp,  2.81663768450139029e+00_dp,  5.00548143863775064e+00_dp]
       aw%aw_erange_matrix(:, 4) = [1.13860760663766201e-03_dp,  6.17299781269263839e-03_dp,  1.60169098352160554e-02_dp,  3.23524323481128584e-02_dp,  5.83702583176669565e-02_dp,&
            9.95777894307948580e-02_dp,  1.64831557562902881e-01_dp,  2.67822190872354815e-01_dp,  4.29360268183013538e-01_dp,  6.80845171457615028e-01_dp,  1.06948381160530315e+00_dp,&
            1.66634821469290184e+00_dp,  2.57980100528938738e+00_dp,  3.98159346669642966e+00_dp,  6.17273411310204700e+00_dp,  9.83997227880138681e+00_dp,  2.94117041553301631e-03_dp,&
            7.25063797608869933e-03_dp,  1.27075060808783998e-02_dp,  2.04817457415186997e-02_dp,  3.24552336489367654e-02_dp,  5.14120457313516355e-02_dp,  8.13399962135207821e-02_dp,&
            1.28053125906785886e-01_dp,  2.00171599487160429e-01_dp,  3.10536809254130741e-01_dp,  4.78375693670618274e-01_dp,  7.33023992755198917e-01_dp,  1.12160597395424810e+00_dp,&
            1.72926445447689736e+00_dp,  2.75168129090199720e+00_dp,  4.93124071847187917e+00_dp]
       aw%aw_erange_matrix(:, 5) = [9.46859903916035537e-04_dp,  5.14616088730276196e-03_dp,  1.34185711537655374e-02_dp,  2.73192349904864268e-02_dp,  4.98287460342099059e-02_dp,&
            8.61193605778245858e-02_dp,  1.44555695695986336e-01_dp,  2.38183015880307103e-01_dp,  3.87041479853388215e-01_dp,  6.21684790258431930e-01_dp,  9.88486563062811463e-01_dp,&
            1.55783889875341841e+00_dp,  2.43776191947990073e+00_dp,  3.80015625761590004e+00_dp,  5.94658657784491584e+00_dp,  9.56369803004218078e+00_dp,  2.44721319362834096e-03_dp,&
            6.06309616212743173e-03_dp,  1.07312550216339935e-02_dp,  1.75540514820946535e-02_dp,  2.83110517763022485e-02_dp,  4.56402961035302507e-02_dp,  7.33688359935863454e-02_dp,&
            1.17168476679268843e-01_dp,  1.85554660029532470e-01_dp,  2.91329319901989403e-01_dp,  4.53792910832546526e-01_dp,  7.02522550597137641e-01_dp,  1.08507304824308415e+00_dp,&
            1.68710567859197180e+00_dp,  2.70447200538390886e+00_dp,  4.87732532908532423e+00_dp]
       aw%aw_erange_matrix(:, 6) = [8.14183837939547509e-04_dp,  4.43467955230156252e-03_dp,  1.16136389856696040e-02_dp,  2.38092630968598784e-02_dp,  4.38383776182723894e-02_dp,&
            7.66089872341315092e-02_dp,  1.30098648127646099e-01_dp,  2.16845526181672299e-01_dp,  3.56280572517574390e-01_dp,  5.78280446078727084e-01_dp,  9.28540305907498920e-01_dp,&
            1.47688490081642176e+00_dp,  2.33103078021899535e+00_dp,  3.66297548958391106e+00_dp,  5.77473392533884855e+00_dp,  9.35293868610848378e+00_dp,  2.10531973679683841e-03_dp,&
            5.23888103224154095e-03_dp,  9.35274075631195578e-03_dp,  1.54958369642747475e-02_dp,  2.53648437706326772e-02_dp,  4.14818886569631451e-02_dp,  6.75490862157180211e-02_dp,&
            1.09125281408510050e-01_dp,  1.74637111405913398e-01_dp,  2.76845040629674743e-01_dp,  4.35096147843626602e-01_dp,  6.79150611008417449e-01_dp,  1.05690673230816246e+00_dp,&
            1.65445677640878741e+00_dp,  2.66782922167630066e+00_dp,  4.83550044561988024e+00_dp]
       aw%aw_erange_matrix(:, 7) = [7.16494901298969080e-04_dp,  3.91018638216612119e-03_dp,  1.02801072464593964e-02_dp,  2.12070464670612545e-02_dp,  3.93748784721239808e-02_dp,&
            6.94755195241083950e-02_dp,  1.19170770235098020e-01_dp,  2.00585127206465486e-01_dp,  3.32649462565160037e-01_dp,  5.44678859853301556e-01_dp,  8.81798125100913932e-01_dp,&
            1.41334590605841526e+00_dp,  2.24676696687539001e+00_dp,  3.55412349045657461e+00_dp,  5.63780628889054558e+00_dp,  9.18448241122484710e+00_dp,  1.85351442545404581e-03_dp,&
            4.63039115192240627e-03_dp,  8.33054786648090156e-03_dp,  1.39590364258402114e-02_dp,  2.31433556018734582e-02_dp,  3.83107456954635550e-02_dp,  6.30622860656869233e-02_dp,&
            1.02863316936792890e-01_dp,  1.66063213575976282e-01_dp,  2.65381378593835426e-01_dp,  4.20195346923853241e-01_dp,  6.60410163061198929e-01_dp,  1.03420797660110741e+00_dp,&
            1.62804896339238914e+00_dp,  2.63813532983522236e+00_dp,  4.80162132438811451e+00_dp]
       aw%aw_erange_matrix(:, 8) = [6.41326809481740758e-04_dp,  3.50617346260536722e-03_dp,  9.25085074482776167e-03_dp,  1.91923279358940310e-02_dp,  3.59033684018405658e-02_dp,&
            6.38944507354568814e-02_dp,  1.10562852458958027e-01_dp,  1.87686236929583999e-01_dp,  3.13773709712884596e-01_dp,  5.17662494297918774e-01_dp,  8.43986668600400392e-01_dp,&
            1.36166018332979322e+00_dp,  2.17788216574139337e+00_dp,  3.46475828890199411e+00_dp,  5.52500051528785985e+00_dp,  9.04533575355827679e+00_dp,  1.65971109796032596e-03_dp,&
            4.16106326244922858e-03_dp,  7.53902014585412523e-03_dp,  1.27615706769678879e-02_dp,  2.13972122588712184e-02_dp,  3.57936419467514491e-02_dp,  5.94677802105419498e-02_dp,&
            9.78053035397581927e-02_dp,  1.59087168480228858e-01_dp,  2.55993100221750536e-01_dp,  4.07920704338065321e-01_dp,  6.44893253755295270e-01_dp,  1.01533352025257706e+00_dp,&
            1.60602200914479964e+00_dp,  2.61332768483982214e+00_dp,  4.77332649007502585e+00_dp]
       aw%aw_erange_matrix(:, 9) = [5.81552858876719573e-04_dp,  3.18458728238287577e-03_dp,  8.43009848407876017e-03_dp,  1.75811659326021198e-02_dp,  3.31156506410349299e-02_dp,&
            5.93885926005389142e-02_dp,  1.03571099986227905e-01_dp,  1.77143894018762488e-01_dp,  2.98252992060085709e-01_dp,  4.95321060119377710e-01_dp,  8.12552457633085590e-01_dp,&
            1.31848446625012738e+00_dp,  2.12009233578978984e+00_dp,  3.38951070393522613e+00_dp,  5.42973079030332695e+00_dp,  8.92755335444040199e+00_dp,  1.50556315952505627e-03_dp,&
            3.78704505063218438e-03_dp,  6.90596865384866029e-03_dp,  1.17983671303826514e-02_dp,  1.99815601654048762e-02_dp,  3.37352970754505530e-02_dp,  5.65047866336671131e-02_dp,&
            9.36063868568455248e-02_dp,  1.53259665040788029e-01_dp,  2.48106420809356160e-01_dp,  3.97557329619784516e-01_dp,  6.31734422799635764e-01_dp,  9.99268430087539294e-01_dp,&
            1.58722306022532145e+00_dp,  2.59212588251235276e+00_dp,  4.74915085228901113e+00_dp]
       aw%aw_erange_matrix(:, 10) = [5.32791449243409957e-04_dp,  2.92201407330914490e-03_dp,  7.75884196839569766e-03_dp,  1.62599925842890798e-02_dp,  3.08208827514734295e-02_dp,&
            5.56612598231712857e-02_dp,  9.77557047572212706e-02_dp,  1.68326485590083619e-01_dp,  2.85201980076353778e-01_dp,  4.76439727683499570e-01_dp,  7.85862548805269845e-01_dp,&
            1.28166964157642660e+00_dp,  2.07063073322832736e+00_dp,  3.32489911123671833e+00_dp,  5.34771226724267912e+00_dp,  8.82595237509298869e+00_dp,  1.37978885020055087e-03_dp,&
            3.48132965159881182e-03_dp,  6.38681382757585753e-03_dp,  1.10042750893626431e-02_dp,  1.88060416343505121e-02_dp,  3.20129266794980244e-02_dp,  5.40079284084068259e-02_dp,&
            9.00461102030354760e-02_dp,  1.48291387217199144e-01_dp,  2.41349478045632310e-01_dp,  3.88639236651869280e-01_dp,  6.20366715867012197e-01_dp,  9.85345131558152731e-01_dp,&
            1.57089164262038694e+00_dp,  2.57368414096267095e+00_dp,  4.72812711274626363e+00_dp]
       aw%aw_erange_matrix(:, 11) = [2.99224120879729519e-04_dp,  1.66026313565520206e-03_dp,  4.51381700033238586e-03_dp,  9.81114129695110151e-03_dp,  1.94616438373382758e-02_dp,&
            3.68864680879441853e-02_dp,  6.79045674012652006e-02_dp,  1.22204135062910973e-01_dp,  2.15692386527333746e-01_dp,  3.74167706912010345e-01_dp,  6.39029285349485887e-01_dp,&
            1.07625266047240209e+00_dp,  1.79115927400814545e+00_dp,  2.95586846091472255e+00_dp,  4.87513923191184961e+00_dp,  8.23667784479432186e+00_dp,  7.76884742069300360e-04_dp,&
            2.00656602380648846e-03_dp,  3.85231021696860266e-03_dp,  7.05187854403546612e-03_dp,  1.28051535867035465e-02_dp,  2.29904356331946634e-02_dp,  4.06223196249400550e-02_dp,&
            7.05684431621601549e-02_dp,  1.20615323824493109e-01_dp,  2.03090767371495673e-01_dp,  3.37395001084139834e-01_dp,  5.54190330170222567e-01_dp,  9.03401381371959022e-01_dp,&
            1.47399421150855536e+00_dp,  2.46379540993549329e+00_dp,  4.60293586094290941e+00_dp]
       aw%aw_erange_matrix(:, 12) = [2.13470521766618681e-04_dp,  1.19443821216121972e-03_dp,  3.30309294815460527e-03_dp,  7.36335748981275899e-03_dp,  1.50440187979848754e-02_dp,&
            2.93726979281437213e-02_dp,  5.55989091923544607e-02_dp,  1.02640726711399635e-01_dp,  1.85414326356829418e-01_dp,  3.28517321642923299e-01_dp,  5.72017178043324859e-01_dp,&
            9.80614752437028669e-01_dp,  1.65873474387425257e+00_dp,  2.77836756683095709e+00_dp,  4.64507203421799542e+00_dp,  7.94720268358463677e+00_dp,  5.55246702745094663e-04_dp,&
            1.45843838376191748e-03_dp,  2.89028275110567992e-03_dp,  5.50037429515352373e-03_dp,  1.03516891504137030e-02_dp,  1.91571445523666846e-02_dp,  3.47438857823840624e-02_dp,&
            6.17661025400090061e-02_dp,  1.07787025740297096e-01_dp,  1.84949410097452444e-01_dp,  3.12595021695347530e-01_dp,  5.21582146906248378e-01_dp,  8.62410403168432427e-01_dp,&
            1.42497657398810884e+00_dp,  2.40787031054696632e+00_dp,  4.53927021665296770e+00_dp]
       aw%aw_erange_matrix(:, 13) = [1.68012592820130133e-04_dp,  9.46540488815995548e-04_dp,  2.65389383834955736e-03_dp,  6.03435111070690922e-03_dp,  1.26044257319316653e-02_dp,&
            2.51439404415568467e-02_dp,  4.85422220818330988e-02_dp,  9.12233302740963620e-02_dp,  1.67457115943608098e-01_dp,  3.01044618457292690e-01_dp,  5.31153332379371790e-01_dp,&
            9.21602589992802423e-01_dp,  1.57617191456792005e+00_dp,  2.66672079636148762e+00_dp,  4.49933223235559598e+00_dp,  7.76286688504890332e+00_dp,  4.37651070085769004e-04_dp,&
            1.16536666410485871e-03_dp,  2.36804563911255589e-03_dp,  4.63798343442946379e-03_dp,  8.95184829397262932e-03_dp,  1.69190097024326525e-02_dp,  3.12441751584375850e-02_dp,&
            5.64366963334440863e-02_dp,  9.99033373206869579e-02_dp,  1.73650154624361980e-01_dp,  2.96961393104604898e-01_dp,  5.00806584552872414e-01_dp,  8.36059273049339358e-01_dp,&
            1.39325367922203336e+00_dp,  2.37154456299757888e+00_dp,  4.49793067920409495e+00_dp]
       aw%aw_erange_matrix(:, 14) = [1.37026975479014590e+00_dp,  2.34515676733101053e+00_dp,  4.46790660194239475e+00_dp,  1.39556140827204678e-04_dp,  7.90875404343269173e-04_dp,&
            2.24373197289569727e-03_dp,  5.18614495108164764e-03_dp,  1.10265465390968261e-02_dp,  2.23694637981097025e-02_dp,  4.38480579093586884e-02_dp,  8.35312227650857608e-02_dp,&
            1.55218699402296184e-01_dp,  2.82125295666550369e-01_dp,  5.02747768057156597e-01_dp,  8.80238193840727434e-01_dp,  1.51787502219682380e+00_dp,  2.58739721912801590e+00_dp,&
            4.39526935349846326e+00_dp,  7.63076233425439288e+00_dp,  3.63984273626696418e-04_dp,  9.80644321112650276e-04_dp,  2.03480199951820533e-03_dp,  4.07735377152630956e-03_dp,&
            8.02412457026831227e-03_dp,  1.54111678077861834e-02_dp,  2.88537422675381669e-02_dp,  5.27529528046750545e-02_dp,  9.43962899936182931e-02_dp,  1.65682189837208782e-01_dp,&
            2.85842929517656941e-01_dp,  4.85920011438755906e-01_dp,  8.17057847522464131e-01_dp]
       aw%aw_erange_matrix(:, 15) = [1.19940633777844367e-04_dp,  6.83291861808967222e-04_dp,  1.95877168414221835e-03_dp,  4.59174524873474151e-03_dp,  9.90855703502338189e-03_dp,&
            2.03808716667070837e-02_dp,  4.04466664137481044e-02_dp,  7.79018628897185372e-02_dp,  1.46181662230467829e-01_dp,  2.68042251689520117e-01_dp,  4.81450674354238051e-01_dp,&
            8.49025951183791872e-01_dp,  1.47363870321694668e+00_dp,  2.52691885693640472e+00_dp,  4.31562663815705783e+00_dp,  7.52937595194574971e+00_dp,  3.13174089302581220e-04_dp,&
            8.52574377712036276e-04_dp,  1.80131247687566765e-03_dp,  3.67844340685374513e-03_dp,  7.35389540699760667e-03_dp,  1.43079355017790150e-02_dp,  2.70861317219600745e-02_dp,&
            5.00039451671855859e-02_dp,  9.02531713884734688e-02_dp,  1.59643928480476205e-01_dp,  2.77362092792709347e-01_dp,  4.74499463661941734e-01_dp,  8.02409594992836150e-01_dp,&
            1.35248644425788744e+00_dp,  2.32469837012121960e+00_dp,  4.44463201570879818e+00_dp]
       aw%aw_erange_matrix(:, 16) = [1.05537307465743284e-04_dp,  6.04114890345993001e-04_dp,  1.74808087768887855e-03_dp,  4.14892351339478469e-03_dp,  9.06777482852746779e-03_dp,&
            1.88708736505370164e-02_dp,  3.78405483771319731e-02_dp,  7.35534743950406605e-02_dp,  1.39149970172078974e-01_dp,  2.57012619692873145e-01_dp,  4.64673706380170914e-01_dp,&
            8.24310774687844305e-01_dp,  1.43845177788936329e+00_dp,  2.47862798314855626e+00_dp,  4.25183840359052656e+00_dp,  7.44799055279019750e+00_dp,  2.75845624421119831e-04_dp,&
            7.58059696898161304e-04_dp,  1.62738916170547445e-03_dp,  3.37735747437889998e-03_dp,  6.84165302145215218e-03_dp,  1.34560408443224477e-02_dp,  2.57094629782190394e-02_dp,&
            4.78470046371534674e-02_dp,  8.69809853452546689e-02_dp,  1.54846909473789957e-01_dp,  2.70589095153322690e-01_dp,  4.65336333654371825e-01_dp,  7.90610641841838824e-01_dp,&
            1.33811980156249577e+00_dp,  2.30814335154912431e+00_dp,  4.42579981986181270e+00_dp]
       aw%aw_erange_matrix(:, 17) = [9.44773691119611941e-05_dp,  5.43193371816229689e-04_dp,  1.58529076001570363e-03_dp,  3.80445119019962551e-03_dp,  8.40830888946824360e-03_dp,&
            1.76766622119811992e-02_dp,  3.57636322027086104e-02_dp,  7.00642034135144226e-02_dp,  1.33472875053466083e-01_dp,  2.48058977084102017e-01_dp,  4.50988013467829851e-01_dp,&
            8.04062319500161493e-01_dp,  1.40951520734543023e+00_dp,  2.43878826484893274e+00_dp,  4.19907917115933049e+00_dp,  7.38055138869627925e+00_dp,  2.47168812970795910e-04_dp,&
            6.85157250381651867e-04_dp,  1.49210873033567303e-03_dp,  3.14045446537675015e-03_dp,  6.43430877951877301e-03_dp,  1.27727320392762647e-02_dp,  2.45972718527957372e-02_dp,&
            4.60935944652694080e-02_dp,  8.43063411502943394e-02_dp,  1.50906614412536083e-01_dp,  2.65001268910812326e-01_dp,  4.57747306434051426e-01_dp,  7.80806606488453681e-01_dp,&
            1.32615265104682423e+00_dp,  2.29433424518916640e+00_dp,  4.41009226224503337e+00_dp]
       aw%aw_erange_matrix(:, 18) = [8.56966014984952726e-05_dp,  4.94737232830936907e-04_dp,  1.45531491856781760e-03_dp,  3.52771939786528143e-03_dp,  7.87462438036366438e-03_dp,&
            1.67031895938322059e-02_dp,  3.40593424378246093e-02_dp,  6.71839347162209166e-02_dp,  1.28761881603225270e-01_dp,  2.40594118418479552e-01_dp,  4.39530255088571464e-01_dp,&
            7.87047459730448873e-01_dp,  1.38512122114885150e+00_dp,  2.40511115764277461e+00_dp,  4.15438397780310265e+00_dp,  7.32332950147354378e+00_dp,  2.24392060575159135e-04_dp,&
            6.27041706761741094e-04_dp,  1.38344066171848458e-03_dp,  2.94819848362962850e-03_dp,  6.10068538780172715e-03_dp,  1.22089286671955115e-02_dp,  2.36739223810270155e-02_dp,&
            4.46301312678028944e-02_dp,  8.20634642574863266e-02_dp,  1.47588508027176962e-01_dp,  2.60278104396839072e-01_dp,  4.51311362761884882e-01_dp,  7.72468954882403636e-01_dp,&
            1.31595387099549921e+00_dp,  2.28255173264817390e+00_dp,  4.39669057356122384e+00_dp]
       aw%aw_erange_matrix(:, 19) = [7.85427260255526331e-05_dp,  4.55192461786559175e-04_dp,  1.34886745971785765e-03_dp,  3.29980061701751507e-03_dp,  7.43216147935275330e-03_dp,&
            1.58908890315543790e-02_dp,  3.26288715086645031e-02_dp,  6.47538075936886781e-02_dp,  1.24768765970660347e-01_dp,  2.34240816007087072e-01_dp,  4.29743078508864784e-01_dp,&
            7.72466595117579313e-01_dp,  1.36415811068297010e+00_dp,  2.37610189373330272e+00_dp,  4.11581100024765245e+00_dp,  7.27387779175559945e+00_dp,  2.05828296233996665e-04_dp,&
            5.79516303470468453e-04_dp,  1.29394653547505538e-03_dp,  2.78840262668174875e-03_dp,  5.82114329324756683e-03_dp,  1.17334510546047814e-02_dp,  2.28910200924036800e-02_dp,&
            4.33834862866021784e-02_dp,  8.01450248378290792e-02_dp,  1.44739972641330816e-01_dp,  2.56210088425984417e-01_dp,  4.45752169295529965e-01_dp,  7.65249583727318994e-01_dp,&
            1.30710669831289317e+00_dp,  2.27232012462184851e+00_dp,  4.38505336694388248e+00_dp]
       aw%aw_erange_matrix(:, 20) = [4.43803089127606113e-05_dp,  2.65218591197643853e-04_dp,  8.30855323495340568e-04_dp,  2.16814201958144461e-03_dp,  5.18489451591576823e-03_dp,&
            1.16752594424327817e-02_dp,  2.50600750841695370e-02_dp,  5.16739374920347702e-02_dp,  1.02949002312831606e-01_dp,  1.99055913849081695e-01_dp,  3.74892690057061606e-01_dp,&
            6.89886804356359851e-01_dp,  1.24433776114628336e+00_dp,  2.20900073280761244e+00_dp,  3.89224221437696771e+00_dp,  6.98596666740116490e+00_dp,  1.17061244040588000e-04_dp,&
            3.49507944241428143e-04_dp,  8.49620646664857337e-04_dp,  1.96981214128917614e-03_dp,  4.35085567522754818e-03_dp,  9.17937119472227517e-03_dp,  1.86109192573347161e-02_dp,&
            3.64635216587984104e-02_dp,  6.93518532719665654e-02_dp,  1.28520713372702128e-01_dp,  2.32797744928449957e-01_dp,  4.13453840598429323e-01_dp,  7.22968566709522364e-01_dp,&
            1.25497538883367321e+00_dp,  2.21182278597783633e+00_dp,  4.31625131487044467e+00_dp]
       aw%aw_erange_matrix(:, 21) = [3.18692979875762945e-05_dp,  1.94924143581001979e-04_dp,  6.34735317845784510e-04_dp,  1.72499618053375062e-03_dp,  4.27312550471167351e-03_dp,&
            9.90891266044059173e-03_dp,  2.17985806983987014e-02_dp,  4.58990884084013243e-02_dp,  9.31095938820044738e-02_dp,  1.82893089930429714e-01_dp,  3.49282689301175675e-01_dp,&
            6.50775264786516261e-01_dp,  1.18688178491725038e+00_dp,  2.12803484113070596e+00_dp,  3.78301819554569896e+00_dp,  6.84446856078756305e+00_dp,  8.44802619933795508e-05_dp,&
            2.63287491421840790e-04_dp,  6.75566783375913349e-04_dp,  1.63333525201483530e-03_dp,  3.72314514391366329e-03_dp,  8.05605050544987676e-03_dp,  1.66815720516301329e-02_dp,&
            3.32777027647976134e-02_dp,  6.42904126378543717e-02_dp,  1.20789761544245547e-01_dp,  2.21475826652405366e-01_dp,  3.97635557280198393e-01_dp,  7.02038439571572748e-01_dp,&
            1.22895802375841612e+00_dp,  2.18148993239703737e+00_dp,  4.28175673827994530e+00_dp]
       aw%aw_erange_matrix(:, 22) = [2.52372605908996745e-05_dp,  1.57392133212888027e-04_dp,  5.28274241848721731e-04_dp,  1.47892945126437171e-03_dp,  3.75547803252764293e-03_dp,&
            8.88646471280357154e-03_dp,  1.98792253903057707e-02_dp,  4.24522975011028258e-02_dp,  8.71647656143440946e-02_dp,  1.73023533565983295e-01_dp,  3.33498452444295046e-01_dp,&
            6.26473072676287823e-01_dp,  1.15092966665799223e+00_dp,  2.07707237149278523e+00_dp,  3.71394768667479935e+00_dp,  6.75468845238541338e+00_dp,  6.71839362983703346e-05_dp,&
            2.16820834544496606e-04_dp,  5.78858894715697709e-04_dp,  1.44079602687706139e-03_dp,  3.35595107506599056e-03_dp,  7.38753519939425174e-03_dp,  1.55169220144455947e-02_dp,&
            3.13310776551212547e-02_dp,  6.11648614294553819e-02_dp,  1.15971107204260557e-01_dp,  2.14360693807584268e-01_dp,  3.87622881472094116e-01_dp,  6.88709210745946487e-01_dp,&
            1.21231184422481064e+00_dp,  2.16203090843149370e+00_dp,  4.25962792921999256e+00_dp]
       aw%aw_erange_matrix(:, 23) = [2.10828735485771043e-05_dp,  1.33747043923274799e-04_dp,  4.60302814175420549e-04_dp,  1.31908976394215505e-03_dp,  3.41373834744931924e-03_dp,&
            8.20208746315155210e-03_dp,  1.85794936296085622e-02_dp,  4.00951119714870186e-02_dp,  8.30646818595997816e-02_dp,  1.66166495086338906e-01_dp,  3.22461707884920790e-01_dp,&
            6.09385272174553516e-01_dp,  1.12552832887090504e+00_dp,  2.04092006169053786e+00_dp,  3.66479300909027206e+00_dp,  6.69064931196403645e+00_dp,  5.63375423327661044e-05_dp,&
            1.87324678348996492e-04_dp,  5.15997238093761422e-04_dp,  1.31297823197896904e-03_dp,  3.10840654255903261e-03_dp,  6.93141540756278107e-03_dp,  1.47143909644481610e-02_dp,&
            2.99783578189846064e-02_dp,  5.89769743650245121e-02_dp,  1.12576359067155851e-01_dp,  2.09319673205829559e-01_dp,  3.80493824364532884e-01_dp,  6.79179041147570661e-01_dp,&
            1.20037201041025687e+00_dp,  2.14804779322759742e+00_dp,  4.24372611094621011e+00_dp]
       aw%aw_erange_matrix(:, 24) = [1.82166686738074864e-05_dp,  1.17355253951603524e-04_dp,  4.12646046678105574e-04_dp,  1.20544501630295061e-03_dp,  3.16766225037333439e-03_dp,&
            7.70401763679718320e-03_dp,  1.76251541149907119e-02_dp,  3.83513160771349129e-02_dp,  8.00120403518244372e-02_dp,  1.61032879231492870e-01_dp,  3.14159010930951887e-01_dp,&
            5.96476492618847165e-01_dp,  1.10626971507662764e+00_dp,  2.01342726666024063e+00_dp,  3.62732282398227612e+00_dp,  6.64174934770401748e+00_dp,  4.88479888159984235e-05_dp,&
            1.66743914977552775e-04_dp,  4.71273806908887769e-04_dp,  1.22055032501193809e-03_dp,  2.92729126196330808e-03_dp,  6.59463811322526239e-03_dp,  1.41173715836293810e-02_dp,&
            2.89656047084140694e-02_dp,  5.73298928164362870e-02_dp,  1.10008370974673766e-01_dp,  2.05490126818393987e-01_dp,  3.75057914236386680e-01_dp,  6.71889473870629361e-01_dp,&
            1.19121736903672493e+00_dp,  2.13731165043817173e+00_dp,  4.23151659571047745e+00_dp]
       aw%aw_erange_matrix(:, 25) = [1.61101137105389739e-05_dp,  1.05257432767853372e-04_dp,  3.77124469837566750e-04_dp,  1.11973962747994172e-03_dp,  2.98014652008201080e-03_dp,&
            7.32119813756530822e-03_dp,  1.68863969203923159e-02_dp,  3.69933310084055861e-02_dp,  7.76226242551877887e-02_dp,  1.56996883399357717e-01_dp,  3.07606512749093397e-01_dp,&
            5.86254946758791418e-01_dp,  1.09097647899171291e+00_dp,  1.99154292498465102e+00_dp,  3.59744001648970313e+00_dp,  6.60269853853355038e+00_dp,  4.33396342454888918e-05_dp,&
            1.51467498773258743e-04_dp,  4.37524678131294127e-04_dp,  1.14987956691945054e-03_dp,  2.78750323004162831e-03_dp,  6.33280212676514374e-03_dp,  1.36504094549384763e-02_dp,&
            2.81694379065950094e-02_dp,  5.60293648763132632e-02_dp,  1.07972916431346322e-01_dp,  2.02444490698196894e-01_dp,  3.70722013872658129e-01_dp,  6.66060575713892322e-01_dp,&
            1.18388318186635288e+00_dp,  2.12870098815666564e+00_dp,  4.22172408565013857e+00_dp]
       aw%aw_erange_matrix(:, 26) = [1.44910830020980627e-05_dp,  9.59247666285387932e-05_dp,  3.49480241611745145e-04_dp,  1.05236428638244149e-03_dp,  2.83143875825483384e-03_dp,&
            7.01541787267422751e-03_dp,  1.62928054973930578e-02_dp,  3.58967694676089050e-02_dp,  7.56850447598008957e-02_dp,  1.53712204828203025e-01_dp,  3.02257000428244105e-01_dp,&
            5.77887188163868748e-01_dp,  1.07842740830373907e+00_dp,  1.97355013597876772e+00_dp,  3.57283298867063293e+00_dp,  6.57050657122702475e+00_dp,  3.91036617280149617e-05_dp,&
            1.39622093468860363e-04_dp,  4.10978281792326609e-04_dp,  1.09367810950585973e-03_dp,  2.67546635017943237e-03_dp,  6.12167259211140406e-03_dp,  1.32720036926534928e-02_dp,&
            2.75215465371370822e-02_dp,  5.49672130356585603e-02_dp,  1.06305300828417465e-01_dp,  1.99942346115711156e-01_dp,  3.67151255105131302e-01_dp,  6.61250501621218900e-01_dp,&
            1.17782146729458970e+00_dp,  2.12157781794781597e+00_dp,  4.21362308983330625e+00_dp]
       aw%aw_erange_matrix(:, 27) = [1.21556594544792030e-05_dp,  8.23983748572736053e-05_dp,  3.08963398371883647e-04_dp,  9.52384499123811097e-04_dp,  2.60842886057528018e-03_dp,&
            6.55290684696903554e-03_dp,  1.53886295835706215e-02_dp,  3.42166248696358297e-02_dp,  7.27014821000612765e-02_dp,  1.48632646931657925e-01_dp,  2.93953603975382449e-01_dp,&
            5.64857136583094444e-01_dp,  1.05883226845267098e+00_dp,  1.94538975252448010e+00_dp,  3.53425049785512657e+00_dp,  6.51996578961256912e+00_dp,  3.29892698526955332e-05_dp,&
            1.22339673980425813e-04_dp,  3.71555506181443741e-04_dp,  1.00911840740269950e-03_dp,  2.50533305997647192e-03_dp,  5.79875266691652017e-03_dp,  1.26898229903717064e-02_dp,&
            2.65198029780881646e-02_dp,  5.33179394599257103e-02_dp,  1.03706252106678443e-01_dp,  1.96029940068529973e-01_dp,  3.61552073806202301e-01_dp,  6.53689885157118789e-01_dp,&
            1.16827593217446624e+00_dp,  2.11034879572048117e+00_dp,  4.20085232088900096e+00_dp]
       aw%aw_erange_matrix(:, 28) = [7.13471002471993354e-06_dp,  5.29467032451044542e-05_dp,  2.18126901539138012e-04_dp,  7.21360517466291921e-04_dp,  2.08015499871736474e-03_dp,&
            5.43528204883592450e-03_dp,  1.31681009285434403e-02_dp,  3.00345719866606173e-02_dp,  6.51901784064533585e-02_dp,  1.35719345130269731e-01_dp,  2.72666080314345083e-01_dp,&
            5.31207153623647232e-01_dp,  1.00790980686110809e+00_dp,  1.87182424824945604e+00_dp,  3.43304177060903370e+00_dp,  6.38699885280501967e+00_dp,  1.98257938552532495e-05_dp,&
            8.40296462119933309e-05_dp,  2.80273970340653956e-04_dp,  8.07328750940328643e-04_dp,  2.09061226948387829e-03_dp,  4.99849164035787510e-03_dp,  1.12274645565650279e-02_dp,&
            2.39748495351119265e-02_dp,  4.90869662060536283e-02_dp,  9.69821658282067484e-02_dp,  1.85832842429148964e-01_dp,  3.46864256074506494e-01_dp,  6.33748523755653492e-01_dp,&
            1.14299369524447547e+00_dp,  2.08053509421593663e+00_dp,  4.16694294663970854e+00_dp]
       aw%aw_erange_matrix(:, 29) = [5.29249514301132187e-06_dp,  4.19343252757951111e-05_dp,  1.82737438901562374e-04_dp,  6.27843358430121520e-04_dp,  1.85983070894205180e-03_dp,&
            4.95818859592519427e-03_dp,  1.22023907425847667e-02_dp,  2.81878008516035054e-02_dp,  6.18305239154504777e-02_dp,  1.29880299744984995e-01_dp,  2.62950034242736719e-01_dp,&
            5.15724357812102130e-01_dp,  9.84317591482867238e-01_dp,  1.83754523881623699e+00_dp,  3.38566863771911741e+00_dp,  6.32456122317518865e+00_dp,  1.49899692977725207e-05_dp,&
            6.93204774283267655e-05_dp,  2.43220905340470661e-04_dp,  7.22473373592021528e-04_dp,  1.91187712314453383e-03_dp,  4.64701311884111248e-03_dp,  1.05752900144337678e-02_dp,&
            2.28253081203215642e-02_dp,  4.71550285030659447e-02_dp,  9.38829556499910095e-02_dp,  1.81094415667536784e-01_dp,  3.39990601639111378e-01_dp,  6.24360483604677663e-01_dp,&
            1.13103660859194521e+00_dp,  2.06639710262585607e+00_dp,  4.15086131910516443e+00_dp]
       aw%aw_erange_matrix(:, 30) = [4.31773736878657279e-06_dp,  3.60303047352619597e-05_dp,  1.63261168167596623e-04_dp,  5.75226168100277158e-04_dp,  1.73379504384428150e-03_dp,&
            4.68178815714303739e-03_dp,  1.16372706574355469e-02_dp,  2.70982187469671100e-02_dp,  5.98347883444798334e-02_dp,  1.26391629758851071e-01_dp,  2.57116152461817626e-01_dp,&
            5.06388230347464385e-01_dp,  9.70039638261550818e-01_dp,  1.81673675871043816e+00_dp,  3.35684316451570375e+00_dp,  6.28650545297204477e+00_dp,  1.24303076540154611e-05_dp,&
            6.12924753893748605e-05_dp,  2.22338565270176626e-04_dp,  6.73725996984385692e-04_dp,  1.80782224834487824e-03_dp,  4.44029321814444489e-03_dp,  1.01885563508556497e-02_dp,&
            2.21389857406311917e-02_dp,  4.59949144505170382e-02_dp,  9.20126454807105032e-02_dp,  1.78222523273414285e-01_dp,  3.35808992061634781e-01_dp,  6.18631233760847832e-01_dp,&
            1.12372183807606851e+00_dp,  2.05773586398375707e+00_dp,  4.14100882646687207e+00_dp]
       aw%aw_erange_matrix(:, 31) = [3.70975867485438758e-06_dp,  3.23095974345838857e-05_dp,  1.50751791796946199e-04_dp,  5.40915866129935237e-04_dp,  1.65069566621838817e-03_dp,&
            4.49801466629874431e-03_dp,  1.12590458181901151e-02_dp,  2.63650642132821425e-02_dp,  5.84859131899247639e-02_dp,  1.24024822968737125e-01_dp,  2.53145535590468929e-01_dp,&
            5.00016351298351092e-01_dp,  9.60271964584979987e-01_dp,  1.80247354207944044e+00_dp,  3.33705426048128562e+00_dp,  6.26035148960221832e+00_dp,  1.08336000906553486e-05_dp,&
            5.61646913829607426e-05_dp,  2.08706414643022598e-04_dp,  6.41498436540636542e-04_dp,  1.73842323177099346e-03_dp,  4.30149368219168854e-03_dp,  9.92748802913993866e-03_dp,&
            2.16736122289964903e-02_dp,  4.52053172738387798e-02_dp,  9.07355616494632117e-02_dp,  1.76256045479438767e-01_dp,  3.32938764907261542e-01_dp,  6.14690683320622044e-01_dp,&
            1.11868286357054081e+00_dp,  2.05176383786785221e+00_dp,  4.13421515627375502e+00_dp]
       aw%aw_erange_matrix(:, 32) = [3.29307433796630773e-06_dp,  2.97376527986368188e-05_dp,  1.41976719845007126e-04_dp,  5.16576765783947141e-04_dp,  1.59126867541210652e-03_dp,&
            4.36579262671051883e-03_dp,  1.09856221527627942e-02_dp,  2.58330132577597980e-02_dp,  5.75039035275383276e-02_dp,  1.22297087801244098e-01_dp,  2.50240360996554667e-01_dp,&
            4.95345024579851689e-01_dp,  9.53099040349520976e-01_dp,  1.79198460975399865e+00_dp,  3.32248583508375628e+00_dp,  6.24108223942480667e+00_dp,  9.73921315654793179e-06_dp,&
            5.25819992873462335e-05_dp,  1.99028066423316980e-04_dp,  6.18407632563333165e-04_dp,  1.68838246973765458e-03_dp,  4.20092557580681921e-03_dp,  9.73759613841949229e-03_dp,&
            2.13340343901398007e-02_dp,  4.46276041650120678e-02_dp,  8.97990158818834650e-02_dp,  1.74811044817542116e-01_dp,  3.30826015322738642e-01_dp,  6.11785841611682035e-01_dp,&
            1.11496412843897841e+00_dp,  2.04735361117476744e+00_dp,  4.12919801960688915e+00_dp]
       aw%aw_erange_matrix(:, 33) = [2.98964795412928161e-06_dp,  2.78510556032888645e-05_dp,  1.35463612561978952e-04_dp,  4.98353624804465101e-04_dp,  1.54649765459536899e-03_dp,&
            4.26571596491992760e-03_dp,  1.07779202919790727e-02_dp,  2.54276640489241416e-02_dp,  5.67539329083084562e-02_dp,  1.20974901513310126e-01_dp,  2.48013235638977037e-01_dp,&
            4.91758596295572781e-01_dp,  9.47584973821341991e-01_dp,  1.78391286926115833e+00_dp,  3.31126541650462869e+00_dp,  6.22623261842660014e+00_dp,  8.94223582088384038e-06_dp,&
            4.99308515006045077e-05_dp,  1.91777151092376909e-04_dp,  6.00986766112024265e-04_dp,  1.65044538838878419e-03_dp,  4.12440019763640146e-03_dp,  9.59267530862847537e-03_dp,&
            2.10742486967102106e-02_dp,  4.41847367419175643e-02_dp,  8.90798160999419775e-02_dp,  1.73699706474474164e-01_dp,  3.29198984665094174e-01_dp,  6.09546349869233395e-01_dp,&
            1.11209472908096241e+00_dp,  2.04394895409445887e+00_dp,  4.12532474780290670e+00_dp]
       aw%aw_erange_matrix(:, 34) = [2.57922219186754929e-06_dp,  2.52766850415510830e-05_dp,  1.26457365708311655e-04_dp,  4.72914025961016545e-04_dp,  1.48357617092218538e-03_dp,&
            4.12436318531398793e-03_dp,  1.04834085373526585e-02_dp,  2.48510894180926055e-02_dp,  5.56843884684864870e-02_dp,  1.19085180222321993e-01_dp,  2.44824194691807717e-01_dp,&
            4.86614940401622065e-01_dp,  9.39665921045545360e-01_dp,  1.77230748120379422e+00_dp,  3.29511861865382860e+00_dp,  6.20484984880878532e+00_dp,  7.86406486178199112e-06_dp,&
            4.62765946095298626e-05_dp,  1.81648002610440557e-04_dp,  5.76466629589813639e-04_dp,  1.59676844408603957e-03_dp,  4.01569466774701501e-03_dp,  9.38616135532111505e-03_dp,&
            2.07030869722131423e-02_dp,  4.35506157524021276e-02_dp,  8.80480980271636099e-02_dp,  1.72102869337548120e-01_dp,  3.26857898204441721e-01_dp,  6.06320214250667622e-01_dp,&
            1.10795741758446264e+00_dp,  2.03903724449288815e+00_dp,  4.11973685029489456e+00_dp]
       aw%aw_erange_matrix(:, 35) = [2.43541362433170772e-06_dp,  2.43674010920485705e-05_dp,  1.23239875127668686e-04_dp,  4.63753248815872328e-04_dp,  1.46079218475904187e-03_dp,&
            4.07296827940922378e-03_dp,  1.03759836334632576e-02_dp,  2.46402394239789019e-02_dp,  5.52924317122943237e-02_dp,  1.18391416193676977e-01_dp,  2.43651638230665685e-01_dp,&
            4.84721244099458271e-01_dp,  9.36747200836526961e-01_dp,  1.76802615796251827e+00_dp,  3.28915764337563843e+00_dp,  6.19695188747453063e+00_dp,  7.48620957173570369e-06_dp,&
            4.49744932307259967e-05_dp,  1.77998490979932098e-04_dp,  5.67577081138817986e-04_dp,  1.57722452679259057e-03_dp,  3.97598562426190998e-03_dp,  9.31052865345715144e-03_dp,&
            2.05668655620557320e-02_dp,  4.33174693969102659e-02_dp,  8.76681895298326436e-02_dp,  1.71514093328614226e-01_dp,  3.25993725425264114e-01_dp,  6.05128200549742812e-01_dp,&
            1.10642760942127394e+00_dp,  2.03722030677132970e+00_dp,  4.11766973357092336e+00_dp]
       aw%aw_erange_matrix(:, 36) = [1.90859617159554337e-06_dp,  2.09950439600139040e-05_dp,  1.11113988674419262e-04_dp,  4.28852559134845432e-04_dp,  1.37333608208660158e-03_dp,&
            3.87459378923412116e-03_dp,  9.95955902729078225e-03_dp,  2.38200692781438822e-02_dp,  5.37634425665394003e-02_dp,  1.15678622021244265e-01_dp,  2.39057295596334041e-01_dp,&
            4.77288361373471137e-01_dp,  9.25274009180120371e-01_dp,  1.75117594726518022e+00_dp,  3.26567406672438754e+00_dp,  6.16581627107343788e+00_dp,  6.10122979734671908e-06_dp,&
            4.00837025306847991e-05_dp,  1.64084444472536599e-04_dp,  5.33400320302549955e-04_dp,  1.50164908261383990e-03_dp,  3.82175791908060704e-03_dp,  9.01575168296964852e-03_dp,&
            2.00344308423492865e-02_dp,  4.24040123406136116e-02_dp,  8.61766786663821643e-02_dp,  1.69198486719865704e-01_dp,  3.22589816911634075e-01_dp,  6.00426913775136684e-01_dp,&
            1.10038807582983877e+00_dp,  2.03004300869215726e+00_dp,  4.10950392701377876e+00_dp]
       aw%aw_erange_matrix(:, 37) = [1.90827994525250899e-06_dp,  2.09929942749567209e-05_dp,  1.11106514447489467e-04_dp,  4.28830855332524641e-04_dp,  1.37328138921740056e-03_dp,&
            3.87446922377897339e-03_dp,  9.95929668391619047e-03_dp,  2.38195511381459925e-02_dp,  5.37624744822230785e-02_dp,  1.15676901105840627e-01_dp,  2.39054376294121851e-01_dp,&
            4.77283632901010835e-01_dp,  9.25266704669341467e-01_dp,  1.75116520865135428e+00_dp,  3.26565907490263108e+00_dp,  6.16579634865748449e+00_dp,  6.10039728638633995e-06_dp,&
            4.00806962962503083e-05_dp,  1.64075783635878373e-04_dp,  5.33378918013125954e-04_dp,  1.50160156415125889e-03_dp,  3.82166062969930154e-03_dp,  9.01556517573429313e-03_dp,&
            2.00340931971503301e-02_dp,  4.24034320065246953e-02_dp,  8.61757293982789441e-02_dp,  1.69197011150964555e-01_dp,  3.22587647195451244e-01_dp,  6.00423914940963921e-01_dp,&
            1.10038421265627195e+00_dp,  2.03003839540528963e+00_dp,  4.10949865437411699e+00_dp]
    case(18)
       aw%energy_range(:) = [100.0_dp, 2.00e+02_dp, 3.00e+02_dp, 4.00e+02_dp, 5.00e+02_dp, 6.00e+02_dp,&
            7.00e+02_dp, 8.00e+02_dp, 9.00e+02_dp, 1.00e+03_dp, 2.00e+03_dp, 3.00e+03_dp, 4.00e+03_dp, 5.00e+03_dp,&
            6.00e+03_dp, 8.00e+03_dp, 9.00e+03_dp, 1.00e+04_dp, 2.00e+04_dp, 3.00e+04_dp, 4.00e+04_dp, 5.00e+04_dp,&
            6.00e+04_dp, 7.00e+04_dp, 8.00e+04_dp, 1.00e+05_dp, 2.00e+05_dp, 3.00e+05_dp, 4.00e+05_dp, 5.00e+05_dp,&
            6.00e+05_dp, 7.00e+05_dp, 1.00e+06_dp, 2.00e+06_dp, 3.00e+06_dp, 4.00e+06_dp, 5.00e+06_dp]
       aw%aw_erange_matrix(:, 1) = [3.13060568305601351e-03_dp,  1.67097480153570073e-02_dp,  4.20603752640300449e-02_dp,  8.09539652999341081e-02_dp,  1.36390967857746681e-01_dp,&
            2.13147798122915927e-01_dp,  3.18509646416849312e-01_dp,  4.63157341438440340e-01_dp,  6.62226034996298507e-01_dp,  9.36677961961052907e-01_dp,  1.31523305493242249e+00_dp,&
            1.83718188625082179e+00_dp,  2.55661335463641270e+00_dp,  3.54919245789090043e+00_dp,  4.92431566493793138e+00_dp,  6.85079762824929883e+00_dp,  9.62538936192941819e+00_dp,&
            1.39441065435998155e+01_dp,  8.05825095210471742e-03_dp,  1.92509609476614607e-02_dp,  3.17394427938591031e-02_dp,  4.65387250173524350e-02_dp,  6.51190142803804162e-02_dp,&
            8.95940655383273415e-02_dp,  1.22892483688214565e-01_dp,  1.68903153525686522e-01_dp,  2.32696487928973766e-01_dp,  3.20948271173013500e-01_dp,  4.42642242720547530e-01_dp,&
            6.10162954106962974e-01_dp,  8.41123679816304204e-01_dp,  1.16188888281340885e+00_dp,  1.61556100548747228e+00_dp,  2.28381103982621791e+00_dp,  3.36465045919326933e+00_dp,&
            5.63501703153524858e+00_dp]
       aw%aw_erange_matrix(:, 2) = [1.77849763282545976e-03_dp,  9.53050331705429001e-03_dp,  2.41715763670106998e-02_dp,  4.70787133826155654e-02_dp,  8.06744954509492351e-02_dp,&
            1.28944316701122441e-01_dp,  1.98099178394377601e-01_dp,  2.97374199692843622e-01_dp,  4.40063137209178501e-01_dp,  6.44984839288004541e-01_dp,  9.38607314927031755e-01_dp,&
            1.35811986020136444e+00_dp,  1.95596687976843242e+00_dp,  2.80694386709650079e+00_dp,  4.02057818466138084e+00_dp,  5.76669629386025306e+00_dp,  8.34281786095414724e+00_dp,&
            1.24402956709732422e+01_dp,  4.58203689637041566e-03_dp,  1.10339170966427605e-02_dp,  1.84719162771828926e-02_dp,  2.77374293691252972e-02_dp,  4.01072366160640584e-02_dp,&
            5.74545178903397985e-02_dp,  8.23776589959942379e-02_dp,  1.18364088609596463e-01_dp,  1.70115491406395952e-01_dp,  2.44089150111693826e-01_dp,  3.49280752689573537e-01_dp,&
            4.98372261284116880e-01_dp,  7.09596261927372463e-01_dp,  1.01023464227912507e+00_dp,  1.44442078136247543e+00_dp,  2.09441041846179798e+00_dp,  3.15665485525111889e+00_dp,&
            5.39538962042540060e+00_dp]
       aw%aw_erange_matrix(:, 3) = [1.27310209329263785e-03_dp,  6.84088991188971929e-03_dp,  1.74418711512609667e-02_dp,  3.42567626710492470e-02_dp,  5.94095603098494657e-02_dp,&
            9.64524637447977395e-02_dp,  1.50974587362519258e-01_dp,  2.31356617556676830e-01_dp,  3.49810807722170936e-01_dp,  5.23902124174831840e-01_dp,  7.78760954826290486e-01_dp,&
            1.15027942021814567e+00_dp,  1.68981403336343616e+00_dp,  2.47148636863273063e+00_dp,  3.60474942090268691e+00_dp,  5.25997727567653595e+00_dp,  7.73535457520057523e+00_dp,&
            1.17204273208081204e+01_dp,  3.28199550905035548e-03_dp,  7.94687934938108353e-03_dp,  1.34463066977488586e-02_dp,  2.05292847448247245e-02_dp,  3.03582077757123363e-02_dp,&
            4.46469181613256619e-02_dp,  6.57772653088819770e-02_dp,  9.69972855612304119e-02_dp,  1.42802353477466282e-01_dp,  2.09516542528400629e-01_dp,  3.06107049805222575e-01_dp,&
            4.45372961500316800e-01_dp,  6.45855025440923547e-01_dp,  9.35353101387836228e-01_dp,  1.35865735200749205e+00_dp,  1.99853353342874773e+00_dp,  3.05090311395216762e+00_dp,&
            5.27388729479473550e+00_dp]
       aw%aw_erange_matrix(:, 4) = [1.00300764061800004e-03_dp,  5.40119285324320610e-03_dp,  1.38290060307211613e-02_dp,  2.73431383817179241e-02_dp,  4.78739878380367509e-02_dp,&
            7.86830586197688475e-02_dp,  1.24932830849546025e-01_dp,  1.94417356236044975e-01_dp,  2.98609672055963393e-01_dp,  4.54213697328763211e-01_dp,  6.85426012155419873e-01_dp,&
            1.02720865762776636e+00_dp,  1.53010666114936233e+00_dp,  2.26769934719810662e+00_dp,  3.34931803735087019e+00_dp,  4.94568863021012639e+00_dp,  7.35552542383112673e+00_dp,&
            1.12674173192650144e+01_dp,  2.58696587386509974e-03_dp,  6.29119659728542141e-03_dp,  1.07351937196825085e-02_dp,  1.66070637685148750e-02_dp,  2.49888153437386115e-02_dp,&
            3.74779891899764961e-02_dp,  5.63035562499744971e-02_dp,  8.45522551019596558e-02_dp,  1.26579337993229679e-01_dp,  1.88608097926375479e-01_dp,  2.79565535852876024e-01_dp,&
            4.12305005143533798e-01_dp,  6.05556651756423370e-01_dp,  8.87470522524977157e-01_dp,  1.30331335096881218e+00_dp,  1.93627015321842832e+00_dp,  2.98203008114882762e+00_dp,&
            5.19487437442852773e+00_dp]
       aw%aw_erange_matrix(:, 5) = [8.33130943394381563e-04_dp,  4.49453339308115464e-03_dp,  1.15484388891811066e-02_dp,  2.29636533575316402e-02_dp,  4.05306238239156805e-02_dp,&
            6.72958002634303637e-02_dp,  1.08102758203152668e-01_dp,  1.70308031845335772e-01_dp,  2.64834042691084737e-01_dp,  4.07739033822954400e-01_dp,  6.22509561219692031e-01_dp,&
            9.43386633099325134e-01_dp,  1.42026920255805522e+00_dp,  2.12628406201697340e+00_dp,  3.17063021819169144e+00_dp,  4.72428168372380153e+00_dp,  7.08638539056717587e+00_dp,&
            1.09449464392502041e+01_dp,  2.14969537437315582e-03_dp,  5.24688620068331120e-03_dp,  9.01722309283595376e-03_dp,  1.41042638297888476e-02_dp,  2.15285615268984536e-02_dp,&
            3.27975625536323784e-02_dp,  5.00253106214112592e-02_dp,  7.61799293424517826e-02_dp,  1.15510957557027463e-01_dp,  1.74159210949979215e-01_dp,  2.61009248396864846e-01_dp,&
            3.88940440982029068e-01_dp,  5.76812654610189601e-01_dp,  8.53035431936163269e-01_dp,  1.26324693268146993e+00_dp,  1.89098401425114604e+00_dp,  2.93182828153962438e+00_dp,&
            5.13733863073762365e+00_dp]
       aw%aw_erange_matrix(:, 6) = [7.15674370843821166e-04_dp,  3.86697304672405486e-03_dp,  9.96677365690930121e-03_dp,  1.99172110614480556e-02_dp,  3.54008150836152211e-02_dp,&
            5.92954333284383614e-02_dp,  9.61933370780173463e-02_dp,  1.53107118543686582e-01_dp,  2.40526092120968232e-01_dp,  3.73997115143040904e-01_dp,  5.76437879681963916e-01_dp,&
            8.81503038816230711e-01_dp,  1.33855571884150937e+00_dp,  2.02033476439687165e+00_dp,  3.03590765912302984e+00_dp,  4.55643391821923860e+00_dp,  6.88142347695170553e+00_dp,&
            1.06984955705750568e+01_dp,  1.84728124663890124e-03_dp,  4.52309511617493120e-03_dp,  7.82185187058320568e-03_dp,  1.23524323213823044e-02_dp,  1.90859782538452365e-02_dp,&
            2.94573075116894845e-02_dp,  4.54900527824695505e-02_dp,  7.00598456607959136e-02_dp,  1.07331414106435472e-01_dp,  1.63375364458601763e-01_dp,  2.47034717198631570e-01_dp,&
            3.71200184596518046e-01_dp,  5.54826538805611946e-01_dp,  8.26526683183469335e-01_dp,  1.23224156051956890e+00_dp,  1.85580993954395712e+00_dp,  2.89276832421468555e+00_dp,&
            5.09260485295158460e+00_dp]
       aw%aw_erange_matrix(:, 7) = [6.29246983573313696e-04_dp,  3.40476518226396794e-03_dp,  8.79983474088586425e-03_dp,  1.76636613430474140e-02_dp,  3.15919107906099256e-02_dp,&
            5.33250278657308244e-02_dp,  8.72499733034381025e-02_dp,  1.40098901437124612e-01_dp,  2.22007419058450256e-01_dp,  3.48102016441458606e-01_dp,  5.40828223715164258e-01_dp,&
            8.33348162653710256e-01_dp,  1.27456789671411364e+00_dp,  1.93688694440292952e+00_dp,  2.92924597136559184e+00_dp,  4.42294883364496982e+00_dp,  6.71781677985601000e+00_dp,&
            1.05012007218785897e+01_dp,  1.62470924325914612e-03_dp,  3.98939943132514165e-03_dp,  6.93740550799082280e-03_dp,  1.10494735645689061e-02_dp,  1.72556556398382119e-02_dp,&
            2.69305189193088587e-02_dp,  4.20241386577638529e-02_dp,  6.53370028900366329e-02_dp,  1.00963057749311427e-01_dp,  1.54911672533170230e-01_dp,  2.35986284011789865e-01_dp,&
            3.57080603033420874e-01_dp,  5.37221991259609011e-01_dp,  8.05188864650826619e-01_dp,  1.20717674174648137e+00_dp,  1.82728826865744765e+00_dp,  2.86104949584900714e+00_dp,&
            5.05629894405748725e+00_dp]
       aw%aw_erange_matrix(:, 8) = [5.62782334945291941e-04_dp,  3.04901907885508743e-03_dp,  7.90029086110356077e-03_dp,  1.59223945992483552e-02_dp,  2.86389033037378754e-02_dp,&
            4.86751317030152050e-02_dp,  8.02458271532852235e-02_dp,  1.29848258845782466e-01_dp,  2.07321142519682061e-01_dp,  3.27435994170185096e-01_dp,  5.12236322594479265e-01_dp,&
            7.94460714322788752e-01_dp,  1.22261725071368121e+00_dp,  1.86880404707785375e+00_dp,  2.84184092769238550e+00_dp,  4.31314738354701088e+00_dp,  6.58281632626634128e+00_dp,&
            1.03380056558369038e+01_dp,  1.45351311063793179e-03_dp,  3.57821112835372780e-03_dp,  6.25389079018070105e-03_dp,  1.00377848387202106e-02_dp,  1.58249375146050225e-02_dp,&
            2.49388369993986600e-02_dp,  3.92681519545044280e-02_dp,  6.15504612191745196e-02_dp,  9.58190040595412740e-02_dp,  1.48028823756269218e-01_dp,  2.26946010516923008e-01_dp,&
            3.45462248099259850e-01_dp,  5.22662230551262152e-01_dp,  7.87462830383376389e-01_dp,  1.18627850542234259e+00_dp,  1.80344595414217435e+00_dp,  2.83450126737101193e+00_dp,&
            5.02592512274666348e+00_dp]
       aw%aw_erange_matrix(:, 9) = [5.09956820844997220e-04_dp,  2.76606079370183720e-03_dp,  7.18379472930052044e-03_dp,  1.45324734401939566e-02_dp,  2.62744457345686147e-02_dp,&
            4.49365179818514496e-02_dp,  7.45860480149161398e-02_dp,  1.21519479481494080e-01_dp,  1.95321069512380402e-01_dp,  3.10456408919682125e-01_dp,  4.88620042582648240e-01_dp,&
            7.62179819446476281e-01_dp,  1.17929178166661575e+00_dp,  1.81178303747056191e+00_dp,  2.76835875929337583e+00_dp,  4.22053315012927488e+00_dp,  6.46863952158576350e+00_dp,&
            1.01996937375922911e+01_dp,  1.31742409134479239e-03_dp,  3.25085143951871257e-03_dp,  5.70821004123319310e-03_dp,  9.22663936664768272e-03_dp,  1.46708066965285135e-02_dp,&
            2.33201522555469623e-02_dp,  3.70110320759733727e-02_dp,  5.84271461966189842e-02_dp,  9.15486203042080171e-02_dp,  1.42281684231895439e-01_dp,  2.19357305638089473e-01_dp,&
            3.35661995709629124e-01_dp,  5.10326837097277775e-01_dp,  7.72386958060629958e-01_dp,  1.16844839325418315e+00_dp,  1.78305793410421654e+00_dp,  2.81177434489785316e+00_dp,&
            4.99993299310682104e+00_dp]
       aw%aw_erange_matrix(:, 10) = [4.66884191696600092e-04_dp,  2.53518315097069156e-03_dp,  6.59842169255152496e-03_dp,  1.33946637142630318e-02_dp,  2.43333256802801684e-02_dp,&
            4.18555359937679630e-02_dp,  6.99004840051102028e-02_dp,  1.14590085605347738e-01_dp,  1.85286893591144752e-01_dp,  2.96188601701286136e-01_dp,  4.68682238029433706e-01_dp,&
            7.34806567718746972e-01_dp,  1.14240233117834866e+00_dp,  1.76305053795450828e+00_dp,  2.70534768772159451e+00_dp,  4.14088691630270844e+00_dp,  6.37021671670584588e+00_dp,&
            1.00802468831582139e+01_dp,  1.20644262202773494e-03_dp,  2.98351806905674134e-03_dp,  5.26144993746265575e-03_dp,  8.55990693672604996e-03_dp,  1.37168167419120662e-02_dp,&
            2.19731075762485857e-02_dp,  3.51198349523668615e-02_dp,  5.57937272538030424e-02_dp,  8.79277306880305448e-02_dp,  1.37383772368286527e-01_dp,  2.12859795610828961e-01_dp,&
            3.27235125271443661e-01_dp,  4.99679091955250343e-01_dp,  7.59329485867318521e-01_dp,  1.15296227944800744e+00_dp,  1.76531471483956559e+00_dp,  2.79197621123604556e+00_dp,&
            4.97729762459709502e+00_dp]
       aw%aw_erange_matrix(:, 11) = [2.60914461772335975e-04_dp,  1.42841100538599729e-03_dp,  3.77931336615040716e-03_dp,  7.87553426551717423e-03_dp,  1.48187271209905720e-02_dp,&
            2.65443710178691790e-02_dp,  4.62373542001488086e-02_dp,  7.89916247748617822e-02_dp,  1.32848433755762302e-01_dp,  2.20379319232295229e-01_dp,  3.61064527641105915e-01_dp,&
            5.84852933079486603e-01_dp,  9.37523669434331519e-01_dp,  1.48897669044511138e+00_dp,  2.34696867784464924e+00_dp,  3.68347884456586971e+00_dp,  5.80045014349923616e+00_dp,&
            9.38451113606470066e+00_dp,  6.75435555884669296e-04_dp,  1.69811101235115263e-03_dp,  3.09356203984489887e-03_dp,  5.27770014065303151e-03_dp,  8.92467772623281759e-03_dp,&
            1.50467300660292111e-02_dp,  2.51712972165478929e-02_dp,  4.16516501616942189e-02_dp,  6.81184703412873321e-02_dp,  1.10132067155941857e-01_dp,  1.76142887783290847e-01_dp,&
            2.78930670620491594e-01_dp,  4.37845260282784421e-01_dp,  6.82625481569703596e-01_dp,  1.06112107980412063e+00_dp,  1.65936002035834029e+00_dp,  2.67334288372550377e+00_dp,&
            4.84179632444405428e+00_dp]
       aw%aw_erange_matrix(:, 12) = [1.85513213773606345e-04_dp,  1.02150286740210769e-03_dp,  2.73448938724222363e-03_dp,  5.80387009885520444e-03_dp,  1.11810481774840837e-02_dp,&
            2.05518300952334462e-02_dp,  3.67304305786214091e-02_dp,  6.43020141414563928e-02_dp,  1.10640253721470419e-01_dp,  1.87473309428166701e-01_dp,  3.13264290932447564e-01_dp,&
            5.16812698313678487e-01_dp,  8.42722150950093285e-01_dp,  1.35988341226895848e+00_dp,  2.17548182992352057e+00_dp,  3.46162154443591952e+00_dp,  5.52102425741810254e+00_dp,&
            9.04041836150807754e+00_dp,  4.80852705753012058e-04_dp,  1.22305403571343449e-03_dp,  2.27947736150288169e-03_dp,  4.01365797781973958e-03_dp,  7.01540376387589735e-03_dp,&
            1.21847611879422838e-02_dp,  2.09200714737653552e-02_dp,  3.54265591278901115e-02_dp,  5.91658737085576508e-02_dp,  9.75188516417223455e-02_dp,  1.58774380033241708e-01_dp,&
            2.55620059373879827e-01_dp,  4.07460398574663651e-01_dp,  6.44326699296321581e-01_dp,  1.01465268223249172e+00_dp,  1.60523186707796173e+00_dp,  2.61244035390791707e+00_dp,&
            4.77231660346555397e+00_dp]
       aw%aw_erange_matrix(:, 13) = [1.45623827102946623e-04_dp,  8.05588152979675896e-04_dp,  2.17691130407093175e-03_dp,  4.68812739584344249e-03_dp,  9.19600123660022604e-03_dp,&
            1.72289055604926133e-02_dp,  3.13676494456035099e-02_dp,  5.58742924602326219e-02_dp,  9.76924100663839612e-02_dp,  1.67998143390820670e-01_dp,  2.84577977288065331e-01_dp,&
            4.75453513865553901e-01_dp,  7.84416664022989374e-01_dp,  1.27964179827845292e+00_dp,  2.06788347256167926e+00_dp,  3.32129384479391954e+00_dp,  5.34312314157560042e+00_dp,&
            8.82025781425350708e+00_dp,  3.77841659955862491e-04_dp,  9.70055271158684097e-04_dp,  1.84098010513958528e-03_dp,  3.32040210608571234e-03_dp,  5.94400030565985817e-03_dp,&
            1.05417311185070812e-02_dp,  1.84298427244651702e-02_dp,  3.17154955608041356e-02_dp,  5.37447671964476903e-02_dp,  8.97721982893909948e-02_dp,  1.47968012029178114e-01_dp,&
            2.40943262811694148e-01_dp,  3.88121908953358907e-01_dp,  6.19717513693844646e-01_dp,  9.84555284879807480e-01_dp,  1.56996784104876874e+00_dp,  2.57264246868716873e+00_dp,&
            4.72694104483554423e+00_dp]
       aw%aw_erange_matrix(:, 14) = [1.20693065016366942e-04_dp,  6.70318312279102792e-04_dp,  1.82598709243442714e-03_dp,  3.98066758906077175e-03_dp,  7.92407149371679963e-03_dp,&
            1.50731508121776207e-02_dp,  2.78434453982067988e-02_dp,  5.02664501056209675e-02_dp,  8.89754985015726030e-02_dp,  1.54744075465572922e-01_dp,  2.64859545012644360e-01_dp,&
            4.46762838614896685e-01_dp,  7.43631633383161961e-01_dp,  1.22308810558274139e+00_dp,  1.99154189668606629e+00_dp,  3.22116127998360158e+00_dp,  5.21559085842792136e+00_dp,&
            8.66187972983845889e+00_dp,  3.13424614215803300e-04_dp,  8.11091344556560262e-04_dp,  1.56293846928815807e-03_dp,  2.87442640486548556e-03_dp,  5.24259999238325686e-03_dp,&
            9.44812750354856110e-03_dp,  1.67483721901341004e-02_dp,  2.91782108273659670e-02_dp,  4.99968956826367938e-02_dp,  8.43623562376343422e-02_dp,  1.40351586287337071e-01_dp,&
            2.30511253319458237e-01_dp,  3.74270678501655019e-01_dp,  6.01971072101519233e-01_dp,  9.62727552491357996e-01_dp,  1.54428582589175911e+00_dp,  2.54359484716486017e+00_dp,&
            4.69383505396399681e+00_dp]
       aw%aw_erange_matrix(:, 15) = [1.03530813604779028e-04_dp,  5.77010708614184996e-04_dp,  1.58297978993598076e-03_dp,  3.48764309027865815e-03_dp,  7.02981637575661022e-03_dp,&
            1.35420129552544823e-02_dp,  2.53143459665430594e-02_dp,  4.62021297583787224e-02_dp,  8.25996052244215434e-02_dp,  1.44967282294702543e-01_dp,  2.50201253823955916e-01_dp,&
            4.25283165263042551e-01_dp,  7.12899874312358128e-01_dp,  1.18022626317392154e+00_dp,  1.93338491947318536e+00_dp,  3.14454465168471442e+00_dp,  5.11766152430480759e+00_dp,&
            8.53993925026136758e+00_dp,  2.69059599396066513e-04_dp,  7.01171220589084324e-04_dp,  1.36917997681255824e-03_dp,  2.55983016868183361e-03_dp,  4.74076816982743621e-03_dp,&
            8.65545451941890018e-03_dp,  1.55159703188033887e-02_dp,  2.73005170505556810e-02_dp,  4.71993736826663846e-02_dp,  8.02927232938843299e-02_dp,  1.34581100450814117e-01_dp,&
            2.22555937276669474e-01_dp,  3.63645272961041655e-01_dp,  5.88286109735375096e-01_dp,  9.45821302421523069e-01_dp,  1.52432961945465073e+00_dp,  2.52098460780209654e+00_dp,&
            4.66807280121724055e+00_dp]
       aw%aw_erange_matrix(:, 16) = [8.12878045927325100e-05_dp,  4.55784176179485265e-04_dp,  1.26575781231635042e-03_dp,  2.83901197267764265e-03_dp,  5.84075866946649497e-03_dp,&
            1.14816659474414502e-02_dp,  2.18703351471255143e-02_dp,  4.06049658428226795e-02_dp,  7.37274839449534453e-02_dp,  1.31232827738574781e-01_dp,  2.29429562298488349e-01_dp,&
            3.94602877306535982e-01_dp,  6.68687020587504510e-01_dp,  1.11816086345712651e+00_dp,  1.84868752331819342e+00_dp,  3.03241562091376515e+00_dp,  4.97377213890380609e+00_dp,&
            8.36023751368250245e+00_dp,  2.11528474078379687e-04_dp,  5.57938342187908878e-04_dp,  1.11429094995911461e-03_dp,  2.13984900872585520e-03_dp,  4.05975600506285235e-03_dp,&
            7.56390783215793276e-03_dp,  1.37976760555232440e-02_dp,  2.46541044572325049e-02_dp,  4.32184500991177195e-02_dp,  7.44508856457123808e-02_dp,  1.26231561940560427e-01_dp,&
            2.10961000314310559e-01_dp,  3.48056000387103548e-01_dp,  5.68089975660581725e-01_dp,  9.20748230015767999e-01_dp,  1.49462504826186793e+00_dp,  2.48726404601868678e+00_dp,&
            4.62966203399261467e+00_dp]
       aw%aw_erange_matrix(:, 17) = [7.36290238032269906e-05_dp,  4.13946476575048573e-04_dp,  1.15578122974341227e-03_dp,  2.61246326353696025e-03_dp,  5.42131632353000555e-03_dp,&
            1.07469311984104646e-02_dp,  2.06290337409884283e-02_dp,  3.85675089612289509e-02_dp,  7.04684426329998531e-02_dp,  1.26145823376294514e-01_dp,  2.21678062600754872e-01_dp,&
            3.83075284348309064e-01_dp,  6.51971738557552305e-01_dp,  1.09456550812149200e+00_dp,  1.81633014027732598e+00_dp,  2.98939917340350014e+00_dp,  4.91838496867421782e+00_dp,&
            8.29089093989333747e+00_dp,  1.91708707114072877e-04_dp,  5.08367399657288512e-04_dp,  1.02527533888157078e-03_dp,  1.99114795509654219e-03_dp,  3.81505964757374331e-03_dp,&
            7.16665373395437236e-03_dp,  1.31655503592034168e-02_dp,  2.36714122682651733e-02_dp,  4.17278947848333456e-02_dp,  7.22471020464889591e-02_dp,  1.23060199748490182e-01_dp,&
            2.06529455242298821e-01_dp,  3.42064141262841903e-01_dp,  5.60288564762608798e-01_dp,  9.11022245959785426e-01_dp,  1.48306650559867004e+00_dp,  2.47412089926975653e+00_dp,&
            4.61469407825248634e+00_dp]
       aw%aw_erange_matrix(:, 18) = [6.73944932443592415e-05_dp,  3.79844572428876927e-04_dp,  1.06590795373273759e-03_dp,  2.42654206896005193e-03_dp,  5.07517853516520952e-03_dp,&
            1.01369719376394104e-02_dp,  1.95925662402830214e-02_dp,  3.68571498016495233e-02_dp,  6.77192628196435775e-02_dp,  1.21835648595960938e-01_dp,  2.15083880414005030e-01_dp,&
            3.73233014220664572e-01_dp,  6.37653103358320594e-01_dp,  1.07429339684447545e+00_dp,  1.78845750203856402e+00_dp,  2.95226237507783207e+00_dp,  4.87048249753405393e+00_dp,&
            8.23083530472446867e+00_dp,  1.75569876637717915e-04_dp,  4.67898292556300052e-04_dp,  9.52228454181657511e-04_dp,  1.86817929682057309e-03_dp,  3.61108514012193401e-03_dp,&
            6.83324047724214335e-03_dp,  1.26319553349792356e-02_dp,  2.28377529856874373e-02_dp,  4.04577800825522693e-02_dp,  7.03617133989330734e-02_dp,  1.20337130462841210e-01_dp,&
            2.02711688147605068e-01_dp,  3.36886623918729866e-01_dp,  5.53529442704170482e-01_dp,  9.02576781682568829e-01_dp,  1.47301298821370485e+00_dp,  2.46267883439995794e+00_dp,&
            4.60166480506323872e+00_dp]
       aw%aw_erange_matrix(:, 19) = [3.77097267144774277e-05_dp,  2.16718158239339242e-04_dp,  6.31964795877831245e-04_dp,  1.51501851013790970e-03_dp,  3.34484667837428931e-03_dp,&
            7.02512458570965417e-03_dp,  1.42014149350935173e-02_dp,  2.78013295560369245e-02_dp,  5.29266569041400847e-02_dp,  9.83025113032961195e-02_dp,  1.78600150657352325e-01_dp,&
            3.18120006405406497e-01_dp,  5.56596891614358547e-01_dp,  9.58408306407735666e-01_dp,  1.62774470686190731e+00_dp,  2.73655217246640481e+00_dp,  4.59058386383132877e+00_dp,&
            7.87837440091345975e+00_dp,  9.86461415774919214e-05_dp,  2.73221143359511415e-04_dp,  5.94197557752134385e-04_dp,  1.24895734154715447e-03_dp,  2.55620318224912286e-03_dp,&
            5.06990830887251494e-03_dp,  9.75621380152377933e-03_dp,  1.82704930302675719e-02_dp,  3.33967548858809735e-02_dp,  5.97405156792770453e-02_dp,  1.04810909008227066e-01_dp,&
            1.80703082801913328e-01_dp,  3.06740164242034985e-01_dp,  5.13823817457257026e-01_dp,  8.52592911322042957e-01_dp,  1.41317820778574244e+00_dp,  2.39437286369400981e+00_dp,&
            4.52390876499952377e+00_dp]
       aw%aw_erange_matrix(:, 20) = [2.68925195549073629e-05_dp,  1.56793785885291212e-04_dp,  4.69893990043240026e-04_dp,  1.16545220385413321e-03_dp,  2.65989506956115698e-03_dp,&
            5.75384007396932337e-03_dp,  1.19342111722500789e-02_dp,  2.38928003384090495e-02_dp,  4.63925218571977285e-02_dp,  8.76901279402622358e-02_dp,  1.61839358502072539e-01_dp,&
            2.92374836693560902e-01_dp,  5.18161174811695435e-01_dp,  9.02717884502531653e-01_dp,  1.54960075834088817e+00_dp,  2.63061729962015844e+00_dp,  4.45202318493723048e+00_dp,&
            7.70286032300627355e+00_dp,  7.05638795782659030e-05_dp,  2.01002867914723093e-04_dp,  4.56947425490451874e-04_dp,  1.00093647179561995e-03_dp,  2.11652844500955601e-03_dp,&
            4.31084332567352024e-03_dp,  8.48457589505386897e-03_dp,  1.62034578815592155e-02_dp,  3.01348775048121756e-02_dp,  5.47430158873886477e-02_dp,  9.73833001049487501e-02_dp,&
            1.70015028445808208e-01_dp,  2.91900568092631596e-01_dp,  4.94043300498388738e-01_dp,  8.27439707654565648e-01_dp,  1.38283937942449198e+00_dp,  2.35959541119381777e+00_dp,&
            4.48433441513418707e+00_dp]
       aw%aw_erange_matrix(:, 21) = [2.11776428406011891e-05_dp,  1.24955957092028758e-04_dp,  3.82749361784648454e-04_dp,  9.73991658367676029e-04_dp,  2.27688417139812168e-03_dp,&
            5.02884426710743966e-03_dp,  1.06183171895938705e-02_dp,  2.15888221418370253e-02_dp,  4.24877264991899983e-02_dp,  8.12706704880782893e-02_dp,  1.51590277234206289e-01_dp,&
            2.76478503601760406e-01_dp,  4.94222230573377497e-01_dp,  8.67763285947123597e-01_dp,  1.50022012401874916e+00_dp,  2.56328999983562467e+00_dp,  4.36355480209440749e+00_dp,&
            7.59041890716973189e+00_dp,  5.57090107245227829e-05_dp,  1.62366769955750792e-04_dp,  3.81782556436269834e-04_dp,  8.61192201413146460e-04_dp,  1.86281232839267670e-03_dp,&
            3.86442117299721881e-03_dp,  7.72478532341631073e-03_dp,  1.49514707528110163e-02_dp,  2.81353063747719796e-02_dp,  5.16464716293368112e-02_dp,  9.27364294632549019e-02_dp,&
            1.63269905535395971e-01_dp,  2.82461831610602510e-01_dp,  4.81374480384390513e-01_dp,  8.11235487340904116e-01_dp,  1.36320833480298265e+00_dp,  2.33703760956925288e+00_dp,&
            4.45866956074495580e+00_dp]
       aw%aw_erange_matrix(:, 22) = [1.76070275101297462e-05_dp,  1.04974632586343552e-04_dp,  3.27524057106157195e-04_dp,  8.50889138429431618e-04_dp,  2.02675943694682068e-03_dp,&
            4.54854527962151971e-03_dp,  9.73548372448280498e-03_dp,  2.00259509380829870e-02_dp,  3.98132333040402478e-02_dp,  7.68361260215089659e-02_dp,  1.44456327953492114e-01_dp,&
            2.65338595024798618e-01_dp,  4.77344481537174625e-01_dp,  8.42986466846138205e-01_dp,  1.46505294151738807e+00_dp,  2.51515068726526270e+00_dp,  4.30009786276047024e+00_dp,&
            7.50957825176482174e+00_dp,  4.64186887815647243e-05_dp,  1.37983797560572785e-04_dp,  3.33452287623846701e-04_dp,  7.69419491405382667e-04_dp,  1.69333629958220424e-03_dp,&
            3.56219435629200097e-03_dp,  7.20463902309312343e-03_dp,  1.40860992359440711e-02_dp,  2.67414967648813770e-02_dp,  4.94717544703552684e-02_dp,  8.94508734055344951e-02_dp,&
            1.58471793174805697e-01_dp,  2.75710974071400383e-01_dp,  4.72269671037240601e-01_dp,  7.99542467873979534e-01_dp,  1.34899902892752999e+00_dp,  2.32068207466329079e+00_dp,&
            4.44006317705370979e+00_dp]
       aw%aw_erange_matrix(:, 23) = [1.51487261945737330e-05_dp,  9.11659324769150721e-05_dp,  2.89040873614000921e-04_dp,  7.64069868490654643e-04_dp,  1.84814200002400239e-03_dp,&
            4.20167024631436757e-03_dp,  9.09161519340235570e-03_dp,  1.88763863743453900e-02_dp,  3.78313678142224094e-02_dp,  7.35284936492460878e-02_dp,  1.39104394958239008e-01_dp,&
            2.56938214106312401e-01_dp,  4.64558752402094766e-01_dp,  8.24140212584070331e-01_dp,  1.43820799153508694e+00_dp,  2.47829256677809928e+00_dp,  4.25139458167740614e+00_dp,&
            7.44742363171281507e+00_dp,  4.00172934108619654e-05_dp,  1.21053459367159757e-04_dp,  2.99362782194228345e-04_dp,  7.03592889127086147e-04_dp,  1.57016975178594681e-03_dp,&
            3.34027399899004991e-03_dp,  6.81941752803501798e-03_dp,  1.34404676386291749e-02_dp,  2.56948821776365501e-02_dp,  4.78293838010395780e-02_dp,  8.69568482518628116e-02_dp,&
            1.54812802519890103e-01_dp,  2.70541537476452487e-01_dp,  4.65272231353635712e-01_dp,  7.90528148997813096e-01_dp,  1.33801932338487473e+00_dp,  2.30802753439873376e+00_dp,&
            4.42566810636102037e+00_dp]
       aw%aw_erange_matrix(:, 24) = [1.33450946857598585e-05_dp,  8.10013345693360817e-05_dp,  2.60505518780499042e-04_dp,  6.99027120294132363e-04_dp,  1.71292672688305443e-03_dp,&
            3.93664118611715408e-03_dp,  8.59572801141966769e-03_dp,  1.79849132808722820e-02_dp,  3.62852280011255329e-02_dp,  7.09344754417242929e-02_dp,  1.34887592254427091e-01_dp,&
            2.50292160328723368e-01_dp,  4.54405961654528678e-01_dp,  8.09126146653293299e-01_dp,  1.41676082143058490e+00_dp,  2.44877472513873817e+00_dp,  4.21231557277193680e+00_dp,&
            7.39748146374000015e+00_dp,  3.53174020538623987e-05_dp,  1.08539211526486186e-04_dp,  2.73819521014837626e-04_dp,  6.53580073098785556e-04_dp,  1.47559217672499333e-03_dp,&
            3.16843822945470479e-03_dp,  6.51906462972331319e-03_dp,  1.29340795221531794e-02_dp,  2.48697209477921517e-02_dp,  4.65285612184684963e-02_dp,  8.49733606273255226e-02_dp,&
            1.51892088420516908e-01_dp,  2.66401503567959441e-01_dp,  4.59651859717771616e-01_dp,  7.83269968829686136e-01_dp,  1.32916215886498579e+00_dp,  2.29780866662279104e+00_dp,&
            4.41404428527587811e+00_dp]
       aw%aw_erange_matrix(:, 25) = [1.19608983643762814e-05_dp,  7.31776981510573726e-05_dp,  2.38397281335537387e-04_dp,  6.48177541128883454e-04_dp,  1.60627044353721924e-03_dp,&
            3.72594606594641407e-03_dp,  8.19885121070576886e-03_dp,  1.72673139868429436e-02_dp,  3.50344193249588626e-02_dp,  6.88267529409492540e-02_dp,  1.31448064305153656e-01_dp,&
            2.44852576105250047e-01_dp,  4.46070906494070551e-01_dp,  7.96766930006198804e-01_dp,  1.39906449480958073e+00_dp,  2.42437075724512852e+00_dp,  4.17995561517234826e+00_dp,&
            7.35607828290079802e+00_dp,  3.17082936442419536e-05_dp,  9.88711098779920632e-05_dp,  2.53846227836985994e-04_dp,  6.14009273933647896e-04_dp,  1.40009229465454287e-03_dp,&
            3.03030469936903983e-03_dp,  6.27622081555165301e-03_dp,  1.25226197820897597e-02_dp,  2.41963451940166926e-02_dp,  4.54629624299898311e-02_dp,  8.33430042783768893e-02_dp,&
            1.49484033643996705e-01_dp,  2.62978817693378941e-01_dp,  4.54994127276706906e-01_dp,  7.77242691667035968e-01_dp,  1.32179571981875110e+00_dp,  2.28930236484464933e+00_dp,&
            4.40436883899004261e+00_dp]
       aw%aw_erange_matrix(:, 26) = [9.96761158553555444e-06_dp,  6.18692914207559080e-05_dp,  2.06170036210568655e-04_dp,  5.73211107890209581e-04_dp,  1.44730240695950876e-03_dp,&
            3.40892099536259157e-03_dp,  7.59685007995568491e-03_dp,  1.61712883720046205e-02_dp,  3.31125600685078861e-02_dp,  6.55713200633647325e-02_dp,  1.26111159618525148e-01_dp,&
            2.36377868660430784e-01_dp,  4.33038093291385484e-01_dp,  7.77379946000164024e-01_dp,  1.37122800415111401e+00_dp,  2.38589243185479205e+00_dp,  4.12883680000907560e+00_dp,&
            7.29058430903771093e+00_dp,  2.65071583106260783e-05_dp,  8.48293254396083508e-05_dp,  2.24390897268131415e-04_dp,  5.54811598906642159e-04_dp,  1.28593422488879972e-03_dp,&
            2.81969169097368326e-03_dp,  5.90338491932312654e-03_dp,  1.18871614049258830e-02_dp,  2.31510113224687453e-02_dp,  4.38012130603939800e-02_dp,  8.07902275831345329e-02_dp,&
            1.45699852659562423e-01_dp,  2.57582690043541174e-01_dp,  4.47629793783756336e-01_dp,  7.67689878785539737e-01_dp,  1.31009898646472545e+00_dp,  2.27578179593081087e+00_dp,&
            4.38899056842124402e+00_dp]
       aw%aw_erange_matrix(:, 27) = [5.69628328201120298e-06_dp,  3.73947477693335357e-05_dp,  1.34808292606433211e-04_dp,  4.02363370425018007e-04_dp,  1.07522362749315568e-03_dp,&
            2.64987937419991189e-03_dp,  6.12770973340106412e-03_dp,  1.34527127599654243e-02_dp,  2.82784222092996897e-02_dp,  5.72823558565759422e-02_dp,  1.12375952980137431e-01_dp,&
            2.14359358899264385e-01_dp,  3.98891047400829091e-01_dp,  7.26205544331862018e-01_dp,  1.29727249730106053e+00_dp,  2.28310280963276657e+00_dp,  3.99168266601377741e+00_dp,&
            7.11430302311417950e+00_dp,  1.53412014655641082e-05_dp,  5.40378710172882411e-05_dp,  1.57188321282197624e-04_dp,  4.15042400610066186e-04_dp,  1.00955395518829635e-03_dp,&
            2.29966856177653885e-03_dp,  4.96771195351118459e-03_dp,  1.02701101357026894e-02_dp,  2.04586605332921639e-02_dp,  3.94755240942939317e-02_dp,  7.40820751149756651e-02_dp,&
            1.35671534087687312e-01_dp,  2.43174288430001523e-01_dp,  4.27834657486577119e-01_dp,  7.41867110367467664e-01_dp,  1.27834520477995417e+00_dp,  2.23898801777038958e+00_dp,&
            4.34714455839110681e+00_dp]
       aw%aw_erange_matrix(:, 28) = [4.13173182523085577e-06_dp,  2.82953569586273491e-05_dp,  1.07357655282983076e-04_dp,  3.34031207388946118e-04_dp,  9.21278013215434373e-04_dp,&
            2.32697217024414134e-03_dp,  5.48821969396552471e-03_dp,  1.22464124910277479e-02_dp,  2.60980188844333050e-02_dp,  5.34905273921795141e-02_dp,  1.06014872445904379e-01_dp,&
            2.04051155883155949e-01_dp,  3.82751435806632623e-01_dp,  7.01813987411379236e-01_dp,  1.26176462906345210e+00_dp,  2.23344686165775030e+00_dp,  3.92510147193946413e+00_dp,&
            7.02842472272909724e+00_dp,  1.12413825198791527e-05_dp,  4.23581397394261224e-05_dp,  1.30257470325819791e-04_dp,  3.56609544464453910e-04_dp,  8.90477280344353396e-04_dp,&
            2.07032072932821203e-03_dp,  4.54706396508409039e-03_dp,  9.53127582160513340e-03_dp,  1.92112591692011961e-02_dp,  3.74468407961127125e-02_dp,  7.09020793561878498e-02_dp,&
            1.30872002390418907e-01_dp,  2.36219607171893220e-01_dp,  4.18208222356725279e-01_dp,  7.29229896403720712e-01_dp,  1.26273069797389059e+00_dp,  2.22084593571100797e+00_dp,&
            4.32651261931560516e+00_dp]
       aw%aw_erange_matrix(:, 29) = [3.30226286847199525e-06_dp,  2.34206306119432754e-05_dp,  9.23040944987100478e-05_dp,  2.95637516785855010e-04_dp,  8.33032454706429562e-04_dp,&
            2.13888506531630974e-03_dp,  5.11084598639201682e-03_dp,  1.15268119959546240e-02_dp,  2.47853719976952012e-02_dp,  5.11897543127538684e-02_dp,  1.02128694597074979e-01_dp,&
            1.97715753164931340e-01_dp,  3.72779679793518004e-01_dp,  6.86673954558245980e-01_dp,  1.23963600788232053e+00_dp,  2.20239644190786343e+00_dp,  3.88335573763512620e+00_dp,&
            6.97447560324605043e+00_dp,  9.06493831009571861e-06_dp,  3.60111158305800768e-05_dp,  1.15103834959029406e-04_dp,  3.22921911156177086e-04_dp,  8.20644989436114134e-04_dp,&
            1.93402803087780505e-03_dp,  4.29437456179440017e-03_dp,  9.08340287300126066e-03_dp,  1.84491991824218791e-02_dp,  3.61990758173190777e-02_dp,  6.89345184382906401e-02_dp,&
            1.27886682133110025e-01_dp,  2.31873470472087867e-01_dp,  4.12167639049623669e-01_dp,  7.21272419162619305e-01_dp,  1.25287237048937405e+00_dp,  2.20937452693632252e+00_dp,&
            4.31346712866726367e+00_dp]
       aw%aw_erange_matrix(:, 30) = [2.78283201387954402e-06_dp,  2.03426037320646153e-05_dp,  8.26259072951729948e-05_dp,  2.70516106805006676e-04_dp,  7.74477217336155344e-04_dp,&
            2.01269269881237858e-03_dp,  4.85538908841575724e-03_dp,  1.10360899520125632e-02_dp,  2.38846577194131125e-02_dp,  4.96025957422899302e-02_dp,  9.94354927347321504e-02_dp,&
            1.93307477671805478e-01_dp,  3.65816591824084103e-01_dp,  6.76069111738649142e-01_dp,  1.22409427413862582e+00_dp,  2.18053922732942196e+00_dp,  3.85391701960755872e+00_dp,&
            6.93638192354861616e+00_dp,  7.70087496722592801e-06_dp,  3.19576584688943520e-05_dp,  1.05177485427069824e-04_dp,  3.00482919789287189e-04_dp,  7.73582704782332952e-04_dp,&
            1.84134068718631730e-03_dp,  4.12126152251564119e-03_dp,  8.77467844465320244e-03_dp,  1.79211328551752251e-02_dp,  3.53304913887004385e-02_dp,  6.75593850637441345e-02_dp,&
            1.25792833963319417e-01_dp,  2.28815579140328212e-01_dp,  4.07905810796353407e-01_dp,  7.15645047122159150e-01_dp,  1.24588833188247139e+00_dp,  2.20123945650827624e+00_dp,&
            4.30421587078781709e+00_dp]
       aw%aw_erange_matrix(:, 31) = [2.42470995219759579e-06_dp,  1.82056418065889971e-05_dp,  7.58071587224136450e-05_dp,  2.52573761105593449e-04_dp,  7.32207996753558832e-04_dp,&
            1.92083837540198534e-03_dp,  4.66820156902059222e-03_dp,  1.06745344941764735e-02_dp,  2.32179642804407754e-02_dp,  4.84231777753803896e-02_dp,  9.74273510544483023e-02_dp,&
            1.90010738320586459e-01_dp,  3.60595635420595140e-01_dp,  6.68099359921788150e-01_dp,  1.21239121059783406e+00_dp,  2.16405314617733913e+00_dp,  3.83168318569659494e+00_dp,&
            6.90758403160364232e+00_dp,  6.75989988981772157e-06_dp,  2.91166320615211813e-05_dp,  9.80814416411888686e-05_dp,  2.84239565715768542e-04_dp,  7.39214990519655962e-04_dp,&
            1.77319565434359968e-03_dp,  3.99328716643299574e-03_dp,  8.54540727765975248e-03_dp,  1.75274395605375770e-02_dp,  3.46807424038581488e-02_dp,  6.65276661394083452e-02_dp,&
            1.24217779070994183e-01_dp,  2.26510012074744777e-01_dp,  4.04685973490304496e-01_dp,  7.11386227023358408e-01_dp,  1.24059585248586801e+00_dp,  2.19507011726273715e+00_dp,&
            4.29720009490146815e+00_dp]
       aw%aw_erange_matrix(:, 32) = [2.16176129771252570e-06_dp,  1.66270788284179806e-05_dp,  7.07073818199545589e-05_dp,  2.39005352223638798e-04_dp,  6.99970597597809083e-04_dp,&
            1.85032202789351969e-03_dp,  4.52374236259756816e-03_dp,  1.03943065527904843e-02_dp,  2.26993688693021452e-02_dp,  4.75029283214516498e-02_dp,  9.58563153562880799e-02_dp,&
            1.87425604388040429e-01_dp,  3.56493299282148790e-01_dp,  6.61826019040598901e-01_dp,  1.20316501743560211e+00_dp,  2.15103939920971232e+00_dp,  3.81411428402921349e+00_dp,&
            6.88481150276458820e+00_dp,  6.06872942426459449e-06_dp,  2.70007474622843711e-05_dp,  9.27112801831565700e-05_dp,  2.71824476999939159e-04_dp,  7.12764856522128025e-04_dp,&
            1.72046973418600829e-03_dp,  3.89384253410110818e-03_dp,  8.36660921401996896e-03_dp,  1.72194799075441295e-02_dp,  3.41711481654501797e-02_dp,  6.57166273395015377e-02_dp,&
            1.22977096456407906e-01_dp,  2.24690622353530628e-01_dp,  4.02141088617422038e-01_dp,  7.08015656612941524e-01_dp,  1.23640293809345647e+00_dp,  2.19017966864169455e+00_dp,&
            4.29163868677014726e+00_dp]
       aw%aw_erange_matrix(:, 33) = [1.66936450608206082e-06_dp,  1.36431464861143047e-05_dp,  6.08878352206036587e-05_dp,  2.12464299388389423e-04_dp,  6.36156479143742287e-04_dp,&
            1.70945176503017405e-03_dp,  4.23305031045998186e-03_dp,  9.82704453711613982e-03_dp,  2.16443420921362956e-02_dp,  4.56228251434010676e-02_dp,  9.26348688449874502e-02_dp,&
            1.82107804189891698e-01_dp,  3.48030917348462088e-01_dp,  6.48853583623900110e-01_dp,  1.18404608576497394e+00_dp,  2.12402377079366600e+00_dp,  3.77759108546833344e+00_dp,&
            6.83742277697884582e+00_dp,  4.77387758909035730e-06_dp,  2.29508126780449446e-05_dp,  8.21955857595700617e-05_dp,  2.47176829848202247e-04_dp,  6.59746554464799278e-04_dp,&
            1.61399895858116817e-03_dp,  3.69183124596445282e-03_dp,  8.00159671421590488e-03_dp,  1.65881388791646808e-02_dp,  3.31226449124670599e-02_dp,  6.40425960094452851e-02_dp,&
            1.20409083459694130e-01_dp,  2.20915444128518351e-01_dp,  3.96849048328325904e-01_dp,  7.00993735471721613e-01_dp,  1.22765555889850098e+00_dp,  2.17996886122467481e+00_dp,&
            4.28002698121303382e+00_dp]
       aw%aw_erange_matrix(:, 34) = [1.05153210682392675e-06_dp,  9.82109359198566600e-06_dp,  4.78395861953558743e-05_dp,  1.76150199721789534e-04_dp,  5.46950548985951032e-04_dp,&
            1.50929468544557942e-03_dp,  3.81467384580032147e-03_dp,  9.00203065583603246e-03_dp,  2.00964950531542334e-02_dp,  4.28440003106925135e-02_dp,  8.78430751011748528e-02_dp,&
            1.74153796774317637e-01_dp,  3.35311955480436741e-01_dp,  6.29273208523922545e-01_dp,  1.15508235334965992e+00_dp,  2.08297103191134481e+00_dp,  3.72195535108324504e+00_dp,&
            6.76510927942180729e+00_dp,  3.14818816312260181e-06_dp,  1.76265786828965847e-05_dp,  6.77797879677525459e-05_dp,  2.12550340542676892e-04_dp,  5.83980993235004228e-04_dp,&
            1.45984592182904255e-03_dp,  3.39626651311348464e-03_dp,  7.46288840499817214e-03_dp,  1.56494990870774189e-02_dp,  3.15539085571265657e-02_dp,  6.15241149934381698e-02_dp,&
            1.16526851395361591e-01_dp,  2.15183704649527130e-01_dp,  3.88784002747290469e-01_dp,  6.90258249868418550e-01_dp,  1.21424952388445284e+00_dp,  2.16429814216163985e+00_dp,&
            4.26220623979455038e+00_dp]
       aw%aw_erange_matrix(:, 35) = [8.34728214621684905e-07_dp,  8.44560904459773364e-06_dp,  4.29584696333223408e-05_dp,  1.62177244224623302e-04_dp,  5.11932680714949351e-04_dp,&
            1.42954228669343788e-03_dp,  3.64602080873342380e-03_dp,  8.66631682566647125e-03_dp,  1.94617241079152355e-02_dp,  4.16968867194957987e-02_dp,  8.58538110406805999e-02_dp,&
            1.70835591676214343e-01_dp,  3.29983260076326335e-01_dp,  6.21039304902132216e-01_dp,  1.14286341091260302e+00_dp,  2.06560541307989309e+00_dp,  3.69837086391296577e+00_dp,&
            6.73440815365611112e+00_dp,  2.57729833765363847e-06_dp,  1.56542937546273492e-05_dp,  6.22229276297930358e-05_dp,  1.98899151081870858e-04_dp,  5.53641700347800485e-04_dp,&
            1.39738260727315100e-03_dp,  3.27536872998534807e-03_dp,  7.24082112567673349e-03_dp,  1.52600396475556441e-02_dp,  3.08993617692109243e-02_dp,  6.04681734753560185e-02_dp,&
            1.14892156390790021e-01_dp,  2.12761131320170715e-01_dp,  3.85363989641657190e-01_dp,  6.85693142737872852e-01_dp,  1.20853665780972341e+00_dp,  2.15761202134704488e+00_dp,&
            4.25460273384813270e+00_dp]
       aw%aw_erange_matrix(:, 36) = [7.31697418235159294e-07_dp,  7.78192913747531435e-06_dp,  4.05556972701898700e-05_dp,  1.55204510399602370e-04_dp,  4.94292159044087270e-04_dp,&
            1.38908477445104676e-03_dp,  3.55999988304802392e-03_dp,  8.49433877643784516e-03_dp,  1.91353729694584383e-02_dp,  4.11053346197430766e-02_dp,  8.48253033248454608e-02_dp,&
            1.69116119572689377e-01_dp,  3.27216548478532321e-01_dp,  6.16756871439927856e-01_dp,  1.13649899521993558e+00_dp,  2.05654909227641935e+00_dp,  3.68605932606823439e+00_dp,&
            6.71837038140371501e+00_dp,  2.30579035272544704e-06_dp,  1.46876721811319836e-05_dp,  5.94476934252411929e-05_dp,  1.92009110580819360e-04_dp,  5.38216569141812995e-04_dp,&
            1.36544905409106596e-03_dp,  3.21328987943083396e-03_dp,  7.12638348625628006e-03_dp,  1.50587343735886312e-02_dp,  3.05601639949946297e-02_dp,  5.99197402403319895e-02_dp,&
            1.14041463521724409e-01_dp,  2.11498245945244356e-01_dp,  3.83578440411476651e-01_dp,  6.83306702782851882e-01_dp,  1.20554729177078546e+00_dp,  2.15411140205395535e+00_dp,&
            4.25062177779508943e+00_dp]
       aw%aw_erange_matrix(:, 37) = [6.83039002664919050e-07_dp,  7.46562050293998484e-06_dp,  3.93979118840847106e-05_dp,  1.51820281594498933e-04_dp,  4.85687667238021291e-04_dp,&
            1.36927867666848582e-03_dp,  3.51776903744843977e-03_dp,  8.40971701364783518e-03_dp,  1.89744917631880054e-02_dp,  4.08132582175073366e-02_dp,  8.43167978175517058e-02_dp,&
            1.68265003230324950e-01_dp,  3.25845673139424641e-01_dp,  6.14633097811190798e-01_dp,  1.13334030627276450e+00_dp,  2.05205152948763603e+00_dp,  3.67994206898681142e+00_dp,&
            6.71039878795908340e+00_dp,  2.17748291977497743e-06_dp,  1.42229341674666827e-05_dp,  5.81001678508209256e-05_dp,  1.88645103315669666e-04_dp,  5.30656526049757909e-04_dp,&
            1.34975291504493789e-03_dp,  3.18270709340716688e-03_dp,  7.06990140878475948e-03_dp,  1.49592223666598269e-02_dp,  3.03922627801653661e-02_dp,  5.96479536663102744e-02_dp,&
            1.13619458264455936e-01_dp,  2.10871203187387918e-01_dp,  3.82691193513678196e-01_dp,  6.82120087405447051e-01_dp,  1.20406013018453062e+00_dp,  2.15236939221153945e+00_dp,&
            4.24864073243265405e+00_dp]
       aw%aw_erange_matrix(:, 38) = [6.72200345974212699e-07_dp,  7.39487832678119611e-06_dp,  3.91377783182234103e-05_dp,  1.51057631809280499e-04_dp,  4.83744654923509341e-04_dp,&
            1.36479949662894596e-03_dp,  3.50820740770161005e-03_dp,  8.39053967695347139e-03_dp,  1.89380042261968438e-02_dp,  4.07469730322969001e-02_dp,  8.42013313845971084e-02_dp,&
            1.68071648316776984e-01_dp,  3.25534113047576457e-01_dp,  6.14150254525752004e-01_dp,  1.13262195300347757e+00_dp,  2.05102842495773885e+00_dp,  3.67855023430316885e+00_dp,&
            6.70858477543888654e+00_dp,  2.14889286843083340e-06_dp,  1.41186087411002464e-05_dp,  5.77964458703214584e-05_dp,  1.87885165416971971e-04_dp,  5.28946021846840391e-04_dp,&
            1.34619737793210893e-03_dp,  3.17577288535281048e-03_dp,  7.05708509626393893e-03_dp,  1.49366276661514613e-02_dp,  3.03541189639940263e-02_dp,  5.95861799519916066e-02_dp,&
            1.13523502921516722e-01_dp,  2.10728576176481969e-01_dp,  3.82489316883349306e-01_dp,  6.81850022811744116e-01_dp,  1.20372159515618860e+00_dp,  2.15197279609530101e+00_dp,&
            4.24818970522041894e+00_dp]
    case(20)
       aw%energy_range(:) = [200.0_dp, 3.00e+02_dp, 4.00e+02_dp, 5.00e+02_dp, 6.00e+02_dp, 7.00e+02_dp,&
            8.00e+02_dp, 9.00e+02_dp, 1.00e+03_dp, 2.00e+03_dp, 3.00e+03_dp, 4.00e+03_dp, 5.00e+03_dp, 6.00e+03_dp,&
            7.00e+03_dp, 8.00e+03_dp, 9.00e+03_dp, 1.00e+04_dp, 2.00e+04_dp, 3.00e+04_dp, 4.00e+04_dp, 5.00e+04_dp,&
            6.00e+04_dp, 7.00e+04_dp, 8.00e+04_dp, 1.00e+05_dp, 2.00e+05_dp, 3.00e+05_dp, 4.00e+05_dp, 5.00e+05_dp,&
            6.00e+05_dp, 7.00e+05_dp, 9.00e+05_dp, 1.00e+06_dp, 2.00e+06_dp, 3.00e+06_dp, 4.00e+06_dp, 8.00e+06_dp, 1.00e+07_dp]
       aw%aw_erange_matrix(:, 1) = [1.59340911033067478e-03_dp,  8.50904946849679461e-03_dp,  2.14381324891662305e-02_dp,  4.13217924134164430e-02_dp,  6.97620860473193438e-02_dp,&
            1.09321104685075207e-01_dp,  1.63918949130833574e-01_dp,  2.39312230571788187e-01_dp,  3.43666004319766016e-01_dp,  4.88296891505629160e-01_dp,  6.88703938259422221e-01_dp,&
            9.66014744733213249e-01_dp,  1.34900932673276852e+00_dp,  1.87699212381449976e+00_dp,  2.60403874113626044e+00_dp,  3.60576914286975558e+00_dp,  4.99149485326237308e+00_dp,&
            6.92991716904237887e+00_dp,  9.71772239452879383e+00_dp,  1.40512612733913116e+01_dp,  4.10193054094624976e-03_dp,  9.80903579051960878e-03_dp,  1.62026542274407610e-02_dp,&
            2.38266973811476421e-02_dp,  3.34742013991377804e-02_dp,  4.62892230812084957e-02_dp,  6.38565714484942287e-02_dp,  8.82787418367577265e-02_dp,  1.22298020153240233e-01_dp,&
            1.69523185339919447e-01_dp,  2.34779037546969105e-01_dp,  3.24592511147649587e-01_dp,  4.47879020689130047e-01_dp,  6.16992015749276979e-01_dp,  8.49517596862311541e-01_dp,&
            1.17178329836823036e+00_dp,  1.62685169309855482e+00_dp,  2.29638632075273952e+00_dp,  3.37853613531567998e+00_dp,  5.65113328705362239e+00_dp]
       aw%aw_erange_matrix(:, 2) = [1.13904499210405719e-03_dp,  6.09552014069879994e-03_dp,  1.54192503088270930e-02_dp,  2.99082651922659183e-02_dp,  5.09493519494348124e-02_dp,&
            8.08010692431461869e-02_dp,  1.22961299281198297e-01_dp,  1.82606873383205665e-01_dp,  2.67142819438789980e-01_dp,  3.86955966557172848e-01_dp,  5.56484587041208290e-01_dp,&
            7.95723471843741592e-01_dp,  1.13232890329597891e+00_dp,  1.60460031003783343e+00_dp,  2.26586404926580043e+00_dp,  3.19138968216569285e+00_dp,  4.49062754712213330e+00_dp,&
            6.33281343681569542e+00_dp,  9.01494311947765325e+00_dp,  1.32306709449480362e+01_dp,  2.93366938570087778e-03_dp,  7.04515555971851719e-03_dp,  1.17321497525370126e-02_dp,&
            1.74722715685951720e-02_dp,  2.49801406266746927e-02_dp,  3.52990340770604344e-02_dp,  4.98753280948631070e-02_dp,  7.06377004364660971e-02_dp,  1.00147769319503158e-01_dp,&
            1.41860186612827538e-01_dp,  2.00495050445585049e-01_dp,  2.82543540531873028e-01_dp,  3.96979859841220306e-01_dp,  5.56341007540550048e-01_dp,  7.78542662514336348e-01_dp,&
            1.09037275616667939e+00_dp,  1.53538382621142788e+00_dp,  2.19546436446452242e+00_dp,  3.26782969634996601e+00_dp,  5.52341118050514801e+00_dp]
       aw%aw_erange_matrix(:, 3) = [8.96430353358386965e-04_dp,  4.80511917695569061e-03_dp,  1.21936515586327544e-02_dp,  2.37702270534453051e-02_dp,  4.07838827125285575e-02_dp,&
            6.52933285267013330e-02_dp,  1.00510331249324444e-01_dp,  1.51213651502473068e-01_dp,  2.24287342990249272e-01_dp,  3.29482475920918672e-01_dp,  4.80507719050899063e-01_dp,&
            6.96566570523348361e-01_dp,  1.00451159229719678e+00_dp,  1.44189952932815313e+00_dp,  2.06147408277824784e+00_dp,  2.93819399840996098e+00_dp,  4.18155568909791153e+00_dp,&
            5.96116486731876360e+00_dp,  8.57432689691118100e+00_dp,  1.27131467807816598e+01_dp,  2.30967192745106529e-03_dp,  5.56509969704642189e-03_dp,  9.32697130056928982e-03_dp,&
            1.40298314904001731e-02_dp,  2.03348345664661201e-02_dp,  2.92125911781597747e-02_dp,  4.20091617449700402e-02_dp,  6.05323709666417131e-02_dp,  8.72222769826312971e-02_dp,&
            1.25427531276734477e-01_dp,  1.79788267361212062e-01_dp,  2.56755085344252509e-01_dp,  3.65322039803622378e-01_dp,  5.18132829129207328e-01_dp,  7.33319053236828111e-01_dp,&
            1.03799272215612737e+00_dp,  1.47607815325437985e+00_dp,  2.12968859060615667e+00_dp,  3.19552335247835018e+00_dp,  5.44013345543908233e+00_dp]
       aw%aw_erange_matrix(:, 4) = [7.43941858108558194e-04_dp,  3.99324506572499953e-03_dp,  1.01604121564421498e-02_dp,  1.98903231588493483e-02_dp,  3.43334526398137996e-02_dp,&
            5.54024931102704218e-02_dp,  8.60964974945682276e-02_dp,  1.30896886303630361e-01_dp,  1.96299511993370007e-01_dp,  2.91581476175378829e-01_dp,  4.29903569927137685e-01_dp,&
            6.29867338197687143e-01_dp,  9.17703244108222349e-01_dp,  1.33038016317608565e+00_dp,  1.92016513297746938e+00_dp,  2.76174277576863370e+00_dp,  3.96461447664841415e+00_dp,&
            5.69866218889320120e+00_dp,  8.26147101907144155e+00_dp,  1.23441248885543207e+01_dp,  1.91738312725215298e-03_dp,  4.63273201663321442e-03_dp,  7.80616390956771766e-03_dp,&
            1.18410761366227798e-02_dp,  1.73585460554474165e-02_dp,  2.52729488781124716e-02_dp,  3.68533483311020288e-02_dp,  5.38175204768781279e-02_dp,  7.85158078111489882e-02_dp,&
            1.14216036069974561e-01_dp,  1.65492589404866192e-01_dp,  2.38756014936469058e-01_dp,  3.43003601684555626e-01_dp,  4.90948202391497612e-01_dp,  7.00877399607738760e-01_dp,&
            1.00015045424850091e+00_dp,  1.43298990052828579e+00_dp,  2.08171532337388987e+00_dp,  3.14270096039701530e+00_dp,  5.37936604706722488e+00_dp]
       aw%aw_erange_matrix(:, 5) = [6.38569097485054823e-04_dp,  3.43174037774286039e-03_dp,  8.75196804462230991e-03_dp,  1.71963112345279133e-02_dp,  2.98398770292170358e-02_dp,&
            4.84818741012941479e-02_dp,  7.59541529063177817e-02_dp,  1.16504064565401771e-01_dp,  1.76322405079390676e-01_dp,  2.64313084573280566e-01_dp,  3.93202447454869652e-01_dp,&
            5.81109560406391834e-01_dp,  8.53759528622831998e-01_dp,  1.24763563082221629e+00_dp,  1.81460168527891219e+00_dp,  2.62909845388257679e+00_dp,  3.80061152976508865e+00_dp,&
            5.49923963973098662e+00_dp,  8.02281714515402378e+00_dp,  1.20616990467191130e+01_dp,  1.64624927347082348e-03_dp,  3.98721122270609320e-03_dp,  6.74992784847414876e-03_dp,&
            1.03137860467780312e-02_dp,  1.52680780395206932e-02_dp,  2.24817028905434819e-02_dp,  3.31621895881786602e-02_dp,  4.89568250767923874e-02_dp,  7.21456267563233278e-02_dp,&
            1.05931034158105800e-01_dp,  1.54831311570967134e-01_dp,  2.25219052919692603e-01_dp,  3.26086701981142257e-01_dp,  4.70194933849082175e-01_dp,  6.75950928297744835e-01_dp,&
            9.70912199566876133e-01_dp,  1.39954954691031519e+00_dp,  2.04436947680555070e+00_dp,  3.10152548232226133e+00_dp,  5.33203772355721828e+00_dp]
       aw%aw_erange_matrix(:, 6) = [5.61072571143229307e-04_dp,  3.01847196714530959e-03_dp,  7.71392131513013325e-03_dp,  1.52066639997109291e-02_dp,  2.65115694541762648e-02_dp,&
            4.33359502045003939e-02_dp,  6.83752103545800793e-02_dp,  1.05685662835659139e-01_dp,  1.61209385496320828e-01_dp,  2.43545204318619424e-01_dp,  3.65061973518195249e-01_dp,&
            5.43478150149877104e-01_dp,  8.04094672373795283e-01_dp,  1.18298183861235096e+00_dp,  1.73165420573701256e+00_dp,  2.52433338301671384e+00_dp,  3.67047792002619833e+00_dp,&
            5.34036265755118400e+00_dp,  7.83204467831639839e+00_dp,  1.18353304830751558e+01_dp,  1.44680898907737616e-03_dp,  3.51167044073153866e-03_dp,  5.96968933905672565e-03_dp,&
            9.18093559184831980e-03_dp,  1.37085277848084802e-02_dp,  2.03834591409695898e-02_dp,  3.03625964952499612e-02_dp,  4.52360196139964740e-02_dp,  6.72262909737048331e-02_dp,&
            9.94809247142534742e-02_dp,  1.46469151754476751e-01_dp,  2.14528074303888566e-01_dp,  3.12641076666117790e-01_dp,  4.53603331993510606e-01_dp,  6.55917496307296788e-01_dp,&
            9.47305530734814605e-01_dp,  1.37245042451229127e+00_dp,  2.01402824955472548e+00_dp,  3.06803524529046046e+00_dp,  5.29356870637847177e+00_dp]
       aw%aw_erange_matrix(:, 7) = [5.01502882940591649e-04_dp,  2.70059099306814735e-03_dp,  6.91448695519962693e-03_dp,  1.36715291090292804e-02_dp,  2.39368946329081636e-02_dp,&
            3.93412656695783861e-02_dp,  6.24656462006050184e-02_dp,  9.72062500368065519e-02_dp,  1.49296807648936825e-01_dp,  2.27079819851898512e-01_dp,  3.42621991546906424e-01_dp,&
            5.13300567195080992e-01_dp,  7.64051947550105304e-01_dp,  1.13058766242498310e+00_dp,  1.66411434419076909e+00_dp,  2.43865478941696301e+00_dp,  3.56363419705310402e+00_dp,&
            5.20947420923795512e+00_dp,  7.67443229279426298e+00_dp,  1.16478865034794978e+01_dp,  1.29348047897017305e-03_dp,  3.14559139318004288e-03_dp,  5.36758143201675371e-03_dp,&
            8.30349608680929288e-03_dp,  1.24943149769587650e-02_dp,  1.87387247506306288e-02_dp,  2.81509204278415082e-02_dp,  4.22732668306473811e-02_dp,  6.32799645640150787e-02_dp,&
            9.42711779089224627e-02_dp,  1.39672583458803179e-01_dp,  2.05788207123199618e-01_dp,  3.01590110107878451e-01_dp,  4.39899072363681753e-01_dp,  6.39296224930650192e-01_dp,&
            9.27643339822016300e-01_dp,  1.34980836215914834e+00_dp,  1.98862199643710014e+00_dp,  3.03996478570268147e+00_dp,  5.26134269478920658e+00_dp]
       aw%aw_erange_matrix(:, 8) = [4.54176798051982500e-04_dp,  2.44789321253706393e-03_dp,  6.27827314840997529e-03_dp,  1.24477642842393075e-02_dp,  2.18795651918432255e-02_dp,&
            3.61390339655896717e-02_dp,  5.77092718818298725e-02_dp,  9.03495739014426036e-02_dp,  1.39615485330027711e-01_dp,  2.13629615927840938e-01_dp,  3.24198064335450342e-01_dp,&
            4.88401802539330132e-01_dp,  7.30858277294287317e-01_dp,  1.08696225289904591e+00_dp,  1.60764513485711924e+00_dp,  2.36674787687761778e+00_dp,  3.47365879336816574e+00_dp,&
            5.09892433582888494e+00_dp,  7.54098359742881197e+00_dp,  1.14888704905340866e+01_dp,  1.17164923379155499e-03_dp,  2.85436374449175379e-03_dp,  4.88752520782221103e-03_dp,&
            7.60157610521120051e-03_dp,  1.15183886939231837e-02_dp,  1.74086397296599645e-02_dp,  2.63499286281822058e-02_dp,  3.98439640943772447e-02_dp,  6.00233171003619198e-02_dp,&
            8.99465532002833024e-02_dp,  1.34000176292588674e-01_dp,  1.98457373277317051e-01_dp,  2.92277680330032319e-01_dp,  4.28301261947665823e-01_dp,  6.25175194562218151e-01_dp,&
            9.10882366673366861e-01_dp,  1.33045440899754253e+00_dp,  1.96686388521136335e+00_dp,  3.01590430475151239e+00_dp,  5.23373285475806327e+00_dp]
       aw%aw_erange_matrix(:, 9) = [2.31389545652061434e-04_dp,  1.25573548361936158e-03_dp,  3.26467418878907721e-03_dp,  6.61525939054778862e-03_dp,  1.19877180510286341e-02_dp,&
            2.05587754848347778e-02_dp,  3.42247963066599770e-02_dp,  5.59267900365687104e-02_dp,  9.01510566310825684e-02_dp,  1.43681976375189924e-01_dp,  2.26707521329035461e-01_dp,&
            3.54426262682979276e-01_dp,  5.49376630765429419e-01_dp,  8.44825940909645468e-01_dp,  1.28978231999914672e+00_dp,  1.95672260232700945e+00_dp,  2.95461562172969705e+00_dp,&
            4.45472865449403788e+00_dp,  6.75680531467232992e+00_dp,  1.05482558845397225e+01_dp,  5.97841988594928681e-04_dp,  1.47677186318177149e-03_dp,  2.59844834407391709e-03_dp,&
            4.21318849959283811e-03_dp,  6.72427730470716387e-03_dp,  1.07286659349290717e-02_dp,  1.70853685841758293e-02_dp,  2.70540197117113690e-02_dp,  4.25071646238493889e-02_dp,&
            6.62263969430600397e-02_dp,  1.02319734751618513e-01_dp,  1.56818913364540680e-01_dp,  2.38545154014374761e-01_dp,  3.60397852411913966e-01_dp,  5.41391349029674895e-01_dp,&
            8.10266992654025575e-01_dp,  1.21316027447477337e+00_dp,  1.83411013311083582e+00_dp,  2.86864360132638829e+00_dp,  5.06499212313868163e+00_dp]
       aw%aw_erange_matrix(:, 10) = [2.31389545652061434e-04_dp,  1.25573548361936158e-03_dp,  3.26467418878907721e-03_dp,  6.61525939054778862e-03_dp,  1.19877180510286341e-02_dp,&
            2.05587754848347778e-02_dp,  3.42247963066599770e-02_dp,  5.59267900365687104e-02_dp,  9.01510566310825684e-02_dp,  1.43681976375189924e-01_dp,  2.26707521329035461e-01_dp,&
            3.54426262682979276e-01_dp,  5.49376630765429419e-01_dp,  8.44825940909645468e-01_dp,  1.28978231999914672e+00_dp,  1.95672260232700945e+00_dp,  2.95461562172969705e+00_dp,&
            4.45472865449403788e+00_dp,  6.75680531467232992e+00_dp,  1.05482558845397225e+01_dp,  5.97841988594928681e-04_dp,  1.47677186318177149e-03_dp,  2.59844834407391709e-03_dp,&
            4.21318849959283811e-03_dp,  6.72427730470716387e-03_dp,  1.07286659349290717e-02_dp,  1.70853685841758293e-02_dp,  2.70540197117113690e-02_dp,  4.25071646238493889e-02_dp,&
            6.62263969430600397e-02_dp,  1.02319734751618513e-01_dp,  1.56818913364540680e-01_dp,  2.38545154014374761e-01_dp,  3.60397852411913966e-01_dp,  5.41391349029674895e-01_dp,&
            8.10266992654025575e-01_dp,  1.21316027447477337e+00_dp,  1.83411013311083582e+00_dp,  2.86864360132638829e+00_dp,  5.06499212313868163e+00_dp]
       aw%aw_erange_matrix(:, 11) = [1.64106873478977560e-04_dp,  8.94356094481026532e-04_dp,  2.34479715468854369e-03_dp,  4.81575108056488989e-03_dp,  8.88843116175322932e-03_dp,&
            1.55765097799911226e-02_dp,  2.65296970497616198e-02_dp,  4.43454032277437946e-02_dp,  7.30554063217297017e-02_dp,  1.18861997867758001e-01_dp,  1.91229739550221989e-01_dp,&
            3.04490668298481315e-01_dp,  4.80195734468752133e-01_dp,  7.50565238695916959e-01_dp,  1.16361704350310702e+00_dp,  1.79107160906205554e+00_dp,  2.74158681135842963e+00_dp,&
            4.18670920782826972e+00_dp,  6.42686158846605249e+00_dp,  1.01490122213306133e+01_dp,  4.24401097476844607e-04_dp,  1.05727589829934018e-03_dp,  1.89172495585899852e-03_dp,&
            3.14454430541335128e-03_dp,  5.16652752063045690e-03_dp,  8.47954378269798539e-03_dp,  1.38517863631827491e-02_dp,  2.24395660034455643e-02_dp,  3.59963271602878956e-02_dp,&
            5.71680679504821404e-02_dp,  8.99177023531648745e-02_dp,  1.40139757841798468e-01_dp,  2.16557288029568279e-01_dp,  3.32058063174325457e-01_dp,  5.05792188779843777e-01_dp,&
            7.66839932609333474e-01_dp,  1.16187995611759587e+00_dp,  1.77553947438556503e+00_dp,  2.80338937361828577e+00_dp,  4.99034689774381057e+00_dp]
       aw%aw_erange_matrix(:, 12) = [1.28568716066204100e-04_dp,  7.03026514521875668e-04_dp,  1.85560664685786446e-03_dp,  3.85208057449708437e-03_dp,  7.21190652297153730e-03_dp,&
            1.28460330737252409e-02_dp,  2.22490411662515028e-02_dp,  3.78016757009908971e-02_dp,  6.32460841694079262e-02_dp,  1.04408351026837459e-01_dp,  1.70278412496789294e-01_dp,&
            2.74610981192654091e-01_dp,  4.38288395194931735e-01_dp,  6.92809118645061894e-01_dp,  1.08549282488148080e+00_dp,  1.68750781965786012e+00_dp,  2.60725900002492317e+00_dp,&
            4.01645618802980753e+00_dp,  6.21599941615518858e+00_dp,  9.89266388698867338e+00_dp,  3.32741224892216562e-04_dp,  8.34534629424527529e-04_dp,  1.51314890244545991e-03_dp,&
            2.56412204028950768e-03_dp,  4.30430865632256462e-03_dp,  7.20813936570132222e-03_dp,  1.19869142915110007e-02_dp,  1.97302600503207887e-02_dp,  3.21119534017258179e-02_dp,&
            5.16847784772300686e-02_dp,  8.23092470536804649e-02_dp,  1.29779454499307473e-01_dp,  2.02740721724053524e-01_dp,  3.14058639815314677e-01_dp,  4.82960697769841130e-01_dp,&
            7.38747791964979639e-01_dp,  1.12847197448742165e+00_dp,  1.73718758800898043e+00_dp,  2.76055445478163319e+00_dp,  4.94138714425316028e+00_dp]
       aw%aw_erange_matrix(:, 13) = [1.06385133376617129e-04_dp,  5.83370449728654429e-04_dp,  1.54858333545919999e-03_dp,  3.24385777928466301e-03_dp,  6.14515163133302119e-03_dp,&
            1.10907193996734339e-02_dp,  1.94655108102521276e-02_dp,  3.34965919996977265e-02_dp,  5.67188754614086563e-02_dp,  9.46864375291209259e-02_dp,  1.56042379020865263e-01_dp,&
            2.54114990033732113e-01_dp,  4.09287211708037946e-01_dp,  6.52511637001359834e-01_dp,  1.03057187695187857e+00_dp,  1.61420249636376889e+00_dp,  2.51159646991932384e+00_dp,&
            3.89457106698527467e+00_dp,  6.06439084840805709e+00_dp,  9.70773966211605277e+00_dp,  2.75500745629308705e-04_dp,  6.94914441903353897e-04_dp,  1.27416915369170597e-03_dp,&
            2.19363072846082878e-03_dp,  3.74574197015587265e-03_dp,  6.37142538505014715e-03_dp,  1.07417435383839119e-02_dp,  1.78979789245490017e-02_dp,  2.94548268302540286e-02_dp,&
            4.78947831060808488e-02_dp,  7.69998423782822516e-02_dp,  1.22485294515095461e-01_dp,  1.92932600352046763e-01_dp,  3.01183266467369593e-01_dp,  4.66514755988375607e-01_dp,&
            7.18387752933783763e-01_dp,  1.10413579938484174e+00_dp,  1.70914727341012851e+00_dp,  2.72917918879560562e+00_dp,  4.90554472200116276e+00_dp]
       aw%aw_erange_matrix(:, 14) = [9.11299266109850996e-05_dp,  5.00955114194532930e-04_dp,  1.33647988051104581e-03_dp,  2.82165743093389315e-03_dp,  5.39955123421062468e-03_dp,&
            9.85332451242546502e-03_dp,  1.74849063769687285e-02_dp,  3.04045333851281283e-02_dp,  5.19884132146004527e-02_dp,  8.75805986006403664e-02_dp,  1.45554197257650425e-01_dp,&
            2.38902835711941874e-01_dp,  3.87614188439427632e-01_dp,  6.22204747423747206e-01_dp,  9.89024982295867328e-01_dp,  1.55845308345162348e+00_dp,  2.43850005542893200e+00_dp,&
            3.80105900780219708e+00_dp,  5.94768748398844416e+00_dp,  9.56502795220571223e+00_dp,  2.36123336764438414e-04_dp,  5.98562968184080508e-04_dp,  1.10826359627376253e-03_dp,&
            1.93399216337780569e-03_dp,  3.34948608200666496e-03_dp,  5.77034764998995628e-03_dp,  9.83706815425649042e-03_dp,  1.65534576760454771e-02_dp,  2.74876882326032593e-02_dp,&
            4.50662590827335952e-02_dp,  7.30078523507230459e-02_dp,  1.16963153229053499e-01_dp,  1.85459608227162981e-01_dp,  2.91315098444254794e-01_dp,  4.53841813215039913e-01_dp,&
            7.02623714126374366e-01_dp,  1.08521855803019984e+00_dp,  1.68728817043916113e+00_dp,  2.70468505928098901e+00_dp,  4.87757399779168743e+00_dp]
       aw%aw_erange_matrix(:, 15) = [7.99513211818079095e-05_dp,  4.40480100273793647e-04_dp,  1.18043287221358643e-03_dp,  2.50972866667220542e-03_dp,  4.84536770340586800e-03_dp,&
            8.92681999663503915e-03_dp,  1.59901889293979985e-02_dp,  2.80527231464049974e-02_dp,  4.83635179114337843e-02_dp,  8.20972791919212935e-02_dp,  1.37407975722330933e-01_dp,&
            2.27015822801634820e-01_dp,  3.70583378740419300e-01_dp,  5.98265757020446043e-01_dp,  9.56051274598360368e-01_dp,  1.51401624986978511e+00_dp,  2.38001241056539525e+00_dp,&
            3.72598892795090819e+00_dp,  5.85374707237895908e+00_dp,  9.44991526480561461e+00_dp,  2.07259523146821357e-04_dp,  5.27743168436076507e-04_dp,  9.85682497105203016e-04_dp,&
            1.74056798836079831e-03_dp,  3.05118589870811367e-03_dp,  5.31311515597368816e-03_dp,  9.14249557319238466e-03_dp,  1.55127983671289656e-02_dp,  2.59540846387241245e-02_dp,&
            4.28465752176474188e-02_dp,  6.98561346118610760e-02_dp,  1.12578862868986593e-01_dp,  1.79495499922559476e-01_dp,  2.83401465126374263e-01_dp,  4.43634265712469833e-01_dp,&
            6.89877063657090339e-01_dp,  1.06987289829179089e+00_dp,  1.66951450142050417e+00_dp,  2.68474521792867948e+00_dp,  4.85481079054310882e+00_dp]
       aw%aw_erange_matrix(:, 16) = [7.13831975957801421e-05_dp,  3.94070557414630566e-04_dp,  1.06039838107656217e-03_dp,  2.26887735894547218e-03_dp,  4.41516125917593150e-03_dp,&
            8.20292284367789629e-03_dp,  1.48143307208321289e-02_dp,  2.61901631484158237e-02_dp,  4.54744113356478005e-02_dp,  7.77009627515651835e-02_dp,  1.30840559929286476e-01_dp,&
            2.17383566833786041e-01_dp,  3.56717684228294363e-01_dp,  5.78690630181714849e-01_dp,  9.28980394500882545e-01_dp,  1.47740187212987562e+00_dp,  2.33166526247446670e+00_dp,&
            3.66376254665971857e+00_dp,  5.77570286165077995e+00_dp,  9.35411658107688737e+00_dp,  1.85129909885371226e-04_dp,  4.73313642877459817e-04_dp,  8.91030234594590402e-04_dp,&
            1.59011031429565509e-03_dp,  2.81702158923678942e-03_dp,  4.95098003577831326e-03_dp,  8.58806747962608148e-03_dp,  1.46764266654769941e-02_dp,  2.47140097130862518e-02_dp,&
            4.10417708233977371e-02_dp,  6.72804167578404944e-02_dp,  1.08978898750293735e-01_dp,  1.74576875600901571e-01_dp,  2.76848637903514494e-01_dp,  4.35150781120790531e-01_dp,&
            6.79248731263868200e-01_dp,  1.05704272734518123e+00_dp,  1.65462493376700914e+00_dp,  2.66802416177299628e+00_dp,  4.83572672360228850e+00_dp]
       aw%aw_erange_matrix(:, 17) = [6.45919046172234337e-05_dp,  3.57244359004111266e-04_dp,  9.64947091415230274e-04_dp,  2.07669093975910773e-03_dp,  4.07020796657402498e-03_dp,&
            7.61911830524246176e-03_dp,  1.38602950606173652e-02_dp,  2.46700717584350873e-02_dp,  4.31034385684379573e-02_dp,  7.40744584914696386e-02_dp,  1.25397230109973362e-01_dp,&
            2.09364698321958947e-01_dp,  3.45127358310070076e-01_dp,  5.62266257195702113e-01_dp,  9.06188494647871989e-01_dp,  1.44647873670279070e+00_dp,  2.29071990594395736e+00_dp,&
            3.61093746195846910e+00_dp,  5.70932120129661769e+00_dp,  9.27251323213914169e+00_dp,  1.67585006389789945e-04_dp,  4.30065142823307851e-04_dp,  8.15501787443688214e-04_dp,&
            1.46924809715015601e-03_dp,  2.62738699777469598e-03_dp,  4.65542668219270928e-03_dp,  8.13250988854261406e-03_dp,  1.39851462776275932e-02_dp,  2.36836605335067965e-02_dp,&
            3.95350223678160792e-02_dp,  6.51206049207332244e-02_dp,  1.05947940902738805e-01_dp,  1.70420066107486362e-01_dp,  2.71291456476858317e-01_dp,  4.27933455588147660e-01_dp,&
            6.70181271059925754e-01_dp,  1.04607118754885398e+00_dp,  1.64187059552129311e+00_dp,  2.65368847252926798e+00_dp,  4.81936838224020203e+00_dp]
       aw%aw_erange_matrix(:, 18) = [5.90671705438591002e-05_dp,  3.27255606259535938e-04_dp,  8.87065468156453198e-04_dp,  1.91938066912922299e-03_dp,  3.78659468047330564e-03_dp,&
            7.13661315013889956e-03_dp,  1.30675308560924963e-02_dp,  2.34003282960270620e-02_dp,  4.11132236983706573e-02_dp,  7.10164792782264631e-02_dp,  1.20787940407201908e-01_dp,&
            2.02548136693446179e-01_dp,  3.35239539268372189e-01_dp,  5.48208274009858720e-01_dp,  8.86621522655313465e-01_dp,  1.41985849853084023e+00_dp,  2.25538654350318835e+00_dp,&
            3.56525777952575273e+00_dp,  5.65182136139268199e+00_dp,  9.20173727079627390e+00_dp,  1.53308858399679005e-04_dp,  3.94802873973349074e-04_dp,  7.53679480203142270e-04_dp,&
            1.36971213154538172e-03_dp,  2.47007058014086459e-03_dp,  4.40855407450766335e-03_dp,  7.74972129914588363e-03_dp,  1.34012766381315894e-02_dp,  2.28093810679012438e-02_dp,&
            3.82511320233823726e-02_dp,  6.32731384704180860e-02_dp,  1.03346060178322777e-01_dp,  1.66839925784102922e-01_dp,  2.66490624991893299e-01_dp,  4.21681106402822914e-01_dp,&
            6.62306829712637768e-01_dp,  1.03652366240097460e+00_dp,  1.63075503979098335e+00_dp,  2.64118514241116120e+00_dp,  4.80510334732246758e+00_dp]
       aw%aw_erange_matrix(:, 19) = [3.28224484567984215e-05_dp,  1.84280197576634352e-04_dp,  5.13129793251420252e-04_dp,  1.15533955303439673e-03_dp,  2.38715227416747520e-03_dp,&
            4.71232336588018729e-03_dp,  9.01068360323859820e-03_dp,  1.67877454786610168e-02_dp,  3.05777352644954897e-02_dp,  5.45820241665686198e-02_dp,  9.56682633955002193e-02_dp,&
            1.64917942299495190e-01_dp,  2.80002569262169099e-01_dp,  4.68811468855826918e-01_dp,  7.74997699140769214e-01_dp,  1.26661171850926046e+00_dp,  2.05033163283972186e+00_dp,&
            3.29831462211391546e+00_dp,  5.31390007833082478e+00_dp,  8.78400792567013156e+00_dp,  8.54353690799433902e-05_dp,  2.25944182794824420e-04_dp,  4.53440750018906298e-04_dp,&
            8.75662275353216932e-04_dp,  1.66953305618841390e-03_dp,  3.12340547841677462e-03_dp,  5.71770714816871497e-03_dp,  1.02484381286729095e-02_dp,  1.80155532583461571e-02_dp,&
            3.11124892427445850e-02_dp,  5.28682739024428569e-02_dp,  8.85173770573338542e-02_dp,  1.46210202743524420e-01_dp,  2.38544300502429640e-01_dp,  3.84945819661920263e-01_dp,&
            6.15657996825351939e-01_dp,  9.79571892515397891e-01_dp,  1.56411282476503422e+00_dp,  2.56602511289843704e+00_dp,  4.71939831326842363e+00_dp]
       aw%aw_erange_matrix(:, 20) = [2.32962402161399536e-05_dp,  1.32055981092677323e-04_dp,  3.74839977969314763e-04_dp,  8.67018033095867083e-04_dp,  1.84480865155746280e-03_dp,&
            3.74520339779335545e-03_dp,  7.34592546659092009e-03_dp,  1.40019747000581871e-02_dp,  2.60313282678624226e-02_dp,  4.73329996204092801e-02_dp,  8.43648984201871011e-02_dp,&
            1.47673591218463379e-01_dp,  2.54264341878251476e-01_dp,  4.31247942064372725e-01_dp,  7.21449996818138595e-01_dp,  1.19217247906242885e+00_dp,  1.94962022268630952e+00_dp,&
            3.16596292239029875e+00_dp,  5.14506868844995147e+00_dp,  8.57409525716754750e+00_dp,  6.07636192166222057e-05_dp,  1.63793630532315700e-04_dp,  3.40174856657600921e-04_dp,&
            6.82306032697698860e-04_dp,  1.34385222906203733e-03_dp,  2.58273566631901609e-03_dp,  4.83830666876542814e-03_dp,  8.85011586892991454e-03_dp,  1.58425745251531969e-02_dp,&
            2.78122580371543207e-02_dp,  4.79708046156028245e-02_dp,  8.14215630554797898e-02_dp,  1.36187506557822940e-01_dp,  2.24776382379014678e-01_dp,  3.66617269529460332e-01_dp,&
            5.92120682929402009e-01_dp,  9.50565358380333736e-01_dp,  1.52993546398228419e+00_dp,  2.52733955994408444e+00_dp,  4.67531311945309902e+00_dp]
       aw%aw_erange_matrix(:, 21) = [1.82768686384135133e-05_dp,  1.04417347919080671e-04_dp,  3.01000951462268346e-04_dp,  7.10847810668938790e-04_dp,  1.54571361298362132e-03_dp,&
            3.20181862357970517e-03_dp,  6.39397242950541685e-03_dp,  1.23832469483872113e-02_dp,  2.33510009743427664e-02_dp,  4.30030599635740085e-02_dp,  7.75327408507942573e-02_dp,&
            1.37137607870212203e-01_dp,  2.38383717160715658e-01_dp,  4.07862967786620456e-01_dp,  6.87842494503462798e-01_dp,  1.14511059809369953e+00_dp,  1.88553654502488044e+00_dp,&
            3.08128096542072605e+00_dp,  5.03656339350511306e+00_dp,  8.43873603575548970e+00_dp,  4.77510514538043561e-05_dp,  1.30725348199526632e-04_dp,  2.78839678510383097e-04_dp,&
            5.74965922084611988e-04_dp,  1.15864133134080285e-03_dp,  2.26903071507058825e-03_dp,  4.31941617512579795e-03_dp,  8.01292114296655450e-03_dp,  1.45246256387368058e-02_dp,&
            2.57871124684656526e-02_dp,  4.49334652585602345e-02_dp,  7.69779065263630213e-02_dp,  1.29854768623762662e-01_dp,  2.16005958323979730e-01_dp,  3.54854744034257708e-01_dp,&
            5.76915783269319360e-01_dp,  9.31723856159322739e-01_dp,  1.50764439582344267e+00_dp,  2.50205346399103279e+00_dp,  4.64650701658826915e+00_dp]
       aw%aw_erange_matrix(:, 22) = [1.51473404015659037e-05_dp,  8.71243659234028809e-05_dp,  2.54468973576208357e-04_dp,  6.11299999793072594e-04_dp,  1.35240591857200892e-03_dp,&
            2.84571706861578774e-03_dp,  5.76205885506093137e-03_dp,  1.12962211654832664e-02_dp,  2.15323161898956957e-02_dp,  4.00375633330651501e-02_dp,  7.28140302835723480e-02_dp,&
            1.29805206598339162e-01_dp,  2.27255100152298689e-01_dp,  3.91372245954157760e-01_dp,  6.64007613693874399e-01_dp,  1.11156238321900980e+00_dp,  1.83964746537561008e+00_dp,&
            3.02040766915205072e+00_dp,  4.95832144405293640e+00_dp,  8.34090257682230707e+00_dp,  3.96314252090897067e-05_dp,  1.09946771533849565e-04_dp,  2.39749996962043585e-04_dp,&
            5.05239856466603251e-04_dp,  1.03620411711276652e-03_dp,  2.05866032990039511e-03_dp,  3.96726052268024654e-03_dp,  7.43881961236985724e-03_dp,  1.36125006461492604e-02_dp,&
            2.43739175756457205e-02_dp,  4.27979881349007454e-02_dp,  7.38322409424063963e-02_dp,  1.25343663588691395e-01_dp,  2.09722480931934130e-01_dp,  3.46383680564075980e-01_dp,&
            5.65915026807195964e-01_dp,  9.18039164611651803e-01_dp,  1.49140766131358671e+00_dp,  2.48360686525505825e+00_dp,  4.62549692830303449e+00_dp]
       aw%aw_erange_matrix(:, 23) = [1.29964471150150904e-05_dp,  7.52038805186812202e-05_dp,  2.22196529687256864e-04_dp,  5.41590274135701007e-04_dp,  1.21550187536022020e-03_dp,&
            2.59071146873320968e-03_dp,  5.30494890727680173e-03_dp,  1.05027659096567447e-02_dp,  2.01940756625821997e-02_dp,  3.78396999976663775e-02_dp,  6.92940663971308296e-02_dp,&
            1.24303453575809700e-01_dp,  2.18860535391796729e-01_dp,  3.78872950063046332e-01_dp,  6.45862912815095225e-01_dp,  1.08592321681505455e+00_dp,  1.80445573105089041e+00_dp,&
            2.97358732052783425e+00_dp,  4.89799912289900607e+00_dp,  8.26534218773699791e+00_dp,  3.40471852492794184e-05_dp,  9.55719144498884892e-05_dp,  2.12380450750809476e-04_dp,&
            4.55655798037782084e-04_dp,  9.47931731475714148e-04_dp,  1.90530071147553015e-03_dp,  3.70815234948269814e-03_dp,  7.01300923792264747e-03_dp,  1.29311616612344006e-02_dp,&
            2.33115500084998054e-02_dp,  4.11833744572942667e-02_dp,  7.14413307639191336e-02_dp,  1.21898463543825414e-01_dp,  2.04902638957595640e-01_dp,  3.39859968590963457e-01_dp,&
            5.57413284554014177e-01_dp,  9.07431857378405593e-01_dp,  1.47879445892678762e+00_dp,  2.46926002025859992e+00_dp,  4.60915876949644421e+00_dp]
       aw%aw_erange_matrix(:, 24) = [1.14207048273847982e-05_dp,  6.64484130614495310e-05_dp,  1.98364705632591312e-04_dp,  4.89680342568533803e-04_dp,  1.11257515254348303e-03_dp,&
            2.39722037536997104e-03_dp,  4.95521165465335599e-03_dp,  9.89119282393999266e-03_dp,  1.91558185967471602e-02_dp,  3.61245234039200835e-02_dp,  6.65327103010643489e-02_dp,&
            1.19966975407711327e-01_dp,  2.12215600247272179e-01_dp,  3.68940412206329449e-01_dp,  6.31393682491580011e-01_dp,  1.06541330659025468e+00_dp,  1.77622627153034740e+00_dp,&
            2.93594113554789704e+00_dp,  4.84940422326652865e+00_dp,  8.20438542819059968e+00_dp,  2.99538235601905597e-05_dp,  8.49804556698071739e-05_dp,  1.92001219094591990e-04_dp,&
            4.18248947198593781e-04_dp,  8.80584218195854191e-04_dp,  1.78723479963482403e-03_dp,  3.50716773455697671e-03_dp,  6.68055972088156622e-03_dp,  1.23961398099230700e-02_dp,&
            2.24730180647289973e-02_dp,  3.99030113422094643e-02_dp,  6.95373455443270422e-02_dp,  1.19144298782168687e-01_dp,  2.01035969122660890e-01_dp,  3.34609664670065254e-01_dp,&
            5.50551680495186879e-01_dp,  8.98850509114371254e-01_dp,  1.46857223003089277e+00_dp,  2.45762167144899246e+00_dp,  4.59590654805552301e+00_dp]
       aw%aw_erange_matrix(:, 25) = [1.02129853665488214e-05_dp,  5.97224033665720453e-05_dp,  1.79968001876572704e-04_dp,  4.49310761967231735e-04_dp,  1.03186306193965745e-03_dp,&
            2.24429055378175648e-03_dp,  4.67683405379132593e-03_dp,  9.40136451442429369e-03_dp,  1.83196550766199957e-02_dp,  3.47364191544349277e-02_dp,  6.42881061852892816e-02_dp,&
            1.16428075481103732e-01_dp,  2.06773452807062580e-01_dp,  3.60779453324731381e-01_dp,  6.19470494647045289e-01_dp,  1.04846815361171242e+00_dp,  1.75284955056996217e+00_dp,&
            2.90470529324109350e+00_dp,  4.80902024684955354e+00_dp,  8.15366880961822815e+00_dp,  2.68148886718955737e-05_dp,  7.68210992565900367e-05_dp,  1.76153395456738691e-04_dp,&
            3.88828307524828219e-04_dp,  8.27109072609790168e-04_dp,  1.69277491364309648e-03_dp,  3.34534816880062945e-03_dp,  6.41142722928096046e-03_dp,  1.19609238164291500e-02_dp,&
            2.17879658757671769e-02_dp,  3.88529268269094813e-02_dp,  6.79702850602889735e-02_dp,  1.16870209587740290e-01_dp,  1.97833926492451712e-01_dp,  3.30250262574636633e-01_dp,&
            5.44840978502778017e-01_dp,  8.91694374810249268e-01_dp,  1.46003514088497854e+00_dp,  2.44789415028531510e+00_dp,  4.58483112275295301e+00_dp]
       aw%aw_erange_matrix(:, 26) = [8.47669796166132796e-06_dp,  5.00242837971973656e-05_dp,  1.53275574109323258e-04_dp,  3.90182608568225638e-04_dp,  9.12420752458163536e-04_dp,&
            2.01578379024756416e-03_dp,  4.25730835582959020e-03_dp,  8.65759910626789746e-03_dp,  1.70415533953586391e-02_dp,  3.26021220288412600e-02_dp,  6.08186688565086689e-02_dp,&
            1.10932128458666804e-01_dp,  1.98285574595964298e-01_dp,  3.48001922484235227e-01_dp,  6.00737297535444026e-01_dp,  1.02176139285064571e+00_dp,  1.71590475217036720e+00_dp,&
            2.85522414136309743e+00_dp,  4.74492660635849450e+00_dp,  8.07306337224151704e+00_dp,  2.22992568351785837e-05_dp,  6.50136849135791761e-05_dp,  1.52942130635250624e-04_dp,&
            3.45130342825854480e-04_dp,  7.46765182067150022e-04_dp,  1.54955133592102182e-03_dp,  3.09811531544227498e-03_dp,  5.99752381829290158e-03_dp,  1.12877049899231616e-02_dp,&
            2.07227873132566805e-02_dp,  3.72125394571043919e-02_dp,  6.55119435253443683e-02_dp,  1.13288963572479862e-01_dp,  1.92773634125936455e-01_dp,  3.23339065790839486e-01_dp,&
            5.35761993096250388e-01_dp,  8.80290439328862417e-01_dp,  1.44640642561162580e+00_dp,  2.43235010495084447e+00_dp,  4.56713502379775083e+00_dp]
       aw%aw_erange_matrix(:, 27) = [4.77162217975052653e-06_dp,  2.91668856331809356e-05_dp,  9.48821379855366293e-05_dp,  2.57598292613655199e-04_dp,  6.37572758202925127e-04_dp,&
            1.47742295887662387e-03_dp,  3.24827632234858232e-03_dp,  6.83617387254335015e-03_dp,  1.38616547779678596e-02_dp,  2.72171464660654339e-02_dp,  5.19550945129050798e-02_dp,&
            9.67332163320293176e-02_dp,  1.76134574373753433e-01_dp,  3.14350903325736952e-01_dp,  5.50994272272958718e-01_dp,  9.50322068117752772e-01_dp,  1.61643700300510540e+00_dp,&
            2.72126761065023093e+00_dp,  4.57063870580321119e+00_dp,  7.85315353792608306e+00_dp,  1.26470920648656757e-05_dp,  3.93709424700992549e-05_dp,  1.00885940425964106e-04_dp,&
            2.43669035312139701e-04_dp,  5.55019876746663788e-04_dp,  1.20021707946449396e-03_dp,  2.48403177561761770e-03_dp,  4.95320350081516082e-03_dp,  9.56550428124965796e-03_dp,&
            1.79641999719573932e-02_dp,  3.29171066132178045e-02_dp,  5.90100020341800327e-02_dp,  1.03730646694039405e-01_dp,  1.79155565358709440e-01_dp,  3.04600128154180194e-01_dp,&
            5.10981306992512874e-01_dp,  8.48989087313390955e-01_dp,  1.40884114975024644e+00_dp,  2.38940742183020793e+00_dp,  4.51825790067075239e+00_dp]
       aw%aw_erange_matrix(:, 28) = [3.42200124612533428e-06_dp,  2.14795977800482488e-05_dp,  7.27931060367611917e-05_dp,  2.05652828701512050e-04_dp,  5.26124616740868763e-04_dp,&
            1.25246741430834582e-03_dp,  2.81569672700486551e-03_dp,  6.03795545760961710e-03_dp,  1.24413051501760675e-02_dp,  2.47714301236555100e-02_dp,  4.78697619177791758e-02_dp,&
            9.01024065490877851e-02_dp,  1.65668103195190375e-01_dp,  2.98282223223030774e-01_dp,  5.27015875188172278e-01_dp,  9.15593507937763729e-01_dp,  1.56772415593837122e+00_dp,&
            2.65525119808757593e+00_dp,  4.48431225145893819e+00_dp,  7.74382455050710927e+00_dp,  9.12267553996344091e-06_dp,  2.97782938235843919e-05_dp,  8.04762008480940360e-05_dp,&
            2.02054552088428699e-04_dp,  4.73662260510720275e-04_dp,  1.04800747447037716e-03_dp,  2.21052298276957397e-03_dp,  4.47925090915589005e-03_dp,  8.77100834263967732e-03_dp,&
            1.66730757520466601e-02_dp,  3.08806180820769202e-02_dp,  5.58915044031746411e-02_dp,  9.90980169561825330e-02_dp,  1.72492449290110456e-01_dp,  2.95352774830862530e-01_dp,&
            4.98659663798053099e-01_dp,  8.33325756198420309e-01_dp,  1.38995327107408162e+00_dp,  2.36775916083012872e+00_dp,  4.49362344166366778e+00_dp]
       aw%aw_erange_matrix(:, 29) = [2.70857528023669478e-06_dp,  1.73824675975106526e-05_dp,  6.08007017131558374e-05_dp,  1.76793729684337786e-04_dp,  4.62880220912142407e-04_dp,&
            1.12249782389384407e-03_dp,  2.56197738480458100e-03_dp,  5.56375798474603462e-03_dp,  1.15881961261663814e-02_dp,  2.32883361269934745e-02_dp,  4.53714740249268472e-02_dp,&
            8.60171390199267055e-02_dp,  1.59176614013866874e-01_dp,  2.88256566568690420e-01_dp,  5.11975058103568248e-01_dp,  8.93705673791643340e-01_dp,  1.53689447832666781e+00_dp,&
            2.61332238621594604e+00_dp,  4.42932837952966896e+00_dp,  7.67404440659009524e+00_dp,  7.25671600966090790e-06_dp,  2.46108812587894022e-05_dp,  6.91283002216229860e-05_dp,&
            1.78280345108654118e-04_dp,  4.26255177147598569e-04_dp,  9.57935241483460109e-04_dp,  2.04659436769854316e-03_dp,  4.19208858133998229e-03_dp,  8.28508624517085290e-03_dp,&
            1.58768608867642276e-02_dp,  2.96155024326393256e-02_dp,  5.39414460482286126e-02_dp,  9.61839308156842016e-02_dp,  1.68278613421956519e-01_dp,  2.89476370736281186e-01_dp,&
            4.90796176703813092e-01_dp,  8.23293620189284980e-01_dp,  1.37782304675651202e+00_dp,  2.35383537187622638e+00_dp,  4.47778069418177083e+00_dp]
       aw%aw_erange_matrix(:, 30) = [2.26251445525014870e-06_dp,  1.48039880636135852e-05_dp,  5.31416128088749591e-05_dp,  1.58040646840163678e-04_dp,  4.21147625899582134e-04_dp,&
            1.03563677296014656e-03_dp,  2.39060962537256377e-03_dp,  5.24060798095646665e-03_dp,  1.10023814471914491e-02_dp,  2.22631671738785647e-02_dp,  4.36345263985256498e-02_dp,&
            8.31622337480630441e-02_dp,  1.54619409838966410e-01_dp,  2.81189489616427979e-01_dp,  5.01334015720054205e-01_dp,  8.78170141334107202e-01_dp,  1.51494988258760133e+00_dp,&
            2.58340536033461099e+00_dp,  4.39002046838892390e+00_dp,  7.62408789263364106e+00_dp,  6.08870912420167293e-06_dp,  2.13307424187180562e-05_dp,  6.17488367021811645e-05_dp,&
            1.62519357537921707e-04_dp,  3.94389143487223006e-04_dp,  8.96733927318684983e-04_dp,  1.93421640753366165e-03_dp,  3.99374279098910039e-03_dp,  7.94726547986469162e-03_dp,&
            1.53201567889741652e-02_dp,  2.87264724416089259e-02_dp,  5.25648940004549425e-02_dp,  9.41185062368712216e-02_dp,  1.65281005493465305e-01_dp,  2.85282262721775703e-01_dp,&
            4.85167472736483585e-01_dp,  8.16094915370970453e-01_dp,  1.36910268768840915e+00_dp,  2.34381536801679990e+00_dp,  4.46638053432672422e+00_dp]
       aw%aw_erange_matrix(:, 31) = [1.95520309160147283e-06_dp,  1.30177455610905032e-05_dp,  4.77699277056753375e-05_dp,  1.44704490268162631e-04_dp,  3.91112695871576065e-04_dp,&
            9.72507920670827395e-04_dp,  2.26505403621023388e-03_dp,  5.00223926417243947e-03_dp,  1.05677607556923341e-02_dp,  2.14987830405207782e-02_dp,  4.23337650579894906e-02_dp,&
            8.10160083972915057e-02_dp,  1.51181699507279416e-01_dp,  2.75842138921684243e-01_dp,  4.93260368570506924e-01_dp,  8.66354247590333748e-01_dp,  1.49822391705281710e+00_dp,&
            2.56056177053762957e+00_dp,  4.35996302045826134e+00_dp,  7.58584741797931628e+00_dp,  5.28329779560503016e-06_dp,  1.90417363895934812e-05_dp,  5.64974872526313138e-05_dp,&
            1.51136234246477072e-04_dp,  3.71130443441160750e-04_dp,  8.51695348297861333e-04_dp,  1.85095701759959981e-03_dp,  3.84595112808428473e-03_dp,  7.69430931514070420e-03_dp,&
            1.49015115640716844e-02_dp,  2.80553763496916748e-02_dp,  5.15222643888510815e-02_dp,  9.25493475814019861e-02_dp,  1.62997391612473730e-01_dp,  2.82079259084697609e-01_dp,&
            4.80859508026839078e-01_dp,  8.10575201442163107e-01_dp,  1.36240694358376646e+00_dp,  2.33611580283512099e+00_dp,  4.45762085471925396e+00_dp]
       aw%aw_erange_matrix(:, 32) = [1.72960139273996152e-06_dp,  1.17001313607044589e-05_dp,  4.37650399987639228e-05_dp,  1.34646135657629157e-04_dp,  3.68237981352100647e-04_dp,&
            9.24047534313781802e-04_dp,  2.16804637027645259e-03_dp,  4.81707060286444468e-03_dp,  1.02285848708239310e-02_dp,  2.08998908680933203e-02_dp,  4.13110880916842740e-02_dp,&
            7.93234571129355220e-02_dp,  1.48463301946358234e-01_dp,  2.71603440324783507e-01_dp,  4.86846776771414413e-01_dp,  8.56949848514978574e-01_dp,  1.48488918635942313e+00_dp,&
            2.54232388180733038e+00_dp,  4.33593842613987768e+00_dp,  7.55525676106186950e+00_dp,  4.69160847264410453e-06_dp,  1.73423940337633487e-05_dp,  5.25344729271504530e-05_dp,&
            1.42442614305894703e-04_dp,  3.53216246020351498e-04_dp,  8.16776990505050119e-04_dp,  1.78605723081466314e-03_dp,  3.73022452802260294e-03_dp,  7.49545986497960476e-03_dp,&
            1.45712907599660872e-02_dp,  2.75244331035289849e-02_dp,  5.06951675615561689e-02_dp,  9.13015702641552934e-02_dp,  1.61177550862612262e-01_dp,  2.79521774387272437e-01_dp,&
            4.77413832677281047e-01_dp,  8.06153910734494272e-01_dp,  1.35703777400223546e+00_dp,  2.32993793434914931e+00_dp,  4.45059264173135372e+00_dp]
       aw%aw_erange_matrix(:, 33) = [1.41886137796995782e-06_dp,  9.87436540219261310e-06_dp,  3.81420863321894053e-05_dp,  1.20328937033829805e-04_dp,  3.35306348287066293e-04_dp,&
            8.53643157287757213e-04_dp,  2.02606093436642374e-03_dp,  4.54436729619164034e-03_dp,  9.72644390103706222e-03_dp,  2.00092321896778690e-02_dp,  3.97841823036489475e-02_dp,&
            7.67876127175006890e-02_dp,  1.44377938952077817e-01_dp,  2.65215781458406519e-01_dp,  4.77157897649237217e-01_dp,  8.42711908654622754e-01_dp,  1.46466244711069460e+00_dp,&
            2.51461522969896301e+00_dp,  4.29939105591348802e+00_dp,  7.50867691488572575e+00_dp,  3.87596278284403415e-06_dp,  1.49686541796921346e-05_dp,  4.68892269201415766e-05_dp,&
            1.29887348425810148e-04_dp,  3.27092391898948858e-04_dp,  7.65470839571249194e-04_dp,  1.69010905179960060e-03_dp,  3.55824376587745303e-03_dp,  7.19863210358792378e-03_dp,&
            1.40764462054979219e-02_dp,  2.67260753852816696e-02_dp,  4.94477050531962614e-02_dp,  8.94144794385312719e-02_dp,  1.58418528983117257e-01_dp,  2.75635868471146572e-01_dp,&
            4.72168173179626172e-01_dp,  7.99411892474264363e-01_dp,  1.34884014613905223e+00_dp,  2.32049905986289229e+00_dp,  4.43985498462144701e+00_dp]
       aw%aw_erange_matrix(:, 34) = [1.30682076215574168e-06_dp,  9.21230103419385707e-06_dp,  3.60777820506542435e-05_dp,  1.15007390574968328e-04_dp,  3.22942749282655722e-04_dp,&
            8.26999800237385391e-04_dp,  1.97198120105341245e-03_dp,  4.43994262135867599e-03_dp,  9.53329117896053665e-03_dp,  1.96653009055034987e-02_dp,  3.91925682314573712e-02_dp,&
            7.58021570974209136e-02_dp,  1.42786148248512379e-01_dp,  2.62721115240734437e-01_dp,  4.73366064322675417e-01_dp,  8.37129449336611819e-01_dp,  1.45671906055837219e+00_dp,&
            2.50371868274963472e+00_dp,  4.28500293152441092e+00_dp,  7.49032450783281689e+00_dp,  3.58166740544859434e-06_dp,  1.41012624824627562e-05_dp,  4.47894722858063098e-05_dp,&
            1.25160938055950030e-04_dp,  3.17174709330518730e-04_dp,  7.45865050527678465e-04_dp,  1.65324808989842547e-03_dp,  3.49187701447795399e-03_dp,  7.08364872000395891e-03_dp,&
            1.38841187666644134e-02_dp,  2.64148762682075572e-02_dp,  4.89601819209699238e-02_dp,  8.86752668909496500e-02_dp,  1.57335503252764047e-01_dp,  2.74107634843242409e-01_dp,&
            4.70101764244850173e-01_dp,  7.96752312857495482e-01_dp,  1.34560294170146966e+00_dp,  2.31676949685429490e+00_dp,  4.43561236801076575e+00_dp]
       aw%aw_erange_matrix(:, 35) = [7.71991367601195506e-07_dp,  6.01188017486886296e-06_dp,  2.58331150414066731e-05_dp,  8.79343948397957349e-05_dp,  2.58798759078993318e-04_dp,&
            6.86622104597263264e-04_dp,  1.68348045801247907e-03_dp,  3.87710431958284253e-03_dp,  8.48313765549411902e-03_dp,  1.77813926363193896e-02_dp,  3.59308996111747034e-02_dp,&
            7.03381717934871697e-02_dp,  1.33915686402270723e-01_dp,  2.48756796209642445e-01_dp,  4.52055656622306401e-01_dp,  8.05644178219993679e-01_dp,  1.41177905572475915e+00_dp,&
            2.44190912873904908e+00_dp,  4.20321643292941616e+00_dp,  7.38584390670966684e+00_dp,  2.17516527710297515e-06_dp,  9.83686937874478960e-06_dp,  3.40911794807122775e-05_dp,&
            1.00516175832159038e-04_dp,  2.64611812117782411e-04_dp,  6.40634437941980267e-04_dp,  1.45335802417366814e-03_dp,  3.12886949168914236e-03_dp,  6.45007574789729140e-03_dp,&
            1.28175787260026396e-02_dp,  2.46794247572051199e-02_dp,  4.62278443048538337e-02_dp,  8.45138014752302991e-02_dp,  1.51214003421640153e-01_dp,  2.65438575788372044e-01_dp,&
            4.58342505053100036e-01_dp,  7.81576747787620318e-01_dp,  1.32709377942230344e+00_dp,  2.29542090956463429e+00_dp,  4.41132831206158560e+00_dp]
       aw%aw_erange_matrix(:, 36) = [5.76735002793307808e-07_dp,  4.81796949814678273e-06_dp,  2.18483421872876011e-05_dp,  7.70173393749095747e-05_dp,  2.32217067617773254e-04_dp,&
            6.27214064392758990e-04_dp,  1.55933122602490491e-03_dp,  3.63157033812529315e-03_dp,  8.01975002563170940e-03_dp,  1.69419691855408096e-02_dp,  3.44652936060531184e-02_dp,&
            6.78648367359473981e-02_dp,  1.29874232623644670e-01_dp,  2.42357827527284242e-01_dp,  4.42240387974227989e-01_dp,  7.91076763029657348e-01_dp,  1.39090428178819203e+00_dp,&
            2.41310249516129938e+00_dp,  4.16499789079741589e+00_dp,  7.33692571660345116e+00_dp,  1.66095137224994890e-06_dp,  8.20074753469969660e-06_dp,  2.97676497989321825e-05_dp,&
            9.02363370065398159e-05_dp,  2.42198175651940829e-04_dp,  5.94996626056037070e-04_dp,  1.36547679821607312e-03_dp,  2.96745695023317713e-03_dp,  6.16563639522362070e-03_dp,&
            1.23347764774671582e-02_dp,  2.38881047565255543e-02_dp,  4.49739612649937420e-02_dp,  8.25931421941693417e-02_dp,  1.48374217827051180e-01_dp,  2.61398494781967083e-01_dp,&
            4.52840084356117167e-01_dp,  7.74451476612179168e-01_dp,  1.31838081518574168e+00_dp,  2.28535676109961017e+00_dp,  4.39988104041656669e+00_dp]
       aw%aw_erange_matrix(:, 37) = [4.73886425613891435e-07_dp,  4.17970580864170017e-06_dp,  1.96623716565723957e-05_dp,  7.09049514450060440e-05_dp,  2.17110309698999125e-04_dp,&
            5.93067710371235650e-04_dp,  1.48733455024355546e-03_dp,  3.48814465718405879e-03_dp,  7.74742832481121156e-03_dp,  1.64461239878085849e-02_dp,  3.35957282494034823e-02_dp,&
            6.63917017136938514e-02_dp,  1.27458933211450770e-01_dp,  2.38522094193151818e-01_dp,  4.36341108323591909e-01_dp,  7.82300621989012179e-01_dp,  1.37830238440942998e+00_dp,&
            2.39568197291720386e+00_dp,  4.14185357093737938e+00_dp,  7.30727204549361353e+00_dp,  1.38993180594434485e-06_dp,  7.30994082205168018e-06_dp,  2.73439544949609298e-05_dp,&
            8.43748407830279597e-05_dp,  2.29265931027044855e-04_dp,  5.68425191544899631e-04_dp,  1.31393787419090916e-03_dp,  2.87222570368213378e-03_dp,  5.99696872720659104e-03_dp,&
            1.20472328507921500e-02_dp,  2.34150217578703822e-02_dp,  4.42218217302560376e-02_dp,  8.14375937677677270e-02_dp,  1.46661112136236865e-01_dp,  2.58955474937018260e-01_dp,&
            4.49505769567694891e-01_dp,  7.70126058061773699e-01_dp,  1.31308443450498591e+00_dp,  2.27923439428326180e+00_dp,  4.39291748104259039e+00_dp]
       aw%aw_erange_matrix(:, 38) = [3.12168708718758378e-07_dp,  3.15509043854656020e-06_dp,  1.60396540039556192e-05_dp,  6.05345347901636982e-05_dp,  1.91046499902649901e-04_dp,&
            5.33409518772520698e-04_dp,  1.36030194508581252e-03_dp,  3.23305392937584934e-03_dp,  7.25986670485826487e-03_dp,  1.55533654998133344e-02_dp,  3.20225029785451004e-02_dp,&
            6.37152356941127068e-02_dp,  1.23054399695399289e-01_dp,  2.31504282444295856e-01_dp,  4.25516417063861396e-01_dp,  7.66155682919092751e-01_dp,  1.35506744495752551e+00_dp,&
            2.36350179853319853e+00_dp,  4.09903554897101507e+00_dp,  7.25235124629250461e+00_dp,  9.63437611266755152e-07_dp,  5.84573868356708477e-06_dp,  2.32262863906807196e-05_dp,&
            7.42268033517151106e-05_dp,  2.06579837360269531e-04_dp,  5.21343456951251210e-04_dp,  1.22188281591533582e-03_dp,  2.70100566659986366e-03_dp,  5.69202387249046534e-03_dp,&
            1.15248770221008552e-02_dp,  2.25520303961786774e-02_dp,  4.28447386130575286e-02_dp,  7.93150026366342964e-02_dp,  1.43505171794848924e-01_dp,  2.54443118346391550e-01_dp,&
            4.43332986627030412e-01_dp,  7.62102886337588492e-01_dp,  1.30324574547687333e+00_dp,  2.26785189738067405e+00_dp,  4.37997143688256863e+00_dp]
       aw%aw_erange_matrix(:, 39) = [2.80580408279310689e-07_dp,  2.95049171159065096e-06_dp,  1.52945248324598467e-05_dp,  5.83575109507800420e-05_dp,  1.85496679406350597e-04_dp,&
            5.20572087731371934e-04_dp,  1.33274249195369592e-03_dp,  3.17734742663118618e-03_dp,  7.15281285816080775e-03_dp,  1.53564408426251769e-02_dp,  3.16741120106560844e-02_dp,&
            6.31205011915101227e-02_dp,  1.22072732313017143e-01_dp,  2.29936027072113913e-01_dp,  4.23091764893167077e-01_dp,  7.62531838111124793e-01_dp,  1.34984278928180279e+00_dp,&
            2.35625468033670904e+00_dp,  4.08938104591259144e+00_dp,  7.23995693521429562e+00_dp,  8.80031312565509921e-07_dp,  5.54661609202853559e-06_dp,  2.23609176671629847e-05_dp,&
            7.20599509266696783e-05_dp,  2.01682267113263946e-04_dp,  5.11094418303308790e-04_dp,  1.20171110412647811e-03_dp,  2.66328344812695808e-03_dp,  5.62453458732293469e-03_dp,&
            1.14088214568662284e-02_dp,  2.23596457931184399e-02_dp,  4.25368372706329823e-02_dp,  7.88391630738977944e-02_dp,  1.42796013385649651e-01_dp,  2.53427036209508516e-01_dp,&
            4.41940442033310288e-01_dp,  7.60290074913774427e-01_dp,  1.30102008571134586e+00_dp,  2.26527528542373968e+00_dp,  4.37704095531072479e+00_dp]
       aw%aw_erange_matrix(:, 40) = [2.50037054543459451e-07_dp,  2.75065850253568672e-06_dp,  1.45580032538501094e-05_dp,  5.61886134141293647e-05_dp,  1.79937558126970633e-04_dp,&
            5.07661812572850549e-04_dp,  1.30494106638445761e-03_dp,  3.12101246683672406e-03_dp,  7.04432961318839871e-03_dp,  1.51565421457647235e-02_dp,  3.13199357226953551e-02_dp,&
            6.25151135010186682e-02_dp,  1.21072355633585996e-01_dp,  2.28336294921602012e-01_dp,  4.20616270923935887e-01_dp,  7.58829132331704792e-01_dp,  1.34450082520074798e+00_dp,&
            2.34884061013481915e+00_dp,  4.07949963320537279e+00_dp,  7.22726712269525429e+00_dp,  7.99319498651264064e-07_dp,  5.25167142347067190e-06_dp,  2.14984315244743870e-05_dp,&
            6.98872794461767599e-05_dp,  1.96751022559891727e-04_dp,  5.00742416654297745e-04_dp,  1.18128601236660593e-03_dp,  2.62500956852358778e-03_dp,  5.55594123795717914e-03_dp,&
            1.12906947598504191e-02_dp,  2.21635794060799952e-02_dp,  4.22226943193413917e-02_dp,  7.83531987853733763e-02_dp,  1.42071127423185201e-01_dp,  2.52387601399961081e-01_dp,&
            4.40514903391555690e-01_dp,  7.58433221310045091e-01_dp,  1.29873933283050214e+00_dp,  2.26263422505046297e+00_dp,  4.37403718786888618e+00_dp]
    case(22)
       aw%energy_range(:) = [300.0_dp, 4.00e+02_dp, 5.00e+02_dp, 6.00e+02_dp, 7.00e+02_dp, 8.00e+02_dp,&
            9.00e+02_dp, 1.00e+03_dp, 2.00e+03_dp, 3.00e+03_dp, 4.00e+03_dp, 5.00e+03_dp, 6.00e+03_dp, 7.00e+03_dp,&
            8.00e+03_dp, 9.00e+03_dp, 1.00e+04_dp, 2.00e+04_dp, 3.00e+04_dp, 4.00e+04_dp, 5.00e+04_dp, 6.00e+04_dp,&
            7.00e+04_dp, 8.00e+04_dp, 9.00e+04_dp, 1.00e+05_dp, 2.00e+05_dp, 3.00e+05_dp, 4.00e+05_dp, 5.00e+05_dp,&
            6.00e+05_dp, 7.00e+05_dp, 9.00e+05_dp, 1.00e+06_dp, 2.00e+06_dp, 3.00e+06_dp, 4.00e+06_dp, 8.00e+06_dp,&
            1.00e+07_dp,  2.00e+07_dp]
       aw%aw_erange_matrix(:, 1) = [1.03063435016644759e-03_dp,  5.49923156302779131e-03_dp,  1.38334982710772401e-02_dp,  2.65993900808953014e-02_dp,  4.47520782299964323e-02_dp,&
            6.98068249252753320e-02_dp,  1.04068428172679431e-01_dp,  1.50908691841177584e-01_dp,  2.15093367125171991e-01_dp,  3.03194997198029947e-01_dp,  4.24153351843991056e-01_dp,&
            5.90049532797139786e-01_dp,  8.17165186315059078e-01_dp,  1.12742439261148752e+00_dp,  1.55037456293163300e+00_dp,  2.12598214734496160e+00_dp,  2.90878150747867537e+00_dp,&
            3.97455237146797069e+00_dp,  5.43242203991751627e+00_dp,  7.45067149032892484e+00_dp,  1.03258659770943311e+01_dp,  1.47568560745174491e+01_dp,  2.65267482488290270e-03_dp,&
            6.33294964486558112e-03_dp,  1.04280009978064310e-02_dp,  1.52601415880402238e-02_dp,  2.12939160800245907e-02_dp,  2.91950765345169380e-02_dp,  3.98847415349164389e-02_dp,&
            5.45833554480214117e-02_dp,  7.48722343170218907e-02_dp,  1.02808246584750670e-01_dp,  1.41104546650114959e-01_dp,  1.93377742947796749e-01_dp,  2.64475170911950552e-01_dp,&
            3.60922373755072967e-01_dp,  4.91572352482467723e-01_dp,  6.68626420794888210e-01_dp,  9.09416270057912035e-01_dp,  1.23993354412780010e+00_dp,  1.70291215537995910e+00_dp,&
            2.37993376114488786e+00_dp,  3.47004010223380899e+00_dp,  5.75693742710772938e+00_dp]
       aw%aw_erange_matrix(:, 2) = [8.10419979097196271e-04_dp,  4.32987536073072780e-03_dp,  1.09190123190569411e-02_dp,  2.10767323714248292e-02_dp,  3.56564928575854728e-02_dp,&
            5.60293239853698402e-02_dp,  8.42994325741821121e-02_dp,  1.23562539505384872e-01_dp,  1.78216928035805383e-01_dp,  2.54372027381100541e-01_dp,  3.60416024188965678e-01_dp,&
            5.07804615065230336e-01_dp,  7.12142098042157556e-01_dp,  9.94656213174979431e-01_dp,  1.38422766463101721e+00_dp,  1.92025211199259704e+00_dp,  2.65686848896032091e+00_dp,&
            3.66971228809781413e+00_dp,  5.06805126036200360e+00_dp,  7.02049940645484494e+00_dp,  9.82370441014125895e+00_dp,  1.41744350886459483e+01_dp,  2.08650546514541199e-03_dp,&
            4.99439128734344585e-03_dp,  8.26521775822509387e-03_dp,  1.21897481171777209e-02_dp,  1.71946451409541179e-02_dp,  2.38974280159545208e-02_dp,  3.31536769147314225e-02_dp,&
            4.60989289766344304e-02_dp,  6.42187687672978874e-02_dp,  8.94764759215750105e-02_dp,  1.24503013523261263e-01_dp,  1.72850667563437199e-01_dp,  2.39329044250052747e-01_dp,&
            3.30466087030480693e-01_dp,  4.55174709176111914e-01_dp,  6.25790536020832566e-01_dp,  8.59861415531779083e-01_dp,  1.18365880545163105e+00_dp,  1.64018693622440770e+00_dp,&
            2.31109169778968848e+00_dp,  3.39467469919791842e+00_dp,  5.66979714295159365e+00_dp]
       aw%aw_erange_matrix(:, 3) = [6.72087511475085949e-04_dp,  3.59470541834720032e-03_dp,  9.08388199334733346e-03_dp,  1.75914399761403981e-02_dp,  2.98985789014098169e-02_dp,&
            4.72721062533955350e-02_dp,  7.16685007640940897e-02_dp,  1.05977773122365365e-01_dp,  1.54324000984959686e-01_dp,  2.22471673239026196e-01_dp,  3.18397403077191776e-01_dp,&
            4.53086738924418142e-01_dp,  6.41628569576203733e-01_dp,  9.04712386590862394e-01_dp,  1.27069340352189464e+00_dp,  1.77850551848010774e+00_dp,  2.48195346725823196e+00_dp,&
            3.45653086090150197e+00_dp,  4.81159232043718621e+00_dp,  6.71601195856402455e+00_dp,  9.46655499820343849e+00_dp,  1.37585735658933945e+01_dp,  1.73078487945685843e-03_dp,&
            4.15198025245653354e-03_dp,  6.89993499781980761e-03_dp,  1.02427683744667868e-02_dp,  1.45792067227524021e-02_dp,  2.04898999177307757e-02_dp,  2.87796793557555178e-02_dp,&
            4.05195853742843989e-02_dp,  5.71242252255813182e-02_dp,  8.04874394188137554e-02_dp,  1.13176946315939941e-01_dp,  1.58691959877238359e-01_dp,  2.21806152200116485e-01_dp,&
            3.09039947394767034e-01_dp,  4.29341930072118849e-01_dp,  5.95141807058347960e-01_dp,  8.24148898888018566e-01_dp,  1.14285360673369163e+00_dp,  1.59448465845054477e+00_dp,&
            2.26077227364725486e+00_dp,  3.33952020201084965e+00_dp,  5.60610623943482977e+00_dp]
       aw%aw_erange_matrix(:, 4) = [5.76541798871297640e-04_dp,  3.08657044423512337e-03_dp,  7.81384360765192271e-03_dp,  1.51747461063655789e-02_dp,  2.58955551659944376e-02_dp,&
            4.11627624208335333e-02_dp,  6.28174199310700121e-02_dp,  9.35877186751942342e-02_dp,  1.37382081158007707e-01_dp,  1.99693938226721324e-01_dp,  2.88175538974597800e-01_dp,&
            4.13439277418744755e-01_dp,  5.90160840003604248e-01_dp,  8.38592901620649034e-01_dp,  1.18665791055058212e+00_dp,  1.67290216164992822e+00_dp,  2.35084112371337550e+00_dp,&
            3.29583373130628576e+00_dp,  4.61729059143473908e+00_dp,  6.48429700742921433e+00_dp,  9.19374382724353545e+00_dp,  1.34399421870293594e+01_dp,  1.48505007631269024e-03_dp,&
            3.56922047817995755e-03_dp,  5.95304204495843321e-03_dp,  8.88728889829167902e-03_dp,  1.27488075761912085e-02_dp,  1.80886261992498042e-02_dp,  2.56706382245647236e-02_dp,&
            3.65147544192421236e-02_dp,  5.19801105704495656e-02_dp,  7.39057929346307041e-02_dp,  1.04808005358250939e-01_dp,  1.48140573971585643e-01_dp,  2.08643744206222864e-01_dp,&
            2.92826030129254233e-01_dp,  4.09658325664241707e-01_dp,  5.71640483938300803e-01_dp,  7.96609091792657598e-01_dp,  1.11123351656180658e+00_dp,  1.55893392619487359e+00_dp,&
            2.22152972695046858e+00_dp,  3.29646371471532484e+00_dp,  5.55643282286291385e+00_dp]
       aw%aw_erange_matrix(:, 5) = [5.06301374348889252e-04_dp,  2.71278761171135814e-03_dp,  6.87856017151506141e-03_dp,  1.33920607281986185e-02_dp,  2.29358966453171338e-02_dp,&
            3.66319603265295932e-02_dp,  5.62274858691858906e-02_dp,  8.43186005607075850e-02_dp,  1.24637851061089364e-01_dp,  1.82457621413614907e-01_dp,  2.65164755739631663e-01_dp,&
            3.83064351283859128e-01_dp,  5.50488938082404622e-01_dp,  7.87324820338791365e-01_dp,  1.12112723035988537e+00_dp,  1.59010865056180872e+00_dp,  2.24752958332770136e+00_dp,&
            3.16862246665946223e+00_dp,  4.46283451629254735e+00_dp,  6.29942745333522058e+00_dp,  8.97541645187698656e+00_dp,  1.31843080894929585e+01_dp,  1.30437265924520491e-03_dp,&
            3.14022289782075158e-03_dp,  5.25443251677055415e-03_dp,  7.88389833135097473e-03_dp,  1.13876295487330640e-02_dp,  1.62920447106396638e-02_dp,  2.33270389319482595e-02_dp,&
            3.34706882965253444e-02_dp,  4.80370967810812288e-02_dp,  6.88203632849411356e-02_dp,  9.82931931226290467e-02_dp,  1.39869704652046317e-01_dp,  1.98259233188294887e-01_dp,&
            2.79956408660673450e-01_dp,  3.93946342038355879e-01_dp,  5.52783442365534894e-01_dp,  7.74408322866289134e-01_dp,  1.08564112055017259e+00_dp,  1.53006862036254487e+00_dp,&
            2.18959872101262132e+00_dp,  3.26139926305611993e+00_dp,  5.51600983923928290e+00_dp]
       aw%aw_erange_matrix(:, 6) = [4.52328997700439893e-04_dp,  2.42541940651550166e-03_dp,  6.15878752614964706e-03_dp,  1.20180968011167049e-02_dp,  2.06500938360118999e-02_dp,&
            3.31230995252827717e-02_dp,  5.11059061643640358e-02_dp,  7.70839435227944486e-02_dp,  1.14642450144781846e-01_dp,  1.68868576944605442e-01_dp,  2.46925992970210867e-01_dp,&
            3.58859892600339014e-01_dp,  5.18710476831461653e-01_dp,  7.46049362900396584e-01_dp,  1.06811334063430730e+00_dp,  1.52282190800534600e+00_dp,  2.16320763058890853e+00_dp,&
            3.06438439565946164e+00_dp,  4.33582313860725588e+00_dp,  6.14693617040813933e+00_dp,  8.79485826575428042e+00_dp,  1.29724514738667107e+01_dp,  1.16552367137457281e-03_dp,&
            2.81018559121979815e-03_dp,  4.71590909909307045e-03_dp,  7.10813611379747506e-03_dp,  1.03309111563121975e-02_dp,  1.48897071521879761e-02_dp,  2.14855604114208808e-02_dp,&
            3.10614767189596394e-02_dp,  4.48939487762343367e-02_dp,  6.47390130701064465e-02_dp,  9.30317245051570513e-02_dp,  1.33150898969299131e-01_dp,  1.89777281414116400e-01_dp,&
            2.69390808048385000e-01_dp,  3.80985576078253563e-01_dp,  5.37159738903385531e-01_dp,  7.55941220449633189e-01_dp,  1.06428001974542874e+00_dp,  1.50591026264861583e+00_dp,&
            2.16282550536570506e+00_dp,  3.23197665458178296e+00_dp,  5.48211161243339973e+00_dp]
       aw%aw_erange_matrix(:, 7) = [4.09463955316871013e-04_dp,  2.19707903409960747e-03_dp,  5.58634642673214599e-03_dp,  1.09238913120095690e-02_dp,  1.88262894374492430e-02_dp,&
            3.03164034487479472e-02_dp,  4.69960276555681095e-02_dp,  7.12558970704633005e-02_dp,  1.06555323301802465e-01_dp,  1.57822973295951186e-01_dp,  2.32030915294803464e-01_dp,&
            3.39000042141027635e-01_dp,  4.92516686674662585e-01_dp,  7.11877254549920457e-01_dp,  1.02403778262496448e+00_dp,  1.46665684376382033e+00_dp,  2.09256091345909478e+00_dp,&
            2.97675266211982104e+00_dp,  4.22871778952681066e+00_dp,  6.01799933870749992e+00_dp,  8.64184581975755783e+00_dp,  1.27925889223381191e+01_dp,  1.05523685978521371e-03_dp,&
            2.54778311387449021e-03_dp,  4.28697639370369601e-03_dp,  6.48857830184702573e-03_dp,  9.48380540077316606e-03_dp,  1.37599740604215665e-02_dp,  1.99932026118929546e-02_dp,&
            2.90965281146010890e-02_dp,  4.23143327934254188e-02_dp,  6.13697396007867879e-02_dp,  8.86646198838301897e-02_dp,  1.27546015735097384e-01_dp,  1.82668158546487353e-01_dp,&
            2.60496080777035011e-01_dp,  3.70029323875206817e-01_dp,  5.23901932671672510e-01_dp,  7.40216597135249899e-01_dp,  1.04603708345079682e+00_dp,  1.48522947792240667e+00_dp,&
            2.13986938637024560e+00_dp,  3.20673200723689300e+00_dp,  5.45304183152807376e+00_dp]
       aw%aw_erange_matrix(:, 8) = [3.74536513689004711e-04_dp,  2.01093874256135082e-03_dp,  5.11931403847935666e-03_dp,  1.00300616007581826e-02_dp,  1.73338888970692613e-02_dp,&
            2.80143887120274729e-02_dp,  4.36151707268864000e-02_dp,  6.64446546939080340e-02_dp,  9.98527469147449664e-02_dp,  1.48630270429679301e-01_dp,  2.19582093557942060e-01_dp,&
            3.22332445416299462e-01_dp,  4.70443843893328828e-01_dp,  6.82968577305359203e-01_dp,  9.86611911362780591e-01_dp,  1.41879734177426342e+00_dp,  2.03216328844818372e+00_dp,&
            2.90160786155497918e+00_dp,  4.13662510506400238e+00_dp,  5.90687310592433601e+00_dp,  8.50970804507050538e+00_dp,  1.26370160085061674e+01_dp,  9.65363281953964957e-04_dp,&
            2.33375825085292959e-03_dp,  3.93654790774339725e-03_dp,  5.98116160943248411e-03_dp,  8.78763358698389681e-03_dp,  1.28273133232257447e-02_dp,  1.87545059713052562e-02_dp,&
            2.74562383607578030e-02_dp,  4.01489971289221217e-02_dp,  5.85269615945393437e-02_dp,  8.49623494432137777e-02_dp,  1.22773325112377513e-01_dp,  1.76589448341746752e-01_dp,&
            2.52860956026609074e-01_dp,  3.60590351441223977e-01_dp,  5.12441666736709212e-01_dp,  7.26582638760519983e-01_dp,  1.03017804776413668e+00_dp,  1.46721338759692377e+00_dp,&
            2.11984251267978685e+00_dp,  3.18469537275312620e+00_dp,  5.42767735620232283e+00_dp]
       aw%aw_erange_matrix(:, 9) = [2.07916311779228946e-04_dp,  1.12155761153414431e-03_dp,  2.88127758734212178e-03_dp,  5.72772751608632748e-03_dp,  1.01054123620637430e-02_dp,&
            1.67704053240377918e-02_dp,  2.69244859195186571e-02_dp,  4.23921907191599640e-02_dp,  6.58805569609397790e-02_dp,  1.01364968397741645e-01_dp,  1.54647961551893698e-01_dp,&
            2.34154774032586194e-01_dp,  3.52057663618104755e-01_dp,  5.25861379354838676e-01_dp,  7.80644517631268742e-01_dp,  1.15226216680138238e+00_dp,  1.69204597817615787e+00_dp,&
            2.47409994481944739e+00_dp,  3.60786218697460548e+00_dp,  5.26369191349306487e+00_dp,  7.73976068285175334e+00_dp,  1.17256205146613937e+01_dp,  5.36467715023434774e-04_dp,&
            1.30915525385516996e-03_dp,  2.24911227056142802e-03_dp,  3.51604898345418779e-03_dp,  5.36313626635470143e-03_dp,  8.16430335527195197e-03_dp,  1.24437195139966619e-02_dp,&
            1.89365370967848681e-02_dp,  2.86935938504337684e-02_dp,  4.32292742095023511e-02_dp,  6.47224871099811999e-02_dp,  9.62938700402608677e-02_dp,  1.42392698698953352e-01_dp,&
            2.09342291317476281e-01_dp,  3.06123801474388446e-01_dp,  4.45550520347793078e-01_dp,  6.46173012089851895e-01_dp,  9.35795948539298372e-01_dp,  1.35921039025971568e+00_dp,&
            1.99918186703179845e+00_dp,  3.05163758801036078e+00_dp,  5.27474337655333336e+00_dp]
       aw%aw_erange_matrix(:, 10) = [1.47171034911424206e-04_dp,  7.96428323837147919e-04_dp,  2.05896664004827238e-03_dp,  4.13466706766572444e-03_dp,  7.39936201103626439e-03_dp,&
            1.24987458722376123e-02_dp,  2.04667409164244057e-02_dp,  3.28906135988269246e-02_dp,  5.21610594386609783e-02_dp,  8.18464632812529458e-02_dp,  1.27238497445389814e-01_dp,&
            1.96136667599684955e-01_dp,  2.99969379508689626e-01_dp,  4.55391258003387089e-01_dp,  6.86560914454681037e-01_dp,  1.02841510570747774e+00_dp,  1.53148292016649346e+00_dp,&
            2.26933300000655347e+00_dp,  3.35128770114537033e+00_dp,  4.94806350992282962e+00_dp,  7.35836582136724626e+00_dp,  1.12707865998321051e+01_dp,  3.80005538578459382e-04_dp,&
            9.33338699389642040e-04_dp,  1.62390227222127871e-03_dp,  2.58855365800755768e-03_dp,  4.04653947253978859e-03_dp,  6.32189653318652577e-03_dp,  9.87455765362500217e-03_dp,&
            1.53655063745645748e-02_dp,  2.37611806119816318e-02_dp,  3.64787245922180148e-02_dp,  5.55869015818103787e-02_dp,  8.40880209433377684e-02_dp,  1.26315924755279474e-01_dp,&
            1.88500889520422404e-01_dp,  2.79583276965536864e-01_dp,  4.12427769529583099e-01_dp,  6.05771789423283358e-01_dp,  8.87768717560709297e-01_dp,  1.30368560346876783e+00_dp,&
            1.93670672248412501e+00_dp,  2.98252430167097149e+00_dp,  5.19544853947986507e+00_dp]
       aw%aw_erange_matrix(:, 11) = [1.15126529151328002e-04_dp,  6.24589880363876355e-04_dp,  1.62281269600238784e-03_dp,  3.28507237039356694e-03_dp,  5.94484779181019777e-03_dp,&
            1.01786277534976731e-02_dp,  1.69147981879632543e-02_dp,  2.75915369674121259e-02_dp,  4.43996514867282185e-02_dp,  7.06474284750039511e-02_dp,  1.11295571810849231e-01_dp,&
            1.73732377650324982e-01_dp,  2.68890258627387913e-01_dp,  4.12848323366157011e-01_dp,  6.29131981503187498e-01_dp,  9.52033554436149010e-01_dp,  1.43150419010024965e+00_dp,&
            2.14070865917387732e+00_dp,  3.18885412672288426e+00_dp,  4.74688455032915613e+00_dp,  7.11389711304269490e+00_dp,  1.09779502387492141e+01_dp,  2.97432196185447480e-04_dp,&
            7.34248784794580079e-04_dp,  1.29035202001703082e-03_dp,  2.08832471267370078e-03_dp,  3.32556285051951008e-03_dp,  5.29418675624997431e-03_dp,  8.41388498269535212e-03_dp,&
            1.32987566459194743e-02_dp,  2.08601018763598482e-02_dp,  3.24495525587031458e-02_dp,  5.00598444054498465e-02_dp,  7.66096125520880494e-02_dp,  1.16348407992360639e-01_dp,&
            1.75434489290140605e-01_dp,  2.62768980870982161e-01_dp,  3.91238667044871424e-01_dp,  5.79697167071256625e-01_dp,  8.56531677721931839e-01_dp,  1.26734410170348299e+00_dp,&
            1.89563508424130456e+00_dp,  2.93699604108030687e+00_dp,  5.14326427364680150e+00_dp]
       aw%aw_erange_matrix(:, 12) = [9.51436729162034249e-05_dp,  5.17270965481586619e-04_dp,  1.34965095006098776e-03_dp,  2.75063616270900500e-03_dp,  5.02409835099588628e-03_dp,&
            8.69766774200029721e-03_dp,  1.46251882146344186e-02_dp,  2.41394814613502019e-02_dp,  3.92893016178502003e-02_dp,  6.31964222262876157e-02_dp,  1.00581828984489030e-01_dp,&
            1.58532956889468474e-01_dp,  2.47615612840909299e-01_dp,  3.83479011279471482e-01_dp,  5.89170141663977631e-01_dp,  8.98487922099273972e-01_dp,  1.36093326932247560e+00_dp,&
            2.04934678905197076e+00_dp,  3.07282922941290959e+00_dp,  4.60248581721676064e+00_dp,  6.93772132176544698e+00_dp,  1.07662537605035951e+01_dp,  2.45921947329527614e-04_dp,&
            6.09681575569105444e-04_dp,  1.08048217433343984e-03_dp,  1.77082905344642201e-03_dp,  2.86240028463815915e-03_dp,  4.62458201556379886e-03_dp,  7.44872589102107702e-03_dp,&
            1.19154813231642353e-02_dp,  1.88958529137463194e-02_dp,  2.96926738305596759e-02_dp,  4.62411918444786615e-02_dp,  7.13957799066545634e-02_dp,  1.09339911550742505e-01_dp,&
            1.66173373117442319e-01_dp,  2.50761842671747781e-01_dp,  3.76001769077057402e-01_dp,  5.60827824679953180e-01_dp,  8.33800292631723394e-01_dp,  1.24077761664007458e+00_dp,&
            1.86551413845751934e+00_dp,  2.90355631432598127e+00_dp,  5.10496090727687246e+00_dp]
       aw%aw_erange_matrix(:, 13) = [8.14132863726718339e-05_dp,  4.43438181799428747e-04_dp,  1.16127534069767604e-03_dp,  2.38070961367043300e-03_dp,  4.38334849656378360e-03_dp,&
            7.65984385472751658e-03_dp,  1.30076358481182111e-02_dp,  2.16797344420144138e-02_dp,  3.56167156386108796e-02_dp,  5.77972661403760996e-02_dp,  9.27570244102206981e-02_dp,&
            1.47348988468907538e-01_dp,  2.31851068382248837e-01_dp,  3.61572161335724507e-01_dp,  5.59177233840892129e-01_dp,  8.58067272951867022e-01_dp,  1.30737537959642269e+00_dp,&
            1.97967156835564628e+00_dp,  2.98395950829788603e+00_dp,  4.49146636044110981e+00_dp,  6.80184903552209796e+00_dp,  1.06025902099772953e+01_dp,  2.10518581262747359e-04_dp,&
            5.23849992285989872e-04_dp,  9.35189048688426995e-04_dp,  1.54940041282643323e-03_dp,  2.53609900434045550e-03_dp,  4.14739470159187530e-03_dp,  6.75323587542941538e-03_dp,&
            1.09086636409994140e-02_dp,  1.74532704715506803e-02_dp,  2.76513431020509702e-02_dp,  4.33922324868079509e-02_dp,  6.74784216164883988e-02_dp,  1.04039232976262802e-01_dp,&
            1.59125353965573757e-01_dp,  2.41570662449016704e-01_dp,  3.64275022925983405e-01_dp,  5.46233634635032272e-01_dp,  8.16142673469200441e-01_dp,  1.22006746819583412e+00_dp,&
            1.84197380919507925e+00_dp,  2.87739093899265930e+00_dp,  5.07500437871142740e+00_dp]
       aw%aw_erange_matrix(:, 14) = [7.13592602953326029e-05_dp,  3.89315053628413934e-04_dp,  1.02290016637576254e-03_dp,  2.10808534696908844e-03_dp,  3.90891228002737939e-03_dp,&
            6.88672932852428445e-03_dp,  1.17942977996034100e-02_dp,  1.98213106598323675e-02_dp,  3.28221100608088456e-02_dp,  5.36606118809696275e-02_dp,  8.67228342890487836e-02_dp,&
            1.38671335005026458e-01_dp,  2.19548687767534084e-01_dp,  3.44383806133186920e-01_dp,  5.35525179860289025e-01_dp,  8.26041520397564399e-01_dp,  1.26475567304876946e+00_dp,&
            1.92400558711946434e+00_dp,  2.91270629065943742e+00_dp,  4.40218119622389903e+00_dp,  6.69230018398852167e+00_dp,  1.04703744397918381e+01_dp,  1.84588073435838182e-04_dp,&
            4.60846669383881493e-04_dp,  8.28098646616922209e-04_dp,  1.38513747790979511e-03_dp,  2.29191443040749473e-03_dp,  3.78682567533227508e-03_dp,  6.22287602595384889e-03_dp,&
            1.01345751815021604e-02_dp,  1.63359744295523690e-02_dp,  2.60597081143932423e-02_dp,  4.11571292968829355e-02_dp,  6.43873797754156846e-02_dp,  9.98340320606822124e-02_dp,&
            1.53505526775757045e-01_dp,  2.34207088139599728e-01_dp,  3.54838537454467418e-01_dp,  5.34442377729807849e-01_dp,  8.01825742248729934e-01_dp,  1.20322663456372658e+00_dp,&
            1.82279189171337763e+00_dp,  2.85604880272480033e+00_dp,  5.05057914863739565e+00_dp]
       aw%aw_erange_matrix(:, 15) = [6.36579983019180099e-05_dp,  3.47816904834585600e-04_dp,  9.16606656190229580e-04_dp,  1.89805540144578791e-03_dp,  3.54186609442470228e-03_dp,&
            6.28539259011753546e-03_dp,  1.08448267084552269e-02_dp,  1.83579477305851300e-02_dp,  3.06080848555685126e-02_dp,  5.03641360602654284e-02_dp,  8.18875824451109208e-02_dp,&
            1.31681632920222336e-01_dp,  2.09590915238371839e-01_dp,  3.30407578483158271e-01_dp,  5.16210949027820387e-01_dp,  7.99785285621254727e-01_dp,  1.22968561961200007e+00_dp,&
            1.87804688652642215e+00_dp,  2.85370281770559808e+00_dp,  4.32805520345455630e+00_dp,  6.60115781335191620e+00_dp,  1.03601921647022532e+01_dp,  1.64721176977347488e-04_dp,&
            4.12482080613346225e-04_dp,  7.45587885479292458e-04_dp,  1.25784498264198616e-03_dp,  2.10122286130215137e-03_dp,  3.50288356884526867e-03_dp,  5.80195979248964312e-03_dp,&
            9.51595545077662391e-03_dp,  1.54375270625704136e-02_dp,  2.47725892091762018e-02_dp,  3.93402058975390809e-02_dp,  6.18624226914841341e-02_dp,  9.63832904369785054e-02_dp,&
            1.48874223255665072e-01_dp,  2.28114450085994347e-01_dp,  3.47001715767874530e-01_dp,  5.24616719284114019e-01_dp,  7.89859775543438225e-01_dp,  1.18911662284867847e+00_dp,&
            1.80669222746527725e+00_dp,  2.83812086290345444e+00_dp,  5.03006762909104310e+00_dp]
       aw%aw_erange_matrix(:, 16) = [5.75572951779637076e-05_dp,  3.14914306640529064e-04_dp,  8.32188482767582728e-04_dp,  1.73080610181602598e-03_dp,  3.24846434323990390e-03_dp,&
            5.80238149711106033e-03_dp,  1.00780714842716203e-02_dp,  1.71696844891436259e-02_dp,  2.88006307318317452e-02_dp,  4.76592602480586866e-02_dp,  7.79009926997707575e-02_dp,&
            1.25892715892636398e-01_dp,  2.01308959280787431e-01_dp,  3.18737484345279931e-01_dp,  5.00024124918876711e-01_dp,  7.77705013695570990e-01_dp,  1.20009993174445717e+00_dp,&
            1.83916349363603615e+00_dp,  2.80365447240209775e+00_dp,  4.26503999352744056e+00_dp,  6.52353530439399965e+00_dp,  1.02662211683425344e+01_dp,  1.48980050649978461e-04_dp,&
            3.74093905974257025e-04_dp,  6.79878494222507942e-04_dp,  1.15594085400318246e-03_dp,  1.94750477749893228e-03_dp,  3.27230714177671918e-03_dp,  5.45783233086765719e-03_dp,&
            9.00715387075903817e-03_dp,  1.46946051562604138e-02_dp,  2.37030724147478736e-02_dp,  3.78236373963166950e-02_dp,  5.97460018280367572e-02_dp,  9.34794912089463720e-02_dp,&
            1.44962599288782995e-01_dp,  2.22950806915248823e-01_dp,  3.40338543176551489e-01_dp,  5.16238115635440420e-01_dp,  7.79629764330003971e-01_dp,  1.17702800798791141e+00_dp,&
            1.79287805320122295e+00_dp,  2.82272660203277903e+00_dp,  5.01245943123840831e+00_dp]
       aw%aw_erange_matrix(:, 17) = [5.25969497635628904e-05_dp,  2.88140299657843612e-04_dp,  7.63388729255190458e-04_dp,  1.59416624183384543e-03_dp,  3.00791751449272894e-03_dp,&
            5.40463761301825715e-03_dp,  9.44360608781482681e-03_dp,  1.61815943494118403e-02_dp,  2.72904897758569354e-02_dp,  4.53890949790402704e-02_dp,  7.45408810429689511e-02_dp,&
            1.20994088473675779e-01_dp,  1.94274624261707080e-01_dp,  3.08790956078508427e-01_dp,  4.86183223357853467e-01_dp,  7.58767923389570109e-01_dp,  1.17465534317426790e+00_dp,&
            1.80563799283809878e+00_dp,  2.76040516925991852e+00_dp,  4.21047946882117508e+00_dp,  6.45621984012329531e+00_dp,  1.01846271205820038e+01_dp,  1.36178924890914869e-04_dp,&
            3.42825234820011090e-04_dp,  6.26191749168496454e-04_dp,  1.07228090118842716e-03_dp,  1.82051245976536883e-03_dp,  3.08056613776433322e-03_dp,  5.16995083867658693e-03_dp,&
            8.57926437455989663e-03_dp,  1.40668760010720529e-02_dp,  2.27955019479989147e-02_dp,  3.65315987641240403e-02_dp,  5.79362557797011499e-02_dp,  9.09878764338887397e-02_dp,&
            1.41595346423459723e-01_dp,  2.18492311409719364e-01_dp,  3.34569115823445562e-01_dp,  5.08964731161898665e-01_dp,  7.70729091745732675e-01_dp,  1.16649062786416691e+00_dp,&
            1.78082044043342025e+00_dp,  2.80928104526444544e+00_dp,  4.99708355849031083e+00_dp]
       aw%aw_erange_matrix(:, 18) = [2.90762890407724913e-05_dp,  1.60820764045495378e-04_dp,  4.34422626290592476e-04_dp,  9.35039866616582996e-04_dp,  1.83288487070204118e-03_dp,&
            3.43151821088961929e-03_dp,  6.24330639046053164e-03_dp,  1.11139604851392074e-02_dp,  1.94205651181870234e-02_dp,  3.33783588073015142e-02_dp,  5.65099671357182226e-02_dp,&
            9.43565235776523542e-02_dp,  1.55546612904553100e-01_dp,  2.53391770241247982e-01_dp,  4.08255853980029115e-01_dp,  6.51070811498433266e-01_dp,  1.02859999602671803e+00_dp,&
            1.61156146938925948e+00_dp,  2.50813991254804813e+00_dp,  3.89015625413403843e+00_dp,  6.05888858834257960e+00_dp,  9.70101820848491236e+00_dp,  7.54396214641075318e-05_dp,&
            1.93611184726491780e-04_dp,  3.67183036170629030e-04_dp,  6.61657087412284291e-04_dp,  1.18342298349698389e-03_dp,  2.09721549528343789e-03_dp,  3.66409371548580276e-03_dp,&
            6.30174076999150123e-03_dp,  1.06730393182044355e-02_dp,  1.78180701123197525e-02_dp,  2.93512452214240056e-02_dp,  4.77539458032805800e-02_dp,  7.68061849693031928e-02_dp,&
            1.22220852210935996e-01_dp,  1.92577294064878640e-01_dp,  3.00716204350791827e-01_dp,  4.65916878838533499e-01_dp,  7.17645882764439302e-01_dp,  1.10324722563679933e+00_dp,&
            1.70812188016783417e+00_dp,  2.72803096337466400e+00_dp,  4.90423336038595714e+00_dp]
       aw%aw_erange_matrix(:, 19) = [2.05653889251502476e-05_dp,  1.14520867388633136e-04_dp,  3.13640994086593816e-04_dp,  6.89261854057424649e-04_dp,  1.38517347545250176e-03_dp,&
            2.66038656199351492e-03_dp,  4.95918459442486713e-03_dp,  9.02797406142399547e-03_dp,  1.61023543090457119e-02_dp,  2.82000330967852458e-02_dp,  4.85739460822395552e-02_dp,&
            8.24063146263888241e-02_dp,  1.37862273521427520e-01_dp,  2.27676785155268485e-01_dp,  3.71530405037607481e-01_dp,  5.99598522246047039e-01_dp,  9.57889945568599410e-01_dp,&
            1.51649820008986214e+00_dp,  2.38328420664463492e+00_dp,  3.73019408610861980e+00_dp,  5.85901523269036240e+00_dp,  9.45637637256199426e+00_dp,  5.34363201296681438e-05_dp,&
            1.39020073506606164e-04_dp,  2.70599284078877720e-04_dp,  5.03939028710325798e-04_dp,  9.29930193231351420e-04_dp,  1.69264740976545792e-03_dp,  3.02631466800271781e-03_dp,&
            5.31244207968048672e-03_dp,  9.16523880225796408e-03_dp,  1.55609829578794105e-02_dp,  2.60333897168352757e-02_dp,  4.29663462371881605e-02_dp,  7.00293507626535416e-02_dp,&
            1.12821990431476962e-01_dp,  1.79827912884988311e-01_dp,  2.83844003822466384e-01_dp,  4.44206488566912749e-01_dp,  6.90593012627218727e-01_dp,  1.07073613696106928e+00_dp,&
            1.67051539850277919e+00_dp,  2.68586871499289925e+00_dp,  4.85609323400620241e+00_dp]
       aw%aw_erange_matrix(:, 20) = [1.60905223191839913e-05_dp,  9.00924705193344226e-05_dp,  2.49480714636845147e-04_dp,  5.57253697520921840e-04_dp,  1.14108958815330652e-03_dp,&
            2.23286406677653416e-03_dp,  4.23520330690799109e-03_dp,  7.83307362353265896e-03_dp,  1.41734445524208229e-02_dp,  2.51488274898480427e-02_dp,  4.38393471602563367e-02_dp,&
            7.51948372263522291e-02_dp,  1.27077300211358046e-01_dp,  2.11840719863816418e-01_dp,  3.48709343563647356e-01_dp,  5.67347420179910911e-01_dp,  9.13246803231961057e-01_dp,&
            1.45606441575002998e+00_dp,  2.30342338171505601e+00_dp,  3.62733903836381089e+00_dp,  5.72994466090799826e+00_dp,  9.29787769290753019e+00_dp,  4.18582630530724392e-05_dp,&
            1.10095201120191568e-04_dp,  2.18729154478433733e-04_dp,  4.17481342981990177e-04_dp,  7.87767058022032255e-04_dp,  1.46107800353917133e-03_dp,  2.65485040360329211e-03_dp,&
            4.72745322818911469e-03_dp,  8.26152870286268534e-03_dp,  1.41915027312551519e-02_dp,  2.39975456485793520e-02_dp,  3.99980426188496635e-02_dp,  6.57871441778337951e-02_dp,&
            1.06885646486166594e-01_dp,  1.71708282235778303e-01_dp,  2.73015955599476445e-01_dp,  4.30175629967851303e-01_dp,  6.73000885562459072e-01_dp,  1.04948552778007187e+00_dp,&
            1.64584195942300959e+00_dp,  2.65815350615402668e+00_dp,  4.82446309349684288e+00_dp]
       aw%aw_erange_matrix(:, 21) = [1.33051281617681520e-05_dp,  7.48448384990370801e-05_dp,  2.09214312792452143e-04_dp,  4.73669311645959765e-04_dp,  9.84726055373060525e-04_dp,&
            1.95547892998873714e-03_dp,  3.75958796998500291e-03_dp,  7.03892644101850040e-03_dp,  1.28777413472578924e-02_dp,  2.30792052353523229e-02_dp,  4.05991719793361294e-02_dp,&
            7.02191345034593045e-02_dp,  1.19579941024401004e-01_dp,  2.00755646924385800e-01_dp,  3.32632747525695771e-01_dp,  5.44494103839673005e-01_dp,  8.81442137232194889e-01_dp,&
            1.41280037704187977e+00_dp,  2.24600452092047931e+00_dp,  3.55311338618090922e+00_dp,  5.63651897644199629e+00_dp,  9.18288725381164284e+00_dp,  3.46469464063064458e-05_dp,&
            9.19802895962243850e-05_dp,  1.85890421128139926e-04_dp,  3.61858574332359778e-04_dp,  6.94740965771247919e-04_dp,  1.30729901127845001e-03_dp,  2.40507869574388064e-03_dp,&
            4.32982418281745076e-03_dp,  7.64129555982906786e-03_dp,  1.32433409792537971e-02_dp,  2.25766959381045185e-02_dp,  3.79110871371775016e-02_dp,  6.27841312538312130e-02_dp,&
            1.02656723796630878e-01_dp,  1.65890020881459044e-01_dp,  2.65214803703477164e-01_dp,  4.20016957188921014e-01_dp,  6.60207961676460342e-01_dp,  1.03397579428113473e+00_dp,&
            1.62778610521243183e+00_dp,  2.63784401884980513e+00_dp,  4.80129185077557441e+00_dp]
       aw%aw_erange_matrix(:, 22) = [1.13934021647398620e-05_dp,  6.43553767195351271e-05_dp,  1.81384735260225273e-04_dp,  4.15465804906859325e-04_dp,  8.74784651968030431e-04_dp,&
            1.75842677544671769e-03_dp,  3.41835170634737342e-03_dp,  6.46392134002142949e-03_dp,  1.19317312559794313e-02_dp,  2.15566465163261507e-02_dp,  3.81989404458580872e-02_dp,&
            6.65099017862893216e-02_dp,  1.13958393754720083e-01_dp,  1.92399599408031863e-01_dp,  3.20454452163599690e-01_dp,  5.27104117055056220e-01_dp,  8.57140736476918685e-01_dp,&
            1.37961952060154536e+00_dp,  2.20182185430469257e+00_dp,  3.49583596141349240e+00_dp,  5.56425878518387496e+00_dp,  9.09379170158253380e+00_dp,  2.96949246120568376e-05_dp,&
            7.94830390495184180e-05_dp,  1.63026224371196949e-04_dp,  3.22611474393445742e-04_dp,  6.28209031130342569e-04_dp,  1.19604481045554301e-03_dp,  2.22261828090361850e-03_dp,&
            4.03689706422688966e-03_dp,  7.18093917005207995e-03_dp,  1.25347945927189886e-02_dp,  2.15083195138626969e-02_dp,  3.63328896886428443e-02_dp,  6.05012296058346846e-02_dp,&
            9.94262052048541650e-02_dp,  1.61425318654521810e-01_dp,  2.59203569677637025e-01_dp,  4.12159396023710212e-01_dp,  6.50279574358099022e-01_dp,  1.02190518592096291e+00_dp,&
            1.61370511273339901e+00_dp,  2.62198875738320014e+00_dp,  4.78320650635398081e+00_dp]
       aw%aw_erange_matrix(:, 23) = [9.99454274093545240e-06_dp,  5.66643636299825617e-05_dp,  1.60896503364809740e-04_dp,  3.72334006752350892e-04_dp,  7.92633889132781234e-04_dp,&
            1.60990407495521142e-03_dp,  3.15902992422392986e-03_dp,  6.02364217988483278e-03_dp,  1.12024081667284076e-02_dp,  2.03755415661601089e-02_dp,  3.63264769194730081e-02_dp,&
            6.36013551904103663e-02_dp,  1.09529542685367035e-01_dp,  1.85787923004832572e-01_dp,  3.10780127529339312e-01_dp,  5.13239254991739280e-01_dp,  8.37700866434439195e-01_dp,&
            1.35299652288265082e+00_dp,  2.16627671791894283e+00_dp,  3.44965052286817064e+00_dp,  5.50588350642492852e+00_dp,  9.02171444171596804e+00_dp,  2.60697394028477843e-05_dp,&
            7.02972872632460762e-05_dp,  1.46084629233613855e-04_dp,  2.93196758798126291e-04_dp,  5.77782380989189191e-04_dp,  1.11092466332648305e-03_dp,  2.08191067723823313e-03_dp,&
            3.80944401039385065e-03_dp,  6.82128745818843651e-03_dp,  1.19781797824074533e-02_dp,  2.06647966320373938e-02_dp,  3.50810773504934836e-02_dp,  5.86827296947812008e-02_dp,&
            9.68427139463775566e-02_dp,  1.57841812595722136e-01_dp,  2.54362549305863661e-01_dp,  4.05812091239185158e-01_dp,  6.42237696997827623e-01_dp,  1.01210598637712601e+00_dp,&
            1.60225484010221209e+00_dp,  2.60908459532920922e+00_dp,  4.76848987721681183e+00_dp]
       aw%aw_erange_matrix(:, 24) = [8.92352094047950570e-06_dp,  5.07651520041585948e-05_dp,  1.45123988619345743e-04_dp,  3.38934589926485311e-04_dp,  7.28555367815529533e-04_dp,&
            1.49318516017513179e-03_dp,  2.95380020435500549e-03_dp,  5.67296401145544159e-03_dp,  1.06181420151690940e-02_dp,  1.94243935039796548e-02_dp,  3.48114124334977515e-02_dp,&
            6.12377859731470867e-02_dp,  1.05916301172055749e-01_dp,  1.80374296040537785e-01_dp,  3.02832467443627407e-01_dp,  5.01814244147640531e-01_dp,  8.21637300725933217e-01_dp,&
            1.33094204170813835e+00_dp,  2.13676550226255024e+00_dp,  3.41123201151645850e+00_dp,  5.45724971517158153e+00_dp,  8.96159470755822163e+00_dp,  2.32930218299722802e-05_dp,&
            6.32360769780289069e-05_dp,  1.32967133627795708e-04_dp,  2.70192493508733436e-04_dp,  5.37965852619124883e-04_dp,  1.04317824981281699e-03_dp,  1.96917378155438572e-03_dp,&
            3.62614732732700114e-03_dp,  6.52996012917771932e-03_dp,  1.15252083208033391e-02_dp,  1.99754326692176479e-02_dp,  3.40540728555096608e-02_dp,  5.71854808137426049e-02_dp,&
            9.47086037923560858e-02_dp,  1.54872615401006386e-01_dp,  2.50340151197668392e-01_dp,  4.00524640047162650e-01_dp,  6.35523448616123310e-01_dp,  1.00390904389308799e+00_dp,&
            1.59266346971105199e+00_dp,  2.59826755537668985e+00_dp,  4.75615524329419248e+00_dp]
       aw%aw_erange_matrix(:, 25) = [8.07533230400322961e-06_dp,  4.60856441621744308e-05_dp,  1.32570852452785267e-04_dp,  3.12210915451310400e-04_dp,  6.76950813769914249e-04_dp,&
            1.39856692266817424e-03_dp,  2.78640694108129661e-03_dp,  5.38534378226993276e-03_dp,  1.01365344492760443e-02_dp,  1.86368240312546410e-02_dp,  3.35517776609026847e-02_dp,&
            5.92653909136533208e-02_dp,  1.02890821649513090e-01_dp,  1.75827211907739561e-01_dp,  2.96137945827540716e-01_dp,  4.92165535839492907e-01_dp,  8.08038902739617870e-01_dp,&
            1.31223194784214559e+00_dp,  2.11168172454495195e+00_dp,  3.37852397717336084e+00_dp,  5.41578989059694660e+00_dp,  8.91029189403326782e+00_dp,  2.10932070918042014e-05_dp,&
            5.76236612613593971e-05_dp,  1.22472462807700111e-04_dp,  2.51622999405283748e-04_dp,  5.05555830895371150e-04_dp,  9.87653780958501438e-04_dp,  1.87624224076666157e-03_dp,&
            3.47429497489486552e-03_dp,  6.28753596109184708e-03_dp,  1.11467641378285454e-02_dp,  1.93973929963348439e-02_dp,  3.31900503795852508e-02_dp,  5.59219848393945895e-02_dp,&
            9.29025902137011472e-02_dp,  1.52353349648122888e-01_dp,  2.46919086314924902e-01_dp,  3.96017811288945254e-01_dp,  6.29789380412422162e-01_dp,  9.96897394001393877e-01_dp,&
            1.58444925174457607e+00_dp,  2.58899789320651852e+00_dp,  4.74558629969863599e+00_dp]
       aw%aw_erange_matrix(:, 26) = [7.38580306967939125e-06_dp,  4.22757469481121925e-05_dp,  1.22319116365676202e-04_dp,  2.90280105497779098e-04_dp,  6.34352507020876050e-04_dp,&
            1.32000166881638282e-03_dp,  2.64665609257590077e-03_dp,  5.14403969909843321e-03_dp,  9.73069986920719991e-03_dp,  1.79705334901513712e-02_dp,  3.24822962283058056e-02_dp,&
            5.75852967059411816e-02_dp,  1.00306058599380868e-01_dp,  1.71931950772977377e-01_dp,  2.90388822316800965e-01_dp,  4.83860549183816491e-01_dp,  7.96309975940985959e-01_dp,&
            1.29606381803504678e+00_dp,  2.08996987178063476e+00_dp,  3.35017255185728491e+00_dp,  5.37981093058197679e+00_dp,  8.86573240555865993e+00_dp,  1.93042785336726517e-05_dp,&
            5.30458699076083204e-05_dp,  1.13860681610436235e-04_dp,  2.36261993628351859e-04_dp,  4.78547024573813535e-04_dp,  9.41102528781277313e-04_dp,  1.79793452155763102e-03_dp,&
            3.34577588500104516e-03_dp,  6.08156170905922461e-03_dp,  1.08240927944734319e-02_dp,  1.89029713356079453e-02_dp,  3.24488645598754247e-02_dp,  5.48352209151617767e-02_dp,&
            9.13453669492429871e-02_dp,  1.50176191039324863e-01_dp,  2.43956403619624701e-01_dp,  3.92107409822559982e-01_dp,  6.24805750964588902e-01_dp,  9.90794782222321646e-01_dp,&
            1.57729255106850563e+00_dp,  2.58091724238665954e+00_dp,  4.73637393673943219e+00_dp]
       aw%aw_erange_matrix(:, 27) = [4.11577647931065565e-06_dp,  2.41109171591163585e-05_dp,  7.28959878731409684e-05_dp,  1.82705558633408622e-04_dp,  4.21135317613602464e-04_dp,&
            9.18895701336235049e-04_dp,  1.92013555113837073e-03_dp,  3.86903859938759207e-03_dp,  7.55495901793930974e-03_dp,  1.43514943327147215e-02_dp,  2.66043538157777967e-02_dp,&
            4.82519876889243582e-02_dp,  8.58062586147143591e-02_dp,  1.49884652982802707e-01_dp,  2.57581294617070133e-01_dp,  4.36111432279205069e-01_dp,  7.28412434315531154e-01_dp,&
            1.20188826287217632e+00_dp,  1.96280936152550511e+00_dp,  3.18334555122622742e+00_dp,  5.16729407011710684e+00_dp,  8.60177692919055659e+00_dp,  1.08103451888892216e-05_dp,&
            3.10771144342180430e-05_dp,  7.16274298056807080e-05_dp,  1.58818698899649766e-04_dp,  3.39017395939608462e-04_dp,  6.95803112746412513e-04_dp,  1.37837010493610415e-03_dp,&
            2.64713734273472811e-03_dp,  4.94736636639465065e-03_dp,  9.02660699562345820e-03_dp,  1.61196197861813856e-02_dp,  2.82360865790384878e-02_dp,  4.86035197591654297e-02_dp,&
            8.23430936734954044e-02_dp,  1.37495298031682200e-01_dp,  2.26580573866090457e-01_dp,  3.69028426862235914e-01_dp,  5.95227692025933064e-01_dp,  9.54405318751613607e-01_dp,&
            1.53446955274810404e+00_dp,  2.53247747978796900e+00_dp,  4.68116716115914766e+00_dp]
       aw%aw_erange_matrix(:, 28) = [2.93059248958698421e-06_dp,  1.74658802185034416e-05_dp,  5.44583745684866936e-05_dp,  1.41377660096951151e-04_dp,  3.36534398962208472e-04_dp,&
            7.54852749661623769e-04_dp,  1.61490243368843443e-03_dp,  3.32054799874063672e-03_dp,  6.59924585107690177e-03_dp,  1.27320754504761414e-02_dp,  2.39302082708908702e-02_dp,&
            4.39420719319380260e-02_dp,  7.90196677371394623e-02_dp,  1.39438282335772434e-01_dp,  2.41862130257866903e-01_dp,  4.12999486066383314e-01_dp,  6.95243226386015767e-01_dp,&
            1.15549789669796255e+00_dp,  1.89970943889035770e+00_dp,  3.10004174254383846e+00_dp,  5.06063560401212964e+00_dp,  8.46879716288626128e+00_dp,  7.72551686138751593e-06_dp,&
            2.29485526199224021e-05_dp,  5.54045251476519742e-05_dp,  1.27743027514059810e-04_dp,  2.80970684003487112e-04_dp,  5.90784411349128818e-04_dp,  1.19440174575451401e-03_dp,&
            2.33442466393772412e-03_dp,  4.43040475364441487e-03_dp,  8.19396116517738353e-03_dp,  1.48113872318636000e-02_dp,  2.62296888928512453e-02_dp,  4.55996442864698009e-02_dp,&
            7.79556120448624684e-02_dp,  1.31252074888239895e-01_dp,  2.17946128537454792e-01_dp,  3.57462889284247576e-01_dp,  5.80294174541846508e-01_dp,  9.35917506305006919e-01_dp,&
            1.51261219411583525e+00_dp,  2.50769258392197258e+00_dp,  4.65293052711519373e+00_dp]
       aw%aw_erange_matrix(:, 29) = [2.30617418414400491e-06_dp,  1.39420777043108131e-05_dp,  4.45428273019938463e-05_dp,  1.18703172459198118e-04_dp,  2.89149766243032937e-04_dp,&
            6.61245134643599955e-04_dp,  1.43788129075881482e-03_dp,  2.99794059227682236e-03_dp,  6.03016118430149163e-03_dp,  1.17572577965464210e-02_dp,  2.23048693646072070e-02_dp,&
            4.12997430637738125e-02_dp,  7.48263377765128745e-02_dp,  1.32937840432216126e-01_dp,  2.32017568873716828e-01_dp,  3.98440204051437541e-01_dp,  6.74237452945035676e-01_dp,&
            1.12597907163621103e+00_dp,  1.85938903873710504e+00_dp,  3.04661998143597090e+00_dp,  4.99203821681522708e+00_dp,  8.38308565543044537e+00_dp,  6.09799434345543036e-06_dp,&
            1.86031294610854214e-05_dp,  4.65026406609495805e-05_dp,  1.10215809968520044e-04_dp,  2.47519551061662203e-04_dp,  5.29229248136807960e-04_dp,  1.08503939977670663e-03_dp,&
            2.14625956536696220e-03_dp,  4.11601348431579837e-03_dp,  7.68278225386337735e-03_dp,  1.40014093755201335e-02_dp,  2.49779158617415425e-02_dp,  4.37124829452176303e-02_dp,&
            7.51816527951001123e-02_dp,  1.27281803376186359e-01_dp,  2.12425880206672552e-01_dp,  3.50032879386126261e-01_dp,  5.70659285004944894e-01_dp,  9.23946461035048960e-01_dp,&
            1.49842149874735231e+00_dp,  2.49157828538030701e+00_dp,  4.63457567934905068e+00_dp]
       aw%aw_erange_matrix(:, 30) = [1.91673417108346766e-06_dp,  1.17329394155408427e-05_dp,  3.82558718053225527e-05_dp,  1.04102752208523592e-04_dp,  2.58167492293393994e-04_dp,&
            5.99208950755224603e-04_dp,  1.31919777523101895e-03_dp,  2.77948005793550099e-03_dp,  5.64143484885461059e-03_dp,  1.10862905776961066e-02_dp,  2.11785606265024878e-02_dp,&
            3.94575988228887883e-02_dp,  7.18869622660941177e-02_dp,  1.28358831044982691e-01_dp,  2.25051956635620498e-01_dp,  3.88096902922114961e-01_dp,  6.59259631949477898e-01_dp,&
            1.10486180022603597e+00_dp,  1.83046068334501411e+00_dp,  3.00819690941418516e+00_dp,  4.94260149290475415e+00_dp,  8.32122291118276891e+00_dp,  5.08183620378214294e-06_dp,&
            1.58611740792196564e-05_dp,  4.07692268035397438e-05_dp,  9.86978499440419532e-05_dp,  2.25198865481160177e-04_dp,  4.87659743499264313e-04_dp,  1.01044451864037857e-03_dp,&
            2.01681198765380019e-03_dp,  3.89810708052533254e-03_dp,  7.32613084356623177e-03_dp,  1.34329356605255353e-02_dp,  2.40946839162048780e-02_dp,  4.23744855460415557e-02_dp,&
            7.32062368579494555e-02_dp,  1.24443048187603433e-01_dp,  2.08464324797154321e-01_dp,  3.44682957393298239e-01_dp,  5.63701175641321828e-01_dp,  9.15279688026507388e-01_dp,&
            1.48812871187359153e+00_dp,  2.47987867707640985e+00_dp,  4.62125107431175497e+00_dp]
       aw%aw_erange_matrix(:, 31) = [1.64895154539128592e-06_dp,  1.02072794924256046e-05_dp,  3.38721200697524231e-05_dp,  9.37926563723827687e-05_dp,  2.36021601897219344e-04_dp,&
            5.54396087801813965e-04_dp,  1.23269227931650425e-03_dp,  2.61902174395846734e-03_dp,  5.35401143155140626e-03_dp,  1.05872805743862670e-02_dp,  2.03365792784754133e-02_dp,&
            3.80741487753279012e-02_dp,  6.96703693849783090e-02_dp,  1.24892908078455087e-01_dp,  2.19761770847518162e-01_dp,  3.80217398314288302e-01_dp,  6.47817938432648677e-01_dp,&
            1.08869000178272612e+00_dp,  1.80825851778046087e+00_dp,  2.97865254527264423e+00_dp,  4.90453114611245944e+00_dp,  8.27352996176200151e+00_dp,  4.38249705452781130e-06_dp,&
            1.39570693499999557e-05_dp,  3.67194437667889460e-05_dp,  9.04326665574478182e-05_dp,  2.08991926420577247e-04_dp,  4.57196086844999120e-04_dp,  9.55358061867149109e-04_dp,&
            1.92058951967479397e-03_dp,  3.73520283693082713e-03_dp,  7.05815542147528608e-03_dp,  1.30038803460141780e-02_dp,  2.34253646377022597e-02_dp,  4.13568231121202931e-02_dp,&
            7.16987462523292624e-02_dp,  1.22270113321089682e-01_dp,  2.05423489126560666e-01_dp,  3.40566056588523680e-01_dp,  5.58334743885169460e-01_dp,  9.08582871030465422e-01_dp,&
            1.48016432544994636e+00_dp,  2.47081889914956276e+00_dp,  4.61093391716888146e+00_dp]
       aw%aw_erange_matrix(:, 32) = [1.45267137242234143e-06_dp,  9.08473901898180976e-06_dp,  3.06194421232814190e-05_dp,  8.60601847394439973e-05_dp,  2.19244189704205446e-04_dp,&
            5.20152574520566825e-04_dp,  1.16610631558480534e-03_dp,  2.49474268811706885e-03_dp,  5.13019808764342191e-03_dp,  1.01968830327122013e-02_dp,  1.96751314298413485e-02_dp,&
            3.69833245921940279e-02_dp,  6.79168533834199339e-02_dp,  1.22142900489391221e-01_dp,  2.15553010782744109e-01_dp,  3.73933331954689918e-01_dp,  6.38672818386253449e-01_dp,&
            1.07573861955972894e+00_dp,  1.79044663631006951e+00_dp,  2.95491509748028536e+00_dp,  4.87390682634371331e+00_dp,  8.23513085914243881e+00_dp,  3.86950818582524928e-06_dp,&
            1.25492735313935511e-05_dp,  3.36812570154890965e-05_dp,  8.41512016483296542e-05_dp,  1.96556699954354561e-04_dp,  4.33646608105545236e-04_dp,  9.12509887585908025e-04_dp,&
            1.84534789011500493e-03_dp,  3.60723263852728761e-03_dp,  6.84679241967494235e-03_dp,  1.26642458832250532e-02_dp,  2.28938244986376313e-02_dp,  4.05462799949124764e-02_dp,&
            7.04948703003382277e-02_dp,  1.20530599439649236e-01_dp,  2.02983789833098893e-01_dp,  3.37256378604028129e-01_dp,  5.54012851349098612e-01_dp,  9.03181465052904309e-01_dp,&
            1.47373337309290209e+00_dp,  2.46349906129618867e+00_dp,  4.60259878286634017e+00_dp]
       aw%aw_erange_matrix(:, 33) = [1.18277247892696558e-06_dp,  7.53382482588371914e-06_dp,  2.60779342135758097e-05_dp,  7.51224967992965151e-05_dp,  1.95227490959034921e-04_dp,&
            4.70636327260251150e-04_dp,  1.06900360456677339e-03_dp,  2.31219715214458750e-03_dp,  4.79941040416772841e-03_dp,  9.61676624548491073e-03_dp,  1.86875564926067758e-02_dp,&
            3.53477669019867657e-02_dp,  6.52776944090153743e-02_dp,  1.17989813958694983e-01_dp,  2.09177270981736163e-01_dp,  3.64387140331187465e-01_dp,  6.24745252980397514e-01_dp,&
            1.05596959718259154e+00_dp,  1.76320425107576462e+00_dp,  2.91854809430863416e+00_dp,  4.82692443114716951e+00_dp,  8.17616067826534554e+00_dp,  3.16347496014357980e-06_dp,&
            1.05923351682571904e-05_dp,  2.93818833567492176e-05_dp,  7.51263108427510873e-05_dp,  1.78491524155511525e-04_dp,  3.99137372795228319e-04_dp,  8.49268614427133983e-04_dp,&
            1.73361549881359978e-03_dp,  3.41618966482252866e-03_dp,  6.52978107886599199e-03_dp,  1.21527333444035497e-02_dp,  2.20903099402139230e-02_dp,  3.93168852348080300e-02_dp,&
            6.86633087650498353e-02_dp,  1.17876752111420763e-01_dp,  1.99252260617322013e-01_dp,  3.32182555274532387e-01_dp,  5.47373749754756389e-01_dp,  8.94869814973328404e-01_dp,&
            1.46382478847394260e+00_dp,  2.45221311683700316e+00_dp,  4.58974844353298117e+00_dp]
       aw%aw_erange_matrix(:, 34) = [1.08558975265753804e-06_dp,  6.97285376172907351e-06_dp,  2.44187131788034063e-05_dp,  7.10783551714022979e-05_dp,  1.86251757654043172e-04_dp,&
            4.51964338749976540e-04_dp,  1.03211326511026759e-03_dp,  2.24240799000784878e-03_dp,  4.67226182859080928e-03_dp,  9.39273150226986077e-03_dp,  1.83045903951385980e-02_dp,&
            3.47112002893365107e-02_dp,  6.42471602021468663e-02_dp,  1.16363354044152947e-01_dp,  2.06673731808339645e-01_dp,  3.60629672694155023e-01_dp,  6.19251357685395520e-01_dp,&
            1.04815633217125259e+00_dp,  1.75241891111554859e+00_dp,  2.90412934660146105e+00_dp,  4.80827507059037096e+00_dp,  8.15273245915029676e+00_dp,  2.90904199273370447e-06_dp,&
            9.88034801454935336e-06_dp,  2.77915178824643994e-05_dp,  7.17425769405372472e-05_dp,  1.71651991907649401e-04_dp,  3.85972311497142693e-04_dp,  8.24990713833095119e-04_dp,&
            1.69049331602863983e-03_dp,  3.34211810029706735e-03_dp,  6.40637181413836274e-03_dp,  1.19528930710990269e-02_dp,  2.17753820228709649e-02_dp,  3.88336452846907987e-02_dp,&
            6.79414859800191867e-02_dp,  1.16828366220876056e-01_dp,  1.97774934638904970e-01_dp,  3.30169852509938300e-01_dp,  5.44735529705501431e-01_dp,  8.91562114870858569e-01_dp,&
            1.45987724968194699e+00_dp,  2.44771417531497759e+00_dp,  4.58462621781103863e+00_dp]
       aw%aw_erange_matrix(:, 35) = [6.22443128153836243e-07_dp,  4.27249258996541033e-06_dp,  1.62529434981895659e-05_dp,  5.06723551372187647e-05_dp,  1.39970205298834170e-04_dp,&
            3.53952707704476986e-04_dp,  8.35587239036498326e-04_dp,  1.86596152975692232e-03_dp,  3.97906594391459704e-03_dp,  8.15997287551761645e-03_dp,  1.61801165734150668e-02_dp,&
            3.11543738785128727e-02_dp,  5.84518687707416690e-02_dp,  1.07163785865984809e-01_dp,  1.92439133160937026e-01_dp,  3.39164408599788414e-01_dp,  5.87732340746192183e-01_dp,&
            1.00315928195193260e+00_dp,  1.69009616012076735e+00_dp,  2.82057188884279064e+00_dp,  4.69995107106125420e+00_dp,  8.01641724611723205e+00_dp,  1.69447527098163592e-06_dp,&
            6.40758836525944905e-06_dp,  1.97579702891286563e-05_dp,  5.41860854219365049e-05_dp,  1.35475897817959490e-04_dp,  3.15281723642006119e-04_dp,  6.92999412035867472e-04_dp,&
            1.45356824534664895e-03_dp,  2.93142672355146397e-03_dp,  5.71664630206478001e-03_dp,  1.08280772977361450e-02_dp,  1.99915467045142618e-02_dp,  3.60808077689588336e-02_dp,&
            6.38082290950276743e-02_dp,  1.10796842461126116e-01_dp,  1.89239110538469507e-01_dp,  3.18495400746003077e-01_dp,  5.29379993990211850e-01_dp,  8.72253926728072893e-01_dp,&
            1.43678397212003950e+00_dp,  2.42136411381618455e+00_dp,  4.55462934975407041e+00_dp]
       aw%aw_erange_matrix(:, 36) = [4.53256279959429321e-07_dp,  3.26878734999866000e-06_dp,  1.31027952196471195e-05_dp,  4.24920984671517627e-05_dp,  1.20820935182527079e-04_dp,&
            3.12359866245833266e-04_dp,  7.50452140340149851e-04_dp,  1.70006788803432618e-03_dp,  3.66912131859901351e-03_dp,  7.60185017767122523e-03_dp,  1.52077471182174816e-02_dp,&
            2.95107269951396245e-02_dp,  5.57508778139598829e-02_dp,  1.02843352284423747e-01_dp,  1.85708083820780356e-01_dp,  3.28951310972175948e-01_dp,  5.72652040159266185e-01_dp,&
            9.81523069005708071e-01_dp,  1.65999785504769770e+00_dp,  2.78006816972400372e+00_dp,  4.64728478552088919e+00_dp,  7.94999503748127889e+00_dp,  1.24971764372527093e-06_dp,&
            5.08693360054884036e-06_dp,  1.65312955368420163e-05_dp,  4.68598876288360330e-05_dp,  1.19967770120401381e-04_dp,  2.84338124589194133e-04_dp,  6.34229301738489055e-04_dp,&
            1.34655468330935183e-03_dp,  2.74363821848860141e-03_dp,  5.39788780956410898e-03_dp,  1.03033356187560935e-02_dp,  1.91523824205403828e-02_dp,  3.47760576778694888e-02_dp,&
            6.18359029296285598e-02_dp,  1.07900957575593051e-01_dp,  1.85117905918771225e-01_dp,  3.12830321739075157e-01_dp,  5.21895291602346822e-01_dp,  8.62807124565198813e-01_dp,&
            1.42545339596598897e+00_dp,  2.40841578774241505e+00_dp,  4.53989122605335726e+00_dp]
       aw%aw_erange_matrix(:, 37) = [3.63740571041784989e-07_dp,  2.73125761452274828e-06_dp,  1.13733685754730894e-05_dp,  3.78950548332579519e-05_dp,  1.09860031801165795e-04_dp,&
            2.88206840734960544e-04_dp,  7.00437661465795267e-04_dp,  1.60167354427463173e-03_dp,  3.48380066085383650e-03_dp,  7.26582778105013850e-03_dp,  1.46188014738536480e-02_dp,&
            2.85099408863252156e-02_dp,  5.40985925072002410e-02_dp,  1.00189350362776639e-01_dp,  1.81557748909266986e-01_dp,  3.22632697144117742e-01_dp,  5.63293870386273099e-01_dp,&
            9.68060181105636075e-01_dp,  1.64122492626440830e+00_dp,  2.75475404530085344e+00_dp,  4.61431574676766854e+00_dp,  7.90836486312216547e+00_dp,  1.01409632024120888e-06_dp,&
            4.36831411403733670e-06_dp,  1.47156694458407933e-05_dp,  4.26471698655039871e-05_dp,  1.10913726307929666e-04_dp,  2.66058980170149045e-04_dp,  5.99179766226317343e-04_dp,&
            1.28222314284284615e-03_dp,  2.62997912749839136e-03_dp,  5.20382011759427769e-03_dp,  9.98220585354878814e-03_dp,  1.86364744122212017e-02_dp,  3.39706152505146533e-02_dp,&
            6.06138426345598294e-02_dp,  1.06100631413093172e-01_dp,  1.82548009071161421e-01_dp,  3.09287976144302701e-01_dp,  5.17203741948951023e-01_dp,  8.56873541617053647e-01_dp,&
            1.41832565744026207e+00_dp,  2.40026352880592597e+00_dp,  4.53061282785091457e+00_dp]
       aw%aw_erange_matrix(:, 38) = [2.19404256848981043e-07_dp,  1.84893021480692055e-06_dp,  8.43625704482602870e-06_dp,  2.98539805089761292e-05_dp,  9.02519223531514164e-05_dp,&
            2.44242657282476964e-04_dp,  6.08129647216171612e-04_dp,  1.41799756751927273e-03_dp,  3.13453794163316875e-03_dp,  6.62735814303910126e-03_dp,  1.34918126483430569e-02_dp,&
            2.65829458200730478e-02_dp,  5.08996208966378191e-02_dp,  9.50257265065573287e-02_dp,  1.73447302844207202e-01_dp,  3.10236178966879494e-01_dp,  5.44868696002544395e-01_dp,&
            9.41469060432773341e-01_dp,  1.60404226603196398e+00_dp,  2.70449668138230370e+00_dp,  4.54873608362962933e+00_dp,  7.82544076598920668e+00_dp,  6.33674353720899892e-07_dp,&
            3.16142029665097347e-06_dp,  1.15345020495897742e-05_dp,  3.50716751174244654e-05_dp,  9.43328038257060746e-05_dp,  2.32109865626505300e-04_dp,  5.33339583218342948e-04_dp,&
            1.16022830478211113e-03_dp,  2.41270135549469158e-03_dp,  4.83023949766437938e-03_dp,  9.36025275674355617e-03_dp,  1.76318741011527759e-02_dp,  3.23946358386633729e-02_dp,&
            5.82122731713006977e-02_dp,  1.02548721779660013e-01_dp,  1.77459667457144227e-01_dp,  3.02251570077447795e-01_dp,  5.07857954639636011e-01_dp,  8.45025080919816096e-01_dp,&
            1.40406690406700996e+00_dp,  2.38393910989210678e+00_dp,  4.51203496554961259e+00_dp]
       aw%aw_erange_matrix(:, 39) = [1.88351987687530526e-07_dp,  1.65516314143432405e-06_dp,  7.76785348869954718e-06_dp,  2.79714222719367443e-05_dp,  8.55649156628721677e-05_dp,&
            2.33566931271817656e-04_dp,  5.85434841966541754e-04_dp,  1.37238105469931603e-03_dp,  3.04706477699641810e-03_dp,  6.46630606172341350e-03_dp,  1.32057750849448336e-02_dp,&
            2.60912206996127931e-02_dp,  5.00794300971465628e-02_dp,  9.36962113326961776e-02_dp,  1.71351144739202293e-01_dp,  3.07021398554227165e-01_dp,  5.40075954336806552e-01_dp,&
            9.34533420128018766e-01_dp,  1.59432099391613225e+00_dp,  2.69133052375000670e+00_dp,  4.53152801808002970e+00_dp,  7.80365554201054845e+00_dp,  5.51739955394908927e-07_dp,&
            2.88965349261451595e-06_dp,  1.07885628952105954e-05_dp,  3.32527414287060681e-05_dp,  9.02854705937388568e-05_dp,  2.23718084617662434e-04_dp,  5.16899809855151391e-04_dp,&
            1.12951230425542559e-03_dp,  2.35760844161404847e-03_dp,  4.73493932271073885e-03_dp,  9.20075307202328566e-03_dp,  1.73730417593908858e-02_dp,  3.19869002325735771e-02_dp,&
            5.75886203437633443e-02_dp,  1.01623229464899281e-01_dp,  1.76129789220076022e-01_dp,  3.00407486887453057e-01_dp,  5.05402669684611339e-01_dp,  8.41905915259029580e-01_dp,&
            1.40030742906647565e+00_dp,  2.37963135568394124e+00_dp,  4.50713289901048153e+00_dp]
       aw%aw_erange_matrix(:, 40) = [1.23463765871970068e-07_dp,  1.24227014851891724e-06_dp,  6.30099821315938331e-06_dp,  2.37495954184466969e-05_dp,  7.48900605176096643e-05_dp,&
            2.08968957693649965e-04_dp,  5.32666280583382590e-04_dp,  1.26553213999849889e-03_dp,  2.84091478504491443e-03_dp,  6.08477333004282805e-03_dp,  1.25251097567092018e-02_dp,&
            2.49165105968512969e-02_dp,  4.81132640892589572e-02_dp,  9.04993115648662916e-02_dp,  1.66296983651955499e-01_dp,  2.99251001169452469e-01_dp,  5.28465936613628950e-01_dp,&
            9.17699371327465596e-01_dp,  1.57068501829206730e+00_dp,  2.65927189410928433e+00_dp,  4.48957841416491110e+00_dp,  7.75050185857576590e+00_dp,  3.80362342875479197e-07_dp,&
            2.29776591986583113e-06_dp,  9.11381832026984379e-06_dp,  2.90972858209795715e-05_dp,  8.09262360808636432e-05_dp,  2.04132278747352609e-04_dp,  4.78246163371334081e-04_dp,&
            1.05685121252171066e-03_dp,  2.22661125745068867e-03_dp,  4.50733829706144711e-03_dp,  8.81836131340562289e-03_dp,  1.67503996067354943e-02_dp,  3.10030960016899580e-02_dp,&
            5.60797619815002549e-02_dp,  9.93786263009129034e-02_dp,  1.72897283618897418e-01_dp,  2.95916166307739981e-01_dp,  4.99412180283538731e-01_dp,  8.34284318373271327e-01_dp,&
            1.39111094836104243e+00_dp,  2.36908715257658553e+00_dp,  4.49513452427299054e+00_dp]
       aw%aw_erange_matrix(:, 41) = [1.02979676562400021e-07_dp,  1.10854968898761468e-06_dp,  5.80971202290333547e-06_dp,  2.23025644309457052e-05_dp,  7.11719810020676382e-05_dp,&
            2.00298943958802087e-04_dp,  5.13894394260241950e-04_dp,  1.22723782279953353e-03_dp,  2.76657497214461779e-03_dp,  5.94647117621023135e-03_dp,  1.22772700541065420e-02_dp,&
            2.44871162678800998e-02_dp,  4.73921055980099926e-02_dp,  8.93231782165937738e-02_dp,  1.64432528157402086e-01_dp,  2.96377581020370262e-01_dp,  5.24163324919575291e-01_dp,&
            9.11448679670500250e-01_dp,  1.56189379176514986e+00_dp,  2.64733075596364120e+00_dp,  4.47393507086348041e+00_dp,  7.73066360217537518e+00_dp,  3.26171811190760098e-07_dp,&
            2.10104899061767922e-06_dp,  8.53914972141398519e-06_dp,  2.76455700276699606e-05_dp,  7.76156655549408933e-05_dp,  1.97138891792050330e-04_dp,  4.64340969794496488e-04_dp,&
            1.03055189846285229e-03_dp,  2.17895340158323885e-03_dp,  4.42417057412375098e-03_dp,  8.67809718203898725e-03_dp,  1.65212418851616687e-02_dp,  3.06399342642522041e-02_dp,&
            5.55212911142715648e-02_dp,  9.85458312366256639e-02_dp,  1.71695339933153363e-01_dp,  2.94242876376668128e-01_dp,  4.97176480282642719e-01_dp,  8.31435706595659907e-01_dp,&
            1.38766992546048185e+00_dp,  2.36513945874929021e+00_dp,  4.49064259579510860e+00_dp]
    case(24)
       aw%energy_range(:) = [700.0_dp, 8.00e+02_dp, 9.00e+02_dp, 1.00e+03_dp, 2.00e+03_dp, 3.00e+03_dp,&
            4.00e+03_dp, 5.00e+03_dp, 6.00e+03_dp, 7.00e+03_dp, 8.00e+03_dp, 9.00e+03_dp, 1.00e+04_dp, 2.00e+04_dp,&
            3.00e+04_dp, 4.00e+04_dp, 5.00e+04_dp, 6.00e+04_dp, 7.00e+04_dp, 8.00e+04_dp, 9.00e+04_dp, 1.00e+05_dp,&
            2.00e+05_dp, 3.00e+05_dp, 4.00e+05_dp, 5.00e+05_dp, 6.00e+05_dp, 7.00e+05_dp, 9.00e+05_dp, 1.00e+06_dp,&
            2.00e+06_dp, 3.00e+06_dp, 4.00e+06_dp, 8.00e+06_dp, 2.00e+07_dp, 3.00e+07_dp, 4.00e+07_dp, 5.00e+07_dp,&
            6.00e+07_dp, 7.00e+07_dp]
       aw%aw_erange_matrix(:, 1) = [4.61317700479576837e-04_dp,  2.46443268044606398e-03_dp,  6.21345008294119528e-03_dp,  1.19897132374405420e-02_dp,  2.02739670923447989e-02_dp,&
            3.18377099027178576e-02_dp,  4.78641607352428139e-02_dp,  7.00932512854925693e-02_dp,  1.00995744524275372e-01_dp,  1.44001017114561369e-01_dp,  2.03812472187446275e-01_dp,&
            2.86844291102829552e-01_dp,  4.01816272151374665e-01_dp,  5.60555452265255383e-01_dp,  7.79074123523174955e-01_dp,  1.07902724507158587e+00_dp,  1.48971094898826495e+00_dp,&
            2.05088065362743333e+00_dp,  2.81692631784065739e+00_dp,  3.86357451267436502e+00_dp,  5.29999983565710409e+00_dp,  7.29459539669274015e+00_dp,  1.01439426048849182e+01_dp,&
            1.45461238319090658e+01_dp,  1.18767724263906926e-03_dp,  2.84226242662972452e-03_dp,  4.70164567550286974e-03_dp,  6.92948268053140016e-03_dp,  9.76556987556314801e-03_dp,&
            1.35567096474764615e-02_dp,  1.87833434726818599e-02_dp,  2.60829208461077489e-02_dp,  3.62884493956485005e-02_dp,  5.04988006632740294e-02_dp,  7.01833384580881903e-02_dp,&
            9.73204001278768216e-02_dp,  1.34577004251765914e-01_dp,  1.85546146090968617e-01_dp,  2.55067662909368387e-01_dp,  3.49675457527506173e-01_dp,  4.78250711655921312e-01_dp,&
            6.53047794479161281e-01_dp,  8.91478429703930542e-01_dp,  1.21963437813600217e+00_dp,  1.68034347718683685e+00_dp,  2.35520511567018120e+00_dp,  3.44298876450068247e+00_dp,&
            5.72565205661532151e+00_dp]
       aw%aw_erange_matrix(:, 2) = [4.11980014757813294e-04_dp,  2.20218766663119207e-03_dp,  5.55862987655223253e-03_dp,  1.07453858648086705e-02_dp,  1.82164123493948234e-02_dp,&
            2.87041419880564239e-02_dp,  4.33359895250264304e-02_dp,  6.37736777702597457e-02_dp,  9.23826170114171730e-02_dp,  1.32457260670822530e-01_dp,  1.88535709332668000e-01_dp,&
            2.66836289436336582e-01_dp,  3.75853089494852632e-01_dp,  5.27160408446637252e-01_dp,  7.36497424890991237e-01_dp,  1.02523787513350029e+00_dp,  1.42240794982540342e+00_dp,&
            1.96753139965474189e+00_dp,  2.71483737564820293e+00_dp,  3.73999376922786686e+00_dp,  5.15223275732461516e+00_dp,  7.12008431709909395e+00_dp,  9.94016617156099969e+00_dp,&
            1.43097188153594352e+01_dp,  1.06080168802775662e-03_dp,  2.54170514486310849e-03_dp,  4.21420577207505098e-03_dp,  6.23348669844608384e-03_dp,  8.82867531556557998e-03_dp,&
            1.23323780583169552e-02_dp,  1.72055167230363085e-02_dp,  2.40604218131138421e-02_dp,  3.37013637092306306e-02_dp,  4.71970817466301068e-02_dp,  6.59863937004311618e-02_dp,&
            9.20172060027153044e-02_dp,  1.27927606892525386e-01_dp,  1.77286101655570294e-01_dp,  2.44917205757472778e-01_dp,  3.37355128111671732e-01_dp,  4.63504613240878216e-01_dp,&
            6.35674025908022466e-01_dp,  8.71362734772751835e-01_dp,  1.19677651746772584e+00_dp,  1.65485388494679975e+00_dp,  2.32722153133878740e+00_dp,  3.41234946707974229e+00_dp,&
            5.69022770301579861e+00_dp]
       aw%aw_erange_matrix(:, 3) = [3.72806520431575163e-04_dp,  1.99388436321399796e-03_dp,  5.03811424246158705e-03_dp,  9.75517179974083257e-03_dp,  1.65765442512157170e-02_dp,&
            2.62016561693737791e-02_dp,  3.97104181416016513e-02_dp,  5.86977067051166548e-02_dp,  8.54388560953810161e-02_dp,  1.23112995790628954e-01_dp,  1.76116725194692542e-01_dp,&
            2.50500112554284782e-01_dp,  3.54562363691661109e-01_dp,  4.99658274778876366e-01_dp,  7.01288277111772440e-01_dp,  9.80578466724679298e-01_dp,  1.36631525250825869e+00_dp,&
            1.89781419909175453e+00_dp,  2.62915704460106348e+00_dp,  3.63595379763080739e+00_dp,  5.02748284753467001e+00_dp,  6.97239513779703390e+00_dp,  9.76735138988749441e+00_dp,&
            1.41088911813171514e+01_dp,  9.60054683003752954e-04_dp,  2.30285140794512459e-03_dp,  3.82626089893532912e-03_dp,  5.67832873667764668e-03_dp,  8.07909101155888758e-03_dp,&
            1.13488960778253562e-02_dp,  1.59317538928749752e-02_dp,  2.24184032766560244e-02_dp,  3.15885952094700703e-02_dp,  4.44852695692428135e-02_dp,  6.25207250741233039e-02_dp,&
            8.76159789193741573e-02_dp,  1.22383066659518827e-01_dp,  1.70367866048186239e-01_dp,  2.36379862462686358e-01_dp,  3.26951588065993282e-01_dp,  4.51006378864590440e-01_dp,&
            6.20898175320340973e-01_dp,  8.54202371646161751e-01_dp,  1.17722564617834635e+00_dp,  1.63300707797638411e+00_dp,  2.30320448441095449e+00_dp,  3.38603970674368115e+00_dp,&
            5.65982601526982343e+00_dp]
       aw%aw_erange_matrix(:, 4) = [3.40894989658156158e-04_dp,  1.82413360015617294e-03_dp,  4.61364657059914075e-03_dp,  8.94685377476417409e-03_dp,  1.52360305182347028e-02_dp,&
            2.41521932880958377e-02_dp,  3.67341083233717822e-02_dp,  5.45185761920902526e-02_dp,  7.97026688006645112e-02_dp,  1.15365325762695883e-01_dp,  1.65780012083858674e-01_dp,&
            2.36849876892240685e-01_dp,  3.36703186808777699e-01_dp,  4.76501207076854616e-01_dp,  6.71532675088139408e-01_dp,  9.42702715780466827e-01_dp,  1.31858217642094422e+00_dp,&
            1.83829760971432665e+00_dp,  2.55579468298687518e+00_dp,  3.54662686975614116e+00_dp,  4.92011019612566258e+00_dp,  6.84500370663874502e+00_dp,  9.61801465982363268e+00_dp,&
            1.39350872540719504e+01_dp,  8.77977092803098593e-04_dp,  2.10811603342637727e-03_dp,  3.50954342145100209e-03_dp,  5.22417896258082007e-03_dp,  7.46417514899641848e-03_dp,&
            1.05391312290683773e-02_dp,  1.48782012757832979e-02_dp,  2.10532969348070191e-02_dp,  2.98229050916155036e-02_dp,  4.22074870861415263e-02_dp,  5.95959414534450033e-02_dp,&
            8.38852069850493515e-02_dp,  1.17663630726851889e-01_dp,  1.64456093340965559e-01_dp,  2.29057495231711011e-01_dp,  3.17997376089129424e-01_dp,  4.40213987897014047e-01_dp,&
            6.08100391202702206e-01_dp,  8.39298888882065830e-01_dp,  1.16020653656844219e+00_dp,  1.61395451350079222e+00_dp,  2.28223380167740286e+00_dp,  3.36305647785424844e+00_dp,&
            5.63328102292555677e+00_dp]
       aw%aw_erange_matrix(:, 5) = [1.88795805568305188e-04_dp,  1.01400115142319413e-03_dp,  2.58300392260153115e-03_dp,  5.06584363608998407e-03_dp,  8.76730367286805874e-03_dp,&
            1.41958066456900449e-02_dp,  2.21500852814415922e-02_dp,  3.38258175595678862e-02_dp,  5.09607130548969528e-02_dp,  7.60446026209399273e-02_dp,  1.12620940361637653e-01_dp,&
            1.65710194214140877e-01_dp,  2.42397023416488389e-01_dp,  3.52640217042394732e-01_dp,  5.10388023217978382e-01_dp,  7.35116678054602901e-01_dp,  1.05396843833842668e+00_dp,&
            1.50477651092529774e+00_dp,  2.14050598056043739e+00_dp,  3.03623428950970142e+00_dp,  4.30143635337295560e+00_dp,  6.10556518287263117e+00_dp,  8.74579010501390641e+00_dp,&
            1.29148008998208805e+01_dp,  4.86654817252449518e-04_dp,  1.17725533742895250e-03_dp,  1.98830281754513094e-03_dp,  3.02701082824698847e-03_dp,  4.45920039261790517e-03_dp,&
            6.52920049323299067e-03_dp,  9.57628576953010319e-03_dp,  1.40613554274847882e-02_dp,  2.06184517578951940e-02_dp,  3.01329443607712862e-02_dp,  4.38468275407232905e-02_dp,&
            6.34991502937741203e-02_dp,  9.15159897521184845e-02_dp,  1.31269865123574248e-01_dp,  1.87436793299735255e-01_dp,  2.66494955363848318e-01_dp,  3.77442432795629668e-01_dp,&
            5.32890432269656378e-01_dp,  7.50891768492044198e-01_dp,  1.05843323548548329e+00_dp,  1.49929102855462681e+00_dp,  2.15548436751696793e+00_dp,  3.22390716218214379e+00_dp,&
            5.47281923289078875e+00_dp]
       aw%aw_erange_matrix(:, 6) = [1.33427979879733433e-04_dp,  7.18433254841546395e-04_dp,  1.83906927024921757e-03_dp,  3.63506515744753747e-03_dp,  6.36149785981150306e-03_dp,&
            1.04491270264946770e-02_dp,  1.65795527241529989e-02_dp,  2.57813106025258330e-02_dp,  3.95671230072276656e-02_dp,  6.01362603417052582e-02_dp,  9.06664868182099210e-02_dp,&
            1.35726987616870548e-01_dp,  2.01856747454321001e-01_dp,  2.98370867095851700e-01_dp,  4.38482092638891707e-01_dp,  6.40861383013148767e-01_dp,  9.31820828562283765e-01_dp,&
            1.34841254883383077e+00_dp,  1.94297380038203182e+00_dp,  2.79022945855977467e+00_dp,  3.99967675626622343e+00_dp,  5.74114862457286090e+00_dp,  8.31217874731993867e+00_dp,&
            1.24040095890847244e+01_dp,  3.44130368286081915e-04_dp,  8.36706427768368899e-04_dp,  1.42714852826577037e-03_dp,  2.20633507238329448e-03_dp,  3.31708266638522549e-03_dp,&
            4.97022951206217267e-03_dp,  7.45981483004476333e-03_dp,  1.11925033131611302e-02_dp,  1.67408732612223804e-02_dp,  2.49202508268891523e-02_dp,  3.68922396160049557e-02_dp,&
            5.43053576506714181e-02_dp,  7.94882733701729322e-02_dp,  1.15716499858170993e-01_dp,  1.67582052787808633e-01_dp,  2.41511378984097774e-01_dp,  3.46509438124794511e-01_dp,&
            4.95283964065632909e-01_dp,  7.06101175609208598e-01_dp,  1.00628208022072796e+00_dp,  1.44000076131204913e+00_dp,  2.08954179767392523e+00_dp,  3.15133029335533754e+00_dp,&
            5.38929409080123012e+00_dp]
       aw%aw_erange_matrix(:, 7) = [1.04250475968719529e-04_dp,  5.62434846586883390e-04_dp,  1.44529215232509214e-03_dp,  2.87438545558675335e-03_dp,  5.07444496778178353e-03_dp,&
            8.42791403765329357e-03_dp,  1.35428891589042914e-02_dp,  2.13430157659925332e-02_dp,  3.31995961636812609e-02_dp,  5.11278905675659201e-02_dp,  7.80713885758039378e-02_dp,&
            1.18306355073952013e-01_dp,  1.78012762503832633e-01_dp,  2.66076339391266126e-01_dp,  3.95212155199796689e-01_dp,  5.83537857385897696e-01_dp,  8.56785151592705407e-01_dp,&
            1.25144833027563851e+00_dp,  1.81940222905018301e+00_dp,  2.63509388185717119e+00_dp,  3.80800630789495509e+00_dp,  5.50822584360313794e+00_dp,  8.03357315982872500e+00_dp,&
            1.20744336863641823e+01_dp,  2.68996632674163355e-04_dp,  6.56625167389525942e-04_dp,  1.12870041866375997e-03_dp,  1.76601824232699143e-03_dp,  2.69672945425687835e-03_dp,&
            4.11011475066255620e-03_dp,  6.27162578892574273e-03_dp,  9.55395941180009305e-03_dp,  1.44904153087516245e-02_dp,  2.18501207381241840e-02_dp,  3.27402220142435363e-02_dp,&
            4.87465497783093477e-02_dp,  7.21287903700857935e-02_dp,  1.06091753018451918e-01_dp,  1.55163298946024042e-01_dp,  2.25725405889390468e-01_dp,  3.26776910843770640e-01_dp,&
            4.71080795343457037e-01_dp,  6.77041874442256497e-01_dp,  9.72210621290131027e-01_dp,  1.40104765578737966e+00_dp,  2.04605142372271365e+00_dp,  3.10338539062153806e+00_dp,&
            5.33417799388986769e+00_dp]
       aw%aw_erange_matrix(:, 8) = [8.60702692156677649e-05_dp,  4.65114939959932621e-04_dp,  1.19907141731334676e-03_dp,  2.39707428680903433e-03_dp,  4.26279719994325425e-03_dp,&
            7.14473113931465495e-03_dp,  1.15991031196004540e-02_dp,  1.84755620863822669e-02_dp,  2.90453917178807412e-02_dp,  4.51928342290498289e-02_dp,  6.96930782895108097e-02_dp,&
            1.06609942703662974e-01_dp,  1.61860507551532584e-01_dp,  2.44013029323841529e-01_dp,  3.65410912297271961e-01_dp,  5.43754167744837758e-01_dp,  8.04331171810440226e-01_dp,&
            1.18320390842443945e+00_dp,  1.73188145034004104e+00_dp,  2.52458166839406806e+00_dp,  3.67076033844195226e+00_dp,  5.34069001995255821e+00_dp,  7.83242599609456835e+00_dp,&
            1.18357747699633205e+01_dp,  2.22168392942659196e-04_dp,  5.44112340898784920e-04_dp,  9.41381930808274890e-04_dp,  1.48771980108678759e-03_dp,  2.30078547368130389e-03_dp,&
            3.55441144055798327e-03_dp,  5.49385058844628626e-03_dp,  8.46781237274265262e-03_dp,  1.29813311762030457e-02_dp,  1.97695679484370945e-02_dp,  2.98989679463123291e-02_dp,&
            4.49078586761989459e-02_dp,  6.70028857513928350e-02_dp,  9.93334764617979371e-02_dp,  1.46375777052357187e-01_dp,  2.14473541561244596e-01_dp,  3.12615211071711419e-01_dp,&
            4.53599442273293074e-01_dp,  6.55931125535817783e-01_dp,  9.47333528911167511e-01_dp,  1.37249031352070938e+00_dp,  2.01407793455765693e+00_dp,  3.06809337921977532e+00_dp,&
            5.29363780354398639e+00_dp]
       aw%aw_erange_matrix(:, 9) = [7.35869645249009983e-05_dp,  3.98222439617148128e-04_dp,  1.02950700202266679e-03_dp,  2.06738844202907076e-03_dp,  3.69979762146187881e-03_dp,&
            6.24960775740759315e-03_dp,  1.02338223116612819e-02_dp,  1.64461400978437935e-02_dp,  2.60820040112540938e-02_dp,  4.09257216998439588e-02_dp,  6.36231823837244415e-02_dp,&
            9.80737721947836233e-02_dp,  1.49989582730209625e-01_dp,  2.27689481649009401e-01_dp,  3.43222681429479259e-01_dp,  5.13955927425115533e-01_dp,  7.64820623024491164e-01_dp,&
            1.13152686170774897e+00_dp,  1.66528148450246660e+00_dp,  2.44010733200013785e+00_dp,  3.56542795473851504e+00_dp,  5.21166099607335731e+00_dp,  7.67705939750003274e+00_dp,&
            1.16510073290945790e+01_dp,  1.90006553496260215e-04_dp,  4.66679698561009014e-04_dp,  8.11972748891158828e-04_dp,  1.29432004627338948e-03_dp,  2.02335419246756076e-03_dp,&
            3.16110715631146965e-03_dp,  4.93756480024014170e-03_dp,  7.68324601351366970e-03_dp,  1.18813996660679481e-02_dp,  1.82405979747101814e-02_dp,  2.77950589275952030e-02_dp,&
            4.20451221695035682e-02_dp,  6.31545619227385918e-02_dp,  9.42274356570295957e-02_dp,  1.39696660640596132e-01_dp,  2.05872670840081984e-01_dp,  3.01732069454644758e-01_dp,&
            4.40098225879962446e-01_dp,  6.39552987516172733e-01_dp,  9.27957121150538633e-01_dp,  1.35017632361663553e+00_dp,  1.98903921323124511e+00_dp,  3.04042851829495486e+00_dp,&
            5.26187676953501082e+00_dp]
       aw%aw_erange_matrix(:, 10) = [6.44514573999347920e-05_dp,  3.49225738643587041e-04_dp,  9.05098491367132977e-04_dp,  1.82487183604718010e-03_dp,  3.28411412095126363e-03_dp,&
            5.58544158928153938e-03_dp,  9.21480458048908220e-03_dp,  1.49216002458564274e-02_dp,  2.38410458106975941e-02_dp,  3.76776768918578378e-02_dp,  5.89735493136651773e-02_dp,&
            9.14951983715417694e-02_dp,  1.40788103345794030e-01_dp,  2.14967214972540538e-01_dp,  3.25839716358234444e-01_dp,  4.90496392665096093e-01_dp,  7.33570921668526221e-01_dp,&
            1.09047737920517407e+00_dp,  1.61216535544398987e+00_dp,  2.37248792813998666e+00_dp,  3.48083475126634134e+00_dp,  5.10774102813810416e+00_dp,  7.55163007644648143e+00_dp,&
            1.15015619316919242e+01_dp,  1.66465131597940633e-04_dp,  4.09900723438320251e-04_dp,  7.16764476367776548e-04_dp,  1.15129869659960074e-03_dp,  1.81671653497322556e-03_dp,&
            2.86564216197924930e-03_dp,  4.51598565603269098e-03_dp,  7.08380410476581145e-03_dp,  1.10347859186615826e-02_dp,  1.70558170237268512e-02_dp,  2.61546028825580430e-02_dp,&
            3.97999906179644716e-02_dp,  6.01199150029828877e-02_dp,  9.01801073240176276e-02_dp,  1.34376404462795612e-01_dp,  1.98989782083773575e-01_dp,  2.92984668785511437e-01_dp,&
            4.29202352242147200e-01_dp,  6.26286398050400872e-01_dp,  9.12211091526156537e-01_dp,  1.33199552795959963e+00_dp,  1.96860108029619463e+00_dp,  3.01782812607916640e+00_dp,&
            5.23594152400877633e+00_dp]
       aw%aw_erange_matrix(:, 11) = [5.74573911142637605e-05_dp,  3.11684623020712826e-04_dp,  8.09635130832271513e-04_dp,  1.63834818633580115e-03_dp,  2.96333997615317278e-03_dp,&
            5.07066619532897059e-03_dp,  8.42087391253820358e-03_dp,  1.37271063464835565e-02_dp,  2.20751546004689554e-02_dp,  3.51037841908612078e-02_dp,  5.52690030169233279e-02_dp,&
            8.62266873666506506e-02_dp,  1.33382837623802902e-01_dp,  2.04680840684917709e-01_dp,  3.11723207399472901e-01_dp,  4.71366094888306608e-01_dp,  7.07988534202323549e-01_dp,&
            1.05674961893049946e+00_dp,  1.56837521684017522e+00_dp,  2.31656811658794703e+00_dp,  3.41068391151613692e+00_dp,  5.02135579453298764e+00_dp,  7.44715654865751109e+00_dp,&
            1.13768882928525468e+01_dp,  1.48438750838859814e-04_dp,  3.66354657591505713e-04_dp,  6.43528711138076251e-04_dp,  1.04077773951916867e-03_dp,  1.65601701224966317e-03_dp,&
            2.63413631151620305e-03_dp,  4.18317594997089440e-03_dp,  6.60730297325672587e-03_dp,  1.03575971916804524e-02_dp,  1.61027348493536226e-02_dp,  2.48280067198096421e-02_dp,&
            3.79754683209380384e-02_dp,  5.76423555260552750e-02_dp,  8.68612913875122794e-02_dp,  1.29995688217542371e-01_dp,  1.93300157685571572e-01_dp,  2.85727088968599663e-01_dp,&
            4.20131176055101696e-01_dp,  6.15206996762805436e-01_dp,  8.99025069289258183e-01_dp,  1.31673677564042046e+00_dp,  1.95142115546416606e+00_dp,  2.99881719873486530e+00_dp,&
            5.21413327144736982e+00_dp]
       aw%aw_erange_matrix(:, 12) = [5.19194844599093182e-05_dp,  2.81938310272364628e-04_dp,  7.33891254313736381e-04_dp,  1.49004253136384920e-03_dp,  2.70751936584285148e-03_dp,&
            4.65849717247361281e-03_dp,  7.78222394784469640e-03_dp,  1.27614307050279169e-02_dp,  2.06403321636805109e-02_dp,  3.30021402713332468e-02_dp,  5.22298565539675783e-02_dp,&
            8.18850557283881259e-02_dp,  1.27254363843749468e-01_dp,  1.96133703097384510e-01_dp,  2.99948891930471528e-01_dp,  4.55352600399399010e-01_dp,  6.86501888848262798e-01_dp,&
            1.02833225371209092e+00_dp,  1.53137179368997578e+00_dp,  2.26918848143912966e+00_dp,  3.35110440223032890e+00_dp,  4.94783619446950329e+00_dp,  7.35808960155657665e+00_dp,&
            1.12704558919221842e+01_dp,  1.34163112709142986e-04_dp,  3.31820004635060867e-04_dp,  5.85292383956380639e-04_dp,  9.52525712116287344e-04_dp,  1.52695895112218939e-03_dp,&
            2.44697373117672189e-03_dp,  3.91234116246991139e-03_dp,  6.21720431912608975e-03_dp,  9.80020195791868332e-03_dp,  1.53143843958859476e-02_dp,  2.37256997067326163e-02_dp,&
            3.64529621596812875e-02_dp,  5.55666229010075563e-02_dp,  8.40702126792006177e-02_dp,  1.26298364826408188e-01_dp,  1.88481852726078480e-01_dp,  2.79561367968087338e-01_dp,&
            4.12401859887934819e-01_dp,  6.05741037023105555e-01_dp,  8.87732639884163621e-01_dp,  1.30364414849327814e+00_dp,  1.93666021655104847e+00_dp,  2.98247296013335905e+00_dp,&
            5.19538978702465215e+00_dp]
       aw%aw_erange_matrix(:, 13) = [4.74186432943696765e-05_dp,  2.57746697458575752e-04_dp,  6.72215423261009561e-04_dp,  1.36904849982412988e-03_dp,  2.49822952531189102e-03_dp,&
            4.32007260453564409e-03_dp,  7.25562191687022153e-03_dp,  1.19615994316679373e-02_dp,  1.94465690339826643e-02_dp,  3.12459287321233924e-02_dp,  4.96795967508927919e-02_dp,&
            7.82273487251369526e-02_dp,  1.22071872432134465e-01_dp,  1.88880212928117241e-01_dp,  2.89923179188923408e-01_dp,  4.41674209821309227e-01_dp,  6.68093992391500135e-01_dp,&
            1.00391918666630020e+00_dp,  1.49950080486156123e+00_dp,  2.22828459865320605e+00_dp,  3.29956003603171188e+00_dp,  4.88411601355048663e+00_dp,  7.28077788598311759e+00_dp,&
            1.11779608830889874e+01_dp,  1.22559089642059892e-04_dp,  3.03711714289307680e-04_dp,  5.37776362192977984e-04_dp,  8.80243545699582047e-04_dp,  1.42070033407478650e-03_dp,&
            2.29195134187792178e-03_dp,  3.68670370867581459e-03_dp,  5.89048389132385431e-03_dp,  9.33114202532726519e-03_dp,  1.46480918768852972e-02_dp,  2.27903209466914676e-02_dp,&
            3.51561724177167217e-02_dp,  5.37923877258781602e-02_dp,  8.16765885661760588e-02_dp,  1.23117559944136062e-01_dp,  1.84324333905412557e-01_dp,  2.74226332070221845e-01_dp,&
            4.05696473576583005e-01_dp,  5.97509585343210525e-01_dp,  8.77892506855791011e-01_dp,  1.29221601746443571e+00_dp,  1.92376053724517249e+00_dp,  2.96818176516477417e+00_dp,&
            5.17900498532960363e+00_dp]
       aw%aw_erange_matrix(:, 14) = [2.61083791822088096e-05_dp,  1.42941529537112346e-04_dp,  3.78238298307792835e-04_dp,  7.88329163685841622e-04_dp,  1.48366706040521873e-03_dp,&
            2.65832050595232044e-03_dp,  4.63167445906838716e-03_dp,  7.91463449746751015e-03_dp,  1.33137928011505726e-02_dp,  2.20900805920563811e-02_dp,  3.61964230826096120e-02_dp,&
            5.86305139732095984e-02_dp,  9.39547021811209387e-02_dp,  1.49057156701848881e-01_dp,  2.34259870304458884e-01_dp,  3.64924444193215181e-01_dp,  5.63775440324342236e-01_dp,&
            8.64276277219052314e-01_dp,  1.31561809087683490e+00_dp,  1.99041393475007755e+00_dp,  2.99768336390155410e+00_dp,  4.50863479760469854e+00_dp,  6.82288522285185906e+00_dp,&
            1.06279521240912498e+01_dp,  6.75881961905316794e-05_dp,  1.69941829583506848e-04_dp,  3.09660645286637180e-04_dp,  5.28449765678521083e-04_dp,  8.93909737450426485e-04_dp,&
            1.50755454518753588e-03_dp,  2.52260544486072663e-03_dp,  4.17515964100183094e-03_dp,  6.82943561603700355e-03_dp,  1.10428692695803531e-02_dp,  1.76609395891629099e-02_dp,&
            2.79551936600397379e-02_dp,  4.38230950360496002e-02_dp,  6.80758902725479981e-02_dp,  1.04851842353276908e-01_dp,  1.60209704371014588e-01_dp,  2.42988684117548870e-01_dp,&
            3.66088449395505444e-01_dp,  5.48495011204098826e-01_dp,  8.18883412230972274e-01_dp,  1.22328644924849317e+00_dp,  1.84563625570733736e+00_dp,  2.88146370066102930e+00_dp,&
            5.07966645973249964e+00_dp]
       aw%aw_erange_matrix(:, 15) = [1.84166877500424847e-05_dp,  1.01338248367429024e-04_dp,  2.70891966570577041e-04_dp,  5.73694488797580346e-04_dp,  1.10214286207274627e-03_dp,&
            2.01979102326085933e-03_dp,  3.59919332013810796e-03_dp,  6.28346253476893234e-03_dp,  1.07836615736264969e-02_dp,  1.82283844401024825e-02_dp,  3.03902691978087669e-02_dp,&
            5.00259315263227727e-02_dp,  8.13821346394924122e-02_dp,  1.30944068492507670e-01_dp,  2.08534415819745522e-01_dp,  3.28919285252216531e-01_dp,  5.14148465757719220e-01_dp,&
            7.96974976579003069e-01_dp,  1.22592434973876929e+00_dp,  1.87310901935587482e+00_dp,  2.84735349060911780e+00_dp,  4.32006788637914774e+00_dp,  6.59132616815617300e+00_dp,&
            1.03482965837111074e+01_dp,  4.77290069970175520e-05_dp,  1.21229034727591870e-04_dp,  2.25323962158315027e-04_dp,  3.95274223967029151e-04_dp,  6.88280321531956432e-04_dp,&
            1.19144209154864065e-03_dp,  2.03962888495879150e-03_dp,  3.44498476015290189e-03_dp,  5.73982220072231236e-03_dp,  9.43952078562288431e-03_dp,  1.53354254075933113e-02_dp,&
            2.46314750288419253e-02_dp,  3.91438412593936233e-02_dp,  6.15909001408859735e-02_dp,  9.60125940056078819e-02_dp,  1.48376380429630472e-01_dp,  2.27458649535560359e-01_dp,&
            3.46156859171918130e-01_dp,  5.23555793005025527e-01_dp,  7.88565881874275010e-01_dp,  1.18758903687419193e+00_dp,  1.80494770837808072e+00_dp,  2.83617741341719309e+00_dp,&
            5.02784446982277977e+00_dp]
       aw%aw_erange_matrix(:, 16) = [1.43794768756607054e-05_dp,  7.94407267743463643e-05_dp,  2.14087932379079663e-04_dp,  4.59136410574220309e-04_dp,  8.96028446508450216e-04_dp,&
            1.66976560805800903e-03_dp,  3.02439562000179516e-03_dp,  5.36142841649881168e-03_dp,  9.33261828251444152e-03_dp,  1.59834223811472532e-02_dp,  2.69719480469838291e-02_dp,&
            4.49000383443473317e-02_dp,  7.38097878200917878e-02_dp,  1.19922436646304767e-01_dp,  1.92730359779763410e-01_dp,  3.06601683045230333e-01_dp,  4.83130261167591724e-01_dp,&
            7.54583091101484027e-01_dp,  1.16902310938102127e+00_dp,  1.79820599818541327e+00_dp,  2.75080499784322763e+00_dp,  4.19835486807251801e+00_dp,  6.44124697922546652e+00_dp,&
            1.01664653479358034e+01_dp,  3.72987233959718080e-05_dp,  9.55023307763634883e-05_dp,  1.80306769262708713e-04_dp,  3.23000597716850072e-04_dp,  5.74384980429340519e-04_dp,&
            1.01281517270836044e-03_dp,  1.76185443937899150e-03_dp,  3.01850376221718007e-03_dp,  5.09455509525917009e-03_dp,  8.47801039225177903e-03_dp,  1.39245855006621139e-02_dp,&
            2.25932205171958285e-02_dp,  3.62453228422488644e-02_dp,  5.75358316723112370e-02_dp,  9.04363388859761169e-02_dp,  1.40849109314211712e-01_dp,  2.17502832012251235e-01_dp,&
            3.33286821824373858e-01_dp,  5.07345903543043319e-01_dp,  7.68745571632986624e-01_dp,  1.16413988586492589e+00_dp,  1.77812850683180224e+00_dp,  2.80627812635434282e+00_dp,&
            4.99364996097974512e+00_dp]
       aw%aw_erange_matrix(:, 17) = [1.18698686593928831e-05_dp,  6.57987372519803132e-05_dp,  1.78547872493372021e-04_dp,  3.86965633398256553e-04_dp,  7.64927524801522433e-04_dp,&
            1.44461477591216773e-03_dp,  2.65033992561638154e-03_dp,  4.75461176520393294e-03_dp,  8.36747964438012796e-03_dp,  1.44754415456443269e-02_dp,  2.46547347360661258e-02_dp,&
            4.13957363940960638e-02_dp,  6.85920817827494478e-02_dp,  1.12272209497015801e-01_dp,  1.81685535435211376e-01_dp,  2.90905349470154673e-01_dp,  4.61185157968011961e-01_dp,&
            7.24426017410617407e-01_dp,  1.12833880171398193e+00_dp,  1.74440310606512305e+00_dp,  2.68116884929595223e+00_dp,  4.11025744803576565e+00_dp,  6.33230139095818423e+00_dp,&
            1.00341731712209583e+01_dp,  3.08118052761882747e-05_dp,  7.94316379583479926e-05_dp,  1.51946049989422485e-04_dp,  2.76867314429344680e-04_dp,  5.00546327724966239e-04_dp,&
            8.95303265024346073e-04_dp,  1.57678048653578742e-03_dp,  2.73118078633072508e-03_dp,  4.65550196358321608e-03_dp,  7.81784620563788155e-03_dp,  1.29478320850938706e-02_dp,&
            2.11711793130857778e-02_dp,  3.42085251400267229e-02_dp,  5.46671320559994553e-02_dp,  8.64666120186466519e-02_dp,  1.35458776456228530e-01_dp,  2.10333988073563849e-01_dp,&
            3.23971999669050770e-01_dp,  4.95558891813246472e-01_dp,  7.54273716590565968e-01_dp,  1.14696009064139304e+00_dp,  1.75843162533584119e+00_dp,  2.78429279390467688e+00_dp,&
            4.96851608304748105e+00_dp]
       aw%aw_erange_matrix(:, 18) = [1.01493431280495660e-05_dp,  5.64287459736858947e-05_dp,  1.54048782241414248e-04_dp,  3.36923608381857085e-04_dp,  6.73292683860254046e-04_dp,&
            1.28578896967230009e-03_dp,  2.38399572301077958e-03_dp,  4.31864823262937802e-03_dp,  7.66826727101123547e-03_dp,  1.33744831265954194e-02_dp,  2.29508473085234047e-02_dp,&
            3.88018974502402081e-02_dp,  6.47063094043532072e-02_dp,  1.06542423418234347e-01_dp,  1.73369498044410736e-01_dp,  2.79028802966647560e-01_dp,  4.44504455672566912e-01_dp,&
            7.01405979754009423e-01_dp,  1.09716150449456173e+00_dp,  1.70302614499239291e+00_dp,  2.62744582849347097e+00_dp,  4.04210657742671842e+00_dp,  6.24783410420217855e+00_dp,&
            9.93142792023963317e+00_dp,  2.63626597308958994e-05_dp,  6.83685184434004267e-05_dp,  1.32281726070828857e-04_dp,  2.44526922853104708e-04_dp,  4.48128899840554611e-04_dp,&
            8.10914556311444754e-04_dp,  1.44254868588091119e-03_dp,  2.52097880352028861e-03_dp,  4.33180300591165697e-03_dp,  7.32769485138720818e-03_dp,  1.22179195528638145e-02_dp,&
            2.01021318534979378e-02_dp,  3.26687760109587166e-02_dp,  5.24871984612843082e-02_dp,  8.34353088058046727e-02_dp,  1.31323922327678949e-01_dp,  2.04811435442679157e-01_dp,&
            3.16767923821470954e-01_dp,  4.86409942277473528e-01_dp,  7.43005052876108962e-01_dp,  1.13354766795262840e+00_dp,  1.74302496515801608e+00_dp,  2.76708003629181443e+00_dp,&
            4.94884391877995977e+00_dp]
       aw%aw_erange_matrix(:, 19) = [8.89159792421571919e-06_dp,  4.95679759385973532e-05_dp,  1.36053541829094280e-04_dp,  2.99977481659800306e-04_dp,  6.05167956636287752e-04_dp,&
            1.16678643116254008e-03_dp,  2.18286506262262972e-03_dp,  3.98697545429172167e-03_dp,  7.13264240424145868e-03_dp,  1.25257332181879580e-02_dp,  2.16295848923911474e-02_dp,&
            3.67796513829504942e-02_dp,  6.16616759570113027e-02_dp,  1.02032133648366691e-01_dp,  1.66795229349862156e-01_dp,  2.69602214582802258e-01_dp,  4.31215560915994778e-01_dp,&
            6.83003680179501083e-01_dp,  1.07215935945554053e+00_dp,  1.66974905151588637e+00_dp,  2.58412884818207900e+00_dp,  3.98703522675006061e+00_dp,  6.17945438240648137e+00_dp,&
            9.84813569764865449e+00_dp,  2.31090359611159263e-05_dp,  6.02520826232466015e-05_dp,  1.17764151694591736e-04_dp,  2.20423029443349397e-04_dp,  4.08645842617014101e-04_dp,&
            7.46742145461516856e-04_dp,  1.33964120871897128e-03_dp,  2.35868484184250304e-03_dp,  4.08029370297597824e-03_dp,  6.94465896923445857e-03_dp,  1.16445004865607972e-02_dp,&
            1.92581817007345599e-02_dp,  3.14477152838842297e-02_dp,  5.07511400667911144e-02_dp,  8.10116930296266996e-02_dp,  1.28005765901508001e-01_dp,  2.00364381438578598e-01_dp,&
            3.10948256083292796e-01_dp,  4.78997560880030648e-01_dp,  7.33851748304701790e-01_dp,  1.12262978179298445e+00_dp,  1.73046443461699351e+00_dp,  2.75303631908657787e+00_dp,&
            4.93279721463345133e+00_dp]
       aw%aw_erange_matrix(:, 20) = [7.92943492811425830e-06_dp,  4.43119886893608735e-05_dp,  1.22228438916328571e-04_dp,  2.71462597082496330e-04_dp,  5.52266425565187515e-04_dp,&
            1.07374597432597870e-03_dp,  2.02455078893148087e-03_dp,  3.72424731470214062e-03_dp,  6.70586488633590980e-03_dp,  1.18458140458168742e-02_dp,  2.05658987513696045e-02_dp,&
            3.51442089237571711e-02_dp,  5.91890323565376700e-02_dp,  9.83549064639916010e-02_dp,  1.61415871849318354e-01_dp,  2.61863081860778169e-01_dp,  4.20271543291650895e-01_dp,&
            6.67804863119458081e-01_dp,  1.05145486973064473e+00_dp,  1.64212563949061630e+00_dp,  2.54809424129414674e+00_dp,  3.94113788444330471e+00_dp,  6.12237951803294411e+00_dp,&
            9.77853300456316532e+00_dp,  2.06192331618514530e-05_dp,  5.40232427598474560e-05_dp,  1.06560123283953070e-04_dp,  2.01663689689283911e-04_dp,  3.77635630469695682e-04_dp,&
            6.95932200646762786e-04_dp,  1.25760040368501206e-03_dp,  2.22852306033233727e-03_dp,  3.87749916160977023e-03_dp,  6.63431088389790976e-03_dp,  1.11778258007327515e-02_dp,&
            1.85685094452199342e-02_dp,  3.04460612739809039e-02_dp,  4.93219665747793509e-02_dp,  7.90098877663644311e-02_dp,  1.25256618914649454e-01_dp,  1.96669278768011990e-01_dp,&
            3.06099682906322057e-01_dp,  4.72806941866720010e-01_dp,  7.26190639008665206e-01_dp,  1.11347540533697198e+00_dp,  1.71991912047126894e+00_dp,  2.74123817138487302e+00_dp,&
            4.91931883283546956e+00_dp]
       aw%aw_erange_matrix(:, 21) = [7.16803627752296239e-06_dp,  4.01472861242540915e-05_dp,  1.11245581550157771e-04_dp,  2.48715393760719119e-04_dp,  5.09832385277932164e-04_dp,&
            9.98664196884188442e-04_dp,  1.89603649313176896e-03_dp,  3.50978968722603793e-03_dp,  6.35572049443416942e-03_dp,  1.12853755554045646e-02_dp,  1.96853755382704262e-02_dp,&
            3.37850528172386894e-02_dp,  5.71266496204200641e-02_dp,  9.52775123910288707e-02_dp,  1.56900006222728733e-01_dp,  2.55347499709513814e-01_dp,  4.11033133743181289e-01_dp,&
            6.54943032172548123e-01_dp,  1.03389409665238108e+00_dp,  1.61864812042625017e+00_dp,  2.51741167839956104e+00_dp,  3.90199569344713248e+00_dp,  6.07364192296369865e+00_dp,&
            9.71903856517508125e+00_dp,  1.86483698557531051e-05_dp,  4.90798524334996915e-05_dp,  9.76228211832427784e-05_dp,  1.86586179919061534e-04_dp,  3.52510875396298037e-04_dp,&
            6.54476032558140510e-04_dp,  1.19026386844131257e-03_dp,  2.12113611193764668e-03_dp,  3.70941336484984733e-03_dp,  6.37599906979002557e-03_dp,  1.07879045037535207e-02_dp,&
            1.79902231510591024e-02_dp,  2.96034208909502425e-02_dp,  4.81160033294198430e-02_dp,  7.73159166920895230e-02_dp,  1.22924046545293070e-01_dp,  1.93526319649241829e-01_dp,&
            3.01966139537300782e-01_dp,  4.67518202278870365e-01_dp,  7.19633505022078235e-01_dp,  1.10562814659246533e+00_dp,  1.71086947711974235e+00_dp,  2.73110774807190726e+00_dp,&
            4.90774746482512736e+00_dp]
       aw%aw_erange_matrix(:, 22) = [6.54949147016424086e-06_dp,  3.67599211858730431e-05_dp,  1.02291500164715865e-04_dp,  2.30098974047354389e-04_dp,  4.74930005243569232e-04_dp,&
            9.36573543839715889e-04_dp,  1.78919693255239561e-03_dp,  3.33062482114044114e-03_dp,  6.06187997524775934e-03_dp,  1.08131197488804381e-02_dp,  1.89406030311630204e-02_dp,&
            3.26314625686266457e-02_dp,  5.53706182403844657e-02_dp,  9.26495368264531882e-02_dp,  1.53033136019531224e-01_dp,  2.49754228873652645e-01_dp,  4.03083928263728630e-01_dp,&
            6.43852135287927729e-01_dp,  1.01872118657447630e+00_dp,  1.59832643757846027e+00_dp,  2.49081087202628870e+00_dp,  3.86801386852845352e+00_dp,  6.03128197184597514e+00_dp,&
            9.66728464063949744e+00_dp,  1.70468480459040007e-05_dp,  4.50533009508492739e-05_dp,  9.03089046067231504e-05_dp,  1.74162345743696236e-04_dp,  3.31659425896218565e-04_dp,&
            6.19857458060724576e-04_dp,  1.13373860134602334e-03_dp,  2.03057927006095170e-03_dp,  3.56709326293176438e-03_dp,  6.15647623537543282e-03_dp,  1.04554160078639425e-02_dp,&
            1.74955809810295078e-02_dp,  2.88805867702902064e-02_dp,  4.70787375241496497e-02_dp,  7.58552793104651651e-02_dp,  1.20908097467212469e-01_dp,  1.90804104268044533e-01_dp,&
            2.98378767154727942e-01_dp,  4.62919868474127638e-01_dp,  7.13923126866968127e-01_dp,  1.09878506894649397e+00_dp,  1.70297020149869560e+00_dp,  2.72226075711566873e+00_dp,&
            4.89764341611014054e+00_dp]
       aw%aw_erange_matrix(:, 23) = [3.62316384899741436e-06_dp,  2.06662589591035961e-05_dp,  5.93867981928769439e-05_dp,  1.39664500034251184e-04_dp,  3.02392953597313812e-04_dp,&
            6.23913220384252542e-04_dp,  1.24158348366475721e-03_dp,  2.39714437510383811e-03_dp,  4.50785328572772450e-03_dp,  8.28120766800075590e-03_dp,  1.48975198963633287e-02_dp,&
            2.62969675682382595e-02_dp,  4.56258770476314496e-02_dp,  7.79234553364625548e-02_dp,  1.31168639852292268e-01_dp,  2.17862779049170785e-01_dp,  3.57407097284327568e-01_dp,&
            5.79664656849280702e-01_dp,  9.30329070134009117e-01_dp,  1.47922848122994299e+00_dp,  2.33408033532110970e+00_dp,  3.66687452965429594e+00_dp,  5.77960966222481343e+00_dp,&
            9.35891569613454344e+00_dp,  9.46282935694829789e-06_dp,  2.58239371914388208e-05_dp,  5.47876477692921878e-05_dp,  1.12357899174281211e-04_dp,  2.25410508844244435e-04_dp,&
            4.39823252088218024e-04_dp,  8.34658513466712024e-04_dp,  1.54415035187728013e-03_dp,  2.79222575559080624e-03_dp,  4.94654471638751418e-03_dp,  8.60219334755443939e-03_dp,&
            1.47099223834576687e-02_dp,  2.47707250906284218e-02_dp,  4.11284759244666259e-02_dp,  6.74067205235111266e-02_dp,  1.09157124591444152e-01_dp,  1.74821633710288610e-01_dp,&
            2.77175755009873737e-01_dp,  4.35575233193674860e-01_dp,  6.79781417253571218e-01_dp,  1.05768662814925518e+00_dp,  1.65537289255531617e+00_dp,  2.66886452835402821e+00_dp,&
            4.83668577282495349e+00_dp]
       aw%aw_erange_matrix(:, 24) = [2.56687094857670747e-06_dp,  1.48140478608297225e-05_dp,  4.35497791130441602e-05_dp,  1.05483081347521106e-04_dp,  2.35279200779714979e-04_dp,&
            4.98716166471356516e-04_dp,  1.01631470784968917e-03_dp,  2.00367258026344771e-03_dp,  3.83831240431013699e-03_dp,  7.16862405397350293e-03_dp,  1.30889059367504789e-02_dp,&
            2.34169831350073999e-02_dp,  4.11292984187428753e-02_dp,  7.10353848964340034e-02_dp,  1.20813060809651041e-01_dp,  2.02583315021552357e-01_dp,  3.35289428889970098e-01_dp,&
            5.48278572538722253e-01_dp,  8.86719055490787444e-01_dp,  1.41999108109747230e+00_dp,  2.25556255062890854e+00_dp,  3.56548544062999051e+00_dp,  5.65210809347542398e+00_dp,&
            9.20209037623522264e+00_dp,  6.72075406470941075e-06_dp,  1.87685583158872635e-05_dp,  4.13668609905263349e-05_dp,  8.80705285290741191e-05_dp,  1.82104310081710000e-04_dp,&
            3.64206856695979675e-04_dp,  7.05845136760333502e-04_dp,  1.33003091463308461e-03_dp,  2.44448090513076869e-03_dp,  4.39402586807451350e-03_dp,  7.74244744853903443e-03_dp,&
            1.33988214598457188e-02_dp,  2.28105603270823097e-02_dp,  3.82556134740365747e-02_dp,  6.32811652381851802e-02_dp,  1.03358286124759155e-01_dp,  1.66857298877244459e-01_dp,&
            2.66514261343302805e-01_dp,  4.21712108726296664e-01_dp,  6.62346025261959515e-01_dp,  1.03657129271470283e+00_dp,  1.63081056713585570e+00_dp,  2.64124764570998494e+00_dp,&
            4.80517466483982947e+00_dp]
       aw%aw_erange_matrix(:, 25) = [2.01189546515375101e-06_dp,  1.17233648244081383e-05_dp,  3.50959117953791644e-05_dp,  8.69336997157480054e-05_dp,  1.98162808962633010e-04_dp,&
            4.28198887734239765e-04_dp,  8.87310707804653417e-04_dp,  1.77499333157769465e-03_dp,  3.44404039243215946e-03_dp,  6.50571406299224695e-03_dp,  1.19998262339639714e-02_dp,&
            2.16660879508208393e-02_dp,  3.83716824509428289e-02_dp,  6.67773897929400395e-02_dp,  1.14364666796375455e-01_dp,  1.93004773521509243e-01_dp,  3.21338201704145010e-01_dp,&
            5.28368394359665539e-01_dp,  8.58910473987573475e-01_dp,  1.38203960453247343e+00_dp,  2.20504874997767697e+00_dp,  3.50002411369846511e+00_dp,  5.56954749675741123e+00_dp,&
            9.10031727857536055e+00_dp,  5.27841311303466373e-06_dp,  1.50189207592488828e-05_dp,  3.40851587092681632e-05_dp,  7.45495484921219001e-05_dp,  1.57451713073147425e-04_dp,&
            3.20381451057778646e-04_dp,  6.30059109418049687e-04_dp,  1.20240733871348647e-03_dp,  2.23481334346221916e-03_dp,  4.05744093651400767e-03_dp,  7.21380013754863766e-03_dp,&
            1.25857693863473327e-02_dp,  2.15855187408655474e-02_dp,  3.64472744823981926e-02_dp,  6.06670935229573008e-02_dp,  9.96614146138858764e-02_dp,  1.61751005459189912e-01_dp,&
            2.59642830354010423e-01_dp,  4.12734473920376044e-01_dp,  6.51007217786149228e-01_dp,  1.02279085179190932e+00_dp,  1.61473916165854003e+00_dp,  2.62315361076397657e+00_dp,&
            4.78453508191765398e+00_dp]
       aw%aw_erange_matrix(:, 26) = [1.66650804957725690e-06_dp,  9.79195922229785925e-06_dp,  2.97671963956997093e-05_dp,  7.50889943280584854e-05_dp,  1.74120549715566273e-04_dp,&
            3.81901310856980159e-04_dp,  8.01589645444458201e-04_dp,  1.62141965041956676e-03_dp,  3.17676475082105583e-03_dp,  6.05256166106276500e-03_dp,  1.12497586142683032e-02_dp,&
            2.04520438710679604e-02_dp,  3.64478368323097671e-02_dp,  6.37901575616463962e-02_dp,  1.09817537982207961e-01_dp,  1.86218609625028619e-01_dp,  3.11411354298047838e-01_dp,&
            5.14145288920909826e-01_dp,  8.38972998045112384e-01_dp,  1.35474093066610513e+00_dp,  2.16860835622923531e+00_dp,  3.45268305902671191e+00_dp,  5.50971944265975022e+00_dp,&
            9.02645357677880433e+00_dp,  4.37996254015510915e-06_dp,  1.26639117157945443e-05_dp,  2.94358206429358304e-05_dp,  6.57479994815268992e-05_dp,  1.41144005156891599e-04_dp,&
            2.91016833733939099e-04_dp,  5.78733039505662308e-04_dp,  1.11517019493301544e-03_dp,  2.09031926898599211e-03_dp,  3.82378124258229632e-03_dp,  6.84438590118669875e-03_dp,&
            1.20142093327273401e-02_dp,  2.07196220387161727e-02_dp,  3.51626591189038648e-02_dp,  5.88014907670197443e-02_dp,  9.70117352483137046e-02_dp,  1.58076631362593256e-01_dp,&
            2.54680227579877017e-01_dp,  4.06229158366321563e-01_dp,  6.42766719215347226e-01_dp,  1.01275123016837187e+00_dp,  1.60300933245512400e+00_dp,  2.60993519962017695e+00_dp,&
            4.76945988476785665e+00_dp]
       aw%aw_erange_matrix(:, 27) = [1.42943695055503615e-06_dp,  8.46165424382126785e-06_dp,  2.60698401100851063e-05_dp,  6.67814196952674276e-05_dp,  1.57061980096056565e-04_dp,&
            3.48699609753426410e-04_dp,  7.39534350906485968e-04_dp,  1.50932350422760685e-03_dp,  2.98025358505948608e-03_dp,  5.71723340985497933e-03_dp,  1.06915099200873675e-02_dp,&
            1.95437765453790706e-02_dp,  3.50017679326257297e-02_dp,  6.15351694963551937e-02_dp,  1.06371585743293251e-01_dp,  1.81057398934298813e-01_dp,  3.03836641250375683e-01_dp,&
            5.03259532047688363e-01_dp,  8.23671639355172513e-01_dp,  1.33373790244342372e+00_dp,  2.14050999882194981e+00_dp,  3.41611042866982118e+00_dp,  5.46342911645605334e+00_dp,&
            8.96923709909895805e+00_dp,  3.76280666490803481e-06_dp,  1.10349192639900207e-05_dp,  2.61749472871830625e-05_dp,  5.94785304240196910e-05_dp,  1.29381396050283660e-04_dp,&
            2.69624782417856625e-04_dp,  5.41030666856124102e-04_dp,  1.05062825583379063e-03_dp,  1.98274016404487120e-03_dp,  3.64883628530559678e-03_dp,  6.56639788172501491e-03_dp,&
            1.15821299145321133e-02_dp,  2.00622942977296825e-02_dp,  3.41837277229397746e-02_dp,  5.73747999305673703e-02_dp,  9.49788229467453393e-02_dp,  1.55249040753075052e-01_dp,&
            2.50850676871953726e-01_dp,  4.01196417731231325e-01_dp,  6.36377279055782896e-01_dp,  1.00495221581774308e+00_dp,  1.59388478284124879e+00_dp,  2.59964533909053719e+00_dp,&
            4.75772623961460805e+00_dp]
       aw%aw_erange_matrix(:, 28) = [1.25593072779015605e-06_dp,  7.48509099235730312e-06_dp,  2.33380690075558333e-05_dp,  6.05862425351964338e-05_dp,  1.44216985724140322e-04_dp,&
            3.23477349893590250e-04_dp,  6.92027615642327380e-04_dp,  1.42292839335073692e-03_dp,  2.82790124968306227e-03_dp,  5.45589697607193076e-03_dp,  1.02544101565129903e-02_dp,&
            1.88296350361601139e-02_dp,  3.38604593210769314e-02_dp,  5.97492896911544571e-02_dp,  1.03633917275519655e-01_dp,  1.76945222267432978e-01_dp,  2.97785582188463460e-01_dp,&
            4.94542408381846377e-01_dp,  8.11391530926075077e-01_dp,  1.31684830884414028e+00_dp,  2.11787478911631055e+00_dp,  3.38660403430587609e+00_dp,  5.42603669207294903e+00_dp,&
            8.92297581960380093e+00_dp,  3.31083107841461289e-06_dp,  9.83462735015609646e-06_dp,  2.37431616737304026e-05_dp,  5.47422903320507240e-05_dp,  1.20404032431273536e-04_dp,&
            2.53165446792662674e-04_dp,  5.11825542614317668e-04_dp,  1.00034096162128451e-03_dp,  1.89849149621014197e-03_dp,  3.51120721241224461e-03_dp,  6.34681078723501029e-03_dp,&
            1.12395614532896839e-02_dp,  1.95393860050951801e-02_dp,  3.34025813261771970e-02_dp,  5.62331329589741413e-02_dp,  9.33477903750834642e-02_dp,  1.52974946772401754e-01_dp,&
            2.47763903298591159e-01_dp,  3.97131605130562226e-01_dp,  6.31207427889815320e-01_dp,  9.98632371583498379e-01_dp,  1.58648264165129627e+00_dp,  2.59129305213222505e+00_dp,&
            4.74820305492783490e+00_dp]
       aw%aw_erange_matrix(:, 29) = [1.01778594648656121e-06_dp,  6.13964731036603379e-06_dp,  1.95437043381015915e-05_dp,  5.18826467221919149e-05_dp,  1.25959840810346862e-04_dp,&
            2.87251594465221492e-04_dp,  6.23174152494021744e-04_dp,  1.29672357902945769e-03_dp,  2.60381073572208722e-03_dp,  5.06916370135518206e-03_dp,  9.60407022854281069e-03_dp,&
            1.77619303644410004e-02_dp,  3.21466090409059171e-02_dp,  5.70568253617527582e-02_dp,  9.94914935344446277e-02_dp,  1.70702308494943306e-01_dp,  2.88571152142299225e-01_dp,&
            4.81231097389002938e-01_dp,  7.92591691975457646e-01_dp,  1.29093228148162753e+00_dp,  2.08307176405047878e+00_dp,  3.34115706531417889e+00_dp,  5.36836180585040168e+00_dp,&
            8.85154518963745929e+00_dp,  2.68997557333974691e-06_dp,  8.17318793142506686e-06_dp,  2.03264410175787538e-05_dp,  4.79845597463935636e-05_dp,  1.07440884457087255e-04_dp,&
            2.29172855787619596e-04_dp,  4.68916988302596819e-04_dp,  9.25955715518831149e-04_dp,  1.77312743317322493e-03_dp,  3.30532793617600807e-03_dp,  6.01677224026092323e-03_dp,&
            1.07224755421114620e-02_dp,  1.87470115600596697e-02_dp,  3.22146730966314590e-02_dp,  5.44912851539835832e-02_dp,  9.08517990792797603e-02_dp,  1.49485166818382670e-01_dp,&
            2.43014842555163008e-01_dp,  3.90863195448717893e-01_dp,  6.23218401902415731e-01_dp,  9.88849326774313808e-01_dp,  1.57500959510431038e+00_dp,  2.57833868291318158e+00_dp,&
            4.73343442147548998e+00_dp]
       aw%aw_erange_matrix(:, 30) = [9.32182506534499083e-07_dp,  5.65427050047378655e-06_dp,  1.81641642924285632e-05_dp,  4.86843580634723421e-05_dp,  1.19179404603004141e-04_dp,&
            2.73671187884923781e-04_dp,  5.97153185443552881e-04_dp,  1.24869564104713331e-03_dp,  2.51801395372686782e-03_dp,  4.92030585912688317e-03_dp,  9.35256125525345955e-03_dp,&
            1.73472617732815110e-02_dp,  3.14784531695425571e-02_dp,  5.60035198389070496e-02_dp,  9.78658552320255670e-02_dp,  1.68245317352748835e-01_dp,  2.84935135760699454e-01_dp,&
            4.75965809488915204e-01_dp,  7.85139110856129308e-01_dp,  1.28063839499368748e+00_dp,  2.06922374707535184e+00_dp,  3.32304676889775541e+00_dp,  5.34535083343202011e+00_dp,&
            8.82301987561054979e+00_dp,  2.46663401521989575e-06_dp,  7.57112122388445892e-06_dp,  1.90707407840366502e-05_dp,  4.54661847163986995e-05_dp,  1.02558395312991546e-04_dp,&
            2.20060394488419609e-04_dp,  4.52506637725504329e-04_dp,  8.97337131127042280e-04_dp,  1.72464367837495440e-03_dp,  3.22533744287678255e-03_dp,  5.88801225850025742e-03_dp,&
            1.05199906179829199e-02_dp,  1.84356786606557875e-02_dp,  3.17464912516325359e-02_dp,  5.38028411029115722e-02_dp,  8.98627198152701229e-02_dp,  1.48098961240044624e-01_dp,&
            2.41124261770076231e-01_dp,  3.88362754098591501e-01_dp,  6.20025938789660280e-01_dp,  9.84934139986923918e-01_dp,  1.57041302902575808e+00_dp,  2.57314565679063278e+00_dp,&
            4.72751471396439804e+00_dp]
       aw%aw_erange_matrix(:, 31) = [5.25590892574421312e-07_dp,  3.33043832380041068e-06_dp,  1.14431096247839162e-05_dp,  3.27431485458530691e-05_dp,  8.46350992273292666e-05_dp,&
            2.03153259621234258e-04_dp,  4.59821285980971557e-04_dp,  9.91650001765859547e-04_dp,  2.05322006704024111e-03_dp,  4.10523807758926031e-03_dp,  7.96233658384974327e-03_dp,&
            1.50356841335305951e-02_dp,  2.77253004357191461e-02_dp,  5.00458991089591035e-02_dp,  8.86130494935644941e-02_dp,  1.54180051460468925e-01_dp,  2.64010700262742870e-01_dp,&
            4.45519145014729834e-01_dp,  7.41854921517251187e-01_dp,  1.22061484027573153e+00_dp,  1.98819245481561180e+00_dp,  3.21675627422806087e+00_dp,  5.20996853074945587e+00_dp,&
            8.65488645350951025e+00_dp,  1.40411871441283978e-06_dp,  4.65945746373287958e-06_dp,  1.28088836351098766e-05_dp,  3.25470686510014471e-05_dp,  7.69739430187918623e-05_dp,&
            1.71503599393232409e-04_dp,  3.63832666091625401e-04_dp,  7.40831450577037333e-04_dp,  1.45671193408331381e-03_dp,  2.77917966751328000e-03_dp,  5.16386766644335356e-03_dp,&
            9.37269395086945054e-03_dp,  1.66596630508670449e-02_dp,  2.90591748479299729e-02_dp,  4.98287886466289229e-02_dp,  8.41233836030797211e-02_dp,  1.40016424411814888e-01_dp,&
            2.30051959107940712e-01_dp,  3.73659596847667475e-01_dp,  6.01186234654401463e-01_dp,  9.61760021921176378e-01_dp,  1.54314545194367869e+00_dp,  2.54230382876726591e+00_dp,&
            4.69236395299902842e+00_dp]
       aw%aw_erange_matrix(:, 32) = [3.77746786484820052e-07_dp,  2.47367412749661625e-06_dp,  8.88910608615204707e-06_dp,  2.64595599516958189e-05_dp,  7.05587839095670583e-05_dp,&
            1.73605015730760731e-04_dp,  4.00918296660801555e-04_dp,  8.79201294501707228e-04_dp,  1.84641001010475377e-03_dp,  3.73718127951488998e-03_dp,  7.32635321852410579e-03_dp,&
            1.39659443490608732e-02_dp,  2.59704105931541744e-02_dp,  4.72342122738790837e-02_dp,  8.42092467768616737e-02_dp,  1.47434278696521814e-01_dp,  2.53904839975671481e-01_dp,&
            4.30720296200788055e-01_dp,  7.20694024878018924e-01_dp,  1.19111682847565215e+00_dp,  1.94818633451092960e+00_dp,  3.16407218709918769e+00_dp,  5.14265022994814824e+00_dp,&
            8.57108216879129792e+00_dp,  1.01675696110901377e-06_dp,  3.56683154388218107e-06_dp,  1.03378032813617490e-05_dp,  2.72303186636337344e-05_dp,  6.61190130474000151e-05_dp,&
            1.50405574194169988e-04_dp,  3.24539504225176522e-04_dp,  6.70313968410577820e-04_dp,  1.33423257207291950e-03_dp,  2.57262674798155143e-03_dp,  4.82482714628009336e-03_dp,&
            8.83010058528159621e-03_dp,  1.58120634727405042e-02_dp,  2.77660334220296082e-02_dp,  4.79020030873376218e-02_dp,  8.13214175964597497e-02_dp,  1.36045354853115003e-01_dp,&
            2.24580177693445515e-01_dp,  3.66354910501169195e-01_dp,  5.91782424535910456e-01_dp,  9.50147106691224264e-01_dp,  1.52944143192368198e+00_dp,  2.52677962908059905e+00_dp,&
            4.67467516563689411e+00_dp]
       aw%aw_erange_matrix(:, 33) = [2.99664078973365364e-07_dp,  2.01676529456486806e-06_dp,  7.49812484072044869e-06_dp,  2.29562789622202959e-05_dp,  6.25512438204072960e-05_dp,&
            1.56516836091155691e-04_dp,  3.66388142810575204e-04_dp,  8.12526363145498094e-04_dp,  1.72258719036366107e-03_dp,  3.51495368214549408e-03_dp,  6.93951365252978413e-03_dp,&
            1.33110102712282775e-02_dp,  2.48897248127692254e-02_dp,  4.54936440738658576e-02_dp,  8.14701439600811622e-02_dp,  1.43220425244155236e-01_dp,  2.47567293637350416e-01_dp,&
            4.21406547480265403e-01_dp,  7.07333082751402054e-01_dp,  1.17243749048047086e+00_dp,  1.92278753935211300e+00_dp,  3.13055087517012032e+00_dp,  5.09974132074687425e+00_dp,&
            8.51759375779859873e+00_dp,  8.11840237974868068e-07_dp,  2.97676437467599049e-06_dp,  8.95877340782960216e-06_dp,  2.41887320656559041e-05_dp,  5.97983195236438640e-05_dp,&
            1.37949802576712747e-04_dp,  3.01077435777808875e-04_dp,  6.27802625559085981e-04_dp,  1.25978454436160171e-03_dp,  2.44616712980497359e-03_dp,  4.61592672479145365e-03_dp,&
            8.49387389545272833e-03_dp,  1.52841394595869661e-02_dp,  2.69568604463215691e-02_dp,  4.66912232329309493e-02_dp,  7.95538490348449423e-02_dp,  1.33531359311761549e-01_dp,&
            2.21104809883150316e-01_dp,  3.61701630907670990e-01_dp,  5.85776206087097573e-01_dp,  9.42713589653869088e-01_dp,  1.52065506289460450e+00_dp,  2.51681761990832609e+00_dp,&
            4.66332558154922250e+00_dp]
       aw%aw_erange_matrix(:, 34) = [1.73730848626341147e-07_dp,  1.26912975751936725e-06_dp,  5.15153823606368156e-06_dp,  1.68583242672310579e-05_dp,  4.82488018910973975e-05_dp,&
            1.25356558872712483e-04_dp,  3.02348850888272086e-04_dp,  6.87116047706990687e-04_dp,  1.48687959331074144e-03_dp,  3.08752828321902546e-03_dp,  6.18872533876025505e-03_dp,&
            1.20297098566864904e-02_dp,  2.27603976223081025e-02_dp,  4.20421469029760714e-02_dp,  7.60071669908969072e-02_dp,  1.34772064548270148e-01_dp,  2.34800518610677439e-01_dp,&
            4.02562837605401957e-01_dp,  6.80194494981714026e-01_dp,  1.13436171838198985e+00_dp,  1.87085280236007812e+00_dp,  3.06182420563061042e+00_dp,  5.01157764804825945e+00_dp,&
            8.40751494962753831e+00_dp,  4.80674357513349563e-07_dp,  1.99285021040878236e-06_dp,  6.55476843504883173e-06_dp,  1.87193473234136633e-05_dp,  4.81793844230035082e-05_dp,&
            1.14656730135680914e-04_dp,  2.56580931723435343e-04_dp,  5.46218825474838882e-04_dp,  1.11545180541612024e-03_dp,  2.19881887010929793e-03_dp,  4.20412520900342954e-03_dp,&
            7.82644901475024538e-03_dp,  1.42296156649352728e-02_dp,  2.53313747253629783e-02_dp,  4.42464196352532271e-02_dp,  7.59679212838059170e-02_dp,  1.28409035336740834e-01_dp,&
            2.13995562863774413e-01_dp,  3.52148516834832837e-01_dp,  5.73406097443571627e-01_dp,  9.27362789204974280e-01_dp,  1.50247436216476671e+00_dp,  2.49618240430330474e+00_dp,&
            4.63981967370555548e+00_dp]
       aw%aw_erange_matrix(:, 35) = [8.83557997605877948e-08_dp,  7.47621325013458939e-07_dp,  3.42094325152354304e-06_dp,  1.21275048693836442e-05_dp,  3.67073078597054470e-05_dp,&
            9.94269390815720596e-05_dp,  2.47729277561915806e-04_dp,  5.77957806383662246e-04_dp,  1.27817905627881727e-03_dp,  2.70349188858008379e-03_dp,  5.50551104637513576e-03_dp,&
            1.08506159368095528e-02_dp,  2.07813764065210640e-02_dp,  3.88056832254963860e-02_dp,  7.08434483001099541e-02_dp,  1.26728575363420248e-01_dp,  2.22565577319307151e-01_dp,&
            3.84396180083315220e-01_dp,  6.53889333621266999e-01_dp,  1.09727575495089158e+00_dp,  1.82005115357550351e+00_dp,  2.99435099495450929e+00_dp,  4.92476615078199753e+00_dp,&
            8.29888537518427327e+00_dp,  2.55529326096967382e-07_dp,  1.28101632753650128e-06_dp,  4.68483136949608877e-06_dp,  1.42645386838839181e-05_dp,  3.84046913218155287e-05_dp,&
            9.45648951403460541e-05_dp,  2.17414031756023513e-04_dp,  4.73182154646426033e-04_dp,  9.84365118873844428e-04_dp,  1.97135109764810497e-03_dp,  3.82125539958089091e-03_dp,&
            7.19986438314273742e-03_dp,  1.32309845000683334e-02_dp,  2.37799409987305317e-02_dp,  4.18963448627306043e-02_dp,  7.24985040317744495e-02_dp,  1.23423627531251801e-01_dp,&
            2.07038657087652705e-01_dp,  3.42753920314871008e-01_dp,  5.61187961935596724e-01_dp,  9.12144801594348853e-01_dp,  1.48440167457457406e+00_dp,  2.47563978283315533e+00_dp,&
            4.61642377288386729e+00_dp]
       aw%aw_erange_matrix(:, 36) = [6.73037114169863018e-08_dp,  6.15083928304524313e-07_dp,  2.95799200667729676e-06_dp,  1.08093200246870833e-05_dp,  3.33939114298216740e-05_dp,&
            9.18125355691171255e-05_dp,  2.31401289449927543e-04_dp,  5.44849270884113237e-04_dp,  1.21410857985279684e-03_dp,  2.58437607064648106e-03_dp,  5.29171190470389356e-03_dp,&
            1.04787688883060739e-02_dp,  2.01529709097529389e-02_dp,  3.77717105057609748e-02_dp,  6.91847136654090700e-02_dp,  1.24131995921297408e-01_dp,  2.18598244610203857e-01_dp,&
            3.78481518936924666e-01_dp,  6.45293540998941872e-01_dp,  1.08511722510447250e+00_dp,  1.80334769771032977e+00_dp,  2.97211117186446261e+00_dp,  4.89609522498911520e+00_dp,&
            8.26295542251170545e+00_dp,  1.99905590548142190e-07_dp,  1.09346719997225291e-06_dp,  4.16270885497327146e-06_dp,  1.29775979642102512e-05_dp,  3.55132473075491176e-05_dp,&
            8.85130922866433315e-05_dp,  2.05444546193259424e-04_dp,  4.50593300850056185e-04_dp,  9.43410547046450560e-04_dp,  1.89966434483849686e-03_dp,  3.69967524627809853e-03_dp,&
            6.99955720877200142e-03_dp,  1.29098322668550324e-02_dp,  2.32783313621694361e-02_dp,  4.11328258393498930e-02_dp,  7.13663368852438390e-02_dp,  1.21790184148070746e-01_dp,&
            2.04750863881002110e-01_dp,  3.39654172854953862e-01_dp,  5.57144662627416243e-01_dp,  9.07096256949488255e-01_dp,  1.47839499420113785e+00_dp,  2.46880540504772039e+00_dp,&
            4.60864108894960012e+00_dp]
       aw%aw_erange_matrix(:, 37) = [5.63756690531640412e-08_dp,  5.45141297997350871e-07_dp,  2.70769884954924823e-06_dp,  1.00840013918214356e-05_dp,  3.15478386877207276e-05_dp,&
            8.75303677559782821e-05_dp,  2.22151558418467874e-04_dp,  5.25982385558769624e-04_dp,  1.17741860773814709e-03_dp,  2.51588033812328270e-03_dp,  5.16832913419365200e-03_dp,&
            1.02635057571827584e-02_dp,  1.97881815370499134e-02_dp,  3.71700157932754807e-02_dp,  6.82173313560840711e-02_dp,  1.22614657374409589e-01_dp,  2.16275737349087366e-01_dp,&
            3.75013409641876516e-01_dp,  6.40245931716625027e-01_dp,  1.07796811297597017e+00_dp,  1.79351481908389587e+00_dp,  2.95900624505579835e+00_dp,  4.87918727523437390e+00_dp,&
            8.24175408051627123e+00_dp,  1.71004560103619930e-07_dp,  9.92697182334500706e-07_dp,  3.87517398119211907e-06_dp,  1.22588587859795228e-05_dp,  3.38825784507339864e-05_dp,&
            8.50746811867344256e-05_dp,  1.98603590662673526e-04_dp,  4.37620082429083308e-04_dp,  9.19792981883775355e-04_dp,  1.85817869104775574e-03_dp,  3.62910029291854761e-03_dp,&
            6.88296869424027957e-03_dp,  1.27224568583438127e-02_dp,  2.29850377491460055e-02_dp,  4.06855229641864269e-02_dp,  7.07018891871147998e-02_dp,  1.20829997461524813e-01_dp,&
            2.03404047259240744e-01_dp,  3.37826922494642301e-01_dp,  5.54758380422972652e-01_dp,  9.04113729252530862e-01_dp,  1.47484379397260934e+00_dp,  2.46476324738877839e+00_dp,&
            4.60403828058722464e+00_dp]
       aw%aw_erange_matrix(:, 38) = [4.97727334423554353e-08_dp,  5.02362703653979704e-07_dp,  2.55210099415985428e-06_dp,  9.62796532877697780e-06_dp,  3.03779168827208642e-05_dp,&
            8.48006137249868207e-05_dp,  2.16228093411334309e-04_dp,  5.13855469129673821e-04_dp,  1.15376339955213739e-03_dp,  2.47160443344679372e-03_dp,  5.08839625681514231e-03_dp,&
            1.01237778225120988e-02_dp,  1.95509905546010845e-02_dp,  3.67781907166757280e-02_dp,  6.75865115296859498e-02_dp,  1.21624002892814623e-01_dp,  2.14757714355392260e-01_dp,&
            3.72744327869088288e-01_dp,  6.36940431071695179e-01_dp,  1.07328260087689120e+00_dp,  1.78706575054210970e+00_dp,  2.95040589323683777e+00_dp,  4.86808565133181492e+00_dp,&
            8.22782834614862679e+00_dp,  1.53527937248053923e-07_dp,  9.30292090694303919e-07_dp,  3.69429667736570611e-06_dp,  1.18027092080259345e-05_dp,  3.28412887152215973e-05_dp,&
            8.28687759523755483e-05_dp,  1.94198515391785591e-04_dp,  4.29240839990999558e-04_dp,  9.04499659731136489e-04_dp,  1.83125625161053240e-03_dp,  3.58321300228387710e-03_dp,&
            6.80703685979662000e-03_dp,  1.26002412612291458e-02_dp,  2.27934815659137542e-02_dp,  4.03930275176479434e-02_dp,  7.02669241279521711e-02_dp,  1.20200804967896063e-01_dp,&
            2.02520698022807338e-01_dp,  3.36627475925464603e-01_dp,  5.53190825239166961e-01_dp,  9.02153291225225029e-01_dp,  1.47250849030304987e+00_dp,  2.46210442577815325e+00_dp,&
            4.60101076371775353e+00_dp]
       aw%aw_erange_matrix(:, 39) = [4.54831153544427612e-08_dp,  4.74311903038898438e-07_dp,  2.44888557777143065e-06_dp,  9.32308394742767545e-06_dp,  2.95915288927230389e-05_dp,&
            8.29584078660507352e-05_dp,  2.12218152772813436e-04_dp,  5.05625512349111269e-04_dp,  1.13767652809357344e-03_dp,  2.44144169165903856e-03_dp,  5.03386058151400473e-03_dp,&
            1.00283211490093185e-02_dp,  1.93887642490112345e-02_dp,  3.65099283872732522e-02_dp,  6.71542261219890135e-02_dp,  1.20944571556333424e-01_dp,  2.13715815831136330e-01_dp,&
            3.71185887070233866e-01_dp,  6.34668776570053494e-01_dp,  1.07006079320012026e+00_dp,  1.78262916583945974e+00_dp,  2.94448692129155454e+00_dp,  4.86044271378568205e+00_dp,&
            8.21823878370653027e+00_dp,  1.42165942952855346e-07_dp,  8.89001610702881171e-07_dp,  3.57333074708115803e-06_dp,  1.14958044585542086e-05_dp,  3.21377578790810807e-05_dp,&
            8.13736727282972984e-05_dp,  1.91205375065229427e-04_dp,  4.23535649468275604e-04_dp,  8.94068906357722956e-04_dp,  1.81286680170237983e-03_dp,  3.55182937124549425e-03_dp,&
            6.75504634547574589e-03_dp,  1.25164764040281099e-02_dp,  2.26620736919062662e-02_dp,  4.01922123332519241e-02_dp,  6.99680751359165881e-02_dp,  1.19768218649619368e-01_dp,&
            2.01912999870729831e-01_dp,  3.35801860811374198e-01_dp,  5.52111298303506470e-01_dp,  9.00802638517182452e-01_dp,  1.47089907493926009e+00_dp,  2.46027174684356087e+00_dp,&
            4.59892398993987772e+00_dp]
       aw%aw_erange_matrix(:, 40) = [4.26332790582793822e-08_dp,  4.55544206136908078e-07_dp,  2.37925183473571949e-06_dp,  9.11626334904722569e-06_dp,  2.90560504435212923e-05_dp,&
            8.17004889875167856e-05_dp,  2.09474117457653813e-04_dp,  4.99983900984721685e-04_dp,  1.12663318539456248e-03_dp,  2.42071040105016049e-03_dp,  4.99633835450934323e-03_dp,&
            9.96258442379749191e-03_dp,  1.92769570672496685e-02_dp,  3.63249095219593948e-02_dp,  6.68558923498574209e-02_dp,  1.20475406766376483e-01_dp,  2.12995986723446162e-01_dp,&
            3.70108684828051648e-01_dp,  6.33097934099508097e-01_dp,  1.06783207893617949e+00_dp,  1.77955909527023692e+00_dp,  2.94038989063624623e+00_dp,  4.85515116651880430e+00_dp,&
            8.21159837079276755e+00_dp,  1.34612764719713526e-07_dp,  8.61194007477554715e-07_dp,  3.49125284802499254e-06_dp,  1.12866852981321436e-05_dp,  3.16569858948818157e-05_dp,&
            8.03497133548326689e-05_dp,  1.89151871671935050e-04_dp,  4.19615901888986101e-04_dp,  8.86893879869551276e-04_dp,  1.80020424723260561e-03_dp,  3.53020009634539807e-03_dp,&
            6.71918701354781538e-03_dp,  1.24586613018964967e-02_dp,  2.25713186132279371e-02_dp,  4.00534441211037942e-02_dp,  6.97614577762599025e-02_dp,  1.19468999123271330e-01_dp,&
            2.01492477091762962e-01_dp,  3.35230321284752497e-01_dp,  5.51363731371360744e-01_dp,  8.99867050312488370e-01_dp,  1.46978400514046514e+00_dp,  2.45900184417820222e+00_dp,&
            4.59747803870346328e+00_dp]
       aw%aw_erange_matrix(:, 41) = [3.97972590548516455e-08_dp,  4.36749112881947908e-07_dp,  2.30901822095820461e-06_dp,  8.90669065507326356e-06_dp,  2.85117175356736206e-05_dp,&
            8.04187756982368461e-05_dp,  2.06673117528046558e-04_dp,  4.94216810283568228e-04_dp,  1.11533067884091630e-03_dp,  2.39947110229914162e-03_dp,  4.95786323959217148e-03_dp,&
            9.89512729906643317e-03_dp,  1.91621474132992256e-02_dp,  3.61348098865640135e-02_dp,  6.65492040412706404e-02_dp,  1.19992874044612341e-01_dp,  2.12255329235387591e-01_dp,&
            3.68999882184251282e-01_dp,  6.31480441291888073e-01_dp,  1.06553645346505910e+00_dp,  1.77639597593752963e+00_dp,  2.93616768736423195e+00_dp,  4.84969691214935050e+00_dp,&
            8.20475280537622176e+00_dp,  1.27091452363502344e-07_dp,  8.33187435856259754e-07_dp,  3.40806720221057817e-06_dp,  1.10739933654662787e-05_dp,  3.11667995071938201e-05_dp,&
            7.93037753836723042e-05_dp,  1.87051235559900289e-04_dp,  4.15601409035354721e-04_dp,  8.79538074509779336e-04_dp,  1.78721154926010824e-03_dp,  3.50799043261123059e-03_dp,&
            6.68234146845954162e-03_dp,  1.23992217962518251e-02_dp,  2.24779652765556351e-02_dp,  3.99106362824236602e-02_dp,  6.95487353045896317e-02_dp,  1.19160818883065692e-01_dp,&
            2.01059207738448181e-01_dp,  3.34641268995125973e-01_dp,  5.50593039253774119e-01_dp,  8.98902290488593048e-01_dp,  1.46863396273117752e+00_dp,  2.45769198685639312e+00_dp,&
            4.59598661029939581e+00_dp]
    case(26)
       aw%energy_range(:) = [799.0_dp, 995.0_dp, 1293.0_dp, 1738.0_dp, 2238.0_dp, 3009.0_dp, 4377.0_dp,&
            6256.0_dp, 9034.0_dp, 15564.0_dp, 19500.0_dp, 22300.0_dp, 24783.0_dp, 41198.0_dp, 94407.0_dp, 189080.0_dp,&
            457444.0_dp, 2101965.0_dp, 14140999.0_dp]
       aw%aw_erange_matrix(:, 1) = [0.00037867004555797054_dp,  0.002019373143542281_dp,  0.005074452374668418_dp,  0.009741339051894624_dp,  0.016351249099763064_dp,&
            0.025426574118427834_dp,  0.03775841037254848_dp,  0.05450081932003549_dp,  0.0772810957375601_dp,  0.1083363446608876_dp,  0.15069589349866513_dp,&
            0.20843076717336742_dp,  0.2869911650766578_dp,  0.3936564974164155_dp,  0.5381309695975518_dp,  0.7333307052011886_dp,  0.996427988952117_dp,&
            1.3502505778649674_dp,  1.8251937089224979_dp,  2.4619239536701665_dp,  3.3154239654767634_dp,  4.461579856601731_dp,  6.009262672635442_dp,&
            8.12630074937428_dp,  11.10929348730981_dp,  15.660501844751538_dp,  0.0009745069197489692_dp,  0.002323920525926464_dp,  0.003818499918990434_dp,&
            0.005569525222864092_dp,  0.007736056032946723_dp,  0.01054486610207425_dp,  0.014309337959120037_dp,  0.01944447015531593_dp,  0.026485944451580964_dp,&
            0.03612533116765179_dp,  0.04926670257709911_dp,  0.06710398765236568_dp,  0.09122022393749045_dp,  0.12371484118454482_dp,  0.16736960998240946_dp,&
            0.2258689406847251_dp,  0.30409900038431836_dp,  0.40856816694415354_dp,  0.548030961075663_dp,  0.7344894715293502_dp,  0.9849728438705977_dp,&
            1.3251021306723128_dp,  1.7972852200464182_dp,  2.4831111234607426_dp,  3.582848552118199_dp,  5.8876320817174905_dp]
       aw%aw_erange_matrix(:, 2) = [0.00031416617596844365_dp,  0.0016768070133771967_dp,  0.004220387402167522_dp,  0.008122019548232636_dp,  0.013681535061623657_dp,&
            0.021375962641606915_dp,  0.03193242611639194_dp,  0.04641604327774391_dp,  0.0663338368572065_dp,  0.09376694799993701_dp,  0.13155072067060036_dp,&
            0.18352265415559924_dp,  0.2548586347835214_dp,  0.3525227140309385_dp,  0.4858648853075939_dp,  0.667414665262307_dp,  0.9139380450657599_dp,&
            1.2478576632771592_dp,  1.699195061182984_dp,  2.3083135493129188_dp,  3.13000622971369_dp,  4.240119346476375_dp,  5.747638276053859_dp,&
            7.820581475731412_dp,  10.755506760817696_dp,  15.2531099085639_dp,  0.0008086633399478686_dp,  0.001931716707111731_dp,  0.0031843549489835176_dp,&
            0.0046679761695975755_dp,  0.006529332077378777_dp,  0.008979243990088065_dp,  0.012309502604654843_dp,  0.016906748893981475_dp,  0.023272740862285972_dp,&
            0.03206195264016037_dp,  0.04413963056111703_dp,  0.060659468387992695_dp,  0.08316331669118938_dp,  0.11371010407957528_dp,  0.15504510373245767_dp,&
            0.21082547573489732_dp,  0.28592663287586667_dp,  0.38687165073350616_dp,  0.5224647206101295_dp,  0.7047999539903935_dp,  0.951040871895601_dp,&
            1.286968657510818_dp,  1.7551267580277088_dp,  2.437087428172129_dp,  3.5325565753903243_dp,  5.829332987897013_dp]
       aw%aw_erange_matrix(:, 3) = [0.000251216262630506_dp,  0.0013422657589387847_dp,  0.0033852831378795366_dp,  0.006535679555855439_dp,  0.01105946790435755_dp,&
            0.017384205344456737_dp,  0.02616631610050564_dp,  0.03837171615696803_dp,  0.05537294469791371_dp,  0.07907682374899021_dp,  0.11210158563747868_dp,&
            0.15802208108125124_dp,  0.22170328658555793_dp,  0.3097484547351596_dp,  0.43109811316492785_dp,  0.5978298541062674_dp,  0.826228906983008_dp,&
            1.1382318360717252_dp,  1.5634041077266412_dp,  2.1417299191833186_dp,  2.9277541939220253_dp,  3.9972537939524453_dp,  5.459336924183542_dp,&
            7.482255655971381_dp,  10.362568049094785_dp,  14.79927594838438_dp,  0.0006467895131307655_dp,  0.0015483789662149572_dp,  0.002562990408244027_dp,&
            0.00378128864955077_dp,  0.005336413195947385_dp,  0.0074211270754708444_dp,  0.010302498712647285_dp,  0.014334977955021215_dp,  0.01998237358756267_dp,&
            0.027857715884000284_dp,  0.038782119259036095_dp,  0.05386231670249176_dp,  0.07459076970770032_dp,  0.10297648694866778_dp,  0.1417181535853474_dp,&
            0.19443583384059848_dp,  0.26598602063677856_dp,  0.36290208001606944_dp,  0.4940399248668749_dp,  0.671597172994337_dp,  0.9128949281590978_dp,&
            1.2439090969479831_dp,  1.7073582868867894_dp,  2.3848234540233704_dp,  3.4754029453297375_dp,  5.763153512206444_dp]
       aw%aw_erange_matrix(:, 4) = [0.0001950376405774879_dp,  0.0010434591619187578_dp,  0.002638226324824606_dp,  0.005113292156951265_dp,  0.00870086598207974_dp,&
            0.013778342596377403_dp,  0.020929407226154117_dp,  0.031017048599119232_dp,  0.04527407877688829_dp,  0.06542623021788685_dp,  0.09386550782858209_dp,&
            0.1338911807639374_dp,  0.19003887760085253_dp,  0.26852546690556994_dp,  0.3778478282417773_dp,  0.5295878007623821_dp,  0.739496104873784_dp,&
            1.0289606432203449_dp,  1.427022553455354_dp,  1.9732190338166054_dp,  2.721788596346289_dp,  3.7484019634398744_dp,  5.162286381612833_dp,&
            7.131961441982249_dp,  9.954041146838616_dp,  14.32582187195048_dp,  0.0005022997712497699_dp,  0.0012056339745159519_dp,  0.0020057022650528562_dp,&
            0.0029823455742529125_dp,  0.004254672143777001_dp,  0.0059963576674781395_dp,  0.008448134528374854_dp,  0.011930608474038368_dp,  0.016868224194481772_dp,&
            0.023830776674747995_dp,  0.03359223660591877_dp,  0.04720772204797841_dp,  0.06611395904656978_dp,  0.09226211932358148_dp,  0.12829508657243463_dp,&
            0.17778587596291995_dp,  0.24556239855995798_dp,  0.3381606409098896_dp,  0.4644850568708377_dp,  0.6368412465900805_dp,  0.8727231844202037_dp,&
            1.198329166055601_dp,  1.6565902038491545_dp,  2.329130996306286_dp,  3.414441737018883_dp,  5.692646250049292_dp]
       aw%aw_erange_matrix(:, 5) = [0.0001570075292014777_dp,  0.0008410019818081912_dp,  0.002131222620982682_dp,  0.004145569234358363_dp,  0.007090677510918224_dp,&
            0.011305441203226727_dp,  0.01731697219552165_dp,  0.02590771238094241_dp,  0.03820108916428952_dp,  0.05578097500511556_dp,  0.08086136666211902_dp,&
            0.11652297811189155_dp,  0.16703767866605654_dp,  0.23830959694579532_dp,  0.33847251734047956_dp,  0.47869779524697986_dp,  0.6742880172162485_dp,&
            0.9461646148413084_dp,  1.3229154365464835_dp,  1.8436824667954244_dp,  2.5624225632785795_dp,  3.5546935048363073_dp,  4.929807210724086_dp,&
            6.856512184015644_dp,  9.631510529999963_dp,  13.950799086937636_dp,  0.0004044673162152666_dp,  0.0009731515874605558_dp,  0.0016264516115000137_dp,&
            0.002435950210582391_dp,  0.0035098078378292706_dp,  0.005006471432500407_dp,  0.0071456103103081205_dp,  0.010221155980366915_dp,  0.01462678603467543_dp,&
            0.02089809034158199_dp,  0.02977080559865047_dp,  0.042257136734592034_dp,  0.059746455952240894_dp,  0.08413972088086426_dp,  0.11803003373030124_dp,&
            0.1649465354635964_dp,  0.22968730801733273_dp,  0.31878354856943114_dp,  0.44117315496449167_dp,  0.6092463295611383_dp,  0.8406397450171417_dp,&
            1.161742484167732_dp,  1.6156774111835495_dp,  2.2841323787971173_dp,  3.36513821401701_dp,  5.635684641787822_dp]
       aw%aw_erange_matrix(:, 6) = [0.00012173433282636892_dp,  0.000653044843078349_dp,  0.0016597094198001826_dp,  0.0032432122902587715_dp,  0.005583731099072486_dp,&
            0.00897974690437299_dp,  0.01389829106398585_dp,  0.021035914001762628_dp,  0.03139940878203811_dp,  0.046421111007702805_dp,  0.06812376578216424_dp,&
            0.09935124887637425_dp,  0.1440867725430818_dp,  0.20788864079466032_dp,  0.29848482929563186_dp,  0.42658280937440796_dp,  0.6069727081348725_dp,&
            0.8600354677167223_dp,  1.213825493528422_dp,  1.7070100456167858_dp,  2.393195963336712_dp,  3.3477843441213753_dp,  4.680165709088339_dp,&
            6.5593534066075065_dp,  9.282192058087176_dp,  13.54332387201712_dp,  0.00031370730222591084_dp,  0.0007570695990484558_dp,  0.0012727251500342805_dp,&
            0.0019236429401252328_dp,  0.0028062999069873792_dp,  0.004062578033501044_dp,  0.00588929507373728_dp,  0.00855186255429553_dp,  0.012411138722817702_dp,&
            0.017965594962442332_dp,  0.025908461038187995_dp,  0.03720320806718786_dp,  0.05318444551258281_dp,  0.0756939715032581_dp,  0.10726485539108421_dp,&
            0.1513713746489183_dp,  0.21277116226058032_dp,  0.297982246389678_dp,  0.41597262011480124_dp,  0.5792223633874778_dp,  0.8055284041136077_dp,&
            1.1215023110944804_dp,  1.5705011278690364_dp,  2.234313406989394_dp,  3.310497107984999_dp,  5.57261913216792_dp]
       aw%aw_erange_matrix(:, 7) = [8.814790587980884e-05_dp,  0.0004738555483452858_dp,  0.001209162224838196_dp,  0.0023779784718090265_dp,  0.004131730232456246_dp,&
            0.006724209366814066_dp,  0.010555208090639503_dp,  0.01622487926307075_dp,  0.02460909648355756_dp,  0.036969426022926015_dp,  0.05511124918766332_dp,&
            0.08160627715945637_dp,  0.12010201314867164_dp,  0.17574961207193285_dp,  0.2557933011845474_dp,  0.37038042948632577_dp,  0.5336738618256365_dp,&
            0.7653829566132899_dp,  1.092887558492255_dp,  1.5542408605150586_dp,  2.2025803308040364_dp,  3.113074354623969_dp,  4.395181735531908_dp,&
            6.218242520601409_dp,  8.87933263258288_dp,  13.071612343557268_dp,  0.0002272629071152242_dp,  0.0005507552663901378_dp,  0.000933441836691016_dp,&
            0.001428839311231769_dp,  0.002120237386026644_dp,  0.0031304749854564546_dp,  0.0046303259129662575_dp,  0.006853192947474817_dp,  0.01012290436688526_dp,&
            0.014894995829624644_dp,  0.021812185159477044_dp,  0.03177875816649789_dp,  0.046061554071504564_dp,  0.06642770441212868_dp,  0.09533262717257988_dp,&
            0.1361769876776064_dp,  0.19366000664329205_dp,  0.27427258699605633_dp,  0.38700765699040135_dp,  0.5444449877078907_dp,  0.7645728576669742_dp,&
            1.0742808530249828_dp,  1.517233705376216_dp,  2.175383868205045_dp,  3.245782737215517_dp,  5.498017132429984_dp]
       aw%aw_erange_matrix(:, 8) = [6.476469600717215e-05_dp,  0.00034891783555501116_dp,  0.0008941569546994692_dp,  0.001770490093242039_dp,  0.003106199645499256_dp,&
            0.0051184335340408015_dp,  0.008151285145167414_dp,  0.012724953439821175_dp,  0.019606591631387022_dp,  0.029915037575018_dp,  0.04527205344500337_dp,&
            0.06801655695473027_dp,  0.10150518730808944_dp,  0.1505317627030831_dp,  0.22191039986921127_dp,  0.32528375402693044_dp,  0.4742415082162232_dp,&
            0.6878698417118773_dp,  0.9929114555738491_dp,  1.426826349178114_dp,  2.0422826887022167_dp,  2.914194983288514_dp,  4.152054985305785_dp,&
            5.925500508772846_dp,  8.531868435284434_dp,  12.663118219514553_dp,  0.00016705896835219716_dp,  0.0004066434009563924_dp,  0.0006951433499719869_dp,&
            0.0010783855735630424_dp,  0.0016285976932411544_dp,  0.002452428674095891_dp,  0.003698822692463904_dp,  0.005574655476173014_dp,  0.008372611367675033_dp,&
            0.012511058447330109_dp,  0.01858785984880576_dp,  0.027453700962179424_dp,  0.04031295449132496_dp,  0.058862610090479196_dp,  0.08548341120436852_dp,&
            0.12350282964756092_dp,  0.17755835622889385_dp,  0.25410556179535176_dp,  0.3621484874152968_dp,  0.5143475290193675_dp,  0.728860394738634_dp,&
            1.0328353641593397_dp,  1.470237966477004_dp,  2.1232087233584265_dp,  3.1884016755763915_dp,  5.431943610280663_dp]
       aw%aw_erange_matrix(:, 9) = [4.714267869850108e-05_dp,  0.00025461261481751654_dp,  0.0006556761719344113_dp,  0.001308468824945838_dp,  0.002321150197926946_dp,&
            0.003878461874698709_dp,  0.006274894154406174_dp,  0.009959418801224819_dp,  0.01560204953263909_dp,  0.02419303582580116_dp,  0.03718681238787033_dp,&
            0.056707527706112246_dp,  0.08584014778037409_dp,  0.12904063424870618_dp,  0.19271144671607915_dp,  0.2860061442236756_dp,  0.4219516424878554_dp,&
            0.6190135338939492_dp,  0.903289679923842_dp,  1.3116275192031754_dp,  1.8961959695325519_dp,  2.7316203672239747_dp,  3.927395419802043_dp,&
            5.653448502577673_dp,  8.207416056488858_dp,  12.280210966173971_dp,  0.00012167157119425256_dp,  0.00029765224752192887_dp,  0.0005138446741296063_dp,&
            0.0008093181914490766_dp,  0.0012462946084647903_dp,  0.0019166924309280847_dp,  0.002949916160920999_dp,  0.004529176144148134_dp,  0.006918766955709136_dp,&
            0.010502242217013015_dp,  0.01583456242948638_dp,  0.02371430118668526_dp,  0.03528415663667773_dp,  0.05217068929971722_dp,  0.07667811497799712_dp,&
            0.11205676769912475_dp,  0.16287592945291882_dp,  0.23554669511627027_dp,  0.3390729578857381_dp,  0.4861840636834821_dp,  0.6951983617349604_dp,&
            0.9935218371108636_dp,  1.4254333766864864_dp,  2.073293734219426_dp,  3.133425653975384_dp,  5.368705541720747_dp]
       aw%aw_erange_matrix(:, 10) = [2.9443353786715123e-05_dp,  0.00015968865887835848_dp,  0.0004146519996900374_dp,  0.0008385653229066657_dp,  0.0015154710444555315_dp,&
            0.002590553782265674_dp,  0.004297458120933096_dp,  0.0069977275055487075_dp,  0.011241365987483699_dp,  0.017857778834567364_dp,  0.02808897944065483_dp,&
            0.043782322061949784_dp,  0.06766742773244867_dp,  0.1037518951041952_dp,  0.1578839462681274_dp,  0.2385489270378087_dp,  0.3579930679384087_dp,&
            0.5338068043298525_dp,  0.791161589813274_dp,  1.1660045885058554_dp,  1.7097467191304874_dp,  2.496543051760848_dp,  3.635839789357852_dp,&
            5.297957374408996_dp,  7.7810148585358885_dp,  11.774676503198675_dp,  7.606243073747058e-05_dp,  0.000187654111092642_dp,  0.00032937906982910266_dp,&
            0.0005320965908695892_dp,  0.0008454741794288284_dp,  0.0013429974722876687_dp,  0.0021300795419892347_dp,  0.003360638155325681_dp,  0.005262693302642904_dp,&
            0.008173980176265023_dp,  0.0125917789247931_dp,  0.01924336061080778_dp,  0.029185562811316818_dp,  0.043945018416338716_dp,  0.0657148857554778_dp,&
            0.09763017054646428_dp,  0.1441532549628787_dp,  0.2116173831358168_dp,  0.3090075514896036_dp,  0.4491309451933545_dp,  0.6505180076674358_dp,&
            0.9409371471889534_dp,  1.3651313188527026_dp,  2.0058258806612064_dp,  3.0589785756980588_dp,  5.283171742769873_dp]
       aw%aw_erange_matrix(:, 11) = [2.4221939880648916e-05_dp,  0.00013162518783252502_dp,  0.0003431050538880391_dp,  0.000698189132475488_dp,  0.001272589233863707_dp,&
            0.0021976433259721703_dp,  0.0036856546266738404_dp,  0.00606744928063478_dp,  0.009850534125738348_dp,  0.015806672884776662_dp,  0.025100760660057786_dp,&
            0.03947837634367904_dp,  0.06153686837154947_dp,  0.09511493193970912_dp,  0.14584951983758487_dp,  0.22196820786998617_dp,  0.335412339591795_dp,&
            0.5034260546260962_dp,  0.7508088209241689_dp,  1.1131398816244866_dp,  1.6415127034352714_dp,  2.409875693114849_dp,  3.527638075049888_dp,&
            5.165269756856301_dp,  7.621098250903887_dp,  11.584360794128232_dp,  6.260081073117763e-05_dp,  0.0001550481237336088_dp,  0.00027425459381644776_dp,&
            0.0004482075757040794_dp,  0.0007220779147665958_dp,  0.0011628086873198505_dp,  0.0018674017508154028_dp,  0.0029793398493427954_dp,  0.004713349903984844_dp,&
            0.00739000908391551_dp,  0.011484659947886356_dp,  0.017697084080236986_dp,  0.0270505598375803_dp,  0.04103207563980036_dp,  0.06178995726179555_dp,&
            0.09241161287658448_dp,  0.13731382654384652_dp,  0.20279428302236951_dp,  0.2978245159544139_dp,  0.43523615879690214_dp,  0.6336386632046422_dp,&
            0.9209431632237726_dp,  1.3420830429471422_dp,  1.9799449671624767_dp,  3.0303736903126226_dp,  5.250335959421828_dp]
       aw%aw_erange_matrix(:, 12) = [2.1564828392874976e-05_dp,  0.00011732900716735743_dp,  0.00030658459101179467_dp,  0.0006263105810174703_dp,  0.0011476627771254888_dp,&
            0.001994368166545555_dp,  0.0033669949117497793_dp,  0.005579455105811666_dp,  0.009115754087433222_dp,  0.014715595238257124_dp,  0.023500724387693545_dp,&
            0.037159447358704495_dp,  0.058214268280899095_dp,  0.09040780191303678_dp,  0.1392561892112249_dp,  0.21283882379303048_dp,  0.32292082611394995_dp,&
            0.48654501997143224_dp,  0.728293233827829_dp,  1.0835278065593024_dp,  1.6031531143244464_dp,  2.360992062122655_dp,  3.466427749396279_dp,&
            5.090015465029667_dp,  7.53020860247837_dp,  11.47601339726731_dp,  5.574871333711635e-05_dp,  0.00013841643320123116_dp,  0.0002460246878681958_dp,&
            0.00040498060578160683_dp,  0.0006579585426571892_dp,  0.0010682911223034433_dp,  0.0017283542359863323_dp,  0.002775834478515568_dp,  0.0044179824445092115_dp,&
            0.006965640555090498_dp,  0.010881623599671406_dp,  0.01684992291318925_dp,  0.025874425224763156_dp,  0.03941906165439105_dp,  0.05960588810571241_dp,&
            0.08949416080199679_dp,  0.13347332726212882_dp,  0.1978191873760846_dp,  0.29149390319797547_dp,  0.4273416663588288_dp,  0.6240165631691874_dp,&
            0.9095125059692878_dp,  1.3288753869034111_dp,  1.9650901753832044_dp,  3.013944114046813_dp,  5.231485480677545_dp]
       aw%aw_erange_matrix(:, 13) = [1.968079412594506e-05_dp,  0.00010718493641102105_dp,  0.0002806353581711279_dp,  0.0005751273705017238_dp,  0.0010584284934163331_dp,&
            0.001848589008502448_dp,  0.003137423022955394_dp,  0.005226209962089254_dp,  0.008581348030605833_dp,  0.013918427654363913_dp,  0.022326614770472653_dp,&
            0.03545081033521797_dp,  0.0557565940780925_dp,  0.0869132477657862_dp,  0.1343443806186961_dp,  0.20601549407747916_dp,  0.31355578173538573_dp,&
            0.4738522008187966_dp,  0.7113173707719063_dp,  1.061144148921342_dp,  1.574088149869358_dp,  2.323872276609698_dp,  3.4198568581275923_dp,&
            5.032662209058464_dp,  7.460841068394957_dp,  11.393228561803523_dp,  5.0889409017427654e-05_dp,  0.00012660472263169967_dp,  0.0002259210980998603_dp,&
            0.0003740655095418279_dp,  0.0006118384940478349_dp,  0.000999874816429794_dp,  0.0016270976310772938_dp,  0.0026268356436642305_dp,  0.004200673380690335_dp,&
            0.006652036004218306_dp,  0.010434155159748769_dp,  0.016218897955811638_dp,  0.02499520138410832_dp,  0.038209141684542064_dp,  0.05796233141783732_dp,&
            0.08729199004516089_dp,  0.13056598592688773_dp,  0.19404256395643912_dp,  0.2866758412418721_dp,  0.4213188646546868_dp,  0.6166595702412992_dp,&
            0.900755794469128_dp,  1.3187413549017004_dp,  1.9536794891939209_dp,  3.00131680399958_dp,  5.21699996325917_dp]
       aw%aw_erange_matrix(:, 14) = [1.2674056285680165e-05_dp,  6.938931077849002e-05_dp,  0.0001836100520485683_dp,  0.0003826779493233582_dp,  0.0007202040757498056_dp,&
            0.0012903838734080182_dp,  0.002248236525495258_dp,  0.00384173613615444_dp,  0.006462366726733287_dp,  0.010722120046078792_dp,  0.017568831387303644_dp,&
            0.028457372697558436_dp,  0.04560190325370437_dp,  0.07234504966297221_dp,  0.11369479964375559_dp,  0.1771007906182374_dp,  0.2735712154870606_dp,&
            0.4192744014302692_dp,  0.637834936286395_dp,  0.9636469742972286_dp,  1.4467538592483378_dp,  2.1603857309953622_dp,  3.2137696680006567_dp,&
            4.777814618678099_dp,  7.151554564591586_dp,  11.0231252332385_dp,  3.2810000033385146e-05_dp,  8.24959243779711e-05_dp,  0.00015031831931530549_dp,&
            0.0002565196061853303_dp,  0.0004339110517853371_dp,  0.0007317640998323022_dp,  0.0012244457171152918_dp,  0.0020265452915832964_dp,  0.0033148331774103383_dp,&
            0.005359851622721186_dp,  0.008571929299584299_dp,  0.013568145467107103_dp,  0.021269239209990003_dp,  0.033039016114448296_dp,  0.05088380967787895_dp,&
            0.07773666161595308_dp,  0.11786137108586371_dp,  0.177428416908905_dp,  0.26534598308886587_dp,  0.3944982542877601_dp,  0.583721306443492_dp,&
            0.8613658745137791_dp,  1.2729807136647084_dp,  1.9020150846924464_dp,  2.9440730604569967_dp,  5.151372469471086_dp]
       aw%aw_erange_matrix(:, 15) = [6.186479912580181e-06_dp,  3.4233291323971405e-05_dp,  9.256119767321995e-05_dp,  0.00019951435503347473_dp,  0.000391782808208518_dp,&
            0.0007348447007241284_dp,  0.0013393430685668998_dp,  0.0023881407715703996_dp,  0.004179359736833124_dp,  0.007193105406189777_dp,  0.012193603475754204_dp,&
            0.020384081500457202_dp,  0.03363963393153156_dp,  0.054854535910931194_dp,  0.0884563818462964_dp,  0.1411619416118249_dp,  0.22308161106495417_dp,&
            0.3493255541517698_dp,  0.5423345187791336_dp,  0.835274429372801_dp,  1.277059229828611_dp,  1.9400953490632433_dp,  2.933324383841704_dp,&
            4.428041894721328_dp,  6.724054874045832_dp,  10.508722000292966_dp,  1.605270034684411e-05_dp,  4.1236777606757607e-05_dp,  7.834646084720584e-05_dp,&
            0.00014151002746774427_dp,  0.0002536786392750537_dp,  0.00045044627173442046_dp,  0.0007883280940822237_dp,  0.0013578695679215443_dp,  0.0023029240253660617_dp,&
            0.0038494157006882636_dp,  0.006348322282988488_dp,  0.010339483088961496_dp,  0.016645606665935727_dp,  0.026509781185170438_dp,  0.04179532069895724_dp,&
            0.06527461060266015_dp,  0.10104501326885583_dp,  0.15512766861607885_dp,  0.23633651295613511_dp,  0.3575717157671407_dp,  0.5378622500972828_dp,&
            0.8059829678217727_dp,  1.2081212436199635_dp,  1.8283703742955213_dp,  2.862256950464903_dp,  5.057682060420358_dp]
       aw%aw_erange_matrix(:, 16) = [3.398642214192787e-06_dp,  1.9025312202899538e-05_dp,  5.266161038810626e-05_dp,  0.00011755445811565422_dp,  0.00024054583577387794_dp,&
            0.00047036774204517446_dp,  0.0008915989199018158_dp,  0.0016480620414409948_dp,  0.00298048536052862_dp,  0.005285920675047294_dp,  0.009210353187127669_dp,&
            0.01579153884075367_dp,  0.026677173353753962_dp,  0.044455103994883116_dp,  0.07314891087057689_dp,  0.11895589362798491_dp,  0.19133835932619525_dp,&
            0.3046280564326354_dp,  0.48037697428284243_dp,  0.7508072430736709_dp,  1.1639388014245502_dp,  1.7914938071058262_dp,  2.7421307814095353_dp,&
            4.18739539739007_dp,  6.427708444690209_dp,  10.150038983021496_dp,  8.840904815568487e-06_dp,  2.3243411245798182e-05_dp,  4.61419948505059e-05_dp,&
            8.798822647972772e-05_dp,  0.0001658923006259312_dp,  0.0003074688101295367_dp,  0.0005583537180990002_dp,  0.0009937301893068423_dp,  0.001735787847791623_dp,&
            0.002980424755862107_dp,  0.005037848905051201_dp,  0.008393759041169294_dp,  0.013800737687712429_dp,  0.022413678806554142_dp,  0.03598900168633015_dp,&
            0.057175813550115986_dp,  0.08993936716796855_dp,  0.1401758115607063_dp,  0.21660925146824841_dp,  0.33212798532910376_dp,  0.5058820683958832_dp,&
            0.7669510501463104_dp,  1.1620122321700583_dp,  1.775691260253936_dp,  2.8035588004499257_dp,  4.990540570288636_dp]
       aw%aw_erange_matrix(:, 17) = [1.5924512725077461e-06_dp,  9.091861526358144e-06_dp,  2.617525769293245e-05_dp,  6.171102340282755e-05_dp,  0.00013395463159750724_dp,&
            0.00027703236128610273_dp,  0.0005524440573740519_dp,  0.0010685815369178362_dp,  0.002012774020332085_dp,  0.0037030176407984823_dp,  0.006670370391832498_dp,&
            0.011788596543354984_dp,  0.020475865129561125_dp,  0.03500529115221168_dp,  0.058978461638664194_dp,  0.09804109351785203_dp,  0.16095595007242294_dp,&
            0.26120029165707_dp,  0.41933283009911826_dp,  0.6664993354060413_dp,  1.0496740893941348_dp,  1.6397469396988835_dp,  2.54498795937397_dp,&
            3.9371777982755236_dp,  6.11745128618483_dp,  9.772519404832513_dp,  4.1599386041678434e-06_dp,  1.1373735340879785e-05_dp,  2.420740702971419e-05_dp,&
            4.980537853880897e-05_dp,  0.00010018420860309139_dp,  0.00019590630897285626_dp,  0.0003724648960236611_dp,  0.0006901948951647627_dp,  0.0012498546464018527_dp,&
            0.0022170391172995803_dp,  0.0038600176308915607_dp,  0.006607746433407134_dp,  0.011137935349822067_dp,  0.018509521510997304_dp,  0.030360279138384877_dp,&
            0.04919938780336098_dp,  0.07883793047658633_dp,  0.12502011000770785_dp,  0.19635093903737422_dp,  0.3056814182538295_dp,  0.47227226126957916_dp,&
            0.7255282521113733_dp,  1.112683210316189_dp,  1.7190059451835937_dp,  2.7402160914459284_dp,  4.918151105513321_dp]
       aw%aw_erange_matrix(:, 18) = [4.376362889407548e-07_dp,  2.643443650482503e-06_dp,  8.433114286038146e-06_dp,  2.2439275175400065e-05_dp,  5.458751916823435e-05_dp,&
            0.00012469615312218161_dp,  0.00027090154346575427_dp,  0.0005643774226213476_dp,  0.001134437804378847_dp,  0.0022105491534259005_dp,  0.004191467146338867_dp,&
            0.007757292968144634_dp,  0.014048601297429358_dp,  0.024949196870902246_dp,  0.04352732410856656_dp,  0.07471686858797275_dp,  0.12635898905420878_dp,&
            0.21078138430975438_dp,  0.3471766909066865_dp,  0.5651735745080563_dp,  0.9102276762862731_dp,  1.451965060215205_dp,  2.2979916853626663_dp,&
            3.6203271934806356_dp,  5.721128905491445_dp,  9.287036012890901_dp,  1.15698350323011e-06_dp,  3.5239168925237668e-06_dp,  8.790907369106611e-06_dp,&
            2.0802473841355257e-05_dp,  4.666225571530836e-05_dp,  9.967519249166123e-05_dp,  0.00020419454443829576_dp,  0.0004036329368680413_dp,  0.0007736164897230748_dp,&
            0.0014432538211834921_dp,  0.002629031662183179_dp,  0.004688130413199208_dp,  0.008201289641287034_dp,  0.014100202125235302_dp,  0.023861417673799547_dp,&
            0.03979881543108159_dp,  0.06550130495364699_dp,  0.10648414776164583_dp,  0.17115716235174305_dp,  0.27227855433947445_dp,  0.42921718067635106_dp,&
            0.6717958687439105_dp,  1.048026560686365_dp,  1.6441450980118213_dp,  2.656245656029467_dp,  4.82228579092364_dp]
       aw%aw_erange_matrix(:, 19) = [9.23626066730038e-08_dp,  6.508986506667573e-07_dp,  2.548256199033377e-06_dp,  8.121244988671366e-06_dp,  2.280050217208556e-05_dp,&
            5.838011937544951e-05_dp,  0.00013919237871694767_dp,  0.0003133658603796936_dp,  0.0006727986081208618_dp,  0.0013877632092122146_dp,  0.002765631333703304_dp,&
            0.005348711634689046_dp,  0.010074592422860395_dp,  0.018535050517138794_dp,  0.03338852029027121_dp,  0.05900918410430352_dp,  0.10249708343303293_dp,&
            0.17523445903983162_dp,  0.2952639242558341_dp,  0.4909040641368631_dp,  0.8062587841364091_dp,  1.3097798505481377_dp,  2.1083909166084847_dp,&
            3.3742290423813217_dp,  5.410341564313699_dp,  8.903545764950158_dp,  2.5312239739000636e-07_dp,  9.960996067355009e-07_dp,  3.1628525158730217e-06_dp,&
            8.836550056217969e-06_dp,  2.2389728532275477e-05_dp,  5.264557648617341e-05_dp,  0.0001166810769940904_dp,  0.00024642731945934143_dp,  0.0004998727804625503_dp,&
            0.0009797084412907111_dp,  0.0018638236468942935_dp,  0.00345441392005227_dp,  0.0062559619517267165_dp,  0.011097497158462704_dp,  0.019322067389699597_dp,&
            0.03307729968505806_dp,  0.05575685817980752_dp,  0.09266621527033955_dp,  0.15202316630925258_dp,  0.24647013442575488_dp,  0.39542567933234785_dp,&
            0.6290352095921129_dp,  0.9959743276132635_dp,  1.583367038071677_dp,  2.5877759334182198_dp,  4.74419254971056_dp]
       aw%aw_erange_matrix(:, 20) = [1.6637944502107968e-08_dp,  1.8303408054343632e-07_dp,  9.687195306133241e-07_dp,  3.7389122528334316e-06_dp,  1.1973447021251308e-05_dp,&
            3.378096920589386e-05_dp,  8.68338189070593e-05_dp,  0.00020767968945116668_dp,  0.00046874696407475663_dp,  0.001008553647726778_dp,  0.0020841046186256115_dp,&
            0.0041598936699362606_dp,  0.008056312504669037_dp,  0.015193061067365586_dp,  0.02798246122774749_dp,  0.050456461866690214_dp,  0.08925401954572831_dp,&
            0.15515900119886547_dp,  0.2654733104360881_dp,  0.44765566677475566_dp,  0.7449030289121367_dp,  1.2248551649261998_dp,  1.9939328236196414_dp,&
            3.2243037705041826_dp,  5.219599665831332_dp,  8.666863277748455_dp,  5.318821639180232e-08_dp,  3.4945656362875746e-07_dp,  1.4305518660978924e-06_dp,&
            4.650456118314243e-06_dp,  1.30922605762957e-05_dp,  3.33205562494026e-05_dp,  7.860562991572967e-05_dp,  0.00017467466389322248_dp,  0.0003697061169016621_dp,&
            0.0007513106176187417_dp,  0.0014748157046539873_dp,  0.0028095645215491876_dp,  0.005213528655212034_dp,  0.009451865748843455_dp,  0.016782910646881463_dp,&
            0.029246610819691533_dp,  0.05010725392331038_dp,  0.08452724443086411_dp,  0.14058736988255555_dp,  0.23083686098028847_dp,  0.37470521981828925_dp,&
            0.6025298048228135_dp,  0.9634165096071392_dp,  1.5450976370872966_dp,  2.5445132697805617_dp,  4.694880341120372_dp]
    case(28)
       aw%energy_range(:) = [1545.0_dp, 2002.0_dp, 2600.0_dp, 3300.0_dp, 4000.0_dp, 5000.0_dp, 5800.0_dp,&
            7000.0_dp, 8500.0_dp, 11000.0_dp, 14000.0_dp, 18000.0_dp, 22000.0_dp, 30000.0_dp, 40000.0_dp, 55000.0_dp,&
            75000.0_dp, 100000.0_dp, 140000.0_dp, 200000.0_dp, 280000.0_dp, 400000.0_dp, 700000.0_dp, 1200000.0_dp,&
            2000000.0_dp, 4500000.0_dp, 10000000.0_dp, 50000000.0_dp]
       aw%aw_erange_matrix(:, 1) = [0.00019912749139307142_dp,  0.0010623606743332262_dp,  0.002671743519332409_dp,  0.00513532276440788_dp,  0.008635195625198482_dp,&
            0.013459847519256876_dp,  0.020047630847673463_dp,  0.029039389901087887_dp,  0.04134031090156478_dp,  0.05819756722342521_dp,  0.0813050912035362_dp,&
            0.11294734291964816_dp,  0.156193889583069_dp,  0.21515890144425773_dp,  0.2953443526600425_dp,  0.4040922882505582_dp,  0.5511803231110346_dp,&
            0.7496069714265129_dp,  1.016632455219541_dp,  1.3751727396754474_dp,  1.8557042554346754_dp,  2.4989586763196456_dp,  3.3599578911008456_dp,&
            4.514593597566258_dp,  6.071708534726409_dp,  8.199087984832916_dp,  11.19334614657497_dp,  15.75712024734422_dp,  0.0005125044577801448_dp,&
            0.0012232237231645984_dp,  0.00201318827111783_dp,  0.0029437832219448537_dp,  0.004103311596300065_dp,  0.005618171589079525_dp,  0.007663188401742024_dp,&
            0.010469929287815255_dp,  0.01433805703894519_dp,  0.01965640843594298_dp,  0.026936172517646257_dp,  0.036855644579373535_dp,  0.05031748334107659_dp,&
            0.06852207829804384_dp,  0.09306268185731213_dp,  0.12604985461633972_dp,  0.1702755103243793_dp,  0.22943160937864643_dp,  0.30840755489079047_dp,&
            0.4137086894086003_dp,  0.5540782272946391_dp,  0.741497238773558_dp,  0.9929644420040254_dp,  1.3340651862992305_dp,  1.8071782497597113_dp,&
            2.4938998954135005_dp,  3.5946345041711467_dp,  5.901304226915114_dp]
       aw%aw_erange_matrix(:, 2) = [0.00015937273545708875_dp,  0.0008511031358508971_dp,  0.0021444475784000504_dp,  0.004133806965295614_dp,  0.006979927658415096_dp,&
            0.010939953042854741_dp,  0.016407375175241108_dp,  0.02396000912675003_dp,  0.034417229669099075_dp,  0.04891406525328216_dp,  0.06900316741446434_dp,&
            0.09679560984224389_dp,  0.13515199610246875_dp,  0.18793838914519878_dp,  0.2603666777263224_dp,  0.3594458206099125_dp,  0.494579482258333_dp,&
            0.6783583159834518_dp,  0.927614454599119_dp,  1.2648378420472655_dp,  1.7201130600602677_dp,  2.3338552489887388_dp,  3.1608903963862245_dp,&
            4.2770721150582895_dp,  5.7913663447139845_dp,  7.871758093638662_dp,  10.814809416909151_dp,  15.321476179970999_dp,  0.00041027809665924954_dp,&
            0.0009811723452880425_dp,  0.001620915236450962_dp,  0.0023840944111134044_dp,  0.0033503311465523967_dp,  0.004634503191430588_dp,  0.006395709602901231_dp,&
            0.008844913900028784_dp,  0.012257024616981905_dp,  0.016993101307172367_dp,  0.02353379728419698_dp,  0.03252363542798268_dp,  0.04482787026200198_dp,&
            0.06160612640371776_dp,  0.08440876245542936_dp,  0.11530375228655552_dp,  0.15704466612137769_dp,  0.21329510792248377_dp,  0.28893382637364007_dp,&
            0.39048261973383075_dp,  0.5267381633327601_dp,  0.7097795124230211_dp,  0.9567475153369529_dp,  1.2933958092367415_dp,  1.762243934642854_dp,&
            2.4448656022179347_dp,  3.5410603027618426_dp,  5.839188641520334_dp]
       aw%aw_erange_matrix(:, 3) = [0.00012724350537371705_dp,  0.0006802420548887551_dp,  0.0017174005833444652_dp,  0.0033210405572595314_dp,  0.005632832416553322_dp,&
            0.008881578544535886_dp,  0.013419698685145537_dp,  0.019766846961404197_dp,  0.028663060595610065_dp,  0.041139669406250204_dp,  0.058618369068556976_dp,&
            0.08304861028830621_dp,  0.11709467333563459_dp,  0.16438742365500805_dp,  0.2298611471707931_dp,  0.32020292243210463_dp,  0.44445136041510835_dp,&
            0.6147946527884036_dp,  0.8476375115705392_dp,  1.1650387509190727_dp,  1.5966797391113152_dp,  2.1826420752053064_dp,  2.9775354931417777_dp,&
            4.057156308900191_dp,  5.530582551166736_dp,  7.5660060038749855_dp,  10.45997985096501_dp,  14.911921279047414_dp,  0.00032764611790808316_dp,&
            0.0007852281095230586_dp,  0.001302499732766977_dp,  0.001927928018050743_dp,  0.0027331746862386277_dp,  0.0038223319490506384_dp,  0.005339653436386936_dp,&
            0.0074768267093066575_dp,  0.010485860861582824_dp,  0.01470204224626301_dp,  0.020577182689456824_dp,  0.028723198501018832_dp,  0.03996852241361688_dp,&
            0.05543195597157087_dp,  0.07662014166796133_dp,  0.10555670439639143_dp,  0.14495398753342661_dp,  0.1984431170697882_dp,  0.27088687577758624_dp,&
            0.36881736713632424_dp,  0.501078546468407_dp,  0.6798426309902373_dp,  0.9223911021644032_dp,  1.2546500752337444_dp,  1.719292335657813_dp,&
            2.3978939243834243_dp,  3.489702010303218_dp,  5.779704886427998_dp]
       aw%aw_erange_matrix(:, 4) = [0.0001035812061619439_dp,  0.000554312958948722_dp,  0.0014022178351092648_dp,  0.0027199186238307522_dp,  0.004633626332072742_dp,&
            0.007348901862042884_dp,  0.011184100569752935_dp,  0.016610348266822107_dp,  0.02430141997172798_dp,  0.03520198814732897_dp,  0.05062394356636958_dp,&
            0.0723803681708885_dp,  0.10296857818003698_dp,  0.14581768397450023_dp,  0.2056217187942231_dp,  0.28878665461351954_dp,  0.4040292970175386_dp,&
            0.5631795391897031_dp,  0.7822574690472879_dp,  1.0829291748663241_dp,  1.4945032140919154_dp,  2.056746330708999_dp,  2.8240560994099004_dp,&
            3.87216062398999_dp,  5.31022885971396_dp,  7.306643557351196_dp,  10.157983272563953_dp,  14.562388075641433_dp,  0.000266779348158189_dp,&
            0.0006406783010995833_dp,  0.0010669475657666657_dp,  0.0015890577230093321_dp,  0.0022720547662622424_dp,  0.003210897076964011_dp,  0.004537212534376952_dp,&
            0.006426464787961385_dp,  0.009111540381231737_dp,  0.012906037854636046_dp,  0.018237038812244945_dp,  0.025687953276114096_dp,  0.036054499502532374_dp,&
            0.05041868971863853_dp,  0.070247180509793_dp,  0.0975222869562315_dp,  0.134916933126155_dp,  0.1860296678962944_dp,  0.2557045471811927_dp,&
            0.3504778528812439_dp,  0.4792309189974202_dp,  0.6542157222577841_dp,  0.8928389893219679_dp,  1.2211855078507017_dp,  1.6820762830637361_dp,&
            2.357109509558417_dp,  3.4450757129111187_dp,  5.728067480861656_dp]
       aw%aw_erange_matrix(:, 5) = [8.771437194659891e-05_dp,  0.00046981282949284237_dp,  0.0011904567715385075_dp,  0.0023152670914235655_dp,  0.003959200577898668_dp,&
            0.006310722793342447_dp,  0.009662905002781183_dp,  0.01445070779560698_dp,  0.021298505075191456_dp,  0.031086296591640365_dp,  0.04504371996352949_dp,&
            0.06488109904164477_dp,  0.09296905461591826_dp,  0.13258247030343728_dp,  0.1882303453180191_dp,  0.26610050569058297_dp,  0.3746580755427752_dp,&
            0.5254504247939676_dp,  0.7341916622638109_dp,  1.0222323068269783_dp,  1.418577696146867_dp,  1.9627342158223895_dp,  2.7089187483169885_dp,&
            3.732793667761796_dp,  5.14359311285564_dp,  7.109854244554064_dp,  9.928196448146211_dp,  14.295810062691444_dp,  0.0002259584596775983_dp,&
            0.0005436014065931008_dp,  0.0009083522994476501_dp,  0.001360022022949313_dp,  0.0019587330256291995_dp,  0.0027925453027995255_dp,  0.003983542841235872_dp,&
            0.005695043198214815_dp,  0.008145671980235551_dp,  0.011632693203529134_dp,  0.01656425108725137_dp,  0.023501595060412054_dp,  0.03321470050609412_dp,&
            0.04675631004359308_dp,  0.06556089291033432_dp,  0.09157703564672734_dp,  0.1274448334217016_dp,  0.17673475828550625_dp,  0.2442731132825866_dp,&
            0.33659611474367285_dp,  0.4626113339076691_dp,  0.6346313704421029_dp,  0.8701615672163753_dp,  1.1954150913540855_dp,  1.6533375661453558_dp,&
            2.325557793916785_dp,  3.410528405591019_dp,  5.688122811462863_dp]
       aw%aw_erange_matrix(:, 6) = [7.231152188478278e-05_dp,  0.00038772726865815517_dp,  0.0009844837786035514_dp,  0.0019209126836657545_dp,  0.003300157154162172_dp,&
            0.005292557663606624_dp,  0.008164163748376724_dp,  0.012311171926660728_dp,  0.018304991861179568_dp,  0.026956176212698496_dp,  0.03940566382368068_dp,&
            0.057252298754729467_dp,  0.08272831849932695_dp,  0.11893897414817305_dp,  0.17018831821850053_dp,  0.2424208413642749_dp,  0.34381888188337045_dp,&
            0.48560994949960345_dp,  0.68315888981334_dp,  0.9574526915358277_dp,  1.3371435383436294_dp,  1.8614300851160115_dp,  2.5843101667934096_dp,&
            3.5813587287033566_dp,  4.961876490500659_dp,  6.894579341534884_dp,  9.676154843478928_dp,  14.002777296643712_dp,  0.00018632498066864197_dp,&
            0.00044921855213653585_dp,  0.0007537638296544074_dp,  0.0011359077842782092_dp,  0.0016504930186307747_dp,  0.002378087896208448_dp,  0.0034304274436559386_dp,&
            0.004957778020056624_dp,  0.007163453510160269_dp,  0.010326973559848835_dp,  0.0148355888498722_dp,  0.02122577871325675_dp,  0.030238442499465622_dp,&
            0.04289294382786412_dp,  0.06058666241299866_dp,  0.08522882557656473_dp,  0.11942055225842414_dp,  0.16669801861501937_dp,  0.23186426244014152_dp,&
            0.32145183283272727_dp,  0.44439450297432675_dp,  0.6130709874752484_dp,  0.8450980113616935_dp,  1.1668375144951064_dp,  1.621384532244084_dp,&
            2.290416624110755_dp,  3.3720270933942214_dp,  5.643641293866133_dp]
       aw%aw_erange_matrix(:, 7) = [6.358752692633583e-05_dp,  0.0003412046362365837_dp,  0.0008676052314903817_dp,  0.001696724618707933_dp,  0.0029245257586060763_dp,&
            0.004710233349785592_dp,  0.007303215812339811_dp,  0.011075692416141687_dp,  0.016566304596355927_dp,  0.024542556688053108_dp,  0.03609017098515814_dp,&
            0.05273820097743335_dp,  0.07663171929303365_dp,  0.11076846568614528_dp,  0.15932179622752599_dp,  0.22808006029257497_dp,  0.32504298834606_dp,&
            0.4612301705022223_dp,  0.6517778001414668_dp,  0.9174329759430668_dp,  1.2866126278634253_dp,  1.7983078281539955_dp,  2.506365504219328_dp,&
            3.4862959694100204_dp,  4.847438621720606_dp,  6.7586271477598405_dp,  9.516604972737115_dp,  13.816917061150304_dp,  0.00016387366227471576_dp,&
            0.0003956833555057065_dp,  0.0006658661861317086_dp,  0.0010080085317879625_dp,  0.0014736812958346364_dp,  0.002138763598196897_dp,  0.00310853235933617_dp,&
            0.004525172218371909_dp,  0.006582500342505047_dp,  0.0095488999069641_dp,  0.013798331312490912_dp,  0.01985134913295637_dp,  0.028429999006008653_dp,&
            0.04053182527263095_dp,  0.05752974961085405_dp,  0.08130676932209915_dp,  0.11443765763930915_dp,  0.16043487908983078_dp,  0.2240845367272352_dp,&
            0.311914711856157_dp,  0.4328741422164402_dp,  0.5993829816920151_dp,  0.8291300891292139_dp,  1.1485759543667498_dp,  1.6009173789914857_dp,&
            2.2678714811822975_dp,  3.3473105643753676_dp,  5.615101807506323_dp]
       aw%aw_erange_matrix(:, 8) = [5.40208922943882e-05_dp,  0.0002901574218009231_dp,  0.0007392141971595068_dp,  0.0014500283280455789_dp,  0.002510177021344463_dp,&
            0.0040657991238261465_dp,  0.006346521670349522_dp,  0.009696151598289092_dp,  0.01461448675071523_dp,  0.021817888400210955_dp,  0.03232620376087572_dp,&
            0.047584791601727995_dp,  0.06963362235719665_dp,  0.10134008926088696_dp,  0.14671822700211287_dp,  0.21136500303928102_dp,  0.30305507802937337_dp,&
            0.43255028902361986_dp,  0.6147016140691686_dp,  0.8699548471471977_dp,  1.2264290200274872_dp,  1.7228495271474296_dp,  2.412866754832892_dp,&
            3.371902180409746_dp,  4.7093379807234985_dp,  6.594155669520039_dp,  9.323180769772668_dp,  13.591212022178699_dp,  0.00013925034927611353_dp,&
            0.00033689758020930645_dp,  0.0005691288628998808_dp,  0.000866760965686723_dp,  0.0012774771286669078_dp,  0.0018715412342993915_dp,  0.0027465275546555167_dp,&
            0.004035039597245034_dp,  0.0059195955044671665_dp,  0.0086551765726244_dp,  0.012599568036048066_dp,  0.018253775832994857_dp,  0.02631653618635365_dp,&
            0.03775822887923194_dp,  0.053921093309633066_dp,  0.07665492497469895_dp,  0.10850075471633389_dp,  0.1529401192419044_dp,  0.2147361484742657_dp,&
            0.30040908293128193_dp,  0.41892390451696465_dp,  0.5827503975591537_dp,  0.8096664957210812_dp,  1.1262568684972982_dp,  1.5758496574356475_dp,&
            2.240219834134439_dp,  3.316979586478506_dp,  5.5800996039424104_dp]
       aw%aw_erange_matrix(:, 9) = [4.564555265477108e-05_dp,  0.0002454346685212948_dp,  0.0006265784407134182_dp,  0.001233158622447077_dp,  0.0021448652153409154_dp,&
            0.0034954182642228226_dp,  0.005495608826652996_dp,  0.00846210660098653_dp,  0.012857585326682552_dp,  0.01934938119006627_dp,  0.028893873468546265_dp,&
            0.04285529874482882_dp,  0.06317112153384097_dp,  0.0925808252186819_dp,  0.13494115158016368_dp,  0.19565897567855495_dp,  0.2822840227867385_dp,&
            0.405318874954234_dp,  0.5793256641233082_dp,  0.8244427232040404_dp,  1.168482383587411_dp,  1.6498933305311403_dp,  2.322117411253598_dp,&
            3.2604766389691147_dp,  4.574390024972901_dp,  6.432989341834178_dp,  9.133196407474784_dp,  13.369095389855303_dp,  0.00011768969816977658_dp,&
            0.00028534927032496446_dp,  0.0004840726346740154_dp,  0.0007420579710874475_dp,  0.0011032591807000774_dp,  0.0016325138217318865_dp,  0.002419994753070502_dp,&
            0.0035891515886961726_dp,  0.005311642196826666_dp,  0.007829394245850389_dp,  0.011484246513613412_dp,  0.016757755663488007_dp,  0.024325289635545783_dp,&
            0.03512979233715042_dp,  0.05048226189921966_dp,  0.07219835845965666_dp,  0.1027839824530896_dp,  0.14568785531725492_dp,  0.20564777829998312_dp,&
            0.2891735001868529_dp,  0.4052436947742419_dp,  0.5663760127138803_dp,  0.7904374777760682_dp,  1.1041398360339376_dp,  1.5509491079980848_dp,&
            2.21270825293388_dp,  3.2867828515308894_dp,  5.5452724054483005_dp]
       aw%aw_erange_matrix(:, 10) = [3.6488006929306264e-05_dp,  0.00019649151810584802_dp,  0.000503107877439482_dp,  0.000994818980265892_dp,  0.0017419330458111762_dp,&
            0.00286324584306399_dp,  0.004546790823266163_dp,  0.0070764254736432365_dp,  0.01086988103008825_dp,  0.01653489379720316_dp,  0.024950193189557313_dp,&
            0.0373800572984858_dp,  0.05563473102373159_dp,  0.08229382069514388_dp,  0.12101608504744633_dp,  0.1769674560118736_dp,  0.25741052665895353_dp,&
            0.3725145015640989_dp,  0.5364671734722614_dp,  0.7690049629152851_dp,  1.0975352324996732_dp,  1.5601365551801154_dp,  2.209966148011392_dp,&
            3.1222026085468673_dp,  4.406302512891578_dp,  6.231592566777427_dp,  8.895138025566414_dp,  13.09015522563044_dp,  9.411058161764564e-05_dp,&
            0.00022887458151702356_dp,  0.00039057680221556677_dp,  0.0006042814819193009_dp,  0.0009094035245976271_dp,  0.0013641332935019103_dp,  0.002049648033435726_dp,&
            0.003078322757949694_dp,  0.004608548064372825_dp,  0.006866051954909783_dp,  0.010172626387128262_dp,  0.014985141010389004_dp,  0.021949052283394388_dp,&
            0.03197188778843743_dp,  0.046323942696241904_dp,  0.0667759221190767_dp,  0.09578682159309504_dp,  0.13676065620689198_dp,  0.19439935510505243_dp,&
            0.27519525733539296_dp,  0.3881406112452913_dp,  0.5458114344320582_dp,  0.7661884133896568_dp,  1.0761497928442625_dp,  1.519347409242135_dp,&
            2.177726198492106_dp,  3.2483564776686458_dp,  5.5009815780981555_dp]
       aw%aw_erange_matrix(:, 11) = [2.9588446308604547e-05_dp,  0.00015957764714927812_dp,  0.00040980061859442377_dp,  0.0008141581268268158_dp,  0.0014351942867949477_dp,&
            0.002379218143911262_dp,  0.003815130228038144_dp,  0.0059991869926145835_dp,  0.009311238414156518_dp,  0.014308514375164259_dp,  0.021803455514145805_dp,&
            0.03297432754720971_dp,  0.04952104060181237_dp,  0.07388347387186651_dp,  0.1095460460641794_dp,  0.16146092263899975_dp,  0.23663401107846255_dp,&
            0.3449341441338246_dp,  0.5002090250697223_dp,  0.7218265976855273_dp,  1.036819021323394_dp,  1.4829173538285452_dp,  2.113006031585928_dp,&
            3.002118887270464_dp,  4.25973634734891_dp,  6.055362139002693_dp,  8.68620971747294_dp,  12.84476212055696_dp,  7.634110961434247e-05_dp,&
            0.00018622528846790902_dp,  0.00031969127466536623_dp,  0.0004991918529001643_dp,  0.0007602900708133781_dp,  0.0011555100382159716_dp,  0.0017584338105163752_dp,&
            0.0026721061529931525_dp,  0.004043591457129711_dp,  0.00608455281208252_dp,  0.009099150682157118_dp,  0.013522328657324851_dp,  0.01997274372862434_dp,&
            0.029325909214924503_dp,  0.04281497070968849_dp,  0.06216914194983893_dp,  0.08980351557167271_dp,  0.12907941549114965_dp,  0.184663246594884_dp,&
            0.2630278258762335_dp,  0.3731736346682672_dp,  0.5277260380561264_dp,  0.7447670823860634_dp,  1.051327938626868_dp,  1.4912362410625788_dp,&
            2.1465432019415753_dp,  3.214074319529515_dp,  5.461496007193266_dp]
       aw%aw_erange_matrix(:, 12) = [2.3779414127343956e-05_dp,  0.00012846422005240743_dp,  0.0003309928831073909_dp,  0.0006610841016675125_dp,  0.001174111517851935_dp,&
            0.001964735642046167_dp,  0.0031839437046973372_dp,  0.005062138452978201_dp,  0.007943587365676456_dp,  0.012337793711020667_dp,  0.01899408520862755_dp,&
            0.02900819673889348_dp,  0.0439733623874622_dp,  0.06619336766731646_dp,  0.09898159274083276_dp,  0.1470789732012216_dp,  0.2172359337388872_dp,&
            0.31902010927146557_dp,  0.465935489961852_dp,  0.676974407399601_dp,  0.9787827564627841_dp,  1.4087293747222094_dp,  2.0194090526711066_dp,&
            2.8856949554637783_dp,  4.117081205191889_dp,  5.883249820883165_dp,  8.481580484406694_dp,  12.603865500485162_dp,  6.137647450224398e-05_dp,&
            0.00015022918522230813_dp,  0.00025961689799477495_dp,  0.00040956404897364297_dp,  0.0006319903102840593_dp,  0.0009740518614949558_dp,  0.0015022032585384295_dp,&
            0.002310722155775527_dp,  0.0035358702066416604_dp,  0.005375683414999454_dp,  0.008117044767788793_dp,  0.012173222749539778_dp,  0.018136165890969762_dp,&
            0.026849214184554587_dp,  0.03950783752538176_dp,  0.057798731689314514_dp,  0.08409142368704033_dp,  0.1217021378276602_dp,  0.17525857508663245_dp,&
            0.2512102337617962_dp,  0.3585618870750593_dp,  0.5099851018896473_dp,  0.7236623896941952_dp,  1.0267809590048702_dp,  1.4633527890610423_dp,&
            2.1155495825866195_dp,  3.1799713652899073_dp,  5.422242322222096_dp]
       aw%aw_erange_matrix(:, 13) = [1.996975719114667e-05_dp,  0.00010803766024790043_dp,  0.00027914946672295566_dp,  0.0005600685201304379_dp,  0.0010010455833094946_dp,&
            0.0016883469263468778_dp,  0.0027600190175514337_dp,  0.004427777354772284_dp,  0.0070100858816421925_dp,  0.010981630036803518_dp,  0.017045370162921285_dp,&
            0.026235975775831418_dp,  0.04006720477930825_dp,  0.060740772707250645_dp,  0.09144094582526643_dp,  0.13674817658247623_dp,  0.20321752055734807_dp,&
            0.3001847449852204_dp,  0.44088738725627163_dp,  0.6440243823419953_dp,  0.9359372859467849_dp,  1.3537064463445627_dp,  1.9496922567626198_dp,&
            2.798633041764183_dp,  4.010025863301125_dp,  5.753689968035089_dp,  8.327144493740365_dp,  12.421679016019592_dp,  5.156001410295644e-05_dp,&
            0.00012656590579908283_dp,  0.00021996531665695752_dp,  0.00035003655094498806_dp,  0.0005460402965281552_dp,  0.0008512186113124776_dp,  0.0013268718440107125_dp,&
            0.0020609116400843674_dp,  0.0031816393309671174_dp,  0.004876914587104125_dp,  0.007420595371219653_dp,  0.011209484450019506_dp,  0.016815094297439877_dp,&
            0.02505596575962014_dp,  0.037098296020097635_dp,  0.05459544297770877_dp,  0.07988080804958117_dp,  0.11623432159124078_dp,  0.16825175472580994_dp,&
            0.2423620424395829_dp,  0.3475704605516449_dp,  0.4965817576093905_dp,  0.7076548141522077_dp,  1.0080986813480735_dp,  1.4420727665642223_dp,&
            2.09185113214404_dp,  3.1538737345296277_dp,  5.392217904025483_dp]
       aw%aw_erange_matrix(:, 14) = [1.5245472100141073e-05_dp,  8.267520676446116e-05_dp,  0.0002146253100574728_dp,  0.00043387740900171205_dp,  0.0007836944027554964_dp,&
            0.0013387930663292789_dp,  0.0022193817580203824_dp,  0.0036113909961874636_dp,  0.005797519906650395_dp,  0.009203847862920226_dp,  0.01446808989009802_dp,&
            0.022538313347044602_dp,  0.034814713569521594_dp,  0.05335209638799308_dp,  0.0811475703355085_dp,  0.12254732266532327_dp,  0.18381921201396537_dp,&
            0.27395570440350625_dp,  0.40579685475434923_dp,  0.5976003453364507_dp,  0.875245772613848_dp,  1.2753705825844541_dp,  1.8499693672606738_dp,&
            2.6735624079289133_dp,  3.855638524201632_dp,  5.566220151857392_dp,  8.103052167483417_dp,  12.156727650422702_dp,  3.9383299458817714e-05_dp,&
            9.713918675705757e-05_dp,  0.000170421510611186_dp,  0.0002751089399893439_dp,  0.0004367526322332042_dp,  0.0006931539398019088_dp,  0.0010985049239842675_dp,&
            0.0017318671480729233_dp,  0.0027102869498213354_dp,  0.004207039027136905_dp,  0.006477134639756053_dp,  0.009893360814081634_dp,  0.014997191697894255_dp,&
            0.02257042180249529_dp,  0.03373549826844379_dp,  0.05009547634855471_dp,  0.07392862462849442_dp,  0.10845863055245018_dp,  0.15823054590725322_dp,&
            0.22963859190692054_dp,  0.33168416368000975_dp,  0.47711716376192037_dp,  0.6843078702870563_dp,  0.9807484854746213_dp,  1.4108257998358733_dp,&
            2.0569814817847933_dp,  3.1154409728311463_dp,  5.348031605731007_dp]
       aw%aw_erange_matrix(:, 15) = [1.1868158588050793e-05_dp,  6.451496656973e-05_dp,  0.00016828367389327262_dp,  0.0003428138578813575_dp,  0.0006257728657443166_dp,&
            0.001082546664145262_dp,  0.0018189122009131555_dp,  0.002999899699939122_dp,  0.004879035508307028_dp,  0.007842398392055541_dp,  0.012473504017331851_dp,&
            0.019647823262643315_dp,  0.030669526100067415_dp,  0.04746815571454119_dp,  0.07287994053111492_dp,  0.11104807754173227_dp,  0.16798957023910677_dp,&
            0.2523945702323838_dp,  0.37675011948627407_dp,  0.5589184164507482_dp,  0.8243607282457994_dp,  1.209308359311203_dp,  1.7654144169432378_dp,&
            2.566987933665634_dp,  3.7234969430883793_dp,  5.405141983044582_dp,  7.9098851591902335_dp,  11.927751732780631_dp,  3.067518035880976e-05_dp,&
            7.602736152767003e-05_dp,  0.00013466099350051709_dp,  0.0002205156203375216_dp,  0.00035609796830194946_dp,  0.0005747749473125263_dp,  0.0009249835582363322_dp,&
            0.0014785254567841132_dp,  0.0023430318256698933_dp,  0.003679400712048183_dp,  0.005726484701954953_dp,  0.008836295353538897_dp,  0.013524092658912494_dp,&
            0.02053931124436599_dp,  0.030965498069383788_dp,  0.04636050178159656_dp,  0.06895241489824191_dp,  0.1019128924749107_dp,  0.14973885102211165_dp,&
            0.2187896005944566_dp,  0.31805835198625043_dp,  0.4603305714250192_dp,  0.6640726964679041_dp,  0.9569407810619277_dp,  1.3835309605116226_dp,&
            2.026448697565121_dp,  3.0817527233540427_dp,  5.309325429687417_dp]
       aw%aw_erange_matrix(:, 16) = [8.994970387322457e-06_dp,  4.9038651219381535e-05_dp,  0.00012866021417541092_dp,  0.00026454506965524195_dp,  0.0004890218504779385_dp,&
            0.0008585083873106919_dp,  0.0014649004957659442_dp,  0.002453066714438582_dp,  0.00404818477662515_dp,  0.006597100721999991_dp,  0.010629669867507806_dp,&
            0.016948840913414677_dp,  0.026762034057250313_dp,  0.041871547214080776_dp,  0.06494887934178976_dp,  0.09992774926372437_dp,  0.15256426013426896_dp,&
            0.2312316469143259_dp,  0.3480438390880272_dp,  0.520441341810499_dp,  0.7734349455842019_dp,  1.1428133579373827_dp,  1.6798513048817534_dp,&
            2.4586158447712725_dp,  3.5885384348523433_dp,  5.240004673003719_dp,  7.7112226946250795_dp,  11.691667476075265_dp,  2.3263971840934205e-05_dp,&
            5.799733961007195e-05_dp,  0.00010391874644151258_dp,  0.0001730995125618299_dp,  0.0002850761818642179_dp,  0.00046892999174830916_dp,  0.0007675526539416234_dp,&
            0.0012456339379473573_dp,  0.0020014080059075953_dp,  0.0031832610399331806_dp,  0.005013553854404307_dp,  0.007822909693482165_dp,  0.012099368861470802_dp,&
            0.018558470273632704_dp,  0.02824263531163439_dp,  0.04266142900862469_dp,  0.06398871559990858_dp,  0.09533906369189366_dp,  0.14115536539598936_dp,&
            0.20775581572804178_dp,  0.3041200099484807_dp,  0.4430661693538553_dp,  0.6431592914657807_dp,  0.9322296232686789_dp,  1.355102086814327_dp,&
            1.9945706936958254_dp,  3.046542150566686_dp,  5.268893998445498_dp]
       aw%aw_erange_matrix(:, 17) = [6.8673479673260504e-06_dp,  3.755590753670896e-05_dp,  9.915139981458229e-05_dp,  0.00020590986120619216_dp,  0.00038570411969655687_dp,&
            0.0006874225835250557_dp,  0.0011912995282688835_dp,  0.0020251938479038355_dp,  0.0033901629916074914_dp,  0.005599342672133319_dp,  0.009136035642934201_dp,&
            0.0147397005849582_dp,  0.023532325447503034_dp,  0.03720291582111918_dp,  0.05827512485526659_dp,  0.09049318420974399_dp,  0.13937535368635193_dp,&
            0.21300368025384273_dp,  0.3231464746438852_dp,  0.4868502227776408_dp,  0.7287007191729763_dp,  1.0840642579491595_dp,  1.6038486600019533_dp,&
            2.3618790984089095_dp,  3.467539083950752_dp,  5.09138219825452_dp,  7.53185924341038_dp,  11.47798005469671_dp,  1.7773461657358013e-05_dp,&
            4.458780785431919e-05_dp,  8.088403265892346e-05_dp,  0.00013715743270713957_dp,  0.00023041539243485957_dp,  0.00038613012816423253_dp,  0.0006425163485643913_dp,&
            0.001058147634119284_dp,  0.0017230321030210073_dp,  0.002774473725192771_dp,  0.004420090523231148_dp,  0.006971221027705693_dp,  0.010891151319944746_dp,&
            0.01686432115872218_dp,  0.025895072083461815_dp,  0.03944781877116618_dp,  0.059645137430023996_dp,  0.0895468281752171_dp,  0.13354285706817015_dp,&
            0.1979094322641605_dp,  0.291608890942118_dp,  0.42748519623553954_dp,  0.6241916121496675_dp,  0.9097205132707596_dp,  1.3291156770565788_dp,&
            1.965360170512719_dp,  3.0142420912237333_dp,  5.231825921337635_dp]
       aw%aw_erange_matrix(:, 18) = [5.34722652436438e-06_dp,  2.9335345788005576e-05_dp,  7.794420445994551e-05_dp,  0.0001635102977617264_dp,  0.0003103380666014091_dp,&
            0.0005612614423847721_dp,  0.0009871305012123896_dp,  0.0017020454525922442_dp,  0.002887383643586052_dp,  0.004828517011721044_dp,  0.007970062065688042_dp,&
            0.012998268728170836_dp,  0.02096295289681554_dp,  0.03345668031415268_dp,  0.052876323139595754_dp,  0.08280245241436206_dp,  0.1285464334202786_dp,&
            0.19793519202552176_dp,  0.30243198793781245_dp,  0.4587328138519049_dp,  0.6910421452320579_dp,  1.0343433822030452_dp,  1.5392071335504423_dp,&
            2.279230497059965_dp,  3.3637429883435237_dp,  4.96344347806973_dp,  7.377008968134632_dp,  11.293074733218035_dp,  1.3848866712786266e-05_dp,&
            3.496420645730977e-05_dp,  6.422528074399328e-05_dp,  0.00011085156828305275_dp,  0.00018979249819623976_dp,  0.0003236153897281577_dp,  0.0005467478277421638_dp,&
            0.000912713310709869_dp,  0.001504624320381209_dp,  0.0024504009658675406_dp,  0.003945073258731463_dp,  0.006283381657271798_dp,  0.009907137885517564_dp,&
            0.015473588963650402_dp,  0.023953503710281564_dp,  0.03677113226590553_dp,  0.056002962059688105_dp,  0.08465896050001478_dp,  0.1270800799647972_dp,&
            0.18950233376937137_dp,  0.2808690547904898_dp,  0.4140431391215438_dp,  0.6077531842367035_dp,  0.8901351666158656_dp,  1.3064316369806335_dp,&
            1.9398043844798791_dp,  2.9859550412644587_dp,  5.199382405454911_dp]
       aw%aw_erange_matrix(:, 19) = [3.991479954961762e-06_dp,  2.198720501971829e-05_dp,  5.890526173006801e-05_dp,  0.00012518031616235489_dp,  0.00024153524567753135_dp,&
            0.00044471016635781095_dp,  0.0007960973675010948_dp,  0.0013958401833713643_dp,  0.0024051630337238337_dp,  0.004080730226383008_dp,  0.0068267899198047955_dp,&
            0.011273593700680132_dp,  0.018394388130399692_dp,  0.029678633788751282_dp,  0.04738664952634921_dp,  0.07492145227987683_dp,  0.11736834467836472_dp,&
            0.18227349630314565_dp,  0.2807619982420454_dp,  0.42913828874392707_dp,  0.65117720616762_dp,  0.9814263687158781_dp,  1.4700673778736164_dp,&
            2.190427750165236_dp,  3.2517634440541547_dp,  4.824930255638227_dp,  7.208868255000471_dp,  11.091834807498854_dp,  1.0346862342302347e-05_dp,&
            2.6338249637987535e-05_dp,  4.916412854301245e-05_dp,  8.674834277534896e-05_dp,  0.00015194758780428323_dp,  0.00026440532385478294_dp,  0.00045469618333409404_dp,&
            0.0007711009718467_dp,  0.0012894728448137978_dp,  0.002127757911813279_dp,  0.0034674947327675617_dp,  0.005585492420042323_dp,  0.008900179331682422_dp,&
            0.014038952538275588_dp,  0.02193542166778625_dp,  0.033968992230147974_dp,  0.05216418521519709_dp,  0.07947410573244414_dp,  0.12018284604662782_dp,&
            0.18047837042907597_dp,  0.26927867836352665_dp,  0.3994632895161757_dp,  0.5898412436801937_dp,  0.8687080586761889_dp,  1.2815327235597582_dp,&
            1.91168824489995_dp,  2.954800144592791_dp,  5.163666162590388_dp]
       aw%aw_erange_matrix(:, 20) = [2.9287097241919463e-06_dp,  1.6211294489452685e-05_dp,  4.3860626639419625e-05_dp,  9.463310124321836e-05_dp,  0.00018604994609815954_dp,&
            0.0003493943765511619_dp,  0.0006375682448360294_dp,  0.0011380795201948043_dp,  0.001993709375951546_dp,  0.00343456904920158_dp,  0.005827197217933671_dp,&
            0.009749051421957427_dp,  0.016100560249175083_dp,  0.026272317263678777_dp,  0.042392659447799845_dp,  0.06769166630053426_dp,  0.10703277867836304_dp,&
            0.16768452950177593_dp,  0.26043499640668427_dp,  0.4011952042193507_dp,  0.6133044483273886_dp,  0.9308639628203111_dp,  1.4036513854446626_dp,&
            2.1047075037398466_dp,  3.143200044406983_dp,  4.6901364769963045_dp,  7.044731962208944_dp,  10.894906305627384_dp,  7.599945789593838e-06_dp,&
            1.9535340209099297e-05_dp,  3.7160656136204285e-05_dp,  6.722566694134913e-05_dp,  0.00012069652213103665_dp,  0.00021459840053591846_dp,  0.0003759981286012543_dp,&
            0.0006483001768113125_dp,  0.0011005125730724488_dp,  0.0018410833738741884_dp,  0.0030385893339244744_dp,  0.004952462825414513_dp,  0.007978278215654642_dp,&
            0.012714011523554197_dp,  0.02005632833363888_dp,  0.03133963591517064_dp,  0.04853577656951428_dp,  0.07453950556949848_dp,  0.11357558363229604_dp,&
            0.17178036896999477_dp,  0.2580420507206907_dp,  0.3852520170265286_dp,  0.5722961014253114_dp,  0.847629061135607_dp,  1.2569519792288701_dp,&
            1.8838624046566568_dp,  2.923931086168498_dp,  5.1282957309537895_dp]
       aw%aw_erange_matrix(:, 21) = [2.187887975241367e-06_dp,  1.217302105778866e-05_dp,  3.3280277384650595e-05_dp,  7.294651585196924e-05_dp,  0.0001461477506435841_dp,&
            0.0002798259055981046_dp,  0.0005201025046636251_dp,  0.0009442991970046586_dp,  0.0016801646722205163_dp,  0.0029359404461932336_dp,  0.005046805926292481_dp,&
            0.008545920293678393_dp,  0.014272096616783494_dp,  0.023531600046055763_dp,  0.038339327744621156_dp,  0.061775614958965434_dp,  0.09851040024837446_dp,&
            0.15556831587766237_dp,  0.243439200558_dp,  0.3776832590482275_dp,  0.5812481524315448_dp,  0.8878298248851654_dp,  1.3468344739190594_dp,&
            2.031033514764075_dp,  3.0495035108172392_dp,  4.573381799063933_dp,  6.902137854735053_dp,  10.723425645457905_dp,  5.683864101635673e-06_dp,&
            1.4761618602067145e-05_dp,  2.863922084439926e-05_dp,  5.31200028912619e-05_dp,  9.765726111159933e-05_dp,  0.00017719091014345152_dp,  0.00031593721698955905_dp,&
            0.0005532594185059319_dp,  0.000952423486460829_dp,  0.001613835283450162_dp,  0.002695001718544619_dp,  0.004440391442639413_dp,  0.007225736946277607_dp,&
            0.01162326005144731_dp,  0.018497020133015078_dp,  0.02914136949940742_dp,  0.04548081139069081_dp,  0.07035707490413387_dp,  0.10794019701910468_dp,&
            0.1643177452299961_dp,  0.24834763708314758_dp,  0.37292767817567907_dp,  0.5570086406904123_dp,  0.829186245882319_dp,  1.2353723738001468_dp,&
            1.8593753882752646_dp,  2.896735710900212_dp,  5.0971507815821235_dp]
       aw%aw_erange_matrix(:, 22) = [1.6070031867276192e-06_dp,  8.996223208889756e-06_dp,  2.4903330256736718e-05_dp,  5.5597271413213874e-05_dp,  0.00011378123225419507_dp,&
            0.00022251943620373886_dp,  0.0004218452833351526_dp,  0.0007798397462569304_dp,  0.001410464737003683_dp,  0.0025017001910060442_dp,  0.0043594012767892145_dp,&
            0.007474944527557248_dp,  0.012628553354571935_dp,  0.021045689348535686_dp,  0.03463175327773065_dp,  0.05632151430959264_dp,  0.09059550579868533_dp,&
            0.1442379871996754_dp,  0.2274428250359289_dp,  0.35541953739182136_dp,  0.5507212109659467_dp,  0.8466317459799636_dp,  1.2921752687688919_dp,&
            1.9598406408284006_dp,  2.9586009577769_dp,  4.459717588193107_dp,  6.762921787948558_dp,  10.555632767123296_dp,  4.180340751600858e-06_dp,&
            1.099130280826382e-05_dp,  2.1822771688601605e-05_dp,  4.1621175974422805e-05_dp,  7.848461568768336e-05_dp,  0.00014548449813908576_dp,  0.00026422505480514413_dp,&
            0.0004703011644186479_dp,  0.0008215666081596167_dp,  0.0014107803843716492_dp,  0.002384836263299515_dp,  0.0039737397649737246_dp,  0.006533894710819361_dp,&
            0.010612242095881175_dp,  0.017040581368346817_dp,  0.027073311442312262_dp,  0.04258730399078407_dp,  0.06637038094501468_dp,  0.10253618298821561_dp,&
            0.15712087871604702_dp,  0.23894860519216926_dp,  0.36091957483420634_dp,  0.5420459855749467_dp,  0.8110632955302136_dp,  1.2140975460551333_dp,&
            1.8351776848641823_dp,  2.8698308488017825_dp,  5.066349620899207_dp]
       aw%aw_erange_matrix(:, 23) = [9.917609137977847e-07_dp,  5.61571565668021e-06_dp,  1.5905578931798044e-05_dp,  3.668045117366575e-05_dp,  7.779911321491552e-05_dp,&
            0.0001574703243782781_dp,  0.0003080352227521647_dp,  0.0005857339949829306_dp,  0.0010866136806263423_dp,  0.0019719871993643633_dp,  0.0035086606810778888_dp,&
            0.006131766091278258_dp,  0.010541856280946737_dp,  0.017853461412239566_dp,  0.029820286970471933_dp,  0.049173663410606715_dp,  0.08012714696416229_dp,&
            0.1291233103698595_dp,  0.20593171918618178_dp,  0.32525471926745003_dp,  0.5090693312937355_dp,  0.7900513389150534_dp,  1.216653152188614_dp,&
            1.8609312710223198_dp,  2.831687087034342_dp,  4.300351181611193_dp,  6.567047622277454_dp,  10.318912411219488_dp,  2.5862090723245625e-06_dp,&
            6.95626109831625e-06_dp,  1.4392071614748994e-05_dp,  2.874925884264212e-05_dp,  5.642970465568392e-05_dp,  0.00010814505912642593_dp,  0.0002021008939835649_dp,&
            0.0003688922773676731_dp,  0.0006591003182679095_dp,  0.0011550942822986209_dp,  0.0019891964272602783_dp,  0.0033713558748037233_dp,  0.0056309115068987_dp,&
            0.00927906547653282_dp,  0.015101574633332329_dp,  0.024295245067122773_dp,  0.03866754865528627_dp,  0.060926803424727255_dp,  0.09510216196058996_dp,&
            0.14715087305959354_dp,  0.22584200604915872_dp,  0.34407208794479094_dp,  0.5209358458550991_dp,  0.7853686557167923_dp,  1.1838124863793347_dp,&
            1.8006332661854303_dp,  2.8313699702065134_dp,  5.0223450191392205_dp]
       aw%aw_erange_matrix(:, 24) = [6.243870742444705e-07_dp,  3.5837547513507616e-06_dp,  1.0424530634626105e-05_dp,  2.4910967538724005e-05_dp,  5.48216536959138e-05_dp,&
            0.0001148065303999889_dp,  0.000231490488679646_dp,  0.00045215860524583124_dp,  0.0008590837338469464_dp,  0.0015927682432411259_dp,  0.0028891299956185297_dp,&
            0.005138253059658019_dp,  0.008976135083193377_dp,  0.015426433468499803_dp,  0.026117293389759495_dp,  0.043609983170991296_dp,  0.0718927383781267_dp,&
            0.11711702885780055_dp,  0.18868749571084845_dp,  0.3008660121451099_dp,  0.47512403493520045_dp,  0.7435972663838382_dp,  1.154222692676153_dp,&
            1.7786576652841575_dp,  2.7255320837393326_dp,  4.166412460588324_dp,  6.401776404663582_dp,  10.11856522638141_dp,  1.6329106731735064e-06_dp,&
            4.511304110662316e-06_dp,  9.770509823422929e-06_dp,  2.0454024246766166e-05_dp,  4.172679855059072e-05_dp,  8.25404415798886e-05_dp,  0.00015847711044459983_dp,&
            0.00029619555479324744_dp,  0.0005404696680087612_dp,  0.000965266351597196_dp,  0.0016909815488903273_dp,  0.0029109600865339395_dp,  0.004931885217935304_dp,&
            0.008234716202420396_dp,  0.01356583086594651_dp,  0.022072242618732946_dp,  0.035500720158786525_dp,  0.05648910800327906_dp,  0.08899040639475064_dp,&
            0.13888891315246274_dp,  0.21489980129365738_dp,  0.32990928766337146_dp,  0.5030773832314516_dp,  0.7635106326257592_dp,  1.157931116384902_dp,&
            1.771014718506682_dp,  2.798340277794168_dp,  4.984574132350961_dp]
       aw%aw_erange_matrix(:, 25) = [4.0370316213559476e-07_dp,  2.3541711422913717e-06_dp,  7.057573612279361e-06_dp,  1.751162431119145e-05_dp,  3.998197703771905e-05_dp,&
            8.651758104084795e-05_dp,  0.00017950101241220228_dp,  0.00035945947508640393_dp,  0.0006981140824218166_dp,  0.001319810424458466_dp,  0.0024361923309759396_dp,&
            0.004401544013890147_dp,  0.007800025204094134_dp,  0.013581611201940205_dp,  0.023271704375636543_dp,  0.03929119474223725_dp,  0.06544075124204898_dp,&
            0.10762751156023793_dp,  0.17494730054790258_dp,  0.2812860986936203_dp,  0.44768004577537496_dp,  0.705795087398322_dp,  1.1031142167130359_dp,&
            1.7109362858917831_dp,  2.6377277178978358_dp,  4.055162275207615_dp,  6.264028054745519_dp,  9.951137323627291_dp,  1.0593262130160056e-06_dp,&
            3.0185707613834373e-06_dp,  6.8658831794879865e-06_dp,  1.5046778554922312e-05_dp,  3.182893187575826e-05_dp,  6.484710011709137e-05_dp,  0.00012766407757989652_dp,&
            0.000243860437909589_dp,  0.00045361234593234906_dp,  0.0008241579727338823_dp,  0.001466238321223863_dp,  0.002559617356616681_dp,  0.004392266266516603_dp,&
            0.0074199303035661236_dp,  0.012355848225194005_dp,  0.02030472924789548_dp,  0.0329612503015901_dp,  0.05290210344877982_dp,  0.08401329609895744_dp,&
            0.13211363792116823_dp,  0.20586780977743016_dp,  0.3181478896142563_dp,  0.48816468770340044_dp,  0.7451687687530838_dp,  1.1361253599700833_dp,&
            1.7459877749472628_dp,  2.7703909434425067_dp,  4.952626774509824_dp]
       aw%aw_erange_matrix(:, 26) = [2.0330900665342417e-07_dp,  1.2259721995909777e-06_dp,  3.9000493864165235e-06_dp,  1.0346548674455454e-05_dp,  2.5104573254227977e-05_dp,&
            5.722321695341548e-05_dp,  0.0001240915794841191_dp,  0.00025812480274594484_dp,  0.0005181574776824829_dp,  0.0010084979886612006_dp,  0.0019102647242876139_dp,&
            0.003532146072512572_dp,  0.0063914997207670885_dp,  0.011342304908756772_dp,  0.019774721642387954_dp,  0.03392304253932974_dp,  0.05733626485492922_dp,&
            0.0955906794116426_dp,  0.15736013123361753_dp,  0.2560121769552906_dp,  0.4119766499624099_dp,  0.6562580036536524_dp,  1.0356912360288544_dp,&
            1.6210528978513052_dp,  2.5205569278227222_dp,  3.906010837038453_dp,  6.07864411392436_dp,  9.725147433824423_dp,  5.372960626792664e-07_dp,&
            1.6313777234005112e-06_dp,  4.053582188809558e-06_dp,  9.562645330678188e-06_dp,  2.140029237042944e-05_dp,  4.5628018410594356e-05_dp,  9.332815320834305e-05_dp,&
            0.00018423715674012751_dp,  0.0003527071740962291_dp,  0.0006573395283833412_dp,  0.0011963286460791483_dp,  0.0021315852595548933_dp,  0.003726211411404738_dp,&
            0.006402085885163769_dp,  0.010827471574921069_dp,  0.018049062427338197_dp,  0.029689317308315376_dp,  0.048239120452325215_dp,  0.0774890829270111_dp,&
            0.12316278045588097_dp,  0.1938483490800667_dp,  0.30239008881049667_dp,  0.4680611192958136_dp,  0.7203071572907666_dp,  1.1064348627930074_dp,&
            1.7118002420359797_dp,  2.732149945273555_dp,  4.908937928661797_dp]
       aw%aw_erange_matrix(:, 27) = [1.0453117858944355e-07_dp,  6.608617412970565e-07_dp,  2.263239779628554e-06_dp,  6.456580799232289e-06_dp,  1.6648773232985477e-05_dp,&
            3.988538010811069e-05_dp,  9.013431830037881e-05_dp,  0.00019412595183632066_dp,  0.00040148594866505004_dp,  0.000801953768960446_dp,  0.0015541047607406542_dp,&
            0.002932474750123789_dp,  0.0054037297272779_dp,  0.0097481213186169_dp,  0.01725076571642057_dp,  0.029999477630049398_dp,  0.05134367891607686_dp,&
            0.08659470336954253_dp,  0.14408515309896788_dp,  0.2367592019652705_dp,  0.3845463783654337_dp,  0.6178989765753549_dp,  0.9831034515051852_dp,&
            1.5504849076462728_dp,  2.4280266299570483_dp,  3.7876319869184405_dp,  5.930901453777237_dp,  9.544473553973464_dp,  2.7911359387760054e-07_dp,&
            9.225699169205882e-07_dp,  2.525970869540034e-06_dp,  6.400448408750558e-06_dp,  1.5105857774384374e-05_dp,  3.360200325419909e-05_dp,  7.118848251951222e-05_dp,&
            0.00014478894137667428_dp,  0.0002844248664965114_dp,  0.0005421792095475284_dp,  0.001006646872918385_dp,  0.0018259034247645597_dp,  0.003243545770723624_dp,&
            0.0056545862514208655_dp,  0.009691209964763319_dp,  0.016353052767086353_dp,  0.027203315594152448_dp,  0.044661579803193296_dp,  0.07243798977464_dp,&
            0.11617410461859103_dp,  0.18438950337219645_dp,  0.2898984327270837_dp,  0.45201785956261764_dp,  0.7003495143727266_dp,  1.0824839695902242_dp,&
            1.6841236033891487_dp,  2.701136203287264_dp,  4.87352184807004_dp]
       aw%aw_erange_matrix(:, 28) = [2.883608355013751e-08_dp,  2.1533521290478e-07_dp,  8.922309745530507e-07_dp,  2.9625449220607536e-06_dp,  8.567451943237967e-06_dp,&
            2.243432347635065e-05_dp,  5.4443793898855104e-05_dp,  0.00012434900771925045_dp,  0.0002702071667839422_dp,  0.0005630773934606482_dp,  0.001132097168889575_dp,&
            0.0022064607569192898_dp,  0.004184511435918261_dp,  0.007745737975074526_dp,  0.014029823575166972_dp,  0.024919304936911416_dp,  0.04348066531140458_dp,&
            0.07464540686704813_dp,  0.12625156513534663_dp,  0.2106228965724005_dp,  0.34694729093744464_dp,  0.5648480693819434_dp,  0.9097754411882123_dp,&
            1.4513508415022998_dp,  2.297177622667234_dp,  3.619275978541861_dp,  5.719806682040045_dp,  9.28540894538464_dp,  8.026935386394373e-08_dp,&
            3.431660125547769e-07_dp,  1.1507249938591508e-06_dp,  3.3255150480524145e-06_dp,  8.631544858985621e-06_dp,  2.0673889190163757e-05_dp,  4.650257994599218e-05_dp,&
            9.941521329490123e-05_dp,  0.0002037413884778986_dp,  0.00040284592225487954_dp,  0.0007722776557802875_dp,  0.0014410293829550617_dp,  0.0026254207291881958_dp,&
            0.0046823958218659345_dp,  0.008192369379080482_dp,  0.014086602667824237_dp,  0.023841088169455484_dp,  0.039769020002557186_dp,  0.06545851893082533_dp,&
            0.1064240118543133_dp,  0.1710745747479101_dp,  0.2721680114354786_dp,  0.42907347105223687_dp,  0.671615155080402_dp,  1.0478076752969256_dp,&
            1.6438903234547084_dp,  2.6559588426369642_dp,  4.821957836337722_dp]
       aw%aw_erange_matrix(:, 29) = [7.2246193648183315e-09_dp,  7.947805205434051e-08_dp,  4.2064242178143404e-07_dp,  1.6235298675242196e-06_dp,  5.1991703168705275e-06_dp,&
            1.4668546651111404e-05_dp,  3.7705424076918065e-05_dp,  9.01796708136201e-05_dp,  0.0002035415234868656_dp,  0.0004379390624442907_dp,  0.0009049701100068852_dp,&
            0.0018063291005268592_dp,  0.0034982504743316482_dp,  0.006597202794116675_dp,  0.012150671665557649_dp,  0.021909404675932484_dp,  0.03875606413936719_dp,&
            0.06737255773930659_dp,  0.11526829896699824_dp,  0.1943501837935314_dp,  0.3233020082512726_dp,  0.5311764928093146_dp,  0.8628395371729328_dp,&
            1.3874103242942566_dp,  2.212207252365476_dp,  3.5093118591573305_dp,  5.581272308767191_dp,  9.114780312141413_dp,  2.3095697901478273e-08_dp,&
            1.5174284298730647e-07_dp,  6.211817019789001e-07_dp,  2.0193446034719265e-06_dp,  5.684986670359784e-06_dp,  1.4468629362219993e-05_dp,  3.413251437719442e-05_dp,&
            7.584802072328596e-05_dp,  0.00016053557840876553_dp,  0.0003262378591667722_dp,  0.0006404016188164239_dp,  0.0012199823490146536_dp,  0.0022638430447923645_dp,&
            0.0041042325427612_dp,  0.007287543243404729_dp,  0.012699528275383098_dp,  0.021757390677572788_dp,  0.03670161910421734_dp,  0.061035614898498174_dp,&
            0.10018369236083766_dp,  0.16247380492926894_dp,  0.26061721476056016_dp,  0.41400956019249524_dp,  0.6526199011491635_dp,  1.0247530046294373_dp,&
            1.617029280212683_dp,  2.625732716141634_dp,  4.78747600908045_dp]
    case(30)
       aw%energy_range(:) = [2906.0_dp, 3236.0_dp, 3810.0_dp, 4405.0_dp, 5400.0_dp, 6800.0_dp, 8400.0_dp,&
            10000.0_dp, 12000.0_dp, 15000.0_dp, 20000.0_dp, 28000.0_dp, 38000.0_dp, 50000.0_dp, 64000.0_dp, 84000.0_dp,&
            110000.0_dp, 160000.0_dp, 220000.0_dp, 370000.0_dp, 520000.0_dp, 700000.0_dp, 1100000.0_dp, 1800000.0_dp,&
            3300000.0_dp, 6000000.0_dp, 18000000.0_dp, 50000000.0_dp]
       aw%aw_erange_matrix(:, 1) = [0.00010719782131833758_dp,  0.0005720967766785331_dp,  0.001439669174001423_dp,  0.002769849121659816_dp,  0.004664000049778321_dp,&
            0.007283241780546586_dp,  0.010873000028774003_dp,  0.015792629918665993_dp,  0.02255038608921401_dp,  0.031847782299297675_dp,  0.04463988143266245_dp,&
            0.06221821662158963_dp,  0.08632310206096853_dp,  0.11929354905847961_dp,  0.16426574047985984_dp,  0.2254347024266385_dp,  0.30839852293765396_dp,&
            0.42061064113062563_dp,  0.5719741696129382_dp,  0.7756243966360713_dp,  1.0489645297600516_dp,  1.415051852321483_dp,  1.9044914330966105_dp,&
            2.558117126903484_dp,  3.4310117702125913_dp,  4.5990752550643395_dp,  6.171106184489477_dp,  8.314824292369494_dp,  11.326871889957468_dp,&
            15.91048796379593_dp,  0.0002759212187476099_dp,  0.0006589920695996667_dp,  0.001085937188673809_dp,  0.0015910122106733914_dp,  0.0022237295869651035_dp,&
            0.003055163774041633_dp,  0.004183672861951183_dp,  0.005739573533927359_dp,  0.007891868776677179_dp,  0.010860734164881057_dp,  0.014936866838559482_dp,&
            0.020507383044696917_dp,  0.02808894961545452_dp,  0.038370325842103176_dp,  0.052267550952526234_dp,  0.07099592893538823_dp,  0.09616415398590554_dp,&
            0.1298977096485147_dp,  0.17500152116499199_dp,  0.235176717264172_dp,  0.3153154828400979_dp,  0.4219165107665381_dp,  0.5637038218742615_dp,&
            0.7526245591453196_dp,  1.005629375577748_dp,  1.3482480324070423_dp,  1.8228149068796895_dp,  2.510939848106289_dp,  3.6132439344998795_dp,  5.9228966909439205_dp]
       aw%aw_erange_matrix(:, 2) = [9.76676051737233e-05_dp,  0.000521439839796676_dp,  0.0013131673080453233_dp,  0.0025293903535565823_dp,  0.0042661250059111795_dp,&
            0.0066765801036800295_dp,  0.009994783165295324_dp,  0.014564000812497294_dp,  0.020870492553693578_dp,  0.02958685422269625_dp,  0.04163138568046643_dp,&
            0.058249942771901375_dp,  0.08112692839971898_dp,  0.11253369957795559_dp,  0.15552549205786806_dp,  0.2142017013001525_dp,  0.29404913356355256_dp,&
            0.4023941070276559_dp,  0.5489978387767454_dp,  0.7468418586806641_dp,  1.0131691905482183_dp,  1.3708791149003934_dp,  1.8504330035916199_dp,&
            2.4925505474173923_dp,  3.3522463174174093_dp,  4.505410618668513_dp,  6.060890691327169_dp,  8.186478838726869_dp,  11.178786177384513_dp,&
            15.74038405881577_dp,  0.00025141352623809566_dp,  0.0006009323005964832_dp,  0.0009917485369505094_dp,  0.0014564055840763494_dp,  0.002042200574311218_dp,&
            0.0028172406584227633_dp,  0.003875825130607345_dp,  0.005342947480157929_dp,  0.00738116098048549_dp,  0.010203263291670922_dp,  0.014091611662835478_dp,&
            0.019423788992341998_dp,  0.026705494627716198_dp,  0.03661296179988794_dp,  0.05004819563708869_dp,  0.06821123613467164_dp,  0.09269486715309114_dp,&
            0.12560901090026905_dp,  0.16974491105273157_dp,  0.22879407342136257_dp,  0.30764598759629597_dp,  0.4128070021079015_dp,  0.5530226357541534_dp,&
            0.7402778334602197_dp,  0.9915767219176012_dp,  1.332510891997349_dp,  1.8054641559432192_dp,  2.49203146209568_dp,  3.5925935804922893_dp,  5.89893592679766_dp]
       aw%aw_erange_matrix(:, 3) = [8.47819458078811e-05_dp,  0.00045292247232938765_dp,  0.0011419488590203569_dp,  0.0022036034544386665_dp,  0.0037263081029974285_dp,&
            0.005851972143509195_dp,  0.008798246612654406_dp,  0.012885194918659285_dp,  0.018567320347579703_dp,  0.026475458982841895_dp,  0.03747475660944132_dp,&
            0.05274484615883288_dp,  0.07388876569982615_dp,  0.103079000554941_dp,  0.14325196624460337_dp,  0.19836605747691327_dp,  0.27374326191660053_dp,&
            0.3765207493043709_dp,  0.5162475872690495_dp,  0.7056742360429414_dp,  0.9618016216492792_dp,  1.3072886416997354_dp,  1.772376230882795_dp,&
            2.39760681930879_dp,  3.237886311434963_dp,  4.369085529708981_dp,  5.900122672480505_dp,  7.998902446784312_dp,  10.962002095892634_dp,&
            15.491026703179935_dp,  0.00021827426323305446_dp,  0.0005223668404015434_dp,  0.0008641224881204865_dp,  0.0012736436118541284_dp,  0.001795042273807457_dp,&
            0.0024921178119473955_dp,  0.00345325315447348_dp,  0.004795711504484412_dp,  0.006672726568911133_dp,  0.009286420295701826_dp,  0.012906984705977642_dp,&
            0.017897959998054307_dp,  0.02474875419566456_dp,  0.0341168643096291_dp,  0.046883167956700916_dp,  0.06422456826044022_dp,  0.08770953467687731_dp,&
            0.11942394754852313_dp,  0.16213754631449626_dp,  0.21952613567742027_dp,  0.2964736644552558_dp,  0.39949638395652903_dp,  0.5373706722452656_dp,&
            0.7221373998987525_dp,  0.9708812677144617_dp,  1.309288667276434_dp,  1.7798217912856973_dp,  2.464060270734846_dp,  3.5620371023687465_dp,  5.8635023001897295_dp]
       aw%aw_erange_matrix(:, 4) = [7.475415175359787e-05_dp,  0.0003995782940364109_dp,  0.0010085401523130205_dp,  0.0019494530926649103_dp,  0.003304487334168418_dp,&
            0.005206188391856581_dp,  0.007858547072014874_dp,  0.011562196417307553_dp,  0.01674502130839572_dp,  0.02400284151062447_dp,  0.03415615402045192_dp,&
            0.04832877588615427_dp,  0.06805486346691048_dp,  0.09542279350363236_dp,  0.13326739942725138_dp,  0.18542592486136436_dp,  0.25707802147355546_dp,&
            0.3551966628680496_dp,  0.48914575629830964_dp,  0.6714729217979144_dp,  0.9189652081581203_dp,  1.2540675821639358_dp,  1.706822983665265_dp,&
            2.3176124905704714_dp,  3.1412403932005213_dp,  4.253555048851896_dp,  5.763534315696107_dp,  7.839184077942836_dp,  10.777063260156016_dp,&
            15.27796168145106_dp,  0.00019248226458298544_dp,  0.0004611671410305988_dp,  0.0007645470771240003_dp,  0.0011307064674945877_dp,  0.001601097912084883_dp,&
            0.002235885143791768_dp,  0.003118439955104845_dp,  0.00435950977219228_dp,  0.006104510985566756_dp,  0.008546583020796826_dp,  0.011945581030119579_dp,&
            0.016652984461283625_dp,  0.02314408570724871_dp,  0.03206002850298206_dp,  0.044263120398424796_dp,  0.06090977430142275_dp,  0.08354673398267808_dp,&
            0.11423814381228614_dp,  0.15573394602932067_dp,  0.2116949419787466_dp,  0.28699873776286255_dp,  0.38816866941911854_dp,  0.5240067210209787_dp,&
            0.7066019007188943_dp,  0.9531097377337013_dp,  1.2893015561438181_dp,  1.7577121859945424_dp,  2.439914418694296_dp,  3.5356481389173076_dp,  5.832916244207001_dp]
       aw%aw_erange_matrix(:, 5) = [6.263685476987877e-05_dp,  0.0003350864107617077_dp,  0.0008471021406263553_dp,  0.0016414732496533968_dp,  0.0027923267352349805_dp,&
            0.004420069092229887_dp,  0.006710856247430801_dp,  0.009939856329159545_dp,  0.014500042255919464_dp,  0.02094127936473627_dp,  0.030025328730095918_dp,&
            0.042802276529506474_dp,  0.06071482338334902_dp,  0.08573906273600417_dp,  0.1205734666631485_dp,  0.168891746293846_dp,  0.23568035070007362_dp,&
            0.3276881055094081_dp,  0.4540246498488577_dp,  0.6269572963379791_dp,  0.8629754834557104_dp,  1.1842241235652444_dp,  1.6204657246854692_dp,&
            2.2118496417292866_dp,  3.0130302596964915_dp,  4.099816741357715_dp,  5.581265514191353_dp,  7.625526452879397_dp,  10.52915117245689_dp,&
            14.99185357007055_dp,  0.00016131234676709978_dp,  0.0003871323803883455_dp,  0.0006438639425224474_dp,  0.0009569827684469891_dp,  0.0013644640382973136_dp,&
            0.0019216628235682472_dp,  0.002705305661659533_dp,  0.003817552287654078_dp,  0.005393559577915718_dp,  0.007614615789728071_dp,  0.0107267793784036_dp,&
            0.01506524759897237_dp,  0.02108608211163133_dp,  0.029407976403737046_dp,  0.040867556214958384_dp,  0.05659266977268607_dp,  0.07809948510740791_dp,&
            0.10742117687653448_dp,  0.1472789549319935_dp,  0.20131104861420793_dp,  0.27438405593923587_dp,  0.3730286538370253_dp,  0.50607994537212_dp,&
            0.6856919919465865_dp,  0.9291183488974969_dp,  1.2622503775094018_dp,  1.7277292939880777_dp,  2.407128771305674_dp,  3.4998023815666013_dp,  5.791398263742947_dp]
       aw%aw_erange_matrix(:, 6) = [5.1260796624319686e-05_dp,  0.00027449998212479283_dp,  0.0006952559402364613_dp,  0.0013512581469068891_dp,  0.0023084695955214183_dp,&
            0.0036748571064951344_dp,  0.005618146035815099_dp,  0.008387084892047715_dp,  0.012338401496919359_dp,  0.01797419704821914_dp,  0.025994966614220735_dp,&
            0.03737346294738748_dp,  0.05345583818085436_dp,  0.0760988446106003_dp,  0.10785503448538585_dp,  0.15222193676863333_dp,  0.2139764359272559_dp,&
            0.29962253812775563_dp,  0.4179901608236006_dp,  0.5810358869798488_dp,  0.8049164836351872_dp,  1.11143825413834_dp,  1.5300435290071273_dp,&
            2.1006127473331047_dp,  2.8776209315806276_dp,  3.93682226153259_dp,  5.387354342884511_dp,  7.397531351751853_dp,  10.263919356557402_dp,&
            14.685102820756857_dp,  0.00013204473011131683_dp,  0.0003175248324260899_dp,  0.0005301211290638112_dp,  0.000792646090309341_dp,  0.0011394726365449568_dp,&
            0.0016209075495792747_dp,  0.0023066917116162182_dp,  0.0032900298122596973_dp,  0.004695415042196518_dp,  0.0066917090579473775_dp,  0.009510290459434545_dp,&
            0.013468802311717997_dp,  0.019002357781084556_dp,  0.02670497738030662_dp,  0.037384817040713114_dp,  0.05213776033026327_dp,  0.072445378072951_dp,&
            0.10030529745315488_dp,  0.13840505606579703_dp,  0.19035539860923475_dp,  0.2610076849435031_dp,  0.3568973694825793_dp,  0.48689310015691706_dp,&
            0.6632189222745253_dp,  0.9032371034193576_dp,  1.2329755271865048_dp,  1.695201274879912_dp,  2.3715025399546223_dp,  3.4608285109987857_dp,  5.746290251951392_dp]
       aw%aw_erange_matrix(:, 7) = [4.264827171158114e-05_dp,  0.00022859921129368268_dp,  0.0005800651655637003_dp,  0.0011306618266057897_dp,  0.001939659276932296_dp,&
            0.0031047255791507274_dp,  0.004778205945319036_dp,  0.007186731964418128_dp,  0.010656673525706605_dp,  0.01565005735652159_dp,  0.02281575734702256_dp,&
            0.03306097534037487_dp,  0.04764947470167637_dp,  0.0683354702649706_dp,  0.09754528626464402_dp,  0.13862301432535307_dp,  0.19616179200890152_dp,&
            0.27644949046799433_dp,  0.3880673249123362_dp,  0.5426935306072133_dp,  0.7561844283103211_dp,  1.0500376608384598_dp,  1.4534003880410857_dp,&
            2.005901250325418_dp,  2.7618425195091647_dp,  3.796918364528742_dp,  5.220333260426152_dp,  7.200552079960992_dp,  10.034172866297915_dp,&
            14.41882124837872_dp,  0.0001098833579123983_dp,  0.0002647437855875365_dp,  0.0004436472669265954_dp,  0.0006672101026873376_dp,  0.0009667895306084336_dp,&
            0.0013884135997529708_dp,  0.0019959070037335724_dp,  0.0028749526284055213_dp,  0.004141093118158889_dp,  0.005952643864305207_dp,  0.008528340123693306_dp,&
            0.01217052007837369_dp,  0.017295858304956554_dp,  0.024476476682491477_dp,  0.03449505017711748_dp,  0.04841859097574648_dp,  0.06769708781585487_dp,&
            0.09429528482789237_dp,  0.13086897935334732_dp,  0.1810021324793278_dp,  0.24952974948475865_dp,  0.3429885311920854_dp,  0.4702742651357602_dp,&
            0.6436716531183656_dp,  0.880640225748484_dp,  1.2073331130934455_dp,  1.6666371474517014_dp,  2.3401654531537157_dp,  3.4265253480645885_dp,  5.706615197039722_dp]
       aw%aw_erange_matrix(:, 8) = [3.6638445936143444e-05_dp,  0.00019654898584533858_dp,  0.0004995365456010348_dp,  0.0009761629469610179_dp,  0.001680692380508944_dp,&
            0.0027030234477272513_dp,  0.004183824314963024_dp,  0.006332898868769434_dp,  0.009453508422633638_dp,  0.013977114132271049_dp,  0.020513053005428484_dp,&
            0.02991804681839935_dp,  0.043392038467264346_dp,  0.06260934488416951_dp,  0.089897358898568_dp,  0.12847922117893293_dp,  0.18280238941596594_dp,&
            0.25898237331386087_dp,  0.36540085871444006_dp,  0.5135113514430796_dp,  0.718925939367594_dp,  1.0028894169830442_dp,  1.3943048957706679_dp,&
            1.9325902027814568_dp,  2.6718997828791795_dp,  3.6878719059499683_dp,  5.089761008341865_dp,  7.046156150294659_dp,  9.853694451194254_dp,&
            14.209263474504755_dp,  9.441684068693651e-05_dp,  0.00022786014262271372_dp,  0.00038307351711976584_dp,  0.0005790223101422384_dp,  0.0008447661561374952_dp,&
            0.0012230417856923205_dp,  0.0017731334822002557_dp,  0.002574990545123123_dp,  0.003737321923840727_dp,  0.005410302324390409_dp,  0.007802786243657_dp,&
            0.011205034742082835_dp,  0.016019061597614896_dp,  0.02279945700439318_dp,  0.032308353668577205_dp,  0.04558931015753348_dp,  0.06406643328220389_dp,&
            0.08967722561429306_dp,  0.1250507942222136_dp,  0.17374800817672645_dp,  0.24058882427995695_dp,  0.3321088613980677_dp,  0.4572237934265981_dp,&
            0.6282658627647734_dp,  0.8627730246821379_dp,  1.187001575527606_dp,  1.6439399241381616_dp,  2.3152296444802993_dp,  3.3992160109205325_dp,  5.6750508010285134_dp]
       aw%aw_erange_matrix(:, 9) = [3.125352048295285e-05_dp,  0.00016781351172929327_dp,  0.00042725280780375096_dp,  0.0008372368297287864_dp,  0.0014472482996344947_dp,&
            0.0023397061489668085_dp,  0.00364397706459525_dp,  0.005553554215846271_dp,  0.008349288769114196_dp,  0.012432933353972381_dp,  0.018375235671074245_dp,&
            0.0269833920064158_dp,  0.03939439104594155_dp,  0.057203310896803944_dp,  0.08263890933961444_dp,  0.118803108858221_dp,  0.16999660677619183_dp,&
            0.24216036024367976_dp,  0.3434728927615001_dp,  0.4851575786991668_dp,  0.6825748514741242_dp,  0.956707159367906_dp,  1.3362022902301018_dp,&
            1.8602550523674573_dp,  2.582860494442485_dp,  3.5795923551771076_dp,  4.959752013207221_dp,  6.892057403111867_dp,  9.673196873736643_dp,&
            13.999332658018544_dp,  8.055654719138439e-05_dp,  0.0001947658525515697_dp,  0.00032859686532794565_dp,  0.0004994306825817649_dp,  0.0007340947583837473_dp,&
            0.0010721035282296792_dp,  0.001568310846736925_dp,  0.002297102568888374_dp,  0.0033605346972129828_dp,  0.00490077013024238_dp,  0.0071168327918336724_dp,&
            0.010286868707377024_dp,  0.01479808821232481_dp,  0.021187271898748812_dp,  0.03019555503700218_dp,  0.042842362400043486_dp,  0.06052496436664667_dp,&
            0.08515234916902564_dp,  0.11932532276255173_dp,  0.16657974696296404_dp,  0.23171843775477596_dp,  0.3212739304330216_dp,  0.4441803236632917_dp,&
            0.6128171239515162_dp,  0.8448023886795951_dp,  1.1664998500858539_dp,  1.6210063652060114_dp,  2.2900001311037035_dp,  3.371570229987856_dp,  5.643112922874634_dp]
       aw%aw_erange_matrix(:, 10) = [2.572460674430881e-05_dp,  0.00013828809639808773_dp,  0.0003528803185321503_dp,  0.0006939963784805861_dp,  0.0012058419667301263_dp,&
            0.00196250811153749_dp,  0.003080710120197529_dp,  0.004735659001322314_dp,  0.007183065998906602_dp,  0.010791253848864667_dp,  0.016087325873200328_dp,&
            0.02382213056319325_dp,  0.03506059129972535_dp,  0.051306532736392435_dp,  0.07467445268948096_dp,  0.10812503757927873_dp,  0.15578699671299615_dp,&
            0.2233955307214805_dp,  0.31888824542436384_dp,  0.45321390622010965_dp,  0.6414306299530386_dp,  0.9042033223791973_dp,  1.2698679478183343_dp,&
            1.7773436933847355_dp,  2.4804246798636003_dp,  3.4545978455659587_dp,  4.80921488681201_dp,  6.713149815396915_dp,  9.463167209838238_dp,&
            13.75460415331797_dp,  6.632324623333035e-05_dp,  0.00016073115506706547_dp,  0.0002724186286744679_dp,  0.00041700914328903177_dp,  0.0006188187442380069_dp,&
            0.000913709771883424_dp,  0.0013515392555132155_dp,  0.002000450746020119_dp,  0.0029549917404197673_dp,  0.004348167160940399_dp,  0.006367634264735562_dp,&
            0.009277411142785488_dp,  0.013447324345434384_dp,  0.01939310256171694_dp,  0.027830892296757826_dp,  0.03975118351588367_dp,  0.056518799068157294_dp,&
            0.08000790205760616_dp,  0.11278427119956595_dp,  0.15835215912014322_dp,  0.2214916578911036_dp,  0.3087289982044895_dp,  0.42901765162073763_dp,&
            0.5947915793660636_dp,  0.8237640916060611_dp,  1.142429482918768_dp,  1.5940199744065582_dp,  2.2602675888442345_dp,  3.3389719364130626_dp,  5.605477212236766_dp]
       aw%aw_erange_matrix(:, 11) = [2.0010353507234397e-05_dp,  0.0001077433422458143_dp,  0.00027580023006758454_dp,  0.0005451258515021734_dp,  0.0009539529098224693_dp,&
            0.0015668441726082855_dp,  0.0024859562396115408_dp,  0.003865438814677545_dp,  0.005931997279200694_dp,  0.009015206545406032_dp,  0.013591237471033118_dp,&
            0.020344680291656994_dp,  0.03025506904355767_dp,  0.04471729454363523_dp,  0.0657084625235162_dp,  0.09601821543882148_dp,  0.13956549956219141_dp,&
            0.2018327748807436_dp,  0.2904595479070126_dp,  0.41605219017177664_dp,  0.5932887745998796_dp,  0.8424313672051914_dp,  1.1914160744119529_dp,&
            1.678803642976038_dp,  2.3581208348346263_dp,  3.3047307785320212_dp,  4.6280385004171025_dp,  6.497115386386612_dp,  9.208843054933286_dp,&
            13.457588010635305_dp,  5.1609539489171625e-05_dp,  0.00012547938250585046_dp,  0.00021401951207906163_dp,  0.0003308508348857692_dp,  0.0004973777574979467_dp,&
            0.0007451991801870302_dp,  0.0011183825299403462_dp,  0.0016778698024847744_dp,  0.0025094583225573333_dp,  0.0037353071207149595_dp,  0.005529432603001048_dp,&
            0.008138731296629963_dp,  0.011911786975259289_dp,  0.0173383968709882_dp,  0.02510367274551181_dp,  0.03616185208550007_dp,  0.05183670042056887_dp,&
            0.07395778747131394_dp,  0.10504533244757883_dp,  0.14856147943385148_dp,  0.20925449434054086_dp,  0.2936386766166653_dp,  0.41068755123213285_dp,&
            0.5728998438821491_dp,  0.7981069410955339_dp,  1.1129693992413612_dp,  1.5608972768887142_dp,  2.223705100788062_dp,  3.298855550400102_dp,  5.559194240832182_dp]
       aw%aw_erange_matrix(:, 12) = [1.4913158562437485e-05_dp,  8.046249863674396e-05_dp,  0.00020679269836799646_dp,  0.00041135606175085387_dp,  0.000726429457279576_dp,&
            0.001206954515218603_dp,  0.0019403041016435166_dp,  0.0030592205520702234_dp,  0.004760844972277948_dp,  0.007334997531745175_dp,  0.011205132165430936_dp,&
            0.016986616421301493_dp,  0.025568952229820977_dp,  0.03823114994605764_dp,  0.056802896237992014_dp,  0.08388874591064059_dp,  0.12317874197383723_dp,&
            0.17987720132408636_dp,  0.26129273365588884_dp,  0.3776482327688284_dp,  0.5431920469707082_dp,  0.7777259064649447_dp,  1.1087233079248426_dp,&
            1.5743236370665366_dp,  2.2277313150621567_dp,  3.144149304477041_dp,  4.433028932012796_dp,  6.263665410565493_dp,  8.933099495687644_dp,&
            13.134684015624826_dp,  3.848089108885685e-05_dp,  9.39454195069688e-05_dp,  0.00016153027576924122_dp,  0.0002528432683507074_dp,  0.0003863006748200446_dp,&
            0.000589103857593431_dp,  0.0008994092337673156_dp,  0.0013708168552981645_dp,  0.0020800635852347526_dp,  0.0031378613819817206_dp,  0.004703620407911122_dp,&
            0.007005705166384156_dp,  0.010369489117192241_dp,  0.015256172448531693_dp,  0.02231631000951314_dp,  0.03246335096484727_dp,  0.04697433098325742_dp,&
            0.06762737851039459_dp,  0.09688924910721004_dp,  0.13817140362709024_dp,  0.19618184162864452_dp,  0.2774160356134996_dp,  0.39086437274546876_dp,&
            0.5490937144681445_dp,  0.7700664939132916_dp,  1.0806338759241187_dp,  1.5244170903885788_dp,  2.183343539374224_dp,  3.254529601017961_dp,  5.508095437648589_dp]
       aw%aw_erange_matrix(:, 13) = [1.1418872198921082e-05_dp,  6.173413750253069e-05_dp,  0.00015929220328335313_dp,  0.00031889503854739523_dp,  0.0005682342649483396_dp,&
            0.0009547547779672074_dp,  0.0015542608919948567_dp,  0.0024827272879236525_dp,  0.003914044995780404_dp,  0.006106526724782397_dp,  0.009441451620849054_dp,&
            0.01447827421468276_dp,  0.022033096235727034_dp,  0.03328960553532699_dp,  0.04995516756713029_dp,  0.07447945346316431_dp,  0.11035934551634245_dp,&
            0.16256247169247554_dp,  0.23811344985212937_dp,  0.3469031346310937_dp,  0.5028042883523934_dp,  0.7252121362263632_dp,  1.0411867545570728_dp,&
            1.4884852069632906_dp,  2.120012518561155_dp,  3.0108136851550755_dp,  4.2703677397197355_dp,  6.068165404997884_dp,  8.701408700808228_dp,&
            12.8626331750244_dp,  2.947787098787719e-05_dp,  7.225954134028555e-05_dp,  0.0001252402865134724_dp,  0.00019846473664424758_dp,  0.00030798161384563543_dp,&
            0.0004775038459907116_dp,  0.0007405481153276563_dp,  0.0011449321771295313_dp,  0.0017601237436588445_dp,  0.0026874815648166833_dp,  0.004074324357547685_dp,&
            0.0061335185303593494_dp,  0.009170846038462778_dp,  0.013623145317805922_dp,  0.020111251332522298_dp,  0.029513178507526473_dp,  0.043064913823154624_dp,&
            0.062498815014491525_dp,  0.09023329634576005_dp,  0.12963292767495377_dp,  0.18536685282870904_dp,  0.26390946533406945_dp,  0.3742607782294386_dp,&
            0.5290426355523791_dp,  0.7463296721416969_dp,  1.0531417380773798_dp,  1.493293292834725_dp,  2.1488272848196677_dp,  3.216586556758033_dp,  5.464388958502095_dp]
       aw%aw_erange_matrix(:, 14) = [8.982546693806235e-06_dp,  4.865818678309805e-05_dp,  0.00012604148335636865_dp,  0.00025390861901678955_dp,  0.0004563995854196066_dp,&
            0.0007750952045840026_dp,  0.0012767265489376275_dp,  0.002064101925548432_dp,  0.0032927700122513484_dp,  0.005195994959981289_dp,  0.008121235122746024_dp,&
            0.012582721029292343_dp,  0.019336695756878207_dp,  0.029488494866509965_dp,  0.044644170175709534_dp,  0.06712415308115761_dp,  0.10026296333569998_dp,&
            0.14882782019535817_dp,  0.2196010954609523_dp,  0.3221882442730784_dp,  0.47013658664261176_dp,  0.6824859983381489_dp,  0.9859314820504801_dp,&
            1.417888163895422_dp,  2.030988118342932_dp,  2.900125529064684_dp,  4.134793295219112_dp,  5.904650857570968_dp,  8.50705581410213_dp,&
            12.633884453832914_dp,  2.3198709666506916e-05_dp,  5.709307522017352e-05_dp,  9.972792415428492e-05_dp,  0.0001599273928986242_dp,  0.0002518605559821019_dp,&
            0.00039647689416403176_dp,  0.0006236485188629505_dp,  0.0009766180462365724_dp,  0.0015189982582493745_dp,  0.0023445004429086033_dp,  0.0035904524574293063_dp,&
            0.005456804956339633_dp,  0.008232881015796438_dp,  0.012334887697009843_dp,  0.01835828332102448_dp,  0.027150564163334993_dp,  0.03991202944665125_dp,&
            0.05833481297012722_dp,  0.08479428042701978_dp,  0.12261246982859136_dp,  0.17642213365813839_dp,  0.2526759210340022_dp,  0.3603782883628389_dp,&
            0.5121951678672628_dp,  0.7262964913958404_dp,  1.0298496814196687_dp,  1.4668430564566999_dp,  2.119432331068027_dp,  3.1842446525056274_dp,  5.427158434113688_dp]
       aw%aw_erange_matrix(:, 15) = [7.238349702296008e-06_dp,  3.928458373506955e-05_dp,  0.00010214563688496669_dp,  0.00020702207021802802_dp,  0.0003752580104799391_dp,&
            0.0006437818300993232_dp,  0.001072114382774453_dp,  0.0017525850080720873_dp,  0.002826070350614258_dp,  0.004505651346498813_dp,  0.007111320652225213_dp,&
            0.011120302372900251_dp,  0.017239502228764955_dp,  0.026509227663294707_dp,  0.04045088318380033_dp,  0.06127621597555976_dp,  0.09218233054747209_dp,&
            0.13776574927878416_dp,  0.2046011860485274_dp,  0.3020478300836994_dp,  0.443370229666864_dp,  0.6472971651730427_dp,  0.9402012658200933_dp,&
            1.3591924682162404_dp,  1.956655369037565_dp,  2.8073424394429862_dp,  4.020750844752612_dp,  5.766686098521229_dp,  8.342653009498491_dp,&
            12.439991369032668_dp,  1.8702020468622055e-05_dp,  4.6203219782190167e-05_dp,  8.131762180666392e-05_dp,  0.00013190143534228278_dp,  0.00021061299324796057_dp,&
            0.0003361903026363539_dp,  0.000535605997735673_dp,  0.0008484276122519987_dp,  0.0013334887847408908_dp,  0.0020781819161196638_dp,  0.003211507934451115_dp,&
            0.004922574199422692_dp,  0.007486789650255751_dp,  0.011302797934300408_dp,  0.016944298667165195_dp,  0.02523242357196762_dp,  0.03733639323117268_dp,&
            0.05491299477721061_dp,  0.08029934786534572_dp,  0.11677913471088484_dp,  0.16895145657274943_dp,  0.24324744526263437_dp,  0.3486724489732179_dp,&
            0.49792796808610074_dp,  0.7092652162431419_dp,  1.009980880755789_dp,  1.4442192775687972_dp,  2.094243792411222_dp,  3.1565101709731045_dp,  5.395251764843703_dp]
       aw%aw_erange_matrix(:, 16) = [5.706429844293361e-06_dp,  3.1040364829701505e-05_dp,  8.107328661660111e-05_dp,  0.00016550326188419503_dp,  0.0003029763316477536_dp,&
            0.0005259009041296794_dp,  0.0008867852843333584_dp,  0.0014677450288040894_dp,  0.002395273728414361_dp,  0.003862517134876226_dp,  0.006162129729070569_dp,&
            0.00973422779264423_dp,  0.015235895683403678_dp,  0.023641332048275565_dp,  0.03638532798309582_dp,  0.055567718600602306_dp,  0.08424323176407787_dp,&
            0.12683043751734444_dp,  0.18968622248619452_dp,  0.2819098113026998_dp,  0.4164652673968472_dp,  0.6117484077902107_dp,  0.893783996100621_dp,&
            1.2993493017998998_dp,  1.8805551654919748_dp,  2.7119921346520504_dp,  3.9031533013982855_dp,  5.623997981838861_dp,  8.172198452849088_dp,&
            12.238558563265647_dp,  1.4751349028333799e-05_dp,  3.660909835718544e-05_dp,  6.501211560655282e-05_dp,  0.00010687536539993863_dp,  0.00017337123549154558_dp,&
            0.00028107588099189824_dp,  0.000454137569267576_dp,  0.0007285000756826098_dp,  0.001158211238344495_dp,  0.0018242672088928367_dp,  0.0028471735084972696_dp,&
            0.004404897269066725_dp,  0.006758454939132606_dp,  0.01028819929336629_dp,  0.01554501319618277_dp,  0.023322182186959424_dp,  0.03475584758118297_dp,&
            0.05146485198022031_dp,  0.07574483876932264_dp,  0.11083732616584702_dp,  0.16130359908425193_dp,  0.23354923720333218_dp,  0.3365774511982118_dp,&
            0.48312462993088157_dp,  0.6915266416555069_dp,  0.9892183849743513_dp,  1.420514754404278_dp,  2.0678032116600904_dp,  3.127373080182709_dp,  5.361746811812883_dp]
       aw%aw_erange_matrix(:, 17) = [4.507925303711096e-06_dp,  2.4580451494530444e-05_dp,  6.451263281768541e-05_dp,  0.00013272058871030806_dp,  0.0002455197572529811_dp,&
            0.0004313910776655633_dp,  0.0007367445062259567_dp,  0.001234786889942828_dp,  0.002039383465959761_dp,  0.00332603532508658_dp,  0.005363004530984576_dp,&
            0.008557044500582709_dp,  0.01352012174944254_dp,  0.021166153619467505_dp,  0.03285042829603399_dp,  0.05056939685207151_dp,  0.07724543673788668_dp,&
            0.11713067199112848_dp,  0.17637697527453522_dp,  0.263837336282913_dp,  0.39218930488595755_dp,  0.5795089364205666_dp,  0.8514842120807159_dp,&
            1.2445671334250805_dp,  1.8105970655011243_dp,  2.6239991397280553_dp,  3.7942542214271437_dp,  5.491467032685958_dp,  8.013480504379215_dp,&
            12.05062051565182_dp,  1.165943020924494e-05_dp,  2.907711562944157e-05_dp,  5.2135381292650646e-05_dp,  8.692931914146652e-05_dp,  0.00014332433921579002_dp,&
            0.00023600989072825049_dp,  0.0003866717964299752_dp,  0.0006280479343018203_dp,  0.0010098870748762492_dp,  0.001607379741603303_dp,  0.00253325997236095_dp,&
            0.003955231709080428_dp,  0.006120960061358173_dp,  0.009393712743103142_dp,  0.014302912174317341_dp,  0.02161546778139668_dp,  0.03243595167457546_dp,&
            0.04834668628642002_dp,  0.07160299218511915_dp,  0.10540485530435402_dp,  0.15427553890661685_dp,  0.22459367842315606_dp,  0.3253574621758251_dp,&
            0.46933370101682803_dp,  0.6749372715340488_dp,  0.9697356715189667_dp,  1.3982111886845372_dp,  2.0428791235793997_dp,  3.099885407679866_dp,  5.330156088439341_dp]
       aw%aw_erange_matrix(:, 18) = [3.2492504706355543e-06_dp,  1.7782935938647487e-05_dp,  4.702094527107548e-05_dp,  9.788765130060788e-05_dp,  0.00018394800566011574_dp,&
            0.0003290226690843259_dp,  0.0005722787919864911_dp,  0.0009762934532250238_dp,  0.0016397346725346576_dp,  0.002716663917457915_dp,  0.004445425742834286_dp,&
            0.007191491449190322_dp,  0.011510534331810433_dp,  0.01824066242763528_dp,  0.02863642038498904_dp,  0.044562287024877995_dp,  0.06877046943314344_dp,&
            0.10529755457345026_dp,  0.16002804271088766_dp,  0.24149146592791995_dp,  0.361986165199179_dp,  0.5391615439925698_dp,  0.7982521139849742_dp,&
            1.1752670593405194_dp,  1.7216703901842698_dp,  2.511651208522495_dp,  3.654660979891775_dp,  5.320994285506342_dp,  7.808735565003434_dp,&
            11.80762543615471_dp,  8.410837332223653e-06_dp,  2.113245841372446e-05_dp,  3.845111058827689e-05_dp,  6.548419810172488e-05_dp,  0.00011052592304298797_dp,&
            0.00018602054544749434_dp,  0.0003107143750285354_dp,  0.0005134448260073493_dp,  0.0008386432327700857_dp,  0.0013542399463937198_dp,  0.002163165180703502_dp,&
            0.0034200707057045_dp,  0.0053555078104983965_dp,  0.008310672199526598_dp,  0.012787041135722151_dp,  0.019516894799063707_dp,  0.029562998187469335_dp,&
            0.044458853860523784_dp,  0.0664052872217014_dp,  0.09854535532252624_dp,  0.14534905940024856_dp,  0.2131554574377401_dp,  0.3109515151654752_dp,&
            0.4515398978356489_dp,  0.653437304234083_dp,  0.9443877723109725_dp,  1.3691021780052584_dp,  2.010279336423766_dp,  3.063897971847719_dp,  5.288820184644417_dp]
       aw%aw_erange_matrix(:, 19) = [2.460537860443357e-06_dp,  1.3513517677623214e-05_dp,  3.598515389781885e-05_dp,  7.575261086818076e-05_dp,  0.0001444216294004159_dp,&
            0.00026247903310060725_dp,  0.0004639039840208787_dp,  0.0008036114726657992_dp,  0.001369205915452374_dp,  0.0022989755449385045_dp,  0.0038090333768977967_dp,&
            0.006233886266199457_dp,  0.01008661073890614_dp,  0.016147469119526157_dp,  0.025593516684082565_dp,  0.04018693519947543_dp,  0.06254703089827915_dp,&
            0.0965408099871576_dp,  0.14784085162679408_dp,  0.22471839016516207_dp,  0.33916659817804756_dp,  0.5084886871639984_dp,  0.7575478060875892_dp,&
            1.1219864437669036_dp,  1.6529529420726783_dp,  2.4244318079577103_dp,  3.5458393070593472_dp,  5.187620119758077_dp,  7.648065546110419_dp,&
            11.61648316806262_dp,  6.374120402922641e-06_dp,  1.6128252577743594e-05_dp,  2.9754113670763418e-05_dp,  5.166430893732005e-05_dp,  8.901502625196717e-05_dp,&
            0.00015264214630912148_dp,  0.0002591687585053359_dp,  0.00043455383137211705_dp,  0.000719240735481554_dp,  0.0011756556233951615_dp,  0.0018992288700902785_dp,&
            0.003034542038483531_dp,  0.004798831948386736_dp,  0.007515980097531093_dp,  0.011665360331427038_dp,  0.017951632069288924_dp,  0.027403921176393425_dp,&
            0.041516070343442624_dp,  0.06244414300326091_dp,  0.0932837802259788_dp,  0.13845969920620585_dp,  0.20427585444540086_dp,  0.2997063234922189_dp,&
            0.43757882986144675_dp,  0.6364895482287001_dp,  0.9243252918542686_dp,  1.3459866886820373_dp,  1.9843322156142236_dp,  3.035224685946777_dp,  5.255903429150008_dp]
       aw%aw_erange_matrix(:, 20) = [1.5637740408395342e-06_dp,  8.644581438556428e-06_dp,  2.3326026659580105e-05_dp,  5.012244143843333e-05_dp,  9.804997776691128e-05_dp,&
            0.000183175648888439_dp,  0.0003325825937550191_dp,  0.0005909069030179598_dp,  0.0010307262850542825_dp,  0.001768639145774748_dp,  0.0029898323691103764_dp,&
            0.004985271727949821_dp,  0.008207526024267482_dp,  0.013353895620644004_dp,  0.0214892751837372_dp,  0.034226463130117915_dp,  0.053989018153179254_dp,&
            0.08439206340451581_dp,  0.1307907673641504_dp,  0.20106621537104283_dp,  0.3067466011130968_dp,  0.46460274135109153_dp,  0.698920779817791_dp,&
            1.0447664262134877_dp,  1.5527831264532448_dp,  2.296617666855539_dp,  3.3856119528633637_dp,  4.990434105875191_dp,  7.4097119078577_dp,&
            11.33215729315795_dp,  4.056801137252665e-06_dp,  1.0400304864325584e-05_dp,  1.968307964805301e-05_dp,  3.537211676997081e-05_dp,  6.309758174077909e-05_dp,&
            0.00011156144898932593_dp,  0.0001945228908869725_dp,  0.0003339605532184735_dp,  0.000564711446670201_dp,  0.0009413729224733118_dp,  0.0015485976328911168_dp,&
            0.0025163527439127716_dp,  0.004042366217657837_dp,  0.006424908889055276_dp,  0.010110351754549359_dp,  0.01576172006986062_dp,  0.02435693199744164_dp,&
            0.03732881710193292_dp,  0.05676374129839576_dp,  0.08568239505343193_dp,  0.12843634401072082_dp,  0.19127041101280928_dp,  0.28313227875730185_dp,&
            0.41688110772833514_dp,  0.6112296495300013_dp,  0.8942832614684186_dp,  1.3112417204467246_dp,  1.9452277892500305_dp,  2.9919599852080703_dp,  5.206267501211972_dp]
       aw%aw_erange_matrix(:, 21) = [1.162904999603526e-06_dp,  6.460220717190712e-06_dp,  1.7606607480068412e-05_dp,  3.8410731277459235e-05_dp,  7.652869912290393e-05_dp,&
            0.00014570244755436322_dp,  0.00026937164379552174_dp,  0.0004866802067040396_dp,  0.0008620687596415197_dp,  0.001500239074857704_dp,  0.0025692139442485516_dp,&
            0.004335527408651135_dp,  0.00721744537817494_dp,  0.011864777494471953_dp,  0.01927761941951377_dp,  0.030981700048419165_dp,  0.04928550606403292_dp,&
            0.07765484694198317_dp,  0.12125517110926898_dp,  0.18773242857967326_dp,  0.2883321960866146_dp,  0.43949878526491704_dp,  0.6651609536255874_dp,&
            1.0000229662476519_dp,  1.4944068851808134_dp,  2.2217383426459816_dp,  3.2913012165448343_dp,  4.873895944019108_dp,  7.268367413333011_dp,&
            11.163103194460474_dp,  3.0200745063435063e-06_dp,  7.819224105572315e-06_dp,  1.5081156733800399e-05_dp,  2.776801766506454e-05_dp,  5.0700455074630125e-05_dp,&
            9.145615638049602e-05_dp,  0.0001622504928743556_dp,  0.00028286367265657205_dp,  0.0004849887879932366_dp,  0.0008187828750348566_dp,  0.0013627221411797283_dp,&
            0.002238314348285273_dp,  0.003631885398877431_dp,  0.005826597760169261_dp,  0.009249172128608455_dp,  0.014537611686674936_dp,  0.0226387619956523_dp,&
            0.03494804598497715_dp,  0.053508604450163914_dp,  0.0812939971568708_dp,  0.12260888215529167_dp,  0.18365876903663883_dp,  0.27337120427563866_dp,&
            0.4046203209300098_dp,  0.5961868839329513_dp,  0.8763095433545912_dp,  1.2903758814089559_dp,  1.9216820534456363_dp,  2.9658783009061263_dp,  5.176364379511127_dp]
       aw%aw_erange_matrix(:, 22) = [8.982200709109713e-07_dp,  5.013746520038623e-06_dp,  1.3797556086375253e-05_dp,  3.053886905121258e-05_dp,  6.188411270912267e-05_dp,&
            0.00011984809712211666_dp,  0.00022515174308936954_dp,  0.00041280335546669807_dp,  0.0007410557365297971_dp,  0.0013054805064640572_dp,  0.0022608163607798406_dp,&
            0.003854540538433318_dp,  0.006477967304857807_dp,  0.010743336152506139_dp,  0.01759914400488243_dp,  0.028501352515678965_dp,  0.04566568530951012_dp,&
            0.07243687914225057_dp,  0.11382566141804518_dp,  0.17728511649027562_dp,  0.2738276317065499_dp,  0.41962643246873177_dp,  0.6383115216108352_dp,&
            0.9642825901967439_dp,  1.4475879782153047_dp,  2.1614613507414213_dp,  3.2151308325865324_dp,  4.779503431415253_dp,  7.153609650797204_dp,&
            11.025589359258367_dp,  2.33509910395076e-06_dp,  6.10398857813977e-06_dp,  1.1988268517456785e-05_dp,  2.2570731247056643e-05_dp,  4.206867163196105e-05_dp,&
            7.722227756857685e-05_dp,  0.00013907449507452585_dp,  0.0002457091386543816_dp,  0.00042636927263886435_dp,  0.0007277240308901886_dp,  0.0012233619517539264_dp,&
            0.0020280492037779985_dp,  0.0033189631613667883_dp,  0.005367062740884365_dp,  0.008583092022062843_dp,  0.0135845766573915_dp,  0.021292762750234086_dp,&
            0.033072042665578866_dp,  0.05092942142345538_dp,  0.07779867772338114_dp,  0.11794435500606822_dp,  0.17753757154997848_dp,  0.26548687080483935_dp,&
            0.3946762763128839_dp,  0.5839408961339644_dp,  0.8616294721408878_dp,  1.27328786626255_dp,  1.902362545255473_dp,  2.944458259522062_dp,  5.1518134931437265_dp]
       aw%aw_erange_matrix(:, 23) = [6.070765776691511e-07_dp,  3.416871225040894e-06_dp,  9.561919657171797e-06_dp,  2.1683051083242104e-05_dp,  4.515689173365717e-05_dp,&
            8.982491421280511e-05_dp,  0.00017296367691311632_dp,  0.00032428348083868487_dp,  0.0005940202656656541_dp,  0.001065793876730367_dp,  0.0018767883279394732_dp,&
            0.0032490764599530236_dp,  0.0055377486897541_dp,  0.009304158830434279_dp,  0.015426407047504075_dp,  0.02526460961553787_dp,  0.0409062041812229_dp,&
            0.06552740429734295_dp,  0.10392219781712_dp,  0.16327176797255086_dp,  0.25425762049527484_dp,  0.3926657116824371_dp,  0.6016960744289409_dp,&
            0.9153062522487351_dp,  1.3831427017478757_dp,  2.0781516959653166_dp,  3.1094698633638336_dp,  4.64815163738497_dp,  6.99350202546865_dp,&
            10.833342615226073_dp,  1.5810344568110277e-06_dp,  4.201983132608034e-06_dp,  8.509301559244783e-06_dp,  1.660217499192091e-05_dp,  3.1937948250709083e-05_dp,&
            6.0196622838881555e-05_dp,  0.00011090164490251824_dp,  0.00019990225033881748_dp,  0.0003531805009952824_dp,  0.000612721804071078_dp,  0.0010454964371682956_dp,&
            0.0017570690249359784_dp,  0.0029120339233426416_dp,  0.004764440773172712_dp,  0.007702737556158165_dp,  0.012315669576916223_dp,  0.019488241944529968_dp,&
            0.03054057337997689_dp,  0.0474276677063864_dp,  0.0730254826754342_dp,  0.11153946595828376_dp,  0.1690888848289202_dp,  0.2545515426791882_dp,&
            0.3808215765752087_dp,  0.5668083660036856_dp,  0.8410170454494971_dp,  1.2492234533619844_dp,  1.8750991975085831_dp,  2.91420232217261_dp,  5.117153113908438_dp]
       aw%aw_erange_matrix(:, 24) = [3.9676424024146875e-07_dp,  2.2573188915559954e-06_dp,  6.453978574008018e-06_dp,  1.5076014172691726e-05_dp,  3.241300804980766e-05_dp,&
            6.644316731632574e-05_dp,  0.00013145859105193587_dp,  0.0002525113816701914_dp,  0.0004726891479758136_dp,  0.0008648183450896165_dp,  0.0015500448519401517_dp,&
            0.00272698303228361_dp,  0.004716941205527944_dp,  0.008033374074156422_dp,  0.013487530814882825_dp,  0.022347744374551653_dp,  0.03657761361377878_dp,&
            0.0591894063537043_dp,  0.09476465831169727_dp,  0.15021601736547668_dp,  0.23589569241536856_dp,  0.3672011309219512_dp,  0.5668972518738795_dp,&
            0.8684896297613345_dp,  1.3212084913196627_dp,  1.9976957850654782_dp,  3.00698161245029_dp,  4.520261622699826_dp,  6.837125864760079_dp,&
            10.645115539809895_dp,  1.0356874216884052e-06_dp,  2.812088667709278e-06_dp,  5.914445821533368e-06_dp,  1.2021336664547268e-05_dp,  2.3940050145157427e-05_dp,&
            4.6430499419310365e-05_dp,  8.765815047744608e-05_dp,  0.00016143932217372545_dp,  0.0002907528424437285_dp,  0.0005132235070598931_dp,  0.0008895970920506905_dp,&
            0.0015167009659352298_dp,  0.0025470709958953243_dp,  0.004218406965056695_dp,  0.006897410357753999_dp,  0.011144514451186464_dp,  0.017808764773569234_dp,&
            0.028165958189904933_dp,  0.044118529654135305_dp,  0.0684833014292294_dp,  0.1054043456969387_dp,  0.16094570015015752_dp,  0.24395004441521287_dp,&
            0.3673168074143_dp,  0.550025709656669_dp,  0.8207374761240367_dp,  1.2254629611475691_dp,  1.8481116761650778_dp,  2.884215749911384_dp,  5.082816358644647_dp]
       aw%aw_erange_matrix(:, 25) = [2.3571329342099874e-07_dp,  1.3632343165414816e-06_dp,  4.023756596461991e-06_dp,  9.795394424152668e-06_dp,  2.195714519062327e-05_dp,&
            4.674841254419276e-05_dp,  9.563512339593217e-05_dp,  0.00018918230274547437_dp,  0.00036348321259617266_dp,  0.0006806544569921736_dp,  0.001245724390748882_dp,&
            0.002233454990133142_dp,  0.00393042938989855_dp,  0.0068003647335751725_dp,  0.011584430850915652_dp,  0.019453873577319414_dp,  0.032240162866761804_dp,&
            0.052779185694875504_dp,  0.08542204924536101_dp,  0.136787796738067_dp,  0.2168656674798965_dp,  0.3406212481786099_dp,  0.5303313641534245_dp,&
            0.8189892522205988_dp,  1.2553469341161432_dp,  1.9116889104572583_dp,  2.896909089589307_dp,  4.382351904816623_dp,  6.667935664555373_dp,&
            10.44093549828013_dp,  6.17436261768095e-07_dp,  1.7313881986083482e-06_dp,  3.841234035608705e-06_dp,  8.228727173271725e-06_dp,  1.709820341911912e-05_dp,&
            3.4332974259821285e-05_dp,  6.676429369632049e-05_dp,  0.00012617607786624823_dp,  0.0002325052400773221_dp,  0.0004189073145508487_dp,  0.000739676578430109_dp,&
            0.0012824873185346604_dp,  0.0021871159459730906_dp,  0.0036737980596718025_dp,  0.006085791120718465_dp,  0.009952725340223944_dp,  0.01608416328295351_dp,&
            0.02570677247015947_dp,  0.04066411629576856_dp,  0.06370600198386596_dp,  0.09890586632566792_dp,  0.15226260044097462_dp,  0.23257478024275838_dp,&
            0.35274192070832605_dp,  0.5318168800175598_dp,  0.7986316416317844_dp,  1.1994633292908101_dp,  1.8185003473260537_dp,  2.851271019221645_dp,  5.045111765749473_dp]
       aw%aw_erange_matrix(:, 26) = [1.4156108142769783e-07_dp,  8.358408090279267e-07_dp,  2.563425668333653e-06_dp,  6.532496957198105e-06_dp,  1.529085429435563e-05_dp,&
            3.381004239065925e-05_dp,  7.145759572627074e-05_dp,  0.0001454048583206386_dp,  0.0002863676783658706_dp,  0.0005481076173216848_dp,  0.0010229120533241523_dp,&
            0.0018664583231752616_dp,  0.0033372345185815654_dp,  0.005858297289124_dp,  0.010112965971934416_dp,  0.017191603273076256_dp,  0.028814641239522522_dp,&
            0.04766849746482397_dp,  0.07790741829272711_dp,  0.12589760019317675_dp,  0.2013131761655887_dp,  0.3187416881227427_dp,  0.5000288557106496_dp,&
            0.7777107650960449_dp,  1.200107192058866_dp,  1.8391727503056268_dp,  2.803666178964093_dp,  4.265054523983134_dp,  6.5235529232359735_dp,&
            10.266242055151874_dp,  3.724391604535185e-07_dp,  1.0869250980985111e-06_dp,  2.5605516562597e-06_dp,  5.7850356623393795e-06_dp,  1.2528619638131183e-05_dp,&
            2.6016291223482876e-05_dp,  5.204832564381964e-05_dp,  0.00010081228246405614_dp,  0.00018982349811898816_dp,  0.0003486351977763963_dp,  0.000626280938172421_dp,&
            0.0011028905979224082_dp,  0.0019076162117445236_dp,  0.0032460080919710396_dp,  0.005441424633595424_dp,  0.00899711363781583_dp,  0.01468851247521531_dp,&
            0.023699427657021866_dp,  0.03782154672437762_dp,  0.059744945747629535_dp,  0.09347917704011861_dp,  0.1449628687820927_dp,  0.2229515840638538_dp,&
            0.34033980700933175_dp,  0.5162398701965166_dp,  0.7796319970356158_dp,  1.1770306615230532_dp,  1.7928810285625805_dp,  2.8227297861916565_dp,  5.012462806749877_dp]
       aw%aw_erange_matrix(:, 27) = [5.619543540742037e-08_dp,  3.5072928313745456e-07_dp,  1.1785442886447028e-06_dp,  3.3029924055616623e-06_dp,  8.394882189434066e-06_dp,&
            1.987896075698813e-05_dp,  4.449626404673206e-05_dp,  9.506973724339386e-05_dp,  0.00019528307697601834_dp,  0.0003877720465039687_dp,  0.0007475784942017128_dp,&
            0.001404160723388905_dp,  0.002576876821658531_dp,  0.004631430044514_dp,  0.008168570072664031_dp,  0.014161927673231893_dp,  0.024169988544510326_dp,&
            0.04065896918886588_dp,  0.06749026098429482_dp,  0.1106503055340562_dp,  0.17933541972048098_dp,  0.28755422315274737_dp,  0.45648527437792397_dp,&
            0.7179486868235401_dp,  1.119577032884951_dp,  1.7327879813628193_dp,  2.6661030665007965_dp,  4.091161983842176_dp,  6.3086506803163065_dp,&
            10.005419833797736_dp,  1.4962249041277701e-07_dp,  4.835223883864694e-07_dp,  1.2927723889950915e-06_dp,  3.221113081192073e-06_dp,  7.5085382788298944e-06_dp,&
            1.653889597388599e-05_dp,  3.475621269796358e-05_dp,  7.020789074538884e-05_dp,  0.000137107896070371_dp,  0.0002600215543190807_dp,  0.00048059369103468994_dp,&
            0.0008682158907070847_dp,  0.0015367276921454673_dp,  0.0026702645844449866_dp,  0.004562839898703722_dp,  0.007678366608423676_dp,  0.012740848501038493_dp,&
            0.02086877544772841_dp,  0.03377384609132226_dp,  0.05405280484981297_dp,  0.08561371875294983_dp,  0.13429708936338725_dp,  0.20878452860262342_dp,&
            0.32195328331951945_dp,  0.4929981089833493_dp,  0.7511227960385115_dp,  1.1432128213684527_dp,  1.7541296769802968_dp,  2.7794877584199225_dp,  4.963023556769552_dp]
       aw%aw_erange_matrix(:, 28) = [2.4343774571442308e-08_dp,  1.6513085830933388e-07_dp,  6.196802532079424e-07_dp,  1.9114039978346155e-06_dp,  5.2375690334248195e-06_dp,&
            1.316277905469794e-05_dp,  3.092030271194152e-05_dp,  6.876819455159673e-05_dp,  0.0001461441409773963_dp,  0.0002988264714800754_dp,  0.0005910302376020617_dp,&
            0.0011354771724870155_dp,  0.0021261659208679307_dp,  0.0038911087717360972_dp,  0.0069760886290369745_dp,  0.012276085772085119_dp,  0.021239226973771258_dp,&
            0.03618004618429005_dp,  0.060756103616981624_dp,  0.10068683961443668_dp,  0.16482922554469034_dp,  0.2667764798918916_dp,  0.4272231990996556_dp,&
            0.6774637303634756_dp,  1.064618278549332_dp,  1.6596947503579922_dp,  2.5710208535654164_dp,  3.9703478472317504_dp,  6.158711044835771_dp,&
            9.822845740299956_dp,  6.607779435757453e-08_dp,  2.4531316662915766e-07_dp,  7.456766806787507e-07_dp,  2.026328956129065e-06_dp,  5.0327732385235214e-06_dp,&
            1.1652295901214502e-05_dp,  2.5505955301673674e-05_dp,  5.331453501898778e-05_dp,  0.0001072058234797249_dp,  0.00020853777085767683_dp,  0.0003941273627591354_dp,&
            0.0007262462873395727_dp,  0.0013084426047463612_dp,  0.00231027419159065_dp,  0.004005535279633006_dp,  0.006830718203653046_dp,  0.011473559821092195_dp,&
            0.019005957730023747_dp,  0.031081856143062504_dp,  0.0502296954993904_dp,  0.08028203266369409_dp,  0.12700459445500437_dp,  0.19901982937944282_dp,&
            0.3091853239783209_dp,  0.4767481905112323_dp,  0.7310696923220488_dp,  1.1193069602208368_dp,  1.72663775937696_dp,  2.74875508795335_dp,  4.927904659791831_dp]
       aw%aw_erange_matrix(:, 29) = [3.3134286206436378e-09_dp,  3.608476731974466e-08_dp,  1.9011771017402648e-07_dp,  7.319641454121685e-07_dp,  2.3402664590560514e-06_dp,&
            6.59508695658557e-06_dp,  1.6937909149354525e-05_dp,  4.048237319663682e-05_dp,  9.132011445721631e-05_dp,  0.00019639172259169963_dp,  0.0004056673541505352_dp,&
            0.0008094360939409007_dp,  0.001567130287522824_dp,  0.0029545924806747004_dp,  0.005440438931022152_dp,  0.009807773172795757_dp,  0.017345795847210112_dp,&
            0.030148043017114434_dp,  0.0515717640546498_dp,  0.08693872933672857_dp,  0.14459504048117927_dp,  0.23750169179013617_dp,  0.3856081802743787_dp,&
            0.6193889538319287_dp,  0.9851526108818242_dp,  1.5532426287786882_dp,  2.431651849778814_dp,  3.7922799603529507_dp,  5.93671227399739_dp,&
            9.551588089060862_dp,  1.0546633313159667e-08_dp,  6.866238130433312e-08_dp,  2.8014791592507246e-07_dp,  9.089927672975243e-07_dp,  2.5558210419120903e-06_dp,&
            6.498693707400478e-06_dp,  1.531991208406891e-05_dp,  3.4023779895267246e-05_dp,  7.197856727334818e-05_dp,  0.00014621506144801751_dp,  0.0002869199340733901_dp,&
            0.0005464264697957605_dp,  0.0010137006051227095_dp,  0.0018373548659257801_dp,  0.0032617470785987366_dp,  0.005682944230668296_dp,  0.009734554219352109_dp,&
            0.016418075032354977_dp,  0.02729906435610205_dp,  0.04479995778261982_dp,  0.07263414004010217_dp,  0.11644650227746163_dp,  0.1847594577631817_dp,&
            0.2903885825624092_dp,  0.4526492240843357_dp,  0.7011369514153092_dp,  1.0834309171758731_dp,  1.6852193328076974_dp,  2.7023644845108916_dp,  4.874923027763007_dp]
    case(32)
       aw%energy_range(:) = [4862.0_dp, 5846.0_dp, 6665.0_dp, 7800.0_dp, 10044.0_dp, 14058.0_dp,&
            19114.0_dp, 25870.0_dp, 35180.0_dp, 58986.0_dp, 85052.0_dp, 126612.0_dp, 247709.0_dp, 452410.0_dp,&
            1104308.0_dp, 2582180.0_dp, 10786426.0_dp, 72565710.0_dp]
       aw%aw_erange_matrix(:, 1) = [6.390102821995496e-05_dp,  0.0003410048490902184_dp,  0.000858015893339577_dp,  0.0016504328288047757_dp,  0.002778247721262644_dp,&
            0.004336751679872466_dp,  0.0064710209548362084_dp,  0.009393407912277671_dp,  0.01340416788754177_dp,  0.018917565422190717_dp,  0.026497301468281335_dp,&
            0.036905213045864395_dp,  0.0511672148624311_dp,  0.07066128158960995_dp,  0.09723384839331731_dp,  0.13335312533591778_dp,  0.18231048817210543_dp,&
            0.24848450523394644_dp,  0.3376865863344385_dp,  0.4576131552357413_dp,  0.6184374387625521_dp,  0.8335859250026539_dp,  1.1207633263472254_dp,&
            1.5033220996389998_dp,  2.0121331479703484_dp,  2.6882384204905283_dp,  3.5868412256539663_dp,  4.783852020373796_dp,  6.387973693919086_dp,&
            8.566792744620582_dp,  11.61702931918549_dp,  16.243244163196902_dp,  0.0001644750107629364_dp,  0.0003927652580734362_dp,  0.0006470526340880139_dp,&
            0.0009476004809462573_dp,  0.0013236672585754251_dp,  0.00181722804655121_dp,  0.0024863648293148356_dp,  0.0034080264366486593_dp,  0.004681956510169319_dp,&
            0.006437986099490384_dp,  0.00884737093390095_dp,  0.01213798662642807_dp,  0.01661375946425665_dp,  0.022679584239104586_dp,  0.030873595466529645_dp,&
            0.0419091652309378_dp,  0.05672961944641861_dp,  0.07657952416894345_dp,  0.10309761695759492_dp,  0.13843826485639682_dp,  0.18543118617833296_dp,&
            0.24779409719067566_dp,  0.3304222201755898_dp,  0.43979746242641576_dp,  0.5846010059990558_dp,  0.7767070228661042_dp,  1.0329650164221098_dp,&
            1.3787901136203442_dp,  1.8564287417380754_dp,  2.547529279131862_dp,  3.6531886321446416_dp,  5.969270620351697_dp]
       aw%aw_erange_matrix(:, 2) = [5.442349002883837e-05_dp,  0.0002906163628933693_dp,  0.0007321293214401846_dp,  0.0014109784539713872_dp,  0.0023816365307190094_dp,&
            0.0037311772208214787_dp,  0.005592773584012946_dp,  0.0081619152463338_dp,  0.011715729132325964_dp,  0.016637929524033673_dp,  0.023453105417011018_dp,&
            0.03287403908552556_dp,  0.045865900699847945_dp,  0.06373214396298486_dp,  0.08822857905561622_dp,  0.12171425464017217_dp,  0.16735049859917261_dp,&
            0.22936294868166074_dp,  0.3133859506800179_dp,  0.4269147709487938_dp,  0.5798994400836499_dp,  0.7855261781163106_dp,  1.0612512378052221_dp,&
            1.4301841263166695_dp,  1.9229772371853768_dp,  2.580501782235826_dp,  3.4578625966778955_dp,  4.630962088019041_dp,  6.208582258228505_dp,&
            8.358418878620247_dp,  11.37712612667173_dp,  15.968170544511102_dp,  0.0001401015051695221_dp,  0.00033499665110023704_dp,  0.0005532522530987489_dp,&
            0.0008133560062402562_dp,  0.0011422450286066607_dp,  0.0015787618621387204_dp,  0.002176697085238142_dp,  0.0030073676766694266_dp,  0.004163649639092577_dp,&
            0.005767402761174793_dp,  0.007980687395399007_dp,  0.011020625281672214_dp,  0.015178459352483943_dp,  0.020844174229750198_dp,  0.028538588286649656_dp,&
            0.03895532923923562_dp,  0.05301575138551127_dp,  0.07194074865569469_dp,  0.09734467595148226_dp,  0.1313584288969531_dp,  0.17679160298883081_dp,&
            0.23734854692003116_dp,  0.31792227850976007_dp,  0.42500882037583165_dp,  0.5673248919963556_dp,  0.7568049338284129_dp,  1.0103817704846647_dp,&
            1.3535647228420928_dp,  1.8286721197765015_dp,  2.517319618502594_dp,  3.6202102280059703_dp,  5.930981680526662_dp]
       aw%aw_erange_matrix(:, 3) = [4.8544808464658475e-05_dp,  0.0002593485712181486_dp,  0.0006539518705617748_dp,  0.0012620998863735048_dp,  0.002134648029009164_dp,&
            0.003353250117842424_dp,  0.005043178699940835_dp,  0.007388686080392843_dp,  0.010651467508498257_dp,  0.01519484467183565_dp,  0.02151725732341947_dp,&
            0.03029858754646109_dp,  0.04246309490817034_dp,  0.05926381239480956_dp,  0.08239494336727046_dp,  0.11414097839482948_dp,  0.15757400511758252_dp,&
            0.21681422548727597_dp,  0.29737332757633034_dp,  0.4066065418252456_dp,  0.5543077141349219_dp,  0.7534939940706474_dp,  1.0214457127670413_dp,&
            1.3810991072747199_dp,  1.8629492914867638_dp,  2.5077428833752067_dp,  3.370510682262872_dp,  4.52714545796296_dp,  6.086483167699423_dp,&
            8.216299008759131_dp,  11.213210923746692_dp,  15.779945298966286_dp,  0.00012498176375406726_dp,  0.000299130758824733_dp,  0.0004949254877793316_dp,&
            0.0007296849915859161_dp,  0.0010288042473983914_dp,  0.0014290235014882077_dp,  0.0019812415860893163_dp,  0.0027529950493393966_dp,  0.00383257137438519_dp,&
            0.005336492795437645_dp,  0.007420608589016774_dp,  0.01029470391501449_dp,  0.014241296894904196_dp,  0.01964004703493962_dp,  0.02699971016577413_dp,&
            0.03700007872604122_dp,  0.050547012803328586_dp,  0.06884458868981981_dp,  0.09348967784530542_dp,  0.12659613160197555_dp,  0.17095862941207848_dp,&
            0.2302711267303353_dp,  0.3094239152326776_dp,  0.4149216994326361_dp,  0.555505084520509_dp,  0.7431502415960526_dp,  0.9948487920453783_dp,&
            1.3361777826685788_dp,  1.809509286088653_dp,  2.496441460806496_dp,  3.597410863544938_dp,  5.9045255378523525_dp]
       aw%aw_erange_matrix(:, 4) = [4.2319791684398945e-05_dp,  0.0002262256833340046_dp,  0.0005710759339812151_dp,  0.0011041000499916853_dp,  0.0018721273080030145_dp,&
            0.002950744254164121_dp,  0.004456329948077625_dp,  0.006560445775378507_dp,  0.009507343812482433_dp,  0.013637283047013359_dp,  0.01941907839765781_dp,&
            0.027495223047590025_dp,  0.03874330338636378_dp,  0.054358588747561776_dp,  0.07596440384363516_dp,  0.10575909660902735_dp,  0.14671121294566578_dp,&
            0.20281810337064493_dp,  0.27944792395127677_dp,  0.38379131316561327_dp,  0.525457692252208_dp,  0.7172635733637082_dp,  0.9762793617085143_dp,&
            1.3252331952635001_dp,  1.7944297509586076_dp,  2.4244629711653745_dp,  3.270270626899914_dp,  4.407729666691881_dp,  5.945738068832148_dp,&
            8.052167730865541_dp,  11.023603896745234_dp,  15.561924493694205_dp,  0.00010896980958811763_dp,  0.0002611185815588283_dp,  0.00043301817888623847_dp,&
            0.0006406820460768114_dp,  0.0009077674409158128_dp,  0.001268624601649587_dp,  0.0017708556976942175_dp,  0.002477704567241762_dp,  0.0034722665030748113_dp,&
            0.004865004260125733_dp,  0.006804656478190395_dp,  0.009492536281842186_dp,  0.013201017163803366_dp,  0.01829768554187978_dp,  0.025277118873799437_dp,&
            0.034802761118126926_dp,  0.04776205831199323_dp,  0.06533896907314592_dp,  0.08910928388293891_dp,  0.1211660799438405_dp,  0.16428554188642505_dp,&
            0.22214824519628396_dp,  0.29963998828115096_dp,  0.40327442471190905_dp,  0.5418192779326029_dp,  0.7272994796496782_dp,  0.9767765646718063_dp,&
            1.315909458684106_dp,  1.7871373292268897_dp,  2.4720433230436907_dp,  3.5707586649853615_dp,  5.873612522584144_dp]
       aw%aw_erange_matrix(:, 5) = [3.393255390660133e-05_dp,  0.00018157221465841256_dp,  0.0004592307676320871_dp,  0.0008905276367436023_dp,  0.0015164746316516245_dp,&
            0.0024038182425049908_dp,  0.003655880996707775_dp,  0.0054255179504103905_dp,  0.007931245931576318_dp,  0.0114792650369607_dp,  0.01649451279684121_dp,&
            0.023563843280162002_dp,  0.033494992691347265_dp,  0.047396265575400595_dp,  0.06678363974811272_dp,  0.093724229945987_dp,  0.13102792038715239_dp,&
            0.18250270781166214_dp,  0.2532942033927338_dp,  0.3503362853107127_dp,  0.4829488331079465_dp,  0.6636312054412428_dp,  0.9091194626135006_dp,&
            1.241807411536441_dp,  1.6916904363164948_dp,  2.299110603722669_dp,  3.1188473050465815_dp,  4.226742880544712_dp,  5.73178901479017_dp,&
            7.802015996740442_dp,  10.73398018079764_dp,  15.228283328315147_dp,  8.739335036060486e-05_dp,  0.00020983760313852035_dp,  0.0003493229451058535_dp,&
            0.0005199664146008647_dp,  0.0007428694438573918_dp,  0.001048824687194698_dp,  0.0014805150854819704_dp,  0.0020948189118065326_dp,  0.0029671598393268173_dp,&
            0.004198990073584458_dp,  0.005928343272370908_dp,  0.008343628060712927_dp,  0.011701624144711492_dp,  0.0163512213147089_dp,  0.022764888824347185_dp,&
            0.03158039794694182_dp,  0.04365605404949107_dp,  0.06014369592103612_dp,  0.08258509281019695_dp,  0.11303931699899293_dp,  0.1542516135087157_dp,&
            0.20987914795565896_dp,  0.28479788380848736_dp,  0.38553266378218126_dp,  0.5208913754190811_dp,  0.7029743313415381_dp,  0.9489538722825864_dp,&
            1.2846216880545815_dp,  1.7525303625505235_dp,  2.4342520367725067_dp,  3.5294590063092537_dp,  5.82574638448254_dp]
       aw%aw_erange_matrix(:, 6) = [2.5286922170354894e-05_dp,  0.00013550300515152938_dp,  0.00034365254863389185_dp,  0.0006692829830793605_dp,  0.0011467768099562_dp,&
            0.0018326842825994733_dp,  0.0028151146148173586_dp,  0.004225026003742279_dp,  0.006250780019705354_dp,  0.009158586718943748_dp,  0.013321604127882514_dp,&
            0.01926054419553751_dp,  0.02769941545264008_dp,  0.03964139074601272_dp,  0.056471580120530336_dp,  0.08009577889831763_dp,  0.11312722824816804_dp,&
            0.15913731138320336_dp,  0.22299123810302657_dp,  0.3112966160076166_dp,  0.43300212495019025_dp,  0.6001966661049284_dp,  0.8291790761739616_dp,&
            1.141900704443846_dp,  1.5679415214227161_dp,  2.147298128643314_dp,  2.934523705189343_dp,  4.00539737112386_dp,  5.469022915149277_dp,&
            7.493643904207673_dp,  10.375817412384437_dp,  14.814601510443076_dp,  6.514769920993364e-05_dp,  0.00015687397983623111_dp,  0.00026260059811101074_dp,&
            0.000394267458907849_dp,  0.000569987485627106_dp,  0.0008163276402528582_dp,  0.0011701237352853642_dp,  0.001680758396099781_dp,  0.002414639429717159_dp,&
            0.0034624986342827427_dp,  0.004949404187288667_dp,  0.00704789552613599_dp,  0.00999535091125554_dp,  0.014117167183015276_dp,  0.019857768872086755_dp,&
            0.02782204007098351_dp,  0.03883055990410155_dp,  0.05399308565310966_dp,  0.07480617533828689_dp,  0.1032828703661818_dp,  0.14212537017885443_dp,&
            0.19495650401901887_dp,  0.2666345532300576_dp,  0.3636933937165475_dp,  0.4949878042793592_dp,  0.6727122940047311_dp,  0.91418280277028_dp,&
            1.2453685853158643_dp,  1.7089820529103965_dp,  2.3866034784779506_dp,  3.4773514427582293_dp,  5.765409584335823_dp]
       aw%aw_erange_matrix(:, 7) = [1.932135319864962e-05_dp,  0.00010368197789193328_dp,  0.0002636661772942731_dp,  0.0005157191470705282_dp,  0.0008891130218806238_dp,&
            0.0014324323017507073_dp,  0.002221781063255329_dp,  0.003370774273254239_dp,  0.005043864110016273_dp,  0.00747544974094905_dp,  0.010997228176632299_dp,&
            0.01607649385487372_dp,  0.023369007932130658_dp,  0.033791453456649585_dp,  0.04862029962984213_dp,  0.06962623074409648_dp,  0.09925634818539467_dp,&
            0.14088037853305638_dp,  0.199122452009721_dp,  0.2803071363072363_dp,  0.39305808457961094_dp,  0.549101229063986_dp,  0.7643445772055675_dp,&
            1.0603390883163382_dp,  1.466283147413628_dp,  2.0218493641999524_dp,  2.781370504056042_dp,  3.8205517999540817_dp,  5.248586694740235_dp,&
            7.23391404025796_dp,  10.073125286074314_dp,  14.464007598891323_dp,  4.9794334064441845e-05_dp,  0.0001202442466871008_dp,  0.00020239142207391234_dp,&
            0.00030648388928353133_dp,  0.00044826394218884584_dp,  0.0006508962458350465_dp,  0.0009465213181019036_dp,  0.001378560941846276_dp,  0.002006235556189542_dp,&
            0.002911588665713435_dp,  0.004208981250342487_dp,  0.006057652087479506_dp,  0.008678530354123182_dp,  0.012376893766972177_dp,  0.01757291406967402_dp,&
            0.02484274557305271_dp,  0.034973648086824426_dp,  0.04903775694722466_dp,  0.0684906283923225_dp,  0.09530279631811456_dp,  0.1321356654214213_dp,&
            0.1825779770666456_dp,  0.2514677535724369_dp,  0.34534170861863805_dp,  0.473091186246975_dp,  0.6469906291688993_dp,  0.8844828931716822_dp,&
            1.2116994219481383_dp,  1.6715060390575855_dp,  2.3455108368533817_dp,  3.432378601360354_dp,  5.713384147526647_dp]
       aw%aw_erange_matrix(:, 8) = [1.4818639796865528e-05_dp,  7.963886363515554e-05_dp,  0.00020311223580610354_dp,  0.00039911404589973475_dp,  0.0006926333595966217_dp,&
            0.0011254943882583104_dp,  0.0017635291623248677_dp,  0.0027054780460561163_dp,  0.004095260696633318_dp,  0.006139843660554759_dp,  0.009134914466544177_dp,&
            0.013500979005727987_dp,  0.019833495884515773_dp,  0.02897206838249403_dp,  0.042095533039198385_dp,  0.06085215831868936_dp,  0.08753730146299196_dp,&
            0.1253350272549887_dp,  0.17864571781396288_dp,  0.2535290933953418_dp,  0.35830209305587557_dp,  0.5043450664128841_dp,  0.7071902955364795_dp,&
            0.9879995840794562_dp,  1.3755955013642696_dp,  1.9093257812455384_dp,  2.6432950751333295_dp,  3.6531218956627853_dp,  5.048075954994877_dp,&
            6.99678689342338_dp,  9.795906582751762_dp,  14.142088288126018_dp,  3.820307739174523e-05_dp,  9.253211423715926e-05_dp,  0.00015666144789002137_dp,&
            0.00023941035175647392_dp,  0.0003544795957843895_dp,  0.0005220706038556393_dp,  0.0007702567427026573_dp,  0.0011373348215882129_dp,  0.001676302988186351_dp,&
            0.002461551034244103_dp,  0.003597851119996275_dp,  0.005232383925906072_dp,  0.007571020540645877_dp,  0.01090045623553385_dp,  0.01561826230147036_dp,&
            0.022273572546892197_dp,  0.03162200007875945_dp,  0.04469955727258082_dp,  0.06292193738822648_dp,  0.08821770562181855_dp,  0.12320712191071415_dp,&
            0.17144328423198826_dp,  0.23774090556227068_dp,  0.3286348439935499_dp,  0.45304683721268363_dp,  0.6233242933376546_dp,  0.8570306244015214_dp,&
            1.1804560303362448_dp,  1.6366228282940412_dp,  2.3071833124934_dp,  3.3903998854623985_dp,  5.664862602289436_dp]
       aw%aw_erange_matrix(:, 9) = [1.1315989220620849e-05_dp,  6.091484824470965e-05_dp,  0.0001558556348063919_dp,  0.0003078203984115187_dp,  0.0005380983662976303_dp,&
            0.0008826023610964867_dp,  0.0013981206622491853_dp,  0.0021702768627684973_dp,  0.0033248504851841115_dp,  0.005044444967959304_dp,  0.007592519217726472_dp,&
            0.011347339662602948_dp,  0.016849450868861544_dp,  0.024867669342019266_dp,  0.03649043474384647_dp,  0.05325176815491175_dp,  0.07730430050505666_dp,&
            0.1116561161775443_dp,  0.16049387125009024_dp,  0.22962230812156062_dp,  0.3270606812998151_dp,  0.46385106272587334_dp,  0.6551545548992531_dp,&
            0.9217445361800276_dp,  1.2920638357850092_dp,  1.8051264992628424_dp,  2.514796276521399_dp,  3.496590833178514_dp,  4.85984548686659_dp,&
            6.773380959237685_dp,  9.533934166411905_dp,  13.837118014874001_dp,  2.918394667912216e-05_dp,  7.092102053120252e-05_dp,  0.0001208490607819795_dp,&
            0.00018654379226223144_dp,  0.00027989427379719806_dp,  0.0004184502969887145_dp,  0.000626673050017551_dp,  0.0009383293128013663_dp,  0.0014008572107836192_dp,&
            0.002081684144431634_dp,  0.0030767293366374213_dp,  0.004521918459768059_dp,  0.006608939226821494_dp,  0.009606834282132472_dp,  0.013891522205312699_dp,&
            0.0199860217642558_dp,  0.028615075699006714_dp,  0.04077910134384887_dp,  0.057854053870927825_dp,  0.08172606721705969_dp,  0.11497300539733377_dp,&
            0.16111009858027334_dp,  0.22492545113901904_dp,  0.3129477675464925_dp,  0.43412426576959084_dp,  0.6008706467915673_dp,  0.8308678912066612_dp,&
            1.1505656278854524_dp,  1.6031493439397482_dp,  2.270331555451258_dp,  3.3500083706514476_dp,  5.6182166490122984_dp]
       aw%aw_erange_matrix(:, 10) = [7.188720818598015e-06_dp,  3.881661788798954e-05_dp,  9.991514616563082e-05_dp,  0.00019924670807668527_dp,  0.0003530921393375012_dp,&
            0.0005892390933533132_dp,  0.0009519594342766803_dp,  0.0015086954155209336_dp,  0.0023599840025008696_dp,  0.003654238190455114_dp,  0.005609186735480063_dp,&
            0.008542461005020551_dp,  0.01291487453690728_dp,  0.01939133485309362_dp,  0.028926175238232574_dp,  0.04288217331115784_dp,  0.06319585740381348_dp,&
            0.09260618135064007_dp,  0.13496966452445772_dp,  0.19569319485572675_dp,  0.2823266941469131_dp,  0.40537308432315655_dp,  0.5793949454394274_dp,&
            0.8245311284762115_dp,  1.1685945009236132_dp,  1.6500342331912714_dp,  2.322292524496194_dp,  3.260691519334744_dp,  4.574650070688025_dp,&
            6.433299519631953_dp,  9.133561259222846_dp,  13.369520380969645_dp,  1.8552570543955455e-05_dp,  4.5365280395614445e-05_dp,  7.824430449992235e-05_dp,&
            0.000123064841905601_dp,  0.00018917614532398957_dp,  0.00029038908164917863_dp,  0.0004461302902503803_dp,  0.0006838555287975979_dp,  0.001043105615047013_dp,&
            0.0015811894241096348_dp,  0.0023809364846653818_dp,  0.003561415027894291_dp,  0.005292829489029218_dp,  0.007817192602841475_dp,  0.011476902733337355_dp,&
            0.01675409107074492_dp,  0.024324590249295588_dp,  0.035131696353015066_dp,  0.050486674490227534_dp,  0.07220539175013203_dp,  0.10279391074071534_dp,&
            0.14570107833907187_dp,  0.20566478637597496_dp,  0.28919483053487227_dp,  0.40526987079618043_dp,  0.5664074675124494_dp,  0.7904744673132619_dp,&
            1.1041823544223337_dp,  1.5509968448017435_dp,  2.2127606942845244_dp,  3.286839828729209_dp,  5.545336982828468_dp]
       aw%aw_erange_matrix(:, 11) = [5.212842853721373e-06_dp,  2.8216485728556984e-05_dp,  7.298119875698969e-05_dp,  0.00014666687860403872_dp,  0.00026275225618308033_dp,&
            0.0004444091505151155_dp,  0.0007287615881022521_dp,  0.0011728502574399147_dp,  0.0018626816535886259_dp,  0.0029267741621968922_dp,  0.004555909601317652_dp,&
            0.007031533755260647_dp,  0.01076629475072859_dp,  0.016361592669832718_dp,  0.02468887356341755_dp,  0.03700391916268461_dp,  0.055106793600662854_dp,&
            0.08156471559554047_dp,  0.12002134058150862_dp,  0.17562433753378992_dp,  0.2556145691836236_dp,  0.370135979817623_dp,  0.5333479158700739_dp,&
            0.7649561302861829_dp,  1.092336920188996_dp,  1.5535402649553307_dp,  2.201701148128128_dp,  3.111986649331865_dp,  4.39385578004782_dp,&
            6.216650142887412_dp,  8.877446914810541_dp,  13.069399675595056_dp,  1.3460633699481115e-05_dp,  3.307685267989567e-05_dp,  5.7604518102217696e-05_dp,&
            9.195641910083116e-05_dp,  0.00014400755053456355_dp,  0.00022539775962319305_dp,  0.0003526662335951264_dp,  0.0005496236389883362_dp,  0.0008511354053799554_dp,&
            0.0013083814172165737_dp,  0.0019961328695681448_dp,  0.0030229467394267007_dp,  0.00454547537639009_dp,  0.00678848631990991_dp,  0.010072747132319853_dp,&
            0.014853695284454384_dp,  0.021774843424492554_dp,  0.03174125164120884_dp,  0.04602026166587432_dp,  0.06637923482681696_dp,  0.0952736156331887_dp,&
            0.13610397708422503_dp,  0.19356942137149777_dp,  0.27416081565736494_dp,  0.3868712620397728_dp,  0.5442810461868627_dp,  0.7643794009427055_dp,&
            1.0740573000929907_dp,  1.5169810386781961_dp,  2.1751039860564134_dp,  3.2454752973208514_dp,  5.497663131109453_dp]
       aw%aw_erange_matrix(:, 12) = [3.6756703015465298e-06_dp,  1.9954750677579616e-05_dp,  5.191549207164041e-05_dp,  0.00010531730513004429_dp,  0.00019114894568566566_dp,&
            0.0003284345206580502_dp,  0.0005478574351058358_dp,  0.0008970642726998205_dp,  0.001448846563954588_dp,  0.0023134440610659145_dp,  0.00365660754627558_dp,&
            0.005725809627536767_dp,  0.008888016831791598_dp,  0.01368381089343313_dp,  0.020904518142872384_dp,  0.031701561841387796_dp,  0.04774073523681854_dp,&
            0.07141882485947987_dp,  0.10616643258939448_dp,  0.15686955518574938_dp,  0.23045435097627895_dp,  0.33669592809026266_dp,  0.48933536087816165_dp,&
            0.7076244370403264_dp,  1.0184762856769092_dp,  1.4595108245228903_dp,  2.0835241856266555_dp,  2.965501956804563_dp,  4.214929989807013_dp,&
            6.001367447071038_dp,  8.622077664029446_dp,  12.76932426608392_dp,  9.497600314818484e-06_dp,  2.347760247868802e-05_dp,  4.136863276834202e-05_dp,&
            6.721947207800727e-05_dp,  0.00010755652914793868_dp,  0.00017204310109274583_dp,  0.00027461020521950513_dp,  0.00043573410701580053_dp,  0.0006859083122114215_dp,&
            0.0010704829571126738_dp,  0.0016564711186942442_dp,  0.0025422076964986724_dp,  0.003871039755848531_dp,  0.005850643295798488_dp,  0.008780144703142966_dp,&
            0.013088020403816946_dp,  0.01938482007375741_dp,  0.02853620387815694_dp,  0.04176374049584569_dp,  0.06078358076425594_dp,  0.087996854859105_dp,&
            0.12675113895160917_dp,  0.18170115882627896_dp,  0.2593129035941683_dp,  0.3685886307380454_dp,  0.5221684374385586_dp,  0.7381657526850111_dp,&
            1.0436599990100175_dp,  1.4825352832265704_dp,  2.136878793265541_dp,  3.2034437811357703_dp,  5.449257489548676_dp]
       aw%aw_erange_matrix(:, 13) = [2.0394837579119305e-06_dp,  1.1136428689231387e-05_dp,  2.9311026954581983e-05_dp,  6.057395938117077e-05_dp,  0.00011273361636596131_dp,&
            0.00019945262227642557_dp,  0.00034307020413915784_dp,  0.0005789986244949678_dp,  0.0009626136389075716_dp,  0.0015797047702976012_dp,  0.0025620328180390273_dp,&
            0.00411027194192014_dp,  0.006527602589322277_dp,  0.010268581467814463_dp,  0.016009798955308914_dp,  0.02475143770962476_dp,  0.03796243042890685_dp,&
            0.057786831913653906_dp,  0.0873357482301367_dp,  0.1310983577206706_dp,  0.19551814727621097_dp,  0.2897978980472306_dp,  0.4270216153720778_dp,&
            0.6257183147230295_dp,  0.9120522680792509_dp,  1.3229345707400562_dp,  1.9105866829861018_dp,  2.7496652071687744_dp,  3.9496659363104385_dp,&
            5.680486965184195_dp,  8.239732353995374_dp,  12.318415719148392_dp,  5.276628830314947e-06_dp,  1.3196668481729192e-05_dp,  2.379458469172375e-05_dp,&
            3.999888381841533e-05_dp,  6.655277116013134e-05_dp,  0.00011053727387478837_dp,  0.00018248747181848964_dp,  0.00029842809341232193_dp,  0.0004828462223872788_dp,&
            0.0007729214376067976_dp,  0.001224631278196218_dp,  0.001921577765363735_dp,  0.0029876866735399437_dp,  0.0046053732161687655_dp,  0.007041380807008615_dp,&
            0.010683336283217822_dp,  0.016091204665789092_dp,  0.02406937359680715_dp,  0.03576720002987578_dp,  0.05281872257231177_dp,  0.07753624384014632_dp,&
            0.11317828438072734_dp,  0.16432147566237754_dp,  0.2373819281103962_dp,  0.34136408980953675_dp,  0.48899079151871744_dp,  0.6985643332266451_dp,&
            0.9974643386084848_dp,  1.4299370040199422_dp,  2.0783190576822954_dp,  3.138964327630411_dp,  5.375073752754771_dp]
       aw%aw_erange_matrix(:, 14) = [1.2029417634752254e-06_dp,  6.609895643735541e-06_dp,  1.761886326860991e-05_dp,  3.714701435066809e-05_dp,  7.09606454750247e-05_dp,&
            0.00012924586800944736_dp,  0.0002289177620104591_dp,  0.00039735575327091586_dp,  0.0006783049971089168_dp,  0.001140918627613451_dp,  0.0018934100817969098_dp,&
            0.0031034810098270562_dp,  0.0050286516433828725_dp,  0.008060966671683137_dp,  0.012792436500279173_dp,  0.020110206423758527_dp,  0.03133410211483852_dp,&
            0.04841425570428367_dp,  0.07421348496567018_dp,  0.11290867950409522_dp,  0.1705586323733128_dp,  0.2559040285419376_dp,  0.38149111837813177_dp,&
            0.5652486706311622_dp,  0.8327086526705059_dp,  1.2201713489652732_dp,  1.7793482729282073_dp,  2.5845851027859807_dp,  3.745354303892851_dp,&
            5.4318267387256425_dp,  7.941927006849686_dp,  11.965772395891321_dp,  3.1166015808204985e-06_dp,  7.893584647537054e-06_dp,  1.4590421523143668e-05_dp,&
            2.540166835137321e-05_dp,  4.38866273658644e-05_dp,  7.544235506820306e-05_dp,  0.00012836853819339647_dp,  0.0002156510284757784_dp,  0.00035755049600582473_dp,&
            0.0005853791673295657_dp,  0.0009470589489445969_dp,  0.0015152726995104847_dp,  0.0023993396733646157_dp,  0.0037624022005128713_dp,  0.005846143988201738_dp,&
            0.009006130762738474_dp,  0.01376205648174844_dp,  0.02086880821287917_dp,  0.03141648831640661_dp,  0.04697057987136505_dp,  0.06976767926454802_dp,&
            0.10298830876584288_dp,  0.1511376630263013_dp,  0.2205811788480397_dp,  0.320313759532162_dp,  0.4631152216015924_dp,  0.6674360776688134_dp,&
            0.9609048344557064_dp,  1.3880820200304151_dp,  2.031544618845653_dp,  3.087377771089118_dp,  5.31578682998986_dp]
       aw%aw_erange_matrix(:, 15) = [5.514768391533348e-07_dp,  3.0661463050989084e-06_dp,  8.370607042832422e-06_dp,  1.830793322995512e-05_dp,  3.65866489044243e-05_dp,&
            6.987140555359488e-05_dp,  0.00012955245746241426_dp,  0.00023469156977383316_dp,  0.00041673066301572737_dp,  0.0007268469188482661_dp,  0.0012473021552990137_dp,&
            0.002108784237240361_dp,  0.0035166643543472435_dp,  0.005790416868049286_dp,  0.009422327737378779_dp,  0.015164272167993003_dp,  0.02415507435487594_dp,&
            0.038106181361843135_dp,  0.05957064947730835_dp,  0.09233052534676135_dp,  0.14195168557080626_dp,  0.2165746550862792_dp,  0.3280373760158957_dp,&
            0.493465990037226_dp,  0.7375325370826453_dp,  1.0956894156446793_dp,  1.6189196862917987_dp,  2.381098691427177_dp,  3.4916207819996172_dp,&
            5.1210060943387665_dp,  7.5676555925110085_dp,  11.520669806145102_dp,  1.4324492833187423e-06_dp,  3.714945365379106e-06_dp,  7.187993199450151e-06_dp,&
            1.3287667789384625e-05_dp,  2.435221365340668e-05_dp,  4.406786544877652e-05_dp,  7.839477316844559e-05_dp,  0.00013700432533564813_dp,  0.00023541888373746023_dp,&
            0.0003982417143724247_dp,  0.0006640223423546365_dp,  0.0010925276402432504_dp,  0.0017755203327472222_dp,  0.0028526152739405167_dp,  0.0045344377578051915_dp,&
            0.007136218994535957_dp,  0.011126225466246156_dp,  0.01719515628222213_dp,  0.026355033830470446_dp,  0.04007940751399853_dp,  0.06050127009572156_dp,&
            0.09069160330701329_dp,  0.13505126572297815_dp,  0.19986523979796597_dp,  0.2940997147198093_dp,  0.4305938358212778_dp,  0.6279833290723216_dp,&
            0.9142278655051918_dp,  1.3343266294937697_dp,  1.971223529289826_dp,  3.0207289388880563_dp,  5.239269717116346_dp]
       aw%aw_erange_matrix(:, 16) = [2.6346628583772034e-07_dp,  1.4870682611802608e-06_dp,  4.18499777462391e-06_dp,  9.565754079998203e-06_dp,  2.0095438915823317e-05_dp,&
            4.030712638205273e-05_dp,  7.820310873891258e-05_dp,  0.0001476189063429688_dp,  0.00027206125557350966_dp,  0.0004908276742423054_dp,  0.0008686422030538531_dp,&
            0.0015106576888148007_dp,  0.0025855700579017434_dp,  0.0043608859416256075_dp,  0.007256247781115456_dp,  0.011923381221441286_dp,  0.01936500732097008_dp,&
            0.031110395247887928_dp,  0.049472724953768164_dp,  0.07792391423249093_dp,  0.12163719777397036_dp,  0.18826820696011917_dp,  0.2890741916661454_dp,&
            0.44051299024821017_dp,  0.6665282145336267_dp,  1.0018392157983365_dp,  1.4967815518731842_dp,  2.2247902473537535_dp,  3.295151802440413_dp,&
            4.87866127574686_dp,  7.274154425483326_dp,  11.170031678415262_dp,  6.865691087424058e-07_dp,  1.8349687179642652e-06_dp,  3.75363535847515e-06_dp,&
            7.406302503889748e-06_dp,  1.4385998863803715e-05_dp,  2.7332856632656454e-05_dp,  5.070293872960844e-05_dp,  9.194731882902252e-05_dp,  0.00016333066755487376_dp,&
            0.0002847458523672339_dp,  0.0004880328440654819_dp,  0.0008235400684252141_dp,  0.001370000313997599_dp,  0.002249268185463258_dp,  0.003648135807576948_dp,&
            0.005850384133560756_dp,  0.009283540163912576_dp,  0.01458663604211278_dp,  0.022707798579110823_dp,  0.03504400012572891_dp,  0.05364018073213217_dp,&
            0.08147187176803067_dp,  0.12284570471579664_dp,  0.18396885882280342_dp,  0.27376977104743744_dp,  0.4051220237008429_dp,  0.5968036286827566_dp,&
            0.8770477402799991_dp,  1.2912341021707199_dp,  1.9226515113345246_dp,  2.9669527288987725_dp,  5.177596037459169_dp]
       aw%aw_erange_matrix(:, 17) = [7.696764691509552e-08_dp,  4.5183075501644766e-07_dp,  1.3712733197245156e-06_dp,  3.452398063116915e-06_dp,  7.99119238288884e-06_dp,&
            1.7499890158507848e-05_dp,  3.668225172908821e-05_dp,  7.411367023252019e-05_dp,  0.0001450609740104884_dp,  0.0002761333471939282_dp,  0.0005128366382617873_dp,&
            0.0009316733536716902_dp,  0.0016592776527650684_dp,  0.0029023227583525545_dp,  0.004993762465329047_dp,  0.008463615785356367_dp,  0.014146318917564842_dp,&
            0.02334212828021178_dp,  0.03805782856888779_dp,  0.0613630042242151_dp,  0.0979136503362365_dp,  0.15471676332581316_dp,  0.24224053787692895_dp,&
            0.3760196167953944_dp,  0.5789730031642062_dp,  0.8847668378302425_dp,  1.3427798213464026_dp,  2.0257631969638736_dp,  3.042786371879784_dp,&
            4.564995894383141_dp,  6.891880055654338_dp,  10.711074682469496_dp,  2.0224960490304485e-07_dp,  5.837519191389681e-07_dp,  1.3534097826701593e-06_dp,&
            3.0163324999016133e-06_dp,  6.464297707355912e-06_dp,  1.331006273228438e-05_dp,  2.6438493784462622e-05_dp,  5.089377871159909e-05_dp,  9.53135908336066e-05_dp,&
            0.00017421956824809404_dp,  0.0003116271902197154_dp,  0.0005466663994455823_dp,  0.0009422361350401303_dp,  0.0015981946213501235_dp,  0.0026712728244169207_dp,&
            0.004404875333973468_dp,  0.00717331310959028_dp,  0.011546952692401411_dp,  0.018387490277830258_dp,  0.028986360940377882_dp,  0.04526460807725196_dp,&
            0.07006006488175816_dp,  0.10753871373789183_dp,  0.16378446149863707_dp,  0.24765288245177752_dp,  0.3720420890669611_dp,  0.5559074354965244_dp,&
            0.8278548563568215_dp,  1.2338117262467065_dp,  1.8576021658437778_dp,  2.8947650724141107_dp,  5.094894347251689_dp]
       aw%aw_erange_matrix(:, 18) = [1.5580960122828055e-08_dp,  1.0192299261551187e-07_dp,  3.657511654714275e-07_dp,  1.0874230387694592e-06_dp,  2.8971440093678986e-06_dp,&
            7.1231112287328e-06_dp,  1.6440376252868036e-05_dp,  3.6035911594207885e-05_dp,  7.564822282610485e-05_dp,  0.0001530604093750471_dp,  0.0002999674552596099_dp,&
            0.0005716626169231291_dp,  0.0010627819854068627_dp,  0.0019325381417571955_dp,  0.0034446462658281212_dp,  0.006029758821968773_dp,  0.010382043759726605_dp,&
            0.017607085158632928_dp,  0.02944628039743525_dp,  0.04861435988540585_dp,  0.07930297685700823_dp,  0.12792652074627453_dp,  0.20421936908257549_dp,&
            0.3228415686086713_dp,  0.5057217947095013_dp,  0.785484422991801_dp,  1.2105330871831885_dp,  1.852886865611021_dp,  2.8213315422092657_dp,&
            4.2873111269979765_dp,  6.550982976987269_dp,  10.299461972660755_dp,  4.192785461816036e-08_dp,  1.4682586910958697e-07_dp,  4.248767433871547e-07_dp,&
            1.117972412187448e-06_dp,  2.71251708562224e-06_dp,  6.166664608205133e-06_dp,  1.3299698724334126e-05_dp,  2.7458360437668858e-05_dp,  5.463565964132874e-05_dp,&
            0.00010531494807998692_dp,  0.0001974602213268062_dp,  0.0003612942851684024_dp,  0.0006468325897487988_dp,  0.001135618706607295_dp,  0.0019588032650003236_dp,&
            0.003324710089253406_dp,  0.005560469162003084_dp,  0.009174344598623402_dp,  0.014948285309521006_dp,  0.024074306780617383_dp,  0.03835406948576763_dp,&
            0.060489181968959096_dp,  0.0945015966761899_dp,  0.1463417320393485_dp,  0.22477371348415012_dp,  0.34269334477792374_dp,  0.5192018690921564_dp,&
            0.7832512032052583_dp,  1.1813099794431254_dp,  1.7977731364021527_dp,  2.828182155100705_dp,  5.018697893750251_dp]
       aw%aw_erange_matrix(:, 19) = [1.4796558719757977e-09_dp,  1.6277762282636202e-08_dp,  8.615102100471783e-08_dp,  3.3251166768869524e-07_dp,  1.064831343891703e-06_dp,&
            3.0042406722730696e-06_dp,  7.722404719947578e-06_dp,  1.8469628120696557e-05_dp,  4.168720676495336e-05_dp,  8.969407716668371e-05_dp,  0.00018534670944540425_dp,&
            0.00036995434452620024_dp,  0.0007164776276992899_dp,  0.0013511756947926401_dp,  0.0024885840769342527_dp,  0.004487272639985271_dp,  0.007937634518341114_dp,&
            0.013798542180092185_dp,  0.023607852053926744_dp,  0.03980353161167122_dp,  0.06620914631345257_dp,  0.10876176019481658_dp,  0.17659514989891748_dp,&
            0.28364164912494744_dp,  0.45099127302133024_dp,  0.7103683461225575_dp,  1.109312567732835_dp,  1.7191679096665389_dp,  2.648421848786194_dp,&
            4.06873544538514_dp,  6.280858212318488_dp,  9.971616727521422_dp,  4.7301698039073534e-09_dp,  3.1078236876760897e-08_dp,  1.2722293347158736e-07_dp,&
            4.135773629354279e-07_dp,  1.1643319139692173e-06_dp,  2.9633030798047164e-06_dp,  6.990659700961397e-06_dp,  1.5534394085699292e-05_dp,  3.28792052583233e-05_dp,&
            6.681670495514343e-05_dp,  0.00013116076656301764_dp,  0.0002498651223560153_dp,  0.0004636589258144925_dp,  0.0008405899279454757_dp,  0.0014925644239063773_dp,&
            0.0026009920182698686_dp,  0.004456120959091793_dp,  0.00751677674389606_dp,  0.012500261250050913_dp,  0.020516489982978243_dp,  0.03326657617193578_dp,&
            0.053334802223457274_dp,  0.08461553683522265_dp,  0.13293582624456382_dp,  0.20696680596446562_dp,  0.31958257452387945_dp,  0.4899879264178536_dp,&
            0.7474157763110646_dp,  1.1388011986618074_dp,  1.7490627041810483_dp,  2.7738272660229994_dp,  4.9565545529977175_dp]
    case(34)
       aw%energy_range(:) = [9649.0_dp, 15161.0_dp, 29986.0_dp, 49196.0_dp, 109833.0_dp, 276208.0_dp, 852991.0_dp]
       aw%aw_erange_matrix(:, 1) = [3.284029233271286e-05_dp,  0.0001753443602861097_dp,  0.0004416382833363001_dp,  0.0008508530310865113_dp,  0.001435500605855746_dp,&
            0.0022474948219129297_dp,  0.003366173371324329_dp,  0.004907942966760921_dp,  0.007037748206337417_dp,  0.009983831117905245_dp,  0.014057973412231587_dp,&
            0.019683414844193538_dp,  0.02743271515183254_dp,  0.03807838736551588_dp,  0.052660091008446035_dp,  0.07257342975355677_dp,  0.09968697353631621_dp,&
            0.13649612296545818_dp,  0.18632500099011615_dp,  0.2535908886542759_dp,  0.3441500984275648_dp,  0.46575004766668_dp,  0.6286204407593304_dp,&
            0.8462483877785438_dp,  1.1364010501626234_dp,  1.5224916494827863_dp,  2.035445381255887_dp,  2.716346186249304_dp,  3.620421040738206_dp,&
            4.823581438788581_dp,  6.434509907627798_dp,  8.620766164346367_dp,  11.679090118639257_dp,  16.31432816585662_dp,  8.4538052355896e-05_dp,&
            0.00020209318539166847_dp,  0.00033361605486797254_dp,  0.0004901319572083779_dp,  0.0006876803644732437_dp,  0.0009493703556928221_dp,  0.0013072005549165328_dp,&
            0.0018035890085942577_dp,  0.0024937295684409103_dp,  0.0034499397324379713_dp,  0.00476826036404788_dp,  0.0065772141488150675_dp,  0.009049030640506878_dp,&
            0.012414126010019873_dp,  0.016979951815415794_dp,  0.02315561592334051_dp,  0.031484041042937_dp,  0.04268391545844468_dp,  0.05770434764295233_dp,&
            0.07779602216248598_dp,  0.10460388481675117_dp,  0.14028819232415718_dp,  0.18768361957484037_dp,  0.2505110517811862_dp,  0.33366599536844854_dp,&
            0.4436264870131247_dp,  0.5890644515470643_dp,  0.7818386670916874_dp,  1.0387776399865274_dp,  1.3852730956246349_dp,  1.8635541689521873_dp,&
            2.5552790973254864_dp,  3.661647560244881_dp,  5.979097365202448_dp]
       aw%aw_erange_matrix(:, 2) = [2.2094687262226044e-05_dp,  0.00011816317101145329_dp,  0.00029854362986100777_dp,  0.0005779739014899124_dp,  0.0009819112670486493_dp,&
            0.0015516034173501654_dp,  0.0023507187818878183_dp,  0.0034732685500535195_dp,  0.00505334279364789_dp,  0.007278229326144087_dp,  0.010406819589614869_dp,&
            0.014795161155330522_dp,  0.020931286923915934_dp,  0.029482154546951286_dp,  0.04135653654321724_dp,  0.05778898092867048_dp,  0.08045158704880996_dp,&
            0.11160243319073283_dp,  0.15428220578421106_dp,  0.2125741178044686_dp,  0.2919468573365305_dp,  0.3997065271731107_dp,  0.545592087818525_dp,&
            0.7425611332686428_dp,  1.0078318394632224_dp,  1.3642789958888877_dp,  1.8423416665523202_dp,  2.48272174251732_dp,  3.34042303033154_dp,&
            4.49133411535853_dp,  6.044309582675748_dp,  8.167153038233048_dp,  11.156471603344267_dp,  15.714736908294974_dp,  5.689777248462453e-05_dp,&
            0.0001364647878286462_dp,  0.0002266959149215116_dp,  0.0003363227347021138_dp,  0.0004783149170860289_dp,  0.0006715498663212837_dp,  0.0009421913049362618_dp,&
            0.0013250426248297155_dp,  0.0018660179676916586_dp,  0.00262648268480602_dp,  0.003689465255982615_dp,  0.005167783411766887_dp,  0.007214591621086576_dp,&
            0.010037222243044243_dp,  0.013915457617770334_dp,  0.01922566776214666_dp,  0.02647264919678102_dp,  0.03633153858613665_dp,  0.049702887216291285_dp,&
            0.06778493328431712_dp,  0.09216841458681874_dp,  0.12496114099205212_dp,  0.1689524368948172_dp,  0.22783243637559455_dp,  0.3064902609254688_dp,&
            0.411433435566918_dp,  0.5514107558972641_dp,  0.7384129467740507_dp,  0.9894523508058952_dp,  1.3301301240547123_dp,  1.8028378573543902_dp,&
            2.489168614889156_dp,  3.5894672490279578_dp,  5.895310588871672_dp]
       aw%aw_erange_matrix(:, 3) = [1.2132019239369146e-05_dp,  6.506554392462203e-05_dp,  0.00016528151320033429_dp,  0.00032271987439700723_dp,  0.0005549904907647636_dp,&
            0.0008912186380916402_dp,  0.0013769333304078336_dp,  0.00207998381188429_dp,  0.0030983248733590404_dp,  0.004571097441508312_dp,  0.006694456763746173_dp,&
            0.009743704668641925_dp,  0.0141037713916676_dp,  0.02031087384545962_dp,  0.029109193553804015_dp,  0.04152772200656733_dp,  0.05898411878582171_dp,&
            0.08342465247911345_dp,  0.11751221427114543_dp,  0.1648782324800013_dp,  0.23045938623741916_dp,  0.3209467823295079_dp,  0.44538447707509593_dp,&
            0.6159672665711226_dp,  0.8491072823533072_dp,  1.1668710896432313_dp,  1.5989474740844776_dp,  2.1854242653839444_dp,  2.9809152725739025_dp,&
            4.061217864274067_dp,  5.535407811609172_dp,  7.571672639611632_dp,  10.466564989305379_dp,  14.91952959384994_dp,  3.126217597574477e-05_dp,&
            7.540587737410632e-05_dp,  0.00012663926218204873_dp,  0.00019110769108438634_dp,  0.00027820581560489743_dp,  0.000401742249839652_dp,  0.0005808693019056328_dp,&
            0.0008413761481210849_dp,  0.0012182358494751703_dp,  0.001759637722089599_dp,  0.00253246317609668_dp,  0.003629502407336348_dp,  0.00517906799440672_dp,&
            0.0073579025027302025_dp,  0.01040852585005309_dp,  0.014662503493069159_dp,  0.020571571811010672_dp,  0.028749161300949972_dp,  0.04002565337439286_dp,&
            0.05552176300115481_dp,  0.07674587007131725_dp,  0.10572312939601615_dp,  0.14516718239506418_dp,  0.19871016099008434_dp,  0.2712154487579484_dp,&
            0.3692151806287191_dp,  0.5015525805409441_dp,  0.6803982020135383_dp,  0.9230308920142081_dp,  1.2553734442165063_dp,  1.7200955901449364_dp,&
            2.3987730554757754_dp,  3.4906629653715_dp,  5.780815874478559_dp]
       aw%aw_erange_matrix(:, 4) = [7.845531705064765e-06_dp,  4.217579566703759e-05_dp,  0.0001076255406955941_dp,  0.0002116704074840506_dp,  0.0003678026352786526_dp,&
            0.000598637239972256_dp,  0.000939800791185946_dp,  0.0014447771966513766_dp,  0.002191627535252642_dp,  0.0032927972200630018_dp,  0.004909216880554122_dp,&
            0.007270144108734435_dp,  0.010700753084513917_dp,  0.015660261119126325_dp,  0.02279439226673758_dp,  0.033007300151380946_dp,  0.04755981744820865_dp,&
            0.06820320985515761_dp,  0.09736066855556012_dp,  0.13837281589809153_dp,  0.19582885861759528_dp,  0.2760121868265214_dp,  0.3874989458973457_dp,&
            0.5419617440352286_dp,  0.7552508475941064_dp,  1.048857678610048_dp,  1.4519234318259462_dp,  2.004071631268318_dp,  2.759600998924181_dp,&
            3.7942044073242056_dp,  5.217087596451319_dp,  7.196718451225344_dp,  10.029695823161129_dp,  14.413626849760721_dp,  2.022742080929783e-05_dp,&
            4.902127023830298e-05_dp,  8.308838409631613e-05_dp,  0.00012719755797047105_dp,  0.00018877141608928316_dp,  0.00027875729779708194_dp,  0.00041237481957666014_dp,&
            0.0006104274314639044_dp,  0.0009017824485847016_dp,  0.0013270526990110587_dp,  0.0019435429478184875_dp,  0.0028318806888766317_dp,  0.004105011117474572_dp,&
            0.0059204430431561185_dp,  0.008496895591229079_dp,  0.012136858469163465_dp,  0.017257067102487582_dp,  0.02442954195917895_dp,  0.03443669419909579_dp,&
            0.04834513188831433_dp,  0.06760433082481605_dp,  0.09417845588835595_dp,  0.13072271761835513_dp,  0.18082056576074482_dp,  0.2493066841224739_dp,&
            0.3427177893721846_dp,  0.46995019394607707_dp,  0.6432897947695325_dp,  0.8801980582788952_dp,  1.2068306280775711_dp,  1.6660767737909483_dp,&
            2.3395502248176463_dp,  3.4258517245052684_dp,  5.7058363778559205_dp]
       aw%aw_erange_matrix(:, 5) = [3.865603619912879e-06_dp,  2.087430645188353e-05_dp,  5.3738065099243575e-05_dp,  0.00010718454630525007_dp,  0.0001900012319697737_dp,&
            0.00031718955789873134_dp,  0.0005126537329759226_dp,  0.0008128168395638415_dp,  0.0012719920833382853_dp,  0.001970384125983329_dp,  0.0030256963224515113_dp,&
            0.00460969356562721_dp,  0.006971643847773047_dp,  0.010471317877795917_dp,  0.01562522911317097_dp,  0.023171141484273704_dp,  0.034157679556524934_dp,&
            0.05006830174859498_dp,  0.0729921440414609_dp,  0.1058585802344639_dp,  0.1527581466742932_dp,  0.2193802673426605_dp,  0.31360877991591823_dp,&
            0.4463309353744743_dp,  0.6325368438825444_dp,  0.8928196332263253_dp,  1.2554443726415525_dp,  1.759267106284206_dp,  2.458035348630705_dp,&
            3.427215114290294_dp,  4.77616847618911_dp,  6.673804689443379_dp,  9.416907604833742_dp,  13.70063546514659_dp,  9.976454765718753e-06_dp,&
            2.4397956666150202e-05_dp,  4.2091633541468636e-05_dp,  6.622948123983615e-05_dp,  0.00010186030376456873_dp,  0.00015644233845744574_dp,  0.0002404690130604266_dp,&
            0.0003687782406776715_dp,  0.0005627493820533361_dp,  0.0008533803933831028_dp,  0.0012854870718438482_dp,  0.0019235107198793406_dp,  0.002859592750775737_dp,&
            0.004224781787747757_dp,  0.0062045295634641385_dp,  0.009060029072853486_dp,  0.0131574822853581_dp,  0.019008095244250185_dp,  0.02732254096178089_dp,&
            0.03908488812876366_dp,  0.055652686745107795_dp,  0.07889222315477536_dp,  0.11136127365662019_dp,  0.15655676737113142_dp,  0.21925337945032786_dp,&
            0.30597553969409247_dp,  0.42568066454670456_dp,  0.5908146138910935_dp,  0.8191119697759165_dp,  1.137096600179962_dp,  1.5880319126786595_dp,&
            2.2536634913094686_dp,  3.331728541732483_dp,  5.597118014202835_dp]
       aw%aw_erange_matrix(:, 6) = [1.7147362511403065e-06_dp,  9.319802731855972e-06_dp,  2.4302616807482706e-05_dp,  4.948261860530359e-05_dp,  9.026404229610094e-05_dp,&
            0.0001560250715046401_dp,  0.00026193159643869907_dp,  0.0004316317370034808_dp,  0.0007014310228413562_dp,  0.0011265811323026616_dp,  0.001790538099368303_dp,&
            0.002818442710487939_dp,  0.004396613985562227_dp,  0.006800572618267542_dp,  0.01043510374307695_dp,  0.015891226467704685_dp,  0.02402679235817231_dp,&
            0.03607995947798214_dp,  0.05382821091103021_dp,  0.07981021388476216_dp,  0.11763406509273454_dp,  0.17240391938168603_dp,  0.25130849721857534_dp,&
            0.36443086121519086_dp,  0.525861591666699_dp,  0.7552321345423689_dp,  1.0798433566303969_dp,  1.5376756421659512_dp,  2.1818096583688225_dp,&
            3.087384050821325_dp,  4.363863206811109_dp,  6.180624421988004_dp,  8.83477361053772_dp,  13.019312520088521_dp,  4.43185996007463e-06_dp,&
            1.0980758773489005e-05_dp,  1.943725634861249e-05_dp,  3.1800270744211854e-05_dp,  5.129667367655266e-05_dp,  8.27095880626353e-05_dp,  0.0001329766243411858_dp,&
            0.00021237174266944332_dp,  0.0003362840713927107_dp,  0.0005277050042553842_dp,  0.0008207467241987074_dp,  0.001265656823748127_dp,  0.001935949524221289_dp,&
            0.002938498181312597_dp,  0.004427745609890722_dp,  0.00662561356009038_dp,  0.009849268137973679_dp,  0.01454967083492188_dp,  0.021364880820575197_dp,&
            0.031193465982641635_dp,  0.04529525929852885_dp,  0.06542926430703719_dp,  0.09404211859223464_dp,  0.13452588222462222_dp,  0.19157264817324873_dp,&
            0.27166955823017896_dp,  0.3838116245690371_dp,  0.5405893273969905_dp,  0.7600124959108089_dp,  1.0690028998164918_dp,  1.5112619233396074_dp,&
            2.168763561959426_dp,  3.238506003394989_dp,  5.489633460260332_dp]
       aw%aw_erange_matrix(:, 7) = [6.35749973501056e-07_dp,  3.492169089246581e-06_dp,  9.302356973224121e-06_dp,  1.9592566972831844e-05_dp,  3.7377729183938204e-05_dp,&
            6.798117415794394e-05_dp,  0.00012023568563760358_dp,  0.0002084236297872551_dp,  0.00035534106272681936_dp,  0.0005969900768905776_dp,  0.0009896587375544644_dp,&
            0.001620507511517086_dp,  0.0026232770839453665_dp,  0.004201426984363522_dp,  0.006661986770200809_dp,  0.010464757550535931_dp,  0.01629338409394917_dp,&
            0.025157418466823473_dp,  0.038538075135755495_dp,  0.05859528378548497_dp,  0.088460352930584_dp,  0.13264771717146925_dp,  0.19763178422753508_dp,&
            0.2926522443111407_dp,  0.4308357751366649_dp,  0.6307586891808089_dp,  0.9186346830441839_dp,  1.3314221217344053_dp,  1.9213813610912545_dp,&
            2.76319200354934_dp,  3.9663504041897766_dp,  5.700732751780247_dp,  8.263919165830904_dp,  12.346999008174697_dp,  1.646994148539095e-06_dp,&
            4.168704436960474e-06_dp,  7.695536879782202e-06_dp,  1.3374173961844378e-05_dp,  2.3064311988245763e-05_dp,  3.958321376425686e-05_dp,  6.725628425988863e-05_dp,&
            0.00011284251346096221_dp,  0.0001868773135187132_dp,  0.0003056293856662244_dp,  0.0004939782736171523_dp,  0.0007896314087630558_dp,  0.0012492643080895956_dp,&
            0.001957402304692391_dp,  0.0030391875844591482_dp,  0.004678623430285177_dp,  0.007144500300310407_dp,  0.010827043576140467_dp,  0.016289458703073287_dp,&
            0.024340088611569204_dp,  0.03613298947254673_dp,  0.05330759009262987_dp,  0.07818208570144934_dp,  0.11402099109647866_dp,  0.16540631928634394_dp,&
            0.23875782212611196_dp,  0.3430802599354601_dp,  0.49109150628018977_dp,  0.7010818269759505_dp,  1.0004112023843146_dp,  1.4333015739126787_dp,&
            2.0820719815510036_dp,  3.1430997613534135_dp,  5.379828456565977_dp]
       aw%aw_erange_matrix(:, 8) = [1.117489994479298e-09_dp,  1.0437767934573854e-08_dp,  5.084528505414382e-08_dp,  1.872213850467172e-07_dp,  5.813350591395648e-07_dp,&
            1.6041946888512065e-06_dp,  4.054564459005386e-06_dp,  9.568204612679587e-06_dp,  2.1360561665920905e-05_dp,  4.553900601252074e-05_dp,  9.336828090817695e-05_dp,&
            0.00018510270750313838_dp,  0.0003563527131882449_dp,  0.0006684922084643366_dp,  0.0012254246986170441_dp,  0.002200236453707954_dp,  0.0038770557829877106_dp,&
            0.006716063190006149_dp,  0.011453417554768287_dp,  0.019253386304452922_dp,  0.0319378960087355_dp,  0.052330033425615166_dp,  0.0847641054822857_dp,&
            0.13583764990140365_dp,  0.21551319196591645_dp,  0.3387243798072503_dp,  0.527711765290698_dp,  0.8154303015314811_dp,  1.2505960123470905_dp,&
            1.9054661547111273_dp,  2.8889237961629872_dp,  4.37232400643935_dp,  6.65560984092506_dp,  10.426038695834771_dp,  3.3457340053937633e-09_dp,&
            1.873386519501546e-08_dp,  7.20396391424945e-08_dp,  2.259124367219339e-07_dp,  6.20700398368113e-07_dp,  1.5516519034576341e-06_dp,  3.6098773524770243e-06_dp,&
            7.932321947059267e-06_dp,  1.6633942623129364e-05_dp,  3.353876533808818e-05_dp,  6.539287755778778e-05_dp,  0.00012384295496778607_dp,  0.00022861478182158466_dp,&
            0.0004125495803926801_dp,  0.0007294839100515848_dp,  0.0012664381661580966_dp,  0.002162272032202245_dp,  0.003635959152010226_dp,  0.006029053426126615_dp,&
            0.009868936147464018_dp,  0.015962272528537477_dp,  0.025532104472748035_dp,  0.040417626126145245_dp,  0.06336363493503006_dp,  0.09843823938908386_dp,&
            0.1516353587304579_dp,  0.23175010955983177_dp,  0.35168177852781385_dp,  0.5304884039781542_dp,  0.7970145382697631_dp,  1.197557199061726_dp,&
            1.8163260289376923_dp,  2.848850144582029_dp,  5.042342061181277_dp]
    end select

  end subroutine set_aw_array_tau

  !> \brief Unpacks the minimax coefficients for the desired energy range
  !! @param[in] k - size of the grid
  !! @param[in] e_range - the selected energy range
  !! @param[inout] ac_we - vector containing coefficients and weights
  !! @param[out] ierr - error code
  subroutine get_points_weights_tau(grid_size, e_range, ac_we, ierr)
    integer, intent(in)                                    :: grid_size
    real(kind=dp), intent(in)                              :: e_range
    real(kind=dp), dimension(2*grid_size), intent(inout)   :: ac_we
    integer, intent(out)                                   :: ierr

    !> Internal variables
    integer                                                :: kloc, bup
    type(er_aw_aux)                                        :: aw
    real(kind=dp)                                          :: e_ratio

    !> Begin work
    ierr = 0
    if (.not. any(tau_npoints_supported == grid_size)) then
       ierr = 1
       _REGISTER_EXC("The grid size you chose is not available.")
       return
    end if

    ! Find location of grid size
    kloc = findloc(tau_npoints_supported, grid_size, 1)
    bup = energy_ranges_grids(kloc)

    ! Allocate and set type elements
    allocate(aw%energy_range(bup))
    allocate(aw%aw_erange_matrix(2*grid_size, bup+1))
    call set_aw_array_tau(grid_size, aw)

    ! Get coefficients and weights
    e_ratio = 1.0_dp
    call aw%get_coeff_weight(grid_size, bup, e_range, ac_we, e_ratio)
    ac_we(:) = ac_we(:) * e_ratio

    ! Deallocate
    deallocate(aw%energy_range)
    deallocate(aw%aw_erange_matrix)

  end subroutine get_points_weights_tau
  
end module minimax_tau
