! ***************************************************************************************************
!  Copyright (C) 2020-2023 GreenX library                                                          
!  This file is distributed under the terms of the APACHE2 License.                                 
!                                                                                                   
! ***************************************************************************************************
!> \brief This module contains the tabulated minimax coefficients that approximate                  
!>        1/x ~ 1/pi \sum_{i} w_i x^2/(x^2+a_i^2)^2 with x \in [1:rc]                                     
!> The arrays containing the coefficients and weights are stored in the `er_aw_aux` derived type.
!> To extend this module, add the new entries to `omega_npoints_supported`, `energy_ranges_grids`,
!> and fill the corresponding arrays in the derived type.
!> reference: [https://doi.org/10.1021/ct5001268](https://doi.org/10.1021/ct5001268)
!> reference: [https://doi.org/10.1103/PhysRevB.94.165109](https://doi.org/10.1103/PhysRevB.94.165109)
! ***************************************************************************************************
module minimax_omega
#include "gx_common.h"
  use kinds,          only: dp
  use error_handling, only: register_exc
  use minimax_utils,  only: er_aw_aux
  implicit none

  private

  !> list wth the number of points supported for the imag-time meshes
  integer, parameter :: ngrids = 15
  integer, parameter, public :: omega_npoints_supported(ngrids) = &
       [6, 8, 10, 12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34]
  integer, parameter :: energy_ranges_grids(ngrids) = &
       [13, 13, 15, 18, 21, 36, 37, 40, 40, 40, 19, 28, 28, 18, 7]

  public :: get_points_weights_omega

contains

  !> \brief Stores the minimax coefficients for all supported grid sizes
  !! @param[in]  grid_size - grid size
  !! @param[inout] aw - derived type of energy ranges and coefficients:weights
  subroutine set_aw_array_omega(grid_size, aw)
    integer, intent(in)            :: grid_size
    type(er_aw_aux), intent(inout) :: aw

    select case(grid_size)
    case(6)
       aw%energy_range(:) = [1.5849_dp,2.5119_dp,3.9811_dp,6.3096_dp,10.0000_dp,15.8489_dp,25.1189_dp,39.8107_dp,63.0957_dp,&
          100.0000_dp,158.4893_dp,251.1886_dp,398.1072_dp]
       aw%aw_erange_matrix(:,1) = [0.1655921867_dp,0.5209012738_dp,0.9601309570_dp,1.5943859351_dp,2.7306701362_dp,5.8749472842_dp,0.3350409678_dp,0.3850084475_dp,0.5102784088_dp,&
          0.8010367883_dp,1.6389781325_dp,6.1530434388_dp]
       aw%aw_erange_matrix(:,2) = [0.1805946879_dp,0.5710657016_dp,1.0624914384_dp,1.7847101379_dp,3.0902608916_dp,6.7008049407_dp,0.3658827843_dp,0.4263996093_dp,0.5761737092_dp,&
          0.9174724599_dp,1.8850148010_dp,7.0551807646_dp]
       aw%aw_erange_matrix(:,3) = [0.2107412898_dp,0.6758353424_dp,1.2899365007_dp,2.2366556977_dp,3.9937219129_dp,8.8543593569_dp,0.4284867076_dp,0.5184556747_dp,0.7378491813_dp,&
          1.2226010100_dp,2.5465173691_dp,9.4627147137_dp]
       aw%aw_erange_matrix(:,4) = [0.2398913915_dp,0.7830273741_dp,1.5440300762_dp,2.7906304931_dp,5.1927614407_dp,11.8692704652_dp,0.4899249120_dp,0.6209200162_dp,0.9423639946_dp,&
          1.6456777707_dp,3.5063104284_dp,12.9449494205_dp]
       aw%aw_erange_matrix(:,5) = [0.2673892983_dp,0.8903650428_dp,1.8225708515_dp,3.4589903359_dp,6.7690150328_dp,16.0814892391_dp,0.5487973414_dp,0.7321630617_dp,1.1933651064_dp,&
          2.2183473080_dp,4.8878751777_dp,17.9923442029_dp]
       aw%aw_erange_matrix(:,6) = [0.2928309791_dp,0.9958642183_dp,2.1219318882_dp,4.2497817154_dp,8.8133486370_dp,21.9427374532_dp,0.6041411724_dp,0.8499697001_dp,1.4915299424_dp,&
          2.9717516759_dp,6.8532141469_dp,25.3187548500_dp]
       aw%aw_erange_matrix(:,7) = [0.3159877346_dp,1.0977576875_dp,2.4369534500_dp,5.1642537452_dp,11.4185878867_dp,30.0488928270_dp,0.6553089559_dp,0.9716623387_dp,1.8340079795_dp,&
          3.9316976361_dp,9.6037639984_dp,35.9602691606_dp]
       aw%aw_erange_matrix(:,8) = [0.3367465699_dp,1.1944388989_dp,2.7609613838_dp,6.1944061883_dp,14.6663452364_dp,41.1631772657_dp,0.7018712769_dp,1.0942251613_dp,2.2141179142_dp,&
          5.1124612568_dp,13.3702621560_dp,51.4089921368_dp]
       aw%aw_erange_matrix(:,9) = [0.3550728269_dp,1.2844528796_dp,3.0859658208_dp,7.3210405275_dp,18.6070113241_dp,56.2194168750_dp,0.7435582547_dp,1.2144539917_dp,2.6213742311_dp,&
          6.5100771106_dp,18.3838109224_dp,73.7809389606_dp]
       aw%aw_erange_matrix(:,10) = [0.3709877153_dp,1.3665230999_dp,3.4030538180_dp,8.5128437088_dp,23.2337294933_dp,76.2759655370_dp,0.7802273935_dp,1.3291370683_dp,3.0419246526_dp,&
          8.0964568401_dp,24.8212721126_dp,105.9887144250_dp]
       aw%aw_erange_matrix(:,11) = [0.3845515600_dp,1.4395817104_dp,3.7029023900_dp,9.7268226520_dp,28.4534055091_dp,102.3628470217_dp,0.8118390150_dp,1.4352343393_dp,3.4593757378_dp,&
          9.8157444418_dp,32.7233122900_dp,151.7982620088_dp]
       aw%aw_erange_matrix(:,12) = [0.3958434305_dp,1.5027520411_dp,3.9761855968_dp,10.9096324549_dp,34.0580373848_dp,135.1079781755_dp,0.8384180383_dp,1.5299753461_dp,3.8557072482_dp,&
          11.5832713988_dp,41.8902966573_dp,215.3882300448_dp]
       aw%aw_erange_matrix(:,13) = [0.4049246607_dp,1.5552016328_dp,4.2134089372_dp,11.9977032386_dp,39.6930350498_dp,173.9106681833_dp,0.8599721981_dp,1.6107309991_dp,4.2114952563_dp,&
          13.2844959178_dp,51.7581905099_dp,299.3780119369_dp]
       aw%aw_erange_matrix(:,14) = [0.4117629031_dp,1.5957145113_dp,4.4032949099_dp,12.9112278362_dp,44.7999640048_dp,215.2118480052_dp,0.8763104024_dp,1.6743960921_dp,4.5039003347_dp,&
          14.7656672369_dp,61.2249538434_dp,398.9837714449_dp]            
    case(8)
       aw%energy_range(:) = [10.0000_dp,15.8489_dp,25.1189_dp,39.8107_dp,63.0957_dp,100.0000_dp,158.4893_dp,251.1886_dp,398.1072_dp,&
          630.9573_dp,1000.0000_dp,1584.8932_dp,2511.8864_dp]
       aw%aw_erange_matrix(:,1) = [0.2071657943_dp,0.6610068185_dp,1.2441397724_dp,2.0831497522_dp,3.3887988553_dp,5.5847789422_dp,9.8270353946_dp,21.7281350381_dp,0.4207454390_dp,&
          0.5014976440_dp,0.6851766586_dp,1.0262600603_dp,1.6496784456_dp,2.9041456023_dp,6.2037554522_dp,23.2093654407_dp]
       aw%aw_erange_matrix(:,2) = [0.2281975509_dp,0.7378108396_dp,1.4244318649_dp,2.4707104240_dp,4.1905418464_dp,7.2131597201_dp,13.1990042083_dp,29.9997176794_dp,0.4649882690_dp,&
          0.5742245008_dp,0.8283885406_dp,1.3139969363_dp,2.2222991557_dp,4.0573273175_dp,8.7886716040_dp,32.6307167054_dp]
       aw%aw_erange_matrix(:,3) = [0.2478325384_dp,0.8123920588_dp,1.6105407106_dp,2.8990088123_dp,5.1406825221_dp,9.2811129613_dp,17.7585094674_dp,41.7034890992_dp,0.5067204192_dp,&
          0.6488380257_dp,0.9885419393_dp,1.6617137904_dp,2.9674466307_dp,5.6611212463_dp,12.5401854734_dp,46.3415559475_dp]
       aw%aw_erange_matrix(:,4) = [0.2659623770_dp,0.8840006194_dp,1.8001085399_dp,3.3647028374_dp,6.2468323093_dp,11.8655812207_dp,23.8618997130_dp,58.2201559306_dp,0.5456512836_dp,&
          0.7242341275_dp,1.1636891030_dp,2.0709124861_dp,3.9122188078_dp,7.8497209223_dp,17.9508953905_dp,66.3317026050_dp]
       aw%aw_erange_matrix(:,5) = [0.2825286714_dp,0.9519740012_dp,1.9904683567_dp,3.8621957498_dp,7.5094353654_dp,15.0358645326_dp,31.9280725161_dp,81.4355610562_dp,0.5815837142_dp,&
          0.7992370771_dp,1.3509789933_dp,2.5397840102_dp,5.0777819156_dp,10.7708899999_dp,25.6817056446_dp,95.5142824872_dp]
       aw%aw_erange_matrix(:,6) = [0.2975081621_dp,1.0157197989_dp,2.1786795733_dp,4.3835258184_dp,8.9196010680_dp,18.8421849707_dp,42.4191477984_dp,113.8785066010_dp,0.6143889942_dp,&
          0.8726271509_dp,1.5467092421_dp,3.0626313217_dp,6.4748941133_dp,14.5714440993_dp,36.5813059769_dp,138.1359857525_dp]
       aw%aw_erange_matrix(:,7) = [0.3109044422_dp,1.0747189671_dp,2.3616257812_dp,4.9184789791_dp,10.4573448605_dp,23.3016349187_dp,55.7970544598_dp,158.8526834842_dp,0.6439946472_dp,&
          0.9431868807_dp,1.7464589798_dp,3.6295451089_dp,8.0992146383_dp,19.3741257946_dp,51.6724717864_dp,200.3405959261_dp]
       aw%aw_erange_matrix(:,8) = [0.3227426886_dp,1.1285376083_dp,2.5361560368_dp,5.4549563099_dp,12.0906636709_dp,28.3830633636_dp,72.4478080141_dp,220.5115689924_dp,0.6703777562_dp,&
          1.0097564124_dp,1.9453052591_dp,4.2264715904_dp,9.9271812582_dp,25.2464912037_dp,72.0737340430_dp,290.8984786126_dp]
       aw%aw_erange_matrix(:,9) = [0.3330653429_dp,1.1768381468_dp,2.6992426824_dp,5.9795753281_dp,13.7758193804_dp,33.9934529728_dp,92.5691845452_dp,303.7688360097_dp,0.6935592742_dp,&
          1.0712866459_dp,2.1381008841_dp,4.8357521746_dp,11.9134085419_dp,32.1651386464_dp,98.8188127849_dp,422.0268465485_dp]
       aw%aw_erange_matrix(:,10) = [0.3419271677_dp,1.2193812021_dp,2.8481167646_dp,6.4783947849_dp,15.4589061277_dp,39.9686099759_dp,116.0253061549_dp,413.8290580287_dp,0.7135957406_dp,&
          1.1268764982_dp,2.3197607958_dp,5.4370721963_dp,13.9903510517_dp,39.9811217952_dp,132.5423013393_dp,609.9378472851_dp]
       aw%aw_erange_matrix(:,11) = [0.3493873557_dp,1.2560051022_dp,2.9803168477_dp,6.9374999179_dp,17.0780527326_dp,46.0690845013_dp,142.1776489457_dp,554.9087117432_dp,0.7305630635_dp,&
          1.1757734018_dp,2.4854579125_dp,6.0084938106_dp,16.0699904831_dp,48.3919200908_dp,173.0191454100_dp,873.9156615559_dp]
       aw%aw_erange_matrix(:,12) = [0.3554939312_dp,1.2865572716_dp,3.0935328862_dp,7.3429185821_dp,18.5642203900_dp,51.9760118440_dp,169.6889674060_dp,727.2704120894_dp,0.7445214007_dp,&
          1.2172980158_dp,2.6305474321_dp,6.5268067690_dp,18.0450614155_dp,56.9170815136_dp,218.5505156517_dp,1230.5349651906_dp]
       aw%aw_erange_matrix(:,13) = [0.3602495964_dp,1.3107201533_dp,3.1850020782_dp,7.6788144410_dp,19.8359971798_dp,57.2679524910_dp,196.2327143675_dp,920.7799560162_dp,0.7554361141_dp,&
          1.2506098187_dp,2.7498665119_dp,6.9656401678_dp,19.7831086916_dp,64.8519815020_dp,265.1056052278_dp,1674.6829767463_dp]
       aw%aw_erange_matrix(:,14) = [0.3635299969_dp,1.3275808457_dp,3.2498551555_dp,7.9215394123_dp,20.7780605758_dp,61.3311078827_dp,217.8756023806_dp,1101.7305989568_dp,0.7629878767_dp,&
          1.2741002658_dp,2.8355822804_dp,7.2879118995_dp,21.0979770379_dp,71.1253081773_dp,304.8441281342_dp,2131.3275131179_dp] 
    case(10)
       aw%energy_range(:) = [10.0000_dp,15.8489_dp,25.1189_dp,39.8107_dp,63.0957_dp,100.0000_dp,158.4893_dp,251.1886_dp,398.1072_dp,&
          630.9573_dp,1000.0000_dp,1584.8932_dp,2511.8864_dp,3981.0717_dp,6309.5734_dp]
       aw%aw_erange_matrix(:,1) = [0.1691355148_dp,0.5284642103_dp,0.9545511712_dp,1.5047418958_dp,2.2597115867_dp,3.3454936705_dp,4.9862966533_dp,7.6650093063_dp,12.7796542255_dp,&
          27.2864801899_dp,0.3417226652_dp,0.3844227596_dp,0.4771272671_dp,0.6363706526_dp,0.8938238488_dp,1.3129879948_dp,2.0414882487_dp,3.5103169114_dp,7.4908908101_dp,28.4904121016_dp]
       aw%aw_erange_matrix(:,2) = [0.1869871309_dp,0.5895532734_dp,1.0837264945_dp,1.7513396673_dp,2.7113373418_dp,4.1525884587_dp,6.4085359717_dp,10.1783866742_dp,17.4346686857_dp,&
          37.8980985815_dp,0.3786450204_dp,0.4367817026_dp,0.5649171097_dp,0.7896364896_dp,1.1607673493_dp,1.7743885110_dp,2.8435711821_dp,4.9711742872_dp,10.6201538876_dp,40.0398066552_dp]
       aw%aw_erange_matrix(:,3) = [0.2039012435_dp,0.6490652017_dp,1.2154617028_dp,2.0166038170_dp,3.2247481157_dp,5.1221232176_dp,8.2115606004_dp,13.5286265845_dp,23.9062720036_dp,&
          53.0749019920_dp,0.4138813037_dp,0.4900973682_dp,0.6611573329_dp,0.9690511968_dp,1.4926655730_dp,2.3817399021_dp,3.9551983098_dp,7.0759673349_dp,15.2045183691_dp,56.8572918710_dp]
       aw%aw_erange_matrix(:,4) = [0.2215612726_dp,0.7130836108_dp,1.3641220347_dp,2.3327570045_dp,3.8722111284_dp,6.4171323475_dp,10.7641098295_dp,18.5487568360_dp,34.1014651224_dp,&
          77.8384401962_dp,0.4509585163_dp,0.5500867058_dp,0.7775819091_dp,1.2005601561_dp,1.9481146439_dp,3.2680298069_dp,5.6771590916_dp,10.5041265527_dp,22.8758677846_dp,84.9127568138_dp]
       aw%aw_erange_matrix(:,5) = [0.2345329395_dp,0.7614514514_dp,1.4815009224_dp,2.5950035869_dp,4.4373238158_dp,7.6084809054_dp,13.2446162815_dp,23.7066487254_dp,45.1296102041_dp,&
          105.6515405245_dp,0.4783942076_dp,0.5972714864_dp,0.8751233602_dp,1.4057763384_dp,2.3748072156_dp,4.1473552541_dp,7.4887345561_dp,14.3092728834_dp,31.6856232830_dp,117.1748283873_dp]
       aw%aw_erange_matrix(:,6) = [0.2464823496_dp,0.8071022381_dp,1.5964864475_dp,2.8627114558_dp,5.0394486965_dp,8.9364966681_dp,16.1470780375_dp,30.0602271613_dp,59.4110826768_dp,&
          143.0796327614_dp,0.5038292544_dp,0.6433038741_dp,0.9752840930_dp,1.6265181595_dp,2.8558127957_dp,5.1896016186_dp,9.7542857946_dp,19.3262236939_dp,43.7567202430_dp, 161.6443914088_dp]
       aw%aw_erange_matrix(:,7) = [0.2654151573_dp,0.8817697294_dp,1.7938027165_dp,3.3470666145_dp,6.1913419928_dp,11.6351237642_dp,22.4546568543_dp,44.9392680518_dp,95.5658998932_dp,&
          244.1384121888_dp,0.5444673083_dp,0.7217806651_dp,1.1573032493_dp,2.0521555932_dp,3.8424395191_dp,7.4786895698_dp,15.1258300988_dp,32.2334439760_dp,77.0350952259_dp,286.6561721471_dp]
       aw%aw_erange_matrix(:,8) = [0.2762854262_dp,0.9260414313_dp,1.9164695671_dp,3.6641228087_dp,6.9876700672_dp,13.6157081584_dp,27.4080928631_dp,57.5664213221_dp,128.9747437104_dp,&
          344.8292437935_dp,0.5679986258_dp,0.7701986486_dp,1.2766274473_dp,2.3475292633_dp,4.5697038635_dp,9.2840896357_dp,19.7029788976_dp,44.2289545620_dp,110.6270863337_dp,417.1647742251_dp]
       aw%aw_erange_matrix(:,9) = [0.2820309257_dp,0.9498823515_dp,1.9843457172_dp,3.8448332437_dp,7.4562107662_dp,14.8232425960_dp,30.5554359026_dp,65.9925282444_dp,152.5750528656_dp,&
          419.9454634243_dp,0.5804976867_dp,0.7968584417_dp,1.3446054795_dp,2.5213920887_dp,5.0132488016_dp,10.4308177021_dp,22.7523418007_dp,52.6831372017_dp,135.7521321651_dp,517.8733383965_dp]
       aw%aw_erange_matrix(:,10) = [0.2916392096_dp,0.9904672599_dp,2.1029049829_dp,4.1695234368_dp,8.3244617815_dp,17.1414352405_dp,36.8575879013_dp,83.7621447531_dp,205.6283870841_dp,&
          600.5018496162_dp,0.6014985087_dp,0.8431887875_dp,1.4665763527_dp,2.8432978010_dp,5.8636306085_dp,12.7210853018_dp,29.1509189638_dp,71.5379751769_dp,195.8981148161_dp,770.3526395770_dp]
       aw%aw_erange_matrix(:,11) = [0.2997413607_dp,1.0254167167_dp,2.2081204281_dp,4.4673434763_dp,9.1504060805_dp,19.4417624082_dp,43.4386668156_dp,103.5471053219_dp,269.7738483758_dp,&
          840.2848331258_dp,0.6193060617_dp,0.8840434321_dp,1.5781675031_dp,3.1488061460_dp,6.7046855839_dp,15.0996376420_dp,36.2060502309_dp,93.9657265595_dp,274.5288412137_dp,1126.0984805870_dp]
       aw%aw_erange_matrix(:,12) = [0.3068119265_dp,1.0564844599_dp,2.3041382741_dp,4.7471116229_dp,9.9518408796_dp,21.7608680962_dp,50.3946041349_dp,125.7778595593_dp,348.0823679645_dp,&
          1165.5956152032_dp,0.6349224147_dp,0.9211050699_dp,1.6826698673_dp,3.4442834592_dp,7.5487363569_dp,17.5953310413_dp,44.0329344049_dp,120.7330042899_dp,378.0891314432_dp,1642.4900496712_dp]
       aw%aw_erange_matrix(:,13) = [0.3128903217_dp,1.0836325633_dp,2.3900037820_dp,5.0037876383_dp,10.7087987937_dp,24.0290199661_dp,57.5035919858_dp,149.8665589334_dp,440.3163793218_dp,&
          1597.3324135008_dp,0.6484056036_dp,0.9540645582_dp,1.7782223062_dp,3.7222989519_dp,8.3698000470_dp,20.1241272281_dp,52.3878207087_dp,151.3903898996_dp,509.3860603144_dp,2383.3770295017_dp]
       aw%aw_erange_matrix(:,14) = [0.3180208381_dp,1.1068742715_dp,2.4649964188_dp,5.2330020840_dp,11.4022590556_dp,26.1726545478_dp,64.4968153514_dp,174.8953978169_dp,544.3139194265_dp,&
          2152.9928502293_dp,0.6598289811_dp,0.9827062576_dp,1.8632606978_dp,3.9759704470_dp,9.1413408907_dp,22.5889526629_dp,60.9288369969_dp,184.8771547581_dp,668.1120868220_dp,3424.9941233446_dp]
       aw%aw_erange_matrix(:,15) = [0.3222327278_dp,1.1261846760_dp,2.5283617927_dp,5.4303608672_dp,12.0125527722_dp,28.1109550492_dp,71.0481159543_dp,199.5351883274_dp,654.9414770847_dp,&
          2834.2166166353_dp,0.6692367974_dp,1.0068010254_dp,1.9362446833_dp,4.1983457774_dp,9.8350334132_dp,24.8769328455_dp,69.1997933226_dp,219.3240772356_dp,848.0876503767_dp,4831.8698300660_dp]
       aw%aw_erange_matrix(:,16) = [0.3255214101_dp,1.1414100173_dp,2.5790085497_dp,5.5905435586_dp,12.5168740676_dp,29.7491494348_dp,76.7532744950_dp,221.9330386256_dp,762.7629878613_dp,&
          3600.8832911895_dp,0.6766014703_dp,1.0259889580_dp,1.9953123273_dp,4.3814587926_dp,10.4182966230_dp,26.8526025391_dp,76.6027833431_dp,251.8170602854_dp,1033.5460450322_dp,6581.5947573428_dp]
    case(12)
       aw%energy_range(:) = [10.0000_dp,15.8489_dp,25.1189_dp,39.8107_dp,63.0957_dp,100.0000_dp,158.4893_dp,251.1886_dp,398.1072_dp,&
          630.9573_dp,1000.0000_dp,1584.8932_dp,2511.8864_dp,3981.0717_dp,6309.5734_dp,10000.0000_dp,15848.9319_dp,25118.8643_dp]
       aw%aw_erange_matrix(:,1) = [0.1429218755_dp,0.4413040082_dp,0.7787608801_dp,1.1864502692_dp,1.7042237511_dp,2.3869386764_dp,3.3160967043_dp,4.6250055507_dp,6.5625649934_dp,&
          9.6884369954_dp,15.6631146136_dp,32.7980178888_dp,0.2879096642_dp,0.3132260593_dp,0.3668155938_dp,0.4551062337_dp,0.5894057676_dp,0.7892350989_dp,1.0903935815_dp,&
          1.5663175771_dp,2.3928796682_dp,4.0893782104_dp,8.7825606869_dp,33.8102751115_dp]
       aw%aw_erange_matrix(:,2) = [0.1584039050_dp,0.4923328393_dp,0.8800124739_dp,1.3655907229_dp,2.0071985270_dp,2.8866096232_dp,4.1262350784_dp,5.9244211841_dp,8.6433443828_dp,&
          13.0785495268_dp,21.5592271650_dp,45.7118338614_dp,0.3196222832_dp,0.3542571191_dp,0.4283564047_dp,0.5523360633_dp,0.7442466481_dp,1.0344524304_dp,1.4766097932_dp,&
          2.1765609805_dp,3.3811001812_dp,5.8128874205_dp,12.4386306554_dp,47.5132218404_dp]
       aw%aw_erange_matrix(:,3) = [0.1732162492_dp,0.5421693623_dp,0.9824597614_dp,1.5548177385_dp,2.3422035365_dp,3.4651351072_dp,5.1078262325_dp,7.5699331662_dp,11.3905202972_dp,&
          17.7236693379_dp,29.8814434759_dp,64.2938771336_dp,0.3501242235_dp,0.3957859959_dp,0.4947508394_dp,0.6635483653_dp,0.9309027084_dp,1.3448855041_dp,1.9886743748_dp,&
          3.0204285513_dp,4.7952476232_dp,8.3319958117_dp,17.8065500802_dp,67.4765078414_dp]
       aw%aw_erange_matrix(:,4) = [0.1872798507_dp,0.5905014926_dp,1.0854206804_dp,1.7532172531_dp,2.7093772498_dp,4.1282041031_dp,6.2843865572_dp,9.6323358121_dp,14.9869260172_dp,&
          24.0540642157_dp,41.6061199649_dp,91.0579838939_dp,0.3792435644_dp,0.4375045382_dp,0.5656048578_dp,0.7888458222_dp,1.1518726945_dp,1.7303560649_dp,2.6551434426_dp,&
          4.1695556253_dp,6.7988935390_dp,12.0017657848_dp,25.7012299118_dp,96.6398811058_dp]
       aw%aw_erange_matrix(:,5) = [0.2005431091_dp,0.6370759298_dp,1.1882131903_dp,1.9596143934_dp,3.1080141107_dp,4.8798810059_dp,7.6781144945_dp,12.1876192174_dp,19.6488402798_dp,&
          32.6228933481_dp,58.0768113011_dp,129.6233187971_dp,0.4068597097_dp,0.4791053751_dp,0.6403871274_dp,0.9279233717_dp,1.4089046869_dp,2.2000500441_dp,3.5065734207_dp,&
          5.7089766044_dp,9.6048855357_dp,17.3210624857_dp,37.3219713868_dp,139.3487102141_dp]
       aw%aw_erange_matrix(:,6) = [0.2129723629_dp,0.6816735477_dp,1.2901205273_dp,2.1725086187_dp,3.5363611280_dp,5.7219435274_dp,9.3081177834_dp,15.3135513250_dp,25.6240110606_dp,&
          44.1260843459_dp,81.1209689443_dp,185.1814120492_dp,0.4328851129_dp,0.5202704606_dp,0.7184166502_dp,1.0800067147_dp,1.7027459853_dp,2.7617892731_dp,4.5740080605_dp,&
          7.7357979157_dp,13.4820663628_dp,24.9778985366_dp,54.4233997075_dp,202.0289741531_dp]
       aw%aw_erange_matrix(:,7) = [0.2245470036_dp,0.7240993142_dp,1.3903836822_dp,2.3900535856_dp,3.9914711962_dp,6.6532201376_dp,11.1882815305_dp,19.0846043138_dp,33.1849165684_dp,&
          59.4159094721_dp,113.1875318972_dp,265.1423802300_dp,0.4572563533_dp,0.5606704196_dp,0.7988654765_dp,1.2438096034_dp,2.0328893255_dp,3.4211580276_dp,5.8867908786_dp,&
          10.3557613983_dp,18.7581531731_dp,35.8993571496_dp,79.5506004677_dp,294.1829550392_dp]
       aw%aw_erange_matrix(:,8) = [0.2352569289_dp,0.7641800006_dp,1.4882116081_dp,2.6100766743_dp,4.4691309002_dp,7.6689956608_dp,13.3248903807_dp,23.5651210903_dp,42.6155684811_dp,&
          79.5012479303_dp,157.4966041219_dp,380.0140548312_dp,0.4799299760_dp,0.5999706068_dp,0.8807759597_dp,1.4175212571_dp,2.3973518587_dp,4.1805461970_dp,7.4696971805_dp,&
          13.6771070681_dp,25.8156997830_dp,51.2989610152_dp,116.3509369773_dp,429.8532978855_dp]
       aw%aw_erange_matrix(:,9) = [0.2451010324_dp,0.8017654177_dp,1.5828017133_dp,2.8301315091_dp,4.9638755099_dp,8.7605637856_dp,15.7142019721_dp,28.8008583603_dp,54.1903386201_dp,&
          105.5249731655_dp,218.1800947562_dp,544.5701280399_dp,0.5008804244_dp,0.6378406025_dp,0.9630900436_dp,1.5988302974_dp,2.7925242877_dp,5.0382073210_dp,9.3394978690_dp,&
          17.8013122631_dp,35.0774426337_dp,72.7082927464_dp,169.9652196055_dp,629.7648944090_dp]
       aw%aw_erange_matrix(:,10) = [0.2540860929_dp,0.8367307788_dp,1.6733652691_dp,3.0475761199_dp,5.4690974483_dp,9.9150059248_dp,18.3402496051_dp,34.8093868001_dp,68.1439425172_dp,&
          138.7068831809_dp,300.3728140052_dp,779.3504064325_dp,0.5200987061_dp,0.6739648288_dp,1.0446871754_dp,1.7849867539_dp,3.2131230326_dp,5.9874580217_dp,11.5012197452_dp,&
          22.8106613081_dp,46.9772400931_dp,101.9720804684_dp,247.4758767966_dp,924.4338269797_dp]
       aw%aw_erange_matrix(:,11) = [0.2622257858_dp,0.8689789279_dp,1.7591538896_dp,3.2596687994_dp,5.9772463916_dp,11.1152640178_dp,21.1732100622_dp,41.5702473402_dp,84.6330027911_dp,&
          180.2402911075_dp,410.1862989334_dp,1112.4892104442_dp,0.5375911789_dp,0.7080527257_dp,1.1244271067_dp,1.9728991568_dp,3.6522685550_dp,7.0161575650_dp,13.9445301437_dp,&
          28.7532181011_dp,61.9137136958_dp,141.1783745973_dp,358.3390054868_dp,1358.6237808356_dp]
       aw%aw_erange_matrix(:,12) = [0.2695397145_dp,0.8984418080_dp,1.8394842699_dp,3.4636731491_dp,6.4801097618_dp,12.3405489690_dp,24.1686845488_dp,49.0161813673_dp,103.6916002257_dp,&
          231.1334699402_dp,554.4649729304_dp,1581.7521816579_dp,0.5533782107_dp,0.7398474700_dp,1.2011938191_dp,2.1592596327_dp,4.1016985956_dp,8.1065950190_dp,16.6408070873_dp,&
          35.6265963621_dp,80.1860664335_dp,192.4911013451_dp,514.6393661077_dp,1997.6015554757_dp]
       aw%aw_erange_matrix(:,13) = [0.2760524229_dp,0.9250808413_dp,1.9137590003_dp,3.6569632156_dp,6.9691521519_dp,13.5670860508_dp,27.2681820098_dp,57.0270405663_dp,125.1857068673_dp,&
          291.9952897702_dp,740.1948138210_dp,2236.4143292529_dp,0.5674926172_dp,0.7691325599_dp,1.2739369142_dp,2.3406863163_dp,4.5521069506_dp,9.2358710776_dp,19.5415050994_dp,&
          43.3627936431_dp,101.9156324042_dp,257.8561654396_dp,730.8786122152_dp,2935.5916108398_dp]
       aw%aw_erange_matrix(:,14) = [0.2817923316_dp,0.9488858958_dp,1.9814813109_dp,3.8371183004_dp,7.4358800759_dp,14.7691411930_dp,30.4009348015_dp,65.4279344217_dp,148.7737052777_dp,&
          362.7781053922_dp,973.4258195636_dp,3138.1182553730_dp,0.5799777558_dp,0.7957356560_dp,1.3417066692_dp,2.5138686631_dp,4.9935763572_dp,10.3767865261_dp,22.5783236670_dp,&
          51.8169502523_dp,126.9620346466_dp,338.5696696685_dp,1022.8563670293_dp,4306.3486764644_dp]
       aw%aw_erange_matrix(:,15) = [0.2867904139_dp,0.9698720269_dp,2.0422603506_dp,4.0019927656_dp,7.8721789655_dp,15.9201852024_dp,33.4868654474_dp,73.9924717306_dp,173.8808134122_dp,&
          442.5068881766_dp,1257.6017567738_dp,4358.8912675126_dp,0.5908848778_dp,0.8195285979_dp,1.4036776623_dp,2.6756952059_dp,5.4160438292_dp,11.4991268023_dp,25.6653392884_dp,&
          60.7626525470_dp,154.8473999059_dp,434.7285925715_dp,1405.0882864622_dp,1293.9857358549_dp]
       aw%aw_erange_matrix(:,16) = [0.2910780860_dp,0.9880717528_dp,2.0957994347_dp,4.1497340080_dp,8.2705289811_dp,16.9939014393_dp,36.4399516317_dp,82.4500835175_dp,199.6913462924_dp,&
          529.0300260431_dp,1591.2609924507_dp,5972.6134097078_dp,0.6002686025_dp,0.8404208888_dp,1.4591515488_dp,2.8233296403_dp,5.8096852863_dp,12.5710278432_dp,28.7025282055_dp,&
          69.8944444213_dp,184.7013394375_dp,544.6171058090_dp,1886.1991561619_dp,1636.7409625631_dp]
       aw%aw_erange_matrix(:,17) = [0.2946830135_dp,1.0035172315_dp,2.1418493923_dp,4.2786919199_dp,8.6239165593_dp,17.9644477245_dp,39.1701771936_dp,90.4925192593_dp,225.1497724816_dp,&
          618.7984682735_dp,1965.0906091672_dp,8032.3776354955_dp,0.6081776611_dp,0.8583399368_dp,1.5075185249_dp,2.9541692000_dp,6.1650041879_dp,13.5597425826_dp,31.5787233026_dp,&
          78.8332286894_dp,215.2252971871_dp,664.0736526166_dp,2461.7976878738_dp,13101.2227769949_dp]
       aw%aw_erange_matrix(:,18) = [0.2976197396_dp,1.0161987870_dp,2.1800846641_dp,4.3870919140_dp,8.9250517003_dp,18.8047527393_dp,41.5801589448_dp,97.7669240973_dp,248.9317783904_dp,&
          706.5698042833_dp,2358.0029706570_dp,10518.0301237170_dp,0.6146340927_dp,0.8731818667_dp,1.5481304343_dp,3.0655438415_dp,6.4721849704_dp,14.4304002566_dp,34.1694206849_dp,&
          87.1199659748_dp,244.6345912064_dp,785.7486816858_dp,3104.0521536434_dp,18366.2833303773_dp]
       aw%aw_erange_matrix(:,19) = [0.2998670956_dp,1.0259644793_dp,2.2097926993_dp,4.4721489070_dp,9.1639446473_dp,19.4800041119_dp,43.5475134513_dp,103.8257383340_dp,269.2803013626_dp,&
          784.6486318135_dp,2730.6250483393_dp,13221.1901021026_dp,0.6195831378_dp,0.8846907737_dp,1.5799656761_dp,3.1538099965_dp,6.7186856110_dp,15.1395754616_dp,36.3189542433_dp,&
          94.1609721175_dp,270.4429681576_dp,897.7357862925_dp,3745.2859315653_dp,24671.3210710722_dp]            
    case(14)
       aw%energy_range(:) = [10.0000_dp,15.8489_dp,25.1189_dp,39.8107_dp,63.0957_dp,100.0000_dp,158.4893_dp,251.1886_dp,398.1072_dp,&
          630.9573_dp,1000.0000_dp,1584.8932_dp,2511.8864_dp,3981.0717_dp,6309.5734_dp,10000.0000_dp,15848.9319_dp,25118.8643_dp,39810.7171_dp,&
          63095.7344_dp,100000.0000_dp]
       aw%aw_erange_matrix(:,1) = [0.1237520189_dp,0.3793210576_dp,0.6597671782_dp,0.9840308063_dp,1.3749190589_dp,1.8615001814_dp,2.4828690979_dp,3.2948316051_dp,4.3831950418_dp,&
          5.8934107334_dp,8.1059865442_dp,11.6655039451_dp,18.5015048397_dp,38.2819532635_dp,0.2488370146_dp,0.2650794331_dp,0.2989413739_dp,0.3533487033_dp,0.4332130096_dp,&
          0.5463612200_dp,0.7054753585_dp,0.9323118806_dp,1.2675835756_dp,1.7965516204_dp,2.7252931763_dp,4.6615549872_dp,10.0839562478_dp,39.1544232431_dp]
       aw%aw_erange_matrix(:,2) = [0.1374094319_dp,0.4232874675_dp,0.7434627635_dp,1.1246519429_dp,1.5998350917_dp,2.2119848549_dp,3.0197111559_dp,4.1068823716_dp,5.6011802735_dp,&
          7.7149478780_dp,10.8492870266_dp,15.9125394356_dp,25.6080428343_dp,53.4755730624_dp,0.2766436526_dp,0.2989467997_dp,0.3458113038_dp,0.4220036888_dp,0.5354349601_dp,&
          0.6984826350_dp,0.9306650650_dp,1.2643532502_dp,1.7581552449_dp,2.5320019406_dp,3.8730686134_dp,6.6286669095_dp,14.2653278687_dp,55.0281302896_dp]
       aw%aw_erange_matrix(:,3) = [0.1505666691_dp,0.4663185031_dp,0.8277068519_dp,1.2712878156_dp,1.8436088079_dp,2.6071668363_dp,3.6492744660_dp,5.0966470423_dp,7.1423813249_dp,&
          10.1035301725_dp,14.5662361716_dp,21.8316004103_dp,35.7316313060_dp,75.4226567085_dp,0.3035404850_dp,0.3330713582_dp,0.3957256454_dp,0.4990979554_dp,0.6558183478_dp,&
          0.8856202117_dp,1.2193001614_dp,1.7067737275_dp,2.4354797344_dp,3.5786067486_dp,5.5425490305_dp,9.5203253363_dp,20.4055935443_dp,78.1654326549_dp]
       aw%aw_erange_matrix(:,4) = [0.1631562520_dp,0.5081770951_dp,0.9120372525_dp,1.4233448685_dp,2.1061443172_dp,3.0494194848_dp,4.3813220072_dp,6.2921913817_dp,9.0755650205_dp,&
          13.2120067673_dp,19.5745958656_dp,30.0570187431_dp,50.1492744094_dp,107.1783297388_dp,0.3293860261_dp,0.3672500562_dp,0.4484587525_dp,0.5846770059_dp,0.7955240571_dp,&
          1.1120199476_dp,1.5828730610_dp,2.2865361693_dp,3.3573874234_dp,5.0527187015_dp,7.9582310915_dp,13.7699199075_dp,29.4466322395_dp,111.9883059949_dp]
       aw%aw_erange_matrix(:,5) = [0.1751328845_dp,0.5486776639_dp,0.9960210811_dp,1.5801351784_dp,2.3869791803_dp,3.5403675980_dp,5.2247498897_dp,7.7222102918_dp,11.4770489386_dp,&
          17.2224496874_dp,26.2781750027_dp,41.4427682146_dp,70.6620174299_dp,153.1839920115_dp,0.3540807601_dp,0.4012905745_dp,0.5037213598_dp,0.6785936111_dp,0.9553641327_dp,&
          1.3815652392_dp,2.0332043125_dp,3.0338539093_dp,4.5937656003_dp,7.1052263662_dp,11.4303112105_dp,20.0061321519_dp,42.7874318856_dp,161.5637932500_dp]
       aw%aw_erange_matrix(:,6) = [0.1864652922_dp,0.5876633466_dp,1.0792191469_dp,1.7408187754_dp,2.6851623062_dp,4.0805890775_dp,6.1868950019_dp,9.4146722805_dp,14.4288638472_dp,&
          22.3464969167_dp,35.1805962456_dp,57.1239822554_dp,99.7934538596_dp,219.8849251090_dp,0.3775511583_dp,0.4349982165_dp,0.5611463585_dp,0.7804726447_dp,1.1356966669_dp,&
          1.6975108670_dp,2.5819061296_dp,3.9814540008_dp,6.2266818203_dp,9.9270851610_dp,16.3796136997_dp,29.1318891212_dp,62.5017658877_dp,234.4013377657_dp]
       aw%aw_erange_matrix(:,7) = [0.1971320791_dp,0.6249956559_dp,1.1611723802_dp,1.9043787816_dp,2.9991713766_dp,4.6693370438_dp,7.2727561475_dp,11.3950369748_dp,18.0156522854_dp,&
          28.8229704372_dp,46.8960302078_dp,78.5862781220_dp,141.0502377875_dp,316.6100822049_dp,0.3997417912_dp,0.4681716805_dp,0.6202849713_dp,0.8896863729_dp,1.3363204068_dp,&
          2.0621694870_dp,3.2396561944_dp,5.1633036255_dp,8.3493271249_dp,13.7533070113_dp,23.3648778784_dp,42.4276062839_dp,91.6530813193_dp,341.6356160655_dp]
       aw%aw_erange_matrix(:,8) = [0.2071196561_dp,0.6605509163_dp,1.2414020419_dp,2.0696221481_dp,3.3268716611_dp,5.3043048684_dp,8.4841749046_dp,13.6840687647_dp,22.3200876467_dp,&
          36.9116410253_dp,62.1534675938_dp,107.7376013130_dp,199.2542570825_dp,456.8148338118_dp,0.4206116673_dp,0.5006042902_dp,0.6806119502_dp,1.0053444289_dp,1.5563837555_dp,&
          2.4765762114_dp,4.0152964555_dp,6.6127134579_dp,11.0633152384_dp,18.8639006227_dp,33.1094249311_dp,61.6803117299_dp,134.7433279119_dp,249.7791775842_dp]
       aw%aw_erange_matrix(:,9) = [0.2164211208_dp,0.6942199530_dp,1.3194178020_dp,2.2351995560_dp,3.6655163476_dp,5.9814567261_dp,9.8190406778_dp,16.2953330419_dp,27.4167071273_dp,&
          46.8819737408_dp,81.7899193598_dp,146.9723500158_dp,280.9494882357_dp,659.8171183904_dp,0.4401325181_dp,0.5320881774_dp,0.7415378106_dp,1.1263002544_dp,1.7943210878_dp,&
          2.9401644200_dp,4.9148026024_dp,8.3597883164_dp,14.4738828904_dp,25.5798337763_dp,46.5220179584_dp,89.3352189162_dp,198.3395493241_dp,733.3143455343_dp]
       aw%aw_erange_matrix(:,10) = [0.2250355201_dp,0.7259090833_dp,1.3947298426_dp,2.3996392595_dp,4.0117873209_dp,6.6949449046_dp,11.2705993355_dp,19.2325359337_dp,33.3642363588_dp,&
          58.9958553993_dp,106.7269043438_dp,199.2107206824_dp,394.8716061946_dp,953.1877173873_dp,0.4582878540_dp,0.5624197640_dp,0.8024260872_dp,1.2511740402_dp,2.0478267482_dp,&
          3.4504904254_dp,5.9402028103_dp,10.4282848334_dp,18.6826463204_dp,34.2518029731_dp,64.7040280062_dp,128.6569747902_dp,291.9231783438_dp,1078.5150247338_dp]
       aw%aw_erange_matrix(:,11) = [0.2329672393_dp,0.7555413998_dp,1.4668625114_dp,2.5613905063_dp,4.3618737487_dp,7.4371311516_dp,12.8269527907_dp,22.4869444262_dp,40.1967444694_dp,&
          73.4837764659_dp,137.9240966582_dp,267.8878409083_dp,552.4401168858_dp,1375.9767397362_dp,0.4750722360_dp,0.5914055029_dp,0.8626137302_dp,1.3783908160_dp,2.3138738004_dp,&
          4.0030452232_dp,7.0885544544_dp,12.8320538455_dp,23.7778167607_dp,45.2396241167_dp,88.9322108883_dp,183.8727618124_dp,428.9924509359_dp,1589.0574882567_dp]
       aw%aw_erange_matrix(:,12) = [0.2402254166_dp,0.7830578426_dp,1.5353678880_dp,2.7188726288_dp,4.7115837714_dp,8.1987224212_dp,14.4708344316_dp,26.0351871851_dp,47.9142866742_dp,&
          90.5147950650_dp,176.3046192376_dp,356.8585531562_dp,768.1850111560_dp,1982.9500242172_dp,0.4904905322_dp,0.6188672200_dp,0.9214329207_dp,1.5062311781_dp,2.5887805355_dp,&
          4.5911873826_dp,8.3511084620_dp,15.5713777752_dp,29.8221820388_dp,58.8806263183_dp,120.6042717758_dp,260.2481833296_dp,628.4008054380_dp,2344.2037701947_dp]
       aw%aw_erange_matrix(:,13) = [0.2468233492_dp,0.8084178279_dp,1.5998381346_dp,2.8705261553_dp,5.0564823086_dp,8.9690218562_dp,16.1797300245_dp,29.8377700205_dp,56.4740220701_dp,&
          110.1618693375_dp,222.6488481804_dp,470.1773374173_dp,1059.9473360129_dp,2849.9290757236_dp,0.5045570755_dp,0.6446466508_dp,0.9782327448_dp,1.6328911714_dp,2.8683231326_dp,&
          5.2062217421_dp,9.7127961169_dp,18.6296360315_dp,36.8397037996_dp,75.4467599949_dp,161.1342170119_dp,364.0212493064_dp,915.8061534622_dp,3460.6173145631_dp]
       aw%aw_erange_matrix(:,14) = [0.2527778876_dp,0.8315993357_dp,1.6599158447_dp,3.0148625096_dp,5.3920458557_dp,9.7362845660_dp,17.9263848498_dp,33.8386264546_dp,65.7830638131_dp,&
          132.3657077044_dp,277.4583246669_dp,611.7132434736_dp,1448.5954171971_dp,4080.0961955192_dp,0.5172947070_dp,0.6686089264_dp,1.0323993525_dp,1.7565471256_dp,3.1478880630_dp,&
          5.8376343335_dp,11.1521569039_dp,21.9708116140_dp,44.8021855325_dp,95.0923352411_dp,211.7879402237_dp,502.0958885177_dp,1324.9016742343_dp,5109.1479217630_dp]
       aw%aw_erange_matrix(:,15) = [0.2581088191_dp,0.8525984201_dp,1.7153018106_dp,3.1505090575_dp,5.7138232801_dp,10.4881555752_dp,19.6796959600_dp,37.9659478012_dp,75.6944278689_dp,&
          156.9018728488_dp,340.7986066435_dp,784.5727195079_dp,1956.8887152134_dp,5810.5946526557_dp,0.5287337062_dp,0.6906448661_dp,1.0833734070_dp,1.8754207705_dp,3.4226532212_dp,&
          6.4734757869_dp,12.6417849586_dp,25.5383524518_dp,53.6179916839_dp,117.7972403305_dp,273.4582120634_dp,681.3862209558_dp,1897.7667932019_dp,7538.0176679465_dp]
       aw%aw_erange_matrix(:,16) = [0.2628382234_dp,0.8714280605_dp,1.7657596164_dp,3.2762461263_dp,6.0175894628_dp,11.2121536455_dp,21.4059326227_dp,42.1343821843_dp,86.0072674859_dp,&
          183.3569410967_dp,412.1386866853_dp,990.3340957722_dp,2607.0355156257_dp,8217.6518865426_dp,0.5389105748_dp,0.7106719317_dp,1.1306636688_dp,1.9878394438_dp,3.6877820943_dp,&
          7.1008636764_dp,14.1493031685_dp,29.2557818247_dp,63.1250213151_dp,143.3135345823_dp,346.3927806276_dp,907.7234332819_dp,2683.2231510780_dp,11103.1863555287_dp]
       aw%aw_erange_matrix(:,17) = [0.2669897168_dp,0.8881160366_dp,1.8111159140_dp,3.3910312600_dp,6.2994739200_dp,11.8961448564_dp,23.0701486723_dp,46.2484254367_dp,96.4719312295_dp,&
          211.1191385393_dp,490.2118567969_dp,1228.1451103779_dp,3416.4888677083_dp,11516.2764247119_dp,0.5478665013_dp,0.7286334755_dp,1.1738551166_dp,2.0922847636_dp,3.9386087483_dp,&
          7.7065456968_dp,15.6387696490_dp,33.0291620045_dp,73.0897431844_dp,171.1251080115_dp,429.9069845345_dp,1184.3032522341_dp,3731.5986458489_dp,16303.6463671380_dp]
       aw%aw_erange_matrix(:,18) = [0.2705873339_dp,0.9027008262_dp,1.8512534027_dp,3.4940013018_dp,6.5560338595_dp,12.5287105641_dp,24.6375162990_dp,50.2063656122_dp,106.7989482477_dp,&
          239.3855189704_dp,572.9217973825_dp,1493.7802560807_dp,4391.6531513823_dp,15946.3157069551_dp,0.5556449687_dp,0.7444951812_dp,1.2126082806_dp,2.1874187044_dp,4.1707779482_dp,&
          8.2774187326_dp,17.0722537538_dp,36.7509601270_dp,83.2126077162_dp,200.4282185106_dp,522.1239375852_dp,1509.7550525312_dp,5083.9915726941_dp,23809.4872933868_dp]
       aw%aw_erange_matrix(:,19) = [0.2736534033_dp,0.9152228943_dp,1.8860889161_dp,3.5844307219_dp,6.7842091626_dp,13.0992262603_dp,26.0740471238_dp,53.9032916651_dp,116.6682197836_dp,&
          267.1788894394_dp,657.2957186602_dp,1778.7428167398_dp,5519.3340205550_dp,21728.1102484379_dp,0.5622871153_dp,0.7582357751_dp,1.2466418911_dp,2.2720643515_dp,4.3802733591_dp,&
          8.8008061721_dp,18.4110184365_dp,40.3038468518_dp,93.1367925368_dp,230.1312226533_dp,619.7711046485_dp,1875.9814388890_dp,6754.1179845509_dp,34441.9288939212_dp]
       aw%aw_erange_matrix(:,20) = [0.2762038751_dp,0.9257049710_dp,1.9155181787_dp,3.6615955500_dp,6.9810205813_dp,13.5972653572_dp,27.3456018037_dp,57.2299813261_dp,125.7291850483_dp,&
          293.3488003957_dp,739.4224775091_dp,2069.3129722892_dp,6755.4499384197_dp,28953.1020102810_dp,0.5678214945_dp,0.7698248185_dp,1.2756800070_dp,2.3450904474_dp,4.5631932604_dp,&
          9.2640941495_dp,19.6151475842_dp,43.5610242519_dp,102.4505562674_dp,258.8483817322_dp,717.9870196681_dp,2265.7579672866_dp,8700.5852876781_dp,23988.4028262284_dp]
       aw%aw_erange_matrix(:,21) = [0.2782372627_dp,0.9341052990_dp,1.9392811143_dp,3.7244178713_dp,7.1426775987_dp,14.0104317718_dp,28.4127187100_dp,60.0605983468_dp,133.5708071158_dp,&
          316.4906323388_dp,814.1646091006_dp,2344.8657889336_dp,8008.3640462208_dp,37336.4453482422_dp,0.5722398731_dp,0.7791695364_dp,1.2993167745_dp,2.4050765797_dp,4.7149464686_dp,&
          9.6528320691_dp,20.6390719566_dp,46.3754930149_dp,110.6595490445_dp,284.8116113011_dp,809.8918933503_dp,2649.3019212461_dp,10784.5707432897_dp,67552.5430413666_dp]
       aw%aw_erange_matrix(:,22) = [0.2797061105_dp,0.9401976208_dp,1.9566150716_dp,3.7705342696_dp,7.2621588744_dp,14.3181598521_dp,29.2146691724_dp,62.2108644129_dp,139.6075090217_dp,&
          334.6128282321_dp,874.0601697758_dp,2573.2407688027_dp,9106.9093547352_dp,45681.2634053412_dp,0.5754349166_dp,0.7859786803_dp,1.3166648423_dp,2.4494118770_dp,4.8279665616_dp,&
          9.9449041050_dp,21.4163941175_dp,48.5390530841_dp,117.0692434004_dp,305.4980103858_dp,885.1734856830_dp,2976.7201952819_dp,12696.1696497372_dp,12778.6824362300_dp]            
    case(16)
       aw%energy_range(:) = [100.0000_dp,200.0000_dp,300.0000_dp,400.0000_dp,500.0000_dp,600.0000_dp,700.0000_dp,800.0000_dp,900.0000_dp,&
          1000.0000_dp,2000.0000_dp,3000.0000_dp,4000.0000_dp,5000.0000_dp,6000.0000_dp,7000.0000_dp,8000.0000_dp,9000.0000_dp,&
          10000.0000_dp,20000.0000_dp,30000.0000_dp,40000.0000_dp,50000.0000_dp,60000.0000_dp,70000.0000_dp,80000.0000_dp,100000.0000_dp,&
          200000.0000_dp,300000.0000_dp,400000.0000_dp,500000.0000_dp,600000.0000_dp,700000.0000_dp,900000.0000_dp,1000000.0000_dp,2000000.0000_dp]
       aw%aw_erange_matrix(:,1) = [0.1658239563_dp,0.5171334987_dp,0.9303765657_dp,1.4570328304_dp,2.1653332837_dp,3.1504614774_dp,4.5493907204_dp,6.5633618109_dp,9.4930875435_dp,&
          13.7964356275_dp,20.1903405767_dp,29.8547208171_dp,44.9220344899_dp,69.9611336073_dp,118.0767404750_dp,254.2593005437_dp,0.3348764695_dp,0.3746857163_dp,&
          0.4602578814_dp,0.6042610194_dp,0.8279633068_dp,1.1647585940_dp,1.6660742054_dp,2.4110700070_dp,3.5231064904_dp,5.2000156651_dp,7.7774760619_dp,&
          11.8871228315_dp,18.9384589784_dp,32.9637281791_dp,70.4783921035_dp,266.9897014276_dp]
       aw%aw_erange_matrix(:,2) = [0.1796188680_dp,0.5640280477_dp,1.0284857024_dp,1.6421539724_dp,2.5006876754_dp,3.7435994693_dp,5.5807790003_dp,8.3344011034_dp,12.5067566445_dp,&
          18.8932813718_dp,28.7803521422_dp,44.3247744642_dp,69.4123519683_dp,112.1580355482_dp,195.1385370534_dp,428.7889331091_dp,0.3633588263_dp,0.4144472317_dp,&
          0.5257979977_dp,0.7172007538_dp,1.0226908688_dp,1.4974394233_dp,2.2295271553_dp,3.3596316448_dp,5.1145469104_dp,7.8671158960_dp,12.2541055605_dp,&
          19.4492942193_dp,31.9606311600_dp,56.6861571917_dp,121.5342919900_dp,456.2757433765_dp]
       aw%aw_erange_matrix(:,3) = [0.1884206558_dp,0.5944581318_dp,1.0939595566_dp,1.7698243503_dp,2.7399902108_dp,4.1815948362_dp,6.3691396726_dp,9.7367276671_dp,14.9817468349_dp,&
          23.2417208469_dp,36.4072939505_dp,57.7165006394_dp,93.0524051206_dp,154.5727044051_dp,275.3415317763_dp,614.8206695657_dp,0.3816114859_dp,0.4409681802_dp,&
          0.5715874075_dp,0.7994100006_dp,1.1698313059_dp,1.7581297911_dp,2.6876523229_dp,4.1609421883_dp,6.5137596498_dp,10.3121683937_dp,16.5396985487_dp,&
          27.0108280205_dp,45.5192320567_dp,82.1625739085_dp,177.0957911240_dp,661.1497609718_dp]
       aw%aw_erange_matrix(:,4) = [0.1942245489_dp,0.6147551504_dp,1.1384592804_dp,1.8585037172_dp,2.9099802892_dp,4.4998377208_dp,6.9552594441_dp,10.8043246093_dp,16.9133032253_dp,&
          26.7254744252_dp,42.6898191774_dp,69.0766171353_dp,113.7276867204_dp,192.8055539393_dp,349.6006065415_dp,790.3452737603_dp,0.3936831697_dp,0.4589820405_dp,&
          0.6036357188_dp,0.8584885627_dp,1.2781808217_dp,1.9547747000_dp,3.0418929157_dp,4.7968259440_dp,7.6549361447_dp,12.3649985948_dp,20.2496262505_dp,&
          33.7670856027_dp,58.0126849704_dp,106.2423790202_dp,230.2949013970_dp,856.7974792736_dp]
       aw%aw_erange_matrix(:,5) = [0.1977793344_dp,0.6272812805_dp,1.1662618497_dp,1.9146970328_dp,3.0192751478_dp,4.7074780059_dp,7.3434611868_dp,11.5225402840_dp,18.2342831138_dp,&
          29.1500315315_dp,47.1447614801_dp,77.2947047525_dp,129.0029463514_dp,221.6596281574_dp,406.7355719018_dp,927.2785886489_dp,0.4010914531_dp,0.4702308484_dp,&
          0.6240364829_dp,0.8967364985_dp,1.3494396924_dp,2.0861507928_dp,3.2824399973_dp,5.2360918865_dp,8.4577514041_dp,13.8375110832_dp,22.9665285321_dp,&
          38.8239553182_dp,67.5699243680_dp,125.0145140343_dp,272.2104357523_dp,1010.7843885329_dp]
       aw%aw_erange_matrix(:,6) = [0.2010199598_dp,0.6387647242_dp,1.1919823186_dp,1.9672237865_dp,3.1225339638_dp,4.9057820342_dp,7.7183469335_dp,12.2242186521_dp,19.5408274593_dp,&
          31.5798875131_dp,51.6732052000_dp,85.7770369270_dp,145.0284244241_dp,252.4408886700_dp,468.6362428400_dp,1077.3185674270_dp,0.4078548995_dp,0.4806325979_dp,&
          0.6431662648_dp,0.9330450826_dp,1.4178702260_dp,2.2137849058_dp,3.5189801563_dp,5.6736133804_dp,9.2684033321_dp,15.3464195837_dp,25.7949391503_dp,&
          44.1777207407_dp,77.8637319323_dp,145.5476370987_dp,318.4847845698_dp,1180.7283739724_dp]
       aw%aw_erange_matrix(:,7) = [0.2036357272_dp,0.6480796035_dp,1.2130110734_dp,2.0105568730_dp,3.2085103104_dp,5.0724515279_dp,8.0364891453_dp,12.8257596802_dp,20.6730636781_dp,&
          33.7101155845_dp,55.6932870488_dp,93.4100340197_dp,159.6610710660_dp,280.9748180399_dp,526.8369133777_dp,1219.8782585539_dp,0.4133212123_dp,0.4891333035_dp,&
          0.6589889393_dp,0.9633962334_dp,1.4756465826_dp,2.3226398100_dp,3.7228583371_dp,6.0549753566_dp,9.9835456561_dp,16.6949014551_dp,28.3582444400_dp,&
          49.1030685081_dp,87.4821676279_dp,165.0097234930_dp,362.7425520277_dp,1343.2809779482_dp]
       aw%aw_erange_matrix(:,8) = [0.2058745597_dp,0.6560851540_dp,1.2312036533_dp,2.0483275158_dp,3.2840295056_dp,5.2200001614_dp,8.3204227320_dp,13.3672158293_dp,21.7015229918_dp,&
          35.6641641476_dp,59.4203432588_dp,100.5691194919_dp,173.5582973450_dp,308.4325898904_dp,583.5453837457_dp,1360.0872526018_dp,0.4180048571_dp,0.4964846624_dp,&
          0.6728092420_dp,0.9901400253_dp,1.5269809838_dp,2.4201774666_dp,3.9071667252_dp,6.4030043433_dp,10.6428407911_dp,17.9517964627_dp,30.7760779219_dp,&
          53.8090647968_dp,96.7970063398_dp,184.0976852129_dp,406.5124764312_dp,1504.1053871788_dp]
       aw%aw_erange_matrix(:,9) = [0.2077912924_dp,0.6629634515_dp,1.2469237706_dp,2.0811759162_dp,3.3501432124_dp,5.3500462336_dp,8.5724304784_dp,13.8513493414_dp,22.6283946159_dp,&
          37.4403000338_dp,62.8396973494_dp,107.2042380768_dp,186.5815850155_dp,334.4658066377_dp,637.9185442861_dp,1495.6706240607_dp,0.4220184019_dp,0.5028346837_dp,&
          0.6848492255_dp,1.0136143492_dp,1.5723626774_dp,2.5070345558_dp,4.0725558633_dp,6.7178768910_dp,11.2446128484_dp,19.1100572735_dp,33.0275110259_dp,&
          58.2410816559_dp,105.6752713292_dp,202.5005558789_dp,449.0416009016_dp,1660.4659036836_dp]
       aw%aw_erange_matrix(:,10) = [0.2094923936_dp,0.6690871465_dp,1.2609893400_dp,2.1107332848_dp,3.4099787231_dp,5.4684406771_dp,8.8032721349_dp,14.2977110381_dp,23.4889292652_dp,&
          39.1018169779_dp,66.0648728347_dp,113.5194187887_dp,199.1000985493_dp,359.7546766569_dp,691.2795357746_dp,1629.7753326901_dp,0.4255833454_dp,0.5085145079_dp,&
          0.6956987766_dp,1.0349067155_dp,1.6137842105_dp,2.5868202378_dp,4.2255077367_dp,7.0111805679_dp,11.8095483158_dp,20.2066645740_dp,35.1789178138_dp,&
          62.5191992847_dp,114.3381260493_dp,220.6457384190_dp,491.2838223352_dp,1815.8906489997_dp]
       aw%aw_erange_matrix(:,11) = [0.2199648199_dp,0.7071951819_dp,1.3500294829_dp,2.3014898753_dp,3.8039161896_dp,6.2640677360_dp,10.3883548409_dp,17.4342263733_dp,29.6892498137_dp,&
          51.4072666001_dp,90.6897262810_dp,163.4029142875_dp,301.8062739164_dp,576.0321730685_dp,1167.1668975223_dp,2866.3462242273_dp,0.4475919613_dp,0.5444247827_dp,&
          0.7660427188_dp,1.1760818110_dp,1.8944331370_dp,3.1396588060_dp,5.3110028691_dp,9.1474459873_dp,16.0425147770_dp,28.6838564013_dp,52.3961981225_dp,&
          98.1027617656_dp,189.5125190165_dp,385.0546230890_dp,886.8755808827_dp,3279.8478272665_dp]
       aw%aw_erange_matrix(:,12) = [0.2254855052_dp,0.7275782696_dp,1.3987483827_dp,2.4085395227_dp,4.0308014702_dp,6.7346910616_dp,11.3525925922_dp,19.4003421223_dp,33.7045605198_dp,&
          59.6659845037_dp,107.8826671839_dp,199.8010945617_dp,380.5441566179_dp,751.1705566466_dp,1574.9757412239_dp,3976.4773412762_dp,0.4592382989_dp,0.5640363685_dp,&
          0.8057309987_dp,1.2580610596_dp,2.0620213178_dp,3.4794870345_dp,5.9992069045_dp,10.5478798269_dp,18.9204492344_dp,34.6829355162_dp,65.1335363101_dp,&
          125.7646065496_dp,251.2571114743_dp,528.1385400286_dp,1248.0730162840_dp,4633.2205691667_dp]
       aw%aw_erange_matrix(:,13) = [0.2291458665_dp,0.7412092104_dp,1.4317646055_dp,2.4821649469_dp,4.1892346702_dp,7.0685405142_dp,12.0481267399_dp,20.8444486250_dp,36.7129393061_dp,&
          65.9914860689_dp,121.3803567467_dp,229.1872120887_dp,446.1810946735_dp,902.6024217645_dp,1941.8011404909_dp,5009.8797981362_dp,0.4669775049_dp,0.5773103204_dp,&
          0.8331017001_dp,1.3155491614_dp,2.1814882044_dp,3.7259467698_dp,6.5076857600_dp,11.6037988956_dp,21.1394644229_dp,39.4249699560_dp,75.4872508046_dp,&
          148.9751731359_dp,304.9770020045_dp,657.7086747979_dp,1587.3127020998_dp,5920.5836116932_dp]
       aw%aw_erange_matrix(:,14) = [0.2318596903_dp,0.7513766684_dp,1.4566226975_dp,2.5381740443_dp,4.3110511476_dp,7.3280974803_dp,12.5953321268_dp,21.9953590831_dp,39.1449985136_dp,&
          71.1874381046_dp,132.6694525539_dp,254.2770222187_dp,503.5763265609_dp,1038.7505242344_dp,2281.9830831059_dp,5995.4201867196_dp,0.4727245350_dp,0.5872949645_dp,&
          0.8539595110_dp,1.3598710730_dp,2.2746638421_dp,3.9205272572_dp,6.9144790809_dp,12.4609259649_dp,22.9699959240_dp,43.4081130047_dp,84.3638508080_dp,&
          169.3475197779_dp,353.4300407609_dp,778.2539995976_dp,1912.5495272325_dp,7170.1862702921_dp]
       aw%aw_erange_matrix(:,15) = [0.2339807060_dp,0.7593601296_dp,1.4762809337_dp,2.5828177926_dp,4.4089425095_dp,7.5384598077_dp,13.0428921163_dp,22.9461450702_dp,41.1765817165_dp,&
          75.5822463322_dp,142.3542055651_dp,276.1554260895_dp,554.5891223527_dp,1162.5126510088_dp,2599.2657303576_dp,6936.9282990729_dp,0.4772216588_dp,0.5951850642_dp,&
          0.8706052760_dp,1.3955573868_dp,2.3503508883_dp,4.0800752099_dp,7.2514580363_dp,13.1790000167_dp,24.5229388830_dp,46.8353673012_dp,92.1257860967_dp,&
          187.4970205995_dp,397.5505430031_dp,890.8446878113_dp,2224.2514849963_dp,8382.1693155268_dp]
       aw%aw_erange_matrix(:,16) = [0.2356793956_dp,0.7657776935_dp,1.4921732725_dp,2.6191360223_dp,4.4890956483_dp,7.7118769938_dp,13.4145537890_dp,23.7420529926_dp,42.8924978643_dp,&
          79.3317990745_dp,150.7127022461_dp,295.2912232099_dp,599.9130242432_dp,1274.5505465661_dp,2892.8155189719_dp,7826.4112995750_dp,0.4808268417_dp,0.6015597181_dp,&
          0.8841592457_dp,1.4248201349_dp,2.4128521625_dp,4.2128183487_dp,7.5341272004_dp,13.7868247088_dp,25.8508308086_dp,49.7997024600_dp,98.9279932108_dp,&
          203.6471318934_dp,437.5258171102_dp,995.0542013082_dp,2519.2792916087_dp,9542.4090958235_dp]
       aw%aw_erange_matrix(:,17) = [0.2371565111_dp,0.7713754938_dp,1.5061014152_dp,2.6511323500_dp,4.5600934351_dp,7.8663613515_dp,13.7476759824_dp,24.4602645785_dp,44.4526453154_dp,&
          82.7702439130_dp,158.4531745840_dp,313.2153957547_dp,642.9446106408_dp,1382.6658986086_dp,3181.5886752821_dp,8718.1400956821_dp,0.4839643244_dp,0.6071435314_dp,&
          0.8961089748_dp,1.4507707931_dp,2.4686060916_dp,4.3319773629_dp,7.7896221881_dp,14.3404299411_dp,27.0706817674_dp,52.5494620743_dp,105.3087850947_dp,&
          218.9962003902_dp,476.1150312763_dp,1097.5429367020_dp,2815.3425231749_dp,10719.5970542372_dp]
       aw%aw_erange_matrix(:,18) = [0.2383791084_dp,0.7760210462_dp,1.5177070330_dp,2.6779122634_dp,4.6197910361_dp,7.9968899954_dp,14.0306238436_dp,25.0738482607_dp,45.7942010596_dp,&
          85.7488108977_dp,165.2155200571_dp,329.0306611577_dp,681.3643289726_dp,1480.5867105295_dp,3447.6614433086_dp,9554.0768038375_dp,0.4865630036_dp,0.6117940844_dp,&
          0.9061164509_dp,1.4726120633_dp,2.5157672599_dp,4.4333150345_dp,8.0081916631_dp,14.8171429018_dp,28.1288792974_dp,54.9549336775_dp,110.9450611137_dp,&
          232.7097323559_dp,511.0662157786_dp,1191.9134190740_dp,3092.9786516246_dp,11835.2685529071_dp]
       aw%aw_erange_matrix(:,19) = [0.2394665902_dp,0.7801625788_dp,1.5280893080_dp,2.7019607355_dp,4.6736119972_dp,8.1150612069_dp,14.2879456072_dp,25.6346618970_dp,47.0272976636_dp,&
          88.5041660059_dp,171.5175017002_dp,343.8975074435_dp,717.8562386955_dp,1574.7785572547_dp,3707.5649743359_dp,10383.6173776339_dp,0.4888758675_dp,0.6159527397_dp,&
          0.9151075636_dp,1.4923189155_dp,2.5585033151_dp,4.5255698396_dp,8.2081862684_dp,15.2558256454_dp,29.1089076738_dp,57.1990378291_dp,116.2478565054_dp,&
          245.7410456839_dp,544.6795257491_dp,1284.0102386442_dp,3368.4543745396_dp,12953.5463215818_dp]
       aw%aw_erange_matrix(:,20) = [0.2439619664_dp,0.7973783256_dp,1.5716141482_dp,2.8037227500_dp,4.9035780994_dp,8.6252020092_dp,15.4113137626_dp,28.1136928196_dp,52.5555501891_dp,&
          101.0585978770_dp,200.7780139937_dp,414.4852435996_dp,895.8803941817_dp,2050.0627104149_dp,5075.6686568281_dp,14954.4580822648_dp,0.4984506753_dp,0.6333689027_dp,&
          0.9531960652_dp,1.5766772529_dp,2.7433912393_dp,4.9292826236_dp,9.0945232101_dp,17.2277759200_dp,33.5858729046_dp,67.6424807184_dp,141.4668848886_dp,&
          309.3410134400_dp,714.0113279167_dp,1766.7070327012_dp,4881.8715715996_dp,19298.8728735009_dp]
       aw%aw_erange_matrix(:,21) = [0.2478279300_dp,0.8123088648_dp,1.6098451410_dp,2.8943658057_dp,5.1114173621_dp,9.0934365297_dp,16.4599619538_dp,30.4718781049_dp,57.9279973917_dp,&
          113.5632674097_dp,230.7738103388_dp,489.3739960868_dp,1092.8458035605_dp,2604.4914515779_dp,6784.1755383131_dp,21126.5938719292_dp,0.5067031432_dp,0.6486421647_dp,&
          0.9871729657_dp,1.6531092746_dp,2.9135882831_dp,5.3073611547_dp,9.9405537607_dp,19.1508072532_dp,38.0595278081_dp,78.3761296950_dp,168.2580432143_dp,&
          379.6405402638_dp,910.5686482995_dp,2362.9849944263_dp,6902.2943103602_dp,28308.3872897900_dp]
       aw%aw_erange_matrix(:,22) = [0.2514110598_dp,0.8262529170_dp,1.6459615928_dp,2.9810777709_dp,5.3128665077_dp,9.5536787342_dp,17.5067289098_dp,32.8670116462_dp,63.4939169303_dp,&
          126.8202428653_dp,263.4528827777_dp,573.6956888284_dp,1323.9079398301_dp,3290.3703991318_dp,9054.9595427138_dp,30108.6774943850_dp,0.5143672124_dp,0.6630486295_dp,&
          1.0197128862_dp,1.7273374322_dp,3.0812675888_dp,5.6857203227_dp,10.8021228802_dp,21.1482386282_dp,42.8128367051_dp,90.0863080323_dp,198.4197507679_dp,&
          461.8721798911_dp,1151.8268578609_dp,3142.5671960808_dp,9776.8791816980_dp,42218.0357732799_dp]
       aw%aw_erange_matrix(:,23) = [0.2528648661_dp,0.8319400523_dp,1.6608070636_dp,3.0170254246_dp,5.3971276906_dp,9.7480354519_dp,17.9534629117_dp,33.9014551716_dp,65.9309509359_dp,&
          132.7183864347_dp,278.2715570858_dp,612.8301683368_dp,1434.3246872856_dp,3630.9670915019_dp,10244.6006180858_dp,35169.9484258880_dp,0.5174810704_dp,0.6689639360_dp,&
          1.0332116800_dp,1.7584224449_dp,3.1521769341_dp,5.8474418989_dp,11.1748131102_dp,22.0240893092_dp,44.9299973167_dp,95.3987073443_dp,212.4065171189_dp,&
          501.0472318215_dp,1270.7717805347_dp,3544.9212020745_dp,11358.6770977554_dp,50438.7867085082_dp]
       aw%aw_erange_matrix(:,24) = [0.2540178287_dp,0.8364626133_dp,1.6726606843_dp,3.0458564268_dp,5.4650232643_dp,9.9054313390_dp,18.3172623041_dp,34.7491988801_dp,67.9427763605_dp,&
          137.6294300508_dp,290.7381148994_dp,646.1741312441_dp,1529.9418440504_dp,3932.3963561222_dp,11330.9240361175_dp,40008.6302545528_dp,0.5199523330_dp,0.6736843844_dp,&
          1.0440414321_dp,1.7834844605_dp,3.2096404326_dp,5.9792371720_dp,11.4804650089_dp,22.7476005967_dp,46.6936240903_dp,99.8680124417_dp,224.3149960761_dp,&
          534.9001420101_dp,1375.5456381947_dp,3908.7242534887_dp,12844.6064444480_dp,58542.0291926889_dp]
       aw%aw_erange_matrix(:,25) = [0.2549866597_dp,0.8402713328_dp,1.6826763795_dp,3.0703054062_dp,5.5228184490_dp,10.0399627163_dp,18.6296349600_dp,35.4808942840_dp,69.6896819013_dp,&
          141.9241704959_dp,301.7339346247_dp,675.8978088757_dp,1616.3466709216_dp,4209.8606318184_dp,12358.3316621397_dp,44778.8527500268_dp,0.5220301394_dp,0.6776710138_dp,&
          1.0532273130_dp,1.8048276749_dp,3.2587821104_dp,6.0924657188_dp,11.7444239751_dp,23.3761442225_dp,48.2363839893_dp,103.8097568816_dp,234.9225954454_dp,&
          565.4312578691_dp,1471.5757374943_dp,4249.6766379633_dp,14284.3739993822_dp,66756.6799545079_dp]
       aw%aw_erange_matrix(:,26) = [0.2557379623_dp,0.8432302665_dp,1.6904785022_dp,3.0894072855_dp,5.5681138327_dp,10.1457517019_dp,18.8761878055_dp,36.0608740066_dp,71.0812157353_dp,&
          145.3652502260_dp,310.6065038413_dp,700.0934038910_dp,1687.4839763069_dp,4441.8642446357_dp,13237.4233285534_dp,49012.9853448564_dp,0.5236421949_dp,0.6807753338_dp,&
          1.0604055083_dp,1.8215607028_dp,3.2974406445_dp,6.1818753132_dp,11.9537421510_dp,23.8770049585_dp,49.4727321296_dp,106.9899559504_dp,243.5512252355_dp,&
          590.5230779693_dp,1551.5664026893_dp,4539.0466303904_dp,15541.6302935884_dp,74231.5026953161_dp]
       aw%aw_erange_matrix(:,27) = [0.2564010066_dp,0.8458454742_dp,1.6973895312_dp,3.1063685052_dp,5.6084358624_dp,10.2401858384_dp,19.0969567918_dp,36.5820317312_dp,72.3367447872_dp,&
          148.4850951264_dp,318.6981956473_dp,722.3220420976_dp,1753.4648026928_dp,4659.8942229397_dp,14080.0786413567_dp,53206.2646298460_dp,0.5250654315_dp,0.6835242044_dp,&
          1.0667801460_dp,1.8364605534_dp,3.3319609425_dp,6.2619625564_dp,12.1418947712_dp,24.3290376869_dp,50.5938079390_dp,109.8898305026_dp,251.4731553408_dp,&
          613.7591553860_dp,1626.4857294511_dp,4814.4131352645_dp,16767.8460227627_dp,81800.7679527001_dp]
       aw%aw_erange_matrix(:,28) = [0.2569579799_dp,0.8480451452_dp,1.7032135982_dp,3.1206920238_dp,5.6425623211_dp,10.3203007309_dp,19.2847495852_dp,37.0266973365_dp,73.4118087721_dp,&
          151.1678172710_dp,325.6920218555_dp,741.6587582025_dp,1811.3449162210_dp,4853.3898389440_dp,14841.2606227166_dp,57109.6782120646_dp,0.5262613911_dp,0.6858400767_dp,&
          1.0721640653_dp,1.8490739264_dp,3.3612548402_dp,6.3301068408_dp,12.3024758825_dp,24.7161813844_dp,51.5578904486_dp,112.3958353629_dp,258.3601782316_dp,&
          634.1127155092_dp,1692.7687392040_dp,5061.4926908256_dp,17892.6585424032_dp,88993.4207075852_dp]
       aw%aw_erange_matrix(:,29) = [0.2601086530_dp,0.8605372769_dp,1.7364832603_dp,3.2030405150_dp,5.8400927889_dp,10.7874442332_dp,20.3888339693_dp,39.6659293840_dp,79.8639942984_dp,&
          167.4841928139_dp,368.9281247815_dp,863.6957532433_dp,2186.7779179576_dp,6158.1308626283_dp,20299.1328844362_dp,88518.0624195757_dp,0.5330336883_dp,0.6990576419_dp,&
          1.1031270006_dp,1.9221308937_dp,3.5321977981_dp,6.7310729459_dp,13.2563078352_dp,27.0409938894_dp,57.4220651473_dp,127.8761716171_dp,301.7231778952_dp,&
          765.4361399106_dp,2134.7178433253_dp,6789.2247063929_dp,26396.7938069641_dp,151820.0966657112_dp]
       aw%aw_erange_matrix(:,30) = [0.2614966852_dp,0.8660674107_dp,1.7513173842_dp,3.2400462999_dp,5.9295952618_dp,11.0010208697_dp,20.8987354919_dp,40.8990199295_dp,82.9197874585_dp,&
          175.3383856309_dp,390.1601646950_dp,925.1616158777_dp,2382.3435121866_dp,6871.1762964399_dp,23520.7435741661_dp,110153.9881294830_dp,0.5360210432_dp,0.7049445153_dp,&
          1.1170457625_dp,1.9552577903_dp,3.6104195243_dp,6.9164166856_dp,13.7023077006_dp,28.1425885947_dp,60.2445555475_dp,135.4684857718_dp,323.4906988269_dp,&
          833.3513502597_dp,2372.6392364149_dp,7775.2837755386_dp,31745.5832863604_dp,200031.9346445932_dp]
       aw%aw_erange_matrix(:,31) = [0.2623927492_dp,0.8696462837_dp,1.7609524952_dp,3.2641782950_dp,5.9882069182_dp,11.1415264621_dp,21.2359218209_dp,41.7193109697_dp,84.9669229685_dp,&
          180.6447640614_dp,404.6557682024_dp,967.6939419883_dp,2520.1487688994_dp,7387.1215399292_dp,25957.6498637107_dp,128242.7476018067_dp,0.5379508276_dp,0.7087659818_dp,&
          1.1261236891_dp,1.9769586430_dp,3.6618994578_dp,7.0390289460_dp,13.9991008027_dp,28.8806875885_dp,62.1510894274_dp,140.6475316153_dp,338.5222526788_dp,&
          881.0021806147_dp,2543.2483567892_dp,8505.7649642908_dp,35939.1316653421_dp,243328.3687408177_dp]
       aw%aw_erange_matrix(:,32) = [0.2627718541_dp,0.8711624407_dp,1.7650423724_dp,3.2744439752_dp,6.0131978634_dp,11.2015882891_dp,21.3804756521_dp,42.0721537208_dp,85.8509744751_dp,&
          182.9472918632_dp,410.9831456456_dp,986.4022841619_dp,2581.3991354344_dp,7619.9982039966_dp,27086.8268950307_dp,137169.1569910549_dp,0.5387675634_dp,0.7103875892_dp,&
          1.1299856469_dp,1.9862130069_dp,3.6839098411_dp,7.0916041097_dp,14.1267854429_dp,29.1994539835_dp,62.9782445560_dp,142.9069894020_dp,345.1258622933_dp,&
          902.1267806327_dp,2619.8368734934_dp,8839.9515605404_dp,37923.2973041788_dp,265681.1476518737_dp]
       aw%aw_erange_matrix(:,33) = [0.2630563241_dp,0.8723010280_dp,1.7681172895_dp,3.2821715705_dp,6.0320340151_dp,11.2469199987_dp,21.4897444333_dp,42.3393399670_dp,86.5218140028_dp,&
          184.6989170267_dp,415.8117268558_dp,1000.7370869878_dp,2628.5923438906_dp,7800.9192070029_dp,27976.7347726542_dp,144460.9017578381_dp,0.5393805485_dp,0.7116065610_dp,&
          1.1328929300_dp,1.9931889395_dp,3.7005241796_dp,7.1313503089_dp,14.2234810120_dp,29.4413473310_dp,63.6074429868_dp,144.6307313309_dp,350.1822634135_dp,&
          918.3799832426_dp,2679.1608813270_dp,9101.4598729715_dp,39504.8220871568_dp,284420.3782975398_dp]
       aw%aw_erange_matrix(:,34) = [0.2631220516_dp,0.8725641954_dp,1.7688284187_dp,3.2839599089_dp,6.0363962050_dp,11.2574261051_dp,21.5150897584_dp,42.4013736123_dp,86.6777403351_dp,&
          185.1066253440_dp,416.9375814523_dp,1004.0867366174_dp,2639.6521074329_dp,7843.5048705938_dp,28187.8193172578_dp,146224.5950682552_dp,0.5395221931_dp,0.7118884373_dp,&
          1.1335657430_dp,1.9948045635_dp,3.7043749873_dp,7.1405702071_dp,14.2459323077_dp,29.4975719842_dp,63.7538844428_dp,145.0325865154_dp,351.3634064255_dp,&
          922.1861683415_dp,2693.1017695894_dp,9163.2515901222_dp,39882.2284393060_dp,289017.1383106848_dp]
       aw%aw_erange_matrix(:,35) = [0.2632430200_dp,0.8730486024_dp,1.7701375706_dp,3.2872526463_dp,6.0444296439_dp,11.2767795705_dp,21.5617948504_dp,42.5157341799_dp,86.9653380131_dp,&
          185.8590678941_dp,419.0170693894_dp,1010.2806915518_dp,2660.1360303686_dp,7922.5633528021_dp,28581.3200056667_dp,149548.2524287735_dp,0.5397828928_dp,0.7124073530_dp,&
          1.1348045430_dp,1.9977798579_dp,3.7114686158_dp,7.1575602956_dp,14.2873222561_dp,29.6012761437_dp,64.0241425762_dp,145.7747325103_dp,353.5470109870_dp,&
          929.2326291739_dp,2718.9608472867_dp,9278.1983512593_dp,40588.0967343748_dp,297748.1292678806_dp]
       aw%aw_erange_matrix(:,36) = [0.2632429401_dp,0.8730483137_dp,1.7701369194_dp,3.2872513050_dp,6.0444269378_dp,11.2767744685_dp,21.5617863587_dp,42.5157223060_dp,86.9653280383_dp,&
          185.8590851153_dp,419.0171719308_dp,1010.2809208136_dp,2660.1364682755_dp,7922.5650536918_dp,28581.3263301165_dp,149548.1582384946_dp,0.5397827248_dp,0.7124070868_dp,&
          1.1348040577_dp,1.9977789035_dp,3.7114667697_dp,7.1575573632_dp,14.2873184980_dp,29.6012739603_dp,64.0241513050_dp,145.7747859397_dp,353.5471227288_dp,&
          929.2327874624_dp,2718.9611614626_dp,9278.2017268714_dp,40588.1035800849_dp,297747.7938530878_dp]
       aw%aw_erange_matrix(:,37) = [0.2632429813_dp,0.8730484658_dp,1.7701372744_dp,3.2872520534_dp,6.0444284162_dp,11.2767770018_dp,21.5617897706_dp,42.5157249296_dp,86.9653208691_dp,&
          185.8590311595_dp,419.0169629233_dp,1010.2803149420_dp,2660.1347243218_dp,7922.5593417351_dp,28581.2923662009_dp,149547.9734559663_dp,0.5397828118_dp,0.7124072317_dp,&
          1.1348043344_dp,1.9977794407_dp,3.7114677002_dp,7.1575584514_dp,14.2873190018_dp,29.6012708257_dp,64.0241309025_dp,145.7747002374_dp,353.5468820873_dp,&
          929.2321366837_dp,2718.9594320326_dp,9278.1929046266_dp,40588.0340115379_dp,297747.5613474760_dp]            
    case(18)
       aw%energy_range(:) = [100.0000_dp,200.0000_dp,300.0000_dp,400.0000_dp,500.0000_dp,600.0000_dp,700.0000_dp,800.0000_dp,900.0000_dp,&
          1000.0000_dp,2000.0000_dp,3000.0000_dp,4000.0000_dp,5000.0000_dp,6000.0000_dp,8000.0000_dp,9000.0000_dp,10000.0000_dp,&
          20000.0000_dp,30000.0000_dp,40000.0000_dp,50000.0000_dp,60000.0000_dp,70000.0000_dp,80000.0000_dp,100000.0000_dp,200000.0000_dp,&
          300000.0000_dp,400000.0000_dp,500000.0000_dp,600000.0000_dp,700000.0000_dp,1000000.0000_dp,2000000.0000_dp,3000000.0000_dp,4000000.0000_dp,5000000.0000_dp&
          ]
       aw%aw_erange_matrix(:,1) = [0.1527981223_dp,0.4736735191_dp,0.8422879268_dp,1.2969949102_dp,1.8866941753_dp,2.6768243548_dp,3.7576314748_dp,5.2560227000_dp,7.3529844088_dp,&
          10.3098202106_dp,14.5091054561_dp,20.5224221475_dp,29.2333380207_dp,42.0933873501_dp,61.7573643645_dp,94.0319878961_dp,155.8524525586_dp,331.6666454386_dp,&
          0.3081114816_dp,0.3389838301_dp,0.4045613209_dp,0.5129049024_dp,0.6772652660_dp,0.9178430315_dp,1.2646393907_dp,1.7618830445_dp,2.4749338974_dp,&
          3.5014291571_dp,4.9905064808_dp,7.1795753170_dp,10.4751687346_dp,15.6616554134_dp,24.5415634965_dp,42.3610859910_dp,90.6499809216_dp,345.5812203285_dp]
       aw%aw_erange_matrix(:,2) = [0.1630595518_dp,0.5078454432_dp,0.9113242587_dp,1.4219143796_dp,2.1032373061_dp,3.0432092324_dp,4.3670504641_dp,6.2565617694_dp,8.9792704978_dp,&
          12.9333140459_dp,18.7182748971_dp,27.2516047571_dp,39.9742332838_dp,59.2607021773_dp,89.4001916667_dp,139.5646279037_dp,236.0119955160_dp,508.8712615468_dp,&
          0.3291860241_dp,0.3669636069_dp,0.4479530140_dp,0.5836830921_dp,0.7934040217_dp,1.1070968212_dp,1.5704132532_dp,2.2524718420_dp,3.2581634715_dp,&
          4.7480031887_dp,6.9716245615_dp,10.3290234975_dp,15.4980119683_dp,23.7506428902_dp,37.9159495927_dp,66.0684302416_dp,141.2646651462_dp,534.8041524444_dp]
       aw%aw_erange_matrix(:,3) = [0.1705157849_dp,0.5329774509_dp,0.9631534900_dp,1.5180492354_dp,2.2742864325_dp,3.3403090241_dp,4.8743423617_dp,7.1117069959_dp,10.4071362841_dp,&
          15.3013839295_dp,22.6281315821_dp,33.6909746028_dp,50.5733327575_dp,76.7403326936_dp,118.4300130927_dp,188.7754427627_dp,324.7362218774_dp,708.1501449317_dp,&
          0.3445471260_dp,0.3879712607_dp,0.4817375016_dp,0.6406265045_dp,0.8896307917_dp,1.2683429214_dp,1.8382595742_dp,2.6945175568_dp,3.9847496713_dp,&
          5.9398737665_dp,8.9260338600_dp,13.5377629091_dp,20.7845131696_dp,32.5347358248_dp,52.8383624414_dp,93.0090646173_dp,199.0924806088_dp,749.7941673117_dp]
       aw%aw_erange_matrix(:,4) = [0.1753724416_dp,0.5494915993_dp,0.9977173837_dp,1.5832956162_dp,2.3925275383_dp,3.5495056900_dp,5.2382028596_dp,7.7366662011_dp,11.4709080813_dp,&
          17.1011211083_dp,25.6620404322_dp,38.7975529653_dp,59.1717536234_dp,91.2571770104_dp,143.1147450183_dp,231.5674047467_dp,403.3643728158_dp,887.0367390720_dp,&
          0.3545753722_dp,0.4019794831_dp,0.5048425080_dp,0.6804591848_dp,0.9583294358_dp,1.3857363627_dp,2.0371295696_dp,3.0294111303_dp,4.5468532602_dp,&
          6.8822916295_dp,10.5070740928_dp,16.1959498886_dp,25.2721771652_dp,40.1736865810_dp,66.1018252049_dp,117.3386399133_dp,251.6255820785_dp,944.3885629150_dp]
       aw%aw_erange_matrix(:,5) = [0.1792634140_dp,0.5628074380_dp,1.0258882361_dp,1.6371516574_dp,2.4914222321_dp,3.7268076313_dp,5.5507285118_dp,8.2808021364_dp,12.4102029358_dp,&
          18.7137008125_dp,28.4225509460_dp,43.5198027864_dp,67.2597852973_dp,105.1576978013_dp,167.1846844824_dp,274.0316583356_dp,482.5800516279_dp,1069.1455217709_dp,&
          0.3626230022_dp,0.4133949264_dp,0.5240129938_dp,0.7140417682_dp,1.0170969827_dp,1.4875843790_dp,2.2121408831_dp,3.3284979466_dp,5.0566306497_dp,&
          7.7508728374_dp,11.9891683644_dp,18.7325758390_dp,29.6347190103_dp,47.7398012261_dp,79.4699929121_dp,142.1908163494_dp,305.5941129510_dp,1143.8185002949_dp]
       aw%aw_erange_matrix(:,6) = [0.1824275671_dp,0.5736934774_dp,1.0491222851_dp,1.6820304473_dp,2.5747196681_dp,3.8777631059_dp,5.8197221117_dp,8.7543843212_dp,13.2371920221_dp,&
          20.1507264819_dp,30.9140562606_dp,47.8396956470_dp,74.7650047430_dp,118.2516977816_dp,190.2115173141_dp,315.2750004868_dp,560.5441858754_dp,1250.0374608106_dp,&
          0.3691763555_dp,0.4228081801_dp,0.5400524031_dp,0.7425032375_dp,1.0674915506_dp,1.5759280281_dp,2.3657256452_dp,3.5941590911_dp,5.5152047015_dp,&
          8.5426973513_dp,13.3594489834_dp,21.1129759784_dp,33.7929507589_dp,55.0675461174_dp,92.6154883865_dp,166.9270513977_dp,359.6149740839_dp,1343.0921742095_dp]
       aw%aw_erange_matrix(:,7) = [0.1845402510_dp,0.5809913394_dp,1.0648021114_dp,1.7125536673_dp,2.6318299464_dp,3.9821018390_dp,6.0071758416_dp,9.0871950403_dp,13.8234621614_dp,&
          21.1788583886_dp,32.7140056593_dp,50.9928543565_dp,80.3035192335_dp,128.0271172500_dp,207.6104411774_dp,346.8108778212_dp,620.7887994110_dp,1390.8547791118_dp,&
          0.3735565554_dp,0.4291597901_dp,0.5509930923_dp,0.7621042619_dp,1.1025040768_dp,1.6378374763_dp,2.4743053650_dp,3.7836997749_dp,5.8455413731_dp,&
          9.1189076723_dp,14.3673717310_dp,22.8839631506_dp,36.9240327578_dp,60.6541254041_dp,102.7587257895_dp,186.2028175932_dp,401.9170351636_dp,1498.9565722365_dp]
       aw%aw_erange_matrix(:,8) = [0.1868028095_dp,0.5888334464_dp,1.0817455973_dp,1.7457516566_dp,2.6943634927_dp,4.0971241083_dp,6.2152460671_dp,9.4592277517_dp,14.4836733720_dp,&
          22.3456860772_dp,34.7736661548_dp,54.6327852171_dp,86.7572435755_dp,139.5319952300_dp,228.3021868469_dp,384.7072757845_dp,693.8618958579_dp,1562.7906525256_dp,&
          0.3782516137_dp,0.4360221814_dp,0.5629206924_dp,0.7836442748_dp,1.1412631241_dp,1.7068676743_dp,2.5962717093_dp,3.9982573034_dp,6.2225285063_dp,&
          9.7821717658_dp,15.5382258481_dp,24.9613535034_dp,40.6349607723_dp,67.3468998832_dp,115.0396342009_dp,209.7488892780_dp,453.8302508246_dp,1690.0724518598_dp]
       aw%aw_erange_matrix(:,9) = [0.1889192323_dp,0.5961942268_dp,1.0977388657_dp,1.7772929339_dp,2.7541781755_dp,4.2078951751_dp,6.4170137568_dp,9.8225699892_dp,15.1332825355_dp,&
          23.5028395309_dp,36.8333949661_dp,58.3054821199_dp,93.3316171825_dp,151.3722280049_dp,249.8269779137_dp,424.5573643490_dp,771.4557948840_dp,1746.6429947131_dp,&
          0.3826473328_dp,0.4424985483_dp,0.5742792596_dp,0.8043204163_dp,1.1787417405_dp,1.7741030137_dp,2.7159546043_dp,4.2104439185_dp,6.5984257859_dp,&
          10.4493076166_dp,16.7268978412_dp,27.0913648258_dp,44.4802785076_dp,74.3592538063_dp,128.0493684396_dp,234.9282829482_dp,509.6295674211_dp,1895.3494531455_dp]
       aw%aw_erange_matrix(:,10) = [0.1903879496_dp,0.6013168240_dp,1.1089208087_dp,1.7994640492_dp,2.7964566007_dp,4.2866286979_dp,6.5612419722_dp,10.0838243357_dp,15.6032531027_dp,&
          24.3454652121_dp,38.3436918964_dp,61.0185260176_dp,98.2269380823_dp,160.2640090902_dp,266.1379252521_dp,455.0313652623_dp,831.2881474395_dp,1889.2621396318_dp,&
          0.3857000407_dp,0.4470258630_dp,0.5822782434_dp,0.8189759799_dp,1.2054673384_dp,1.8223342042_dp,2.8023370561_dp,4.3645792132_dp,6.8733432774_dp,&
          10.9407585128_dp,17.6093055967_dp,28.6856426725_dp,47.3838669100_dp,79.7036000150_dp,138.0572536990_dp,254.4553065018_dp,553.1007480388_dp,2055.1987987535_dp]
       aw%aw_erange_matrix(:,11) = [0.2004773481_dp,0.6368375672_dp,1.1876500661_dp,1.9583390236_dp,3.1049903474_dp,4.8719302829_dp,7.6540035084_dp,12.1029440095_dp,19.3126222915_dp,&
          31.1473126488_dp,50.8338732988_dp,84.0516485483_dp,140.9901400583_dp,240.3905954302_dp,418.1349129991_dp,749.0950031399_dp,1427.9283160510_dp,3346.5458889158_dp,&
          0.4067217569_dp,0.4788809259_dp,0.6399266301_dp,0.9268650214_dp,1.4061641204_dp,2.1918280699_dp,3.4779960725_dp,5.5969933640_dp,9.1237199299_dp,&
          15.0663243585_dp,25.2211938823_dp,42.8502473564_dp,74.0218425104_dp,130.4575379986_dp,236.5689891584_dp,453.0888523964_dp,1004.4155925382_dp,3714.2671841220_dp]
       aw%aw_erange_matrix(:,12) = [0.2060923936_dp,0.6568656934_dp,1.2329832605_dp,2.0520357907_dp,3.2914706387_dp,5.2345868421_dp,8.3485657989_dp,13.4209241076_dp,21.8030961178_dp,&
          35.8535966396_dp,59.7602909927_dp,101.0994830552_dp,173.8694957238_dp,304.6097011437_dp,545.5713806587_dp,1007.6354663732_dp,1977.0084005764_dp,4735.0152742876_dp,&
          0.4184608131_dp,0.4972036411_dp,0.6741675114_dp,0.9927793132_dp,1.5320651526_dp,2.4298648191_dp,3.9254906978_dp,6.4374740045_dp,10.7070328659_dp,&
          18.0675541150_dp,30.9613891161_dp,53.9559505932_dp,95.8098141785_dp,173.9182100598_dp,325.1068088959_dp,640.0565130514_dp,1442.8622811605_dp,5330.2153116126_dp]
       aw%aw_erange_matrix(:,13) = [0.2094752506_dp,0.6690253334_dp,1.2608469589_dp,2.1104329482_dp,3.4093675338_dp,5.4672211845_dp,8.8008565440_dp,14.2928786861_dp,23.4788549362_dp,&
          39.0785473296_dp,65.9992861070_dp,113.2752218097_dp,197.9172373939_dp,352.8303735122_dp,644.0840718013_dp,1213.8952851159_dp,2429.0637615777_dp,5906.9084339158_dp,&
          0.4255474045_dp,0.5084570324_dp,0.6955884802_dp,1.0346890513_dp,1.6133572157_dp,2.5859849971_dp,4.2238534821_dp,7.0077645581_dp,11.8017672094_dp,&
          20.1852821928_dp,35.1022677213_dp,62.1633910973_dp,112.3459056401_dp,207.8857691341_dp,396.5505381224_dp,795.8587346363_dp,1817.2294197627_dp,6715.7635445298_dp]
       aw%aw_erange_matrix(:,14) = [0.2121581617_dp,0.6787202020_dp,1.2832499726_dp,2.1578321904_dp,3.5059957828_dp,5.6597904823_dp,9.1791392965_dp,15.0301826515_dp,24.9125941632_dp,&
          41.8731378865_dp,71.4815357600_dp,124.1393652143_dp,219.7421301739_dp,397.4317148687_dp,737.1595777719_dp,1413.3997553296_dp,2876.9782299657_dp,7091.0678107240_dp,&
          0.4311754820_dp,0.5174998412_dp,0.7130167803_dp,1.0691630744_dp,1.6809311462_dp,2.7171607638_dp,4.4774063857_dp,7.4983357144_dp,12.7559194999_dp,&
          22.0575880369_dp,38.8208607896_dp,69.6617108245_dp,127.7446017458_dp,240.1988094408_dp,466.1397468408_dp,951.4027569398_dp,2198.4961458203_dp,8133.3203336836_dp]
       aw%aw_erange_matrix(:,15) = [0.2139921648_dp,0.6853739849_dp,1.2987228567_dp,2.1908018772_dp,3.5736987910_dp,5.7957235572_dp,9.4482495044_dp,15.5590537094_dp,25.9502231069_dp,&
          43.9153424776_dp,75.5306015660_dp,132.2581172941_dp,236.2663898368_dp,431.6998885572_dp,809.8664998975_dp,1572.1602361118_dp,3240.3954623280_dp,8067.4677286845_dp,&
          0.4350267749_dp,0.5237425120_dp,0.7251601402_dp,1.0933802360_dp,1.7287741808_dp,2.8107871087_dp,4.6599346945_dp,7.8547530404_dp,13.4560860148_dp,&
          23.4465065014_dp,41.6124750965_dp,75.3655610751_dp,139.6318053004_dp,265.5594759721_dp,521.7790434550_dp,1078.2496818124_dp,2514.6604163544_dp,9314.1638997611_dp]
       aw%aw_erange_matrix(:,16) = [0.2178151179_dp,0.6993140505_dp,1.3313988591_dp,2.2610539640_dp,3.7192949294_dp,6.0908275329_dp,10.0383009651_dp,16.7310269976_dp,28.2761942458_dp,&
          48.5511764282_dp,84.8506107486_dp,151.2366932549_dp,275.5700358601_dp,514.8276950982_dp,990.2516885162_dp,1976.2621831459_dp,4191.2288257724_dp,10683.1908182285_dp,&
          0.4430653606_dp,0.5369178579_dp,0.7510882812_dp,1.1456226846_dp,1.8330157272_dp,3.0168938111_dp,5.0661839848_dp,8.6574888928_dp,15.0535279971_dp,&
          26.6605642029_dp,48.1744338564_dp,89.0091722681_dp,168.6305683277_dp,328.8251790857_dp,664.1592767767_dp,1412.0406993233_dp,3367.6131241992_dp,12525.3812822271_dp]
       aw%aw_erange_matrix(:,17) = [0.2188081567_dp,0.7029508216_dp,1.3399817439_dp,2.2796476603_dp,3.7581319758_dp,6.1701789054_dp,10.1983028044_dp,17.0517005110_dp,28.9188643373_dp,&
          49.8458188665_dp,87.4842442568_dp,156.6703747359_dp,286.9896650454_dp,539.3866234032_dp,1044.5712481684_dp,2100.6359589853_dp,4490.8966264193_dp,11524.8126775818_dp,&
          0.4451558102_dp,0.5403766820_dp,0.7579621105_dp,1.1595935419_dp,1.8611279523_dp,3.0729639927_dp,5.1777339300_dp,8.8801319309_dp,15.5014537915_dp,&
          27.5726472053_dp,50.0613660602_dp,92.9907021237_dp,177.2345753094_dp,347.9535003042_dp,708.1450322595_dp,1517.6510781274_dp,3643.4817836702_dp,13572.1857422280_dp]
       aw%aw_erange_matrix(:,18) = [0.2197828445_dp,0.7065268576_dp,1.3484452335_dp,2.2980407479_dp,3.7966747852_dp,6.2491922090_dp,10.3581829359_dp,17.3733379970_dp,29.5661089226_dp,&
          51.1555523006_dp,90.1619002253_dp,162.2258144685_dp,298.7390660052_dp,564.8375634919_dp,1101.3350723375_dp,2231.8691674585_dp,4810.4999157353_dp,12431.0893265124_dp,&
          0.4472086016_dp,0.5437865853_dp,0.7647663233_dp,1.1734727600_dp,1.8891535196_dp,3.1290645559_dp,5.2897782356_dp,9.1047033174_dp,15.9553383683_dp,&
          28.5015525163_dp,51.9939423727_dp,97.0943433317_dp,186.1660494056_dp,367.9733742722_dp,754.6199482807_dp,1630.4422765799_dp,3941.1471257651_dp,14706.2839701511_dp]
       aw%aw_erange_matrix(:,19) = [0.2265068924_dp,0.7313723956_dp,1.4079029715_dp,2.4288666490_dp,4.0743494431_dp,6.8260310413_dp,11.5419454962_dp,19.7913469464_dp,34.5140986987_dp,&
          61.3557522089_dp,111.4538331601_dp,207.4531772156_dp,397.0044159553_dp,784.4559544064_dp,1609.5435792804_dp,3459.5173734479_dp,7955.7224057844_dp,21787.7872792344_dp,&
          0.4613964399_dp,0.5677182986_dp,0.8132820575_dp,1.2738440119_dp,2.0946622147_dp,3.5464814496_dp,6.1366541222_dp,10.8315244763_dp,19.5122061582_dp,&
          35.9359276337_dp,67.8312378331_dp,131.6382289194_dp,263.7064908785_dp,548.1543124891_dp,1191.1152093944_dp,2744.1365262322_dp,7034.9730156755_dp,26779.6427117294_dp]
       aw%aw_erange_matrix(:,20) = [0.2299046347_dp,0.7440466445_dp,1.4386817413_dp,2.4977004608_dp,4.2229117582_dp,7.1400486674_dp,12.1983208840_dp,21.1590457313_dp,37.3746256619_dp,&
          67.3972149915_dp,124.4113415746_dp,235.8339174556_dp,460.8695878332_dp,933.1280623325_dp,1970.5451213787_dp,4383.4382370187_dp,10490.0415539656_dp,29853.7138953906_dp,&
          0.4685835491_dp,0.5800895256_dp,0.8388841363_dp,1.3277922637_dp,2.2071337130_dp,3.7792979087_dp,6.6187541544_dp,11.8367180574_dp,21.6341446873_dp,&
          40.4936762994_dp,77.8419038960_dp,154.2432177310_dp,316.5085486601_dp,676.6859045395_dp,1520.1522622007_dp,3640.7662046546_dp,9709.9025797711_dp,37642.0149894778_dp]
       aw%aw_erange_matrix(:,21) = [0.2322426137_dp,0.7528155675_dp,1.4601566795_dp,2.5461768342_dp,4.3285472611_dp,7.3655789295_dp,12.6748091407_dp,22.1635689596_dp,39.5028735027_dp,&
          71.9574649193_dp,134.3529357777_dp,258.0191755122_dp,511.8853109587_dp,1054.9582203366_dp,2275.6093922057_dp,5194.4156325965_dp,12821.7854876095_dp,37657.6423999949_dp,&
          0.4735360781_dp,0.5887137683_dp,0.8569421263_dp,1.3662449059_dp,2.2881386212_dp,3.9488343848_dp,6.9740393893_dp,12.5872963374_dp,23.2418684237_dp,&
          44.0038125005_dp,85.6955724242_dp,172.3579359302_dp,359.8809360759_dp,785.4069036400_dp,1808.5700733323_dp,4462.1506943232_dp,12290.4942808444_dp,48494.3206443843_dp]
       aw%aw_erange_matrix(:,22) = [0.2336768436_dp,0.7582143877_dp,1.4734520876_dp,2.5763743626_dp,4.3947703836_dp,7.5079066797_dp,12.9776624895_dp,22.8070387800_dp,40.8780295306_dp,&
          74.9328525019_dp,140.9115314301_dp,272.8423532693_dp,546.4815978838_dp,1139.0497298808_dp,2490.7560668022_dp,5782.0074376657_dp,14570.4471420771_dp,43741.3576559383_dp,&
          0.4765770918_dp,0.5940500004_dp,0.8682017317_dp,1.3903872862_dp,2.3393490281_dp,4.0568008584_dp,7.2021080821_dp,13.0733720716_dp,24.2932834607_dp,&
          46.3248131822_dp,90.9542469493_dp,184.6642780706_dp,389.8515787306_dp,862.0796631928_dp,2017.1246766216_dp,5075.1705599863_dp,14292.7434800951_dp,57167.3354554488_dp]
       aw%aw_erange_matrix(:,23) = [0.2349711741_dp,0.7630994958_dp,1.4855312811_dp,2.6039326902_dp,4.4554861518_dp,7.6390323637_dp,13.2581387984_dp,23.4063874175_dp,42.1670929633_dp,&
          77.7420965786_dp,147.1549158039_dp,287.0876041224_dp,580.1016244922_dp,1221.8669259734_dp,2706.1509321370_dp,6382.6921105031_dp,16407.5280492903_dp,50341.3630128390_dp,&
          0.4793233813_dp,0.5988959516_dp,0.8784840550_dp,1.4125452718_dp,2.3865869706_dp,4.1569264641_dp,7.4148536739_dp,13.5297327259_dp,25.2875843255_dp,&
          48.5377313573_dp,96.0150836930_dp,196.6366941974_dp,419.3845618102_dp,938.8088067222_dp,2229.8751236449_dp,5716.1223474154_dp,16452.9800581583_dp,66773.6908077037_dp]
       aw%aw_erange_matrix(:,24) = [0.2362056864_dp,0.7677702998_dp,1.4971240753_dp,2.6304912868_dp,4.5142511221_dp,7.7665188164_dp,13.5321609484_dp,23.9950874146_dp,43.4408449937_dp,&
          80.5367989944_dp,153.4142279527_dp,301.4979966589_dp,614.4734307780_dp,1307.6229833800_dp,2932.7423350003_dp,7027.5640383226_dp,18433.7806581014_dp,57865.4795639180_dp,&
          0.4819444399_dp,0.6035448238_dp,0.8883992010_dp,1.4340115990_dp,2.4325645800_dp,4.2548680283_dp,7.6240978812_dp,13.9813064760_dp,26.2781501086_dp,&
          50.7593479718_dp,101.1408282912_dp,208.8878226568_dp,449.9752198065_dp,1019.4696542107_dp,2457.7060905133_dp,6419.2205665980_dp,18898.5337435476_dp,77962.5285029020_dp]
       aw%aw_erange_matrix(:,25) = [0.2373811804_dp,0.7722283067_dp,1.5082286277_dp,2.6560325699_dp,4.5709979484_dp,7.8901603872_dp,13.7991626271_dp,24.5716663778_dp,44.6955816594_dp,&
          83.3078107108_dp,159.6671289369_dp,316.0196927701_dp,649.4705607609_dp,1396.0394640284_dp,3170.0394143638_dp,7716.7206819492_dp,20659.4302682909_dp,66423.1120212093_dp,&
          0.4844417397_dp,0.6079960880_dp,0.8979397304_dp,1.4547589152_dp,2.4772012954_dp,4.3504088347_dp,7.8292862308_dp,14.4267153785_dp,27.2616042688_dp,&
          52.9814469879_dp,106.3116632913_dp,221.3708125829_dp,481.5183314923_dp,1103.8598269916_dp,2700.4771705858_dp,7186.6895812688_dp,21655.8394871780_dp,90981.2260411802_dp]
       aw%aw_erange_matrix(:,26) = [0.2384929321_dp,0.7764541060_dp,1.5187910501_dp,2.6804190928_dp,4.6253918957_dp,8.0091653367_dp,14.0573010465_dp,25.1318593329_dp,45.9214125315_dp,&
          86.0320248361_dp,165.8589709257_dp,330.5213437011_dp,684.7716456399_dp,1486.3194127934_dp,3416.0807835360_dp,8445.7682576954_dp,23080.0500376109_dp,76077.3463064641_dp,&
          0.4868050225_dp,0.6122283656_dp,0.9070534966_dp,1.4746621676_dp,2.5202048636_dp,4.4428752109_dp,8.0288699041_dp,14.8623807033_dp,28.2295988616_dp,&
          55.1842978329_dp,111.4801345970_dp,233.9693080953_dp,513.7236927079_dp,1191.2537920844_dp,2956.4629917081_dp,8015.5672953765_dp,24733.8193867627_dp,106025.7763539200_dp]
       aw%aw_erange_matrix(:,27) = [0.2424066958_dp,0.7914046541_dp,1.5564443104_dp,2.7680823572_dp,4.8226295707_dp,8.4446731203_dp,15.0114711050_dp,27.2256530508_dp,50.5608704286_dp,&
          96.4912212100_dp,190.0298069297_dp,388.2508224812_dp,828.6545663149_dp,1865.1697805908_dp,4487.8156685859_dp,11785.0441837860_dp,35002.1836037296_dp,128962.1250768669_dp,&
          0.4951355072_dp,0.6273021161_dp,0.9398489002_dp,1.5469558320_dp,2.6778932564_dp,4.7854195711_dp,8.7766219472_dp,16.5153607224_dp,31.9551714346_dp,&
          63.8025814509_dp,132.0903298321_dp,285.3568780777_dp,648.7347131305_dp,1570.3680700535_dp,4117.2842170109_dp,12010.0206109856_dp,40950.7873618795_dp,194612.8629584392_dp]
       aw%aw_erange_matrix(:,28) = [0.2451483463_dp,0.8019477304_dp,1.5832664989_dp,2.8312249320_dp,4.9663414572_dp,8.7658865407_dp,15.7246398040_dp,28.8139006699_dp,54.1393916659_dp,&
          104.7144822098_dp,209.4610558075_dp,435.8986297272_dp,951.2536527997_dp,2201.0294283516_dp,5487.9035786263_dp,15127.5077224134_dp,48255.1157456922_dp,198725.3328400740_dp,&
          2.0039254651_dp,2.5521059930_dp,3.8540019382_dp,6.3989622653_dp,11.1779286325_dp,20.1685457731_dp,37.3818682463_dp,71.1764172917_dp,139.5702311295_dp,&
          282.9901494567_dp,596.5096927657_dp,1316.5702025323_dp,3072.3924881061_dp,7685.7784941115_dp,21038.1983744905_dp,65151.8935038286_dp,242901.2154403277_dp,1303987.3158731237_dp]
       aw%aw_erange_matrix(:,29) = [0.2451584911_dp,0.8019868539_dp,1.5833664624_dp,2.8314613818_dp,4.9668822755_dp,8.7671015990_dp,15.7273526805_dp,28.8199804209_dp,54.1531881809_dp,&
          104.7464454026_dp,209.5372970267_dp,436.0876685648_dp,951.7466778391_dp,2202.4031805614_dp,5492.0861687473_dp,15141.9292444349_dp,48315.1164798901_dp,199069.2488759779_dp,&
          0.5010030137_dp,0.6380664445_dp,0.9635890952_dp,1.5999393770_dp,2.7949236246_dp,5.0431140248_dp,9.3476469670_dp,17.7990399522_dp,34.9039874892_dp,&
          70.7748108912_dp,149.1950441262_dp,329.3185677466_dp,768.5855483743_dp,1922.9069771868_dp,5264.4349177856_dp,16306.9754356859_dp,60818.5342949878_dp,326683.0762412191_dp]
       aw%aw_erange_matrix(:,30) = [0.2460843831_dp,0.8055606841_dp,1.5925097532_dp,2.8531188820_dp,5.0164916215_dp,8.8787427624_dp,15.9770766339_dp,29.3807725841_dp,55.4287162064_dp,&
          107.7095964209_dp,216.6287632197_dp,453.7426000720_dp,998.0221583737_dp,2332.1646550623_dp,5890.4995870220_dp,16531.9236622109_dp,54208.2506864180_dp,234158.2961103953_dp,&
          0.5029791750_dp,0.6417196055_dp,0.9717065651_dp,1.6181802341_dp,2.8354963081_dp,5.1331308781_dp,9.5487949715_dp,18.2555079161_dp,35.9638756655_dp,&
          73.3120612863_dp,155.5108113290_dp,345.8365421606_dp,814.5932497073_dp,2061.9760197334_dp,5733.6867171101_dp,18158.2802893164_dp,70123.8908441199_dp,398633.9384505207_dp]
       aw%aw_erange_matrix(:,31) = [0.2466623194_dp,0.8077948498_dp,1.5982387997_dp,2.8667234346_dp,5.0477367999_dp,8.9492548457_dp,16.1352880115_dp,29.7372894100_dp,56.2428185727_dp,&
          109.6094612719_dp,221.1999709041_dp,465.1963104174_dp,1028.2814580196_dp,2417.8718375738_dp,6157.2042805604_dp,17480.4794754592_dp,58357.0349287896_dp,260473.4413372097_dp,&
          0.5042131777_dp,0.6440079330_dp,0.9768069389_dp,1.6296735867_dp,2.8611344928_dp,5.1901916339_dp,9.6767456775_dp,18.5470129577_dp,36.6437851669_dp,&
          74.9482070859_dp,159.6086502713_dp,356.6335737870_dp,844.9445707055_dp,2154.8129629221_dp,6052.0114227348_dp,19444.0387880265_dp,76846.9611556469_dp,455064.7940782208_dp]
       aw%aw_erange_matrix(:,32) = [0.2470052603_dp,0.8091218154_dp,1.6016463057_dp,2.8748276155_dp,5.0663795081_dp,8.9913991392_dp,16.2300284650_dp,29.9512323720_dp,56.7325330921_dp,&
          110.7554906838_dp,223.9664660987_dp,472.1556042268_dp,1046.7571381205_dp,2470.5309237515_dp,6322.4522482374_dp,18075.3893191007_dp,61011.6261370216_dp,278043.4319918001_dp,&
          0.5049456007_dp,0.6453687263_dp,0.9798456467_dp,1.6365329129_dp,2.8764626879_dp,5.2243720830_dp,9.7535545588_dp,18.7224245427_dp,37.0540406668_dp,&
          75.9386048198_dp,162.0985889041_dp,363.2242952357_dp,863.5775619736_dp,2212.2293119703_dp,6250.8786633660_dp,20259.3672008996_dp,81220.2624367206_dp,493903.7336261885_dp]
       aw%aw_erange_matrix(:,33) = [0.2479070591_dp,0.8126156863_dp,1.6106355109_dp,2.8962520852_dp,5.1157722654_dp,9.1033195075_dp,16.4822737221_dp,30.5225047448_dp,58.0445152466_dp,&
          113.8376373968_dp,231.4406236551_dp,491.0606539737_dp,1097.2889246287_dp,2615.8193585517_dp,6783.7921251546_dp,19765.1665816340_dp,68773.2087073112_dp,332879.9902833110_dp,&
          0.5068722333_dp,0.6489576683_dp,0.9878804553_dp,1.6547124799_dp,2.9171851328_dp,5.3154164832_dp,9.9587430808_dp,19.1925744522_dp,38.1578179311_dp,&
          78.6150396384_dp,168.8626882537_dp,381.2426698660_dp,914.9252925831_dp,2372.0982841524_dp,6812.5177792103_dp,22611.4620307756_dp,94313.9725576750_dp,621003.7893413518_dp]
       aw%aw_erange_matrix(:,34) = [0.2488445564_dp,0.8162546763_dp,1.6200245149_dp,2.9186987450_dp,5.1676891801_dp,9.2213672091_dp,16.7493449509_dp,31.1299547030_dp,59.4464473127_dp,&
          117.1499428429_dp,239.5273357413_dp,511.6827842253_dp,1152.9725592643_dp,2778.0302179810_dp,7308.1217912983_dp,21736.8287450385_dp,78245.3963530631_dp,407389.7568922244_dp,&
          0.5088761210_dp,0.6527048135_dp,0.9963009145_dp,1.6738302826_dp,2.9601613259_dp,5.4118721369_dp,10.1770679896_dp,19.6952763247_dp,39.3446624314_dp,&
          81.5118069612_dp,176.2408673541_dp,401.0839570649_dp,972.1435868244_dp,2553.0292718077_dp,7461.9282580860_dp,25420.7597614315_dp,110881.6155222195_dp,807717.4768339451_dp]
       aw%aw_erange_matrix(:,35) = [0.2489027717_dp,0.8164809134_dp,1.6206092698_dp,2.9200994031_dp,5.1709349593_dp,9.2287620348_dp,16.7661108868_dp,31.1681785725_dp,59.5348987369_dp,&
          117.3595707997_dp,240.0409997763_dp,512.9984965777_dp,1156.5449506809_dp,2788.5118341189_dp,7342.3362280433_dp,21867.3537661850_dp,78888.3507304854_dp,412773.3816334762_dp,&
          0.5090005943_dp,0.6529381345_dp,0.9968264487_dp,1.6750258908_dp,2.9628544011_dp,5.4179295425_dp,10.1908113350_dp,19.7270039685_dp,39.4197954981_dp,&
          81.6958381899_dp,176.7115790119_dp,402.3563056081_dp,975.8367478350_dp,2564.8072390811_dp,7504.7021546451_dp,25609.0943418721_dp,112028.7196634320_dp,821824.7982251805_dp]
       aw%aw_erange_matrix(:,36) = [0.2489028793_dp,0.8164812826_dp,1.6206100265_dp,2.9201007372_dp,5.1709371443_dp,9.2287653856_dp,16.7661154320_dp,31.1681839603_dp,59.5349042318_dp,&
          117.3595737483_dp,240.0409935634_dp,512.9985002870_dp,1156.5449881503_dp,2788.5116971549_dp,7342.3361726431_dp,21867.3597042521_dp,78888.3522147178_dp,412773.5597732165_dp,&
          0.5090008175_dp,0.6529384478_dp,0.9968269205_dp,1.6750265874_dp,2.9628554209_dp,5.4179308019_dp,10.1908123913_dp,19.7270045689_dp,39.4197948530_dp,&
          81.6958325873_dp,176.7115700855_dp,402.3563424594_dp,975.8367333780_dp,2564.8068659078_dp,7504.7037477325_dp,25609.1035816155_dp,112028.6869791661_dp,821825.9737785851_dp]
       aw%aw_erange_matrix(:,37) = [0.2489028079_dp,0.8164810378_dp,1.6206095245_dp,2.9200998161_dp,5.1709354667_dp,9.2287624833_dp,16.7661110594_dp,31.1681786578_dp,59.5349016573_dp,&
          117.3595876258_dp,240.0410761551_dp,512.9988191375_dp,1156.5459366962_dp,2788.5140947174_dp,7342.3407649382_dp,21867.3686070922_dp,78888.4332660082_dp,412773.9966006952_dp,&
          0.5090006693_dp,0.6529382408_dp,0.9968266019_dp,1.6750260368_dp,2.9628544290_dp,5.4179293831_dp,10.1908109785_dp,19.7270045640_dp,39.4198017403_dp,&
          81.6958641187_dp,176.7116945056_dp,402.3567263825_dp,975.8376809767_dp,2564.8088840884_dp,7504.7057927441_dp,25609.1162370374_dp,112028.8631073939_dp,821826.1873703946_dp]
       aw%aw_erange_matrix(:,38) = [0.2489028522_dp,0.8164811882_dp,1.6206098122_dp,2.9201002719_dp,5.1709361783_dp,9.2287636832_dp,16.7661132249_dp,31.1681828409_dp,59.5349119329_dp,&
          117.3596202973_dp,240.0411572863_dp,512.9989857692_dp,1156.5464150529_dp,2788.5158335943_dp,7342.3468180366_dp,21867.3834916361_dp,78888.4353980769_dp,412773.6537501516_dp,&
          0.5090007613_dp,0.6529383634_dp,0.9968267522_dp,1.6750262318_dp,2.9628547692_dp,5.4179300607_dp,10.1908123018_dp,19.7270076400_dp,39.4198129704_dp,&
          81.6958999381_dp,176.7117546132_dp,402.3568620295_dp,975.8382656500_dp,2564.8112405019_dp,7504.7120444986_dp,25609.1291848948_dp,112028.7748460801_dp,821825.5760266429_dp]            
    case(20)
       aw%energy_range(:) = [100.0000_dp,200.0000_dp,300.0000_dp,400.0000_dp,500.0000_dp,600.0000_dp,700.0000_dp,800.0000_dp,900.0000_dp,&
          1000.0000_dp,2000.0000_dp,3000.0000_dp,4000.0000_dp,5000.0000_dp,6000.0000_dp,7000.0000_dp,8000.0000_dp,9000.0000_dp,&
          10000.0000_dp,20000.0000_dp,30000.0000_dp,40000.0000_dp,50000.0000_dp,60000.0000_dp,70000.0000_dp,80000.0000_dp,100000.0000_dp,&
          200000.0000_dp,300000.0000_dp,400000.0000_dp,500000.0000_dp,600000.0000_dp,700000.0000_dp,900000.0000_dp,1000000.0000_dp,2000000.0000_dp,&
          3000000.0000_dp,4000000.0000_dp,8000000.0000_dp,10000000.0000_dp]
       aw%aw_erange_matrix(:,1) = [0.1269005661_dp,0.3893436383_dp,0.6784223260_dp,1.0143054963_dp,1.4209575480_dp,1.9280640442_dp,2.5734629819_dp,3.4063474143_dp,4.4916150233_dp,&
          5.9159470021_dp,7.7966091574_dp,10.2948349674_dp,13.6376091270_dp,18.1564082014_dp,24.3637900705_dp,33.1239654661_dp,46.0884630118_dp,67.0263507212_dp,&
          107.1915270177_dp,222.9766291753_dp,0.2552297265_dp,0.2726311907_dp,0.3088610134_dp,0.3668677723_dp,0.4513440201_dp,0.5691227872_dp,0.7298027520_dp,&
          0.9466832844_dp,1.2381561701_dp,1.6298372023_dp,2.1580142079_dp,2.8756651728_dp,3.8639634663_dp,5.2565377559_dp,7.2960714266_dp,10.4818575419_dp,&
          16.0127746431_dp,27.4249166322_dp,59.1658806072_dp,228.8684844393_dp]
       aw%aw_erange_matrix(:,2) = [0.1487641846_dp,0.4603640033_dp,0.8158259805_dp,1.2500369197_dp,1.8069581625_dp,2.5446723364_dp,3.5422243445_dp,4.9092359416_dp,6.7996884178_dp,&
          9.4319683501_dp,13.1184778521_dp,18.3103368732_dp,25.6672898525_dp,36.1736796938_dp,51.3499975303_dp,73.6954851986_dp,107.7903192646_dp,163.6788697578_dp,&
          270.7085752167_dp,575.2960452142_dp,0.2998464753_dp,0.3282616729_dp,0.3884117226_dp,0.4872546392_dp,0.6361565358_dp,0.8522889325_dp,1.1608604696_dp,&
          1.5984996809_dp,2.2183218944_dp,3.0975820152_dp,4.3495025604_dp,6.1423404725_dp,8.7323602516_dp,12.5272476594_dp,18.2263003194_dp,27.1833348194_dp,&
          42.5198181117_dp,73.3349718913_dp,156.9707549667_dp,598.8882107641_dp]
       aw%aw_erange_matrix(:,3) = [0.1553033321_dp,0.4819730674_dp,0.8589054503_dp,1.3267340899_dp,1.9376352225_dp,2.7619553370_dp,3.8974115944_dp,5.4823168482_dp,7.7149415235_dp,&
          10.8822280753_dp,15.4029303766_dp,21.8935434075_dp,31.2718335439_dp,44.9272234982_dp,65.0249340347_dp,95.1241869292_dp,141.6840250994_dp,218.6556268996_dp,&
          366.3368632313_dp,785.0727107663_dp,0.3132497414_dp,0.3457182041_dp,0.4148355207_dp,0.5294098044_dp,0.7039668881_dp,0.9607439790_dp,1.3329053603_dp,&
          1.8694832105_dp,2.6428897343_dp,3.7604095010_dp,5.3820811386_dp,7.7494027866_dp,11.2340289890_dp,16.4283684885_dp,24.3388578990_dp,36.8755634811_dp,&
          58.3510265348_dp,101.1969993687_dp,216.3941760866_dp,821.7863004547_dp]
       aw%aw_erange_matrix(:,4) = [0.1602004477_dp,0.4982768628_dp,0.8918282994_dp,1.3862726480_dp,2.0407769840_dp,2.9363514353_dp,4.1872918884_dp,5.9579161540_dp,8.4874770477_dp,&
          12.1277050228_dp,17.4000555539_dp,25.0843007374_dp,36.3587398714_dp,53.0308860036_dp,77.9444988857_dp,115.7893855068_dp,175.0326856823_dp,273.7531584568_dp,&
          463.6315343486_dp,1000.6412716698_dp,0.3233065868_dp,0.3590615676_dp,0.4355111553_dp,0.5631070573_dp,0.7592170706_dp,1.0507088367_dp,1.4781533654_dp,&
          2.1023610716_dp,3.0144546403_dp,4.3514790279_dp,6.3209308272_dp,9.2402745759_dp,13.6035123636_dp,20.2025549258_dp,30.3787724867_dp,46.6452428752_dp,&
          74.5772897925_dp,130.0596055803_dp,278.0987366596_dp,1052.3224746964_dp]
       aw%aw_erange_matrix(:,5) = [0.1638688488_dp,0.5105604532_dp,0.9168785055_dp,1.4321163876_dp,2.1211989903_dp,3.0740630119_dp,4.4191033387_dp,6.3431018079_dp,9.1212764540_dp,&
          13.1631392903_dp,19.0832782264_dp,27.8120992982_dp,40.7725452278_dp,60.1717228114_dp,89.5126977791_dp,134.5978846704_dp,205.8797927146_dp,325.4902463212_dp,&
          556.1414701978_dp,1207.3263758191_dp,0.3308512935_dp,0.3692149624_dp,0.4515227451_dp,0.5896214449_dp,0.8033146592_dp,1.1234887339_dp,1.5972344368_dp,&
          2.2958886770_dp,3.3275669505_dp,4.8568023156_dp,7.1357039513_dp,10.5544529160_dp,15.7262899198_dp,23.6407452337_dp,35.9746460773_dp,55.8450726898_dp,&
          90.0739692140_dp,157.8883317051_dp,337.7570557426_dp,1274.5553902054_dp]
       aw%aw_erange_matrix(:,6) = [0.1666812473_dp,0.5200197880_dp,0.9363159150_dp,1.4680135301_dp,2.1847779446_dp,3.1839864675_dp,4.6059283508_dp,6.6565650054_dp,9.6421957068_dp,&
          14.0228927018_dp,20.4957937657_dp,30.1266222062_dp,44.5611562236_dp,66.3756371885_dp,99.6904809947_dp,151.3616213170_dp,233.7317810725_dp,372.7770246586_dp,&
          641.5652904001_dp,1399.5025565565_dp,0.3366421328_dp,0.3770934988_dp,0.4641134201_dp,0.6107224468_dp,0.8387891450_dp,1.1826400637_dp,1.6950085513_dp,&
          2.4564493367_dp,3.5901441623_dp,5.2853183249_dp,7.8347083231_dp,11.6956640563_dp,17.5931650358_dp,26.7044655011_dp,41.0283931704_dp,64.2634427909_dp,&
          104.4206964123_dp,183.8648608116_dp,393.5981950763_dp,1482.1123469086_dp]
       aw%aw_erange_matrix(:,7) = [0.1689983670_dp,0.5278413162_dp,0.9524859226_dp,1.4980942342_dp,2.2384628937_dp,3.2775178203_dp,4.7661176411_dp,6.9274306303_dp,10.0959126890_dp,&
          14.7778992447_dp,21.7468510348_dp,32.1949647809_dp,47.9786541782_dp,72.0271500965_dp,109.0579600700_dp,166.9560061338_dp,259.9205313825_dp,417.6959062761_dp,&
          723.4174819692_dp,1584.7298779218_dp,0.3414175972_dp,0.3836475849_dp,0.4746985841_dp,0.6286311009_dp,0.8691546306_dp,1.2336868862_dp,1.7800761528_dp,&
          2.5973123241_dp,3.8225027660_dp,5.6679375388_dp,8.4647255049_dp,12.7343955668_dp,19.3099663809_dp,29.5523018192_dp,45.7781719211_dp,72.2624507871_dp,&
          118.1883304765_dp,208.9732397638_dp,447.7166658434_dp,1682.9251682933_dp]
       aw%aw_erange_matrix(:,8) = [0.1709975801_dp,0.5346104341_dp,0.9665527031_dp,1.5244238634_dp,2.2857569714_dp,3.3604499415_dp,4.9090787048_dp,7.1707601199_dp,10.5062642169_dp,&
          15.4655323863_dp,22.8946056552_dp,34.1070735500_dp,51.1634950623_dp,77.3386634166_dp,117.9404792765_dp,181.8804039669_dp,285.2201402775_dp,461.4812139354_dp,&
          803.8200405044_dp,1767.6358274642_dp,0.3455411323_dp,0.3893490233_dp,0.4839888077_dp,0.6444742812_dp,0.8962114315_dp,1.2794858141_dp,1.8569268957_dp,&
          2.7254731305_dp,4.0354666725_dp,6.0213163325_dp,9.0512798536_dp,13.7096395972_dp,20.9361451675_dp,32.2748999074_dp,50.3627737688_dp,80.0575861694_dp,&
          131.7236728332_dp,233.8208784489_dp,501.4126707244_dp,1881.8947304241_dp]
       aw%aw_erange_matrix(:,9) = [0.1728363280_dp,0.5408533960_dp,0.9795862984_dp,1.5489540702_dp,2.3300726590_dp,3.4386091506_dp,5.0445951101_dp,7.4027796571_dp,10.8999168753_dp,&
          16.1293383882_dp,24.0098862171_dp,35.9779368508_dp,54.3023627872_dp,82.6138138091_dp,126.8337910507_dp,196.9496532881_dp,310.9861906317_dp,506.4458754055_dp,&
          886.9824447716_dp,1957.7555762036_dp,0.3493363943_dp,0.3946314940_dp,0.4926645366_dp,0.6593741291_dp,0.9218200047_dp,1.3231002939_dp,1.9305653016_dp,&
          2.8490587713_dp,4.2421870714_dp,6.3667075825_dp,9.6287355654_dp,14.6770820798_dp,22.5622795853_dp,35.0204218560_dp,55.0265945555_dp,88.0576357175_dp,&
          145.7292992131_dp,259.6935178652_dp,557.4706539806_dp,2089.3700179412_dp]
       aw%aw_erange_matrix(:,10) = [0.1747236608_dp,0.5472787064_dp,0.9930617083_dp,1.5744526151_dp,2.3763969335_dp,3.5207727637_dp,5.1878638223_dp,7.6494891918_dp,11.3209836695_dp,&
          16.8437657056_dp,25.2179952404_dp,38.0183385206_dp,57.7503322676_dp,88.4526053005_dp,136.7566607627_dp,213.9055578161_dp,340.2293509804_dp,557.9081484565_dp,&
          982.8584902244_dp,2178.0487007882_dp,0.3532346686_dp,0.4000927534_dp,0.5017031300_dp,0.6750038465_dp,0.9488503163_dp,1.3694128804_dp,2.0092328227_dp,&
          2.9819085667_dp,4.4658471680_dp,6.7429430083_dp,10.2622527490_dp,15.7464359524_dp,24.3739941067_dp,38.1048989064_dp,60.3120644386_dp,97.2046014714_dp,&
          161.8768332801_dp,289.7172339532_dp,622.7088267238_dp,2330.5550510347_dp]
       aw%aw_erange_matrix(:,11) = [0.1838745625_dp,0.5786892739_dp,1.0598468503_dp,1.7028865109_dp,2.6137008180_dp,3.9489011076_dp,5.9473718311_dp,8.9806897085_dp,13.6350803488_dp,&
          20.8464416741_dp,32.1256332940_dp,49.9393672334_dp,78.3638910186_dp,124.2286594356_dp,199.1775712702_dp,323.5964676730_dp,535.0206455703_dp,910.8130205444_dp,&
          1657.5903483082_dp,3757.0281223492_dp,0.3721759923_dp,0.4271526160_dp,0.5475253024_dp,0.7558746146_dp,1.0913474028_dp,1.6180556646_dp,2.4394980262_dp,&
          3.7226755839_dp,5.7384807986_dp,8.9299655994_dp,14.0291217392_dp,22.2595825062_dp,35.6953004506_dp,57.9143208283_dp,95.2571205121_dp,159.5310630972_dp,&
          275.1830699133_dp,505.5360584493_dp,1097.2825503989_dp,4079.8052751862_dp]
       aw%aw_erange_matrix(:,12) = [0.1893442233_dp,0.5976752765_dp,1.1009673746_dp,1.7836840623_dp,2.7663449388_dp,4.2305124459_dp,6.4583627568_dp,9.8972867136_dp,15.2672324439_dp,&
          23.7416603604_dp,37.2568521772_dp,59.0482829106_dp,94.5956167624_dp,153.3195984480_dp,251.7106390248_dp,419.3652380359_dp,711.7922522672_dp,1243.8026770051_dp,&
          2317.1948214930_dp,5340.4387162258_dp,0.3835304794_dp,0.4438057686_dp,0.5765838506_dp,0.8085345155_dp,1.1864117381_dp,1.7879159406_dp,2.7406303285_dp,&
          4.2543159880_dp,6.6762179522_dp,10.5868161488_dp,16.9678208963_dp,27.5015100275_dp,45.1142123920_dp,74.9889317930_dp,126.5354006363_dp,217.5697005416_dp,&
          384.9447949438_dp,721.8571663357_dp,1582.0813807270_dp,5862.7984242797_dp]
       aw%aw_erange_matrix(:,13) = [0.1929640538_dp,0.6103308334_dp,1.1287011703_dp,1.8389229760_dp,2.8721747439_dp,4.4285304006_dp,6.8228632844_dp,10.5608772296_dp,16.4673389679_dp,&
          25.9056654089_dp,41.1595610556_dp,66.1066026800_dp,107.4272765066_dp,176.8178959446_dp,295.1453267194_dp,500.5708301967_dp,865.7939502029_dp,1542.1363147961_dp,&
          2923.8663861836_dp,6825.3663623913_dp,0.3910589308_dp,0.4550328658_dp,0.5965434966_dp,0.8453048888_dp,1.2538080536_dp,1.9101646448_dp,2.9607357991_dp,&
          4.6492516408_dp,7.3848369300_dp,11.8617470151_dp,19.2731872933_dp,31.6994620652_dp,52.8262769166_dp,89.3073795371_dp,153.4500850616_dp,268.9102192056_dp,&
          484.8471515734_dp,923.9508809596_dp,2042.4381629022_dp,7555.7714291597_dp]
       aw%aw_erange_matrix(:,14) = [0.1953268529_dp,0.6186315144_dp,1.1470347510_dp,1.8757697123_dp,2.9434263672_dp,4.5631048003_dp,7.0729626672_dp,11.0207404044_dp,17.3077365919_dp,&
          27.4379440803_dp,43.9558913837_dp,71.2286774216_dp,116.8675691848_dp,194.3646884356_dp,328.1086307600_dp,563.2980208060_dp,987.0560823558_dp,1781.8374413443_dp,&
          3420.8334598571_dp,8059.7253799248_dp,0.3959791965_dp,0.4624522078_dp,0.6098967742_dp,0.8701722334_dp,1.2998481007_dp,1.9945168283_dp,3.1141917800_dp,&
          4.9276143462_dp,7.8900918050_dp,12.7820305921_dp,20.9592839967_dp,34.8134090969_dp,58.6347666239_dp,100.2713734263_dp,174.4335470338_dp,309.7252860155_dp,&
          565.9174595153_dp,1091.1791332967_dp,2428.3939374675_dp,8976.2684789511_dp]
       aw%aw_erange_matrix(:,15) = [0.1979220756_dp,0.6277857441_dp,1.1673867666_dp,1.9169822219_dp,3.0237404335_dp,4.7159901539_dp,7.3593849231_dp,11.5518207748_dp,18.2869048651_dp,&
          29.2401475740_dp,47.2782513535_dp,77.3810455573_dp,128.3416896459_dp,215.9685447344_dp,369.2709270409_dp,642.8540289853_dp,1143.4943281987_dp,2096.7545149013_dp,&
          4085.4764949230_dp,9733.4147125769_dp,0.4013891591_dp,0.4706859123_dp,0.6248675868_dp,0.8983032593_dp,1.3523696718_dp,2.0915568230_dp,3.2922785955_dp,&
          5.2536478204_dp,8.4877039551_dp,13.8820055023_dp,22.9974199128_dp,38.6235354068_dp,65.8361819586_dp,114.0616190251_dp,201.2456490940_dp,362.7865858475_dp,&
          673.2812412339_dp,1316.6864159845_dp,2955.5779328169_dp,10919.2910161878_dp]
       aw%aw_erange_matrix(:,16) = [0.1993791819_dp,0.6329426794_dp,1.1789139470_dp,1.9404690900_dp,3.0698025097_dp,4.8042386799_dp,7.5258072374_dp,11.8625304683_dp,18.8639570543_dp,&
          30.3105291734_dp,49.2680571847_dp,81.0992054850_dp,135.3443840989_dp,229.2954621089_dp,394.9635543219_dp,693.1603194264_dp,1243.8428121004_dp,2301.9027515682_dp,&
          4525.1220971822_dp,10853.8603929865_dp,0.4044292682_dp,0.4753481665_dp,0.6334154455_dp,0.9144830930_dp,1.3827858098_dp,2.1481427993_dp,3.3968720385_dp,&
          5.4465903028_dp,8.8442306363_dp,14.5439310045_dp,24.2353524597_dp,40.9611235878_dp,70.3029252891_dp,122.7179376050_dp,218.2984984777_dp,397.0258012296_dp,&
          743.6524001242_dp,1466.8140504486_dp,3310.5991720942_dp,12229.9813326092_dp]
       aw%aw_erange_matrix(:,17) = [0.2007287666_dp,0.6377302769_dp,1.1896559856_dp,1.9624505391_dp,3.1131031720_dp,4.8875693338_dp,7.6836814310_dp,12.1587068457_dp,19.4168411849_dp,&
          31.3417174027_dp,51.1963607081_dp,84.7255865155_dp,142.2220551115_dp,242.4849819026_dp,420.6068715213_dp,743.8425862321_dp,1345.9981578952_dp,2513.1236504618_dp,&
          4982.9639601912_dp,12031.2882728258_dp,0.4072467615_dp,0.4796920152_dp,0.6414256270_dp,0.9297223696_dp,1.4115709668_dp,2.2019529685_dp,3.4968358422_dp,&
          5.6319773197_dp,9.1887497064_dp,15.1874794416_dp,25.4468668705_dp,43.2652212867_dp,74.7401103499_dp,131.3908637934_dp,235.5463785324_dp,432.0212860913_dp,&
          816.4059531201_dp,1623.8371874752_dp,3685.2386391427_dp,13615.2774199275_dp]
       aw%aw_erange_matrix(:,18) = [0.2020840295_dp,0.6425489310_dp,1.2005071815_dp,1.9847477410_dp,3.1572137609_dp,4.9728273747_dp,7.8459300539_dp,12.4645182026_dp,19.9905584018_dp,&
          32.4174866719_dp,53.2196564781_dp,88.5545015972_dp,149.5336572481_dp,256.6123898725_dp,448.3028300934_dp,799.0904867364_dp,1458.5112786281_dp,2748.4098173074_dp,&
          5498.8688318388_dp,13370.3635889921_dp,0.4100777806_dp,0.4840791375_dp,0.6495606546_dp,0.9452750654_dp,1.4410840136_dp,2.2573816221_dp,3.6003093288_dp,&
          5.8248681410_dp,9.5492020231_dp,15.8648082978_dp,26.7302003384_dp,45.7230169360_dp,79.5095689997_dp,140.7919497274_dp,254.4173532130_dp,470.7090887768_dp,&
          897.7590223867_dp,1801.4936356453_dp,4113.0305585569_dp,15200.0410499012_dp]
       aw%aw_erange_matrix(:,19) = [0.2032205695_dp,0.6465984181_dp,1.2096570722_dp,2.0036212512_dp,3.1946985238_dp,5.0455691472_dp,7.9849323186_dp,12.7276523813_dp,20.4864941198_dp,&
          33.3520284721_dp,54.9867771812_dp,91.9181816995_dp,155.9979832028_dp,269.1908123424_dp,473.1550907773_dp,849.0992659852_dp,1561.3529337435_dp,2965.7991208260_dp,&
          5980.8252030269_dp,14632.6142302186_dp,0.4124532089_dp,0.4877777262_dp,0.6564540765_dp,0.9585134479_dp,1.4663123889_dp,2.3049676386_dp,3.6895439071_dp,&
          5.9920148700_dp,9.8631540337_dp,16.4580266269_dp,27.8609119922_dp,47.9026672970_dp,83.7695736201_dp,149.2552286193_dp,271.5551657080_dp,506.1887949137_dp,&
          973.1779944256_dp,1968.0569962117_dp,4517.7640074971_dp,16702.4332984174_dp]
       aw%aw_erange_matrix(:,20) = [0.2100467885_dp,0.6710867958_dp,1.2655966277_dp,2.1204488023_dp,3.4297162999_dp,5.5076322035_dp,8.8799491815_dp,14.4464326282_dp,23.7761722206_dp,&
          39.6552702974_dp,67.1241456556_dp,115.4870441158_dp,202.3027424707_dp,361.5321008587_dp,660.7054799726_dp,1238.4882754172_dp,2391.2466429102_dp,4792.5946650321_dp,&
          10209.5610089504_dp,26121.5962145944_dp,0.4267457680_dp,0.5103745955_dp,0.6992682155_dp,1.0419398350_dp,1.6275173138_dp,2.6133681708_dp,4.2765680933_dp,&
          7.1093051388_dp,11.9982862164_dp,20.5686675844_dp,35.8579383327_dp,63.6688292548_dp,115.3603001063_dp,213.7851141281_dp,406.3852699746_dp,795.4467675420_dp,&
          1613.4496474889_dp,3445.5938150773_dp,8247.6252410914_dp,30701.4510546252_dp]
       aw%aw_erange_matrix(:,21) = [0.2134211415_dp,0.6832999754_dp,1.2938913455_dp,2.1804863813_dp,3.5524727238_dp,5.7530169518_dp,9.3635167855_dp,15.3921413624_dp,25.6219059850_dp,&
          43.2672961425_dp,74.2412586858_dp,129.6605522363_dp,230.9357343846_dp,420.4305728470_dp,784.5694186442_dp,1506.0086109926_dp,2987.7753404713_dp,6175.4472528435_dp,&
          13596.3071435874_dp,35790.3633594939_dp,0.4338273088_dp,0.5217934532_dp,0.7213589758_dp,1.0857823424_dp,1.7137308375_dp,2.7812811282_dp,4.6022721177_dp,&
          7.7418600433_dp,13.2336587334_dp,23.0037453201_dp,40.7184816207_dp,73.5253160601_dp,135.7348748315_dp,256.8755076852_dp,500.0120134270_dp,1005.4531221143_dp,&
          2102.5694231474_dp,4639.9562502971_dp,11424.8081725858_dp,42857.5681328368_dp]
       aw%aw_erange_matrix(:,22) = [0.2154048306_dp,0.6905139694_dp,1.3107300197_dp,2.2165178156_dp,3.6267841342_dp,5.9028827283_dp,9.6615935230_dp,15.9808544186_dp,26.7831616219_dp,&
          45.5662657275_dp,78.8290086443_dp,138.9260518495_dp,249.9487133875_dp,460.2341641642_dp,869.9648330428_dp,1694.7284101600_dp,3420.0061213002_dp,7209.4431806666_dp,&
          16220.7724974093_dp,43535.7255823378_dp,0.4379955245_dp,0.5285852418_dp,0.7346430220_dp,1.1124030451_dp,1.7665688206_dp,2.8851827907_dp,4.8058758827_dp,&
          8.1416390320_dp,14.0237512217_dp,24.5814305810_dp,43.9127669455_dp,80.1057147054_dp,149.5786505165_dp,286.7398293421_dp,566.3838524293_dp,1158.2631897128_dp,&
          2469.5012711559_dp,5568.0985493326_dp,13982.7476296939_dp,52802.4632251246_dp]
       aw%aw_erange_matrix(:,23) = [0.2173211080_dp,0.6975072808_dp,1.3271437824_dp,2.2518575469_dp,3.7001326120_dp,6.0517729437_dp,9.9597591924_dp,16.5740573443_dp,27.9625628780_dp,&
          47.9214414702_dp,83.5739597163_dp,148.6111402314_dp,270.0601611920_dp,502.9071881088_dp,962.9363686967_dp,1903.8962513689_dp,3909.2840000751_dp,8409.8263874927_dp,&
          19358.8810606264_dp,53065.9842160918_dp,0.4420257835_dp,0.5352028042_dp,0.7476902021_dp,1.1387347329_dp,1.8191920194_dp,2.9893972965_dp,5.0116395287_dp,&
          8.5489641930_dp,14.8359261888_dp,26.2190424023_dp,47.2641244408_dp,87.0925493281_dp,164.4757637376_dp,319.3690989999_dp,640.1806662632_dp,1331.6759700693_dp,&
          2896.1226974247_dp,6678.6718268326_dp,17138.0059712719_dp,65264.0692155336_dp]
       aw%aw_erange_matrix(:,24) = [0.2193531513_dp,0.7049495461_dp,1.3447091867_dp,2.2899142584_dp,3.7796302540_dp,6.2142182128_dp,10.2873452565_dp,17.2306802824_dp,29.2786972691_dp,&
          50.5731813132_dp,88.9693208847_dp,159.7456029322_dp,293.4694015758_dp,553.2823373222_dp,1074.4843057459_dp,2159.6665041804_dp,4521.2568082323_dp,9952.9945444859_dp,&
          23528.1129162730_dp,66154.9862522529_dp,0.4463035027_dp,0.5422814483_dp,0.7617594998_dp,1.1673332908_dp,1.8767443882_dp,3.1041993470_dp,5.2400633384_dp,&
          9.0049401523_dp,15.7534327805_dp,28.0876681074_dp,51.1309398709_dp,95.2546322855_dp,182.1240661173_dp,358.6470914471_dp,730.6730596026_dp,1548.9997428759_dp,&
          3444.9107894353_dp,8153.0596082736_dp,21475.7473625549_dp,82745.2901188683_dp]
       aw%aw_erange_matrix(:,25) = [0.2199556850_dp,0.7071616248_dp,1.3499499029_dp,2.3013165037_dp,3.8035517381_dp,6.2633169276_dp,10.3868212036_dp,17.4310747137_dp,29.6825632027_dp,&
          51.3917750880_dp,90.6459552077_dp,163.2314157746_dp,300.8594912800_dp,569.3374889456_dp,1110.4290304790_dp,2243.1545546858_dp,4724.1213696404_dp,10474.2783230759_dp,&
          24969.1774183049_dp,70787.9056406355_dp,0.4475727160_dp,0.5443927270_dp,0.7659785620_dp,1.1759505156_dp,1.8941667611_dp,3.1391208583_dp,5.3099073825_dp,&
          9.1451444214_dp,16.0372798748_dp,28.6696636488_dp,52.3442994499_dp,97.8372524796_dp,187.7612122269_dp,371.3292221168_dp,760.2592827024_dp,1621.1098798282_dp,&
          3630.2728339545_dp,8661.9987430013_dp,23010.5776793044_dp,89027.1877264543_dp]
       aw%aw_erange_matrix(:,26) = [0.2213967225_dp,0.7124620182_dp,1.3625440438_dp,2.3288072141_dp,3.8614201529_dp,6.3825038823_dp,10.6291843551_dp,17.9212449302_dp,30.6746932010_dp,&
          53.4122903462_dp,94.8062798825_dp,171.9324347205_dp,319.4305524136_dp,609.9958576948_dp,1202.2760280137_dp,2458.7628453062_dp,5254.7959061760_dp,11859.8272869300_dp,&
          28876.4300087964_dp,83621.3584888344_dp,0.4506096931_dp,0.5494651356_dp,0.7761574759_dp,1.1968178614_dp,1.9365098729_dp,3.2243155683_dp,5.4809935891_dp,&
          9.4901034635_dp,16.7390513969_dp,30.1162910985_dp,55.3783262185_dp,104.3386324165_dp,202.0605295036_dp,403.7825310523_dp,836.7510481214_dp,1809.8393464491_dp,&
          4122.7132961454_dp,10039.4586895308_dp,27256.9773349684_dp,106669.4482880922_dp]
       aw%aw_erange_matrix(:,27) = [0.2224633678_dp,0.7163943625_dp,1.3719211590_dp,2.3493577781_dp,3.9048579408_dp,6.4723510220_dp,10.8127104194_dp,18.2942246887_dp,31.4336342187_dp,&
          54.9669673773_dp,98.0283202070_dp,178.7205394600_dp,334.0395318624_dp,642.2863900989_dp,1276.0355268848_dp,2634.2100758602_dp,5693.5921229264_dp,13028.5986277815_dp,&
          32256.4716147261_dp,95035.7667695921_dp,0.4528589947_dp,0.5532406648_dp,0.7837727453_dp,1.2125008092_dp,1.9684751543_dp,3.2889301656_dp,5.6114015966_dp,&
          9.7544762384_dp,17.2800949143_dp,31.2389616706_dp,57.7502559085_dp,109.4634285227_dp,213.4387314049_dp,429.8879637881_dp,899.0683013216_dp,1965.9491929210_dp,&
          4537.6897182456_dp,11227.6210172707_dp,31024.3140825160_dp,122643.2181496294_dp]
       aw%aw_erange_matrix(:,28) = [0.2270035179_dp,0.7332198213_dp,1.4123703151_dp,2.4388103627_dp,4.0957062773_dp,6.8709440976_dp,11.6353156671_dp,19.9847521336_dp,34.9159532511_dp,&
          62.1982419296_dp,113.2457066364_dp,211.3399667308_dp,405.6484519624_dp,804.2585035719_dp,1656.2452930299_dp,3568.8525746747_dp,8128.5951388481_dp,19864.2862011517_dp,&
          53453.5351595583_dp,172915.1665532315_dp,0.4624461756_dp,0.5695146586_dp,0.8169775175_dp,1.2815895432_dp,2.1107247940_dp,3.5795452147_dp,6.2047044555_dp,&
          10.9724629989_dp,19.8075121695_dp,36.5649147988_dp,69.1995242361_dp,134.6929562291_dp,270.7382314800_dp,564.8796459398_dp,1231.6238153148_dp,2831.6002200052_dp,&
          6952.2631308159_dp,18588.0043317405_dp,56341.6438905135_dp,237851.7740726325_dp]
       aw%aw_erange_matrix(:,29) = [0.2282452994_dp,0.7378467772_dp,1.4235873662_dp,2.4638483499_dp,4.1496383234_dp,6.9847050121_dp,11.8725768460_dp,20.4779361720_dp,35.9446606584_dp,&
          64.3642984509_dp,117.8753890465_dp,221.4402103430_dp,428.2726175934_dp,856.6427196043_dp,1782.6521146754_dp,3890.1080292655_dp,9000.8493152889_dp,22446.8911166671_dp,&
          62057.6209967001_dp,207544.2604152353_dp,0.4650721038_dp,0.5740240208_dp,0.8262871486_dp,1.3011642069_dp,2.1514462179_dp,3.6636459744_dp,6.3784203874_dp,&
          11.3336694651_dp,20.5676692843_dp,38.1920219064_dp,72.7592346003_dp,142.6946633080_dp,289.3308298026_dp,609.8630677897_dp,1345.9979183502_dp,3140.9672008704_dp,&
          7857.7024751677_dp,21526.9070605909_dp,67335.1753726742_dp,292199.2083754797_dp]
       aw%aw_erange_matrix(:,30) = [0.2302329800_dp,0.7452757671_dp,1.4416828789_dp,2.5044527384_dp,4.2375756404_dp,7.1712444839_dp,12.2639786439_dp,21.2968875142_dp,37.6653148941_dp,&
          68.0168392117_dp,125.7541243801_dp,238.8096013994_dp,467.6539561913_dp,949.1365523813_dp,2009.6995661194_dp,4479.4250976071_dp,10644.4747666966_dp,27491.3402238993_dp,&
          79749.2290866879_dp,284201.2216034857_dp,0.4692787280_dp,0.5812951424_dp,0.8413981073_dp,1.3331256476_dp,2.2183275752_dp,3.8026339752_dp,6.6674495133_dp,&
          11.9391083775_dp,21.8523111879_dp,40.9671715187_dp,78.8940703298_dp,156.6502558501_dp,322.2102932417_dp,690.7273514479_dp,1555.7163249990_dp,3722.3682098598_dp,&
          9614.2346334070_dp,27480.1654067523_dp,91042.1984510180_dp,418626.7623824707_dp]
       aw%aw_erange_matrix(:,31) = [0.2309997652_dp,0.7481491652_dp,1.4487102104_dp,2.5202919496_dp,4.2720372785_dp,7.2446994523_dp,12.4188978358_dp,21.6228557718_dp,38.3544509116_dp,&
          69.4899002151_dp,128.9566358732_dp,245.9334937859_dp,483.9748566103_dp,987.9423670191_dp,2106.3746675355_dp,4734.9715827774_dp,11374.0526670871_dp,29802.1776879041_dp,&
          88233.9284246859_dp,323606.8205726882_dp,0.4709026293_dp,0.5841176915_dp,0.8472969826_dp,1.3456653710_dp,2.2446991698_dp,3.8577278692_dp,6.7826779609_dp,&
          12.1820097136_dp,22.3713323301_dp,42.0972287354_dp,81.4145349861_dp,162.4426397032_dp,336.0203223899_dp,725.1737456466_dp,1646.5888196690_dp,3979.7077292905_dp,&
          10413.4545368449_dp,30293.3774670340_dp,102890.7613584087_dp,486716.6945973909_dp]
       aw%aw_erange_matrix(:,32) = [0.2317216671_dp,0.7508582400_dp,1.4553502446_dp,2.5352946574_dp,4.3047609754_dp,7.3146332056_dp,12.5668057721_dp,21.9350283572_dp,39.0166700294_dp,&
          70.9108372686_dp,132.0592352592_dp,252.8695660393_dp,499.9578397739_dp,1026.2066992849_dp,2202.4979190057_dp,4991.7130843994_dp,12116.9785521321_dp,32199.1397717070_dp,&
          97281.6652623537_dp,367557.7408342897_dp,0.4724320498_dp,0.5867840700_dp,0.8528864803_dp,1.3575799638_dp,2.2698244257_dp,3.9103698421_dp,6.8931246234_dp,&
          12.4156370380_dp,22.8724611966_dp,43.1930603877_dp,83.8707280295_dp,168.1194331310_dp,349.6448317936_dp,759.4272357179_dp,1737.8300995737_dp,4241.2567183729_dp,&
          11238.8274817755_dp,33264.2891459751_dp,115840.3068652384_dp,565032.8320210141_dp]
       aw%aw_erange_matrix(:,33) = [0.2315723578_dp,0.7502976548_dp,1.4539752009_dp,2.5321852206_dp,4.2979726514_dp,7.3001120222_dp,12.5360618709_dp,21.8700659116_dp,38.8786859171_dp,&
          70.6143296572_dp,131.4107403383_dp,251.4170699709_dp,496.6034330584_dp,1018.1548395679_dp,2182.2059654894_dp,4937.2959217347_dp,11958.6904408133_dp,31684.7806037803_dp,&
          95319.1174596458_dp,357852.8059662599_dp,0.4721156826_dp,0.5862319504_dp,0.8517278701_dp,1.3551078642_dp,2.2646060976_dp,3.8994247207_dp,6.8701337691_dp,&
          12.3669407660_dp,22.7678540238_dp,42.9639318185_dp,83.3561980628_dp,166.9277005859_dp,346.7773613512_dp,752.1960779938_dp,1718.4964408708_dp,4185.5739886276_dp,&
          11062.0231565802_dp,32622.3714329241_dp,113004.7096290576_dp,547527.8009971447_dp]
       aw%aw_erange_matrix(:,34) = [0.2330327465_dp,0.7557879577_dp,1.4674695396_dp,2.5627685619_dp,4.3648927073_dp,7.4436029263_dp,12.8406272949_dp,22.5154022539_dp,40.2536360812_dp,&
          73.5790953005_dp,137.9204825461_dp,266.0638269379_dp,530.6088760669_dp,1100.2979423094_dp,2390.8193260591_dp,5502.1832614508_dp,13622.9172653753_dp,37190.3510177558_dp,&
          116916.8682410683_dp,470023.6490242586_dp,0.4752111280_dp,0.5916491579_dp,0.8631275974_dp,1.3794916206_dp,2.3162037201_dp,4.0079285275_dp,7.0986967091_dp,&
          12.8525666419_dp,23.8146804455_dp,45.2658378904_dp,88.5485224119_dp,179.0166268464_dp,376.0424512008_dp,826.5356805498_dp,1919.0447455394_dp,4769.8029212309_dp,&
          12945.6412313694_dp,39612.4676004553_dp,144982.7424613823_dp,756968.8285960874_dp]
       aw%aw_erange_matrix(:,35) = [0.2336917809_dp,0.7582706845_dp,1.4735909944_dp,2.5766905568_dp,4.3954654329_dp,7.5094042557_dp,12.9808579282_dp,22.8138487792_dp,40.8926315209_dp,&
          74.9645608438_dp,140.9816947594_dp,273.0014821883_dp,546.8530172131_dp,1139.9352317226_dp,2492.7321322551_dp,5782.4620143348_dp,14465.6745115322_dp,40059.1967586687_dp,&
          128683.0742356157_dp,536270.9828905459_dp,0.4766087737_dp,0.5941057386_dp,0.8683196623_dp,1.3906408024_dp,2.3398881961_dp,4.0579408165_dp,7.2045237306_dp,&
          13.0785379764_dp,24.3044984032_dp,46.3496670744_dp,91.0107736800_dp,184.7968202259_dp,390.1716259317_dp,862.8470043196_dp,2018.4159457724_dp,5064.6094943918_dp,&
          13919.4946699187_dp,43354.1266009085_dp,163081.8283592247_dp,887800.6101104312_dp]
       aw%aw_erange_matrix(:,36) = [0.2355385238_dp,0.7652446928_dp,1.4908503515_dp,2.6161051844_dp,4.4823892214_dp,7.6973271787_dp,13.3832773956_dp,23.6748478611_dp,42.7470247408_dp,&
          79.0122012776_dp,149.9936478274_dp,293.6070803411_dp,595.6059838121_dp,1260.3986842078_dp,2807.2890323238_dp,6664.8121324150_dp,17189.9402657865_dp,49693.4223340807_dp,&
          170733.9762369749_dp,805469.2230494008_dp,0.4805277478_dp,0.6010292090_dp,0.8830277392_dp,1.4223703274_dp,2.4076048048_dp,4.2016395524_dp,7.5102407058_dp,&
          13.7352562402_dp,25.7375980707_dp,49.5449228420_dp,98.3333406215_dp,202.1619239973_dp,433.1318992402_dp,974.8691200400_dp,2330.5777436894_dp,6012.6125605664_dp,&
          17152.3502183949_dp,56371.8063813081_dp,231269.1995400906_dp,1471413.4587235544_dp]
       aw%aw_erange_matrix(:,37) = [0.2362071507_dp,0.7677758110_dp,1.4971376489_dp,2.6305221598_dp,4.5143190664_dp,7.7666656655_dp,13.5324758503_dp,23.9957637090_dp,43.4423108596_dp,&
          80.5400224950_dp,153.4214474265_dp,301.5145297428_dp,614.5118221402_dp,1307.7048556028_dp,2932.7523689171_dp,7023.8190577394_dp,18328.4549698330_dp,53879.2052977139_dp,&
          190218.9242868384_dp,949194.0448591732_dp,0.4819475446_dp,0.6035502702_dp,0.8884107039_dp,1.4340363557_dp,2.4326174364_dp,4.2549803904_dp,7.6243378838_dp,&
          13.9818257869_dp,26.2792936842_dp,50.7619154150_dp,101.1467263538_dp,208.9017043246_dp,450.0072447952_dp,1019.5178319373_dp,2457.2729303758_dp,6406.5081871167_dp,&
          18539.3546761464_dp,62228.8072704350_dp,264565.1715539864_dp,1816006.0736520144_dp]
       aw%aw_erange_matrix(:,38) = [0.2364530992_dp,0.7687077649_dp,1.4994560497_dp,2.6358469269_dp,4.5261314955_dp,7.7923613317_dp,13.5878673870_dp,24.1151440977_dp,43.7015244540_dp,&
          81.1110277037_dp,154.7061920184_dp,304.4880688618_dp,621.6490505745_dp,1325.6482610671_dp,2980.6219613941_dp,7161.8389722969_dp,18770.6868607551_dp,55530.0093352317_dp,&
          198106.2027454207_dp,1011151.4363131844_dp,0.4824699375_dp,0.6044797423_dp,0.8903993119_dp,1.4383537855_dp,2.4418905796_dp,4.2747928499_dp,7.6668029553_dp,&
          14.0737977482_dp,26.4818471532_dp,51.2182598778_dp,102.2051146882_dp,211.4467869967_dp,456.4084213622_dp,1036.5472513717_dp,2505.9293668235_dp,6559.1442235077_dp,&
          19083.5104881533_dp,64570.1582319044_dp,278327.6560799017_dp,1971462.0517839019_dp]
       aw%aw_erange_matrix(:,39) = [0.2367095486_dp,0.7696799444_dp,1.5018761682_dp,2.6414095143_dp,4.5384812877_dp,7.8192485816_dp,13.6458809992_dp,24.2403057633_dp,43.9736166609_dp,&
          81.7112244189_dp,156.0587389971_dp,307.6242756745_dp,629.1931118456_dp,1344.6642439415_dp,3031.5179204175_dp,7309.2032431469_dp,19245.6104053433_dp,57318.3302885437_dp,&
          206780.8472123449_dp,1081955.9030865165_dp,0.4830146984_dp,0.6054499197_dp,0.8924769416_dp,1.4428683492_dp,2.4515956228_dp,4.2955476337_dp,7.7113346404_dp,&
          14.1703631996_dp,26.6948129146_dp,51.6988123571_dp,103.3216424614_dp,214.1373371973_dp,463.1924064512_dp,1054.6496457913_dp,2557.8478071162_dp,6722.8307602149_dp,&
          19671.1937760559_dp,67126.1132235567_dp,293647.8995180877_dp,2154156.0464118524_dp]
       aw%aw_erange_matrix(:,40) = [0.2367095349_dp,0.7696798865_dp,1.5018760060_dp,2.6414091090_dp,4.5384803880_dp,7.8192469055_dp,13.6458785809_dp,24.2403033433_dp,43.9736154891_dp,&
          81.7112237157_dp,156.0587358919_dp,307.6242853845_dp,629.1931876082_dp,1344.6644722167_dp,3031.5183208785_dp,7309.2050560073_dp,19245.6245161903_dp,57318.4221315288_dp,&
          206781.1357473792_dp,1081957.0745627352_dp,0.4830146683_dp,0.6054498545_dp,0.8924767853_dp,1.4428679989_dp,2.4515949740_dp,4.2955467799_dp,7.7113341374_dp,&
          14.1703638145_dp,26.6948145327_dp,51.6988107401_dp,103.3216426489_dp,214.1373680273_dp,463.1925209623_dp,1054.6498000269_dp,2557.8481281643_dp,6722.8344505458_dp,&
          19671.2220756444_dp,67126.2577886249_dp,293648.2400787623_dp,2154159.1938336445_dp]
       aw%aw_erange_matrix(:,41) = [0.2367095541_dp,0.7696799621_dp,1.5018762083_dp,2.6414096107_dp,4.5384815233_dp,7.8192491231_dp,13.6458822537_dp,24.2403086901_dp,43.9736233707_dp,&
          81.7112406038_dp,156.0587780238_dp,307.6243783226_dp,629.1933873974_dp,1344.6649288229_dp,3031.5198680274_dp,7309.2110293459_dp,19245.6405004830_dp,57318.4440887523_dp,&
          206781.2190072980_dp,1081956.3896388158_dp,0.4830147094_dp,0.6054499345_dp,0.8924769754_dp,1.4428684363_dp,2.4515958270_dp,4.2955480752_dp,7.7113357241_dp,&
          14.1703656022_dp,26.6948186934_dp,51.6988265456_dp,103.3216780396_dp,214.1374401511_dp,463.1926703671_dp,1054.6502565123_dp,2557.8502733234_dp,6722.8416355782_dp,&
          19671.2347474855_dp,67126.2610550484_dp,293648.2839700870_dp,2154156.9017522712_dp]
            
    case(22)
       aw%energy_range(:) = [300.0000_dp,400.0000_dp,500.0000_dp,600.0000_dp,700.0000_dp,800.0000_dp,900.0000_dp,1000.0000_dp,2000.0000_dp,&
          3000.0000_dp,4000.0000_dp,5000.0000_dp,6000.0000_dp,7000.0000_dp,8000.0000_dp,9000.0000_dp,10000.0000_dp,20000.0000_dp,&
          30000.0000_dp,40000.0000_dp,50000.0000_dp,60000.0000_dp,70000.0000_dp,80000.0000_dp,90000.0000_dp,100000.0000_dp,200000.0000_dp,&
          300000.0000_dp,400000.0000_dp,500000.0000_dp,600000.0000_dp,700000.0000_dp,900000.0000_dp,1000000.0000_dp,2000000.0000_dp,3000000.0000_dp,&
          4000000.0000_dp,8000000.0000_dp,10000000.0000_dp,20000000.0000_dp]
       aw%aw_erange_matrix(:,1) = [0.1441170079_dp,0.4451145641_dp,0.7857939550_dp,1.1973661610_dp,1.7186472507_dp,2.4001934207_dp,3.3098070861_dp,4.5401370249_dp,6.2193688435_dp,&
          8.5264401723_dp,11.7129087711_dp,16.1347214090_dp,22.2990405865_dp,30.9348424021_dp,43.1034312355_dp,60.3826338002_dp,85.2055156575_dp,121.5752734763_dp,&
          176.8552726753_dp,267.2687084529_dp,440.3633877802_dp,933.5558599550_dp,0.2903383428_dp,0.3160954628_dp,0.3704109236_dp,0.4591352090_dp,0.5917629166_dp,&
          0.7825054649_dp,1.0519884735_dp,1.4297869951_dp,1.9581425612_dp,2.6974086133_dp,3.7340920396_dp,5.1929188031_dp,7.2554447161_dp,10.1901370325_dp,&
          14.4047979662_dp,20.5484937206_dp,29.7390957561_dp,44.1549395243_dp,68.8480988193_dp,118.5837954971_dp,253.9501486803_dp,970.2789416090_dp]
       aw%aw_erange_matrix(:,2) = [0.1474105568_dp,0.4559130086_dp,0.8070286742_dp,1.2345389542_dp,1.7808470248_dp,2.5017389105_dp,3.4727994837_dp,4.7983711562_dp,6.6242944473_dp,&
          9.1562340866_dp,12.6862864221_dp,17.6319936874_dp,24.5942916675_dp,34.4452843471_dp,48.4656274751_dp,68.5719674822_dp,97.7297737183_dp,140.8119314461_dp,&
          206.7281558429_dp,314.9556889586_dp,522.2764283993_dp,1111.8020434921_dp,0.2970754887_dp,0.3246975236_dp,0.3831023542_dp,0.4789076184_dp,0.6229013216_dp,&
          0.8313334849_dp,1.1279698914_dp,1.5471809908_dp,2.1385090603_dp,2.9734175546_dp,4.1553361018_dp,5.8348147393_dp,8.2329276737_dp,11.6789033110_dp,&
          16.6748494761_dp,24.0192841580_dp,35.0782737284_dp,52.4847352813_dp,82.2838180992_dp,142.0622184717_dp,303.9925224140_dp,1158.6905294384_dp]
       aw%aw_erange_matrix(:,3) = [0.1509114548_dp,0.4674397882_dp,0.8298629576_dp,1.2748765694_dp,1.8490035523_dp,2.6141134833_dp,3.6549567374_dp,5.0898154801_dp,7.0858322045_dp,&
          9.8813161595_dp,13.8184986598_dp,19.3920793856_dp,27.3220131651_dp,38.6645168307_dp,54.9862100984_dp,78.6510794212_dp,113.3349529261_dp,165.0780268890_dp,&
          244.8627444176_dp,376.4911344565_dp,628.9041822634_dp,1345.1582530811_dp,0.3042445363_dp,0.3339492071_dp,0.3969424873_dp,0.5007484871_dp,0.6576940599_dp,&
          0.8864769553_dp,1.2146730855_dp,1.6825377112_dp,2.3486802916_dp,3.2985363257_dp,4.6570969536_dp,6.6082616541_dp,9.4248453780_dp,13.5167161989_dp,&
          19.5126964524_dp,28.4139492475_dp,41.9238890770_dp,63.2887057983_dp,99.8733667932_dp,172.9695531491_dp,369.9196209268_dp,1406.2736900862_dp]
       aw%aw_erange_matrix(:,4) = [0.1536955769_dp,0.4766431857_dp,0.8482212221_dp,1.3075836427_dp,1.9047706222_dp,2.7069072191_dp,3.8067527379_dp,5.3349075831_dp,7.4775506122_dp,&
          10.5024969217_dp,14.7978220987_dp,20.9296285029_dp,29.7293786467_dp,42.4279775826_dp,60.8668544973_dp,87.8452195912_dp,127.7374606155_dp,187.7396398310_dp,&
          280.8870352463_dp,435.2340572550_dp,731.5679695302_dp,1571.1092032473_dp,0.3099516325_dp,0.3413883875_dp,0.4082149191_dp,0.5187488871_dp,0.6866731163_dp,&
          0.9328608381_dp,1.2883077496_dp,1.7986072367_dp,2.5306860309_dp,3.5829468654_dp,5.1006370721_dp,7.2993849475_dp,10.5018746881_dp,15.1967144192_dp,&
          22.1379816124_dp,32.5291277259_dp,48.4117813933_dp,73.6440274365_dp,116.8901378887_dp,203.0436360961_dp,434.1404306477_dp,1646.8793814655_dp]
       aw%aw_erange_matrix(:,5) = [0.1558744528_dp,0.4838689714_dp,0.8627147674_dp,1.3335806630_dp,1.9494171648_dp,2.7817388039_dp,3.9300549891_dp,5.5354412574_dp,7.8004098099_dp,&
          11.0183226847_dp,15.6173196140_dp,22.2265048555_dp,31.7767107976_dp,45.6561173235_dp,65.9561614006_dp,95.8760736092_dp,140.4382319943_dp,207.9178440912_dp,&
          313.2691722510_dp,488.5011213669_dp,825.3330516836_dp,1778.4598955036_dp,0.3144217358_dp,0.3472620250_dp,0.4172059508_dp,0.5332404256_dp,0.7101979337_dp,&
          0.9708090378_dp,1.3490122248_dp,1.8950335727_dp,2.6830828484_dp,3.8230245213_dp,5.4781864563_dp,7.8928076393_dp,11.4350158276_dp,16.6659401779_dp,&
          24.4562148221_dp,36.1991864700_dp,54.2555265945_dp,83.0592301482_dp,132.4851645401_dp,230.7461798524_dp,493.3677623531_dp,1868.3614859391_dp]
       aw%aw_erange_matrix(:,6) = [0.1577420296_dp,0.4900788355_dp,0.8752275409_dp,1.3561497684_dp,1.9884059014_dp,2.8474770181_dp,4.0390177175_dp,5.7137100310_dp,8.0891555723_dp,&
          11.4824948537_dp,16.3594459705_dp,23.4086773838_dp,33.6557463687_dp,48.6400404477_dp,70.6954704396_dp,103.4126190545_dp,152.4529012532_dp,227.1621219673_dp,&
          344.4017643056_dp,540.0960501577_dp,916.7176409366_dp,1981.3800393377_dp,0.3182558030_dp,0.3523332401_dp,0.4250332577_dp,0.5459520475_dp,0.7309733738_dp,&
          1.0045360104_dp,1.4033026897_dp,1.9818183126_dp,2.8211326611_dp,4.0419585198_dp,5.8248758088_dp,8.4416547480_dp,12.3045371057_dp,18.0456850793_dp,&
          26.6509067698_dp,39.7026540781_dp,59.8808816713_dp,92.1955098840_dp,147.7226935005_dp,257.9376341497_dp,551.5733506510_dp,2085.6903898363_dp]
       aw%aw_erange_matrix(:,7) = [0.1595766639_dp,0.4961941481_dp,0.8876018082_dp,1.3785833858_dp,2.0273709206_dp,2.9135345279_dp,4.1491070977_dp,5.8948089586_dp,8.3841166359_dp,&
          11.9593604585_dp,17.1263515456_dp,24.6377784386_dp,35.6217919538_dp,51.7828553129_dp,75.7217693626_dp,111.4634347167_dp,165.3839526969_dp,248.0336779114_dp,&
          378.4248655535_dp,596.8834139329_dp,1017.8967204650_dp,2206.9416060218_dp,0.3220246206_dp,0.3573485404_dp,0.4328331959_dp,0.5587069224_dp,0.7519487226_dp,&
          1.0387865473_dp,1.4587538002_dp,2.0709759914_dp,2.9638074450_dp,4.2696295778_dp,6.1877188747_dp,9.0199170047_dp,13.2270451038_dp,19.5201486658_dp,&
          29.0140129036_dp,43.5044754869_dp,66.0336664421_dp,102.2649561638_dp,164.6285305303_dp,288.2428905701_dp,616.5303843661_dp,2327.8897713507_dp]
       aw%aw_erange_matrix(:,8) = [0.1607758370_dp,0.5001994629_dp,0.8957348627_dp,1.3933903711_dp,2.0532041805_dp,2.9575267169_dp,4.2227520566_dp,6.0165023287_dp,8.5832297537_dp,&
          12.2827798171_dp,17.6490051764_dp,25.4796436320_dp,36.9754961105_dp,53.9586928177_dp,79.2215722409_dp,117.1028760760_dp,174.4985213677_dp,262.8395830695_dp,&
          402.7145263321_dp,637.6680025120_dp,1090.9284951990_dp,2370.3030519817_dp,0.3244893294_dp,0.3606449683_dp,0.4379919839_dp,0.5671906911_dp,0.7659710933_dp,&
          1.0617934070_dp,1.4961780450_dp,2.1314381034_dp,3.0610397916_dp,4.4255789634_dp,6.4375758500_dp,9.4203116621_dp,13.8694767492_dp,20.5531330096_dp,&
          30.6799434891_dp,46.2020717059_dp,70.4282404222_dp,109.5032207396_dp,176.8498238540_dp,310.2367273619_dp,663.7309286332_dp,2503.6840921749_dp]
       aw%aw_erange_matrix(:,9) = [0.1702949317_dp,0.5322291046_dp,0.9615962494_dp,1.5151290039_dp,2.2690279176_dp,3.3310549033_dp,4.8582988517_dp,7.0841296034_dp,10.3597776853_dp,&
          15.2192192939_dp,22.4814415016_dp,33.4131871038_dp,49.9905945622_dp,75.3247673562_dp,114.3645047607_dp,175.0835271486_dp,270.5694444089_dp,423.0035691868_dp,&
          672.4894120225_dp,1102.2467053651_dp,1941.0601164726_dp,4300.3213909263_dp,0.3440915125_dp,0.3873401394_dp,0.4807065242_dp,0.6388630857_dp,0.8866070222_dp,&
          1.2631916428_dp,1.8295180866_dp,2.6796325717_dp,3.9590078574_dp,5.8937480925_dp,8.8376065366_dp,13.3484020449_dp,20.3136743061_dp,31.1605701810_dp,&
          48.2150157017_dp,75.3433791336_dp,119.1971624550_dp,192.0205174951_dp,319.6439631329_dp,571.9142828487_dp,1229.1457147735_dp,4600.6335286739_dp]
       aw%aw_erange_matrix(:,10) = [0.1751737113_dp,0.5488134918_dp,0.9962897074_dp,1.5805812543_dp,2.3875698497_dp,3.5406582649_dp,5.2226560654_dp,7.7095997631_dp,11.4238934505_dp,&
          17.0188181974_dp,25.5143694717_dp,38.5178939127_dp,58.5863953296_dp,89.8287076831_dp,138.9210470214_dp,216.8546735931_dp,342.0459845127_dp,546.2432659295_dp,&
          887.3346565379_dp,1484.7821847441_dp,2661.6442082050_dp,5969.3805448616_dp,0.3541646554_dp,0.4014009373_dp,0.5038786064_dp,0.6787816639_dp,0.9554085499_dp,&
          1.3806897245_dp,2.0284537527_dp,3.0144712625_dp,4.5208083482_dp,6.8354919016_dp,10.4176848007_dp,16.0058837975_dp,24.7997029446_dp,38.7698494521_dp,&
          61.1977786202_dp,97.6480183184_dp,157.8383419143_dp,259.7304550554_dp,440.7316633955_dp,799.5695689941_dp,1726.6346830857_dp,6437.6178577764_dp]
       aw%aw_erange_matrix(:,11) = [0.1786605245_dp,0.5607391016_dp,1.0214944059_dp,1.6287106244_dp,2.4758421281_dp,3.6987251620_dp,5.5009421394_dp,8.1935410949_dp,12.2582875674_dp,&
          18.4496574426_dp,27.9611671890_dp,42.6996718386_dp,65.7430067681_dp,102.1132676984_dp,160.1028425852_dp,253.5924066567_dp,406.2255798146_dp,659.3610703780_dp,&
          1089.1052916976_dp,1852.3179473042_dp,3368.2046865402_dp,7629.7200789319_dp,0.3613752591_dp,0.4116146059_dp,0.5210026545_dp,0.7087356034_dp,1.0077570304_dp,&
          1.4712983380_dp,2.1839573403_dp,3.2798930404_dp,4.9726725077_dp,7.6045852128_dp,11.7289136623_dp,18.2487103549_dp,28.6538458757_dp,45.4319734700_dp,&
          72.7949838902_dp,118.0021317674_dp,193.9061669689_dp,324.4297576502_dp,559.1145859642_dp,1026.4106597221_dp,2227.1693511420_dp,8282.0067772426_dp]
       aw%aw_erange_matrix(:,12) = [0.1813450973_dp,0.5699633994_dp,1.0411401995_dp,1.6665645720_dp,2.5459211440_dp,3.8253991316_dp,5.7260836687_dp,8.5888766643_dp,12.9467842565_dp,&
          19.6427530017_dp,30.0240408832_dp,46.2666324055_dp,71.9233970381_dp,112.8627012275_dp,178.9001052545_dp,286.6890467935_dp,464.9859737318_dp,764.7335015874_dp,&
          1280.5197283134_dp,2207.4752523151_dp,4062.5186031124_dp,9281.1213264381_dp,0.3669335038_dp,0.4195744584_dp,0.5345185794_dp,0.7326456820_dp,1.0499742563_dp,&
          1.5451051803_dp,2.3119173440_dp,3.5006092308_dp,5.3525783932_dp,8.2587061985_dp,12.8577717274_dp,20.2045879871_dp,32.0610577862_dp,51.4077134976_dp,&
          83.3599685472_dp,136.8550046716_dp,227.9113435888_dp,386.5773340372_dp,674.9690116950_dp,1252.0133781118_dp,2729.4415777034_dp,10130.7415306161_dp]
       aw%aw_erange_matrix(:,13) = [0.1833338883_dp,0.5768212745_dp,1.0558321359_dp,1.6950684509_dp,2.5990673867_dp,3.9221558102_dp,5.8993011088_dp,8.8953008809_dp,13.4845628578_dp,&
          20.5822211246_dp,31.6622793027_dp,49.1250261247_dp,76.9238552397_dp,121.6495322931_dp,194.4351535622_dp,314.3668670048_dp,514.7553317662_dp,855.2146140510_dp,&
          1447.2985333385_dp,2521.5749798449_dp,4685.0915653480_dp,10776.8769976891_dp,0.3710549661_dp,0.4255263433_dp,0.5447226473_dp,0.7508512219_dp,1.0823705115_dp,&
          1.6021755303_dp,2.4116319975_dp,3.6739975702_dp,5.6535533366_dp,8.7815543992_dp,13.7686065252_dp,21.7985240141_dp,34.8673581133_dp,56.3856031945_dp,&
          92.2682670682_dp,152.9604066795_dp,257.3711559175_dp,441.2259101090_dp,778.3984647651_dp,1456.1530645300_dp,3187.5508156520_dp,11816.0652107334_dp]
       aw%aw_erange_matrix(:,14) = [0.1846993927_dp,0.5815420244_dp,1.0659886574_dp,1.7148709205_dp,2.6361792700_dp,3.9900699229_dp,6.0215196900_dp,9.1126696936_dp,13.8681827815_dp,&
          21.2563233179_dp,32.8450833612_dp,51.2024131733_dp,80.5836340585_dp,128.1290319014_dp,205.9837254840_dp,335.1221389073_dp,552.4298912987_dp,924.4085445080_dp,&
          1576.2391837470_dp,2767.1649646297_dp,5177.0285760558_dp,11968.0101921659_dp,0.3738866518_dp,0.4296404103_dp,0.5518247314_dp,0.7635998287_dp,1.1051835638_dp,&
          1.6425850104_dp,2.4826334701_dp,3.7981811199_dp,5.8704424825_dp,9.1607763090_dp,14.4337771750_dp,22.9710593563_dp,36.9478243061_dp,60.1067835636_dp,&
          98.9873310955_dp,165.2254581116_dp,280.0407946667_dp,483.7490844027_dp,859.8059192899_dp,1618.5092634315_dp,3554.2352326421_dp,13164.8584113213_dp]
       aw%aw_erange_matrix(:,15) = [0.1863159286_dp,0.5871435638_dp,1.0780861353_dp,1.7385623737_dp,2.6807832835_dp,4.0720717291_dp,6.1697823293_dp,9.3776312873_dp,14.3381504702_dp,&
          22.0865346935_dp,34.3099852655_dp,53.7906461618_dp,85.1725466571_dp,136.3093079270_dp,220.6715201873_dp,361.7306746046_dp,601.1493487618_dp,1014.7342015612_dp,&
          1746.2754646089_dp,3094.4730042350_dp,5839.2462359963_dp,13583.4717321030_dp,0.3772409197_dp,0.4345401521_dp,0.5603352643_dp,0.7789597942_dp,1.1328071284_dp,&
          1.6917566707_dp,2.5694667575_dp,3.9508550874_dp,6.1385688770_dp,9.6323320726_dp,15.2660503121_dp,24.4478778325_dp,39.5867564818_dp,64.8627666955_dp,&
          107.6453534898_dp,181.1704182944_dp,309.7963420236_dp,540.1439781093_dp,968.9375786961_dp,1838.3380293709_dp,4053.8917273431_dp,15002.9058091478_dp]
       aw%aw_erange_matrix(:,16) = [0.1874491096_dp,0.5910786161_dp,1.0866144797_dp,1.7553323939_dp,2.7124897315_dp,4.1306103807_dp,6.2760804391_dp,9.5684449515_dp,14.6781771453_dp,&
          22.6901511739_dp,35.3806113482_dp,55.6927681520_dp,88.5650444889_dp,142.3954823450_dp,231.6746888189_dp,381.8130740649_dp,638.2186219844_dp,1084.0696401078_dp,&
          1878.0508173881_dp,3350.6845228755_dp,6362.5940066825_dp,14869.3790003703_dp,0.3795935452_dp,0.4379939369_dp,0.5663682251_dp,0.7899025747_dp,1.1525774075_dp,&
          1.7271089569_dp,2.6321873130_dp,4.0616703072_dp,6.3341813840_dp,9.9782253786_dp,15.8800577265_dp,25.5440999260_dp,41.5585006333_dp,68.4414805289_dp,&
          114.2100287782_dp,193.3604391435_dp,332.7498029047_dp,584.0715532503_dp,1054.8129830325_dp,2012.9829311354_dp,4453.3508278722_dp,16472.7282706270_dp]
       aw%aw_erange_matrix(:,17) = [0.1885601309_dp,0.5949435551_dp,1.0950152212_dp,1.7719071153_dp,2.7439358430_dp,4.1888715677_dp,6.3822508125_dp,9.7597284816_dp,15.0203489378_dp,&
          23.3000361001_dp,36.4670111742_dp,57.6317844147_dp,92.0403898602_dp,148.6633790991_dp,243.0714308414_dp,402.7434914100_dp,677.1160380394_dp,1157.3658398981_dp,&
          2018.4815769762_dp,3626.0594871088_dp,6929.7141747184_dp,16271.5410347054_dp,0.3819012251_dp,0.4413957194_dp,0.5723379721_dp,0.8007750327_dp,1.1722948409_dp,&
          1.7624982958_dp,2.6952142291_dp,4.1734732902_dp,6.5323723645_dp,10.3302491537_dp,16.5079275583_dp,26.6707828007_dp,43.5961111463_dp,72.1615147521_dp,&
          121.0774167617_dp,206.2009442956_dp,357.1109028044_dp,631.0761653313_dp,1147.5021742117_dp,2203.0459385795_dp,4890.5029612894_dp,18081.8150431117_dp]
       aw%aw_erange_matrix(:,18) = [0.1949721439_dp,0.6173833608_dp,1.1442706687_dp,1.8701975876_dp,2.9326176744_dp,4.5426256257_dp,7.0347798366_dp,10.9502950579_dp,17.1785341300_dp,&
          27.2014489279_dp,43.5223948903_dp,70.4304089244_dp,115.3855662997_dp,191.5763875326_dp,322.7363825403_dp,552.4214919992_dp,962.3597980213_dp,1710.0488185397_dp,&
          3110.5415198588_dp,5840.0703282200_dp,11642.5774305799_dp,28229.4187113292_dp,0.3952402409_dp,0.4613337535_dp,0.6078754830_dp,0.8663943938_dp,1.2928300232_dp,&
          1.9816150738_dp,3.0906376694_dp,4.8847277598_dp,7.8119316436_dp,12.6390187248_dp,20.6958382995_dp,34.3232956305_dp,57.7096225293_dp,98.4829095756_dp,&
          170.8120264454_dp,301.6028591803_dp,543.2970597500_dp,1001.7216600548_dp,1903.5448151789_dp,3806.3180661427_dp,8669.3134062261_dp,32033.8629627412_dp]
       aw%aw_erange_matrix(:,19) = [0.1989936579_dp,0.6315770411_dp,1.1758570143_dp,1.9342303779_dp,3.0575467110_dp,4.7807182690_dp,7.4813737603_dp,11.7794206688_dp,18.7093013688_dp,&
          30.0230402179_dp,48.7323254479_dp,80.0951812638_dp,133.4456756804_dp,225.6565231393_dp,387.8397394358_dp,678.6406164985_dp,1211.3802849063_dp,2211.4874488749_dp,&
          4144.7367834599_dp,8037.5098416462_dp,16548.4305196300_dp,41164.1260029741_dp,0.4036247252_dp,0.4741118580_dp,0.6311438163_dp,0.9101749351_dp,1.3746722656_dp,&
          2.1330207366_dp,3.3688668648_dp,5.3948238511_dp,8.7483618785_dp,14.3654969511_dp,23.9006459132_dp,40.3265272169_dp,69.0821489178_dp,120.3188199799_dp,&
          213.4039721814_dp,386.2091970505_dp,714.9239674599_dp,1358.3716125206_dp,2666.1544529873_dp,5503.7069646357_dp,12825.2482547198_dp,47498.4176536332_dp]
       aw%aw_erange_matrix(:,20) = [0.2012398251_dp,0.6395460651_dp,1.1937403319_dp,1.9708322316_dp,3.1296625400_dp,4.9195321781_dp,7.7444221494_dp,12.2730231350_dp,19.6309633750_dp,&
          31.7425193808_dp,51.9487229388_dp,86.1461083728_dp,144.9264642124_dp,247.6850987875_dp,430.6983554712_dp,763.4326099256_dp,1382.4880756646_dp,2564.8940904068_dp,&
          4894.8478751235_dp,9683.4857210923_dp,20349.8357700459_dp,51476.5126403194_dp,0.4083141186_dp,0.4813434144_dp,0.6444825088_dp,0.9355576386_dp,1.4226280417_dp,&
          2.2226888379_dp,3.5354855195_dp,5.7039073132_dp,9.3229230420_dp,15.4391028073_dp,25.9225083298_dp,44.1734243794_dp,76.4943715915_dp,134.8171272371_dp,&
          242.2667924135_dp,444.8560814685_dp,836.9407376956_dp,1619.2611464210_dp,3242.1602051841_dp,6830.5350237777_dp,16170.9299807350_dp,60052.6640642910_dp]
       aw%aw_erange_matrix(:,21) = [0.2031013903_dp,0.6461734099_dp,1.2086954079_dp,2.0016344619_dp,3.1907461078_dp,5.0378865217_dp,7.9702265965_dp,12.6997643083_dp,20.4338313337_dp,&
          33.2525770312_dp,54.7982458408_dp,91.5581069943_dp,155.3022643479_dp,267.8221547321_dp,470.3748416022_dp,843.0417524393_dp,1545.6991555430_dp,2908.1006079514_dp,&
          5638.4548607852_dp,11354.0307723722_dp,24307.6261873841_dp,62454.1953729316_dp,0.4122040611_dp,0.4873890311_dp,0.6557280892_dp,0.9571166377_dp,1.4636458120_dp,&
          2.2999290014_dp,3.6800777992_dp,5.9742484168_dp,9.8297094546_dp,16.3946685455_dp,27.7397415678_dp,47.6678799194_dp,83.3060772782_dp,148.3119784438_dp,&
          269.5146984226_dp,501.1037686841_dp,956.0713638982_dp,1879.2062241708_dp,3829.5600216847_dp,8218.7718324805_dp,19754.2291589833_dp,73606.6628531452_dp]
       aw%aw_erange_matrix(:,22) = [0.2044706274_dp,0.6510614115_dp,1.2197740916_dp,2.0245667788_dp,3.2364565258_dp,5.1269163503_dp,8.1409989236_dp,13.0243295830_dp,21.0481499450_dp,&
          34.4155005946_dp,57.0081317402_dp,95.7873860342_dp,163.4785724775_dp,283.8372264599_dp,502.2535952563_dp,907.7398648202_dp,1680.0621771135_dp,3194.8362150066_dp,&
          6270.3626608683_dp,12801.8342624381_dp,27813.0700007564_dp,72367.8368479170_dp,0.4150672770_dp,0.4918664757_dp,0.6641121553_dp,0.9732840170_dp,1.4945761136_dp,&
          2.3585007678_dp,3.7903687893_dp,6.1817486925_dp,10.2213048087_dp,17.1383810531_dp,29.1651807511_dp,50.4323715012_dp,88.7453993863_dp,159.1994923049_dp,&
          291.7511186854_dp,547.5995278200_dp,1055.9921467960_dp,2100.9062518821_dp,4340.2867613719_dp,9452.2928804385_dp,23004.4310131155_dp,85998.7587267308_dp]
       aw%aw_erange_matrix(:,23) = [0.2056339913_dp,0.6552234254_dp,1.2292398492_dp,2.0442370789_dp,3.2758225079_dp,5.2039030315_dp,8.2892965859_dp,13.3074393893_dp,21.5865610271_dp,&
          35.4399702397_dp,58.9657788358_dp,99.5567529494_dp,170.8145030769_dp,298.3125292928_dp,531.3048638729_dp,967.2458165086_dp,1804.9420001729_dp,3464.5536118564_dp,&
          6873.1194736988_dp,14205.5621734136_dp,31274.7335599056_dp,82323.4357501457_dp,0.4175013571_dp,0.4956912645_dp,0.6713113061_dp,0.9872300013_dp,1.5213724870_dp,&
          2.4094684040_dp,3.8867869924_dp,6.3640474403_dp,10.5671697265_dp,17.7990155080_dp,30.4392843147_dp,52.9201923354_dp,93.6770014619_dp,169.1526099203_dp,&
          312.2675177799_dp,590.9469880749_dp,1150.2563016995_dp,2312.9292394903_dp,4836.5436768091_dp,10672.8741971187_dp,26278.5272782612_dp,98576.8398088366_dp]
       aw%aw_erange_matrix(:,24) = [0.2069775040_dp,0.6600403189_dp,1.2402327852_dp,2.0671701032_dp,3.3219021337_dp,5.2943872625_dp,8.4643316662_dp,13.6430845236_dp,22.2279304584_dp,&
          36.6666360357_dp,61.3229363072_dp,104.1231900943_dp,179.7617000289_dp,316.0993748034_dp,567.3008660624_dp,1041.6720817014_dp,1962.8128315737_dp,3809.7592672350_dp,&
          7655.8017849026_dp,16059.6754897271_dp,35937.1573256609_dp,95979.7168344628_dp,0.4203139414_dp,0.5001321869_dp,0.6797133590_dp,1.0035802839_dp,1.5529244825_dp,&
          2.4697458496_dp,4.0013459453_dp,6.5817175578_dp,10.9823423117_dp,18.5966034510_dp,31.9871611128_dp,55.9633339786_dp,99.7550714105_dp,181.5226720347_dp,&
          338.0059839916_dp,645.9068723725_dp,1271.2310300312_dp,2588.8867103792_dp,5493.2080250041_dp,12319.3081132178_dp,30781.7220454709_dp,116032.3361356684_dp]
       aw%aw_erange_matrix(:,25) = [0.2077600768_dp,0.6628512249_dp,1.2466664953_dp,2.0806362775_dp,3.3490518982_dp,5.3478846618_dp,8.5681905110_dp,13.8429992394_dp,22.6114911707_dp,&
          37.4034452225_dp,62.7455533529_dp,106.8936002490_dp,185.2211900665_dp,327.0222518995_dp,589.5644501511_dp,1088.0778452197_dp,2062.1588170631_dp,4029.3210732029_dp,&
          8159.8868646223_dp,17271.7242812273_dp,39038.0060180248_dp,105212.7558571776_dp,0.4219530067_dp,0.5027307873_dp,0.6846512915_dp,1.0132264590_dp,1.5716075243_dp,&
          2.5055718323_dp,4.0697026631_dp,6.7121470095_dp,11.2322439839_dp,19.0790457755_dp,32.9284400592_dp,57.8247358066_dp,103.4968820433_dp,189.1927368246_dp,&
          354.0939232709_dp,680.5742458585_dp,1348.3404179074_dp,2766.9321874856_dp,5923.0003828403_dp,13415.2124592168_dp,33832.0226596393_dp,127958.1253110201_dp]
       aw%aw_erange_matrix(:,26) = [0.2081933907_dp,0.6644092995_dp,1.2502387307_dp,2.0881274955_dp,3.3641848468_dp,5.3777628442_dp,8.6263151792_dp,13.9551256504_dp,22.8271222335_dp,&
          37.8187145077_dp,63.5495604636_dp,108.4640724552_dp,188.3263905304_dp,333.2579881587_dp,602.3274434799_dp,1114.8061969430_dp,2119.6881360598_dp,4157.2632433470_dp,&
          8455.8029227834_dp,17989.5483700177_dp,40893.5201427240_dp,110793.3937650666_dp,0.4228608173_dp,0.5041734701_dp,0.6873996295_dp,1.0186071731_dp,1.5820510036_dp,&
          2.5256408657_dp,4.1080813740_dp,6.7855534851_dp,11.3732572424_dp,19.3520470382_dp,33.4627328234_dp,58.8849027429_dp,105.6360383871_dp,193.5959479043_dp,&
          363.3729426820_dp,700.6757280481_dp,1393.3257585172_dp,2871.5488135792_dp,6177.6869430943_dp,14071.1666852181_dp,35677.2966879028_dp,135212.4893551906_dp]
       aw%aw_erange_matrix(:,27) = [0.2126896792_dp,0.6806463088_dp,1.2877207566_dp,2.1673388521_dp,3.5254761417_dp,5.6988175431_dp,9.2562253644_dp,15.1813044173_dp,25.2082899090_dp,&
          42.4533225674_dp,72.6275529540_dp,126.4250934438_dp,224.3499352542_dp,406.7658702139_dp,755.5306376950_dp,1442.3710717637_dp,2841.9040752803_dp,5809.6717047741_dp,&
          12410.4327513260_dp,27993.7566576718_dp,68112.6855855200_dp,197105.0711734123_dp,0.4322912965_dp,0.5193038647_dp,0.7165165519_dp,1.0761259374_dp,1.6946552790_dp,&
          2.7439544295_dp,4.5295089336_dp,7.5997918750_dp,12.9546020382_dp,22.4502390722_dp,39.6060527282_dp,71.2522576889_dp,130.9956580492_dp,246.7504440960_dp,&
          477.7177064352_dp,954.3214052744_dp,1976.8634044630_dp,4273.8859897270_dp,9730.3487791900_dp,23679.3321160828_dp,64242.9910056185_dp,251314.2383457562_dp]
       aw%aw_erange_matrix(:,28) = [0.2148506623_dp,0.6884960952_dp,1.3060105620_dp,2.2063965034_dp,3.6058618945_dp,5.8605887338_dp,9.5772645824_dp,15.8138619829_dp,26.4528236083_dp,&
          44.9102492173_dp,77.5153928277_dp,136.2629278459_dp,244.4603760083_dp,448.6865704893_dp,845.0327763266_dp,1639.0867126265_dp,3289.7162486028_dp,6873.6362626559_dp,&
          15075.1917476659_dp,35124.4234356608_dp,88918.3748025578_dp,268262.3870982124_dp,0.4368306957_dp,0.5266819660_dp,0.7309096004_dp,1.1049021959_dp,1.7516438527_dp,&
          2.8557587918_dp,4.7480597843_dp,8.0277823096_dp,13.7980148178_dp,24.1290973780_dp,42.9934113931_dp,78.2036161505_dp,145.5571962110_dp,278.0120655497_dp,&
          546.8198535035_dp,1112.4669786357_dp,2354.1730370757_dp,5220.6504649706_dp,12258.1003979030_dp,30975.0629494139_dp,87667.7472601300_dp,351779.6569114250_dp]
       aw%aw_erange_matrix(:,29) = [0.2162960739_dp,0.6937634781_dp,1.3183457165_dp,2.2328880527_dp,3.6607039037_dp,5.9716175520_dp,9.7989909978_dp,16.2536756436_dp,27.3244349773_dp,&
          46.6446372977_dp,80.9960047169_dp,143.3364569524_dp,259.0772035160_dp,479.5297940043_dp,911.8075091408_dp,1788.2313178298_dp,3635.7123210357_dp,7714.5068495501_dp,&
          17240.5593049632_dp,41126.8844351203_dp,107256.8315932670_dp,334468.4142116168_dp,0.4398695068_dp,0.5316560297_dp,0.7406846089_dp,1.1245732962_dp,1.7908465995_dp,&
          2.9331717561_dp,4.9004358219_dp,8.3284198219_dp,14.3952910496_dp,25.3286146616_dp,45.4374254892_dp,83.2738985787_dp,156.3084281805_dp,301.4138423068_dp,&
          599.3712464305_dp,1234.9595130365_dp,2652.8146881089_dp,5989.8447382865_dp,14379.3880386348_dp,37357.2157063584_dp,109268.7175994240_dp,448590.5277955089_dp]
       aw%aw_erange_matrix(:,30) = [0.2175489615_dp,0.6983403824_dp,1.3291049343_dp,2.2560941408_dp,3.7089561760_dp,6.0697478824_dp,9.9958909398_dp,16.6462313546_dp,28.1066854919_dp,&
          48.2106183031_dp,84.1596278834_dp,149.8135414594_dp,272.5731644807_dp,508.2772508988_dp,974.7206949381_dp,1930.5269348597_dp,3970.7603418492_dp,8543.5263891771_dp,&
          19423.6436846440_dp,47355.7019649240_dp,127040.1189612220_dp,409445.8705308242_dp,0.4425052341_dp,0.5359932906_dp,0.7492554445_dp,1.1419058702_dp,1.8255532742_dp,&
          3.0020442357_dp,5.0367139881_dp,8.5988243515_dp,14.9358296058_dp,26.4215582445_dp,47.6810139402_dp,87.9673293454_dp,166.3542765990_dp,323.5145185104_dp,&
          649.6137627437_dp,1353.7622928283_dp,2947.4487643582_dp,6764.6820381226_dp,16572.9303100728_dp,44186.7283424959_dp,133460.3979620122_dp,561771.4053367316_dp]
       aw%aw_erange_matrix(:,31) = [0.2183655321_dp,0.7013290003_dp,1.3361511974_dp,2.2713420130_dp,3.7407681622_dp,6.1346684432_dp,10.1266305250_dp,16.9079042504_dp,28.6303408601_dp,&
          49.2638104409_dp,86.2982926807_dp,154.2174006332_dp,281.8088166601_dp,528.0951327582_dp,1018.4600600214_dp,2030.4371999963_dp,4208.7872511490_dp,9140.9572351626_dp,&
          21025.2887042307_dp,52033.6921893655_dp,142382.9745163791_dp,470099.2643897415_dp,0.4442239173_dp,0.5388331067_dp,0.7548910589_dp,1.1533454759_dp,1.8485433178_dp,&
          3.0478384059_dp,5.1276930632_dp,8.7801354603_dp,15.3000002759_dp,27.1617591539_dp,49.2093322210_dp,91.1852548813_dp,173.2924797777_dp,338.9060844958_dp,&
          684.9422045330_dp,1438.2483388607_dp,3159.8300824946_dp,7332.5436483025_dp,18214.6557127919_dp,49442.5755149292_dp,152802.2864061145_dp,655909.8326738909_dp]
       aw%aw_erange_matrix(:,32) = [0.2186742698_dp,0.7024601361_dp,1.3388223923_dp,2.2771328530_dp,3.7528722252_dp,6.1594172318_dp,10.1765707675_dp,17.0080735857_dp,28.8312664887_dp,&
          49.6689563903_dp,87.1233422844_dp,155.9217005828_dp,285.3957794177_dp,535.8233082895_dp,1035.5965855717_dp,2069.7953197016_dp,4303.1660507614_dp,9379.7304298714_dp,&
          21671.8682357735_dp,53947.3406519953_dp,148776.3411909367_dp,496015.8330232533_dp,0.4448739108_dp,0.5399095191_dp,0.7570321760_dp,1.1577006749_dp,1.8573135211_dp,&
          3.0653442416_dp,5.1625488328_dp,8.8497656811_dp,15.4402227276_dp,27.4475943113_dp,49.8013936782_dp,92.4363071552_dp,176.0007433716_dp,344.9417174036_dp,&
          698.8697765041_dp,1471.7638618206_dp,3244.7139487741_dp,7561.6130733739_dp,18884.7796936697_dp,51622.1097406976_dp,161002.6247092503_dp,696808.1581583555_dp]
       aw%aw_erange_matrix(:,33) = [0.2198782128_dp,0.7068770903_dp,1.3492753836_dp,2.2998478985_dp,3.8004682907_dp,6.2569829743_dp,10.3739770360_dp,17.4051752184_dp,29.6303106739_dp,&
          51.2857373122_dp,90.4284787704_dp,162.7785957362_dp,299.8977953452_dp,567.2430641136_dp,1105.7187063146_dp,2232.0769799967_dp,4695.8805796643_dp,10384.5178072463_dp,&
          24432.0291527249_dp,62274.4829269457_dp,177368.6923233238_dp,616547.3782229358_dp,0.4474095073_dp,0.5441210056_dp,0.7654350773_dp,1.1748395273_dp,1.8919185945_dp,&
          3.1346104489_dp,5.3008774428_dp,9.1269978871_dp,16.0004962841_dp,28.5941391536_dp,52.1866082452_dp,97.5010391597_dp,187.0257125595_dp,369.6683449598_dp,&
          756.3504831659_dp,1611.3010131586_dp,3601.8754227676_dp,8538.2086436403_dp,21790.5978465254_dp,61293.6413919722_dp,198624.4160270221_dp,892097.8845087624_dp]
       aw%aw_erange_matrix(:,34) = [0.2205755009_dp,0.7094397114_dp,1.3553564847_dp,2.3131026874_dp,3.8283285292_dp,6.3142767965_dp,10.4902942063_dp,17.6400128585_dp,30.1047304678_dp,&
          52.2498851800_dp,92.4090505664_dp,166.9098935930_dp,308.6890319929_dp,586.4239656771_dp,1148.8758631538_dp,2332.9165048231_dp,4942.7373086278_dp,11025.1617954126_dp,&
          26224.1209959338_dp,67814.4643245868_dp,197071.8130837686_dp,704030.7938736863_dp,0.4488787179_dp,0.5465704995_dp,0.7703414465_dp,1.1848813813_dp,1.9122625275_dp,&
          3.1754743020_dp,5.3827921486_dp,9.2918390922_dp,16.3351246999_dp,29.2823043794_dp,53.6260747158_dp,100.5763289562_dp,193.7665976955_dp,384.9076036113_dp,&
          792.1068957790_dp,1699.0621792049_dp,3829.5233339028_dp,9171.0744405107_dp,23714.4471176181_dp,67887.1739965544_dp,225396.6303648227_dp,1038874.3973506133_dp]
       aw%aw_erange_matrix(:,35) = [0.2224377059_dp,0.7162996642_dp,1.3716949966_dp,2.3488612926_dp,3.9038067049_dp,6.4701726928_dp,10.8082522130_dp,18.2851453162_dp,31.4151173088_dp,&
          54.9289395078_dp,97.9492814189_dp,178.5534824685_dp,333.6786520013_dp,641.4848041205_dp,1274.1891627742_dp,2629.7223575866_dp,5681.3924050273_dp,12981.9776395896_dp,&
          31846.2925691313_dp,85845.5733578560_dp,264842.4043736593_dp,1033196.5811019712_dp,0.4528048661_dp,0.5531496173_dp,0.7835887038_dp,1.2121210746_dp,1.9676997164_dp,&
          3.2873595156_dp,5.6082247602_dp,9.7480209040_dp,17.2668500230_dp,31.2113983728_dp,57.6918295511_dp,109.3367311298_dp,213.1561628256_dp,429.2351241074_dp,&
          897.4862306011_dp,1961.7878918285_dp,4524.1985462233_dp,11149.5494849911_dp,29923.5093028414_dp,90140.2431887631_dp,322195.0066980188_dp,1626621.7226316575_dp]
       aw%aw_erange_matrix(:,36) = [0.2239628349_dp,0.7219355060_dp,1.3851835473_dp,2.3785430503_dp,3.9668088400_dp,6.6010539366_dp,11.0768290063_dp,18.8336926711_dp,32.5373962751_dp,&
          57.2417989927_dp,102.7750045174_dp,188.7976056543_dp,355.9168822786_dp,691.1313918785_dp,1388.9298160111_dp,2906.5063814445_dp,6385.7594355512_dp,14901.0391217622_dp,&
          37566.6308767189_dp,105158.5712504577_dp,343432.4005668496_dp,1472923.8844171131_dp,0.4560229807_dp,0.5585787528_dp,0.7945963525_dp,1.2348954597_dp,2.0143300102_dp,&
          3.3820668359_dp,5.8003474242_dp,10.1396846683_dp,18.0733281755_dp,32.8962669883_dp,61.2787261993_dp,117.1534472128_dp,230.6822681245_dp,469.9066152974_dp,&
          995.8900705536_dp,2212.3499471505_dp,5204.1049563543_dp,13150.8079295981_dp,36485.4121522435_dp,115173.4190205108_dp,442490.0578465670_dp,2493513.1430867943_dp]
       aw%aw_erange_matrix(:,37) = [0.2243965013_dp,0.7235409420_dp,1.3890367197_dp,2.3870485328_dp,3.9849206258_dp,6.6388051324_dp,11.1545712186_dp,18.9930833620_dp,32.8648647383_dp,&
          57.9198028342_dp,104.1969950480_dp,191.8339523060_dp,362.5524557845_dp,706.0600231939_dp,1423.7460810181_dp,2991.4083594542_dp,6604.7171596209_dp,15507.7421901125_dp,&
          39415.9362265085_dp,111603.1061980504_dp,371012.6224243028_dp,1643128.6270796654_dp,0.4569384765_dp,0.5601292542_dp,0.7977525237_dp,1.2414486178_dp,2.0277943095_dp,&
          3.4095131325_dp,5.8562441841_dp,10.2541265145_dp,18.3100902522_dp,33.3934954384_dp,62.3434883801_dp,119.4892526136_dp,235.9593169549_dp,482.2607809330_dp,&
          1026.0926003988_dp,2290.2227401019_dp,5418.7139363672_dp,13795.1304786747_dp,38655.0410926747_dp,123775.6738481261_dp,486536.0450457267_dp,2852917.4730972610_dp]
       aw%aw_erange_matrix(:,38) = [0.2257280868_dp,0.7284787343_dp,1.4009186212_dp,2.4133523642_dp,4.0410989331_dp,6.7562599561_dp,11.3972381152_dp,19.4923653756_dp,33.8946132905_dp,&
          60.0610038276_dp,108.7094229805_dp,201.5218299780_dp,383.8565871173_dp,754.3383757573_dp,1537.3076264585_dp,3271.2067186961_dp,7335.5742038181_dp,17566.3480868639_dp,&
          45831.7044920070_dp,134698.6726713842_dp,475373.8675466694_dp,2370246.2019322505_dp,0.4597507654_dp,0.5649093236_dp,0.8075184620_dp,1.2617918420_dp,2.0697260288_dp,&
          3.4952761608_dp,6.0315410632_dp,10.6144464483_dp,19.0587908729_dp,34.9734966865_dp,65.7453442538_dp,126.9982293865_dp,253.0444886075_dp,522.5921175813_dp,&
          1125.6665078537_dp,2550.0437280359_dp,6145.5294802653_dp,16020.1320351972_dp,46350.6273161777_dp,155535.3340619753_dp,660935.4431146942_dp,4531788.9980076430_dp]
       aw%aw_erange_matrix(:,39) = [0.2260960944_dp,0.7298455047_dp,1.4042154577_dp,2.4206704955_dp,4.0567721899_dp,6.7891239920_dp,11.4653468388_dp,19.6329684749_dp,34.1856733039_dp,&
          60.6687050686_dp,109.9960138655_dp,204.2984775948_dp,389.9991164296_dp,768.3550821883_dp,1570.5493708793_dp,3353.9236728895_dp,7554.3047013994_dp,18192.2530611664_dp,&
          47824.6037415822_dp,142101.8482716335_dp,510652.7132165922_dp,2648668.5120964348_dp,0.4605283085_dp,0.5662353265_dp,0.8102368088_dp,1.2674717305_dp,2.0814689820_dp,&
          3.5193707950_dp,6.0809598880_dp,10.7164098245_dp,19.2715422705_dp,35.4245562861_dp,66.7215685549_dp,129.1658164223_dp,258.0099439913_dp,534.4067343872_dp,&
          1155.1115416141_dp,2627.7595629551_dp,6366.0648192168_dp,16707.9785338324_dp,48791.6363938491_dp,166005.7221901024_dp,722469.0060948996_dp,5234589.6473626727_dp]
       aw%aw_erange_matrix(:,40) = [0.2261733010_dp,0.7301323991_dp,1.4049080386_dp,2.4222091516_dp,4.0600702464_dp,6.7960450259_dp,11.4797022884_dp,19.6626311552_dp,34.2471415656_dp,&
          60.7971921298_dp,110.2683862617_dp,204.8871446678_dp,391.3035452124_dp,771.3373955617_dp,1577.6379738323_dp,3371.6096725859_dp,7601.2284769245_dp,18327.1102478198_dp,&
          48256.5019054435_dp,143720.2228231070_dp,518483.1882992422_dp,2712897.2449349295_dp,0.4606914567_dp,0.5665138676_dp,0.8108084502_dp,1.2686672317_dp,2.0839426746_dp,&
          3.5244506649_dp,6.0913885366_dp,10.7379499327_dp,19.3165403313_dp,35.5200796859_dp,66.9286031901_dp,129.6262695848_dp,259.0667452007_dp,536.9266561480_dp,&
          1161.4076992782_dp,2644.4284351048_dp,6413.5530026239_dp,16856.8500540382_dp,49323.6295552938_dp,168312.3706533907_dp,736293.7184796637_dp,5401331.4693767307_dp]
       aw%aw_erange_matrix(:,41) = [0.2261732461_dp,0.7301322119_dp,1.4049076526_dp,2.4222084544_dp,4.0600690988_dp,6.7960433225_dp,11.4796999614_dp,19.6626279855_dp,34.2471373492_dp,&
          60.7971896142_dp,110.2683938690_dp,204.8871558975_dp,391.3034734250_dp,771.3370001893_dp,1577.6366809335_dp,3371.6062497252_dp,7601.2208578517_dp,18327.0899173436_dp,&
          48256.4331612689_dp,143719.9828438671_dp,518482.1905220594_dp,2712896.0135127371_dp,0.4606913432_dp,0.5665137093_dp,0.8108082029_dp,1.2686668512_dp,2.0839421606_dp,&
          3.5244500779_dp,6.0913878513_dp,10.7379488866_dp,19.3165397141_dp,35.5200850043_dp,66.9286165836_dp,129.6262502847_dp,259.0665725933_dp,536.9261347651_dp,&
          1161.4063004605_dp,2644.4255245315_dp,6413.5465926378_dp,16856.8275378510_dp,49323.5406084456_dp,168312.0958090605_dp,736292.1370502859_dp,5401335.3836557837_dp]            
    case(24)
       aw%energy_range(:) = [700.0000_dp,800.0000_dp,900.0000_dp,1000.0000_dp,2000.0000_dp,3000.0000_dp,4000.0000_dp,5000.0000_dp,6000.0000_dp,&
          7000.0000_dp,8000.0000_dp,9000.0000_dp,10000.0000_dp,20000.0000_dp,30000.0000_dp,40000.0000_dp,50000.0000_dp,60000.0000_dp,&
          70000.0000_dp,80000.0000_dp,90000.0000_dp,100000.0000_dp,200000.0000_dp,300000.0000_dp,400000.0000_dp,500000.0000_dp,600000.0000_dp,&
          700000.0000_dp,900000.0000_dp,1000000.0000_dp,2000000.0000_dp,3000000.0000_dp,4000000.0000_dp,8000000.0000_dp,20000000.0000_dp,30000000.0000_dp,&
          40000000.0000_dp,50000000.0000_dp,60000000.0000_dp,70000000.0000_dp]
       aw%aw_erange_matrix(:,1) = [0.1444831841_dp,0.4463129305_dp,0.7881428810_dp,1.2014614143_dp,1.7254689683_dp,2.4112775480_dp,3.3275084061_dp,4.5680241415_dp,6.2628063920_dp,&
          8.5934240114_dp,11.8152023682_dp,16.2892192514_dp,22.5288206461_dp,31.2678471963_dp,43.5619559201_dp,60.9420651793_dp,85.6545500191_dp,121.0591608460_dp,&
          172.3522673692_dp,248.0734198626_dp,363.8458976581_dp,553.8515140977_dp,917.7986693324_dp,1952.9117915519_dp,0.2910870254_dp,0.3170469290_dp,0.3718060277_dp,&
          0.4612957722_dp,0.5951464219_dp,0.7877810899_dp,1.0601464492_dp,1.4422957175_dp,1.9771680040_dp,2.7260990272_dp,3.7768785615_dp,5.2556016576_dp,&
          7.3442618251_dp,10.3072523710_dp,14.5322648586_dp,20.5960578770_dp,29.3777063234_dp,42.2751385759_dp,61.6812840242_dp,92.2133252902_dp,144.4837916238_dp,&
          249.3832447505_dp,533.6819506240_dp,2034.6812130298_dp]
       aw%aw_erange_matrix(:,2) = [0.1462351224_dp,0.4520540968_dp,0.7994228466_dp,1.2211864610_dp,1.7584357941_dp,2.4650348251_dp,3.4136941332_dp,4.7044124886_dp,6.4764241313_dp,&
          8.9252888104_dp,12.3275308604_dp,17.0764098214_dp,23.7342251851_dp,33.1094526118_dp,46.3718374940_dp,65.2272650233_dp,92.1916981548_dp,131.0436704122_dp,&
          187.6411062375_dp,271.6082468252_dp,400.4981511560_dp,612.5417457455_dp,1018.8940111719_dp,2173.3280071422_dp,0.2946702604_dp,0.3216162729_dp,0.3785364862_dp,&
          0.4717652242_dp,0.6116114582_dp,0.8135669770_dp,1.1002227739_dp,1.5041403839_dp,2.0720738283_dp,2.8711592052_dp,3.9980154578_dp,5.5922033331_dp,&
          7.8563040850_dp,11.0862799686_dp,15.7184068280_dp,22.4046116197_dp,32.1413789683_dp,46.5137216101_dp,68.2261257303_dp,102.4653370494_dp,161.0806535929_dp,&
          278.4572085100_dp,595.6911596593_dp,2267.9697886947_dp]
       aw%aw_erange_matrix(:,3) = [0.1480254215_dp,0.4579337645_dp,0.8110189757_dp,1.2415600901_dp,1.7926594407_dp,2.5211286991_dp,3.5040867940_dp,4.8481890429_dp,6.7027698187_dp,&
          9.2787577535_dp,12.8761137851_dp,17.9239105861_dp,25.0392947209_dp,35.1149478848_dp,49.4502635769_dp,69.9514249484_dp,99.4452959336_dp,142.1967669711_dp,&
          204.8365962245_dp,298.2605268262_dp,442.2824727194_dp,679.8553440442_dp,1135.4135593105_dp,2428.1922750715_dp,0.2983340021_dp,0.3263141520_dp,0.3855062739_dp,&
          0.4826798657_dp,0.6288796171_dp,0.8407612053_dp,1.1427161320_dp,1.5700685414_dp,2.1738000022_dp,3.0275157594_dp,4.2377477964_dp,5.9592805615_dp,&
          8.4181383582_dp,11.9464976910_dp,17.0367899874_dp,24.4284761387_dp,35.2557057237_dp,51.3239814800_dp,75.7055579826_dp,114.2569409241_dp,180.2699406534_dp,&
          312.1779089127_dp,667.6444661491_dp,2538.2837097538_dp]
       aw%aw_erange_matrix(:,4) = [0.1495207465_dp,0.4628547668_dp,0.8207591322_dp,1.2587484912_dp,1.8216692929_dp,2.5689048829_dp,3.5814431792_dp,4.9718158210_dp,6.8983254114_dp,&
          9.5856273277_dp,13.3547404058_dp,18.6671136196_dp,26.1897918363_dp,36.8925672577_dp,52.1943604128_dp,74.1872982909_dp,105.9888838572_dp,152.3217269704_dp,&
          220.5481261826_dp,322.7717685540_dp,480.9542755105_dp,742.5159256330_dp,1244.3915104090_dp,2667.3021019704_dp,0.3013957086_dp,0.3302604655_dp,0.3914004560_dp,&
          0.4919676878_dp,0.6436556862_dp,0.8641511062_dp,1.1794484220_dp,1.6273447109_dp,2.2626275452_dp,3.1647612174_dp,4.4493120040_dp,6.2850287038_dp,&
          8.9195842042_dp,12.7188298679_dp,18.2277869335_dp,26.2684744543_dp,38.1057440783_dp,55.7555316005_dp,82.6418345370_dp,125.2598873133_dp,198.2669609027_dp,&
          343.9024155387_dp,735.3761760093_dp,2792.4007975812_dp]
       aw%aw_erange_matrix(:,5) = [0.1581584545_dp,0.4914655700_dp,0.8780290019_dp,1.3612185024_dp,1.9971910555_dp,2.8623379317_dp,4.0637284684_dp,5.7542616489_dp,8.1550244560_dp,&
          11.5886381531_dp,16.5294112875_dp,23.6793452859_dp,34.0841888721_dp,49.3120741540_dp,71.7310702306_dp,104.9457479288_dp,154.4966301940_dp,229.0167848121_dp,&
          342.2593271762_dp,517.0527707275_dp,794.5036692063_dp,1261.3893969466_dp,2162.7398061845_dp,4705.9350887607_dp,0.3191110394_dp,0.3534686602_dp,0.4267939039_dp,&
          0.5488233603_dp,0.7356835113_dp,1.0122083638_dp,1.4156919494_dp,2.0016798733_dp,2.8528008777_dp,4.0922437829_dp,5.9044201005_dp,8.5668449726_dp,&
          12.4997077073_dp,18.3436967442_dp,27.0838216744_dp,40.2505374352_dp,60.2583466839_dp,91.0142703018_dp,139.1437232012_dp,216.9327357162_dp,351.1339455834_dp,&
          616.8079434404_dp,1319.8772466218_dp,4975.5101757964_dp]
       aw%aw_erange_matrix(:,6) = [0.1631097620_dp,0.5080135567_dp,0.9116667485_dp,1.4225389648_dp,2.1043236614_dp,3.0450359010_dp,4.3700131540_dp,6.2611211321_dp,8.9856050855_dp,&
          12.9399049409_dp,18.7167879480_dp,27.2090533360_dp,39.7710395669_dp,58.4723061861_dp,86.4999013261_dp,128.8036555283_dp,193.1479506070_dp,291.8704906235_dp,&
          444.9645785607_dp,685.9916443587_dp,1075.4395233632_dp,1739.5672411434_dp,3029.3298737363_dp,6660.6016806705_dp,0.3292892917_dp,0.3671025078_dp,0.4481712623_dp,&
          0.5840409015_dp,0.7939862869_dp,1.1080155816_dp,1.5717812452_dp,2.2542645172_dp,3.2596882303_dp,4.7460262182_dp,6.9539264000_dp,10.2522278622_dp,&
          15.2101805373_dp,22.7131385208_dp,34.1501527103_dp,51.7234046577_dp,78.9730765643_dp,121.7140919219_dp,189.8748316532_dp,301.7465519058_dp,496.3642328717_dp,&
          880.9495966272_dp,1889.0610962342_dp,7090.4217269333_dp]
       aw%aw_erange_matrix(:,7) = [0.1664674025_dp,0.5192992114_dp,0.9348305991_dp,1.4652599863_dp,2.1798809460_dp,3.1754829050_dp,4.5914060961_dp,6.6320592545_dp,9.6011642410_dp,&
          13.9544169906_dp,20.3812229064_dp,29.9326640661_dp,44.2235898076_dp,65.7546627134_dp,98.4306545184_dp,148.4042522886_dp,225.4705402984_dp,345.4244262409_dp,&
          534.2099602523_dp,835.8233970236_dp,1329.8280854082_dp,2181.2822208070_dp,3843.6799664110_dp,8519.1634651685_dp,0.3362016095_dp,0.3764914681_dp,0.4631460168_dp,&
          0.6090928797_dp,0.8360361451_dp,1.1780256329_dp,1.6873334714_dp,2.4437398939_dp,3.5690964004_dp,5.2502409163_dp,7.7752875394_dp,11.5915422657_dp,&
          17.3987403579_dp,26.3006419804_dp,40.0544557406_dp,61.4877246077_dp,95.2120979201_dp,148.8986862848_dp,235.7507129855_dp,380.0737118713_dp,633.1136329283_dp,&
          1133.3186348854_dp,2436.0577961080_dp,9116.6780863972_dp]
       aw%aw_erange_matrix(:,8) = [0.1690776973_dp,0.5281095095_dp,0.9530417927_dp,1.4991313490_dp,2.2403191273_dp,3.2807596388_dp,4.7716782841_dp,6.9368306537_dp,10.1115966056_dp,&
          14.8036958093_dp,21.7883944857_dp,32.2591622719_dp,48.0682608207_dp,72.1146215914_dp,108.9756534578_dp,165.9485193173_dp,254.7915087282_dp,394.6980676439_dp,&
          617.5622271710_dp,977.9811277731_dp,1575.1218646365_dp,2613.9759091892_dp,4652.4661733029_dp,10382.7978511251_dp,0.3415811579_dp,0.3838728955_dp,0.4750640433_dp,&
          0.6292516218_dp,0.8702095713_dp,1.2354628915_dp,1.7830337174_dp,2.6021851733_dp,3.8304277561_dp,5.6805493356_dp,8.4838802661_dp,12.7601201248_dp,&
          19.3310759058_dp,29.5078437842_dp,45.4025661560_dp,70.4555770009_dp,110.3464794351_dp,174.6287171469_dp,279.8799006894_dp,456.6692570002_dp,768.9396428512_dp,&
          1387.0711545806_dp,2989.0749703627_dp,11161.0556901874_dp]
       aw%aw_erange_matrix(:,9) = [0.1709557703_dp,0.5344686559_dp,0.9662573049_dp,1.5238691353_dp,2.2847567751_dp,3.3586881106_dp,4.9060235024_dp,7.1655139823_dp,10.4972872973_dp,&
          15.4500901467_dp,22.8675204274_dp,34.0574489068_dp,51.0647967947_dp,77.1150960111_dp,117.3434868279_dp,180.0072675174_dp,278.5322573782_dp,435.0365515056_dp,&
          686.6040010667_dp,1097.1993584122_dp,1783.4898352383_dp,2986.2142227836_dp,5356.0836020504_dp,12016.9059606308_dp,0.3454548628_dp,0.3892293024_dp,0.4837928296_dp,&
          0.6441385331_dp,0.8956351410_dp,1.2785039189_dp,1.8552630076_dp,2.7226521906_dp,4.0306368631_dp,6.0128362011_dp,9.0356096383_dp,13.6779344741_dp,&
          20.8626467845_dp,32.0743836708_dp,49.7259301183_dp,77.7832698071_dp,122.8543665835_dp,196.1515671140_dp,317.2657255647_dp,522.4150070505_dp,887.0057238075_dp,&
          1609.9115965938_dp,3477.1227916280_dp,12962.7512514722_dp]
       aw%aw_erange_matrix(:,10) = [0.1726447830_dp,0.5402022810_dp,0.9782242070_dp,1.5463843073_dp,2.3254181485_dp,3.4303775442_dp,5.0302797273_dp,7.3781823188_dp,10.8579865286_dp,&
          16.0581304869_dp,23.8888077362_dp,35.7702460115_dp,53.9381215294_dp,81.9441090303_dp,125.4853484427_dp,193.7959335739_dp,302.0155051485_dp,475.3005938585_dp,&
          756.1877714365_dp,1218.5936533093_dp,1997.9473426011_dp,3373.4421522655_dp,6095.0561348602_dp,13744.7879009066_dp,0.3489409145_dp,0.3940794590_dp,0.4917547726_dp,&
          0.6578067504_dp,0.9191179687_dp,1.3184835161_dp,1.9227402786_dp,2.8358581620_dp,4.2199323354_dp,6.3290212303_dp,9.5641245019_dp,14.5633247217_dp,&
          22.3510726502_dp,34.5881630863_dp,53.9955472147_dp,85.0836236152_dp,135.4324631636_dp,218.0114762829_dp,355.6392245410_dp,590.6401417408_dp,1010.8425872098_dp,&
          1845.7325256463_dp,3995.9445980019_dp,14876.1560237048_dp]
       aw%aw_erange_matrix(:,11) = [0.1738605393_dp,0.5443380620_dp,0.9868867334_dp,1.5627507674_dp,2.3551044797_dp,3.4829469465_dp,5.1217979994_dp,7.5355209805_dp,11.1260743095_dp,&
          16.5122194112_dp,24.6553358520_dp,37.0625585194_dp,56.1180969954_dp,85.6293729569_dp,131.7375179578_dp,204.4544488438_dp,320.2960832789_dp,506.8809564298_dp,&
          811.2057377731_dp,1315.4042726890_dp,2170.5222479635_dp,3687.8741411859_dp,6700.0179154473_dp,15167.5878795238_dp,0.3514515514_dp,0.3975902285_dp,0.4975525841_dp,&
          0.6678130090_dp,0.9363925476_dp,1.3480308458_dp,1.9728444581_dp,2.9203241384_dp,4.3618812340_dp,6.5673687306_dp,9.9647275133_dp,15.2383215088_dp,&
          23.4927406755_dp,36.5287652598_dp,57.3142122789_dp,90.7993513300_dp,145.3568831305_dp,235.4026772409_dp,386.4380005836_dp,645.9038040534_dp,1112.0689367858_dp,&
          2039.9867949721_dp,4425.0672231763_dp,16457.6314110225_dp]
       aw%aw_erange_matrix(:,12) = [0.1753655164_dp,0.5494678874_dp,0.9976670809_dp,1.5831986606_dp,2.3923461136_dp,3.5491675380_dp,5.2375601555_dp,7.7353810978_dp,11.4680982283_dp,&
          17.0941702077_dp,25.6423723038_dp,38.7349608389_dp,58.9541546985_dp,90.4506116309_dp,139.9656122303_dp,218.5705032899_dp,344.6710685128_dp,549.2960926611_dp,&
          885.6771136835_dp,1447.5423259491_dp,2408.1530281330_dp,4124.7049256973_dp,7547.3043273828_dp,17171.9709510711_dp,0.3545610482_dp,0.4019591263_dp,0.5048080352_dp,&
          0.6803975269_dp,0.9582167603_dp,1.3855239964_dp,2.0367043909_dp,3.0284718185_dp,4.5444937849_dp,6.8755236067_dp,10.4853717613_dp,16.1204210430_dp,&
          24.9933884816_dp,39.0953047224_dp,61.7321009802_dp,98.4615076338_dp,158.7605363404_dp,259.0791626459_dp,428.7271349072_dp,722.4697634703_dp,1253.5792039363_dp,&
          2313.6572543644_dp,5032.2139735216_dp,18693.9190060109_dp]
       aw%aw_erange_matrix(:,13) = [0.1762525278_dp,0.5524966749_dp,1.0040508933_dp,1.5953494848_dp,2.4145565226_dp,3.5888042694_dp,5.3071037767_dp,7.8558935977_dp,11.6751259028_dp,&
          17.4478358462_dp,26.2447351358_dp,39.7600958375_dp,60.7007012587_dp,93.4344188867_dp,145.0847228505_dp,227.4021853772_dp,360.0129611396_dp,576.1649809108_dp,&
          933.1802189583_dp,1532.4570953189_dp,2562.0641335204_dp,4409.9027665207_dp,8104.5389998819_dp,18497.1969702348_dp,0.3563945776_dp,0.4045461520_dp,0.5091256006_dp,&
          0.6879192306_dp,0.9713130436_dp,1.4081098242_dp,2.0753236739_dp,3.0941375016_dp,4.6558387310_dp,7.0642416323_dp,10.8056938539_dp,16.6657683060_dp,&
          25.9259213466_dp,40.6989072978_dp,64.5084644211_dp,103.3064950012_dp,167.2921824588_dp,274.2567753884_dp,456.0426876422_dp,772.3234133156_dp,1346.4655933627_dp,&
          2494.5638380344_dp,5435.1707231009_dp,20177.5066401676_dp]
       aw%aw_erange_matrix(:,14) = [0.1829783682_dp,0.5755938137_dp,1.0531970558_dp,1.6899438035_dp,2.5894885088_dp,3.9046729855_dp,5.8679231710_dp,8.8396478506_dp,13.3866247544_dp,&
          20.4106346920_dp,31.3621347581_dp,48.5995152028_dp,76.0007561643_dp,120.0187127092_dp,191.5278286467_dp,309.1059913741_dp,504.9671486648_dp,835.8995406056_dp,&
          1403.9494370002_dp,2397.0991426389_dp,4175.7480862539_dp,7492.4429100080_dp,14302.6666713534_dp,33555.9419248088_dp,0.3703179667_dp,0.4244588968_dp,0.5428864332_dp,&
          0.7475654171_dp,1.0765075808_dp,1.5918195796_dp,2.3934883989_dp,3.6423586825_dp,5.5984672689_dp,8.6855481066_dp,13.6007516181_dp,21.5035284761_dp,&
          34.3450835002_dp,55.4512312859_dp,90.5696137480_dp,149.7836724584_dp,251.0801516726_dp,427.1647777877_dp,738.9719004826_dp,1304.3562792987_dp,2368.9196212347_dp,&
          4543.2054945270_dp,10079.2529978940_dp,37268.9585459828_dp]
       aw%aw_erange_matrix(:,15) = [0.1864328591_dp,0.5875492898_dp,1.0789643041_dp,1.7402865658_dp,2.6840380111_dp,4.0780712696_dp,6.1806590053_dp,9.3971229425_dp,14.3728222885_dp,&
          22.1479648759_dp,34.4187048094_dp,53.9832910445_dp,85.5148902410_dp,136.9198284881_dp,221.7622052391_dp,363.6659505460_dp,604.4572621267_dp,1019.5465755022_dp,&
          1747.7402439870_dp,3051.1515592744_dp,5443.8681058120_dp,10014.0254933408_dp,19571.8504999681_dp,46737.2209709122_dp,0.3774836319_dp,0.4348958056_dp,0.5609552035_dp,&
          0.7800821684_dp,1.1348314252_dp,1.6953702560_dp,2.5758665583_dp,3.9621412099_dp,6.1584508458_dp,9.6674081750_dp,15.3281438612_dp,24.5583220846_dp,&
          39.7842045296_dp,65.2168705851_dp,108.2771491585_dp,182.2605473204_dp,311.4266558086_dp,540.9645390539_dp,957.1793336441_dp,1730.8219657637_dp,3222.4346623368_dp,&
          6320.4828281551_dp,14214.0487649746_dp,52510.9710846642_dp]
       aw%aw_erange_matrix(:,16) = [0.1888673828_dp,0.5960135969_dp,1.0973453046_dp,1.7765141441_dp,2.7526956080_dp,4.2051369061_dp,6.4119579634_dp,9.8133747332_dp,15.1165441060_dp,&
          23.4719283795_dp,36.7740207159_dp,58.1812475623_dp,93.0278994207_dp,150.4486063818_dp,246.3207344899_dp,408.6913848578_dp,687.9884842007_dp,1176.6680482315_dp,&
          2048.0301817657_dp,3635.6622808178_dp,6606.1147808342_dp,12388.8768429516_dp,24670.4465803243_dp,59765.6786084483_dp,0.3825395972_dp,0.4423391997_dp,0.5739985133_dp,&
          0.8038071031_dp,1.1778066684_dp,1.7724143723_dp,2.7129170522_dp,4.2049553639_dp,6.5883275304_dp,10.4299109641_dp,16.6861925793_dp,26.9915765371_dp,&
          44.1776987280_dp,73.2245357881_dp,123.0341810517_dp,209.8052795947_dp,363.5965618859_dp,641.4271543159_dp,1154.3058667782_dp,2125.9959039706_dp,4035.3547351848_dp,&
          8060.2352858889_dp,18344.4026094871_dp,67781.7202153203_dp]
       aw%aw_erange_matrix(:,17) = [0.1908012661_dp,0.6027605095_dp,1.1120797355_dp,1.8057447039_dp,2.8084669625_dp,4.3090570644_dp,6.6024368735_dp,10.1586238065_dp,15.7380385569_dp,&
          24.5870918153_dp,38.7746508920_dp,61.7792631772_dp,99.5296982089_dp,162.2789883408_dp,268.0395021411_dp,449.0011876172_dp,763.7792359281_dp,1321.3436607123_dp,&
          2329.0736920909_dp,4192.7194711633_dp,7736.4036043617_dp,14750.2565581519_dp,29855.8684036316_dp,73259.4938942780_dp,0.3865594411_dp,0.4483047301_dp,0.5845463388_dp,&
          0.8231452604_dp,1.2130930613_dp,1.8361346851_dp,2.8271157811_dp,4.4088710629_dp,6.9523315126_dp,11.0812505492_dp,17.8571269115_dp,29.1105993227_dp,&
          48.0449841567_dp,80.3549054679_dp,136.3396775154_dp,234.9807659012_dp,411.9945806478_dp,736.1685089361_dp,1343.6208414133_dp,2513.2762031319_dp,4849.9948833345_dp,&
          9844.1013924928_dp,22656.5439149310_dp,83782.4350216886_dp]
       aw%aw_erange_matrix(:,18) = [0.1919275086_dp,0.6066993775_dp,1.1207162119_dp,1.8229571440_dp,2.8414643308_dp,4.3708369707_dp,6.7162289720_dp,10.3659192449_dp,16.1131786879_dp,&
          25.2639983229_dp,39.9963115681_dp,63.9904453046_dp,103.5530041949_dp,169.6542328021_dp,281.6889602727_dp,474.5581547092_dp,812.2969292007_dp,1414.9478643552_dp,&
          2513.0666715026_dp,4562.2632949378_dp,8497.4370986425_dp,16366.6024633931_dp,33466.2430685965_dp,82787.2172804636_dp,0.3889019836_dp,0.4518008519_dp,0.5907670865_dp,&
          0.8346137832_dp,1.2341277170_dp,1.8743137280_dp,2.8959011651_dp,4.5323761503_dp,7.1740882935_dp,11.4805267880_dp,18.5796868308_dp,30.4275119206_dp,&
          50.4668006262_dp,84.8571339354_dp,144.8165065913_dp,251.1773091777_dp,443.4670252683_dp,798.5123604322_dp,1469.8554010678_dp,2775.3655096322_dp,5410.4529573927_dp,&
          11092.7716344120_dp,25718.0669014744_dp,95181.4081571741_dp]
       aw%aw_erange_matrix(:,19) = [0.1933564752_dp,0.6117072285_dp,1.1317332773_dp,1.8449986465_dp,2.8838871676_dp,4.4505826874_dp,6.8637146235_dp,10.6357387185_dp,16.6036560695_dp,&
          26.1532312813_dp,41.6093312507_dp,66.9259189279_dp,108.9256358365_dp,179.5658252755_dp,300.1599611170_dp,509.4060406167_dp,879.0070210811_dp,1544.8462284673_dp,&
          2771.0527089735_dp,5086.4824495012_dp,9591.3232342662_dp,18724.5089389590_dp,38815.6860863636_dp,97090.0884551632_dp,0.3918757640_dp,0.4562600553_dp,0.5987432017_dp,&
          0.8493866271_dp,1.2613397417_dp,1.9239173760_dp,2.9856672819_dp,4.6943066179_dp,7.4662809377_dp,12.0094024620_dp,19.5421859073_dp,32.1923730307_dp,&
          53.7336445114_dp,90.9733989748_dp,156.4213655496_dp,273.5386766621_dp,487.3256351897_dp,886.2968738300_dp,1649.6781570207_dp,3153.6366120257_dp,6231.3588092874_dp,&
          12950.7918869555_dp,30335.2973550919_dp,112436.4040300664_dp]
       aw%aw_erange_matrix(:,20) = [0.1943108649_dp,0.6150583398_dp,1.1391286283_dp,1.8598474852_dp,2.9125723120_dp,4.5047064536_dp,6.9641970949_dp,10.8202994673_dp,16.9405572126_dp,&
          26.7667538902_dp,42.7275330578_dp,68.9713190578_dp,112.6899259480_dp,186.5519734475_dp,313.2643525875_dp,534.3064613806_dp,927.0515751223_dp,1639.2210763092_dp,&
          2960.3313694703_dp,5475.3667191272_dp,10413.0770973923_dp,20521.1923064040_dp,42954.0503775990_dp,108298.4636116199_dp,0.3938629079_dp,0.4592529596_dp,0.6041227721_dp,&
          0.8593932698_dp,1.2798463359_dp,1.9577875801_dp,3.0472157781_dp,4.8058207173_dp,7.6684349009_dp,12.3771167228_dp,20.2149315842_dp,33.4329555659_dp,&
          56.0441348430_dp,95.3280102273_dp,164.7438650793_dp,289.7034968450_dp,519.3108399542_dp,950.9467013146_dp,1783.5755312295_dp,3438.8306422667_dp,6859.0622879598_dp,&
          14393.4131448216_dp,33968.5865839197_dp,126069.5260881044_dp]
       aw%aw_erange_matrix(:,21) = [0.1949032214_dp,0.6171408932_dp,1.1437339237_dp,1.8691160596_dp,2.9305208092_dp,4.5386549110_dp,7.0273811323_dp,10.9366543075_dp,17.1535354611_dp,&
          27.1557321237_dp,43.4386900404_dp,70.2765199610_dp,115.1006922353_dp,191.0437625747_dp,321.7261864634_dp,550.4612767752_dp,958.3845356844_dp,1701.1269394308_dp,&
          3085.3013461297_dp,5734.0280135783_dp,10964.2880847546_dp,21737.9946794511_dp,45785.8936787092_dp,116037.0486525428_dp,0.3950966654_dp,0.4611165613_dp,0.6074832097_dp,&
          0.8656616533_dp,1.2914696376_dp,1.9791158688_dp,3.0860784239_dp,4.8764334118_dp,7.7968297202_dp,12.6114183250_dp,20.6450781310_dp,34.2291299363_dp,&
          57.5329369882_dp,98.1462887434_dp,170.1559468014_dp,300.2707826044_dp,540.3423289538_dp,993.7338507104_dp,1872.8451331777_dp,3630.5621595573_dp,7285.0828648038_dp,&
          15382.7674510084_dp,36483.7054953274_dp,135535.8212170132_dp]
       aw%aw_erange_matrix(:,22) = [0.1957720327_dp,0.6201990235_dp,1.1505096734_dp,1.8827832009_dp,2.9570477503_dp,4.5889452522_dp,7.1212022418_dp,11.1098543744_dp,17.4713901640_dp,&
          27.7378660787_dp,44.5061580411_dp,72.2419557300_dp,118.7435604274_dp,197.8568938439_dp,334.6141501693_dp,575.1781571063_dp,1006.5656809942_dp,1796.8561741546_dp,&
          3279.7752019921_dp,6139.4481789634_dp,11835.3955721937_dp,23679.2326211424_dp,50350.4509662074_dp,128623.6218266693_dp,0.3969067823_dp,0.4638582278_dp,0.6124418747_dp,&
          0.8749358787_dp,1.3087093028_dp,2.0108284560_dp,3.1440110917_dp,4.9819816166_dp,7.9893021889_dp,12.9637375819_dp,21.2940339582_dp,35.4345964000_dp,&
          59.7958078474_dp,102.4478997776_dp,178.4546346000_dp,316.5566527953_dp,572.9382481426_dp,1060.4671926503_dp,2013.0696387129_dp,3934.1906223460_dp,7966.0451652464_dp,&
          16980.6073115596_dp,40584.4378242144_dp,151021.1309224219_dp]
       aw%aw_erange_matrix(:,23) = [0.2002461459_dp,0.6360169682_dp,1.1858073518_dp,1.9545646981_dp,3.0975480162_dp,4.8575927560_dp,7.6268085948_dp,12.0518525036_dp,19.2170535566_dp,&
          30.9684436655_dp,50.4969794088_dp,83.4072432416_dp,139.7137783912_dp,237.6491988999_dp,411.0974939497_dp,724.4779974750_dp,1303.4072415926_dp,2399.9188301826_dp,&
          4536.3545712665_dp,8836.7244095185_dp,17832.0695821586_dp,37588.7313101731_dp,84568.0625046655_dp,226966.2639682454_dp,0.4062390180_dp,0.4781358064_dp,0.6385508701_dp,&
          0.9242446811_dp,1.4012091057_dp,2.1825540574_dp,3.4607422084_dp,5.5649291451_dp,9.0639245813_dp,14.9538420687_dp,25.0059868007_dp,42.4241007821_dp,&
          73.1120947228_dp,128.1758008948_dp,228.9860170869_dp,417.7161578158_dp,779.9730587579_dp,1495.1570001170_dp,2953.2211352557_dp,6039.1568430082_dp,12875.3710433885_dp,&
          29029.3067895134_dp,72895.8090520912_dp,275268.7460731898_dp]
       aw%aw_erange_matrix(:,24) = [0.2024815412_dp,0.6439643614_dp,1.2037021062_dp,1.9913302780_dp,3.1702718090_dp,4.9981368390_dp,7.8942337200_dp,12.5558382127_dp,20.1624293735_dp,&
          32.7408610623_dp,53.8300577411_dp,89.7139523154_dp,151.7555001583_dp,260.9145776443_dp,456.7106658891_dp,815.4955041133_dp,1488.8648679717_dp,2787.2541806911_dp,&
          5369.2458241369_dp,10690.6943830916_dp,22133.0553106771_dp,48080.4578752378_dp,111921.3929908081_dp,310054.3982273569_dp,0.4109084631_dp,0.4853706722_dp,0.6519640922_dp,&
          0.9498844425_dp,1.4498568430_dp,2.2739075720_dp,3.6312570729_dp,5.8827522877_dp,9.6577525920_dp,16.0695496943_dp,27.1196226053_dp,46.4715804298_dp,&
          80.9658775306_dp,143.6572985116_dp,260.0725160256_dp,481.4950700111_dp,914.1408442148_dp,1785.7458912894_dp,3604.4557769198_dp,7558.6415962404_dp,16595.2369123843_dp,&
          38690.8383591849_dp,100368.4160876864_dp,384003.2781817770_dp]
       aw%aw_erange_matrix(:,25) = [0.2039567119_dp,0.6492254626_dp,1.2156080576_dp,2.0159319152_dp,3.2192215462_dp,5.0933017063_dp,8.0764294302_dp,12.9014270844_dp,20.8151563281_dp,&
          33.9736801250_dp,56.1669868274_dp,94.1743421713_dp,160.3531893150_dp,277.7002901640_dp,490.0030483859_dp,882.7913553629_dp,1627.9958775646_dp,3082.6829710924_dp,&
          6016.7364088544_dp,12164.4767840963_dp,25644.1479578213_dp,56926.2601799215_dp,135896.2211163915_dp,385806.8250424920_dp,0.4139924228_dp,0.4901828907_dp,0.6609540956_dp,&
          0.9671847902_dp,1.4828904550_dp,2.3363392233_dp,3.7485736650_dp,6.1029857769_dp,10.0723994400_dp,16.8550410343_dp,28.6209903116_dp,49.3745811661_dp,&
          86.6589100855_dp,155.0112635701_dp,283.1683034569_dp,529.5717113954_dp,1016.9467821925_dp,2012.6150198099_dp,4124.0260826563_dp,8802.2467366642_dp,19734.3484631149_dp,&
          47151.5875767643_dp,125438.4106855630_dp,485666.9083783065_dp]
       aw%aw_erange_matrix(:,26) = [0.2055673467_dp,0.6549847994_dp,1.2286964022_dp,2.0431060098_dp,3.2735552217_dp,5.1994615206_dp,8.2807259499_dp,13.2910468289_dp,21.5553230270_dp,&
          35.3804009379_dp,58.8516753405_dp,99.3364742574_dp,170.3845470209_dp,297.4612814803_dp,529.5892192070_dp,963.7092597510_dp,1797.4232859338_dp,3447.7052295662_dp,&
          6830.3684580497_dp,14053.7281490869_dp,30254.7503089476_dp,68893.4388476543_dp,169553.8298265210_dp,496454.7630005633_dp,0.4173618876_dp,0.4954716961_dp,0.6708971787_dp,&
          0.9864262758_dp,1.5198254021_dp,2.4065203711_dp,3.8811991441_dp,6.3534602006_dp,10.5470373047_dp,17.7604653058_dp,30.3647360665_dp,52.7741998671_dp,&
          93.3866305692_dp,168.5641608069_dp,311.0474288038_dp,588.3411042745_dp,1144.4310035522_dp,2298.6212533427_dp,4791.7758490115_dp,10437.6142304121_dp,23979.2675978929_dp,&
          58996.0415985972_dp,161990.8672374849_dp,638001.6820435197_dp]
       aw%aw_erange_matrix(:,27) = [0.2064011738_dp,0.6579726530_dp,1.2355091212_dp,2.0573041779_dp,3.3020544276_dp,5.2553652812_dp,8.3887498750_dp,13.4979535159_dp,21.9502037788_dp,&
          36.1346314572_dp,60.2988976236_dp,102.1356640703_dp,175.8594793244_dp,308.3241340257_dp,551.5250341372_dp,1008.9534798882_dp,1893.1294938710_dp,3656.3350617894_dp,&
          7301.8176749966_dp,15166.3913332697_dp,33024.3878930755_dp,76262.2133270944_dp,190937.6852233650_dp,569244.0191144008_dp,0.4191072142_dp,0.4982240484_dp,0.6760976197_dp,&
          0.9965344591_dp,1.5393100459_dp,2.4437019466_dp,3.9517791205_dp,6.4873953644_dp,10.8021459673_dp,18.2498228687_dp,31.3128861331_dp,54.6349452492_dp,&
          97.0957788302_dp,176.0965088937_dp,326.6818011124_dp,621.6340733884_dp,1217.4896215046_dp,2464.7258506438_dp,5185.7023064458_dp,11420.5986557085_dp,26590.0760656226_dp,&
          66494.1460453540_dp,185954.5871745703_dp,740487.4545925040_dp]
       aw%aw_erange_matrix(:,28) = [0.2069473623_dp,0.6599321428_dp,1.2399855137_dp,2.0666533148_dp,3.3208617706_dp,5.2923403084_dp,8.4603636295_dp,13.6354580461_dp,22.2133210058_dp,&
          36.6386187468_dp,61.2689400365_dp,104.0182476066_dp,179.5553404692_dp,315.6874040485_dp,566.4625636771_dp,1039.9234228171_dp,1959.0297899986_dp,3800.9768763157_dp,&
          7631.3020520032_dp,15951.4927035377_dp,35001.7471899578_dp,81601.8175684925_dp,206733.5066518533_dp,624217.5061446414_dp,0.4202508244_dp,0.5000323042_dp,0.6795239323_dp,&
          1.0032108704_dp,1.5522101072_dp,2.4683780768_dp,3.9987402083_dp,6.5767534844_dp,10.9728473608_dp,18.5783075397_dp,31.9515376989_dp,55.8930431704_dp,&
          99.6140917521_dp,181.2342367570_dp,337.4010521710_dp,644.5943900368_dp,1268.2134649454_dp,2580.9523062512_dp,5463.8818279494_dp,12122.4744053872_dp,28479.9495951574_dp,&
          72017.3729303258_dp,203996.8322474206_dp,819009.0853796824_dp]
       aw%aw_erange_matrix(:,29) = [0.2080560400_dp,0.6639152616_dp,1.2491054372_dp,2.0857494935_dp,3.3593782386_dp,5.3682673170_dp,8.6078320121_dp,13.9194491543_dp,22.7584698362_dp,&
          37.6864141852_dp,63.2932307438_dp,107.9629979857_dp,187.3348116528_dp,331.2648288771_dp,598.2430837050_dp,1106.2373789992_dp,2101.1756501406_dp,4115.6267659699_dp,&
          8355.2595234098_dp,17697.4412971776_dp,39464.9865942383_dp,93885.6595351764_dp,243995.2377014678_dp,757857.4157596451_dp,0.4225730372_dp,0.5037157949_dp,0.6865270616_dp,&
          1.0168977099_dp,1.5787311035_dp,2.5192573189_dp,4.0958664533_dp,6.7621752902_dp,11.3283177149_dp,19.2649813825_dp,33.2922034421_dp,58.5462453372_dp,&
          104.9520621780_dp,192.1864445864_dp,360.3978380572_dp,694.2110326979_dp,1378.7413088996_dp,2836.6754539486_dp,6083.0125894334_dp,13706.5409808708_dp,32820.1208767645_dp,&
          84990.3921886387_dp,247623.1992522239_dp,1013683.5125962318_dp]
       aw%aw_erange_matrix(:,30) = [0.2086229906_dp,0.6659551693_dp,1.2537871783_dp,2.0955787863_dp,3.3792578586_dp,5.4075645609_dp,8.6843760162_dp,14.0673029317_dp,23.0432110022_dp,&
          38.2356161107_dp,64.3583119800_dp,110.0472074752_dp,191.4640570211_dp,339.5755459375_dp,615.2958976748_dp,1142.0521032722_dp,2178.5187904791_dp,4288.3166661044_dp,&
          8756.6648990544_dp,18677.4814402983_dp,42008.7829868896_dp,101025.2813056025_dp,266225.5339394037_dp,840189.0337543851_dp,0.4237610217_dp,0.5056064334_dp,0.6901342543_dp,&
          1.0239693692_dp,1.5924740338_dp,2.5457011394_dp,4.1465052178_dp,6.8591733952_dp,11.5149439368_dp,19.6269041586_dp,34.0018349981_dp,59.9571832820_dp,&
          107.8053907815_dp,198.0745027431_dp,372.8413526334_dp,721.2561856370_dp,1439.4986038208_dp,2978.6388931596_dp,6430.7685878544_dp,14609.0370583744_dp,35337.3204457132_dp,&
          92690.8378643312_dp,274318.1245022812_dp,1136158.3378886473_dp]
       aw%aw_erange_matrix(:,31) = [0.2111806944_dp,0.6751827547_dp,1.2750561783_dp,2.1404498524_dp,3.4704630723_dp,5.5887796899_dp,9.0392391198_dp,14.7566598510_dp,24.3789215199_dp,&
          40.8290449665_dp,69.4244694660_dp,120.0406239881_dp,211.4397779839_dp,380.1812178791_dp,699.5546530760_dp,1321.2923396911_dp,2571.3552923339_dp,5180.7616058073_dp,&
          10874.4834095190_dp,23980.9003142699_dp,56222.5464472466_dp,142643.8057631811_dp,403694.9767809444_dp,1391473.9581879203_dp,0.4291241952_dp,0.5141930313_dp,0.7066212751_dp,&
          1.0564735310_dp,1.6559845279_dp,2.6685873696_dp,4.3832162627_dp,7.3154660928_dp,12.3988842590_dp,21.3539379109_dp,37.4158312967_dp,66.8065297429_dp,&
          121.7960864484_dp,227.2709704917_dp,435.3318534744_dp,859.0638901184_dp,1754.3542748555_dp,3729.1133253954_dp,8313.7392748885_dp,19642.6835388834_dp,49919.8202587492_dp,&
          139631.7262233275_dp,449045.0624672886_dp,2001280.6863545526_dp]
       aw%aw_erange_matrix(:,32) = [0.2125969665_dp,0.6803102019_dp,1.2869401101_dp,2.1656777276_dp,3.5220698592_dp,5.6919883734_dp,9.2427260150_dp,15.1548178567_dp,25.1564155014_dp,&
          42.3514215141_dp,72.4259368188_dp,126.0217534493_dp,223.5309953320_dp,405.0717238833_dp,751.9446450040_dp,1434.5659541384_dp,2824.3239866350_dp,5768.2796567234_dp,&
          12306.0386561942_dp,27684.0954913213_dp,66565.5855619688_dp,174641.4608164631_dp,518013.8791092461_dp,1904957.6860556419_dp,0.4320966442_dp,0.5189888763_dp,0.7159049223_dp,&
          1.0749081224_dp,1.6922530620_dp,2.7392608109_dp,4.5203739564_dp,7.5819868126_dp,12.9196935934_dp,22.3811392637_dp,39.4674900536_dp,70.9698418255_dp,&
          130.4085513688_dp,245.5007781725_dp,474.9819812083_dp,948.1270588067_dp,1962.2383742802_dp,4237.2879515551_dp,9628.2131752047_dp,23291.8408956365_dp,61017.6014557670_dp,&
          177786.8621323129_dp,605226.9344436220_dp,2870733.8946965942_dp]
       aw%aw_erange_matrix(:,33) = [0.2132695741_dp,0.6827498287_dp,1.2926110628_dp,2.1777560455_dp,3.5468611522_dp,5.7417402458_dp,9.3411717059_dp,15.3481851416_dp,25.5355758791_dp,&
          43.0971936674_dp,73.9035641690_dp,128.9823922196_dp,229.5525727490_dp,417.5523004302_dp,778.4177778483_dp,1492.3138719166_dp,2954.6271780646_dp,6074.6245254553_dp,&
          13063.5733753494_dp,29679.9486357327_dp,72273.4652868895_dp,192878.7961952659_dp,586367.5442187003_dp,2236309.3100171653_dp,0.4335089877_dp,0.5212769423_dp,0.7203531437_dp,&
          1.0837744875_dp,1.7097604899_dp,2.7735040793_dp,4.5870953547_dp,7.7121931725_dp,13.1753123415_dp,22.8878541390_dp,40.4851894216_dp,73.0476851150_dp,&
          134.7363351500_dp,254.7319156092_dp,495.2360230429_dp,994.0826779543_dp,2070.7747619650_dp,4506.3494196423_dp,10336.1536309671_dp,25299.7051970274_dp,67297.3647441372_dp,&
          200234.1922210731_dp,702701.4995745739_dp,3461740.1959740943_dp]
       aw%aw_erange_matrix(:,34) = [0.2150829046_dp,0.6893415019_dp,1.3079869009_dp,2.2106327256_dp,3.6146141065_dp,5.8782715972_dp,9.6125021144_dp,15.8835991874_dp,26.5906849420_dp,&
          45.1838301234_dp,78.0627755497_dp,137.3716442069_dp,246.7428674929_dp,453.4824822938_dp,855.3653288675_dp,1662.0339954080_dp,3342.5859977612_dp,7000.9659576450_dp,&
          15398.1175701049_dp,35979.9458938123_dp,90869.7484760846_dp,255001.8023268931_dp,836010.4680315829_dp,3610137.9489567750_dp,0.4373188176_dp,0.5274790190_dp,0.7324720192_dp,&
          1.1080393825_dp,1.7578825773_dp,2.8680510468_dp,4.7721982641_dp,8.0752861022_dp,13.8921279284_dp,24.3175279960_dp,43.3760413994_dp,78.9944241952_dp,&
          147.2269685231_dp,281.6291372125_dp,554.8973653689_dp,1131.1713880854_dp,2399.3995675088_dp,5335.7345961901_dp,12567.2387424551_dp,31809.1061357633_dp,88442.4269198594_dp,&
          280054.4392423225_dp,1081281.8907916807_dp,6144477.1931226198_dp]
       aw%aw_erange_matrix(:,35) = [0.2165930652_dp,0.6948475016_dp,1.3208906322_dp,2.2383689182_dp,3.6720827635_dp,5.9947219726_dp,9.8452725753_dp,16.3457796585_dp,27.5076111590_dp,&
          47.0105467502_dp,81.7334634348_dp,144.8422973338_dp,262.2054726695_dp,486.1705763020_dp,926.2835742482_dp,1820.8219585072_dp,3712.0243465929_dp,7901.9899443997_dp,&
          17728.9181774639_dp,42482.4642094690_dp,110935.1954153121_dp,326443.5813006668_dp,1154434.8260509006_dp,5781356.9989355495_dp,0.4404941569_dp,0.5326820484_dp,0.7427082354_dp,&
          1.1286586347_dp,1.7990134429_dp,2.9493499546_dp,4.9323877193_dp,8.3916905095_dp,14.5214900744_dp,25.5831661414_dp,45.9585680087_dp,84.3608268729_dp,&
          158.6270259471_dp,306.4949315504_dp,610.8703111254_dp,1262.0054175612_dp,2719.4503872294_dp,6163.5122768420_dp,14862.8018785892_dp,38774.3852334501_dp,112303.9756007481_dp,&
          377443.5489992495_dp,1608248.1327597764_dp,11094076.8939570524_dp]
       aw%aw_erange_matrix(:,36) = [0.2168964316_dp,0.6959553706_dp,1.3234936713_dp,2.2439801696_dp,3.6837434635_dp,6.0184220373_dp,9.8927970601_dp,16.4404642464_dp,27.6961512705_dp,&
          47.3876757393_dp,82.4946502898_dp,146.3991269027_dp,265.4456163087_dp,493.0633053106_dp,941.3454924190_dp,1854.8287099511_dp,3791.9280212223_dp,8099.1945515460_dp,&
          18246.6079122569_dp,43954.2423941564_dp,115594.1123838407_dp,343661.4351591033_dp,1236140.9579337058_dp,6425268.6480215443_dp,0.4411323020_dp,0.5337314275_dp,0.7447804190_dp,&
          1.1328464721_dp,1.8073938481_dp,2.9659692183_dp,4.9652493357_dp,8.4568455273_dp,14.6516263430_dp,25.8460501330_dp,46.4976527613_dp,85.4872484983_dp,&
          161.0348837836_dp,311.7843337008_dp,622.8743701127_dp,1290.3324929360_dp,2789.5315523077_dp,6347.2730727057_dp,15381.2037871690_dp,40382.7525270133_dp,117984.7797854022_dp,&
          401713.0322907210_dp,1750477.9329989499_dp,12721107.1410644483_dp]
       aw%aw_erange_matrix(:,37) = [0.2169478873_dp,0.6961433326_dp,1.3239354887_dp,2.2449330249_dp,3.6857245817_dp,6.0224507836_dp,9.9008803711_dp,16.4565786906_dp,27.7282595944_dp,&
          47.4519471087_dp,82.6244852079_dp,146.6649395716_dp,265.9994656071_dp,494.2429805873_dp,943.9269276343_dp,1860.6668801483_dp,3805.6732878833_dp,8133.1970629416_dp,&
          18336.1264451855_dp,44209.7091040348_dp,116407.0903538253_dp,346690.1240705280_dp,1250715.2222315413_dp,6544211.3271539956_dp,0.4412405490_dp,0.5339095341_dp,0.7451323337_dp,&
          1.1335580802_dp,1.8088186847_dp,2.9687965656_dp,4.9708434948_dp,8.4679443019_dp,14.6738099962_dp,25.8909020511_dp,46.5897246998_dp,85.6798576521_dp,&
          161.4471299613_dp,312.6911520817_dp,624.9356458304_dp,1295.2064355812_dp,2801.6167510162_dp,6379.0437501817_dp,15471.1386798570_dp,40663.0522792135_dp,118981.2717130096_dp,&
          406012.7170565017_dp,1776130.8402275355_dp,13029418.8102731034_dp]
       aw%aw_erange_matrix(:,38) = [0.2169478264_dp,0.6961431310_dp,1.3239350833_dp,2.2449322868_dp,3.6857232781_dp,6.0224484819_dp,9.9008762124_dp,16.4565713050_dp,27.7282478624_dp,&
          47.4519313029_dp,82.6244671381_dp,146.6649193671_dp,265.9994282692_dp,494.2428842899_dp,943.9266830835_dp,1860.6659661581_dp,3805.6698149069_dp,8133.1886582707_dp,&
          18336.1134176163_dp,44209.6890770845_dp,116407.0371097728_dp,346689.9548800496_dp,1250714.0256065591_dp,6544214.1689795908_dp,0.4412404242_dp,0.5339093704_dp,0.7451320789_dp,&
          1.1335576530_dp,1.8088179504_dp,2.9687952328_dp,4.9708410108_dp,8.4679403632_dp,14.6738054972_dp,25.8908986825_dp,46.5897234836_dp,85.6798521392_dp,&
          161.4470960848_dp,312.6910643546_dp,624.9353852151_dp,1295.2050923634_dp,2801.6128421333_dp,6379.0383109776_dp,15471.1352540443_dp,40663.0366129218_dp,118981.2187605095_dp,&
          406012.4514282376_dp,1776130.0998965928_dp,13029440.2357217539_dp]
       aw%aw_erange_matrix(:,39) = [0.2169478601_dp,0.6961432398_dp,1.3239352879_dp,2.2449326098_dp,3.6857237227_dp,6.0224490422_dp,9.9008770007_dp,16.4565726436_dp,27.7282499624_dp,&
          47.4519338851_dp,82.6244700195_dp,146.6649265568_dp,265.9994706539_dp,494.2430824024_dp,943.9273027902_dp,1860.6676639493_dp,3805.6748495209_dp,8133.2017341120_dp,&
          18336.1361661298_dp,44209.7150067573_dp,116407.0961493534_dp,346690.3346687911_dp,1250716.5858198206_dp,6544226.9576467071_dp,0.4412404929_dp,0.5339094543_dp,0.7451321872_dp,&
          1.1335577780_dp,1.8088180645_dp,2.9687953689_dp,4.9708413746_dp,8.4679410855_dp,14.6738061972_dp,25.8908989301_dp,46.5897243043_dp,85.6798634775_dp,&
          161.4471702996_dp,312.6913259474_dp,624.9360109932_dp,1295.2068805749_dp,2801.6182433885_dp,6379.0486819907_dp,15471.1415467669_dp,40663.0440192628_dp,118981.2970290090_dp,&
          406013.2421766578_dp,1776133.1830551089_dp,13029460.0603005476_dp]
       aw%aw_erange_matrix(:,40) = [0.2169479967_dp,0.6961436853_dp,1.3239361471_dp,2.2449340725_dp,3.6857261237_dp,6.0224529463_dp,9.9008832907_dp,16.4565827169_dp,27.7282659546_dp,&
          47.4519585470_dp,82.6245082082_dp,146.6649938493_dp,265.9996029798_dp,494.2433169496_dp,943.9277081020_dp,1860.6683344416_dp,3805.6754034417_dp,8133.2001170045_dp,&
          18336.1289895845_dp,44209.6996877465_dp,116407.1332170651_dp,346690.5901280560_dp,1250716.7713186527_dp,6544221.5728685968_dp,0.4412407721_dp,0.5339098040_dp,0.7451326784_dp,&
          1.1335585175_dp,1.8088192407_dp,2.9687972497_dp,4.9708443493_dp,8.4679458110_dp,14.6738133960_dp,25.8909093105_dp,46.5897425393_dp,85.6799078188_dp,&
          161.4472551451_dp,312.6914494586_dp,624.9362441543_dp,1295.2071101789_dp,2801.6174908422_dp,6379.0447749927_dp,15471.1343512902_dp,40663.0398564033_dp,118981.4429058655_dp,&
          406013.4704579106_dp,1776132.5999892899_dp,13029447.5373880584_dp]
       aw%aw_erange_matrix(:,41) = [0.2169478985_dp,0.6961433640_dp,1.3239355155_dp,2.2449329447_dp,3.6857241342_dp,6.0224494299_dp,9.9008770705_dp,16.4565717454_dp,27.7282471829_dp,&
          47.4519283504_dp,82.6244613012_dp,146.6649201632_dp,265.9994918861_dp,494.2431774153_dp,943.9275927639_dp,1860.6684146201_dp,3805.6761428686_dp,8133.2032452987_dp,&
          18336.1404962747_dp,44209.7307288540_dp,116407.0902591648_dp,346690.0500622534_dp,1250715.6210436232_dp,6544220.2872860357_dp,0.4412405713_dp,0.5339095492_dp,0.7451322969_dp,&
          1.1335578770_dp,1.8088181090_dp,2.9687952461_dp,4.9708407975_dp,8.4679396767_dp,14.6738038573_dp,25.8908957844_dp,46.5897217627_dp,85.6798744744_dp,&
          161.4472174171_dp,312.6914357359_dp,624.9363266513_dp,1295.2074549019_dp,2801.6186711074_dp,6379.0488934272_dp,15471.1492766609_dp,40663.0531671503_dp,118981.2047286658_dp,&
          406012.8144159144_dp,1776131.7484223465_dp,13029447.0094524790_dp]    
    case(26)
       aw%energy_range(:) = [799.0_dp, 995.0_dp, 1293.0_dp, 1738.0_dp, 2238.0_dp, 3009.0_dp, 4377.0_dp,&
            6256.0_dp, 9034.0_dp, 15564.0_dp, 19500.0_dp, 22300.0_dp, 24783.0_dp, 41198.0_dp, 94407.0_dp, 189080.0_dp,&
            457444.0_dp, 2101965.0_dp, 14140999.0_dp]
       aw%aw_erange_matrix(:, 1) = [0.13642311899920728_dp,  0.4200545572769177_dp,  0.7370807753440111_dp,  1.1133066772659823_dp,  1.5801544604816498_dp,  2.1776122735126364_dp,&
            2.958051608229127_dp,  3.9913517870782127_dp,  5.371912501848569_dp,  7.228355464212482_dp,  9.73704628190353_dp,  13.14105442578561_dp,  17.776888498885384_dp,&
            24.112423571820134_dp,  32.80110412846477_dp,  44.760201041042265_dp,  61.285574367976004_dp,  84.22431526497385_dp,  116.24572841708785_dp,  161.2969910194418_dp,&
            225.45259795885693_dp,  318.7313048003067_dp,  459.67927623297453_dp,  689.4586002694374_dp,  1129.2608611575954_dp,  2384.899332663137_dp,  0.27462662125713716_dp,&
            0.2963690140650455_dp,  0.34194449526494647_dp,  0.4156960072544763_dp,  0.5246016522241007_dp,  0.678958500149969_dp,  0.8934517120897347_dp,  1.1887228663585705_dp,&
            1.5936139137203535_dp,  2.1483557077661164_dp,  2.9090998337926104_dp,  3.9543808256482187_dp,  5.394380930104983_dp,  7.384320521229908_dp,  10.144053822937831_dp,&
            13.98732687567056_dp,  19.366933030106217_dp,  26.948262498364663_dp,  37.739388497689305_dp,  53.348600455664965_dp,  76.56744981421966_dp,  112.8945538518047_dp,&
            175.19453971323657_dp,  301.19411259128674_dp,  645.6383267719649_dp,  2472.504957810595_dp]
       aw%aw_erange_matrix(:, 2) = [0.13925292153649474_dp,  0.4292451559612877_dp,  0.7548554587233549_dp,  1.1437833384686722_dp,  1.6300177124420845_dp,  2.2571802345230276_dp,&
            3.0828977198982144_dp,  4.1847303613644495_dp,  5.668362141207399_dp,  7.679038990085008_dp,  10.417660954546289_dp,  14.163558801605095_dp,  19.30691297965789_dp,&
            26.395168657294754_dp,  36.199950721972876_dp,  49.8144257942317_dp,  68.79692224946095_dp,  95.38754743793149_dp,  132.8473999791564_dp,  186.0241881676569_dp,&
            262.3929640997449_dp,  374.2539456946303_dp,  544.2375437714703_dp,  822.2186120363165_dp,  1354.4009319403576_dp,  2870.8477591124674_dp,  0.28040112472679574_dp,&
            0.30356564912420003_dp,  0.3522269185689752_dp,  0.43123890171761686_dp,  0.5484284775560514_dp,  0.7154038747678296_dp,  0.9488235797546415_dp,  1.2722683343829815_dp,&
            1.7189393959653365_dp,  2.3355248974949743_dp,  3.1877439258332987_dp,  4.368323267393725_dp,  6.008533513106221_dp,  8.294995170661158_dp,  11.494429371223015_dp,&
            15.990739727277512_dp,  22.342169324408708_dp,  31.373674894712472_dp,  44.337963681573235_dp,  63.228794541463415_dp,  91.4807612744294_dp,  135.78931506653154_dp,&
            211.68822580031673_dp,  364.5832859080767_dp,  780.7919711700442_dp,  2983.4784258039044_dp]
       aw%aw_erange_matrix(:, 3) = [0.14255313239112444_dp,  0.44000221057671707_dp,  0.7757923584694003_dp,  1.1799690097399198_dp,  1.6897357815984675_dp,  2.3533173555606655_dp,&
            3.235072042865805_dp,  4.422504788444178_dp,  6.036065942606674_dp,  8.242988234525232_dp,  11.27697154139003_dp,  15.466358681870163_dp,  21.274694591939472_dp,&
            29.359452647866682_dp,  40.65762974807096_dp,  56.51155510141434_dp,  78.85603619054034_dp,  110.50208558142461_dp,  155.58114995208211_dp,  220.28032446631136_dp,&
            314.1757282936284_dp,  453.002579503336_dp,  665.5199679038552_dp,  1014.5604898901777_dp,  1683.22275383598_dp,  3584.316130956811_dp,  0.28714175969757383_dp,&
            0.31204447582876227_dp,  0.364492437091783_dp,  0.4499982622781763_dp,  0.5774890351263188_dp,  0.7602856703230421_dp,  1.0176474576111718_dp,  1.3770666782034746_dp,&
            1.8776058066032693_dp,  2.5747281832062225_dp,  3.5472996309663594_dp,  4.907773444426638_dp,  6.817081323539761_dp,  9.506552191680779_dp,  13.31048343115922_dp,&
            18.71524624188938_dp,  26.435090191201706_dp,  37.53395349827001_dp,  53.63476429545202_dp,  77.3183278694657_dp,  112.99764481556716_dp,  169.1712283981583_dp,  265.33605383413214_dp,  458.1910477032067_dp,  980.4158381921823_dp,  3736.258588894085_dp]
       aw%aw_erange_matrix(:, 4) = [0.1461752355773306_dp,  0.45185762356828396_dp,  0.7990360544025357_dp,  1.2205083800767933_dp,  1.7572993236504337_dp,  2.4631760346842047_dp,&
            3.4107040936751862_dp,  4.699662235621062_dp,  6.468947005916596_dp,  8.913592437239087_dp,  12.309286000158947_dp,  17.047905694553386_dp,  23.689335953611156_dp,&
            33.03746908843982_dp,  46.25240948793399_dp,  65.01743260557059_dp,  91.79007233646534_dp,  130.18588342772145_dp,  185.58109233196095_dp,  266.1051496392308_dp,&
            384.41585349757804_dp,  561.3175220599338_dp,  834.5950790025629_dp,  1285.9791179040656_dp,  2151.82484272534_dp,  4607.654058324236_dp,  0.2945477387281676_dp,&
            0.3214595830325172_dp,  0.37830480422867613_dp,  0.4714034978877635_dp,  0.611040549973395_dp,  0.8126695125533263_dp,  1.0988217520443189_dp,  1.5019658160165765_dp,&
            2.06870896475044_dp,  2.8659496939699607_dp,  3.9899051060400774_dp,  5.579409370592633_dp,  7.835605731464965_dp,  11.051295593199974_dp,  15.655055606015468_dp,&
            22.278373441162756_dp,  31.859818549870184_dp,  45.8120996920796_dp,  66.3057001417783_dp,  96.79792736663595_dp,  143.16536816614507_dp,  216.57964119429673_dp,  342.31559687568875_dp,  593.3221880856304_dp,  1268.7961696472178_dp,  4820.528614472729_dp]
       aw%aw_erange_matrix(:, 5) = [0.1491837006061548_dp,  0.46174475862674996_dp,  0.8185592742668346_dp,  1.2548602679290635_dp,  1.8150957152976515_dp,  2.558059962915574_dp,&
            3.563852447313441_dp,  4.943651210588896_dp,  6.8536849594075475_dp,  9.515417267221789_dp,  13.244928718595732_dp,  18.495967806145995_dp,  25.923415200427836_dp,&
            36.47741618080311_dp,  51.54392422008612_dp,  73.15615247652946_dp,  104.31586449506281_dp,  149.48862759379944_dp,  215.38537956732102_dp,  312.2481145191122_dp,&
            456.12776079236704_dp,  673.453709288462_dp,  1012.0267276035421_dp,  1574.3691121807644_dp,  2654.80297068303_dp,  5713.432382523477_dp,  0.30070547042796897_dp,&
            0.329369148940076_dp,  0.3900659972080265_dp,  0.4898601972791792_dp,  0.6402960685231848_dp,  0.8588226443075349_dp,  1.1710637934386325_dp,  1.6142423603884266_dp,&
            2.242256373868115_dp,  3.133186411937803_dp,  4.400427168655394_dp,  6.20926516954263_dp,  8.801701004315937_dp,  12.533857850605154_dp,  17.932816993690953_dp,&
            25.783927946483768_dp,  37.26719820010372_dp,  54.17638997139517_dp,  79.2886731198742_dp,  117.0433439522649_dp,  174.96435451384102_dp,  267.2144601065969_dp,  425.4335136047249_dp,  740.2128400786926_dp,  1582.657563073928_dp,  5997.255668095278_dp]
       aw%aw_erange_matrix(:, 6) = [0.15260234927102537_dp,  0.47302539641929003_dp,  0.840991102837249_dp,  1.294672645194693_dp,  1.8827021467743683_dp,  2.6700932738662866_dp,&
            3.7463755633773785_dp,  5.2371602474102446_dp,  7.32087360313924_dp,  10.253221414128866_dp,  14.403238412842787_dp,  20.306771564613143_dp,  28.74634048361317_dp,&
            40.87115597284819_dp,  58.378725955266525_dp,  83.7917688772186_dp,  120.88443420446742_dp,  175.34684950976114_dp,  255.8424057631557_dp,  375.7503122503565_dp,&
            556.2295831578734_dp,  832.2606405042899_dp,  1266.9035893238452_dp,  1994.1233884282024_dp,  3394.9032163950633_dp,  7352.339298447235_dp,  0.3077100277629248_dp,&
            0.33845848997185407_dp,  0.4037598644074573_dp,  0.5116119745332622_dp,  0.6751498929108865_dp,  0.9143651994460601_dp,  1.2588616224335032_dp,  1.75204662427369_dp,&
            2.4574104002902457_dp,  3.4679202527071955_dp,  4.920122312459791_dp,  7.015412894339497_dp,  10.052310636158587_dp,  14.475746343180617_dp,  20.952944399378328_dp,&
            30.491434160561496_dp,  44.62526181307512_dp,  65.71573855063913_dp,  97.45740287039392_dp,  145.7937686814701_dp,  220.7911788117634_dp,  341.2160648337932_dp,  548.3643432689413_dp,  959.1555990081102_dp,  2051.35284281865_dp,  7749.544195896773_dp]
       aw%aw_erange_matrix(:, 7) = [0.15677016271388225_dp,  0.4868453590049744_dp,  0.8687054372436632_dp,  1.3443711902060458_dp,  1.9680308145076961_dp,  2.8130758201236987_dp,&
            3.981916598375189_dp,  5.620150667525299_dp,  7.937365822573636_dp,  11.238009575994626_dp,  15.967571787550442_dp,  22.782179390039534_dp,  32.65420224580025_dp,&
            47.03362680645636_dp,  68.09645542153785_dp,  99.13012280380028_dp,  145.13706650883265_dp,  213.79117632911775_dp,  316.97982833553084_dp,  473.36316641843865_dp,&
            712.8558730426033_dp,  1085.3059686529182_dp,  1680.4570998839688_dp,  2686.878722527825_dp,  4633.878278204835_dp,  10122.376598223758_dp,  0.3162602891789823_dp,&
            0.3496899146467389_dp,  0.4209457453791445_dp,  0.5393025302878296_dp,  0.7200886783450724_dp,  0.9868388578481941_dp,  1.3747701483151744_dp,  1.9361268964127185_dp,&
            2.748293611495538_dp,  3.9261112821432245_dp,  5.640652928475169_dp,  8.14800606157719_dp,  11.833678475242102_dp,  17.28159907773421_dp,  25.382209512440305_dp,&
            37.50344417920538_dp,  55.76466551122398_dp,  83.48401940950261_dp,  125.93307609242711_dp,  191.6876355183379_dp,  295.31901285869344_dp,  463.7601116874953_dp,  755.1838770980437_dp,  1331.5355795873638_dp,  2851.124163900499_dp,  10729.678598478598_dp]
       aw%aw_erange_matrix(:, 8) = [0.16057542542593983_dp,  0.49952962028824754_dp,  0.8943731123892699_dp,  1.3909076292240965_dp,  2.048865907808532_dp,  2.950126882998398_dp,&
            4.2103425278857385_dp,  5.995954911767919_dp,  8.549525380892058_dp,  12.227842251849554_dp,  17.559745488437873_dp,  25.33455063970661_dp,  36.7382813247586_dp,&
            53.565155655152715_dp,  78.54849962850257_dp,  115.8829384695536_dp,  172.05630085125_dp,  257.19020516627944_dp,  387.2342312585102_dp,  587.6466072063987_dp,&
            899.8509121387558_dp,  1393.5887995892674_dp,  2194.6568910457167_dp,  3565.0884460947423_dp,  6230.642206140841_dp,  13732.602514515214_dp,  0.32407734329056537_dp,&
            0.3600930324125088_dp,  0.43712641231000066_dp,  0.5657644526879636_dp,  0.7636093738551866_dp,  1.0579110134655951_dp,  1.4898487496937127_dp,  2.121183742243155_dp,&
            3.0444831771204277_dp,  4.398856321226892_dp,  6.39429509803701_dp,  9.349562092561689_dp,  13.751584428416358_dp,  20.349284881552315_dp,  30.303092531346753_dp,&
            45.425519514426874_dp,  68.57296149770674_dp,  104.29448877473207_dp,  159.93517513497875_dp,  247.604939611224_dp,  388.0261636148984_dp,  619.3536447537377_dp,  1022.6891506041642_dp,  1819.6861630933722_dp,  3904.600906393468_dp,  14642.034530501202_dp]
       aw%aw_erange_matrix(:, 9) = [0.16432089011213946_dp,  0.5120783448939726_dp,  0.9199886972681118_dp,  1.437840597555715_dp,  2.1313000192264977_dp,  3.0914596821926974_dp,&
            4.448548492729215_dp,  6.392274593692749_dp,  9.202519740089308_dp,  13.296161754347343_dp,  19.29914269890146_dp,  28.158315667958256_dp,  41.31644063618593_dp,&
            60.98825519945388_dp,  90.59976907891678_dp,  135.49373172032819_dp,  204.0739991922362_dp,  309.6835071732142_dp,  473.73172471065976_dp,  731.0162782781778_dp,&
            1139.1233191834983_dp,  1796.2958792734264_dp,  2880.674779835234_dp,  4760.907205234219_dp,  8443.643014901543_dp,  18797.512140801202_dp,  0.33178166646984186_dp,&
            0.3704756224159047_dp,  0.4535273760622626_dp,  0.5929656383094599_dp,  0.8089125230587538_dp,  1.1327811095321214_dp,  1.6125162306884913_dp,  2.320823479534045_dp,&
            3.3679786225905852_dp,  4.921818456475184_dp,  7.239122346149252_dp,  10.715224167142727_dp,  15.963061445035835_dp,  23.94020351071544_dp,  36.154875568991734_dp,&
            55.00354905209596_dp,  84.33076754977621_dp,  130.3718204724196_dp,  203.3763607524491_dp,  320.5163292162362_dp,  511.499766113826_dp,  831.0632228122023_dp,  1393.996082618058_dp,  2507.6502273684464_dp,  5398.730047404666_dp,  20174.243482162154_dp]
       aw%aw_erange_matrix(:, 10) = [0.16955168876185606_dp,  0.5297128492681893_dp,  0.9563682595025127_dp,  1.5053456637551013_dp,  2.251458851423233_dp,  3.30025374096381_dp,&
            4.805213971653435_dp,  6.993792120211177_dp,  10.207452358120111_dp,  14.963970947103403_dp,  22.05530884512796_dp,  32.702779510861056_dp,  48.80530736719183_dp,&
            73.34037086293017_dp,  111.01793202252156_dp,  169.3581666315736_dp,  260.4877531652313_dp,  404.17142433017716_dp,  633.0038005727723_dp,  1001.4735710632112_dp,&
            1602.2710504984482_dp,  2597.339796633601_dp,  4284.439326490744_dp,  7277.272159321957_dp,  13217.442145549_dp,  29915.954039362532_dp,  0.34255855985607125_dp,&
            0.3852211686852873_dp,  0.4772548827347379_dp,  0.6329784737782622_dp,  0.8765598435301294_dp,  1.246188563782287_dp,  1.800992159648441_dp,  2.6320669330081983_dp,&
            3.8799679680225974_dp,  5.762558150515838_dp,  8.61966780966633_dp,  12.985306239805702_dp,  19.705483921340985_dp,  30.13234025723871_dp,  46.44736905029676_dp,&
            72.20575596588289_dp,  113.2647934625349_dp,  179.39118814888906_dp,  287.0985893630822_dp,  464.81061857234823_dp,  762.7938761510428_dp,  1274.5462739445218_dp,  2193.7537015644107_dp,  4023.287609783773_dp,  8726.293698066565_dp,  32456.229019326278_dp]
       aw%aw_erange_matrix(:, 11) = [0.17161078656390408_dp,  0.5366905364814994_dp,  0.97088881076514_dp,  1.5325699975422944_dp,  2.300445459098221_dp,  3.386304928549832_dp,&
            4.953814656226931_dp,  7.2471781401179225_dp,  10.635564848787308_dp,  15.682784328062063_dp,  23.257647540954054_dp,  34.710448891778356_dp,  52.15789633691571_dp,&
            78.94769861589022_dp,  120.42398853039909_dp,  185.20235595500606_dp,  287.3191078945018_dp,  449.9016881055523_dp,  711.5314989639854_dp,  1137.4846615524127_dp,&
            1840.156204841028_dp,  3018.1229428253846_dp,  5039.367148043016_dp,  8662.764242204496_dp,  15902.181332715702_dp,  36264.30820408944_dp,  0.3468065253991383_dp,&
            0.3911064628011852_dp,  0.48686770710592453_dp,  0.649407004504934_dp,  0.9046709269666874_dp,  1.2938615472397417_dp,  1.8811396766420316_dp,  2.7659895528533784_dp,&
            4.102970732081106_dp,  6.133424978400673_dp,  9.236763352774814_dp,  14.014155411820974_dp,  21.42640568242634_dp,  33.02345872650458_dp,  51.33068406408745_dp,&
            80.5068763005674_dp,  127.47998440706951_dp,  203.93737778272407_dp,  329.88071553711586_dp,  540.1552172919626_dp,  897.0521095218941_dp,  1517.2143251608513_dp,  2641.8070030495096_dp,  4889.503800534578_dp,  10648.322208945656_dp,  39537.49857015622_dp]
       aw%aw_erange_matrix(:, 12) = [0.1728061198835349_dp,  0.5407506981101403_dp,  0.9793714038026303_dp,  1.5485484444330022_dp,  2.3293373186010964_dp,  3.4373065991570115_dp,&
            5.0423230776903445_dp,  7.398853217497418_dp,  10.89314740028624_dp,  16.11757919438818_dp,  23.988966085187993_dp,  35.93874008813103_dp,  54.22162060992446_dp,&
            82.42176042102257_dp,  126.29163129215709_dp,  195.15830646295916_dp,  304.3099266577351_dp,  479.10015694274045_dp,  762.1158665504651_dp,  1225.9324082258843_dp,&
            1996.438395305476_dp,  3297.6059283833983_dp,  5546.6394027927_dp,  9604.798156883016_dp,  17747.506304598486_dp,  40662.28777164996_dp,  0.3492740223484516_dp,&
            0.3945444048035226_dp,  0.4925209274047293_dp,  0.6591264626127379_dp,  0.9213922685098963_dp,  1.322366949443488_dp,  1.929314142208889_dp,  2.8469205908136956_dp,&
            4.238488142375349_dp,  6.360114867321287_dp,  9.616265458690265_dp,  14.650939404307358_dp,  22.498728594279285_dp,  34.837787125550584_dp,  54.418346394612065_dp,&
            85.79761930703064_dp,  136.61734637002158_dp,  219.8591273308433_dp,  357.9019839857931_dp,  590.021281618985_dp,  986.9076588661666_dp,  1681.5547940952313_dp,  2948.8741403686568_dp,  5489.390378606207_dp,  11987.379922744534_dp,  44468.08958434164_dp]
       aw%aw_erange_matrix(:, 13) = [0.1737308948597355_dp,  0.5438966877521114_dp,  0.9859610380912929_dp,  1.560999073537973_dp,  2.3519219694926194_dp,  3.4773019976149877_dp,&
            5.11195442725365_dp,  7.518569316391805_dp,  11.09714005655843_dp,  16.463120094991794_dp,  24.572288754382765_dp,  36.922235617447974_dp,  55.880761448416834_dp,&
            85.22675085299979_dp,  131.05081429199765_dp,  203.27261201403803_dp,  318.2293711657402_dp,  503.152754643253_dp,  804.032555977885_dp,  1299.6927362845154_dp,&
            2127.667695664749_dp,  3534.0326704862105_dp,  5979.169285255363_dp,  10414.573972160117_dp,  19345.73831740431_dp,  44492.46471079593_dp,  0.3511837705016231_dp,&
            0.39721506693266434_dp,  0.4969316438045135_dp,  0.6667392102402926_dp,  0.9345354080893498_dp,  1.3448487389874917_dp,  1.967438938031642_dp,  2.9111947746582425_dp,&
            4.346509228913013_dp,  6.541503533662434_dp,  9.921152543346885_dp,  15.164693048542738_dp,  23.367741212788335_dp,  36.315072183771754_dp,  56.94501925463534_dp,&
            90.15017067236717_dp,  144.17713392303253_dp,  233.11210787240515_dp,  381.3788199371866_dp,  632.0944337435812_dp,  1063.2960978183357_dp,  1822.393451001988_dp,  3214.1964330189876_dp,  6011.5393853613305_dp,  13158.00196550311_dp,  48777.36298097777_dp]
       aw%aw_erange_matrix(:, 14) = [0.17799130240706984_dp,  0.558445410809649_dp,  1.0166297309491044_dp,  1.6193831652945954_dp,  2.4586620686801695_dp,  3.667829742572521_dp,&
            5.446314891054136_dp,  8.098126747261656_dp,  12.093029411971516_dp,  18.164916061856836_dp,  27.47177125227526_dp,  41.85862387738217_dp,  64.29463269296241_dp,&
            99.6081132229676_dp,  155.73791067776023_dp,  245.8920402977068_dp,  392.3244219552829_dp,  633.0449562544749_dp,  1033.945999167842_dp,  1711.154006548549_dp,&
            2873.283514523331_dp,  4904.516412355985_dp,  8541.213772971501_dp,  15320.756510308098_dp,  29238.418877095413_dp,  68583.17313114721_dp,  0.35999057854489624_dp,&
            0.40964342132227255_dp,  0.517678596262729_dp,  0.702891167526621_dp,  0.9974953889055737_dp,  1.4534562037793455_dp,  2.1531958092699743_dp,  3.2271377404365795_dp,&
            4.882410917802139_dp,  7.450138129922876_dp,  11.464071575277625_dp,  17.792734308409212_dp,  27.864023043096953_dp,  44.05169019349174_dp,  70.34922266622794_dp,&
            113.5616876658134_dp,  185.44652039869408_dp,  306.62505535472695_dp,  513.8711614457693_dp,  874.0321926895938_dp,  1511.615678189863_dp,  2667.372553488233_dp,  4842.950527196566_dp,  9285.588160319077_dp,  20597.325965171356_dp,  76161.8596274348_dp]
       aw%aw_erange_matrix(:, 15) = [0.18427327646299554_dp,  0.5800678142111884_dp,  1.062813174396542_dp,  1.7086710138090095_dp,  2.6245433407677226_dp,  3.9687456084576795_dp,&
            5.983088088759827_dp,  9.044214821988016_dp,  13.747180049983248_dp,  21.04333745984166_dp,  32.47068559616262_dp,  50.54348632696267_dp,  79.41986612597532_dp,&
            126.06149615339397_dp,  202.27775198215787_dp,  328.3861574350742_dp,  539.8798783942364_dp,  899.8005204338677_dp,  1522.1727316803415_dp,  2617.4198392181756_dp,&
            4582.775104882628_dp,  8188.716947049966_dp,  14986.46895717419_dp,  28322.082953313376_dp,  56826.75793518779_dp,  138481.00974898663_dp,  0.3730028368021069_dp,&
            0.428354171574684_dp,  0.549600032397171_dp,  0.7595995851576278_dp,  1.098014096850626_dp,  1.629865924998882_dp,  2.460250070930056_dp,  3.7589670939362154_dp,&
            5.801833730520023_dp,  9.040587178848838_dp,  14.222505427016364_dp,  22.59766467101562_dp,  36.28293067415338_dp,  58.9106289633826_dp,  96.80295342684235_dp,&
            161.1348984666336_dp,  271.99263116463095_dp,  466.14531748937003_dp,  812.2703291175023_dp,  1441.5731111728833_dp,  2611.4422570763995_dp,  4844.665731670444_dp,  9266.732583950343_dp,  18647.213606093355_dp,  42656.46566405554_dp,  157660.6216097217_dp]
       aw%aw_erange_matrix(:, 16) = [0.1889166187549335_dp,  0.5961850834729495_dp,  1.097718788260886_dp,  1.7772527519739671_dp,  2.754100363430344_dp,  4.207746123352419_dp,&
            6.416725271991069_dp,  9.821987600643263_dp,  15.131996317599018_dp,  23.4995566780628_dp,  36.8234002615115_dp,  58.269695998899806_dp,  93.18703062085261_dp,&
            150.7367390335279_dp,  246.84666046331515_dp,  409.6601639519172_dp,  689.7883982951697_dp,  1180.0211938503542_dp,  2054.1392300559787_dp,  3645.406178397627_dp,&
            6610.227338646847_dp,  12282.21946267068_dp,  23479.16184253136_dp,  46536.49778293873_dp,  97988.005994822_dp,  248271.71627050647_dp,  0.3826418964555519_dp,&
            0.44249042636795055_dp,  0.5742647482387918_dp,  0.8042933870412514_dp,  1.1786909803221888_dp,  1.7740059766551828_dp,  2.7157599189739634_dp,  4.210014007474866_dp,&
            6.5973245402497005_dp,  10.445947244529027_dp,  16.71490137908719_dp,  27.043296616534693_dp,  44.27161732479843_dp,  73.39668617871985_dp,  123.35300047165461_dp,&
            210.40156572097126_dp,  364.7172011277595_dp,  643.5002779012159_dp,  1157.7720495453327_dp,  2128.791663589068_dp,  4011.0945220895405_dp,  7773.563988357228_dp,  15593.849284273198_dp,  32917.0364973997_dp,  78051.06376348506_dp,  289942.76429117593_dp]
       aw%aw_erange_matrix(:, 17) = [0.194043824675455_dp,  0.6141201650984125_dp,  1.137056339966442_dp,  1.8556822490348746_dp,  2.9045171634501883_dp,  4.489491230295626_dp,&
            6.935918070559926_dp,  10.768297969516013_dp,  16.84551695032909_dp,  26.59345448264271_dp,  42.41124163468271_dp,  68.39190043336116_dp,  111.6218630548935_dp,&
            184.5662784702303_dp,  309.53243627996767_dp,  527.1995714305152_dp,  913.3020625088166_dp,  1612.1087655214071_dp,  2905.5631924229647_dp,  5360.760592207567_dp,&
            10156.351679516332_dp,  19836.90395856863_dp,  40153.56249059423_dp,  84945.40622185206_dp,  191879.29446083526_dp,  516853.91463486024_dp,  0.3933068197531215_dp,&
            0.45841433657798863_dp,  0.6026132459496706_dp,  0.8565818421331466_dp,  1.2746407100646033_dp,  1.948249294221966_dp,  3.029862020316641_dp,  4.774339119270673_dp,&
            7.611287961919275_dp,  12.27301847355483_dp,  20.02418710434074_dp,  33.08062796253295_dp,  55.38677548001659_dp,  94.08664339680347_dp,  162.36610582016837_dp,&
            285.0728404570405_dp,  510.11475389805196_dp,  932.2410214966416_dp,  1744.2541855756215_dp,  3351.323885452711_dp,  6636.946607638909_dp,  13613.456185258428_dp,  29125.998944110084_dp,  65928.95446339808_dp,  166172.86831676983_dp,  628359.6031461136_dp]
       aw%aw_erange_matrix(:, 18) = [0.20097761439605452_dp,  0.6386142361636943_dp,  1.1916436123207983_dp,  1.9665277909771697_dp,  3.121155070595262_dp,  4.903104495370994_dp,&
            7.713190752815356_dp,  12.214218517736429_dp,  19.52076540766317_dp,  31.536135148981113_dp,  51.56105718002733_dp,  85.41349834924547_dp,  143.5295342840862_dp,&
            244.98992662983036_dp,  425.42112573916177_dp,  752.9079358406352_dp,  1360.9881333864396_dp,  2519.3466781523043_dp,  4791.042590791367_dp,  9397.53189950536_dp,&
            19109.699703842572_dp,  40556.572222445735_dp,  90659.45721129453_dp,  216333.95478046086_dp,  564416.1292362113_dp,  1760880.0389147545_dp,  0.4077664557826492_dp,&
            0.48049567628653145_dp,  0.6429124643807497_dp,  0.9325592299631679_dp,  1.4169440263827862_dp,  2.2120247106687776_dp,  3.515599466670522_dp,  5.666879358451872_dp,&
            9.253815059676851_dp,  15.309414383937733_dp,  25.677138365111702_dp,  43.7042055643289_dp,  75.58521070695693_dp,  133.0274420624619_dp,  238.67600100704755_dp,&
            437.47737171537983_dp,  821.2587051299821_dp,  1583.8578781748606_dp,  3150.0096236720124_dp,  6491.449657956827_dp,  13947.036522523289_dp,  31498.820504827483_dp,  75639.29476513214_dp,  196582.0757874882_dp,  575257.5915715584_dp,  2362446.5347060743_dp]
       aw%aw_erange_matrix(:, 19) = [0.20658002404396444_dp,  0.6586140986839257_dp,  1.2369737792708104_dp,  2.060361497488803_dp,  3.3082012274969337_dp,  5.267442774090238_dp,&
            8.412127582372415_dp,  13.542811939893562_dp,  22.035981804979933_dp,  36.29881109275795_dp,  60.6146407237101_dp,  102.74788434800739_dp,  177.06018691028456_dp,&
            310.7136329238608_dp,  556.36637029124_dp,  1018.9766332082758_dp,  1914.422668305446_dp,  3702.980284069834_dp,  7407.809066534502_dp,  15417.95314386633_dp,&
            33651.57917590195_dp,  77878.97148771118_dp,  194227.98208713965_dp,  535582.0636490324_dp,  1709364.267502628_dp,  7045890.294848618_dp,  0.4194816618416714_dp,&
            0.49881572197479607_dp,  0.6772179216261252_dp,  0.9987160310392028_dp,  1.5435226281295777_dp,  2.451754987162261_dp,  3.9670946613559326_dp,  6.516517085204095_dp,&
            10.85773397839894_dp,  18.356701405781113_dp,  31.520492951672992_dp,  55.0435050002148_dp,  97.91266305771113_dp,  177.760944100452_dp,  330.14945470086406_dp,&
            629.0496827419165_dp,  1233.841581178721_dp,  2502.1075901274335_dp,  5274.877655599449_dp,  11644.098372122406_dp,  27178.121421602995_dp,  68003.68316809284_dp,  186203.45469776812_dp,  576894.7309753906_dp,  2152261.8479348915_dp,  11566361.80487531_dp]
       aw%aw_erange_matrix(:, 20) = [0.20878089337233605_dp,  0.6665236543193817_dp,  1.2550931614146674_dp,  2.0983237232110734_dp,  3.38481582751189_dp,  5.418564331713917_dp,&
            8.705828067862388_dp,  14.108794066294884_dp,  23.12322697734966_dp,  38.390182043454885_dp,  64.65856322054098_dp,  110.63583466643804_dp,  192.63262310028742_dp,&
            341.932785309222_dp,  620.1449956190338_dp,  1152.2653509765064_dp,  2200.646613149521_dp,  4337.910771163601_dp,  8872.44998086692_dp,  18961.533603283864_dp,&
            42748.40639563531_dp,  103069.49259277884_dp,  271388.86064721714_dp,  808265.503410061_dp,  2915885.1156875044_dp,  15257006.288806498_dp,  0.4240919425078822_dp,&
            0.5061337957364646_dp,  0.6911418800611648_dp,  1.0259473122312546_dp,  1.5963227579992894_dp,  2.5531163460188835_dp,  4.160724269135366_dp,  6.8864489147035135_dp,&
            11.567503857113842_dp,  19.729006828740097_dp,  34.20240654961173_dp,  60.35680029031647_dp,  108.61537925165752_dp,  199.75021678633257_dp,  376.3927619922551_dp,&
            728.9995944299609_dp,  1456.9587839825_dp,  3019.6115675071173_dp,  6531.619658687757_dp,  14871.950539519852_dp,  36069.05923591306_dp,  94800.92746541071_dp,  277390.3096379278_dp,  946568.3924295901_dp,  4140822.150621181_dp,  30376446.323901277_dp]       
    case(28)
       aw%energy_range(:) = [1545.0_dp, 2002.0_dp, 2600.0_dp, 3300.0_dp, 4000.0_dp, 5000.0_dp, 5800.0_dp,&
            7000.0_dp, 8500.0_dp, 11000.0_dp, 14000.0_dp, 18000.0_dp, 22000.0_dp, 30000.0_dp, 40000.0_dp, 55000.0_dp,&
            75000.0_dp, 100000.0_dp, 140000.0_dp, 200000.0_dp, 280000.0_dp, 400000.0_dp, 700000.0_dp, 1200000.0_dp,&
            2000000.0_dp, 4500000.0_dp, 10000000.0_dp, 50000000.0_dp]
       aw%aw_erange_matrix(:, 1) = [0.13575757270404953_dp,  0.4178973639556045_dp,  0.7329236428021971_dp,  1.1062108040376035_dp,  1.568601536611355_dp,  2.1592688100674207_dp,&
            2.9294121393197234_dp,  3.9472075381811034_dp,  5.304565115169143_dp,  7.126455157872397_dp,  9.58387216560659_dp,  12.911957380816203_dp,  17.435457004483258_dp,&
            23.60465355900629_dp,  32.04632634372974_dp,  43.63642652454945_dp,  59.604434449969176_dp,  81.68464372636545_dp,  112.33867792077912_dp,  155.0906277872644_dp,&
            215.05233203699643_dp,  299.80262448398395_dp,  421.01528178406727_dp,  597.918699882043_dp,  865.992956895906_dp,  1303.7011000121163_dp,  2141.5524242895876_dp,&
            4531.217316508373_dp,  0.27326920724684195_dp,  0.2946860969732406_dp,  0.33955686148055997_dp,  0.4121109865279064_dp,  0.5191386834268464_dp,  0.6706479068869163_dp,&
            0.8808905660301858_dp,  1.1698656397567786_dp,  1.5654653694813896_dp,  2.106516824061551_dp,  2.847090389745759_dp,  3.8626161289484946_dp,  5.258596350357734_dp,&
            7.183071722294598_dp,  9.844561359899942_dp,  13.538086075854713_dp,  18.68336222220931_dp,  25.881909474247266_dp,  36.00509685683712_dp,  50.336922172287224_dp,&
            70.82460332971439_dp,  100.57024668126095_dp,  144.9365352509728_dp,  214.43007706895824_dp,  333.52874619054023_dp,  573.9031161499247_dp,  1229.5973040920514_dp,  4703.444164101694_dp]
       aw%aw_erange_matrix(:, 2) = [0.13875490228933057_dp,  0.4276255171652803_dp,  0.7517156037290086_dp,  1.1383835985980468_dp,  1.6211543280535077_dp,  2.24298954813131_dp,&
            3.0605574074054775_dp,  4.150010628234262_dp,  5.614955400623468_dp,  7.597559757812928_dp,  10.294150717899024_dp,  13.977239772443514_dp,  19.02678365968016_dp,&
            25.97476748389089_dp,  35.56910288047172_dp,  48.86570866761254_dp,  67.36207051671413_dp,  93.19263977809487_dp,  129.41834545397384_dp,  180.464373940652_dp,&
            252.8054276668391_dp,  356.10337604440286_dp,  505.2839065614399_dp,  724.884011732378_dp,  1059.8824174872211_dp,  1608.9472923295727_dp,  2660.4222496990938_dp,&
            5652.953965854954_dp,  0.27938451496965105_dp,  0.3022942729129253_dp,  0.3504019114933159_dp,  0.42846797473594944_dp,  0.5441638095088057_dp,  0.7088566053659555_dp,&
            0.9388406258537662_dp,  1.2571515140355833_dp,  1.696177262039284_dp,  2.3013920551824687_dp,  3.1366969256786756_dp,  4.292078154026608_dp,  5.894629362616917_dp,&
            8.124486230083422_dp,  11.237991762535799_dp,  15.601617466229358_dp,  21.74214403705774_dp,  30.42206941171729_dp,  42.75588434748671_dp,  60.397316412895705_dp,&
            85.86308174160227_dp,  123.15573491240343_dp,  179.14126278900252_dp,  267.1160865820656_dp,  417.7474719630637_dp,  720.4456660885186_dp,  1542.1360738637661_dp,  5884.200372654236_dp]
       aw%aw_erange_matrix(:, 3) = [0.14170133198228974_dp,  0.4372217263238123_dp,  0.7703667410153948_dp,  1.170561871979771_dp,  1.6741573415215183_dp,  2.328150315477383_dp,&
            3.1950959957750866_dp,  4.359823742790623_dp,  5.938793138824031_dp,  8.093266245009781_dp,  11.047988350138315_dp,  15.117836905873796_dp,  20.74603727817547_dp,&
            28.559241927240333_dp,  39.447320753058946_dp,  54.67958659957536_dp,  76.07517146731833_dp,  106.25479226978082_dp,  149.01700315289193_dp,  209.91049251369597_dp,&
            297.1309477297457_dp,  423.00157872210633_dp,  606.6410313436483_dp,  879.4603870251076_dp,  1298.7028908996313_dp,  1988.89250689582_dp,  3311.7439315777756_dp,&
            7068.809688414042_dp,  0.28540132117732764_dp,  0.30984707978323456_dp,  0.36129794839002666_dp,  0.44508976371473236_dp,  0.5698536211227944_dp,  0.7484481979075869_dp,&
            0.9994281881329805_dp,  1.349221631283907_dp,  1.8352885387908624_dp,  2.5106794264046326_dp,  3.4506197143865025_dp,  4.7620484853517535_dp,  6.597491611853674_dp,&
            9.175333108649932_dp,  12.809599627131021_dp,  17.954022803045103_dp,  25.267822915246338_dp,  35.715254750483545_dp,  50.719550256407835_dp,  72.40992923579321_dp,&
            104.04357266683634_dp,  150.80209138086602_dp,  221.51983619636096_dp,  333.11222703164094_dp,  524.1575321727823_dp,  906.5045080905089_dp,  1939.0883159241562_dp,&
            7379.912745615771_dp]
       aw%aw_erange_matrix(:, 4) = [0.1443213439813841_dp,  0.4457831750819166_dp,  0.7871040955754438_dp,  1.199649305615617_dp,  1.7224480814853638_dp,  2.406363780432981_dp,&
            3.3196484980978123_dp,  4.555610260714856_dp,  6.243390548485977_dp,  8.563273777666188_dp,  11.768595390094813_dp,  16.217282570542377_dp,  22.41751110034019_dp,&
            31.094143991937496_dp,  43.28592662828489_dp,  60.4884812383459_dp,  84.86601967785074_dp,  119.56722092591784_dp,  169.20143472321413_dp,  240.5679228390866_dp,&
            343.80218836354004_dp,  494.2629646656425_dp,  715.8986735090695_dp,  1048.0827251136477_dp,  1562.2463425593953_dp,  2412.584383401865_dp,  4044.261841910583_dp,&
            8670.083651639381_dp,  0.29075611057246903_dp,  0.3166261585220409_dp,  0.3711885602570686_dp,  0.46033854458071854_dp,  0.5936453499685163_dp,  0.7854359258485714_dp,&
            1.0565085213633747_dp,  1.436688222670177_dp,  1.968561659570244_dp,  2.712912087952586_dp,  3.7566401596502166_dp,  5.224345933312585_dp,  7.295336462823744_dp,&
            10.228723138518374_dp,  14.400665352704436_dp,  20.36000930380217_dp,  28.912119534169648_dp,  41.246715659956386_dp,  59.136507003494074_dp,  85.2557783449816_dp,&
            123.72029659330352_dp,  181.09033026269378_dp,  268.50693202646386_dp,  407.097383174319_dp,  644.5187891323013_dp,  1118.0760747835973_dp,  2390.794104288915_dp,  9077.664025334678_dp]
       aw%aw_erange_matrix(:, 5) = [0.14638817647917096_dp,  0.4525562304919969_dp,  0.8004113661058508_dp,  1.2229192620590597_dp,  1.7613392882634402_dp,  2.46978131492719_dp,&
            3.4213217718648257_dp,  4.716508123125715_dp,  6.495398953009317_dp,  8.954788755571927_dp,  12.373034589954802_dp,  17.14606176978465_dp,  23.83987749306696_dp,&
            33.26757985490672_dp,  46.60291342057746_dp,  65.54869201567364_dp,  92.58820799313293_dp,  131.36337282507247_dp,  187.24910181582314_dp,  268.2384452991122_dp,&
            386.33916361080026_dp,  559.8711155854782_dp,  817.5377393277938_dp,  1206.597963677244_dp,  1812.5358477762866_dp,  2818.7486911520996_dp,  4751.872883766852_dp,&
            10224.75043407236_dp,  0.2949833929923809_dp,  0.3220167305947149_dp,  0.37912854664712514_dp,  0.4726893344740349_dp,  0.6130690229748341_dp,  0.815855353412895_dp,&
            1.103786627022251_dp,  1.5096476978683913_dp,  2.0805275568641997_dp,  2.8840571372482406_dp,  4.0175653372807405_dp,  5.6215679248366985_dp,  7.899729561825618_dp,&
            11.148556975090484_dp,  15.801822722811611_dp,  22.49747280636011_dp,  32.179229771450665_dp,  46.2526111584764_dp,  66.82858029050779_dp,  97.11489993975235_dp,&
            142.07650817080304_dp,  209.6486242985047_dp,  313.28112229681227_dp,  478.2993974050695_dp,  761.3056079394062_dp,  1324.411398249371_dp,  2831.738972704671_dp,  10731.431828374814_dp]
       aw%aw_erange_matrix(:, 6) = [0.14873276811274175_dp,  0.4602604435552678_dp,  0.8156202251610892_dp,  1.2496712569060364_dp,  1.8063333929867138_dp,  2.5436216992100222_dp,&
            3.5404623300065525_dp,  4.906249870587573_dp,  6.794488603605298_dp,  9.422467064090277_dp,  13.09985016154952_dp,  18.270496883841385_dp,  25.573981534747755_dp,&
            35.93662127761815_dp,  50.706943480264954_dp,  71.85851387250909_dp,  102.29562759788192_dp,  146.31752861965646_dp,  210.33006795552657_dp,  303.95126003728876_dp,&
            441.76539937885485_dp,  646.211368042272_dp,  952.6710416909058_dp,  1419.5536773254755_dp,  2152.239930114691_dp,  3375.245128381901_dp,  5728.981212116223_dp,&
            12382.677743050794_dp,  0.29978212482122407_dp,  0.3281783565393207_dp,  0.38828614617008406_dp,  0.4870536747556686_dp,  0.6358284060736311_dp,  0.8517463850003115_dp,&
            1.159944468384693_dp,  1.5968934307453053_dp,  2.2153322272555918_dp,  3.0915535322808045_dp,  4.336186457224601_dp,  6.110220047392288_dp,  8.648930492647603_dp,&
            12.297804765083454_dp,  17.566800104918254_dp,  25.212855803295792_dp,  36.36638248551193_dp,  52.727201300860486_dp,  76.87269143077049_dp,  112.75446596355195_dp,&
            166.53426131891115_dp,  248.1027496502268_dp,  374.209851157723_dp,  576.1670830958_dp,  923.1971063861915_dp,  1612.0060664213056_dp,  3447.1144071775916_dp,  13034.665823606138_dp]
       aw%aw_erange_matrix(:, 7) = [0.1502608626762777_dp,  0.46529381929136415_dp,  0.8255984287192656_dp,  1.2673137696821588_dp,  1.836171369209371_dp,  2.5928648745864002_dp,&
            3.620360959868694_dp,  5.03420711947541_dp,  6.997322718003583_dp,  9.741445894989209_dp,  13.598470784165613_dp,  19.046535582715773_dp,  26.77821561373657_dp,&
            37.802020863101404_dp,  53.59437014367614_dp,  76.32857147441665_dp,  109.22216155171537_dp,  157.0678149109126_dp,  227.05242355977109_dp,  330.03689277986626_dp,&
            482.59561279417204_dp,  710.3800419453736_dp,  1054.0275615855337_dp,  1580.7806999365232_dp,  2411.814227215001_dp,  3804.1445503572704_dp,  6487.450697686814_dp,&
            14065.757793553257_dp,  0.30291165139633314_dp,  0.33222125521586104_dp,  0.39434230166254536_dp,  0.49662265274183637_dp,  0.6510887543947779_dp,  0.8759576781503452_dp,&
            1.1980509194264035_dp,  1.65644466400815_dp,  2.3078985102205145_dp,  3.2349123916804046_dp,  4.557716389303409_dp,  6.452190401185147_dp,  9.17678285087093_dp,&
            13.11317617239606_dp,  18.828100900733247_dp,  27.16794809144496_dp,  39.40475025321865_dp,  57.46370759694749_dp,  84.28289069496383_dp,  124.394912066618_dp,&
            184.90546198899864_dp,  277.2608566102862_dp,  420.8506637137679_dp,  651.7708792506397_dp,  1049.2452864243335_dp,  1837.0994387864066_dp,  3929.4241467171732_dp,  14836.654624922767_dp]
       aw%aw_erange_matrix(:, 8) = [0.15216067096721872_dp,  0.471565223326128_dp,  0.8380780093696487_dp,  1.28948169785262_dp,  1.8738495345723247_dp,  2.6553598559323834_dp,&
            3.7222688062985685_dp,  5.19822827508094_dp,  7.258634966635491_dp,  10.154495182186366_dp,  14.247532860539385_dp,  20.062185372368273_dp,  28.363094298620563_dp,&
            40.27130480928199_dp,  57.43960199384115_dp,  82.31880941446532_dp,  118.5650689039139_dp,  171.66758912191602_dp,  249.9251651508212_dp,  365.9837744748739_dp,&
            539.3021607201113_dp,  800.2307440291746_dp,  1197.160479701725_dp,  1810.4543604483329_dp,  2784.8104361839814_dp,  4425.491363293105_dp,  7593.770462224588_dp,&
            16532.026633916616_dp,  0.30680462287335947_dp,  0.33727803632012604_dp,  0.40197064053453857_dp,  0.5087540933684152_dp,  0.6705478410902993_dp,  0.9069974242331109_dp,&
            1.247162385748679_dp,  1.7336002070451948_dp,  2.428475465424688_dp,  3.422685558563218_dp,  4.849534981128361_dp,  6.905320699402122_dp,  9.88048894377267_dp,&
            14.207072801318644_dp,  20.53138622494952_dp,  29.82621577753499_dp,  43.56537666330866_dp,  63.997988617140564_dp,  94.58522278172896_dp,  140.7101667657315_dp,&
            210.87312430131763_dp,  318.83821988553984_dp,  487.951265473417_dp,  761.4827237292491_dp,  1233.5478315642981_dp,  2167.935258865102_dp,  4639.411026254049_dp,  17485.041970829057_dp]
       aw%aw_erange_matrix(:, 9) = [0.1540799216139811_dp,  0.47791627634651673_dp,  0.850769563354445_dp,  1.3121430573119903_dp,  1.9125794571765222_dp,  2.71995838496797_dp,&
            3.8281934943752236_dp,  5.369664505300357_dp,  7.533297508669478_dp,  10.591136793513773_dp,  14.937707413108372_dp,  21.14872971257977_dp,  30.069271553254573_dp,&
            42.94695048821831_dp,  61.63452932012581_dp,  88.90016193695817_dp,  128.90594661034356_dp,  187.95199381913332_dp,  275.6440545272999_dp,  406.74745153174376_dp,&
            604.1811788842098_dp,  903.9925961038582_dp,  1364.0684178021952_dp,  2080.9739321821835_dp,  3228.5707596988204_dp,  5171.757203514712_dp,  8933.185118640264_dp,&
            19534.126420802_dp,  0.31073990046537614_dp,  0.3424211000331794_dp,  0.40978979014005723_dp,  0.5212783171859456_dp,  0.6907658913737462_dp,  0.9394414583345747_dp,&
            1.2987972772142504_dp,  1.8151994912799103_dp,  2.556766327946643_dp,  3.6237110927440117_dp,  5.163950286772281_dp,  7.396774913472435_dp,  10.648956311521417_dp,&
            15.410163693505844_dp,  22.418599460431437_dp,  32.7943079459921_dp,  48.24841432234958_dp,  71.4147546141822_dp,  106.38218646397881_dp,  159.56520008768115_dp,&
            241.17359283608667_dp,  367.8412274406162_dp,  567.849453755969_dp,  893.4358642502303_dp,  1457.1994566202313_dp,  2571.939694554795_dp,  5508.2172091289185_dp,  20720.162172438253_dp]
       aw%aw_erange_matrix(:, 10) = [0.15656226451877622_dp,  0.4861541997731515_dp,  0.8673131850431656_dp,  1.3418609364967635_dp,  1.9636960012374267_dp,  2.805769901664884_dp,&
            3.969811694463799_dp,  5.600354203716589_dp,  7.905313246191792_dp,  11.186501541338686_dp,  15.88524322405582_dp,  22.65105407014549_dp,  32.44576099810582_dp,&
            46.702425471088645_dp,  67.5696101585034_dp,  98.28954732885543_dp,  143.78811859550652_dp,  211.6029698300622_dp,  313.35735776243826_dp,  467.1276302319483_dp,&
            701.3067172895998_dp,  1061.0705297754337_dp,  1619.7132862760398_dp,  2500.373676035913_dp,  3925.0320573338013_dp,  6356.748668419306_dp,  11081.327581159301_dp,&
            24381.679371671577_dp,  0.3158335009912212_dp,  0.34912565992572775_dp,  0.4200753104160631_dp,  0.5378896525589358_dp,  0.7177805773200047_dp,  0.9830934888999047_dp,&
            1.368743715763372_dp,  1.9264972868488104_dp,  2.732981028075031_dp,  3.9018333932401776_dp,  5.602211794669996_dp,  8.087135781098235_dp,  11.737161876852092_dp,&
            17.128136373382_dp,  25.137060567441367_dp,  37.10879805120811_dp,  55.12084225228906_dp,  82.40796079967193_dp,  124.05157534923438_dp,  188.11812693397164_dp,&
            287.5913676589109_dp,  443.82063547247185_dp,  693.2803515796006_dp,  1103.1543539828035_dp,  1816.6489638049509_dp,  3226.5521973584873_dp,  6920.118346610322_dp,  25967.01374573565_dp]
       aw%aw_erange_matrix(:, 11) = [0.15881564616129193_dp,  0.4936556598653273_dp,  0.8824588841815081_dp,  1.3692457113825114_dp,  2.0111264485492355_dp,  2.885950423092618_dp,&
            4.103059689428933_dp,  5.818927316225091_dp,  8.260287362001002_dp,  11.75871117291467_dp,  16.802727948824526_dp,  24.116981065069147_dp,  34.78330983777523_dp,&
            50.42729724265466_dp,  73.50780317707246_dp,  107.76979904893427_dp,  158.95837591286625_dp,  235.95442692125644_dp,  352.598636504137_dp,  530.6546935517891_dp,&
            804.6962971730163_dp,  1230.3548358289224_dp,  1898.8241278088317_dp,  2964.5082086762377_dp,  4706.457126412182_dp,  7704.024872038893_dp,  13551.675236040213_dp,&
            30000.20611112028_dp,  0.3204610031116597_dp,  0.3552640894565443_dp,  0.4295842351097504_dp,  0.5533833088806153_dp,  0.7431779324839743_dp,  1.0244390610175549_dp,&
            1.4354819295430314_dp,  2.0334801056157183_dp,  2.903655037921023_dp,  4.1733290625525665_dp,  6.0335143863200065_dp,  8.772275269918612_dp,  12.82662526184919_dp,&
            18.863835903341464_dp,  27.909803652722584_dp,  41.55338127934832_dp,  62.27454140110857_dp,  93.97671287155796_dp,  142.86069930469114_dp,  218.88238290006603_dp,&
            338.2447047196054_dp,  527.8483016510384_dp,  833.9328380491752_dp,  1341.6124089588006_dp,  2230.6304097502575_dp,  3987.780431784136_dp,  8568.294988757902_dp,&
            32079.26783423429_dp]
       aw%aw_erange_matrix(:, 12) = [0.1610939781339949_dp,  0.5012631407780889_dp,  0.8978984268667602_dp,  1.3973374389835533_dp,  2.0601052994909983_dp,  2.9693037686025625_dp,&
            4.2425076374898465_dp,  6.049209333425932_dp,  8.63683304888055_dp,  12.369947100756928_dp,  17.78988041970464_dp,  25.70606845547581_dp,  37.33707453593626_dp,&
            54.529958798718575_dp,  80.10419787720568_dp,  118.39528522114581_dp,  176.12137581924918_dp,  263.7777454271281_dp,  397.90331695679896_dp,  604.8077130867456_dp,&
            926.7903239830885_dp,  1432.7407682152939_dp,  2236.883538210146_dp,  3534.396483826615_dp,  5679.4632854654_dp,  9404.641424310748_dp,  16707.201153452228_dp,&
            37236.60159840964_dp,  0.3251433866999734_dp,  0.3615219000652313_dp,  0.43936851892508155_dp,  0.5694605971527937_dp,  0.7697318471818748_dp,  1.0679770987543238_dp,&
            1.5062564601488997_dp,  2.147747329286371_dp,  3.0872938959644456_dp,  4.467674438671821_dp,  6.504813446418115_dp,  9.527106074077976_dp,  14.03718906734335_dp,&
            20.809713515195543_dp,  31.04736871222941_dp,  46.63203851421735_dp,  70.53290909444102_dp,  107.47651145953795_dp,  165.06008557957134_dp,  255.62996703107464_dp,&
            399.52200422464244_dp,  630.8696403435848_dp,  1008.8071793310638_dp,  1642.3274927796938_dp,  2759.7196504068343_dp,  4970.842422998521_dp,  10706.437123051388_dp,  39993.678157447954_dp]
       aw%aw_erange_matrix(:, 13) = [0.16286202517738316_dp,  0.5071829401058993_dp,  0.9099691167949224_dp,  1.4194239255890508_dp,  2.0988437736405245_dp,  3.0356259048953707_dp,&
            4.35412840156482_dp,  6.2346505351972965_dp,  8.941919073444248_dp,  12.868307590765278_dp,  18.599987337738977_dp,  27.01899385998513_dp,  39.46194816028856_dp,&
            57.968842841688435_dp,  85.67620223471917_dp,  127.4437361103542_dp,  190.86221942438462_dp,  287.8902318415413_dp,  437.5401948107147_dp,  670.3390916786248_dp,&
            1035.844707784025_dp,  1615.5694884106663_dp,  2545.9657775567157_dp,  4062.078331546931_dp,  6592.278861303402_dp,  11020.719723335082_dp,  19740.079474062593_dp,&
            44247.269653501586_dp,  0.32877960783872057_dp,  0.36641441495071897_dp,  0.44708193429055737_dp,  0.5822306885078613_dp,  0.7909661025753427_dp,  1.1030159511231001_dp,&
            1.5635761654481883_dp,  2.2408877406025356_dp,  3.237972207764307_dp,  4.710846444376257_dp,  6.896950066089331_dp,  10.159810780070979_dp,  15.059747172258659_dp,&
            22.466688314211762_dp,  33.741746573072334_dp,  51.032123644139766_dp,  77.75480860637037_dp,  119.39847679866706_dp,  184.86920459563467_dp,  288.78280596198823_dp,&
            455.4522326064931_dp,  726.0680539550964_dp,  1172.5112874975482_dp,  1927.6011121765132_dp,  3268.0690080523195_dp,  5925.0502577809875_dp,  12791.74491637296_dp,  47700.6553314684_dp]
       aw%aw_erange_matrix(:, 14) = [0.16550588765942553_dp,  0.516062009336416_dp,  0.9281674926129342_dp,  1.4529297711024542_dp,  2.1579962800811443_dp,  3.1375655705474386_dp,&
            4.526824919336864_dp,  6.52346781619548_dp,  9.420301579076966_dp,  13.655208883851635_dp,  19.888399623526567_dp,  29.122872846718497_dp,  42.89382564991708_dp,&
            63.56900112741913_dp,  94.82915018373326_dp,  142.44372123087916_dp,  215.53542531034506_dp,  328.6632843331655_dp,  505.2926101128234_dp,  783.6477805337622_dp,&
            1226.7296666006187_dp,  1939.7944616575992_dp,  3101.785623951648_dp,  5025.174302162332_dp,  8284.281861937603_dp,  14062.83179723175_dp,  25528.48971632664_dp,&
            57759.22356371914_dp,  0.3342213065985371_dp,  0.3737907231275688_dp,  0.45881732537071723_dp,  0.601819261468133_dp,  0.8237789857593867_dp,  1.1575413887012014_dp,&
            1.6533971399634884_dp,  2.3878799654780147_dp,  3.4775169315434225_dp,  5.100383812329383_dp,  7.530103230545625_dp,  11.189861262594912_dp,  16.738922378384462_dp,&
            25.212398544065515_dp,  38.249144800530594_dp,  58.46701561201135_dp,  90.0873039981285_dp,  139.9857380571293_dp,  219.48397400182492_dp,  347.4492001516725_dp,&
            555.763013631551_dp,  899.2679993238643_dp,  1474.9019487688392_dp,  2462.9199651441854_dp,  4236.78509184323_dp,  7766.730019457065_dp,  16842.450828736164_dp,  62648.76273159001_dp]
       aw%aw_erange_matrix(:, 15) = [0.1678624984386396_dp,  0.5240039277065677_dp,  0.9445413343768262_dp,  1.4832899296027031_dp,  2.2119941166231323_dp,  3.231318139361639_dp,&
            4.686840531125701_dp,  6.793101100181927_dp,  9.870359826324044_dp,  14.401421486392543_dp,  21.120311419307942_dp,  31.151884233378865_dp,  46.23355566119178_dp,&
            69.07063891977168_dp,  103.9111329114676_dp,  157.48467618090592_dp,  240.55243987990312_dp,  370.4940225061482_dp,  575.6771054267826_dp,  902.9360411163456_dp,&
            1430.5655099944815_dp,  2291.326618763138_dp,  3714.3291401860997_dp,  6105.216711407023_dp,  10216.855593912702_dp,  17602.252858531945_dp,  32376.971831996485_dp,&
            73939.47243943921_dp,  0.3390761234013535_dp,  0.38042748095334855_dp,  0.46948517950905133_dp,  0.6197911244215307_dp,  0.8541350450818376_dp,  1.2083859869249218_dp,&
            1.737819635649183_dp,  2.5271580680576275_dp,  3.7063925373131634_dp,  5.475812712926834_dp,  8.145864684746673_dp,  12.201123189749161_dp,  18.403853838774207_dp,&
            27.96322210175245_dp,  42.81450321088113_dp,  66.08467665891509_dp,  102.87759879730325_dp,  161.61423975462415_dp,  256.35102809823303_dp,  410.8517040007441_dp,&
            665.8762082773686_dp,  1092.5932138672465_dp,  1818.4906655133614_dp,  3082.6261009452624_dp,  5379.192173747507_dp,  9973.306180469173_dp,  21737.4521360814_dp,  80688.08263620315_dp]
       aw%aw_erange_matrix(:, 16) = [0.17036496307430077_dp,  0.5324663182632273_dp,  0.9620895318113227_dp,  1.516053047783045_dp,  2.2706890746569957_dp,  3.333970091290067_dp,&
            4.863327619652776_dp,  7.092693187729123_dp,  10.374220724548097_dp,  15.243403064520715_dp,  22.521699980194235_dp,  33.47979649400302_dp,  50.09973367935207_dp,&
            75.49983243992041_dp,  114.63042911159955_dp,  175.42509396717406_dp,  270.7269739485002_dp,  421.55091333978646_dp,  662.6791216153256_dp,  1052.3953320146977_dp,&
            1689.6875070139436_dp,  2745.231543799062_dp,  4518.674628346486_dp,  7549.367766712861_dp,  12851.219414149484_dp,  22522.974006977485_dp,  42073.064211266406_dp,&
            97154.3497922795_dp,  0.3442359726520383_dp,  0.38754007185862954_dp,  0.4810326806451499_dp,  0.6394198647471336_dp,  0.8875587091716534_dp,  1.2648037235106746_dp,&
            1.8322244813392865_dp,  2.684145960116962_dp,  3.966500195829279_dp,  5.906140569869111_dp,  8.858010554137483_dp,  13.381685635639329_dp,  20.36673130214467_dp,&
            31.24005706702999_dp,  48.31255482932005_dp,  75.36517063974274_dp,  118.6518389635584_dp,  188.6382478887288_dp,  303.059337317903_dp,  492.38182357709826_dp,&
            809.7503916539654_dp,  1349.5688287135445_dp,  2283.7129470839764_dp,  3938.318929170714_dp,  6988.303443289154_dp,  13136.528535241805_dp,  28826.90535778617_dp,  106792.5668680263_dp]
       aw%aw_erange_matrix(:, 17) = [0.17269556592412857_dp,  0.5403748964029321_dp,  0.9785852679690707_dp,  1.5470653817948434_dp,  2.3266514376923366_dp,  3.4325577583865416_dp,&
            5.0340686756366715_dp,  7.384684578834457_dp,  10.86904453036137_dp,  16.076821539981736_dp,  23.92028394592648_dp,  35.82315617144631_dp,  54.02701058174775_dp,&
            82.09339856008381_dp,  125.73557580722034_dp,  194.2117754907147_dp,  302.6873267691643_dp,  476.2910454873438_dp,  757.1780279061045_dp,  1217.01315551735_dp,&
            1979.4070403165824_dp,  3261.0324543098973_dp,  5448.912148191038_dp,  9251.68321181564_dp,  16020.743424436212_dp,  28570.382753480066_dp,  54229.78154218437_dp,&
            126696.56228762324_dp,  0.3490457636631589_dp,  0.39422579301772903_dp,  0.49199588231797214_dp,  0.6582220201025798_dp,  0.9198335281974843_dp,  1.3197051837498925_dp,&
            1.9248079614290126_dp,  2.839336845230337_dp,  4.225765459102832_dp,  6.338790677137095_dp,  9.580492391710184_dp,  14.590781522893115_dp,  22.39718076042754_dp,&
            34.665500315997214_dp,  54.12415292925981_dp,  85.29113057050759_dp,  135.73577535381042_dp,  218.29950108939332_dp,  355.0638815393454_dp,  584.5607922611614_dp,&
            975.1302787959035_dp,  1650.2897030531005_dp,  2838.7772270165033_dp,  4980.693094157311_dp,  8991.001468037319_dp,  17151.64888313862_dp,  37936.68288424044_dp,  140330.3757792011_dp]
       aw%aw_erange_matrix(:, 18) = [0.17476493191141415_dp,  0.5474193940515534_dp,  0.9933573948794411_dp,  1.5750134784958338_dp,  2.3774182245590976_dp,  3.5225875266569684_dp,&
            5.191031053231731_dp,  7.65493753438325_dp,  11.330235095244555_dp,  16.85924390942949_dp,  25.24328503055809_dp,  38.05761248430254_dp,  57.80337969002844_dp,&
            88.49020666120597_dp,  136.61135242458343_dp,  212.79570615428042_dp,  334.6422749519929_dp,  531.6505866599738_dp,  853.9241940929868_dp,  1387.782907449122_dp,&
            2284.2677480183024_dp,  3812.2510844331973_dp,  6459.925409369088_dp,  11136.186670622694_dp,  19600.27283250913_dp,  35545.19130000765_dp,  68536.45729392112_dp,&
            161999.07322111004_dp,  0.35331994251863175_dp,  0.4002125878121666_dp,  0.5019021590207834_dp,  0.675348992397178_dp,  0.9494484180546069_dp,  1.370438471536318_dp,&
            2.010972421797265_dp,  2.984828447386733_dp,  4.47068346087602_dp,  6.7507629145842705_dp,  10.274207336138101_dp,  15.761962819310062_dp,  24.382228769987734_dp,&
            38.04723731915354_dp,  59.92133877283344_dp,  95.30210365832416_dp,  153.1695274726372_dp,  248.9507054460044_dp,  409.5337510421667_dp,  682.5234878783859_dp,&
            1153.6752246200758_dp,  1980.5529864545388_dp,  3459.8436855153436_dp,  6170.831685857496_dp,  11326.739601304256_dp,  21929.575378095506_dp,  48923.087093047834_dp,  180801.94440922138_dp]
       aw%aw_erange_matrix(:, 19) = [0.17707349703529857_dp,  0.5553034658173043_dp,  1.0099791730991756_dp,  1.6066610726001294_dp,  2.4352859938075695_dp,  3.6258937322871367_dp,&
            5.372347774805723_dp,  7.969256176272777_dp,  11.870406472074928_dp,  17.78239069764118_dp,  26.81626145155005_dp,  40.73582235231918_dp,  62.36857184593184_dp,&
            96.29361499410518_dp,  150.00711005408417_dp,  235.9221311383991_dp,  374.8476146182405_dp,  602.1299577709411_dp,  978.6687811138307_dp,  1611.0196047848676_dp,&
            2688.7838466688286_dp,  4555.662141059458_dp,  7848.009593576401_dp,  13774.881010275332_dp,  24721.57253813353_dp,  45756.96334453548_dp,  89961.78780041506_dp,&
            215802.3342656405_dp,  0.3580921418111161_dp,  0.40694849130748023_dp,  0.5131489467923581_dp,  0.6949501550098907_dp,  0.9835895138470994_dp,  1.429340141748686_dp,&
            2.1117265083379655_dp,  3.156213050221744_dp,  4.761412846546327_dp,  7.243742288102652_dp,  11.111375537133872_dp,  17.187982478782033_dp,  26.82207794053154_dp,&
            42.2454800002888_dp,  67.19501324394297_dp,  108.0057380925622_dp,  175.56201559076507_dp,  288.83560003555107_dp,  481.41389562366794_dp,  813.7768209755345_dp,&
            1396.8818687993016_dp,  2438.628712875254_dp,  4338.525137576797_dp,  7891.642771451579_dp,  14783.538737749648_dp,  29163.479733223845_dp,  65827.4645722146_dp,  243179.26518144744_dp]
       aw%aw_erange_matrix(:, 20) = [0.17939121623291196_dp,  0.5632460962554191_dp,  1.0268207360191903_dp,  1.6389441207178375_dp,  2.494730975718965_dp,  3.732765899075506_dp,&
            5.561259778907539_dp,  8.299124280524993_dp,  12.441566280835362_dp,  18.766165667383223_dp,  28.506386171958923_dp,  43.638584384879934_dp,  67.36224738878316_dp,&
            104.91328976480246_dp,  164.9591150046168_dp,  262.0244972943479_dp,  420.7723136732652_dp,  683.6778754860417_dp,  1125.0251624529246_dp,  1876.914526701929_dp,&
            3178.594976788772_dp,  5472.221732744173_dp,  9593.809960559516_dp,  17167.666568195575_dp,  31469.64998425534_dp,  59575.37273864424_dp,  119738.62770948543_dp,&
            292181.0961057968_dp,  0.36288753563601406_dp,  0.4137727803438694_dp,  0.5246525197888982_dp,  0.7151692528977265_dp,  1.0190794875976732_dp,  1.4910294440355354_dp,&
            2.2180532993489983_dp,  3.3385033549868535_dp,  5.073189714925689_dp,  7.776982900408857_dp,  12.025161831532559_dp,  18.759485999109437_dp,  29.538246648034107_dp,&
            46.969812152842266_dp,  75.47470761821671_dp,  122.64510922855702_dp,  201.70899555206492_dp,  336.0740868831789_dp,  567.8651913799991_dp,  974.2972338202875_dp,&
            1699.8017696089582_dp,  3020.746535916867_dp,  5480.209144976741_dp,  10183.145007147052_dp,  19511.673781213012_dp,  39328.72129742758_dp,  90071.79118231079_dp,  332938.41349157074_dp]
       aw%aw_erange_matrix(:, 21) = [0.18145787532761345_dp,  0.5703516980411364_dp,  1.041969987100468_dp,  1.668169762478659_dp,  2.548905102202213_dp,  3.8308153868700057_dp,&
            5.73575077606394_dp,  8.605924764080692_dp,  12.976605854940628_dp,  19.694664135422833_dp,  30.114196205027547_dp,  46.4231488999098_dp,  72.19529201520032_dp,&
            113.33495114684222_dp,  179.71600233232834_dp,  288.0668684898181_dp,  467.1287314532969_dp,  767.0353863539326_dp,  1276.6863337842099_dp,  2156.5782398644915_dp,&
            3702.2394624534754_dp,  6469.865405622618_dp,  11532.348640847953_dp,  21019.959826658564_dp,  39325.54785388118_dp,  76112.75783759716_dp,  156400.96081191118_dp,&
            388429.5049686922_dp,  0.3671671269112481_dp,  0.41991063535051504_dp,  0.5350925824270352_dp,  0.7336661302058223_dp,  1.051784230902112_dp,  1.5482836110025917_dp,&
            2.31745277890254_dp,  3.510201538782171_dp,  5.369167048208645_dp,  8.287401135959756_dp,  12.907497398486063_dp,  20.29096603628255_dp,  32.211289400182395_dp,&
            51.667875628946675_dp,  83.80064093288466_dp,  137.54326768587123_dp,  228.6628977463769_dp,  385.45278105561033_dp,  659.609584732442_dp,  1147.48083087963_dp,&
            2032.5976860626631_dp,  3673.224580494971_dp,  6788.760553221407_dp,  12875.900072438904_dp,  25223.35236302233_dp,  51966.7592745666_dp,  120920.32428947231_dp,  447736.17541344964_dp]
       aw%aw_erange_matrix(:, 22) = [0.18352365749998711_dp,  0.5774767415391759_dp,  1.0572402465072317_dp,  1.6978091130093156_dp,  2.604194412635273_dp,  3.931521110681753_dp,&
            5.916123772874016_dp,  8.925163169995924_dp,  13.537158542380958_dp,  20.674441611900086_dp,  31.823697567083265_dp,  49.40770243187122_dp,  77.4199205005401_dp,&
            122.5224666728988_dp,  195.97318566547537_dp,  317.0604624119025_dp,  519.3281914726538_dp,  862.0631903466981_dp,  1451.9161062243747_dp,  2484.485489661112_dp,&
            4326.234527544991_dp,  7680.233304404955_dp,  13931.912588144245_dp,  25897.50172535608_dp,  49530.68109870176_dp,  98224.55152386216_dp,  206939.49053745338_dp,&
            524566.9562885154_dp,  0.3714484036559812_dp,  0.4260967455633218_dp,  0.5457049480928435_dp,  0.7526107268819361_dp,  1.085512849361448_dp,  1.6077308109599637_dp,&
            2.421373349769565_dp,  3.690999486571998_dp,  5.683179798418424_dp,  8.83322260839823_dp,  13.858958047223895_dp,  21.957146639909727_dp,  35.147034069885805_dp,&
            56.87996069901934_dp,  93.13791144680825_dp,  154.44652085662315_dp,  259.63116034555344_dp,  442.9649858732788_dp,  768.0687208011793_dp,  1355.5892058006805_dp,&
            2439.7899569103974_dp,  4487.76072780871_dp,  8459.538519483716_dp,  16402.660399711818_dp,  32921.61554906603_dp,  69533.14758216766_dp,  164947.39394785516_dp,  612800.2731138045_dp]
       aw%aw_erange_matrix(:, 23) = [0.1865127174002091_dp,  0.5878264119237888_dp,  1.0795642254870847_dp,  1.7414647079050736_dp,  2.686262513059428_dp,  4.082172829391877_dp,&
            6.188096843488031_dp,  9.410456034264666_dp,  14.396547620814117_dp,  22.190018755083887_dp,  34.49317521293508_dp,  54.11536762065149_dp,  85.74999790790363_dp,&
            137.3406746169067_dp,  222.5209261857046_dp,  365.04500319056837_dp,  606.9830522872659_dp,  1024.1840598017022_dp,  1756.0972227796592_dp,  3064.6905800265477_dp,&
            5454.026428632921_dp,  9920.369530831516_dp,  18493.617558448037_dp,  35457.83700573299_dp,  70250.70208206157_dp,  144978.57163654402_dp,  318662.0049240475_dp,&
            837598.2931640423_dp,  0.3776493979310916_dp,  0.435138768215793_dp,  0.5613788375562351_dp,  0.7808493636204112_dp,  1.1362155303193089_dp,  1.6978417615255126_dp,&
            2.5802450840754427_dp,  3.9698657023594675_dp,  6.172064932079413_dp,  9.69144120022167_dp,  15.370728380706238_dp,  24.634198247900454_dp,  39.92036731837176_dp,&
            65.46330289853823_dp,  108.72733676148118_dp,  183.09024363050807_dp,  312.9622216519181_dp,  543.7670826405075_dp,  961.8761857704002_dp,  1735.4901623013577_dp,&
            3201.003196409115_dp,  6051.665762920312_dp,  11766.014824513828_dp,  23628.41759091035_dp,  49336.106221828624_dp,  108694.3328979675_dp,  267237.7291173849_dp,  1002021.1700926761_dp]
       aw%aw_erange_matrix(:, 24) = [0.18910898992309827_dp,  0.5968553778314455_dp,  1.099179600396227_dp,  1.7801437973508598_dp,  2.7596026949493577_dp,  4.217972884397996_dp,&
            6.435421861591648_dp,  9.855784799002635_dp,  15.192663835695727_dp,  23.608087390187727_dp,  37.01748316805854_dp,  58.61754075464745_dp,  93.81326580419346_dp,&
            151.87161660723567_dp,  248.92100011282085_dp,  413.49172640188107_dp,  696.9547246299024_dp,  1193.621061792798_dp,  2080.3822432854854_dp,  3696.998493291677_dp,&
            6713.636172345608_dp,  12492.44299238403_dp,  23898.55659235671_dp,  47203.17808379973_dp,  96805.96561293019_dp,  207957.8998427237_dp,  477933.1522768045_dp,&
            1308114.0012041177_dp,  0.3830416352614052_dp,  0.4430819114148896_dp,  0.5753071318246739_dp,  0.8061988913511784_dp,  1.182158486497417_dp,  1.7802506103346156_dp,&
            2.726920016454761_dp,  4.229882717633601_dp,  6.632680284308672_dp,  10.509000784328796_dp,  16.82784984813855_dp,  27.24691097632485_dp,  44.64167580356611_dp,&
            74.07592303550199_dp,  124.61429974186086_dp,  212.77537227876334_dp,  369.25379542370973_dp,  652.3218043365148_dp,  1175.259654699904_dp,  2164.1710677218566_dp,&
            4083.922692671305_dp,  7922.927093368247_dp,  15865.524351752772_dp,  32963.81111451602_dp,  71590.44785506214_dp,  164831.2803483035_dp,  422312.83970923733_dp,  1607194.846517409_dp]
       aw%aw_erange_matrix(:, 25) = [0.19132297837375925_dp,  0.6045842212401715_dp,  1.116075239909931_dp,  1.8137003319341403_dp,  2.823703852560562_dp,  4.3375571365379555_dp,&
            6.654879506061997_dp,  10.25406160102038_dp,  15.910566848875042_dp,  24.8980512789274_dp,  39.33518360062533_dp,  62.79250184241327_dp,  101.37073372724154_dp,&
            165.64870251297975_dp,  274.26536197966556_dp,  460.63593723457706_dp,  785.8148240324404_dp,  1363.7143426301643_dp,  2411.848280798477_dp,  4356.437870983105_dp,&
            8057.345689393333_dp,  15307.674322294604_dp,  29991.967793708438_dp,  60909.637552873915_dp,  129092.68815909834_dp,  288420.9208707836_dp,  693804.4807934643_dp,&
            1984726.5657356782_dp,  0.38764444414599436_dp,  0.4499221966702515_dp,  0.5874206914731522_dp,  0.8284384818447332_dp,  1.2227914287059967_dp,  1.8537196608025412_dp,&
            2.858764193877508_dp,  4.46563291856233_dp,  7.054128800394659_dp,  11.264308056241072_dp,  18.18795719360849_dp,  29.712689970242995_dp,  49.150504691840844_dp,&
            82.40656726563064_dp,  140.1950401391153_dp,  242.32987003504195_dp,  426.2290660460464_dp,  764.2086384943232_dp,  1399.6697911963972_dp,  2625.280715869226_dp,&
            5058.106540783201_dp,  10048.454253234977_dp,  20681.146673725165_dp,  44371.40116757773_dp,  100098.51304747425_dp,  240936.64519318528_dp,  645993.0255202315_dp,  2511145.760363479_dp]
       aw%aw_erange_matrix(:, 26) = [0.19436594608582472_dp,  0.6152519388154075_dp,  1.1395565637382394_dp,  1.860708304267537_dp,  2.914238339347287_dp,  4.507855666735311_dp,&
            6.970054206984331_dp,  10.831076997871907_dp,  16.960267570555793_dp,  26.802718362766388_dp,  42.793217314755324_dp,  69.09173097076558_dp,  112.91204543512235_dp,&
            186.9652299761567_dp,  314.0415178600245_dp,  535.7867637150474_dp,  929.9119488429988_dp,  1644.8265501191415_dp,  2971.387935288634_dp,  5496.481343756791_dp,&
            10444.205167643195_dp,  20466.948914951925_dp,  41573.49202713281_dp,  88115.73197659155_dp,  196655.41477188517_dp,  468312.0718363519_dp,  1218658.020501648_dp,&
            3790777.501461529_dp,  0.39397762301074635_dp,  0.4594261340249453_dp,  0.6044348272087624_dp,  0.8599749829785104_dp,  1.2809242553853217_dp,  1.959763999342232_dp,&
            3.0508139495699504_dp,  4.812352417460658_dp,  7.680299553389941_dp,  12.398744349822774_dp,  20.2545889010679_dp,  33.50625966016763_dp,  56.1810053198472_dp,&
            95.58665349028226_dp,  165.23942057994483_dp,  290.66767214215037_dp,  521.2161338636482_dp,  954.7499892270533_dp,  1791.0386147663526_dp,  3451.3628492991857_dp,&
            6857.78940921672_dp,  14117.121205871057_dp,  30292.355037472884_dp,  68308.93744588633_dp,  163720.9776403595_dp,  424479.54068993696_dp,  1238540.16298379_dp,&
            5075528.538071685_dp]
       aw%aw_erange_matrix(:, 27) = [0.19682681061058516_dp,  0.6239175967168831_dp,  1.1587697829037367_dp,  1.899493134270955_dp,  2.9895781441779548_dp,  4.650804912675836_dp,&
            7.236967535355107_dp,  11.324261660542758_dp,  17.866223883633104_dp,  28.463640303603217_dp,  45.84226954217273_dp,  74.71248548146892_dp,  123.3437137988759_dp,&
            206.5036090144391_dp,  351.0605506141016_dp,  606.9104628799121_dp,  1068.8355016906876_dp,  1921.4860373237475_dp,  3534.938103259401_dp,  6675.151370185142_dp,&
            12987.084625450672_dp,  26158.425843963676_dp,  54887.04107632314_dp,  120986.04511920984_dp,  283491.0124779256_dp,  718729.5124126623_dp,  2032037.8454332922_dp,&
            6994905.371452202_dp,  0.39910525153739623_dp,  0.46720011780848775_dp,  0.618510057951796_dp,  0.8863247571224414_dp,  1.3299488054374997_dp,  2.050026502879363_dp,&
            3.21586179306573_dp,  5.113357675674665_dp,  8.229788884254283_dp,  13.40574530515295_dp,  22.111763971910367_dp,  36.960758258119995_dp,  62.6753790751042_dp,&
            107.95244981242995_dp,  189.13916327630196_dp,  337.6660556617939_dp,  615.5008714320224_dp,  1148.2967230500535_dp,  2199.040277147706_dp,  4338.370096055334_dp,&
            8857.05088458885_dp,  18820.230015785593_dp,  41940.375294296646_dp,  99039.96110832106_dp,  251532.47511127745_dp,  702946.2892638468_dp,  2257925.6950304266_dp,  10050972.42119658_dp]
       aw%aw_erange_matrix(:, 28) = [0.20028753559878754_dp,  0.636163845466183_dp,  1.1861370761903154_dp,  1.9552398102278608_dp,  3.0988787201672205_dp,  4.860155327990237_dp,&
            7.631666931241998_dp,  12.060973589284364_dp,  19.234093080159262_dp,  31.000249030237566_dp,  50.55650397769404_dp,  83.5192836105384_dp,  139.92647812445165_dp,&
            238.0575763348749_dp,  411.8925953402226_dp,  726.0520810993172_dp,  1306.5848049114_dp,  2406.4753954062408_dp,  4550.1807639341105_dp,  8866.114556558668_dp,&
            17889.50386772233_dp,  37615.62178522411_dp,  83127.72557060457_dp,  195408.6566631889_dp,  497429.1250321897_dp,  1411467.9993363773_dp,  4708143.792268462_dp,&
            20982067.79050034_dp,  0.4063254337661295_dp,  0.47826913702275475_dp,  0.6387969282983723_dp,  0.9247131283500699_dp,  1.4020945643989537_dp,  2.1842105605751816_dp,&
            3.4638219546199718_dp,  5.570645148922087_dp,  9.074555435905518_dp,  14.97371681097282_dp,  25.043435353654875_dp,  42.49539020969764_dp,  73.2495448121482_dp,&
            128.4448601273465_dp,  229.52203941450034_dp,  418.8057941608578_dp,  782.2389088299047_dp,  1499.9808360221182_dp,  2963.6575509358563_dp,  6060.980474110256_dp,&
            12905.219416347047_dp,  28829.611589852855_dp,  68285.64328511298_dp,  174085.66633020202_dp,  488713.1004068534_dp,  1569108.161138_dp,  6194788.779558858_dp,  36612022.104547605_dp]
       aw%aw_erange_matrix(:, 29) = [0.20148359962188625_dp,  0.6404126846191118_dp,  1.1956914499421463_dp,  1.9748404662282764_dp,  3.1375902789765524_dp,  4.934852000261182_dp,&
            7.773570595927496_dp,  12.327952344136781_dp,  19.733992303134418_dp,  31.935665896373646_dp,  52.31190662715458_dp,  86.8332262967865_dp,  146.23818287831133_dp,&
            250.21865465340818_dp,  435.66229015608553_dp,  773.3205522921141_dp,  1402.527891101972_dp,  2605.9768107615196_dp,  4977.008179815615_dp,  9810.671652583082_dp,&
            20066.047813611207_dp,  42883.66712086875_dp,  96680.36874016587_dp,  233103.28859658248_dp,  613776.5842567491_dp,  1827984.087654119_dp,  6594610.0314337425_dp,&
            34505523.772945166_dp,  0.4088233220420713_dp,  0.48213225469004667_dp,  0.6459447763036226_dp,  0.9383525242997238_dp,  1.427930574234908_dp,  2.232645680425932_dp,&
            3.5540692271124623_dp,  5.738543059876525_dp,  9.387632026047086_dp,  15.56066926581158_dp,  26.1527798090333_dp,  44.61430119999274_dp,  77.34960393660401_dp,&
            136.50197172084523_dp,  245.64488988592876_dp,  451.7568092688549_dp,  851.2535497022337_dp,  1648.713481816509_dp,  3295.0758202094717_dp,  6829.19231168343_dp,&
            14772.00579885476_dp,  33634.6146271831_dp,  81574.2303691521_dp,  214402.9977956818_dp,  627349.5338890508_dp,  2140772.611660053_dp,  9364952.860952146_dp,  68700022.28914484_dp]
    case(30)
       aw%energy_range(:) = [2906.0_dp, 3236.0_dp, 3810.0_dp, 4405.0_dp, 5400.0_dp, 6800.0_dp, 8400.0_dp,&
            10000.0_dp, 12000.0_dp, 15000.0_dp, 20000.0_dp, 28000.0_dp, 38000.0_dp, 50000.0_dp, 64000.0_dp, 84000.0_dp,&
            110000.0_dp, 160000.0_dp, 220000.0_dp, 370000.0_dp, 520000.0_dp, 700000.0_dp, 1100000.0_dp, 1800000.0_dp,&
            3300000.0_dp, 6000000.0_dp, 18000000.0_dp, 50000000.0_dp]
       aw%aw_erange_matrix(:, 1) = [0.13472077973006066_dp,  0.41454012407362284_dp,  0.7264650127115554_dp,  1.0952103821233763_dp,  1.5507340744641087_dp,  2.1309684610148_dp,&
            2.8853353291317902_dp,  3.879435757744625_dp,  5.201430031729745_dp,  6.970811684768931_dp,  9.350560848436539_dp,  12.564071001580649_dp,  16.918834424102887_dp,&
            22.83972781830551_dp,  30.915985544342572_dp,  41.967774219335894_dp,  57.14098640727171_dp,  78.0429357516686_dp,  106.93789608418125_dp,  147.0314580515426_dp,&
            202.88978553122783_dp,  281.07180694626186_dp,  391.1193573348081_dp,  547.2093677381681_dp,  771.1975057919235_dp,  1099.0536535908907_dp,  1596.9929628828843_dp,&
            2411.043015679036_dp,  3969.4542024452535_dp,  8410.91429702379_dp,  0.2711551412973589_dp,  0.2920716478256109_dp,  0.33586025075301174_dp,  0.40657866325776965_dp,&
            0.5107331186087983_dp,  0.6578956545676304_dp,  0.8616669035192411_dp,  1.141083396755036_dp,  1.5226214332468109_dp,  2.0430277403045745_dp,  2.7533146911523487_dp,&
            3.7244090770834286_dp,  5.055161408861251_dp,  6.883741530882366_dp,  9.403917834043737_dp,  12.888423735149207_dp,  17.72269370396895_dp,  24.453949230243484_dp,&
            33.86342292727139_dp,  47.07450697595598_dp,  65.71944005664618_dp,  92.20885123968911_dp,  130.2022391295589_dp,  185.52477786139306_dp,  268.21783125503725_dp,&
            397.8749440116057_dp,  619.9912456625073_dp,  1067.5959324975324_dp,  2286.544806677275_dp,  8738.901002766614_dp]
       aw%aw_erange_matrix(:, 2) = [0.13585569926755156_dp,  0.41821529920891193_dp,  0.7335359382994969_dp,  1.1072550786213582_dp,  1.5703001886119063_dp,  2.1619633106722445_dp,&
            2.9336148368913126_dp,  3.9536785393619445_dp,  5.3144254385553475_dp,  7.141352466099918_dp,  9.606222474304257_dp,  12.945295441239312_dp,  17.484938603694847_dp,&
            23.67775987140482_dp,  32.153806670690884_dp,  43.793428768676115_dp,  59.83145780414573_dp,  82.00695648247473_dp,  112.7798752913388_dp,  155.64727131571763_dp,&
            215.6109099360864_dp,  299.8834977532089_dp,  418.99426494865514_dp,  588.6278555996877_dp,  832.992868332105_dp,  1191.901753683428_dp,  1738.4527532218908_dp,&
            2633.334968802445_dp,  4346.804005514949_dp,  9226.029101357823_dp,  0.27346932244426364_dp,  0.29493396442164976_dp,  0.3399080685990791_dp,  0.4126376624973753_dp,&
            0.5199403220488905_dp,  0.6718660354991877_dp,  0.8827295318060852_dp,  1.1726225736322562_dp,  1.5695736101023368_dp,  2.11260882730488_dp,  2.8560883332106437_dp,&
            3.8758604781327994_dp,  5.278022513223343_dp,  7.211435940783844_dp,  9.88567929424256_dp,  13.596923594598676_dp,  18.765442234455477_dp,  25.990394261374576_dp,&
            36.130687740769254_dp,  50.42631140632084_dp,  70.68536095476419_dp,  99.58565375015075_dp,  141.19809034005954_dp,  201.9985700024835_dp,  293.1161927876947_dp,&
            436.16688367969607_dp,  681.1401603164769_dp,  1173.9561106510982_dp,  2513.427330670278_dp,  9596.45252078042_dp]
       aw%aw_erange_matrix(:, 3) = [0.1375558839742156_dp,  0.4237299246561157_dp,  0.7441765936480988_dp,  1.1254465790607633_dp,  1.599969145991199_dp,  2.209153355047187_dp,&
            3.007419074036812_dp,  4.067629377796827_dp,  5.488551916391047_dp,  7.4052176805776915_dp,  10.003405965567303_dp,  13.540012927877152_dp,  18.37183281919031_dp,&
            24.996410070481446_dp,  34.11028053997399_dp,  46.692377235692085_dp,  64.1240439733692_dp,  88.36264595714124_dp,  122.19431621878961_dp,  169.60498916584984_dp,&
            236.33171640990545_dp,  330.6978635702647_dp,  464.9206246434375_dp,  657.2768359572758_dp,  936.038129172727_dp,  1347.6645495728708_dp,  1977.1418257598443_dp,&
            3010.365247643635_dp,  4989.502375747524_dp,  10618.098959488718_dp,  0.2769375636736402_dp,  0.2992417954352494_dp,  0.3460350190138061_dp,  0.4218590106585124_dp,&
            0.5340215517074247_dp,  0.6933277649140891_dp,  0.9152251840626363_dp,  1.221486913213276_dp,  1.642626003597483_dp,  2.2213351486133592_dp,  3.0173862657746264_dp,&
            4.114624855968743_dp,  5.630986359973453_dp,  7.732893888537386_dp,  10.656041641143203_dp,  14.735541137742192_dp,  20.449897303637602_dp,  28.485619304084782_dp,&
            39.83308706520946_dp,  55.9309260308313_dp,  78.88877991500787_dp,  111.84546669541834_dp,  159.5854467970953_dp,  229.71687105279605_dp,  335.26062431596165_dp,&
            501.33247053904597_dp,  785.6426382482606_dp,  1356.1465778665718_dp,  2902.1064912290262_dp,  11063.595563099649_dp]
       aw%aw_erange_matrix(:, 4) = [0.13904406912216055_dp,  0.4285658262322887_dp,  0.7535381269110286_dp,  1.1415170467922802_dp,  1.6262961987891298_dp,  2.251219295815267_dp,&
            3.073508947298677_dp,  4.170131059288361_dp,  5.64589044652921_dp,  7.644726401563794_dp,  10.365587059435269_dp,  14.084865660094467_dp,  19.18826867042609_dp,&
            26.216290635576655_dp,  35.929400137120936_dp,  49.401912630111426_dp,  68.15783436974851_dp,  94.36850091147679_dp,  131.14193494596577_dp,  182.94990655704783_dp,&
            256.26545423428075_dp,  360.5318037841861_dp,  509.6803886172214_dp,  724.6381601758845_dp,  1037.8525105472484_dp,  1502.6329781048792_dp,  2216.1957399484836_dp,&
            3390.239707940744_dp,  5640.184567731188_dp,  12031.914357631216_dp,  0.2799747756857045_dp,  0.3030322366372613_dp,  0.35146079871870245_dp,  0.4300750514997268_dp,&
            0.5466362705834609_dp,  0.7126509028520336_dp,  0.9446233777431962_dp,  1.265903264001014_dp,  1.7093454326345185_dp,  2.3211172345101936_dp,  3.1661481858067866_dp,&
            4.3359496851147625_dp,  5.959873147875833_dp,  8.221381778008848_dp,  11.381671588892244_dp,  15.81412949615409_dp,  22.054886195529974_dp,  30.877498691173102_dp,&
            43.404310509478876_dp,  61.27480112850241_dp,  86.90579792265169_dp,  123.90910951709412_dp,  177.80590692076998_dp,  257.378170487146_dp,  377.6084511567668_dp,&
            567.2233674632031_dp,  891.8325474154066_dp,  1541.8004917177743_dp,  3298.2637895129933_dp,  12556.72637834854_dp]
       aw%aw_erange_matrix(:, 5) = [0.1410960680209205_dp,  0.43524771510986004_dp,  0.7665207024486466_dp,  1.1639061919509923_dp,  1.6631581543024647_dp,  2.3104184708475035_dp,&
            3.1669893883276354_dp,  4.31584603104468_dp,  5.870689930586957_dp,  7.98866892968443_dp,  10.88837830270718_dp,  14.875483870146642_dp,  20.379381338093896_dp,&
            28.00589822144426_dp,  38.61341811457132_dp,  53.4233672664893_dp,  74.1813854349557_dp,  103.39372381185264_dp,  144.67616311753628_dp,  203.27301335960897_dp,&
            286.8373320413796_dp,  406.623030913939_dp,  579.3566390283918_dp,  830.3204498005787_dp,  1198.869283622746_dp,  1749.6835713037394_dp,  2600.2624010801505_dp,&
            4004.8623767257905_dp,  6698.999233522065_dp,  14341.174561026246_dp,  0.28416489317358423_dp,  0.30828949437487013_dp,  0.3590402711124012_dp,  0.441630361459561_dp,&
            0.5644857180354212_dp,  0.7401452754725963_dp,  0.9866774703026394_dp,  1.329777825854361_dp,  1.8058067012349928_dp,  2.4661647204139823_dp,  3.3835989597835665_dp,&
            4.661316876386105_dp,  6.44620592460885_dp,  8.948088111649072_dp,  12.46789325682324_dp,  17.439091025312962_dp,  24.488970722743385_dp,  34.52989393893134_dp,&
            48.896371819916574_dp,  69.55355936542944_dp,  99.42066613014704_dp,  142.88959294052594_dp,  206.70580521988288_dp,  301.61265861851626_dp,  445.87454058368684_dp,&
            674.2295764664547_dp,  1065.3146704916662_dp,  1846.16967438266_dp,  3948.0236905716197_dp,  15001.510222225108_dp]
       aw%aw_erange_matrix(:, 6) = [0.1433674220737173_dp,  0.4426628898148459_dp,  0.7809932553639084_dp,  1.1890061029798735_dp,  1.704736489813709_dp,  2.377608587440356_dp,&
            3.273747520531666_dp,  4.483286536624551_dp,  6.130604696754322_dp,  8.388821954366914_dp,  11.500472923115659_dp,  15.807168838438361_dp,  21.792381891352697_dp,&
            30.143468661033495_dp,  41.842032467696654_dp,  58.29624138024914_dp,  81.53560861416956_dp,  114.49935191584215_dp,  161.4660067158691_dp,  228.69835105975287_dp,&
            325.4212244904479_dp,  465.3259909564963_dp,  668.9419565107365_dp,  967.5375179808443_dp,  1410.04407287766_dp,  2076.991268832287_dp,  3114.140011343559_dp,&
            4834.667209637358_dp,  8139.076115188743_dp,  17497.251222371135_dp,  0.28880597793065726_dp,  0.3141509726409205_dp,  0.3675650580459323_dp,  0.45473435293913267_dp,&
            0.5848768476213745_dp,  0.7717685864556612_dp,  1.0353639404508452_dp,  1.40420681219775_dp,  1.9189447923955225_dp,  2.637427383030786_dp,  3.6421108679020224_dp,&
            5.050847015942239_dp,  7.032663994123608_dp,  9.830952206979282_dp,  13.797695578641337_dp,  19.444278287883428_dp,  27.51733560480189_dp,  39.112759774672554_dp,&
            55.84848632682428_dp,  80.12935385306108_dp,  115.56018244354962_dp,  167.60893912635234_dp,  244.72674037424497_dp,  360.41056089443987_dp,  537.5456031771105_dp,&
            819.2944118660395_dp,  1302.3421015152915_dp,  2264.016465253404_dp,  4840.7455216590715_dp,  18353.395351716823_dp]
       aw%aw_erange_matrix(:, 7) = [0.14540131106958792_dp,  0.4493201231807535_dp,  0.7940459513320671_dp,  1.2117724207523894_dp,  1.742680949975074_dp,  2.4393090166647937_dp,&
            3.372391950439403_dp,  4.6389574815944155_dp,  6.373746357911509_dp,  8.76549528730057_dp,  12.080324874755885_dp,  16.69553805190378_dp,  23.14873214645153_dp,&
            32.20953897394508_dp,  44.9849668143921_dp,  63.07490757920756_dp,  88.80297222972374_dp,  125.5611681322551_dp,  178.32791033803593_dp,  254.45242267177184_dp,&
            364.85390471544207_dp,  525.8798118900401_dp,  762.2489971360726_dp,  1111.8955650866828_dp,  1634.515787335805_dp,  2428.5690684021565_dp,  3671.812768394607_dp,&
            5743.738068689888_dp,  9729.027159680023_dp,  20999.800616241264_dp,  0.29296462129985257_dp,  0.3194381187890608_dp,  0.3753221134330865_dp,  0.46675631543264745_dp,&
            0.603721438689452_dp,  0.8011914963522677_dp,  1.0809588799458272_dp,  1.4743627941460153_dp,  2.0262889424868957_dp,  2.8010104053825082_dp,  3.8907309938858594_dp,&
            5.428124636584_dp,  7.604818108996212_dp,  10.698763243148507_dp,  15.114982641820642_dp,  21.44655967928129_dp,  30.56648418002324_dp,  43.766908758055855_dp,&
            62.972012402581825_dp,  91.06685250734347_dp,  132.41339099298463_dp,  193.68120824830095_dp,  285.2463053053247_dp,  423.7403809729896_dp,  637.3320623502533_dp,&
            978.785932422185_dp,  1565.1330028308446_dp,  2729.7495636084445_dp,  5836.893913938098_dp,  22085.748007810864_dp]
       aw%aw_erange_matrix(:, 8) = [0.14704498509396693_dp,  0.4547122051327194_dp,  0.8046596707087634_dp,  1.2303750029956346_dp,  1.7738485727322384_dp,  2.490259810187349_dp,&
            3.4542822843702057_dp,  4.768871567754432_dp,  6.577734807920994_dp,  9.08321025306724_dp,  12.572096360608057_dp,  17.453197931663386_dp,  24.312214399521167_dp,&
            33.99241851749717_dp,  47.713888969755104_dp,  67.25069977138067_dp,  95.19575945076387_dp,  135.35908014067192_dp,  193.37069282272213_dp,  277.60030747331535_dp,&
            400.57327496453024_dp,  581.1788761653216_dp,  848.1843494228331_dp,  1246.0269352662644_dp,  1844.99711942575_dp,  2761.311771739057_dp,  4204.4686241418385_dp,&
            6619.444613532952_dp,  11271.466932643574_dp,  24413.71197303721_dp,  0.2963273364593187_dp,  0.32373778596338965_dp,  0.38167760520153354_dp,  0.47667488142293696_dp,&
            0.6193657530314871_dp,  0.8257586307206471_dp,  1.1192417599126105_dp,  1.533595893064965_dp,  2.1174318721030514_dp,  2.94070601543572_dp,  4.104305072137064_dp,&
            5.7541994984779885_dp,  8.10243649038172_dp,  11.458433528239942_dp,  16.275877851241827_dp,  23.223421109964608_dp,  33.291908731012114_dp,  47.958153257319175_dp,&
            69.43709408889302_dp,  101.07403138845517_dp,  147.96365128967585_dp,  217.94992175702825_dp,  323.3082261097551_dp,  483.7891734632719_dp,  732.8442123233287_dp,&
            1132.825686861123_dp,  1820.9007070932764_dp,  3185.3327939434785_dp,  6812.567149441103_dp,  25734.79348292465_dp]
       aw%aw_erange_matrix(:, 9) = [0.14873028953425532_dp,  0.46025228447257244_dp,&
            0.815604068464183_dp,  1.249642730565338_dp,  1.806285223311792_dp,  2.5435423302102835_dp,  3.540333750899514_dp,  4.906044230436624_dp,  6.79416292983886_dp,&
            9.421954935784605_dp,  13.09904827265635_dp,  18.269242442139955_dp,  25.57201171534198_dp,  35.93349371466727_dp,  50.70186253663502_dp,  71.84990624465202_dp,&
            102.27999990001246_dp,  146.28615444437523_dp,  210.25885598550244_dp,  303.76857082731135_dp,  441.24677896342706_dp,  644.626277933146_dp,  947.5662954369608_dp,&
            1402.4345459606109_dp,  2092.551420913222_dp,  3156.1169200796808_dp,  4842.013686573819_dp,  7676.179724663517_dp,  13145.496536528115_dp,  28580.5493144719_dp,&
            0.2997770495160067_dp,  0.328171810325839_dp,  0.3882763609129264_dp,  0.4870382457606423_dp,  0.6358038475084181_dp,  0.8517074887441279_dp,  1.1598833349774023_dp,&
            1.5967979705738689_dp,  2.2151837931186287_dp,  3.0913230528117115_dp,  4.335827775714694_dp,  6.109657533128254_dp,  8.648033701785389_dp,  12.296330771130819_dp,&
            17.564247818840375_dp,  25.20806484627957_dp,  36.35635494507822_dp,  52.70347606492981_dp,  76.80980640661534_dp,  112.57226010676658_dp,  165.9717478515792_dp,&
            246.28595384282085_dp,  368.1305098723015_dp,  555.1307149475209_dp,  847.33732032532_dp,  1319.0786553040762_dp,  2132.484724545273_dp,  3743.164994620193_dp,&
            8008.935958082978_dp,  30201.87390773344_dp]
       aw%aw_erange_matrix(:, 10) = [0.15074658869167354_dp,  0.4668958035836203_dp,  0.8287813045866178_dp,  1.27295686026729_dp,  1.8457432235406386_dp,&
            2.6087085124986893_dp,  3.6461433188342913_dp,  5.0756183993344335_dp,  7.063160044261513_dp,  9.845291593081901_dp,  13.761294933072154_dp,  19.30074127352979_dp,&
            27.17394477351595_dp,  38.41701402640067_dp,  54.549426069506936_dp,  77.81182846954663_dp,  111.5272001496651_dp,  160.6530382259186_dp,  232.63167019101493_dp,&
            338.7100316718215_dp,  496.0070705705839_dp,  730.7907240768792_dp,  1083.762366034212_dp,  1618.8284659000788_dp,  2438.461166724189_dp,  3713.4370228988228_dp,&
            5751.214494532453_dp,  9197.732198940443_dp,  15865.762408736662_dp,  34662.09472159822_dp,  0.3039067430763809_dp,  0.3335109318096939_dp,  0.39628223332684037_dp,&
            0.4996995618831505_dp,  0.6560124551532797_dp,  0.8837941444637961_dp,  1.2104228747326704_dp,  1.6758387356208266_dp,  2.338139093731629_dp,  3.2818964868464273_dp,&
            4.630558114772758_dp,  6.565010733264908_dp,  9.3515186170747_dp,  13.384001926625498_dp,  19.24839955061762_dp,  27.821277621574968_dp,  40.421935481490465_dp,&
            59.048766447593984_dp,  86.74957609916163_dp,  128.20747215218674_dp,  190.67974791015314_dp,  285.53140615722333_dp,  430.82223445682513_dp,  655.9371796248371_dp,&
            1010.808613556427_dp,  1587.720772783595_dp,  2585.9493214732233_dp,  4560.104639212192_dp,  9764.47140618184_dp,  36744.78826999934_dp]
       aw%aw_erange_matrix(:, 11) = [0.15327074211168326_dp,  0.47523666558201105_dp,  0.8454081293698346_dp,  1.302555423844387_dp,  1.8961670213435633_dp,  2.69253915656829_dp,&
            3.7831603662440982_dp,  5.296661745560639_dp,  7.416146989170787_dp,  10.404588465380735_dp,  14.642337577033341_dp,  20.682906910118618_dp,  29.33645605992322_dp,&
            41.795526321167316_dp,  59.825609960791006_dp,  86.05587769109584_dp,  124.42572946943761_dp,  180.87560328043566_dp,  264.42380444802404_dp,  388.8590073673682_dp,&
            575.4255155780847_dp,  857.1327939358282_dp,  1285.7823929714118_dp,  1943.721808327101_dp,  2964.44501552976_dp,  4572.099023467233_dp,  7170.690868688525_dp,&
            11603.357270785074_dp,  20212.957283840064_dp,  44452.062155362604_dp,  0.3090804319893144_dp,  0.34024841302588593_dp,  0.40647905218403607_dp,  0.5159643143279251_dp,&
            0.6821714531077357_dp,  0.9256259273559609_dp,  1.2767723822584245_dp,  1.7803337442227432_dp,  2.5018545143392776_dp,  3.5375123917367213_dp,  5.028880015326183_dp,&
            7.185241661144039_dp,  10.31751858363557_dp,  14.890153020130612_dp,  21.600969658960963_dp,  31.504943516808023_dp,  46.20741046677315_dp,  68.16803820969625_dp,&
            101.1831518566797_dp,  151.15859371819528_dp,  227.3632897864881_dp,  344.4976384132564_dp,  526.2026890949384_dp,  811.3228331489025_dp,  1266.1935178202223_dp,&
            2013.0176783429627_dp,  3312.4890760215126_dp,  5880.358930845362_dp,  12610.246229383727_dp,  47327.426186591925_dp]
       aw%aw_erange_matrix(:, 12) = [0.15611582449200168_dp,  0.484670676531795_dp,  0.864327133672447_dp,  1.3364820958197925_dp,  1.95441681134768_dp,  2.7901462861902027_dp,&
            3.9439511540731202_dp,  5.558103736628507_dp,  7.836974880015686_dp,  11.076799625053408_dp,  15.710099980914652_dp,  22.372456900097028_dp,  32.00355723960249_dp,&
            46.00113943456652_dp,  66.45713829064987_dp,  96.52248517478913_dp,  140.97482131096695_dp,  207.10885220939085_dp,  306.14421022273984_dp,  455.47137403605_dp,&
            682.2697601790953_dp,  1029.4036326614776_dp,  1565.1742345818764_dp,  2399.817998955818_dp,  3714.5558190868474_dp,  5816.906307974656_dp,  9263.140996927303_dp,&
            15206.919687605141_dp,  26815.704784746697_dp,  59463.77740012989_dp,  0.31491712377093267_dp,  0.3479154795072989_dp,  0.41821106711586165_dp,  0.5348674813300344_dp,&
            0.7128490804971861_dp,  0.9750995854659087_dp,  1.3558946548404465_dp,  1.9059877530566083_dp,  2.700404389007717_dp,  3.8502466936019597_dp,  5.520642891704094_dp,&
            7.958186966900626_dp,  11.533140951457444_dp,  16.804769365680052_dp,  24.623195574865484_dp,  36.28937749112529_dp,  53.80834509662708_dp,  80.29343692857348_dp,&
            120.617438528961_dp,  182.4720644848956_dp,  278.11257974792727_dp,  427.27607514947135_dp,  662.181110971826_dp,  1036.4628530654331_dp,  1642.486743242303_dp,&
            2650.295375405498_dp,  4418.177298699454_dp,  7913.127380453787_dp,  17012.044780129974_dp,  63654.92138377425_dp]
       aw%aw_erange_matrix(:, 13) = [0.1585986414808211_dp,  0.49293227874864953_dp,  0.8809949627642996_dp,  1.3665913620994945_dp,  2.0065154205489626_dp,  2.878132174317013_dp,&
            4.090028118330613_dp,  5.797486964911083_dp,  8.225361402603866_dp,  11.702236433624119_dp,  16.71188568127808_dp,  23.971353089725582_dp,  34.550285638470136_dp,&
            50.05461492230114_dp,  72.91135545553885_dp,  106.8135350467453_dp,  157.4208108370035_dp,  233.47191953414887_dp,  348.56667909998555_dp,  524.0475847624567_dp,&
            793.707794772235_dp,  1211.5739798923714_dp,  1864.96076865716_dp,  2896.8251945846673_dp,  4545.4128751191465_dp,  7219.5552340802415_dp,  11662.784361729111_dp,&
            19411.052011165215_dp,  34635.22084899565_dp,  77428.2594678933_dp,  0.32001521125658394_dp,  0.35467076049069735_dp,  0.4286612826978466_dp,  0.5518737641051952_dp,&
            0.7406950741154976_dp,  1.020384176642642_dp,  1.4289161118477156_dp,  2.022921463432152_dp,  2.8867554481509226_dp,  4.146355805574743_dp,  5.990514160302193_dp,&
            8.703718091097011_dp,  12.717190750178169_dp,  18.688773175675315_dp,  27.628891977363935_dp,  41.10078967616605_dp,  61.541495564791326_dp,  92.78097905404866_dp,&
            140.8899743107899_dp,  215.57923784028515_dp,  332.53694700008424_dp,  517.3923979879823_dp,  812.5836504864853_dp,  1289.6958474565083_dp,  2073.21922438688_dp,&
            3392.879910503737_dp,  5728.411572233089_dp,  10353.785071731596_dp,  22327.676016416055_dp,  83326.1396670659_dp]
       aw%aw_erange_matrix(:, 14) = [0.16074983925508995_dp,  0.5001125329781305_dp,  0.895558007331798_dp,  1.3930676119416314_dp,  2.0526395034160516_dp,  2.9565619374491274_dp,&
            4.221130086493037_dp,  6.013805694103401_dp,  8.578774475403922_dp,  12.275421409740702_dp,  17.636750591062544_dp,  25.45877687591181_dp,  36.93833585853616_dp,&
            53.88713953905948_dp,  79.0668373760298_dp,  116.71771479662618_dp,  173.39998521474163_dp,  259.34441610997936_dp,  390.64111209658677_dp,  592.8223625140175_dp,&
            906.7916067039687_dp,  1398.75497341123_dp,  2177.101203426909_dp,  3421.6595296219302_dp,  5436.041196216437_dp,  8747.131818317119_dp,  14319.470865868023_dp,&
            24141.727891729002_dp,  43561.727407829894_dp,  98145.10564234719_dp,  0.3244358799675633_dp,  0.36057328739839345_dp,  0.43787941475989645_dp,  0.5670049256958328_dp,&
            0.7656628944752096_dp,  1.061285309171881_dp,  1.4953457271663428_dp,  2.130078015174346_dp,  3.0588087175505323_dp,  4.421869344637797_dp,  6.431225106069265_dp,&
            9.408834909529705_dp,  13.846816687671257_dp,  20.50252534817982_dp,  30.550011110026805_dp,  45.82334962368959_dp,  69.2111001525887_dp,  105.3018847464044_dp,&
            161.45180420057966_dp,  249.56900420929912_dp,  389.13450782321206_dp,  612.3924255583513_dp,  973.4482751978251_dp,  1564.7379361211877_dp,  2548.6891529503337_dp,&
            4226.38253248531_dp,  7222.917612404246_dp,  13174.2778180717_dp,  28509.267725054422_dp,  106160.2639173599_dp]
       aw%aw_erange_matrix(:, 15) = [0.16262039381403579_dp,  0.5063730793021797_dp,  0.9083149017898943_dp,  1.4163907532992552_dp,  2.093511982304147_dp,  3.026477274832255_dp,&
            4.338696841349741_dp,  6.208955650732167_dp,  8.899549165899632_dp,  12.798932651615898_dp,  18.486939408625744_dp,  26.835311895791346_dp,  39.16387831263876_dp,&
            57.485091296067274_dp,  84.89004001294464_dp,  126.1629602905402_dp,  188.7682361032193_dp,  284.4506192138729_dp,  431.85558174323637_dp,  660.8659291039313_dp,&
            1019.8615285065216_dp,  1588.0275062829162_dp,  2496.5319476854356_dp,  3965.666107658971_dp,  6371.910543745203_dp,  10375.825366387413_dp,  17195.468551114795_dp,&
            29341.488754297978_dp,  53508.65421134318_dp,  121456.79271852584_dp,  0.3282825301877526_dp,  0.36574392060492084_dp,  0.44602159399289126_dp,  0.5804703376001655_dp,&
            0.7880316454913989_dp,  1.0981622518906817_dp,  1.5556172675403404_dp,  2.227923946607758_dp,  3.2169479016852027_dp,  4.676829032127763_dp,  6.84194682233671_dp,&
            10.070815237141659_dp,  14.915491370586214_dp,  22.232202221987517_dp,  33.35916502004518_dp,  50.40499220758785_dp,  76.72092077578877_dp,  117.6818705337546_dp,&
            181.9923044138381_dp,  283.8951043119145_dp,  446.95633356501054_dp,  710.6481252216735_dp,  1142.0217013584413_dp,  1857.0207285861836_dp,  3061.524750375996_dp,&
            5139.401571578885_dp,  8885.07671917959_dp,  16351.40165928286_dp,  35518.56312434257_dp,  132016.67889946306_dp]
       aw%aw_erange_matrix(:, 16) = [0.16461121992013572_dp,  0.5130537467027886_dp,  0.9219891409850267_dp,  1.4415265282665501_dp,  2.137812541238478_dp,  3.1026919987933703_dp,&
            4.4675924708665224_dp,  6.4241492550129164_dp,  9.255358150849865_dp,  13.383148444589107_dp,  19.441687490427764_dp,  28.39127642576762_dp,  41.696760806942606_dp,&
            61.60931997659668_dp,  91.61536108245578_dp,  137.1580410041917_dp,  206.80796953910746_dp,  314.1817503489433_dp,  481.1195168715122_dp,  743.0074290063551_dp,&
            1157.8021962841462_dp,  1821.5380800469534_dp,  2895.3812180016926_dp,  4653.7282926606185_dp,  7572.086677425676_dp,  12495.669401832365_dp,  20997.801241262867_dp,&
            36325.86747707651_dp,  67064.21998527332_dp,  153559.877190311_dp,  0.3323792923495753_dp,  0.37128645855707815_dp,  0.45481882638693466_dp,  0.5951234815811263_dp,&
            0.8125303911554078_dp,  1.1387980677420937_dp,  1.6224364113285599_dp,  2.3370716045920283_dp,  3.394480042467526_dp,  4.96495158621143_dp,  7.309291980811711_dp,&
            10.829475690871233_dp,  16.149448348937934_dp,  24.245113751975293_dp,  36.65532694022094_dp,  55.82764892756104_dp,  85.69068300209298_dp,  132.6116968691035_dp,&
            207.0173551055982_dp,  326.1707659062317_dp,  518.9945518804797_dp,  834.5771689653424_dp,  1357.4620315784393_dp,  2235.8803375612997_dp,  3736.3891876038815_dp,&
            6360.155290450391_dp,  11143.129516033901_dp,  20727.354533996564_dp,  45246.467731381585_dp,  167864.82966862986_dp]
       aw%aw_erange_matrix(:, 17) = [0.16651351865477473_dp,  0.519454577907408_dp,  0.9351507515667731_dp,  1.4658532410614418_dp,  2.1809354081217633_dp,  3.1773124769855223_dp,&
            4.594526615255376_dp,  6.637313429476753_dp,  9.609926239283139_dp,  13.968927280965282_dp,  20.40513523133305_dp,  29.97193207470911_dp,  44.28787090328388_dp,&
            65.85939691635402_dp,  98.59953714664259_dp,  148.66946924606546_dp,  225.85763585192822_dp,  345.86375336652645_dp,  534.124192482264_dp,  832.2963196222343_dp,&
            1309.3929484951884_dp,  2081.173047394442_dp,  3344.449587785483_dp,  5438.954165136708_dp,  8961.84030797197_dp,  14989.196056927289_dp,  25545.816609735895_dp,&
            44824.33409960151_dp,  83822.32422493344_dp,  193711.93304782244_dp,  0.3362966051832095_dp,  0.37662123463393204_dp,  0.4633544143824645_dp,  0.609443679214809_dp,&
            0.8366282415179814_dp,  1.1790166265088777_dp,  1.6889775685600297_dp,  2.4464495867510347_dp,  3.5735433123097637_dp,  5.257520611088955_dp,  7.787186659800698_dp,&
            11.61095992362215_dp,  17.43030221471878_dp,  26.351381738650314_dp,  40.13360778161903_dp,  61.60103361444086_dp,  95.33067886989048_dp,  148.81734792065342_dp,&
            234.46891374272712_dp,  373.0691144154962_dp,  599.8733847399587_dp,  975.5122812884488_dp,  1605.8677591621924_dp,  2679.2461731874328_dp,  4538.878632523566_dp,&
            7836.630003949259_dp,  13921.767968244894_dp,  26195.561124088643_dp,  57512.80156760765_dp,  213037.54770361463_dp]
       aw%aw_erange_matrix(:, 18) = [0.1690396225959909_dp,  0.527980783630836_dp,  0.9527749551685267_dp,  1.498633358447118_dp,  2.2394273289730373_dp,  3.279200504430522_dp,&
            4.768998475289711_dp,  6.932282588250648_dp,  10.103948332171994_dp,  14.790913525485701_dp,  21.76710746310645_dp,  32.22374652919386_dp,  48.00922529285711_dp,&
            72.01561958382231_dp,  108.80743703784105_dp,  165.65487919516372_dp,  254.2501943193653_dp,  393.59144812337_dp,  614.8871476380654_dp,  970.0073946246416_dp,&
            1546.2520771713355_dp,  2492.5552724924996_dp,  4066.776314102136_dp,  6722.740655991635_dp,  11274.547185878206_dp,  19219.24710678121_dp,  33422.55081167871_dp,&
            59863.52522850395_dp,  114088.22675193982_dp,  267340.0098022138_dp,  0.34150265594038326_dp,  0.3837647419711019_dp,  0.4748885595852012_dp,  0.62895347380013_dp,&
            0.8697020693702873_dp,  1.234606483406498_dp,  1.7816009492315084_dp,  2.599802736845317_dp,  3.8264796009278195_dp,  5.674012679777352_dp,  8.473042796813875_dp,&
            12.742076623550743_dp,  19.300782196734378_dp,  29.456166689431363_dp,  45.31165528195391_dp,  70.28579444325638_dp,  109.99285877629794_dp,  173.7564076580376_dp,&
            277.24527030010586_dp,  447.12984357739214_dp,  729.4347143254225_dp,  1204.7788753069976_dp,  2016.732652224046_dp,  3425.9003452972343_dp,  5916.967870704429_dp,&
            10425.95017181254_dp,  18902.03087469423_dp,  36195.04534310194_dp,  80228.85952678222_dp,  296685.58400527935_dp]
       aw%aw_erange_matrix(:, 19) = [0.17108080746256005_dp,  0.5348926066785868_dp,  0.9671403722271738_dp,  1.5255266347919436_dp,  2.28774274844055_dp,  3.3639396483885546_dp,&
            4.91510333449395_dp,  7.181015523939525_dp,  10.52351093743281_dp,  15.494175591137205_dp,  22.941350060033994_dp,  34.180859122853406_dp,  51.27100453309755_dp,&
            77.4598244795078_dp,  117.91998358022018_dp,  180.96932452902817_dp,  280.12130623254893_dp,  437.57174786980636_dp,  690.205645858275_dp,  1100.09105655256_dp,&
            1773.0902705202968_dp,  2892.412355453728_dp,  4780.204848654172_dp,  8012.96799945039_dp,  13643.383830950022_dp,  23642.754995570554_dp,  41847.26051761937_dp,&
            76336.01713020806_dp,  148006.18917830326_dp,  351302.6976256991_dp,  0.3457128544559845_dp,  0.38958721362433624_dp,  0.4843783651820528_dp,  0.6451406758484797_dp,&
            0.8973522100189902_dp,  1.2814196455666504_dp,  1.8601713021665005_dp,  2.7308645051656586_dp,  4.044329620340544_dp,  6.035636552378946_dp,  9.073587781716466_dp,&
            13.741287830611094_dp,  20.968542428515345_dp,  32.25164234427676_dp,  50.02217350804501_dp,  78.2730704718351_dp,  123.63448072667782_dp,  197.24709108663288_dp,&
            318.07033201598585_dp,  518.8138973365631_dp,  856.7493308578589_dp,  1433.7763216800754_dp,  2434.4354710513157_dp,  4199.723043713225_dp,  7375.479667359867_dp,&
            13229.498032764022_dp,  24425.382905474227_dp,  47540.939727448866_dp,  106401.9838008185_dp,  393145.5653859997_dp]
       aw%aw_erange_matrix(:, 20) = [0.174208015078396_dp,  0.5455214527091232_dp,  0.9893700986809595_dp,  1.567453246082248_dp,  2.3636540955315315_dp,  3.498122562251638_dp,&
            5.14827993215702_dp,  7.581158591694967_dp,  11.204028439530992_dp,  16.64459737206355_dp,  24.87939231168654_dp,  37.44134013217713_dp,  56.75883515046388_dp,&
            86.71544891414382_dp,  133.58416570470814_dp,  207.60520215097725_dp,  325.68438918806163_dp,  516.070276345991_dp,  826.5786325694985_dp,  1339.2814449968866_dp,&
            2197.1869318584554_dp,  3653.596263478819_dp,  6165.272491162445_dp,  10572.339869517313_dp,  18455.027976540176_dp,  32866.476778294564_dp,  59929.80935369267_dp,&
            112813.36919811858_dp,  225478.89700441147_dp,  547806.2274616612_dp,  0.35216932807662665_dp,  0.39859666877705746_dp,  0.4992199916127204_dp,  0.6706989633151823_dp,&
            0.9413877281467774_dp,  1.3565962959075781_dp,  1.9874058221916224_dp,  2.944935529470972_dp,  4.403353007447031_dp,  6.637198331969463_dp,  10.082430143316417_dp,&
            15.437217064917919_dp,  23.830070641189668_dp,  37.103429138667835_dp,  58.297656696714924_dp,  92.48760514385602_dp,  148.24827340732497_dp,  240.26021813127747_dp,&
            394.0140410806184_dp,  654.4499869389458_dp,  1102.1183917626631_dp,  1884.023044363102_dp,  3273.8242918006918_dp,  5792.447112731419_dp,  10457.841973367156_dp,&
            19329.53596485672_dp,  36827.67063798521_dp,  73825.92204301585_dp,  168436.45444117606_dp,  622444.873239739_dp]
       aw%aw_erange_matrix(:, 21) = [0.1761202582043389_dp,  0.5520447609872519_dp,  1.0030974507555772_dp,  1.5935326203454305_dp,  2.4112315066636985_dp,  3.582863330501751_dp,&
            5.296667728366527_dp,  7.8377867885008925_dp,  11.643980957560537_dp,  17.394560044285978_dp,  26.15386564019807_dp,  39.605200340498676_dp,  60.43629423022499_dp,&
            92.98154542968216_dp,  144.30465853295624_dp,  226.04648771845612_dp,  357.6210989688087_dp,  571.8230985597171_dp,  924.8176309939128_dp,  1514.236505987768_dp,&
            2512.5546787169756_dp,  4229.875717696101_dp,  7234.632430202447_dp,  12591.32549473927_dp,  22342.113889996825_dp,  40517.55575462074_dp,  75377.59785941406_dp,&
            144997.0428070577_dp,  296108.0338286232_dp,  731745.2865811108_dp,  0.35612112320338896_dp,  0.4041597774113941_dp,  0.5084797084927667_dp,  0.6867923770493396_dp,&
            0.9693484716162668_dp,  1.404717445752288_dp,  2.0695156681544655_dp,  3.084248878260065_dp,  4.639047881996104_dp,  7.035740134386317_dp,  10.757234904755125_dp,&
            16.58309947892346_dp,  25.78417424843358_dp,  40.454121614614245_dp,  64.0813539216139_dp,  102.54891398088944_dp,  165.90782834877612_dp,  271.5712510504401_dp,&
            450.164506454231_dp,  756.4348015438803_dp,  1290.0017302932613_dp,  2235.688380530208_dp,  3943.8151743642957_dp,  7094.4981359781395_dp,  13045.295369347426_dp,&
            24603.05448593095_dp,  47901.44750228873_dp,  98085.01445426073_dp,  227192.3617986822_dp,  840731.2172617969_dp]
       aw%aw_erange_matrix(:, 22) = [0.1777047921517639_dp,  0.5574641492548221_dp,  1.0145511230139452_dp,  1.615403373339322_dp,  2.4513426308010158_dp,  3.654686527513127_dp,&
            5.423110673160765_dp,  8.057659233477818_dp,  12.023051097483231_dp,  18.04454685946229_dp,  27.2652711716243_dp,  41.50449808496695_dp,  63.68639951776318_dp,&
            98.56001627479766_dp,  153.92310734481697_dp,  242.72975295820916_dp,  386.770803794943_dp,  623.1980350390293_dp,  1016.2779319339597_dp,  1678.941978088846_dp,&
            2813.0616487372854_dp,  4786.311559672938_dp,  8282.29270001942_dp,  14601.419569806514_dp,  26282.105739415867_dp,  48430.57601768038_dp,  91721.68927815263_dp,&
            179924.08934574723_dp,  374820.9154378848_dp,  941327.26358562_dp,  0.35939787631562287_dp,  0.40880114349624547_dp,  0.5162611101798964_dp,  0.700403353403251_dp,&
            0.9931343712493289_dp,  1.4458855436673317_dp,  2.14016413580574_dp,  3.204825966103843_dp,  4.844304322301563_dp,  7.385060426051078_dp,  11.352727207647774_dp,&
            17.601567628385595_dp,  27.534196851429424_dp,  43.479077462900264_dp,  69.34760901334614_dp,  111.79404993830953_dp,  182.29420743169334_dp,  300.93178966239077_dp,&
            503.4177074926195_dp,  854.3541827114868_dp,  1472.829099303335_dp,  2582.950642714769_dp,  4616.207797210845_dp,  8424.894875912756_dp,  15742.71644167282_dp,&
            30226.322987810392_dp,  60011.99599547493_dp,  125336.8664775516_dp,  294698.3519483209_dp,  1092989.8333233916_dp]
       aw%aw_erange_matrix(:, 23) = [0.1799647702096782_dp,  0.5652158330702483_dp,  1.0310122635781176_dp,  1.6470123533176932_dp,  2.5096521841629245_dp,  3.7597091666054_dp,&
            5.609096447864183_dp,  8.383032013517075_dp,  12.587531728381435_dp,  19.018811032717153_dp,  28.942669406068813_dp,  44.39198046554904_dp,  68.66582954328437_dp,&
            107.17728460643622_dp,  168.91213211075208_dp,  268.97398648093684_dp,  433.091609012251_dp,  705.730911521849_dp,  1164.951694181309_dp,  1950.1393511855931_dp,&
            3314.8576251656086_dp,  5729.913200878512_dp,  10089.505182983685_dp,  18135.458661616507_dp,  33359.02655596435_dp,  62993.26164148739_dp,  122648.53064183371_dp,&
            248143.1247122177_dp,  533911.066633673_dp,  1377658.9174851538_dp,  0.3640748855027726_dp,  0.4154710858336259_dp,  0.5275322007755151_dp,  0.7202571525801186_dp,&
            1.0280528203703319_dp,  1.5066998100771394_dp,  2.2451905408110053_dp,  3.385256860173649_dp,  5.153564269391184_dp,  7.915191281602387_dp,  12.263349591995896_dp,&
            19.17157925153088_dp,  30.255048026265875_dp,  48.22502671655241_dp,  77.69046144427213_dp,  126.59311361740465_dp,  208.81898401605096_dp,  349.03416680624724_dp,&
            591.8119175008418_dp,  1019.2202578014185_dp,  1785.4939265361284_dp,  3187.1280052231455_dp,  5808.604372665699_dp,  10835.102219391534_dp,  20748.74858833901_dp,&
            40952.890888395414_dp,  83848.098471279_dp,  180849.87459593892_dp,  436512.7424947387_dp,  1628215.5545881148_dp]
       aw%aw_erange_matrix(:, 24) = [0.1822263529525982_dp,  0.5729996377421059_dp,  1.047635792810745_dp,  1.6791461382690935_dp,  2.569340077952974_dp,  3.8679621955823156_dp,&
            5.80214952385766_dp,  8.723197827758348_dp,  13.18207930971986_dp,  20.052986490730195_dp,  30.737880819936766_dp,  47.50921787860396_dp,  74.09130058582036_dp,&
            116.65928109072428_dp,  185.57975117283885_dp,  298.4887547568673_dp,  485.82262550322196_dp,  800.9299575569713_dp,  1338.911182611906_dp,  2272.4501320329528_dp,&
            3921.5269940453863_dp,  6892.49799005733_dp,  12363.437660742366_dp,  22688.291515470326_dp,  42722.85096985635_dp,  82860.32561548792_dp,  166362.18596925205_dp,&
            348621.725686658_dp,  779233.374778165_dp,  2079017.4769696286_dp,  0.36875937076082166_dp,  0.4222059972744914_dp,  0.5390198837588891_dp,  0.7406604412767378_dp,&
            1.064209794623797_dp,  1.5701371533427577_dp,  2.3555718741622864_dp,  3.5763689918945434_dp,  5.483812634926247_dp,  8.486176027552165_dp,  13.253067856267064_dp,&
            20.89439819576877_dp,  33.271277856794725_dp,  53.54362530829514_dp,  87.14918047880543_dp,  143.58213829294814_dp,  239.6811218766154_dp,  405.8231322234427_dp,&
            697.8346358530627_dp,  1220.419781249127_dp,  2174.402936582649_dp,  3954.658279362839_dp,  7359.4771270599895_dp,  14054.111690526057_dp,  27639.364929808464_dp,&
            56238.866456371696_dp,  119209.6009779448_dp,  267043.87988619285_dp,  666555.4805450162_dp,  2511681.1233505663_dp]
       aw%aw_erange_matrix(:, 25) = [0.1847328901237733_dp,  0.5816579181920506_dp,  1.06623831659397_dp,  1.7153584342875687_dp,  2.6370944394526834_dp,  3.991747606377202_dp,&
            6.0245444078219315_dp,  9.118059462860709_dp,  13.87771304303639_dp,  21.273099986416568_dp,  32.874558748673664_dp,  51.25417761001875_dp,  80.67448644264461_dp,&
            128.28771388301732_dp,  206.25529162006907_dp,  335.55403757007736_dp,  552.9304377149555_dp,  923.8479731240166_dp,  1567.0928663554876_dp,  2702.6050494655915_dp,&
            4746.810112095555_dp,  8507.992746711141_dp,  15599.418914679965_dp,  29344.431295226907_dp,  56841.83138821593_dp,  113909.37304425648_dp,  237628.48164125465_dp,&
            520905.746109544_dp,  1225334.6216953252_dp,  3428882.660853226_dp,  0.3739561299590209_dp,  0.42974153402664883_dp,  0.5519996672430176_dp,  0.7639144679035607_dp,&
            1.1057476916121531_dp,  1.643586260939031_dp,  2.4843963114705385_dp,  3.801270659603255_dp,  5.875848620822421_dp,  9.170241595714248_dp,  14.45037787184633_dp,&
            23.0002074091981_dp,  36.99882559646296_dp,  60.19432989625214_dp,  99.12731724056621_dp,  165.39119652215672_dp,  279.8872363721547_dp,  481.0007472363202_dp,&
            840.6692554162208_dp,  1496.7525126524065_dp,  2720.0603815860054_dp,  5057.476332433218_dp,  9648.32206240805_dp,  18951.758348951535_dp,  38497.25395781284_dp,&
            81332.10767021636_dp,  180141.95260617262_dp,  424305.6328571203_dp,  1112162.6633453001_dp,  4275158.808925236_dp]
       aw%aw_erange_matrix(:, 26) = [0.18691650786522696_dp,  0.5892282516421157_dp,  1.082601142199495_dp,  1.7474335701008017_dp,  2.697541917641801_dp,  4.102986967183923_dp,&
            6.2258726307243295_dp,  9.478229744358938_dp,  14.51725016133735_dp,  22.40416043035721_dp,  34.87275063631003_dp,  54.7892770133363_dp,  86.95106742502657_dp,&
            139.4937652327716_dp,  226.41043496205668_dp,  372.1384211072813_dp,  620.0671404995333_dp,  1048.6405945196989_dp,  1802.5180601563457_dp,  3154.370380781851_dp,&
            5630.817881197405_dp,  10277.053936391727_dp,  19232.435067105616_dp,  37032.92799563058_dp,  73696.42898519848_dp,  152437.06029497125_dp,  330262.7698185401_dp,&
            757951.6008420573_dp,  1883277.8699494905_dp,  5569496.903527443_dp,  0.37848766297101066_dp,  0.4363686681953113_dp,  0.5635257557534139_dp,  0.7847411896337352_dp,&
            1.1432429581246502_dp,  1.710400928686577_dp,  2.6025143390201344_dp,  4.00918762919726_dp,  6.241432379311925_dp,  9.814018030878975_dp,  15.58816401063637_dp,&
            25.02210792050149_dp,  40.617544040137886_dp,  66.72766316858582_dp,  111.04481815761915_dp,  187.3905962133754_dp,  321.0551681264537_dp,  559.2435025237448_dp,&
            992.0158501613989_dp,  1795.4100023823648_dp,  3322.951465507801_dp,  6306.521335539891_dp,  12314.481572253086_dp,  24843.55105667421_dp,  52058.072852874495_dp,&
            114094.52399515717_dp,  264080.7768830504_dp,  655640.4058250692_dp,  1818602.1997626221_dp,  7205611.5064693205_dp]
       aw%aw_erange_matrix(:, 27) = [0.19022370155321527_dp,  0.600743360402157_dp,  1.1076668599949147_dp,  1.7969727227449328_dp,  2.7916954361292103_dp,  4.277740702902653_dp,&
            6.5449132123072955_dp,  10.054129405857609_dp,  15.549497842664334_dp,  24.247957163099166_dp,  38.164643624522434_dp,  60.67913113804187_dp,  97.53572421208379_dp,&
            158.6390445053919_dp,  261.3334003305023_dp,  436.50559155590565_dp,  740.1775348487877_dp,  1276.0295073067164_dp,  2240.2658443373566_dp,  4013.504032417865_dp,&
            7354.933813953658_dp,  13827.327012835409_dp,  26765.8040824929_dp,  53592.215438201536_dp,  111658.9332077399_dp,  244020.31011080078_dp,  565646.7162849859_dp,&
            1414014.9652098955_dp,  3921844.1534676724_dp,  13169687.030866666_dp,  0.38535856099370613_dp,  0.44651820489987665_dp,  0.581378827915768_dp,  0.817323895709002_dp,&
            1.2024468005348359_dp,  1.8168664498604732_dp,  2.792503569715967_dp,  4.346916881821315_dp,  6.841455614222962_dp,  10.882312536529671_dp,  17.498453583809766_dp,&
            28.45950172040968_dp,  46.85274666913898_dp,  78.1487335864584_dp,  132.20639852667364_dp,  227.12421582085403_dp,  396.8047794537329_dp,  706.1854709081963_dp,&
            1282.7567112370582_dp,  2383.8137152178447_dp,  4544.9881609113345_dp,  8921.411882096174_dp,  18107.46271796117_dp,  38212.670216679006_dp,  84456.33216515457_dp,&
            197425.42988053817_dp,  494979.5222323457_dp,  1359946.1830275727_dp,  4269995.133188314_dp,  18594665.56920044_dp]
       aw%aw_erange_matrix(:, 28) = [0.19253210390672987_dp,  0.6088167991143922_dp,  1.1253694421581641_dp,  1.8322551698762215_dp,  2.8593371177556604_dp,  4.4043903646211_dp,&
            6.77820202691528_dp,  10.479140516198576_dp,  16.3186936413819_dp,  25.636021161233007_dp,  40.670025252886504_dp,  65.21432629001508_dp,  105.78867675687461_dp,&
            173.7699859169831_dp,  289.3412272171402_dp,  488.9577195527415_dp,  839.7790283388382_dp,  1468.253800848674_dp,  2618.293272053489_dp,  4773.313983139066_dp,&
            8921.221516951595_dp,  17152.781586299636_dp,  34075.58424824942_dp,  70338.2733815119_dp,  151993.0955958592_dp,  347390.7830575113_dp,  852464.8697898219_dp,&
            2298933.9421763406_dp,  7097047.894166372_dp,  27714975.772786345_dp,  0.39015997333677005_dp,  0.4536843511969253_dp,  0.5941303836763525_dp,  0.8408337569147337_dp,&
            1.2455691251977075_dp,  1.8951408073458809_dp,  2.933537085150634_dp,  4.600165407270463_dp,  7.296212807798878_dp,  11.701195546464378_dp,  18.980540228606937_dp,&
            31.16106108953815_dp,  51.82170019429503_dp,  87.38781813144793_dp,  149.60556002756704_dp,  260.3773737102548_dp,  461.44327006014964_dp,  834.290139541255_dp,&
            1542.3362969521413_dp,  2923.3653677825737_dp,  5699.927184905687_dp,  11479.628646828858_dp,  24006.639124934227_dp,  52485.55856206102_dp,  121068.69686949911_dp,&
            298453.56221816584_dp,  801312.2270425739_dp,  2415145.5599759216_dp,  8639654.807896439_dp,  43666635.56753933_dp]
       aw%aw_erange_matrix(:, 29) = [0.1949107095730563_dp,  0.6171672239307736_dp,  1.143792172436484_dp,  1.8692333576553928_dp,  2.930748110931505_dp,  4.539085142263593_dp,&
            7.028182432141186_dp,  10.938130943606705_dp,  17.15624033998357_dp,  27.16067680457394_dp,  43.44774061879075_dp,  70.29315104327573_dp,  115.13144509398658_dp,&
            191.1011162579555_dp,  321.8342697698806_dp,  550.6671938372235_dp,  958.7797787732911_dp,  1701.8776841547697_dp,  3086.6003811493197_dp,  5735.080964660857_dp,&
            10953.106276804325_dp,  21590.73855228189_dp,  44160.136736286004_dp,  94375.76532091481_dp,  212768.38665855065_dp,  513000.1708834057_dp,  1350765.9399022928_dp,&
            4022932.679434288_dp,  14513078.288369589_dp,  75937900.62142041_dp,  0.39511226230548274_dp,  0.461140130776742_dp,  0.6075257413400078_dp,  0.8657410584741931_dp,&
            1.2916169964663862_dp,  1.979386468292054_dp,  3.0865718388636862_dp,  4.877330577353741_dp,  7.798462568355467_dp,  12.614401907780815_dp,  20.650563498437013_dp,&
            34.23929473043507_dp,  57.55196112216066_dp,  98.18231951367711_dp,  170.22503623388062_dp,  300.40452026165275_dp,  540.6000901136043_dp,  994.1995519608768_dp,&
            1873.38945083301_dp,  3628.392490521107_dp,  7251.604873762116_dp,  15029.267131045708_dp,  32509.31717011665_dp,  74021.0484408064_dp,  179523.81893802425_dp,&
            471846.2096834248_dp,  1380639.0831843547_dp,  4711303.6808049455_dp,  20609904.76278654_dp,  151191206.84136662_dp]
    case(32)
       aw%energy_range(:) = [4862.0_dp, 5846.0_dp, 6665.0_dp, 7800.0_dp, 10044.0_dp, 14058.0_dp,&
            19114.0_dp, 25870.0_dp, 35180.0_dp, 58986.0_dp, 85052.0_dp, 126612.0_dp, 247709.0_dp, 452410.0_dp,&
            1104308.0_dp, 2582180.0_dp, 10786426.0_dp, 72565710.0_dp]
       aw%aw_erange_matrix(:, 1) = [0.13254275519610229_dp,  0.4075002434334006_dp,  0.7129653611128071_dp,  1.072311142230351_dp,  1.5137054634325515_dp,  2.072585907106472_dp,&
            2.7948198255241965_dp,  3.7408887157784005_dp,  4.991538722697482_dp,  6.655495576177275_dp,  8.880073834050316_dp,  11.865844968082882_dp,  15.88700871736706_dp,&
            21.319800434807853_dp,  28.682257688821725_dp,  38.69010148421391_dp,  52.335569763753064_dp,  70.99909077535013_dp,  96.60819689305963_dp,  131.86486082729905_dp,&
            180.57288090293983_dp,  248.11368921054333_dp,  342.14755601137466_dp,  473.67067120472257_dp,  658.6709548634489_dp,  920.892639935345_dp,  1296.9341360028814_dp,&
            1847.0377800992435_dp,  2682.1510880264454_dp,  4047.085323969744_dp,  6660.040272218769_dp,  14108.029689918747_dp,  0.2667161148094765_dp,  0.2866077185534088_dp,&
            0.32818422173681944_dp,  0.3951613742572797_dp,  0.4934816587166093_dp,  0.6318550781930461_dp,  0.8226002283812441_dp,  1.0828685966380414_dp,  1.4363782680324575_dp,&
            1.915847231861476_dp,  2.566403777298568_dp,  3.4503727336112044_dp,  4.654008952170379_dp,  6.29699863203509_dp,  8.545911800328465_dp,  11.633320983339523_dp,&
            15.885087444621595_dp,  21.759494015934315_dp,  29.903701501622987_dp,  41.235837219460294_dp,  57.06571237167945_dp,  79.27552554087899_dp,  110.59838219031958_dp,&
            155.06890815199284_dp,  218.81060764214297_dp,  311.5722127601456_dp,  450.16760079454133_dp,  667.4313363099551_dp,  1039.6543728256195_dp,  1789.9748349736915_dp,  3833.9681364720745_dp,  14655.44632366239_dp]
       aw%aw_erange_matrix(:, 2) = [0.1343248849882633_dp,  0.4132592119099152_dp,  0.7240043390799359_dp,  1.091026925935713_dp,  1.54395256157537_dp,  2.120249021729213_dp,&
            2.8686740645792272_dp,  3.8538693799190424_dp,  5.162601596809094_dp,  6.912333890906867_dp,  9.263083009950755_dp,  12.433910552841194_dp,  16.725965711187758_dp,&
            22.554817278001472_dp,  30.49600178512726_dp,  41.3494455589017_dp,  56.23095197989688_dp,  76.70267881628598_dp,  104.96008395875586_dp,  144.10119183719314_dp,&
            198.5169327875596_dp,  274.4618860978532_dp,  380.899499080189_dp,  530.7797694607106_dp,  743.0392454072074_dp,  1045.9229630462034_dp,  1483.0537807692358_dp,&
            2126.1516611014854_dp,  3106.7411184121765_dp,  4713.500361879859_dp,  7790.3641571104345_dp,  16548.445240300603_dp,  0.2703480614473483_dp,  0.2910756261579897_dp,&
            0.3344559767841716_dp,  0.40448278489060463_dp,  0.5075565457043502_dp,  0.6530872691721998_dp,  0.8544340076849473_dp,  1.130277093651096_dp,  1.506570063148876_dp,&
            2.019293595727479_dp,  2.718337239493882_dp,  3.6729796521022116_dp,  4.9796461563368934_dp,  6.772925073652045_dp,  9.241267903229764_dp,  12.649452794111886_dp,&
            17.370863762271927_dp,  23.934101209739133_dp,  33.090676974430735_dp,  45.914061903125_dp,  63.94611611042478_dp,  89.41703574194196_dp,  125.58445451312652_dp,&
            177.27970461634078_dp,  251.853187763381_dp,  360.99548168525195_dp,  524.7699272889096_dp,  782.0601891451947_dp,  1222.611932252175_dp,  2108.162233431274_dp,  4512.831122628449_dp,  17222.120678400817_dp]
       aw%aw_erange_matrix(:, 3) = [0.13557338601660668_dp,  0.41730065260389615_dp,  0.7317746815552322_dp,  1.104251698157746_dp,  1.5654155709323194_dp,  2.1542162086848187_dp,&
            2.9215329135167067_dp,  3.9350769825526832_dp,  5.286080227577014_dp,  7.0985194219082555_dp,  9.541929524180782_dp,  12.849299970750083_dp,  17.34218791649807_dp,&
            23.466110116962277_dp,  31.840626080497724_dp,  43.33050125803531_dp,  59.147245510736425_dp,  80.99462553628598_dp,  111.2781774935937_dp,  153.40846663158962_dp,&
            212.24282951584453_dp,  294.7341924733544_dp,  410.8955699804921_dp,  575.2616255690891_dp,  809.1768735737788_dp,  1144.5847770269388_dp,  1630.9076924124556_dp,&
            2349.3603614162894_dp,  3448.4571118322583_dp,  5252.914668389623_dp,  8709.488526056672_dp,  18538.811858380694_dp,  0.27289359484351916_dp,  0.294220982001665_dp,&
            0.3388980759862916_dp,  0.411123396942743_dp,  0.5176359133047135_dp,  0.668364809184891_dp,  0.8774440884042964_dp,  1.1646980641075588_dp,  1.557761149115098_dp,&
            2.0950797638638385_dp,  2.8301605859021666_dp,  3.837593566983341_dp,  5.221615251219452_dp,  7.128324609566002_dp,  9.763174149466698_dp,  13.41611573645175_dp,&
            18.497910846798998_dp,  25.59282079046985_dp,  35.53550575895753_dp,  49.524062051914576_dp,  69.28781614975094_dp,  97.34010675862754_dp,  137.36838088542098_dp,&
            194.86122705934486_dp,  278.18695350675137_dp,  400.6526832866276_dp,  585.0253370519614_dp,  875.1939500761971_dp,  1371.947991018934_dp,  2368.536327804507_dp,&
            5068.399861166353_dp,  9319.551508880766_dp]
       aw%aw_erange_matrix(:, 4) = [0.1370499857872464_dp,  0.422087838650514_dp,  0.7410040914005517_dp,  1.1200141103213002_dp,  1.5910937845067552_dp,  2.195011651443817_dp,&
            2.985262954814932_dp,  4.033361919610758_dp,  5.436098124858981_dp,  7.325593527943965_dp,  9.883341376758114_dp,  13.359912878263156_dp,  18.102752475144438_dp,&
            24.595554809189636_dp,  33.51428674450282_dp,  45.80722857230063_dp,  62.80982762248363_dp,  86.41024933404324_dp,  119.28915593140628_dp,  165.26885298524112_dp,&
            229.82488386010735_dp,  320.8416671697241_dp,  449.7412806501914_dp,  633.1998473772219_dp,  895.83705332351_dp,  1274.6569305047908_dp,  1827.0549682010767_dp,&
            2647.329246149724_dp,  3907.374072374449_dp,  5981.275709049701_dp,  9956.014230886252_dp,  21245.91059831021_dp,  0.27590538134484827_dp,  0.2979573565068322_dp,&
            0.3442035946496803_dp,  0.41909606429356827_dp,  0.5297935391581631_dp,  0.6868712744253185_dp,  0.9054313778178337_dp,  1.2067332819592969_dp,  1.6205293980185511_dp,&
            2.1883874721831376_dp,  2.9684149635372044_dp,  4.041990934394181_dp,  5.523387463777131_dp,  7.573569384258536_dp,  10.420063818936963_dp,  14.385689720800096_dp,&
            19.930287584151156_dp,  27.711636209408702_dp,  38.67488869797654_dp,  54.18479605908046_dp,  76.22300391975581_dp,  107.6864815178738_dp,  152.8489241029493_dp,&
            218.10116939555124_dp,  313.21671931714536_dp,  453.742571335189_dp,  666.1923552433152_dp,  1001.3603174380673_dp,  1575.1576866889702_dp,  2723.745097867967_dp,&
            5826.475195617796_dp,  22177.608473258828_dp]
       aw%aw_erange_matrix(:, 5) = [0.1393751435556022_dp,  0.4296428076126454_dp,  0.7556269118458908_dp,  1.1451112107395522_dp,  1.6321993263496475_dp,  2.260676043411754_dp,&
            3.088405025081973_dp,  4.193293537779408_dp,  5.681535826776683_dp,  7.69912759768035_dp,  10.448065969109644_dp,  14.209271435146666_dp,  19.375182226620723_dp,&
            26.496316789495793_dp,  36.34808912459801_dp,  50.02711707757524_dp,  69.09067912961464_dp,  95.75961436726789_dp,  133.21501966925618_dp,  186.03449395457477_dp,&
            260.83759278866273_dp,  367.24818341626917_dp,  519.3442915671452_dp,  737.8771467239039_dp,  1053.7584086797096_dp,  1513.801330668974_dp,  2190.9702417788335_dp,&
            3205.2140171776487_dp,  4774.210825118251_dp,  7368.122740715144_dp,  12344.973991941335_dp,  26456.259009955764_dp,  0.28065064408654583_dp,  0.3038780328549319_dp,&
            0.3526759573849975_dp,  0.4319215261801991_dp,  0.5494801124103226_dp,  0.7170195083647463_dp,  0.9512879181334657_dp,  1.2759995304848823_dp,  1.7245524024350398_dp,&
            2.3439221516534534_dp,  3.2002414143320297_dp,  4.386814440304756_dp,  6.0356680674184835_dp,  8.334263247492462_dp,  11.549773796002393_dp,  16.064517095499955_dp,&
            22.427907607712154_dp,  31.43303552982579_dp,  44.23019793781067_dp,  62.496359675110796_dp,  88.69028623168306_dp,  126.44134851448199_dp,  181.15338754218374_dp,&
            260.9731813081323_dp,  378.4325232446994_dp,  553.5022272702486_dp,  820.1059662618102_dp,  1242.61943354538_dp,  1966.3786822155184_dp,  3410.3262356194664_dp,&
            7292.46886903304_dp,  27693.855354838164_dp]
       aw%aw_erange_matrix(:, 6) = [.14237486453944107_dp,  0.4394200534952092_dp,  0.7746555330683098_dp,  1.177996095520943_dp,  1.6864652619295697_dp,&
            2.3480282638346726_dp,  3.2266616421606034_dp,  4.409302651924302_dp,  6.015552554226239_dp,  8.21136864768518_dp,  11.228525969594424_dp,  15.392442739194209_dp,&
            21.162171526463748_dp,  29.18815796828059_dp,  40.3960633542662_dp,  56.109002958965746_dp,  78.22669193938121_dp,  109.4893460132257_dp,  153.8685432400606_dp,&
            217.14949285761446_dp,  307.80403247089794_dp,  438.30940204930323_dp,  627.1596980492438_dp,  901.9765906170177_dp,  1304.4290970945126_dp,  1898.3283702431997_dp,&
            2783.931515564688_dp,  4126.483644157721_dp,  6224.503686674229_dp,  9716.401957272288_dp,  16429.986790645627_dp,  35423.70971974651_dp,  0.28677747480312377_dp,&
            0.3115840468325672_dp,  0.36382211720758184_dp,  0.44896687419484355_dp,  0.5758826693409299_dp,  0.7577922978063393_dp,  1.013805148375157_dp,  1.371186351736884_dp,&
            1.8686546654955012_dp,  2.56115176168904_dp,  3.5267457265125604_dp,  4.876653159913234_dp,  6.769843587920379_dp,  9.434411669061411_dp,  13.198996941760914_dp,&
            18.53920060615783_dp,  26.14651426672167_dp,  37.03124772871653_dp,  52.677154132184704_dp,  75.27530196334217_dp,  108.08071153726374_dp,  155.96210569315744_dp,&
            226.2630374093172_dp,  330.18690270710874_dp,  485.1305151943868_dp,  718.9484199953382_dp,  1078.8257499526349_dp,  1653.3065930339428_dp,  2639.3168711629464_dp,  4598.936377351543_dp,  9833.306219117254_dp,  7228.01191754706_dp]
       aw%aw_erange_matrix(:, 7) = [.14502370796551703_dp,  0.4480829010539378_dp,  0.7916157981286331_dp,  1.2075243508496802_dp,&
            1.7355837902840658_dp,  2.4277406075189805_dp,  3.353852391880919_dp,  4.609630408244423_dp,  6.327831185413123_dp,  8.694192211759745_dp,  11.970290957285025_dp,&
            16.5265343114819_dp,  22.890028099278148_dp,  31.81440016230547_dp,  44.38216963745914_dp,  62.15561490757339_dp,  87.40030591467784_dp,  123.41813625304623_dp,&
            175.04637990163923_dp,  249.4101615146111_dp,  357.06464925805403_dp,  513.7420529668599_dp,  743.0485870384754_dp,  1080.6808579217216_dp,  1581.14809900435_dp,&
            2328.8609813458747_dp,  3457.607083546514_dp,  5188.8198870627575_dp,  7921.499237107771_dp,  12501.58289355634_dp,  21329.60428145102_dp,  46259.92356298488_dp,&
            0.29219234212764755_dp,  0.31845370690034347_dp,  0.373872890610569_dp,  0.4645031477238152_dp,  0.600179586011295_dp,  0.7956470306437238_dp,  1.0723453581784659_dp,&
            1.4610762243702629_dp,  2.0059080326461616_dp,  2.76987145000088_dp,  3.8432790806788373_dp,  5.355919395975122_dp,  7.495003555741856_dp,  10.531701795474737_dp,&
            14.860576373030604_dp,  21.05848895536974_dp,  29.97309227512008_dp,  42.856552877705376_dp,  61.56891170854085_dp,  88.88951722712217_dp,  128.99783246328127_dp,&
            188.2232678787817_dp,  276.231225060434_dp,  407.9416932740777_dp,  606.7569403074244_dp,  910.3888055268573_dp,  1382.7234338831393_dp,  2142.64856112768_dp,&
            3450.91828637881_dp,  6043.794023233803_dp,  12927.644737686549_dp,  8805.08862191054_dp]
       aw%aw_erange_matrix(:, 8) = [0.14754727473790832_dp,  0.4563621690490255_dp,  0.8079150128057052_dp,  1.2360970460871308_dp,  1.783465172289044_dp,  2.506029552409794_dp,&
            3.479706891724579_dp,  4.809331033064525_dp,  6.641460873789643_dp,  9.182776952331544_dp,  12.726704342125874_dp,  17.692182381254717_dp,  24.680446495334998_dp,&
            34.55864851933651_dp,  48.58367650175289_dp,  68.58652178045523_dp,  97.24837691414776_dp,  138.51657135506775_dp,  198.2350828975321_dp,  285.1066307856165_dp,&
            412.17114696470446_dp,  599.09566525532_dp,  875.7537331156108_dp,  1287.8873903467286_dp,  1906.2215953314396_dp,  2841.576751154979_dp,  4271.337323387628_dp,&
            6490.79405022356_dp,  10031.563416886413_dp,  16011.899712868733_dp,  27575.24726251585_dp,  60178.67874227437_dp,  0.2973552991659925_dp,  0.3250566186141874_dp,&
            0.38363560044751194_dp,  0.4797430677530813_dp,  0.6242227281583522_dp,  0.8334115127308077_dp,  1.131206159458274_dp,  1.552168109039008_dp,  2.146103694832248_dp,&
            2.984800054553381_dp,  4.171951997411542_dp,  5.857847187535488_dp,  8.26118966072219_dp,  11.701694869826639_dp,  16.64903553098479_dp,  23.796760202379794_dp,&
            34.17459871027205_dp,  49.32017560264589_dp,  71.54333486665355_dp,  104.33600828116667_dp,  153.01300195482676_dp,  225.72365375346706_dp,  335.069793993653_dp,&
            500.74431450896293_dp,  753.9795215702264_dp,  1145.5159990918794_dp,  1761.5202623948937_dp,  2761.370072156297_dp,  4489.970449108518_dp,  7909.344598672393_dp,  16932.78473977757_dp,  3749.23270457554_dp]
       aw%aw_erange_matrix(:, 9) = [.15002348519433922_dp,  0.46451130342750646_dp,  0.824045021042684_dp,  1.2645625089537342_dp,  1.8315097984767872_dp,  2.5851574108560396_dp,&
            3.6078322639753786_dp,  5.014105379427872_dp,  6.965398429552594_dp,  9.69114538724383_dp,  13.519686414362274_dp,  18.92366324500486_dp,  26.587123387509088_dp,&
            37.50529299176741_dp,  53.13378944809952_dp,  75.61310381203099_dp,  108.10840448253303_dp,  155.3272050619175_dp,  224.31329381924482_dp,  325.67298187252476_dp,&
            475.48339517776736_dp,  698.2866475093558_dp,  1031.8319129009446_dp,  1534.6726862536486_dp,  2298.5349133548734_dp,  3468.9808108623843_dp,  5281.614870909423_dp,&
            8131.595592028999_dp,  12730.978340941307_dp,  20567.038495110464_dp,  35778.24682408639_dp,  78609.82719130651_dp,  0.3024254049895312_dp,  0.3315918385852816_dp,&
            0.3933970226398002_dp,  0.49512550705651104_dp,  0.6486960402797961_dp,  0.872153928090401_dp,  1.1920524512393371_dp,  1.6470520511792885_dp,  2.2932691781075087_dp,&
            3.212208165551602_dp,  4.522554416231983_dp,  6.3977829741741665_dp,  9.092580992760151_dp,  12.982711954934162_dp,  18.625509909639966_dp,  26.85224510411176_dp,&
            38.910067639696216_dp,  56.68163261798167_dp,  83.02738125985056_dp,  122.3231205179819_dp,  181.31153080692263_dp,  270.46417120445864_dp,  406.1868615016362_dp,&
            614.4531917978084_dp,  936.963281227043_dp,  1442.136769945214_dp,  2246.721112418363_dp,  3565.852508027438_dp,  5859.209649832797_dp,  10391.307380953114_dp,&
            22278.529118569048_dp,  3643.6346001724_dp]
       aw%aw_erange_matrix(:, 10) = [0.15399032986872938_dp,  0.4776194738561411_dp,  0.8501753043803365_dp,  1.31107946792835_dp,  1.9107571034709487_dp,  2.7169110257810085_dp,&
            3.823183728851228_dp,  5.361535227120927_dp,  7.52023882937859_dp,  10.570320241106216_dp,  14.904710488340205_dp,  21.09662727004116_dp,  29.987193874300804_dp,&
            42.81777942605248_dp,  61.43118108216903_dp,  88.57949619935079_dp,  128.39852284791314_dp,  187.14404553452533_dp,  274.34323674515576_dp,  404.60959924450174_dp,&
            600.5280841395823_dp,  897.2831647081848_dp,  1350.157326096996_dp,  2046.8215895330711_dp,  3127.779028909418_dp,  4821.115088237592_dp,  7503.844806420884_dp,&
            11818.550736004621_dp,  18929.880007178966_dp,  31248.80735377127_dp,  55365.10950684564_dp,  123170.41015076623_dp,  0.310556146577657_dp,  0.3421802780606353_dp,&
            0.4094223614818588_dp,  0.5206878687071103_dp,  0.6898099116933477_dp,  0.9379031112155564_dp,  1.2963422286973751_dp,  1.8113088637921357_dp,  2.5506317768186104_dp,&
            3.6140696050359202_dp,  5.148822519751217_dp,  7.37304808270909_dp,  10.611713921777906_dp,  15.351596184949763_dp,  22.32620193123881_dp,  32.64780730303991_dp,&
            48.01425448983265_dp,  71.03533163849936_dp,  105.75208085448396_dp,  158.47022734372413_dp,  239.1101888774951_dp,  363.4164108225496_dp,  556.6165909250806_dp,&
            859.5761486038269_dp,  1339.388089086674_dp,  2108.33111127726_dp,  3360.4896315872797_dp,  5453.393419146713_dp,  9137.412009720016_dp,  16424.470778988587_dp,&
            35351.12874442453_dp,  32131.62635064372_dp]
       aw%aw_erange_matrix(:, 11) = [.15665201491118563_dp,  0.48645255086696204_dp,  0.8679140908633145_dp,  1.3429441991327895_dp,  1.9655663020388237_dp,  2.8089215705883355_dp,&
            3.9750326498505615_dp,  5.608890975167092_dp,  7.919132231547877_dp,  11.208702795546053_dp,  15.920717165510954_dp,  22.707526807683728_dp,  32.53546396340545_dp,&
            46.84477048297584_dp,  67.79548225092432_dp,  98.64818052206019_dp,  144.35799385904798_dp,  212.50854417045272_dp,  314.7924737632916_dp,  469.3788283542409_dp,&
            704.7334196247324_dp,  1065.8446744796906_dp,  1624.4783236718592_dp,  2496.2720234668195_dp,  3869.6487189479203_dp,  6055.704970336431_dp,  9577.158206331736_dp,&
            15337.311175310337_dp,  24985.53443387876_dp,  41924.21370939059_dp,  75335.12561757576_dp,  169240.0831820107_dp,  0.31601774300631147_dp,  0.34936919542211453_dp,&
            0.42045090124222617_dp,  0.5384991728556235_dp,  0.718776108006642_dp,  0.9847086381286597_dp,  1.3713420192235894_dp,  1.9306481293657263_dp,  2.7395796341433103_dp,&
            3.912291447920088_dp,  5.6187616588370295_dp,  8.113318173320359_dp,  11.778610539345083_dp,  17.193842137070394_dp,  25.24140130107934_dp,  37.2747560295197_dp,&
            55.384907761264955_dp,  82.82679809483118_dp,  124.70772472447071_dp,  189.1091967049713_dp,  288.93348305516747_dp,  444.97622020020765_dp,  691.102363652001_dp,&
            1083.0962406902304_dp,  1714.1073266384424_dp,  2742.5244927502104_dp,  4445.571757485345_dp,  7336.150374476061_dp,  12481.117979830125_dp,  22687.396569080272_dp,&
            49029.27213151443_dp,  82714.99577064996_dp]
       aw%aw_erange_matrix(:, 12) = [.15940855323823325_dp,  0.4956331404107984_dp,  0.8864643437510178_dp,  1.3765162067718617_dp,  2.023770980185183_dp,  2.9074148057363085_dp,&
            4.1388784725835_dp,  5.877927890230674_dp,  8.356513563090692_dp,  11.914499292816961_dp,  17.053639088844527_dp,  24.519740335146867_dp,  35.428643273777766_dp,&
            51.46079251611351_dp,  75.16399270031552_dp,  110.42815348333873_dp,  163.2357643618038_dp,  242.85834932021987_dp,  363.7810629576585_dp,  548.8279687880828_dp,&
            834.2882284949019_dp,  1278.4147835625017_dp,  1975.6780750770085_dp,  3080.979098507853_dp,  4851.394465485563_dp,  7719.529300718981_dp,  12426.190192755064_dp,&
            20273.565432739022_dp,  33665.94446711274_dp,  57559.451254663276_dp,  105152.47765026288_dp,  238976.3949138361_dp,  0.32167917419720454_dp,  0.35688752020406733_dp,&
            0.432113620138954_dp,  0.557526309412786_dp,  0.750001170356205_dp,  1.0355963446410574_dp,  1.4535706405145328_dp,  2.0626057961394766_dp,  2.9503327570367945_dp,&
            4.247930889038919_dp,  6.152607117843223_dp,  8.96241660484181_dp,  13.130563431815835_dp,  19.350693737814375_dp,  28.691905293869226_dp,  42.814199349842106_dp,&
            64.3152735879299_dp,  97.29381534174325_dp,  148.2732887039295_dp,  227.73244606015456_dp,  352.66878271895433_dp,  550.9427692926408_dp,  868.7360090031099_dp,&
            1383.5494745755586_dp,  2227.296811182226_dp,  3628.5346977825025_dp,  5993.829605644713_dp,  10082.06745111222_dp,  17463.261532301312_dp,  32186.387908159864_dp,&
            69962.03966745277_dp,  59964.04274551442_dp]
       aw%aw_erange_matrix(:, 13) = [.16373958436697705_dp,  0.5101264978608379_dp,  0.9159896069529507_dp,  1.430480778525928_dp,  2.1183125355919272_dp,  3.069088155985158_dp,&
            4.410666493484344_dp,  6.328951180857867_dp,  9.097685942967209_dp,  13.123807198212786_dp,  19.017093457142934_dp,  27.69800527238066_dp,  40.56599533589728_dp,&
            59.76432998550568_dp,  88.60026000403526_dp,  132.21751623180904_dp,  198.68257999925436_dp,  300.7563968967366_dp,  458.81385584717975_dp,  705.7073030471846_dp,&
            1094.9631974180384_dp,  1714.7688190287618_dp,  2712.160181788416_dp,  4335.464784701027_dp,  7009.996439266402_dp,  11475.67124355327_dp,  19043.69699662228_dp,&
            32095.955514205307_dp,  55144.14939722096_dp,  97554.96015781426_dp,  183803.08980484723_dp,  427090.13572662364_dp,  0.33058526273756267_dp,  0.36885467782088144_dp,&
            0.450950093626708_dp,  0.5886660848688742_dp,  0.8017140595780422_dp,  1.12082540025602_dp,  1.592831379794522_dp,  2.2886261360881694_dp,  3.31553715895999_dp,&
            4.836588787162195_dp,  7.100671060940492_dp,  10.490115312417089_dp,  15.596294033961502_dp,  23.34074274734257_dp,  35.17091805226463_dp,  53.37952681146768_dp,&
            81.6305668168434_dp,  125.83495501612735_dp,  195.62426652036729_dp,  306.86047742079927_dp,  485.9640372839356_dp,  777.4791364469528_dp,  1257.4930839302167_dp,&
            2057.832096842602_dp,  3410.5310726374632_dp,  5731.642662527518_dp,  9785.587565752354_dp,  17033.465557765263_dp,  30505.194880449075_dp,  57794.75930208428_dp,&
            127350.9867284555_dp,  71374.61882211326_dp]
       aw%aw_erange_matrix(:, 14) = [.16729345587752817_dp,  0.5220837915443625_dp,  0.9405741148525995_dp,  1.475915186417087_dp,  2.1988425579314264_dp,  3.2084229320437876_dp,&
            4.647658877867247_dp,  6.726900296718572_dp,  9.759556549152265_dp,  14.217184101692544_dp,  20.815259881214573_dp,  30.6479058416218_dp,  45.40134046704283_dp,&
            67.69506474900591_dp,  101.63225933231537_dp,  153.69629208332117_dp,  234.2259466666511_dp,  359.8693374590021_dp,  557.713000374505_dp,  872.3154950503812_dp,&
            1377.8469922115526_dp,  2199.339147354942_dp,  3550.4526300007215_dp,  5801.72143903217_dp,  9606.137468107807_dp,  16135.098020267218_dp,  27533.125876830458_dp,&
            47827.00298089529_dp,  84872.60988452473_dp,  155240.74372944486_dp,  301769.0915913182_dp,  717663.1356247213_dp,  0.3379034645137452_dp,  0.37881946738547795_dp,&
            0.46689089916190624_dp,  0.615406140323357_dp,  0.8467064090676832_dp,  1.1959082144868687_dp,  1.7170430113398156_dp,  2.4927824943410823_dp,  3.6497349710027573_dp,&
            5.382588868417295_dp,  7.992470539282596_dp,  11.948356757923296_dp,  17.986233607448835_dp,  27.27066375498558_dp,  41.66060391661398_dp,  64.15126447641241_dp,&
            99.61667730605177_dp,  156.07223690236458_dp,  246.84813859319397_dp,  394.3808745412634_dp,  636.9222168901913_dp,  1040.5940426531995_dp,  1721.4116028413248_dp,&
            2886.2600258571556_dp,  4910.7228383421725_dp,  8490.620025490385_dp,  14947.387548118924_dp,  26881.538126356885_dp,  49765.1843652489_dp,  97101.2613405646_dp,&
            217655.43736295513_dp,  04161.4023302288_dp]
       aw%aw_erange_matrix(:, 15) = [.17199827765823103_dp,  0.5380059294727404_dp,  0.9736341997615289_dp,  1.5377352694938258_dp,  2.3097736290470463_dp,  3.402751176442763_dp,&
            4.982320081056548_dp,  7.295965784769458_dp,  10.718311338355647_dp,  15.822271186795035_dp,  23.491931948889743_dp,  35.10335805529794_dp,  52.81700264781087_dp,&
            80.05534788452564_dp,  122.29130031272524_dp,  188.3639662992407_dp,  292.70049836028585_dp,  459.11521059936763_dp,  727.3932946321102_dp,  1164.8655376002584_dp,&
            1887.0757702166622_dp,  3095.3089278858897_dp,  5145.990609712843_dp,  8681.612837930576_dp,  14883.222751239353_dp,  25969.45893893789_dp,  46211.24410112112_dp,&
            84069.60121476448_dp,  156951.2764731812_dp,  303071.86527144857_dp,  621318.3413283181_dp,  1540176.9421143653_dp,  0.3476062896043872_dp,  0.39221916200807294_dp,&
            0.4886942701051685_dp,  0.6525426235716102_dp,  0.9100580818924567_dp,  1.3030331602543401_dp,  1.8966193817904524_dp,  2.7919595402699877_dp,  4.146395650315994_dp,&
            6.205957727039082_dp,  9.35800145395578_dp,  14.217249893697428_dp,  21.767802513575838_dp,  33.599962887862425_dp,  52.309603978141894_dp,  82.17963968964122_dp,&
            130.3576745067898_dp,  208.91882658105692_dp,  338.53188644986636_dp,  555.0775251553036_dp,  921.7987850368266_dp,  1552.0204912904064_dp,  2652.486928566717_dp,&
            4607.857141150226_dp,  8149.500672123086_dp,  14702.176180519695_dp,  27120.413678845984_dp,  51329.09608829233_dp,  100319.89810256775_dp,  206207.35016280194_dp,  478990.19443877833_dp,  773158.149536881_dp]
       aw%aw_erange_matrix(:, 16) = [.17588045499160682_dp,  0.5512257021498627_dp,  1.0013703118630188_dp,  1.5902434264445422_dp,  2.405215799012508_dp,  3.572121574953908_dp,&
            5.277810308974558_dp,  7.8050898047384285_dp,  11.587777615192842_dp,  17.298487998001825_dp,  25.990127895451295_dp,  39.32634673857857_dp,  59.96085024576093_dp,&
            92.16864779688744_dp,  142.9088285424604_dp,  223.63608103002358_dp,  353.42944347083966_dp,  564.4730793406918_dp,  911.8045474251959_dp,  1490.9411871748462_dp,&
            2470.3259539694427_dp,  4152.227298440316_dp,  7089.513439000162_dp,  12314.898215191599_dp,  21802.90624200313_dp,  39428.49288596958_dp,  73023.82463664327_dp,&
            138969.14164413267_dp,  272986.4175263669_dp,  557893.4292914037_dp,  1213363.1419136985_dp,  3160013.276178309_dp,  0.35562539358245254_dp,  0.4034598630828778_dp,&
            0.507310698035364_dp,  0.6847544241793833_dp,  0.965797905499124_dp,  1.3985904597574628_dp,  2.0590328917447116_dp,  3.0664136286150927_dp,  4.608786136806865_dp,&
            6.9844140449501895_dp,  10.67005177539468_dp,  16.434554413045298_dp,  25.529954614876807_dp,  40.0165712983268_dp,  63.323071537954775_dp,  101.22419148541576_dp,&
            163.5720951785129_dp,  267.4096498030514_dp,  442.6620147937182_dp,  742.7299414402248_dp,  1264.594124324258_dp,  2187.7911393276054_dp,  3851.763119153975_dp,&
            6913.383865787345_dp,  12677.15458167547_dp,  23810.25092763602_dp,  45950.63865686885_dp,  91498.37485707413_dp,  189213.13598878586_dp,  412506.46573426365_dp,  1004859.506854019_dp,  757500.169762731_dp]
       aw%aw_erange_matrix(:, 17) = [.18118431919794764_dp,  0.569409896731795_dp,  1.0399575954447753_dp,  1.6642774050832718_dp,  2.5416705589038666_dp,  3.817686030211855_dp,&
            5.712321227186271_dp,  8.564613330770575_dp,  12.904350409701964_dp,  19.568888331266418_dp,  29.895695056866206_dp,  46.04343093576417_dp,  71.5338357343432_dp,&
            112.17788900752778_dp,  177.6801488903931_dp,  284.45814060206254_dp,  460.674367791145_dp,  755.3691485577447_dp,  1255.340718976754_dp,  2116.97044167198_dp,&
            3627.5472563405187_dp,  6326.311406095248_dp,  11249.843141984617_dp,  20444.63279555149_dp,  38074.51328278192_dp,  72907.33134133912_dp,  144152.9579642091_dp,&
            295906.8845605476_dp,  635237.5946890834_dp,  1441541.5594042293_dp,  3533032.483017847_dp,  10300282.297035703_dp,  0.3666004567229466_dp,  0.41909533951682115_dp,&
            0.5337007693305699_dp,  0.7311922617920863_dp,  1.047397173851464_dp,  1.5405811924145458_dp,  2.3040410310537687_dp,  3.4869635940224217_dp,  5.328980129783192_dp,&
            8.21786366292428_dp,  12.786862454319088_dp,  20.0807884362153_dp,  31.842977578575024_dp,  51.017784629932365_dp,  82.64330507271109_dp,  135.4622543365706_dp,&
            224.87803543778475_dp,  378.47920097589025_dp,  646.570836391214_dp,  1122.6914600829598_dp,  1984.5495359592944_dp,  3577.888523900694_dp,  6593.427321251175_dp,&
            12452.706002367568_dp,  24181.569964546125_dp,  48474.222324463255_dp,  100819.77620242331_dp,  219012.9151368635_dp,  501508.5608275271_dp,  1228961.4319415095_dp,  3359678.97519788_dp,  3198823.17240385_dp]
       aw%aw_erange_matrix(:, 18) = [0.1860758083606257_dp,  0.5863105731285033_dp,  1.0762838189916888_dp,  1.735025194760996_dp,  2.6741091733730666_dp,  4.059774820324427_dp,&
            6.147499799667448_dp,  9.337719533159323_dp,  14.267190355767958_dp,  21.960866171726565_dp,  34.08763874663045_dp,  53.39659047772489_dp,  84.47130014903539_dp,&
            135.05292206821736_dp,  218.3968931937055_dp,  357.54107328410043_dp,  593.1761358820205_dp,  998.4464642388938_dp,  1707.390489925032_dp,  2970.9030847374506_dp,&
            5269.79573455244_dp,  9550.044752723195_dp,  17728.825622420598_dp,  33825.049900245765_dp,  66597.83794457617_dp,  136026.81652248927_dp,  290225.5203619125_dp,&
            652973.7303844218_dp,  1570277.6349969232_dp,  4120560.3068352453_dp,  12225526.961836666_dp,  44976022.1592941_dp,  0.37674253105568495_dp,  0.43381021502950023_dp,&
            0.5590636422872557_dp,  0.7766587789599726_dp,  1.1286591474829935_dp,  1.68435592509438_dp,  2.5563667570796293_dp,  3.9277652235068583_dp,  6.097912064812507_dp,&
            9.560625610789286_dp,  15.139088266420053_dp,  24.221728239197674_dp,  39.180550580116204_dp,  64.12452706067008_dp,  106.27918892115885_dp,  178.55890322928792_dp,&
            304.458022589578_dp,  527.5532342378145_dp,  930.4049107379313_dp,  1673.1477374753613_dp,  3074.6018027427394_dp,  5788.391058089275_dp,  11199.697804802257_dp,&
            22357.499058820344_dp,  46274.36537417859_dp,  99934.30172443499_dp,  227100.46858913172_dp,  549456.1902431552_dp,  1439654.5954696422_dp,  4195690.360297783_dp,&
            14287820.480068853_dp,  67797329.79665162_dp]
       aw%aw_erange_matrix(:, 19) = [.18894944364096963_dp,  0.5962994473518639_dp,  1.0979679811546372_dp,  1.7777457879798018_dp,  2.755038431602313_dp,  4.2094890725295695_dp,&
            6.419910672165012_dp,  9.827743718777555_dp,  15.14232519057394_dp,  23.518028627832294_dp,  36.85642384608609_dp,  58.3288648866487_dp,  93.29351960290545_dp,&
            150.92965910813132_dp,  247.19917473345893_dp,  410.3110727485798_dp,  691.0051346282855_dp,  1182.327758062675_dp,  2058.578907101761_dp,  3654.0712480702805_dp,&
            6627.183883848432_dp,  12313.689493852524_dp,  23517.234882837332_dp,  46357.167452919886_dp,  94815.65398959102_dp,  202633.14120244567_dp,  456833.45962230937_dp,&
            1101461.096498392_dp,  2900225.249530281_dp,  8637618.748843128_dp,  31160984.654377617_dp,  163046130.3896004_dp,  0.3827101036463066_dp,  0.4425913279562002_dp,&
            0.5744425048624656_dp,  0.804618202373891_dp,  1.179281838121873_dp,  1.7750696801469283_dp,  2.717660210867893_dp,  4.213395860766345_dp,  6.603340508278996_dp,&
            10.45667355729473_dp,  16.73411042964203_dp,  27.07791316760568_dp,  44.334509463025526_dp,  73.51209156253726_dp,  123.56723754128413_dp,  210.80455542346078_dp,&
            365.486536699493_dp,  644.9928757856525_dp,  1160.7129102321292_dp,  2134.627971456765_dp,  4022.319779194768_dp,  7790.4608937563735_dp,  15569.825611679273_dp,&
            32269.164071811185_dp,  69800.37245131386_dp,  158930.03677550773_dp,  385455.2889064046_dp,  1013100.9672579938_dp,  2964362.2649549907_dp,  10115616.277159294_dp,&
            44251495.17814687_dp,  24622278.5292175_dp]
    case(34)
       aw%energy_range(:) = [9649.0_dp, 15161.0_dp, 29986.0_dp, 49196.0_dp, 109833.0_dp, 276208.0_dp, 852991.0_dp]
       aw%aw_erange_matrix(:, 1) = [0.13207515772844727_dp,  0.405991108403864_dp,  0.7100791180957539_dp,  1.0674316993343833_dp,  1.5058443663983385_dp,  2.0602382660689633_dp,&
            2.7757484556313985_dp,  3.711806532052663_dp,  4.947645044945255_dp,  6.589801356244469_dp,  8.782420666111134_dp,  11.721478364771365_dp,  15.674500523350034_dp,&
            21.008017787373205_dp,  28.22593014264274_dp,  38.02331749903241_dp,  51.362195208386105_dp,  69.57856923496088_dp,  94.53432041984598_dp,  128.83359338669658_dp,&
            176.13249956696887_dp,  241.58470670964914_dp,  332.48668514742263_dp,  459.220222708152_dp,  636.6470551378719_dp,  886.216031469228_dp,  1239.261662676278_dp,&
            1742.4871050895674_dp,  2467.9964867366607_dp,  3534.353825869125_dp,  5159.135288744986_dp,  7820.30687542924_dp,  12915.999073999303_dp,  27423.79759942945_dp,&
            0.26576346823795405_dp,  0.2854396551016808_dp,  0.32655197737161484_dp,  0.39274601046287516_dp,  0.48984881388427076_dp,  0.6263944482633245_dp,  0.8144408041785135_dp,&
            1.070757575293203_dp,  1.4185065739762008_dp,  1.889596892023973_dp,  2.527980829463319_dp,  3.3942720864211324_dp,  4.572231822130947_dp,  6.17790432405361_dp,&
            8.372526211327902_dp,  11.380831596076106_dp,  15.517105781570788_dp,  21.222416096494825_dp,  29.118047224727174_dp,  40.08257127813093_dp,  55.363655573112425_dp,&
            76.74148031407067_dp,  106.77011264742791_dp,  149.13984256372788_dp,  209.23574802895516_dp,  295.0380382861162_dp,  418.6824781799572_dp,  599.4725232857328_dp,&
            870.5618605429147_dp,  1296.282804430787_dp,  2025.283801158086_dp,  3491.3059333297665_dp,  7474.317631875756_dp,  28531.55532533201_dp]
       aw%aw_erange_matrix(:, 2) = [0.13599210879419024_dp,  0.41865733842571035_dp,  0.734387462880487_dp,  1.1087078459161832_dp,  1.5726641901039529_dp,  2.165714712843095_dp,&
            2.9394684766946595_dp,  3.962695691901657_dp,  5.328172864333448_dp,  7.162136411063943_dp,  9.637432516710518_dp,  12.991909466518228_dp,  17.55426383461847_dp,&
            23.78051806766095_dp,  32.30571538388694_dp,  44.01747962910964_dp,  60.161119876385705_dp,  82.49044121504559_dp,  113.48508380736385_dp,  156.6651752262542_dp,&
            217.04910217389076_dp,  301.823097180824_dp,  421.326779743335_dp,  590.5151234901978_dp,  831.1508175496598_dp,  1175.1475564541768_dp,  1669.8147612822702_dp,&
            2386.5033267913464_dp,  3436.11940867051_dp,  5000.901252994878_dp,  7412.324148921864_dp,  11389.23639096364_dp,  19014.75702535834_dp,  40656.29906587002_dp,&
            0.2737475207928825_dp,  0.2952786790800227_dp,  0.34039675533342323_dp,  0.4133708800225637_dp,  0.5210568682574819_dp,  0.6735634934463747_dp,  0.885293451950756_dp,&
            1.1764687033967232_dp,  1.575309481781747_dp,  2.1211239192365996_dp,  2.8686862980856427_dp,  3.8944530711305823_dp,  5.305413278430533_dp,  7.2517322156689525_dp,&
            9.944880861346203_dp,  13.683733937475726_dp,  18.89229644939318_dp,  26.1744814467069_dp,  36.39402095230821_dp,  50.791640045991635_dp,  71.15786208341221_dp,&
            100.0896105547862_dp,  141.37465650014659_dp,  200.57507440601208_dp,  285.9310387743719_dp,  409.81030920422364_dp,  591.1755952564811_dp,  860.2110797442753_dp,&
            1268.287932013144_dp,  1913.3787096119436_dp,  3017.958277485201_dp,  5225.5076295652025_dp,  11175.8295539236_dp,  42493.69287150706_dp]
       aw%aw_erange_matrix(:, 3) = [0.14157456993085618_dp,  0.43680814305293825_dp,  0.7695603918544728_dp,  1.1691652838645754_dp,  1.6718472598074612_dp,  2.3244228725405374_dp,&
            3.189182451328711_dp,  4.350562914610808_dp,  5.924439306359134_dp,  8.071200190010469_dp,  11.014281956686672_dp,  15.066594360092068_dp,  20.668390911055305_dp,&
            28.44180291698298_dp,  39.26974002093017_dp,  54.41058484826287_dp,  75.66572466277076_dp,  105.6254757771726_dp,  148.03193881969935_dp,  208.3172512208018_dp,&
            294.4065207981018_dp,  417.9228078083311_dp,  596.0074159115596_dp,  854.090540688523_dp,  1230.1481604663468_dp,  1781.3282413698837_dp,  2594.479625814838_dp,&
            3803.497041610086_dp,  5619.84470555754_dp,  8392.880364891958_dp,  12752.616596160462_dp,  20039.19394160478_dp,  34069.0659733131_dp,  73716.52254207557_dp,&
            0.28514234691822854_dp,  0.30952051330129193_dp,  0.36082397822605955_dp,  0.444362626815503_dp,  0.5687241475224296_dp,  0.7466995085435535_dp,  0.9967403173431321_dp,&
            1.3451190863662297_dp,  1.8290618703769976_dp,  2.50126712171007_dp,  3.4364289749111423_dp,  4.740680380007788_dp,  6.565312747281522_dp,  9.12679076184029_dp,&
            12.73608082512356_dp,  17.84183040178123_dp,  25.094256075252144_dp,  35.44016472753401_dp,  50.26505189600494_dp,  71.60683168542646_dp,  102.47931445154013_dp,&
            147.3651890657808_dp,  212.9734421023194_dp,  309.41511024819323_dp,  452.05559786284186_dp,  664.5034127165376_dp,  983.6387878300005_dp,  1468.7306525598128_dp,&
            2220.152861116728_dp,  3425.255864392015_dp,  5497.180237430633_dp,  9607.611108068526_dp,  20546.39934902344_dp,  77652.80170951472_dp]
       aw%aw_erange_matrix(:, 4) = [0.14537972365704172_dp,  0.44924937732846576_dp,  0.7939069438582139_dp,  1.2115293253508186_dp,  1.742274641402974_dp,  2.438646450121879_dp,&
            3.3713296629150684_dp,  4.637276320033501_dp,  6.371112976811227_dp,  8.761403563453904_dp,  12.07400661126176_dp,  16.685826322050936_dp,  23.133850907384492_dp,&
            32.18677637709266_dp,  44.95016073622998_dp,  63.02161541345043_dp,  88.72107581753308_dp,  125.43437293871429_dp,  178.12881222660047_dp,  254.13153300040636_dp,&
            364.31201181146474_dp,  524.8905530158796_dp,  760.2239712046822_dp,  1107.130506542137_dp,  1621.6666690030702_dp,  2389.8720059435745_dp,  3545.100289215905_dp,&
            5296.792046393856_dp,  7980.859332173258_dp,  12157.208254096504_dp,  18833.87560950116_dp,  30128.29725464392_dp,  51981.56843007888_dp,  113581.24285918285_dp,&
            0.29292046810329686_dp,  0.31938180850986814_dp,  0.3752391615051481_dp,  0.4666272739869208_dp,  0.6035184960523118_dp,  0.8008736626361799_dp,  1.0804648717121432_dp,&
            1.4736003482843745_dp,  2.025118651024602_dp,  2.7992210995810174_dp,  3.8880019577883864_dp,  5.423967095382597_dp,  7.598483671540265_dp,  10.689098259656856_dp,&
            15.100189404799776_dp,  21.423785021626927_dp,  30.53105704736921_dp,  43.71077401338752_dp,  62.88009577250318_dp,  90.90758923336087_dp,  132.11161052090677_dp,&
            193.0349397957077_dp,  283.65875554453504_dp,  419.32384092692257_dp,  623.8101438332048_dp,  934.3678665192813_dp,  1410.2058150916341_dp,  2147.682363184511_dp,&
            3310.5494277589387_dp,  5201.420046514296_dp,  8473.557800314775_dp,  14944.543756287507_dp,  32001.381350595566_dp,  120417.37064242488_dp]
       aw%aw_erange_matrix(:, 5) = [0.15111478683237578_dp,  0.46811081455805653_dp,  0.8311975658154092_dp,  1.277245657117571_dp,  1.8530267811044683_dp,  2.6207793743639676_dp,&
            3.6658104514987024_dp,  5.107246498350614_dp,  7.113506495693816_dp,  9.924804821974032_dp,  13.886131394604528_dp,  19.49590677142994_dp,  27.47820253689281_dp,&
            38.89058150232696_dp,  55.28609346493793_dp,  78.95811810682723_dp,  113.31277487728954_dp,  163.43910700431258_dp,  236.9880966222402_dp,  345.5375832850856_dp,&
            506.7277789266936_dp,  747.6291874168176_dp,  1110.0992865921862_dp,  1659.3801603581437_dp,  2498.036749955731_dp,  3788.815996745633_dp,  5792.691112251481_dp,&
            8933.562233913393_dp,  13912.5117339868_dp,  21925.010151757728_dp,  35138.25619155491_dp,  58037.885439910475_dp,  102878.64576803103_dp,  228949.87062864064_dp,&
            0.30466116249925723_dp,  0.33448999997528_dp,  0.39775748201433164_dp,  0.5020431600763641_dp,  0.6597680435379347_dp,  0.8897794356898496_dp,  1.2198845885842693_dp,&
            1.6906902612632702_dp,  2.361327809416546_dp,  3.317974468669039_dp,  4.686572913471436_dp,  6.651901833209751_dp,  9.486316315170766_dp,  13.593305414380533_dp,&
            19.57390295671377_dp,  28.32858472187177_dp,  41.21465830698257_dp,  60.291072612452794_dp,  88.70200998589976_dp,  131.28259205540442_dp,  195.52312215176198_dp,&
            293.11741299959453_dp,  442.472251954802_dp,  672.8167062941227_dp,  1031.0121854686051_dp,  1593.012339907218_dp,  2483.590841950321_dp,  3911.6669142200035_dp,&
            6238.5231601672385_dp,  10129.751559788456_dp,  16981.74160635827_dp,  30536.056841551963_dp,  65732.37502124789_dp,  245660.71368655932_dp]
       aw%aw_erange_matrix(:, 6) = [0.15704353232796245_dp,  0.4877544479660704_dp,  0.8705376156017336_dp,  1.3476766920864_dp,  1.9737426791979278_dp,  2.8227090946785225_dp,&
            3.997888328321955_dp,  5.646288443196994_dp,  7.979713585250564_dp,  11.306104853980733_dp,  16.076474036543857_dp,  22.955699193585055_dp,  32.930053373920856_dp,&
            47.47165037958756_dp,  68.79170082433475_dp,  100.23337165106483_dp,  146.88580759062535_dp,  216.55141603131435_dp,  321.2823482312382_dp,  479.84253008685556_dp,&
            721.6893163095895_dp,  1093.4758256486132_dp,  1669.780298225661_dp,  2571.0074565627824_dp,  3993.6144689187036_dp,  6261.819958404189_dp,  9917.405887043775_dp,&
            15878.648337580931_dp,  25729.53619615957_dp,  42273.37458242063_dp,  70711.43133488682_dp,  121772.29407575646_dp,  223895.20974520067_dp,  511167.1805753637_dp,&
            0.3168215245682693_dp,  0.35043246766637764_dp,  0.42209228667081694_dp,  0.5411651568675625_dp,  0.7231338255849368_dp,  0.9917838145432007_dp,  1.3827323888424163_dp,&
            1.948858523758149_dp,  2.7685520945936393_dp,  3.958248763359216_dp,  5.69155685455881_dp,  8.22860474914602_dp,  11.961351425605036_dp,  17.48401674923339_dp,&
            25.70334691811683_dp,  38.01255695679674_dp,  56.567920087611036_dp,  84.73230593273999_dp,  127.79279074297057_dp,  194.13262416830213_dp,  297.16429850617567_dp,&
            458.5516775688101_dp,  713.6415082198464_dp,  1120.7309215735056_dp,  1777.0914240859645_dp,  2847.100049696002_dp,  4612.5746467497465_dp,  7565.363105491919_dp,&
            12586.173245720442_dp,  21325.994478121225_dp,  37197.937321555175_dp,  68943.1032010462_dp,  150241.71258102785_dp,  557720.3208697784_dp]
       aw%aw_erange_matrix(:, 7) = [0.16336922513552726_dp,  0.5088837963863347_dp,  0.9134464251208241_dp,  1.4258068793865757_dp,  2.110076760757019_dp,  3.0549223319570658_dp,&
            4.386714080773768_dp,  6.288970678299134_dp,  9.031595056512524_dp,  13.015314763001339_dp,  18.83983365553746_dp,  27.40919724979278_dp,  40.09598751880866_dp,&
            58.99925287825027_dp,  87.35305062053737_dp,  130.1791624944713_dp,  195.3394286468929_dp,  295.2486883643026_dp,  449.6914179453214_dp,  690.5032975197853_dp,&
            1069.4420394766496_dp,  1671.582707097392_dp,  2638.4152977365293_dp,  4208.234704650762_dp,  6787.845837025887_dp,  11081.975915584215_dp,  18331.11464078846_dp,&
            30757.7543831196_dp,  52425.18373362508_dp,  90952.31947001838_dp,  161183.38574164535_dp,  294404.0420740371_dp,  571514.7455391894_dp,  1357785.7335867172_dp,&
            0.3298231494705116_dp,  0.3678238583365475_dp,  0.44931443538568944_dp,  0.5859423647480262_dp,  0.7971613283485441_dp,  1.1132755348536536_dp,  1.5804195782403758_dp,&
            2.26835630112171_dp,  3.2825754864813064_dp,  4.7831079906178315_dp,  7.01394730221803_dp,  10.349374600517876_dp,  15.367451600728_dp,  22.967563824366035_dp,&
            34.56004919104946_dp,  52.374936525753725_dp,  79.96951316778983_dp,  123.07138037510275_dp,  190.99409829094074_dp,  299.04206165572833_dp,  472.6466080518463_dp,&
            754.5744583501951_dp,  1217.6699793620016_dp,  1987.724673849413_dp,  3285.207068806235_dp,  5502.805705516988_dp,  9352.508394580818_dp,  16151.576514128332_dp,&
            28398.10726253618_dp,  51001.74394184054_dp,  94285.04834791286_dp,  183730.02847568074_dp,  411506.64532360405_dp,  1520426.766419325_dp]
       aw%aw_erange_matrix(:, 8) = [0.1835103101234003_dp,  0.5774305061474206_dp,  1.057140472868453_dp,  1.697614005082806_dp,  2.6038279115366483_dp,  3.930849397069668_dp,&
            5.914914021102437_dp,  8.92301131893533_dp,  13.533362340789761_dp,  20.667776233995074_dp,  31.812016357593226_dp,  49.387219863328404_dp,  77.3839092638016_dp,&
            122.4588606715459_dp,  195.86011110608575_dp,  316.8577691536953_dp,  518.9611571628808_dp,  861.3906597021048_dp,  1450.6667756885272_dp,  2482.126411324662_dp,&
            4321.685991308644_dp,  7671.18540800507_dp,  13912.797854700362_dp,  25850.79390530169_dp,  49371.03198418239_dp,  97320.21724486594_dp,  199052.03066138524_dp,&
            425399.7542270562_dp,  959057.1525395216_dp,  2312357.9736451125_dp,  6088604.151380904_dp,  18133512.159556706_dp,  65418675.293680795_dp,  342296590.6151682_dp,&
            0.37142071027541074_dp,  0.4260563273244676_dp,  0.5456348676942346_dp,  0.752484593582803_dp,  1.0852868522175914_dp,  1.6073303748410037_dp,  2.4206700397850627_dp,&
            3.6897704463837764_dp,  5.681035810947625_dp,  8.82947962399161_dp,  13.852405454162984_dp,  21.945623429966453_dp,  35.126642589694754_dp,  56.843590502572916_dp,&
            93.0724205769061_dp,  154.32725344417676_dp,  259.4111633804356_dp,  442.5533297274132_dp,  767.2854489636625_dp,  1354.0674261637732_dp,  2436.7445344093326_dp,&
            4481.335149019925_dp,  8444.272788875625_dp,  16354.953153653125_dp,  32686.61977168015_dp,  67744.54837929386_dp,  146536.159793689_dp,  333651.2446730961_dp,&
            809207.8445405798_dp,  2126855.0754410354_dp,  6223266.011027118_dp,  21236461.045184538_dp,  92901077.9747871_dp,  681507114.6255051_dp]
    end select

  end subroutine set_aw_array_omega

  !> \brief Unpacks the minimax coefficients for the desired energy range
  !! @param[in] grid_size - size of the grid
  !! @param[in] e_range - the selected energy range
  !! @param[inout] ac_we - vector containing coefficients and weights
  !! @param[out] ierr - error code
  subroutine get_points_weights_omega(grid_size, e_range, ac_we, ierr)
    integer, intent(in)                                    :: grid_size
    real(kind=dp), intent(in)                              :: e_range
    real(kind=dp), dimension(2* grid_size), intent(inout)  :: ac_we
    integer, intent(out)                                   :: ierr

    !> Internal variables
    integer                                                :: kloc, bup
    type(er_aw_aux)                                        :: aw
    real(kind=dp)                                          :: e_ratio

    !> Begin work
    ierr = 0
    if (.not. any(omega_npoints_supported == grid_size)) then
       ierr = 1
       _REGISTER_EXC("The grid size you chose is not available.")
       return
    end if

    ! Find location of grid size
    kloc = findloc(omega_npoints_supported, grid_size, 1)
    bup = energy_ranges_grids(kloc)

    ! Allocate and set type elements
    allocate(aw%energy_range(bup))
    allocate(aw%aw_erange_matrix(2*grid_size, bup+1))
    call set_aw_array_omega(grid_size, aw)

    ! Get coefficients and weights
    e_ratio = 1.0_dp
    call aw%get_coeff_weight(grid_size, bup, e_range, ac_we, e_ratio)
    ac_we(:) = ac_we(:) / e_ratio

    ! Deallocate
    deallocate(aw%energy_range)
    deallocate(aw%aw_erange_matrix)

  end subroutine get_points_weights_omega

end module minimax_omega
