/*
 * Copyright (c) 2020 ETH Zurich, Simon Frasch
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holder nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
#ifndef SPLA_BLOCK_HPP
#define SPLA_BLOCK_HPP

#include "spla/config.h"
#include "spla/types.h"
#include "util/common_types.hpp"

namespace spla {

template <typename T>
struct IsDisjointGenerator {
  static constexpr bool value = false;
};

struct BlockInfo {
  IntType globalRowIdx, globalColIdx;  // Indices of first element in block in global matrix
  IntType globalSubRowIdx,
      globalSubColIdx;  // Indices of first element in block in global matrix without offset
  IntType localRowIdx, localColIdx;  // Indices of first element in block on assigned mpi rank
  IntType numRows, numCols;          // Size of block
  IntType mpiRank;                   // negative value indicates mirrored on all ranks
};

struct Block {
  IntType row;
  IntType col;
  IntType numRows;
  IntType numCols;
};

inline auto block_is_active(const Block &block, IntType rowOffset, IntType colOffset,
                            SplaFillMode mode) -> bool {
  if (mode == SplaFillMode::SPLA_FILL_MODE_UPPER) {
    return block.col + block.numCols + colOffset > block.row + rowOffset;
  } else if (mode == SplaFillMode::SPLA_FILL_MODE_LOWER) {
    return block.row + block.numRows + rowOffset > block.col + colOffset;
  }
  return true;
}

}  // namespace spla
#endif
