#include <sirius.hpp>

using namespace sirius;
using namespace sddk;

/* reference values are generated with the following Mathematica script:

(* definition of real spherical harmonics, use Plm(l,m) for m >= 0 only *)

norm[l_, m_] :=
 Sqrt[4*Pi*
   Integrate[
    LegendreP[l, Abs[m], x]*LegendreP[l, Abs[m], x], {x, 0, 1}]]
Plm[l_, m_, x_] := LegendreP[l, Abs[m], x]/norm[l, m]

(* reference definition *)

RRlm[l_, m_, th_, ph_] :=
 If[m > 0, Sqrt[2]*ComplexExpand[Re[SphericalHarmonicY[l, m, th, ph]]
    ], If[m < 0,
   Sqrt[2]*ComplexExpand[Im[SphericalHarmonicY[l, m, th, ph]]],
   If[m == 0, ComplexExpand[Re[SphericalHarmonicY[l, 0, th, ph]]]]]]

(* definition without ComplexExpand *)

Rlm[l_, m_, th_, ph_] :=
 If[m > 0, Plm[l, m, Cos[th]]*Sqrt[2]*Cos[m*ph],
  If[m < 0, (-1)^m*Plm[l, m, Cos[th]]*Sqrt[2]*(-Sin[Abs[m]*ph]),
   If[m == 0, Plm[l, 0, Cos[th]]]]]

(* check that both definitions are identical *)
Do[
 Print[FullSimplify[Rlm[l, m, a, b] - RRlm[l, m, a, b],
   Assumptions -> {0 <= a <= Pi, 0 <= b <= 2*Pi}]], {l, 0, 5}, {m, -l,
   l}]

(* r,theta, phi as functions of x,y,z *)

r[x_, y_, z_] := Sqrt[x*x + y*y + z*z]
t[x_, y_, z_] := ArcCos[z/r[x, y, z]]
p[x_, y_, z_] := ArcTan[x, y]

coord = {{-1, 0, 0}, {1, 0, 0}, {0, -1, 0}, {0, 1, 0}, {0, 0, -1}, {0,
     0, 1}};
lmax = 8;
lm = {};
Do[AppendTo[lm, {l, m}], {l, 0, lmax}, {m, -l, l}]
Table[
  Limit[
   Limit[
    Limit[
     N[
      D[Rlm[lm[[i]][[1]], lm[[i]][[2]], t[x, y, z],
        p[x, y, z]], {{x, y, z}}]
      ],
     x -> coord[[v]][[1]]],
    y -> coord[[v]][[2]]],
   z -> coord[[v]][[3]]],
  {i, 1, (lmax + 1)^2}, {v, 1, 6}] // Chop


*/

int test1()
{
    std::vector<vector3d<double>> v({{1, 2, 3}, {-2, 2, 0.12}, {2, -1, 0}, {0, 1, -5}, {-4, -4, -2}, {-2, 0, 1}});

    int lmax = 8;
    int lmmax = utils::lmmax(lmax);

    mdarray<double, 2> data(lmmax, 3);

    double ref[][6][3] ={{{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
   {{0.018654930590386358, -0.09327465295193178, 0.05596479177115907}, {-0.08614085243649676, -0.0864509595052682, 
     0.005168451146189806}, {-0.08740387444736633, -0.17480774889473266, 0}, {0, -0.09213734255116271, -0.018427468510232543}, 
    {0.03619277865947554, -0.04524097332434443, 0.018096389329737773}, {0, -0.21850968611841579, 0}}, 
   {{-0.027982395885579534, -0.05596479177115907, 0.04663732647596589}, {0.005168451146189807, -0.005168451146189807, 
     0.17228170487299357}, {0, 0, 0.2185096861184158}, {0, 0.018427468510232543, 0.0036854937020465092}, 
    {-0.018096389329737773, -0.018096389329737773, 0.07238555731895109}, {0.08740387444736633, 0, 0.17480774889473266}}, 
   {{-0.12125704883751132, 0.018654930590386358, 0.027982395885579534}, {-0.08645095950526815, -0.08614085243649679, 
     -0.005168451146189806}, {-0.04370193722368317, -0.08740387444736633, 0}, {-0.09582283625320923, 0, 0}, 
    {-0.045240973324344416, 0.036192778659475545, 0.018096389329737773}, {-0.043701937223683165, 0, -0.08740387444736633}}, 
   {{-0.1337814404806628, -0.03344536012016567, 0.06689072024033137}, {-0.0004898816326723754, 0.0004898816326722752, 
     -0.01632938775574418}, {-0.1311058116710495, -0.262211623342099, 0}, {-0.04202109348431074, 0, 0}, 
    {0.013488252229531863, 0.01348825222953182, -0.05395300891812736}, {0, 0.4370193722368317, 0}}, 
   {{0.0668907202403314, -0.1003360803604971, 0.044593813493554235}, {-0.01632938775574418, -0.000029392897960339528, 
     -0.271666580963064}, {0, 0, 0.21850968611841584}, {0, 0.1939435083891265, 0.0387887016778253}, 
    {-0.05395300891812738, 0.006744126114765919, 0.09441776560672288}, {0, -0.21850968611841584, 0}}, 
   {{-0.08689359450834735, -0.1737871890166947, 0.1448226575139123}, {0.000848499877483283, -0.000848499877483283, 
     0.02828332924944204}, {0, 0, 0}, {0, -0.06998333548502664, -0.013996667097005339}, 
    {0.023362338166853326, 0.023362338166853326, -0.09344935266741333}, {0.15138795132120964, 0, 0.30277590264241927}}, 
   {{-0.20067216072099414, 0.06689072024033138, 0.022296906746777118}, {-0.000029392897960339524, -0.016329387755744176, 
     0.271666580963064}, {0, 0, -0.4370193722368317}, {0.2101054674215537, 0, 0}, 
    {0.006744126114765923, -0.053953008918127356, 0.09441776560672288}, {0.1311058116710495, 0, 0.262211623342099}}, 
   {{0.09476185367380276, -0.12263298710727419, 0.05016804018024854}, {-0.27264634422840867, -0.27264634422840867, 0}, 
    {0.17480774889473266, 0.3496154977894653, 0}, {0, -0.040404897581068015, -0.008080979516213603}, 
    {-0.12139427006578656, 0.12139427006578656, 0}, {-0.08740387444736633, 0, -0.17480774889473266}}, 
   {{-0.1399952355967818, 0.09172101642547784, -0.014482265751391271}, {0.3126356119635676, 0.3115141609840251, 
     0.018690849659041497}, {-0.06333012368467125, -0.1266602473693425, 0}, {0, 0.012838432147130865, 0.002567686429426173}, 
    {-0.14568977529052926, 0.11655182023242343, 0.05827591011621169}, {0, -0.6333012368467128, 0}}, 
   {{-0.2601438503421481, -0.023649440940195287, 0.1024809107408462}, {0.015206075187813067, -0.01520607518781308, 
     0.5068691729271024}, {0, 0, 0.5170882945826409}, {0.10901837390709153, 0, 0}, 
    {0.035686561020253735, 0.035686561020253776, -0.142746244081015}, {0, 0.517088294582641, 0}}, 
   {{0.1508186325006462, 0.03116087448360464, -0.07104679382261853}, {0.07840570872182752, 0.08158875528462432, 
     -0.05305077604661299}, {0.08175883811466263, 0.16351767622932523, 0}, {0, -0.2950233344570272, -0.059004666891405454}, 
    {0.022570162936516824, 0.056425407341292075, -0.1579911405556178}, {0, 0, 0}}, 
   {{-0.14197053090816286, -0.2839410618163257, 0.23661755151360483}, {-0.011736019120160595, 0.011736019120160595, 
     -0.39120063733868654}, {0, 0, -0.5006685883593409}, {0, 0.16077080119501963, 0.03215416023900392}, 
    {0.018428460868647672, 0.018428460868647672, -0.07371384347459069}, {0, 0, 0}}, 
   {{-0.19506707426736475, 0.15081863250064623, -0.03552339691130922}, {0.08158875528462431, 0.07840570872182753, 
     0.05305077604661299}, {0.04087941905733133, 0.08175883811466263, 0}, {-0.3412989046707362, 0, 0}, 
    {0.05642540734129207, 0.022570162936516827, -0.1579911405556178}, {0.32703535245865045, 0, 0.6540707049173008}}, 
   {{0.21875732869680628, -0.22466968893185513, 0.07686068305563466}, {-0.030576969335911554, -0.030576969335911548, 0}, 
    {0, 0, 0.3878162209369807}, {0, 0.10272885233552854, 0.02054577046710571}, {0.10705968306076123, -0.10705968306076125, 0}, 
    {0.10341765891652821, 0, 0.20683531783305642}}, {{0.07482503971552143, 0.08206617259121686, -0.07965246163265173}, 
    {-0.311514160984025, -0.3126356119635675, 0.018690849659041508}, {-0.34831568026569215, -0.6966313605313843, 0}, 
    {0.0133519694330161, 0, 0}, {-0.11655182023242337, 0.1456897752905292, -0.05827591011621172}, 
    {-0.12666024736934256, 0, -0.2533204947386851}}, {{0.0036491879618028513, 0.09670348098777354, -0.06568538331244998}, 
    {-0.6235887953581897, -0.6235887953581897, 0}, {0.14018720474061552, 0.28037440948123105, 0}, 
    {0.0037031700322436457, 0, 0}, {0.2472437473379461, -0.24724374733794613, 0}, {0, 0.8010697413749455, 0}}, 
   {{-0.3406082530771876, 0.21288015817324252, -0.02838402108976575}, {0.02655170469627208, 0.05281819116027866, 
     -0.43777477440010926}, {0, 0, 0.7788575387031695}, {0, -0.03726379067015102, -0.007452758134030204}, 
    {0.10683916854638816, -0.15540242697656462, 0.09712651686035277}, {0, -0.8496627694943667, 0}}, 
   {{-0.31860984653060703, 0.07241132875695616, 0.057929063005564885}, {0.0033833276208770995, -0.0033833276208769802, 
     0.11277758736256854}, {0.1135409634909072, 0.22708192698181442, 0}, {-0.20855033974537937, 0, 0}, 
    {-0.07527864520430519, -0.07527864520430518, 0.30111458081722076}, {0, 0.1513879513212096, 0}}, 
   {{0.2457721996741063, 0.2764937246333695, -0.26625321631361526}, {0.029747462022899764, 0.00017954123525413702, 
     0.49279867979409414}, {0, 0, -0.4014279261343736}, {0, 0.37647191145776, 0.07529438229155205}, 
    {0.04772348458844862, -0.03487485412232783, -0.025697260932241547}, {0, 0.21409489393833256, 0}}, 
   {{-0.1457249880974753, -0.2914499761949506, 0.24287498016245884}, {-0.002834023572740485, 0.0028340235727404295, 
     -0.09446745242468199}, {0, 0, 0}, {0, -0.2919093664876769, -0.0583818732975353}, 
    {-0.05804419583824653, -0.05804419583824653, 0.23217678335298614}, {-0.270811000102923, 0, -0.541622000205846}}, 
   {{-0.09216457487778984, 0.24577219967410632, -0.13312660815680763}, {0.00017954123525413697, 0.02974746202289977, 
     -0.49279867979409414}, {0, 0, 0.8028558522687472}, {0.48001120358770005, 0, 0}, 
    {-0.03487485412232782, 0.04772348458844862, -0.025697260932241547}, {0.17127591515066604, 0, 0.34255183030133207}}, 
   {{0.35240179995051973, -0.24137109585652042, 0.043446797254173664}, {0.23314891077956723, 0.2331489107795672, 0}, 
    {-0.1513879513212096, -0.3027759026424192, 0}, {0, -0.1911083392091112, -0.03822166784182224}, 
    {0.023362338166853316, -0.02336233816685335, 0}, {0.393608673435145, 0, 0.78721734687029}}, 
   {{0.15869248154732635, 0.15482193321690346, -0.15611211599371105}, {-0.05281819116027864, -0.026551704696272053, 
     -0.43777477440010965}, {0, 0, -0.14161046158239435}, {-0.03927804962529431, 0, 0}, 
    {0.1554024269765645, -0.10683916854638809, -0.09712651686035285}, {0.05664418463295776, 0, 0.11328836926591553}}, 
   {{-0.134107657596252, 0.03831647359892921, 0.01915823679946454}, {0.0011204430341834406, -0.0011204430341830163, 
     0.03734810113944039}, {0.48064184482496725, 0.9612836896499345, 0}, {0, 0.00356074041561889, 0.0007121480831237781}, 
    {0.027471527481994107, 0.027471527481993913, -0.10988610992797605}, {-0.1602139482749891, 0, -0.3204278965499782}}, 
   {{0.09525654908287237, 0.007032362683970412, -0.03644042481693773}, {0.5765249674083147, 0.5786004572909845, 
     -0.034591498044498965}, {0.4461853290849484, 0.8923706581698969, 0}, {0, -0.0009155034717906946, 
     -0.00018310069435813894}, {-0.19208254063909835, 0.2401031757988731, -0.09604127031954948}, {0, -0.9393375349156808, 0}}, 
   {{-0.004851989108448127, 0.22804348809707106, -0.15041166236189807}, {-0.08766785580525867, -0.08766785580525867, 0}, 
    {0, 0, 0.8911338306064477}, {-0.012043516894865273, 0, 0}, {-0.273338247227133, 0.27333824722713307, 0}, 
    {0, 1.1881784408085971, 0}}, {{-0.5709546815376395, 0.3330807230802957, -0.0317355882076506}, 
    {-0.2578164376731516, -0.2513324925202005, -0.10806575254918624}, {0.052510564547213666, 0.10502112909442715, 0}, 
    {0, 0.0790191406672567, 0.015803828133451336}, {0.06442644272398355, 0.06442644272398357, -0.2577057708959343}, 
    {0, -0.420084516377709, 0}}, {{-0.24931545239023173, 0.26612323569743823, -0.09431033966821496}, 
    {-0.02480766871478883, 0.02480766871478886, -0.8269222904929615}, {0, 0, -0.8574939283076917}, {0.3407130583407733, 0, 0}, 
    {0.01315099255136719, 0.013150992551367088, -0.05260397020546856}, {0, -0.3429975713230767, 0}}, 
   {{0.3007848273498754, 0.5194251330621052, -0.44654503115802857}, {-0.07385560769619406, -0.08212745397860433, 
     0.13786410470683833}, {-0.08102556018035718, -0.16205112036071445, 0}, {0, -0.4206240871341129, -0.08412481742682264}, 
    {-0.07952972750761206, -0.05716199164609616, 0.2733834383074165}, {0, 0.19446134443285723, 0}}, 
   {{-0.06817379234673782, -0.13634758469347572, 0.11362298724456285}, {0.018091030150437403, -0.018091030150437403, 
     0.6030343383479129}, {0, 0, 0.7845266129891254}, {0, 0.46009362974826284, 0.09201872594965242}, 
    {0.019251081885337168, 0.019251081885337168, -0.07700432754134887}, {-0.30125821938782393, 0, -0.6025164387756484}}, 
   {{0.06824789203729204, 0.3007848273498754, -0.22327251557901429}, {-0.08212745397860438, -0.07385560769619415, 
     -0.13786410470683833}, {-0.04051278009017864, -0.08102556018035713, 0}, {-0.6177374547798631, 0, 0}, 
    {-0.057161991646096186, -0.07952972750761206, 0.2733834383074165}, {-0.32410224072142885, 0, -0.6482044814428574}}, 
   {{0.45030852777224417, -0.11905513175938026, -0.07073275475116106}, {0.050432845931909, 0.050432845931909, 0}, 
    {0, 0, -0.6431204462307688}, {0, 0.30099885962522793, 0.06019977192504558}, {-0.11835893296230426, 0.11835893296230429, 
     0}, {0.3429975713230767, 0, 0.6859951426461532}}, {{0.21199944735110696, 0.15581887903756297, -0.1745457351420776}, 
    {0.2513324925202004, 0.2578164376731517, -0.1080657525491863}, {0.2888081050096751, 0.5776162100193499, 0}, 
    {0.08473472203365516, 0, 0}, {-0.06442644272398357, -0.06442644272398364, 0.25770577089593444}, 
    {0.42008451637770905, 0, 0.8401690327554181}}, {{-0.3523757090010594, 0.11038275221719948, 0.04387006818888684}, 
    {-0.02172006569261845, 0.021720065692618525, -0.7240021897539496}, {0, 0, -0.2599140339268807}, 
    {0, -0.011464501659535215, -0.0022929003319070427}, {0.03037091635857022, 0.030370916358570456, -0.12148366543428134}, 
    {0, 0, 0}}, {{0.0444317460487223, -0.0811918237149315, 0.039317300460380226}, 
    {0.5786004572909862, 0.576524967408316, 0.034591498044498875}, {-0.48141048664428643, -0.9628209732885729, 0}, 
    {-0.0009521236106623225, 0, 0}, {0.2401031757988736, -0.19208254063909902, -0.09604127031954922}, 
    {-0.1878675069831362, 0, -0.3757350139662724}}, {{0.06146096073363322, -0.05185768561900311, 0.01408480350145767}, 
    {0.0018313755354815164, -0.0018313755354801842, 0.061045851182691444}, {0.7673523869515583, 1.5347047739031168, 0}, 
    {-0.00023322170181790426, 0, 0}, {-0.03998516939394854, -0.03998516939394808, 0.15994067757579336}, 
    {0, 1.0493707855747803, 0}}, {{0.268351548070635, 0.006283713935538142, -0.09363965864723717}, 
    {0.10573451583971016, 0.07080688077399247, 0.5821272510952946}, {0, 0, 0.7762510852120147}, 
    {0, 0.003210899216991016, 0.0006421798433982034}, {0.27702537975826597, -0.24239720728848269, -0.0692563449395667}, 
    {0, -1.5146362638283215, 0}}, {{-0.044551336507492156, 0.3485301490456427, -0.21750298719459804}, 
    {0.49281672912561447, 0.4928167291256146, 0}, {-0.11302253776323412, -0.22604507552646863, 0}, 
    {0.028592776577481185, 0, 0}, {0.044296507059860595, -0.04429650705986052, 0}, {0, 0.7750116875193203, 0}}, 
   {{-0.7501818340968572, 0.3940828622154376, -0.012661296778006065}, {-0.04145236883136774, -0.08164746385020885, 
     0.6699182503140181}, {0, 0, -1.2159909425596995}, {0, -0.14139328302225226, -0.02827865660445045}, 
    {-0.14826897638283176, 0.094352984970893, 0.10783198282387763}, {0, 0.35374281965373067, 0}}, 
   {{-0.050645187112024204, 0.5040059387881263, -0.31912223015474295}, {-0.007772458201043442, 0.007772458201043414, 
     -0.25908194003477336}, {-0.11054463114179097, -0.22108926228358167, 0}, {-0.503214138404801, 0, 0}, 
    {0.1010924838973853, 0.10109248389738533, -0.40436993558954115}, {0, -0.4716570928716408, 0}}, 
   {{0.26188842378106675, 0.6046728895153537, -0.490411400937258}, {-0.04260373313755643, -0.0005450569146093035, 
     -0.7009779370491195}, {0, 0, 0.5826213625187315}, {0, 0.41193621159492433, 0.08238724231898487}, 
    {0.012787300137585366, 0.0799206258599083, -0.1854158519949872}, {0, -0.037287767201198814, 0}}, 
   {{0.07943833621563337, 0.15887667243126674, -0.1323972270260554}, {0.005921319521655366, -0.005921319521655366, 
     0.19737731738850772}, {0, 0, 0}, {0, -0.6580175305189975, -0.13160350610379945}, 
    {0.068365232617038, 0.068365232617038, -0.2734609304681519}, {0.06834960627815412, 0, 0.13669921255630813}}, 
   {{0.21184025384375368, 0.2618884237810667, -0.245205700468629}, {-0.0005450569146092878, -0.042603733137556415, 
     0.7009779370491195}, {0, 0, -1.165242725037463}, {0.7466742256903975, 0, 0}, {0.07992062585990825, 0.01278730013758538, 
     -0.1854158519949872}, {-0.6040618286594208, 0, -1.2081236573188419}}, 
   {{0.46642970384279414, 0.12579765700268875, -0.23934167261605732}, {-0.2224770360613787, -0.22247703606137872, 0}, 
    {0.14739284152238774, 0.2947856830447755, 0}, {0, -0.4242907220431785, -0.08485814440863569}, 
    {0.06065549033843119, -0.060655490338431164, 0}, {-0.18866283714865623, 0, -0.3773256742973128}}, 
   {{0.17956020885172233, 0.014675593992688562, -0.06963713227903312}, {0.08164746385020881, 0.041452368831367756, 
     0.6699182503140184}, {0, 0, 0.22108926228358156}, {-0.15487946869973826, 0, 0}, 
    {-0.09435298497089292, 0.14826897638283174, -0.10783198282387768}, {0.49523994751522293, 0, 0.9904798950304456}}, 
   {{-0.638245178727916, 0.2239650324663217, 0.06343837126509094}, {-0.005844864649685866, 0.005844864649685422, 
     -0.19482882165618587}, {-0.3875058437596602, -0.7750116875193203, 0}, {0, 0.026885778099384725, 0.005377155619876945}, 
    {-0.10335851647300795, -0.10335851647300798, 0.41343406589203185}, {0.41333956667697086, 0, 0.8266791333539417}}, 
   {{0.13602392519282866, -0.2195603575123377, 0.10103226327728226}, {0.0708068807739928, 0.10573451583971034, 
     -0.582127251095293}, {0, 0, 0.7194522253184529}, {0.0033662653081357433, 0, 0}, 
    {-0.24239720728848344, 0.2770253797582668, -0.0692563449395665}, {-0.06058545055313288, 0, -0.12117090110626576}}, 
   {{0.04876329697095564, 0.031797510935108825, -0.03745277294705776}, {1.0192622285803383, 1.0192622285803383, 0}, 
    {0.2885769660330644, 0.5771539320661291, 0}, {0, -0.00022425163636336952, -0.00004485032727267389}, 
    {0.35986652454553497, -0.3598665245455351, 0}, {-0.20987415711495605, 0, -0.4197483142299121}}, 
   {{-0.011640133065507175, -0.037262201115025706, 0.02872151176518619}, {-0.871140983011134, -0.8680161249612732, 
     -0.05208096749767642}, {0.9846846813890253, 1.9693693627780506, 0}, {0, 0.00005311012799990418, 0.000010622025599980836}, 
    {0.32191119663772705, -0.2575289573101816, -0.12876447865509086}, {0, -1.1334499929657846, 0}}, 
   {{0.193284274560294, -0.1561736938447178, 0.03968770437638053}, {-0.027537792867305046, 0.02753779286730523, 
     -0.9179264289101711}, {0, 0, 0.41652497255278825}, {0.0008857229950846502, 0, 0}, 
    {-0.017206877239782474, -0.01720687723978299, 0.06882750895913084}, {0, 1.8175635165939856, 0}}, 
   {{0.5219426200306858, -0.023299656031194754, -0.15844776932276555}, {-0.4408783832780276, -0.45099780333466244, 
     0.16865700094392216}, {-0.35274047461505253, -0.7054809492301048, 0}, {0, -0.008178590199037114, -0.0016357180398074232}, 
    {-0.15522912805491515, -0.003374546262063524, 0.3172073486339573}, {0, -1.1881784408085971, 0}}, 
   {{-0.13464269775944043, 0.36753817496496005, -0.20014455072349313}, {0.13047791276266785, 0.13047791276266787, 0}, 
    {0, 0, -1.3367007459096718}, {-0.05720670525061005, 0, 0}, {0.2125964145099923, -0.21259641450999245, 0}, 
    {0, -0.23763568816171943, 0}}, {{-0.7947823086559406, 0.3490458974166294, 0.03223017127422717}, 
    {0.24538708008355514, 0.23157103595225198, 0.2302674021883817}, {-0.050378804899665786, -0.10075760979933092, 0}, 
    {0, 0.22601519509201237, 0.04520303901840249}, {0.020858011606699117, -0.15770691702626197, 0.2736978108391254}, 
    {0, 0.7523234865016727, 0}}, {{0.20908771178571048, 0.6753649466028249, -0.5199392016637868}, 
    {0.0335409292821347, -0.03354092928213467, 1.1180309760711558}, {0, 0, 1.1874398190875866}, {0.6916705319266754, 0, 0}, 
    {-0.10180297878325863, -0.10180297878325882, 0.4072119151330348}, {0, -0.1013281978954741, 0}}, 
   {{0.10461886243298435, 0.4218803357601576, -0.31612651131776676}, {0.06816566187463455, 0.0836871619372889, 
     -0.2586916677109073}, {0.08079504602853764, 0.16159009205707536, 0}, {0, -0.33797359005844685, -0.06759471801168937}, 
    {0.09075841444657133, 0.023350465852758687, -0.22821776059865967}, {0, -0.227518849616362, 0}}, 
   {{0.24111405709778477, 0.48222811419556927, -0.40185676182964153}, {-0.024062575824870657, 0.02406257582487067, 
     -0.8020858608290221}, {0, 0, -1.0688179947876377}, {0, 0.8748934775409316, 0.1749786955081865}, 
    {-0.07926429705255747, -0.07926429705255743, 0.31705718821022993}, {0.48156663573151776, 0, 0.9631332714630357}}, 
   {{0.2649520421106811, 0.10461886243298449, -0.15806325565888338}, {0.08368716193728899, 0.06816566187463471, 
     0.2586916677109073}, {0.04039752301426882, 0.08079504602853782, 0}, {-0.8597644956627176, 0, 0}, 
    {0.02335046585275866, 0.09075841444657128, -0.22821776059865967}, {-0.1944251987630724, 0, -0.3888503975261461}}, 
   {{0.37114038636545243, 0.39936140868903364, -0.3899544012478401}, {-0.0692938134696695, -0.06929381346966947, 0}, 
    {0, 0, 0.8905798643156897}, {0, 0.5499231308406849, 0.10998462616813702}, {0.07446366292627049, -0.07446366292627049, 0}, 
    {-0.6890317456892235, 0, -1.378063491378447}}, {{0.03463029041166999, -0.2832140582182086, 0.17726594200824874}, 
    {-0.231571035952252, -0.24538708008355467, 0.23026740218838182}, {-0.2770834269481606, -0.5541668538963216, 0}, 
    {0.2540291205765657, 0, 0}, {0.1577069170262619, -0.02085801160669911, -0.2736978108391256}, 
    {-0.007164985585730252, 0, -0.014329971171460502}}, {{-0.9083833358973094, 0.36662842700712667, 0.05837549396101882}, 
    {0.03181541472857263, -0.0318154147285727, 1.0605138242857555}, {0, 0, 0.38987105089032115}, 
    {0, -0.0530088447944671, -0.010601768958893422}, {0.06411637897920408, 0.06411637897920405, -0.25646551591681616}, 
    {0.6178527892204705, 0, 1.235705578440941}}, {{0.2930651963055997, -0.4029678037672758, 0.17095680374298397}, 
    {-0.45099780333466416, -0.4408783832780288, -0.16865700094392172}, {0.38058840682150397, 0.7611768136430078, 0}, 
    {-0.008656277768184416, 0, 0}, {-0.0033745462620633003, -0.15522912805491512, 0.3172073486339565}, 
    {0.3802171010587513, 0, 0.7604342021175025}}, {{0.14496320592022083, 0.08581821790477041, -0.10553321390992058}, 
    {0.16733134388070936, 0.16733134388070933, 0}, {0, 0, -1.1075777679244603}, 
    {0, 0.00084597901453598, 0.000169195802907196}, {-0.4645856854741334, 0.4645856854741333, 0}, 
    {-0.12117090110626566, 0, -0.2423418022125315}}, {{-0.055411165983434735, 0.023211389603999422, 0.002996128925145289}, 
    {0.8680161249612736, 0.8711409830111341, -0.0520809674976764}, {0.10271890561252443, 0.20543781122504895, 0}, 
    {0.000055234533119900344, 0, 0}, {0.2575289573101817, -0.32191119663772716, 0.12876447865509083}, 
    {-0.22668999859315703, 0, -0.45337999718631405}}, {{-0.03855630397426216, 0.0028841329744527123, 0.010929346008452242}, 
    {1.44740384138081, 1.44740384138081, 0}, {0.9834095357750664, 1.9668190715501324, 0}, {0.000012760874272168864, 0, 0}, 
    {-0.45506532541931155, 0.45506532541931155, 0}, {0, 1.1942734400304413, 0}}, 
   {{-0.033959039383405436, -0.11414010459422307, 0.08741308285728389}, {-0.1305784232145287, -0.17337638347544146, 
     0.713299337681879}, {0, 0, -0.13528878812844872}, {0, -0.0002134992426305176, -0.00004269984852610352}, 
    {-0.3918618079999626, 0.4045025114838325, -0.025281406967739534}, {0, -2.0899785200532723, 0}}, 
   {{0.4256650875388649, -0.3257161028957617, 0.0752557060842195}, {-0.008510698360103673, 0.008510698360102986, 
     -0.28368994533677666}, {-0.5979161628380782, -1.1958323256761565, 0}, {-0.0024393071236316175, 0, 0}, 
    {0.11770117349444328, 0.11770117349444348, -0.4708046939777736}, {0, 1.6353262573349148, 0}}, 
   {{0.8003846012504346, -0.11134413595968035, -0.19256544311035795}, {-0.15228694354066433, -0.10275215595208899, 
     -0.8255797931429211}, {0, 0, -1.1315706840481794}, {0, 0.017459520320437565, 0.003491904064087514}, 
    {-0.10469677445511005, 0.231828572007744, -0.2542635951052683}, {0, 0, 0}}, 
   {{-0.2786327216401915, 0.20464425540703357, -0.04355192972462496}, {-0.4545294747918102, -0.4545294747918102, 0}, 
    {0.10716530154609016, 0.21433060309218024, 0}, {0.10218341112576966, 0, 0}, {-0.20533774328947957, 0.20533774328947949, 
     0}, {0, -0.9797970427071093, 0}}, {{-0.6550247392474989, 0.18930562238161397, 0.09213783149475704}, 
    {0.05557948234966326, 0.10800222691962512, -0.8737124094993652}, {0, 0, 1.6305513249901185}, 
    {0, -0.33244461002114456, -0.06648892200422885}, {0.14861099709088296, 0.040165134348887244, -0.37755226287954025}, 
    {0, 0.37947376290679113, 0}}, {{0.41600450878953316, 0.6576746918204932, -0.5771179641435065}, 
    {0.013650559769780468, -0.013650559769780357, 0.45501865899267924}, {0.10947654202437826, 0.21895308404875657, 0}, 
    {-0.8998729663583077, 0, 0}, {-0.052900653114570206, -0.05290065311457026, 0.21160261245828113}, 
    {0, 0.3736799301098779, 0}}, {{-0.141650787292172, -0.021038871613978466, 0.06124284350670972}, 
    {0.054802124438322705, 0.0012175260162869824, 0.8930766403672616}, {0, 0, -0.7632887212914596}, 
    {0, 0.1903370763941856, 0.03806741527883717}, {-0.09371981425349929, -0.1155247395117219, 0.41848910753044255}, 
    {0, -0.167487353700526, 0}}, {{0.3371949038190417, 0.6743898076380834, -0.5619915063650696}, 
    {-0.010065419327041192, 0.010065419327041192, -0.33551397756803636}, {0, 0, 0}, 
    {0, -1.097121114646837, -0.21942422292936747}, {-0.02907323367763026, -0.02907323367763026, 0.11629293471052118}, 
    {0.4019696488812623, 0, 0.8039392977625248}}, {{0.19143730932427963, -0.14165078729217212, 0.03062142175335486}, 
    {0.0012175260162870915, 0.05480212443832282, -0.8930766403672616}, {0, 0, 1.5265774425829193}, {0.9508382993605917, 0, 0}, 
    {-0.11552473951172204, -0.09371981425349919, 0.41848910753044255}, {0.5917886497418585, 0, 1.1835772994837173}}, 
   {{0.16672477679333914, 0.5658953212647757, -0.43283847310763}, {0.21427216298281074, 0.2142721629828108, 0}, 
    {-0.14596872269917105, -0.2919374453983422, 0}, {0, -0.6645998338336429, -0.13291996676672851}, 
    {-0.11123969112876927, 0.11123969112876927, 0}, {-0.48578390914284125, 0, -0.9715678182856825}}, 
   {{-0.1950108365962159, -0.662631691533635, 0.5067580732211623}, {-0.10800222691962509, -0.055579482349663235, 
     -0.8737124094993657}, {0, 0, -0.29646387727093043}, {-0.3853178593768674, 0, 0}, 
    {-0.04016513434888728, -0.1486109970908831, 0.37755226287954047}, {-0.6830527732322241, 0, -1.3661055464644485}}, 
   {{-1.0525263173302686, 0.5072091894106108, 0.012702646169682363}, {0.011831534897441542, -0.01183153489744121, 
     0.3943844965813762}, {0.3674238910151659, 0.734847782030332, 0}, {0, 0.09307103447687513, 0.018614206895375022}, 
    {0.08503886338251149, 0.08503886338251168, -0.34015545353004684}, {0.19595940854142194, 0, 0.3919188170828439}}, 
   {{0.5100739142321601, -0.5666889242552123, 0.20776797809275482}, {-0.10275215595208947, -0.1522869435406646, 
     0.8255797931429191}, {0, 0, -1.0487728291178249}, {0.01868485169832293, 0, 0}, 
    {0.23182857200774445, -0.10469677445511068, -0.2542635951052676}, {0.7065416954056924, 0, 1.413083390811385}}, 
   {{0.29712701126938645, 0.15160414022395394, -0.20011176390576477}, {-0.7727977852535528, -0.7727977852535528, 0}, 
    {-0.22485736038355097, -0.4497147207671012, 0}, {0, -0.0023118846443940834, -0.0004623769288788167}, 
    {0.18693715790293958, -0.18693715790293944, 0}, {0.3270652514669831, 0, 0.6541305029339659}}, 
   {{-0.18270873967106058, 0.07767642341721442, 0.009118630945543906}, {0.17337638347544146, 0.13057842321452878, 
     0.7132993376818787}, {0, 0, 1.2969063137830574}, {-0.00022331529976295515, 0, 0}, 
    {-0.4045025114838326, 0.3918618079999629, 0.025281406967739506}, {-0.1791410160045662, 0, -0.3582820320091324}}, 
   {{0.010116150620918567, -0.030771313464273323, 0.01714215876920936}, {-0.002600645752132684, 0.002600645752130557, 
     -0.08668819173772069}, {-0.626993556015982, -1.2539871120319641, 0}, {0, 0.000012270071415546983, 2.454014283109397e-6}, 
    {-0.0505628139354794, -0.05056281393547872, 0.20225125574191624}, {-0.23885468800608828, 0, -0.47770937601217656}}};

    double diff = 0;
    for (int iv = 0; iv < 6; iv++) {
        sf::dRlm_dr(lmax, v[iv], data);
        for (int x = 0; x < 3; x++) {
            for (int lm = 0; lm < lmmax; lm++) {
                diff += std::abs(data(lm, x) - ref[lm][iv][x]);
            }
        }
    }
    printf("total diff: %18.12f\n", diff);
    if (diff > 1e-10) {
        return 1;
    } else {
        return 0;
    }
}

int test2()
{
    std::vector<vector3d<double>> v({{-1, 0, 0}, {1, 0, 0}, {0, -1, 0}, {0, 1, 0}, {0, 0, -1}, {0, 0, 1}});

    int lmax = 8;
    int lmmax = utils::lmmax(lmax);

    mdarray<double, 2> data(lmmax, 3);

    double ref[][6][3] = {{{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, -0.4886025119029198, 0}, {0, -0.4886025119029198, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, -0.4886025119029199, 0}, {0, -0.4886025119029199, 0}}, 
  {{0, 0, 0.4886025119029199}, {0, 0, 0.4886025119029199}, {0, 0, 0.4886025119029199}, 
   {0, 0, 0.4886025119029199}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {-0.4886025119029199, 0, 0}, {-0.4886025119029199, 0, 0}, 
   {-0.4886025119029199, 0, 0}, {-0.4886025119029199, 0, 0}}, 
  {{0, 1.0925484305920792, 0}, {0, -1.0925484305920792, 0}, {1.0925484305920792, 0, 0}, 
   {-1.0925484305920792, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 1.0925484305920792}, {0, 0, -1.0925484305920792}, 
   {0, 1.0925484305920792, 0}, {0, -1.0925484305920792, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 1.0925484305920792}, {0, 0, -1.0925484305920792}, {0, 0, 0}, {0, 0, 0}, 
   {1.0925484305920792, 0, 0}, {-1.0925484305920792, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, -1.7701307697799304, 0}, {0, -1.7701307697799304, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}}, {{0, 0.4570457994644658, 0}, {0, 0.4570457994644658, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, -1.828183197857863, 0}, {0, -1.828183197857863, 0}}, 
  {{0, 0, -1.1195289977703462}, {0, 0, -1.1195289977703462}, 
   {0, 0, -1.1195289977703462}, {0, 0, -1.1195289977703462}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0.4570457994644658, 0, 0}, {0.4570457994644658, 0, 0}, 
   {-1.828183197857863, 0, 0}, {-1.828183197857863, 0, 0}}, 
  {{0, 0, 1.445305721320277}, {0, 0, 1.445305721320277}, {0, 0, -1.445305721320277}, 
   {0, 0, -1.445305721320277}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {1.7701307697799304, 0, 0}, {1.7701307697799304, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 2.5033429417967046, 0}, {0, -2.5033429417967046, 0}, 
   {-2.5033429417967046, 0, 0}, {2.5033429417967046, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, -1.7701307697799304}, {0, 0, 1.7701307697799304}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, -0.9461746957575601, 0}, {0, 0.9461746957575601, 0}, 
   {-0.9461746957575601, 0, 0}, {0.9461746957575601, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, -2.0071396306718676}, {0, 0, 2.0071396306718676}, 
   {0, 2.676186174229157, 0}, {0, -2.676186174229157, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, -2.0071396306718676}, {0, 0, 2.0071396306718676}, {0, 0, 0}, {0, 0, 0}, 
   {2.676186174229157, 0, 0}, {-2.676186174229157, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 1.7701307697799304}, {0, 0, -1.7701307697799304}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}}, {{0, -3.2819102842008507, 0}, {0, -3.2819102842008507, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 1.467714898305751, 0}, {0, 1.467714898305751, 0}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, {{0, -0.45294665119569694, 0}, 
   {0, -0.45294665119569694, 0}, {0, 0, 0}, {0, 0, 0}, {0, -3.623573209565575, 0}, 
   {0, -3.623573209565575, 0}}, {{0, 0, 1.754254836801354}, {0, 0, 1.754254836801354}, 
   {0, 0, 1.754254836801354}, {0, 0, 1.754254836801354}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {-0.45294665119569694, 0, 0}, {-0.45294665119569694, 0, 0}, 
   {-3.6235732095655755, 0, 0}, {-3.6235732095655755, 0, 0}}, 
  {{0, 0, -2.396768392486662}, {0, 0, -2.396768392486662}, {0, 0, 2.396768392486662}, 
   {0, 0, 2.396768392486662}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {-1.467714898305751, 0, 0}, {-1.467714898305751, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, 2.0756623148810416}, {0, 0, 2.0756623148810416}, 
   {0, 0, 2.0756623148810416}, {0, 0, 2.0756623148810416}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {-3.2819102842008507, 0, 0}, {-3.2819102842008507, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 4.099104631151485, 0}, {0, -4.099104631151485, 0}, 
   {4.099104631151485, 0, 0}, {-4.099104631151485, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 2.366619162231752}, {0, 0, -2.366619162231752}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, -2.0182596029148967, 0}, {0, 2.0182596029148967, 0}, 
   {2.0182596029148967, 0, 0}, {-2.0182596029148967, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 2.7636157785447706}, {0, 0, -2.7636157785447706}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0.9212052595149236, 0}, {0, -0.9212052595149236, 0}, 
   {0.9212052595149236, 0, 0}, {-0.9212052595149236, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 2.9131068125936572}, {0, 0, -2.9131068125936572}, 
   {0, 4.660970900149851, 0}, {0, -4.660970900149851, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 2.9131068125936572}, {0, 0, -2.9131068125936572}, {0, 0, 0}, {0, 0, 0}, 
   {4.660970900149851, 0, 0}, {-4.660970900149851, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, -2.7636157785447706}, {0, 0, 2.7636157785447706}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}}, {{0, 0, 2.366619162231752}, {0, 0, -2.366619162231752}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, -4.950139127672173, 0}, {0, -4.950139127672173, 0}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, {{0, 2.5945778936013015, 0}, 
   {0, 2.594577893601302, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, -1.4081304047606464, 0}, {0, -1.4081304047606462, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}}, {{0, 0.4516580379125865, 0}, {0, 0.4516580379125865, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, -5.781222885281108, 0}, {0, -5.781222885281108, 0}}, 
  {{0, 0, -2.3899496919201733}, {0, 0, -2.3899496919201733}, 
   {0, 0, -2.3899496919201733}, {0, 0, -2.3899496919201733}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0.4516580379125865, 0, 0}, {0.4516580379125865, 0, 0}, 
   {-5.781222885281108, 0, 0}, {-5.781222885281108, 0, 0}}, 
  {{0, 0, 3.3189951933373694}, {0, 0, 3.3189951933373694}, {0, 0, -3.3189951933373694}, 
   {0, 0, -3.3189951933373694}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {1.4081304047606462, 0, 0}, {1.4081304047606462, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, -3.1134934723215624}, {0, 0, -3.1134934723215624}, 
   {0, 0, -3.1134934723215624}, {0, 0, -3.1134934723215624}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {2.594577893601302, 0, 0}, {2.594577893601302, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, 2.6459606618019}, {0, 0, 2.6459606618019}, 
   {0, 0, -2.6459606618019005}, {0, 0, -2.6459606618019005}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {4.950139127672173, 0, 0}, {4.950139127672173, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 5.831413281398639, 0}, {0, -5.831413281398639, 0}, 
   {-5.831413281398639, 0, 0}, {5.831413281398639, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, -2.9157066406993195}, {0, 0, 2.9157066406993195}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, -3.1939965963572554, 0}, {0, 3.1939965963572554, 0}, 
   {-3.1939965963572554, 0, 0}, {3.1939965963572554, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, -3.4499106220981077}, {0, 0, 3.4499106220981077}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 1.9136660990373227, 0}, {0, -1.9136660990373227, 0}, 
   {-1.9136660990373227, 0, 0}, {1.9136660990373227, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, -3.705798465886633}, {0, 0, 3.705798465886633}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, -0.912304516869819, 0}, {0, 0.912304516869819, 0}, 
   {-0.912304516869819, 0, 0}, {0.912304516869819, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, -3.816443606457298}, {0, 0, 3.816443606457298}, 
   {0, 6.978639737521918, 0}, {0, -6.978639737521918, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, -3.816443606457298}, {0, 0, 3.816443606457298}, {0, 0, 0}, {0, 0, 0}, 
   {6.978639737521917, 0, 0}, {-6.978639737521917, 0, 0}}, 
  {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, 
  {{0, 0, 3.705798465886633}, {0, 0, -3.7057984658866325}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}}, {{0, 0, -3.449910622098108}, {0, 0, 3.449910622098108}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}}, {{0, 0, 2.9157066406993195}, {0, 0, -2.9157066406993195}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}, {{0, 0, 0}, {0, 0, 0}, {0, 0, 0}, 
   {0, 0, 0}, {0, 0, 0}, {0, 0, 0}}};

    double diff = 0;
    for (int iv = 0; iv < 6; iv++) {
        sf::dRlm_dr(lmax, v[iv], data);
        for (int x = 0; x < 3; x++) {
            for (int lm = 0; lm < lmmax; lm++) {
                diff += std::abs(data(lm, x) - ref[lm][iv][x]);
            }
        }
    }
    printf("total diff: %18.12f\n", diff);
    if (diff > 1e-10) {
        return 1;
    } else {
        return 0;
    }
}

int test3()
{
    int lmax = 8;
    int lmmax = utils::lmmax(lmax);

    mdarray<double, 2> data(lmmax, 3);
    double max_diff{0};

    for (int i = 0; i < 300; i++) {
        vector3d<double> v(1 + 10 * utils::random<double>(),
                           1 + 10 * utils::random<double>(),
                           1 + 10 * utils::random<double>());

        auto rtp = SHT::spherical_coordinates(v);

        double dr = 1e-5 * rtp[0];

        sf::dRlm_dr(lmax, v, data);

        mdarray<double, 2> drlm(lmmax, 3);
        for (int x = 0; x < 3; x++) {
            vector3d<double> v1 = v;
            v1[x] += dr;
            vector3d<double> v2 = v;
            v2[x] -= dr;

            auto vs1 = SHT::spherical_coordinates(v1);
            auto vs2 = SHT::spherical_coordinates(v2);
            std::vector<double> rlm1(lmmax);
            std::vector<double> rlm2(lmmax);

            sf::spherical_harmonics(lmax, vs1[1], vs1[2], &rlm1[0]);
            sf::spherical_harmonics(lmax, vs2[1], vs2[2], &rlm2[0]);

            for (int lm = 0; lm < lmmax; lm++) {
                drlm(lm, x) = (rlm1[lm] - rlm2[lm]) / 2 / dr;
            }
        }

        for (int x = 0; x < 3; x++) {
            for (int lm = 0; lm < lmmax; lm++) {
                max_diff = std::max(std::abs(drlm(lm, x) - data(lm, x)), max_diff);
            }
        }
    }
    printf("maximum numerical derivative error: %18.12f\n", max_diff);
    if (max_diff > 1e-8) {
        return 1;
    } else {
        return 0;
    }
}

int main(int argn, char** argv)
{
    cmd_args args;

    args.parse_args(argn, argv);
    if (args.exist("help")) {
        printf("Usage: %s [options]\n", argv[0]);
        args.print_help();
        return 0;
    }

    sirius::initialize(1);
    int result = test1() + test2() + test3();

    if (result) {
        printf("\x1b[31m" "Failed" "\x1b[0m" "\n");
    } else {
        printf("\x1b[32m" "OK" "\x1b[0m" "\n");
    }
    sirius::finalize();
    return 0;
}
