import sys
import re

# @fortran begin function                                    {type} {name} {doc-string}
# @fortran       argument {in|out|inout} {required|optional} {type} {name} {doc-string}
# [@fortran details]
# [detailed documentation]
# @fortran end

in_type_map = {
    'void*'   : 'type(C_PTR)',
    'int'     : 'integer(C_INT)',
    'double'  : 'real(C_DOUBLE)',
    'string'  : 'character(C_CHAR)',
    'bool'    : 'logical(C_BOOL)',
    'complex' : 'complex(C_DOUBLE)'
}


def write_str_to_f90(o, string):
    p = 0
    subs = string
    while (True):
        p = subs.find(',', p)
        # no more commas left in the string or string is short
        if p == -1 or len(subs) <= 80:
            o.write(subs)
            o.write('\n')
            break;
        # position after comma
        p += 1
        if p >= 80:
            o.write(subs[:p] + '&\n&')
            subs = subs[p:]
            p = 0


def write_function(o, func_name, func_type, func_args, func_doc, details):
    o.write('!> @brief ' + func_doc + '\n')
    if details:
        o.write('!> @details ' + details[0] + '\n')
        for i in range(1, len(details)):
            o.write('!> ' + details[i] + '\n')

    for a in func_args:
        o.write('!> @param [' + a['intent'] + '] ' + a['name'] + ' ' + a['doc'] + '\n')

    if func_type == 'void':
        string = 'subroutine '
    else:
        string = 'function '
    string = string + func_name + '('
    va = [a['name'] for a in func_args]
    string = string + ','.join(va)
    string = string + ')'
    if func_type != 'void':
        string = string + ' result(res)'
    write_str_to_f90(o, string)
    o.write('implicit none\n')

    for a in func_args:
        o.write(in_type_map[a['type']])
        if not a['required']:
            o.write(', optional, target')
        if a['type'] == 'string':
            o.write(', dimension(*)')
        o.write(', intent(' + a['intent'] + ') :: ' + a['name'])
        o.write('\n')

    if func_type != 'void':
        o.write(in_type_map[func_type] + ' :: res\n')

    for a in func_args:
        if not a['required']:
            o.write('type(C_PTR) :: ' + a['name'] + '_ptr\n')
# create interface section
    o.write('interface\n')

    if func_type == 'void':
        string = 'subroutine '
    else:
        string = 'function '
    string = string + func_name + '_aux('
    va = [a['name'] for a in func_args]
    string = string + ','.join(va)
    string = string + (')')
    if (func_type == 'void'):
        string = string + '&'
    else:
        string = string + ' result(res)&'
    write_str_to_f90(o, string)
    o.write('&bind(C, name="'+func_name+'")\n')

    o.write('use, intrinsic :: ISO_C_BINDING\n')
    for a in func_args:
        if not a['required']:
            o.write('type(C_PTR)')
            o.write(', value :: ' + a['name'])
        else:
            o.write(in_type_map[a['type']])
            if a['type'] == 'string':
                o.write(', dimension(*)')
            o.write(', intent(' + a['intent'] + ') :: ' + a['name'])
        o.write('\n')

    if func_type != 'void':
        o.write(in_type_map[func_type] + ' :: res\n')

    if func_type == 'void':
        o.write('end subroutine\n')
    else:
        o.write('end function\n')
    o.write('end interface\n\n')

    for a in func_args:
        if not a['required']:
            o.write(a['name'] + '_ptr = C_NULL_PTR\n')
            o.write('if (present('+a['name']+')) ' + a['name'] + '_ptr = C_LOC(' + a['name'] + ')\n\n')

    if (func_type == 'void'):
        string = 'call '
    else:
        string = 'res = '
    string = string + func_name + '_aux('
    va = []
    for a in func_args:
        if not a['required']:
            va.append(a['name'] + '_ptr')
        else:
            va.append(a['name'])
    string = string + ','.join(va)
    string = string + ')'
    write_str_to_f90(o, string)

    if func_type == 'void':
        o.write('end subroutine ')
    else:
        o.write('end function ')
    o.write(func_name + '\n\n')


def main():
    f = open(sys.argv[1], 'r') 
    o = open('generated.f90', 'w')
    o.write('! Warning! This file is automatically generated using cpp_f90.py script!\n\n')
    o.write('!> @file generated.f90\n')
    o.write('!! @brief Autogenerated interface to Fortran.\n')

    o.write('!\n')
    while (True):
        line = f.readline()
        if not line: break

        # parse @fortran begin function {type} {name} {doc-string}
        m = re.search('@fortran\s+begin\s+function\s+(\w+\*?)\s+(\w+)\s+((\w|\W|\s)*)', line)
        # match is successful
        if m:
            # we need to set the following variables:
            #   func_type
            #   func_name
            #   func_args
            #   func_doc
            #   details
            func_type = m.group(1)
            func_name = m.group(2)
            func_doc = m.group(3).strip()
            func_args = []
            details = []

            # parse strings until @fortran end is encountered
            while (True):
                line = f.readline()
                # check for @fortran details
                m = re.search('@fortran\s+details', line)
                if m:
                    while (True):
                        line = f.readline()
                        # check for @fortran end
                        m = re.search('@fortran', line)
                        if m: break
                        details.append(line.strip())

                # check for @fortran end
                m = re.search('@fortran\s+end', line)
                if m: break

                # parse @fortran argument {in|out|inout} {required|optional} {type} {name} {doc-string}
                m = re.search('@fortran\s+argument\s+(\w+)\s+(\w+)\s+(\w+\*?)\s+(\w+)\s+((\w|\W|\s)*)', line)
                if m: func_args.append({'type'     : m.group(3),
                                        'intent'   : m.group(1),
                                        'required' : m.group(2) == 'required',
                                        'name'     : m.group(4),
                                        'doc'      : m.group(5).strip()})

            write_function(o, func_name, func_type, func_args, func_doc, details)


    f.close()
    o.close()

if __name__ == "__main__":
    main()
