# HQ XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
# HQ X
# HQ X   quippy: Python interface to QUIP atomistic simulation library
# HQ X
# HQ X   Copyright James Kermode 2010
# HQ X
# HQ X   These portions of the source code are released under the GNU General
# HQ X   Public License, version 2, http://www.gnu.org/copyleft/gpl.html
# HQ X
# HQ X   If you would like to license the source code under different terms,
# HQ X   please contact James Kermode, james.kermode@gmail.com
# HQ X
# HQ X   When using this software, please cite the following reference:
# HQ X
# HQ X   http://www.jrkermode.co.uk/quippy
# HQ X
# HQ XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

from quippy import *
import unittest, quippy
from quippytest import *

if hasattr(quippy, 'Potential'):

   class TestPotentialForceMixing(QuippyTestCase):

      def setUp(self):

         xml ="""<quip_params>
<SW_params n_types="2" label="PRB_31_plus_H">
<comment> Stillinger and Weber, Phys. Rev. B  31 p 5262 (1984), extended for other elements </comment>
<per_type_data type="1" atomic_num="1" />
<per_type_data type="2" atomic_num="14" />
<per_pair_data atnum_i="1" atnum_j="1" AA="0.0" BB="0.0"
      p="0" q="0" a="5.0" sigma="1.0" eps="0.0" />
<per_pair_data atnum_i="1" atnum_j="14" AA="8.581214" BB="0.0327827"
      p="4" q="0" a="1.25" sigma="2.537884" eps="2.1672" />
<per_pair_data atnum_i="14" atnum_j="14" AA="7.049556277" BB="0.6022245584"
      p="4" q="0" a="1.80" sigma="2.0951" eps="2.1675" />

<!-- triplet terms: atnum_c is the center atom, neighbours j and k -->
<per_triplet_data atnum_c="1"  atnum_j="1"  atnum_k="1"
      lambda="21.0" gamma="1.20" eps="2.1675" />
<per_triplet_data atnum_c="1"  atnum_j="1"  atnum_k="14"
      lambda="21.0" gamma="1.20" eps="2.1675" />
<per_triplet_data atnum_c="1"  atnum_j="14" atnum_k="14"
      lambda="21.0" gamma="1.20" eps="2.1675" />

<per_triplet_data atnum_c="14" atnum_j="1"  atnum_k="1"
      lambda="21.0" gamma="1.20" eps="2.1675" />
<per_triplet_data atnum_c="14" atnum_j="1"  atnum_k="14"
      lambda="21.0" gamma="1.20" eps="2.1675" />
<per_triplet_data atnum_c="14" atnum_j="14" atnum_k="14"
      lambda="21.0" gamma="1.20" eps="2.1675" />
</SW_params>

<SW_params n_types="2" label="eps_2.6">

<per_type_data type="1" atomic_num="1" />
<per_type_data type="2" atomic_num="14" />
<per_pair_data atnum_i="1" atnum_j="1" AA="0.0" BB="0.0"
      p="0" q="0" a="5.0" sigma="1.0" eps="0.0" />
<per_pair_data atnum_i="1" atnum_j="14" AA="8.581214" BB="0.0327827"
      p="4" q="0" a="1.25" sigma="2.537884" eps="2.3" />
<per_pair_data atnum_i="14" atnum_j="14" AA="7.049556277" BB="0.6022245584"
      p="4" q="0" a="1.80" sigma="2.0951" eps="2.3" />

<!-- triplet terms: atnum_c is the center atom, neighbours j and k -->
<per_triplet_data atnum_c="1"  atnum_j="1"  atnum_k="1"
      lambda="21.0" gamma="1.20" eps="2.3" />
<per_triplet_data atnum_c="1"  atnum_j="1"  atnum_k="14"
      lambda="21.0" gamma="1.20" eps="2.3" />
<per_triplet_data atnum_c="1"  atnum_j="14" atnum_k="14"
      lambda="21.0" gamma="1.20" eps="2.3" />

<per_triplet_data atnum_c="14" atnum_j="1"  atnum_k="1"
      lambda="21.0" gamma="1.20" eps="2.3" />
<per_triplet_data atnum_c="14" atnum_j="1"  atnum_k="14"
      lambda="21.0" gamma="1.20" eps="2.3" />
<per_triplet_data atnum_c="14" atnum_j="14" atnum_k="14"
      lambda="21.0" gamma="1.20" eps="2.3" />
</SW_params>

</quip_params>"""

         system_reseed_rng(1984068303)
         self.pot1 = Potential('IP SW label="PRB_31_plus_H"', param_str=xml)
         self.pot2 = Potential('IP SW label="eps_2.6"', param_str=xml)
         self.pot3 = Potential('IP SW label="eps_2.6" little_clusters=T', param_str=xml)

         dia = diamond(5.44, 14)
         self.at = supercell(dia, 4, 4, 4)
         randomise(self.at.pos, 0.1)
         self.at.set_cutoff(self.pot1.cutoff()+2.0)
         self.at.calc_connect()

         self.at.add_property('hybrid', 0)

         self.embedlist = Table(4,0,0,0)
         self.embedlist.append((1,0,0,0))
         self.at.bfs_grow_list(self.embedlist, 2, nneighb_only=True)
         self.at.hybrid[self.embedlist.int[1,:]] = 1

         self.at.add_property('hybrid_mark', HYBRID_NO_MARK)
         self.at.hybrid_mark[self.at.hybrid != 0] = HYBRID_ACTIVE_MARK

      def test_embedlist(self):
         self.assertArrayAlmostEqual(self.embedlist.int[:,1], string_to_array("""
        1        0        0        0
        2        0        0        0
      128        0        0        0
      414        0        0        0
      484        0        0        0
        3        0        0        0
        5        0        0        0
        7        0        0        0
       29        0        0        0
       99        0        0        0
      127        0        0        0
       31        0        0        0
      387        0        0        0
      413        0        0        0
      103        0        0        0
      389        0        0        0
      483        0        0        0""")[:,0])

      def test_no_multiple_images(self):
         self.assert_(not multiple_images(self.embedlist))

      def test_single_cluster(self):
         create_hybrid_weights(self.at, buffer_hops=3)
         self.at.add_property('force', 0.0, n_cols=3)
         f = fzeros((3,self.at.n))
         self.pot2.calc(self.at, force=f, args_str="single_cluster=T cluster_calc_connect=T")

         self.assertArrayAlmostEqual(f, string_to_array("""
       0.4264523725      -1.4614174250      -0.5027997001
       0.1036734719       1.3014620397       0.5790116361
      -0.2033194347      -1.0012536621       0.3714197275
       0.0000000000       0.0000000000       0.0000000000
       0.7389708093       0.3235243113      -0.1013679954
       0.0000000000       0.0000000000       0.0000000000
       0.4497622462      -0.7812570900      -1.4634404144
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
      -0.9418822157       0.6452137054       0.0281627232
       0.0000000000       0.0000000000       0.0000000000
      -0.7451434920      -0.0977202946       1.3616524382
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
      -0.0963896218       0.1896612029       0.4521041053
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.3209645384       0.4566641581      -0.1493028985
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
      -1.2790590262      -0.0540530475       0.2425983847
      -0.1468022014       0.6673532015       0.4173861854
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
      -1.4652278976      -0.5036399289      -0.7525592536
       0.0000000000       0.0000000000       0.0000000000
       0.3410559167      -0.2866148098      -0.2727499670
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.7730300149       0.5887616027       0.0119796713
      -0.1907992296       0.7697343736      -0.5885002234
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.9212085885       0.5689732460      -1.0798266352
       0.1287647513       0.6455767652       0.4496673831
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
         """))

      def test_little_clusters(self):
         create_hybrid_weights(self.at, buffer_hops=3)
         self.at.add_property('force', 0.0, n_cols=3)
         f = fzeros((3,self.at.n))
         self.pot3.calc(self.at, force=f, args_str="little_clusters=T cluster_calc_connect=T buffer_hops=3")

         self.assertArrayAlmostEqual(f, string_to_array("""
       0.4264523725      -1.4614174250      -0.5027997001
       0.1036734719       1.3014620397       0.5790116361
      -0.2033194347      -1.0012536621       0.3714197275
       0.0000000000       0.0000000000       0.0000000000
       0.7389708093       0.3235243113      -0.1013679954
       0.0000000000       0.0000000000       0.0000000000
       0.4497622462      -0.7812570900      -1.4634404144
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
      -0.9418822157       0.6452137054       0.0281627232
       0.0000000000       0.0000000000       0.0000000000
      -0.7451434920      -0.0977202946       1.3616524382
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
      -0.0963896218       0.1896612029       0.4521041053
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.3209645384       0.4566641581      -0.1493028985
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
      -1.2790590262      -0.0540530475       0.2425983847
      -0.1468022014       0.6673532015       0.4173861854
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
      -1.4652278976      -0.5036399289      -0.7525592536
       0.0000000000       0.0000000000       0.0000000000
       0.3410559167      -0.2866148098      -0.2727499670
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.7730300149       0.5887616027       0.0119796713
      -0.1907992296       0.7697343736      -0.5885002234
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.9212085885       0.5689732460      -1.0798266352
       0.1287647513       0.6455767652       0.4496673831
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000
       0.0000000000       0.0000000000       0.0000000000"""))        


      def test_force_mixing_default(self):
         pot = Potential('ForceMixing method=force_mixing buffer_hops=3 qm_args_str={single_cluster=T cluster_calc_connect=T}',
                                 pot1=self.pot1, pot2=self.pot2)
         f = fzeros((3,self.at.n))
         pot.calc(self.at, force=f)
         self.assertArrayAlmostEqual(f, string_to_array("""
       0.4264523725      -1.4614174250      -0.5027997001
       0.1036734719       1.3014620397       0.5790116361
      -0.2033194347      -1.0012536621       0.3714197275
      -0.5245825704       0.8960803203       0.4934865652
       0.7389708093       0.3235243113      -0.1013679954
       0.2003221240      -0.2864186909      -0.0947513360
       0.4497622462      -0.7812570900      -1.4634404144
       0.6019511572       0.8831087095       0.6664193544
       0.1996628605      -0.4063227338       0.9379857553
      -1.3248850421       1.4944091274      -0.4250453013
      -1.0386430202      -0.8327580294      -1.1221580950
       1.2806408905       0.9584571071       0.9251532756
      -0.4275075993      -0.8490839599       1.6854842830
       0.9797860906      -1.4924996916       0.4866199697
      -0.0774059084      -1.0125839823      -1.3729628654
       0.6046978108      -0.4596437086       1.2641279513
      -0.1687565674      -1.0969723940      -0.3842891730
       0.8878858523       0.5704335744       0.0626633177
      -1.6549415859       0.2936701172       0.2233292416
       0.9462083392      -0.0183471351       0.2397467768
      -0.6916953284      -0.8215245039      -0.6936364931
       0.9108910841      -0.5593117381       0.2205325148
       0.6499684705      -0.2749568235      -0.3413196704
      -0.3303027914      -0.3207325836      -0.1479270535
      -0.5667138803       0.7322039313      -0.8678686665
       1.2731696455      -0.3433079580      -0.5612530327
       0.0557364123       0.4942113243       1.1278913916
      -0.3206716366      -0.1011284545      -0.9777876268
      -0.9418822157       0.6452137054       0.0281627232
       0.0725889177       0.7778033379       0.0252922111
      -0.7451434920      -0.0977202946       1.3616524382
      -0.0101174000       0.0246620015      -0.6006780530
      -0.1101727998      -0.5019479740       1.0440629746
       0.5984765646      -0.0222507246      -1.1565074693
      -0.7920878880       0.2574642596       0.4078113196
      -0.2479603743      -0.0113193798       0.7261522437
       0.5199882158      -1.0123168683       0.0171865328
       0.4063345546      -0.3645232848      -0.0169863361
      -1.2644748466       0.1209719037      -0.4211531489
       0.0847556718       0.6886342823       0.8048738274
       0.4955284831      -0.6044312104       0.2756818027
      -0.8870650253      -0.6863127779       1.1547035096
       0.6433382890       0.9806688549       0.0216514099
      -0.5353665164      -0.4617343368      -0.1613414214
       0.1959655960       0.8974868988      -1.0012293577
       0.4845678565      -0.5889861298       0.2301549439
       0.5431519504      -0.8772383408       0.2562822162
      -0.5381379423       0.3331465308      -0.5715374433
       0.0670764045       0.6387920731       0.1931628424
      -0.3159727728      -0.5658863814      -0.3940299558
       0.2914339961      -0.4485525707       0.3030152307
      -0.7033829031       0.1676218213       0.7165203012
       0.1932402239       0.5517486137      -0.0405472958
       0.2018132534      -0.1642468321       0.1621702594
      -1.0283800637      -0.2547204898       0.0804222308
       0.6841988573       0.4578362905      -0.1361427443
      -0.9035964527       0.1491083533       0.8338121351
      -0.2457839638      -0.5195442681       0.2843614116
      -0.1994861303      -0.1028513871      -0.0680356109
       1.5046724873       0.0625282893      -0.8696553550
      -0.8191224887       0.2017712369      -0.1532604653
       1.2642428311      -0.0511269234      -0.1351239468
       0.6067702128       0.2207479198       0.4414286530
      -0.4142329503      -1.0617051314       0.0232009807
      -0.3084314897       0.3272521389      -0.5038411705
      -0.6956821163       0.4884802556       0.8268653626
      -0.2087247220      -1.0827985521      -0.5974180167
       0.6055721313       1.3748088024       0.9247642977
      -0.3264243337      -0.5134852909      -0.6687405666
       0.7044940415       0.7431717092       0.5663886490
       1.2678392825      -0.2509357664      -0.9478430526
      -0.8100255017       0.5175899065      -0.3192781865
       0.5481291715      -1.0175229191       0.2589871329
      -0.5071825256      -0.0110708458       0.1801922262
       1.2350964155      -0.1228434213      -0.1948671655
      -1.6765364759      -0.9016886187       0.9711298860
      -0.6013547408      -0.2595545264      -0.2583185843
       0.2341630484       0.6594705870       0.1097921682
       0.6693589875       0.3296997855      -0.1458804854
      -0.7589894501      -0.3745649690      -0.3667759859
      -0.5694852848      -0.7014635598      -0.1461097503
       0.7730908568       0.8093832389       0.5295262483
       0.4209519005      -0.5194158374       0.7966301027
      -0.0392331858      -0.2764819144       0.6958776605
      -0.4909968831      -0.4923922268      -0.8561343746
       0.6501881645       0.1745445680       0.0614042461
       0.0128641544      -0.8679793853       0.3963337302
      -0.2765288639       0.4539347387       0.5232976221
      -1.1272737948       0.0275057610      -0.2805328027
      -0.3928919625       0.7916570515       0.0988272183
       0.0226100502       0.4300623434       0.4747028016
      -0.4206448532       0.1177890632      -0.5973639564
      -0.3541823529       0.7071381329       0.8219415747
       0.7222869220       0.7623896535      -0.7785195027
       1.0322501093      -1.3308120711       0.9978569007
      -0.7104940170       0.2126673906      -0.0816016581
      -0.4469268276      -0.1878681425      -0.8246124404
      -0.3575820874       1.0302782268      -0.1797380803
      -0.0963896218       0.1896612029       0.4521041053
      -0.4812963519       0.0017738445       0.3531352507
       1.2957831680      -1.1200977802       0.4208342856
      -0.7361652046      -0.0277427390      -0.4010114018
       0.3209645384       0.4566641581      -0.1493028985
      -0.2862328779      -0.4026882082      -0.7193717315
      -0.0251956935       0.0367296046      -0.4441213729
       0.7335055811       0.3455323215      -0.0405007408
      -0.1470021053      -0.4994040503       0.7371020361
       1.2174042135      -0.8904863610      -0.5940327072
       1.0004385695       0.2636684826      -0.6222405623
      -0.9225440987      -0.8458433714       0.5251729866
      -0.9018053436      -0.8343720373       0.2772738438
       0.1467104314       1.6363537219      -0.1192763901
       0.1375797680       0.3952815984       0.2169061303
      -0.9948474794       0.3731644229       0.6689147361
       0.7844773575      -0.3287068088      -0.1957736715
      -0.1929442054       0.4616996515       0.7270212434
       0.3711794070      -0.4353541860      -0.6553434221
      -0.0396505998       0.5284833024       0.2273302714
       0.6194449312      -0.3566245192      -0.8709005963
       0.0080990438       1.2834554066      -0.8386154046
       0.8105888026      -0.5072576205       0.2455561168
      -0.1492175104      -0.1477547920      -0.7920764090
       1.4288522747      -0.6912847924       1.3481148429
      -0.7810183239      -1.3845700074      -0.5432584532
       0.8938224858      -0.0416309068       0.1831852280
       0.1233814331      -0.3248505582      -0.5300100879
      -1.2790590262      -0.0540530475       0.2425983847
      -0.1468022014       0.6673532015       0.4173861854
       0.3536539051      -0.9472074470      -0.5796559031
       0.4489341251       0.4440808828       0.2723006210
      -0.2817650153      -0.6345108669      -0.3788047709
       0.1757083678      -0.1549804725      -0.0837922823
       0.4502951269       0.2204200279      -0.2542721079
      -0.3751514576      -0.1629133184       0.1067852125
      -0.2272781808       0.0650198767       0.3410677135
      -0.6317511038      -0.2529048571       0.5313331844
      -0.9237468423       0.5667604973       0.2931270170
       0.0621925176      -0.7810508654       0.1063761368
       0.5951553619      -0.0980480084      -0.5521529492
      -0.1650778649       0.0475456126       0.0587579741
       0.6945646936       1.2134905209       0.5012191922
      -0.5716532195      -0.7623471499      -0.9926478026
       0.0850517855       0.8305751359      -0.3780855324
      -0.3051958935       0.0031584084      -0.2080642235
      -1.2962301797       1.3850822547      -0.0261462486
      -0.3078908796       0.2815335377      -0.5983793706
       0.2377619284       0.3592806329       0.7850315036
       0.3299779028      -0.5423753477       0.1215289087
       0.3841272443       0.7815864809       0.9331242845
      -0.7615150255      -0.0676118304      -0.4598453288
      -0.5836351535       0.0215820267      -0.2661491757
      -0.5318196685       0.5251973975       0.7106654989
       0.5170912493       1.1063278286      -0.7824708446
      -0.0754669384       0.1064057446      -0.1534873670
      -0.0834245669      -0.1851478887       0.3220846804
       0.2674550984      -1.0939849475       0.6208718763
       0.2556876562       0.3479856515      -0.5970862947
      -0.2768875317       0.1051844398       0.7453012587
       0.3735629630      -0.9800750344       0.2091939697
      -0.1921238195       0.2900117093       0.4529769580
       0.0925841675      -0.2674307153      -0.3055637026
      -0.4179878915       0.5320039567      -0.2290212049
      -0.5716706411       0.1745806743       0.6252742365
       0.0627737257       0.4781856642      -0.8819156843
       1.0748479830       0.4092062333      -0.2790947138
      -0.7510127662       0.0676120299       0.0805716301
      -0.4256288467      -0.4172977676      -0.4358743051
       0.1788322191       1.2499559012       0.1093858741
      -0.4166370974       0.3646495667       0.4704513513
      -0.5998522228      -0.6082744439      -0.4374893586
       0.8663438408      -0.4479802470       0.2843485010
      -1.0014927794      -0.3151333043       0.3024239863
       0.5882483833       0.6552237595       0.2346142140
      -0.2270325060      -0.0140706615      -1.0976682495
       0.3960876339      -0.4195249583       0.3256153244
      -0.7577469424       0.9519031138       0.5421036906
      -0.8308992989      -0.1121315762      -0.6826916148
       0.7190497946       1.0518377401       0.3805267467
      -0.3423727602      -0.8980530355       0.1936265160
      -0.6189394836       0.4694633710       0.0959273870
      -0.4540892999      -0.6631196110      -0.7593072180
      -0.2847863714       1.0806017663       0.7083967209
      -0.0145627490      -0.4943551189      -0.1627482284
      -0.3650799614       0.6530513489       0.7663886870
       0.2564363022      -0.3239061732      -0.8243097129
      -0.1216911573       0.2035546815       0.2945941816
       0.5206544070      -0.3202733262      -0.9617731206
      -0.3980826209       0.7477555154       0.7269618113
       0.1806777941       0.6599140223       0.6474307565
      -0.0712984096       0.3393653399      -1.7634377440
      -1.4493026577      -1.3135350711      -0.8270765183
       1.2847815039       0.9706051586       0.9981580928
       0.7103345481       0.5181358452      -0.1681756413
       0.2489349690      -1.5096214530      -0.4169437061
      -0.2251843094       0.4653542430      -0.3442371614
      -0.7968190627       0.9212573940       0.1017382098
      -1.2388626213      -0.2737272650      -0.0513236006
       0.8012979326       0.9186570350      -0.6288551450
      -0.8965317163       0.6240114711       1.1352840524
      -0.3835992881       0.6537426564      -1.0189271392
       0.1734104497      -0.2508494471      -0.9466780536
       0.4786136337       1.1109541887      -0.5424722636
      -0.6869527493      -0.8771896313       1.1413261585
      -0.6591367055       0.2288833104       1.2257722939
       0.0453034271      -0.7861600681      -0.4454067257
       0.4579362010       0.7703492371      -0.0852902979
       0.1184380497      -0.0079425370      -0.2652510596
       0.4935267722       0.0684878486       0.3795899720
       1.0294167074      -0.5778194918      -0.7512835441
      -0.6757715023      -0.2533864317       1.1257467535
       0.8876045854       1.1731990539      -0.1251610949
      -1.0537440056      -1.0205203310      -0.0892193182
      -0.3302022814      -0.1025526578      -0.6524217389
       0.0495030877       0.3407922226       0.7845133508
      -0.4208068311      -0.3281788900      -1.1552290936
       0.2084139301      -0.7780428280       1.1174046050
      -0.3879056385      -1.3496984515      -0.1625722890
       0.6502024050      -0.4950114585       0.9821444646
      -0.2104565439       0.6764625318      -1.1813728552
       0.3947525493       0.0951878993       0.9047855059
      -0.7583321692      -0.2925510721      -1.2555599413
       0.8884329676      -0.6437753083       1.1073166736
      -1.2878059269      -0.8532485634       0.4623460511
       0.6186539135       1.0922716175       1.1425413415
       0.1182051725      -0.8667233811      -0.5368210539
      -0.6276741412       0.5982437502      -0.6305561310
       0.6248085292      -0.8292575491       0.8753926261
      -0.0964373617       0.7370003457      -0.3611856962
       1.0096496330      -0.7669690714       0.8950294985
      -0.4990769626      -1.3391879703       0.1677737084
       0.3226687889       0.3318289067       0.0162175503
      -0.0225269367      -0.8309721867      -0.2060166356
       1.4839083822       0.8443765519      -0.3144498205
       0.5915881096      -0.6103759161      -1.1866341109
      -0.3968945279       0.4370106006       0.5145252484
      -0.7809129828      -0.2968961387      -0.3511239661
       0.5030211948      -0.5010186124      -0.2180476285
      -0.2926834824       0.4963148534      -0.8758087974
      -0.5294845309       1.7237935440       0.1610149713
       0.9862492123      -1.5544690628      -0.2442722304
      -0.0219370536       0.3930230875      -0.1252209756
       0.1987872132      -0.3799163861      -0.3251680497
      -0.5643914238       0.5041027887       0.2083547867
       0.3175451874      -0.8015721327      -0.6684966798
       0.0737992783       0.9885875911      -0.4987657683
      -0.3002824762       0.5520407731      -0.0447827999
      -1.0742736464      -0.8640440586      -0.7980111604
       0.5672646921       0.4014474549       1.3909689707
      -0.0388305765      -0.4597124294       0.1640936607
       0.1323602986       0.4064025430      -0.0528930025
      -0.1054630546      -0.5320024506      -0.9938522025
       0.8189555992       0.3592961573       0.8071945973
      -0.4076659891      -0.9058536973      -0.5833085385
       0.3333957893      -0.2483437291       0.8208240103
      -0.0751761819       0.0345089133       0.4418756409
      -1.3611991862       0.4209286535      -0.2064455482
      -0.0947561671      -0.7342228271      -0.1210329728
      -0.1574323927      -0.3058227403       0.8729752396
       0.6265781270       0.3371223560       0.0600586257
      -0.3550049035      -0.0210357977       0.2085147342
      -0.4002681697      -0.7222301051      -0.2708414338
      -0.0919717074       0.3246644134       0.1094411165
       0.4426719587       0.7235097499       0.2078417820
       0.8791625955      -0.8566651515      -1.6362428232
      -0.0403019748       0.1853084854      -0.0298100210
       1.0076501527      -0.9890149084      -0.4408791674
       0.0174505139      -0.1184141778       0.3030053268
       0.3185650296      -0.0758026843       0.5634540910
       0.3982018074       0.5172191041      -0.5574754578
      -0.6470989319       0.1608487472       0.6537454800
      -0.5411865522       1.0147480249       0.9611051548
      -1.2925669963       0.4238677273      -0.6201855306
       0.0365196965      -0.3987583667      -0.0476108677
       0.4964252117       0.2728857001      -0.6238698690
       0.9963037083      -0.8563146937       1.0752575512
      -0.0690182897      -0.6947831149      -0.1589518496
       0.2428743737      -0.6779066530      -0.7766913043
      -0.2576363134       0.5887088721       1.0414401956
      -0.8364265359       0.1732226970      -0.4392220044
      -0.1284122789       0.9830844001       0.1539918817
      -0.0577681967      -0.7230214214      -0.3178246537
       0.6362870804      -0.3242780205       1.1223562346
       0.8247944817      -0.5937218235      -0.4923539086
      -0.8776531138      -0.8994758861       0.4475433275
      -0.3290870870       0.4452088619      -0.2559229151
      -0.2368422842      -0.1220458300      -0.5428636225
       0.1155873668       0.8927765400      -0.8581169223
      -0.7687824636       0.4434201759      -0.5350779717
       0.7341306774      -0.8068129418       0.4911966079
       0.1856247932       1.0775255536      -0.4461180522
      -0.8450272650      -0.5003522420       0.0481832672
       0.4739632461       0.0748200340       0.1771206986
      -0.8155388458      -1.1341650815       0.2646764175
       0.5774073944       1.2456522485       0.6727630565
       0.3500435381      -0.9167273876       0.1083621477
       0.2708526297      -0.2043748965       0.1155741637
      -0.3946094303       1.1029311399       1.0210233320
       0.2130605737      -0.9230561014       0.5580242758
       0.1562061807      -0.6470001238      -0.8966709639
       0.3106377257       0.7575530941       0.2099476233
       0.0623447884       0.4805201538      -1.0502340432
      -0.4157371598       0.0675036706      -0.1806255637
      -0.1828097831      -0.3550720107       0.4348325601
       0.4417574907       0.3835959405      -0.0859919337
       0.4578350441      -0.3908495575       0.2394967523
       0.4103325343      -0.0369691850       0.4616871953
      -0.3096191584      -0.5071267420      -0.3464619258
      -0.3483214162       0.9666760911       0.5114543063
      -0.4240784318       0.2634935499      -0.2497376106
       0.9132361757      -0.5057096972      -0.3153571294
       0.1396940320      -0.8503815809       0.3917979784
       0.6844931020       0.8385545797      -0.2128769068
       0.9611719139       0.2528150866      -0.4089220271
       0.6930717116      -1.0018466359      -0.6520184475
      -0.7030362348       0.5310857552       0.5256108828
       0.2719741184       0.0967482878       0.0957909738
       0.1706776667       0.6816281664       0.5223458764
       0.1030682486      -0.3547380118      -0.7732236125
      -0.9909160524      -0.3665243925       0.9087729040
      -0.2625157322       0.0408187568       1.0518148532
      -0.7136625646       0.0206364323      -1.0951915274
       0.8224648243      -0.4496785862       0.6926748178
       0.1499216641      -0.1909247743      -0.9465979425
      -0.3419873296      -0.0118744814       0.5369071417
      -0.2571500080       0.7033353477      -0.4815254136
       1.2125750007      -0.3826582428       0.1359120019
      -0.4595052556      -0.8866497797       0.7725525088
       0.4370978681       0.1024796395      -0.0436639355
       0.9996616659       0.6755231821       0.5205201634
      -0.8371568810       0.2974997994      -0.1937677206
       0.1270815546       0.7866227751       0.7281537608
      -0.4565656440      -0.4196711691      -0.9603939291
       0.0168731944      -0.3020035458       0.6205274525
      -0.6546851564      -0.0154722150      -0.1939505203
      -0.1957644636      -0.7368453489      -0.1907592379
      -0.4429101026       0.0266332942      -0.0586836411
      -0.0564517318      -0.0943887863      -0.6709152522
       0.4421794119       0.9920284613       0.2662719692
      -0.0704626076      -0.4582744355       0.6321049116
      -0.1210263693      -0.6618347484      -0.0006015552
      -0.0019039610      -0.7512548730       0.3660509575
      -0.6131162829       0.1295985088      -0.3297440199
       0.5656517205      -0.6837943656      -0.4504544089
      -0.1073922329       0.5322951811       1.1872621546
       0.1412625308      -0.4719860872      -0.6636351370
      -0.5111405375      -0.2995157354       1.5765527797
      -0.2885369488      -0.0064778578      -1.0867907778
      -0.1383317175      -0.8012064067       1.1185825501
       0.1331517465       0.6196993918      -1.1671799710
      -0.4185581709      -0.2803293173       0.9654069463
       0.2891684445      -0.2223992658      -1.5526063730
       0.5121187931       0.0317532024       0.8582219010
      -0.6916166757       0.1541166064      -0.5662505434
       0.5158556597       0.9330794614       0.8433307876
      -1.4333843868      -0.7751828136      -0.5931404431
       0.6274684375      -0.1912130611       1.8407614736
       0.2438290594       0.2745557244      -0.2340882520
      -0.3031978644      -0.4425733392      -0.7593208531
       0.8797196046      -0.4444508953      -0.8612104040
       0.6783446600       0.6535088317      -0.4080609837
      -0.4736645849       0.3212480648      -1.3207269019
       0.8172287743       0.8622678937       0.5239714275
      -0.5588229764      -0.2797436575       0.5345152331
       0.4520561504       0.6342153757      -0.3431477400
       0.4804498494      -0.0970030678      -0.0110793614
      -0.6784447769       0.3889842133       0.6218198308
       0.4403160834      -0.4569978667       0.8552579880
      -1.2827449911       0.0835786645      -0.5348860889
       0.5654437574      -0.3818186718      -0.6450267594
       0.1063397239       0.7676675798       0.7748246659
       0.9779799928      -1.0722288418       0.4804304202
      -0.4431473715       0.6249865886       0.0062820160
      -0.4532916178       0.2922925087      -0.7511619659
       0.2822495157      -0.7310218070       1.2305974163
       0.2458025954      -0.2160526186      -0.3312833889
      -0.0566020167      -0.3239713899       0.9183817976
       0.6914806699      -0.5333967724      -0.2734452360
      -0.3128582967      -0.8036779632       0.2589066758
       0.3142152759       0.5639004529      -0.7432134113
      -0.2002788379      -0.4185672204       0.1465107042
      -0.2812190940       0.9447417547      -0.4501576405
      -0.0924441255      -0.3223172328      -0.3131943232
      -0.2598018362       0.4473520527      -0.6520206179
       0.1007962859       0.3897796979       0.5429768646
       0.6039199702       0.2460253404      -0.6140008581
       0.8893263653       0.1144415115      -0.2695892132
      -1.4652278976      -0.5036399289      -0.7525592536
       0.7727543213       0.4631296818       0.3840756715
       0.3410559167      -0.2866148098      -0.2727499670
      -1.1192834872       0.1962662569       0.1739712528
      -0.3335926309       0.0862642015      -0.0720512638
      -0.6639449339       0.2738497495       0.6980402239
      -0.8643612974      -0.7065710689       0.3427920922
       1.2082512141       0.0987160059      -0.9799534185
       0.0903235786      -1.0024636428       0.2396235988
       0.6558299172       0.0654347499       0.4920877531
       0.2926001985       0.1046396129      -0.1342635079
       0.0254095857       0.5555332623       0.3879846161
      -0.7050521986       1.3673906595       0.0717638291
      -0.2005773982      -0.6462319970       0.1184834869
      -0.8893688666      -0.8790924235      -0.3806443176
       0.4995270654       1.0684442626       0.8912643206
       0.0429559538      -0.1878910264      -0.2423293870
      -0.1607618914       0.1115922880      -0.3665693295
       0.4430181013      -0.1008639937      -0.3465068109
      -0.4651991505      -0.3180555146       1.3305940709
       0.1114194035       0.3044286643      -0.6265699077
      -0.5719516124       0.7810551007      -0.6573022576
       0.5337541736       0.6869734653       0.1055752838
      -0.8189086485      -1.2330037341      -0.0057317437
       1.3631464050      -0.5148427186      -0.6520993338
       0.2843235982       1.0692151965      -0.3880141953
       0.7730300149       0.5887616027       0.0119796713
      -0.1907992296       0.7697343736      -0.5885002234
      -0.1451141981       0.5452646321       0.2286848560
       0.4150917325      -0.6175302302       0.7665094103
       0.3662366662       0.4862135062       0.0838935872
       0.3556368710      -0.7141544912       0.1890028983
      -0.0473394799      -0.4079106089      -0.6022746841
       0.1466401973      -0.0769152284       1.1159721885
       0.5712524588       1.1986583170      -0.8430935721
      -0.8658728688       0.1685743372      -0.3093141628
      -1.2525537636      -0.4953948859       0.1505707521
       0.7975322538      -0.3171217288       1.4832909413
      -0.5354765054      -0.6987844430      -0.6882801269
       0.3776369013      -0.0929715981       0.2613226712
      -0.3846189668       0.4559804229      -1.1004356028
      -0.1708836301       0.9998088766       0.2403569417
      -0.0140674285       0.3735691148       0.0565647703
       0.3617669546       0.4806526543      -0.4208374560
       0.3506644837       0.2308810007       0.2746750303
      -0.3181261679      -0.3432812455      -0.4445514136
       0.7646221116       0.0813366110       0.5232259526
      -1.2848735102      -0.7357784443       0.7893067749
       0.9246959483       0.6389238493      -0.9855451165
       0.8510648481      -0.3353329293      -0.1027977051
      -0.9204339434      -0.9593967038      -0.3616183495
       1.3345189472       0.9459010592       0.3445523958
       0.3162091461      -0.2856811079       0.8124495481
      -1.2176949680       0.0300349276      -1.0490954672
       0.9508699590       1.1608035247      -0.4472881017
       0.0302037127      -0.4775158885      -1.2298852361
       0.1336275769      -0.8806942372       0.8714613859
       0.2802888752       0.7299265074      -0.2116016094
       0.4619129739      -0.5782862178      -0.6190305904
      -0.1433364103       0.9531997024      -0.2173838638
      -1.1338150435       0.2831131995       0.1828237535
       0.6229075365      -0.1348378054      -0.0865167516
      -0.7246003844       0.5669277101      -0.0350160579
       0.2981855373       0.0856158124      -0.0314836208
       0.2560982449      -0.0647542689      -0.2878426051
       0.4161846501      -0.4005298101      -0.0387405539
       0.7805481140       0.3712222836      -0.7840857994
      -1.4226227555       0.4383419833       0.7444717708
      -0.7641748309      -0.7018783565       0.2037890337
       0.1238503030       1.4180132506       0.0288459603
      -1.2175309570      -0.4373937760      -1.0739436723
       0.6734576242       0.5872376638      -0.0832530373
       0.4062263977      -1.3679326069      -0.2011289620
      -0.6582182823       0.9085087708       0.4365307355
       0.3138266146      -0.3452846887       0.0699097723
      -0.4834304237      -0.1339924672       0.0945589968
       0.2930819118       0.6797778876       0.3789905746
      -0.3736017032      -0.5810323272       0.4364035703
       0.3539225095      -0.2071524010      -0.0722781654
      -1.1481784182       0.8307401736       0.1708571185
       0.1692674660       0.1594278147      -0.0781465041
      -0.4122704510       0.4241505365      -0.4754624881
       0.4850805928      -0.3096152250       0.7464120753
       0.3586649868       0.4967797982       0.3268572160
       0.6846997541       0.0670659428      -0.6242849554
       0.4523618309      -0.2955266451      -1.1113979940
       0.7269511790      -0.3448130485       0.9428358557
      -0.2866917157      -0.1730786641      -0.3437054096
       0.9514404814      -0.3297669412       0.6503184050
      -1.1866196979       0.3241362879      -1.5436892953
       0.2143124196      -0.5232059055       0.1382490572
       0.2516056445       0.5398190748      -0.0800572490
      -0.0557306480       0.3384195946      -0.5150003935
      -0.4303835332      -0.1513581177       0.1520507230
       0.8905247386       0.0683537831       0.3298628834
      -1.0698021163      -0.8232083038       0.0797007636
       0.9212085885       0.5689732460      -1.0798266352
       0.1287647513       0.6455767652       0.4496673831
      -0.7653186179      -0.1873038057       0.5811720505
       0.3473644948      -0.7823919466       0.5639912840
      -0.7800409000       0.5774671992      -0.7574287056
       0.6671197164      -0.0981747213       0.8093831193
       0.1963190783      -0.7842955200       0.5915968580
       0.1328513107       0.6711169952      -0.8784983356
      -1.0625598008       0.2825364752      -0.0461049237
       0.2144763796       0.4268605194      -0.2653315569
       0.6675846384      -0.9965523174      -0.7837810393
       0.4124585069       0.8984726405       0.4106543952
      -0.1935309814      -0.0670739496      -0.4179830821
       0.4836644011      -0.3199786071       0.8604545955
       0.0737657423       0.1057154197      -0.8297993009
      -0.2517757296       0.5399366077       0.8118759564
       0.2893709222      -0.0571665852      -0.6493503665
      -0.0045567244       0.4796897230      -0.3411535019
       0.2149382333      -0.4196790641       0.0033827306
      -0.7667709320      -0.2583330052       0.2660077398
       0.3198758473      -0.1541300758      -1.2107943381
       0.1228389832       0.0495184471       0.7673709224
      -0.0337205142       2.1259998535       0.3166728280
      -0.5597661255       0.0545664132      -0.8682465258
      -0.0591138343       0.2799463324       0.0586204425
       0.0141773614      -0.9490747159       0.3646657753
       0.7178560724       0.2797766288       0.6585706876
       0.2974871598       0.3188808666      -0.0703750046
       0.4415116443       0.1135808859      -0.2920584839
      -0.9099268121      -0.4814231815       0.4761962741"""))
         

      def test_force_mixing_hop_ramp(self):
         pot = Potential('ForceMixing method=force_mixing_smooth transition_hops=0 buffer_hops=1 qm_args_str={single_cluster=T cluster_calc_connect=T}',
                                 pot1=self.pot1, pot2=self.pot2)
         f = fzeros((3,self.at.n))
         pot.calc(self.at, force=f)
         self.assertArrayAlmostEqual(f, string_to_array("""
       0.4264523725      -1.4614174250      -0.5027997001
       0.1036734719       1.3014620397       0.5790116361
      -0.2009062837      -0.9968901351       0.3658118437
      -0.5245825704       0.8960803203       0.4934865652
       0.7422660010       0.3208355127      -0.1014390480
       0.2003221240      -0.2864186909      -0.0947513360
       0.4412832202      -0.7742234852      -1.4612859562
       0.6019511572       0.8831087095       0.6664193544
       0.1996628605      -0.4063227338       0.9379857553
      -1.3248850421       1.4944091274      -0.4250453013
      -1.0386430202      -0.8327580294      -1.1221580950
       1.2806408905       0.9584571071       0.9251532756
      -0.4275075993      -0.8490839599       1.6854842830
       0.9797860906      -1.4924996916       0.4866199697
      -0.0774059084      -1.0125839823      -1.3729628654
       0.6046978108      -0.4596437086       1.2641279513
      -0.1687565674      -1.0969723940      -0.3842891730
       0.8878858523       0.5704335744       0.0626633177
      -1.6549415859       0.2936701172       0.2233292416
       0.9462083392      -0.0183471351       0.2397467768
      -0.6916953284      -0.8215245039      -0.6936364931
       0.9108910841      -0.5593117381       0.2205325148
       0.6499684705      -0.2749568235      -0.3413196704
      -0.3303027914      -0.3207325836      -0.1479270535
      -0.5667138803       0.7322039313      -0.8678686665
       1.2731696455      -0.3433079580      -0.5612530327
       0.0557364123       0.4942113243       1.1278913916
      -0.3206716366      -0.1011284545      -0.9777876268
      -0.9443607477       0.6420913153       0.0289439159
       0.0725889177       0.7778033379       0.0252922111
      -0.7358223563      -0.0926289276       1.3602675226
      -0.0101174000       0.0246620015      -0.6006780530
      -0.1101727998      -0.5019479740       1.0440629746
       0.5984765646      -0.0222507246      -1.1565074693
      -0.7920878880       0.2574642596       0.4078113196
      -0.2479603743      -0.0113193798       0.7261522437
       0.5199882158      -1.0123168683       0.0171865328
       0.4063345546      -0.3645232848      -0.0169863361
      -1.2644748466       0.1209719037      -0.4211531489
       0.0847556718       0.6886342823       0.8048738274
       0.4955284831      -0.6044312104       0.2756818027
      -0.8870650253      -0.6863127779       1.1547035096
       0.6433382890       0.9806688549       0.0216514099
      -0.5353665164      -0.4617343368      -0.1613414214
       0.1959655960       0.8974868988      -1.0012293577
       0.4845678565      -0.5889861298       0.2301549439
       0.5431519504      -0.8772383408       0.2562822162
      -0.5381379423       0.3331465308      -0.5715374433
       0.0670764045       0.6387920731       0.1931628424
      -0.3159727728      -0.5658863814      -0.3940299558
       0.2914339961      -0.4485525707       0.3030152307
      -0.7033829031       0.1676218213       0.7165203012
       0.1932402239       0.5517486137      -0.0405472958
       0.2018132534      -0.1642468321       0.1621702594
      -1.0283800637      -0.2547204898       0.0804222308
       0.6841988573       0.4578362905      -0.1361427443
      -0.9035964527       0.1491083533       0.8338121351
      -0.2457839638      -0.5195442681       0.2843614116
      -0.1994861303      -0.1028513871      -0.0680356109
       1.5046724873       0.0625282893      -0.8696553550
      -0.8191224887       0.2017712369      -0.1532604653
       1.2642428311      -0.0511269234      -0.1351239468
       0.6067702128       0.2207479198       0.4414286530
      -0.4142329503      -1.0617051314       0.0232009807
      -0.3084314897       0.3272521389      -0.5038411705
      -0.6956821163       0.4884802556       0.8268653626
      -0.2087247220      -1.0827985521      -0.5974180167
       0.6055721313       1.3748088024       0.9247642977
      -0.3264243337      -0.5134852909      -0.6687405666
       0.7044940415       0.7431717092       0.5663886490
       1.2678392825      -0.2509357664      -0.9478430526
      -0.8100255017       0.5175899065      -0.3192781865
       0.5481291715      -1.0175229191       0.2589871329
      -0.5071825256      -0.0110708458       0.1801922262
       1.2350964155      -0.1228434213      -0.1948671655
      -1.6765364759      -0.9016886187       0.9711298860
      -0.6013547408      -0.2595545264      -0.2583185843
       0.2341630484       0.6594705870       0.1097921682
       0.6693589875       0.3296997855      -0.1458804854
      -0.7589894501      -0.3745649690      -0.3667759859
      -0.5694852848      -0.7014635598      -0.1461097503
       0.7730908568       0.8093832389       0.5295262483
       0.4209519005      -0.5194158374       0.7966301027
      -0.0392331858      -0.2764819144       0.6958776605
      -0.4909968831      -0.4923922268      -0.8561343746
       0.6501881645       0.1745445680       0.0614042461
       0.0128641544      -0.8679793853       0.3963337302
      -0.2765288639       0.4539347387       0.5232976221
      -1.1272737948       0.0275057610      -0.2805328027
      -0.3928919625       0.7916570515       0.0988272183
       0.0226100502       0.4300623434       0.4747028016
      -0.4206448532       0.1177890632      -0.5973639564
      -0.3541823529       0.7071381329       0.8219415747
       0.7222869220       0.7623896535      -0.7785195027
       1.0322501093      -1.3308120711       0.9978569007
      -0.7104940170       0.2126673906      -0.0816016581
      -0.4469268276      -0.1878681425      -0.8246124404
      -0.3575820874       1.0302782268      -0.1797380803
      -0.0985128696       0.1883877241       0.4552622674
      -0.4812963519       0.0017738445       0.3531352507
       1.2957831680      -1.1200977802       0.4208342856
      -0.7361652046      -0.0277427390      -0.4010114018
       0.3260581967       0.4518549208      -0.1482981276
      -0.2862328779      -0.4026882082      -0.7193717315
      -0.0251956935       0.0367296046      -0.4441213729
       0.7335055811       0.3455323215      -0.0405007408
      -0.1470021053      -0.4994040503       0.7371020361
       1.2174042135      -0.8904863610      -0.5940327072
       1.0004385695       0.2636684826      -0.6222405623
      -0.9225440987      -0.8458433714       0.5251729866
      -0.9018053436      -0.8343720373       0.2772738438
       0.1467104314       1.6363537219      -0.1192763901
       0.1375797680       0.3952815984       0.2169061303
      -0.9948474794       0.3731644229       0.6689147361
       0.7844773575      -0.3287068088      -0.1957736715
      -0.1929442054       0.4616996515       0.7270212434
       0.3711794070      -0.4353541860      -0.6553434221
      -0.0396505998       0.5284833024       0.2273302714
       0.6194449312      -0.3566245192      -0.8709005963
       0.0080990438       1.2834554066      -0.8386154046
       0.8105888026      -0.5072576205       0.2455561168
      -0.1492175104      -0.1477547920      -0.7920764090
       1.4288522747      -0.6912847924       1.3481148429
      -0.7810183239      -1.3845700074      -0.5432584532
       0.8938224858      -0.0416309068       0.1831852280
       0.1233814331      -0.3248505582      -0.5300100879
      -1.2780063247      -0.0547432768       0.2408337051
      -0.1468022014       0.6673532015       0.4173861854
       0.3536539051      -0.9472074470      -0.5796559031
       0.4489341251       0.4440808828       0.2723006210
      -0.2817650153      -0.6345108669      -0.3788047709
       0.1757083678      -0.1549804725      -0.0837922823
       0.4502951269       0.2204200279      -0.2542721079
      -0.3751514576      -0.1629133184       0.1067852125
      -0.2272781808       0.0650198767       0.3410677135
      -0.6317511038      -0.2529048571       0.5313331844
      -0.9237468423       0.5667604973       0.2931270170
       0.0621925176      -0.7810508654       0.1063761368
       0.5951553619      -0.0980480084      -0.5521529492
      -0.1650778649       0.0475456126       0.0587579741
       0.6945646936       1.2134905209       0.5012191922
      -0.5716532195      -0.7623471499      -0.9926478026
       0.0850517855       0.8305751359      -0.3780855324
      -0.3051958935       0.0031584084      -0.2080642235
      -1.2962301797       1.3850822547      -0.0261462486
      -0.3078908796       0.2815335377      -0.5983793706
       0.2377619284       0.3592806329       0.7850315036
       0.3299779028      -0.5423753477       0.1215289087
       0.3841272443       0.7815864809       0.9331242845
      -0.7615150255      -0.0676118304      -0.4598453288
      -0.5836351535       0.0215820267      -0.2661491757
      -0.5318196685       0.5251973975       0.7106654989
       0.5170912493       1.1063278286      -0.7824708446
      -0.0754669384       0.1064057446      -0.1534873670
      -0.0834245669      -0.1851478887       0.3220846804
       0.2674550984      -1.0939849475       0.6208718763
       0.2556876562       0.3479856515      -0.5970862947
      -0.2768875317       0.1051844398       0.7453012587
       0.3735629630      -0.9800750344       0.2091939697
      -0.1921238195       0.2900117093       0.4529769580
       0.0925841675      -0.2674307153      -0.3055637026
      -0.4179878915       0.5320039567      -0.2290212049
      -0.5716706411       0.1745806743       0.6252742365
       0.0627737257       0.4781856642      -0.8819156843
       1.0748479830       0.4092062333      -0.2790947138
      -0.7510127662       0.0676120299       0.0805716301
      -0.4256288467      -0.4172977676      -0.4358743051
       0.1788322191       1.2499559012       0.1093858741
      -0.4166370974       0.3646495667       0.4704513513
      -0.5998522228      -0.6082744439      -0.4374893586
       0.8663438408      -0.4479802470       0.2843485010
      -1.0014927794      -0.3151333043       0.3024239863
       0.5882483833       0.6552237595       0.2346142140
      -0.2270325060      -0.0140706615      -1.0976682495
       0.3960876339      -0.4195249583       0.3256153244
      -0.7577469424       0.9519031138       0.5421036906
      -0.8308992989      -0.1121315762      -0.6826916148
       0.7190497946       1.0518377401       0.3805267467
      -0.3423727602      -0.8980530355       0.1936265160
      -0.6189394836       0.4694633710       0.0959273870
      -0.4540892999      -0.6631196110      -0.7593072180
      -0.2847863714       1.0806017663       0.7083967209
      -0.0145627490      -0.4943551189      -0.1627482284
      -0.3650799614       0.6530513489       0.7663886870
       0.2564363022      -0.3239061732      -0.8243097129
      -0.1216911573       0.2035546815       0.2945941816
       0.5206544070      -0.3202733262      -0.9617731206
      -0.3980826209       0.7477555154       0.7269618113
       0.1806777941       0.6599140223       0.6474307565
      -0.0712984096       0.3393653399      -1.7634377440
      -1.4493026577      -1.3135350711      -0.8270765183
       1.2847815039       0.9706051586       0.9981580928
       0.7103345481       0.5181358452      -0.1681756413
       0.2489349690      -1.5096214530      -0.4169437061
      -0.2251843094       0.4653542430      -0.3442371614
      -0.7968190627       0.9212573940       0.1017382098
      -1.2388626213      -0.2737272650      -0.0513236006
       0.8012979326       0.9186570350      -0.6288551450
      -0.8965317163       0.6240114711       1.1352840524
      -0.3835992881       0.6537426564      -1.0189271392
       0.1734104497      -0.2508494471      -0.9466780536
       0.4786136337       1.1109541887      -0.5424722636
      -0.6869527493      -0.8771896313       1.1413261585
      -0.6591367055       0.2288833104       1.2257722939
       0.0453034271      -0.7861600681      -0.4454067257
       0.4579362010       0.7703492371      -0.0852902979
       0.1184380497      -0.0079425370      -0.2652510596
       0.4935267722       0.0684878486       0.3795899720
       1.0294167074      -0.5778194918      -0.7512835441
      -0.6757715023      -0.2533864317       1.1257467535
       0.8876045854       1.1731990539      -0.1251610949
      -1.0537440056      -1.0205203310      -0.0892193182
      -0.3302022814      -0.1025526578      -0.6524217389
       0.0495030877       0.3407922226       0.7845133508
      -0.4208068311      -0.3281788900      -1.1552290936
       0.2084139301      -0.7780428280       1.1174046050
      -0.3879056385      -1.3496984515      -0.1625722890
       0.6502024050      -0.4950114585       0.9821444646
      -0.2104565439       0.6764625318      -1.1813728552
       0.3947525493       0.0951878993       0.9047855059
      -0.7583321692      -0.2925510721      -1.2555599413
       0.8884329676      -0.6437753083       1.1073166736
      -1.2878059269      -0.8532485634       0.4623460511
       0.6186539135       1.0922716175       1.1425413415
       0.1182051725      -0.8667233811      -0.5368210539
      -0.6276741412       0.5982437502      -0.6305561310
       0.6248085292      -0.8292575491       0.8753926261
      -0.0964373617       0.7370003457      -0.3611856962
       1.0096496330      -0.7669690714       0.8950294985
      -0.4990769626      -1.3391879703       0.1677737084
       0.3226687889       0.3318289067       0.0162175503
      -0.0225269367      -0.8309721867      -0.2060166356
       1.4839083822       0.8443765519      -0.3144498205
       0.5915881096      -0.6103759161      -1.1866341109
      -0.3968945279       0.4370106006       0.5145252484
      -0.7809129828      -0.2968961387      -0.3511239661
       0.5030211948      -0.5010186124      -0.2180476285
      -0.2926834824       0.4963148534      -0.8758087974
      -0.5294845309       1.7237935440       0.1610149713
       0.9862492123      -1.5544690628      -0.2442722304
      -0.0219370536       0.3930230875      -0.1252209756
       0.1987872132      -0.3799163861      -0.3251680497
      -0.5643914238       0.5041027887       0.2083547867
       0.3175451874      -0.8015721327      -0.6684966798
       0.0737992783       0.9885875911      -0.4987657683
      -0.3002824762       0.5520407731      -0.0447827999
      -1.0742736464      -0.8640440586      -0.7980111604
       0.5672646921       0.4014474549       1.3909689707
      -0.0388305765      -0.4597124294       0.1640936607
       0.1323602986       0.4064025430      -0.0528930025
      -0.1054630546      -0.5320024506      -0.9938522025
       0.8189555992       0.3592961573       0.8071945973
      -0.4076659891      -0.9058536973      -0.5833085385
       0.3333957893      -0.2483437291       0.8208240103
      -0.0751761819       0.0345089133       0.4418756409
      -1.3611991862       0.4209286535      -0.2064455482
      -0.0947561671      -0.7342228271      -0.1210329728
      -0.1574323927      -0.3058227403       0.8729752396
       0.6265781270       0.3371223560       0.0600586257
      -0.3550049035      -0.0210357977       0.2085147342
      -0.4002681697      -0.7222301051      -0.2708414338
      -0.0919717074       0.3246644134       0.1094411165
       0.4426719587       0.7235097499       0.2078417820
       0.8791625955      -0.8566651515      -1.6362428232
      -0.0403019748       0.1853084854      -0.0298100210
       1.0076501527      -0.9890149084      -0.4408791674
       0.0174505139      -0.1184141778       0.3030053268
       0.3185650296      -0.0758026843       0.5634540910
       0.3982018074       0.5172191041      -0.5574754578
      -0.6470989319       0.1608487472       0.6537454800
      -0.5411865522       1.0147480249       0.9611051548
      -1.2925669963       0.4238677273      -0.6201855306
       0.0365196965      -0.3987583667      -0.0476108677
       0.4964252117       0.2728857001      -0.6238698690
       0.9963037083      -0.8563146937       1.0752575512
      -0.0690182897      -0.6947831149      -0.1589518496
       0.2428743737      -0.6779066530      -0.7766913043
      -0.2576363134       0.5887088721       1.0414401956
      -0.8364265359       0.1732226970      -0.4392220044
      -0.1284122789       0.9830844001       0.1539918817
      -0.0577681967      -0.7230214214      -0.3178246537
       0.6362870804      -0.3242780205       1.1223562346
       0.8247944817      -0.5937218235      -0.4923539086
      -0.8776531138      -0.8994758861       0.4475433275
      -0.3290870870       0.4452088619      -0.2559229151
      -0.2368422842      -0.1220458300      -0.5428636225
       0.1155873668       0.8927765400      -0.8581169223
      -0.7687824636       0.4434201759      -0.5350779717
       0.7341306774      -0.8068129418       0.4911966079
       0.1856247932       1.0775255536      -0.4461180522
      -0.8450272650      -0.5003522420       0.0481832672
       0.4739632461       0.0748200340       0.1771206986
      -0.8155388458      -1.1341650815       0.2646764175
       0.5774073944       1.2456522485       0.6727630565
       0.3500435381      -0.9167273876       0.1083621477
       0.2708526297      -0.2043748965       0.1155741637
      -0.3946094303       1.1029311399       1.0210233320
       0.2130605737      -0.9230561014       0.5580242758
       0.1562061807      -0.6470001238      -0.8966709639
       0.3106377257       0.7575530941       0.2099476233
       0.0623447884       0.4805201538      -1.0502340432
      -0.4157371598       0.0675036706      -0.1806255637
      -0.1828097831      -0.3550720107       0.4348325601
       0.4417574907       0.3835959405      -0.0859919337
       0.4578350441      -0.3908495575       0.2394967523
       0.4103325343      -0.0369691850       0.4616871953
      -0.3096191584      -0.5071267420      -0.3464619258
      -0.3483214162       0.9666760911       0.5114543063
      -0.4240784318       0.2634935499      -0.2497376106
       0.9132361757      -0.5057096972      -0.3153571294
       0.1396940320      -0.8503815809       0.3917979784
       0.6844931020       0.8385545797      -0.2128769068
       0.9611719139       0.2528150866      -0.4089220271
       0.6930717116      -1.0018466359      -0.6520184475
      -0.7030362348       0.5310857552       0.5256108828
       0.2719741184       0.0967482878       0.0957909738
       0.1706776667       0.6816281664       0.5223458764
       0.1030682486      -0.3547380118      -0.7732236125
      -0.9909160524      -0.3665243925       0.9087729040
      -0.2625157322       0.0408187568       1.0518148532
      -0.7136625646       0.0206364323      -1.0951915274
       0.8224648243      -0.4496785862       0.6926748178
       0.1499216641      -0.1909247743      -0.9465979425
      -0.3419873296      -0.0118744814       0.5369071417
      -0.2571500080       0.7033353477      -0.4815254136
       1.2125750007      -0.3826582428       0.1359120019
      -0.4595052556      -0.8866497797       0.7725525088
       0.4370978681       0.1024796395      -0.0436639355
       0.9996616659       0.6755231821       0.5205201634
      -0.8371568810       0.2974997994      -0.1937677206
       0.1270815546       0.7866227751       0.7281537608
      -0.4565656440      -0.4196711691      -0.9603939291
       0.0168731944      -0.3020035458       0.6205274525
      -0.6546851564      -0.0154722150      -0.1939505203
      -0.1957644636      -0.7368453489      -0.1907592379
      -0.4429101026       0.0266332942      -0.0586836411
      -0.0564517318      -0.0943887863      -0.6709152522
       0.4421794119       0.9920284613       0.2662719692
      -0.0704626076      -0.4582744355       0.6321049116
      -0.1210263693      -0.6618347484      -0.0006015552
      -0.0019039610      -0.7512548730       0.3660509575
      -0.6131162829       0.1295985088      -0.3297440199
       0.5656517205      -0.6837943656      -0.4504544089
      -0.1073922329       0.5322951811       1.1872621546
       0.1412625308      -0.4719860872      -0.6636351370
      -0.5111405375      -0.2995157354       1.5765527797
      -0.2885369488      -0.0064778578      -1.0867907778
      -0.1383317175      -0.8012064067       1.1185825501
       0.1331517465       0.6196993918      -1.1671799710
      -0.4185581709      -0.2803293173       0.9654069463
       0.2891684445      -0.2223992658      -1.5526063730
       0.5121187931       0.0317532024       0.8582219010
      -0.6916166757       0.1541166064      -0.5662505434
       0.5158556597       0.9330794614       0.8433307876
      -1.4333843868      -0.7751828136      -0.5931404431
       0.6274684375      -0.1912130611       1.8407614736
       0.2438290594       0.2745557244      -0.2340882520
      -0.3031978644      -0.4425733392      -0.7593208531
       0.8797196046      -0.4444508953      -0.8612104040
       0.6783446600       0.6535088317      -0.4080609837
      -0.4736645849       0.3212480648      -1.3207269019
       0.8172287743       0.8622678937       0.5239714275
      -0.5588229764      -0.2797436575       0.5345152331
       0.4520561504       0.6342153757      -0.3431477400
       0.4804498494      -0.0970030678      -0.0110793614
      -0.6784447769       0.3889842133       0.6218198308
       0.4403160834      -0.4569978667       0.8552579880
      -1.2827449911       0.0835786645      -0.5348860889
       0.5654437574      -0.3818186718      -0.6450267594
       0.1063397239       0.7676675798       0.7748246659
       0.9779799928      -1.0722288418       0.4804304202
      -0.4431473715       0.6249865886       0.0062820160
      -0.4532916178       0.2922925087      -0.7511619659
       0.2822495157      -0.7310218070       1.2305974163
       0.2458025954      -0.2160526186      -0.3312833889
      -0.0566020167      -0.3239713899       0.9183817976
       0.6914806699      -0.5333967724      -0.2734452360
      -0.3128582967      -0.8036779632       0.2589066758
       0.3142152759       0.5639004529      -0.7432134113
      -0.2002788379      -0.4185672204       0.1465107042
      -0.2812190940       0.9447417547      -0.4501576405
      -0.0924441255      -0.3223172328      -0.3131943232
      -0.2598018362       0.4473520527      -0.6520206179
       0.1007962859       0.3897796979       0.5429768646
       0.6039199702       0.2460253404      -0.6140008581
       0.8893263653       0.1144415115      -0.2695892132
      -1.4627109206      -0.5016401326      -0.7489157797
       0.7727543213       0.4631296818       0.3840756715
       0.3385238672      -0.2805877511      -0.2678341673
      -1.1192834872       0.1962662569       0.1739712528
      -0.3335926309       0.0862642015      -0.0720512638
      -0.6639449339       0.2738497495       0.6980402239
      -0.8643612974      -0.7065710689       0.3427920922
       1.2082512141       0.0987160059      -0.9799534185
       0.0903235786      -1.0024636428       0.2396235988
       0.6558299172       0.0654347499       0.4920877531
       0.2926001985       0.1046396129      -0.1342635079
       0.0254095857       0.5555332623       0.3879846161
      -0.7050521986       1.3673906595       0.0717638291
      -0.2005773982      -0.6462319970       0.1184834869
      -0.8893688666      -0.8790924235      -0.3806443176
       0.4995270654       1.0684442626       0.8912643206
       0.0429559538      -0.1878910264      -0.2423293870
      -0.1607618914       0.1115922880      -0.3665693295
       0.4430181013      -0.1008639937      -0.3465068109
      -0.4651991505      -0.3180555146       1.3305940709
       0.1114194035       0.3044286643      -0.6265699077
      -0.5719516124       0.7810551007      -0.6573022576
       0.5337541736       0.6869734653       0.1055752838
      -0.8189086485      -1.2330037341      -0.0057317437
       1.3631464050      -0.5148427186      -0.6520993338
       0.2843235982       1.0692151965      -0.3880141953
       0.7655396807       0.5789422856       0.0145827525
      -0.1907992296       0.7697343736      -0.5885002234
      -0.1451141981       0.5452646321       0.2286848560
       0.4150917325      -0.6175302302       0.7665094103
       0.3662366662       0.4862135062       0.0838935872
       0.3556368710      -0.7141544912       0.1890028983
      -0.0473394799      -0.4079106089      -0.6022746841
       0.1466401973      -0.0769152284       1.1159721885
       0.5712524588       1.1986583170      -0.8430935721
      -0.8658728688       0.1685743372      -0.3093141628
      -1.2525537636      -0.4953948859       0.1505707521
       0.7975322538      -0.3171217288       1.4832909413
      -0.5354765054      -0.6987844430      -0.6882801269
       0.3776369013      -0.0929715981       0.2613226712
      -0.3846189668       0.4559804229      -1.1004356028
      -0.1708836301       0.9998088766       0.2403569417
      -0.0140674285       0.3735691148       0.0565647703
       0.3617669546       0.4806526543      -0.4208374560
       0.3506644837       0.2308810007       0.2746750303
      -0.3181261679      -0.3432812455      -0.4445514136
       0.7646221116       0.0813366110       0.5232259526
      -1.2848735102      -0.7357784443       0.7893067749
       0.9246959483       0.6389238493      -0.9855451165
       0.8510648481      -0.3353329293      -0.1027977051
      -0.9204339434      -0.9593967038      -0.3616183495
       1.3345189472       0.9459010592       0.3445523958
       0.3162091461      -0.2856811079       0.8124495481
      -1.2176949680       0.0300349276      -1.0490954672
       0.9508699590       1.1608035247      -0.4472881017
       0.0302037127      -0.4775158885      -1.2298852361
       0.1336275769      -0.8806942372       0.8714613859
       0.2802888752       0.7299265074      -0.2116016094
       0.4619129739      -0.5782862178      -0.6190305904
      -0.1433364103       0.9531997024      -0.2173838638
      -1.1338150435       0.2831131995       0.1828237535
       0.6229075365      -0.1348378054      -0.0865167516
      -0.7246003844       0.5669277101      -0.0350160579
       0.2981855373       0.0856158124      -0.0314836208
       0.2560982449      -0.0647542689      -0.2878426051
       0.4161846501      -0.4005298101      -0.0387405539
       0.7805481140       0.3712222836      -0.7840857994
      -1.4226227555       0.4383419833       0.7444717708
      -0.7641748309      -0.7018783565       0.2037890337
       0.1238503030       1.4180132506       0.0288459603
      -1.2175309570      -0.4373937760      -1.0739436723
       0.6734576242       0.5872376638      -0.0832530373
       0.4062263977      -1.3679326069      -0.2011289620
      -0.6582182823       0.9085087708       0.4365307355
       0.3138266146      -0.3452846887       0.0699097723
      -0.4834304237      -0.1339924672       0.0945589968
       0.2930819118       0.6797778876       0.3789905746
      -0.3736017032      -0.5810323272       0.4364035703
       0.3539225095      -0.2071524010      -0.0722781654
      -1.1481784182       0.8307401736       0.1708571185
       0.1692674660       0.1594278147      -0.0781465041
      -0.4122704510       0.4241505365      -0.4754624881
       0.4850805928      -0.3096152250       0.7464120753
       0.3586649868       0.4967797982       0.3268572160
       0.6846997541       0.0670659428      -0.6242849554
       0.4523618309      -0.2955266451      -1.1113979940
       0.7269511790      -0.3448130485       0.9428358557
      -0.2866917157      -0.1730786641      -0.3437054096
       0.9514404814      -0.3297669412       0.6503184050
      -1.1866196979       0.3241362879      -1.5436892953
       0.2143124196      -0.5232059055       0.1382490572
       0.2516056445       0.5398190748      -0.0800572490
      -0.0557306480       0.3384195946      -0.5150003935
      -0.4303835332      -0.1513581177       0.1520507230
       0.8905247386       0.0683537831       0.3298628834
      -1.0698021163      -0.8232083038       0.0797007636
       0.9184662389       0.5657545724      -1.0833796510
       0.1287647513       0.6455767652       0.4496673831
      -0.7653186179      -0.1873038057       0.5811720505
       0.3473644948      -0.7823919466       0.5639912840
      -0.7800409000       0.5774671992      -0.7574287056
       0.6671197164      -0.0981747213       0.8093831193
       0.1963190783      -0.7842955200       0.5915968580
       0.1328513107       0.6711169952      -0.8784983356
      -1.0625598008       0.2825364752      -0.0461049237
       0.2144763796       0.4268605194      -0.2653315569
       0.6675846384      -0.9965523174      -0.7837810393
       0.4124585069       0.8984726405       0.4106543952
      -0.1935309814      -0.0670739496      -0.4179830821
       0.4836644011      -0.3199786071       0.8604545955
       0.0737657423       0.1057154197      -0.8297993009
      -0.2517757296       0.5399366077       0.8118759564
       0.2893709222      -0.0571665852      -0.6493503665
      -0.0045567244       0.4796897230      -0.3411535019
       0.2149382333      -0.4196790641       0.0033827306
      -0.7667709320      -0.2583330052       0.2660077398
       0.3198758473      -0.1541300758      -1.2107943381
       0.1228389832       0.0495184471       0.7673709224
      -0.0337205142       2.1259998535       0.3166728280
      -0.5597661255       0.0545664132      -0.8682465258
      -0.0591138343       0.2799463324       0.0586204425
       0.0141773614      -0.9490747159       0.3646657753
       0.7178560724       0.2797766288       0.6585706876
       0.2974871598       0.3188808666      -0.0703750046
       0.4415116443       0.1135808859      -0.2920584839
      -0.9099268121      -0.4814231815       0.4761962741"""))

      def test_force_mixing_distance_ramp(self):
         pot = Potential('ForceMixing method=force_mixing_super_smooth transition_hops=2 cluster_hopping_nneighb_only=T cluster_heuristics_nneighb_only=T buffer_hops=0 qm_args_str={single_cluster=T cluster_calc_connect=T}',
                                 pot1=self.pot1, pot2=self.pot2)
         f = fzeros((3,self.at.n))
         pot.calc(self.at, force=f)
         self.assertArrayAlmostEqual(f, string_to_array("""
4.264523724577462538e-01 -1.461417424951666044e+00 -5.027997001248233344e-01
1.036734719171164415e-01 1.301462039650632452e+00 5.790116361205615014e-01
-2.033194347303170224e-01 -1.001253662097094876e+00 3.714197275474521165e-01
-5.417083399115099063e-01 9.262684835601788791e-01 5.100573147093399040e-01
7.389708092872480094e-01 3.235243112762054007e-01 -1.013679953758621355e-01
2.060440319289136291e-01 -2.957811273153642118e-01 -9.782502242209623056e-02
4.497622462233324403e-01 -7.812570899758540799e-01 -1.463440414387585431e+00
6.218315281619145596e-01 9.118190860074992798e-01 6.883430195659274720e-01
2.055650048680677799e-01 -4.175681300051410916e-01 9.739422515450043560e-01
-1.979846656403488669e+00 8.814605006475649818e-01 -1.012784452781553668e+01
-1.837016432286894929e+00 -1.494307308667541800e+00 -4.606041705202654768e+00
1.280640890507959151e+00 9.584571071391411179e-01 9.251532756239110311e-01
-4.275075993424560239e-01 -8.490839599085078238e-01 1.685484283008453987e+00
9.797860906157742633e-01 -1.492499691634875614e+00 4.866199696811048803e-01
-7.740590842637798397e-02 -1.012583982347062372e+00 -1.372962865406739175e+00
6.046978108008170816e-01 -4.596437085921544208e-01 1.264127951293718244e+00
-1.687565674174736075e-01 -1.096972393996429096e+00 -3.842891729525352851e-01
8.878858522834061073e-01 5.704335744123517626e-01 6.266331770442300220e-02
-1.654941585934271986e+00 2.936701172125089632e-01 2.233292416320496743e-01
9.462083392129717607e-01 -1.834713514645846741e-02 2.397467768271771826e-01
-6.916953283992086643e-01 -8.215245039037718522e-01 -6.936364930887356506e-01
6.050512837264874211e-01 -5.409458056209190957e-01 1.942164951160376685e+00
6.499684704987189754e-01 -2.749568234637983499e-01 -3.413196704057441244e-01
-2.785780484171618832e-01 -5.820799323550516213e-01 1.534857106442360619e+00
-5.893911768428141684e-01 7.623555451571591979e-01 -9.019987370974874574e-01
1.341559255972587517e+00 -3.617490962483748684e-01 -5.914013137168052658e-01
5.546774680224253462e-02 5.042908005170925190e-01 1.154730796277210869e+00
-6.478831945876053489e-01 -4.007012028451316654e-01 -6.734591940003895605e-01
-9.418822156581555527e-01 6.452137054422718565e-01 2.816272322541943027e-02
7.643592837003303220e-02 8.190247506074977979e-01 2.663262790606644309e-02
-7.451434920081070334e-01 -9.772029459971118270e-02 1.361652438245677388e+00
-1.064910232061465171e-02 2.595806986184332615e-02 -6.322456383663785928e-01
-1.093004183390654249e-01 -5.230550988852095129e-01 1.081449910300946726e+00
-1.210784917925732951e-01 -9.993759887931966901e+00 -1.653410859129610611e+00
-7.920878879724502220e-01 2.574642595826655556e-01 4.078113196329200552e-01
-2.479603742738925864e-01 -1.131937976586185057e-02 7.261522436735844410e-01
-3.379771258573797676e-01 -4.480156711605385311e+00 -9.407342313150620638e-01
4.063345546220891724e-01 -3.645232848005280357e-01 -1.698633609738976064e-02
-1.264474846648252493e+00 1.209719037470197711e-01 -4.211531488571876314e-01
8.475567183940402316e-02 6.886342823214567366e-01 8.048738273859653569e-01
-1.948549101357259783e-01 -1.856527857216288879e+00 -8.240164644793412929e-01
-8.870650253174126032e-01 -6.863127779250932647e-01 1.154703509589253052e+00
6.433382890361667439e-01 9.806688548852262599e-01 2.165140988043774695e-02
-5.353665164158689649e-01 -4.617343368109147472e-01 -1.613414213788071649e-01
1.959655959902439903e-01 8.974868988115998203e-01 -1.001229357678060650e+00
4.845678565251248515e-01 -5.889861298387168986e-01 2.301549438598680386e-01
5.431519503862616283e-01 -8.772383407610607753e-01 2.562822162139495874e-01
-5.381379422757590625e-01 3.331465307562240241e-01 -5.715374433376686492e-01
6.707640448878336714e-02 6.387920731209375536e-01 1.931628424434708169e-01
-3.159727727719864876e-01 -5.658863813688465028e-01 -3.940299557896927984e-01
2.914339960636450622e-01 -4.485525706877185326e-01 3.030152307350109098e-01
-7.033829030962795992e-01 1.676218212704428667e-01 7.165203012356260448e-01
1.932402239371935615e-01 5.517486136725523460e-01 -4.054729583385843772e-02
2.018132534090342489e-01 -1.642468321196975944e-01 1.621702594159017963e-01
-1.028380063734548600e+00 -2.547204898324186284e-01 8.042223081127633511e-02
6.841988573283455111e-01 4.578362904778195408e-01 -1.361427443330771869e-01
-1.623444816685342129e-01 -9.527248147501473641e-01 2.034269172305364748e+00
-2.357755471625296861e-01 -1.947274651937064949e+00 5.074848094423303202e-01
-1.994861302948306159e-01 -1.028513870674774844e-01 -6.803561090449658511e-02
1.504672487299640649e+00 6.252828933419601887e-02 -8.696553550132667398e-01
-8.363166732321877239e-01 2.062388669840952149e-01 -1.575230873614250071e-01
9.695862930883376585e-01 -1.741902577386499118e+00 -1.430532164134301065e-01
6.067702127780362220e-01 2.207479198375607576e-01 4.414286529612553855e-01
-4.142329503218606379e-01 -1.061705131429709015e+00 2.320098071968541148e-02
-3.084314897166687830e-01 3.272521388777862028e-01 -5.038411704945316982e-01
-6.956821162988992002e-01 4.884802555508295230e-01 8.268653625691332332e-01
-2.087247220045442830e-01 -1.082798552054643704e+00 -5.974180167378839723e-01
3.505792172111327210e-01 2.988275312144897278e+00 1.004256476519634234e+00
-3.264243336566745590e-01 -5.134852908932960869e-01 -6.687405665766810170e-01
7.044940414510036941e-01 7.431717091703157196e-01 5.663886490353532421e-01
1.267839282485339281e+00 -2.509357663988042719e-01 -9.478430525800458639e-01
-7.284762022038437923e-01 2.034812704267763284e+00 -5.915172172314326904e-01
5.481291715318089564e-01 -1.017522919100658729e+00 2.589871328665361294e-01
-5.071825255907731478e-01 -1.107084581061848483e-02 1.801922261667820435e-01
1.235096415547707771e+00 -1.228434212789415586e-01 -1.948671654516492269e-01
-1.676536475931804215e+00 -9.016886187237324046e-01 9.711298860149324419e-01
-6.013547408197126298e-01 -2.595545263985038487e-01 -2.583185843418818184e-01
2.341630484147757263e-01 6.594705869630634876e-01 1.097921682483426475e-01
6.693589874909163795e-01 3.296997855331681437e-01 -1.458804853942877455e-01
-7.589894501056104748e-01 -3.745649690286956823e-01 -3.667759858825365704e-01
-5.694852848149050129e-01 -7.014635598260714477e-01 -1.461097503171637502e-01
7.730908568336684050e-01 8.093832389471493638e-01 5.295262483279533638e-01
4.209519004515664409e-01 -5.194158374462115590e-01 7.966301027410100932e-01
-3.923318580727763738e-02 -2.764819143578973182e-01 6.958776604972746194e-01
-4.909968830817875718e-01 -4.923922268489280274e-01 -8.561343746371828933e-01
6.501881644695083340e-01 1.745445680288313373e-01 6.140424614998627600e-02
1.286415442329588536e-02 -8.679793853433597217e-01 3.963337301815616498e-01
-2.765288638522067632e-01 4.539347387355590624e-01 5.232976221217662260e-01
-1.127273794828009379e+00 2.750576099420087156e-02 -2.805328026667539554e-01
-3.928919625208990851e-01 7.916570514650297374e-01 9.882721828703197597e-02
2.261005015263778395e-02 4.300623433623462222e-01 4.747028015686140523e-01
-4.206448531977434735e-01 1.177890631780072056e-01 -5.973639563813376574e-01
-3.541823528650547281e-01 7.071381328999946092e-01 8.219415746833688141e-01
7.222869219841110411e-01 7.623896534686371851e-01 -7.785195027488702735e-01
1.032250109321413589e+00 -1.330812071107438488e+00 9.978569006954014498e-01
-1.186408106815805441e+00 1.002389284809950531e+01 4.865207055068743736e-01
-4.629937794923395522e-01 -1.951965886576619846e-01 -8.542709886348229809e-01
-3.763022666946727046e-01 1.084215473144183983e+00 -1.891477493498474050e-01
-9.638962182466709327e-02 1.896612029081254647e-01 4.521041052933073012e-01
-5.065002076178554136e-01 1.866734750982312462e-03 3.716277447699235514e-01
1.325457114024300997e+00 -1.145714957595413530e+00 4.303191721056462971e-01
-1.107082195115890677e+00 2.387634147838106347e-01 -7.483403415399882963e-01
3.209645384099155674e-01 4.566641580647250453e-01 -1.493028984700823514e-01
-3.012757181642460869e-01 -4.238513059021735696e-01 -7.571779894027360625e-01
5.802672352949492796e-01 1.272577308046133782e+00 -1.597488743070913042e+00
7.153958484694633579e-01 5.746424818993072225e-01 -1.944062538937482376e+00
-1.488781327070437532e-01 -5.117840066745741323e-01 7.534656610540353405e-01
9.708470495759989882e-01 -9.036548662812098343e-01 -2.372822600126840875e+00
1.000438569483613538e+00 2.636684826319803387e-01 -6.222405622511095746e-01
-9.225440987488081612e-01 -8.458433714342770138e-01 5.251729865924709140e-01
-9.018053436031100434e-01 -8.343720372888882020e-01 2.772738437803483613e-01
1.467104313667306481e-01 1.636353721892447322e+00 -1.192763901167255836e-01
1.375797679827665265e-01 3.952815983653040766e-01 2.169061302719349493e-01
-9.948474794196370263e-01 3.731644229187117690e-01 6.689147360754290617e-01
7.844773574801600091e-01 -3.287068088000901223e-01 -1.957736715454670706e-01
-1.929442054271401918e-01 4.616996515299482406e-01 7.270212434007218283e-01
3.711794069940699825e-01 -4.353541859812537496e-01 -6.553434221318726882e-01
-3.965059984626821687e-02 5.284833023948120045e-01 2.273302713596475344e-01
6.194449312100943628e-01 -3.566245191698468320e-01 -8.709005962740447027e-01
-7.033079712499187286e-01 1.758858381554761818e+00 9.099921720614057463e+00
9.092224404172555285e-02 7.417500407586337108e-01 1.484015777830457017e+00
-1.542280644196856176e-01 -1.528827650845354658e-01 -8.186012234615521743e-01
4.327316072877185338e-01 3.102493925637284655e-01 4.881641140660991418e+00
-8.074512598544134612e-01 -1.429559773944449219e+00 -5.604801408001656160e-01
-5.833885692753337793e-02 3.373344444163158151e+00 1.054708581833963432e+00
1.284348939079641150e-01 -3.366407072712243043e-01 -5.485834321224520860e-01
-1.279059026246071706e+00 -5.405304746514018321e-02 2.425983847166533480e-01
-1.468022014093981753e-01 6.673532015082527247e-01 4.173861854240406011e-01
3.656520874812618227e-01 -9.837346309967065494e-01 -6.039738528382387894e-01
-9.217047044795641852e+00 -1.993350378952962187e-01 -5.049338704141356216e-01
-2.817650153407556535e-01 -6.345108668687764464e-01 -3.788047709089076287e-01
1.757083678086171219e-01 -1.549804725216352153e-01 -8.379228227939797269e-02
4.502951268613961666e-01 2.204200279462115164e-01 -2.542721078589882389e-01
-3.751514576225576381e-01 -1.629133184155396874e-01 1.067852124637380107e-01
-3.713173377927931007e+00 -8.078244702262824539e-01 -5.249771281780486465e-01
-6.317511038262370038e-01 -2.529048571334780449e-01 5.313331843925576115e-01
-1.998487575858927778e+00 -1.254297785464804638e-01 -8.315386597959549819e-01
6.219251763769301916e-02 -7.810508654121064431e-01 1.063761367858362533e-01
5.951553618946365809e-01 -9.804800838600008506e-02 -5.521529492261383343e-01
-1.650778648983032837e-01 4.754561263683098477e-02 5.875797408835763125e-02
6.945646935848399650e-01 1.213490520861109312e+00 5.012191922300252012e-01
-5.716532194725910676e-01 -7.623471498574926342e-01 -9.926478025720827381e-01
8.505178546542808971e-02 8.305751359422630653e-01 -3.780855323875466634e-01
-3.051958934840572346e-01 3.158408410292166697e-03 -2.080642235492024261e-01
-1.296230179681692185e+00 1.385082254687335590e+00 -2.614624855168790410e-02
-3.078908795695567968e-01 2.815335376971425663e-01 -5.983793706366083986e-01
2.377619284478596495e-01 3.592806329012592226e-01 7.850315035733826941e-01
3.299779027696430056e-01 -5.423753476975051724e-01 1.215289086683355585e-01
3.841272443127268787e-01 7.815864809256257706e-01 9.331242844698134364e-01
-7.615150254573621602e-01 -6.761183038360754627e-02 -4.598453288091062241e-01
-5.836351535343852204e-01 2.158202665178872090e-02 -2.661491756862466862e-01
-5.318196684587472856e-01 5.251973975122382621e-01 7.106654989142943135e-01
-6.455901327456046168e-01 1.765324187712553172e+00 5.171929351893566063e-01
-1.812696434636362763e+00 1.097414004724502762e-01 1.195430607284005964e-01
-8.342456685496997537e-02 -1.851478886797742751e-01 3.220846803502028788e-01
2.674550984338109894e-01 -1.093984947506438266e+00 6.208718763152653031e-01
2.556876562470923320e-01 3.479856515357956370e-01 -5.970862947292824296e-01
-2.768875316840152534e-01 1.051844398053896068e-01 7.453012586760410230e-01
3.819732230234543779e-01 -1.001504197830396015e+00 2.130979451979518602e-01
-1.806126263641727014e+00 2.522658005010225413e-02 4.423672769305200814e-01
-1.094140453207652630e+00 -1.330749514219140783e+00 -9.483549438297738110e-01
-4.179878915131227668e-01 5.320039567279214454e-01 -2.290212048823968283e-01
-5.716706410870491251e-01 1.745806743428103547e-01 6.252742365278609604e-01
6.277372572808334605e-02 4.781856641877967329e-01 -8.819156843483131336e-01
1.074847983048968558e+00 4.092062332926198476e-01 -2.790947138283408901e-01
-7.510127662164268747e-01 6.761202986701994599e-02 8.057163006688215778e-02
-4.256288467150591281e-01 -4.172977675679779841e-01 -4.358743050642643890e-01
1.788322191354368151e-01 1.249955901241237566e+00 1.093858740941769381e-01
-4.166370974441244712e-01 3.646495666524470214e-01 4.704513513005470493e-01
-5.998522227584349720e-01 -6.082744438683845845e-01 -4.374893585940776153e-01
8.663438407955620368e-01 -4.479802470288466454e-01 2.843485010399623980e-01
-1.001492779413569956e+00 -3.151333042933955331e-01 3.024239863404430406e-01
5.882483833119812600e-01 6.552237594841570090e-01 2.346142140377798224e-01
-2.270325060220937941e-01 -1.407066146947973800e-02 -1.097668249497706805e+00
3.960876338685003217e-01 -4.195249583386050363e-01 3.256153244206766995e-01
-7.577469424423826094e-01 9.519031137861008185e-01 5.421036906446741188e-01
-8.308992988911071409e-01 -1.121315761880319339e-01 -6.826916147722443817e-01
7.190497946047312849e-01 1.051837740110172614e+00 3.805267467374159884e-01
-3.423727602310845941e-01 -8.980530354722062736e-01 1.936265160473200908e-01
-6.189394835598026612e-01 4.694633709607154981e-01 9.592738703807372758e-02
-4.540892998656984592e-01 -6.631196109657279125e-01 -7.593072180245805658e-01
-2.847863713776441763e-01 1.080601766316876100e+00 7.083967208775298063e-01
-1.456274897792881641e-02 -4.943551189035880755e-01 -1.627482284023355208e-01
-3.650799613921690279e-01 6.530513489120022275e-01 7.663886870336080737e-01
2.564363021525573050e-01 -3.239061732446765762e-01 -8.243097128933898343e-01
-1.216911572560392951e-01 2.035546815277790855e-01 2.945941816250650103e-01
5.206544069620443471e-01 -3.202733262034493378e-01 -9.617731206488837614e-01
-3.980826209169975316e-01 7.477555153672919008e-01 7.269618113072547905e-01
1.806777941016647349e-01 6.599140223255672844e-01 6.474307565428746303e-01
-7.129840960502918656e-02 3.393653398991834935e-01 -1.763437743994018758e+00
-1.449302657674113703e+00 -1.313535071099646734e+00 -8.270765182781196767e-01
1.284781503854362406e+00 9.706051586228255923e-01 9.981580928134502484e-01
7.103345480749432284e-01 5.181358451719511038e-01 -1.681756412732751116e-01
2.489349689895739293e-01 -1.509621452977644385e+00 -4.169437061273204015e-01
-2.251843093786650429e-01 4.653542430141744402e-01 -3.442371614453878514e-01
-7.968190627052106345e-01 9.212573939663155631e-01 1.017382098175726585e-01
-1.238862621292364574e+00 -2.737272649828852433e-01 -5.132360061910314075e-02
8.012979325917728435e-01 9.186570350252198924e-01 -6.288551450122059627e-01
-8.965317162625581204e-01 6.240114710673102527e-01 1.135284052390594933e+00
-3.835992881450501213e-01 6.537426564139771079e-01 -1.018927139151212247e+00
1.734104497069120787e-01 -2.508494471066370357e-01 -9.466780535582255407e-01
4.786136336574763894e-01 1.110954188655924435e+00 -5.424722636494304639e-01
-6.869527492626544829e-01 -8.771896313298231940e-01 1.141326158459191609e+00
-6.591367054663420655e-01 2.288833103928941404e-01 1.225772293920781131e+00
4.530342714925360975e-02 -7.861600681269188540e-01 -4.454067257164995297e-01
4.579362009608381467e-01 7.703492370575789616e-01 -8.529029789681707996e-02
1.184380496651963044e-01 -7.942537009182598640e-03 -2.652510596177231061e-01
4.935267722228544041e-01 6.848784857581566299e-02 3.795899719645806636e-01
1.029416707372149631e+00 -5.778194918095601595e-01 -7.512835441103710155e-01
-6.757715022807504468e-01 -2.533864316951060514e-01 1.125746753471171147e+00
8.876045854067670104e-01 1.173199053919610213e+00 -1.251610949183842580e-01
-1.053744005649027038e+00 -1.020520331013767601e+00 -8.921931822894364761e-02
-3.302022814305777820e-01 -1.025526577584838922e-01 -6.524217389119120192e-01
4.950308773044625327e-02 3.407922226022034318e-01 7.845133508334536376e-01
-4.208068310672750112e-01 -3.281788899932651660e-01 -1.155229093587997857e+00
2.084139300802926864e-01 -7.780428280337201574e-01 1.117404605012024543e+00
-3.879056385026804499e-01 -1.349698451504580898e+00 -1.625722889718968778e-01
6.502024049532538763e-01 -4.950114585155072078e-01 9.821444645571102416e-01
-2.104565439231418089e-01 6.764625317840867735e-01 -1.181372855151703982e+00
3.947525492914332745e-01 9.518789934887393611e-02 9.047855058638966108e-01
-7.583321691844631118e-01 -2.925510721338584474e-01 -1.255559941300687488e+00
8.884329675679281113e-01 -6.437753083416705158e-01 1.107316673632091186e+00
-1.287805926944953772e+00 -8.532485634222345405e-01 4.623460511379891003e-01
6.186539135143089840e-01 1.092271617513082882e+00 1.142541341484214712e+00
-1.013240302187123687e+00 4.120087992058356630e-01 2.303528752548655301e-01
-2.023855644566625589e+00 7.888162452003839586e-01 -5.754434350969757084e-01
6.248085291649343453e-01 -8.292575491114215769e-01 8.753926260685158667e-01
-9.643736172415809382e-02 7.370003456933270636e-01 -3.611856962404150773e-01
1.009649632953468101e+00 -7.669690713884438349e-01 8.950294985332295505e-01
-4.990769626443498641e-01 -1.339187970343349798e+00 1.677737083506397053e-01
3.289671445912725201e-01 3.397075454823056395e-01 1.575661886348517227e-02
-1.490408319416640204e+00 -8.179392617613709948e-01 -4.107114015545114682e-01
1.483908382224434996e+00 8.443765519243959794e-01 -3.144498205171965544e-01
5.915881096226632829e-01 -6.103759160953848850e-01 -1.186634110899313699e+00
-3.968945279026790396e-01 4.370106005721658970e-01 5.145252483941445965e-01
-7.809129828203688062e-01 -2.968961386702597682e-01 -3.511239661308343529e-01
5.030211947978555642e-01 -5.010186123666685587e-01 -2.180476285108364776e-01
-2.926834823630138360e-01 4.963148534229764319e-01 -8.758087974277840582e-01
-5.294845308853726173e-01 1.723793544029895708e+00 1.610149713293278495e-01
9.862492123047826853e-01 -1.554469062835517112e+00 -2.442722303964492414e-01
-2.193705359673306030e-02 3.930230875028008497e-01 -1.252209755813182557e-01
1.987872132036888617e-01 -3.799163860929255510e-01 -3.251680497280641235e-01
-5.643914238352213397e-01 5.041027887327287349e-01 2.083547867406655985e-01
3.175451874138348618e-01 -8.015721327364461191e-01 -6.684966797658589144e-01
7.379927829409771434e-02 9.885875911295218810e-01 -4.987657683366173433e-01
-3.002824762188817664e-01 5.520407730941372382e-01 -4.478279988903046543e-02
-1.074273646377990943e+00 -8.640440586200862638e-01 -7.980111604323683050e-01
5.672646920934101278e-01 4.014474549141425763e-01 1.390968970659230086e+00
-3.883057648900063719e-02 -4.597124293836185038e-01 1.640936606519886598e-01
1.323602985540662524e-01 4.064025430006657502e-01 -5.289300251087403204e-02
-1.054630546410656178e-01 -5.320024505525354774e-01 -9.938522025272702454e-01
8.189555991946368962e-01 3.592961572919559088e-01 8.071945973078272774e-01
-4.076659890680046638e-01 -9.058536972724663405e-01 -5.833085384660041672e-01
3.333957892631601583e-01 -2.483437290504480754e-01 8.208240103375621377e-01
-3.478121933119680875e+00 9.843169471342846366e-01 1.338183614473114957e+00
-1.160751314309953486e+01 1.023414068850539094e+00 4.992881495452685670e-01
-9.475616707503284331e-02 -7.342228271042007259e-01 -1.210329728308888186e-01
-1.574323926561632669e-01 -3.058227402931272820e-01 8.729752395605669957e-01
6.265781269895130956e-01 3.371223559755559540e-01 6.005862574220832695e-02
1.601463477410860703e+00 -3.123788926040118596e-01 2.135069773744087684e-01
-4.002681697064349176e-01 -7.222301050852578674e-01 -2.708414338311486147e-01
1.770970783914052493e+00 2.701896334113365050e-01 -1.328293739039118904e-01
4.426719586653620198e-01 7.235097499069650384e-01 2.078417819980600545e-01
8.791625954672132037e-01 -8.566651514628855768e-01 -1.636242823242679112e+00
-4.030197475898486537e-02 1.853084854066014142e-01 -2.981002096127764386e-02
1.007650152690180434e+00 -9.890149084307844340e-01 -4.408791673658193355e-01
1.745051386017246881e-02 -1.184141777975084436e-01 3.030053268448473225e-01
3.185650296373108703e-01 -7.580268434979210712e-02 5.634540910161381921e-01
3.982018073503090783e-01 5.172191041145611790e-01 -5.574754577634762898e-01
-6.470989318784083899e-01 1.608487471715200623e-01 6.537454799606243760e-01
-5.411865522329967204e-01 1.014748024862251574e+00 9.611051548478867224e-01
-1.292566996257260570e+00 4.238677273199573659e-01 -6.201855306498323950e-01
3.651969651200688560e-02 -3.987583666545082561e-01 -4.761086767710571988e-02
4.964252116639761847e-01 2.728857001241031810e-01 -6.238698690131789792e-01
9.963037083053878362e-01 -8.563146937336958420e-01 1.075257551208915086e+00
-6.901828973272468304e-02 -6.947831149079114477e-01 -1.589518496303543027e-01
2.428743736554311028e-01 -6.779066529632798987e-01 -7.766913043028974561e-01
-2.576363133734139166e-01 5.887088721309329253e-01 1.041440195581706085e+00
-8.364265359061732186e-01 1.732226969730837340e-01 -4.392220043689097353e-01
-1.284122788960731232e-01 9.830844001088642203e-01 1.539918816863473094e-01
-5.776819670948910718e-02 -7.230214213975038051e-01 -3.178246536871497385e-01
6.362870804123177404e-01 -3.242780204502666574e-01 1.122356234629851901e+00
8.247944817014043473e-01 -5.937218234619741075e-01 -4.923539085540294957e-01
-8.776531138418166567e-01 -8.994758861243253856e-01 4.475433274538200279e-01
-3.290870870385728120e-01 4.452088618979578305e-01 -2.559229150792174390e-01
1.018362066578065317e+01 -5.614297431894605950e-01 -4.433829474453754305e-02
1.155873667991225845e-01 8.927765400363557591e-01 -8.581169223134549329e-01
-7.687824635535369655e-01 4.434201759076258353e-01 -5.350779716739955116e-01
7.341306774461907603e-01 -8.068129418338463310e-01 4.911966079325135737e-01
1.856247931970889109e-01 1.077525553571460959e+00 -4.461180522143058380e-01
-8.450272650085275394e-01 -5.003522420471898657e-01 4.818326723398837219e-02
4.739632461209055680e-01 7.482003401333851833e-02 1.771206986161173447e-01
-8.155388458247256889e-01 -1.134165081472000569e+00 2.646764175423743648e-01
5.774073944394041424e-01 1.245652248529817374e+00 6.727630565114199879e-01
3.500435381030095883e-01 -9.167273876335602090e-01 1.083621476935215794e-01
2.708526296811382128e-01 -2.043748965008413687e-01 1.155741637137054079e-01
-3.946094302555943623e-01 1.102931139868811528e+00 1.021023331995940309e+00
2.130605737043491099e-01 -9.230561014425258115e-01 5.580242757561805211e-01
1.562061807168470262e-01 -6.470001238498351848e-01 -8.966709639172344426e-01
3.106377257299882677e-01 7.575530940580254002e-01 2.099476232591802050e-01
6.234478840687766171e-02 4.805201537633743136e-01 -1.050234043190108713e+00
-4.157371597615807102e-01 6.750367056955364209e-02 -1.806255636968122080e-01
-1.828097830852277172e-01 -3.550720106954454725e-01 4.348325600921387202e-01
4.417574906882179020e-01 3.835959404994875310e-01 -8.599193366277163575e-02
4.578350441096957768e-01 -3.908495574719224597e-01 2.394967523431329182e-01
4.103325342645752771e-01 -3.696918504410799666e-02 4.616871953362203795e-01
-3.096191583522650559e-01 -5.071267419554449374e-01 -3.464619258310020489e-01
-3.483214162275364290e-01 9.666760911436369508e-01 5.114543063471563844e-01
-4.240784317824294325e-01 2.634935499347590970e-01 -2.497376105890040110e-01
9.132361756717584100e-01 -5.057096972052835815e-01 -3.153571294375223655e-01
1.396940320138637115e-01 -8.503815809397490266e-01 3.917979784330778426e-01
6.844931020402079680e-01 8.385545797350930641e-01 -2.128769068414811672e-01
9.611719138914202043e-01 2.528150865531350044e-01 -4.089220271480119018e-01
6.930717116173021886e-01 -1.001846635930519591e+00 -6.520184475057861917e-01
-7.030362347747360907e-01 5.310857552457483033e-01 5.256108827830540120e-01
2.719741183722945710e-01 9.674828783034430946e-02 9.579097378511061689e-02
1.706776666859597114e-01 6.816281664380754446e-01 5.223458763785157055e-01
1.030682486287328586e-01 -3.547380117606486749e-01 -7.732236125403576521e-01
-9.909160524382382151e-01 -3.665243925078590648e-01 9.087729039730300107e-01
-2.625157321981537173e-01 4.081875683716737480e-02 1.051814853219199764e+00
-7.136625646077544571e-01 2.063643230078416463e-02 -1.095191527375675156e+00
8.224648242597620840e-01 -4.496785862263695255e-01 6.926748178347219564e-01
1.499216641193156230e-01 -1.909247743156939903e-01 -9.465979425030474070e-01
-3.419873296357303638e-01 -1.187448143531804300e-02 5.369071417029902538e-01
-2.571500080166001512e-01 7.033353476791570635e-01 -4.815254135927956591e-01
1.212575000656788138e+00 -3.826582427955471100e-01 1.359120018736598434e-01
-4.595052555943857087e-01 -8.866497797474021114e-01 7.725525087779917532e-01
4.370978681308833047e-01 1.024796395319872711e-01 -4.366393549079068681e-02
9.996616659445316522e-01 6.755231820538697418e-01 5.205201634156516599e-01
-8.371568809936310585e-01 2.974997994345126573e-01 -1.937677206033354693e-01
1.270815545740662800e-01 7.866227750533039131e-01 7.281537608012500495e-01
-4.565656440254791115e-01 -4.196711691098067298e-01 -9.603939290780300064e-01
1.687319443835802457e-02 -3.020035458184471833e-01 6.205274525092786542e-01
-6.546851564039080085e-01 -1.547221497291920503e-02 -1.939505202861015742e-01
-1.957644636347407041e-01 -7.368453489401050049e-01 -1.907592379423255768e-01
-4.429101025774621614e-01 2.663329416832537280e-02 -5.868364114482941402e-02
-5.645173177253901692e-02 -9.438878631107570916e-02 -6.709152522423043052e-01
4.421794118868404011e-01 9.920284613467491575e-01 2.662719692076563893e-01
-7.046260758904257515e-02 -4.582744355468169051e-01 6.321049115594401524e-01
-1.210263693036207067e-01 -6.618347484310406204e-01 -6.015551766352898733e-04
-1.903961019656995671e-03 -7.512548730101640260e-01 3.660509575419086947e-01
-6.131162828906532303e-01 1.295985087836839800e-01 -3.297440198769815978e-01
5.656517205227088452e-01 -6.837943656047891228e-01 -4.504544088822735048e-01
-1.073922328676160109e-01 5.322951810849209098e-01 1.187262154558764049e+00
1.412625307715518197e-01 -4.719860872090713144e-01 -6.636351369794095678e-01
-5.111405374942998847e-01 -2.995157354278761286e-01 1.576552779666841486e+00
-2.885369487988617143e-01 -6.477857838726022721e-03 -1.086790777772382643e+00
-1.383317174975124686e-01 -8.012064066586744859e-01 1.118582550087154459e+00
1.331517465458125216e-01 6.196993917938076191e-01 -1.167179971000892547e+00
-4.185581708969031700e-01 -2.803293172903453256e-01 9.654069463114325345e-01
2.891684445411247806e-01 -2.223992657823818109e-01 -1.552606372985388949e+00
5.121187931020773076e-01 3.175320241066466320e-02 8.582219010231463274e-01
-6.916166756892420109e-01 1.541166063876278081e-01 -5.662505433947062627e-01
5.158556597411489975e-01 9.330794613831870921e-01 8.433307875790165831e-01
-1.433384386813046030e+00 -7.751828135975895595e-01 -5.931404430554138019e-01
1.092687370814494230e+01 3.365279536826007445e-01 1.310697990283286929e+00
2.438290593704618425e-01 2.745557243997956975e-01 -2.340882520268953715e-01
-3.031978644084458718e-01 -4.425733391776464631e-01 -7.593208530964121739e-01
8.797196046316319018e-01 -4.444508953495558412e-01 -8.612104039667409783e-01
6.783446599919420450e-01 6.535088316902024630e-01 -4.080609837039154142e-01
-4.736645849363074756e-01 3.212480648120297211e-01 -1.320726901897364636e+00
8.172287743322041997e-01 8.622678937322915127e-01 5.239714274986043696e-01
-5.588229764188556992e-01 -2.797436575228534061e-01 5.345152331273335111e-01
4.520561504329716485e-01 6.342153757377150258e-01 -3.431477400256217059e-01
4.804498494328064973e-01 -9.700306780488597569e-02 -1.107936140812436701e-02
-6.784447769390583582e-01 3.889842132732092006e-01 6.218198307816902304e-01
4.403160834157948678e-01 -4.569978666855354366e-01 8.552579880135975365e-01
-1.282744991092363840e+00 8.357866450669437786e-02 -5.348860889190462098e-01
5.654437573889976587e-01 -3.818186717903611638e-01 -6.450267593940177679e-01
1.063397239470473266e-01 7.676675797850328609e-01 7.748246659482270760e-01
9.779799928054780933e-01 -1.072228841817149725e+00 4.804304202274189417e-01
-4.431473714662532148e-01 6.249865886243908264e-01 6.282016030869511450e-03
-4.532916178163957088e-01 2.922925086915372783e-01 -7.511619659205923671e-01
2.822495156670893346e-01 -7.310218070000613633e-01 1.230597416273400579e+00
2.458025953590928236e-01 -2.160526186246828628e-01 -3.312833888975396102e-01
-5.660201674765712870e-02 -3.239713899224356264e-01 9.183817975760346641e-01
6.914806698842155264e-01 -5.333967723905498337e-01 -2.734452359559312873e-01
-3.128582967406948745e-01 -8.036779631708694804e-01 2.589066758233744547e-01
3.142152759344206125e-01 5.639004528675488626e-01 -7.432134112848883811e-01
1.623035445174080049e+00 -4.162946094213578196e-01 4.022448718480995389e-01
-2.812190940152197172e-01 9.447417546532745902e-01 -4.501576405182688978e-01
1.759679531720110601e+00 -1.001380166909082714e-02 -3.836614254287412651e-01
-2.598018362340941412e-01 4.473520527314203132e-01 -6.520206178749270443e-01
1.007962858913334092e-01 3.897796979236707315e-01 5.429768645575979713e-01
6.303643771883180591e-01 2.570254552474242615e-01 -6.364260878275880273e-01
9.358395559288680277e-01 1.204269855205591655e-01 -2.836891600078101927e-01
-1.465227897642179755e+00 -5.036399289449303707e-01 -7.525592535817768880e-01
8.133517437961690355e-01 4.874606636538111881e-01 4.042534716845865295e-01
3.410559166820648080e-01 -2.866148098057275106e-01 -2.727499670401681287e-01
-1.178722922244529681e+00 2.066889563689756071e-01 1.832099783699944473e-01
-3.408927556410876747e-01 8.914574798396908417e-02 -7.314201846518622596e-02
-2.912229626601164911e-01 -5.559355271591071013e-02 3.598137248567607749e-01
3.678899574423010188e-01 2.797857174251483503e-02 -9.328942977942600301e-01
1.395215804072579280e+00 1.302653342356119293e-01 -2.458759291288746596e+00
9.203752177546027646e-02 -1.024741818623062128e+00 2.453045849775132847e-01
6.818208579127810198e-01 -2.365784752496902232e-01 -1.339683047402742577e+00
2.926001985236509895e-01 1.046396128891602245e-01 -1.342635079320988190e-01
2.540958574107302775e-02 5.555332622996099756e-01 3.879846160678810940e-01
-7.050521986389689522e-01 1.367390659515424822e+00 7.176382912348910903e-02
-2.005773981562402608e-01 -6.462319970009233749e-01 1.184834869041869881e-01
-8.893688666219752292e-01 -8.790924235369277362e-01 -3.806443176122427663e-01
4.995270654274009958e-01 1.068444262623897023e+00 8.912643206473698232e-01
4.295595378075598575e-02 -1.878910264172521938e-01 -2.423293870238045489e-01
-1.607618913821012574e-01 1.115922880432281411e-01 -3.665693294542934355e-01
4.430181013240467314e-01 -1.008639937365819195e-01 -3.465068109226636106e-01
2.856420435861056206e-01 -1.045079187776361884e+00 1.091623281678821122e+01
1.114194034705784930e-01 3.044286643282227978e-01 -6.265699076936188661e-01
-5.719516124485439779e-01 7.810551007166748327e-01 -6.573022576123650484e-01
1.668826500515275146e+00 2.595728547032838396e-02 1.303339443820394461e+00
-8.452986718147659229e-01 -1.273701164375713457e+00 -5.891182964122407793e-03
2.081185294111802975e+00 -1.240895382577879680e+00 2.652412039012530975e+00
2.934620272557852960e-01 1.104782800751184624e+00 -4.011256882239302435e-01
7.730300149476072580e-01 5.887616027452664547e-01 1.197967131854676091e-02
-1.907992296059834170e-01 7.697343735747455007e-01 -5.885002234036382784e-01
3.158641008162099695e+00 -4.245321986091974420e-01 1.210175824161682012e+00
4.277887345297187172e-01 -6.361574912134458915e-01 7.906597662986198927e-01
1.561407897879141471e+00 -7.272223073876002974e-01 6.897399277742104484e-01
5.725444033109049480e-01 -2.435792481209592797e+00 2.020733324327091962e-01
-4.733947991348438911e-02 -4.079106088500174532e-01 -6.022746841380099969e-01
1.466401972583158964e-01 -7.691522838736886136e-02 1.115972188548215449e+00
5.823444583902761540e-01 1.226933039024810101e+00 -8.601828412468086604e-01
-8.835201385841391364e-01 -1.607719533235206555e+00 -6.050963270780354009e-01
-1.252553763608436288e+00 -4.953948858962569091e-01 1.505707520545639111e-01
7.975322538195679378e-01 -3.171217287982055488e-01 1.483290941338858193e+00
-5.354765054284538195e-01 -6.987844430376362048e-01 -6.882801269191650695e-01
3.776369012544030790e-01 -9.297159808588909002e-02 2.613226712049613876e-01
-3.846189668247247395e-01 4.559804229056801383e-01 -1.100435602777888988e+00
-1.708836300626205562e-01 9.998088766456472154e-01 2.403569417198676283e-01
-1.406742849479356017e-02 3.735691148052467381e-01 5.656477029792008476e-02
3.617669546424485127e-01 4.806526543067285995e-01 -4.208374560189673264e-01
3.506644836535414700e-01 2.308810006934635672e-01 2.746750302809532474e-01
-3.181261679011593846e-01 -3.432812454717571460e-01 -4.445514135648620524e-01
7.646221115969550342e-01 8.133661104716159718e-02 5.232259525710885040e-01
-1.284873510193715429e+00 -7.357784442820832771e-01 7.893067749321380777e-01
9.246959482540936248e-01 6.389238492897012511e-01 -9.855451164556061627e-01
8.510648480879231537e-01 -3.353329292653670080e-01 -1.027977050552975752e-01
-9.204339434311002988e-01 -9.593967038056809837e-01 -3.616183495011407456e-01
1.334518947153911839e+00 9.459010591533745238e-01 3.445523957633164214e-01
3.162091460762520567e-01 -2.856811078608429977e-01 8.124495480653025625e-01
-1.217694967993947053e+00 3.003492755815027893e-02 -1.049095467166936579e+00
9.508699589673278441e-01 1.160803524661410346e+00 -4.472881016719925662e-01
3.020371271939210553e-02 -4.775158885129618946e-01 -1.229885236057653097e+00
1.336275768532788621e-01 -8.806942371905960298e-01 8.714613859200828205e-01
2.802888751558198499e-01 7.299265074496380512e-01 -2.116016094420788141e-01
1.321079343659997463e+00 -4.009119071367740261e+00 2.375341428467159788e-01
4.715408821645937221e-01 -9.144310679440618372e+00 4.312172249761209164e-01
-1.133815043471922523e+00 2.831131994508082617e-01 1.828237535138109160e-01
6.229075365225479333e-01 -1.348378054409842264e-01 -8.651675158522528775e-02
-7.246003844145549877e-01 5.669277101089621462e-01 -3.501605789724410861e-02
2.981855373375430496e-01 8.561581236367099557e-02 -3.148362079902419092e-02
2.560982448917111798e-01 -6.475426892746205532e-02 -2.878426050566392780e-01
1.225149683851908256e+00 9.725848425269640884e+00 -8.645249220624614539e-01
7.805481139738741359e-01 3.712222836359741485e-01 -7.840857993570695683e-01
-1.422622755484940438e+00 4.383419833355784445e-01 7.444717707662644735e-01
-7.641748309322924371e-01 -7.018783565385344758e-01 2.037890336941079739e-01
1.238503029745259831e-01 1.418013250599065467e+00 2.884596032773638352e-02
-1.217530957022830984e+00 -4.373937759500636235e-01 -1.073943672264604077e+00
6.734576242469100249e-01 5.872376637858527371e-01 -8.325303727441689394e-02
4.062263977099130896e-01 -1.367932606885492008e+00 -2.011289620025262792e-01
-6.582182822947131307e-01 9.085087708261082584e-01 4.365307354515648441e-01
3.138266145638534121e-01 -3.452846886699336149e-01 6.990977226028956271e-02
-4.834304237306294993e-01 -1.339924672418061624e-01 9.455899680604051838e-02
2.930819118337494311e-01 6.797778876499411815e-01 3.789905746273671272e-01
-3.736017031604514371e-01 -5.810323271739199491e-01 4.364035702672791617e-01
3.539225095205827398e-01 -2.071524010427686913e-01 -7.227816537302299282e-02
-1.148178418188557037e+00 8.307401736428322581e-01 1.708571185387147973e-01
1.692674660380427687e-01 1.594278146611517455e-01 -7.814650411469550861e-02
-4.122704510441110304e-01 4.241505364979569204e-01 -4.754624880926419106e-01
4.850805928324446459e-01 -3.096152249939264078e-01 7.464120753285151677e-01
3.586649867776586231e-01 4.967797981639291982e-01 3.268572159581494274e-01
6.846997541281283350e-01 6.706594277036430352e-02 -6.242849554487719388e-01
4.523618308767418106e-01 -2.955266451043994103e-01 -1.111397994015497659e+00
7.269511790168354803e-01 -3.448130484687252362e-01 9.428358557439971976e-01
-2.866917157291452578e-01 -1.730786641333962594e-01 -3.437054095533464970e-01
9.514404813612088452e-01 -3.297669412417463808e-01 6.503184050387987858e-01
-1.176993065512746961e+00 2.310524556091410719e+00 -1.222541699488027156e+00
2.143124195836977608e-01 -5.232059054774338414e-01 1.382490571791070710e-01
2.516056444733403286e-01 5.398190748070709200e-01 -8.005724898484341268e-02
-5.573064801692383802e-02 3.384195946192440907e-01 -5.150003934596268929e-01
-1.303242004411371280e-01 1.773499919360682853e+00 1.348128847263856123e-01
2.141859729117580713e+00 1.299941437503515207e+00 -3.215580061975952253e-01
-1.105833691141036024e+00 -8.499884288675906285e-01 8.215319881507728017e-02
9.212085885318147538e-01 5.689732459669812892e-01 -1.079826635215518804e+00
1.287647513113359399e-01 6.455767652058296280e-01 4.496673831092660722e-01
1.484559361824731583e-01 3.366579368906049474e+00 -3.203888319890710323e-01
3.610928035002125558e-01 -8.078292299003106969e-01 5.821705877813139196e-01
2.501828060666980491e+00 1.429885776106284201e+00 -1.564605149662646078e+00
6.894546002819953445e-01 -1.016079599917668885e-01 8.367841872959220773e-01
1.963190782882631713e-01 -7.842955200123939985e-01 5.915968580447524827e-01
1.328513106586486359e-01 6.711169951832630653e-01 -8.784983355957852735e-01
-8.394757322274049027e-02 1.247274946209144364e+00 -3.352178986761702895e+00
6.469968312043516967e-01 9.782193949236465125e-01 -1.022048625910630193e+01
6.675846384349970464e-01 -9.965523173847374849e-01 -7.837810393207418347e-01
4.124585068662090515e-01 8.984726404843631498e-01 4.106543951809077941e-01
-1.935309814069811618e-01 -6.707394959181985084e-02 -4.179830820923250423e-01
4.836644011100099672e-01 -3.199786070631257529e-01 8.604545954552842435e-01
7.376574227598972633e-02 1.057154196841280369e-01 -8.297993008681903149e-01
-2.517757295724515343e-01 5.399366077148580745e-01 8.118759564353966729e-01
2.893709221553919519e-01 -5.716658519871208044e-02 -6.493503665491312526e-01
-4.556724439240003588e-03 4.796897230372460519e-01 -3.411535018611069892e-01
2.149382332855715450e-01 -4.196790641009530143e-01 3.382730609710525582e-03
-7.683189732313403519e-01 -3.059643868461919824e-02 2.068464453205993259e+00
3.198758472887517912e-01 -1.541300757727575022e-01 -1.210794338123831748e+00
3.796974890992130813e-01 2.065575433699326849e-02 2.355633276316693614e+00
-3.372051419273255912e-02 2.125999853498957926e+00 3.166728280378126592e-01
-1.731488454162247725e-01 4.607626525601524192e-01 -5.468745868446112102e-01
-6.265340873808841748e-02 2.843196497419710478e-01 6.013458451718600717e-02
1.492818596644569611e-02 -9.993371411301636975e-01 3.839782551237944652e-01
7.308328247746672668e-01 2.865379029030928892e-01 6.694141567833261641e-01
3.130549516245322850e-01 3.355682118011748538e-01 -7.405779693450385837e-02
4.519737936037638848e-01 1.158480587575801135e-01 -2.990231587073537423e-01
-9.586703023778072597e-01 -5.072123392572581713e-01 5.017054338476402142e-01

"""))

      def test_force_mixing_conserve_momentum(self):
         pot = Potential('ForceMixing method=conserve_momentum fit_hops=2 buffer_hops=1 qm_args_str={single_cluster=T cluster_calc_connect=T}',
                                 pot1=self.pot1, pot2=self.pot2)
         f = fzeros((3,self.at.n))
         pot.calc(self.at, force=f)
         self.assertArrayAlmostEqual(f, string_to_array("""
       0.4270785085      -1.4627721014      -0.5021788398
       0.1042996079       1.3001073632       0.5796324965
      -0.2002801477      -0.9982448115       0.3664327041
      -0.5239564343       0.8947256439       0.4941074255
       0.7428921370       0.3194808362      -0.1008181877
       0.2009482601      -0.2877733674      -0.0941304756
       0.4419093562      -0.7755781616      -1.4606650959
       0.6025772932       0.8817540331       0.6670402148
       0.2002889966      -0.4076774103       0.9386066157
      -1.3248850421       1.4944091274      -0.4250453013
      -1.0380168842      -0.8341127059      -1.1215372347
       1.2806408905       0.9584571071       0.9251532756
      -0.4275075993      -0.8490839599       1.6854842830
       0.9797860906      -1.4924996916       0.4866199697
      -0.0774059084      -1.0125839823      -1.3729628654
       0.6046978108      -0.4596437086       1.2641279513
      -0.1687565674      -1.0969723940      -0.3842891730
       0.8878858523       0.5704335744       0.0626633177
      -1.6549415859       0.2936701172       0.2233292416
       0.9462083392      -0.0183471351       0.2397467768
      -0.6916953284      -0.8215245039      -0.6936364931
       0.9108910841      -0.5593117381       0.2205325148
       0.6499684705      -0.2749568235      -0.3413196704
      -0.3303027914      -0.3207325836      -0.1479270535
      -0.5660877443       0.7308492548      -0.8672478062
       1.2737957816      -0.3446626344      -0.5606321723
       0.0563625483       0.4928566479       1.1285122520
      -0.3206716366      -0.1011284545      -0.9777876268
      -0.9437346117       0.6407366389       0.0295647762
       0.0732150537       0.7764486615       0.0259130714
      -0.7351962202      -0.0939836041       1.3608883830
      -0.0094912640       0.0233073250      -0.6000571927
      -0.1095466637      -0.5033026505       1.0446838350
       0.5984765646      -0.0222507246      -1.1565074693
      -0.7920878880       0.2574642596       0.4078113196
      -0.2479603743      -0.0113193798       0.7261522437
       0.5206143518      -1.0136715447       0.0178073931
       0.4063345546      -0.3645232848      -0.0169863361
      -1.2644748466       0.1209719037      -0.4211531489
       0.0847556718       0.6886342823       0.8048738274
       0.4961546192      -0.6057858869       0.2763026630
      -0.8870650253      -0.6863127779       1.1547035096
       0.6433382890       0.9806688549       0.0216514099
      -0.5353665164      -0.4617343368      -0.1613414214
       0.1959655960       0.8974868988      -1.0012293577
       0.4845678565      -0.5889861298       0.2301549439
       0.5431519504      -0.8772383408       0.2562822162
      -0.5381379423       0.3331465308      -0.5715374433
       0.0670764045       0.6387920731       0.1931628424
      -0.3159727728      -0.5658863814      -0.3940299558
       0.2914339961      -0.4485525707       0.3030152307
      -0.7033829031       0.1676218213       0.7165203012
       0.1932402239       0.5517486137      -0.0405472958
       0.2018132534      -0.1642468321       0.1621702594
      -1.0283800637      -0.2547204898       0.0804222308
       0.6841988573       0.4578362905      -0.1361427443
      -0.9029703167       0.1477536769       0.8344329955
      -0.2457839638      -0.5195442681       0.2843614116
      -0.1994861303      -0.1028513871      -0.0680356109
       1.5046724873       0.0625282893      -0.8696553550
      -0.8184963526       0.2004165605      -0.1526396049
       1.2642428311      -0.0511269234      -0.1351239468
       0.6067702128       0.2207479198       0.4414286530
      -0.4142329503      -1.0617051314       0.0232009807
      -0.3084314897       0.3272521389      -0.5038411705
      -0.6956821163       0.4884802556       0.8268653626
      -0.2087247220      -1.0827985521      -0.5974180167
       0.6055721313       1.3748088024       0.9247642977
      -0.3264243337      -0.5134852909      -0.6687405666
       0.7044940415       0.7431717092       0.5663886490
       1.2678392825      -0.2509357664      -0.9478430526
      -0.8100255017       0.5175899065      -0.3192781865
       0.5481291715      -1.0175229191       0.2589871329
      -0.5071825256      -0.0110708458       0.1801922262
       1.2350964155      -0.1228434213      -0.1948671655
      -1.6765364759      -0.9016886187       0.9711298860
      -0.6013547408      -0.2595545264      -0.2583185843
       0.2341630484       0.6594705870       0.1097921682
       0.6693589875       0.3296997855      -0.1458804854
      -0.7589894501      -0.3745649690      -0.3667759859
      -0.5694852848      -0.7014635598      -0.1461097503
       0.7730908568       0.8093832389       0.5295262483
       0.4209519005      -0.5194158374       0.7966301027
      -0.0392331858      -0.2764819144       0.6958776605
      -0.4909968831      -0.4923922268      -0.8561343746
       0.6501881645       0.1745445680       0.0614042461
       0.0128641544      -0.8679793853       0.3963337302
      -0.2765288639       0.4539347387       0.5232976221
      -1.1272737948       0.0275057610      -0.2805328027
      -0.3928919625       0.7916570515       0.0988272183
       0.0226100502       0.4300623434       0.4747028016
      -0.4206448532       0.1177890632      -0.5973639564
      -0.3541823529       0.7071381329       0.8219415747
       0.7222869220       0.7623896535      -0.7785195027
       1.0322501093      -1.3308120711       0.9978569007
      -0.7104940170       0.2126673906      -0.0816016581
      -0.4463006915      -0.1892228189      -0.8239915800
      -0.3569559514       1.0289235503      -0.1791172200
      -0.0978867335       0.1870330476       0.4558831278
      -0.4806702159       0.0004191681       0.3537561111
       1.2964093040      -1.1214524567       0.4214551460
      -0.7361652046      -0.0277427390      -0.4010114018
       0.3266843328       0.4505002443      -0.1476772672
      -0.2856067419      -0.4040428846      -0.7187508711
      -0.0245695574       0.0353749281      -0.4435005125
       0.7335055811       0.3455323215      -0.0405007408
      -0.1463759693      -0.5007587267       0.7377228965
       1.2174042135      -0.8904863610      -0.5940327072
       1.0004385695       0.2636684826      -0.6222405623
      -0.9225440987      -0.8458433714       0.5251729866
      -0.9018053436      -0.8343720373       0.2772738438
       0.1467104314       1.6363537219      -0.1192763901
       0.1375797680       0.3952815984       0.2169061303
      -0.9948474794       0.3731644229       0.6689147361
       0.7844773575      -0.3287068088      -0.1957736715
      -0.1929442054       0.4616996515       0.7270212434
       0.3711794070      -0.4353541860      -0.6553434221
      -0.0396505998       0.5284833024       0.2273302714
       0.6194449312      -0.3566245192      -0.8709005963
       0.0080990438       1.2834554066      -0.8386154046
       0.8112149386      -0.5086122970       0.2461769771
      -0.1485913744      -0.1491094684      -0.7914555486
       1.4294784108      -0.6926394689       1.3487357032
      -0.7803921879      -1.3859246838      -0.5426375928
       0.8944486218      -0.0429855832       0.1838060883
       0.1240075691      -0.3262052347      -0.5293892276
      -1.2773801886      -0.0560979532       0.2414545655
      -0.1461760654       0.6659985251       0.4180070458
       0.3542800412      -0.9485621234      -0.5790350427
       0.4489341251       0.4440808828       0.2723006210
      -0.2817650153      -0.6345108669      -0.3788047709
       0.1757083678      -0.1549804725      -0.0837922823
       0.4502951269       0.2204200279      -0.2542721079
      -0.3751514576      -0.1629133184       0.1067852125
      -0.2266520448       0.0636652003       0.3416885739
      -0.6317511038      -0.2529048571       0.5313331844
      -0.9231207063       0.5654058209       0.2937478773
       0.0621925176      -0.7810508654       0.1063761368
       0.5951553619      -0.0980480084      -0.5521529492
      -0.1650778649       0.0475456126       0.0587579741
       0.6945646936       1.2134905209       0.5012191922
      -0.5716532195      -0.7623471499      -0.9926478026
       0.0850517855       0.8305751359      -0.3780855324
      -0.3051958935       0.0031584084      -0.2080642235
      -1.2962301797       1.3850822547      -0.0261462486
      -0.3078908796       0.2815335377      -0.5983793706
       0.2377619284       0.3592806329       0.7850315036
       0.3299779028      -0.5423753477       0.1215289087
       0.3841272443       0.7815864809       0.9331242845
      -0.7615150255      -0.0676118304      -0.4598453288
      -0.5836351535       0.0215820267      -0.2661491757
      -0.5318196685       0.5251973975       0.7106654989
       0.5177173854       1.1049731521      -0.7818499843
      -0.0754669384       0.1064057446      -0.1534873670
      -0.0834245669      -0.1851478887       0.3220846804
       0.2674550984      -1.0939849475       0.6208718763
       0.2556876562       0.3479856515      -0.5970862947
      -0.2768875317       0.1051844398       0.7453012587
       0.3741890990      -0.9814297108       0.2098148300
      -0.1921238195       0.2900117093       0.4529769580
       0.0932103035      -0.2687853917      -0.3049428422
      -0.4179878915       0.5320039567      -0.2290212049
      -0.5716706411       0.1745806743       0.6252742365
       0.0627737257       0.4781856642      -0.8819156843
       1.0748479830       0.4092062333      -0.2790947138
      -0.7510127662       0.0676120299       0.0805716301
      -0.4256288467      -0.4172977676      -0.4358743051
       0.1788322191       1.2499559012       0.1093858741
      -0.4166370974       0.3646495667       0.4704513513
      -0.5998522228      -0.6082744439      -0.4374893586
       0.8663438408      -0.4479802470       0.2843485010
      -1.0014927794      -0.3151333043       0.3024239863
       0.5882483833       0.6552237595       0.2346142140
      -0.2270325060      -0.0140706615      -1.0976682495
       0.3960876339      -0.4195249583       0.3256153244
      -0.7577469424       0.9519031138       0.5421036906
      -0.8308992989      -0.1121315762      -0.6826916148
       0.7190497946       1.0518377401       0.3805267467
      -0.3423727602      -0.8980530355       0.1936265160
      -0.6189394836       0.4694633710       0.0959273870
      -0.4540892999      -0.6631196110      -0.7593072180
      -0.2847863714       1.0806017663       0.7083967209
      -0.0145627490      -0.4943551189      -0.1627482284
      -0.3650799614       0.6530513489       0.7663886870
       0.2564363022      -0.3239061732      -0.8243097129
      -0.1216911573       0.2035546815       0.2945941816
       0.5206544070      -0.3202733262      -0.9617731206
      -0.3980826209       0.7477555154       0.7269618113
       0.1806777941       0.6599140223       0.6474307565
      -0.0712984096       0.3393653399      -1.7634377440
      -1.4493026577      -1.3135350711      -0.8270765183
       1.2847815039       0.9706051586       0.9981580928
       0.7103345481       0.5181358452      -0.1681756413
       0.2489349690      -1.5096214530      -0.4169437061
      -0.2251843094       0.4653542430      -0.3442371614
      -0.7968190627       0.9212573940       0.1017382098
      -1.2388626213      -0.2737272650      -0.0513236006
       0.8012979326       0.9186570350      -0.6288551450
      -0.8965317163       0.6240114711       1.1352840524
      -0.3835992881       0.6537426564      -1.0189271392
       0.1734104497      -0.2508494471      -0.9466780536
       0.4786136337       1.1109541887      -0.5424722636
      -0.6869527493      -0.8771896313       1.1413261585
      -0.6591367055       0.2288833104       1.2257722939
       0.0453034271      -0.7861600681      -0.4454067257
       0.4579362010       0.7703492371      -0.0852902979
       0.1184380497      -0.0079425370      -0.2652510596
       0.4935267722       0.0684878486       0.3795899720
       1.0294167074      -0.5778194918      -0.7512835441
      -0.6757715023      -0.2533864317       1.1257467535
       0.8876045854       1.1731990539      -0.1251610949
      -1.0537440056      -1.0205203310      -0.0892193182
      -0.3302022814      -0.1025526578      -0.6524217389
       0.0495030877       0.3407922226       0.7845133508
      -0.4208068311      -0.3281788900      -1.1552290936
       0.2084139301      -0.7780428280       1.1174046050
      -0.3879056385      -1.3496984515      -0.1625722890
       0.6502024050      -0.4950114585       0.9821444646
      -0.2104565439       0.6764625318      -1.1813728552
       0.3947525493       0.0951878993       0.9047855059
      -0.7583321692      -0.2925510721      -1.2555599413
       0.8884329676      -0.6437753083       1.1073166736
      -1.2878059269      -0.8532485634       0.4623460511
       0.6186539135       1.0922716175       1.1425413415
       0.1188313085      -0.8680780575      -0.5362001936
      -0.6276741412       0.5982437502      -0.6305561310
       0.6248085292      -0.8292575491       0.8753926261
      -0.0964373617       0.7370003457      -0.3611856962
       1.0096496330      -0.7669690714       0.8950294985
      -0.4990769626      -1.3391879703       0.1677737084
       0.3232949249       0.3304742303       0.0168384106
      -0.0225269367      -0.8309721867      -0.2060166356
       1.4839083822       0.8443765519      -0.3144498205
       0.5915881096      -0.6103759161      -1.1866341109
      -0.3968945279       0.4370106006       0.5145252484
      -0.7809129828      -0.2968961387      -0.3511239661
       0.5030211948      -0.5010186124      -0.2180476285
      -0.2926834824       0.4963148534      -0.8758087974
      -0.5294845309       1.7237935440       0.1610149713
       0.9862492123      -1.5544690628      -0.2442722304
      -0.0219370536       0.3930230875      -0.1252209756
       0.1987872132      -0.3799163861      -0.3251680497
      -0.5643914238       0.5041027887       0.2083547867
       0.3175451874      -0.8015721327      -0.6684966798
       0.0737992783       0.9885875911      -0.4987657683
      -0.3002824762       0.5520407731      -0.0447827999
      -1.0742736464      -0.8640440586      -0.7980111604
       0.5672646921       0.4014474549       1.3909689707
      -0.0388305765      -0.4597124294       0.1640936607
       0.1323602986       0.4064025430      -0.0528930025
      -0.1054630546      -0.5320024506      -0.9938522025
       0.8189555992       0.3592961573       0.8071945973
      -0.4076659891      -0.9058536973      -0.5833085385
       0.3333957893      -0.2483437291       0.8208240103
      -0.0745500459       0.0331542368       0.4424965013
      -1.3611991862       0.4209286535      -0.2064455482
      -0.0947561671      -0.7342228271      -0.1210329728
      -0.1574323927      -0.3058227403       0.8729752396
       0.6265781270       0.3371223560       0.0600586257
      -0.3550049035      -0.0210357977       0.2085147342
      -0.4002681697      -0.7222301051      -0.2708414338
      -0.0919717074       0.3246644134       0.1094411165
       0.4426719587       0.7235097499       0.2078417820
       0.8791625955      -0.8566651515      -1.6362428232
      -0.0403019748       0.1853084854      -0.0298100210
       1.0076501527      -0.9890149084      -0.4408791674
       0.0174505139      -0.1184141778       0.3030053268
       0.3185650296      -0.0758026843       0.5634540910
       0.3982018074       0.5172191041      -0.5574754578
      -0.6470989319       0.1608487472       0.6537454800
      -0.5411865522       1.0147480249       0.9611051548
      -1.2925669963       0.4238677273      -0.6201855306
       0.0365196965      -0.3987583667      -0.0476108677
       0.4964252117       0.2728857001      -0.6238698690
       0.9963037083      -0.8563146937       1.0752575512
      -0.0690182897      -0.6947831149      -0.1589518496
       0.2428743737      -0.6779066530      -0.7766913043
      -0.2576363134       0.5887088721       1.0414401956
      -0.8364265359       0.1732226970      -0.4392220044
      -0.1284122789       0.9830844001       0.1539918817
      -0.0577681967      -0.7230214214      -0.3178246537
       0.6362870804      -0.3242780205       1.1223562346
       0.8247944817      -0.5937218235      -0.4923539086
      -0.8776531138      -0.8994758861       0.4475433275
      -0.3290870870       0.4452088619      -0.2559229151
      -0.2368422842      -0.1220458300      -0.5428636225
       0.1155873668       0.8927765400      -0.8581169223
      -0.7687824636       0.4434201759      -0.5350779717
       0.7341306774      -0.8068129418       0.4911966079
       0.1856247932       1.0775255536      -0.4461180522
      -0.8450272650      -0.5003522420       0.0481832672
       0.4739632461       0.0748200340       0.1771206986
      -0.8155388458      -1.1341650815       0.2646764175
       0.5774073944       1.2456522485       0.6727630565
       0.3500435381      -0.9167273876       0.1083621477
       0.2708526297      -0.2043748965       0.1155741637
      -0.3946094303       1.1029311399       1.0210233320
       0.2130605737      -0.9230561014       0.5580242758
       0.1562061807      -0.6470001238      -0.8966709639
       0.3106377257       0.7575530941       0.2099476233
       0.0623447884       0.4805201538      -1.0502340432
      -0.4157371598       0.0675036706      -0.1806255637
      -0.1828097831      -0.3550720107       0.4348325601
       0.4417574907       0.3835959405      -0.0859919337
       0.4578350441      -0.3908495575       0.2394967523
       0.4103325343      -0.0369691850       0.4616871953
      -0.3096191584      -0.5071267420      -0.3464619258
      -0.3483214162       0.9666760911       0.5114543063
      -0.4240784318       0.2634935499      -0.2497376106
       0.9132361757      -0.5057096972      -0.3153571294
       0.1396940320      -0.8503815809       0.3917979784
       0.6844931020       0.8385545797      -0.2128769068
       0.9611719139       0.2528150866      -0.4089220271
       0.6930717116      -1.0018466359      -0.6520184475
      -0.7030362348       0.5310857552       0.5256108828
       0.2719741184       0.0967482878       0.0957909738
       0.1706776667       0.6816281664       0.5223458764
       0.1030682486      -0.3547380118      -0.7732236125
      -0.9909160524      -0.3665243925       0.9087729040
      -0.2625157322       0.0408187568       1.0518148532
      -0.7136625646       0.0206364323      -1.0951915274
       0.8224648243      -0.4496785862       0.6926748178
       0.1499216641      -0.1909247743      -0.9465979425
      -0.3419873296      -0.0118744814       0.5369071417
      -0.2571500080       0.7033353477      -0.4815254136
       1.2125750007      -0.3826582428       0.1359120019
      -0.4595052556      -0.8866497797       0.7725525088
       0.4370978681       0.1024796395      -0.0436639355
       0.9996616659       0.6755231821       0.5205201634
      -0.8371568810       0.2974997994      -0.1937677206
       0.1270815546       0.7866227751       0.7281537608
      -0.4565656440      -0.4196711691      -0.9603939291
       0.0168731944      -0.3020035458       0.6205274525
      -0.6546851564      -0.0154722150      -0.1939505203
      -0.1957644636      -0.7368453489      -0.1907592379
      -0.4429101026       0.0266332942      -0.0586836411
      -0.0564517318      -0.0943887863      -0.6709152522
       0.4421794119       0.9920284613       0.2662719692
      -0.0704626076      -0.4582744355       0.6321049116
      -0.1210263693      -0.6618347484      -0.0006015552
      -0.0019039610      -0.7512548730       0.3660509575
      -0.6131162829       0.1295985088      -0.3297440199
       0.5656517205      -0.6837943656      -0.4504544089
      -0.1073922329       0.5322951811       1.1872621546
       0.1412625308      -0.4719860872      -0.6636351370
      -0.5111405375      -0.2995157354       1.5765527797
      -0.2885369488      -0.0064778578      -1.0867907778
      -0.1383317175      -0.8012064067       1.1185825501
       0.1331517465       0.6196993918      -1.1671799710
      -0.4185581709      -0.2803293173       0.9654069463
       0.2891684445      -0.2223992658      -1.5526063730
       0.5121187931       0.0317532024       0.8582219010
      -0.6916166757       0.1541166064      -0.5662505434
       0.5158556597       0.9330794614       0.8433307876
      -1.4333843868      -0.7751828136      -0.5931404431
       0.6274684375      -0.1912130611       1.8407614736
       0.2438290594       0.2745557244      -0.2340882520
      -0.3031978644      -0.4425733392      -0.7593208531
       0.8797196046      -0.4444508953      -0.8612104040
       0.6783446600       0.6535088317      -0.4080609837
      -0.4736645849       0.3212480648      -1.3207269019
       0.8172287743       0.8622678937       0.5239714275
      -0.5588229764      -0.2797436575       0.5345152331
       0.4520561504       0.6342153757      -0.3431477400
       0.4804498494      -0.0970030678      -0.0110793614
      -0.6784447769       0.3889842133       0.6218198308
       0.4403160834      -0.4569978667       0.8552579880
      -1.2827449911       0.0835786645      -0.5348860889
       0.5654437574      -0.3818186718      -0.6450267594
       0.1063397239       0.7676675798       0.7748246659
       0.9779799928      -1.0722288418       0.4804304202
      -0.4431473715       0.6249865886       0.0062820160
      -0.4532916178       0.2922925087      -0.7511619659
       0.2822495157      -0.7310218070       1.2305974163
       0.2458025954      -0.2160526186      -0.3312833889
      -0.0566020167      -0.3239713899       0.9183817976
       0.6914806699      -0.5333967724      -0.2734452360
      -0.3128582967      -0.8036779632       0.2589066758
       0.3142152759       0.5639004529      -0.7432134113
      -0.2002788379      -0.4185672204       0.1465107042
      -0.2812190940       0.9447417547      -0.4501576405
      -0.0924441255      -0.3223172328      -0.3131943232
      -0.2598018362       0.4473520527      -0.6520206179
       0.1007962859       0.3897796979       0.5429768646
       0.6045461063       0.2446706640      -0.6133799977
       0.8899525013       0.1130868351      -0.2689683528
      -1.4620847845      -0.5029948091      -0.7482949194
       0.7733804573       0.4617750053       0.3846965318
       0.3391500032      -0.2819424276      -0.2672133069
      -1.1186573511       0.1949115804       0.1745921131
      -0.3329664949       0.0849095250      -0.0714304034
      -0.6639449339       0.2738497495       0.6980402239
      -0.8637351614      -0.7079257453       0.3434129525
       1.2082512141       0.0987160059      -0.9799534185
       0.0909497146      -1.0038183193       0.2402444592
       0.6558299172       0.0654347499       0.4920877531
       0.2926001985       0.1046396129      -0.1342635079
       0.0254095857       0.5555332623       0.3879846161
      -0.7050521986       1.3673906595       0.0717638291
      -0.2005773982      -0.6462319970       0.1184834869
      -0.8893688666      -0.8790924235      -0.3806443176
       0.4995270654       1.0684442626       0.8912643206
       0.0429559538      -0.1878910264      -0.2423293870
      -0.1607618914       0.1115922880      -0.3665693295
       0.4430181013      -0.1008639937      -0.3465068109
      -0.4651991505      -0.3180555146       1.3305940709
       0.1114194035       0.3044286643      -0.6265699077
      -0.5719516124       0.7810551007      -0.6573022576
       0.5343803096       0.6856187888       0.1061961441
      -0.8182825125      -1.2343584106      -0.0051108834
       1.3637725410      -0.5161973951      -0.6514784735
       0.2849497343       1.0678605201      -0.3873933349
       0.7661658168       0.5775876092       0.0152036128
      -0.1901730936       0.7683796971      -0.5878793631
      -0.1444880620       0.5439099556       0.2293057163
       0.4157178686      -0.6188849066       0.7671302707
       0.3668628023       0.4848588297       0.0845144476
       0.3556368710      -0.7141544912       0.1890028983
      -0.0473394799      -0.4079106089      -0.6022746841
       0.1466401973      -0.0769152284       1.1159721885
       0.5718785948       1.1973036406      -0.8424727117
      -0.8658728688       0.1685743372      -0.3093141628
      -1.2525537636      -0.4953948859       0.1505707521
       0.7975322538      -0.3171217288       1.4832909413
      -0.5354765054      -0.6987844430      -0.6882801269
       0.3776369013      -0.0929715981       0.2613226712
      -0.3846189668       0.4559804229      -1.1004356028
      -0.1708836301       0.9998088766       0.2403569417
      -0.0140674285       0.3735691148       0.0565647703
       0.3617669546       0.4806526543      -0.4208374560
       0.3506644837       0.2308810007       0.2746750303
      -0.3181261679      -0.3432812455      -0.4445514136
       0.7646221116       0.0813366110       0.5232259526
      -1.2848735102      -0.7357784443       0.7893067749
       0.9246959483       0.6389238493      -0.9855451165
       0.8510648481      -0.3353329293      -0.1027977051
      -0.9204339434      -0.9593967038      -0.3616183495
       1.3345189472       0.9459010592       0.3445523958
       0.3162091461      -0.2856811079       0.8124495481
      -1.2176949680       0.0300349276      -1.0490954672
       0.9508699590       1.1608035247      -0.4472881017
       0.0302037127      -0.4775158885      -1.2298852361
       0.1336275769      -0.8806942372       0.8714613859
       0.2802888752       0.7299265074      -0.2116016094
       0.4625391100      -0.5796408942      -0.6184097301
      -0.1433364103       0.9531997024      -0.2173838638
      -1.1338150435       0.2831131995       0.1828237535
       0.6229075365      -0.1348378054      -0.0865167516
      -0.7246003844       0.5669277101      -0.0350160579
       0.2981855373       0.0856158124      -0.0314836208
       0.2560982449      -0.0647542689      -0.2878426051
       0.4161846501      -0.4005298101      -0.0387405539
       0.7805481140       0.3712222836      -0.7840857994
      -1.4226227555       0.4383419833       0.7444717708
      -0.7641748309      -0.7018783565       0.2037890337
       0.1238503030       1.4180132506       0.0288459603
      -1.2175309570      -0.4373937760      -1.0739436723
       0.6734576242       0.5872376638      -0.0832530373
       0.4062263977      -1.3679326069      -0.2011289620
      -0.6582182823       0.9085087708       0.4365307355
       0.3138266146      -0.3452846887       0.0699097723
      -0.4834304237      -0.1339924672       0.0945589968
       0.2930819118       0.6797778876       0.3789905746
      -0.3736017032      -0.5810323272       0.4364035703
       0.3539225095      -0.2071524010      -0.0722781654
      -1.1481784182       0.8307401736       0.1708571185
       0.1692674660       0.1594278147      -0.0781465041
      -0.4122704510       0.4241505365      -0.4754624881
       0.4850805928      -0.3096152250       0.7464120753
       0.3586649868       0.4967797982       0.3268572160
       0.6846997541       0.0670659428      -0.6242849554
       0.4523618309      -0.2955266451      -1.1113979940
       0.7269511790      -0.3448130485       0.9428358557
      -0.2866917157      -0.1730786641      -0.3437054096
       0.9514404814      -0.3297669412       0.6503184050
      -1.1866196979       0.3241362879      -1.5436892953
       0.2143124196      -0.5232059055       0.1382490572
       0.2516056445       0.5398190748      -0.0800572490
      -0.0557306480       0.3384195946      -0.5150003935
      -0.4303835332      -0.1513581177       0.1520507230
       0.8911508747       0.0669991067       0.3304837438
      -1.0691759803      -0.8245629802       0.0803216239
       0.9190923750       0.5643998959      -1.0827587906
       0.1293908873       0.6442220888       0.4502882435
      -0.7646924818      -0.1886584821       0.5817929108
       0.3479906308      -0.7837466231       0.5646121444
      -0.7794147639       0.5761125227      -0.7568078453
       0.6677458525      -0.0995293977       0.8100039796
       0.1963190783      -0.7842955200       0.5915968580
       0.1328513107       0.6711169952      -0.8784983356
      -1.0619336648       0.2811817987      -0.0454840633
       0.2144763796       0.4268605194      -0.2653315569
       0.6675846384      -0.9965523174      -0.7837810393
       0.4124585069       0.8984726405       0.4106543952
      -0.1935309814      -0.0670739496      -0.4179830821
       0.4836644011      -0.3199786071       0.8604545955
       0.0737657423       0.1057154197      -0.8297993009
      -0.2517757296       0.5399366077       0.8118759564
       0.2893709222      -0.0571665852      -0.6493503665
      -0.0045567244       0.4796897230      -0.3411535019
       0.2149382333      -0.4196790641       0.0033827306
      -0.7667709320      -0.2583330052       0.2660077398
       0.3198758473      -0.1541300758      -1.2107943381
       0.1228389832       0.0495184471       0.7673709224
      -0.0337205142       2.1259998535       0.3166728280
      -0.5597661255       0.0545664132      -0.8682465258
      -0.0584876983       0.2785916560       0.0592413028
       0.0148034975      -0.9504293923       0.3652866357
       0.7184822084       0.2784219523       0.6591915480
       0.2981132959       0.3175261901      -0.0697541443
       0.4421377803       0.1122262095      -0.2914376236
      -0.9093006761      -0.4827778579       0.4768171344"""))

if __name__ == '__main__':
   unittest.main()


