# HQ XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
# HQ X
# HQ X   quippy: Python interface to QUIP atomistic simulation library
# HQ X
# HQ X   Copyright James Kermode 2010
# HQ X
# HQ X   These portions of the source code are released under the GNU General
# HQ X   Public License, version 2, http://www.gnu.org/copyleft/gpl.html
# HQ X
# HQ X   If you would like to license the source code under different terms,
# HQ X   please contact James Kermode, james.kermode@gmail.com
# HQ X
# HQ X   When using this software, please cite the following reference:
# HQ X
# HQ X   http://www.jrkermode.co.uk/quippy
# HQ X
# HQ XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

from quippy import *
from quippy.structures import quartz_params
import unittest, quippy, numpy
from quippytest import *

if hasattr(quippy, 'Potential'):
   try:
      p1 = Potential('IP ASAP', param_str='')
      got_asap1 = True
   except RuntimeError:
      got_asap1 = False

   try:
      p2 = Potential('IP TS', param_str='')
      got_asap2 = True
   except RuntimeError:
      got_asap2 = False

   # If possible, validate TS against original ASAP potential.
   # Otherwise, we compare to reference data in this file.
   do_compare_p1_p2 = got_asap1 and got_asap2

   if got_asap2:
      class PotTestMixin:
         def compare_p1_p2(self, at, debug=True):
            e1 = farray(0.0)
            e2 = farray(0.0)
            f1 = fzeros((3, at.n))
            f2 = fzeros((3, at.n))
            v1 = fzeros((3,3))
            v2 = fzeros((3,3))
            df1 = fzeros((3, at.n))
            df2 = fzeros((3, at.n))
            local_e2 = fzeros((at.n,))

            at1 = at.copy()
            at1.set_cutoff(self.cutoff)
            at1.calc_connect()

            at2 = at.copy()
            at2.set_cutoff(self.cutoff)
            at2.calc_connect()

            self.p1.calc(at1, energy=e1, force=f1, virial=v1, calc_dipoles=True)
            self.p2.calc(at2, energy=e2, force=f2, virial=v2, local_energy=local_e2)

            if debug:
               print 'e1 = ', e1
               print 'e2 = ', e2
               print 'f1 = ', f1
               print 'f2 = ', f2
               print 'v1 = ', v1
               print 'v2 = ', v2
               if hasattr(at1, 'dipoles'):
                  print 'dip1 = ', at1.dipoles
               if hasattr(at2, 'dipoles'):
                  print 'dip2 = ', at2.dipoles

            self.assertAlmostEqual(e1, e2)
            self.assertAlmostEqual(e2, sum(local_e2))
            self.assertArrayAlmostEqual(f1, f2, tol=1e-6)
            self.assertArrayAlmostEqual(v1, v2, tol=1e-6)
            if hasattr(at1, 'dipoles') and hasattr(at2, 'dipoles'):
               self.assertArrayAlmostEqual(at1.dipoles, at2.dipoles, tol=1e-6)

            return e2, f2.T, v2, local_e2, at2.dipoles.copy()


         def compare_ref(self, at, ref):
            e = farray(0.0)
            f = fzeros((3, at.n))
            v = fzeros((3,3))
            df = fzeros((3, at.n))
            local_e = fzeros((at.n,))
            at.calc_connect()
            self.p2.calc(at, energy=e, force=f, virial=v, local_energy=local_e)

            e_ref, f_ref, v_ref, local_e_ref, dip_ref = ref

            self.assertAlmostEqual(e, e_ref)
            self.assertAlmostEqual(e, sum(local_e))
            self.assertArrayAlmostEqual(local_e, local_e_ref, tol=1e-5)
            self.assertArrayAlmostEqual(f, f_ref, tol=1e-5)
            self.assertArrayAlmostEqual(v, v_ref, tol=1e-5)
            self.assertArrayAlmostEqual(at.dipoles, dip_ref*BOHR, tol=1e-5)

         def test_dimer(self):
            dimer = Atoms(n=2, lattice=100.0*fidentity(3))
            dimer.pos[1] = [0.0,0.0,0.0]
            dimer.pos[2] = [3.042*BOHR, 0.0, 0.0]
            dimer.set_atoms([14, 8])
            dimer.set_cutoff(self.cutoff)
            if do_compare_p1_p2:
               self.compare_p1_p2(dimer, debug=self.debug)
            else:
               self.compare_ref(dimer, self.dimer_ref)

         def test_trimer(self):
            trimer = Atoms(n=3, lattice=100.0*fidentity(3))
            trimer.pos[1] = [0.0,0.0,0.0]
            trimer.pos[2] = [3.042*BOHR, 0.0, 0.0]
            trimer.pos[3] = [2.0*3.042*BOHR, 0.0, 0.0]
            trimer.set_atoms([8, 14, 8])
            trimer.set_cutoff(self.cutoff)
            if do_compare_p1_p2:
               self.compare_p1_p2(trimer, debug=self.debug)
            else:
               self.compare_ref(trimer, self.trimer_ref)

         def test_quartz(self):
            quartz = alpha_quartz(**quartz_params['CASTEP_LDA'])
            quartz.set_cutoff(self.cutoff)
            if do_compare_p1_p2:
               self.compare_p1_p2(quartz, debug=self.debug)
            else:
               self.compare_ref(quartz, self.quartz_ref)

         def __test_bigquartz(self):
            """This test fails due to round-off errors in original ASAP implementation
               If two atoms' positions differ by exactly half a unit cell, i.e. |s_i - s_j| = 0.5,
               then conversion from scaled to absolute coordinates is unstable.
               (short_range.f:229, nnlist.f:48)"""
            quartz = alpha_quartz(**quartz_params['CASTEP_LDA'])
            bigquartz = supercell(quartz, 2, 1, 1)
            bigquartz.set_cutoff(self.cutoff)

            if do_compare_p1_p2:
               self.compare_p1_p2(bigquartz, debug=self.debug)
            else:
               self.compare_ref(bigquartz, self.bigquartz_ref)

         def test_bigquartz_randomise(self):
            quartz = alpha_quartz(**quartz_params['CASTEP_LDA'])
            bigquartz = supercell(quartz, 2, 1, 1)
            numpy.random.seed(1)
            bigquartz.pos += numpy.random.uniform(-0.1,0.1,size=3*bigquartz.n).reshape(3,bigquartz.n)
            bigquartz.set_cutoff(self.cutoff)
            if do_compare_p1_p2:
               self.compare_p1_p2(bigquartz, debug=self.debug)
            else:
               self.compare_ref(bigquartz, self.bigquartz_ref)

         def test_amorphous(self):
            am_xyz = """864
   n_efield_old=3 Lattice="19.363450 0.000000 0.000000 0.000000 25.153859 0.000000 0.000000 0.000000 21.313991" Properties=species:S:1:pos:R:3:Z:I:1
   Si             -0.20088783      1.16952395     -3.97162095      14
   Si             -0.91813079      2.90685298      0.50379101      14
   Si             -1.68977219      0.60971366     -1.46288876      14
   O               4.89847105      0.96212232     -0.00048749       8
   O               0.43966322      2.31994698      1.13783703       8
   O              -1.57754551      1.72063236     -0.34000921       8
   O               3.02635667      3.48148498      0.85184974       8
   O              -0.87103747      4.17262853     -0.47913618       8
   O               1.99020253     -4.27063355     -5.19874959       8
   O               1.99833275      5.18741413      4.93044048       8
   O               4.62653669      1.57979898     -6.04500334       8
   Si              0.58216110     -1.47784242      0.81845036      14
   O               1.05664349     -0.14013916      0.07908196       8
   Si              2.19459677      0.02516912     -1.01830032      14
   O               3.44270832     -1.04784470     -0.55412062       8
   O               2.19768797     -0.80156505     -2.44062869       8
   Si              0.33486117      4.71419955     -1.37889421      14
   O               1.00553633      6.00806304     -0.77000192       8
   Si              0.91437268      1.97311308      2.64357695      14
   Si             -0.96318972      4.13901685      9.76756873      14
   Si             -1.49487240      1.10056787      4.43285753      14
   O               0.65608831      3.29500195      6.14045868       8
   O              -2.14507501      3.27755072     10.47800717       8
   O              -0.63589581      4.79639963      4.55663060       8
   O              -0.20593091      3.27558097      8.62918075       8
   O               1.12943624      3.30876067      3.46340669       8
   O              -0.58762603     -1.08063617      1.84808482       8
   O              -1.50188368      5.58063921      9.31827469       8
   O               3.44424650      3.11465691      4.71903786       8
   Si             -2.20951128     -2.80768990      7.66778614      14
   O              -1.49960055     -1.59479951      8.45754847       8
   Si             -0.00369825     -3.02224464      5.67733853      14
   O               6.54205889     -5.30352206      6.30783793       8
   O               0.05812978     -2.03316107      4.40924655       8
   Si              0.81385948      4.15967841      4.77647057      14
   O               1.59423153      4.92459951      7.94926780       8
   Si              1.81606597     -1.31314221     10.64686431      14
   Si              0.97585842      3.39777736     -9.64382996      14
   Si             -0.16124312      0.49574271     -9.71668578      14
   O               4.82878535      4.15756795    -10.28225706       8
   O              -2.20663823      0.27071779     -6.48011306       8
   O              -5.56907212      0.40719849    -10.06533663       8
   O               1.01564221      0.08246347     10.56977704       8
   O              -1.56439737      1.10609737      8.53301137       8
   O              -2.19456012     -4.09749389      8.61757239       8
   O               1.87157708      6.44614569     -9.35296402       8
   O               1.13753148      7.07767176      9.41755680       8
   Si              0.29920621     -3.53240859     -9.39119313      14
   O               0.01121826      2.09603873     -9.64447931       8
   Si             -1.57685177     -4.83310649      9.91326269      14
   O               1.12689448      3.95627052     -8.14588629       8
   O               3.23376717     -0.83625019    -10.11319238       8
   Si             -3.07573647      2.94193539     -9.54585774      14
   O              -2.54685112      3.76604995     -8.24281552       8
   Si              3.27628336     -1.68142727     -6.28871498      14
   Si              2.83033795      3.01889895     -0.66970842      14
   Si             -3.14438280     -2.30976017     -3.67571112      14
   O               3.58493823     -0.48898475     -7.33902912       8
   O              -0.39311986      0.57314461     -2.46424904       8
   O               2.06860438     -1.71659725     -5.25310465       8
   O               1.45880242      3.57999097     -1.26726794       8
   O              -6.61253190     -1.61588093     -3.16569690       8
   O               0.93934994     -2.07568495     -9.55877565       8
   O               4.05551888      3.60137820     -1.54712997       8
   O               2.40406737      3.02240467    -10.28760791       8
   Si              1.10412555     -2.98359122     -4.78671432      14
   O              -0.34244830     -2.95558163     -5.46164927       8
   Si             -1.19958941     -3.59155467     -6.67661085      14
   O               4.61574169      1.48673546    -10.40789678       8
   O               3.54079668     -3.12234099     -6.87008991       8
   Si             -2.68851603      0.31477814     -4.94141408      14
   O              -4.15256392      0.78672126     -4.45452955       8
   Si              4.32565795      6.74036699     -0.77689991      14
   Si             -0.44789980     11.95886040      1.51741968      14
   Si             -3.66104174      9.55705882      2.21000356      14
   O               5.34844023      4.28302975      1.43509132       8
   O               0.30478106     -9.91904085      5.05983753       8
   O              -2.95767447      8.93900052      3.52467073       8
   O               2.03250608      9.31672839      2.27796816       8
   O              -5.07602980      8.88017659      1.85917052       8
   O               1.09159817      8.54387970     -0.47188508       8
   O              -0.95073998    -11.69770503      1.85708411       8
   O               1.70472989     -9.59543759     -1.46085072       8
   Si             -1.75988489      7.88277383      3.60783061      14
   O              -1.78077226      3.39173537      1.75688099       8
   Si              0.55786132      7.17823340      0.20430836      14
   O               1.23361972      6.94181774      1.64431711       8
   O               3.66747197      6.01599168      0.49672042       8
   Si             -2.48513735     11.95673954     -0.82142756      14
   O              -1.49859141     11.53019577      0.36187905       8
   Si              1.17526084      8.03408734      2.80995608      14
   Si              1.71686847     11.00607500      8.56310665      14
   Si             -5.46217200      8.87463925      5.88253984      14
   O               3.30569920      5.09515302      2.96282731       8
   O              -0.70311790      9.61239587     10.06554939       8
   O               2.80221191      9.22866005      6.00239539       8
   O               1.01747323     11.80341541      0.86302675       8
   O               1.59063799     12.39901024      7.76414714       8
   O              -0.42123080      8.40060606      2.88908861       8
   O               1.12879195     11.90386990      4.91369940       8
   O               2.94427109     10.76385124     -0.67162696       8
   Si             -2.27147800      6.88737846      8.74953076      14
   O              -3.78127354      5.87373451     -9.10571120       8
   Si             -1.88977879      5.75380755      4.64198386      14
   O               1.68418514      7.62129417      4.27279128       8
   O              -1.45757972      7.28265730      5.12432668       8
   Si             -0.07971046     10.91651297      4.50387681      14
   O              -1.51056862     11.16964336      5.14787995       8
   Si              1.41811741      8.41635721      5.66779264      14
   Si              2.63292127     12.49210722     -8.28831430      14
   Si             -1.31368734      7.49848634     -8.59838918      14
   O              -2.45308006      8.51851434     -9.08376170       8
   O               1.64557408      9.39245198     -2.97575846       8
   O              -1.35292880      6.86272905     -4.68664789       8
   O               2.24545307     11.00426963     -8.75131048       8
   O               1.30903042      8.93462869    -10.00243538       8
   O               0.68638838      9.92227181      7.89950482       8
   O               3.34662517    -11.99402318     -9.57941794       8
   O               3.16739944     10.34183372      8.73030522       8
   Si             -2.98579421      5.26131331     -7.79444661      14
   O              -0.37872079      8.42083531     -7.71953488       8
   Si              3.02941182      6.75688538     -8.26440492      14
   O               0.40750966      8.54657467     -5.29245597       8
   O              -0.54249770      7.06165519     -9.93539942       8
   Si             -3.37164695      8.75969755    -10.36054282      14
   O              -0.00027948    -11.42486089     -9.86272541       8
   Si              0.71824948      9.08109917     -6.76389741      14
   Si             -2.02140366     10.68261246     -3.66929355      14
   Si              0.13329512      7.36814287     -4.19327543      14
   O               6.74627526     10.68343420     -6.02390608       8
   O              -2.34733270     11.49210664     -2.36000387       8
   O              -2.10518471      9.09887329     -3.65218051       8
   O              -0.78867243     11.11928243     -4.54676423       8
   O              -0.46281773     11.01877171     -9.17748077       8
   O               0.67135781      4.35763994     -5.54182426       8
   O              -1.64232613    -11.73624277     -4.59894262       8
   O               3.21222460     12.33345804     -6.80395299       8
   Si              0.39574420      9.15681597     -1.83041893      14
   O              -0.06176561      7.80259143     -2.67796452       8
   Si              1.03116481      4.02546485     -4.00563359      14
   O               3.74143645      6.07026884     -2.12625861       8
   O               2.58447511      4.24563591     -3.67802905       8
   Si             -0.84838499     11.29170233     -7.65904820      14
   O               1.43597676    -11.59571635     -7.89804412       8
   Si              0.43063545     -8.94492459     -2.18885792      14
   Si              1.85566019     -3.83524328      1.94151855      14
   Si             -2.08095224     -6.21419929      0.91056245      14
   O              -1.30741912     -9.62688046      0.20713665       8
   O               0.78874766     -6.59560190      3.65635543       8
   O              -1.03850898     -7.04632694      1.80304330       8
   O              -1.16160649     -5.68913963     -0.28129874       8
   O              -1.26773415     -5.82151327     -2.84098231       8
   O              -0.62792362    -10.16396129     -2.16877256       8
   O               1.59876050     -2.26747169      1.77767274       8
   O               2.88468353     -3.37537959     -1.63769706       8
   Si              0.37393047     -6.34423064      2.11186103      14
   O              -2.85734331     -9.82356805      2.18445167       8
   Si             -1.33280290    -10.13855271      1.72283840      14
   O               3.75100987     -8.12714229      0.88986957       8
   O               4.75642825    -10.50915519      1.96694636       8
   Si              0.32216764     -3.36619310     -1.79705257      14
   O               0.56508473     -2.53174137     -0.43648023       8
   Si              2.32207480    -12.56103512      4.06532815      14
   Si              1.28322351     -5.93294821      9.17560878      14
   Si             -0.73662259    -10.29247827      6.26339592      14
   O               4.43021543     -9.16894218      4.13755258       8
   O              -0.31661397     -5.71520710      9.39281837       8
   O               0.03963856     -6.97707019      6.03040238       8
   O               2.35314410     -8.15710137      5.56012607       8
   O              -1.55037165     -5.87124350      4.08110032       8
   O              -0.30465490     -9.40026340      2.67107284       8
   O               0.88261256     -2.64074165      6.94395637       8
   O               2.58467495     -4.00378825      3.35295188       8
   Si             -1.44055163    -11.31090978    -10.53544966      14
   O               0.30497249    -10.40514253      7.52873920       8
   Si              1.67605811    -11.15992816      8.01465287      14
   O              -1.28750191    -10.15274642      9.66253353       8
   O               1.52342645    -11.41106139      3.26135023       8
   Si             -0.13092791     -5.94189435      4.80024329      14
   O              -1.50262663     -3.11918585      6.26269927       8
   Si              1.37880733     -7.55500972      6.62167814      14
   Si             -1.85781217     -8.63764085      9.70248978      14
   Si             -3.96096560     -5.31247353     -9.53118883      14
   O               0.40644298     -4.50903651      5.21956349       8
   O              -1.79338417     -7.69336820      8.38566193       8
   O              -1.81240076     12.42452213     10.14556937       8
   O               1.44750991     -4.55413259     -9.78573215       8
   O              -3.42060475     -8.98347371      9.58974489       8
   O              -1.55000434    -11.64879567      6.09382841       8
   O              -0.94665765     -3.73831174    -10.39909187       8
   O               1.82025027     -4.61643283      8.39365639       8
   Si              0.82421628    -10.45778221     -8.85040067      14
   O              -2.53280628    -10.91070165     -9.41049438       8
   Si              0.43803544     10.28429565    -10.28990338      14
   O               3.30593959     -9.14338841     -6.83466338       8
   O               2.06011704     -6.51659962     10.39868330       8
   Si             -2.71226440     -7.92917725      7.07370908      14
   O              -2.80033153     -5.61300080    -10.65090211       8
   Si              4.86480409     -8.87304612     -7.00256530      14
   Si             -1.82641295     -6.62467148     -6.49462210      14
   Si             -4.47205125     -6.69844146     -4.94246712      14
   O               1.03445678     -9.48130283     -5.87704080       8
   O              -1.80779347     -5.02704342     -6.33031229       8
   O              -3.61860818     -6.43093435     -3.60256419       8
   O              -1.59568789     -7.07341138     -8.03115526       8
   O              -3.18640356     -7.24862930     -5.92463896       8
   O               2.06174981     -9.74415882     -9.62653508       8
   O              -2.41160755     -2.58936958     -7.00334239       8
   O              -1.17819025     -7.93894092    -10.34787920       8
   Si             -0.16932942     -8.62559669     -5.21703205      14
   O              -1.51318957     -9.42127605     -5.57939760       8
   Si             -0.46536483     12.53763928     -5.27167674      14
   O               0.35387781     -8.50190938     -3.70807788       8
   O               1.04764055    -12.23568214     -5.11322955       8
   Si             -5.49756878     -4.22308381     -6.31450169      14
   O              -0.18083440     -3.70989253     -7.89655169       8
   Si              5.88526232      1.35947685      1.27299841      14
   Si              4.41115984      1.91885786      5.16793808      14
   Si              3.80642341      4.76081791      1.50603604      14
   O              -8.07281434      1.53471646      3.89395443       8
   O               3.96662629      0.54688740      4.39501488       8
   O               7.20206569      2.26625745      1.13504784       8
   O               9.47579681      2.67281809      5.24072251       8
   O               7.39997639      0.82549643     -0.87372180       8
   O               8.64190882      0.92505570      3.70237238       8
   O               5.28958032      4.88385596      4.28836403       8
   O               7.16906694      4.72947486     -2.26675149       8
   Si              3.69926935      0.57064214      2.79589456      14
   O               3.88748481     -2.14053028      4.67806475       8
   Si              4.03642919     -3.54889734      3.87070286      14
   O               7.18075944      0.65907629      6.15900970       8
   O               4.80295534      1.73208155      2.40450431       8
   Si              7.68360620      3.66506996     -3.35998874      14
   O               7.30200178      3.80994605      3.12633518       8
   Si              8.25161957      0.15471683      5.07658323      14
   Si              7.21774341      6.25691931      8.21983854      14
   Si              1.10533903      3.39829873      7.67710455      14
   O               1.86354595     -2.15579976      9.28801833       8
   O               7.55138580      7.62509000      8.95621827       8
   O               2.20665460      1.04785477      2.47473114       8
   O               4.29115147      1.64153037      6.74550953       8
   O               3.29674906      6.96741243      7.99201161       8
   O               5.17155455     -0.96301210      6.64951612       8
   O               2.03707155      2.09228108      7.95491187       8
   O               7.90992019      6.44259335      6.76238594       8
   Si              2.03765264     -3.07111528      7.98325391      14
   O               3.39040256     -2.79634964      7.16532172       8
   Si              3.75059361     -1.54417938      6.19368724      14
   O               7.61974867     -2.41736015     10.07103804       8
   O               6.50607400      0.49276487      8.60567660       8
   Si              4.46458855      6.44048224      7.01302064      14
   O               1.05726612      7.35281339      6.81337731       8
   Si              7.33404386     -2.83671733      8.56233880      14
   Si              7.81429089      3.65343842     -8.30933365      14
   Si             -1.90385602     -0.27797055      9.31112344      14
   O               3.68894522      0.20593032      8.89113581       8
   O               8.47713610      4.66796006     -7.22918786       8
   O               2.69571764     -0.35482332      6.63037331       8
   O               6.26950614      4.08823233     -8.28239916       8
   O               4.31082191     10.10426907     -7.38248474       8
   O               7.48838936     -1.62321135      7.53653719       8
   O               8.33894785      3.88833845     -9.80287183       8
   O               8.13314568      2.40912482      9.46317877       8
   Si              4.32189835      0.00916988     10.35760018      14
   O              -0.41128557      2.25643099     -6.82767924       8
   Si              7.03057612     -1.33118462    -10.20565565      14
   O               6.27078918      0.20320628     -7.57876692       8
   O               3.95300099      2.61281414     -8.28732475       8
   Si              1.75909143      6.52842589      8.03869198      14
   O               4.09221119      5.54322232     -8.30978411       8
   Si              4.94152826      0.18105707     -6.71570449      14
   Si              9.02478792      7.73647113     -4.98081472      14
   Si              3.10313362      0.93051483      7.57660107      14
   O               6.30748839     -0.23054809      1.41609463       8
   O               5.13566349      4.45602526     -3.79996754       8
   O               4.50444485     -2.07448113     -3.14459191       8
   O               4.80228194     -1.01005212     -5.45149398       8
   O               0.19481869      4.49879490    -10.48410974       8
   O               5.56437875     -0.97410096     10.56834177       8
   O               7.36968801      3.91987109     -4.91673599       8
   O               8.51046006      7.42612946     -8.45716010       8
   Si              3.88023245      4.60680863     -2.79454562      14
   O               0.75219428      2.43580190     -3.85571749       8
   Si              5.60631559     -1.25611537     -3.99076155      14
   O               9.21132065      0.57229755      1.01863954       8
   O               5.87716369      0.04763345     -3.10751539       8
   Si              0.80910943      3.28569052     -6.72708159      14
   O               6.17533592      5.67824230     -6.14660280       8
   Si              7.25602501      5.61212968     -0.93754691      14
   Si              5.75883095     10.94519513      0.70216194      14
   Si              1.51267880     11.51052015     -0.65591465      14
   O               7.60230637      4.66301718      0.31623614       8
   O               4.59633011     10.23884604      1.56989992       8
   O               4.86152805      7.62831544      5.97010580       8
   O               5.81869835    -10.29698808     -0.41951053       8
   O               0.40920395     10.78161801     -1.52578963       8
   O               5.91941859      6.45000071     -0.73699340       8
   O               5.59736497     12.53879750      0.72873715       8
   O               8.50962594     12.19243953     -1.19384446       8
   Si              3.60389084      4.76390792      4.58941925      14
   O               9.17994722      2.73280920     -0.60059656       8
   Si              8.34421368      1.60956592      0.16331666      14
   O               8.56865158      8.24766132      1.86113052       8
   O               7.20367584      2.19451084     -3.03586571       8
   Si              2.49804219    -10.98771092     -1.24764035      14
   O               1.51376062    -12.20662220     -1.37349128       8
   Si              6.98754865     10.13743514      5.07939543      14
   Si              5.71325202     12.48673138      8.37827991      14
   Si              4.28799987      9.01143820      5.38586854      14
   O               4.09291901      9.07279990      3.78746341       8
   O               7.04746944     11.91919585      7.68767964       8
   O               4.01019498     10.01193548    -10.03934526       8
   O               5.12700120     11.74909463      9.66074105       8
   O              -3.12106833      6.22250850      7.57473875       8
   O               3.93234304      5.21882024      6.17161689       8
   O               4.52107334    -12.55096567      7.28317577       8
   O               7.25365882     10.57806967      1.13201016       8
   Si              5.40732461      5.16371774     -9.12405071      14
   O               8.42002375     11.11600652     -8.44635482       8
   Si              7.10742235      9.02712310      9.55812770      14
   O               8.21217707     10.14673866      9.12688313       8
   O               5.58152825      9.34600549      9.15303162       8
   Si              3.44300084     10.00861443      2.65345354      14
   O               3.46813569    -11.88875791      4.95969474       8
   Si              3.27479792      9.62666584     -8.55640609      14
   Si              7.23303206    -10.40681064     -6.77353692      14
   Si              0.99285264      7.34975961    -10.33407852      14
   O               6.13622319    -11.23132109      8.88782910       8
   O              -7.81128554    -10.54112135     -1.94604135       8
   O               2.10268320      8.97033697     -7.50025244       8
   O               6.04127782     11.56938041     -8.35975192       8
   O               4.55261048    -10.03076546     10.63609897       8
   O               3.80699555      8.04919493     -8.77951590       8
   O               3.16123638    -11.05483063     -5.04168574       8
   O               9.14685029    -11.02849104      9.11765997       8
   Si              2.74415676      7.13405978     -5.26289190      14
   O               2.43745281      6.63247169     -6.76583905       8
   Si              7.03821075      7.88298221     -8.94018116      14
   O               6.50608119      8.83719932     -7.76792648       8
   O               6.28732234      6.50407387     -9.34734309       8
   Si              3.09850380    -10.65819371    -10.46241036      14
   O               2.27529156    -10.97709806      9.49306709       8
   Si              5.88650309     10.28121330     -7.35159843      14
   Si              8.04716075    -12.53327961     -2.71566491      14
   Si              2.91004646      9.80127139     -3.86821320      14
   O               2.84469202     11.28897864     -4.45907139       8
   O               9.23720371    -11.88682818     -3.63942396       8
   O               3.96586136      8.29103586     -0.69477402       8
   O               5.50177757     10.43490891     -0.78835615       8
   O               4.15643190      9.75405682     -2.85478230       8
   O               4.08061362      6.39721847     -4.82280925       8
   O              -9.25480001     -9.37161944     -5.20470904       8
   O              -8.66660774     12.16305567     -8.26302239       8
   Si              3.68750264      2.71612645     -6.71450927      14
   O               2.14816889      2.40857951     -6.40106360       8
   Si              4.94489537      5.13584909     -5.26816031      14
   O               7.52304360     11.34335701     -3.53405972       8
   O               3.03103516      8.70018274     -5.03533409       8
   Si              2.62814490    -12.55091448     -5.33362702      14
   O               0.28811224     10.62248364     -6.73029025       8
   Si              4.09038539     -7.45083796     -0.53799817      14
   Si              6.51603887     -0.35069308      7.21755354      14
   Si             -0.30031784     -6.08423548     -1.58068830      14
   O               5.63934739     -6.98489246     -0.46726660       8
   O               6.33082054     -6.33788980      3.98343722       8
   O               0.80006421     -4.89823850     -1.57591535       8
   O               7.20192667     -3.59115603      3.60008737       8
   O               0.13728373     -7.59676324     -1.36076305       8
   O               3.00736625     -8.83234945     -4.11513267       8
   O               5.01228759     -3.89244502      2.58417420       8
   O               6.84956641     -3.25312635      0.88405092       8
   Si              2.72342156     -7.82118566      2.08162529      14
   O               3.36390954    -11.18203280      0.08517540       8
   Si              4.89874825    -11.19354117      0.54036598      14
   O               6.39166333    -10.79862051      4.05211035       8
   O               3.54768947    -10.95791879     -2.46684500       8
   Si              3.35966701     -1.90040460     -2.04590017      14
   O               0.88328354     -2.84237517     -3.20365217       8
   Si              5.52743618     -9.81110921      3.12650792      14
   Si              8.92942790     -6.66514420      9.57547217      14
   Si              3.66324042     -7.86704188      4.65131971      14
   O               4.58355149     -7.04366556      5.67703379       8
   O               7.61893816     -7.19103922      8.84000061       8
   O               5.22575341    -10.28357021      6.29065613       8
   O               6.50591644     -8.59052046      2.69039038       8
   O               2.11154321     -8.99022652      3.03103196       8
   O               3.42832493     -6.98358733      3.30036481       8
   O               7.91057247     -4.34242097      8.20512997       8
   O               9.62216720     -0.01687466      5.89816436       8
   Si              4.02309286    -11.33504925      6.35540219      14
   O               2.77589736    -10.55515647      7.00532457       8
   Si              0.92026689     -9.94958950      3.54326596      14
   O              -9.32423518     -6.46635212      8.43271792       8
   O               7.68370254    -11.09642083      6.29068565       8
   Si              5.48590940     -5.78193370      5.23235123      14
   O               4.46748547     -4.59770388      4.99172887       8
   Si              6.03418101     -7.35463162      9.09495872      14
   Si              2.99574153     -4.63093542     -6.43773084      14
   Si              2.09563826     -5.96446800     -9.24102698      14
   O               5.61519315    -10.30532696     -6.86662372       8
   O               2.22809705     -5.36435576     -7.59182020       8
   O               1.54079719     -7.17285407      8.10825052       8
   O               5.71518550     -7.48681925    -10.65415742       8
   O               5.45205662     -5.99842101      8.49061152       8
   O               5.68326961     -8.83244760      8.61000525       8
   O               6.17576070     -5.55040440     -9.15675989       8
   O               8.41724869     -5.44898150     10.51130109       8
   Si             -0.46207678     -7.80245053     -8.89042059      14
   O               0.89686967     -6.99638831     -8.83998679       8
   Si              7.39350402     12.33639552     -8.71128034      14
   O              -9.56053343     -6.79558182     -4.05485029       8
   O              -9.60306906    -10.92508721     -9.71861409       8
   Si              6.36590975     -4.86246526      7.82005892      14
   O              -6.03438278     -6.05436819     -4.21149119       8
   Si              8.47220290    -10.98162947     10.59090452      14
   Si              6.21943077     -3.16755781     -0.57922569      14
   Si              2.57202432     -9.61758159     -5.48375045      14
   O               7.52981165     -8.08433065     -4.60207318       8
   O               1.55186370     -7.05090639      1.30128655       8
   O               5.31771296     -8.03827462     -5.67593471       8
   O               5.43241340     -9.59155836     -3.59588214       8
   O               3.69216452     -6.23471443     -9.23991910       8
   O               1.19285645     11.33183096     10.06914700       8
   O               5.63102761     -1.67270635     -0.58804633       8
   O               4.38429260     -5.29247372     -5.91744588       8
   Si              3.94316079     -9.50582381     -2.98155490      14
   O               3.98560121     -8.56600626     -1.67391245       8
   Si              9.14603527    -10.39692286     -4.32600157      14
   O               7.27939411     -8.59879074     -1.70958532       8
   O               7.56116188     -8.82922370     -7.09470854       8
   Si              5.14400286     -6.80276079     -9.32541919      14
   O               6.29335998     -3.66959251     -6.73794470       8
   Si              9.62235496     -0.87132225      1.57900166      14
   Si             -8.98437551      4.70656150      3.34591214      14
   Si             -9.56682973      2.06150592      3.77037775      14
   O              -8.48271171     -0.58880883      2.54194157       8
   O               9.59672534      5.15262463      4.68500999       8
   O               5.66933637      6.06047303      7.96962293       8
   O              -9.57190869      3.30410219      2.72153022       8
   O              -7.75996233      3.08494122      0.13388406       8
   O               8.34324696     -0.36632079     -2.79474530       8
   O              -7.55767007      4.17037437      3.93509513       8
   O              -6.20053462      4.12913173     -1.58533416       8
   Si              8.27691975     -2.45923859      3.88136896      14
   O               7.66916011     -1.30681045      4.77437552       8
   Si              6.58628698     -4.12433133      2.21468415      14
   O              -7.49165076     -2.82003999      3.54781040       8
   O              -9.24851262     -1.79909219      0.38658134       8
   Si             -8.79423875      3.37479305     -1.07005156      14
   O               8.50128828      6.65071697     -1.10852573       8
   Si             -8.61729312     -3.91848617      3.82339930      14
   Si             -8.83305251      5.23577761      8.30333421      14
   Si              6.44988002      3.83585185      4.52085498      14
   O              -6.86473594      2.92736841      6.48640931       8
   O              -9.27592646      4.20873282      9.49564658       8
   O               5.78602517     -3.40973699      8.28505482       8
   O              -9.43592787      4.45492302      7.05817720       8
   O               5.91123222      2.35028773      4.79886840       8
   O               9.55293041     -3.17287395      4.57614983       8
   O              -7.25832797      5.63762883      7.98794374       8
   O              -5.61667538      4.88376540      5.24542838       8
   Si             -9.26813827      0.23853952      7.40422924      14
   O              -8.77163603      1.76885010      7.42032774       8
   Si             -6.76705567     -1.33767133      8.20647985      14
   O              -3.72735402     -2.51119002      7.16291798       8
   O              -5.34644516     -4.66889566      6.66398791       8
   Si              9.03844395      4.12241001      5.78854159      14
   O               7.44121319      4.16575568      5.75010341       8
   Si             -8.94289155      0.06268243     -8.80631507      14
   Si             -8.95564276      2.74827431     -3.82112613      14
   Si              3.95504025      2.85505123     -9.88453617      14
   O              -8.99711099      1.58026379     -9.47676285       8
   O               8.97127690     -0.13813946     -8.13406609       8
   O               9.05559518      0.14900882      8.60672513       8
   O              -7.99352682      3.58944463     -4.82266021       8
   O               4.11218556      4.12367636     -6.15914942       8
   O              -9.30118854     -6.70101404      5.80205459       8
   O               9.28250114      3.82330849     -3.26759435       8
   O              -6.83367223      3.48853405      8.91668473       8
   Si              7.87056048      0.84490985      9.43304410      14
   O               6.96521396     -1.93232077     -4.49405293       8
   Si              7.50665188     -3.08669851     -7.65267109      14
   O              -8.03198777      2.92695394     -7.49204497       8
   O              -8.56970355     -1.16186685     -9.79101126       8
   Si              8.54543882      3.88442376      9.91490061      14
   O               7.74586307      5.02076727      9.09949920       8
   Si              8.33161789     -1.42837660     -5.28392158      14
   Si             -6.22795204      3.48617915     -0.10582121      14
   Si              7.85177438      0.63407976     -7.28325656      14
   O              -9.52603346      1.47624342     -4.47368729       8
   O              -8.20032819      2.64389660     -2.36698897       8
   O               2.66358063      1.49714612     -1.06676227       8
   O              -6.18193661      1.18826144      1.98289590       8
   O               7.93541896      2.17879434     -7.69291225       8
   O               7.80570025      0.09162314    -10.44433033       8
   O               7.96279721      8.85938782     -4.55329251       8
   O              -6.11493826      4.21333658     -6.45997134       8
   Si              7.21271944      0.71607468     -2.46391934      14
   O               7.02988706     -3.32401325     -1.96448979       8
   Si              4.91889996     -0.65021072      0.46168979      14
   O               8.64993526     -1.79680083      2.45618649       8
   O               3.92711505     -0.71627377      1.81992828       8
   Si              7.68242573      5.13567294     -5.91757678      14
   O               8.44081565      6.27611659     -5.12862359       8
   Si             -8.35764735      9.69149842      2.59272042      14
   Si             -8.89632200    -10.29278337      5.23683144      14
   Si              6.86716934      4.62061846      1.77765024      14
   O              -5.81573094      7.43350021     -0.18617564       8
   O               7.19756263      8.89740647    -10.16111103       8
   O               5.45246745     10.11924324      5.58146624       8
   O              -7.61847375      8.83772044      1.42976473       8
   O               8.49114766    -10.57696127     -0.44068275       8
   O              -9.57155387      8.21545329     -6.30292330       8
   O               9.10213187    -11.85433675      1.61159889       8
   O              -6.25103063     12.49943065     -1.28078580       8
   Si              8.48929042      6.89391843      2.72081527      14
   O              -9.36738868      7.69483388      6.10811586       8
   Si             -8.16579625      7.96989621      5.11434374      14
   O              -7.11357844      8.87472301      5.91182048       8
   O              -8.74518884      8.66175893      3.77934634       8
   Si              7.17503072    -10.15419098     -1.27442828      14
   O               9.54953401     -9.26815061      1.70442770       8
   Si             -8.30136657      7.65903510      9.42497785      14
   Si             -8.35201039     11.97090441      7.19037428      14
   Si              4.41511838     10.29864705      9.77649252      14
   O               7.32621496      9.76876475      3.55172967       8
   O              -5.07172097      9.04950609      7.44640944       8
   O              -9.19034459      8.83181699     10.03100187       8
   O              -5.38674633    -11.55901294      6.82033933       8
   O               3.03250716     11.47927777      3.17248666       8
   O               7.17621115      6.13577130      2.24575024       8
   O               7.53427519     11.62986542      5.21424776       8
   O              -3.72112177     11.81008059      6.29641692       8
   Si             -9.58298242     10.29442548      9.49121826      14
   O               7.75305646      9.06732414      6.02648783       8
   Si              8.41898835      7.61847765      5.80751226      14
   O              -7.80884279      6.64901119     10.58351960       8
   O               8.24581329      7.24451531      4.26007704       8
   Si             -9.56142975    -10.69775021      2.49603587      14
   O              -9.36375509    -10.07129710      6.75545649       8
   Si             -7.79310594      5.86374026     -9.31490819      14
   Si             -8.31247962    -11.60876123     -9.02111247      14
   Si             -6.40023313     10.94986514     -7.17479615      14
   O              -4.80487719      9.06295123     -9.67723224       8
   O              -7.77481274     -7.19140002     -5.87476342       8
   O              -9.05566210      9.40071047     -8.52321455       8
   O               9.52601558     12.32845208      6.67747584       8
   O               7.32753110     -9.80824139     10.59560601       8
   O              -9.31304837     11.20715282    -10.52058416       8
   O              -7.58819845    -10.59377449     -7.99051839       8
   O              -6.31982615     10.17780508      9.93946717       8
   Si             -9.43866018     10.89706652     -8.92989310      14
   O               9.33309642     10.81635441     -5.33766592       8
   Si             -9.45122311      7.97948483     -7.88864185      14
   O              -5.83865703      6.83721231     -5.65570487       8
   O              -8.37387483      6.84219960     -8.19519170       8
   Si              5.90690443     -9.94364417      9.78057257      14
   O               7.64320195    -12.29633273    -10.22804669       8
   Si             -8.48385233     10.85397841     -4.97430663      14
   Si              9.13654474    -12.01688019     -0.02215013      14
   Si              4.10782663      9.78607178     -1.26176006      14
   O              -8.15999658      9.85588004     -3.68580752       8
   O               9.64632256     10.46466285      2.21224359       8
   O              -8.42369256      7.78279141     -1.53992829       8
   O              -7.99327120     12.37820592     -4.73822216       8
   O               7.08301232    -11.23750749     -2.48728250       8
   O              -6.34865982     12.52553423     -6.78343172       8
   O              -5.33857313     -9.72513595     -2.61479281       8
   O              -4.92073553     11.88104935    -10.07069762       8
   Si             -9.31818859      6.50302780     -1.28678455      14
   O              -8.80546974      5.02536063     -1.16115413       8
   Si              7.80484300     10.45116809     -4.84675785      14
   O              -9.51849571      5.96807038      2.44541789       8
   O              -8.63620368    -12.15149854     -0.19392759       8
   Si             -8.64668270     -9.18878773     -2.21462905      14
   O              -7.77686486     -8.37807754     -3.31549355       8
   Si             -6.96100738     -9.73531277      1.62676797      14
   Si              9.15805577     -5.20679058      1.29742433      14
   Si             -9.10709563     -8.13529349      0.74604748      14
   O              -8.26727445     -6.90765473      3.20087367       8
   O              -9.27017131     -4.51261629      2.45139358       8
   O               9.65969342     -6.68805872      0.96207146       8
   O              -5.95258264     -5.11538553      2.87681931       8
   O               3.24639011     -6.13578528     -0.94637846       8
   O              -7.90489845    -10.61468706      2.55773018       8
   O               7.63465994     -5.34389997      1.84813777       8
   O              -5.79800930     -4.15582898     -2.35188180       8
   Si              6.64028833    -10.16760926      5.54100367      14
   O               9.45654922    -10.85293449      4.07478225       8
   Si             -4.23400173    -10.18401319      1.47592588      14
   O              -6.61395449    -10.64117286      0.35691857       8
   O               9.28483558     -4.25956118     -0.01103482       8
   Si              3.44619566     -4.56751462     -0.66860987      14
   O               2.83551771     -4.30412362      0.77409464       8
   Si             -9.25785643     -7.43005908      4.37651974      14
   Si             -6.64510541     -4.34069695      7.58828362      14
   Si              7.17481690     -7.69890286      3.85832657      14
   O              -6.99029287     -6.85603485      7.19090482       8
   O               8.27098330     -4.27793745     -8.39937855       8
   O               7.08347150     -8.63513337      5.21741434       8
   O              -8.11143976     -4.64682225      7.06275213       8
   O               8.70316746     -7.36272226      3.65234379       8
   O              -8.59082006     -8.83674247      4.65842388       8
   O              -6.32520052     -2.67532272      7.28494469       8
   O              -7.70890069     -5.02489783      4.59497723       8
   Si             -6.07931571    -10.50718700      5.85228819      14
   O              -5.93283636     -8.29528911      9.11017111       8
   Si              7.95905353     12.49002657      6.48805491      14
   O              -8.28604276     -9.19652303      8.93561615       8
   O              -7.45682623    -11.03257100      5.20300462       8
   Si             -8.52521315     -6.21025149      7.09350655      14
   O              -6.41341199     -4.75920190      9.06361322       8
   Si             -8.14421763     -8.13680047     10.18929096      14
   Si              9.58853751     -7.23731939     -8.00368173      14
   Si              7.66712579     -7.52077328     -6.12014902      14
   O              -7.55950507     -6.59948312     10.41378320       8
   O               9.16484982     -7.07785808     -6.43724135       8
   O              -7.38055006     -9.00925703    -10.01647418       8
   O              -8.98947722     -8.58363921     -8.16976492       8
   O               5.31973285     -7.95188055     -8.21247366       8
   O               8.23460390     -6.91065861     -8.76299199       8
   O               8.44225679     -2.35871638     -6.58436726       8
   O              -6.87296213     -1.81518391      9.70377895       8
   Si             -7.46584936     -9.03633131     -8.38348854      14
   O               7.73626135    -11.52668362     -7.78972639       8
   Si             -8.93885337    -10.51773372      8.25412162      14
   O              -7.51131347     10.31148824     -6.17281097       8
   O               9.64897024     -7.80235611     10.48697924       8
   Si              7.77406008     -5.46843721     -9.33220455      14
   O               6.99823066     -2.01079943     -8.75146711       8
   Si              7.31517489     -4.53688244     -2.99267608      14
   Si             -9.14329575     -3.06937084     -0.57478177      14
   Si              5.96790407     -8.21848841     -4.21379686      14
   O               6.86139513     -6.12897456     -6.21971660       8
   O               5.02475660     -4.24891824     -0.64873816       8
   O               5.69214001     -7.10628012     -3.10248353       8
   O              -9.50268693     -2.46706713     -2.02254065       8
   O               6.20150033     -4.58684322     -4.17859627       8
   O               7.49847641     -5.82550103     -2.01431588       8
   O               9.66789591     -1.64095165     -4.42525667       8
   O              -4.46028890     -1.97063428     -4.50392707       8
   Si              6.60991219     -7.12826565     -1.73226358      14
   O               7.86806051    -10.62100964     -5.30156003       8
   Si             -8.56921168     -7.95741463     -4.68396078      14
   O              -7.53876375     -8.31603272      1.20233983       8
   O               9.29763586     -9.55226344     -2.92459543       8
   Si              5.93367592     -4.89558168     -5.75311281      14
   O              -7.63391991     -3.61888996     -0.49743302       8
   Si             -2.27331860     -2.25173147     -1.01510522      14
   Si             -2.99439145      2.89057432      2.66210321      14
   Si             -6.54963906      1.17078336      3.55650076      14
   O              -1.57441145     -0.83029790     -0.76986339       8
   O              -2.49895490      2.30874586      4.05910869       8
   O              -6.37793231     -0.38476181      3.98340895       8
   O              -4.51922209     -0.27501067      0.43126071       8
   O              -3.61055992      1.62805405      1.89759029       8
   O              -6.74205853     -0.74930128     -0.72796856       8
   O              -5.19652025      2.24830594     -0.10668298       8
   O               0.07675678      4.82801268     -2.96923498       8
   Si             -7.15713725     -1.38231299      2.97616605      14
   O              -6.32548982     -1.70927528      1.67154575       8
   Si             -5.66766327     -1.39657052      0.27070384      14
   O              -1.89433827     -2.85450632      3.05202286       8
   O               0.14009046      0.27579673     -5.31702020       8
   Si             -4.85845446      1.17042895      1.04412862      14
   O              -5.85256980      4.51414513      1.05954663       8
   Si             -1.05688749     -1.51198565      3.34238972      14
   Si             -2.15776481      1.87813043      7.23042720      14
   Si             -7.38285284      2.32204612      7.93833389      14
   O              -0.18632759      1.16088454      3.49751241       8
   O              -5.30362074      5.44939252      9.85557624       8
   O              -7.18041855      1.18230931      9.16279982       8
   O              -1.27976380      1.36307781      5.99704772       8
   O              -5.71642401      2.34725755      4.27732620       8
   O              -5.16556721     -2.96921587      4.75730597       8
   O              -7.38239876      6.66266306      4.60597243       8
   O              -4.00387013      4.11682935      2.79693419       8
   Si             -6.11851423      1.62013682     10.34110278      14
   O              -3.64975289      1.21661221      7.11442427       8
   Si             -4.91654409     -3.12004843      6.31725245      14
   O              -3.51880672     -0.36125087      9.31339172       8
   O              -1.97557715     -0.40365392      4.07383061       8
   Si             -6.48998112      3.54648967      5.03739549      14
   O              -4.81085202      4.36572628      7.54733715       8
   Si             -4.40808587      0.59672951      8.38641926      14
   Si             -0.64393094      0.65924357     -6.73800449      14
   Si             -7.21617003      3.08194871     -6.12317130      14
   O              -1.44686223     -0.10055099    -10.48336195       8
   O              -3.35640752      1.39760344     -9.20754920       8
   O              -7.79631634      0.20455705     -7.69418031       8
   O              -6.57550579      1.63077493     -5.82954920       8
   O              -6.77451797      2.65673468     -9.93796501       8
   O              -4.84410237      1.85366430      9.31477751       8
   O              -0.15819207      0.03538117     -8.14826384       8
   O              -3.07358591      4.99950719      5.40230525       8
   Si             -6.54263375      0.27396193     -6.71626366      14
   O              -6.16899676     -2.07703933     -9.07948318       8
   Si             -4.43456032      0.16994894     -8.92583626      14
   O              -1.76234669      1.49801668     -4.29141646       8
   O              -8.13268343     -0.86490402      7.57966030       8
   Si             -8.11968333      2.90659662     -9.08935316      14
   O              -2.42606486      3.46957797      7.37450943       8
   Si             -6.03598786     -1.93618336     -4.61720606      14
   Si             -4.36251591      1.41230781     -2.95659373      14
   Si             -9.60915634     -1.10177522     -2.90141444      14
   O              -6.64831986     -5.01403037     -7.06039242       8
   O              -3.17740746      0.75524474     -2.07584526       8
   O              -8.46863764     -0.05815318     -2.47443148       8
   O              -5.82296067      0.85428074     -2.55093393       8
   O               7.95383134      0.07337901     -5.76886933       8
   O              -5.14371070      0.38370863     -7.49538226       8
   O              -4.25585708      2.99711133     -2.79920985       8
   O              -2.21632046     -1.08820786     -4.20654465       8
   Si             -6.90599944     -0.29571367     -2.25736266      14
   O              -6.31648605     -3.58672041     -4.88062213       8
   Si             -6.58327698     -4.72678082     -3.60367964      14
   O              -2.87837571     -3.89553630     -3.82443199       8
   O              -6.59108798     -0.98415789     -5.73135212       8
   Si             -5.26300259      5.30706480     -5.67826371      14
   O              -6.62965004      7.71713121     -3.28495417       8
   Si             -2.05580118      8.38777251     -0.22526708      14
   Si             -2.49632496     12.38162388      5.44050638      14
   Si             -6.57499131     11.24586109      4.57277058      14
   O              -3.96800729     11.54465111     -0.33422179       8
   O              -4.60807907    -11.67062110      1.89549426       8
   O              -6.12455571     10.04494576     -0.27725622       8
   O              -7.57481414     11.57658230      5.80249969       8
   O              -3.36607678      7.85996776     -0.99587834       8
   O              -2.63021142      9.23171880      1.02776939       8
   O              -5.68289275     12.50074967      4.04949179       8
   O              -2.52883844    -11.58729114     -0.92097310       8
   Si             -6.18239756      8.75870647      0.68967438      14
   O              -7.32754858     10.72365105      3.26290883       8
   Si             -4.38697136      6.70365573     -0.58307665      14
   O              -2.31260612      6.36755983      3.16087150       8
   O              -1.06559327      7.23783738      0.33125172       8
   Si             -5.34836088     11.19040024     -1.06164062      14
   O              -3.93785039     11.10381853      2.47107918       8
   Si             -4.70147934      5.26169632      1.92249002      14
   Si             -0.79894771      9.53007672      8.43644541      14
   Si              8.16774500      9.79266940      2.17850164      14
   O              -1.30849598      8.03774613      8.15964834       8
   O              -7.15234036    -11.96067490    -10.08414503       8
   O              -7.20161801      8.38283512      8.49292664       8
   O               0.34568736      9.50996145      5.15746299       8
   O              -6.07377813     11.21900215      7.71085018       8
   O              -5.40207918      6.26149275      2.95549667       8
   O              -3.24520142     10.16378460     10.15088090       8
   O              -0.35143835     11.13327617      2.90331162       8
   Si             -5.97082006      4.81987880      8.53362648      14
   O              -4.98441573      7.48758204      5.22981399       8
   Si             -5.80791762      6.29500227      4.50104151      14
   O              -5.36546511     10.19972969      4.95396727       8
   O              -3.86166569      6.04449931      0.80334302       8
   Si             -4.80917228     12.21852961      7.41108175      14
   O              -3.95667921     12.26411783      8.76631580       8
   Si             -4.97584577      5.14372358     -9.91155151      14
   Si             -3.41835499     10.06740018     -6.40666921      14
   Si             -5.63646845     10.41950413     -9.95590936      14
   O              -6.27220403      5.43835650     -9.03159390       8
   O              -5.53441443     12.14961374     -4.40941895       8
   O              -9.24453202      7.52242862     -3.81524192       8
   O              -2.31705144     10.70582162     -7.33086194       8
   O              -8.60847926     10.75503496      8.25837887       8
   O              -9.33816410      6.72972292      8.58311172       8
   O              -3.19307423      7.55045156      9.90599927       8
   O              -1.82528885     10.74202654      8.09839196       8
   Si             -8.08764562      8.34733981     -3.06474155      14
   O              -8.55169469      4.43659581     -9.41752005       8
   Si             -3.36377101      4.74350326      6.97035406      14
   O              -3.84271429      5.36148946     -6.44131123       8
   O              -4.51706543      3.58793347     -9.92511603       8
   Si             -6.24470566      9.69751391      8.41883756      14
   O              -6.65405603     10.59497115     -8.70996917       8
   Si             -2.59223069      7.86294678     -4.58430988      14
   Si             -4.91152608     10.76271048     -4.04303152      14
   Si             -7.29530154    -11.54304442     -0.81293757      14
   O               1.51506825      6.54168623     -4.36999717       8
   O              -5.42993105      9.39169742     -4.69525104       8
   O              -4.31705814      5.65052873     -1.78887217       8
   O              -3.39024111     10.86103140     -4.78156175       8
   O              -2.99918345      8.57617051     -6.01802121       8
   O              -1.74745138      6.26143556     -7.69355236       8
   O              -4.93699233     10.54281387     -2.48146476       8
   O              -4.87434049     10.43985352     -6.90003840       8
   Si             -5.48827965      7.79091067     -4.40732456      14
   O              -3.99233952      7.37717375     -3.96077554       8
   Si             -4.98340413      4.43038704     -2.61562794      14
   O              -1.26139507      9.28818052     -1.24721989       8
   O              -5.28796132      4.94776095     -4.09636497       8
   Si             -6.59691533    -12.11421395     -5.25897215      14
   O              -6.36929532    -10.57489096     -4.91622016       8
   Si             -1.88967178    -10.15836320     -1.21291512      14
   Si             -2.80731757     -3.41360637      1.86913116      14
   Si             -4.41453158     -6.22806808     -0.66509316      14
   O              -3.42619019     -6.94040549      0.42518196       8
   O              -5.16182008     -2.79115199     -0.31454340       8
   O              -5.43388184     -9.26677179      2.32580381       8
   O              -2.55083966     -4.98573955      1.81070318       8
   O              -5.60446994     -5.34154696     -0.04476559       8
   O              -5.02278452     -7.50539209     -1.40384396       8
   O              -3.44452801     -2.05611080     -2.10047299       8
   O              -1.29768170     -3.26282601     -1.84206101       8
   Si             -6.93612814     -6.12360372      3.63104162      14
   O              -4.36759383     -9.69148018     -0.01913858       8
   Si             -4.33590619     12.48763455      3.14716184      14
   O              -3.58767268     -7.70163849      3.87422352       8
   O              -2.88958818     -9.06826132     -1.84870856       8
   Si             -6.11371718     -3.99552008     -0.77234296      14
   O              -2.58197461     -2.83986263      0.41348313       8
   Si             -4.97691718     -8.50473705      3.85514698      14
   Si             -2.98716254     -6.34858414      4.58166624      14
   Si             -5.92994817     -8.05131904      7.52420935      14
   O              -6.08272146     -7.39488388      4.19566807       8
   O              -2.65813975     -6.63857941      6.12733934       8
   O              -4.34803353     -8.07341138      7.12335790       8
   O              -3.95784856     -5.10596257      4.38595154       8
   O              -6.57805327     -9.27776897      6.74815032       8
   O              -5.09426275     -9.83366719      4.73704347       8
   O              -5.59415176     -0.31935678      7.81882182       8
   O               0.50003062     -4.73387815      1.89291772       8
   Si             -4.92101351     -9.10256533     10.07403018      14
   O              -5.22592389    -10.68069376     10.13734997       8
   Si             -2.70002853     11.40825675      9.29481457      14
   O              -7.96209204    -11.81588479      7.99859007       8
   O              -3.12104819    -12.16440439      4.07341142       8
   Si             -4.90799788     -4.07692610      3.57669123      14
   O              -4.33047913     -3.26060101      2.34412106       8
   Si             -4.70792446     -7.92343959     -8.37079853      14
   Si             -3.71588438     -2.58310280     -7.99787286      14
   Si             -6.88224532     -5.12768032     10.55782013      14
   O              -5.45847645     -5.30859364    -10.07067023       8
   O              -3.07998310    -11.23975181     -6.78253287       8
   O              -8.65839799     -6.06710820     -8.11877465       8
   O              -4.09929134     -6.43321417     -8.35851094       8
   O              -7.73290479     -3.85348175    -10.24523899       8
   O              -4.92836255     -8.23787262     -9.92402290       8
   O              -3.62703848     -1.21129343     -8.82287987       8
   O              -1.89414646     -9.16801650      6.39466914       8
   Si             -3.53291738    -10.60969844     -8.17283294      14
   O              -4.97058821    -11.13727438     -8.67886984       8
   Si             -5.55395530    -11.78587353    -10.04240596      14
   O              -0.85056584    -12.46239941     -6.86595641       8
   O              -3.64738462     -9.02952353     -7.89540142       8
   Si             -7.33385539     -2.23677101    -10.08013467      14
   O              -3.54981726     -3.90653608     -8.90171913       8
   Si             -2.52842466    -10.63259384     -5.38361259      14
   Si             -2.76344536     -5.25764317     -2.96411076      14
   Si             -7.99786541     -5.76190460     -6.67864240      14
   O              -0.18743014     -9.31268668     -8.36115916       8
   O              -3.47000768     -5.18481548     -1.48731755       8
   O              -9.04920561     -8.53272524     -0.80912455       8
   O              -8.88497128     -4.86032181     -5.68517521       8
   O              -4.91581574     -2.90074800     -6.94686911       8
   O              -6.21803975     -8.20075271     -7.78681233       8
   O              -8.13655600     -4.68281888     -3.26293083       8
   O              -4.43515961     -5.26321424     -5.80908653       8
   Si             -4.46104874     -8.99824475     -1.49557869      14
   O              -5.19882077     -8.18108190     -4.87034692       8
   Si             -5.21416187     -9.64882392     -4.23969400      14
   O              -3.79250654    -10.35863278     -4.47033151       8
   O              -0.48844206     -7.15499828     -5.73762948       8
   Si             -9.36610495     -5.19243639     -4.17139294      14
   O               8.67861563     -4.38862016     -3.81287652       8"""
            am = Atoms(am_xyz, format='string')
            am.set_cutoff(self.cutoff)         
            if do_compare_p1_p2:
               numpy.set_printoptions(threshold=1e6)
               print self.compare_p1_p2(am, debug=self.debug)
            else:
               self.compare_ref(am, self.am_ref)




      class TestMorseStretch(QuippyTestCase, PotTestMixin):
         """ Turn off charge and dipole terms, to test short-range part of potential only"""

         dimer_ref =  (FortranArray(1.3232765968383859),
                       FortranArray([[-9.05935698,  0.        ,  0.        ],
                                     [ 9.05935698,  0.        ,  0.        ]]).T,
                       FortranArray([[ 14.58336505,   0.        ,   0.        ],
                                     [  0.        ,   0.        ,   0.        ],
                                     [  0.        ,   0.        ,   0.        ]]).T,
                       FortranArray([ 0.6616383,  0.6616383]),
                       FortranArray([[ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.]]).T)

         trimer_ref = (FortranArray(2.6914040732260349),
                       FortranArray([[-9.23936758,  0.        ,  0.        ],
                                     [ 0.        ,  0.        ,  0.        ],
                                     [ 9.23936758,  0.        ,  0.        ]]).T,
                       FortranArray([[ 29.7462768,   0.       ,   0.       ],
                                     [  0.       ,   0.       ,   0.       ],
                                     [  0.       ,   0.       ,   0.       ]]).T,
                       FortranArray([ 0.68406374,  1.3232766 ,  0.68406374]),
                       FortranArray([[ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.]]).T)

         quartz_ref = (FortranArray(20.41542250348801),
                       FortranArray([[  1.81666108e-01,  -3.14654930e-01,   1.35239216e-14],
                                     [  1.81666108e-01,   3.14654930e-01,   1.23107109e-14],
                                     [ -3.63332217e-01,  -3.73615451e-15,  -2.09875030e-16],
                                     [ -2.81150866e+00,   5.89289282e+00,  -3.78417660e+00],
                                     [ -3.69764055e+00,  -5.38128433e+00,  -3.78417660e+00],
                                     [  6.50914921e+00,  -5.11608487e-01,  -3.78417660e+00],
                                     [ -2.81150866e+00,  -5.89289282e+00,   3.78417660e+00],
                                     [  6.50914921e+00,   5.11608487e-01,   3.78417660e+00],
                                     [ -3.69764055e+00,   5.38128433e+00,   3.78417660e+00]]).T,
                       FortranArray([[  7.34602052e+01,  -1.54217752e-14,  -1.11188130e-14],
                                     [ -1.54217752e-14,   7.34602052e+01,  -2.65643286e-14],
                                     [ -1.11188130e-14,  -2.65643286e-14,   7.44956117e+01]]).T,
                       FortranArray([ 1.9532191 ,  1.9532191 ,  1.9532191 ,  2.42596087,  2.42596087,
                                      2.42596087,  2.42596087,  2.42596087,  2.42596087]),
                       FortranArray([[ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.],
                                     [ 0.,  0.,  0.]]).T)

         bigquartz_ref = (FortranArray(48.876005374180011),
                          FortranArray([[  2.93277763,   9.88802741,  -9.49992649],
                                        [-12.29984225,   3.03696733,  -4.61268954],
                                        [  0.32474038,  -1.28242997,  -0.50325171],
                                        [ -6.83023167,   0.39617117,  -7.36381674],
                                        [  7.6798915 ,  -8.27684431, -13.76672841],
                                        [  3.94615918,  -0.08369327,  -2.24527527],
                                        [ -0.33796743, -12.20849207,  -2.44265921],
                                        [  5.93572874,  -1.42901596,   8.40091225],
                                        [  6.62918157,   9.24560364,  12.51037759],
                                        [ -7.01698404,   2.63741145,   3.81348624],
                                        [  7.1530025 ,   5.51186507,  -9.07658237],
                                        [ -5.06562611,   5.24298752,   1.23723701],
                                        [ -2.08984681,   7.55793921,  -2.67194417],
                                        [-17.9623249 ,  -4.44680332,   2.97420223],
                                        [ 10.54089403, -11.89494979,   4.63150636],
                                        [ -1.08234574,  -7.14925412,   2.66404441],
                                        [  7.87211358,  -4.18856031,  10.10962134],
                                        [ -0.32932015,   7.4430703 ,   5.84148648]]).T,
                          FortranArray([[ 165.26199497,    3.69267784,    2.42134071],
                                        [   3.69267784,  158.95382262,    6.49024662],
                                        [   2.42134071,    6.49024662,  167.64378429]]).T,
                          FortranArray([ 1.84093094,  2.94860751,  1.58428019,  2.08952622,  3.18160937,
                                         1.82653314,  2.95589861,  2.80423499,  3.3284506 ,  3.27660168,
                                         2.87698788,  2.80079856,  2.63617854,  3.03799766,  2.51515247,
                                         2.5992419 ,  3.74776611,  2.82520902]),
                          FortranArray([[  0.00000000e+000,   0.00000000e+000,   7.95602526e-316],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  7.78282165e-316,   4.94065646e-324,   2.33639533e-310],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  1.05637316e-312,   7.95602447e-316,   0.00000000e+000],
                                        [  0.00000000e+000,   8.07576622e-316,   0.00000000e+000],
                                        [  0.00000000e+000,   4.94065646e-324,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                                        [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000]]).T)

         am_ref = (FortranArray(1904.8921692888641), FortranArray([[  8.65548611e-01,  -2.98949106e+00,  -8.12889157e-01],
                 [  1.49449797e+00,  -8.62728916e-02,  -2.84218202e-01],
                 [  5.12539984e-01,   2.58793058e-01,  -3.69476982e+00],
                 [ -4.76051314e+00,   6.75735438e+00,  -9.31570870e+00],
                 [  5.71895969e+00,  -1.22416619e+00,  -5.41079638e+00],
                 [ -3.96193639e+00,   2.43061643e-01,   2.52538008e+00],
                 [ -3.67472582e+00,  -4.31052853e+00,   6.68316183e+00],
                 [ -8.65683810e+00,   5.52881393e+00,  -8.37246385e-01],
                 [ -5.32443051e-01,  -7.25275505e+00,   5.18923593e+00],
                 [  5.73902591e-01,   1.01476541e+01,   3.07344703e+00],
                 [  4.02962829e+00,   3.42502139e+00,   1.01465431e+01],
                 [ -1.95675559e+00,  -8.23545024e-01,  -1.35068203e+00],
                 [ -4.80474535e+00,   8.19468855e+00,   2.74116534e+00],
                 [ -3.34047623e-01,  -4.54056605e+00,  -1.25545909e+00],
                 [  1.15687357e+00,  -2.09965450e+00,   5.26988010e+00],
                 [ -6.38999866e+00,   2.73332887e+00,  -1.09193969e+01],
                 [ -2.09179571e+00,   5.96715097e-01,  -1.52630812e+00],
                 [  8.41855193e+00,   1.19504988e+00,  -2.47993027e+00],
                 [ -6.24397911e-01,  -5.77092618e-01,  -1.11258111e+00],
                 [ -1.29451992e+00,  -1.48209794e+00,  -7.40467679e-01],
                 [ -4.93739427e-01,  -2.55043184e-01,  -3.06627474e-01],
                 [ -3.72086063e+00,  -6.27581141e+00,  -2.15503783e+00],
                 [ -1.86658397e+00,  -3.13704804e+00,  -3.65849177e+00],
                 [ -6.61938793e-01,  -2.38865273e+00,  -2.36019035e+00],
                 [ -2.37282191e+00,  -5.95498214e+00,  -1.04624639e+00],
                 [  3.38919733e+00,   4.05834466e+00,  -3.66414658e+00],
                 [ -5.20771789e+00,   5.61514058e+00,  -2.25935358e+00],
                 [  9.83526176e-01,   1.40079489e+00,   1.68473620e-01],
                 [ -7.29453833e+00,  -1.21708909e+00,  -2.03490861e+00],
                 [ -3.77716385e-01,   1.25198870e+00,   8.19106498e-01],
                 [  7.84611664e+00,  -6.76402188e-02,  -6.48686398e-02],
                 [  1.69924623e-02,   7.63904667e-01,  -1.45287429e+00],
                 [  9.89455210e+00,   5.46417997e-01,  -3.20584197e+00],
                 [  8.00892311e+00,   4.12466381e+00,  -1.74801345e+00],
                 [ -3.09414260e-01,  -1.31972766e+00,   1.24829820e+00],
                 [  2.37792549e+00,  -1.12593319e+00,   1.36951472e+00],
                 [  4.11472465e-01,  -1.17179707e+00,  -1.29272344e+00],
                 [  1.06533928e+00,  -1.31580359e+00,   3.40579336e-01],
                 [ -1.76044511e-01,  -1.09697792e+00,   1.76037222e+00],
                 [  2.29642400e+00,   2.93947657e+00,  -1.08260402e+01],
                 [ -6.80786682e+00,  -2.46504866e+00,  -8.85616729e+00],
                 [ -3.62020355e+00,  -6.87553002e+00,  -8.78354979e-01],
                 [  2.49129736e+00,   7.07063749e+00,  -7.98008847e+00],
                 [  6.23132652e+00,   3.20072406e+00,   3.21723614e+00],
                 [ -4.37725998e+00,  -4.35503800e+00,  -2.05829489e+00],
                 [ -7.11034455e-01,  -8.38953169e+00,  -3.09872153e-01],
                 [ -2.43397843e+00,   1.64653684e+00,  -1.93266153e+00],
                 [ -1.28183265e+00,  -1.79790724e-03,  -1.18025407e+00],
                 [ -4.93106889e+00,   2.19141795e+00,   5.18706373e-01],
                 [ -3.76595340e-01,  -1.81484471e+00,   8.26864767e-01],
                 [  2.98916023e+00,   8.78619578e+00,   5.38865207e-01],
                 [  2.92766422e+00,  -2.56109114e+00,   9.92489667e+00],
                 [  1.54189462e+00,  -5.95829331e-02,  -1.28053290e-01],
                 [  6.18425033e+00,  -4.70130968e+00,   5.67426022e+00],
                 [  2.72561665e+00,   2.96872574e+00,   2.44519456e+00],
                 [  1.51191951e+00,   1.09266556e+00,   2.15119936e+00],
                 [  3.53581588e+00,  -1.10388049e+00,   8.41720247e-01],
                 [ -6.88013914e+00,   3.84493743e+00,  -1.11155769e+01],
                 [  6.86280496e+00,  -3.65202444e+00,   3.05698915e+00],
                 [ -3.46747208e+00,   6.82843084e+00,   4.47109579e+00],
                 [ -2.02762383e+00,  -3.87751110e+00,  -3.00141874e+00],
                 [ -2.52905343e+00,  -5.35886853e+00,   4.72476493e+00],
                 [ -1.18953511e+00,   4.57237794e+00,   6.71426687e+00],
                 [  9.90249343e+00,  -3.19338440e+00,   2.45191609e+00],
                 [ -7.44427777e-01,  -1.27110606e+00,  -7.55091329e+00],
                 [  1.35347828e+00,   8.00543251e-01,  -1.00176248e+00],
                 [ -4.57505947e+00,   4.17550204e+00,   3.09610511e+00],
                 [ -4.36214069e-01,   2.08038878e-01,   7.69095041e-01],
                 [  6.88609561e+00,   1.05560999e+00,  -6.18291404e-02],
                 [  4.61619637e+00,  -2.97869979e+00,  -7.76204472e+00],
                 [ -8.19446924e-01,  -2.17399344e+00,  -6.03462226e-02],
                 [ -8.44008138e+00,  -1.22019503e-01,  -6.48919978e+00],
                 [  5.57906841e-01,  -6.97348914e-01,  -7.88624289e-01],
                 [ -3.48150953e-01,  -5.17499893e-01,   3.11667609e-01],
                 [ -5.11093444e-02,  -5.11178692e-01,  -3.18896537e-02],
                 [ -1.14558762e+00,  -4.97610312e+00,  -2.93828632e+00],
                 [  2.24035324e+00,   2.01733023e+00,   1.43277453e+00],
                 [ -3.65631072e+00,   3.21063370e+00,   8.59732468e+00],
                 [ -3.85220530e+00,   3.01177239e+00,  -6.03086825e+00],
                 [ -2.33157292e+00,  -3.78313391e+00,   5.50399175e+00],
                 [  7.76356842e+00,   5.29914717e+00,   3.93516557e+00],
                 [ -8.19422115e-01,  -6.22175743e-01,   3.07489827e+00],
                 [  2.81084752e+00,   4.40625301e+00,   4.28556329e+00],
                 [  4.40329994e-01,  -1.15833502e+00,  -8.75856860e-02],
                 [  2.72702687e+00,   7.60576408e+00,   2.58052362e+00],
                 [ -1.63805625e+00,   1.48632083e+00,   9.50836848e-01],
                 [  4.93515338e+00,  -9.94121264e+00,   1.87758763e+00],
                 [ -5.26618924e+00,   3.39274750e+00,   2.62345020e+00],
                 [  1.76479206e+00,  -1.70943749e+00,  -7.16819102e-01],
                 [  4.48407488e-01,  -5.35597541e+00,   1.18383957e+00],
                 [ -2.42673644e-01,   1.24497183e+00,   8.68223906e-01],
                 [ -3.12802924e-01,  -4.89952502e-01,  -6.18365917e-01],
                 [  1.30877994e+00,   3.21073710e-01,   2.50660184e-01],
                 [ -5.16336080e+00,   4.86744205e+00,   2.62599479e+00],
                 [ -5.91337088e+00,  -3.72998386e+00,   3.78212846e+00],
                 [ -9.16045937e-01,   6.54155349e+00,   6.05201622e+00],
                 [  6.89483411e+00,   9.58279415e-01,   5.14788444e+00],
                 [ -1.70134694e+00,  -1.31596471e+00,  -7.15604087e+00],
                 [ -5.70638083e-01,   5.08641699e+00,  -4.65196279e+00],
                 [  2.51612774e-01,   2.15158177e+00,   8.65287695e+00],
                 [  2.17671519e+00,   1.55749175e+00,   3.27740713e+00],
                 [  8.54820934e-01,   1.10414559e+00,   1.67308209e+00],
                 [  3.11143408e+00,   8.45627350e+00,  -2.10740965e+00],
                 [ -1.29406666e+00,   2.31607979e+00,   9.57242379e-02],
                 [  5.25729764e+00,  -7.83053477e+00,   9.46404955e-01],
                 [  3.62214530e+00,   4.94369336e+00,   1.11926970e+01],
                 [  1.50998673e+00,  -1.63389062e-01,  -2.72496318e+00],
                 [ -3.53864591e+00,  -7.29421173e+00,   2.65561772e+00],
                 [  1.88744104e-01,   4.50807763e-01,   1.45339238e+00],
                 [ -6.84603294e-02,   1.11153217e+00,  -9.87799667e-01],
                 [ -1.29340792e+00,  -1.62391223e+00,  -1.16886592e+00],
                 [ -1.47801193e+00,   5.09131305e+00,   5.92400308e+00],
                 [ -2.79170300e+00,  -1.56636361e+00,   1.10400733e+00],
                 [  6.07299480e-02,  -1.11388089e+01,  -3.64451720e+00],
                 [ -6.40432569e+00,  -3.35760007e+00,  -3.55680303e+00],
                 [  6.85012611e+00,   2.18941640e-01,   2.44365264e+00],
                 [  2.87855329e+00,  -3.74602482e+00,  -7.08285351e+00],
                 [  6.76144945e+00,  -4.43936400e+00,  -2.30369417e+00],
                 [  2.29711333e+00,  -4.63972958e+00,  -5.26658096e+00],
                 [ -1.63386245e+00,  -1.51987538e+00,  -9.14653299e-01],
                 [ -3.19802744e-01,   1.67185316e+00,   3.36957381e-01],
                 [ -1.48121359e+00,   1.35593529e+00,   1.31061214e+00],
                 [ -9.58804940e-01,   3.18803687e+00,   3.73694420e+00],
                 [ -5.05938343e+00,  -4.92736180e+00,  -5.98961340e+00],
                 [  6.68049728e-01,  -5.10298033e-01,  -1.76028455e+00],
                 [  5.25738738e+00,  -6.57606055e+00,  -1.78438907e+00],
                 [  1.39161893e+00,  -1.07239590e+00,   1.64973096e+00],
                 [ -3.05618936e+00,   9.52865344e-01,  -1.11840408e+00],
                 [ -4.33410292e-01,  -1.43497862e+00,  -1.13610181e+00],
                 [ -2.99189285e+00,   4.40788157e+00,   3.78998461e-01],
                 [ -1.23702565e+00,   3.74240441e+00,   6.70637636e-03],
                 [  1.99789153e+00,  -4.13463678e+00,   5.89368385e+00],
                 [  8.04717787e+00,  -5.20290134e+00,  -1.21011968e+00],
                 [ -3.40634837e+00,   3.09494046e+00,  -3.16690836e+00],
                 [ -4.20722226e+00,   9.52916917e+00,  -2.65907222e+00],
                 [ -1.73175258e+00,  -9.85886191e-01,   1.01613368e+01],
                 [  8.78343400e+00,  -2.52637220e+00,   4.28367439e-01],
                 [ -1.57200818e+00,   2.37907726e-01,  -1.81162962e+00],
                 [ -3.73116427e+00,  -5.23725815e+00,   6.52490086e+00],
                 [ -3.30867391e-01,   1.19576224e+00,   4.54357596e-01],
                 [ -4.76356584e+00,   5.70918829e+00,  -4.37008243e+00],
                 [  2.29775647e+00,  -1.69766681e+00,  -3.55898299e+00],
                 [ -3.43226144e-01,   4.53147097e-01,  -1.52998942e+00],
                 [ -2.20271169e+00,  -1.39469316e+00,   8.48029378e+00],
                 [  4.60589576e-01,  -2.27787870e-01,   9.89646444e-01],
                 [ -2.14833460e+00,  -1.14248706e+00,  -3.21224580e-01],
                 [ -8.33177410e-01,  -1.56362200e+00,  -7.17974745e-01],
                 [  4.38632005e+00,   7.01893170e+00,  -1.59177997e+00],
                 [  8.67162117e+00,  -5.73894980e+00,   2.33570646e+00],
                 [ -2.60062555e+00,  -1.06688379e+01,   3.40961359e+00],
                 [  7.41115878e-01,   6.19658439e+00,   5.34375561e-01],
                 [  4.38784550e+00,  -2.77308763e+00,  -8.45285161e+00],
                 [  2.50087918e+00,  -8.85978567e+00,  -7.51461260e+00],
                 [  5.29540237e+00,   6.01964473e+00,   5.92730966e+00],
                 [ -5.36106482e+00,  -3.48894952e+00,  -3.25504292e+00],
                 [ -2.09947311e+00,   1.78360641e+00,  -2.05097814e-01],
                 [  8.76825564e-01,   4.56506211e+00,   8.28574221e+00],
                 [ -1.40040835e+00,  -1.00830600e+00,  -5.91964455e-01],
                 [  4.89714391e+00,  -6.38828505e+00,   1.25683596e+00],
                 [ -7.19437787e+00,  -1.30474665e+00,   1.27661724e+00],
                 [  1.15352937e+00,  -6.44593478e-03,   1.84638011e-01],
                 [  1.25899369e+00,  -8.71781238e-01,   1.76022838e+00],
                 [ -7.07714337e-01,   2.70735265e-01,   1.48335345e-01],
                 [ -2.02544076e+00,   1.92449049e+00,  -1.39178817e+00],
                 [  1.64980390e+00,   1.37306212e+00,   3.36271721e-01],
                 [ -1.49251015e+00,  -5.09424882e+00,   2.43515506e+00],
                 [ -1.61865304e+00,  -3.92064248e+00,  -1.89206013e+00],
                 [ -9.08820860e+00,  -2.05751088e+00,   3.64665257e+00],
                 [ -8.91252411e-02,  -6.35058060e+00,  -2.00744083e+00],
                 [  9.55475669e-02,   4.14752391e+00,  -7.98597722e+00],
                 [ -8.70297676e-01,   9.04058565e+00,   1.23963166e+00],
                 [ -1.33035759e+00,   5.98451566e+00,   2.14291726e+00],
                 [ -4.97977604e+00,  -4.25001141e+00,   6.83358370e+00],
                 [ -1.63890560e+00,   1.06827624e+00,  -6.70260700e-02],
                 [ -2.07875098e+00,   3.69804679e+00,   4.88906329e+00],
                 [ -2.48955513e+00,   3.79918682e-01,   6.32996050e-01],
                 [  4.91517260e+00,  -2.34493543e+00,  -7.71774149e+00],
                 [ -8.74289934e-01,  -2.43431329e+00,  -7.65273513e+00],
                 [ -8.99386503e-01,  -4.85101967e-01,   1.44681394e+00],
                 [ -5.20175523e+00,  -2.90604286e+00,  -6.23065461e+00],
                 [  4.97648068e-01,  -2.74355745e+00,  -7.50588037e-01],
                 [  1.39591243e+00,   7.75670157e-01,   2.01930815e-01],
                 [  2.15733181e+00,  -1.86303427e+00,  -9.40820529e-01],
                 [  7.04557474e+00,   5.92524893e-01,   2.37636588e-01],
                 [  6.40320165e+00,   7.69322335e+00,   4.89122738e-02],
                 [  3.60297620e+00,  -2.44910133e+00,   1.24702856e+00],
                 [  5.16780767e+00,   1.49715369e+00,  -6.00682751e+00],
                 [  5.96328771e-01,  -1.57680229e+00,  -4.21867613e+00],
                 [  3.80515755e-01,  -2.40318594e+00,   3.54007710e+00],
                 [ -4.19806920e+00,   6.36854134e+00,   1.16147859e-01],
                 [  2.51579553e+00,  -1.92777551e+00,  -2.16910896e+00],
                 [  2.80940983e-01,   1.04742565e+00,   7.44281524e-01],
                 [ -5.33752071e-01,   4.80256766e-01,  -5.39557095e-01],
                 [  2.08746922e+00,  -5.12612068e-01,   2.54191067e+00],
                 [ -6.55141698e+00,   1.68442480e+00,  -8.50474765e+00],
                 [  6.87826315e+00,  -7.65015069e+00,   3.59275771e+00],
                 [ -1.68338839e+00,  -2.25144602e+00,   1.73820111e+00],
                 [ -9.68367189e-01,  -6.59613969e+00,  -2.24198985e+00],
                 [  1.23314399e+00,  -8.08225362e-01,  -1.25392541e-01],
                 [ -3.90396405e-01,  -5.11548149e-01,  -3.88457686e-02],
                 [ -3.67978434e+00,   7.56675961e-01,  -2.57538036e+00],
                 [ -3.33366745e+00,  -4.25514679e+00,  -7.00932755e+00],
                 [ -3.56223885e+00,   6.72658396e-01,   3.46577348e+00],
                 [ -1.58572832e-01,  -7.57804394e+00,   4.15888052e+00],
                 [ -5.77570508e+00,   2.71167788e+00,  -3.68586915e+00],
                 [ -3.10465841e+00,  -7.29377919e+00,   1.32683311e-01],
                 [  1.15664060e+00,   1.12457534e+01,   6.89080438e-01],
                 [ -2.93859819e-01,   7.04927035e+00,   3.48266252e+00],
                 [  7.23422025e-01,   3.84793777e+00,  -6.96951624e-02],
                 [  1.77926387e-01,  -9.28162462e-01,  -1.04085423e-01],
                 [ -2.14072301e+00,   3.04651817e+00,  -3.55529557e+00],
                 [ -1.32396968e+00,  -1.24351340e+00,  -1.83549091e+00],
                 [  2.78716053e+00,   4.68949975e+00,  -9.72044687e-01],
                 [  1.99710769e+00,   4.95712920e+00,   3.43405800e+00],
                 [ -2.13735671e+00,   2.13062774e+00,   3.16337728e+00],
                 [  3.51299366e+00,  -2.31025848e+00,   2.77999338e+00],
                 [  5.94110684e-01,   1.04636581e+00,  -1.07051883e+00],
                 [ -2.07145053e-01,  -1.35856570e+00,  -8.70296188e-01],
                 [ -2.19454903e-01,  -5.58898905e-01,  -1.08368373e+00],
                 [  4.29299790e-01,  -1.49117919e+00,   3.27640974e+00],
                 [ -1.04930167e+00,  -8.32906448e+00,   5.04716060e+00],
                 [  1.90258420e+00,   8.64529596e+00,   4.58751689e+00],
                 [  3.44741742e-01,  -5.38610180e+00,   5.53885500e+00],
                 [ -4.63790657e+00,  -4.67693721e+00,   3.19902859e+00],
                 [ -5.57904308e+00,  -2.21528773e+00,  -9.16081675e+00],
                 [ -9.42540499e-01,   9.15544261e+00,  -3.01486988e+00],
                 [ -4.01272237e+00,   1.03400674e+00,  -1.85650905e+00],
                 [  1.81979412e+00,  -1.18946946e+00,  -2.34187507e-01],
                 [ -1.39581139e-01,   5.02749975e+00,  -4.56665280e+00],
                 [  2.19411999e+00,   3.44489135e+00,  -2.46238075e+00],
                 [ -2.55813745e+00,   1.07643263e+01,   4.92646152e-01],
                 [ -8.26138565e-01,   8.86636135e+00,   4.98802255e+00],
                 [ -9.69816535e-01,   1.61705940e+00,   2.58784982e-02],
                 [  7.24186067e+00,  -4.76657850e+00,   3.56274430e-01],
                 [ -3.05814543e-02,   2.82211440e+00,  -1.96183093e+00],
                 [  2.11436999e+00,  -1.09880522e+00,  -8.01557885e-01],
                 [  1.94049841e+00,   1.92629120e-01,   1.18514148e+00],
                 [ -7.04374608e-01,   4.21687118e-01,  -2.49297170e-01],
                 [  5.02439320e+00,  -5.07087276e-02,   1.31026778e+00],
                 [ -1.18663620e+00,  -3.32921468e+00,  -2.84730223e+00],
                 [  7.06303734e+00,   4.00646600e+00,   5.73590487e+00],
                 [  3.30903849e+00,   6.73996341e+00,   6.84220724e+00],
                 [  1.26056695e-01,  -1.05501767e+00,  -1.30473461e+00],
                 [ -9.16616242e-01,  -7.56920580e-01,   4.24011176e+00],
                 [ -1.04613997e-01,  -6.45533493e+00,  -2.37465832e+00],
                 [  6.48912315e-01,  -6.40346356e-01,   4.89736434e-01],
                 [  6.95821819e+00,  -6.37623469e+00,   6.05585344e-01],
                 [ -2.49960101e+00,   8.23314128e-01,  -2.10644537e+00],
                 [  6.60091191e+00,  -4.56752089e+00,   4.08053057e+00],
                 [ -8.38974746e+00,   4.06957453e+00,   3.45237589e+00],
                 [  5.97316201e-02,   2.20562942e+00,  -1.07891885e+00],
                 [ -6.75760539e+00,  -2.83162938e+00,   2.79831418e-01],
                 [  2.61514084e-01,  -1.13497366e+00,  -6.45245275e-01],
                 [  1.58413207e+00,  -4.06609531e-01,  -4.21027060e-01],
                 [  1.81059804e+00,  -5.49961478e-01,  -7.92247230e-01],
                 [ -5.09587762e-01,  -3.29671732e+00,  -8.62897060e-01],
                 [  1.04642540e+01,   3.32493219e+00,  -2.02883910e+00],
                 [ -9.60494552e+00,  -4.31391850e-01,  -3.27179017e+00],
                 [ -4.52999838e+00,  -4.47131377e+00,   5.42963903e+00],
                 [ -4.74009325e+00,   1.19713368e+00,   7.67976720e+00],
                 [  7.24851312e+00,  -1.72259404e-01,  -5.49530737e+00],
                 [  1.44795041e+00,   1.74127399e+00,   2.12781091e-01],
                 [ -1.40115329e+00,   1.21384700e+00,  -3.21264005e+00],
                 [  3.05286093e-01,   8.13966588e-01,  -9.80091377e-01],
                 [ -7.63586096e+00,   3.62325164e+00,  -1.24767857e+00],
                 [  1.91832798e+00,   1.38210847e+00,   1.43429667e+00],
                 [  1.04985267e+00,  -1.56252797e+00,  -8.18621615e+00],
                 [  1.50569880e+00,  -2.70111910e+00,  -6.28027763e-01],
                 [ -1.49752067e+00,  -8.84074368e-01,  -1.33640128e+00],
                 [ -2.68749626e+00,  -4.50453802e+00,   4.90518178e+00],
                 [  3.22599374e-01,  -3.33324564e+00,   1.14974864e+00],
                 [ -2.53838210e-01,   1.91067863e+00,  -3.72101488e-01],
                 [ -2.02600708e+00,  -4.22734397e-01,  -7.23567873e-01],
                 [  8.28295232e+00,  -7.61365143e+00,   5.61184435e+00],
                 [  8.79162977e+00,  -5.58816119e+00,   2.50831410e+00],
                 [  6.04244397e-01,  -6.98655419e+00,  -2.37657456e+00],
                 [ -9.35481067e-01,  -1.68277088e+00,  -2.60602597e-01],
                 [  2.12244746e+00,   1.01207677e+01,   9.04767233e-01],
                 [ -1.73310465e+00,  -5.20960673e+00,  -1.98739275e+00],
                 [ -3.46666609e+00,  -7.52202896e+00,  -4.23583964e+00],
                 [  4.86151892e-01,  -6.92600447e+00,  -8.44409449e-01],
                 [  4.37022499e-02,   7.13904454e-01,  -1.85391877e-01],
                 [  5.22779663e+00,  -1.28714872e+00,   2.02852040e+00],
                 [ -5.37819675e-01,  -2.36859742e-01,  -1.84992335e+00],
                 [  3.47321632e+00,   2.81568321e+00,   1.42389137e+00],
                 [ -7.02295267e+00,   5.63344067e+00,   2.21840352e+00],
                 [ -2.75850145e-02,  -6.41626701e-02,   5.53993028e-01],
                 [ -1.31450852e+00,   8.35004978e+00,  -7.36341277e+00],
                 [  8.85366263e-01,  -2.19404250e-01,   7.88184083e-01],
                 [ -4.09484775e-01,  -1.15339194e-01,   1.58478337e+00],
                 [  1.89882034e+00,  -1.16910074e+00,   2.12896789e+00],
                 [  6.81878135e+00,  -5.52308566e+00,  -3.31470259e-01],
                 [  7.58832417e-01,  -3.34709062e+00,  -1.53665544e+00],
                 [  6.62252444e+00,  -2.23342333e+00,  -2.60683319e+00],
                 [ -2.34207523e+00,   5.06303415e+00,  -5.17391946e-01],
                 [ -7.62572995e+00,   4.59432555e+00,  -5.28305474e+00],
                 [  3.20310618e-01,   4.95631391e+00,   1.70114868e+00],
                 [  3.93408882e+00,   9.53507500e-01,   1.66331553e+00],
                 [ -1.18511868e+00,  -9.17155465e+00,   1.57402149e+00],
                 [  8.86354108e-01,   1.84943637e+00,  -9.43594963e-01],
                 [ -3.36105235e+00,   3.00585665e+00,  -2.27390839e+00],
                 [  2.40577926e-01,   6.26136930e-01,   9.20612609e-01],
                 [  2.61320590e+00,  -4.36626981e-01,  -9.00521689e+00],
                 [ -3.88570761e+00,  -3.08894241e-01,  -1.81040971e+00],
                 [  4.04019865e-01,   2.28529889e+00,   9.19144362e-01],
                 [ -7.92969556e+00,   3.21310738e-01,  -6.37510485e+00],
                 [ -8.92812986e-01,  -1.57974988e+00,   2.78665509e-01],
                 [ -6.54664435e-01,  -2.14875599e+00,  -1.90623992e+00],
                 [ -8.96570000e-01,   4.28654144e-01,   1.28012203e+00],
                 [  3.54927652e+00,  -6.58074512e+00,  -3.31599304e+00],
                 [  2.79407404e+00,  -7.66332210e+00,   3.14283830e+00],
                 [  8.81586491e-01,  -2.45378938e-01,   3.93576815e+00],
                 [  2.65532361e-01,   3.98640202e+00,   9.16813217e+00],
                 [ -4.75488320e+00,   5.43278287e+00,  -4.57923237e+00],
                 [ -1.81745430e+00,  -7.57606139e+00,   2.58006925e+00],
                 [ -4.13574955e+00,  -7.83333588e+00,  -8.84001556e-01],
                 [  4.25462843e+00,   2.71211258e+00,  -4.61403037e+00],
                 [  1.50824274e+00,   9.34132438e-01,  -2.56053715e+00],
                 [ -3.82010596e+00,  -6.49623704e+00,   5.30488143e+00],
                 [ -9.53929569e-02,   2.44548773e+00,  -9.89474453e-01],
                 [ -3.26269635e+00,   5.89267231e+00,  -4.99154630e+00],
                 [ -3.28679171e+00,  -4.30749643e+00,  -6.75059892e+00],
                 [ -9.73395287e-01,  -3.93798872e-01,  -6.24181193e-01],
                 [  4.45869395e+00,   5.05937964e-01,  -3.69287610e+00],
                 [ -3.35705616e+00,   5.60998072e-01,  -2.70499519e+00],
                 [  3.25980606e-01,   9.24322570e-01,   5.44950266e-01],
                 [  6.16129885e-01,   2.13704153e+00,   3.92342464e-01],
                 [  4.74495790e+00,   1.13797535e+00,  -2.75572863e+00],
                 [  1.67840522e+00,  -2.27876458e+00,  -6.57345211e+00],
                 [  5.04290045e+00,  -3.11029211e+00,  -4.84579740e-01],
                 [ -9.30314073e+00,   2.64921405e+00,  -3.68755903e+00],
                 [  8.92175851e-01,   3.80133948e+00,   4.58178512e+00],
                 [  9.06167239e+00,   2.00596685e-01,  -5.16000851e+00],
                 [  7.96222690e+00,  -2.89984413e-01,   4.28284973e+00],
                 [ -3.74599796e+00,  -3.48305726e+00,  -3.95003415e+00],
                 [ -2.82688603e+00,   1.57534973e+00,  -3.51028189e-01],
                 [ -6.09281921e+00,  -4.56709803e+00,  -1.11233711e-01],
                 [ -1.90429957e+00,  -1.99081994e+00,   1.20445703e-01],
                 [ -2.30278883e-01,  -2.80444636e+00,   6.12148224e+00],
                 [  6.50571115e-01,  -1.60564336e-01,  -4.61421601e+00],
                 [ -3.08571540e-01,  -3.28199849e-01,  -5.80296532e-01],
                 [ -1.05548649e+00,  -4.39551831e-01,   1.25419807e+00],
                 [  2.23228221e+00,   5.86074885e-01,  -2.76096750e-02],
                 [  4.12106918e-01,   5.62414617e-01,   1.20268581e+00],
                 [ -4.72420285e-01,  -1.03207730e-01,  -9.50497017e-01],
                 [  1.57217092e+00,   9.98851359e-01,   2.51205409e+00],
                 [  7.73881565e+00,  -5.09722316e+00,  -1.29481719e+00],
                 [ -3.30507842e+00,   9.21557911e-01,   4.49419479e+00],
                 [  8.39400108e+00,   6.48258162e-01,  -8.22401092e+00],
                 [  9.56223862e+00,  -6.74214612e-01,  -5.12971729e+00],
                 [  4.22476598e+00,   4.85573441e+00,   6.04791299e+00],
                 [  2.29277712e+00,  -3.03095416e+00,  -8.03243444e+00],
                 [  1.58099876e+00,  -6.10928724e-01,   9.71378187e+00],
                 [ -5.13478534e-01,  -1.21293236e+00,  -1.30998602e-01],
                 [ -2.53473319e+00,  -8.01906603e+00,   4.22153834e+00],
                 [  2.68484555e+00,  -1.17716608e+00,   7.21257362e-01],
                 [ -6.25939461e+00,  -3.02651922e+00,   3.41891898e+00],
                 [  3.81959333e+00,   3.23741835e+00,  -5.89787194e+00],
                 [ -6.81777837e-02,   9.49906375e-02,  -1.79524016e-02],
                 [  5.49565039e+00,   5.83552757e+00,   5.74853502e+00],
                 [ -1.03478454e+00,   7.69348632e-01,   9.90815039e-01],
                 [  6.02372882e-01,   1.35842045e-01,   1.19065381e+00],
                 [  1.29741596e+00,  -1.95974558e-01,  -1.49937805e+00],
                 [  3.94108010e+00,   4.04838338e+00,   1.00744383e+01],
                 [ -1.55442768e-01,   5.38002770e+00,  -7.63843894e+00],
                 [  1.00454163e+01,  -2.94671932e+00,   8.77178192e-01],
                 [ -3.79900696e+00,  -4.60524138e+00,   8.01465872e+00],
                 [  1.36483798e+00,  -1.79984449e+00,   7.07965424e+00],
                 [ -3.70390974e+00,   1.00948541e+01,   6.48206351e-01],
                 [ -4.15188302e+00,  -3.82306739e-01,  -5.25475816e+00],
                 [  6.16166388e+00,   5.42178254e+00,   1.42012105e+00],
                 [  4.06836924e-01,  -5.61075459e-01,   1.30029606e+00],
                 [ -5.28946244e+00,  -1.65396290e+00,   6.10439625e+00],
                 [  3.54603492e-01,  -7.18632986e-01,  -2.17868050e+00],
                 [  4.74666169e+00,  -1.16228220e+01,  -3.22702987e+00],
                 [  4.49395955e+00,  -1.16051828e+01,  -4.39401254e+00],
                 [  7.66206308e-01,   2.44793926e+00,   2.33124800e+00],
                 [  1.34170289e+00,   4.06249293e+00,   5.78714789e-01],
                 [  1.51037573e+00,   1.86136036e+00,   1.06557566e+00],
                 [  8.88258345e-01,   1.29651076e+00,   1.77228637e+00],
                 [ -1.93274180e+00,   2.49678797e-01,  -1.33679865e+00],
                 [  3.63016051e-01,  -2.67401446e+00,   1.08623751e+01],
                 [  1.21337338e+00,  -3.49456801e+00,  -7.55472703e+00],
                 [ -9.63917737e-01,   6.55513690e+00,   5.05501958e+00],
                 [  2.03928457e+00,   1.84402910e+00,  -1.14799059e+01],
                 [  3.57823691e+00,  -1.83304136e+00,   1.21578646e+00],
                 [  3.47998597e+00,   1.12138222e+01,  -1.52074589e+00],
                 [  1.15407408e+01,  -5.03917958e+00,  -6.64483589e-01],
                 [  6.14990625e+00,  -3.75591944e+00,  -5.28957280e+00],
                 [  3.58654680e-01,  -5.43098610e-01,  -2.11075647e-01],
                 [ -8.54980533e-01,   9.64422414e+00,  -2.00179370e+00],
                 [  2.41333207e-01,  -2.09775032e-01,   2.48603154e+00],
                 [  9.31755643e-01,  -1.13350270e-01,   1.81424272e+00],
                 [  5.13093734e+00,   4.03687639e+00,   4.40290832e+00],
                 [ -1.47477810e-01,  -1.23192049e+00,  -1.11653846e+00],
                 [ -4.79378628e+00,   1.62704874e+00,   6.65572283e+00],
                 [ -7.26618950e-01,   6.31699772e-01,  -7.94619311e-01],
                 [  1.04408723e+00,   1.13524747e+00,   1.38624430e+00],
                 [ -1.58607228e-01,   3.91257136e-02,   4.14948189e-01],
                 [ -5.82420173e+00,  -9.57311321e+00,  -1.20679476e-02],
                 [ -5.74447076e+00,  -3.86695069e+00,  -2.29807524e+00],
                 [  3.22154471e+00,  -4.02560114e+00,   7.15610741e+00],
                 [  1.18386297e+00,  -4.98385499e+00,   1.41098940e+00],
                 [ -1.13395650e+01,   2.02050141e+00,   5.58166724e-01],
                 [ -4.42000347e+00,  -3.14810255e+00,  -1.13506786e+01],
                 [ -5.05059781e+00,   8.05203341e+00,   1.85179416e+00],
                 [  1.54476278e+00,   8.50148715e+00,  -3.59294652e+00],
                 [ -1.85376855e-01,  -1.38103217e+00,  -9.58651853e-01],
                 [  2.75109800e+00,  -6.91804006e-01,   3.28418768e+00],
                 [  6.59366210e-02,   1.81480602e+00,  -6.12751711e-01],
                 [ -6.40683998e+00,  -3.32942403e+00,   5.75697310e+00],
                 [ -7.19883195e-01,   4.85774549e+00,   1.47513926e+00],
                 [  1.37920804e+00,   3.28657092e-01,   2.36782119e+00],
                 [  1.69516597e-01,  -9.45067565e+00,  -2.11340860e+00],
                 [ -3.15588859e-01,   8.33230190e-01,   4.56323580e-01],
                 [  1.39971949e+00,  -1.18235591e+00,  -9.16215203e-01],
                 [  2.51941183e-01,   3.00741421e-01,   1.30010583e+00],
                 [  8.97016541e+00,  -2.25616372e+00,   7.24808812e+00],
                 [  1.69852318e-01,   3.61207762e-02,  -1.16720300e+01],
                 [ -3.11910798e+00,   6.00861840e+00,  -1.07186272e+00],
                 [  5.83956164e+00,  -1.04562564e+01,  -4.77924889e-01],
                 [ -1.08684433e+00,   2.91722861e+00,   5.87419388e-01],
                 [  2.02875579e+00,   9.41396798e+00,   2.64241406e+00],
                 [  1.70373326e+00,   3.84579483e+00,  -6.20881422e+00],
                 [ -2.43595918e+00,  -7.36890250e+00,   2.30327487e+00],
                 [ -8.49452729e-01,   8.23067995e-01,  -1.03764963e+00],
                 [ -5.01999423e-01,  -1.15871138e+00,   5.63894906e-01],
                 [  1.31650713e+00,   2.23072302e-01,  -5.39227087e-01],
                 [  4.99051128e+00,   1.20548679e+00,  -1.59592183e+00],
                 [  1.84154845e+00,   4.02877306e-01,  -8.02914749e+00],
                 [  1.81512181e+00,   1.33533625e-02,   3.07869409e-01],
                 [ -5.27986787e+00,   4.53757231e+00,  -1.11049773e+00],
                 [ -2.10040092e+00,   1.11801597e+00,   6.04660391e-01],
                 [  2.27793967e+00,   4.27829806e-01,  -2.49128636e+00],
                 [ -1.80684401e+00,   7.36713806e-02,  -2.41040513e+00],
                 [ -7.32118751e-01,   6.65317303e+00,   3.28859971e+00],
                 [ -1.26382435e+00,   9.04790800e+00,   1.78442413e+00],
                 [ -3.06172762e+00,  -4.81487309e+00,   5.67847708e+00],
                 [ -2.90678477e+00,  -5.84658776e-01,  -1.09735320e+01],
                 [ -3.96658540e+00,  -5.41245327e+00,   1.04630138e+01],
                 [ -1.90131762e+00,  -1.96545307e+00,  -7.72019299e-01],
                 [  2.83954374e+00,  -1.32263571e-02,  -3.06559755e+00],
                 [ -7.21484998e+00,   2.88423583e+00,  -3.57526487e+00],
                 [  2.52283265e+00,  -1.68223781e-01,  -5.34531328e-01],
                 [ -9.18996611e+00,  -1.65764846e+00,   4.18551162e+00],
                 [ -1.75786923e+00,  -3.52275169e+00,  -6.81830500e-01],
                 [  4.61367060e+00,  -2.40541562e+00,   2.39503727e+00],
                 [  2.59887256e+00,   2.39876143e+00,  -2.06208446e+00],
                 [ -9.03136241e-01,   2.60219589e+00,  -1.74801133e+00],
                 [ -5.04740023e+00,   8.32043283e+00,   3.78009948e-01],
                 [ -8.66430910e-01,  -2.87664082e+00,  -3.69891993e-01],
                 [  2.34641480e+00,  -6.61104629e-01,   2.04358111e+00],
                 [  1.30597092e+00,  -9.94007897e-02,  -3.87221961e-01],
                 [  6.03867020e-01,  -1.28243592e+00,   8.48022334e-01],
                 [  6.38564844e+00,  -4.20013693e+00,   4.36165561e+00],
                 [ -1.16271902e+01,   5.85217025e+00,   1.94128382e+00],
                 [  2.04368365e+00,  -3.02364442e+00,   2.54116283e-01],
                 [  6.40439406e+00,  -6.83444646e+00,  -7.80468520e-01],
                 [ -3.74594782e-01,   6.91458614e-01,   1.08431272e+01],
                 [  1.12157822e+00,   7.56169900e+00,  -6.16562633e+00],
                 [  6.58286006e+00,  -3.99719932e-01,   5.92394961e+00],
                 [ -6.31035391e-01,   7.01447190e-01,   3.95028704e-01],
                 [ -7.49037381e+00,   5.86866597e+00,  -3.16728621e+00],
                 [  1.78799827e+00,  -9.68042119e-01,  -2.72010679e-01],
                 [ -3.89861991e-01,   6.60093699e+00,   3.07311098e+00],
                 [  6.28540214e+00,  -1.16917995e+01,  -3.35516799e+00],
                 [ -1.59733080e+00,   4.17303512e-02,   8.69180584e-01],
                 [ -5.30610408e+00,   1.85083673e+00,   8.28366507e+00],
                 [ -3.12367969e-01,   1.09217133e+00,  -2.12368037e+00],
                 [  1.20196996e+00,   2.36265110e+00,   1.92764531e+00],
                 [  1.52934128e-01,   1.31856993e+00,  -8.15095232e-01],
                 [ -5.78996420e+00,   2.02355357e-01,  -6.56252702e+00],
                 [ -1.17856978e+00,  -6.62069493e+00,  -8.27881063e-01],
                 [  1.40237234e+00,  -5.44634946e+00,   3.52017449e+00],
                 [  1.66423524e-01,   9.10696378e+00,   2.79393920e+00],
                 [ -3.01619753e+00,   3.19032615e+00,  -1.64384082e+00],
                 [ -7.16891260e+00,   1.24784208e+00,   6.53675650e-01],
                 [  4.32206276e+00,   7.13870601e+00,   2.61485456e+00],
                 [ -1.62127057e+00,   1.90144528e-01,   8.16980396e+00],
                 [ -1.70145748e+00,  -8.42720141e-01,   1.02619208e+00],
                 [  2.24169639e+00,  -7.46990550e+00,   8.03042809e-01],
                 [ -1.58499602e+00,  -1.49585722e-01,  -7.08244854e-01],
                 [ -5.60986328e+00,  -6.64785919e-01,   2.08762973e+00],
                 [ -4.33177116e+00,  -2.83950531e+00,  -4.90669341e+00],
                 [  9.39186294e-01,   1.06035641e+00,   3.60529630e-01],
                 [ -2.55838129e+00,  -1.19978538e+00,   9.29738733e-01],
                 [ -1.15995636e+00,   1.17999829e+00,   1.98170283e+00],
                 [  9.49189487e-01,  -5.47645004e-01,  -4.38391118e-01],
                 [ -1.28713115e+00,  -2.55204238e-01,   3.90802106e-01],
                 [ -5.02744369e+00,  -1.15807138e+01,  -6.78520400e+00],
                 [  8.54262247e+00,  -6.38311417e+00,  -9.96489585e-01],
                 [  2.67061119e+00,   1.32722316e+00,  -4.08634874e+00],
                 [ -6.11160775e+00,   4.96358158e-01,  -4.41286056e+00],
                 [  1.61861658e+00,   5.64184209e-01,   1.68543404e+00],
                 [  4.13703714e+00,   4.05358387e+00,   7.52659006e+00],
                 [ -7.38226497e+00,  -2.50428456e+00,   5.39766993e+00],
                 [  1.22067123e+00,  -6.74745007e-01,  -7.90378629e+00],
                 [ -7.37785839e-01,  -5.38403371e-01,  -1.47757327e-01],
                 [  3.81339099e+00,   6.96132533e+00,  -2.14851093e+00],
                 [ -1.56456705e+00,  -9.01319285e-01,   1.00092630e+00],
                 [ -4.39364078e+00,  -1.77050382e+00,  -3.29551486e+00],
                 [ -3.02602221e+00,  -8.68744855e+00,   1.76592336e+00],
                 [ -1.70854156e-01,  -9.62076379e-01,  -2.27672913e+00],
                 [  1.22913771e+00,  -2.55191349e+00,   4.79765875e+00],
                 [ -1.00421441e-01,  -6.45120029e-01,  -1.58977366e+00],
                 [ -1.79450061e+00,  -3.55426190e+00,   4.16970967e-01],
                 [  2.11632114e+00,  -2.60017918e+00,   1.11504268e-01],
                 [ -5.97972355e+00,  -4.42457285e+00,  -3.39016862e+00],
                 [  1.52811901e+00,   6.50048989e+00,   2.09117257e+00],
                 [ -2.95075954e+00,   7.45273073e+00,   4.86363094e+00],
                 [ -5.08427650e+00,  -5.26712133e+00,  -2.93469510e+00],
                 [  4.11598818e+00,   5.83390862e+00,   3.17583306e+00],
                 [  4.60440245e+00,   4.40370432e+00,   1.42006450e+00],
                 [ -5.46911904e+00,  -7.17224432e+00,   4.46895860e+00],
                 [  1.17310276e+00,   9.81183117e-01,  -5.09537133e+00],
                 [  4.12223349e+00,  -1.59848436e+00,   2.31118271e-01],
                 [  1.84485504e+00,  -9.34957354e-01,   9.56048702e+00],
                 [  4.91777887e-01,  -9.52471079e-03,  -1.04542153e+00],
                 [ -2.31547566e+00,   5.67599740e+00,   5.34414759e+00],
                 [ -6.02910790e+00,  -1.70663986e+00,  -9.00914409e-01],
                 [  4.66177993e-01,   1.99452235e-01,  -1.31189811e+00],
                 [ -6.43778076e+00,   1.40929912e+00,   1.29791209e+00],
                 [  2.02935490e+00,  -1.23457451e+00,  -9.89134054e-01],
                 [  4.19708206e+00,  -7.99523256e-01,   5.09562278e-01],
                 [ -6.00789221e-01,  -9.79876088e-01,  -1.15798798e+00],
                 [ -3.83579856e+00,  -2.43829138e+00,  -1.35318304e+00],
                 [  1.03629971e+01,  -3.11396306e+00,   4.85271781e+00],
                 [ -3.67850045e+00,  -1.72717705e+00,   8.27040905e+00],
                 [  1.77031858e+00,   2.10861319e+00,   3.54353342e+00],
                 [  2.76259157e+00,   2.03548367e+00,  -3.00211408e+00],
                 [ -8.54830462e+00,   4.89963007e+00,  -5.21069984e-01],
                 [  9.96030838e-01,   4.89741615e+00,  -8.80095617e+00],
                 [ -1.03229188e+00,  -6.78626271e+00,  -1.27621881e+00],
                 [  2.05652892e-01,   1.77846473e+00,  -2.33391822e+00],
                 [  6.09820573e-01,   3.05048520e+00,   8.05500321e+00],
                 [ -9.56100989e-01,   5.09623175e-01,   2.86266783e-01],
                 [  3.28172559e+00,  -8.75399236e-01,  -9.84721145e-01],
                 [ -3.94716154e+00,   7.74646658e-02,   1.79917231e-01],
                 [  1.95530107e-01,   1.03926789e+00,  -3.10307810e-01],
                 [ -6.13059450e+00,   5.18667224e+00,   9.75341097e-01],
                 [ -1.21624030e+00,  -3.83715910e-01,  -7.51155334e-01],
                 [ -1.38233564e+00,   3.62777266e-01,  -2.16717392e-01],
                 [ -6.90558797e-01,   9.02581573e-02,   1.15708249e+00],
                 [ -4.67218546e+00,  -7.80025294e+00,   6.73295641e+00],
                 [  6.93011109e+00,  -3.31615951e+00,  -2.74584189e+00],
                 [  5.32721984e+00,  -1.17043368e+00,  -1.56959671e+00],
                 [  1.48339805e+00,   3.55378197e-01,  -2.27536053e+00],
                 [  1.67838602e+00,   7.89798179e+00,   5.32694529e+00],
                 [  2.47479881e+00,   8.08216126e+00,  -1.82217948e+00],
                 [  3.54203980e+00,  -3.47962899e+00,   1.01656138e+01],
                 [ -5.50975635e+00,   1.50415444e+00,   1.02002235e+00],
                 [  2.37060146e+00,  -1.99272303e-01,  -8.60253357e-01],
                 [ -2.83718205e+00,   2.46840521e+00,  -5.84290478e+00],
                 [ -6.12733008e-01,   7.06286704e-01,  -4.20641691e-01],
                 [ -6.63283809e+00,   3.15191961e+00,  -9.19924889e+00],
                 [  4.08077983e+00,  -1.33201649e+00,   6.17337844e+00],
                 [  1.12576638e+00,   2.91841232e+00,   2.98592087e+00],
                 [ -3.89861831e+00,  -5.12853679e+00,  -6.79356070e+00],
                 [  1.50726660e+00,  -1.26634537e+00,   1.30856895e+00],
                 [ -2.77782836e+00,  -1.78356897e+00,   1.11848002e+00],
                 [ -2.12155106e+00,   3.41191941e-01,   1.18360386e+00],
                 [  9.49316576e-01,   4.23612667e+00,   4.01602156e+00],
                 [  2.99860985e-01,   1.01131806e+01,  -2.52110267e+00],
                 [  4.47897348e+00,   7.08319688e+00,   7.13318834e+00],
                 [ -5.28894189e+00,   6.17995016e+00,   4.60319393e+00],
                 [ -7.04058552e+00,   9.98712622e-02,  -5.89002033e+00],
                 [  2.21036991e+00,   6.94369857e+00,  -6.64952104e+00],
                 [ -6.64919931e+00,  -5.79948456e+00,   2.06337249e+00],
                 [  8.58688051e+00,  -5.26957016e-01,  -1.91045901e-01],
                 [  1.91448374e+00,   6.84986460e-01,  -8.52795362e-01],
                 [  3.96373308e+00,  -1.95597958e+00,   7.30007504e-01],
                 [  6.88400289e-01,   1.31107169e-01,   1.57966380e-01],
                 [  4.64571187e+00,   2.58106964e+00,  -7.93726605e+00],
                 [  1.68088841e+00,  -6.22892333e+00,   3.03267368e+00],
                 [  1.98387811e-02,   8.98614618e-01,  -5.98783171e-01],
                 [  1.04276351e+01,   3.21967578e+00,   1.40559575e+00],
                 [  3.66243123e+00,   8.35980731e-01,   2.81143528e+00],
                 [ -3.38873819e-01,   1.60068195e+00,   1.01570390e+00],
                 [ -1.49166096e-01,  -1.88105339e+00,   1.38868487e+00],
                 [ -2.49382429e+00,  -1.55242819e+00,  -1.03493474e+01],
                 [  1.62561921e+00,   7.05347911e-01,  -1.80980468e+00],
                 [ -1.28295561e+00,   7.72037812e-01,  -1.87122305e+00],
                 [  3.73063099e-01,   2.53984998e-01,  -1.03556848e+01],
                 [ -7.01280127e+00,  -2.40472882e+00,  -5.24997986e+00],
                 [  2.32361535e+00,  -6.74593130e+00,   6.56521501e+00],
                 [ -4.08190425e+00,  -8.18284932e+00,   7.23765059e-01],
                 [  7.62017086e+00,   2.80356325e+00,  -8.84705407e-01],
                 [  5.95633030e-02,   2.04825880e+00,  -7.41667740e-01],
                 [ -8.66088537e+00,  -4.86765253e+00,   3.02119568e+00],
                 [ -2.44650844e+00,   1.44062018e+00,   6.54247681e-01],
                 [  5.55718251e+00,  -1.17692382e+00,  -1.94449734e+00],
                 [ -5.23563428e+00,  -2.08527115e+00,  -5.16959388e+00],
                 [ -1.17638656e+00,   1.73907920e-01,  -2.88885500e+00],
                 [  2.57696440e+00,  -1.55296690e+00,   2.50964214e+00],
                 [  4.83171400e-01,   1.86052383e+00,   5.25107287e-01],
                 [  1.54344187e-01,   2.13429218e+00,  -3.10447085e+00],
                 [ -1.76036601e+00,  -1.40307236e+00,   1.39904343e+00],
                 [  1.63565938e+00,   3.32870165e+00,  -1.74414724e+00],
                 [  9.65288757e+00,   7.29950725e-01,   1.42491886e+00],
                 [  2.86768405e+00,   4.56505117e+00,   6.45017161e+00],
                 [ -9.05839209e+00,   8.91058740e+00,  -3.25969184e+00],
                 [  5.41891625e-01,   3.03555073e+00,  -6.67999988e+00],
                 [  6.95302596e+00,  -1.74907056e-01,  -1.63016588e+00],
                 [ -1.70751053e+00,   1.85329599e+00,  -2.55253855e+00],
                 [  1.64597410e+00,  -6.59310901e-01,   9.89210506e+00],
                 [  1.12335473e+00,  -9.19309036e-01,  -2.54199676e+00],
                 [ -5.35162039e+00,   3.79554907e+00,   3.77578824e+00],
                 [ -9.31267018e-01,   5.70037573e-01,   1.27498227e+00],
                 [  2.45875560e+00,   1.78251573e+00,  -2.72215152e+00],
                 [ -2.27330416e-01,  -8.23341484e+00,  -4.55935085e+00],
                 [  2.62043744e+00,  -1.06171672e+00,  -7.69274883e-01],
                 [  5.46250113e+00,  -8.41634844e-01,   2.58421959e+00],
                 [  1.54248161e+00,   1.88465475e+00,  -3.26625621e+00],
                 [ -7.29214355e-01,   7.49349771e-01,  -7.66893456e-01],
                 [ -3.69859513e+00,  -1.52184732e+00,  -5.70402451e-01],
                 [ -9.93612825e-01,  -4.57578293e-01,   9.34828945e-01],
                 [  6.98920783e+00,   4.22286026e+00,   6.35125638e+00],
                 [  5.21086083e+00,  -5.97385645e+00,  -1.79667359e+00],
                 [ -4.50825742e+00,  -7.80101939e+00,   4.29450694e-01],
                 [  3.67816056e+00,  -1.09030558e+00,  -1.91274372e+00],
                 [ -6.82667960e+00,  -6.94080478e+00,  -2.35005392e+00],
                 [  8.38548565e+00,  -1.67553617e+00,  -1.42403974e+00],
                 [  1.89009610e+00,  -1.51803972e+00,   3.54909736e+00],
                 [  2.40908398e+00,   1.54335241e+00,   7.99067098e-01],
                 [  5.67174886e+00,   6.26975401e-02,  -1.00696620e+00],
                 [  1.11683090e+00,  -7.83702322e-01,   9.44585140e-01],
                 [ -1.32993587e+00,  -7.57415094e+00,  -1.35933174e+00],
                 [ -5.45888052e+00,  -5.72427794e+00,   7.47312845e+00],
                 [ -3.15814742e-01,   1.71630114e-01,  -1.18727983e+00],
                 [ -3.83322814e+00,   1.40256666e+00,   2.46483856e+00],
                 [ -2.07155740e+00,  -7.98447223e-01,   2.86498806e-01],
                 [ -7.34952219e-01,  -1.33689070e+00,  -1.11933470e+00],
                 [  2.68694723e-01,  -3.46442254e-01,  -2.48016243e-01],
                 [  1.16193591e+00,   1.73799859e+00,  -3.80637140e+00],
                 [  2.90521842e+00,  -5.19895572e+00,  -3.00060478e-01],
                 [ -7.76875693e+00,   8.58278286e+00,  -1.87041462e-01],
                 [ -1.79013794e+00,  -4.24761414e+00,  -4.02994132e+00],
                 [ -5.76156548e+00,   1.49763333e+00,   2.61825312e+00],
                 [  7.91339633e+00,   7.71025856e-01,   4.58488924e+00],
                 [  9.65684290e+00,  -4.80950975e+00,  -4.64613997e+00],
                 [  2.54657234e+00,   2.40584790e+00,  -5.10611937e+00],
                 [  5.82601097e-01,  -9.51425422e-01,  -3.63218299e-01],
                 [ -4.16890534e+00,  -3.81993902e+00,   3.03711762e+00],
                 [ -1.08546471e+00,  -8.16841267e-01,  -1.06307277e+00],
                 [  5.38912139e+00,   9.70579364e+00,  -1.39534706e+00],
                 [ -7.26093381e+00,   2.63398750e+00,   3.36393647e+00],
                 [  6.32456458e-01,   6.43049021e-01,   9.16280337e-01],
                 [ -9.07342810e-01,  -1.56993628e+00,   2.70113589e+00],
                 [ -4.55136313e-02,  -4.51144339e-01,  -1.28828753e+00],
                 [ -6.36723088e-01,   5.48966679e-01,   5.13540440e-01],
                 [  1.14111874e+00,  -5.67219269e-01,   2.40836117e-02],
                 [  5.00222309e+00,  -1.23907987e-01,   6.49108521e+00],
                 [ -3.22773399e+00,   3.65182692e+00,   7.91029493e+00],
                 [  6.91048967e+00,  -3.44459757e+00,   1.02114067e+01],
                 [  1.02790443e+01,  -2.65949510e+00,  -2.88734223e+00],
                 [  5.29923494e+00,  -5.76452505e+00,   1.27874627e+00],
                 [ -6.45631393e+00,   1.80805798e+00,   3.90278011e+00],
                 [  4.92489814e+00,  -9.78108966e-01,  -7.51587651e+00],
                 [ -8.99990078e+00,   5.45600635e+00,  -3.56668176e+00],
                 [  5.68623639e-03,   1.86096764e+00,   1.77874983e+00],
                 [  1.08933167e+00,  -4.57143865e+00,   6.70639048e-01],
                 [  1.20458515e+00,   7.11621189e-01,  -1.02627755e+00],
                 [  6.07106328e-01,  -4.96347356e+00,   6.69715110e+00],
                 [  6.64330235e+00,  -7.30433044e+00,  -1.96170108e-01],
                 [ -1.86014183e+00,  -7.84701646e-01,  -8.03108186e-01],
                 [ -5.12424946e+00,   2.52694139e+00,   2.78482507e+00],
                 [  1.95669842e+00,   1.87828867e+00,  -1.96674961e-01],
                 [  2.27723260e-01,   1.94725411e+00,   8.66738456e-01],
                 [  3.51466356e+00,  -8.85634302e-01,   3.13290788e-01],
                 [  1.91209317e+00,  -4.68440177e+00,  -3.25027026e-01],
                 [  2.63459143e+00,   6.91787214e+00,  -1.63896386e+00],
                 [ -5.90774281e+00,  -8.29046347e+00,   1.30866861e-01],
                 [  8.10031949e+00,  -2.47837218e+00,   2.33679488e+00],
                 [  1.12591192e+01,  -5.09384340e-01,   2.11430350e-01],
                 [ -2.69699772e+00,   8.37056166e+00,  -4.24864419e+00],
                 [ -4.56335369e+00,  -5.86348012e+00,  -2.51095271e+00],
                 [ -2.32248979e+00,   6.05300141e-01,   6.75241328e+00],
                 [  1.44295114e-01,   1.94554878e+00,  -1.47717467e+00],
                 [ -3.99326994e+00,   1.64625013e-01,  -1.03126567e+01],
                 [ -1.63005209e+00,   6.91704725e-01,   8.42380649e-01],
                 [ -4.51842339e+00,  -7.68587871e+00,   6.75716433e+00],
                 [ -1.01956001e+01,  -2.58326417e+00,   2.83924230e+00],
                 [ -2.90047597e+00,  -2.70573766e-01,  -1.69103520e-01],
                 [ -9.55073767e-01,  -5.90490684e+00,  -2.58351003e+00],
                 [ -3.80812302e-01,   1.06440644e+00,   8.50336325e-01],
                 [  7.02676508e-01,  -1.61947526e+00,   1.49520421e+00],
                 [ -8.59300650e-02,  -5.55677600e-01,   4.17835984e-01],
                 [ -5.08561703e+00,  -3.24457775e+00,   5.79264333e+00],
                 [  4.06414846e+00,  -3.75630220e+00,   1.29339689e+00],
                 [ -1.38861383e+00,  -2.64351371e-01,   7.83683454e-01],
                 [  3.92661475e+00,  -6.70802309e-01,   8.15277139e+00],
                 [  3.83832326e+00,   4.67317384e+00,   2.09521390e+00],
                 [  5.16108739e+00,   9.25288467e+00,   1.23196359e-01],
                 [  3.37607818e+00,  -7.17838915e+00,   6.58809498e-01],
                 [ -5.72018339e+00,  -3.63355100e+00,  -4.92801492e+00],
                 [  3.06431330e-01,   3.60566208e-01,  -1.00003511e-01],
                 [  9.99849501e+00,   8.86885131e-01,   9.11101368e+00],
                 [  2.99733781e-01,   6.02114938e-01,  -1.75678422e+00],
                 [ -4.22714966e+00,   1.05667361e+01,   1.60292528e+00],
                 [ -2.86005052e+00,  -4.87884479e+00,  -3.70202245e+00],
                 [ -1.54240095e+00,  -4.04776607e-01,  -1.35895594e+00],
                 [  3.64961763e+00,   2.60541633e+00,   2.83706537e+00],
                 [ -1.72046185e+00,  -3.40796021e+00,  -6.09100788e-01],
                 [ -5.61335582e-03,  -4.59239385e-01,  -6.80622494e-01],
                 [  1.94339762e+00,  -8.18649378e-01,   5.18419530e-01],
                 [  1.41074929e+00,  -6.31563531e-02,  -9.06308082e+00],
                 [ -2.12922622e+00,  -3.04586771e+00,   2.14404194e+00],
                 [ -3.43945168e+00,   8.79820991e+00,   1.54577383e+00],
                 [ -2.63090631e+00,   4.17619029e+00,   4.79373867e-01],
                 [ -1.46366521e+00,   6.40590275e+00,   6.84369308e+00],
                 [  5.07217242e+00,   2.37222610e+00,   4.21970519e+00],
                 [  6.13176771e+00,   2.43080945e+00,  -4.66299479e-01],
                 [  8.93252740e+00,  -4.44218524e-01,   6.67962978e-01],
                 [  1.05071869e+00,   9.81115293e-01,  -1.37833020e+00],
                 [ -3.47834233e+00,  -9.62527899e-01,  -1.61193435e+00],
                 [ -3.08375281e-02,   3.18235473e+00,  -2.80383563e+00],
                 [  1.11853739e+00,  -1.44569618e+00,  -6.99928276e+00],
                 [ -5.55989799e+00,  -1.15662492e+00,  -2.07405483e+00],
                 [  4.10396305e-01,   1.21999103e+00,   9.33281631e-01],
                 [  2.36368851e+00,  -5.34499806e+00,   6.34855339e+00],
                 [ -1.11531296e-01,  -1.38928684e+00,   1.60222037e+00],
                 [ -1.22253043e+00,   1.46353041e+00,   8.69852242e-02],
                 [  7.07054180e-03,  -4.00866186e-01,   2.66531759e+00],
                 [ -4.62688523e-01,  -2.31143726e-01,   8.38206145e+00],
                 [ -4.27294114e+00,  -5.06056514e+00,  -5.05296672e+00],
                 [ -5.79382767e+00,   3.49888901e-01,  -5.94435165e-01],
                 [ -2.68991518e+00,  -4.36223221e-01,  -1.10692539e+00],
                 [ -1.48880494e+00,   4.84762530e+00,  -8.43662028e+00],
                 [  3.37545218e+00,   2.90359380e+00,  -2.98361859e-02],
                 [ -2.98302063e+00,   7.85475599e+00,   1.91459299e+00],
                 [ -4.76495480e+00,   3.52718683e-01,   1.03703372e+00],
                 [ -3.52244851e-03,  -1.71280684e+00,  -4.84327539e-01],
                 [  1.54699035e+00,   3.82823477e+00,  -5.06917624e+00],
                 [ -3.25958240e+00,  -4.78077315e-01,   9.81096861e-01],
                 [ -4.59210196e+00,  -4.15709524e+00,  -1.04741273e+01],
                 [ -3.62406198e+00,  -7.62617832e+00,   4.28166145e+00],
                 [ -4.10774619e-02,   1.05619846e+00,  -3.20838206e-01],
                 [  1.03457146e+00,   1.03858143e+00,  -2.84151798e+00],
                 [  1.21890811e+00,   1.46058935e-01,  -2.75530407e-01],
                 [  5.26784588e-01,   2.73650301e-01,  -1.27082784e+00],
                 [  7.76440883e-01,  -5.58396908e-01,   5.83337078e-01],
                 [  2.33896917e+00,  -2.54640304e+00,  -6.43260234e+00],
                 [ -3.27226194e+00,  -3.60142126e+00,  -8.10125589e+00],
                 [  1.32140022e+00,  -3.91692793e+00,  -5.04756610e+00],
                 [ -4.15777411e+00,  -2.17605305e+00,   1.58132789e+00],
                 [ -5.43126833e+00,   1.58648235e+00,  -8.37937648e-01],
                 [ -2.27818705e+00,   7.04902208e+00,  -3.71762938e+00],
                 [ -2.65479249e+00,   1.76107586e-01,   4.41664363e-01],
                 [ -1.27188776e+00,  -4.06896770e+00,  -4.95489485e-01],
                 [ -3.90915130e-01,   1.16427174e+00,  -1.45413161e-01],
                 [  8.89932475e+00,  -1.68470954e+00,   4.77144761e-01],
                 [  3.01735763e-01,  -2.51071728e-02,   7.42077934e-01],
                 [  8.08007893e+00,   1.72600780e+00,  -3.99797546e+00],
                 [  9.65428014e+00,   8.64898898e-01,   9.19034681e-01],
                 [ -2.04086414e+00,  -1.53240882e+00,   1.02547582e-01],
                 [ -2.32978469e+00,   6.60613647e+00,   5.33477036e+00],
                 [  9.49126467e-01,   2.16413326e-01,  -1.14903864e+00],
                 [ -5.47930052e-01,   7.86459573e-01,   5.41095834e+00],
                 [  1.82085529e+00,   1.03273126e+00,   1.13223853e+00],
                 [  1.27018280e+00,  -2.09735457e-01,   8.68018773e+00],
                 [  1.61570154e+00,   6.57046375e+00,   2.77707674e+00],
                 [ -4.46113598e-01,  -7.24688652e+00,   2.39173376e+00],
                 [ -1.27695878e+00,   1.64481473e+00,  -5.58301354e+00],
                 [  3.09512378e+00,  -4.32387569e+00,  -1.04536828e+00],
                 [ -9.87185897e+00,   4.55920105e+00,  -2.83137162e+00],
                 [ -5.15619340e+00,  -3.74901926e+00,   8.43025699e-02],
                 [ -8.33905626e-01,   3.38243656e+00,  -1.02024328e+01],
                 [ -5.16158005e-02,   3.29001091e+00,   1.93893038e+00],
                 [ -7.32331290e+00,   9.26677341e-01,  -1.86864934e+00],
                 [  1.00213506e+00,  -9.64899860e-01,  -4.01108090e-02],
                 [  3.92399501e+00,   6.11265132e-01,   4.47494658e+00],
                 [ -5.25224418e+00,  -5.30137475e+00,  -1.89890079e+00],
                 [ -5.53754441e-01,   1.37541182e+00,  -3.30746860e+00],
                 [ -7.82836632e+00,  -2.04557572e+00,  -3.13708739e+00],
                 [ -3.55238754e-01,   2.51672716e-02,  -1.59353326e+00],
                 [  2.89933474e+00,  -1.40077238e+00,  -2.01724925e+00],
                 [  1.24961895e+00,   3.21363896e-01,   1.67278798e-01],
                 [  1.38824901e+00,  -7.63260171e+00,   4.52171698e+00],
                 [ -3.70092791e+00,  -1.85250926e-01,  -5.67472247e+00],
                 [  5.48603046e+00,   2.34246632e-01,  -2.82304105e+00],
                 [  2.90276935e+00,   4.00255298e+00,  -2.55047322e+00],
                 [  7.34645654e-01,  -7.26917722e+00,  -5.75139507e+00],
                 [  6.41786780e+00,  -2.25290782e+00,   7.56799600e+00],
                 [  2.27221265e+00,  -6.28983557e+00,   2.85522749e+00],
                 [ -1.57472167e+00,  -3.82748582e-01,  -3.30949252e+00],
                 [  5.24453956e-01,   1.48960098e-01,  -4.99710756e-01],
                 [  8.00560818e-01,  -6.07899205e+00,   7.36475347e+00],
                 [  4.23641562e-01,  -4.47116169e-01,   3.67688850e-01],
                 [ -3.45810149e-01,   6.66673643e+00,  -4.92865160e+00],
                 [ -2.43964709e+00,   6.06925039e-01,  -5.16027155e-01],
                 [ -1.12750237e+00,   4.35926448e-01,  -1.48896859e+00],
                 [ -5.74701822e+00,   5.47425737e+00,  -2.20313712e+00],
                 [ -1.89798411e+00,   2.45199578e-01,   1.03445880e+00],
                 [  2.93269944e-01,   1.35390330e+00,  -2.90589521e-02],
                 [  1.41952400e+00,   1.24656948e+00,  -4.71798034e-01],
                 [ -3.25168550e+00,  -9.32747926e+00,   3.78347845e+00],
                 [  9.58456252e+00,  -1.59178515e+00,  -7.74632600e-01],
                 [ -5.90161314e-01,   3.29859080e+00,   8.93075944e-01],
                 [ -1.96213911e+00,  -2.51706976e+00,   5.47303726e+00],
                 [ -5.26279676e+00,  -3.42108229e+00,   9.45313068e+00],
                 [ -9.17716041e+00,   1.63799341e+00,  -4.73423172e+00],
                 [ -9.93845929e+00,   3.26624655e+00,   2.40603051e+00],
                 [ -3.67282587e+00,  -5.00262905e+00,  -5.20303996e+00],
                 [  7.28675636e-01,   3.54216921e-02,   4.52830648e-01],
                 [ -3.15743238e-01,  -2.10280019e-01,  -2.74872468e+00],
                 [ -9.23655245e-01,   6.04828664e-01,   9.97483364e-01],
                 [  6.31903281e+00,  -2.33815986e+00,  -4.01665534e+00],
                 [  3.97996328e+00,   7.32416373e+00,  -7.47581814e+00],
                 [ -2.17467036e-01,  -9.02368415e-01,  -2.91144630e-01],
                 [ -3.64615784e-01,  -1.37406194e-01,   3.71645056e-01],
                 [ -1.36002166e-01,  -1.64173833e+00,  -4.20154516e+00],
                 [ -1.74843944e+00,  -2.92221849e+00,  -1.31013332e+00],
                 [  1.36144845e+00,   3.28820070e+00,  -6.30887039e-01],
                 [ -2.06424173e+00,   2.63795551e-01,   6.07939541e+00],
                 [  2.22664496e+00,   7.69208963e+00,   4.38416150e+00],
                 [  1.26044063e-01,  -8.58163574e-01,  -2.07042082e+00],
                 [ -6.13408252e-01,   2.38910516e+00,   3.68009515e+00],
                 [ -7.84674830e+00,  -4.54276215e-01,   7.23989607e-01],
                 [  5.66127507e+00,  -5.38742337e+00,  -4.15917708e-03],
                 [  2.12211135e-01,   9.02083019e-01,  -7.21800592e+00],
                 [ -7.81654901e+00,   4.63798401e+00,  -1.70315972e+00],
                 [ -2.90859827e+00,  -2.28004361e+00,  -6.97119768e-01],
                 [ -6.00067633e-03,  -3.75626026e+00,  -7.39419862e+00],
                 [  2.79720965e-01,  -7.59775098e-01,  -9.55510894e-01],
                 [  8.23441035e+00,   8.38899499e-01,   4.38269300e+00],
                 [  4.57681190e+00,   7.50294671e+00,  -2.48376128e+00],
                 [ -2.54482052e+00,   2.72732796e-02,   1.59723101e+00],
                 [ -5.72677831e+00,   7.40293226e+00,  -5.61259375e+00],
                 [ -1.31862116e+00,   1.14591513e+00,   3.64129459e+00],
                 [ -1.59402393e+00,   8.39171241e-01,   1.06668508e-01],
                 [ -3.58168604e+00,  -6.11912277e-01,   1.95575290e+00],
                 [ -4.50363167e-01,  -7.94504260e-01,   1.32048468e+00],
                 [ -1.08479013e+00,  -8.52937013e+00,   1.18412635e+00],
                 [  2.71741570e+00,   6.01460297e+00,  -1.10022056e+01],
                 [  2.73267494e+00,   3.20251617e+00,   7.16033521e+00],
                 [ -8.30607588e+00,   3.06829929e-01,   3.12386377e+00],
                 [ -8.40612077e-01,   4.83649001e+00,  -3.85221683e-01],
                 [  6.36836620e+00,  -3.32740603e-01,  -5.41784744e+00],
                 [ -3.01839301e+00,  -5.76136388e-01,  -3.63518901e+00],
                 [  3.88007535e-01,  -2.73843355e-01,  -1.10785922e+00],
                 [ -5.47785367e+00,   6.50046171e-01,   5.72140322e+00],
                 [  7.06435250e-01,  -2.18705417e-01,  -5.98781454e-01],
                 [ -2.12838929e+00,   1.07430800e+01,  -4.26585786e+00],
                 [  6.34541845e+00,   2.78114656e+00,   5.14045199e+00],
                 [ -4.97436053e+00,  -1.16677416e+00,  -2.39240246e+00],
                 [  3.92960215e+00,   1.68438461e+00,  -1.82970192e+00],
                 [  2.95400222e-01,  -3.40791015e-01,  -1.91016171e+00],
                 [  6.04656626e-01,   3.43991939e+00,   1.38759000e+00],
                 [ -1.04049084e+00,  -2.64170426e+00,   1.72021382e+00],
                 [ -5.25586933e+00,  -1.77034760e+00,   6.82754912e+00],
                 [  1.50252442e+00,   6.78618097e+00,   3.68457288e+00],
                 [ -2.08431849e+00,   2.82181418e+00,  -1.17686890e+00],
                 [ -3.19687217e+00,   9.28023303e+00,  -3.39170296e+00],
                 [ -2.80491734e+00,   8.01333285e+00,   8.83527622e-01],
                 [ -3.70090276e-01,   3.68855886e+00,   7.50397100e+00],
                 [ -3.32956685e+00,   3.11488467e+00,   8.98427128e+00],
                 [  8.30070212e+00,  -4.12934296e-01,  -6.85566034e-01],
                 [ -2.25649607e-01,  -8.32133117e-01,   1.12701086e+00],
                 [ -3.76360938e+00,   1.13393341e+00,  -4.30776100e+00],
                 [ -1.82076376e+00,   4.06000550e-01,   5.95648039e-01],
                 [  2.71205612e-01,  -2.66983059e+00,   5.23120469e+00],
                 [  6.72583375e+00,   8.04158291e+00,   1.11648149e+00],
                 [  4.24981223e-01,  -1.61484726e+00,  -4.97982903e-01],
                 [ -8.77083157e-02,   6.88161166e+00,  -3.31611666e+00]]).T, FortranArray([[ 6942.11286233,    -7.73764131,   -37.89817885],
                 [   -7.73764131,  7070.1293673 ,    21.52299297],
                 [  -37.89817885,    21.52299297,  6993.8316329 ]]), FortranArray([ 1.49655178,  1.95169154,  1.67704201,  2.38996075,  2.53179385,
                  2.59693502,  2.37437019,  2.38895577,  2.10422669,  3.01612062,
                  2.37358524,  1.46872105,  2.71678557,  1.48329825,  2.68299576,
                  2.23583865,  1.95890033,  2.79236814,  1.87185514,  1.52875268,
                  1.88135998,  2.44502723,  2.24278856,  2.53680088,  2.61125155,
                  2.81482859,  2.39364398,  2.51268511,  2.55867587,  1.90161403,
                  2.17707954,  1.96581063,  2.63765236,  2.18117012,  1.92910023,
                  2.18609488,  1.97104439,  1.74569546,  1.5857536 ,  2.11026499,
                  2.21503073,  2.46585758,  2.53469267,  2.44180139,  2.30833273,
                  2.4924067 ,  2.50400249,  2.08083121,  2.51138504,  1.65593815,
                  2.47668503,  2.58240672,  1.46665976,  2.100809  ,  1.44280208,
                  1.83886736,  1.38798981,  2.1517283 ,  2.19387943,  2.17605435,
                  2.92549763,  2.49237424,  2.58848861,  2.36010829,  2.23637234,
                  1.75481536,  2.38212791,  1.90336102,  2.36226   ,  2.29627891,
                  1.27916621,  2.31826221,  1.88131926,  1.5974489 ,  1.77924559,
                  2.67405546,  2.16940501,  2.35030279,  2.28460217,  2.41940209,
                  2.48538431,  2.38383588,  2.58859933,  1.27452693,  2.59987878,
                  1.636901  ,  2.4428593 ,  2.44286129,  1.67060836,  2.62021963,
                  1.50498654,  1.50196317,  1.40609642,  2.60059694,  2.72589899,
                  2.19307022,  2.32534363,  2.31465348,  2.41917192,  2.36256706,
                  2.26323983,  1.73753347,  2.27129854,  1.52077418,  2.57619832,
                  1.94602708,  1.66856594,  2.72215276,  1.41753393,  1.41583383,
                  1.86405983,  2.73969988,  2.57685968,  2.21502278,  2.70076838,
                  2.473813  ,  2.2493476 ,  2.26058045,  2.24742367,  1.76239981,
                  2.95989496,  1.55485774,  2.72287936,  2.72974564,  1.70189603,
                  2.65863645,  1.68753344,  1.3987595 ,  1.3010234 ,  2.18288808,
                  2.61256633,  2.93045839,  2.75997592,  2.62000018,  2.45277798,
                  2.51236968,  2.693802  ,  0.77885066,  2.88237863,  1.57325968,
                  2.38161124,  2.69618646,  1.49111428,  2.58212319,  1.92472452,
                  1.7257953 ,  1.74412006,  2.4796022 ,  2.78565494,  2.69803081,
                  2.873778  ,  2.44480401,  2.64480132,  2.67629556,  2.61841262,
                  1.38802671,  2.49194469,  1.85399674,  2.37985579,  2.90495243,
                  1.40454943,  2.56432711,  1.83500198,  1.83465191,  1.71053566,
                  2.86101095,  2.32388891,  2.52436676,  2.58610369,  2.92117378,
                  2.78655541,  2.56033458,  2.59683475,  1.77705374,  2.09607989,
                  1.70177956,  2.43588559,  2.42826174,  1.79371277,  2.45513386,
                  1.92565999,  1.73856693,  1.72952095,  2.49759162,  2.46485114,
                  2.56391863,  2.75241668,  2.90719452,  2.3974315 ,  2.30695362,
                  2.17131547,  1.51181569,  2.24981671,  1.13989943,  2.45742732,
                  2.69055837,  1.57230025,  2.02116115,  1.6911682 ,  1.60974373,
                  0.87131137,  2.59386724,  2.55697991,  2.91172937,  2.64129575,
                  2.68141227,  2.26623503,  2.3955129 ,  2.38191133,  1.92419129,
                  2.60666943,  1.48625636,  2.69785187,  2.69779352,  1.49344021,
                  2.62083084,  1.07026143,  1.56126566,  1.6892355 ,  2.88011783,
                  2.35820888,  2.61062345,  2.797635  ,  2.73500982,  2.31022187,
                  2.55745396,  2.57687687,  1.35636238,  2.06722941,  1.67060689,
                  2.71813064,  2.45243621,  1.85884202,  2.57107347,  1.61146132,
                  2.0229015 ,  1.65999304,  2.34130219,  2.78074437,  2.52782821,
                  2.46299111,  2.44907604,  2.74862898,  2.00750902,  2.74516425,
                  1.96115101,  2.4347091 ,  1.48015847,  2.33604379,  2.41813549,
                  2.0817423 ,  2.34377871,  1.56498762,  1.79066918,  1.6953569 ,
                  2.49486484,  2.30658049,  2.11040649,  2.76877972,  2.86412243,
                  2.49438896,  2.4792435 ,  2.44552749,  1.873819  ,  2.47536911,
                  1.71626509,  2.28210688,  2.65438714,  1.728832  ,  2.82399038,
                  1.41890786,  2.10732459,  1.61218835,  2.09238646,  2.63355852,
                  2.3880453 ,  2.01986304,  2.75932091,  2.70382677,  2.6559722 ,
                  2.39952437,  1.54052398,  2.47094067,  1.47734804,  2.44192733,
                  2.34522366,  1.72500024,  2.52368346,  1.91884432,  2.15844168,
                  1.74447484,  2.56563901,  2.43331721,  2.2162539 ,  2.46767006,
                  3.08138637,  2.34397538,  2.40725222,  1.99400425,  0.8928534 ,
                  2.56063037,  1.65261311,  2.38147434,  2.65451327,  1.95083952,
                  2.68156392,  1.8695588 ,  1.86074867,  1.72451256,  2.46671387,
                  2.5569003 ,  2.5590344 ,  2.52678189,  2.52131249,  2.80717577,
                  2.34226671,  2.41033787,  1.65669313,  2.67141187,  1.81619393,
                  2.31901549,  2.18413444,  1.86108894,  2.42273547,  0.69279069,
                  1.58033739,  1.51157295,  3.04717723,  2.57087383,  3.21462531,
                  2.36375334,  2.34204311,  2.77056257,  2.6784625 ,  2.18091185,
                  1.50861583,  2.57859853,  1.81566987,  2.38742868,  2.08605031,
                  1.78554018,  2.2820605 ,  1.67499251,  1.51775874,  1.91398689,
                  2.37969757,  2.3864381 ,  2.44429957,  2.50327127,  2.42797149,
                  2.87758859,  2.05794426,  2.53816761,  1.95034581,  2.398965  ,
                  1.73453104,  2.16154975,  2.55321271,  1.81298166,  3.0463891 ,
                  1.71039681,  1.55880061,  1.76713721,  2.40921778,  2.58260176,
                  2.50166009,  2.71327262,  2.46311377,  2.35738376,  2.12249721,
                  2.66797934,  1.44164309,  2.49518871,  2.06487727,  2.36604389,
                  2.69054941,  1.05484983,  2.66103748,  1.80651341,  1.66191149,
                  1.4647266 ,  2.49855175,  2.58900472,  2.75073202,  2.22026942,
                  2.73383721,  2.34126365,  2.0448338 ,  2.67673786,  1.78552304,
                  2.54438556,  1.62780339,  2.71229766,  2.74815499,  2.01334256,
                  2.73164831,  1.98849255,  1.83935274,  1.39186478,  2.25801892,
                  2.42629498,  2.35842326,  2.72140443,  2.3233311 ,  2.51246796,
                  2.36614599,  2.40687769,  1.75668831,  2.64169699,  1.68293761,
                  2.48936395,  2.26041107,  1.62636948,  2.56803497,  1.30675188,
                  1.69454284,  1.61691613,  2.72648163,  2.59618548,  2.76191093,
                  2.70814414,  2.87274602,  2.47338497,  2.77702066,  2.2823238 ,
                  1.5345193 ,  2.8690927 ,  1.03432267,  2.49742634,  2.56229517,
                  2.05124189,  2.37026108,  1.83484382,  1.13126853,  1.3698083 ,
                  2.77214446,  3.00908939,  2.62691318,  2.27147036,  2.50956299,
                  2.81885393,  2.36528027,  2.6024879 ,  1.94073665,  2.70308922,
                  1.45190679,  2.88148212,  2.69194598,  1.55469194,  2.28653808,
                  1.60155516,  1.24131107,  1.58294577,  2.27625967,  2.54817127,
                  2.09571273,  2.83814038,  2.70947702,  2.29250876,  2.35633723,
                  2.82172692,  1.9362703 ,  2.94899607,  1.91032617,  2.34291626,
                  2.33589188,  1.62101507,  2.49978107,  1.62828843,  1.8309712 ,
                  1.81411572,  2.07494088,  2.63930685,  2.68790609,  2.26887587,
                  2.98931612,  2.51930303,  2.54321398,  3.00995483,  1.70528353,
                  2.48404565,  1.81875608,  2.57068678,  1.90625798,  1.77191633,
                  2.37312319,  1.47745872,  1.76108325,  1.5615446 ,  1.76758687,
                  2.58083318,  3.08323973,  2.35650048,  2.34712756,  2.30793237,
                  2.228699  ,  2.47332735,  1.86385394,  2.39912826,  0.5424308 ,
                  2.55470068,  2.19796167,  1.88665452,  2.70769633,  1.84538985,
                  1.4469089 ,  1.36771693,  1.99993052,  2.58408667,  2.27237845,
                  2.1573973 ,  2.56316096,  2.51532445,  2.15515763,  2.3006435 ,
                  1.66198559,  2.75629618,  1.72123048,  2.66869003,  2.51521982,
                  1.42867185,  2.18468662,  1.61157864,  1.21656663,  1.74030694,
                  2.4202554 ,  2.15480331,  2.71044843,  2.53482081,  2.42677562,
                  2.61527769,  2.56529492,  2.64920485,  1.48557523,  2.25821069,
                  1.72053351,  2.37544613,  2.82180788,  1.25249614,  2.57982901,
                  1.8227514 ,  1.45003847,  1.40474188,  2.40324191,  2.52120701,
                  2.55309614,  2.72551041,  2.04504228,  2.383149  ,  2.5850519 ,
                  2.69883183,  1.55768965,  2.37485269,  1.92251776,  2.17486366,
                  2.38069513,  1.64341651,  2.2736706 ,  1.46474243,  1.51330467,
                  1.91706151,  1.86832926,  2.26535234,  2.0984096 ,  2.3592383 ,
                  2.41029403,  2.4790709 ,  2.45185983,  2.43298713,  1.93467184,
                  2.22955421,  1.87946804,  1.93101844,  2.53983234,  1.55458655,
                  2.29468813,  1.29999817,  1.62674173,  1.66635372,  3.03988928,
                  2.1825214 ,  2.57850364,  2.41061776,  2.3481621 ,  2.38835828,
                  2.28153973,  2.68416596,  1.59248208,  2.15602608,  1.46859138,
                  2.67784221,  2.2929428 ,  1.70848371,  2.7165277 ,  1.79454614,
                  1.62515887,  1.63463517,  2.18817658,  2.44557139,  2.07187981,
                  2.58093798,  2.83069697,  3.02122658,  2.39945686,  2.62947794,
                  1.59762513,  2.80455921,  1.93448473,  2.36155422,  2.7520599 ,
                  1.69878542,  2.85325828,  0.99000645,  1.52703834,  1.07691935,
                  2.33206477,  3.27749053,  2.7680032 ,  2.90680244,  2.6195284 ,
                  2.83738423,  2.59520405,  2.37364483,  1.38986911,  2.26091578,
                  1.3199265 ,  2.33012858,  2.52703008,  1.73523564,  2.39984654,
                  1.55498308,  1.82428859,  1.40691854,  2.60862433,  2.50497712,
                  2.55513017,  2.39952634,  2.25935936,  2.29573797,  2.38489291,
                  3.04644676,  1.55324807,  2.36267482,  0.90137757,  2.51467801,
                  2.72668843,  1.60622035,  2.46233261,  1.54408731,  2.01308328,
                  1.67944278,  2.76776642,  2.48946778,  2.21055483,  2.36423254,
                  2.64909932,  2.57363308,  2.40395342,  2.3624974 ,  1.93117264,
                  3.00196832,  1.8310605 ,  2.56407873,  1.99485006,  1.80438033,
                  2.41344036,  1.66329785,  1.55167685,  1.04991053,  2.46914676,
                  2.34965581,  2.43620526,  2.47647065,  2.3291748 ,  2.39067221,
                  2.58097323,  2.64631051,  1.03626581,  2.29319135,  1.41596175,
                  2.54208254,  2.19001083,  1.31268546,  2.80061288,  1.61992449,
                  1.37677854,  1.96081417,  2.49766256,  2.21164711,  2.55602644,
                  2.35048657,  2.77051998,  2.49179992,  2.09533192,  2.57261284,
                  1.94849694,  1.88866891,  1.49101434,  2.50239521,  2.71812503,
                  1.48039853,  2.3040262 ,  1.67893413,  1.65257846,  1.67658012,
                  2.60495829,  2.51671175,  2.5718832 ,  2.33402181,  2.38636025,
                  2.42686818,  2.30442103,  2.25976179,  1.70187858,  2.32056992,
                  1.55686096,  2.41638992,  2.58400973,  1.90138418,  2.45152123,
                  1.71660239,  1.92540042,  1.39290223,  2.58967655,  2.52956248,
                  2.41901864,  2.45501768,  2.52155803,  2.57820003,  2.4821146 ,
                  2.51500825,  1.76020851,  2.41883637,  1.64594117,  1.96544472,
                  2.23676488,  1.98874246,  2.6214626 ,  1.77444211,  1.39194311,
                  1.94620321,  2.387613  ,  2.50079669,  2.55397713,  2.663755  ,
                  2.47090201,  2.53558298,  2.69263439,  2.27020205,  1.49913212,
                  2.20898814,  1.57019926,  2.43055358,  2.38454442,  1.58521004,
                  2.55201807,  1.66614524,  1.34634131,  1.60674801,  2.60273189,
                  2.68502402,  2.52020414,  2.590856  ,  2.1862059 ,  2.80479398,
                  2.27739425,  2.25209492,  1.67129759,  2.63652928,  1.58634165,
                  2.24153959,  2.69952317,  1.07805444,  2.64495723,  1.45472434,
                  1.56496249,  1.93693767,  2.64149727,  2.77650597,  2.40789977,
                  2.21035762,  2.28379184,  2.51492035,  2.6157964 ,  2.58075013,
                  1.60506461,  2.50802727,  1.89555528,  2.98088016,  2.41880948,
                  1.57954746,  2.21619283,  1.84292464,  1.96854832,  1.52557322,
                  2.28515462,  2.90451286,  2.03909926,  2.90052605,  2.29018225,
                  2.51849737,  2.63210649,  2.6695093 ,  1.3497822 ,  2.72427057,
                  1.67632063,  2.10473263,  2.43836306,  1.75688728,  2.80754955,
                  1.24584768,  1.75382614,  1.54865756,  2.33097035,  2.65282314,
                  2.13196787,  2.66179151,  2.91856546,  2.31330352,  2.63567447,
                  2.1943056 ,  1.87455273,  2.35335371,  1.69951548,  2.1875354 ,
                  2.3375476 ,  1.48939246,  2.79453343,  1.46966354,  1.5945907 ,
                  1.86702216,  2.67961304,  2.49449559,  2.44163577,  2.59455462,
                  1.87030499,  3.09970019,  2.42096136,  2.19749902,  1.71165432,
                  2.51478669,  1.50450348,  2.70451473,  2.58253798,  1.79146512,
                  2.45739439,  1.85565042,  1.48407688,  1.40429743,  2.36406787,
                  2.47573061,  2.42687537,  2.43474156,  2.71729432,  2.14362533,
                  2.48360345,  2.91490312,  1.7072477 ,  2.53915469,  1.69726723,
                  2.59278551,  2.39599003,  1.64592508,  2.64549863]), FortranArray([[ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.]]).T)


         def setUp(self):
            self.xml = """<TS_params 
        betapol="0.75" 
        cutoff="20.0 18.0 18.0 18.0"
        cutoff_ms="18.0"
        cutoff_coulomb="20.0"
        tolpol="0.0005" 
        yuksmoothlength="20.0" 
        iesr="2 2 2" 
        a_ew="1e-06" 
        n_types="2" 
        gcut="0.0" 
        pred_order="2" 
        maxipol="60" 
        raggio="0.0" 
        tewald="F" 
        yukalpha="1e-06">

        <per_type_data atomic_num="8" pol="0.0" z="0.0" type="1" />
        <per_type_data atomic_num="14" pol="0.0" z="0.0" type="2" />

        <per_pair_data atnum_i="8"  atnum_j="8"  C_pol="0.46009932"  D_ms="0.00018650185" gamma_ms="11.642637" B_pol="0.87357114" R_ms="8.0465068" />
        <per_pair_data atnum_i="8"  atnum_j="14" C_pol="-1.5091142"  D_ms="0.0053600978"  gamma_ms="10.405794" B_pol="1.977039"   R_ms="4.193651"  />
        <per_pair_data atnum_i="14" atnum_j="14" C_pol="0.0"         D_ms="-0.0021645401" gamma_ms="4.5784138" B_pol="0.0"        R_ms="13.113727" />
      </TS_params>
      """
            if do_compare_p1_p2:
               self.p1 = Potential('IP ASAP', param_str=self.xml)
            self.p2 = Potential('IP TS', param_str=self.xml)
            self.cutoff = 20.0*BOHR
            self.debug = False



      class TestCharge(QuippyTestCase, PotTestMixin):
         """ Turn off short-range and dipole terms, to test charge part of potential only"""

         dimer_ref = (FortranArray(-24.070897337877238), FortranArray([[ 21.07413139,   0.        ,   0.        ],
                 [-21.07413139,   0.        ,   0.        ]]).T, FortranArray([[-33.92423456,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ]]).T, FortranArray([-12.03544867, -12.03544867]), FortranArray([[ 0.,  0.,  0.],
                 [ 0.,  0.,  0.]]).T)

         trimer_ref = (FortranArray(-44.403166958656648), FortranArray([[ 18.63466235,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [-18.63466235,   0.        ,   0.        ]]).T, FortranArray([[-59.99456346,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ]]), FortranArray([-10.16613481, -24.07089734, -10.16613481]), FortranArray([[ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.]]).T)

         quartz_ref = (FortranArray(-181.6944694918098), FortranArray([[ -5.40952808e-01,   9.36957749e-01,   7.09756958e-15],
                 [ -5.40952808e-01,  -9.36957749e-01,   8.09518074e-15],
                 [  1.08190562e+00,   6.52367311e-15,  -1.95299023e-14],
                 [  4.07411523e+00,  -6.85592161e+00,   4.46551787e+00],
                 [  3.90034466e+00,   6.95624809e+00,   4.46551787e+00],
                 [ -7.97445989e+00,  -1.00326485e-01,   4.46551787e+00],
                 [  4.07411523e+00,   6.85592161e+00,  -4.46551787e+00],
                 [ -7.97445989e+00,   1.00326485e-01,  -4.46551787e+00],
                 [  3.90034466e+00,  -6.95624809e+00,  -4.46551787e+00]]).T, FortranArray([[ -7.64106751e+01,   5.42848848e-16,  -9.31988864e-15],
                 [  5.42848848e-16,  -7.64106751e+01,   2.78505061e-14],
                 [ -9.31988864e-15,   2.78505061e-14,  -7.58877866e+01]]).T, FortranArray([-36.12479391, -36.12479391, -36.12479391, -12.22001463,
                 -12.22001463, -12.22001463, -12.22001463, -12.22001463, -12.22001463]), FortranArray([[ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.]]).T)

         bigquartz_ref = (FortranArray(-367.16126747929781), FortranArray([[-2.35632642, -2.0209376 ,  3.87904176],
                 [ 2.84952011, -0.45094289, -0.53113981],
                 [ 2.420267  , -1.71737752,  4.63039802],
                 [ 6.93931868, -4.53731247,  4.68397283],
                 [-1.1283966 ,  7.02435327,  9.35725181],
                 [-5.29444952, -0.69398915,  1.94212592],
                 [ 4.23143492,  9.56380949,  0.31116211],
                 [-8.9420606 , -0.94686359, -8.73981904],
                 [ 0.02200083, -8.73150269, -6.55068154],
                 [ 3.67019499,  1.31856074, -1.61204147],
                 [-2.22233694, -3.71938351,  3.10773226],
                 [ 0.72156838, -4.56370697,  0.0819926 ],
                 [ 4.22696576, -4.80580269,  4.27477823],
                 [ 9.99852044,  5.66166145,  2.73743777],
                 [-8.56627422,  6.52755751, -0.38682621],
                 [ 0.88500014,  8.13213328, -5.01001697],
                 [-7.85653112,  2.24543026, -7.62948936],
                 [ 0.40158417, -8.28568694, -4.54587894]]).T, FortranArray([[-153.94991269,   -2.45634663,   -5.5806971 ],
                 [  -2.45634663, -152.08622231,   -4.68870731],
                 [  -5.5806971 ,   -4.68870731, -154.96812059]]).T, FortranArray([-33.85355213, -37.66036671, -35.88787751, -11.60745322,
                 -12.0585835 , -10.22794598, -12.31926618, -12.35573703,
                 -13.36160802, -38.11538713, -36.36248279, -36.88969989,
                 -12.32753866, -13.23357389, -11.7024293 , -12.1274268 ,
                 -14.4787731 , -12.59156565]), FortranArray([[  5.43472210e-323,   2.18550278e-315,   2.05297892e-315],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [  2.04652552e-315,   2.17363545e-315,   0.00000000e+000],
                 [  4.01707838e-057,   5.64697363e-038,   6.76931563e-043],
                 [  1.58687490e-047,   1.08756584e-071,   4.04676264e-086],
                 [  7.06264092e-077,   4.91414628e-062,   2.19926270e-313],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [  4.05003039e-116,   2.33761622e-310,   2.14844351e-315],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [ -9.91682126e-270,   2.05286699e-315,   2.14843521e-315],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [  4.94065646e-324,   3.88558520e-317,   1.53160350e-322],
                 [  1.45003458e-047,   3.82048470e-086,   1.58687490e-047],
                 [  1.05637317e-312,   7.06264092e-077,   4.91414628e-062],
                 [  2.33761623e-310,   0.00000000e+000,   2.04652694e-315],
                 [  1.45189716e-047,   3.45374415e-086,   9.05449679e-043],
                 [  4.91414628e-062,   2.19926270e-313,   2.16781531e-315]]).T)

         am_ref = (FortranArray(-17171.773070464682), FortranArray([[  2.24451530e+00,   1.31619737e+00,   2.52545725e+00],
                 [ -1.85086423e+00,   1.16395382e-01,  -2.68795571e-02],
                 [ -4.55704191e-02,  -1.41530452e+00,   6.36612100e+00],
                 [  6.94768003e+00,  -8.67571700e+00,   1.42155810e+01],
                 [ -8.42795260e+00,   3.12859483e+00,   6.84744065e+00],
                 [  4.64714673e+00,   1.09301440e+00,  -3.25892830e+00],
                 [  4.35115155e+00,   6.27511619e+00,  -8.74882645e+00],
                 [  1.11027158e+01,  -6.43702574e+00,   1.40886211e-01],
                 [  7.83641536e-01,   9.61855635e+00,  -7.02021754e+00],
                 [ -1.32947255e+00,  -1.26706446e+01,  -2.11173892e+00],
                 [ -4.62168151e+00,  -4.75169053e+00,  -1.22787305e+01],
                 [  1.31979764e+00,   3.46136922e+00,   1.46349473e+00],
                 [  6.87912754e+00,  -1.10947941e+01,  -4.39383309e+00],
                 [ -1.32871515e+00,   7.04753236e+00,   1.46996253e+00],
                 [ -2.46771013e+00,   2.47844234e+00,  -8.46616977e+00],
                 [  1.00669025e+01,  -2.29016230e+00,   1.54256722e+01],
                 [  1.46489271e+00,   6.87309776e-01,   3.58045148e-01],
                 [ -1.07704594e+01,  -2.05983161e+00,   3.17162485e+00],
                 [  2.90121762e+00,  -3.62281463e-01,  -1.27551301e+00],
                 [  6.80207330e-01,   1.63167193e+00,   8.47702619e-02],
                 [  1.38234877e-01,   3.48937982e-01,   1.81046574e+00],
                 [  5.49040424e+00,   8.27698457e+00,   2.01298924e+00],
                 [  1.76004734e+00,   5.03387348e+00,   5.45400533e+00],
                 [  3.64196989e-01,   3.97712277e+00,   2.95896677e+00],
                 [  4.30312443e+00,   8.88662422e+00,   1.80864074e+00],
                 [ -5.60875411e+00,  -5.61289875e+00,   4.26589539e+00],
                 [  5.85366108e+00,  -8.13197094e+00,   3.90930304e+00],
                 [ -2.63388006e+00,  -2.39905504e+00,  -1.85509616e+00],
                 [  9.92336678e+00,   5.08771276e-01,   3.30407673e+00],
                 [ -2.26053453e-01,  -1.81350508e+00,  -1.20216523e+00],
                 [ -9.71759967e+00,   6.96757103e-01,   1.42114264e-01],
                 [  9.09946583e-01,  -4.35453271e-01,   2.18908267e-01],
                 [ -1.20924276e+01,  -5.21305197e-01,   4.58878825e+00],
                 [ -9.97327734e+00,  -4.00621401e+00,   2.21358408e+00],
                 [  6.67983363e-01,   1.72974385e+00,  -2.24486916e+00],
                 [ -2.89372582e+00,   7.64666705e-01,  -1.43226465e+00],
                 [ -3.93503553e-02,   1.87116248e+00,   2.13491061e+00],
                 [ -1.04648816e+00,   3.78999191e+00,  -9.76563600e-01],
                 [ -1.40208671e+00,   6.37365697e-01,  -1.46188332e+00],
                 [ -2.58872945e+00,  -2.67269498e+00,   1.27337385e+01],
                 [  9.16279960e+00,   4.31128929e+00,   1.23382408e+01],
                 [  3.79401472e+00,   9.42478791e+00,   8.84753103e-01],
                 [ -3.07668818e+00,  -8.62489078e+00,   9.15378550e+00],
                 [ -9.72811787e+00,  -6.38761595e+00,  -4.18738015e+00],
                 [  5.16703423e+00,   5.85005977e+00,   2.58948736e+00],
                 [  2.33630384e+00,   1.16523476e+01,  -6.29475499e-02],
                 [  4.29801549e+00,  -1.45166682e+00,   2.53500205e+00],
                 [  1.25724271e+00,   1.27762696e+00,   2.12625767e+00],
                 [  6.49573317e+00,  -2.21124359e+00,   9.57796634e-02],
                 [ -1.06125571e+00,  -3.59956377e-01,  -1.96366016e+00],
                 [ -3.28313735e+00,  -1.26473147e+01,  -3.30099012e-01],
                 [ -4.14207571e+00,   3.06023288e+00,  -1.28566174e+01],
                 [ -6.01809607e-01,  -3.20567702e+00,  -8.48555871e-01],
                 [ -8.99276018e+00,   6.55931515e+00,  -7.92838061e+00],
                 [ -2.59565850e+00,  -1.68189623e+00,  -3.27268697e+00],
                 [ -1.01166618e+00,  -4.21734619e+00,  -1.35346740e+00],
                 [ -3.58182447e+00,   1.61675353e-01,  -2.25123364e+00],
                 [  8.68461167e+00,  -4.21539950e+00,   1.54182152e+01],
                 [ -9.20473749e+00,   5.58068862e+00,  -4.53258824e+00],
                 [  5.81289781e+00,  -1.06419576e+01,  -7.48683723e+00],
                 [  1.85789683e+00,   5.90939648e+00,   3.30509561e+00],
                 [  2.47506751e+00,   7.72511280e+00,  -5.62896191e+00],
                 [  1.62430676e+00,  -5.64240144e+00,  -8.87167052e+00],
                 [ -1.22787163e+01,   5.47706363e+00,  -2.65744840e+00],
                 [  9.78612997e-01,   1.38995397e+00,   9.46790649e+00],
                 [ -2.79857017e+00,   1.85055778e+00,   3.95375239e+00],
                 [  6.62140823e+00,  -6.23780086e+00,  -4.31895510e+00],
                 [ -1.01060878e+00,   1.08828217e+00,   1.33918344e+00],
                 [ -8.43742990e+00,  -1.09774995e+00,  -3.29215641e-01],
                 [ -7.53069087e+00,   3.33162030e+00,   1.02728770e+01],
                 [ -1.52531846e+00,   9.71989772e-01,  -1.83132255e+00],
                 [  1.13768211e+01,   6.02652996e-02,   8.25096665e+00],
                 [ -2.60998238e+00,   2.38990449e+00,   1.45776820e+00],
                 [  6.90009848e-01,  -1.59308138e+00,   7.84083089e-01],
                 [  2.38392675e-01,   1.95473376e+00,   1.28381369e+00],
                 [  9.63965303e-01,   7.32912313e+00,   3.35096877e+00],
                 [ -3.23802338e+00,  -3.49811419e+00,  -2.10926849e+00],
                 [  5.02972674e+00,  -4.78465445e+00,  -1.08103112e+01],
                 [  4.80454220e+00,  -4.74913435e+00,   8.85013674e+00],
                 [  2.79823346e+00,   4.85658948e+00,  -7.56854547e+00],
                 [ -1.08530923e+01,  -6.25793591e+00,  -4.18573384e+00],
                 [  8.19151082e-01,   1.47010065e+00,  -4.48199167e+00],
                 [ -3.23513344e+00,  -6.55026368e+00,  -5.57063441e+00],
                 [ -1.24562184e+00,   4.35047236e+00,   1.53093221e-01],
                 [ -3.11888663e+00,  -9.95224602e+00,  -2.50345959e+00],
                 [  4.41393361e+00,  -2.67221022e+00,  -4.61219111e-01],
                 [ -7.47775082e+00,   1.21296486e+01,  -1.83317995e+00],
                 [  7.84276456e+00,  -5.34792072e+00,  -3.31894934e+00],
                 [ -2.02373442e+00,   9.92266096e-02,   1.16218669e+00],
                 [  6.88933906e-01,   8.54936121e+00,  -6.69761071e-01],
                 [  1.17714541e+00,  -1.88772469e+00,  -2.15012676e+00],
                 [  1.73476233e+00,   5.66140737e-01,  -2.51010705e+00],
                 [  1.94629379e+00,   2.19861492e-01,   8.36280379e-01],
                 [  7.86338163e+00,  -5.65269120e+00,  -3.01694106e+00],
                 [  8.05791358e+00,   4.44968102e+00,  -6.04283282e+00],
                 [  1.67801091e+00,  -8.60537496e+00,  -8.92154293e+00],
                 [ -8.72572483e+00,  -3.75309834e-01,  -7.33588230e+00],
                 [  2.19867610e+00,   2.07908904e+00,   9.19002746e+00],
                 [  9.43657797e-01,  -8.01376423e+00,   5.95798266e+00],
                 [ -3.03107675e-01,  -2.21613192e+00,  -1.13717501e+01],
                 [ -2.36023659e+00,  -7.76955165e-01,  -5.42217789e+00],
                 [ -3.12624292e-01,  -1.11557754e+00,  -1.87654259e+00],
                 [ -2.98861265e+00,  -1.18402499e+01,   2.99583452e+00],
                 [ -2.17349837e-01,  -5.18422725e+00,  -2.19590160e-02],
                 [ -6.41572353e+00,   1.21564246e+01,  -1.72576675e+00],
                 [ -5.19891659e+00,  -7.86946743e+00,  -1.64323511e+01],
                 [ -3.00317830e+00,  -1.86300133e+00,   3.93460633e+00],
                 [  4.28504776e+00,   9.73171811e+00,  -4.13219311e+00],
                 [ -2.56366470e+00,  -8.48478872e-01,   9.91230913e-01],
                 [  8.30212453e-01,  -3.56345636e+00,   1.51143446e+00],
                 [  6.04433686e-01,   1.92295989e+00,   2.31358123e+00],
                 [  1.90657188e+00,  -6.73894434e+00,  -7.53065148e+00],
                 [  3.86147218e+00,   1.47455668e+00,  -4.98069596e-01],
                 [  1.53890069e-01,   1.39031191e+01,   4.63078935e+00],
                 [  6.61685191e+00,   4.51916077e+00,   4.07563329e+00],
                 [ -8.71602695e+00,  -1.25181062e+00,  -4.28166221e+00],
                 [ -4.10164423e+00,   5.27075768e+00,   1.09247819e+01],
                 [ -9.22024511e+00,   6.77077396e+00,   3.84261898e+00],
                 [ -2.98342523e+00,   6.04664172e+00,   7.93409488e+00],
                 [  3.90922427e+00,   2.84667717e+00,   3.52200372e+00],
                 [  1.78609952e+00,  -1.73251164e+00,  -4.22165066e-01],
                 [  2.53132517e-01,  -3.62336434e+00,  -1.03164477e-01],
                 [  1.47488225e+00,  -5.66789119e+00,  -4.16251497e+00],
                 [  5.63179148e+00,   7.66830082e+00,   8.44152759e+00],
                 [ -8.99994631e-01,   2.69107812e-02,   3.36523055e+00],
                 [ -5.66164866e+00,   9.99011577e+00,   2.50455559e+00],
                 [ -7.63976119e-01,   2.19132774e+00,  -7.98297753e-01],
                 [  4.63261189e+00,  -1.08730036e+00,   3.12771664e+00],
                 [ -5.62165454e-01,  -9.24907923e-01,   2.10964614e+00],
                 [  2.79997598e+00,  -5.00693808e+00,  -4.58489928e-01],
                 [  6.26510820e-01,  -3.76794833e+00,  -4.21323341e-01],
                 [ -3.79423842e+00,   5.00216764e+00,  -1.00628336e+01],
                 [ -1.05039259e+01,   7.17987763e+00,   1.60077475e+00],
                 [  4.39535410e+00,  -4.38209866e+00,   5.01605591e+00],
                 [  4.34196844e+00,  -1.36948649e+01,   2.73155557e+00],
                 [  2.64214402e+00,   7.43691050e-01,  -1.21473715e+01],
                 [ -1.07070070e+01,   4.48408982e+00,  -5.82938574e-01],
                 [  5.86900374e-02,   1.12747974e+00,   5.80657799e-01],
                 [  5.01665093e+00,   5.48334100e+00,  -7.01096997e+00],
                 [  9.73505311e-01,   1.60063595e+00,  -1.18470293e+00],
                 [  6.62788205e+00,  -7.51749569e+00,   6.04010159e+00],
                 [ -1.81752076e+00,   8.86818394e-01,   5.06503853e+00],
                 [ -2.62453963e-01,   8.81682335e-01,  -3.19965416e-01],
                 [  4.96442674e+00,   1.68046171e+00,  -1.14967245e+01],
                 [  8.21020321e-01,   6.76970466e-01,   6.69900550e-02],
                 [  2.64926782e+00,   1.71703003e+00,   1.32575112e+00],
                 [  5.59505305e-01,   3.28037601e-01,   3.12322089e+00],
                 [ -5.49671151e+00,  -1.00540893e+01,   9.90148396e-01],
                 [ -1.02821201e+01,   7.25148574e+00,  -2.99131448e+00],
                 [  3.35529618e+00,   1.40544447e+01,  -5.64728029e+00],
                 [ -1.28122665e+00,  -8.20359772e+00,  -3.38254438e-01],
                 [ -4.39482360e+00,   2.72412041e+00,   1.06071564e+01],
                 [ -2.65182999e+00,   1.05975368e+01,   8.99123710e+00],
                 [ -7.50014475e+00,  -7.51112331e+00,  -7.24249781e+00],
                 [  8.07263441e+00,   4.65774478e+00,   4.55617549e+00],
                 [  1.08048726e+00,  -3.23838454e+00,  -2.04908171e-01],
                 [ -1.25624875e+00,  -6.87778633e+00,  -1.10246936e+01],
                 [  1.30308416e+00,   4.41786460e+00,   2.60284840e+00],
                 [ -6.33755349e+00,   8.50484127e+00,  -1.03144295e+00],
                 [  1.00045690e+01,   8.94565706e-01,  -8.51090339e-01],
                 [  1.43042376e+00,   1.89394674e-01,  -3.92296250e-01],
                 [ -2.03216587e+00,   1.49107249e+00,  -1.22519890e+00],
                 [  1.76595733e+00,   4.50496650e-01,  -1.20397444e+00],
                 [  2.46951693e+00,  -4.30660482e+00,   1.28431739e+00],
                 [ -1.45919704e+00,  -8.06322301e-01,   5.60548259e-01],
                 [  4.07072902e+00,   7.19475141e+00,  -3.69979070e+00],
                 [  3.27558935e+00,   6.74285659e+00,   3.19233907e+00],
                 [  1.22440950e+01,   3.06424038e+00,  -4.55248336e+00],
                 [  2.33064752e-01,   9.61585417e+00,   3.23087803e+00],
                 [  1.17600032e+00,  -5.07677896e+00,   1.07553977e+01],
                 [  1.98674327e+00,  -1.27139703e+01,  -1.08446908e+00],
                 [  1.48121785e+00,  -7.76179243e+00,  -2.68601716e+00],
                 [  5.64135191e+00,   5.15696229e+00,  -8.34384565e+00],
                 [  2.79601830e+00,  -3.85470799e+00,  -4.97366415e-01],
                 [  3.03392402e+00,  -4.32491250e+00,  -6.39195395e+00],
                 [  9.38034041e-01,   6.02095767e-01,  -6.95488840e-01],
                 [ -6.40240742e+00,   3.23774738e+00,   1.03530344e+01],
                 [  2.23749302e-01,   2.67065551e+00,   9.60933426e+00],
                 [  6.32135895e-01,   3.56135383e-01,  -1.77454301e+00],
                 [  6.59416267e+00,   3.50157393e+00,   7.15182420e+00],
                 [ -1.18515887e+00,   1.89242836e-01,   1.38159545e+00],
                 [  1.05449311e-01,   1.23204092e+00,  -1.24590627e+00],
                 [ -1.39202755e+00,   6.19782243e-01,   1.74378785e+00],
                 [ -8.59049307e+00,  -1.17923854e+00,   7.87737272e-01],
                 [ -8.72839066e+00,  -1.07004514e+01,  -3.77530848e-02],
                 [ -4.71766423e+00,   3.41002066e+00,  -2.54400086e+00],
                 [ -6.64312249e+00,  -8.70895500e-01,   5.82662740e+00],
                 [ -5.12947659e-02,   2.03145153e+00,   4.85528694e+00],
                 [ -7.21599759e-01,   2.62321010e+00,  -4.38133934e+00],
                 [  5.60987275e+00,  -7.98926219e+00,  -3.82701271e-01],
                 [ -3.46089325e+00,   2.20578686e+00,   2.62909654e+00],
                 [ -1.41060137e+00,  -1.04235567e+00,   2.03196577e+00],
                 [  2.24895946e-01,  -1.72130647e-02,  -7.17146368e-02],
                 [ -2.22247198e+00,   3.00387054e+00,   6.79519371e-01],
                 [  8.46700021e+00,  -1.79214059e+00,   1.05686174e+01],
                 [ -7.89073106e+00,   9.15707176e+00,  -3.32547722e+00],
                 [  3.01921233e+00,   3.77065782e+00,  -2.29971884e+00],
                 [  6.11061118e-01,   8.58447181e+00,   2.74274333e+00],
                 [ -1.91358192e+00,   1.46429598e+00,  -1.83869151e+00],
                 [ -2.08456334e+00,   7.41890480e-01,  -5.08507180e-01],
                 [  2.83388177e+00,  -2.95410062e+00,   1.56513660e+00],
                 [  4.10215852e+00,   5.77793953e+00,   9.80464786e+00],
                 [  5.87630683e+00,  -7.57993241e-01,  -4.95157853e+00],
                 [  6.46971387e-01,   8.85749237e+00,  -4.00543776e+00],
                 [  8.86827272e+00,  -2.17738771e+00,   4.95604095e+00],
                 [  4.04030097e+00,   8.40733842e+00,   1.63017483e+00],
                 [ -1.84159831e+00,  -1.46542051e+01,  -4.47366857e-01],
                 [  3.92364575e-01,  -9.23042274e+00,  -4.93136441e+00],
                 [ -1.94061124e-01,  -6.45531258e+00,   3.77770428e-01],
                 [ -3.42983880e-02,   1.23845417e+00,  -1.35591642e+00],
                 [  2.69431382e+00,  -3.89526135e+00,   5.42096729e+00],
                 [  2.77186765e+00,   1.57568012e+00,   1.81596613e+00],
                 [ -3.71924212e+00,  -5.77529929e+00,   1.02214089e+00],
                 [ -1.80913635e+00,  -5.62164534e+00,  -3.24099695e+00],
                 [  2.71510380e+00,  -1.17027673e+00,  -4.10257255e+00],
                 [ -4.58858711e+00,   2.39493068e+00,  -2.95900187e+00],
                 [  1.44712887e+00,   3.00708757e+00,   1.40116098e+00],
                 [  1.68274121e+00,  -7.98874660e-01,   1.45632250e+00],
                 [ -2.02908905e+00,  -5.18408283e-02,   2.34399705e+00],
                 [  6.75164012e-01,   2.93132875e+00,  -3.66520413e+00],
                 [  1.13839410e+00,   1.10667999e+01,  -6.64432426e+00],
                 [ -3.72008730e+00,  -1.21808634e+01,  -6.44278725e+00],
                 [ -1.71368456e+00,   7.91830192e+00,  -9.16838849e+00],
                 [  6.33875278e+00,   6.22496550e+00,  -3.69279564e+00],
                 [  6.65816816e+00,   2.34510124e+00,   1.19967728e+01],
                 [  9.82635125e-01,  -1.15610441e+01,   3.48151841e+00],
                 [  5.75073029e+00,  -5.93829500e-01,   2.72898855e+00],
                 [ -4.67397279e+00,  -5.10347343e-03,   1.19518869e+00],
                 [  1.52269160e-01,  -7.00550190e+00,   5.21867841e+00],
                 [ -3.81725393e+00,  -3.23213138e+00,   2.45756833e+00],
                 [  4.86721682e+00,  -1.35747772e+01,   4.25716307e-01],
                 [  9.97426833e-01,  -1.31007727e+01,  -6.29678541e+00],
                 [ -2.15087166e+00,  -1.35285693e+00,  -1.17939065e-01],
                 [ -9.42975299e+00,   7.74498407e+00,   1.75661342e+00],
                 [ -1.16776198e+00,  -1.88305815e+00,   1.22845023e+00],
                 [ -3.72043348e-01,  -3.35565914e-01,   8.62064591e-01],
                 [ -2.96540600e+00,  -2.04229552e+00,  -3.64710938e-01],
                 [  1.06242261e+00,  -6.99173871e-01,   1.41198611e+00],
                 [ -7.79840345e+00,   6.71690739e-01,  -1.01522341e+00],
                 [  2.50503539e+00,   3.72514286e+00,   4.94227090e+00],
                 [ -9.52869790e+00,  -3.81052219e+00,  -6.95943596e+00],
                 [ -3.96737999e+00,  -8.20731979e+00,  -8.57220067e+00],
                 [ -6.19852931e-02,   1.32446031e+00,   1.89661649e+00],
                 [  2.01628156e+00,   1.16333961e+00,  -5.89608780e+00],
                 [ -1.36188931e+00,   8.22366493e+00,   3.10152740e+00],
                 [  8.99508066e-02,   7.39509458e-01,  -4.15595211e-01],
                 [ -9.82112958e+00,   8.02538506e+00,  -1.27031312e+00],
                 [  3.43621251e+00,  -7.50057899e-01,   7.61375378e-01],
                 [ -8.31070200e+00,   5.72969116e+00,  -5.68295538e+00],
                 [  1.10292448e+01,  -5.60932565e+00,  -3.68791748e+00],
                 [  7.49178025e-01,  -2.55384186e+00,   2.00437889e+00],
                 [  9.72592083e+00,   2.41554203e+00,  -1.42046831e-01],
                 [  2.02253004e+00,   1.81987749e+00,   1.11846772e+00],
                 [  2.06877768e-01,   3.64395877e-02,   5.32080965e-01],
                 [ -8.57594130e-01,   2.65357914e-01,   3.45765173e+00],
                 [ -1.39978495e-01,   3.71554776e+00,   1.66683674e+00],
                 [ -1.29498764e+01,  -4.11337581e+00,   3.05650566e+00],
                 [  1.12765535e+01,   8.73935172e-01,   4.22781907e+00],
                 [  6.60208233e+00,   5.19044021e+00,  -7.19072470e+00],
                 [  6.92874117e+00,  -2.15481879e+00,  -9.00385010e+00],
                 [ -9.00700573e+00,  -1.50743760e-01,   7.53744478e+00],
                 [ -4.06216737e+00,  -2.31412163e+00,  -3.59640180e-01],
                 [  1.18201006e+00,  -1.44522072e+00,   3.69797560e+00],
                 [  1.29463697e+00,  -1.54781758e+00,   1.42912639e+00],
                 [  9.69341315e+00,  -5.10288658e+00,   2.39005450e+00],
                 [ -1.64293700e+00,  -9.83566715e-01,   1.23142321e+00],
                 [ -1.40669335e+00,   2.67237565e+00,   1.11637277e+01],
                 [ -2.96798592e+00,   2.47089691e+00,   5.86251182e-01],
                 [ -5.88623460e-01,   1.81925980e+00,   1.59639115e+00],
                 [  3.21187067e+00,   6.67380854e+00,  -7.64222875e+00],
                 [  8.61793906e-02,   3.69164417e+00,  -2.49569047e+00],
                 [  3.97078324e-01,  -2.99606222e+00,   1.32477678e+00],
                 [  1.68603432e+00,  -6.80748080e-02,   1.97483152e+00],
                 [ -1.25004240e+01,   1.07615376e+01,  -6.59531484e+00],
                 [ -1.11770097e+01,   6.97347518e+00,  -3.70349066e+00],
                 [ -1.35126300e+00,   8.74404291e+00,   2.77309708e+00],
                 [ -7.23048983e-02,   2.60523949e+00,  -1.41071548e+00],
                 [ -2.72461882e+00,  -1.37282497e+01,  -1.84892824e+00],
                 [  2.99458202e+00,   5.46772196e+00,   1.79287620e+00],
                 [  5.14672696e+00,   9.45443739e+00,   3.64404948e+00],
                 [ -3.28300146e-01,   8.79400190e+00,   1.20808271e+00],
                 [  9.53190937e-01,  -6.69951255e-01,  -9.33403016e-01],
                 [ -7.36879972e+00,   1.90481356e+00,  -3.11174874e+00],
                 [  3.64642047e-01,  -1.47891546e+00,   2.54972245e+00],
                 [ -4.17292839e+00,  -3.47101192e+00,  -3.27613933e+00],
                 [  8.00879660e+00,  -7.28599343e+00,  -3.35132251e+00],
                 [ -6.00399360e-01,   2.87617318e+00,  -6.25220492e-01],
                 [  2.70885413e+00,  -9.24426810e+00,   9.20675116e+00],
                 [ -9.19006010e-01,   4.48739079e-01,  -1.27935004e+00],
                 [  1.92868774e+00,   5.89539265e-01,  -2.34308348e+00],
                 [ -2.61854204e+00,   9.45677869e-01,  -1.40167633e+00],
                 [ -8.53546121e+00,   8.91978644e+00,   5.70139081e-01],
                 [  1.56217148e-01,   3.97043279e+00,   1.64961259e+00],
                 [ -8.19044982e+00,   2.54376037e+00,   3.26323922e+00],
                 [  3.74980334e+00,  -6.85576340e+00,   9.07857452e-01],
                 [  1.04089585e+01,  -4.93636532e+00,   5.92720655e+00],
                 [ -1.55586268e+00,  -6.00992497e+00,  -1.83865228e+00],
                 [ -4.77215047e+00,  -7.67308049e-01,  -2.38568789e+00],
                 [  1.36044817e+00,   1.27512311e+01,  -2.26547023e+00],
                 [ -3.50096264e+00,   2.94202473e-01,  -1.17803116e-02],
                 [  4.52440430e+00,  -5.07303446e+00,   8.72821327e-01],
                 [  3.23873385e-01,   1.34663860e+00,   1.39764644e+00],
                 [ -3.38750561e+00,   1.36885717e+00,   1.04164094e+01],
                 [  6.02503393e+00,  -6.40340890e-03,   3.10955969e+00],
                 [ -8.82092695e-01,  -2.00825164e+00,  -3.89188687e-01],
                 [  9.87268829e+00,  -1.26546763e-01,   7.94597699e+00],
                 [  5.39254369e-01,   2.72764642e+00,   2.15497304e-01],
                 [  1.96520839e+00,   8.48821157e-01,   3.22767072e+00],
                 [  1.61942059e+00,   2.07590252e-01,  -1.91806522e-01],
                 [ -4.64108760e+00,   7.89921855e+00,   4.45699275e+00],
                 [ -4.46134763e+00,   1.00961239e+01,  -5.34739569e+00],
                 [ -8.26318681e-01,   4.39306323e-02,  -5.14241915e+00],
                 [ -8.67491221e-01,  -5.28687701e+00,  -1.13821536e+01],
                 [  5.68011366e+00,  -7.55379850e+00,   5.57524022e+00],
                 [  3.01557995e+00,   8.36572342e+00,  -2.42393258e+00],
                 [  5.56731403e+00,   1.03531090e+01,   1.43661524e+00],
                 [ -5.37209161e+00,  -4.01846097e+00,   6.11697752e+00],
                 [ -1.84089040e+00,  -1.33562849e+00,   3.43346695e+00],
                 [  4.37212256e+00,   8.09178075e+00,  -6.68686826e+00],
                 [  1.14568213e+00,  -1.27062226e+00,  -3.64126675e-01],
                 [  4.90722479e+00,  -7.91954793e+00,   7.05576048e+00],
                 [  3.91697966e+00,   4.92033176e+00,   8.73379531e+00],
                 [  9.32451850e-01,   9.54383265e-01,  -2.01173555e+00],
                 [ -5.23265949e+00,  -8.76580330e-01,   4.69666924e+00],
                 [  2.80087011e+00,  -4.14321426e-02,   2.86683047e+00],
                 [  4.49817220e-01,  -1.78718233e+00,  -9.26657476e-01],
                 [ -1.81533990e+00,  -4.62434497e+00,  -1.37900900e+00],
                 [ -6.26709755e+00,  -1.66269882e+00,   3.66893940e+00],
                 [ -3.99776042e+00,   2.53718667e+00,   8.76641534e+00],
                 [ -6.43851336e+00,   2.22862403e+00,   1.27911053e-02],
                 [  1.16690957e+01,  -2.81877872e+00,   4.64803382e+00],
                 [ -9.97441826e-01,  -5.04911325e+00,  -5.99649080e+00],
                 [ -1.02858227e+01,  -2.20716564e+00,   5.79853906e+00],
                 [ -9.30676221e+00,   1.43403935e+00,  -5.50563700e+00],
                 [  5.33417578e+00,   4.36609236e+00,   5.47780163e+00],
                 [  3.19114865e+00,  -2.84904525e+00,   5.56637809e-01],
                 [  7.46744557e+00,   6.60664399e+00,   5.72281692e-01],
                 [  2.06187508e+00,   1.40765271e+00,   4.91049641e-01],
                 [  1.04247799e-01,   3.24124806e+00,  -6.74946299e+00],
                 [ -1.29942311e+00,  -2.37032347e-01,   4.95088405e+00],
                 [  1.20433832e+00,   8.30071330e-01,   1.62664125e-02],
                 [  2.16151474e+00,   8.32466536e-01,  -1.80916664e+00],
                 [ -3.44149330e+00,   9.47863027e-02,   7.21904485e-01],
                 [ -1.45147884e+00,  -2.40458514e+00,  -2.21052250e+00],
                 [  1.00706775e+00,   9.77840700e-01,  -1.14057923e+00],
                 [ -1.74438167e+00,  -1.19370812e+00,  -2.20963277e+00],
                 [ -1.03587581e+01,   6.56052045e+00,   2.74466820e+00],
                 [  5.18902921e+00,  -6.14999667e-01,  -6.06021652e+00],
                 [ -1.01691712e+01,  -7.86714171e-01,   9.77055785e+00],
                 [ -1.21037113e+01,   3.34709678e-01,   5.93007257e+00],
                 [ -4.27129763e+00,  -5.49914051e+00,  -7.46502135e+00],
                 [ -2.55335661e+00,   2.92990022e+00,   1.13882852e+01],
                 [ -3.50256513e+00,   1.00510588e+00,  -1.15420471e+01],
                 [  1.45410768e+00,   2.20778167e+00,   6.82148650e-01],
                 [  2.07471074e+00,   9.66816627e+00,  -5.70862211e+00],
                 [ -2.84526637e+00,   1.38172531e+00,  -1.68300880e+00],
                 [  7.88174563e+00,   3.81317220e+00,  -4.38145694e+00],
                 [ -4.96185703e+00,  -4.72445704e+00,   8.17718657e+00],
                 [  8.62586632e-01,  -5.02177319e-01,   6.65044439e-01],
                 [ -6.56093335e+00,  -6.14096951e+00,  -7.66478664e+00],
                 [  1.65745771e-01,  -1.95887322e+00,  -1.52825081e-02],
                 [ -2.29022490e+00,   1.10733560e+00,  -2.53072211e+00],
                 [  7.23370542e-01,   4.14577321e-01,   2.21535162e+00],
                 [ -4.58682913e+00,  -4.93042659e+00,  -1.27034951e+01],
                 [ -1.37133020e+00,  -7.31680967e+00,   1.00585987e+01],
                 [ -1.32859257e+01,   2.95802445e+00,  -1.69478630e+00],
                 [  4.31449598e+00,   5.77299244e+00,  -9.64753908e+00],
                 [ -2.71189844e+00,   2.12968305e+00,  -9.67055976e+00],
                 [  5.03455334e+00,  -1.22052099e+01,  -1.43685257e+00],
                 [  5.31937618e+00,   9.45425532e-01,   6.65672038e+00],
                 [ -7.97577029e+00,  -7.21997918e+00,  -1.70230062e+00],
                 [  1.04309256e-01,  -2.16138729e-01,  -1.85891578e+00],
                 [  5.51230798e+00,   8.24415399e-01,  -9.08098582e+00],
                 [ -2.19893731e+00,   1.96151493e+00,   3.69872781e+00],
                 [ -5.74721016e+00,   1.49934910e+01,   4.09091013e+00],
                 [ -6.16158200e+00,   1.33150836e+01,   6.63161856e+00],
                 [ -1.69309925e+00,  -4.33218772e+00,  -4.11783842e+00],
                 [ -2.88605299e+00,  -6.41046787e+00,  -8.89580876e-01],
                 [ -3.82216135e+00,  -3.34316784e+00,  -3.16310948e+00],
                 [ -4.57858643e-01,  -7.88621613e-01,  -2.82611046e+00],
                 [  1.26339181e-01,  -2.24783449e+00,   2.72978948e+00],
                 [ -1.56078026e-01,   3.76077521e+00,  -1.34148521e+01],
                 [ -2.75435333e+00,   4.68111373e+00,   9.10379402e+00],
                 [  2.20529692e+00,  -9.72767416e+00,  -6.86482284e+00],
                 [ -2.94596924e+00,  -3.58444007e+00,   1.53388485e+01],
                 [ -3.91171072e+00,   2.81097862e+00,  -2.56136362e+00],
                 [ -2.87825784e+00,  -1.37578977e+01,   1.61743720e+00],
                 [ -1.66780048e+01,   7.38128741e+00,   5.71917054e-01],
                 [ -8.65198706e+00,   3.96091880e+00,   5.39560535e+00],
                 [ -1.37020072e+00,   2.22186984e+00,  -9.15675530e-01],
                 [  9.74368286e-01,  -1.22693563e+01,   3.34934947e+00],
                 [  8.40804948e-01,   2.16282117e+00,  -1.94925676e+00],
                 [ -7.42333147e-01,  -4.11583260e-01,  -1.48213988e+00],
                 [ -8.13139999e+00,  -5.35937154e+00,  -5.25302030e+00],
                 [  7.39314619e-01,   2.03176647e+00,   1.20459544e+00],
                 [  6.37764539e+00,  -2.11406464e+00,  -7.61021230e+00],
                 [ -5.63619755e-02,  -2.53798228e+00,  -6.48015737e-01],
                 [ -3.18190136e-01,  -3.49569535e+00,  -1.65511035e+00],
                 [  1.08509691e+00,  -1.34335341e+00,   1.38607615e-01],
                 [  7.52852914e+00,   1.19419708e+01,   1.26919096e-02],
                 [  6.79948529e+00,   4.96006275e+00,   2.77355789e+00],
                 [ -3.78045586e+00,   4.96932169e+00,  -9.91986469e+00],
                 [ -1.47880522e+00,   7.06481562e+00,  -3.45901108e+00],
                 [  1.43848105e+01,  -2.07878060e+00,  -1.98938075e-01],
                 [  4.91592897e+00,   4.94076693e+00,   1.45194147e+01],
                 [  6.23182325e+00,  -8.67421809e+00,  -3.34547107e+00],
                 [ -2.48584069e+00,  -1.13307716e+01,   5.17376602e+00],
                 [ -9.29479015e-01,   1.89452349e+00,   1.29533458e+00],
                 [ -4.55625768e+00,   5.30385876e-01,  -3.91025753e+00],
                 [ -1.33310929e+00,  -2.69847940e+00,  -8.26250481e-03],
                 [  1.00120445e+01,   5.02738124e+00,  -6.15856423e+00],
                 [  5.62433249e-01,  -6.87903599e+00,  -2.45494068e+00],
                 [ -2.11715046e+00,  -1.27251302e+00,  -3.77053543e+00],
                 [  1.36532918e-01,   1.33789162e+01,   2.01208123e+00],
                 [  1.19073802e-01,  -5.13094728e-02,  -9.84282023e-01],
                 [  6.18129290e-01,   2.62318321e+00,  -3.55607404e-01],
                 [ -1.45105017e+00,  -5.23116541e-01,  -2.78576941e+00],
                 [ -1.30399562e+01,   2.87689742e+00,  -1.00166400e+01],
                 [  7.69033367e-02,   2.58859345e-01,   1.45428291e+01],
                 [  4.69355043e+00,  -9.10155596e+00,   2.27510475e+00],
                 [ -8.04013873e+00,   1.51672753e+01,  -5.90896736e-01],
                 [  2.35424354e+00,  -3.42711096e+00,  -4.44257709e-01],
                 [ -2.47816231e+00,  -1.28288140e+01,  -5.36495452e+00],
                 [ -2.73034531e+00,  -5.22369951e+00,   7.74028730e+00],
                 [  1.86090754e+00,   9.09374565e+00,  -3.02355140e+00],
                 [  1.42905801e+00,  -2.37239448e+00,   1.56103346e+00],
                 [  1.55313053e+00,   1.30622801e+00,  -1.67646000e+00],
                 [ -2.47444895e+00,  -3.40262694e-01,   4.93017133e-02],
                 [ -8.33996752e+00,   8.96836454e-02,   2.99823551e+00],
                 [ -2.62610667e+00,  -1.41252994e+00,   1.16964575e+01],
                 [ -1.37448124e+00,  -5.42390143e-01,   1.05643382e+00],
                 [  7.07898492e+00,  -5.53518249e+00,   1.15485668e+00],
                 [  1.24530879e+00,   1.23560004e+00,  -9.34988704e-01],
                 [ -1.18443665e+00,   1.74791832e+00,   1.79257512e-01],
                 [  9.50781319e-01,  -2.36502511e+00,   6.45129322e-01],
                 [  1.32490920e+00,  -9.68925118e+00,  -3.51430667e+00],
                 [  1.29529631e+00,  -1.29062831e+01,  -2.92116531e+00],
                 [  3.56660827e+00,   5.95815823e+00,  -6.67872887e+00],
                 [  4.98989824e+00,   1.30906515e+00,   1.48825760e+01],
                 [  5.23976016e+00,   5.83954525e+00,  -1.23015586e+01],
                 [  2.03125849e+00,   2.63415814e+00,   1.27429876e+00],
                 [ -3.75493045e+00,  -9.96050710e-01,   3.81245167e+00],
                 [  9.76745229e+00,  -2.74959357e+00,   5.04240916e+00],
                 [ -4.45093621e+00,  -3.81809660e-01,   2.32164306e+00],
                 [  1.09061903e+01,   2.50449904e+00,  -5.46438153e+00],
                 [  1.42958942e+00,   1.84965060e+00,   1.43584283e+00],
                 [ -7.72064543e+00,   3.03109190e+00,  -3.39708065e+00],
                 [ -3.49162789e+00,  -3.04843732e+00,   2.62648829e+00],
                 [  1.23353120e+00,  -1.79793891e+00,   2.72515488e+00],
                 [  6.66494385e+00,  -1.04227557e+01,  -7.47934193e-01],
                 [  2.07165640e+00,   2.76712637e+00,   2.71721841e+00],
                 [ -2.51711463e+00,   8.37792400e-01,  -3.28314257e+00],
                 [  4.22542339e-01,  -6.16152911e-01,   6.54847683e-01],
                 [ -9.63447292e-01,   6.07723479e-01,  -4.78673978e-01],
                 [ -8.26625751e+00,   5.88577239e+00,  -6.95513178e+00],
                 [  1.75377638e+01,  -7.77607936e+00,  -1.50131658e+00],
                 [ -2.27819217e+00,   3.91907968e+00,   2.77565760e-01],
                 [ -9.19089987e+00,   9.98994711e+00,   1.08882273e-01],
                 [  1.82156739e-01,  -3.10683602e-01,  -1.41948709e+01],
                 [ -1.43732538e+00,  -9.76024042e+00,   7.59625190e+00],
                 [ -9.82145024e+00,   1.37845663e+00,  -8.04281629e+00],
                 [  9.10609615e-01,  -1.35817345e+00,  -1.25955820e+00],
                 [  9.14202675e+00,  -8.12018875e+00,   5.45314709e+00],
                 [ -2.03410408e+00,   1.17801243e+00,  -1.10578610e+00],
                 [  5.50868885e-01,  -8.14965590e+00,  -3.69128786e+00],
                 [ -7.47139214e+00,   1.54731369e+01,   4.68112383e+00],
                 [  1.35638580e+00,   8.48911970e-01,   2.51732948e+00],
                 [  6.06841130e+00,  -3.29237028e+00,  -1.05907978e+01],
                 [ -1.85154066e+00,  -1.64583324e+00,   6.56067146e-01],
                 [  6.35679832e-01,  -1.89777318e+00,  -1.05719326e+00],
                 [  1.05583018e+00,  -1.80803697e+00,  -7.29914442e-03],
                 [  7.99941021e+00,  -5.20798699e-01,   7.99813275e+00],
                 [  2.48598634e+00,   8.26503827e+00,   2.49576527e+00],
                 [ -2.23990927e+00,   6.67722249e+00,  -4.32866547e+00],
                 [ -8.68814548e-01,  -1.15420959e+01,  -3.72643719e+00],
                 [  3.30197542e+00,  -4.40171633e+00,   3.59974515e+00],
                 [  8.02514683e+00,  -2.39312747e+00,  -1.39866433e+00],
                 [ -4.79578275e+00,  -9.60297833e+00,  -3.28876534e+00],
                 [  2.08293607e+00,   8.51398607e-02,  -1.07270856e+01],
                 [  2.18960150e+00,   2.07966201e+00,   3.33515021e-01],
                 [ -2.32393133e+00,   1.03773492e+01,  -1.82919393e+00],
                 [  2.15483248e+00,   1.31281256e+00,   1.06862524e+00],
                 [  6.66235582e+00,   1.55800673e-01,  -2.12733169e+00],
                 [  6.13372444e+00,   2.75604072e+00,   6.40884295e+00],
                 [  1.55957726e-01,  -1.47786311e+00,   4.47016516e-01],
                 [  3.43703478e+00,   7.81158589e-01,  -5.31892943e-01],
                 [  3.08294177e+00,  -2.81139602e+00,  -1.01158246e+00],
                 [ -3.35985200e+00,   4.67556059e-02,   4.26136406e-02],
                 [  9.93900826e-01,   2.96284518e-01,  -1.69749389e+00],
                 [  6.13855093e+00,   1.42395819e+01,   7.24833836e+00],
                 [ -1.13737231e+01,   7.03817542e+00,   4.07406596e-01],
                 [ -3.65601682e+00,  -2.05014951e+00,   3.99305708e+00],
                 [  8.54472747e+00,  -1.05511856e+00,   5.29452252e+00],
                 [ -1.73591827e+00,  -6.20193188e-01,  -1.72451773e+00],
                 [ -6.00354615e+00,  -5.03305471e+00,  -1.02113131e+01],
                 [  8.98360997e+00,   3.11264195e+00,  -6.30976590e+00],
                 [ -1.14740047e+00,  -1.23845970e-01,   9.94609570e+00],
                 [ -1.56526600e+00,   5.90506092e-02,  -1.00302231e-01],
                 [ -5.21133814e+00,  -9.06409731e+00,   3.60569192e+00],
                 [  1.84384807e-01,  -1.78907007e+00,  -3.37611463e+00],
                 [  6.57498171e+00,   1.20110195e+00,   5.44176307e+00],
                 [  6.73545326e+00,   1.19377485e+01,  -2.05951831e+00],
                 [  7.45645609e-01,   1.10551230e+00,   1.54546121e+00],
                 [ -2.11067369e+00,   2.86350308e+00,  -6.19880370e+00],
                 [ -4.83744891e-01,   1.06958946e+00,  -1.36141043e-01],
                 [  1.67872512e+00,   2.27362637e+00,  -3.70464460e-01],
                 [ -6.87269978e-01,  -8.25731362e-01,  -1.00596266e+00],
                 [  8.73661520e+00,   4.66472264e+00,   4.72729122e+00],
                 [ -2.27851076e+00,  -7.52944681e+00,  -3.12444947e+00],
                 [  3.17525200e+00,  -9.19614265e+00,  -6.47695179e+00],
                 [  5.97783864e+00,   6.73150938e+00,   4.09434379e+00],
                 [ -5.41851539e+00,  -9.40713989e+00,  -4.03322688e+00],
                 [ -6.60669907e+00,  -5.94375019e+00,  -1.05846401e+00],
                 [  6.41696047e+00,   9.51733595e+00,  -6.10221296e+00],
                 [ -1.44333205e+00,  -1.97008261e+00,   6.01594541e+00],
                 [ -3.35229293e+00,   2.10723263e+00,  -6.05856969e-01],
                 [ -2.23481340e+00,   2.44234537e+00,  -1.28047414e+01],
                 [ -2.88934332e+00,  -1.51070129e+00,   2.86837596e+00],
                 [  4.55658891e+00,  -7.03545594e+00,  -8.67064415e+00],
                 [  7.64665859e+00,   2.94469674e+00,   2.06606665e+00],
                 [ -5.27826497e-01,   3.03905172e+00,  -9.30365531e-01],
                 [  9.40822829e+00,  -3.40853878e+00,  -1.09355548e+00],
                 [ -2.79256107e+00,   8.53602723e-01,  -4.61724601e-01],
                 [ -2.00363602e+00,  -1.23571074e-01,  -1.00113930e+00],
                 [  4.98547121e-01,   9.04745564e-01,   1.23245476e+00],
                 [  4.36661923e+00,   2.54698884e+00,   1.55071712e+00],
                 [ -1.20101986e+01,   4.50300512e+00,  -5.44827383e+00],
                 [  4.59389526e+00,   2.48404161e+00,  -9.68554808e+00],
                 [ -5.74517731e-01,  -2.32199319e+00,  -4.19651864e+00],
                 [ -2.97073588e+00,  -3.25524363e+00,   4.64992920e+00],
                 [  9.50533056e+00,  -6.43817751e+00,  -6.12129973e-02],
                 [ -1.29940413e+00,  -5.92379218e+00,   1.05818036e+01],
                 [  6.11547164e-01,   9.34236163e+00,   2.31384618e+00],
                 [ -1.19211592e-01,  -7.15828151e-01,   1.79122494e+00],
                 [ -6.15242241e-01,  -3.53580436e+00,  -1.02272108e+01],
                 [  1.66470320e+00,  -2.52828389e+00,   1.39245693e+00],
                 [ -4.04695287e+00,   1.54391231e+00,   1.21215099e+00],
                 [  4.08517429e+00,   1.13886835e+00,  -2.33902387e-01],
                 [ -2.07926719e+00,  -2.87396356e+00,   2.20117508e+00],
                 [  8.23958174e+00,  -5.87066318e+00,  -1.31072838e+00],
                 [  7.63210284e-01,  -2.93354693e-01,   1.70702659e+00],
                 [  2.10111142e+00,  -7.36376296e-01,   1.59696145e+00],
                 [ -2.78474849e-01,  -1.02843907e+00,  -2.91150443e+00],
                 [  4.64551564e+00,   9.27405169e+00,  -8.76141193e+00],
                 [ -1.00205519e+01,   5.96428742e+00,   3.31187914e+00],
                 [ -7.59405826e+00,   9.99000540e-01,   1.42678506e+00],
                 [ -1.13390947e+00,  -9.67459694e-01,   3.17053978e+00],
                 [ -2.32183965e+00,  -1.06228177e+01,  -6.66047142e+00],
                 [ -3.93205294e+00,  -9.33014202e+00,   2.44311815e+00],
                 [ -4.81607930e+00,   5.07538287e+00,  -1.29099149e+01],
                 [  7.19195443e+00,  -2.97300109e+00,  -9.69131399e-01],
                 [ -4.23893180e-02,  -8.84048733e-01,   2.09313949e+00],
                 [  2.34037076e+00,  -3.36871743e+00,   7.81829157e+00],
                 [  2.56842704e+00,  -5.92549942e-01,   8.10366986e-01],
                 [  8.00974157e+00,  -3.57976003e+00,   1.12140769e+01],
                 [ -4.63146670e+00,   1.45724348e+00,  -7.80469015e+00],
                 [ -6.35926009e-01,  -1.83792447e+00,  -3.23664404e+00],
                 [  5.10007080e+00,   6.63501244e+00,   8.94129483e+00],
                 [ -2.39776085e+00,   1.30514596e+00,  -1.55220625e+00],
                 [  3.01743316e+00,  -3.53234829e-01,   1.04437625e+00],
                 [  2.86649584e+00,  -6.53753878e-01,  -4.42217214e-01],
                 [ -1.93704436e+00,  -5.23515934e+00,  -4.52329642e+00],
                 [ -2.82487551e-01,  -1.28541318e+01,   3.44114826e+00],
                 [ -6.98606711e+00,  -8.05564841e+00,  -9.28291329e+00],
                 [  8.33923059e+00,  -8.31994521e+00,  -6.95912538e+00],
                 [  1.02984180e+01,  -2.20194680e+00,   9.02023760e+00],
                 [ -2.47185648e+00,  -1.03571124e+01,   1.02137370e+01],
                 [  9.57624960e+00,   6.78207567e+00,  -2.26107162e+00],
                 [ -1.07134331e+01,   4.65535172e-01,   1.70938381e+00],
                 [ -7.05689355e-01,  -5.80834828e-02,   1.70150607e+00],
                 [ -5.67327950e+00,   3.02153623e+00,  -5.26261597e-01],
                 [ -1.90438166e+00,   3.22593069e-01,  -5.61576617e-01],
                 [ -6.00370962e+00,  -3.56046660e+00,   8.96565110e+00],
                 [ -1.96252488e+00,   7.98501091e+00,  -3.78799843e+00],
                 [  1.06389379e-01,  -4.32150353e-01,   1.33645311e+00],
                 [ -1.46735494e+01,  -3.54461704e+00,  -1.83283353e+00],
                 [ -2.80258842e-01,  -1.09342767e-01,  -1.29762381e+00],
                 [ -1.08805355e-01,  -4.08727342e+00,  -1.11110311e+00],
                 [ -1.49243161e+00,   4.60409693e+00,  -5.78692621e-01],
                 [  3.82772445e+00,   3.39151109e+00,   1.36167162e+01],
                 [ -2.02525127e+00,   5.16129987e-01,   2.78753390e+00],
                 [  1.44824260e+00,  -3.15983098e-01,   2.34261569e-01],
                 [ -9.83844454e-01,  -1.44806728e+00,   1.38665805e+01],
                 [  9.84451524e+00,   2.33330146e+00,   6.09982434e+00],
                 [ -4.29809805e+00,   8.73550465e+00,  -1.00384773e+01],
                 [  4.24743408e+00,   1.14683595e+01,  -1.79548301e+00],
                 [ -9.85269607e+00,  -5.07486347e+00,   8.62961629e-01],
                 [  1.31109950e+00,  -1.40170981e+00,   3.43542678e+00],
                 [  1.06380391e+01,   7.03286874e+00,  -3.72313044e+00],
                 [  1.02947308e+00,  -1.09299988e+00,  -6.40588695e-01],
                 [ -8.85703254e+00,   1.88540664e+00,   2.77287262e+00],
                 [  6.64704381e+00,   2.55615234e+00,   6.59186928e+00],
                 [ -1.07909073e+00,  -1.36339655e+00,   2.19886043e+00],
                 [ -3.32763603e+00,   1.91101114e+00,  -2.56889027e+00],
                 [ -5.24295101e-01,  -1.35816779e+00,  -2.34437024e+00],
                 [ -8.82039155e-01,  -2.91610572e+00,   1.74160150e+00],
                 [  3.36587083e+00,   4.32385582e+00,  -4.07679802e+00],
                 [ -2.52799290e+00,  -2.97302210e+00,   2.21373764e+00],
                 [ -1.19512278e+01,  -1.09254823e+00,  -1.86082603e+00],
                 [ -3.48517333e+00,  -5.34975231e+00,  -9.73386749e+00],
                 [  1.12370470e+01,  -1.02340763e+01,   6.11179031e+00],
                 [ -1.03816407e+00,  -4.53576885e+00,   9.77886489e+00],
                 [ -9.14325959e+00,  -2.84070024e-03,   1.83650141e+00],
                 [  3.37676430e+00,  -4.54483814e+00,   3.32432995e+00],
                 [ -1.23264714e+00,  -1.06038780e+00,  -1.36941488e+01],
                 [ -1.40646949e+00,  -1.15995268e+00,   1.60574757e+00],
                 [  7.44791194e+00,  -4.34736332e+00,  -5.43396665e+00],
                 [  2.97293931e+00,   1.47388681e-01,  -1.19105233e+00],
                 [ -4.66099075e+00,  -2.12249339e+00,   4.05270491e+00],
                 [ -7.53606308e-01,   1.05108050e+01,   6.74859570e+00],
                 [ -3.58479895e+00,   1.73581107e+00,  -2.12650789e-01],
                 [ -6.93119119e+00,   1.99234206e+00,  -3.58621884e+00],
                 [  4.70171327e-01,  -2.23180008e+00,   2.22497477e+00],
                 [ -2.50042595e-01,  -2.66090212e+00,  -4.95774256e-01],
                 [  5.16364222e+00,   2.43721420e+00,  -1.59142826e-01],
                 [ -9.71433181e-02,   3.59725267e-01,  -8.31261559e-01],
                 [ -7.46299068e+00,  -4.26232079e+00,  -8.82523001e+00],
                 [ -6.32191742e+00,   6.43387643e+00,   3.93453283e+00],
                 [  6.32801223e+00,   1.03034869e+01,  -6.71339979e-01],
                 [ -5.78817245e+00,   1.15610885e+00,   2.51342325e+00],
                 [  8.58066878e+00,   1.00923264e+01,   2.85619797e+00],
                 [ -1.04365427e+01,   2.40005869e+00,   1.71478336e+00],
                 [ -2.14990107e+00,   6.68253943e-01,  -3.94993627e+00],
                 [ -2.29829600e+00,  -2.79315880e+00,   1.39679219e+00],
                 [ -7.31127444e+00,  -6.84336432e-01,   9.49089231e-01],
                 [ -7.30569184e-01,   1.50119118e+00,  -1.60139350e+00],
                 [  2.16079205e+00,   1.08359430e+01,   2.46104889e+00],
                 [  6.94800425e+00,   6.99923778e+00,  -9.13138444e+00],
                 [  5.02374949e-01,   3.33045186e-01,   1.66714301e+00],
                 [  4.50259286e+00,  -3.42534455e+00,  -3.35568215e+00],
                 [  2.96247038e+00,   3.72710446e+00,  -6.95587139e-01],
                 [  3.81786236e-01,   7.61004095e-01,  -3.89024646e-01],
                 [  6.07974573e-01,   1.58116267e-02,   1.15545392e+00],
                 [ -3.26943106e-01,  -1.99236914e+00,   5.27521394e+00],
                 [ -3.10810858e+00,   7.02120874e+00,   1.42653121e-01],
                 [  9.34649954e+00,  -1.07934704e+01,   7.48769123e-01],
                 [  2.74797487e+00,   6.68418127e+00,   5.95077702e+00],
                 [  7.10301423e+00,  -2.73358042e+00,  -3.16867901e+00],
                 [ -9.96037499e+00,  -8.98768827e-01,  -5.78627429e+00],
                 [ -1.06858111e+01,   6.74264318e+00,   5.33033968e+00],
                 [ -2.99928616e+00,  -2.48738629e+00,   6.63185040e+00],
                 [  1.67094018e+00,   1.49309350e+00,   1.48196296e+00],
                 [  5.01223886e+00,   4.22555171e+00,  -4.65171236e+00],
                 [  2.12932525e+00,  -5.20511857e-02,  -7.49610450e-01],
                 [ -6.69604599e+00,  -1.33913837e+01,   3.53002796e-01],
                 [  1.09590323e+01,  -4.55930348e+00,  -5.65011999e+00],
                 [ -1.94687840e+00,  -5.77685202e-01,  -7.17751805e-01],
                 [  1.44344801e+00,   1.53864741e+00,  -3.19717755e+00],
                 [ -2.25467337e+00,   2.43634738e-03,   3.22604462e+00],
                 [  4.62650606e-01,   9.06183996e-01,   2.36556242e+00],
                 [ -1.32952845e+00,   2.08201157e-01,   8.38799650e-01],
                 [ -6.14143907e+00,  -1.20344005e+00,  -8.73508550e+00],
                 [  5.05243758e+00,  -5.43536741e+00,  -9.72757531e+00],
                 [ -7.98983010e+00,   5.29960563e+00,  -1.26928708e+01],
                 [ -1.35015867e+01,   3.25097645e+00,   4.17480316e+00],
                 [ -6.76582160e+00,   7.64613725e+00,  -1.10606550e+00],
                 [  8.39041379e+00,  -1.92810043e+00,  -4.32034962e+00],
                 [ -5.47610093e+00,   2.18551839e+00,   1.02293860e+01],
                 [  1.01346272e+01,  -7.86690263e+00,   5.27735842e+00],
                 [  2.16525156e+00,  -3.30218649e+00,  -7.70586313e-02],
                 [ -2.28361914e+00,   7.26059291e+00,  -1.27442340e+00],
                 [  9.34723095e-01,  -2.67035432e+00,   1.09661840e+00],
                 [ -8.36454984e-01,   6.74794532e+00,  -8.14933343e+00],
                 [ -1.02241238e+01,   1.09645028e+01,  -2.56015827e-01],
                 [  3.54480716e+00,  -2.87887007e-01,  -1.29112422e+00],
                 [  7.47260469e+00,  -3.22832569e+00,  -3.95969375e+00],
                 [ -1.92532603e+00,  -1.89099643e+00,   3.18794656e-01],
                 [  2.28296070e+00,   3.52196900e-01,  -7.48203374e-01],
                 [ -4.98764963e+00,   2.02724592e+00,  -8.22974473e-01],
                 [ -2.96002524e+00,   6.22020223e+00,   7.58771840e-01],
                 [ -3.02051408e+00,  -8.15365536e+00,   1.99162186e+00],
                 [  7.64290006e+00,   9.64378666e+00,  -8.47343275e-01],
                 [ -1.00639687e+01,   1.62281540e+00,  -3.08320630e+00],
                 [ -1.41943182e+01,  -3.65140112e-01,  -7.23474255e-01],
                 [  4.57117665e+00,  -1.09205162e+01,   5.99360380e+00],
                 [  6.62489270e+00,   8.74204745e+00,   3.48657268e+00],
                 [  2.15057441e+00,  -1.44462077e+00,  -1.02139744e+01],
                 [  4.88149968e-02,   1.02108262e-01,   2.14158499e+00],
                 [  6.08241183e+00,  -8.00960319e-01,   1.30468709e+01],
                 [  1.42669924e+00,   8.14499486e-01,  -2.07410466e+00],
                 [  5.77739617e+00,   9.63346390e+00,  -9.49128249e+00],
                 [  1.20344367e+01,   3.26766746e+00,  -3.03491919e+00],
                 [  5.41997199e+00,   7.43207741e-01,   1.35865436e+00],
                 [  1.72912247e+00,   7.33471777e+00,   4.68850427e+00],
                 [ -8.20006124e-01,  -1.29004576e+00,  -2.10393399e+00],
                 [ -9.86517036e-01,  -7.06225658e-01,  -3.73813423e+00],
                 [  5.40243658e-01,   1.56940364e+00,  -6.24170031e-01],
                 [  7.83069006e+00,   3.48434706e+00,  -7.82213340e+00],
                 [ -6.05713810e+00,   5.35710848e+00,  -1.38470795e+00],
                 [  1.39260722e+00,   6.59048990e-01,  -1.70311953e+00],
                 [ -5.95994017e+00,   1.85840083e+00,  -9.78407762e+00],
                 [ -5.86185573e+00,  -7.45860525e+00,  -1.10069067e+00],
                 [ -6.60583321e+00,  -1.26769722e+01,  -6.26998596e-01],
                 [ -3.22573802e+00,   9.88040847e+00,  -4.30331837e-01],
                 [  9.22537250e+00,   5.26217811e+00,   6.50803364e+00],
                 [  9.41469258e-01,  -1.41765667e+00,   1.79628273e+00],
                 [ -1.10719826e+01,  -1.36983841e+00,  -9.88822793e+00],
                 [  2.32072569e+00,  -2.13404807e+00,   1.75479638e+00],
                 [  5.66223433e+00,  -1.28832682e+01,  -2.34704746e+00],
                 [  3.31707510e+00,   5.88565228e+00,   4.92687262e+00],
                 [ -2.66570911e-01,   2.04879858e+00,  -4.02216647e-02],
                 [ -4.93609389e+00,  -2.27309182e+00,  -3.85419647e+00],
                 [  1.60278026e+00,   2.78246837e+00,  -5.12570870e-01],
                 [ -1.45521237e+00,   6.15548358e-01,   1.45063888e+00],
                 [ -2.47598776e+00,  -2.63816039e-01,  -1.84326246e+00],
                 [ -1.14976541e+00,   6.35730698e-01,   1.17079314e+01],
                 [  3.27822000e+00,   3.70046455e+00,  -3.86654708e+00],
                 [  4.11805203e+00,  -1.17409683e+01,  -1.42054336e+00],
                 [  3.10987863e+00,  -6.12574300e+00,  -9.41779117e-01],
                 [ -3.63731948e-02,  -8.65922641e+00,  -9.03009271e+00],
                 [ -6.69651142e+00,  -2.22437689e+00,  -5.84788142e+00],
                 [ -7.10266454e+00,  -1.97668566e+00,   1.96458423e-01],
                 [ -1.10944749e+01,   2.12622819e+00,   1.91872469e-01],
                 [ -1.90366670e+00,   5.64371337e-01,   2.51736569e+00],
                 [  4.61820693e+00,  -5.20272370e-01,   1.80182545e+00],
                 [  4.83303337e-01,  -1.91336680e+00,   3.24714032e+00],
                 [ -1.70952644e+00,   2.23388310e+00,   9.11533742e+00],
                 [  5.82056416e+00,   1.47454008e+00,   3.17216533e+00],
                 [ -7.10442118e-01,  -5.23619371e-01,  -1.72430483e+00],
                 [ -2.94750094e+00,   6.63665950e+00,  -8.55380920e+00],
                 [ -7.74785634e-01,   2.92560769e+00,  -2.59576619e+00],
                 [  1.60139029e+00,  -3.08454875e+00,   6.77275260e-01],
                 [ -9.41625206e-03,   3.25860865e+00,  -2.52095667e+00],
                 [ -1.75512903e-01,  -1.18115211e-01,  -1.08481677e+01],
                 [  5.73503392e+00,   6.77209942e+00,   6.45757705e+00],
                 [  6.84349083e+00,  -4.92914223e-01,   1.55813590e+00],
                 [  2.72354043e+00,  -4.04745565e-01,   5.00583825e-01],
                 [  1.42364376e+00,  -5.38438096e+00,   1.15983589e+01],
                 [ -4.83239463e+00,  -5.35752114e+00,   3.43572655e-01],
                 [  4.72085504e+00,  -1.00853831e+01,  -2.32219797e+00],
                 [  6.36367684e+00,   5.33489392e-01,  -5.24344561e-01],
                 [ -2.41275032e+00,   3.55123130e-01,  -8.83631474e-02],
                 [ -1.90458832e+00,  -5.63593328e+00,   6.10660755e+00],
                 [  3.66885349e+00,   1.17994497e+00,  -3.44338974e+00],
                 [  7.08716987e+00,   6.95634320e+00,   1.50186323e+01],
                 [  5.65510075e+00,   9.93937052e+00,  -6.20023054e+00],
                 [ -1.02558474e-02,  -1.15375968e+00,   7.40227689e-01],
                 [ -1.44308098e+00,  -1.26363413e+00,   4.44559794e+00],
                 [ -6.08592034e-01,   1.59568831e+00,   2.44408684e+00],
                 [ -4.04410371e-01,  -2.33397700e+00,  -2.21287003e+00],
                 [ -4.00248792e-01,   1.86340234e+00,  -2.13493851e+00],
                 [ -4.12306246e+00,   4.14937726e+00,   8.84134236e+00],
                 [  4.01631431e+00,   4.45532687e+00,   9.88706086e+00],
                 [ -1.42117999e+00,   5.04349932e+00,   8.00189142e+00],
                 [  6.39415740e+00,   2.24875952e+00,  -3.37746990e+00],
                 [  7.59443091e+00,  -2.95413106e+00,   1.13016589e+00],
                 [  1.98918875e+00,  -8.47353105e+00,   4.66663636e+00],
                 [  2.56316829e+00,  -1.47814043e+00,  -1.40226432e+00],
                 [  1.57103658e+00,   5.85011570e+00,   1.80957182e+00],
                 [  9.81804029e-01,  -1.06119693e+00,   7.65489598e-02],
                 [ -1.26564461e+01,   2.61626858e+00,  -5.36123301e-01],
                 [  1.72843677e+00,  -1.39541063e+00,  -9.44047868e-01],
                 [ -1.09950336e+01,  -2.99596917e+00,   4.52562253e+00],
                 [ -1.27692433e+01,  -1.35025909e+00,  -2.55367190e+00],
                 [  1.64146404e-01,   2.42526984e+00,   3.45871275e-01],
                 [  3.41477362e+00,  -8.07638166e+00,  -7.78252163e+00],
                 [ -2.26207311e+00,   3.10762177e-01,   2.90389522e-01],
                 [  2.52682617e-01,  -6.59428640e-01,  -7.94017385e+00],
                 [ -2.93408142e+00,  -1.01024830e+00,  -7.55999495e-01],
                 [ -1.29751594e+00,   5.88192679e-01,  -1.07851501e+01],
                 [ -2.90547820e+00,  -7.75991892e+00,  -4.01401440e+00],
                 [  9.75911816e-01,   9.28074905e+00,  -1.54333984e+00],
                 [  2.13008701e+00,  -1.03064494e+00,   7.48059595e+00],
                 [ -4.73217539e+00,   4.82948376e+00,   1.52856349e+00],
                 [  1.23759563e+01,  -6.60747023e+00,   5.17104703e+00],
                 [  6.36749043e+00,   5.54766221e+00,  -4.23753747e-01],
                 [  1.33115229e+00,  -4.35409863e+00,   1.34580424e+01],
                 [  1.80676573e+00,  -3.65069363e+00,   3.54840250e-02],
                 [  1.00083884e+01,  -6.74874220e-01,   3.67202116e+00],
                 [ -3.13938001e+00,   1.54551839e-01,  -1.73458325e+00],
                 [ -4.96564264e+00,  -1.45330569e+00,  -5.49893677e+00],
                 [  8.47996377e+00,   7.76931542e+00,   3.85245278e+00],
                 [ -9.59076782e-01,  -9.44323734e-01,   2.20141033e+00],
                 [  1.06091143e+01,   2.57396926e+00,   2.98967613e+00],
                 [  1.36037022e+00,  -3.00587711e+00,   2.56258988e+00],
                 [ -2.73991634e+00,   2.03642799e+00,   2.43712850e+00],
                 [  1.17745654e+00,  -9.39575827e-01,  -1.88980811e+00],
                 [ -1.56097211e+00,   1.14473679e+01,  -5.52615695e+00],
                 [  5.56822537e+00,  -9.68439481e-02,   7.49765661e+00],
                 [ -6.89972535e+00,  -2.10357079e+00,   4.86606760e+00],
                 [ -2.71440557e+00,  -6.81516347e+00,   2.87933864e+00],
                 [ -1.25079619e+00,   8.90439318e+00,   8.95824427e+00],
                 [ -7.51095391e+00,   2.08585052e+00,  -9.89533107e+00],
                 [ -2.68381819e+00,   7.56623161e+00,  -3.96927563e+00],
                 [  1.38503943e+00,   1.06211946e+00,   3.55926994e+00],
                 [ -1.53346889e+00,  -9.86914354e-01,   1.21279261e+00],
                 [ -1.28091254e+00,   8.40411296e+00,  -1.08071707e+01],
                 [ -3.62760907e-01,   1.45634753e+00,  -1.36246509e+00],
                 [  1.53489260e+00,  -9.21763359e+00,   6.89490151e+00],
                 [  2.63190881e+00,  -1.27752866e+00,  -6.20978917e-01],
                 [  3.02406013e-01,   3.13334446e+00,   1.51282712e+00],
                 [  7.45761308e+00,  -6.95032266e+00,   2.18802121e+00],
                 [  2.04958688e+00,   1.44711954e-01,  -2.40912627e+00],
                 [ -1.16051978e+00,  -1.65039999e+00,  -4.13595891e-01],
                 [ -3.10216557e+00,  -1.42099968e+00,   1.52651024e+00],
                 [  3.87721308e+00,   1.17119192e+01,  -5.49962729e+00],
                 [ -1.33713772e+01,   2.09879211e+00,   1.07207647e+00],
                 [  5.27983421e-01,  -3.95790251e+00,  -5.73664767e-01],
                 [  1.15284717e+00,   2.62870364e+00,  -7.87863286e+00],
                 [  6.20726562e+00,   3.48216248e+00,  -1.25049697e+01],
                 [  1.10497631e+01,  -1.38409751e+00,   6.22822359e+00],
                 [  1.32934432e+01,  -3.12708898e+00,  -4.51909221e+00],
                 [  4.48354932e+00,   6.64686908e+00,   5.84069174e+00],
                 [ -6.87495374e-01,  -1.09121622e+00,  -1.01731729e+00],
                 [  1.71864250e+00,   8.65720312e-01,   2.48731896e+00],
                 [  1.91079610e-01,   3.41000766e-01,  -2.43316171e+00],
                 [ -8.67667945e+00,   4.06821720e+00,   6.13296813e+00],
                 [ -4.64373801e+00,  -8.44450984e+00,   8.81880082e+00],
                 [  2.05332914e+00,   1.68980831e+00,   1.52351362e+00],
                 [  2.16182653e+00,  -9.61196625e-02,   4.76442952e-01],
                 [  9.21156608e-01,   4.92190220e-03,   3.00766210e+00],
                 [  2.69997334e+00,   3.92578086e+00,   4.09975209e-01],
                 [ -3.28805398e+00,  -2.30366592e+00,   7.35284447e-01],
                 [  3.39444704e+00,   6.79039001e-01,  -7.34431156e+00],
                 [ -3.23470905e+00,  -8.92116703e+00,  -6.07914255e+00],
                 [ -3.28725090e-01,   6.37641622e-01,   2.61768672e+00],
                 [ -1.81336576e-01,  -2.37033984e+00,  -5.12674771e+00],
                 [  1.01277716e+01,   2.18715064e-01,  -1.49670748e+00],
                 [ -6.84472471e+00,   7.21423408e+00,   9.48138661e-01],
                 [ -2.99344908e-01,  -1.52729241e-01,   9.02307573e+00],
                 [  9.41440453e+00,  -5.95304887e+00,   2.19250771e+00],
                 [  7.26407755e-01,   2.73831610e+00,  -3.47785207e-01],
                 [  5.68857605e-01,   4.95208620e+00,   9.95346514e+00],
                 [ -8.76967173e-01,   1.83129686e+00,   2.32921511e+00],
                 [ -1.10207576e+01,  -4.59580836e-01,  -5.17675735e+00],
                 [ -5.09628185e+00,  -9.32885649e+00,   3.16358973e+00],
                 [  1.00193030e+00,   2.03304211e+00,  -3.19950468e+00],
                 [  8.70169183e+00,  -8.77653942e+00,   7.53352095e+00],
                 [  2.24253788e+00,   1.42621457e+00,  -7.59072341e-01],
                 [  1.89039533e+00,   1.56342534e+00,  -1.01500773e-01],
                 [  3.44043684e+00,  -1.14478949e+00,  -9.76878222e-01],
                 [  9.09956865e-01,   1.79905455e+00,  -1.11546387e+00],
                 [ -2.86890921e-02,   1.05651263e+01,  -8.37738999e-01],
                 [ -4.31098308e+00,  -8.58674571e+00,   1.28956810e+01],
                 [ -4.46698102e+00,  -3.29597011e+00,  -9.63669552e+00],
                 [  1.03282485e+01,   6.72051008e-01,  -3.93659496e+00],
                 [  2.52004596e+00,  -6.41927358e+00,   1.88353531e+00],
                 [ -8.03252764e+00,  -2.17648077e-01,   6.19461416e+00],
                 [  2.73354031e+00,   5.93004218e-01,   4.66584310e+00],
                 [  6.42863077e-01,  -6.30714761e-01,   2.60378882e+00],
                 [  7.58150829e+00,   5.43639845e-01,  -8.39127157e+00],
                 [ -1.12557645e+00,  -6.51054847e-01,  -9.47357361e-01],
                 [  2.21885441e+00,  -1.32062178e+01,   5.82166167e+00],
                 [ -8.80657002e+00,  -5.25634953e+00,  -7.52707283e+00],
                 [  4.16438820e+00,   1.64569427e+00,   1.84641291e+00],
                 [ -4.82552439e+00,  -1.80615822e+00,   2.25731028e+00],
                 [ -3.51117596e-01,  -4.92978287e-01,   2.19861909e+00],
                 [ -3.38202246e-02,  -2.50150480e+00,  -2.53738267e+00],
                 [  2.07006909e+00,   2.82799063e+00,  -4.20042305e+00],
                 [  5.99141282e+00,   2.04327629e+00,  -9.69776335e+00],
                 [ -1.87027780e+00,  -9.59143599e+00,  -3.94151433e+00],
                 [  4.23762422e+00,  -3.15063262e+00,   1.71443911e+00],
                 [  5.14792645e+00,  -1.20694019e+01,   4.66953492e+00],
                 [  3.88375277e+00,  -1.05823884e+01,  -9.51663545e-01],
                 [  1.03173891e+00,  -4.90411474e+00,  -1.07839333e+01],
                 [  4.81072814e+00,  -5.76116684e+00,  -1.10569465e+01],
                 [ -1.19129561e+01,   1.14339316e+00,   1.77931115e+00],
                 [ -1.76250756e+00,   6.48220119e-01,  -1.46682044e+00],
                 [  5.99798607e+00,  -2.90301792e+00,   5.15024027e+00],
                 [ -1.01725872e+00,   1.88264079e-01,  -3.36875264e-01],
                 [ -8.64357907e-01,   3.74161565e+00,  -6.60122140e+00],
                 [ -7.88990729e+00,  -9.24310794e+00,  -1.90939250e+00],
                 [ -3.13885705e+00,   4.43574658e+00,   1.09845394e+00],
                 [ -5.43958268e-01,  -9.85050929e+00,   3.36980741e+00]]).T, FortranArray([[ -7.07779245e+03,   5.30957603e+01,   4.92534567e+01],
                 [  5.30957603e+01,  -7.35692662e+03,   4.91391846e+00],
                 [  4.92534567e+01,   4.91391846e+00,  -7.24629732e+03]]), FortranArray([-34.20441181, -36.11914502, -35.86923811, -11.32892774,
                 -11.58417857, -12.19608218, -11.44407094, -12.08422206,
                 -12.04252822, -10.10435147, -12.63973715, -35.55371483,
                 -12.02983904, -34.2437527 , -13.64052149, -11.59590519,
                 -35.60071395, -12.17933889, -37.17701538, -36.30187584,
                 -35.46095231, -11.65235944, -11.93140498, -12.30988255,
                 -11.51617732, -11.68182037, -11.93997317, -11.87529396,
                 -11.81875035, -34.9797412 , -12.52638293, -35.65229321,
                 -12.67235206, -11.9165326 , -37.59321411, -11.9479405 ,
                 -35.70127765, -35.63533238, -35.81499979, -11.60351249,
                 -11.5950533 , -11.97559554, -11.95042156, -11.9048626 ,
                 -12.25963624, -11.68779381, -11.94457786, -36.19215894,
                 -12.43951015, -35.78969265, -12.04455238, -11.5606028 ,
                 -35.15718853, -12.10232307, -31.88345441, -36.7512163 ,
                 -35.23106987, -12.28181662, -11.99302631, -12.23974759,
                 -12.65413915, -12.7163347 , -12.20816799, -11.85132215,
                 -12.25020357, -35.66645206, -11.69648751, -37.24755921,
                 -12.00985909, -12.59913876, -33.84096648, -11.92847822,
                 -36.46061536, -35.92239277, -36.2731673 , -12.19051669,
                 -12.26454783, -11.88675659, -11.77446101, -12.38759526,
                 -11.38926927, -12.86799396, -12.47568033, -33.80562241,
                 -12.33948154, -35.88118993, -11.49063464, -11.6197523 ,
                 -36.33198424, -11.86086036, -35.34870166, -35.10708206,
                 -34.02914678, -10.28485711, -12.77375217, -11.82592826,
                 -11.6336728 , -12.14343204, -12.39189235, -12.19744252,
                 -12.07582549, -36.65520494, -12.40550141, -34.41189856,
                 -11.77356544, -11.06165875, -36.00395034, -11.71149538,
                 -36.15998925, -36.19274062, -36.44064423, -11.97348016,
                 -11.27973619, -11.18608398, -10.82483171, -12.53694574,
                 -12.01171457, -11.75439814, -11.73038612, -35.05224379,
                 -12.65975971, -36.73357897, -12.32304191, -11.83039717,
                 -35.77788124, -12.01696986, -36.41232896, -33.9822649 ,
                 -36.18679634, -11.64380359, -12.52365348, -12.66770131,
                 -12.42929331, -12.15070085, -11.52549709, -11.8127946 ,
                 -11.22228343, -37.34871504, -11.1895854 , -36.12545002,
                 -11.82403184, -12.19008589, -35.29667676, -11.79670055,
                 -35.55876917, -35.77649989, -35.00762925, -12.71084032,
                 -12.68068906, -11.84302825, -12.88208305, -12.25142263,
                 -12.26575128, -12.00059024, -12.63967079, -34.27414952,
                 -12.63712087, -35.1175862 , -12.01840299, -12.29252072,
                 -35.31093579, -12.70115264, -35.83146404, -36.73688965,
                 -35.39745761, -12.93924596, -11.71531052, -13.27412084,
                 -13.17748689, -12.5231803 , -12.12779152, -12.28171953,
                 -11.69465321, -35.64939484, -11.82192811, -35.32759206,
                 -12.19656797, -12.22048236, -34.40143439, -12.48026057,
                 -31.76449931, -35.16807228, -36.59258649, -12.68387667,
                 -11.69386981, -12.2432995 , -11.83364303, -13.08821237,
                 -12.47694076, -12.32669595, -12.18141497, -35.29573214,
                 -12.74481041, -34.99851002, -12.19718401,  -9.87999327,
                 -35.17872658, -11.56445264, -34.83036088, -37.95029575,
                 -38.31826572, -11.85754799, -11.38225341, -11.77958077,
                 -11.30271726,  -9.93885849, -11.8044717 , -10.93400806,
                 -12.12401164, -36.52112742, -11.99481134, -35.64563294,
                 -12.45470214, -11.89376919, -34.41444505, -12.3748044 ,
                 -33.27705615, -35.42854056, -37.19099904, -12.69626491,
                 -11.83551417, -11.9192392 , -12.48445199, -12.3841356 ,
                 -11.9356134 , -11.32542723, -12.10060201, -34.90039683,
                 -11.319152  , -36.06935802, -11.85612755, -12.31257481,
                 -36.56684406, -11.71912946, -35.17166276, -37.24732569,
                 -36.41428059, -12.51656178, -12.02875286, -11.7086833 ,
                 -11.89609181, -11.70034205, -12.11137616, -11.59531397,
                 -11.75267375, -36.25275459, -11.35988023, -36.52665688,
                 -12.51417551, -12.06286623, -36.82989694, -11.19172764,
                 -33.3108042 , -35.7592875 , -35.04536962, -12.17428273,
                 -11.9281688 , -11.24628499, -12.28075543, -10.40669223,
                 -12.56045421, -12.10593843, -12.19433959, -36.5459669 ,
                 -12.55670072, -35.86262239, -12.44939923, -12.49173434,
                 -36.70087863, -11.80920488, -32.1823639 , -35.27280159,
                 -35.85594549, -10.89641299, -11.70543585, -12.97867401,
                 -14.34296726, -11.67704629, -12.03378108, -12.25166113,
                 -12.02938357, -35.57261063, -12.49383098, -34.1092973 ,
                 -11.97706306, -12.03174639, -35.11813656, -12.13053699,
                 -35.61406699, -36.46505135, -36.31652464, -11.69978274,
                 -12.18721493, -12.12186861, -12.50159015, -10.93506019,
                 -12.38728664, -12.44839768, -11.99190522, -38.06602268,
                 -12.28051059, -35.86239912, -12.03236244, -12.0458173 ,
                 -35.17611606, -12.22371301, -35.57828024, -35.90361296,
                 -35.28050374, -12.32064995, -11.81911785, -10.71890976,
                 -11.69283991, -11.80870372, -11.08202582, -12.41226875,
                 -11.9270683 , -35.47340065, -12.33580957, -36.3180329 ,
                 -11.47560549, -11.70241166, -36.09650851, -12.31324343,
                 -38.69196721, -34.71793377, -35.86161954, -12.45507124,
                 -12.21855111, -11.28804971, -11.28105975, -12.25729043,
                 -10.92982947, -11.8756918 , -11.96848151, -36.14250098,
                 -11.60251115, -36.26889293, -11.72505292, -12.18289852,
                 -35.58797478, -12.36095388, -37.18149534, -34.86155224,
                 -37.12387088, -11.53752433, -11.63598199, -11.68800815,
                 -11.43708384, -11.53173063, -12.12233667, -11.539117  ,
                 -11.97083313, -35.84234138, -12.43902242, -35.89239812,
                 -11.78476885, -11.66244754, -37.34213908, -12.389248  ,
                 -35.79768353, -35.44097219, -35.28788666, -11.98776197,
                 -12.50874802, -11.83370538, -12.34214736, -12.24724184,
                 -12.44238334, -11.83550207, -12.3438814 , -34.64671468,
                 -12.21707831, -36.26146763, -12.16451387, -12.12142645,
                 -32.48376725, -11.911959  , -34.44355479, -35.28998197,
                 -34.35945834, -12.50892274, -12.65096039, -12.12350863,
                 -12.22944551, -12.39061145, -12.18220623, -11.38382073,
                 -12.0800385 , -35.49174908, -12.09790682, -35.48457923,
                 -12.76360485, -12.11082453, -35.04431592, -12.43037448,
                 -34.69899983, -35.96593922, -39.39020738, -12.14937794,
                 -10.29183458, -12.4942097 , -12.66041593, -12.77773713,
                 -11.48041838, -12.06240049, -12.34149859, -36.80013731,
                 -11.10514592, -35.33824334, -11.98422364, -12.19964858,
                 -33.06717242, -11.11471413, -35.20079581, -35.25834099,
                 -34.90883852, -12.76185155, -12.37974948, -13.47232048,
                 -12.45102263, -11.24219564, -12.19341305, -12.41274767,
                 -11.65931096, -34.49940653, -12.66157743, -34.58231917,
                 -11.99092562, -12.4659324 , -37.02612717, -11.91049138,
                 -36.30986833, -33.87098846, -34.87586945, -12.56385775,
                 -12.64581819, -11.8452037 , -11.72455641, -11.63571963,
                 -12.28023628, -12.59750977, -12.17133749, -35.3256986 ,
                 -12.32544078, -35.15348997, -12.65354271, -12.43923652,
                 -35.87465016, -12.94629804, -34.39090241, -34.76495413,
                 -35.52187206, -12.16072292, -11.67695554, -11.45285271,
                 -12.56335438, -11.8109176 , -12.21580396, -12.12167861,
                 -11.81601154, -36.39423296, -12.06040715, -35.38027303,
                 -12.99756221, -12.31376338, -35.57900104, -11.94768143,
                 -36.33316999, -36.65274321, -35.79796966, -11.52832508,
                 -11.7466643 , -12.41042302, -11.7864101 , -12.27886146,
                 -12.67523537, -12.22520089, -12.80614312, -35.54089702,
                 -12.10492218, -36.25529827, -12.37707966, -11.06557108,
                 -36.47395104, -11.98629928, -36.02602181, -34.74083003,
                 -35.6637114 ,  -7.9879792 , -11.3816715 , -12.21618497,
                 -12.95505748, -12.26535796, -11.68337895, -12.13862351,
                 -12.03275296, -36.58025361, -11.82149183, -33.63910704,
                 -12.47390398, -11.53360064, -35.02743981, -12.84677073,
                 -35.12772157, -35.24567579, -35.92917962, -11.75122393,
                 -12.30814265, -11.91477452, -11.88589149, -12.62788111,
                 -12.39425351, -11.72340812, -12.37883297, -35.64721807,
                 -11.93326282, -34.48281565, -13.00924768, -13.07500063,
                 -35.41140822, -12.24670271, -35.69590784, -35.87246096,
                 -36.69909579, -12.49143937, -11.78233624, -12.34485084,
                 -12.43122878, -12.2085188 , -12.15598397, -11.9542177 ,
                 -12.1819202 , -35.328548  , -12.18371721, -36.23354676,
                 -12.50962301, -12.38504414, -33.83177791, -11.85574496,
                 -36.38451019, -35.58240012, -34.8156599 , -11.78000895,
                 -11.80292622, -11.90693094, -12.28891575, -11.44891093,
                 -12.21981279, -11.42147911, -12.61313478, -35.59113068,
                 -12.19098039, -36.54442553, -11.79885041, -11.78681619,
                 -35.68134241, -11.96871765, -34.53989512, -34.12611003,
                 -36.52585107, -12.75256292, -11.70363161, -12.53166965,
                 -11.95642865, -11.78838962, -11.84503162, -11.45608965,
                 -12.05182852, -31.92550067, -12.8205017 , -36.00974556,
                 -11.72046911, -12.79581004, -35.31306143, -11.45937842,
                 -32.09420635, -36.24895638, -35.72973093, -12.34353162,
                 -12.63794276, -11.76335153, -12.38928928, -11.7754704 ,
                 -12.82681282, -11.23783884, -12.88342714, -34.8332522 ,
                 -11.86306886, -33.14743938, -13.15492724, -11.81278245,
                 -35.78905099, -12.4601453 , -34.9384327 , -32.7803722 ,
                 -34.45739338, -12.44960591, -12.22250672, -12.59049229,
                 -12.1273155 , -12.637408  , -12.92950198, -14.25996347,
                 -12.49544318, -35.53675706, -12.57548869, -36.43897493,
                 -11.77774334, -11.88353527, -35.25912163, -13.06652098,
                 -35.14110214, -34.33467299, -33.44474909, -11.66761602,
                 -13.16581744, -12.49943279, -12.46954319, -12.06077043,
                 -12.58942095, -11.66839854, -12.0269529 , -35.12437108,
                 -12.56339614, -35.76652453, -12.22916739, -11.97508207,
                 -34.79711978, -11.86533259, -35.93192578, -36.50818191,
                 -32.72663136, -12.77186374, -12.77220065, -12.83423871,
                 -11.46773172, -11.81184222, -12.12713251, -11.08502077,
                 -12.73908145, -34.62464752, -12.24264722, -34.87046494,
                 -12.51258358, -11.94685084, -35.89480239, -12.48882741,
                 -35.52269554, -35.83511191, -34.59709859, -12.16522342,
                 -12.02514098, -11.67259595, -12.09009976, -12.8740637 ,
                 -12.44929139, -12.75630083, -11.702849  , -35.61565776,
                 -12.48791615, -35.85864274, -12.262037  , -11.5677839 ,
                 -34.41727977, -12.78002279, -35.34888648, -35.69394042,
                 -33.54756586, -12.43453596, -12.34297064, -12.27860763,
                 -11.90924017, -12.02230394, -12.2753157 , -12.83801268,
                 -12.23048111, -34.47209225, -12.139587  , -32.44210941,
                 -12.41367574, -11.91812014, -35.09067334, -12.02825996,
                 -33.98216214, -34.54165596, -35.98046956, -11.67836145,
                 -11.30369207, -12.3191327 , -11.62202351, -12.24316165,
                 -12.2701367 , -11.69072513, -11.68946451, -37.17769052,
                  -7.88183788, -37.55236132, -12.81145713, -12.24224561,
                 -35.72620188, -11.91162519, -34.93015689, -35.77230956,
                 -36.61234946, -12.52842947, -12.34902917, -11.92502443,
                 -12.09309727, -11.92739298, -11.45784234, -12.11548106,
                 -12.14774875, -35.55640139, -14.34775163, -33.42081052,
                 -11.97339378, -12.12750941, -35.74720218, -12.53591287,
                 -36.54847477, -36.1422244 , -35.1262028 , -11.95307905,
                 -12.52674305, -12.45512185, -11.8317528 , -11.72831496,
                 -12.10749667, -11.89818395, -12.87676102, -35.55410058,
                 -12.62091915, -35.40569036, -11.19617101, -11.86174997,
                 -35.69218255, -12.24180938, -35.66317315, -34.71276611,
                 -35.75968778, -11.64331531, -11.95890924, -11.85063199,
                 -11.87113544, -12.09023973, -12.19366516, -12.65740856,
                 -11.85878151, -35.20571134, -12.32278784, -35.04869451,
                 -12.23406199, -12.09019897, -35.35788567, -12.23865441,
                 -35.00836707, -33.05543335, -35.36371051, -12.10536172,
                 -13.30738049, -13.05766725, -12.7721027 , -12.14808966,
                 -11.61871329, -12.07673437, -11.9957513 , -33.61659896,
                 -11.36945391, -36.21406072, -12.16781547, -12.36321586,
                 -34.41818822, -12.18149986, -34.60643962, -32.4014347 ,
                 -35.50145239, -11.57865702, -13.60362232, -11.84253684,
                 -14.65410753, -12.88020312, -11.7837745 , -13.1847539 ,
                 -13.01159635, -34.34735548, -12.30630236, -36.12327858,
                 -10.39225885, -12.22951435, -35.14168487, -11.50573931,
                 -34.53346552, -36.2158524 , -34.1174105 , -12.60337429,
                 -11.67288595, -14.36747729, -12.51179664, -11.92982579,
                 -12.29653512, -11.96113326, -12.78260679, -33.11909452,
                 -13.11000263, -35.62691737, -12.5902323 , -12.4676411 ,
                 -35.58651912, -12.50733041, -32.69650227, -35.21821033,
                 -34.91565993, -13.3301198 , -12.19274802, -12.65011861,
                 -12.84485275, -12.65469141, -12.7204872 , -12.74421905,
                 -12.44642045, -35.53924261, -12.1466693 , -35.49236402,
                 -11.4942427 , -12.21940664, -34.85706196, -11.80150147,
                 -36.09976298, -37.70679553, -36.73752118, -12.00623293,
                 -11.8409943 , -12.11292762, -11.27941336, -10.98568298,
                 -12.81060193, -10.62317341, -12.15695415, -35.48449965,
                 -12.30616909, -35.37009123, -11.57406889, -11.93505583,
                 -37.67615339, -11.96173074, -36.70379702, -35.73571254,
                 -34.64136564, -11.89806349, -12.47872766, -12.32435675,
                 -12.25163286, -11.83017974, -11.71443864, -12.68853373,
                 -11.4798028 , -35.25170718, -10.4284074 , -37.29875727,
                 -11.98401241, -11.36708122, -35.21460102, -11.93419636]).T, FortranArray([[ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.],
                 [ 0.,  0.,  0.]]).T)


         def setUp(self):
            self.xml = """<TS_params 
        betapol="0.75" 
        cutoff="20.0 18.0 18.0 18.0"
        cutoff_ms="18.0"
        cutoff_coulomb="20.0"
        tolpol="0.0005" 
        yuksmoothlength="20.0" 
        iesr="2 2 2" 
        a_ew="1e-06" 
        n_types="2" 
        gcut="0.0" 
        pred_order="2" 
        maxipol="60" 
        raggio="0.0" 
        tewald="F" 
        yukalpha="1e-06">

        <per_type_data atomic_num="8" pol="0.0" z="-1.95853" type="1" />
        <per_type_data atomic_num="14" pol="0.0" z="3.91706" type="2" />

        <per_pair_data atnum_i="8"  atnum_j="8"  C_pol="0.46009932"  D_ms="0.0" gamma_ms="11.642637" B_pol="0.87357114" R_ms="8.0465068" />
        <per_pair_data atnum_i="8"  atnum_j="14" C_pol="-1.5091142"  D_ms="0.0" gamma_ms="10.405794" B_pol="1.977039"   R_ms="4.193651"  />
        <per_pair_data atnum_i="14" atnum_j="14" C_pol="0.0"         D_ms="0.0" gamma_ms="4.5784138" B_pol="0.0"        R_ms="13.113727" />
      </TS_params>
      """
            if do_compare_p1_p2:
               self.p1 = Potential('IP ASAP', param_str=self.xml)
            self.p2 = Potential('IP TS', param_str=self.xml)
            self.cutoff = 20.0*BOHR
            self.debug = False


      class TestDipoleLongRange(QuippyTestCase, PotTestMixin):
         """ Turn off short-range terms, to test charge and dipole parts of potential only"""

         dimer_ref =  (FortranArray(-17.920706383643051), FortranArray([[ 19.45266941,   0.        ,   0.        ],
                 [-19.45266941,   0.        ,   0.        ]]).T, FortranArray([[-31.31407448,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ]]).T, FortranArray([-9.04196871, -8.87873767]), FortranArray([[ 0.        ,  0.        ,  0.        ],
                 [ 0.10953198,  0.        ,  0.        ]]).T)

         trimer_ref = (FortranArray(-33.77012116015446), FortranArray([[ 17.66745928,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [-17.66745928,   0.        ,   0.        ]]).T, FortranArray([[-56.88063925,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ]]).T, FortranArray([ -7.85260528, -18.0649106 ,  -7.85260528]), FortranArray([[-0.10314826,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [ 0.10314826,  0.        ,  0.        ]]).T)

         quartz_ref = (FortranArray(-145.00951258119775), FortranArray([[ -4.56597548e-01,   7.90850151e-01,   1.69993886e-15],
                 [ -4.56597548e-01,  -7.90850151e-01,  -1.58513935e-17],
                 [  9.13195095e-01,   2.46191320e-15,  -7.56265619e-15],
                 [  3.73701627e+00,  -6.41941407e+00,   4.19227411e+00],
                 [  3.69086753e+00,   6.44605806e+00,   4.19227411e+00],
                 [ -7.42788380e+00,  -2.66439852e-02,   4.19227411e+00],
                 [  3.73701627e+00,   6.41941407e+00,  -4.19227411e+00],
                 [ -7.42788380e+00,   2.66439852e-02,  -4.19227411e+00],
                 [  3.69086753e+00,  -6.44605806e+00,  -4.19227411e+00]]).T, FortranArray([[ -7.33343215e+01,  -5.19794384e-15,   1.59466046e-14],
                 [ -5.21772326e-15,  -7.33343215e+01,   3.55057139e-14],
                 [  1.60222203e-14,   3.55414169e-14,  -7.32106130e+01]]).T, FortranArray([-28.2876303 , -28.2876303 , -28.2876303 , -10.02443695,
                 -10.02443695, -10.02443695, -10.02443695, -10.02443695, -10.02443695]), FortranArray([[ 0.        ,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [-0.0232381 ,  0.04181159, -0.02749311],
                 [-0.02459085, -0.04103058, -0.02749311],
                 [ 0.04782895, -0.00078101, -0.02749311],
                 [-0.0232381 , -0.04181159,  0.02749311],
                 [ 0.04782895,  0.00078101,  0.02749311],
                 [-0.02459085,  0.04103058,  0.02749311]]).T)

         bigquartz_ref =  (FortranArray(-293.89596056205966), FortranArray([[ -2.08556343,  -2.53157461,   4.07065982],
                 [  3.67263959,  -0.65153177,  -0.25212019],
                 [  1.99745414,  -1.38116121,   4.08378903],
                 [  6.43626849,  -4.01107178,   4.61390629],
                 [ -1.57184014,   6.63777767,   9.11923249],
                 [ -4.98131305,  -0.551726  ,   1.85144973],
                 [  3.51302957,   9.30028   ,   0.51941844],
                 [ -8.23922897,  -0.88572301,  -8.20950357],
                 [ -0.53908634,  -8.09690393,  -6.56122849],
                 [  3.63724393,   0.95543137,  -1.65454499],
                 [ -2.55779223,  -3.62022476,   3.48268632],
                 [  0.96176958,  -4.7668353 ,  -0.09161431],
                 [  3.98097028,  -4.57664084,   4.04336359],
                 [ 10.0179143 ,   5.25617998,   2.16209757],
                 [ -8.21974883,   6.70521117,  -0.7001308 ],
                 [  0.88801158,   7.58080766,  -4.64732635],
                 [ -7.26778637,   2.39395518,  -7.36287336],
                 [  0.3570579 ,  -7.7562498 ,  -4.46726124]]).T, FortranArray([[-148.19601394,   -2.36596991,   -5.22831181],
                 [  -2.36596991, -146.31561484,   -4.64900502],
                 [  -5.22831181,   -4.64900502, -149.86176414]]).T, FortranArray([-26.2926636 , -29.6335351 , -27.89610351,  -9.39674925,
                  -9.99340869,  -8.27479928, -10.14468882, -10.23484308,
                 -11.11317274, -30.14839652, -28.72089926, -29.19998377,
                 -10.15690765, -10.87159232,  -9.46575002,  -9.96148855,
                 -12.05372921, -10.33724919]).T, FortranArray([[ 0.        ,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [-0.04004351,  0.02478763, -0.03143275],
                 [ 0.01007097, -0.04307067, -0.05899246],
                 [ 0.03190406,  0.00266745, -0.01270899],
                 [-0.02042961, -0.06111135, -0.00274793],
                 [ 0.05186195,  0.00611388,  0.05287925],
                 [ 0.00476878,  0.05182657,  0.04460869],
                 [ 0.        ,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [-0.02529707,  0.03199534, -0.02651793],
                 [-0.06687741, -0.03356669, -0.01308603],
                 [ 0.05273026, -0.04459676,  0.00453838],
                 [-0.00629478, -0.04849379,  0.03014353],
                 [ 0.04890792, -0.01452389,  0.04788236],
                 [-0.00444397,  0.05008518,  0.03009146]]).T)

         am_ref = (FortranArray(-13680.186076886164), FortranArray([[  1.33021125e+00,   1.32847826e+00,   2.14609979e+00],
                 [ -1.63234146e+00,  -7.84575326e-02,   7.87640281e-02],
                 [ -8.04056765e-02,  -1.52705552e+00,   5.45934544e+00],
                 [  6.71592182e+00,  -8.08518020e+00,   1.34119425e+01],
                 [ -7.83545717e+00,   2.91504237e+00,   6.49489385e+00],
                 [  4.37397405e+00,   9.99542805e-01,  -2.98566342e+00],
                 [  4.01624687e+00,   5.88351772e+00,  -8.19430791e+00],
                 [  1.03639113e+01,  -5.93209116e+00,   1.49448083e-01],
                 [  6.00909293e-01,   8.71935181e+00,  -6.52383428e+00],
                 [ -1.47060847e+00,  -1.20261249e+01,  -2.21370703e+00],
                 [ -4.32355172e+00,  -4.42794389e+00,  -1.14864318e+01],
                 [  1.45890175e+00,   2.73490184e+00,   1.11651881e+00],
                 [  6.42178906e+00,  -1.02528902e+01,  -4.02462147e+00],
                 [ -8.94180442e-01,   5.72259003e+00,   1.47734216e+00],
                 [ -2.48578151e+00,   2.42400136e+00,  -8.00596864e+00],
                 [  9.43196881e+00,  -2.03515120e+00,   1.45048760e+01],
                 [  1.70803054e+00,   2.84524649e-01,   5.04050367e-01],
                 [ -1.01075949e+01,  -1.87921426e+00,   2.99424935e+00],
                 [  2.30998581e+00,  -2.04826015e-01,  -1.02011689e+00],
                 [  6.89598613e-01,   1.49167624e+00,   2.68126185e-01],
                 [  1.22619282e-01,   4.43857798e-01,   1.35609456e+00],
                 [  5.17439849e+00,   7.69463107e+00,   1.92470166e+00],
                 [  1.74580685e+00,   4.61305650e+00,   5.02861994e+00],
                 [  3.73971406e-01,   3.74326016e+00,   2.70711617e+00],
                 [  4.10581460e+00,   8.21570362e+00,   1.62821773e+00],
                 [ -5.14050079e+00,  -5.16408042e+00,   3.92293172e+00],
                 [  5.53076596e+00,  -7.42698585e+00,   3.68939569e+00],
                 [ -2.29005450e+00,  -2.18489676e+00,  -1.70475534e+00],
                 [  9.27641379e+00,   4.65484448e-01,   3.14262892e+00],
                 [ -9.64228657e-02,  -1.70340193e+00,  -1.17040902e+00],
                 [ -9.01994721e+00,   6.80963145e-01,   1.86136230e-01],
                 [  5.16453106e-01,  -7.37012892e-01,   3.80329682e-01],
                 [ -1.12956861e+01,  -4.00470252e-01,   4.30705690e+00],
                 [ -9.25560608e+00,  -3.65613588e+00,   2.16962220e+00],
                 [  7.77184520e-01,   1.57796534e+00,  -1.84801438e+00],
                 [ -2.58030857e+00,   7.58504367e-01,  -1.33097618e+00],
                 [ -1.40846450e-01,   1.95791608e+00,   1.99831913e+00],
                 [ -1.06065666e+00,   3.20778431e+00,  -7.34704805e-01],
                 [ -9.83489426e-01,   7.75457487e-01,  -1.53125245e+00],
                 [ -2.49950551e+00,  -2.54102598e+00,   1.17656603e+01],
                 [  8.50900525e+00,   3.99210345e+00,   1.14671457e+01],
                 [  3.64992530e+00,   8.57822553e+00,   9.70889018e-01],
                 [ -2.93122291e+00,  -8.13569794e+00,   8.61120042e+00],
                 [ -8.81463734e+00,  -5.90319048e+00,  -3.97289634e+00],
                 [  4.78890816e+00,   5.42741833e+00,   2.47421881e+00],
                 [  2.15152417e+00,   1.07972532e+01,  -9.72111861e-02],
                 [  4.07374790e+00,  -1.52150126e+00,   2.32489387e+00],
                 [  1.24385324e+00,   1.01455417e+00,   1.58264097e+00],
                 [  6.02823247e+00,  -2.12400112e+00,  -4.70520981e-03],
                 [ -7.33593887e-01,  -5.27644585e-02,  -1.78934137e+00],
                 [ -3.22888241e+00,  -1.16050027e+01,  -2.70816111e-01],
                 [ -3.87748771e+00,   2.73969080e+00,  -1.20557920e+01],
                 [ -9.07068111e-01,  -2.33548328e+00,  -9.33078366e-01],
                 [ -8.34858650e+00,   6.08896820e+00,  -7.15720692e+00],
                 [ -2.25430345e+00,  -1.70305248e+00,  -2.82783612e+00],
                 [ -9.77295532e-01,  -3.44600975e+00,  -1.35729879e+00],
                 [ -3.55883000e+00,   4.26614375e-01,  -1.49573252e+00],
                 [  8.16424850e+00,  -4.15078850e+00,   1.44575670e+01],
                 [ -8.47604648e+00,   5.28419371e+00,  -4.26726781e+00],
                 [  5.30043104e+00,  -9.77117422e+00,  -7.07220670e+00],
                 [  1.74051106e+00,   5.49150461e+00,   3.16566131e+00],
                 [  2.25487619e+00,   7.28146827e+00,  -5.46649949e+00],
                 [  1.54677780e+00,  -5.34870679e+00,  -8.20037514e+00],
                 [ -1.15178153e+01,   4.96119216e+00,  -2.60948035e+00],
                 [  9.75840207e-01,   1.39820447e+00,   8.69405138e+00],
                 [ -2.24396975e+00,   1.57110324e+00,   3.42818002e+00],
                 [  6.15008986e+00,  -5.82401825e+00,  -4.09259109e+00],
                 [ -9.34100985e-01,   9.57265056e-01,   1.08104204e+00],
                 [ -7.86121047e+00,  -9.81170692e-01,  -4.38464075e-01],
                 [ -7.01765722e+00,   3.20164492e+00,   9.38302134e+00],
                 [ -9.13546986e-01,   9.38917566e-01,  -9.67295104e-01],
                 [  1.06915061e+01,   2.62605239e-01,   7.50042911e+00],
                 [ -2.25219134e+00,   1.58551700e+00,   1.30770649e+00],
                 [  5.82565325e-01,  -1.03388737e+00,   5.34039489e-01],
                 [  9.17381176e-02,   1.57942158e+00,   9.64222852e-01],
                 [  9.01726864e-01,   6.83976817e+00,   3.17498050e+00],
                 [ -3.08016997e+00,  -3.25899172e+00,  -1.95816481e+00],
                 [  4.82162595e+00,  -4.48916460e+00,  -1.00349386e+01],
                 [  4.60411957e+00,  -4.50120591e+00,   8.05950765e+00],
                 [  2.67291693e+00,   4.53371645e+00,  -7.03523899e+00],
                 [ -1.00765084e+01,  -6.06421058e+00,  -3.92920703e+00],
                 [  8.38866673e-01,   1.28851942e+00,  -4.09665907e+00],
                 [ -3.11700423e+00,  -6.04550282e+00,  -5.13643639e+00],
                 [ -1.12766484e+00,   3.57147316e+00,   1.06887572e-01],
                 [ -2.88338156e+00,  -9.13957839e+00,  -2.38492609e+00],
                 [  3.69776317e+00,  -2.13416617e+00,  -5.87743498e-01],
                 [ -6.87266385e+00,   1.13697845e+01,  -1.72876906e+00],
                 [  7.17481434e+00,  -4.78706844e+00,  -2.97955043e+00],
                 [ -2.07134728e+00,   3.67016864e-01,   1.13063721e+00],
                 [  6.13303379e-01,   7.78958283e+00,  -6.62430733e-01],
                 [  8.58943711e-01,  -1.41488903e+00,  -1.71715596e+00],
                 [  1.20902781e+00,   5.73479572e-01,  -1.66604009e+00],
                 [  7.57735641e-01,   2.69659910e-01,   6.19283405e-01],
                 [  7.21555474e+00,  -5.18936749e+00,  -3.10387077e+00],
                 [  7.56361730e+00,   4.17209593e+00,  -5.73223361e+00],
                 [  1.62387672e+00,  -8.09564896e+00,  -8.20833381e+00],
                 [ -8.10121545e+00,  -5.01264008e-01,  -6.87408873e+00],
                 [  2.02862308e+00,   1.84781029e+00,   8.56756249e+00],
                 [  7.80246117e-01,  -7.50939675e+00,   5.58741498e+00],
                 [ -2.74689690e-01,  -2.20442539e+00,  -1.05650260e+01],
                 [ -2.17963552e+00,  -8.36068088e-01,  -5.02277800e+00],
                 [ -3.80543173e-01,  -1.11632297e+00,  -1.50534861e+00],
                 [ -3.00181140e+00,  -1.10102472e+01,   2.87857272e+00],
                 [ -2.00236543e-01,  -4.29148531e+00,  -1.07132686e-01],
                 [ -6.05106958e+00,   1.10797690e+01,  -1.69111581e+00],
                 [ -5.00161051e+00,  -7.40328396e+00,  -1.55154449e+01],
                 [ -2.50495377e+00,  -1.22846252e+00,   3.43930519e+00],
                 [  3.98112543e+00,   8.89148137e+00,  -3.71565151e+00],
                 [ -1.90064605e+00,  -5.66285055e-01,   3.70181980e-01],
                 [  1.63355505e-01,  -3.10632865e+00,   1.24543988e+00],
                 [  6.72620722e-01,   2.04505289e+00,   1.80627704e+00],
                 [  1.64170799e+00,  -6.40008617e+00,  -7.06414290e+00],
                 [  3.88354236e+00,   1.47470621e+00,  -6.77741265e-01],
                 [  7.78288317e-02,   1.28146048e+01,   4.42626753e+00],
                 [  6.59738134e+00,   4.51508250e+00,   3.98109432e+00],
                 [ -8.05766986e+00,  -1.17339746e+00,  -4.04646361e+00],
                 [ -3.86033191e+00,   4.90074059e+00,   1.00112398e+01],
                 [ -8.44413243e+00,   6.24894779e+00,   3.68379456e+00],
                 [ -2.73096473e+00,   5.54192727e+00,   7.28713779e+00],
                 [  3.29213956e+00,   2.65230689e+00,   2.42476807e+00],
                 [  1.60357846e+00,  -1.86442298e+00,  -3.31398261e-01],
                 [  3.53218908e-01,  -2.97534212e+00,  -3.00543741e-01],
                 [  1.47993009e+00,  -5.23851256e+00,  -3.91845178e+00],
                 [  5.37562782e+00,   6.98005104e+00,   7.99119395e+00],
                 [ -6.59592958e-01,   3.13850616e-01,   3.02229625e+00],
                 [ -5.35734800e+00,   9.32342573e+00,   2.30095421e+00],
                 [ -1.07947081e+00,   2.11953468e+00,  -1.10964974e+00],
                 [  3.83727636e+00,  -1.12966536e+00,   2.22058036e+00],
                 [ -3.71741376e-01,  -1.98996930e-01,   1.80006982e+00],
                 [  2.40577872e+00,  -4.66707469e+00,  -2.17878225e-01],
                 [  8.36456002e-01,  -3.55086991e+00,  -3.94392372e-01],
                 [ -3.36292922e+00,   4.56516783e+00,  -9.06866249e+00],
                 [ -9.65009895e+00,   6.75093770e+00,   1.79379858e+00],
                 [  4.09754256e+00,  -3.95370546e+00,   4.67955598e+00],
                 [  3.95145523e+00,  -1.27033157e+01,   2.55887142e+00],
                 [  2.57808885e+00,   9.32814106e-01,  -1.14184061e+01],
                 [ -9.83646549e+00,   4.06843443e+00,  -4.60431112e-01],
                 [  3.26629542e-02,   9.43471263e-01,   8.51081934e-01],
                 [  4.73280227e+00,   4.96418265e+00,  -6.79460519e+00],
                 [  9.65936678e-01,   1.10064952e+00,  -1.01488036e+00],
                 [  6.17401104e+00,  -6.87184696e+00,   5.64054062e+00],
                 [ -1.75544443e+00,   8.58292300e-01,   4.69689072e+00],
                 [ -1.11743867e-01,   4.94531284e-01,   2.84969479e-01],
                 [  4.63943804e+00,   1.71475857e+00,  -1.08124378e+01],
                 [  4.45067499e-01,   3.69420709e-01,   1.06686263e-01],
                 [  2.33460619e+00,   1.47224456e+00,   9.88823762e-01],
                 [  5.74165993e-01,   4.83715629e-01,   2.62514020e+00],
                 [ -5.00786407e+00,  -9.28190550e+00,   1.01875301e+00],
                 [ -9.84885666e+00,   6.94186796e+00,  -2.83713908e+00],
                 [  3.15623227e+00,   1.32193648e+01,  -5.32431296e+00],
                 [ -1.12344659e+00,  -7.73064804e+00,  -2.76873540e-01],
                 [ -4.02695975e+00,   2.49140249e+00,   9.68777389e+00],
                 [ -2.55072162e+00,   9.97767679e+00,   8.39832081e+00],
                 [ -6.97087345e+00,  -6.95858786e+00,  -6.62305138e+00],
                 [  7.57317022e+00,   4.20093557e+00,   4.06951039e+00],
                 [  1.21643163e+00,  -2.90847584e+00,   1.19619392e-02],
                 [ -1.10830918e+00,  -6.37145677e+00,  -1.02633877e+01],
                 [  1.06635146e+00,   3.70359974e+00,   2.07752524e+00],
                 [ -5.93351151e+00,   7.97628559e+00,  -1.01188530e+00],
                 [  9.32616012e+00,   7.81619532e-01,  -7.65465724e-01],
                 [  8.57639269e-01,   2.04178383e-01,  -4.44764361e-02],
                 [ -1.88893729e+00,   1.36544959e+00,  -1.19160865e+00],
                 [  1.22937458e+00,   3.87234658e-01,  -8.67946670e-01],
                 [  2.04100801e+00,  -3.70621655e+00,   1.19076902e+00],
                 [ -1.25163652e+00,  -7.18532110e-01,   3.34985471e-01],
                 [  3.58638532e+00,   6.62477052e+00,  -3.43708713e+00],
                 [  3.02749282e+00,   6.14023937e+00,   3.01815974e+00],
                 [  1.15643921e+01,   2.82824389e+00,  -4.12199812e+00],
                 [  1.75546473e-01,   8.92876571e+00,   3.30948889e+00],
                 [  1.04793623e+00,  -4.73519934e+00,   1.01788163e+01],
                 [  1.75878278e+00,  -1.18881687e+01,  -1.06155761e+00],
                 [  1.43857707e+00,  -7.06967180e+00,  -2.58392076e+00],
                 [  5.36073340e+00,   4.98655765e+00,  -7.78853313e+00],
                 [  2.49179687e+00,  -3.14616974e+00,  -3.35766327e-01],
                 [  2.81977446e+00,  -4.08812928e+00,  -5.87168351e+00],
                 [  1.14127051e+00,   3.14461747e-01,  -9.38964706e-01],
                 [ -5.89180227e+00,   3.01554452e+00,   9.67019205e+00],
                 [  4.08077343e-01,   2.52463679e+00,   8.91821184e+00],
                 [  6.65577289e-01,   1.61422271e-01,  -1.84976964e+00],
                 [  6.20129741e+00,   3.36883562e+00,   6.70029873e+00],
                 [ -1.31240968e+00,   6.56513205e-01,   8.59734065e-01],
                 [ -3.45187704e-01,   6.98302789e-01,  -1.16316293e+00],
                 [ -1.32191890e+00,   4.42315007e-01,   1.50586770e+00],
                 [ -7.97164624e+00,  -1.00063406e+00,   6.90778330e-01],
                 [ -8.12138054e+00,  -9.96620616e+00,   2.45090007e-02],
                 [ -4.36745854e+00,   3.22648987e+00,  -2.36803416e+00],
                 [ -6.13105904e+00,  -8.01140779e-01,   5.67701598e+00],
                 [ -7.49503327e-02,   1.91062960e+00,   4.57839225e+00],
                 [ -6.50370916e-01,   2.43521155e+00,  -3.98787105e+00],
                 [  5.16441123e+00,  -7.41010606e+00,  -2.32083035e-01],
                 [ -2.96647259e+00,   2.06985488e+00,   2.55433112e+00],
                 [ -1.34386088e+00,  -1.29756004e+00,   1.36220943e+00],
                 [  3.07282155e-01,  -1.08377335e-01,   9.93292861e-02],
                 [ -2.21141257e+00,   2.35679159e+00,   5.59301500e-02],
                 [  7.83210394e+00,  -1.61189840e+00,   9.79228875e+00],
                 [ -7.42954120e+00,   8.62291612e+00,  -3.42431251e+00],
                 [  2.66868071e+00,   3.38543184e+00,  -2.19701996e+00],
                 [  6.05720560e-01,   8.10037157e+00,   2.62202019e+00],
                 [ -1.29586724e+00,   1.33562970e+00,  -1.01587266e+00],
                 [ -1.86960620e+00,   5.05838078e-01,  -4.09669787e-01],
                 [  2.74081406e+00,  -2.16354052e+00,   1.80285836e+00],
                 [  3.87604148e+00,   5.56031143e+00,   9.08401578e+00],
                 [  5.53134841e+00,  -7.38225510e-01,  -4.62275857e+00],
                 [  7.06959912e-01,   8.20377134e+00,  -4.00635518e+00],
                 [  8.35367506e+00,  -2.06195111e+00,   4.75046173e+00],
                 [  3.98630133e+00,   7.95206372e+00,   1.34323921e+00],
                 [ -1.60107742e+00,  -1.36221798e+01,  -4.18177826e-01],
                 [  4.52673288e-01,  -8.61132153e+00,  -4.69821843e+00],
                 [ -1.16191217e-01,  -5.84044597e+00,   2.94958617e-01],
                 [ -7.16728778e-02,   8.86396684e-01,  -1.00976205e+00],
                 [  2.51038606e+00,  -3.56251690e+00,   4.99769151e+00],
                 [  1.97596508e+00,   1.15335194e+00,   1.39260931e+00],
                 [ -3.44932421e+00,  -5.22696557e+00,   8.94661353e-01],
                 [ -1.75391947e+00,  -5.32602860e+00,  -2.97273101e+00],
                 [  2.57985014e+00,  -1.50003164e+00,  -3.25032911e+00],
                 [ -4.29047891e+00,   2.23497704e+00,  -2.75601548e+00],
                 [  1.02006717e+00,   1.94080339e+00,   1.16594612e+00],
                 [  1.43013506e+00,  -4.77006008e-01,   1.02615031e+00],
                 [ -1.43416398e+00,  -1.87871417e-01,   1.93335641e+00],
                 [  5.61863955e-01,   2.53744051e+00,  -3.46087707e+00],
                 [  1.02482655e+00,   1.03670185e+01,  -6.28406046e+00],
                 [ -3.50823131e+00,  -1.15869892e+01,  -6.07370150e+00],
                 [ -1.66203414e+00,   7.33835970e+00,  -8.38248702e+00],
                 [  5.98478899e+00,   5.86361555e+00,  -3.57383978e+00],
                 [  6.19234054e+00,   2.26045287e+00,   1.13341924e+01],
                 [  1.06900685e+00,  -1.08239581e+01,   3.36573076e+00],
                 [  5.23651833e+00,  -5.65556236e-01,   2.54803717e+00],
                 [ -3.73669492e+00,   2.18508368e-01,   7.85314626e-01],
                 [  1.22121478e-01,  -6.46857481e+00,   4.92227503e+00],
                 [ -3.45402869e+00,  -3.39223541e+00,   2.21638997e+00],
                 [  4.37249394e+00,  -1.27366506e+01,   3.22917505e-01],
                 [  9.34094684e-01,  -1.22680980e+01,  -6.01458057e+00],
                 [ -1.38295310e+00,  -1.14771443e+00,  -1.02982253e-01],
                 [ -9.07977360e+00,   7.16640799e+00,   1.49663287e+00],
                 [ -7.46607972e-01,  -1.86414020e+00,   1.32696348e+00],
                 [ -2.09865377e-01,   7.06086160e-02,   5.77287667e-01],
                 [ -2.96184785e+00,  -1.59840981e+00,  -4.76481973e-01],
                 [  1.05962055e+00,  -6.18305489e-01,   1.29090141e+00],
                 [ -7.32426946e+00,   5.72009892e-01,  -9.46453962e-01],
                 [  2.23285843e+00,   3.50961694e+00,   4.63667598e+00],
                 [ -8.88335582e+00,  -3.49418219e+00,  -6.45600535e+00],
                 [ -3.68249610e+00,  -7.82350892e+00,  -7.94588649e+00],
                 [ -3.78090188e-02,   1.20910022e+00,   1.70444258e+00],
                 [  1.84436069e+00,   1.03724784e+00,  -5.47802912e+00],
                 [ -1.35826734e+00,   7.78111157e+00,   2.96650087e+00],
                 [ -2.24240013e-01,   6.68199969e-01,  -5.88395366e-01],
                 [ -9.15430251e+00,   7.51770133e+00,  -1.07187423e+00],
                 [  3.08283344e+00,  -7.35182422e-01,   1.01854881e+00],
                 [ -7.62619815e+00,   5.26605031e+00,  -5.27192737e+00],
                 [  1.03101588e+01,  -5.03864972e+00,  -3.59900653e+00],
                 [  4.67779739e-01,  -2.20977972e+00,   1.52344117e+00],
                 [  8.98479864e+00,   2.29371462e+00,  -2.02695617e-01],
                 [  1.24764131e+00,   1.59327738e+00,   9.24645632e-01],
                 [ -2.59300290e-01,   2.98138016e-01,   5.98283580e-01],
                 [ -1.23179465e+00,   5.10813769e-01,   2.78014213e+00],
                 [ -9.26089259e-02,   3.58562351e+00,   1.50291831e+00],
                 [ -1.21332723e+01,  -3.89334761e+00,   2.88765594e+00],
                 [  1.05243548e+01,   8.58210596e-01,   3.91778520e+00],
                 [  6.22941194e+00,   4.89111053e+00,  -6.81195879e+00],
                 [  6.70803979e+00,  -1.90255199e+00,  -8.27246335e+00],
                 [ -8.41633713e+00,  -3.72063290e-01,   7.01988071e+00],
                 [ -3.69798113e+00,  -2.12503231e+00,  -3.16549189e-01],
                 [  9.73924751e-01,  -1.34344291e+00,   3.44138370e+00],
                 [  9.14486740e-01,  -1.38603280e+00,   1.40238039e+00],
                 [  8.91053772e+00,  -4.59981143e+00,   2.18628408e+00],
                 [ -1.67230357e+00,  -7.40416328e-01,   6.48173283e-01],
                 [ -1.40635907e+00,   2.52538690e+00,   1.03534314e+01],
                 [ -2.70108992e+00,   2.37686173e+00,   5.43936998e-01],
                 [ -3.18576254e-01,   1.79634507e+00,   1.37965787e+00],
                 [  3.02943375e+00,   6.31595697e+00,  -7.18077522e+00],
                 [ -1.02167784e-01,   3.38458344e+00,  -2.09487782e+00],
                 [  6.65393764e-01,  -2.52007310e+00,   1.04772462e+00],
                 [  1.86306271e+00,  -2.31745592e-01,   1.54842820e+00],
                 [ -1.17532922e+01,   1.02834898e+01,  -6.19648395e+00],
                 [ -1.06704106e+01,   6.54342615e+00,  -3.42165503e+00],
                 [ -1.24473712e+00,   8.14090681e+00,   2.59151074e+00],
                 [ -2.60530664e-02,   2.47756318e+00,  -1.27002820e+00],
                 [ -2.48112496e+00,  -1.29022398e+01,  -1.67189588e+00],
                 [  2.65855561e+00,   5.04171383e+00,   1.79594667e+00],
                 [  4.84683393e+00,   8.74882278e+00,   3.49684591e+00],
                 [ -2.92321010e-01,   8.17155276e+00,   1.06533574e+00],
                 [  8.13801037e-01,  -6.15317662e-01,  -7.13645123e-01],
                 [ -6.82983738e+00,   1.70737542e+00,  -2.85006015e+00],
                 [  3.80287667e-01,  -1.20641612e+00,   2.27402737e+00],
                 [ -3.87066927e+00,  -3.16803138e+00,  -3.03623423e+00],
                 [  7.55412394e+00,  -6.65508767e+00,  -3.11247393e+00],
                 [ -1.45728330e-01,   2.06308912e+00,  -5.88934700e-01],
                 [  2.44078664e+00,  -8.61522556e+00,   8.77574861e+00],
                 [ -6.95631856e-01,   3.39764709e-02,  -9.96139912e-01],
                 [  1.39347175e+00,   6.15417023e-01,  -2.08656865e+00],
                 [ -2.72091032e+00,   9.85934995e-01,  -1.44125905e+00],
                 [ -8.10817934e+00,   8.31999262e+00,   4.77479876e-01],
                 [  1.40766850e-01,   3.62145947e+00,   1.50609971e+00],
                 [ -7.70086736e+00,   2.36642341e+00,   3.05360017e+00],
                 [  3.49471432e+00,  -6.45701668e+00,   9.17281030e-01],
                 [  9.97191312e+00,  -4.57428599e+00,   5.63957542e+00],
                 [ -1.33541689e+00,  -5.38082355e+00,  -1.78960566e+00],
                 [ -4.42909908e+00,  -8.35271763e-01,  -2.17187515e+00],
                 [  1.22640561e+00,   1.15766138e+01,  -2.06898436e+00],
                 [ -2.88294171e+00,  -2.54230166e-01,   1.26854981e-01],
                 [  4.24356982e+00,  -4.69635104e+00,   1.02534581e+00],
                 [  1.64607574e-03,   1.00065592e+00,   1.03214914e+00],
                 [ -3.31207909e+00,   1.30137918e+00,   9.56370148e+00],
                 [  5.42596405e+00,   2.39674461e-03,   2.78692295e+00],
                 [ -6.91259587e-01,  -1.78024502e+00,  -5.43252540e-01],
                 [  9.38520038e+00,  -1.26541059e-01,   7.34775049e+00],
                 [  6.51704372e-01,   2.34387048e+00,   3.08163701e-01],
                 [  1.60331857e+00,   1.13110271e+00,   2.77111309e+00],
                 [  1.30296242e+00,   2.41878658e-01,  -4.20644399e-01],
                 [ -4.22453958e+00,   7.26246762e+00,   4.03875331e+00],
                 [ -4.14567341e+00,   9.41997427e+00,  -4.90687768e+00],
                 [ -5.64137711e-01,   1.34521251e-01,  -5.03546623e+00],
                 [ -7.22412708e-01,  -5.00380674e+00,  -1.06099987e+01],
                 [  5.39059075e+00,  -6.88210461e+00,   5.18439872e+00],
                 [  2.90788225e+00,   8.01613643e+00,  -2.00390049e+00],
                 [  5.25298992e+00,   9.58057329e+00,   1.28505425e+00],
                 [ -4.97232746e+00,  -3.77811532e+00,   5.56017461e+00],
                 [ -1.74427044e+00,  -1.49414901e+00,   3.55105408e+00],
                 [  4.10035336e+00,   7.60413480e+00,  -6.20069579e+00],
                 [  1.12477052e+00,  -1.28630768e+00,  -1.25654713e-01],
                 [  4.52444720e+00,  -7.40637388e+00,   6.56826717e+00],
                 [  3.66521907e+00,   4.57574644e+00,   8.16344480e+00],
                 [  6.22328161e-01,   9.06130245e-01,  -1.23001532e+00],
                 [ -4.82064190e+00,  -8.95012420e-01,   4.33405168e+00],
                 [  2.39783401e+00,  -1.95007287e-01,   2.64381909e+00],
                 [  3.32406391e-01,  -9.50900873e-01,  -6.77070445e-01],
                 [ -1.89880919e+00,  -3.72000023e+00,  -9.30089106e-01],
                 [ -5.79728402e+00,  -1.51262669e+00,   3.46551594e+00],
                 [ -3.51282669e+00,   2.37417063e+00,   8.01597648e+00],
                 [ -6.06225703e+00,   2.26846969e+00,   3.27780719e-01],
                 [  1.08694260e+01,  -2.57103129e+00,   4.40762853e+00],
                 [ -9.98472783e-01,  -4.65720559e+00,  -5.48795709e+00],
                 [ -9.70844361e+00,  -2.13480577e+00,   5.53772010e+00],
                 [ -8.68190147e+00,   1.02458534e+00,  -5.24049139e+00],
                 [  4.87046874e+00,   4.10352904e+00,   5.06392947e+00],
                 [  2.71291530e+00,  -2.76201645e+00,   4.17768104e-01],
                 [  7.03545936e+00,   6.07336856e+00,   5.45280665e-01],
                 [  2.17177149e+00,   1.51491491e+00,   4.60591758e-01],
                 [ -7.19395526e-02,   3.04461231e+00,  -6.25395549e+00],
                 [ -1.25110531e+00,  -1.86451930e-01,   4.53867595e+00],
                 [  7.91783868e-01,   4.43557011e-01,   1.22615676e-01],
                 [  1.98390702e+00,   8.89652930e-01,  -1.66652795e+00],
                 [ -2.88619697e+00,  -5.70696430e-02,   1.29292840e-01],
                 [ -8.08862571e-01,  -1.35517259e+00,  -1.90703375e+00],
                 [  4.11780164e-01,   5.68544243e-01,  -6.03742639e-01],
                 [ -1.38440938e+00,  -1.08476655e+00,  -2.11297616e+00],
                 [ -9.76875899e+00,   6.05237721e+00,   2.45535674e+00],
                 [  4.83277986e+00,  -5.90041021e-01,  -5.65258665e+00],
                 [ -9.44913262e+00,  -8.49006075e-01,   9.07420871e+00],
                 [ -1.11281374e+01,   3.80668411e-01,   5.51266820e+00],
                 [ -3.97625037e+00,  -4.93498292e+00,  -7.01588964e+00],
                 [ -2.16833028e+00,   2.64007306e+00,   1.06273177e+01],
                 [ -3.31490937e+00,   1.00446824e+00,  -1.08455114e+01],
                 [  1.23401512e+00,   1.81723312e+00,   4.97963302e-01],
                 [  2.06320654e+00,   9.13213512e+00,  -5.28850782e+00],
                 [ -2.63763825e+00,   1.12195072e+00,  -1.46881456e+00],
                 [  7.06109639e+00,   3.44539347e+00,  -3.99837125e+00],
                 [ -4.34862075e+00,  -4.29933476e+00,   7.67673868e+00],
                 [  2.48630224e-01,  -1.12699797e-01,   2.58789706e-01],
                 [ -5.94024463e+00,  -5.88708874e+00,  -7.19774496e+00],
                 [  4.61212810e-01,  -1.67039090e+00,  -1.88208081e-01],
                 [ -1.62632766e+00,   8.89631605e-01,  -2.10057849e+00],
                 [  7.47456229e-02,   6.62607720e-01,   2.35948895e+00],
                 [ -4.20183158e+00,  -4.67768848e+00,  -1.18105197e+01],
                 [ -1.10258915e+00,  -6.76333847e+00,   9.37793797e+00],
                 [ -1.24520261e+01,   2.65987692e+00,  -1.78788637e+00],
                 [  4.07165498e+00,   5.47594954e+00,  -8.93807698e+00],
                 [ -2.33672052e+00,   2.06977065e+00,  -9.11320652e+00],
                 [  4.60980618e+00,  -1.13878609e+01,  -1.36679078e+00],
                 [  5.03162518e+00,   8.63929366e-01,   6.28088715e+00],
                 [ -7.44404931e+00,  -6.73140273e+00,  -1.64675867e+00],
                 [ -1.14085819e-01,   1.05861809e-01,  -1.38021688e+00],
                 [  5.16854149e+00,   7.70719688e-01,  -8.39938269e+00],
                 [ -1.85539170e+00,   1.98056346e+00,   3.20338204e+00],
                 [ -5.42582494e+00,   1.39233539e+01,   3.74802455e+00],
                 [ -5.71460265e+00,   1.24014951e+01,   6.13163616e+00],
                 [ -1.62365397e+00,  -3.74025845e+00,  -3.40029530e+00],
                 [ -2.70456298e+00,  -5.99186443e+00,  -9.95019213e-01],
                 [ -3.22805700e+00,  -2.68904102e+00,  -2.36372125e+00],
                 [ -5.12881879e-01,  -1.06217206e+00,  -2.34447031e+00],
                 [  4.01874549e-01,  -1.77263308e+00,   2.11289272e+00],
                 [ -1.62708368e-01,   3.55104715e+00,  -1.24506849e+01],
                 [ -2.48359596e+00,   4.25025675e+00,   8.42486141e+00],
                 [  2.05508775e+00,  -8.98304375e+00,  -6.26484965e+00],
                 [ -2.68193957e+00,  -3.23867913e+00,   1.41813824e+01],
                 [ -3.67033593e+00,   2.46148385e+00,  -2.52166326e+00],
                 [ -2.73004936e+00,  -1.29862269e+01,   1.46465635e+00],
                 [ -1.57393759e+01,   7.22271368e+00,   4.68534151e-01],
                 [ -8.05270006e+00,   3.60918588e+00,   5.15810459e+00],
                 [ -1.23065385e+00,   1.97668357e+00,  -6.30171982e-01],
                 [  9.70997565e-01,  -1.14875894e+01,   3.21837737e+00],
                 [  5.84600091e-01,   1.86084091e+00,  -1.87285075e+00],
                 [ -8.49768102e-01,  -1.99141129e-01,  -1.50558357e+00],
                 [ -7.58006336e+00,  -4.98656316e+00,  -4.92702371e+00],
                 [  5.32508366e-01,   1.53227783e+00,   9.48908288e-01],
                 [  5.96828611e+00,  -1.88934788e+00,  -7.05949134e+00],
                 [  3.87181374e-01,  -2.19617336e+00,  -3.55215543e-01],
                 [ -2.08158941e-01,  -2.58538982e+00,  -1.49083363e+00],
                 [  6.46086504e-01,  -9.43449214e-01,  -1.00869956e-01],
                 [  6.98189914e+00,   1.09767721e+01,  -1.05845268e-01],
                 [  6.47823249e+00,   4.61768244e+00,   2.94149789e+00],
                 [ -3.40423820e+00,   4.44308203e+00,  -9.31620159e+00],
                 [ -1.59903060e+00,   6.54659191e+00,  -3.15457462e+00],
                 [  1.33467687e+01,  -1.98822153e+00,  -1.62869751e-01],
                 [  4.63799011e+00,   4.75069507e+00,   1.37030238e+01],
                 [  5.69139055e+00,  -8.09037222e+00,  -3.18469864e+00],
                 [ -2.17540310e+00,  -1.04009299e+01,   4.87514992e+00],
                 [ -1.01387365e+00,   1.49746784e+00,   1.05427688e+00],
                 [ -4.21041251e+00,   3.98149072e-01,  -3.52474585e+00],
                 [ -7.27685781e-01,  -2.52351529e+00,  -5.19698301e-02],
                 [  9.39685864e+00,   4.81014158e+00,  -5.59281367e+00],
                 [  5.26040184e-01,  -6.36819352e+00,  -2.23174392e+00],
                 [ -1.75617342e+00,  -1.04056564e+00,  -3.35108570e+00],
                 [ -7.51636686e-02,   1.25467687e+01,   2.15051755e+00],
                 [  2.16760290e-01,  -1.64184902e-01,  -7.93669330e-01],
                 [ -6.88467114e-03,   2.38131429e+00,  -1.97054925e-01],
                 [ -1.01816478e+00,  -5.73489669e-01,  -2.19278500e+00],
                 [ -1.22646972e+01,   2.77129731e+00,  -9.32353827e+00],
                 [  7.89768936e-02,   1.42895773e-01,   1.36244455e+01],
                 [  4.04973373e+00,  -8.37444841e+00,   2.09772241e+00],
                 [ -7.62387608e+00,   1.39932027e+01,  -5.37291999e-01],
                 [  2.21714536e+00,  -3.25076149e+00,  -5.11714717e-01],
                 [ -2.26389716e+00,  -1.19763879e+01,  -4.91205482e+00],
                 [ -2.25167224e+00,  -4.91113509e+00,   7.28860907e+00],
                 [  1.56268303e+00,   8.49212985e+00,  -2.84920540e+00],
                 [  1.45559972e+00,  -1.95756358e+00,   1.36969119e+00],
                 [  1.27432848e+00,   1.31891367e+00,  -1.43224873e+00],
                 [ -2.34683983e+00,   1.01710802e-01,   3.42752933e-01],
                 [ -7.57731136e+00,   1.45299168e-01,   2.93250142e+00],
                 [ -2.41220688e+00,  -1.43010986e+00,   1.06882749e+01],
                 [ -8.28492578e-01,  -4.29820937e-01,   1.00317075e+00],
                 [  6.57530514e+00,  -5.19182181e+00,   9.82138032e-01],
                 [  1.43990475e+00,   7.24164888e-01,  -6.30143140e-01],
                 [ -1.52762140e+00,   1.30092577e+00,   9.04970152e-01],
                 [  9.95606108e-01,  -1.63609184e+00,   1.04004647e+00],
                 [  1.22888501e+00,  -9.03124400e+00,  -3.36747324e+00],
                 [  1.19744256e+00,  -1.21376229e+01,  -2.62797699e+00],
                 [  3.24722988e+00,   5.42528983e+00,  -6.15159787e+00],
                 [  4.65285627e+00,   1.09081781e+00,   1.38403395e+01],
                 [  4.89047407e+00,   5.52991909e+00,  -1.16830188e+01],
                 [  2.01424253e+00,   2.48534589e+00,   1.16976602e+00],
                 [ -3.43960277e+00,  -8.63386043e-01,   3.43665439e+00],
                 [  9.13122540e+00,  -2.61472343e+00,   4.58523606e+00],
                 [ -3.77576815e+00,  -3.12245495e-01,   1.68441044e+00],
                 [  1.02032737e+01,   2.24189417e+00,  -5.15331527e+00],
                 [  1.55796095e+00,   2.15588804e+00,   1.28611875e+00],
                 [ -7.08822133e+00,   2.99047126e+00,  -3.10531373e+00],
                 [ -3.32249034e+00,  -2.79677497e+00,   2.35627172e+00],
                 [  1.23098439e+00,  -1.89003067e+00,   2.43051342e+00],
                 [  6.29673389e+00,  -9.51247471e+00,  -6.94773979e-01],
                 [  1.91754036e+00,   2.53200573e+00,   2.15390366e+00],
                 [ -2.14170050e+00,   7.98931726e-01,  -2.88865907e+00],
                 [  3.66475903e-01,  -4.99373573e-01,   2.39515685e-01],
                 [ -7.73161304e-01,   5.49242315e-01,  -6.40173769e-01],
                 [ -7.97634923e+00,   5.52243389e+00,  -6.42167646e+00],
                 [  1.65171967e+01,  -7.50366578e+00,  -1.46786228e+00],
                 [ -2.30084187e+00,   3.66609249e+00,   2.99882064e-01],
                 [ -8.63362292e+00,   9.33065190e+00,   3.05910317e-01],
                 [  1.37551217e-01,  -3.48272380e-01,  -1.30481810e+01],
                 [ -1.22753310e+00,  -9.11737759e+00,   7.12985673e+00],
                 [ -9.00253101e+00,   1.22695237e+00,  -7.59328457e+00],
                 [  8.86954245e-01,  -8.25104168e-01,  -1.15260624e+00],
                 [  8.55578344e+00,  -7.68084735e+00,   5.08653925e+00],
                 [ -1.69580362e+00,   1.29845249e+00,  -8.74092332e-01],
                 [  4.40198809e-01,  -7.61329892e+00,  -3.57666941e+00],
                 [ -7.08307998e+00,   1.47869859e+01,   4.57818964e+00],
                 [  1.60393989e+00,   8.21997179e-01,   1.42125878e+00],
                 [  5.54704927e+00,  -3.13859029e+00,  -9.76313157e+00],
                 [ -1.47397748e+00,  -1.04297417e+00,   6.45658670e-01],
                 [  1.20905554e-01,  -2.05011868e+00,  -9.53254237e-01],
                 [  7.86058320e-01,  -1.69636231e+00,   5.54250675e-01],
                 [  7.40707535e+00,  -5.29204889e-01,   7.56296179e+00],
                 [  2.41394363e+00,   7.67985831e+00,   2.29475118e+00],
                 [ -1.95819648e+00,   6.20301253e+00,  -3.94151599e+00],
                 [ -7.89920788e-01,  -1.08571360e+01,  -3.58352375e+00],
                 [  3.05777034e+00,  -4.08985427e+00,   3.31123036e+00],
                 [  7.27185268e+00,  -2.08600536e+00,  -1.11557617e+00],
                 [ -4.56740187e+00,  -9.04805637e+00,  -3.30394590e+00],
                 [  1.92398078e+00,   9.89692497e-02,  -1.00951780e+01],
                 [  1.99565925e+00,   1.53110579e+00,   2.77440126e-02],
                 [ -2.13693474e+00,   9.71000729e+00,  -1.58329737e+00],
                 [  1.72050156e+00,   9.65226572e-01,   1.22583928e+00],
                 [  6.16178302e+00,   2.58991340e-01,  -1.87396684e+00],
                 [  5.61928594e+00,   2.46759695e+00,   6.11769004e+00],
                 [  4.77600342e-01,  -1.29855812e+00,   2.47176117e-01],
                 [  2.93595676e+00,   7.27027515e-01,  -4.97775495e-01],
                 [  2.16230415e+00,  -2.51398824e+00,  -9.62340907e-01],
                 [ -2.77443753e+00,   6.62170501e-02,   1.79724693e-01],
                 [  6.66188976e-01,   7.57735012e-02,  -1.42275563e+00],
                 [  6.05707848e+00,   1.38824435e+01,   7.00560974e+00],
                 [ -1.06881000e+01,   6.65836795e+00,   3.69052799e-01],
                 [ -3.41197409e+00,  -1.91156516e+00,   3.68018544e+00],
                 [  7.92515437e+00,  -9.66569010e-01,   5.00322639e+00],
                 [ -1.55586344e+00,  -6.21173974e-01,  -1.57331679e+00],
                 [ -5.55506703e+00,  -4.74945299e+00,  -9.58819756e+00],
                 [  8.09800151e+00,   2.91925138e+00,  -5.79273516e+00],
                 [ -1.08089445e+00,  -6.31047961e-03,   9.39653267e+00],
                 [ -1.17654032e+00,  -1.14299255e-01,   1.04306055e-01],
                 [ -4.73148414e+00,  -8.46735295e+00,   3.37500453e+00],
                 [  3.13729972e-01,  -1.14697851e+00,  -2.99502688e+00],
                 [  5.95750821e+00,   1.28462368e+00,   4.98136723e+00],
                 [  6.13751101e+00,   1.12399475e+01,  -1.65820117e+00],
                 [  3.95712896e-01,   9.37228991e-01,   1.48341469e+00],
                 [ -1.94360225e+00,   2.68481887e+00,  -5.75919835e+00],
                 [ -2.77581001e-01,   7.23928836e-01,   4.90183623e-01],
                 [  1.93475116e+00,   2.57839538e+00,  -4.99119590e-01],
                 [ -6.57287064e-01,  -4.02240796e-01,  -7.17569187e-01],
                 [  8.06158507e+00,   4.43453168e+00,   4.36296853e+00],
                 [ -2.21777306e+00,  -7.12012708e+00,  -2.94773337e+00],
                 [  2.98588431e+00,  -8.66743826e+00,  -6.01002227e+00],
                 [  5.49916108e+00,   6.27943989e+00,   3.66227006e+00],
                 [ -5.27185260e+00,  -8.70566549e+00,  -3.74502420e+00],
                 [ -6.12470883e+00,  -5.66753563e+00,  -1.10286924e+00],
                 [  5.89925994e+00,   8.69058955e+00,  -5.51439658e+00],
                 [ -1.44049227e+00,  -1.76558078e+00,   5.62705792e+00],
                 [ -2.99040107e+00,   1.72559471e+00,  -2.83364037e-01],
                 [ -2.14260306e+00,   2.29713859e+00,  -1.19065411e+01],
                 [ -2.29486905e+00,  -1.32024060e+00,   2.44212048e+00],
                 [  4.10280005e+00,  -6.70940406e+00,  -7.97417988e+00],
                 [  7.15406816e+00,   2.77857613e+00,   1.80657610e+00],
                 [ -3.23316350e-01,   2.41816086e+00,  -7.79944976e-01],
                 [  8.44998788e+00,  -3.01158966e+00,  -1.10652635e+00],
                 [ -2.63904175e+00,   8.37880684e-01,  -4.97092101e-01],
                 [ -2.00405008e+00,   8.09983377e-02,  -8.11549514e-01],
                 [  4.22218621e-01,   1.05565573e+00,   1.10403255e+00],
                 [  3.97489534e+00,   2.53414237e+00,   1.35637482e+00],
                 [ -1.11924333e+01,   4.05199859e+00,  -5.22465450e+00],
                 [  4.29888610e+00,   2.24000706e+00,  -9.13910268e+00],
                 [ -5.93083945e-01,  -2.17997614e+00,  -3.80741579e+00],
                 [ -2.71623119e+00,  -2.91038117e+00,   4.18498796e+00],
                 [  8.82407086e+00,  -5.86049913e+00,  -5.02829237e-02],
                 [ -1.21064074e+00,  -5.51540096e+00,   9.85079423e+00],
                 [  7.74364826e-01,   8.59709785e+00,   2.15501209e+00],
                 [ -4.75523927e-02,  -9.52437460e-01,   1.88502904e+00],
                 [ -6.31999302e-01,  -3.31229651e+00,  -9.49938014e+00],
                 [  1.84861748e+00,  -2.17061933e+00,   9.70213600e-01],
                 [ -3.77336376e+00,   1.54347973e+00,   1.25655839e+00],
                 [  3.59874139e+00,   1.01387648e+00,  -2.15815068e-01],
                 [ -1.30086902e+00,  -2.01342583e+00,   1.70489411e+00],
                 [  7.52315887e+00,  -5.51603083e+00,  -1.18023252e+00],
                 [  8.18570817e-01,  -4.39368853e-01,   1.14517781e+00],
                 [  2.09020165e+00,  -6.70704385e-01,   1.27184769e+00],
                 [ -9.01590765e-02,  -6.55495734e-01,  -2.22278257e+00],
                 [  4.34602047e+00,   8.61326310e+00,  -8.15637431e+00],
                 [ -9.08529582e+00,   5.32459186e+00,   3.19951871e+00],
                 [ -7.00123368e+00,   9.19917545e-01,   1.37501238e+00],
                 [ -1.24213572e+00,  -9.47716035e-01,   2.86010975e+00],
                 [ -2.23473381e+00,  -1.00020142e+01,  -6.27018254e+00],
                 [ -3.66180427e+00,  -8.87375382e+00,   2.21551926e+00],
                 [ -4.56922307e+00,   4.80458622e+00,  -1.20015703e+01],
                 [  6.65843748e+00,  -2.72911761e+00,  -8.51242867e-01],
                 [ -3.37665156e-01,  -4.68610191e-01,   1.72853532e+00],
                 [  2.17132506e+00,  -2.98763877e+00,   7.24918532e+00],
                 [  1.95048332e+00,  -4.46936340e-01,   7.14128083e-01],
                 [  7.47188675e+00,  -3.50624286e+00,   1.04592754e+01],
                 [ -4.22821698e+00,   1.31563273e+00,  -7.21367386e+00],
                 [ -5.20727925e-01,  -1.82749004e+00,  -3.18849129e+00],
                 [  4.62408396e+00,   6.17545217e+00,   8.38264601e+00],
                 [ -1.71233229e+00,   1.10743580e+00,  -1.48575958e+00],
                 [  2.97915942e+00,   6.63110515e-01,   5.39320887e-01],
                 [  2.37841140e+00,  -4.66946587e-01,  -3.78413878e-01],
                 [ -1.94711469e+00,  -4.88086394e+00,  -4.03844698e+00],
                 [ -3.42343793e-01,  -1.19831370e+01,   3.06795934e+00],
                 [ -6.65151965e+00,  -7.69861773e+00,  -8.50409569e+00],
                 [  7.64139349e+00,  -7.62791906e+00,  -6.50492068e+00],
                 [  9.41795060e+00,  -2.24155758e+00,   8.30463034e+00],
                 [ -2.39896038e+00,  -9.46484839e+00,   9.51824945e+00],
                 [  8.94348034e+00,   6.49732733e+00,  -2.23512572e+00],
                 [ -1.01438792e+01,   4.99714844e-01,   1.53094225e+00],
                 [ -8.11441735e-01,  -2.50018587e-01,   1.33680228e+00],
                 [ -5.26229987e+00,   3.06359498e+00,  -5.12256926e-01],
                 [ -9.54466024e-01,   1.61307924e-01,  -6.82084631e-01],
                 [ -5.59041980e+00,  -3.29806840e+00,   8.30409853e+00],
                 [ -1.83345818e+00,   7.11570076e+00,  -3.59442065e+00],
                 [ -1.18936991e-01,  -7.46552420e-01,   1.10622501e+00],
                 [ -1.36929230e+01,  -3.39301059e+00,  -1.71347795e+00],
                 [ -8.43824275e-01,  -1.03199591e-01,  -1.30706614e+00],
                 [ -3.40257957e-02,  -3.30661392e+00,  -7.45459083e-01],
                 [ -7.87879400e-01,   3.46592074e+00,  -2.81590719e-01],
                 [  3.55873624e+00,   3.26366463e+00,   1.27518693e+01],
                 [ -1.88356538e+00,   3.71555986e-01,   2.50846291e+00],
                 [  1.23805375e+00,  -3.44639511e-01,   1.34972972e-01],
                 [ -9.85820369e-01,  -1.25730607e+00,   1.27946023e+01],
                 [  9.10770121e+00,   2.18454331e+00,   5.63336158e+00],
                 [ -3.94361880e+00,   7.97519745e+00,  -9.36429844e+00],
                 [  4.00729107e+00,   1.05889871e+01,  -1.75501455e+00],
                 [ -9.34905760e+00,  -4.75438100e+00,   8.18099933e-01],
                 [  8.61511569e-01,  -1.42565974e+00,   2.51785092e+00],
                 [  9.93337095e+00,   6.30036085e+00,  -3.57913684e+00],
                 [  1.10185139e+00,  -9.81937381e-01,  -3.98950527e-01],
                 [ -8.21152156e+00,   1.67745617e+00,   2.51108190e+00],
                 [  6.24705896e+00,   2.30475247e+00,   6.02917838e+00],
                 [ -6.80860056e-01,  -8.43537506e-01,   2.30577665e+00],
                 [ -3.04846184e+00,   1.71271427e+00,  -2.38072957e+00],
                 [ -3.73074672e-02,  -1.46990318e+00,  -2.15379631e+00],
                 [ -5.82293445e-01,  -2.90412843e+00,   1.41948129e+00],
                 [  2.79807946e+00,   3.41819170e+00,  -3.35693891e+00],
                 [ -2.34067690e+00,  -2.87343154e+00,   2.15453404e+00],
                 [ -1.10514325e+01,  -7.90701965e-01,  -1.85459618e+00],
                 [ -3.20252320e+00,  -5.05557341e+00,  -8.78928627e+00],
                 [  1.04276290e+01,  -9.50982304e+00,   5.79735338e+00],
                 [ -1.11338468e+00,  -4.11400648e+00,   9.02970782e+00],
                 [ -8.69374939e+00,   8.82034144e-03,   1.83071926e+00],
                 [  2.96928052e+00,  -4.11736138e+00,   3.03668375e+00],
                 [ -1.32168491e+00,  -7.02437243e-01,  -1.29179207e+01],
                 [ -1.47856067e+00,  -5.29521921e-01,   1.46165671e+00],
                 [  7.03138667e+00,  -4.08892638e+00,  -5.10068263e+00],
                 [  2.45183854e+00,   7.04297083e-02,  -1.05452010e+00],
                 [ -4.39871859e+00,  -2.13850981e+00,   3.75261461e+00],
                 [ -7.30211527e-01,   9.76898750e+00,   6.29605336e+00],
                 [ -2.73450869e+00,   1.46533677e+00,  -2.47436983e-01],
                 [ -6.41990560e+00,   1.91517940e+00,  -3.35203853e+00],
                 [  1.51459595e-01,  -2.13100801e+00,   2.30768969e+00],
                 [ -9.85149904e-02,  -2.14383640e+00,   3.44043840e-02],
                 [  4.78381897e+00,   2.06699108e+00,   1.57303493e-01],
                 [  3.22485608e-02,   5.78117722e-01,  -6.75927106e-01],
                 [ -7.22208995e+00,  -4.11912410e+00,  -8.45429508e+00],
                 [ -6.04660071e+00,   6.11985625e+00,   3.63257225e+00],
                 [  5.83350744e+00,   9.57342379e+00,  -5.54843425e-01],
                 [ -5.57458009e+00,   1.02596451e+00,   2.16957655e+00],
                 [  8.14224955e+00,   9.37580035e+00,   2.59877394e+00],
                 [ -9.45914650e+00,   2.20784671e+00,   1.53708033e+00],
                 [ -2.05756000e+00,   6.89483761e-01,  -3.67386021e+00],
                 [ -2.26725299e+00,  -2.48542666e+00,   6.18422299e-01],
                 [ -6.91614869e+00,  -6.48521373e-01,   9.92523947e-01],
                 [ -5.44478876e-01,   1.31275526e+00,  -1.59226921e+00],
                 [  1.84754387e+00,   1.00409658e+01,   2.05700697e+00],
                 [  6.45557172e+00,   6.60079533e+00,  -8.71027231e+00],
                 [  3.87968117e-01,  -1.99561456e-01,   1.72988116e+00],
                 [  4.25591562e+00,  -3.18974067e+00,  -3.24641299e+00],
                 [  2.53725449e+00,   3.00470544e+00,  -6.63764312e-01],
                 [  4.96515465e-01,   9.19791225e-01,  -3.29936764e-02],
                 [  7.39321459e-01,   5.42811358e-02,   9.24758729e-01],
                 [ -4.21932594e-01,  -1.69835222e+00,   4.82425592e+00],
                 [ -2.86958638e+00,   6.44272917e+00,   1.72003676e-01],
                 [  8.71919984e+00,  -1.01031305e+01,   6.37230178e-01],
                 [  2.49505047e+00,   6.13354090e+00,   5.53839144e+00],
                 [  6.58288336e+00,  -2.45537715e+00,  -2.95496186e+00],
                 [ -9.20342537e+00,  -9.07113890e-01,  -5.26526126e+00],
                 [ -1.01002372e+01,   6.39490759e+00,   4.87804837e+00],
                 [ -2.83285159e+00,  -2.29536267e+00,   5.94826604e+00],
                 [  9.31007923e-01,   1.26052433e+00,   8.49591953e-01],
                 [  4.63932634e+00,   3.66603920e+00,  -4.36755126e+00],
                 [  1.43885819e+00,   2.73684868e-01,  -5.48995574e-01],
                 [ -6.28463944e+00,  -1.24058170e+01,   2.41411000e-01],
                 [  1.03388875e+01,  -4.13051811e+00,  -5.26694725e+00],
                 [ -1.44506402e+00,  -7.21670682e-01,  -6.53153981e-01],
                 [  1.31323189e+00,   1.31180090e+00,  -2.95932501e+00],
                 [ -1.75398714e+00,   2.86831883e-02,   2.42449981e+00],
                 [  3.42746496e-01,   5.85975547e-01,   1.69829375e+00],
                 [ -1.36634984e+00,   4.04675690e-01,   5.61733806e-01],
                 [ -5.86755071e+00,  -9.54109893e-01,  -8.01890523e+00],
                 [  4.61217813e+00,  -5.05458329e+00,  -9.08762792e+00],
                 [ -7.47266893e+00,   4.92964489e+00,  -1.18841742e+01],
                 [ -1.26256033e+01,   3.04723959e+00,   4.01033300e+00],
                 [ -6.24283315e+00,   7.00605659e+00,  -1.02284071e+00],
                 [  7.76254982e+00,  -1.72744788e+00,  -4.06485044e+00],
                 [ -5.19053879e+00,   1.97480200e+00,   9.51242736e+00],
                 [  9.44862133e+00,  -7.33110642e+00,   4.90441929e+00],
                 [  1.59963719e+00,  -2.82982225e+00,  -4.19512346e-01],
                 [ -1.99396575e+00,   6.63570636e+00,  -1.12561452e+00],
                 [  6.01497371e-01,  -2.26060542e+00,   8.04504523e-01],
                 [ -8.77373721e-01,   6.42607716e+00,  -7.61326136e+00],
                 [ -9.46734093e+00,   1.01717848e+01,  -2.29675457e-01],
                 [  2.92192858e+00,   1.81529127e-03,  -1.01037152e+00],
                 [  6.80808972e+00,  -2.96048417e+00,  -3.69033874e+00],
                 [ -1.78983949e+00,  -2.07276761e+00,   1.38248607e-01],
                 [  1.49869694e+00,  -8.59686184e-02,  -4.37807602e-01],
                 [ -4.53005341e+00,   1.90051676e+00,  -4.60681963e-01],
                 [ -2.62558925e+00,   5.76418690e+00,   7.97219204e-01],
                 [ -2.77062178e+00,  -7.48318580e+00,   1.85943433e+00],
                 [  7.08811191e+00,   9.08849513e+00,  -5.62212219e-01],
                 [ -9.32404005e+00,   1.56434893e+00,  -2.90198928e+00],
                 [ -1.32055512e+01,  -4.25929061e-01,  -5.35772602e-01],
                 [  4.31969379e+00,  -9.96484307e+00,   5.68136744e+00],
                 [  6.22064050e+00,   8.19495812e+00,   3.25758227e+00],
                 [  2.24545872e+00,  -1.28342095e+00,  -9.37230017e+00],
                 [  6.97602254e-02,   3.30642343e-02,   1.53742596e+00],
                 [  5.65640601e+00,  -7.23287584e-01,   1.20556082e+01],
                 [  1.14850761e+00,   4.08747691e-02,  -1.72784164e+00],
                 [  5.45601862e+00,   8.96212091e+00,  -8.73134881e+00],
                 [  1.12887480e+01,   3.09469574e+00,  -2.85333645e+00],
                 [  4.61271335e+00,   8.03138497e-01,   1.01603107e+00],
                 [  1.71990936e+00,   6.93431562e+00,   4.25882150e+00],
                 [ -6.50165939e-01,  -9.24537978e-01,  -1.62403991e+00],
                 [ -7.97961586e-01,  -2.21376878e-01,  -3.02571054e+00],
                 [  5.55490277e-01,   1.29374455e+00,  -6.30058824e-01],
                 [  7.25301991e+00,   3.18960386e+00,  -7.19972044e+00],
                 [ -5.50066744e+00,   4.93589373e+00,  -1.11752754e+00],
                 [  1.27872030e+00,   3.37806334e-01,  -1.41125839e+00],
                 [ -5.54902921e+00,   1.70557955e+00,  -9.19488476e+00],
                 [ -5.51685363e+00,  -6.95149327e+00,  -9.67783415e-01],
                 [ -6.09108661e+00,  -1.20052401e+01,  -5.40450089e-01],
                 [ -3.19249144e+00,   9.07501122e+00,  -3.77778935e-01],
                 [  8.67330462e+00,   4.91174437e+00,   6.03244026e+00],
                 [  8.95801507e-01,  -9.06456093e-01,   1.45284984e+00],
                 [ -1.09914517e+01,  -1.43946579e+00,  -9.69546085e+00],
                 [  1.69626392e+00,  -1.45830245e+00,   1.46480783e+00],
                 [  5.30126974e+00,  -1.19067589e+01,  -2.34437981e+00],
                 [  3.03908256e+00,   5.29779952e+00,   4.55635386e+00],
                 [  1.68608184e-01,   1.67280529e+00,  -1.41228114e-01],
                 [ -4.56597471e+00,  -2.18535146e+00,  -3.73201434e+00],
                 [  1.84646669e+00,   2.49547606e+00,   1.73841470e-01],
                 [ -1.10171063e+00,   1.32256938e-01,   1.35323717e+00],
                 [ -2.26609091e+00,  -2.10317868e-01,  -1.46267583e+00],
                 [ -1.07943416e+00,   7.18935746e-01,   1.07413520e+01],
                 [  3.00094644e+00,   3.63485360e+00,  -3.44053712e+00],
                 [  3.86782705e+00,  -1.09917584e+01,  -1.46642263e+00],
                 [  2.75877777e+00,  -5.56739943e+00,  -9.47745697e-01],
                 [  3.84805580e-01,  -7.99482480e+00,  -8.46877151e+00],
                 [ -6.15750968e+00,  -2.24134738e+00,  -5.38425385e+00],
                 [ -6.54412478e+00,  -1.78429854e+00,   4.50145242e-02],
                 [ -1.03911027e+01,   1.74632991e+00,  -1.12345250e-01],
                 [ -1.45259604e+00,   1.90289752e-02,   2.38617984e+00],
                 [  4.18530655e+00,  -2.09445957e-01,   1.49037189e+00],
                 [  6.06834896e-01,  -1.99605412e+00,   2.91114672e+00],
                 [ -1.44821049e+00,   2.04253980e+00,   8.30757494e+00],
                 [  5.34275460e+00,   1.33500884e+00,   2.82676935e+00],
                 [ -7.37307390e-01,  -3.55229661e-01,  -1.49860109e+00],
                 [ -2.77087716e+00,   6.00661800e+00,  -7.92719059e+00],
                 [ -5.90432544e-01,   2.83557910e+00,  -2.46735248e+00],
                 [  1.34909827e+00,  -2.58801777e+00,   3.09353166e-01],
                 [  2.73308695e-01,   2.59231758e+00,  -2.24446908e+00],
                 [ -1.66170409e-01,  -1.11173187e-01,  -1.01391869e+01],
                 [  5.38281208e+00,   6.20474879e+00,   5.99247877e+00],
                 [  6.26926267e+00,  -4.39482703e-01,   1.43392916e+00],
                 [  2.44632243e+00,  -2.76837532e-01,   3.96215178e-01],
                 [  1.45616414e+00,  -5.27144165e+00,   1.07779933e+01],
                 [ -4.42383215e+00,  -4.90794718e+00,   3.59879981e-01],
                 [  4.33920723e+00,  -9.55278849e+00,  -2.31146942e+00],
                 [  5.95811709e+00,   4.52549375e-01,  -7.30969922e-01],
                 [ -1.67873394e+00,   5.46084920e-01,   5.80238784e-02],
                 [ -1.89844036e+00,  -5.05529510e+00,   5.58921187e+00],
                 [  3.28499103e+00,   1.18106028e+00,  -2.66887603e+00],
                 [  6.86077970e+00,   6.52092496e+00,   1.42983925e+01],
                 [  5.25472639e+00,   9.18866603e+00,  -5.81398952e+00],
                 [  3.01077954e-01,  -1.12421751e+00,   5.29255402e-01],
                 [ -1.27257462e+00,  -1.22852224e+00,   4.09462439e+00],
                 [ -8.68361952e-01,   1.20770084e+00,   1.96035939e+00],
                 [ -2.81094631e-01,  -1.69096624e+00,  -1.02578660e+00],
                 [ -1.79649276e-01,   1.36076177e+00,  -1.40819087e+00],
                 [ -3.84218829e+00,   3.80920665e+00,   7.96259446e+00],
                 [  3.75843515e+00,   4.16870276e+00,   9.19685691e+00],
                 [ -1.35982392e+00,   4.64030875e+00,   7.50299266e+00],
                 [  5.75676602e+00,   2.09064792e+00,  -2.88547694e+00],
                 [  7.18977972e+00,  -2.62131262e+00,   1.20402579e+00],
                 [  2.00726365e+00,  -7.96142788e+00,   4.36203912e+00],
                 [  2.43219721e+00,  -1.34240429e+00,  -1.28277988e+00],
                 [  1.43695602e+00,   5.30989460e+00,   1.55782297e+00],
                 [  5.11168374e-01,  -1.38447545e+00,   1.51668802e-01],
                 [ -1.15516567e+01,   2.48442392e+00,  -4.08561680e-01],
                 [  1.01554160e+00,  -1.18514982e+00,  -7.98473493e-01],
                 [ -1.01065186e+01,  -2.90520117e+00,   4.30881330e+00],
                 [ -1.17854889e+01,  -1.28936389e+00,  -2.31681375e+00],
                 [  3.20644569e-01,   2.04823568e+00,   2.17917116e-01],
                 [  3.17617256e+00,  -7.54267282e+00,  -7.34964117e+00],
                 [ -1.91570930e+00,  -1.06936399e-01,   2.46301983e-01],
                 [  4.09524798e-01,  -5.55645469e-01,  -6.70474309e+00],
                 [ -2.45394487e+00,  -7.82197436e-01,  -7.80513615e-01],
                 [ -1.26092893e+00,   6.87268293e-01,  -9.96350684e+00],
                 [ -2.62972134e+00,  -7.06754507e+00,  -3.58661415e+00],
                 [  8.84293383e-01,   8.59593084e+00,  -1.52281672e+00],
                 [  1.83112977e+00,  -9.96662357e-01,   6.78815402e+00],
                 [ -4.54844621e+00,   4.60642697e+00,   1.46348040e+00],
                 [  1.16718238e+01,  -6.20815450e+00,   4.82355297e+00],
                 [  5.86301437e+00,   5.12942971e+00,  -4.46347246e-01],
                 [  1.26935525e+00,  -4.19154401e+00,   1.24933775e+01],
                 [  1.60869401e+00,  -3.39148150e+00,  -5.56755659e-01],
                 [  9.32669211e+00,  -5.84802973e-01,   3.54673917e+00],
                 [ -2.81386062e+00,   2.38706939e-01,  -1.29725321e+00],
                 [ -4.50386469e+00,  -1.44378678e+00,  -5.05632308e+00],
                 [  7.97285533e+00,   7.27318366e+00,   3.66945590e+00],
                 [ -7.26299869e-01,  -8.50932095e-01,   2.00994930e+00],
                 [  9.93378590e+00,   2.26318863e+00,   2.84619811e+00],
                 [  1.01270262e+00,  -1.75363717e+00,   2.17606529e+00],
                 [ -2.49751620e+00,   1.61434852e+00,   2.01504258e+00],
                 [  6.90438403e-01,  -4.41709239e-01,  -1.31967576e+00],
                 [ -1.54411173e+00,   1.08015511e+01,  -5.19996697e+00],
                 [  5.06412457e+00,  -2.41259889e-01,   7.02109333e+00],
                 [ -6.23990507e+00,  -1.85741897e+00,   4.36468063e+00],
                 [ -2.61221107e+00,  -6.08248411e+00,   2.78983135e+00],
                 [ -1.16043165e+00,   8.14084386e+00,   8.11178246e+00],
                 [ -7.01251946e+00,   1.83277685e+00,  -9.12029670e+00],
                 [ -2.55133001e+00,   7.07628600e+00,  -3.62494678e+00],
                 [  1.40582256e+00,   8.41232105e-01,   3.09361749e+00],
                 [ -1.18879303e+00,  -4.17295560e-01,   1.04616230e+00],
                 [ -1.05363715e+00,   7.59595125e+00,  -9.94365808e+00],
                 [ -6.57549981e-01,   1.19132945e+00,  -7.81578705e-01],
                 [  1.20938091e+00,  -8.85242595e+00,   6.70484201e+00],
                 [  2.53699026e+00,  -1.28271895e+00,  -5.03882665e-01],
                 [  4.84463507e-01,   1.84937329e+00,   1.39973033e+00],
                 [  6.93937501e+00,  -6.43525148e+00,   1.99514405e+00],
                 [  1.88055133e+00,   1.83448986e-01,  -1.76081668e+00],
                 [ -9.54086078e-01,  -1.85094206e+00,  -3.14900482e-01],
                 [ -2.42924044e+00,  -1.27295937e+00,   1.35374011e+00],
                 [  3.69833454e+00,   1.11374742e+01,  -5.04912239e+00],
                 [ -1.25999939e+01,   2.00255130e+00,   1.10794404e+00],
                 [  5.69094105e-01,  -3.71694926e+00,  -6.83395770e-01],
                 [  1.12410034e+00,   2.52808454e+00,  -7.30652347e+00],
                 [  5.72784761e+00,   3.31021905e+00,  -1.17133565e+01],
                 [  1.02412955e+01,  -1.37493471e+00,   5.71606139e+00],
                 [  1.25157336e+01,  -3.02225462e+00,  -4.14643506e+00],
                 [  4.31341633e+00,   6.27897605e+00,   5.48219388e+00],
                 [ -7.44406180e-01,  -9.86563733e-01,  -6.01539412e-01],
                 [  1.47111603e+00,   7.22251543e-01,   2.32512658e+00],
                 [  2.21788697e-01,   4.68658727e-01,  -1.96867848e+00],
                 [ -7.96587672e+00,   3.67382798e+00,   5.64497459e+00],
                 [ -4.40733873e+00,  -8.00450202e+00,   8.10571674e+00],
                 [  2.07401061e+00,   1.63386279e+00,   1.11466670e+00],
                 [  1.81819278e+00,   4.11408724e-02,   5.09291904e-01],
                 [  5.35020690e-01,   2.48888946e-01,   3.07829986e+00],
                 [  2.32180776e+00,   3.45562319e+00,   5.20841057e-01],
                 [ -2.94980537e+00,  -2.46720668e+00,   5.91631523e-01],
                 [  3.17939790e+00,   5.99511684e-01,  -6.89183098e+00],
                 [ -2.97922061e+00,  -8.31694216e+00,  -5.55784990e+00],
                 [ -2.66207362e-01,   6.81514957e-01,   2.52240399e+00],
                 [  2.53676596e-02,  -2.15081564e+00,  -4.75431088e+00],
                 [  9.51173773e+00,   1.53795344e-01,  -1.23208186e+00],
                 [ -6.31645457e+00,   6.66063057e+00,   8.59413563e-01],
                 [ -2.44317635e-01,  -2.87748874e-01,   8.29062293e+00],
                 [  8.93405410e+00,  -5.53476677e+00,   2.10060103e+00],
                 [  8.94923533e-01,   2.47848016e+00,  -1.76347279e-01],
                 [  4.17740976e-01,   4.49770224e+00,   9.19624471e+00],
                 [ -7.65369020e-01,   1.51215907e+00,   2.10260091e+00],
                 [ -1.03532808e+01,  -5.06583995e-01,  -4.74857343e+00],
                 [ -4.79269576e+00,  -8.84792690e+00,   3.02081168e+00],
                 [  1.08971540e+00,   1.30493084e+00,  -2.76052402e+00],
                 [  8.15039734e+00,  -8.19961525e+00,   7.13349007e+00],
                 [  1.94281790e+00,   7.61663135e-01,  -1.49875416e+00],
                 [  1.80291941e+00,   9.92683632e-01,   1.52129688e-01],
                 [  3.28520097e+00,  -1.11983031e+00,  -1.34974747e+00],
                 [  8.31771170e-01,   1.78278082e+00,  -9.46955474e-01],
                 [ -3.30041187e-02,   1.00933502e+01,  -8.35667416e-01],
                 [ -3.94406467e+00,  -7.80749963e+00,   1.21931571e+01],
                 [ -4.16468974e+00,  -3.13501880e+00,  -8.96583686e+00],
                 [  9.61405753e+00,   6.38681998e-01,  -3.52900131e+00],
                 [  2.32775709e+00,  -5.81202670e+00,   1.73798282e+00],
                 [ -7.53177430e+00,  -2.32928503e-01,   5.86434846e+00],
                 [  2.63551512e+00,   4.68524387e-01,   4.33866092e+00],
                 [  3.48709805e-01,  -4.49999339e-01,   2.01260736e+00],
                 [  7.08550021e+00,   3.98903562e-01,  -7.62560358e+00],
                 [ -9.73539642e-01,  -4.13726355e-01,  -5.22612667e-01],
                 [  2.23613765e+00,  -1.24177340e+01,   5.41973002e+00],
                 [ -8.19789203e+00,  -4.74048194e+00,  -6.81171053e+00],
                 [  4.53562343e+00,   1.67727134e+00,   1.78830566e+00],
                 [ -4.56150765e+00,  -1.57281167e+00,   2.08320055e+00],
                 [ -2.34308174e-01,  -7.36936357e-01,   2.02853608e+00],
                 [ -3.46793784e-01,  -2.36409800e+00,  -1.82663413e+00],
                 [  1.51379918e+00,   2.22978497e+00,  -3.37223029e+00],
                 [  5.79976296e+00,   2.03214112e+00,  -8.90079246e+00],
                 [ -1.84484734e+00,  -8.98151718e+00,  -3.76292385e+00],
                 [  3.80495474e+00,  -2.68745719e+00,   1.46034904e+00],
                 [  4.77441732e+00,  -1.09906631e+01,   4.23368063e+00],
                 [  3.54148784e+00,  -9.89771997e+00,  -1.09235593e+00],
                 [  1.00856954e+00,  -4.62464359e+00,  -9.85750057e+00],
                 [  4.38810199e+00,  -5.32434868e+00,  -1.03330321e+01],
                 [ -1.11009661e+01,   1.12296181e+00,   1.38368027e+00],
                 [ -1.16959329e+00,   6.58863997e-01,  -1.08755009e+00],
                 [  5.52821915e+00,  -2.86869092e+00,   4.81943638e+00],
                 [ -5.17890246e-01,   1.40080237e-02,  -1.88177259e-01],
                 [ -7.65607019e-01,   3.57364649e+00,  -6.21557661e+00],
                 [ -7.33145596e+00,  -8.69800358e+00,  -1.75813120e+00],
                 [ -2.52361925e+00,   3.48667090e+00,   7.13152215e-01],
                 [ -3.40690727e-01,  -9.03412965e+00,   3.25031144e+00]]).T, FortranArray([[ -6.81602685e+03,   4.38305915e+01,   5.20127007e+01],
                 [  4.38305915e+01,  -7.07514025e+03,   4.78970292e+00],
                 [  5.20127007e+01,   4.78970292e+00,  -6.96902054e+03]]).T, FortranArray([-26.68356909, -28.17728902, -28.20400895,  -9.09467718,
                  -9.52842189, -10.02457348,  -9.41642536, -10.01915238,
                  -9.94276552,  -8.31650019, -10.33502139, -27.94094479,
                  -9.8126945 , -26.8322076 , -11.03831764,  -9.34725019,
                 -27.69541057, -10.12453301, -28.96620552, -28.40044452,
                 -27.53695233,  -9.65426829,  -9.70230233, -10.2455966 ,
                  -9.47452507,  -9.71741171,  -9.75853523,  -9.72473172,
                  -9.72234156, -27.39017099, -10.25611073, -27.9031725 ,
                 -10.45157057,  -9.78997496, -29.27342645,  -9.91136763,
                 -27.84034326, -27.83941047, -27.8732908 ,  -9.58946495,
                  -9.56787693,  -9.79734375,  -9.81640595,  -9.72886637,
                 -10.02362381,  -9.61531173,  -9.85523095, -28.18596487,
                 -10.19867724, -27.92717386,  -9.91137008,  -9.51080635,
                 -27.50501126,  -9.94673263, -24.72069634, -28.72418717,
                 -27.75717741, -10.00335264,  -9.72517111, -10.00193343,
                 -10.42823457, -10.35630063, -10.1054401 ,  -9.75494251,
                 -10.09202616, -27.78697939,  -9.67539407, -28.91326522,
                  -9.91037787, -10.34059038, -26.36329975,  -9.75480894,
                 -28.28766803, -28.18385791, -28.44990768, -10.01475097,
                  -9.89523781,  -9.77387022,  -9.67010347, -10.0768466 ,
                  -9.35794039, -10.41156694, -10.21482866, -26.17934094,
                 -10.18329547, -27.88824113,  -9.55473208,  -9.61456827,
                 -28.61760636,  -9.63905501, -27.47884622, -27.4639815 ,
                 -26.65417394,  -8.48908428, -10.28924421,  -9.69078566,
                  -9.5043863 ,  -9.90457573, -10.17171575,  -9.91999258,
                  -9.99643684, -28.62430216, -10.12325014, -26.64275237,
                  -9.69942934,  -9.06173049, -28.16658215,  -9.63439975,
                 -28.1479868 , -28.16950416, -28.68531994,  -9.73115593,
                  -9.2339512 ,  -9.23435422,  -8.7895856 , -10.16789628,
                  -9.82033761,  -9.69082656,  -9.63797403, -27.21887257,
                 -10.30925949, -28.71880847, -10.06510976,  -9.62564168,
                 -28.09897004,  -9.76523375, -28.70788903, -26.77297019,
                 -28.18889107,  -9.75922622, -10.18595877, -10.25972004,
                 -10.08407358,  -9.8433952 ,  -9.54060294,  -9.61297276,
                  -9.34801192, -29.22919091,  -9.15580493, -28.1037396 ,
                  -9.76588099, -10.03421317, -27.90027371,  -9.63852088,
                 -27.80186285, -28.20386302, -27.72505742, -10.26295098,
                 -10.20273917,  -9.54430026, -10.4071989 , -10.02395433,
                 -10.02533013,  -9.77053355, -10.25280501, -27.07753948,
                 -10.1833175 , -27.76175072,  -9.81954424, -10.11744742,
                 -27.81732479, -10.22771276, -28.12639119, -28.63669936,
                 -27.75049218, -10.48106972,  -9.56916783, -10.80638367,
                 -10.75621857, -10.14069162,  -9.841347  , -10.10817288,
                  -9.53398248, -28.07314383,  -9.59908384, -27.6499041 ,
                  -9.8882707 ,  -9.92791916, -27.16033518, -10.18144805,
                 -24.5953126 , -27.62145107, -28.48968011, -10.3407023 ,
                  -9.47469641,  -9.95552474,  -9.80405146, -10.59994165,
                 -10.20529158, -10.14999325, -10.01864864, -27.54531926,
                 -10.3348522 , -27.57397963, -10.09146352,  -8.12061948,
                 -27.70645912,  -9.45262774, -27.27537853, -29.48476931,
                 -29.93887877,  -9.81604248,  -9.46738462,  -9.62009147,
                  -9.35943873,  -8.13472111,  -9.75421691,  -9.05606296,
                  -9.94222305, -28.35046377,  -9.92701696, -28.00094036,
                 -10.3065875 ,  -9.80260449, -26.77763355, -10.33372883,
                 -25.98207692, -27.58567775, -28.99833363, -10.32065362,
                  -9.73317045,  -9.75804342, -10.09142788, -10.13188105,
                  -9.71482781,  -9.25444422,  -9.9480682 , -27.17591302,
                  -9.30771112, -28.17459663,  -9.74038501, -10.04126089,
                 -28.59221713,  -9.47752797, -27.53849643, -29.00829587,
                 -28.20198788, -10.34400469,  -9.93797444,  -9.68581491,
                  -9.87634808,  -9.73849407, -10.02447206,  -9.64112525,
                  -9.63366394, -28.24553516,  -9.37870025, -28.34276173,
                 -10.37318617,  -9.90379049, -28.59064284,  -9.25517605,
                 -25.76359089, -27.92183519, -27.36331649, -10.06217461,
                  -9.78346897,  -9.26407019, -10.04075078,  -8.68407194,
                 -10.31896259,  -9.97133174, -10.07634627, -28.50845579,
                 -10.33079554, -27.88321543, -10.15985317, -10.20049572,
                 -28.46027538,  -9.70028908, -25.07298571, -27.58079427,
                 -27.73234372,  -8.68312498,  -9.60087467, -10.54889717,
                 -11.71163468,  -9.59759952,  -9.92042991, -10.06771136,
                  -9.86269553, -27.72079155, -10.27171077, -26.7201448 ,
                  -9.82059891,  -9.84013655, -27.38891544,  -9.94204494,
                 -27.75621136, -28.20859748, -28.38870513,  -9.54162455,
                 -10.12882276, -10.02281964, -10.21740402,  -8.90240475,
                 -10.2752121 , -10.32465041,  -9.87573861, -29.60529639,
                 -10.07765955, -28.05271975,  -9.9133582 ,  -9.97768828,
                 -27.52839592, -10.05607611, -27.74608765, -28.11018345,
                 -27.44866965, -10.15283674,  -9.70298789,  -8.78907302,
                  -9.63914768,  -9.71340923,  -9.15699853, -10.13419697,
                 -10.00245752, -27.61270842, -10.1108845 , -28.31057212,
                  -9.38717341,  -9.67636714, -28.12501829, -10.05774529,
                 -30.20067831, -27.17438451, -28.07913311, -10.26046401,
                  -9.93972485,  -9.18749507,  -9.36706844, -10.16001872,
                  -8.93946345,  -9.80620606,  -9.77140454, -28.10098388,
                  -9.53659905, -28.28138496,  -9.69437841, -10.06058335,
                 -27.65474671, -10.13840674, -28.70502286, -27.01913912,
                 -28.72997209,  -9.69636064,  -9.4368453 ,  -9.78722217,
                  -9.61499082,  -9.73107563, -10.05781834,  -9.30665727,
                  -9.74683072, -28.16217186, -10.14810942, -28.07081202,
                  -9.87984531,  -9.72410059, -28.93634625, -10.05107243,
                 -28.0422647 , -27.62909497, -27.70417837,  -9.79172336,
                 -10.22466449,  -9.62917472, -10.12233701, -10.03310214,
                 -10.16729455,  -9.61412916, -10.07424351, -27.2176383 ,
                 -10.03728598, -28.36444176,  -9.89758367,  -9.96753768,
                 -25.43793934,  -9.73077648, -27.08161073, -27.64075661,
                 -27.05971992, -10.18393681, -10.37065021,  -9.88684775,
                  -9.95106466, -10.03272973,  -9.8052985 ,  -9.29999228,
                  -9.86063283, -27.89194689,  -9.85236336, -27.95190738,
                 -10.45382813,  -9.94133143, -27.52184096, -10.20856347,
                 -27.04974697, -27.93072864, -30.54566697,  -9.92747893,
                  -8.49918723, -10.32571394, -10.4627211 , -10.56934731,
                  -9.43355516,  -9.88079087, -10.12748339, -28.62522339,
                  -9.26015326, -27.51191633,  -9.65700431,  -9.91076823,
                 -25.61863028,  -8.90337979, -27.4796684 , -27.70079131,
                 -27.16539613, -10.21423717, -10.04241355, -10.90344421,
                 -10.10090601,  -9.41494935,  -9.88435569, -10.04583336,
                  -9.56014311, -27.07686201, -10.31719999, -27.05746733,
                  -9.73492739, -10.02667283, -28.85989994,  -9.69996561,
                 -28.54264033, -26.68036535, -27.4483566 , -10.21700033,
                 -10.15382829,  -9.93348735,  -9.44984466,  -9.49381591,
                 -10.11243059, -10.10171896,  -9.94016098, -27.69796142,
                 -10.14910584, -27.44567298, -10.33057306, -10.17094778,
                 -28.05783998, -10.73746033, -27.02199882, -27.35037991,
                 -27.8360476 ,  -9.77807829,  -9.43633462,  -9.36874422,
                 -10.21604286,  -9.69815308, -10.00456075,  -9.75800339,
                  -9.58114561, -28.58293391,  -9.7770808 , -27.65262924,
                 -10.63703958,  -9.88500777, -28.08750261,  -9.78968338,
                 -28.14721869, -28.69716799, -27.86881261,  -9.40283564,
                  -9.74334319, -10.19320222,  -9.76071784, -10.09268261,
                 -10.31487275, -10.00682323, -10.29731095, -27.70010428,
                  -9.83083799, -28.27846314, -10.21078712,  -9.19791309,
                 -28.44903026,  -9.9258837 , -28.0331092 , -27.30384304,
                 -27.71054005,  -6.28675092,  -9.30729765, -10.08942791,
                 -10.52725752, -10.12331918,  -9.59200912, -10.06275871,
                  -9.96716363, -28.58814785,  -9.63768482, -26.3828831 ,
                 -10.15201861,  -9.31055644, -27.38714186, -10.60968439,
                 -27.48889572, -27.824784  , -28.12722242,  -9.55168458,
                 -10.14954625,  -9.79263494,  -9.73350868, -10.22061208,
                 -10.14982637,  -9.58166326, -10.10263234, -27.96904206,
                  -9.66572288, -27.3202208 , -10.48062003, -10.58694366,
                 -27.80389965,  -9.83757031, -28.03171789, -28.11251   ,
                 -28.58230436, -10.2909587 ,  -9.57582785, -10.08902964,
                 -10.09268625, -10.01916882,  -9.97345193,  -9.87061407,
                  -9.92960449, -27.62488147,  -9.9815118 , -28.48775639,
                 -10.21888338, -10.08441133, -26.49858652,  -9.610171  ,
                 -28.4127051 , -27.93583583, -27.42408571,  -9.60732217,
                  -9.55746411,  -9.75837347, -10.08042449,  -9.32769941,
                  -9.95671943,  -9.30311599, -10.29438415, -27.90584479,
                  -9.98330885, -28.58092638,  -9.67127027,  -9.75457942,
                 -27.79784967,  -9.79747347, -27.10680754, -26.59277142,
                 -28.16483229, -10.29664523,  -9.65248488, -10.30108849,
                  -9.70709737,  -9.61794472,  -9.56479242,  -9.37761171,
                  -9.74069136, -24.52453149, -10.52893877, -27.79991377,
                  -9.44489789, -10.44762948, -27.6868443 ,  -9.36012725,
                 -25.37050294, -28.33179075, -28.06422442,  -9.88626551,
                 -10.23947007,  -9.68341492,  -9.9994953 ,  -9.63484236,
                 -10.29325919,  -9.16241731, -10.43481036, -27.29305295,
                  -9.63462535, -26.18609536, -10.61031773,  -9.67883855,
                 -28.1308924 , -10.1191196 , -27.67548245, -25.66842421,
                 -27.03504401,  -9.97318396, -10.08535334, -10.21316694,
                  -9.89228421, -10.31903406, -10.39458154, -11.76883941,
                 -10.02510151, -28.17444514, -10.17247516, -28.51216526,
                  -9.48197226,  -9.55031031, -27.73030462, -10.72143217,
                 -27.63227792, -27.11205065, -26.53221282,  -9.42073078,
                 -10.59042525, -10.09904112, -10.15946798,  -9.90439221,
                 -10.22999672,  -9.65552429,  -9.91730839, -27.54559246,
                 -10.32852471, -28.05663871,  -9.84698254,  -9.69101131,
                 -27.23601104,  -9.78789241, -28.20749374, -28.5739543 ,
                 -25.89963913, -10.27849567, -10.37786542, -10.36271618,
                  -9.39492128,  -9.56047535,  -9.88346277,  -9.18102021,
                 -10.40789571, -27.15862571,  -9.94893648, -27.28830724,
                 -10.05298213,  -9.63275075, -28.23306156, -10.23066991,
                 -28.09666728, -28.03806738, -27.11262133,  -9.87014813,
                  -9.92825767,  -9.5532344 ,  -9.847984  , -10.56437157,
                 -10.12024107, -10.37237407,  -9.71373196, -28.00626716,
                 -10.2076549 , -28.28341075, -10.01607532,  -9.40970342,
                 -27.03362327, -10.37656132, -27.54046605, -27.83421796,
                 -26.37438842, -10.28421078, -10.03569453,  -9.8456012 ,
                  -9.87018081,  -9.79530797, -10.02127428, -10.33218329,
                 -10.02843618, -27.14078576,  -9.89930465, -25.2248324 ,
                 -10.1438828 ,  -9.84705448, -27.59583889,  -9.79192888,
                 -26.74271624, -26.79154749, -27.95883227,  -9.63357742,
                  -9.26830995, -10.25380144,  -9.6156695 ,  -9.92317868,
                  -9.8397802 ,  -9.62963257,  -9.6299247 , -28.85122241,
                  -6.30552578, -29.15611398, -10.45643384, -10.09196063,
                 -27.94735101,  -9.75311688, -27.27995467, -28.00242857,
                 -28.56614787, -10.36684473, -10.00985043,  -9.81230355,
                  -9.86569146,  -9.80769569,  -9.51781239,  -9.99834644,
                  -9.82469685, -27.87973409, -11.82331659, -26.28550622,
                  -9.73108722, -10.12677742, -27.73058658, -10.22332927,
                 -28.68990877, -28.38003963, -27.63246545,  -9.66829349,
                 -10.10269528, -10.16997332,  -9.5510339 ,  -9.61228514,
                  -9.88805721,  -9.59829214, -10.44205241, -27.84235758,
                 -10.2983802 , -27.64331752,  -9.13155325,  -9.76956684,
                 -28.13299389,  -9.99475193, -28.00189334, -27.12168931,
                 -27.71732164,  -9.6024655 ,  -9.71059744,  -9.5972655 ,
                  -9.76564502,  -9.74712185,  -9.93354927, -10.26589056,
                  -9.65535415, -27.77596846, -10.0607406 , -27.61264142,
                  -9.87955854,  -9.89915691, -27.77747381,  -9.95026423,
                 -27.47281322, -26.08243793, -27.77063769,  -9.94193066,
                 -10.76870796, -10.69714212, -10.30052728,  -9.89152893,
                  -9.42908661,  -9.87337313,  -9.77434556, -26.24237285,
                  -9.27261529, -28.26849173, -10.0991968 , -10.0150825 ,
                 -26.89360411,  -9.9267911 , -27.10513867, -25.69633019,
                 -27.93197215,  -9.54919243, -10.91283516,  -9.75754624,
                 -11.80409782, -10.40359135,  -9.73129423, -10.64949916,
                 -10.50643134, -26.98260368, -10.05458765, -28.08697752,
                  -8.33895298, -10.12514772, -27.52848282,  -9.49569667,
                 -27.23091878, -28.53445548, -26.88157924, -10.14219842,
                  -9.45817713, -11.61084343, -10.12718911,  -9.69289478,
                 -10.02501145,  -9.60058922, -10.30639018, -26.18618843,
                 -10.58058528, -28.08700527, -10.10376372, -10.13580758,
                 -27.99455725, -10.20413565, -25.8241145 , -27.87277167,
                 -27.628315  , -10.65083473,  -9.846272  , -10.13977174,
                 -10.37330973, -10.17028542, -10.19491441, -10.39975175,
                 -10.03662497, -28.16493942,  -9.82118877, -27.94051622,
                  -9.29094121,  -9.88325782, -27.397331  ,  -9.64191018,
                 -28.20879097, -29.15347652, -28.67417703,  -9.91158153,
                  -9.6716271 ,  -9.87330421,  -9.28239061,  -9.11064454,
                 -10.41463457,  -8.89864758,  -9.82949683, -27.86202051,
                  -9.9718543 , -27.83395382,  -9.32720391,  -9.77837899,
                 -29.19645813,  -9.95727326, -28.7347581 , -28.15364039,
                 -27.1252583 ,  -9.80642783, -10.01149684, -10.05241045,
                  -9.99841383,  -9.83674734,  -9.57882398, -10.37547889,
                  -9.36258726, -27.69556881,  -8.58627561, -29.04964875,
                  -9.87111552,  -9.49470206, -27.62991597,  -9.78595532]), FortranArray([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.28080482e-02,   4.92292131e-02,  -8.18208976e-02],
                 [  4.93985215e-02,  -1.79186532e-02,  -4.27439585e-02],
                 [ -2.89012237e-02,  -5.26855554e-03,   1.90017683e-02],
                 [ -2.56691633e-02,  -3.80095854e-02,   5.16132407e-02],
                 [ -6.52795705e-02,   3.74741264e-02,  -1.50860256e-03],
                 [ -3.22153727e-03,  -5.26318751e-02,   4.07466754e-02],
                 [  8.87153974e-03,   7.68974266e-02,   1.65884874e-02],
                 [  2.82984121e-02,   2.60941294e-02,   7.25676303e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.03780116e-02,   6.49172875e-02,   2.46126796e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.52849425e-02,  -1.72428451e-02,   4.97895171e-02],
                 [ -5.78784138e-02,   1.11137713e-02,  -8.90002756e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.55890251e-02,   1.09983528e-02,  -2.00147481e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.34810942e-02,  -4.93813544e-02,  -1.25949954e-02],
                 [ -1.19377697e-02,  -2.89823048e-02,  -3.16739189e-02],
                 [ -3.97633745e-03,  -2.39466151e-02,  -1.72239589e-02],
                 [ -2.71316082e-02,  -5.21132511e-02,  -1.03941815e-02],
                 [  3.25463067e-02,   3.21659258e-02,  -2.50218514e-02],
                 [ -3.62613752e-02,   4.61891707e-02,  -2.33833989e-02],
                 [  1.38734104e-02,   1.33900480e-02,   1.03286905e-02],
                 [ -5.90852601e-02,  -4.83345103e-03,  -2.00207907e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.77353079e-02,  -4.66066323e-03,  -1.77128525e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.20269689e-02,   2.20012499e-03,  -2.68757541e-02],
                 [  5.85984264e-02,   2.32793873e-02,  -1.43294678e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.60780082e-02,  -5.30895797e-03,   8.39494894e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.63059938e-02,   1.64849238e-02,  -7.35183407e-02],
                 [ -5.29184355e-02,  -2.44213405e-02,  -7.07368993e-02],
                 [ -2.52588907e-02,  -5.31999578e-02,  -7.88842132e-03],
                 [  1.89194980e-02,   5.20043762e-02,  -5.51126126e-02],
                 [  5.37690785e-02,   3.67327148e-02,   2.59331504e-02],
                 [ -3.06266492e-02,  -3.36313744e-02,  -1.63895185e-02],
                 [ -1.36372444e-02,  -6.81316855e-02,   2.49341685e-04],
                 [ -2.69985858e-02,   1.15434895e-02,  -1.46296358e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.89221940e-02,   1.47245370e-02,   1.40655861e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.23799569e-02,   7.16275534e-02,   1.77867941e-03],
                 [  2.39582176e-02,  -1.68476435e-02,   7.64386851e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.19738243e-02,  -3.78475142e-02,   4.36992950e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.03103622e-02,   2.68270466e-02,  -8.87853231e-02],
                 [  5.28445020e-02,  -3.33594678e-02,   2.68173936e-02],
                 [ -3.06261455e-02,   6.06967765e-02,   4.34598905e-02],
                 [ -1.21302540e-02,  -3.45314058e-02,  -2.20622810e-02],
                 [ -1.42130908e-02,  -4.79930539e-02,   3.56747808e-02],
                 [ -1.05553081e-02,   3.51484041e-02,   5.18260065e-02],
                 [  7.28962745e-02,  -2.87062038e-02,   1.77710948e-02],
                 [ -6.64443491e-03,  -1.00721952e-02,  -5.49153657e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.85197098e-02,   3.71555670e-02,   2.64044100e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.06991466e-02,   6.20705659e-03,   3.29182676e-03],
                 [  4.43397272e-02,  -1.89495515e-02,  -5.72645416e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.74302281e-02,  -3.52209734e-03,  -4.65029825e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.16383364e-03,  -4.45051741e-02,  -2.13264894e-02],
                 [  2.06511795e-02,   2.08554841e-02,   1.34236792e-02],
                 [ -3.07228556e-02,   2.80036349e-02,   6.36016719e-02],
                 [ -3.05843496e-02,   2.92709679e-02,  -4.94729979e-02],
                 [ -1.75376483e-02,  -2.95540300e-02,   4.47876524e-02],
                 [  6.32649956e-02,   3.94458350e-02,   2.56046671e-02],
                 [ -6.04327440e-03,  -7.26027439e-03,   2.59645785e-02],
                 [  2.19181655e-02,   3.85047743e-02,   3.18557730e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.83916879e-02,   5.72661729e-02,   1.63022323e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.24130173e-02,  -7.19236856e-02,   1.11789978e-02],
                 [ -4.51237922e-02,   2.92166316e-02,   1.73743829e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.24476581e-03,  -4.89880835e-02,   4.66038494e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.49617175e-02,   3.24301170e-02,   1.98933731e-02],
                 [ -4.86844473e-02,  -2.72995813e-02,   3.62194807e-02],
                 [ -1.02232607e-02,   5.12602984e-02,   5.05637622e-02],
                 [  5.17114440e-02,   4.23422340e-03,   4.40501514e-02],
                 [ -1.24160411e-02,  -1.09169630e-02,  -5.50776736e-02],
                 [ -4.85284261e-03,   4.82587521e-02,  -3.56907757e-02],
                 [  1.86417989e-03,   1.48337071e-02,   6.68017257e-02],
                 [  1.47265772e-02,   6.36175476e-03,   3.18476230e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.07754294e-02,   6.86081951e-02,  -1.88400791e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.90111831e-02,  -6.82599199e-02,   1.09026441e-02],
                 [  3.20970437e-02,   4.48518739e-02,   9.47310419e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.52645706e-02,  -5.59510623e-02,   2.24832326e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -9.53434396e-03,   4.18925511e-02,   4.54168564e-02],
                 [ -2.43310518e-02,  -1.04218447e-02,   4.32856451e-03],
                 [ -7.10683658e-04,  -7.91332054e-02,  -2.87805079e-02],
                 [ -4.54367181e-02,  -2.93537617e-02,  -2.63059635e-02],
                 [  5.13300163e-02,   7.57157143e-03,   2.64921182e-02],
                 [  2.45377147e-02,  -3.14563502e-02,  -6.15794467e-02],
                 [  5.24643615e-02,  -3.84978068e-02,  -2.38915816e-02],
                 [  1.66630319e-02,  -3.47973011e-02,  -4.60275890e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -9.46328854e-03,   1.43725631e-02,   1.05331691e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -9.05469645e-03,   3.36366418e-02,   2.58467856e-02],
                 [ -3.56126750e-02,  -4.27185637e-02,  -5.17641112e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.53928526e-02,  -5.89753931e-02,  -1.44255448e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.37184987e-02,   3.04768519e-02,   5.30430323e-04],
                 [ -7.90125686e-03,   2.28396495e-02,   5.23932793e-04],
                 [  2.03496429e-02,  -2.79223053e-02,   5.53911474e-02],
                 [  5.96428266e-02,  -4.50634818e-02,  -1.38424467e-02],
                 [ -2.64047637e-02,   2.45255702e-02,  -2.98085602e-02],
                 [ -2.41707102e-02,   7.91729136e-02,  -1.60455741e-02],
                 [ -1.65509011e-02,  -7.77828761e-03,   7.30558467e-02],
                 [  6.20500641e-02,  -2.53050978e-02,   2.19975639e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.05477343e-02,  -3.17472192e-02,   4.42766887e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.94165527e-02,   4.27065758e-02,  -3.60918049e-02],
                 [  1.20184154e-02,  -5.38873220e-03,  -3.00459073e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.86428708e-02,  -1.14058579e-02,   6.91411865e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.09207646e-02,   5.81324735e-02,  -7.85670218e-03],
                 [  6.47313433e-02,  -4.50938515e-02,   1.86594827e-02],
                 [ -1.97330529e-02,  -8.33130501e-02,   3.37194683e-02],
                 [  6.11517278e-03,   5.08464883e-02,   1.60642404e-03],
                 [  2.59285951e-02,  -1.58014876e-02,  -6.03454807e-02],
                 [  1.66241221e-02,  -6.33809235e-02,  -5.32985843e-02],
                 [  4.38178199e-02,   4.41117481e-02,   4.13001145e-02],
                 [ -4.89709630e-02,  -2.43613773e-02,  -2.51269069e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.75576214e-03,   3.94078508e-02,   6.45258264e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.81190521e-02,  -5.16685219e-02,   7.41248819e-03],
                 [ -5.98246878e-02,  -4.19127738e-03,   5.75460081e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.25731790e-02,  -8.70955290e-03,   8.20708095e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.08927102e-02,  -4.17319249e-02,   2.23887868e-02],
                 [ -1.88377024e-02,  -3.79319383e-02,  -1.87211055e-02],
                 [ -7.32189152e-02,  -1.85150025e-02,   2.66515674e-02],
                 [ -3.25555930e-03,  -5.65572949e-02,  -2.17692029e-02],
                 [ -5.10171720e-03,   3.08291251e-02,  -6.63594624e-02],
                 [ -1.01458338e-02,   7.57203495e-02,   7.04333517e-03],
                 [ -1.02694921e-02,   4.46007207e-02,   1.71264343e-02],
                 [ -3.54059021e-02,  -3.33109960e-02,   4.97719112e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.78403084e-02,   2.74405977e-02,   3.71288584e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.71279381e-02,  -1.87168116e-02,  -6.11622697e-02],
                 [ -4.95417338e-03,  -1.66895534e-02,  -5.67881970e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.01479852e-02,  -2.19824421e-02,  -4.32361819e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.16779093e-02,   4.95882870e-03,  -3.59451394e-03],
                 [  5.15425380e-02,   6.24138205e-02,  -1.49298468e-04],
                 [  2.84832261e-02,  -2.16421780e-02,   1.49538023e-02],
                 [  3.71284723e-02,   6.50337324e-03,  -3.91966630e-02],
                 [  9.23055668e-04,  -1.27211355e-02,  -3.01522846e-02],
                 [  4.54982754e-03,  -1.57034361e-02,   2.54441621e-02],
                 [ -3.23466726e-02,   4.71678453e-02,   8.27261178e-04],
                 [  1.70503792e-02,  -1.32764791e-02,  -1.74279542e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.00781875e-03,   2.43190605e-03,  -2.35346565e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.92897646e-02,   9.63242848e-03,  -6.18149771e-02],
                 [  4.61784812e-02,  -5.38713265e-02,   2.16584723e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.56229460e-03,  -5.23727939e-02,  -1.66330694e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.44086393e-02,  -3.68291929e-02,  -5.75089977e-02],
                 [ -3.58267139e-02,   5.63691135e-03,   2.94100354e-02],
                 [ -4.68987050e-03,  -5.18854451e-02,   2.89101811e-02],
                 [ -5.35859066e-02,   1.36845071e-02,  -3.19938490e-02],
                 [ -2.54123584e-02,  -5.16612330e-02,  -8.29867697e-03],
                 [  9.33684066e-03,   8.49495435e-02,   3.06531439e-03],
                 [ -2.85180953e-03,   5.43747026e-02,   3.07735600e-02],
                 [  2.68080604e-04,   3.61900152e-02,  -2.07552070e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.58758584e-02,   2.32153650e-02,  -3.21076575e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.29206858e-02,   3.25949637e-02,  -5.12867260e-03],
                 [  1.12673422e-02,   3.58322030e-02,   1.88250766e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.85996563e-02,  -1.53443119e-02,   1.78413221e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.68138643e-03,  -1.48726966e-02,   2.32204836e-02],
                 [ -6.45459830e-03,  -6.55175364e-02,   4.07831845e-02],
                 [  2.08115555e-02,   7.42768824e-02,   3.86805293e-02],
                 [  1.10753817e-02,  -4.65799809e-02,   5.19225263e-02],
                 [ -3.85437903e-02,  -3.77564425e-02,   2.47474414e-02],
                 [ -3.88580139e-02,  -1.52835710e-02,  -7.20827742e-02],
                 [ -5.92041421e-03,   6.74442948e-02,  -2.22165688e-02],
                 [ -3.31432743e-02,   4.88577725e-03,  -1.61739486e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.42426691e-04,   4.13272792e-02,  -3.21251944e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.65170484e-02,   8.11076031e-02,  -1.19730985e-03],
                 [ -5.99255172e-03,   7.73389319e-02,   3.87103330e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.95115266e-02,  -4.48651193e-02,  -7.83472786e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -7.62768980e-03,   3.89498580e-03,  -7.35530232e-03],
                 [  4.76652345e-02,  -3.28670437e-03,   6.56705837e-03],
                 [ -1.31176235e-02,  -2.32105568e-02,  -3.00277002e-02],
                 [  5.62954628e-02,   2.15644382e-02,   4.06948065e-02],
                 [  2.30655356e-02,   5.09036816e-02,   4.97722857e-02],
                 [  5.15499288e-04,  -6.68743399e-03,  -9.87657710e-03],
                 [ -1.08042824e-02,  -6.45630705e-03,   3.49140183e-02],
                 [  8.84357074e-03,  -5.12729479e-02,  -2.00098152e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.79164600e-02,  -4.80174590e-02,   6.42419731e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.76251434e-02,  -3.35018103e-02,   3.25890295e-02],
                 [ -6.54710952e-02,   2.93322208e-02,   2.43693115e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.66409472e-02,  -1.51338764e-02,   1.51840777e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.27612511e-04,  -2.46896489e-02,  -9.71188530e-03],
                 [  7.64124394e-02,   2.54112549e-02,  -1.79788589e-02],
                 [ -6.62194909e-02,  -5.23280506e-03,  -2.43258173e-02],
                 [ -4.00179903e-02,  -3.16136252e-02,   4.45207230e-02],
                 [ -4.29874275e-02,   1.21253632e-02,   5.30100248e-02],
                 [  5.40373370e-02,   3.79263786e-03,  -4.36025948e-02],
                 [  2.28936835e-02,   1.36717137e-02,   2.62031328e-03],
                 [ -5.95341556e-03,   8.18462705e-03,  -2.25380047e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.59756949e-02,   2.83232445e-02,  -1.33504620e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.05127224e-03,  -1.71910753e-02,  -6.47926568e-02],
                 [  1.67458549e-02,  -1.60439605e-02,  -3.26665635e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.87209276e-02,  -4.18621273e-02,   4.62975185e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.14151728e-02,  -6.29768064e-02,   3.79709789e-02],
                 [  6.91103759e-02,  -4.06992219e-02,   2.13264513e-02],
                 [  7.46543853e-03,  -5.25031730e-02,  -1.63942134e-02],
                 [ -6.37575620e-04,  -1.59806121e-02,   5.86183872e-03],
                 [  1.56562891e-02,   8.21147345e-02,   1.02459446e-02],
                 [ -1.50749290e-02,  -3.26441322e-02,  -1.37933670e-02],
                 [ -3.15970830e-02,  -5.44435986e-02,  -2.40178215e-02],
                 [  1.67640630e-03,  -5.25909628e-02,  -6.20066667e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.28677913e-02,  -1.06338444e-02,   1.75914721e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.49727744e-02,   2.04546936e-02,   1.92082939e-02],
                 [ -4.93374284e-02,   4.01778722e-02,   1.87179778e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.44194569e-02,   5.46178779e-02,  -5.70341884e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.24684969e-02,  -5.22572440e-02,  -3.83502836e-03],
                 [ -2.37154990e-04,  -2.32081082e-02,  -1.04937022e-02],
                 [  4.96791908e-02,  -1.47909453e-02,  -2.00393900e-02],
                 [ -2.27414090e-02,   4.16688908e-02,  -6.63606962e-03],
                 [ -6.37957418e-02,   2.94682161e-02,  -3.55132380e-02],
                 [  8.19574853e-03,   3.31147177e-02,   1.21759255e-02],
                 [  2.90643699e-02,   6.33295178e-03,   1.32220640e-02],
                 [ -7.89392702e-03,  -7.05250879e-02,   1.31381688e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.85761670e-02,   3.04691224e-02,  -9.65802519e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.22656602e-02,  -7.91277536e-03,  -6.01225084e-02],
                 [ -3.29707476e-02,  -2.85982578e-05,  -1.68711946e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.01088927e-02,   1.96080628e-03,  -4.66372744e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.60114442e-02,  -4.59511480e-02,  -2.48565390e-02],
                 [  2.60205275e-02,  -6.03513068e-02,   3.04508846e-02],
                 [  3.45208963e-03,  -1.28195604e-03,   3.14166710e-02],
                 [  4.24183537e-03,   3.29905405e-02,   6.70971867e-02],
                 [ -3.47043878e-02,   4.24845497e-02,  -3.31664770e-02],
                 [ -1.89740897e-02,  -5.16238283e-02,   1.32536504e-02],
                 [ -3.39332133e-02,  -6.03886925e-02,  -7.65484536e-03],
                 [  3.16323221e-02,   2.45087703e-02,  -3.50081339e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.68324402e-02,  -4.93140216e-02,   3.95631479e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.82939890e-02,   4.77867362e-02,  -4.15532943e-02],
                 [ -2.29586791e-02,  -2.94880917e-02,  -5.23749278e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.11836734e-02,   6.39269764e-03,  -2.75765184e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.73520987e-02,   9.63662961e-03,  -2.29558759e-02],
                 [  2.05060658e-02,  -1.52813327e-02,  -4.99154499e-02],
                 [  3.73765076e-02,  -1.77924307e-02,  -2.59803005e-03],
                 [ -6.81399821e-02,   1.66093888e-02,  -2.85129267e-02],
                 [  6.74675325e-03,   2.98357422e-02,   3.50319523e-02],
                 [  6.21307875e-02,   1.20009038e-02,  -3.60207427e-02],
                 [  5.54416464e-02,  -2.85369582e-03,   3.45288212e-02],
                 [ -3.10163524e-02,  -2.71218270e-02,  -3.22130527e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.55308524e-02,  -3.74383598e-02,  -2.72373399e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.07765161e-03,  -2.01195421e-02,   4.02980744e-02],
                 [  8.18749084e-03,   4.34461573e-04,  -2.94587010e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.22557608e-02,  -6.79875658e-03,   1.05410901e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.56394562e-03,   6.86729864e-03,   1.45761611e-02],
                 [  6.22439173e-02,  -3.84950631e-02,  -1.42830766e-02],
                 [ -3.10526363e-02,   4.40438365e-03,   3.63340116e-02],
                 [  5.92573826e-02,   6.17722298e-03,  -5.67757800e-02],
                 [  6.95405416e-02,  -3.02254346e-03,  -3.43071772e-02],
                 [  2.57103246e-02,   3.04235136e-02,   4.63006761e-02],
                 [  1.20410909e-02,  -1.60459029e-02,  -6.67267363e-02],
                 [  2.10534576e-02,  -7.03498323e-03,   6.94373772e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.46972543e-02,  -5.89253903e-02,   3.31736199e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.33020491e-02,  -2.16536542e-02,   2.52857981e-02],
                 [  2.52794113e-02,   2.64852261e-02,  -4.99194633e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.65960248e-02,   3.93080353e-02,   4.65176635e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.61046777e-02,   3.03284347e-02,   7.36031978e-02],
                 [  5.00494657e-03,   4.26705888e-02,  -5.94884207e-02],
                 [  7.88777654e-02,  -1.59385669e-02,   1.23865415e-02],
                 [ -2.63016510e-02,  -3.56897659e-02,   5.67300842e-02],
                 [  1.28802475e-02,  -1.37607816e-02,   5.88303782e-02],
                 [ -2.85298854e-02,   7.16139139e-02,   8.86831093e-03],
                 [ -3.21154646e-02,  -5.43200967e-03,  -4.10284078e-02],
                 [  4.80812564e-02,   4.30385999e-02,   1.09272207e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.37439398e-02,  -5.66118963e-03,   5.22742756e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.40740219e-02,  -8.57770848e-02,  -2.28717850e-02],
                 [  3.52204519e-02,  -7.78036508e-02,  -3.80691961e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.81530219e-02,   3.86785321e-02,   7.99682393e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.34037955e-03,  -2.27215192e-02,   7.81303098e-02],
                 [  1.43238712e-02,  -2.53892610e-02,  -5.33964656e-02],
                 [ -1.27244026e-02,   5.63381986e-02,   3.90121033e-02],
                 [  1.59584857e-02,   1.90214810e-02,  -8.69677467e-02],
                 [  2.52138294e-02,  -1.38395638e-02,   1.77232142e-02],
                 [  1.78049768e-02,   8.22261572e-02,  -8.24241804e-03],
                 [  9.57527045e-02,  -4.61693954e-02,  -1.54586703e-03],
                 [  5.05256318e-02,  -2.25252080e-02,  -3.45515931e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -7.02751102e-03,   7.31944664e-02,  -2.06738484e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.97488332e-03,  -2.24453684e-04,   1.16378744e-02],
                 [  4.81694943e-02,   3.20915276e-02,   3.18797534e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.77016344e-02,   1.10319761e-02,   4.60925498e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.39965744e-02,  -6.78000132e-02,   1.69989981e-03],
                 [ -4.18602301e-02,  -2.74857615e-02,  -1.82604588e-02],
                 [  2.14273775e-02,  -2.91402897e-02,   5.62534932e-02],
                 [  1.25241228e-02,  -4.21787050e-02,   1.85053103e-02],
                 [ -8.32383913e-02,   1.32623311e-02,   1.41573885e-03],
                 [ -2.92808605e-02,  -3.00456822e-02,  -8.62484837e-02],
                 [ -3.45042405e-02,   5.23645197e-02,   2.00045456e-02],
                 [  1.15895003e-02,   6.53196809e-02,  -3.14946526e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.46080607e-02,  -2.72429741e-03,   2.24382494e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.96538627e-02,  -3.04959181e-02,   3.38402282e-02],
                 [ -3.25447289e-03,   4.02194384e-02,   1.40638060e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.36613448e-03,  -7.54728163e-02,  -1.49158821e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.66007635e-02,  -1.80527319e-02,   5.78099964e-02],
                 [  1.30062997e-04,   2.56520228e-04,  -8.64905718e-02],
                 [ -2.27743114e-02,   5.29054164e-02,  -1.25869254e-02],
                 [  4.87759565e-02,  -8.58304354e-02,   3.04383163e-03],
                 [ -1.21459339e-02,   2.11754071e-02,   4.07560408e-03],
                 [  1.41638599e-02,   7.54081291e-02,   2.98617212e-02],
                 [  1.11307146e-02,   3.09716792e-02,  -4.82386506e-02],
                 [ -9.31348782e-03,  -5.43564910e-02,   1.81591934e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.70714216e-03,  -1.01463828e-02,   8.02683628e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.67336862e-02,  -6.25818987e-04,  -2.06564491e-02],
                 [  1.38427085e-02,   1.04879937e-02,  -6.59108609e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.23958817e-02,   3.40695477e-02,  -5.56680419e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -7.28327784e-03,   5.76560879e-02,   2.31742080e-02],
                 [ -8.00889159e-03,   7.76066188e-02,   1.53271659e-02],
                 [ -2.06272343e-02,  -3.38955038e-02,   3.94944009e-02],
                 [ -2.94944589e-02,  -5.70488328e-03,  -8.50734334e-02],
                 [ -3.00688116e-02,  -3.51699810e-02,   7.46123212e-02],
                 [ -1.43651314e-02,  -1.65790974e-02,  -8.04817729e-03],
                 [  2.14764581e-02,   4.70646210e-03,  -2.13963125e-02],
                 [ -5.85921197e-02,   1.69218552e-02,  -2.77672935e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.52835081e-02,  -1.36361477e-02,   3.37650583e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.48611031e-02,  -2.08272545e-02,   1.91764943e-02],
                 [  2.23541087e-02,   1.77938953e-02,  -1.45302986e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.84480672e-02,   6.01102270e-02,   3.76987711e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.07214860e-03,  -3.35912465e-03,   5.49714895e-03],
                 [  5.27955886e-02,  -3.47510324e-02,   4.01721905e-02],
                 [ -1.00881530e-01,   4.71816458e-02,   9.28761745e-03],
                 [  1.62181709e-02,  -2.44129361e-02,  -1.29163690e-03],
                 [  5.46777449e-02,  -5.93421614e-02,  -3.13610298e-03],
                 [ -4.20668510e-04,   2.92869960e-03,   8.02401625e-02],
                 [  7.10287257e-03,   5.76638032e-02,  -4.47877040e-02],
                 [  5.66563056e-02,  -6.98329055e-03,   4.91238702e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.41815779e-02,   5.00259680e-02,  -3.24727409e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.28578543e-03,   4.94296757e-02,   2.37315025e-02],
                 [  4.38248083e-02,  -9.19765921e-02,  -2.94949773e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.47399884e-02,   2.06975676e-02,   6.14567223e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.62183540e-02,   3.83499702e-03,  -4.80384236e-02],
                 [ -1.67245948e-02,  -4.97980843e-02,  -1.40512911e-02],
                 [  1.04455709e-02,  -4.08084749e-02,   2.46333405e-02],
                 [  5.64098706e-03,   6.99279528e-02,   2.26647453e-02],
                 [ -2.04728792e-02,   2.68853672e-02,  -2.15623892e-02],
                 [ -4.57471451e-02,   1.25783449e-02,   5.47501230e-03],
                 [  2.93191531e-02,   5.86734263e-02,   2.34741342e-02],
                 [ -1.29962574e-02,  -1.57903968e-03,   6.52988409e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.21414309e-02,  -6.14581365e-02,   9.23584376e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.93937105e-02,  -3.55940990e-03,   1.16290547e-02],
                 [ -3.57884456e-02,  -1.44357435e-02,  -3.95051731e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.69698760e-02,  -4.73712290e-03,   3.54698382e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.81337398e-02,  -8.58154973e-02,  -4.30092519e-02],
                 [  6.80658558e-02,  -4.23592089e-02,  -1.47003501e-03],
                 [  2.20048744e-02,   1.07699976e-02,  -2.39409651e-02],
                 [ -5.01417973e-02,   5.64264836e-03,  -3.28227957e-02],
                 [  9.92148212e-03,   4.27027689e-03,   1.00693540e-02],
                 [  3.47742307e-02,   3.02406327e-02,   6.08529272e-02],
                 [ -4.94084743e-02,  -1.90506677e-02,   3.64824817e-02],
                 [  7.79038986e-03,  -7.47791929e-04,  -6.13715506e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.88758881e-02,   5.47340160e-02,  -2.11229463e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.66380639e-02,  -1.07878701e-02,  -3.09302652e-02],
                 [ -3.73679458e-02,  -7.01370301e-02,   8.49904581e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.20182809e-02,  -1.77739383e-02,   3.76824426e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.02468813e-02,  -2.90841827e-02,  -2.71696598e-02],
                 [  1.48332254e-02,   4.69769866e-02,   1.94090270e-02],
                 [ -1.91515229e-02,   5.54989440e-02,   3.76554856e-02],
                 [ -3.50680094e-02,  -4.01371526e-02,  -2.19268446e-02],
                 [  3.56250430e-02,   5.40629192e-02,   2.36526603e-02],
                 [  3.80013511e-02,   3.79787766e-02,   9.23854253e-03],
                 [ -3.68851376e-02,  -5.34466577e-02,   3.32433046e-02],
                 [  9.70565071e-03,   1.08395005e-02,  -3.68544868e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.44469647e-02,  -1.44529941e-02,   7.51938575e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.45910482e-02,   4.45002929e-02,   4.91310072e-02],
                 [ -4.67052701e-02,  -1.79169808e-02,  -1.06413466e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.10219072e-02,   1.68788458e-02,   7.68447351e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.54716193e-02,  -1.82923106e-02,  -8.37715063e-03],
                 [  7.00897876e-02,  -2.37312054e-02,   3.37375111e-02],
                 [ -2.79238817e-02,  -1.36327826e-02,   5.99542608e-02],
                 [  4.56769127e-03,   1.40689267e-02,   2.38165544e-02],
                 [  1.71816475e-02,   1.74077130e-02,  -2.56437632e-02],
                 [ -5.62675938e-02,   3.66240285e-02,  -7.97097190e-05],
                 [  7.31549013e-03,   3.49295259e-02,  -6.28447577e-02],
                 [ -7.02562221e-03,  -5.47434516e-02,  -1.35809539e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.02212525e-03,   2.13442331e-02,   6.00973501e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.46886543e-02,  -1.10483805e-02,  -9.70653971e-03],
                 [ -2.21531431e-02,  -5.45053004e-03,   1.08313813e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.76612262e-02,   3.51423172e-02,   6.51845650e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.78614536e-02,  -5.40025924e-02,   5.09587309e-02],
                 [  5.56871028e-02,  -3.11953986e-02,  -2.12540493e-02],
                 [  4.43228656e-02,  -5.53870929e-03,  -9.79397799e-03],
                 [  9.10148714e-03,   7.10595985e-03,  -1.77819800e-02],
                 [  1.38235506e-02,   6.35345981e-02,   3.95756842e-02],
                 [  2.27485236e-02,   5.83409155e-02,  -1.33324904e-02],
                 [  3.00529850e-02,  -3.06502736e-02,   7.46962649e-02],
                 [ -4.27672049e-02,   1.68907767e-02,   4.70051997e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.36369315e-02,   1.75201890e-02,  -4.59703258e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.64538258e-02,   2.32320813e-02,  -6.50475081e-02],
                 [  2.65994526e-02,  -8.45775561e-03,   4.57190123e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.84568182e-02,  -3.92641436e-02,  -5.31629528e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.29251741e-02,   3.04221920e-02,   2.52218517e-02],
                 [  2.47294710e-03,   7.53967933e-02,  -1.84393029e-02],
                 [  4.14393864e-02,   4.84134311e-02,   5.30784442e-02],
                 [ -4.74749114e-02,   4.68613513e-02,   4.11079906e-02],
                 [ -5.77153847e-02,   1.47061757e-02,  -5.16175571e-02],
                 [  1.55538544e-02,   5.78544828e-02,  -5.97501876e-02],
                 [ -5.66250996e-02,  -4.23074097e-02,   1.62218053e-02],
                 [  6.59035088e-02,  -3.77797228e-03,  -9.53488437e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.28686006e-02,  -1.80347288e-02,   2.75375468e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.48618064e-02,   2.07307928e-02,  -5.21478626e-02],
                 [  1.14988073e-02,  -4.28306969e-02,   2.37546513e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.53918431e-02,   2.14803049e-02,   1.07389745e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.20812434e-02,  -2.04565793e-02,  -8.07670088e-02],
                 [  1.25178839e-02,  -3.02268685e-04,  -1.46976349e-02],
                 [ -7.18360166e-03,   2.22971358e-03,  -5.72309807e-04],
                 [  5.58096074e-03,   6.80662317e-03,  -7.94786303e-02],
                 [ -5.74027887e-02,  -1.42192560e-02,  -3.54043260e-02],
                 [  2.48021612e-02,  -4.82773935e-02,   5.82634224e-02],
                 [ -2.55994136e-02,  -6.56305661e-02,   1.19249912e-02],
                 [  6.08676235e-02,   2.99067938e-02,  -7.19790029e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.34864660e-02,  -3.74544762e-02,   2.30385555e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.25460889e-02,  -8.50964265e-03,  -1.44661499e-02],
                 [ -4.08421977e-02,  -1.37142626e-02,  -3.76785510e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.98990583e-02,  -1.06453170e-02,   1.55054965e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.49030697e-02,   2.04837906e-02,  -1.39850220e-02],
                 [  6.98035586e-02,   2.97496046e-03,   1.21212039e-02],
                 [  1.93974526e-02,   3.26971927e-02,   5.29298774e-02],
                 [ -6.35621060e-02,   6.01731115e-02,  -3.63150722e-02],
                 [  8.15404909e-03,   2.54460278e-02,  -5.74711490e-02],
                 [  5.70766884e-02,   1.72198176e-04,  -1.33400295e-02],
                 [ -1.70279319e-02,   2.34212241e-02,  -1.93384301e-02],
                 [  9.49498950e-03,   2.07356477e-03,   8.20877378e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.60813561e-02,   2.69022762e-02,   3.29563602e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.85861104e-02,   1.52150820e-02,  -2.41376193e-02],
                 [  4.38978401e-03,  -6.25728285e-02,  -3.97189485e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.13477290e-02,  -1.13341543e-02,   1.91572194e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.81727208e-03,  -4.82067294e-03,   3.36672793e-03],
                 [  4.83129467e-02,   2.75898508e-02,   5.51554800e-02],
                 [  4.04676308e-02,  -4.03202407e-02,  -2.25175873e-02],
                 [ -3.70789395e-02,  -6.01977830e-02,   2.89862037e-03],
                 [  3.74217009e-02,  -7.23170370e-03,  -1.19074239e-02],
                 [ -5.26602010e-02,  -5.89831349e-02,  -1.59250973e-02],
                 [  5.85676787e-02,  -1.29205960e-02,  -1.00385386e-02],
                 [  1.42940766e-02,  -4.95121024e-03,   2.09871833e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.54432154e-02,   4.04159589e-03,  -7.55474552e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.04489126e-02,  -6.29108817e-02,  -1.07577934e-02],
                 [ -4.06510225e-02,  -4.13227014e-02,   5.64003386e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.77695378e-02,   2.10240207e-02,   2.18420670e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.88917713e-03,   9.30453371e-03,  -3.08963425e-02],
                 [  1.88903485e-02,  -4.05962521e-02,  -1.42559652e-03],
                 [ -5.53450252e-02,   6.31886613e-02,  -4.09865572e-03],
                 [ -1.51612738e-02,  -3.89378305e-02,  -3.50312258e-02],
                 [ -4.19805944e-02,   1.49690843e-02,   1.90053643e-02],
                 [  5.77934734e-02,   5.98367799e-03,   3.29077383e-02],
                 [  6.46251785e-02,  -4.07660307e-02,  -3.04233466e-02],
                 [  1.77904266e-02,   1.49406585e-02,  -3.82404960e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.05457989e-02,  -2.17699683e-02,   2.80599839e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.05993853e-02,   7.57910978e-02,  -3.71632948e-04],
                 [ -6.60264221e-02,   2.50458033e-02,   3.33571241e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -7.63675375e-03,  -7.66218551e-03,   1.90970742e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.93496972e-02,   4.00979642e-03,   5.05491112e-02],
                 [ -2.79072035e-02,   3.21872579e-02,   5.79319557e-02],
                 [  4.68367550e-02,  -3.02497694e-02,   7.38586107e-02],
                 [  7.93980565e-02,  -1.86035642e-02,  -2.56465699e-02],
                 [  3.96254149e-02,  -4.39482104e-02,   6.52946607e-03],
                 [ -4.99023000e-02,   1.04679612e-02,   2.67914116e-02],
                 [  3.40837896e-02,  -1.15100657e-02,  -6.05176733e-02],
                 [ -5.91847677e-02,   4.60643558e-02,  -3.03902918e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.16599273e-02,  -4.18465958e-02,   7.02616288e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.51119298e-03,  -4.19088517e-02,   4.91429589e-02],
                 [  5.75337517e-02,  -6.23626430e-02,   1.07050769e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.20725670e-02,   1.83310956e-02,   2.29123834e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.50546443e-02,  -3.73589761e-02,  -5.32315720e-03],
                 [  1.76617402e-02,   4.77908385e-02,  -1.16758722e-02],
                 [ -4.34092273e-02,  -5.86153948e-02,   1.56264661e-03],
                 [  5.97882536e-02,  -1.05094098e-02,   1.83770330e-02],
                 [  8.30695708e-02,   2.67545232e-03,   1.30789906e-03],
                 [ -2.80728664e-02,   6.20884076e-02,  -3.49798857e-02],
                 [ -3.97959760e-02,  -5.21608295e-02,  -2.11329188e-02],
                 [ -1.61342665e-02,   7.36692200e-03,   5.89254432e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.52102849e-02,   3.66599603e-03,  -7.48018244e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.52518257e-02,  -5.62916002e-02,   5.34668399e-02],
                 [ -7.14078851e-02,  -1.96407238e-02,   1.83270007e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.23351290e-02,  -4.57329228e-02,  -2.66026907e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.58487956e-02,  -2.02367830e-02,   4.51713248e-02],
                 [  3.49028806e-02,  -2.97239090e-02,   5.75001426e-03],
                 [ -8.11692222e-03,   5.99690379e-04,   7.63069276e-03],
                 [  3.49987637e-02,  -9.81677235e-03,   5.89384828e-02],
                 [  3.64702935e-02,   4.47060488e-02,   6.68978870e-03],
                 [  3.77441241e-02,   7.59521414e-02,   2.64470572e-03],
                 [  2.22251946e-02,  -5.67773690e-02,   2.57607365e-03],
                 [ -5.47414521e-02,  -3.09489330e-02,  -3.92288040e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.95971430e-02,   9.55136886e-03,   6.11470213e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.32514422e-02,   7.41089715e-02,   1.56186131e-02],
                 [ -1.82788206e-02,  -3.35255672e-02,  -2.86983643e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.02677922e-02,   1.52082475e-02,   2.52921338e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.76119947e-03,  -5.06560601e-03,  -6.69899368e-02],
                 [ -1.85460429e-02,  -2.48825741e-02,   1.98617815e-02],
                 [ -2.47052617e-02,   7.03820306e-02,   1.02073120e-02],
                 [ -1.73404505e-02,   3.48479864e-02,   6.39635332e-03],
                 [ -6.24635025e-03,   4.98399347e-02,   5.39575216e-02],
                 [  3.85944234e-02,   1.58118740e-02,   3.45055885e-02],
                 [  4.21503449e-02,   1.13001771e-02,   3.25883662e-04],
                 [  6.65392739e-02,  -8.58145474e-03,   2.87249932e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.63574875e-02,  -1.15272056e-03,  -9.13135237e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.23959343e-03,  -1.27331859e-02,  -5.25893983e-02],
                 [ -3.42285415e-02,  -9.79119626e-03,  -1.63856699e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.74732058e-02,  -3.75511989e-02,   5.03548115e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.12491579e-04,   1.87321922e-04,   6.52537392e-02],
                 [ -3.46938239e-02,  -3.74402601e-02,  -3.92899493e-02],
                 [ -3.98854025e-02,   2.94183996e-03,  -9.01695128e-03],
                 [ -1.43863156e-02,   3.67262356e-04,  -2.38121874e-03],
                 [ -1.08122194e-02,   3.48456874e-02,  -6.76172266e-02],
                 [  2.78680273e-02,   3.07619760e-02,  -1.63485837e-03],
                 [ -2.66720566e-02,   6.16462768e-02,   1.62224864e-02],
                 [ -3.86526993e-02,  -5.78705403e-04,   7.17955788e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.33626272e-02,   3.01211167e-02,  -3.53314621e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.28290639e-02,  -3.97039936e-02,  -8.85368573e-02],
                 [ -3.25787712e-02,  -5.77106990e-02,   3.72561440e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.76880171e-03,   8.37222638e-03,  -2.62445933e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.51307268e-02,  -2.35720373e-02,  -4.87150949e-02],
                 [ -2.41709369e-02,  -2.67489199e-02,  -5.83386042e-02],
                 [  9.11114897e-03,  -2.90152566e-02,  -4.82701048e-02],
                 [ -3.55460821e-02,  -1.42572535e-02,   1.53988554e-02],
                 [ -4.65534263e-02,   1.53393314e-02,  -9.52730230e-03],
                 [ -1.39474947e-02,   5.19781266e-02,  -2.81701549e-02],
                 [ -1.68847697e-02,   7.83575176e-03,   7.26442359e-03],
                 [ -8.71005185e-03,  -3.37802194e-02,  -8.79337291e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.14135639e-02,  -1.53888385e-02,   1.86789494e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.28495708e-02,   1.87193347e-02,  -2.77530083e-02],
                 [  7.38896481e-02,   8.48142231e-03,   1.39350374e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.05241635e-02,   4.77866754e-02,   4.74145155e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.98760729e-03,  -6.08814732e-03,   6.35402000e-02],
                 [  1.71814641e-02,   4.25078993e-02,   2.22531377e-02],
                 [ -5.22572887e-03,  -5.45414714e-02,   1.20097122e-02],
                 [ -1.15883162e-02,   6.01295785e-03,  -4.13964119e-02],
                 [  3.04079630e-02,  -3.10251165e-02,  -9.20129631e-03],
                 [ -7.50010997e-02,   3.83058585e-02,  -3.00568634e-02],
                 [ -3.73789143e-02,  -3.23411084e-02,   3.62600714e-03],
                 [ -8.09461832e-03,   2.71879332e-02,  -7.78539754e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.99460040e-02,   3.46991418e-03,  -2.37370594e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.82168896e-02,   9.57413048e-03,   3.22268580e-02],
                 [ -5.11193258e-02,  -4.64161090e-02,  -2.30027037e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.43063911e-02,  -1.28049988e-02,  -1.86165502e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.06115219e-02,  -6.89160873e-02,   3.35240751e-02],
                 [ -3.06390125e-02,   4.01735362e-03,  -4.59792688e-02],
                 [  3.88217914e-02,   1.07686369e-02,  -2.65612874e-02],
                 [  1.65096811e-02,   3.63044306e-02,  -1.83191806e-02],
                 [  6.69530293e-03,  -4.98393672e-02,  -4.93580643e-02],
                 [  4.42520822e-02,  -1.15224911e-02,   5.74442073e-02],
                 [  1.75552906e-02,  -4.63252847e-02,   2.07161279e-02],
                 [ -8.80569860e-03,  -4.72776953e-03,  -1.69854752e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.19044888e-03,  -4.67791534e-02,   6.17168193e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -7.14599482e-03,   5.67572037e-02,  -4.25893975e-02],
                 [ -1.70660335e-02,   9.04896953e-03,   2.27320290e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.47390884e-02,   4.02212212e-02,  -1.29702355e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.35532644e-02,  -7.18526200e-02,   3.03748450e-02],
                 [  8.07107419e-02,  -1.27153179e-02,  -8.07828697e-03],
                 [ -4.17449341e-03,   2.34155647e-02,   4.29984095e-03],
                 [ -7.60640529e-03,  -1.71250283e-02,   4.73039541e-02],
                 [ -3.56834402e-02,  -2.14447372e-02,   7.35575223e-02],
                 [ -6.43951860e-02,   9.66567943e-03,  -3.62088976e-02],
                 [ -7.94691655e-02,   2.00405261e-02,   2.50440565e-02],
                 [ -2.87808375e-02,  -4.11882970e-02,  -3.60009886e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -8.22203882e-03,  -5.01321325e-03,  -1.39755375e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.89382271e-02,  -2.32446338e-02,  -3.55133798e-02],
                 [  2.87472235e-02,   5.18552169e-02,  -5.06246464e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -9.81748500e-03,  -1.26885524e-03,  -3.10433001e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.01780935e-02,  -4.09809363e-03,   4.50120236e-02],
                 [  1.92423247e-02,   5.33087157e-02,   3.41667409e-02],
                 [  9.76036633e-04,  -5.56150090e-03,  -1.69484172e-02],
                 [ -2.24470021e-03,   1.42731873e-02,   3.10067313e-02],
                 [ -6.19099203e-02,  -6.05412763e-04,   6.76195378e-03],
                 [  4.04836651e-02,  -4.14874413e-02,  -6.45150803e-03],
                 [  1.14349433e-03,   2.86589763e-03,  -5.23137773e-02],
                 [ -5.83744097e-02,   3.51880572e-02,  -1.36656834e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.33872661e-03,  -2.76474070e-02,  -5.75218710e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.57444678e-02,   3.52681176e-03,   2.89551681e-02],
                 [  3.10688248e-02,   5.73184286e-02,  -1.99184941e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.18999246e-02,   5.23530112e-02,  -4.54782848e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.01765623e-03,  -1.20240308e-02,   6.27962719e-03],
                 [  7.25258913e-05,  -6.59869146e-02,   5.41906292e-03],
                 [  2.40356735e-02,   4.68930062e-02,  -7.74778452e-02],
                 [  2.63978210e-02,   2.06367940e-02,   5.75526092e-02],
                 [ -6.07735296e-02,  -4.70278697e-03,   2.06555032e-02],
                 [ -1.50512463e-02,   3.61035732e-02,  -1.07555993e-02],
                 [  4.86148933e-02,   1.26708352e-03,  -3.81007194e-02],
                 [ -1.73427890e-02,  -2.80263248e-03,  -2.76112050e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.56814392e-02,  -4.61309943e-04,   4.72464063e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.51866070e-02,   7.85038697e-02,  -3.37376549e-02],
                 [  5.24763936e-02,   2.87635608e-02,   4.18440090e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.05205969e-02,   8.80308946e-03,  -1.32719332e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.84884268e-02,  -1.39244917e-02,   5.57270049e-02],
                 [  1.23761227e-02,   5.70008827e-02,   2.54692511e-02],
                 [ -2.33646554e-02,   1.46066056e-02,  -8.33704780e-03],
                 [ -2.83786067e-02,   6.72792560e-02,  -2.57958410e-02],
                 [ -2.38471118e-02,   6.25172755e-02,   1.01899717e-02],
                 [ -7.15516448e-03,   2.98687135e-02,   6.11836665e-02],
                 [ -2.57734373e-02,   3.34647116e-02,   6.56794705e-02],
                 [  6.93791691e-02,  -4.95134782e-03,  -7.90937611e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.52162276e-02,   1.77740734e-02,  -3.08923585e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.91463757e-03,  -2.36245247e-02,   4.06012332e-02],
                 [  4.71624151e-02,   5.51903859e-02,   1.09073649e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.24619478e-03,   5.69622791e-02,  -2.21662935e-02]]).T)


         def setUp(self):
            self.xml = """<TS_params 
        betapol="0.75" 
        cutoff="20.0 18.0 18.0 18.0"
        cutoff_ms="18.0"
        cutoff_coulomb="20.0"
        tolpol="1e-15" 
        yuksmoothlength="20.0" 
        iesr="2 2 2" 
        a_ew="1e-06" 
        n_types="2" 
        gcut="0.0" 
        pred_order="2" 
        maxipol="60" 
        raggio="0.0" 
        tewald="F" 
        yukalpha="0.1"
        tdip_sr="F">

        <per_type_data atomic_num="8" pol="1.0" z="-1.95853" type="1" />
        <per_type_data atomic_num="14" pol="0.0" z="3.91706" type="2" />

        <per_pair_data atnum_i="8"  atnum_j="8"  C_pol="0.46009932"  D_ms="0.0" gamma_ms="11.642637" B_pol="0.87357114" R_ms="8.0465068" />
        <per_pair_data atnum_i="8"  atnum_j="14" C_pol="-1.5091142"  D_ms="0.0" gamma_ms="10.405794" B_pol="1.977039"   R_ms="4.193651"  />
        <per_pair_data atnum_i="14" atnum_j="14" C_pol="0.0"         D_ms="0.0" gamma_ms="4.5784138" B_pol="0.0"        R_ms="13.113727" />
      </TS_params>
      """
            if do_compare_p1_p2:
               self.p1 = Potential('IP ASAP', param_str=self.xml)
            self.p2 = Potential('IP TS', param_str=self.xml)
            self.cutoff = 20.0*BOHR
            self.debug = False


      class TestDipoleShortRange(QuippyTestCase, PotTestMixin):

         dimer_ref = (FortranArray(-17.811006977457581), FortranArray([[ 18.94240662,   0.        ,   0.        ],
                 [-18.94240662,   0.        ,   0.        ]]).T, FortranArray([[-30.49267528,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ]]).T, FortranArray([-8.93226931, -8.87873767]), FortranArray([[ 0.       ,  0.       ,  0.       ],
                 [ 0.0627256,  0.       ,  0.       ]]).T)

         trimer_ref = (FortranArray(-33.563525193963763), FortranArray([[ 17.19744833,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [-17.19744833,   0.        ,   0.        ]]).T, FortranArray([[-55.36743225,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ],
                 [  0.        ,   0.        ,   0.        ]]).T, FortranArray([ -7.85585209, -17.85182102,  -7.85585209]), FortranArray([[-0.05527469,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [ 0.05527469,  0.        ,  0.        ]]).T)

         quartz_ref = (FortranArray(-144.79740179687568), FortranArray([[ -4.92225246e-01,   8.52559136e-01,   2.42526623e-15],
                 [ -4.92225246e-01,  -8.52559136e-01,   1.04930082e-15],
                 [  9.84450493e-01,   2.37044520e-15,  -7.23613974e-15],
                 [  3.78732762e+00,  -6.47630354e+00,   4.23003219e+00],
                 [  3.71497958e+00,   6.51807370e+00,   4.23003219e+00],
                 [ -7.50230719e+00,  -4.17701600e-02,   4.23003219e+00],
                 [  3.78732762e+00,   6.47630354e+00,  -4.23003219e+00],
                 [ -7.50230719e+00,   4.17701600e-02,  -4.23003219e+00],
                 [  3.71497958e+00,  -6.51807370e+00,  -4.23003219e+00]]).T, FortranArray([[ -7.29045857e+01,  -5.77736150e-15,   1.57770946e-14],
                 [ -5.80585293e-15,  -7.29045857e+01,   3.50407202e-14],
                 [  1.57152070e-14,   3.50435723e-14,  -7.26577486e+01]]).T, FortranArray([-28.21066338, -28.21066338, -28.21066338, -10.02756861,
                 -10.02756861, -10.02756861, -10.02756861, -10.02756861, -10.02756861]), FortranArray([[ 0.        ,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [ 0.        ,  0.        ,  0.        ],
                 [-0.01017154,  0.01531192, -0.01028628],
                 [-0.00817474, -0.01646477, -0.01028628],
                 [ 0.01834628,  0.00115285, -0.01028628],
                 [-0.01017154, -0.01531192,  0.01028628],
                 [ 0.01834628, -0.00115285,  0.01028628],
                 [-0.00817474,  0.01646477,  0.01028628]]).T)

         bigquartz_ref = (FortranArray(-293.30383805940176), FortranArray([[-2.08415672, -2.23007886,  3.74869236],
                 [ 3.16451334, -0.59185079, -0.31440948],
                 [ 2.04159262, -1.41858379,  4.18632826],
                 [ 6.41344975, -4.14899642,  4.54377578],
                 [-1.16409577,  6.64240981,  8.88166689],
                 [-5.01788994, -0.59216369,  1.88547306],
                 [ 3.73907778,  9.16476719,  0.30075182],
                 [-8.31315417, -0.92273074, -8.22990847],
                 [-0.21573686, -8.15142253, -6.35533748],
                 [ 3.3386908 ,  1.24920556, -1.50772306],
                 [-2.38418468, -3.48960376,  3.15056973],
                 [ 0.91358002, -4.37357124, -0.04218135],
                 [ 4.0269014 , -4.67971601,  4.05462354],
                 [ 9.64370758,  5.31059828,  2.45489693],
                 [-8.08774173,  6.32318443, -0.42988191],
                 [ 0.93668411,  7.60451508, -4.71403328],
                 [-7.48810435,  2.14860634, -7.22439801],
                 [ 0.53686683, -7.84456887, -4.38890531]]).T, FortranArray([[-147.13389319,   -2.39672914,   -5.20789094],
                 [  -2.39672914, -145.18823505,   -4.52055773],
                 [  -5.20789094,   -4.52055773, -148.43587042]]).T, FortranArray([-26.29431962, -29.49983881, -27.84408267,  -9.39524437,
                 -10.00607828,  -8.27638007, -10.15057588, -10.24336127,
                 -11.11087648, -29.96689858, -28.56601455, -29.07165033,
                 -10.16627649, -10.86878858,  -9.47024297,  -9.96695783,
                 -12.05319564, -10.35305564]), FortranArray([[  0.00000000e+000,   2.61854792e-322,   0.00000000e+000],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [ -1.29260253e-002,   1.63677234e-002,  -3.84011947e-003],
                 [ -8.14834924e-003,  -1.16193389e-002,  -1.26546262e-002],
                 [  1.24785221e-002,   2.56647966e-003,  -2.70009262e-003],
                 [ -1.39423455e-002,  -1.59807080e-002,   1.77527876e-003],
                 [  2.33497524e-002,   7.38173497e-003,   1.88430019e-002],
                 [ -9.62827443e-003,   1.65234068e-002,   4.18689651e-003],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [  0.00000000e+000,   0.00000000e+000,   0.00000000e+000],
                 [ -1.35080460e-002,   3.33361771e-003,  -1.22585519e-002],
                 [ -8.26055231e-003,  -1.37492167e-002,  -1.53129065e-002],
                 [  1.32799696e-002,  -4.61174276e-003,  -6.72856511e-003],
                 [ -1.06221191e-003,  -1.78793839e-002,   1.50772946e-002],
                 [  1.57100029e-002,  -3.03022644e-003,   1.28736848e-002],
                 [ -1.61885301e-004,   1.84258788e-002,   7.61293906e-003]]).T)

         am_ref = (FortranArray(-13653.518031289135), FortranArray([[  1.66636276e+00,   1.32054000e+00,   2.32275027e+00],
                 [ -1.74795115e+00,  -6.83025394e-02,   1.71340684e-02],
                 [ -1.16291579e-01,  -1.51074776e+00,   5.65960963e+00],
                 [  6.58502390e+00,  -7.98750704e+00,   1.32519932e+01],
                 [ -7.85518557e+00,   2.93361193e+00,   6.55128359e+00],
                 [  4.44504100e+00,   1.03311156e+00,  -3.01803227e+00],
                 [  4.05463032e+00,   5.94933045e+00,  -8.17955791e+00],
                 [  1.04042646e+01,  -5.97867716e+00,   1.66460556e-01],
                 [  6.36152639e-01,   8.77000471e+00,  -6.60040426e+00],
                 [ -1.27214827e+00,  -1.19657296e+01,  -2.18143567e+00],
                 [ -4.36786797e+00,  -4.34842740e+00,  -1.15081705e+01],
                 [  1.33770419e+00,   2.85034906e+00,   1.07371650e+00],
                 [  6.42631114e+00,  -1.03098944e+01,  -4.01749952e+00],
                 [ -9.68310915e-01,   6.06608100e+00,   1.52017861e+00],
                 [ -2.33328914e+00,   2.46369297e+00,  -7.91551835e+00],
                 [  9.33033968e+00,  -2.02007643e+00,   1.43393525e+01],
                 [  1.65267588e+00,   3.42807557e-01,   4.48448679e-01],
                 [ -1.01682990e+01,  -1.86938595e+00,   3.02729816e+00],
                 [  2.45282982e+00,  -3.50405586e-01,  -1.12508366e+00],
                 [  5.19820690e-01,   1.53491002e+00,   1.62469784e-01],
                 [  1.58060711e-01,   4.31385614e-01,   1.47538968e+00],
                 [  5.21559105e+00,   7.77283150e+00,   1.89545881e+00],
                 [  1.76924400e+00,   4.67808399e+00,   5.08754730e+00],
                 [  4.62437458e-01,   3.75987588e+00,   2.75851761e+00],
                 [  4.14050350e+00,   8.28516299e+00,   1.62898525e+00],
                 [ -5.19935359e+00,  -5.18926697e+00,   4.01221701e+00],
                 [  5.56027093e+00,  -7.49427913e+00,   3.73263110e+00],
                 [ -2.34309706e+00,  -2.19446480e+00,  -1.74438386e+00],
                 [  9.28233053e+00,   5.98857249e-01,   3.12994329e+00],
                 [ -2.22096254e-01,  -1.71615647e+00,  -1.20627065e+00],
                 [ -9.09618178e+00,   6.91896366e-01,   1.91479658e-01],
                 [  6.42530728e-01,  -7.23329259e-01,   3.20907562e-01],
                 [ -1.13646114e+01,  -4.07215079e-01,   4.32969859e+00],
                 [ -9.32961101e+00,  -3.66422211e+00,   2.15314447e+00],
                 [  4.96233372e-01,   1.49280345e+00,  -1.98861271e+00],
                 [ -2.62710935e+00,   7.57526791e-01,  -1.36419448e+00],
                 [ -1.37657184e-01,   1.85261035e+00,   1.93183626e+00],
                 [ -1.00784528e+00,   3.27249131e+00,  -7.71140125e-01],
                 [ -1.11309979e+00,   8.10606097e-01,  -1.44247955e+00],
                 [ -2.48430058e+00,  -2.51626198e+00,   1.17927478e+01],
                 [  8.50973481e+00,   3.99104242e+00,   1.14207508e+01],
                 [  3.69470582e+00,   8.62566809e+00,   1.01080957e+00],
                 [ -2.93711807e+00,  -8.13391835e+00,   8.62482968e+00],
                 [ -8.89896642e+00,  -5.92732746e+00,  -4.00564039e+00],
                 [  4.85907186e+00,   5.47514975e+00,   2.53289265e+00],
                 [  2.20196022e+00,   1.08643964e+01,  -6.79376003e-02],
                 [  4.13608038e+00,  -1.54459651e+00,   2.31740845e+00],
                 [  1.18219795e+00,   1.16211372e+00,   1.71048325e+00],
                 [  6.10270192e+00,  -2.14077754e+00,   1.55038926e-03],
                 [ -8.60570644e-01,  -1.80502876e-01,  -1.92370643e+00],
                 [ -3.21726230e+00,  -1.16908706e+01,  -2.91353049e-01],
                 [ -3.84704900e+00,   2.77603382e+00,  -1.20877646e+01],
                 [ -7.72237505e-01,  -2.58928109e+00,  -1.01488870e+00],
                 [ -8.37904527e+00,   6.09699520e+00,  -7.20964830e+00],
                 [ -2.18384365e+00,  -1.60334079e+00,  -2.80076246e+00],
                 [ -1.04945239e+00,  -3.80163798e+00,  -1.39972147e+00],
                 [ -3.55492039e+00,   3.44680863e-01,  -1.72987327e+00],
                 [  8.08930372e+00,  -4.06429199e+00,   1.43213278e+01],
                 [ -8.53392335e+00,   5.30557501e+00,  -4.26727382e+00],
                 [  5.21606314e+00,  -9.89976047e+00,  -7.02210966e+00],
                 [  1.79416299e+00,   5.56083761e+00,   3.20448821e+00],
                 [  2.25445327e+00,   7.36635846e+00,  -5.43908781e+00],
                 [  1.54177437e+00,  -5.38338693e+00,  -8.27162428e+00],
                 [ -1.15490373e+01,   5.00130475e+00,  -2.57674272e+00],
                 [  9.78922810e-01,   1.40291162e+00,   8.78737098e+00],
                 [ -2.33892974e+00,   1.75581135e+00,   3.68116575e+00],
                 [  6.16769437e+00,  -5.89551194e+00,  -4.16651461e+00],
                 [ -8.97539560e-01,   1.00847638e+00,   1.21484223e+00],
                 [ -7.94763277e+00,  -9.88352714e-01,  -4.36745296e-01],
                 [ -7.07384310e+00,   3.02475074e+00,   9.43211172e+00],
                 [ -1.09481580e+00,   8.55846044e-01,  -1.08395402e+00],
                 [  1.06580602e+01,   2.55076897e-01,   7.54568573e+00],
                 [ -2.28260302e+00,   1.79242174e+00,   1.38846676e+00],
                 [  5.13490534e-01,  -1.09549915e+00,   6.87560609e-01],
                 [  6.21371420e-02,   1.75392961e+00,   1.02768475e+00],
                 [  8.52224051e-01,   6.92078986e+00,   3.20574860e+00],
                 [ -3.10982449e+00,  -3.30873084e+00,  -1.99736556e+00],
                 [  4.80709982e+00,  -4.48559325e+00,  -1.01006163e+01],
                 [  4.61156501e+00,  -4.54231580e+00,   8.12952604e+00],
                 [  2.69599696e+00,   4.57760552e+00,  -7.11349762e+00],
                 [ -1.00928548e+01,  -6.00132599e+00,  -3.96251884e+00],
                 [  8.60092714e-01,   1.27662094e+00,  -4.16756507e+00],
                 [ -3.13145830e+00,  -6.11538874e+00,  -5.20586388e+00],
                 [ -1.11507570e+00,   3.96865501e+00,  -9.68846083e-03],
                 [ -2.89600923e+00,  -9.23916633e+00,  -2.40234889e+00],
                 [  3.86582125e+00,  -2.33486000e+00,  -5.40476866e-01],
                 [ -6.89754355e+00,   1.13673007e+01,  -1.74752536e+00],
                 [  7.24993720e+00,  -4.86782281e+00,  -3.00871981e+00],
                 [ -2.06922855e+00,   3.49100096e-01,   1.06637417e+00],
                 [  6.46708320e-01,   7.88232490e+00,  -6.53046882e-01],
                 [  8.63100879e-01,  -1.40931868e+00,  -1.79899174e+00],
                 [  1.33084252e+00,   6.04497417e-01,  -1.96258164e+00],
                 [  1.03807804e+00,   2.69608198e-01,   6.28488683e-01],
                 [  7.26140546e+00,  -5.24389913e+00,  -2.88326316e+00],
                 [  7.59911142e+00,   4.18844594e+00,  -5.73260226e+00],
                 [  1.59035680e+00,  -8.14162306e+00,  -8.25507913e+00],
                 [ -8.13378007e+00,  -5.03815182e-01,  -6.92783379e+00],
                 [  2.05573880e+00,   1.87318245e+00,   8.64775723e+00],
                 [  8.61646724e-01,  -7.57688812e+00,   5.61232708e+00],
                 [ -2.74701587e-01,  -2.20538296e+00,  -1.06357189e+01],
                 [ -2.17947971e+00,  -8.62094846e-01,  -5.10639286e+00],
                 [ -3.62257364e-01,  -1.13786729e+00,  -1.50002026e+00],
                 [ -2.94719730e+00,  -1.10394303e+01,   2.91318613e+00],
                 [ -2.11010301e-01,  -4.73853983e+00,  -3.24749703e-02],
                 [ -6.05173250e+00,   1.11532209e+01,  -1.64022651e+00],
                 [ -4.90232432e+00,  -7.31474964e+00,  -1.53128377e+01],
                 [ -2.62066482e+00,  -1.43238378e+00,   3.48727916e+00],
                 [  4.01575722e+00,   8.98062326e+00,  -3.76382123e+00],
                 [ -2.11538201e+00,  -5.51694772e-01,   5.40797778e-01],
                 [  2.99545962e-01,  -3.14029175e+00,   1.27620664e+00],
                 [  6.44201774e-01,   2.13969971e+00,   1.99822131e+00],
                 [  1.69055354e+00,  -6.45894194e+00,  -7.10589355e+00],
                 [  3.75744870e+00,   1.43257886e+00,  -5.38890205e-01],
                 [  1.67981950e-01,   1.28070089e+01,   4.42752481e+00],
                 [  6.58142826e+00,   4.33989833e+00,   3.95109872e+00],
                 [ -8.08206175e+00,  -1.20094990e+00,  -4.04457059e+00],
                 [ -3.84855937e+00,   4.91107122e+00,   1.00494580e+01],
                 [ -8.51238673e+00,   6.28798609e+00,   3.70643470e+00],
                 [ -2.71482063e+00,   5.58921010e+00,   7.38841212e+00],
                 [  3.41033813e+00,   2.70173824e+00,   2.73281319e+00],
                 [  1.65859600e+00,  -1.86238406e+00,  -3.28373031e-01],
                 [  2.35128025e-01,  -3.29381849e+00,  -1.92626279e-01],
                 [  1.48301386e+00,  -5.34322641e+00,  -3.91162724e+00],
                 [  5.40276220e+00,   7.02598606e+00,   7.98491342e+00],
                 [ -6.94366267e-01,   3.04779364e-01,   3.03414480e+00],
                 [ -5.33664059e+00,   9.39566955e+00,   2.34074459e+00],
                 [ -1.13063171e+00,   2.07736910e+00,  -9.85650638e-01],
                 [  3.80397256e+00,  -1.09118109e+00,   2.39995041e+00],
                 [ -4.45681297e-01,  -3.41386035e-01,   1.78155093e+00],
                 [  2.41011856e+00,  -4.73616212e+00,  -2.94520123e-01],
                 [  8.18012969e-01,  -3.56207018e+00,  -2.76150890e-01],
                 [ -3.39474082e+00,   4.53177936e+00,  -9.19195926e+00],
                 [ -9.63572271e+00,   6.83929421e+00,   1.71028225e+00],
                 [  4.18423584e+00,  -4.03882884e+00,   4.68175032e+00],
                 [  3.96934326e+00,  -1.27192465e+01,   2.59143250e+00],
                 [  2.56389178e+00,   9.30044017e-01,  -1.14474955e+01],
                 [ -9.91749145e+00,   4.09580810e+00,  -4.64207604e-01],
                 [ -4.51543236e-02,   1.05253408e+00,   6.35693026e-01],
                 [  4.77399143e+00,   5.06787075e+00,  -6.69191158e+00],
                 [  1.07177609e+00,   1.14352264e+00,  -1.03695014e+00],
                 [  6.20777289e+00,  -6.92307791e+00,   5.67504488e+00],
                 [ -1.78635654e+00,   8.76800005e-01,   4.75886482e+00],
                 [ -3.27039119e-01,   5.40037985e-01,   9.75282211e-03],
                 [  4.70571086e+00,   1.66338655e+00,  -1.08111887e+01],
                 [  6.26964713e-01,   4.80815601e-01,   1.42702481e-01],
                 [  2.39103812e+00,   1.45497568e+00,   1.00216153e+00],
                 [  5.08185546e-01,   4.70977506e-01,   2.75530287e+00],
                 [ -5.05231651e+00,  -9.35841573e+00,   9.79582689e-01],
                 [ -9.80010733e+00,   6.90869565e+00,  -2.85215132e+00],
                 [  3.16174347e+00,   1.31865189e+01,  -5.31091816e+00],
                 [ -1.11770298e+00,  -7.80763922e+00,  -3.26493167e-01],
                 [ -4.06563829e+00,   2.51176287e+00,   9.75447730e+00],
                 [ -2.49756821e+00,   1.00051764e+01,   8.37746360e+00],
                 [ -7.01342388e+00,  -6.98623289e+00,  -6.66547093e+00],
                 [  7.60563629e+00,   4.17716651e+00,   4.15920915e+00],
                 [  1.14936429e+00,  -2.93708515e+00,  -6.65497763e-02],
                 [ -1.02280477e+00,  -6.41203002e+00,  -1.03057566e+01],
                 [  1.06099800e+00,   3.83057123e+00,   2.20545463e+00],
                 [ -5.94925703e+00,   8.02657670e+00,  -1.05021283e+00],
                 [  9.40914522e+00,   7.43813386e-01,  -8.66869957e-01],
                 [  1.05533826e+00,   3.37126361e-01,  -6.75215795e-02],
                 [ -1.90730661e+00,   1.40391363e+00,  -1.17886597e+00],
                 [  1.41526188e+00,   4.35562588e-01,  -9.66004635e-01],
                 [  2.01119875e+00,  -3.81101964e+00,   8.81254471e-01],
                 [ -1.28458258e+00,  -6.72091691e-01,   3.76187873e-01],
                 [  3.68287080e+00,   6.68653545e+00,  -3.48859280e+00],
                 [  3.08764423e+00,   6.22696012e+00,   3.06617068e+00],
                 [  1.15219200e+01,   2.89894274e+00,  -4.21296083e+00],
                 [  3.20161642e-01,   9.00478077e+00,   3.18528889e+00],
                 [  1.03657221e+00,  -4.76605096e+00,   1.02036141e+01],
                 [  1.80822412e+00,  -1.19441105e+01,  -1.04726444e+00],
                 [  1.48831760e+00,  -7.16719641e+00,  -2.56413226e+00],
                 [  5.39889527e+00,   5.00166535e+00,  -7.81424122e+00],
                 [  2.44794304e+00,  -3.39426145e+00,  -3.61623604e-01],
                 [  2.84397956e+00,  -4.12380011e+00,  -5.93672526e+00],
                 [  9.85873518e-01,   4.28022871e-01,  -8.19430616e-01],
                 [ -5.91677503e+00,   3.03243236e+00,   9.71238926e+00],
                 [  3.97229880e-01,   2.51351403e+00,   9.00278928e+00],
                 [  7.01862167e-01,   1.97564707e-01,  -1.73738568e+00],
                 [  6.25844493e+00,   3.38749822e+00,   6.73521171e+00],
                 [ -1.30096026e+00,   3.68973939e-01,   8.98451183e-01],
                 [ -3.05200754e-01,   8.19985997e-01,  -1.18464062e+00],
                 [ -1.32500491e+00,   4.51088553e-01,   1.49054044e+00],
                 [ -8.06467516e+00,  -9.52405891e-01,   7.25139474e-01],
                 [ -8.12706571e+00,  -9.98006127e+00,   2.65371486e-02],
                 [ -4.41868443e+00,   3.27341509e+00,  -2.39790823e+00],
                 [ -6.11482221e+00,  -9.34455060e-01,   5.70031993e+00],
                 [ -3.93826570e-02,   1.95015916e+00,   4.64350814e+00],
                 [ -6.78866309e-01,   2.43594960e+00,  -4.06380731e+00],
                 [  5.22587265e+00,  -7.48227683e+00,  -2.32513705e-01],
                 [ -3.03434086e+00,   2.08385812e+00,   2.60117750e+00],
                 [ -1.46756873e+00,  -1.27217270e+00,   1.55898824e+00],
                 [  3.05109281e-01,  -9.52201568e-02,   9.66834094e-02],
                 [ -2.27325487e+00,   2.51685328e+00,   3.43035489e-01],
                 [  7.82249916e+00,  -1.63708652e+00,   9.84075408e+00],
                 [ -7.35132289e+00,   8.61429617e+00,  -3.12858116e+00],
                 [  2.60973692e+00,   3.40401409e+00,  -2.28046982e+00],
                 [  6.04685177e-01,   8.15384495e+00,   2.65543821e+00],
                 [ -1.33123614e+00,   1.47768318e+00,  -1.19410354e+00],
                 [ -1.91225525e+00,   6.50361231e-01,  -4.65498247e-01],
                 [  2.60558196e+00,  -2.33580525e+00,   1.74193821e+00],
                 [  3.84227454e+00,   5.58971098e+00,   9.13903962e+00],
                 [  5.58464620e+00,  -7.85476329e-01,  -4.68505388e+00],
                 [  6.54007629e-01,   8.23680427e+00,  -4.07341626e+00],
                 [  8.39258441e+00,  -2.04886717e+00,   4.81855597e+00],
                 [  3.86061501e+00,   7.98233692e+00,   1.44582197e+00],
                 [ -1.62478997e+00,  -1.36168951e+01,  -4.08063243e-01],
                 [  3.90532995e-01,  -8.65292530e+00,  -4.75585610e+00],
                 [ -7.64183306e-02,  -5.90850277e+00,   3.82324564e-01],
                 [ -4.09783810e-02,   8.76676883e-01,  -1.00099323e+00],
                 [  2.55406207e+00,  -3.59195691e+00,   5.06596746e+00],
                 [  2.15204837e+00,   1.15310997e+00,   1.52940547e+00],
                 [ -3.51254254e+00,  -5.31137648e+00,   8.63697515e-01],
                 [ -1.63198017e+00,  -5.38051447e+00,  -3.02583644e+00],
                 [  2.46827545e+00,  -1.33218622e+00,  -3.30875577e+00],
                 [ -4.36394192e+00,   2.25550176e+00,  -2.77503885e+00],
                 [  1.16125849e+00,   2.42092695e+00,   1.29501127e+00],
                 [  1.51162268e+00,  -6.42122648e-01,   1.12410335e+00],
                 [ -1.51661998e+00,  -2.06148196e-01,   1.90471820e+00],
                 [  5.79940295e-01,   2.59361836e+00,  -3.51337058e+00],
                 [  1.02939017e+00,   1.03704406e+01,  -6.23905048e+00],
                 [ -3.46237549e+00,  -1.14902386e+01,  -6.07757113e+00],
                 [ -1.68030789e+00,   7.37155713e+00,  -8.46141126e+00],
                 [  6.00229849e+00,   5.90426898e+00,  -3.60148182e+00],
                 [  6.19770736e+00,   2.24124688e+00,   1.13049933e+01],
                 [  8.49818304e-01,  -1.07918394e+01,   3.35918532e+00],
                 [  5.32611437e+00,  -6.04626917e-01,   2.54878897e+00],
                 [ -4.01656772e+00,   1.12431875e-01,   8.77809971e-01],
                 [  1.25145372e-01,  -6.51840640e+00,   4.98399943e+00],
                 [ -3.51800389e+00,  -3.33689286e+00,   2.24348058e+00],
                 [  4.39246061e+00,  -1.27437170e+01,   3.49319225e-01],
                 [  8.92526774e-01,  -1.22551804e+01,  -5.94527481e+00],
                 [ -1.71206046e+00,  -1.13811042e+00,  -6.21535333e-02],
                 [ -9.06500525e+00,   7.21364055e+00,   1.51970746e+00],
                 [ -8.25195543e-01,  -1.82805694e+00,   1.22930332e+00],
                 [ -1.19189612e-01,  -1.09606096e-01,   6.90749664e-01],
                 [ -2.91984880e+00,  -1.72618316e+00,  -3.64979005e-01],
                 [  1.08046693e+00,  -6.51735848e-01,   1.28684931e+00],
                 [ -7.41962781e+00,   5.68867607e-01,  -9.64675913e-01],
                 [  2.25481899e+00,   3.57305399e+00,   4.69508227e+00],
                 [ -8.93014864e+00,  -3.52557475e+00,  -6.48151322e+00],
                 [ -3.67715454e+00,  -7.84276370e+00,  -8.00465688e+00],
                 [ -9.87854510e-02,   1.21202719e+00,   1.72373122e+00],
                 [  1.85846407e+00,   1.07595953e+00,  -5.55567881e+00],
                 [ -1.37834291e+00,   7.81578587e+00,   3.01969339e+00],
                 [ -2.80895874e-01,   6.42033197e-01,  -4.46797749e-01],
                 [ -9.16452449e+00,   7.57692173e+00,  -1.11639696e+00],
                 [  3.24400357e+00,  -7.21156490e-01,   8.57375551e-01],
                 [ -7.70309181e+00,   5.34199860e+00,  -5.26494389e+00],
                 [  1.03173791e+01,  -5.08337359e+00,  -3.57143638e+00],
                 [  6.34715548e-01,  -2.19057593e+00,   1.67312524e+00],
                 [  9.08500891e+00,   2.27296242e+00,  -1.64565745e-01],
                 [  1.57254576e+00,   1.81532960e+00,   1.03236099e+00],
                 [ -7.02729166e-02,   1.26419858e-01,   5.39637906e-01],
                 [ -1.06165239e+00,   3.74751252e-01,   2.93772782e+00],
                 [ -1.02745405e-01,   3.62922692e+00,   1.51156028e+00],
                 [ -1.21367684e+01,  -3.91129619e+00,   2.85167614e+00],
                 [  1.05563138e+01,   8.51809282e-01,   3.93253388e+00],
                 [  6.22524146e+00,   4.93913291e+00,  -6.85017831e+00],
                 [  6.61044887e+00,  -1.94859395e+00,  -8.37724761e+00],
                 [ -8.48433731e+00,  -2.83704261e-01,   7.02751661e+00],
                 [ -3.75652121e+00,  -2.15880233e+00,  -3.59750665e-01],
                 [  1.00852458e+00,  -1.33750816e+00,   3.50028806e+00],
                 [  1.09365497e+00,  -1.43687231e+00,   1.36131501e+00],
                 [  8.98479757e+00,  -4.68010171e+00,   2.21017371e+00],
                 [ -1.69213716e+00,  -8.33283979e-01,   8.72112614e-01],
                 [ -1.24568294e+00,   2.56066073e+00,   1.03787835e+01],
                 [ -2.74691742e+00,   2.42387348e+00,   5.09061380e-01],
                 [ -5.42008359e-01,   1.77365956e+00,   1.51112581e+00],
                 [  3.00965781e+00,   6.37842878e+00,  -7.21333109e+00],
                 [  2.06821518e-02,   3.33461731e+00,  -2.08852818e+00],
                 [  6.72385511e-01,  -2.72055166e+00,   1.19193822e+00],
                 [  1.77541643e+00,  -2.03492433e-01,   1.78342696e+00],
                 [ -1.16206944e+01,   1.00960172e+01,  -6.12917693e+00],
                 [ -1.06078150e+01,   6.53634242e+00,  -3.42671103e+00],
                 [ -1.21829460e+00,   8.23489081e+00,   2.59050918e+00],
                 [ -2.65645103e-02,   2.46857103e+00,  -1.24465159e+00],
                 [ -2.51114143e+00,  -1.28928628e+01,  -1.71080885e+00],
                 [  2.69591907e+00,   5.12080667e+00,   1.79731175e+00],
                 [  4.87816725e+00,   8.79134932e+00,   3.54406241e+00],
                 [ -3.16404431e-01,   8.25816113e+00,   1.07212182e+00],
                 [  8.00932505e-01,  -6.28757280e-01,  -8.23790594e-01],
                 [ -6.89256439e+00,   1.81691877e+00,  -2.89425270e+00],
                 [  1.66624800e-01,  -1.36124742e+00,   2.19536434e+00],
                 [ -3.92179768e+00,  -3.23750498e+00,  -3.07251542e+00],
                 [  7.64427735e+00,  -6.65748873e+00,  -3.08578144e+00],
                 [ -2.78134810e-01,   2.19176169e+00,  -6.20213742e-01],
                 [  2.45112334e+00,  -8.62139850e+00,   8.75319819e+00],
                 [ -7.20438947e-01,   2.08485205e-01,  -1.13058762e+00],
                 [  1.55291358e+00,   7.21851797e-01,  -2.00583714e+00],
                 [ -2.58591201e+00,   1.07059702e+00,  -1.35006742e+00],
                 [ -8.13315314e+00,   8.33786024e+00,   5.32396386e-01],
                 [  1.45098804e-01,   3.68457701e+00,   1.51688331e+00],
                 [ -7.76514455e+00,   2.34266206e+00,   3.11432506e+00],
                 [  3.53759204e+00,  -6.52302719e+00,   9.07117801e-01],
                 [  9.91697717e+00,  -4.69267312e+00,   5.58789998e+00],
                 [ -1.42411137e+00,  -5.45506586e+00,  -1.80597414e+00],
                 [ -4.49420712e+00,  -8.74170419e-01,  -2.21088167e+00],
                 [  1.22013535e+00,   1.16441813e+01,  -2.11138591e+00],
                 [ -2.97491319e+00,  -6.33738648e-02,   1.03282308e-01],
                 [  4.30164865e+00,  -4.75752844e+00,   1.00142128e+00],
                 [  1.68711904e-01,   1.11110241e+00,   1.12219078e+00],
                 [ -3.32589427e+00,   1.33236881e+00,   9.63867771e+00],
                 [  5.54002738e+00,   9.24967707e-04,   2.84022894e+00],
                 [ -8.19948926e-01,  -1.71638295e+00,  -4.07911601e-01],
                 [  9.39553792e+00,  -2.06802623e-01,   7.41005034e+00],
                 [  6.31200991e-01,   2.40124067e+00,   2.32376277e-01],
                 [  1.69952391e+00,   1.11163188e+00,   2.83977083e+00],
                 [  1.40677215e+00,   2.78171940e-01,  -3.85718811e-01],
                 [ -4.27709471e+00,   7.33428066e+00,   4.09038957e+00],
                 [ -4.17436751e+00,   9.46963080e+00,  -4.95304997e+00],
                 [ -6.76938068e-01,   8.31738748e-02,  -4.82583617e+00],
                 [ -7.60846659e-01,  -5.06228411e+00,  -1.06289321e+01],
                 [  5.41947662e+00,  -6.96832553e+00,   5.20592038e+00],
                 [  2.86293166e+00,   7.93700976e+00,  -2.20035247e+00],
                 [  5.29025994e+00,   9.63189806e+00,   1.32783388e+00],
                 [ -5.04660221e+00,  -3.81746644e+00,   5.62508617e+00],
                 [ -1.66909745e+00,  -1.42092038e+00,   3.51069612e+00],
                 [  4.07743566e+00,   7.63734303e+00,  -6.22884835e+00],
                 [  1.18749549e+00,  -1.25536805e+00,  -2.10902040e-01],
                 [  4.51895741e+00,  -7.46046249e+00,   6.60984686e+00],
                 [  3.65358436e+00,   4.62242870e+00,   8.21987005e+00],
                 [  7.22759651e-01,   9.91301174e-01,  -1.44259338e+00],
                 [ -4.88995477e+00,  -8.99692817e-01,   4.39398769e+00],
                 [  2.34522186e+00,  -1.95102540e-01,   2.56544783e+00],
                 [  4.60716939e-01,  -1.14506741e+00,  -7.24217745e-01],
                 [ -1.90483070e+00,  -3.85669154e+00,  -1.12629080e+00],
                 [ -5.87674207e+00,  -1.53256935e+00,   3.51352454e+00],
                 [ -3.59129087e+00,   2.42071434e+00,   8.09791975e+00],
                 [ -5.89033101e+00,   2.30268148e+00,   1.51692850e-01],
                 [  1.08428688e+01,  -2.65437585e+00,   4.45834400e+00],
                 [ -9.95160572e-01,  -4.71554859e+00,  -5.56938284e+00],
                 [ -9.68100215e+00,  -1.97686001e+00,   5.52602096e+00],
                 [ -8.72800139e+00,   1.03541112e+00,  -5.25255840e+00],
                 [  4.92499119e+00,   4.13717747e+00,   5.10778129e+00],
                 [  2.79692653e+00,  -2.78298078e+00,   4.13176813e-01],
                 [  7.10039527e+00,   6.13683988e+00,   5.32283267e-01],
                 [  2.14615161e+00,   1.38835901e+00,   4.77632063e-01],
                 [ -7.91128306e-02,   3.12127002e+00,  -6.30446057e+00],
                 [ -1.25363832e+00,  -1.67352128e-01,   4.61131410e+00],
                 [  9.41061252e-01,   5.00949658e-01,  -6.35438729e-02],
                 [  2.03713602e+00,   8.97055876e-01,  -1.65683239e+00],
                 [ -2.83274991e+00,  -3.04333334e-02,   3.33082894e-01],
                 [ -9.32439635e-01,  -1.65037087e+00,  -1.95489373e+00],
                 [  5.02534525e-01,   7.32563861e-01,  -8.32292774e-01],
                 [ -1.42479377e+00,  -1.12669039e+00,  -2.13782545e+00],
                 [ -9.78625797e+00,   6.07590254e+00,   2.46961246e+00],
                 [  4.88785584e+00,  -5.70185875e-01,  -5.73195342e+00],
                 [ -9.48294325e+00,  -8.52831845e-01,   9.07338608e+00],
                 [ -1.11949104e+01,   3.67902717e-01,   5.51533857e+00],
                 [ -4.02726112e+00,  -4.99469142e+00,  -7.06872456e+00],
                 [ -2.19563586e+00,   2.66669457e+00,   1.06605539e+01],
                 [ -3.31536109e+00,   9.87723074e-01,  -1.08634599e+01],
                 [  1.32519286e+00,   1.88554312e+00,   5.19083403e-01],
                 [  2.01535237e+00,   9.18536291e+00,  -5.32023766e+00],
                 [ -2.67906992e+00,   1.14577822e+00,  -1.60064568e+00],
                 [  7.14984787e+00,   3.47902045e+00,  -4.05609869e+00],
                 [ -4.41508308e+00,  -4.32640080e+00,   7.74513505e+00],
                 [  3.34129213e-01,  -1.81138289e-01,   4.61426103e-01],
                 [ -5.95154652e+00,  -5.88449849e+00,  -7.19814835e+00],
                 [  3.37498142e-01,  -1.85337219e+00,  -1.75299121e-01],
                 [ -1.84056996e+00,   8.76322811e-01,  -2.19932370e+00],
                 [  2.04647464e-01,   5.80194275e-01,   2.42332901e+00],
                 [ -4.25060180e+00,  -4.67494954e+00,  -1.18081496e+01],
                 [ -1.12323733e+00,  -6.80265642e+00,   9.40821551e+00],
                 [ -1.24373708e+01,   2.67536223e+00,  -1.76745379e+00],
                 [  4.06138697e+00,   5.47303576e+00,  -9.00821891e+00],
                 [ -2.37350434e+00,   2.03857937e+00,  -9.20503994e+00],
                 [  4.61006753e+00,  -1.14212393e+01,  -1.39928076e+00],
                 [  5.03955652e+00,   8.82397558e-01,   6.34579565e+00],
                 [ -7.48708042e+00,  -6.79197127e+00,  -1.62245444e+00],
                 [ -3.00126586e-02,  -6.23829609e-02,  -1.44501954e+00],
                 [  5.24034558e+00,   7.64478642e-01,  -8.46957199e+00],
                 [ -1.94007764e+00,   2.00921943e+00,   3.34113552e+00],
                 [ -5.38194538e+00,   1.38675197e+01,   3.78310898e+00],
                 [ -5.71215434e+00,   1.23585289e+01,   6.13631652e+00],
                 [ -1.51611287e+00,  -3.98192352e+00,  -3.54653534e+00],
                 [ -2.73395621e+00,  -6.07142616e+00,  -1.01159630e+00],
                 [ -3.51844879e+00,  -2.79915487e+00,  -2.54892339e+00],
                 [ -4.41407934e-01,  -9.47890748e-01,  -2.57499541e+00],
                 [  2.12656379e-01,  -2.02703033e+00,   2.33888630e+00],
                 [ -1.47118867e-01,   3.56356230e+00,  -1.24754680e+01],
                 [ -2.53675815e+00,   4.30074480e+00,   8.51092048e+00],
                 [  2.08237344e+00,  -9.06770842e+00,  -6.32923580e+00],
                 [ -2.68827571e+00,  -3.25579991e+00,   1.41801318e+01],
                 [ -3.72001836e+00,   2.50281822e+00,  -2.52996176e+00],
                 [ -2.70867539e+00,  -1.29361395e+01,   1.45244763e+00],
                 [ -1.55383432e+01,   7.07411357e+00,   4.82983430e-01],
                 [ -8.10827692e+00,   3.64789584e+00,   5.19416084e+00],
                 [ -1.27043565e+00,   2.05291110e+00,  -7.86197514e-01],
                 [  9.77079431e-01,  -1.15302039e+01,   3.21710986e+00],
                 [  7.00038191e-01,   2.00499255e+00,  -1.79257990e+00],
                 [ -8.81985488e-01,  -2.25103177e-01,  -1.47274763e+00],
                 [ -7.62611135e+00,  -5.07537114e+00,  -4.94727621e+00],
                 [  5.84172514e-01,   1.66047713e+00,   9.87215907e-01],
                 [  6.00700183e+00,  -1.86681296e+00,  -7.15136797e+00],
                 [  3.29036257e-01,  -2.20478844e+00,  -4.12896224e-01],
                 [ -1.20231433e-01,  -2.70601573e+00,  -1.38988789e+00],
                 [  7.95722188e-01,  -1.04159835e+00,   1.12403763e-02],
                 [  6.98355925e+00,   1.09981095e+01,  -9.71341480e-02],
                 [  6.40868316e+00,   4.51854585e+00,   2.64253885e+00],
                 [ -3.44753339e+00,   4.66445876e+00,  -9.10382575e+00],
                 [ -1.61962204e+00,   6.63384114e+00,  -3.15488546e+00],
                 [  1.33848719e+01,  -1.96699131e+00,  -1.65182207e-01],
                 [  4.60562970e+00,   4.65509238e+00,   1.36542228e+01],
                 [  5.68297906e+00,  -8.15426828e+00,  -3.19487150e+00],
                 [ -2.19821321e+00,  -1.04824061e+01,   4.86660519e+00],
                 [ -1.10413732e+00,   1.52888076e+00,   9.72302232e-01],
                 [ -4.14215806e+00,   5.13237166e-01,  -3.60656655e+00],
                 [ -7.72728095e-01,  -2.45180310e+00,  -4.47827493e-02],
                 [  9.44462397e+00,   4.76141874e+00,  -5.62968150e+00],
                 [  5.13052250e-01,  -6.44785334e+00,  -2.28343731e+00],
                 [ -1.86389814e+00,  -1.31003680e+00,  -3.50928679e+00],
                 [  1.70561600e-01,   1.23150482e+01,   1.96000377e+00],
                 [  3.04018784e-01,  -1.13089460e-01,  -8.22776050e-01],
                 [  1.25971102e-01,   2.42956540e+00,  -2.88252876e-01],
                 [ -1.09977622e+00,  -5.23239964e-01,  -2.30837015e+00],
                 [ -1.21748047e+01,   2.73118592e+00,  -9.29638540e+00],
                 [  9.89623516e-02,   1.56063266e-01,   1.36214868e+01],
                 [  4.09459597e+00,  -8.49245975e+00,   2.04313384e+00],
                 [ -7.61951588e+00,   1.39548937e+01,  -5.12014398e-01],
                 [  2.05467333e+00,  -3.24345893e+00,  -5.03123750e-01],
                 [ -2.25587311e+00,  -1.19957219e+01,  -4.94640451e+00],
                 [ -2.31175336e+00,  -4.90707504e+00,   7.35366345e+00],
                 [  1.57045811e+00,   8.56545218e+00,  -2.88615060e+00],
                 [  1.46655550e+00,  -1.90916377e+00,   1.50697618e+00],
                 [  1.31473297e+00,   1.31597846e+00,  -1.47446758e+00],
                 [ -2.44424085e+00,  -2.98229316e-04,   1.96809147e-01],
                 [ -7.67898443e+00,   9.29051289e-02,   2.96002611e+00],
                 [ -2.42958723e+00,  -1.44489551e+00,   1.07693504e+01],
                 [ -8.20804085e-01,  -5.24507422e-01,   1.05470857e+00],
                 [  6.65156484e+00,  -5.22477814e+00,   9.88423215e-01],
                 [  1.37068976e+00,   8.88736448e-01,  -7.07004793e-01],
                 [ -1.25692823e+00,   1.48325498e+00,   6.83256534e-01],
                 [  1.08009095e+00,  -1.87896597e+00,   9.55236395e-01],
                 [  1.21158682e+00,  -9.08951554e+00,  -3.39465095e+00],
                 [  1.17751263e+00,  -1.21373450e+01,  -2.64996064e+00],
                 [  3.27259933e+00,   5.49747596e+00,  -6.23389096e+00],
                 [  4.63941432e+00,   1.11448540e+00,   1.38068333e+01],
                 [  4.86337460e+00,   5.50580107e+00,  -1.16683081e+01],
                 [  2.03408955e+00,   2.52860773e+00,   1.17200493e+00],
                 [ -3.45878191e+00,  -8.98194586e-01,   3.51853196e+00],
                 [  9.18144594e+00,  -2.62456603e+00,   4.59263347e+00],
                 [ -3.91615853e+00,  -3.42319281e-01,   1.80018137e+00],
                 [  1.02388149e+01,   2.29598289e+00,  -5.14431743e+00],
                 [  1.43638688e+00,   2.02432121e+00,   1.31798016e+00],
                 [ -7.18675106e+00,   2.99418370e+00,  -3.14557070e+00],
                 [ -3.36524665e+00,  -2.86014112e+00,   2.38533980e+00],
                 [  1.07519627e+00,  -1.66575513e+00,   2.53850506e+00],
                 [  6.14676370e+00,  -9.64545605e+00,  -6.79728907e-01],
                 [  2.01139489e+00,   2.52366225e+00,   2.21579537e+00],
                 [ -2.13297100e+00,   6.48512306e-01,  -3.08370287e+00],
                 [  5.69361725e-01,  -5.76850816e-01,   3.50302698e-01],
                 [ -8.11193663e-01,   5.41627495e-01,  -5.93006028e-01],
                 [ -7.97837786e+00,   5.52893451e+00,  -6.43306756e+00],
                 [  1.63126990e+01,  -7.34443488e+00,  -1.42673533e+00],
                 [ -2.33438031e+00,   3.72302114e+00,   2.84321065e-01],
                 [ -8.66814190e+00,   9.36456136e+00,   2.81612364e-01],
                 [  1.03488996e-01,  -3.17875022e-01,  -1.31078580e+01],
                 [ -1.28891414e+00,  -9.12307521e+00,   7.14882809e+00],
                 [ -9.04528865e+00,   1.22257589e+00,  -7.59878724e+00],
                 [  9.03723818e-01,  -1.02198692e+00,  -1.18808834e+00],
                 [  8.52751761e+00,  -7.74119873e+00,   5.09518062e+00],
                 [ -1.68151648e+00,   1.17867869e+00,  -1.08120873e+00],
                 [  5.60759380e-01,  -7.66888199e+00,  -3.53852953e+00],
                 [ -6.98872997e+00,   1.46078658e+01,   4.49802944e+00],
                 [  1.63282945e+00,   8.57002544e-01,   1.83452781e+00],
                 [  5.54159396e+00,  -3.14938668e+00,  -9.82337999e+00],
                 [ -1.58734560e+00,  -1.12358073e+00,   6.42003518e-01],
                 [  4.10434326e-01,  -1.61817186e+00,  -7.62675377e-01],
                 [  8.41365966e-01,  -1.83000262e+00,   4.89073618e-01],
                 [  7.43071822e+00,  -5.70087109e-01,   7.59438293e+00],
                 [  2.45323420e+00,   7.74694956e+00,   2.31819529e+00],
                 [ -2.00974876e+00,   6.28225909e+00,  -3.98715180e+00],
                 [ -8.34336826e-01,  -1.09116302e+01,  -3.54480916e+00],
                 [  3.11212876e+00,  -4.11710349e+00,   3.37964986e+00],
                 [  7.37006690e+00,  -2.13213541e+00,  -1.15733620e+00],
                 [ -4.51968460e+00,  -9.11423923e+00,  -3.28704227e+00],
                 [  1.97019339e+00,   1.19569241e-01,  -1.01100004e+01],
                 [  2.08685423e+00,   1.73481702e+00,   9.53691862e-02],
                 [ -2.04486641e+00,   9.76588661e+00,  -1.65096466e+00],
                 [  1.74603907e+00,   1.08280878e+00,   1.18861412e+00],
                 [  6.24880369e+00,   2.41758533e-01,  -1.90318281e+00],
                 [  5.73250709e+00,   2.42583399e+00,   6.13012818e+00],
                 [  3.64671638e-01,  -1.37046160e+00,   3.41245867e-01],
                 [  2.99659477e+00,   7.55239767e-01,  -5.12728891e-01],
                 [  2.23810517e+00,  -2.64732571e+00,  -9.55580907e-01],
                 [ -2.79586478e+00,   1.14986952e-01,   2.32932948e-01],
                 [  6.11838344e-01,   1.95328628e-01,  -1.54761100e+00],
                 [  5.84901010e+00,   1.34183691e+01,   6.77759901e+00],
                 [ -1.06904643e+01,   6.63382427e+00,   3.01482986e-01],
                 [ -3.43942041e+00,  -1.79001204e+00,   3.76514804e+00],
                 [  7.99924339e+00,  -9.85138469e-01,   5.04249712e+00],
                 [ -1.58024748e+00,  -6.25806549e-01,  -1.60438253e+00],
                 [ -5.57257566e+00,  -4.76976495e+00,  -9.58126370e+00],
                 [  8.15273588e+00,   2.98308655e+00,  -5.84247929e+00],
                 [ -1.11846389e+00,  -5.88140533e-02,   9.45478279e+00],
                 [ -1.31950037e+00,  -1.22513716e-01,   6.89345409e-02],
                 [ -4.76102667e+00,  -8.52722904e+00,   3.37729565e+00],
                 [  7.41492184e-02,  -1.58751661e+00,  -3.25481546e+00],
                 [  6.04804332e+00,   1.28095737e+00,   5.04555921e+00],
                 [  6.19000859e+00,   1.12203562e+01,  -1.72764596e+00],
                 [  5.44995359e-01,   1.04044244e+00,   1.51080550e+00],
                 [ -1.97423988e+00,   2.72932297e+00,  -5.83947175e+00],
                 [ -2.55735680e-01,   7.62406732e-01,   3.81836581e-01],
                 [  1.84797470e+00,   2.52655010e+00,  -4.88369727e-01],
                 [ -6.07445597e-01,  -6.07212719e-01,  -8.18845389e-01],
                 [  8.14139313e+00,   4.43391423e+00,   4.38653873e+00],
                 [ -2.23753691e+00,  -7.18152646e+00,  -2.97384874e+00],
                 [  2.99612142e+00,  -8.70571082e+00,  -6.04369732e+00],
                 [  5.55283149e+00,   6.33244697e+00,   3.71024059e+00],
                 [ -5.26894802e+00,  -8.77116227e+00,  -3.77158307e+00],
                 [ -6.19142901e+00,  -5.70389524e+00,  -1.14247418e+00],
                 [  5.90862389e+00,   8.71736996e+00,  -5.58450300e+00],
                 [ -1.44610000e+00,  -1.80341416e+00,   5.69956630e+00],
                 [ -2.98064742e+00,   1.81380791e+00,  -2.86755793e-01],
                 [ -2.20840486e+00,   2.29425864e+00,  -1.19430753e+01],
                 [ -2.43624750e+00,  -1.35850692e+00,   2.48002210e+00],
                 [  4.19948535e+00,  -6.69676369e+00,  -8.00697748e+00],
                 [  7.20191637e+00,   2.81410546e+00,   1.81531770e+00],
                 [ -3.86418666e-01,   2.70657095e+00,  -8.28279911e-01],
                 [  8.55683622e+00,  -3.04269040e+00,  -1.11291762e+00],
                 [ -2.56556546e+00,   7.97666961e-01,  -4.47329900e-01],
                 [ -1.89631566e+00,  -7.59272390e-02,  -9.70577026e-01],
                 [  3.93928627e-01,   9.72912661e-01,   1.04481292e+00],
                 [  4.04091179e+00,   2.55483583e+00,   1.37939699e+00],
                 [ -1.11775075e+01,   4.08045740e+00,  -5.16473106e+00],
                 [  4.30538769e+00,   2.27985574e+00,  -9.17859367e+00],
                 [ -5.90136354e-01,  -2.24283612e+00,  -3.85887517e+00],
                 [ -2.74655881e+00,  -3.00471315e+00,   4.23589788e+00],
                 [  8.85933886e+00,  -5.94254208e+00,  -6.54605679e-02],
                 [ -1.22031526e+00,  -5.54766299e+00,   9.90947829e+00],
                 [  7.58838886e-01,   8.69085866e+00,   2.17341528e+00],
                 [  5.71050849e-03,  -8.20676491e-01,   1.80098131e+00],
                 [ -6.45992004e-01,  -3.31682103e+00,  -9.57514376e+00],
                 [  1.80799036e+00,  -2.29100295e+00,   1.01484655e+00],
                 [ -3.82685847e+00,   1.53653216e+00,   1.28489237e+00],
                 [  3.67033287e+00,   1.03791382e+00,  -2.20116875e-01],
                 [ -1.46016141e+00,  -2.11964021e+00,   1.80293894e+00],
                 [  7.60243188e+00,  -5.56497321e+00,  -1.14922545e+00],
                 [  8.04440315e-01,  -3.65085975e-01,   1.20525569e+00],
                 [  2.00363062e+00,  -6.78063872e-01,   1.30321859e+00],
                 [ -2.01510810e-02,  -8.58948176e-01,  -2.33423484e+00],
                 [  4.35462019e+00,   8.61620633e+00,  -8.17169641e+00],
                 [ -9.16139835e+00,   5.40213401e+00,   3.19029672e+00],
                 [ -7.10688889e+00,   9.11164998e-01,   1.39289145e+00],
                 [ -1.21320159e+00,  -9.60799053e-01,   2.92874632e+00],
                 [ -2.24028453e+00,  -1.00105738e+01,  -6.28762477e+00],
                 [ -3.69381874e+00,  -8.89846313e+00,   2.25081322e+00],
                 [ -4.55814512e+00,   4.77789425e+00,  -1.20067331e+01],
                 [  6.74580423e+00,  -2.76334086e+00,  -8.63961983e-01],
                 [ -1.65928671e-01,  -6.25410789e-01,   1.83698011e+00],
                 [  2.08193705e+00,  -3.05055187e+00,   7.33448695e+00],
                 [  2.14705367e+00,  -4.43045378e-01,   7.73441127e-01],
                 [  7.47498990e+00,  -3.53973366e+00,   1.04377973e+01],
                 [ -4.28893155e+00,   1.33640342e+00,  -7.29338611e+00],
                 [ -5.71124924e-01,  -1.85568938e+00,  -3.13160341e+00],
                 [  4.65295418e+00,   6.20680368e+00,   8.40445853e+00],
                 [ -1.80717038e+00,   1.02644464e+00,  -1.40689464e+00],
                 [  3.03386450e+00,   4.31800584e-01,   7.23010350e-01],
                 [  2.37963559e+00,  -4.98254551e-01,  -2.81087431e-01],
                 [ -1.97177631e+00,  -4.88538371e+00,  -4.14425161e+00],
                 [ -3.68120082e-01,  -1.20148609e+01,   3.08512818e+00],
                 [ -6.58527307e+00,  -7.58537482e+00,  -8.60072140e+00],
                 [  7.67981163e+00,  -7.65376023e+00,  -6.53349459e+00],
                 [  9.46801965e+00,  -2.22766285e+00,   8.35278892e+00],
                 [ -2.39470657e+00,  -9.48307870e+00,   9.52446688e+00],
                 [  8.95868802e+00,   6.49333556e+00,  -2.31215260e+00],
                 [ -1.01842672e+01,   4.75546667e-01,   1.54241730e+00],
                 [ -7.10052146e-01,  -2.36922203e-01,   1.43039437e+00],
                 [ -5.29318121e+00,   2.83283248e+00,  -5.02356386e-01],
                 [ -1.10478731e+00,   2.27223535e-01,  -6.75313555e-01],
                 [ -5.64859474e+00,  -3.28230002e+00,   8.33496500e+00],
                 [ -1.87816535e+00,   7.22560559e+00,  -3.62949805e+00],
                 [ -9.03056213e-02,  -6.96377026e-01,   1.33418684e+00],
                 [ -1.36583225e+01,  -3.37182330e+00,  -1.73531913e+00],
                 [ -4.04811414e-01,  -1.01496886e-01,  -1.19292738e+00],
                 [  7.68331607e-02,  -3.51278031e+00,  -7.93573882e-01],
                 [ -9.74226750e-01,   3.71430302e+00,  -3.67007930e-01],
                 [  3.50056826e+00,   3.23251193e+00,   1.27296720e+01],
                 [ -1.94878364e+00,   3.69368167e-01,   2.51375355e+00],
                 [  1.24381970e+00,  -2.97653108e-01,   1.60384197e-01],
                 [ -9.24026275e-01,  -1.19856356e+00,   1.28425332e+01],
                 [  9.17765242e+00,   2.18562359e+00,   5.67355540e+00],
                 [ -4.07621731e+00,   7.98037318e+00,  -9.34955872e+00],
                 [  3.99873514e+00,   1.06474486e+01,  -1.74325606e+00],
                 [ -9.37789947e+00,  -4.77241890e+00,   9.09066282e-01],
                 [  1.01323494e+00,  -1.30162365e+00,   2.81055742e+00],
                 [  9.93768189e+00,   6.33760295e+00,  -3.54992830e+00],
                 [  9.07910970e-01,  -8.95700793e-01,  -5.05911634e-01],
                 [ -8.30941535e+00,   1.64161042e+00,   2.45480878e+00],
                 [  6.28688907e+00,   2.33389947e+00,   6.09805654e+00],
                 [ -7.97693686e-01,  -8.95309323e-01,   2.26950999e+00],
                 [ -3.10852075e+00,   1.74774311e+00,  -2.41242804e+00],
                 [ -1.32694344e-01,  -1.41903595e+00,  -2.19562737e+00],
                 [ -6.06728363e-01,  -2.82505186e+00,   1.51813310e+00],
                 [  2.97963897e+00,   3.66094501e+00,  -3.49461300e+00],
                 [ -2.40015491e+00,  -2.87553974e+00,   2.17993996e+00],
                 [ -1.11318567e+01,  -7.98951360e-01,  -1.83717157e+00],
                 [ -3.23002059e+00,  -5.06854432e+00,  -8.88781526e+00],
                 [  1.03496942e+01,  -9.52289085e+00,   5.77603459e+00],
                 [ -1.11314249e+00,  -4.16053292e+00,   9.12188125e+00],
                 [ -8.76624533e+00,  -2.77334155e-02,   1.84747323e+00],
                 [  3.01730006e+00,  -4.07254197e+00,   3.13015941e+00],
                 [ -1.30651521e+00,  -7.44795742e-01,  -1.28754560e+01],
                 [ -1.33638603e+00,  -6.98784490e-01,   1.44994437e+00],
                 [  7.04435244e+00,  -4.11730416e+00,  -5.08654439e+00],
                 [  2.70700032e+00,   1.44384110e-01,  -1.06306866e+00],
                 [ -4.44172393e+00,  -2.12384364e+00,   3.82283935e+00],
                 [ -7.45897690e-01,   9.79422822e+00,   6.33673303e+00],
                 [ -2.89929701e+00,   1.39950542e+00,  -2.51081733e-01],
                 [ -6.53346135e+00,   1.89059512e+00,  -3.18757857e+00],
                 [  3.72751879e-01,  -2.13407388e+00,   2.17126376e+00],
                 [ -2.01600104e-01,  -2.36560191e+00,  -1.12299563e-01],
                 [  4.78048131e+00,   2.24994180e+00,   3.59986371e-02],
                 [  9.34840255e-03,   5.30020492e-01,  -6.94572290e-01],
                 [ -7.11554745e+00,  -4.08757559e+00,  -8.41159727e+00],
                 [ -6.04869956e+00,   6.12995542e+00,   3.68452081e+00],
                 [  5.95306433e+00,   9.60454302e+00,  -5.74374486e-01],
                 [ -5.63323732e+00,   1.10177613e+00,   2.15489959e+00],
                 [  8.10998300e+00,   9.45986732e+00,   2.59630304e+00],
                 [ -9.56410282e+00,   2.24657135e+00,   1.56445610e+00],
                 [ -2.11858989e+00,   6.37876444e-01,  -3.51359083e+00],
                 [ -2.27302030e+00,  -2.49493160e+00,   7.86355128e-01],
                 [ -6.98917685e+00,  -7.01777917e-01,   9.56125001e-01],
                 [ -5.67526891e-01,   1.42980193e+00,  -1.58517364e+00],
                 [  1.85175974e+00,   1.01216464e+01,   2.10345144e+00],
                 [  6.45847398e+00,   6.59666103e+00,  -8.67279320e+00],
                 [  4.09007709e-01,  -6.68738847e-02,   1.69372597e+00],
                 [  4.31645857e+00,  -3.24357878e+00,  -3.26129414e+00],
                 [  2.57799761e+00,   3.18842003e+00,  -6.89706687e-01],
                 [  5.51722866e-01,   8.64273398e-01,  -9.05453231e-02],
                 [  6.96373445e-01,   1.08040521e-01,   8.63361020e-01],
                 [ -4.28911017e-01,  -1.70889138e+00,   4.89218650e+00],
                 [ -2.91339817e+00,   6.51928153e+00,   1.72193443e-01],
                 [  8.73311945e+00,  -1.00700529e+01,   7.24570482e-01],
                 [  2.51790955e+00,   6.23219715e+00,   5.56142255e+00],
                 [  6.66024121e+00,  -2.48723966e+00,  -2.98879263e+00],
                 [ -9.25571295e+00,  -8.62613459e-01,  -5.33432621e+00],
                 [ -1.00722003e+01,   6.40781837e+00,   4.90550298e+00],
                 [ -2.83034505e+00,  -2.32279988e+00,   6.05273623e+00],
                 [  1.19799426e+00,   1.30560799e+00,   9.67495941e-01],
                 [  4.70912262e+00,   3.72672322e+00,  -4.39012732e+00],
                 [  1.52800192e+00,   2.16522398e-01,  -7.10013787e-01],
                 [ -6.37092710e+00,  -1.23608494e+01,   2.10149956e-01],
                 [  1.03346163e+01,  -4.17883363e+00,  -5.27710026e+00],
                 [ -1.51132333e+00,  -7.65918404e-01,  -6.30994638e-01],
                 [  1.27165541e+00,   1.37009234e+00,  -3.00467603e+00],
                 [ -1.86931160e+00,  -7.10224515e-02,   2.73514670e+00],
                 [  3.04997055e-01,   6.78089066e-01,   1.86421481e+00],
                 [ -1.44516618e+00,   4.36315575e-01,   5.79721337e-01],
                 [ -5.87320708e+00,  -9.49524425e-01,  -8.08741040e+00],
                 [  4.67043175e+00,  -5.09436325e+00,  -9.11653560e+00],
                 [ -7.44102852e+00,   4.91206736e+00,  -1.18411785e+01],
                 [ -1.26292294e+01,   3.03544160e+00,   3.99514901e+00],
                 [ -6.28019359e+00,   7.11442382e+00,  -1.00773423e+00],
                 [  7.83707511e+00,  -1.74746136e+00,  -4.11044184e+00],
                 [ -5.20380219e+00,   1.98867878e+00,   9.55303229e+00],
                 [  9.46377920e+00,  -7.32903215e+00,   4.89016180e+00],
                 [  1.74984201e+00,  -3.05188951e+00,  -3.65187878e-01],
                 [ -2.04430885e+00,   6.74656101e+00,  -1.14572291e+00],
                 [  6.51300784e-01,  -2.36159403e+00,   9.07263679e-01],
                 [ -8.60722774e-01,   6.48012155e+00,  -7.65119015e+00],
                 [ -9.48697778e+00,   1.01719425e+01,  -2.50622501e-01],
                 [  3.05058216e+00,  -2.83165475e-02,  -1.08814891e+00],
                 [  6.93075306e+00,  -2.97009577e+00,  -3.70370860e+00],
                 [ -1.78234866e+00,  -2.07485460e+00,   1.46258926e-01],
                 [  1.73966237e+00,   1.08236132e-01,  -4.51578634e-01],
                 [ -4.52688539e+00,   1.98959671e+00,  -6.18419855e-01],
                 [ -2.64307535e+00,   5.85016802e+00,   7.77084519e-01],
                 [ -2.80304719e+00,  -7.57567654e+00,   1.88274441e+00],
                 [  7.07604531e+00,   9.04285669e+00,  -6.22850590e-01],
                 [ -9.40200958e+00,   1.57389462e+00,  -2.91552070e+00],
                 [ -1.32011244e+01,  -3.95066049e-01,  -5.48752399e-01],
                 [  4.32843094e+00,  -1.00534214e+01,   5.58676360e+00],
                 [  6.26428622e+00,   8.21475806e+00,   3.26816970e+00],
                 [  2.24269292e+00,  -1.29018723e+00,  -9.47388954e+00],
                 [  5.91938046e-02,   1.66452726e-01,   1.68492922e+00],
                 [  5.71527334e+00,  -7.02952101e-01,   1.20566902e+01],
                 [  1.18326686e+00,   3.75883572e-01,  -1.72957201e+00],
                 [  5.47459357e+00,   8.96364245e+00,  -8.73519233e+00],
                 [  1.13120143e+01,   3.10328793e+00,  -2.85593481e+00],
                 [  4.66988010e+00,   8.11516173e-01,   1.13707435e+00],
                 [  1.76935000e+00,   6.99264943e+00,   4.29548303e+00],
                 [ -8.40211395e-01,  -1.02331715e+00,  -1.82294200e+00],
                 [ -7.61201145e-01,  -3.28538497e-01,  -3.23934324e+00],
                 [  5.71233504e-01,   1.42112374e+00,  -7.35647109e-01],
                 [  7.33078717e+00,   3.22703939e+00,  -7.23424856e+00],
                 [ -5.62547214e+00,   4.88581035e+00,  -1.11928200e+00],
                 [  1.29779491e+00,   3.59680395e-01,  -1.46293525e+00],
                 [ -5.58532030e+00,   1.68165574e+00,  -9.24373496e+00],
                 [ -5.56798561e+00,  -7.01808732e+00,  -9.71225486e-01],
                 [ -6.11184162e+00,  -1.19658483e+01,  -5.10301360e-01],
                 [ -3.19635777e+00,   9.15377161e+00,  -4.20232939e-01],
                 [  8.69501562e+00,   4.92305778e+00,   6.10678708e+00],
                 [  8.67606301e-01,  -1.06562406e+00,   1.57576181e+00],
                 [ -1.05825688e+01,  -1.37535187e+00,  -9.36608657e+00],
                 [  1.89595441e+00,  -1.48260025e+00,   1.44449759e+00],
                 [  5.28156496e+00,  -1.19240263e+01,  -2.32843319e+00],
                 [  3.03016442e+00,   5.41046665e+00,   4.61357216e+00],
                 [  4.75828552e-02,   1.79674684e+00,  -1.38644379e-01],
                 [ -4.61614333e+00,  -2.19826856e+00,  -3.76433291e+00],
                 [  1.78074864e+00,   2.37937111e+00,   2.28012136e-02],
                 [ -1.21238863e+00,   1.98937429e-01,   1.47337572e+00],
                 [ -2.35092617e+00,  -2.52823829e-01,  -1.42858883e+00],
                 [ -1.11235686e+00,   6.96008144e-01,   1.08314020e+01],
                 [  3.04824899e+00,   3.67105337e+00,  -3.50886059e+00],
                 [  3.85430689e+00,  -1.10328796e+01,  -1.47597556e+00],
                 [  2.83713557e+00,  -5.62962690e+00,  -9.63590785e-01],
                 [  3.45029767e-01,  -8.04944439e+00,  -8.48734104e+00],
                 [ -6.22595511e+00,  -2.26202513e+00,  -5.45990321e+00],
                 [ -6.63882156e+00,  -1.78083246e+00,   5.63905408e-02],
                 [ -1.04211969e+01,   1.81771589e+00,  -9.49151412e-02],
                 [ -1.47305741e+00,   1.65929648e-01,   2.46990439e+00],
                 [  4.26082546e+00,  -2.61846379e-01,   1.52849191e+00],
                 [  5.84498398e-01,  -1.70132875e+00,   2.95066649e+00],
                 [ -1.47913790e+00,   2.04722876e+00,   8.39484820e+00],
                 [  5.40702223e+00,   1.44617248e+00,   2.79110676e+00],
                 [ -6.61645553e-01,  -3.96745792e-01,  -1.51118945e+00],
                 [ -2.79034487e+00,   6.05501730e+00,  -7.99697101e+00],
                 [ -7.87222262e-01,   2.89554715e+00,  -2.34731461e+00],
                 [  1.40291693e+00,  -2.62059070e+00,   4.40366068e-01],
                 [  1.56900710e-01,   2.76294377e+00,  -2.25657481e+00],
                 [ -1.18106891e-01,  -1.13139256e-01,  -1.01973328e+01],
                 [  5.42055551e+00,   6.17479934e+00,   6.09710443e+00],
                 [  6.35614460e+00,  -4.80450937e-01,   1.48478154e+00],
                 [  2.46036536e+00,  -2.75574232e-01,   4.51557907e-01],
                 [  1.47760434e+00,  -5.25222701e+00,   1.08392957e+01],
                 [ -4.46340176e+00,  -5.00352808e+00,   3.27714285e-01],
                 [  4.36287820e+00,  -9.58759484e+00,  -2.28296486e+00],
                 [  6.02310954e+00,   3.81523019e-01,  -7.10648837e-01],
                 [ -1.90848619e+00,   5.01455914e-01,   2.76309774e-03],
                 [ -1.90935313e+00,  -5.13929702e+00,   5.66067319e+00],
                 [  3.26605680e+00,   1.13130987e+00,  -2.90000977e+00],
                 [  6.73905912e+00,   6.44929601e+00,   1.40908879e+01],
                 [  5.29113390e+00,   9.23111531e+00,  -5.83647091e+00],
                 [  1.73037579e-01,  -1.15918771e+00,   4.95912753e-01],
                 [ -1.28818835e+00,  -1.26944288e+00,   4.16102896e+00],
                 [ -9.02251707e-01,   1.27139281e+00,   2.09236050e+00],
                 [ -2.93388334e-01,  -1.89020309e+00,  -1.31955379e+00],
                 [ -2.65268826e-01,   1.47851807e+00,  -1.53759091e+00],
                 [ -3.89198556e+00,   3.82011196e+00,   8.06744903e+00],
                 [  3.76957869e+00,   4.18834985e+00,   9.26775481e+00],
                 [ -1.35241291e+00,   4.70276896e+00,   7.55733552e+00],
                 [  5.85724732e+00,   2.10686942e+00,  -2.94764180e+00],
                 [  7.18611366e+00,  -2.69194334e+00,   1.20565251e+00],
                 [  2.02577500e+00,  -8.02650773e+00,   4.38841663e+00],
                 [  2.46112810e+00,  -1.37906748e+00,  -1.29761280e+00],
                 [  1.46565731e+00,   5.37682033e+00,   1.61131747e+00],
                 [  6.44636268e-01,  -1.31050855e+00,   1.56699927e-01],
                 [ -1.16280936e+01,   2.45911061e+00,  -4.37196971e-01],
                 [  1.14605665e+00,  -1.18945669e+00,  -8.45516931e-01],
                 [ -1.01674077e+01,  -2.87498755e+00,   4.29353068e+00],
                 [ -1.18402759e+01,  -1.27485231e+00,  -2.34887538e+00],
                 [  1.78357403e-01,   2.20019734e+00,   1.93358392e-01],
                 [  3.21102475e+00,  -7.59777606e+00,  -7.36767462e+00],
                 [ -1.99517040e+00,   1.75890844e-02,   1.52118845e-01],
                 [  3.13578623e-01,  -4.50512419e-01,  -6.88054168e+00],
                 [ -2.61559571e+00,  -8.01992490e-01,  -8.07849033e-01],
                 [ -1.29078183e+00,   6.79441398e-01,  -1.00413513e+01],
                 [ -2.75398296e+00,  -7.04682905e+00,  -3.70140629e+00],
                 [  8.47194824e-01,   8.66893934e+00,  -1.55790989e+00],
                 [  1.97118549e+00,  -9.68882472e-01,   6.85040701e+00],
                 [ -4.56913886e+00,   4.64184627e+00,   1.47466166e+00],
                 [  1.16104795e+01,  -6.15390867e+00,   4.84450161e+00],
                 [  5.95546263e+00,   5.16764596e+00,  -4.71406464e-01],
                 [  1.26847435e+00,  -4.17649949e+00,   1.24979464e+01],
                 [  1.69746749e+00,  -3.34102998e+00,  -3.37636572e-01],
                 [  9.38343686e+00,  -5.97775310e-01,   3.56126455e+00],
                 [ -2.94880631e+00,   1.47613262e-01,  -1.37630756e+00],
                 [ -4.58568362e+00,  -1.45319106e+00,  -5.11146498e+00],
                 [  7.99131955e+00,   7.29608828e+00,   3.67885587e+00],
                 [ -8.65177189e-01,  -8.01441279e-01,   2.05466785e+00],
                 [  9.97802102e+00,   2.29767343e+00,   2.81365970e+00],
                 [  9.90384207e-01,  -1.92130321e+00,   2.21775873e+00],
                 [ -2.46098805e+00,   1.66740381e+00,   2.01680081e+00],
                 [  9.09270925e-01,  -5.54017250e-01,  -1.41349917e+00],
                 [ -1.51080153e+00,   1.07716358e+01,  -5.19354609e+00],
                 [  5.11106676e+00,  -3.07126268e-01,   7.07747701e+00],
                 [ -6.32828956e+00,  -1.90612277e+00,   4.41273345e+00],
                 [ -2.55371248e+00,  -6.19715387e+00,   2.77258316e+00],
                 [ -1.13694954e+00,   8.11714926e+00,   8.22931443e+00],
                 [ -7.04386405e+00,   1.85082265e+00,  -9.18331909e+00],
                 [ -2.58716039e+00,   7.16596128e+00,  -3.55340172e+00],
                 [  1.29717354e+00,   9.02219141e-01,   3.11765618e+00],
                 [ -1.28807861e+00,  -3.52130334e-01,   1.12355423e+00],
                 [ -1.07684688e+00,   7.62761654e+00,  -9.99912449e+00],
                 [ -5.79496558e-01,   1.27659941e+00,  -9.44181054e-01],
                 [  1.42373347e+00,  -8.77439728e+00,   6.56666918e+00],
                 [  2.56793821e+00,  -1.28705719e+00,  -5.48382469e-01],
                 [  5.02116672e-01,   2.11102434e+00,   1.49377767e+00],
                 [  7.03299982e+00,  -6.41426280e+00,   2.04219338e+00],
                 [  1.83122150e+00,   1.25579939e-01,  -1.79543159e+00],
                 [ -9.23769673e-01,  -1.84134040e+00,  -4.39536931e-01],
                 [ -2.56321266e+00,  -1.30721162e+00,   1.42426598e+00],
                 [  3.66666494e+00,   1.11153056e+01,  -5.10179386e+00],
                 [ -1.26182840e+01,   1.98308885e+00,   1.09611031e+00],
                 [  5.48148338e-01,  -3.72368265e+00,  -6.03864832e-01],
                 [  1.12423768e+00,   2.56320852e+00,  -7.38994820e+00],
                 [  5.72827038e+00,   3.32316019e+00,  -1.17239833e+01],
                 [  1.02970407e+01,  -1.38753295e+00,   5.72983625e+00],
                 [  1.24971085e+01,  -2.99942206e+00,  -4.17611361e+00],
                 [  4.30636711e+00,   6.28593653e+00,   5.48859245e+00],
                 [ -6.07596798e-01,  -1.15794226e+00,  -6.36662499e-01],
                 [  1.50792267e+00,   7.94178132e-01,   2.22486921e+00],
                 [  2.03539489e-01,   5.03246309e-01,  -2.19244531e+00],
                 [ -7.97937207e+00,   3.77176357e+00,   5.71811069e+00],
                 [ -4.40726858e+00,  -7.99785437e+00,   8.12097798e+00],
                 [  2.02433108e+00,   1.63884666e+00,   1.14734057e+00],
                 [  1.86742405e+00,   2.97877256e-02,   5.10830834e-01],
                 [  5.77526496e-01,   1.08292300e-01,   2.89757947e+00],
                 [  2.29323012e+00,   3.57138241e+00,   4.18135960e-01],
                 [ -3.01480736e+00,  -2.30614518e+00,   5.30611256e-01],
                 [  3.16330994e+00,   6.60624929e-01,  -6.96021961e+00],
                 [ -2.98964355e+00,  -8.37996334e+00,  -5.59922200e+00],
                 [ -2.42068197e-01,   6.86538116e-01,   2.55253730e+00],
                 [ -7.08410527e-03,  -2.15632798e+00,  -4.84152424e+00],
                 [  9.56495917e+00,   1.43197748e-01,  -1.25838046e+00],
                 [ -6.41127662e+00,   6.65832483e+00,   9.29420652e-01],
                 [ -2.50733294e-01,  -2.79714951e-01,   8.40427942e+00],
                 [  8.95509456e+00,  -5.54738946e+00,   2.10400650e+00],
                 [  8.44427184e-01,   2.54462368e+00,  -2.54473182e-01],
                 [  4.22229098e-01,   4.52415292e+00,   9.28901467e+00],
                 [ -8.32157759e-01,   1.60168651e+00,   2.28877895e+00],
                 [ -1.03842933e+01,  -4.89125079e-01,  -4.77421046e+00],
                 [ -4.79555030e+00,  -8.89405775e+00,   3.04785429e+00],
                 [  9.91278352e-01,   1.39423711e+00,  -2.74927108e+00],
                 [  8.16601584e+00,  -8.18443295e+00,   7.11327031e+00],
                 [  1.96693659e+00,   9.08303147e-01,  -1.34292263e+00],
                 [  1.81934599e+00,   1.08831927e+00,   1.03430913e-01],
                 [  3.37570740e+00,  -1.29718581e+00,  -1.36722190e+00],
                 [  8.29993451e-01,   1.81391901e+00,  -9.66098942e-01],
                 [ -1.38868576e-02,   1.01498804e+01,  -7.72468766e-01],
                 [ -3.92784229e+00,  -7.79424643e+00,   1.21166913e+01],
                 [ -4.17559121e+00,  -3.11546005e+00,  -9.00975915e+00],
                 [  9.65993903e+00,   7.41715695e-01,  -3.53228706e+00],
                 [  2.36915533e+00,  -5.88755260e+00,   1.80897206e+00],
                 [ -7.60058727e+00,  -2.27195984e-01,   5.90208378e+00],
                 [  2.65329850e+00,   4.97769585e-01,   4.40290861e+00],
                 [  4.62219118e-01,  -5.77083059e-01,   2.00629998e+00],
                 [  7.11423920e+00,   4.22158790e-01,  -7.71237548e+00],
                 [ -9.50712715e-01,  -4.61710650e-01,  -6.65030481e-01],
                 [  2.21929132e+00,  -1.23313662e+01,   5.45332828e+00],
                 [ -8.23996944e+00,  -4.76202546e+00,  -6.85740029e+00],
                 [  4.11259937e+00,   1.66037716e+00,   1.36992850e+00],
                 [ -4.61711110e+00,  -1.56547898e+00,   2.14109772e+00],
                 [ -2.96699751e-01,  -7.27333282e-01,   2.09227881e+00],
                 [ -2.61228547e-01,  -2.28448667e+00,  -1.90631649e+00],
                 [  1.69811998e+00,   2.32077585e+00,  -3.54525931e+00],
                 [  5.79633489e+00,   2.07272329e+00,  -8.96021366e+00],
                 [ -1.81904950e+00,  -9.02186964e+00,  -3.80858488e+00],
                 [  3.87136569e+00,  -2.75389202e+00,   1.49107744e+00],
                 [  4.79127194e+00,  -1.10513652e+01,   4.29267866e+00],
                 [  3.65623431e+00,  -9.87418097e+00,  -1.15533354e+00],
                 [  1.05602675e+00,  -4.62017697e+00,  -9.92983183e+00],
                 [  4.35909437e+00,  -5.36315159e+00,  -1.03727730e+01],
                 [ -1.10672861e+01,   9.39702044e-01,   1.51401225e+00],
                 [ -1.34119300e+00,   5.89726474e-01,  -9.35017671e-01],
                 [  5.63120431e+00,  -2.75727531e+00,   4.83693056e+00],
                 [ -6.88563252e-01,  -6.06673837e-02,  -1.92049528e-01],
                 [ -8.15372944e-01,   3.63934408e+00,  -6.27873505e+00],
                 [ -7.40238053e+00,  -8.68346863e+00,  -1.80984383e+00],
                 [ -2.72003926e+00,   3.75128681e+00,   8.18884894e-01],
                 [ -3.86316436e-01,  -9.12082066e+00,   3.27711421e+00]]).T, FortranArray([[ -6.75472167e+03,   4.72732216e+01,   5.01740690e+01],
                 [  4.72732216e+01,  -7.01383802e+03,   4.63948748e+00],
                 [  5.01740690e+01,   4.63948748e+00,  -6.91123350e+03]]).T, FortranArray([-26.53533866, -28.08517982, -28.13120429,  -9.11340672,
                  -9.52780782, -10.02664296,  -9.41825244, -10.01858901,
                  -9.94883111,  -8.31248824, -10.33945088, -27.85249809,
                  -9.81143437, -26.67893944, -11.06348011,  -9.34622838,
                 -27.56264348, -10.13111822, -28.91328145, -28.32754248,
                 -27.43619044,  -9.6583742 ,  -9.70944819, -10.25358923,
                  -9.46993507,  -9.72135672,  -9.75843297,  -9.72657679,
                  -9.7425837 , -27.30901007, -10.26375963, -27.81763884,
                 -10.4573631 ,  -9.78842877, -29.18065369,  -9.91718175,
                 -27.72415531, -27.72272388, -27.78585077,  -9.58757488,
                  -9.56712284,  -9.81190747,  -9.82230058,  -9.73608117,
                 -10.02658343,  -9.61536753,  -9.85447285, -28.07990363,
                 -10.21257994, -27.86002015,  -9.91558688,  -9.50637972,
                 -27.3951164 ,  -9.95229566, -24.5437875 , -28.64743992,
                 -27.64788038, -10.00194212,  -9.73070332, -10.00191691,
                 -10.44597593, -10.36887836, -10.10824709,  -9.76050931,
                 -10.09922671, -27.71271788,  -9.66930413, -28.85457429,
                  -9.90964154, -10.35316032, -26.17731857,  -9.758712  ,
                 -28.22339694, -28.12749192, -28.38187506, -10.02587091,
                  -9.89875081,  -9.77209665,  -9.67664072, -10.08793848,
                  -9.37034781, -10.42554673, -10.22111453, -25.97024195,
                 -10.18990292, -27.74081692,  -9.5506328 ,  -9.6157263 ,
                 -28.59133112,  -9.64157965, -27.3479587 , -27.34357419,
                 -26.50374915,  -8.48196023, -10.31891301,  -9.6988986 ,
                  -9.50591801,  -9.90874685, -10.18661609,  -9.92999135,
                 -10.0078889 , -28.58160759, -10.13642422, -26.4527198 ,
                  -9.70639267,  -9.06699781, -28.09607788,  -9.62589501,
                 -28.05797949, -28.03545416, -28.58520777,  -9.73275352,
                  -9.25326692,  -9.22312547,  -8.79696409, -10.18856669,
                  -9.82436232,  -9.69729796,  -9.63880203, -27.10554026,
                 -10.32779354, -28.60809033, -10.08294872,  -9.6335028 ,
                 -28.0181649 ,  -9.77331774, -28.59687731, -26.67285138,
                 -28.07610205,  -9.75302136, -10.1846614 , -10.27423271,
                 -10.08975601,  -9.84728985,  -9.53475039,  -9.61357341,
                  -9.3446248 , -29.16744848,  -9.16190865, -28.01757018,
                  -9.77470654, -10.04624464, -27.78264572,  -9.64969865,
                 -27.71506123, -28.09523026, -27.58644944, -10.2760171 ,
                 -10.21847911,  -9.54104513, -10.42784206, -10.03002945,
                 -10.02949571,  -9.77432123, -10.25694376, -26.89713286,
                 -10.18307505, -27.66488475,  -9.81670135, -10.11166402,
                 -27.72712033, -10.24676756, -28.05265677, -28.56221409,
                 -27.71500093, -10.50004395,  -9.57507854, -10.81130204,
                 -10.75432646, -10.14557863,  -9.83637199, -10.11227259,
                  -9.52927259, -27.99045178,  -9.59094587, -27.57461359,
                  -9.89337651,  -9.93116725, -27.02078439, -10.18854871,
                 -24.46435647, -27.52570952, -28.43872692, -10.34803273,
                  -9.46747454,  -9.95665256,  -9.81698029, -10.61740376,
                 -10.20746901, -10.16362617, -10.02396943, -27.39672485,
                 -10.35033189, -27.45473823, -10.09438773,  -8.1119513 ,
                 -27.64477698,  -9.45050299, -27.14526601, -29.40361308,
                 -29.94443825,  -9.81526617,  -9.46531332,  -9.62792298,
                  -9.35931971,  -8.12631213,  -9.75901289,  -9.04875938,
                  -9.94841536, -28.2726245 ,  -9.93235934, -27.87677389,
                 -10.31125202,  -9.80579202, -26.63685268, -10.3427307 ,
                 -25.7098347 , -27.42553355, -28.91280928, -10.33445813,
                  -9.73749198,  -9.75598738, -10.11067352, -10.14288698,
                  -9.715212  ,  -9.27173075,  -9.95064697, -27.02985168,
                  -9.29754509, -28.11090768,  -9.74201981, -10.0639008 ,
                 -28.51740238,  -9.48804615, -27.39161025, -28.97515354,
                 -28.1555096 , -10.35121144,  -9.93903614,  -9.68496233,
                  -9.88107538,  -9.74087519, -10.03124316,  -9.63560555,
                  -9.63166   , -28.19362757,  -9.37411185, -28.28404877,
                 -10.3798053 ,  -9.91247689, -28.47935604,  -9.2494661 ,
                 -25.54460773, -27.84798751, -27.24073379, -10.0673284 ,
                  -9.78276376,  -9.26455795, -10.05417114,  -8.6744686 ,
                 -10.33456617,  -9.9673588 , -10.07362258, -28.4495991 ,
                 -10.3423904 , -27.81092471, -10.16500325, -10.21652404,
                 -28.40635735,  -9.70595627, -24.89244506, -27.48988515,
                 -27.65604041,  -8.70375922,  -9.6011289 , -10.55232767,
                 -11.72985216,  -9.59648911,  -9.92277673, -10.0739856 ,
                  -9.8697622 , -27.58834637, -10.28021916, -26.63320531,
                  -9.81588587,  -9.83839193, -27.24829894,  -9.94966208,
                 -27.67625644, -28.14639489, -28.27087372,  -9.54389859,
                 -10.13241628, -10.03398511, -10.22738635,  -8.90182127,
                 -10.28020724, -10.32803574,  -9.8787375 , -29.57502629,
                 -10.07900222, -27.9725049 ,  -9.91302595,  -9.97845899,
                 -27.382176  , -10.0573472 , -27.64529812, -27.99075002,
                 -27.32969555, -10.16397981,  -9.70415555,  -8.78972961,
                  -9.64173467,  -9.7082201 ,  -9.1642392 , -10.14781041,
                  -9.99008917, -27.50760962, -10.12533992, -28.22635935,
                  -9.38373159,  -9.67579126, -28.06089121, -10.06156821,
                 -30.20022705, -27.06659274, -27.97658524, -10.27054606,
                  -9.94480646,  -9.20068955,  -9.36320967, -10.16094179,
                  -8.94433929,  -9.8101368 ,  -9.77498455, -27.98840428,
                  -9.54045645, -28.21624068,  -9.70046068, -10.06483614,
                 -27.56396741, -10.14658599, -28.64439558, -26.89695375,
                 -28.64708027,  -9.69040049,  -9.44199421,  -9.78472485,
                  -9.60669533,  -9.72739191, -10.06390278,  -9.31295041,
                  -9.7504831 , -28.06601524, -10.16287686, -27.94884107,
                  -9.87019261,  -9.7260738 , -28.89818801, -10.06938647,
                 -27.95817976, -27.51002156, -27.59317269,  -9.79174016,
                 -10.2296258 ,  -9.62434217, -10.1272846 , -10.03045643,
                 -10.18080185,  -9.61788303, -10.08143449, -27.08892086,
                 -10.03909902, -28.31021423,  -9.90758106,  -9.97312159,
                 -25.27869811,  -9.72393779, -26.91530751, -27.54338246,
                 -26.93678818, -10.19165674, -10.38208986,  -9.8912535 ,
                  -9.95989434, -10.03915619,  -9.81655859,  -9.30428921,
                  -9.86545838, -27.77753294,  -9.85338248, -27.88712114,
                 -10.45705155,  -9.93964366, -27.3791666 , -10.21190498,
                 -26.89792769, -27.8248683 , -30.59364954,  -9.93633472,
                  -8.49578694, -10.31875567, -10.47130493, -10.57677875,
                  -9.42657348,  -9.88859712, -10.13389448, -28.55098446,
                  -9.25725505, -27.38989992,  -9.67036276,  -9.91934843,
                 -25.38501282,  -8.9162071 , -27.39008103, -27.60199193,
                 -27.01738774, -10.23816763, -10.04928166, -10.9332127 ,
                 -10.11413533,  -9.40494083,  -9.8992389 , -10.05509454,
                  -9.55736749, -26.90934068, -10.33159982, -26.94934543,
                  -9.73765686, -10.0467949 , -28.81355202,  -9.70610573,
                 -28.48967113, -26.53729553, -27.31176599, -10.23082183,
                 -10.18062036,  -9.92709175,  -9.45304119,  -9.49043516,
                 -10.12306217, -10.12262961,  -9.94775057, -27.56597678,
                 -10.15208808, -27.35033036, -10.33699975, -10.18057317,
                 -27.97407105, -10.73487772, -26.89705614, -27.21478657,
                 -27.65452137,  -9.78809482,  -9.43833068,  -9.38219394,
                 -10.22233911,  -9.70478359, -10.00843449,  -9.76723161,
                  -9.57840756, -28.50632119,  -9.78105146, -27.60376499,
                 -10.64039163,  -9.89105012, -27.96897247,  -9.79243051,
                 -28.08117123, -28.47660784, -27.78724093,  -9.40761435,
                  -9.73965073, -10.20280437,  -9.76625414, -10.10152564,
                 -10.31797272, -10.02555479, -10.31526493, -27.60026325,
                  -9.83906399, -28.21138553, -10.22137233,  -9.19551516,
                 -28.42554245,  -9.92017015, -27.92608341, -27.15446286,
                 -27.64845089,  -6.29327354,  -9.31130909, -10.08398593,
                 -10.54601379, -10.12485982,  -9.59197354, -10.06620393,
                  -9.96738644, -28.541378  ,  -9.63134586, -26.21868706,
                 -10.16649708,  -9.31923499, -27.24645836, -10.61426371,
                 -27.39402205, -27.71139957, -28.02726776,  -9.55221178,
                 -10.16085166,  -9.79530857,  -9.72684961, -10.24150961,
                 -10.15839826,  -9.57542596, -10.09771822, -27.8679615 ,
                  -9.66808138, -27.1667186 , -10.50411982, -10.6094072 ,
                 -27.71108664,  -9.85281726, -27.92930148, -28.08281226,
                 -28.49249574, -10.30205565,  -9.57557525, -10.09784363,
                 -10.09241951, -10.02454579,  -9.98364299,  -9.86966683,
                  -9.93112168, -27.52165845,  -9.9895256 , -28.40185615,
                 -10.22934161, -10.0995813 , -26.37237945,  -9.61129956,
                 -28.34922973, -27.81475227, -27.3287575 ,  -9.60230009,
                  -9.56649453,  -9.75622772, -10.08339706,  -9.32656232,
                  -9.96592511,  -9.29608045, -10.30101238, -27.79446094,
                  -9.99157066, -28.52615682,  -9.67015799,  -9.74494533,
                 -27.69527435,  -9.80405621, -27.0037111 , -26.47852307,
                 -28.06915767, -10.30969798,  -9.64393705, -10.28802418,
                  -9.7076849 ,  -9.62419643,  -9.56752809,  -9.36848335,
                  -9.74685086, -24.40352971, -10.53136595, -27.74934507,
                  -9.44694173, -10.45978499, -27.57345934,  -9.35896156,
                 -25.2058819 , -28.293245  , -28.01586193,  -9.89375713,
                 -10.25776703,  -9.66865609, -10.00245734,  -9.62735271,
                 -10.31096856,  -9.15555035, -10.44911248, -27.15408749,
                  -9.63198632, -26.07758707, -10.62210188,  -9.67406165,
                 -28.07448342, -10.12955482, -27.56319684, -25.47932642,
                 -26.8994445 ,  -9.98106222, -10.08556863, -10.22961109,
                  -9.88628574, -10.32001915, -10.41370875, -11.77632918,
                 -10.03693138, -28.09682953, -10.17644945, -28.42760125,
                  -9.48428292,  -9.55137372, -27.66227133, -10.73431667,
                 -27.55904809, -26.91750088, -26.35074117,  -9.42582104,
                 -10.61933533, -10.11378185, -10.16720958,  -9.90607311,
                 -10.24313072,  -9.65112383,  -9.92578856, -27.42126528,
                 -10.33423211, -27.98292178,  -9.85537495,  -9.69765542,
                 -27.08680897,  -9.78209326, -28.13691851, -28.5339756 ,
                 -25.65852241, -10.29951594, -10.39045367, -10.38464304,
                  -9.39036075,  -9.56241958,  -9.88402768,  -9.17037873,
                 -10.41888754, -27.02036913,  -9.95650083, -27.15659261,
                 -10.05899935,  -9.64536692, -28.16343991, -10.23564115,
                 -27.99345549, -27.9362645 , -26.98174169,  -9.87177419,
                  -9.92763239,  -9.54700859,  -9.84756367, -10.57628723,
                 -10.13120659, -10.38795638,  -9.70818102, -27.89240983,
                 -10.21362618, -28.16798093, -10.01960397,  -9.41532616,
                 -26.89458388, -10.3902232 , -27.43809147, -27.74084353,
                 -26.25127012, -10.29432722, -10.04145743,  -9.86600382,
                  -9.86510754,  -9.79418197, -10.01530382, -10.35027164,
                 -10.03011446, -26.99715902,  -9.90754535, -25.07976261,
                 -10.15878664,  -9.84429507, -27.50796195,  -9.79018784,
                 -26.55397617, -26.65542074, -27.84846875,  -9.6286393 ,
                  -9.26378104, -10.2674193 ,  -9.61858222,  -9.93483891,
                  -9.8604411 ,  -9.62662149,  -9.62247709, -28.80533442,
                  -6.30596604, -29.11758588, -10.4817088 , -10.08574423,
                 -27.85539847,  -9.75519622, -27.20369991, -27.93428414,
                 -28.48183548, -10.36998775, -10.02173176,  -9.81337653,
                  -9.86866691,  -9.81349522,  -9.5174622 ,  -9.99738397,
                  -9.83831304, -27.79242695, -11.8401142 , -26.09459937,
                  -9.73611269, -10.12087693, -27.65924874, -10.22716347,
                 -28.5496265 , -28.31231809, -27.54926257,  -9.66709227,
                 -10.10190835, -10.17075106,  -9.54914706,  -9.61436911,
                  -9.89842956,  -9.59620868, -10.452698  , -27.77299354,
                 -10.30660363, -27.53359631,  -9.13977039,  -9.78041416,
                 -28.07540925,  -9.99511594, -27.91122438, -26.98136952,
                 -27.64382083,  -9.59649706,  -9.71853137,  -9.59988541,
                  -9.76611168,  -9.7540934 ,  -9.93442529, -10.2792472 ,
                  -9.6595393 , -27.66866304, -10.06409339, -27.495008  ,
                  -9.88985338,  -9.90386076, -27.7026945 ,  -9.95085419,
                 -27.3468151 , -25.98630417, -27.66925637,  -9.94345125,
                 -10.78469918, -10.70566775, -10.30764126,  -9.89135139,
                  -9.43082656,  -9.87282165,  -9.77911848, -26.14764247,
                  -9.26890923, -28.21357134, -10.09753307, -10.03038537,
                 -26.81738216,  -9.92427747, -26.97363776, -25.54999052,
                 -27.87216552,  -9.5525754 , -10.93738206,  -9.75281927,
                 -11.83014996, -10.41911476,  -9.72439775, -10.66120084,
                 -10.5142012 , -26.85005873, -10.05987179, -28.04911284,
                  -8.34206522, -10.12528021, -27.43152273,  -9.47978984,
                 -27.08339146, -28.45481707, -26.72523867, -10.15079445,
                  -9.4537857 , -11.6155498 , -10.13370803,  -9.68726905,
                 -10.02630801,  -9.60765697, -10.3247376 , -26.03139395,
                 -10.58909552, -28.00909179, -10.10249135, -10.14607674,
                 -27.89191337, -10.20335293, -25.74907237, -27.79426813,
                 -27.5565103 , -10.66409874,  -9.84353846, -10.14911636,
                 -10.38070206, -10.18074687, -10.1970299 , -10.40450398,
                 -10.05092337, -28.07309243,  -9.82558786, -27.84906206,
                  -9.29182569,  -9.8875696 , -27.30191932,  -9.63294069,
                 -28.12270327, -29.12213479, -28.63925008,  -9.91356145,
                  -9.67065471,  -9.87907658,  -9.2770582 ,  -9.10669634,
                 -10.43404321,  -8.8785096 ,  -9.82958584, -27.76275153,
                  -9.98227465, -27.72142493,  -9.33171159,  -9.78323092,
                 -29.07659581,  -9.96021962, -28.65644084, -28.04387256,
                 -26.97723407,  -9.80930083, -10.03128067, -10.05558852,
                 -10.00990722,  -9.83291555,  -9.57792365, -10.3843642 ,
                  -9.37221359, -27.59645339,  -8.5680878 , -29.0007317 ,
                  -9.87142555,  -9.49118833, -27.49874148,  -9.78147588]), FortranArray([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.87705907e-02,   1.78809195e-02,  -3.64520523e-02],
                 [  2.28183723e-02,  -1.15061107e-02,  -1.74125840e-02],
                 [ -1.04110462e-02,  -6.00538967e-03,   8.34981372e-03],
                 [ -8.88594500e-03,  -1.73614831e-02,   2.14373453e-02],
                 [ -2.49750921e-02,   1.29712663e-02,   1.91329607e-03],
                 [ -1.05729048e-03,  -1.95399301e-02,   1.59888450e-02],
                 [  6.41571374e-03,   3.02682446e-02,   3.03544776e-03],
                 [  9.83035116e-03,   1.19735154e-02,   2.62141105e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.80150963e-02,   2.70333780e-02,   1.19611273e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.94568137e-03,  -6.31916802e-03,   2.38545477e-02],
                 [ -2.76601942e-02,  -1.45308660e-04,  -3.72995837e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.76212809e-02,   6.15797160e-03,  -8.91237119e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.52966339e-02,  -1.96491632e-02,  -4.07558128e-03],
                 [ -3.29657444e-03,  -1.42350019e-02,  -1.46399239e-02],
                 [ -4.95118203e-04,  -1.26269798e-02,  -6.70909653e-03],
                 [ -1.45361687e-02,  -2.33579256e-02,  -4.73049209e-03],
                 [  1.68132697e-02,   1.39695079e-02,  -8.75360081e-03],
                 [ -1.24654630e-02,   2.02247114e-02,  -1.22301579e-02],
                 [  8.89520432e-03,   7.18966488e-03,   8.59299808e-03],
                 [ -2.46685667e-02,   1.90147207e-03,  -1.06960484e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.19217747e-02,  -3.46287290e-03,  -7.89579208e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.74044491e-02,  -7.65772697e-06,  -1.22440998e-02],
                 [  2.18825883e-02,   5.15869999e-03,  -6.92911364e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.12951778e-03,  -4.11950170e-04,   1.72317220e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.85758096e-03,   3.78165524e-03,  -2.34419588e-02],
                 [ -2.12483220e-02,  -1.28735244e-02,  -2.98325546e-02],
                 [ -8.35330010e-03,  -2.19962246e-02,  -3.25134707e-03],
                 [  7.45013794e-03,   2.01175344e-02,  -1.93906565e-02],
                 [  2.40523432e-02,   2.07726560e-02,   1.13266215e-02],
                 [ -1.02249110e-02,  -1.41019732e-02,  -6.65881446e-03],
                 [ -9.19993840e-03,  -2.88053157e-02,   2.49366712e-03],
                 [ -1.55779495e-02,   3.85880544e-03,  -6.08760100e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.63865688e-02,   5.21834983e-03,  -1.12818825e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.72383864e-03,   3.14815522e-02,  -6.13559633e-04],
                 [  1.02835187e-02,  -4.64774586e-03,   3.07823761e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.31578292e-02,  -1.60954230e-02,   1.65207273e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.88423404e-02,   9.97499812e-03,  -3.80128874e-02],
                 [  2.09115661e-02,  -1.60752903e-02,   1.23727524e-02],
                 [ -1.57947784e-02,   2.73731800e-02,   2.30274882e-02],
                 [ -2.95261573e-03,  -1.59116197e-02,  -8.00264648e-03],
                 [ -3.18763905e-03,  -2.18649424e-02,   1.49334986e-02],
                 [ -4.92793614e-03,   1.38355462e-02,   2.08123128e-02],
                 [  2.74684639e-02,  -1.34789983e-02,   6.82600858e-03],
                 [ -3.53702611e-03,  -3.90329165e-03,  -2.02292723e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.80844377e-02,   1.74213589e-02,   1.17544962e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.92731891e-02,   9.58566269e-04,   2.98819873e-03],
                 [  2.17078997e-02,  -8.10719530e-03,  -2.21843987e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.80954319e-02,  -2.98633362e-03,  -1.67110364e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.02768313e-03,  -2.07964708e-02,  -7.12012707e-03],
                 [  1.01049524e-02,   1.07991771e-02,   5.79059520e-03],
                 [ -1.36795467e-02,   1.29666106e-02,   2.41303179e-02],
                 [ -1.32554451e-02,   1.42235294e-02,  -2.15705397e-02],
                 [ -6.37029279e-03,  -1.21442096e-02,   1.91949936e-02],
                 [  2.67685045e-02,   1.53484854e-02,   7.14435683e-03],
                 [ -2.45538331e-03,  -4.51782288e-03,   1.15759668e-02],
                 [  8.68210069e-03,   1.72252808e-02,   1.24578161e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.33243760e-03,   2.27879209e-02,   5.53178668e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.96709547e-02,  -2.67666380e-02,   2.59648434e-03],
                 [ -2.10650626e-02,   1.23955418e-02,   5.85419404e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.17168673e-03,  -2.39662783e-02,  -1.64599472e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.08962832e-02,   1.00156155e-02,   9.43034094e-03],
                 [ -2.05193112e-02,  -1.03210656e-02,   1.78624876e-02],
                 [ -6.34042807e-03,   2.03468347e-02,   2.24024500e-02],
                 [  2.02247910e-02,  -6.34642677e-04,   1.93200237e-02],
                 [ -4.40232948e-03,  -5.05948567e-03,  -2.22025227e-02],
                 [ -1.71706914e-03,   2.33718120e-02,  -1.42667551e-02],
                 [  6.52396045e-04,   4.88775915e-03,   2.71091660e-02],
                 [  4.05537046e-03,  -1.05854390e-03,   1.57471017e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.02298448e-03,   2.89688360e-02,  -8.57170103e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.46588444e-02,  -3.11570682e-02,   6.34544311e-03],
                 [  1.45662633e-02,   2.09930600e-02,   4.14098037e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -9.66123973e-03,  -2.30216241e-02,   1.02073780e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.23172831e-03,   1.79426361e-02,   1.80705468e-02],
                 [ -1.24145323e-02,  -3.60330598e-03,   7.25730051e-04],
                 [  8.34840856e-04,  -2.83382001e-02,  -1.16349322e-02],
                 [ -1.53920835e-02,  -1.41668564e-02,  -9.46962185e-03],
                 [  1.92924779e-02,   5.60273827e-03,   1.41708591e-02],
                 [  1.08536693e-02,  -1.38094678e-02,  -2.77163707e-02],
                 [  2.07248436e-02,  -1.86249590e-02,  -1.21888325e-02],
                 [  6.53264216e-03,  -1.34958905e-02,  -2.06068587e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.35855612e-03,   6.87450287e-03,   1.59726232e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.74273531e-03,   1.77125155e-02,   9.91738689e-03],
                 [ -1.24131035e-02,  -1.91672308e-02,  -2.34954429e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.24145515e-02,  -2.77629313e-02,  -5.76499028e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.50880780e-03,   1.02066421e-02,  -2.33650885e-03],
                 [ -1.37280796e-03,   5.98292609e-03,   1.88896194e-03],
                 [  1.06341686e-02,  -1.06249543e-02,   2.72739921e-02],
                 [  2.43597250e-02,  -1.98793077e-02,  -7.58249763e-03],
                 [ -9.68784802e-03,   9.93502972e-03,  -1.45881676e-02],
                 [ -5.08057446e-03,   3.47606804e-02,  -4.27701039e-03],
                 [ -7.98330558e-03,  -2.71391883e-03,   2.64857115e-02],
                 [  2.19688771e-02,  -1.32930893e-02,   2.44776938e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.23585070e-02,  -8.42213926e-03,   1.56327977e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.78553526e-02,   1.61679880e-02,  -1.55447278e-02],
                 [  1.99335239e-03,   2.08352286e-03,  -1.32837653e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.67147453e-02,  -5.40555895e-03,   2.95540655e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.08149102e-02,   2.60464406e-02,  -1.33848866e-03],
                 [  2.46140093e-02,  -1.84974136e-02,   6.79694163e-03],
                 [ -7.71242501e-03,  -3.40317346e-02,   1.67484580e-02],
                 [  2.65620393e-03,   2.23671947e-02,  -3.38012088e-04],
                 [  6.39682775e-03,  -3.53969783e-03,  -2.19448071e-02],
                 [  5.83782418e-03,  -2.34850636e-02,  -1.97958304e-02],
                 [  1.96034129e-02,   1.73102778e-02,   1.44412498e-02],
                 [ -2.29898919e-02,  -9.13119811e-03,  -8.96055803e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.42965435e-03,   1.81269250e-02,   2.62697618e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.56676613e-02,  -2.22433637e-02,   1.49546088e-03],
                 [ -2.71282577e-02,   1.16194886e-03,   2.01771287e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.46193784e-03,  -4.17423641e-03,   8.95183938e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.28428064e-02,  -1.82491946e-02,   1.02849091e-02],
                 [ -1.00698997e-02,  -1.91914781e-02,  -8.77709877e-03],
                 [ -3.26778952e-02,  -8.34194122e-03,   8.97843616e-03],
                 [ -8.00112393e-04,  -2.76102836e-02,  -1.38557034e-02],
                 [ -4.80905207e-03,   1.23177046e-02,  -2.91407888e-02],
                 [ -5.63502728e-03,   3.41423567e-02,   1.96062877e-03],
                 [ -3.48822700e-03,   1.74352684e-02,   7.52184586e-03],
                 [ -1.30758984e-02,  -1.39572385e-02,   1.87847097e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -8.01298526e-03,   9.65636373e-03,   1.42346650e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.46463398e-02,  -7.86698765e-03,  -2.48480895e-02],
                 [ -5.85524566e-04,  -5.97162993e-03,  -2.18628357e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.54233027e-02,  -8.94435552e-03,  -1.57624856e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.94509455e-02,   3.06322280e-03,  -4.84911880e-03],
                 [  2.20174547e-02,   2.67142927e-02,  -3.25020634e-04],
                 [  1.20195509e-02,  -1.04385999e-02,   8.96779792e-03],
                 [  1.40860348e-02,  -9.42430328e-04,  -1.15165842e-02],
                 [ -1.18095942e-03,  -5.19804728e-03,  -1.04381746e-02],
                 [  2.41437798e-03,  -5.92001390e-03,   9.32641001e-03],
                 [ -1.27511832e-02,   1.79013174e-02,   6.16284301e-04],
                 [  5.48527849e-03,  -4.49118066e-03,  -6.35555332e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.07706778e-04,   4.72211592e-04,   2.93524113e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.95612252e-02,   1.90454365e-03,  -2.31231706e-02],
                 [  1.56349001e-02,  -1.98161286e-02,   8.77126083e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.74845291e-04,  -2.19564367e-02,  -5.46002908e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.01958798e-02,  -1.71287918e-02,  -2.50471377e-02],
                 [ -1.92358270e-02,   1.94857414e-03,   1.31394714e-02],
                 [ -4.29170026e-03,  -1.85453425e-02,   8.64172065e-03],
                 [ -2.62321342e-02,   2.15432999e-03,  -1.30770837e-02],
                 [ -1.13921012e-02,  -1.84176773e-02,  -7.20602622e-03],
                 [  3.50652678e-03,   3.31017087e-02,   1.08469029e-03],
                 [ -2.21840804e-03,   2.20465828e-02,   1.41570173e-02],
                 [ -2.16226906e-03,   1.81011446e-02,  -7.98132555e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.40023956e-03,   9.13177849e-03,  -1.57173814e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  9.40758815e-03,   1.15729952e-02,  -1.65024930e-03],
                 [  3.49697296e-03,   1.28715486e-02,   3.02632196e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.18100546e-02,  -5.50137617e-03,   5.93342382e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.26468857e-03,  -7.76003745e-03,   8.44362153e-03],
                 [ -1.26588605e-03,  -2.59108407e-02,   1.70776713e-02],
                 [  1.16568811e-02,   3.22830908e-02,   1.69774429e-02],
                 [  8.38699296e-03,  -2.06791445e-02,   2.48618805e-02],
                 [ -1.71110909e-02,  -1.61439669e-02,   9.72501010e-03],
                 [ -1.35734785e-02,  -4.64752920e-03,  -2.93805452e-02],
                 [ -3.23848882e-03,   2.56520445e-02,  -8.58212610e-03],
                 [ -1.41650356e-02,  -7.01548344e-04,  -7.44927510e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.21492120e-05,   1.75461511e-02,  -1.08990591e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.37089836e-02,   3.17097927e-02,  -2.59309679e-03],
                 [ -2.82354784e-03,   3.52576043e-02,   1.50583247e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.43200130e-02,  -2.15773784e-02,  -7.90791624e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.00156756e-03,   2.42529503e-03,  -5.70095365e-03],
                 [  2.38641135e-02,  -2.46193036e-03,   8.55844483e-04],
                 [ -6.85699527e-03,  -7.53046958e-03,  -1.64274208e-02],
                 [  2.37398429e-02,   3.71729429e-03,   1.42891383e-02],
                 [  8.02989328e-03,   2.02215410e-02,   1.76016506e-02],
                 [ -4.11774704e-04,  -1.99875337e-03,  -4.07171017e-03],
                 [ -6.52457403e-03,  -2.76486995e-03,   1.50069244e-02],
                 [  8.34043329e-03,  -2.10383903e-02,  -7.99036633e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.56431470e-02,  -1.78030662e-02,   3.05082048e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.76938754e-02,  -1.19918107e-02,   1.39243074e-02],
                 [ -2.62841222e-02,   1.07226101e-02,   8.08399919e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.46341270e-02,  -3.11375822e-03,   4.06913838e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.53709275e-03,  -9.60323243e-03,  -5.37932076e-03],
                 [  2.84672737e-02,   9.96268520e-03,  -9.15500819e-03],
                 [ -2.14284248e-02,  -3.39204147e-03,  -9.09214851e-03],
                 [ -1.89584457e-02,  -1.09731073e-02,   1.95247828e-02],
                 [ -2.18688433e-02,   5.40959247e-03,   1.74756471e-02],
                 [  1.99203791e-02,   4.75039348e-03,  -1.87677059e-02],
                 [  1.50172187e-02,   5.64329363e-03,   1.43957299e-03],
                 [  7.74880441e-04,   2.85847948e-03,  -8.00508165e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.14765873e-02,   1.13202915e-02,  -7.33197681e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.13271637e-03,  -8.75633190e-03,  -2.74774870e-02],
                 [  9.22537570e-03,  -3.07125051e-03,  -5.96563813e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -7.31527427e-03,  -1.96830731e-02,   2.31595802e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.18641184e-02,  -2.70973161e-02,   1.18858322e-02],
                 [  2.72764695e-02,  -1.52381163e-02,   9.12082398e-03],
                 [  5.10282384e-03,  -2.05284885e-02,  -5.26144411e-03],
                 [  3.07555245e-03,  -7.85843760e-03,   5.72835540e-03],
                 [  5.55948101e-03,   3.54547555e-02,   5.24291314e-03],
                 [ -6.63281973e-03,  -9.10389529e-03,  -4.79051570e-03],
                 [ -1.44815204e-02,  -2.08624021e-02,  -5.37099549e-03],
                 [ -1.33097654e-04,  -2.03113474e-02,  -2.41726859e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.89083532e-02,  -3.78155655e-03,   8.03757358e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.53898797e-03,   7.75047882e-03,   1.20455867e-02],
                 [ -1.63165776e-02,   1.46853224e-02,   9.38634257e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -8.18361898e-03,   1.69037144e-02,  -2.35055835e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.05124086e-02,  -2.57850310e-02,  -1.12280805e-03],
                 [ -3.26774401e-03,  -7.84026491e-03,  -4.08189827e-03],
                 [  1.88827732e-02,  -5.61632377e-03,  -7.29362191e-03],
                 [ -1.14985343e-02,   1.85643532e-02,  -4.46135604e-03],
                 [ -2.80100806e-02,   8.25656474e-03,  -1.24036943e-02],
                 [  4.57668601e-03,   1.03615014e-02,   4.85308585e-03],
                 [  1.06702770e-02,   1.72222511e-03,   5.26423991e-03],
                 [ -3.05865628e-03,  -2.70864013e-02,   5.24964506e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.26811733e-02,   1.59287774e-02,  -2.24850257e-05],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  9.77543585e-03,  -5.19537833e-03,  -1.89663749e-02],
                 [ -1.44010602e-02,   1.09876421e-03,  -8.24098870e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.42751974e-02,  -3.60652053e-04,  -1.79382072e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  9.33541607e-03,  -1.63413073e-02,  -9.38901297e-03],
                 [  1.24867572e-02,  -2.43283689e-02,   1.52245654e-02],
                 [ -9.84944460e-04,   3.60918737e-04,   1.54347374e-02],
                 [  2.50789615e-03,   1.36906408e-02,   2.62992104e-02],
                 [ -1.35229398e-02,   1.73701044e-02,  -1.27235333e-02],
                 [ -1.03869707e-02,  -1.85355793e-02,   1.15793452e-03],
                 [ -1.40241429e-02,  -2.42842686e-02,  -3.06215227e-03],
                 [  1.22729368e-02,   1.17620004e-02,  -1.38504105e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -9.52192506e-03,  -1.94583860e-02,   1.51801292e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.28967248e-02,   1.96368081e-02,  -1.75817373e-02],
                 [ -8.17464192e-03,  -9.53303163e-03,  -2.08836077e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.10012530e-02,   4.46937399e-03,  -1.05979567e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.59851664e-02,   4.14718953e-03,  -1.03754543e-02],
                 [  1.16102802e-02,  -4.60238879e-03,  -2.06119818e-02],
                 [  1.54988109e-02,  -2.36548313e-03,  -1.54590652e-03],
                 [ -2.67747403e-02,   3.68054501e-03,  -1.15293486e-02],
                 [  3.31785819e-03,   1.22508869e-02,   1.37463196e-02],
                 [  2.11614398e-02,   1.06575760e-02,  -1.18236230e-02],
                 [  1.95043492e-02,  -1.87771092e-03,   1.34554098e-02],
                 [ -1.34514876e-02,  -1.09353082e-02,  -1.40154524e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.74647179e-02,  -1.65891322e-02,  -2.04691403e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.65648892e-03,  -6.38692023e-03,   1.26183924e-02],
                 [  4.55054151e-03,   1.06231525e-03,  -8.04359421e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.85303879e-03,  -4.92555176e-03,   5.23328192e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.67311679e-04,   1.66618669e-03,   3.19095921e-03],
                 [  2.54850773e-02,  -1.42436596e-02,  -7.31406605e-03],
                 [ -1.63555099e-02,   1.10777088e-03,   1.53544661e-02],
                 [  2.09060324e-02,   3.03609290e-03,  -1.99129327e-02],
                 [  2.49711572e-02,  -4.04627718e-04,  -1.16201978e-02],
                 [  6.18594198e-03,   8.67937518e-03,   1.80048130e-02],
                 [  2.34806388e-03,  -1.73990001e-03,  -2.83863974e-02],
                 [  1.25622377e-02,  -4.02321757e-03,   2.49503423e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.20361524e-03,  -2.13607093e-02,   1.36978419e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.48939429e-02,  -7.51145084e-03,   9.25358060e-03],
                 [  7.75269267e-03,   1.14161658e-02,  -2.22706005e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.33704025e-02,   1.27165661e-02,   1.87116875e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.16859658e-03,   1.12653865e-02,   2.74367863e-02],
                 [  4.87689424e-03,   1.78524298e-02,  -2.43619025e-02],
                 [  3.15737794e-02,  -2.94130868e-03,   8.18406467e-03],
                 [ -9.09574394e-03,  -1.50269221e-02,   2.02822915e-02],
                 [  6.53350196e-03,  -5.82445723e-03,   2.61776584e-02],
                 [ -1.11027068e-02,   2.50272002e-02,   5.21082577e-03],
                 [ -1.32944862e-02,  -3.42515809e-03,  -1.61025646e-02],
                 [  1.99817335e-02,   1.83640889e-02,   4.66497498e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.03918145e-02,   1.49783429e-03,   2.30398718e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.27006792e-02,  -3.25747891e-02,  -7.21893886e-03],
                 [  1.46515815e-02,  -2.58185802e-02,  -1.69928528e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.00287606e-02,   1.89365858e-02,   4.95921775e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.91209339e-04,  -1.04827800e-02,   2.84427669e-02],
                 [  8.12248037e-03,  -1.01815168e-02,  -1.87568300e-02],
                 [ -7.21727438e-03,   2.57689162e-02,   1.61174455e-02],
                 [  6.24503818e-03,   1.00673372e-02,  -3.42180643e-02],
                 [  8.86829793e-03,  -5.26272239e-03,   1.05004764e-02],
                 [  2.75267806e-03,   3.08173849e-02,  -1.81956249e-03],
                 [  4.05489792e-02,  -2.16913331e-02,   8.53592798e-04],
                 [  2.11455095e-02,  -5.62936262e-03,  -1.13643720e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.58107044e-03,   2.85156544e-02,  -1.12113432e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.91757118e-03,   4.12133964e-04,   4.00917040e-03],
                 [  2.33075169e-02,   1.28473272e-02,   1.23637908e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.57969550e-02,   4.14269068e-03,   1.56097586e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.71997948e-02,  -2.39696783e-02,   1.81469166e-03],
                 [ -1.67660740e-02,  -1.10155288e-02,  -9.80905421e-03],
                 [  6.07008472e-03,  -9.23117292e-03,   2.65066309e-02],
                 [  6.84524060e-03,  -1.88501013e-02,   1.15129641e-02],
                 [ -3.13659870e-02,   4.14184712e-03,  -1.02898140e-03],
                 [ -9.27802388e-03,  -1.55385125e-02,  -3.42057604e-02],
                 [ -1.19104995e-02,   1.52408280e-02,   1.16060097e-02],
                 [  5.28416376e-03,   2.53261156e-02,  -1.41618228e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.25504805e-02,   6.44923989e-04,   7.56988184e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.81726210e-02,  -1.50958084e-02,   8.45894109e-03],
                 [ -1.44871325e-04,   1.75725549e-02,   6.69961079e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.28314127e-04,  -3.42461613e-02,  -6.04943994e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.46547952e-02,  -7.40867120e-03,   2.41880928e-02],
                 [ -1.04847381e-03,   1.10268676e-04,  -3.32982182e-02],
                 [ -9.11365582e-03,   2.37125229e-02,  -7.12389416e-03],
                 [  2.12449745e-02,  -3.69842548e-02,   4.29979142e-03],
                 [ -8.22743328e-03,   7.77988723e-03,   1.67427505e-03],
                 [  5.14229221e-03,   3.15833942e-02,   1.71605827e-02],
                 [  5.30543472e-03,   1.38756543e-02,  -1.90034685e-02],
                 [  7.91977682e-04,  -2.08839763e-02,   6.67181255e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.43879384e-03,  -4.93594342e-03,   4.91178228e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.23431082e-02,  -5.22856431e-03,  -1.24298745e-02],
                 [  6.05151964e-03,   7.63392941e-03,  -2.77138889e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.72214397e-02,   1.30591360e-02,  -7.76672706e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.88633372e-03,   2.56404839e-02,   7.78822636e-03],
                 [ -2.42269762e-03,   3.40839145e-02,   7.92710758e-03],
                 [ -6.66773527e-03,  -1.20936213e-02,   1.33677737e-02],
                 [ -1.52597494e-02,  -3.19350659e-03,  -3.39836091e-02],
                 [ -1.19887402e-02,  -1.17475511e-02,   2.71353203e-02],
                 [ -5.36915972e-03,  -7.07281042e-03,  -4.53179837e-03],
                 [  8.45088175e-03,   3.72022605e-03,  -6.82430221e-03],
                 [ -2.48093520e-02,   3.86553774e-03,  -1.16548641e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.38209333e-02,  -5.42224930e-03,   1.43319786e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.23335318e-02,  -9.30392241e-03,   7.91678698e-03],
                 [  1.04986552e-02,   6.66975108e-03,  -5.29976314e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.76976123e-02,   2.08544851e-02,   2.45683248e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  9.28080422e-04,   2.17023390e-03,   1.79583367e-03],
                 [  2.25601071e-02,  -1.46218297e-02,   1.86370261e-02],
                 [ -4.47938645e-02,   2.05804625e-02,   5.93176788e-04],
                 [  6.68747104e-03,  -9.73580928e-03,  -2.26774567e-03],
                 [  2.40496157e-02,  -2.64933902e-02,  -6.51066480e-04],
                 [  4.64377981e-04,   3.03086301e-04,   3.06748733e-02],
                 [  1.62144700e-03,   2.22397182e-02,  -1.69544544e-02],
                 [  2.53420690e-02,  -4.69688471e-03,   2.02812043e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.16579348e-02,   2.21488159e-02,  -1.66400958e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.29625813e-03,   1.93226459e-02,   1.02164257e-02],
                 [  1.52286784e-02,  -3.83107876e-02,  -1.39240794e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.10504616e-02,   1.11408353e-02,   2.27059023e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.94129112e-02,   2.30814066e-03,  -1.78737621e-02],
                 [ -1.01189577e-02,  -1.91015246e-02,  -9.36806766e-03],
                 [  4.41880325e-03,  -1.53845914e-02,   8.94322775e-03],
                 [  3.62022663e-03,   2.67841632e-02,   1.05090662e-02],
                 [ -7.15468435e-03,   1.28882580e-02,  -1.26973488e-02],
                 [ -1.37283056e-02,   6.24398563e-03,   2.06741758e-03],
                 [  1.00554163e-02,   2.38610239e-02,   1.00239003e-02],
                 [ -4.74850242e-03,  -1.03181861e-03,   2.72272800e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.48251010e-03,  -2.67365179e-02,   5.19867767e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.43401452e-02,  -5.28043211e-04,   2.33716132e-03],
                 [ -1.40663899e-02,  -2.33254991e-03,  -1.65904969e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.96325486e-03,   5.61962180e-04,  -6.06734876e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.62327823e-02,  -3.55971872e-02,  -1.37141072e-02],
                 [  2.80941598e-02,  -1.44809953e-02,   1.10403517e-03],
                 [  9.73857221e-03,   5.79754309e-03,  -5.46390497e-03],
                 [ -2.17657391e-02,   3.19656879e-03,  -1.25239144e-02],
                 [  3.16000446e-03,   1.59336095e-03,   2.43543440e-03],
                 [  1.49788166e-02,   1.11851893e-02,   2.53926875e-02],
                 [ -1.61076527e-02,  -6.73093969e-03,   1.13117965e-02],
                 [  1.59294279e-03,   1.57522119e-03,  -2.50475807e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.19243526e-02,   2.25580024e-02,  -1.10024064e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.65108113e-02,  -2.42791343e-03,  -1.50957087e-02],
                 [ -2.09955044e-02,  -2.90772045e-02,   7.33614294e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.91411085e-03,  -6.81886246e-03,   1.52194975e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.09056714e-02,  -9.21490220e-03,  -1.15963663e-02],
                 [  7.64605297e-03,   1.79002437e-02,   9.00257406e-03],
                 [ -6.01748496e-03,   2.12527392e-02,   1.58541651e-02],
                 [ -1.20515183e-02,  -1.59382929e-02,  -7.97828816e-03],
                 [  1.60413838e-02,   2.62334913e-02,   8.15929986e-03],
                 [  1.64880812e-02,   1.74035445e-02,   2.27476063e-03],
                 [ -1.23060439e-02,  -2.06359133e-02,   1.26797556e-02],
                 [  4.84042518e-03,   6.42762971e-03,  -1.37786131e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  5.38224998e-03,  -1.01771334e-02,   3.13932380e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.26091085e-02,   1.78575721e-02,   2.41087691e-02],
                 [ -1.87586956e-02,  -9.94744123e-03,  -5.96582136e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.11368270e-02,   1.01074240e-02,   2.36517515e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -8.14130073e-03,  -7.34002175e-03,  -2.30403109e-03],
                 [  2.30699539e-02,  -9.22531898e-03,   1.17341248e-02],
                 [ -1.11567066e-02,  -5.58477515e-03,   2.21641716e-02],
                 [ -3.31506846e-03,   3.88634008e-03,   8.06118345e-03],
                 [  4.34512318e-03,   7.50109575e-03,  -1.21771515e-02],
                 [ -1.86887536e-02,   1.29835256e-02,   1.91143526e-03],
                 [  2.85840362e-03,   1.35152447e-02,  -2.25387334e-02],
                 [ -2.66397870e-03,  -2.35513669e-02,  -7.22106579e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.21198148e-03,   7.31791895e-03,   2.34800222e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  9.36865635e-03,  -6.66863828e-03,  -5.03523212e-03],
                 [ -4.84690578e-03,  -5.30685623e-03,   6.98423543e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.84772003e-02,   1.14466842e-02,   2.46334578e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -7.27710860e-03,  -1.85300630e-02,   2.00859835e-02],
                 [  2.39012385e-02,  -1.49530572e-02,  -8.20526247e-03],
                 [  1.92447871e-02,  -3.02123736e-04,  -2.79885605e-03],
                 [  2.91575074e-03,   5.20337737e-03,  -6.91793147e-03],
                 [  6.04621719e-03,   2.66054565e-02,   1.47834021e-02],
                 [  1.09999338e-02,   2.11657465e-02,  -4.68955586e-03],
                 [  1.34426626e-02,  -1.41191540e-02,   2.81211534e-02],
                 [ -1.77930338e-02,   1.01245210e-02,   7.63332095e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.74296630e-03,   6.56273168e-03,  -1.95364644e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.59482505e-02,   8.34075808e-03,  -2.33266225e-02],
                 [  8.38055540e-03,  -2.94543949e-03,   1.73144861e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.04114412e-02,  -1.58501753e-02,  -2.21378375e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  8.03081327e-03,   1.13790673e-02,   5.64689820e-03],
                 [  5.10043790e-04,   2.88439536e-02,  -6.67014608e-03],
                 [  2.14386150e-02,   1.75228345e-02,   1.96473631e-02],
                 [ -2.14858837e-02,   1.85899129e-02,   1.83147713e-02],
                 [ -2.40555894e-02,   1.31299799e-02,  -2.33203513e-02],
                 [  5.17518232e-03,   2.52712563e-02,  -2.80330673e-02],
                 [ -2.56625441e-02,  -1.54199573e-02,   5.50388335e-03],
                 [  2.54524578e-02,  -1.46546400e-03,  -7.63287170e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.44019903e-02,  -1.23954794e-02,  -4.34403500e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.22143242e-02,   9.06546222e-03,  -1.54221177e-02],
                 [  3.89170789e-03,  -1.44090602e-02,   9.35491805e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.67174764e-02,   6.93286098e-03,   3.00763160e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.08769174e-02,  -1.22768329e-02,  -3.27832751e-02],
                 [  4.48367534e-03,  -3.27305210e-03,  -6.40923327e-03],
                 [ -2.02485816e-03,  -4.92324332e-04,   7.23722252e-03],
                 [  3.89381829e-03,   6.09672160e-03,  -3.19325802e-02],
                 [ -2.48448834e-02,  -4.11561888e-03,  -1.13136908e-02],
                 [  1.22373499e-02,  -1.80064453e-02,   2.74866109e-02],
                 [ -7.33793503e-03,  -2.71026266e-02,   7.94833740e-03],
                 [  2.62423289e-02,   1.67228918e-02,  -2.32065962e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.33397255e-02,  -1.47908216e-02,   9.94468790e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.47352858e-02,  -3.87346298e-03,  -6.50601242e-03],
                 [ -1.69624143e-02,  -4.02915288e-03,  -1.39839896e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.30476932e-03,  -3.48328449e-03,   4.60276445e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.01238151e-03,   5.18344116e-03,  -5.75004955e-03],
                 [  2.60721463e-02,  -1.74599777e-04,   6.04083430e-03],
                 [  6.01548397e-03,   1.16438162e-02,   2.23818282e-02],
                 [ -2.31845230e-02,   1.95144471e-02,  -1.99405205e-02],
                 [  5.11770858e-03,   1.14277511e-02,  -2.71151291e-02],
                 [  2.53681539e-02,   2.47952553e-04,  -5.50312325e-03],
                 [ -8.33969006e-03,   1.29580049e-02,  -7.50117545e-03],
                 [  2.86100178e-03,   4.05710256e-03,   3.47621968e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.07661073e-02,   9.28179643e-03,   1.52737874e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.57937307e-02,   6.67581068e-03,  -1.05066491e-02],
                 [  4.41620765e-03,  -2.42544386e-02,  -1.76836440e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.58792011e-02,  -7.85093335e-03,   8.90827122e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.56392503e-03,  -3.61307306e-03,  -1.68236026e-04],
                 [  1.64527026e-02,   8.03523606e-03,   2.37586241e-02],
                 [  1.59123729e-02,  -1.36333099e-02,  -1.24478276e-02],
                 [ -1.61572792e-02,  -2.48364180e-02,   1.38655933e-03],
                 [  1.96321095e-02,  -1.51080711e-03,  -4.07979970e-03],
                 [ -2.14620236e-02,  -2.64670318e-02,  -5.57184025e-03],
                 [  2.07603838e-02,  -5.25915447e-03,  -3.03774357e-03],
                 [  5.61258474e-03,   1.57287554e-03,   8.20324841e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.90718553e-02,   2.58239966e-03,  -3.43227901e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.76659184e-03,  -2.66908393e-02,  -5.21134987e-03],
                 [ -1.55535525e-02,  -1.59455276e-02,   2.21519591e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -9.58483469e-03,   1.31408795e-02,   1.03918718e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.02046941e-04,   1.83669079e-03,  -1.35859041e-02],
                 [  6.18961451e-03,  -1.62136398e-02,   7.25883963e-05],
                 [ -1.94945206e-02,   2.37982979e-02,  -2.25631342e-03],
                 [ -6.77574169e-03,  -1.82918200e-02,  -1.64752811e-02],
                 [ -1.49808994e-02,   7.75943240e-03,   6.89794976e-03],
                 [  2.17868846e-02,   2.91818538e-03,   1.15436937e-02],
                 [  2.11933699e-02,  -1.78323179e-02,  -9.17858861e-03],
                 [  6.72873959e-03,   4.26282661e-03,  -1.49498254e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.15273971e-02,  -4.35685591e-03,   1.39793855e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.44455780e-02,   3.13947178e-02,   4.42636762e-03],
                 [ -3.11152961e-02,   1.24336779e-02,   1.61755254e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.56096091e-03,  -1.18272299e-05,   6.54271436e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.56665284e-02,   3.83990005e-03,   2.07920029e-02],
                 [ -1.25044631e-02,   1.53863214e-02,   2.21892267e-02],
                 [  1.51453587e-02,  -1.41276295e-02,   2.66643757e-02],
                 [  3.19383469e-02,  -7.37677588e-03,  -1.27220557e-02],
                 [  1.58333611e-02,  -1.70972970e-02,   1.22499542e-03],
                 [ -2.00703542e-02,   2.17165224e-03,   9.41647047e-03],
                 [  1.14888241e-02,  -7.07415198e-03,  -2.57430881e-02],
                 [ -1.75866512e-02,   2.06580343e-02,  -1.36802136e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.20203209e-03,  -2.01553180e-02,   3.36603628e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.57193632e-03,  -1.92117921e-02,   1.86181987e-02],
                 [  2.53078862e-02,  -2.74574459e-02,   1.64714433e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.79586357e-02,   7.01139164e-03,   1.04627346e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.64888280e-03,  -1.45958577e-02,  -4.06850018e-03],
                 [  5.59360547e-03,   1.61614497e-02,  -4.26622298e-03],
                 [ -1.66582739e-02,  -2.02327308e-02,   2.15809392e-03],
                 [  2.29446651e-02,   6.85882973e-04,   7.75273402e-03],
                 [  3.17399650e-02,   5.02200474e-03,   5.00257947e-04],
                 [ -1.51027431e-02,   2.29994903e-02,  -1.67084415e-02],
                 [ -1.83595250e-02,  -2.41398452e-02,  -8.77530587e-03],
                 [ -5.47344473e-03,   4.65593296e-03,   2.78047987e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.55391916e-02,   3.69105615e-03,  -2.74996422e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.43712373e-02,  -2.05568617e-02,   2.29757225e-02],
                 [ -2.48462273e-02,  -7.59044021e-03,   5.35657021e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -6.85015612e-03,  -1.89143825e-02,  -1.36399124e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.15515041e-02,  -5.46005440e-03,   1.92467420e-02],
                 [  1.51106401e-02,  -1.38608235e-02,   2.18814962e-04],
                 [ -1.83169602e-03,   1.08185071e-03,   3.33748105e-03],
                 [  1.59270161e-02,  -6.56954531e-03,   2.18232099e-02],
                 [  1.68993160e-02,   2.17619158e-02,  -2.06880238e-03],
                 [  1.41561615e-02,   3.26425686e-02,   2.98420288e-03],
                 [  6.17574196e-03,  -2.32523485e-02,  -7.57393344e-04],
                 [ -2.73503337e-02,  -1.41020884e-02,  -1.65279045e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.66962041e-02,   5.82531681e-03,   2.21748260e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.32891300e-02,   2.55543513e-02,   7.34769333e-03],
                 [ -6.20418326e-03,  -1.08140481e-02,  -1.13614174e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.25973111e-02,   3.57551513e-03,   1.13168762e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.39277268e-03,  -4.84352128e-03,  -2.60288792e-02],
                 [ -8.75499276e-03,  -1.10830159e-02,   9.98143584e-03],
                 [ -9.62278455e-03,   2.92171605e-02,   2.93828700e-03],
                 [ -5.64153842e-03,   1.54366347e-02,   4.17050844e-03],
                 [  1.07636284e-03,   1.97872553e-02,   2.26455962e-02],
                 [  1.54518331e-02,   3.93447955e-03,   1.42480023e-02],
                 [  1.37019966e-02,   9.38331656e-04,   2.39747991e-03],
                 [  2.53634271e-02,  -5.75390949e-03,  -4.35901792e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -8.97782727e-03,   1.89386312e-03,  -1.92699239e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.90749538e-03,  -6.03507041e-03,  -2.05945282e-02],
                 [ -9.26883869e-03,  -3.52781505e-03,  -7.72788824e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  7.77225441e-03,  -1.36196676e-02,   2.08648151e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.78105725e-03,   8.06026390e-04,   2.69446429e-02],
                 [ -1.43527386e-02,  -1.52571140e-02,  -1.54483181e-02],
                 [ -1.31636278e-02,   2.87536427e-04,  -5.14872527e-03],
                 [ -2.54919691e-03,   2.43283729e-03,   2.26415705e-03],
                 [ -3.22602445e-03,   1.36510438e-02,  -2.88478277e-02],
                 [  1.25000217e-02,   1.63750713e-02,  -2.01018446e-03],
                 [ -1.21936529e-02,   2.51915159e-02,   7.48534049e-03],
                 [ -1.64209838e-02,  -2.31523595e-03,   2.44043329e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  6.28017635e-03,   1.23187686e-02,  -1.20689490e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.05039439e-02,  -1.89115764e-02,  -3.85391328e-02],
                 [ -1.48690473e-02,  -2.20259534e-02,   1.65646047e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.47117318e-03,   3.22078756e-03,  -1.25801856e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.37443121e-02,  -1.16934383e-02,  -1.91016070e-02],
                 [ -9.42602892e-03,  -1.02870515e-02,  -2.12410102e-02],
                 [  2.01503220e-03,  -1.12656987e-02,  -2.36024308e-02],
                 [ -1.57652522e-02,  -4.85883138e-03,   8.65531626e-03],
                 [ -2.01932818e-02,   7.49184558e-03,  -4.43108216e-03],
                 [ -3.18719837e-03,   1.95970183e-02,  -1.07522279e-02],
                 [ -4.22161271e-03,   6.08589745e-03,   5.01189421e-03],
                 [ -2.83041496e-03,  -1.51576444e-02,  -5.58784868e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  3.02557638e-02,  -7.90216835e-03,  -3.43535630e-04],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.53234008e-02,   1.03637775e-02,  -9.54048721e-03],
                 [  2.98210677e-02,   4.89252278e-03,   9.21436999e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -9.69288656e-03,   1.73936703e-02,   2.20928398e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.69581803e-03,  -5.17559331e-03,   2.38950829e-02],
                 [  8.17472935e-03,   1.53816709e-02,   8.55716081e-03],
                 [ -2.88960677e-03,  -2.08912232e-02,   5.77670666e-04],
                 [ -3.73510559e-03,  -4.51419653e-04,  -1.67628075e-02],
                 [  1.50477789e-02,  -1.07746227e-02,  -3.93245845e-03],
                 [ -2.95335982e-02,   1.71640076e-02,  -1.60740224e-02],
                 [ -1.30454663e-02,  -1.46423011e-02,   2.90893444e-03],
                 [ -3.91492244e-03,   1.10736333e-02,  -3.05187440e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.47010394e-02,  -4.32460696e-04,  -1.34559785e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.01680719e-02,   6.86773023e-03,   1.17132432e-02],
                 [ -2.47359112e-02,  -2.09144444e-02,  -1.34724136e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.76123216e-02,  -3.42671422e-03,  -5.66738150e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  4.35902976e-03,  -3.11237171e-02,   1.29233477e-02],
                 [ -1.36593287e-02,   2.99360077e-03,  -1.89153470e-02],
                 [  1.36012692e-02,   8.96338851e-03,  -1.33144475e-02],
                 [  4.52831633e-03,   1.63420261e-02,  -6.67051163e-03],
                 [  4.21161872e-03,  -1.82826048e-02,  -2.15057801e-02],
                 [  1.59535295e-02,  -1.34472183e-03,   2.28477631e-02],
                 [  7.40742516e-03,  -1.79081916e-02,   8.93456045e-03],
                 [ -3.37584374e-03,  -2.17752133e-03,  -3.95044543e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.75268363e-03,  -1.83908464e-02,   2.74756819e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.87989800e-03,   2.50142332e-02,  -1.96627711e-02],
                 [ -6.38669554e-03,   5.88384720e-03,   3.52390455e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.78825412e-02,   1.63798112e-02,  -2.88470246e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -8.37624049e-03,  -2.83223518e-02,   1.27175286e-02],
                 [  3.56649433e-02,  -5.45378152e-03,  -4.06899289e-03],
                 [ -1.42939730e-03,   7.44884205e-03,   5.94766061e-04],
                 [  6.10994572e-04,  -5.85559699e-03,   2.12351036e-02],
                 [ -1.12066050e-02,  -5.35791105e-03,   3.00418259e-02],
                 [ -2.28437652e-02,   2.48402430e-03,  -1.54676864e-02],
                 [ -3.35483697e-02,   5.41709290e-03,   1.29245759e-02],
                 [ -1.15388649e-02,  -1.75047829e-02,  -1.15529983e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -5.82802685e-03,  -2.83820511e-03,  -3.79367218e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.08435531e-02,  -1.06900506e-02,  -1.60767826e-02],
                 [  1.03266303e-02,   1.89313687e-02,  -1.74220524e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -7.65253373e-03,  -8.81601009e-04,  -4.44874797e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.08867584e-02,  -3.83798164e-03,   1.76440412e-02],
                 [  8.79388799e-03,   1.89819628e-02,   1.39569643e-02],
                 [  4.19111415e-04,  -1.75893266e-03,  -7.01699490e-03],
                 [  5.56903638e-04,   4.05266827e-03,   1.27942112e-02],
                 [ -2.63280539e-02,   5.38332113e-04,   3.42268457e-03],
                 [  1.38581189e-02,  -1.78030342e-02,  -4.41771788e-03],
                 [  4.10809086e-04,  -6.74836427e-04,  -1.87110240e-02],
                 [ -2.22682838e-02,   1.34523795e-02,  -5.66527870e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.84209223e-03,  -1.06232139e-02,  -2.29765884e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.72711495e-02,   2.53152449e-04,   9.11621050e-03],
                 [  1.01210739e-02,   2.22819172e-02,  -8.21635775e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.42475866e-02,   1.87462361e-02,  -2.01689215e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -3.05095913e-03,  -7.93927506e-03,   1.61431341e-03],
                 [  4.11012338e-03,  -2.62029777e-02,   1.31493124e-03],
                 [  1.05806232e-02,   1.85634705e-02,  -2.76277423e-02],
                 [  1.32047953e-02,   6.29726296e-03,   2.44801139e-02],
                 [ -2.17467986e-02,  -4.26870153e-03,   7.52639678e-03],
                 [ -1.01002072e-02,   1.47817036e-02,  -7.05227994e-03],
                 [  1.88299293e-02,   2.72894343e-03,  -1.33240770e-02],
                 [ -4.08727752e-03,   6.09388993e-04,  -1.07529016e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -2.05004560e-02,  -2.88572113e-03,   2.05531703e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -4.40481310e-03,   2.99444175e-02,  -1.34324124e-02],
                 [  2.29546569e-02,   1.57124044e-02,   1.82080712e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.20338704e-02,   2.23800841e-03,  -3.83254567e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.46456084e-02,  -5.48832200e-03,   2.44278540e-02],
                 [  5.38201138e-03,   2.56862025e-02,   8.14154019e-03],
                 [ -1.35018422e-02,   2.10487528e-03,  -3.35637874e-03],
                 [ -1.37934932e-02,   2.47467697e-02,  -9.99062180e-03],
                 [ -9.19426485e-03,   2.64727008e-02,   4.18649629e-03],
                 [ -4.08357312e-03,   1.26206404e-02,   2.59122224e-02],
                 [ -1.14132567e-02,   1.82376113e-02,   2.42191272e-02],
                 [  3.14208479e-02,  -4.21510377e-03,  -3.58551379e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [ -1.67920347e-02,   1.21901131e-02,  -1.15043313e-02],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  2.88619295e-03,  -1.02230123e-02,   1.63226488e-02],
                 [  1.54221746e-02,   1.94894521e-02,   4.63645328e-03],
                 [  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
                 [  1.37858915e-03,   2.58158826e-02,  -7.19496824e-03]]).T)


         def setUp(self):
            self.xml = """<TS_params 
        betapol="0.75" 
        cutoff="20.0 18.0 18.0 18.0"
        cutoff_ms="18.0"
        cutoff_coulomb="20.0"
        tolpol="1e-15" 
        yuksmoothlength="20.0" 
        iesr="2 2 2" 
        a_ew="1e-06" 
        n_types="2" 
        gcut="0.0" 
        pred_order="2" 
        maxipol="60" 
        raggio="0.0" 
        tewald="F" 
        yukalpha="0.1"
        tdip_sr="T">

        <per_type_data atomic_num="8" pol="1.0" z="-1.95853" type="1" />
        <per_type_data atomic_num="14" pol="0.0" z="3.91706" type="2" />

        <per_pair_data atnum_i="8"  atnum_j="8"  C_pol="0.46009932"  D_ms="0.0" gamma_ms="11.642637" B_pol="0.87357114" R_ms="8.0465068" />
        <per_pair_data atnum_i="8"  atnum_j="14" C_pol="-1.5091142"  D_ms="0.0" gamma_ms="10.405794" B_pol="1.977039"   R_ms="4.193651"  />
        <per_pair_data atnum_i="14" atnum_j="14" C_pol="0.0"         D_ms="0.0" gamma_ms="4.5784138" B_pol="0.0"        R_ms="13.113727" />
      </TS_params>
      """
            if do_compare_p1_p2:
               self.p1 = Potential('IP ASAP', param_str=self.xml)
            self.p2 = Potential('IP TS', param_str=self.xml)
            self.cutoff = 20.0*BOHR
            self.debug = False


      if do_compare_p1_p2:
         class TestMD(QuippyTestCase):

            def setUp(self):
               self.xml = """<TS_params 
           betapol="0.75" 
           cutoff="20.0 18.0 18.0 18.0"
           cutoff_ms="18.0"
           cutoff_coulomb="20.0"
           tolpol="1e-5" 
           yuksmoothlength="10.0" 
           iesr="2 2 2" 
           a_ew="1e-06" 
           n_types="2" 
           gcut="0.0" 
           pred_order="2" 
           maxipol="60" 
           raggio="0.0" 
           tewald="F" 
           yukalpha="0.1"
           tdip_sr="T">

           <per_type_data atomic_num="8" pol="14.131863" z="-1.4295594" type="1" />
           <per_type_data atomic_num="14" pol="0.0" z="2.8591188" type="2" />

           <per_pair_data C_pol="0.44302622" atnum_j="8" atnum_i="8" D_ms="0.00030700577" gamma_ms="12.165654" B_pol="1.1221903" R_ms="7.0252019" />
           <per_pair_data C_pol="-1.5003213" atnum_j="8" atnum_i="14" D_ms="0.0020129372" gamma_ms="11.350477" B_pol="1.973181" R_ms="4.5780828" />
           <per_pair_data C_pol="0.0" atnum_j="14" atnum_i="14" D_ms="0.33967532" gamma_ms="-0.17694797" B_pol="0.0" R_ms="-0.085202834" />
         </TS_params>
         """
               if do_compare_p1_p2:
                  self.p1 = Potential('IP ASAP', param_str=self.xml)
               self.p2 = Potential('IP TS', param_str=self.xml)

               self.p1.print_()
               self.p2.print_()

               self.at = alpha_quartz(**quartz_params['CASTEP_LDA'])
               self.at.cutoff = 20.0*BOHR

               self.ds1 = DynamicalSystem(self.at)
               self.ds1.rescale_velo(300.0)
               self.ds1.zero_momentum()
               self.ds1.atoms.calc_connect()
               self.ds1.avg_temp = 0.0

               self.ds2 = DynamicalSystem(self.ds1.atoms.copy())
               self.ds2.atoms.calc_connect()
               self.ds2.avg_temp = 0.0


            def test_md_1step(self):

               self.p1.calc(self.ds1.atoms, args_str="force")
               self.p2.calc(self.ds2.atoms, args_str="force")

               self.assertArrayAlmostEqual(self.ds1.atoms.force, self.ds2.atoms.force)

               self.ds1.advance_verlet1(1.0)
               self.ds2.advance_verlet1(1.0)

               self.assertArrayAlmostEqual(self.ds1.atoms.pos, self.ds2.atoms.pos)

               self.p1.calc(self.ds1.atoms, args_str="force")
               self.p2.calc(self.ds2.atoms, args_str="force")

               self.assertArrayAlmostEqual(self.ds1.atoms.force, self.ds2.atoms.force)

               self.ds1.advance_verlet2(1.0, self.ds1.atoms.force)
               self.ds2.advance_verlet2(1.0, self.ds1.atoms.force)

               self.assertArrayAlmostEqual(self.ds1.atoms.velo, self.ds2.atoms.velo)

               self.ds1.print_status()
               self.ds2.print_status()


            def test_md_2step(self):

               self.p1.calc(self.ds1.atoms, args_str="force")
               self.p2.calc(self.ds2.atoms, args_str="force")

               self.assertArrayAlmostEqual(self.ds1.atoms.force, self.ds2.atoms.force)

               for i in range(2):
                  self.ds1.advance_verlet1(1.0)
                  self.ds2.advance_verlet1(1.0)

                  self.assertArrayAlmostEqual(self.ds1.atoms.pos, self.ds2.atoms.pos)

                  self.p1.calc(self.ds1.atoms, args_str="force")
                  self.p2.calc(self.ds2.atoms, args_str="force")

                  self.assertArrayAlmostEqual(self.ds1.atoms.force, self.ds2.atoms.force)

                  self.ds1.advance_verlet2(1.0, self.ds1.atoms.force)
                  self.ds2.advance_verlet2(1.0, self.ds1.atoms.force)

                  self.ds1.print_status()
                  self.ds2.print_status()

                  self.assertArrayAlmostEqual(self.ds1.atoms.velo, self.ds2.atoms.velo)


            def test_md_10step(self):

               traj1 = self.ds1.run(self.p1, dt=0.5, n_steps=10, save_interval=1)
               traj2 = self.ds2.run(self.p2, dt=0.5, n_steps=10, save_interval=1)

               import itertools

               for i, (at1, at2) in enumerate(itertools.izip(traj1, traj2)):
                  self.assertArrayAlmostEqual(at1.force, at2.force)
                  self.assertArrayAlmostEqual(at1.pos, at2.pos)

class TestPseudise(QuippyTestCase):

   def setUp(self):
      self.xml = """<TS_params label="screened_LDA" betapol="0.75" cutoff="20.0 20.0 18.0 0.0" cutoff_coulomb="20.0" cutoff_ms="18.0" tolpol="1e-10" yuksmoothlength="10.0" iesr="-1 -1 -1" a_ew="1e-06" n_types="2" gcut="0.0" pred_order="2" maxipol="60" raggio="0.0" tewald="F" yukalpha="0.1">

     <per_type_data pseudise_sigma="1.0" atomic_num="8" pol="14.131863" z="-1.4295594" type="1" />
     <per_type_data pseudise_sigma="1.0" atomic_num="14" pol="0.0" z="2.8591188" type="2" />

     <per_pair_data C_pol="0.44302622" atnum_j="8" atnum_i="8" D_ms="0.00030700577" gamma_ms="12.165654" B_pol="1.1221903" R_ms="7.0252019" />
     <per_pair_data C_pol="-1.5003213" atnum_j="8" atnum_i="14" D_ms="0.0020129372" gamma_ms="11.350477" B_pol="1.973181" R_ms="4.5780828" />
     <per_pair_data C_pol="0.0" atnum_j="14" atnum_i="14" D_ms="0.33967532" gamma_ms="-0.17694797" B_pol="0.0" R_ms="-0.085202834" />

     </TS_params>"""

      self.energy_ref = -173.214019568
      self.force_ref = farray([[  8.61080094e-03, -1.49143447e-02, -4.14564379e-15],
                               [  8.61080094e-03,  1.49143447e-02,  4.19267327e-14],
                               [ -1.72216019e-02, -1.54343323e-15,  3.51674061e-15],
                               [ -1.00690873e-01,  2.46829816e-01, -1.65689273e-01],
                               [ -1.63415455e-01, -2.10615762e-01, -1.65689273e-01],
                               [  2.64106328e-01, -3.62140542e-02, -1.65689273e-01],
                               [ -1.00690873e-01, -2.46829816e-01,  1.65689273e-01],
                               [  2.64106328e-01,  3.62140542e-02,  1.65689273e-01],
                               [ -1.63415455e-01,  2.10615762e-01,  1.65689273e-01]]).T
      self.local_energy_ref = farray([-36.64386835, -36.64386835, -36.64386835, -10.54706909, -10.54706909,
                                      -10.54706909, -10.54706909, -10.54706909, -10.54706909])

   def test_pseudise_alpha_quartz(self):
      p = Potential('IP TS', param_str=self.xml)
      a = alpha_quartz(**quartz_params['ASAP_JRK'])
      a.set_cutoff(p.cutoff())
      a.calc_connect()
      p.calc(a, force=True)
      p.calc(a, args_str="force energy local_energy calc_charge=T calc_short_range=T calc_sc_dipoles=F calc_dipoles=T pseudise=T")
      self.assertAlmostEqual(a.energy, self.energy_ref)
      self.assertArrayAlmostEqual(a.force, self.force_ref)
      self.assertArrayAlmostEqual(a.local_energy, self.local_energy_ref)
      self.assertAlmostEqual(sum(a.local_energy), a.energy)

if __name__ == '__main__':
   unittest.main()
