#### LIBRARIES NEEDED
#!python
from timeit import default_timer as timer
from ase.io import read,write
from ase.visualize import view
from ase.data import atomic_masses,atomic_numbers
import numpy as np
from numpy.linalg import norm
import matplotlib.pyplot as plt
from scipy.constants import physical_constants
import warnings
import matplotlib.cbook
warnings.filterwarnings("ignore",category=matplotlib.cbook.mplDeprecation)
#### END LIBRARIES NEEDED

#### FUNCTION CALCULATION OF Kinetic energy
def ekin(m,v):
    ek=0.5*np.sum([m[i]*np.dot(v[i],v[i]) for i in range(len(m))])
    return ek
#### END FUNCTION CALCULATION OF Kinetic energy

#### FUNCTION CALCULATION OF FORCES and potential energy
def forces_pot(pos,sigma,epsilon):
    f=np.zeros_like(pos)
    ep=0.0
    for i in range(len(pos)-1):
        for j in range(i+1,len(pos)):
            dij=pos[i]-pos[j]
            d=norm(dij)
            #### FAST
#           sd=sigma/d
#           sd6=sd**6
#           sd12=sd6*sd6
#           ep=ep+4.0*epsilon*(sd12 - sd6)
#           fr=4*epsilon/d*(12.0*sd12 -6.0*sd6)
            #### ENDFAST

            #### SLOW
            ep=ep+4.0*epsilon*(sigma**12/d**12 - sigma**6/d**6)
            fr=4*epsilon*(12.0*sigma**12/d**13 -6.0*sigma**6/d**7)
            ####END SLOW
            f[i]=f[i]+fr*dij/d
            f[j]=f[j]-f[i]
    return f,ep          
#### END FUNCTION  CALCULATION OF FORCES and potential energy

#### INPUT paramaters
filein =raw_input("name of geometry file ")
dt =float(raw_input("timestep in [fs] "))
nsteps=int(raw_input("number of steps "))
T0=float(raw_input("Initial temperature in K "))
#### END INPUT paramaters


#### ATOMIC UNITS
kb=physical_constants['Boltzmann constant in eV/K'][0]
hartree=physical_constants['hartree-electron volt relationship'][0]
kb=kb/hartree
bohr=physical_constants['Bohr radius'][0]
hartree_joule=physical_constants['hartree-joule relationship'][0]
amc=physical_constants['atomic mass constant'][0]
aum=physical_constants['atomic unit of mass'][0]
me=amc/aum
tfact=physical_constants['atomic unit of time'][0]
tfact=tfact * 10**15
dt=dt / tfact 
epsilon=1.67 * 10**-21 #[J]
epsilon=epsilon/hartree_joule
sigma=3.405 * 10**-10    #[m]
sigma=sigma/bohr
#### ATOMIC UNITS

#### READING initial geometry and conversion to atomic units
geo=read(filein)
natoms=len(geo)
pos=geo.get_positions()
pos=pos/(bohr*10**10)
m=np.array([atomic_masses[atomic_numbers[a.symbol]] for a in geo ])
m=m*me
#### END READING initial geometry


#### Initialization plotting vectors
ekin_plot=[]
epot_plot=[]
etot_plot=[]
#graphs="False"   ### PUT False for performance analysis
graphs="True"   ### PUT False for performance analysis
prints="True"  ### Put False for performance analysis
#### END Initialization plotting vectors

#### Initial values
#v=np.zeros_like(pos)

#### Initailize random velocities
v=np.random.rand(natoms,3)
v=v-0.5
#### END Initailize random velocities

#### Removing linear momentum
vm=np.sum(np.array([m[i]*v[i] for i in range(natoms)]),axis=0)
v=v-vm/(np.sum(m))
####END  Removing linear momentum

#### RESCALING initial velocities to desired T
ekini=ekin(m,v)
tini=ekini*2.0/3.0/kb/natoms
v=v*np.sqrt(T0/tini)
#### END RESCALING initial velocities to desired T

ek=ekin(m,v)
#print "tini, T0, ekin ",tini,T0,ek
t=0
f,epot0=forces_pot(pos,sigma,epsilon)
a=[f[i]/m[i] for i in range(natoms)]
ek=ekin(m,v)
etot0=epot0+ek
ekin_plot.append([t,ek])
epot_plot.append([t,epot0])
etot_plot.append([t,epot0+ek])
#### END Initial values

#### Initialization of the images
fig,(ax1,ax2)=plt.subplots(1,2,figsize=(10,5))
plt.ion()
x,y,z=zip(*pos)
emin=epot0*1.05
emax=epot0+ek*1.15
xmin=np.amin(x)-5
xmax=np.amax(x)+5
ymin=np.amin(y)-5
ymax=np.amax(y)+5
#### END Initialization of the images


#### OPEN out file
if prints=="True":
    out=open("data.out",'w')


#### DYNAMICS LOOP
if prints=="True":
    out.write(" STEP    Time[fs]     Ekin[Hartree]  Epot[Hartree]     Etot[Hartree]    Epot[LJ]      T[K] \n")   
print " STEP    Time[fs]     Ekin[Hartree]  Epot[Hartree]     Etot[Hartree]    Epot[LJ]      T[K] \n"   

#### PERFORMANCE, START the time counter
start = timer()
for i in range(nsteps):
    t=t+dt
    f,ep=forces_pot(pos,sigma,epsilon)
    a=np.array([f[j]/m[j] for j in range(natoms)])
    ek=ekin(m,v)
    temper=ek*2.0/3.0/kb/natoms

    #### PRINTING to screen and to file, comment for PERF ANALYSIS
    if prints=="True":
        #### Print on screen
        if (np.mod(i,20)==0 or i==nsteps-1):
            print "{:^7d} {:^12.6f} {:^12.6f} {:^18.6f} {:^18.6f} {:^8.2f}\
            {:^4.2f}".format(i,t*tfact,ek,ep,ek+ep,ep/epsilon,temper)

        #### Print on file
        out.write("{:^7d} {:^12.6f} {:^12.6f} {:^18.6f} {:^18.6f} {:^8.2f}\
        {:^4.2f} \n" .format(i,t*tfact,ek,ep,ek+ep,ep/epsilon,temper))
    ####END PRINTING to screen and to file, comment for PERF ANALYSIS

    ekin_plot.append([t*tfact,ek])
    epot_plot.append([t*tfact,ep])
    etot_plot.append([t*tfact,ep+ek])
    pos=pos + dt * v + 0.5 * dt * dt * a
    f,ep=forces_pot(pos,sigma,epsilon)
    a1=np.array([f[j]/m[j] for j in range(natoms)])
    v=v+0.5 * dt * (a + a1)
    a=np.copy(a1)

    #### PLOTS SECTION
    if (np.mod(i,20)==0 or i==nsteps-1) and graphs=="True" :
        ax1.set_title('Energies')
#       ax1.set_aspect(1)
        ax1.axes.set_xlim([0,dt*nsteps*tfact])
        ax1.axes.set_ylim([emin,emax])
#       ax1.invert_yaxis()
        ax1.set_xlabel('time [fs]')
        ax1.set_ylabel('Energy [a.u.]')
        x,y=zip(*epot_plot)
        ax1.plot(x,y,'b-',label='epot')
        x,y=zip(*etot_plot)
        ax1.plot(x,y,'r-',label='etot')
        ax1.legend(loc="lower right")

        x,y,z=zip(*pos)
        for j in range (natoms):
            xi=x[j]
            yi=y[j]
            ax2.add_artist(plt.Circle((xi,yi),1.5 , color='r', fill=False, clip_on=True))
#       for j in range (na):
#           ax1.scatter(x.tolist()[j][i],y.tolist()[j][i],s=40)
#       ax1.legend(loc="upper right")
        ax2.set_title('positions')
        ax2.axes.set_xlim([xmin,xmax])
        ax2.axes.set_ylim([ymin,ymax])
        ax2.set_xlabel('position [a.u.]')
        ax2.set_ylabel('position [a.u.]')
        ax2.set_aspect(1)
#       ax2.scatter(cm.tolist()[i],pp.tolist()[i],color='red',s=40)
        if i == nsteps - 1:
            fig.savefig("plots.png", dpi=200)
        plt.show()
        plt.pause(0.1)
        ax1.clear()
        ax2.clear()
#       ax3.clear()
    #### END PLOTS SECTION

#### END DYNAMICS LOOP

#### CLOSE OUT FILE
if prints=="True":
    out.close()

end = timer()
print "time for execution", end - start
####END PERFORMANCE, STOP the time counter
geo.set_positions(pos*(bohr*10**10))
geo.write("final.xyz")

